//---------------------------------------------------------------------------
// File   : C:\pic\dsPIC33EP_Si5351_Synth\si5351.h
// Author : Wolfgang Buescher, DL4YHF
// Date   : 2021-10-31
// Purpose: Functions to initialise and control an Si5351A/B/C,
//          and possibly similar RF synthesisers by Silicon Labs.
//          Further details in the *.c module.
// Disclaimer / Term of use : See "implementation", si5351.c .

#ifndef  si5351_h_included  // prevent multiple inclusion...
# define si5351_h_included  // ... and let other modules know "we've been included"

#ifndef  SWI_SI5351_REFERENCE_FREQUENCY_HZ // <- should have been defined in SWITCHES.H ...
# define SWI_SI5351_REFERENCE_FREQUENCY_HZ 25000000 // ...otherwise assume 25 MHz as on the Adafruit board
#endif

#ifndef  SWI_SI5351_READ_BACK_REGISTER // <- if not defined in SWITCHES.H ...
# define SWI_SI5351_READ_BACK_REGISTER 0 // assume we don't need to READ BACK (and VERIFY) registers via I2C
#endif

#ifndef  SWI_SI5351_SUPPORT_FREQUENCY_MODULATION // <- if not defined in SWITCHES.H ...
# define SWI_SI5351_SUPPORT_FREQUENCY_MODULATION 1 // 0=no FM, 1=preferred via PLL feedback, 2=preferred via OUTPUT divider
        // (compatible with FM_METHOD_OFF(0), FM_METHOD_PLL_FEEDBACK(1), FM_METHOD_OUTPUT_DIV(2) )
#endif

#ifndef  SWI_SI5351_ALLOW_POKING_REGISTERS_FROM_MENU // <- if not defined in SWITCHES.H ...
# define SWI_SI5351_ALLOW_POKING_REGISTERS_FROM_MENU 1 // 1="allow poking registers", e.g. from the "Synth Registers" menu
#endif

#define  SI5351_OUTPUT_FREQUENCY_MIN 2000      // min 2 kHz (slighty exceeding the spec, but not problem)
#define  SI5351_OUTPUT_FREQUENCY_MAX 200000000 // max 200 MHz (if you need more, use an ADF4351 instead)

#define  SI5351_NUM_PLLS_AND_OSCILLATORS   2   
#define  SI5351_NUM_ACTIVE_OUTPUT_CHANNELS 2   

#define  SI5351_NUM_REGISTERS 188 // number of 'known' 8-bit registers in an Si5351 (0="Device Status"..187="Fanout Enable")

typedef struct tSi5351_FractionalDividerParams
{
  // Basic function of an OUTPUT fractional divider in an Si5351:
  //  > f_out = f_in / ( ( a + b / c  )  *  r_div )
  //     |       '--- for OUTPUT dividers, f_in = f_vco (600 .. 900 MHz)
  //     '----------- "frequency available at the OUTPUT pins" (e.g. 10.14 MHz)
  //     Explained in AN619, page 6,
  //      > "4.1.2. Output Multisynth Divider Equations (Fout <= 150 MHz)"
  //
  // Basic function of a PLL FEEDBACK fractional divider in an Si5351:
  //  > f_out = f_in *   ( a + b / c  )
  //     |       '--- for PLL FEEDBACK dividers, f_in  = f_ref (e.g. 25 MHz)
  //     '----------- for PLL FEEDBACK dividers, f_out = f_vco (600 .. 900 MHz)
  //     Explained in AN619, page 3,
  //     > "3.2. Feedback Multisynth Divider Equations"
   
  double f_in, f_out; 
  DWORD a, b, c; // fractional divider parameters shown above (see AN619); later replaced by p1,p2,p3
  DWORD r_div;   // 1,2,4..128; can be used to generate frequencies below about 500 kHz
} T_Si5351_FractionalDividerParams;

typedef struct tSi5351_FractionalDividerParamsP1P2P3
{
  DWORD p1, p2, p3; // 18-bit integer part, 20-bit numerator, 20-bit denominator
                 // for any Si5351 'Multisynth' (PLL-feedback and OUTPUT divider)
  DWORD r_div;   // 1,2,4..128; can be used to generate frequencies below about 500 kHz
                 // (only available in OUTPUT dividers but not in PLL-feedback-dividers)
  // ('thank you, Silicon Labs' for these intuitive register/bitgroup names,
  //   and all those 'special cases' above 112.5 and 150 MHz.. )
} T_Si5351_FractionalDividerParamsP1P2P3;

typedef union tSi5351_FeedbackDividerRegs // // Register-like struct; one per *PLL* (incl. VCO)
{
  struct
   { BYTE P3_M;  // e.g. Register 26, bits 15..8 of 20-bit denominator for the fractional part of a PLL FEEDBACK divider
     BYTE P3_L;  // e.g. Register 27, bits 7..0 of 20-bit denominator for the fractional part of a PLL FEEDBACK divider
     BYTE P1_H;  // e.g. Register 28, bits 1..0 = bits 17..16 of 18-bit integer part of a PLL FEEDBACK divider
     BYTE P1_M;  // e.g. Register 29, bits 15..8 of 18-bit integer part of a PLL FEEDBACK divider
     BYTE P1_L;  // e.g. Register 30, bits 7..0 of 18-bit integer part of a PLL FEEDBACK divider
     BYTE P32_H; // e.g. Register 31, bits7..4 = bits 19..16 of 20-bit DENOMINATOR 'P3',
                 //                   bits3..0 = bits 19..16 of 20-bit NUMERATOR 'P2'.
     BYTE P2_M;  // e.g. Register 32, bits 15..8 of 20-bit numerator of the fractional part 'P2'
     BYTE P2_L;  // e.g. Register 33, bits 7..0 of 20-bit numerator of the fractional part  'P2'
   } b; // ".b" = NAMED BYTE registers (debug-friendly when 'watched')
  BYTE bArray[8]; // access THE SAME REGISTERS as a byte-array
} T_Si5351_FeedbackDividerRegs;

typedef union tSi5351_OutputDivider // Register-like struct; one per OUTPUT CHANNEL
{
  struct
   { BYTE P3_M;    // e.g. Register 42, bits 15..8 of 20-bit denominator for the fractional part of an output MultiSynth divider)
     BYTE P3_L;    // e.g. Register 43, bits 7..0 of 20-bit denominator for the fractional part of an output MultiSynth divider)
     BYTE P1H_DIV; // e.g. Register 44, bits 6..4 : R0 Output Divider (div-by-1,2,4,8,16,32,64,128),
                   //                   bits 3..2 : "MS0 Divide by 4 Enable". 11=div by 4,  00=div by anything else,
                   //                   bits 1..0 = bits 17..16 of 18-bit integer part of an output MultiSynth divider
     BYTE P1_M;    // e.g. Register 45, bits 15..8 of 18-bit integer part of an output MultiSynth divider
     BYTE P1_L;    // e.g. Register 46, bits 7..0 of 18-bit integer part of an output MultiSynth divider
     BYTE P32_H;   // e.g. Register 47, bits7..4 = bits 19..16 of 20-bit DENOMINATOR,
                   //                   bits3..0 = bits 19..16 of 20-bit NUMERATOR.
     BYTE P2_M;    // e.g. Register 48, bits 15..8 of 20-bit numerator of the fractional part
     BYTE P2_L;    // e.g. Register 49, bits 7..0 of 20-bit numerator of the fractional part
   } b; // ".b" = NAMED BYTE registers (debug-friendly when 'watched')
  BYTE bArray[8]; // access THE SAME REGISTERS as a byte-array
} T_Si5351_OutputDividerRegs; 

typedef union tSi5351_Regs // Register-like struct; one for the ENTIRE CHIP
{
  struct 
   { BYTE DeviceStatus;          // [0] = "Device Status" (AN619 page 15)
     BYTE InterruptStatusSticky; // [1] = "Interrupt Status Sticky" (AN619 p 16)
     BYTE InterruptStatusMask;   // [2] = "Interrupt Status Mask" (AN619 p 17)
     BYTE OutputEnableControl;   // [3] = "Output Enable Controls" (page 18)
     BYTE b4_8Rsvd[ 8-4 + 1];    // [4..8]
     BYTE OEBPinEnableControlMask; // [9] 
     BYTE b10_15Rsvd[14-10 + 1]; // [10..14]
     BYTE PLLInputSource;        // [15]
     BYTE b8CLKControl[8];       // [16..23] (AN619 pages 20..27) "Multisynth x" (output) INTEGER MODE, etc etc
     BYTE CLK3_0_DisableState;   // [24] (AN619 page 28)
     BYTE CLK7_4_DisableState;   // [25] (AN619 page 28)
     T_Si5351_FeedbackDividerRegs FeedbackDivider[2]; // [26..33], [34..41]
          // aka "Multisynth NA,NB Parameters" (whatever that stands for - it's the PLL FEEDBACK)
     T_Si5351_OutputDividerRegs OutputDivider[6]; // [42..49], [50..57], [58..65] .. [82..89]  
          // aka "Multisynth0..5 Parameters" (funny names for the FRACTIONAL OUTPUT DIVIDERS)
     BYTE b2IntegerOutputDividerRegs[2]; // [90..91] "Multisynth6,7" (can only divide by INTEGERS)
     BYTE CLK7_6_OutputDivider;  // [92] (AN619 page 53)
     BYTE b93_148_Rsvd[148-93 + 1]; // [93..148] RESERVED (what a waste..)
     BYTE bSpreadSpectrumParams[161-149 + 1]; // [149..161] = "Spread Spectrum Parameters" (pages 54..57)
     BYTE bVCXOParams[164-162 + 1]; // [162..164] = "VCXO Parameters" (pages 57..58)
     BYTE b6CLKInitialPhaseOffset[170-165+1]; // [165..170] = "CLK0..5 Initial Phase Offset" (page 58)
     BYTE b171_176_Rsvd[176-171 + 1];     // [171..176] RESERVED/wasted RAM :o(
     BYTE PLL_Reset;                      // [177] "PLL Reset" (AN619 page 61)
     BYTE b178_182_Rsvd[182-178 + 1];     // [178..182] even more wasted RAM :o(
     BYTE CrystalLoadCapacitance;         // [183] AN619 page 61: "2 bits determine the internal load capacitance value for the crystal".
     BYTE b184_186_Rsvd[186-184 + 1];     // [184..186] ..and the waste goes on, but THESE unknown registers contained neither 0x00 nor 0xFF !
     BYTE FanoutEnable; // [187] AN619 page 62: "enable fanout of XYZ to clock output multiplexers. Set this to 1b. Reset value undefined." Buaaaah.
                        //   ,------------------------------------'
                        //   '--> XYZ="CLKIN" (bit 7), "XO" (bit 6), "Multisynth0 and ..4" (bit 4). What the heck.. ? ?
                        // When READ after power-on, FanoutEnable (register #187) contained 0x02 .
     // This was the last DOCUMENTED register number in AN619 .
     // On a PIC, we don't waste even more precious RAM for the remaining
     // "reserved" registers (indices 188..255, AN619 page 13) !
   } g; // ".g" = NAMED GROUPS of registers (debug-friendly when 'watched')
  BYTE bArray[SI5351_NUM_REGISTERS]; // access THE SAME REGISTERS as a byte-array

} T_Si5351_Regs; // struct containing ALL DOCUMENTED registers in an Si5357
extern T_Si5351_Regs Si5351Regs; // 'shadow registers' of the Si5357 in CPU RAM 

extern double g_dblReferenceFrequency_Hz; // EDITABLE at runtime ! 
  // SWI_SI5351_REFERENCE_FREQUENCY_HZ is just a default for this variable.
  // g_dblReferenceFrequency_Hz has no module prefix because it will be used
  // in other 'synthesizer drivers' too, for example in ADF4351.C .

extern BYTE Si5351_bTotalNumErrors; // summed-up Si5351-related errors counted since init 
            // (maxes out near 255, but the lower 4 bits always 'stay alive')
extern BYTE Si5351_bLastErrorCode;  // 0 = "no error code yet", or one of the following:
#  define Si5351_ERROR_NONE     0
#  define Si5351_ERROR_I2C_FAIL 1 // Error when trying to communicate with the chip via I2C bus
#  define Si5351_ERROR_VERIFY   2 // unexpected register content when READING BACK / VERIFYING a register content.
                                // When set, Si5351_bLastErrorParam contains the REGISTER NUMBER.
extern BYTE Si5351_bLastErrorParam; // 'additional info', meaning depends on Si5351_bLastErrorCode .

#if( SWI_SI5351_ALLOW_POKING_REGISTERS_FROM_MENU )
extern BOOL Si5351_fPokingAround; // flag controlled by Si5351_RWAccessForMenuItems(), TRUE when "poking around" in certain registers (P1,P2,P3).
#endif // SWI_SI5351_ALLOW_POKING_REGISTERS_FROM_MENU ?

#if( SWI_SI5351_SUPPORT_FREQUENCY_MODULATION )
# if( SWI_SI5351_SEND_FM_STREAM_IN_I2C_COMPLETION_CALLBACK ) 
extern BYTE  Si5351_bFMStreamState; // may contain one of the following:
#       define FM_STREAM_OFF    0 // stream of FM samples to the Si5351 permanently 'off'
#       define FM_STREAM_ON     1 // stream of FM samples is 'running'
#       define FM_STREAM_ERROR  2 // sample stream broken, e.g. due to I2C bus error.  
#       define FM_STREAM_PAUSED 3 // sample stream temporarily paused, e.g. to 'QSY'.
                                  // Similar effect as FM_STREAM_OFF, but will
                                  // automatically turn the 'FM stream' on again
                                  // as soon as possible.
# endif // SWI_SI5351_SEND_FM_STREAM_IN_I2C_COMPLETION_CALLBACK ?
#endif // SWI_SI5351_SUPPORT_FREQUENCY_MODULATION ?

//---------------------------------------------------------------------------
BOOL Si5351_Init(void); // returns TRUE if an Si5351 was detected on the I2C bus
void Si5351_ClearErrors(void);  // Clears Si5351_bTotalNumErrors, etc (?)
BOOL Si5351_SetFrequency( // programs one of the first two outputs for a given frequency
        int    iOutputIndex,           // [in] zero-based index for PLL *and* output divider
        double dblWantedFrequency_Hz,  // [in] "wanted" frequency in Hertz
        int    iFreqModDeviation_Hz ); // [in] deviation for frequency modulation, 0=none
BOOL Si5351_SetVcoPhaseOffset(  // See AN619 pages 10 and 58 ! 
        int iOutputIndex, // [in] zero-based index for the PLL *and* the output divider
        BYTE bOffset);    // [in] 0..127 times 1 / (4*Fcvo) -> ca. 0 ... 35 ns .

#if(SWI_SI5351_SUPPORT_FREQUENCY_MODULATION) 
//------------------------------------------------------------------------
extern WORD APPL_GetModulatorSampleByTimestamp( WORD wTimestamp_us ); // really 'extern' !
void Si5351_SendFMSample(WORD u12AudioSample); // [in] 12-bit audio sample, 0..4095
void Si5351_PauseSendingFM(void);
void Si5351_StartSendingFM(void);
#endif // SWI_SI5351_SUPPORT_FREQUENCY_MODULATION ?

#if( SWI_SI5351_ALLOW_POKING_REGISTERS_FROM_MENU ) // ..actually "peeking and poking"..
//------------------------------------------------------------------------
int Si5351_RWAccessForMenuItems( // service for the 'Hardware Diagnostics'; explained further below.
              BYTE bParamID,   // [in] "parameter identifier", here: bitwise combined as explained BELOW
              BYTE bAccess, BYTE bDataType, BYTE bMaxLength, void *pvValue, void *pvMenuItem);
  // Invoked via function pointer in a T_MENU_ITEM with data_type=DTYPE_GET_LONG.
  // The function prototype must be compatible with T_SL_GET_SET_ANY_FUNC (see micro_string.h).
  // Parameter-IDs implemented for this function so far:
# define Si5351_PARAM_NONE 0
# define Si5351_PARAM_FRACTIONAL_DIV_A  1 // fractional divider parameter 'A' (in A+B/C)
# define Si5351_PARAM_FRACTIONAL_DIV_B  2 // fractional divider parameter 'B' (in A+B/C)
# define Si5351_PARAM_FRACTIONAL_DIV_C  3 // fractional divider parameter 'B' (in A+B/C)
# define Si5351_PARAM_FRACTIONAL_DIV_P1 4 // bitgroup 'P1', mostly the integer part 'A'
# define Si5351_PARAM_FRACTIONAL_DIV_P2 5 // bitgroup 'P2', mostly the numerator 'B'
# define Si5351_PARAM_FRACTIONAL_DIV_P3 6 // bitgroup 'P3', mostly the denominator 'C'
# define Si5351_PARAM_VCO_FREQUENCY     7 // theoretic VCO frequency, calc'd from P1,P2,P3
# define Si5351_PARAM_OUTPUT_FREQUENCY  8 // theoretic OUTPUT frequency, calc'd from P1,P2,P3   
        // The above 'fractional divider parameters' occurr multiple times in an Si5351.
        // The UPPER BITS in the 8-bit 'Parameter-ID' define WHERE:
# define Si5351_PARAM_PLL1_FEEDBACK 0x10 // in SiLabs geek speak, "Multisynth NA"
# define Si5351_PARAM_PLL2_FEEDBACK 0x20 // in SiLabs geek speak, "Multisynth NB"
# define Si5351_PARAM_CLK0_OUTPUT   0x30 // in SiLabs geek speak, "Multisynth 0"
# define Si5351_PARAM_CLK1_OUTPUT   0x40 // in SiLabs geek speak, "Multisynth 1"
# define Si5351_PARAM_CLK2_OUTPUT   0x50 // in SiLabs geek speak, "Multisynth 2"
#endif // SWI_SI5351_ALLOW_POKING_REGISTERS_FROM_MENU ?




#endif // ndef  si5351_h_included ?
