/*========================================================================*/
/* File:    C:\pic\dsPIC33EP_Si5351_Synth\si5351.c                        */
/* Date:    2021-11-15                                                    */
/* Author:  Wolfgang Buescher (DL4YHF) .                                  */
/* Purpose: Functions to initialise and control an Si5351A/B/C,           */
/*          and possibly similar RF synthesisers by Silicon Labs.         */
/*        Tries to realize the best possible setting for a given          */
/*        'wanted' frequency by determining numerator and denominator     */
/*        using the Farey algorithm.                                      */
/*        Details in C:\elektronik_projekte\Si5351_Clock_Generator\       */
/*                Documentation\Si5351_Calculations.txt .                 */
/*      Later, an optional 'frequency modulation via software' was added, */
/*                controlled via SWI_SI5351_SUPPORT_FREQUENCY_MODULATION. */
/*                Details about the frequency modulation principle        */
/*                in C:\Elektronik_Projekte\Si5351_Clock_Generator\       */
/*                       Documentation\Si5351_Frequency_Modulation.txt .  */
/*                                                                        */
/*        This code was originally written to run on a dsPIC33EP, with    */
/*        sufficient RAM and ROM for 'double' (64-bit) floating point     */
/*        numbers to make life easier than with pure integer arithmetics. */
/*                                                                        */
/*  Used at least in the following projects (by DL4YHF):                  */
/*        - 'Si5351A dual frequency synthesiser'; will possibly be used   */
/*          used in a 23 cm FM repeater to eliminate custom-made crystals */
/*          near 1298 MHz divided by 128 = circa 10.14xxx MHz output from */
/*          the TWO (not THREE) independent VCOs in an Si5351A .          */
/*                                                                        */
/* Literature:                                                            */ 
/* [Si535DS] Si5351A/B/C datasheet Rev. 1.3 3/20, titled                  */
/*          "I2C-PROGRAMMABLE ANY-FREQUENCY CMOS CLOCK GENERATOR + VCXO", */
/*          locally saved as C:\datasheets\PLL_and_RF_Special_ICs\        */
/*             Si5351_A_B_C_I2C_Programmable_Clock_Generator.pdf          */
/*          Take the claimed "0 ppm error" with a grain of salt !         */
/*  [AN619] SiLab's AN619 Rev. 0.82, "Conceptualizing a Frequency Plan",  */ 
/*          locally saved as C:\datasheets\PLL_and_RF_Special_ICs\        */
/*             Si5351_Manually_Generating_a_Register_Settings_AN619.pdf . */
/* [BITX20] Experiences by Hans Summers G0UPL, posted on the BITX20 group,*/
/*          forwarded to                                                  */
/*          groups.io/g/BITX20/topic/si5351a_facts_and_myths/5430607 .    */
/* [Si5351_FM] Experiments with Frequency Modulation via Software         */
/*          by DL4YHF, original project notes, equations, calculations,   */
/*          and test results documented in C:\pic\dsPIC33EP_Si5351_Synth\ */
/*                            Documentation\Si5351_FM_Tests_2021_11.txt . */
/*========================================================================*/
//
// Disclaimer / Terms of use
//  Copyright (c) 2020, Wolfgang Buescher (DL4YHF) .
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR ''AS IS'' AND ANY EXPRESS OR
// IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
// OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
// IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
// INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
// NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
// THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//

#include "switches.h" // project specific 'compiler' switches & options 
#include "iop.h"      // old-fashioned data types and target-specific "I/O port definitions"
#if( SWI_USE_PROGRAMMED_I2C ) // use DL4YHF's "software" I2C bus master ?
# include "i2c_soft_master.h" // (fast enough to read from EEPROM, configure PLL, etc)
#endif // SWI_USE_PROGRAMMED_I2C ?
#if( SWI_USE_HARDWARE_I2C ) // use DL4YHF's "hardware" I2C bus master for dsPIC33EP ?
# include "i2c.h" // (fast enough to 'QSY' with an Si5351 thousands of times per second)
#endif // SWI_USE_HARDWARE_I2C ?
#include "micro_string.h"  // SL_FillMem(), etc
#include "micro_math.h"    // MM_AbsInt() [absolute integer] and other low-level 'math' functions
#include "Farey.h"    // convert floating point ratio into a fraction of integers
#include "si5351.h"   // header file for THIS module (Si5351A/B/C driver)

#if( SWI_SI5351_ALLOW_POKING_REGISTERS_FROM_MENU ) 
# include "micro_menu.h" // may be required for the IMPLEMENTATION of Si5351_RWAccessForMenuItems()
#endif // SWI_SI5351_ALLOW_POKING_REGISTERS_FROM_MENU ?


//#if( sizeof(double) != 8 )
//#  error "Please tell the stupid compiler to use 64-bit 'double' !"
//  (unfortunately, unlike Borland C, the compiler (XC16) was also too stupid
//   to evaluate 'sizeof( SOME-STANDARD-IN-DATA-TYPE )' in a preprocessor condition)
//#endif


// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
// Constant definitions, register numbers, register bitgroup def's, etc :
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 

#define SI5351_I2C_7BIT_SLAVE_ADDRESS 0x60 // 7-bit slave address [Si5351DS page 18]
  // When used in I2C_SendSlaveAddress, shift the 7-bit address left into bits 7..1,
  //  and append the usual "Read / NotWrite"-flag in bit zero:
  //  Bit 0 = 0 : WRITE (single byte or "burst with auto address increment"),
  //  Bit 0 = 1 : READ  (single byte or "burst with auto address increment").
  // The datasheet [Si5351DS pages 18..19] also clearly shows when to send
  // ACKKNOWLEGDE ("A"), NOT-ACKNOWLEDGE ("N"), and STOP ("P") via I2C bus.

// About the Si5351 register names: There's a long list of registers
// [AN619 page 13] but none of those registers has a UNIQUE NAME.
// Only BITS or BITGROUPS have names, but not very intuitive ones, e.g.
//  Register 0 :  bit 7 = SYS_INIT,  bit 6 = LOL_B,  bit 5 = LOL_A, etc.
//  Lots Of Laughter, A and B ? Oh well. The "register summary" doesn't 
//  show any register name, but [AN619] chapter 9 "Register Descriptions"
//  has a few names which we will use where applicable.
// Naming conventions used in THIS module:
// Si5351_REG_xyz is a REGister number,
// Si5351_OFS_xyz is an address OFFSET in a group of registers with the same type,
// Si5351_MSK_xyz is a bitmask for a bit (or group of bits) in register 'xyz',
// Si5351_VAL_xyz is a BITGROUP VALUE that may be written into 'xyz',
//             bitwise ORed with other bitgroup values in the same register.
// 
#define Si5351_REG_DEVICE_STATUS 0 // b7="SYS_INIT", b6="LOL_B", b5="LOL_A", .. :
#define Si5351_MSK_DEVICE_STATUS_SYS_INIT  (1<<7) // "System Initialization Status"
#define Si5351_MSK_DEVICE_STATUS_LOL_B     (1<<6) // "PLLB Loss Of Lock Status"
#define Si5351_MSK_DEVICE_STATUS_LOL_A     (1<<5) // "PLLA Loss Of Lock Status"
#define Si5351_MSK_DEVICE_STATUS_LOS_CLKIN (1<<4) // "CLKIN Loss Of Signal (Si5351C Only)"
#define Si5351_MSK_DEVICE_STATUS_LOS_XTAL  (1<<3) // "Crystal Loss of Signal"
#define Si5351_MSK_DEVICE_STATUS_REVISION  (3<<0) // bits 1..0 : "Revision number of the device"

#define Si5351_REG_INT_STATUS_STICKY 1 // "Interrupt Status Sticky" (what a name..)
#define Si5351_MSK_INT_STATUS_STICKY_SYS_INIT (1<<7) // "System Calibration Status Sticky Bit."
#define Si5351_MSK_INT_STATUS_STICKY_LOL_B    (1<<6) // "PLLB Loss Of Lock Status Sticky Bit."
#define Si5351_MSK_INT_STATUS_STICKY_LOL_A    (1<<5) // "PLLA Loss Of Lock Status Sticky Bit."
#define Si5351_MSK_INT_STATUS_STICKY_LOS_CLKIN (1<<4) // "CLKIN Loss Of Signal Sticky Bit (Si5351C)."
#define Si5351_MSK_INT_STATUS_STICKY_LOS_XTAL (1<<3) // "Crystal Loss of Signal Sticky Bit."

#define Si5351_REG_INT_STATUS_MASK 2 // "Interrupt Status Mask" (not very useful in an Si5351A..)
#define Si5351_MSK_INT_STATUS_MASK_SYS_INIT (1<<7) // "System Initialization Status Mask."
#define Si5351_MSK_INT_STATUS_MASK_LOL_B    (1<<6) // "PLLB Loss Of Lock Status Mask."
#define Si5351_MSK_INT_STATUS_MASK_LOL_A    (1<<5) // "PLLA Loss Of Lock Status Mask."
#define Si5351_MSK_INT_STATUS_MASK_LOS_CLKIN (1<<4) // "CLKIN Loss Of Signal Mask (Si5351C)."
#define Si5351_MSK_INT_STATUS_MASK_LOS_XTAL (1<<3) // "Crystal Loss of Signal Mask."

#define Si5351_REG_OUTPUT_DISABLE 3 // "Output Enable Control" [AN619 page 18]. CLEARED bits are ENABLED !
#define Si5351_MSK_OUTPUT_DISABLE_CLK0 (1<<0) // "Output DISABLE for CLK0" (0=enable=default)
#define Si5351_MSK_OUTPUT_DISABLE_CLK1 (1<<1) // "Output DISABLE for CLK1" (0=enable=default)
#define Si5351_MSK_OUTPUT_DISABLE_CLK2 (1<<2) // "Output DISABLE for CLK2" (0=enable=default)
#define Si5351_MSK_OUTPUT_DISABLE_CLK3 (1<<3) // "Output DISABLE for CLK3" (0=enable=default)
#define Si5351_MSK_OUTPUT_DISABLE_CLK4 (1<<4) // "Output DISABLE for CLK4" (0=enable=default)
#define Si5351_MSK_OUTPUT_DISABLE_CLK5 (1<<5) // "Output DISABLE for CLK5" (0=enable=default)
#define Si5351_MSK_OUTPUT_DISABLE_CLK6 (1<<6) // "Output DISABLE for CLK6" (0=enable=default)
#define Si5351_MSK_OUTPUT_DISABLE_CLK7 (1<<7) // "Output DISABLE for CLK7" (0=enable=default)

#define Si5351_REG_OEB_PIN_ENABLE 9 // "OEB Pin Enable Control Mask" [AN619 page 18]. CLEARED bits are ENABLED !
#define Si5351_MSK_OEB_PIN_ENABLE_CLK1 (1<<1) // "OEB pin enable control of CLK1" (0=OEN pin controls enable/disable of CLK0 output=default)
#define Si5351_MSK_OEB_PIN_ENABLE_CLK0 (1<<0) // "OEB pin enable control of CLK0" 

#define Si5351_REG_PLL_INPUT_SOURCE 15 // "Register 15. PLL Input Source" [AN619 page 19]:
#define Si5351_MSK_PLL_INPUT_SOURCE_CLKIN_DIV (3<<6) // bits 7..6: "ClKIN Input Divider."
#define Si5351_VAL_PLL_INPUT_SOURCE_CLKIN_DIV_BY_1 (0<<6) // "ClKIN Input Divide by 1"
#define Si5351_VAL_PLL_INPUT_SOURCE_CLKIN_DIV_BY_2 (0<<6) // "ClKIN Input Divide by 2"
#define Si5351_VAL_PLL_INPUT_SOURCE_CLKIN_DIV_BY_4 (0<<6) // "ClKIN Input Divide by 4"
#define Si5351_VAL_PLL_INPUT_SOURCE_CLKIN_DIV_BY_8 (0<<6) // "ClKIN Input Divide by 8"
#define Si5351_MSK_PLL_INPUT_SOURCE_PLLB_SRC (1<<3) // bit 3 = "Input Source Select for PLLB" : 0 = "XTAL" (nothing else on Si5351A)
#define Si5351_MSK_PLL_INPUT_SOURCE_PLLA_SRC (1<<2) // bit 2 = "Input Source Select for PLLA" : 0 = "XTAL" (nothing else on Si5351A)

#define Si5351_REG_CLK0_CTRL 16 // "Register 16. CLK0 Control" [AN619 page 20]:
#define Si5351_MSK_CLK_CTRL_PDN    (1<<7) // "Clock x Power Down" : 0=powered up, 1=powered down
#define Si5351_MSK_CLK_CTRL_MS_INT (1<<6) // "MultiSynth x Integer Mode" : 1="force Integer mode to improve jitter performance"
#define Si5351_MSK_CLK_CTRL_MS_SRC (1<<5) // "MultiSynth Source Select for CLK0": 0=PLLA, 1=PLLB
#define Si5351_VAL_CLK_CTRL_MS_SRC_PLLA (0<<5) // "MultiSynth Source Select for CLK0": 0=PLLA
#define Si5351_VAL_CLK_CTRL_MS_SRC_PLLB (1<<5) // "MultiSynth Source Select for CLK0": 1=PLLB
#define Si5351_MSK_CLK_CTRL_INV    (1<<4) // "Output Clock 0/1/2/..7 Invert"
#define Si5351_MSK_CLK_CTRL_SRC    (3<<2) // "Output Clock Input Source" (buah..but wait:)
#define Si5351_VAL_CLK_CTRL_SRC_SYNTH (3<<2) // "11: Select MultiSynth as the source for CLK0"
#define Si5351_VAL_CLK_CTRL_SRC_CLKIN (1<<2) // "01: Select CLKIN as the clock source for CLK0"
#define Si5351_VAL_CLK_CTRL_SRC_XTAL  (0<<2) // "00: Select the XTAL as the clock source for CLK0"
  // > This option by-passes both synthesis stages (PLL/VCXO 
  // > & MultiSynth) and connects CLK0 directly to the oscillator 
  // > which generates an output frequency determined by the XTAL frequency.
  // (remember this; it may be helpful for troubleshooting JITTER / PHASE NOISE)
#define Si5351_MSK_CLK_CTRL_IDRV     (3<<0) // "CLK0 Output Rise and Fall time / Drive Strength Control."
#define Si5351_VAL_CLK_CTRL_IDRV_2mA (0<<0) // "Drive Strength : 2 mA"
#define Si5351_VAL_CLK_CTRL_IDRV_4mA (1<<0) // "Drive Strength : 4 mA"
#define Si5351_VAL_CLK_CTRL_IDRV_6mA (2<<0) // "Drive Strength : 6 mA"
#define Si5351_VAL_CLK_CTRL_IDRV_8mA (3<<0) // "Drive Strength : 8 mA"
  // 
#define Si5351_REG_CLK1_CTRL 17 // "Register 17. CLK1 Control" [AN619 page 21]:
  // On closer inspection, all bits in THIS register (17)
  // seemed to be compatible with the former Si5351_MSK/VAL_CLK0_-defines above,
  // so don't brainlessly copy (duplicate) them all. The same applies 
  // to other 'CLK'-outputs (#2 .. #7) of the bigger brother chips.
  // BUT: Even on the tiny "10-MSOP"-device (Si5153A), we need ALL THESE,
  //      because (from AN619 page 27 about "Clock x>2 Power Down"):
  // > Write this bit to a 1 when programming a Si5351A-B-GT 10-MSOP Device. 
  // (i.e. for all "CLK n CTRL" registers of NON-IMPLEMENTED
  //  channels, those POWER-DOWN FLAGS must be SET,
  //  because their 'reset values' are all ZERO = POWERED UP ! )
#define Si5351_REG_CLK2_CTRL 18 // "Register 18. CLK2 Control" [AN619 page 27]
#define Si5351_REG_CLK3_CTRL 19 // "Register 19. CLK3 Control" ..
#define Si5351_REG_CLK4_CTRL 20 // "Register 20. CLK4 Control"
#define Si5351_REG_CLK5_CTRL 21 // "Register 21. CLK5 Control" 
#define Si5351_REG_CLK6_CTRL 22 // "Register 22. CLK6 Control" 
#define Si5351_REG_CLK7_CTRL 23 // "Register 23. CLK7 Control" 

#define Si5351_REG_CLK3_0_DISABLE_STATE 24 // "Register 24. CLK3-0 Disable State". AN619 page 28..
#define Si5351_REG_CLK7_4_DISABLE_STATE 25 // "Register 25. CLK7-4 Disable State".
               // Each of these registers has TWO bits per "CLK"-channel:
#define Si5351_VAL_CLK_DISABLE_STATE_LOW   0 // CLKx is set to a LOW state when disabled.
#define Si5351_VAL_CLK_DISABLE_STATE_HIGH  1 // CLKx is set to a HIGH state when disabled.
#define Si5351_VAL_CLK_DISABLE_STATE_HI_Z  2 // CLKx is set to a HIGH IMPEDANCE state when disabled.
#define Si5351_VAL_CLK_DISABLE_STATE_NEVER 3 // CLKx is NEVER DISABLED.

#define Si5351_REG_MSYNTH_PLL_A   26 // BASE address of registers for "PLL A FEEDBACK Multisynth Parameters" 
#define Si5351_OFS_MSYNTH_PLL_P3_M  0 // e.g. "Register 26. Multisynth NA Parameter 3" (bits 15..8 of 20-bit denominator of PLLA Feedback Multisynth Divider)
#define Si5351_OFS_MSYNTH_PLL_P3_L  1 // e.g. "Register 27. Multisynth NA Parameter 3" (bits 7..0 of 20-bit denominator of PLLA Feedback Multisynth Divider)
#define Si5351_OFS_MSYNTH_PLL_P1_H  2 // e.g. "Register 28. Multisynth NA Parameter 1" (bits 17..16 of 18-bit integer part of PLLA Feedback Multisynth Divider)
#define Si5351_OFS_MSYNTH_PLL_P1_M  3 // e.g. "Register 29. Multisynth NA Parameter 1" (bits 15..8 of 18-bit integer part of PLLA Feedback Multisynth Divider)
#define Si5351_OFS_MSYNTH_PLL_P1_L  4 // e.g. "Register 30. Multisynth NA Parameter 1" (bits 7..0 of 18-bit integer part of PLLA Feedback Multisynth Divider)
#define Si5351_OFS_MSYNTH_PLL_P32_H 5 // e.g.  "Register 31. Multisynth NA Parameter 3 and 2" :
                                      // (b7..4 = bits 19..16 of 20-bit DENOMINATOR of PLLB Feedback Multisynth Divider,
                                      //  b3..0 = bits 19..16 of 20-bit NUMERATOR of PLLB Feedback Multisynth Divider.)
#define Si5351_OFS_MSYNTH_PLL_P2_M  6 // e.g. "Register 32. Multisynth NA Parameter 2" (bits 15..8 of 20-bit numerator of PLLA Feedback Multisynth Divider)
#define Si5351_OFS_MSYNTH_PLL_P2_L  7 // e.g. "Register 33. Multisynth NA Parameter 2" (bits 7..0 of 20-bit numerator of PLLA Feedback Multisynth Divider)
        // Si5351_REG_MSYNTH_PLL_A(26) + Si5351_OFS_MSYNTH_PLL_P2_L(7) = 33 . NEXT register:
#define Si5351_REG_MSYNTH_PLL_B   34 // BASE address of registers for "PLL B FEEDBACK Multisynth Parameters",
                                     // with the same REGISTER OFFSETS as above for "PLL A".

#define Si5351_REG_OUTPUT_MSYNTH0 42 // BASE address of 8 registers for "Multisynth0" (output divider from f_vco to f_out)
#define Si5351_REG_OUTPUT_MSYNTH1 50 // BASE address of 8 registers for "Multisynth1" (output divider from f_vco to f_out)
#define Si5351_REG_OUTPUT_MSYNTH2 58 // BASE address of 8 registers for "Multisynth2" (output divider from f_vco to f_out)
#define Si5351_REG_OUTPUT_MSYNTH3 66 // BASE address of 8 registers for "Multisynth3" (output divider from f_vco to f_out)
#define Si5351_REG_OUTPUT_MSYNTH4 74 // BASE address of 8 registers for "Multisynth4" (output divider from f_vco to f_out)
#define Si5351_REG_OUTPUT_MSYNTH5 82 // BASE address of 8 registers for "Multisynth5" (output divider from f_vco to f_out)
  // Each of the ABOVE "OUTPUT Multisynths" occupies EIGHT registers with the following ADDRESS OFFSETS:
#define Si5351_OFS_OUTPUT_MSYNTH_P3_M 0  // e.g. "Register 42. Multisynth0 Parameter 3 mid" (bits 15..8 of 20-bit denominator for the fractional part of an output MultiSynth divider)
#define Si5351_OFS_OUTPUT_MSYNTH_P3_L 1  // e.g. "Register 43. Multisynth0 Parameter 3 low" (bits 7..0 of 20-bit denominator for the fractional part of an output MultiSynth divider)
#define Si5351_OFS_OUTPUT_MSYNTH_P1H_DIV 2  // e.g. "Register 44. Multisynth0 Parameters" :
#define Si5351_MSK_OUTPUT_MSYNTH_P1H_DIV_R0  (7<<4) // bits 6..4 : R0 Output Divider (div-by-1,2,4,8,16,32,64,128)
#define Si5351_MSK_OUTPUT_MSYNTH_P1H_DIV_BY4 (3<<2) // bits 3..2 : "MS0 Divide by 4 Enable". 11=div by 4,  00=div by anything else
#define Si5351_MSK_OUTPUT_MSYNTH_P1H_DIV_P1H (3<<0) // bits 1..0 : bits 17..16 of 18-bit integer part of an output MultiSynth divider
#define Si5351_OFS_OUTPUT_MSYNTH_P1_M 3  // e.g. "Register 45. Multisynth0 Parameter 1", middle part (bits 15..8 of 18-bit integer part of an output MultiSynth divider)
#define Si5351_OFS_OUTPUT_MSYNTH_P1_L 4  // e.g. "Register 46. Multisynth0 Parameter 1", lower part (bits 7..0 of 18-bit integer part of an output MultiSynth divider)
        // (Note the lots of copy-and-paste errors in AN619 Rev. 0.8, not only on page 36)
#define Si5351_OFS_OUTPUT_MSYNTH_P32_H 5  // e.g. "Register 47. Multisynth0 Parameter 3 and 2, upper bits" :
#define Si5351_MSK_OUTPUT_MSYNTH_P32_H_P3 (15<<4) // b7..4 = bits 19..16 of 20-bit DENOMINATOR of Multisynth0 Divider
#define Si5351_MSK_OUTPUT_MSYNTH_P32_H_P2 (15<<0) // b3..0 = bits 19..16 of 20-bit NUMERATOR of Multisynth0 Divider
#define Si5351_OFS_OUTPUT_MSYNTH_P2_M 6  // e.g. "Register 48. Multisynth0 Parameter 2 mid" (bits 15..8 of 20-bit numerator of an output Multisynth Divider)
#define Si5351_OFS_OUTPUT_MSYNTH_P2_L 7  // e.g. "Register 49. Multisynth0 Parameter 2 low" (bits 7..0 of 20-bit numerator of an output Multisynth Divider)
  // Note for FREQUENCY MODULATION via software, using a fractional OUTPUT DIVIDER:
  //   the 16 lower bits of the NUMERATOR (e.g. reg 48,49) and the DENOMINATOR (reg 45,46)
  //   are too widely spaced to reprogram them several thousand times per second.
  //   Read a long story with musing about 'fast' frequency modulation in
  //   C:\Elektronik_Projekte\Si5351_Clock_Generator\Documentation\Si5351_Frequency_Modulation.txt . 
  // The next two "OUTPUT Multisynths" (#6 and #7) don't have fractional dividers:
#define Si5351_REG_OUTPUT_MSYNTH6 90 // Address of ONLY ONE REGISTER for "Multisynth6" :
        // > This 8-bit number is the Multisynth6 divide ratio. 
        // > Multisynth6 divide ratio can only be even integers 
        // > greater than or equal to 6. All other divide values are invalid.
#define Si5351_REG_OUTPUT_MSYNTH7 91 // Address of ONLY ONE REGISTER for "Multisynth7" (same limitations as for "Multisynth6")
#define Si5351_REG_OUTPUT_CLK6_7_DIV 92 // "Register 92. Clock 6 and 7 Output Divider"
        // Note the very large gap between register #92 and #149.
#define Si5351_REG_SPREAD_SPECTRUM 149 // "Register 149. Spread Spectrum Parameters"
        // Since we don't want to "spread our spectrum", ignore the rest.
        // There are already too many "spread spectrum jammers" out there,
        // inspread of properly filtering their electronic gadgets.
        // Someone wrote that certain batches of the Si5351 have Spread Spectrum
        // enabled by default, so this beast is disabled further below.
#define Si5351_REG_VCXO_PARAMS 162 // "Register 162. VCXO Parameter".
        // Unfortunately the widely available Si5351A doesn't support
        // a true (analog) voltage controlled oscillator. If it had,
        // the repeater's FREQUENCY MODULATION would be much easier to implement.
#define Si5351_REG_CLK0_PHASE_OFFSET 165 // "Register 165. CLK0 Initial Phase Offset".
        // If you think you need this, continue reading in AN619 page 10 and 58: 
        // > Outputs 0-5 of the Si5351 can be programmed with an independent 
        // > *initial* phase offset. The phase offset only works when MS0-5 
        // > are set as fractional dividers (divider values greater than 8). 
        // > The phase offset parameter is an unsigned integer where each LSB 
        // > represents a phase difference of a quarter of the VCO period, 
        // > T VCO /4. Use the equation below to determine the register value. 
        // > Also, remember that any divider using the phase offset feature
        // > needs the MSx_INT bit set to 0 .
        // > 
        // > CLKx_PHOFF = Round( DesiredOffset[sec] * 4 * Fvco )
        // > 
        // > CLKx_PHOFF[6:0] is an unsigned integer with one LSB equivalent 
        // > to a time delay of Tvco/4, where Tvco is the period 
        // > of the VCO/PLL associated with this output.
        // Ok, so there is a SEVEN-bit "offset". What's the phase-adjustment-
        // range in seconds .. is it enough for frequency modulation ?
        // With Fvco = 900 MHz : t_phase = 0..127 / (4*Fvco) = 0 .. 35 ns.
#define Si5351_REG_CLK1_PHASE_OFFSET 166
#define Si5351_REG_CLK2_PHASE_OFFSET 167
#define Si5351_REG_CLK3_PHASE_OFFSET 168
#define Si5351_REG_CLK4_PHASE_OFFSET 169
#define Si5351_REG_CLK5_PHASE_OFFSET 170
        // Outputs "CLK6" and "CLK7" have no phase adjustment because they
        // don't have fractional output dividers - only simple INTEGER dividers.

#define Si5351_REG_PLL_RESET 177 // "Register 177. PLL Reset" :
#define Si5351_MSK_PLL_RESET_PLLB (1<<7) // "Writing a 1 to this bit will reset PLLB. This is a self clearing bit."
#define Si5351_MSK_PLL_RESET_PLLA (1<<5) // "Writing a 1 to this bit will reset PLLA. This is a self clearing bit."

#define Si5351_REG_CRYSTAL_LOAD 183 // "Register 183. Crystal Internal Load Capacitance". AN619 page 61.
#define Si5351_VAL_CRYSTAL_LOAD_6pF  (1<<6)  // internal CL = 6 pF
#define Si5351_VAL_CRYSTAL_LOAD_8pF  (2<<6)  // internal CL = 8 pF
#define Si5351_VAL_CRYSTAL_LOAD_10pF (3<<6)  // internal CL = 10 pF
#define Si5351_VAL_CRYSTAL_LOAD_RSVD_BITS 0b010010 // "Bits 5:0 should be written to 010010b"
        
#define Si5351_REG_FANOUT 187 // "Register 187. Fanout Enable"
#define Si5351_VAL_FANOUT_CLKIN (1<<7) // "Enable fanout of CLKIN to clock output multiplexers. Set this bit to 1b."
#define Si5351_VAL_FANOUT_XO    (1<<6) // "Enable fanout of XO to clock output multiplexers. Set this bit to 1b."
#define Si5351_VAL_FANOUT_MS    (1<<4) // "Enable fanout of Multisynth0 and Multisynth4 to all output multiplexers. Set this bit to 1b." 

#define Si5351_GET_REG_FROM_ADDRESS 0xFF // dummy-register-number explained in Si5351_WriteMultipleRegisters()

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
// Global (or at least static) variables :
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 

BYTE Si5351_bTotalNumErrors; // summed-up Si5351-related errors counted since init (maxes out near 255)
BYTE Si5351_bLastErrorCode;  // Si5351_ERROR_NONE, Si5351_ERROR_VERIFY, Si5351_ERROR_PLL_UNLOCKED, ...
BYTE Si5351_bLastErrorParam; // 'additional info', meaning depends on Si5351_bLastErrorCode, e.g. REGISTER NUMBER


float Si5351_fltPrevVcoFreq[2]; // Previous internal VCO frequencies for "PLLA" and "PLLB".
   // Used in Si5351_SetFrequency() to decide if we need a "PLL Reset" or not 
   // - if that's what the bits in "Register 177. PLL Reset" REALLY DO !
   //     See notes by G0UPL - the "PLL Reset" is most important to maintain
   //     a KNOWN PHASE RELATION between *OUTPUT MULTISYNTH STAGES* driven
   //     by the same PLL ("PLLA" or "PLLB").  

   // Structure with a 'copy' of ALL *current register values* in RAM;
   //     inspectable via table-driven menu in the main application : 
T_Si5351_Regs Si5351Regs;  // circa 198 BYTE-registers in a *UNION*

#if( SWI_SI5351_ALLOW_POKING_REGISTERS_FROM_MENU )
 BOOL Si5351_fPokingAround; // flag controlled by Si5351_RWAccessForMenuItems(),
        // TRUE when "poking around" in certain registers like P1,P2,P3.
        // While "poking around" (editing via 'Synth Registers' menu), the
        // FREQUENCY MODULATOR and other periodic register-writers are disabled. 
#endif // SWI_SI5351_ALLOW_POKING_REGISTERS_FROM_MENU ?

#if( SWI_SI5351_SUPPORT_FREQUENCY_MODULATION )
  BYTE  Si5351_bFreqModulationMethod[SI5351_NUM_ACTIVE_OUTPUT_CHANNELS]; // ... one of the following:
#       define FM_METHOD_OFF          0 // no frequency modulation at all
#       define FM_METHOD_PLL_FEEDBACK 1 // modulate the Si5351's *PLL FEEDBACK* divider
#       define FM_METHOD_OUTPUT_DIV   2 // modulate the Si5351's *FRACTIONAL OUTPUT* divider 
               // (unfortunately the FRACTIONAL output divider cannot be used above 112.5 MHz "out")  
  WORD  Si5351_wFreqModulationOffset[SI5351_NUM_ACTIVE_OUTPUT_CHANNELS]; // 16-bit offset added to each 12-bit modulation sample  
  DWORD Si5351_FM_Carrier_dwPLLFeedbackIntegerPart;
  DWORD Si5351_FM_Carrier_dwPLLFeedbackNumerator;
  DWORD Si5351_FM_Carrier_dwPLLFeedbackDenominator;
# if( SWI_SI5351_SEND_FM_STREAM_IN_I2C_COMPLETION_CALLBACK ) 
  BYTE  Si5351_bFMStreamState;   // may contain one of the following:
        // FM_STREAM_OFF   : stream of FM samples to the Si5351 permanently 'off'
        // FM_STREAM_ON    : stream of FM samples is 'running'
        // FM_STREAM_ERROR : sample stream broken, e.g. due to I2C bus error.
# endif // SWI_SI5351_SEND_FM_STREAM_IN_I2C_COMPLETION_CALLBACK ?
#endif // SWI_SI5351_SUPPORT_FREQUENCY_MODULATION ?


// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
// Internal function prototypes
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
  
BOOL Si5351_ReadRegister_8Bit( BYTE bRegAddress, BYTE *pbData);
BOOL Si5351_ReadMultipleRegisters(BYTE bRegAddress, BYTE *pbData, BYTE nBytes);
BOOL Si5351_WriteRegister_8Bit(BYTE bRegAddress, BYTE bData);
BOOL Si5351_WriteMultipleRegisters( BYTE bRegAddress, BYTE *pbData, BYTE nBytes);
BOOL Si5351_SetPLLFeedbackRegisters(BYTE bPLLBaseReg, DWORD dwA, DWORD dwB, DWORD dwC, T_Si5351_FeedbackDividerRegs *pRegs);
BOOL Si5351_SetPLLFeedbackRegsP1P2P3(BYTE bPLLBaseReg,T_Si5351_FractionalDividerParamsP1P2P3 *pPLLDivider,T_Si5351_FeedbackDividerRegs *pRegs);
BOOL Si5351_SetOutputDividerRegisters(BYTE bOutputBaseReg,T_Si5351_FractionalDividerParams *pParams,T_Si5351_OutputDividerRegs *pRegs);
BOOL Si5351_SetOutputDividerRegsP1P2P3(BYTE bOutputBaseReg,T_Si5351_FractionalDividerParamsP1P2P3 *pOutDivider,T_Si5351_OutputDividerRegs *pRegs);
void Si5351_CalcVcoFreqFromP1P2P3(T_Si5351_FractionalDividerParamsP1P2P3 *p1p2p3, double *pdbVcoFreq_Hz );

#if( SWI_SI5351_READ_BACK_REGISTER ) // READ BACK and VERIFY registers via I2C ?
 BOOL Si5351_VerifyRegisters(BYTE bRegAddress, BYTE *pbData, BYTE nBytes);
#endif
#if( SWI_SI5351_SUPPORT_FREQUENCY_MODULATION ) 
 BOOL Si5351_GetPLLAndFractionalOutputDividerParamsForFM( int iOutputIndex, // preferred but unusable above 112 MHz
          double dblWantedFrequency_Hz, int iFreqModDeviation_Hz,
          T_Si5351_FractionalDividerParamsP1P2P3 *pPLLDivider,  // [out] integer divider params FOR THE PLL (VCO "A" or "B")
          T_Si5351_FractionalDividerParamsP1P2P3 *pOutDivider); // [out] fractional divider params FOR THE OUTPUT ("CLK0", "CLK1", "CLK2")
 BOOL Si5351_GetPLLFractionalFeedbackDividerAndIntegerOutputParamsForFM( int iOutputIndex, // MUST be used above 112 MHz (works up to 200 MHz)
          double dblWantedFrequency_Hz, int iFreqModDeviation_Hz,
          T_Si5351_FractionalDividerParamsP1P2P3 *pPLLDivider,  // [out] fractional divider params FOR THE PLL (VCO "A" or "B")
          T_Si5351_FractionalDividerParamsP1P2P3 *pOutDivider); // [out] integer divider params FOR THE OUTPUT ("CLK0", "CLK1", "CLK2")
#endif  // support "FM via software" ?

     // Callback functions for I2C-transfer-completion :
void Si5351_OnWriteMultipleRegistersComplete(T_I2C_TransferInstance *pXfer, int iResult ); 
void Si5351_OnReadMultipleRegistersComplete(T_I2C_TransferInstance *pXfer, int iResult ); 
void Si5351_OnSendFMSampleComplete(T_I2C_TransferInstance *pXfer, int iResult ); 


// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
// Implementation of functions
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 

//---------------------------------------------------------------------------
BOOL Si5351_Init(void) // returns TRUE if an Si5351 was detected on I2C .
                       // Does *not* turn on any output !
{
  BOOL fOk, fResult;
  BYTE bRegValue;
  int  i = 3; 
  
  Si5351_ClearErrors(); // -> Si5351_bTotalNumErrors=0, etc (?)

#if( SWI_SI5351_SEND_FM_STREAM_IN_I2C_COMPLETION_CALLBACK ) 
  Si5351_bFMStreamState = FM_STREAM_OFF; // stream of FM samples to the Si5351 still 'off'
#endif // SWI_SI5351_SEND_FM_STREAM_IN_I2C_COMPLETION_CALLBACK ?
  
  // Force at least ONE "PLL Reset" for "PLLA" [0] and "PLLB" [1] in 
  Si5351_fltPrevVcoFreq[0] = Si5351_fltPrevVcoFreq[1] = 0.0f;
  
  // If the I2C bus fails to co-operate, try up to THREE times;
  for( i=0; i<3; ++i )
   {
     // Wait until the Si5351 has finished its internal 'system initialisation' ?
     // From AN619 page 14, "Register 0. Device Status", bit 7 = "SYS_INIT" :
     // > During power up the device copies the content of the NVM into RAM
     // > and performs a system initialization. The device is not operational 
     // > until initialization is complete. It is not recommended to read or write 
     // > registers in RAM through the I2C interface until initialization 
     // > is complete.
     // The DATASHEET (page 9) promises a power-up time of 10 ms or less.. anyway:
     fOk = Si5351_ReadRegister_8Bit( Si5351_REG_DEVICE_STATUS, &bRegValue );
     if( bRegValue & Si5351_MSK_DEVICE_STATUS_SYS_INIT ) // "SYS_INIT" not finished ?!
      { fOk = FALSE;  // eat up ONE of the three repetition loops
      }
     // When trying this with the Banggood-clones or Adafruit's Si5351A-board,
     // got here with fOk=TRUE and bRegValue=0x11 = "LOS_CLKIN (Si5351C Only)"
     //                                           + "REVID 0b01" ....
     // With bit 3 set in the 'DEVICE STATUS' register, the chip indicates
     //     "Crystal Loss of Signal" (AN619 Rev 0.8 page 15) - a bad thing :
     if( bRegValue & Si5351_MSK_DEVICE_STATUS_LOS_XTAL ) // "Crystal Loss of Signal" ?
      { // > A loss of signal status indicates that the crystal failed 
        // > to meet the minimum requirements of a valid input signal 
        // > as specified in the Si5351 datasheet.
        fOk = FALSE;  // eat up ONE of the three repetition loops
      }
     if( fOk )  // no need to try again..
      { break;
      }
     IOP_WaitMilliseconds( 10 ); // give the Si5351 some time for "Sys Init" and detect "Crystal Signal" ?
   } // end for < some repetions to wait until 'Device Status' is HEALTHY >

  fResult = fOk;  // final return value only TRUE=ok when ALL STEPS were successful..
  
  // At this point, the Si5351 should be 'accessable' via I2C bus.
  // For testing/debugging purposes, read ALL KNOWN REGISTERS,
  // and (if we can afford the code memory size) immediately VERIFY them.
  // Like most 'vital' steps, this is repeated up to three times
  // because if anything goes wrong, we may 'transmit' on a wrong frequency.
  for( i=0; i<3; ++i ) // Three more repetitions to READ ALL initial register values..
   {
     SL_FillMem( Si5351Regs.bArray, sizeof(Si5351Regs), 0xDB/*Dead Beef*/ );
     fOk = Si5351_ReadMultipleRegisters( Si5351_GET_REG_FROM_ADDRESS, Si5351Regs.bArray, SI5351_NUM_REGISTERS );
     // Seen at this point (in MPLAB's "Watch"-window, enter 'Si5351Regs') : 
     //   Si5351Regs.DeviceStatus          = 0x11
     //   Si5351Regs.InterruptStatusSticky = 0xF9, sometimes 0xF8 ("cold start"?)
     //   Si5351Regs.InterruptStatusMask   = 0x03
     //   Si5351Regs.OutputEnableControl   = 0xFC, sometimes 0x00 
     //   Si5351Regs.CrystalLoadCapacitance= 0b11010010
     //               (bits 7:6 set for "10 pF") |____|
     //    ,----------------------------------------'
     //    '--> ok, "Bits 5:0 should be written to 010010b" (AN619 page 61)
     //    ...
     //   Si5351Regs.FanoutEnable          = 0x02 (last DOCUMENTED register).
     // (-> see screenshot Si5351_Default_Register_Values_inspected_via_MPLAB.png )
     //   Added the following, optional test to check if this was 'real' :
#   if( SWI_SI5351_READ_BACK_REGISTER ) // READ BACK and VERIFY registers via I2C ?
     fOk &= Si5351_VerifyRegisters( Si5351_GET_REG_FROM_ADDRESS, Si5351Regs.bArray, SI5351_NUM_REGISTERS );
             //    '--> increments Si5351_bTotalNumErrors on 'verify errors' .
     // 2021-11-14: fOk = TRUE at this point. Confirms that all register values 
     //             read from the Si5351 are 'real', and that there are some 
     //             secrets hidden in the 'Reserved' registers, e.g.:
     //  Si5351Regs.bArray[  4..  8] ("Reserved") : values 0x00, 0xFF, 0x00, 0x00, 0x90
     //  Si5351Regs.bArray[ 10.. 15] ("Reserved") : values 0x00, 0x00, 0x00, 0x00, 0x00
     //  Si5351Regs.bArray[171..176] ("Reserved") : values 0x00, ... , 0x00, 0xFF  
     //  Si5351Regs.bArray[178..182] ("Reserved") : values 0x00, 0x00, 0x00, 0x30, 0x1D  
     //  Si5351Regs.bArray[184..186] ("Reserved") : values 0x60, 0x60, 0xB8
#   endif // SWI_SI5351_READ_BACK_REGISTER ?

     if( fOk )  // no need to try again..
      { break;
      }
     IOP_WaitMilliseconds( 10 ); // give the Si5351 some time to breathe, then try again
   } // end for < some repetions to wait until 'Device Status' is HEALTHY >
     
  fResult &= fOk;  // final return value only TRUE=ok if the above step was ok
  
  for( i=0; i<3; ++i ) // Three more repetitions to init the most vital registers..
   {
     // Only initialize a few registers that will not be configured
     //  in Si5351_SetFrequency() -> Si5351_SetPLLFeedbackRegisters(),
     //                              Si5351_SetOutputDividerRegisters():
     fOk = Si5351_WriteRegister_8Bit( Si5351_REG_PLL_INPUT_SOURCE, // AN619 page 19..
                 Si5351_VAL_PLL_INPUT_SOURCE_CLKIN_DIV_BY_1 // "ClKIN Input Divide by 1"
          | (0 * Si5351_MSK_PLL_INPUT_SOURCE_PLLB_SRC )   // "Input Source Select for PLLB" : 0 = "XTAL" (nothing else on Si5351A)
          | (0 * Si5351_MSK_PLL_INPUT_SOURCE_PLLA_SRC )); // "Input Source Select for PLLA" : 0 = "XTAL" (nothing else on Si5351A)

     fOk &= Si5351_WriteRegister_8Bit( Si5351_REG_OUTPUT_DISABLE, // AN619 page 18..
            ( 0 * Si5351_MSK_OUTPUT_DISABLE_CLK0 ) // "Output DISABLE for CLK0" (0=enable=default) ?
          | ( 0 * Si5351_MSK_OUTPUT_DISABLE_CLK1 ) // "Output DISABLE for CLK1" (0=enable=default)
          | ( 1 * Si5351_MSK_OUTPUT_DISABLE_CLK2 ) // "Output DISABLE for CLK2" (1=disable, output not USED here)
          | ( 1 * Si5351_MSK_OUTPUT_DISABLE_CLK3 ) // "Output DISABLE for CLK3" (1=DISABLE non-existing output on Si5351A)
          | ( 1 * Si5351_MSK_OUTPUT_DISABLE_CLK4 ) // "Output DISABLE for CLK4" 
          | ( 1 * Si5351_MSK_OUTPUT_DISABLE_CLK5 ) // "Output DISABLE for CLK5" 
          | ( 1 * Si5351_MSK_OUTPUT_DISABLE_CLK6 ) // "Output DISABLE for CLK6" 
          | ( 1 * Si5351_MSK_OUTPUT_DISABLE_CLK7 ) // "Output DISABLE for CLK7"
                                    ); // <- end of the "Output Enable Control"-value

     fOk &= Si5351_WriteRegister_8Bit( Si5351_REG_OEB_PIN_ENABLE, // AN619 page 18..
                      0xFF );  // H : "OEB pin does NOT enable/disable CLKx"
     
     fOk &= Si5351_WriteRegister_8Bit( Si5351_REG_CRYSTAL_LOAD, // AN619 page 61...
                                      Si5351_VAL_CRYSTAL_LOAD_6pF 
                                    | Si5351_VAL_CRYSTAL_LOAD_RSVD_BITS );
     // Tried various crystal load capacitor settings with the "violet"
     // Banggood-clones and the "blue" original Adafruit Si5351A-boards,
     // and measured the resulting "25 MHz" reference frequencies 
     // with the stock crystals at all three selectable load settings. Result:
     // 
     //  Test sample |   CL = 6 pF  | delta |   CL = 8 pF  | delta |  CL = 10 pF
     //  ------------+--------------+-------+--------------+-------+------------
     //  Banggood #1 | 25001118 Hz  | 959 Hz| 25000159 Hz  | 616 Hz| 24999543 Hz
     //  ------------+--------------+-------+--------------+-------+------------
     //  Banggood #2 |              |       |              |       |          Hz
     //  ------------+--------------+-------+--------------+-------+------------
     //  Adafruit #1 | 25000011 Hz  | 849 Hz| 24999162 Hz  | 536 Hz| 24998626 Hz
     //
     //
     
     fOk &= Si5351_WriteRegister_8Bit( Si5351_REG_SPREAD_SPECTRUM, 0 ); // AN619 page 54
            // (Note: The dreadful Spread Spectrum mode (which mostly exits
            //   to 'beautify' EMI test results without reducing the total 
            //   emitted unwanted power at all) is enabled by an OR(!)-
            //   combination of bit 7 ("Spread Spectrum Enable", cleared above)
            //   and the SSEN-pin ! )
         
     if( fOk )  // no need to try again..
      { break;
      }
   } // end for < try again up to 3 times >

  fResult &= fOk;  // final return value only TRUE=ok if the above step was ok
  
  return fResult;  // TRUE=success, FALSE=error (I2C bus trouble?)
  
} // end Si5351_Init()

//------------------------------------------------------------------------
void Si5351_InitStruct_FractionalDividerParamsP1P2P3(
          T_Si5351_FractionalDividerParamsP1P2P3 *pParamsP1P2P3 ) // [out]
  // This structure can be used for a PLL-feedback-divider
  // as well as for an integer- or fractional OUTPUT divider .
  // Equations (derived and explained in DL4YHF's Si5351_FM_Tests_2021_11.txt):
  //  ----------------------------------------------------------
  //     f_vco = f_ref * ( 512 + P1 + P2/P3 ) / 128     (1.1)
  //     f_out = f_vco * 128 / ( 512 + p1 + p2/p3 )     (1.2)
  //  ----------------------------------------------------------
{
  pParamsP1P2P3->p1 = 0; // 18-bit integer part
  pParamsP1P2P3->p2 = 0; // 20-bit numerator
  pParamsP1P2P3->p3 = 1; // 20-bit denominator. MUST NEVER BE ZERO HERE !
  pParamsP1P2P3->r_div = 1;  // only for OUTPUT DIVIDERS: 1,2,4..128; can be used to generate frequencies below about 500 kHz
} // end Si5351_InitStruct_FractionalDividerParamsP1P2P3()


//------------------------------------------------------------------------
BOOL Si5351_SetFrequency( // A 'core function' of this module ...
        int iOutputIndex, // [in] 0-based index for PLL *and* output divider
        double dblWantedFrequency_Hz, // [in] "wanted" frequency in Hertz
        int iFreqModDeviation_Hz ) // [in] optional headroom for FM,
                                   // 0=no frequency modulation .. 32767 Hz="WFM" .
  // This function tries to hide many of the ugly details of the Si5351
  // register programming from the "application". But it doesn't use all 
  // features of say an Si5351A : 
  //  - The Si5351A has THREE outputs but Si5351_SetFrequency() only uses
  //                    TWO of them, because to keep things simple,
  //                    we use PLL "A" for the FIRST output (iOutputIndex=0),
  //                           PLL "B" for the SECOND output (iOutputIndex=1),
  //     so there's no other PLL to *independently* feed the THIRD output !
  //  - if the VCO frequency step between THIS and the PREVIOUS call
  //    is 'sufficiently small' we get along *without* resetting the PLL. 
  //    See notes by G0UPL !
  // LIMITATIONS: See Si5351A datasheet and 'AN619' : 
  //  * dblWantedFrequency_Hz must be between 2.5 kHz and 200 MHz. 
  //    From AN619 page 2, "Conceptualizing a Frequency Plan" (buaah..) : 
  //    > The device consists of two PLLs - PLLA and PLLB. 
  //    > Each PLL generates an intermediate VCO frequency in the
  //    > range of 600 to 900 MHz using a Feedback Multisynth. 
  //    > These VCO frequencies can then be divided down by
  //    > individual output Multisynth dividers to generate any frequency 
  //    > between 500 kHz and 200 MHz. Additionally, the R dividers
  //    > can be used to generate any output frequency down to 2.5 kHz.
  // (to achive 150 MHz .. 200 MHz with an Si5351A, forget about
  //  FRACTIONAL OUTPUT DIVERS, and forget about PROGRAMMABLE output dividers:
  //  For such 'Very High Frequencies', the OUTPUT DIVIDER stage must be
  //  configured to use a specialized, FIXED "divide-by-FOUR" value).
{
  BOOL  fOk = TRUE;
  DWORD dwApproxVcoFreq, dwVcoOutputDivider, dwA;
  T_Si5351_FractionalDividerParamsP1P2P3 pll_p1p2p3; // parameters for the VCO (PLL feedback divider)
  T_Si5351_FractionalDividerParamsP1P2P3 out_p1p2p3; // parameters for a fractional output divider
  double f_vco, pll_feedback_ratio, fractional_part;
  float  fltTemp;
  int  n_iterations; // number of iterations in the Farey algorithm (for testing)
  long i32Numerator, i32Denominator;
# if( SWI_SI5351_SUPPORT_FREQUENCY_MODULATION )  
   T_Si5351_FractionalDividerParams output_divider_params; // fractional divider parameters (a+b/c and r_div) for the OUTPUT divider
   long i32Factor;
# endif // SWI_SI5351_SUPPORT_FREQUENCY_MODULATION ? 
  BYTE bPLLBaseAddr, bOutDivBaseAddr, bClockCtrlRegAddr, bClockCtrlRegValue, bPLLResetRegValue;
  BOOL pll_params_ready = FALSE;
  
  if( dblWantedFrequency_Hz < 2e3 ) 
   {  dblWantedFrequency_Hz = 2e3; // lower limit with maximum "R divider"
      // (only "slightly" violates the VCO frequency range; details below)
   }
  if( dblWantedFrequency_Hz > 200e6 ) 
   {  dblWantedFrequency_Hz = 200e6; // upper frequency limit for Si5351
   }

  if( g_dblReferenceFrequency_Hz <= 0.0 ) // Init this if the application didn't..
   {  g_dblReferenceFrequency_Hz = (double)SWI_SI5351_REFERENCE_FREQUENCY_HZ;
      // SWI_SI5351_REFERENCE_FREQUENCY_HZ is just a default for this variable.
   }     

  // Set 'output divider parameters' to meaningful default:
  output_divider_params.r_div = 1;
  output_divider_params.a = 8;
  output_divider_params.b = 0;
  output_divider_params.c = 1;
  output_divider_params.f_out = dblWantedFrequency_Hz;

  // Prepare "channel dependent" parameters (register addresses, special bits, etc)
  switch( iOutputIndex )
   { case 0 :  // FIRST output, driven by PLL A, software-frequency-modulatable ...
       bPLLBaseAddr    = Si5351_REG_MSYNTH_PLL_A;   // "PLL A FEEDBACK Parameters" 
       bPLLResetRegValue=Si5351_MSK_PLL_RESET_PLLA; // IF we were to reset the PLL, THIS would be the register value
       bOutDivBaseAddr = Si5351_REG_OUTPUT_MSYNTH0; // "Multisynth0" (output div)
       bClockCtrlRegAddr  = Si5351_REG_CLK0_CTRL;   // "CLK Control" [AN619 page 21]
       bClockCtrlRegValue = Si5351_VAL_CLK_CTRL_MS_SRC_PLLA; // "MultiSynth Source Select for CLK0": 0=PLLA
       dwApproxVcoFreq = 900000000UL;   // 1st internal VCO (the one for "PLLA")
       break;
     case 1 :  // SECOND output, driven by PLL B : 
       bPLLBaseAddr    = Si5351_REG_MSYNTH_PLL_B;   // "PLL B feedback div.."
       bPLLResetRegValue=Si5351_MSK_PLL_RESET_PLLB; // 
       bOutDivBaseAddr = Si5351_REG_OUTPUT_MSYNTH1; // "Multisynth1" (output div)
       bClockCtrlRegAddr  = Si5351_REG_CLK1_CTRL;   // "CLK Control" [AN619 page 21]
       bClockCtrlRegValue = Si5351_VAL_CLK_CTRL_MS_SRC_PLLB; // "MultiSynth Source Select for CLK0": 1=PLLB
#     if(0) // Avoid two VCO frequencies being TOO CLOSE to each other ? 0=no, 1=yes
       dwApproxVcoFreq = 777777777UL;   // 2nd internal VCO (the one for "PLLB") runs near 777 MHz (or LESS)
       // to avoid 'cross modulation' or two VCOs affecting each other (crosstalk?)
#     else  // dont care if two VCO frequencies are only separated by a few Hertz ?
       dwApproxVcoFreq = 900000000UL;   // 2nd internal VCO (the one for "PLLB") runs near 900 MHz, too 
       // Does this make a difference ? Even with the two OUTPUT frequencies
       // separated by only 0.01 Hz, it didn't. Obviously no "cross-talk" 
       // between the two VCOs and their PLL's phase comparators. Great.
#     endif // avoid having the two VCOs (and PLLs) running too close to another ?       
       break;
     default:  // bail out due to invalid "channel number"      
       return FALSE; 
   } 
  output_divider_params.f_in = dwApproxVcoFreq;


# if( SWI_SI5351_SUPPORT_FREQUENCY_MODULATION )    
  Si5351_PauseSendingFM(); // .. and, if necessary, wait until an ongoing I2C bus transaction is finished
         // '--> Si5351_bFMStreamState = FM_STREAM_PAUSED when only 'temporarily' paused !
# endif // SWI_SI5351_SUPPORT_FREQUENCY_MODULATION 
  
  Si5351_bFreqModulationMethod[iOutputIndex] = FM_METHOD_OFF; // guess no frequency modulation at all
#if( SWI_SI5351_SUPPORT_FREQUENCY_MODULATION ) 
  if( iFreqModDeviation_Hz != 0 )
   { // FREQUENCY-MODULATE the output .. how ?
     // First try to generate FREQUENCY MODULATION via the FRACTIONAL OUTPUT DIVIDER,  
     //  hoping this generates less unwanted products than when modulating the PLL
     //  (with an unknown loop bandwidth, and unpredictable 'swing-in' effects 
     //   after large sudden steps in the modulator signal). 
     if( Si5351_GetPLLAndFractionalOutputDividerParamsForFM( // -> TRUE=ok, FALSE='try something else' !
              iOutputIndex, // [in] zero-based index for the PLL *and* the output divider
              dblWantedFrequency_Hz, // [in] "wanted" frequency in Hertz, 64 bit float
              iFreqModDeviation_Hz,  // [in] wanted deviation (+/- N Hz around the carrier)
              &pll_p1p2p3,  // [out] parameters for the VCO (PLL feedback divider, here usually INTEGER)
              &out_p1p2p3)) // [out] parameters for the OUTPUT DIVIDER (here: FRACTIONAL)
      { // SUCCESS : can use a FRACTIONAL output divider to generate FM, so use it:
        Si5351_wFreqModulationOffset[iOutputIndex] = (WORD)out_p1p2p3.p2; // save for later
        out_p1p2p3.p2 += 2047;  // add 'half modulator swing' to generate the center frequency
        Si5351_bFreqModulationMethod[iOutputIndex] = FM_METHOD_OUTPUT_DIV; // modulate the Si5351's *FRACTIONAL OUTPUT* divider 
        
        // Split the PLL' fractional divider parameters (AN619 : "P1".."P3")
        // into various bitgroups for EIGHT adjacent PLL-Feedback-registers 
        // and write them into the Si5351 via I2C : 
        fOk &= Si5351_SetPLLFeedbackRegsP1P2P3( bPLLBaseAddr, &pll_p1p2p3,
               &Si5351Regs.g.FeedbackDivider[iOutputIndex] ); // [out] 
        // Similar for the fractional OUTPUT divider:
        fOk &= Si5351_SetOutputDividerRegsP1P2P3( bOutDivBaseAddr, &out_p1p2p3, 
               &Si5351Regs.g.OutputDivider[iOutputIndex] ); // [out] 
        // Calculate the new internal *VCO* frequency, used further below to decide for a 'PLL reset':
        Si5351_CalcVcoFreqFromP1P2P3( &pll_p1p2p3, &f_vco ); 
        output_divider_params.f_in = f_vco;
        pll_params_ready = TRUE; // bypass the code fragment below to program the PLL (etc) 'A','B','C' !
      }  
     else // cannot use a fractional OUTPUT divider for FM, so try a fractional PLL FEEDBACK divider instead:
     if( Si5351_GetPLLFractionalFeedbackDividerAndIntegerOutputParamsForFM( // -> TRUE=ok, FALSE='try something else' !
              iOutputIndex, // [in] zero-based index for the PLL *and* the output divider
              dblWantedFrequency_Hz, // [in] "wanted" frequency in Hertz, 64 bit float
              iFreqModDeviation_Hz,  // [in] wanted deviation (+/- N Hz around the carrier)
              &pll_p1p2p3,  // [out] parameters for the VCO (PLL feedback) divider
              &out_p1p2p3)) // [out] parameters for the OUTPUT DIVIDER (p1,p2,p3,r_div)
      { // SUCCESS -> frequency-modulate the PLL FEEDBACK (i.e. even the VCO frequency is modulated)
 
        Si5351_wFreqModulationOffset[iOutputIndex] = (WORD)pll_p1p2p3.p2; // save for later
        pll_p1p2p3.p2 += 2047;  // add 'half modulator swing' for the center frequency
        Si5351_bFreqModulationMethod[iOutputIndex] = FM_METHOD_PLL_FEEDBACK; // modulate the Si5351's *PLL FEEDBACK* divider 
        
        // Split the PLL's fractional divider parameters (AN619 : "P1".."P3")
        // into various bitgroups for EIGHT adjacent PLL-Feedback-registers 
        // and write them into the Si5351 via I2C : 
        fOk &= Si5351_SetPLLFeedbackRegsP1P2P3( bPLLBaseAddr, &pll_p1p2p3,
               &Si5351Regs.g.FeedbackDivider[iOutputIndex] ); // [out] 
        // Similar for the fractional OUTPUT divider (here, usually INTEGER in the interest of low phase noise):
        fOk &= Si5351_SetOutputDividerRegsP1P2P3( bOutDivBaseAddr, &out_p1p2p3, 
               &Si5351Regs.g.OutputDivider[iOutputIndex] ); // [out] 
        // And again, the annoying 'special cases' above 112.5 and 150 MHz:
        if( (out_p1p2p3.p2 == 0 ) // no NUMERATOR (no 'fractional' part), and...
         &&((out_p1p2p3.p1 & 0x007F)==0) ) // even INTEGER part ? (note the bizzare format, "p1 = A * 128", thus bit 7 in 'p1' is bit 0 in 'A')
         { bClockCtrlRegValue |= Si5351_MSK_CLK_CTRL_MS_INT; // "MultiSynth x [OUTPUT] Integer Mode" : 1="force Integer mode", HERE: because there's NO FRACTIONAL PART ("p2") .
           // 2022-02-16 : Generating a 'very high frequency' (VHF: 162.29 MHz),
           //              frequency-modulated by the PLL *feedback* divider,
           //              and (as required) "even integer" + "DIVBY4" mode enabled,
           //  the spectrum around 162.29 MHz wasn't spectacularly clean.
           //  Measured with a Rigol MSO5104 using the scope's FFT, 500 MSamples/sec,
           //  Center at 162.5 MHz, Span=5MHz, RBW=500 Hz (!), using the scope's
           //  "Peak Search" feature with Threshold around -45 dBV :
           //     * main peak at 162.29 MHz : -13 dBV (due to software-bandwidth-limit)
           //     * strongest spurs at   161.74 and 162.83 MHz : Both 30 dB below carrier.
           //     * next weaker spurs at 160.66 and 163.92 MHz : Both 42 dB below carrier.
           //   No change after turning the frequency modulation off
           //   (only in the narrow-band spectrum around the carrier of course).
           //   NARROW-BAND spectrum around 162.29 MHz, Span=100 kHz, unmodulated : 
           //     * main peak at 162.29 MHz : still around -13 dBV (at RBW=500 Hz)
           //     * strongest spurs : over 60 dB "below" the carrier !
           //   Frequency modulation BACK ON again, looking at the NARROW-BAND spectrum:
           //     * broader 'peak' as expected, now around -14 dBV (at RBW=500 Hz)
           //     * any 'close-in' spur in the 100 kHz span: still >= 60 dB below the "carrier" !
         }
        // Calculate the new internal *VCO* frequency, used further below to decide for a 'PLL reset':
        Si5351_CalcVcoFreqFromP1P2P3( &pll_p1p2p3, &f_vco ); 
        output_divider_params.f_in = f_vco;
        pll_params_ready = TRUE; // bypass the code fragment below to program the PLL (etc) 'A','B','C' !
      }  
     else
      { // NONE of the frequency modulation methods is applicable ->
        Si5351_bFreqModulationMethod[iOutputIndex] = FM_METHOD_OFF; // don't try to modulate the Si5351 at all     
#   endif // SWI_SI5351_SUPPORT_FREQUENCY_MODULATION ?
      } // end else < Si5351_GetPLLAndFractionalOutputDividerForFM() said 'NO !' >
   }  // end else < iFreqModDeviation_Hz != 0 > 
  else     
   { Si5351_bFMStreamState = FM_STREAM_OFF;
   }
  
  
  // Continue in AN619 page 2, "Conceptualizing a Frequency Plan" :
  // > The relationship between the VCO and output frequencies is given below
  // >
  // >               f_vco
  // >  fout_x = -------------------------------
  // >             Output_Multisynth_x * R_x   
  // >
  // >
  // >  f_vco = f_ref * Feedback_Multisynth
  // >
  // > (...)
  // >
  // > 2.1.1. Selecting the Proper VCO Frequencies and Divide Ratios
  // >
  // > The general criteria below may be used to set the VCO frequencies. 
  // > This is a general model, and individual applications may require 
  // > some modification.
  // >  1.  Valid Multisynth [-OUTPUT-] divider ratios are 4, 6, 8, 
  // >      and any fractional value between 8  +  1 / 1,048,575 
  // >      and 2048. This means that if any output is greater than 
  // >      112.5 MHz (900 MHz/8), then this output frequency sets 
  // >      one of the VCO frequencies.
  if( ! pll_params_ready )  // still need to determine the VCO / PLL parameters ?
   {
     if( dblWantedFrequency_Hz <= 150e6 ) // less than 150 MHz "out" ? 
      { dwVcoOutputDivider = ( dwApproxVcoFreq / (DWORD)dblWantedFrequency_Hz )
                          & ~1UL;  // we even want an EVEN output divider !
        // Example with the lowest possible output frequency (only slightly violating the specs):
        //    f_vco = 128 * 2048 * 2 kHz = 524.288 MHz (acceptable) .
        // Example (DB0BI 23cm) : dblWantedFrequency_Hz = 1298.350 MHz / 128 = 10.143359 .. MHz
        //                        dwApproxVcoFreq = 900000000 (900 MHz)
        //                        dwVcoOutputDivider = 88
        // LATER: f_vco = dwVcoOutputDivider * dblWantedFrequency_Hz = 892.61 MHz
        //  (unless the output shall be FREQUENCY MODULATED via software)
        // For low output frequencies that need a lot of "division",
        //     dwVcoOutputDivider should be a multiple of 128 .
        // But for VERY low output frequencies, we cannot clear the
        //     least significant EIGHT bits in dwVcoOutputDivider, 
        //     because that would result in a VCO too large to divide down
        //     to the 'wanted' output frequency !  Example:
        //     dblWantedFrequency_Hz = 2500.0 
        //     -> dwVcoOutputDivider = 800 MHz / 2.5 kHz = 320000 . 
        //        But together with the maximum 'R' divider ratio (128),
        //        the maximum possible dwVcoOutputDivider is 128 * 2048 = 262144 !
        // For HIGH output frequencies (between 112.5 and 150 MHz),
        //     it's impossible to use a FRACTIONAL output divider anyway, because:
        //  > "4.1.2. Output Multisynth Divider Equations (Fout <= 150 MHz)" :
        //  > Divider represented a fractional number,  a + b/c
        //  > between 8 + 1/1048575 and 2048 .
        // ,----------'
        // '--> that's the MINIMUM for 'a' !  900 MHz / 8 = 112.5 MHz. Sigh..
        if( dwVcoOutputDivider <= 8 ) // no FRACTIONAL output divider below div-by-8 !
         { // > Valid INTEGER Multisynth divider ratios are 4, 6, 8 
           //    (FRACTIONAL divider ratios are only possible ABOVE 8)
           dwVcoOutputDivider &= 0x00E; // turn the VCO output divider ratio into an EVEN INTEGER (4,6,8).
         }
        if( dwVcoOutputDivider > (128*2048UL) )
         {  dwVcoOutputDivider = (128*2048UL); // max. divider from "output MSynth" * "R Divider"
         }
        else 
        if( dwVcoOutputDivider >= 900 )   // sufficiently large to strip EIGHT LSBITS ?
         {  dwVcoOutputDivider &= ~255UL; // NOW we can divide by 2^7 without loss,
            // the remaining dwVcoOutputDivider would still be an EVEN number,
            // and the internal VCO frequency 'f_vco' calculated further below 
            //   ( f_vco = dwVcoOutputDivider / dblWantedFrequency_Hz )
            // will be BELOW 900 MHz because we TRUNCATED dwVcoOutputDivider here.
            // If we had 'rounded' dwVcoOutputDivider above, we'd violate the specs.
         }
        // Found the "final" VCO output divider ratio, so calculate the internal VCO frequency:
        f_vco = (double)dwVcoOutputDivider * dblWantedFrequency_Hz;
        // In the interest of low jitter / phase noise,
        //      f_vco should be divided DOWN to the 'wanted output frequency', 
        //      by an EVEN INTEGER  :
        // [in] dwVcoOutputDivider = product of what AN619 calls
        //       "Output Multisynth Divider" (here: integer, not fractional)
        //       and "R Divider" (necessary for frequencies below 500 kHz).
        // [out] output_divider_params.a,b,c,r_div : "Output Multisynth" params.
        //         Based on THESE (especially a=EVEN, b=0, c=1),
        //         we will later decide if 'Integer Divide Values'
        //         can be used (inside Si5351_SetOutputDividerRegisters).
        //         The 'R Divider' can only divide by powers of two, 
        //         maximum = divide by 128 .
        // The 'Output Multisynth Divider' is assumed to divide between 8 and 2048,
        // but that's when dividing by a+b/c,  not with b=0, c=1. So ASSUME that,
        // when using "Integer Divide Values" (chapter 4.1.2.1), the maximum of 'a'
        // is 2047. 
      }
     else // "wanted" output frequency ABOVE 150 MHz -> can only divide the VCO by FOUR !
      { dwVcoOutputDivider = 4; // [AN619] page 6, "4.1.3 Output Multisynth ... for 150 to 200 MHz"
        // -> f_vco = f_out * 4; "MSx_P1=0", "MSx_P2=0", "MSx_P3=1", "MSx_INT=1", "MSx_DIVBY4=0b11" !
        //    (only SOME of the above FIVE special conditions for 150 MHz < Fout <= 200 MHz
        //     can be taken care of in Si5351_SetOutputDividerRegisters(),
        //     which will recognize output_divider_params.a = 4
        //     as a 'special case', and set the FIVE special conditions listed above)
        bClockCtrlRegValue |= Si5351_MSK_CLK_CTRL_MS_INT; // "MultiSynth x [OUTPUT] Integer Mode" : 1="force Integer mode", HERE: along with "MSx_DIVBY4==0b11" .
        f_vco = 4.0 * dblWantedFrequency_Hz;
      }
     output_divider_params.r_div = 1;
     output_divider_params.a = dwVcoOutputDivider;
     while( (output_divider_params.a>2048) // 'A' too large but still EVEN, and still not the maximum 'R Divide' ?
        && ((output_divider_params.a & 1)==0)  // still an EVEN integer divider ?
        && (output_divider_params.r_div<128) ) // still not the maximum for the 'R' divider ?
      { output_divider_params.a >>= 1;
        output_divider_params.r_div <<= 1; // next power of two for the 'R divider'
      }
     // >  2.  For the frequencies where jitter is a concern make the 
     // >      output Multisynth divide ratio an integer. 
     // >      If possible, make both output and feedback Multisynth ratios 
     // >      integers.
     // >  3.  Once criteria 1 and 2 are satisfied, try to select as many
     // >      integer output Multisynth ratios as possible.
     // >
     // > (...)
     // >
     // > 3.2.  Feedback Multisynth Divider Equations
     // >
     // >
     // > f_vco = f_ref * ( a + b/c )
     // >
     // > The fractional ratio 
     // >            b
     // >     a  +  ---
     // >            c
     // > has a valid range of 15 + 0/1,048,575 and 90 and is represented 
     // > in the Si5351 register space using the equations below...
     //   (the conversion of a,b,c into BITGROUP VALUES,
     //    and the distribution of those BITGROUP VALUES into REGISTERS
     //    happens in Si5351_SetPLLFeedbackRegisters(). 
     //    Don't care about "MSNx_P1[bits 17..0]", "MSNx_P2[bits 19..0]"
     //    and "MSNx_P3[bits 17..0]" at this point)
     // > 
     // > 2.1. Integer Divide Values (FBx_INT)
     // >  If a + b/c is an EVEN integer, integer mode may be enabled for PLLA 
     // >  or PLLB by setting parameter FBA/B_INT. In most cases setting this bit
     // >  will improve jitter when using even integer divide values. (...)
     //   
     // > Each PLL generates an intermediate VCO frequency 
     // > in the range of 600 to 900 MHz...
     // Don't violate the specs "too severely" :
     if( f_vco < 524e6 )  // low enough for 2 kHz (with all output dividers)
      {  f_vco = 524e6;
         fOk   = FALSE;
      }
     if( f_vco > 900e6 )
      {  f_vco = 900e6;
         fOk   = FALSE;
      }
     output_divider_params.f_in = f_vco;
     pll_feedback_ratio = f_vco / g_dblReferenceFrequency_Hz;

     // pll_feedback_ratio is the "ideal" value for the PLL feedback divider.
     // In an Si5351, this ratio can only be APPROXIMATED by a 
     // fractional expression in the form 
     //              pll_feedback_ratio = a + b/c  .
     // See "Feedback Multisynth Divider Equations" quoted from AN619 above.
     // In THIS implementation, we use the glorious Farey algorithm to find
     // the BEST (really) possible values for b and c (a is the INTEGER part):
     dwA = (DWORD)pll_feedback_ratio;  // <- TRUNCATED, not ROUNDED !
     fractional_part = pll_feedback_ratio - (double)dwA;

     // Find the best fraction (b/c) for the fractional part - thanks to Farey:
     n_iterations = Farey( fractional_part, // [in] double x, between 0.0 and 1.0
                          1048575,  // [in] i32MaxDenominator (*)
                    &i32Numerator,  // [out] numerator (Dividend),  integer
                  &i32Denominator); // [out] denominator (Divisor), integer
     // (*) Must fit in 20 bits.  1048575 = 2^20-1 .  
     //     If the NUMERATOR shall be used to frequency-modulate the PLL, 
     //     we'll reprogram only SIXTEEN bits of the numerator for a sufficient
     //     sampling rate. Details about the experimental frequency modulation later.
     // Example (DB0BI 23cm) : dblWantedFrequency_Hz = 1298.350 MHz / 128 = 10.143359 MHz
     //                        dwApproxVcoFreq = 900000000 (900 MHz)
     //                        dwVcoOutputDivider = 88
     //           f_vco = dwVcoOutputDivider * dblWantedFrequency_Hz = 892.6156254 MHz
     //           fractional_part := 0.7046250000000001 (shown by MPLAB X)
     //           i32Integer      := 35     (aka 'A') \  '''''''''''''''''':
     //           i32Numerator    := 1813   (aka 'B')  > displayed in menu :
     //           i32Denominator  := 2573   (aka 'C') /  "Synth Registers" :
     //           n_iterations = 23 (amazingly low..)
     //  ( 25MHz * (i32Integer + i32Numerator/i32Denominator ) / 88 ) * 128
     //            = 1298.349998 MHz . Ok.
     //
#if( SWI_SI5351_SUPPORT_FREQUENCY_MODULATION  )  
     // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
     // About 'FREQUENCY MODULATION' using only the PLL FEEDBACK'S NUMERATOR : 
     // EXAMPLE: [in] dblWantedFrequency = 145 MHz, iFreqModDeviation_Hz = 2500 ->
     //     dwVcoOutputDivider = circa 900 MHz / 145 MHz = 6.2
     //        -> nearest even integer = 6; f_vco = 145 MHz * 6 = 870 MHz .
     //     f_vco = f_ref * ( a + b/c ), here: f_vco = 25 MHz * ( 34 + 4/5 ) .
     //       ,---------------------------------------------------'   
     //       '--> a = 34,  b=4,  c=5 
     //            (lowest possible numerator / denominator from Farey() ).
     //     With such a low numerator (4), the resulting stepwidth for FM
     //     would be too large. So extend numerator and denominator, i.e.
     //     multiply b and c by the same factor so that 'c' gets as close 
     //     to 2^20 (minus some headroom) as allowed by the Si5351, 
     //     with just enough headroom for 16 bit samples
     //     (minus the sign bit, thus 2^15) ?
     // factor := int((2^20-2^15)/c)  =: 203161.000000
     // 
     // b := b*factor  =: 812644.000000
     // c := c*factor  =: 1015805.000000
     // 
     // f1 := (f_ref * (a+(b-2)/c) ) / output_div  =: 144999991.796326
     // f2 := (f_ref * (a+(b-1)/c) ) / output_div  =: 144999995.898163
     // f3 := (f_ref * (a+(b+0)/c) ) / output_div  =: 145000000.000000
     // f4 := (f_ref * (a+(b+1)/c) ) / output_div  =: 145000004.101837
     // f5 := (f_ref * (a+(b+2)/c) ) / output_div  =: 145000008.203674
     // 
     // ; Resulting modulation 'stepwidth' (least significant bit of the modulator):
     // f4-f3  =:   4.101837 
     //     Stepwidth in the VHF range of ~~4 Hz, ok for VOICE (not HiFi).
     //     Time for a first test, also to find out if the numerator (b)
     //     may 'slighly exceed' the denominator (c) at the POSITIVE PEAK
     //     of the modulation (f_momentary = f_center + iFreqModDeviation_Hz) :
     if( (iFreqModDeviation_Hz != 0 ) // modify numerator & denominator for FM ?
      && (iOutputIndex==0)    // .. only intended for 'VFO A' alias 'TX'
      &&(Si5351_bFreqModulationMethod[iOutputIndex]==FM_METHOD_PLL_FEEDBACK) ) 
      { i32Factor = (1048576/*2^21*/-32768/*2^15*/) / i32Denominator;
        i32Denominator *= i32Factor; // -> e.g. 203161 * 5 = 1015805
        i32Numerator   *= i32Factor; // -> e.g. 203161 * 4 =  812644
        // See THEORETIC output frequencies / stepwith of FM samples in Hz,
        // calculated with DL4YHF's CalcEd, and documented in file
        //                 Si5351_Frequency_Modulation.txt
        // 
      }
     Si5351_FM_Carrier_dwPLLFeedbackIntegerPart = dwA;
     Si5351_FM_Carrier_dwPLLFeedbackNumerator   = (DWORD)i32Numerator;
     Si5351_FM_Carrier_dwPLLFeedbackDenominator = (DWORD)i32Denominator;
     // < - - - - - - end of "FM using the PLL *FEEDBACK* divider" - - - - - -
#endif // frequency modulation using the PLL FEEDBACK divider ?

     // Configure "PLL A" or "PLL B" with the FRACTIONAL FEEDBACK parameters for the VCO.
     // This what AN619 calls "Feedback Multisynth Divider" on page 3 :
     fOk &= Si5351_SetPLLFeedbackRegisters(  // -> returns TRUE when OK, FALSE on any error
             bPLLBaseAddr, // [in] Si5351_REG_MSYNTH_PLL_A or Si5351_REG_MSYNTH_PLL_B
             dwA,          // [in] integer part 'a' in 'a + b/c', a = 15 .. 90 (?)
            (DWORD)i32Numerator,    // [in] numerator    'b' in 'a + b/c'
            (DWORD)i32Denominator,  // [in] denominator  'c' in 'a + b/c'
            &Si5351Regs.g.FeedbackDivider[iOutputIndex] ); // [out] register latches

     // Split the parameters a,b,c for the OUTPUT DIVIDER (fractional or integer)
     // into the strange bitgroups p1,p2,p3 for the "output multisynth", and set them:
     fOk &= Si5351_SetOutputDividerRegisters( // -> Si5351_SetOutputDividerRegsP1P2P3()
        bOutDivBaseAddr, // [in] Si5351_REG_MSYNTH0 (42) or Si5351_REG_MSYNTH1 (50)
        &output_divider_params, // [in] f_in, f_out, a,b,c, r_div (parameters for the OUTPUT divider)
        &Si5351Regs.g.OutputDivider[iOutputIndex] ); // [out] register latches
   } // end if( ! pll_params_ready ) ?
  
  // As noted by Hans, G0UPL, resetting the PLL causes a glitch, but WITHOUT
  // resetting the PLL after a 'significant' change of the parameters,
  // the PLL won't start at all. The big question is : What IS a 'significant'
  // change in frequency for the VCO / the Phase-Locked-Loop to fail ?
  // From G0UPL [BITX20]:
  // > You do NOT need to do a PLL reset for subsequent PLL feedback divider 
  // > changes! I have not been able to find any size of frequency change where
  // > a PLL reset is required. Certainly incrementally tuning across 
  // > an amateur band requires no PLL reset. Given the datasheet comments 
  // > and the fact that evidently you do at least need one PLL reset 
  // > at the start, I remained a bit nervous and in my code I do generate 
  // > a PLL reset when there is a "large" frequency step... what does 
  // > "large" mean, who knows. I arbitrarily set it to 10kHz and no QRP Labs 
  // > customer has ever had a problem with it to the best of my knowledge.
  fltTemp = (float)f_vco - Si5351_fltPrevVcoFreq[iOutputIndex];
  if( (fltTemp < -10e3) || (fltTemp > 10e3) ) // *VCO* frequency off by more than 10 kHz ?
   {   
     fOk &= Si5351_WriteRegister_8Bit( Si5351_REG_PLL_RESET, bPLLResetRegValue );
     // DL4YHF: "Resetting" the PLL this way caused a temporary outage of the
     // output signal, often about 1.42 ms long, sometimes only 0.71 ms.
   }     
  if( fOk )  // remember the NEW internal VCO frequency (600..900 MHz) for the next call
   { Si5351_fltPrevVcoFreq[iOutputIndex] = (float)f_vco; // 'float' resultion is ok here
   }
   
  // Close an internal switch between "Synthesis Stage 1" (=the PLL or VCO ?)
  // and "Synthesis Stage 2" (=the Output Divider ?) via the "Clock Control" register.
  // The same register ("Register 16. CLK0 Control") is also used to configure
  // "MultiSynth x Integer Mode".  Use that IF POSSIBLE: 
  if( Si5351_bFreqModulationMethod[iOutputIndex] == FM_METHOD_OUTPUT_DIV )
   { // Need a FRACTIONAL output divider for software frequency modulation:     
   }
  else // can POSSIBLY use ""MultiSynth x Integer Mode" (for this OUTPUT divider):
  if( ((output_divider_params.a & 1)==0) // even INTEGER part ? 
    && (output_divider_params.b == 0) )  // no NUMERATOR for the fractional part ?
   { bClockCtrlRegValue |= Si5351_MSK_CLK_CTRL_MS_INT; // "MultiSynth x [OUTPUT] Integer Mode" : 1="force Integer mode to improve jitter performance"
     // AN619 page 6 (about "MSx_INT" ) : 
     // > If any of the MS0-MS5 is an even(!!!) integer, Multisynth integer mode
     // > may be enabled by setting MSx_INT=1 (see registers 16-21, bit 6). 
     // > In most cases setting this bit will improve jitter when using 
     // > even integer divide values.
     //   WB: Above 112.5 MHz, the OUTPUT DIVIDER *must* use 
     //       'even integer mode'. Above 150 MHz, it *must* additionally only
     //       divide BY FOUR, and besides "MSx_INT"=1, "MSx_DIVBY4" must be set to 0b11 !
     // AN619 page 20 (about Register 16 = "CLK0 Control" ) : 
     // > This bit can be used to force MS0 into Integer mode to improve 
     // > jitter performance. Note that the fractional mode is necessary 
     // > when a delay offset is specified for CLK0.
   }   
  // No-No: bClockCtrlRegValue |= Si5351_MSK_CLK_CTRL_PDN; // "Clock x Power Down" : 0=powered up, 1=powered down
  // Already set or cleared : Si5351_MSK_CLK_CTRL_MS_SRC = "MultiSynth Source Select for CLK0": 0=PLLA, 1=PLLB
  // No-No: bClockCtrlRegValue |= Si5351_MSK_CLK_CTRL_INV; // "Output Clock 0/1/2/..7 Invert"
  bClockCtrlRegValue |= ( Si5351_VAL_CLK_CTRL_SRC_SYNTH  // "Select MultiSynth as the source for CLKx"
                        | Si5351_VAL_CLK_CTRL_IDRV_8mA); // "Drive Strength : 8 mA"
  fOk &= Si5351_WriteRegister_8Bit( bClockCtrlRegAddr, bClockCtrlRegValue );

  // Timing seen on the I2C bus during a complete Si5351_SetFrequency(),
  //     using a dsPIC33EPxxMC502's hardware-I2C at 943 kBit/sec:
  //       _             ______             _______         _______________
  // SCL :  |||||||||||||      |||||||||||||       |||||||||
  //            (A)        (B)      (C)       (D)     (E)  
  //        :<--112us-->:<-24->:<--112us-->:<-40u->:<-32u->:
  // (A) Si5351_SetPLLFeedbackRegisters() consumed 112 us per call;
  // (B) 24 us gap between ..SetPLLFeedback() and ..SetOutputDivider()
  // (C) Si5351_SetOutputDividerRegisters() also consumed 112 us per call;
  // (D) 40 us gap between SetOutputDivider() and setting 'Clock Control Reg'
  // (E) 32 us consumed by Si5351_WriteRegister_8Bit(),
  //     consisting of 8 I2C bits for the SLAVE ADDRESS,
  //                   8 I2C bits for the 8-bit REGISTER NUMBER,
  //                   8 I2C bits for the 8-bit REGISTER VALUE,
  //           plus a few I2C clock cycles for 'START','ACKs', and 'STOP'.
  //     32 us * 943 kBit/sec = circa 30 I2C bit clock cycles; ok.

# if( SWI_SI5351_SUPPORT_FREQUENCY_MODULATION )    
  if( Si5351_bFMStreamState == FM_STREAM_PAUSED ) // frequency modulation was only TEMPORARILY paused ->
   { Si5351_StartSendingFM(); // continue sending frequency modulation samples
   }
# endif // SWI_SI5351_SUPPORT_FREQUENCY_MODULATION 
  
  return fOk; // hopefully TRUE = "all ok" (including the I2C bus transfer)
   
} // end Si5351_SetFrequency()


//---------------------------------------------------------------------------
void Si5351_GetRegNumberFromDataAddress( BYTE *pbRegAddress, BYTE *pbData )
  // If *pbRegAddress is Si5351_GET_REG_FROM_ADDRESS, determines the
  //   REAL register number from pbData (address in RAM, points to Si5351Regs)
{ if(*pbRegAddress == Si5351_GET_REG_FROM_ADDRESS ) // determine the register number from pRegs (address in RAM) ?
   { *pbRegAddress = pbData - Si5351Regs.bArray;    // -> 0..187 (hopefully)
   }
} // end Si5351_GetRegNumberFromDataAddress()


//---------------------------------------------------------------------------
BOOL Si5351_WriteRegister_8Bit(BYTE bRegAddress, BYTE bData)
  // Note:
  //  * Si5351_WriteRegister_8Bit() will always WAIT FOR COMPLETION.
  //    Thus it must NOT (never, ever) be called from interrupt handlers !
{
  BOOL fOk;

#if( SWI_USE_PROGRAMMED_I2C ) // use DL4YHF's "software" I2C bus master ?  
  // [Si5351DS] page 19 : Write Operation - Single Write .
  // First Slave Select, with the R/!W-bit cleared:
  fOk = I2C_SendSlaveAddress( (SI5351_I2C_7BIT_SLAVE_ADDRESS<<1) | 0/*0=WRITE,1=READ*/);
  // Note: I2C_SendSlaveAddress() ends with 'get acknowledge'.
  //       At THIS point, the bus is still occupied, both SDA and SCL are LOW.
  if( fOk )  // initial 'SendSlaveAddress' ok ?
   {
     // Next: Send the 8-bit 'Reg Addr', followed by another ACK:
     fOk &= I2C_WriteByte( bRegAddress );
     fOk &= I2C_GetAcknowledge();
     // Next: Send the 8-bit 'Data', followed by another ACK (even if it's the LAST byte):
     fOk &= I2C_WriteByte( bData );
     fOk &= I2C_GetAcknowledge();
   }
  // We (the master) SENT the last byte so send STOP condition ("P"):
  fOk &= I2C_Stop();  // -> TRUE when successful 
  // Note: It's important to call I2C_Stop() even if the slave didn't respond !
  //       After I2C_Stop(), the bus is back in idle, both SDA and SCL are HIGH.
#elif( SWI_USE_HARDWARE_I2C )  
  I2C_WriteBlock( SI5351_I2C_7BIT_SLAVE_ADDRESS, bRegAddress, &bData, 1,
# if( SWI_SI5351_SEND_FM_STREAM_IN_I2C_COMPLETION_CALLBACK )
                 Si5351_OnWriteMultipleRegistersComplete );     
# else // ! SWI_SI5351_SEND_FM_STREAM_IN_I2C_COMPLETION_CALLBACK :
                 NULL );  // no 'on completion'-callback required here
# endif //  SWI_SI5351_SEND_FM_STREAM_IN_I2C_COMPLETION_CALLBACK ?
  fOk = I2C_WaitForCompletion();
#endif // SWI_USE_SOFTWARE/HARDWARE_I2C ?
  
  if( fOk ) // update the 'shadow registers' (inspectable via debugger and 'HW Diagnostics'):
   { if( bRegAddress < SI5351_NUM_REGISTERS )
      { Si5351Regs.bArray[bRegAddress] = bData;
      }
   } 
  return fOk;
} // end Si5351_WriteRegister_8Bit()

//---------------------------------------------------------------------------
BOOL Si5351_WriteMultipleRegisters(
       BYTE bRegAddress, // [in] e.g. Si5351_REG_MSYNTH_PLL_A . 
                         // May be 0xFF = Si5351_GET_REG_FROM_ADDRESS for
                         // auto-detection (if pbData points into Si5351_Regs)
       BYTE *pbData,     // [in] source address; OFTEN (but not always) points
                         //      into Si5351Regs ['shadow registers' in RAM].
       BYTE nBytes)      // [in] number of bytes to write in a single block.
  // Note:
  //  * Si5351_WriteMultipleRegisters() will always WAIT FOR COMPLETION.
  //    Thus it must NOT (never, ever) be called from interrupt handlers !
{
  BOOL fOk;
  
  // If bRegAddress is Si5351_GET_REG_FROM_ADDRESS, determine the
  //    register number from pbData (address in RAM, points to Si5351Regs): 
  Si5351_GetRegNumberFromDataAddress( &bRegAddress, pbData ); 

#if( SWI_USE_PROGRAMMED_I2C ) // use DL4YHF's "software" I2C bus master ?  
  // [Si5351DS] page 19 : Write Operation - Single Write .
  // First Slave Select, with the R/!W-bit cleared:
  fOk = I2C_SendSlaveAddress( (SI5351_I2C_7BIT_SLAVE_ADDRESS<<1) | 0/*0=WRITE,1=READ*/);
  // See oscillogram sketched as 'ASCII ART' in I2C_SendSlaveAddress() !
  // The ACKNOWLEDGE (from receiving slave to transmitting master)
  // appears like an "unhealthy short spike" on SDA, even BEFORE
  // the master sends the 9th clock bit on SCL clock-in the ACK !
  
  if( fOk )  // initial 'SendSlaveAddress' ok ?
   {
     // Next: Send the 8-bit 'Reg Addr', followed by another ACK:
     fOk &= I2C_WriteByte( bRegAddress );
     fOk &= I2C_GetAcknowledge();
     // Next: Send data (byte by byte), followed by ACKs from the receiver:
     while( fOk && (nBytes>0) )
      {
       fOk &= I2C_WriteByte( *pbData );
       fOk &= I2C_GetAcknowledge();
       if( fOk ) // update the 'shadow registers' (inspectable via debugger and 'HW Diagnostics'):
        { if( bRegAddress < SI5351_NUM_REGISTERS )
           { Si5351Regs.bArray[bRegAddress] = *pbData;
           }
        } 
       ++bRegAddress;
       ++pbData;
       --nBytes;
      }
   }  
  // Finally send the STOP condition ("P") - even if the slave didn't react
  fOk &= I2C_Stop();  // -> TRUE when successful 
#elif( SWI_USE_HARDWARE_I2C )  
  I2C_WriteBlock( SI5351_I2C_7BIT_SLAVE_ADDRESS, bRegAddress, pbData, nBytes,
# if( SWI_SI5351_SEND_FM_STREAM_IN_I2C_COMPLETION_CALLBACK )
                 Si5351_OnWriteMultipleRegistersComplete );     
# else // ! SWI_SI5351_SEND_FM_STREAM_IN_I2C_COMPLETION_CALLBACK :
                 NULL/*OnCompletion-Callback*/);
# endif //  SWI_SI5351_SEND_FM_STREAM_IN_I2C_COMPLETION_CALLBACK ?
  fOk = I2C_WaitForCompletion();
  if( fOk ) // update the 'shadow registers' (inspectable via debugger and 'HW Diagnostics'):
   { while( nBytes-- )
      { if( bRegAddress < SI5351_NUM_REGISTERS )
         { Si5351Regs.bArray[bRegAddress] = *pbData;
         }
       ++bRegAddress;
       ++pbData;
       --nBytes;
      }
   } 
#endif // SWI_USE_SOFTWARE/HARDWARE_I2C ?
  
  return fOk;
} // end Si5351_WriteMultipleRegisters()

//---------------------------------------------------------------------------
BOOL Si5351_ReadRegister_8Bit( BYTE bRegAddress, BYTE *pbData)
  // Note:
  //  * Si5351_ReadRegister_8Bit() will always WAIT FOR COMPLETION.
  //    Thus it must NOT (never, ever) be called from interrupt handlers !
{
  BOOL fOk;

  // If bRegAddress is Si5351_GET_REG_FROM_ADDRESS, determine the
  //    register number from pbData (address in RAM, points to Si5351Regs): 
  Si5351_GetRegNumberFromDataAddress( &bRegAddress, pbData ); 
  
  
#if( SWI_USE_PROGRAMMED_I2C ) // use DL4YHF's "software" I2C bus master ?  
  // On entry, both SCL and SDA should be high.
  //
  // From the M24128 datasheet (EEPROM) : "Random Address Read"
  // > A dummy Write is performed to load the address
  // > into the address counter (as shown in Figure 9) but
  // > without sending a Stop condition. Then, the bus
  // > master sends another Start condition, and repeats
  // > the Device Select Code, with the RW bit set to 1.
  // > The device acknowledges this, and outputs the
  // > contents of the addressed byte. The bus master
  // > must not acknowledge the byte, and terminates
  // > the transfer with a Stop condition.
  // > (...)
  //
  // An article about the "Repeated Start Condition" at www.i2c-bus.org said:
  // > Instead of sending the stop condition it is also allowed to send 
  // > another start condition again followed by an address (and of course 
  // > including a read/write bit) and more data. This is defined recursively 
  // > allowing any number of start conditions to be sent. 
  // > The purpose of this is to allow combined write/read operations 
  // > to one or more devices without releasing the bus and thus 
  // > with the guarantee that the operation is not interrupted.
  // 
  // Sounded good. So any "I2C"-compatible chip should support both 
  //     (STOP and START between writing the REGISTER ADDRESS and READING DATA,
  //      or NO STOP but a REPEATED START to have both in a "single over") ?
  //  Tried something like this ... did NOT to work with an Si5351:
  //
  //  Start   "Write"     REPEATED Start  "Read"    Not ACK StoP
  //  |          |              |          |              | |
  // ,-,--------,-,-,--------,-,-,--------,-,-,----------,-,-,
  // |S|Slv Addr|0|A|Reg Addr|A|S|Slv Addr|1|A|Data[7..0]|N|P|
  // '-'--------'-'-'--------'-'-'--------'-'-'----------'-'-'
  //   |<--1st--->|              |<--2nd--->|
  //    SlaveSelect               SlaveSelect
  // (Reading from a register via I2C using a "REPEATED START",
  //    without STOP between writing the register address 
  //                     and reading the data byte[s]  )
  //  The above didn't work with an Si5351, there was no ACK for the 
  //  2nd slave select (the one with R/!W = 1), so tried the following instead:
  //   
  // The principle used in the Si5351A appeared a bit different than the above
  // description quoted from the datasheet of a serial EEPROM.
  // See "Si5351A/B/C-B" datasheet Rev 1.3, page 19, "Read Operation - Single Byte":
  //
  //  Start   "Write"        Stop    Start     "Read"     Not ACK StoP
  //  |          |              |     |          |              | |
  // ,-,--------,-,-,--------,-,-,   ,-,--------,-,-,----------,-,-,
  // |S|Slv Addr|0|A|Reg Addr|A|P|---|S|Slv Addr|1|A|Data[7..0]|N|P|
  // '-'--------'-'-'--------'-'-' : '-'--------'-'-'----------'-'-'
  //   |<--1st--->||         / /|\ :   |<--2nd--->||            |
  //   SlaveSelect |        /   |  :    SlaveSelect|      NACK from MASTER
  //   to WRITE    |       /    |  :    to READ    |
  //             "Ack"    /     |  :             "ACK" from SLAVE
  //             from Slave  |  |  '.. Bus is "released" (idle) here.
  //   (Si5351 ended ACK-ing |  |
  //    about 400 ns after   | "STOP" between WRITING the register address
  //    the falling edge of  |      and sending a 2nd slave-select to READ .
  //    the 9-th clock bit)  |
  //    
  // First Slave Select, with the R/!W-bit cleared, to *write* the REGISTER address
  fOk = I2C_SendSlaveAddress( // .. including START and ACKNOWLEDGE (1st call)
           (SI5351_I2C_7BIT_SLAVE_ADDRESS<<1) | 0/*0=WRITE,1=READ*/);
  // At this step, SCL=LOW + SDA=HIGH, because the bus is still occupied !
  if( fOk )  // initial 'SendSlaveAddress' ok ?
   {
     fOk &= I2C_WriteByte(bRegAddress); // Send register address (only 8 bits in a Si5351)
     fOk &= I2C_GetAcknowledge();       // wait(?) for ACK from slave (at least provide a 9th clock bit)
            // 2021-08-22: Sometimes(!) did NOT get an acknowledge here,
            //             even though the initial I2C_SendSlaveAddress() DID .
     fOk &= I2C_Stop(); // unnecessary for M24128, but required for Si5351 !

     // Second Slave Select, now with the R/!W-bit set, to READ further below .
     if( ! I2C_SendSlaveAddress( (SI5351_I2C_7BIT_SLAVE_ADDRESS<<1) | 1/*0=WRITE,1=READ*/ ) ) // 2nd call
      { fOk = FALSE;   // <- debug-friendly .. set a breakpoint HERE !
      }
     
     // Waveform seen at this point, when stopped via breakpoint BELOW :     .
     //                                                                      .
     //         ____________                               _____             .
     // SDA ___|   1     1  |__0_____0_____0_____0_____0__|  1  \_ _ACK_ _ _ .
     //            __    __    __    __    __    __    __    __    __        .
     // SCL ______|A6|__|A5|__|A4|__|A3|__|A2|__|A1|__|A0|__|RD|__|AC|_____  .
     //                                                     /|\   /|\        .
     //           |___I2C slave address of Si5351: 0x60 __|  |     |         .
     //                                                    1=Read, |         .
     //                                                not WRITE   |         .
     //                                                         Slave ACKs   . 
     //
     
     // Read the data byte from the I2C slave, with the master sending NOT ACK..
     *pbData = I2C_ReadByte(I2C_SEND_NACK);
     // > A master receiver must signal an end of data to the slave transmitter
     // > by NOT GENERATING an acknowledge on the last byte that has been
     // > clocked out of the slave. In this case the transmitter must leave
     // > the data line High to enable the master to generate the STOP condition.
   } // end if < successfully selected the I2C-slave >

  // Finally send the STOP condition ("P") - even if the slave didn't react
  fOk &= I2C_Stop();
  // 2007-03-05: Failed in I2C_Stop() after "reading the byte with NOT-ACK".
  //  BUT AGAIN, the problem only seemed to be caused by the debugger,
  //    which "peeked" the I2C status register, with ugly side effects.
  //    In fact, when running in 'real time', I2C_Stop() didn't fail .
  //  (don't remove this comment; save time if the problem happens again
  //   when switching from the SOFTWARE I2C MASTER to the dreadful I2C hardware)
#elif( SWI_USE_HARDWARE_I2C ) // much faster, but caused more headaches :
  I2C_ReadBlock( SI5351_I2C_7BIT_SLAVE_ADDRESS, bRegAddress, pbData, 1, NULL );
  fOk = I2C_WaitForCompletion(); // 2021-11-14: FAILED to "wait for completion" when trying to read register #0 ?!
      // At THIS point: I2C_bMasterState = 0xFF . 
#endif // SWI_USE_SOFTWARE/HARDWARE_I2C ?
  
  
  return fOk;  // TRUE=success, FALSE=error (I2C bus trouble?)
} // end Si5351_ReadRegister_8Bit()

//---------------------------------------------------------------------------
BOOL Si5351_ReadMultipleRegisters(
       BYTE bRegAddress, // [in] e.g. Si5351_REG_MSYNTH_PLL_A . 
                         // May be 0xFF = Si5351_GET_REG_FROM_ADDRESS for
                         // auto-detection (if pbData points into Si5351_Regs)
       BYTE *pbData,     // [in] destination address; OFTEN (but not always) points
                         //      into Si5351Regs ['shadow registers' in RAM].
       BYTE nBytes)      // [in] number of bytes to read in a single block.
  // Notes:
  //  * Si5351_ReadMultipleRegisters() will always WAIT FOR COMPLETION.
  //    Thus it must NOT (never, ever) be called from interrupt handlers !
  //  * The read-out register values are *NOT* additionally stored
  //    in Si5351Regs.bArray[ bRegAddress .. +nBytes-1], 
  //    because Si5351_ReadMultipleRegisters() may be called 
  //    from Si5351_VerifyRegisterValues(), where Si5351Regs.bArray[]
  //    contains the 'wanted' values that must not be modified .
{
  BOOL fOk;
  
  // If bRegAddress is Si5351_GET_REG_FROM_ADDRESS, determine the
  //    register number from pbData (address in RAM, points to Si5351Regs): 
  Si5351_GetRegNumberFromDataAddress( &bRegAddress, pbData ); 

  
#if( SWI_USE_PROGRAMMED_I2C ) // use DL4YHF's "software" I2C bus master ?  
  // [Si5351DS] page 19 : Read Operation - Burst (Auto Address Increment) .
  // First Slave Select, with the R/!W-bit cleared, to *write* the REGISTER address
  fOk = I2C_SendSlaveAddress( // .. including START and ACKNOWLEDGE (1st call)
           (SI5351_I2C_7BIT_SLAVE_ADDRESS<<1) | 0/*0=WRITE,1=READ*/);
  // At this step, SCL=LOW + SDA=HIGH, because the bus is still occupied !
  if( fOk )  // initial 'SendSlaveAddress' ok ?
   {
     fOk &= I2C_WriteByte(bRegAddress); // Send register address (only 8 bits in a Si5351)
     fOk &= I2C_GetAcknowledge();       // wait(?) for ACK from slave (at least provide a 9th clock bit)
         // 2021-08-22: Sometimes(!) did NOT get an acknowledge here,
         //             even though the initial I2C_SendSlaveAddress() DID .
         // At this step, SCL=LOW + SDA=HIGH, because the bus is still occupied !
     fOk &= I2C_Stop(); // unnecessary for M24128, but required for Si5351 !
         // At this step, both SCL and SDA are HIGH, the bus is in idle state.
        
     // Second Slave Select, now with the R/!W-bit set, to READ further below .
     if( ! I2C_SendSlaveAddress( (SI5351_I2C_7BIT_SLAVE_ADDRESS<<1) | 1/*0=WRITE,1=READ*/ ) ) // 2nd call
      { fOk = FALSE;   // <- debug-friendly .. set a breakpoint HERE !
      }
     
     // Read the data bytes from the I2C slave (up to nBytes=189 to read ALL REGISTERS in an Si5351):
     while( fOk && (nBytes>0) )
      { --nBytes;
        *(pbData++) = I2C_ReadByte((nBytes==0)?I2C_SEND_NACK:I2C_SEND_ACK);
        // > A master receiver must signal an end of data to the slave transmitter
        // > by NOT GENERATING an acknowledge on the last byte that has been
        // > clocked out of the slave. In this case the transmitter must leave
        // > the data line High to enable the master to generate the STOP condition.
      }
   } // end if < successfully selected the I2C-slave >
  // Finally send the STOP condition ("P") - even if the slave didn't react
  fOk &= I2C_Stop();  // -> TRUE when successful 
#elif( SWI_USE_HARDWARE_I2C )  
  I2C_ReadBlock( SI5351_I2C_7BIT_SLAVE_ADDRESS, bRegAddress, pbData, nBytes, 
# if( SWI_SI5351_SEND_FM_STREAM_IN_I2C_COMPLETION_CALLBACK )
                 Si5351_OnReadMultipleRegistersComplete );     
# else // ! SWI_SI5351_SEND_FM_STREAM_IN_I2C_COMPLETION_CALLBACK :
                 NULL/*OnCompletion-Callback*/);
# endif //  SWI_SI5351_SEND_FM_STREAM_IN_I2C_COMPLETION_CALLBACK ?
  fOk = I2C_WaitForCompletion();
  // Note: No incrementing of Si5351_bTotalNumErrors here - that's done on a
  //       'higher layer' only, to avoid incrementing the error counter twice
  //       when in fact there was only ONE problem with the I2C bus .
#endif // SWI_USE_SOFTWARE/HARDWARE_I2C ?

  return fOk;
} // end Si5351_ReadMultipleRegisters()

#if( SWI_SI5351_READ_BACK_REGISTER ) // READ BACK and VERIFY registers via I2C ?
//---------------------------------------------------------------------------
BOOL Si5351_VerifyRegisters(
       BYTE bRegAddress, // [in] e.g. Si5351_REG_MSYNTH_PLL_A . 
                         // May be 0xFF = Si5351_GET_REG_FROM_ADDRESS for
                         // auto-detection (if pbData points into Si5351_Regs)
       BYTE *pbData,     // [in] source for comparison; OFTEN (but not always) points
                         //      into Si5351Regs ['shadow registers' in RAM].
       BYTE nBytes)      // [in] number of bytes to compare.
  // Note:
  //  * Si5351_VerifyRegisters() will always WAIT FOR COMPLETION.
  //    Thus it must NOT (never, ever) be called from interrupt handlers !
{
  BYTE nBytesPerBlock, bByteIndex;
  BYTE b8ReadData[8];

  // If bRegAddress is Si5351_GET_REG_FROM_ADDRESS, determine the
  //    register number from pbData (address in RAM, points to Si5351Regs): 
  Si5351_GetRegNumberFromDataAddress( &bRegAddress, pbData ); 
   
  while( nBytes > 0 ) 
   { nBytesPerBlock = nBytes;
     if( nBytesPerBlock > 8 )  // need some time to breathe ?
      {  nBytesPerBlock = 8;
      }
     if( ! Si5351_ReadMultipleRegisters( bRegAddress, b8ReadData, nBytesPerBlock ) )
      { Si5351_bLastErrorCode  = Si5351_ERROR_I2C_FAIL;
        Si5351_bLastErrorParam = bRegAddress;
        I2C_IncErrorCounter( &Si5351_bTotalNumErrors );
        return FALSE;  // VERIFY failed because reading via I2C bus failed
      }
     for( bByteIndex=0; bByteIndex<nBytesPerBlock; ++bByteIndex)
      { if( b8ReadData[bByteIndex] != *(pbData++) ) 
         { Si5351_bLastErrorCode  = Si5351_ERROR_VERIFY;
           Si5351_bLastErrorParam = bRegAddress;
           I2C_IncErrorCounter( &Si5351_bTotalNumErrors );
           return FALSE; // VERIFY error : read-back register different from the EXPECTED one
         } // end if( b8ReadData[bByteIndex] != *pbData )
        ++bRegAddress;
      } // end < for all bytes in the current 8-byte-block >
     nBytes -= nBytesPerBlock;
   } // end while( nBytes > 0 )
  return TRUE;  
} // end Si5351_VerifyRegisters()
#endif // SWI_SI5351_READ_BACK_REGISTER ?


//------------------------------------------------------------------------
void Si5351_ClearErrors(void)  // Clears Si5351_bTotalNumErrors, etc (?)
{ Si5351_bTotalNumErrors = Si5351_bLastErrorCode = Si5351_bLastErrorParam = 0;
} // end Si5351_ClearErrors()

//------------------------------------------------------------------------
BOOL Si5351_SetPLLFeedbackRegisters(  // -> returns TRUE when OK, FALSE on any error
        BYTE bPLLBaseReg, // [in] Si5351_REG_MSYNTH_PLL_A or Si5351_REG_MSYNTH_PLL_B .
                          // May be 0xFF = Si5351_GET_REG_FROM_ADDRESS for
                          // auto-detection (if pRegs points into Si5351_Regs)
        DWORD dwA,        // [in] integer part 'a' in 'a + b/c', a = 15 .. 90
        DWORD dwB,        // [in] numerator    'b' in 'a + b/c', b = 0..1048575 (2^20-1)
        DWORD dwC,        // [in] denominator  'c' in 'a + b/c', c = 1..1048575 (2^20-1)
        T_Si5351_FeedbackDividerRegs *pRegs) // [out] latch for 8 registers   
   // Splits up integer part 'a', numerator 'b', and denominator 'c'
   // into bitgroup values 'P1'..'P3' in an Si5351, and writes those values
   // into eight adjacent registers of a *PLL FEEDBACK DIVIDER* via I2C bus .
   //
   // Basically, a + b/c  ranges from 15 to 90.
   // With b = 0 ("integer 'a' only), there MAY be less jitter / phase noise
   //        on the VCO signal (and thus on the final output on CLK0/1/2)
   //        but so far nobody actually came up with in-depth measurements.
   // 
   // Note: We use a similar terminology as in AN619, page 3, 
   // > 3.2 "Feedback Multisynth Divider Equations"
   // > -------------------------------------------
   // > .. the two VCO frequencies (..) can be generated using the following
   // > equation. Each feedback divider essentially multiplies(!) the source
   // > frequency such that
   // >
   // >    f_vco = f_xtal * ( a + b/c )
   // > 
   // >   and/or
   // > 
   // >    f_vco = (f_clkin / CLKIN_DIV) * (a + b/c)
   // > 
   // The conversion of the fractional parameter a,b,c
   // into "register value" for the various bitgroups
   //      'P1' (18 bits),  'P2' (20 bits),
   //  and 'P3' (20 bits, more or less equals 'c') 
   // is a little bizarre, and quoted further below.
{
  T_Si5351_FractionalDividerParamsP1P2P3  p1p2p3; // <- struct containing...
      // PLL configuration parameter 'P1': 18 bits spread across bitgroups in multiple registers
      // PLL configuration parameter 'P2': 20 bits spread across bitgroups in multiple registers
      // PLL configuration parameter 'P3': 20 bits spread across bitgroups in multiple registers
  DWORD dwFloorTerm; // result from "the strange Floor() term" - details below.

  Si5351_InitStruct_FractionalDividerParamsP1P2P3( &p1p2p3 );
  
  // If bRegAddress is Si5351_GET_REG_FROM_ADDRESS, determine the
  //    register number from pbData (address in RAM, points to Si5351Regs): 
  Si5351_GetRegNumberFromDataAddress( &bPLLBaseReg, (BYTE*)pRegs );   


   // AN619 page 3 (continued):
   // > The fractional ratio        a + b/c
   // > has a valid range of 15 + 0/1,048,575 and 90 
   // > and is represented in the Si5351 register space 
   // > using the equations below.
   // > MSNx_P1[bits 17..0] = 128 * a + Floor( 128 * b/c ) - 512
   // > MSNx_P2[bits 19..0] = 128 * b - c * Floor( 128 * b/c )
   // > MSNx_P3[bits 19..0] = c
   // 
   // The math 'Floor' function truncates to the nearest integer. It is
   // absolutely unnecessary here. We don't even need float or double.
   // 32-bit INTEGER arithmetic is sufficient because all 'inputs'
   // (dwA, dwB, dwC) fit in 20 bit; enough headroom to multiply by 128 :
   dwFloorTerm = ( 128UL * dwB ) / dwC;      // <- theoretic maximum is 127 because b<c !
   p1p2p3.p1 = 128UL * dwA + dwFloorTerm/*0..127*/ - 512UL; // <- mainly the INTEGER part, but with a tiny part of the FRACTION !
   p1p2p3.p2 = 128UL * dwB - dwC * dwFloorTerm/*0..127*/;   // <- mainly the NUMERATOR ?
        // ,-----------|_______________|
        // '---> This is strange, but for some reason it works.
        //       Got to understand why, exactly, for the frequency modulation.
   
   p1p2p3.p3 = dwC;  // <- 'P3' is just the denominator
      // Note for frequency modulation via the PLL-FEEDBACK-divider: 
      //  Unfortunately, the numerator (b) not only affects bitgroup "P2"
      //  but also bitgroup "P1"; and the denominator (c) affects ALL three.
      //  The dreadful 'floor term' only disappears if b < (c/128);
      //  but that means the fractional part is almost zero; so WITH FAST
      // FREQUENCY MODULATION, the precise wanted carrier frequency can only
      // be realized with a FRACTIONAL OUTPUT DIVIDER . 
      //

   // Split the PLL' fractional divider parameters (AN619 : "P1".."P3")
   // into various bitgroups for EIGHT adjacent PLL-Feedback-registers 
   // and write them into the Si5351 via I2C : 
   return Si5351_SetPLLFeedbackRegsP1P2P3(bPLLBaseReg, &p1p2p3, pRegs);
 
} // end Si5351_SetPLLFeedbackRegisters()

//------------------------------------------------------------------------
BOOL Si5351_SetPLLFeedbackRegsP1P2P3(  // -> returns TRUE when OK, FALSE on any error
        BYTE bPLLBaseReg, // [in] Si5351_REG_MSYNTH_PLL_A or Si5351_REG_MSYNTH_PLL_B .
                          // May be 0xFF = Si5351_GET_REG_FROM_ADDRESS for
                          // auto-detection (if pRegs points into Si5351_Regs)
        T_Si5351_FractionalDividerParamsP1P2P3 *pPLLDivider, // [in] integer- or fractional divider params FOR THE PLL (VCO "A" or "B"),
        T_Si5351_FeedbackDividerRegs *pRegs) // [out] latch for 8 registers   
   // Splits up the 3 bitgroup values 'P1'..'P3' into Si5351 registers,
   // and writes those values into eight adjacent registers of a 
   // *PLL FEEDBACK DIVIDER* via I2C bus .
   // Called from Si5351_SetPLLFeedbackRegisters() after converting 
   //    'a','b', and 'c' (fractional divider params a+b/c) into "P1".."P3".
{

  // If bRegAddress is Si5351_GET_REG_FROM_ADDRESS, determine the
  //    register number from pbData (address in RAM, points to Si5351Regs): 
  Si5351_GetRegNumberFromDataAddress( &bPLLBaseReg, (BYTE*)pRegs );   


  // Split the PLL' fractional divider parameters (AN619 : "P1".."P3")
  // into various bitgroups for EIGHT adjacent PLL-Feedback-registers :
  pRegs->b.P3_M = (BYTE)(pPLLDivider->p3 >> 8);  // b15..8 of 20-bit denominator
  pRegs->b.P3_L = (BYTE)(pPLLDivider->p3 >> 0);  // b7..0 of 20-bit denominator
  pRegs->b.P1_H = (BYTE)(pPLLDivider->p1 >> 16); // b17..16 of 18-bit integer part
  pRegs->b.P1_M = (BYTE)(pPLLDivider->p1 >> 8);  // b15..8 of 18-bit integer part
  pRegs->b.P1_L = (BYTE)(pPLLDivider->p1 >> 0);  // b7..0 of 18-bit integer part
  pRegs->b.P32_H // Upper bits from "P3" AND "P2" ...
     = ((pPLLDivider->p3 & 0x000F0000) >> 12)  // b7..4 = bits 19..16 of the 20-bit denominator
     | ((pPLLDivider->p2 & 0x000F0000) >> 16); // b3..0 = bits 19..16 of the 20-bit numerator
  pRegs->b.P2_M = (BYTE)(pPLLDivider->p2 >> 8); // b15..8 of 20-bit numerator
  pRegs->b.P2_L = (BYTE)(pPLLDivider->p2 >> 0); // b7..0 of 20-bit numerator

  return Si5351_WriteMultipleRegisters( bPLLBaseReg, pRegs->bArray, 8/*nBytes*/ );
 
} // end Si5351_SetPLLFeedbackRegsP1P2P3()


//------------------------------------------------------------------------
BOOL Si5351_SetOutputDividerRegisters( // -> returns TRUE when OK, FALSE on any error
        BYTE bOutputBaseReg, // [in] Si5351_REG_MSYNTH0 (42), 
                             //      Si5351_REG_MSYNTH1 (50), etc.
                             // May be 0xFF = Si5351_GET_REG_FROM_ADDRESS for
                             // auto-detection (if pRegs points into Si5351_Regs)
           // For Si5351A (3rd output), Si5351_REG_MSYNTH2 (58) is the maximum.
        T_Si5351_FractionalDividerParams *pParams, // [in] the following parameters:
           // [in] DWORD pParams->a : INTEGER part (optional fractional part further below).
           //      For outputs equipped with a FRACTIONAL OUTPUT DIVIDER,
           //      this parameter ranges from 8 to 2048 (AN619 page 6).
           //      For output frequencies above 150 MHz, INTEGER dividers
           //      available (see AN619 page 6, especially chapter 4.1.3: "DIVBY4").
           // [in] DWORD pParams->b : numerator    'b' in 'a + b/c', b = 0..1048575 (2^20-1) ?
           // [in] DWORD pParams->c : denominator  'c' in 'a + b/c', c = 1..1048575 (2^20-1) ?
           //      For "lower jitter" INTEGER output dividers, use b=0 and c=1 !
           // [in] DWORD pParams->r_div : "R Divider" ratio for f_out < 500 kHz, only 1,2,4,8,..128 !
        T_Si5351_OutputDividerRegs *pRegs) // [out] latch for 8 registers   
  // Sets an INTEGER OUTPUT DIVIDER (the thing called "Synthesis state 2"
  //                  in AN619 figure 1 "Generalized Si5351 Block Diagram"),
  // with up to EIGHT instances called "Multi Synth 0" to "Multi Synth 7",
  // each feeding its "Output Stage" called "R0" to "R7" widh additional
  // "R Dividers" for an extra divide-by-power-of-two (1,2,..,128) .
  // The Si5351A has THREE of these output dividers (because it has
  // THREE OUTPUTS), but there are only TWO PLLS ("PLLA" and "PLLB").
  //   Note: AN619 uses 'a + b/c' for the PLL FEEDBACK DIVIDERS
  //         as well as for the FRACTIONAL OUTPUT DIVIDERS.
  //         Other authors use 'd + e/f' for the fractional OUTPUT dividers
  //         to avoid confusion when putting all into a single formula
  //         for the Si5351's OUTPUT FREQUENCY:
  //                               a + b/c  (PLL FEEDBACK)
  //          f_out = f_ref * -----------------------------------------
  //                             d + e/f  (OUTPUT DIVIDER) * RDivider
  // 
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  //   These OUTPUT DIVIDERS must divide the frequency of an internal VCO
  //   (f_vco, ranging from 600 MHz to 900 MHz) down to an OUTPUT FREQUENCY
  //   (f_out, ranging from 2.5 kHz to 200 MHz).
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // 
  // > 4.1.2.1. Integer Divide Values(MSx_INT)
  // > If any of the MS0-MS5 is an even integer, Multisynth integer mode 
  // > may be enabled by setting MSx_INT=1 (see registers 16-21, bit 6). 
  // > In most cases setting this bit will improve jitter when using 
  // > even integer divide values.
  // "In most cases ?" - Na toll. Etwas genauer bitte !
  //           Bei Ausgangsfrequenzen ber 112.5 MHz MUSS "MSx_INT"
  //           gesetzt sein, bei Frequenzen ber 150 MHz darberhinaus
  //           auch noch die Bitgruppe "MSx_DIVBY4". 
  //           Beide befinden sich nervttenderweise in vllig unterschiedlichen
  //           Registergruppen (P1P2P3 vs. "Clock Control"). Suche nach..
  //           > bClockCtrlRegValue |= Si5351_MSK_CLK_CTRL_MS_INT; 
  // 
  // > 4.1.3. Output Multisynth Divider Equations (150 MHz <Fout<=200 MHz)
  // > Output frequencies greater than 150 MHz are available 
  // > on Multisynths 0-5. For this frequency range a divide value
  // > of 4 must be used by setting ..
  // >  MSx_P1=0, PSx_P2=0, MSx_P3=1, MSx_INT=1, MSx_DIVBY4=0b11 .
  //
  // Call Stack : 
  //    Si5351_SetFrequency() or Si5351_RWAccessForMenuItems() 
  //      -> Si5351_SetOutputDividerRegisters()
  //        -> Si5351_SetOutputDividerRegsP1P2P3()
{
  T_Si5351_FractionalDividerParamsP1P2P3 p1p2p3; // <- struct containing ...
             // PLL configuration parameter 'P1': 18 bit integer part, 'encoded'
             // PLL configuration parameter 'P2': 20 bit numerator, 'encoded'
             // PLL configuration parameter 'P3': 20 bit denominator, 'encoded'
  DWORD dwFloorTerm; // result from "the strange Floor() term" - see Si5351_SetPLLFeedbackRegisters()
  BYTE  bRDivRegValue; // value for the OUTPUT-DIVIDER specific "R-divider" (what a name..)

  // AN619 page 6 (continued):
  // > 4.1.2. Output Multisynth Divider Equations (Fout <= 150 MHz)
  // > Once the PLL source for the output Multisynth is selected,
  // > the divide ratio can be set using the equations below.
  // > Divider represented as fractional number,
  // >         a +  b / c      (same principle as for the PLL FEEDBACK - WB ...)
  // > between 8 + 1/1048575 and 2048.  (...but these limits are very different)
  // >
  // > MSx_P1[bits 17..0]  = 128 * a + Floor( 128 * b/c ) - 512
  // > MSx_P2[bits 19..0]  = 128 * b - c * Floor( 128 * b/c )
  // > MSx_P3[bits 19..0]  = c  ( the denominator )
  //   ( '--<-- x=0,1,2 for Si5351A, as in 'CLKx' )
  // >  ...
  // > 4.1.2.1. Integer Divide Values(MSx_INT)
  // > If any of the MS0-MS5 is an even integer, Multisynth integer mode 
  // > may be enabled by setting MSx_INT=1 (see registers 16-21, bit 6). 
  // > In most cases setting this bit will improve jitter when using even 
  // > integer divide values.
  // > Multisynths 6 and 7 inherently operate in integer mode, and so 
  // > there is no register to turn integer mode on or off.
  // > However, it's important to note that Multisynth integer mode 
  // > cannot be used when adding phase offsets in NVM. (WB: Non-Volatile Memory???)
  // > In other words, MSx_INT needs to be set to 0 if phase offsets
  // > need to be enabled.
  // WB: Hmmm... so not only INTEGER but also an EVEN divisor
  //     in the interest of low jitter / phase noise ! 
  //     If dwA and dwB permit, Si5351_SetFrequency() will set 'MSx_INT'
  //     when writing 'bClockCtrlRegValue' into 'bClockCtrlRegAddr' .
  //
  // ex: dwFloorTerm = (DWORD)( 128.0 * (double)dwB / (double)dwC ); 
  dwFloorTerm = ( 128UL * pParams->b ) / pParams->c;         // <- theoretic maximum is 127
  p1p2p3.p1 = 128UL * pParams->a + dwFloorTerm/*0..127*/ - 512UL; // <- mainly the INTEGER part, but with a tiny part of the FRACTION !
  p1p2p3.p2 = 128UL * pParams->b - pParams->c * dwFloorTerm/*0..127*/;   // <- mainly the NUMERATOR
  p1p2p3.p3 = pParams->c;  // exactly the DENOMINATOR .. but keep this as-is for clarity

  
  
  // Convert the "R-Divider" - RATIO (1,2,4,8,..128) into the
  //  3-bit "register value" (the latter is actually two's logarithm of the ratio)
  bRDivRegValue = 0;  // 2^0 = 1 (the lowest possible "R divider" ratio)
  while( (bRDivRegValue<7) && ( (1<<bRDivRegValue) < pParams->r_div ) )
   { ++bRDivRegValue;
   }
  p1p2p3.r_div = bRDivRegValue; // .. for completeness, also in the struct
  
  pRegs->b.P1H_DIV // e.g. "Register 44": VARIOUS BITGROUPS ... AN619 page 35:
    = ((BYTE)(bRDivRegValue<<4) & Si5351_MSK_OUTPUT_MSYNTH_P1H_DIV_R0)   // b6..4 = R0 Output Divider (2^0 .. 2^3 to divide by 1,2,4,8,16,32,64,128)
          | (pRegs->b.P1H_DIV & (~Si5351_MSK_OUTPUT_MSYNTH_P1H_DIV_R0)); // leave all other bits UNMODIFIED
  
  if( (pParams->a==4) && (pParams->b==0) ) // exactly divide by FOUR ?
   { // This is important for OUTPUT FREQUENCIES above 150 MHz. AN619 page 6:
     // > 4.1.3. Output Multisynth Divider Equations (150 MHz <Fout<=200 MHz)
     // > Output frequencies greater than 150 MHz are available on Multisynths 0-5.
     // > For this frequency range a divide value of 4 must be used by setting
     // >   (1)  MSx_P1=0,
     // >   (2)  MSx_P2=0,
     // >   (3)  MSx_P3=1,
     // >   (4)  MSx_INT=1, and
     // >   (5)  MSx_DIVBY4[1:0]=11b.
     // > Set the appropriate feedback Multisynth to generate f VCO =Fout*4.
     p1p2p3.p1 = 0;    // (1)
     p1p2p3.p2 = 0;    // (2)
     p1p2p3.p3 = 1;    // (3)
     pRegs->b.P1H_DIV |= Si5351_MSK_OUTPUT_MSYNTH_P1H_DIV_BY4; // (5) b3..2 = "MS0 Divide by 4 Enable" : 11=div by 4,  00=div by anything else     
          // NOT HERE: // (4)  "MSx_INT" is in register Si5351_REG_CLKx_CTRL, bitmask Si5351_MSK_CLK_CTRL_MS_INT .
          // This bit in one of the "Clock Control"-registers is set in Si5351_SetFrequency(),
          // depending on output_divider_params.a and output_divider_params.b !
          // Search for places like ...
          //   > bClockCtrlRegValue |= Si5351_MSK_CLK_CTRL_MS_INT;
          // or (if the OUTPUT DIVIDER shall use FRACTIONAL mode) : 
          //   > bClockCtrlRegValue &= ~Si5351_MSK_CLK_CTRL_MS_INT;
          //        
          // At least the Clkctrl0 register is shown in BINARY form
          // in the 'Synth Registers' menu, e.g:
          // > "R16:Clk0ct 01001111"
          //               ||||||\|__ bits 1..0 = "CLK0_IDRV": 11 = 8 mA output drive strength
          //               ||||\|____ bits 3..2 = "CLK0_SRC" : 11 = "MultiSynth0 is the source for CLK0"
          //               ||||______ bit  4 = "CLK0_INV" : 0 = "Output Clock 0 is not inverted"
          //               |||_______ bit  5 = "MS0_SRC"  : 0 = "Select PLLA as the source for MultiSynth0"
          //               ||________ bit  6 = "MS0_INT"  : 1 = "MS0 operates in integer mode"
          //               |_________ bit  7 = "CLK0_PDN" : 0 = "CLK0 is powered up"
   } // exactly divide by FOUR ?
  else 
   { pRegs->b.P1H_DIV &= (~Si5351_MSK_OUTPUT_MSYNTH_P1H_DIV_BY4); // turn "MS0 Divide by 4 Enable" *off* (to divide by "anything")
   }  

  return Si5351_SetOutputDividerRegsP1P2P3( bOutputBaseReg, &p1p2p3, pRegs );
  
} // end Si5351_SetOutputDividerRegisters()


//------------------------------------------------------------------------
BOOL Si5351_SetOutputDividerRegsP1P2P3(BYTE bOutputBaseReg,
        T_Si5351_FractionalDividerParamsP1P2P3 *p1p2p3, // [in] integer- or fractional divider params p1, p2, p3 FOR A CLOCK OUTPUT ("clk0".."clk2")
        T_Si5351_OutputDividerRegs *pRegs)   // [out]
{
  // Similar as for the *PLL FEEDBACK* divider in Si5351_SetPLLFeedbackRegisters(),
  // even if USUALLY NOT FRACTIONAL, split the 3 "fractional" divider parameters
  // (plus the "R-Divider"-register-value) into various bitgroups in EIGHT(!)
  // adjacent registers, using the register-address-OFFSETs defined somewhere else:

  // Each of these "OUTPUT Multisynths" also occupies EIGHT registers 
  // with the ADDRESS OFFSETS like Si5351_OFS_MSYNTH0_P3_M (0),
  // Si5351_OFS_MSYNTH0_P3_L (1), Si5351_OFS_MSYNTH0_P1H_DIV (2),
  // .. Si5351_OFS_MSYNTH0_P2_L (7). 
  // So again, similar as in Si5351_SetPLLFeedbackRegisters(),
  // here's another bit-fiddling orgy to spread P1, P2, P3
  // over EIGHT ADJACENT REGISTERS for a single "output Multisynth": 
  pRegs->b.P3_M = (BYTE)(p1p2p3->p3 >> 8); // e.g. "Register 42": Bits 15..8 of 20-bit denominator 'P3'
  pRegs->b.P3_L = (BYTE)(p1p2p3->p3 >> 0); // e.g. "Register 43": Bits 7..0 of 20-bit denominator 'P3'
  
  pRegs->b.P1H_DIV // e.g. "Register 44": VARIOUS BITGROUPS ... AN619 page 35:
    = ((BYTE)(p1p2p3->p1 >> 16) & Si5351_MSK_OUTPUT_MSYNTH_P1H_DIV_P1H) // b1..0 = bits 17..16 of 18-bit integer part 'P1'
    | ((BYTE)(pRegs->b.P1H_DIV &(~Si5351_MSK_OUTPUT_MSYNTH_P1H_DIV_P1H))); // leave all other bits UNMODIFIED
  pRegs->b.P1_M = (BYTE)(p1p2p3->p1 >> 8); // e.g. "Register 45": Bits 15..8 of 18-bit integer part 'P1'
  pRegs->b.P1_L = (BYTE)(p1p2p3->p1 >> 0); // e.g. "Register 46": Bits 7..0 of 18-bit integer part 'P1'
     
  pRegs->b.P32_H // e.g.  "Register 47": Upper bits from "P3" AND "P2" ...
      = ((p1p2p3->p3 & 0x000F0000UL) >> 12)  // b7..4 = bits 19..16 of 20-bit DENOMINATOR 'P3'
      | ((p1p2p3->p2 & 0x000F0000UL) >> 16); // b3..0 = bits 19..16 of 20-bit NUMERATOR 'P2'
  pRegs->b.P2_M = (BYTE)(p1p2p3->p2 >> 8); // e.g. "Register 48": Bits 15..8 of 20-bit numerator 'P2'
  pRegs->b.P2_L = (BYTE)(p1p2p3->p2 >> 0); // e.g. "Register 49": Bits 7..0 of 20-bit numerator 'P2'

  // Also here, care for the annoying 'special cases' above 112.5 and 150 MHz
  //       ("no FRACTIONAL but only INTEGER output dividers" or even 
  //        "only divide-by-four, with a SPECIAL output divider") .
  if( (p1p2p3->p1==0) && (p1p2p3->p2==0) ) // hey, it's the crazy "divide-by-FOUR" case !
   { pRegs->b.P1H_DIV |= Si5351_MSK_OUTPUT_MSYNTH_P1H_DIV_BY4; // b3..2 = "MS0 Divide by 4 Enable" : 11=div by 4,  00=div by anything else
     // .. and again, this is NOT THE ONLY annoying 'special case' ! 
     //    Besides this "DIVBY4"-thing, there is this "MSx_INT"-thing,
     //    ("integer divide mode", mandatory above 112.5 MHz) - look for places
     //    that SET or CLEAR Si5351_MSK_CLK_CTRL_MS_INT in a 'clock control' register.
   }
  else // do NOT divide by (exactly) FOUR but by "anything else" ->
   { pRegs->b.P1H_DIV &= (~Si5351_MSK_OUTPUT_MSYNTH_P1H_DIV_BY4); // turn "MS0 Divide by 4 Enable" *off* (to divide by "anything") 
   }
  
  return Si5351_WriteMultipleRegisters( bOutputBaseReg, pRegs->bArray, 8/*nBytes*/ );

} // end Si5351_SetOutputDividerRegsP1P2P3()


//------------------------------------------------------------------------
void Si5351_ConvertP1P2P3ToFractionalDividerParamsABC(
        T_Si5351_FractionalDividerParamsP1P2P3 *p1p2p3, // [in] integer- or fractional divider params p1, p2, p3 (18..20 bit each)
        long *pi32A, // [out] 'A' in fractional division ratio 'A+B/C'
        long *pi32B, // [out] 'B' in fractional division ratio 'A+B/C'
        long *pi32C) // [out] 'C' in fractional division ratio 'A+B/C'
{
   // What follows is the INVERSE to  AN619 page 3 :
   // > The fractional ratio        a + b/c
   // > has a valid range of 15 + 0/1,048,575 and 90 
   // > and is represented in the Si5351 register space 
   // > using the equations below.
   // > P1[bits 17..0] = 128 * a + Floor( 128 * b/c ) - 512
   // > P2[bits 19..0] = 128 * b - c * Floor( 128 * b/c )
   // > P3[bits 19..0] = c             |________________|
   //                  ,----------------' 
   // The mysterious 'floor term' sits in the lower 7 bits of P1, range 0..127:
   DWORD dwFloorTerm = p1p2p3->p1 & 0x0000007F;
   *pi32C = (long)p1p2p3->p3;   // 'P3' is simply the DENOMINATOR 'C' 
   //           P1 = 128 * a + Floor( 128 * b/c ) - 512
   //    ->     a * 128 = P1 - dwFloorTerm + 512  -> 
   *pi32A = (long)( (p1p2p3->p1 + 512 - dwFloorTerm) >> 7 );
   //           P2 = 128 * b - c * Floor( 128 * b/c )
   //    ->     b * 128 = P2 + c * dwFloorTerm = P2 + P3 * dwFloorTerm -> 
   *pi32B = (long)( (p1p2p3->p2 + p1p2p3->p3 * dwFloorTerm) >> 7 );
   // If all went well, the output in *pdwA, *pdwB, *pdwC is the same as
   // the INPUT parameters dwA, dwB, dwC in e.g. Si5351_SetPLLFeedbackRegisters().
} // end Si5351_ConvertP1P2P3ToFractionalDividerParamsABC()
   

//------------------------------------------------------------------------
void Si5351_GetP1P2P3FromPLLFeedbackDivider(
       T_Si5351_FeedbackDividerRegs *pRegs, // [in] values in e.g. registers 26..33 for the 1st PLL
       T_Si5351_FractionalDividerParamsP1P2P3 *p1p2p3 ) // [out] fractional divider params "P1".."P3"
   // Kind of 'inverse' to Si5351_SetPLLFeedbackRegisters() .
   // First used for troubleshooting the experimental FREQUENCY MODULATION.
   // The results can be inspected (displayed) in the 'HW Diagnostics' menu.
{
   // Re-combine the PLL's fractional divider parameters (AN619 : "P1".."P3")
   // from bitgroups in EIGHT adjacent PLL-Feedback-registers (e.g. #26..#33):
   p1p2p3->p3 = 
      (DWORD)((unsigned int)pRegs->b.P3_L << (unsigned)0)  // b7..0 of 20-bit denominator
    | (DWORD)((unsigned int)pRegs->b.P3_M << (unsigned)8)  // b15..8 of 20-bit denominator
    | (DWORD)(((unsigned long)pRegs->b.P32_H & 0x0F0UL) << (unsigned)12); // b7..4 -> bits 19..16 of 20-bit denominator
   p1p2p3->p2 = 
      (DWORD)((unsigned int)pRegs->b.P2_L << (unsigned)0)  // b7..0 of 20-bit numerator
    | (DWORD)((unsigned int)pRegs->b.P2_M << (unsigned)8)  // b15..8 of 20-bit numerator
    | (DWORD)(((unsigned long)pRegs->b.P32_H & 0x0FUL) << (unsigned)16); // b3..0 -> bits 19..16 of 20-bit numerator
       // (without this megaton of casts to UNSIGNED, there was a stupid SIGN EXPANSION,
       //  resulting in *pdwP2 = 0xFFFFA3D0 instead of 0x0000A3D0 .
  p1p2p3->p1 = 
      (DWORD)((unsigned int)pRegs->b.P1_L << (unsigned)0)  // b7..0 of 18-bit integer part
    | (DWORD)((unsigned int)pRegs->b.P1_M << (unsigned)8)  // b15..8 of 18-bit integer part
    | (DWORD)(((unsigned long)pRegs->b.P1_H & 3UL) << (unsigned)16); // b17..16 of 18-bit integer part
  
  p1p2p3->r_div = 1; // no "R divider" here.. if there was, it would divide by one
   
} // end Si5351_GetP1P2P3FromPLLFeedbackDivider()

//------------------------------------------------------------------------
void Si5351_GetP1P2P3FromOutputDivider(
       T_Si5351_OutputDividerRegs *pRegs, // [in] values in e.g. registers 42..49 for output "CLK0"
       T_Si5351_FractionalDividerParamsP1P2P3 *p1p2p3 ) // [out] fractional divider params "P1".."P3"
   // Kind of 'inverse' to Si5351_SetOutputDividerRegisters() .
   // First used for troubleshooting the experimental FREQUENCY MODULATION.
   // The results can be inspected (displayed) in the 'HW Diagnostics' menu.
{
   // On first glance, a 'Multisynth' for an OUTPUT DIVIDER has the same
   // structure as a 'Multisynth' for the PLL FEEDBACK "divider", 
   // but in fact, there's more to an "Output Multisynth",
   // thus the 1st argument (pRegs) is A DIFFERENT STRUCTURE !
   p1p2p3->p3 =
      (DWORD)((unsigned int)pRegs->b.P3_L << (unsigned)0)  // b7..0 of 20-bit denominator
    | (DWORD)((unsigned int)pRegs->b.P3_M << (unsigned)8)  // b15..8 of 20-bit denominator
    | (DWORD)(((unsigned long)pRegs->b.P32_H & 0x0F0UL) << (unsigned)12); // b7..4 -> bits 19..16 of 20-bit denominator
   p1p2p3->p2 =
      (DWORD)((unsigned int)pRegs->b.P2_L << (unsigned)0)  // b7..0 of 20-bit numerator
    | (DWORD)((unsigned int)pRegs->b.P2_M << (unsigned)8)  // b15..8 of 20-bit numerator
    | (DWORD)(((unsigned long)pRegs->b.P32_H & 0x0FUL) << (unsigned)16); // b3..0 -> bits 19..16 of 20-bit numerator
      // (without this megaton of casts to UNSIGNED, there was a stupid SIGN EXPANSION,
      //  resulting in *pdwP2 = 0xFFFFA3D0 instead of 0x0000A3D0 .
   p1p2p3->p1 =
      (DWORD)((unsigned int)pRegs->b.P1_L << (unsigned)0)  // b7..0 of 18-bit integer part
    | (DWORD)((unsigned int)pRegs->b.P1_M << (unsigned)8)  // b15..8 of 18-bit integer part
    | (DWORD)(((unsigned long)pRegs->b.P1H_DIV & 3UL) << (unsigned)16); // b17..16 of 18-bit integer part
   
   // Because the funny-named "R"-divider is an important feature of each 
   // 'output' divider, T_Si5351_FractionalDividerParamsP1P2P3 has the
   // ".r_div" component. Get THAT important parameter from the registers, too:
   // pRegs->b.P1H_DIV (e.g. Register 44),
   //       bits 6..4 : R0 Output Divider (div-by-1,2,4,8,16,32,64,128),
   //       bits 3..2 : "MS0 Divide by 4 Enable". 11=div by 4,  00=div by anything else,
   //       bits 1..0 = bits 17..16 of 18-bit integer part of an output MultiSynth divider   
   p1p2p3->r_div = 1 << ( (pRegs->b.P1H_DIV >> 4) & 7); // real DIVIDER RATIO, no bloody 3-bit value
   
} // end Si5351_GetParamsFromOutputDivider()
        

//------------------------------------------------------------------------
void Si5351_CalcVcoFreqFromP1P2P3(
        T_Si5351_FractionalDividerParamsP1P2P3 *p1p2p3, // [in] 'PLL feedback multisynth' register params
        double *pdbVcoFreq_Hz ) // [out] "theoretic" internal VFO frequency.
   // [in] g_dblReferenceFrequency_Hz : reference frequency fed into the Si5351
   // Uses this gloriously simple formula, derived as explained in 
   //    dsPIC33EP_Si5351_Synth/Documentation/Si5351_FM_Tests_2021_11.txt :
   // --------------------------------------------------
   //    f_vco = f_ref * ( 512 + P1 + P2/P3 ) / 128
   // --------------------------------------------------
{
  if( g_dblReferenceFrequency_Hz <= 0.0 ) // Init this if the application didn't..
   {  g_dblReferenceFrequency_Hz = (double)SWI_SI5351_REFERENCE_FREQUENCY_HZ;
      // SWI_SI5351_REFERENCE_FREQUENCY_HZ is just a default for this variable.
   }     
  if( p1p2p3->p3 == 0 )
   {  p1p2p3->p3 = 1;   // avoid div-by-zero
   }
  *pdbVcoFreq_Hz = g_dblReferenceFrequency_Hz * (1.0 / 128.0)
                 * ( (double)( 512 + p1p2p3->p1 ) + (double)p1p2p3->p2 / (double)p1p2p3->p3 );

} // end Si5351_CalcVcoFreqFromP1P2P3()


//------------------------------------------------------------------------
void Si5351_CalcOutputFreqFromVcoAndP1P2P3( 
        double *pdbVcoFreq_Hz,     // [in] VCO(!) frequency
        T_Si5351_FractionalDividerParamsP1P2P3 *p1p2p3, // [in] 'output multisynth' register params (fractional divider)
        double *pdbOutputFreq_Hz ) // [out] "theoretic" output frequency.
   // Calculates    f_out = f_vco * 128 / ( 512 + p1 + p2/p3 )
   // here:          |       |
   //     *pdbOutputFreq_Hz  *pdbVcoFreq_Hz
   // Used to find the 'best' method for frequency modulation
   //      AND for the 'Synth Registers' menu (-> Si5351_RWAccessForMenuItems) .
{
  if( p1p2p3->p3 == 0 )
   {  p1p2p3->p3 = 1; // avoid div-by-zero (affecting the caller's variable)
   }
  *pdbOutputFreq_Hz = *pdbVcoFreq_Hz * 128.0  / ( (double)( 512 + p1p2p3->p1 )
                         + (double)p1p2p3->p2 / (double)p1p2p3->p3 );
} // end Si5351_CalcOutputFreqFromVcoAndP1P2P3()


#if(0) // API to set the Si5351's "phase offset" registers ? Useless here.
//------------------------------------------------------------------------
BOOL Si5351_SetVcoPhaseOffset(  // See AN619 pages 10 and 58 !
        int iOutputIndex, // [in] zero-based index for the PLL *and* the output divider
        BYTE bOffset)     // [in] Offset for the *VCO FREQUENCY* (ca 900 MHz)
                          //      in 128 steps (0..127), with one step (LSbit)
                          //      representing ONE QUARTER of the VCO period !
   // > bOffset is an unsigned integer ranging from 0 to 127,
   // > with one LSB equivalent to a time delay of Tvco/4, where Tvco 
   // > is the period of the VCO/PLL associated with this output.
   // Hmm... let's get this straight. With Fvco in the 900 MHz range,
   // 127 steps a 1 / ( 4*Fvco ) = 127 * 0.277 ns = 35 ns, this is not really
   // suited for FREQUENCY-MODULATING one of the outputs via software.
{
  return Si5351_WriteRegister_8Bit(
           Si5351_REG_CLK0_PHASE_OFFSET+iOutputIndex, bOffset );
  // Only used for an I2C interface 'speed test', stupidly ramping up the phase
  // as fast at the interface permitted (using a dsPIC33EP's hardware I2C master).
} // end Si5351_SetVcoPhaseOffset()
#endif // (0)


#if( SWI_SI5351_SUPPORT_FREQUENCY_MODULATION ) 
//------------------------------------------------------------------------
int Si5351_CalcOutputDividerParamsForFM( // ... for a given INTEGER PLL feedback divider :
          int    iPLLDivider,  // [in] integer, even PLL FEEDBACK DIVIDER. f_vco = f_ref * iPLLDivider
          double dblWantedFrequency_Hz, // [in] "wanted" carrier frequency in Hertz, 64 bit float
          int    iFreqModDeviation_Hz,  // [in] wanted deviation (+/- N Hz around the carrier)
          T_Si5351_FractionalDividerParamsP1P2P3 *pOutDivider) // [out] integer- or fractional divider params (p1,p2,p3,r_div) FOR THE OUTPUT ("CLK0", "CLK1", "CLK2")
  // Subroutine for Si5351_GetPLLAndFractionalOutputDividerForFM().
  // Called *multiple times* from there, to find the *best* solution,
  // and (after picking the best) calculate all outputs for that solution again.
  // When successfull (for the given inputs), returns the FM DEVIATION 
  //                  (+/- X Hertz) realized with the input parameters. 
  // If the fractional part (p2/p3) doesn't give enough headroom for the
  // modulation, the return value may be LESS than the 'wanted' deviation.
  // Otherwise (completely unusable solution), returns ZERO .
  // 
  // Formula for the OUTPUT FREQUENCY of a fractional output divider
  // configured with the output parameters calculated by this subroutine:
  //    >  f_out = f_vco * 128 / ( 512 + p1 + (p2_offset+modulation)/p3 )
  //       where   modulation = 0..4095 (for audio samples from a 12-bit ADC).
  //               modulation = 2047    should give f_out as close as possible
  //                                    to dblWantedFrequency_Hz .

{
  long   p1, p2_offset, p3;
  double temp, f_out_min, f_out_max, f_out_center;
  double p2_div_p3_lo, p2_div_p3_hi;  // p2/p3 as a floating point number (not a fraction = Bruch yet)
  double f_vco = iPLLDivider * g_dblReferenceFrequency_Hz;
  int    realized_deviation;
  if( (f_vco < 599E6) || (f_vco > 902E6) ) // internal VCO frequency "out of spec" ?
   { return 0;  // let the caller (Si5351_GetPLLAndFractionalOutputDividerForFM) try something else
     // (allow 1 MHz tolerance here; just because the calibrated reference frequency
     //  is 25.0001 MHz instead of 25.0 MHz we don't want to reject iPLLDivider=36 and f_vco=900.00x MHz)
   }
  
  // The 'direct' formula for the Fractional Ouput Divider's (FOD's) output frequency,
  // 'directly' calculated from the bitgroups 'p1' ('encoded' integer part),
  // 'p2' ('encoded' numerator), and 'p3' (same as the denominator, 'c') is:
  // 
  //   f_out = f_vco * 128 / ( 512 + p1 + p2/p3 )      [Si5351_FM equation 1.2]
  //                                 |    |___|
  //                                 |      |
  //                      integer part      fractional part, 0.1 ... 0.9, ideally 0.5
  // 
  // For reasons explained in dsPIC33EP_Si5351_Synth/Documentation/Si5351_FM_Tests_2021_11.txt,
  // we're going to inject the MODULATING SIGNAL into the lower 12 bits 
  // of the fractional part's numerator (p2, a 20-bit value) .
  // The derivation of the following formulas from eqn 1.1 is in Si5351_FM_Tests_2021_11.txt:
  //       f_out = f_vco * 128 / ( 512 + p1 + p2/p3 )      [Si5351_FM equ. 1.2]
  //       p1 = int(f_vco * 128 / f_out - 512 - p2/p3)     [Si5351_FM equ. 1.3]
  //       p2/p3 = f_vco * 128 / f_out - 512 - p1          [Si5351_FM equ. 1.4]
  //       p2 = p3 * ( 128 * f_vco/f_out - 512 - p1 )      [Si5351_FM equ. 1.5]
  //   HERE: p2 = p2_offset (constant part) + modulation (0..4095 from a 12-bit ADC)
  //   p2/p3 = p2_div_p3 is the "modulated fractional part" for a FRACTIONAL OUTPUT DIVDER.
  //   In this case injecting the modulator into the NUMERATOR (p2),
  //           resulting in the following 'low' and 'high' edge values:
  //       p2_div_p3_lo = (p2_offset+0   ) / p3            [Si5351_FM equ. 5.1]
  //       p2_div_p3_hi = (p2_offset+4095) / p3            [Si5351_FM equ. 5.2]
  // 
  //     
  // Coarsely estimate the integer part 'p1' of the output divider,
  // assuming the fractional part (p2/p3) is somewhere between 0 and 0.9999 :
  temp = 128.0 * f_vco / dblWantedFrequency_Hz - 512.0;
  p1 = (long)temp;  // [Si5351_FM equation 1.3 without the fractional part]
  if( p1 > 262143 ) // oops.. 'p1' doesn't fit inside the Si5351 18-bit group !
   { return 0;
   }   

  // "23cm"-Example : dblWantedFrequency_Hz = 1298.350 MHz / 128 = 10143359.375
  //    f_vco = 24 * f_ref [25MHz] = 600 MHz
  //    -> p1 = int(128*f_vco/dblWantedFrequency_Hz-512) = 7059
  //    f_vco = 36 * f_ref [25MHz] = 900 MHz
  //    -> p1 = int(128*f_vco/dblWantedFrequency_Hz-512) = 10845
  
  // To reach the wanted FM deviation (+/- iFreqModDeviation_Hz), 
  // p2_div_p3 must be varied to give output frequencies from equation 1.1
  // in the following range:
  f_out_min = dblWantedFrequency_Hz - iFreqModDeviation_Hz;
  f_out_max = dblWantedFrequency_Hz + iFreqModDeviation_Hz;
  
  // Apply equation 1.3 again, (p2/p3 = f_vco * 128 / f_out - 512 - p1) for both min,max:
  p2_div_p3_lo = 128.0 * f_vco / f_out_max - 512.0 - p1;
  p2_div_p3_hi = 128.0 * f_vco / f_out_min - 512.0 - p1;
  
  // "23cm"-Example (shows why we need to get as close to p3_div_p3 = 0.5 sometimes):
  //  p2_div_p3_lo := 900MHz * 128 / f_out_max - 512 - p1  =: 0.15000
  //  p2_div_p3_hi := 900MHz * 128 / f_out_min - 512 - p1  =: 0.21945
  
  if( p2_div_p3_lo < 0.0 ) // cannot reach the wanted MAXIMUM(!) frequency !
   { temp = p2_div_p3_lo;
     p2_div_p3_lo = 0;     // reduce the MAXIMUM frequency (and inc the min value for p2/p3)
     p2_div_p3_hi += temp; // limit the other endstop by the same amount
   }
  if( p2_div_p3_hi > 1.0 ) // cannot reach the wanted MINMUM(!) frequency !
   { temp = p2_div_p3_hi - 1.0;
     p2_div_p3_hi = 1.0;   // increase the MINIMUM frequency (and dec the max value for p2/p3)
     p2_div_p3_lo += temp; // limit the other endstop by the same amount
   }
  
  // At this point, we only know the fractional part as p2/p3 between 0 and 0.999 .
  // Next: Find a suitable fraction from integers 'p2_offset' and 'p3' for a 
  //       "modulatable fractional part" with integers p2_offset and p3, so that..
  //    p2_div_p3_lo = (p2_offset+0   ) / p3                         (5.1)
  //    p2_div_p3_hi = (p2_offset+4095) / p3                         (5.2)
  //         ,--------------------|__|
  //         '---> 12-bit "Modulator" HERE, added to the NUMERATOR .
  // Again, the derivation of these formulas is in Si5351_FM_Tests_2021_11.txt !
  //    p2_offset = 4095 / ( (p2_div_p3_hi / p2_div_p3_lo) - 1 )     (5.3)
  //    p3        = p2_offset / p2_div_p3_lo                         (5.4)
  // Apply equations 5.3 and 5.4 :
  p2_offset = 4095 / ( (p2_div_p3_hi / p2_div_p3_lo) - 1 );
  p3        = p2_offset / p2_div_p3_lo;

  // "23cm"-Example (1st call in a loop, with f_vco=900 MHz):
  //  p1        = 10845
  //  p2_offset = 4095 / ( ( 0.2194/0.1500) - 1 ) = 8850 ( + 2047 = 10897 for CENTER frequency )
  //  p3        = 8850 / 0.1500                   = 58984

  // Pass the many results back in a struct : (ok for dsPIC, but not for PIC16 ! )
  pOutDivider->p1 = p1;
  pOutDivider->p2 = p2_offset; // here: LOWEST value for p2, with headroom to add the modulator (0..4095)
  pOutDivider->p3 = p3;
  pOutDivider->r_div = 1;    // FM for frequencies below 500 kHz ? Forget it.


  
  // As a sanity check, calculate the MIN- and MAX output frequency 
  // resulting from p1, p2_offset + 0, p2_offset+4095, and p3 again.
  // The DIFFERENCE between both frequencies is the (double) deviation,
  // the MEAN value is the 'unmodulated carrier' frequency.
  // Remember: f_out = f_vco * 128 / ( 512 + p1 + p2/p3 ),
  //           thus THE LOWER 'p2', the HIGHER f_out ! 
  Si5351_CalcOutputFreqFromVcoAndP1P2P3( &f_vco, pOutDivider, &f_out_max );
  pOutDivider->p2 = p2_offset + 4095;
  Si5351_CalcOutputFreqFromVcoAndP1P2P3( &f_vco, pOutDivider, &f_out_min );
  pOutDivider->p2 = p2_offset; // back to the LOWEST value for p2, with headroom to add the modulator (0..4095)
  
  temp = 0.5 * (f_out_max - f_out_min);  // -> deviation "to ONE side" (+/- x Hz)
  realized_deviation = (int)temp;
  f_out_center = 0.5 * (f_out_min + f_out_max); // -> "realized" center frequency

  // "23cm"-Example (1st call in a loop, with f_vco=900 MHz):
  //  f_out_max    = 1.0143390374556607E7
  //  f_out_min    = 1.0143328368971938E7
  //  f_out_center          = 1.0143359371764272E7
  //  dblWantedFrequency_Hz = 1.0143359375E7
  //  realized_deviation    = 31
  //  iFreqModDeviation_Hz  = 31 ("wanted" deviation in Hz)

  
  if(  (f_out_center < (dblWantedFrequency_Hz - 0.25 * temp) )
     ||(f_out_center > (dblWantedFrequency_Hz + 0.25 * temp) ) )
   { return 0;  // unacceptably "off-center", discard the result
   }
  if(  (realized_deviation < (iFreqModDeviation_Hz - iFreqModDeviation_Hz/4) )
     ||(realized_deviation > (iFreqModDeviation_Hz + iFreqModDeviation_Hz/4) ) )
   { return 0;  // unacceptably "wide" or "narrow" modulation, discard the result
   }
  
  return realized_deviation; // ... in Hertz, acceptably close to the 'wanted' FM-deviation.
  // (since Si5351_GetPLLAndFractionalOutputDividerForFM() calls this
  //  function in a loop, for all possible EVEN INTEGER PLL FEEDBACK DIVIDERS,
  //  chances are good to find a 'realized deviation' that is really close
  //  to the 'wanted' value, e.g. 3 or 4 kHz for narrow-band FM)
  
   
} // end Si5351_CalcOutputDividerParamsForFM()


//------------------------------------------------------------------------
BOOL Si5351_GetPLLAndFractionalOutputDividerParamsForFM( // -> TRUE=ok, FALSE='try something else' !
          int    iOutputIndex, // [in] zero-based index for the PLL *and* the output divider
          double dblWantedFrequency_Hz, // [in] "wanted" carrier frequency in Hertz, 64 bit float
          int    iFreqModDeviation_Hz,  // [in] wanted deviation (+/- N Hz around the carrier)
          T_Si5351_FractionalDividerParamsP1P2P3 *pPLLDivider, // [out] integer- or fractional divider params FOR THE PLL (VCO "A" or "B")
          T_Si5351_FractionalDividerParamsP1P2P3 *pOutDivider) // [out] integer- or fractional divider params (p1,p2,p3,r_div) FOR THE OUTPUT ("CLK0", "CLK1", "CLK2")
  // Subroutine to determine parameters for one of the Si5351's VCOs, and one of the 
  //    *FRACTIONAL* output dividers for SOFTWARE-GENERATED frequency modulation. 
  //    Details and principle explained in Si5351_CalcOutputDividerParamsForFM().
  // Returns TRUE if this is possible (using a constant internal VFO frequency
  //         but a FRACTIONAL output divider to 'modulate'), keeping the
  //         carrier frequency acceptable close to the 'wanted' frequency.
  //    
{ 
  int    iPLLDivider, iMinPLLDivider, iBestPLLDivider;
  int    realized_deviation, best_realized_deviation;

  
  // 150 MHz is too much for a FRACTIONAL divider's output in general.
  // But with a maximum VFO frequency of 900 MHz, and a MINIMUM divide-ratio
  // of "8 + 1/1048575" in *fractional* (non-integer) mode, the max frequency
  // reached without exceeding the specs is 900 MHz / 8.x = 112.5 MHz ! Thus:
  if( ((DWORD)dblWantedFrequency_Hz + (DWORD)iFreqModDeviation_Hz) > 112500000UL ) // f_out exceeding 112.5 MHz ?
   { return FALSE;  // bail out; illegal output frequency for an 'Output Multisynth' in FRACTIONAL divider mode !
   }   
  
  if( g_dblReferenceFrequency_Hz <= 0.0 ) // Init this if the application didn't..
   {  g_dblReferenceFrequency_Hz = (double)SWI_SI5351_REFERENCE_FREQUENCY_HZ;
      // SWI_SI5351_REFERENCE_FREQUENCY_HZ is just a default for this variable.
   }     
  // For a start, assume f_vco = 900 MHz, or at least 'close to the maximum'.
  iPLLDivider = (int)( 900.1E6/*f_vco*/ / g_dblReferenceFrequency_Hz + 0.5) & 0x7FFE;
       //  '-> e.g. i = 900MHz / 25MHz = 36
  iMinPLLDivider = (int)( 599.9E6/*f_vco*/ / g_dblReferenceFrequency_Hz + 0.5) & 0x7FFE;
       //  '-> e.g. i = 600MHz / 25MHz = 24  
  // In the interest of low phase noise FROM THE PLL, the PLL feedback divider
  // should be an EVEN INTEGER. Try all valid VFO frequencies, and
  // pick the one that gives a fractional part for the OUTPUT DIVIDER (p2/p3)
  // as close to 0.5 as possible, because that would give a good 'headroom'
  // for the modulation:
  iBestPLLDivider = -1;
  best_realized_deviation = 0;
  while( iPLLDivider >= iMinPLLDivider )
   { realized_deviation = Si5351_CalcOutputDividerParamsForFM( iPLLDivider,
                           dblWantedFrequency_Hz, iFreqModDeviation_Hz,
                           pOutDivider );      
     if(  MM_AbsInt( iFreqModDeviation_Hz-realized_deviation ) 
        < MM_AbsInt( iFreqModDeviation_Hz-best_realized_deviation ) ) 
      { // Bingo.. found a [better] solution with this PLL-divider -> remember it
        best_realized_deviation = realized_deviation;
        iBestPLLDivider = iPLLDivider;
      }
     iPLLDivider -= 2;
   } // end while < repeat for all even integer PLL dividers >
  
  if( iBestPLLDivider > 0 )  // found ANY solution ? Use the BEST:
   { Si5351_InitStruct_FractionalDividerParamsP1P2P3( pPLLDivider );  // parameters for the PLL feedback divider (here: integer) 
     pPLLDivider->p1 = (DWORD)(iBestPLLDivider << 7) - 512;  // equation 1.1, with the fractional part (P2) = ZERO
     Si5351_InitStruct_FractionalDividerParamsP1P2P3( pOutDivider );  // parameters for the 'Multisynth' output (here: FRACTIONAL divider for FM)
     Si5351_CalcOutputDividerParamsForFM( iBestPLLDivider, // update pOutDivider again, now for the 'best' solution:
        dblWantedFrequency_Hz, iFreqModDeviation_Hz, pOutDivider );
     // "23cm"-Example :
     //  dblWantedFrequency_Hz = 1298.350 MHz / 128 = 10143359 Hz [ideally]
     //  pOutDivider->p1 = 10845
     //  pOutDivider->p2 = 8850 ( + 2047 = 10897 for the CENTER frequency )
     //  pOutDivider->p3 = 58984
     //  f_vco = 900 MHz
     //  f_out = f_vco   * 128 / (512+ p1    + p2    /   p3 )
     //        = 900 MHz * 128 / (512+ 10845 + 10897 / 58984) = 10143358.8 Hz
     return TRUE;
   }
  else // didn't find ANY solution -> let the caller try a different method
   { return FALSE;
   }
} // end Si5351_GetPLLAndFractionalOutputDividerParamsForFM()

//------------------------------------------------------------------------
int Si5351_CalcPLLFeedbackDividerParamsForFM( // ... for a given INTEGER PLL feedback divider :
          int    iOutputDivider,  // [in] integer, even OUTPUT DIVIDER, for f_vco = dblWantedFrequency_Hz * iOutputDivider
          double dblWantedFrequency_Hz, // [in] "wanted" carrier frequency in Hertz, 64 bit float
          int    iFreqModDeviation_Hz,  // [in] wanted deviation (+/- N Hz around the carrier)
          T_Si5351_FractionalDividerParamsP1P2P3 *pPLLDivider) // [out] fractional divider params (p1,p2,p3) FOR THE PLL FEEDBACK (f_vco_a or f_vco_b)
  // Subroutine for Si5351_GetPLLFractionalFeedbackDividerAndIntegerOutputParamsForFM().
  // Called *multiple times* from there, to find the *best* solution,
  // and (after picking the best) calculate all outputs for that solution again.
  // When successfull (for the given inputs), returns the FM DEVIATION 
  //                  (+/- X Hertz) realized with the input parameters. 
  // If the fractional part (p2/p3) doesn't give enough headroom for the
  // modulation, the return value may be LESS than the 'wanted' deviation.
  // Otherwise (completely unusable solution), returns ZERO .
  // 
  // Formula for the INTERNAL VCO FREQUENCY using a fractional PLL feedback divider:
  //    >  f_vco = f_ref * ( 512 + P1 + (P2_offset+modulation)/P3 ) / 128  [Si5351_FM equ. 1.1]
  //       where   modulation = 0..4095 (for audio samples from a 12-bit ADC).
  //               modulation = 2047    should give f_out as close as possible
  //                                    to dblWantedFrequency_Hz .
{
  long   p1, p2_offset, p3;
  double temp;
  double p2_div_p3_lo, p2_div_p3_hi;  // p2/p3 as a floating point number (not a fraction = Bruch yet)
  double f_vco = iOutputDivider * dblWantedFrequency_Hz;
  double f_dev_vco = (double)iOutputDivider * iFreqModDeviation_Hz;
  double f_vco_min = f_vco - f_dev_vco;
  double f_vco_max = f_vco + f_dev_vco;
  double f_out_center;
  int    realized_deviation;
  // "439.250 MHz FM" example:
  //   [in] iOutputDivider = ca 900 MHz / 146.41666 MHz = 6,
  //   [in] dblWantedFrequency_Hz = 439.250 MHz / 3 = 146.41666666 MHz ("carrier"),
  //   [in] iFreqModDeviation_Hz  = 4000 Hz / 3     = 1333 Hz,
  //        f_vco     = 6 * 146.41666666 MHz = 878.5000000 MHz (well within the 'spec') 
  //        f_dev_vco = 6 * 1333 Hz = 7998.0 Hz
  
  if( (f_vco_min < 599E6) || (f_vco_max > 901E6) ) // internal VCO frequency "out of spec" ?
   { return 0;  // let the caller try something else !
   }
  
  // The 'direct' formula for the PLL's Fractional Feedback Divider output frequency,
  // 'directly' calculated from the bitgroups 'P1' ('encoded' integer part),
  // 'P2' ('encoded' numerator), and 'P3' (same as the denominator, 'C') is:
  // 
  //   f_vco = f_ref * ( 512 + P1 + P2/P3 ) / 128      [Si5351_FM equation 1.1]
  //                                 |    |___|
  //                                 |      |
  //                      integer part      fractional part, 0.1 ... 0.9, ideally 0.5
  // 
  // As explained in dsPIC33EP_Si5351_Synth/Documentation/Si5351_FM_Tests_2021_11.txt,
  // we're going to inject the MODULATING SIGNAL into the lower 12 bits 
  // of the fractional part's numerator (p2, a 20-bit value) .
  // The derivation of the following equations from 1.1 is in Si5351_FM_Tests_2021_11.txt:
  //   P1    = int(128*f_vco/f_ref - 512)              (1.1.1)
  //   P2/P3 = 128*f_vco/f_ref - 512 - P1              (1.1.2)
  //     
  // Coarsely estimate the integer part 'p1' of the output divider,
  // assuming the fractional part (p2/p3) is somewhere between 0 and 0.9999 :
  temp = 128.0 * f_vco_min / g_dblReferenceFrequency_Hz - 512.0; // (1.1.1) ...
  p1 = (long)temp;  // .. now without the fractional part (TRUNCATED, not ROUNDED)
  if( (p1<0) || (p1>262143) ) // oops.. 'p1' doesn't fit inside the Si5351 18-bit group !
   { return 0;
   }   
  // "439.250 MHz FM" example (at THIS point) : 
  // > f_vco_min = f_vco_center - f_dev_vco  = 878500000 - 7998 = 878492002 
  // > f_vco_max = f_vco_center + f_dev_vco  = 878500000 + 7998 = 878507998
  // > temp = 3985.878 (oops.. damned close to the NEXT integer)
  // > p1   = 3985
  //
  p2_div_p3_lo = 128.0*f_vco_min/g_dblReferenceFrequency_Hz - 512.0 - p1;
  p2_div_p3_hi = 128.0*f_vco_max/g_dblReferenceFrequency_Hz - 512.0 - p1;
  
  // "439.250 MHz FM" example (values seen in the debugger at THIS point) : 
  //  P2_div_P3_lo = 0.8788
  //  P2_div_P3_hi = 0.9607 (we're lucky and don't hit the NEXT integer for 'p1')
  
  if( p2_div_p3_lo < 0.0 ) // aargh ... cannot reach the wanted MINIMUM frequency !
   { temp = p2_div_p3_lo;
     p2_div_p3_lo = 0;     // reduce the MAXIMUM frequency (and inc the min value for p2/p3)
     p2_div_p3_hi += temp; // limit the other endstop by the same amount to keep the CENTER
   }
  if( p2_div_p3_hi > 1.0 ) // aargh ... cannot reach the wanted MAXIMUM frequency !
   { temp = p2_div_p3_hi - 1.0;
     p2_div_p3_hi = 1.0;   // increase the MINIMUM frequency (and dec the max value for p2/p3)
     p2_div_p3_lo += temp; // limit the other endstop by the same amount to keep the CENTER
   }
  
  // At this point, we only know the fractional part as p2/p3 between 0 and 0.999 .
  // Next: Find a suitable fraction from integers 'p2_offset' and 'p3' for a 
  //       "modulatable fractional part" with integers p2_offset and p3, so that..
  //  P2_div_P3_lo = (P2_offset+0   ) / P3                       (1.1.3)
  //  P2_div_P3_hi = (P2_offset+4095) / P3                       (1.1.4)
  //    // ,--------------------|__|
  //    // '---> 12-bit "Modulator" FOR THE PLL FEEDBACK "DIVIDER" 

  // Again, the derivation of these formulas is in Si5351_FM_Tests_2021_11.txt !
  //  P2_offset = 4095 / ( (P2_div_P3_hi / P2_div_P3_lo) - 1 )   (1.1.5)
  //  P3 = P2_offset / P2_div_P3_lo                              (1.1.6)
  // Apply the above equations (here, again, for the PLL FEEDBACK DIVIDER) :
  p2_offset = 4095 / ( (p2_div_p3_hi / p2_div_p3_lo) - 1 );
  p3        = p2_offset / p2_div_p3_lo;

  // "439.250 MHz FM" example (values seen in the debugger at THIS point) :
  //  p1        =  3985
  //  p2_offset = 43940
  //  p3        = 49999

  // Pass the results back in a struct : (ok for dsPIC, but not for PIC16 ! )
  Si5351_InitStruct_FractionalDividerParamsP1P2P3( pPLLDivider );
  pPLLDivider->p1 = p1;
  pPLLDivider->p2 = p2_offset; // here: LOWEST value for p2, with headroom to add the modulator (0..4095)
  pPLLDivider->p3 = p3;
  
  // As a sanity check, to find out if the FM signal will REALLY be ok,
  // calculate the MIN- and MAX output frequency resulting from
  //            p1, p2_offset + 0, p2_offset+4095, and p3 again.
  // The DIFFERENCE between both frequencies is the (double) deviation,
  // the MEAN value is the 'unmodulated carrier' frequency.
  // Remember: f_vco = f_ref * ( 512 + P1 + P2/P3 ) / 128 [Si5351_FM equation 1.1]
  //           thus THE HIGHER 'P2', the HIGHER f_vco and f_out ! 
  Si5351_CalcVcoFreqFromP1P2P3( pPLLDivider, &f_vco_min );
  pPLLDivider->p2 = p2_offset + 4095; // <- would be nice to SUPPRESS CARRY from bit 15 into bit 16 here..
  Si5351_CalcVcoFreqFromP1P2P3( pPLLDivider, &f_vco_max );
  pPLLDivider->p2 = p2_offset; // back to the LOWEST value for p2, with headroom to add the modulator (0..4095)
  
  temp = 0.5 * (f_vco_max - f_vco_min) / iOutputDivider;  // -> deviation "to ONE side" (+/- x Hz)
  realized_deviation = (int)temp;
  f_out_center = 0.5 * (f_vco_min + f_vco_max) / iOutputDivider; // -> "realized" OUTPUT center frequency

  // "439.250 MHz FM" example (values seen in the debugger at THIS point) :
  //  f_vco_min             = 8.784920054020339E8
  //  f_vco_max             = 8.785080018166016E8
  //  f_out_center          = 1.4641666726821962E8
  //  dblWantedFrequency_Hz = 1.4641666666666666E8
  //  realized_deviation    = 1333 == iFreqModDeviation_Hz; Great..
  //       and the IC-9700's waterfall confirmed a low-frequency sinewave modulation
  //       with the above parameters did indeed sweep between 439.246 and 439.254 MHz.
  //    With a couple of attenuators in series, the IC-9700 indicated the following
  //    'relative fieldstrenghts' (S-meter readings, all in *FM*):
  //    439.250 MHz : S9 + 34dB
  //    439.200 MHz : S0 
  //    439.165 MHz : S4 (strongest frequency-modulated "sideband" observed NEAR the carrier)
  //    439.331 MHz : S3..S4
  //    439.712 MHz : S2..S3 
  //    435.668 MHz : S9 in FM ! (3.5 MHz below the carrier)
  //    433.150 MHz : S9 in FM ! (6.1 MHz below the carrier)
  // Turned the modulation (and the 'periodic reprogramming' of the Si5351) off
  //    and measured again, for comparison:
  //    439.250 MHz : S9 + 34dB  (no change between modulated/unmodulated),
  //    439.165 MHz : S4         (no change between modulated/unmodulated),
  //    435.668 MHz : S9         (no change between modulated/unmodulated),
  //    433.150 MHz : S9         (no change between modulated/unmodulated).
  // Displayed in the 'Synth Registers' menu:
  //    R16:Clk0ct ("Clock 0 Control") = 0b01001111
  // ,--------------------------------------' (see AN619 page 20) :                                       
  // '--> Bit 6 set: "MS0 operates in integer mode." 
  //      -> These unwanted birdies don't originate in the OUTPUT DIVIDER,
  //         but possibly in the VCO itself, maybe injected from a noisy supply.
  //  ! For an FM repeater, the output from an Si5351 (modulated or not)
  //  ! MUST be 'cleaned' by a traditional PLL to remove those birdies
  //  ! several Megahertz away from the nominal ouput frequency !
  
  
  if(  (f_out_center < (dblWantedFrequency_Hz - 0.25 * temp) )
     ||(f_out_center > (dblWantedFrequency_Hz + 0.25 * temp) ) )
   { return 0;  // unacceptably "off-center", discard the result
   }
  if(  (realized_deviation < (iFreqModDeviation_Hz - iFreqModDeviation_Hz/4) )
     ||(realized_deviation > (iFreqModDeviation_Hz + iFreqModDeviation_Hz/4) ) )
   { return 0;  // unacceptably "wide" or "narrow" modulation, discard the result
   }
  
  return realized_deviation; // ... in Hertz, acceptably close to the 'wanted' FM-deviation.
  // (since Si5351_GetPLLAndFractionalOutputDividerForFM() calls this
  //  function in a loop, for all possible EVEN INTEGER PLL FEEDBACK DIVIDERS,
  //  chances are good to find a 'realized deviation' that is really close
  //  to the 'wanted' value, e.g. 3 or 4 kHz for narrow-band FM)
   
} // end Si5351_CalcPLLFeedbackDividerParamsForFM()


//------------------------------------------------------------------------
BOOL Si5351_GetPLLFractionalFeedbackDividerAndIntegerOutputParamsForFM( // -> TRUE=ok, FALSE='try something else' !
          int    iOutputIndex, // [in] zero-based index for the PLL *and* the output divider
          double dblWantedFrequency_Hz, // [in] "wanted" carrier frequency in Hertz, 64 bit float
          int    iFreqModDeviation_Hz,  // [in] wanted deviation (+/- N Hz around the carrier)
          T_Si5351_FractionalDividerParamsP1P2P3 *pPLLDivider, // [out] fractional divider params FOR THE PLL (VCO "A" or "B")
          T_Si5351_FractionalDividerParamsP1P2P3 *pOutDivider) // [out] integer divider params (p1,p2,p3,r_div) FOR THE OUTPUT ("CLK0", "CLK1", "CLK2")
  // Subroutine to determine parameters for one of the Si5351's VCOs, and one of the 
  //    output dividers for SOFTWARE-GENERATED frequency modulation, using the
  //    FRACTIONAL PLL FEEDBACK dividers .
  //    Details and principle explained in Si5351_CalcPLLFeedbackDividerParamsForFM().
  //    Used for OUTPUT frequencies above 112 MHz, where the OUTPUT DIVIDER
  //    cannot be *fractional* anymore (a hardware limitation of the Si5351A),
  //    for example to generate FM for 439.250 MHz (=3 * 146.41666666.. MHz) .
  // Returns TRUE if this is possible (using a modulated VFO frequency and an
  //         EVEN INTEGER output divider ratio to keep the phase noise low).
  //    
{ 
  int    iOutputDivider, iMinOutDivider, iBestOutDivider;
  int    realized_deviation, best_realized_deviation;

    
  if( g_dblReferenceFrequency_Hz <= 0.0 ) // Init this if the application didn't..
   {  g_dblReferenceFrequency_Hz = (double)SWI_SI5351_REFERENCE_FREQUENCY_HZ;
      // SWI_SI5351_REFERENCE_FREQUENCY_HZ is just a default for this variable.
   }     
  // For a start, assume f_vco = 900 MHz, or at least 'close to the maximum'.
  iOutputDivider = (int)( 900.1E6/*f_vco*/ / dblWantedFrequency_Hz + 0.5) & 0x7FFE;
  iMinOutDivider = (int)( 599.9E6/*f_vco*/ / dblWantedFrequency_Hz + 0.5) & 0x7FFE;
  // "439.250 MHz FM" example: iOutputDivider = ca 900 MHz / 146.41666 MHz = 6,
  //                           iMinOutDivider = ca 600 MHz / 146.41666 MHz = 4 .             
  // In the interest of low phase noise, the output divider ratio
  // should be an EVEN INTEGER. Above 112 MHz, it MUST. And above 150 MHz,
  // the only allowed OUTPUT DIVIDER is FOUR (grep for "MSx_DIVBY4" ).
  if( dblWantedFrequency_Hz >= 150e6 )
   { iOutputDivider = iMinOutDivider = 4; // above 150 MHz, the only allowed output divider is FOUR !
   }
  // Try all valid output dividers (and thus VCO frequencies), and pick the one 
  // that gives a fractional part for the PLL FEEDBACK DIVIDER (P2/P3) 
  // as close to 0.5 as possible, because that would give the most 'headroom'
  // for the modulation:
  iBestOutDivider = -1;
  best_realized_deviation = 0;
  while( iOutputDivider >= iMinOutDivider )
   { realized_deviation = Si5351_CalcPLLFeedbackDividerParamsForFM( iOutputDivider,
                           dblWantedFrequency_Hz, iFreqModDeviation_Hz,
                           pPLLDivider );      
     if(  MM_AbsInt( iFreqModDeviation_Hz-realized_deviation ) 
        < MM_AbsInt( iFreqModDeviation_Hz-best_realized_deviation ) ) 
      { // Bingo.. found a [better] solution with this PLL-divider -> remember it
        best_realized_deviation = realized_deviation;
        iBestOutDivider = iOutputDivider;
      }
     iOutputDivider -= 2;
   } // end while < repeat for all 'allowed' even(!) integer output dividers >
  
  if( iBestOutDivider > 0 )  // found ANY solution ? Use the BEST:
   { Si5351_InitStruct_FractionalDividerParamsP1P2P3( pOutDivider );  // parameters for the 'Multisynth' output divider (here: integer) 
     pOutDivider->p1 = (DWORD)(iBestOutDivider << 7) - 512;
     // Note: The "special case above 150 MHz", bitgroup "MSx_DIVBY4" set to 0b11,
     //       can be detected by pOutDivider->p1 == (4*128) - 512 == ZERO (!),
     //                    *and* pOutDivider->p2 == ZERO (no fractional part),
     //       when the T_Si5351_FractionalDividerParamsP1P2P3 struct is evaluated
     //       in Si5351_SetOutputDividerRegsP1P2P3() .
     Si5351_CalcPLLFeedbackDividerParamsForFM( iBestOutDivider, // update pPLLDivider again, now for the 'best' solution:
        dblWantedFrequency_Hz, iFreqModDeviation_Hz, pPLLDivider );
     // For a 'working example', see Documentation\Si5351_FM_Tests_2021_11.txt !
     return TRUE;
   }
  else // didn't find ANY solution -> let the caller try a different method
   { return FALSE;
   }
} // end Si5351_GetPLLFractionalFeedbackDividerAndIntegerOutputParamsForFM()

//------------------------------------------------------------------------
void Si5351_SendFMSample( // Highly experimental ! See details below.
      WORD u12AudioSample) // [in] next 12-bit 'audio sample' for FM.
                           //  (0...4095;  2047 = "unmodulated carrier")
  // Caller: If SWI_SI5351_SEND_FM_STREAM_IN_I2C_COMPLETION_CALLBACK == 1,
  //          only the first sample is sent when called from MAIN LOOP.
  //         After that, Si5351_SendFMSample() is called from the I2C-Bus-
  //         -transfer-complete-handler ( -> Si5351_OnSendFMSampleComplete )
  //         to achieve an almost gapless stream of samples, at the highest
  //         possible sampling rate. 
  //         For that purpose, Si5351_SendFMSample() must be non-blocking
  //         (returns almost immediately, while the next I2C multi-byte
  //         transaction keeps running in the background.
  //        
{
  BYTE *pbData;  
  
#if( SWI_SI5351_ALLOW_POKING_REGISTERS_FROM_MENU )
  if( Si5351_fPokingAround ) // "poking around" in the Si5351 via 'Synth Registers'-menu ?
   { return;  // don't "frequency modulate" while poking around (P1,P2,P3) !
   }
#endif // SWI_SI5351_ALLOW_POKING_REGISTERS_FROM_MENU ?

  if( u12AudioSample > 4095 ) // oops.. invalid sample rate (12 bit unsigned)
   {  u12AudioSample = 4095;  // prevent overflow when adding the OFFSET below
   }
  u12AudioSample += Si5351_wFreqModulationOffset[0]; // add 16-bit offset to the 12-bit modulation sample
  
#if( SWI_USE_HARDWARE_I2C ) // frequency modulation only available with 'fast' I2C-HARDWARE (not bit-banged software I2C)
  switch( Si5351_bFreqModulationMethod[0/*1st channel*/] )
   { case FM_METHOD_OUTPUT_DIV: // modulate the Si5351's *FRACTIONAL OUTPUT* divider (lower 16 bits of numerator 'P2')
        pbData = &Si5351Regs.g.OutputDivider[0].b.P2_M; // <- pointer to THE FIRST of TWO 8-bit registers (H,L)
        pbData[0] = (BYTE)(u12AudioSample >> 8);
        pbData[1] = (BYTE)(u12AudioSample >> 0);
        I2C_WriteBlock( SI5351_I2C_7BIT_SLAVE_ADDRESS, 
              Si5351_REG_OUTPUT_MSYNTH0 + Si5351_OFS_OUTPUT_MSYNTH_P2_M,
              pbData, 2/*nBytes*/,
#           if( SWI_SI5351_SEND_FM_STREAM_IN_I2C_COMPLETION_CALLBACK )
              Si5351_OnSendFMSampleComplete ); // <- invoked later from 
#           else        
              NULL/* no I2C transfer 'completion callback' */ );        
#           endif // SWI_SI5351_SEND_FM_STREAM_IN_I2C_COMPLETION_CALLBACK ?
        break;   
     case FM_METHOD_PLL_FEEDBACK: // modulate the Si5351's *PLL FEEDBACK* divider
        pbData = &Si5351Regs.g.FeedbackDivider[0].b.P2_M; // <- pointer to THE FIRST of TWO 8-bit registers (H,L)
        pbData[0] = (BYTE)(u12AudioSample >> 8); // "Register 32": Bits 15..8 of 20-bit numerator 'P2'
        pbData[1] = (BYTE)(u12AudioSample >> 0); // "Register 33": Bits 7..0 of 20-bit numerator 'P2'
#      if( SWI_USE_HARDWARE_I2C )
        I2C_WriteBlock( SI5351_I2C_7BIT_SLAVE_ADDRESS, 
           Si5351_REG_MSYNTH_PLL_A + Si5351_OFS_MSYNTH_PLL_P2_M/*bRegAddress*/, 
           pbData, 2/*nBytes*/,
#           if( SWI_SI5351_SEND_FM_STREAM_IN_I2C_COMPLETION_CALLBACK )
              Si5351_OnSendFMSampleComplete );
#           else        
              NULL/* no I2C transfer 'completion callback' */ );        
#           endif // SWI_SI5351_SEND_FM_STREAM_IN_I2C_COMPLETION_CALLBACK ?
#      endif // < using a HARDWARE I2C bus > ?
        break;   
     default:
        return;
   } 

# if( ! SWI_SI5351_SEND_FM_STREAM_IN_I2C_COMPLETION_CALLBACK ) 
   I2C_WaitForCompletion();  // <- remove this when called from an interrupt !
# endif // ! SWI_SI5351_SEND_FM_STREAM_IN_I2C_COMPLETION_CALLBACK ?
   // 2021-08-23 : With BRG=53 (circa 943 kBit/second for I2C = max. speed),
   //              a 4-byte I2C bus transaction (including slave address,
   //              register address, and 16 bits per 'FM sample) took about
   //              45.2 us (from I2C-START to I2C-STOP), and when called from
   //              THE MAIN LOOP(!), could be called approximately 18000 times
   //              per second -> A netto 'sampling rate' of 20 kHz may just be
   //              achieveable with a dsPIC33EPxxMC502 at 60 MIPS .
   // 2021-08-17 : Another show-stopper: After running for a couple of hours,
   //              the I2C-SDA(?) line got stuck at low, and only a power cycle
   //              could make the Si5351A accessable again .
   //       I2C register contents seen at this point when the FM was 'alive' :
   //       I2C2CON  = 0xD240 = I2CEN + SCLREL + DISSLW + STREN
   //       I2C2STAT = 0x0410 = BCL + P
   //       I2C_bMasterState = 0xFF = I2C_STATE_FINISHED
#endif // SWI_USE_HARDWARE_I2C ?
   
} // end Si5351_SendFMSample()

# if( SWI_SI5351_SEND_FM_STREAM_IN_I2C_COMPLETION_CALLBACK ) 
//---------------------------------------------------------------------------
// Callback functions for I2C-transfer-completion :
//---------------------------------------------------------------------------

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
void Si5351_OnWriteMultipleRegistersComplete(T_I2C_TransferInstance *pXfer, int iResult )
{
} // end Si5351_OnWriteMultipleRegistersComplete() 

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
void Si5351_OnReadMultipleRegistersComplete(T_I2C_TransferInstance *pXfer, int iResult )
{
} // end Si5351_OnReadMultipleRegistersComplete()

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
void Si5351_OnSendFMSampleComplete(T_I2C_TransferInstance *pXfer, int iResult )
  // Call stack: I2C_Master_Interrupt() -> Si5351_OnSendFMSampleComplete()
{
  // To send FM samples as fast as the I2C bus permits, immediately send the
  // NEXT FM sample from here - unless the application wants to write some
  // other registers inside the Si5351.
  if( Si5351_bFMStreamState == FM_STREAM_ON )
   { Si5351_SendFMSample( APPL_GetModulatorSampleByTimestamp(M_IOP_GET_CURRENT_TIMESTAMP) );
   }
  else // FM_STREAM_OFF, FM_STREAM_ERROR ->
   { // Let the 'main loop' care for this !
   }
} // end Si5351_OnSendFMSampleComplete()

//---------------------------------------------------------------------------
void Si5351_PauseSendingFM(void) // temporarily paused sending FM samples,
  // and (if the transmission of FM samples in the I2C-bus-transfer callback
  // WAS active on entry) waits until the I2C bus isn't busy anymore.
  // 'Does no harm' (and not even waste time) when NOT sending FM at all.
{
  if( Si5351_bFMStreamState == FM_STREAM_ON )  
   {  Si5351_bFMStreamState = FM_STREAM_PAUSED; // prevent the next call of Si5351_SendFMSample()
      // from I2C_Master_Interrupt() -> Si5351_OnSendFMSampleComplete() 
      I2C_WaitForCompletion(); // wait until the current I2C bus transation is finished.
      // After this, it's safe to call e.g. Si5351_SetFrequency() .
   }
  
} // end Si5351_PauseSendingFM()            

//---------------------------------------------------------------------------
void Si5351_StartSendingFM(void) // 'does no harm' when ALREADY SENDING FM samples
{
  if( Si5351_bFMStreamState != FM_STREAM_ON )  
   { Si5351_bFMStreamState = FM_STREAM_ON;  
     Si5351_SendFMSample( APPL_GetModulatorSampleByTimestamp(M_IOP_GET_CURRENT_TIMESTAMP) );
     // All further FM samples are sent from within the I2C-transfer-
     // completion callback, Si5351_OnSendFMSampleComplete() .
   }
} // end Si5351_StartSendingFM()            
        
        
# endif // SWI_SI5351_SEND_FM_STREAM_IN_I2C_COMPLETION_CALLBACK ?

#endif // SWI_SI5351_SUPPORT_FREQUENCY_MODULATION ?

#if( SWI_SI5351_ALLOW_POKING_REGISTERS_FROM_MENU ) // ..for "peeking and poking"..
//---------------------------------------------------------------------------
int Si5351_RWAccessForMenuItems( // 'get/set function'. Service for the 'Hardware Diagnostics' in DL4YHF's syntheseizer firmware.
              BYTE bParamID,   // [in] "parameter identifier", here: bitwise combined as explained BELOW
              BYTE bAccess,    // [in] SL_ACCESS_READ, SL_ACCESS_WRITE, maybe more (see micro_string.h)
              BYTE bDataType,  // [in] usually DTYPE_LONG, but may be DTYPE_BYTE, etc; applies to pvValue !
              BYTE bMaxLength, // [in] maximum number of characters to be placed in pvValue (if it's a STRING)
              void *pvValue,   // [in,out] source for SL_ACCESS_WRITE, destination for SL_ACCESS_READ
              void *pvMenuItem)// [in] optional pointer to a T_MENU_ITEM ("future reserve")
  // Invoked via function pointer in a T_MENU_ITEM with data_type=DTYPE_GET_SET_ANY.
  // This function must be painfully compatible with T_SL_GET_SET_ANY_FUNC,
  //      defined in micro_string.h but mainly used in micro_menu.h (so far).
  // It was used a lot during the development of the experimental frequency
  // modulation, and to validate the 'direct' conversion formulas from 
  // parameters P1,P2,P3 into fractional division ratios and back.
  // 
  // [in] bParamID : Bitwise combination of the 4 upper bits .. 
  //                 Si5351_PARAM_PLL1_FEEDBACK, Si5351_PARAM_PLL2_FEEDBACK,
  //                 Si5351_PARAM_CLK0/1/2_OUTPUT (all called "Multisynths").
  //      The four LOWER bits identify e.g. a Multisynth's PARAMETER:
  //                 Si5351_PARAM_FRACTIONAL_DIV_A : integer part in A+B/C
  //                 Si5351_PARAM_FRACTIONAL_DIV_B : numerator in    A+B/C
  //                 Si5351_PARAM_FRACTIONAL_DIV_C : denominator in  A+B/C .
  // [in] Si5351Regs : Global variable with current REGISTER VALUES inside the Si5351
  //                   (last written to, or read from, the chip via I2C bus).
{ long i32Value, i32A, i32B, i32C;
  T_Si5351_FractionalDividerParams divider_params; // fractional divider parameters (a+b/c and possibly r_div for OUTPUT dividers)
  T_Si5351_FractionalDividerParamsP1P2P3 pll_p1p2p3; // parameters for the VCO (PLL feedback divider)
  T_Si5351_FractionalDividerParamsP1P2P3 out_p1p2p3; // parameters for a fractional output divider  
  double dblTemp;
  int  iPLL, iOut;
  BOOL access_pll = FALSE;  // TRUE=access a register belonging to a PLL / VCO; FALSE=access a register belonging to an OUTPUT DIVIDER
  
  if( bAccess & SL_ACCESS_BEGIN_EDIT ) // operator wants to EDIT a register :
   { Si5351_fPokingAround = TRUE;   // don't let others (like the FREQUENCY MODULATOR) interfere while "poking around" !
   }
  if( bAccess & SL_ACCESS_END_EDIT ) // operator wants to FINISH EDITING :
   { // ex: Si5351_fPokingAround = FALSE;  // allow e.g. the FREQUENCY MODULATOR to modify Si5351 registers
     // Keep Si5351_fPokingAround SET until leaving the 'Synth Registers' menu !
     // See implementation of the menu in the main module.
   }
  if( bAccess & SL_ACCESS_GET_TYPE ) // caller wants to know our DATA TYPE :
   { return DTYPE_LONG;  // this function only supports bDataType==DTYPE_LONG
   }
  if( bDataType != DTYPE_LONG )
   { return SL_ERROR_TYPE_CONFLICT;
   }
  switch( bParamID & 0xF0 ) // which FUNCTION BLOCK (PLL, output divider) ?
   { case Si5351_PARAM_PLL1_FEEDBACK:
     case Si5351_PARAM_PLL2_FEEDBACK:
        iPLL = ( (bParamID & 0xF0) - Si5351_PARAM_PLL1_FEEDBACK) >> 4; // -> PLL index, 0..1 
        iOut = iPLL;
        access_pll = TRUE;
        break; // end case < access a parameter in one of the PLL FEEDBACK dividers >
        
     case Si5351_PARAM_CLK0_OUTPUT :
     case Si5351_PARAM_CLK1_OUTPUT :
     case Si5351_PARAM_CLK2_OUTPUT :
        iOut = ( (bParamID & 0xF0) - Si5351_PARAM_CLK0_OUTPUT) >> 4; // -> CLKOUT index, 0..2
        iPLL = (iOut<=1) ? iOut : 1;
        access_pll = FALSE;
        break; // end case < access a parameter in one of the FRACTIONAL OUTPUT DIVIDERS >
        
     default: return SL_ERROR_ACCESS_DENIED;
   } // end switch( bParamID & 0xF0 )

  // Even when WRITING a certain parameter, many others must be valid, so update them all:
  Si5351_GetP1P2P3FromPLLFeedbackDivider(&Si5351Regs.g.FeedbackDivider[iPLL], &pll_p1p2p3 );
  Si5351_GetP1P2P3FromOutputDivider( &Si5351Regs.g.OutputDivider[iOut], &out_p1p2p3 );
        
  // Convert P1,P2,P3 back into the fractional divider parameters A+B/C :
  if( access_pll ) 
   { Si5351_ConvertP1P2P3ToFractionalDividerParamsABC( &pll_p1p2p3, &i32A, &i32B, &i32C);
   }
  else
   { Si5351_ConvertP1P2P3ToFractionalDividerParamsABC( &out_p1p2p3, &i32A, &i32B, &i32C);
   }

  // Now decide to do with all these values (pll_p1p2p3, out_p1p2p3, A,B,C) : 
  if( bAccess & SL_ACCESS_READ ) // just a harmless READ-access ... return ONE of them:
   { switch( bParamID & 0x0F )   // which PARAMETER (A,B,C in 'A+B/C' or 'encoded' P1..P3) ?
      { case Si5351_PARAM_FRACTIONAL_DIV_A : i32Value = i32A; break;
        case Si5351_PARAM_FRACTIONAL_DIV_B : i32Value = i32B; break;
        case Si5351_PARAM_FRACTIONAL_DIV_C : i32Value = i32C; break;
        case Si5351_PARAM_FRACTIONAL_DIV_P1:
             i32Value = (long)(access_pll ? pll_p1p2p3.p1 : out_p1p2p3.p1 );
             break;
        case Si5351_PARAM_FRACTIONAL_DIV_P2: 
             i32Value = (long)(access_pll ? pll_p1p2p3.p2 : out_p1p2p3.p2 );
             break;
        case Si5351_PARAM_FRACTIONAL_DIV_P3: 
             i32Value = (long)(access_pll ? pll_p1p2p3.p3 : out_p1p2p3.p3 );
             break;
        case Si5351_PARAM_VCO_FREQUENCY: // theoretic VCO frequency, calc'd from P1,P2,P3:
             Si5351_CalcVcoFreqFromP1P2P3( &pll_p1p2p3, &dblTemp );
             i32Value = (long)( dblTemp + 0.5 ); // round to integer [frequency in Hertz]
             break;
        case Si5351_PARAM_OUTPUT_FREQUENCY: // theoretic 'CLKx' OUTPUT frequency.
             Si5351_CalcVcoFreqFromP1P2P3( &pll_p1p2p3, &dblTemp );
             Si5351_CalcOutputFreqFromVcoAndP1P2P3( &dblTemp/*f_vco*/, &out_p1p2p3, &dblTemp/*f_out*/ );
             i32Value = (long)( dblTemp + 0.5 ); // round to integer [freq in Hertz]
             // When tested with the 1298.350 MHz settings, got here with ...
             //    dwP1=10752, dwP2=0, dwP3=1 (OUTPUT divider parameters)
             //    dwP1_PLL=4059, dwP2_PLL=618, dwP3_PLL=3223,
             //    f_vco = 892615625 Hz,   f_out = 10143359 Hz, 
             //    f_out * 128 = 1298349952 (ok, accept the rounding error).
             break;
             
        default: return SL_ERROR_ACCESS_DENIED;
      }
     // ex: return i32Result; .. when this function was a simple T_SL_GET_LONG_FUNC..
     *(long*)pvValue = i32Value; // we're running on a PIC .. don't waste memory to check for NULL pointers here.
     return SL_GET_SET_OK;
   }
  else // no READ- but a WRITE-access to a fractional divider (aka "Multisynth") ?
  if( bAccess & SL_ACCESS_WRITE )  // a potentially dangerous WRITE-access from the 'Synth Registers' menu:
   { // 'Poking around' in the registers of an Si5351 should be done CAREFULLY !
     if( ! Si5351_fPokingAround ) 
      { return SL_ERROR_ACCESS_DENIED;
      }
     i32Value = *(long*)pvValue;  // [in] NEW 18..20-bit-group value .
     // Life would be easy if i32Value could simply be written into one or two
     // subsequent registers in the Si5351. Unfortunately that's NOT the case.
     switch( bParamID & 0xF0 ) // which FUNCTION BLOCK (PLL, output divider) ?
      { case Si5351_PARAM_PLL1_FEEDBACK:
        case Si5351_PARAM_PLL2_FEEDBACK:
           iPLL = ( (bParamID & 0xF0) - Si5351_PARAM_PLL1_FEEDBACK) >> 4; // -> PLL index, 0..1 
           switch( bParamID & 0x0F ) // overwrite which PARAMETER (A,B,C in 'A+B/C' or 'encoded' P1..P3) ?
            { case Si5351_PARAM_FRACTIONAL_DIV_A : i32A = i32Value; break;
              case Si5351_PARAM_FRACTIONAL_DIV_B : i32B = i32Value; break;
              case Si5351_PARAM_FRACTIONAL_DIV_C : i32C = i32Value; break;
              case Si5351_PARAM_FRACTIONAL_DIV_P1: pll_p1p2p3.p1 = (DWORD)i32Value; break;
              case Si5351_PARAM_FRACTIONAL_DIV_P2: pll_p1p2p3.p2 = (DWORD)i32Value; break;
              case Si5351_PARAM_FRACTIONAL_DIV_P3: pll_p1p2p3.p3 = (DWORD)i32Value; break;
              default: break; // don't overwrite ANY (A,B,C,P1,P2,P3) !
            }
           switch( bParamID & 0x0F ) // branch again, now to invoke the suitable 'register-write' function..
            { case Si5351_PARAM_FRACTIONAL_DIV_A :
              case Si5351_PARAM_FRACTIONAL_DIV_B :
              case Si5351_PARAM_FRACTIONAL_DIV_C : // "poke" A,B,C :
                 if( Si5351_SetPLLFeedbackRegisters( Si5351_GET_REG_FROM_ADDRESS,
                      (DWORD)i32A, (DWORD)i32B, (DWORD)i32C, &Si5351Regs.g.FeedbackDivider[iPLL] ) )
                  { return SL_GET_SET_OK;
                  }
                 break;
              case Si5351_PARAM_FRACTIONAL_DIV_P1:
              case Si5351_PARAM_FRACTIONAL_DIV_P2:
              case Si5351_PARAM_FRACTIONAL_DIV_P3: // "poke" P1,P2,P3 :
                 if( Si5351_SetPLLFeedbackRegsP1P2P3( Si5351_GET_REG_FROM_ADDRESS,
                        &pll_p1p2p3, &Si5351Regs.g.FeedbackDivider[iPLL] ) )
                  { return SL_GET_SET_OK;
                  }
                 break;         
              default: 
                 break;
            }
           break; // end case < WRITE ("poke") into one of the PLL FEEDBACK dividers >
        
        case Si5351_PARAM_CLK0_OUTPUT :
        case Si5351_PARAM_CLK1_OUTPUT :
        case Si5351_PARAM_CLK2_OUTPUT :
           iOut = ( (bParamID & 0xF0) - Si5351_PARAM_CLK0_OUTPUT) >> 4; // -> CLKOUT index, 0..2
           switch( bParamID & 0x0F ) // overwrite which PARAMETER (A,B,C in 'A+B/C' or 'encoded' P1..P3) ?
            { case Si5351_PARAM_FRACTIONAL_DIV_A : i32A = i32Value; break;
              case Si5351_PARAM_FRACTIONAL_DIV_B : i32B = i32Value; break;
              case Si5351_PARAM_FRACTIONAL_DIV_C : i32C = i32Value; break;
              case Si5351_PARAM_FRACTIONAL_DIV_P1: out_p1p2p3.p1 = (DWORD)i32Value; break;
              case Si5351_PARAM_FRACTIONAL_DIV_P2: out_p1p2p3.p2 = (DWORD)i32Value; break;
              case Si5351_PARAM_FRACTIONAL_DIV_P3: out_p1p2p3.p3 = (DWORD)i32Value; break;
              default: break; // don't overwrite ANY (A,B,C,P1,P2,P3) !
            }
           switch( bParamID & 0x0F ) // branch again, now to invoke the suitable 'register-write' function..
            { case Si5351_PARAM_FRACTIONAL_DIV_A :
              case Si5351_PARAM_FRACTIONAL_DIV_B :
              case Si5351_PARAM_FRACTIONAL_DIV_C : // poke a,b,c (OUTPUT DIVIDER parameter, designer form)
                 divider_params.a = (DWORD)i32A;
                 divider_params.b = (DWORD)i32B;
                 divider_params.c = (DWORD)i32C;
                 divider_params.r_div = 1;
                 if( Si5351_SetOutputDividerRegisters( Si5351_GET_REG_FROM_ADDRESS,
                     &divider_params, // [in] a,b,c, r_div (parameters for the OUTPUT divider)
                     &Si5351Regs.g.OutputDivider[iOut] ) )
                  { return SL_GET_SET_OK;
                  }
                 break;
              case Si5351_PARAM_FRACTIONAL_DIV_P1:
              case Si5351_PARAM_FRACTIONAL_DIV_P2:
              case Si5351_PARAM_FRACTIONAL_DIV_P3: // poke p1,p2,p3 (OUTPUT DIVIDER parameter, register bitgroup form)
                 if( Si5351_SetOutputDividerRegsP1P2P3( Si5351_GET_REG_FROM_ADDRESS,
                         &out_p1p2p3, &Si5351Regs.g.OutputDivider[iOut] ) )
                  { return SL_GET_SET_OK;
                  }
                 break;         
              default: 
                 break;
            }
           break; // end case < WRITE ("poke") into one of the CLOCKOUT DIVIDER dividers >
        
        default:
           break;
      } // end switch( bParamID & 0xF0 )
   } // end if < READ- or WRITE-access ? > 
  
  // Arrived here ? Access to this register or function block not supported yet !
  return SL_ERROR_ACCESS_DENIED;
  
} // end Si5351_RWAccessForMenuItems()
#endif // SWI_SI5351_ALLOW_POKING_REGISTERS_FROM_MENU ?

// EOF < C:\pic\dsPIC33EP_Si5351_Synth\si5351.c >
