// File: C:\cbproj\Remote_CW_Keyer\dsound_wrapper.c
// Purpose: Wrapper between non-MFC applications and "Direct Sound" .
// Based on dsound_wrapper.c,v 1.1.1.1 2002/01/22 00:52:45,
//          "Simplified DirectSound interface",
//          by Phil Burk & Robert Marsanyi .
//   (in 2023, no traces of the above file were found in the WWW anymore)
//    Original description quoted below - DON'T MODIFY -,
//    plus various additions by DL4YHF who tried to understand
//    how this is supposed to work (any why it didn't work)
//    *further below* .
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
// PortAudio Portable Real-Time Audio Library
// For more information see: http://www.softsynth.com/portaudio/
// DirectSound Implementation
// Copyright (c) 1999-2000 Phil Burk & Robert Marsanyi
//
// Permission is hereby granted, free of charge, to any person obtaining
// a copy of this software and associated documentation files
// (the "Software"), to deal in the Software without restriction,
// including without limitation the rights to use, copy, modify, merge,
// publish, distribute, sublicense, and/or sell copies of the Software,
// and to permit persons to whom the Software is furnished to do so,
// subject to the following conditions:
//
// The above copyright notice and this permission notice shall be
// included in all copies or substantial portions of the Software.
//
// Any person wishing to distribute modifications to the Software is
// requested to send the modifications to the original developer so that
// they can be incorporated into the canonical version.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
// IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR
// ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
// CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
// WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
//
// Notes (by DL4YHF, 2023..2024):
//   * As already expected, when really INVOKING any of the functions in
//     'dsound_wrapper.c', Borland C++ Builder 6 complained :
//       > Unresolved external 'DirectSoundEnumerateA referenced from .. bla..'
//       > Unresolved external 'DirectSoundCreate referenced from .. blubb..'
//       > Unresolved external 'DirectSoundCaptureCreate referenced from .. '
//     Microsoft (formerly MSDN, not "learn.microsoft.com") suggested
//     to use 'Link Library: Dsound.lib'. A-ha. Where's that ?
//     Fortunately, even the stoneage Borland C++Builder 6 was shipped with
//     this library, so added C:\CBuilder6\Lib\Psdk\dsound.lib to the project,
//     hoping THAT LIBRARY uses a calling convention compatible with BCB .
//
//   * Even though many (including Microsoft) say 'DirectSound is a dead horse',
//     some say it's better than the stoneage WMME 'Wave Audio API',
//     as long as low audio latency is concerned. For the planned use
//     (generation of a low-latency 'Morse sidetone output' for an electronic
//      keyer), low audio latency was the primary goal, WITHOUT having to dig
//      into the hyped "AudioGraph settings or the WASAPI IAudioClient3 interface",
//      even a dead horse may be better than a dead dinosaur. Time will tell.
//      Audio latency test results are possibly documented in
//      C:\cbproj\Remote_CW_Keyer\KeyerThread.c .
//
//   * When trying to build a project with dsound_wrapper.c using
//     "Embarcadero C++Builder V12 Community Edition" instead of the old
//     "Borland C++Builder V6", hell broke lose again:
//       > [ilink32 Error] Fatal: Illegal VIRDEF fixup index in module
//       >                 '..\dsound_wrapper.c'
//     (Ah, cryptic LINKER errors, after fighting with the COMPILER..)
//     Even though Google isn't your friend, searched for the error, and found:
//      * "You may do better asking in a Borland group." (a-ha, it's an OLD one)
//      * "VIRDEF/COMDEF is a record type in the OMF object code generated
//         by the compiler, hence it's a compiler issue."
//     (No, in this case it wasn't a COMPILER- but a stupid LIBRARY issue.)
//     Added ..hold your breath but SEARCH THIS FILE YOURSELF.. :
//     C:\Program Files (x86)\Embarcadero\Studio\23.0\lib\win32\release\psdk\dsound.lib )
//      to the Embarcadero *.cbproj (Add.. Static Library Files (*.lib).. yaddayadda).
//     That didn't fix the "Illegal VIRDEF fixup index".
//     Compared the function used dsound_wrapper.c (listed by tdump with the
//     real function names, in this case the 'Ansi' versions ending with 'A')
//     against the functions listed by "tdump" for dsound.lib . Result:
//       DirectSoundEnumerate()        : exists as DirectSoundEnumerateA()/..W()
//       DirectSoundCreate()           : ok, exists as DirectSoundCreate() and ..8() [?!]
//       DirectSoundCaptureCreate()    : exists as "Internal Name: DirectSoundCaptureCreate"
//       DirectSoundCaptureEnumerate() : exists as DirectSoundCaptureEnumerateA()/..W()
//     Temporarily "commented out" references to anything suspected to cause
//     the "Illegal VIRDEF fixup index" (controlled by _TEST_VIRDEF_TROUBLE_)...
#define _TEST_VIRDEF_TROUBLE_ 0 // omit anything that may have caused the
       //  "Illegal VIRDEF fixup index" in C++Builder 12 ?
       //   1=yes (at the risk of NON-FUNCTIONING CODE), 0=no=normal compilation

#include "switches.h" // project specific 'compilation switches' like
                      // SWI_USE_DSOUND_WRAPPER, SWI_HARDCORE_DEBUGGING, etc.

#ifdef __BORLANDC__
// prevent BCB V6 from bugging us with "initialized data in header, cannot pre-compile..." :
# pragma hdrstop
#endif


#include <stdio.h>
#include <stdlib.h>
#include <math.h>

// #define INITGUID  // Formerly needed to build IID_IDirectSoundNotify.
  // But when compiling with Embarcadero C++Builder V12, this caused
  // the ilink32 error "Illegal VIRDEF fixup index". See objbase.h for info, and
  // guiddef_replacement_for_CBuilder12.h for an attempt to fix the "Illegal VIRDEF fixup index" .
#include <objbase.h> // e.g. in c:\cbuilder6\include\objbase.h (a windows thing)
#include <unknwn.h>  // e.g. in c:\cbuilder6\include\unknwn.h  (also a windows thing)

#include "dsound_wrapper.h"


// ex: #include "pa_trace.h"
#define AddTraceMessage(msg,data) /* no 'trace' available */

//----------------------------------------------------------------------------
// Global variables for hardcore debugging / post-mortem crash analysis.
//        Useful when the debugger's call stack shows nothing but garbage,
//        e.g. after an exception at the infamous address 0xFEEEFEEE .
//        ( C++Builder's debugger could still inspect SIMPLE GLOBAL
//          VARIABLES after most kinds of exceptions, but of course nothing
//          stack-based because the CPU- or task-stack was usually trashed.)
//----------------------------------------------------------------------------
#if(SWI_HARDCORE_DEBUGGING) // (1) = hardcore-debugging, (0)=normal compilation
 int DSW_iLastSourceLine = 0; // WATCH THIS after crashing with e.g. "0xFEEEFEEE"  ...
# define HERE_I_AM__DSW()  DSW_iLastSourceLine=__LINE__
  // (to also WATCH the instance of DSoundWrapper that most likely crashed,
  //  in Borland C++Builder, add a 'watch' for MyDirectSound ,
  //  especially examine ..
  //     MyDirectSound.fInputOpened   MyDirectSound.pDirectSoundIn
  //     MyDirectSound.pInputBuffer   MyDirectSound.dwInputFifoSizeInBytes
  //     MyDirectSound.dwInputBufferTailIndex .
  // See also: Complete list of other XYZ_iLastSourceLine variables to watch
  //     in C:\cbproj\Remote_CW_Keyer\Keyer_Main.cpp, near GUI_iLastSourceLine)
#else
# define HERE_I_AM__DSW()
#endif // SWI_HARDCORE_DEBUGGING ?


// internal function prototypes (may be decorated with 'static'...)
void DSW_OnOutputError( DSoundWrapper *dsw, char *pszInfo, HRESULT hr );
void DSW_SetFirstOutputError( DSoundWrapper *dsw, char *pszInfo );
void DSW_OnInputError( DSoundWrapper *dsw, char *pszInfo, HRESULT hr );
void DSW_SetFirstInputError( DSoundWrapper *dsw, char *pszInfo );
void DSW_RestartSampleRateMeasurement( DSoundWrapper *dsw );
void DSW_CalibrateInputSampleRate( DSoundWrapper *dsw, double dblCurrentTime_s,
                                   DWORD dwNumSamplesSinceLastCall);


//---------------------------------------------------------------------------
// Common functions used for audio-in- as well as -output:
//---------------------------------------------------------------------------

static LONGLONG DSW_i64HighResTimerOffset = 0; // kludge to let our 'internal timestamps' start at ZERO

//---------------------------------------------------------------------------
double DSW_ReadHighResTimestamp_s( void ) // <- the "_s" suffix is the physical unit
  // Actually just a 'thin abstraction layer' to the
  //  "QueryPerformanceCounter"-thingy in Windows, with a conversion into
  //  the one-and-only SI unit for time (no machine-depending 'tick frequency').
{
  LONGLONG i64, i64Freq;
  static double dblConversionFactor = 0.0;

  QueryPerformanceCounter( (LARGE_INTEGER*)&i64 );
        // '--> call KERNEL32.QueryPerformanceCounter
        //    -> ntdll.RtlQueryPerformanceCounter
        //     -> ntdll.RtlWow64GetProcessMachines, etc etc ...
        // Is this safe to be called from DIFFERENT THREADS ? Yes. From Microsoft:
        // > In general, the performance counter results are consistent across
        // > all processors in multi-core and multi-processor systems,
        // > even when measured on different threads or processes.
        // > Here are some exceptions to this rule:
        // >    * Pre-Windows Vista operating systems that run on certain
        // >      processors might violate this consistency .
        // >    * When you compare performance counter results that are
        // >      acquired from different threads, consider values
        // >      that differ by +/- 1 tick to have an ambiguous ordering.
        // >      If the time stamps are taken from the same thread,
        // >      this +/- 1 tick uncertainty doesn't apply.
        //  (both of the above exceptions can be safely ignored anno 2025)
        //

  if( dblConversionFactor <= 0.0 )  // Determine the "conversion factor" when not done yet:
   { QueryPerformanceFrequency( (LARGE_INTEGER*)&i64Freq ); // will the FREQUENCY change when speed-stepping ? (hope not..)
     // '--> Also a possibly 'expensive' KERNEL call, involving 'Windows on Windows 64' ("Wow64").
     //      So only call QueryPerformanceFrequency() when necessary.
     if( i64Freq > 0 )
      { dblConversionFactor = 1.0 / (double)i64Freq;
      }
   } // end if < need to determine the 'conversion factor' (from timer ticks to SECONDS) > ?

  if( DSW_i64HighResTimerOffset == 0 )
   {  DSW_i64HighResTimerOffset = i64; // kludge to let OUR timestamp start at ZERO when the program is launched
   }
  // Use floating point to convert from timer ticks to seconds:
  return (double)(i64 - DSW_i64HighResTimerOffset) * dblConversionFactor;
  //  ,----------|_______________________________|
  //  '--> difference in timer ticks (tick frequency MAY be many Megahertz..
  //       thus do the SCALING into seconds with 64-bit *double* )
} // end DSW_ReadHighResTimestamp_s()

//---------------------------------------------------------------------------
int CALLBACK DSW_EnumOutputsProc(  // *internal* - no API function !
        LPGUID lpGUID,      // [in] > Address of the GUID that identifies the
                            //      > DirectSound driver being enumerated.
                            //      > This value can be passed to DirectSoundCreate() ...
        LPCTSTR lpszDesc,   // [in] > Address of a null-terminated string
                            //      > that provides a textual description
                            //      > of the DirectSound OUTPUT device.
        LPCTSTR lpszDrvName,// [in] > Address of a null-terminated string that
                            //      > specifies the module name of the DirectSound
                            //      > OUTPUT driver corresponding to this device.
                            // (if you think this is worth being displayed anywhere: It's not.
                            //  When tested, got stuff like this in lpszDrvName:
                            //  "{0.0.0.00000000}.{d2063819-bf5d-4a9b-9ffa-fdef51b1ed7}" .
                            // )
        LPVOID lpContext )  // [in] our application-defined "instance data"
{ DSoundWrapper *dsw = (DSoundWrapper*)lpContext;
  // > Remarks
  // > The application can save the strings passed in the lpcstrDescription
  // > and lpcstrModule parameters by copying them to memory allocated from
  // > the heap. The memory used to pass the strings to this callback function
  // > is valid only while this callback function is running.
  if( (dsw!=NULL) && (lpszDesc!=NULL) )
   {
     if( dsw->m_nOutputDevices < DSW_MAX_ENUMERATED_DEVICES )
      { strncpy( dsw->m_sOutputDevices[dsw->m_nOutputDevices].sz255DevName,
                 lpszDesc, 255 ); // seen here: "Primary Sound Driver",
                 // but the Primary Sound Driver's GUID was a NULL pointer.
                 // Don't try to COPY the "GUID" from such a pointer ... :
        if( lpGUID == NULL ) // this guy doesn't have a guid .. guess he is the "Primary Sound Driver" (output; "renderer")
         { memset( &dsw->m_sOutputDevices[dsw->m_nOutputDevices].guid,
                   0, sizeof( GUID ) ); // "all zeroes" in here will later be replaced by a NULL pointer
         }
        else // not sure if the "GUID pointer" remains valid until we need it,
         {   // so BLOCK-COPY this 128-bit "globally unique identifier" :
           dsw->m_sOutputDevices[dsw->m_nOutputDevices].guid = *lpGUID;
           // The first (output) devices that DID have a "GUID" were :
           //  * "MAGIC LCD (Intel(R) Display-Audio)"
           //  * "Lautsprecher (USB Audio CODEC)"
           //  * "Lautsprecher (Realtek High Definition Audio)"
           //  * "Lautsprecher/Kopfhrer (VB-Audio Virtual Audio Cable)"
         }
        ++dsw->m_nOutputDevices;
        // > Return Values (of the DSEnumCallback) :
        // > Returns TRUE to continue enumerating drivers, or FALSE to stop.
        return ( dsw->m_nOutputDevices < DSW_MAX_ENUMERATED_DEVICES );
      }
   }
  // Arrived here ? Too many devices, or we're unhappy with what was passed in !
  return FALSE; // stop enumerating more DirectSound OUTPUT devices !
} // end DSW_EnumOutputsProc()

//---------------------------------------------------------------------------
int CALLBACK DSW_EnumInputsProc(  // *internal* - no API function !
        LPGUID lpGUID,      // .. same meaning as in DSW_EnumOutputsProce() ..
        LPCTSTR lpszDesc, LPCTSTR lpszDrvName, LPVOID lpContext )
  // (if you think the 'DrvName' is worth being displayed anywhere: It's not.
  //  When tested, got stuff like this in lpszDrvName:
  //  "{0.0.1.00000000}.{6a609b65-8348-4412-8c49-66301a6f8843}",
  //  for the IC-7300 (with lpszDesc = "Mikrofon (USB Audio CODEC)". Baaaah.
  // )

{ DSoundWrapper *dsw = (DSoundWrapper*)lpContext;
  if( (dsw!=NULL) && (lpszDesc!=NULL) )
   {
     if( dsw->m_nInputDevices < DSW_MAX_ENUMERATED_DEVICES )
      { strncpy( dsw->m_sInputDevices[dsw->m_nInputDevices].sz255DevName,
                 lpszDesc, 255 );
        if( lpGUID == NULL ) // this guy doesn't have a guid .. guess he is the "Primary Sound Capture Driver"..
         { memset( &dsw->m_sInputDevices[dsw->m_nInputDevices].guid,
                   0, sizeof( GUID ) ); // "all zeroes" in here will later be replaced by a NULL pointer
         }
        else // this audio "capturing" (input) decive has a guid, so COPY it (128 bits, not just the address):
         { dsw->m_sInputDevices[dsw->m_nInputDevices].guid = *lpGUID;
           // The first (input-)"capturing" devices that DID have a "GUID" were :
           //  * "Onboard-Microphone (Realtek Hig" [same stupid truncation as in the WMM "wave audio" device)
           //  * "CABLE Output (VB-Audio Virtual "
           //  * "Mikrofon (USB Audio CODEC)"  [believe it or not, THIS was an Icom IC-7300]
         }
        ++dsw->m_nInputDevices;
        // > Return Values (of the DSEnumCallback) :
        // > Returns TRUE to continue enumerating drivers, or FALSE to stop.
        return ( dsw->m_nInputDevices < DSW_MAX_ENUMERATED_DEVICES );
      }
   }
  return FALSE; // stop enumerating more DirectSound INPUT devices !
} // end DSW_EnumInputsProc()


//---------------------------------------------------------------------------
BOOL DSW_EnumerateDevices( DSoundWrapper *dsw )                       // API
  // Tries to enumerate all DirectSound capable audio in- and output devices.
  // Originally a thin wrapper for Microsoft's "DirectSoundEnumerate()" .
  // To also enumerate INPUT devices, "DirectSoundCaptureEnumerate()" is used.
  // After this, an application can list all 'DirectSound' capable devices
  //    in e.g. combo lists for the IN- and OUTPUT device .
  //
  // Returns TRUE on success, FALSE otherwise.
  // Does NOT interfere when called while devices are already opened for use.
  //
  //  [out] dsw->m_nOutputDevices : number of OUTPUT devices (for "rendering"),
  //        dsw->m_nInputDevices  : number of INPUT  devices (for "capturing"),
  //        dsw->m_sOutputDevices[] : names and GUIDs of OUTPUT devices,
  //        dsw->m_sInputDevices[]  : names and GUIDs of INPUT devices.
  //
  // We avoid the name "capture", because from an application's point of view,
  // those are INPUT devices for almost anything (radio receivers, SDRs,
  // microphones, "line inputs", mosquito detectors) - not only for "capturing".
{
  dsw->m_nOutputDevices = dsw->m_nInputDevices = 0;
#if( ! _TEST_VIRDEF_TROUBLE_ )
  DirectSoundEnumerate( DSW_EnumOutputsProc, (void*)dsw );
  DirectSoundCaptureEnumerate( DSW_EnumInputsProc, (void*)dsw );
#endif // _TEST_VIRDEF_TROUBLE_ ?
  return ( dsw->m_nOutputDevices + dsw->m_nInputDevices ) > 0;

} // end DSW_EnumerateDevices()

//---------------------------------------------------------------------------
void DSW_Term( DSoundWrapper *dsw )                                   // API
{
  // Cleanup the sound buffers, for both in- and output
  DSW_CloseInput( dsw );   // <- doesn't do harm if NOT opened at all ..
  DSW_CloseOutput( dsw );
} // end DSW_Term()


//---------------------------------------------------------------------------
BOOL DSW_Init( DSoundWrapper *dsw )                                   // API
{
  memset( dsw, 0, sizeof(DSoundWrapper) );
  dsw->dblLastOutputTime_s = dsw->dblLastInputTime_s = dsw->dblTimestampAtInputBufferTail_s = DSW_ReadHighResTimestamp_s(); // here: set in DSW_Init()
  return TRUE;
}

//---------------------------------------------------------------------------
const char *DSW_ErrorCodeToString( HRESULT hr )
{
  switch( hr )
   { case DS_OK : return "";

     case DS_NO_VIRTUALIZATION : // The function completed successfully, but we had to substitute the 3D algorithm
          return "no virtualization";

     case DSERR_ALLOCATED:    // The call failed because resources (such as a priority level)
          return "allocated"; // were already being used by another caller

     case DSERR_CONTROLUNAVAIL: // The control (vol,pan,etc.) requested by the caller is not available
          return "control unavailable";

     case DSERR_INVALIDPARAM: // An invalid parameter was passed to the returning function
          return "invalid parameter";

     case DSERR_INVALIDCALL:  // This call is not valid for the current state of this object
          return "invalid call";

     case DSERR_GENERIC:      // An undetermined error occured inside the DirectSound subsystem
          return "generic error";

     case DSERR_PRIOLEVELNEEDED: // The caller does not have the priority level required for the function to succeed
          return "insufficient priority level";

     case DSERR_OUTOFMEMORY:  // Not enough free memory is available to complete the operation
          return "out of memory";

     case DSERR_BADFORMAT:    // The specified WAVE format is not supported
          return "bad wave format";

     case DSERR_UNSUPPORTED:  // The function called is not supported at this time
          return "unsupported";

     case DSERR_NODRIVER:     // No sound driver is available for use
          return "no driver available";

     case DSERR_ALREADYINITIALIZED: // This object is already initialized
          return "already initialized";

     case DSERR_NOAGGREGATION: // This object does not support aggregation
          return "no aggregation";

     case DSERR_BUFFERLOST:    // The buffer memory has been lost, and must be restored
          return "buffer lost";

     case DSERR_OTHERAPPHASPRIO: // Another app has a higher priority level, preventing this call from succeeding
          return "other app has priority";

     case DSERR_UNINITIALIZED: // This object has not been initialized
          return "uninitialized";

     case DSERR_NOINTERFACE:   // The requested COM interface is not available
          return "no COM interface";

     case DSERR_ACCESSDENIED:  // Access is denied
          return "access denied";

     default:  return "???";   // looks like our copy of "dsound.h" is outdated :)
   } // end switch
} // end DSW_ErrorCodeToString()

//---------------------------------------------------------------------------
// Audio OUTPUT ("Rendering Audio" a la DirectSound) ... usage summary :
// 1.) DSW_OpenOutputDevice() with the DESCRIPTIVE NAME (not a "GUID") .
// 2.) Set sample rate, number of channels, and buffer size (thus latency)
//        per DSW_InitOutputBuffer(). In Microsoft geek speak, that's
//        IDirectSound_CreateSoundBuffer() for the rarely-used "primary",
//        and (more important) for the "secondary" buffer.
//        DSW_InitOutputBuffer() fills the buffer with 'silence'
// 4.) Start "streaming out" per DSW_StartOutput() [ -> IDirectSoundBuffer_Play(), Loop ].
//        From this moment on, periodically 'feed' the buffer,
//        e.g. by calling DSW_WriteBlock() [ -> IDirectSoundBuffer_Lock(), Write,
//                                              IDirectSoundBuffer_Unlock() ]
// Quoted from an old "Microsoft DirectX 9.0" SDK:
//  > Using Streaming Buffers
//  > A streaming buffer plays a long sound that cannot all fit into the buffer
//  > at once. As the buffer plays, old data is periodically replaced with new data.
//  > To play a streaming buffer, call the IDirectSoundBuffer8::Play() method,
//  > specifying DSBPLAY_LOOPING in the dwFlags parameter.
//  > To halt playback, call the IDirectSoundBuffer8::Stop() method.
//  > This method stops the buffer immediately, so you need to be sure that
//  > all data has been played. This can be done by polling the play position
//  > or by setting a notification position.
//  > Streaming into a buffer requires the following steps:
//  > 1. Ascertain whether the buffer is ready to receive new data.
//  >    This can be done either by polling the play cursor
//  >    or by waiting for a notification.
//  > 2. Lock a portion of the buffer by using IDirectSoundBuffer8::Lock().
//  >    This method returns one or two addresses where data can now be written.
//  > 3. Write the audio data to the returned address or addresses
//  >    by using a standard memory-copy routine.
//  > 4. Unlock the buffer using IDirectSoundBuffer8::Unlock() .
//  > The reason IDirectSoundBuffer8::Lock might return two addresses is that
//  > you can lock any number of bytes, up to the size of the buffer,
//  > regardless of the start point. If necessary, the locked portion
//  > wraps around to the beginning of the buffer. If it does, you have to
//  > perform two separate memory copies.
//  > For example, say you lock 30,000 bytes beginning at offset 20,000
//  > in a 40,000-byte buffer. When you call Lock in this case,
//  > it returns four values:
//  >   * The memory address of offset 20,000.
//  >   * The number of bytes locked from that point to the end of the buffer;
//  >     that is, 20,000 bytes. You write this number of bytes to the first address.
//  >   * The memory address of offset 0.
//  >   * The number of bytes locked from that point; that is, 10,000 bytes.
//  >     You write this number of bytes to the second address.
//  > If no wraparound is necessary, the last two values are NULL and 0 respectively.
//  > Although it's possible to lock the entire buffer, you must not do so
//  > while it is playing. Normally you refresh only a fraction of the buffer
//  > each time. For example, you might lock and write to the first quarter
//  > of the buffer as soon as the play cursor reaches the second quarter,
//  > and so on. You must never write to the part of the buffer that lies
//  > between the play cursor and the write cursor.
//     (On first glance,
//      the "write cursor" looks like a circular buffer's "FIFO HEAD INDEX",
//      the "play cursor" looks like a circular buffer's "FIFO TAIL INDEX".
//      but that's NOT the case - so read on):
//  > DirectSound maintains two pointers into the buffer: the play cursor
//  > and the write cursor. These positions are byte offsets into the buffer,
//  > not absolute memory addresses.  (...)
//  > The write cursor is the point after which it is safe to write data into the buffer.
//  > The block between the play cursor and the write cursor is already committed
//  > to be played, and cannot be changed safely.
//  >
//  > You might visualize the buffer as a clock face, with data written to it
//  > in a clockwise direction. The play cursor and the write cursor are like
//  > two hands sweeping around the face at the same speed, the write cursor
//  > always keeping a little ahead of the play cursor. If the play cursor
//  > points to the 1 and the write cursor points to the 2, it is only safe
//  > to write data after the 2. Data between the 1 and the 2 may already
//  > have been queued for playback by DirectSound and should not be touched.
//  >
//  > The write cursor moves with the play cursor, not with data written to the buffer.
//  > If you're streaming data, you are responsible for maintaining
//  > your own pointer into the buffer to indicate where the next block of data
//  > should be written.
//  >
//  > An application can retrieve the play and write cursors by calling
//  > the IDirectSoundBuffer8::GetCurrentPosition() method.
//  > The IDirectSoundBuffer8::SetCurrentPosition() method lets you move
//  > the play cursor. Applications do not control the position of the write cursor.
//  >
//  > To ensure that the play cursor is reported as accurately as possible,
//  > always specify the DSBCAPS_GETCURRENTPOSITION2 flag when creating
//  > a secondary buffer. For more information, see DSBUFFERDESC.
//  >
//  > Play Buffer Notification
//  >
//  > DirectSound can notify your application when the play cursor reaches
//  > certain points in a buffer, or when the buffer stops. This can be useful
//  > in two common situations:
//  >  * The application must take some action, such as playing another buffer, 7
//  >    when a sound has finished playing.
//  >  * The application is streaming data to the buffer, and needs to know
//  >    when the play cursor reaches predefined points so that data
//  >    that has been played can be replaced by new data.
//  > Using the IDirectSoundNotify8::SetNotificationPositions() method, you can
//  > set any number of points within the buffer where events are to be signaled.
//  > You cannot do this while the buffer is playing. (...)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
BOOL DSW_OpenOutputDevice(
           DSoundWrapper *dsw,
     const char *sz255DevName ) // [in] descriptive audio *DEVICE NAME* (not a stuid 'GUID')
  // Return value : TRUE = ok,   FALSE = failure (e.g. device name not found) .
  //     If the function failed, dsw->sz255LastOutputError may give a clue why.
  //
  // Microsoft explains, about DirectSoundCreate() :
  // > This function creates and initializes an IDirectSound interface.
  // > Parameters:
  // > * lpGuid
  // >      Address of the GUID that identifies the sound device.
  // >      The value of this parameter must be one of the GUIDs
  // >      returned by DirectSoundEnumerate(), or NULL for the
  // >      default device.
  // > * ppDS (2nd argument)
  // >      Address of a pointer to a DirectSound object
  // >      created in response to this function.
  // > * pUnkOuter
  // >      Controlling unknown of the aggregate. Its value must be NULL.
  // >      (no idea what an "aggregate" is, guess a Microsoft 'COM' thing)
  // > Remarks
  // >   The application must call IDirectSound::SetCooperativeLevel()
  // >   immediately after creating a DirectSound object.
  //     (WB: .. with DSSCL_EXCLUSIVE, DSSCL_NORMAL, etc.
  //      In this 'wrapper', that happens in DSW_InitOutputBuffer(). )
{
  int  i;
  GUID *pGUID = NULL;
  HRESULT hr;
  BOOL found_device = FALSE;


  if( sz255DevName != NULL ) // use a specific device (not the "Default"-thingy) ?
   { if( dsw->sz255OutputDeviceName != sz255DevName ) // <- compare ADDRESS, not STRING CONTENT
      { strncpy( dsw->sz255OutputDeviceName, sz255DevName, 255 );
        // '--> dinosaur will crash when copying a string to itself !
      }
     for(i=0;( (i<dsw->m_nOutputDevices) && (i<DSW_MAX_ENUMERATED_DEVICES) ); ++i)
      { if( strcmp(sz255DevName, dsw->m_sOutputDevices[i].sz255DevName) == 0 )
         { if( dsw->m_sOutputDevices[i].guid.Data1 != 0 )
            { pGUID = &dsw->m_sOutputDevices[i].guid;
            }
           found_device = TRUE;
           break;
         }
      }
   }    // end if( sz255DevName != NULL )
  else
   { dsw->sz255OutputDeviceName[0] = '\0';
   }

  if( ! found_device )
   { strcpy( dsw->sz255LastOutputError, "OpenOutput: Device not found" );
     return FALSE;
   }

  // Create the DS object
#if( ! _TEST_VIRDEF_TROUBLE_ )
  hr = DirectSoundCreate( pGUID, &dsw->pDirectSoundOut, NULL );
#else //  _TEST_VIRDEF_TROUBLE_ : don't call the above 'suspect' at all
  hr = !DS_OK;
#endif // _TEST_VIRDEF_TROUBLE_ ?
  if( hr == DS_OK )
   { return TRUE;
   }
  return FALSE;
} // end DSW_OpenOutputDevice()


//---------------------------------------------------------------------------
BOOL DSW_InitOutputBuffer(
           DSoundWrapper *dsw,
           unsigned long nFrameRate, // Here, a "frame" is a "SAMPLE POINT", possibly with MULTIPLE CHANNELS,
                                     // all sampled at the same time (if the hardware permits).
           int nChannels,
           int bytesPerBuffer )
{
  DWORD          dwDataLen;
  DWORD          playCursor;
  HRESULT        hr;
  LPDIRECTSOUNDBUFFER pPrimaryBuffer;
  HWND           hWnd;
  WAVEFORMATEX   wfFormat;
  DSBUFFERDESC   primaryDesc;
  DSBUFFERDESC   secondaryDesc;
  unsigned char* pDSBuffData;
  LARGE_INTEGER  counterFrequency;

  // Sanity check (by DL4YHF) to catch 'application errors' :
  if( dsw->pDirectSoundOut == NULL ) // oops.. output device not opened !
   { DSW_SetFirstOutputError( dsw, "InitBuffer called but Output not open" );
     return FALSE;
   }


  dsw->iOutputSize = bytesPerBuffer;
  dsw->fOutputOpened  = FALSE;
  dsw->nOutputUnderflows = 0;
  dsw->dblSamplesWrittenToOutput = 0;
  dsw->dwBytesPerFrame_out = nChannels * sizeof(short);
  dsw->dblLastOutputTime_s = DSW_ReadHighResTimestamp_s();


  // We were using getForegroundWindow() but sometimes the ForegroundWindow may not be the
  // applications's window. Also if that window is closed before the Buffer is closed
  // then DirectSound can crash. (Thanks for Scott Patterson for reporting this.)
  // So we will use GetDesktopWindow() which was suggested by Miller Puckette.
  // hWnd = GetForegroundWindow();
#if( ! _TEST_VIRDEF_TROUBLE_ )
  hWnd = GetDesktopWindow();
#else //  _TEST_VIRDEF_TROUBLE_ : don't call the above 'suspect' at all
  hWnd = NULL;
#endif // _TEST_VIRDEF_TROUBLE_ ?

  // Set cooperative level to DSSCL_EXCLUSIVE so that we can get 16 bit output, 44.1 KHz.
  // Exclusive also prevents unexpected sounds from other apps during a performance.
  if ((hr = IDirectSound_SetCooperativeLevel( dsw->pDirectSoundOut,
              hWnd, DSSCL_EXCLUSIVE)) != DS_OK)
   { DSW_OnOutputError( dsw, "SetCooperativeLevel", hr);
     return FALSE;
   }
  // -----------------------------------------------------------------------
  // Create primary buffer and set format just so we can specify our custom format.
  // Otherwise we would be stuck with the default which might be 8 bit or 22050 Hz.
  // Setup the primary buffer description
  ZeroMemory(&primaryDesc, sizeof(DSBUFFERDESC));
  primaryDesc.dwSize        = sizeof(DSBUFFERDESC);
  primaryDesc.dwFlags       = DSBCAPS_PRIMARYBUFFER; // all panning, mixing, etc done by synth
  primaryDesc.dwBufferBytes = 0;
  primaryDesc.lpwfxFormat   = NULL;
  // Create the buffer
  if ((hr = IDirectSound_CreateSoundBuffer( dsw->pDirectSoundOut,
                  &primaryDesc, &pPrimaryBuffer, NULL)) != DS_OK)
   { DSW_OnOutputError( dsw, "CreateSoundBuffer", hr);
     return FALSE;
   }

  // Define the buffer format
  wfFormat.wFormatTag = WAVE_FORMAT_PCM;
  wfFormat.nChannels = nChannels;
  wfFormat.nSamplesPerSec = nFrameRate;
  wfFormat.wBitsPerSample = 8 * sizeof(short);
  wfFormat.nBlockAlign = wfFormat.nChannels * wfFormat.wBitsPerSample / 8;
  wfFormat.nAvgBytesPerSec = wfFormat.nSamplesPerSec * wfFormat.nBlockAlign;
  wfFormat.cbSize = 0;  /* No extended format info. */

  // Set the primary buffer's format
  if((hr = IDirectSoundBuffer_SetFormat( pPrimaryBuffer, &wfFormat)) != DS_OK)
   { DSW_OnOutputError( dsw, "SetFormat", hr);
     return FALSE;
   }

  // ----------------------------------------------------------------------
  // Setup the secondary buffer description
  ZeroMemory(&secondaryDesc, sizeof(DSBUFFERDESC));
  secondaryDesc.dwSize = sizeof(DSBUFFERDESC);
  secondaryDesc.dwFlags =  DSBCAPS_GLOBALFOCUS | DSBCAPS_GETCURRENTPOSITION2;
  secondaryDesc.dwBufferBytes = bytesPerBuffer;
  secondaryDesc.lpwfxFormat = &wfFormat;

  // Create the secondary buffer
  if ((hr = IDirectSound_CreateSoundBuffer( dsw->pDirectSoundOut,
                  &secondaryDesc, &dsw->pOutputBuffer, NULL)) != DS_OK)
   { DSW_OnOutputError( dsw, "CreateSoundBuffer", hr);
     return FALSE;
   }

  // Lock the DS buffer
  if ((hr = IDirectSoundBuffer_Lock( dsw->pOutputBuffer, 0, dsw->iOutputSize, (LPVOID*)&pDSBuffData,
                                           &dwDataLen, NULL, 0, 0)) != DS_OK)
   { DSW_OnOutputError( dsw, "LockBuffer", hr);
     return FALSE;
   }

  // Zero the DS buffer
  ZeroMemory(pDSBuffData, dwDataLen);

  // Unlock the DS buffer .. we'll "Unlock" buffers in other functions, too,
  //                         so here's what "Unlock" means in this context.
  //     From the horse's mouth ("learn.microsoft.com", which replaced MSDN) :
  // > The IDirectSoundBuffer's Unlock() method releases a locked sound buffer.
  // >  [in] lpvAudioPtr1 :
  // >       Address of the value retrieved in the lplpvAudioPtr1 parameter
  // >       of the IDirectSoundBuffer::Lock method.
  // >  [in] dwAudioBytes1
  // >       Number of bytes actually written to the lpvAudioPtr1 parameter.
  // >       It should not exceed the number of bytes returned by the IDirectSoundBuffer::Lock method.
  // >  [in] lpvAudioPtr2
  // >       Address of the value retrieved in the lplpvAudioPtr2 parameter
  // >       of the IDirectSoundBuffer::Lock method.
  // >  [in] dwAudioBytes2
  // >       Number of bytes actually written to the lpvAudioPtr2 parameter.
  // >       It should not exceed the number of bytes returned by the
  // >       IDirectSoundBuffer::Lock method.
  // >
  // > Remarks (still about IDirectSoundBuffer::Unlock(Ptr1,Bytes1,Ptr2,Bytes2) ) :
  // > An application must pass both pointers, lpvAudioPtr1 and lpvAudioPtr2,
  // > returned by the IDirectSoundBuffer::Lock method to ensure the correct
  // > pairing of IDirectSoundBuffer::Lock and IDirectSoundBuffer::Unlock.
  // > The second pointer is needed even if 0 bytes were written to the second pointer.
  // > Applications must pass the number of bytes actually written to the two
  // > pointers in the parameters dwAudioBytes1 and dwAudioBytes2.
  // > Make sure the sound buffer does not remain locked for long periods of time.
  // >
  if ((hr = IDirectSoundBuffer_Unlock( dsw->pOutputBuffer, pDSBuffData, dwDataLen, NULL, 0)) != DS_OK)
   { DSW_OnOutputError( dsw, "UnlockBuffer", hr);
     return FALSE;
   }

  if( QueryPerformanceFrequency( &counterFrequency ) )
   {
     int framesInBuffer = bytesPerBuffer / (nChannels * sizeof(short));
     dsw->i64CounterTicksPerOutputBuffer.QuadPart = (counterFrequency.QuadPart * framesInBuffer) / nFrameRate;
     AddTraceMessage("i64CounterTicksPerOutputBuffer = %d\n", dsw->i64CounterTicksPerOutputBuffer.LowPart );
   }
  else
   {
     dsw->i64CounterTicksPerOutputBuffer.QuadPart = 0;
   }

  // Let DSound set the starting write position because if we set it to zero, it looks like the
  // buffer is full to begin with. This causes a long pause before sound starts when using large buffers.
  hr = IDirectSoundBuffer_GetCurrentPosition( dsw->pOutputBuffer, &playCursor, &dsw->dwWriteOffset );
  if( hr != DS_OK )
   { DSW_OnOutputError( dsw, "GetCurrentPosition", hr);
     return FALSE;
   }
  dsw->dblSamplesWrittenToOutput = dsw->dwWriteOffset / dsw->dwBytesPerFrame_out;
  /* printf("DSW_InitOutputBuffer: playCursor = %d, writeCursor = %d\n", playCursor, dsw->dwWriteOffset ); */

  return TRUE;
} // end DSW_InitOutputBuffer()


//---------------------------------------------------------------------------
BOOL DSW_StartOutput( DSoundWrapper *dsw )
{
  HRESULT  hr;
  QueryPerformanceCounter( &dsw->i64LastPlayTime );
  dsw->uLastPlayCursor = 0;
  dsw->dblSamplesPlayedToOutput = 0;
  hr = IDirectSoundBuffer_SetCurrentPosition( dsw->pOutputBuffer, 0 );
  if( hr != DS_OK )
   { DSW_OnOutputError( dsw, "SetCurrentPosition", hr);
     return FALSE;
   }

  // Start the buffer playback in a loop.
  if( dsw->pOutputBuffer != NULL )
   {
     hr = IDirectSoundBuffer_Play( dsw->pOutputBuffer, 0, 0, DSBPLAY_LOOPING );
     if( hr != DS_OK )
      { DSW_OnOutputError( dsw, "Play", hr);
        return FALSE;
      }
     dsw->fOutputOpened = TRUE;
   }

  return TRUE;
} // end DSW_StartOutput()


//---------------------------------------------------------------------------
BOOL DSW_StopOutput( DSoundWrapper *dsw )
{
  HRESULT  hr;

  if( dsw->fOutputOpened ) // <- kludge added 2024-01-06 to reduce multithreading issues
   {  dsw->fOutputOpened = FALSE;  // <- DO NOT CALL ANY AUDIO-OUTPUT-RELATED functions now !
      Sleep(50); // give worker threads and DSW_QueryOutputSpace(), etc, the chance
                 // to take notice, and STOP CALLING any DirectSound-OUTPUT-function.
   }

  // Stop the buffer playback
  if( dsw->pOutputBuffer != NULL )
   {
     hr = IDirectSoundBuffer_Stop( dsw->pOutputBuffer );
     if( hr != DS_OK )
      { DSW_OnOutputError( dsw, "Stop", hr);
        return FALSE;
      }
   }
  return TRUE;
} // end DSW_StopOutput()

//---------------------------------------------------------------------------
void DSW_CloseOutput( DSoundWrapper *dsw )                            // API
{ // Cleanup the sound OUTPUT buffers and the release related resources

  if( dsw->fOutputOpened ) // <- kludge added 2024-01-06 - details in DSW_StopOutput() ...
   {  dsw->fOutputOpened = FALSE;  // <- DO NOT CALL ANY AUDIO-OUTPUT-RELATED functions now !
      Sleep(50); // give worker threads and DSW_QueryOutputSpace(), etc, the chance
                 // to take notice, and STOP CALLING any DirectSound-OUTPUT-function.
   }

  if (dsw->pOutputBuffer)
   {
     IDirectSoundBuffer_Stop( dsw->pOutputBuffer );
     IDirectSoundBuffer_Release( dsw->pOutputBuffer );
     dsw->pOutputBuffer = NULL;
   }
  if (dsw->pDirectSoundOut != NULL )
   {
     IDirectSound_Release( dsw->pDirectSoundOut );
     dsw->pDirectSoundOut = NULL;
   }

} // end DSW_CloseOutput()


//---------------------------------------------------------------------------
BOOL DSW_ZeroEmptySpaceInOutputBuffer( DSoundWrapper *dsw )
{
  HRESULT hr;
  LPBYTE lpbuf1 = NULL;
  LPBYTE lpbuf2 = NULL;
  DWORD dwsize1 = 0;
  DWORD dwsize2 = 0;
  long  bytesEmpty;

  if( ! DSW_QueryOutputSpace( dsw, &bytesEmpty ) ) // updates dblSamplesPlayedToOutput (!)
   { return FALSE; // "last error" (string) already set because the above is an API function
   }
  if( bytesEmpty == 0 )
   { return TRUE;
   }
  // Lock free space in the DS
  hr = IDirectSoundBuffer_Lock( dsw->pOutputBuffer, dsw->dwWriteOffset, bytesEmpty, (void **) &lpbuf1, &dwsize1,
                                  (void **) &lpbuf2, &dwsize2, 0);
  if(hr != DS_OK )
   { DSW_OnOutputError( dsw, "LockOutputBuffer", hr);
     return FALSE;
   }

  // Copy the buffer into the DS
  ZeroMemory(lpbuf1, dwsize1);
  if(lpbuf2 != NULL)
   {
     ZeroMemory(lpbuf2, dwsize2);
   }

  // Update our buffer offset and unlock sound buffer
  dsw->dwWriteOffset = (dsw->dwWriteOffset + dwsize1 + dwsize2) % dsw->iOutputSize;
  IDirectSoundBuffer_Unlock( dsw->pOutputBuffer, lpbuf1, dwsize1, lpbuf2, dwsize2);
  dsw->dblSamplesWrittenToOutput += bytesEmpty / dsw->dwBytesPerFrame_out;
  return TRUE;
} // end DSW_ZeroEmptySpaceInOutputBuffer()

//---------------------------------------------------------------------------
BOOL DSW_QueryOutputSpace( DSoundWrapper *dsw, long *bytesEmpty )      // API
  // Typically called SHORTLY BEFORE 'generating data' for the output,
  //                  and writing those samples via DSW_WriteBlock() .
  // [out] *bytesEmpty : number of BYTES (not *AUDIO SAMPLE POINT*)
  //                  that may be appended to the circular buffer
  //                  via DSW_WriteBlock( with numBytes = *bytesEmpty ) .
  // Return value :
  //   TRUE on succees (even if *bytesEmpty is ZERO = "buffer completely full"),
  //   FALSE if there was a problem, and *bytesEmpty is invalid
  //         (dws->sz255LastOutputError may give a clue about WHAT went wrong).
{
  HRESULT hr;
  DWORD   playCursor;
  DWORD   writeCursor;
  long    numBytesEmpty;
  long    playWriteGap;

  // Protect against "application errors" (avoid crashing with a NULL pointer reference) :
  if( (dsw==NULL) || (dsw->pDirectSoundOut==NULL) ) // POINTERS ARE NOT BOOLEAN !
   { return FALSE;
   }
  if( (dsw->pOutputBuffer==NULL) || ( !dsw->fOutputOpened ) )
   { // ( avoid DSW_QueryOutputSpace() still being called from a worker thread,
     //   when another thread / the "main thread" has already CLOSED or STOPPED the output )
     DSW_SetFirstOutputError( dsw, "QueryOutputSpace called but Output not open" );
     return FALSE; // No valid output buffer ? Refuse to even 'query for output space' !
   }

  // Query to see how much room is in buffer.
  // Note: Even though writeCursor is not used, it must be passed to prevent DirectSound from dieing
  // under WinNT. The Microsoft documentation says we can pass NULL but apparently not.
  // Thanks to Max Rheiner for the fix.
  hr = IDirectSoundBuffer_GetCurrentPosition( dsw->pOutputBuffer, &playCursor, &writeCursor );
  if( hr != DS_OK )
   { DSW_OnOutputError( dsw, "GetCurrentPosition", hr);
     return FALSE;
   }
  // AddTraceMessage("playCursor", playCursor);
  // AddTraceMessage("dwWriteOffset", dsw->dwWriteOffset);

  // Determine size of gap between playIndex and WriteIndex that we cannot write into.
  playWriteGap = writeCursor - playCursor;
  if( playWriteGap < 0 )
   {  playWriteGap += dsw->iOutputSize; // unwrap
   }

  // DirectSound may not have a large enough playCursor so we cannot detect wrap-around.
  // Attempt to detect playCursor wrap-around and correct it.
  if( dsw->fOutputOpened && (dsw->i64CounterTicksPerOutputBuffer.QuadPart != 0) )
   {
     // How much time has elapsed since the last check ?
     LARGE_INTEGER   currentTime;
     LARGE_INTEGER   elapsedTime;
     long            bytesPlayed;
     long            bytesExpected;
     long            buffersWrapped;
     QueryPerformanceCounter( &currentTime );
     elapsedTime.QuadPart = currentTime.QuadPart - dsw->i64LastPlayTime.QuadPart;
     dsw->i64LastPlayTime = currentTime;

     // How many bytes does DirectSound say have been played ?
     bytesPlayed = playCursor - dsw->uLastPlayCursor;
     if( bytesPlayed < 0 )
      {  bytesPlayed += dsw->iOutputSize; // unwrap
      }
     dsw->uLastPlayCursor = playCursor;

     // Calculate how many bytes we would have expected to been played by now.
     bytesExpected = (long) ((elapsedTime.QuadPart * dsw->iOutputSize) / dsw->i64CounterTicksPerOutputBuffer.QuadPart);
     buffersWrapped = (bytesExpected - bytesPlayed) / dsw->iOutputSize;
     if( buffersWrapped > 0 )
      {
        // AddTraceMessage("playCursor wrapped! bytesPlayed", bytesPlayed );
        // AddTraceMessage("playCursor wrapped! bytesExpected", bytesExpected );
        playCursor += (buffersWrapped * dsw->iOutputSize);
        bytesPlayed += (buffersWrapped * dsw->iOutputSize);
      }

      // Bookkeeping for the number of sample points actually 'played to the ouput':
      dsw->dblSamplesPlayedToOutput += (bytesPlayed / dsw->dwBytesPerFrame_out);
   }
  numBytesEmpty = playCursor - dsw->dwWriteOffset;
  if( numBytesEmpty < 0 ) numBytesEmpty += dsw->iOutputSize; // unwrap offset

  /* Have we underflowed? */
  if( numBytesEmpty > (dsw->iOutputSize - playWriteGap) )
   {
     if( dsw->fOutputOpened )
      {
        ++dsw->nOutputUnderflows;
        // AddTraceMessage("underflow detected! numBytesEmpty", numBytesEmpty );
      }
     dsw->dwWriteOffset = writeCursor;
     numBytesEmpty = dsw->iOutputSize - playWriteGap;
   }
  *bytesEmpty = numBytesEmpty;
  return TRUE;
} // end DSW_QueryOutputSpace()

//---------------------------------------------------------------------------
BOOL DSW_WriteBlock( DSoundWrapper *dsw, char *buf, long numBytes )    // API
  // Returns TRUE when successful, otherwise FALSE
  //  (and, in the latter case, error-info in dsw->sz255LastOutputError)
  //
{
  HRESULT hr;
  LPBYTE lpbuf1 = NULL;
  LPBYTE lpbuf2 = NULL;
  DWORD dwsize1 = 0;
  DWORD dwsize2 = 0;

  // Protect against "application errors" (avoid crashing with a NULL pointer reference) :
  if( (dsw==NULL) || (dsw->pDirectSoundOut==NULL) ) // POINTERS ARE NOT BOOLEAN !
   { return FALSE;
   }
  if( (dsw->pOutputBuffer==NULL) || ( !dsw->fOutputOpened ) )
   { // ( avoid DSW_QueryOutputSpace() still being called from a worker thread,
     //   when another thread / the "main thread" has already CLOSED or STOPPED the output )
     DSW_SetFirstOutputError( dsw, "WriteBlock called but Output not open" );
     return FALSE; // No valid output buffer ? Refuse to even 'query for output space' !
   }

  // Lock free space in the DS
  hr = IDirectSoundBuffer_Lock( dsw->pOutputBuffer,
         dsw->dwWriteOffset,   // [in] dwWriteCursor :
             // > Offset, in bytes, from the start of the buffer to where the lock begins.
             // > This parameter is ignored if DSBLOCK_FROMWRITECURSOR is specified in the dwFlags parameter.
         numBytes,               // [in] dwWriteBytes :
             // > Size, in bytes, of the portion of the buffer to lock.
             // > Note that the sound buffer is conceptually circular.
         (void **) &lpbuf1,      // [out] lplpvAudioPtr1 :
             // > Address of a pointer to contain the first block of the sound buffer to be locked.
         &dwsize1,               // [out] lpdwAudioBytes1 :
             // > Address of a variable to contain the number of bytes pointed to
             // > by the lplpvAudioPtr1 parameter. If this value is less than the
             // > dwWriteBytes parameter, lplpvAudioPtr2 will point to a second block of sound data.
         (void **) &lpbuf2,      // [out] lplpvAudioPtr2 :
             // > Address of a pointer to contain the second block of the
             // > sound buffer to be locked. If the value of this parameter is NULL,
             // > the lplpvAudioPtr1 parameter points to the entire
             // > locked portion of the sound buffer.
         &dwsize2,               // [out] lpdwAudioBytes2 :
             // > Address of a variable to contain the number of bytes
             // > pointed to by the lplpvAudioPtr2 parameter.
             // > If lplpvAudioPtr2 is NULL, this value will be 0.
         0 ); // [in] dwFlags : Flags modifying the lock event. The following flags are defined:
             // > DSBLOCK_FROMWRITECURSOR :	Locks from the current write position,
             // >   making a call to IDirectSoundBuffer::GetCurrentPosition unnecessary.
             // >   If this flag is specified, the dwWriteCursor parameter is ignored.
             // > DSBLOCK_ENTIREBUFFER : Locks the entire buffer.
             // >   The dwWriteBytes parameter is ignored.
  if(hr != DS_OK )
   { DSW_OnOutputError( dsw, "LockOutputBuffer", hr);
     return FALSE;
   }

  // Copy the buffer into the DS  (possibly "before" and "after" the circular wrapping index)
  CopyMemory(lpbuf1, buf, dwsize1);
  if(lpbuf2 != NULL)
   {
     CopyMemory(lpbuf2, buf+dwsize1, dwsize2);
   }

  // Update our buffer offset and unlock sound buffer
  dsw->dwWriteOffset = (dsw->dwWriteOffset + dwsize1 + dwsize2) % dsw->iOutputSize;
  hr = IDirectSoundBuffer_Unlock( dsw->pOutputBuffer, lpbuf1, dwsize1, lpbuf2, dwsize2);
  dsw->dblSamplesWrittenToOutput += numBytes / dsw->dwBytesPerFrame_out;
  if(hr != DS_OK )
   { DSW_OnOutputError( dsw, "UnlockLockOutputBuffer", hr);
     return FALSE;
   }
  else
   { return TRUE;
   }

} // end DSW_WriteBlock()


//---------------------------------------------------------------------------
DWORD DSW_GetOutputStatus( DSoundWrapper *dsw )
  // Straight from the horse's mouth (besides "DirectSound is a legacy feature"):
  // > This method retrieves the current status of the sound buffer.
  // A-ha. And what does this status tell us; the number of UNUSED BYTES or SAMPLES ?
  //       Of course not. Microsoft's "dwStatus" ..
  // > ... can be a combination of the following flags :
  // >     Flag              Description
  // > DSBSTATUS_BUFFERLOST  The buffer is lost and must be restored before it can be played or locked.
  // > DSBSTATUS_LOOPING     The buffer is being looped. If this value is not set,
  // >                       the buffer will stop when it reaches the end of the sound data.
  // >                       Note that if this value is set, the buffer must also be playing.
  // > DSBSTATUS_PLAYING 	  The buffer is playing. If this value is not set,
  // >                       the buffer is stopped.
{
  DWORD status;

  // Protect against "application errors" (avoid crashing with a NULL pointer reference) :
  if( (dsw==NULL) || (dsw->pDirectSoundOut==NULL) ) // POINTERS ARE NOT BOOLEAN !
   { return DSERR_INVALIDPARAM;
   }
  if( (dsw->pOutputBuffer==NULL) || ( !dsw->fOutputOpened ) )
   { // ( avoid DSW_QueryOutputSpace() still being called from a worker thread,
     //   when another thread / the "main thread" has already CLOSED or STOPPED the output )
     DSW_SetFirstOutputError( dsw, "GetOutputStatus called but Output not open" );
     return DSERR_INVALIDPARAM; // No valid output buffer ? Refuse to even 'query for output status' !
   }

  if (IDirectSoundBuffer_GetStatus( dsw->pOutputBuffer, &status ) != DS_OK)
   { return( DSERR_INVALIDPARAM );
   }
  else
   { return( status );
   }
} // end DSW_GetOutputStatus()



//---------------------------------------------------------------------------
// Audio INPUT ("Capture" a la DirectSound) ... usage summary :
// 1.) Call DSW_OpenInputDevice() with the DESCRIPTIVE NAME (not a "GUID") .
// 2.) Set sample rate, number of channels, and buffer size (thus latency)
//        per DSW_InitInputBuffer(). In Microsoft geek speak, that's
//        IDirectSoundCapture_CreateCaptureBuffer() . Someone (not MS) explained:
//     > The DirectSound capture buffer object, represented by the
//     > IDirectSoundCaptureBuffer8 interface, represents a buffer used for
//     > receiving data from the input device. Like playback buffers,
//     > this buffer is conceptually circular: when input reaches the end
//     > of the buffer, it automatically starts again at the beginning.
//         What follows is from an old "Microsoft DirectX 9.0" SDK, renumbered:
//     > Capturing a sound consists of the following steps:
// 3.) Start the buffer by calling the IDirectSoundCaptureBuffer8::Start method.
//       Normally you should pass DSCBSTART_LOOPING in the dwFlags parameter
//       so that the buffer will keep running continuously rather than stopping
//       when it reaches the end. Audio data from the input device begins filling
//       the buffer from the beginning.
// 4.) Wait until the desired amount of data is available.
//       For one method of determining when the capture cursor reaches a certain point,
//       see Capture Buffer Notification.
// 5.) When sufficient data is available, lock a portion of the capture buffer
//       by calling the IDirectSoundCaptureBuffer8::Lock() method.
//       To make sure you are not attempting to lock a portion of memory
//       that is about to be used for capture, you can first obtain the position
//       of the read cursor by calling IDirectSoundCaptureBuffer8::GetCurrentPosition().
//       As parameters to the Lock method, you pass the size and offset
//       of the block of memory you want to read. The method returns a pointer
//       to the address where the memory block begins, and the size of the block.
//       If the block wraps around from the end of the buffer to the beginning,
//       two pointers are returned, one for each section of the block.
//       The second pointer is NULL if the locked portion of the buffer does not wrap around.
// 6.) Copy the data from the buffer, using the addresses and block sizes
//       returned by the Lock method.
// 7.) Unlock the buffer with the IDirectSoundCaptureBuffer8::Unlock method.
// 8.) Repeat steps 2 to 5 until you are ready to stop capturing data.
//       Then call the IDirectSoundCaptureBuffer8::Stop() method.
//---------------------------------------------------------------------------



//---------------------------------------------------------------------------
BOOL DSW_OpenInputDevice(     // aka "Create the capture device object" ...
           DSoundWrapper *dsw,
     const char *sz255DevName ) // audio *DEVICE NAME* (not a stuid 'GUID')
  // Return value : TRUE = ok,   FALSE = failure (e.g. device name not found) .
  //     If the function failed, dsw->sz255LastInputError may give a clue why.

{
  GUID *pGUID = NULL;
  HRESULT hr;
  BOOL found_device = FALSE;
  int i;

  strncpy( dsw->sz255InputDeviceName, sz255DevName, 255 );

  if( sz255DevName != NULL ) // use a specific device (not the "Default"-thingy) ?
   { if( dsw->sz255InputDeviceName != sz255DevName ) // <- compare ADDRESS, not STRING CONTENT
      { strncpy( dsw->sz255InputDeviceName, sz255DevName, 255 );
        // '--> this dinosaur will also crash when copying a string to itself !
      }
     for( i=0; (i<dsw->m_nInputDevices) && (i<DSW_MAX_ENUMERATED_DEVICES); ++i)
      { if( strcmp(sz255DevName, dsw->m_sInputDevices[i].sz255DevName) == 0 )
         { if( dsw->m_sInputDevices[i].guid.Data1 != 0 )
            { pGUID = &dsw->m_sInputDevices[i].guid;
            }
           found_device = TRUE;
           break;
         }
      }
   }    // end if( sz255DevName != NULL )
  else
   { dsw->sz255InputDeviceName[0] = '\0';
   }

  if( ! found_device )
   { strcpy( dsw->sz255LastInputError, "OpenInput: Device not found" );
     return FALSE;
   }

#if( ! _TEST_VIRDEF_TROUBLE_ )
  hr = DirectSoundCaptureCreate( pGUID, &dsw->pDirectSoundIn, NULL );
      // ,------------------------------------'                   |
      // '--> "Address of a pointer to a DirectSoundCapture       |
      //       object created in response to this function."      |
      //                                                          |
      //  ,-------------------------------------------------------'
      //  '--> "pUnkOuter: Controlling unknown of the aggregate.
      //                   Its value must be NULL." (the joys of Windows..)
#else //  _TEST_VIRDEF_TROUBLE_ : don't call the above 'suspect' at all
  hr = !DS_OK;
#endif // _TEST_VIRDEF_TROUBLE_ ?
  return ( hr == DS_OK );
} // end DSW_OpenInputDevice()


//---------------------------------------------------------------------------
BOOL DSW_InitInputBuffer(  // aka "Create a Capture Buffer" ...
        DSoundWrapper *dsw,
        unsigned long nFrameRate,
        int nChannels,
        int bytesPerBuffer ) // <- make this AT LEAST long enough for 20 ms of buffering,
        // i.e. nFrameRate * 2 (bytes per 'short') * nChannels / (1000ms/20m),
        // e.g. 48000 Hz   * 2 bytes * 1 channel * (20ms/1000ms) = 1920 sample, "round up" to 2048 samples.
{
  DSCBUFFERDESC  captureDesc;
    // > One of the parameters to the method is a DSCBUFFERDESC structure that
    // > describes the characteristics of the desired buffer.
    // > The last member of this structure is a *pointer to* a WAVEFORMATEX structure,
    // > which must be initialized with the details of the desired WAV format.
    // (a left-over from the ancient Windows "MultiMedia" system)
  WAVEFORMATEX   wfFormat;
  HRESULT        hr;
  double dblCurrentTime;

  if( (dsw==NULL) || (dsw->pDirectSoundIn==NULL) )
   { return FALSE; // "application error" (avoid crashing with a NULL pointer reference)
   }

  // Define the buffer format  (in the DirectX SDK: "Set up WAV format structure", e.g. in CreateBasicBuffer)
  memset( &wfFormat, 0, sizeof(WAVEFORMATEX) );
  wfFormat.wFormatTag      = WAVE_FORMAT_PCM;
  wfFormat.nChannels       = nChannels;
  wfFormat.nSamplesPerSec  = nFrameRate;
  wfFormat.wBitsPerSample  = 8 * sizeof(short);
  wfFormat.nBlockAlign     = wfFormat.nChannels * (wfFormat.wBitsPerSample / 8);
  wfFormat.nAvgBytesPerSec = wfFormat.nSamplesPerSec * wfFormat.nBlockAlign;
  wfFormat.cbSize          = 0;   /* No extended format info. */
  dsw->dwInputFifoSizeInBytes = bytesPerBuffer;
  dsw->dwBytesPerFrame_in  = nChannels * sizeof(short);


  // ----------------------------------------------------------------------
  // Setup the secondary buffer description
  ZeroMemory(&captureDesc, sizeof(DSCBUFFERDESC));
  captureDesc.dwSize = sizeof(DSCBUFFERDESC);
  captureDesc.dwFlags =  0;
  captureDesc.dwBufferBytes = bytesPerBuffer;
  captureDesc.lpwfxFormat = &wfFormat;

  // Create the capture buffer
  if ((hr = IDirectSoundCapture_CreateCaptureBuffer( dsw->pDirectSoundIn,
                &captureDesc, &dsw->pInputBuffer, NULL)) != DS_OK)
   { DSW_OnInputError( dsw, "CreateCaptureBuffer", hr );
     return FALSE;
   }
  dsw->dwInputBufferTailIndex = 0;  // reset last read position to start of buffer (ex: "uReadOffset").
       //  Somehow similar with a FIFO tail index in a classic lock-free circular FIFO.
       //  The FIFO head index is SIMILAR TO what Microsoft calls "capture cursor";
       //  see text quoted from a stoneage 'Direct-something SDK' further above)
       //  The FIFO tail index is where we ask Mr. Direct Sound to "lock" a part
       //  of the entire FIFO, so we can 'memcopy' samples from it in DSW_ReadBlock().

  // Prepare our 'helpers for bookkeeping' / detect timing problems, overflows, etc
  dblCurrentTime = DSW_ReadHighResTimestamp_s();
  dsw->nInputOverflows         = 0;
  dsw->dblInputSamplingRate_Hz = (double)nFrameRate;
  dsw->dblInputWrapInterval_s  = (double)bytesPerBuffer / (double)( sizeof(short) * nFrameRate * nChannels );
       // '--> if the recommended buffer size was specified,
       //      the result should be at least 0.02 seconds !
  dsw->dblLastInputTime_s = dsw->dblTimeAtLastCaptureHistoryEntry_s = dsw->dblTimestampAtInputBufferTail_s = dblCurrentTime; // here: set in DSW_InitInputBuffer()

  // Prepare the precise measurement of the INPUT SAMPLING RATE (actually, dblInputSecondsPerSample):
  DSW_RestartSampleRateMeasurement( dsw ); // (called again from DSW_StartInput() .. anyway)

  // Do NOT set dsw->fInputOpened here yet, because without also calling
  // DSW_StartInput(), some mimosa deep inside DirectSound crashed
  //     when a worker thread tried to call e.g. DSW_ReadBlock() !


  return TRUE;
} // end DSW_InitInputBuffer()


//---------------------------------------------------------------------------
BOOL DSW_StartInput( DSoundWrapper *dsw )
{
  HRESULT hr;

  if( (dsw==NULL) || (dsw->pDirectSoundIn==NULL) )
   { return FALSE; // "application error" (avoid crashing with a NULL pointer reference)
   }

  dsw->dwCaptureHistoryIndex = 0;

  // Prepare the precise measurement of the INPUT SAMPLING RATE (actually, dblInputSecondsPerSample):
  DSW_RestartSampleRateMeasurement( dsw ); // (called again from DSW_StartInput() .. anyway)

  // Start the capturing into the input buffer
  if( dsw->pInputBuffer != NULL )
   {
     hr = IDirectSoundCaptureBuffer_Start( dsw->pInputBuffer, DSCBSTART_LOOPING );
     dsw->dblLastInputTime_s = dsw->dblTimestampAtInputBufferTail_s = DSW_ReadHighResTimestamp_s(); // here: initialized again in DSW_StartInput()
     if( hr != DS_OK )
      { DSW_OnInputError( dsw, "StartInput", hr );
        return FALSE;
      }
     dsw->fInputOpened = TRUE;  // allow calling e.g.
          // IDirectSoundCaptureBuffer_GetCurrentPosition(),
          // IDirectSoundCaptureBuffer_Lock(),
          // IDirectSoundCaptureBuffer_Unlock(),
          // and who-knows-what may be called from a 'worker thread'.

     return TRUE;
   }
  return FALSE;  // nothing to START
} // end DSW_StartInput()


//---------------------------------------------------------------------------
BOOL DSW_StopInput( DSoundWrapper *dsw )
{
  HRESULT hr;

  if( dsw->fInputOpened ) // <- kludge added 2024-01-06 to reduce multithreading issues
   {  dsw->fInputOpened = FALSE;  // <- DO NOT CALL ANY AUDIO-INPUT-RELATED functions now !
      Sleep(50); // give some worker thread the chance to stop calling us
                 // before we REALLY "stop the input"; and prevent any of our
                 // 'wrapper functions' to invoke INPUT-related DirectSound functions.
   }

  if( (dsw==NULL) || (dsw->pDirectSoundIn==NULL) )
   { return FALSE; // "application error" (avoid crashing with a NULL pointer reference)
   }

  // Stop capturing" (feeding samples into the input buffer)
  if( dsw->pInputBuffer != NULL )
   {
     hr = IDirectSoundCaptureBuffer_Stop( dsw->pInputBuffer );
     if( hr != DS_OK )
      { DSW_OnInputError( dsw, "StopInput", hr );
        return FALSE;
      }
     return TRUE;
   }
  return FALSE;  // nothing to STOP
} // end DSW_StopInput()

//---------------------------------------------------------------------------
BOOL DSW_IsInputOpened( DSoundWrapper *dsw ) // special service for RCW Keyer's DspThread()...
{ if( dsw != NULL )
   { return dsw->fInputOpened;
   }
  else
   { return FALSE; // If there is no "Direct Sound Wrapper", the input isn't open :)
   }
} // end DSW_IsInputOpened()


//---------------------------------------------------------------------------
void DSW_RestartSampleRateMeasurement( DSoundWrapper *dsw )
  // Prepare the precise measurement of the INPUT SAMPLING RATE (actually, dblInputSecondsPerSample)
{
  dsw->dblInputSecondsPerSample = 1.0 / dsw->dblInputSamplingRate_Hz; // "initial value" before a sufficient number of samples have been captured
  dsw->dblInputSRCalibTime_s = 0.0;  // there's no valid START TIME for the sample rate measurement yet !
         // (the timestamp for this MUST be taken in DSW_QueryInputFilled(),
         //  as close to the circular buffer index wrap-around as possible)
  dsw->dwInputSRCalibSampleCounter = 0; // nothing counted in the current 'gate time' yet
  dsw->iInputSRCalib_AverageFilterNumEntries = 0;
  dsw->dblInputSRCalibError_ppm = 0.0;  // nothing MEASURED yet, assume "no error" (zero ppm for the input sampling rate)
} // end DSW_RestartSampleRateMeasurement()


//---------------------------------------------------------------------------
BOOL DSW_QueryInputFilled( DSoundWrapper *dsw,
        long *bytesFilled ) // [out] number of BYTES (not sample points alias "frames") available for reading
  // This function must be sufficiently robust to NOT CRASH
  //      when called even if Mr. DirectSound's *input* is not open at all !
  //      For reliable bookkeeping (watching for possible input overruns, etc),
  //      the time between calls of DSW_QueryInputFilled() should not exceed
  //      HALF the buffer time (=time it takes to fill the entire buffer
  //      allocated by DSW_InitInputBuffer() .. e.g. at least every 10 ms
  //      when DirectSound's input buffer ("capture buffer") can store 20 ms.
{
  HRESULT hr;
  DWORD dwCapturePos;
  DWORD dwReadPos;
  long  filled, nSamplesAvailable;
  double d, dblDeltaT_s, dblCurrentTime_s, dblSecondsSinceLastRefill;
  int    i,n;
  // ex: int nBytesAddedSinceLastCall;
  T_DSW_InputCaptureHistoryEntry *pCH;


  dblCurrentTime_s = DSW_ReadHighResTimestamp_s();


  // Protect against "application errors" (avoid crashing with a NULL pointer reference) :
  if( (dsw==NULL) || (dsw->pDirectSoundIn==NULL) )
   { return FALSE;  // <- got here when NO INPUT DEVICE was selected or opened.
                    //    "That's a feature, not a bug" (B. Gates)
                    //  In RCW Keyer / CwDSP.c, called from DspThread() .
   }
  if( dsw->pInputBuffer==NULL )
   { DSW_SetFirstOutputError( dsw, "DSW_QueryInputFilled called but Input not open" );
     return FALSE; // No valid input buffer ? Refuse to even 'query for input' !
   }

  // Query to see how much data is in buffer.
  // > We don't need the capture position but sometimes DirectSound
  // > doesn't handle NULLS correctly, so let's pass a pointer just to be safe.
  hr = IDirectSoundCaptureBuffer_GetCurrentPosition( dsw->pInputBuffer, &dwCapturePos, &dwReadPos );
  // ,----------'
  // '--> From a MORE RECENT description (about "IDirectSoundCaptureBuffer8", note the EIGHT) :
  // > Capture Buffer Notification
  // > To copy data out of the capture buffer at regular intervals,
  // > your application has to know when the read cursor, which is the point
  // > up to which it is safe to read data, reaches certain points in the buffer.
  // > One way to get this information is to poll the position of the read cursor
  // > using IDirectSoundCaptureBuffer8::GetCurrentPosition(). [WB: .. as used here..]
  // >
  // > A more efficient way is to use notifications. (..)  [WB: We don't use that. KISS.]
  // >
  if( hr != DS_OK ) // IDirectSoundCaptureBuffer_GetCurrentPosition() didn't co-operate ->
   { DSW_OnInputError( dsw, "GetCurrentInputPos", hr );
     return FALSE;
   }

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // Relation between DirectSound's "dwCapturePos", "dwReadPos",            .
  //              and our own TAIL INDEX into the circular capture buffer:  .
  //                                                                        .
  //          |<----Block from DSW_InitInputBuffer(), e.g. 8192 byte---->|  .
  //          |     for a maximum of CWDSP_INPUT_BUFFER_NSAMPLES = 4096  |  .
  //           __________________________________________________________   .
  //          |    Circular "Capture Buffer"                             |  .
  //          |             |<-------- "still"  VALID------------>|      |  .
  //          | ............|  Already read   Waiting to be read  |......|  .
  //          | being filled|<------------->:<------------------->|      |  .
  //          |_____________________________:____________________________|  .
  //                       /|\             /|\                   /|\        .
  //                        |               |                     |         .
  //               dwCapturePos:            |                     |         .
  //           ~~~  In use by the           |<------"filled"----->|         .
  //                audio driver ?      dwInputBufferTailIndex    |         .
  //      (CapturePos must  :           (our FIFO tail index)    dwReadPos  .
  //       NEVER catch up   :                                     :(if our TAIL index
  //       with our TAIL ;) :             ........................: catches up with ReadPos,
  //    ....................:             :                         no samples are available for READING)
  //    :                               > "The read cursor, which is the point
  //    :                               > up to which it is safe to read data.
  //    :
  // > "The capture cursor is ahead of the read cursor. (???)
  // Somewhere else:
  // > "The capture cursor is at the end of the block of data that is
  // >              currently being copied into the buffer".
  // >  The data after the read position up to
  // >  and including the capture position
  // >  is not necessarily valid data."
  // And even somewhere else ("learn.microsoft.com" about "Direct Sound" / "Capture Buffer Information"):
  // >  You can copy data from the buffer only **UP TO** the "Read Cursor".
  //
  //     Hmmm.  See the TEST RESULTS from 'watching' all these indices further below:
  //  * "dwCapturePos" incremented in FINER STEPS, sometimes by 960 samples (@ fs=48 kHz)
  //  * "dwReadPos" indeed LAGS "dwCapturePos", but at COARSER STEPS,
  //                often incremented in 3000-sample-steps, i.e. every 62.5 milliseconds
  //    Depending on where the leading "Capture Position" is, relative to the wrapping point,
  //    the situation may be any of the following:
  //      __________________________________________________________   .
  //     |    Circular "Capture Buffer"                             |  .
  // (1) |----valid--->|.......INVALID DATA...........|<---valid----|  .
  //     |__________________________________________________________|  .
  //       /|\        /|\                            /|\               .
  //        |- - - - > |                              |                .
  //      Tail-    READ POS                       CAPTURE POS          .
  //      index                                                        .
  //      __________________________________________________________   .
  //     |    Circular "Capture Buffer"                             |  .
  // (2) |...INVALID...|<----------valid samples----->|...INVALID...|  .
  //     |__________________________________________________________|  .
  //                  /|\                 /|\        /|\               .
  //                   |                   |- - - - > |                .
  //            CAPTURE POS              Tail-       READ POS          .
  //                                     index                         .
  //                            (not seen by DirectSound but 'local')  .
  //
  //
  // Accidentally found at "learn.microsoft.com", DirectSound / Programming Guide / Capturing Waveforms /
  //                       "Capture Buffer Information" :
  // > The IDirectSoundCaptureBuffer8::GetCurrentPosition method returns the
  // > offsets of the read and capture cursors within the buffer.
  // > The read cursor is at the end of the data that has been fully captured
  // > into the buffer at this point. The capture cursor is at the end of the block
  // > of data that is currently being copied from the hardware.
  // > You can safely copy data from the buffer only up to the read cursor.
  //     (A-ha. A "Cursor" is just a "Position". )
  //
  // Summary: "The data after [in time] dwReadPos, up to (but not including) "CapturePos"
  //     CONTAIN VALID DATA that we may LOCK / READ / UNLOCK ?
  // ex: if( dwCapturePos != dsw->dwPrevInputCapturePos ) // <- often stepped by  960 BYTES = 480 SAMPLES  = 10 milliseconds (but USELESS for us)
  // ex: if( dwReadPos != dsw->dwPrevInputReadPos ) // <- often stepped by 3000 BYTES = 1500 SAMPLES = 31.25 milliseconds (*)
  if( dwReadPos < dsw->dwPrevInputReadPos ) // <- this should happen every 4096 samples / 48 kHz = 85.333 milliseconds
   { // The "Read Position" aka "Read Cursor" (funny name for the HEAD INDEX)
     // has WRAPPED AROUND again, so emit this to the "capture history index" :
     pCH = &dsw->sCaptureHistory[ dsw->dwCaptureHistoryIndex ];
     dsw->dwCaptureHistoryIndex = ( dsw->dwCaptureHistoryIndex+1 ) & 255;
#   if(0)  // OLD STUFF (when trying to track EVERY step of dwCapturePos and dwReadPos) :
     nBytesAddedSinceLastCall = (int)dwReadPos - (int)dsw->dwPrevInputReadPos;
     if( nBytesAddedSinceLastCall < 0 )
      {  nBytesAddedSinceLastCall += dsw->dwInputFifoSizeInBytes; // circular FIFO wrap-around
      }
     dsw->dblSamplesCapturedFromInput += nBytesAddedSinceLastCall / dsw->dwBytesPerFrame_in;
#   else  // simplified 2025-04 (when getting here only ONCE for every circular buffer wrap-around) :
     dsw->dblSamplesCapturedFromInput += dsw->dwInputFifoSizeInBytes / dsw->dwBytesPerFrame_in;
#   endif
     dsw->dblLastInputTime_s = dblCurrentTime_s; // here: set in DSW_QueryInputFilled() when Mr. Direct Sound promised to have more "READABLE" bytes

     pCH->dwCapturePos = dwCapturePos;
     pCH->dwReadPos    = dwReadPos;
     pCH->dwTotalSamplesRead = dsw->dwTotalSamplesReadFromInput;
     pCH->dblTimestamp_s = dblCurrentTime_s; // here: unfiltered timestamp close to the 'step' in dwReadPos
     // When inspected on the 'Debug' tab at random intervals, got surprising results:
     //  * dwReadPos ALWAYS stepped by 3000 bytes, even when wrapping around
     //              from the end of the buffer to the start.
     //              With the buffer size = 2*CWDSP_INPUT_BUFFER_NSAMPLES = 8192 bytes,
     //              one would expect dwReadPos to wrap from ( 6000 + 3000 ) & 8191 = 808.
     //              But instead it wrapped from 6000 to ZERO !
     //               ( So only 8192-6000 = 2192 bytes in the last fragment )

#   if( DSW_INPUT_TIMESTAMPING_METHOD==0)  // 0=do not provide TIMESTAMPS for input audio samples at all
     // That's easy, but it's impossible to precisely align data from DIFFERENT KINDS OF INPUTS without timestamps
#   elif(DSW_INPUT_TIMESTAMPING_METHOD==1) // 1=based on counted samples and 'self-calibrated' sampling rate:
     // At THIS POINT, shortly after the "Read Position" aka "Read Cursor",
     // reported by DirectSound for the circular input buffer, has wrapped around
     // from e.g. byte index 8191 to zero. With 16 bit/sample, mono, and fs=48 kHz
     // this theoretically happens every 4096 samples / 48 kHz = 85.33 milliseconds.
     // But due to the uncertainty in the timestamps of at least +/-10 milliseconds,
     //  the measurement of the SAMPING RATE or (inverse) SAMPLING PERIOD
     //  needs a much longer gate time, as implemented in the following function:
     DSW_CalibrateInputSampleRate(dsw, dblCurrentTime_s,
         dsw->dwInputFifoSizeInBytes / dsw->dwBytesPerFrame_in); // [in] nSamplesSinceLastCall
#   else // DSW_INPUT_TIMESTAMPING_METHOD==2 : old stuff with too much 'timestamp jitter'
     // Prevent the sample-counter-based dsw->dblTimestampAtInputBufferTail_s 'drifting away' .
     // The goal is an accurate, low-jitter timestamp of the sample at dsw->dwInputBufferTailIndex(!).
     //   [in]  dwCapturePos, dwReadPos : BYTE-INDICES into DirecSound's circular buffer; see ASCII sketch above.
     //             ("Capture Pos" *LEADS* "Read Pos" in a CAPTURE buffer;
     //              again, see the ASCII sketch above for which part of the buffer may actually be READ FROM).
     //   [in]  dblCurrentTime_s : current timestamp; unfortunately we cannot accurately predict
     //              to which DirectSound-Capture-Buffer-Sample-Index it applies to ! (details below)
     //   [out] dsw->dblTimestampAtInputBufferTail_s : Applies to the already acquired sample at dsw->dwInputBufferTailIndex
     //
     // The closest we can get (for a relation between (DSound-)INPUT BUFFER INDEX
     //         and TIMESTAMP) is most likely at 'dwCapturePos' (not 'dwReadPos')
     //         - see ASCII sketch of the Circular "Capture Buffer" further below.
     nSamplesAvailable = dwCapturePos - dsw->dwInputBufferTailIndex; // here: still a BYTE OFFSET,
        // '--> Measured between dwCapturePos (which 'leads' dwReadPos)
        //         and dwInputBufferTailIndex (which 'lags'  dwReadPos, and at most catch up with it).
     if( nSamplesAvailable < 0 ) // circular buffer wrap between the momentary "Capture Posision" and the application's FIFO TAIL index
      {  nSamplesAvailable += dsw->dwInputFifoSizeInBytes; // unwrap the circular buffer offset
      }
     if( dsw->dwBytesPerFrame_in > 1 ) // convert the BYTE OFFSET into a NUMBER OF AUDIO SAMPLES
      {  nSamplesAvailable /= dsw->dwBytesPerFrame_in;  // NOW it's really a number of sample points, not BYTES
      }
     if( dsw->dblInputSamplingRate_Hz > 0 ) // convert the NUMBER OF SAMPLES into an interval in SECONDS
      { d = dblCurrentTime_s - (double)nSamplesAvailable / dsw->dblInputSamplingRate_Hz;
        // '--> d = value "expected" in dsw->dblTimestampAtInputBufferTail_s
        //       (the sample at dwInputBufferTailIndex always LAGS the "CapturePos",
        //        because the sample at "CapturePos" is not available yet.
        //        Thus 'd' must be OLDER and thus LOWER than dblCurrentTime_s.)
        dblDeltaT_s = d - dsw->dblTimestampAtInputBufferTail_s; // -> difference in seconds, ideally ZERO.
        // '---> Results are stored in dsw->sCaptureHistory[].iTimestampJitter_ms,
        //       and displayed as "TS Jitter" on the 'Debug' tab in the GUI.
        //       Example with alpha = 0.001 for the lowpass filter below:
        //    > TS Jitter : 29 -11 -12 3 25 -14 -21 0 22 38  ms (that's an AWFUL LOT)
        //       Expected timestamp jitter:
        //       6 milliseconds due to the calling interval (DSP thread)
        //     + 10 milliseconds because dwCapturePos seemed to increment by 960-byte-steps,
        //          i.e. 480-sample-steps at fs=48 kHz -> Windows only seems to
        //          update dwCapturePos every 480 / 48 kHz = 10 ms .
#      define L_MAX_TIMESTAMP_ERROR_IN_SECONDS 0.5
        if( (dblDeltaT_s >= -L_MAX_TIMESTAMP_ERROR_IN_SECONDS) && (dblDeltaT_s <= L_MAX_TIMESTAMP_ERROR_IN_SECONDS) )
         { // reasonably small 'delta T' -> slowly steer dsw->dblTimestampAtInputBufferTail_s to minimize the error:
           dsw->dblTimestampAtInputBufferTail_s += 0.01 * dblDeltaT_s;
         }
        else // too large error -> Restart with the 'new' calculated timestamp (d)
         { dsw->dblTimestampAtInputBufferTail_s = d;
           // (should get here after being paused on a breakpoint,
           //  because dblCurrentTime_s (from "QueryPerformanceCounter") keeps ticking,
           //  but audio samples 'captured' from DirectSound will be SKIPPED)
         }
      } // end if( dsw->dblInputSamplingRate_Hz > 0 )
     else // without a valid sampling rate, cannot correct dblTimestampAtInputBufferTail_s
      { dblDeltaT_s = 0.0;
      }
#   endif // DSW_INPUT_TIMESTAMPING_METHOD==?

     pCH->iDeltaT_ms   = (int)( 0.5 + 1e3 * (dblCurrentTime_s - dsw->dblTimeAtLastCaptureHistoryEntry_s) );
     pCH->iTimestampJitter_ms  = (int)( 1e3 * dblDeltaT_s );
     dsw->dblTimeAtLastCaptureHistoryEntry_s = dblCurrentTime_s;
     if( dsw->dwCaptureHistoryIndex == 255 )
      {  dsw->dwCaptureHistoryIndex = dsw->dwCaptureHistoryIndex; // <- place for a "conditional breakpoint" !
      }
     // WATCHED the result with Borland's debugger ( "MyDirectSound.sCaptureHistory[0] .. [10]" ),
     //   when using CW_DSP_INPUT_BUFFER_NSAMPLES = 4096 = circa 85 milliseconds "bufferable",
     //   and when adding a few history entry on each step of the CAPTURE- (not READ-) "cursor":
     //  ->     dwCapturePos  dwReadPos TotalSamplesRead iDeltaT_ms  iTimestampJitter_ms
     //             [BYTES]     [BYTES]    [SAMPLES]      /  Theory
     //    [0] = {    960,         0,        0        ,    26 / 10     33
     //    [1] = {   1920,         0,        0        ,     7 / 10       50
     //    [2] = {   2880,         0,        0        ,     6 / 10       66
     //    [3] = {   3840,      3000,        0        ,    12 / 10       88
     //    [4] = {   4800,      3000,        0        ,    12 / 10       47
     //    [5] = {   5760,      6000,      1500       ,     6 ...        63
     //    [6] = {   6720,      6000,      1500       ,    12 ...        85
     //    [7] = {   7680,      6000,      3000       ,    12            44
     //    [8] = {    448,         0,      3000       ,     6            60
     //    [9] = {   1408,         0,      4096       ,    12            37
     //               |                                     |
     //               |                               6 or 12 ms due to the
     //               |                               calling interval - see DspThread() : "Sleep(5)" in each loop
     //               '--> Ok, confirms "dwCapturePos" *LEADS* "dwReadPos" .
     //   Notes: (1) "dwReadPos" doesn't have anything to do
     //              with the application's own BUFFER TAIL INDEX !
     //          (2) "dwCapturePos" obviously *LEADS* "dwReadPos"
     //              by up to (11520-9000=)2520 samples. Those samples are INVALID
     //              in the buffer.
     //
     //          (3) If only the samples between
     //
     //          (4) DSW_InitInputBuffer() had been called with bytesPerBuffer=4096,
     //
     //                     dsw->uReadOffset, but renamed to dsw->dwInputBufferTailIndex
     //                     because that's what it would be in a simple
     //                     lock-free, classic circular FIFO).
     //     So "dwCapturePos" is THE REAL BUFFER HEAD INDEX ("occupied by the hardware").
     //     But why the heck does "dwReadPos" always step by 3000, if it steps at all ? ?
     //     At fsample=48 kHz, 3000 samples are 62.5 bloody milliseconds ! ! ? !
     //     As a test, using the 'Remote CW Keyer' application,
     //     the almost-latency-free SIDETONE FROM THE COM-PORT'S TXD OUTPUT
     //     was 'echoed back' into the PC's onboard microphone, and plotted
     //     along with the original keying signal in the program's timing scope.
     //     Result: Total latency of originally 138 ms - see screenshot in
     //             Remote_CW_Keyer/manual/MIC_input_latency_with_DirectSound.png !
     //
   } // end if < "capure pos" or "read pos" have changed since the last call >
  dsw->dwPrevInputCapturePos = dwCapturePos;
  dsw->dwPrevInputReadPos    = dwReadPos;


  filled = dwReadPos - dsw->dwInputBufferTailIndex; // <- again, treat the "read pos", not the "capture offset", as the FIFO head index !
  if( filled < 0 )
   {  filled += dsw->dwInputFifoSizeInBytes; // unwrap offset
   }
  *bytesFilled = filled;


  // Timestamp-based check for DSoundWrapper.nInputOverflows :
  dblSecondsSinceLastRefill = dblCurrentTime_s - dsw->dblLastInputTime_s;
  if( dblSecondsSinceLastRefill >= (dsw->dblInputWrapInterval_s + 50e-3/*50 ms for thread-switching*/) )
   { // Calling interval is too slow  ..
     ++dsw->nInputOverflows;
   }

  return TRUE;
} // end DSW_QueryInputFilled()


//---------------------------------------------------------------------------
void DSW_CalibrateInputSampleRate( // .. against the "QueryPerformanceCounter"-thingy in Windows
        DSoundWrapper *dsw,       // [in,out] instance data of the 'Direct Sound Wrapper'
        double dblCurrentTime_s,  // [in] timestamp of the 'Circular Buffer Wrap'
        DWORD  dwNumSamplesSinceLastCall) // [in] usually 4096 audio samples between calls,
                                          //      but don't bet on that !
  // Called from DSW_QueryInputFilled(), shortly after the "Read Position" aka "Read Cursor",
  // reported by DirectSound for the circular input buffer, has wrapped around
  // from e.g. byte index 8191 to zero. With 16 bit/sample, mono, and fs=48 kHz
  // this theoretically happens every 4096 samples / 48 kHz = 85.33 milliseconds.
  //
  // But due to the uncertainty in the timestamps of at least +/-10 milliseconds,
  //  the measurement of the SAMPING RATE or (inverse) SAMPLING PERIOD
  //  needs a much longer gate time, as implemented in this function.
  //
  // So, measure the sampling frequency every <DSW_INPUT_SR_CALIB_GATE_TIME_S> seconds.
  //  Example: DSW_INPUT_SR_CALIB_GATE_TIME_S = 10 [seconds] :
  //            10 seconds * 48 kHz = 480000 samples in CIRCA 10 seconds;
  //        480000 samples / (10000 - 10 ms) = 48.048 kHz
  //        480000 samples / (10000 + 10 ms) = 47.952 kHz -> still not "good",
  //                       thus the need for the additional AVERAGE FILTER
  //                       with e.g. DSW_INPUT_SR_CALIB_AVERAGE_LENGTH = 100 .
  // [in]      dblCurrentTime_s : value returned by DSW_ReadHighResTimestamp_s() when the DirectSound "Read Position" wrapped around
  // [in]      dsw->dblInputSRCalibTime_s : value returned by DSW_ReadHighResTimestamp_s() when the current counting interval began
  // [in,out]  dsw->dwInputSRCalibSampleCounter : number of AUDIO SAMPLE POINTS counted in the current counting interval
  // [out]     dsw->dblInputSecondsPerSample : PRECISE ("calibrated") stepwidth of the timestamp for each audio sample
  // [out]     dsw->dblInputSRCalibError_ppm : Simply the 'error' in dblInputSecondsPerSample, scaled into ppm
  //                                           to have easier-to-interpret values for the 'Debug' display.
{
  double dblDeltaT_s, dblSecondsPerSample, dblShortTermSRCalibError_ppm;
  int i;

  if( dsw->dblInputSRCalibTime_s <= 0.0 ) // no valid START TIME for the sample rate measurement yet ?
   { // (Will get here in the first call from DSW_QueryInputFilled(),
     //  when there is no PREVIOUS TIMESTAMP because there was no
     //                   PREVIOUS circular buffer wrap-around)
     // Just prepare the next 'frequency measuring gate time' and return :
     dsw->dwInputSRCalibSampleCounter = 0;
     dsw->dblInputSRCalibTime_s = dblCurrentTime_s; // <- NOW we've got a valid timestamp for the begin of input SR measurement
     return;
     // With the above kludge, even the INITIAL READINGS of the SR (SampleRate) error
     // were low enough to be useful (to provide a sample-counter-based timestamp)
     // Displayed on the RCW Keyer's "Debug" tab :
     // >
     // > Input from 'Mikrofon (USB Audio CODEC )' : (actually an IC-7300's built-in USB audio device)
     // > 836 kSamples read, 836 kS captured, 0 Overflows, SR error=-15.3 ppm
     // > 6981 kSamples read, 6980 kS captured, 0 Overflows, SR error=+4.0 ppm
     // > 9081 kSamples read, 9080 kS captured, 0 Overflows, SR error=-1.0 ppm
     // > 11401 kSamples read, 11400 kS captured, 0 Overflows, SR error=-11.5 ppm
     // > 15132 kSamples read, 15132 kS captured, 0 Overflows, SR error=+7.9 ppm
     // > 22197 kSamples read, 22196 kS captured, 0 Overflows, SR error=+5.3 ppm
     // > 32594 kSamples read, 32592 kS captured, 0 Overflows, SR error=-7.0 ppm
     // >
     // > |____|-- At 48000 kSamples read from the audio input,
     //            dsw->dblInputSRCalib_AverageFilterForSecondsPerSample[]
     //            is completely filled, and the "low-pass filtered" readings
     //               of  dsw->dblInputSecondsPerSample (inverse of the SAMPLING RATE)
     //               and dsw->dblInputSRCalibError_ppm
     //            will not become more accurate without a further increase of
     //               DSW_INPUT_SR_CALIB_GATE_TIME_S * DSW_INPUT_SR_CALIB_AVERAGE_LENGTH .

   } // end if < no valid START TIME for the sampling rate measurement / "calibration" against the Windows "Performance Counter" >

  dsw->dwInputSRCalibSampleCounter += dwNumSamplesSinceLastCall; // count SAMPLES during the measuring interval
  if( (dblDeltaT_s = dblCurrentTime_s - dsw->dblInputSRCalibTime_s) >= DSW_INPUT_SR_CALIB_GATE_TIME_S )
   { // Time for action.. finished another "input sample rate calibration" interval ?
     if( dsw->dwInputSRCalibSampleCounter > 0 ) // avoid div-by-zero ..
      {
        dblSecondsPerSample = dblDeltaT_s / (double)dsw->dwInputSRCalibSampleCounter;
        // '--> NEW value for dsw->dblInputSecondsPerSample, before averaging.
        //      Got to excluse grossly wrong values of 'd' from entering the
        //      average filter because that would 'spoil the soup' for a long
        //      time (until the 'bad measurement', caused by LOST INPUT SAMPLES,
        //      has fallen out of the average buffer), e.g. after
        //   DSW_INPUT_SR_CALIB_GATE_TIME_S * DSW_INPUT_SR_CALIB_AVERAGE_LENGTH
        //        = 10 seconds * 100 elements in the average filter
        //        = 10000 seconds = 16.7 minutes !
        dblShortTermSRCalibError_ppm = 1e6 * ( dblSecondsPerSample * dsw->dblInputSamplingRate_Hz - 1.0 );
        // What can be EXPECTED, and what must be REJECTED (e.g. disturbed by
        //   Windows taking the CPU aways for too long, wrecking the calling
        //   interval of DSW_QueryInputFilled() -> DSW_CalibrateInputSampleRate() ) ?
        // * Assume neither the soundcard's crystal oscillator,
        //          nor the oscillator feeding windows' "Performance Counter"
        //          are off by more than 100 ppm (that's a cheap 'computer grade' crystal).
        // * Assume the 10-second (DSW_INPUT_SR_CALIB_GATE_TIME_S) measuring interval
        //          has a maximum "timing jitter" of  +/-10 milliseconds
        //          (most of that being the unpredictable times at which the
        //           'Direct Sound' emulation updates the "Read Position" aka "Read Cursor").
        // With those assumptions:
        //    20 milliseconds "error" in a 10-second interval : 1e6 * 20e-3 / 10 = 2000 ppm
        //       plus 100 ppm for the 'cheap crystals'
        //          -> Maximum dblShortTermSRCalibError_ppm = 2100 .
        // Actually SEEN HERE when hitting a breakpoint :
        //       dblShortTermSRCalibError_ppm = -318.4, -125.4, +302.3, ...
        //

        // Plausible result in THIS ten-second interval (DSW_INPUT_SR_CALIB_GATE_TIME_S) ?
        if( (dblShortTermSRCalibError_ppm > -2100.0 ) && (dblShortTermSRCalibError_ppm < 2100.0 ) )
         { // Add the new value (dblSecondsPerSample) to the average filter (simplistic lowpass):
           if( dsw->iInputSRCalib_AverageFilterNumEntries < DSW_INPUT_SR_CALIB_AVERAGE_LENGTH )
            { dsw->dblInputSRCalib_AverageFilterForSecondsPerSample[dsw->iInputSRCalib_AverageFilterNumEntries++] = dblSecondsPerSample;
            }
           else
            { dsw->iInputSRCalib_AverageFilterNumEntries = DSW_INPUT_SR_CALIB_AVERAGE_LENGTH;
              for(i=0; i<(DSW_INPUT_SR_CALIB_AVERAGE_LENGTH-1); ++i)
               { // index ZERO shall contain the OLDEST entry, so "scroll up":
                 dsw->dblInputSRCalib_AverageFilterForSecondsPerSample[i]
                  = dsw->dblInputSRCalib_AverageFilterForSecondsPerSample[i+1];
               }
              dsw->dblInputSRCalib_AverageFilterForSecondsPerSample[DSW_INPUT_SR_CALIB_AVERAGE_LENGTH-1] = dblSecondsPerSample;
            }
           // Update the "long term averaged" audio-input-sampling-interval
           //  (here, for convenience, measured in seconds per sample, not samples per second):
           dblSecondsPerSample = 0.0;
           for( i=0; i < dsw->iInputSRCalib_AverageFilterNumEntries; ++i)
            { dblSecondsPerSample += dsw->dblInputSRCalib_AverageFilterForSecondsPerSample[i];
            }
           dsw->dblInputSecondsPerSample = dblSecondsPerSample / (double)dsw->iInputSRCalib_AverageFilterNumEntries;
           dsw->dblInputSRCalibError_ppm = 1e6 * ( dsw->dblInputSecondsPerSample * dsw->dblInputSamplingRate_Hz - 1.0 );
         } // end if < dblShortTermSRCalibError_ppm plausible > ?
      }   // end if( dsw->dwInputSRCalibSampleCounter > 0 )

     // Prepare the next "timestamp"-based sample-rate-measurement interval:
     dsw->dwInputSRCalibSampleCounter = 0;
     dsw->dblInputSRCalibTime_s = dblCurrentTime_s;

     if( dsw->iInputSRCalib_AverageFilterNumEntries == DSW_INPUT_SR_CALIB_AVERAGE_LENGTH )
      { dsw->iInputSRCalib_AverageFilterNumEntries = dsw->iInputSRCalib_AverageFilterNumEntries;  // <- place for a BREAKPOINT
        // Test results with different gate times, with input from an IC-7300
        // at f_sample = 48 kHz -> Ideally dsw->dblInputSecondsPerSample = 2.08333e-6 (20.8333 us),
        // when dsw->iInputSRCalib_AverageFilterNumEntries reached DSW_INPUT_SR_CALIB_AVERAGE_LENGTH...
        //   DSW_INPUT_SR_CALIB_GATE_TIME_S = 10, DSW_INPUT_SR_CALIB_AVERAGE_LENGTH = 10 (takes 1.6 minutes to reach "full accuracy"):
        //    |- dsw->dblInputSecondsPerSample = 2.08360e-5,
        //    |- dsw->dblInputSRCalibError_ppm = 131.08,
        //
        //   DSW_INPUT_SR_CALIB_GATE_TIME_S = 10, DSW_INPUT_SR_CALIB_AVERAGE_LENGTH = 100 (takes 16 minutes to reach "full accuracy"):
        //    |- dsw->dblInputSecondsPerSample =
        //    |- dsw->dblInputSRCalibError_ppm = 15.0
      }
   } // end if < finished another "input sample rate calibration" interval > ?
} // end DSW_CalibrateInputSampleRate()


//---------------------------------------------------------------------------
BOOL DSW_ReadBlock( DSoundWrapper *dsw, char *buf, long numBytes, double *pdblTimestamp_s )
  // [out, optional] *pdblTimestamp_s = timestamp that applies to the
  //                                    first byte (or "sample point") in buf[0] .
  //                  pdblTimestamp_s = NULL if the caller doesn't need a timestamp.
  //       The Remote CW Keyer uses these timestamps to vertically align
  //       channels on the 'timing scope', collected from dozens of
  //       different sources (serial port, network, audio input, etc).
  // Caller (in the Remote CW Keyer): CwDSP.c : DspThread(),
  //           asking for as many bytes as DSW_QueryInputFilled() has reported
  //           to be available at the moment (but limited by the application's
  //           audio sample buffer size) .
  //
{
  HRESULT hr; // <- this "H" does NOT mean Handle. It doesn't mean anything .. Microsoft legacy.
  LPBYTE  lpbuf1 = NULL;
  LPBYTE  lpbuf2 = NULL;
  DWORD   dwsize1 = 0;
  DWORD   dwsize2 = 0;
  int     nSamplePoints;
#if ( SWI_AUDIO_INPUT_DUMMY )
  int     i, iSample;
#endif // SWI_AUDIO_INPUT_DUMMY ?


  HERE_I_AM__DSW(); // -> DSW_iLastSourceLine = __LINE__ (when compiled for "hardcore debugging")

  if( (dsw==NULL) || (dsw->pDirectSoundIn==NULL) )
   { HERE_I_AM__DSW();
     DSW_SetFirstInputError( dsw, "Called DSW_ReadBlock() but there's no DSOUND instance." );
     return FALSE; // "application error" (avoid crashing with a NULL pointer reference)
   }

  if( dsw->pInputBuffer==NULL )
   { HERE_I_AM__DSW();
     DSW_SetFirstInputError( dsw, "Called DSW_ReadBlock() but there's no INPUT BUFFER." );
     return FALSE; // "application error" (avoid crashing in ..GetCurrentPosition() )
   }

  if( (!dsw->fInputOpened ) // oops.. in the meantime, someone has called DSW_StopInput() ?!
    || (dsw->dwBytesPerFrame_in == 0 ) ) // on this occasion, also avoid crashing from div-by-zero
   { // (this may indicate a multi-threading issue, so beware..)
     HERE_I_AM__DSW();
     DSW_SetFirstInputError( dsw, "Called DSW_ReadBlock() but input isn't open." );
     return FALSE;
   }

  // Refuse to read a number of bytes that is NOT an integer multiple
  //  of a "frame" (aka "sample point, with N channels") :
  nSamplePoints = numBytes / dsw->dwBytesPerFrame_in;
  if( numBytes != (long)( nSamplePoints * dsw->dwBytesPerFrame_in ) )
   { HERE_I_AM__DSW();
     DSW_SetFirstInputError( dsw, "Called DSW_ReadBlock() with a broken blocksize." );
     return FALSE;
   }


  // Lock free space in the DS  (WB: let Direct Sound know that we read from it, and don't want DS to interfere)
  HERE_I_AM__DSW();  // <- 2024-01-06 : Last sign of life HERE, before crashing deep inside in IDirectSoundCaptureBuffer_Lock() !
  hr = IDirectSoundCaptureBuffer_Lock( dsw->pInputBuffer,
             dsw->dwInputBufferTailIndex, // [in] DWORD "dwReadCursor" : Equivalent of a circular FIFO's TAIL INDEX ?
                                  // See ASCII sketch in DSW_QueryInputFilled() !
                                  //    > Offset, in bytes, from the start of the buffer to where the lock begins.
             numBytes,            // [in] DWORD dwReadBytes  :   BYTES, not AUDIO SAMPLES !
                                  //    > Size, in bytes, of the portion of the buffer to lock.
                                  //    > Note that the capture buffer is conceptually circular.
             (void **) &lpbuf1,   // [out] LPVOID* lplpvAudioPtr1  : Address of a pointer to contain
                                  //       the first block of the capture buffer to be locked
             &dwsize1,            // [out] LPDWORD lpdwAudioBytes1 : Address of a variable to contain
                                  //       the number of bytes pointed to by the lplpvAudioPtr1 parameter.
                                  //       If this value is less than the dwReadBytes parameter,
                                  //       lplpvAudioPtr2 will point to a second block of data.
             (void **) &lpbuf2,   // [out] LPVOID* lplpvAudioPtr2  : Address of a pointer to contain
                                  //       the second block of the capture buffer to be locked.
                                  //       If the value of this parameter is NULL, the lplpvAudioPtr1
                                  //       parameter points to the entire locked portion of the capture buffer.
             &dwsize2,            // [out] LPDWORD lpdwAudioBytes2 : Address of a variable to contain
                                  //       the number of bytes pointed to by the lplpvAudioPtr2 parameter.
                                  //       If lplpvAudioPtr2 is NULL, this value will be 0.
             0 );                 // [in] dwFlags : 0 or DSCBLOCK_ENTIREBUFFER (to lock the ENTIRE buffer).
  HERE_I_AM__DSW();
  if (hr == DS_OK)
   {
     HERE_I_AM__DSW();
     // Out of curiosity.. do 'lpbuf1' and 'lpbuf2' really point into dsw->pInputBuffer ?
     // After hitting a breakpoint on the "here I am" above, got here with ...
     //   (1st call) :   dsw->pInputBuffer  = 0x00CACCD0  (LPDIRECTSOUNDCAPTUREBUFFER is an obscure struct, members unknown)
     //                  lpbuf1 = 0x057C4FF0  dwsize1 = 3000 (==numBytes)
     //                  lpbuf2 = NULL        dwsize2 = 0
     //      -> dsw->pInputBuffer doesn't point to the SAMPLE BUFFER ITSELF .
     //
     // Copy from DS (lpbuf1 and/or lbbuf2) to the caller's buffer :
     CopyMemory( buf, lpbuf1, dwsize1); // first sample in the caller's destination buffer was taken from dsw->dwInputBufferTailIndex
     HERE_I_AM__DSW();
     // > If the block wraps around from the end of the buffer to the beginning,
     // > two pointers are returned, one for each section of the block.
     // > The second pointer is NULL if the locked portion of the buffer does not wrap around.
     if( (lpbuf2 != NULL) && (dwsize2>0) )
      { HERE_I_AM__DSW();
        CopyMemory( buf+dwsize1, lpbuf2, dwsize2); // copy a few more samples AFTER the DirectSound circular buffer's wrapping point
        HERE_I_AM__DSW();
      }

     nSamplePoints = ( dwsize1 + dwsize2 ) / dsw->dwBytesPerFrame_in;
     // '--> e.g. 3000 + 0 bytes / two bytes per sample point ("frame") -> nSamplePoints = 1500 .

     // Unlock the sound buffer (as early as possible, see old Microsoft documentation):
     HERE_I_AM__DSW(); // shortly before calling IDirectSoundCaptureBuffer_Unlock() ..
#  if( SWI_HARDCORE_DEBUGGING )
     // Take a snapshot of the parameters passed to IDirectSoundCaptureBuffer_Unlock() below...
     dsw->dwDebug[0] = __LINE__;
     dsw->dwDebug[1] = (DWORD)dsw->pInputBuffer;
     dsw->dwDebug[2] = (DWORD)lpbuf1;
     dsw->dwDebug[3] = (DWORD)dwsize1;
     dsw->dwDebug[4] = (DWORD)lpbuf2;
     dsw->dwDebug[5] = (DWORD)dwsize2;
     //  WATCHED the following global variables after getting stuck in CwDSP_Stop()
     //  because DSW_ReadBlock() -> IDirectSoundCaptureBuffer_Unlock() didn't return:
     //    MyDirectSound.fInputOpened    = 1 (ok)
     //    MyDirectSound.fOutputOpened   = 1 (ok)
     //    MyDirectSound.pInputBuffer = 0x00C3D8E0 (ok)
     //    MyDirectSound.dwInputBufferTailIndex = 1536       (ok)
     //    MyDirectSound.dwInputFifoSizeInBytes = 4096
     //    MyDirectSound.dwDebug[0]  = 1227 = __LINE__ (above,ok)
     //    MyDirectSound.dwDebug[1]  = 0x00C3D8E0 = dsw->pInputBuffer (ok)
     //    MyDirectSound.dwDebug[2]  = 0x00C3CC30 = lpbuf1
     //    MyDirectSound.dwDebug[3]  = 0x00000200 = dwsize1
     //    MyDirectSound.dwDebug[4]  = 0x00000000 = lpbuf2
     //    MyDirectSound.dwDebug[5]  = 0x00000000 = dwsize2
#  endif // SWI_HARDCORE_DEBUGGING ?
     IDirectSoundCaptureBuffer_Unlock( dsw->pInputBuffer, lpbuf1, dwsize1, lpbuf2, dwsize2);
     HERE_I_AM__DSW();   // survived the call of IDirectSoundCaptureBuffer_Unlock()
#   if(SWI_AUDIO_INPUT_DUMMY) // (1)=TEST with an "artifical input signal" (sawtooth), (0)=normal compilation
     for( i=0; i<nSamplePoints; ++i )
      { iSample = (int)( (dsw->dwTotalSamplesReadFromInput % dsw->dwInputFifoSizeInBytes )
                            * 65535 / dsw->dwInputFifoSizeInBytes ) - 32767;
        buf[ i*dsw->dwBytesPerFrame_in + 0 ] = iSample & 0x00FF; // lower 8 bits
        buf[ i*dsw->dwBytesPerFrame_in + 1 ] = iSample >> 8;     // upper 8 bits
        ++dsw->dwTotalSamplesReadFromInput;
      }
#   else // no TEST but a NORMAL COMPILATION :
     dsw->dwTotalSamplesReadFromInput += nSamplePoints; // <- overflows after a few hours
#   endif  // SWI_AUDIO_INPUT_DUMMY ?
     dsw->dblSamplesReadFromInput += nSamplePoints; // <- won't overflow during the lifetime of any computer

     // Provide a timestamp for the first audio sample copied into the caller's buffer.
     // The FIRST sample 'locked' by Direct Sound was at dsw->dwInputBufferTailIndex,
     //           so the timestamp for that sample is in dsw->dblTimestampAtInputBufferTail_s :
     if( pdblTimestamp_s != NULL ) // caller wants to know the timestamp of the first sample's "origin" ->
      { // [in] dsw->dblTimestampAtInputBufferTail_s (tracked here,
        //      but additionally "drift eliminated" in DSW_QueryInputFilled()
        *pdblTimestamp_s = dsw->dblTimestampAtInputBufferTail_s;
      }

     // Adjust the application's input buffer TAIL INDEX for the next call:
     dsw->dwInputBufferTailIndex = (dsw->dwInputBufferTailIndex + dwsize1 + dwsize2) % dsw->dwInputFifoSizeInBytes;

     // Along with dsw->dwInputBufferTailIndex (incremented above),
     // so also INCREMENT dsw->dblTimestampAtInputBufferTail_s for the next call:
     dsw->dblTimestampAtInputBufferTail_s += (double)nSamplePoints * dsw->dblInputSecondsPerSample;


     HERE_I_AM__DSW();
     return TRUE;
   }
  else // IDirectSoundCaptureBuffer_Lock() refused to lock the wanted number of BYTES, beginning at our FIFO TAIL INDEX ...
   { HERE_I_AM__DSW();
     return FALSE;
   }
} // end DSW_ReadBlock()

//---------------------------------------------------------------------------
void DSW_CloseInput( DSoundWrapper *dsw )                            // API
{ // Cleanup the sound INPUT buffers and the release related resources

  if( dsw->fInputOpened ) // <- similar as in DSP_StopInput() .. avoid multithreading issues
   {  dsw->fInputOpened = FALSE;  // <- DO NOT CALL ANY AUDIO-INPUT-RELATED functions now !
      Sleep(50); // wait until 'worker threads' have taken notice,
                 // and e.g. DSW_ReadBlock() refuses to use pInputBuffer,
                 // because (in contrast to its name), pInputBuffer
                 // isn't just the circular input buffer, but also contains
                 // the pointers to the 'IDirectSound'-object's METHOD TABLE !
                 // See definition of e.g. IDirectSoundCaptureBuffer_Lock()
                 // in Microsoft's "dsound.h" :
                 // #define IDirectSoundCaptureBuffer_Lock(p,a,b,c,d,e,f,g)  (p)->lpVtbl->Lock(p,a,b,c,d,e,f,g)
                 //                                        '--------->--------'
                 // The 'p' in the definition above is our
                 // "LPDIRECTSOUNDCAPTUREBUFFER pInputBuffer"
                 //    |______________________|
   } // end if( dsw->fInputOpened )



  if( dsw->pInputBuffer != NULL ) // <- this is a pointer, not a BOOLEAN
   {
     IDirectSoundCaptureBuffer_Stop( dsw->pInputBuffer );
     IDirectSoundCaptureBuffer_Release( dsw->pInputBuffer );
     dsw->pInputBuffer = NULL;
   }
  if (dsw->pDirectSoundIn != NULL )
   {
     IDirectSoundCapture_Release( dsw->pDirectSoundIn );
     dsw->pDirectSoundIn = NULL;
   }
} // end DSW_CloseInput()


// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
// Extra stuff for troubleshooting / development :
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -



//---------------------------------------------------------------------------
double DSW_GetMeasuredInputSampleRate( DSoundWrapper *dsw ) // .. in samples/second [Hz]
{
  if( dsw->dblInputSecondsPerSample > 0.0 )
   { return 1.0 / dsw->dblInputSecondsPerSample;
   }
  else // the audio-INPUT's sampling rate has not been MEASURED yet ->
   { return 0.0;
   }
} // end DSW_GetMeasuredInputSampleRate()


//---------------------------------------------------------------------------
double DSW_GetInputSampleRateFromHistory( DSoundWrapper *dsw,
           int iHistoryIndex )  // [in] 0 = most recent reading, 1 = one interval before, etc
  // This "history" is actually a queue of timestamp-based readings of the
  // input sampling interval (internally using SECCONDS PER SAMPLE, not samples per second),
  // with new measurements taken every <DSW_INPUT_SR_CALIB_GATE_TIME_S> seconds.
  // Up to <DSW_INPUT_SR_CALIB_AVERAGE_LENGTH> measurements are available.
  // The AVERAGE from all readings is returned by DSW_GetMeasuredInputSampleRate().
{
  double d;
  int j = dsw->iInputSRCalib_AverageFilterNumEntries - 1 - iHistoryIndex;
  if( (j>=0) && (j<DSW_INPUT_SR_CALIB_AVERAGE_LENGTH) )
   { d = dsw->dblInputSRCalib_AverageFilterForSecondsPerSample[j];
     if( d > 0.0 )
      { return 1.0 / d; // convert "seconds per sample" to "samples per second" [Hz]
      }
   }
  return 0;
} // end DSW_GetInputSampleRateFromHistory()



//---------------------------------------------------------------------------
void DSW_OnOutputError( DSoundWrapper *dsw, char *pszInfo, HRESULT hr )
  // Invoked from a couple of OUTPUT functions after problems
  // with a DSound function call... good place for a breakpoint !
{
  sprintf( dsw->sz255LastOutputError, "%s:%d", pszInfo, (int)hr );
}

//---------------------------------------------------------------------------
void DSW_SetFirstOutputError( DSoundWrapper *dsw, char *pszInfo )
{ if( dsw->sz255LastOutputError[0] == '\0' ) // only if this the FIRST error..
   { strcpy( dsw->sz255LastOutputError, pszInfo );
   }
}

//---------------------------------------------------------------------------
void DSW_OnInputError( DSoundWrapper *dsw, char *pszInfo, HRESULT hr )
  // Invoked from a couple of INPUT functions after problems
  // with a DSound function call... good place for a breakpoint !
{
  sprintf( dsw->sz255LastInputError, "%s:%d", pszInfo, (int)hr );
}

//---------------------------------------------------------------------------
void DSW_SetFirstInputError( DSoundWrapper *dsw, char *pszInfo )
{ if( dsw->sz255LastInputError[0] == '\0' ) // only if this the FIRST error..
   { strcpy( dsw->sz255LastInputError, pszInfo );
   }
}


