//---------------------------------------------------------------------------
// File: C:\cbproj\Remote_CW_Keyer\Yaesu5Byte.h
// Authors: Wolfgang Buescher (DL4YHF)
// Date:   2025-06-22
// Purpose:
//  Limited support for Yaesu's ugly, impractical, and outdated
//          "5-byte" CAT Control. Only tested with an FT-817ND so far.
//  Can only be used together with RigControl.c, which implements
//  the 'API' towards the application.
//  THIS module (Yaesu5Byte.c) implements the protocol
//  to communicate with the radio.
//
// Literature  and  Revision History :  See Yaesu5Byte.c  !
//
//---------------------------------------------------------------------------

#ifndef   _YAESU_5_BYTE_H_   // prevent multiple inclusion of 'Yaesu5Byte.h' ..
# define  _YAESU_5_BYTE_H_   //  .. and let other header files know we've been included

// "Command bytes" in this stoneage "Five-Byte Yaesu CAT" protocol (meow !);
// from "The KA7OEI FT-817 pages / CAT Interface Programming" [KA7OEI_FT817_CAT].
// Thankfully, someone sorted the list BY HEXADECIMAL COMMAND VALUE. So do we:

#define YAESU5B_CMD_LOCK_ON 0x00
        // > This command is equivalent to turning the dial/panel lock on.
        // > This precise effect of this setting depends on the state of
        // > submenu item #32 (stored at memory location 5E HEX - see below.)
        // > This command returns 00 if it was unlocked, and F0 if already locked.

#define YAESU5B_CMD_SET_FREQ 0x01
        // > This commands sets the current frequency (see Yaesu5Byte_DecodeMessage)

#define YAESU5B_CMD_SPLIT_ON 0x02
        // > This command enables split operation.  Use the Read Transmitter Status
        // > command (below) to determine current split status.
        // > This command returns 00 if not already on, and F0 (HEX) if already on.

#define YAESU5B_CMD_READ_FREQ_AND_MODE 0x03
        // > This command returns five bytes (see Yaesu5Byte_DecodeMessage)

        // Command 0x04 has been eaten by the CAT. Meow !

#define YAESU5B_CMD_RIT_ON 0x05
        // > This command turns the clarifier (a.k.a. RIT) on.
        // > This command returns 00 if not already on, and F0 (HEX) if already on.

#define YAESU5B_CMD_SET_OP_MODE 0x07  // .. with b[0] = OPMODE
        // > The mode byte is described below (YAESU5B_OPMODE_xxx)

#define YAESU5B_CMD_SET_PTT_ON  0x08
        // > This "keys" the FT-817.  In CW, this sets the radio to transmit mode,
        // > but does (not?) key the transmitter.
        // > Note that keying and unkeying the radio's PTT line will cancel
        // > the transmit mode (i.e. put it back into receive)
        // > and effectively override this command.
        // > This command returns 00 if the '817 was unkeyed, and F0 if already keyed.

#define YAESU5B_CMD_SET_REPEATER_DIR 0x09  // .. with b[0] = offset-DIRECTION (sign and OnOff flag, but NOT the OFFSET FREQUENCY)
        // > The byte "Offset" has the following function:
        // > 09 = - (minus) shift    49 = + (positive) shift    89 = simplex

#define YAESU5B_CMD_SET_DCS_CTCSS_MODE 0x0A // .. with the "mode" in b[0]
        // > The byte "Mode" has the following function:
        // > 0A = DCS Enable  2A = CTCSS Enable 4A = DCS/CTCSS Encoder  enable
        // > 8A = DCS/CTCSS Encoder Disable

#define YAESU5B_CMD_SET_CTCSS_TONE_FREQ 0x0B // .. with the tone freq in b[0..1]
        // > This command sets the CTCSS frequency

#define YAESU5B_CMD_SET_DCS_CODE        0x0C

        // Commands 0x0D and 0x0E have been chased away by the CAT. Meow !

#define YAESU5B_CMD_TURN_ON             0x0F

#define YAESU5B_CMD_READ_TX_KEYED_STATE 0x10 // ... undocumented ..
        // > This command returns 00 if the FT-817 is unkeyed,
        // > and nonzero (F0 HEX) if it is keyed

#define YAESU5B_CMD_LOCK_OFF 0x80
        // > This command is equivalent to turning the dial/panel lock off.
        // > This command returns 00 if the '817 was already locked, and F0 (HEX) if already unlocked.

#define YAESU5B_CMD_TOGGLE_VFO_A_B 0x81

#define YAESU5B_CMD_SPLIT_OFF      0x82

#define YAESU5B_CMD_RIT_OFF        0x85

#define YAESU5B_CMD_SET_PTT_OFF    0x88

#define YAESU5B_CMD_TURN_OFF       0x8F

#define YAESU5B_CMD_RADIO_CONFIG   0xA7  // better leave this to Simon's "FT817 Commander" !

#define YAESU5B_CMD_UNKNOWN_STATUS 0xBA

#define YAESU5B_CMD_READ_EEPROM    0xBB  // Simon's "FT817 Commander" uses this a lot; we don't

#define YAESU5B_CMD_WRITE_EEPROM   0xBC  // Simon's "FT817 Commander" uses this; we don't.
        // > Note that careless use of this command can cause the FT-817's CPU
        // > to crash and/or cause a complete wipe of all EEPROM data including
        // > configuration, software calibration/alignment, and memories
        // > - the effect being similar to that of the "Reset to Factory Defaults" command.
        // > If you insist on using this command without first recording the
        // > 76 "soft calibration" settings and noting what is in your radio's
        // > memories, you are an idiot!   (Am I clear on this point?)
        // (Sure, Clint. Before any experiments, "our" FT-817ND's "calibration"
        //  has been read-out with Simon's "FT817 Commander", and stored
        //  as a file in C:\afusoft\FT-817_Commander\FT-817 Calibration.txt,
        //  and even PRINTED ON PAPER + glued into the FT-817ND manual )

#define YAESU5B_CMD_TX_METERS      0xBD  // returns Forward power, VSWR, ALC, and Modulation

#define YAESU5B_CMD_FACTORY_DEFAULT 0xBE // MEEP MEEP MEEP ..  NEVER SEND THIS COMMAND !
        // > This command completely resets all configuration settings
        // > and factory calibration parameters to their factory (pre-alignment)
        // > values as well as erasing all channel memories.
        // > Be very careful when using this command!  (YHF: We'll NEVER send this)

#define YAESU5B_CMD_GET_RX_STATUS  0xE7
        // > This command returns one byte containing receiver status (see below)

#define YAESU5B_CMD_SET_RIT_OFFSET 0xF5
        // > This command sets the clarifier (a.k.a. RIT) direction and offset amount

#define YAESU5B_CMD_GET_TX_STATUS  0xF7
        // > This command returns one byte containing transmitter status

#define YAESU5B_CMD_SET_REPEATER_OFFSET 0xF9
        // > This command sets the magnitude of the repeater offset.
        // > It may range from 0 to 99.99 MHz

// A few(!) EEPROM locations in the FT-817 regularly polled by "FT817 Commander".
// So far, only used to make the "CAT log" on RCW Keyer's "Debug"-tab
//         a bit more readable (and to understand why Simon't program
//         keeps shelling out YAESU5B_CMD_READ_EEPROM a go-go) :
//  > 19:34:22.5 COM8: TX[05] 0x00 00 00 00 F7   ; Get TX Status
//  > 19:34:22.5 COM8: TX[05] 0x00 00 00 00 E7   ; Get RX Status
//  > 19:34:22.5 COM8: TX[05] 0x00 55 00 00 BB   ; Read EEPROM
//  > 19:34:22.6 COM8: TX[05] 0x00 57 00 00 BB   ; Read EEPROM
//  > 19:34:22.6 COM8: TX[05] 0x00 59 00 00 BB   ; Read EEPROM
//  > 19:34:22.6 COM8: TX[05] 0x00 5F 00 00 BB   ; Read EEPROM
//  > 19:34:22.7 COM8: TX[05] 0x00 65 00 00 BB   ; Read EEPROM
//  > 19:34:22.7 COM8: TX[05] 0x00 79 00 00 BB   ; Read EEPROM
//  > 19:34:22.7 COM8: TX[05] 0x03 55 00 00 BB   ; Read EEPROM
//  > 19:34:22.8 COM8: TX[05] 0x03 5F 00 00 BB   ; Read EEPROM
//  > 19:34:22.8 COM8: TX[05] 0x03 61 00 00 BB   ; Read EEPROM
//  > 19:34:22.8 COM8: TX[05] 0x01 B6 00 00 BB   ; Read EEPROM
//  > 19:34:22.8 COM8: TX[05] 0x01 B8 00 00 BB   ; Read EEPROM
//  > 19:34:22.9 COM8: TX[05] 0x01 BB 00 00 BB   ; Read EEPROM
//  > 19:34:22.9 COM8: TX[05] 0x00 00 00 00 F7   ; Get TX Status
#define FT817_EEPROM_ADDR_MEMO_VFO_SELECT 0x0055 // b0:VFO_A/B, b1:"MTQMB", b2="QMB", b3="?", b4="Home Select", b5="Memory/MTUNE Select", b6="?", b7="MEMORY/VFO Select"
#define FT817_EEPROM_ADDR_AGC_DSP_PBT_NB_LOCK_FST 0x0057 // b1..0="AGC mode", b2="DSP on", b4="Passband Tuning",b5="Noiseblanker", b6="Lock", b7="Fast Tuning"
#define FT817_EEPROM_ADDR_VFO_BAND_SELECT 0x0059 // bits 7..4 : "VFO B", bits 3..0 : "VFO A"
#define FT817_EEPROM_ADDR_CW_PITCH_ETC    0x005E // b3..0=CW Pitch, b5..4=Lock Mode, b7..6=Op Filter (why cram all this in ONE BYTE?)
#define FT817_EEPROM_ADDR_CW_WEIGHT_ETC   0x005F // b4..0=CW Weight, b5="ARS(2), b6="ARS(1)", b7="Sql/RF-G" (what the heck..)
#define FT817_EEPROM_ADDR_SIDETONE_VOL    0x0061
#define FT817_EEPROM_ADDR_CW_SPEED        0x0062 // b5..0=CW Speed, b7..6="Batt Charge Time"
#define FT817_EEPROM_ADDR_VOX_GAIN_ETC    0x0063 // b6..0=VOX Gain, b7="Disable AM/FM Dial"
#define FT817_EEPROM_ADDR_VOX_DELAY_ETC   0x0064 // b4..0=VOX Gain, b5="Emergency", b7..6="CAT Rate" (4800,9600,38400)
#define FT817_EEPROM_ADDR_APO_TIME_ETC    0x0065 // b2..0="APO Time", b4=Mem Group On/Off, b7..5="Dig Mode"
#define FT817_EEPROM_ADDR_TX_POWER_ETC    0x0079 // b1..0=TX Power, b3="PRI on/off", etc etc


// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
// Instance data to talk to or decode data from the "Yaesu 5-byte CAT" (meow)
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

typedef void (*T_Y5B_OnDecodeCallback)( // may be invoked ONCE or even MULTIPLE TIMES from Yaesu5Byte_ProcessData() ..
  struct t_Yaesu5ByteControl *pYC, // [in] instance that 'decoded' the message,
                      //      possibly also connected to a T_RigControl instance.
  int iRigCtrlPort,   // [in] e.g. RIGCTRL_PORT_RADIO, RIGCTRL_PORT_AUX_COM_1, ..
  int iRigCtrlOrigin, // [in] e.g. RIGCTRL_ORIGIN_RADIO, RIGCTRL_ORIGIN_CONTROLLER,
                      //      or sometimes just RIGCTRL_ORIGIN_COM_PORT_RX/TX
                      //      if we don't know "who is what" on a Serial Port Tunnel.
  BYTE *pbMessage, int iMsgLength, // [in] raw message data and length in bytes
  char *pszComment ); // [in] human readable "comment" for the log


typedef struct t_Yaesu5ByteControl
{

  int iExpectingResponseForCmd; // negative when NOT expecting to receive the response for a command,
                                // otherwise a YAESU5B_CMD_.. code
  int iExpectedResponseLength; // <- set along with iExpectingResponseForCmd
                   // if we *KNOW* the expected number of bytes IN THE RESPONSE.

# define YAESU5B_RX_BUFFER_SIZE 8 // MAXIMUM size, in bytes, or any COMMAND or RESPONSE of this protocol
  BYTE bRxBuffer[ YAESU5B_RX_BUFFER_SIZE ];
  int  nBytesInRxBuffer;

  void *pvRigControl; // actually a "T_RigCtrlInstance *" (may be NULL for "stand-alone use").
     // "void*" is the usual kludge in "C" to avoid circular dependencies:
     //  T_Yaesu5ByteControl needs this 'link' to a T_RigCtrlInstance,
     //  T_RigCtrlInstance CONTAINS a T_Yaesu5ByteControl (as a struct member).
  BOOL fListenOnlyMode; // FALSE="active mode" (where this instance may send commands),
     // TRUE ="receive-only" aka "eavesdropping" mode (this instance MUST NOT send anything).


  // Replacement for an awful list of FUNCTION ARGUMENTS passed from Yaesu5Byte_ProcessData()
  // to dozens of COMMAND- and RESPONSE parser functions (invoked internally via function pointer):
  int iRigCtrlPort;   // [in] e.g. RIGCTRL_PORT_RADIO, RIGCTRL_PORT_AUX_COM_1, ..
  int iRigCtrlOrigin; // [in] e.g. RIGCTRL_ORIGIN_RADIO, RIGCTRL_ORIGIN_CONTROLLER,
                      //      or sometimes just RIGCTRL_ORIGIN_COM_PORT_RX/TX
                      //      if we don't know "who is what" on a Serial Port Tunnel.
  char sz80Comment[84]; // [out] string buffer for COMMENTS, long enough for all 'parsers'
                        //       invoked via function pointer in Yaesu5Byte_ProcessData() .
  int iCurrentMsgType; // <- input for pOnDecodeCallback, e.g. RIGCTRL_MSGTYPE_FREQUENCY_REPORT
        // | RIGCTRL_MSGTYPE_FLAG_RX | RIGCTRL_MSGTYPE_FLAG_PERIODIC_POLL | ...
        // RigControl.c : RigCtrl_Y5B_OnDecodeCallback() uses this info
        // to decide whether to LOG/DISPLAY the message or not.
  T_Y5B_OnDecodeCallback pOnDecodeCallback; // <- if non-NULL, may be invoked ONCE or even MULTIPLE TIMES from Yaesu5Byte_ProcessDat

  BOOL fBusy; // FALSE= *not* busy from sending a command (from THIS end), TRUE= "busy; cannot send from this end now".
  T_TIM_Stopwatch sw_BusyTimer; // timer started when setting fBusy=TRUE. Used to check for RESPONSE TIMEOUTS.
      // (Note that many of the "Yaesu-5-Byte-CAT"-commands don't send a response at all.
      //  In such cases sw_BusyTimer creates the important GAP between
      //  the transmission of two FIVE-BYTE-BLOCKS; say at least 20 ms @ 4800 baud, because:
      // 5 bytes * (1+8+2)bits_per_byte / 4800 bits_per_second = 11.5 ms per 5-byte-block
# define Y5B_BUSY_TIMER_TIMEOUT_MS 20

  int iParameterPollingState; // related to (and often the same as) pvRigControl->iParameterPollingState .
      // '--> Values def'd in RigControl.h, e.g. RIGCTRL_POLLSTATE_READ_RADIO_ID .
      //      'Read-only' externally; writeable via Yaesu5Byte_SwitchPollingState() .
  int iParameterPollingSubState;  // sub-index or similar; meaning depends on iParameterPollingState
  T_TIM_Stopwatch sw_ParameterPolling;  // "stopwatch" for timed transitions of iParameterPollingState,
                  // independent of the Yaesu5Byte_Handler() calling interval.
  BOOL fRadioTurnedON; // <- affects the state machine in Yaesu5Byte_Handler()..



} T_Yaesu5ByteControl;


// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
// Functions from Yaesu5Byte.c,  mostly used in RigControl.c :
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

//--------------------------------------------------------------------------
CPROT void Yaesu5Byte_InitStruct( T_Yaesu5ByteControl *pYC, // [out] 'Yaesu 5-byte CAT' decoder or controller instance
       void *pvRigControl, // [in] optional T_RigCtrlInstance* (may be NULL for "stand-alone use")
       T_Y5B_OnDecodeCallback pOnDecodeCallback ); // [in] optional callback for 'decoded COMMANDS or RESPONSES'

//--------------------------------------------------------------------------
CPROT void Yaesu5Byte_ForgetAllParams( T_Yaesu5ByteControl *pYC );


//--------------------------------------------------------------------------
CPROT int Yaesu5Byte_ProcessData( T_Yaesu5ByteControl *pYC,   // [in,out] 'Yaesu 5-byte CAT' instance
        int iRigCtrlPort,   // [in] e.g. RIGCTRL_PORT_RADIO, RIGCTRL_PORT_AUX_COM_1, ..
        int iRigCtrlOrigin, // [in] e.g. RIGCTRL_ORIGIN_RADIO, RIGCTRL_ORIGIN_CONTROLLER, RIGCTRL_ORIGIN_COM_PORT_RX/TX
        BYTE *pbRxData, int nBytes); // [in] chunk received data (not necessarily a complete, nor a SINGLE 'Message')

//--------------------------------------------------------------------------
CPROT int Yaesu5Byte_Handler( T_Yaesu5ByteControl *pYC ); // periodically called every 50(?) milliseconds
      // from the same task / thread / interrupt handler that also calls Yaesu5Byte_ProcessData().
      // The RETURN VALUE is one of the RIGCTRL_POLLSTATE_.. valued defined in RigControl.h

//--------------------------------------------------------------------------
CPROT int Yaesu5Byte_SwitchPollingState( T_Yaesu5ByteControl *pYC, int iNewPollingState );
      // Called from same task / thread / interrupt handler that also calls Yaesu5Byte_ProcessData(),
      // whenever module RigControl.c wants to "begin something", e.g.:
      //  * RIGCTRL_POLLSTATE_READ_RADIO_ID : Try to identify the radio model,
      //          which in Icom's CI-V protocol was performed by reading
      //          the radio's model-depending "default address" aka numeric ID.
      //  * etc ..  see implementation in the *.c module .
      // Also here, the RETURN VALUE is one of the RIGCTRL_POLLSTATE_.. valued defined in RigControl.h,
      // in most (but not all) cases the same as iNewPollingState .


//--------------------------------------------------------------------------
CPROT BOOL Yaesu5Byte_TurnPowerOnOrOff( T_Yaesu5ByteControl *pYC, BOOL fPowerOn ); // [in] TRUE=turn on, FALSE=turn off

//--------------------------------------------------------------------------
CPROT BOOL Yaesu5Byte_SetVFOFrequency( T_Yaesu5ByteControl *pYC, double dblFreqHz ); // [in] freq in HERTZ

//--------------------------------------------------------------------------
CPROT BOOL Yaesu5Byte_SetOpMode( T_Yaesu5ByteControl *pYC, int iRigControlOpMode ); // [in] "op-mode" a la RigControl.h (!)

//--------------------------------------------------------------------------
CPROT BOOL Yaesu5Byte_AskForParameter( T_Yaesu5ByteControl *pYC, int iUnifiedPN ); // [in] "unified parmater number" a la RigControl.c,


#endif // ndef   _YAESU_5_BYTE_H_ ?

/* EOF < C:\cbproj\Remote_CW_Keyer\Yaesu5Byte.h > */


