//---------------------------------------------------------------------------
// File: C:\cbproj\Remote_CW_Keyer\Yaesu5Byte.c
// Authors: Wolfgang Buescher (DL4YHF)
// Date:   2025-06-22
// Purpose:
//  Limited support for Yaesu's ugly, impractical, and outdated
//          "5-byte" CAT Control. Only tested with an FT-817ND so far.
//  In Remote CW Keyer, used together with RigControl.c, which implements
//                      the 'API' towards the application.
//  THIS module (Yaesu5Byte.c) implements the protocol to communicate with the radio,
//       or to eavesdrop on the communication between an external 'Controller'
//       and a real radio . 
//
// Literature:
//  [KA7OEI_FT817_CAT]: "The KA7OEI FT-817 pages / CAT Interface Programming"
//              (lacking a good specification by Yaesu itself, mostly based
//               on trial and error, with MUCH MORE than in the poor documents
//               listed further below). Locally saved as
//   file:///C:/datasheets/Rig_Manuals/Yaesu_FT-817_and_CAT_protocol/KA7OEI_FT-817_CAT_Interface_Programming.htm
//               See rants and notes by KA7OEI, like the following:
//      > Important:  This document details aspects of CAT interface commands
//      > that are specific to the FT-817.  Furthermore, some of these parameters
//      > (specifically memory locations) may be pertinent only to the version
//      > of the software in my radio and may not apply to other software versions!
//      > Finally, careless use of these commands may result in an unusable radio,
//      > requiring complete recalibration!  You have been warned!
//      >             (...)
//      > Note: It is specified in the manual that these 5 bytes are to be sent
//      >       in quick succession - within 200 milliseconds of each other.
//      >       Experimentation reveals, however, that this would be too slow -
//      >       it is more likely that all 5 bytes must be sent within a
//      >       200 millisecond period.
//            (DL4YHF: That's because there is NOTHING in this stupid protocol
//                     -except the pauses between 5-byte-blocks-
//                     that allow detection of the START and END of a block.
//              "Fortunately", the protocol is so stupid that it doesn't seem
//              to allow sending unsolicited reports, such as VFO report sent
//              'automatically' when turning the VFO knob.)
//      > Some commands (such as frequency entry) may take a few 10's of milliseconds
//      > to take effect, so be prepared to simply wait for the radio to finish.
//
//
//  [FT897_CAT]: "Official" CAT Data Protocol by Yaesu, for their FT-897 .
//          C:\datasheets\Rig_Manuals\Yaesu_FT-817_and_CAT_protocol\FT-897D_Yaesu-official_CAT-Commands.pdf .
//               Also uses the ancient 5-byte commands without a clear delimiter.
//               On first sight, basic commands like "Set Frequency" (0x01 in the FIFTH byte)
//               and "Read Frequency and mode" (0x03 in the FIFTH byte)
//               seem to be compatible with the FT-817 .
//
//  [FT817_ACC_Adapter]: Since Yeasu's CAT is "serial" but not RS-232 compatible,
//               an interface like the one shown HERE (for the FT-812)
//               is required between the RS-232 9-pin "COM port"
//               and the radio's "8 pin mini-DIN" receptacle, aka "ACC-connector".
//               Slightly modified (added wire colours of the homemade interface,
//               with two NPN transistors hidden in a tiny blob of hot-melt glue) :
//   C:\datasheets\Rig_Manuals\Yaesu_FT-817_and_CAT_protocol\FT-817_to_Serial_Port_Adapter.png
//
//
// Revision History (latest entry first):
//
//   2025-06-22: Added limited supported for the old Yaeasu-"5-byte"-CAT
//               control, in this module (Yeasu5Byte.c) .
//
//---------------------------------------------------------------------------

#include "switches.h" // project specific 'compilation switches'

#include <string.h>     // memset() isn't really a STRING function, but..
#include <stdio.h>      // snprintf() isn't "I/O", but declared in string.h
#include <math.h>       // fmodl() ..
#include "Timers.h"     // high-resolution stopwatches, etc
#include "Utilities.h"  // UTL_FormatDateAndTime(), UTL_GetCurrentUnixDateAndTime[_Fast](), etc
#include "QFile.h"      // "Quick File" helper functions by DL4YHF
#include "Inet_Tools.h" // INET_GetU16FromLE(), etc
#include "RigControl.h" // header file for the 'Rig Control' module,
                        // without Yaesu-specific details, but focused on ICOM.
#include "Yaesu5Byte.h" // Yaesu-FIVE-BYTE-CAT-specific details HERE !



//----------------------------------------------------------------------------
// Global variables for hardcore debugging / post-mortem crash analysis.
//        Useful when the debugger's call stack shows nothing but garbage,
//        e.g. after an exception at the infamous address 0xFEEEFEEE .
//        ( C++Builder's debugger could still inspect SIMPLE GLOBAL
//          VARIABLES after most kinds of exceptions, but of course nothing
//          stack-based because the CPU- or task-stack was usually trashed.)
//----------------------------------------------------------------------------
#if(SWI_HARDCORE_DEBUGGING) // (1) = hardcore-debugging, (0)=normal compilation
 int Yaesu5B_iLastSourceLine = 0; // WATCH THIS after crashing with e.g. "0xFEEEFEEE"  ...
# define HERE_I_AM__RIGCTRL()  Yaesu5B_iLastSourceLine=__LINE__
# if( 0 )
    If certain worker threads of the application don't seem to terminate
    "politely", watch the following (complete list not HERE but in Keyer_Main.cpp):
     * GUI_iLastSourceLine
     * DSW_iLastSourceLine
     * DSP_iLastSourceLine
     * Keyer_iLastSourceLine
     * CwNet_iLastSourceLine
     * HttpSrv_iLastSourceLine
     * RigCtrl_iLastSourceLine
     * Yaesu5B_iLastSourceLine
#  endif // (0) .. kludge to inspect the above variables in Borland via mouse-over ...
#else
# define HERE_I_AM__RIGCTRL()  /* nuffink   */
#endif // SWI_HARDCORE_DEBUGGING ?

//----------------------------------------------------------------------------
// Constant Tables for decoding 'Yaesu-5-Byte'-messages and FT-817 EEPROM data
//----------------------------------------------------------------------------

typedef int (*T_Y5B_Parser)( // COMMAND parser method, used internally, in T_Y5B_CommandTabEntry Y5B_CommandTable[]
           struct t_Yaesu5ByteControl *pYC); // [in,out] instance data, rx buffer   [out] decoded value[s]

typedef struct t_Y5B_CommandInfo // used as variable or member pCmdInfo :
{
  BYTE bCmd;   // 0x00 = YAESU5B_CMD_LOCK_ON ..  0xF9 = YAESU5B_CMD_SET_REPEATER_OFFSET
  T_Y5B_Parser pCmdParser;
  T_Y5B_Parser pRespParser;

} T_Y5B_CommandTabEntry;

const T_Y5B_CommandTabEntry Y5B_CommandTable[]; // implemented way further below, AFTER the dozens of 'Command Parser' and 'Response Parser' functions

const T_SL_TokenList FT817_EEPROM_Addresses[] = // only exist because the author HAD an FT-817[ND] ..
{ // pszKeyword, iTokenValue :
 { "Memo/VFO Select",  FT817_EEPROM_ADDR_MEMO_VFO_SELECT },
 { "AGC_DSP_LOCK_etc", FT817_EEPROM_ADDR_AGC_DSP_PBT_NB_LOCK_FST },
 { "VFO A/B band sel", FT817_EEPROM_ADDR_VFO_BAND_SELECT },
 { "CW Pitch, etc",    FT817_EEPROM_ADDR_CW_PITCH_ETC    },
 { "CW Weight, etc",   FT817_EEPROM_ADDR_CW_WEIGHT_ETC   },
 { "Sidetone Volume",  FT817_EEPROM_ADDR_SIDETONE_VOL    },
 { "CW Speed",         FT817_EEPROM_ADDR_CW_SPEED        },
 { "VOX Gain, etc",    FT817_EEPROM_ADDR_VOX_GAIN_ETC    },
 { "VOX Delay, etc",   FT817_EEPROM_ADDR_VOX_DELAY_ETC   },
 { "APO Time, etc",    FT817_EEPROM_ADDR_APO_TIME_ETC    },
 { "TX Power, etc",    FT817_EEPROM_ADDR_TX_POWER_ETC    },
 { NULL, 0 } // "all zeros" mark the end of the list
}; // end FT817_EEPROM_Addresses[]



//----------------------------------------------------------------------------
// Global variables ...
//----------------------------------------------------------------------------




//--------------------------------------------------------------------------
// Internal function prototypes (forward declarations) ...
//--------------------------------------------------------------------------

static int OnRespPowerOnOff( struct t_Yaesu5ByteControl *pYC );
static BOOL Y5B_StartReadingNextParamForRigcontrol( T_Yaesu5ByteControl *pYC );
static void Y5B_PresetCommentForMessageOrigin( T_Yaesu5ByteControl *pYC, int iRigCtrlOrigin );
static void Y5B_PresetMsgTypeForMessageOrigin( T_Yaesu5ByteControl *pYC, int iRigCtrlOrigin );




//--------------------------------------------------------------------------
// Implementation of functions
//--------------------------------------------------------------------------

//--------------------------------------------------------------------------
void Yaesu5Byte_InitStruct( T_Yaesu5ByteControl *pYC, // [out] 'Yaesu 5-byte CAT' decoder or controller instance
       void *pvRigControl, // [in] optional T_RigCtrlInstance* (may be NULL for "stand-alone use")
       T_Y5B_OnDecodeCallback pOnDecodeCallback ) // [in] optional callback for 'decoded COMMANDS or RESPONSES'
{
  memset( pYC, 0, sizeof(T_Yaesu5ByteControl) );

  pYC->iExpectingResponseForCmd = -1; // <0 when NOT expecting to received the response for a command

  pYC->pvRigControl      = pvRigControl;
  pYC->pOnDecodeCallback = pOnDecodeCallback;

} // end Yaesu5Byte_InitStruct()


//--------------------------------------------------------------------------
void Yaesu5Byte_ForgetAllParams( T_Yaesu5ByteControl *pYC )
{
  // Since T_Yaesu5ByteControl doesn't store any parameters internally yet
  //  (not even the radio's current VFO frequency and operating mode / modulation),
  // there's nothing to do here  ...  at least NOT YET.
  // See also: RigCtrl_ForgetAllParams() [calls Yaesu5Byte_ForgetAllParams()] .

} // end Yaesu5Byte_ForgetAllParams()


//--------------------------------------------------------------------------
static const T_Y5B_CommandTabEntry* Y5B_GetCommandInfo( BYTE bCmd )
  // May return NULL of the command byte (bCmd) is unknown ! ! !
{ const T_Y5B_CommandTabEntry *pCmdInfo = Y5B_CommandTable;
  while( (pCmdInfo->bCmd != 0x00) || (pCmdInfo->pCmdParser != NULL) )
   { if( pCmdInfo->bCmd == bCmd )
      { return pCmdInfo;
      }
     ++pCmdInfo;
   }
  return NULL;
} // end Y5B_GetCommandInfo()

//--------------------------------------------------------------------------
static DWORD Y5B_ParseBigEndian16Bit( BYTE *pbSource ) // ... aka "MSByte first"
{
  return ( (DWORD)pbSource[0] << 8 ) | pbSource[1];
} // end Y5B_ParseBigEndian16Bit()

//--------------------------------------------------------------------------
static DWORD Y5B_ParseBigEndian32Bit( BYTE *pbSource ) // ... aka "MSByte first"
{
  return ( (DWORD)pbSource[0] << 24 ) | ( (DWORD)pbSource[1] << 16 )
      |  ( (DWORD)pbSource[2] << 16 ) | ( (DWORD)pbSource[3] << 0 );
} // end Y5B_ParseBigEndian32Bit()

//--------------------------------------------------------------------------
static DWORD Y5B_ParseBigEndianBCD( BYTE *pbSource, int nDigits )
{ DWORD dwResult = 0;
  BYTE b;

  while( nDigits > 0 )
   {
     b = *(pbSource++);
     --nDigits;
     dwResult = (10 * dwResult) + ( b >> 4 ); // most significant 4 bits FIRST

     if( nDigits > 0 ) // also get the lower DIGIT from the same byte ?
      { dwResult = (10 * dwResult) + ( b & 15 ); // least significant 4 bits LAST
        --nDigits;
      }
     // Worried about 'pseudo tetrades' ? Nope. Garbage in, Garbage out !
   }

  return dwResult;
}

//---------------------------------------------------------------------------
static BYTE Y5B_ByteToBCD( BYTE bValue ) // subroutine for Y5B_Uint32ToBCD8(), details THERE.
{ // Note: With 'garbage in' (bValue > 99), there will be 'garbage out' !
  return (bValue % 10) | ( (bValue/10) << 4);
  //     |___________|   |_________________|----- upper digit in bits 7..4 (0..9)
  //           '--------------------------------- lower digit in bits 3..0 (0..9)
} // end Y5B_ByteToBCD()


//---------------------------------------------------------------------------
DWORD Y5B_Uint32ToBCD8( DWORD dwValue )
  // Converts an unsigned 32-bit unsigned integer into the 8-bit BINARY CODED DECIMAL
  // as required for the FT-818's command 0x01 = YAESU5B_CMD_SET_FREQ .
  // The result is a DWORD (Microsoft Windows term for unsigned 32 bit),
  // suitable for transmission via Yaesu5Byte_SendAndLogCommand( dwParameter ) .
  //
  // [in]    : dwValue = an ordinaty 32-bit unsigned integer in THE HOST'S ENDIANNESS.
  // [return]: bits  7..0 are in the byte SENT FIRST in Yaesu5Byte_SendAndLogCommand(!)
  //            bits 15..8 are sent next,  (..)
  //            bits 31..24 are sent last in Yaesu5Byte_SendAndLogCommand(!)
{
  union
   { BYTE b4[4];
     DWORD dw;
   } u;


  // Begin with the least significant TWO DECIMAL DIGITS, as a "Binary Coded Decimal" (00..99):
  u.b4[3] = Y5B_ByteToBCD( (BYTE)( dwValue % 100UL ) );
  dwValue /= 100UL;  // what remains are the original "higher digits"..

  u.b4[2] = Y5B_ByteToBCD( (BYTE)( dwValue % 100UL ) );
  dwValue /= 100UL;  // you can imagine how badly an 8-bit PIC microcontroller will chew on this ..

  u.b4[1] = Y5B_ByteToBCD( (BYTE)( dwValue % 100UL ) ); // SOOO many divisions and modulo-ten operations :)
  dwValue /= 100UL;

  // Finish with the most significant TWO DIGITS, as a "Binary Coded Decimal":
  u.b4[0] = Y5B_ByteToBCD( (BYTE)( dwValue % 100UL ) );

  return u.dw;  // return the 8-digit BINARY CODED DECIMAL in a 32-bit (4-byte) value

} // end Y5B_Uint32ToBCD8()

//---------------------------------------------------------------------------
BOOL Y5B_OpModeToRigControl( BYTE bYaesuOpMode, int *piRigCtrlOpMode, const char **ppszMode )
{
  char *pszMode = NULL;
  int  iRigControlOpMode = RIGCTRL_OPMODE_UNKNOWN;
  BOOL fRecognized = TRUE;
  switch( bYaesuOpMode )
   { case 0x00: iRigControlOpMode = RIGCTRL_OPMODE_LSB; pszMode = "LSB"; break;
     case 0x01: iRigControlOpMode = RIGCTRL_OPMODE_USB; pszMode = "USB"; break;
     case 0x02: iRigControlOpMode = RIGCTRL_OPMODE_CW;  pszMode = "CW"; break;
     case 0x03: iRigControlOpMode = RIGCTRL_OPMODE_CW | RIGCTRL_OPMODE_REVERSE; pszMode = "CWR"; break;
     case 0x04: iRigControlOpMode = RIGCTRL_OPMODE_AM;  pszMode = "AM"; break;
     case 0x06: iRigControlOpMode = RIGCTRL_OPMODE_FM_WIDE; pszMode = "FM-WIDE"; break;
          // Note the absence of case 0x07.  Eaten by the CAT !
     case 0x08: iRigControlOpMode = RIGCTRL_OPMODE_FM;   pszMode = "FM"; break;
     case 0x0A: iRigControlOpMode = RIGCTRL_OPMODE_RTTY; pszMode = "DIG"; break; // When Yaesu says "DIG", Icom says "RTTY" :o)
     case 0x0C: pszMode = "PKT"; break;
     case 0x82: iRigControlOpMode = RIGCTRL_OPMODE_CW | RIGCTRL_OPMODE_NARROW;  pszMode = "CWN"; break;
     case 0x83: iRigControlOpMode = RIGCTRL_OPMODE_CW | RIGCTRL_OPMODE_REVERSE | RIGCTRL_OPMODE_NARROW;
                pszMode = "CWRN"; break;

     default:   fRecognized = FALSE;
                break;
   }
  // Pass back the results ? (both outputs are OPTIONAL)
  if( piRigCtrlOpMode != NULL )
   { *piRigCtrlOpMode = iRigControlOpMode;
   }
  if( ppszMode != NULL )
   { *ppszMode = pszMode;
   }
  return fRecognized; // -> TRUE=ok ("recognized the Yeasu Op-Mode and converted it to a RigControl-Op-Mode")
} // end Y5B_OpModeToRigControl()


//--------------------------------------------------------------------------
int Yaesu5Byte_ProcessData( T_Yaesu5ByteControl *pYC, // [in,out] 'Yaesu 5-byte CAT' instance
        int iRigCtrlPort,   // [in] e.g. RIGCTRL_PORT_RADIO, RIGCTRL_PORT_AUX_COM_1, ..,
                            //      but also RIGCTRL_PORT_AUX_COM_1/2/3 to "eavesdrop" on a SERIAL PORT TUNNEL or similar
        int iRigCtrlOrigin, // [in] e.g. RIGCTRL_ORIGIN_RADIO, RIGCTRL_ORIGIN_CONTROLLER, RIGCTRL_ORIGIN_COM_PORT_RX/TX
                            // (especially important in "Listen-Only", aka "eavesdropping" mode!)
        BYTE *pbData, int nBytes ) // [in] chunk of received data (not necessarily a complete, nor a SINGLE 'Message')
                            // [out] usually pYC->bRxBuffer[pYC->nBytesInRxBuffer++],
                            //       until pYC->nBytesInRxBuffer reaches pYC->iExpectedResponseLength,
                            //       or a VALID COMMAND has been detected.
        // When that happens, the command specific parser will be called,
        //   and optionally the caller's pYC->OnDecodeCallback is invoked.
        //   Yaesu5Byte_ProcessData() may invoke pYC->OnDecodeCallback
        //   a couple of times, e.g. to emit decoded COMMANDS and RESPONSES
        //   to the log, etc (that's the caller's decision).
        // When used on a 'serial port tunnel', commands and responses may
        //   arrive in both directions. In that case, iOrigin differentiates
        //   data in ONE direction from data travelling in the OTHER direction.
        //   See caller, e.g. AuxComPorts.c : AuxCom_RunSerialPortTunnel() .
  // [return] iMsgType or -1 when nothing decodeable
  // Callers:
  //   (a) after RECEPTION of data from a 'local' COM port for 'Serial Port Tunnelling':
  //       AuxPortThread() -> ? ..
  //   (b) shortly before TRANSMISSION ("WriteFile()") of data to a 'local' COM port :
  //       AuxComThread() -> AuxComPassRxOrTxDataToProtocolSpecificDecoder()
  //                             with iRigCtrlOrigin = RIGCTRL_ORIGIN_COM_PORT_TX
{
  const T_Y5B_CommandTabEntry *pCmdInfo;
  int   iParserResult = 0;
  BOOL  fProcessNow;
  int   nExcessiveBytes;

  // If this Y5B-protocol-instance is now fed with data travelling e.g.
  //  on a DIFFERENT PORT, or in a DIFFERENT DIRECTION,
  //  or data received from the OTHER end of a 'serial port tunnel',
  //  this marks the end of a PREVIOUS COMMAND or RESPONSE :
  if( (pYC->iRigCtrlPort   != iRigCtrlPort)
   || (pYC->iRigCtrlOrigin != iRigCtrlOrigin) )  // finish previous block because e.g. the direction of data flow (rx,tx) has changed ?
   {
     if( pYC->nBytesInRxBuffer > 0 )  // something left to process (or show as hex dump) ?
      {           // (often got here with pYC->nBytesInRxBuffer = 1..2 = a RESPONSE from the radio)
        // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
        // Example:
        // When called from AuxComPorts.c : AuxCom_PassRxOrTxDataToProtocolSpecificDecoder(),
        //      shortly before SENDING e.g. a 5-byte-command to an FT-817 via 'WriteFile',
        //   pYC->nBytesInRxBuffer will often be 2 (!) at this point,
        //  * first for the OLD data (still waiting in the decoder's internal buffer mentioned above,
        //      because the not-yet-flushed bytes were a two-byte response
        //      from the 'READ EEPROM' command shelled out by "FT817 Commander".
        //      When displayed on the RCW keyer's "Debug" tab, those two bytes
        //      shall appear as e.g.
        //       > COM8: RX[02] 0x48 00          ; Read EEPROM response
        //  * then for the NEW data ( pbData, often with nBytes = 5 = a complete "Yaesu 5-Byte COMMAND".
        //       > COM8: TX[05] 0x01 B6 00 00 BB ; Read EEPROM, addr=0x01B6
        //
        // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

        if( pYC->iExpectingResponseForCmd >= 0 ) // parse an EXPECTED RESPONSE ? (here: because the direction of data (rx,tx) has changed)
         { Y5B_PresetMsgTypeForMessageOrigin( pYC, pYC->iRigCtrlOrigin ); // set pYC->iCurrentMsgType to a meaningful default
           pCmdInfo = Y5B_GetCommandInfo( pYC->iExpectingResponseForCmd );
           if( (pCmdInfo!=NULL) && (pCmdInfo->pRespParser != NULL) )    // note the EVALUATION SEQUENCE (left-to-right) !
            { iParserResult = pCmdInfo->pRespParser( pYC ); // [in] pYC->iRigCtrlOrigin, pYC->bRxBuffer, pYC->nBytesInRxBuffer, ..
                 // ,----------------------'                // [in,out] pYC->iCurrentMsgType
                 // '--> e.g. OnRespReadFreq(), for YAESU5B_CMD_READ_FREQ_AND_MODE .
                 // Overwrites the 'default comment' with something more useful like .. guess what.
            }
         } // end if( pYC->iExpectingResponseForCmd >= 0 )

        // Before discarding pYC->bRxBuffer[0..pYC->nBytesInRxBuffer-1] further below,
        // allow THE CALLER to emit e.g. a commented hex dump to a logfile, or show it on RCW-Keyer's 'DEBUG' tab, etc.
        // Because iRigCtrlOrigin has changed, iParserResult doesn't matter here .. it's ALWAYS the end of a block.
        if( pYC->pOnDecodeCallback != NULL )
         {  pYC->pOnDecodeCallback( pYC, iRigCtrlPort, pYC->iRigCtrlOrigin/*old!*/, pYC->bRxBuffer, pYC->nBytesInRxBuffer, pYC->sz80Comment );
            // '--> e.g. AuxComPorts.c : AuxCom_Y5B_OnDecodeCallback() -> AuxCom_DumpMessageToLog() -> ShowError() [thread-safe]
         }
        pYC->nBytesInRxBuffer = 0;           // discard everything in pYC->bRxBuffer[]
        if( pYC->iExpectingResponseForCmd >= 0 ) // <- only to set a breakpoint below.. WHEN do we clear this ? WATCH pYC->sz80Comment !
         {  pYC->iExpectingResponseForCmd = pYC->iExpectingResponseForCmd; // <- place for a breakpoint (no other purpose)
         }
        pYC->iExpectingResponseForCmd = -1;  // "done" (until the NEXT command)
        pYC->fBusy = FALSE; // not "busy from transmission of a command" anymore
      } // end if( pYC->nBytesInRxBuffer > 0 )
     else // the direction of data flow (rx,tx) has changed, but NOTHING has been received yet ->
      { // don't clear pYC->iExpectingResponseForCmd and pYC->iExpectedResponseLength here !
      }
   } // end < called on behalf of a DIFFERENT "Rig Control Port" or "Origin" > ?


  // Prepare former FUNCTION ARGUMENTS for any 'parser' invoked from the loop below:
  pYC->iRigCtrlPort  = iRigCtrlPort;   // [in] former FUNCTION ARGUMENT for any 'parser' invoked below
  pYC->iRigCtrlOrigin= iRigCtrlOrigin; // [in] e.g. RIGCTRL_ORIGIN_RADIO, RIGCTRL_ORIGIN_CONTROLLER, RIGCTRL_ORIGIN_COM_PORT_RX/TX

  // Process as many byte as we have at the moment (not related to boundaries between COMMANDS and RESPONSES) :
  while( nBytes-- )
   {
     if( pYC->nBytesInRxBuffer >= YAESU5B_RX_BUFFER_SIZE )  // buffer already FULL ?
      { // This means in the previous loops or calls, garbage was received,
        // and we're still waiting for a valid COMMAND BYTE (*after* the parameters).
        // "Move up" the NON-CIRCULAR buffer to give room for the new received byte.
        // Because source and destination overlap, use memmove(), not memcpy() :
        memmove( pYC->bRxBuffer/*destination*/, pYC->bRxBuffer+1/*source*/,
                 YAESU5B_RX_BUFFER_SIZE-1/* number of bytes to copy */ );
        pYC->nBytesInRxBuffer = YAESU5B_RX_BUFFER_SIZE-1;
      }
     // After the above 'buffer scroll', the next byte can be APPENDED:
     pYC->bRxBuffer[pYC->nBytesInRxBuffer++] = *(pbData++);

     if( (pYC->iParameterPollingState == RIGCTRL_POLLSTATE_TURN_RIG_ON )
      ||( pYC->iParameterPollingState == RIGCTRL_POLLSTATE_WAIT_TURN_ON)
      ||( pYC->iParameterPollingState == RIGCTRL_POLLSTATE_TURN_RIG_OFF)
      ||( pYC->iParameterPollingState == RIGCTRL_POLLSTATE_WAIT_TURN_OFF)
      ||( pYC->iParameterPollingState == RIGCTRL_POLLSTATE_TURNED_OFF  ) )
      { // YAESU5B_CMD_TURN_ON/OFF sometimes causes a response, sometimes not...
        Y5B_PresetMsgTypeForMessageOrigin( pYC, pYC->iRigCtrlOrigin ); // set pYC->iCurrentMsgType to a meaningful default
        Y5B_PresetCommentForMessageOrigin( pYC, pYC->iRigCtrlOrigin ); // set pYC->sz80Comment to a meaningful default
        OnRespPowerOnOff( pYC );    // update sz80Comment, and stop waiting (?)
        if( pYC->pOnDecodeCallback != NULL ) // not really a "decode" but optionally show this somewhere
         {  pYC->pOnDecodeCallback( pYC, iRigCtrlPort, iRigCtrlOrigin, pYC->bRxBuffer, pYC->nBytesInRxBuffer, pYC->sz80Comment );
         }
        pYC->nBytesInRxBuffer = 0;
        pYC->iExpectingResponseForCmd = -1;  // "done" (until the NEXT command)
        pYC->fBusy = FALSE; // not "busy from transmission of a command" anymore
        // Results (shown on RCW Keyer's DEBUG tab) :
        // >    1 22:01:04.4 TX 005 FF FF FF FF FF   ; Sync before Power-On
        // >    2 22:01:04.5 TX 005 00 00 00 00 0F          ; Turn Power ON
        // >    3 22:01:04.6 RX 001 00                  ; Power-ON response
        // ,-----------------|_______|
        // '--> This single-byte RESPONSE only arrived when the FT-817 was PREVIOUSLY OFF.
        // >    4 22:04:31.5 TX 005 00 00 00 00 8F         ; Turn Power OFF
        // >    5 22:04:31.6 RX 001 00                 ; Power-OFF response
        // ,-----------------|_______|
        // '--> This single-byte RESPONSE only arrived when the FT-817 was PREVIOUSLY ON.
        //
      } // end if < in one of the special "Turning Power ON" / "Turning Power OFF" states > ?

     if( pYC->iExpectingResponseForCmd >= 0 )
      { // '--> When 'eavesdropping' on "FT817 Commander",
        //          iExpectingResponseForCmd was set in Yaesu5Byte_ProcessData() ON RECEPTION of a COMMAND.
        //      When actively controlling e.g. an FT-817 from RCW Keyer,
        //          iExpectingResponseForCmd was set in Yaesu5Byte_AskForParameter() when SENDING a COMMAND.
        //      Similar applies to iExpectedResponseLength, which is -if known and "predictable"-
        //          set along with pYC->iExpectingResponseForCmd .
        pCmdInfo = NULL;
        if( pYC->iExpectedResponseLength > 0 ) // if we now the LENGTH of the response..
         { fProcessNow = ( pYC->nBytesInRxBuffer >= pYC->iExpectedResponseLength);
         }
        else // "expecing a response" but no "expected response length" specified:
         { fProcessNow = ( pYC->nBytesInRxBuffer > 0 ); // oh well... we KNOW nBytesInRxBuffer is > 0, but keep this for future modifications
         }
        if( fProcessNow )
         {
           // In the 'Yaesu-5-byte-COMMAND' protocol the RESPONSE may have *any* length !
           // One of the facts that make this protocol so ugly :
           //  The number of bytes in the RESPONSE is not alway predictable !
           //  For example (from KA7OEI) :
           // > Command BD - Read TX Metering:  This command returns one byte (always 00)
           // > when the FT-817 is in receive mode (and should thus be ignored)
           // > but it returns two bytes (each containing two BCD digits)
           // > when it is transmitting.
           //  Thus, a fully-fledged implementation of the "Yaesu-5-byte-COMMAND-protocol"
           //  would need a timeout monitor for responses with a variable length !
           pCmdInfo = Y5B_GetCommandInfo( pYC->iExpectingResponseForCmd );
           if( pCmdInfo != NULL )
            { // Got all we need, so let the optional RESPONSE-parser take a look:
              iParserResult = 0;
              Y5B_PresetMsgTypeForMessageOrigin( pYC, pYC->iRigCtrlOrigin ); // set pYC->iCurrentMsgType to a meaningful default
              // No-No (not HERE): Y5B_PresetCommentForMessageOrigin( pYC, pYC->iRigCtrlOrigin );
              if( pCmdInfo->pRespParser != NULL )
               { iParserResult = pCmdInfo->pRespParser( pYC ); // [in] pYC->iRigCtrlOrigin, pYC->bRxBuffer, pYC->nBytesInRxBuffer, ..
                 // ,----------------------'
                 // '--> e.g. with pYC->iExpectingResponseForCmd = 3 = YAESU5B_CMD_READ_FREQ_AND_MODE,
                 //                pRespParser points to OnRespReadFreq() .
                 //  The PARSED RESULTS (e.g. frequency+mode) are stored in the
                 //  T_RigCtrlInstance (e.g. .dblVfoFrequency), so the GUI can show them.
                 //  The following call (also via function pointer, lands in RigControl.c, too)
                 //  allows the Rig-Control-module to do whatever it needs to,
                 //  for example report the NEW frequency to remote clients, etc.
                 //  Besides that, the call below optionally queues up the message
                 //  in a FIFO for the display in the log / on the DEBUG tab.
               }
              if( pYC->pOnDecodeCallback != NULL ) // e.g. RigCtrl_Y5B_OnDecodeCallback() ...
               {  pYC->pOnDecodeCallback( pYC, iRigCtrlPort, iRigCtrlOrigin,
                       pYC->bRxBuffer, pYC->nBytesInRxBuffer, pYC->sz80Comment );
               }
            }   // end if < pCmdInfo != NULL >
           else // was "expecting a RESPONSE for a certain command", but no entry in the command-table for it ->
            { sprintf( pYC->sz80Comment, "expecting response for cmd 0x%02X, but missing a RESPONSE-PARSER.",
                      (unsigned int)pYC->iExpectingResponseForCmd );
              if( pYC->pOnDecodeCallback != NULL )
               {  pYC->pOnDecodeCallback( pYC, iRigCtrlPort, iRigCtrlOrigin, pYC->bRxBuffer, pYC->nBytesInRxBuffer, pYC->sz80Comment );
               }
            } // end else  < pCmdInfo == NULL  for the "command that expected a RESPONSE >
           pYC->nBytesInRxBuffer = 0;
           if( pYC->iExpectingResponseForCmd >= 0 ) // <- only to set a breakpoint below.. WHEN do we clear this ?
            { pYC->iExpectingResponseForCmd = pYC->iExpectingResponseForCmd; // <- place for a breakpoint (no other purpose)
            }
           pYC->iExpectingResponseForCmd = -1;  // "done" (until the NEXT command)
           pYC->fBusy = FALSE; // not "busy from transmission of a command" anymore
         }  // end if < fProcessNow > (A)
      }    // end if( pYC->iExpectingResponseForCmd >= 0 )
     else // not expecting a RESPONSE at the moment, so this may be a (5-byte-)COMMAND ->
      { if( pYC->nBytesInRxBuffer >= 5 )
         { Y5B_PresetMsgTypeForMessageOrigin( pYC, pYC->iRigCtrlOrigin ); // set pYC->iCurrentMsgType to a meaningful default
           Y5B_PresetCommentForMessageOrigin( pYC, pYC->iRigCtrlOrigin ); // set pYC->sz80Comment .. if pCmdParser() doesn't set it
           // Check the **LAST** byte in the buffer .. that should be the COMMAND(!) in this crazy protocol !
           pCmdInfo = Y5B_GetCommandInfo( pYC->bRxBuffer[pYC->nBytesInRxBuffer-1] );
           if( pCmdInfo!=NULL ) // note the EVALUATION SEQUENCE (left-to-right) !
            { if( pYC->iExpectingResponseForCmd >= 0 ) // <- only to set a breakpoint below.. WHEN do we clear this ?
               {  pYC->iExpectingResponseForCmd = pYC->iExpectingResponseForCmd; // <- place for a breakpoint (no other purpose)
               }
              pYC->iExpectingResponseForCmd = -1;  // <- set default HERE, just in case pCmdInfo->pCmdParser() leaves this untouched
              if( pCmdInfo->pCmdParser != NULL )
               { iParserResult = pCmdInfo->pCmdParser( pYC ); // [in]  pYC->iRigCtrlOrigin, pYC->bRxBuffer, pYC->nBytesInRxBuffer, ..
                                                              // [out] pYC->iExpectingResponseForCmd, pYC->iExpectedResponseLength (!)
                 // For example, OnCmdReadEEPROM() [called above via function pointer]
                 // has set pYC->iExpectingResponseForCmd to YAESU5B_CMD_READ_EEPROM
                 //     and pYC->iExpectedResponseLength  to 2 (!) [bytes] .
                 // That way, we can parse the response as soon as it arrives,
                 // instead of having to wait for '(pYC->iRigCtrlOrigin != iRigCtrlOrigin)' !
               }
            }
           if( pYC->pOnDecodeCallback != NULL )
            {  pYC->pOnDecodeCallback( pYC, iRigCtrlPort, iRigCtrlOrigin, pYC->bRxBuffer, pYC->nBytesInRxBuffer, pYC->sz80Comment );
            }
           pYC->nBytesInRxBuffer = 0;
           pYC->fBusy = FALSE; // not "busy from transmission of a command" anymore
         } // end if( pYC->nBytesInRxBuffer >= 5 )
      }   // end else < not expecting a RESPONSE, so check for complete reception of a 5-byte-COMMAND >
   }     // end while( nBytes-- )
  (void)iParserResult; // ... assigned a value that is never used ..
                       // shut up, better than using a non-initialized value
  return pYC->iCurrentMsgType;
} // end Yaesu5Byte_ProcessData()

//---------------------------------------------------------------------------
static void Y5B_PresetCommentForMessageOrigin( T_Yaesu5ByteControl *pYC, int iRigCtrlOrigin )
  // [in]  iRigCtrlOrigin      : e.g. RIGCTRL_ORIGIN_COM_PORT_RX / .. TX / ...
  // [out] pYC->sz80Comment    : e.g. "received response ?
  // Note: If everything works as planned, the "default comments" set below
  //       will never appear on the log. Instead, when powering on an FT-817,
  //       the 'Rig Control traffic log' on RCW Keyer's "DEBUG" tab was:
  // >    1 21:11:42.5 TX 005 FF FF FF FF FF      ; Sync before Power-On
  // >    2 21:11:42.6 TX 005 00 00 00 00 0F             ; Turn Power ON
  // >    3 21:11:42.7 RX 001 00                     ; Power-ON response  <--- only appeared when the FT-817 *WAS OFF*
  // >    4 21:11:42.8 TX 005 00 00 00 00 03              ; read VFOfreq
  // >    5 21:11:43.2 RX 005 14 43 00 00 01        ; 144.30000 MHz, USB
  // human-friendly comments from various parsers ----|________________|
{
  switch( iRigCtrlOrigin )
   { case RIGCTRL_ORIGIN_RADIO      : // pYC->bRxBuffer[] contains data originating from a RADIO:
        // Since we're usually not a radio, "Origin=RADIO" means we RECEIVED this
        strcpy( pYC->sz80Comment, "response from radio ?" );
        break;
     case RIGCTRL_ORIGIN_CONTROLLER : // pYC->bRxBuffer[] contains data originating from a CONTROLLER:
        // Since we're most likely a controller (not a radio), treat the buffer content like been SENT.
        strcpy( pYC->sz80Comment, "%d-byte command from controller ?" );
        break;
     case RIGCTRL_ORIGIN_COM_PORT_RX: // pYC->bRxBuffer[] contains data RECEIVED FROM a local 'COM' port
        strcpy( pYC->sz80Comment, "received response ?" );
        break;
     case RIGCTRL_ORIGIN_COM_PORT_TX: // pYC->bRxBuffer[] contains data SENT TO a local 'COM' port
        strcpy( pYC->sz80Comment, "sent command ?" );
        break;
     default:
        strcpy( pYC->sz80Comment, "unknown 'Origin' ?" );
        break;
   } // end switch( pYC->iRigCtrlOrigin )
} // end Y5B_PresetCommentForMessageOrigin()

//--------------------------------------------------------------------------
static void Y5B_PresetMsgTypeForMessageOrigin( T_Yaesu5ByteControl *pYC, int iRigCtrlOrigin )
  // [in]  iRigCtrlOrigin      : e.g. RIGCTRL_ORIGIN_COM_PORT_RX / .. TX / ...
  // [out] pYC->iCurrentMsgType: e.g. RIGCTRL_MSGTYPE_FLAG_RX |
  //            iCurrentMsgType is only important for the LOG/DEBUG display.
{
  switch( iRigCtrlOrigin )
   { case RIGCTRL_ORIGIN_RADIO      : // pYC->bRxBuffer[] contains data originating from a RADIO:
        // Since we're usually not a radio, "Origin=RADIO" means we RECEIVED this
        pYC->iCurrentMsgType = RIGCTRL_MSGTYPE_FLAG_RX | RIGCTRL_MSGTYPE_OTHER;
         // ,--------------------------------------------|___________________|
         // '--> "The message may an acknowledge / response / value for 'set', with some 'other' parameter".
         //      The parser called via function pointer later may replace this bitflag
         //      by e.g. RIGCTRL_MSGTYPE_FREQUENCY_REPORT, or whatever fits best.
        break;
     case RIGCTRL_ORIGIN_CONTROLLER : // pYC->bRxBuffer[] contains data originating from a CONTROLLER:
        // Since we're most likely a controller (not a radio), treat the buffer content like been SENT.
        pYC->iCurrentMsgType = RIGCTRL_MSGTYPE_FLAG_TX | RIGCTRL_MSGTYPE_OTHER;
        break;
     case RIGCTRL_ORIGIN_COM_PORT_RX: // pYC->bRxBuffer[] contains data RECEIVED FROM a local 'COM' port
     default:
        pYC->iCurrentMsgType = RIGCTRL_MSGTYPE_FLAG_RX | RIGCTRL_MSGTYPE_OTHER;
        break;
     case RIGCTRL_ORIGIN_COM_PORT_TX: // pYC->bRxBuffer[] contains data SENT TO a local 'COM' port
        pYC->iCurrentMsgType = RIGCTRL_MSGTYPE_FLAG_TX | RIGCTRL_MSGTYPE_OTHER;
        break;
   } // end switch( pYC->iRigCtrlOrigin )
} // end Y5B_PresetMsgTypeForMessageOrigin()

//--------------------------------------------------------------------------
static BOOL Yaesu5Byte_SendAndLogMessage(
        T_Yaesu5ByteControl *pYC, // [in,out] 'Yaesu 5-byte CAT' instance.
                                  //   In many cases, connected to a T_RigCtrlInstance.
        BYTE *pbMessage, int iMsgLength, // [in] message to send
        int iMsgType,     // [in] message type category and 'flags', only for the traffic log.
                          //      (see RigControl.c : RigCtrl_ReadTrafficLog() for details
                          //       about how this message appears 'in the log'.
             // In case of doubt, use iMsgType = RIGCTRL_MSGTYPE_OTHER to show a HEX DUMP.
        char *pszComment) // [in] human readable info, short enough for the traffic log
  // Sends a "Yaesu-5-Byte"-CAT-message or (for special applications) an N-byte-RESPONSE,
  // and -when successful- counts and optionally logs it.
  // (The real transmission on e.g. a serial port runs in a separate thread
  //  or interrupt handler, see RigControl.c )
{
  int iResult, iErrorCode, iNumMsWaited; // optional results from IO_SendToCOMPort()
  char sz255Msg[256];
  T_RigCtrlInstance *pRC = (T_RigCtrlInstance*)pYC->pvRigControl; // <- this may be NULL !

  if( pRC != NULL )
   { return RigCtrl_SendAndLogMessage( pRC, pYC->iRigCtrlPort, RIGCTRL_ORIGIN_CONTROLLER,
          pbMessage, iMsgLength, RIGCTRL_MSGTYPE_FLAG_TX | iMsgType, pszComment );
   }
  return FALSE; // cannot SEND without a RigControl instance (only "eavesdrop" on received data)

} // end Yaesu5Byte_SendAndLogMessage()

//--------------------------------------------------------------------------
static BOOL Yaesu5Byte_SendAndLogCommand(
        T_Yaesu5ByteControl *pYC, // [in,out] 'Yaesu 5-byte CAT' instance.
                                  //   In many cases, connected to a T_RigCtrlInstance.
        BYTE bOpcode,      // [in] 8-bit "Opcode" (terminology from the FT-817ND Operating Manual)
        DWORD dwParameter, // [in] 32-bit "Parameter", LEAST SIGNIFICANT BYTE SENT FIRST !
        int iMsgType,     // [in] message type category and 'flags', only for the traffic log
        char* pszComment) // [in] human readable info, short enough for the traffic log,
                          //      e.g. "read VFOfreq",
{
  BYTE b8TxBuffer[8];
  BYTE *pbBufPtr = b8TxBuffer;

  // Convert from the HOST BYTE ORDER (here: Intel, least significant byte first)
  // into the dreadful Big Endian format (most significant byte first) ?
  // NO, because most commands only use the FIRST of the FOUR PARAMETER BYTES,
  //     and we don't want to have that in the MOST SIGNIFICANT BYTE of dwParameter.
  // ... so beware with the Set Frequency command in an FT-817, which uses the
  //     dreadful BIG ENDIAN format. DWORD Y5B_Uint32ToBCD8() takes care of that.
  *pbBufPtr++ = (BYTE)dwParameter;
  *pbBufPtr++ = (BYTE)(dwParameter>>8);
  *pbBufPtr++ = (BYTE)(dwParameter>>16);
  *pbBufPtr++ = (BYTE)(dwParameter>>24);

  // We will never understand why Yeasu expects the COMMAND *after* the command's parameters.
  // The receiver must pull all parameter bytes from the UART before he knows how to process them.
  *pbBufPtr++ = bOpcode;
  pYC->fBusy = TRUE;          // now busy from the transmission of a COMMAND 
  TIM_StartStopwatch( &pYC->sw_BusyTimer );
  return Yaesu5Byte_SendAndLogMessage( pYC, b8TxBuffer,
             pbBufPtr-b8TxBuffer/*iMsgLength*/, iMsgType, pszComment );
} // end Yaesu5Byte_SendAndLogCommand()


//--------------------------------------------------------------------------
int Yaesu5Byte_Handler( T_Yaesu5ByteControl *pYC ) // Periodically called every 50(?) milliseconds
  // from the same task / thread / interrupt handler that also calls Yaesu5Byte_ProcessData().
  // When used for RIG CONTROL in the Remote CW Keyer, called from RigCtrl_Handler() .
{
  BOOL fTimeout = (TIM_ReadStopwatch_ms(&pYC->sw_ParameterPolling) > 100);
  switch( pYC->iParameterPollingState )  // see 'specifications' of these states in RigControl.h ..
   { // the states below are roughly ordered by the sequence we run through them...
     case RIGCTRL_POLLSTATE_PASSIVE  : // initial state: have neither READ nor WRITTEN any setting from/to the remote rig
        break;
     case RIGCTRL_POLLSTATE_TURN_RIG_ON:   // special service for rigs that can be turned on / off via CAT (e.g. IC-7300 via USB/CI-V)
        if( fTimeout ) // waited "long enough" after the last normal traffic ?
         { TIM_StartStopwatch(&pYC->sw_ParameterPolling);
           Yaesu5Byte_SendAndLogCommand( pYC,
                   0xFF,         // undocumented "sync pattern" sent BEFORE the "Power ON" command
                   0xFFFFFFFFUL, // 4-byte parameter: also filled with the "sync pattern"
                RIGCTRL_MSGTYPE_FLAG_TX | RIGCTRL_MSGTYPE_POWER_ON_OFF, // [in] iMsgType, only for the traffic log
                "Sync before Power-On" ); // [in] char *pszComment
           pYC->iParameterPollingState = RIGCTRL_POLLSTATE_WAIT_TURN_ON;
           pYC->iParameterPollingSubState = 0;
         } // end if < waited long enough .. >
        break; // end case RIGCTRL_POLLSTATE_TURN_RIG_ON
     case RIGCTRL_POLLSTATE_WAIT_TURN_ON:  // waiting for an "OK"-response after the command to "turn on"
        if( fTimeout ) // waited "long enough" after the last normal traffic ?
         { TIM_StartStopwatch(&pYC->sw_ParameterPolling);
           switch( pYC->iParameterPollingSubState )
            { case 0 :
                 Yaesu5Byte_SendAndLogCommand( pYC,
                      YAESU5B_CMD_TURN_ON, // [in] 8-bit "Opcode" (terminology from the FT-817ND Operating Manual):
                      0x00000000UL, // [in] 32-bit "Parameter", ignored for the above opcode
                      RIGCTRL_MSGTYPE_FLAG_TX | RIGCTRL_MSGTYPE_POWER_ON_OFF, // [in] iMsgType, only for the traffic log
                      "Turn Power ON" ); // [in] char *pszComment
                  ++pYC->iParameterPollingSubState;
                  break;
              case 1 : // been waiting long enough for the unpredictable response for YAESU5B_CMD_TURN_ON (*)
                  pYC->iParameterPollingState = RIGCTRL_POLLSTATE_START_RD; // ex: RIGCTRL_POLLSTATE_DONE
                  // (*) If the FT-817 is already on when you ask it to turn on,
                  //     it doesn't respond with "hey, I'm already on" !
                  //     Instead, it responds with ...  NOTHING !
                  //     Be prepared for more fun like this.
                  pYC->fRadioTurnedON = TRUE; // <- ASSUMPTION, even if we didn't get a response ! (reasons in the sarcastic comment above)
                  break;
            } // end switch( pYC->iParameterPollingSubState ) after time-out in RIGCTRL_POLLSTATE_WAIT_TURN_ON
         } // end if < waited long enough .. >
        break; // end case RIGCTRL_POLLSTATE_WAIT_TURN_ON
     case RIGCTRL_POLLSTATE_START_RD : // begin to read all we need to know about/from the remote rig
        TIM_StartStopwatch(&pYC->sw_ParameterPolling);
        pYC->iParameterPollingState = RIGCTRL_POLLSTATE_READ_RADIO_ID;
        pYC->iParameterPollingSubState = 0;
        break; // end case RIGCTRL_POLLSTATE_START_RD
     case RIGCTRL_POLLSTATE_READ_RADIO_ID:  // (try to) read the radio's "default" ID, as a connection test and to check its type
        // That's a problem with this stonage Yaesu CAT protocol...
        // at least there was nothing in the (official) specification for this purpose !
        // Maybe, in a future version, Yaesu5Byte_Handler() will "try something"
        // to find out which model of Yaesu radio we've got out there,
        TIM_StartStopwatch(&pYC->sw_ParameterPolling);
        pYC->iParameterPollingState = RIGCTRL_POLLSTATE_PARAMS_FOR_RIGCTL;
        pYC->iParameterPollingSubState = 0;  // here: enumerator for RigCtrl_EnumerateRequiredPNs()
        break; // end case RIGCTRL_POLLSTATE_READ_RADIO_ID
     case RIGCTRL_POLLSTATE_PARAMS_FOR_RIGCTL: // try to read the parameters required for RigControl.c itself
        if( ! Y5B_StartReadingNextParamForRigcontrol( pYC ) ) // [in,out] pYC->iParameterPollingSubState++
         { // seems we're through with all 'essential' parameters for rig-control ->
           pYC->iParameterPollingState = RIGCTRL_POLLSTATE_DONE;
         }
        break; // end case RIGCTRL_POLLSTATE_PARAMS_FOR_RIGCTL
   //   case RIGCTRL_POLLSTATE_TX_BAND_EDGES:      // (try to) retrieve all of the radio's TRANSMIT BANDS (with edge frequencies)
   //   case RIGCTRL_POLLSTATE_BAND_STACKING_REGS: // (try to) read out the 'band stacking registers' (special VFO memories)
   //   case RIGCTRL_POLLSTATE_PARAMS_FOR_HAMLIB:  // (try to) read a few more params used by HamlibServer.c
   //   case RIGCTRL_POLLSTATE_CONFIGURE_SPECTRUM: // (try to) activate transmission of "spectrum scope waveform data"
   //        (the above states that are meaningless for this primitive radio
   //         are handled in the 'default' case further below,
   //         and end up immediately in RIGCTRL_POLLSTATE_DONE)
     case RIGCTRL_POLLSTATE_DONE: // finished polling "all we need to know" about the radio,
        // but since the stoneage Yaesu radios aren't capable of sending UNSOLICITED REPORTS,
        // we must keep polling the radio for at least the VFO FREQUENCY, the S-meter, and who-knows-what.
        break;
   //   case RIGCTRL_POLLSTATE_FIND_VFO_FOR_FREQ: // special state for e.g. IC-9700 to "find" the suitable VFO for a given band/frequency
     case RIGCTRL_POLLSTATE_TURN_RIG_OFF:  // send command to turn the radio off (as soon as possible, but not IMMEDIATELY)
        if( fTimeout ) // waited "long enough" after the last normal traffic ?
         { TIM_StartStopwatch(&pYC->sw_ParameterPolling);
           Yaesu5Byte_SendAndLogCommand( pYC,
               YAESU5B_CMD_TURN_OFF, // [in] 8-bit "Opcode" (terminology from the FT-817ND Operating Manual):
               0x00000000UL, // [in] 32-bit "Parameter", ignored for the above opcode
               RIGCTRL_MSGTYPE_FLAG_TX | RIGCTRL_MSGTYPE_POWER_ON_OFF, // [in] iMsgType, only for the traffic log
               "Turn Power OFF" ); // [in] char *pszComment
           pYC->iParameterPollingState = RIGCTRL_POLLSTATE_WAIT_TURN_OFF;

      // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      // Caution ... you may WRECK THE EXTERNAL LI-ION BATTERY that powers your FT-8x7 !
      //             "Power OFF" state is in fact "standby" ... from KA7OEI :
      //
      // > Command 8F - Turn Off FT-817:  Note that when this command is used,
      // > the CPU remains active in order to be able to detect the "On" command (0F)
      // > and a small amount of current is continuously drawn it should be used
      // > with caution when operating the FT-817 from via the rear power connector
      // > as noted several times below:
      // >  * That the FT-817 will respond to some commands after the OFF command is executed
      // >    and the '817 is powered from the rear power connector.
      // >  * Repeatedly sending the OFF command will cause the FT-817 to send
      // >    a string of unknown data - only if the '817 is powered from the
      // >    rear power connector.
      // >  * When this command is executed using internal batteries the '817
      // >    will shut completely down and will not respond to an ON command.
      // >    If, after an OFF command was sent with the '817 operating on internal batteries,
      // >    the configuration is changed such that power is supplied via the rear connector,
      // >    the '817 will begin to consume 10 milliamps when off (having been shut off
      // >    with the OFF command) and it will respond to the ON command.
      // >  * Note that even if the OFF command was used, the '817 will not consume
      // >    10 milliamps when off if running on internal batteries only.
      // >   (Remember, again, that the ON command won't work when on internal batteries.)
      // >  * When this command is executed using with power supplied at the
      // >    rear power connector the current consumption will be approximately
      // >    10 milliamps, depending on battery voltage.
      // Observations by DL4YHF:
      //   * An FT-817ND *always* drew 9.2 mA when fed externally,
      //     regardless of being TURNED OFF VIA COMMAND, or the power button.
      //   * Without sending 0xFF 0xFF 0xFF 0xFF 0xFF (another undocumented feature),
      //     an FT-817ND did NOT react on the "Power On" sequence !
      //   * WITH sending 0xFF 0xFF 0xFF 0xFF 0xFF but no pause afterwards
      //     caused the FT-817ND to draw ~~190 mA for a second, but then
      //     switched back to ~10 mA in "standby" .
      //     Engine Start Abort ? Turbine Stall ? APU Power Fail ?  :-)
      //   * Sometimes, there's a RESPONSE for a Power ON/OFF command,
      //     sometimes not. Possibly only if there was a CHANGE of state.
      //   * For that reason, Yaesu5Byte_ProcessData() also checks
      //     pYC->iPowerControlState, and if it's != Y5B_PCS_PASSIVE,
      //     may(!) cause a state transition - see OnRespPowerOnOff() .
      // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

         } // end if < waited long enough .. >
        break; // end case RIGCTRL_POLLSTATE_TURN_RIG_OFF
     case RIGCTRL_POLLSTATE_WAIT_TURN_OFF: // waiting for the rig to turn off
        if( fTimeout ) // waited "long enough" after the last normal traffic ?
         { // been waiting long enough for the unpredictable response for YAESU5B_CMD_TURN_OFF
           pYC->iParameterPollingState = RIGCTRL_POLLSTATE_TURNED_OFF;
           pYC->fRadioTurnedON = FALSE; // <- ASSUMPTION, don't always expect a response for Yaesu's "POWER OFF" command !
         } // end if < waited long enough .. >
        break; // end case RIGCTRL_POLLSTATE_WAIT_TURN_OFF
     case RIGCTRL_POLLSTATE_TURNED_OFF: // after RIGCTRL_POLLSTATE_TURN_RIG_OFF / WAIT_TURN_OFF, the only allowed command is to "turn on"
        break; // end case RIGCTRL_POLLSTATE_TURNED_OFF
     default: // anything else we don't support here ?
        pYC->iParameterPollingState = RIGCTRL_POLLSTATE_DONE;
        break;
   } // end switch( pYC->iParameterPollingState )


  return pYC->iParameterPollingState;

} // end Yaesu5Byte_Handler()

//--------------------------------------------------------------------------
int Yaesu5Byte_SwitchPollingState( T_Yaesu5ByteControl *pYC, int iNewPollingState )
      // Called from same task / thread / interrupt handler
      //        that also calls Yaesu5Byte_ProcessData(),
      // whenever module RigControl.c wants to "begin something", e.g.:
      //  * RIGCTRL_POLLSTATE_READ_RADIO_ID : Try to identify the radio model,
      //          which in Icom's CI-V protocol was performed by reading
      //          the radio's model-depending "default address" aka numeric ID.
      //          See RigControl.c : RigCtrl_AskForRadioID().
      //  * etc ..  see implementation in the *.c module .
      // Also here, the RETURN VALUE is one of the RIGCTRL_POLLSTATE_.. valued defined in RigControl.h,
      // in most (but not all) cases the same as iNewPollingState .
{
  if( iNewPollingState != pYC->iParameterPollingState )
   { // Regardless of the new state: Restart the TIMER for parameter-polling..
     TIM_StartStopwatch( &pYC->sw_ParameterPolling ); // <- checked in Yaesu5Byte_Handler()

     // Any special actions on the transition of the parameter polling state machine ?
     switch( iNewPollingState )
      { case RIGCTRL_POLLSTATE_START_RD :  // RigControl.c asks us to "read all we need"..
           pYC->iParameterPollingSubState = 0;  // start a "sub-state-machine"; the rest happens in Yaesu5Byte_Handler()
           if( ! pYC->fRadioTurnedON )
            {  iNewPollingState = RIGCTRL_POLLSTATE_TURN_RIG_ON; // must "turn on" before "starting to read" !
            }
           else
            { // leave iNewPollingState as-is (RIGCTRL_POLLSTATE_START_RD)
            }
           break; // end case RIGCTRL_POLLSTATE_START_RD
        default: // "no special action" required for the state transition - Yaesu5Byte_Handler() will do the rest
           break;
      }
     pYC->iParameterPollingState = iNewPollingState;
   }
  return pYC->iParameterPollingState;

} // end Yaesu5Byte_SwitchPollingState()


//---------------------------------------------------------------------------
static BOOL Y5B_StartReadingNextParamForRigcontrol( T_Yaesu5ByteControl *pYC )
  // Called from Yaesu5Byte_Handler() in RIGCTRL_POLLSTATE_PARAMS_FOR_RIGCTL,
  // repeatedly until this function returns TRUE, which means "done, finished".
  // [in,out] pYC->iParameterPollingSubState
{ BOOL fDone = TRUE;
  int  iUnifiedPN;
  T_RigCtrl_ParamInfo *pPI;
  T_RigCtrlInstance *pRC = (T_RigCtrlInstance*)pYC->pvRigControl; // <- this may be NULL !

  if( pRC != NULL )  // this only works with an associated RIG CONTROL INSTANCE
   {                 // (because where else should we store the VALUE later)..
     while( (iUnifiedPN=RigCtrl_EnumerateRequiredPNs( pYC->iParameterPollingSubState++)) > 0 )
      { pPI = RigCtrl_GetInfoForUnifiedParameterNumber( iUnifiedPN );
        if( pPI != NULL )
         { if( !RigCtrl_HaveValidParamValue( pRC, pPI ) ) // skip parameters for which we ALREADY HAVE a "valid value" from the radio
            { if( Yaesu5Byte_AskForParameter( pYC, iUnifiedPN ) )  // .. by sending the appropriate READ-command ..
               { fDone = FALSE;
                 break;
               }
              else // obviously a parameter that doesn't exist in stoneage Yaesu radios..
               { // .. continue in the EnumerateRequiredPNs loop
               }
            } // end if < RigControl.c does NOT have a "valid VALUE" for this parameter yet >
         }   // end if( pPI != NULL )
      }     // end while < more "required parameter numbers" to query >
   }       // end if < got a valid T_RigCtrlInstance pointer > ?
  return fDone;
} // end Y5B_StartReadingNextParamForRigcontrol()


//---------------------------------------------------------------------------
BOOL Yaesu5Byte_TurnPowerOnOrOff( T_Yaesu5ByteControl *pYC, // API ("public")
        BOOL fPowerOn )  // [in] TRUE=turn on, FALSE=turn off
{
  TIM_StartStopwatch(&pYC->sw_ParameterPolling);
  pYC->iParameterPollingState = fPowerOn ? RIGCTRL_POLLSTATE_TURN_RIG_ON : RIGCTRL_POLLSTATE_TURN_RIG_OFF;
  return TRUE;  // the rest happens "in the background" - see details in Yaesu5Byte_Handler() !
} // end Yaesu5Byte_TurnPowerOnOrOff()

//---------------------------------------------------------------------------
BOOL Yaesu5Byte_SetVFOFrequency( T_Yaesu5ByteControl *pYC,  // API ("public")
        double dblFreqHz )  // [in] freq in HERTZ
  // Non-blocking ! The return value only indicates ...
  //   TRUE = "We're optimistic and guess the radio will understand"
  //   FALSE= "Something wrong. We're not going to send anything"
{
  return Yaesu5Byte_SendAndLogCommand( pYC,
             YAESU5B_CMD_SET_FREQ,        // [in] 8-bit "Opcode"
             Y5B_Uint32ToBCD8( (DWORD)(dblFreqHz*0.1) ), // [in] 32-bit "Parameter", here: 8-digit BINARY CODED DECIMAL !
             RIGCTRL_MSGTYPE_FLAG_TX | RIGCTRL_MSGTYPE_POWER_ON_OFF, // [in] iMsgType, only for the traffic log
             "Set VFO Freq" ); // [in] char *pszComment

} // end Yaesu5Byte_SetVFOFrequency()

//---------------------------------------------------------------------------
BOOL Yaesu5Byte_SetOpMode( T_Yaesu5ByteControl *pYC,        // API ("public")
        int iRigControlOpMode ) // [in] "op-mode" a la RigControl.h (!)
  // Non-blocking ! See Yaesu5Byte_SetVFOFrequency() about the return value.
  //
{
  return FALSE;
} // end Yaesu5Byte_SetOpMode()

//---------------------------------------------------------------------------
BOOL Yaesu5Byte_AskForParameter( T_Yaesu5ByteControl *pYC,  // API ("public")
        int iUnifiedPN ) // [in] "unified parmater number" a la RigControl.c,
                         //       e.g. RIGCTRL_PN_FREQUENCY, etc.
                         // [out] pYC->iExpectingResponseForCmd, etc.
  // Sends a read-request, without waiting for a response, as the name should imply.
  //     Like any other response from the radio, the response will be analysed
  //     later, in Yaesu5Byte_ProcessData() -> pYC->pOnDecodeCallback() .
  // Returns TRUE if a READ REQUEST for the specified parameter could be sent,
  //         FALSE otherwise (usual reason: iUnifiedPN has no equivalent
  //                                        in the currently connected rig).
  // Note: Unlike RigCtrl_QueueUpCmdToReadUnifiedPN( T_RigCtrlInstance *pRC, int iUnifiedPN ),
  //              Yaesu5Byte_AskForParameter() will send the request IMMEDIATELY.
  //              The caller should check for pYC->fBusy before the call, which is
  //              what RigCtrl_QueueUpCmdToReadUnifiedPN() does internally.
{
  BYTE b16TxBuffer[16];
  BYTE *pbBufPtr = b16TxBuffer;
  int  i, iMsgLength;
  int  iY5B_ReadCmd;
  char sz80Comment[84];
  char *pszDest=sz80Comment, *pszEndstop=sz80Comment+80;
  T_RigCtrl_ParamInfo *pPI = RigCtrl_GetInfoForUnifiedParameterNumber( iUnifiedPN );
  DWORD dwParam = 0; // 4-byte PARAMETER, sent *before* the 1-byte OPCODE
  int iMsgType  = RIGCTRL_MSGTYPE_FLAG_TX | RIGCTRL_MSGTYPE_OTHER;

  if( pYC->fBusy && (TIM_ReadStopwatch_ms(&pYC->sw_BusyTimer) < Y5B_BUSY_TIMER_TIMEOUT_MS) )
   { return FALSE;
   }

  if( (pPI=RigCtrl_GetInfoForUnifiedParameterNumber(iUnifiedPN)) != NULL )
   { if( pPI->pszToken != NULL )
      { SL_AppendPrintf( &pszDest, pszEndstop, "read %s", pPI->pszToken );
      }
     else
      { SL_AppendPrintf( &pszDest, pszEndstop, "read #%d", (int)pPI->iUnifiedPN );
      }
     // Having a valid T_RigCtrl_ParamInfo from module RigControl.c doesn't mean
     // we can READ this parameter from a stoneage YAESU radio.
     // Comparing a modern ICOM radio (which is what RigControl.c was written for)
     // with a dinosaur like FT-817, FT-857, or FT-897 would be unfair.
     // Keep it simple, and use a simple switch/case list for a few COMMANDS
     // that exist in an FT-817 (without the undocumented 'READ EEPROM', the number
     // of parameters we can READ via Yaesu-5-Byte-CAT is disappointingly low):
     switch( iUnifiedPN )  // how can we READ this parameter from an FT-8x7 ?
      { case RIGCTRL_PN_FREQUENCY : // 'VFO' frequency in Hertz: will later be stored in T_RigCtrlInstance.dblVfoFrequency
        case RIGCTRL_PN_OP_MODE   : // 'operation mode' (USB/LSB/CW..): will later be stored in T_RigCtrlInstance.iOpMode
           iY5B_ReadCmd = YAESU5B_CMD_READ_FREQ_AND_MODE;
           pYC->iExpectedResponseLength = 5; // <- import for Yaesu5Byte_ProcessData() : expecting 4 bytes for the FREQUENCY + 1 byte for the MODE now
           break;
        case RIGCTRL_PN_FILTER_BANDWIDTH: // (IF-) filter bandwidth in Hz : No such parameter (only, maybe, an EEPROM location)
        case RIGCTRL_PN_DATA_MODE : // 0=no data mode, 1=data mode.   Equivalent of CI-V cmd 0x1A, subcommand 0x06.
        case RIGCTRL_PN_TRANSMITTING: // 0=receiving, 1=transmitting. Equivalent of CI-V cmd 0x1C, sub 0x00, no data = "Read the transceiver's status".
        default:  // a parameter with this 'Unified Number' cannot be read !
           iY5B_ReadCmd = -1;
           break;
      } // end switch( iUnifiedPN )
     if( iY5B_ReadCmd >= 0 ) // bingo, got a YAESU-5-BYTE-COMMAND to read that parameter...
      { pYC->iExpectingResponseForCmd = iY5B_ReadCmd; // <- import for Yaesu5Byte_ProcessData(), to parse the response
        return Yaesu5Byte_SendAndLogCommand( pYC,
          (BYTE)iY5B_ReadCmd/*Opcode*/, dwParam, iMsgType, sz80Comment );
      } // end if( iY5B_ReadCmd >= 0 )

   } // end if( pPI != NULL )

  (void)pbBufPtr; // ... assigned a value that is never used ..

  return FALSE;
} // end Yaesu5Byte_AskForParameter()


// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
// Parsers for received COMMANDS and received RESPONSES.
//                        | ,--------------------'
//                        | '-> called via Y5B_CommandTable[].pRespParser()
//                        '---> called via Y5B_CommandTable[].pCmdParser()
//  We need BOTH, to 'eavesdrop' on Yeasu-5-Byte-CAT-traffic through 'serial port tunnels'.
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

//--------------------------------------------------------------------------
static int OnCmdLockOn( struct t_Yaesu5ByteControl *pYC )
{
  // > This command is equivalent to turning the dial/panel lock on.
  // > This precise effect of this setting depends on the state of
  // > submenu item #32 (stored at memory location 5E HEX - see below.)
  // > This command returns 00 if it was unlocked, and F0 if already locked.
  strcpy( pYC->sz80Comment, "Lock ON");
  pYC->iExpectedResponseLength  = 1;
  pYC->iExpectingResponseForCmd = YAESU5B_CMD_LOCK_ON; // call OnRespLockOn() when the response arrives !
  return 0;
} // end OnCmdLockOn()

//--------------------------------------------------------------------------
static int OnRespLockOn( struct t_Yaesu5ByteControl *pYC )
{
  return 0;
}

//--------------------------------------------------------------------------
static int OnCmdSetFreq( struct t_Yaesu5ByteControl *pYC ) // parse cmd YAESU5B_CMD_SET_FREQ ..
{
  // > This commands sets the current frequency (see Yaesu5Byte_DecodeMessage).
  // > The current frequency is set using 4 BCD bytes.  To set a frequency
  // > such as 435.12345 MHz, data bytes 1-4 would be [43][51][23][45] followed by
  // > the the set frequency command [01]
  // ( do we expect a RESPONSE for this command ? )
  strcpy( pYC->sz80Comment, "Set Frequency");
  pYC->iExpectedResponseLength  = 1;
  pYC->iExpectingResponseForCmd = YAESU5B_CMD_SET_FREQ; // call OnRespSetFreq() when the response arrives !
  return 0;
} // end OnCmdSetFreq()

//--------------------------------------------------------------------------
static int OnRespSetFreq( struct t_Yaesu5ByteControl *pYC )
{
   return 0;
}

//--------------------------------------------------------------------------
static int OnCmdSplitOn( struct t_Yaesu5ByteControl *pYC ) // parse cmd YAESU5B_CMD_SPLIT_ON
{
  // > This command enables split operation.  Use the Read Transmitter Status
  // > command (below) to determine current split status.
  // > This command returns 00 if not already on, and F0 (HEX) if already on.
  strcpy( pYC->sz80Comment, "Split ON" );
  pYC->iExpectedResponseLength  = 5;
  pYC->iExpectingResponseForCmd = YAESU5B_CMD_SPLIT_ON; // call OnRespSplitOn() when the response arrives !
  return 0;
}

//--------------------------------------------------------------------------
static int OnRespSplitOn( struct t_Yaesu5ByteControl *pYC ) // got the response for YAESU5B_CMD_SPLIT_ON :
{
  return 0;
}

//--------------------------------------------------------------------------
static int OnCmdReadFreq( struct t_Yaesu5ByteControl *pYC ) // parse cmd YAESU5B_CMD_READ_FREQ_AND_MODE ..
{
  // > This command returns five bytes .. ( see OnRespReadFreq() )
  strcpy( pYC->sz80Comment, "Read Frequency and Mode" );
  pYC->iExpectedResponseLength  = 5;
  pYC->iExpectingResponseForCmd = YAESU5B_CMD_READ_FREQ_AND_MODE; // call OnRespReadFreq() when the response arrives !
  return 0;
}

//--------------------------------------------------------------------------
static int OnRespReadFreq( struct t_Yaesu5ByteControl *pYC ) // parse response after YAESU5B_CMD_READ_FREQ_AND_MODE
{
  // > Command 03 - Read frequency and mode:
  // > This command returns 5 bytes with the first 4 bytes containing
  // > the current frequency in the same format as the Set frequency command
  // > while the last byte contains the current mode as follows:
  // > 00 - LSB, 01 - USB, 02 - CW, 03 - CWR, 04 - AM, 06 - WFM, 08 - FM, 0A - DIG, 0C - PKT
  // Addition by DL4YHF :  An FT-817ND sent MODE = 0x82 in CW-NARROW
  //                                           and 0x83 in CW-REVERSE-NARROW.

  T_RigCtrlInstance *pRC = (T_RigCtrlInstance*)pYC->pvRigControl; // <- this may be NULL !
  char *pszDest = pYC->sz80Comment;
  const char *pszEnd = pYC->sz80Comment + 80;
  const char *pszMode= NULL;
  DWORD dwValue = Y5B_ParseBigEndianBCD( pYC->bRxBuffer+0,  8/*digits*/ );
        // '--> here, in TEN-HERTZ-UNITS. With a maximum of say 440 MHz,
        // the result easily fits inside 31(!) bits - so even a signed 'long' would be ok.
  double dblVfoFrequency = (double)dwValue * 10.0;
  int    iRigControlOpMode = -1;
  if( pRC != NULL )
   { iRigControlOpMode = pRC->iOpMode;
   }
  pYC->iCurrentMsgType &= ~RIGCTRL_MSGTYPE_OTHER;            // it's not "some other message", but ..
  pYC->iCurrentMsgType |=  RIGCTRL_MSGTYPE_FREQUENCY_REPORT; // it's a FREQUENCY REPORT !
     // (this allows filtering for certain types of messages on the DEBUG tab.
     //  Leave RIGCTRL_MSGTYPE_FLAG_TX / RIGCTRL_MSGTYPE_FLAG_RX unchanged here
     //   - those flags have already been set in Yaesu5Byte_ProcessData(),
     //     based on iRigCtrlOrign, e.g. RIGCTRL_ORIGIN_COM_PORT_RX !


  SL_AppendPrintf( &pszDest, pszEnd, "%.5lf MHz", (double)(dblVfoFrequency * 1e-6) );
  Y5B_OpModeToRigControl( pYC->bRxBuffer[4], &iRigControlOpMode, &pszMode );
  if( pszMode != NULL )
   { SL_AppendPrintf( &pszDest, pszEnd, ", %s", pszMode );
   }
  if( pRC != NULL )
   { if( pRC->dblVfoFrequency != dblVfoFrequency )
      { pRC->dblVfoFrequency = dblVfoFrequency;
        ++pRC->iFrequencyModifiedByRadio_cnt;
        // Similar as in RigControl.c : RigCtrl_OnVfoReportFromNetwork(),
        // this counter is polled in a GUI-thread somewhere.
      }

     if( iRigControlOpMode > 0 )
      { if( pRC->iOpMode != iRigControlOpMode )
         {  pRC->iOpMode = iRigControlOpMode;
            // Set a 'signal' for the GUI ? Not yet...
         }
      }
   }

  return 0;
} // end OnRespReadFreq() [ .. "and mode", but not all we want to know, e.g. "CW Narrow" or "CW Wide" ]

//--------------------------------------------------------------------------
static int OnCmdRitOn( struct t_Yaesu5ByteControl *pYC )
{
  // > This command turns the clarifier (a.k.a. RIT) on.
  // > This command returns 00 if not already on, and F0 (HEX) if already on.
  strcpy( pYC->sz80Comment, "RIT ON" );
  pYC->iExpectedResponseLength  = 1;
  pYC->iExpectingResponseForCmd = YAESU5B_CMD_RIT_ON; // call OnRespRitOn() when the response arrives !
  return 0;
}

//--------------------------------------------------------------------------
static int OnRespRitOn( struct t_Yaesu5ByteControl *pYC )
{
   return 0;
}

//--------------------------------------------------------------------------
static int OnCmdSetOpMode( struct t_Yaesu5ByteControl *pYC )
{
  // > The mode byte is described below (YAESU5B_OPMODE_xxx)
  // > Command 07 - Set operation mode:
  // > 00 - LSB, 01 - USB, 02 - CW, 03 - CWR, 04 - AM, 08 - FM, 0A - DIG, 0C - PKT.
  // > Note: Setting other than one of the listed modes can "crash" the FT-817,
  // > necessitating removal of power.  Note that the WFM mode (08) cannot be set
  // > using this command, rather that the mode is implicit when selecting
  // > the frequency in the range correlating with the FM broadcast band (i.e. 76-108 MHz.)
  // ( Do we expect a RESPONSE for this command ? How else to find out if the CPU has crashed ? :~)
  char *pszDest = pYC->sz80Comment;
  const char *pszEnd = pYC->sz80Comment + 80;
  const char *pszMode= NULL;
  int  iRigControlOpMode;
  T_RigCtrlInstance *pRC = (T_RigCtrlInstance*)pYC->pvRigControl; // <- this may be NULL !

  Y5B_OpModeToRigControl( pYC->bRxBuffer[0], &iRigControlOpMode, &pszMode );
  SL_AppendPrintf( &pszDest, pszEnd, "Set OpMode: 0x%02X", (unsigned int)pYC->bRxBuffer[0] );
  if( pszMode != NULL )
   { SL_AppendPrintf( &pszDest, pszEnd, " = %s", pszMode );
   }
  if( pRC != NULL )
   { if( iRigControlOpMode > 0 )
      { if( pRC->iOpMode != iRigControlOpMode )
         {  pRC->iOpMode = iRigControlOpMode;
            // Set a 'signal' for the GUI ? Not yet...
         }
      }
   }
  pYC->iExpectedResponseLength  = 1;
  pYC->iExpectingResponseForCmd = YAESU5B_CMD_SET_OP_MODE; // call OnRespSetOpMode() when the response arrives !
  return 0;
}

//--------------------------------------------------------------------------
static int OnRespSetOpMode( struct t_Yaesu5ByteControl *pYC )
{
  return 0;
}

//--------------------------------------------------------------------------
static int OnCmdSetPttOn( struct t_Yaesu5ByteControl *pYC )
{
  // > This "keys" the FT-817.  In CW, this sets the radio to transmit mode,
  // > but does (not?) key the transmitter.
  // > Note that keying and unkeying the radio's PTT line will cancel
  // > the transmit mode (i.e. put it back into receive)
  // > and effectively override this command.
  // > This command returns 00 if the '817 was unkeyed, and F0 if already keyed.
  strcpy( pYC->sz80Comment, "PTT ON" );
  pYC->iExpectedResponseLength  = 1;
  pYC->iExpectingResponseForCmd = YAESU5B_CMD_SET_PTT_ON; // call OnRespSetPttOn() when the response arrives !
  return 0;
}

//--------------------------------------------------------------------------
static int OnRespSetPttOn( struct t_Yaesu5ByteControl *pYC )
{
  return 0;
}

//--------------------------------------------------------------------------
static int OnCmdSetRptrDir( struct t_Yaesu5ByteControl *pYC ) // cmd YAESU5B_CMD_SET_REPEATER_DIR
{
  // > The byte "Offset" has the following function:
  // > 09 = - (minus) shift    49 = + (positive) shift    89 = simplex
  // ( Do we expect a RESPONSE for this command ? )
  strcpy( pYC->sz80Comment, "Set Rptr offset-DIRECTION" );
  pYC->iExpectedResponseLength  = 1;
  pYC->iExpectingResponseForCmd = YAESU5B_CMD_SET_REPEATER_DIR; // call OnRespSetPttOn() when the response arrives !

  return 0;
}

//--------------------------------------------------------------------------
static int OnRespSetRptrDir( struct t_Yaesu5ByteControl *pYC ) // resp YAESU5B_CMD_SET_REPEATER_DIR
{
   return 0;
}

//--------------------------------------------------------------------------
static int OnCmdSetCTCSSMode( struct t_Yaesu5ByteControl *pYC )
{
  // > The byte "Mode" has the following function:
  // > 0A = DCS Enable  2A = CTCSS Enable 4A = DCS/CTCSS Encoder  enable
  // > 8A = DCS/CTCSS Encoder Disable
  strcpy( pYC->sz80Comment, "Set CTCSS mode" );
  return 0;
}

//--------------------------------------------------------------------------
static int OnRespSetCTCSSMode( struct t_Yaesu5ByteControl *pYC )
{
  return 0;
}

//--------------------------------------------------------------------------
static int OnCmdSetCTCSSFreq( struct t_Yaesu5ByteControl *pYC )
{
  strcpy( pYC->sz80Comment, "Set CTCSS freq" );
  return 0;
}

//--------------------------------------------------------------------------
static int OnRespSetCTCSSFreq( struct t_Yaesu5ByteControl *pYC )
{
  return 0;
}

//--------------------------------------------------------------------------
static int OnCmdSetDCSCode( struct t_Yaesu5ByteControl *pYC )
{
  strcpy( pYC->sz80Comment, "Set DCS code" );
  return 0;
}

//--------------------------------------------------------------------------
static int OnRespSetDCSCode( struct t_Yaesu5ByteControl *pYC )
{
  return 0;
}

//--------------------------------------------------------------------------
static int OnCmdTurnOn( struct t_Yaesu5ByteControl *pYC )
{
  strcpy( pYC->sz80Comment, "Turn radio ON" );
  return 0;
}

//--------------------------------------------------------------------------
static int OnRespPowerOnOff( struct t_Yaesu5ByteControl *pYC )
{
  // YAESU5B_CMD_TURN_ON/OFF sometimes causes a response, sometimes not.
  //  IF a response arrives when we're waiting for a few hundred milliseconds
  //  after sending such a command, show what happened and stop waiting:
  switch( pYC->iParameterPollingState )
   { default :  strcpy( pYC->sz80Comment, "Unexpected Power on/off response" );
          // no change of pYC->iParameterPollingState; wait for the time-out in Yaesu5Byte_Handler()
          break;
     case RIGCTRL_POLLSTATE_WAIT_TURN_ON  : // "waiting after sending YAESU5B_CMD_TURN_ON"
          strcpy( pYC->sz80Comment, "Power-ON response" );
          pYC->fRadioTurnedON = TRUE;
          pYC->iParameterPollingState = RIGCTRL_POLLSTATE_START_RD; // ex: RIGCTRL_POLLSTATE_DONE
          TIM_StartStopwatch(&pYC->sw_ParameterPolling);
          break;
     case RIGCTRL_POLLSTATE_WAIT_TURN_OFF : // "waiting after sending YAESU5B_CMD_TURN_OFF"
          strcpy( pYC->sz80Comment, "Power-OFF response" );
          pYC->fRadioTurnedON = FALSE;
          pYC->iParameterPollingState = RIGCTRL_POLLSTATE_TURNED_OFF; // FT-817 can be woken up via command now, at the expense of drawing ~10 mA ..
          break;
   }
  return 0;
} // end OnRespPowerOnOff()

//--------------------------------------------------------------------------
static int OnCmdReadTxState( struct t_Yaesu5ByteControl *pYC )
{
  strcpy( pYC->sz80Comment, "Read TX State" );
  return 0;
}

//--------------------------------------------------------------------------
static int OnRespReadTxState( struct t_Yaesu5ByteControl *pYC )
{
  return 0;
}

//--------------------------------------------------------------------------
static int OnCmdLockOff( struct t_Yaesu5ByteControl *pYC )
{
  strcpy( pYC->sz80Comment, "Lock OFF" );
  return 0;
}

//--------------------------------------------------------------------------
static int OnRespLockOff( struct t_Yaesu5ByteControl *pYC )
{
   return 0;
}

//--------------------------------------------------------------------------
static int OnCmdToggleVFOs( struct t_Yaesu5ByteControl *pYC )
{
  strcpy( pYC->sz80Comment, "Toggle VFO A/B" );
  return 0;
}

//--------------------------------------------------------------------------
static int OnRespToggleVFOs( struct t_Yaesu5ByteControl *pYC )
{
   return 0;
}

//--------------------------------------------------------------------------
static int OnCmdSplitOff( struct t_Yaesu5ByteControl *pYC )
{
  strcpy( pYC->sz80Comment, "Split mode OFF" );
  return 0;
}

//--------------------------------------------------------------------------
static int OnRespSplitOff( struct t_Yaesu5ByteControl *pYC )
{
   return 0;
}

//--------------------------------------------------------------------------
static int OnCmdRitOff( struct t_Yaesu5ByteControl *pYC )
{
  strcpy( pYC->sz80Comment, "RIT OFF" );
  return 0;
}

//--------------------------------------------------------------------------
static int OnRespRitOff( struct t_Yaesu5ByteControl *pYC )
{
   return 0;
}

//--------------------------------------------------------------------------
static int OnCmdPttOff( struct t_Yaesu5ByteControl *pYC )
{
  strcpy( pYC->sz80Comment, "PTT OFF" );
  return 0;
}

//--------------------------------------------------------------------------
static int OnRespPttOff( struct t_Yaesu5ByteControl *pYC )
{
   return 0;
}

//--------------------------------------------------------------------------
static int OnCmdTurnOff( struct t_Yaesu5ByteControl *pYC )
{
  strcpy( pYC->sz80Comment, "Turn radio OFF" );
  return 0;
}

//--------------------------------------------------------------------------
static int OnRespTurnOff( struct t_Yaesu5ByteControl *pYC )
{
   return 0;
}

//--------------------------------------------------------------------------
static int OnCmdRadioConfig( struct t_Yaesu5ByteControl *pYC )
{
  strcpy( pYC->sz80Comment, "Radio Config" );
  return 0;
}

//--------------------------------------------------------------------------
static int OnRespRadioConfig( struct t_Yaesu5ByteControl *pYC )
{
   return 0;
}

//--------------------------------------------------------------------------
static int OnCmdUnknownStatus( struct t_Yaesu5ByteControl *pYC )
{
  strcpy( pYC->sz80Comment, "Unknown 'Status' command" );
  return 0;
}

//--------------------------------------------------------------------------
static int OnRespUnknownStatus( struct t_Yaesu5ByteControl *pYC )
{
   return 0;
}

//--------------------------------------------------------------------------
static int OnCmdReadEEPROM( struct t_Yaesu5ByteControl *pYC ) // cmd YAESU5B_CMD_READ_EEPROM .
  // Should be called with pYC->nBytesInRxBuffer == 5 [bytes], not more, not less,
  //        because anything else means we've already missed the RESPONSE !
{
  char *pszDest = pYC->sz80Comment;
  const char *pszEnd = pYC->sz80Comment + 80;
  const char *pszDetails;
  int iValue = (int)Y5B_ParseBigEndian16Bit( pYC->bRxBuffer+0 );

  // > This (undocumented + FT812 only) command causes two bytes
  // > of EEPROM data to be returned, beginning with the address in data bytes 1 and 2.
  // > (Approximately 6.25k of EEPROM data may be accessed - See below)
  // > The FT-817 uses an 64 kbit (8 Kbyte) EEPROM to store internal configuration
  // > and memories and most bytes in this EEPROM may be read with this command,
  // > 2 bytes at a time.  The address to be read is a 16 bit address with the
  // > upper 8 bits in the first byte and the lower 8 bits in the second byte.
  // > The first of the two bytes to be returned contains the data at the address
  // > in bytes 1 and 2 while the second of the two bytes returns the data at the
  // > next address up from that specified.
  // > Addresses from 0 to 1925 (HEX) (just over 6.25k bytes) are valid
  // > and attempts to retrieve addresses out of this range
  // > appear to return one byte - a zero.
  //           [WB: Which confirms that even for WELL-KNOWN commands like this,
  //                the number of bytes to expect IN THE RESPONSE is unknown.
  //                This, in turn, confirms that there can NEVER be more than
  //                ONE command, and more than ONE response "pending",
  //                and unsolicited messages from the radio are forbidden.]
  // Anyway, we expect the EEPROM address to be VALID, thus a TWO-BYTE response:
  pYC->iExpectedResponseLength  = 2;
  pYC->iExpectingResponseForCmd = YAESU5B_CMD_READ_EEPROM; // call OnRespReadEEPROM() when the response arrives !

  SL_AppendString( &pszDest, pszEnd, "Read EEPROM: " );
  if( (pszDetails=SL_GetStringFromTokenList( FT817_EEPROM_Addresses, iValue)) != NULL )
   { SL_AppendPrintf( &pszDest, pszEnd, "%s", pszDetails );
   }
  else // if we have no human-friendly "details", show the requested EEPROM address instead:
   { SL_AppendPrintf( &pszDest, pszEnd, "addr=0x%04X", (unsigned int)iValue );
   }
  return 0;  // "nothing else to do" (for the caller)
} // end OnCmdReadEEPROM()

//--------------------------------------------------------------------------
static int OnRespReadEEPROM( struct t_Yaesu5ByteControl *pYC )  // resp YAESU5B_CMD_READ_EEPROM
  // Should be called with pYC->nBytesInRxBuffer == 2 [bytes], not more, not less;
  //  because anything else means we've already missed a part of the RESPONSE !

{
  char *pszDest = pYC->sz80Comment;
  const char *pszEnd = pYC->sz80Comment + 80;
  char *pszDetails;

  SL_AppendPrintf( &pszDest, pszEnd, "EEPROM data: 0x%02X 0x%02X",
     (unsigned int)pYC->bRxBuffer[0], (unsigned int)pYC->bRxBuffer[1] );
  // Result on the RCW Keyer's "DEBUG"-tab, when eavesdropping "FT817 Commander" traffic:
  //  > 20:58:33.7 COM8: TX[05] 0x00 79 00 00 BB                ; Read EEPROM: TX Power, etc
  //  > 20:58:33.7 COM8: RX[02] 0x08 80                         ; EEPROM data : 0x08 0x80
  //  > 20:58:33.7 COM8: TX[05] 0x03 55 00 00 BB                ; Read EEPROM: addr=0x0355
  //  > 20:58:33.8 COM8: RX[02] 0x45 05                         ; EEPROM data : 0x45 0x05
  //  > 20:58:33.8 COM8: TX[05] 0x03 5F 00 00 BB                ; Read EEPROM: addr=0x035F
  //  > 20:58:33.8 COM8: RX[02] 0x02 9E                         ; EEPROM data : 0x02 0x9E

  return 0;  // "nothing else to do" (for the caller)
} // end OnRespReadEEPROM()

//--------------------------------------------------------------------------
static int OnCmdWriteEEPROM( struct t_Yaesu5ByteControl *pYC )
{
  strcpy( pYC->sz80Comment, "Write EEPROM" );
  return 0;
} // end OnCmdWriteEEPROM(

//--------------------------------------------------------------------------
static int OnRespWriteEEPROM( struct t_Yaesu5ByteControl *pYC )
{
  return 0;
} // end OnRespWriteEEPROM()

//--------------------------------------------------------------------------
static int OnCmdTxMeters( struct t_Yaesu5ByteControl *pYC )
{
  strcpy( pYC->sz80Comment, "Send Tx Meters" );
  return 0;
} // end OnCmdTxMeters()

//--------------------------------------------------------------------------
static int OnRespTxMeters( struct t_Yaesu5ByteControl *pYC )
{
  return 0;
} // end OnRespTxMeters()

//--------------------------------------------------------------------------
static int OnCmdGetRxStatus( struct t_Yaesu5ByteControl *pYC )
{
  strcpy( pYC->sz80Comment, "Get Rx Status" );
  return 0;
} // end OnCmdGetRxStatus()

//--------------------------------------------------------------------------
static int OnRespGetRxStatus( struct t_Yaesu5ByteControl *pYC )
{
  return 0;
} // end OnRespGetRxStatus()

//--------------------------------------------------------------------------
static int OnCmdSetRitOffset( struct t_Yaesu5ByteControl *pYC )
{
  strcpy( pYC->sz80Comment, "Set RIT Offset" );
  return 0;
} // end OnCmdSetRitOffset()

//--------------------------------------------------------------------------
static int OnRespSetRitOffset( struct t_Yaesu5ByteControl *pYC )
{
  return 0;
} // end OnRespSetRitOffset()

//--------------------------------------------------------------------------
static int OnCmdGetTxStatus( struct t_Yaesu5ByteControl *pYC )
{
  strcpy( pYC->sz80Comment, "Get Tx Status" );
  return 0;
} // end OnCmdGetTxStatus()

//--------------------------------------------------------------------------
static int OnRespGetTxStatus( struct t_Yaesu5ByteControl *pYC )
{
  return 0;
} // end OnRespGetTxStatus()

//--------------------------------------------------------------------------
static int OnCmdSetRptrOffset( struct t_Yaesu5ByteControl *pYC )
{
  strcpy( pYC->sz80Comment, "Set Repeater Offset" );
  return 0;
} // end OnCmdSetRptrOffset()

const T_Y5B_CommandTabEntry Y5B_CommandTable[] =
{  //   bCmd                           pCmdParser         pRespParser
  { YAESU5B_CMD_LOCK_ON,               OnCmdLockOn,       OnRespLockOn      },
  { YAESU5B_CMD_SET_FREQ,              OnCmdSetFreq,      OnRespSetFreq     },
  { YAESU5B_CMD_SPLIT_ON,              OnCmdSplitOn,      OnRespSplitOn     },
  { YAESU5B_CMD_READ_FREQ_AND_MODE,    OnCmdReadFreq,     OnRespReadFreq    },
  { YAESU5B_CMD_RIT_ON,                OnCmdRitOn,        OnRespRitOn       }, // ancient name: "Clarifier" (from the days of synthesizer with 100 Hz stepwidth)
  { YAESU5B_CMD_SET_OP_MODE,           OnCmdSetOpMode,    OnRespSetOpMode   },
  { YAESU5B_CMD_SET_PTT_ON,            OnCmdSetPttOn,     OnRespSetPttOn    },
  { YAESU5B_CMD_SET_REPEATER_DIR,      OnCmdSetRptrDir,   OnRespSetRptrDir  },
  { YAESU5B_CMD_SET_DCS_CTCSS_MODE,    OnCmdSetCTCSSMode, OnRespSetCTCSSMode},
  { YAESU5B_CMD_SET_CTCSS_TONE_FREQ,   OnCmdSetCTCSSFreq, OnRespSetCTCSSFreq},
  { YAESU5B_CMD_SET_DCS_CODE,          OnCmdSetDCSCode,   OnRespSetDCSCode  },
  { YAESU5B_CMD_TURN_ON,               OnCmdTurnOn,       OnRespPowerOnOff  },
  { YAESU5B_CMD_READ_TX_KEYED_STATE,   OnCmdReadTxState,  OnRespReadTxState },
  { YAESU5B_CMD_LOCK_OFF,              OnCmdLockOff,      OnRespLockOff     },
  { YAESU5B_CMD_TOGGLE_VFO_A_B,        OnCmdToggleVFOs,   OnRespToggleVFOs  },
  { YAESU5B_CMD_SPLIT_OFF,             OnCmdSplitOff,     OnRespSplitOff    },
  { YAESU5B_CMD_RIT_OFF,               OnCmdRitOff,       OnRespRitOff      },
  { YAESU5B_CMD_SET_PTT_OFF,           OnCmdPttOff,       OnRespPttOff      },
  { YAESU5B_CMD_TURN_OFF,              OnCmdTurnOff,      OnRespPowerOnOff  },
  { YAESU5B_CMD_RADIO_CONFIG,          OnCmdRadioConfig,  OnRespRadioConfig },
  { YAESU5B_CMD_UNKNOWN_STATUS,        OnCmdUnknownStatus,OnRespUnknownStatus},
  { YAESU5B_CMD_READ_EEPROM,           OnCmdReadEEPROM,   OnRespReadEEPROM  },
  { YAESU5B_CMD_WRITE_EEPROM,          OnCmdWriteEEPROM,  OnRespWriteEEPROM },
  { YAESU5B_CMD_TX_METERS,             OnCmdTxMeters,     OnRespTxMeters    },
  { YAESU5B_CMD_FACTORY_DEFAULT,       NULL,              NULL              },
  { YAESU5B_CMD_GET_RX_STATUS,         OnCmdGetRxStatus,  OnRespGetRxStatus },
  { YAESU5B_CMD_SET_RIT_OFFSET,        OnCmdSetRitOffset, OnRespSetRitOffset},
  { YAESU5B_CMD_GET_TX_STATUS,         OnCmdGetTxStatus,  OnRespGetTxStatus },
  { YAESU5B_CMD_SET_REPEATER_OFFSET,   OnCmdSetRptrOffset,NULL              },

  // "all zero" marks the end of the table:
  { 0,                                 NULL,               NULL        }
}; // end Y5B_CommandTable[]



/* EOF < C:\cbproj\Remote_CW_Keyer\Yaesu5Byte.c > */



