//---------------------------------------------------------------------------
// File:  C:\cbproj\Remote_CW_Keyer\VorbisStream.h
// Date:  2024-02-02
// Author: Wolfgang Buescher (DL4YHF)
// The ugly DETAILS are in the implementation (VorbisStream.c) !
//
// Short overview / most important functions for AUDIO STREAMING :
// ----------------------------------------------------------------------
//
// [A] Ogg/Vorbis ENCODER API ("compress once, send to multiple receivers")
//
//  (1) To ENCODE audio, initialise an instance of struct T_VorbisStream,
//         by calling VorbisStream_InitEncoder() with the required
//         a sampling rate and number of channels per sample point.
//  (2) Repeatedly (or periodically) call VorbisStream_EncodeSamples(),
//         with a reasonable number of samples (the maximum depends
//         on an internal buffer for "Ogg Pages", see definition of
//         VORBIS_STREAM_BUFFER_MAX_BYTES in VorbisStream.h) .
//         So don't try to feed in more than a few thousand audio samples
//         in a singe call.
//  (3) After calling VorbisStream_EncodeSamples(),
//         check for 'fInitialHeadersReady'. If that flag is set
//         FOR THE FIRST TIME, or when a NEW remote client has requested
//         the 'already running stream',
//          call VorbisStream_GetInitialHeadersFromEncoder() to retrieve
//               the FIRST FEW BYTES that must be delivered
//         (if the encoder output shall be safed as a *.ogg file,
//          write the bytes delivered by GetInitialHeadersForStreamOutput()
//          to the BEGIN of the output file, i.e. at byte-offset ZERO).
//
//  (4) After each call of VorbisStream_EncodeSamples(), send as many
//         "pages" as T_VorbisStream.BufferPages[ iBufferPagesHead ]
//         indicates, by calling VorbisStream_GetNextPagesFromEncoder().
//         Each STREAM SENDER (or FILE WRITER)
//           has its own instance of a TAIL INDEX (iBufferPagesTail),
//           compared against T_VorbisStream.iBufferPagesHead internally.
//         This allows a (theoretically) unlimited number of 'listeners'
//         for the same T_VorbisStream instance, i.e. the ENCODING
//         only happens once. For that reason, the T_VorbisStream itself
//         does NOT provide any extra buffering, and it's the application's
//         job to "distribute" the encoded Ogg pages after each call of
//         VorbisStream_EncodeSamples() .
//   (5) Back to step 2 of the ENCODING process.
//
//  Whenever a client connects, e.g. asking for "_audiostream.ogg" in SL,
//             call GetInitialHeadersForStreamOutputServer() as explained
//             in that method.
//
//  As long as the stream (e.g. TCP socket) is open and 'ready to send',
//             call VorbisStream_GetNextPagesForStreamOutputServer() as explained there.
//
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
// [B] Ogg/Vorbis DECODER API (to read/play Ogg audio files or a received stream)
//
//  (1) Initialise an instance of struct T_VorbisStream,
//         by calling VorbisStream_InitDecoder().
//         (sampling rate and number of channels cannot be specified;
//          they will be known LATER when a sufficient number of
//          "Ogg pages" with "Vorbis headers" have been fed
//          into VorbisStream_DecodeBytes() )
//  (2) Push a limited number of bytes (received from a stream or read
//          from a file into VorbisStream_DecodeBytes().
//  (3) Pull as many AUDIO SAMPLE POINTS from an internal buffer
//          as you can via VorbisStream_ReadSamples() .
//          (depending on the complex Ogg/Vorbis decoding process,
//           there may be NO NEW SAMPLES in the buffer at all,
//           depending on how the stream was encoded, 
//           and on how many bytes have been pushed into
//           VorbisStream_DecodeBytes() in previous calls ).
//  (4) Back to step 2 of the DECODING process.

//
//---------------------------------------------------------------------------

#ifndef _VORBIS_STREAM_H_
#define _VORBIS_STREAM_H_


// Note: it is NOT necessary to include the Vorbis headers here, because
//       the 'user' of VorbisStream.h should not need to know the details.
//       All Vorbis (vorbisfile,  vorbisenc) specific structs are hidden
//       in ::m_pVorbisFileHelper



/*----------- Constant Definitions ----------------------------------------*/

// Possible values for the parameter 'file_mode' in OutOpen() :
#define WAVE_FILE_MODE_OVERWRITE 0
#define WAVE_FILE_MODE_APPEND    1



/*----------- Data Types --------------------------------------------------*/

#ifndef  TRUE  /* types that used to be defined in windef.h : */
# define TRUE  1
# define FALSE 0
  typedef int            BOOL;
  typedef unsigned char  BYTE;
  typedef unsigned short WORD;
  typedef unsigned long  DWORD; // compatible with windef.h, StringLib.h, YHF_Help.h, etc
#endif


typedef struct t_VorbisStream
{

  int   nChannelsPerSample;
  DWORD dwCurrentSampleIndex;
  DWORD dwCurrFilePos;     // current file access position
  DWORD dwFileSizeInBytes; // contains what the name says (even after closing a file)
  float fltSampleRate;


# define C_VORBIS_STREAM_SIZEOF_BUFFER_FOR_INITIAL_FILE_DATA 4096
  BYTE  bBufferForInitialFileData[C_VORBIS_STREAM_SIZEOF_BUFFER_FOR_INITIAL_FILE_DATA]; // added 2023-02-01 to diagnose problems with "unrecognized stream formats". GREP FOR THE DATE !
  int   BufferUsageForInitialFileData;
  DWORD dwMaxSamplesReturnedByVorbisSynthesisPcmOut; // <- only for debugging;
   // full story in cbproj/SpecLab/bugfixes_and_notes/2023_02_Vorbis_File_IO_Problems.txt .

  BOOL   fEndOfOutput;     // nothing more to be expected from vorbisfile::ov_read()

  char  sz255FileName[256];    // ... or URL
  char  sz255ErrorString[256]; // contains last error description as a string

# define VORBIS_STREAM_BUFFER_MAX_BYTES 65536 // buffer data and headers, for received and transmitted streams
  BYTE   bBuffer[VORBIS_STREAM_BUFFER_MAX_BYTES]; // used for the DECODER (between VorbisStream_DecodeBytes() and VorbisRead()),
                                   // or as the semi-circular FIFO for an outgoing stream with 'multiple readers'.
  DWORD  dwBufferUsage;            // 0 ... VORBIS_STREAM_BUFFER_MAX_BYTES
  DWORD  dwBufferIndex;            // 0 ... dwBufferUsage-1 (steps by one for each byte consumed from the buffer)
  DWORD  dwNumTotalBytesProcessed; // for an extra criterion to process bytes "only in the VERY first byte"
# define VORBIS_STREAM_BUFFER_MAX_PAGES 64 // max number of Ogg pages (or similar) in bBuffer[]
  struct
   { int iBufIndex;     // (byte-)index into T_VorbisStream.bBuffer[] where this page begins
     int iSize_Byte;    // size of this Ogg page, measured in byte
     DWORD dwPageIndex; // unique serial number, starts counting OGG pages at zero
   }BufferPages[VORBIS_STREAM_BUFFER_MAX_PAGES]; // "directory" of all Ogg pages
                           // currently stored in bBuffer.
                           // Usage explained in GetNextPagesForStreamOutputServer().
  int    iBufferPagesHead; // index into BufferPages[] for the next page written in WriteOggPage()
  int    nBufferPagesUsed; // only BufferPages[0..nBufferPagesUsed-1] may be read in GetNextPagesForStreamOutputServer()
  DWORD  dwPageIndex;      // unique serial number, incremented in WriteOggPage(),
                           // copied to BufferPages[iBufferPagesHead++]

  // Info about the received audio stream;
  //  set as soon as we know it from the ogg/vorbis headers :
  int    nChannelsPerSampleInStream;  // nomenclature: a "sample" (sample point) may contain more than one channel
  int    nChannelsPerSampleRequested; // number of active channels (per sample) requested by the caller.
                                      // Usually set in C_VorbisFileIO::ReadSampleBlocks(), but may be ZERO
                                      // if the application 'will accept anything' (since 2021-05-20).
  int    nSamplesPerHeaderInStream;   // number of sample-points (with one or more channels per point) per header or VT_BLOCK
  int    iSampleCountdownBetweenHeaders; // "countdown" for sample-points following a stream-header or VT_BLOCK .
                             // As long as this is nonzero, we expect another SAMPLE POINT
                             // in ReadAndDecodeNonCompressed(), but don't check
                             // for the next stream-header or VT_BLOCK yet .
  BOOL   fGotAllHeaders;     // flag set when received enough data to know the basic Vorbis parameters
                             // (or, for 'uncompressed streams with headers', got the relevant headers)
  BOOL   have_vorbis;        // added 2011-12-15
  BOOL   fEncoderInit2;      // flag for the 'postponed' part of decoder-init


  #define VORBIS_STREAM_MAX_DECODER_OUTPUT_SIZE (32768) // <- ex: 4*65536*4 bytes (far too much)
  float  fltDecoderOutput[VORBIS_STREAM_MAX_DECODER_OUTPUT_SIZE]; // buffer for the output from Vorbis decompressor
  int    nDecoderOutputBufferUsage;     // number of single floats READ INTO fltDecoderOutput[]
  int    nDecoderOutputBufferPeakUsage; // peak value of the above, only for diagnostics
  int    iDecoderOutputReadIndex; // index to read the next sample from fltDecoderOutput[]
  DWORD  dwNumSamplesDecoded;     // ... internally, in handle_vorbis_packet()

  // For INTERNET STREAMS, the following buffer (with the "initial headers")
  //              must be sent to EACH NEW READER that joins the stream.
  // For OGG files (with Vorbis audio content),
  //              bInitialHeaderBuffer[0..nInitialHeaderBufferedBytes-1]
  //              must be written to the begin of the saved file.
  // Note: These "initial headers" (actually, Ogg-"pages" describing the stream)
  //       are NOT written to the streaming output buffer (bBuffer[]) !
  BOOL  fStoreInitialHeaders; // flag controlling when encoded "Ogg pages" go...
  #define VORBIS_INITIAL_HEADER_BUFFER_SIZE 4096  // details in GetInitialHeadersForStreamOutputServer() ...
  BYTE  bInitialHeaderBuffer[VORBIS_INITIAL_HEADER_BUFFER_SIZE];
  int   nInitialHeaderBufferedBytes;

  // For debugging:
  int   nWriteErrors;
  int   nSuccessfullCallsOfReadAndDecode;
  int   nMaxSamplesReturnedByVorbisSynthesisPcmOut;


  DWORD  dwNumSamplesWritten;

   //  Because VorbisFileIO.H shall be 'includeable' WITHOUT
   //  including an army of headers from libogg and libvorbis,
   //  all Vorbis-specific data structures for the IMPLEMENTATION (*.c)
   //  are hidden here :
  struct tVorbisHelper *pVH; // actually a pointer to a
     // T_VorbisHelper struct. That type is only known and defined in the
     // IMPLEMENTATION (VorbisStream.c), not in this HEADER (VorbisStream.h).
  DWORD dwInitMagicE; // only trust pLibVorbisHelper if this 'magic' contains 0x27182818  ..
# define VORBIS_MAGIC_E 0x27182818
        // (the application's T_VorbisStream should be ZEROED
        //  before passing its address to any VorbisStream-API-function,
        //  thus the above 'magic' would be zero, and pLibVorbisHelper would
        //  initially be NULL.)


} T_VorbisStream;

//---------------------------------------------------------------------------
// Functions (prototypes, plain C, may even be moved into a DLL one day..)
//---------------------------------------------------------------------------

#ifndef CPROT  // For peaceful co-existence of "C" and "C++" ....
# ifdef __cplusplus
#   define CPROT extern "C"
# else
#   define CPROT
# endif  // ! "cplusplus"
#endif  // ndef CPROT


//---------------------------------------------------------------------------
CPROT void VorbisStream_Init( T_VorbisStream *pVS );  // Only call this ONCE per instance !
  // Formerly a class constructor. Now a simple 'struct initializer'.
  // Should only be called ONCE for a certain instance to prevent memory leaks.


//---------------------------------------------------------------------------
CPROT void VorbisStream_Exit( T_VorbisStream *pVS );
  // Closes, cleans up, and frees everything that may have been allocated.

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  //  Ogg/Vorbis ENCODER API ("compress once, send to multiple receivers")
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

//---------------------------------------------------------------------------
CPROT BOOL VorbisStream_InitEncoder( // Prepare ENCODING to write files or send streams
        T_VorbisStream *pVS,
        float fltSampleRate,    // number of sample points per second
        int nChannelsPerSample, // number of audio channels PER SAMPLE POINT
        float base_quality); // Desired quality level, currently from -0.1 to 1.0 (lo to hi).
  // Returns TRUE on success and FALSE on any error.
  // [in] base_quality:  Desired quality level, currently from -0.1 to 1.0
  //                     (lo to hi). A good starting point seems to be 0.5  .
  // [in] nChannelsPerSample:  1=mono, 2=stereo
  // [in] fltSampleRate: number of sample points per second.
  //                     Caution - Ogg Vorbis only supports a limited set
  //                     of sample rates, officially ranging from 8 to 192 kHz.
  // Details in VorbisStream.h, "Ogg/Vorbis ENCODER API" (1) .
  // For more details, see the Ogg Vorbis / "libvorbisenc" documentation .
  // Don't miss http://www.xiph.org/vorbis/doc/vorbisenc/examples.html
  //      and   http://www.xiph.org/vorbis/doc/vorbisenc/overview.html


//----------------------------------------------------------------------------------------
CPROT long VorbisStream_EncodeSamples( // .. into VORBIS audio, in OGG "pages" ...
        T_VorbisStream *pVS,
        float *pfltSrc,         // [in] caller's source buffer, 32-bit floats, possibly interleaved
        int nChannelsPerSample, // [in] number of audio channels PER SAMPLE POINT in pfltSrc
        int nSamplePoints );    // [in] number of sample points in pfltSrc
  // Writes some samples in the 'normalized' floating point format (-1.0 ... +1.0)
  //       to an audio file which has been opened for WRITING;
  //       or to an OUTGOING audio stream .
  // Returns the number of SAMPLE POINTS successfully encoded,
  //       or a NEGATIVE value if there was an error .
  // Details in VorbisStream.h, "Ogg/Vorbis ENCODER API" (2) .


//----------------------------------------------------------------------------------------
CPROT int VorbisStream_GetLengthOfInitialHeadersFromEncoder( T_VorbisStream *pVS );
CPROT int VorbisStream_GetInitialHeadersFromEncoder( T_VorbisStream *pVS,
        BYTE *pbDest,   int iMaxDestLength, // [out] network buffer, with limited length
        DWORD *pdwPageIndex); // [out] indicator for the next call of
                              //       of VorbisStream_GetNextPagesFromEncoder()
  // Used to implement a multi-stream-server.
  //   "Compress once" (via WriteSamples_Float),
  //   "ready multiple" (via GetInitialHeaders..() + VorbisStream_GetNextPagesFromEncoder() ).
  // Returns the number of bytes actually placed in pbDest .
  // Details and 'when to call' in VorbisStream.h, "Ogg/Vorbis ENCODER API" (3).
  //

//----------------------------------------------------------------------------
CPROT int VorbisStream_GetNextPagesFromEncoder(
        T_VorbisStream *pVS,
        BYTE *pbDest, int iMaxDestLength,
        DWORD *pdwPageIndex); // [in,out] indicator for the next call
  // Retrieves as many samples as possible (limited to complete Ogg pages)
  //           for one of many possible STREAM READERS .
  // First used to implement a multi-stream-server in Spectrum Lab. Principle:
  //    "one writer/compressor"  (in WriteSamples_Float),
  //    "multiple readers" (GetInitialHeadersForStreamOutputServer
  //                      + VorbisStream_GetNextPagesFromEncoder() ).
  // Returns the number of bytes actually placed in pbDest .
  // Details and 'when to call' in VorbisStream.h, "Ogg/Vorbis ENCODER API" (4).
  //



  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  //  Ogg/Vorbis DECODER API (to read/play Ogg audio files or a received stream)
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

//----------------------------------------------------------------------------------------
CPROT BOOL VorbisStream_InitDecoder( T_VorbisStream *pVS );
  // Prepares both Ogg and Vorbis for DECODING a stream or file.
  // Returns TRUE on success and FALSE on any error .
  // Details in VorbisStream.h, "Ogg/Vorbis DECODER API" (1) .

//---------------------------------------------------------------------------
CPROT int VorbisStream_DecodeBytes( T_VorbisStream *pVS,
        BYTE *pbSource, int nBytes );
  // Pushes a few more bytes into the decoder.
  // Returns the NUMBER OF BYTES that could actually be "pushed" into a buffer.
  // Details in VorbisStream.h, "Ogg/Vorbis DECODER API" (2) .

//----------------------------------------------------------------------------
CPROT long VorbisStream_ReadSamples( T_VorbisStream *pVS,
        int nChannelsWanted, // number of channels 'wanted'
                             // (the stream may be 'stereo' but we only want 'mono')
        int nSamplesWanted,  // number of samples 'wanted' (per destination block, length of destination arrays)
        float *pfltDest);    // destination block (decoded samples)
  // Reads some samples from the decoder's output buffer .
  // Details in VorbisStream.h, "Ogg/Vorbis DECODER API" (3) .
  // Return value: Number of samples ACTUALLY PLACED in *pfltDest++



   //- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
   // Added 2014-04 to support MULTIPLE STREAM CLIENTS ('receivers')
   //               fed by a SINGLE STREAM SERVER ('transmitter')
   //               which only holds the latest few seconds (actually, Ogg pages)
   //               in memory, and a buffer with the three mandatory headers
   //               which will be sent first when a client connects
   //               to SL's built-in HTTP server.
   //- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

//----------------------------------------------------------------------------
int VorbisStream_GetInitialHeadersForStreamOutputServer(
        T_VorbisStream *pVS,
        BYTE *pbDest,  int iMaxDestLength,  // Called by the HTTP server for each new client which attaches to the outbound stream.
        DWORD *pdwPageIndex); // [out] indicator for the next call of GetNextPagesForStreamOutputServer()
     // Returns the number of bytes actually placed in pbDest

//----------------------------------------------------------------------------
int VorbisStream_GetNextPagesForStreamOutputServer(
        T_VorbisStream *pVS,
        BYTE *pbDest, int iMaxDestLength, // Periodically called by the HTTP server to send more samples to a remote client.
        DWORD *pdwPageIndex); // [in,out] indicator for the next call of GetNextPagesForStreamOutputServer()
     // Returns the number of bytes actually placed in pbDest

//----------------------------------------------------------------------------
int VorbisStream_GetNumPagesAvailableForStreamOutput(
        T_VorbisStream *pVS,
        DWORD dwPageIndex);   // [in] indicator for the next call of GetNextPagesForStreamOutputServer()

//----------------------------------------------------------------------------
BOOL VorbisStream_CloseFile( T_VorbisStream *pVS );


#endif // _VORBIS_STREAM_H_

/* EOF <VorbisStream.h> */