//---------------------------------------------------------------------------
// File:  C:\cbproj\Remote_CW_Keyer\VorbisStream.c
// Date:  2024-02-02
// Author: Wolfgang Buescher (DL4YHF)
// Purpose: Interface between the Ogg / Vorbis encoder/decoder
//          (where Ogg is just a container, and Vorbis is the audio format)
//          and our 'audio streaming server', with the intention
//          of NOT having to pull in dozens of the original
//          Vorbis- and Ogg-specific header files in the application
//          (to have the option to compile / link this into an
//           easy-to-use, "standalone" library one day).
//    Uses libogg V1.2.0 and libvorbis V1.3.1, locally saved in            
//           C:\cbproj\ogg_vorbis\libvorbis_1_3_1\*.*   (audio codec)      
//      and  C:\cbproj\ogg_vorbis\libogg_1_2_0\*.*      (stream container) 
//    Does NOT use any windows gimmics, just standard file I/O.            
//    The INTERFACE (*.h) doesn't even use the OGG / VORBIS headers !      
//                                                                         
//  Based on a more complex module for DL4YHF's Spectrum Lab,
//   but greatly simplified by omitting the nanosecond-accurate
//   timestamps embedded in the Ogg stream (container).
//
// The "Short overview / most important functions for AUDIO STREAMING"
//   has been moved from VorbisStream.c (which may be compiled into a DLL)
//                  into VorbisStream.h (available even when USING the DLL).
//   
//
// Revision History :
//
//  2024-11-25: Tried to used this module in a C++Builder V12 project.
//              Had to add the following under "Project".."Options"...
//              "C++ Compiler" .. "Directories and Conditionals" :
//               * C:\cbproj\ogg_vorbis\libvorbis_1_3_1\include
//  2024-02-02: Based on C:\cbproj\SoundUtl\VorbisFileIO.cpp, created
//                       C:\cbproj\Remote_CW_Keyer\VorbisStream.c .
//---------------------------------------------------------------------------


#include "SWITCHES.H"  // project specific compiler switches ("options")
                       // must be included before anything else !
                       // (use the same SWITCHES when compiling SL and the DLL,
                       //  most likely \CBproj\SpecLab\SWI_V01\SWITCHES.H  )

#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#define _I_AM_VORBIS_STREAM_ 1
#include "VorbisStream.h"      // header for THIS file

#pragma hdrstop

#define OV_EXCLUDE_STATIC_CALLBACKS /* don't need OV_CALLBACKS_DEFAULT & Co .. */
#include "vorbis/codec.h"      // C:\cbproj\ogg_vorbis\libvorbis_1_3_1\include\vorbis\codec.h
#include "vorbis/vorbisfile.h" // used for an 'easy start' to DECODE ogg files with vorbis audio
#include "vorbis/vorbisenc.h"


long VorbisStream_i32DebugDummy = 0;


// Internal data for any VorbisStream_ object. Later used for streams, too.
//    (We don't want to declare this stuff in the header VorbisStream.h
//     because it's purely INTERNAL. )

struct tStreamDemux
{
   int serial;            // 2013-07-05: tStreamDemux.serial=-1 now marks an UNUSED entry
   ogg_stream_state os;   // type declared in c:\cbproj\ogg_vorbis\libogg_1_2_0\include\ogg\ogg.h
   int (*handler)(struct tVorbisFileHelper *pvfh, ogg_packet * op);
   // 2024-11-25: Mr Pedantic (BCB V12, possibly similar as "clang")
   //             complained about the ABOVE:
   // > [bcc32c Warning] : declaration of 'struct tVorbisFileHelper'
   // >                    will not be visible outside of this function
   int activity_detector; // added 2013-07-05 to recycle the OLDEST (orphaned) entry
};

typedef struct tVorbisHelper // T_VorbisHelper ...
{

  // bitstream settings for the encoder and decoder :
  ogg_stream_state osv;  // Stream state - vorbis stream
    // ogg_stream_state ost;  // former   timestamp stream (removed)
          // > An ogg_stream_state takes physical pages,
          // > weld into a logical stream of packets .
  ogg_sync_state   oy;   // oy: not a fictional creature by Stephen King, but :
          // > ogg_sync_state tracks the synchronization of the current page.
          // > It is used during decoding to track the status of data
          // > as it is read in, synchronized, verified, and parsed into pages
          // > belonging to the various logical bitstreams in the current
          // > physical bitstream link.
          //
  vorbis_info      vinfo;
  vorbis_comment   vcomment; // struct that stores all the user comments
  vorbis_dsp_state vdsp;     // central working state for the packet->PCM decoder
  vorbis_block     vblock;   // local working space for packet->PCM decode
  int   nRcvdVorbisHeaders;  // Number of vorbis headers received (important: 3 headers when opening)
                             // Note: A "Vorbis Header" is not an "Ogg Page" !

  ogg_packet header;
  ogg_packet header_comm; // not sure if it's ok to use a LOCAL (stack) var for this
  ogg_packet header_code;
  DWORD dwNumSamplesWrittenAfterLastOggPage; // added 2024-02-16 in an attempt to reduce stream latency

  #define MAX_DEMUX 10 // Added 2011-12-15, to demultiplex streams as in Paul's vtvorbis.c
  struct tStreamDemux demux[MAX_DEMUX];
  int    ndemux; // number of 'installed' demultiplexers in demux[]
                 // (2013-07-05: this may give problems with the Palaoa streams,
                 //              which seems to use new 'serials' after a bunch
                 //              of seconds ... grep for the modification date !
                 // 2013-07-05: tStreamDemux.serial=-1 now marks an UNUSED entry,
                 //             and tStreamDemux.activity_detector will be used
                 //             to recycle the OLDEST (orphaned) entry .

  T_VorbisStream *pVS;    // link back from the T_VorbisHelper to its 'parent',
                          // the T_VorbisStream (exposed in VorbisStream.h)

} T_VorbisHelper;


/*----- Internal functions / Callbacks / forward declarations ----*/
/*----------- NO CLASS MEMBERS OR CLASS METHODS HERE ! -----------*/


char* VorbisErrorCodeToString( int vorbis_error_code )
{
  static char sz40UnknownCode[44];
  static char sz40NotAnError[44];
  if( vorbis_error_code > 0 )
   { sprintf(sz40NotAnError, "Code %d is not an error !",(int)vorbis_error_code);
     return sz40NotAnError;  // stupid caller, kick the bucket :)
   }
  // For an explanation, see vorbis(libvorbis_1_3_1)/doc/vorbis-errors.txt :
  switch( vorbis_error_code )
   {
     case 0         : return "ok (no error)";
     case OV_FALSE  : return "FALSE / info not available";    // not available because no playback, etc
     case OV_EOF    : return "End of file";
     case OV_HOLE   : return "lost packet(s)";  // 'hole in the data'
     case OV_EREAD  : return "media read error";
     case OV_EFAULT : return "internal fault";
     case OV_EIMPL  : return "feature not implemented";
     case OV_EINVAL : return "invalid argument value";
     case OV_ENOTVORBIS: return "not a Vorbis file/stream/packet";
     case OV_EBADHEADER: return "invalid Vorbis bitstream header";
     case OV_EVERSION  : return "Vorbis version mismatch";
     case OV_ENOTAUDIO : return "packet is no audio data";
     case OV_EBADPACKET: return "invalid packet submitted";
     case OV_EBADLINK  : return "invalid stream section / bad link";
     case OV_ENOSEEK   : return "bitstream is not seekable";
     // In additon to the above 'Ogg/Vorbis' error codes, added the following
     // more 'general' errors in 2020-10-11 :
     case OV_ERR_TIMEOUT: return "timeout";

     default: sprintf(sz40UnknownCode, "unknown error code (%d)", (int)vorbis_error_code );
        return sz40UnknownCode; // caller should process this result a.s.a.p. !
   }
} // end VorbisErrorCodeToString()


//---------------------------------------------------------------------------
static T_VorbisHelper *VorbisStream_AllocHelper( T_VorbisStream *pVS )
  // Allocates a 'vorbis helper structure' .
  // Contains everything we don't want to expose in VorbisFileIO.h,
  //   because if we did, any application using VorbisStream.h
  //   would also have to #include an army of Ogg- and Vorbis specific
  //   headers.
{
  // Is the 'helper struct' with the Ogg/Vorbis-specific stuff already valid ?
  if( (pVS->dwInitMagicE == VORBIS_MAGIC_E) && (pVS->pVH != NULL) )
   { return pVS->pVH; // the "internal helper" is already allocated,
                      // so don't allocate it again !
   }
  pVS->pVH = malloc( sizeof(T_VorbisHelper) );
  if( pVS->pVH != NULL )
   { // Ok, successfully allocated a block for all those Ogg/Vorbis specific structs..
     memset( pVS->pVH, 0, sizeof(T_VorbisHelper) ); // clear old garbage, including pVH->stream_header_1
     pVS->pVH->pVS = pVS; // "link back" from the T_VorbisHelper to its 'parent',
                          //         which is the T_VorbisStream .

     // > It is safe to call ogg_stream_clear on the same structure more than once.
     // Ok, so let's make liberal use of it, despite the 'memset' further above:
     ogg_stream_clear( &pVS->pVH->osv );  // Ogg stream for Vorbis (audio)

     pVS->dwInitMagicE = VORBIS_MAGIC_E; // "open for business" !
     return pVS->pVH;
   } // end if( pVS->pLibVorbisHelper != NULL )

  // Arrived here ? Something went BADLY WRONG ! Here's what:
  strcpy( pVS->sz255ErrorString, "Failed to allocate the Ogg/Vorbis structs" );
  return NULL;

} // end VorbisStream_AllocHelper()

//---------------------------------------------------------------------------
void VorbisStream_FreeHelper( T_VorbisStream *pVS )
{
  if( (pVS->dwInitMagicE == VORBIS_MAGIC_E) && (pVS->pVH != NULL) )
   {
     free( pVS->pVH );
     pVS->pVH = NULL;
   }
} // end VorbisStream_FreeHelper()


//---------------------------------------------------------------------------
// Vorbis callback (and former file I/O) functions.
//  As their names suggest, they are expected to work in exactly
//  the same way as normal c io functions (fread, fclose etc.).
//  Its up to us to return the information that the libs need to parse
//  the file from memory, stream, 'real files', ROM, or whatever .
// > The function prototypes for the callbacks are basically the same as for
// > the stdio functions fread, fseek, fclose, ftell.
// > The one difference is that the FILE* arguments have been replaced with
// > a void* - this is to be used as a pointer to whatever internal data these
// > functions might need.
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// Function: VorbisRead
// Purpose: Callback for the Vorbis read function.
//            The return value indicates how many samples have actually been returned.
//---------------------------------------------------------------------------
static size_t VorbisRead( // in THIS implementation, only "reads" from T_VorbisStream.bBuffer[] !
        void *ptr,         // ptr to the data that the vorbis files need
        size_t byteSize,   // how big a byte is  ;)   [usually one]
        size_t sizeToRead, // How much we can read
        void *datasource)  // a pointer to the data we passed into ov_open_callbacks
{
  T_VorbisStream *pVS = (T_VorbisStream*)datasource;
  int nBytesRead = 0;
  if( (pVS!=NULL) && (pVS->dwInitMagicE == VORBIS_MAGIC_E) ) // Is the "datasource" really our T_VorbisStream ?
   {
   } // end if ( pvfh != NULL )
  return nBytesRead;
} // end VorbisRead()


//----------------------------------------------------------------------------
static int VorbisSeek(
   void *datasource,   // [in] pointer to the data we passed into ov_open_callbacks
   ogg_int64_t offset, // [in] offset from the point we wish to seek to
   int  whence )       // [in] where we want to seek to (whence = "from where")
  // Callback for the Vorbis seek function
  //  Sets the file pointer associated with stream to a new position
  //  that is offset bytes from the file location given by whence.
  //  fseek (!) returns 0 if the pointer is successfully moved
  //  and nonzero on failure.
  //   ( Hmmm. Not many streams, especially not AUDIO STREAMS, are SEEKABLE !
  //     In Vorbis parlance, "stream" often means "file" .
  //     In SpecLab parlance, "stream" is an ENDLESS stream (never seekable)
  //               and "file" is a file on a local storage medium (seekable).
{
  return -1; // Let vorbisfile.c know WE CANNOT SEEK !
} // end VorbisSeek()


//---------------------------------------------------------------------------
static int VorbisClose(
   void *datasource) // pointer to the data we passed into ov_open_callbacks
  // Callback for the Vorbis close function
  //    Like fclose(), returns 0 on success (!) .
{
  // ex: return fclose( (FILE*)datasource );
  return 0;  // no error
} // end VorbisClose()

//---------------------------------------------------------------------------------
static long VorbisTell(
   void *datasource) // pointer to the data we passed into ov_open_callbacks
  //  Callback for the Vorbis tell function
  //  Returns the current file pointer for stream.
  //  The offset is measured in bytes from the beginning of the file
  //   (btw the file IS ALWAYS binary). The value returned by ftell
  //   can be used in a subsequent call to fseek.
  // NOTE (WB): The vorbisfile API seems to use seek & tell
  //               to determine the TOTAL LENGTH of the file,
  //               making it useless to process REAL AUDIO STREAMS
  //               (with a length unknown in advance) .
  //            But it only does this when VorbisSeek(), called 1st,
  //            indicates by a non-negative result that the 'stream' (file)
  //            is seekable. Found this in the caller of VorbisTell() :
  //     /* we can seek, so set out learning all about this file */
  // and /* If seek_func is implemented, tell_func must also be implemented */
  //     In other words: we don't really need VorbisSeek & VorbisTell
  //     seeking just to play files (without 'hopping around').
  //     Only Spectrum Lab needs seeking, if an audio file is to be played
  //     in an 'endless loop' .  For (endless) streams, this makes no sense anyway.
{
  long i32Result = -1;
  T_VorbisHelper *pVH = (T_VorbisHelper*)datasource;
  if( pVH )
   {
#   if (VORBIS_USE_QFILE)
     if( pVH->m_QFile.iHandle > 0 )
      { i32Result = QFile_Seek( &pVH->m_QFile, 0/*offset*/, QFILE_SEEK_CUR/*fromwhere*/ );
        // QFile_Seek( pqFile, 0, QFILE_SEEK_CUR )  returns the current file position .
      }
#   endif // (VORBIS_USE_QFILE)
   } // end if ( pVH != NULL )
  return i32Result;
} // end VorbisTell()

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // 	End of Vorbis callback functions
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -



//----------------------------------------------------------------------------
static int handle_vorbis_packet(
              T_VorbisHelper *pVH, // [in,out] ogg/vorbis-related states
              ogg_packet *op)      // [in] Ogg packet with Vorbis content
  //  Attempt to decode a vorbis packet, returns zero if failed.
  //  In SpectrumLab, called through
  //   TSoundThread::Execute() -> C_AnyAudioFileIO::ReadSampleBlocks()
  //    -> VorbisStream_ReadSampleBlocks() -> ReadAndDecodePages()
  //          -> DecodeOggPage() -> handle_vorbis_packet()
  //               -> vorbis_synthesis(), vorbis_synthesis_blockin(),
  //                  implemented in a DLL so cannot step into them from here.
  //
{
  int nSamplePoints, nSamplesConsumed, nChannelsPerSample, vorbis_error_code;
  float **ppfltPCM;
  float *pfltDest;
  T_VorbisStream *pVS = pVH->pVS;
  // long double timestamp;
  // double srcal;
  // int tbreak;

  if( pVS == NULL ) // oops..
   { return 0;
   }

  // > Decoding workflow (using Libvorbis, *WITHOUT* vorbisfile.c ) :
  // > 1. When reading the header packets of an Ogg stream, you can use
  // >    vorbis_synthesis_idheader to check whether a stream might be Vorbis.
  // > 2. Initialize a vorbis_info and a vorbis_comment structure using
  // >    the appropriate vorbis_*_init functions, then pass the first
  // >    three packets from the stream (the Vorbis stream header packets)
  // >    to vorbis_synthesis_headerin in order. At this point,
  // >    you can see the comments and basic parameters of the Vorbis stream.
  // > 3. Initialize a vorbis_dsp_state for decoding based on
  // >    the parameters in the vorbis_info by using vorbis_synthesis_init.
  // > 4. Initialize a vorbis_block structure using vorbis_block_init.
  // > 5. While there are more packets to decode:
  // >    5.1. Decode the next packet into a block using vorbis_synthesis.
  // >    5.2. Submit the block to the reassembly layer using vorbis_synthesis_blockin.
  // >    5.3. Obtain some decoded audio using vorbis_synthesis_pcmout
  // >         and vorbis_synthesis_read. Any audio data returned
  // >         but not marked as consumed using vorbis_synthesis_read
  // >         carries over to the next call to vorbis_synthesis_pcmout.
  // > 6. Destroy the structures using the appropriate vorbis_*_clear functions.
  //



  if( pVH->nRcvdVorbisHeaders <= 0)    // First time through?
   {
      vorbis_info_init( &pVH->vinfo);  //  (2)
      vorbis_comment_init( &pVH->vcomment );
      pVH->nRcvdVorbisHeaders = 0;
   }

  if( pVH->nRcvdVorbisHeaders < 3)     // Collect 3 vorbis headers
   {
     // > vorbis_synthesis_headerin() decodes a header packet from a Vorbis stream
     // > and applies the contents to the given
     // > vorbis_info structure (to provide codec parameters to the decoder) and
     // > vorbis_comment structure (to provide access to the embedded Vorbis comments).
     // > Once the three Vorbis header packets (info, comments, and codebooks,
     // > in that order) have been passed to this function,
     // > the vorbis_info structure is ready to be used in a call to vorbis_synthesis_init.
     vorbis_error_code = vorbis_synthesis_headerin(
            &pVH->vinfo/*out*/, &pVH->vcomment/*out*/, op/*in: ogg_packet to decode*/);
     if( vorbis_error_code < 0)
      {
        if( pVH->nRcvdVorbisHeaders==0 )
         {
           return 0;
         }
      }
      // VT_report( 2, "got vorbis header %d", nRcvdVorbisHeaders + 1);

      if( ++pVH->nRcvdVorbisHeaders == 3)  // All mandatory headers received ?
      {
         // Initialise vorbis decoder
         vorbis_synthesis_init( &pVH->vdsp, &pVH->vinfo);        // (3)
          // |__ initializes a vorbis_dsp_state structure for decoding
          //     and allocates internal storage for it.
          // (don't forget to call vorbis_dsp_clear( &pVH->vdsp )
          //      = "cleanup for vorbis_synthesis_init()" later )
         vorbis_block_init( &pVH->vdsp, &pVH->vblock);           // (4)
          // |__ initializes a vorbis_block structure and allocates
          //     its internal storage. A vorbis_block is used to represent
          //     a particular block of input audio
          //     which can be analyzed and coded as a unit.

         // Save all the application needs to know in our own struct:
         pVH->pVS->fltSampleRate      = pVH->vinfo.rate;
         pVH->pVS->nChannelsPerSample = pVH->vinfo.channels;
         pVH->pVS->fGotAllHeaders = TRUE;  // ready for business from VORBIS' point of view
      }
   } // end if( pVH->nRcvdVorbisHeaders < 3)
  else   // Deal with vorbis audio data
   {
     if( vorbis_synthesis( &pVH->vblock, op) == 0)            //   (5.1)
      { // |_ decodes a Vorbis packet into a block of data.
        //    The vorbis_block should then be submitted to the vorbis_dsp_state
        //    for the decoder instance using vorbis_synthesis_blockin
        //    to be assembled into the final decoded audio
        vorbis_synthesis_blockin( &pVH->vdsp, &pVH->vblock); //   (5.2)
        // |_ submits a vorbis_block for assembly into the final decoded audio.
        //    After this, decoded audio can be obtained with vorbis_synthesis_pcmout
      }

     // About decoding Vorbis audio into PCM :
     // > vorbis_synthesis_pcmout retrieves buffers containing decoded audio samples.
     // > The application is not required to make use of all of the samples
     // > made available to it by one call to this function
     // > before it continues to decode. Use vorbis_synthesis_read to inform
     // > the decoder of how many samples were actually used.
     // > Any unused samples will be included in the buffers output
     // > by the next call to this function.
     nSamplesConsumed = 0;
#   ifdef __BORLANDC__  // "... is assigned a value that is never used".. shut up ..
     (void)nSamplesConsumed;
#   endif

     while( (nSamplePoints=vorbis_synthesis_pcmout( &pVH->vdsp, &ppfltPCM)) > 0)
      {
        int i, j;

        // Since we ideally WANT TO consume "all" nSamplePoints offered by
        //   vorbis_synthesis_pcmout() here, what's the MAXIMUM number of
        //   samples to expect when calling
        // ReadAndDecodePages() -> DecodeOggPage() -> handle_vorbis_packet() ?
        //   Just detect the maximum, and store it in an "easy-to-inspect" variable:
        if( nSamplePoints > pVH->pVS->nMaxSamplesReturnedByVorbisSynthesisPcmOut )
         { pVH->pVS->nMaxSamplesReturnedByVorbisSynthesisPcmOut = nSamplePoints; // <- only for debugging;
            // full story in cbproj/SpecLab/bugfixes_and_notes/2023_02_Vorbis_File_IO_Problems.txt .
            // TEST RESULTS (values seen in dwMaxSamplesReturnedByVorbisSynthesisPcmOut) :
            //   Sarabande_Album_komplett.ogg : 1024 ,
            // but decoding an entire OGG-PAGE usually emitted up to 20480 (!) new entries
            //   into pVS->fltDecoderOutput[] .
            // Thus, when transporting VORBIS, an OGG-PAGE seems to contain 20 OGG-PACKETS.
            // (but don't assume anything, use a sufficiently large pVS->fltDecoderOutput[]).
         } // end if < new "maximum" for the number of samples delivered by vorbis_synthesis_pcmout() >

        // Copy as many samples into pVS->fltDecoderOutput[] as fit in there.
        // Note that the *CHANNELS* in pVS->fltDecoderOutput[] are interleaved;
        // i.e. pVS->fltDecoderOutput[0] = left channel, first sample,
        //      pVS->fltDecoderOutput[1] = right channel, first sample,
        //      pVS->fltDecoderOutput[0] = left channel, second sample,
        //      pVS->fltDecoderOutput[1] = right channel, second sample;  etc.
        nChannelsPerSample     = pVS->nChannelsPerSampleRequested; // <- may be ZERO..
        if( nChannelsPerSample > pVS->nChannelsPerSample )
         {  nChannelsPerSample = pVS->nChannelsPerSample;
         }
        if( nChannelsPerSample < 1 )
         {  nChannelsPerSample = 1;  // avoid div-by-zero further below
         }
        // May have to limit the number of sample points which can be 'consumed' here:
        nSamplesConsumed = ( VORBIS_STREAM_MAX_DECODER_OUTPUT_SIZE
                              - (int)pVS->nDecoderOutputBufferUsage)
                         / nChannelsPerSample;
        // 2021-05-21 : Exception from 'divide by zero' here . nChannelsPerSample = 0 ?
        if( nSamplesConsumed > nSamplePoints )
         {  nSamplesConsumed = nSamplePoints;   // <- this is "normal" : Less samples available than our buffer could "consume"
         }
        else if( nSamplePoints > nSamplesConsumed )
         { strcpy( pVS->sz255ErrorString, "vorbis_synthesis_pcmout() delivered more samples than we can handle" );
           // Anyway, we'll inform Vorbis that we consumed less samples than expected.
           // Shouldn't be a problem, though .. re-read the documentation
           // about vorbis_synthesis_pcmout() quoted further above !
         }
        pfltDest = pVS->fltDecoderOutput + pVS->nDecoderOutputBufferUsage;
        pVS->nDecoderOutputBufferUsage += nSamplesConsumed * nChannelsPerSample;
        if( pVS->nDecoderOutputBufferUsage > pVS->nDecoderOutputBufferPeakUsage )
         {  pVS->nDecoderOutputBufferPeakUsage = pVS->nDecoderOutputBufferUsage;
         }
        for( j=0; j<nSamplesConsumed; j++)
         { for( i = 0; i < nChannelsPerSample; i++)
            { // frame[i] = ppfltPCM[i][j];
              *pfltDest++ = ppfltPCM[i][j];
            }
         }
        if( pVS->dwNumSamplesDecoded==0  )
         { // ex: CallBack_Info( pVH, "Vorbis decoded first %ld PCM samples, %ld timestamp(s).",
           //                   (long)nSamplesConsumed, (long)pVH->i64TimestampCounter );
         }

        pVS->dwNumSamplesDecoded += nSamplesConsumed;  // formerly important for processing received timestamps


        // inform the decoder of how many samples were actually used :
        vorbis_synthesis_read( &pVH->vdsp, nSamplesConsumed );
        if( nSamplesConsumed < nSamplePoints )  // cannot accept more samples
         {  break;
         }
      } // end while vorbis_synthesis_pcmout(..)
   } // end else < Vorbis samples >

  return 1;
} // end handle_vorbis_packet()


//----------------------------------------------------------------------------------------
static int handle_dummy_packet(
              T_VorbisHelper *pVH, // [in,out] ogg/vorbis-related states
              ogg_packet *op  )
  // Dummy handler for ignoring unrecognised logical streams in the ogg stream
{
   return 1;
} // end handle_dummy_packet()

//----------------------------------------------------------------------------------------
static BOOL DecodeOggPage( // Called after ogg_sync_pageout() told us "another page is ready"
              T_VorbisHelper *pVH, // [in,out] ogg/vorbis-related states
              ogg_page *opage) // [in] received Ogg page with header="OggS", header_len=e.g. 28, body, body_len=e.g. 64.
  // [out] PCM-samples may be appended to our pVH->fltDecoderOutput[ nDecoderOutputBufferUsage++ ]
{
  int i, serial;
  int oldest_active_demux, oldest_activity_detector; // added 2013-07-05
  struct tStreamDemux *dm;
  ogg_packet opack; // because the stupid debugger couldn't find the type definition:

                    // It's in c:\cbproj\ogg_vorbis\libogg_1_2_0\include\ogg\ogg.h !

                    // Note: "opack.packet" is just a BYTE POINTER (aka "unsigned char").

  BOOL fResult = FALSE;

  // BOOL fNewSerial = FALSE;


  // 2020-10-11: Crashed somewhere inside DecodeOggPage(), or one of the
  //  DLL functions invoked from there, with a 'NonDelphiException',
  //  when fed with data originating from a raw stream with VT_BLOCK structs only,
  //  i.e. a stream that did NOT contain any Ogg/Vorbis pages at all.
  //  Tried to make this thing a bit more robust then. Grep for date 2020-10-11 .
  //  Used DK7FC's test stream at tcp://129.206.29.82:49335 which had been
  //  identified as AUDIO_FILE_FORMAT_STREAM_WITH_VLF_RX_BLOCKS
  //  (audio file/stream types defined in c:\cbproj\SoundUtl\AudioFileDefs.h).
  // Call stack (shortly before the suspected crash) :
  //  C_AnyAudioFileIO::InOpen() -> VorbisStream_InOpen() -> VorbisStream_InOpenInternal()
  //   -> ReadAndDecodePages() -> DecodeOggPage( opage->header = NULL, header_len=0, body=NULL, body_len=0 )
  if( opage->header == NULL ) // avoid crashing in ogg_page_serialno() [details below]
   { return FALSE;  // doesn't look like a valid ogg_page - ogg_sync_pageout() not successfully called ?
   }

  serial = ogg_page_serialno( opage );
  // ogg_page_serialno(ogg_page *og) :
  // > Returns the unique serial number for the logical bitstream of this page.
  // > Each page contains the serial number for the logical bitstream that it belongs to.
  // If the stupid IDE (Borland) cannot find the implementation of "ogg_page_serialno":
  // It's in C:\cbproj\ogg_vorbis\libogg_1_2_0\src\framing.c .
  // ogg_page_serialno() doesn't perform any error checking at all. 
  // Instead, when fed with garbage, it simply crashes. Here's an example why :
  // > OV_API int ogg_page_serialno(const ogg_page *og)
  // > {
  // >   return(og->header[14] |
  // >         (og->header[15]<<8) |
  // >         (og->header[16]<<16) |
  // >         (og->header[17]<<24));
  // > }
  // Guess what happens when opage.header is NULL .. (WB, 2020-10-11) .


  // Find the demux entry for this serial, or allocate if not seen before
  dm = pVH->demux;       // in T_VorbisHelper :  demux[MAX_DEMUX];  MAX_DEMUS used to be 10 ..
  for( i = 0; i<pVH->ndemux && dm->serial != serial; i++, dm++)
   {
   }
  if( i == pVH->ndemux)  // First time this serial has been seen?
   {
     // 2013-07-05 : Repeatedly got here after playing the 'Palaoa' audio stream
     //       for a minute, with the following .....
     //   serial = 12316 (etc, the INITIAL stream serial was 27529 in that case)
     //   pVH->ndemux = 1
     //   pVH->demux->serial = -1
     // An excerpt from SL's "web stream control panel" showed:
     // > 08:25:32.1 Found vorbis stream, serial 13977
     // > 08:25:32.2 Vorbis decoded first 2048 PCM samples, 0 timestamp(s).
     // > 08:26:26.2 Found vorbis stream, serial 31673
     // > 08:27:27.8 Found vorbis stream, serial 5021
     // > 08:28:28.0 Found vorbis stream, serial 26924
     // > 08:29:27.5 Found vorbis stream, serial 6270
     // > 08:30:25.9 Found vorbis stream, serial 26777
     // > 08:31:26.9 Found vorbis stream, serial 5097
     // Looks like the AWI stream server uses a new 'serial' for the Vorbis-encoded
     // audio stream once a minute. SL would run out of 'demultiplexers' after ten minutes
     // without the following modifications...
     pVH->ndemux++;
     if( pVH->ndemux >= MAX_DEMUX )
      { // ex: (before 2013-07-05): return FALSE;  // Ignore stream  (and the entire Ogg page)
        // 2013-07-05: tStreamDemux.serial=-1 now marks an UNUSED entry,
        //             and tStreamDemux.activity_detector will be used
        //             to recycle the OLDEST (orphaned) entry :
        for( i = 0; i<MAX_DEMUX; i++ )
         { dm = &pVH->demux[i]; // in T_VorbisHelper :  demux[MAX_DEMUX];  MAX_DEMUS used to be 10 ..
           if( dm->serial < 0 ) // this entry has been FREED so it can be 'recycled' now !
            { break;
            }
         }
        if( i<MAX_DEMUX ) // still no luck.. 'recycle' the OLDEST (orphaned) entry
         { oldest_active_demux = 0;
           oldest_activity_detector = pVH->demux[oldest_active_demux].activity_detector;
           for( i = 1; i<MAX_DEMUX; i++ )
            { if( pVH->demux[i].activity_detector < oldest_activity_detector )
               { oldest_activity_detector = pVH->demux[i].activity_detector;
                 oldest_active_demux = i;
               }
            }
           dm = &pVH->demux[oldest_active_demux];  // recycle the 'oldest unused' demultiplexer, regardless of its type
         }
        if( dm->serial > 0 )  // delete the orphaned demultiplexer / ogg_stream_state struct ?
         { // About ogg_stream_destroy():
          // > This function frees the internal memory used by the
          // > ogg_stream_state struct as well as the structure itself.
          // > This should be called when you are done working with an ogg stream.
          // > It can also be called to make sure that the struct does not exist.
          // > It calls free() on its argument, so if the ogg_stream_state
          // > is not malloc()'d or will otherwise be freed by your own code,
          // > use ogg_stream_clear instead.
          ogg_stream_clear( &dm->os );
         }
      } // end if( pVH->ndemux >= MAX_DEMUX )
     if( dm==NULL )
      { return FALSE;
      }
     ogg_stream_init( &dm->os, serial);
     dm->handler = NULL;
     dm->serial = serial;
     // fNewSerial = TRUE;  // added 2013-07-05
   } // end if( i == pVH->ndemux), i.e. "first time this serial has been seen"
  else // this serial has been seen before (which is the NORMAL case) ...
   { // fNewSerial = FALSE;
   }

  // Added 2013-07-05: Detect activity for THIS serial's multiplexer,
  //                   and decrement the activitiy-detector for all others
  //                   to find those which are not alive anymore (when needed):
  for( i = 1; i<MAX_DEMUX; i++ )
   { if( pVH->demux[i].serial == serial )
      {  pVH->demux[i].activity_detector = 10000;
      }
     else
     if( pVH->demux[i].activity_detector > 0 )
      { --pVH->demux[i].activity_detector;
      }
     // When running out of 'demultiplexers', the one with the LOWEST
     // activity-detector-value will be considered as 'orphaned' (not active anymore),
     // and recycled for a new 'serial' (ID). This hopefully cures the problem
     // with the 'PALAOA Neumayer Satellite Stream', which use a new stream ID each minute.
     // 2013-07-05: It did; now http://icecast.awi.de:8000/PALAOA1.OGG
     //             kept running for over 10 minutes.
   } // end for( i = 1; i<MAX_DEMUX; i++ )

  // ogg_stream_pagein : adds a complete page to the bitstream.
  // > In a typical decoding situation, this function would be called
  // > after using ogg_sync_pageout to create a valid ogg_page struct.
  // > Internally, this function breaks the page into packet segments
  // > in preparation for outputting a valid packet to the codec decoding layer.
  if( ogg_stream_pagein( &dm->os, opage) < 0)
   { // VT_bailout( "Error reading bitstream data");
     return FALSE;
   }

  // Read all available packets in this page
  //  (processing a COMPLETE page was fine in Paul's vtvorbis.c
  //   because it could push as many decompressed PCM audio samples
  //   into the output as it wanted.
  //   But here, in VorbisFileIO.cpp, there's only a limited buffer.
  //   The maximum number of audio samples decoded from a single Ogg page
  //   needs yet to be found out ! )
  while( ogg_stream_packetout( &dm->os, &opack) == 1) // <- "assembles a data packet for output to the codec decoding engine".
   {        // ,--------------------'      |
            // |   ,-----------------------'
            // |   '-> "Pointer to the packet to be filled in with pointers to the new data.
            // |        This will typically be submitted to a codec for decode after this function is called.
            // |        The pointers are only valid until the next call on this stream state."
            // '-> "Pointer to a previously declared ogg_stream_state struct.
            //      Before this function is called, an ogg_page should be submitted
            //      to the stream using ogg_stream_pagein()."
            //
     handle_vorbis_packet( pVH, &opack);
   } // end while( ogg_stream_packetout( ... )

  return fResult;

} // end DecodePage()


//----------------------------------------------------------------------------------------
static int ReadAndDecodePages(
              T_VorbisStream *pVS) // [in,out] ogg/vorbis-related states, e.g.:
                                 // [out] pVH->fltDecoderOutput[filled in vorbis_synthesis_pcmout()],
                                 // [out] pVH->nDecoderOutputBufferUsage, ...
                                 // [in]  pVH->iDecoderOutputReadIndex (important because SOME samples in fltDecoderOutput[] haven't been read yet)
  // Main part of the OGG(!) DECODER. Does what the name says ...
  // Details in the function body. Call stack: See handle_vorbis_packet() .
  // Return value: number of pages actually read.
  //               Zero    : nothing read but no error.
  //               Negative: one of the error codes defined in ogg.h -> codec.h,
  //                         e.g. OV_ENOTVORBIS, OV_EBADHEADER, OV_EBADPACKET .
  // Caller :  C_AnyFileIO::ReadSampleBlock() -> VorbisStream_ReadSampleBlocks()
  //            -> ReadAndDecodePages()
  //                |-> ogg_sync_pageout(), ogg_sync_buffer(), ogg_sync_wrote()
  //                |-> DecodeOggPage() [repeated as long as ogg_sync_pageout() has more "pages" to decode]
  //                |     |-> DecodeOggPage() -> handle_vorbis_packet() -> ...., vorbis_synthesis_pcmout()
  //                |     |                        |- fills pVH->fltDecoderOutput[ pVH->nDecoderOutputBufferUsage++ ]
  //                Called in a loop until ReadSampleBlocks() has "enough" samples,
  //                or (for web audio streams only) a read-timeout occurred .

{
  T_VorbisHelper *pVH = pVS->pVH;
  int  nPagesRead, iOffset, nLoops;
  int  ov_result; // vorbis result code (when negative, it's an error code)
  int  pageout_result;  // value returned by ogg_sync_pageout(), indicates readiness of another "page"
  size_t nBytesRead, nBytesLeft;
  ogg_page opage; // ogg_page op only contains a POINTER to the page's netto data,
                  // it's a small object so it's ok to use a local stack var here.
  char *buffer;
  DWORD *pdw;
  // BOOL found_uncompressed_header;  // future plan: also allow UNCOMPRESSED "auxiliary" channels

  if( (pVH==NULL) || (pVS->dwInitMagicE!=VORBIS_MAGIC_E) )
   { return OV_ENOTVORBIS;   // failed to allocate the internal T_VorbisHelper ..
   }

  memset( &opage, 0, sizeof(opage) ); // don't leave anything to fate - clear garbage


  nLoops = nPagesRead = 0;
  while( (nLoops<1000) && (nPagesRead<10) )
   {
     // We want to obtain a complete page of Ogg data.
     // This is held in an ogg_page structure.
     // The process of obtaining this structure is to do the following steps:
     // 1. Call ogg_sync_pageout. This will take any data currently stored
     //    in the ogg_sync_state object and store it in the ogg_page.
     //    It will return a result indicating when the entire pages data
     //    has been read and the ogg_page can be used.
     //    It needs to be called first to initialize buffers.
     //    It gets called repeatedly as we read data from the file.
     // 2. Call ogg_sync_buffer to obtain an area of memory we can
     //    reading data from the file into. We pass the size of the buffer.
     //    This buffer is reused on each call and will be resized if needed
     //    if a larger buffer size is asked for later.
     // 3. Read data from the file into the buffer obtained above.
     // 4. Call ogg_sync_wrote to tell libogg how much data
     //    we copied into the buffer.
     // 5. Resume from the first step, calling ogg_sync_buffer.
     //    This will copy the data from the buffer into the page,
     //    and return '1' if a full page of data is available.

     // About ogg_sync_pageout() :
     //  > This function takes the data stored in the buffer of the
     //  > ogg_sync_state struct and inserts them into an ogg_page.
     //  > Caution: This function should be called BEFORE(!!!) reading
     //  >          into the buffer to ensure that data does not remain
     //  >          in the ogg_sync_state struct.
     //  >          Failing to do so may result in a memory leak.
     //  >          See the example code below for details.
     //  > Return values :
     //  > -1 : stream has not yet captured sync (bytes were skipped).
     //  >  0 : more data needed or an internal error occurred.
     //  >  1 : indicated a page was synced and returned.
     //  > Example Usage
     //  >  if (ogg_sync_pageout(&oy, &og) != 1)
     //  >   {	buffer = ogg_sync_buffer(&oy, 8192);
     //  >  	bytes = fread(buffer, 1, 8192, stdin);
     //  >  	ogg_sync_wrote(&oy, bytes);
     //  >   }
     //
     // Repeat until ogg_sync_pageout indicates a page was synced and returned .
     // Note that ogg_sync_pageout() reads blocks of data from the incoming stream,
     //           which is a bit surprising for a function named '.._pageout' !
     // Call stack: VorbisStream_ReadSampleBlocks() -> ReadAndDecodePages() .
     // Note: Neither 'ogg_sync_pageout' nor 'ogg_sync_buffer' call any file-I/O-function !
     //
     pageout_result = ogg_sync_pageout(&pVH->oy/*"state"*/, &opage);
     // '-->  Return values :
     //    > -1 : stream has not yet captured sync (bytes were skipped).
     //    >  0 : more data needed or an internal error occurred.        [got THIS in the first loop, when no data had been fed in via ogg_sync_wrote()]
     //    >  1 : indicated a page was synced and returned.              [got THIS in the first loop, with a "good" OGG file, w/o leading junk]
     if( pageout_result != 1 )  // "no new page was synced and return" ->
      {
        //  Call ogg_sync_buffer to obtain an area of memory we can
        //    read data from the file into. We pass the size of the buffer.
        //    This buffer is reused on each call and will be resized if needed
        //    if a larger buffer size is asked for later.
        buffer = ogg_sync_buffer(&pVH->oy, 4096);  // <- funny name for a function that ALLOCATES a buffer
        if( buffer==NULL )
         { return 0;
         }
        // Read data from the "file" (in this case, T_VorbisStream.bBuffer[VORBIS_STREAM_BUFFER_MAX_BYTES] )
        nBytesRead = VorbisRead( // << this may already take awfully long, see YHF_Inet.c::INET_ReadDataFromServer( .., pVH->inetTimeout_ms ) !
           buffer,          // ptr to the data that the vorbis files need
           1,               // how big a byte is  ;)   [usually one]
           4096,            // How much we can read (same size as passed to "ogg_sync_page" .. what a name)
           (void *)pVS );   // from where to read (once a 'disk file', but now 'our buffer in RAM')
        if(nBytesRead <= 0)
         { // End of file, or (if it's a stream) must wait a bit longer :
           return nPagesRead;
         }

        // Call ogg_sync_wrote to tell libogg how much data we copied into the buffer.
        ov_result = ogg_sync_wrote(&pVH->oy, nBytesRead);   // -> ov_result = usually ZERO = "no error"
        if( ov_result < 0 )
         { return ov_result;  // negative = error code like OV_ENOTVORBIS, OV_EBADHEADER, OV_EBADPACKET
                              // (not sure if the caller really cares, but anyway..)
         }

      }    // end if( pageout_result != 1 )
     else // ogg_sync_pageout() returned pageout_result=1 : "a new page was synced and returned", so PROCESS IT.
      { // In the "Example Usage" about ogg_sync_pageout(), we're HERE : --,
        //  > Example Usage                                                |
        //  >  if (ogg_sync_pageout(&oy, &og) != 1)                        |
        //  >   { buffer = ogg_sync_buffer(&oy, 8192);                     |
        //  >     bytes = fread(buffer, 1, 8192, stdin);                   |
        //  >     ogg_sync_wrote(&oy, bytes);                              |
        //  >   }                                                          |
        //  >  else  ,-----------------------------------------------------'
        //  >   { // ogg_sync_pageout indicated a page was synced and returned .
        //  >   }
        // Decode those pages, until ogg_sync_pageout says we got them all .
        //  Not sure if there will be more than one page pouring out of the buffer ("state")
        //  here, but doing this repeatedly cannot hurt :
        // Added for safety to avoid crashing in DecodeOggPage() :
        if( opage.header==NULL )
         { // doesn't look like a valid ogg_page - ogg_sync_pageout() failed ?
           // 2023-02-01 : Bailed out here with pVH->dwBufferUsage = nBytesRead = 4096,
           //              fReadSamples = FALSE,
           //              opage.header = NULL (should have been set non-NULL WHERE ? )
           //  buffer = "HTTP/1.0 200 OK\r\nServer: Icecast 2.4.4\r\n ...  " .
           //  Oh well. That's NOT the begin of an OggVorbis file or stream.
           //           That's the response from an HTTP SERVER !
           //  To give some more info about WHY 'Analyse / play audio stream' failed,
           //  Spectrum Lab's webstream control panel now SHOWS THE INITIALLY RECEIVED BYTES,
           //  similar as already implemented FOR OTHER FILE/STREAM TYPES
           //  in TSpectrumLab::OpenInputFileAnalysis() .  Result (with the new "Hex Dump") :
           //  > Could not open "http://5.9.106.210/vlf41" for analysis. Possibly wrong file format ?
           //  >  (OV error: missing headers)
           //  > Dump of the initial bytes below (4096 bytes received, 512 bytes displayed) :
           //  > 0000: 48 54 54 50 2F 31 2E 30 20 32 30 30 20 4F 4B 0D  HTTP/1.0 200 OK.
           //  > 0010: 0A 53 65 72 76 65 72 3A 20 49 63 65 63 61 73 74  .Server: Icecast
           //  > 0020: 20 32 2E 34 2E 34 0D 0A 43 6F 6E 6E 65 63 74 69   2.4.4..Connecti
           //  > 0030: 6F 6E 3A 20 43 6C 6F 73 65 0D 0A 44 61 74 65 3A  on: Close..Date:
           //  > 0040: 20 57 65 64 2C 20 30 31 20 46 65 62 20 32 30 32   Wed, 01 Feb 202
           //  > 0050: 33 20 32 32 3A 35 36 3A 34 34 20 47 4D 54 0D 0A  3 22:56:44 GMT..
           //  > 0060: 43 6F 6E 74 65 6E 74 2D 54 79 70 65 3A 20 61 70  Content-Type: ap
           //  > 0070: 70 6C 69 63 61 74 69 6F 6E 2F 6F 67 67 0D 0A 43  plication/ogg..C
           //  > 0080: 61 63 68 65 2D 43 6F 6E 74 72 6F 6C 3A 20 6E 6F  ache-Control: no
           //  > 0090: 2D 63 61 63 68 65 2C 20 6E 6F 2D 73 74 6F 72 65  -cache, no-store
           //  > 00A0: 0D 0A 45 78 70 69 72 65 73 3A 20 4D 6F 6E 2C 20  ..Expires: Mon,
           //  > 00B0: 32 36 20 4A 75 6C 20 31 39 39 37 20 30 35 3A 30  26 Jul 1997 05:0
           //  > 00C0: 30 3A 30 30 20 47 4D 54 0D 0A 50 72 61 67 6D 61  0:00 GMT..Pragma
           //  > 00D0: 3A 20 6E 6F 2D 63 61 63 68 65 0D 0A 41 63 63 65  : no-cache..Acce
           //  > 00E0: 73 73 2D 43 6F 6E 74 72 6F 6C 2D 41 6C 6C 6F 77  ss-Control-Allow
           //  > 00F0: 2D 4F 72 69 67 69 6E 3A 20 2A 0D 0A 69 63 79 2D  -Origin: *..icy-
           //  > 0100: 6E 61 6D 65 3A 6E 6F 20 6E 61 6D 65 0D 0A 69 63  name:no name..ic
           //  > 0110: 79 2D 70 75 62 3A 30 0D 0A 0D 0A 4F 67 67 53 00  y-pub:0....OggS.   <--- THIS is where the Ogg/Vorbis begins !
           //  > 0120: 02 00 00 00 00 00 00 00 00 BE 0D 0F 75 00 00 00  ............u...
           //  > 0130: 00 A1 61 C1 A9 01 1E 01 76 6F 72 62 69 73 00 00  ..a.....vorbis..
           //  > 0140: 00 00 01 00 7D 00 00 00 00 00 00 B0 30 01 00 00  ....}.......0...
           //  > 0150: 00 00 00 B8 01 4F 67 67 53 00 00 00 00 00 00 00  .....OggS.......
           //  > 0160: 00 00 00 BE 0D 0F 75 01 00 00 00 3B CA 85 CD 0F  ......u....;....
           //  > 0170: 5B FF FF FF FF FF FF FF FF FF FF FF FF FF 91 03  [...............
           //  > 0180: 76 6F 72 62 69 73 34 00 00 00 58 69 70 68 2E 4F  vorbis4...Xiph.O
           //  > 0190: 72 67 20 6C 69 62 56 6F 72 62 69 73 20 49 20 32  rg libVorbis I 2
           //  > 01A0: 30 32 30 30 37 30 34 20 28 52 65 64 75 63 69 6E  0200704 (Reducin
           //  > 01B0: 67 20 45 6E 76 69 72 6F 6E 6D 65 6E 74 29 01 00  g Environment)..
           //  > 01C0: 00 00 13 00 00 00 65 6E 63 6F 64 65 64 20 62 79  ......encoded by
           //  > 01D0: 20 76 74 76 6F 72 62 69 73 01 05 76 6F 72 62 69   vtvorbis..vorbi
           //  > 01E0: 73 26 42 43 56 01 00 08 00 00 80 22 4C 18 C4 80  s&BCV......"L...
           //  > 01F0: D0 90 55 00 00 10 00 00 A0 AC 37 96 7B C8 BD F7  ..U.......7.{...
           //
           // 2023-02-01 : Looking at the above hex dump, shouldn't the decoder
           //   be smart enough to skip trailing junk (here: the HTTP "GET"-header,
           //          up to and including the two CR+NLs near offset 0x0100) ?
           //   The description of ogg_sync_pageout() ...
           //      > Return values :
           //      > -1 : stream has not yet captured sync (bytes were skipped).
           //   ... sounds like it SHOULD BE ABLE to skip trailing junk .
           //
           //
         } // end if( opage.header==NULL )
        else  // opage.header!=NULL -> It's a "good Ogg page", so process (decode) it
         { ++nPagesRead;
           DecodeOggPage( pVH, &opage );
           // 2023-02-04: It took  __3__  decoded Ogg-pages until pVH->fGotAllHeaders was TRUE.
           //             MULTIPLE pages were "emitted" this way (from ogg_sync_pageout());
           //             i.e. all those "required headers" seemed to be contained
           //             in the 4096 bytes, read in the FIRST CALL of VorbisRead().
           //             It's important to NOT call VorbisRead() again, before
           //             *ALL* pages have been pulled out [via ogg_sync_pageout()] !
           // Need even more pages (because we still didn't get what the caller asked for) ?
           //  or return from the page-decoding-loop now (to let ReadSampleBlock() drain fltDecoderOutput[] ?
           // fltDecoderOutput[] should have been filled in handle_vorbis_packet():
           if( pVS->nDecoderOutputBufferUsage > pVS->iDecoderOutputReadIndex ) // anything to process in fltDecoderOutput[] now ?
            { pVS->nDecoderOutputBufferUsage = pVS->nDecoderOutputBufferUsage; // <- place for a breakpoint (2023-02-02, see 2023_02_Vorbis_File_IO_Problems.txt)
              break; // return from this loop; there are at least a few PCM samples ready for processing
            }
         }
        // (2023-02-02: Moved the while( .. ogg_sync_pageout() ) to the BEGIN of the above loop)
      } // end else < ogg_sync_pageout() returned pageout_result=1 >

    ++nLoops;

   } // end while < push more data [samples], and pull more pages from Ogg >


  // EX: AFTER the try-to-read-more-pages-and-analyse-them loop : Decide if it's not Ogg/Vorbis at all
  // if( (!pVH->fGotAllHeaders) && (pVH->dwBufferUsage >= 64/*bytes*/) )
  //  { // Read and analysed a 'fair amount' of bytes, but didn't get all OggVorbis headers yet:
  //  }

  return nPagesRead;
} // end ReadAndDecodePages()


//----------------------------------------------------------------------------------------
void VorbisStream_Init( T_VorbisStream *pVS )  // Only call this ONCE per instance !
  // Formerly a class constructor. Now a simple 'struct initializer'.
  // Should only be called ONCE for a certain instance to prevent memory leaks.
{
  int i;

  VorbisStream_i32DebugDummy = sizeof(T_VorbisStream);
    // '---> 2024-02-24: Result = 1123716 = 0x112584 bytes = ?!

  if( (pVS->dwInitMagicE == VORBIS_MAGIC_E) && (pVS->pVH != NULL) )
   { // VorbisStream_Init() was called even though already initialized ->
     return;  // "do nothing" (important to avoid memory leaks,
              //     because the memset() below would kill all pointers)
     // 2024-02-24: When stepping over this in disassembly view (Borland: "CPU"),
     // saw this :  cmp [eax+0x00112580], 0x27182818 <--- that's VORBIS_MAGIC_E...
     //                      |________|---> is THAT the offset of 'dwInitMagicE'
     //                                     within struct T_VorbisStream ?!         
   }

  memset( pVS, 0, sizeof( T_VorbisStream ) ); // -> pVS->pVH = NULL = NOT ALLOCATED YET

  // Make the few FLOATING POINT numbers in our struct valid:
  pVS->fltSampleRate = 48000.0f;  // initial guess
  for( i=0; i<VORBIS_STREAM_MAX_DECODER_OUTPUT_SIZE; ++i)
   { pVS->fltDecoderOutput[i] = 0.0f; // Hello Mr Pedantic ("armclang"), this 'f' is especially for you !
   }
  pVS->nChannelsPerSample = 1;  // should be properly set before opening such files

  pVS->dwInitMagicE = VORBIS_MAGIC_E; // may trust all struct members now
     // ( VorbisStream_AllocHelper() won't be called before we NEED it)


} // end VorbisStream_Init()

//---------------------------------------------------------------------------
void VorbisStream_Exit( T_VorbisStream *pVS )
  // Closes, cleans up, and frees everything that may have been allocated.
{
  if( (pVS->dwInitMagicE == VORBIS_MAGIC_E) && (pVS->pVH != NULL) )
   { VorbisStream_FreeHelper( pVS );
     // -> calls whatever it needs to (in the Ogg- and Vorbis library modules)
     //    to "destroy the structures using the appropriate vorbis_*_clear functions",
     //    then finally sets pVS->pVH to NULL .
   }
  pVS->dwInitMagicE = 0; // the magic smoke has escaped, we're out of business
} // end VorbisStream_Exit()



  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  //  Ogg/Vorbis ENCODER ("compress once, send to multiple receivers")
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


//----------------------------------------------------------------------------------------
static BOOL WriteOggPage( T_VorbisStream *pVS, ogg_page *pOggPage )
  // Writes an Ogg page (usually with Vorbis data) to what
  // may eventually become a file or stream .
  //
  //  Calling sequence: FIRST THREE CALLS from VorbisStream_InitEncoder(),
  //                    to retrieve the "initial headers" (that EVERY RECEIVER needs,
  //                    no matter how late he starts, and regardless of how many
  //                    'Ogg pages' have been sent out to others in the meantime) :
  //   1.) with pOggPage->header="OggS", header_len=28, body="\x01vorbis..", body_len=30,
  //            pVS->dwPageIndex = 0 -> 1
  //   2.) with pOggPage->header="OggS", header_len=38..43, body="\x03vorbis*.", body_len=2562..3857
  //            pVS->dwPageIndex = 1 -> 2
  //  [3.) with pOggPage->header="OggS", header_len=57, body=<binary stuff>, body_len=4308
  //            pVS->dwPageIndex = 2 -> 2 ] ... not ALWAYS the case,
  //                  so don't assume anything about the NUMBER OF OGG PAGES
  //                  delivered by VorbisStream_GetInitialHeadersFromEncoder() !
  //
  //  Call stack (after initialisation, when really 'encoding AUDIO', not the initial three pages):
  //   e.g. CwNet_Server_UpdateVorbisStreamIfNecessary() -> VorbisStream_EncodeSamples()
  //     -> WriteOggPage() with ....
  //   1.) pVS->dwPageIndex = 2   [MANY SECONDS after beginning to feed audio into the encoder]
  //             pOggPage->header="OggS", header_len=282, body="\0\0\0\0\0..", body_len=255,
  //   2.) pVS->dwPageIndex = 3   [again MANY SECONDS later.. ?!]
  //             pOggPage->header="OggS", header_len=282, body="\0\0\0\0\0..", body_len=255,

{
  BOOL fResult = FALSE;
  int i,j, nBytesForHeaderAndBody;
  DWORD dw;
  T_VorbisHelper *pVH = pVS->pVH;
  if( pVH!=NULL && pOggPage!=NULL )
   { nBytesForHeaderAndBody = pOggPage->header_len + pOggPage->body_len;

     // ex: pVH->nBitrateCounter += 8 * nBytesForHeaderAndBody;
     ++pVS->dwPageIndex; // unique Ogg Page number, here incremented in WriteOggPage()
                         // (rolls over from 0xFFFFFFFF to zero after a LONG time)

     // store this Ogg-page in the buffer for the 'initial header' sent to each new remote client ?
     if( pVS->fStoreInitialHeaders )
      {
        if( (pVS->nInitialHeaderBufferedBytes + nBytesForHeaderAndBody ) < VORBIS_INITIAL_HEADER_BUFFER_SIZE )
         { dw = pVS->nInitialHeaderBufferedBytes;
           memcpy( pVS->bInitialHeaderBuffer + dw, pOggPage->header, pOggPage->header_len );
           dw += pOggPage->header_len;
           memcpy( pVS->bInitialHeaderBuffer + dw, pOggPage->body,   pOggPage->body_len );
           dw += pOggPage->body_len;
           pVS->nInitialHeaderBufferedBytes = dw;  // THIS will make the "initial header" available...
           // ... for the 'multiple readers', which will possibly access the buffer IN ANOTHER THREAD(!) .
         }
        // 2024-02-10: When pVS->fStoreInitialHeaders was cleared in VorbisStream_InitEncoder(),
        //             pVS->nInitialHeaderBufferedBytes was 2658 bytes .
        //
      } // end if < store initial headers in bInitialHeaderBuffer[] >
     else // the 'initial headers' have been placed in a persistent buffer, but what about the 'AUDIO SAMPLES' ?
      { // Store the last few seconds of compressed audio in a circular buffer (FIFO-like),
        // from where they can be "pulled out" by MULTIPLE READERS .
        // Details about the one-writer, multiple-reader buffer in VorbisStream_GetNextPagesFromEncoder() :
        // >            _____________ ___________ ________ _______ ________
        // > bBuffer : | Ogg-page[n] | page[n-4] | [n-3]  | [n-1] | unused |
        // >           |_____________|___________|________|_______|________|
        // >                         |           |        |       |
        // >                     dwBufferIndex   |        |    dwBufferUsage  (from writer)
        // >          /|\                        |         to be read from the buffer
        // >           |           circular wrap                  |
        // >           '--------------<<<<<-----------------------'
        // >
        // Because especially for protocols like UDP (but also for TCP),
        // a single Ogg page shall be send 'in a single over' (if possible),
        // thus the FIFO buffer must be aware of Ogg page boundaries,
        // and an Ogg page must not cross the wrapping-point (at the upper end of the buffer).
        // So will the new Ogg page fit in the buffer without wrapping ?
        if( (pVS->dwBufferIndex + nBytesForHeaderAndBody ) >= VORBIS_STREAM_BUFFER_MAX_BYTES ) // circular buffer wrap ?
         { // pVS->dwBufferIndex + nBytesForHeaderAndBody would exceed bBuffer[VORBIS_STREAM_BUFFER_MAX_BYTES] :
           // buffer-head-index wraps around, but dwBufferUsage indicates the "current end" for all readers:
           pVS->dwBufferUsage = pVS->dwBufferIndex;
           pVS->dwBufferIndex = 0;  // index into pVS->bBuffer[] (head)
           // pVH->iBufferPagesHead = the index into BufferPages[] wraps around, too !
           // Make the end of the 'Ogg page directory' invalid (=the "unused" rest in the diagram above):
           pVS->nBufferPagesUsed = pVS->iBufferPagesHead; // only BufferPages[0..nBufferPagesUsed-1]
                       // may be read in VorbisStream_GetNextPagesFromEncoder() !
           pVS->iBufferPagesHead = 0;
         } // end if < circular buffer wrap > ?

        if( pVS->iBufferPagesHead >= VORBIS_STREAM_BUFFER_MAX_PAGES ) // for safety only:
         { // (this will rarely ever happen because a 64 kByte buffer should be large enough
           //  to keep more than a few dozen Ogg pages in pVH->bBuffer[],
           //  and VORBIS_STREAM_BUFFER_MAX_PAGES (64?) is larger than we ever need)
           pVS->nBufferPagesUsed = pVS->iBufferPagesHead; // only BufferPages[0..nBufferPagesUsed-1]
                        // may be read in VorbisStream_GetNextPagesFromEncoder()
           pVS->iBufferPagesHead = 0;
         }

        // Make sure bBuffer[0..VORBIS_STREAM_BUFFER_MAX_BYTES-1] is not exceeded:
        if( (pVS->dwBufferIndex + nBytesForHeaderAndBody ) < VORBIS_STREAM_BUFFER_MAX_BYTES )
         { dw = pVS->dwBufferIndex;
           memcpy( pVS->bBuffer + dw, pOggPage->header,  pOggPage->header_len );
           dw += pOggPage->header_len;
           memcpy( pVS->bBuffer + dw, pOggPage->body,    pOggPage->body_len );
           dw += pOggPage->body_len;
           pVS->dwNumTotalBytesProcessed += (pOggPage->header_len + pOggPage->body_len);
           // Also update the "directory" of the buffered Ogg pages:
           pVS->BufferPages[ pVS->iBufferPagesHead ].iBufIndex = pVS->dwBufferIndex; // index into bBuffer[] where this page begins
           pVS->BufferPages[ pVS->iBufferPagesHead ].iSize_Byte= nBytesForHeaderAndBody;
           pVS->BufferPages[ pVS->iBufferPagesHead ].dwPageIndex = pVS->dwPageIndex; // unique index (or "counter") for this Ogg page
           ++pVS->iBufferPagesHead;  // new 'head index' for the Ogg page directory (pVH->BufferPages[])
           // >  BufferPages[] shall only contain info about those Ogg pages
           // >  which are still available in pVH->bBuffer[] !
           if( pVS->nBufferPagesUsed < pVS->iBufferPagesHead )
            {  pVS->nBufferPagesUsed = pVS->iBufferPagesHead;
            }
           pVS->dwBufferIndex = dw;  // THIS will make the new Ogg page available...
           // ...for the multiple readers, which will possibly read the FIFO
           // in ANOTHER TREAD. No problem because the buffer is large enough
           // so none of the multiple readers will still need the data
           // which have just been overwritten by the memcpy()s above.
           // When a NEW READER attaches, VorbisFileIO will set 'his' stream-sample-position
           // to the NEWEST (! - not the OLDEST - !) part of the buffer for that reason.
           // See details in VorbisStream_GetNextPagesFromEncoder() .
         }

      } // end else < ! pVH->fStoreInitialHeaders >

   } // end if ( pVH != NULL )

  return fResult;
} // end WriteOggPage()


//---------------------------------------------------------------------------
BOOL VorbisStream_InitEncoder( // Prepare ENCODING Ogg/Vorbis files or streams
        T_VorbisStream *pVS,
        float fltSampleRate,    // number of sample points per second
        int nChannelsPerSample, // number of audio channels PER SAMPLE POINT
        float base_quality) // Desired quality level, currently from -0.1 to 1.0 (lo to hi).
  // Returns TRUE on success and FALSE on any error.
  // [in] base_quality:  Desired quality level, currently from -0.1 to 1.0
  //                     (lo to hi). A good starting point seems to be 0.5  .
  // [in] nChannelsPerSample:  1=mono, 2=stereo
  // [in] fltSampleRate: number of sample points per second.
  //                     Caution - Ogg Vorbis only supports a limited set
  //                     of sample rates, officially ranging from 8 to 192 kHz.
  // For details, see the Ogg Vorbis / "libvorbisenc" documentation .
  // Don't miss http://www.xiph.org/vorbis/doc/vorbisenc/examples.html
  //      and   http://www.xiph.org/vorbis/doc/vorbisenc/overview.html
{
  // Some experiences by Paul Nicholson, 2010-06-02 :
  // > PS, I switched to using variable bitrate encoding which
  // > appears to remove the artifacts I noticed, and also reduces
  // > the CPU load considerably - now about the same as libmp3lame.
  // >
  // > Was calling:
  // >    vorbis_encode_init( &vi, chans, sample_rate, -1, bps, -1)
  // >
  // > Now calls instead:
  // >    Q = 0.5; // quality factor, -0.1 (low) to 1.0 (high)
  // >    vorbis_encode_init_vbr( &vi, chans, sample_rate, Q)
  // >
  // > See http://www.xiph.org/vorbis/doc/vorbisenc/vorbis_encode_init_vbr.html
  // >
  // > Seems like the encoder had to work extra hard to produce the
  // > constant bitrate stream. With reasonable amount of buffer,
  // > VBR should be no problem at all, since we don't have to run
  // > in 'real time' or worry about latency.
  // >
  // > Currently encoding a 5-channel stream at 32k frames/sec,
  // > uses 13% of 2GHz CPU.
  // ex: vorbis_info vi;  // moved into a static structure (see notes below)
  int v_result;    // vorbis result code / error code; 0 = "no error"
  int n;  // number of bytes, or result code used in YHF_inet.h (def'd in c:\cbproj\SoundUtl\ErrorCodes.h)
  int coupling_mode;
  // BOOL ok = FALSE;
  ogg_page opage; // type ogg_page only contains a POINTER to the page's netto data,
                  // it's a small object so it's ok to use a local stack var here.
  ogg_packet sr_header;

  T_VorbisHelper *pVH = VorbisStream_AllocHelper( pVS );
  if( pVH == NULL )
   { return FALSE; // pVS->sz255ErrorString already set !
   }

  pVS->nChannelsPerSample = nChannelsPerSample; // save for later ..
  pVS->fltSampleRate = fltSampleRate;
  pVS->nSamplesPerHeaderInStream = 0; // use the same size as the feeding-chunk-size, as soon as we know it
  pVS->iSampleCountdownBetweenHeaders = 0; // here in OutOpen() : emit the initial header or VT_BLOCK a.s.a.p. !
  pVS->dwNumSamplesWritten = 0;

  pVS->dwCurrFilePos = pVS->dwFileSizeInBytes = 0;


  vorbis_info_init(&pVH->vinfo);

  // Workflow (using libvorbisenc, quoted from doc/vorbisenc/overview.html
  // > Libvorbisenc is used only during encoder setup; its function is
  // > to automate initialization of a multitude of settings in a
  // > vorbis_info structure which libvorbis then uses as a reference
  // > during the encoding process.
  // > Libvorbisenc plays no part in the encoding process after setup.
  // >
  // > Encode setup using libvorbisenc consists of three steps:
  // >
  // >  1. high-level initialization of a vorbis_info structure
  // >     by calling one of vorbis_encode_setup_vbr() [....]
  // >  2. optional adjustment of the basic setup defaults
  // >     using vorbis_encode_ctl()
  // >  3. calling vorbis_encode_setup_init() to finalize the high-level setup
  // >     into the detailed low-level reference values needed by libvorbis
  // >     to encode audio. The vorbis_info structure is then ready to use
  // >     for encoding by libvorbis.
  //      (so better do NOT use a local 'stack' variable for vorbis_info - WB)
  // > These three steps can be collapsed into a single call by using
  // > vorbis_encode_init_vbr  to set up a quality-based VBR stream [....] .
  // The latter is used here, for an easy start :

  // Prepare encoding using a VBR quality mode.  The usable range is -.1
  // (lowest quality, smallest file) to 1.0 (highest quality, largest file).
  // Example quality mode .4: 44kHz stereo coupled, roughly 128kbps VBR
  // EX: v_result = vorbis_encode_init_vbr( &pVH->vinfo, // [in] vorbis_info *vi,
  //                    channels, nominal_sample_rate, base_quality);
  // Removed 2011-12-21 because vorbis_encode_init_vbr() internally calls
  //         vorbis_encode_setup_init(), and after that,
  //         vorbis_encode_ctl() is impossible. Use ..._.._setup_vbr() instead:
  v_result = vorbis_encode_setup_vbr( &pVH->vinfo, // [in] vorbis_info *vi,
                nChannelsPerSample, fltSampleRate, base_quality);

  // Do not continue if setup failed; this can happen if we ask for a
  // mode that libVorbis does not support (eg, too low a quality mode, etc,
  // will return 'OV_EIMPL')
  if( v_result != 0 )
   { sprintf( pVS->sz255ErrorString,"vorbis_encode_setup_vbr(): %s",
                                     VorbisErrorCodeToString( v_result ) );
     return FALSE;
   }

  // Call vorbis_encode_ctl(), described as follows:
  // > This function implements a generic interface to miscellaneous
  // > encoder settings similar to the clasasic UNIX 'ioctl()' system call.
  // With 'number' (2nd argument) = OV_ECTL_COUPLING_SET :
  // >  Argument: int *
  // >  Enables/disables channel coupling in multichannel encoding according to arg.
  // >  *arg of zero disables all channel coupling, nonzero allows the encoder
  // >  to use coupling if a coupled mode is available for the input.
  // >  At present, coupling is available for stereo and 5.1 input modes.
  if( nChannelsPerSample >= 2 )
   { if( vorbis_encode_ctl( &pVH->vinfo, OV_ECTL_COUPLING_GET, &coupling_mode) < 0)
      { strcpy( pVS->sz255ErrorString,"vorbis_encode_ctr(OV_ECTL_COUPLING_GET) failed" );
        return FALSE;
      }
     if( coupling_mode != 0 )  // we want INDEPENDENT encoding of the channels, so:
      {
        coupling_mode = 0;
        if( vorbis_encode_ctl( &pVH->vinfo, OV_ECTL_COUPLING_SET, &coupling_mode) < 0)
         { strcpy( pVS->sz255ErrorString, "OV_ECTL_COUPLING_SET failed" );
           return FALSE;
           // 2011-12-21: 'OV_ECTL_COUPLING_SET' failed because some odd flag
           //             named 'set_in_stone' was already set in vorbis_encode_ctl().
           //  Reason: vorbis_encode_setup_init() had already been called,
           //          INTERNALLY, camouflaged in vorbis_encode_init_vbr() !
         }
      }
   }

  // > vorbis_encode_setup_init() : performs the last stage of three-step encoding setup,
  // > as described in the API overview under managed bitrate modes.
  // > Before this function is called, the vorbis_info struct should be initialized
  // > by using vorbis_info_init() from the libvorbis API, (....) .
  // > vorbis_encode_setup_init() finalizes the highlevel encoding structure
  // > into a complete encoding setup after which the application may make no further setup changes.
  // > After encoding, vorbis_info_clear should be called.
  if( (v_result=vorbis_encode_setup_init( &pVH->vinfo)) < 0)
   { sprintf( pVS->sz255ErrorString,"VorbisEncodeSetup failed: %s",
                                     VorbisErrorCodeToString( v_result ) );
     return FALSE;
   }

  // Setup a stream encoder ("stream 0" for vorbis) :
  pVS->fStoreInitialHeaders = TRUE;  // store the initial headers for remote clients which may connect the server LATER
  pVS->nInitialHeaderBufferedBytes = 0;  // nothing cached yet

  // Setup one or two stream encoders, stream 0 for vorbis, stream 1 not used here :)
  if( (v_result=ogg_stream_init( &pVH->osv, 0/*serialno*/) ) < 0)
   { sprintf( pVS->sz255ErrorString,"Can't init ogg-stream #1 (audio): %s",
                                     VorbisErrorCodeToString( v_result ) );
     return FALSE;
   }

  // The following code was originally based on
  //     libvorbisXYZ/examples/encoder_example.c ,
  // > Add a (vorbis-) comment :
  vorbis_comment_init(&pVH->vcomment);
  vorbis_comment_add_tag(&pVH->vcomment,"ENCODER","VorbisStream.cpp");

  // > set up the analysis state and auxiliary encoding storage
  vorbis_analysis_init(&pVH->vdsp,&pVH->vinfo);
  vorbis_block_init(&pVH->vdsp, &pVH->vblock); // don't forget vorbis_block_clear( &pVH->vblock ) later !

  // > Vorbis streams begin with three headers:
  // >   (1) the initial header (with most of the codec setup parameters)
  // >       which is mandated by the Ogg bitstream spec.
  // >   (2) The second header holds any comment fields.
  // >   (3) The third header holds the bitstream codebook.
  // > We merely need to make the headers, then pass them to libvorbis
  // > one at a time; libvorbis handles the additional Ogg bitstream constraints .
  // About vorbis_analysis_headerout :
  // > This function creates and returns the three header packets
  // > needed to configure a decoder to accept compressed data.
  // > Should be called after all encoder initialization and configuration is complete.
  // > The output packets should be placed in order at the start of the
  // > compressed vorbis stream, prior to the first data packet.
  vorbis_analysis_headerout(
        &pVH->vdsp,    // [in] vorbis_dsp_state with the encoder configuration
        &pVH->vcomment,// [in] vorbis_comment structure with the metadata
                        //      associated with the stream being encoded
        &pVH->header,      // [out] ogg_packet to be filled out with the stream ID header
        &pVH->header_comm, // [out] ogg_packet to be filled out with the serialied vorbis_comment data
        &pVH->header_code);// [out] ogg_packet to be filled out with the codebooks,
                        //       mode descriptions, etc.
                        //       which will be used encoding the stream
  ogg_stream_packetin(&pVH->osv,&pVH->header); /* automatically placed in its own page */
  ogg_stream_packetin(&pVH->osv,&pVH->header_comm);
  ogg_stream_packetin(&pVH->osv,&pVH->header_code);
  // About ogg_stream_packetin() :
  // > This function submits a packet to the bitstream for page encapsulation.
  // > After this is called, more packets can be submitted, or pages can be written out.
  // > In a typical encoding situation, this should be used after filling
  // > a packet with data. The data in the packet is copied into the internal
  // > storage managed by the ogg_stream_state, so the caller is free
  // > to alter the contents of op after this call has returned. (..)

  // > ogg_stream_flush() checks for remaining packets inside the stream
  // > and forces remaining packets into a page, regardless of the size of the page.
  // > ! This should only be used when you want to flush an undersized page
  // > ! from the middle of the stream.
  // ( well, not really. In libvorbis_1_3_1/examples/encoder_example.c ,
  //   ogg_stream_flush() is also called to flush the above 'three headers' !
  while(1)
   { v_result = ogg_stream_flush( &pVH->osv/*in*/, &opage/*out*/);
     if( v_result==0 ) // no more 'flushed' pages to write at the moment..
      { break;
      }
     WriteOggPage( pVS, &opage ); // append to data in our "initial header buffer"
   }

  // 2024-02-14 : At this point, pVS->dwPageIndex was only TWO, not THREE. Why ?
  //              Shouldn't there be THREE Ogg-"pages" for THREE Ogg-"packets"
  //                            with THREE Vorbis-"headers" ? Confusing !

  pVS->fStoreInitialHeaders = FALSE;  // finished storing the initial headers
      // for multiple stream receivers (and/or the "file writer").
      // Clear fStoreInitialHeaders to let WriteOggPage() know that
      // whatever is emitted NEXT will not go into pVS->bInitialHeaderBuffer[]
      // but into pVS->bBuffer[] .
      // It's completely up to the caller ("application") about how and when
      // to SEND or WRITE pVS->bInitialHeaderBuffer[] (possibly NOW already)
      // or pVS->bBuffer[] (not possible NOW but later, as Ogg pages with
      // 'dynamic' content have been emitted).
      // Since C++Builder (V6) didn't have uVision's glorious 'Memory Window',
      // here's how to inpect the "initial header buffer" with BCB V6:
      //  Step 1: Add a "watch" for pVS->bInitialHeaderBuffer , display format = "Pointer".
      //  Step 2: Open Borland's "CPU"-window (!), maximize the hex-dump-like
      //          anonymous stuff in the lower left part of the window,
      //          right-click, select "Go To Address", and enter the hex address
      //          shown in the "watch" window (e.g. pVS->bInitialHeaderBuffer = 0x0076EE10):
      // 0076EE10: 4F 67 67 53 00 02 00 00  "OggS...."
      // 0076EE18: 00 00 00 00 00 00 00 00  "........"
      // 0076EE20: 00 00 00 00 00 00 2E CD  "........"
      // 0076EE28: 0C 18 01 1E 01 76 6F 72  ".....vor"
      // 0076EE30: 62 69 73 00 00 00 00 01  "bis....."
      // 0076EE38: 40 1F 00 00 00 00 00 00  "@......."
      // 0076EE40: 70 62 00 00 00 00 00 00  "pb......"
      // 0076EE48: 99 01 4F 67 67 53 00 00  "~.OggS.."
      // 0076EE50: 00 00 00 00 00 00 00 00  "........"
      // 0076EE58: 00 00 00 00 01 00 00 00  "........"
      // 0076EE60: BE 54 59 EF 0B 56 FF FF  "~~~~.~~~"
      //    ...
      // 0076EEA8: 00 18 00 00 00 45 4E 43  ".....ENC"
      // 0076EEB0: 4F 44 45 52 3D 56 6F 72  "ODER=Vor"
      // 0076EEB8: 62 69 73 53 73 72 65 61  "bisStrea"  (a-ha, here is the "header_comm")
      // 0076EEC0: 6D 2E 63 70 70 01 05 76  "m.cpp..v"
      //    ...
      // The rest looked like "binary stuff with high entropy", possibly the
      // codebook, with NO THIRD "OggS" in pVS->bInitialHeaderBuffer .

  return (pVS->nInitialHeaderBufferedBytes > 0); // -> TRUE when successful
            //  '--> e.g. 2658 bytes
            // -> Thus VORBIS_INITIAL_HEADER_BUFFER_SIZE = at least 4096 bytes .
            //    Similar requirement for the deeply embedded WEB SERVER !

} // end VorbisStream_InitEncoder()


//----------------------------------------------------------------------------------------
long VorbisStream_EncodeSamples( // .. into VORBIS audio, in OGG "pages" ...
        T_VorbisStream *pVS,
        float *pfltSrc,         // [in] caller's source buffer, 32-bit floats, possibly interleaved
        int nChannelsPerSample, // [in] number of audio channels PER SAMPLE POINT in pfltSrc
        int nSamplePoints )     // [in] number of sample points in pfltSrc
  // Writes some samples in the 'normalized' floating point format (-1.0 ... +1.0)
  //       to an audio file which has been opened for WRITING;
  //       or to an OUTGOING audio stream .
  // Returns the number of SAMPLE POINTS successfully encoded,
  //       or a NEGATIVE value if there was an error .
{
  T_VorbisHelper *pVH = pVS->pVH;
  int v_result;    // vorbis result code / error code; 0 = "no error"
  int i, n;
  int iChannel;
  int nDstChannelsPerSample, nDstBytesPerSample, nCommonChannelsPerSample;
  int iSample, nSamplesInVorbisBlock;
  double d;
  char sz80[84];
  float *pfltSource2;
  float **ppfltBuffer;
  float *pfltDst;
  long nSamplesWritten = 0;
  ogg_page opage; // ogg_page only contains a POINTER to the page's netto data,
                  // it's a small object so it's ok to use a local stack var here.
  ogg_packet opack;

  memset( &opage, 0, sizeof(opage) ); // don't leave anything to fate - clear garbage
  memset( &opack, 0, sizeof(opack) );

  if( (pVH==NULL) || (pVS->dwInitMagicE!=VORBIS_MAGIC_E) )
   { return -1; // failed to allocate the internal T_VorbisHelper ..
   }

  nDstChannelsPerSample = pVS->nChannelsPerSample;
  nCommonChannelsPerSample = nChannelsPerSample; // who knows.. caller may deliver less than initially expected
  if( nCommonChannelsPerSample > nDstChannelsPerSample )
   {  nCommonChannelsPerSample = nDstChannelsPerSample;
   }


  // Push the samples into Vorbis' own buffer (later used as input for encoding)
  while( nSamplePoints > 0 )
   { nSamplesInVorbisBlock = (nSamplePoints>1024) ? 1024 : nSamplePoints;
     nSamplePoints -= nSamplesInVorbisBlock;

     ppfltBuffer = vorbis_analysis_buffer( &pVH->vdsp, nSamplesInVorbisBlock );
     // About vorbis_analysis_buffer() :
     // > requests a buffer array for delivering audio to the encoder for compression.
     // > The Vorbis encoder expects the caller to write audio data
     // > as non-interleaved floating point samples into its internal buffers.
     // > The general procedure is to call this function with
     // > the number of samples you have available ("1024 is a reasonable choice").
     // > The encoder will arrange for that much internal storage
     // > and return an array of buffer pointers, one for each channel of audio.
     // > The caller must then write the audio samples into those buffers,
     // > as float values, and finally call vorbis_analysis_wrote()
     // > to tell the encoder the data is available for analysis.
     // > Returns a **float where the first index is the channel,
     // >                    and the second is the sample index.
     iChannel=0;
     while( iChannel<nCommonChannelsPerSample )
      { if( (pfltDst = ppfltBuffer[iChannel]) != NULL )
         { pfltSource2 = pfltSrc + iChannel;  // source: INTERLEAVED, float
           for( iSample=0; iSample<nSamplesInVorbisBlock; ++iSample )
            { *pfltDst++ = *pfltSource2;  // destination: non interleaved
              pfltSource2 += nChannelsPerSample;
            }
         }
        ++iChannel;
      }
     // If the caller now delivers less channels than specified in OutOpen(),
     //  fill the remaining channels with silence :
     while( iChannel<nDstChannelsPerSample )
      { if( (pfltDst=ppfltBuffer[iChannel]) != NULL )
         { for( iSample=0; iSample<nSamplesInVorbisBlock; ++iSample )
            { *pfltDst++ = 0.0f;
            }
         }
        ++iChannel;
      }
     // Inform the Vorbis encoder that we have finished writing into the buffer:
     v_result = vorbis_analysis_wrote( &pVH->vdsp, nSamplesInVorbisBlock );
     if( v_result<0 ) // failure !
      { nSamplesWritten = 0;
        sprintf( pVS->sz255ErrorString,
                 "Cannot send: %s", (char*)VorbisErrorCodeToString( v_result ) );
        break;
      }
     else // ok, the Vorbis encoder has 'accepted' the buffered data
      { nSamplesWritten += nSamplesInVorbisBlock;
        pVS->dwNumSamplesWritten += nSamplesInVorbisBlock;
        pVH->dwNumSamplesWrittenAfterLastOggPage += nSamplesInVorbisBlock;
      }

     // prepare the (interleaved) source-pointer for the SAMPLE POINT (with <nChannelsPerSample>):
     pfltSrc += nSamplesInVorbisBlock * nChannelsPerSample;
   } // end while( nSamplePoints > 0 )

  // Encode what we can (in the Vorbis audio file or stream) .
  //    About vorbis_analysis_blockout :
  // > examines the available uncompressed data and tries to break it
  // > into appropriate sized blocks. It should be called in a loop
  // > after adding new data with vorbis_analysis_buffer()/vorbis_analysis_wrote()
  // > until it returns zero (need more data) or an negative value (error).
  // > Each block returned should be passed to vorbis_analysis() for transform and coding.
  // > Returns 1 for "success, more blocks are available",
  // >         0 for "success, this is the last block from the current input",
  // >         negative values for failure  (but nobody seems to care)
  while( vorbis_analysis_blockout( &pVH->vdsp, &pVH->vblock) == 1)
   {
     v_result = vorbis_analysis( &pVH->vblock, NULL);  // 2011-12-22: Crashed with access violation (in analysis.c -> psy.c; floor[0] corrupted)
     // -> Once the uncompressed audio data has been divided into blocks,
     //  > vorbis_analysis() is called on each block. It looks up
     //  > the encoding mode and dispatches the block to the
     //  > forward transform provided by that mode. (..)
#ifdef __BORLANDC__  // "... is assigned a value that is never used".. shut up ..
     (void)v_result;
#endif

     v_result = vorbis_bitrate_addblock( &pVH->vblock);
     // -> ..addblock() submits a transformed block to the bitrate management engine
     //  > for final encoding. Packets are buffered and the packet boundaries
     //  > adjusted and padded to meet the target bitrate, if any.
     //  > After calling vorbis_bitrate_addblock(), the passed vorbis_block
     //  > structure can be reused in another call to vorbis_analysis_blockout().
     //  > Call vorbis_bitrate_flushpacket() to obtain the final compressed data.
     (void)v_result;
     while( ( v_result = vorbis_bitrate_flushpacket( &pVH->vdsp, &opack )) > 0 )
      {
        // -> ..flushpacket() returns the next available completed packet from the
        //  > bitrate management engine. It should be called in a loop
        //  > after any call to vorbis_bitrate_addblock() until it returns
        //  > either 0 (more data needed) or a negative value (error).
        //  > The data returned in the ogg_packet structure can be copied
        //  > to the final compressed output stream.

        ogg_stream_packetin( &pVH->osv, &opack);
        // About ogg_stream_packetin() :
        // > This function submits a packet to the bitstream for page encapsulation.
        // > After this is called, more packets can be submitted, or pages can be written out.
        // > In a typical encoding situation, this should be used after filling
        // > a packet with data. The data in the packet is copied into the internal
        // > storage managed by the ogg_stream_state, so the caller is free
        // > to alter the contents of op after this call has returned. (..)

        // write out pages (if any)
        while( ogg_stream_pageout( &pVH->osv, &opage) )
         { WriteOggPage( pVS, &opage );
           pVH->dwNumSamplesWrittenAfterLastOggPage = 0; // no need to flush an Ogg page immediately after writing !
         }

      } // end while( .. vorbis_bitrate_flushpacket() .. "should be called in a loop".... )

   } // end while( vorbis_analysis_blockout() .. )

# define SAMPLES_PER_OGG_PAGE 4000
# if( SAMPLES_PER_OGG_PAGE > 0 )   // try to LIMIT the size of an Ogg page (trying to reduce stream latency) ?
  if( pVH->dwNumSamplesWrittenAfterLastOggPage >= SAMPLES_PER_OGG_PAGE ) // try to send SMALLER pages to reduce stream latency
   { v_result = ogg_stream_flush( &pVH->osv/*in*/, &opage/*out*/);
     // > ogg_stream_flush() checks for remaining packets inside the stream
     // > and forces remaining packets into a page, regardless of the size of the page.
     // > ! This should only be used when you want to flush an undersized page
     // > ! from the middle of the stream.
     // > If there are no packets or partial packets to flush,
     // > ogg_stream_flush returns 0.
     if( v_result > 0 ) // have 'flushed' a page -> "write" it to the output (stream buffer)
      { WriteOggPage( pVS, &opage ); // emit our 'shorter page' to the stream buffer
        pVH->dwNumSamplesWrittenAfterLastOggPage = 0; // no need to flush again (without 'X' more AUDIO SAMPLES fed in)
      }
     // Test results: ( "<audio>" latency guesstimated from Firefox built-in player)
     // f_samp | quality | SAMPLES_PER_OGG_PAGE | avrg bytes/page | <audio> latency
     // -------+---------+----------------------+-----------------+----------------
     //  8 kHz |   0.8   | 0 (no forced flush)  |     4230        |   21 seconds
     //  8 kHz |   0.8   | 8000 ( 1  second )   |     1670        |   20 seconds
     //  8 kHz |   0.8   | 4000 (0.5 seconds)   |      900        |   20 seconds
     //  8 kHz |   0.8   | 2000 (0.25 seconds)  |   413 .. 551    |   19 seconds


   } // end if( pVH->dwNumSamplesWrittenAfterLastOggPage >= SAMPLES_PER_OGG_PAGE )
# endif // SAMPLES_PER_OGG_PAGE > 0 ?

  return nSamplesWritten;

} // end VorbisStream_EncodeSamples


//----------------------------------------------------------------------------------------
int VorbisStream_GetLengthOfInitialHeadersFromEncoder( T_VorbisStream *pVS )
  // Delivers the same NUMBER OF BYTES that VorbisStream_GetInitialHeadersFromEncoder()
  // would deliver.
  // Designed to to make life of the HTTP server easier, because
  // for "Transfer-Encoding: chunked", the server must "print" the chunk's
  // content-length IN HEXADECIMAL STRING FORM before the chunk (here: Ogg/Vorbis).
  // Details in HttpSrv_StartStreamingLiveAudio() .
{
  if( pVS->dwInitMagicE == VORBIS_MAGIC_E )   // open for business ?
   { return pVS->nInitialHeaderBufferedBytes; // returns the number of bytes actually placed in pbDest
   }
  else
   { return 0; // Cannot send any Ogg pages with the "initial Vorbis headers" yet
   }
} // end VorbisStream_GetLengthOfInitialHeadersFromEncoder()


//----------------------------------------------------------------------------------------
int VorbisStream_GetInitialHeadersFromEncoder(
        T_VorbisStream *pVS,
        BYTE *pbDest,   int iMaxDestLength, // [out] network buffer, with limited length
        DWORD *pdwPageIndex ) // [out] indicator for the next call of
                              //       of VorbisStream_GetNextPagesFromEncoder()
  // Used to implement a multi-receiver audio server.
  //   "Compress once" (via WriteSamples_Float),
  //   "ready multiple" (via GetInitialHeaders..() + VorbisStream_GetNextPagesFromEncoder() ).
  //
  // Returns the number of bytes actually placed in pbDest .
  //
  // Details: Added 2014-04 to support MULTIPLE STREAM CLIENTS ('receivers')
  //               fed by a SINGLE STREAM SERVER ('transmitter')
  //               which only holds the latest few seconds (actually, Ogg pages)
  //               in memory, and a buffer with the three mandatory headers
  //               which will be sent first when a client connects
  //               to our audio-streaming-capable HTTP server.
{
  if( pVS->dwInitMagicE == VORBIS_MAGIC_E )   // open for business ?
   {
     if( pdwPageIndex != NULL )
      { *pdwPageIndex = pVS->dwPageIndex; // begin reading samples at the current OGG PAGE WITH AUDIO(!) that becomes available
        //    ,--------------|_________|
        //    '--> has been PRE-INCREMENTED in WriteOggPage() when emitting the last page.
        //         (we don't want to send "more backlog" to a remote client,
        //          but we also don't want to let him wait for the NEXT (future)
        //          Ogg page that crawls out of the encoder so very slowly...
        //          ... with often less than 0.5 pages per second on average)
      }
     // The following header-block has already been prepared ONCE,
     // when the Ogg/Vorbis encoder was initialized (OutOpen)
     // and all 'initial' headers have been produced (EncoderInit2) :
     if(   pVS->nInitialHeaderBufferedBytes > 0
       &&  pVS->nInitialHeaderBufferedBytes <= iMaxDestLength )
      { memcpy( pbDest, pVS->bInitialHeaderBuffer, pVS->nInitialHeaderBufferedBytes );
        return pVS->nInitialHeaderBufferedBytes; // returns the number of bytes actually placed in pbDest
      }
   }
  return 0;  // "nothing copied"
} // end VorbisStream_GetInitialHeadersFromEncoder()


//----------------------------------------------------------------------------------------
int VorbisStream_GetNumPagesAvailableForStreamOutput(
        T_VorbisStream *pVS,
        DWORD dwPageIndex) // [in] our "Ogg Page counter" for the next call of VorbisStream_GetNextPagesFromEncoder().
                           //      For each of the remote audio receivers,
                           //      the "sender" (e.g. the HTTP server) keeps
                           //      track of OGG PAGES, not VORBIS PACKETS !
  // Implemented for the 'audio stream server statistics' in AudioStreamServer_GUI.h .
{
  int iPage, nPages = 0;
  if( pVS->dwInitMagicE!=VORBIS_MAGIC_E )
   {
     // Similar loop as in VorbisStream_GetNextPagesFromEncoder() :
     for(iPage=0; iPage<pVS->nBufferPagesUsed; ++iPage)
      { if( (pVS->BufferPages[iPage].dwPageIndex >= dwPageIndex ) // bingo, the requested page is still available (in the buffer) ..
         && (pVS->BufferPages[iPage].iSize_Byte > 0 ) )
         { ++nPages;  // count the number of pages which THIS reader would be able to send now
         }
      }
   }
  return nPages;
} // end VorbisStream_GetNumPagesAvailableForStreamOutput()


//----------------------------------------------------------------------------
int VorbisStream_GetNextPagesFromEncoder(
        T_VorbisStream *pVS,
        BYTE *pbDest, int iMaxDestLength,
        DWORD *pdwPageIndex) // [in,out] indicator for the next call
  // Retrieves as many samples as possible (limited to complete Ogg pages)
  //           for one of many possible STREAM READERS .
  // First used to implement a multi-stream-server in Spectrum Lab. Principle:
  //    "one writer/compressor"  (in WriteSamples_Float),
  //    "multiple readers" (GetInitialHeadersForStreamOutputServer
  //                      + VorbisStream_GetNextPagesFromEncoder() ).
  // Returns the number of bytes actually placed in pbDest .
  //
{
  DWORD dwPageIndex = 0;
  int   iPage,page_size,page_buf_idx, nBytesRead=0, nPagesRead=0;
  BOOL  page_ok;

  (void)dwPageIndex; // "assigned a value that is never used" (shut up..)

  if( pVS->dwInitMagicE == VORBIS_MAGIC_E )
   {
     if( pdwPageIndex != NULL )
      { dwPageIndex = *pdwPageIndex; // for Ogg/Vorbis encoded streams, this is actually the Ogg Page counter
                                     // which started at zero when the 'COMPRESSOR' was initialized .
        // Because the first THREE Ogg pages are the 'constant INITIAL headers'
        // that every stream receiver will need (delivered by VorbisStream_GetInitialHeadersFromEncoder() ),
        // the FIRST PAGE WITH REAL "AUDIO" will have dwPageIndex >= 3 .
      }
     else // Ogg page index not specified -> wait for the next page which becomes available
      { dwPageIndex = pVS->dwPageIndex;  // <- this is the Ogg-page-number from
      }

     // Is the reader's "sample position" (= the FIFO TAIL INDEX) still valid,
     // i.e. does dwPageIndex still point to a valid Ogg page boundary
     // (or to the header of an un-compressed stream) ?
     page_ok = FALSE;
     for(iPage=0; iPage<pVS->nBufferPagesUsed; ++iPage)
      { if( (pVS->BufferPages[iPage].dwPageIndex == dwPageIndex ) // bingo, the requested page is still available (in the buffer) ..
         && (pVS->BufferPages[iPage].iSize_Byte > 0 ) )
         { page_ok = TRUE;
           break;  // which means all 'later' pages (if any, in the buffer) are also still valid
         }
      }
     if( !page_ok )
      { //  Most likely reason: the reader (web server?) didn't call
        //  VorbisStream_GetNextPagesFromEncoder() frequently enough.
        //  'His' buffer-tail-index will be reset to the new buffer-head-index.
        if( dwPageIndex == (pVS->dwPageIndex+1) )
         {   // reader must wait for the NEXT OGG PAGE. Leave dwPageIndex unchanged.
         }
        else // something wrong, discard this caller's Ogg page counter (he will lose a few pages!)
         { dwPageIndex = pVS->dwPageIndex;  // <<< set breakpoint here
         }
      }
     else // page_ok :
      {   // pVH->BufferPages[iPage] is the next page to be copied to pbDest,
          // and there is AT LEAST ONE more complete Ogg page available for this reader.
          // pVH->dwBufferIndex is the index of the next byte WRITTEN INTO the buffer (in WriteOggPage),
          // it also marks the CURRENT END-STOP for all readers.
          //  Example : THREE Ogg-pages currently available, each 4000 bytes long; [in reality, an Ogg page was 4..5 kByte large]
          //            reader starts with pdwPageIndex = 0
          //
          //     pdwPageIndex=0  ..4000    ..8000     pVH->dwBufferIndex=1200
          //           |         |         |         |
          //           |         |         |         |
          //          \|/_______\|/_______\|/_______\|/_______________
          // .bBuffer | page[0] | page[1] | page[2] |not available yet|
          //          |_________|_________|_________|_________________|
          //                                         filled in future calls
          //                                         of WriteOggPage
          //
        // Repeat this for as many Ogg pages as possible :
        while( (page_size=pVS->BufferPages[iPage].iSize_Byte) <= iMaxDestLength)
         { page_buf_idx = pVS->BufferPages[iPage].iBufIndex;
           if( page_size<=0 || page_buf_idx<0 )  // page invalid ?
            { break;
            }
           if( pVS->BufferPages[iPage].dwPageIndex!=dwPageIndex ) // wrong page (doesn't belong to the sequence,
            { break;       // most likely it's an OLDER page in the circular FIFO)
            }
           memcpy( pbDest, &pVS->bBuffer[page_buf_idx], page_size );
           pbDest      += page_size;
           iMaxDestLength -= page_size;
           nBytesRead  += page_size;
           ++iPage;
           if( iPage >= pVS->nBufferPagesUsed )
            {  iPage = 0;
            }
           ++nPagesRead;
           ++dwPageIndex;  // the caller's (reader's) Ogg page counter
           (void)nPagesRead;
           (void)dwPageIndex; // "assigned a value that is never used" (shut up, we used it FOR TESTING)
         }
      } // end else < pdwPageIndex points to the begin of a valid page in the buffer >
     if( pdwPageIndex != NULL )
      { *pdwPageIndex = dwPageIndex;
      }
   }
  return nBytesRead;
} // end VorbisStream_GetNextPagesFromEncoder()


  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  //  Ogg/Vorbis DECODER API (to read/play Ogg audio files or a received stream)
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

//---------------------------------------------------------------------------
BOOL VorbisStream_InitDecoder( T_VorbisStream *pVS )
  // Prepares both Ogg and Vorbis for DECODING a stream or file.
  // Returns TRUE on success and FALSE on any error .
  // Details in VorbisStream.h, "Ogg/Vorbis DECODER API" (1) .
{
  T_VorbisHelper *pVH = VorbisStream_AllocHelper( pVS );
  if( pVH == NULL )
   { return FALSE;
   }

  // Formerly used ov_open_callbacks() for this purpose.
  // Replaced 2011-12-14 with direct use of the Libvorbis API, because:
  // > Libvorbis is the reference implementation of the Vorbis codec.
  // > It is the lowest-level interface to the Vorbis encoder and decoder,
  // > working with packets directly.
  //
  //
  // A nicely explained, example for an Ogg/Vorbis decoder
  // using libogg was found at
  //  www.bluishcoder.co.nz/2009/06/24/reading-ogg-files-using-libogg.html
  //
  // The first thing we need to do when reading an Ogg file is find
  // the first page of data. We use an ogg_sync_state structure
  // to keep track of search for the page data. This needs to be
  // initialized with ogg_sync_init and later cleaned up with ogg_sync_clear:
  if(ogg_sync_init( &pVH->oy )!=0) // init our instance of 'ogg_sync_state'
   { strcpy( pVS->sz255ErrorString, "error in ogg_sync_init" );
     return FALSE;
   }
  return TRUE;
  // ALL THE REST (e.g. the internal calls of ReadAndDecodePages() )
  // cannot take place in VorbisStream_InitDecoder() because
  // we don't use a "pull" model (with direct access to a file or network socket)
  // but a "push" model (where THE APPLICATION repeatedly PUSHES IN new data
  // into the decoder as soon as it has them).
  // Thus, as explained in the
  //      "Short overview / most important functions for AUDIO STREAMING"
  // in VorbisStream.h, the next step (2) happens in VorbisStream_DecodeBytes() !

} // end VorbisStream_InitDecoder()


//---------------------------------------------------------------------------
int VorbisStream_DecodeBytes( T_VorbisStream *pVS, // actually "buffers" FOR THE DECODER..
        BYTE *pbSource, int nBytes )
  // Pushes a few more bytes into the decoder (here: into the BUFFER for decoding).
  // Returns the NUMBER OF BYTES that could actually be "pushed" into a buffer.
  // Details in VorbisStream.h, "Ogg/Vorbis DECODER API" (2) .
{
  // The Ogg/Vorbis decoder wants to PULL bytes from what it thinks
  // is a FILE. Thus, all we can do HERE is append the caller's
  // block of bytes (which is hopefully much smaller than
  // VORBIS_STREAM_BUFFER_MAX_BYTES) into pVS->bBuffer[] .
  // LATER, when VorbisStream_ReadSamples() is called to actually "decode samples",
  // VorbisRead() will drain pVS->bBuffer[] again.
  //  [out] pVS->bBuffer[VORBIS_STREAM_BUFFER_MAX_BYTES]
  //  [in]  pVS->dwBufferUsage : 0 ... VORBIS_STREAM_BUFFER_MAX_BYTES
  //            = number of bytes that MUST REMAIN AVAILABLE in the buffer.
  //  [in]  pVS->dwBufferIndex : 0 ... dwBufferUsage-1 (steps by one for each byte
  //                                consumed from the buffer in VorbisRead() )
  int nBytesNotConsumedYet = pVS->dwBufferUsage - pVS->dwBufferIndex;
  if( nBytesNotConsumedYet < 0 ) // oops.. "buffer-READ-index" has exceeded the "buffer usage" ?!
   {  nBytesNotConsumedYet = 0;  // assume the buffer is completely empty
   }
  if((nBytesNotConsumedYet > 0 ) && (pVS->dwBufferIndex > 0) )
   { // move whatever has "not been consumed" by VorbisRead() yet
     // to the BEGIN of our buffer (pVS->bBuffer[0..nBytesNotConsumedYet-1]) :
     memmove( pVS->bBuffer,                      // destination
              pVS->bBuffer + pVS->dwBufferIndex, // source
              nBytesNotConsumedYet ); // number of BYTES(!) to copy
   }
  pVS->dwBufferIndex = 0;  // new SOURCE BYTE INDEX for VorbisRead()
  pVS->dwBufferUsage = (DWORD)nBytesNotConsumedYet;
  if( (nBytes+nBytesNotConsumedYet) <= VORBIS_STREAM_BUFFER_MAX_BYTES ) // ok..
   { memmove( pVS->bBuffer + nBytesNotConsumedYet, // destination
              pbSource, // source
              nBytes ); // number of "NEW BYTES"(!) to append to the buffer feeding the DEODER
     pVS->dwBufferUsage = (DWORD)(nBytes+nBytesNotConsumedYet);
     return nBytes;     // ok, pushed all the bytes that the caller wants us to "decode"
   }
  else
   { return 0; // error; please read the notes in VorbisStream.h, "Ogg/Vorbis DECODER API",
               // about how to avoid this problem (call VorbisStream_ReadSamples().. )
   }
} // end VorbisStream_DecodeBytes()

//----------------------------------------------------------------------------
long VorbisStream_ReadSamples( T_VorbisStream *pVS,
        int nChannelsWanted, // number of channels 'wanted'
                             // (the stream may be 'stereo' but we only want 'mono')
        int nSamplesWanted,  // number of samples 'wanted' (per destination block, length of destination arrays)
        float *pfltDest)     // destination block (decoded samples)
  // Reads some samples from an audio file which has been opened for READING
  // and converts them to floating point values .
  // Details in VorbisStream.h, "Ogg/Vorbis DECODER API" (3) .
  // Return value: Number of samples ACTUALLY PLACED in *pfltDest++
{
  int  n, n2, nSamplePointsReturned;
  // long nBytesReadFromDecoder;
  // int  current_section;
  int  ov_result;      // vorbis result code (when negative, it's an error code)
  int  nPagesDecoded;
  int  iWatchdogCount=2000;   // shouldn't be necessary (but prevents endless loops)
  T_VorbisHelper* pVH = pVS->pVH;
  struct VT_OGG_BLK* pTS;
  long double ldblTimestamp;  // beware of stupid Microsoft compilers:
                              // we need at least 80 bit precision here !
  double dblSrCal; int tbreak; // other results from tq_get()
  double d;
  long i32Counter;
  // char sz80[84];
  // LONGLONG i64TimeToLeave, i64Tnow, i64Frequency; // removed because this turned debugging into a nightmare

  if( pVH==NULL ) // no valid 'file helper' allocated !
   { return -1;
   }

  pVS->nChannelsPerSampleRequested = nChannelsWanted;

  nSamplePointsReturned = 0;


  while( (nSamplePointsReturned<nSamplesWanted)
      && (pVS->nChannelsPerSample>0)   // <- prevent div-by-zero below
      && (iWatchdogCount > 0)      )   // prevents crashing for some other reason
   {
     // As long as there are still some samples waiting in fltDecoderOutput[],
     //  take the samples from there, instead of reading the next Ogg page.
     if ( pVS->iDecoderOutputReadIndex < pVS->nDecoderOutputBufferUsage )  // still something left in pVH->fltDecoderOutput[] ?
      { // there's at least ONE sample remaining in pVH->fltDecoderOutput[]..
        while( ( nSamplePointsReturned < nSamplesWanted )
            && ( pVS->iDecoderOutputReadIndex < pVS->nDecoderOutputBufferUsage )
            && ( pVS->iDecoderOutputReadIndex < VORBIS_STREAM_MAX_DECODER_OUTPUT_SIZE ) )
         { *pfltDest++ = pVS->fltDecoderOutput[ pVS->iDecoderOutputReadIndex ];
           //
           // If the caller expects to read a 2nd channel : ...
           if( nChannelsWanted >= 2 )
            { if( pVS->nChannelsPerSample  >= 2 )
               { *pfltDest++ = pVS->fltDecoderOutput[ pVS->iDecoderOutputReadIndex+1 ];
               }
              else
               { *pfltDest++ = 0.0;
               }
            }
           pVS->iDecoderOutputReadIndex += pVS->nChannelsPerSample;
           ++nSamplePointsReturned;
         }
      } // end if(pVH->iDecoderOutputReadIndex<pVH->nDecoderOutputBufferUsage)

     // Request new PCM samples from the vorbis decoder
     // if our internal buffer (pVH->fltDecoderOutput[]) is SUFFICIENTLY EMPTY
     //           (not "completely empty", as it used to be before 2022-03-12).
     n = pVS->nDecoderOutputBufferUsage - pVS->iDecoderOutputReadIndex;
         // -> n = number of single floats STILL WAITING in pVH->fltDecoderOutput[]
     n2 = (VORBIS_STREAM_MAX_DECODER_OUTPUT_SIZE-n)
         / pVS->nChannelsPerSample; // -> #sample points FREE in pVH->fltDecoderOutput[]
     if( n2 >= 16 )
      { // Can 'accept' at least 16 more SAMPLE POINTS, i.e. pVH->fltDecoderOutput[] is 'sufficiently empty' ->
        // Try to read the next block of samples from the file or stream to keep the 'network buffer' usage as low as possible.
        if( n <= 0 ) // pVH->fltDecoderOutput[] COMPLETELY empty ?
         {
           pVS->iDecoderOutputReadIndex   = 0;  // no CPU-hogging "memmove" required to "scroll up" the buffer
           pVS->nDecoderOutputBufferUsage = 0;
         }
        else  // n>0 : Some sample-points remaining in pVH->fltDecoderOutput[] ->
         { // Scroll up the remaining buffer content so the OLDEST sample
           // will be at pVH->fltDecoderOutput[0] :
           memmove( &pVS->fltDecoderOutput[0], // destination
                    &pVS->fltDecoderOutput[pVS->iDecoderOutputReadIndex], // source
                    n * sizeof(float) );        // number of BYTES(!) to copy
           pVS->iDecoderOutputReadIndex   = 0;
           pVS->nDecoderOutputBufferUsage = n; // number of SINGLE FLOATS (not "sample points") remaining in pVH->fltDecoderOutput[]
               //  '--> this is also the buffer index where handle_vorbis_packet() will APPEND the next sample(s),
               //       deep inside the following call of ReadAndDecodePages() ...
         }
        // ----- read COMPRESSED Ogg/Vorbis stream (here: from our internal buffer) ---->
        nPagesDecoded = ReadAndDecodePages( pVS );
                          // [in]  pVS->iDecoderOutputReadIndex,
                          // [out] pVS->nDecoderOutputBufferUsage, pVH->fltDecoderOutput[]
        // ReadAndDecodePages() -> .. -> handle_vorbis_packet() should have filled up
        //  as many samples as possible in pVH->fltDecoderOutput[],
        //     incremented pVH->nDecoderOutputBufferUsage,
        //       but left  pVH->iDecoderOutputReadIndex unchanged !
        // Typical increments seen in pVH->nDecoderOutputBufferUsage when STEPPING OVER ReadAndDecodePages() :
        //   0 -> 19584 (first call when playing "Sarabande", that's 19584/2 = 9792 SAMPLE POINTS (a 2 channels)
        //   0 -> 20480 (second call of ReadAndDecodePages(), but still in the 1st call of ReadSampleBlocks(nSamplesWanted=16384), nSamplePointsReturned now 9792)
        //   7296 -> 7296 (third call of ReadAndDecodePages(), nPagesDecoded=1, but NO NEW SAMPLE in pVH->fltDecoderOutput[])
        //               First call of ReadSampleBlocks() returned with:
        //                     nSamplePointsReturned = nSamplesWanted = 16384 (ok),
        //                     pVH->iDecoderOutputReadIndex = 0,
        //                     pVH->nDecoderOutputBufferUsage = 7296 .
        // SECOND call of ReadSampleBlocks() :
        //   nSamplePointsReturned | nDecoderOutputBufferUsage |
        //      3648               |   0 ->  20480             |    (1st call of ReadAndDecodePages() )
        //      13888              |   0 ->  20480             |    (2nd call of ReadAndDecodePages() )
        //      16384              |   15488 ->  20480         |    (3rd call of ReadAndDecodePages() )
        //
        // THIRD call of ReadSampleBlocks() :
        //   nSamplePointsReturned | nDecoderOutputBufferUsage |
        //      7744               |   0 ->  20480             |    (1st call of ReadAndDecodePages() )
        //      16484              |   3200 ->  3200           |    (2nd call of ReadAndDecodePages(), ZERO PAGES decoded !)
        //
        //
        //
        //
        if( nPagesDecoded <= 0 )
         { // "error" or "nothing available at the moment" ?
           break;
         }
      } // end  if( m_dwDecoderOutputByteIndex >= m_dwDecoderOutputNumBytes )
     else  // there are still some bytes waiting in m_c4kDecoderOutput[]
      {    // which have not been processed yet : leave this loop !
        break;
      }
     --iWatchdogCount;
   } // end  while( (nSamplePointsReturned<nSamplesWanted) && ... )

  // Return THE NUMBER OF SAMPLE-POINTS read into the caller's buffer(s) :
  return nSamplePointsReturned;

} // VorbisStream_ReadSampleBlocks(..)



// About floating point exceptions... WB, 2020-10-11 :
// From http://www.virtualdub.org/blog/pivot/entry.php?id=53 :
//
// > In general, you don't want to be tripping floating-point exceptions,
// > even if they are masked. The reason is that when the FPU hits one,
// > the fast hardware can't handle it and punts to the microcode, which then
// > takes about twenty times longer. This is especially bad with NaNs since
// > any operation with a NaN produces another NaN, causing them to spread
// > throughout your calculations (NaN disease) and slow down everything massively.
// > You can even crash due to NaNs blowing past clamp expressions,
// > since any comparison with a NaN is false and converting one to integer
// > form results in integer indefinite (0x80000000).
// > Despite the erroneous results, though, NaNs can appear sporadically
// > in a large Win32 program (such as ours, WB..) without anyone knowing,
// > and may go unnoticed in a code base for years.
// >
// > How Borland C/C++ factors into the picture
// >
// > The Borland DLL run-time library, as it turns out, enables floating-point
// > exceptions on initialization. This happens even if you simply load the DLL!
// > Because Windows programs generally don't touch the floating-point control word,
// > the effects of this can persist long after the DLL has been unloaded.
// > For instance, you could:
// >   * launch an Open file dialog in a program and hover over an AVI file in the list,
// >     ...thus causing Explorer to load its shell media extension
// >        to retrieve the file's properties for the tooltip,
// >     ...causing a codec search for the video stream in the file,
// >     ...thus loading and unloading the Borland-compiled DLL.
// > It is possible to disable this behavior of the Borland run-time library
// > and avoid this problem, but most people aren't aware of it,
// > and unintentionally release DLLs that cause this issue. (....)
// > The best way to fix it is to not modify the control word, but I don't know
// > if that is possible; barring that, a usable workaround is to remask
// > the exceptions with _controlfp(), as noted at
// >   http://homepages.borland.com/ccalvert/TechPapers/FloatingPoint.html .
//     (of course that page doesn't exist anymore, but hold on..)
// > Sadly, strictly speaking you cannot simply recover by remasking
// > the FPU exceptions and restarting the faulting instruction.
// > The problem is that the x87 FPU doesn't signal the interrupt
// > until the next floating-point instruction, at which point
// > necessary information to retry the instruction is irretrievably lost.
// > Take this instruction sequence for example:
// >   FDIV DWORD PTR [EAX]
// >   XOR  EAX, EAX
// >   FSTP DWORD PTR [EDX]
// > A divide-by-zero error here will actually result in the FSTP instruction
// > faulting, not the FDIV.


/* EOF <VorbisStream.c> */



