//---------------------------------------------------------------------------
// File: C:\cbproj\Remote_CW_Keyer\Utilities.c
//   (stuff collected from other projects, e.g. DL4YHF's Spectrum Lab)
//
// Most recent modifications:
//   2025-04-15 : Removed the RichEdit_-functions from Remote_CW_Keyer/Utilities.c,
//                because those functions already exist in YHF_WinStuff.c
//                (since two decades), and THAT module has been added to the
//                Remote CW Keyer application. So, now
//                 RichEdit_SetSelBgColor(), RichEdit_SetSelColors(), etc,
//                are now exclusively implemented in
//                  C:\cbproj\YHF_Tools\YHF_WinStuff.c .
//
//   2025-01-02 : Drilled up UTL_FormatDateAndTime() for the DAY-OF-WEEK .
//
//   2024-11-26: Got most modules to compile with
//         "Embarcadero C++Builder V12 Community Edition" instead of the old
//         "Borland C++Builder V6", but then hell broke lose again:
//       > [ilink32 Error] Fatal: Illegal VIRDEF fixup index in module
//       >          '..\Utilities.c'
//                   (simlar also in 'dsound_wrapper.c'; details THERE.).
//
//   2024-11-25: Replaced HUNDREDS of "char pointers" by "const char *"
//               because 'modern compilers' (e.g. "ISO C++11") are
//               very pedantic about char pointers, and refuse to
//               use STRING LITERALS as a "char *" !
//         After that, most of the non-VCL-modules could be compiled
//         with the old Borland C++Builder V6 as well as with
//         Embarcadero C++Builder V12 "Community Edition".
//               Details about the migration process from BCB V6 to V12
//               in DL4YHF's C:\cbproj\YHF_Tools\StringLib.h .
//
//---------------------------------------------------------------------------

#if(defined(__clang__))  // "__clang__" seemed to indicate 'C++ Builder 12'..
# define _TEST_VIRDEF_TROUBLE_ 1 // omit anything that may have caused the
       //  "Illegal VIRDEF fixup index" in C++Builder 12 ?
       //   1=yes (at the risk of generating LINKABLE, but NON-FUNCTIONAL CODE),
       //   0=no=normal compilation
#else  // no "Clang-enhanced compiler", so guess this is old Borland C++Builder:
# define _TEST_VIRDEF_TROUBLE_ 0 // don't omit anything to avoid the mysterious 'VIRDEF fixup trouble'
#endif


//---------------------------------------------------------------------------
#include <string.h>
#include <stdio.h>     // no 'standard I/O' here, but using sprintf() ..
#include <math.h>
#include <windows.h>

#if(1) // (! _TEST_VIRDEF_TROUBLE_ ) // maybe the inclusion of the headers below
                  // already causes the "Illegal VIRDEF fixup index" .. ?
     // (YES. The inclusion of one of these obscure header files (*) seemed to
     //  cause Embarcadero's "Illegal VIRDEF fixup index" (LINKER trouble).
     // At least the following message DISAPPEARED when *not* #including <initguid.h>:
     // > [ilink32 Error] Fatal: Illegal VIRDEF fixup index in module '..\Utilities.c'
     //
# include <richedit.h>  // RichEdit, part of windoze, no 'special' DLL
# include <commctrl.h>  // Common Controls, part of windoze, no 'special' DLL
# if(defined(__clang__)) // "__clang__" seemed to indicate 'C++ Builder 12'..
//#  define INITGUID     // <- from "initguid.h", before "#include guiddef.h"
//#  include <guiddef_replacement_for_CBuilder12.h>  // replacement for "initguid.h" without "Illegal VIRDEF fixup index" ?
# else // !defined __clang__  -> guess we're compiling with the ancient Borland C++ Builder V6
//#  include <initguid.h>  // EVIL for C++Builder V12(!), but formerly required to retrieve a COM port's "friendly name"
# endif // def'd __clang__ ?
# include <devguid.h>   // required to retrieve a COM port's "friendly name" ?
# include <setupapi.h>  // required to retrieve a COM port's "friendly name" ?
# include <io.h>        // to write the optional logfile ..
# include <fcntl.h>     // it's a mystery why we need another stupid header for O_RDWR (!)

   // (*) tried to find out what's so evil about <iniguid.h>, located in
   // c:\program files (x86)\embarcadero\studio\23.0\include\windows\sdk\initguid.h .
   //
#endif // ! _TEST_VIRDEF_TROUBLE_ ?

# pragma hdrstop

#include "switches.h"  // Application-specific "compilation switches",
                       //  e.g. SWI_APP_EXE_NAME to detect UTL_iAppInstance.
#include "StringLib.h" // "Old school C string library" for embedded systems, etc
#include "Utilities.h" // verify prototypes implemented in THIS module

//-------------- Internal constants and look-up tables ----------------------
const char *UTL_months1[12]=
   { "JAN","FEB","MAR","APR","MAY","JUN","JUL","AUG","SEP","OCT","NOV","DEC" };
const char *UTL_months2[12]=
   { "Jan","Feb","Mar","Apr","May","Jun","Jul","Aug","Sep","Oct","Nov","Dec" };
const char *UTL_weekdays[7]  = { "Mo", "Tu", "We", "Th", "Fr", "Sa", "Su" };
const char *UTL_wochentage[7]= { "Mo", "Di", "Mi", "Do", "Fr", "Sa", "So" };


const T_SL_TokenList WindowsProgramFolderNames[] =
{ { "Program Files",       1  },
  { "Program Files (x86)", 2  },
  { "Programme",           3  },
  { "Programme (x86)",     4  },
  { "Programmi",           5  },
  { "Programmi (x86)",     6  },
  { NULL, 0 } // "all zeros" mark the end of the list
};


//-------------- Old-fashioned global variables -----------------------------
//   (use with discretion; don't wrap each and everything into C++ classes)
// Initialized ONLY to prevent chaos if the application
// "forgets" to call UTL_Init() before any other function of this module.
int UTL_iDesktopXMin = 0;    // typically something like ZERO
int UTL_iDesktopXMax = 1920; // typically something like 1920
int UTL_iDesktopYMin = 0;    // typically something like ZERO
int UTL_iDesktopYMax = 1080; // typically something like 1080
int UTL_iWindowsVersion = 8; // keep it simple: 0=95/98, 1=XP, 2="Vista" (uurgh..),
                             //  7=WIN7, 8=WIN8, 10=WIN10, .. ?

static DWORD  UTL_dwInitMagicPi = 0;
#define C_UTL_INIT_MAGIC 0x31415926
static HANDLE UTL_hInstanceDetectionMutex = NULL;
int  UTL_iAppInstance = -1;  // zero-based application "instance index".  Possible values:
             // -1 = "Shame on you, haven't called UTL_Init() yet",
             //  0 = "I am the FIRST running instance of this program",
             //  1 = "I am the SECOND running instance of this program", etc ..

char UTL_sz255LastError[256]; // only for debugging .. contains the last error
             // in human readable form after e.g. complex functions like
             // UTL_EnumerateComPorts() experienced unexpected trouble
             // with OS function calls [for example, being unable to extract
             // a COM port's FRIENDLY NAME, which occasionally happened,
             // depending on the Windows version and who-knows-what-else].
char       UTL_sz255LogFile[256] = "";  // name optional name of LOGFILE (""=don't TRY TO log)
static int UTL_iLogFileNrOfErrors = 0;  // Don't let the logfile get too large
BOOL       UTL_fRunLogOpened = FALSE;   // TRUE when a 'run-log' shall be written
CRITICAL_SECTION UTL_csRunLog; // same here: UTL_WriteRunLogEntry() should be thread-safe

static LONGLONG UTL_i64HighResTimerOffset = 0;
static LONGLONG UTL_i64HighResTimerFreq   = 0;


//-------------- Internal function prototypes -------------------------------
BOOL CALLBACK UTL_MonitorEnumProc( HMONITOR hMonitor, HDC hdcMonitor,
                                   LPRECT lprcMonitor, LPARAM dwData );
static void UTL_InitCRC32(void); // must be called ONCE, *before* UTL_CRC32() !


//-------------- Implementation of functions --------------------------------

//---------------------------------------------------------------------------
void UTL_Init(void)
  // Initializes the module. Call 'as early as you can'.
{
  HWND hwndDesktop;
  RECT rc;
  OSVERSIONINFO windoze_version_info;
  char sz80[84];

  if( UTL_dwInitMagicPi == C_UTL_INIT_MAGIC ) // don't run the code below TWICE !
   { return;
   }
  UTL_dwInitMagicPi = C_UTL_INIT_MAGIC;
  memset( UTL_sz255LastError, 0, sizeof(UTL_sz255LastError) );

  UTL_InitCRC32(); // initialize a look-up table for faster 32-bit CRC calculation

  CoInitialize(0);  // Init COM (formerly done in VARIOUS other modules).
      // 2012-08-09 : Moved this obscure CoInitialize/CoUninitialize-stuff
      //              HERE, to ensure this ugly stuff is called exactly ONCE.


  InitializeCriticalSection( &UTL_csRunLog );


  // Get the desktop rectangle (to center the splash window)
  hwndDesktop = GetDesktopWindow();
  GetWindowRect(hwndDesktop, &rc);
  // For dual-monitor systems,  see MSDN on "The Virtual Screen",
  //  and google for EnumDisplayMonitors() ...
  UTL_iDesktopXMin = rc.left;   // typically something like ZERO
  UTL_iDesktopXMax = rc.right;  // typically something like 1023
  UTL_iDesktopYMin = rc.top;    // typically something like ZERO
  UTL_iDesktopYMax = rc.bottom; // typically something like 767
  // > To enumerate the devices on the desktop <faselblah>,
  // > call EnumDisplayMonitors. This returns the HMONITOR handle to each
  // > monitor, which is used with GetMonitorInfo. To enumerate all the devices
  // > in the virtual screen, use EnumDisplayMonitors...
  EnumDisplayMonitors(NULL, NULL, UTL_MonitorEnumProc, 0);
  // After this UTL_iDesktopXXXX should contain the "total" desktop area,
  // possibly spread over various monitors ! (for example, UTL_iDesktopXMax=2560=2*1280)
  // Note that EnumDisplayMonitors also seems to use the damned "bounding rectangle",
  // which means that any pixel with x=UTL_iDesktopXMax is *invisible* , etc ...
  --UTL_iDesktopXMax; // to defeat the "bounding rectangle"-crap mentioned above
  --UTL_iDesktopYMax; // result: often 799 (=correct value for a display height of 800 pixels)

#if(! _TEST_VIRDEF_TROUBLE_ ) // maybe the stuff below caused the linker's "VIRDEF fixup" trouble ?
  // Find out the windoze version (MAJOR version, at least)
  // because of a zillion of subtle and annoying differences
  //  (especially between XP and "Vista"-yucc-makes me vomit- and Windows 7)
  // We don't want to have dozens of GetVersionEx-calls all over the place.
  windoze_version_info.dwOSVersionInfoSize = sizeof( OSVERSIONINFO );
  if( GetVersionEx( &windoze_version_info ) )  // BCB V12, Microsoft: "deprecated". WB: "Oh Shut up!"
   {
     switch( windoze_version_info.dwMajorVersion )
      { case 5: // Win2000, Win XP, etc..  , but definitely NOT win98:
           UTL_iWindowsVersion = 1;
           break;
        case 6: // "Major Version 6" doesn't mean a lot..
           switch( windoze_version_info.dwMinorVersion ) // .. got to look at THIS !
            { case 0 : // "Vista" .. crap, we're going to crash very soon
                 UTL_iWindowsVersion = 2; // (worse than even "Window 3.0")
                 break;
              case 1 : // "Version 6.1" is "WIN7", decent Multimedia API..
                 UTL_iWindowsVersion = 7;
                 break;
              default: // "Version 6.2 ... 6.3" was "WIN8", may still be around
                 UTL_iWindowsVersion = 8;
                 break;
            }
           break;
        case 10: // "Major Version 10" may actually be "WIN10" or "WIN11" ..
           UTL_iWindowsVersion = 10; // We don't care.
           // Instead of the nerve-wrecking Window 11, try Linux / WINE :o)
           break;
        default: // anything else ? Expect the worst; don't assume anything.
           UTL_iWindowsVersion = 0;  // hallo, "alte Labormhre !"
           break;
       }
   }
#endif // ! _TEST_VIRDEF_TROUBLE_ ?

#if(! _TEST_VIRDEF_TROUBLE_ ) // maybe the stuff below caused the linker's "VIRDEF fixup" trouble ?
  // Detect the "instance index" of the currently running instance.
  // Some applications used UTL_iAppInstance to run WITH DIFFERENT
  // CONFIGURATIONS, managed internally.
  // For that purpose, UTL_iAppInstance could be used
  // as a simple ARRAY INDEX (running from 0 to UTL_MAX_APP_INSTANCES).
  UTL_iAppInstance = 0;
  while(UTL_iAppInstance<UTL_MAX_APP_INSTANCES) // look for mutexes of already running instances..
   { sprintf( sz80, "%s%d", SWI_APP_EXE_NAME, (int)(UTL_iAppInstance) ); // create a unique, global Mutex name
#   ifdef __clang__ // guess we're not compiling with an old BORLAND compiler but e.g. with BCB V12 ->
     UTL_hInstanceDetectionMutex = CreateMutex( NULL, FALSE, sz80 );
#   else // __clang__ is undefined -> guess we're compiling with old Borland C++ V6 :
     UTL_hInstanceDetectionMutex = ::CreateMutex( NULL, FALSE, sz80 );
#   endif // "Clang" / "Codegear" / "Embarcadero" or "Borland" ?
     if( GetLastError()==ERROR_ALREADY_EXISTS )
      { // the mutex for this instance already exists :
        //   FORGET the handle (it's not ours!) and try the next mutex in the next loop:
        if(UTL_hInstanceDetectionMutex!=NULL)
         { CloseHandle(UTL_hInstanceDetectionMutex); // closes THIS HANDLE to the mutex object,
           // but doesn't remove the mutex itself, because (from Win32 API):
           // > The mutex object is destroyed when its LAST handle has been closed.
         }
        ++UTL_iAppInstance;  // 1 = "I AM THE *SECOND* INSTANCE" (array index!!), etc
        // '--> This "automatically detected instance number" may be overridden
        //      via command line argument as explained in the manual,
        //      SpecLab/html/startins.htm#multiple_instances (e.g. "/inst=5").
      }
     else
      { // Mutex did NOT exist: break loop; APPL_iInstanceNr is valid ,
        // and leave "our" instance-detection-mutex as it is.
        // It will be removed when THIS instance terminates.
        break;  // found an "unoccupied" instance (-mutex)
      }
   } // end while < loop to detect the instance number, using MUTEXES >
#endif // ! _TEST_VIRDEF_TROUBLE_ ?


} // end UTL_Init()

//---------------------------------------------------------------------------
void UTL_Exit(void)  // Cleans up. Call 'as late as you can'.
  // (NOT calling UTL_Exit() makes UTL_iAppInstance run wild)
{

  if( UTL_dwInitMagicPi != C_UTL_INIT_MAGIC ) // don't let UTL_Exit() do anything without a previous UTL_Init() !
   { return;
   }
  UTL_dwInitMagicPi = 0;



  if( UTL_hInstanceDetectionMutex != NULL ) // only if UTL_Init() was called..
   { CloseHandle(UTL_hInstanceDetectionMutex); // closes THIS HANDLE .. (*)
     UTL_hInstanceDetectionMutex = NULL;
     // (*) Windows seems to close / delete the Mutex when this instance
     //     terminates itself. But keep this code as-is; who knows....
   }
  UTL_iAppInstance = -1;  // indicate "we're out of business"

#if(! _TEST_VIRDEF_TROUBLE_ )
  CoUninitialize();  // De-Init COM (formerly done in VARIOUS other modules).
#endif

} // end UTL_Exit()


//---------------------------------------------------------------------------
void UTL_LimitInteger( int *pI, int iMin, int iMax )
{
  if(*pI<iMin)
     *pI=iMin;
  if(*pI>iMax)
     *pI=iMax;
}


//---------------------------------------------------------------------------
// CRC32 calculation (more reliable than the crude old 'checksum',
//                    used for example to find out if the content
//                    of a simple C structure has been modified,
//                    and needs to be 'saved on exit' in a file, etc, etc..)
//
// Author :  Unknown (found on the net, but the poster wasn't the author).
// Purpose:  32-bit CRC with a polynomial that seems to be 'widely used' .
//       Obviously based on rosettacode.org/wiki/CRC-32#Implementation_2 .
//       After some difficulties, compatible with the 32-bit CRC that an LPC824
//       can calculate 'in hardware', with the following HARD-WIRED polynomial :
// x^32+ x^26 + x^23 + x^22 + x^16 + x^12 + x^11 + x^10 + x^8 + x^7 + x^5 + x^4 + x^2 + x + 1
//
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
#if( ! SWI_USE_HARDWARE_CRC )
 DWORD CRC32_table[256]; // occupies ONE KILOBYTE of precious RAM,
                         // but we don't want to such an incredible amount
                         // of memory in a tiny Cortex-M controller.
  // Ex: By "exposing" CRC32_table[] in the header, we can
  //     easily 'recycle' it for other purposes, between two CRC32-calculations,
  //     for the price of being unsafe for use in interrupts or worker threads.
  //     Thus, in THIS copy of WB's original code, CRC32_table[] isn't used
  //     for anything but the 32-bit-CRC-calculation .
#endif // ! SWI_USE_HARDWARE_CRC ?

#if( SWI_CRC_NEED_BIT_REVERSAL )
 extern BYTE EightBitMirror[256]; // C:\drivers1\Tables\EightBitMirror.c
#endif // SWI_CRC_NEED_BIT_REVERSAL ?


//---------------------------------------------------------------------------
void UTL_InitCRC32(void)  // must be called ONCE, *before* UTL_CRC32() !
{
#ifndef  SWI_CRC_SELF_TEST
# define SWI_CRC_SELF_TEST 1
#endif
#if( SWI_CRC_SELF_TEST )
  DWORD dwSelfTestResult;
  BYTE *pb4ByteTest = (BYTE*)"\x04\x10\x00\x00";
#endif // SWI_CRC_SELF_TEST ?

  // In THIS stripped-down implementation, only for for targets WITHOUT
  //    a suitable 'CRC engine' (like many Cortex-M microcontrollers):
  // Build a lookup table (in RAM) for the CRC32 .
  DWORD rem;
  int   i, j;

  /* Calculate CRC table. */
  for (i = 0; i < 256; i++)
   {
     rem = i;  /* remainder from polynomial division */
     for (j = 0; j < 8; j++)
      { if (rem & 1)
         {  rem >>= 1;
            rem ^= 0xedb88320;
            // On first glance, this didn't look like the polynomial used by NXP's "ISP CRC checksum" [UM10800],
            // and also not like the STM32F4'2 "CRC-32 (Ethernet) polynomial: 0x4C11DB7" [RM0090 Rev 13 page 113].
            // But thanks to Wikipedia, we can shed light on this : It's indeed the "common Ethernet" CRC-32 :
            //  Polynomial representation, Normal              = 0x04C11DB7  (used in ST's description)
            //  Polynomial representation, Reversed            = 0xEDB88320  (used in THIS implementation)
            //  Polynomial representation, Reversed reciprocal = 0x82608EDB  (used in Koopman's article on CRC)
            // It's also the same as used by Keil in their application note AN277 "ROM Self-Test in MDK-ARM".
         }
        else
         {  rem >>= 1;
         }
      }
     CRC32_table[i] = rem;
   }
  // If all went well, CRC32_table[] should now contain :
  //    0x00000000 0x77073096 0xEE0E612C 0x990951BA 0x076DC419 0x706AF48F ..
  // An LPC1765 (@100MHz) performed this in less than 3 milliseconds .
  // The online CRC calculator at www.sunshine2k.de/coding/javascript/crc/crc_js.html
  //    shows the same 'lookup table' when option 'Show REFLECTED lookup table' is SET .


#if( SWI_CRC_SELF_TEST )
  // Perform a few tests for CRC32() with various seeds, block sizes, and 'fragmented' blocks:
  dwSelfTestResult = UTL_CRC32( 0, pb4ByteTest, 4 ); // TEST #1a
  if( dwSelfTestResult != 0xB200EB3B )  // panic .. there's a bug in the CRC calculation !
   { strcpy( UTL_sz255LastError, "UTL_CRC32-self-test failed (#1a)" );
   }
  dwSelfTestResult = UTL_CRC32( 1, pb4ByteTest, 4 ); // TEST #1b : same data, asymmetric seed
  if( dwSelfTestResult != 0x0ABC8C5E )  // aargh !
   { strcpy( UTL_sz255LastError, "UTL_CRC32-self-test failed (#1b)" );
   }
  dwSelfTestResult = UTL_CRC32( 1UL<<31, pb4ByteTest, 4 ); // TEST #1c : same data, another seed
  if( dwSelfTestResult != 0x5FB8681B )  // eeek  !
   { strcpy( UTL_sz255LastError, "UTL_CRC32-self-test failed (#1c)" );
   }

  dwSelfTestResult = UTL_CRC32( 0xFFFFFFFF, (BYTE*)"\x01\x02\x03\x04\x05\x06\x07\x08\x09", 9 ); // TEST #2
  if( dwSelfTestResult != 0x591940CF )  // oops !
   { strcpy( UTL_sz255LastError, "UTL_CRC32-self-test failed (#2)" );
   }

  // Same as above, but in TWO calls, passing in the intermediate result as NEW SEED .
  // *Must* give the same result, to process very large blocks that don't fit in a buffer :
  dwSelfTestResult = UTL_CRC32( 0xFFFFFFFF, (BYTE*)"\x01\x02\x03\x04\x05\x06", 6 ); // -> 0xCFCB2978 ( TEST #3a )
  dwSelfTestResult = UTL_CRC32( dwSelfTestResult, (BYTE*)"\x07\x08\x09", 3 ); // -> 0x591940CF ( TEST #3b )
  if( dwSelfTestResult != 0x591940CF )  // ouch !
   { strcpy( UTL_sz255LastError, "UTL_CRC32-self-test failed (#3)" );
   }

  // Test #4 : simulate response from an 'internal bus' (SPI interface)
  dwSelfTestResult = UTL_CRC32( 0, (BYTE*)"\x0C\x10\x01\x00\x00\x01\x10\x00\x00\x00\x00\x00", 12 );
  if( dwSelfTestResult != 0x9A4A2D34 )  // f**k !
   { strcpy( UTL_sz255LastError, "UTL_CRC32-self-test failed (#4)" );
   }
  // Arrived here ? Congrats, you finally got the 'hardware CRC engine' compatible
  //                with WB's gold standard "software CRC" !
#endif // SWI_CRC_SELF_TEST ?

} // end CRC32_Init()


//---------------------------------------------------------------------------
DWORD UTL_CRC32( DWORD crc/*~seed*/, const BYTE *data, int len )
  // 'Litmus tests' using www.sunshine2k.de/coding/javascript/crc/crc_js.html :
  //                      Settings in that online CRC calculator :
  //    ( ) CRC-8    () CRC-16     (*) CRC-32
  //    CRC parametrization :      ( *not* predefined )
  //         -> Input reflected:  [v] (yes),
  //            Result reflected: [v] (yes),
  //            Polynomial :       0x4C11DB7 (supported by LPC824, too)
  //            Initial value:    0x00000000 ( complement of ~seed = 0xFFFFFFFF )
  //            Final Xor Value:  0xFFFFFFFF
  //    CRC Input Data : BYTES   0x04 0x10 0x00 0x00 (first IBUS packet)
  // ->  crc32 = 0x6CBBCBD8  ("initial value" = 0xFFFFFFFF would give 0xB200EB3B ! )
  //    CRC Input Data : BYTES   0x01 0x02 0x03 0x04 0x05 0x06 0x07 0x08 0x09  (9 bytes)
  // ->  crc32 = 0x591940CF = UTL_CRC32( 0xFFFFFFFF, "\x01\x02\x03\x04\x05\x06\x07\x08\x09", 9 );
  //
{
  // For targets WITHOUT a suitable 'CRC engine' ... :
  BYTE  octet;
  const BYTE  *p, *q;

  crc = ~crc;   // invert on INPUT - don't ask why ! (must be an implementation-specific thing)
  q = data + len;
  for (p = data; p < q; p++)
   { octet = *p;  /* Cast to unsigned octet (formerly known as BYTE - WB) */
     crc = (crc >> 8) ^ CRC32_table[(crc & 0xff) ^ octet];
   }
  return ~crc;  // also invert on OUTPUT - otherwise 'fragmented blocks' wouldn't be possible !

} // end UTL_CRC32()



//---------------------------------------------------------------------------
double UTL_ConvertYMDhmsToUnix( int year, int month, int day,
                                int hour, int minute, double dblSecond )
  // Converts a date into the UNIX format, which is explained below.
  // Shall be a better(?) replacement for the old "mktime" function.
  //      NO lousy global variables like "_timezone" and "_daylight" are used.
  //      We always use UTC (~GMT) here, and double values are much better
  //      in the year 2038 (when programs using the old "mktime" will fail).
  //  Input parameters: full year (like 2001),
  //                    month: 1..12, day: 1..31,
  //                    hour: 0..23, minute: 0..59, second: 0..59 .
  //  Return:  UNIX-Format = SECONDS elapsed since January 1st, 1970, 00:00:00 .
  //  Example: 22.10.2001, 11:12:13  returns 1003749133 UNIX-SECONDS.
  //          (DD.MM.YYYY, hh:mm:ss)
{
  // Nach Infos von http://www.pci.uzh.ch/pfister/formelsammlung.html ,
  //                  (Lokal gespeichert unter "Rolfs Formelsammlung")
  //  Julianisches Datum (JD)
  // JD = Anzahl Tage seit 4713 vor Christus
  // bis und mit 4.Oktober 1582: B = int((y+4716)/4) - 1181
  //        ab 15.Oktober 1582: B = int(y/400)-int(y/100)+int(y/4)
  // (Wegen der Gregoranischen Kalenderreform folgte auf den 4.Okt. sogleich der 15.)
  //  JD = 365*y - 679004 + B + int(30.6*m) + d + h/24 + 2400000.5
  // MJD = JD - 2400000.5  (Modifiziertes Julianisches Datum)
  //          (WB: MJD = Anzahl Tage seit 17. November 17 1858 Mitternacht)
  // h = Tageszeit in Stunden
  // d = Tag
  // Y = fr Jahre vor Christus: Y = -1 - Jahreszahl
  //     fr Jahre nach Christus: Y = Jahreszahl
  // wenn Monat<=2 : y = Y-1   m = Monat+13
  //          sonst: y = Y     m = Monat+1
  // int() = Abrundung auf ganze Zahl
  //
  // ... Conversion into C :
  // The "Modified Julian Date" is the number of DAYS passed since
  //     >>  Midnight, November 17, 1858  <<
  // The UNIX date-and-time format is the number of SECONDS passed since
  //     >>  Midnight, January 1, 1970    <<
  long   b, m, y;
  double mjd;    // Modified Julian Date ... the unit is DAYS

  if( month<= 2) { y=year-1; m=month+13; }
         else    { y=year;   m=month+1;  }
  if( (year>=1582) || (month>=10) || (day>=15) )
       b = (int)(y/400) - (int)(y/100) + (int)(y/4);
   else // before October 4, 1582:
       b = (int)((y+4716)/4) - 1181;
   mjd = 365.0*(double)y - 679004 + b + (int)((double)m*30.6) + day;
   // Now convert the result from Modified Julian DAYS (w/o hours)
   // to UNIX-Date in "SECONDS since January 1, 1970",
   //   because the second is the SI unit for the time !
   // The MJD of Jan 1, 1970 has been calculated by the above routine
   // as:  40587 [days between Jan1, 1970 00:00:00  and  Nov 17, 1858, 00:00:00]
   return ((double)mjd - 40587.0) * 86400.0/* seconds per day*/
         + (double)hour*3600.0 + (double)minute*60.0 + dblSecond;
   // Note: In contrast to UNIX, this routine may also return valid negative
   //       numbers (there were good times even before January 1, 1970 ;-)
   //       and -hopefully- this routine will still work after year 2038 .
} // end UTL_ConvertYMDhmsToUnix()

//---------------------------------------------------------------------------
int UTL_GetDayOfWeekFromUnixDateTime( // -> 0=Monday...6=Sunday
    double dblUnixDateTime ) // [in] SECONDS passed since Midnight, January 1, 1970, UTC
{
  long unix_day = (long)(dblUnixDateTime / 86400.0); // scale from seconds to days .
                 // Unix day "zero" (1970-01-01) was a THURSDAY, thus..
  return (unix_day + 3) % 7;
                 //  '--> Adjustment to place the Unix birthdate on a THURSDAY.
} // end UTL_GetDayOfWeekFromUnixDateTime()

//---------------------------------------------------------------------------
void UTL_SplitUnixDateTimeToYMDhms( double dblUnixDateTime,
         int *piYear, int *piMonth, int *piDay, // [out] "calendar day-of-year"
         int *piHour, int *piMinute, double *pdblSecond ) // [out] time of day
  // Inverse function to UTL_ConvertYMDhmsToUnix() .
  // Splits a date-and-time in UNIX format (seconds since Jan 1st, 1970, 00:00:00 UTC)
  // into year, month, day, hour, minute, and (fractional) seconds .
  // Much better than the ugly, thread-unsafe, global-variable-dependent,
  //                      stupid, stonage function named 'gmtime()' !
{
  double dbl, dblUnixDay, dblSecondsOfDay, dblHour, dblMin;
  long   jdayno, j, d, m, y;

  // Split DAY and SECONDS-OF-DAY (here: still UNIX, not MJD, and not JD):
  dblUnixDay = floor( dblUnixDateTime / 86400.0/*seconds per day*/ );
  dblSecondsOfDay = dblUnixDateTime - dblUnixDay * 86400.0;
  dbl = dblSecondsOfDay / 3600.0;      // -> hours of day, fractional
  dblHour = floor(dbl); dbl=60.0*(dbl-dblHour);  // -> hours of day
  dblMin  = floor(dbl); dbl=60.0*(dbl-dblMin);   // -> minutes of hour
  if( piHour != NULL )
   { *piHour = (int)dblHour;
   }
  if( piMinute!= NULL )
   { *piMinute = (int)dblMin;
   }
  if( pdblSecond!= NULL )
   { *pdblSecond = dbl;   // remaining "seconds of minute", fractional
   }

  // Now for the trickier part: Split year, month of year, and day of month.
  // From   http://en.wikipedia.org/wiki/Julian_day :
  // > Unix Time (seconds since January 1st, 1970, UTC)
  // >  calculated from Julian Day by : (JD - 2440587.5) * 86400
  // So, calculate 'jdayno' (Julian Day, JD) for the algo further below.
  //  "UnixDay" = JulianDay - 2440587.5, so :
  jdayno = (long)( dblUnixDay + 2440587.5 );  // -> Julian Day, INTEGER

  // Die folgende Berechnung stammt aus Google Code Search, Stichwort
  //  "jdater - sets day, month and year given jdayno" (haystack.edu .. date.c)
  //  "inverse of jday method" .
  // Was in der Beschreibung leider komplett fehlt, ist eine Spezifikation
  //  der ZAEHLWEISE, speziell fuer jdayno (0...x oder 1...x),
  //  Monat(0..11 oder 1..12) und Tag (0..30 oder 1..31) !
  // Von gmtime() ist man ja Elendes gewohnt (Tage von 1..31 aber Monate von 0..11).
  // Hier scheint aber alles AB EINS gezaehlt zu werden (Pascal-Freaks?),
  // auch der 'jdayno' beginnt die Zaehlung scheinbar bei EINS, nicht NULL, darum:
  ++jdayno;
  // Original formula from haystack.edu starts here:
  j = jdayno - 1721119;
  y = (4*j - 1)/146097;
  j  =  4*j - 1 - 146097*y;
  d = j/4;
  j = (4*d + 3)/1461;
  d = 4*d + 3 - 1461*j;
  d = (d + 4)/4;
  m = (5*d - 3)/153;
  d = 5*d - 3 - 153*m;
  d = (d + 5)/5;
  y = 100*y + j;
  if (m < 10)
   {  m = m + 3;
   }
  else
   {  m = m - 9;
      y = y + 1;
   }

  if( piYear != NULL )
   { *piYear = y;
   }
  if( piMonth != NULL )
   { *piMonth = m;
   }
  if( piDay != NULL )
   { *piDay = d;
   }
} // end UTL_SplitUnixDateTimeToYMDhms()



//---------------------------------------------------------------------------
double UTL_GetCurrentUnixDateAndTime(void)
  // Returns "Seconds elapsed since 00:00:00 GMT(=UTC), January 1, 1970" ,
  // as a simple 64-bit "double" which makes calculations very simple.
  // Forget about a bazillion of funny structures that have plagued
  // developers over decades.
  // See also (less 'expensive', but not precise in the long run, without GetSystemTime() ):
  //  double UTL_GetCurrentUnixDateAndTime_Fast() .
  //
{
  // Tried this NON-VCL implementation, after reading megatons of info
  // about the 'date and time' API in windoze .. the journey started
  // at  http://www.codeproject.com/KB/datetime/datetimedisc.aspx  .
  // After that, some of those functions were looked up in the Win32 manual.
  // > The GetSystemTime function retrieves the current system date and time.
  // > The system time is expressed in Coordinated Universal Time (UTC).
  // Ok, sounds nice to have UTC. But what's this 'SYSTEMTIME' thingy ?
  // Of course it's not a nice Unix-like time (seconds counter), but
  // split into year,month,day,hour,minute,second, and milliseconds..
  SYSTEMTIME stm;
  GetSystemTime( &stm );
  // now convert date+time from Windows into Unix format
  return UTL_ConvertYMDhmsToUnix( stm.wYear, stm.wMonth, stm.wDay,
                                  stm.wHour, stm.wMinute,
                                  (double)stm.wSecond
                                + (double)stm.wMilliseconds * 1e-3 );
} // end  UTL_GetCurrentUnixDateAndTime(void)

//---------------------------------------------------------------------------
double UTL_GetCurrentUnixDateAndTime_Fast(void)
  // Almost the same as UTL_GetCurrentUnixDateAndTime(), but less expensive
  //  (uses the windows "performance counter", converted into an absolute
  //   date and time, simply by adding an offset).
  // Also returns "Seconds elapsed since 00:00:00 GMT(=UTC), January 1, 1970" ,
  // as a simple 64-bit "double" which makes calculations very simple.
  // Forget about a bazillion of funny structures that have plagued
  // developers over decades.
  // See also (more 'expensive', but more precise in the long run due to Windows time-sync services):
  //  double UTL_GetCurrentUnixDateAndTime() .
  //
{
  LONGLONG i64;
  double dblUnixSeconds;
  if( UTL_i64HighResTimerFreq == 0 )
   { QueryPerformanceFrequency( (LARGE_INTEGER*)&UTL_i64HighResTimerFreq ); // will the FREQUENCY change when speed-stepping ? (hope not..)
     if( UTL_i64HighResTimerFreq == 0 ) // oops..  we're running on a target without this "Performance Counter"-thingy !
      { return UTL_GetCurrentUnixDateAndTime();
      }
   }
  QueryPerformanceCounter( (LARGE_INTEGER*)&i64 );
  if( UTL_i64HighResTimerOffset == 0 )
   {  dblUnixSeconds = UTL_GetCurrentUnixDateAndTime();
      // Can a 64-bit integer store the "Unix time" in PERFORMANCE TIMER TICKS ?
      // For worst case, assume UTL_i64HighResTimerFreq is a Gigahertz.
      // If we want our program to run until year 2100, the counter value will be
      // (2100-1970)*365*24*60*60 seconds * 1 GHz = 4e18.  4e18 / 2^63 = 0.44
      //          -> We're safe.. but not much headroom .
      UTL_i64HighResTimerOffset = (LONGLONG)(dblUnixSeconds * (double)UTL_i64HighResTimerFreq - (double)i64);
      // '--> Inverse to the calculation below :
      //  dblUnixSeconds = (i64 + UTL_i64HighResTimerOffset) / UTL_i64HighResTimerFreq;
      //  dblUnixSeconds * UTL_i64HighResTimerFreq = i64 + UTL_i64HighResTimerOffset;
      //  UTL_i64HighResTimerOffset = dblUnixSeconds * UTL_i64HighResTimerFreq - i64;
   }
  // ex: i64 = (i64 * 1000000LL) / i64Freq; // even a 64-bit product may be insufficient..
  // thus use floating point to scale from timer ticks into microseconds:
  dblUnixSeconds = (double)(i64 + UTL_i64HighResTimerOffset) / (double)UTL_i64HighResTimerFreq;
  return dblUnixSeconds;
} // end UTL_GetCurrentUnixDateAndTime_Fast()

//---------------------------------------------------------------------------
void UTL_FormatDateAndTime( const char* format, double dblUnixDateTime, char* dest)
  // Example for format:  "YYYY-MM-DD hh:mm:ss"  (preferred ISO8601)
  //           ->result:  "2003-06-01 14:28:02"  (placed in dest)
  // The input parameter dblDateTime is similar to the UNIX time:
  //         "SECONDS elapsed since January 1st, 1970, 00:00:00" .
{
 char   fmt,fmt2,fmt3;
 int    i;
 int    year,month,day,hour,min,weekday;
 double dblSec, dblSec2;

  UTL_SplitUnixDateTimeToYMDhms( dblUnixDateTime,
                                 &year, &month, &day, &hour, &min, &dblSec );
  while( (fmt = *format++) != 0)
   {
    fmt2 = *format; // to look ahead at following letters in the format string
    fmt3 = *(format+1);
    switch(fmt)
     {
       case 'Y':  // year  (1 to 4 digits)
            if(fmt2=='Y')
             {
              ++format;  // skip 2nd letter in format string
              if(fmt3=='Y')
               {
                ++format;  // skip 3rd letter in format string
                if(*format=='Y')
                 {
                  ++format;  // skip 4th letter in format string
                  sprintf(dest,"%04d",(int)year );
                 }
                else // 3-letter format (??)
                 {
                  sprintf(dest,"%03d",(int)(year%1000) );
                 }
               }
              else // 2-letter format:
               {
                sprintf(dest,"%02d",(int)(year%100) );
               }
             } // end if (fmt2..)
            else // 1-letter format (used for automatic filenames somewhere)
             {
              sprintf(dest,"%01d",(int)(year%10) );
             }
            dest += strlen(dest);
            fmt = 0; // format letter has been handled
            break;

       case 'M':  // month (2 digits or THREE LETTERS (Jan,Feb, etc) )
            if(fmt2=='M' || fmt2=='m')
             {
              ++format;  // skip 2nd letter in format string
              if(fmt3=='M') // "MMM" = { JAN, FEB, .. }
               {
                ++format;  // skip 3rd letter in format string
                if( month<1 ) month=1;
                strcpy(dest,UTL_months1[(month-1) % 12]);
               }
              else
              if(fmt3=='m') // "Mmm" = { Jan, Feb, .. }
               {
                ++format;  // skip 3rd letter in format string
                if( month<1 ) month=1;
                strcpy(dest,UTL_months2[(month-1) % 12]);
               }
              else // 2-letter format: numerical, 1..12
               {
                sprintf(dest,"%02d",(int)month );
               }
             } // end if (fmt2..)
            else // 1-letter format (??)
             {
              sprintf(dest,"%01d",(int)month );
             }
            dest += strlen(dest);
            fmt = 0; // format letter has been handled
            break;

       case 'D':  // day (2 digits)
            if(fmt2=='D')
             {
              ++format;  // skip 2nd letter in format string
              sprintf(dest,"%02d",(int)day);
             } // end if (fmt2..)
            else // 1-letter format (??)
             {
              sprintf(dest,"%01d",(int)day);
             }
            dest += strlen(dest);
            fmt = 0; // format letter has been handled
            break;


       case 'h':  // hour
            if(fmt2=='h')
             {
              ++format;  // skip 2nd letter in format string
              sprintf(dest,"%02d",(int)hour);
             } // end if (fmt2..)
            else // 1-letter format (??)
             {
              sprintf(dest,"%01d",(int)hour);
             }
            dest += strlen(dest);
            fmt = 0; // format letter has been handled
            break;


       case 'm':  // minute (always 2 digits)
            if(fmt2=='m')
             {
              ++format;  // skip 2nd letter in format string
              sprintf(dest,"%02d",(int)min);
             } // end if (fmt2..)
            else // 1-letter format (??)
             {
              sprintf(dest,"%01d",(int)min);
             }
            dest += strlen(dest);
            fmt = 0; // format letter has been handled
            break;


       case 's':  // second (at least 2 digits, possible 'fractional' !)
            if(fmt2=='s')
             {
              ++format;  // skip 2nd letter in format string
              sprintf(dest,"%02d",(int)dblSec );
             } // end if (fmt2..)
            else // 1-letter format (??)
             {
              sprintf(dest,"%01d",(int)dblSec );
             }
            dest += strlen(dest);
            if( format[0]=='.' && format[1]=='s')
             { // looks like tenths of a second after a decimal point !
               format+=2; // skip ".s" from the format string
               dblSec2 = 10.0 * dblSec;
               i = fmod( dblSec2, 10.0);
               if (i<0)   // ??
                   i=0;
               if (i>9)   // ????
                   i=9;   // set breakpoint here if fmod behaves strange...
               *dest++ = '.';
               *dest++ = (char)('0'+i);  // append 1/10 second(s)
               // Modified 2015-01 : Even more decimal places (1/100, 1/1000 s)
               while(*format=='s')
                 { // gosh.. he wants to see hundreds of a second :
                   ++format; // skip another "s" from the format string
                   dblSec2 *= 10.0;
                   i = fmod( dblSec2, 10.0);
                   *dest++ = (char)('0'+i);  // append 1/100, 1/1000 second(s), any maybe even more
                   // (see AFilePrp.cpp: GPS-sync'd recordings may have timestamps
                   //      with microsecond accuracy, or at least resolution )
                 }
               *dest   = '\0';
             }
            fmt = 0; // format letter has been handled
            break;

       case 'W': // "Wd" week day (day of week, "Mo","Tu","We","Th","Fr","Sa","Su",
                 // "Wt" Wochentag (germanski,  "Mo","Di","Mi","Do","Fr","Sa","So")
            if(fmt2=='t')
             { ++format;  // skip 2nd letter in format string
               weekday = UTL_GetDayOfWeekFromUnixDateTime(dblUnixDateTime); // -> 0=Monday...6=Sunday
               strcpy(dest,UTL_weekdays[weekday] );
               dest += strlen(dest);
               fmt = 0; // format letter has been handled
             } // end if (fmt2..)
            else if(fmt2=='d') // das gleiche mit deutschen Wochentagen..
             { ++format;  // skip 2nd letter in format string
               weekday = UTL_GetDayOfWeekFromUnixDateTime(dblUnixDateTime);
               strcpy(dest,UTL_wochentage[weekday] );
               dest += strlen(dest);
               fmt = 0; // format letter has been handled
             }
            break;

       default: // all other characters will be passed 1:1 to "dest"
            break;
     } // end switch(fmt)

    if (fmt) // character from format string could not be handled:
     {
      *dest++ = fmt;
     }
   } // end while(fmt)
  *dest = '\0';  // null-terminate the destination string
} // end UTL_FormatDateAndTime()


//---------------------------------------------------------------------------
void UTL_LastWindowsErrorCodeToString( DWORD dwWindowsErrorCode, char* pszDest, int iMaxDestLen )
  // Converts whatever Windows has vomited out via 'GetLastError()'
  // into a human-readable string ... WITHOUT THE FOOLISH '\n' CHARACTER AT THE END.
{
  LPVOID lpMsgBuf;
  DWORD  dwLength;

  // Try to let Windows convert the GetLastError()-code into a readable string:
  if( FormatMessage(FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM,
        NULL,
        dwWindowsErrorCode,
        0,
        (LPTSTR)&lpMsgBuf,  // [out] a string buffer allocated by FormatMessage() itself (!)
        0,
        NULL) == 0)
   { /* failed to format the message */
     snprintf(pszDest, iMaxDestLen, "unknown error #%ld",(long)dwWindowsErrorCode );
   }
  else // FormatMessage() successfull ..
   {
     // Convert the string and free the message's buffer.
     SL_strncpy(pszDest, (char*)lpMsgBuf, iMaxDestLen-1 );

     // Free the buffer that been allocated somehow in FormatMessage(!)
     LocalFree( lpMsgBuf );

     // If the result string contains this foolish CR and/or NL characters at the end,
     // trow them out.. they are UTTERLY USELESS in 99.9999 % of all cases !
     dwLength = strlen(pszDest);
     if( dwLength < (DWORD)iMaxDestLen ) // ok, try to remove the trailing junk that Windows may have appended :
      { if( (dwLength>0) && (pszDest[dwLength-1] == '\r') )  // crazy #1
         { --dwLength;
           pszDest[dwLength] =  '\0';
         }
        if( (dwLength>0) && (pszDest[dwLength-1] == '\n') )  // crazy #2
         { --dwLength;
           pszDest[dwLength] =  '\0';
         }
        if( (dwLength>0) && (pszDest[dwLength-1] == '\r') )  // crazy #3
         { --dwLength;
           pszDest[dwLength] =  '\0';
         }
      }
   }
} // end UTL_LastWindowsErrorCodeToString()


//---------------------------------------------------------------------------
void UTL_OpenRunLogFile( const char* pszLogFileName )
{
  SYSTEMTIME ti;
  if( pszLogFileName && (pszLogFileName[0]>0) )
   {  DeleteFile( pszLogFileName ); // write a new, "clean" file when it's due
      SL_strncpy( UTL_sz255LogFile, pszLogFileName, 255);
      GetSystemTime( &ti );       // retrieves the system time IN UTC (hooray)
      UTL_fRunLogOpened = TRUE;     // allow UTL_WriteRunLogEntry() to write..
      if( ! UTL_WriteRunLogEntry( "Logfile created on %04d-%02d-%02d %02d:%02d:%02d UTC",
          (int)ti.wYear, (int)ti.wMonth, (int)ti.wDay,
          (int)ti.wHour, (int)ti.wMinute,(int)ti.wSecond ) )
       { // Most likely, the never-ending trouble with missing write permissions
         // for "our" folder. Give up, and simply don't try again:
         UTL_fRunLogOpened = FALSE;
       }
   }
} // end UTL_OpenRunLogFile()

//---------------------------------------------------------------------------
BOOL UTL_WriteRunLogEntry( const char* pszFormatString, ... )
  // Opens the "run-log"-file, appends a line of text, and closes again (!)
  //    because this program / the OS / the PC may crash and we don't want
  //    to lose the debugging info .
  // To avoid flooding the harddisk with error messages (if the same error
  //  is detected & written over and over again), the logfile is
  //  deliberately limited to a certain number of messages.
  // See also: ShowError(), which "prints into the GUI", not into a logfile.
{
  va_list arglist;  // very 'arglistig' if not handled properly (crash,bang,boom)
  SYSTEMTIME ti;
  char temp[1024];
  char* cp;
  BOOL fResult = FALSE;   // -> TRUE=success, FALSE=couldn't write

  if( (UTL_dwInitMagicPi != C_UTL_INIT_MAGIC) || (!UTL_fRunLogOpened) )
   { return FALSE;
   }

  if( UTL_iLogFileNrOfErrors > 2000 )   // 2009-07 : increased from 1000 to 2000
   { return FALSE;
   }

  EnterCriticalSection( &UTL_csRunLog );

  cp = temp;
  GetSystemTime( &ti );
  wsprintf( cp, "%02d:%02d:%02d.%01d ",
                (int)ti.wHour,  (int)ti.wMinute,
                (int)ti.wSecond,(int)ti.wMilliseconds / 100 );
  cp += strlen(cp);

  // Print to string and append to edit control
  va_start(arglist, pszFormatString);
  vsprintf(cp, pszFormatString, arglist);
  va_end(arglist);    // MUST be called, always !
  cp += strlen(cp);

#if(0)  // Send this message to the SYSTEM DEBUGGER ?
  if( UTL_fRunLog_UseOutputDebugString )
   { OutputDebugString( temp );  // here: WITHOUT trailing CR+NL  !
     // > The OutputDebugString function sends a string to the debugger for the current application.
     // (In Borland C++Builder erscheinen diese Strings im 'Ereignisprotokoll'
     //  mit dem Vorsatz "ODS:", was wohl "Output Debug String" heissen soll)
   }
#endif // (0)

  // Append this message to the "run-log" file ?
  if( UTL_sz255LogFile[0] > 32 )
   { int iHandle = _rtl_open(UTL_sz255LogFile, O_RDWR);
     if(iHandle>=0)
      { // successfully OPENED the file, jump to its end :
        lseek( iHandle, 0, SEEK_END );
      }
     else // could not OPEN the file, so CREATE it :
      { iHandle = _rtl_creat(UTL_sz255LogFile, 0/*attrib*/ );
      }
     if(iHandle>=0)
      { strcpy( cp, "\r\n" );  // actually APPENDS CR+NL to 'temp'
        _rtl_write( iHandle, temp, strlen(temp) );
        _rtl_close( iHandle );
        fResult = TRUE;
        ++UTL_iLogFileNrOfErrors;
      }
   } // end if < also log the message text AS A FILE ? >

  LeaveCriticalSection( &UTL_csRunLog );


  return fResult;

} // end UTL_WriteRunLogEntry()


//---------------------------------------------------------------------------
void UTL_ClipPointToDesktop( int *piX, int *piY)
  // Clips a pixel coordinate into the "total" desktop area
  // (which btw is called "The Virtual Screen" in the MSDN documentation)
{
  if( *piX < UTL_iDesktopXMin)
   {  *piX = UTL_iDesktopXMin;
   }
  if( *piX > (UTL_iDesktopXMax-24) )
   {  *piX =  UTL_iDesktopXMax-24;
   }
  if( *piY < UTL_iDesktopYMin)
   {  *piY = UTL_iDesktopYMin;
   }
  if( *piY > (UTL_iDesktopYMax-12) )
   {  *piY =  UTL_iDesktopYMax-12;
   }
} // end UTL_ClipPointToDesktop()

//---------------------------------------------------------------------------
BOOL UTL_ClipXYWHToDesktop( int *piX, int *piY, int *piWidth, int *piHeight )
  // This function was implemented (decades ago) as a workaround for
  // YET ANOTHER BUG in the VCL:
  //    If the monitor resolution was modified, and the position of the main window
  //    was saved between sessions, the window could be COMPLETELY OUT OF SIGHT
  //    (on the desktop) - no chance to grab it with the mouse, and no chance
  //    to make the program usable again (without deleting the configuration file).
  // Clips an entire window (defined by x1, y1, width, height in pixels)
  // to the desktop, but (since 2015-02) allows SLIGHTLY NEGATIVE coordinates,
  // as long as the window remains 'partially visible',
  // and can be 'grabbed' with the mouse :
  //  - under Windows XP with 120 DPI (adjustable under 'Desktop'..'Properties'),
  //    a window can just be grabbed and moved with the mouse if y>-20 ;
  //
  //  [in] UTL_iDesktopXMin, UTL_iDesktopXMax,
  //       UTL_iDesktopYMin, UTL_iDesktopYMax, all set in an obfuscated
  //               "MonitorEnumProc()"-callback function .
  //
  // Similar incarnations of this function exist in :
  //  - C:\cbproj\SpecLab\Config.cpp          [master]
  //  - C:\cbproj\WinPicPr\WinPicPr.cpp       [copy]
  //  - C:\cbproj\Remote_CW_Keyer\Utilities.c [stripped-down copy]
{ BOOL fClipped = FALSE;
  int x1 = *piX;
  int w  = *piWidth;
  int x2;
  int y1 = *piY;
  int h  = *piHeight;
# define UTL_MIN_WINDOW_WIDTH  20
# define UTL_MIN_WINDOW_HEIGHT 20

  // Check for VERTICAL settings .. only accept positions "slightly off" the desktop
  if( w > (UTL_iDesktopXMax - UTL_iDesktopXMin) )
   {  w =  UTL_iDesktopXMax - UTL_iDesktopXMin;
      fClipped = TRUE;
   }
  if( w < UTL_MIN_WINDOW_WIDTH )
   {  w = UTL_MIN_WINDOW_WIDTH;
      fClipped = TRUE;
   }
  if( x1 < (UTL_iDesktopXMin-20) )
   {  x1 =  UTL_iDesktopXMin-20;
      fClipped = TRUE;
   }
  x2 = x1 + w - 1;
  if( x2 < (UTL_iDesktopXMin+20) ) // at least 20 pixels at the RIGHT(!) edge of the window
   {                               //     must be visible at the LEFT(!) edge of the screen
     x2 = UTL_iDesktopXMin+20;     // because otherwise it's difficult to 'grab' with the mouse
     x1 = x2 - w + 1;
     fClipped = TRUE;
   }
  if( x1 > (UTL_iDesktopXMax-20) ) // at least 20 pixels at the LEFT(!) edge of the window
   {                               //   must be visible at the RIGHT(!) edge of the screen
     x1 =  UTL_iDesktopXMax-20;    // .. etc, etc
     fClipped = TRUE;
   }
  *piX     = x1;
  *piWidth = w;

  // Similar for the VERTICAL settings .. only accept positions "slightly off" the desktop
  if( h > (UTL_iDesktopYMax - UTL_iDesktopYMin) )
   {  h =  UTL_iDesktopYMax - UTL_iDesktopYMin;
      fClipped = TRUE;
   }
  if( h < UTL_MIN_WINDOW_HEIGHT )
   {  h = UTL_MIN_WINDOW_HEIGHT;
      fClipped = TRUE;
   }
  if( y1 < UTL_iDesktopYMin-29) // not more than 29 pixels of the window title may be "out of sight"
   {  y1 = UTL_iDesktopYMin-29; // (this is the "most negative coord" windows XP allowed at 120 DPI)
      fClipped = TRUE;
   }
  if( y1 > (UTL_iDesktopYMax-10) )
   {  y1 =  UTL_iDesktopYMax-10;
      fClipped = TRUE;
   }
  *piY     = y1;
  *piHeight= h;
  return fClipped;  // returns TRUE when clipped, FALSE when there was no need for clipping

} // end UTL_ClipXYWHToDesktop()


//---------------------------------------------------------------------------
BOOL CALLBACK UTL_MonitorEnumProc(
  HMONITOR hMonitor,  // > "handle to display monitor"
  HDC hdcMonitor,     // > "handle to monitor DC"
  LPRECT lprcMonitor, // > "monitor intersection rectangle"
  LPARAM dwData)      // > "data" (hmm..)
{
  if( lprcMonitor->left < UTL_iDesktopXMin )
     UTL_iDesktopXMin = lprcMonitor->left;
  if( lprcMonitor->right> UTL_iDesktopXMax )
     UTL_iDesktopXMax = lprcMonitor->right;
  if( lprcMonitor->top < UTL_iDesktopYMin )
     UTL_iDesktopYMin = lprcMonitor->top;
  if( lprcMonitor->bottom > UTL_iDesktopYMax )
     UTL_iDesktopYMax = lprcMonitor->bottom;
  return TRUE;  // "To continue the enumeration, return TRUE."
} // end UTL_MonitorEnumProc()


//--------------------------------------------------------------------------
// Function to enumerate all "COM ports" (serial ports) on a Windoze PC
//--------------------------------------------------------------------------

int EnumerateComPortsWithFriendlyNames( // ... using the registry + funny "SetupDi"-stuff ...
       int iMaxComPorts,   // [in] Maximum number of NAMES / PORTS to retrieve,
                           //      limited by the fixed-size TWO-DIMENSIONAL arrays
       int iSizePerPerName, // [in] Maximum name of a SINGLE NAME, INCLUDING the space for each trailing zero
       char  *pPortName,   // [out] TWO-DIMENSIONAL char array (details below)
                           //       for the COM port's UNFRIENDLY names
       char  *pFriendName) // [out] TWO-DIMENSIONAL char array (details below)
                           //       for the COM port's FRIENDLY names
    // Return value: ZERO when there is "not a single COM port" (but no error),
    //               NEGATIVE when something went really wrong (hey, it's windows),
    //               or THE NUMBER OF COM PORTS successfully enumerated.
    // To store the results, the caller needs to provide
    //    TWO fixed-size, two-dimensional character arrays like:
    //    #define MAX_PORT_NUM 20
    //    #define MAX_STR_LEN  80
    //    int nPorts;
    //    char portNames[MAX_PORT_NUM][MAX_STR_LEN];
    //    char friendlyNames[MAX_PORT_NUM][MAX_STR_LEN];
    //    nPorts = EnumerateComPortsWithFriendlyNames(
    //                 MAX_PORT_NUM, MAX_STR_LEN, portNames, friendlyNames );
{
  // Based on "Enumerating com ports in Windows",
  // "EnumerateComPortSetupAPISetupDiClassGuidsFromNamePort()" [what a name ..]
  //  > original by PJ Naughter, 1998 - 2013
  //  > http://www.naughter.com/enumser.html ,
  //  > reorganized by Gaiger Chen , Jul, 2015,
  // .. and heavily modified / simplified / shortened / commented by DL4YHF, 2024.
  int iSetupItem = 0, nComPortsFound = 0;

  char* pTempPortName;
  HMODULE hLibrary;
  char szFullPath[_MAX_PATH];
  char* pszDest;
  const char* pszEndstop;

  GUID *pGuid;
  DWORD dwGuids;
  HDEVINFO hDevInfoSet;


  // FUNCTION POINTER types for all those funny functions extracted from
  //   "SETUPAPI.DLL" (not really required for Borland because they had
  //   an import library to call those functions 'directly', but since the
  //   code from PJ Naughter or Gaiger Chen already had the dynamic loading,
  //   why throw them out ?
 typedef HKEY (__stdcall SetupDiOpenDevRegKeyFunType)
                (HDEVINFO, PSP_DEVINFO_DATA, DWORD, DWORD, DWORD, REGSAM);
 typedef BOOL (__stdcall SetupDiClassGuidsFromNameFunType)(LPCTSTR, LPGUID, DWORD, PDWORD);
 typedef BOOL (__stdcall SetupDiDestroyDeviceInfoListFunType)(HDEVINFO);
 typedef BOOL (__stdcall SetupDiEnumDeviceInfoFunType)(HDEVINFO, DWORD, PSP_DEVINFO_DATA);
 typedef HDEVINFO (__stdcall SetupDiGetClassDevsFunType)(LPGUID, LPCTSTR, HWND, DWORD);
 typedef BOOL (__stdcall SetupDiGetDeviceRegistryPropertyFunType)
                (HDEVINFO, PSP_DEVINFO_DATA, DWORD, PDWORD, PBYTE, DWORD, PDWORD);

  SetupDiOpenDevRegKeyFunType* SetupDiOpenDevRegKeyFunPtr;
  SetupDiClassGuidsFromNameFunType *SetupDiClassGuidsFromNameFunPtr;
  SetupDiGetClassDevsFunType *SetupDiGetClassDevsFunPtr;
  SetupDiGetDeviceRegistryPropertyFunType *SetupDiGetDeviceRegistryPropertyFunPtr;
  // SetupDiDestroyDeviceInfoListFunType *SetupDiDestroyDeviceInfoListFunPtr;
  SetupDiEnumDeviceInfoFunType *SetupDiEnumDeviceInfoFunPtr;

  SP_DEVINFO_DATA devInfo;

  szFullPath[0] = '0';

  // Get the Windows System32 directory

  // ex: if(0 == GetSystemDirectory(szFullPath, _countof(szFullPath)))
  // "_countof()" ? Who the heck is that ? A, yet another Microsoft thing:
  // > Computes the number of elements in a statically allocated array .
  // There's no such thing in Borland, so "away with this proprietary stuff":
  if(0 == GetSystemDirectory(szFullPath, sizeof(szFullPath)))
   { // "_tprintf()" - who the heck is that ? No such thing in BCB V6 .. away with it !
     // _tprintf(TEXT("CEnumerateSerial::UsingSetupAPI1 failed, Error:%u\n"), GetLastError());
     return -1;  // ERROR !
   }


  // Setup the full path and delegate to LoadLibrary
  pszDest    = szFullPath;
  pszEndstop = szFullPath + sizeof(szFullPath) - 1;
  SL_SkipToEndOfString( (const char**)&pszDest );
  SL_AppendString(&pszDest, pszEndstop, "\\SETUPAPI.DLL" );
  hLibrary = LoadLibrary(szFullPath);


  SetupDiOpenDevRegKeyFunPtr = (SetupDiOpenDevRegKeyFunType*)
      GetProcAddress(hLibrary, "SetupDiOpenDevRegKey");

  // Ancient Borland doesn't support "UNICODE" ("wide strings", not UTF-8),
  //  so use the old -A (ANSI?) variants:
  SetupDiClassGuidsFromNameFunPtr = (SetupDiClassGuidsFromNameFunType*)GetProcAddress(hLibrary, "SetupDiClassGuidsFromNameA");
  SetupDiGetClassDevsFunPtr = (SetupDiGetClassDevsFunType*)GetProcAddress(hLibrary, "SetupDiGetClassDevsA");
  SetupDiGetDeviceRegistryPropertyFunPtr = (SetupDiGetDeviceRegistryPropertyFunType*)GetProcAddress(hLibrary, "SetupDiGetDeviceRegistryPropertyA");
  // SetupDiDestroyDeviceInfoListFunPtr = (SetupDiDestroyDeviceInfoListFunType*)GetProcAddress(hLibrary, "SetupDiDestroyDeviceInfoList");
  SetupDiEnumDeviceInfoFunPtr = (SetupDiEnumDeviceInfoFunType*)GetProcAddress(hLibrary, "SetupDiEnumDeviceInfo");

  // First need to convert the name "Ports" to a GUID using SetupDiClassGuidsFromName.
  // Warum einfach, wenn's auch kompliziert geht...
  dwGuids = 0;
  SetupDiClassGuidsFromNameFunPtr( "Ports", NULL, 0, &dwGuids);

  if(0 == dwGuids)
   { return -2;  // ERROR !
   }

  // Allocate the needed memory
  pGuid = (GUID*)HeapAlloc(GetProcessHeap(), HEAP_GENERATE_EXCEPTIONS, dwGuids * sizeof(GUID));

  if(NULL == pGuid)
   { return -3;  // ERROR !
   }

  // Call the function again, this time not to FIND THE LENGTH but GET THE GUID
  if (FALSE == SetupDiClassGuidsFromNameFunPtr("Ports", pGuid, dwGuids, &dwGuids))
   { return -4;  // ERROR !
   }

  hDevInfoSet = SetupDiGetClassDevsFunPtr(pGuid, NULL, NULL, DIGCF_PRESENT );
  if (INVALID_HANDLE_VALUE == hDevInfoSet)
   { return -5;  // ERROR !
   } // end if < hDevInfoSet INVALID >

  devInfo.cbSize = sizeof(SP_DEVINFO_DATA);

  do
   {
     HKEY hDeviceKey;
     BOOL foundPort = FALSE;

     if( ! SetupDiEnumDeviceInfoFunPtr(hDevInfoSet, iSetupItem++, &devInfo) )
      { break;  // through with all "setup items" that we may TEST
      }

     hDeviceKey = SetupDiOpenDevRegKeyFunPtr(hDevInfoSet, &devInfo,
                       DICS_FLAG_GLOBAL, 0, DIREG_DEV, KEY_QUERY_VALUE);

     if (INVALID_HANDLE_VALUE != hDeviceKey)
      {
        size_t nLen;
        LPTSTR pszPortName;
        DWORD  dwType, dwDataSize, dwAllocatedSize, dwReturnedSize;
        LONG   err;

        pszPortName = NULL;
#      ifdef __BORLANDC__  // "... is assigned a value that is never used".. oh shut up, we know what we're doing !
        (void)pszPortName;
#      endif

        // First query for the size of the registry value
        dwType = 0; dwDataSize = 0;
        err = RegQueryValueEx(hDeviceKey, "PortName", NULL,
                                 &dwType, NULL, &dwDataSize);
           // -> often got dwType=1=REG_SZ and dwDataSize=6 at this point,
           //    when pszPortName (retrieved further below) was e.g. "COM20".

        if (ERROR_SUCCESS != err)
         { continue;
         }
        // Ensure the value is a string
        if (dwType != REG_SZ)
         { continue;
         }

        // Allocate enough bytes for the return value
        dwAllocatedSize = dwDataSize + sizeof(char);
        // +sizeof(char) is to allow us to NULL terminate
        //   the data if it is not null terminated in the registry

        pszPortName = (LPTSTR)LocalAlloc(LMEM_FIXED, dwAllocatedSize);

        if (pszPortName == NULL)
         { continue;
         }
        memset( pszPortName, 0, dwAllocatedSize ); // clear garbage, and
        // provide a trailing zero for the C string (in case the REGISTRY won't)

        // Recall RegQueryValueEx to return the data
        dwReturnedSize = dwAllocatedSize;

        err = RegQueryValueEx(hDeviceKey, "PortName", NULL,
                     &dwType, (LPBYTE)pszPortName, &dwReturnedSize);

        if (ERROR_SUCCESS != err)
         {
           LocalFree(pszPortName);
           pszPortName = NULL;
           (void)pszPortName;  // "assigned a value that is never used" .. oh, shut up, Mr Pedantic !

           continue;
         }

        // Handle the case where the data just returned is the same size
        // as the allocated size. This could occur where the data
        // has been updated in the registry with a non null terminator
        // between the two calls to ReqQueryValueEx above. Rather than
        // return a potentially non-null terminated block of data,
        // just fail the method call (.. hmm.. not a method call anymore..) :
        if (dwReturnedSize >= dwAllocatedSize)
         { continue;  // alarm alarm .. something is really broken !
         }

        // Because the entire ALLOCATED BLOCK had been zeroed further above,
        // there's no need to speculate here - it WILL have a trailing zero-byte.

        // If it looks like "COMX" then add it to the array which will be returned
        nLen = strlen(pszPortName);

        if( (nLen > 3) && ((int)nLen < iSizePerPerName) ) // "COM" followed by at least one digit (but expect more)
         {
           if( (SL_strnicmp(pszPortName,"COM",3)==0) && SL_IsDigit(pszPortName[3]) )
            { // Pass back the (unfriendly) port name :
              SL_strncpy(pPortName + nComPortsFound*iSizePerPerName, pszPortName, nLen+1 );
              foundPort = TRUE;
            }
         } // end if < long enough for a "COM port name", but not too long for the caller's array > ?

        LocalFree(pszPortName);

        // Close the key now that we are finished with it
        RegCloseKey(hDeviceKey);
      } // end if < SetupDiOpenDevRegKey() successful >


     if( foundPort )// If the port was a serial port, then also try to get its friendly name
      { char szFriendlyName[1024];
        DWORD dwSize = sizeof(szFriendlyName);
        DWORD dwType = 0;
        memset( szFriendlyName, 0, sizeof(szFriendlyName) ); // trailing zero, if not set by "Setup Di Get Device Blabla"..
        if( SetupDiGetDeviceRegistryPropertyFunPtr(hDevInfoSet, &devInfo,
                      SPDRP_DEVICEDESC, &dwType, (PBYTE)(szFriendlyName),
                      dwSize, &dwSize )
            && (REG_SZ == dwType) )
         { SL_strncpy(pFriendName + nComPortsFound*iSizePerPerName, szFriendlyName, iSizePerPerName );
         }
        else
         {
           *(pFriendName + nComPortsFound*iSizePerPerName) = '\0'; // no "friendly name" (empty string in the output)
         } // end if < SetupDiGetDeviceRegistryProperty() successful >
        nComPortsFound++;
      } // end if < found a valid COM port (at least the "unfriendly" name) >
   }while( nComPortsFound < iMaxComPorts );

  HeapFree(GetProcessHeap(), 0, pGuid);

  return nComPortsFound;
} // end of the former "EnumerateComPortSetupAPISetupDiClassGuidsFromNamePort()"

//---------------------------------------------------------------------------
BOOL UTL_EnumerateComPorts( // Older implementations caused SOOO MANY HEADACHES ..
        int  *piEnumerator,   // [in,out] zero-based index as "enumerator"
        char* pszComPortName, // [out] with iMaxLen between 6 and 10, only "COM0".."COM255";
                              //       with iMaxLen > 10, possibly with a
                              //       shortened "friendly name" in parentheses
                              //       (easy to skip when parsing), e.g.:
                              //             "COM11 (USB Serial Port)"
                              //             instead of just "COM11" .
        int iMaxLen )         // [in] max length of pszComPortName;
                              //      also controls appending a "friendly name" !
  // How to use (like other 'simple enumerators' that don't need a callback):
  //  > int iEnumerator = 0;  // initialize 'enumerator'
  //  > char sz80[84];
  //  > while( UTL_EnumerateComPorts( &iEnumerator, sz80, 80 ) )
  //  >  { // .. do something with the "COM port name", e.g. "COM11" ..
  //  >  )
  // Don't assume ANYTHING about the enumerator's value and "stepwidth" !
  // Just keep calling UTL_EnumerateComPorts( &iEnumerator, .. ) until that function
  // returns FALSE .
  // First used in Spectrum Lab (2023-08), where 'COM ports' are listed in the
  //  "Audio Input Devices" (where they may be connected to an A/C converter,
  //  interfaced with a simple microcontroller with a UART. With the
  //  "CB_SETDROPPEDWIDTH" trick, even a narrow TComboBox can show the complete
  //  name (with the "friendly part appended") in full length when DROPPED DOWN.
  // Since 2024-01-05 (when the 'Remote CW Keyer' was unable to retrieve the
  //  'friendly' names of COM ports under Windows 11), UTL_EnumerateComPorts()
  //  tries to leave a note about what went wrong in UTL_sz255LastError[] .
  //  For some reason, the problem disappeared after rebuilding all with an
  //       absurdly large 'min stack size' and 'min heap size' in Borland's
  //       *LINKER* options for the project !
{
#  define MAX_PORT_NUM 20
#  define MAX_STR_LEN  84 // 2 * 20 * 84 = 3360 bytes for the static arrays below
  static int nPorts;
  static char portNames[MAX_PORT_NUM][MAX_STR_LEN];
  static char friendlyNames[MAX_PORT_NUM][MAX_STR_LEN];
  char* pszDest    = pszComPortName;
  char* pszEndstop = pszComPortName + iMaxLen - 1;

  *pszDest = '\0';
  if( *piEnumerator == 0 ) // Caller asks for the FIRST entry ?
   { nPorts = EnumerateComPortsWithFriendlyNames( MAX_PORT_NUM, MAX_STR_LEN, portNames[0], friendlyNames[0] );
   }
  if( (*piEnumerator >= 0 ) && (*piEnumerator < nPorts ) && (*piEnumerator < MAX_PORT_NUM ) )
   { SL_AppendString( &pszDest, pszEndstop, portNames[*piEnumerator] );  // hopefully something like "COM1", "COM2", ..
     if( friendlyNames[*piEnumerator][0] != '\0' )  // Got a "friendly name", too ?
      { SL_AppendString( &pszDest, pszEndstop, " (" );
        SL_AppendString( &pszDest, pszEndstop, friendlyNames[*piEnumerator] );
        SL_AppendString( &pszDest, pszEndstop, ")" );
      }
     (*piEnumerator)++;  // increment the 'enumerator' for the NEXT call
     return TRUE;
   }
  return FALSE;
} // end UTL_EnumerateComPorts()


//--------------------------------------------------------------------------
int UTL_ParseNumberFromComPortName( const char* pszComPortName)
   // Returns a NON-NEGATIVE "COM Port number" if pszComPortName
   // is obviously the name of a COM port (with an without a 'friendly name'),
   // otherwise a negative result indicating NOT THE NAME OF A COM PORT
   // (neither "friendly" nor "short").
   // [in] pszComPortName : e.g. "COM11" or "COM11 (USB Serial Port)"
   // [return] COM port "number", e.g. 11 for the above example
   //
   // Note: Because Windows, or certain "virtual COM port" / "virtual serial port emulators",
   //       sometines NEITHER appear in the 'Device Manager',
   //       NOR will they be enumerated by UTL_EnumerateComPorts(),
   //       the user MAY have to type the entire COM port name into the
   //       COMBO LISTS (with VCL: "TComboBox.Style" set to 'csDropDown',
   //       not 'csDropDownList'). Because of that, expect users to misspell
   //       e.g. "COM6" as "com7" or even "Com7" here ! Thus,
   //       the token "COM" (before the "COM port number") must be skipped
   //       in a case-insensitive way. See details about that annoyance in
   //       C:\cbproj\Remote_CW_Keyer\Keyer_GUI.cpp : FillComboWithSerialPorts() .

{
  int iComPortNumber = -1;
  const char* cp = pszComPortName;
  // "cp" is a pointer to a "constant char" (here: an array of chars that are CONSTANT).
  //  > int SL_SkipString(const char **ppszSrc, const char *pszStringToSkip);
  //     ,----------------|____________|
  //     '--> expects  "a pointer to a pointer to a const char".
  //          Below, we take the address of a "const char*" (cp),
  //                 which should result in a "const char **",
  //                 but BCB6 complained "suspicious pointer conversion" !
  if( SL_SkipString_AnyCase( &cp, "COM" ) ) // ,------'-----,
   { iComPortNumber = SL_ParseInteger(         (const char**)  &cp );
     // Depending on the presence of a "friendly name" in parentheses,
     // the COM port number (already parsed) may be followed by any of
     // the following delimiters:
     if( (*cp!='\0') && (*cp!=' ') && (*cp!='(') && (*cp!='[') && (*cp!=':')
      && (*cp!=';')  && (*cp!=',') && (*cp!='*') )
      { iComPortNumber = -1; // not the expected delimiter -> not the name of a "COM port" !
        // For example, the Remote CW Keyer's function FillComboWithSerialPorts()
        // will add a warning after the COM port string, e.g.
        //      "COM234 *NOT ENUMERATED*"
        // if the CURRENTLY CONFIGURED port was not enumerated by UTL_EnumerateComPorts().
        // Thus the SPACE CHARACTER, the ASTERISK, BRACKETS / BRACES, etc,
        // are all considered "valid delimiters" in the long list above.
      }
   }
  return iComPortNumber;
} // end UTL_ParseNumberFromComPortName()


//--------------------------------------------------------------------------
int UTL_CompareComPortNames( const char* pszComPortName1, const char* pszComPortName2 )
   // Added 2023-08-16 because UTL_EnumerateComPorts() *may* have appended
   //      a COM port's "friendly name" AFTER the significant part,
   //      e.g. "COM11 (USB Serial Port)" instead of "COM11",
   //      compare the names with UTL_CompareComPortNames() .
   // Returns the "COM port number" as an integer if BOTH NAMES actually
   // mean "the same port"; otherwise a negative result indicating NO MATCH.
{ int n1 = UTL_ParseNumberFromComPortName(pszComPortName1);
  int n2 = UTL_ParseNumberFromComPortName(pszComPortName2);
  return (n1==n2) ? n1 : -1;
} // end UTL_CompareComPortNames()


//--------------------------------------------------------------------------
// REMOVED FROM Remote_CW_Keyer/Utilities.c :
//    Helpers for Microsoft's "RichText" edit control (without VCL)
//--------------------------------------------------------------------------



//--------------------------------------------------------------------------
// Helpers to deal with old-fashioned INI-files (for 'small config files')
//--------------------------------------------------------------------------

//--------------------------------------------------------------------------
const char* UTL_IntToStr(long i32)  // not really safe for MULTI-THREADING ...
{ static char sz15Result[16];
  wsprintf(sz15Result,"%ld",(long)i32 );
  return sz15Result;
}

//--------------------------------------------------------------------------
long UTL_ReadIntFromIniFile(
       const char* pszIniFileName, // [in] INI file name, these days WITHOUT A PATH,
                              //      see details in the implementation below.
       const char* pszSection, const char* pszKeyName, long i32Default)
{
  return GetPrivateProfileInt(  // what an ugly argument sequence in this WinAPI routine..
    // section, key name,   default value, ini-file name :
    pszSection, pszKeyName, i32Default,    pszIniFileName);
  // Because the days where Windows allowed an application to place
  // "anything it needs to work" in ITS OWN folder (e.g. "all in one place")
  // are over, don't specify a path in pszIniFileName. That way, NO AVERAGE
  // USER will ever find the file (been there, tried that), but at least,
  // the INI-file will be writeable, and at least the dreadful operating system
  // will be able to find it.
  // For example, when running an application under debugger control
  // with Borland C++ Builder V6 (which needs to be run 'as admin' because
  // otherwise, almost nothing works at all), a file named "Remote_CW_Keyer.ini"
  // was written by Windows into
  //

} // end ..ReadIntFromIniFile()

//--------------------------------------------------------------------------
void UTL_WriteIntToIniFile( const char* pszIniFileName, const char* pszSection,
                            const char* pszKeyName, long i32Value)
{ char sz31Temp[32];
  wsprintf(sz31Temp, "%ld", (long)i32Value);
  WritePrivateProfileString( pszSection, pszKeyName, sz31Temp, pszIniFileName);
} // end ..WriteIntToIniFile()

//--------------------------------------------------------------------------
void UTL_ReadStringFromIniFile(const char* pszIniFileName, const char* pszSection,
                               const char* pszKeyName, const char* pszDefault,
                               char* pszResult, int iMaxLen )
{
  GetPrivateProfileString(  // what an ugly argument sequence in this WinAPI routine..
    // section, key name,   default value,  returnedString, maxLen, ini-file name :
    pszSection, pszKeyName, pszDefault,      pszResult,   iMaxLen,   pszIniFileName);
} // end ..ReadStringFromIniFile()

//--------------------------------------------------------------------------
void UTL_WriteStringToIniFile(const char* pszIniFileName, const char* pszSection,
                              const char* pszKeyName, const char* pszValue )
{
  WritePrivateProfileString( pszSection, pszKeyName, pszValue, pszIniFileName);
} // end UTL_ReadStringFromIniFile()


//---------------------------------------------------------------------------
BOOL UTL_GetPathAndNameOfExecutable( char* pszDest, int iMaxLen )
        // -> e.g. C:\Remote_CW_Keyer\Remote_CW_Keyer.exe
{
  memset( pszDest, 0, iMaxLen );  // make sure there's a TRAILING ZERO !
  if( GetModuleFileName(0, pszDest, iMaxLen ) > 0 )
   { // '--> Retrieves the fully qualified path for the file that contains
     //    > the specified module. The module must have been loaded by the
     //    > current process.
     //    > If the first parameter ("hModule") is NULL, GetModuleFileName
     //    > retrieves the path of the executable file of the current process.
     return TRUE;
   }
  else
   { return FALSE;
   }
} // end UTL_GetPathAndNameOfExecutable()



//--------------------------------------------------------------------------
// Helpers to deal with files in what MS now call "AppData" or "ProgramData"
//    (for 'larger config files', and anything that we do NOT
//     want to store in a windoze-specific *.ini file or even
//     in the triple-stupid "registry", e.g. ..
//       * logfiles
//       * configuration files that can be easily exchanged via email, etc
//       * HTML templates accessed by an application's built-in web server
//       * HTML files used as the application's own, built-in help system
//--------------------------------------------------------------------------

  // From an explanation about where applications should store "their files"
  // on windows (last not least because the windoze-"Programs" folder is
  // usually NOT WRITEABLE by the application itself):
  //
  // >  "AppData" (in three flavours, "Local, LocalLow, and Roaming")
  // >  ---------
  // > Windows applications often store their data and settings in an AppData
  // > folder, and each Windows user account has its own.
  // > It's a hidden folder, so you'll only see it if you show hidden files
  // > in the file manager.
  //               [ OH MY GOD. "Our users are all dumb; taking a look at their
  //                             files in 'Exlorer' would only confuse them,
  //                             so we better hide the files that THEY SAVED."
  //                             .. Besides that, they shall save EVERYTHING
  //                             in our cloud server, not on their harddisks.]
  //                 This is almost as stupid as hiding known file extensions
  //                 per default in the triple-stupid 'File Explorer'. ]
  // > There are actually three folders inside AppData, and different programs
  // > store different types of settings in each. Open your AppData folder and
  // > you'll see [nothing, because it's hidden] Local, LocalLow, and Roaming
  // > folders.  (....)
  // >
  // > "ProgramData" (formerly known as "All Users")
  // > -------------
  // > If a program wants to have a single set of settings or files
  // > that are used by multiple users, it should use the ProgramData folder
  // > instead. This was known as the "All Users" AppData folder in previous
  // > versions of Windows. For example, an antivirus application might keep
  // > its scan logs and settings in ProgramData and share them with all users
  // > on the PC.
  // >
  // > Back in the days of Windows 95, 98, and XP, programs often stored
  // > their settings and other data in their own folders. So, if you installed
  // > a program named "Example" to C:\Program Files\Example, that application
  // > might just store its own settings and other data files at
  // > C:\Program Files\Example, too. This isn't great for security.
  // > Modern versions of Windows limit the permissions programs have,
  // > and applications shouldn't be able to write to system folders
  // > during normal operation.
  //            [That's why some developers were fed up, and installed
  //             their programs to e.g. C:\MyWonderfulProgram,
  //             and decided to have ALL IN ONE PLACE, which was beautifully
  //             simple, because this allowed copying THE ENTIRE APPLICTION,
  //             along with EVERYTHING the application needed to e.g. a
  //             memory stick, to carry it along run it ANYWHERE YOU WANTED.
  //             In other words, create a "portable app" that doesn't rely on
  //                             all those messy special folders in Windows.
  //             But hold on, it gets even more silly if you care about
  //             compatibility, or still have a cloud of 'old workhorses'
  //             in the lab which you don't want to kill by installing Win10,
  //             or throw aways just because they won't run Win11. OMG. ]
  // > On Windows XP, there was no C:\ProgramData folder. Instead, there was a
  // >     "C:\Documents and Settings\All Users\Application Data" folder.
  // > Starting with Windows Vista, the All Users application data folder
  // >     was moved to "C:\ProgramData".
  // > You can still see this today. If you plug C:\Users\All Users\ into
  // > File Explorer or Windows Explorer on Windows 10, Windows will
  // > automatically redirect you to the C:\Program Data folder.
  // > It'll redirect any program that tries to write to C:\Users\All Users\ to
  // > the C:\ProgramData folder, too.
  //     The question is if all those portable low-level file I/O functions
  //     in the C runtime library can to the same 'redirection' trick...
  //     .. but even if they did, we want to know the PHYSICAL LOCATION
  //     of the files we READ AND WRITE in this application .
//---------------------------------------------------------------------------
BOOL UTL_GetPathToMyDataFiles( // see long explanation further above
       char* pszDest, int iMaxLen ) // [out] path to whatever Windows thinks we SHOULD use these days
  // Return value:
  //     TRUE  = "I am quite sure that pszDest contains a 'valid file path' ."
  //     FALSE = "Not really sure about pszDest ... had to speculate "

{
  char sz255[256], sz255EnvValue[256];
  char* pszSrc, c, *cp;
  const char *ccp;
  char* pszPathToExecutable;
  int  i,n;
  BOOL fAppInstalledUnderPrograms = FALSE;
  BOOL fResult = TRUE;

  // > The GetModuleFileName() function retrieves the full path and filename
  // > for the executable file containing the specified module...
  // We use this as the first clue for where "our files" (along with our application)
  // may be, including a refreshingly old-school 'complete directory path' :
  memset( sz255, 0, sizeof(sz255) );  // make sure there's a TRAILING ZERO !
  if( GetModuleFileName(0, sz255, 255 ) > 0 ) // <- this is WINDOWS, not BORLAND
   { // '--> Retrieves the fully qualified path for the file that contains
     //    > the specified module. The module must have been loaded by the
     //    > current process.
     //    > If the first parameter ("hModule") is NULL, GetModuleFileName
     //    > retrieves the path of the executable file of the current process.
     // Example:
     //   sz255 = "C:\\cbproj\\Remote_CW_Keyer\\Remote_CW_Keyer.exe"
     //            |__________________________|-- C++Builder was so kind
     //           to place the EXECUTABLE file in our 'project directory',
     //           instead of some ever-changing crappy place where
     //           Microsoft / Windows would like to have it (in a place
     //           where by default we don't even have WRITE PERMISSIONS).
     if( (cp=strrchr(sz255, '.')) != NULL )
      { *cp = '\0'; // cut away the trailing ".exe"
        // What remains is e.g. sz255 = "C:\\cbproj\\Remote_CW_Keyer\\Remote_CW_Keyer"
      }

     pszPathToExecutable = sz255;   // -> "path to", without "name of" the executable
     if( (cp=strrchr(sz255, '\\')) != NULL )
      { cp[1] = '\0'; // truncate the FOLDER AFTER the last backward slash
      }
     SL_strncpy( pszDest, sz255, iMaxLen ); // <- initial guess with "ALL IN ONE PLACE"...
        // ,------'
        // '--> e.g. "C:\\cbproj\\Remote_CW_Keyer\\" ,
        //  unless fAppInstalledUnderPrograms is set to TRUE further below.
        //  In that case, the FINAL RESULT (pszDest) WILL BE different,
        //  because for unexperienced windoze users, it's difficult
        //  to make the application's folder for "data files" WRITEABLE .

     // If the user decided NOT to install our application where MS/Windows wants
     // it to be, he's not a dedicated follower of fashion (..compliments..).
     // In that case, he certainly doesn't want to waste hours searching for
     // HIS FILES in a crappy ever-changing virtual folder, which depends on
     // on the windows version, and on the user that had originally installed
     // the application on the ten-year-old "laborary workhorse" .
     cp = sz255;  // Does the "module file name" look familiar (a "full path") ?
     if(  SL_IsLetter(cp[0]) && (cp[1]==':') && (cp[2]=='\\') )
      { // Looks like the begin of an old-fashioned, full directory path.
        // Instead of wasting days finding out how to do this properly
        // (with a set of obscure API functions full of funny geek speak),
        // try a few names that MS have been using over the decades
        // for places where THEY wanted us to install our programs :
        ccp = sz255 + 3; // look for e.g. "Program Files" (or who-knows-what) in "C:\Program Files\MyWonderfulApp" ..
        if( SL_SkipOneOfNTokens( &ccp, WindowsProgramFolderNames) > 0 )
         { // Gotcha... no need to fool around with environment variables ...
           fAppInstalledUnderPrograms = TRUE;
         }
        else // Not in "Program Files"/"Programme"/"Programmi"/.... ["(x86)"],
         { // which doesn't mean a lot because there are hundreds of other languages
           // (besides enlish, german, italian). Try to match the path against
           // an 'environment' variable :
           // As long as this program is compiled into a 32-bit Win32 application,
           // look for it under "Programs (x86)" in the first place..
           // .. but who knows, Microsoft may decide to get rid of this one day.
           if( (n=GetEnvironmentVariable("PROGRAMFILES(X86)", sz255EnvValue, 255 )) > 0 )
            { //   '--> On Win10 : "C:\\Program Files (x86)"  ...
              if( SL_strnicmp( pszPathToExecutable, sz255EnvValue, n) == 0 )
               { fAppInstalledUnderPrograms = TRUE;
                 fResult = TRUE; // executable path is correct, and "all files still in our own folder"
               }
            }
         }
      } // end if < result from GetModuleFileName() looks like a classic path in "DOS", e.g. C:\Something >
   } // end if < GetModuleFileName() claimed success > ?
  else  // unable to retrieve our own name and path ->
   { fAppInstalledUnderPrograms = TRUE;  // try environment variables like "PROGRAMDATA" further below
     fResult = FALSE;
   }

  if( fAppInstalledUnderPrograms )
   { // The program has been installed under
     //  "Programme" / "Programmi" / "Program Files" or their "(x86)" cousins.
     //  For good reasons, THOSE FOLDERS are all NOT WRITEABLE with normal 'user' privileges.
     //  So back to the long introduction .. where do we find a WRITEABLE folder for us ?
     //  In the shell, there is "%APPDATA%", "%HOMEPATH%", "%LOCALAPPDATA%",
     //                         "%PROGRAMDATA%" .  The author decided to use the latter:
     if( (n=GetEnvironmentVariable("PROGRAMDATA", pszDest, iMaxLen-20 )) > 0 )
      { //   '--> On Win10 : "C:\\Program Files (x86)"  ...
        if( ((c=SL_GetLastChar(pszDest,iMaxLen-20)) != '\\') && (c!='/') )
         { cp = pszDest;
           SL_AppendChar( &cp, pszDest+iMaxLen-1, '\\' );
         }
      }
   }
  return fResult;
} // end UTL_GetPathToMyDataFiles()


/* EOF <Utilities.c> */

