//---------------------------------------------------------------------------
// File:     C:\cbproj\Remote_CW_Keyer\Translator.cpp
// Purpose:  Translates individual strings
//           or an entire Borland VCL "form"
//           frome one language (english) into another language.
// Notes/Details:
//  * Unfortunately neither Borland's VCL = "Visual Component Library"
//    nor their IDE [Borland C++ Builder V6] supported UTF-8,
//    thus we're limited to the "Ansi Character Set").
//  * To avoid having to store all strings in a separate file
//    (*required* to load at run-time), the APPLICTION SPECIFIC TRANSLATIONS
//    are located in a simple C (not C++) module
//     - see C:\cbproj\Remote_CW_Keyer\Translations.c .
//  * To simplify testing new translations, the translation table
//    can be loaded AT RUNTIME, if the *C MODULE* (Translations.c)
//    is located in the directory of the executable binary (*.exe).
//    For that purpose, the 'Translator' module contains a simplistic parser
//    for the string table (const char *TranslationTable[] in Translations.c)
//     - see APPL_LoadTranslationsFromFile() .
//
//---------------------------------------------------------------------------



//---------------------------------------------------------------------------
#include <vcl.h>
#include <buttons.hpp>   // TBitBtn
#include <ComCtrls.hpp>  // TTabSheet
#include <CheckLst.hpp>  // TCheckListBox
#include <Grids.hpp>     // TStringGrid
#include <IniFiles.hpp>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <stdarg.h>

#pragma hdrstop

#include "Translator.h"
#include "Translations.h" // implementation of const char *TranslationTable[]
#include "StringLib.h"    // DL4YHF's portable 'string library'
#include "QFile.h"        // DL4YHF's portable 'Quick File I/O'

extern "C" void ShowError( int iErrorClass, const char * pszFormat, ... );
  //                             |   ,------|__________|
  // ,---------------------------'   '--> Not just "char *" anymore, for reasons
  // |                                   explained in ?/YHF_Tools/StringLib.h !
  // '--> ERROR_CLASS_FATAL, ERROR_CLASS_ERROR, ERROR_CLASS_WARNING, ERROR_CLASS_INFO .
  //      If an optional 'debug run log' (file) is supported,
  //      iErrorClass bitmask SHOW_ERROR_IN_RUN_LOG can be used to emit
  //      the same message in "debug_run_log.txt", besides the GUI's "Debug" tab.


//---------------------------------------------------------------------------
//   Variables (global, say "yucc" if you feel better then ...)
//---------------------------------------------------------------------------

int  APPL_iLanguage = TRANSLATOR_LANGUAGE_ENGLISH;
int  APPL_iPrevLanguage = TRANSLATOR_LANGUAGE_ENGLISH;

int  APPL_iLanguageTestMode = 0;  // 0=normal operation,
      //      1=highlight translated controls and list missing translations .
      //  Will be set in the main form via command line switch "\xlate" .
      //  If this flag is set, a list of all untranslated strings will be
      //  dumped into the main window's error log, along with the MODULE NAME.
      //  Missing strings can easily be fixed in TRANSLATIONS.C then . )
char APPL_sz255UntranslatedString[256];
int  APPL_iTranslationOk = 0;     // additional result from APPL_TranslateString()

#define MAX_LINES_IN_TRANSLATION_TABLE 2048
static char *TranslationTableFromFile[MAX_LINES_IN_TRANSLATION_TABLE] = { 0 };

//---------------------------------------------------------------------------
//  Subroutines
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// void  APPL_TranslationError( char *pszUnknownId )
// {
//
// } // end APPL_TranslationError()

//---------------------------------------------------------------------------
const char *APPL_CountryCallingCodeToPrefix( int iLanguage )
  // [in] iLanguage : NUMERIC "calling calling code" as shown at
  //         https://en.wikipedia.org/wiki/List_of_country_calling_codes,
  //         shown in parenthesis in the list below .
  // Returns a string prefix in the format used in TranslationTable[],
  //   e.g.: "fr:" France (33),  "es:" Spain (34),   "pt:" Portugal (351),
  //         "it:" Italy  (39),  "nl:" Netherlands (31),
  //         "dk:" Denmark(45),  "se:" Sweden(46),   "no:" Norway (47),
  //         "de:" Germanly(49), "pl:" Poland(48),
  //         "gb:" Great Britain (44), politically incorrect treated as United Kingdom AND the USA (1),
  //          etc.
  //         The TWO-LETTER CODE is actually ISO 3166-1 "alpha-2",
  //         see https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2 .
  // Note: Due to the C compiler (Borland C++), which doesn't support anything
  //       but "ANSI", only roman languages are supported.
  // If iLanguage is not recognized (or not supported),
  //    APPL_CountryCallingCodeToPrefix() returns and EMPTY STRING .
{
  switch( iLanguage )
   { case 31: return "nl:";
     case 33: return "fr:";
     case 34: return "es:";  // Espana, not Estonia !
     case 39: return "it:";
     case 44: return "gb:";  // "Great Britain" (we "speak British", not US english, sorry fellas)
     case 45: return "dk:";
     case 46: return "se:";  // Sweden
     case 47: return "no:";  // Norway
     case 48: return "pl:";  // Poland
     case 49: return "de:";  // Germanly
     case 351: return "pt:"; // Portugal
     case 358: return "fi:"; // Finland
     case 420: return "cz:"; // Czech Republic
     case 421: return "sk:"; // Slovakia ("SL" is Sierra Leone..)

     // Note: Many countries not listed here are impossible to support
     //       due to the 8-bit ANSI restriction in Borland C++ / C++Builder V6 !

     default: return "";
   } // end switch( iLanguage )
} // end APPL_CountryCallingCodeToPrefix()


//---------------------------------------------------------------------------
const char *APPL_TranslateString( int iTranslateFrom,
                     const char * pszString2Translate )
  // [in] APPL_iLanguage: language TO TRANSLATE INTO, e.g. 49 = TRANSLATOR_LANGUAGE_GERMAN.
{ // Called via macro TS(x) = TranslateString  and  TE(x) = TranslateFromEnglish.
  // If the string-identifer is not found, the identifier itself is returned,
  // because in that case it's most likely that no "translation" is avaliable.
  // In language test mode (APPL_iLanguageTestMode>0),
  //    the string is copied to APPL_sz255UntranslatedString if it looks
  //    like something which 'should be translatable'.
  const char **ppsz = TranslationTable;  // <- "default" : Hard-coded table in Translations.c
  const char *psz;
  const char **ppszSubStrings;
  int  iTranslateTo = APPL_iLanguage;
  int  iSubstringIndex;
  const char *pszFromPrefix = APPL_CountryCallingCodeToPrefix(iTranslateFrom);
  const char *pszToPrefix   = APPL_CountryCallingCodeToPrefix(iTranslateTo);
  BOOL found_match;

  if( TranslationTableFromFile[0] != NULL )  // use the "loaded" table ..
   { ppsz = (const char**)TranslationTableFromFile; // .. from APPL_LoadTranslationsFromFile() ?
   }


  // Note: The FIRST TWO ENTRIES per language don't use the "alpha-2"
  //       prefixes, but fixed offset:
  //        * the first sub-string is ENGLISH
  //        * the second sub-string is GERMAN
  //        * all other sub-strings may be any other language
  //        * NULL marks the end of a set of sub-strings
  //        * empty strings in TranslationTable[] are "placeholders"
  //           for NOT-YET-TRANSLATED entries .

  APPL_iTranslationOk = 1;                 // by default: "translation was ok"

  if(APPL_iLanguageTestMode)
   { APPL_sz255UntranslatedString[0]='\0';
   }

  // Some "special" strings must never be translated..
  if(APPL_iLanguage==iTranslateFrom)
    return pszString2Translate;            // no need to translate
  if(pszString2Translate[0]=='_')
    ++pszString2Translate;                 // not able to translate, skip "error"-marker
  if(strlen(pszString2Translate)<=2)
    return pszString2Translate;            // not worth to translate


  // Repeat search loop until translation found or end of table..
  while( ((psz=*ppsz)!=NULL) && (*(ppsz+1)!=NULL) ) // end of table marked by a "DOUBLE NULL" (pointer)
   {
     found_match = FALSE;

     if( (strcmp( psz, pszString2Translate)==0)
       &&(psz[0]!='\0') ) // don't accept empty translations (sub-strings)
      { // Ok, found the matching table entry for ENGLISH MASTER ->
        found_match = TRUE;
      }
     if( (!found_match )  // Compare the SUB-STRINGS against pszString2Translate ?
      && ( (iTranslateFrom != 0) && (iTranslateFrom != TRANSLATOR_LANGUAGE_ENGLISH) ) )
      { // yes, because the (Borland-VCL-)"Form" HAS ALREADY been translated...
        ppszSubStrings = ppsz+1;
        while( (!found_match) && (psz=*(ppszSubStrings++))!=NULL )
         { if( strcmp( psz, pszString2Translate)==0 )
            { found_match = TRUE; // bingo, found a match WITHOUT language-prefix
              // (originally used for GERMAN in the 1st sub-string, but ANY sub-string will do)
            }
           else if( strncmp(psz, pszFromPrefix, 3) == 0 ) // at least the LANGUAGE is ok..
            { if( strcmp( psz+3, pszString2Translate)==0 ) // .. and THE REST OF THE STRING matches..
               { found_match = TRUE; // bingo, found a match WITHOUT the correct language-prefix
               }
            }
         } // end while( (!found_match) && ...
      } // end if < compare SUB-STRINGS against pszString2Translate > ?

     if( found_match ) // this string-table-entry matches pszString2Translate ->
      { // Run through all sub-strings to find the right language..
        if( (iTranslateFrom<=0) || (iTranslateFrom == TRANSLATOR_LANGUAGE_ENGLISH) )
         { ppszSubStrings = ppsz+1;  // exclude the ENGLISH string (main string) in the loop below
           iSubstringIndex = 1;  // 1 = index of the german translation (first substring)
         }
        else // INCLUDE english in the search-loop further below
         { ppszSubStrings = ppsz;
           iSubstringIndex = 0;  // 0 = index of the english text ("master")
         }
        while( (psz=*(ppszSubStrings++))!=NULL )
         { if( strncmp(psz, pszToPrefix, 3) == 0 ) // bingo, found the matching prefix ..
            { return psz+3; // return whatever follows AFTER the prefix
            }
           // Extrawurst fr die DEUTSCHE bersetzung :
           if( iTranslateTo==TRANSLATOR_LANGUAGE_GERMAN )
            { if( (iSubstringIndex==1) && (psz[0]!='\0') )  // valid sub-string for GERMAN ?
               { return psz; // return the "lazy" german translation w/o language prefix
               }
            }
           // 'Extra sausage' to translate BACK INTO ENGLISH :
           if( (iTranslateTo==TRANSLATOR_LANGUAGE_ENGLISH ) || (iTranslateTo<=0) )
            { if( (iSubstringIndex==0) && (psz[0]!='\0') ) // valid sub-string for ENGLISH ?
               { return psz; // return the "master" string (english text)
               }
            }

           ++iSubstringIndex;
         }
        // Arrived here ? Found the matching string table entry for pszString2Translate,
        //   but there's no translation into the wanted language !
        //   We'll care for that AFTER the loop (-> ..UntranslatedString ).
      } // end if( found_match )

     // Skip all sub-strings (translations) for this entry:
     while( *ppsz!=NULL )
      { ++ppsz;
      }
     ++ppsz; // skip the SINGLE NULL POINTER that marks the end of SUB-STRINGS
     // (but not the DOUBLE NULL POINTER that marks the end of the entire table)

   } // end while( (psz=*(ppsz))!=NULL )



  APPL_iTranslationOk = 0;

 // if(APPL_iLanguageTestMode)
 //  {
      strncpy(APPL_sz255UntranslatedString, pszString2Translate, 255);
      APPL_sz255UntranslatedString[255]='\0'; // always terminate, sz255 has 256 byte
 //  }


  return pszString2Translate; // no string alternative found; use the "identifier" itself
} // end APPL_TranslateString()



//---------------------------------------------------------------------------
// int APPL_TranslateMainMenu( TMainMenu *pMenu )
// Tried to translate all strings in a Borland VCL-style "main menu" .
// But unfortunately there seems to be NO WAY to loop through a Borland-Style
// "TMainMenu" to modify all entries. What a shame !
// {
//   return 0;
// } // end APPL_TranslateMainMenu()


int  APPL_iCountTranslationErrors;

//---------------------------------------------------------------------------
void APPL_ShowTranslationError( AnsiString sFormName, TComponent *pComp )
{
  if(APPL_iCountTranslationErrors==0)
   { ShowError( 127/*level:IMPORTANT*/,
     "------ UNTRANSLATED STRINGS FOUND (FromLang=%d, ToLang=%d) ------",
     (int)APPL_iPrevLanguage, (int)APPL_iLanguage );
   }
  ++APPL_iCountTranslationErrors;

  // Format the untranslated string so it can be easily copied into the
  // translation table (like in TRANSLATION.C) .
  // A valuable help for internationalizing the whole project .
  ShowError( 127/*level:IMPORTANT*/,"  \"%s\",  // in %s.%s",
      APPL_sz255UntranslatedString,
      sFormName.c_str(), pComp->Name.c_str()  );
} // end APPL_ShowTranslationError()


//---------------------------------------------------------------------------
void APPL_TranslateComponent( AnsiString sFormName, TComponent *pComp )
{
  int iItemIndex;
  AnsiString s;

  /* Check if the component is a TLabel:   */
  if (pComp->ClassNameIs("TLabel") )
   {
           // cast the component to a TStringGrid pointer
           TLabel *label = (TLabel*)pComp;
           // Translate the label's CAPTION :
           label->Caption = TS(AnsiString(label->Caption).c_str());
           //   ,--------------|_________|
           //   '--> Important when compiling with e.g. C++Builder V12 "Athens",
           //    where regardless of the mapping of _TCHAR to wchar_t or char,
           //    many of these VCL functions don't return an AnsiString anymore
           //    but a UnicodeString. The "c_str()" method of the latter
           //    always returns a pointer to a 16-bit "wide character" string,
           //    not the nice old UTF-8 compatible 8-bit "C string" that
           //    every microcontroller developer loves so dearly ;o)
           // Without this conversion to AnsiString before the c_str(),
           // Embarcadero C++Builder V12 "Athens" complained:
           //    "no matching function for call to 'APPL_TranslateString()' ",
           // because that function expects a 'const char *'.
           // not a 'wide char pointer' (whatever a 'wide char' is, 16 or 32 bit).
           // When compiling with the venerable Borland C++Builder V6,
           // all those shiny VCL functions and methods returned an
           // 'AnsiString' because at that time, there wasn't much else,
           // so calling AnsiString() as a constructor turns an AnsiString
           // into a (temporary) AnsiString that hopefully exists long enough
           // for the c_str() method's return value (a char *) until
           //  TE() ["Translate From English", -> APPL_TranslateString()]
           // doesn't need the string anymore. Expect more fun like this
           // in hundreds of similar cases in a complex project.
           //
           if(APPL_iLanguageTestMode)
            { if(APPL_iTranslationOk)
                label->Color =  clGreen;
              else
                label->Color =  clRed;
            }
           else // no 'test mode'; mark missing translations with an underscore
           if(!APPL_iTranslationOk)
            { label->Caption = "_"+label->Caption;
            }
   } /* end if <component is a TLabel>       */
  else
  if (pComp->ClassNameIs("TButton") )
   { TButton *btn = (TButton *)pComp;
     btn->Caption = TS(AnsiString(btn->Caption).c_str());
     if(APPL_iLanguageTestMode)
      { if(!APPL_iTranslationOk)
          btn->Font->Style = TFontStyles() << fsItalic;
      }
     else // no 'test mode'; mark missing translations with an underscore
     if(!APPL_iTranslationOk)
      { btn->Caption = "_"+btn->Caption;
      }
     if(btn->Hint != "")  // translate the button's "HINT" also ?
        btn->Hint = TS(AnsiString(btn->Hint).c_str());
   } /* end if <component is a TButton>           */
  else
  if (pComp->ClassNameIs("TBitBtn") )
   { TBitBtn *btn = (TBitBtn *)pComp;
     btn->Caption = TS(AnsiString(btn->Caption).c_str());
     if(APPL_iLanguageTestMode)
      { if(!APPL_iTranslationOk)
           btn->Font->Style = TFontStyles() << fsItalic;
      }
     else // no 'test mode'; mark missing translations with an underscore
     if(!APPL_iTranslationOk)
      { btn->Caption = "_"+btn->Caption;
      }
     if(btn->Hint != "")  // translate the button's "HINT" also ?
        btn->Hint = TS(AnsiString(btn->Hint).c_str());
   } /* end if <component is a TBitBtn>           */
  else
  if (pComp->ClassNameIs("TRadioButton") )
   { TRadioButton *btn = (TRadioButton *)pComp;
     btn->Caption = TS(AnsiString(btn->Caption).c_str());
     if(APPL_iLanguageTestMode)
      { if(!APPL_iTranslationOk)
           btn->Font->Style = TFontStyles() << fsItalic;
      }
     else // no 'test mode'; mark missing translations with an underscore
     if(!APPL_iTranslationOk)
      { btn->Caption = "_"+btn->Caption;
      }
   } /* end if <component is a TRadioButton>     */
  else
  if (pComp->ClassNameIs("TTabSheet") )
   { TTabSheet *ts = (TTabSheet *)pComp;
     ts->Caption = TS(AnsiString(ts->Caption).c_str());
     // if(APPL_iLanguageTestMode)
      { if(!APPL_iTranslationOk)
            ts->Caption = "_"+ts->Caption;
      }
   } /* end if <component is a TTabSheet>           */
  else
  if (pComp->ClassNameIs("TCheckBox") )
   { TCheckBox *chk = (TCheckBox *)pComp;
     chk->Caption = TS(AnsiString(chk->Caption).c_str());
     if(APPL_iLanguageTestMode)
      { if(APPL_iTranslationOk)
           chk->Color =  clGreen;
        else
           chk->Color =  clRed;
      }
     else // no 'test mode'; mark missing translations with an underscore
     if(!APPL_iTranslationOk)
      { chk->Caption = "_"+chk->Caption;
      }
     if(chk->Hint != "")  // translate the control's "HINT" also ?
        chk->Hint = TS(AnsiString(chk->Hint).c_str());
   } /* end if <component is a TTabSheet>           */
  else
  if (pComp->ClassNameIs("TMenuItem") )
   { // works for entries in a "TMainMenu" and "TPopupMenu" .
     TMenuItem *mi = (TMenuItem *)pComp;
     mi->Caption = TS(AnsiString(mi->Caption).c_str());
     // if(APPL_iLanguageTestMode)
      { if(!APPL_iTranslationOk)
         { if(mi->Caption.c_str()[0]!='_')
              mi->Caption =  "_"+mi->Caption;
         }
      }
   }
  else
  if (pComp->ClassNameIs("TGroupBox") )
   { TGroupBox *gb = (TGroupBox *)pComp;
     gb->Caption = TS(AnsiString(gb->Caption).c_str());
     // if(APPL_iLanguageTestMode)
      { if(!APPL_iTranslationOk)
         { if(gb->Caption.c_str()[0]!='_')
              gb->Caption =  "_"+gb->Caption;
         }
      }
   }
  else
  if (pComp->ClassNameIs("TListBox") )
   { TListBox *lb = (TListBox *)pComp;
     if(lb->Tag & 0x0008)  // tag 8 = translEIGHT ;-)   ["do NOT translate" is tag=16]
      {
        for(iItemIndex=0; iItemIndex<lb->Items->Count; ++iItemIndex)
         { // translate all 'Items' (TStrings) in this listbox..
           s = lb->Items->Strings[iItemIndex];
           s = TS(s.c_str());
           // if(APPL_iLanguageTestMode)
           if(!APPL_iTranslationOk)
            { if(s.c_str()[0]!='_')
                 s =  s+"_";  // APPEND(!) the 'translation error marker' here !
              // (why ? see LB_CompareOperators in NewEventDialog..
              //  the first few characters are used to identify the selected entry)
              APPL_ShowTranslationError(sFormName,pComp);
              APPL_iTranslationOk = 1;  // don't show this error again further down..
            }
           lb->Items->Strings[iItemIndex] = s;
         } // end for <all items>
        if(lb->Hint != "")  // also translate the control's "HINT" ?
         { lb->Hint = TS(AnsiString(lb->Hint).c_str());
         }
      } // end if <component's tag says 'transl8 me'>
   } // end if <component is a TListBox>
  else
  if (pComp->ClassNameIs("TComboBox") )
   { TComboBox *cb = (TComboBox *)pComp;
     if(cb->Tag & 0x0008)  // tag 8 = translEIGHT ;-)
      {
        for(iItemIndex=0; iItemIndex<cb->Items->Count; ++iItemIndex)
         { // translate all 'Items' (TStrings) in this listbox..
           s = cb->Items->Strings[iItemIndex];
           s = TS(s.c_str());
           // if(APPL_iLanguageTestMode)
           if(!APPL_iTranslationOk)
            { if(s.c_str()[0]!='_')
                 s =  s+"_";  // APPEND(!) the 'translation error marker' here !
                 // (why ? see LB_CompareOperators in NewEventDialog..
                 //  the first few characters are used to identify the selected entry)
              APPL_ShowTranslationError(sFormName,pComp);
              APPL_iTranslationOk = 1;  // don't show this error again further down..
            }
           cb->Items->Strings[iItemIndex] = s;
         } // end for <all items>
        if(cb->Hint != "")  // translate the control's "HINT" also ?
         { cb->Hint = TS(AnsiString(cb->Hint).c_str());
         }
       } // end if <component's tag says 'transl8 me'>
   } // end if <component is a TComboBox>
  else
  if (pComp->ClassNameIs("TRadioGroup") )
          { TRadioGroup *rg = (TRadioGroup *)pComp;
            if(rg->Tag & 0x0008)  // tag 8 = translEIGHT ;-)
             {
              rg->Caption = TS(AnsiString(rg->Caption).c_str());
              for(iItemIndex=0; iItemIndex<rg->Items->Count; ++iItemIndex)
               { // translate all 'Items' (TStrings) in this radio group..
                s = rg->Items->Strings[iItemIndex];
                s = TS(s.c_str());
              // if(APPL_iLanguageTestMode)
                if(!APPL_iTranslationOk)
                { if(s.c_str()[0]!='_')
                       s =  s+"_";  // APPEND(!) the 'translation error marker' here !
                    // (why ? see LB_CompareOperators in NewEventDialog..
                    //  the first few characters are used to identify the selected entry)
                  APPL_ShowTranslationError(sFormName,pComp);
                  APPL_iTranslationOk = 1;  // don't show this error again further down..
                }
                rg->Items->Strings[iItemIndex] = s;
               } // end for <all items>
             } // end if <component's tag says 'transl8 me'>
   } // end if <component is a TRadioGroup>
  else
  if (pComp->ClassNameIs("TCheckListBox") )
          { TCheckListBox *cl = (TCheckListBox *)pComp;
            if(cl->Tag & 0x0008)  // tag 8 = translEIGHT ;-)
             {
              for(iItemIndex=0; iItemIndex<cl->Items->Count; ++iItemIndex)
               { // translate all 'Items' (TStrings) in this checklist..
                s = cl->Items->Strings[iItemIndex];
                s = TS(s.c_str());
                // if(APPL_iLanguageTestMode)
                if(!APPL_iTranslationOk)
                  { if(s.c_str()[0]!='_')
                       s =  s+"_";  // APPEND(!) the 'translation error marker' here !
                    // (why ? see LB_CompareOperators in NewEventDialog..
                    //  the first few characters are used to identify the selected entry)
                    APPL_ShowTranslationError(sFormName,pComp);
                    APPL_iTranslationOk = 1;  // don't show this error again further down..
                  }
                cl->Items->Strings[iItemIndex] = s;
               } // end for <all items>
              if(cl->Hint != "")  // translate the control's "HINT" also ?
               { cl->Hint = TS(AnsiString(cl->Hint).c_str());
               }
             } // end if <component's tag says 'transl8 me'>
   } // end if <component is a TCheckListBox>
  else
  if (pComp->ClassNameIs("TStringGrid") )
          { TStringGrid *sgr = (TStringGrid *)pComp;
            if(sgr->Tag & 0x0008)  // tag 8 = translEIGHT ;-)
             {
              if(sgr->Hint != "")  // translate the button's "HINT" also ?
               { sgr->Hint = TS(AnsiString(sgr->Hint).c_str());
               }
             }
          } /* end if <component is a TStringGrid>    */
  else
  if (pComp->ClassNameIs("TEdit") )
   { TEdit *ed = (TEdit *)pComp;
     if(ed->Tag & 0x0008)  // tag 8 = translEIGHT ;-)
      {
        if(ed->Hint != "")  // translate the button's "HINT" also ?
         { ed->Hint = TS(AnsiString(ed->Hint).c_str());
         }
      }
   } /* end if <component is a TEdit>    */
} // end APPL_TranslateComponent()


//---------------------------------------------------------------------------
void APPL_TranslateForm(TForm *pForm) // [in] APPL_iLanguage, APPL_iPrevLanguage.
  // Translates a single 'Form' (window) of the application into <APPL_iLanguage>.
{
 int iForm,iComp,iComp2,count;
 int iItemIndex;
 TComponent *pComp, *pComp2;
 AnsiString s;


  // The "Tag" value of certain VCL components defines if translation is used
  // or not:
  //    (Tag & 8) set means: Translate ALSO THE STRINGS IN THE LIST
  //    (Tag & 16) means:    DO NOT translate me !
     if((pForm->Tag & 16)==0)
      { pForm->Caption = TS(AnsiString(pForm->Caption).c_str());
        if(APPL_iLanguageTestMode && !APPL_iTranslationOk)
         { pForm->Caption = "_"+pForm->Caption;
           if(APPL_iCountTranslationErrors==0)
            { ShowError( 127/*level:IMPORTANT*/,
               "------ UNTRANSLATED STRINGS FOUND (FromLang=%d, ToLang=%d) ------",
               (int)APPL_iPrevLanguage, (int)APPL_iLanguage );
            }
           ++APPL_iCountTranslationErrors;

           // Format the untranslated string so it can be easily copied into the
           // translation table (like in TRANSLATION.C) .
           // A valuable help for internationalizing the whole project .
           ShowError( 127/*level:IMPORTANT*/,"  \"%s\",  // in %s",
              APPL_sz255UntranslatedString, pForm->Name.c_str() );
         }
      }

     for( iComp=0; (iComp<pForm->ComponentCount); iComp++ )
      {
       APPL_iTranslationOk = 1;

       pComp = pForm->Components[iComp];
       if((pComp->Tag & 16) == 0)
        { // Only if this VCL component MAY be translated :
          // Check the type of this component; most need special treatment..

         // Check if the component is a TFrame (needs extra treatment,
         //   because everything located on the 'Script Editor Frame'
         //   was not automatically translated without this) :
    //   if (pComp->ClassNameIs("TFrame") )
    //    { iComp = iComp;  // << set breakpoint here  ... it NEVER fired !
    //    }
         if (pComp->InheritsFrom( __classid(TFrame)) )
          { iComp = iComp;  // << set breakpoint here  ... this one fired !
            // (with pComp = an instance of a TPageEditorFrame;
            //  so iterate through all of this 'frames' components
            //  just as if it was a normal TForm (a la Borland VCL) ?
            //  But a TFrame is not a TForm but a TComponent, but fortunately
            //  a TComponent can have a (formal) array of TComponents itself...
            for( int iComp2=0; (iComp2<pComp->ComponentCount); iComp2++ )
             { // Translate all components of the TFrame thingy :
               APPL_iTranslationOk = 1;
               pComp2 = pComp->Components[iComp2];
               if((pComp2->Tag & 16) == 0)
                { // Only if this VCL component MAY be translated :
                  APPL_TranslateComponent( pComp->Name/*name of the TFrame*/, pComp2 );
                }
               if(  ( APPL_iLanguageTestMode>0 ) // from ini file, [ToolSettings], "LanguageTestMode"
                 && (!APPL_iTranslationOk )
                 && ( APPL_sz255UntranslatedString[0]>0 ) )
                {
                  APPL_ShowTranslationError( pComp->Name/*name of the TFrame*/, pComp2);
                } // end if <show untranslatable string in global error list>
             }
            APPL_iTranslationOk = 1;
          }
         else // not a TFrame, but a normal TComponent :
          { APPL_TranslateComponent( pForm->Name, pComp );
          }

         if(  ( APPL_iLanguageTestMode>0 ) // from ini file, [ToolSettings], "LanguageTestMode"
           && (!APPL_iTranslationOk )
           && ( APPL_sz255UntranslatedString[0]>0 ) )
          {
            APPL_ShowTranslationError(pForm->Name,pComp);
          } // end if <show untranslatable string in global error list>
        } // end if <may translate this component, due to its 'Tag' value>
      } // end for( iComp=0; (iComp<pForm->ComponentCount); iComp++ )


} // end APPL_TranslateForm()



//---------------------------------------------------------------------------
void APPL_TranslateAllForms(void) // [in] APPL_iLanguage, APPL_iPrevLanguage .
  // Translates all 'Forms' (windows) of the application into <APPL_iLanguage> .
  // If the forms have already been tranlated (at runtime),
  //    APPL_iPrevLanguage contains the OLD ("previous") language.
  // Note: The application should not call APPL_TranslateAllForms()
  //                                before APPL_LoadTranslationsFromFile() ;o)
{
 int iForm,iComp,count;
 int iItemIndex;
 TForm *pForm;
 TComponent *pComp;
 AnsiString s;

 APPL_iCountTranslationErrors=0;


  // Loop through all 'Forms' and all their 'Components', to find out which
  // of them need to be translated; using strings from a very own table.
  //  (to avoid the crashy Resource-DLL-Wizard which is too stupid to produce path names)
  // The "Tag" value of certain VCL components defines if translation is used
  // or not:
  //    (Tag & 8) set means: Translate ALSO THE STRINGS IN THE LIST
  //    (Tag & 16) means:    DO NOT translate me !
  count = Screen->FormCount;  // does this work ?
  for(iForm=0; iForm<count; ++iForm)
   {
     APPL_TranslateForm( Screen->Forms[iForm] );
   } // end for<all forms of the application>

  if( (APPL_iLanguageTestMode>0)
     && (APPL_iCountTranslationErrors>0) )
   {
    ShowError( 127/*level:IMPORTANT*/,
      "Found %d untranslated strings in all forms.",(int)APPL_iCountTranslationErrors );
   }

  APPL_iPrevLanguage = APPL_iLanguage;

} // end APPL_TranslateAllForms()


//---------------------------------------------------------------------------
int APPL_LoadTranslationsFromFile( char* pszPathAndFilename )
  // [out] TranslationTableFromFile[] : array of pointers to dynamically allocated strings.
  // Returns the NUMBER OF LINES appended to TranslationTableFromFile[],
  // including the important NULL-pointers that mark the end of a group
  // of sub-strings (strings with different languages for the same english 'master').
  // When successful, char *TranslationTableFromFile[]
  // may be used instead of the hard-coded (linked) const char *TranslationTable[].
{ T_QFile qf;
  BOOL fOk = TRUE;
  int  iCharsPerLine;
  int  nLinesLoaded = 0;
  char sz255Src[256], sz255Dst[256], *cpDst, *cpDstEndstop;
  const char *cpSrc, *cpSrcEndstop;
  char *cp;
  const char *ccp;
  BOOL fInTranslationTable = FALSE;

  APPL_UnloadTranslationsFromFile(); // free old entries in TranslationTableFromFile[],
                                     // and set all string pointers to NULL .
  if( QFile_Open( &qf, pszPathAndFilename, QFILE_O_RDONLY ) )
   { // Successfully opened the file (e.g. "Translations.c") so process it,
     // line-by-line, to keep things simple ..
     while( ( (iCharsPerLine=QFile_ReadLine( &qf, sz255Src, 255/*iMaxLen*/ )) >= 0 )
         && (nLinesLoaded < MAX_LINES_IN_TRANSLATION_TABLE) )
      { if( iCharsPerLine > 0 )  // non-empty line -> parse it !
         { cpSrc        = sz255Src;
           cpSrcEndstop = sz255Src+iCharsPerLine;
           cpDst        = sz255Dst;
           cpDstEndstop = sz255Dst+255;
           SL_SkipSpaces( &cpSrc );
           if( (cpSrc[0] == '/') && (cpSrc[1] == '/') ) // it's a C++ comment so ignore..
            {
            }
           else // Arrived here ? There MAY be a valid C statement in this line..
            {
              //
              // All we're interested in (in Translations.c) is the STRING ARRAY:
              // > const char *TranslationTable[] = // -> fInTranslationTable = TRUE
              // >  {
              // >    "I/O Config",      // comments are skipped ..
              // >        "I/O-Konfig.",
              // >        "fr:Config. I/O",
              // >        "it:Config. I/O",
              // >        "nl:I/O Config",
              // >        NULL,  // NULL marks the end of the substrings (tranlations) for ONE TEXT
              // >     ...
              // >    NULL, NULL // end of table  marked by TWO 'NULL' pointers
              // >  }; // end TranslationTable[]
              if( ! fInTranslationTable ) // look for the begin of the string table :
               { ccp = SL_strnstr_skip( cpSrc, "const char *TranslationTable[]",
                     cpSrcEndstop-cpSrc/*HaystackLength*/, SL_COMPARE_OPTION_NORMAL );
                 if( ccp != NULL ) // found and skipped the array declaraction...
                  { fInTranslationTable = TRUE;
                  }
               }
              else // already INSIDE the translation table :
               { // Look for the next double-quoted string literal, or "NULL" (macro for a NULL-pointer)
                 if( *cpSrc=='"' )
                  { int iLen = SL_ParseDoubleQuotedString( &cpSrc, cpDst, cpDstEndstop-cpDst/*iMaxLen*/ );
                    if( iLen >= 0 )
                     { // Keep it simple, assume any translation (string) fits inside a single line !
                       if( TranslationTableFromFile[nLinesLoaded] == NULL ) // ok to allocate a NEW STRING ?
                        { TranslationTableFromFile[nLinesLoaded] = (char*)malloc( iLen + 1 ); // "plus one" for the trailing zero
                          if( TranslationTableFromFile[nLinesLoaded] != NULL ) // successfully allocated ?
                           { if( iLen>0 ) // REALLY something to copy from cpDst ?
                              { memcpy( TranslationTableFromFile[nLinesLoaded], cpDst, iLen );
                              }
                             TranslationTableFromFile[nLinesLoaded][iLen] = '\0'; // make sure C-strings are ALWAYS terminated
                             // (this includes EMPTY STRINGS for "not-yet-translated" entries,
                             //  which must not be NULL because NULL marks the end
                             //  of a block of sub-strings in different languages)
                           }
                        }
                       ++nLinesLoaded;
                     } // end if < successfully parsed a DOUBLE-QUOTED string >
                    else
                     { // oops... syntax error: Line begins with '"'
                       // but SL_ParseDoubleQuotedString() returned a negative error code
                     }
                  } // end if < first non-space character is DOUBLE QUOTE -> string literal >
                 else // not a double-quoted string literal, but maybe it's a "NULL"-pointer in the C source:
                 if( SL_SkipToken( &cpSrc, "NULL" ) )
                  { // > NULL marks the end of the substrings (tranlations) for ONE TEXT ..
                    // Leave the pointer in TranslationTableFromFile[nLinesLoaded] NULL !
                    ++nLinesLoaded;
                  } // end if < NULL pointer in the C source, marks the end of a list of sub-strings >
               }   // end if < fInTranslationTable >
            }     // end else < line may contain a C statement >
         }       // end if < non-empty line >
      }         // end while < more lines read from the text file >
     QFile_Close( &qf );
   } // end if < successfully opened the file >
  if( fOk )
   { return APPL_GetNumEntriesInTranslationTable( (const char**)TranslationTableFromFile );
   }
  return 0;
} // end APPL_LoadTranslationsFromFile()


//---------------------------------------------------------------------------
void APPL_UnloadTranslationsFromFile(void) // Call this on exit !
   // Frees all dynamically allocated strings in TranslationTableFromFile[] .
{ int i;
  for(i=0; i<MAX_LINES_IN_TRANSLATION_TABLE; ++i)
   { if( TranslationTableFromFile[i] != NULL )  // highly portable stoneage "C" ..
      { free( TranslationTableFromFile[i] );
        TranslationTableFromFile[i] = NULL;
      }
   }
} // end APPL_UnloadTranslationsFromFile()


//---------------------------------------------------------------------------
int APPL_GetNumEntriesInTranslationTable( const char **cppSrc )
{ int nEntries = 0;
  // Count the number of entries until hitting TWO subsequent NULL pointers:
  while( (cppSrc[nEntries] != NULL) || (cppSrc[nEntries+1] != NULL) )
   { ++nEntries;
   }
  return nEntries;
} // end APPL_GetNumEntriesInBuiltInTranslationTable()


