//---------------------------------------------------------------------------
// File: C:\cbproj\Remote_CW_Keyer\Timers.c
// Date: 2024-01-13
//       Provides high-resolution "timers" and "stopwatches" .
//       For Windows, based on the "QueryPerformance"-thingy .
//       For Linux,   not sure about what to use instead .
//       Formerly a part of DL4YHF'S Spectrum Lab .
//       Modified / 'downsized' copies of this file may exist in:
//          * C:\cbproj\SpecLab\Timers.c
//          * C:\cbproj\Remote_CW_Keyer\Timers.c
//          * C:\cbproj\SoundUtl\Timers.c
//          * ... ?
// Latest modifications:
//   2024-01-13 : Created this spin-off for the Remote CW Keyer (running on windows).
//   2024-12-07 : Modified TIM_ReadStopwatch_ms() to return zero ONLY when
//                the specified stopwatch has not been STARTED yet,
//                to make it compatible with the incarnations for embedded systems.
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
#include <windows.h>
// #include <vcl.h>   // try to keep Borland-VCL and other depencencies out of this module !
// #include <time.h>  // _timezone and related stuff (for the 'large' implementation only)
#include <math.h>


#pragma hdrstop

#include "Timers.h"   // verify prototypes


static LONGLONG TIM_i64HighResTimerOffset = 0;
static LONGLONG TIM_i64HighResTimerFreq   = 0;


//-------------- Implementation of functions ---------------------------------

//---------------------------------------------------------------------------
void TIM_Init(void)
  // Initializes the module.  Call this once before any other function in here.
{
  TIM_ReadHighResTimer_s(); // let our 'high-res timer' start ticking (at ZERO)
  QueryPerformanceFrequency( (LARGE_INTEGER*)&TIM_i64HighResTimerFreq );
               // will this FREQUENCY change when speed-stepping ? (hope not..)

} // end TIM_Init()


//---------------------------------------------------------------------------
CPROT double TIM_ReadHighResTimer_s( void )
  // Returns the value from Windows' "PerformanceCounter", converted into
  // SECONDS (because that's the SI unit for time).
  // Don't assume anything about when this timer started at ZERO,
  // and don't assume anything about when this timer will WRAP AROUND
  // from some 'high value' to zero - it's HARDWARE SPECIFIC !
{
  LONGLONG i64, i64Freq;
  double dblSeconds;

  QueryPerformanceCounter( (LARGE_INTEGER*)&i64 );
  QueryPerformanceFrequency( (LARGE_INTEGER*)&i64Freq ); // will the FREQUENCY change when speed-stepping ? (hope not..)
  if( TIM_i64HighResTimerOffset == 0 )
   {  TIM_i64HighResTimerOffset = i64;
   }
  if( i64Freq > 0 )
   { // ex: i64 = (i64 * 1000000LL) / i64Freq; // even a 64-bit product may be insufficient..
     // thus use floating point to scale from timer ticks into microseconds:
     dblSeconds = (double)(i64 - TIM_i64HighResTimerOffset) / (double)i64Freq;
     //  ,----------------|_______________________________|
     //  '--> difference in timer ticks (tick frequency MAY be many Megahertz..
     //       thus do the SCALING into seconds with 64-bit *double* )
   }
  else  // there doesn't seem to be a 'performance counter / timer' on this system !
   { dblSeconds = 0; // can this happen ? guess not, but Microsoft said "be prepared for the worst"..
   }
  return dblSeconds;
} // end TIM_ReadHighResTimer_s()

//---------------------------------------------------------------------------
CPROT long TIM_ReadHighResTimer_ms( void )
{ return (long)( fmodl( TIM_ReadHighResTimer_s() * 1e3, 2147483647.0 ) );
  // ,--------------------------------------------------|_________|
  // '--> The result shall "roll over" from 2^31 to ZERO,
  //      so as a 32-bit integer, it never gets NEGATIVE.
  //      This happens every 2^31 milliseconds = circa 24.8 days.
} // end TIM_ReadHighResTimer_ms()

//---------------------------------------------------------------------------
CPROT void TIM_SyncHighResTimer_ms( long i32_ms )
  // Adjusts TIM_i64HighResTimerOffset so that TIM_ReadHighResTimer_ms(),
  // if called NOW, would deliver <i32_ms> (as a value in MILLISECONDS).
  // Intended for synchronisation via network, to make timestamps 'comparable'.
{
  LONGLONG i64Freq;
  long i32DeltaT_ms = TIM_ReadHighResTimer_ms() - i32_ms;
  QueryPerformanceFrequency( (LARGE_INTEGER*)&i64Freq );
  if( i64Freq > 0 )
   { // Note : In TIM_ReadHighResTimer_s(), TIM_i64HighResTimerOffset is
     //        SUBTRACTED from the QueryPerformanceCounter() thingy, thus:
     TIM_i64HighResTimerOffset += ((LONGLONG)i32DeltaT_ms * i64Freq) / 1000LL;
   }

} // end TIM_SyncHighResTimer_ms()

//---------------------------------------------------------------------------
CPROT void TIM_StartStopwatch( T_TIM_Stopwatch *pStopwatch )
  // Stopuhr; ursprunglich implementiert fr diverse Mikrocontroller-Projekte.
  // Erwies sich allerdings auch bei anderen hartnaeckigen Problemen
  // (z.B. Messen der Task-Wechsel-Zeiten des Keil-RTX) als sehr hilfreich,
  // und ist daher in ALLEN Programmen fr LPC2xxx & Co vorhanden .
  //  HIER : Variante fr PC (mit Windows) .
  //
  // Prinzip:
  //   1.) Starten des Stopuhr-Intervalls :
  //       static T_TIM_Stopwatch myStopwatch;
  //       TIM_StartStopwatch( &myStopwatch );
  //   2.) Ablesen der Zeit im MIKROSEKUNDEN ...             (*)
  //       i32T_us = TIM_ReadStopwatch_us( &myStopwatch );
  //       ... oder, wenn's ausreicht, in MILLISEKUNDEN :
  //       i32T_ms = TIM_ReadStopwatch_ms( &myStopwatch );
  // (*) Note: The english Wikipedia said about 'Microsecond' :
  //       The symbol is 's', sometimes simplified to 'us'
  //       when Unicode is not available. Thus (here): 'us', not 'usec'.
  //
{
  QueryPerformanceCounter( (LARGE_INTEGER*)pStopwatch );
} // end TIM_StartStopwatch()

//---------------------------------------------------------------------------
CPROT void TIM_StopStopwatch( T_TIM_Stopwatch *pStopwatch )
  // After STOPPING a stopwatch, TIM_ReadStopwatch_s(), etc, all return
  //       ZERO. A few applications use this feature, for example if a
  //       stopwatch shall not 'run' at all.
{
  *pStopwatch = 0;
} // end TIM_StopStopwatch()


//---------------------------------------------------------------------------
CPROT double TIM_ReadStopwatch_s( T_TIM_Stopwatch *pStopwatch )
  // Return value :
  //   > 0 : Time between  TIM_StartStopwatch() and this call
  //         IN SECONDS (thus the the return value is floating point) .
  //         *pStopwatch  will not be modified, so we can take more than
  //         one reading with the same stopwatch .
  //   < 0 : Time from a stopwatch that is RUNNING (not STOPPED),
  //         but has been 'adjusted' via TIM_AdjustStopwatch_ms()
  //         as if the time when it was started hasn't come yet (!)
  //   = 0 : The specified stopwatch has not been started at all.
  //
  // Not sure about the MAXIMUM interval that can be measured between
  //     TIM_StartStopwatch() and TIM_ReadStopwatch_sec()
  //     - it's HARDWARE SPECIFIC !
{
  LONGLONG i64;
  double dblSeconds;

  if( *pStopwatch == 0 )  // stopwatch not running at all ?
   { return 0.0;
   }


  QueryPerformanceCounter( (LARGE_INTEGER*)&i64 );
  if( TIM_i64HighResTimerFreq > 0 )
   { // ex: i64 = (i64 * 1000000LL) / i64Freq; // even a 64-bit product may be insufficient..
     // thus use floating point to scale from timer ticks into microseconds:
     dblSeconds = (double)(i64 - *pStopwatch) / (double)TIM_i64HighResTimerFreq;
     //  ,----------------|_________________|
     //  '--> difference in timer ticks (tick frequency MAY be many Megahertz..
     //       thus do the SCALING into seconds with 64-bit *double* )
   }
  else  // there doesn't seem to be a 'performance counter / timer' on this system !
   { dblSeconds = 0; // can this happen ? guess not, but Microsoft said "be prepared for the worst"..
   }
  return dblSeconds;
} // end TIM_ReadStopwatch_s()

//---------------------------------------------------------------------------
CPROT long TIM_ReadStopwatch_ms( T_TIM_Stopwatch *pStopwatch )
  // Return value : Time difference in MILLISECONDS between 'now' and the start.
  //
  //   > 0 : Time between  TIM_StartStopwatch() and this call
  //         IN MILLISECONDS (thus a 32-bit 'long' should be ok) .
  //         *pStopwatch  will not be modified, so we can take more than
  //         one reading with the same stopwatch .
  //   < 0 : Time from a stopwatch that is RUNNING (not STOPPED),
  //         but has been 'adjusted' via TIM_AdjustStopwatch_ms()
  //         as if the time when it was started hasn't come yet (!)
  //   = 0 : The specified stopwatch has not been started at all.
  //
  // To tell a not-yet-running stopwatch from a stopwatch that has
  //    JUST BEEN STARTED, the mimimum interval that can be measured
  //    by TIM_ReadStopwatch_ms() is ONE MILLISECOND :
  //    Even when called IMMEDIATELY AFTER TIM_StartStopwatch( &myStopwatch ),
  //    TIM_ReadStopwatch_ms( &myStopwatch ) would return ONE, not ZERO !
  // *pStopwatch  will not be modified, so we can take more than
  // one reading with the same stopwatch .
  // For details, see TIM_StartStopwatch() and TIM_ReadStopwatch_s() .
{
  long i32Time_ms;
  if( *pStopwatch == 0 )  // stopwatch not running at all ?
   { return 0;
   }
  i32Time_ms = (long)( TIM_ReadStopwatch_s(pStopwatch) * 1e3 );
  // Note the possibly NEGATIVE value here, which must remain negative;
  //      and the avoidance of ZERO which would mean "stopped".
  //      At *this* point, we know the stopwatch isn't stopped
  //      so the return value must be negative or positive but not zero.
  if( i32Time_ms == 0 ) // oops... too close to the time of starting ..
   {  i32Time_ms = 1; // This is important to tell a NON-STARTED stopwatch
                      // from a stopwatch that has JUST BEEN STARTED !
   }
  return i32Time_ms;
} // end TIM_ReadStopwatch_ms()

//---------------------------------------------------------------------------
CPROT void TIM_AdjustStopwatch_ms( T_TIM_Stopwatch *pStopwatch, int nMilliseconds )
  // Adjusts an already running stopwatch (with TIM_StartStopwatch)
  // as if it was started "precisely <nMilliseconds> LATER" than it really was,
  // without stopping and restarting it.
  // If TIM_ReadStopwatch_ms() was called after TIM_AdjustStopwatch_ms(),
  // the value (number of milliseconds returned by TIM_AdjustStopwatch_ms() )
  // would be exactly <nMilliseconds> LESS than without the 'adjustment' .
  // Added 2024-01-01 to realize a sequence of precisely timed actions
  // using a single stopwatch, WITHOUT ANY SLIP, in the Remote CW Keyer.
  //    Note: After "adjusting" a stopwatch this way,
  //          the value returned by e.g. TIM_ReadStopwatch_ms() may even
  //          be NEGATIVE (= "stopwatch not expired yet").
{
  if( *pStopwatch == 0 )  // stopwatch not running at all ?
   { return;              //  .. then we cannot ADJUST it by any positive or negative interval !
   }

  *pStopwatch += ((LONGLONG)nMilliseconds * TIM_i64HighResTimerFreq + 500/*round*/) / 1000LL;
} // end TIM_AdjustStopwatch_ms()


//---------------------------------------------------------------------------
CPROT long TIM_ReadStopwatch_us( T_TIM_Stopwatch *pStopwatch )
  // Return value : Time difference in MICROSECONDS between 'now' and the start.
  //
  //   > 0 : Time between  TIM_StartStopwatch() and this call
  //         IN MICROSECONDS (thus, with a 32-bit return value, only ok
  //         for short intervals, up to 2^31 us = circa 30 minutes).
  //         *pStopwatch  will not be modified, so we can take more than
  //         one reading with the same stopwatch .
  //   < 0 : Time from a stopwatch that is RUNNING (not STOPPED),
  //         but has been 'adjusted' via TIM_AdjustStopwatch_ms()
  //         as if the time when it was started hasn't come yet (!)
  //   = 0 : The specified stopwatch has not been started at all.
  //
  // *pStopwatch  will not be modified, so we can take more than
  // one reading with the same stopwatch .
  // For details, see TIM_StartStopwatch() and TIM_ReadStopwatch_s() .
{
  long i32Time_us;
  if( *pStopwatch == 0 )  // stopwatch not running at all ?
   { return 0;
   }
  i32Time_us = (long)( TIM_ReadStopwatch_s(pStopwatch) * 1e6 );
  if( i32Time_us == 0 ) // oops... too close to the time of starting ..
   {  i32Time_us = 1; // This is important to tell a NON-STARTED stopwatch
                      // from a stopwatch that has JUST BEEN STARTED !
   }
  return i32Time_us;
} // end TIM_ReadStopwatch_us()

//---------------------------------------------------------------------------
CPROT long TIM_ReadAndRestartStopwatch_us( T_TIM_Stopwatch *pStopwatch )
  // Returns the time between TIM_StartStopwatch
  //                   or the previous call of TIM_ReadAndRestartStopwatch_us
  //         and THIS call of TIM_ReadAndRestartStopwatch_us(),
  //         converted into IN MICROSECONDS,
  //         and restarts the particular stopwatch *AT THE VERY SAME TIME* .
  // (which cannot be achieved, at least not PRECISELY, with
  //    TIM_ReadStopwatch_us() followed by TIM_StartStopwatch() )
{
  LONGLONG i64, i64Freq;
  double dblMicroseconds;
  long i32Time_us;
  if( *pStopwatch == 0 )  // stopwatch not running at all ?
   { TIM_StartStopwatch( pStopwatch );
     return 0;
   }

  QueryPerformanceCounter( (LARGE_INTEGER*)&i64 );
  if( TIM_i64HighResTimerFreq > 0 )
   { // scale from timer ticks into microseconds with floating point, even though our API uses integer only:
     dblMicroseconds = (double)(i64 - *pStopwatch) * 1e6 / (double)TIM_i64HighResTimerFreq;
   }
  else // oops.. there doesn't seem to be a 'performance counter / timer' on this system !
   { dblMicroseconds = 0; // can this happen ? guess not, but Microsoft said "be prepared for the worst"..
   }
  *(LONGLONG*)pStopwatch = i64;  // <- here is the "Restart", without any slip

  i32Time_us = (long)dblMicroseconds;
  if( i32Time_us == 0 ) // oops... too close to the time of starting ..
   {  i32Time_us = 1; // This is important to tell a NON-STARTED stopwatch
                      // from a stopwatch that has JUST BEEN STARTED !
   }
  return i32Time_us;

} // end TIM_ReadAndRestartStopwatch_us()




/* EOF <timers.c> */

