// File: C:\cbproj\Remote_CW_Keyer\TIconToFavicon.cpp
// Date: 2024-02-12
// Author: Wolfgang Buescher (DL4YHF)
//
// Purpose: Converts a Borland-VCL-"TIcon" into a simple byte array
//          with a 'Favicon.ico' compatible structure for our embedded
//          web server. The source (TIcon a la Borland/Embarcadero VCL)
//          can be retrieved from THE MAIN FORM's "Icon" or Application->Icon
//          (both are Borland/Embarcadero VCL objects of type "TIcon").
//  For some reason, "Application->Icon" retrieved a 40*40-pixel-icon,
//  while e.g. "KeyerMainForm->Icon" retrieved the original 32*32-pixels.


#include <windows.h> // Must be included BEFORE vcl.h for some strange reason.
#include <vcl.h>     // Visual Component Library with stuff like "TIcon"
#include "Inet_Tools.h" // helpers like INET_AppendBlock(), to assemble binary data


//---------------------------------------------------------------------------
int TIconToFavicon( TIcon *pIcon, BYTE *pbFavicon, int iMaxSize)
  // Return value: number of byte placed in bpFavicon[] when successful .
  // About the *.ico file format GENERATED here: Start grokking at Wikipedia:
  // > Icon file structure
  // > An ICO or CUR file is made up of an ICONDIR ("Icon directory") structure,
  // > containing an ICONDIRENTRY structure for each image in the file,
  // > followed by a contiguous block of all image bitmap data (which may be
  // > in either Windows BMP format, excluding the BITMAPFILEHEADER structure,
  // > or in PNG format, stored in its entirety).[3]
  //
{
  BYTE *pbDest    = pbFavicon;
  BYTE *pbEndstop = pbFavicon + iMaxSize;
  int x,y,iWidth, iAlignedWidthInBytes1, iAlignedWidthInBytes2, iHeight;
  Graphics::TBitmap *pBmp;
  TColor color;   // Borland's "TColor" format is 0x00BBGGRR as a 32-bit value !
  WORD w3IconDir[3];
  BOOL fBuggyStructAlignment = FALSE;
#pragma pack(push,1) // Thank you, Microsoft, for NOT properly aligning DWORDs in your *.ico file formats...
  struct
   { //   Hexadecimal number in squared brackets below are BYTE OFFSETS within the *.ico-"file"
     // 6-byte "ICONDIR" struct (three 16-bit WORDs below) ..
     WORD  wReserved;     // [00-01] "Reserved. Must always be 0" (in an "ICONDIR"-struct)
     WORD  wImageType;    // [02-03] "Specifies image type: 1 for icon (.ICO) image, 2 for cursor (.CUR) image."
     WORD  wNumImages;    // [04-05] "Specifies number of images in the file." (all in LITTLE-ENDIAN byte order, hooray)
           // Note how this crappy SIX-BYTE "ICONDIR" destroys
           // the alignment of any 32-bit fields that follow further below.

     // 16-byte "ICONDIRENTRY" below .. (too tired of trying to find this in the Windows headerfile swamp)
     BYTE  bWidth;        // [06] Width, in pixels, of the image
     BYTE  bHeight;       // [07] Height, in pixels, of the image
     BYTE  bColorCount;   // [08] Number of colors in image ("0 if >=8bpp")
     BYTE  bReserved;     // [09] Reserved ( must be 0)
     WORD  wPlanes;       // [0A..0B] Color Planes
     WORD  wBitsPerPixel; // [0C..0D] Bits per pixel (formerly known as "BitCount"..what the heck is that)
     DWORD dwBytesInRes;  // [0E..11] "How many bytes are in this resource?" (Microsoft) / "Size of (InfoHeader + ANDbitmap + XORbitmap)" (somebody else)
     DWORD dwImageOffset; // [12..15] Where in the file is this image ? (guess what.. at 0x00000016, but that's ..TADA.. NOT THE PIXEL DATA YET)
           // At this point, at BYTE-OFFSET #6 + #16 = #22 = 0x16 ...

     // 40-byte "Variant of a BMP Info Header" (this is NOT a "BITMAPFILEHEADER" but a "BITMAPINFOHEADER" .. PFUI DEIBEL)
     DWORD dwSizeofBmpInfoHeader; // [0x16..0x19] : ALWAYS must contain sizeof(BITMAPINFOHEADER) = #40 = 0x28 00 00 00 (low byte first)
     DWORD dwWidth;        // [1A..1D] : ALSO the WIDTH IN PIXELS.
     DWORD dwDoubleHeight; // [1E..21] added height of the 24-bpp-"XOR-Bitmap" *plus* the 1-bpp-"AND-Bitmap"
     WORD  wPlanes2;       // [22..23] what we already have in the "ICONDIRENTRY" must be duplicated here...
     WORD  wBitsPerPixel2; // [24..25] ... but here, as 32-bit DWORDS, not as 16-bit WORDS..
     DWORD dwCompression;  // [26..29] 0=no compression; the memset(0) takes care of this crap
     DWORD dwSizeImage;    // [2A..2D] 0 when not compressed; " " "
     DWORD dwXPelsPerMeter; // [2E..31]
     DWORD dwYPelsPerMeter; // [32..35]
     DWORD dwColorsUsed;    // [36..39]
     DWORD dwColorsImportant; // [3A..3D]

   } iconHdr;
#pragma pack(pop)
  if( pIcon != NULL )
   { iWidth = pIcon->Width;
     iHeight= pIcon->Height;
     // The application's 'program icon', "morse_key.ico" ORIGINALLY had 32*32 pixels.
     // But here, in Borland's "Application.Icon", iWidth=40, iHeight=40. Strange.
     // The number of BYTES PER LINE OF PIXELS must be an
     //     integer multiple of FOUR, for both the 24-bit-per-pixel
     //     COLOUR BITMAP and the 1-bit-per-pixel "transparency" bitmap.
     // Example: 40 * 40 pixel *.ico file with 24 bits per pixel generated by
     //          IrfanView : 5182 bytes in file = sum of the following :
     //                         6 bytes for the stupid "ICONDIR" (which destroys DWORD-alignment)
     //                    +   16 bytes for a single   "ICONDIRENTRY"
     //                    +   40 bytes for a single   "BITMAPINFOHEADER"
     //                    + 4800 bytes for 40 lines * 40 pixels * 3 bytes_per_pixel = 40 * 120 bytes_per_line for the COLOUR BITMAP
     //                    +  320 bytes for 40 lines * 40 pixels / 8 pixels_per_byte = 5 bytes, aligned to 8 bytes per line -> 40 * 8 = 320 .
     //                  ---------
     //               Sum  = 5192 bytes in the file (ok),
     //                      40+4800+320 = 5160 of those bytes are the
     //                      "bytes in this resource", whatever that means.
     //
     iAlignedWidthInBytes1 = (iWidth*3/*24bit/pix*/ + 3)      & 0x7FFFC;
       // '--> e.g. (40*3 + 3) & 0x7FFFC = 120 [DWORD-aligned bytes per 24-BIT-COLOR PIXEL LINE]
     iAlignedWidthInBytes2 = ((iWidth /*1 bit/pix*/ +31) / 8) & 0x7FFFC;
       // '--> e.g. (40+31)/8  & 0x7FFFC = 8   [DWORD-aligned bytes per MONOCHROME PIXEL LINE]

     memset( &iconHdr, 0, sizeof(iconHdr) ); // <- this clears all the "Reserved" and "meaningless" fields
     iconHdr.wImageType = 1;  // "Specifies image type: 1 for icon (.ICO) image, 2 for cursor (.CUR) image."
     iconHdr.wNumImages = 1;  // "Specifies number of images in the file." (all in LITTLE-ENDIAN byte order, hooray)
     iconHdr.bWidth  = (BYTE)iWidth;
     iconHdr.dwWidth = (DWORD)iWidth;
     iconHdr.bHeight = (BYTE)iHeight;
     iconHdr.dwDoubleHeight = (DWORD)iHeight*2; // added height of XOR-Bitmap and AND-Bitmap (buaah !)
     iconHdr.wPlanes = iconHdr.wPlanes2 = 1;  // "Specifies color planes. Should be 0 or 1."
     iconHdr.wBitsPerPixel=iconHdr.wBitsPerPixel2 = 24; // "Specifies bits per pixel."
     iconHdr.dwSizeofBmpInfoHeader = 40;  // [0x16..0x19] : 0x28 00 00 00, NICELY MISALIGNED, thank you.
     iconHdr.dwXPelsPerMeter = iconHdr.dwYPelsPerMeter = 0x60; // same as in the IrfanView-generated test file
     iconHdr.dwBytesInRes  = (DWORD)( iHeight * (iAlignedWidthInBytes1+iAlignedWidthInBytes2))
                           + iconHdr.dwSizeofBmpInfoHeader;
              // '--> "Specifies the size of the resource, in bytes."
              //   e.g. [at 0x0E..0x11]  : 0x28 0x14 0x00 0x00 = 0x001428 = 5160 bytes,
              //          seen in a 40*40 * 24bit *.ico generated by IrfanView,
              //          favicon_40_40_24bit.ico, with FILE SIZE = 5182 bytes .
              // This answers the question WHAT ACTUALLY IS this "resource"-thingy:
              // 5182 (bytes in file) = 5160 "bytes in the "resource" + 6+16+40 bytes in our iconHdr.
              //
              // Example: The various headers seen in the IrfanView-generated "favico_40_40_24bit.ico" :
              //      (+0  1  2  3  4  5  6  7   8  9  A  B  C  D  E  F)
              // 0000: 00 00 01 00 01 00 28 28  00 00 01 00 18 00 28 14
              //       |__"ICONDIR"____| |_____"ICONDIRENTRY"..._______
              //                                                  |___dwBytesInRes..
              //      (+0  1  2  3  4  5  6  7   8  9  A  B  C  D  E  F)
              // 0010: 00 00 16 00 00 00 28 00  00 00 28 00 00 00 50 00
              //       ..._____________| |_____"BITMAPINFOHEADER"...___
              //       ..__| |ImageOffs| |SizeBmpHdr| |_dwWidth_| |_dwDoubleHeight..
              //
              //      (+0  1  2  3  4  5  6  7   8  9  A  B  C  D  E  F)
              // 0020: 00 00 01 00 18 00 00 00  00 00 00 00 00 00 60 00
              //       .... Planes BitPP  Compression |SizeImage| |XPels...
              //       --__| |___| |___| |__________| |_________| |_____..
              //
              //      (+0  1  2  3  4  5  6  7   8  9  A  B  C  D  E  F)
              // 0030: 00 00 60 00 00 00 00 00  00 00 00 00 00 00 00 80 FF
              //       ..../ | YPels/m | |ColorsUsed| |ColorsImp| |1st PIXEL|
              //             |_________| |__________| |_________| |______|
              // The "orange" = 0x00 (blue?) 0x80 (green?) 0xFF (red?) pattern
              // began at 0x003E and ended (last 0xFF) at 0x12FD :
              //      (+0  1  2  3  4  5  6  7   8  9  A  B  C  D  E  F)
              // 12F0: 80 FF 00 80 FF 00 80 FF  00 80 FF 00 80 FF 00 00 .
              //       ..last few orange COLOUR PIXELS   (B G R)| |"AND"-mask ?
              //   (   0x12FD (end of LAST colour pixel)
              //     - 0x003E (begin of FIRST colour pixel)
              //     + 1      (the old stuff with ADDRESS DIFFERENCES, 0x00..0x00 spans ONE BYTE)
              //   ) / 3  =  1600 of these THREE-BYTE-PIXELS, as expected.
     iconHdr.dwImageOffset =  6 + 16;  // "number of bytes from the beginning of the file to the BITMAPINFOHEADER"

     if( (pBmp = new Graphics::TBitmap() ) != NULL )
      { pBmp->Width  = iWidth;
        pBmp->Height = iHeight;
        pBmp->Canvas->Draw( 0,0, pIcon ); // Can we draw a "TIcon" into a "TBitmap" ?
        // > Images with less than 32 bits of color depth follow a particular format:
        // > the image is encoded as a single image consisting of a color mask
        // > (the "XOR mask") together with an opacity mask (the "AND mask").
        // > The XOR mask must precede the AND mask inside the bitmap data;
        // > if the image is stored in bottom-up order (which it most likely is),
        // > the XOR mask would be drawn below the AND mask.
        // > The AND mask is 1 bit per pixel, regardless of the color depth
        // > specified by the BMP header, and specifies which pixels are
        // > fully transparent(1) and which are fully opaque(0).
        // > The XOR mask conforms to the bit depth specified in the BMP header
        // > and specifies the numerical color or palette value for each pixel.
        // > Together, the AND mask and XOR mask make for a non-transparent
        // > image representing an image with 1-bit transparency; they also
        // > allow for inversion of the background. The height for the image
        // > in the ICONDIRENTRY structure of the ICO/CUR file takes on that
        // > of the intended image dimensions (after the masks are composited),
        // > whereas the height in the BMP header takes on that of the two mask
        // > images combined (before they are composited). Therefore, the masks
        // > must each be of the same dimensions, and the height specified
        // > in the BMP header must be exactly twice the height specified
        // > in the ICONDIRENTRY structure.[..]    OH MY GOD.
        INET_AppendBlock( &pbDest, pbEndstop, (BYTE*)&iconHdr, sizeof(iconHdr) );
          // '--> this emits the "ICONDIR" + "ICONDIRENTRY" + "BITMAPINFOHEADER"
          //                     (6 bytes) +   (16 bytes)   +   (40 bytes) .
          // What remains (to be emitted further below) are the funny-named
          //  "XOR bitmap" (that's the actual COLOUR BITMAP) plus the
          //  "AND bitmap" (with only 1 bit per pixels, but DWORD-aligned,
          //                even though the crappy Microsoft header garbage
          //                makes sure the data are NOT DWORD-aligned in the file)
        if( (pbDest - pbFavicon) != (6+16+40) )
         { fBuggyStructAlignment = TRUE;  // the result isn't a valid *.ico a la Microsoft-stoneage !
         }

        // Grab the COLOUR bitmap from Borland's TCanvas ...
        //   same as in the stupid "*.bmp", the bitmap is vertically flipped
        //   in an "icon". To compensate this utter nonsense (thank you, MS..),
        //   we run through the pixels in the source bitmap ("TCanvas")
        //   from BOTTOM to TOP :
        // ex: for(y=0; y<iHeight; ++y)
        for(y=iHeight-1; y>=0; --y)
         {
           for(x=0; x<iWidth; ++x)
            { color = pBmp->Canvas->Pixels[x][y]; // TColor a la Borland: 0x00BBGGRR (in a 32-bit DWORD)
              // Our old-school "favicon.ico" shall be a 24(!) bit-per-pixel thingy !
              INET_AppendByte( &pbDest, pbEndstop, (BYTE)(color >> 16) ); // "BLUE" byte first (found out by trial-and-error)
              INET_AppendByte( &pbDest, pbEndstop, (BYTE)(color >> 8)  ); // "GREEN" byte next
              INET_AppendByte( &pbDest, pbEndstop, (BYTE)(color >> 0)  ); // "RED" byte last
            }
         }

        // Append DUMMY BYTES for the 1-bit-per-pixel "AND"-mask-bitmap ...
        // Note: The IrfanView-generated *.ico had
        //       THE ENTIRE MONOCHROME BITMAP (320 bytes) filled with ZEROs.
        // So a ZERO-BIT (or ZERO-pixel) in the "AND"-mask seems to mean
        //    "NON-transparent pixel in the COLOUR bitmap".
        for(y=0; y<iHeight; ++y)
         {
           for(x=0; x<iAlignedWidthInBytes2; ++x)
            { INET_AppendByte( &pbDest, pbEndstop, 0x00 ); // 8 pixels per byte in the "AND"-bitmap
            }
         }

        delete pBmp; // delete the VCL-style BITMAP with its CANVAS that we used to RENDER the "TIcon"
      }


   } // end if( pIcon != NULL )
  if( fBuggyStructAlignment )
   { return 0;
   }
  else
   { return pbDest - pbFavicon; // returns THE NUMBER OF BYTE emitted into pbFavicon
     // 2024-02-12 : After some hassle, the pointer difference was 5182 bytes,
     // exactly the same as the IrfanView-generated 40*40 pixel 'Favicon.ico'.
   }

} // TIconToFavicon()


/* end < TIconToFavicon.cpp > */
