// File:  C:\cbproj\Remote_CW_Keyer\StraightKeyDecoder.c
// Date:  2024-05-09
// Author:  Wolfgang Buescher (DL4YHF)
// Purpose: Morse code decoder for a 'straight keying' signal.
//          First used in the author's "Remote CW Keyer" project 
//          to check the CW keying signal on the SERVER SIDE
//          (which drives the radio with a "straight keying" signal,
//           not via separate "dash" and "dot" signals).
//          The API (callable "C" functions) is similar as the
//          "Elbug" module (Remote_CW_Keyer/Elbug.c), which can 
//          also emit finished characters (at the end of an inter-
//          character-space) in the same 'CW_CHR_...' format as specified
//          in Remote_CW_Keyer/Elbug.h .
//
// Revision History (latest entry FIRST):
//   2024-05-13: Removed the 'contact debouncing' here, because it caused the
//               decoder to malfunction at insanely high speeds, for example
//               for the 'high-speed network-keying stress test',
//               when a remote client tried to send with 150 WPM,
//               which the server failed to decode when configured for 110 WPM
//               (a 3-dot gap between characters at 150 WPM looks like
//                a 3*110/150 = 2.2-dot-gap when "decoded" at 110 WPM,
//                which THEORETICALLY should work, but didn't. Scope display:
//                   ___     _     _   _   _     ___
//               ___|   |___| |___| |_| |_| |___|   |__________
//                            :   :
//                  '--- N ------''--------V------------'
//                            :   :
// Should have been '--T---''-E--''-----S------''---T---'   !
//                            :   :
//                            :   :
//                          ->|---|<- Measured this interval on the scope via mouse.
//                                    On the SERVER side (cfg'd for 110 WPM),
//                                    the result was "dt=28ms = 2.5 dots" (ok)
//                                    but the decoder didn't recognize this
//                                    as a valid gap !
//
//
//   2024-05-09: Create this module for the "Remote CW Keyer" project.

#include <string.h> // no string functions used here, but memset()

#include "Debouncer.h" // 'debouncer' for paddle inputs (T_Debouncer in T_ElbugInstance)
#include "Elbug.h" // header file for Elbug.c; also used for the "Straight Keying Decoder" !
     //    '--> stuff like Elbug_WordsPerMinuteToDotTimeInMilliseconds(),
     //                    Elbug_DotTimeInMillisecondsToWordsPerMinute(),
     //                    Elbug_MorseCodePatternToASCII().
     // Some of the above 'elbug' functions are also used by the
     // 'straight keying' decoder, and we don't want to reinvent the wheel.

#include "StraightKeyDecoder.h" // header file especially for StraightKeyDecoder.c


// Internal function prototypes
void StraightKeyDecoder_AppendToHistory( T_StraightKeyDecoder *pDecoder,
                    int nMicrosecondsBeforeNewState, BOOL fKeyDown,
                    T_StraightKeyDecoderState iState, WORD wShiftReg, WORD wEmittedCode );



//---------------------------------------------------------------------------
int StraightKeyDecoder( T_StraightKeyDecoder *pDecoder,
       BOOL fKeyDown, // [in] input from straight key: TRUE = "key down", FALSE = "key up"
       long nMicrosecondsSinceLastCall ) // [in] ideally 2000 us, if windoze permits (*)
  // Values returned by StraightKeyDecoder() : One of the CW_CHR-codes (More code
  // pattern emitted at the end of a single character or space),
  // or one of the ELBUG_RESULT_.. values defined in Elbug.h, for example
  //               ELBUG_RESULT_NOTHING_DECODED  ( == 0 ) .
  // Called periodically from KeyerThread() to decode what has been SENT,
  //        but also from DspThread() -> CwDSP_ProcessAudioSpectrumForDecoder()
  //        after finishing another AUDIO FFT. In that case, the calling
  //        interval (*) may be as long as 8, 16, or even 32 milliseconds !
{
  int iResult = ELBUG_RESULT_NOTHING_DECODED;
  long i32TimeAccuForHistory;

  pDecoder->i32TimeAccu_us += nMicrosecondsSinceLastCall;
  if( pDecoder->i32TimeAccu_us > 10000000L )  // avoid 32-bit integer overflow (limit to 10 seconds)
   {  pDecoder->i32TimeAccu_us = 10000000L;
   }
  i32TimeAccuForHistory = pDecoder->i32TimeAccu_us;
  switch( pDecoder->iState )
   {
     case DS_WAITING:   // waiting for anything to happen (key was "up" so far)
        if( fKeyDown )
         {
           // Prepare to decode the next character:
           pDecoder->i32TimeAccu_us = 0;
           pDecoder->iState = DS_RCV_DOT;
           pDecoder->wShiftReg = 1; // place a new "start bit" in the shift register
           // (the position of the startbit encodes the number of dots and dashes
           //   in the LOWER BITS of the shift register later)

           // Let the caller of StraightKeyDecoder() know we just STARTED a new character:
           iResult |= ELBUG_RESULT_BEGIN_NEW_CHAR; // indicate "BEGIN of a new character"
                                                   // in the 'Timing Scope'.
           // Add this 'state transition' to the history for debugging:
           StraightKeyDecoder_AppendToHistory( pDecoder, i32TimeAccuForHistory, fKeyDown, pDecoder->iState, pDecoder->wShiftReg, 0 );
         }
        break;
     case DS_RCV_DOT:    // currently receiving a dot (which MAY turn out a DASH soon)
        if( ! fKeyDown ) // key JUST RELEASED (debounced) -> time for action ..
         { pDecoder->wShiftReg = (pDecoder->wShiftReg << 1) | ( pDecoder->i32TimeAccu_us >= (pDecoder->iDotTime_ms * 2000) );
           // ,-----------------------------------------------'
           // '--> Bit values in the dash/dot-matrix : 0=dot 1=dash .
           //    > Bit 0 always contains the LAST TRANSMITTED dash/dot.
           //  Anything shorter than TWO NOMINAL DOT-TIMES is considered a dot.
           //  Anything else can only be a DASH. To test the decoder's tolerance,
           //  use e.g. 20 WPM on the CLIENT SIDE (sending from memory),
           //       and 35 WPM on the SERVER SIDE (decoding the 'straight key'
           //           signal from the timestamped keying commands).
           //    In that case, a single dot time becomes 35/20 = 1.75 dot times
           //    from the receiver's point of view, and if all works as planned
           //    will still be decoded properly.
           pDecoder->iState = DS_RCV_GAP;
           pDecoder->i32TimeAccu_us = 0;

           // Add this 'state transition' to the history for debugging:
           StraightKeyDecoder_AppendToHistory( pDecoder, i32TimeAccuForHistory, fKeyDown, pDecoder->iState, pDecoder->wShiftReg, 0 );
         }
        break; // end case DS_RCV_DOT
     case DS_RCV_GAP:  // Receiving a GAP WITHIN A CHARACTER (which may turn out a SPACE later)
        // BEFORE checking if the key is still UP or already DOWN again:
        // Check if the "gap" is the end of a character :
        // 3 dots is the ideal spacing of characters within a word,
        // 1 dot is the ideal spacing between dots and dashes within a character,
        // thus the only "threshold" can be a length of TWO dots:
        if( pDecoder->i32TimeAccu_us >= (pDecoder->iDotTime_ms * 2000) )
         { // two-dot gap -> reached the end of a CHARACTER, so "emit" it:
           pDecoder->i32TimeAccu_us -= pDecoder->iDotTime_ms * 2000; // do NOT reset the time completely, before detecting e.g. a SPACE CHARACTER in DS_RCV_SPACE !
           iResult |= pDecoder->wShiftReg;
           pDecoder->iState = DS_RCV_SPACE;  // finished receiving a CHARACTER, but now check for a SPACE after the character
           StraightKeyDecoder_AppendToHistory( pDecoder, i32TimeAccuForHistory, fKeyDown, pDecoder->iState, pDecoder->wShiftReg, pDecoder->wShiftReg );
           pDecoder->wShiftReg = 1;          // prepare reception of the next character
         } // end if < "GAP" long enough for the END OF A CHARACTER > ?

        if( fKeyDown ) // key just PRESSED AGAIN -> next element ..
         { pDecoder->iState = DS_RCV_DOT;
           pDecoder->i32TimeAccu_us = 0;
           // NOT here: pDecoder->wShiftReg = 1; because this element *may* not be the FIRST of a character !
           StraightKeyDecoder_AppendToHistory( pDecoder, i32TimeAccuForHistory, fKeyDown, pDecoder->iState, pDecoder->wShiftReg, 0 );
         }
        break; // end case DS_RCV_GAP

     case DS_RCV_SPACE: // finished receiving a CHARACTER but didn't emit a trailing SPACE yet
        // BEFORE checking if the key is still UP or already DOWN again:
        // Check the "space" is between characters or between words :
        // 3 dots is the ideal spacing of characters within a word,
        // 5 dots or more (ideally 6) means emit a SPACE character.
        // 2 dot-times (!) have already been spent in DS_RCV_GAP, thus:
        if( pDecoder->i32TimeAccu_us >= (pDecoder->iDotTime_ms * 3000) )
         { // time to emit a SPACE :
           iResult |= CW_CHR_SPACE;
           pDecoder->iState = DS_WAITING;
           pDecoder->i32TimeAccu_us = 0;
           StraightKeyDecoder_AppendToHistory( pDecoder, i32TimeAccuForHistory, fKeyDown, pDecoder->iState, pDecoder->wShiftReg, CW_CHR_SPACE );
         }

        if( fKeyDown ) // key just PRESSED AGAIN before the WORD-space expired -> new character but NO SPACE (between words)
         { iResult = ELBUG_RESULT_BEGIN_NEW_CHAR;
           pDecoder->wShiftReg = 1;       // prepare reception of the next character
           pDecoder->i32TimeAccu_us = 0;
           pDecoder->iState = DS_RCV_DOT; // receivr the next CHARACTER, without a space in between
           StraightKeyDecoder_AppendToHistory( pDecoder, i32TimeAccuForHistory, fKeyDown, pDecoder->iState, pDecoder->wShiftReg, 0 );
         }
        break; // end case DS_RCV_SPACE

     default:  // oops.. obviously a bug in the state machine
        break;
   } // end switch( pDecoder->iState )

  return iResult;

} // end StraightKeyDecoder()


//---------------------------------------------------------------------------
void StraightKeyDecoder_Init( T_StraightKeyDecoder *pDecoder, int iDefaultDotTime_ms )
{ memset( pDecoder, 0, sizeof(T_StraightKeyDecoder) );
  pDecoder->iDotTime_ms = iDefaultDotTime_ms;
}


//---------------------------------------------------------------------------
void StraightKeyDecoder_AppendToHistory( T_StraightKeyDecoder *pDecoder,
                    int nMicrosecondsBeforeNewState, // [in] what the name says..
                    BOOL fKeyDown,    // [in] FALSE=key up,  TRUE=key down
                    T_StraightKeyDecoderState iState, // [in] DS_WAITING, ..
                    WORD wShiftReg, WORD wEmittedCode )
{ T_StraightKeyDecoderHistoryEntry *pHE = &pDecoder->sHistory[pDecoder->iHistoryHeadIndex];
  pDecoder->iHistoryHeadIndex = (pDecoder->iHistoryHeadIndex+1) % SKD_HISTORY_N_ENTRIES;
  pHE->nMicrosecondsBeforeNewState = nMicrosecondsBeforeNewState;
  pHE->fKeyDown  = fKeyDown;
  pHE->iState    = iState;
  pHE->wShiftReg = wShiftReg;
  pHE->wEmittedCode = wEmittedCode;
  // The "History" filled this way can be converted into human-readable text
  // via the main menu, "Settings" .. "Report Test Results (on the Debug tab)".
  // Example of a high-speed signal that SHOULD have been decoded as "test",
  // SENT at 150 WPM by a 'remote client' via TCP/IP, DECODED at 109 WPM,
  //      which should THEORETICALLY be possible without errors
  //      (but with a 2-millisecond loop in KeyerThread(), things aren't as
  //       easy as they seem to be on first glance) :
  // > 'Straight Key' decoder: t_dot=11 ms; history:
  // >  #0: Key Down after 10000000 us : ShiftReg=0x0001, State=DOT
  // >  #1: Key Up after 24083 us : ShiftReg=0x0003, State=GAP    (ok, it actually was a DASH. The DASH is in bit 0 of the shift register, 0x0003 = "T")
  // >  #2: Key Down after 23975 us : ShiftReg=0x0003, State=DOT  (wrong: 23975 us are JUST ABOVE 2*t_dot, missed the end of a character)
  // >  #3: Key Up after 9953 us : ShiftReg=0x0006, State=GAP     (ok, that's a DOT, the "e" in "test")
  // >  #4: Key Down after 24017 us : ShiftReg=0x0006, State=DOT  (wrong again: this gap was above 2*t_dot, so it's a new character)
  // >  #5: Key Up after 7894 us : ShiftReg=0x000C, State=GAP     (ok, that's a suprisingly SHORT DOT, but ok..)
  // >  #6: Key Down after 10018 us : ShiftReg=0x000C, State=DOT  (ok, that's a valid inter-element-space within a character)
  // >  #7: Key Up after 8033 us : ShiftReg=0x0018, State=GAP     (ok, this is the 2nd "dot"/"dit" of the "s")
  // >  #8: Key Down after 9964 us : ShiftReg=0x0018, State=DOT   (ok, another valid inter-element-space)
  // >  #9: Key Up after 7919 us : ShiftReg=0x0030, State=GAP     (ok, this is the 3rd "dot"/"dit" of the "s")
  // >  #10: Key Down after 23253 us : ShiftReg=0x0030, State=DOT (again, JUST ABOVE 2*t_dot, so this also the END of a character)
  // >  #11: Key Up after 26049 us : ShiftReg=0x0061, State=GAP   (that's a valid "dash", actually the 2nd "t" in "test")
  // >  #12: Key Up after 25010 us : ShiftReg=0x0061, State=SPACE (ok, the decoder switched from "gap-between-chars" to "space between words")
  // >  #13: Key Up after 56800 us : ShiftReg=0x0001, State=WAITING (ok, enough time has passed to switch back to the WAITING state)
} // end StraightKeyDecoder_AppendToHistory()


//---------------------------------------------------------------------------
char* StraightKeyDecoder_StateToString( T_StraightKeyDecoderState iState )
{ switch( iState )
   { case DS_WAITING  : return "WAITING";
     case DS_RCV_DOT  : return "DOT";
     case DS_RCV_GAP  : return "GAP";
     case DS_RCV_SPACE: return "SPACE";
     default: return "???";
   }
} // end StraightKeyDecoder_StateToString()

/* EOF < StraightKeyDecoder.c > */

