//---------------------------------------------------------------------------
// File: C:\cbproj\Remote_CW_Keyer\SpecDisp.h
// Date: 2024-02-26
// Author: Wolfgang Buescher (DL4YHF)
// Purpose: Spectrum Display (waterfall and/or spectrum graph)
//          for the 'Remote CW Keyer' .
//      Unlike the main module for the GUI,
//      *this* module shall not have any dependencies
//      on Borland's VCL, Microsoft's don't-know-what-dot-something-framework,
//      or Qt   -    just PLAIN OLD WINDOWS GDI functions !
//---------------------------------------------------------------------------

#ifndef  SPEC_DISP_INCLUDED  // this conditional ends near -you guessed it- END OF FILE
# define SPEC_DISP_INCLUDED

//---------------------------------------------------------------------------
// External References (global variables and objects that formerly existed
//                      in Keyer_Main.cpp only, but we need them for plotting.
//                      This is incredibly ugly, but anything else would add
//                      and awful lot of extra work)
//---------------------------------------------------------------------------

extern T_CwGen CwKeyer_Gen; // 'CW Generator' instance, to convert text to Morse code
extern T_CwDSP CwKeyer_DSP; // 'CW Digital Signal Processor', with sidetone generator, CW filter/decoder, etc
extern T_CwNet MyCwNet;     // A single instance representing the "CW Network". Owns a "Rig Control" instance.


//---------------------------------------------------------------------------
// Constants
//---------------------------------------------------------------------------

#define SPECTRUM_MIN_DB -110.0
#define SPECTRUM_MAX_DB 0.0


extern unsigned long SunriseColors[256]; // imported from SunriseColors.c
extern unsigned long LinradColors[256];  // imported from LinradColors.c

// Visible controls on the spectrum/spectrogram/frequency scale :
#define GUI_CONTROL_NONE        0
#define GUI_CONTROL_SPECTRUM    1
#define GUI_CONTROL_SPECTROGRAM 2 // .. aka 'waterfall'
#define GUI_CONTROL_FREQ_SCALE  3
#define GUI_CONTROL_FREQ_INFO   4 // .. shows band indicators, "frequencies of interest", and maybe a web-sdr-like passband indicator


// Codes for various 'GUI event handlers' :
#define GUI_EVENT_NONE       0
#define GUI_EVENT_MOUSE_DOWN 1 // abstraction of Borland's "OnMouseDown" (i.e. button JUST PRESSED or "pen down")
#define GUI_EVENT_MOUSE_MOVE 2 // abstraction of Borland's "OnMouseMove" (i.e. mouse coord changed while button pressed)
#define GUI_EVENT_MOUSE_UP   3 // abstraction of Borland's "OnMouseUp"   (i.e. mouse button released or "pen up")

// Bitwise combineable flags for function argument 'iButtonsAndShiftKeys',
//   used in mouse-event-handlers as a thin abstraction layer :
#define GUI_KEY_FLAGS_NONE               0
#define GUI_KEY_FLAGS_LEFT_MOUSE_BUTTON  0x0001
#define GUI_KEY_FLAGS_RIGHT_MOUSE_BUTTON 0x0002
#define GUI_KEY_FLAGS_SHIFT              0x0004
#define GUI_KEY_FLAGS_CONTROL            0x0008

//---------------------------------------------------------------------------
// Data types
//---------------------------------------------------------------------------

  // T_RGBColor : Has been moved from Remote_CW_Keyer\SpecDisp.h
  //                             into cbproj\YHF_Tools\yhf_type.h !

typedef struct t_SpecDispMarker // a "marker" isn't a visual component (aka "windowed control"),
{        // but in most cases a 'clickable area' somewhere in a bitmap graphic.
  BOOL fVisible;
  int x1,y1,x2,y2;
  int iGuiControl; // GUI_CONTROL_NONE, GUI_CONTROL_SPECTRUM / ..SPECTROGRAM / ..FREQ_SCALE / ..FREQ_INFO
  T_RGBColor color;
  void *pvObject;  // pointer to e.g. a T_FreqListEntry (with iGuiControl==GUI_CONTROL_FREQ_INFO)
} T_SpecDispMarker;


typedef struct t_SpecDispControl
{
  // Configuration parameters
  int iFreqScaleHeight;
  int iWFBrightness_Percent, iWFContrast_Percent;
  int iWFColorPalette; // one of the following values:
#   define WF_PALETTE_SUNRISE 0 // author's favourite, taken from Spectrum Lab
#   define WF_PALETTE_LINRAD  1 // "rainbow-like" palette, inspired by Linrad
#   define WF_PALETTE_RED     2 // black -> dark red -> light red -> almost white
#   define WF_PALETTE_GREEN   3 // black -> dark green -> light green -> almost white
#   define WF_PALETTE_BLUE    4 // black -> dark blue -> light blue -> almost white
  float fltSpectrumAmplMin_dB, fltSpectrumAmplMax_dB; // 0..127 dB on the "Icom scale"

  int iColourScheme;  // colour scheme for most GUI elements (and the frequency scale),
                      // may contain one of the following:
#   define COLOUR_SCHEME_DEFAULT 0
#   define COLOUR_SCHEME_DARK    1

  int iDisplayOptions; // controls various 'options' for the spectrum/spectrogram display, bitwise combineable:
#   define SPEC_DISP_OPTIONS_NONE 0 // no "special" options (show spectrum and spectrogram without extras)
#   define SPEC_DISP_OPTIONS_TIME_MARKERS 16 // draw a tiny 'time tick' labelled with the time in UTC
           // on the left side of the SPECTROGRAM. Added 2024-12 to monitor the international NCDXF/IARU beacons.
#   define SPEC_DISP_OPTIONS_CW_DECODER_SPECTRUM 32 // show the coarse 'audio spectrum', calculated for/by the AUDIO CW DECODER


  int iBottomPanelMode; // controls the display in the optional panel on the bottom
     // of the 'TRX' tab (usually related with the FREQUENCY SCALE, thus directly below it):
#   define SPEC_DISP_PANEL_MODE_OFF 0 // nothing displayed on the panel (so HIDE it)
#   define SPEC_DISP_PANEL_MODE_FREQUENCY_INFO       1 // indicator for band edge frequencies and 'special frequencies of interest'
#   define SPEC_DISP_PANEL_MODE_MULTI_FUNCTION_METER 2 // S-Meter/Power, ALC, Compressor, SWR, Drain current, Supply Voltage, PA Temperature
#   define SPEC_DISP_PANEL_MODE_CHATBOX              3 // small text box primarily intended for messages from the SYSOP to other users
#   define SPEC_DISP_PANEL_MODE_AUTOMATIC            8 // automatically show e.g. "meters" during TX and "frequency info" during RX

  // Flags indicating which of the three parts (Spectrum, Waterfall, Frequency Scale)
  //   needs to be updated on the display:
  BOOL fUpdateSpectrum, fUpdateWaterfall, fClearWaterfall, fUpdateFreqScale, fUpdateFrequencyInfo;
  long i32AudioSpectrumUpdateCounter; // compared against the same member in a T_CwDSP to update the AUDIO SPECTRUM

  // Internal values ... most of them updated in each call of SpecDisp_UpdateWaterfall() / SpecDisp_UpdateSpectrum()
  DWORD clButtonFace; // ex: clBtnFace; actually the VCL form's BACKGROUND COLOUR
  DWORD clWindowText; // ex: clWindowText;
  DWORD clWindowBackground; // ex: clWindow; per default, WHITE background in e.g. combos and edit fields
  DWORD clActiveCaption;    // ex: clActiveCaption;   "color of the active window's title bar"
  DWORD clInactiveCaption;  // ex: clInactiveCaption; "color of inactive windows' title bar"
  DWORD clButtonFont;       // ex: the same as clWindowText

  DWORD dwBackgndColor/*for SPECTRUM DISPLAYS*/, dwGridColor, dwCurveColor,
        dwAuxCurveColor, dwAuxBkgndColor; // "auxiliary" curve- and polygon filling colour, e.g. for the AUDIO SPECTRUM in the background

  Graphics::TCanvas *Canvas; // A TCanvas can be painted on. A TBitmap is only the "pixel storage". Welcome to the funny world of Borland's VCL.
  int    iCanvasWidth, iCanvasHeight;
  double fmin_from_RigCtrl, fmax_from_RigCtrl;  // <- updated from the most recent T_RigCtrl_Spectrum
  BOOL   fixedEdgeMode;   // .. also info delivered by RigControl.c
  double fmin_on_FreqScale, fmax_on_FreqScale;  // <- currently displayed on the FREQUENCY SCALE
  int    nNewWaterfallLines;
  BOOL   fAudioSpectrumVisible;

  // Stuff used in various GUI event handlers (to display spectrum, VFO, etc) :
  double dblDisplayedVfoFreq_Hz; // VFO frequency (in Hertz) currently displayed in the GUI
  int    iVFOEditTimer_ms;       // "countdown" in milliseconds, > 0 while USER IS EDITING Ed_VFO in any way
  BOOL   fDraggingVfoFrequency;  // DRAGGING the VFO frequency (via mouse or touchscreen)
                                 // also temporarily stops updating the VFO display from 'RigControl'.
  double fmin_at_drag_start, fmax_at_drag_start;
  int    iDraggingOffsetX; // horizontal offset, in pixels, between initial mouse pointer and the dragged object (e.g. VFO)
  double dblClickedVfoFreq_Hz;   // "clicked" VFO frequency (in Hertz)
# define SPECDISP_NUM_MARKERS 4
# define SPECDISP_MARKER_VFO  0
  T_SpecDispMarker marker[SPECDISP_NUM_MARKERS];

  // Internal stuff for the optional "time markers" on the spectrogram:
  int iLastTimeMarkerTick_10s;  // ten-second time slot of the LAST time-marker shown on the screen
  int iTimeMarkerCounter;       // counter for spectrogram lines plotted since the LAST time-marker

  // Stuff for the optional "frequeny info display" (to avoid unnecessary updates):
  double fmin_on_FreqInfo, fmax_on_FreqInfo;  // <- currently displayed on the FREQUENCY INFO panel
  double dblUnixDateAndTimeOfFreqInfoUpdate;
# define SPECDISP_NUM_FREQ_INFO_MARKERS 128  // <- enough for 1 MHz bandwidth overcrowded with broadcast stations, spaced 10 kHz, sometimes two stations on the same QRG
  T_SpecDispMarker freqInfoClickableMarker[SPECDISP_NUM_FREQ_INFO_MARKERS];
     // ,----------'
     // '--> Filled with the CURRENTLY VISIBLE entries from the frequency database
     //      in SpecDisp_UpdateFrequencyInfo(). Used by the GUI in a mouse event handler.

} T_SpecDispControl;



//---------------------------------------------------------------------------
// Variables
//---------------------------------------------------------------------------



//---------------------------------------------------------------------------
// Function prototypes (details only in the implementation, SpecDisp.c)
//---------------------------------------------------------------------------
#ifndef CPROT  // For peaceful co-existence of "C" and "C++" ....
# ifdef __cplusplus
#   define CPROT extern "C"
# else
#   define CPROT
# endif  // ! "cplusplus"
#endif  // ndef CPROT

CPROT float SpecDisp_Voltage_to_dBfs( float fltNormalizedVoltage/*0..1*/ );
CPROT float SpecDisp_Power_to_dBfs( float fltNormalizedPower/*0..1*/ );
CPROT void GetStepAndSubstepForScale( double dblMinStep, double *pdblLargeStep, double *pdblSubStep);
CPROT void SpecDisp_InitControl( T_SpecDispControl *pDispCtrl, int iFreqScaleHeight );
CPROT void SpecDisp_UpdateVFODisplay( double dblVfoFrequency_Hz ); // -> Ed_VFO (in the GUI)
CPROT void SpecDisp_BrightnessToWaterfallColour( int iWFColorPalette, int iBrightness, T_RGBColor *pColour);
CPROT void SpecDisp_UpdateWaterfall(
                 Graphics::TBitmap *pbmpDest,   // [out] Borland-VCL-style "TBitmap"
                 T_SpecDispControl *pCtrl,
                 T_RigCtrlInstance *pRigCtrl,
                 int iScopeDisplayTailIndex, int nSpectra );
CPROT void SpecDisp_UpdateSpectrum(
                 Graphics::TBitmap *pbmpDest,   // [out] Borland-VCL-style "TBitmap"
                 T_SpecDispControl *pCtrl,
                 T_RigCtrlInstance *pRigCtrl);
CPROT void SpecDisp_UpdateFreqScale(
                 Graphics::TBitmap *pbmpDest,   // [out] Borland-VCL-style "TBitmap"
                 T_SpecDispControl *pCtrl,
                 T_RigCtrlInstance *pRigCtrl );
CPROT void SpecDisp_UpdateFrequencyInfo( // Renders the "band- and frequency info" into a bitmap graphics
        Graphics::TBitmap *pbmpDest,    // [out] Borland-VCL-style "TBitmap"
        T_SpecDispControl *pDispCtrl,   // [in] struct with the displayed frequency range, colour scheme, etc
        T_RigCtrlInstance *pRigCtrl,    // [in] all we know about "the rig", bands, and frequencies
        int x1, int y1, int x2, int y2); // [in] graphic area to use within the above bitmap

CPROT T_SpecDispMarker* SpecDisp_CheckCoordForClickableMarker( // .. in certain mouse event handlers
        T_SpecDispControl *pDispCtrl, // [in] struct to control the 'spectrum display'
        int iClientX, int iClientY,   // [in] client coordinate from a mouse event handler
        int iGuiControl); // [in] GUI_CONTROL_SPECTRUM / ..SPECTROGRAM / ..FREQ_SCALE / ..FREQ_INFO ?

CPROT BOOL SpecDisp_HandleMouseEvent( T_SpecDispControl *pDispCtrl,
                                      T_RigCtrlInstance *pRigCtrl,
        int iGuiEvent, int iGuiControlIndex, int iButtonsAndShiftKeys,
        int iClientX,  int iClientY);

#endif // ndef SPEC_DISP_INCLUDED ?


/* EOF < Remote_CW_Keyer/SpecDisp.h > */

