//---------------------------------------------------------------------------
// File: C:\cbproj\Remote_CW_Keyer\SpecDisp.cpp
// Date: 2024-02-26
// Author: Wolfgang Buescher (DL4YHF)
// Purpose: Spectrum Display (waterfall and/or spectrum graph)
//          for the 'Remote CW Keyer' .
//---------------------------------------------------------------------------

#include "switches.h"  // project specific compiler switches ("options"),
                       // must be included before anything else !
#include "yhf_type.h"  // classic types like BYTE, WORD, DWORD, BOOL, ..
#include <windows.h>   // Must be included BEFORE vcl.h for some strange reason.
                       // Contains stuff like WAVEINCAPS, WAVEOUTCAPS, etc^255 .
#include <vcl.h>       // Borland's stoneage Visual Component Library
#include <math.h>
#include <string.h>
#include <stdio.h>     // no "standard I/O" but "sprintf" used here

#ifdef __BORLANDC__
// prevent that bugger from bugging us with 'initialized data in header, cannot pre-compile..." :
# pragma hdrstop
#endif

#include "Utilities.h"  // UTL_FormatDateAndTime(), UTL_GetCurrentUnixDateAndTime[_Fast](), etc
#include "RigControl.h" // T_RigCtrlInstance "delivers" the spectra to display
          // (Icom call them "waveform data" but that's too ambiguous)
          // in a small circular FIFO with T_RigCtrl_Spectrum objects,
          // each of them with up to <RIGCTRL_MAX_FREQ_BINS_PER_SPECTRUM>.
#include "FreqList.h"  // import 'frequencies of interest' from e.g. the EiBi frequency list
#include "CwGen.h"     // CW generator (converts text to Morse code)
#include "CwDSP.h"     // CW-'Digital Signal Processor' / sidetone generator
#include "CwNet.h"     // Socket-based 'Client or Server' for the Remote CW Keyer. Tied to a "Rig Control" instance.
#include "SpecDisp.h"  // header for THIS module ('Spectrum/Waterfall display')



//---------------------------------------------------------------------------
// Internal function prototypes
//---------------------------------------------------------------------------

static void SpecDisp_DrawAudioSpectrumFromCwDSP( T_SpecDispControl *pDispCtrl,
                        T_RigCtrlInstance *pRigCtrl, T_CwDSP *pCwDSP );



//---------------------------------------------------------------------------
// Variables
//---------------------------------------------------------------------------

int SpecDisp_iLastErrorLine = 0; // <- for debugging only.. set to __LINE__ for error analysis after certain functions bailed out



//---------------------------------------------------------------------------
// Implementation of functions
//---------------------------------------------------------------------------

float SpecDisp_Voltage_to_dBfs( float fltNormalizedVoltage/*0..1*/ )
{
  if( fltNormalizedVoltage >= 1e-10 )
   {  return 20.0 * log10( fltNormalizedVoltage ); // ->  -200 to 0 dB "over" full scale
   }
  else
   {  return -200.0; // dB "over" full scale
   }
} // end SpecDisp_Voltage_to_dBfs()

//---------------------------------------------------------------------------
float SpecDisp_Power_to_dBfs( float fltNormalizedPower/*0..1*/ )
{
  if( fltNormalizedPower >= 1e-10 )
   {  return 10.0 * log10( fltNormalizedPower ); // ->  -200 to 0 dB "over" full scale
   }
  else
   {  return -200.0; // dB "over" full scale
   }
} // end SpecDisp_Power_to_dBfs()


//---------------------------------------------------------------------------
static void LimitInteger( int *piValue, int iMin, int iMax )
{ if( *piValue > iMax )
   {  *piValue = iMax;
   }
  if( *piValue < iMin )
   {  *piValue = iMin;
   }
} // end LimitInteger()

//---------------------------------------------------------------------------
static void LimitFloat( float *pfltValue, float fltMin, float fltMax )
{ if( *pfltValue > fltMax )
   {  *pfltValue = fltMax;
   }
  if( *pfltValue < fltMin )
   {  *pfltValue = fltMin;
   }
} // end LimitFloat()


//---------------------------------------------------------------------------
void GetStepAndSubstepForScale( double dblMinStep, double *pdblLargeStep, double *pdblSubStep)
  // Retrieves a "nice" stepwidth for drawing a scale, with a given MINIMUM stepwidth:
  //  [in] dblMinStep = MINIMUM stepwidth, dictated by the displayed bandwidth
  //                       and the number of pixels along the frequency scale.
  //                    Example: IC-7300 spectrum display set to "+/-25k"
  //                              = 50 kHz displayed bandwidth,
  //                              at least 80 pixels per "step" (labelled tick),
  //                              1200 pixels wide 'drawing canvas' for the scale,
  //   -> dblMinStep = 80 min_pixels_per_step * 50 kHz / 1200 pixels = 3333 [Hz].
  //

{ double step, substep, fact_pow10 = 1.0;

  while( dblMinStep > 1000.0/*Hz*/ )
   { fact_pow10 *= 10.0;
     dblMinStep *=  0.1;
   }

  if     ( dblMinStep <=  1.0 ) {  step =  1.0;   substep=0.25; }
  else if( dblMinStep <=  2.0 ) {  step =  2.0;   substep=0.5;  } // not 2.5 - no decimals in the labels !
  else if( dblMinStep <=  5.0 ) {  step =  5.0;   substep=1.0;  }
  else if( dblMinStep <= 10.0 ) {  step = 10.0;   substep=2.5;  }
  else if( dblMinStep <= 25.0 ) {  step = 25.0;   substep=5.0;  }
  else if( dblMinStep <= 50.0 ) {  step = 50.0;   substep=10.0; }
  else if( dblMinStep <= 100.0) {  step = 100.0;  substep=25.0; }
  else if( dblMinStep <= 250.0) {  step = 250.0;  substep=50.0; }
  else if( dblMinStep <= 500.0) {  step = 500.0;  substep=100.0; }
  else                          {  step = 1000.0; substep=100.0; }
  step    *= fact_pow10;
  substep *= fact_pow10;

  if( pdblLargeStep != NULL )
   { *pdblLargeStep = step;
   }
  if( pdblSubStep != NULL )
   { *pdblSubStep = substep;
   }
}

//---------------------------------------------------------------------------
void SpecDisp_InitControl( T_SpecDispControl *pDispCtrl, int iFreqScaleHeight )
{
  memset( pDispCtrl, 0, sizeof(T_SpecDispControl) );
  pDispCtrl->fltSpectrumAmplMin_dB = 0.0;   // 0..127 dB on the "Icom scale" ..
  pDispCtrl->fltSpectrumAmplMax_dB = 100.0; // .. but this seemed to be a realistic maximum
  pDispCtrl->fmin_from_RigCtrl = 7.000e6;  // <- phantasy frequency for testing w/o a real rig
  pDispCtrl->fmax_from_RigCtrl = 7.040e6;
  pDispCtrl->dblDisplayedVfoFreq_Hz = (pDispCtrl->fmin_from_RigCtrl + pDispCtrl->fmax_from_RigCtrl) / 2.0;
  pDispCtrl->fixedEdgeMode     = FALSE;


  pDispCtrl->iFreqScaleHeight = iFreqScaleHeight;
  pDispCtrl->iWFBrightness_Percent = pDispCtrl->iWFContrast_Percent = 50;
  pDispCtrl->iWFColorPalette = WF_PALETTE_SUNRISE; // author's favourite
  pDispCtrl->iColourScheme = COLOUR_SCHEME_DEFAULT;
  // Configure some of the 'markers' in various parts of the GUI (spectrum displays):
  pDispCtrl->marker[SPECDISP_MARKER_VFO].iGuiControl=GUI_CONTROL_SPECTRUM;

} // end SpecDisp_InitControl()

//---------------------------------------------------------------------------
BOOL SpecDisp_PrepareDrawing(
                 Graphics::TBitmap *pbmpDest,  // [in,out] Borland-VCL-style "TBitmap"
                 T_SpecDispControl *pDispCtrl, // [in,out] layout info, frequency range, etc
                 T_RigCtrlInstance *pRigCtrl)  // [in] "Rig Control" instance
{
  // 2024-03-01: Crash-landed in "__InitExceptBlockLDTC()" [wtf..?] shortly after
  //     entering SpecDisp_UpdateFreqScale(), thus the following pointer-checks:
  if( pbmpDest==NULL )
   { SpecDisp_iLastErrorLine = __LINE__;
     return FALSE; // VCL-style TBitmap not properly initialized !
   }
  if( pDispCtrl==NULL )
   { SpecDisp_iLastErrorLine = __LINE__;
     return FALSE;  // missing 'Display Control' instance !
   }
  if( pRigCtrl==NULL )
   { SpecDisp_iLastErrorLine = __LINE__;
     return FALSE;  // no valid RigControl instance !
   }
#ifdef __BORLANDC__  // here the implementation for Borland/Embarcadero C++Builder/VCL :
  if( pbmpDest->PixelFormat!=pf32bit )  // <- THE ENTIRE BORLAND IDE (!) crashed HERE when single-stepping
   { SpecDisp_iLastErrorLine = __LINE__;
     return FALSE; // Not compatible with our RGB-quad-mapper, T_RGBColor !
   }
  pDispCtrl->Canvas = pbmpDest->Canvas;
  if( pDispCtrl->Canvas==NULL )
   { SpecDisp_iLastErrorLine = __LINE__;
     return FALSE; // oops.. no TCanvas (~~~ graphic device context ?) for this TBitmap ?!
   }
  // Since we already have the VCL-TCanvas, prepare 'Brush'-, 'Pen'-, 'Font'-,
  // and some other colours, depending on the GUI's colour scheme:
  pDispCtrl->Canvas->Brush->Style = bsSolid;
  switch( pDispCtrl->iColourScheme )
   { case COLOUR_SCHEME_DEFAULT : // black text on white background
     default:
        pDispCtrl->Canvas->Brush->Color  = clWhite;
        pDispCtrl->Canvas->Pen->Color    = clBlack;
        pDispCtrl->Canvas->Font->Color   = clBlack;
        pDispCtrl->dwBackgndColor = RGB( 0xFF, 0xFF, 0xFF );
        pDispCtrl->dwGridColor    = RGB( 0x60, 0x60, 0x60 );
        pDispCtrl->dwCurveColor   = RGB( 0x00, 0x00, 0xFF );
        pDispCtrl->dwAuxCurveColor= RGB( 0x00, 0xFF, 0x00 ); // "auxiliary" curve colour, e.g. for the AUDIO SPECTRUM in the background
        pDispCtrl->dwAuxBkgndColor= RGB( 0x00, 0xC0, 0x00 ); // polygon-filling colour for the 1st "auxiliary" curve
        break;
     case COLOUR_SCHEME_DARK    : // white text on black background
        pDispCtrl->Canvas->Brush->Color  = clBlack;
        pDispCtrl->Canvas->Pen->Color    = clWhite;
        pDispCtrl->Canvas->Font->Color   = clWhite;
        pDispCtrl->dwBackgndColor = RGB( 0x00, 0x00, 0x00 );
        pDispCtrl->dwGridColor    = RGB( 0x60, 0x60, 0x60 );
        pDispCtrl->dwCurveColor   = RGB( 0xC0, 0xC0, 0xFF );
        pDispCtrl->dwAuxCurveColor= RGB( 0xC0, 0xFF, 0xC0 );
        pDispCtrl->dwAuxBkgndColor= RGB( 0x00, 0x50, 0x80 ); // polygon-filling colour for the 1st "auxiliary" curve
        break;
   } // end switch( pDispCtrl->iColourScheme )

  if( !RigCtrl_GetDisplayableRadioFrequencyRange( pRigCtrl,
            &pDispCtrl->fmin_from_RigCtrl, &pDispCtrl->fmax_from_RigCtrl,
            &pDispCtrl->fixedEdgeMode ) )
   { // failed to retrieve the spectrum / waterfall display range,
     // so (to see anything on the frequency axis at all), use this instead:
     // pDispCtrl->fmin_from_RigCtrl = ?;  // stick to the defaults from SpecDisp_InitControl(),
     // pDispCtrl->fmax_from_RigCtrl = ?;  // or whatever THE APPLICATION has set after that.
     // pDispCtrl->fixedEdgeMode     = ?;
   }
  pDispCtrl->iCanvasWidth  = pbmpDest->Width;
  pDispCtrl->iCanvasHeight = pbmpDest->Height;
  return TRUE;
#else
#  error "Sorry, this module hasn't been ported to other targets yet."
#endif
} // end SpecDisp_PrepareDrawing()

//---------------------------------------------------------------------------
void SpecDisp_FillRect(
      T_SpecDispControl *pDispCtrl, // [in,out] layout info, config data, etc
      DWORD dwColor,                // [in] colour as 24-bit value 0x00BBGGRR
      int x1, int y1, int x2, int y2 )
  //
{
#ifdef __BORLANDC__  // here the implementation for Borland/Embarcadero C++Builder/VCL :
  if( pDispCtrl->Canvas != NULL )
   { pDispCtrl->Canvas->Brush->Color = (TColor)dwColor;
     pDispCtrl->Canvas->FillRect( TRect(x1,y1,x2+1/*!*/,y2+1/*!*/) );
   }
#else
#  error "Sorry, this module hasn't been ported to other targets yet."
#endif
} // end SpecDisp_FillRect()

//---------------------------------------------------------------------------
void SpecDisp_FillPolygon(
      T_SpecDispControl *pDispCtrl, // [in,out] layout info, config data, etc
      TPoint *poly_points, int nPoints )
  // From the WIN32 'GDI' (not 'GDI+') documentation:
  // > The Polygon function draws a polygon consisting of two or more vertices
  // > connected by straight lines. The polygon is outlined by using the current
  // > pen and filled by using the current brush and polygon fill mode.
  // > Remarks
  // >  The polygon is closed automatically by drawing a line
  // >  from the last vertex to the first.
  //   Note (by WB): Windows defines a "POINT" struct, with two members: "LONG x; LONG y;".
  //                 Borland's VCL uses a "TPoint" instead, most likely compatible.
{
#ifdef __BORLANDC__  // here the implementation for Borland/Embarcadero C++Builder/VCL :
  if( pDispCtrl->Canvas != NULL )
   { pDispCtrl->Canvas->Polygon( poly_points, nPoints );
   }
#else
#  error "Sorry, this module hasn't been ported to other targets yet."
#endif

} // end SpecDisp_FillPolygon()


//-------------------------------------------------------------------------
void SpecDisp_DrawCircle( T_SpecDispControl *pDispCtrl,
                          int x,int y, int radius)
{
#ifdef __BORLANDC__  // here the implementation for Borland/Embarcadero C++Builder/VCL :
  if( pDispCtrl->Canvas != NULL )
   { pDispCtrl->Canvas->Ellipse( x-radius, y-radius, x+radius, y+radius );
   }
#else
#  error "Sorry, this module hasn't been ported to other targets yet."
#endif
} // end SpecDisp_DrawCircle()


//-------------------------------------------------------------------------
void SpecDisp_DrawLine(T_SpecDispControl *pDispCtrl, int x1, int y1, int x2, int y2)
{
#ifdef __BORLANDC__  // here the implementation for Borland/Embarcadero C++Builder/VCL :
  if( pDispCtrl->Canvas != NULL )
   { pDispCtrl->Canvas->MoveTo(x1,y1);
     pDispCtrl->Canvas->LineTo(x2,y2); // this does not include the end coord !
   }
#else
#  error "Sorry, this module hasn't been ported to other targets yet."
#endif
} // end SpecDisp_DrawLine()

//-------------------------------------------------------------------------
void SpecDisp_DrawText(T_SpecDispControl *pDispCtrl, int x, int y, const char *pszText)
{
#ifdef __BORLANDC__
  if( pDispCtrl->Canvas != NULL )
   { pDispCtrl->Canvas->TextOut(x,y,pszText);
   }
#else
#  error "Sorry, this module hasn't been ported to other targets yet."
#endif
} // end SpecDisp_DrawText()

//-------------------------------------------------------------------------
void SpecDisp_GetTextExtent( T_SpecDispControl *pDispCtrl,
                             const char *pszString, int *piWidth, int *piHeight)
  // Computes the width and height of the specified string of text,
  // as if it was drawn into the destination (bitmap) via SpecDisp_DrawText().
{
#ifdef __BORLANDC__
  TSize textSize;
  textSize = pDispCtrl->Canvas->TextExtent( pszString );  // the "VCL" way of doing things..
  if(piWidth)  *piWidth = textSize.cx; // the name (TSize) has changed, but not the principle
  if(piHeight) *piHeight= textSize.cy;
#elif( defined __WIN32__ ) // here the implementation for "bare Win32" (no VCL):
  HDC  hdc = ?;  // how to retrieve the Win32 GDI "device context" for what used to be the TCanvas ?
  SIZE size;
  if(GetTextExtentPoint32( hdc, pszString, strlen(pszString), &size) )
   { if(piWidth) *piWidth=size.cx;
     if(piHeight)*piHeight=size.cy;
   }
  else // function failed..
   { if(piWidth) *piWidth=1;
     if(piHeight)*piHeight=1;
   }
#else
#  error "Sorry, this module hasn't been ported to other targets yet."
#endif
} // end SpecDisp_GetTextExtent()


//---------------------------------------------------------------------------
static int SpecDisp_FreqToScreenX( T_SpecDispControl *pDispCtrl, double freq_Hz )
{ // Converts a frequency [Hz] into a screen offset (client abscissa, "x"),
  // for the waterfall area (and the associated frequency scale).
  // Note: To avoid desaster with invalid coordinates, the result will be
  //       clipped "almost" inside the 'waterfall coordinate range' :
  //   Return value = -1 : "frequency is TOO LOW to be displayed"
  //                  0 .. pDispCtrl->iCanvasWidth - 1 : "frequency is ON-SCREEN"
  //                  pDispCtrl->iCanvasWidth : .. TOO HIGH to be displayed .
  double fmin = pDispCtrl->fmin_from_RigCtrl;
  double fmax = pDispCtrl->fmax_from_RigCtrl;
  double span = fmax - fmin;
  int x = 0;
  if( span > 0 )
   { x = (int)( ( (freq_Hz-fmin) * (double)pDispCtrl->iCanvasWidth ) / span );
   }
  if( x < -1 )
   {  x = -1;
   }
  if( x > pDispCtrl->iCanvasWidth )
   {  x = pDispCtrl->iCanvasWidth;
   }
  return x;
} // SpecDisp_FreqToScreenX()

//---------------------------------------------------------------------------
static double ScreenXToFreq( T_SpecDispControl *pDispCtrl, int x )
{ // Inverse function to SpecDisp_FreqToScreenX() .
  double fmin = pDispCtrl->fmin_from_RigCtrl;
  double fmax = pDispCtrl->fmax_from_RigCtrl;
  double span = fmax - fmin;
  if( pDispCtrl->iCanvasWidth > 0 )
   { // inverse to x = (int)( ( (freq_Hz-fmin) * (double)pDispCtrl->iCanvasWidth ) / span ) ,
     // solved for freq_Hz :
     return fmin + ((double)x * span) / (double)pDispCtrl->iCanvasWidth;
   }
  else
   { return 0.0;
   }
} // ScreenXToFreq()


//---------------------------------------------------------------------------
void SpecDisp_GetFreqScaleParams( T_SpecDispControl *pDispCtrl,
                  double *pdblStartFreq,
                  double *pdblLargeSteps_Hz, double *pdblSmallSteps_Hz,
                  char   *psz16FormatString )
{ // Select a suitable frequency step for the ticks .. must depend on the
  //  frequency span, AND the display width.
  double fmin = pDispCtrl->fmin_from_RigCtrl;
  double fmax = pDispCtrl->fmax_from_RigCtrl;
  double freq; // 'minimum spacing' between labels [Hz]

  if( pDispCtrl->iCanvasWidth > 0 )
   { freq = 80/*pixels*/ * (fmax - fmin) / pDispCtrl->iCanvasWidth;
     // '--> min value of "Hertz per tick", but often a very odd number
   }
  else
   { freq = 1000.0; // Hz
   }
  GetStepAndSubstepForScale( freq, pdblLargeSteps_Hz, pdblSmallSteps_Hz);
  // Initial test with an IC-7300 :
  //    fmin = 7031970 [Hz],  fmin = 7041970 [Hz], pDispCtrl->iCanvasWidth = 1366 [pixels]
  // -> fstep=     500 [Hz],  substep = 100 [Hz] .  That's 10 kHz / 500 Hz = 20 'labelled ticks' .
  freq  = fmin;
  freq -= fmod( freq, *pdblLargeSteps_Hz );
  *pdblStartFreq = freq;

  if( psz16FormatString != NULL )  // caller needs a FORMAT STRING for the labels..
   { strcpy( psz16FormatString, "%.5lf" );
     // ToDo: Let the number of decimals (fractional digits) depend on the
     //       'large stepwidth' !
   }

} // end SpecDisp_GetFreqScaleParams()

//----------------------------------------------------------------------------
void SpecDisp_BrightnessToWaterfallColour(
                int iWFColorPalette, // [in] WF_PALETTE_RED, WF_PALETTE_GREEN, WF_PALETTE_BLUE,
                                     //      WF_PALETTE_SUNRISE, WF_PALETTE_LINRAD, ...(?)
                int iBrightness,     // [in] brightness (aka "intensity") ranging from 0 to 255
                T_RGBColor *pColour) // [out] red, green, and blue colour component, all ranging from 0..255
{ int red,green,blue;
  LimitInteger( &iBrightness, 0, 255 ); // prevent array index violations with the 256-colour palette arrays
  switch( iWFColorPalette )
   {
     case WF_PALETTE_SUNRISE: // 'sunrise' palette from Spectrum Lab
     default:
          red   = SunriseColors[iBrightness] & 0xFF;
          green =(SunriseColors[iBrightness] >> 8) & 0xFF;
          blue  =(SunriseColors[iBrightness] >> 16) & 0xFF;
          break;
     case WF_PALETTE_LINRAD : // 'Linrad'-like palette (rainbow-like)
          red   = LinradColors[iBrightness] & 0xFF;
          green =(LinradColors[iBrightness] >> 8) & 0xFF;
          blue  =(LinradColors[iBrightness] >> 16) & 0xFF;
          break;
     case WF_PALETTE_RED    : // "Shades of Red" waterfall display ...
          red   = iBrightness * 2;
          // .. but turning WHITE towards the endstop :
          if( iBrightness >= 128 )
           { green = blue = (iBrightness-128) * 2;
           }
          else
           { green = blue = 0;
           }
          break;
     case WF_PALETTE_GREEN  : // "Shades of Green" waterfall display
          green = iBrightness * 2;
          if( iBrightness >= 128 )
           { red = blue = (iBrightness-128) * 2;
           }
          else
           { red = blue = 0;
           }
          break;
     case WF_PALETTE_BLUE :     // "Shades of Blue" waterfall display
          blue  = iBrightness * 2;
          if( iBrightness >= 128 )
           { red = green = (iBrightness-128) * 2;
           }
          else
           { red = green = 0;
           }
          break;
   }
  LimitInteger( &red,   0, 255 );
  LimitInteger( &green, 0, 255 );
  LimitInteger( &blue,  0, 255 );
  pColour->b[RGB_BYTE_RED]   = red;   // write directly into the memory-mapped, off-screen bitmap
  pColour->b[RGB_BYTE_GREEN] = green;
  pColour->b[RGB_BYTE_BLUE]  = blue;
  pColour->b[RGB_BYTE_ALPHA] = 0x00;
} // end SpecDisp_BrightnessToWaterfallColour()


//---------------------------------------------------------------------------
void SpecDisp_UpdateWaterfall( // aka spectrogram ..
        Graphics::TBitmap *pbmpDest,  // [out] Borland-VCL-style "TBitmap"
        T_SpecDispControl *pDispCtrl, // [in] layout info, config data, options, etc
        T_RigCtrlInstance *pRigCtrl,  // [in] "Rig Control" instance
        int iScopeDisplayTailIndex,   // [in] index of the OLDEST spectrum to plot
        int nSpectra )                // [in] number of spectra (waterfall lines) to plot
{
  double freq, fstep, substep;
  int iBrightness;
  int i,n,x,y,w,h,tw,th,x1,y1,x2,y2;
  int iLine, iTenSecondTick;
  BOOL first;
  T_RigCtrl_Spectrum *pSpectrum;
  float flt, fltFreqToBinIdx, fltPeak;
  int   iBinIdx, iPrevBinIdx;
  long  i32SpectrumIndex;
  T_RGBColor *pPixels;  // <- this only works with bitmap 'PixelFormat' = 'pf32bit' !
  char  szLabel[44];

  if( ! SpecDisp_PrepareDrawing( pbmpDest, pDispCtrl, pRigCtrl ) )
   { return; // something wrong with any of the input arguments -> bail out
   }

  if( pDispCtrl->fClearWaterfall )
   {
     SpecDisp_FillRect( pDispCtrl, pDispCtrl->dwBackgndColor,
         0,0, pDispCtrl->iCanvasWidth, pDispCtrl->iCanvasHeight );
     pDispCtrl->fClearWaterfall = FALSE;
   }

  // How many lines of the waterfall must be updated ?
  if( nSpectra > 0 )
   {
     // Scroll the (old) waterfall image down by as many lines as required:
     if( nSpectra < pDispCtrl->iCanvasHeight )
      { TRect rctSource = TRect( 0/*left*/, 0/*top*/, pbmpDest->Width/*right*/,
                                 pbmpDest->Height-nSpectra/*bottom*/);
        TRect rctDest   = TRect( 0/*left*/, nSpectra/*top*/, pbmpDest->Width/*right*/,
                                 pbmpDest->Height/*bottom*/);
        pDispCtrl->Canvas->CopyRect( // <- wrapper for BitBlt() with parameter 'SRCCOPY' ?
                                 rctDest, pDispCtrl->Canvas/*Source*/, rctSource );
      }

     // Plot a few more spectrogram lines into the "waterfall"
     //   (begin with most recent on top, thus the name .. "like water, falling down")
     for(iLine=0; iLine < nSpectra; ++iLine )
      { int iTailIndex = iScopeDisplayTailIndex-(nSpectra-1-iLine);
        pSpectrum = RigCtrl_GetSpectrumFromFIFO( pRigCtrl, &iTailIndex );
        pPixels   = (T_RGBColor*)pbmpDest->ScanLine[iLine];
        if( (pSpectrum != NULL) && (pPixels!=NULL) )
         { // Since neither the width of the window, not the FFT size is fixed anymore,
           // the ratio between 'screen pixels' and 'FFT frequency bins' is arbitrary.
           //
           // Principle used below:
           // Running along the frequency axis, the program looks into
           //    a RANGE(!) of FFT bins(!) to get the color of a pixel.
           // There may be less FFT bins than pixels, there may be more,
           // and only if we're lucky there is a 'perfect match' (one pixel per bin).
           if( pSpectrum->dblBinWidth_Hz > 0.0 )
            { fltFreqToBinIdx = 1.0 / pSpectrum->dblBinWidth_Hz;
            }
           else // avoid div-by-zero..
            { fltFreqToBinIdx = 0.0;
            }
           iPrevBinIdx = -1;
           for(x = 0; x < pDispCtrl->iCanvasWidth; ++x)
            {
              // Convert the x-coord to a frequency (in Hertz):
              freq = ScreenXToFreq( pDispCtrl, x );
              // Convert the pixel's center frequency to an FFT frequency bin index:
              iBinIdx = (int)( 0.5 + (freq - pSpectrum->dblFmin_Hz) * fltFreqToBinIdx);
              if( iBinIdx>=0 && iBinIdx<pSpectrum->nBinsUsed )  // an IC-7300 delivered 475 frequency bins...
               { // .. which is FIVE LESS than expected for a 480-pixel-wide TFT panel (IC-7300 & Co)
                 if( iPrevBinIdx < 0 )
                  {  iPrevBinIdx = iBinIdx-1;  // 1st loop, without a "previous" frequency bin
                     if( iPrevBinIdx < 0 )
                      {  iPrevBinIdx = 0;
                      }
                     fltPeak = pSpectrum->fltMagnitudes_dB[iBinIdx/*!*/];
                  }
                 if( iPrevBinIdx < iBinIdx ) // Is this a new FFT-bin at all ?
                  { fltPeak = pSpectrum->fltMagnitudes_dB[++iPrevBinIdx];
                    while( iPrevBinIdx < iBinIdx ) // MORE THAN ONE BIN per pixel : get the peak..
                     { flt = pSpectrum->fltMagnitudes_dB[++iPrevBinIdx];
                       if( flt>fltPeak )
                        {  fltPeak = flt;
                        }
                     }
                  } // end if( iBinIndex > iPrevBinIdx )
                 else  // not a new FFT frequency bin index: use the same 'fltPeak' as for the left neighbour
                  {
                  }
               } // end if( iBinIdx>0 && iBinIdx<pSpectrum->nBinsUsed )

              // Convert fltPeak with Icom's "magnitude in dB" into a pixel colour .
              //  'iWFBrightness_Percent'  = trackbar position, ranging from 0 to 100 .
              //  'fltPeak' = FFT bin amplitude (proportional to a voltage),
              //         normalized to 0....1 for a full-swing sinewave w/o AGC .
              //  'iBrightness' = 0..255 .
              //  The factor below was empirically found to have a similar
              //  brightness as on an IC-7300's own display, with both
              //  'Contrast' and 'Brightness' set to mid-range (50 %).
              // How to "apply" the contrast and brightness percentage
              //     to the logarithmized voltages (here: in dB) ?
              // - 'Contrast' is a kind of GAIN adjustment, i.e. multiplied with the dB value
              // - 'Brightness' is an OFFSET adjustment, i.e. added or subtracted from the brightness
              flt = (  fltPeak/* from Icom: 0..127 dB relative to a phantasy reference (*) */
                     + 2.0*(float)pDispCtrl->iWFBrightness_Percent/* adds +/-100 dB to the "phantasy reference" */
                     - 127.0
                    ) // now MULTIPLY with the 'contrast' control value:
                  * ( 0.5 + 0.1 * (float)pDispCtrl->iWFContrast_Percent );
                 // (*) More on Icom's "phantasy unit" in their "waveform data"
                 //     in RigControl.c : RigCtrl_ParseSpectrumData_CIV() .....
              iBrightness/*0..255*/ = 127/* mid range of 0..255*/ + (int)flt;
              SpecDisp_BrightnessToWaterfallColour( pDispCtrl->iWFColorPalette, iBrightness, pPixels );
              ++pPixels;
            }  // end for x...

           // After plotting another line of pixels: Draw a "time marker" into the spectrogram ?
           ++pDispCtrl->iTimeMarkerCounter;
           if( pDispCtrl->iDisplayOptions & SPEC_DISP_OPTIONS_TIME_MARKERS )
            { // Unfortunately, we cannot really control the 'scope update rate',
              // so draw a time marker only if ...
              //   (a) enough spectra have been plotted since the last time marker,
              //       to avoid overlap between the labels
              //   (b) the spectrum's timestamp (in UTC) has crossed a 10-second gap,
              //       or whatever appears appropriate for the "measured" interval
              // That way, with the remote receiver tuned to one of the five
              // NCDXF/IARU international beacon frequencies, traces of the currently
              // active beacon (with the callsign shown in the 'bottom bar' below
              // the frequency) can be easily identified on the waterfall...
              // and allow checking the remote receiver's system time.
              //    (Why ? Because pSpectrum->dblUnixTime is provided by the
              //     SERVER, so it's completely independent from the
              //     CLIENT's system time in UTC.)
              // Besides that, it's fun to monitor to NCDXF/IARU International Beacons
              // for a short while, to decide if a band is "sufficiently open".
              iTenSecondTick = (int)fmodl( pSpectrum->dblUnixTime * 0.1, 100.0 );
              if(  (iTenSecondTick != pDispCtrl->iLastTimeMarkerTick_10s )
                && ( (pDispCtrl->iTimeMarkerCounter >= 20) || (pDispCtrl->iTimeMarkerCounter < 0 ) )
                )
               { // Typically got every 10 seconds with pDispCtrl->iTimeMarkerCounter = 40 +/- a few..
                 pDispCtrl->iTimeMarkerCounter = 0;
                 pDispCtrl->iLastTimeMarkerTick_10s = iTenSecondTick;

                 UTL_FormatDateAndTime( "hh:mm:ss", pSpectrum->dblUnixTime, szLabel );
                 // Prepare a sufficiently SMALL font for the TIME LABEL :
                 pDispCtrl->Canvas->Font->Name  = "Arial";
                 pDispCtrl->Canvas->Font->Size  = 8;
                 // Already set by SpecDisp_PrepareDrawing():
                 //    pDispCtrl->Canvas->Font->Color, pDispCtrl->Canvas->Pen->Color,
                 //    etc, depending on the currently selected COLOUR SCHEME.
                 SpecDisp_GetTextExtent( pDispCtrl, szLabel, &tw, &th );  // typical width ("23:59:59") : 54 pixels
                 SpecDisp_DrawText( pDispCtrl, 0/*x*/, iLine/*y*/, szLabel );
                 // To keep it simple, the label cannot be vertically centered
                 // on the line of pixels it belongs to (for such a feature,
                 // it would have to be DRAWN REPEATEDLY until completely visible,
                 // as the waterfall scrolls DOWN). To indicate the label text
                 // applies to the line of pixels at its TOP, draw this TICK,
                 // directly ABOVE the text :
                 SpecDisp_DrawLine( pDispCtrl, 0/*x1*/, iLine/*y1*/, tw/*x2*/, iLine/*y2*/ );
               }
            }  // end if < SPEC_DISP_OPTIONS_TIME_MARKERS > ?
         }    // end if( pSpectrum != NULL ) && ...
      }      // end  for(iLine... )
     pDispCtrl->fUpdateWaterfall = FALSE; // "done" (waterfall display is up-to-date now)
   }       // end if( nSpectra > 0 )
} // end SpecDisp_UpdateWaterfall()

//---------------------------------------------------------------------------
void SpecDisp_UpdateSpectrum(
        Graphics::TBitmap *pbmpDest,  // [out] Borland-VCL-style "TBitmap"
        T_SpecDispControl *pDispCtrl, // [in] layout info, config data, etc
        T_RigCtrlInstance *pRigCtrl)  // [in] "Rig Control" instance (provides
             // spectrum data, VFO frequency, and maybe even RIT / XIT offsets)
{
  T_RigCtrl_Spectrum *pSpectrum;
  double freq, fmax, fstep, substep;
  float flt, fltFreqToBinIdx, fltPeak, fltScalingFactor;
  int   i, n, tw, th, thh, iBinIdx, iPrevBinIdx, x,y, prev_x, prev_y, x_covered;
  TPoint *pPoints, poly_points[4];
  int     nPoints = 0;
  char   szLabel[44], sz16FormatString[16+4];
  T_SpecDispMarker *pMarker;

  if( ! SpecDisp_PrepareDrawing( pbmpDest, pDispCtrl, pRigCtrl ) )
   { return; // something wrong with any of the input arguments -> bail out
   }
  fmax = pDispCtrl->fmax_from_RigCtrl; // <- updated in SpecDisp_PrepareDrawing()..


  pPoints = (TPoint*)malloc( pDispCtrl->iCanvasWidth * sizeof(POINT) );
  if(pPoints==NULL) // oops .. malloc() failed !
   { return;
   }
  // NO "LAZY RETURN" after this point !

  fltScalingFactor = (float)pDispCtrl->iCanvasHeight;
  flt = pDispCtrl->fltSpectrumAmplMax_dB - pDispCtrl->fltSpectrumAmplMin_dB;
  if( flt > 0.0 ) // avoid div-by-zero
   { fltScalingFactor /= flt;
   }

  // First clear the background, then draw stuff "way in the background",
  //             then draw the grid, then draw other curve(s)...
  pDispCtrl->Canvas->Brush->Color = (TColor)pDispCtrl->dwBackgndColor;
  pDispCtrl->Canvas->Pen->Color   = (TColor)pDispCtrl->dwGridColor;
  SpecDisp_FillRect( pDispCtrl, pDispCtrl->dwBackgndColor,
     0/*x1*/,0/*y1*/, pDispCtrl->iCanvasWidth, pDispCtrl->iCanvasHeight );

  // Draw the AUDIO SPECTRUM (from the CW decoder) into the background ?
  if( ( pDispCtrl->iDisplayOptions & SPEC_DISP_OPTIONS_CW_DECODER_SPECTRUM )
   && ( CwKeyer_DSP.AudioCwDecoder[0].iCwDecoderState != CWDSP_DECODER_STATE_OFF) )
   { pDispCtrl->fAudioSpectrumVisible = TRUE;
     SpecDisp_DrawAudioSpectrumFromCwDSP( pDispCtrl, pRigCtrl, &CwKeyer_DSP );
   } // end if < show the spectrum calculated by the AUDIO CW DECODER > ?
  else
   { pDispCtrl->fAudioSpectrumVisible = FALSE;
   }

  for(i=0; i<SPECDISP_NUM_MARKERS; ++i)
   { pMarker = &pDispCtrl->marker[i];
     if( pMarker->fVisible && (pMarker->iGuiControl==GUI_CONTROL_SPECTRUM) )
      { // Adjust the position of certain "special" markers here:
        switch(i)
         { case SPECDISP_MARKER_VFO: // area that allows dragging the VFO (frequency) ..
              x = SpecDisp_FreqToScreenX( pDispCtrl, pRigCtrl->dblVfoFrequency );
              pMarker->x1 = x-5;
              pMarker->x2 = x+5;
              pMarker->y1 = 0;
              pMarker->y2 = pDispCtrl->iCanvasHeight-1;
              if( pDispCtrl->iColourScheme == COLOUR_SCHEME_DARK )
               { // white text on black background -> darker marker
                 if( pDispCtrl->fDraggingVfoFrequency )
                  { pMarker->color.dw = RGB( 0x00, 0xA0, 0x00 );
                  }
                 else
                  { pMarker->color.dw = RGB( 0x00, 0x80, 0x00 );
                  }
               }
              else // COLOUR_SCHEME_DEFAULT : black text on white background
               { if( pDispCtrl->fDraggingVfoFrequency )
                  { pMarker->color.dw = RGB( 0xC0, 0xFF, 0xC0 );
                  }
                 else
                  { pMarker->color.dw = RGB( 0xC0, 0xFF, 0xC0 );
                  }
               }
              break; // end case SPECDISP_MARKER_VFO
           default:
              break;
         }
        SpecDisp_FillRect( pDispCtrl, pMarker->color.dw,
                                      pMarker->x1, pMarker->y1,
                                      pMarker->x2, pMarker->y2);
      }
   }

  // Prepare drawing colours for the grid :
  pDispCtrl->Canvas->Brush->Color = (TColor)pDispCtrl->dwBackgndColor;
  pDispCtrl->Canvas->Pen->Color   = (TColor)pDispCtrl->dwGridColor;

  // Prepare a sufficiently SMALL font for the grid :
  pDispCtrl->Canvas->Font->Name  = "Arial";
  pDispCtrl->Canvas->Font->Size  = 8;
  pDispCtrl->Canvas->Font->Color = (TColor)pDispCtrl->dwGridColor;
  th = pDispCtrl->Canvas->TextHeight( "Q" );
  thh= th / 2; // half text height for alignment


  // The HORIZONTAL grid lines mark 10 dB steps ..
  flt = 0.0;
  prev_y = pDispCtrl->iCanvasHeight;
  while( flt <= 100.0/*dB*/ )
   { y = (int)(( flt - pDispCtrl->fltSpectrumAmplMin_dB ) * fltScalingFactor);
     y = pDispCtrl->iCanvasHeight - 1 - y; // same as in the curve-plotting loop below..
     SpecDisp_DrawLine( pDispCtrl, 0, y, pDispCtrl->iCanvasWidth, y );
     if( (y<(prev_y-thh)) && (y>thh) && (y<(pDispCtrl->iCanvasHeight-thh) ) )
      { sprintf( szLabel, "%d dB", (int)flt );
        prev_y = y-thh;
        SpecDisp_DrawText( pDispCtrl, pDispCtrl->iCanvasWidth - 40, prev_y, szLabel );
      }
     flt += 10.0/*dB*/;
   }

  // The VERTICAL grid lines must match the 'large ticks' on the frequency scale,
  //   thus use a common subroutine to calculate their positions.
  //   Considered "more important" than the amplitude grid, thus drawn afterwards.
  SpecDisp_GetFreqScaleParams( pDispCtrl, &freq, &fstep, &substep, sz16FormatString );
  n = 0;
  x_covered = 0;
  while( ((n++)<100) && (freq<fmax) ) // vertical grid lines ...
   { x = SpecDisp_FreqToScreenX( pDispCtrl, freq );
     SpecDisp_DrawLine( pDispCtrl, x, 0/*y1*/, x, pDispCtrl->iCanvasHeight/*y2*/ );
     sprintf( szLabel, sz16FormatString, (double)(freq*1e-6) ); // Megahertz !
     tw = pDispCtrl->Canvas->TextWidth( szLabel );
     if( (x+tw/2) > x_covered )
      { SpecDisp_DrawText( pDispCtrl, x-tw/2, 2/*y:top*/, szLabel ); // frequency [MHz]
        x_covered = x+tw/2;
      }
     freq += fstep;
   }


  // If the "VFO frequency" is within the displayed range, show it similar to the display in an IC-7300:
  //  * The  "RX" frequency marker is GREEN, and starts with a small triangle
  //      pointing down from the upper end of the SPECTRUM display.
  //  * The  "TX" frequency marker is ORANGE, and when equal to the "RX" frequency,
  //      appears as a green/orange dashed line .
  //  * If the mouse pointer is close enough to the VFO frequency to "drag" it,
  //      the marker
  x = SpecDisp_FreqToScreenX( pDispCtrl, pRigCtrl->dblVfoFrequency );
      // ,--------------------------------|_____________|
      // '--> 2024-07: When inspecting pRigCtrl->dblVfoFrequency in the IDE here, the value was ok.
      //               But after STEPPING INTO SpecDisp_FreqToScreenX(), the argument 'double freq_Hz' was trashed.
      //               Fixed by doing a "build all" with Borland C++ Builder V6 .
      //               Reason: Caching of pre-compiled HEADERS doesn't work
      //                       reliably anymore, at least not under "modern Windows".
      //                       Remember this when a similar bug bites again.
  if( (x>=0) && (x<pDispCtrl->iCanvasWidth) )
   { // the "current" VFO frequency is within the displayed range,
     // but is it the TRANSMIT and/or RECEIVE frequency ? (care for this later) ..
     pDispCtrl->Canvas->Pen->Color = (TColor)0x00C0FF; // "clOrange" doesn't exist in the VCL,
            // so use a hex constant(0xBBGRR; the eight RED BITS are the LSByte)
     SpecDisp_DrawLine( pDispCtrl, x, 0/*y1*/, x, pDispCtrl->iCanvasHeight/*y2*/ );
     // > Use Polygon() to draw a closed, many-sided shape on the canvas,
     // > using the value of Pen. After drawing the complete shape,
     // > Polygon() fills the shape using the value of Brush.
     // > The Points parameter is an array of points that give the vertices of the polygon.
     // > Note: The Points_Size parameter is the index of the last point
     // >       in the array (one less than the total number of points).
     // > The first point is always connected to the last point .
     poly_points[0] = Point( x-4, 0 ); // upper left corner of the triangle
     poly_points[1] = Point( x+4, 0 ); // upper right corner of the triangle
     poly_points[2] = Point( x ,  4 ); // lower corner (tip pointing down)
     pDispCtrl->Canvas->Brush->Color = pDispCtrl->Canvas->Pen->Color;
     pDispCtrl->Canvas->Polygon( poly_points, 2/*!*/ ); // draw triangle as a filled polygon
   } // end if < pRigCtrl->dblVfoFrequency within the displayed frequency range > ?

  // Plot the MOST RECENT spectrum as a curve into the FOREGROUND (thus last)
  int iTailIndex = RIGCTRL_GET_LATEST_ENTRY;
  pSpectrum = RigCtrl_GetSpectrumFromFIFO( pRigCtrl, &iTailIndex );
  if( pSpectrum != NULL ) // ok, got a valid spectrum so PLOT IT:
   { // Since neither the width of the window, not the FFT size is fixed anymore,
     // the ratio between 'screen pixels' and 'FFT frequency bins' is arbitrary.
     // Use the same principle to transform pixel coordinate (x)
     // into frequency bin numbers as in SpecDisp_UpdateWaterfall() :
     if( pSpectrum->dblBinWidth_Hz > 0.0 )
      { fltFreqToBinIdx = 1.0 / pSpectrum->dblBinWidth_Hz;
      }
     else // avoid div-by-zero..
      { fltFreqToBinIdx = 0.0;
      }
     iPrevBinIdx = -1;
     fltPeak = pSpectrum->fltMagnitudes_dB[0];
     for(int x = 0; x < pDispCtrl->iCanvasWidth; ++x)
      {
        // Convert the x-coord to a frequency (in Hertz):
        freq = ScreenXToFreq( pDispCtrl, x );
        // Convert the pixel's center frequency to an FFT frequency bin index:
        iBinIdx = (int)( 0.5 + (freq - pSpectrum->dblFmin_Hz) * fltFreqToBinIdx);
        if( iBinIdx>=0 && iBinIdx<pSpectrum->nBinsUsed )  // an IC-7300 delivered 475 frequency bins...
         {
           if( iPrevBinIdx < 0 )
            {  iPrevBinIdx = iBinIdx-1;  // 1st loop
               if( iPrevBinIdx < 0 )
                {  iPrevBinIdx = 0;
                }
            }
           if( iPrevBinIdx < iBinIdx ) // Is this a new FFT-bin at all ?
            { fltPeak = pSpectrum->fltMagnitudes_dB[++iPrevBinIdx];
              while( iPrevBinIdx < iBinIdx ) // MORE THAN ONE BIN per pixel : get the peak..
               { flt = pSpectrum->fltMagnitudes_dB[++iPrevBinIdx];
                 if( flt>fltPeak )
                  {  fltPeak = flt;
                  }
               }
            }
           // Convert Icom's "magnitude in dB" into a vertical pixel offset.
           //
           flt = ( fltPeak - pDispCtrl->fltSpectrumAmplMin_dB ) * fltScalingFactor;
           if( flt < 0.0 )
            {  flt = 0.0;
            }
           if( flt >= pDispCtrl->iCanvasHeight )
            {  flt =  pDispCtrl->iCanvasHeight-1;
            }
           y = pDispCtrl->iCanvasHeight - 1 - (int)flt;
           pPoints[nPoints].x   = x;  // append another point to the polyline
           pPoints[nPoints++].y = y;
         } // end if < valid FFT bin index >
      }  // end for x...

     if( nPoints >= 2 ) // draw the entire "curve" in a single GDI function call
      { pDispCtrl->Canvas->Pen->Color = (TColor)pDispCtrl->dwCurveColor;
        pDispCtrl->Canvas->Polyline( pPoints, nPoints-1 );
        // ,----------------------------------|_______|
        // '--> funny but true: Not the NUMBER OF POINTS,
        //                      but the NUMBER OF POINTS MINUS ONE !
      }
   } // end if( pSpectrum != NULL ) && ...
  pDispCtrl->fUpdateSpectrum = FALSE; // "done" (spectrum graph is up-to-date now)
  free(pPoints);

} // end SpecDisp_UpdateSpectrum()

//---------------------------------------------------------------------------
void SpecDisp_UpdateFreqScale(
        Graphics::TBitmap *pbmpDest,  // [out] Borland-VCL-style "TBitmap"
        T_SpecDispControl *pDispCtrl, // [in] layout info, config data, etc
        T_RigCtrlInstance *pRigCtrl)  // [in] "Rig Control" instance (with audio passband parameters)
{
  double freq, fmin, fmax, fstep, substep;
  int i,n,x,y,x1,y1,x2,y2,tw,th,x_covered,iDecoderIndex;
  int iLine;
  BOOL first;
  float flt, fltFreqToBinIdx, fltPeak, fltPeak_dBfs;
  int   iBinIdx, iPrevBinIdx;
  char  szLabel[80];    // also used for the yellow info text
  char  sz16FormatString[16+4];

  if( ! SpecDisp_PrepareDrawing( pbmpDest, pDispCtrl, pRigCtrl ) )
   { return; // something wrong with any of the input arguments -> bail out
   }

  fmin = pDispCtrl->fmin_on_FreqScale = pDispCtrl->fmin_from_RigCtrl; // <- polled LATER..
  fmax = pDispCtrl->fmax_on_FreqScale = pDispCtrl->fmax_from_RigCtrl; // .. to check if re-draw required,
     // just in case the user/operator changed the displayed range ON THE RADIO .
     //  fmin_from_RigCtrl and fmax_from_RigCtrl have been updated
     //  in SpecDisp_PrepareDrawing() already .

  pDispCtrl->Canvas->Brush->Color = (TColor)pDispCtrl->dwBackgndColor;
  pDispCtrl->Canvas->Pen->Color   = (TColor)pDispCtrl->dwGridColor;
  SpecDisp_FillRect( pDispCtrl, pDispCtrl->dwBackgndColor,
     0/*x1*/,0/*y1*/, pDispCtrl->iCanvasWidth, pDispCtrl->iCanvasHeight );

  SpecDisp_GetFreqScaleParams( pDispCtrl, &freq, &fstep, &substep, sz16FormatString );

  x1 = 0;
  x2 = pDispCtrl->iCanvasWidth - 1;
  y1 = 0;
  y2 = pDispCtrl->iCanvasHeight / 2;
# define L_HORZ_CLEARING 35
  x_covered = x1+L_HORZ_CLEARING;
  while( freq < fmax ) // draw frequency markers (vertical lines + labels) ...
   { x = SpecDisp_FreqToScreenX( pDispCtrl, freq );
     if( x>x1 && x<x2 )  // inside the frequency scale
      {
        SpecDisp_DrawLine(pDispCtrl, x, y1, x, y2);  // draw frequency marker ("large tick")
        if( (x>x_covered) && (x+L_HORZ_CLEARING)<x2 )  // enough clearance for a frequency label ?
         { sprintf( szLabel, "%d Hz", (int)(freq+0.5) );
           SpecDisp_GetTextExtent( pDispCtrl, szLabel, &tw, &th );  // typical width ("1000 Hz") : 54 pixels
           SpecDisp_DrawText( pDispCtrl, x-(tw/2), y2, szLabel );
           x_covered = x + tw + 3/*pixels*/;  // just in case a label is 'surprisingly long'
         }
      }
     freq += fstep;
   }
  // Similar for the 'sub-steps' (with smaller ticks and no labels) :
  freq  = fmin;
  freq -= fmod( freq, substep );
  y2 = pDispCtrl->iCanvasHeight / 4;
  while( freq < fmax ) // draw frequency markers (vertical lines + labels) ...
   { x = SpecDisp_FreqToScreenX( pDispCtrl, freq );
     if( x>x1 && x<x2 )  // inside the frequency scale
      { SpecDisp_DrawLine( pDispCtrl, x, y1, x, y2 ); // draw "small tick"
      }
     freq += substep;
   }
  pDispCtrl->fUpdateFreqScale = FALSE; // "done" (frequency scale is up-to-date now)


} // end SpecDisp_UpdateFreqScale()

//---------------------------------------------------------------------------
const char *SpecDisp_GetTextFromFreqListEntry( T_FreqListEntry *pFLE )
{ if( pFLE != NULL )
   { if( pFLE->sz7Callsign[0] != '\0' ) // If the entry has a CALLSIGN, show it because it's shorter
      { return pFLE->sz7Callsign;
      }
     if( pFLE->sz31StationName[0] != '\0' ) // second best alternative: "Station Name"
      { return pFLE->sz31StationName;
      }
   }
  return "";  // no valid entry -> return an EMPTY STRING (not a NULL pointer)
}

//---------------------------------------------------------------------------
const char *SpecDisp_ReplaceStationNameByNCDXFBeaconSchedule(
                const char *pszDummyName, // [in] dummy name like "NCDXF beacon", returned if the frequency is unknown
                long i32Frequency_Hz )
  // Quick-and-dirty "beacon clock" for NCDXF beacons on 20, 17, 15, 12, and 10 meters.
  // Even though the NCDXF/IARU International Beacon Project isn't as complete
  // as it used to be (when established in the 1990s), it's still usedful.
  // Worldwide beacon transmit cycle (musings for the simple implementation further below):
  //   - A full world-wide cycle takes THREE MINUTES, synchronized to UTC.
  //     This full cycle consists of 18 time slots a 10 seconds.
  //   - EACH beacon transmits for 10 seconds per band, in its exclusive time slot:
  //      e.g. (4U1UN with iBeaconIndex=0) :
  //            on 00:00 (iTimeSlot=0) on 14.100 MHz (iBandIndex=0),
  //               00:10 (iTimeSlot=1) on 18.110 MHz (iBandIndex=1),
  //               00:20 (iTimeSlot=2) on 21.150 MHz (iBandIndex=2),
  //               00:30 (iTimeSlot=3) on 24.930 MHz (iBandIndex=3),
  //               00:40 (iTimeSlot=4) on 28.200 MHz (iBandIndex=4).
  //   - After transmitting on 28.2 MHz, each beacon "takes a rest" until the next
  //      3-minute cyle. Thus, starting on 00:00 means it will start on 03:00 again, etc.
  //   - 4U1UN (iBeaconIndex=0)  starts its cycle at 00:00 (iTimeSlot=0) on iBandIndex=0,
  //     VE8AT (iBeaconIndex=1)  starts its cycle at 00:10 (iTimeSlot=1) on iBandIndex=0,
  //      W6WX (iBeaconIndex=2)  starts its cycle at 00:20 (iTimeSlot=2) on iBandIndex=0,
  //     KH6WO (iBeaconIndex=3)  starts its cycle at 00:30 (iTimeSlot=3) on iBandIndex=0,
  //      ZL6B (iBeaconIndex=4)  starts its cycle at 00:40 (iTimeSlot=4) on iBandIndex=0,
  //    VK6BRP (iBeaconIndex=5)  starts its cycle at 00:50 (iTimeSlot=5) on iBandIndex=0,
  //    JA2IGY (iBeaconIndex=6)  starts its cycle at 01:00 (iTimeSlot=6) on iBandIndex=0,
  //      RR9O (iBeaconIndex=7)  starts its cycle at 01:10 (iTimeSlot=7) on iBandIndex=0,
  //      VR2B (iBeaconIndex=8)  starts its cycle at 01:20 (iTimeSlot=8) on iBandIndex=0,
  //      4S7B (iBeaconIndex=9)  starts its cycle at 01:30 (iTimeSlot=9) on iBandIndex=0,
  //     ZS6DN (iBeaconIndex=10) starts its cycle at 01:40 (iTimeSlot=10) on iBandIndex=0,
  //      5Z4B (iBeaconIndex=11) starts its cycle at 01:50 (iTimeSlot=11) on iBandIndex=0,
  //     4X6TU (iBeaconIndex=12) starts its cycle at 02:00 (iTimeSlot=12) on iBandIndex=0,
  //      OH2B (iBeaconIndex=13) starts its cycle at 02:10 (iTimeSlot=13) on iBandIndex=0,
  //      CS3B (iBeaconIndex=14) starts its cycle at 02:20 (iTimeSlot=14) on iBandIndex=0,
  //     LU4AA (iBeaconIndex=15) starts its cycle at 02:30 (iTimeSlot=15) on iBandIndex=0,
  //      OA4B (iBeaconIndex=16) starts its cycle at 02:40 (iTimeSlot=16) on iBandIndex=0,
  // and  YV5B (iBeaconIndex=17) starts its cycle at 02:50 (iTimeSlot=17) on iBandIndex=0.
  //     Then the cycle repeats after 180 seconds = 18 "ten-second time slots"
  //     with 4U1UN at 03:00, etc etc.
  //       Thus (ignoring the SILENT time slots for a simple start) :
  //       iBandIndex = iTimeSlot - iBeaconIndex .
  //     Test/example: CS3B on 14.10 MHz (iBeaconIndex=14 iBandIndex=0)
  //                   should start at 02:20 (or 05:20, 08:20,.. iTimeSlot=14):
  //                    -> iBandIndex = 14 - 14 = 0. Ok. No higher maths involved.
  //                   CS3B on 18.11 MHz (iBeaconIndex=14 iBandIndex=1)
  //                   should start at 02:30 (or 05:30, 08:30,.. iTimeSlot=15):
  //                    -> iBandIndex = 15 - 14 = 1. Ok.
  //       So resolve for the 'wanted' beacon index:
  //        iBeaconIndex = (18 + iTimeSlot - iBandIndex) modulo 18
  //             |                  |           |
  //            0..17              0..17       0..4
  //
  // Note: The CURRENT STATUS can be checked at https://www.ncdxf.org/beacon/
  //     (this hard-coded "beacon schedule" doesn't care
  //      which of the beacons is actually still "on air", and on WHICH BAND)
{
  double dblUnixDateAndTime = UTL_GetCurrentUnixDateAndTime_Fast(); // .. in SECONDS, because that's the one-and-only SI unit for time
  int iTimeSlot  = (int)fmodl( dblUnixDateAndTime * 0.1, 18.0 ); // -> 0..17 (per 10-second interval)
  int iBandIndex = -1;
  int iBeaconIndex; // combination of time slot (0..17) and band index (0..4),
         // which (after a trick for the 'silent' slots) indicates THE BEACON.

#ifdef __BORLANDC__  // "... is assigned a value that is never used".. shut up ..
  (void)iBandIndex;
#endif


  switch( i32Frequency_Hz )
   { case 14100000L: iBandIndex=0; break;
     case 18110000L: iBandIndex=1; break;
     case 21150000L: iBandIndex=2; break;
     case 24930000L: iBandIndex=3; break;
     case 28200000L: iBandIndex=4; break;
     default: return pszDummyName;  // not an NCDFX/IARU International Beacon frequency !
   }
  iBeaconIndex = (18/*time slots*/ + iTimeSlot - iBandIndex) % 18;
     //           '--> This avoids a negative result,           |
     //                 This limits the index range to 0..17 <--'
  switch( iBeaconIndex )
   { case  0: return "4U1UN New York";
     case  1: return "VE8AT Inuvik";
     case  2: return "W6WX California";
     case  3: return "KH6RS Hawaii";
     case  4: return "ZL6B New Zealand";
     case  5: return "VK6BRP Australia";
     case  6: return "JA2IGY Japan";
     case  7: return "RR9O Siberia";
     case  8: return "VR2B Hong Kong";
     case  9: return "4S7B Sri Lanka";
     case 10: return "ZS6DN South Africa";
     case 11: return "5Z4B Kenya";
     case 12: return "4X6TU Israel";
     case 13: return "OH2B Finland";
     case 14: return "CS3B Madeira";
     case 15: return "LU4AA Argentinia";
     case 16: return "OA4B Peru";
     case 17: return "YV5B Venezuela";
     default: return pszDummyName;  // oops.. something wrong with the 'modulo trick' ?
   }

} // end SpecDisp_ReplaceStationNameByNCDXFBeaconSchedule()

//---------------------------------------------------------------------------
T_SpecDispMarker* SpecDisp_CheckCoordForClickableMarker( // .. in certain mouse event handlers
        T_SpecDispControl *pDispCtrl, // [in] struct to control the 'spectrum display'
        int iClientX, int iClientY,   // [in] client coordinate from a mouse event handler
        int iGuiControl) // [in] GUI_CONTROL_SPECTRUM / ..SPECTROGRAM / ..FREQ_SCALE / ..FREQ_INFO ?
  // Using this kludge to reduce the number of Borland VCL dependencies.
  // Each of the "GUI controls" (Spectrum, Spectrogram, Frequency Scale, Frequency Info bar)
  // is in fact an own VCL control (with a VCL-"TBitmap" for the graphics),
  // but we don't want to bother with that in this function.
{
  T_SpecDispMarker *pClickableMarker = NULL;
  int i, nMarkersToCheck = 0;

  switch( iGuiControl )
   { case GUI_CONTROL_SPECTRUM    : // looking for a "clickable marker" in the spectrum ?
     case GUI_CONTROL_SPECTROGRAM : // looking for a "clickable marker" in the spectrogram ?
     case GUI_CONTROL_FREQ_SCALE  : // looking for a "clickable marker" on the normal frequency scale ?
          pClickableMarker = pDispCtrl->marker;    // <- an array SHARED by the above GUI control elements ...
          nMarkersToCheck  = SPECDISP_NUM_MARKERS; //  .. and not put into service yet
          break;
     case GUI_CONTROL_FREQ_INFO   : // looking for a "clickable marker" in the optional "frequency info" (stations, callsigns, etc) ?
          pClickableMarker = pDispCtrl->freqInfoClickableMarker;
          nMarkersToCheck  = SPECDISP_NUM_FREQ_INFO_MARKERS;
          break;
     default:   // nothing "clickable" managed by SpecDisp.cpp itself (only use the VCL's "OnClick"-stuff for it)
          break;
   } // end switch( iGuiControl )


  if( pClickableMarker != NULL )
   { for( i=0; i<nMarkersToCheck; ++i )
      { if(  pClickableMarker->fVisible
           &&(iClientX >= pClickableMarker->x1)
           &&(iClientX <= pClickableMarker->x2)
           &&(iClientY >= pClickableMarker->y1)
           &&(iClientY <= pClickableMarker->y2) )
         { return pClickableMarker;
         }
        ++pClickableMarker;
      }
   }
  return NULL;  // whatever it was, the mouse click was NOT on any "clickable marker" !
} // end SpecDisp_CheckCoordForClickableMarker()


//---------------------------------------------------------------------------
void SpecDisp_UpdateFrequencyInfo( // Renders the "band- and frequency info" into a bitmap graphics
        Graphics::TBitmap *pbmpDest,    // [out] Borland-VCL-style "TBitmap"
        T_SpecDispControl *pDispCtrl,   // [in] struct with the displayed frequency range, colour scheme, etc
           // |-- [in]  dblUnixDateAndTimeOfFreqInfoUpdate : important to suppress
           // |         database entries that are currently NOT TRANSMITTING
           // |         (the EiBi database often lists the same station multiple times
           // |          on the same frequency, but with different times of day
           // |          and different program languages, target areas, etc)
           // '-- [out] fUpdateFrequencyInfo (cleared),
           //           fmin_on_FreqInfo, fmax_on_FreqInfo (updated here) .
        T_RigCtrlInstance *pRigCtrl,    // [in] all we know about "the rig", bands, and frequencies
        int x1, int y1, int x2, int y2) // [in] graphic area to use within the above bitmap
  // Only update the "frequency info" when really necessary !
  // For example, showing all entries in 1 MHz of bandwidth on 40 meters
  //     with hundreds of entries loaded from the EiBi database,
  //     and lazily updating the frequency-info after each update of the spectrum
  //     or spectrogram, the CPU usage from a single Remote Keyer Instance
  //     exploded from 2.8 % (with an EMPTY frequency-info-panel)
  //                to 5.3 % (with an overcrowed frequency-info-panel).
  //     Fixed by introducing the flag T_SpecDispControl.fUpdateFrequencyInfo,
  //     which is *polled* in the GUI thread, *cleared* in SpecDisp_UpdateFrequencyInfo(),
  //          and *set* whenever there is reason to believe an update 'may be necessary'
  //          - including a periodic update, every TEN SECONDS, whenever
  //            UTL_GetCurrentUnixDateAndTime_Fast() indicates a new 10-second-slot
  //            as in SpecDisp_ReplaceStationNameByNCDXFBeaconSchedule() .
{
  double freq, fmin, fmax;
  int x,y,xf2,x_end,tw,th,th2,xt,yt,iSingleCharWidth;
  int nTextLines;  // maximum number of text lines to "stack" entries for the same QRG.
                   // If nStackedEntries exceeds nTextLines,
                   // there may even be multiple COLUMNS of text for the same
                   // frequency (or frequencies resulting in the same 'x' position)
  int i, iPass, iLine, iLine2, nLinesInCurrentColumn;
  const char *pszText, *pszPrevText;
  T_FreqListEntry *pFLE, *pFLE2;
# define L_MAX_STACKED_FREQ_LIST_ENTRIES 8
  T_FreqListEntry *pStack[L_MAX_STACKED_FREQ_LIST_ENTRIES];
  int  iStackIndex, nStackedEntries; // .. with the same 'x' coordinate
  int  x_occupied[L_MAX_STACKED_FREQ_LIST_ENTRIES]; // index: TEXT LINE, 0..nStackedEntries-1
  int  nClickableMarkers = 0; // 0 .. SPECDISP_NUM_FREQ_INFO_MARKERS; index into pDispCtrl->freqInfoClickableMarker[]
  T_SpecDispMarker *pClickableMarker;

  if( ! SpecDisp_PrepareDrawing( pbmpDest, pDispCtrl, pRigCtrl ) )
   { return; // something wrong with any of the input arguments -> bail out
     // (2024-12-30 : Got here with SpecDisp_iLastErrorLine indicating
     //               trouble in SpecDisp_PrepareDrawing() with the
     //               dreadful pbmpDest->PixelFormat (!= pf32bit) .
     //  Fixed by explicitly setting "Img_TRX_Bottom->Picture->Bitmap->PixelFormat"
     //   to "pf32bit" in TKeyerMainForm::AdaptImageSizesOnTRXTab(),
     //   along with some other kludges, described only there. )
   }

  for(i=0; i<SPECDISP_NUM_FREQ_INFO_MARKERS; ++i) // forget old "clickable markers" on the frequency info panel
   { memset( &pDispCtrl->freqInfoClickableMarker[i], 0, sizeof(T_SpecDispMarker) );
   }

  pDispCtrl->fUpdateFrequencyInfo = FALSE;  // "done" (when returning from SpecDisp_UpdateFrequencyInfo() )
  pDispCtrl->fmin_on_FreqInfo = pDispCtrl->fmin_on_FreqScale; // now the "frequency info panel"...
  pDispCtrl->fmax_on_FreqInfo = pDispCtrl->fmax_on_FreqScale; // ... is synchronized with the "frequency scale" again


  pDispCtrl->Canvas->Brush->Color = (TColor)pDispCtrl->dwBackgndColor;
  pDispCtrl->Canvas->Pen->Color   = (TColor)pDispCtrl->clWindowText; // ex: dwGridColor;
  SpecDisp_FillRect( pDispCtrl, pDispCtrl->dwBackgndColor, x1, y1, x2, y2);

  // Prepare a sufficiently SMALL font for the grid,
  //  with a different background colour to give it a 'block like' appearance
  //  (Remember, if the output mode is opaque, the text background
  //   will be filled with pDispCtrl->Canvas->Brush->Color )
  pDispCtrl->Canvas->Font->Name  = "Arial";
  pDispCtrl->Canvas->Font->Size  = 8;
  pDispCtrl->Canvas->Font->Color = (TColor)pDispCtrl->dwBackgndColor; // <- compatible with the current 'colour scheme'
  pDispCtrl->Canvas->Brush->Color= (TColor)pDispCtrl->clWindowText; // (!)
  SpecDisp_GetTextExtent( pDispCtrl, "Q", &iSingleCharWidth, &th ); // -> circa
  th += 3;  // add a few pixels to separate the 'vertically stacked entries'
  nTextLines = (y2-y1-8) / th; // -> e.g. (76-8) pixels / 19 = circa 3 lines
                  //  '--> headroom for a vertical line indicating the precise frequency

  for( iPass=0; iPass<=1; ++iPass ) // pass 0 : draw the "connecting lines" into the background;
   {                                // pass 1 : draw the LABELS into the front
     fmin = pDispCtrl->fmin_on_FreqInfo;
     fmax = pDispCtrl->fmax_on_FreqInfo;
     pFLE = FreqList_GetEntryByFrequency( fmin, // get the "next higher entry" from e.g. EiBi's frequency database
                        pDispCtrl->dblUnixDateAndTimeOfFreqInfoUpdate );
     if( pFLE != NULL ) // is this entry inside the currently displayed frequency range ?
      { if( pFLE->dblFrequency_Hz > fmax )
         { pFLE = NULL; // not a single item from this database visible -> can use the "screen space" for something else
         }
      }
     memset( x_occupied, 0, sizeof(x_occupied) );
     // Show what we have from the EiBi database:
     while( ( pFLE != NULL ) && (pFLE->dblFrequency_Hz<=fmax) )
      { // .. which, by the way, seems to be what the multi-band Web SDR from Twente shows, too .
        x = SpecDisp_FreqToScreenX( pDispCtrl, pFLE->dblFrequency_Hz );
        if( (x>x1) && (x<x2) )  // graphic position also inside the frequency scale ?
         { //   '--> don't "smear" multiple entries for the same frequency over each other.

           // Before drawing, check how many database entries are on the same frequency
           //  (more precisely, would share the same horizontal position) :
           nStackedEntries = 0;
           pStack[nStackedEntries++] = pFLE;  // <pFLE> will be the top of the "stack", but how many more ?
           pszPrevText = SpecDisp_GetTextFromFreqListEntry( pFLE );
           while( (nStackedEntries < L_MAX_STACKED_FREQ_LIST_ENTRIES) // look ahead for more entries on the same horizontal position
             &&   ( (pFLE2 = FreqList_GetNextEntry( pFLE,pDispCtrl->dblUnixDateAndTimeOfFreqInfoUpdate)
                     ) != NULL ) )
            { xf2 = SpecDisp_FreqToScreenX( pDispCtrl, pFLE2->dblFrequency_Hz );
              pszText = SpecDisp_GetTextFromFreqListEntry( pFLE2 );
              // ex: if( xf2 != x )  // this placed a lot of "Russion Channel Markers" in individual columns,
                    // which almost overlapped. Thus if xf2 overlaps with the current column,
                    // also stack those extra entries even if they are on SLIGHTLY different
                    // frequencies, as sketched below:
                    //        (7.0385 MHz)
                    //    ,-----------------------------------------------------,
                    //    |       | | '-------------------------,               |
                    //    |   OK0EU Beacon D  Channel Marker D  OK0EPB Pendulum |
                    //    |   OK0EU Beacon M  Channel Marker P                  |
                    //    |   OK0EU Beacon U  Channel Marker S                  |
                    //    '-----------------------------------------------------'
                    //
              if( xf2 > (int)(x+iSingleCharWidth*strlen(pszText)/2) )
               { // Got all entries that should be squeezed into the current "vertical stack",
                 // and the next entry to process in the OUTER LOOP is pFLE again (not pFLE2).
                 break;
               }
              else
               {// Avoid dupe names for the same frequency (or horz position),
                 //   because there were thousands of dupe entries in the EiBi database
                 //   for the same frequency but different time-of-day, etc etc:
                 if( SL_strnicmp(pszText,pszPrevText,31) != 0 ) // prevent the "Voice of Korea" or "China Radio Int." from appearing MULTIPLE TIMES on the same frequency
                  {  pStack[nStackedEntries++] = pFLE2;
                     pszPrevText = pszText;
                  }
                 pFLE = pFLE2; // <- also for the OUTER LOOP; further below: pFLE = FreqList_GetNextEntry( pFLE );
               }
              // For example, THIS (inner) loop also skipped a dozen of stations all named
              // "Bangkok Meteorolog.Radio", on 3298.67651, 3298.67651, 3298.67651, etc kHz.
              // Even though they ware NOT on the same QRG, the 'x' coordinate was,
              // so all those dupes were excluded from the display. That's INTENDED.
              // After this loop, pFLE->sz31StationName was still "Bangkok Meteorolog.Radio"
              // because that's the element skipped in the OUTER LOOP by
              //   > pFLE = FreqList_GetNextEntry( pFLE ); // skip another database entry (here in the OUTER loop)
              // The next with a DIFFERENT NAME (and thus 'added to the stack')
              // was "E06 Russian Spy Numbers" on 3306 kHz in the EiBi database.
            } // end while( nStackedEntries < L_MAX_STACKED_FREQ_LIST_ENTRIES ) ..

           // Render the <nStackedEntries> as graphics into the bitmap:
           // Example (from the EiBi database, ordered by frequency) :
           // kHz           Time(UTC) Days  ITU Station                Lang. Target   Remarks
           // ================================================================================
           // 7038.5        0000-2400       CZE OK0EU Beacon D           -CW Eu          dl
           // 7038.5        0000-2400       CZE OK0EU Beacon M           -CW Eu          pv
           // 7038.5        0000-2400       CZE OK0EU Beacon U           -CW Eu          pr
           //  |   ,----------------------------|____________|
           //  |   '--> here: pFLE->sz31StationName[], because EiBi's database doesn't know CALLSIGNS
           //  '------> here: THREE "stations" on the same frequency, thus nStackedEntries = 3

           for( iStackIndex=0; iStackIndex<nStackedEntries; ++iStackIndex )
            { pFLE2 = pStack[iStackIndex];
              iLine = (iStackIndex % nTextLines);
              if( iLine==0 ) // begin of a new TEXT COLUMN ("from top to bottom") ?
               { nLinesInCurrentColumn = nStackedEntries - iStackIndex;
                 if( nLinesInCurrentColumn > nTextLines )
                  {  nLinesInCurrentColumn = nTextLines;
                  }
               }
              iLine2 = (nLinesInCurrentColumn - iLine); // index into x_occupied[iLine2]
              pszText = SpecDisp_GetTextFromFreqListEntry( pFLE2 );
              // Little gadget added 2024-12-30: The *manually edited* file
              //  contains a couple of entries with station name = "NCDXF Beacons".
              //  If those strings are encountered here, AND the frequency
              //  is indeed correct for that worldwide network of amateur radio beacons,
              //  replace the display text by the CALLSIGN of the beacon
              //  that is "theoretically" transmitting at the moment:
              if( strncmp( pszText, "NCDXF", 5) == 0 )
               { pszText = SpecDisp_ReplaceStationNameByNCDXFBeaconSchedule( pszText, (long)pFLE2->dblFrequency_Hz );
               }
              SpecDisp_GetTextExtent( pDispCtrl, pszText, &tw, &th2/*!*/ );
              xf2 = SpecDisp_FreqToScreenX( pDispCtrl, pFLE2->dblFrequency_Hz ); // position indicating the exact FREQUENCY
              xt = xf2 - tw/2; // horizontal position for the TEXT (ideally, but may have to be shifted right to avoid overlap)
              yt = y2 - iLine2 * th/*!*/;  // th > th2 !
              if( iPass==0 )
               { if( xt < x_occupied[iLine2] )
                  { // Oops.. it's getting crowded ! Avoid overlap:
                    xt = x_occupied[iLine2];
                    // Draw a diagonal line from the frequency to the center of the text
                    SpecDisp_DrawLine( pDispCtrl, xf2, y1, xt+tw/2, yt );
                  }
                 else
                  { SpecDisp_DrawLine( pDispCtrl, xf2, y1, xf2, yt ); // draw straight vertical line
                  }
               }
              if( xt < x_occupied[iLine2] )
               { xt = x_occupied[iLine2];
               }
              if( iPass==1 )
               { SpecDisp_DrawText( pDispCtrl, xt+1, yt, pszText );
                 // Store the first few entries in an array to make them "clickable":
                 if( nClickableMarkers < SPECDISP_NUM_FREQ_INFO_MARKERS )
                  { pClickableMarker = &pDispCtrl->freqInfoClickableMarker[nClickableMarkers++];
                    pClickableMarker->fVisible = TRUE;
                    pClickableMarker->x1 = xt+1;
                    pClickableMarker->y1 = yt;
                    pClickableMarker->x2 = xt+tw;
                    pClickableMarker->y2 = yt+th2;
                    pClickableMarker->iGuiControl = GUI_CONTROL_FREQ_INFO;
                    pClickableMarker->pvObject = (void*)pFLE2;
                  }
               }

              x_end = xt + tw + 4;  // "end coordinate" occupied by the text printed above, with a little separator between labels
              if( x_end > x_occupied[iLine2] )
               { x_occupied[iLine2] = x_end;
               }
            }   // end for( iStackIndex .. )
         }     // end if < x ok ? >
        pFLE = FreqList_GetNextEntry( pFLE, pDispCtrl->dblUnixDateAndTimeOfFreqInfoUpdate );
      }
   }    // end for < two "passes" >
} // end SpecDisp_UpdateFrequencyInfo()


//---------------------------------------------------------------------------
static void SpecDisp_DrawAudioSpectrumFromCwDSP( // .. into an off-screen bitmap, for the "TRX" tab
        T_SpecDispControl *pDispCtrl, // [in] layout info, config data, etc
                                      // [out] pDispCtrl->Canvas (for VCL, "the thing to draw into")
        T_RigCtrlInstance *pRigCtrl,  // [in] all we know about "the rig" and audio- vs radio-frequencies
        T_CwDSP *pCwDSP ) // [in] 'CW DSP' with the most recent audio spectra for the CW Decoder,
                          // [in] pCwDSP->fltCwDecoderSNR_dB : momentary signal-to-noise ratio,
                          // [in] pCwDSP->fltCwDecoderCenterFrequency : momentary audio peak frequency in Hertz, subject to AFC
  // This function was used to develop the CW Decoder (mostly the "Demodulator")
  // in cbproj/Remote_CW_Keyer/CwDSP.c, where different principles were tried
  // to implement a kind of 'matched filter with automatic frequency control'.
  // Most important was the AUDIO SPECTRUM, plotted from a few 'frequency bins'
  // in the CW decoder (either with an array of 'Goertzel' filters or a short-
  // time FFT, whatever works best..) .
  //
  // See also : Keyer_GUI.cpp : ScopeDisplay_DrawAudioSpectrumAndDecoderInfo()
  //            (shows a fast-running audio SPECTROGRAM in the timing scope)
  //
{
  double freq, fmax, fstep, substep;
  float flt, fltFreqToBinIdx, fltPeak, fltScalingFactor;
  int   i, n, tw, th, thh, iBinIdx, x,y, prev_x, prev_y, x_covered;
  TPoint poly_points[CWDSP_AUDIO_SPECTRUM_NUM_FREQUENCY_BINS+4];
         // "plus 4" to join the last and first point at the BASE LINE.
  int    nPoints = 0;  // number of polygon points actually used in poly_points[]
  char   szLabel[44], sz16FormatString[16+4];
  T_SpecDispMarker *pMarker;
  TColor oldTextColor = pDispCtrl->Canvas->Font->Color;

  pDispCtrl->i32AudioSpectrumUpdateCounter = pCwDSP->i32AudioSpectrumUpdateCounter;

  for( iBinIdx=0; iBinIdx<CWDSP_AUDIO_SPECTRUM_NUM_FREQUENCY_BINS; ++iBinIdx )
   {
     freq = (float)iBinIdx * pCwDSP->fltAudioSpectrumBinWidth_Hz;
     // At this point, 'freq' is an "audio" frequency. But for SpecDisp_FreqToScreenX(),
     //      we need the RADIO FREQUENCY. The relation between them is not
     //      as easy as you may guess (it's not just "add the VFO frequency",
     //      consider operating modes like "LSB", "CW Reverse", etc).
     // Only the "RigControl" module knows about all these annoying rig-specific
     // special cases, e.g. "dial frequency" versus "VFO frequency", "CW pitch",
     // so:
     freq = RigCtrl_AudioToRadioFrequency( pRigCtrl, freq );

     x = SpecDisp_FreqToScreenX( pDispCtrl, freq );
     if( (x>=0) && (x<pDispCtrl->iCanvasWidth) ) // inside the "plotting area" ?
      {
        flt  = pCwDSP->fltAudioPowerSpectrum[iBinIdx];  // power densities, normalized to 0.0 .. 1.0 (not complex, not logarithmized)
        flt  = SpecDisp_Power_to_dBfs( flt/* fltNormalizedPower, 0..1 */ );
        // '---> the interesting range for the audio CW decoder is only -50 to 0 dB "over" full scale .
        // Convert the 'dB' value into a vertical pixel offset.
        // This doesn't have anything to do with the vertical scaling of Icom's
        // "scope data" anyway (the audio amplitude depends on the AGC,
        //                      while the "scope data" should not),
        // so keep it simple:  With 8 bits per audio sample over the network,
        // the dynamic range is only ~50 dB, so the 'canvas height' represents 50 dB:
        flt = (float)pDispCtrl->iCanvasHeight * (flt + 50.0f/*dB*/) / 50.0f;
        LimitFloat( &flt, 0.0f, pDispCtrl->iCanvasHeight-1 );
        y = pDispCtrl->iCanvasHeight - 1 - (int)flt;
        poly_points[nPoints].x   = x;  // append another point to the polyline
        poly_points[nPoints++].y = y;
      }
   }  // end for < all available "audio frequency bins" >


  if( nPoints >= 2 ) // draw the entire "curve" in a single GDI function call
   { pDispCtrl->Canvas->Pen->Color = (TColor)pDispCtrl->dwAuxCurveColor;
     pDispCtrl->Canvas->Brush->Color = (TColor)pDispCtrl->dwAuxBkgndColor;
#   if(0) // Polyline or filled polygon ?
     pDispCtrl->Canvas->Polyline( poly_points, nPoints-1 );
     // ,--------------------------------------|_______|
     // '--> funny but true: Not the NUMBER OF POINTS,
     //                      but the NUMBER OF POINTS MINUS ONE !
#   else // not just a "polyline" but a FILLED polygon :
     pDispCtrl->Canvas->Brush->Color = (TColor)pDispCtrl->dwAuxBkgndColor;
     poly_points[nPoints].x   = poly_points[nPoints-1].x;
     poly_points[nPoints++].y = pDispCtrl->iCanvasHeight-1;
     poly_points[nPoints].x   = poly_points[0].x;
     poly_points[nPoints++].y = pDispCtrl->iCanvasHeight-1;
     SpecDisp_FillPolygon( pDispCtrl, poly_points, nPoints-1/*!*/ );
#   endif // just a "polyline" or a FILLED polygon ?
   }

  // Also show the CW decoder's measured audio center frequency and SNR,
  //   as a little 'ball swimming in the waves' of the spectrum display:
  freq = RigCtrl_AudioToRadioFrequency( pRigCtrl, pCwDSP->AudioCwDecoder[0].fltCwDecoderCenterFrequency );
  x = SpecDisp_FreqToScreenX( pDispCtrl, freq );
  if( (x>=0) && (x<pDispCtrl->iCanvasWidth) ) // inside the "plotting area" ?
   { flt = SpecDisp_Power_to_dBfs( pCwDSP->AudioCwDecoder[0].fltCwDecoderSignalPower );
     sprintf(szLabel, "%.1f dB", (float)flt );
     flt = (float)pDispCtrl->iCanvasHeight * (flt + 50.0f) / 50.0f;
     LimitFloat( &flt, 0.0f, pDispCtrl->iCanvasHeight-1 );
     y = pDispCtrl->iCanvasHeight - 1 - (int)flt;
     pDispCtrl->Canvas->Brush->Color = (TColor)pDispCtrl->dwAuxBkgndColor;
     SpecDisp_DrawCircle( pDispCtrl, x,y, 5/*radius in pixels*/ );
     SpecDisp_GetTextExtent( pDispCtrl, szLabel, &tw, &th );
     if( y > (th+5) )  // text ABOVE or BELOW the "swimming ball" ?
      {  y -= (th+5);  // ABOVE (preferred) [note that y=0 is on top, thus subtracting moves the text UP on the screen]
      }
     else  // text BELOW the "swimming ball", so add the radius (but not the text height)
      {  y += 5;
      }
     x -= tw/2;
     pDispCtrl->Canvas->Font->Color  = (TColor)pDispCtrl->dwAuxCurveColor;
     pDispCtrl->Canvas->Brush->Color = (TColor)pDispCtrl->dwBackgndColor;
     SpecDisp_DrawText( pDispCtrl, x,y, szLabel );
   }
  pDispCtrl->Canvas->Font->Color = oldTextColor;


} // end SpecDisp_DrawAudioSpectrumFromCwDSP()

//---------------------------------------------------------------------------
BOOL SpecDisp_HandleMouseEvent(  // processes most(*) mouse-related event in the spectrum, spectrogram, or frequency scale
        T_SpecDispControl *pDispCtrl, // [in] layout info, config data, etc
        T_RigCtrlInstance *pRigCtrl,  // [in,out] "Rig Control" instance
        int iGuiEvent,                // [in] GUI_EVENT_MOUSE_DOWN, GUI_EVENT_MOUSE_MOVE, GUI_EVENT_MOUSE_UP
        int iGuiControlIndex,         // [in] GUI_CONTROL_SPECTRUM, GUI_CONTROL_SPECTROGRAM,
                                      //      GUI_CONTROL_FREQ_SCALE, GUI_CONTROL_FREQ_INFO, .. (?)
        int iButtonsAndShiftKeys,     // [in] GUI_KEY_FLAGS_NONE, GUI_KEY_FLAGS_LEFT_MOUSE_BUTTON, etc etc (bitwise combined)
        int iClientX, int iClientY)   // [in] client coordinate of the mouse pointer (0,0 = upper left pixel of the visible control)
  // (*) If a mouse event isn't processed here, the function returns FALSE,
  //     and *the caller* may decide what to do - with a lot of VCL dependencies,
  //     for example:
  // * Open a context menu with e.g. "things to do with the clicked frequency",
  // * Open a info popup with more info about the "radio station name" that was clicked,
  // * etc.
  //     SpecDisp_HandleMouseEvent() itself will not invoke any obscure
  //     VCL handler / C++ method.
{
  // Common treatment for ALL THREE PARTS of the screen, and ALL THREE MOUSE-RELATED EVENTS:
  double freq  = ScreenXToFreq( pDispCtrl, iClientX );
  double span;
  int    x_vfo = SpecDisp_FreqToScreenX( pDispCtrl, pRigCtrl->dblVfoFrequency );
  BOOL   fMouseNearVFO = (iClientX >= (x_vfo-5) ) && (iClientX <= (x_vfo+5) );
  BOOL   fHandledEvent = TRUE;
  BOOL   fVisible;

  if( pDispCtrl->iCanvasWidth <= 0 ) // avoid div-by-zero anywhere below
   { return FALSE;
   }

  // Specific treatment for mouse down/move/up :
  switch( iGuiEvent )
   { case GUI_EVENT_MOUSE_DOWN:
        // Stuff used in SpecDisp_HandleMouseEvent() :
        pDispCtrl->dblClickedVfoFreq_Hz = freq; // remember this for later (e.g. ..MOUSE_UP)
        if( fMouseNearVFO )
         { pDispCtrl->iVFOEditTimer_ms = -1; // avoid interference from the edit field
           pDispCtrl->fDraggingVfoFrequency = TRUE;
           pDispCtrl->iDraggingOffsetX = iClientX - x_vfo;
           pDispCtrl->fmin_at_drag_start = pDispCtrl->fmin_from_RigCtrl;
           pDispCtrl->fmax_at_drag_start = pDispCtrl->fmax_from_RigCtrl;
           // Similar as when typing a new frequency into edit field Ed_VFO
           // (and setting pDispCtrl->iVFOEditTimer_ms > 0),
           // fDraggingVfoFrequency = TRUE also temporarily cuts the connection
           // between 'RigControl' and the frequency displayed on the GUI.
           // In both cases, the background colour of the Ed_VFO (display field)
           // indicates "the new frequency may not have arrived
           //             on the remotely controlled radio yet".
         }
        else
         { pDispCtrl->fDraggingVfoFrequency = FALSE;
         }
        break;

     case GUI_EVENT_MOUSE_MOVE:
        fVisible = ( fMouseNearVFO || pDispCtrl->fDraggingVfoFrequency );
        if( pDispCtrl->marker[SPECDISP_MARKER_VFO].fVisible != fVisible )
         { pDispCtrl->marker[SPECDISP_MARKER_VFO].fVisible = fVisible;
           pDispCtrl->fUpdateSpectrum = TRUE;
         }
        if( pDispCtrl->fDraggingVfoFrequency )
         { // Similar to freq = ScreenXToFreq( pDispCtrl, iClientX );
           // but here using pDispCtrl->fmin_at_drag_start, fmax_at_drag_start
           //     instead of pDispCtrl->fmin_from_RigCtrl,  fmin_from_RigCtrl:
           span = pDispCtrl->fmax_at_drag_start - pDispCtrl->fmin_at_drag_start;
           if( span > 0.0 )
            { freq = pDispCtrl->fmin_at_drag_start
                      + ((double)(iClientX-pDispCtrl->iDraggingOffsetX) * span)
                       / (double)pDispCtrl->iCanvasWidth;
              pDispCtrl->iVFOEditTimer_ms = -1; // avoid interference from the edit field (user isn't TYPING there anymore)
              SpecDisp_UpdateVFODisplay(freq);  // -> Ed_VFO (a visible control element in the GUI)
              RigCtrl_SetVFOFrequency( pRigCtrl, freq );
            }
         }
        break;
     case GUI_EVENT_MOUSE_UP  :
        if( pDispCtrl->fDraggingVfoFrequency || pDispCtrl->marker[SPECDISP_MARKER_VFO].fVisible )
         { pDispCtrl->fUpdateSpectrum = TRUE;
         }
        if( pDispCtrl->fDraggingVfoFrequency )
         {  pDispCtrl->iVFOEditTimer_ms = -1; // edit field (Ed_VFO) back in default state
            // (may report a new frequency from RigControl now)
         }
        else // "mouse-up" event, but not after dragging the VFO frequency ->
         { fHandledEvent = FALSE; // let e.g. Keyer_Main.cpp handle this event, and open a context menu
         }
        pDispCtrl->fDraggingVfoFrequency = FALSE;
        pDispCtrl->marker[SPECDISP_MARKER_VFO].fVisible = FALSE;
        break;
     default:
        break;
   }


  return fHandledEvent;


} // end SpecDisp_HandleMouseEvent()

