//---------------------------------------------------------------------------
// File  :  C:\cbproj\Remote_CW_Keyer\SampleRateConv.c
// Date  :  2023-12
// Author:  Wolfgang Buescher  (DL4YHF)
//
// Description:
//     Sample Rate conversions for the 'Remote CW Keyer'.
//     Mainly used to convert audio sample streams from 48 kSamples/second
//      (as used by e.g. 'DirectSound' audio devices)
//      to 8 kSamples/seconds (as used 'over the network') and back.
//     This includes proper low-pass filtering (to decimate or interpolate
//                                              by 3 and by 2).
//
// Revision history (yyyy-mm-dd):
//   2023-12:  Created this module, based on various modules 'recycled'
//             from Spectrum Lab .
//
//---------------------------------------------------------------------------


#include "switches.h"  // project specific compiler switches ("options"),
                       // must be included before anything else !
#include "yhf_type.h"  // classic types like BYTE, WORD, DWORD, BOOL, ..
#include <math.h>

#pragma hdrstop   // no precompiled headers after this point


#include "SampleRateConv.h"

//---------------------------------------------------------------------------
// Options for software tests, optionally defined in SWITCHES.H :
//    If NOT defined in the project-specific switches.h, define them here:
#ifndef  SWI_SRCONV_SKIP_DONT_DECIMATE
   // Let SampleRateConv.c simply "skip samples" in the input
   //     instead of properly DECIMATING them ?
   //  1 = yes (TEST, sounds ugly),  0 = no (normal compilation)
# define SWI_SRCONV_SKIP_DONT_DECIMATE 0
#endif // ndef SWI_SRCONV_SKIP_DONT_DECIMATE ?
#ifndef  SWI_SRCONV_DUPLICATE_DONT_INTERPOLATE
   // Let SampleRateConv.c simply "duplicate samples" in the input
   //     instead of properly INTERPOLATE them, using low-pass filters ?
   // 1 = yes (TEST, sounds ugly),  0 = no (normal compilation)
# define SWI_SRCONV_DUPLICATE_DONT_INTERPOLATE 0
#endif // ndef SWI_SRCONV_DUPLICATE_DONT_INTERPOLATE ?

//---------------------------------------------------------------------------
// Constant tables with filter coefficients, etc
//---------------------------------------------------------------------------


const float Dec2FilterCoeffs[DEC_FIR_LENGTH] =
{ // Reasonably short 'decimate by 2' - filter (FIR)
  // Design method: Parks-McClellan
  //  25 coefficients,
  //  2 bands,
  // Band Lower Upper Value Weight
  //  1  0.000 0.120 1.000  001.0
  //  2  0.250 0.500 0.0001 100.0
    -0.001086881568768   ,
    -0.00367797341826391 ,
    -0.0047920411648502  ,
     0.00129370756629773 ,
     0.0145245113499293  ,
     0.0211455982811658  ,
     0.00276341141492466 ,
    -0.0369726519862278  ,
    -0.0567732234406836  ,
    -0.0041615169748691  ,
     0.128367871654144   ,
     0.275162804208724   ,
     0.339561154995869   ,
     0.275162804208724   ,
     0.128367871654144   ,
    -0.0041615169748691  ,
    -0.0567732234406836  ,
    -0.0369726519862278  ,
     0.00276341141492466 ,
     0.0211455982811658  ,
     0.0145245113499293  ,
     0.00129370756629773 ,
    -0.0047920411648502  ,
    -0.00367797341826391 ,
    -0.001086881568768
}; // end Dec2FilterCoeffs[]

const float Dec3FilterCoeffs[DEC_FIR_LENGTH] =
{ // Reasonably short 'decimate by 3' - filter (FIR)
  // Design method: Parks-McClellan, 25 coefficients, 2 bands .
    -0.000286440582680371,
    -0.00188366938636472 ,
    -0.00506620307896015 ,
    -0.00974817248307781 ,
    -0.0139407470519893  ,
    -0.0139210800449785  ,
    -0.00502634467052337 ,
     0.0163468116202039  ,
     0.0504694707843042  ,
     0.0928642782132562  ,
     0.134771696647059   ,
     0.16568873828291    ,
     0.177106618781497   ,
     0.16568873828291    ,
     0.134771696647059   ,
     0.0928642782132562  ,
     0.0504694707843042  ,
     0.0163468116202039  ,
    -0.00502634467052337 ,
    -0.0139210800449785  ,
    -0.0139407470519893  ,
    -0.00974817248307781 ,
    -0.00506620307896015 ,
    -0.00188366938636472 ,
    -0.000286440582680371
}; // end Dec3FilterCoeffs[]



//---------------------------------------------------------------------------
void SRConv_InitDecimator( T_SRCONV_DECIMATOR *pDecimator,
                                  int decimation_ratio )
  // Must be called for each of the stages later used in
  //  SRConv_TwoStageDecimator() or SRConv_TwoStageInterpolator() .
  //  The same data type, T_SRCONV_DECIMATOR, is used for both .
{
  int i;
  // Initialize all decimator stages, clear the circular delay line.
  for(i=0;i<DEC_FIR_LENGTH;++i)
   {
     pDecimator->queue[i] = 0.0; // also good for COMPLEX processing !
   }
  pDecimator->ratio = decimation_ratio;
  pDecimator->count = 0;
  pDecimator->inptr = pDecimator->queue;
  switch(decimation_ratio)
   { case 3:
        pDecimator->coeffs = Dec3FilterCoeffs;
        break;
     case 2:
     default:
        pDecimator->coeffs= Dec2FilterCoeffs;
        break;
   }

} // end SRConv_InitDecimator()


//---------------------------------------------------------------------------
int SRConv_TwoStageDecimator(   // e.g. decimate 48 kSamples/sec to 8 .
      T_SRCONV_DECIMATOR *pTwoDecimators, // [in,out] array with TWO 'decimators',
                            // the first decimating by 3, the 2nd by 2.
      float *pfltInput,     // [in] values sampled at 48 kS/sec
      int   nSamplesIn,     // [in] number of samples in pfltInput
      float *pfltOutput )   // [out] values sampled at 8 kS/sec
   // Return value: Number of samples placed in pfltOutput .
{
  int nSamplesEmitted = 0;
  float acc;
  BOOL filter_output_ready;
  int  stage;
  T_SRCONV_DECIMATOR *pDecimator;

#if( SWI_SRCONV_SKIP_DONT_DECIMATE )
  while( (nSamplesIn--) > 0 )
   { acc = *pfltInput++;
     filter_output_ready = TRUE;
     for(stage=0; (stage<2) && filter_output_ready; ++stage )
      {
        pDecimator = pTwoDecimators+stage; // address of the "current" decimator
        if( (++pDecimator->count) >= pDecimator->ratio)
         { // if we would properly decimate, THIS STAGE would run the decimation filter now,
           // and emit another sample for the NEXT stage (see original code further below)
           pDecimator->count = 0;
         }
        else
         { // Do not emit a sample from this stage to any following state (or to the output):
           filter_output_ready = FALSE; // no output from the filter this time
         }
      } // end for <all "sample-skipping stages">
     if (filter_output_ready)
      { // "decimated" sample would be complete (here: only another "non-skipped" sample)
        // Emit the value into the destination block..
        pfltOutput[nSamplesEmitted++] = acc;
      }
   } // end while( (nSamplesIn--) > 0 )
#else // SWI_SRCONV_SKIP_DONT_DECIMATE==0 -> implement a "proper DECIMATOR" :
  float *firptr;
  const float *kptr; // pointer to FIR-filter coefficients
  int   i, j, qlen;  // Length of filter queue + number of coeffs
  float *inptr; // local copy for better speed.. "input pointer" ....
  float *qptr;  //                               "queue pointer" (for the FIR filter)

  while( (nSamplesIn--) > 0 )
   { acc = *pfltInput++;
     // Run the sample (acc) through a chain of decimation stages
     //     with decent low-pass filtering to avoid aliasing .
     // Principle (for decimate-by-SIX) : Cascaded decimating filters..
     //  The 1st filter runs at f_sample, and cuts off everything
     //         above f_sample/6.
     //     The 2nd filter runs at f_sample/3, so the first (FIR!-)filter
     //         only needs to calculate every 3rd output value.
     //         Every (2nd+1)-th sample is only shifted through the
     //         filter's chain but no MAC's are needed then ...
     //     and so on.
     //  For a simple halfband filter, we don't need a long FIR kernel
     //      (here: SoundTab_DEC_FIR_LENGTH = 25 ) .
     //  If the necessary lowpass-filtering would be performed in a
     //  SINGLE FIR-stage, a very long kernel would be required
     //  for large decimation ratios.
     //
     //   Here: Only decimation but no frequency shifting .
     filter_output_ready = TRUE;
     for(stage=0; (stage<2) && filter_output_ready; ++stage )
      {
        pDecimator = pTwoDecimators+stage; // address of the "current" decimator
        inptr = pDecimator->inptr;    // local copy for speed: pointer to current element in circular queue
        qptr  = pDecimator->queue;    //   "     "   "    "  : pointer to the queue itself
        qlen  = DEC_FIR_LENGTH;       // queue length, aka 'number of taps', for example 25 (see ... )
        if(--inptr < qptr)            // deal with FIR pointer wrap around
         { inptr = qptr+qlen-1;
         }
        *inptr = acc;  // place in circular Queue (here only a 'real', not 'complex', sample)
        if( (++pDecimator->count) >= pDecimator->ratio)
         { // calculate the decimation filter now (example decimate by two: only get here for every 2nd sample)
           pDecimator->count = 0;
           // Prepare decimation (by 2 or 3).
           kptr = pDecimator->coeffs; // typically points to SoundTab_Dec2FilterCoeffs[SoundTab_DEC_FIR_LENGTH]
           acc  = 0.0;
           firptr = inptr;
           for(j=0; j<qlen; ++j )     // do the MACs (Multiply and Accumulate)
            {
              acc += (*firptr) * (*kptr++);
              if( (++firptr) >= qptr+qlen ) //deal with wraparound
               { firptr  =  qptr;
               }
            }
           // filter output for the next stage now in acc.re .
         } // end if ( ..m_Decimator[].count >= .ratio )
        else
         { // Do not CALCULATE the decimation filter now (only place
           // the sample in its queue, which already happened further above),
           // because the result would be thrown away anyway .. save a couple of MAC's .
           // (example decimate by two: only get here for every 2nd sample)
           filter_output_ready = FALSE; // no output from the filter this time
         }
        // save position in circular delay line
        pDecimator->inptr = inptr;
      } // end for <all lowpass & decimation stages>
     if (filter_output_ready)
      { // "decimated" sample is complete.
        // Emit the value into the destination block..
        pfltOutput[nSamplesEmitted++] = acc;
      }
   } // end while <more input samples to process>
#endif // SWI_SRCONV_SKIP_DONT_DECIMATE ?
  return nSamplesEmitted;
} // end SRConv_TwoStageDecimator()


//---------------------------------------------------------------------------
int SRConv_RunSingleUpsamplingStage(
     T_SRCONV_DECIMATOR *pUpsampler, // "decimator abused as interpolator/upsampler",
                                     // includes pointer to FIR coeffs and queue
          float fltInputSample,      // input: a single sample to be processed
          float *pfltOutputSamples ) // output: two or three samples,
                                     // depending on pUpsampler->ratio .
  // Returns THE NUMBER OF upsampled/interpolated samples in pfltOutputSamples.
  // Note that in contrast to a decimator, the "upsampler" always emits
  // the same number of OUTPUT samples for a single input sample,
  // because the input sampling rate is MULTIPLED, not DIVIDED, by an integer ratio.
{
#if(1) // (1)=normal compilation, (0)=TEST with just 'sample duplication'
  // Use local copies for speed...
  const float *coeffs;               // pointer to a table of coefficients
  float *inptr = pUpsampler->inptr;  // input position in circular delay line
  float *qptr  = pUpsampler->queue;  // first element of circular delay line
  float *firptr;
  int   qlen   = DEC_FIR_LENGTH;
  int   iLoop, iRatio = pUpsampler->ratio;
  int   j;

  if(iRatio>3)
   { iRatio = 3; // '3' is the maximum output/input ratio we can handle (with a single stage)
   }
  if(iRatio <=1)
   {  // This stage looks like a "pass-through" stage. Not much to do:
     *pfltOutputSamples = fltInputSample;
     // ex: pUpsampler->count = 1;
     return 1;
   }
  else
  for( iLoop=0; iLoop<iRatio; ++iLoop) // e.g. to upsample by THREE, run the same sample through the lowpass THREE TIMES,
   { // generating THREE DIFFERENT (interpolated) output sample from ONE input sample.
     float acc; // "accumulator" for output from digital filter
     // (Finite Impulse Response lowpass prepared by SRConv_InitDecimator() )

     inptr--;
     if(inptr < qptr)          // deal with FIR pointer wrap around
        inptr = qptr+qlen-1;   // (2 indices per COMPLEX sample!)
     *inptr = fltInputSample;  // place sample in circular Queue (two or three times THE SAME sample)

     // Run interpolating filter (an ordinary FIR lowpass, 1/2 or 1/3 band)
     //      Prepare complex MAC's for FIR lowpass
     coeffs = pUpsampler->coeffs; // pointer to filter coeffs
     acc    = 0.0;                // clear output accumulator
     firptr = inptr;
     for(j=0; j<qlen; ++j )   // do the MACs (Multiply and Accumulate)
      {
        acc += (*firptr) * (*coeffs++);
        if( (++firptr) >= qptr+qlen ) // deal with wraparound
               firptr  =  qptr;
      }

     // The filtered output for the next interpolator stage is now in acc .
     // Store it in *pfltOutputSamples .
     // For ratio=2 or 3,
     //              the 1st interpolated sample is in pfltOutputSamples[0],
     //              the 2nd interpolated sample is in pfltOutputSamples[1].
     // For ratio=3, the 3rd interpolated sample is in pfltOutputSamples[2].
     pfltOutputSamples[ iLoop ] = acc;

   } // end for(iUpsLoop=0; ... )

  // save position in circular delay line
  pUpsampler->inptr = inptr;
  pUpsampler->count = iRatio; // 'pfltOutputSamples[]' now filled with 2 or 3 samples
#else // ABOVE: normal compilation,  BELOW: TEST with just 'sample duplication'
  int   i, iRatio = pUpsampler->ratio;
  for(i=0; i<iRatio; ++i)
   { pfltOutputSamples[i] = fltInputSample;  // only 'duplicate', don't 'interpolate'
   }
#endif // test or normal compilation ?
  return iRatio;  // returns the number of samples placed in pfltOutputSamples[]
} // end SRConv_RunSingleUpsamplingStage()


//---------------------------------------------------------------------------
int SRConv_TwoStageUpsampler(   // e.g. upsample (interpolate) fro 8 to 48 kSamples/sec .
      T_SRCONV_DECIMATOR *pTwoInterpolators, // [in,out] array with TWO 'decimators',
                            // (here used as interpolators),
                            // the first interpolating by 3, the 2nd by 2.
      float *pfltInput,     // [in] values sampled at e.g. 8 kS/sec
      int   nSamplesIn,     // [in] number of samples in pfltInput
      float *pfltOutput )   // [out] values sampled at e.g. 48 kS/sec .
             // This array must have a capacity of at least
             // nSamplesIn * pTwoInterpolators[0].ratio * pTwoInterpolators[1].ratio !
   // Return value: Number of samples placed in pfltOutput .
{
  float fltSamplesFromFirstState[4];
  int   iSample, nSamples, nSamplesEmitted = 0;
  while( (nSamplesIn--) > 0 )
   { // Run the next INPUT sample through the chain of interpolator stages,
     // each of them with decent low-pass filtering to avoid aliasing .
     nSamples = SRConv_RunSingleUpsamplingStage(
          &pTwoInterpolators[0], *pfltInput++, fltSamplesFromFirstState );
     for(iSample=0; iSample<nSamples; ++iSample )
      { nSamplesEmitted += SRConv_RunSingleUpsamplingStage(
          &pTwoInterpolators[1],
          fltSamplesFromFirstState[iSample], // [in]  sample from 1st stage
          pfltOutput+nSamplesEmitted );      // [out] sample from 2nd stage
      }
   } // end while <more input samples to process>
  return nSamplesEmitted;
} // end SRConv_TwoStageUpsampler()


/* EOF <SampleRateConv.cpp> */

