//---------------------------------------------------------------------------
// File: C:\cbproj\Remote_CW_Keyer\RigControl_CIV_Server.c
// Authors: Wolfgang Buescher (DL4YHF)
//       (the stripped-down variant for DL4YHF's "Remote CW Keyer" [RCW]
//        does NOT support the ugly and hypercomplicated "RS-BA1 protocol") .
// Date:   2025-07-13
// Purpose:
//  Addition to RigControl.c . Allows an application to mimick an Icom-radio
//  on an "Additional COM Port", to allow THIRD PARTY APPLICATIONS to control
//  the 'real' radio (that is controlled by RigControl's CIV *CLIENT*, with
//  core functions like RigCtrl_ParseCIV() implemented in RigControl.c).
//
// Principle and interaction with RigControl.c, CwNet.c, and 3rd party apps
// -------------------------------------------------------------------------
//
//  _______                ____________               _______
// | 3rd   |   com0com    | Remote    |  RS-232, USB | Real  |
// | party |<------------>|   CW      |<------------>| Rig   |
// | app   |         COMx |  Keyer    |              |(Icom) |
// |_______|              |___________|              |_______|
//    :  \________  _______/:       :\_______  _______/   :
//    :           \/        : Data- :        \/           :
//    :   "Aux COM"-port    : base  :   "RigControl"      :
//    :   and worker thread :       :   RADIO PORT        :
//    :                     :       :  and worker thread  :
//    : COMMAND (Get,Set)   :       :                     :
//    :-------------------->: _ _ _ : COMMAND (Get,Set)   :
//    :                     : (1)   :-------------------->: (2a)
//    :                     :       :                     :  (radio working...)
//    :    (waiting....)    :       :  RESPONSE           :
//    :                     : _ _ _ :<--------------------: (2b)
//    :                     :(store):                     :
//    :  RESPONSE (+data?)  :       :                     :
//    :<--------------------: (3)   :                     :
//
//  Ideal case: the to-be-read parameter is already in memory (Rig Control "database"),
//              so the "AuxCOM"-port-worker-thread doesn't need to wait
//              for anything:
//    : COMMAND (Get)       :       :                     :
//    :-------------------->: Read  :                     :
//    :                     : from  :                     :
//    :                     : data- :                     : (nothing to do here)
//    :                     : base  :                     :
//    :  RESPONSE (+data)   :  (4)  :                     :
//    :<--------------------:       :                     :
//
//  Similar for WRITE-COMMANDS (received on an "Aux COM" port):
//     If the to-be-SET parameter's value, e.g. VFO frequency,
//     equals the database entry (e.g. addressed by RIGCTRL_PN_FREQUENCY),
//     the "AuxCOM"-port-worker-thread shall immediately respond with an
//     "OK"-response. Nothing to do on the 'real' RADIO CONTROL (right side).
//
//
// (1) If the SERVER SIDE (left) cannot immediately send a response
//     because the command most be forwarded to the REAL RADIO first
//     (possibly using a different command, and different 'addresses'),
//     iServerState switches from RIGCTRL_SSTATE_PARSING_COMMAND
//                             to RIGCTRL_SSTATE_WAIT_RADIO_NBUSY  .
//      [that happens inside RigCtrl_ParseCIV(), which 'knows command-specific
//       details' in a MONSTEROUS switch/case list for dozens of CI-V commands
//       and hundreds of sub-commands. Grep for 'pPort->fActAsServer' ... ]
// (2a) As soon as the RADIO CONTROL PORT isn't busy, "forward" the command
//      on behalf of the remote client to the real radio (and SEND it) ->
//        iServerState switches from RIGCTRL_SSTATE_WAIT_RADIO_NBUSY
//                                to RIGCTRL_SSTATE_FWD2RADIO_BUSY .
//        [supposed to happen in RigCtrl_Handler() after checking RigCtrl_IsPortBusy()]
// (2b) When the RESPONSE FROM THE REAL RADIO arrives (read-response with data,
//                  or write-response with OK / NOT_OK) ->
//     iServerState switches from RIGCTRL_SSTATE_FWD2RADIO_BUSY to RIGCTRL_SSTATE_FWD2RADIO_DONE .
//     Note that THESE transitions (2a and 2b) doesn't happen in the Aux-COM-Port's worker thread
//     but in whatever thread "communicates with the radio" (maybe even via CwNet.c)
// (3) When the Aux-COM-Port's worker thread notices RIGCTRL_SSTATE_FWD2RADIO_DONE,
//     it "finishes" the response message (including the READ DATA, which may
//     even be a verbatim copy of the read-response FROM THE RADIO, if both
//     [Aux COM port and the real radio] use the same protocol).
//     The transition from RIGCTRL_SSTATE_FWD2RADIO_DONE to RIGCTRL_SSTATE_RESPONSE_READY
//     was supposed to happen in
//
// (4) If RigCtrl_ParseCIV(), called on the SERVER side, can immediately
//     read the requested value from e.g. pRC->dblVfoFrequency (etc^10),
//     it fills out a complete response in pServerPortInstance already,
//     which the Aux-COM-Worker-Thread can send IMMEDIATELY, without waiting
//     for anything (unfortunately, this is only the "ideal" situation).
//
// SERVER STATE DIAGRAM for a port instance that acts 'as SERVER' (iServerState, with fActAsServer=TRUE),
// with the same labels in parentheses as used above for the COMMAND / RESPONSE sequence :
//----------------------------------------------------------------------------,
//                                                                            .
//                             Start                                          .
//                               |                                            .
//                              \|/                                           .
//                    ,--------------------------------,                      .
//  ,---------------> | RIGCTRL_SSTATE_PASSIVE         |                      .
//  |                 '--------------------------------'                      .
//  |                            |                                            .
//  |                            | In RigControl.c : RigCtrl_ProcessData(),   .
//  |                            | shortly before calling RigCtrl_ParseCIV(): .
//  |                            |    RigCtrl_CIV_Server_PrepareParsing()     .
//  |                           \|/                                           .
//  |                 ,--------------------------------,                      .
//  |                 | RIGCTRL_SSTATE_PARSING_COMMAND |                      .
//  |                 '--------------------------------'                      .
//  |                            |                                            .
//  |               ,------------------------------------------------------,  .
//  |              /  Can RigCtrl_ParseCIV() fill out a response already ?  \ .
//  |              \           Yes                       No                 / .
//  |               '------------------------------------------------------'  .
//  |                            |(4)                    |(1)                 .
//  |                            |                       |                    .
//  |                           \|/                     \|/                   .
//  |     ,-------------------------------, ,-------------------------------, .
//  |     | RIGCTRL_SSTATE_RESPONSE_READY | |RIGCTRL_SSTATE_WAIT_RADIO_NBUSY| .
//  |     '-------------------------------' '-------------------------------' .
//  |                   |                               | when NOT BUSY,      .
//  |                   |                               | Radio-Port-Thread   .
//  |                   |                               | starts 'forwarding' .
//  |                   |                               |     (2a)            .
//  |                   |                              \|/                    .
//  |                   |                   ,-------------------------------, .
//  |                   |                   | RIGCTRL_SSTATE_FWD2RADIO_BUSY | .
//  |                   |                   '-------------------------------' .
//  |                   |                               | Radio-Port-Thread   .
//  |                   |                               | got the response    .
//  |                   |                               |     (2b)            .
//  |                   |                              \|/                    .
//  |                   |                   ,-------------------------------, .
//  |                   |                   | RIGCTRL_SSTATE_FWD2RADIO_DONE | .
//  |                   |                   '-------------------------------' .
//  |                  \|/                              | Aux-COM-Thread      .
//  |              Aux-COM-Thread                       | notices this (3)    .
//  '---------<--- sends ITS resonse,..  <--------------'                     .
//              [ using RigCtrl_SendServerResponse(?) ]                       .
//                                                                            .
//                                                                            .
//----------------------------------------------------------------------------'
//
//  Notes (originally just a 'collection of ideas'):
//   * The "unified parameter number" alone (e.g. RIGCTRL_PN_FREQUENCY)
//     doesn't provide enough information to assemble a RESPONSE
//     on the AUX COM PORT (left side), because:
//       - in Icom's CI-V protcol for 'modern' radios, there are multiple
//         commands / sub-commands to read / set e.g. the VFO frequency.
//       - the READ RESPONSE not only contains the read-out VALUE,
//         but also the COMMAND, SUB-COMMAND, SUB-SUB-COMMAND,
//         and who-know-what; so we need a universal way to store
//         "everything we need to know LATER, for the response"
//         already in the monsterous CI-V parser ( RigCtrl_ParseCIV() ).
//       - RigCtrl_ParseCIV() MUST NEVER BLOCK the caller, so if it cannot
//         immediately fill out a complete response, it will prepare a
//         "template" in T_RigCtrlInstance.PortInstance[x].sTxMsg,
//         and remain in RIGCTRL_SSTATE_WAIT_RADIO_NBUSY, until the worker thread
//         for the REAL RADIO PORT switches it to RIGCTRL_SSTATE_FWD2RADIO_DONE.
//         Shortly before that, the "original read-response" from the RADIO
//         is stored in pServerPortInstance->sRxMsg[1], as documented in
//         RigControl.c : T_RigCtrl_PortInstance.sRxMsg[] !
// - Example (seen 2025-07-27 when WSJT-X started BANGING on the door of a SERVER PORT):
//       0006695  RigControl: Init done, IC-7300 on COM5, 5352000.00000 , CWN
//       0006695  Traffic Log paused on Radio Port after initialisation.
//  3725 0074825 r2 006 FE FE 94 E0 03 FD                                             ; VFOfreq ?
//  3726 0074825 t2 00B FE FE E0 94 03 00 20 35 05 00 FD                   ; VFOfreq 5352000.0 Hz
//  3727 0074856 r2 007 FE FE 94 E0 25 00 FD                                       ; SelVFOFreq ?
//  3730 0074866 TX 007 FE FE 94 E0 25 00 FD                                       ; SelVFOFreq ?
//  3731 0074930 RX 03C FE FE E0 94 27 00 00 04 11 00 00 00 12 11 07 11 10 00 ..  ; fragment 4/11
//  3732 1074862 RX 00C FE FE E0 94 25 00 00 20 35 05 00 FD                  ; SelVFOFreq:5352000
//       '--- had been paused on a breakpoint here, thus timestamps are "corrupted"
//  3740 1074862 t2 00C FE FE E0 94 25 00 00 20 35 05 00 FD                  ; SelVFOFreq:5352000
//       In the above snippet from the TRAFFIC LOG, ...
//       Message #3725 could be answered immeditately because the associated value
//                     was already present in RAM.
//       Message #3727 from WSJT-X had to be forwarded to the radio (#3730),
//       Messages #3728, 3729, 3731 (and more) were UNSOLITITED MESSAGES from the radio,
//       Message #3732 contains the READ RESPONSE FROM THE RADIO
//                      (it caused the transition to RIGCTRL_SSTATE_FWD2RADIO_DONE),
//       Message #3740 is the response on the SERVER PORT to the REMOTE CLIENT
//         (causing the final transition to RIGCTRL_SSTATE_PASSIVE,
//          shortly before WSJT-X banged on the door again, asking for more)...
//
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

#include "switches.h" // project specific 'compilation switches', e.g.:
         // #define SWI_RIGCTRL_ACT_AS_SERVER 1 // Act as server emulating e.g.
         // an Icom- radio with CI-V on 'Additional COM Ports' ? 0=no, 1=yes
         //   (optional; added 2025-07; details in RigControl_CIV_Server.c )

#include <string.h>     // memset() isn't really a STRING function, but..
#include <stdio.h>      // snprintf() isn't "I/O", but declared in string.h
#include <math.h>       // fmodl() ..
#include "Timers.h"     // high-resolution stopwatches, etc
#include "Utilities.h"  // UTL_FormatDateAndTime(), UTL_GetCurrentUnixDateAndTime(), etc
#include "QFile.h"      // "Quick File" helper functions by DL4YHF
#include "Inet_Tools.h" // INET_GetU16FromLE(), etc
#include "RigControl.h" // header file for THIS module


// internal forward references :
static int  RigCtrl_CIV_Server_MapDataIntoReadResponse(T_RigCtrl_PortInstance *pServerPortInstance, long i32Value);

//---------------------------------------------------------------------------
void RigCtrl_CIV_Server_PrepareParsing( T_RigCtrl_PortInstance *pServerPortInstance )
  // Called shortly BEFORE RigCtrl_ParseCIV() from AuxComThread() -> ..
  //                  -> AuxCom_PassRxOrTxDataToProtocolSpecificDecoder() .
  // See server state diagram explained at the begin of RigControl_CIV_Server.c !
{
  RigCtrl_SetServerState(pServerPortInstance, RIGCTRL_SSTATE_PARSING_COMMAND); // here: set in RigCtrl_CIV_Server_PrepareParsing()
  pServerPortInstance->iStartOfDataField = 4; // .. after the PREAMBLE and ADDRESSES
  pServerPortInstance->iLengthOfDataField= 0;
} // end RigCtrl_CIV_Server_PrepareParsing()


//---------------------------------------------------------------------------
void RigCtrl_CIV_PatchAddressesInResponse( T_RigCtrl_PortInstance *pServerPortInstance,
        BYTE *pbResponseMessage )
  // In CI-V, if the COMMAND had the "To"-address set to 0x00 (= "Broadcast", aka "To ANY"),
  //      the RESPONSE must have the "From"-address set to the radio's REAL CI-V address !
  // Example:
  // Msg# time/ms dir len pre  to fm cmd .. postable (FD)  ; comment
  // Byte index :         0  1  2  3  4  5  6  7  8  9 10 11 ...
  // -------------------------------------------------------------------
  // >  1 0000939 TX 007 FE FE 00 E0 19 00 FD             ; get radio ID
  //                           '-- RIGCTRL_DEF_ADDR_AUTO_DETECT
  // >  3 0000948 RX 008 FE FE E0 94 19 00 94 FD           ; Rig=IC-7300
  //                              '-- used as "FROM"-address further below
  // >  5 0000973 TX 006 FE FE 94 E0 03 FD                ; read VFOfreq
  // >  7 0001037 RX 00B FE FE E0 94 03 87 89 09 50 00 FD  ; VFOfreq=...
  // -------------------------------------------------------------------
{
  T_RigCtrlInstance *pRC = pServerPortInstance->pRC;
  T_RigCtrl_PortInstance *pRadioPortInstance = &pRC->PortInstance[RIGCTRL_PORT_RADIO];
  if( pbResponseMessage[3] == 0x00 ) // "From"-address still 0x00 (copied from the "To"-address in the COMMAND) ?
   { if( (pServerPortInstance->iRadioDeviceAddr != RIGCTRL_DEF_ADDR_AUTO_DETECT/*0x00*/ )
      && (pServerPortInstance->iRadioDeviceAddr != RIGCTRL_DEF_ADDR_NO_REMOTE_CTRL/*<0*/)
       )  // The "From"-address [3] must REALLY be patched... and we have a CONFIGURED address for this port:
      { // (which is rarely the case; usually pServerPortInstance->iRadioDeviceAddr is CONFIGURED(!) to RIGCTRL_DEF_ADDR_AUTO_DETECT.
        //  Thus, the 'Additional COM Port' configured as 'Virtual Radio' (emulator)
        //  will often use pRadioPortInstance->iRealRadioAddress (further below) !
        pbResponseMessage[3] = (BYTE)pServerPortInstance->iRadioDeviceAddr;
      }
     else // pServerPortInstance->iRadioDeviceAddr is not a valid "From"-address, so use THIS:
     if( pRadioPortInstance->iRealRadioAddress > 0 )
      { pbResponseMessage[3] = (BYTE)pRadioPortInstance->iRealRadioAddress;
        // ,---------------------------------------------|_______________|
        // '--> This has been "captured" from a RESPONSE (not a COMMAND)
        //      received from the REAL RADIO, in RigCtrl_ParseCIV(),
        // e.g. 0x94 if the REAL RADIO is an IC-7300 using its DEFAULT CI-V-address.
      }
   }

} // end RigCtrl_CIV_PatchAddressesInResponse()


//---------------------------------------------------------------------------
void RigCtrl_CIV_Server_OnReadCommand( T_RigCtrl_PortInstance *pServerPortInstance,
        T_RigCtrl_ParamInfo *pPI, // [in] e.g. parameter info from RigCtrl_ParameterInfo[], but beware:
                         // * may be NULL if the remote client asked for something we don't know,
                         // * MULTIPLE ICOM CI-V COMMANDS may refer to the same "unified parameter number" !
                         // [in] pPI->pszToken   = e.g. "UnselVfoOpMode" (WFView often wanted to know this),
                         //       '-->iUnifiedPN = e.g. RIGCTRL_PN_UNSEL_VFO_OP_MODE, ....
        BYTE *pbMessage, // [in] the complete RECEIVED CI-V MESSAGE, including preable, addresses, command, sub-command[s], etc
        int iMsgLength,  // [in] length of pbMessage in bytes, including pre- and postable .
                         //      Important for forwarding "completely unknown" commands
                         //      from a SERVER PORT to the REAL RADIO !
        int iMsgType,    // [in] RIGCTRL_MSGTYPE_xxx (with protocol-independent info from a parser)
        int iCIVDataType)// [in] type of data to be filled in in the READ-RESPONSE
                         //      (a kludge, since RigCtrl_ParseCIV() already contains an
                         //       enormous switch/case lists for commands and sub-commands,
                         //       it can easily PASS IN this info, so we don't need to guess HERE)
        // [in]  : pServerPortInstance->sz255CommentFromParser : e.g. "Transmitting ?"
        // [out] : pServerPortInstance->sTxMsg : message to send in response,
        //                                       but in a some cases WITHOUT
        //                                       valid DATA. This is indicated by..
        // [out] : pServerPortInstance->iServerState:
        //             RIGCTRL_SSTATE_RESPONSE_READY when the DATA are already filled in;
        //             RIGCTRL_SSTATE_WAIT_RADIO_NBUSY when another thread must read
        //                                           the data from the REAL RADIO first.
        // [out] : pServerPortInstance->dwExpectedResponse_CmdAndSubcode :
        //             Kind of "filter" for RigCtrl_CheckAndForwardResponseToClients(),
        //             because there may be OTHER MESSAGES arriving from the
        //             RADIO PORT (like unsolicited messages, or other responses).
        //
  // Stores the message 'for later' (parts of the READ-COMMAND are copied into
  //                                 what eventually becomes the READ-RESPONSE).
  // Called from RigCtrl_ParseCIV() as soon as the 'COMMAND' has been identified
  // as a READ COMMAND,  BEFORE the received message has been 'echoed',
  //                 and BEFORE the received message has been added to the
  //                     traffic log (and before there's a COMMENT for the log).
  //
{
  int nBytesForData = RigCtrl_GetSizeOfCIVDataType( iCIVDataType );
  int nBytesMapped;
  int iUnifiedPN;
  long i32Value;
  int iRigCtrlPort = RigCtrl_PortInstancePtrToIndex( pServerPortInstance );
  T_RigCtrlInstance *pRC = pServerPortInstance->pRC;
  BOOL fOk = FALSE;

  if( ! pServerPortInstance->fActAsServer ) // <- checked HERE because there are DOZENS of callers that don't care
   { return;
   }
  if( ! pRC->fGotBasicParams )  // RigControl.c hasn't read all "basic" parameters FROM THE RADIO yet -> cannot "serve" any external client yet !
   { return;
   }
  // "Garbage in -> NOTHING out" ...
  if( (nBytesForData < 0) || (iMsgLength < 3) || ((iMsgLength+nBytesForData) >= RIGCTRL_MAX_MESSAGE_LENGTH) ) // reject nonsense ..
   { return;  // ... do NOTHING instead of crashing with an access violation further below
   }

  if( pPI != NULL ) // RigCtrl_ParseCIV() has already found the PARAMETER INFO -> STORE IT for later
   { pServerPortInstance->sParamInfoForPendingCommand = *pPI;
     // Note: There are often MULTIPLE CI-V commands to read the same thing,
     //       e.g. the VFO frequency. So don't trust the PARAMETER INFO (*pPI) alone !
     //       The COMMAND, SUB-COMMAND, SUB-SUB-COMMAND, and who-knows-what
     //       are in pbMessage[] at this point, and THEY will be use for the
     //       response (if e.g. something was READ via cmd 0x03, the READ-RESPONSE
     //       must use command-byte 0x03, too ! See example sketched below.
   }
  else // no PARAMETER INFO passed in by the caller, so fill out a dummy:
   { // clear unknown/"future" members of the struct:
     memset( &pServerPortInstance->sParamInfoForPendingCommand, 0, sizeof(T_RigCtrl_ParamInfo) );
     // '--> e.g. iUnifiedPN, pszToken, pszUnit, pStringValueTable,
     //           iByteOffsetIntoRigCtrlInstance, iRigCtrlDataType,
     //           iAccessFlags, pbCIVReadCmd, iCIVReadCmdLength, ...
     pServerPortInstance->sParamInfoForPendingCommand.iUnifiedPN = RIGCTRL_PN_UNKNOWN;
     pServerPortInstance->sParamInfoForPendingCommand.iCIVDataType = iCIVDataType;
   }
  iUnifiedPN = pServerPortInstance->sParamInfoForPendingCommand.iUnifiedPN;
  if( iUnifiedPN == RIGCTRL_PN_UNSEL_VFO_OP_MODE ) // breakpoint for a "problematic" parameter:
   { iUnifiedPN = iUnifiedPN;  // <- set breakpoint HERE
   }

  // Store the entire received READ COMMAND from the remote client,
  //   because we may have to forward it to the RADIO PORT (e.g. if we
  //   don't understand its meaning, or don't have a READ-OUT VALUE yet):
  memcpy( pServerPortInstance->sRxMsg[1].bData, pbMessage, iMsgLength ); // here: original READ REQUEST from a remote client, stored for 'later forwarding' in RigCtrl_CIV_Server_OnReadCommand()
  pServerPortInstance->sRxMsg[1].iLength = iMsgLength;
  RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_CMD);
  pServerPortInstance->sRxMsg[1].iType = iMsgType;

  // Fill out the READ-RESPONSE (as a 'template', with dummy bytes for the data):
  // The first part of the READ-RESPONSE equals the READ COMMAND, e.g. :
  //   b[0] [1] [2] [3] [4] [5] [6] [7] [8] [9 [10]
  //  > FE  FE  94  E0  03  FD                     ("Read operating frequency", iMsgLength=6)
  //  > FE  FE  E0  94  03  00  00  00  07  00  FD (Read-Response WITH AN UNPREDICTABLE NUMBER OF DATA BYTES)
  //            |   |   |   |________________| iCIVDataType=CIV_DTYPE_BCD10, here: 07.000000 MHz for a SHORTWAVE radio
  //         "To"   |  cmd  :
  //              "From"    With iMsgLength=6 from the READ COMMAND,
  //                        the DATA FIELD begins here.
  memcpy( pServerPortInstance->sTxMsg.bData, pbMessage, iMsgLength ); // <- starting point for a READ RESPONSE...
  // As shown in the example above, the "TO" and "FROM"-addresses
  // must be SWAPPED in the response ....  :
  pServerPortInstance->sTxMsg.bData[2] = pbMessage[3];   // "From"-address[3] becomes "To"-address[2]
  pServerPortInstance->sTxMsg.bData[3] = pbMessage[2];   // "To"-address[2] becomes "From"-address[3]
  pServerPortInstance->iStartOfDataField = iMsgLength-1; // e.g. with the READ-COMMAND's iMsgLength=6, DATA IN THE RESPONSE will begin at byte[5]
  pServerPortInstance->iLengthOfDataField= nBytesForData;
  memset( pServerPortInstance->sTxMsg.bData + pServerPortInstance->iStartOfDataField, 0, nBytesForData );
  pServerPortInstance->sTxMsg.bData[iMsgLength+nBytesForData-1] = 0xFD; // new postamble
  pServerPortInstance->sTxMsg.iLength = iMsgLength + nBytesForData;
    // Example: WSJT-X asks for           |            |
    //          the VFO frequency :       6 bytes  +   5 bytes
    //                                                 '--> set to i32Value in BCD10 format below.
    // Example: WFView asks for the "Unselected VFO Operating Mode, Data Mode, and Filter Number
    //          (all in one command, which gave some headaches and required extra work in 2025-08):
    //  READ REQUEST : > r2 007 FE FE 94 E0 26 01 FD          ; UnselVfoOpMode ?
    //  READ RESPONSE: > t2 00A FE FE E0 94 26 01 03 00 02 FD ; ....
    //                        |              |  |  |  |  |
    //                        |              |  |  |  |  '--> Filter (1..3)
    //                        |              |  |  |  '--> DataMode, 0=off, 1=on
    //                        |              |  |  '--> OpMode, 3=CW
    //                        |              |  | \_________________/
    //                        |              |  |  iCIVDataType=CIV_DTYPE_BCD6(!)
    //                        |              |  |
    //                    iMsgLength=10    cmd  0=selected VFO,
    //                                          1=unselected VFO.
    //  (0x26 0x01 -> RIGCTRL_PN_UNSEL_VFO_OP_MODE,
    //                now in pServerPortInstance->sParamInfoForPendingCommand )
  // Also here: If the COMMAND had the "To"-address set to 0x00 (= "Broadcast", "To ANY"),
  //   the RESPONSE must have the "From"-address set to the radio's REAL CI-V address !
  RigCtrl_CIV_PatchAddressesInResponse( pServerPortInstance, pServerPortInstance->sTxMsg.bData );

  // Can the read response's DATA FIELD be filled in already ?
  // (to minimize the read-latency; if the calling thread doesn't need to wait)
  // Notes:
  //   * Parameters that are KNOWN to change their values autonomously,
  //     like the S-meter reading, are POLLED from the real radio anyway.
  //   * Esoteric Icom commands like "Get Selected VFO Frequency" (cmd 0x25 0x00)
  //     can only be forwarded to the radio if the radio is LOCALLY CONNECTED,
  //     and is controlled via CI-V. If the 'Virtual Radio emulation' runs on
  //     the CLIENT SIDE, CI-V will *not* be forwarded to the radio !
  //     Thus, if the FIRST call of RigCtrl_GetParamValue_Int() further below
  //     fails (e.g. to read the 'Selected VFO Frequency'
  if( iUnifiedPN == RIGCTRL_PN_FILTER_BANDWIDTH ) // "problematic" parameter ? (2025-08-03: "FilterBW" was wrong,
   { // after WSJT-X fiddled with the VFO. Full story in RigCtrl_OnVfoSwitch(!) .
     iUnifiedPN = iUnifiedPN;  // <-- set a breakpoint HERE, and step through the rest..
   }
  if( iUnifiedPN == RIGCTRL_PN_UNSEL_VFO_OP_MODE ) // "problematic" parameter ?
   { iUnifiedPN = iUnifiedPN;  // <-- set a breakpoint HERE, and step through the rest..
     // > 755 13:07:30.3 r2 007 FE FE 94 E1 26 01 FD  ; UnselVFOOpModeMode ?
     // > 756 13:07:30.3 t2 006 FE FE E1 94 FA FD     ; UnselVFOOpModeMode=CW
   }

  i32Value = RigCtrl_GetParamValue_Int( pServerPortInstance->pRC,
               &pServerPortInstance->sParamInfoForPendingCommand );
  if( i32Value != RIGCTRL_NOVALUE_INT )
   { nBytesMapped = RigCtrl_CIV_Server_MapDataIntoReadResponse( pServerPortInstance, i32Value );
                    // '-- [in] 
     // If all worked as planned, nBytesMapped should equal nBytesForData :
     if( nBytesMapped == nBytesForData )
      { RigCtrl_SetServerState(pServerPortInstance, RIGCTRL_SSTATE_RESPONSE_READY); // here: set in RigCtrl_CIV_Server_OnReadCommand(), with the read-value immediately available (without forwarding to the 'real radio')
        // Next steps : TRANSMISSION of the response from server to client,
        //      in RigCtrl_SendServerResponse(), called from RigCtrl_ProcessData(),
        //      with iServerState == RIGCTRL_SSTATE_RESPONSE_READY .
        //  ( RigCtrl_SendServerResponse() also patches sz255CommentFromParser,
        //    so we don't need to fiddle with that here to make the traffic log
        //    look nice, e.g. request "Transmitting ?" -> response "Transmitting=0" )
        fOk = TRUE; // response for the SERVER PORT is "ok", no need to forward to the RADIO PORT
      }
   }
  if( !fOk )  // parameter isn't available yet -> FORWARD the read-command to the radio port !
   {          // (fasten seat belts, this is another place where it gets really dirty)
     // Forward the message, and "Wait for the radio":
     RigCtrl_SetServerState(pServerPortInstance, RIGCTRL_SSTATE_WAIT_RADIO_NBUSY);  // here: set in RigCtrl_CIV_Server_OnReadCommand()
     // '--> Call RigCtrl_ForwardCommandFromServerPortToRadio() a.s.a.p.,
     //      not from the thread that "runs the Virtual Radio Server",
     //      but from the thread that "talks to the REAL RADIO" !
     pServerPortInstance->dwExpectedResponse_CmdAndSubcode = // prepare info for RigCtrl_CheckAndForwardResponseToClients():
        RigCtrl_GetCombinedCmdAndSubcodeForCIV( pbMessage, iMsgLength, iMsgType );
     // Example seen when WFView (as test client) started to ask questions:
     //  pServerPortInstance->sParamInfoForPendingCommand.iUnifiedPN = RIGCTRL_PN_ATU_ENABLED
     //  pServerPortInstance->sParamInfoForPendingCommand.pszToken = "ATU_enabled"
     //  pServerPortInstance->sRxMsg[1].iLength = 7
     //  pServerPortInstance->sRxMsg[1].bData = 0xFE 0xFE 0x94 0xE1 0x1C 0x01 0xFD
     //                                                             ,--'    |
     //                                                             | ,-----'
     //  pServerPortInstance->dwExpectedResponse_CmdAndSubcode = 0x1C01FFFF
     //
     // Next steps in RigCtrl_Handler() -> RigCtrl_ForwardCommandFromServerPortToRadio(),
     //               called as soon as the RADIO PORT is *not* busy,
     //               sends pServerPortInstance->sRxMsg[1] .
     // If all works as planned, the traffic log will shown the following (for the above example):
     // > r2 007 FE FE 94 E1 1C 01 FD ; ATU_enabled ? (READ COMMAND received on server port)
     // > TX 007 FE FE 94 E1 1C 01 FD ; ATU_enabled ? (READ COMMAND SENT TO RADIO PORT)
     // >
     // >
     //
   } // end if < need to forward the READ COMMAND from an external client to the REAL RADIO > ?


  // For the decision if e.g. WFView's "initialisation phase" is over :
  //  (similar also for "Serial Port Tunnels", see AuxComPort.c : AuxCom_RigControlCallback() )
  switch( iUnifiedPN )  // criterion for RIGCTRL_TMON_DISPLAY_OPTION_PAUSE_ON_INIT_DONE ?
   { case RIGCTRL_PN_S_METER_LEVEL_DB :
        if( pServerPortInstance->iNumSMeterReports < 32767 )
         { ++pServerPortInstance->iNumSMeterReports;
         }
        break;
     case RIGCTRL_PN_FREQUENCY :
        if( pServerPortInstance->iNumVfoFreqReports < 32767 )
         { ++pServerPortInstance->iNumVfoFreqReports;
         }
        break;
     default:
        break;
   }

  if( (RigCtrl_TrafficMonitor.iDisplayOptions & RIGCTRL_TMON_DISPLAY_OPTION_PAUSE_ON_INIT_DONE ) // here: for a SERVER PORT ..
   && (!pServerPortInstance->fTrafficMonitorPausedOnTrigger) )
   { // In contrast to the RADIO CONTROL PORT, it's difficult to find out
     // when the REMOTE CLIENT has finished the 'initialisation phase', i.e.
     // after all parameters have been read ONCE. For exampl, if the remote client
     // is WFVIEW, one of the FIRST parameters read during the init-phase
     // is the Icom radio's 'Default CI-V Address' (to detect the RIG TYPE).
     // C:\cbproj\Remote_CW_Keyer\Extra_Literature\Trouble_with_WFVIEW_on_CLIENT_PORT.txt
     // contains a few examples (traffic log extracts) with the MANY parameters
     // read by WFView before beginning the "normal polling cycle".
     // Indicators for the begin of WFView's "normal polling cycle" were:
     //  * repeated "read S-Meter" (cmd 0x15.02 = RIGCTRL_PN_S_METER_LEVEL_DB)
     //  * repeated "read OpMode"  (cmd 0x04    = RIGCTRL_PN_OP_MODE)
     // Indicators for the begin of N1MM Logger's normal polling cycle was
     //    repeated reading the VFO FREQUENCY .
     // Thus:
     if( (  (pServerPortInstance->iNumSMeterReports >= 2 )
         || (pServerPortInstance->iNumVfoFreqReports>= 2) )
       &&( pServerPortInstance->dwNumMessagesSent > 10 )
       &&( pServerPortInstance->dwNumMessagesRcvd > 10 ) )
      { char sz255[256];
        // Before pausing the log FOR THIS PORT, leave a note explaining WHY the traffic monitor is paused:
        sprintf(sz255, "Traffic Log paused on %s after initialisation.",
           RigCtrl_GetDescriptivePortName(pServerPortInstance) ); // <- delivers e.g. "Virtual Radio Port #2"
           // (this shall resemble the menu item 'Pause when initialisation done' )
        RigCtrl_AddToTrafficLog( pServerPortInstance->pRC, iRigCtrlPort, RIGCTRL_ORIGIN_CONTROLLER, NULL/*pbMessage*/, 0/*iMsgLength*/,
                                 RIGCTRL_MSGTYPE_INFO, sz255 );

        RigCtrl_fUpdateTrafficLog = TRUE; // now "ready for being UPDATED on-screen"
        pServerPortInstance->fTrafficMonitorPausedOnTrigger = TRUE;
      }
   } // end RIGCTRL_TMON_DISPLAY_OPTION_PAUSE_ON_INIT_DONE


} // end RigCtrl_CIV_Server_OnReadCommand()

//---------------------------------------------------------------------------
void RigCtrl_CIV_Server_OnWriteCommand( T_RigCtrl_PortInstance *pServerPortInstance,
        T_RigCtrl_ParamInfo *pPI, // [in] e.g. parameter info for "VFOfreq", but beware:
                         // * may be NULL if the remote client tries to SET something we don't know,
                         // * MULTIPLE ICOM CI-V COMMANDS may refer to the same "unified parameter number" !
        BYTE *pbMessage, // [in] the complete RECEIVED CI-V MESSAGE, including preable, addresses, command, sub-command[s], etc
        int iMsgLength,  // [in] length pbMessage, from (and including) pre- and postable
        int iMsgType,    // [in] RIGCTRL_MSGTYPE_xxx (with control-independent info from a parser)
        int iCIVDataType)// [in] type of data to be filled in in the READ-RESPONSE
                         //      (a kludge, since RigCtrl_ParseCIV() already contains such
                         //       enormous switch/case lists for commands and sub-commands,
                         //       it can easily PASS IN this info, so we don't need to guess HERE)

  // Called from RigCtrl_ParseCIV() as soon as the 'Message' has been identified
  //    as a WRITE COMMAND, received on any of the SERVER PORTS (aka 'Additional COM Ports').
  //
  // - - - Example - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // Using WSJT-X as remote client, got here for the first time with ...
  //  pPI          = NULL  (before adding support for at least VFO "A" and "B" in Remote CW Keyer)
  //  pbMessage    = FE FE 94 E0 07 00 FD = "Set Selected VFO to 'A'"
  //                                \|__ there's more to this byte than you expect !
  //  iMsgLength   = 7
  //  iMsgType     = 0x03001F = RIGCTRL_MSGTYPE_FLAG_WRITE_CMD | RIGCTRL_MSGTYPE_OTHER
  //  iCIVDataType = 2 = CIV_DTYPE_BCD2 ("two-digit BCD", ok; see "A7292-4EX-11" page 161
  //  pServerPortInstance->sz255CommentFromParser = "SelVFO:00"
  //  Caller: RigCtrl_ParseCIV() [somewhere after 'case 0x07:']
  //  More on this particular case in RigCtrl_ParseCIV(), somewhere after 'case 0x07:' !
  //
  // If all works as planned, the entire sequence (with forwarding to the RADIO PORT)
  // should look as follows in RCW Keyer's Traffic Log:
  //  > r2 007 FE FE 94 E0 07 00 FD  ; SelVFO:00
  //  > TX 007 FE FE 94 E0 07 00 FD  ; SelVFO:00
  //  > RX 006 FE FE E0 94 FB FD     ; SelVFO:ok
  //  > t2 006 FE FE E0 94 FB FD     ; SelVFO:ok
  //
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
{
  int nBytesForData = RigCtrl_GetSizeOfCIVDataType( iCIVDataType );
  int nBytesMapped;
  int i32Value;
  BYTE *pbBufPtr;
  T_RigCtrlInstance *pRC = pServerPortInstance->pRC;

  if( ! pServerPortInstance->fActAsServer ) // <- checked HERE because there are DOZENS of callers that don't !
   { return;
   }
  if( ! pRC->fGotBasicParams ) // RigControl.c hasn't read all "basic" parameters FROM THE RADIO yet -> cannot "serve" any external client yet !
   { return;
   }

  // "Garbage in -> NOTHING out" :
  if( (nBytesForData < 0) || (iMsgLength < 3) || ((iMsgLength+nBytesForData) >= RIGCTRL_MAX_MESSAGE_LENGTH) ) // reject nonsense ..
   { return;  // instead of crashing with an access violation below
   }

  if( pPI != NULL ) // RigCtrl_ParseCIV() has already found the PARAMETER INFO -> STORE IT (for later)
   { pServerPortInstance->sParamInfoForPendingCommand = *pPI;
     // Note: pPI->pbCIVReadCmd may not be correct here, because
     //       there are so many CIV-commands to read e.g. the VFO frequency.
     //       The COMMAND, SUB-COMMAND, SUB-SUB-COMMAND, and who-knows-what
     //       are in pbMessage[] at this point ! See example sketched below.
   }
  else // no PARAMETER INFO passed in by the caller, so fill out a dummy:
   { // clear unknown/"future" members of the struct:
     memset( &pServerPortInstance->sParamInfoForPendingCommand, 0, sizeof(T_RigCtrl_ParamInfo) );
     // '--> e.g. iUnifiedPN, pszToken, pszUnit, pStringValueTable,
     //           iByteOffsetIntoRigCtrlInstance, iRigCtrlDataType,
     //           iAccessFlags, pbCIVReadCmd, iCIVReadCmdLength, ...
     pServerPortInstance->sParamInfoForPendingCommand.iUnifiedPN = RIGCTRL_PN_UNKNOWN;
     pServerPortInstance->sParamInfoForPendingCommand.iCIVDataType = iCIVDataType;
   }

  // Store the received WRITE COMMAND from the remote client, because
  // we may have to forward it to the RADIO PORT as an almost verbatim copy:
  memcpy( pServerPortInstance->sRxMsg[1].bData, pbMessage, iMsgLength );
  pServerPortInstance->sRxMsg[1].iLength = iMsgLength;
  RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_WRITE_CMD);
  pServerPortInstance->sRxMsg[1].iType = iMsgType;


  // Similar as in RigCtrl_CIV_Server_OnReadCommand(), preset the COMPLETE RESPONSE
  //     (in this case for a WRITE-COMMAND) on the SERVER PORT ?
  // NO.. because in CI-V, a WRITE-RESPONSE looks totally different then a
  //                         READ-RESPONSE !
  // (a READ-RESPONSE contains at least the command, and maybe the sub-command,
  //    from the original READ-COMMAND.
  //  A WRITE-RESPONSE does not.. it only says "FB" (Fine Business?), and that's it ! )
  // So here we are, be optimistic, and assemble a WRITE RESPONSE saying "Fine Business" a la CI-V:
  pbBufPtr = pServerPortInstance->sTxMsg.bData;
  *pbBufPtr++ = 0xFE;  // fixed preamble code for CI-V protocol (term used by Icom .. NOT "Begin of Message" ! )
  *pbBufPtr++ = 0xFE;  // fixed preamble code for CI-V protocol
  // Also here, the "TO" and "FROM"-addresses must be SWAPPED in the response:
  *pbBufPtr++ = pbMessage[3];  // "to" transceiver's address
  *pbBufPtr++ = pbMessage[2];  // "from" controller's address
  *pbBufPtr++ = 0xFB;  // "Fine Business" (the only alternative would be 0xFA = "Fail")
  *pbBufPtr++ = 0xFD;  // CI-V code for "End Of Message" (term used by Icom .. NOT "postamble", even though 0xFE is called "preamble" ! )
  pServerPortInstance->sTxMsg.iLength = (int)(pbBufPtr - pServerPortInstance->sTxMsg.bData);
  // Again: The WRITE RESPONSE doesn't have a data field,
  //   only the WRITE COMMAND FORWARDED TO THE RADIO PORT has ..
  pServerPortInstance->iStartOfDataField = pServerPortInstance->iLengthOfDataField = 0;

  // Also here: If the COMMAND had the "To"-address set to 0x00 (= "Broadcast", "To ANY"),
  //   the RESPONSE must have the "From"-address set to the radio's REAL CI-V address !
  RigCtrl_CIV_PatchAddressesInResponse( pServerPortInstance, pServerPortInstance->sTxMsg.bData );


  RigCtrl_SetServerState(pServerPortInstance, RIGCTRL_SSTATE_WAIT_RADIO_NBUSY); // here: set in RigCtrl_CIV_Server_OnWriteCommand()
     // '--> call RigCtrl_ForwardCommandFromServerPortToRadio() a.s.a.p.,
     //      not from the thread that "runs the Virtual Radio Server",
     //      but from the thread that "talks to the REAL RADIO".
  // Prepare info for RigCtrl_CheckAndForwardResponseToClients():
  pServerPortInstance->dwExpectedResponse_CmdAndSubcode = RCTL_COMBINE_CMD_ONLY(0xFB); // be optimistic, expect "Fine Business" from the REAL RADIO

  // Next steps in RigCtrl_Handler() -> RigCtrl_ForwardCommandFromServerPortToRadio(),
  //               called as soon as the RADIO PORT is *not* busy .


} // end RigCtrl_CIV_Server_OnWriteCommand()


//---------------------------------------------------------------------------
void RigCtrl_CIV_Server_PrepareResponse_OkOrNotOk( // .. with cmd = 0xFA or 0xFB ..
       T_RigCtrl_PortInstance *pServerPortInstance, BOOL fOk)
  // [out] Icom's CI-V 'error' or 'ok' response, in pServerPortInstance->sTxMsg .
  // Also called from RigControl.c, for example if a command could not
  // be forwarded to the external radio, or returned an ERROR, thus not 'static' !
{
  BYTE *pbBufPtr = pServerPortInstance->sTxMsg.bData;

  *pbBufPtr++ = 0xFE;  // fixed preamble code for CI-V protocol (term used by Icom .. NOT "Begin of Message" ! )
  *pbBufPtr++ = 0xFE;  // fixed preamble code for CI-V protocol
  *pbBufPtr++ = (BYTE)pServerPortInstance->iRadioMasterAddr;  // "to" the EXTERNAL CLIENT'S address (Icom slang: "Controller", sometimes "Master")
  *pbBufPtr++ = (BYTE)pServerPortInstance->iRadioDeviceAddr;  // "from" the emulated radio's address
  *pbBufPtr++ = fOk ? 0xFB : 0xFA;  // 0xFB = "Fine Business" (OK), 0xFA = "FAilure" (NOT OK)
  *pbBufPtr++ = 0xFD;  // CI-V code for "End Of Message" (term used by Icom .. NOT "postamble", even though 0xFE is called "preamble" ! )
  pServerPortInstance->sTxMsg.iLength = pbBufPtr - pServerPortInstance->sTxMsg.bData;
  if( fOk )
   { pServerPortInstance->sTxMsg.iType = RIGCTRL_MSGTYPE_OK;
   }
  else
   { pServerPortInstance->sTxMsg.iType = RIGCTRL_MSGTYPE_NOT_OK;
   }
  // If the COMMAND had the "To"-address set to 0x00 (= "Broadcast", aka "To ANY"),
  //    the RESPONSE must have the "From"-address set to the radio's REAL CI-V address !
  RigCtrl_CIV_PatchAddressesInResponse( pServerPortInstance, pServerPortInstance->sTxMsg.bData );


} // end RigCtrl_CIV_Server_PrepareResponse_OkOrNotOk()


//---------------------------------------------------------------------------
static int RigCtrl_CIV_Server_MapDataIntoReadResponse(
               T_RigCtrl_PortInstance *pServerPortInstance,
               // '-- [in] pServerPortInstance->sParamInfoForPendingCommand.iUnifiedPN, etc
               long i32Value )  // [in] e.g. VFO frequency in Hertz
    // [out]    pServerPortInstance->sTxMsg.bData[ pServerPortInstance->iStartOfDataField .. ]
    // [return] number of bytes actually mapped into pServerPortInstance->sTxMsg
{
  BYTE *pbDest = pServerPortInstance->sTxMsg.bData + pServerPortInstance->iStartOfDataField;
  int  nBytesMapped;
  // The "value" for certain READ-RESPONSES cannot be transported in a simple 32-bit integer (i32Value).
  // In such cases, RigCtrl_CIV_Server_MapDataIntoReadResponse() assembles the data
  // from one or even multiple members of the T_RigCtrlInstance(!) :
  T_RigCtrlInstance *pRC = pServerPortInstance->pRC;

  // Anything special about the 'command' and 'subcommand' in CI-V to *WRITE* this parameter ?
  //  Beware: Similar special cases must be handled when sending a
  //       WRITE COMMAND on the CIV-CLIENT side [RigCtrl_SendWriteCommandForUnifiedPN()]
  //   and when assembling a
  //       READ RESPONSE on the CIV-SERVER size [RigCtrl_CIV_Server_MapDataIntoReadResponse()] !
  switch( pServerPortInstance->sParamInfoForPendingCommand.iUnifiedPN ) // anything SPECIAL ?
   { case RIGCTRL_PN_FREQUENCY: // One of the dreadful special cases where the
           // WRITE-command isn't just the READ-command follwed by WRITE-DATA:
           // A7508-3EX-4 page 4 : 0x03 = "READ the operating frequency".
           //                      0x05 = "SET the operating frequency".
           // The parameter info table (RigCtrl_ParameterInfo[]) only contains the code for the READ-command (0x03).
           // Since there was already an internal 'Set'-function use it:
           // see also: RigCtrl_SetVFOFrequency_Internal( pRC, (double)i32Value/*dblFreqHz*/ );
           break;
     case RIGCTRL_PN_OP_MODE :
           // see also: RigCtrl_SetOperatingMode_Internal( pRC, i32Value/*iOpMode*/ );
           break;
     case RIGCTRL_PN_TRANSMIT_REQUEST :
           break;
     case RIGCTRL_PN_SCOPE_SPAN :
           // see also: RigCtrl_SetScopeSpan_Internal( pRC );
           break;
     case RIGCTRL_PN_AUDIO_VOLUME_PERCENT :  // send a command to SET the radio's own loudspeaker volume..
           // What's the format for the 'AF Volume' ? A7508-3EX-4 page 4 said
           //       about command 0x14, sub command 0x01 :
           // > Send/read the AF level (0000=Minimum to 0255=Maximum) .
           //       Here for comparison the READ COMMAND, and an IC-7300's response:
           // > TX 007 FE FE 00 E0 14 01 FD       ; read AudioVolume
           // > RX 009 FE FE E0 94 14 01 00 34 FD ; AudioVolume:13 %
           //                      |  |  |___|
           //                    CMD SUB  DATA ("0000".."0255", NOT PERCENT).
           break;

     case RIGCTRL_PN_FILTER_BANDWIDTH: // "FilterBW" (audio filter bandwidth, quite FLEXIBLE for *DSP* rigs)
           // As the name (suffix) implies, T_RigCtrlInstance.iFilterBW_Hz
           // uses the one-and-only SI unit for any kind of frequency: HERTZ.
           // But Icom's cmd 0x1A subcmd 0x03 (*) uses an esoteric two-digit
           // CODE instead of a bandwidth in Hertz - see implementation of
           //      RigCtrl_CIV_AudioFilterBandwidthCodeToFrequency()
           // and  RigCtrl_CIV_FrequencyToAudioFilterBandwidthCode()
           i32Value/* Icom's "Filter Code" */
              = RigCtrl_CIV_FrequencyToAudioFilterBandwidthCode(
                    i32Value ); // [in] bandwidth in Hertz, no stupid "code"
           break; // end case RIGCTRL_PN_FILTER_BANDWIDTH

     case RIGCTRL_PN_CW_PITCH_HZ: // here: Convert from HERTZ into the CI-V internal unit:
           // > 0000 = 300 Hz,  0128 = 600 Hz, 0255 = 900 Hz.
           // A7380-7EX-2 page 3 tells a story about "5 Hz steps"
           // but THAT'S NOT THE SCALING STEP. Expect more fun like this !
           // From the inverse function in RigCtrl_ParseCIV() :
           // > dblConvFactor = 600.0/*Hz*/ / 256.0/*steps*/;
           // > dblConvOffset = 300.0/*Hz*/;
           // > dblValue [Hz] = (double)i32Value ["code" a la Icom] * dblConvFactor + dblConvOffset;
           i32Value/*"code" a la Icom*/ = (i32Value/*Hertz*/ - 300) * 256 / 600;
           RigCtrl_LimitInt32( &i32Value, 0, 255 );
           break; // end case RIGCTRL_PN_CW_PITCH_HZ


     case RIGCTRL_PN_TUNING_STEP: // Icom: cmd 0x10 (NO subcodes. Note: T_RigCtrlInstance.iTuningStep_Hz
           // uses HERTZ. Icom's command does NOT. Their stupid CODES are not only
           // manufacturer-specific, but even RIG-specific ! See developer's rant
           // in RigCtrl_CIV_TuningStepCodeToFrequency() ... here's the INVERSE function:
           i32Value = RigCtrl_CIV_TuningStepFrequencyToCode( pRC->iDefaultAddress, i32Value );
           break;

     case RIGCTRL_PN_SEL_VFO_OP_MODE: // Icom: cmd 0x26, sub 0x00 = "SELECTED VFO" OpMode+DataMode+FilterBW (!)
           // What's the data format for the 'SelAF Volume' ? A7508-3EX-4 page "25/28" said:
           // > Selected or unselected VFOs operating mode and filter settings (Only MAIN band)
           // >          pbDest[0]   [1]    [2]
           // >  ,-------,-------,-------,-------,
           // >  | X : X | X : X | X : X | X : X |
           // >  '-------'-------'-------'-------'
           // >      |       |       |       '-- (3) "Filter setting" (01="FIL1" .. 02="FIL2"; here: flags RIGCTRL_OPMODE_NARROW, etc)
           // >      |       |       '---------- (2) "Data mode setting" (00=Data mode OFF, 01=Data mode ON)
           // >      |       '------------------ (1) "Operating mode setting" (see codes in RigCtrl_IcomToRigCtrlOpmode)
           // >      |
           // >     00 = Selected VFO,  \ In THIS implementation, considered
           // >     11 = Unselected VFO / the SUB-COMMAND for "command 0x26",
           //                             NOT PART OF THE DATA FIELD !
           i32Value = RigCtrl_RigCtrlOpmodeToIcom( pRC->iOpMode );
           if( i32Value == RIGCTRL_NOVALUE_INT )
            { i32Value = 0x03; // Icom's code for CW
            }
           pbDest[0] = i32Value; // (1) "Operating mode setting" from cmd 0x26
           pbDest[1] = pRC->iSelVfoDataMode; // "Data mode setting" .. what the heck is that ? "USBD" instead of "USB", etc ? Heavens, no.
           pbDest[2] = RigCtrl_RigCtrlOpmodeToIcomFilterNumber( pRC->iOpMode );
           return 3; // end case RIGCTRL_PN_SEL_VFO_OP_MODE

     case RIGCTRL_PN_UNSEL_VFO_OP_MODE: // Icom: cmd 0x26, sub 0x01 = "UNSELECTED VFO" OpMode+DataMode+FilterBW (!)
           // Almost the same as RIGCTRL_PN_SEL_VFO_OP_MODE, but other members in pRC :
           i32Value = RigCtrl_RigCtrlOpmodeToIcom( pRC->iUnselVfoOpMode );
           if( i32Value == RIGCTRL_NOVALUE_INT )
            { i32Value = 0x03; // Icom's code for CW
            }
           pbDest[0] = i32Value; // (1) "Operating mode setting" from cmd 0x26
           pbDest[1] = pRC->iUnselVfoDataMode; // "Data mode setting" .. what the heck is that ? "USBD" instead of "USB", etc ? Heavens, no.
           pbDest[2] = RigCtrl_RigCtrlOpmodeToIcomFilterNumber( pRC->iUnselVfoOpMode );
           return 3; // end case RIGCTRL_PN_UNSEL_VFO_OP_MODE

     default:  // nothing special, so use the 'parameter info' from RigCtrl_ParameterInfo[],
           // which have already been copied into pServerPortInstance->sParamInfoForPendingCommand .
           // For example, when WFView (as external client) asked for the radio's
           // "Transceiver ID" alias "Default CI-V Address" via cmd 0x19 subcode 0x00,
           // RIGCTRL_PN_TRANSCEIVER_ID, got here with i32Value = 148 = 0x94 and
           // pServerPortInstance->sParamInfoForPendingCommand.iCIVDataType = 2 = CIV_DTYPE_BCD2 .
           // From the glorious TRAFFIC LOG:
           // ; Nr time/ms tr len pream to fm CIV-cmd, params..  postamble=FD
           //    1 0000967 TX 007 FE FE 00 E0 19 00 FD           ; get radio ID
           //    3 0000977 RX 008 FE FE E0 94 19 00 94 FD        ; Rig=IC-7300
           //  ,------------------------------------'
           //  '--> that's 0x94, not "interpreted as BCD" which would result in #94 (decimal).
           //       With an IC-9700, the result would be 0xA2, which clearly indicates
           //       that the ICOM DATA TYPE for this command (0x19 0x00)
           //       is not CIV_DTYPE_BCD2 but CIV_DTYPE_HEX_1BYTE !
           //   -> Fixed that in RigCtrl_ParameterInfo[] ... and added support
           //      for that new CIV_DTYPE_HEX_1BYTE in RigCtrl_MapCIVDataToMessage() .
           break;
   } // end switch( iUnifiedPN )

  nBytesMapped = RigCtrl_MapCIVDataToMessage( pbDest,
     pServerPortInstance->sParamInfoForPendingCommand.iCIVDataType, i32Value );

  return nBytesMapped;
} // end RigCtrl_CIV_Server_MapDataIntoReadResponse()

//---------------------------------------------------------------------------
long RigCtrl_CIV_TuningStepCodeToFrequency(
         int iDefaultAddress,   // [in] e.g. RIGCTRL_DEF_ADDR_IC_9700, RIGCTRL_DEF_ADDR_IC_7300, etc,
         // because the dreadful "tuning step codes" are not only MANUFACTURER-SPECIFIC
         // but even RIG-SPECIFIC ! For example, iIcomTuningStepCode = 0x02
         // means 500 Hz in an IC-9700, but 1000 Hz in an IC-7300 .
         int iIcomTuningStepCode )
  // Converts Icom's stupid RIG-SPECIFIC "code" for the tuning stepwith (in cmd 0x10)
  // into the one-and-only SI unit for frequencies:  Hertz ! (as used in
  //                                   T_RigCtrlInstance.iTuningStep_Hz) .
{
  switch( iDefaultAddress )
   { case RIGCTRL_DEF_ADDR_IC_7300:
     default :              // Assume the same applies to MANY OTHER shortwave radios, too:
        // Possible values taken from Icom's "A7292-4EX-11" (for the IC-7300):
        switch( iIcomTuningStepCode )
         { case 0x00: return 10;   // 0 Hertz: "Send/read the tuning step OFF" (guess this means 10 Hz or even less)
           case 0x01: return 100; // "Send/read the 100 Hz tuning step" (what they mean is GET or SET)
                // Note the absence of 500 Hz here (for the IC-7300) !
           case 0x02: return 1000;
           case 0x03: return 5000;
           case 0x04: return 9000;   // 9 kHz, for AM somewhere.. THIS stepwidth ONLY exists in an IC-7300 but NOT in an IC-9700 !
           case 0x05: return 10000;  // 10 kHz, for AM somewhere else
           case 0x06: return 12500;  // 12.5 kHz, for NBFM
           case 0x07: return 20000;
           case 0x08: return 25000;
           default  : break; // THAT'S ALL (for an IC-7300) !
           // Note the absence of "tuning step codes" for 10 or even 1 Hz steps !
         }
        return 100;
     case RIGCTRL_DEF_ADDR_IC_9700:
        // The following stepwidths were taken from "A7508-3EX-4", page 4 (for the IC-9700).
        // Note the shocking incompatibility between IC-7300 (where "0x02" means 1000 Hz)
        //                                       and IC-9700 (where "0x02" means 500 Hz) !
        // One more reason to get rid of these crappy rig-specific codes as soon as possible.
        switch( iIcomTuningStepCode )
         { case 0x00: return 10;  // "0 = OFF (which means step by 10 Hz or 1 Hz)
           case 0x01: return 100; // "Send/read the 100 Hz tuning step" (what they mean is GET or SET)
           case 0x02: return 500;
           case 0x03: return 1000;
           case 0x04: return 5000;
           case 0x05: return 6250; // <- another TUNING STEPWIDTH that does NOT exist in an IC-7300 !
           case 0x06: return 10000;  // 10 kHz .. and, you guessed it, INCOMPATIBLE with the IC-7300 !
           case 0x07: return 12500;  // 12.5 kHz, for NBFM
           case 0x08: return 20000;
           case 0x09: return 25000;
           case 0x10: return 50000;  // 50 kHz (IC-9700 only, presumable for FM broadcast)
           case 0x11: return 100000; // 100 kHz
           default:   break;
         }
        return 100;
   } // end switch( iDefaultAddress )

} // end RigCtrl_CIV_TuningStepCodeToFrequency()

//---------------------------------------------------------------------------
int RigCtrl_CIV_TuningStepFrequencyToCode(
         int iDefaultAddress,   // [in] e.g. RIGCTRL_DEF_ADDR_IC_9700, RIGCTRL_DEF_ADDR_IC_7300, etc,
         // because the dreadful "tuning step codes" are not only MANUFACTURER-SPECIFIC
         // but even RIG-SPECIFIC ! For example, iIcomTuningStepCode = 0x02
         // means "500 Hz" in an IC-9700, but "1000 Hz" in an IC-7300 ! Thank you, Icom.
         long i32TuningStep_Hz ) // [in] tuning stepwidth in HERTZ (SI unit, yeah !).
  // Inverse to RigCtrl_CIV_TuningStepCodeToFrequency(). Required by this server
  // to convert T_RigCtrlInstance.iTuningStep_Hz [in Hertz] into one of the
  // 'enum-like' codes used by CI-V command 0x10, when e.g. WFVIEW(!) asks
  // for it, and we don't want to tunnel the command from SERVER- to the RADIO-PORT
  // (to avoid unnecessary traffic on the REMOTE connection).
{ int iIcomTuningStepCode;
  if( i32TuningStep_Hz < 100 )  // CI-V command 0x10 cannot encode anything below 100 Hz, so:
   { return 0;  // code for "the finest available tuning-step-width" 
   }
  for( iIcomTuningStepCode=0; iIcomTuningStepCode<=8; ++iIcomTuningStepCode )
   { if( RigCtrl_CIV_TuningStepCodeToFrequency( iDefaultAddress, iIcomTuningStepCode ) == i32TuningStep_Hz )
      { return iIcomTuningStepCode;
      }
   }
  // Arrived here: The tuning step (e.g. 10 Hz or 50 Hz, as supported by any modern all-mode rig)
  //               is not supported by CI-V command 0x10 - so ..
  return 1; // "A7292-4EX-11" (IC-7300)  : "Send/read the 100 Hz tuning step"

} // end RigCtrl_CIV_TuningStepFrequencyToCode()

