//---------------------------------------------------------------------------
// File: C:\cbproj\Remote_CW_Keyer\RigControl.h
// Authors: Wolfgang Buescher (DL4YHF)    .
//       (this stripped-down variant for DL4YHF's "Remote CW Keyer"
//        does NOT support the ugly "RS-BA1 protocol" and thus is MUCH
//        easier to port to a simple microcontroller firmware) .
// Date:   2021-09-16
// Purpose:
//  "Rig Control" for a few transceivers, mostly Icom, and a few others.
//  Started 2018-12 with a parser for Icom'c CI-V protocol, implemented in
//          c:\cbproj\SpecLab\RigControl\CIV_parser.c .
//          Written in C, not C++, to MAYBE use parts of this in a micro-
//          controller firmare (a plan that didn't materialize yet...) .
//  First used in DL4YHF's Spectrum Lab .
//  Later (2024) stripped down for use in DL4YHF's "Remote CW Keyer",
//               without support for the poorly specified "RS-BA1" protocol
//               by Icom ..
//
// Details / Literature / Revsision History : See implementation (*.c) !
//---------------------------------------------------------------------------

#ifndef RigControlH
#define RigControlH

#ifndef _DATA_TYPES_H_
# include "yhf_type.h" // old fashioned stuff like BYTE, WORD, DWORD, etc^10
#endif
#ifndef _YHF_TIMERS_H  // C:\cbproj\SpecLab\Timers.h not included yet ? Include it here:
# include "Timers.h"   // TIM_GetCurrentUTC() for the 'traffic log', stopwatches, etc
#endif
#ifndef  CIRCULAR_FIFO_INCLUDED
# include "CircularFifo.h" // T_CFIFO is used as thread-safe buffer for serial ports
#endif
#ifndef  _STRING_LIB_H_
# include "StringLib.h"  // DL4YHF's string library (originally designed for Microcontrollers, thus pure "C")
#endif

#if(SWI_SUPPORT_YAESU_5_BYTE_CAT) // Support the OLD UGLY "YAESU 5-BYTE-CAT" (see KA7OEI, "FT817 Meow") ?
# ifndef _YAESU_5_BYTE_H_ // Note the chicken-and-egg problem with Yaesu5Byte.h and RigControl.h :
#  include "Yaesu5Byte.h" // 'RigControl.h' *may* #include 'Yaesu5Byte.h', but not vice versa !
# endif
#endif // SWI_SUPPORT_YAESU_5_BYTE_CAT ?



/*-------------- Macros --------------------------------------------------*/

#ifndef  RIGCONTROL_SUPPORT_ICOM_UDP    // support communicating with Icom-radios via UDP (LAN,WLAN) ?
# define RIGCONTROL_SUPPORT_ICOM_UDP 0  // 1=yes (omg..); 0= HEAVENS, NO ! :o)
#endif // ndef  RIGCONTROL_SUPPORT_ICOM_UDP ?


/*-------------- Constants -----------------------------------------------*/


// Values for iRadioCtrlProtocol :
#define RIGCTRL_PROTOCOL_NONE         0
#define RIGCTRL_PROTOCOL_ICOM_CI_V    1 // supported since the early beginnings
#define RIGCTRL_PROTOCOL_KENWOOD      2 // not supported here yet
#define RIGCTRL_PROTOCOL_YAESU_5_BYTE 3 // Ugly CAT with FIVE BYTES per block, and the COMMAND in the LAST(!) byte. See Yaesu5Byte.c
#define RIGCTRL_PROTOCOL_YAESU_ASCII  4
#define RIGCTRL_PROTOCOL_ELECRAFT     5 // why did they have to (ele-)CRAFT their own ? Not supported here.
#define RIGCTRL_PROTOCOL_AD9832_I2C   6 // supported somewhere else (in SpecLab) :)
#define RIGCTRL_PROTOCOL_HAMLIB_RIGCTLD 7 // first used in the Remote CW Keyer, 2024-06. Only via TCP/IP !
//#define RIGCTRL_PROTOCOL_FLRIG_XMLRPC 8 // FLRIG has an easy-to-use GUI, but XML-RPC is sooo wasteful
//  and completely unecessary for this purpose (why not just send commands via TCP/IP ?),
//  and FLRIG always wrecked the settings in an IC-9700 -> removed again.
// 


// Values for dwRigControlFlags (bitwise combineable) :
#define RIGCTRL_FLAGS_NONE                    0x00
#define RIGCTRL_FLAG_SEND_SETTINGS_ON_STARTUP 0x01   // if not set, "read settings on startup" (but don't modify anything yet)
#define RIGCTRL_FLAG_WANT_SPECTRUM_DATA       0x02   // if set, convince the radio to periodically send SPECTRUM DATA (IC-7300)
#define RIGCTRL_FLAG_SIDETONE_WHEN_KEYED_ON_RIG 0x04 // if set, sidetone-output ON THE RIG'S USB-AUDIO-OUTPUT depends on WHO keys the radio ("sysop, locally" or "remotely")

// Values for iRigCtrlPort (to identify the "radio control" port, or any of the optional "duplicator ports"):
#define RIGCTRL_PORT_RADIO    0 // communication port that "talks to the radio"
#define RIGCTRL_PORT_AUX_COM_1 1 // 1st "additional COM" port, allows sharing the radio with other apps ...
#define RIGCTRL_PORT_AUX_COM_2 2 // 2nd "additional COM" port, " " " .. but also used INTERNALLY for the ADDITIONAL ports (in Listen-Only / Serial Port Tunnel mode)
#define RIGCTRL_PORT_AUX_COM_3 3 // 3rd "additional COM" port, for another application
#define RIGCTRL_MAX_CLIENT_PORTS  3
#define RIGCTRL_PORT_AUX_COM_LAST 3

// Values for iRigCtrlOrigin (to tell the "origin" or "flow direction" of data
// received from both sides of a 'serial port tunnel'. First used in
// AuxComPorts.c : AuxCom_RunSerialPortTunnel(), to let the RIG CONTROL MODULE
// 'eavesdrop' messages travelling between e.g. an external rig control application
// (like "FT817 Commander") and the radio itself (e.g. a Yaesu FT-817)
// over a TCP/IP tunnel provided by RCW Keyer .
#define RIGCTRL_ORIGIN_RADIO       0 // "we know the origin of data fed into a protocol analyser or packet decoder is THE RADIO"
#define RIGCTRL_ORIGIN_CONTROLLER  1 // "we know the origin of data fed into a protocol analyser or packet decoder is A CONTROLLER"
#define RIGCTRL_ORIGIN_COM_PORT_RX 2 // "we only know this was RECEIVED on a local COM port" (but no idea about what it connects to)
#define RIGCTRL_ORIGIN_COM_PORT_TX 3 // "we only know this was SENT TO a local COM port" (but no idea about what it connects to)


// Return values from most parsing functions indicate the message type (iMsgType):
#define RIGCTRL_MSGTYPE_INFO      0 // not a MESSAGE but just an INFO, e.g. an important state transition. No "Packet Data".
#define RIGCTRL_MSGTYPE_RESERVED  1 //
#define RIGCTRL_MSGTYPE_FREQUENCY_REPORT 2 // "frequency report" (from radio to user / application)
#define RIGCTRL_MSGTYPE_OP_MODE   3 // "mode data" like USB/LSB/CW/FM/AM (often unsolicited, "on change")
#define RIGCTRL_MSGTYPE_SPECTRUM  4 // "waveform data" (always unsolicited, for Icom's spectrum scope)
#define RIGCTRL_MSGTYPE_SPECTRUM_CONFIG 5 // configure / start / stop transmission of spectrum data (not spectrum data itself)
#define RIGCTRL_MSGTYPE_AUDIO     6 // "waveform data" (always unsolicited, for Icom's spectrum scope)
#define RIGCTRL_MSGTYPE_IDLE_ETC  7 // any kind of 'idle' message (extra flag to selectively REJECT them from logging)
#define RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND 8 // ANY KNOWN COMMAND (extra flag to selectively REJECT them from logging; what remains are commands that we need to add in the code)
#define RIGCTRL_MSGTYPE_UNKNOWN   9 // a well-formatted message, but the parser didn't recognize the command.
#define RIGCTRL_MSGTYPE_RADIO_ID  10    // "something to IDENTIFY the radio"
#define RIGCTRL_MSGTYPE_SET_FREQUENCY 11 // command to SET a new frequency (from user to radio)
#define RIGCTRL_MSGTYPE_RIT_XIT   12 // RIT or XIT data (often unsolicited, "on change")
#define RIGCTRL_MSGTYPE_RF_GAIN   13 // anything that affects the RF gain (preamp, attenuator, "AIP", ..)
#define RIGCTRL_MSGTYPE_SCOPE_SPAN 14 // flag for pRC->iPostponedSetMessages to send pRC->dblScopeSpan_Hz a.s.a.p. (when not waiting for a response)
#define RIGCTRL_MSGTYPE_POWER_ON_OFF 15 // Packet to turn remotely controlled equipment on/off (or similar)
#define RIGCTRL_MSGTYPE_POTI_SETTING 16 // some 'front-panel poti setting', e.g. Audio Volume, RF Power, RF Gain, CW Pitch, etc.
#define RIGCTRL_MSGTYPE_READ_METER  17 // message to read some kind of "meter" value (incl. squelch status, S-Meter, SWR-Meter, ALC-Meter, etc etc)
#define RIGCTRL_MSGTYPE_UDP_CONTROL 20 // Packet to/from Icom's UDP "Control" port (50001 by default)
#define RIGCTRL_MSGTYPE_UDP_SERIAL  21 // Packet to/from Icom's UDP "Serial" port (50002 by default), but none of the well-known CI-V message types like RIGCTRL_MSGTYPE_SPECTRUM
#define RIGCTRL_MSGTYPE_UDP_AUDIO   22 // Packet to/from Icom's UDP "Audio" port (50003 by default)
#define RIGCTRL_MSGTYPE_OTHER    31 // the message is acknowledge / response / date for 'set', read "some other parameter", etc pp.
#define RIGCTRL_MSGMASK_BASIC   0x1F // mask to selectively access the above "basic info" in e.g. RigCtrl_ModifyMsgType()
#define RIGCTRL_MSGTYPE_OK      0x20 // <- bitwise combineable (for the log). In CI-V : unspecific "OK", cmd 0xFB.  Without "Real data" (neither _TX nor _RX) : "info, all OK here."
#define RIGCTRL_MSGTYPE_NOT_OK  0x40 // <- bitwise combineable (for the log). In CI-V : unspecific "NG" / "Not Good", cmd 0xFA
#define RIGCTRL_MSGMASK_OK_NOK  0x60 // mask to selectively access the "ok/not ok" status bits

// Add new message types also in RigCtrl_MessageTypeToString() !
#define RIGCTRL_MSGMASK_FLAGS 0xFF00 // bitwise-AND mask for the following flags:
#define RIGCTRL_MSGTYPE_FLAG_TX     (1<<8) // For the log: Real "data", TRANSMITTED from controller to radio, bitwise OR-able to the above for the "CAT traffic log"
#define RIGCTRL_MSGTYPE_FLAG_RX     (1<<9) // For the log: Real "data", RECEIVED by the controller from the radio, "   "   "   "  "   ...
#define RIGCTRL_MSGMASK_FLAGS_RX_TX (RIGCTRL_MSGTYPE_FLAG_RX|RIGCTRL_MSGTYPE_FLAG_TX)
#define RIGCTRL_MSGTYPE_FLAG_EXPECTED (1<<10) // EXPECTED message (response matching earlier request, shown in GREEN colour in the traffic log)
#define RIGCTRL_MSGTYPE_FLAG_CIV    (1<<11) // for the TRAFFIC LOG: pure CI-V payload, without overhead from any "transport layer" (and no "UDP-CIV-Header")
#define RIGCTRL_MSGTYPE_FLAG_UDP    (1<<12) // not from/to a SERIAL PORT but UDP (datagram, usually Icom RS-BA compatible)
#define RIGCTRL_MSGTYPE_FLAG_ECHO   (1<<13) // "echo" from an old CI-V interface (with a SINGLE DATA LINE)
#define RIGCTRL_MSGTYPE_FLAG_ERROR  (1<<14) // <- set, for example, on response-reception-TIMEOUT.
#define RIGCTRL_MSGTYPE_FLAG_PERIODIC_POLL (1<<15) // "periodic polled" message, e.g for the S-meter. Often rejected for the LOG
#define RIGCTRL_MSGMASK_FLAGS_RW_UNKNOWN (0<<16) // no idea if the message is a READ/WRITE COMMAND/RESPONSE, or UNSOLICITED, or any of the following:
#define RIGCTRL_MSGTYPE_FLAG_READ_CMD    (1<<16) // important for e.g. RigCtrl_CIV_Server_OnReadCommand(), when STORING a message for forwarding
#define RIGCTRL_MSGTYPE_FLAG_READ_RESP   (2<<16) // important  "   "     "  ...
#define RIGCTRL_MSGTYPE_FLAG_WRITE_CMD   (3<<16) // important for e.g. RigCtrl_CIV_Server_OnWriteCommand(), when STORING a message for forwarding
#define RIGCTRL_MSGTYPE_FLAG_WRITE_RESP  (4<<16) // important  "   "     "  ...
#define RIGCTRL_MSGTYPE_FLAG_OTHER_RESP  (5<<16) // look at the bits in RIGCTRL_MSGMASK_OK_NOK ..
#define RIGCTRL_MSGTYPE_FLAG_UNSOLICITED (6<<16) // message sent by the radio without being asked (causes shock and horror for some poorly written 'libraries')
#define RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP (7<<16) // mask to extract bitfield with "READ/WRITE_COMMAND" or ..RESPONSE from iMsgType
            // '--> If neither _READ_CMD_READ_CMD nor .._WRITE_CMD is set,
            // the message isn't a COMMAND but a READ-RESPONSE, "OK" for write, "NOT OK", or an UNSOLICITED REPORT.
            // Then RIGCTRL_MSGTYPE_OK or RIGCTRL_MSGTYPE_NOT_OK indicate what the stored/forwarded message is.
#define RIGCTRL_MSGTYPE_FLAG_IS_ALIAS    (1<<20) // "this message (e.g. 'Read SELECTED VFO frequency') is just an other method to access a parameter we usually read/write through a DIFFERENT command (e.g. 'Read VFO Frequency')"
#define RIGCTRL_MSGTYPE_FLAG_SUGGEST_NEW_HEADLINE_FOR_TRAFFIC_LOG (1<<21) // *suggestion* for the GUI to emit another line generated by RigCtrl_GetHeadlineForTrafficLog()




// Possible operating modes, not protocol-specific, and different from CI-V :
//   (a less ambiguous name would be "modulation", but "op mode" includes
//   "wide" or "narrow" settings or even "reverse", for historic reasons)
#define RIGCTRL_OPMODE_UNKNOWN 0  // possible values for iOpMode ...
#define RIGCTRL_OPMODE_CW      1  //  BITWISE COMBINEABLE to use this for a "list of supported modes", too
#define RIGCTRL_OPMODE_LSB     2  // To convert all these code into "Icom equivalents",
#define RIGCTRL_OPMODE_USB     4  // use
#define RIGCTRL_OPMODE_AM      8
#define RIGCTRL_OPMODE_FM      16
#define RIGCTRL_OPMODE_FM_WIDE 32
#define RIGCTRL_OPMODE_RTTY    64
#define RIGCTRL_OPMODE_PSK     128
#define RIGCTRL_OPMODE_DIGITAL_VOICE 256 // "Digital Voice", presumably D-Star, not DMR
#define RIGCTRL_OPMODE_DIGITAL_DATA  512 // "DD" seen in the IC-9700 CI-V Reference Guide (no idea what "DD" really stands for.. 128 kbp "Digital Data" on 23 cm ?)
#define RIGCTRL_OPMODE_RESERVED_1   1024
#define RIGCTRL_OPMODE_RESERVED_2   2048
#define RIGCTRL_OPMODE_ALL          4095 // combination of "all" modes (including those we don't even know yet)
#define RIGCTRL_OPMODE_MASK_TO_STRIP_FLAGS 4095
#define RIGCTRL_OPMODE_REVERSE     4096 // bitwise combineable with some op-modes (CW, RTTY, PSK)
#define RIGCTRL_OPMODE_NARROW      8192 // bitwise combineable with some op-modes ...
#define RIGCTRL_OPMODE_VERY_NARROW 16384

// Possible values for T_RigCtrlInstance.iParameterPollingState (for CI-V sequencing):
#define RIGCTRL_POLLSTATE_PASSIVE             0 // initial state: have neither READ nor WRITTEN any setting from/to the remote rig
#define RIGCTRL_POLLSTATE_START_RD            1 // begin reading all we need to know about/from the remote rig
#define RIGCTRL_POLLSTATE_READ_RADIO_ID       2 // (try to) read the radio's "default" ID, as a connection test and to check its type
#define RIGCTRL_POLLSTATE_PARAMS_FOR_RIGCTL   3 // (try to) read the parameters required for RigControl.c itself
#define RIGCTRL_POLLSTATE_TX_BAND_EDGES       4 // (try to) retrieve all of the radio's TRANSMIT BANDS (with edge frequencies)
#define RIGCTRL_POLLSTATE_BAND_STACKING_REGS  5 // (try to) read out the 'band stacking registers' (special VFO memories)
#define RIGCTRL_POLLSTATE_PARAMS_FOR_HAMLIB   6 // (try to) read a few more params used by HamlibServer.c
#define RIGCTRL_POLLSTATE_CONFIGURE_SPECTRUM  7 // (try to) activate transmission of "spectrum scope waveform data"
#define RIGCTRL_POLLSTATE_DONE               10 // finished polling "all we need to know" about the radio
#define RIGCTRL_POLLSTATE_FIND_VFO_FOR_FREQ  11 // special state for e.g. IC-9700 to "find" the suitable VFO for a given band/frequency
#define RIGCTRL_POLLSTATE_AFTER_VFO_SWITCH   12 // read certain parmameters again after e.g. exchanging "VFO A" and "VFO B"
#define RIGCTRL_POLLSTATE_TURN_RIG_ON        13 // special service for modern rigs that can be turned on / off via CAT (e.g. IC-7300 via USB/CI-V)
#define RIGCTRL_POLLSTATE_WAIT_TURN_ON       14 // waiting for an "OK"-response after the command to "turn on"
#define RIGCTRL_POLLSTATE_TURN_RIG_OFF       15 // send command to turn the radio off (as soon as possible)
#define RIGCTRL_POLLSTATE_WAIT_TURN_OFF      16 // waiting for the rig to turn off
#define RIGCTRL_POLLSTATE_TURNED_OFF         17 // after RIGCTRL_POLLSTATE_TURN_RIG_OFF / WAIT_TURN_OFF, the only allowed command is to "turn on"
        // When adding new POLLSTATEs here, also add 'cases' for them in ...
        //     (1) RigCtrl_ParameterPollingStateToString()
        //     (2) RigCtrl_Handler()
        //     (3) Yaesu5Byte_Handler()
        //     (4) and possibly in similar modules, implementing other 'Rig Control' protocols
        // Protocol-specific modules like 'Yaesu5Byte.c' return these codes
        // from their 'periodically called handlers', e.g. Yaesu5Byte_Handler().
        // See also: Yaesu5Byte_

// CI-V default addresses, also used internally to identify a certain radio model .
//  Hint: Many of these are 'exposed' via RigCtrl_RadioInfo_CIV[] .
#define RIGCTRL_DEF_ADDR_NO_REMOTE_CTRL -1   // special dummy for .iDefaultAddress or CwKeyer_Config.iRadioCIVAddress if there shall be NO REMOTE CONTROL at all
#define RIGCTRL_DEF_ADDR_AUTO_DETECT    0    // special dummy for .iDefaultAddress or CwKeyer_Config.iRadioCIVAddress for 'auto-detect'
#define RIGCTRL_DEF_ADDR_ICOM_LOWEST    0x01
#define RIGCTRL_DEF_ADDR_ICOM_HIGHEST   0xFF
#define RIGCTRL_DEF_ADDR_IC_271         0x20 // VHF Allmode
#define RIGCTRL_DEF_ADDR_IC_275         0x10 // VHF Allmode
#define RIGCTRL_DEF_ADDR_IC_375         0x12 // 220MHz Allmode
#define RIGCTRL_DEF_ADDR_IC_471         0x22 // UHF Allmode
#define RIGCTRL_DEF_ADDR_IC_475         0x14 // UHF Allmode
#define RIGCTRL_DEF_ADDR_IC_575         0x16 // 10m & 6m Allmode
#define RIGCTRL_DEF_ADDR_IC_703         0x68 // HF Allmode QRP
#define RIGCTRL_DEF_ADDR_IC_705         0xA4 // HF/VHF/UHF portable QRP DSP transceiver, spectrum via CI-V, CI-V via USB & WLAN
#define RIGCTRL_DEF_ADDR_IC_905         0xAC // VHF/UHF/SHF portable QRP DSP transceiver, spectrum via CI-V, CI-V via USB & WLAN
#define RIGCTRL_DEF_ADDR_IC_706         0x48 // HF/VHF Allmode
#define RIGCTRL_DEF_ADDR_IC_706MkII     0x4E // HF/VHF Allmode
#define RIGCTRL_DEF_ADDR_IC_706MkIIG    0x58 // HF/VHF/UHF Allmode
#define RIGCTRL_DEF_ADDR_IC_707         0x3e // HF Allmode
#define RIGCTRL_DEF_ADDR_IC_718         0x5E // HF Allmode
#define RIGCTRL_DEF_ADDR_IC_725         0x28 // HF Allmode
#define RIGCTRL_DEF_ADDR_IC_726         0x30 // HF/6m Allmode
#define RIGCTRL_DEF_ADDR_IC_728         0x38 // HF Allmode
#define RIGCTRL_DEF_ADDR_IC_729         0x3A // HF/6m Allmode
#define RIGCTRL_DEF_ADDR_IC_735         0x04 // HF Allmode (possibly the first radio with CI-V ever made)
#define RIGCTRL_DEF_ADDR_IC_736         0x40 // HF/6m Allmode
#define RIGCTRL_DEF_ADDR_IC_737         0x3C // HF Allmode
#define RIGCTRL_DEF_ADDR_IC_738         0x44 // HF Allmode
#define RIGCTRL_DEF_ADDR_IC_746         0x56 // HF/VHF Allmode
#define RIGCTRL_DEF_ADDR_IC_751A        0x1C // HF Allmode
#define RIGCTRL_DEF_ADDR_IC_756         0x50 // HF/6m Allmode
#define RIGCTRL_DEF_ADDR_IC_756Pro      0x5C // HF/6m Allmode
#define RIGCTRL_DEF_ADDR_IC_756ProII    0x64 // HF/6m Allmode
#define RIGCTRL_DEF_ADDR_IC_756ProIII   0x6e // HF/6m Allmode
#define RIGCTRL_DEF_ADDR_IC_761         0x1E // HF Allmode
#define RIGCTRL_DEF_ADDR_IC_765         0x2C // HF Allmode
#define RIGCTRL_DEF_ADDR_IC_775         0x46 // HF Allmode
#define RIGCTRL_DEF_ADDR_IC_78          0x62 // HF Allmode
#define RIGCTRL_DEF_ADDR_IC_7000        0x70 // HF/VHF/UHF Allmode
#define RIGCTRL_DEF_ADDR_IC_7100        0x88 // HF/VHF/UHF
#define RIGCTRL_DEF_ADDR_IC_7200        0x76 // HF Allmode
#define RIGCTRL_DEF_ADDR_IC_7300        0x94 // HF/6m/4m, full DSP, spectrum data via CI-V)
#define RIGCTRL_DEF_ADDR_IC_9700        0xA2 // 2m/70cm/23cm, full DSP transceiver, spectrum via CI-V, CI-V via USB or LAN)
#define RIGCTRL_DEF_ADDR_IC_7400        0x66 // same ID as IC-746Pro ?
#define RIGCTRL_DEF_ADDR_IC_7600        0x7A // HF Allmode (non-DSP, no spectrum data via CI-V)
#define RIGCTRL_DEF_ADDR_IC_7610        0x98 // HF/6m Allmode, full DSP, spectrum data,..)
#define RIGCTRL_DEF_ADDR_IC_7700        0x74 // HF Allmode
#define RIGCTRL_DEF_ADDR_IC_7760        0xB2 // HF/6m Allmode, anno 2025
#define RIGCTRL_DEF_ADDR_IC_7800        0x6A // HF Allmode
#define RIGCTRL_DEF_ADDR_IC_7851        0x8F // HF/6m Allmode, IF DSP, spectrum via CI-V
#define RIGCTRL_DEF_ADDR_IC_781         0x26 // HF Allmode
#define RIGCTRL_DEF_ADDR_IC_820         0x42 // VHF/UHF Allmode
#define RIGCTRL_DEF_ADDR_IC_821         0x4C // VHF/UHF Allmode
#define RIGCTRL_DEF_ADDR_IC_910         0x60 // VHF/UHF Allmode
#define RIGCTRL_DEF_ADDR_IC_970         0x2E // VHF/UHF Allmode
#define RIGCTRL_DEF_ADDR_IC_1271        0x24 // SHF
#define RIGCTRL_DEF_ADDR_IC_1275        0x18 // SHF
#define RIGCTRL_DEF_ADDR_IC_R10         0x52 // Handheld scanner
#define RIGCTRL_DEF_ADDR_IC_R20         0x6C // Handheld scanner
#define RIGCTRL_DEF_ADDR_IC_R71         0x1A // General Coverage Receiver
#define RIGCTRL_DEF_ADDR_IC_R72         0x32 // General Coverage Receiver
#define RIGCTRL_DEF_ADDR_IC_R75         0x5A // General Coverage Receiver
#define RIGCTRL_DEF_ADDR_IC_R7000       0x08 // General Coverage Receiver
#define RIGCTRL_DEF_ADDR_IC_R7100       0x34 // General Coverage Receiver
#define RIGCTRL_DEF_ADDR_IC_R8500       0x4A // Gen.Cov.RX
#define RIGCTRL_DEF_ADDR_IC_R8600       0x96 // DSP receiver, supports spectrum data via CI-V
#define RIGCTRL_DEF_ADDR_IC_R9000       0x2A // General Coverage Receiver
#define RIGCTRL_DEF_ADDR_IC_R9500       0x72 // General Coverage Receiver
#define RIGCTRL_DEF_ADDR_IC_RX7         0x78 // Handheld scanner

// Yaesu radios with the terrible binary "5-byte-command" CAT protocol, e.g. FT-817,
// cannot be identified by reading their 'Default Address'. The following
// constants are only used in the GUI, values stored in the configuration.
// Since ICOM RADIOS use eight-bit 'Radio'- and 'Controller' addresses in CI-V,
// the following values (0x3nn = "third choice" ) won't collide with any of the codes above.
#define RIGCTRL_DEF_ADDR_UNKNOWN_YAESU 0x300
#define RIGCTRL_DEF_ADDR_YAESU5B_LOWEST  0x300
#define RIGCTRL_DEF_ADDR_YAESU5B_HIGHEST 0x3FF
#define RIGCTRL_DEF_ADDR_YAESU_FT_817    0x317

// "Unified" parameter numbers, used within RigControl.c but NOT (always) by Icom.
// Some of the parameters, transceiver settings, or current mode can be read
//   can be read via CI-V, e.g. command 0x1C for "transceiver status (RX/TX)".
// Some of the RIG-specific 4-digit subcodes for Icom's command 0x1A 0x05
//   can also be translated into one of these.
// Note: When adding new PNs here, also add them in RigCtrl_ParameterInfo[] !
#define RIGCTRL_PN_UNKNOWN                   0
#define RIGCTRL_PN_TRANSCEIVER_ID            1 // T_RigCtrlInstance.iDefaultAddress : "Transceiver ID" alias "Default CI-V Address" : Icom: cmd 0x19, subcommand 0x00.
                                               // e.g.
#define RIGCTRL_PN_FREQUENCY                 2 // .dblVfoFrequency, alias "Selected VFO Freq",  various CI-V commands !
#define RIGCTRL_PN_OP_MODE                   3 // .iOpMode (USB/LSB/CW..), sometimes decorated with a "filter number" (e.g. for "CWN" or even "CWNN" instead of just "CW")
#define RIGCTRL_PN_FILTER_BANDWIDTH          4 // (IF-) filter bandwidth in Hz. Icom: cmd 0x1A sub 0x03.
#define RIGCTRL_PN_DATA_MODE                 5 // 0=no data mode, 1=data mode.  Icom: cmd 0x1A sub 0x06.
#define RIGCTRL_PN_SEL_VFO_FREQUENCY         6 // alias for .dblVfoFreq, but a different cmd: 0x25 sub 0x00.
#define RIGCTRL_PN_SEL_VFO_OP_MODE           7 // alias for .iOpMode, but a different CI-V cmd: 0x26 sub 0x00.
#define RIGCTRL_PN_UNSEL_VFO_FREQUENCY       8 // .dblUnselVfoFreq : polled by WFView, cmd 0x25 sub 0x01.
#define RIGCTRL_PN_UNSEL_VFO_OP_MODE         9 // .iUnselVfoOpMode : polled by WFView, cmd 0x26 sub 0x01.

#define RIGCTRL_PN_TRANSMIT_REQUEST         10 // 0=WE want the rig to receive, 1=WE want the rig to transmit.
                                               //   With CI-V, may send cmd 0x1C, sub 0x00 + data (00=RX, 01=TX) = "Send the transceiver's status".
#define RIGCTRL_PN_TRANSMITTING             11 // 0=receiving, 1=transmitting. CI-V: cmd 0x1C, sub 0x00, no data = "Read the transceiver's status".
                 // Note that when the rig's local operator presses the PTT on the hand mic,
                 //   TRANSMIT_REQUEST may be 0 (from our point of view)
                 //     while TRANSMITTING is 1 - that's NOT an error, and causes
                 //     the remote CW Keyer's GUI to indicate 'On Air' even if
                 //     it's not 'requested' to transmit remotely.
                 // Since 2025-09-18, 'ON AIR' in *orange* colour
                 // indicates being 'on air' on behalf of someone else, including
                 // the 'sysop' keying his own radio locally, w/o RCW Keyer.
                 // To achieve that, RIGCTRL_PN_TRANSMITTING -> T_RigCtrlInstance.iTransmitting
                 // is polled PERIODICALLY, along with other parameters, by the KEYER GUI .
#define RIGCTRL_PN_CIV_TRANSCEIVE_MODE      12 // 0=UNSOLICITED REPORTS diabled, 1=enabled.
                                               //    In Icom geek speak, this is "CI-V Transceive" mode.
#define RIGCTRL_PN_ATU_ENABLED              17 // 0=automatic antenna tuner disabled, 1=enabled. Icom: 0x017 0x01
#define RIGCTRL_PN_XFC_PRESSED              18 // 0="transmit frequency monitor" off, 1=on. Icom: 0x17 0x02.
                 // Guess this is what others call the "XFC button". Useful in SPLIT mode to monitor the own TX-frequency.
#define RIGCTRL_PN_TRANSMIT_FREQ            19 // Icom: 0x17 subcommand 0x03. They call it "Read transmit frequency".
                 // Guess the response contains the frequency displayed on the radio while pressing the 'XFC' button.
#define RIGCTRL_PN_REPEATER_TONE_FREQ       25 // Repeater Tone frequency in Hertz
#define RIGCTRL_PN_TONE_SQUELCH_FREQ        26 // Tone Squelch frequency in Hertz

#define RIGCTRL_PN_NUM_TX_BANDS             30 // Number of TRANSMIT bands supported by connected radio.
#define RIGCTRL_PN_TX_BAND_EDGES            31 // dummy represending T_RigCtrlFrequencyRange TxBandEdges[RIGCTRL_PN_TX_BAND_EDGES]
#define RIGCTRL_PN_BAND_STACKING_REGS       32 // dummy represending T_RigCtrlFreqMemEntry BandStackingRegs[RIGCTRL_NUM_BAND_STACKING_REGS]



        //--- SPECTRUM SCOPE related (with CI-V, exchanged via command 0x27) ---
#define RIGCTRL_PN_SCOPE_ON_OFF             40 // pRC->iScopeOnOff
#define RIGCTRL_PN_SCOPE_MODE               41 // pRC->iScopeMode
#define RIGCTRL_PN_SCOPE_SPAN               42 // pRC->dblScopeSpan_Hz
#define RIGCTRL_PN_SCOPE_CENTER_FREQUENCY   43 // pRC->dblSpectrumCenterFreq_Hz ( there is no CI-V equivalent for this, but we use it for CwNet.c, and let the remote server decide how to modify the scope center frequency without QSY-ing )
#define RIGCTRL_PN_SCOPE_EDGE_NR            44 // pRC->iScopeEdgeNumber
#define RIGCTRL_PN_SCOPE_HOLD_FLAG          45 // pRC->iScopeHoldFlag
#define RIGCTRL_PN_SCOPE_ATTEN              46 // pRC->dblScopeAttenuator_dB
#define RIGCTRL_PN_SCOPE_REF_LEVEL          47 // pRC->dblScopeRefLevel_dB
#define RIGCTRL_PN_SCOPE_SPEED              48 // pRC->iScopeSpeed (RIGCTRL_SCOPE_SPEED_FAST/MID/SLOW)
#define RIGCTRL_PN_SCOPE_DURING_TX          49 // pRC->iScopeDuringTX
#define RIGCTRL_PN_SCOPE_CENTER_TYPE        50 // pRC->iScopeCenterType
#define RIGCTRL_PN_SCOPE_VBW                51 // pRC->iScopeVBW ("visual bandwidth", kind of smoothing/averaging)
#define RIGCTRL_PN_SCOPE_FIXED_EDGES        52 // pRC->ScopeFreqRange[][] (a TWO-DIMENSIONAL ARRAY in Icom rigs)
#define RIGCTRL_PN_SCOPE_MARKER_POS_TYPE    53 // pRC->iScopeMarkerPosType
#define RIGCTRL_PN_SCOPE_DATA               54 // just a dummy for the log
#define RIGCTRL_PN_WATERFALL_SPEED          55 // pRC->iWaterfallSpeed (values also RIGCTRL_SCOPE_SPEED_FAST/MID/SLOW,
                                               // even though not compatible with what Icom uses internally)


        //--- Various "poti" settings (with CI-V, exchanged via command 0x14; ---
        //    from server to client, sent in a T_RigCtrl_PotiReport :
#define RIGCTRL_PN_AUDIO_VOLUME_PERCENT     60 // pRC->iAudioVolume_Percent
#define RIGCTRL_PN_RF_POWER_SETTING_PERCENT 61 // pRC->iRFPowerSetting_Percent;  "setting" to avoid confusion with the POWER METER MEASUREMENT !
#define RIGCTRL_PN_RF_GAIN_PERCENT          62 // pRC->iRFGain_Percent
#define RIGCTRL_PN_SQUELCH_LEVEL_PERCENT    63 // pRC->iSquelchLevel_Percent
#define RIGCTRL_PN_NOISE_BLANKER_PERCENT    64 // pRC->iNoiseBlanker_Percent;
#define RIGCTRL_PN_NOISE_REDUCTION_PERCENT  65 // pRC->iNoiseReduction_Percent
#define RIGCTRL_PN_PASSBAND_TUNING_POS1     66 // pRC->iPassbandTuningPos1_Percent; "inner ring" of "TWIN PBT", 50 % = center
#define RIGCTRL_PN_PASSBAND_TUNING_POS2     67 // pRC->iPassbandTuningPos2_Percent; "outer ring" of "TWIN PBT", 50 % = center
#define RIGCTRL_PN_CW_PITCH_HZ              68 // pRC->iCWPitch_Hz; here: in Hertz, not the strange CI-V internal unit
#define RIGCTRL_PN_MIC_GAIN_PERCENT         69 // pRC->iMicGain_Percent
#define RIGCTRL_PN_KEYER_SPEED_WPM          70 // pRC->iKeyerSpeed_WPM;  applies to the rig's BUILT-IN keyer
#define RIGCTRL_PN_NOTCH_POS_PERCENT        71 // pRC->iNotchPos_Percent;
#define RIGCTRL_PN_COMP_SETTING_PERCENT     72 // pRC->iCompSetting_Percent; audio compressor LEVEL SETTING (not "measurement")
#define RIGCTRL_PN_BREAK_IN_DELAY_PERCENT   73 // pRC->iBreakInDelay_Percent;
#define RIGCTRL_PN_MONITOR_GAIN_PERCENT     74 // pRC->iMonitorGain_Percent;
#define RIGCTRL_PN_VOX_GAIN_PERCENT         75 // pRC->iVoxGain_Percent;
#define RIGCTRL_PN_ANTI_VOX_GAIN_PERCENT    76 // pRC->iAntiVoxGain_Percent;
#define RIGCTRL_PN_BRIGHTNESS_PERCENT       77 // pRC->iBrightness_Percent;

        //------ Various "meter" readings (with CI-V, exchanged via command 0x15) -----
        //       Not to be confused with the "poti settings" further above.
        //       A "meter" changes its value without intervention from the user.
        //       Thus, if the radio supports 'meter readings', the GUI(!)
        //       will periodically read them when time permits
        // - see TKeyerMainForm::Timer1Timer() -> RigCtrl_QueueUpCmdToReadUnifiedPN() .
#define RIGCTRL_PN_SQUELCH_STATUS           90 // pRC->iSquelchStatus;  (0 or 1) from 0x1501
#define RIGCTRL_PN_S_METER_LEVEL_DB         91 // pRC->iSMeterLevel_dB; here: "decibel over S0",  not the strange CI-V internal unit
#define RIGCTRL_PN_POWER_METER_PERCENT      92 // pRC->iPowerMeterLevel_Percent; not the strange CI-V internal unit
#define RIGCTRL_PN_SWR_METER_VALUE          93 // pRC->dblSWRMeterValue;  dimensionless, usual range 1.0 (perfect) to 3.0 (poor)
#define RIGCTRL_PN_ALC_METER_LEVEL          94 // pRC->iALCMeterLevel;  here: 0 = min, 100 = max, not the strange CI-V internal unit
#define RIGCTRL_PN_COMP_METER_LEVEL_DB      95 // pRC->iCompMeterLevel_dB; here: in decibel, 0=no compression, 30 dB = maximum for IC-7300)
#define RIGCTRL_PN_SUPPLY_VOLTAGE_mV        96 // pRC->iSupplyVoltage_mV;  here: in millivolts, not the strange CI-V internal unit
#define RIGCTRL_PN_DRAIN_CURRENT_mA         97 // pRC->iDrainCurrent_mA;   here: in milliamps,  not the strange CI-V internal unit
#define RIGCTRL_PN_TEMPERATURE_CELSIUS      98 // pRC->iTemperature_Celsius; one fine day Icom will discover they've FORGOTTEN this parameter
#define RIGCTRL_PN_ADC_OVERFLOW_STATUS      99 // pRC->iAdcOverflowStatus; cmd 0x15 sub 0x07 in an IC-7300

        //-- extended Preamp, AGC speed, Noise blanker, Audio peak filter,
        // noise reduction, auto notch, etc. With CI-V, via command 0x16) -----
#define RIGCTRL_PN_PREAMP_SETTING          100 // CI-V : 0x16 0x02  IC-9700: 0=internal+external OFF, 1=internal, 2=external, 3=BOTH preamps on
                                               //                   IC-7300: 0=preamp off, 1=preamp low gain, 2=preamp high gain, 3=NOT ALLOWED
#define RIGCTRL_PN_AGC_SPEED               101 // CI-V : 0x16 0x12  1=fast 2=mid 3=slow  . Other radios: 0 = off (MANUAL gain control)
#define RIGCTRL_PN_NOISE_BLANKER_ON        102 // CI-V : 0x16 0x22  0=off  1=on
#define RIGCTRL_PN_NOISE_REDUCTION_ON      103 // CI-V : 0x16 0x40  0=off  1=on
#define RIGCTRL_PN_AUTO_NOTCH_ON           104 // CI-V : 0x16 0x41  0=off  1=on
#define RIGCTRL_PN_RPTR_TONE_ON            105 // CI-V : 0x16 0x42  0=off  1=on
#define RIGCTRL_PN_TONE_SQUELCH_ON         106 // CI-V : 0x16 0x43  0=off  1=on
#define RIGCTRL_PN_COMP_ON                 107 // CI-V : 0x16 0x44  0=off  1=on
#define RIGCTRL_PN_MONITOR_ON              108 // CI-V : 0x16 0x45  0=off  1=on
#define RIGCTRL_PN_VOX_ON                  109 // CI-V : 0x16 0x46  0=off  1=on
#define RIGCTRL_PN_BK_IN_MODE              110 // CI-V : 0x16 0x47  0=off 1=semi-BK 2=full-BK
#define RIGCTRL_PN_MANUAL_NOTCH_ON         111 // CI-V : 0x16 0x48  0=off  1=on
#define RIGCTRL_PN_MANUAL_NOTCH_WIDTH      112 // CI-V : 0x16 0x57  0=wide 1=mid 2=narrow
#define RIGCTRL_PN_TWIN_PEAK_FILTER_ON     113 // CI-V : 0x16 0x4F  0=off  1=on (only for standard AMATEUR RADIO RTTY)
#define RIGCTRL_PN_DIAL_LOCK_ON            114 // CI-V : 0x16 0x50  0=off  1=on
#define RIGCTRL_PN_DSP_SHARP_SOFT          115 // CI-V : 0x16 0x56  0=sharp 1=soft
#define RIGCTRL_PN_SSB_TX_BANDWIDTH        116 // CI-V : 0x16 0x58  0=wide 1=mid 2=narrow (the bandwidths are configurable in an extra commands)
#define RIGCTRL_PN_SUBBAND_ON              117 // CI-V : 0x16 0x59  0=off  1=dual watch / subband ON
#define RIGCTRL_PN_SATELLITE_MODE          118 // CI-V : 0x16 0x5A  0=off  1=on
#define RIGCTRL_PN_IMPROVED_IP3            119 // CI-V : 0x16 0x65  0=off  1=on. Known as "IP+", "improved IP3", maybe just another "gain stage off", or ADC dithering ON. We don't care.

        //-- "Special" settings, audio equalizer, connector settings, etc -----
        // Many of these exchanged via command 0x1A with an awful number of subcodes.
        // Especially annoying: Command 0x1A subcommand 0x05 is often followed
        //                      by a MODEL-SPECIFIC 4-digit sub-subcode
        //   - see RigCtrl_GetCrazyModelDependingSubSubcodeForCmd1A_05() !
#define RIGCTRL_PN_SSB_RX_HPF_LPF          130
#define RIGCTRL_PN_SSB_RX_BASS_LEVEL       131
#define RIGCTRL_PN_SSB_RX_TREBLE_LEVEL     132
#define RIGCTRL_PN_SSB_CW_SYNC_TUNING      133 // "SSB/CW synchronous tuning". In IC-7300, cmd 0x1A sub 5 PN #0052. Used by RS-BA1.
#define RIGCTRL_PN_CALIBRATION_MARKER      134 // "Calibration marker".  In IC-7300, cmd 0x1A sub 5 PN #0057. Used by RS-BA1.
#define RIGCTRL_PN_AF_OUTPUT_LEVEL_TO_USB  135 // .. or "to ACC or USB". In IC-7300, cmd 0x1A sub 5 PN #0060
#define RIGCTRL_PN_REFERENCE_FREQ_OFFSET   136 // .iReferenceFrequencyOffset.    IC-7300: 0x1A 0x05 0058
#define RIGCTRL_PN_AF_IF_OUT_SELECTOR      137 // .iAF_IF_OutputSelector (ACC,USB). IC-7300: 0x1A 0x05 0059
#define RIGCTRL_PN_SQUELCH_FOR_AF_ON_USB   138 // "Squelch function for the AF signal output to ACC/USB"
#define RIGCTRL_PN_SIDETONE_ON_USB_AUDIO   139 // .iSidetone_on_USB: "Beep and speech output setting to ACC/USB" (not only "signal beeps" but also the CW SIDETONE, at least in an IC-7300)
                // '--> Note: The IC-7300 has a single command to control what they call "Beeps" on both ACC and USB.
                //            The IC-9700 has separate commands for ACC, USB, and LAN(!).

#define RIGCTRL_PN_IF_OUTPUT_LEVEL_TO_ACC    140 // "IF signal output level to ACC/USB" (0..255 for 100%)
#define RIGCTRL_PN_MOD_INPUT_LEVEL_FROM_ACC  141 // "MOD input level from ACC" (0..255 for 100%)
#define RIGCTRL_PN_MOD_INPUT_LEVEL_FROM_USB  142 // "MOD input level from USB" (0..255 for 100%)
#define RIGCTRL_PN_MOD_IN_CONN_FOR_DATA_OFF  143 // "MOD input connector during DATA OFF" (0=MIC, 1=ACC, 2=MIC/ACC, 3=USB)
#define RIGCTRL_PN_MOD_IN_CONN_FOR_DATA      144 // "MOD input connector during DATA" (0=MIC, 1=ACC, 2=MIC/ACC, 3=USB)
#define RIGCTRL_PN_CIV_TRANSCEIVE          150
#define RIGCTRL_PN_CIV_ADDR_USB_TO_REMOTE  151
#define RIGCTRL_PN_CIV_OUTPUT_FOR_ANT      152 // Icom: 0x17 sub 4. They call it "CI-V output for ANT".
#define RIGCTRL_PN_CIV_UNLINK_FROM_REMOTE  153
#define RIGCTRL_PN_CIV_USB_ECHO            154 // "CI-V USB Echo Back" : 0075 in IC-7300, 0116 in IC-7610
#define RIGCTRL_PN_CIV_PTT_CONTROL_LINE    155 // .iCW_PTTControlLineOnUSB; IC-7300: 0x1A 0x05 #0078
#define RIGCTRL_PN_CIV_CW_KEYING_LINE      156 // .iCW_KeyingLineOnUSB;     IC-7300: 0x1A 0x05 #0079
#define RIGCTRL_PN_CIV_RTTY_KEYING_LINE    157 // .iRTTY_KeyingLineOnUSB;   IC-7300: 0x1A 0x05 #0080


#define RIGCTRL_PN_IP_PLUS                 160 // IC-7300 : 0x1A sub 0x07 (totally different in IC-7610)
#define RIGCTRL_PN_NTP_SERVER_ACCESS       161 // IC-7610 : 0x1A sub 0x07 (!)
#define RIGCTRL_PN_NTP_ACCESS_RESULT       162 // IC-7610 : 0x1A sub 0x08
#define RIGCTRL_PN_AF_MUTE                 163 // IC-7610 : 0x1A sub 0x09

        //-- Even more settings (in CI-V, controlled via command 0x0E..0x10, etc) --
#define RIGCTRL_PN_SCAN_MODE               170  // Icom: 0x0E sub 0
#define RIGCTRL_PN_SPLIT_MODE              171  // Icom: 0x0F (NO subcodes here)
#define RIGCTRL_PN_TUNING_STEP             172  // Icom: 0x10 (NO subcodes here). Note: T_RigCtrlInstance.iTuningStep_Hz uses HERTZ. Icom's command does NOT.
#define RIGCTRL_PN_ATTENUATOR_DB           173  // Icom: 0x11 (NO subcodes here)

        // "RIT" / "XIT" related .. even WSJT-X wanted to play with these via Hamlib:
#define RIGCTRL_PN_RIT_FREQ    180 // 'Receiver Incremental Tuning' in Hertz.    IC-7300: Cmd 0x21, Sub 0x00
#define RIGCTRL_PN_RIT_ENABLED 181 // 'Receiver Incremental Tuning' ON/OFF.      IC-7300: Cmd 0x21, Sub 0x01
#define RIGCTRL_PN_XIT_ENABLED 182 // 'Transmitter Incremental Tuning' ON/OFF.   IC-7300: Cmd 0x21, Sub 0x02
#define RIGCTRL_PN_XIT_FREQ    183 // 'Transmitter Incremental Tuning' in Hertz. IC-7300: alias "delta TX", but:
        // (accidentally found in ?/Hamlib/SOURCE/rigs/icom/icom.c after wasting an hour on this):
        // > The Icom rigs have only one register for both RIT and Delta TX.
        // > you can turn one or both on -- but both end up just being in sync.
        // OMG. Who made this crazy decision ?

// RigControl error codes (actually, many functions return a NEGATVE integer when something went wrong,
//                         but the error codes defined below are NON-NEGATIVE.
// The first few codes may be compatible with hamlib, but don't take this for granted.
#define RIGCTRL_ERROR_OK    0  // No error, operation completed sucessfully
#define RIGCTRL_ERROR_INVALID_PARAM   1
#define RIGCTRL_ERROR_INVALID_CONFIG  2
#define RIGCTRL_ERROR_OUT_OF_MEMORY   3 // here, also used when a string buffer is too short for a response
#define RIGCTRL_ERROR_NOT_IMPLEMENTED 4 // "function not implemented, but will be" (o-tone Hamlib)
#define RIGCTRL_ERROR_TIMEOUT         5 // communication timed out
#define RIGCTRL_ERROR_INPUT_OUTPUT    6 // IO error, including open failed
#define RIGCTRL_ERROR_INTERNAL        7 // "Internal Hamlib error, huh!"
#define RIGCTRL_ERROR_PROTOCOL        8 // Protocol error
#define RIGCTRL_ERROR_CMD_REJECTED    9 // Command rejected by the rig
#define RIGCTRL_ERROR_TRUNCATED      10 // Command performed, but argument truncated
#define RIGCTRL_ERROR_NOT_AVAILABLE  11 // function not available (for the selected rig?)
#define RIGCTRL_ERROR_NOT_TARGETABLE 12 // "VFO not targetable" (?)
#define RIGCTRL_ERROR_BUS            13 // Error talking on the bus
#define RIGCTRL_ERROR_RIG_BUS_BUSY   14 // Collision on the bus
#define RIGCTRL_ERROR_INVALID_POINTER 15 // NULL RIG handle or any invalid pointer parameter in get arg
#define RIGCTRL_ERROR_INVALID_VFO    16 // Invalid VFO
#define RIGCTRL_ERROR_OUT_OF_RANGE   17 // "Argument out of domain of func"
#define RIGCTRL_ERROR_RESERVE_18     18
#define RIGCTRL_ERROR_RESERVE_19     19
        // Above: error codes compatible with Hamlib / "rigctl" / "rigctld" .
        // Below: error codes specifically for the Remote CW Keyer / RigControl.c :
#define RIGCTRL_ERROR_CMD_IN_PROGRESS 20 // returned by KeyerGUI_ExecuteCommand() if a command involves a lengthy network operation,
           // and the response didn't arrive within a few hundred milliseconds (would block the GUI for too long)



// Scaling units / physical units for the DISPLAY. Only defined HERE
//  (in DL4YHF's RigControl/RigControl.h) if not already defined
//   in FFT_FILTER_DLL.H or UTILITY1.H :
#ifndef  SCALE_UNIT_BLANK
# define SCALE_UNIT_BLANK   -2  // for UTL_FloatToTechNotation(): tech exponent but a BLANK unit (appended as string by caller)
# define SCALE_UNIT_NONE    -1
# define SCALE_UNIT_UNKNOWN  0  // in FiltCtrl.cpp : "amplitude unit as currently configured for the display" (dB or linear scale)
  // Voltages, Power units ...
# define SCALE_UNIT_NORM_V   1  // normalized ADC units (voltage), +/- 1.0  for full A/D converter input range
# define SCALE_UNIT_PERCENT  2  // linear, reference is "full scale" (100 % = ADC clipping)
# define SCALE_UNIT_VOLT     3  // voltage (V), 'calibrated' using UTL_dblAdcMaxInputVoltage
# define SCALE_UNIT_WATT     4  // power (W), 'calibrated' with UTL_dblAdcMaxInputVoltage + UTL_dblAdcInputImpedance
# define SCALE_UNIT_NORM_PWR 5  // power, normalized to 0...1 for full A/D converter input range
# define SCALE_UNIT_AMPERE   6  // current (A), just for completeness.. and CLI_ParseTechUnit()

  // Logarithmic units, dB, relative to "something" ...
# define SCALE_UNIT_MASK_dB 0x18  // mask to detect unit "dB" in all of its flavours..
# define SCALE_UNIT_dB      0x08  // dB , unknown reference level (including Icom's flavour for their 'Spectrum Scope')
# define SCALE_UNIT_dBfs    0x09  // dB , reference level is  "full scale" (0 dBfs = clipping point)
# define SCALE_UNIT_dBV     0x0A  // dB , reference level is 1 volt
# define SCALE_UNIT_dBuV    0x0B  // dB , reference level is 1 microvolt rms across any impedance (dBV)
# define SCALE_UNIT_dBm     0x0C  // dB , reference level is 1 milliwatt (often across 50 or 600 ohms)
  // Other UNIT CODES, which are NO amplitudes/magnitudes ...
# define SCALE_UNIT_DEGREE  0x10
# define SCALE_UNIT_HERTZ   0x11
# define SCALE_UNIT_SECONDS 0x12
# define SCALE_UNIT_TIME    0x13  // almost the same as SCALE_UNIT_SECONDS, but UTL_FloatToTechNotation() may decide to use MINUTES or HOURS for the display
# define SCALE_UNIT_SAMPLES_PER_SECOND 0x14 // technically the same as 'Hz' but used for SAMPLING RATES in combo lists
#endif // ndef SCALE_UNIT_BLANK ?


/*-------------- Data types ----------------------------------------------*/

typedef enum // switch-case friendly enums for the key=value pairs in a "Frequency Memory" entry, for RigCtrl_StringToFreqMemEntry() :
{ RIGCTRL_FREQ_MEM_TOKEN_INDEX=0, // "i=" -> array index (not a part of the struct, but an extra function argument)
  RIGCTRL_FREQ_MEM_TOKEN_N_ITEMS, // "n=" -> total number of items in the array (also not a part of the struct but " " ")
  RIGCTRL_FREQ_MEM_TOKEN_NAME,    // "na=" -> pFreqMemEntry->c16MemoryName
  RIGCTRL_FREQ_MEM_TOKEN_FREQ,    // "fr=" -> pFreqMemEntry->RxTx[0].dblOperatingFreq_Hz
  RIGCTRL_FREQ_MEM_TOKEN_RX_FREQ, // "rx=" -> pFreqMemEntry->RxTx[1].dblOperatingFreq_Hz
  RIGCTRL_FREQ_MEM_TOKEN_OP_MODE,   // "mo=" -> pFreqMemEntry->RxTx[0].iOpMode
  RIGCTRL_FREQ_MEM_TOKEN_DATA_MODE, // "dm=" -> pFreqMemEntry->RxTx[0].iDataMode
  RIGCTRL_FREQ_MEM_TOKEN_DUPL_MODE, // "du=" -> pFreqMemEntry->RxTx[0].iDuplexMode
  RIGCTRL_FREQ_MEM_TOKEN_TONE_TYPE, // "to=" -> pFreqMemEntry->RxTx[0].iToneType
  RIGCTRL_FREQ_MEM_TOKEN_DIG_SQLCH, // "ds=" -> pFreqMemEntry->RxTx[0].iDigitalSquelch
  RIGCTRL_FREQ_MEM_TOKEN_RPTR_TONE, // "tf=" -> pFreqMemEntry->RxTx[0].iRepeaterToneFreq_Hz
  RIGCTRL_FREQ_MEM_TOKEN_TSQL_FREQ, // "ts=" -> pFreqMemEntry->RxTx[0].iToneSquelchFreq_Hz
  RIGCTRL_FREQ_MEM_TOKEN_DTCS_CODE, // "sc=" -> pFreqMemEntry->RxTx[0].iDTCS_Code
  RIGCTRL_FREQ_MEM_TOKEN_D_C_SQLCH, // "cs=" -> pFreqMemEntry->RxTx[0].iDV_DigitalCodeSquelch
  RIGCTRL_FREQ_MEM_TOKEN_DUP_OFFS,  // "of=" -> pFreqMemEntry->RxTx[0].iDuplexOffsetFreq_Hz
  RIGCTRL_FREQ_MEM_TOKEN_DEST_CALL, // "UR=" -> pFreqMemEntry->RxTx[0].c8DestCall        aka "UR" for digital voice
  RIGCTRL_FREQ_MEM_TOKEN_RPTR_CALL, // "R1=" -> pFreqMemEntry->RxTx[0].c8AccessRptrCall  aka "R1" for digital voice
  RIGCTRL_FREQ_MEM_TOKEN_GTWY_CALL  // "R2=" -> pFreqMemEntry->RxTx[0].c8GatewayRptrCall aka "R2" for digital voice
} RigCtrl_FreqMemTokens;



typedef struct tRigCtrl_ParamInfo // -> const T_RigCtrl_ParamInfo RigCtrl_ParameterInfo[] in RigControl.c ...
{
  int iUnifiedPN;  // Unified parameter number (not Icom-specific), e.g. RIGCTRL_PN_TRANSMITTING.
                   // ZERO marks the end of the lookup table RigCtr_ParameterInfo[] .
  const char *pszToken;  // SHORT but descriptive, unique token like "transmitting" (used for access via interpreter)
  const char *pszUnit;   // physical unit where applicable, e.g. "Hz", "dB", "%", often empty ("") but not NULL.
  const T_SL_TokenList *pStringValueTable; // string/value lookup table for certain parameters,
                   // e.g. "CW","USB","LSB",.. for RIGCTRL_PN_OP_MODE . NULL for most other PNs.

  int  iByteOffsetIntoRigCtrlInstance; // hacky way to retrieve the address of e.g. T_RigCtrlInstance.iTransmitting,
                   // RELATIVE TO THE BEGIN of struct T_RigCtrlInstance, using offsetof().
  int  iRigCtrlDataType; // data type for acrobatic pointer arithmetics; may be one of the following:
#       define RIGCTRL_DT_UNKNOWN    0
#       define RIGCTRL_DT_INT        1
#       define RIGCTRL_DT_DOUBLE     2
#       define RIGCTRL_DT_STRING     3
#       define RIGCTRL_DT_FREQ_RANGE 4
#       define RIGCTRL_DT_FREQ_MEM   5 // e.g T_RigCtrlFreqMemEntry BandStackingRegs[RIGCTRL_NUM_BAND_STACKING_REGS]
#       define RIGCTRL_DT_OTHER    255
  // Note: When adding new data types above, consider supporting them
  //       in Remote_CW_Keyer\CwNet.c : CwNet_OnPoll(), too !
  // Parameters accessable via iByteOffsetIntoRigCtrlInstance
  //  will have the following values, as long as not set via CI-V parser or the application:
#       define RIGCTRL_NOVALUE_INT    -2147483647  // == 0x80000001 = "almost the most negative integer" (ex: INVALID_INT_VALUE)
#       define RIGCTRL_NOVALUE_DOUBLE -99e99       // ex: INVALID_DOUBLE_VALUE

  int  iAccessFlags;     // bitwise combination of the following:
#       define RIGCTRL_AF_READWRITE 0
#       define RIGCTRL_AF_READONLY  1  // only *readable* parameters, e.g. T_RigCtrlInstance.iSMeterLevel_dB



  // Info to read or write the parameter via Icom's CI-V protocol :
  const BYTE *pbCIVReadCmd; // byte-sequence for RigCtrl_SendAndLogMessage() to READ
              // this parameter via Icom CI-V protocol, beginning with
              // the "Command number", sometimes followed by a
              // "Sub command number".
              // The READ RESPONSE or WRITE COMMAND always has a
              // "Data area" . In some cases, even the READ REQUEST
              // has a "Data area", for example a "memory number".
              // NULL when the command format is "too special",
              //  and requires special treatment in an extra function.
              //  For example, cmd 0x03 READS the "operating frequency",
              //               cmd 0x05 WRITES the same parameter in CI-V.
  int  iCIVReadCmdLength; // length of the above sequence in bytes (w/o "Data Area") .
              // Note: strlen(pbCIVReadCmd) doen't work because
              //       the "sub command" (part of the byte sequence)
              //       may be 0x00 .
  int  iCIVDataType;      // Icom-specific data type :
#       define CIV_DTYPE_NONE  0   // dummy for "no data"
#       define CIV_DTYPE_BCD1  1   // BCD with only ONE digit (in the data byte bits 3..0)
#       define CIV_DTYPE_BCD2  2   // two-digit BCD (tens in the bytes bits 7..4, ones in bits 3..0)
#       define CIV_DTYPE_BCD3  3   // three-digit BCD (also least significant digit FIRST)
#       define CIV_DTYPE_BCD4  4   // FOUR-DIGIT BCD .. warning; often bitwise ORed to some of the flags further below !
#       define CIV_DTYPE_BCD5  5   // etc, etc, ...
#       define CIV_DTYPE_BCD6  6   // SIX-DIGIT BCD as used for "Duplex Offset frequency setting" in IC-9700
#       define CIV_DTYPE_BCD7  7
#       define CIV_DTYPE_BCD8  8
#       define CIV_DTYPE_BCD9  9
#       define CIV_DTYPE_BCD10 10  // TEN-digit BCD as used for VFO- and "Transmit"-frequencies in IC-7300, IC-9700, etc)
#       define CIV_DTYPE_BCD11 11
#       define CIV_DTYPE_BCD12 12  // future reserve / maybe for some exotic "Multi-Gigahertz-Radios" ?
        // Don't rearrange the above "basic BCD types" ! We calculate the number
        // of DIGITS as follows (somewhere in RigControl.c) :
        //  > nDigits = 1 + (iCIVBasicType - CIV_DTYPE_BCD1);
#       define CIV_DTYPE_HEX_1BYTE              13 // a single byte, interpreted as "hexadecimal value" (e.g. 0xA2 = "default CI-V address" for an IC-9700. That's NOT BCD !)
#       define CIV_DTYPE_BAND_EDGE_FREQUENCIES  14 // 2-digit "edge number" + 12..14-digit "Lower edge" + 0x2D as "Separator" + 12..14-digit "Higher edge"
#       define CIV_DTYPE_SCOPE_EDGE_FREQUENCIES 15 // 2-digit "edge number" + 10..12-digit "Lower edge", NO "Separator", 10..12-digit "Higher edge"
#       define CIV_DTYPE_MASK_FOR_BASIC_TYPE 0x7F
#       define CIV_DTYPE_SUFFIX_BCD2_SIGN    0x80 // Flag bitwise ORed to e.g. 'CIV_DTYPE_BCD4'. Adds a TWO-digit BCD suffix(!) indicating "plus" (0x00) or "minus" (0x01).
#       define CIV_DTYPE_FLAG_MSBYTE_FIRST  0x100 // e.g. bitwise ORed to e.g. 'CIV_DTYPE_BCD4', for parameters like the 'Scope Reference Level'.
               // '--> WITHOUT this flag, assume the LEAST SIGNIFICANT BYTE
               //      (with two BCD digits per byte) is sent first.
               // LEAST SIGNIFICANT BYTE FIRST
               //        seems to be common for FREQUENCIES.
               // MOST SIGNIFICANT BYTE FIRST
               //        seems to be common for LEVELS, COLOURS, VOLUME POT SETTINGS, etc.
               // What a mess.
#       define CIV_DTYPE_FLAG_PERCENT255    0x200 // e.g. bitwise ORed to e.g. 'CIV_DTYPE_BCD4', but PERCENTAGES (0..100 in the internal value) must be scaled to 0..255 for CI-V !
   // Examples for the above, from the "IC-7300 Full Manual", A-7292-4EX-6
   //       (the -6 seems to be the document version):
   // * "Read transmit frequency" (A-7292-4EX-6 page 19-2) :
   //        pbCIVReadCmd = { 0x1C, 0x03 },  iCIVReadCmdLength = 3,
   //       iCIVDataType = CIV_DTYPE_BCD10
   // * "Send/read the Scope Reference level setting" (A-7292-4EX-6 page 19-12) :
   //       iUnifiedPN = RIGCTRL_PN_SCOPE_REF_LEVEL,
   //       pbCIVReadCmd = { 0x27, 0x19 },  iCIVReadCmdLength = 3,
   //       iCIVDataType = CIV_DTYPE_BCD4 | CIV_DTYPE_SUFFIX_BCD2_SIGN (!)
   //           ,----------,     |     ,---------------------------'
   //           |         \|/   \|/   \|/
   //           |        ,---,-------,---,
   //           |        |0:0|X:X|X:X|S:S|  Sign as SUFFIX ! 00 = positive
   //           |        '---'-------'---'                   01 = negative
   //        "Fixed to zero", but that only applies to the IC-7300.
   //        For other rigs (e.g. IC-7610), this two-digit BCD value
   //        is the SCOPE NUMBER ! 00 = Main Scope;  01 = Sub Scope.
   //        We treat this field like a "sub-sub-index", not part of the VALUE.
   //
   int iMsgType; // e.g. RIGCTRL_MSGTYPE_FREQUENCY_REPORT | RIGCTRL_MSGTYPE_FLAG_PERIODIC_POLL
                 // (mainly used for the TRAFFIC LOG)

} T_RigCtrl_ParamInfo; // -> const T_RigCtrl_ParamInfo RigCtrl_ParameterInfo[] in RigControl.c

typedef struct tRigCtrlTrafficLogEntry // used by RigCtrl_AddToTrafficLog(), RigCtrl_ReadTrafficLog(), RigCtrl_ClearTrafficLog(), etc.
{
  long i32MessageNr;    // unique "message number", assigned in RigCtrl_AddToTrafficLog(), details there.
                        // Only incremented when logging PACKETS, for reasons explained in RigCtrl_AddToTrafficLog().
  double dblUnixTime;   // date and time of transmission or reception
  BYTE b80Message[80];  // first 80 bytes of the message data
  int  iMsgLength;      // full length of the message (including pre- and postamble). Zero if the T_RigCtrlTrafficLogEntry doesn't contain 'packet data'.
  int  iMsgType;        // e.g. RIGCTRL_MSGTYPE_FREQUENCY_REPORT (if the parser 'understood')
  int  iRigCtrlPort;    //  0=RIGCTRL_PORT_RADIO, 1=RIGCTRL_PORT_AUX_COM_1, etc
# define RIGCTRL_MAX_COMMENT_LENGTH 120
  char szComment[ RIGCTRL_MAX_COMMENT_LENGTH+4 ]; // optional comment, possibly "human readable" message text
} T_RigCtrlTrafficLogEntry;
#define RIGCTRL_MAX_TRAFFIC_LOG_ENTRIES 2048

typedef struct tRigCtrlFrequencyRange
{ double dblFmin_Hz;
  double dblFmax_Hz;
} T_RigCtrlFrequencyRange; // -> TxBandEdges[], ScopeFreqRange[], .. (?)
#define RIGCTRL_MAX_FIXED_EDGE_SCOPE_FREQ_RANGES 15 /* deliberately 2 more than used by IC-7300 */
#define RIGCTRL_MAX_FIXED_EDGES_PER_FREQ_RANGE    3


typedef struct tRigCtrlUserDefinedBand
{ double dblFmin_Hz;
  double dblFmax_Hz;
  double dblFdef_Hz; // "default frequency" when switching to this band in the GUI
                     // (see details in Keyer_GUI.cpp )
  // Above: Struct members that most likely must be aligned to 8-byte .
  // Below: Struct members that only need to be aligned to 4-byte (at least for ARM/Cortex-M).
  int   dwDefOpMode; // "default op-mode", usually RIGCTRL_OPMODE_CW .
  DWORD dwOpModes; // Bitwise combination of RIGCTRL_OPMODE_CW, .._LSB, .._USB,
                   //  .._AM, .._FM, .._FM_WIDE, .._RTTY, and maybe even .._PSK .
                   // Some "extra flags" in iOpMode, bits 16 and above, shall be
                   // stripped by bitwise ANDing with RIGCTRL_OPMODE_MASK_TO_STRIP_FLAGS.
  DWORD dwPermissions; // bitwise combination of the following:
# define RIGCTRL_PERMISSION_NONE      0
# define RIGCTRL_PERMISSION_TRANSMIT  (1<<0) // may TRANSMIT in this band

  char sz16Name[16+4]; // custom band- or sub-band name, long enough for e.g. "60 m EU" and "60 m UK"

} T_RigCtrlUserDefinedBand; // -> T_RigCtrlInstance.UserDefinedBands[RIGCTRL_NUM_USER_DEFINED_BANDS]

typedef struct tRigCtrlFreqMemEntry // .. for classic "memory channels" and Icom's "Band stacking register"
{ // What belongs to Icom's "Memory content" (for a single entry) can be seen
  // in the IC-9700 'CI-V Reference Guide', document #A7508-3EX-4, page 14:
  // Listed for "Command: 1A 00", but the same format is used for
  // the response to 0x1A 0x01 = "Read Band stacking register", too.
  // Except for the first few bytes (with two-digit BCD each), e.g.
  //  (1) = "Frequency band" [this first byte is completely missing in the IC-7300's "Memory content", see A7292-4EX-6 page "19-10"]
  //  (2)+(3) = "Memory channel number" [this also exists in an IC-7300, but there, in bytes (1)+(2).. bleah !
  //  (4) = "Select memory setting" (??),
  // everything is stored in a struct member that resembles the name by Icom:
  struct // sub-struct because bytes (5)..(51) "white" are duplicated for SPLIT MODE [(5)..(51) "black"]
   { double dblOperatingFreq_Hz;  // (5)..(9)  [IC-7300: (4)..(8) ]
     int    iOpMode;     // IC-9700: (10)..(11), but Icom's code is translated into our own RIGCTRL_OPMODE_CW/LSB/USB/etc
                         // IC-7300: "Operating mode setting" in bytes (9)+(10), but "(10") is the FILTER NUMBER
     BOOL   fDataMode;   // IC-9700: (12) "Data mode setting" : FALSE=Data mode OFF, TRUE=Data mode ON.
     int    iDuplexMode; // IC-9700: (13, upper BCD digit) : 0=not Duplex, 1=Duplex-, 2=Duplex+, 3="RPS"(?)
     int    iToneType;   // IC-9700: (13, lower BCD digit) : 0=off, 1=TONE, 2=TSQL, 3=DTCS
                         // IC-7300: (11) = "Data mode       and  tone type settings"
                         //                 |_upper digit_|       |___lower digit__|
#            define RIGCTRL_TONE_TYPE_OFF  0
#            define RIGCTRL_TONE_TYPE_TONE 1  // guess what they mean by this is the 1750 Hz repeater tone (?)
#            define RIGCTRL_TONE_TYPE_TSQL 2  // guess what they mean by this is a sub-audio "Tone Squelch" (?)
#            define RIGCTRL_TONE_TYPE_DTCS 3  // seen in an IC-9700. Guess this is what the rest of the world calls "DCS" for "Digital-Coded Squelch" (except for ICOM)

     int    iDigitalSquelch; // IC-9700: (14, only the UPPER BCD DIGIT in use): 0=off, 1=Digital Call Sign Squelch (DSQL), 2=Digital Code Squelch (CSQL)
     int    iRepeaterToneFreq_Hz; // IC-9700: (15)..(17), IC-7300: (12)..(14)
     int    iToneSquelchFreq_Hz;  // IC-9700: (18)..(20), IC-7300: (15)..(17)
        // The above ("Tone squelch frequency setting") is the last common item
        // that exists in an IC-9700 *and* an IC-7300 .. but at different byte-offsets. AAAARGH !
        // You will have a lot of fun taking care of all these annoying little differences
        // when writing a parser for ALL Icom radios, since there are no
        // "separator characters" that would help to identify different fields.
        // Cramming all this stuff into stupid, unformatted BCD (Binary-Coded Decimal)
        // is an anachronism, but WE cannot change this, and Icom won't switch
        // to an easily parsable format or even something JSON-ish, or at least
        // a "memory content" consisting of key=value pairs.
     int    iDTCS_Code;  // (21)..(23) "DTCS code settings" (never used, and most likely never really supported here)
     int    iDV_DigitalCodeSquelch; // (24) .. will share the same fate as the "DTCS", whatever that is
     int    iDuplexOffsetFreq_Hz;   // (25)..(27)
     char   c8DestCall[8+4];        // (28)..(35) "UR (Destination) call sign setting, 8 chars, fixed" (here: plus trailing zero for "C")
     char   c8AccessRptrCall[8+4];  // (36)..(43) "R1 (Access repeater) call sign setting, 8 chars, fixed" (here: plus trailing zero for "C")
     char   c8GatewayRptrCall[8+4]; // (44)..(51) "R2 (Gateway/Link repeater) call sign setting, 8 chars, fixed" (here: plus trailing zero for "C")
   } RxTx[2];  // [0] for the RECEIVER, [1] for the TRANSMITTER (TX settings "when the Split function is ON")
  char sz16Name[16+4]; // (52)..(67) "Memory name setting, 16 chars, fixed" (here: plus trailing zeros for "C")
       // '--> the MEMORY NAME ("channel name") is only stored ONCE .
  // From A7508-3EX-4 page 14:
  // > NOTE: (about the "Band stacking register", Icom's command 0x1A 0x01) :
  // >  * The same data as (5) ~ (51) "white" are stored in (5) ~ (51) "black".
  // >  * When the Split function is ON, the data of (5) ~ (51) "black"
  // >    is used for transmit.
  // >  * Even if the Split function is OFF, enter the data into (5) ~ (51) "black"
  // >    to match your transceiver. We recommend that you set the same data
  // >    as (5) ~ (51) "black".
  //    WB: "black" are the circled numbers with a BLACK BACKGROUND, aka "inverse".
  //        Very unfriendly for developers using plain text editors for code.
  //        For the BAND STACKING REGISTERS, "(5) ~ (51) black" seem to be
  //        non-existent, as Icom wrote in A7508-3EX-4 page 15:
  // > NOTE:
  // >    When sending the contents, the codes, such as operating frequency
  // >    and operating mode*, should be added after the frequency band code
  // >    and the register code, as shown below.
  // >  * See (5) to (51) [WB: "white"] on 'Memory content setting.' (p. 14)

} T_RigCtrlFreqMemEntry;

typedef struct tRigCtrlAudioFilterParams
{ // Used in an ARRAY for Icom's "RX HPF/LPF setting for each operating mode",
  // to avoid unnecessary polling of the filter bandwidth when e.g.
  // WSJT-X asks for it, in every period !
  int iLowerEdge_Hz;  // lower cut-off frequency in Hertz
  int iUpperEdge_Hz;  // upper cut-off frequency in Hertz
  int iFilterIndex;   // 0 for Icom's "FIL1" .. 2 for Icom's "FIL3", or maybe more
  int iOpMode;        // RIGCTRL_OPMODE_CW/LSB/USB, because there are DIFFERENT
      // FILTERS for each of the basic modes. In Icom terms ("A7292-4EX-11" page 167):
      //  > RX HPF/LPF setting for each operating mode
      //  > Command : 1A 05 0001 : "Send/read SSB RX HPF/LPF settings"
      //                 05 0004 : "Send/read AM RX HPF/LPF settings"
      //                 05 0007 : "Send/read FM RX HPF/LPF settings"
      //                 05 0010 : "Send/read CW RX HPF/LPF settings"
      //                 05 0011 : "Send/read RTTY RX HPF/LPF settings"
      //                  | |__|-- FOUR-DIGIT sub-subcode, possibly RIG-SPECIFIC
      //                  '------- sub-command for CI-V monster command 0x01A
} T_RigCtrlAudioFilterParams; // -> T_RigCtrlInstance.sAudioFilterParams[RIGCTRL_NUM_AUDIO_FILTER_PARAMS]



typedef struct tRigCtrlSpectrum
{ // Simplified variant of T_SPECTRUM in C:\CBproj\SpecLab\TSpectrum.h,
  // to eliminate as many dependencies from non-standard headers as possible.
  // Since 2019-01, also used in Spectrum Lab's built-in OpenWebRX server,
  //        see c:\cbproj\SpecLab\http_server_sourcecode\OpenWebRX_Server.c .
  // Doesn't use dynamically allocated memory to keep this module suitable
  // for embedded systems (microcontrollers without an operating system) .
  //
  //     Because Icom radios (and most likely all similar radios with
  //                          with built-in waterfall or spectrum display)
  //     only spit out amplitude spectra, keep it simple, and keep everything
  //     in the same "decibel" unit (possibly relative to the receiver's
  //     noise floor.. ?) as used by the radio itself.
  //     Attenuator-, preamplifier- or reference level settings are considered
  //     when READING a T_RigCtrlSpectrum from the radio. There's no need
  //     to store such information in the header.
  int    nBinsUsed;      // may range from 1 to RIGCTRL_MAX_FREQ_BINS_PER_SPECTRUM
  double dblBinWidth_Hz; // 'width' of a single FFT bin in Hertz (=resolution
  double dblFmin_Hz;     // "radio frequency" stored in the first frequency bin.
     //  dblFmax_Hz would be dblFmin_Hz + (nBinsUsed - 1) * dblBinWidth_Hz !
  double dblCenterFreq_Hz;
      // THE VFO FREQUENCY IS NOT ALWAYS EQUAL TO the above "center" frequency !
      // That's why some applications need to know the current 'fixed/center' setting:
  BOOL   fFixedEdgeMode; // TRUE=fixed frequency edges, FALSE=centered on VFO-frequency
  double dblUnixTime;    // timestamp in UNIX format (second elapsed since 1970-01-01 00:00:00 UTC).
         // If the SOURCE doesn't support this, use the system time in UTC instead.
#define RIGCTRL_MAX_FREQ_BINS_PER_SPECTRUM 1024 // at least as many frequency bins
         // as the width of the spectrum display in the "large" Icom radio.
         // For example, an IC-7851 sends 704-byte-messages via LAN ->
         // a MAXIMUM of 1024 frequency bins per spectrum should be enough.
  float  fltMagnitudes_dB[ RIGCTRL_MAX_FREQ_BINS_PER_SPECTRUM ];
  int    iAmplitudeUnit; // e.g. SCALE_UNIT_dB = "Icom's dB scale where ZERO is near the NOISE FLOOR" (below "S0")
  float  fltAmplOffset_dBm; // number to add to fltMagnitudes_dB[] to convert into dBm (dB "over" one milliwatt)

} T_RigCtrl_Spectrum;

typedef struct tRigCtrl_MsgBuffer
{
#define RIGCTRL_MAX_MESSAGE_LENGTH 2048
  BYTE bData[RIGCTRL_MAX_MESSAGE_LENGTH];
  int  iLength;    // index into bData[], with post-increment
  int  iType;      // bitwise combination of RIGCTRL_MSGTYPE_..., including flags,
                   //
} T_RigCtrl_MsgBuffer;


typedef struct tRigCtrl_MsgFilter // .. for client message forwarding
{
  int iCmd;      // [in] CI-V main command like 0x03 = "Read operating frequency" (response on request)
  int iSubCmd;   // [in] optional sub-command. -1 to omit or ignore.
  int iDataB1;   // [in] optional 1st parameter byte. -1 to omit or ignore.
  int iDataB2;   // [in] optional 2nd parameter byte. -1 to omit or ignore.
  int iMsgTypes; // [in] additional message types like RIGCTRL_MSGTYPE_OK | RIGCTRL_MSGTYPE_NOT_OK
} T_RigCtrl_MsgFilter;


typedef int (*T_RigCtrl_AppCallback)( // may be invoked ONCE or even MULTIPLE TIMES from RigCtrl_ProcessData() ..
  //     '--> return value depends on iRigCtrlCbkEvent, NEGATIVE in case of errors.
  struct tRigCtrl_PortInstance *pPortInstance, // [in] RigControl port instance that 'decoded' the message, with ...
                      // [in] pPortInstance->iMsgTypeFromParser, iUnifiedParameterNumberFromParser, etc
  int iRigCtrlOrigin, // [in] e.g. RIGCTRL_ORIGIN_RADIO, RIGCTRL_ORIGIN_CONTROLLER,
                      //      or sometimes just RIGCTRL_ORIGIN_COM_PORT_RX/TX
                      //      if we don't know "who is what" on a Serial Port Tunnel.
  int iRigCtrlCbkEvent,  // Reason for invoking this callback: One of the values below:
#  define RIGCTRL_CBK_DUMMY    0 // dummy call without any of the following (but e.g. a change of state)
#  define RIGCTRL_CBK_SEND_MSG 1 // "please SEND A MESSAGE on this port" .
          // '--> This call may be made from a DIFFERENT THREAD than the one
          //      that called RigCtrl_ProcessData(), for reasons explained
          //      in the SERVER STATE DIAGRAM in RigControl_CIV_Server.c !
          //      In RCW Keyer, this problem is solved with a lock-free circular
          //      FIFO in AuxComPorts.c : AuxCom_RigControlCallback() .
#  define RIGCTRL_CBK_DECODED  2 // "have decoded a message (received on this port)
          // '--> Note that this may happen MULTIPLE TIMES during a single call
          //      of RigCtrl_ProcessData(), if the passed-in block contains
          //      multiple messages !
  BYTE *pbMessage, int iMsgLength, // [in] raw message data and length in bytes
  char *pszComment ); // [in] human readable "comment" for the log


typedef struct tRigCtrl_PortInstance // -> T_RigCtrlInstance.PortInstance[RIGCTRL_NUM_PORT_INSTANCES]
{ // Instance data to communicate with the 'real radio' ( .PortInstance[RIGCTRL_PORT_RADIO] )
  //   or to forward data to/from other applications ( .PortInstance[RIGCTRL_PORT_AUX_COM_1 .. 3] .
  // Multiple of these 'Port Instances' may be used to forward requests
  //    and responses from other applications (CLIENTS) that need to control the radio,
  //    e.g. WSJT-X . In Remove CW Keyer, the name of ports configured that way
  //    are 'Virtual Rig Ports'. They may even run a DIFFERENT PROTCOL
  //    than the one used on PortInstance[RIGCTRL_PORT_RADIO] .
  //    Usually connected via virtual "NULL-modem" cable (extra COM ports created by com0com).
  int iPortUsage; // purpose of a certain RIG CONTROL PORT,
     // and (linked via pointer T_AuxComPortInstance.pRigctrlPort) an AUXILIARY COM PORT:
#    define RIGCTRL_PORT_USAGE_NONE              0 // Don't modify these values - they are used in CONFIGURATION FILES !
#    define RIGCTRL_PORT_USAGE_WINKEYER_HOST     1 // a HOST using K1EL's "CW Keyer IC for Windows", aka "Winkeyer"/"Winkeyer2"
#    define RIGCTRL_PORT_USAGE_WINKEYER_EMULATOR 2 // a crude EMULATOR for K1EL's "CW Keyer IC for Windows", used e.g. to fool the N1MM Logger
#    define RIGCTRL_PORT_USAGE_RADIO_CONTROL     3 // port controlling the 'real radio'
            // Codes 4..9 are reserved for other 'smart Morse key adapters'
#    define RIGCTRL_PORT_USAGE_ECHO_TEST        10 // send ANYTHING received on this port back AS FAST AS POSSIBLE
#    define RIGCTRL_PORT_USAGE_TX_STRESS_TEST   11 // try to flood TXD with data (256-step 'ramp up')
#    define RIGCTRL_PORT_USAGE_TEXT_TERMINAL    20 // .. e.g. to control other 'station equipment' via commands entered on the 'Debug' tab
#    define RIGCTRL_PORT_USAGE_SERIAL_TUNNEL    21 // two serial ports "communicating with each other"
            // between the Remote CW Keyer's CLIENT SIDE ("local shack")
            //   and the SERVER SIDE ("remotely controlled radio site").
            // Note: A 'Serial Tunnel' may be local (between COM ports on one machine)
            //       or on two sides of the TCP/IP connection between RCW Keyer's
            //       SERVER and CLIENT(s). Thus the term 'SERIAL_TUNNEL' here.
            // file:///C:/cbproj/Remote_CW_Keyer/manual/Remote_CW_Keyer.htm#Serial_Port_Tunnels !
#    define RIGCTRL_PORT_USAGE_VIRTUAL_RIG      22 // successor to RIGCTRL_PORT_USAGE_SERIAL_TUNNEL,
            // but must be able to DECODE and ENCODE the radio control prototcol ! See
            // file:///C:/cbproj/Remote_CW_Keyer/manual/Remote_CW_Keyer.htm#Virtual_Rig_Ports .

  // Initially, only T_RigCtrlInstance.PortInstance[RIGCTRL_PORT_RADIO] is the instance "talking to the radio".
  // The 'real radio' may use a different protocol than the other ports, thus:
  int iRadioCtrlProtocol;  // protocol used on a particular port, e.g. RIGCTRL_PROTOCOL_ICOM_CI_V
  int iRadioMasterAddr;    // for CI-V, this may be 0xE0=RIGCTRL_CIV_MASTER_ADDR_MIN .. 0xEF=RIGCTRL_CIV_MASTER_ADDR_MAX (but don't bet on that)
#define RIGCTRL_CIV_MASTER_ADDR_MIN 0xE0
#define RIGCTRL_CIV_MASTER_ADDR_MAX 0xEF
  int iRadioDeviceAddr;    // for an IC-7300, this may be 0x94 (but don't bet on that). May be 0x00 = RIGCTRL_DEF_ADDR_AUTO_DETECT !
  int iRealRadioAddress;   // "FROM"-address copied from the RESPONSE received on the RADIO CONTROL PORT. Used by e.g. the "Virtual Radio" emulation in RigControl_CIV_Server.c .

  int iCIVTransceiveMode;  // Accessable via RIGCTRL_PN_CIV_TRANSCEIVE_MODE.
      // 0 = FALSE = UNSOLICITED frequency- and operating-mode-reports disabled;
      // 1 = TRUE  = UNSOLICITED frequency- and operating-mode-reports ENABLED.
      // NOT PART OF T_RigCtrlInstance, but part of T_RigCtrl_PortInstance(!),
      // because Icom's "CI-V Transceive Mode" may be disabled for
      // 'Additional COM Port' (aka "client ports" when configured as "Virtual Radio" port)
      // if a poorly written client software cannot handle unsolicited reports.
      // But we will NEVER, EVER allow a poorly written client software (using outdated Hamlib?)
      // to turn "CI-V Transceive" OFF inside the REAL RADIO ! That's why this
      // "rig control parameter" (iCIVTransceiveMode) is different from others.

  DWORD dwPortErrorFlags; // 0 = "all ok", or any BITWISE combination of the following:
#define RIGCTRL_PORT_ERROR_FLAG_NOT_OPEN 0x01  // the "communication port" is currently NOT OPEN
#define RIGCTRL_PORT_ERROR_FLAGS_ALL 0xFFFFFFFF // <- special combination for

  T_RigCtrl_AppCallback pAppCallback; // <- if non-NULL, may be invoked ONCE
        // or even MULTIPLE TIMES from a single call of RigCtrl_ProcessData() .

  // The following members (iResponseCountdown_ms..iExpectResponseForUnifiedPN)
  // indicate if the T_RigCtrl_PortInstance is currently BUSY, for example
  // because a COMMAND has been sent from this side for which we "expect"
  // a response, but that response has not arrived yet. See implementation
  // of BOOL RigCtrl_IsPortBusy( T_RigCtrl_PortInstance *pPortInstance ) :
  int iResponseCountdown_ms; // timer to skip parameters if the radio doesn't answer.
                             // As long as iResponseCountdown_ms is positive,
                             // periodic polling (also from RigCtrl_SendReadCommandFromFIFO() )
                             // is disabled, but requests from e.g. RigCtrl_QueueUpCmdToReadUnifiedPN()
                             // can still be queued up in iReadPNFifo[] .
  DWORD dwExpectedResponse_CmdAndSubcode; // e.g. set to 0x03****** when asking for
         // a VFO frequency via CI-V,
         // or 0xFFFFFFFF or RIGCTRL_NOVALUE_INT when NOT expecting a certain response.
         // Also set (>=0) when sending a command TO THE RADIO PORT on behalf of a client !
         // iExpectedResponseCmd is only "valid" as long as iResponseCountdown_ms is nonzero.
         // (but it doesn't hurt to set iExpectedResponseCmd to -1 when the response arrives,
         //  to avoid MISINTERPRETING the next unspecific "OK" or "NOT OK" from the radio)
         // Macros to assemble i32ExpectedResponse_CmdAndSubcode for CI-V commands
         // WITHOUT a "Sub cmd." (that's the column title in Icom's "Command table"),
         // with a SINGLE-BYTE "Sub cmd." (as in 0x1A 0x04),
         //   or a  THREE-BYTE "Sub cmd." (as in 0x1A 0x05 with HUNDREDS of four-digit indices)
# define RCTL_COMBINE_CMD_ONLY(cmd)                (((DWORD)cmd<<24)|0x00FFFFFFUL)
# define RCTL_COMBINE_CMD_AND_SUB(cmd,sub)         (((DWORD)cmd<<24)|((DWORD)sub<<16)|0x0000FFFFUL)
  // problematic (if the 4-digit "index" is considered DECIMAL, not HEX):
  // ex: # define RCTL_COMBINE_CMD_SUB_AND_IDX(cmd,sub,idx) (((DWORD)cmd<<24)|((DWORD)sub<<16)|((DWORD)idx))

  int iExpectResponseForUnifiedPN; // kind of replacement for dwExpectedResponse_CmdAndSubcode .
         // Would contain e.g. RIGCTRL_PN_FREQUENCY after asking the radio for the VFO frequency,
         // regardless of the PROTOCOL in use .
         // iExpectResponseForUnifiedPN <=(!) RIGCTRL_PN_UNKNOWN(0) when "not busy from a PN".
  int numResponseTimeouts;    // Number of response timeouts in a row.
         // Too many of these causes an attempt to turn the radio on again
         // (or whatever is connected to this T_RigCtrl_PortInstance), etc.


  T_RigCtrl_MsgBuffer sRxMsg[2]; // since 2025-07, TWO buffers for received messages.
            // Messages travelling through a serial port tunnel
            //    in one direction are stored in sRcvdMsg[0] before they are complete;
            //    messages in the opposite direction are stored in sRcvdMsg[1] -
            //    see details in AuxCom_PassRxOrTxDataToProtocolSpecificDecoder().
            // With .fActAsServer==TRUE, sRxMsg[1] stores a copy of the message
            //    from the external client when iServerState switches
            //    from RIGCTRL_SSTATE_PARSING_COMMAND to RIGCTRL_SSTATE_WAIT_RADIO_NBUSY.
            // Later, when switching from RIGCTRL_SSTATE_FWD2RADIO_BUSY to RIGCTRL_SSTATE_FWD2RADIO_DONE,
            //    pServerPortInstance->sRxMsg[1] stores the RESPONSE(!) from the REAL RADIO
            //    (copied there in RigCtrl_CheckAndForwardResponseToClients()
            //    when the received message is identified as the 'expected' response) .
            // pPortInstance->sRxMsg[0] is always received by THE PORT ITSELF.
  T_RigCtrl_MsgBuffer sTxMsg; // buffer to assemble a message for TRANSMISSION .
            // For instances RIGCTRL_PORT_AUX_COM_1..RIGCTRL_PORT_AUX_COM_3,
            // sTxMsg may contain the to-be-sent RESPONSE after RigCtrl_ParseCIV(),
            // depending on .iServerState (RIGCTRL_SSTATE_RESPONSE_READY) .

  BOOL fLastSendFailed; // to avoid flooding log with "TX error" messages when a client (like RS-BA1) goes away
  int  iClientOptions;  // bitwise combination of the following flags :
#define RIGCTRL_CLIENT_OPTION_REJECT_UNSOLICITED_MSGS 0x0001
#define RIGCTRL_CLIENT_OPTION_EMULATE_CIV_ECHO        0x0002
  // From a 3rd party client's point of view, the 'local CI-V echo'
  // can be modified via CI-V itself, to emulate the behaviour of modern rigs
  // where CI-V isn't a simple SINGLE-WIRE interface ("always with an echo")
  // but an USB port or LAN with (logically) separated "TxD" and "RxD" lines,
  // which eliminates the need for collision detection on the physical layer.
  // The following flag controls (and indicates) if the 'local CI-V echo'
  // is currently active:
  BOOL fEmulateCIVEcho;  // "initial setting" copied from iClientOptions !

  // Currently registered responses (expected by the client on a certain port,
  //   used by RigCtrl_RegisterReponseForClient) :
#define RIGCTRL_MAX_REGISTERED_MESSAGES_PER_CLIENT 32 /* who knows.. without buying "RS-BA1" V2 */
  T_RigCtrl_MsgFilter RegisteredMessages[RIGCTRL_MAX_REGISTERED_MESSAGES_PER_CLIENT];

  // For debugging / troubleshooting / statistics in SL's "Debug"-window :
  DWORD dwNumGarbageBytes;
  DWORD dwNumMessagesSent, dwNumMessagesRcvd;
  char sz255CommentFromParser[256]; // for a human-readable traffic log ... set in e.g. RigCtrl_ParseCIV()
        // '--> member of T_RigCtrl_PortInstance because e.g. AuxComPorts.c needs this
        //      for the traffic log, to emit info about a response sent on an 'Additional COM Port'
        //      AT EXACTLY THE RIGHT TIME .
  int  iMsgTypeFromParser; // <- for the traffic log AND TO FORWARD MESSAGES from server- to RADIO-port.
        // '--> Set to the value returned by e.g. RigCtrl_ParseCIV().
  int  iUnifiedParameterNumberFromParser; // <- also mainly for the traffic log,
        // but also to avoid yet another function parameter for T_RigCtrl_AppCallback .
#if( SWI_RIGCTRL_ACT_AS_SERVER ) // only when RigControl.c isn't only a CLIENT (controlling a radio)
                                 // but also a SERVER (emulating a radio, controlled by external apps):
  BOOL fActAsServer; // TRUE when using e.g. RigCtrl_CIV_Server.c to mimick an Icom radio, etc.
  int  iServerState; // States for the 'machine' sketched (with transitions) in RigControl_CIV_Server.c :
# define RIGCTRL_SSTATE_PASSIVE         0 // may be "acting as server" but got nothing to do at the moment
# define RIGCTRL_SSTATE_PARSING_COMMAND 1 // state entered immediately before calling e.g. RigCtrl_ParseCIV(), SERVER-SIDE
# define RIGCTRL_SSTATE_WAIT_RADIO_NBUSY 2 // state entered in RigCtrl_ParseCIV(), SERVER-SIDE, if a RESPONSE cannot be assembled yet
                                          // because e.g. the to-be-read data are not available yet.
# define RIGCTRL_SSTATE_FWD2RADIO_BUSY 3 // a COMMAND (read or write) has been sent on behalf of a remote client,
                                         // now waiting for the RESPONSE received on the RADIO PORT .
# define RIGCTRL_SSTATE_FWD2RADIO_DONE 4 // state entered when the to-be-read data are available now,
                                         // or the to-be-written data have been forwarded
                                         // from the 'Virtual Rig' to the REAL radio and the "OK" arrived.
                                         // (that's when pRadioPortInstance->iWorkingForServerPort will be CLEARED)
# define RIGCTRL_SSTATE_RESPONSE_READY 5 // .. again: see STATE DIAGRAM in RigControl_CIV_Server.c .. "Ready but not SENT yet"
  int nServerStateErrors; // <- number of INVALID state transitions, counted in RigCtrl_SetServerState()
  int iWorkingForServerPort; // e.g. RIGCTRL_PORT_AUX_COM_1 in .PortInstance[RIGCTRL_PORT_RADIO],
         // |    if the RADIO CONTROL PORT is currently 'working' for a remote client .
         // '--> When NONZERO, messages sent to / received from the REAL RADIO
         //      may be shown in the traffic log even though the traffic log
         //      has already been PAUSED (e.g. after finishing the init-phase).
         // Set in     RigCtrl_ForwardCommandFromServerPortToRadio(),
         // cleared in RigCtrl_CheckAndForwardResponseToClients() .

  T_RigCtrl_ParamInfo sParamInfoForPendingCommand;
  T_RigCtrl_MsgBuffer sResponseFromRealRadio; // buffer for the last response from the REAL RADIO,
      // filled when T_RigCtrlInstance.PortInstance[RIGCTRL_PORT_RADIO]
      // has relayed a READ-REQUEST on behalf of a 'Virtual Rig Port'
      // to the REAL RADIO. Again: All the details are in RigCtrl_CIV_Server.c !
  int iStartOfDataField;   // .. within .sTxMsg, set e.g. in RigCtrl_CIV_Server_OnReadCommand()
  int iLengthOfDataField;  // .. within .sTxMsg, set e.g. in RigCtrl_CIV_Server_OnReadCommand()
  // 'Statictics' to decide if e.g. WFView's "initialisation phase" is over :
  int iNumSMeterReports, iNumVfoFreqReports; // <- purpose/details in RigControl_CIV_Server.c (criterion for RIGCTRL_TMON_DISPLAY_OPTION_PAUSE_ON_INIT_DONE)
#endif // SWI_RIGCTRL_ACT_AS_SERVER ?
  BOOL fTrafficMonitorEnabled; // here: in T_RigCtrl_PortInstance.
       // Checked in RigCtrl_IsMsgTypeRejectedForLog(),
       // set in Keyer_Main.cpp : StartKeyerAndShowInfo(), after KeyerGUI_ParseAuxComParams().
  BOOL fTrafficMonitorPausedOnTrigger; // FALSE=not "automatically paused on a trigger event", TRUE="something happened, PLEASE TAKE A LOOK.."
            // '--> Reason for "pausing" / "stopping" : e.g. RIGCTRL_TMON_DISPLAY_OPTION_PAUSE_ON_FULL_FIFO / .. STOP_ON_ERROR / ...
  int  iTrafficMonitorUnpauseCountdown; // special service to debug 'forwarding' messages
       // from 'Virtual Rig' ports to the real radio, to see commands sent to the
       // radio ON BEHALF OF AN EXTERNAL CLIENT (e.g. WFView) regardless of
       //  despite fTrafficMonitorPausedOnTrigger fTrafficMonitorEnabled on the RADIO PORT.
       // Details in RigCtrl_ForwardCommandFromServerPortToRadio() .



  struct tRigCtrlInstance *pRC; // "link back" from the T_RigCtrl_PortInstance to its parent (the entire "Rig Control" instance)

} T_RigCtrl_PortInstance; // -> T_RigCtrlInstance.PortInstance[RIGCTRL_NUM_PORT_INSTANCES]


typedef struct tRigCtrlInstance // T_RigCtrlInstance
{
  // "Config data". Usually set once (in or shortly after RigCtrl_Init() ):
  // ex: int iRadioDeviceAddr;   // "to" transceiver's address (selected in the GUI; may be RIGCTRL_DEF_ADDR_AUTO_DETECT when beginning a session with fListenOnlyMode = TRUE)
  // ex: int iRadioMasterAddr;   // "from" controller's address (often >= 0xE0; may be RIGCTRL_DEF_ADDR_AUTO_DETECT when beginning a session with fListenOnlyMode = TRUE)
  // ex: int iRadioCtrlProtocol; // RIGCTRL_PROTOCOL_ICOM_CI_V, possibly more one day
  //     '--> To avoid confusion / reduce complexity,
  //          the above members now ONLY exist in the T_RigCtrl_PortInstance,
  //          not in the T_RigCtrlInstance anymore !
  DWORD dwRigControlFlags; // RIGCTRL_FLAG_SEND_SETTINGS_ON_STARTUP,
                          // RIGCTRL_FLAG_WANT_SPECTRUM_DATA,
                          // RIGCTRL_FLAG_SIDETONE_WHEN_KEYED_ON_RIG, etc (bit combination)
  int iCapabilities;      // RIGCTRL_CAPS_POWER_ON_OFF_VIA_CAT, RIGCTRL_CAPS_SPECTRUM_VIA_CAT, etc (bit combination)
  DWORD dwAvailableBands; // bitwise combination of RIGCTRL_BAND_...,
                          //  initialized in RigCtrl_Init() but restricted/revised later
  int iMaxResponseDelay_ms; // maximum delay for getting a response, in milliseconds
  int iMaxTimeToSendMsg_ms; // maximum time spent waiting for message-transmission, in milliseconds
  int iTransmitReqstCountdown_ms; // "countdown timer" reloaded as long as iTransmitReqst is set.
         // '--> Important for the 'Sidetone when keyed ON THE RIG',
         // to tell if the rig transmits on behalf of a REMOTE CLIENT
         // or because THE SYSOP has connected his Morse key *directly* to the radio.

  T_CFIFO *pRadioPortTxFifo; // thread-safe circular FIFO to TRANSMIT data to the radio
  T_CFIFO *pRadioPortRxFifo; // thread-safe circular FIFO to RECEIVE data from the radio

  uint32_t u32RxAudioSamplesPerSecond; // for Icom transceivers with AUDIO VIA UDP, possibly fixed to 48 kHz !
  uint32_t u32TxAudioSamplesPerSecond;

  BOOL fListenOnlyMode; // <- term inspired by CAN (Controller Area Network) :
                        // FALSE="active mode" (where a "Controller" may send commands),
                        // TRUE ="receive-only" aka "eavesdropping" mode (where this instance
                        //        MUST NOT send anything, just like CAN doesn't even send "ACK" in "LOM")

  // Replacement for an awful list of FUNCTION ARGUMENTS passed from RigCtrl_ProcessData()
  // -> RigCtrl_XYZ() to the optional 'On-Decode'-callback (invoked via pOnDecodeCallback):
  int iRigCtrlPort;   // [in] e.g. RIGCTRL_PORT_RADIO, RIGCTRL_PORT_AUX_COM_1, .. RIGCTRL_PORT_AUX_COM_1, ..
  int iRigCtrlOrigin; // [in] e.g. RIGCTRL_ORIGIN_RADIO, RIGCTRL_ORIGIN_CONTROLLER,
                      //      or sometimes just RIGCTRL_ORIGIN_COM_PORT_RX/TX
                      //      if we don't know "who is what" on a Serial Port Tunnel.
  char sz80Comment[84]; // [out] string buffer for COMMENTS, long enough for all 'parsers'
                        //       invoked via function pointer in Yaesu5Byte_ProcessData() .
  int iCurrentMsgType; // <- input for pOnDecodeCallback, e.g. RIGCTRL_MSGTYPE_FREQUENCY_REPORT
        // | RIGCTRL_MSGTYPE_FLAG_RX | RIGCTRL_MSGTYPE_FLAG_PERIODIC_POLL | ...
        // RigControl.c : RigCtrl_Y5B_OnDecodeCallback() uses this info
        // to decide whether to LOG/DISPLAY the message or not.

  // Variables exchanged with the remote rig (and a few "update counters" for them):
  int iDefaultAddress;   // For Icom radios, the DEFAULT CI-V ADDRESS also identifies the model !
                         // RigControl.c tries to READ THIS FROM THE RADIO,
                         // because e.g. iRadioDeviceAddr(!) = 0x48 doesn't mean it's an IC-706.
                         // Only iDefaultAddress(!) = 0x48 means "it's REALLY an IC-706.
  double dblVfoFrequency; // current 'VFO' frequency (interpretation depends on the radio.
                         // Most use the (suppressed) carrier frequency for USB + LSB,
                         // some always use the center frequency (even for USB + LSB).
                         // Accessable as parameter # RIGCTRL_PN_FREQUENCY .
  double dblTxFrequency; // frequency (in Hz) that WOULD BE USED for transmission.
                         // Accessable as parameter # RIGCTRL_PN_TX_FREQUENCY .
                         // Preferred API to set: RigCtrl_SetOperatingMode() .
  int    iFrequencyModifiedByRadio_cnt; // counter incremented whenever .dblFrequency was modified BY THE RADIO,
                                        // for example after turning the VFO knob. Polled in a GUI-thread somewhere.
  int    iOpMode;        // RIGCTRL_OPMODE_CW/LSB/USB/.., combineable with _REVERSE, _NARROW, etc.
                         // Accessable as parameter # RIGCTRL_PN_OP_MODE .
                         // Preferred API to set: RigCtrl_SetOperatingMode()
  int    iCurrVfoIndex, iPrevVfoIndex; // 0="VFO A", 1="VFO B". Used to keep track of changes from e.g. cmd 0x07, in RigCtrl_ParseCIV()
  BOOL   fVfoSwitchPending; // flag set on e.g. CI-V cmd 0x07, "Select the VFO mode" (includes "Exchange VFO A and VFO B")
  double dblUnselVfoFreq; // frequently polled by WFView, so keep this parameter prepared in memory
  int    iUnselVfoOpMode; // also frequently polled by WFView - no idea why
  int    iSelVfoDataMode, iUnselVfoDataMode; // <- so far, only exist to implement command 0x26 sub 0x00 / 0x01 properly

  double dblDemodBasebandFreqOffset, dblDemodBasebandFreqFactor; // f(iOpMode), see RigCtrl_UpdateBasebandFrequencyConversionParams()
  int    iFilterBW_Hz; // may be readable via cmd 0x1A, subcmd 0x03 (polled by WSJT-X) .
                       // Since 2025-08-02, also updated by a few OTHER READ RESPONSES in RigCtrl_ParseCIV(). Grep for the date.
# define RIGCTRL_NUM_AUDIO_FILTER_PARAMS 16 /* Icom: THREE FILTERS per CW and SSB, and maybe one for FM+AM+RTTY */
  T_RigCtrlAudioFilterParams sAudioFilterParams[RIGCTRL_NUM_AUDIO_FILTER_PARAMS];
  int    iDataMode;    // 0=no data mode, 1=data mode. Icom: cmd 0x1A, subcommand 0x06 ("").
                       // Accessable as parameter # RIGCTRL_PN_DATA_MODE .
# define RIGCTRL_NUM_TX_BAND_EDGES 32
  T_RigCtrlFrequencyRange TxBandEdges[RIGCTRL_NUM_TX_BAND_EDGES]; // read from THE RIG ... not from a configuration file.
  int    iNumTxBands;  // number of valid entries in TxBandEdges[]; read via CI-V cmd 0x1E subcmd 0x00
  BOOL   fTxBandsModified; // <- application may POLL and CLEAR this flag after taking notice
  BOOL   fTxBandsModifiedForCwNet; // <- similar flag to be "consumed" by CwNet.c,
          // when sending the radio's transmit-band-edges from server to clients.

# define RIGCTRL_NUM_USER_DEFINED_BANDS 100 // .. bands or sub-bands. There may be MULTIPLE for a single amateur radio band !
  T_RigCtrlUserDefinedBand UserDefinedBands[RIGCTRL_NUM_USER_DEFINED_BANDS]; // <- loaded from a configuration file (TxBandEdges[] and the "bandplan" is only a DEFAULT for UserDefinedBands[])
  int    iNumUserDefinedBands;  // number of valid entries in UserDefinedBands[]; loaded from file (SERVER) or received via TCP/IP (CLIENT)
  BOOL   fUserDefinedBandsModified; // <- application may POLL and CLEAR this flag after taking notice
  BOOL   fUserDefinedBandsModifiedForCwNet; // <- similar flag to be "consumed" by CwNet.c,
          // when sending the "sysop-enabled bands" from server to clients.

# define RIGCTRL_NUM_USER_DEFINED_FREQUENCIES 100
  T_RigCtrlFreqMemEntry UserDefinedFrequencies[RIGCTRL_NUM_USER_DEFINED_FREQUENCIES];
  int    iNumUserDefinedFrequencies; // number of valid entries in UserDefinedFrequencies[]; also loaded from a file (SERVER side)
  BOOL   fUserDefinedFrequenciesModified; // <- application may POLL and CLEAR this flag after taking notice
  BOOL   fUserDefinedFrequenciesModifiedForCwNet; // <- similar flag to be "consumed" by CwNet.c,
          // when sending the "sysop-suggested frequencies" from server to clients.



# define RIGCTRL_NUM_BAND_STACKING_REGS 64 // Consider an HF/VHF/UHF "all-mode": (10+3+2) bands * 3 "stack levels" per band plus a few entries for what Icom calls "GENE" (generic frequencies)
  T_RigCtrlFreqMemEntry BandStackingRegs[RIGCTRL_NUM_BAND_STACKING_REGS];
  int    iNumBandStackingRegs; // number of valid entries in BandStackingRegs[]; determined in state RIGCTRL_POLLSTATE_BAND_STACKING_REGS
  BOOL   fBandStackingRegsModified;         // <- application may POLL and CLEAR this flag after taking notice
  BOOL   fBandStackingRegsModifiedForCwNet; // <- similar flag to be "consumed" by CwNet.c,
          // when sending band stacking registers from server to clients.


  //------ SPECTRUM SCOPE related (with CI-V, exchanged via command 0x27) -----
  int    iScopeOnOff;     // Icom: "Send/read the Scope ON/OFF status"
  int    iScopeMode;      // HERE, independently of the protocol, one of the following:
#        define RIGCTRL_SCOPE_MODE_OFF    0  // spectrum scope currently off, or not available
#        define RIGCTRL_SCOPE_MODE_FIXED  1  // "fixed frequency range"
#        define RIGCTRL_SCOPE_MODE_CENTER 2  // "centered around the VFO frequency"
  double dblScopeSpan_Hz;       // last value reported for Icom's "Scope span settings" (cmd 0x27 0x15)
  double dblScopeAttenuator_dB; // "Scope Attenuator setting" (IC-7851, n/a on IC-7300)
  double dblScopeRefLevel_dB;   // "Scope Reference level setting" in dB [RIGCTRL_PN_SCOPE_REF_LEVEL]
  double dblSpectrumCenterFreq_Hz;
  int    iScopeEdgeNumber; // "Edge number setting in the Fixed mode Scope", runs from 1 to 3 FOR EACH BAND
  int    iScopeHoldFlag;   // "Scope hold function status" : 0=hold OFF, 1=hold ON
  T_RigCtrlFrequencyRange ScopeFreqRange[RIGCTRL_MAX_FIXED_EDGE_SCOPE_FREQ_RANGES][RIGCTRL_MAX_FIXED_EDGES_PER_FREQ_RANGE];
  int    iScopeNumFreqRanges;
  int    iScopeNumEdgesPerFreqRange;
  int    iScopeDataCountdown_ms; // timer to monitor periodic 'scope data' reception
#        define RIGCTRL_SPECTRUM_TIMEOUT_MS 6000 /* large enough to tolerate "interference" from RS-BA1 */
  int    iScopeSpeed;      // Icom's crazy "Scope Sweep speed setting" (0x27 0x1A), may contain one of the following :
#        define RIGCTRL_SCOPE_SPEED_FAST 0x00 // Note the mad incompatibility with RIGCTRL_WATERFALL_SPEED_.. !
#        define RIGCTRL_SCOPE_SPEED_MID  0x01
#        define RIGCTRL_SCOPE_SPEED_SLOW 0x02
  int    iScopeCenterType; // may contain one of the following:
#        define RIGCTRL_SCOPE_CT_FILTER_CENTER 0
#        define RIGCTRL_SCOPE_CT_CARRIER_POINT 1
#        define RIGCTRL_SCOPE_CT_CARRIER_ABS   2 // .. whatever that means !
  int    iScopeVideoBW;    // may contain one of the following:
#        define RIGCTRL_SCOPE_VBW_NARROW 0
#        define RIGCTRL_SCOPE_VBW_WIDE   1
  int    iScopeDuringTX;   // 0=scope only on during receive, 1=scope also on during transmit
  int    iScopeMarkerPosType; // 0="filter center", 1="carrier point"
  int    iWaterfallSpeed;  // Icom's crazy command 0x1A, subcommand 0x05, and RIG-DEPENDENT sub-sub-command (BCD) = "Waterfall Speed"
          // (possible values are also RIGCTRL_SCOPE_SPEED_FAST/MID/SLOW,
          //  even though not compatible with what Icom uses internally.
          //  Since the four-digit-BCD "sub-sub-command" depends on the RIG MODEL
          //  -it's different for e.g. IC-7300 and IC-9700-, we need an extra
          //  READ/WRITE ACCESS FUNCTION to handle all these specialities anyway)

  //--- "Transceiver status" (RX/TX). With CI-V, exchanged via command 0x1C ---
  int    iTransmitting;  // Set when THE RIG ITSELF transmits:    0=receive, 1=transmit.  With CI-V, parsed from the respose for a READ with cmd=0x1C, sub=0x00.
                         // Accessable via RIGCTRL_PN_TRANSMITTING (read-only) .
                         // iTransmitting < 0 means "I haven't got a clue if the rig transmits or receives at the moment".
  int    iTransmitReqst; // Set when WE want the rig to transmit: 0="shall receive", 1="shall transmit".
                         // If RigCtrl_SetTransmitRequest() is called with a different state,
                         // RigControl will send a CI-V SET command with cmd=0x1C, sub=0x00,
                         //     and data=0x00 ("stop transmitting") or 0x01 ("start transmitting").
                         // Accessable via RIGCTRL_PN_TRANSMIT_REQUEST (read- and writeable parameter).
                         // Replaces what used to be BOOL fLocalPTTFlag .
                         // For more 'Hamlib' compatibility, iTransmitReqst and iTransmitting
                         // are not limited to 0 = "receive" or 1 = "transmit", but:
                         //     0 = RX,  1 =  TX,  2 = "TX mic", or 3 = "TX data".
                         // When treated as a kind of BOOLEAN value (on read),
                         //     anything POSITIVE means "transmit" .
  int    iATU_enabled;   // 0=automatic antenna tuner disabled, 1=enabled
  int    iXFC_pressed;   // 0='XFC'-button not pressed, 1=button pressed
  double dblRepeaterTone_Hz; // "repeater tone frequency". Icom: 0x1B 0x00 .
  double dblToneSquelch_Hz;  // "TSQL tone frequency".     Icom: 0x1B 0x01 .

  //----- Various "poti" settings: With CI-V, exchanged via command 0x14 .. ---
  //      From server to client(s), some of these settings are exchanged
  //      in a bandwidth-saving T_RigCtrl_PotiReport.
  int    iAudioVolume_Percent;
  int    iRFGain_Percent;
  int    iSquelchLevel_Percent;
  int    iNoiseReduction_Percent;
  int    iNoiseBlanker_Percent;
  int    iPassbandTuningPos1_Percent; // "inner ring" of "TWIN PBT", 50 % = center
  int    iPassbandTuningPos2_Percent; // "outer ring" of "TWIN PBT", 50 % = center
  int    iCWPitch_Hz; // here: in Hertz, not the strange CI-V internal unit
  int    iRFPowerSetting_Percent;
  int    iMicGain_Percent;
  int    iKeyerSpeed_WPM;      // range of Icom's built-in keyer : "0 %" = 6 WPM, "100 %" = 48 WPM
  int    iNotchPos_Percent;
  int    iCompSetting_Percent; // audio compressor SETTING (not "measurement")
  int    iBreakInDelay_Percent;
  int    iMonitorGain_Percent;
  int    iVoxGain_Percent;
  int    iAntiVoxGain_Percent;
  int    iBrightness_Percent;


  //------ Various "meter" readings (with CI-V, exchanged via command 0x15) -----
  //      From server to client(s), exchanged in a T_RigCtrl_MultiFunctionMeterReport..
  int    iSquelchStatus;           // (0 or 1) from 0x1501
  int    iSMeterLevel_dB;          // here: "decibel over S0",  not the strange CI-V internal unit
  int    iPowerMeterLevel_pcnt;    // *measured* output power in PERCENT, not the strange CI-V internal unit
  double dblSWRMeterValue;         // dimensionless "SWR" value, usual range 1.0 (perfect) to 3.0 (poor)
  int    iALCMeterLevel_pcnt;      // here: 0 = min, 100 = max, not the strange CI-V internal unit
  int    iCompMeterLevel_dB;       // here: in decibel, 0=no compression, 30 dB = maximum reading for IC-7300)
  int    iSupplyVoltage_mV;        // here: in millivolts, not the strange CI-V internal unit (241 = 16 V)
  int    iDrainCurrent_mA;         // here: in milliamps,  not the strange CI-V internal unit (241 = 25 A)
  int    iPATemperature_degC;      // power amplifier temperature in C (not sure how to read this via CI-V yet...)
  int    iADCOverflowStatus;       // ADC Overflow Flag, cmd 0x15 sub 0x07 in an IC-7300. Frequently polled by WFView. 


  //-- Preamp step, AGC speed, Noise blanker, Audio peak filter,
  //   noise reduction, auto notch, etc. With CI-V, via command 0x16) -----
  int    iPreampSet;        // RIGCTRL_PN_PREAMP_SETTING,       CI-V : 0x16 0x02
  int    iAGCSpeed;         // RIGCTRL_PN_AGC_SPEED,            CI-V : 0x16 0x12
  int    iNoiseBlankerON;   // RIGCTRL_PN_NOISE_BLANKER_ON,     CI-V : 0x16 0x22
  int    iNoiseReductON;    // RIGCTRL_PN_NOISE_REDUCT_ON,      CI-V : 0x16 0x40
  int    iAutoNotchON;      // RIGCTRL_PN_AUTO_NOTCH_ON,        CI-V : 0x16 0x41
  int    iRptrToneON;       // RIGCTRL_PN_RPTR_TONE_ON,         CI-V : 0x16 0x42
  int    iToneSquelchON;    // RIGCTRL_PN_TONE_SQUELCH_ON,      CI-V : 0x16 0x43
  int    iCompressON;       // RIGCTRL_PN_COMP_ON,              CI-V : 0x16 0x44
  int    iMonitorON;        // RIGCTRL_PN_MONITOR_ON,           CI-V : 0x16 0x45
  int    iVoxON;            // RIGCTRL_PN_VOX_ON,               CI-V : 0x16 0x46
  int    iBreakInMode;      // RIGCTRL_PN_BK_IN_MODE,           CI-V : 0x16 0x47 0=off, 1=semi-BK, 2=full-BK
  int    iManualNotchON;    // RIGCTRL_PN_MANUAL_NOTCH_ON,      CI-V : 0x16 0x48
  int    iManualNotchWidth; // RIGCTRL_PN_MANUAL_NOTCH_WIDTH,   CI-V : 0x16 0x57
  int    iTwinPeakFilterON; // RIGCTRL_PN_TWIN_PEAK_FILTER_ON,  CI-V : 0x16 0x4F
  int    iDialLockON;       // RIGCTRL_PN_DIAL_LOCK_ON,         CI-V : 0x16 0x50
  int    iDSP_SharpSoft;    // RIGCTRL_PN_DSP_SHARP_SOFT,       CI-V : 0x16 0x56
  int    iSSB_Tx_BW;        // RIGCTRL_PN_SSB_TX_BANDWIDTH,     CI-V : 0x16 0x58
  int    iSubbandON;        // RIGCTRL_PN_SUBBAND_ON,           CI-V : 0x16 0x59
  int    iSatelliteMode;    // RIGCTRL_PN_SATELLITE_MODE,       CI-V : 0x16 0x5A
  int    iImprovedIP3;      // RIGCTRL_PN_IMPROVED_IP3,         CI-V : 0x16 0x65. 0=off 1=on. Known as "IP+", "improved IP3", maybe just another "gain stage off", or ADC dithering ON. We don't care.




  //---- Various parameters accessed via RIG-SPECIFIC(!) 4-digit subcodes -----
  //    ...after Icom's CI-V command 0x1A 0x05. These 4-digit subcodes
  //       must be translated into PRIGCTRL_PN_xyz by
  //       RigCtrl_SubcodeFrom1A05_to_UnifiedParameterNumber() !
  int    iSSB_RX_HPF_LPF;
  int    iSSB_RX_BASS_LEVEL;
  int    iSSB_RX_TREBLE_LEVEL;
  int    iSSB_CW_sync_tuning;
  int    iCalibMarker;
  int    iAF2ACC_level;
  int    iSquelch_for_USB;
  int    iSidetone_on_USB; // Funny name in the IC-7300's "Setup" / "Connectors" : "ACC/USB AF Beep/Speech Output".
           // The Remote CW Keyer needs to know, and even CONTROL this setting,
           // because what Icom calls "Beep/Speech" includes the CW SIDETONE
           // generated LOCALLY inside the rig. We want to ACTIVATE the sidetone
           // on the "ACC/USB AF"-output when the SYSOP (local operator of the radio)
           // transmits in CW with the key directly plugged into the transceiver,
           // but DEACTIVATE the rig's sidetone-on-USB when keyed remotely,
           // because the 'echo' (operator's local sidetone plus audio played back
           // through via network while transmitting) is annoying. Thus the
           // implementation of the "Sidetone when keyed ON THE RIG"-option
           //                in RigCtrl_OnTransmitFlagChange() .
  int    iCIV_transceive; // IC-7300: "Send/read the CI-V transceive setting (0=off, 1=on)" 1A 05 0071
  int    iCIV_addr_USB_Rem;
  int    iCIV_for_ANT;
  int    iCIV_unlink_Remote;
  int    iCIV_USB_echo;
  int    iCW_PTTControlLineOnUSB; // RIGCTRL_PN_CIV_PTT_CONTROL_LINE. IC-7300: 0x1A 0x05 #0078
  int    iCW_KeyingLineOnUSB;     // RIGCTRL_PN_CIV_CW_KEYING_LINE.   IC-7300: 0x1A 0x05 #0079
  int    iRTTY_KeyingLineOnUSB;   // RIGCTRL_PN_CIV_RTTY_KEYING_LINE. IC-7300: 0x1A 0x05 #0080

  int    iIP_Plus;
  int    iNTP_Server_Access, iNTP_Access_Result;
  int    iAF_mute;
  int    iReferenceFrequencyOffset;    // IC-7300: "Send/read reference frequency" 1A 05 0058 (0..255)
  int    iAF_IF_OutputSelectOnACC_USB; // IC-7300: "Send/read AF/IF signal output to ACC/USB" 1A 05 0059 (0=AF, 1=IF)
  int    iAF_OutputLevelOnACC_USB;     // IC-7300: "Send/read AF output level to ACC/USB" 1A 05 0060 (0000..0255)
  int    iSquelchSignalOutputOnACC_USB;// IC-7300: "Send/read squelch function for the AF signal output to ACC/USB" 1A 05 0061 (0=off, 1=on)
  int    iIF_OutputLevelOnACC_USB; // IC-7300: "Send/read IF output level to ACC/USB" 1A 05 0063 (0000..0255)
  int    iModInputLevelFromACC;    // IC-7300: "Send/read MOD input level from ACC"   1A 05 0064 (0000..0255)
  int    iModInputLevelFromUSB;    // IC-7300: "Send/read MOD input level from USB"   1A 05 0065 (0000..0255)
  int    iModInputConnector_NoData;// IC-7300: "Send/read MOD input connector during DATA OFF" 1A 05 0066 : 0=MIC, 1=ACC, 2=MIC/ACC, 3=USB, 4=MIC/USB
  int    iModInputConnector_Data;  // IC-7300: "Send/read MOD input connector during DATA" 1A 05 0067 : 0=MIC, 1=ACC, 2=MIC/ACC, 3=USB, 4=MIC/USB


  //-- Even more settings (in CI-V, controlled via command 0x0E..0x10) --
  int    iScanMode;
  int    iSplitMode;     // one of the following :
#        define RIGCTRL_SPLIT_MODE_OFF      0 // neither "Split" not "Duplex for Repeater operation"
#        define RIGCTRL_SPLIT_MODE_ON       1 // classic "Split" mode, usually a few kHz
#        define RIGCTRL_SPLIT_MODE_DUP_NEG  2 // "DUP- operation" (IC-9700)
#        define RIGCTRL_SPLIT_MODE_DUP_POS  3 // "DUP+ operation" (IC-9700)
#        define RIGCTRL_SPLIT_MODE_REPEATER_SIMPLEX 4 // <- IC-9700 extravaganza ? "DD Repeater Simplex mode (RPS)"
  int    iTuningStep_Hz;
  int    iAttenuator_dB;
  int    iRIT_freq_Hz;   // RIGCTRL_PN_RIT_FREQ, 'Receiver Incremental Tuning' in Hertz.
  int    iXIT_freq_Hz;   // RIGCTRL_PN_XIT_FREQ, 'Transmitter Incremental Tuning' in Hertz.
  int    iRIT_ON;        // RIGCTRL_PN_RIT_ENABLED, 'Receiver Incremental Tuning' ON(1) / OFF(0).
  int    iXIT_ON;        // RIGCTRL_PN_XIT_ENABLED, 'Transmitter Incremental Tuning' ON(1) / OFF(0).

  // GPS data (from radios with built-in GPS receivers)
  double dblGpsLat_deg, dblGpsLon_deg;

  // Lock-free buffer (circular FIFO) for a few spectra stored by RigControl .
  // There may(!) be MULTIPLE READERS, each of them has his own "head index"
  // but there is only one "tail index". If head_index == tail_index,
  // there are no new spectra available for the display. The entry at <head_index>
  // should not be accessed (read) by the application because RigControl.c
  // may already be "filling" it in another thread or task.
# define RIGCTRL_SPECTRUM_FIFO_SIZE 4
  T_RigCtrl_Spectrum SpectrumFifo[ RIGCTRL_SPECTRUM_FIFO_SIZE ];
  int    nSpectrumBinsUsed;    // number of frequency bins acually used in the LASTEST entry in SpectrumFifo[]
  int    iScopeFifoHeadIndex;  // index for WRITING the next entry into SpectrumFifo[]
  BOOL   fScopeFifoFull;  // Set to TRUE when iScopeFifoHeadIndex has wrapped around.
         // Then all entries (except the one at iScopeFifoHeadIndex) are valid.
  int    iScopeBinCounter; // frequency bin counter for assembling 'fragments' of spectra
  T_RigCtrlFrequencyRange ScopeFreqRangeFromFirstFragment;

  // Internal states, used for reading a sequence of parameters from the "radio" port, etc..
  enum // tracks calls of RigCtrl_Init() and RigCtrl_Close().
   { initState_ColdStart = 0,    // didn't call RigCtrl_Init() yet
     initState_Opened,           // last call was RigCtrl_Init() or RigCtrl_ReConnect(); allow processing UDP packets or chunks of CI-V
     initState_Closed            // last call was RigCtrl_Close()
   } initState;
  int    iParameterPollingState; // RIGCTRL_POLLSTATE_PASSIVE, RIGCTRL_POLLSTATE_START_RD, etc. Used shortly after connecting, to poll the most important parameters
  int    iParameterPollingSubState;  // sub-index or similar; meaning depends on iParameterPollingState
  int    iParameterPollingArrayIndex; // index for reading e.g. TWELVE "Fixed edge frequency settings",
                                      //  or THREE "Stack levels" per Band Stacking Register, etc, etc.
  BOOL fGotBasicParams; // TRUE when the "basic" parameters have been read from the radio
                        //      (on the 'radio control port' and ready to 'serve' external clients.
                        // Related to iParameterPollingState, but fGotBasicParams remains SET
                        // even if iParameterPollingState switches from RIGCTRL_POLLSTATE_DONE to something else.
  int    iPostponedSetMessages;  // bitwise combination of (1<<RIGCTRL_MSGTYPE_RADIO_ID) .. (1<<RIGCTRL_MSGTYPE_BAND_SCOPE).
                                 // Only applies to "Set"-commands for the LOCALLY CONNECTED RADIO, i.e. on the server side.
                                 // Setting a flag in iPostponedSetMessages does not cause transmissions via CwNet.c (TCP/IP).
  double dblPostponedFrequency;  // VFO frequency that RigCtrl_SetVFOFrequency() should set but couldn't
  int    iPostponedOpMode;       // .. etc, etc, (parameters pending to be SET)
# define RIGCTRL_READ_WRITE_PN_FIFO_SIZE 5 // FIFO for RigCtrl_QueueUpCmdToReadUnifiedPN() and RigCtrl_QueueUpCmdToWriteUnifiedPN()
  int    iReadWritePNFifo[RIGCTRL_READ_WRITE_PN_FIFO_SIZE]; // FIFO to queue up parameter numbers to read or write
  int    iReadWritePNFifoHead, iReadWritePNFifoTail;  // circular buffer indices into iReadWritePNFifo[]

  // Multiple *SERIAL* port instances, to allow 3rd party applications
  // to control the same radio over a serial port or 'RIGCTLD'-compatible TCP/IP.
  // Spectrum Lab (or the Remote CW Keyer) lets those clients think
  // they talk to the radio "directly" (via virtual NULL-modem like com0com):
# define RIGCTRL_NUM_PORT_INSTANCES 1+RIGCTRL_MAX_CLIENT_PORTS  // [0] for the RADIO PORT, the others for EXTERNAL CLIENTS / "Aux COM Ports"
  T_RigCtrl_PortInstance PortInstance[RIGCTRL_NUM_PORT_INSTANCES];
    // Index into the above array is iRigControlPort :
    //        0 = RIGCTRL_PORT_RADIO, 1 = RIGCTRL_PORT_AUX_COM_1, ... !
    // Note : To keep those ugly details about accessing serial ports
    //        under windows out of this sourcecode module, the windows-specific
    //        "COM port settings" (and the COM port interface itself) are NOT
    //        part of RigControl.c / RigControl.h !


  // Transmit FIFO for the RADIO port. Used for buffering messages that
  // need to be forwarded from CLIENTS(!) to the radio, if the RADIO port
  // is currently occupied (radio still busy from a previously transmitted
  // command, for which the response is still pending).
  // Filled      by RigCtrl_AppendMessageToRadioPortTxFifo(),
  // and drained by RigCtrl_SendMessageFromRadioPortTxFifo().
#define RIGCTRL_RADIO_PORT_TX_FIFO_SIZE 2048 /* # bytes, must be a power of two */
  BYTE bRadioPortTxFifo[RIGCTRL_RADIO_PORT_TX_FIFO_SIZE];
  int  iRadioPortTxFifoHead, iRadioPortTxFifoTail;

  // Debugging stuff, including a FIFO for spectrum lab's "CAT traffic monitor":
  char sz255LastError[256]; // buffer for 'sprintf'-generated error messages
  T_RigCtrlTrafficLogEntry TrafficLog[RIGCTRL_MAX_TRAFFIC_LOG_ENTRIES];
  int    iTrafficLogHeadIndex; // circular FIFO head index (post-incremented, wraps from RIGCTRL_MAX_TRAFFIC_LOG_ENTRIES-1 to zero)

  // Because there may be multiple readers for the 'Traffic Log', each of them
  // has its own TAIL INDEX into TrafficLog[], and the tail indices are an array:
# define RIGCTRL_NUM_TRAFFIC_LOG_READERS 2
  int    iTrafficLogTailIndex[RIGCTRL_NUM_TRAFFIC_LOG_READERS];
  // Each traffic-log-reader identifies himself as parameter 'iTrafficReader'
  // in RigCtrl_GetNumEntriesInTrafficLog() and RigCtrl_ReadTrafficLog(), using one of these values:
# define RIGCTRL_TRAFFIC_READER_GUI            0
# define RIGCTRL_TRAFFIC_READER_LOGFILE_WRITER 1

  BOOL   fTrafficLogFull;      // TRUE when all <RIGCTRL_MAX_TRAFFIC_LOG_ENTRIES> entries are IN USE.
  // (but in most cases, the circular head index 'iTrafficLogHeadIndex' keeps incrementing,
  //  unless RigCtrl_TrafficMonitor.iDisplayOptions.RIGCTRL_TMON_DISPLAY_OPTION_PAUSE_ON_FULL_FIFO is set.)

  DWORD  dwMessageFilterForLog;  // bit combination of the following :
# define RIGCTRL_MSGFILTER_ECHO              (1<<0)
# define RIGCTRL_MSGFILTER_PERIODIC_POLL     (1<<1)
# define RIGCTRL_MSGFILTER_SPECTRUM          (1<<2)
# define RIGCTRL_MSGFILTER_FREQUENCY_REPORT  (1<<3)
# define RIGCTRL_MSGFILTER_ANY_KNOWN_COMMAND (1<<8)




  // ex: char* pszCommentForTxFunction; // <- added 2023-07 to avoid yet another parameter for the (UDP-) transmit function called from RigCtrl_SendAndLogMessage(),
                                        // in combination with RIGCTRL_TMON_DISPLAY_OPTION_SHOW_UDP_PAYLOAD  .
  // ex: int iMsgTypeForTxFunction; // <- added 2023-07 so RigCtrl_TxFunctionForCivViaUDP() can pass 'iMsgType' on to RigCtrl_AddToTrafficLog(),
                                    //    especially for flags like RIGCTRL_MSGTYPE_FLAG_SUGGEST_NEW_HEADLINE_FOR_TRAFFIC_LOG .

  long   nCompleteSpectraReceived; // counter for COMPLETELY received spectra (not fragments) for WF speed check
  double dblUnixTimeOfFirstSpectrum; // = TIM_GetCurrentUTC() on reception of the FIRST spectrum

#if( SWI_USE_HAMLIB_SERVER ) // build an application with integrated "Hamlib Net rigctld"-compatible server ?
  void *pvHamlibServer;      // 'link'  from the T_RigCtrlInstance to a Hamlib-Server (T_HLSrv*)
#endif // SWI_USE_HAMLIB_SERVER ?

  // Queue for all 'Unified parameter numbers' that still need to be sent from
  // CLIENT to SERVER using the TCP/IP based protocol implemented in e.g.
  // Remote_CW_Keyer/CwNet.c .   Details/Purpose in the implementation
  //                             of CwNet_OnPoll() .
  //     If there is sufficient space in the outbound network buffer, CwNet_OnPoll()
  //     will compare the queue's HEAD index (iTxParameterQueueHeadIndex) with its own,
  //     "per-client" TAIL index, and if they are different, append another "SET xyz"-command,
  //     along with the CURRENT VALUE retrieved via 'unified parameter number' in this queue:
# define RIGCTRL_TX_PARAMETER_QUEUE_LENGTH 16
  int iTxParameterQueue[RIGCTRL_TX_PARAMETER_QUEUE_LENGTH];   // <- only contains PARAMETER NUMBER, no VALUES !
  int iTxParameterQueueHeadIndex, iTxParameterQueueTailIndex; // <- 0..RIGCTRL_TX_PARAMETER_QUEUE_LENGTH-1
      // '--> New items (unified parameter numbers) are added to this queue
      //      via RigCtrl_AddParamToTxQueueForCwNet().
  T_TIM_Stopwatch sw_LastUpdateOfBasicParams;

#if(SWI_SUPPORT_YAESU_5_BYTE_CAT) // support the OLD, UGLY, "YAESU 5-BYTE-CAT" (see KA7OEI:"Meow")
  T_Yaesu5ByteControl Y5B; // instance data for module 'Yaesu5Byte.c', optionally running under the hood of RigControl.c
#endif // SWI_SUPPORT_YAESU_5_BYTE_CAT ?


  CRITICAL_SECTION criticalSection; // Win32 object for mutual exclusion, here:
                     // to protect the RADIO PORT, the TRAFFIC MONITOR, and maybe
                     // a few other elements from multithreading issues.
                     // See RigCtrl_Enter/LeaveCriticalSection().
  DWORD dwInitMagic; // == DSP_INIT_MAGIC when initialized
                     // (don't assume the content of T_CwDSP is nicely filled
                     //  with zeros when the application calls CwDSP_InitInstanceWithDefaults(),
                     //  and don't assume the application will call that function only ONCE)
#  define RIGCTRL_INIT_MAGIC 0x31415925  // not magic but .. almost PI !



} T_RigCtrlInstance; // -> main instance = 'RigControlData'



typedef struct tRigCtrl_RadioInfo // -> T_RigCtrl_RadioInfo RigCtrl_RadioInfo_CIV[], RigCtrl_RadioInfo_Y5B[], etc .
{ int  iDefaultAddress; // for Icom radios, the DEFAULT CI-V ADDRESS also identifies the model !
                        // Possible values :  iDefaultAddress = -1 = "OFF (No RigControl)",
                        //                    iDefaultAddress = 0  = "Auto-Detect",
                        //  or RIGCTRL_DEF_ADDR_IC_7300, etc etc

  const char *pszName;  // user-friendly name like "IC-7300"
  int  iCapabilities;   // <- don't take anything for granted. Almost nothing could be tested.
                        // Zero means (you guessed it) : "I don't know anything about this radio".
# define RIGCTRL_CAPS_NONE                 0x0000
# define RIGCTRL_CAPS_ALL                  0xFFFF
# define RIGCTRL_CAPS_POWER_ON_OFF_VIA_CAT 0x0010 /* set if the rig can be powered on/off via CAT command*/
# define RIGCTRL_CAPS_SPECTRUM_VIA_CAT     0x0100
# define RIGCTRL_CAPS_HAS_IQ_OUTPUT        0x0200 /* the IC-7300 definitely does NOT ! */
# define RIGCTRL_CAPS_HAS_NO_IQ_OUTPUT     0x0400 /* flag only set if we SURELY know.. */

   DWORD dwBands;  // bitwise combination of the following amateur radio bands.
         // For traditional reasons, most use APPROXIMATE WAVELENGTHS, not FREQUENCIES.
# define RIGCTRL_BAND_160M   (1UL<<0)
# define RIGCTRL_BAND_80M    (1UL<<1)
# define RIGCTRL_BAND_60M    (1UL<<2)
# define RIGCTRL_BAND_40M    (1UL<<3)
# define RIGCTRL_BAND_30M    (1UL<<4)
# define RIGCTRL_BAND_20M    (1UL<<5)
# define RIGCTRL_BAND_17M    (1UL<<6)
# define RIGCTRL_BAND_15M    (1UL<<7)
# define RIGCTRL_BAND_12M    (1UL<<8)
# define RIGCTRL_BAND_10M    (1UL<<9)
# define RIGCTRL_BAND_GENERIC (1UL<<10) // dummy for Icom's "GENE" (generic) for any TX band outside the ones listed above
# define RIGCTRL_BAND_RSV11  (1UL<<11) // reserve for a future SHORTWAVE Ham radio band
# define RIGCTRL_BAND_6M     (1UL<<12) // low VHF bands: 50 MHz, ..
# define RIGCTRL_BAND_4M     (1UL<<13) // 70 MHz, only 'temporarily tolerated' in DL !
# define RIGCTRL_BAND_2M     (1UL<<14) // 144 .. 145 MHz, more in some ITU zones
# define RIGCTRL_BAND_R_VHF  (1UL<<15) // reserve for a future VHF Ham radio band
# define RIGCTRL_BAND_70CM   (1UL<<16) // UHF bands: 430 .. 440 MHz, more in some ITU zones, less in some countries
# define RIGCTRL_BAND_23CM   (1UL<<17) // circa 1.2 GHz but beware.. MANY national specialities!
# define RIGCTRL_BAND_13CM   (1UL<<18) // circa 2.3 GHz (satellite uplink a bit higher)
# define RIGCTRL_BAND_9CM    (1UL<<19) // circa 3.4 GHz
# define RIGCTRL_BAND_6CM    (1UL<<20) // circa 5.6 GHz (supported by IC-905 ?)
# define RIGCTRL_BAND_3CM    (1UL<<21) // circa 10 GHz  (also downlink for QO-100)
  // "Exotic bands" (not microwave) use even higher-valued bits:
# define RIGCTRL_BAND_2200M  (1UL<<24) // Longwave (136 kHz)
# define RIGCTRL_BAND_600M   (1UL<<25) // Medium wave (472 kHz)
# define RIGCTRL_BAND_MAX_BITNUM   25

  // Frequenctly seen combinations of the above bits:
# define RIGCTRL_BAND_ALL_SHORTWAVE (RIGCTRL_BAND_160M|RIGCTRL_BAND_80M|RIGCTRL_BAND_40M|RIGCTRL_BAND_30M|RIGCTRL_BAND_20M|RIGCTRL_BAND_17M|RIGCTRL_BAND_15M|RIGCTRL_BAND_12M|RIGCTRL_BAND_10M)
         // '---> Note the absence of e.g. the 60-meter-band and 11 meter (CB) here !


} T_RigCtrl_RadioInfo; // -> T_RigCtrl_RadioInfo RigCtrl_RadioInfo_CIV[], RigCtrl_RadioInfo_Y5B[], etc .


//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
// Structures used in UDP packets compatible with Icom's "RS-BA" software.
//   Based on Wfview's "packettypes.h", but heavily modified to get rid of
//   the Qt-specific data types, for example "quint8/16/32".
//   Kudos to whoever spent days reverse-engineering Icom's "RS-BA" protocol
//   with Wireshark !  It's a shame that this information isn't officially
//   available from Icom.
// Anything that is really Icom-specific has the prefix "ICOM_"
//   or "ICOM_PKT_" for "Icom-specific UDP packet" .
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


#pragma pack(push, 1)

// Fixed Size Packets
#define ICOM_PKT_CONTROL_SIZE            0x10
#define ICOM_PKT_WATCHDOG_SIZE           0x14
#define ICOM_PKT_PING_SIZE               0x15
#define ICOM_PKT_CIV_HEADER_SIZE         ICOM_PKT_PING_SIZE // actually, almost the same HEADER(!) as for "Ping" !
#define ICOM_PKT_OPENCLOSE_SIZE          0x16
#define ICOM_PKT_RETRANSMIT_RANGE_SIZE   0x18
#define ICOM_PKT_TOKEN_SIZE              0x40
#define ICOM_PKT_STATUS_SIZE             0x50
#define ICOM_PKT_LOGIN_RESPONSE_SIZE     0x60
#define ICOM_PKT_LOGIN_SIZE              0x80
#define ICOM_PKT_CONNINFO_SIZE           0x90
#define ICOM_PKT_CAPABILITIES_SIZE       0xA8

// Variable size packets + payload (via Icom's UDP..)
#define ICOM_MIN_PACKET_SIZE_FOR_CIV     0x16 /* 0x15 = 21 would be an "empty CI-V packet", not distnguishable from "PING"! */
#define ICOM_PKT_AUDIO_SIZE              0x18
#define ICOM_PKT_DATA_SIZE               0x15



// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
// UDP packet types ordered by their PAYLOAD LENGTH, converted from WFview,
//     with packet numbers in parentheses from a Wireshark dump shown in
//     RigControl.c : RigCtrl_Handler(), RIGCTRL_CONNSTATE_CONNECTING .
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
// 0x10 length control packet (connect/disconnect/idle.)  (35)(36)(37)(38)
typedef union tIcomControlPacket
{
  struct             // Byte-Offset into UDP payload | [ : possible values]
   { uint32_t len;   // 0x00
     uint16_t type;  // 0x04 : 0x00="idle", 0x01="missing packet/please re-send", 0x03="connect",
                     //        0x04="connected", 0x05="disconnect", 0x06="are you ready / I am ready" ?
#      define ICOM_CONTROL_PACKET_TYPE_IDLE       0x00
#      define ICOM_CONTROL_PACKET_TYPE_RESEND     0x01
#      define ICOM_CONTROL_PACKET_TYPE_CONNECT    0x03 // possibly also means "Are You There ?" ..
#      define ICOM_CONTROL_PACKET_TYPE_CONNECTED  0x04 // possibly also means "I Am Here", but this is NOT a "ping" (connection speed test)
#      define ICOM_CONTROL_PACKET_TYPE_DISCONNECT 0x05
#      define ICOM_CONTROL_PACKET_TYPE_ARE_YOU_READY 0x06 // udphandler.cpp line 1068: 'sendControl(false, 0x06, 0x01); // Send Are you ready - untracked'
#      define ICOM_CONTROL_PACKET_TYPE_I_AM_READY    0x06 // udphandler.cpp line 160 : 'This is "I am ready" in response to "Are you ready" so send login'

     uint16_t seq;     // 0x06 "seq" = sequence number of RE-TRANSMISSION,
     // see IcomPkt_SendControl() -> IcomPkt_Send() with SEND_PKT_TRACKED !

     uint32_t sentid;  // 0x08
     uint32_t rcvdid;  // 0x0c
   } s;  // .s = "structured"
  BYTE packet[ICOM_PKT_CONTROL_SIZE];
} T_IcomControlPacket;


// 0x14 length watchdog packet        (39)(42)
typedef union tIcomWatchdogPacket {
    struct {                 // Byte-Offset into UDP payload:
        uint32_t len;        // 0x00
        uint16_t type;       // 0x04
        uint16_t seq;        // 0x06
        uint32_t sentid;     // 0x08
        uint32_t rcvdid;     // 0x0c
        uint16_t secondsa;   // 0x10
        uint16_t secondsb;   // 0x12
    } s;
    BYTE packet[ICOM_PKT_WATCHDOG_SIZE];
} T_IcomWatchdogPacket;


// 0x15 length ping packet
// Also used for the slightly different civ header packet.
typedef union tIcomPingPacket
{
  struct // Details ONLY in file:///C:/cbproj/SpecLab/RigControl/Icom_UDP_Info.htm#CIV_data_packet_header !
    {                       // Byte-Offset into the UDP payload for "Ping" and "CI-V over UDP":
        uint32_t len;       // 0x00
        uint16_t type;      // 0x04
#      define ICOM_PING_PACKET_TYPE_PING  0x07
#      define ICOM_PING_PACKET_TYPE_CIV   0x00
        uint16_t seq;       // 0x06
        uint32_t sentid;    // 0x08  // set to pUdpPortInst->myId somewhere, which in fact depends on "my IP address and my ephemeral UDP port number"
        uint32_t rcvdid;    // 0x0c  // set to pUdpPortInst->remoteId somewhere, which is what THE PEER has decided to use in certain 'Control' packets
        BYTE     reply;     // 0x10 .. one of the following:
#      define ICOM_PING_PACKET_REPLY_PING_REQUEST  0x00
#      define ICOM_PING_PACKET_REPLY_PING_RESPONSE 0x01
#      define ICOM_PING_PACKET_REPLY_CIV 0xC1   // CI-V-"header" (followed by payload) ? another of those former 'magic numbers' in *.cpp; used in udpCivData::send()
        union { // This contains differences between the send/receive packet
            struct { // 'Ping'
                uint32_t time;    // 0x11..0x14
            };
            struct { // 'Data' (CI-V-"header" for UDP; payload follows in the same datagram)
                uint16_t datalen; // 0x11..0x12  <- number of CI-V PAYLOAD bytes that follow. LITTLE ENDIAN.
                uint16_t sendseq; // 0x13..0x14  <- STUPID BIG ENDIAN .. WHY ?!
            };
        } u;
    } s;
    BYTE packet[ICOM_PKT_PING_SIZE]; // <- this does NOT include the CI-V payload - ONLY THE "CI-V Header" !
} T_IcomPingPacket, T_IcomCiVPacketHeader;

// 0x16 length open/close packet .  For speculations about the purpose, see IcomPkt_SendOpenClose().
typedef union tIcomOpenClosePacket
{
  struct
   {                      // Byte-Offset into UDP payload:
     uint32_t len;        // 0x00
     uint16_t type;       // 0x04
     uint16_t seq;        // 0x06
     uint32_t sentid;     // 0x08
     uint32_t rcvdid;     // 0x0c
     uint16_t data;       // 0x10
#      define ICOM_OPENCLOSE_PACKET_DATA_01C0 0x01C0 // <- no idea what this means - nothing else was seen yet
              // replace the above with more descriptive names as soon as we know the purpose
     BYTE     unused;     // 0x12 (not 0x11 as shown in WFview's "packettypes.h")
     uint16_t sendseq;    // 0x13
     char magic;          // 0x15
#      define ICOM_OPENCLOSE_PACKET_MAGIC_CLOSE 0x00
#      define ICOM_OPENCLOSE_PACKET_MAGIC_OPEN  0x04

    } s;
    BYTE packet[ICOM_PKT_OPENCLOSE_SIZE];
} T_IcomOpenClosePacket;


// 0x18 length audio packet
typedef union tIcomAudioPacket
{
  struct
   {                      // Byte-Offset into UDP payload:
     uint32_t len;        // 0x00
     uint16_t type;       // 0x04
     uint16_t seq;        // 0x06
     uint32_t sentid;     // 0x08
     uint32_t rcvdid;     // 0x0c
     uint16_t ident;      // 0x10
     uint16_t sendseq;    // 0x12
     uint16_t unused;     // 0x14
     uint16_t datalen;    // 0x16
   } s;
   BYTE packet[ICOM_PKT_AUDIO_SIZE];
} T_IcomAudioPacket;

// 0x18 length retransmit_range packet
typedef union tIcomRetransmitRangePacket
{
  struct
   {                      // Byte-Offset into UDP payload:
     uint32_t len;        // 0x00
     uint16_t type;       // 0x04
     uint16_t seq;        // 0x06
     uint32_t sentid;     // 0x08
     uint32_t rcvdid;     // 0x0c
     uint16_t first;      // 0x10
     uint16_t second;     // 0x12
     uint16_t third;      // 0x14
     uint16_t fourth;     // 0x16
   } s;
  BYTE packet[ICOM_PKT_RETRANSMIT_RANGE_SIZE];
} T_IcomRetransmitRangePacket;


// 0x40 length token packet
typedef union tIcomTokenPacket
{
  struct
   {                            // Byte-Offset into UDP payload:
     uint32_t len;              // 0x00  // value always ICOM_PKT_TOKEN_SIZE = 64 ?
     uint16_t type;             // 0x04  // leave this ZERO when SENDING this from client to server ?
#      define ICOM_TOKEN_PACKET_TYPE_IGNORE_RESPONSE 0x01 // in udpHandler::dataReceived() : response only processed if in->type != 0x01 ?!
     uint16_t seq;              // 0x06  // in udpHandler::sendToken(), left zero; but in sendTrackedPacket(), set to 'sendSeq' !
     uint32_t sentid;           // 0x08  // in udpHandler::sendToken(), set to 'myId'
     uint32_t rcvdid;           // 0x0c  // in udpHandler::sendToken(), set to 'remoteId'
     BYTE     unuseda[3];       // 0x10
     uint16_t code;             // 0x13  : 0x0130 for "token removal / request"
#      define ICOM_TOKEN_PACKET_CODE_S2C   0x0130 // value used in all(?) Token packets from client to server ?
     uint16_t magic; /*ex:res*/ // 0x15  : 1="token removal", 2="token request/update/renew" ?
#      define ICOM_TOKEN_PACKET_MAGIC_REMOVAL  0x01
#      define ICOM_TOKEN_PACKET_MAGIC_REQUEST  0x02
#      define ICOM_TOKEN_PACKET_MAGIC_RESPONSE 0x05 // response to a 'Token request' ?
     // Seen somewhere in the Qt-stuff: " connect(tokenTimer, &QTimer::timeout, this, std::bind(&udpHandler::sendToken, this, 0x05));"
     // Oh dear. Too obfuscated for non-Qt-experts.
     BYTE     innerseq;         // 0x17  // in udpHandler::sendToken(), set to authSeq++
     BYTE     unusedb;          // 0x18
     BYTE     unusedc;          // 0x19
     uint16_t tokrequest;       // 0x1a  // in udpHandler::dataReceived(), copied from in->tokrequest, filled in in sendToken()
     uint32_t token;            // 0x1c  // in udpHandler::dataReceived(), copied from in->token, filled in in sendToken()
     BYTE     unusedd[7];       // 0x20
     uint16_t commoncap;        // 0x27
     BYTE     unuseddd[2];      // 0x29
     BYTE     identa;           // 0x2b
     uint32_t identb;           // 0x2c
     uint32_t response;         // 0x30  // possibly a SIGNED int32, -1 = 0xFFFFFFFF = "Radio rejected token renewal" ?
#      define ICOM_TOKEN_PACKET_RESPONSE_ACCEPTED 0x00000000 // "Radio accepted token renewal" ?
#      define ICOM_TOKEN_PACKET_RESPONSE_REJECTED 0xFFFFFFFF // "Radio rejected token renewal" ?

     BYTE     unusede[12];      // 0x34
    } s;
  BYTE packet[ICOM_PKT_TOKEN_SIZE];
} T_IcomTokenPacket;


// 0x50 length login status packet
typedef union tIcomStatusPacket
{
  struct
   {                            // Byte-Offset into UDP payload:
     uint32_t len;              // 0x00
     uint16_t type;             // 0x04
#      define ICOM_STATUS_PACKET_TYPE_IGNORE_ME 0x01 // in udpHandler::dataReceived() : response only processed if in->type != 0x01 ?!
     uint16_t seq;              // 0x06
     uint32_t sentid;           // 0x08
     uint32_t rcvdid;           // 0x0c
     BYTE     unuseda[3];       // 0x10
     uint16_t code;             // 0x13
     uint16_t res;              // 0x15
     BYTE     innerseq;         // 0x17
     BYTE     unusedb;          // 0x18
     BYTE     unusedc;          // 0x19
     uint16_t tokrequest;       // 0x1a
     uint32_t token;            // 0x1c
     BYTE     unusedd[6];       // 0x20
     uint16_t unknown;          // 0x26
     BYTE     unusede;          // 0x28
     BYTE     unusedf[2];       // 0x29
     BYTE     identa;           // 0x2b
     uint32_t identb;           // 0x2c
     uint32_t error;            // 0x30
     BYTE     unusedg[12];      // 0x34
     BYTE     disc;             // 0x40
     BYTE     unusedh;          // 0x41
     uint16_t civport;          // 0x42 // Sent bigendian
     uint16_t unusedi;          // 0x44 // Sent bigendian
     uint16_t audioport;        // 0x46 // Sent bigendian
     BYTE     unusedj[7];       // 0x49
    } s;
  BYTE packet[ICOM_PKT_STATUS_SIZE];
} T_IcomStatusPacket;

// 0x60 length login status packet
typedef union tIcomLoginResponsePacket
{
  struct
   {                            // Byte-Offset into UDP payload:
     uint32_t len;              // 0x00
     uint16_t type;             // 0x04
#      define ICOM_LOGIN_RESPONSE_TYPE_IGNORE_ME 0x01 // haven't got a clue about what this '0x01' REALLY means !
     uint16_t seq;              // 0x06
     uint32_t sentid;           // 0x08
     uint32_t rcvdid;           // 0x0c
     BYTE     unuseda[3];       // 0x10
     uint16_t code;             // 0x13
     uint16_t res;              // 0x15
     BYTE     innerseq;         // 0x17
     BYTE     unusedb;          // 0x18
     BYTE     unusedc;          // 0x19
     uint16_t tokrequest;       // 0x1a
     uint32_t token;            // 0x1c
     uint16_t authstartid;      // 0x20
     BYTE     unusedd[14];      // 0x22
     uint32_t error;            // 0x30
#      define ICOM_LOGIN_RESPONSE_ERROR_NO_ERROR                 0x00000000
#      define ICOM_LOGIN_RESPONSE_ERROR_INVALID_USER_OR_PASSWORD 0xfeffffff
     BYTE     unusede[12];      // 0x34
     BYTE     connection[16];   // 0x40
     BYTE     unusedf[16];      // 0x50
    } s;
  BYTE packet[ICOM_PKT_LOGIN_RESPONSE_SIZE];
} T_IcomLoginResponsePacket;


// 0x80 length login packet     (41)
typedef union tIcomLoginPacket
{
  struct
   {                            // Byte-Offset into UDP payload:
     uint32_t len;              // 0x00
     uint16_t type;             // 0x04
     uint16_t seq;              // 0x06
     uint32_t sentid;           // 0x08
     uint32_t rcvdid;           // 0x0c
     BYTE     unuseda[3];       // 0x10
     uint16_t code;             // 0x13
     uint16_t res;              // 0x15
     BYTE     innerseq;         // 0x17
     BYTE     unusedaa;         // 0x18;
     BYTE     unusedb;          // 0x19
     uint16_t tokrequest;       // 0x1a
     uint32_t token;            // 0x1c
     BYTE     unusedc[32];      // 0x20
     BYTE     username[16];     // 0x40
     BYTE     password[16];     // 0x50
     char     name[16];         // 0x60
     BYTE     unusedf[16];      // 0x70
   } s;
  BYTE packet[ICOM_PKT_LOGIN_SIZE];
} T_IcomLoginPacket;

// length 0x90 = #144 : "conninfo and stream request packet"
typedef union tIcomConnInfoPacket
{
  struct
   {                            // Byte-Offset into UDP payload:
     uint32_t len;              // 0x00
     uint16_t type;             // 0x04
#      define ICOM_CONNINFO_TYPE_IGNORE_ME 0x01 // <- not sure what this REALLY means
     uint16_t seq;              // 0x06
     uint32_t sentid;           // 0x08
     uint32_t rcvdid;           // 0x0c
     BYTE     unuseda[3];       // 0x10
     uint16_t code;             // 0x13
     uint16_t res;              // 0x15
     BYTE     innerseq;         // 0x17
     BYTE     unusedaa;         // 0x18
     BYTE     unusedb;          // 0x19
     uint16_t tokrequest;       // 0x1a
     uint32_t token;            // 0x1c
     uint16_t authstartid;      // 0x20
     BYTE     unusedd[5];       // 0x22
     uint32_t commoncap;        // 0x27
     BYTE     identa;           // 0x2b
     uint32_t identb;           // 0x2c
     BYTE     unusedf[16];      // 0x30
     char     name[16];         // 0x40
     char     unusedg[16];      // 0x50
     union { // This contains differences between the send/receive packet
      struct { // Receive
        uint32_t busy;          // 0x60
        char     computer[16];  // 0x64
        BYTE     unusedi[16];   // 0x74
        uint32_t ipaddress;     // 0x84
        BYTE     unusedj[8];    // 0x78
       }rx;
      struct { // Send
        char     username[16];  // 0x60
        BYTE     rxenable;      // 0x70
        BYTE     txenable;      // 0x71
        BYTE     rxcodec;       // 0x72
        BYTE     txcodec;       // 0x73
        uint32_t rxsample;      // 0x74
        uint32_t txsample;      // 0x78
        uint32_t civport;       // 0x7c
        uint32_t audioport;     // 0x80
        uint32_t txbuffer;      // 0x84
        BYTE     convert;       // 0x88
        BYTE     unusedl[7];    // 0x89
       }tx;
     }u;
   } s;
  BYTE packet[ICOM_PKT_CONNINFO_SIZE];
} T_IcomConnInfoPacket;


// 0xA8 length capabilities packet
typedef union tIcomCapabilitiesPacket
{
  struct
   {                            // Byte-Offset into UDP payload:
     uint32_t len;              // 0x00
     uint16_t type;             // 0x04
#      define ICOM_CAPABILITIES_TYPE_IGNORE_ME 0x01 // <- not sure what this REALLY means
     uint16_t seq;              // 0x06
     uint32_t sentid;           // 0x08
     uint32_t rcvdid;           // 0x0c
     BYTE     unuseda[3];       // 0x10
     uint16_t code;             // 0x13
     uint16_t res;              // 0x15
     BYTE     innerseq;         // 0x17
     BYTE     unusedb;          // 0x18
     BYTE     unusedc;          // 0x19
     uint16_t tokrequest;       // 0x1a
     uint32_t token;            // 0x1c
     BYTE     unusedd[33];      // 0x20
     BYTE     capa;             // 0x41
     BYTE     unusede[7];       // 0x42
     uint16_t commoncap;        // 0x49
     BYTE     unused;           // 0x4b
     BYTE     macaddress[6];    // 0x4c
     char     name[32];         // 0x52 // "Name" ? What name ? Whose name, mine, yours ? Difference to the one in "ConnInfo" ?
     char     audio[32];        // 0x72
     uint16_t conntype;         // 0x92
     BYTE     civ_addr;         // 0x94 // ex: just "civ". Too short for me. In fact, the CI-V Address (0xA4 for IC-705)
     uint16_t rxsample;         // 0x95
     uint16_t txsample;         // 0x97
     BYTE     enablea;          // 0x99
     BYTE     enableb;          // 0x9a
     BYTE     enablec;          // 0x9b
     uint32_t baudrate;         // 0x9c
     uint16_t capf;             // 0xa0
     BYTE     unusedi;          // 0xa2
     uint16_t capg;             // 0xa3
     BYTE     unusedj[3];       // 0xa5
    } s;
  BYTE packet[ICOM_PKT_CAPABILITIES_SIZE];
} T_IcomCapabilitiesPacket;

#pragma pack(pop)

typedef union uIcomPackets // union with POINTERS TO(!) any kind of 'Icom Packet'
{ T_IcomControlPacket         *control;
  T_IcomWatchdogPacket        *watchdog;
  T_IcomPingPacket            *ping;
  T_IcomCiVPacketHeader       *civ;  // <- "immediately" followed by the CI-V PAYLOAD, in the same datagram !
  T_IcomOpenClosePacket       *open_close;
  T_IcomAudioPacket           *audio;
  T_IcomRetransmitRangePacket *retransmit;
  T_IcomTokenPacket           *token;
  T_IcomStatusPacket          *status;
  T_IcomLoginPacket           *login;
  T_IcomLoginResponsePacket   *login_resp;
  T_IcomConnInfoPacket        *conninfo;
  T_IcomCapabilitiesPacket    *capabilities;
  BYTE                        *packet;
} U_IcomPackets;



//---------------------------------------------------------------------------
// Structures used in DL4YHF's "Remote CW Keyer" for communication
// between client (without direct connection to a real radio)
//     and server (usually connected to his 'local' radio via USB/serial/Ci-V):
//---------------------------------------------------------------------------


typedef struct t_RigCtrl_VfoReport // struct to report the radio's VFO frequency, etc. NOT ICOM-SPECIFIC.

{ // This packet travels "both ways" via TCP/IP, from client to server and server to client,
  // where it is passed on to RigControl.c (which either passes on the command
  // to the connected radio [server side: -> RigCtrl_SetVFOFrequency(), etc],
  // or just stores the data for the display in the GUI [client side]).
  BYTE bStructSize; // [0] set to sizeof(struct..) to allow future extensions
                    // at the end of the struct. With this, the receiver
                    // knows which members of the "new" component are valid.
  // If bStructSize exceeds EIGHT, the payload after CWNET_CMD_FREQ_REPORT
  // may contain any of the "Rig Control Parameters" identified by the following number:
  BYTE bVfoIndex;   // [1] for multi-VFO-radios like the IC-7610 : [0]=main VFO, [1]=sub VFO
  BYTE bTrxStatus;  // [2] Bit combination of the following flags:
#      define RIGCTRL_TRX_STATUS_TRANSMITTING  0x01 // H=transmitting, L=receiving
#      define RIGCTRL_TRX_STATUS_SQUELCH_MUTED 0x02 // H=audio muted,  L=not muted ("squelch open")
  signed char i8SMeterLevel_dB; // [3] "decibel over S0", signed 8 bit, may be "slightly negative".
                                // i8SMeterLevel_dB = +127 = "S9 plus 73 dB" .. that's enough.
                                // 0x80 = -128 means "no valid S-meter report".
  // Note the STRUCT ALIGNMENT: four bytes above, and a 32-bit field below !
  double dblVfoFrequency;     // [8..15]  current "VFO frequency" in Hertz(!)
} T_RigCtrl_VfoReport;


typedef struct t_RigCtrl_ParamReport_Integer // short struct to report an 'int' parameter, identified by 'parameter number'

{ // Like similar structs defined further below, this packet travels "both ways"
  // via TCP/IP, from client to server and server to client,
  // where it is passed on to RigControl.c .
  // NO NEED FOR AN EXTRA BYTE WITH THE 'STRUCT SIZE' HERE, to keep it as short as possible!
  BYTE bRigControlParameterNumber; // byte [0] : RIGCTRL_PN_xyz
  BYTE bSubindex;             // [1] future reserve for "array-like" parameters or similar
  BYTE b2AlignmentDummy[2];   // [2..3] future reserve / dummy for 4-byte alignment for the following 32-bit parameter:
  long i32ParameterValue;     // [4..7] : 32-bit signed value of the parameter identified by bRigControlParameterNumber
} T_RigCtrl_ParamReport_Integer;


typedef struct t_RigCtrl_ParamReport_Double // struct to report a 'double' parameter, identified by 'parameter number'

{ BYTE bStructSize;         // [0] struct size in bytes
  BYTE bRigControlParameterNumber; // [1] RIGCTRL_PN_UNKNOWN(0), RIGCTRL_PN_FREQUENCY(1), ...
  BYTE bSubindex;           // [2] future reserve for "array-like" parameters or similar
  BYTE b5AlignmentDummy[5]; // [3..7] future reserve / dummy for 8-byte alignment for the following 64-bit parameter:
  double dblParameterValue; // [8..11] value of the parameter identified by bRigControlParameterNumber.
} T_RigCtrl_ParamReport_Double;


typedef struct t_RigCtrl_ParamReport_String // very short struct to report a 'string' parameter, identified by 'parameter number'

{ BYTE bStructSize;         // [0] USED struct size (depends on the actual string length; 255 characters is the MAXIMUM)
  BYTE bRigControlParameterNumber; // byte[1] : RIGCTRL_PN_xyz
  char sz255ParameterValue[256]; // byte [2..bStructSize-1] : zero-terminated "C" string, 8 bits per character
} T_RigCtrl_ParamReport_String;



typedef struct t_RigCtrl_MultiFunctionMeterReport // short struct to report most that an IC-7300 shows on the "Mult-Function Meter" screen
{ BYTE bStructSize;  // [0] USED struct size (depends on the SOFTWARE VERSION of the *sender*; 255 bytes is the MAXIMUM)
  signed char i8PowerMeterLevel_pcnt; // [1] ICOM-like: They measure output power in PERCENTS, not WATTS !
                                      //     ~~ RIGCTRL_PN_POWER_METER_PERCENT
  signed char i8SWRMeterValue_x10;    // [2] dimensionless "SWR" value, multiplied by 10 to send it as an 8-bit integer
                                      //     ~~ RIGCTRL_PN_SWR_METER_VALUE
  signed char i8ALCMeterLevel_pcnt;   // [3] 0 ... 100 % of whatever the MAXIMUM may be (don't ask me; ask Icom)
                                      //     ~~ RIGCTRL_PN_ALC_METER_LEVEL
  signed char i8CompMeterLevel_dB;    // [4] compression meter level (measured); Icom measures from 0 to 30 dB. We send it "in dB", not "0241 = 30 dB".
                                      //     ~~ RIGCTRL_PN_COMP_METER_LEVEL_DB
  BYTE u8SupplyVoltage_100mV; // [5] measured supply voltage in 100-mV-units
                              //             ~~ RIGCTRL_PN_SUPPLY_VOLTAGE_mV
  BYTE u8DrainCurrent_100mA;  // [6] measured drain current in 100-mA-units
                              //             ~~ RIGCTRL_PN_DRAIN_CURRENT_mA
  BYTE u8PATemperature_degC;  // [7] measured power amplifier temperature in degrees CELSIUS; 0 when invalid; 255 = *OUCH* !
       // Note: The Icom engineers have forgotten to implement a CI-V command
       //       to query the POWER AMPLIFIER TEMPERATURE.
       //       At least an IC-7300 can only display that important parameter LOCALLY !
  BYTE bReserve[15-7+1];      // [x..15] reserved for future 'meter readings', and to align the struct size for 8 bytes

} T_RigCtrl_MultiFunctionMeterReport;

typedef struct t_RigCtrl_PotiReport // short struct to report most "Poti Settings" in a short block
{ BYTE bStructSize;  // [0] USED struct size (depends on the SOFTWARE VERSION of the *sender*; 255 bytes is the MAXIMUM)

  // Struct with "poti settings" (in contrast to  METER READINGS) :
  //  Expected to change rarely, thus not part of the "multi function meter report".
  // Struct members ordered APPROXIMATELY by their 'unified parameter numbers',
  // but don't assume anything.
  // Here, to conserve bandwidth, 'poti-like settings' are 8-bit signed integers.
  //       Negative = UNKNOWN (ok for "percentages" but not for "dB values") .
  signed char i8AudioVolume_pcnt;         // [1] the radio's LOCAL AUDIO VOLUME SETTING
              //     '~~ RIGCTRL_PN_AUDIO_VOLUME_PERCENT : pRC->iAudioVolume_Percent
  signed char i8RFPower_pcnt;             // [2] ICOM-like: They also SET the output power in PERCENTS, not WATTS !
              //     '~~ RIGCTRL_PN_RF_POWER_SETTING_PERCENT : pRC->iRFPowerSetting_Percent;  "setting" to avoid confusion with the POWER METER MEASUREMENT !
  signed char i8RFGain_pcnt;              // [3] etc, etc.. see T_RigCtrlInstance.iRFGain_Percent ...
              //     '~~ RIGCTRL_PN_RF_GAIN_PERCENT       : pRC->iRFGain_Percent
  signed char i8SquelchLevel_pcnt;        // [4]
              //     '~~ RIGCTRL_PN_SQUELCH_LEVEL_PERCENT : pRC->iSquelchLevel_Percent
  signed char i8NoiseBlanker_pcnt;        // [5]
              //     '~~ RIGCTRL_PN_NOISE_BLANKER_PERCENT : pRC->iNoiseBlanker_Percent
  signed char i8NoiseReduction_pcnt;      // [6]
              //     '~~ RIGCTRL_PN_NOISE_REDUCTION_PERCENT : pRC->iNoiseReduction_Percent
  signed char i8PassbandTuningPos1_pcnt;  // [7]
              //     '~~ RIGCTRL_PN_PASSBAND_TUNING_POS1  : pRC->iPassbandTuningPos1_Percent; "inner ring" of "TWIN PBT", 50 % = center
  signed char i8PassbandTuningPos2_pcnt;  // [8]
              //     '~~ RIGCTRL_PN_PASSBAND_TUNING_POS2  : pRC->iPassbandTuningPos2_Percent; "outer ring" of "TWIN PBT", 50 % = center
  signed char i8CWPitch_25Hz;             // [9]
              //     '~~ RIGCTRL_PN_CW_PITCH_HZ           : pRC->iCWPitch_Hz; here: multiplied by 25 Hz to fit inside 7 bits
  signed char i8MicGain_pcnt;             // [10]
              //     '~~ RIGCTRL_PN_MIC_GAIN_PERCENT      : pRC->iMicGain_Percent
  signed char i8KeyerSpeed_WPM;           // [11]
              //     '~~ RIGCTRL_PN_KEYER_SPEED_PERCENT   : pRC->iKeyerSpeed_WPM;  typical range: 6 WPM ... 48 WPM (Icom's crazy range "0 .. 100 %")
  signed char i8NotchPos_pcnt;            // [12]
              //     '~~ RIGCTRL_PN_NOTCH_POS_PERCENT     : pRC->iNotchPos_Percent
  signed char i8Comp_pcnt;                // [13]
              //     '~~ RIGCTRL_PN_COMP_SETTING_PERCENT  : pRC->iCompSetting_Percent
  signed char i8BreakInDelay_pcnt;        // [14]
              //     '~~ RIGCTRL_PN_BREAK_IN_DELAY_PERCENT: pRC->iBreakInDelay_Percent
  signed char i8MonitorGain_pcnt;         // [15]
              //     '~~ RIGCTRL_PN_MONITOR_GAIN_PERCENT  : pRC->iMonitorGain_Percent
  signed char i8VoxGain_pcnt;             // [16]
              //     '~~ RIGCTRL_PN_VOX_GAIN_PERCENT      : pRC->iVoxGain_Percent
  signed char i8AntiVoxGain_pcnt;         // [17]
              //     '~~ RIGCTRL_PN_ANTI_VOX_GAIN_PERCENT : pRC->iAntiVoxGain_Percent
  signed char i8Brightness_pcnt;          // [18]
              //     '~~ RIGCTRL_PN_BRIGHTNESS_PERCENT    : pRC->iBrightness_Percent; only interesting for the sysop :)

  BYTE bReserve[23-19+1];  // [19..23] reserved for future 'poti settings', and to align the struct size for 8 bytes


} T_RigCtrl_PotiReport;



/*-------------- global variables ---------------------------------------*/

extern const T_RigCtrl_RadioInfo RigCtrl_RadioInfo_CIV[]; // radios compatible with Icom CI-V ("Computer Interface Vee")
#if( SWI_SUPPORT_YAESU_5_BYTE_CAT )
extern const T_RigCtrl_RadioInfo RigCtrl_RadioInfo_Y5B[]; // radios with stoneage Yaesu "5-Byte-Command" CAT [Meow!]
#endif // SWI_SUPPORT_YAESU_5_BYTE_CAT ?
extern const T_RigCtrl_ParamInfo RigCtrl_ParameterInfo[];
extern const T_SL_TokenList RigCtrl_OpModes[];
extern const T_SL_TokenList RigCtrl_BandNames[];
extern const T_SL_TokenList RigCtrl_FreqMemEntryTokens[]; // .. to convert T_RigCtrlFreqMemEntry into a string and back
   // The above 'constant tables' are exposed via header
   // so the main application (e.g. Spectrum Lab) can use it
   // in a selection list for LAN- or WLAN-capable radios .

extern BOOL RigCtrl_fUpdateTrafficLog; // flag signalling "call TDebugForm::UpdateTrafficLog() a.s.a.p" (or whoever DISPLAYS the traffic log on-screen, and CLEARS that flag when 'done')
extern long RigCtrl_i32MessageNrForTrafficLog;
extern double RigCtrl_TrafficMon_dblUnixTimeOfStart;
typedef struct t_RigCtrl_TrafficMonitorConfig
{
   int iDisplayOptions; // 0=RIGCTRL_TMON_DISPLAY_OPTION_NORMAL; 1=RIGCTRL_TMON_DISPLAY_OPTION_SHOW_UDP_PAYLOAD = 1, etc, bitwise combineable:
#      define RIGCTRL_TMON_DISPLAY_OPTION_DISABLED                0 // if NONE of the following bits are set, RigCtrl_AddToTrafficLog() shows NOTHING..
#      define RIGCTRL_TMON_DISPLAY_OPTION_ENABLE                  1 // when cleared, RigCtrl_AddToTrafficLog() won't log "normal traffic" (only report "problems")
#      define RIGCTRL_TMON_DISPLAY_OPTION_SHOW_UDP_PAYLOAD        2
#      define RIGCTRL_TMON_DISPLAY_OPTION_PAUSE_ON_FULL_FIFO      4
#      define RIGCTRL_TMON_DISPLAY_OPTION_PAUSE_ON_ERROR          8
#      define RIGCTRL_TMON_DISPLAY_OPTION_PAUSE_ON_INIT_DONE     16
#      define RIGCTRL_TMON_DISPLAY_OPTION_PAUSE_ON_EXT_TRIGGER   32
#      define RIGCTRL_TMON_DISPLAY_OPTION_OMIT_PING_IDLE_AND_CO  64

   int iExtraColumns;   // 0=RIGCTRL_TMON_EXTRA_COLUMN_NONE; etc, bitwise combineable:
#      define RIGCTRL_TMON_EXTRA_COLUMN_NONE           0
#      define RIGCTRL_TMON_EXTRA_COLUMN_TIME           1 // show "time of day" (in UTC) ?
#      define RIGCTRL_TMON_EXTRA_COLUMN_TX_RX          2 // show "TX"/"RX" indicator ?
#      define RIGCTRL_TMON_EXTRA_COLUMN_PAYLOAD_LENGTH 4 // show payload length ("len") ?
#      define RIGCTRL_TMON_EXTRA_COLUMN_COMMENTS       8 // show "machine-generated comments, parameter names, etc" ?
#      define RIGCTRL_TMON_EXTRA_COLUMNS_ALL         255

   int iTimestampFormat; // usually 0 = RIGCTRL_TMON_TIMESTAMP_TIME_OF_DAY, or one of the following:
#      define RIGCTRL_TMON_TIMESTAMP_TIME_OF_DAY       0 // "time of day" (in UTC)
#      define RIGCTRL_TMON_TIMESTAMP_MILLISECONDS      1 // "milliseconds since CLEARING or STARTING the traffic monitor"
   // When modifying the sequence of members, also modify the struct initializer,
   //    because the stoneage C compiler (originally used for this project)
   //    only supported "Positional Initialization" - bleah !
} T_RigCtrl_TrafficMonitorConfig;
extern T_RigCtrl_TrafficMonitorConfig RigCtrl_TrafficMonitor;

extern BOOL RigCtrl_fTrafficMonitorPausedByUser;    // FALSE=not "paused by user", TRUE="paused by user"

extern int  RigCtrl_ITU_Region; // <- INITIALIZED global variable for simplicity, default = 1 :
       // ITU region 1 = Europe, Africa, FSU, Mongolia, Middle East, Iraq
       // ITU region 2 = North- and South America, Greenland, some east pacific islands
       // ITU region 3 = Far East, non-FSU Asia, Iran, Australia, Oceania

#if(SWI_HARDCORE_DEBUGGING) // (1) = hardcore-debugging, (0)=normal compilation
  extern int RigCtrl_iLastSourceLine;
  extern int RigCtrl_iConditionalBreakpointStep;  // see "conditional breakpoint sequences" listed in RigControl.c !
#endif // SWI_HARDCORE_DEBUGGING ?


/*-------------- prototypes ----------------------------------------------*/
#ifndef CPROT  // For peaceful co-existence of "C" and "C++" ....
// Note on "extern": do not trust the lousy example from Borland !
//   >extern "c" = wrong    (not recognized by Borland C++ V4.0)
//   >extern "C" = correct  (though the help system says something different)
#ifdef __cplusplus
 #define CPROT extern "C"
#else
 #define CPROT
#endif  /* ! "cplusplus" */
#endif

// External functions (used by RigControl.c, but not implemented in RigControl.c) :


//--- Common functions (not specific to CI-V or whatever may be added later) ----
      // Note: Radio address and protocol are filled in by the caller
      //       AFTER RigCtrl_Init(). Avoid a bloated API, and don't
      //       use 'getter' and 'setter' functions for each and everything.
      //       The same applies to many other members of T_RigCtrlInstance.
CPROT void  RigCtrl_Init( // initialize everything for a "single instance"
               T_RigCtrlInstance *pRC,    // [out] instance data
               int iRadioControlProtocol, // [in] e.g. RIGCTRL_PROTOCOL_NONE, RIGCTRL_PROTOCOL_ICOM_CI_V,
                                          //      or   RIGCTRL_PROTOCOL_HAMLIB_RIGCTLD (using TCP/IP, not "COM")
               int iRadioDefaultAddress,  // [in] e.g. RIGCTRL_DEF_ADDR_NO_REMOTE_CTRL, RIGCTRL_DEF_ADDR_AUTO_DETECT, RIGCTRL_DEF_ADDR_IC_7300, etc^10 ...
               T_CFIFO *pRadioPortRxFifo, // [in] thread-safe circular FIFO to RECEIVE data from the radio
               T_CFIFO *pRadioPortTxFifo, // [in] thread-safe circular FIFO to TRANSMIT data to the radio
               int nClientPorts,          // [in] 0..RIGCTRL_MAX_CLIENT_PORTS
               int *piClientOptions); // [in] array, one per client
CPROT void  RigCtrl_Close( T_RigCtrlInstance *pRC );     // Closes handles, sockets, frees resources
CPROT void  RigCtrl_ForgetAllParams( T_RigCtrlInstance *pRC );
CPROT void  RigCtrl_ReConnect( T_RigCtrlInstance *pRC ); // Restart connection. Called e.g. when clicking the "Apply"-button on SL's 'Radio Control' panel
CPROT void  RigCtrl_SetClientOptions( T_RigCtrlInstance *pRC, int iRigCtrlPort, int iClientOptions );
CPROT void  RigCtrl_SetRadioControlProtocol( T_RigCtrlInstance *pRC, int iRadioCtrlProtocol );
CPROT void  RigCtrl_SetPortErrorFlags( T_RigCtrlInstance *pRC, int iRigCtrlPort, DWORD dwPortErrorFlags );
CPROT void  RigCtrl_ClearPortErrorFlags( T_RigCtrlInstance *pRC, int iRigCtrlPort, DWORD dwPortErrorFlags );
CPROT int   RigCtrl_PortInstancePtrToIndex( T_RigCtrl_PortInstance *pPortInstance );
CPROT T_RigCtrl_PortInstance* RigCtrl_IndexToPortInstancePtr( T_RigCtrlInstance *pRC, int iRigCtrlPort );


CPROT void  RigCtrl_StartReading( T_RigCtrlInstance *pRC ); // begins "communicating" actively
CPROT int   RigCtrl_SwitchPollingState( T_RigCtrlInstance *pRC, int iNewPollingState );
      // May be called from the appliction, to modify pRC->iParameterPollingState
      //  (which, since the introduction of other Rig-Control-Protocols besides CI-V,
      //   may involve a bit more than just setting pRC->iParameterPollingState)
      // The RETURN VALUE is one of the RIGCTRL_POLLSTATE_.. valued defined in RigControl.h,
      // in most (but not all) cases the same as iNewPollingState .

CPROT void  RigCtrl_Handler( T_RigCtrlInstance *pRC ); // periodically called every 50(?) milliseconds
#           define C_RIGCTRL_HANDLER_CALLING_INTERVAL_MS 50 // calling interval for RigCtrl_Handler() in milliseconds
CPROT int   RigCtrl_EnumerateRequiredPNs( int iZeroBasedIndex ); // used by RigControl.c to poll all *mandatory* parameters
CPROT int   RigCtrl_ProcessData( T_RigCtrlInstance *pRC, // <- suited for 'eavesdropping' on a SERIAL PORT TUNNEL or similar
               int iRigCtrlPort, int iRigCtrlOrigin,
               T_RigCtrl_AppCallback pAppCallback, BYTE *pbData, int nBytes );
CPROT void  RigCtrl_SetTransmitRequest( T_RigCtrlInstance *pRC, int iTransmitReqst);
CPROT void  RigCtrl_TurnPowerOnOrOff( T_RigCtrlInstance *pRC,BOOL fPowerOn );

CPROT BOOL  RigCtrl_SetVFOFrequency( T_RigCtrlInstance *pRC, double dblFreqHz );
CPROT BOOL  RigCtrl_SwitchToBand( T_RigCtrlInstance *pRC, DWORD dwNewBand/* RIGCTRL_BAND_xyz(only ONE bit)*/ );
CPROT BOOL  RigCtrl_SwitchToFreqMemEntry( T_RigCtrlInstance *pRC, T_RigCtrlFreqMemEntry *pFreqMemEntry );
CPROT DWORD RigCtrl_GetAvailableBands( T_RigCtrlInstance *pRC ); // -> bitwise combination of RIGCTRL_BAND_xyz
CPROT DWORD RigCtrl_FrequencyToBand( double dblVfoFrequency ); // -> RIGCTRL_BAND_xyz (not aware of user defined [sub-]bands; thus the RCW Keyer GUI uses KeyerGUI_BandComboInfoTable[][] instead)
CPROT void  RigCtrl_BandToFrequencyRange(DWORD dwBand, double *pdblFmin_Hz, double *pdblFmax_Hz );


CPROT void  RigCtrl_AssembleVfoReport( T_RigCtrlInstance *pRC, T_RigCtrl_VfoReport *pVfoReport);
CPROT void  RigCtrl_OnVfoReportFromNetwork( T_RigCtrlInstance *pRC, T_RigCtrl_VfoReport *pVfoReport);
CPROT void  RigCtrl_OnParamReport_Integer( T_RigCtrlInstance *pRC, T_RigCtrl_ParamReport_Integer *pReport);
CPROT void  RigCtrl_OnParamReport_Double(T_RigCtrlInstance *pRC,T_RigCtrl_ParamReport_Double *pReport);
CPROT void  RigCtrl_OnParamReport_String(T_RigCtrlInstance *pRC,T_RigCtrl_ParamReport_String *pReport);
CPROT void  RigCtrl_AssembleMultiFunctionMeterReport(T_RigCtrlInstance *pRC, T_RigCtrl_MultiFunctionMeterReport *pMeterReport);
CPROT void  RigCtrl_OnMultiFunctionMeterReportFromNetwork(T_RigCtrlInstance *pRC, T_RigCtrl_MultiFunctionMeterReport *pMeterReport);
CPROT void  RigCtrl_AssemblePotiReport(T_RigCtrlInstance *pRC, T_RigCtrl_PotiReport *pPotiReport);
CPROT void  RigCtrl_OnPotiReportFromNetwork(T_RigCtrlInstance *pRC, T_RigCtrl_PotiReport *pPotiReport);
CPROT void  RigCtrl_OnBandStackingRegisterReportFromNetwork( T_RigCtrlInstance *pRC, const char *pszSource );
CPROT void  RigCtrl_OnUserDefinedBandReportFromNetwork( T_RigCtrlInstance *pRC, const char *pszSource );

CPROT BOOL  RigCtrl_QueueUpCmdToReadUnifiedPN( T_RigCtrlInstance *pRC, int iUnifiedPN );
CPROT BOOL  RigCtrl_QueueUpCmdToWriteUnifiedPN( T_RigCtrlInstance *pRC, int iUnifiedPN );
CPROT int   RigCtrl_GetNumQueuedUpCommandsPending( T_RigCtrlInstance *pRC );

CPROT int   RigCtrl_GetAgeOfBasicParams_ms( T_RigCtrlInstance *pRC );
CPROT BOOL  RigCtrl_QueueUpCmdToReadBasicParams( T_RigCtrlInstance *pRC );

CPROT BOOL  RigCtrl_SetOperatingMode( T_RigCtrlInstance *pRC, int iOpMode ); // .. and SEND TO RADIO ..
CPROT BOOL  RigCtrl_SetScopeRefLevel_dB( T_RigCtrlInstance *pRC, double dblNewValue ); // .. and SEND TO RADIO ..
CPROT BOOL  RigCtrl_SetScopeSpan_Hz( T_RigCtrlInstance *pRC, double dblNewDisplayWidth_Hz ); // .. and SEND TO RADIO ..
CPROT BOOL  RigCtrl_SetScopeCenterFrequency( T_RigCtrlInstance *pRC, double dblFreq_Hz ); // .. and SEND TO RADIO (or server) ..


CPROT void  RigCtrl_UpdateBasebandFrequencyConversionParams( T_RigCtrlInstance *pRC ); // -> DemodBasebandFreqOffset, ..Factor
CPROT double RigCtrl_AudioToRadioFrequency( T_RigCtrlInstance *pRC, double dblAudioFreq_Hz );
CPROT double RigCtrl_RadioToAudioFrequency( T_RigCtrlInstance *pRC, double dblRadioFreq_Hz );

CPROT T_RigCtrl_RadioInfo *RigCtrl_GetRadioInfoByDefaultAddress( int iRadioCtrlProtocol, int iDefaultAddress );

CPROT void  RigCtrl_InitFreqMemEntry( T_RigCtrlFreqMemEntry *pFreqMemEntry );
CPROT void  RigCtrl_InitUserDefinedBand( T_RigCtrlUserDefinedBand *pUserDefdBand );


  // "Stringifiers" and their inverse functions (for configuration files, use TEXT, NOTHING BINARY):
CPROT int   RigCtrl_FreqMemEntryToString(T_RigCtrlFreqMemEntry *pFreqMemEntry,
                            int iArrayIndex, int nEntriesInArray, char *pszDest, int iMaxDestLen );
CPROT int   RigCtrl_StringToFreqMemEntry( const char **cppSrc, T_RigCtrlFreqMemEntry *pFreqMemEntry,
                            int *piArrayIndex, int *pnEntriesInArray );
CPROT const char* RigCtrl_PermissionsToString( DWORD dwPermissions ); // [in] RIGCTRL_PERMISSION_NONE, RIGCTRL_PERMISSION_TRANSMIT, ..
CPROT const char* RigCtrl_VfoFrequencyToString( double dblVfoFrequency, char *psz15Buffer ); // [in] frequency in HERTZ, [out] e.g. "144.050 MHz"
CPROT const char* RigCtrl_OperatingModeToString( int iOpMode ); // e.g. "USB", "LSB", "CW", ...
CPROT int   RigCtrl_StringToOperatingMode(const char **ppszSource); // -> BIT COMBINATION of RIGCTRL_OPMODE_CW, RIGCTRL_OPMODE_NARROW, ..

CPROT int   RigCtrl_UserDefinedBandToString( T_RigCtrlUserDefinedBand *pUserDefdBand,
                            int iArrayIndex, int nEntriesInArray, char *pszDest, int iMaxDestLen );
CPROT BOOL  RigCtrl_StringToUserDefinedBand( const char **cppSrc, T_RigCtrlUserDefinedBand *pUserDefdBand,
                            int *piArrayIndex, int *pnEntriesInArray );
CPROT int   RigCtrl_MultiFunctionMeterReportToString(T_RigCtrl_MultiFunctionMeterReport *pMeterReport, char *pszDest, int iMaxDestLen );
CPROT int   RigCtrl_PotiReportToString(T_RigCtrl_PotiReport *pPotiReport, char *pszDest, int iMaxDestLen );

CPROT const char* RigCtrl_ErrorCodeToString( int iRigctrlErrorCode );
CPROT const char* RigCtrl_DefaultAddressToString( int iRadioCtrlProtocol, int iDefaultAddress );
CPROT const char* RigCtrl_RigControlPortNrToString( int iRigCtrlPort );
CPROT const char* RigCtrl_PortUsageToString( int iPortUsage );
CPROT const char* RigCtrl_ControlConnStateToString( T_RigCtrlInstance *pRC ); // more verbose than RigCtrl_ConnStateToString(), but only for the "CONTROL"-UDP-port
CPROT const char* RigCtrl_ParameterPollingStateToString(int iParameterPollingState);
CPROT const char* RigCtrl_GetRadioControlPortAsString(T_RigCtrlInstance *pRC); // -> e.g. "COM5", if that's the port "talking to the radio" .
               // Since RighControl.c doesn't know anything about serial ports,
               // RigCtrl_GetRadioControlPortAsString() must be provided by the application.
CPROT const char* RigCtrl_GetDescriptivePortName(T_RigCtrl_PortInstance *pPortInstance); // <- returns e.g. "Radio Control Port" or even "COM5:IC-7300", "Virtual Radio Port #2", etc


   // API to access parameters (stored in the instance data) via 'Unified Parameter Number'
   // (to eliminate any IcomKenwoodYeasu-specific addressing of settings/parameter/readout data)
CPROT int     RigCtrl_EnumerateUnifiedParameters( int iPreviousUnifiedPN );
CPROT T_RigCtrl_ParamInfo *RigCtrl_GetInfoForUnifiedParameterNumber(int iUnifiedPN);
CPROT T_RigCtrl_ParamInfo *RigCtrl_GetInfoForUnifiedParameterByName(const char **cpSrc);
CPROT T_RigCtrl_ParamInfo *RigCtrl_GetInfoForCIVCommandAndSubcode(DWORD dwCombinedCmdAndSubcode);
CPROT const char* RigCtrl_UnifiedParameterNumberToString(int iUnifiedPN);
CPROT int*    RigCtrl_GetIntValuePtrForUnifiedParameterNumber( T_RigCtrlInstance *pRC, int iUnifiedPN );
CPROT double* RigCtrl_GetDblValuePtrForUnifiedParameterNumber( T_RigCtrlInstance *pRC, int iUnifiedPN );
CPROT BOOL    RigCtrl_HaveValidParamValue(T_RigCtrlInstance *pRC, T_RigCtrl_ParamInfo *pPI);
CPROT BOOL    RigCtrl_GetParamValueAsString(T_RigCtrlInstance *pRC, T_RigCtrl_ParamInfo *pPI, char *pszDest, int iMaxLen );
CPROT BOOL    RigCtrl_SetParamValueFromString(T_RigCtrlInstance *pRC, T_RigCtrl_ParamInfo *pPI, const char **cppSrc );
CPROT long    RigCtrl_GetParamValue_Int( T_RigCtrlInstance *pRC, T_RigCtrl_ParamInfo *pPI );
CPROT BOOL    RigCtrl_SetParamValue_Int( T_RigCtrlInstance *pRC, T_RigCtrl_ParamInfo *pPI, long i32Value );
CPROT BOOL    RigCtrl_SetParamValue_Double(T_RigCtrlInstance *pRC,T_RigCtrl_ParamInfo *pPI,double dblValue);
CPROT long    RigCtrl_GetParamByPN_Int( T_RigCtrlInstance *pRC, int iUnifiedPN );
CPROT double  RigCtrl_GetParamByPN_Double( T_RigCtrlInstance *pRC, int iUnifiedPN );
CPROT void    RigCtrl_SetParamByPN_Int( T_RigCtrlInstance *pRC, int iUnifiedPN, long i32Value );
CPROT void    RigCtrl_SetParamByPN_Double( T_RigCtrlInstance *pRC, int iUnifiedPN, double dblValue );
CPROT BOOL    RigCtrl_AddParamToTxQueueForCwNet(T_RigCtrlInstance *pRC, int iUnifiedPN); // .. for a later transmission via CwNet.c !
CPROT int     RigCtrl_GetParamNrFromTxQueueForCwNet(T_RigCtrlInstance *pRC);
CPROT void    RigCtrl_ModifyMsgType(int *piMsgType, int iMsgMask, int iNewType );


CPROT T_RigCtrl_Spectrum *RigCtrl_AppendSpectrumToFIFO( T_RigCtrlInstance *pRC,
                   double dblBinWidth_Hz, double dblFmin_Hz, int nFrequencyBins);
CPROT int     RigCtrl_GetNumSpectraAvailableInFIFO( T_RigCtrlInstance *pRC, int iTailIndex );
CPROT T_RigCtrl_Spectrum *RigCtrl_GetSpectrumFromFIFO( T_RigCtrlInstance *pRC, int *piTailIndex );
#      define RIGCTRL_GET_OLDEST_ENTRY -1 // dummy FIFO index (iTailIndex) for "the oldest available entry"
#      define RIGCTRL_GET_LATEST_ENTRY -2 // dummy FIFO index (iTailIndex) for "the most recent entry"

CPROT BOOL  RigCtrl_GetDisplayableRadioFrequencyRange( T_RigCtrlInstance *pRC,
        double *pdblRadioFreqMin, double *pdblRadioFreqMax, // [out] : RADIO  frequencies
        BOOL   *pfFixedEdgeMode );  // [out,optional: TRUE= "fixed edges", FALSE="centered on VFO"
CPROT BOOL RigCtrl_MayTransmitOnFrequencyAndMode( T_RigCtrlInstance *pRC,
        double dblOperatingFreq_Hz, // [in] operating frequency, in most cases: CARRIER FREQUENCY
        int    iOpMode ); // [in] RIGCTRL_OPMODE_CW/LSB/USB/.., combineable with _REVERSE, _NARROW, etc.
CPROT int  RigCtrl_LoadBandsAndFrequenciesFromFile(T_RigCtrlInstance *pRC, const char *pszPathAndFilename );
CPROT void RigCtrl_InitUserDefinedBandsFromRigInfo(T_RigCtrlInstance *pRC);
CPROT T_RigCtrlUserDefinedBand *RigCtrl_GetUserDefinedBandByFrequencyRange(
           T_RigCtrlInstance *pRC, double dblFmin_Hz, double dblFmax_Hz );
CPROT T_RigCtrlUserDefinedBand *RigCtrl_GetUserDefinedBandByName(
           T_RigCtrlInstance *pRC, const char* pszName );

CPROT void RigCtrl_ClearTrafficLog( T_RigCtrlInstance *pRC );
CPROT void RigCtrl_AddToTrafficLog( T_RigCtrlInstance *pRC, int iRigControlPort, int iRigCtrlOrigin,
              BYTE *pbMessage, int iMsgLength, int iMsgType, const char *pszComment );
CPROT void RigCtrl_UnPauseTrafficLogOnAllPorts( T_RigCtrlInstance *pRC ); // -> fTrafficMonitorPausedOnTrigger = FALSE;
CPROT int  RigCtrl_GetNumEntriesInTrafficLog( T_RigCtrlInstance *pRC, int iTrafficReader );
CPROT BOOL RigCtrl_ReadTrafficLog( T_RigCtrlInstance *pRC, int iTrafficReader,
                char *pszDest, int iWantedCharsPerLine, int *piMsgType, DWORD *pdwColor );
CPROT void RigCtrl_GetHeadlineForTrafficLog(int iMsgType, char *pszDest, int iWantedCharsPerLine);

CPROT void RigCtrl_EnterErrorHistory( const char *pszFormat,  // located "somewhere in the application" !
                                       ... );  // optional argument list (printf-style)
CPROT void RigCtrl_LimitDouble( double *pdblValue, double dblMin, double dblMax );


//--- NON-CI-V specific "low-level" functions ----------------------------
CPROT void RigCtrl_LimitInt32( long *pi32Value, long i32Min, long i32Max );
CPROT void RigCtrl_ModifyBitInDWORD( DWORD *pdwDest, DWORD dwBitmask, int iNewBitState );

CPROT BOOL RigCtrl_SendAndLogMessage( T_RigCtrlInstance *pRC, // API (also called from e.g. Yaesu5Byte.c)
        int iRigCtrlPort,   // [in] RIGCTRL_PORT_RADIO / RIGCTRL_PORT_AUX_COM_1 / 2 / 3 ?
        int iRigCtrlOrigin, // [in] usually RIGCTRL_ORIGIN_CONTROLLER, but who knows.. we may pretend to "be a radio"
        BYTE *pbMessage, int iMsgLength, // [in] message, including pre- and postamble but not the UDP-CIV-HEADER (!)
        int iMsgType,     // [in] message type category and 'flags', only for the traffic log,
                          //      e.g. RIGCTRL_MSGTYPE_FLAG_CIV | RIGCTRL_MSGTYPE_FLAG_TX .
        char *pszComment); // [in] human readable info, short enough for the traffic log



//--- CI-V specific functions -------------------------------------------------
CPROT int RigCtrl_ParseCIV( T_RigCtrlInstance *pRC, int iRigCtrlPort, int iRigCtrlOrigin, BYTE *pbMessage, int iMsgLength );
CPROT int RigCtrl_GetSizeOfCIVDataType( int iCIVDataType ); // .. in BYTES (what else) !
CPROT DWORD RigCtrl_GetCombinedCmdAndSubcodeForCIV( BYTE *pbMessage, int iMsgLength, int iMsgType );
CPROT int RigCtrl_MapCIVDataToMessage( BYTE *pbDest, int iCIVDataType, long i32Value );
CPROT int RigCtrl_CIV_AudioFilterBandwidthCodeToFrequency( int iBandwidthCode );
CPROT int RigCtrl_CIV_FrequencyToAudioFilterBandwidthCode( int iFrequency_Hz );
CPROT long RigCtrl_CIV_TuningStepCodeToFrequency(int iDefaultAddress, int iIcomTuningStepCode);
CPROT int  RigCtrl_CIV_TuningStepFrequencyToCode(int iDefaultAddress, long i32TuningStep_Hz);
CPROT int RigCtrl_IcomToRigCtrlOpmode( BYTE bIcomsOpMode );
CPROT int RigCtrl_RigCtrlOpmodeToIcom( int iRigCtrlOpmode );
CPROT int RigCtrl_RigCtrlOpmodeToIcomFilterNumber( int iRigCtrlOpmode );


#if( SWI_RIGCTRL_ACT_AS_SERVER ) // OPTIONAL functions, only used when RigControl can also act as SERVER (i.e. "emulate a radio"):
CPROT void RigCtrl_SetServerState( T_RigCtrl_PortInstance *pServerPortInstance,int iNewServerState);
CPROT const char* RigCtrl_ServerStateToString(int iServerState);
CPROT void RigCtrl_CIV_Server_PrepareParsing( T_RigCtrl_PortInstance *pServerPortInstance ); // called BEFORE parsing a message
CPROT void RigCtrl_CIV_Server_OnReadCommand( T_RigCtrl_PortInstance *pServerPortInstance,    // called FROM RigCtrl_ParseCIV()
                T_RigCtrl_ParamInfo *pPI, BYTE *pbMessage, int iMsgLength, int iMsgType, int iCIVDataType );
CPROT void RigCtrl_CIV_Server_OnWriteCommand( T_RigCtrl_PortInstance *pServerPortInstance,   // called FROM RigCtrl_ParseCIV()
                T_RigCtrl_ParamInfo *pPI, BYTE *pbMessage, int iMsgLength, int iMsgType, int iCIVDataType );
CPROT void RigCtrl_CIV_Server_PrepareResponse_OkOrNotOk(T_RigCtrl_PortInstance *pServerPortInstance, BOOL fOk); // .. with cmd = 0xFA or 0xFB ..


#endif // SWI_RIGCTRL_ACT_AS_SERVER ?


#endif //  RigControlH

/* EOF <RigControl.h> */
