//---------------------------------------------------------------------------
// File: C:\cbproj\Remote_CW_Keyer\RigControl.c
// Authors: Wolfgang Buescher (DL4YHF)
//       (the stripped-down variant for DL4YHF's "Remote CW Keyer" [RCW]
//        does NOT support the hypercomplicated "RS-BA1 protocol" via UDP) .
// Date:   2025-06-22
// Purpose:
//  Initially just a parser for Icon's CI-V protocol, used for rig control
//  over serial ports (including 'virtual COM ports' for USB).
//
//  Optional modules for Rig Control may be used in the project:
//    * RigControl_CIV_Server.c : Used for 'Additional COM Ports' mimicking
//                                Icom radios with CI-V protocol.
//                           Interacts heavily with RigCtrl_ParseCIV() when
//                           T_RigCtrl_PortInstance.fActAsServer == TRUE.
//
// Literature:
//  [IC7300FM]: Icom IC-7300 "Full Manual", Section 19 : "Control Command"
//              (contains the full set of CI-V commands in this radio)
//               C:\datasheets\Rig_Manuals\IC-7300\IC-7300_Full_Manual.pdf
//  [IC7610CIV]: Icom IC-7610 "CI-V Reference Guide", #A7380-7EX (2017)
//  [IC-R8600CIV]: IC-R8600 (RX) "CI-V Reference..",  #A7375-2EX-3a (2018)
//  [CIV_REF_MAN_V3_2]: "CI-V Reference Manual ver 3.2" from 2002, saved as
//               C:\datasheets\Rig_Manuals\IC-706\Icom_CI-V_manual.pdf .
//  [G3NRW_CIV]: IC-7300 TechNote: CI-V Controls - The Big Picture, G3NRW
//  [DF4OR_CIV]: http://www.plicht.de/ekki/civ/civ-p33.html .
//
// Revision History (latest entry first):
//
//   2025-07-09: New option SWI_RIGCTRL_ACT_AS_SERVER, for ADDITIONAL COM PORTS.
//               Details only in RigControl_CIV_Server.c .
//
//   2025-07-09: After a long time, tested WSJT-X v2.6 again, now using the
//               'Serial Port Tunnel'. Didn't work in contrast to WFView,
//               which worked perfectly with the same RCW Keyer configuration.
//        Details not HERE (because they are most likely already outdated) but in
//    Remote_CW_Keyer/Extra_Literature/Trouble_with_WSJTX_via_Serial_Tunnel.txt .
//               Added an alternative timestamp formats for the traffic log:
//    RigCtrl_TrafficMonitor.iTimestampFormat = RIGCTRL_TMON_TIMESTAMP_MILLISECONDS .
//
//   2025-06-22: Added limited supported for the old Yaeasu-"5-byte"-CAT
//               control, in a separate module: Yeasu5Byte.c .
//   2024-11-25: Replaced HUNDREDS of "char pointers" by "const char *"
//               because modern compilers (e.g. "ISO C++11") are
//               very pedantic about char pointers, and refuse to
//               treat STRING LITERALS as a "char *" !
//             After that, most of the non-VCL-modules could be compiled
//             with the old Borland C++Builder V6 as well as with
//             Embarcadero C++Builder V12 "Athens", "Community Edition".
//               Details about the migration process from BCB V6 to V12
//               in DL4YHF's C:\cbproj\YHF_Tools\StringLib.h .
//               (Porting of the project to C++Builder V12 was abandoned,
//                it never worked reliably, and the Embarcadero price
//                for a NON-EXPIRING version of C++Builder is a bad joke,
//                especially if you want CROSS-PLATFORM capability.)
//
//   2024-11 : Entries in T_RigCtrlInstance.BandStackingRegs[RIGCTRL_NUM_BAND_STACKING_REGS]
//             shall now be distributed from the RCW-server to all RCW-clients
//             via the one-and-only TCP/IP connection between client and server,
//             so that all users (clients AND the server's "sysop") have the
//             same list of available BANDS or even BAND-STACKING REGISTERS.
//             To do this in the background, without causing a burst of network
//             traffic in CwNet.c, added T_CwNetClient.iBandStackingRegsTxIndex .
//
//   2024-06 : Added a "Hamlib Net rigctld"-compatible SERVER, which shares the
//             data with RigControl.c (e.g. current frequency, mode, and who-
//             knows-what). To avoid crowding RigControl.c even more, the
//             Hamlib-"rigctld"-emulation is in Remote_CW_Keyer/HamlibServer.c .
//
//   2024-02 : Made a stripped-down variant for the "Remote CW Keyer":
//              C:\cbproj\Remote_CW_Keyer\RigControl.c ( + *.h ) .
//              Threw out the support for Icom's undocumented/proprietary
//              "RS-BA1" protocol (that's the thing with those THREE
//              UDP PORTS used when connected to the radio via LAN (IC-9700)
//              or WLAN (IC-705). If you need all these whistles and bells,
//              try to let WFVIEW run side-by-side with the Remote CW Keyer.
//   2018-12 : Created for Spectrum Lab by DL4YHF . Original location:
//              C:\cbproj\SpecLab\RigControl\RigControl.c .
//
//---------------------------------------------------------------------------

#include "switches.h" // project specific 'compilation switches', e.g.:
         // #define SWI_RIGCTRL_ACT_AS_SERVER 1 // Act as server emulating e.g.
         // an Icom- radio with CI-V on 'Additional COM Ports' ? 0=no, 1=yes
         //   (optional; added 2025-07; details in RigControl_CIV_Server.c )

#include <string.h>     // memset() isn't really a STRING function, but..
#include <stdio.h>      // snprintf() isn't "I/O", but declared in string.h
#include <math.h>       // fmodl() ..
#include "Timers.h"     // high-resolution stopwatches, etc
#include "Utilities.h"  // UTL_FormatDateAndTime(), UTL_GetCurrentUnixDateAndTime(), etc
#include "QFile.h"      // "Quick File" helper functions by DL4YHF
#include "Inet_Tools.h" // INET_GetU16FromLE(), etc
#include "RigControl.h" // header file for THIS module
#include "Yaesu5Byte.h" // you guessed it.. Yaesu-FIVE-BYTE-CAT-specific details HERE !
         // (Note the unhealthy dependence: RigControl.c depends on Yaesu5Byte.c,
         //                                 Yaesu5Byte.c depends on RigControl.c.
         //  Didn't turn the protocol-specific modules into derived C++ classes
         //  because the rig-specific procotol parsers / hosts were written
         //  with a simple microcontroller firmware in mind, possibly running
         //  on a Raspberry Pico W (W for WiFi), or even a PIC with USB.
         //  Such a uC firmware would only support ONE rig control protocol,
         //  may it be CI-V, Yaesu "CAT" 5-byte (meow!), "CAT ASCII" (aka Kenwood),
         //  or whatever comes next.
         // )


#if( SWI_USE_HAMLIB_SERVER ) // build an application with integrated "Hamlib Net rigctld"-compatible server ?
# include "HamlibServer.h"
  // HLSRV_EnumerateRequiredPNs() is called from RigControl.c during the init-
  // phase to have all parameters cached when anyone wants them, instead of
  // repeatedly reading them from the radio. RigControl.c (not HamlibServer.c)
  // is the module that keeps track of everything.
  // Parameters that are expected to change (like S-meter, or SWR during transmit)
  // will be polled at a moderate speed by RigControl.c anyway, thus there is
  // no need to issue a real 'READ'-command to the radio on behalf of any
  // remote Hamlib client. If we allow a remote (and insecure) Hamlib client
  // to CONTROL (set, change) anything inside the radio is another issue.
#endif // SWI_USE_HAMLIB_SERVER ?

#if( SWI_NUM_AUX_COM_PORTS > 0 ) // compile with support for 'Auxiliary / Additional COM ports' / 'Rig Control Clients on SERIAL PORTS' ?
# include "AuxComPorts.h" // structs and API functions for the "Auxiliary" (later: "Additional") COM ports
#endif // SWI_NUM_AUX_COM_PORTS ?


// Internal defines (only used within THIS module, thus not in the header file)
#define RIGCTRL_MAX_PINGS_LOGGED 5
#define RIGCTRL_MAX_IDLES_LOGGED 5

// Internal forward references (because C compilers are still stupid..)
static void RigCtrl_ForgetScopeFrequencyRanges( T_RigCtrlInstance *pRC );
static BOOL RigCtrl_SetVFOFrequency_Internal( T_RigCtrlInstance *pRC, double dblFreqHz );
static BOOL RigCtrl_SetOperatingMode_Internal( T_RigCtrlInstance *pRC, int iOpMode );
static BOOL RigCtrl_SetScopeSpan_Internal( T_RigCtrlInstance *pRC );
static void RigCtrl_SendReadWriteCommandFromFIFO( T_RigCtrlInstance *pRC );
static BOOL RigCtrl_SendReadCommandForUnifiedPN(  T_RigCtrlInstance *pRC, int iUnifiedPN );
static BOOL RigCtrl_SendWriteCommandForUnifiedPN( T_RigCtrlInstance *pRC, int iUnifiedPN );
static int  RigCtrl_GetCrazyModelDependingSubSubcodeForCmd1A_05( int iDefaultAddress, int iUnifiedPN );
static void RigCtrl_OnTransmitFlagChange( T_RigCtrlInstance *pRC ); // ( RIGCTRL_PN_TRANSMITTING; the rig itself changed from RX to TX or back )

#if( SWI_SUPPORT_YAESU_5_BYTE_CAT )
static void RigCtrl_Y5B_OnDecodeCallback( T_Yaesu5ByteControl *pYC, int iRigCtrlPort,
               int iRigCtrlOrigin, BYTE *pbMessage, int iMsgLength, char *pszComment );
#endif // SWI_SUPPORT_YAESU_5_BYTE_CAT ?
static void RigCtrl_OnPollingStateChangeInSubmodule( T_RigCtrlInstance *pRC, int iPollingStateFromSubmodule );
static int  RigCtrl_ProcessRxData( T_RigCtrlInstance *pRC, int iRigCtrlPort );
#if( SWI_RIGCTRL_ACT_AS_SERVER )
static BOOL RigCtrl_SendServerResponse( T_RigCtrl_PortInstance *pServerPortInstance );
#endif // SWI_RIGCTRL_ACT_AS_SERVER ?



// Not necessarily "internal" functions ... may have to call them from other modules:
BOOL RigCtrl_RegisterCIVReponseForClient(T_RigCtrlInstance *pRC, int iRigCtrlClientPort, T_RigCtrl_MsgFilter *pFilter);
BOOL RigCtrl_IsNotOkResponse( int iRadioCtrlProtocol, BYTE *pbMessage, int iMsgLength );
BOOL RigCtrl_IsOkResponse( int iRadioCtrlProtocol, BYTE *pbMessage, int iMsgLength );
T_RigCtrl_ParamInfo * RigCtrl_GetInfoForUnifiedParameterNumber(int iUnifiedPN);
T_RigCtrl_ParamInfo * RigCtrl_GetInfoForCIVCommandAndSubcode(DWORD dwCombinedCmdAndSubcode);
T_RigCtrl_RadioInfo * RigCtrl_GetRadioInfoByDefaultAddress( int iRadioCtrlProtocol, int iDefaultAddress );
BYTE RigCtrl_IntToBCD2( int iValue );  // converts an integer between 0 and 99 into BCD (single byte)
void RigCtrl_IntToBCD_CIV_LSByteFirst( long i32Value, BYTE *pbDest, int nDigits );
void RigCtrl_IntToBCD_CIV_MSByteFirst( long i32Value, BYTE *pbDest, int nDigits );
void RigCtrl_FindVfoForNewFrequency( T_RigCtrlInstance *pRC, int iMsgType ); // non-blocking, state machine driven subroutine
void RigCtrl_UpdateParamsAfterVfoSwitch( T_RigCtrlInstance *pRC ); // also a non-blocking state machine, runs shortly after e.g. "exchange VFO A / B"
void RigCtrl_OnVfoSwitch( T_RigCtrlInstance *pRC, int iNewVfoIndex );
#     define VFO_SWITCH_EQUALIZE_A_B 0xA0 // special value for 'iNewVfoIndex' in RigCtrl_OnVfoSwitch(), "Inspired by Icom"
#     define VFO_SWITCH_EXCHANGE_A_B 0xB0 // special value for 'iNewVfoIndex' in RigCtrl_OnVfoSwitch(), "Inspired by Icom"

T_RigCtrlAudioFilterParams *RigCtrl_GetAudioFilterParams(T_RigCtrlInstance *pRC,int iFilterIndex,int iOpMode);
T_RigCtrlAudioFilterParams *RigCtrl_AllocAudioFilterParam(T_RigCtrlInstance *pRC);
int  RigCtrl_GetAudioFilterBandwidth( T_RigCtrlInstance *pRC, int iFilterIndex, int iOpMode );
void RigCtrl_InitAudioFilterParamsForFilterIndexAndOpMode( T_RigCtrlAudioFilterParams *pAFP,int iFilterIndex,int iOpMode);

BOOL IcomPkt_SendToken(T_RigCtrlInstance *pRC, BYTE magic, char *pszComment);
BOOL IcomPkt_SendConnInfo( T_RigCtrlInstance *pRC, int iPortType );
void IcomPkt_PeriodicHandler( T_RigCtrlInstance *pRC, int iPortType );
int _stdcall RigCtrl_TxFunctionForCivViaUDP( T_RigCtrlInstance *pRC, int iRigCtrlPort, BYTE *source, int n_bytes,
                              int *piErrorCode, int *piNumMsWaited);

//----------------------------------------------------------------------------
// Global variables for hardcore debugging / post-mortem crash analysis.
//        Useful when the debugger's call stack shows nothing but garbage,
//        e.g. after an exception at the infamous address 0xFEEEFEEE .
//        ( C++Builder's debugger could still inspect SIMPLE GLOBAL
//          VARIABLES after most kinds of exceptions, but of course nothing
//          stack-based because the CPU- or task-stack was usually trashed.)
//----------------------------------------------------------------------------
#if(SWI_HARDCORE_DEBUGGING) // (1) = hardcore-debugging, (0)=normal compilation
 int RigCtrl_iLastSourceLine = 0; // WATCH THIS after crashing with e.g. "0xFEEEFEEE"  ...
# define HERE_I_AM__RIGCTRL()  RigCtrl_iLastSourceLine=__LINE__
 int RigCtrl_iConditionalBreakpointStep = 0; // helper for 'very complex conditional breakpoints', hard coded:
     // '--> Sequences / "Step" numbers for different use cases :
     //       1..4 : Trouble with forwarding 'Set SplitMode' from a "Virtual Rig" to a "Real Radio"
     //       5..9 : Trouble with forwarding 'Set SelectVFO' from a "Virtual Rig" to a "Real Radio"
     //
# if( 0 )
    If certain worker threads of the application don't seem to terminate
    "politely", watch the following (complete list not HERE but in Keyer_Main.cpp):
     * GUI_iLastSourceLine
     * DSW_iLastSourceLine
     * DSP_iLastSourceLine
     * Keyer_iLastSourceLine
     * CwNet_iLastSourceLine
     * HttpSrv_iLastSourceLine
     * RigCtrl_iLastSourceLine
#  endif // (0) .. kludge to inspect the above variables in Borland via mouse-over ...
  CPROT void CheckSystemHealth(const char *pszModuleName, int iSourceLine); // <- periodically called via macro CHECK_SYSTEM_HEALTH()
  static  const char *s_ModuleName = "RigCtrl";
# define CHECK_SYSTEM_HEALTH() CheckSystemHealth(s_ModuleName,__LINE__)
  BYTE *RigCtrl_pbSource = NULL;
  int   RigCtrl_iScopeBin= 0;
  float *RigCtrl_pfltDest= NULL;
#else
# define HERE_I_AM__RIGCTRL()  /* nuffink   */
# define CHECK_SYSTEM_HEALTH() /* hamwanich */
#endif // SWI_HARDCORE_DEBUGGING ?


//----------------------------------------------------------------------------
// Global variables ...
//----------------------------------------------------------------------------

BOOL RigCtrl_fUpdateTrafficLog = FALSE; // flag signalling "call TDebugForm::UpdateTrafficLog() a.s.a.p" (or whoever DISPLAYS the traffic log on-screen, and CLEARS that flag when 'done')
long RigCtrl_i32MessageNrForTrafficLog = 0;
double RigCtrl_TrafficMon_dblUnixTimeOfStart = 0.0;
T_RigCtrl_TrafficMonitorConfig RigCtrl_TrafficMonitor =
{ // Struct initialisation for a stoneage C compiler; modern ones can do better than this:
  /* .iDisplayOptions = */ RIGCTRL_TMON_DISPLAY_OPTION_DISABLED, // RigCtrl_AddToTrafficLog() disabled unless the application sets at least RIGCTRL_TMON_DISPLAY_OPTION_ENABLE
  /* .iExtraColumns   = */ RIGCTRL_TMON_EXTRA_COLUMN_TIME
                       | RIGCTRL_TMON_EXTRA_COLUMN_TX_RX
                       | RIGCTRL_TMON_EXTRA_COLUMN_PAYLOAD_LENGTH
                       | RIGCTRL_TMON_EXTRA_COLUMN_COMMENTS,
  /* .iTimestampFormat = */ RIGCTRL_TMON_TIMESTAMP_TIME_OF_DAY

  // Members of this struct above are stored in e.g. Remote CW Keyer's configuration file.
  // The above INITIALISATION VALUES will be used as DEFAULTS when LOADING such a file.
};
BOOL RigCtrl_fTrafficMonitorPausedByUser    = FALSE; // FALSE=not "paused by user", TRUE="paused by user"
     // '--> Reason for "pausing" / "stopping" : e.g. RIGCTRL_TMON_DISPLAY_OPTION_PAUSE_ON_FULL_FIFO / .. STOP_ON_ERROR / ...

int  RigCtrl_ITU_Region = 1; // <- INITIALIZED global variable for simplicity :
       // ITU region 1 = Europe, Africa, FSU, Mongolia, Middle East, Iraq
       // ITU region 2 = North- and South America, Greenland, some east pacific islands
       // ITU region 3 = Far East, non-FSU Asia, Iran, Australia, Oceania




//---------------------------------------------------------------------------
// Implementation of functions
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
int  RigCtrl_PortInstancePtrToIndex( T_RigCtrl_PortInstance *pPortInstance )
{ T_RigCtrlInstance *pRC = pPortInstance->pRC;
  int iRigCtrlPort = (int)( pPortInstance - &pRC->PortInstance[0] );
  // ,--------------------|_______________________________________|
  // '--> This "pointer difference" is not an OFFSET IN BYTES, but an array index !
  if((iRigCtrlPort<0) || (iRigCtrlPort>=RIGCTRL_NUM_PORT_INSTANCES) )
   {  iRigCtrlPort = 0;  // prevent chaos, use the instance "talking to the radio"
   }
  return iRigCtrlPort;
}

//---------------------------------------------------------------------------
T_RigCtrl_PortInstance* RigCtrl_IndexToPortInstancePtr( T_RigCtrlInstance *pRC, int iRigCtrlPort )
{ if((iRigCtrlPort<0) || (iRigCtrlPort>=RIGCTRL_NUM_PORT_INSTANCES) )
   {  iRigCtrlPort = 0;  // prevent chaos, use the instance "talking to the radio"
   }
  return &pRC->PortInstance[iRigCtrlPort];
}


//--------------------------------------------------------------------------
void RigCtrl_Init( T_RigCtrlInstance *pRC,
        int iRadioControlProtocol, // [in] e.g. RIGCTRL_PROTOCOL_ICOM_CI_V
        int iRadioDefaultAddress,  // [in] e.g. RIGCTRL_DEF_ADDR_NO_REMOTE_CTRL, RIGCTRL_DEF_ADDR_AUTO_DETECT, RIGCTRL_DEF_ADDR_IC_7300, etc^10 ...
        T_CFIFO *pRadioPortRxFifo, // [in] thread-safe circular FIFO to RECEIVE data from the radio
        T_CFIFO *pRadioPortTxFifo, // [in] thread-safe circular FIFO to TRANSMIT data to the radio
        int nClientPorts,          // [in] 0..RIGCTRL_MAX_CLIENT_PORTS
        int *piClientOptions )     // [in] array with one element per client (*)
  // Initializes everything for a "single instance", but DOES NOT COMMUNICATE yet.
  //
  // (*) These "clients" were initially just "CI-V port duplicators",
  //     connected to extra "COM ports" (in most cases, "com0com" virtual NULL MODEM CABLES).
  //     Later renamed into "Client #1" and "Client #2", at least for the
  //     configuration dialog in Spectrum Lab. Each of those clients can have
  //     its individual configuration. Thus using an ARRAY here !
{
  int i, iClient, iRigCtrlPort, iClientOptions;
  char   sz255Msg[256];
  T_RigCtrl_ParamInfo *pPI;
  T_RigCtrl_RadioInfo *pRadioInfo;
  T_RigCtrl_PortInstance *pPortInstance;

  if( nClientPorts > RIGCTRL_MAX_CLIENT_PORTS )
   {  nClientPorts = RIGCTRL_MAX_CLIENT_PORTS; // oops.. more than expected !?
   }

  memset( pRC, 0, sizeof( T_RigCtrlInstance ) );
  InitializeCriticalSection( &pRC->criticalSection );
  pRC->dwInitMagic = RIGCTRL_INIT_MAGIC;
  pRC->iCurrVfoIndex = pRC->iPrevVfoIndex = RIGCTRL_NOVALUE_INT; // got NO IDEA about which VFO ("A" or "B") is currently in use
  RigCtrl_ForgetAllParams( pRC ); // Set everything that we MAY(!) read from the radio via CI-V to "unknown" / "not read yet".
  pRC->initState = initState_ColdStart;     // initState_ColdStart /  _Opened / _Closed ?
  pRC->iDefaultAddress  = iRadioDefaultAddress; // RIGCTRL_DEF_ADDR_NO_REMOTE_CTRL, RIGCTRL_DEF_ADDR_AUTO_DETECT, RIGCTRL_DEF_ADDR_IC_7300, etc^10 ...
  pRC->pRadioPortTxFifo = pRadioPortTxFifo; // thread-safe circular FIFO to TRANSMIT data to the radio
  pRC->pRadioPortRxFifo = pRadioPortRxFifo; // thread-safe circular FIFO to RECEIVE data from the radio
  for( i=0; i<RIGCTRL_NUM_PORT_INSTANCES; ++i )
   { pPortInstance = &pRC->PortInstance[i];
     pPortInstance->pRC = pRC; // "link back" from the T_RigCtrl_PortInstance to its parent.
     // '--> Saves an extra function argument in dozens of subroutines:
     //      It we pass-in a T_RigCtrl_PortInstance*, the callee also knows the T_RigCtrlInstance* !
     pPortInstance->iRadioCtrlProtocol = iRadioControlProtocol;
     pPortInstance->iRadioDeviceAddr = iRadioDefaultAddress; // <- preferred : 0 = RIGCTRL_DEF_ADDR_AUTO_DETECT;
         //  ,--------'
         //  '--> For the RADIO CONTROL PORT, set to the "From"-address in the
         //       first RESPONSE from the "real radio", in RigCtrl_ParseCIV().
         //
     pPortInstance->iRadioMasterAddr = RIGCTRL_CIV_MASTER_ADDR_MIN; // Icom-slang: 0xE0 = "controller" (in this case, the PC controlling the radio)
   }

  pRC->u32RxAudioSamplesPerSecond = pRC->u32TxAudioSamplesPerSecond = 48000;
  pRC->dblUnixTimeOfFirstSpectrum = 0.0;

  for(iClient=0; iClient<nClientPorts; ++iClient)
   { iRigCtrlPort = iClient + RIGCTRL_PORT_AUX_COM_1;
     pPortInstance = &pRC->PortInstance[iRigCtrlPort];
     iClientOptions = piClientOptions[iClient]; // individual settings for each client !
     pPortInstance->iClientOptions = iClientOptions;
     pPortInstance->fEmulateCIVEcho = (iClientOptions & RIGCTRL_CLIENT_OPTION_EMULATE_CIV_ECHO) != 0;
   }
  pRC->dwRigControlFlags = RIGCTRL_FLAGS_NONE; // overwritten in Keyer_Main.cpp : TKeyerMainForm::FormCreate()
  pRC->iCapabilities    = RIGCTRL_CAPS_ALL;   // be optimistic about FUTURE rigs .. IC-9700 anyone ?
  pRC->dwAvailableBands = RIGCTRL_BAND_ALL_SHORTWAVE | RIGCTRL_BAND_6M | RIGCTRL_BAND_2M | RIGCTRL_BAND_70CM;  // <- that's an IC-705 (until we know better)
       // '--> For the application, the set of 'available bands' may be limited,
       //      or in a few cases extended, in RigCtrl_GetAvailableBands() .
  pRC->iMaxResponseDelay_ms = 1500; // maximum delay for getting a response, in milliseconds
  // ex: pRC->dblVfoFrequency = RIGCTRL_NOVALUE_DOUBLE; // <- unnecessary, should have been set in the "while( pPI->iUnifiedPN != 0 )" loop above
  // ex: pRC->iOpMode = RIGCTRL_NOVALUE_INT; // later: USB, LSB, CW, ...
  // ex: pRC->iTransmitting = pRC->iTransmitReqst = 0; // RIGCTRL_NOVALUE_INT is ok as long as THE RIG didn't report these parameters
  pRC->iParameterPollingState = RIGCTRL_POLLSTATE_PASSIVE; // don't do anything until the application calls RigCtrl_StartReading()
  // ex: pRC->iScopeSpeed = RIGCTRL_SCOPE_SPEED_FAST; // <- now READ FROM THE RADIO when "starting up" (same for the "Waterfall Speed")
  // ex: pRC->iResponseCountdown_ms = 0; // now part of a T_RigCtrl_PortInstance (and zeroed anyway)


  // Since the application (e.g. Remote CW Keyer) has already passed in
  //       iRadioControlProtocol (e.g. RIGCTRL_PROTOCOL_ICOM_CI_V)
  //   and iRadioDefaultAddress  (e.g. RIGCTRL_DEF_ADDR_IC_7300, RIGCTRL_DEF_ADDR_UNKNOWN_YAESU, etc^10),
  // look for info about the device capabilities.
  pRadioInfo = RigCtrl_GetRadioInfoByDefaultAddress( iRadioControlProtocol, iRadioDefaultAddress );
  if( pRadioInfo != NULL )
   { // ok, we know "a bit more" about the radio even before communicating with it -> USE THAT INFO !
     pRC->iCapabilities    = pRadioInfo->iCapabilities; // <- RIGCTRL_CAPS_.... (bitwise combineable)
     pRC->dwAvailableBands = pRadioInfo->dwBands; // <- may change later, after RIGCTRL_POLLSTATE_TX_BAND_EDGES
   } // end if < RigCtrl_GetRadioInfoByDefaultAddress() successful >

# if( SWI_SUPPORT_YAESU_5_BYTE_CAT )
  Yaesu5Byte_InitStruct( &pRC->Y5B, // [out] 'Yaesu 5-byte CAT' decoder or controller instance
       (void *)pRC, // [in] optional T_RigCtrlInstance* (here MANDATORY to call RigControl.c from Yaesu5Byte.c)
       RigCtrl_Y5B_OnDecodeCallback ); // [in] optional callback for 'decoded COMMANDS or RESPONSES'
# endif // SWI_SUPPORT_YAESU_5_BYTE_CAT ?


  RigCtrl_ForgetScopeFrequencyRanges( pRC );
  pRC->initState = initState_Opened; // initState_ColdStart /  _Opened / _Closed ? here: 'Opened' when returning from RigCtrl_Init()

  if( RigCtrl_TrafficMon_dblUnixTimeOfStart <= 0.0 )
   {  RigCtrl_TrafficMon_dblUnixTimeOfStart = UTL_GetCurrentUnixDateAndTime_Fast();
   }

  snprintf( sz255Msg, 255, "RigControl: Configured for %s on %s",
      RigCtrl_DefaultAddressToString(pRC->PortInstance[RIGCTRL_PORT_RADIO].iRadioCtrlProtocol, pRC->iDefaultAddress),
      RigCtrl_GetRadioControlPortAsString(pRC) ); // -> e.g. "COM5", if that's the port "talking to the radio"
  // Emit a single line with 'info' in the error history, with a summary about the remote rig:
  ShowError( ERROR_CLASS_INFO | SHOW_ERROR_TIMESTAMP, sz255Msg );


} // end RigCtrl_Init()

//-------------------------------------------------------------------- API --
void RigCtrl_EnterCriticalSection( T_RigCtrlInstance *pRC )
{
  // Avoid crashing without InitializeCriticalSection() [shouldn't happen, but..]
  if( pRC->dwInitMagic==RIGCTRL_INIT_MAGIC )
   { EnterCriticalSection( &pRC->criticalSection );
   }

} // end RigCtrl_EnterCriticalSection()

//-------------------------------------------------------------------- API --
void RigCtrl_LeaveCriticalSection( T_RigCtrlInstance *pRC )
{
  if( pRC->dwInitMagic==RIGCTRL_INIT_MAGIC )
   { LeaveCriticalSection( &pRC->criticalSection );
   }

} // end RigCtrl_LeaveCriticalSection()


//-------------------------------------------------------------------- API --
void RigCtrl_ForgetAllParams( T_RigCtrlInstance *pRC )
  // Sets everything that we MAY(!) read from the radio via CI-V to "unknown" / "not read yet".
  // This way, the "report" in SL's debug-window / CAT traffic monitor
  //      can spit out THOSE PARAMETERS THAT HAVE REALLY BEEN ACCESSED,
  //      and omits unused entries (in RigCtrl_ParameterInfo[]) .
{
  T_RigCtrl_ParamInfo *pPI = (T_RigCtrl_ParamInfo*)RigCtrl_ParameterInfo;

  while( pPI->iUnifiedPN != 0 )
   { if( ( pPI->iByteOffsetIntoRigCtrlInstance > 0 )
       &&( pPI->iByteOffsetIntoRigCtrlInstance < sizeof(T_RigCtrlInstance) ) )
      { switch ( pPI->iRigCtrlDataType )
         { case RIGCTRL_DT_INT  :
              *(int*)( (BYTE*)pRC + pPI->iByteOffsetIntoRigCtrlInstance ) = RIGCTRL_NOVALUE_INT; // default for MOST integer parameters (to poll them later; except for those set further below)
              break;
           case RIGCTRL_DT_DOUBLE:
              *(double*)( (BYTE*)pRC + pPI->iByteOffsetIntoRigCtrlInstance ) = RIGCTRL_NOVALUE_DOUBLE;
              break;
           default:
              break;
         } // end switch ( pPI->iRigCtrlDataType )
      }
     ++pPI;
   } // end while < all "unified parameters" >

  // Also "forget" parameters that are not referenced in RigCtrl_ParameterInfo[] :
  pRC->iPATemperature_degC = RIGCTRL_NOVALUE_INT; // no change to read this from Icom transceivers yet :(

# if( SWI_SUPPORT_YAESU_5_BYTE_CAT )
  Yaesu5Byte_ForgetAllParams( &pRC->Y5B );
# endif // SWI_SUPPORT_YAESU_5_BYTE_CAT ?

  pRC->fGotBasicParams = FALSE; // here: cleared in RigCtrl_ForgetAllParams()

} // end RigCtrl_ForgetAllParams()


//-------------------------------------------------------------------- API --
void RigCtrl_Close( T_RigCtrlInstance *pRC ) // closes handles, sockets, frees resources
  // Called internally when necessary, or via SL's main menu: "Start/Stop" .. "Rig Control" .. "Start" .
{
  // Won't stop the threads running in IOaccess.cpp, but will stop functions called from there:
  //    (closing those threads, and closing "COM" ports isn't RigControl's part
  //     because RigControl neither opened those ports or created those threads) :
  if( pRC->initState == initState_Opened )
   {  pRC->initState = initState_Closed; // here: 'Closed' in
     // Give any RigControl-subroutines and "handlers" the chance to return,
     // before closing sockets, freeing memory, and making pointers invalid:
     Sleep( 50 ); // should be long enough to return from e.g RigCtrl_Handler() to IO_RadioCommThdFunc() .
   }

  // The RCW-Keyer GUI checks for RIGCTRL_POLLSTATE_DONE, to switch from the
  // 'Debug'- to the 'TRX'-tab when all required parameters are through. Thus:
  RigCtrl_SwitchPollingState( pRC, RIGCTRL_POLLSTATE_PASSIVE );

} // end RigCtrl_Close()


//-------------------------------------------------------------------- API --
void RigCtrl_ReConnect( T_RigCtrlInstance *pRC )
  // Politely closes an existing connection (if any),
  // clears the 'counters' that limit certain messages from the 'traffic log',
  // and finally tries to re-connect the rig.
  // Keeps all previous settings from RigCtrl_SetLANParams() INTACT !
  //
  // Called for example when clicking the "Apply"-button on SL's 'Radio Control' panel,
  //        or via SL's main menu: "Start/Stop" .. "Rig Control" .. "Start" .
{ char sz80Msg[84], *cp;
  int iMsgTypeForTrafficLog;

  RigCtrl_Close( pRC );   // <- doesn't do harm if already closed
  if( pRC->initState == initState_Closed )
   {  pRC->initState =  initState_Opened; // allow e.g. RigCtrl_ProcessIcomUDP(), RigCtrl_Handler(), and RigCtrl_ProcessRxData() to "work"
   }

  strcpy( sz80Msg, "RigControl: Re-connect" );  // show what's going on ..
  iMsgTypeForTrafficLog = RIGCTRL_MSGTYPE_FLAG_ERROR;
  RigCtrl_AddToTrafficLog( pRC, 0/*iRigCtrlPort*/, 0/*iRigCtrlOrigin*/, NULL, 0, iMsgTypeForTrafficLog, sz80Msg );

  RigCtrl_StartReading( pRC ); // -> RigCtrl_OpenSocket() -> RigCtrl_LaunchUDPThread() ...

} // end RigCtrl_ReConnect()


//-------------------------------------------------------------------- API --
void RigCtrl_SetClientOptions( T_RigCtrlInstance *pRC, int iRigCtrlPort,
         int iClientOptions ) // [in] bitwise combineable options like ..
         //   RIGCTRL_CLIENT_OPTION_REJECT_UNSOLICITED_MSGS ,
         //   RIGCTRL_CLIENT_OPTION_EMULATE_CIV_ECHO, ..(?)
{
  if( (iRigCtrlPort>=RIGCTRL_PORT_RADIO) && (iRigCtrlPort<=/*!*/RIGCTRL_MAX_CLIENT_PORTS) )
   { pRC->PortInstance[iRigCtrlPort].iClientOptions = iClientOptions;
     pRC->PortInstance[iRigCtrlPort].fEmulateCIVEcho = (iClientOptions & RIGCTRL_CLIENT_OPTION_EMULATE_CIV_ECHO) != 0;
   }
} // end RigCtrl_SetClientOptions()

//--------------------------------------------------------------------------
void RigCtrl_SetRadioControlProtocol( T_RigCtrlInstance *pRC,
  int iRadioCtrlProtocol ) // [in] RIGCTRL_PROTOCOL_ICOM_CI_V, etc(?) (*)
  // Only required to change the protocol 'on the fly', for example when
  // clicking 'Apply new settings and Start' on the GUI's "I/O Config" tab.
{

  // (*) .. About other "protocols" (Yeasu's name : "CAT Operation") :
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // The "PC CONTROL COMMAND REFERENCE FOR THE TS-480HX/ SAT TRANSCEIVER" says:
  // > A computer control command is composed of an alphabetical command,
  // > various parameters, and the terminator that signals the end
  // > of the control command.    (WB: The TS-480's terminator was a SEMICOLON)
  // > A command consists of 2 alphabetical characters.
  // > You may use either lower or upper case characters.
  // (the "Read command" omits the parameter, the "Set command" has it,
  //  and the "Answer command" looks exactly like a "Set command")
  //
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // The Yaesu "FT-857 Operating Manual" / "CAT Operation" shows a BINARY protocol, with..
  // > There are 17 instruction opcodes for the FT-857, listed in (..).
  // > Many of these opcodes are On/Off toggle commands for the same action
  // > (e.g. "PTT On" and "PTT Off"). Most of these commands require
  // > some parameter or parameters to be set. Irrespective of the number
  // > of parameters present, every Command Block sent must consist of
  // > five bytes.
  // For obscure reasons, Yaesu decided to send the parameters BEFORE the command,
  // which is one of the reasons why RCW Keyer will possibly never support this.
  // Commands and parameters cannot be told from each other by their BYTE VALUES.
  // A much better explanation than in Yaesu's "Operating Manuals"
  // used to be at https://www.ka7oei.com/ft817_meow.html .
  // > Note that careless use of this command can cause the FT-817's CPU
  // > to crash and/or cause a complete wipe of all EEPROM data including
  // > configuration, software calibration/alignment, and memories -
  // > the effect being similar to that of the "Reset to Factory Defaults"
  // > command (below!)  If you insist on using this command without
  // > first recording the 76 "soft calibration" settings and noting
  // > what is in your radio's memories, you are an idiot!
  // >   (Am I clear on this point?)    [thanks Clint; absolutely.)



  if( (pRC->PortInstance[RIGCTRL_PORT_RADIO].iRadioCtrlProtocol != iRadioCtrlProtocol) // REALLY restart ?
    ||(pRC->PortInstance[RIGCTRL_PORT_RADIO].dwPortErrorFlags & RIGCTRL_PORT_ERROR_FLAG_NOT_OPEN)
    )
   { RigCtrl_Close( pRC );       // stop communicating with the "old" protocol
     pRC->PortInstance[RIGCTRL_PORT_RADIO].iRadioCtrlProtocol = iRadioCtrlProtocol;
     pRC->PortInstance[RIGCTRL_PORT_RADIO].dwPortErrorFlags &= (~RIGCTRL_PORT_ERROR_FLAG_NOT_OPEN);
     // '--> Because BorlandC++Builder V6 is too stupid to evaluate the above
     //      expression in the debugger, WATCH the following instead:

     RigCtrl_ReConnect( pRC );   // start communicating with the "new" protocol
   }
} // end RigCtrl_SetRadioControlProtocol()

//--------------------------------------------------------------------------
void RigCtrl_SetPortErrorFlags( T_RigCtrlInstance *pRC,
        int iRigCtrlPort,        // [in] e.g. RIGCTRL_PORT_RADIO, RIGCTRL_PORT_AUX_COM_1, etc
        DWORD dwPortErrorFlags ) // [in] e.g. RIGCTRL_PORT_ERROR_FLAG_NOT_OPEN
                                 //      when there's trouble with the particular serial port
  // Added 2025-01 . First used to inform RigControl.c about trouble with
  //                 e.g. the serial port controlling the radio (iRgCtrlPort=RIGCTRL_PORT_RADIO).
  // Called from a 'housekeeping task', or from a timer in the main task,
  //   when one of the tasks chewing on I/O on the SERIAL PORTS gets in trouble.
  //   Example (in the Remote CW Keyer application):
  //      TKeyerMainForm::Timer1Timer(), when Keyer_hComPortRadioKeyingAndControl == INVALID_HANDLE_VALUE .
{
  DWORD dwNewErrorFlags;
  if( (iRigCtrlPort>=0 ) && (iRigCtrlPort <=/*!*/ RIGCTRL_MAX_CLIENT_PORTS ) )
   { dwNewErrorFlags = pRC->PortInstance[iRigCtrlPort].dwPortErrorFlags | dwPortErrorFlags;
     if( dwNewErrorFlags != pRC->PortInstance[iRigCtrlPort].dwPortErrorFlags )
      {  pRC->PortInstance[iRigCtrlPort].dwPortErrorFlags = dwNewErrorFlags; // <- set a breakpoint HERE !
      }
   }
} // end RigCtrl_SetPortErrorFlags()

//--------------------------------------------------------------------------
void RigCtrl_ClearPortErrorFlags( T_RigCtrlInstance *pRC,
        int iRigCtrlPort,        // [in] e.g. RIGCTRL_PORT_RADIO, RIGCTRL_PORT_AUX_COM_1, etc
        DWORD dwPortErrorFlagsToClear ) // [in] e.g. RIGCTRL_PORT_ERROR_FLAG_NOT_OPEN
        // to *selectively* clear that flag (and leave the others unchanged),
        // or RIGCTRL_PORT_ERROR_FLAGS_ALL to clear ALL 'error flags' internally.
  // Like RigCtrl_SetPortErrorFlags(), possibly called from a 'housekeeping task',
  // or from a timer in the main task, etc etc.
{
  DWORD dwNewErrorFlags;
  if( (iRigCtrlPort>=0 ) && (iRigCtrlPort <=/*!*/ RIGCTRL_MAX_CLIENT_PORTS ) )
   { dwNewErrorFlags = pRC->PortInstance[iRigCtrlPort].dwPortErrorFlags & (~dwPortErrorFlagsToClear);
     if( dwNewErrorFlags != pRC->PortInstance[iRigCtrlPort].dwPortErrorFlags )
      {  pRC->PortInstance[iRigCtrlPort].dwPortErrorFlags = dwNewErrorFlags; // <- set a breakpoint HERE !
      }
   }
} // end RigCtrl_ClearPortErrorFlags()


//--------------------------------------------------------------------------
static void RigCtrl_ForgetScopeFrequencyRanges( T_RigCtrlInstance *pRC )
{  int iRange, iEdge;
   pRC->iScopeNumFreqRanges = pRC->iScopeNumEdgesPerFreqRange = 0;
   for( iRange=0; iRange<RIGCTRL_MAX_FIXED_EDGE_SCOPE_FREQ_RANGES; ++iRange)
    { for( iEdge=0; iEdge<RIGCTRL_MAX_FIXED_EDGES_PER_FREQ_RANGE; ++iEdge)
       { pRC->ScopeFreqRange[iRange][iEdge].dblFmin_Hz = 0.0;
         pRC->ScopeFreqRange[iRange][iEdge].dblFmin_Hz = 0.0;
       }
    }
} // end RigCtrl_ForgetScopeFrequencyRanges()

//--------------------------------------------------------------------------
int RigCtrl_SwitchPollingState( T_RigCtrlInstance *pRC, int iNewPollingState )
      // May be called from the appliction, to modify pRC->iParameterPollingState
      //  (which, since the introduction of other Rig-Control-Protocols besides CI-V,
      //   may involve a bit more than just setting pRC->iParameterPollingState)
      // The RETURN VALUE is one of the RIGCTRL_POLLSTATE_.. valued defined in RigControl.h,
      // in most (but not all) cases the same as iNewPollingState .
{
  if( pRC->iParameterPollingState != iNewPollingState )
   { pRC->iParameterPollingSubState = pRC->iParameterPollingArrayIndex = 0;
   }

  pRC->iParameterPollingState = iNewPollingState; // e.g. RIGCTRL_POLLSTATE_START_RD

# if( SWI_SUPPORT_YAESU_5_BYTE_CAT )
  if( pRC->PortInstance[RIGCTRL_PORT_RADIO].iRadioCtrlProtocol == RIGCTRL_PROTOCOL_YAESU_5_BYTE )
   {  pRC->iParameterPollingState = Yaesu5Byte_SwitchPollingState( &pRC->Y5B, pRC->iParameterPollingState );
   }
# endif // SWI_SUPPORT_YAESU_5_BYTE_CAT ?

  return pRC->iParameterPollingState;

} // end RigCtrl_SwitchPollingState()


//--------------------------------------------------------------------------
void RigCtrl_StartReading( T_RigCtrlInstance *pRC )
  // Prepares reading "all we need to know" about the remotely controlled rig,
  //  but doesn't really 'communicate' via serial port, LAN or WLAN yet.
  //  For Icom radios connected via LAN or WLAN, also tries to open the
  //  required ports (in this case UDP sockets), and prepares to log-in .
  // Called from the application (e.g. TConfigDlg::ApplyMyControls() ),
  //        or (if a simple SERIAL PORT is used for CI-V) IOaccess.cpp : IO_OpenComPorts(),
  //        or (if UDP is used for CI-V) called from "RigControl.c" itself,
  //   to begin the first PARAMETER POLLING CYCLE via CI-V .
  // Communication parameters like pRC->iRadioDeviceAddr, iRadioCtrlProtocol
  //        and  [removed] iInetProtocol ( INET_PROTOCOL_OFF/HTTP/TCP/UDP/RAW/COM_PORT)
  //   must have been set by the application prior to this.
  // The rest (communication) happens in RigCtrl_Handler(); details there.
{
  T_RigCtrl_PortInstance *pPortInstance = &pRC->PortInstance[RIGCTRL_PORT_RADIO];

  // 'Forget' various counters and values read during a previous session:
  pRC->iRadioPortTxFifoHead = pRC->iRadioPortTxFifoTail = 0;
  pRC->nCompleteSpectraReceived   = 0;
  pRC->dblUnixTimeOfFirstSpectrum = 0.0;

  pRC->iParameterPollingSubState = pRC->iParameterPollingArrayIndex = 0; // don't leave anything to fate !
  pPortInstance->iResponseCountdown_ms = 0;
  RigCtrl_ForgetAllParams( pRC ); // forget "everything" previously read from the rig, e.g. VFO frequency, Op-Mode, etc
  TIM_StartStopwatch( &pRC->sw_LastUpdateOfBasicParams );
  RigCtrl_SwitchPollingState( pRC, RIGCTRL_POLLSTATE_START_RD );

} // end RigCtrl_StartReading()


//--------------------------------------------------------------------------
void RigCtrl_ModifyMsgType(int *piMsgType, // [in,out] message type (to/from a radio or client port)
                           int iMsgMask,   // [in]
                           int iNewType )
  // Copies only those bits from iNewType to *piMsgType that are SET(!) in iMsgMask.
  //      Formerly only important for the traffic log, the "message type"
  //      now plays an important role for storing/forwarding messages
  //      between 'additional ports' for external clients and the REAL RADIO.
  // Sample usage to change the basic type to "Frequency Report"
  // without modifying the 'Flags' and the indicators for 'Ok/Not Ok':
  //  > RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_BASIC, RIGCTRL_MSGTYPE_FREQUENCY_REPORT );
{ *piMsgType = (*piMsgType & (~iMsgMask) ) | ( iNewType & iNewType );
}

//---------------------------------------------------------------------------
BOOL RigCtrl_IsTrafficMonitorEnabled( T_RigCtrl_PortInstance *pPortInstance )
{ int iRigCtrlPort = RigCtrl_PortInstancePtrToIndex( pPortInstance );
  if( iRigCtrlPort == RIGCTRL_PORT_RADIO )
   { // for the RADIO CONTROL PORT, the traffic monitor (ex: "CI-V log")
     // is enabled/disabled via the following flag:
     if( RigCtrl_TrafficMonitor.iDisplayOptions & RIGCTRL_TMON_DISPLAY_OPTION_ENABLE )
      { return TRUE;
      }
     // After a command has been forwarded from an external client to the RADIO PORT,
     // the traffic monitor may have been "temporarily unpaused" :
     if( pPortInstance->iTrafficMonitorUnpauseCountdown > 0 ) // here: checked in RigCtrl_IsTrafficMonitorEnabled()
      { return TRUE;
        // 2025-09-05 : After receiving a command from a REMOTE CLIENT on an 'Additional COM Port' (N1MM Logger+),
        //              fTrafficMonitorTemporarilyUnpaused was SET for the RADIO CONTROL PORT
        //              when forwarding a command "on behalf of the remote client" to the radio.
        //              That's ok. But fTrafficMonitorTemporarilyUnpaused was NEVER CLEARED,
        //              even when the remote client stopped sending further commands,
        //              and the traffic log was flooded by SPECTRUM SCOPE FRAGMENTS again.
        //  Fixed by replacing the former FLAG (fTrafficMonitorTemporarilyUnpaused)
        //                      by a COUNTDOWN (iTrafficMonitorUnpauseCountdown),
        //              decremented on each message received or sent on that port.
      }
   }
  else // for all other ("additional") COM ports, the traffic monitor
   { // is individually enabled/disabled via the "debug" option, via this flag:
     if( pPortInstance->fTrafficMonitorEnabled )
      { return TRUE;
      }
   }
  return FALSE;
} // end RigCtrl_IsTrafficMonitorEnabled()


//---------------------------------------------------------------------------
BOOL RigCtrl_IsMsgTypeRejectedForLog( T_RigCtrl_PortInstance *pPortInstance, // API (public)
        int iMsgType) // [in] specific TYPE CODE in lower bits (e.g. 4=RIGCTRL_MSGTYPE_SPECTRUM),
        //  additional TYPE FLAGS in higher bits (e.g. RIGCTRL_MSGTYPE_FLAG_RX,TX,...) .
        //  Example for iMsgType (seen very often, also on VIRTUAL RADIO ports):
        //  iMsgType = 0x0001801F = RIGCTRL_MSGTYPE_FLAG_READ_CMD(1<<16) | RIGCTRL_MSGTYPE_FLAG_PERIODIC_POLL(1<<15) | RIGCTRL_MSGTYPE_OTHER(31) .
  // [in] pPortInstance->pRC->dwMessageFilterForLog : bitwise combineable flags like ..
  //         RIGCTRL_MSGFILTER_ECHO              (1<<0),
  //         RIGCTRL_MSGFILTER_PERIODIC_POLL     (1<<1),
  //         RIGCTRL_MSGFILTER_SPECTRUM          (1<<2),
  //         RIGCTRL_MSGFILTER_FREQUENCY_REPORT  (1<<3),
  //         RIGCTRL_MSGFILTER_ANY_KNOWN_COMMAND (1<<8).
  // [return] TRUE  = "reject THIS MESSAGE from the traffic log",
  //          FALSE = "don't reject from the log" (but show it somewhere).
  //
  // Note: If messages received on / sent to a VIRTUAL RIG port ('Additional COM Port' with that configuration),
  //       do NOT appear in the 'Traffic Log' even though they should,
  //       check if the "debug" token is present in the 'Additional COM Port' dialog-window !
  //
{
  BOOL fReject = FALSE;
  DWORD dwMessageFilterForLog = pPortInstance->pRC->dwMessageFilterForLog;
  int iBasicMsgType = iMsgType & RIGCTRL_MSGMASK_BASIC;
  int iRigCtrlPort = RigCtrl_PortInstancePtrToIndex( pPortInstance );

  if( iRigCtrlPort == RIGCTRL_PORT_RADIO )
   { // for the RADIO CONTROL PORT, the traffic monitor (ex: "CI-V log")
     // is enabled/disabled via the following flag:
     if( (RigCtrl_TrafficMonitor.iDisplayOptions & RIGCTRL_TMON_DISPLAY_OPTION_ENABLE) == 0 )
      { // After a command has been forwarded from an external client to the RADIO PORT,
        // the traffic monitor may have been "temporarily unpaused" :
        if( pPortInstance->iTrafficMonitorUnpauseCountdown > 0 ) // here: checked in RigCtrl_IsMsgTypeRejectedForLog()
         { // special case: Do NOT reject in the traffic log because it was 'forwarded' on behalf of an external client,
           // and already 'filtered' in RigCtrl_ForwardCommandFromServerPortToRadio().
         }
        else
         { return TRUE; // reject traffic on the RADIO CONTROL PORT from the log (traffic monitor) because the monitor is NOT ENABLED
         }
      }
   }
  else // for all other ("additional") COM ports, the traffic monitor
   { // is individually enabled/disabled via the "debug" option, via this flag:
     if( !pPortInstance->fTrafficMonitorEnabled )
      { // '--> At the moment, all port instances are controlled by a single flag
        //      in RigCtrl_TrafficMonitor.iDisplayOptions, bit RIGCTRL_TMON_DISPLAY_OPTION_ENABLE.
        return TRUE; // reject from the log (traffic monitor) because the monitor is NOT ENABLED at all
      }
   }

  if( dwMessageFilterForLog & RIGCTRL_MSGFILTER_ECHO )
   { if( iMsgType & RIGCTRL_MSGTYPE_FLAG_ECHO )
      { fReject = TRUE;
      }
   }

  if( dwMessageFilterForLog & RIGCTRL_MSGFILTER_FREQUENCY_REPORT )
   { if( iBasicMsgType == RIGCTRL_MSGTYPE_FREQUENCY_REPORT )
      { fReject = TRUE;
      }
   }

  if( dwMessageFilterForLog & RIGCTRL_MSGFILTER_SPECTRUM )
   { if( iBasicMsgType == RIGCTRL_MSGTYPE_SPECTRUM )
      { fReject = TRUE;
      }
   }

  if( dwMessageFilterForLog & RIGCTRL_MSGFILTER_PERIODIC_POLL )
   { if( iMsgType & RIGCTRL_MSGTYPE_FLAG_PERIODIC_POLL )
      { fReject = TRUE;
      }
   }


  return fReject;
} // end RigCtrl_IsMsgTypeRejectedForLog()


//--------------------------------------------------------------------------
BOOL RigCtrl_SendAndLogMessage( T_RigCtrlInstance *pRC,      // API (public)
        int iRigCtrlPort,   // [in] RIGCTRL_PORT_RADIO / RIGCTRL_PORT_AUX_COM_1 / 2 ?
        int iRigCtrlOrigin, // [in] usually RIGCTRL_ORIGIN_CONTROLLER, but who knows.. we may pretend to "be a radio"
        BYTE *pbMessage, int iMsgLength, // [in] message, including pre- and postamble but not the UDP-CIV-HEADER (!)
        int iMsgType,     // [in] message type category and 'flags', only for the traffic log
                          //      e.g. RIGCTRL_MSGTYPE_FLAG_CIV | RIGCTRL_MSGTYPE_FLAG_TX .
        char* pszComment) // [in] human readable info, short enough for the traffic log
  // Sends any message (not limited to CI-V), counts, and optionally logs it.
  // (The real transmission on e.g. a serial port runs in a separate thread
  //  or interrupt handler.
  //  In the Remote CW Keyer, pRC->pRadioPortTxFifo is drained in KeyerThread.c :
  //  KeyerThread() -> CFIFO_GetNumBytesReadable() + CFIFO_Read() + WriteFile().
  //  That way, the various serial port API functions are called from the same
  //  thread, without the need to wait for a critical section, Mutex, etc.  )
  // Since 2025-06, also called from e.g. Yaesu5Byte.c, and possibly other
  //                Rig-Control-submodules for other radio control protocols.
  //
  // Messages SENT this way MAY be displayed by [v] Show CAT traffic,
  //    highlighted with a YELLOW background - see RigCtrl_ReadTrafficLog().
  //
{
  T_RigCtrl_PortInstance *pPortInstance = &pRC->PortInstance[iRigCtrlPort];
  int iResult;
  char sz255Msg[256];
  BOOL fResult=TRUE;
  BOOL fShowInLog;

  RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RX_TX, RIGCTRL_MSGTYPE_FLAG_TX );
  if( pPortInstance->iRadioCtrlProtocol == RIGCTRL_PROTOCOL_ICOM_CI_V )
   { iMsgType |= RIGCTRL_MSGTYPE_FLAG_CIV; // <- for the TRAFFIC LOG
   }

  fShowInLog = RigCtrl_IsTrafficMonitorEnabled( pPortInstance );
  if( RigCtrl_IsMsgTypeRejectedForLog( pPortInstance, iMsgType) ) // here: called from RigCtrl_SendAndLogMessage()
   { fShowInLog = FALSE;
   }

#if(0) // TEST (when suspecting responses were sent on a SERVER PORT without being displayed)
  if( iRigCtrlPort != RIGCTRL_PORT_RADIO )
   { fShowInLog = TRUE;
   }
#endif // TEST ?

  if( (iRigCtrlPort==RIGCTRL_PORT_RADIO) && (pRC->pRadioPortTxFifo!=NULL) )
   {
#   if(SWI_HARDCORE_DEBUGGING)
     if( RigCtrl_iConditionalBreakpointStep == 3 ) // forwarding a "SET Split Mode" command from an AUX COM PORT to the real radio ?
      { RigCtrl_iConditionalBreakpointStep = 4;   // just entered RigCtrl_SendWriteCommandForUnifiedPN() for the forwarded "Split Mode"-command !
        RigCtrl_AddToTrafficLog( pRC, iRigCtrlPort, iRigCtrlOrigin, NULL/*pbMessage*/, 0/*iMsgLength*/,
              RIGCTRL_MSGTYPE_FLAG_ERROR, "Hit conditional breakpoint step 3 in RigCtrl_SendAndLogMessage" );
        // 2025-10-19 : Got here when firing up N1MM Logger+, but the FORWARDED "Set Split Mode"-command
        //              wasn't sent to the real radio (IC-7300) . EXPECTED to see the follwing at this point:
        // iRigCtrlPort   = 0 = RIGCTRL_PORT_RADIO        (ok)
        // iRigCtrlOrigin = 1 = RIGCTRL_ORIGIN_CONTROLLER (ok)
        // iMsgLength = 7  (ok)
        // iMsgType   = 0x0030908 = RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND (lower byte) | RIGCTRL_MSGTYPE_FLAG_TX (bit 8) | RIGCTRL_MSGTYPE_FLAG_CIV (bit 11) | RIGCTRL_MSGTYPE_FLAG_WRITE_CMD (3<<16)
        // pbMessage[0..7] = 0xFE 0xFE 0x94 0xE0 0x0F 0x00 0xFD  (ok)
        //                                            /|\         .
        //        0x00 = split mode OFF, 0x01 = ON ----'
        // In the CI-V traffic log:
        // > 826 0012807 r2 007 FE FE 94 E0 0F 00 FD  ; SplitMode:off  (command received on a "Virtual Rig" port) ---, 45 ms wasted..
        // > 831 0012852 TX 007 FE FE 94 E0 0F 00 FD  ; write SplitMode (command forwarded to the "real radio") -----' in between !
        // > 832 0012912 RX 03C FE FE E0 94 27 00 00 06 11 00 04 03 0B 05 0B 0D 00 0A 00 00 00 09 0B ..  ; spectrum fragment 6/11
        // > 833 0012912 RX 006 FE FE E0 94 FB FD     ; SplitMode : OK (response from the "real radio" arrived 60 ms after the command)
        // > 834 0012912 t2 006 FE FE E0 94 FB FD     ; SplitMode : OK (response sent by the "Virtual Rig", with almost no extra delay)
        // > 930 0014461 r2 007 FE FE 94 E0 07 00 FD  ; SelectVFO:00 (next command sent by N1MM 1.5 seconds after the previous response)
        // > 934 0014475 TX 007 FE FE 94 E0 07 00 FD  ; SelectVFO:00 (this time, less delay between the threads for all these ports)
        //               ; but note the missing response ("RX" from the IC-7300) here.
        //               ; See "serial port tunnel" for comparison: the "OK" for "SelectVFO" already arrived after only FOUR ms)
        // >     0014527  Illegal transition from server state forward_busy to parsing_cmd
        // > 935 0014527 r2 007 FE FE 94 E0 0F 00 FD  ; SplitMode:off  (why did N1MM Logger+ send the same command again, only 66 ms after the previous command ?
        // Compare the above with the timing when RCWK was running as "serial port tunnel"
        // between N1MM and an IC-7300, saved in 2025_08_16_N1MM and RCWK with winkeyer emulation.txt :
        // >   2 0016157 t3 007 FE FE 94 E0 0F 00 FD  ; SplitMode:off    ("t3","r3" = serial port tunnel with an IC-7300)
        // >   3 0016159 r3 006 FE FE E0 94 FB FD     ; SplitMode : OK --, 1.8 seconds between THIS RESPONSE
        // >   6 0017975 t3 007 FE FE 94 E0 07 00 FD  ; SelectVFO:00   --' .. and the NEXT COMMAND (sent here)
        // >   7 0017979 r3 006 FE FE E0 94 FB FD     ; OK for cmd 0x700FFFF -, only 71 ms
        // >  10 0018050 t3 007 FE FE 94 E0 0F 00 FD  ; SplitMode:off  -------' in between !
        // >  11 0018057 r3 006 FE FE E0 94 FB FD     ; SplitMode : OK -------, only 62 ms
        // >  14 0018119 t3 007 FE FE 94 E0 07 00 FD  ; SelectVFO:00   -------' in between !
        // >  15 0018127 r3 006 FE FE E0 94 FB FD     ; OK for cmd 0x700FFFF -, 116 ms
        // >  22 0018243 t3 006 FE FE 94 E0 03 FD     ; VFOfreq ? ------------' in between !
        // >  23 0018251 r3 00B FE FE E0 94 03 00 65 03 07 00 FD ; VFOfreq=7036500.0 Hz
        // Hmm.. maybe N1MM Logger+ doesn't wait for responses, before sending the next command ?
        //       That would be a show stopper when operating "truely remote" !
      }
#   endif // SWI_HARDCORE_DEBUGGING ?

#   if(1)
     if( (pbMessage[4]==0x25) && (pbMessage[5]==0x00) ) // sending cmd "SelVFOFreq" to the REAL RADIO (e.g. forward from WSJT-X) ?
      { pbMessage[4] = pbMessage[4]; // place for a 'conditional' breakpoint
        // (seen in the traffic log:
        //    r2 00C FE FE 94 E0 25 00 00 01 00 00 00 FD ; SelVFOFreq:100 (from WSJT-X)
        //    TX 00C FE FE 94 E0 25 00 00 01 00 00 00 FD ; SelVFOFreq:100 (to the REAL RADIO)
        //     > Illegal transition from server state forward_busy to parsing_cmd
        //   (the response from the real radio was supressed, without the
        //    special treatment with 'if( pRadioPortInstance->iWorkingForServerPort .. )'
        // )
      }
#   endif
     iResult = CFIFO_Write( pRC->pRadioPortTxFifo, pbMessage, iMsgLength, 0.0/*no timestamp*/ );
     // '--> only a NEGATIVE result indicates a 'real' error. Zero = FALSE = "not finished yet"
     if( iResult >= 0 )
      { // Modified 2023-08-15 : If the port is a COM port, RIGCTRL_TMON_DISPLAY_OPTION_SHOW_UDP_PAYLOAD
        // is meaningless . Break up the too complex 'if' condition further below.
        if( pPortInstance->pAppCallback != NULL ) // did THE USER of RigControl.c register a callback ?
         { // Let some other module (e.g. AuxComPorts.c) know we have SENT another message on this port:
           pPortInstance->pAppCallback( pPortInstance, pRC->iRigCtrlOrigin, RIGCTRL_CBK_SEND_MSG, pbMessage, iMsgLength, pszComment );
         }
        ++pPortInstance->dwNumMessagesSent;    // here: in RigCtrl_SendAndLogMessage(), for RIGCTRL_PORT_RADIO
        ++RigCtrl_i32MessageNrForTrafficLog;   // increment regardless of being DISPLAYED in the log or not !
        pPortInstance->fLastSendFailed = FALSE;
        fResult = TRUE;
      }
     else // CFIFO_Write( pRC->pRadioPortTxFifo ) failed .. "tried to send more than possible" ?
      { if( !pPortInstance->fLastSendFailed )
         {   pPortInstance->fLastSendFailed = TRUE;
           snprintf( sz255Msg,250, "Failed to send to %s",
                RigCtrl_RigControlPortNrToString(iRigCtrlPort) );
           RigCtrl_AddToTrafficLog( pRC, iRigCtrlPort, iRigCtrlOrigin, NULL, 0, RIGCTRL_MSGTYPE_FLAG_ERROR, sz255Msg );
         }
      }
   } // end if( (iRigCtrlPort==RIGCTRL_PORT_RADIO) && (pRC->pRadioPortTxFifo!=NULL) )
  else if( (iRigCtrlPort>=RIGCTRL_PORT_AUX_COM_1) && (iRigCtrlPort<=RIGCTRL_PORT_AUX_COM_LAST) )
   { // The 'Aux COM Ports' (renamed to 'Additional COM Ports' for the GUI)
     // run in their own threads, and have their own FIFOs for transmission.
     // So to SEND on any of them, use their callback functions and RIGCTRL_CBK_SEND_MSG.
     // To-be-sent data are passed MESSGAGE BY MESSAGE, because for some protocols
     // (higher-level or transport layer), messages may have to be wrapped in
     // packets, datagrams, or whatever their name will be:
     if( pPortInstance->pAppCallback != NULL )
      { // Let some other module (e.g. AuxComPorts.c) know we have SENT another message on this port:
        iResult = pPortInstance->pAppCallback( pPortInstance, iRigCtrlOrigin,
                    RIGCTRL_CBK_SEND_MSG, pbMessage, iMsgLength, pszComment );
        fResult = (iResult >= 0);
        if( fResult )  // here: transmission on an Additional COM Port successful, so count it, too:
         { ++pPortInstance->dwNumMessagesSent;  // here: in RigCtrl_SendAndLogMessage(), for RIGCTRL_PORT_AUX_COM_1..AUX_COM_LAST
           ++RigCtrl_i32MessageNrForTrafficLog;
           pPortInstance->fLastSendFailed = FALSE;
         }
      }
   } // end if < don't send on the RADIO PORT but on one of the AUX COM PORTS > ?

  if( fResult && fShowInLog )
   { RigCtrl_AddToTrafficLog( pRC, iRigCtrlPort, iRigCtrlOrigin, pbMessage,
              iMsgLength, iMsgType | RIGCTRL_MSGTYPE_FLAG_TX, pszComment );
   }


  return fResult;

} // end RigCtrl_SendAndLogMessage()


//--------------------------------------------------------------------------
BOOL RigCtrl_SendShortCIVMessage( T_RigCtrlInstance *pRC,
        int iRigCtrlPort, // [in] RIGCTRL_PORT_RADIO / RIGCTRL_PORT_AUX_COM_1 / 2 ?
        int iRigCtrlOrigin, // [in] usually RIGCTRL_ORIGIN_CONTROLLER, but who knows..
        int iMsgType,     // [in] message type category and 'flags', only for the traffic log
        char *pszComment, // [in] human readable info, short enough for the traffic log
        int iCmd,         // [in] CI-V main command like 0x27 for "Scope waveform data"
        int iSubCmd,      // [in] optional sub-command.        -1 to omit.
        int iDataByte1,   // [in] optional 1st parameter byte. -1 to omit.
        int iDataByte2 )  // [in] optional 2nd parameter byte. -1 to omit.
  // Sends a 'short' CI-V message, usually a 'request' for something.
  // To send WITHOUT a "sub-command" (byte), use iSubCmd = -1.
  // To send WITHOUT a "data byte", use iDataByte1 = -1,  etc.
  // Returns TRUE when successfully sent, otherwise FALSE.
  //
  // About the TRAFFIC LOG: See RigCtrl_SendAndLogMessage() [called from here].
  //
{
  BYTE b16TxBuffer[16];
  BYTE *pbBufPtr = b16TxBuffer;
  int  iMsgLength;
  T_RigCtrl_PortInstance *pPortInstance = &pRC->PortInstance[iRigCtrlPort];


  *pbBufPtr++ = 0xFE;  // fixed preamble code for CI-V protocol (term used by Icom .. NOT "Begin of Message" ! )
  *pbBufPtr++ = 0xFE;  // fixed preamble code for CI-V protocol
  *pbBufPtr++ = (BYTE)pPortInstance->iRadioDeviceAddr;  // "to" the transceiver's address
                       // (may be 0x00 = RIGCTRL_DEF_ADDR_AUTO_DETECT initially)
  *pbBufPtr++ = (BYTE)pPortInstance->iRadioMasterAddr;  // "from" the controller's address
                       // (typically 0xE0 or 0xE1; the radio seems to accept anything)
  if( iCmd >= 0 )
   { *pbBufPtr++ = (BYTE)iCmd;  // e.g. 0x03 = CI-V command "Read operating frequency"
   }
  if( iSubCmd >= 0 )
   { *pbBufPtr++ = (BYTE)iSubCmd;
   }
  if( iDataByte1 >= 0 )
   { *pbBufPtr++ = (BYTE)iDataByte1;
   }
  if( iDataByte2 >= 0 )
   { *pbBufPtr++ = (BYTE)iDataByte2;
   }
  *pbBufPtr++ = 0xFD;  // CI-V code for "End Of Message" (term used by Icom .. NOT "postamble", even though 0xFE is called "preamble" ! )

  iMsgLength = pbBufPtr - b16TxBuffer;
  return RigCtrl_SendAndLogMessage( pRC, iRigCtrlPort, iRigCtrlOrigin, b16TxBuffer, iMsgLength,
         RIGCTRL_MSGTYPE_FLAG_CIV | iMsgType, pszComment );
} // end RigCtrl_SendShortCIVMessage()


//--------------------------------------------------------------------------
BOOL RigCtrl_SendMidSizeCIVMessage( T_RigCtrlInstance *pRC,
        int iRigCtrlPort, // [in] RIGCTRL_PORT_RADIO / RIGCTRL_PORT_AUX_COM_1 / 2 ?
        int iRigCtrlOrigin, // [in] usually RIGCTRL_ORIGIN_CONTROLLER, but who knows..
        int iMsgType,     // [in] message type category and 'flags', only for the traffic log
        char *pszComment, // [in] human readable info, short enough for the traffic log
        int iCmd,         // [in] CI-V main command like 0x27 for "Scope waveform data"
        int iSubCmd,      // [in] optional sub-command.        -1 to omit.
        int iDataByte1,   // [in] optional 1st parameter byte. -1 to omit.
        int iDataByte2,   // [in] optional 2nd parameter byte. -1 to omit.
        int iDataByte3,   // [in] optional 3rd parameter byte. -1 to omit.
        int iDataByte4 )  // [in] optional 4th parameter byte. -1 to omit.
  // Sends a 'mid-sized' CI-V message, for example a "Scope Reference Level Setting".
  // To send WITHOUT a "sub-command" (byte), use iSubCmd = -1.
  // To send WITHOUT a "data byte", use iDataByte1 = -1,  etc.
  // Returns TRUE when successfully sent, otherwise FALSE. KISS.
  //
  // About the TRAFFIC LOG: See RigCtrl_SendAndLogMessage() [called from here].
  //
{
  BYTE b16TxBuffer[16];
  BYTE *pbBufPtr = b16TxBuffer;
  int  iMsgLength;
  T_RigCtrl_PortInstance *pPortInstance = &pRC->PortInstance[iRigCtrlPort];

  *pbBufPtr++ = 0xFE;  // fixed preamble code for CI-V protocol (term used by Icom .. NOT "Begin of Message" ! )
  *pbBufPtr++ = 0xFE;  // fixed preamble code for CI-V protocol
  *pbBufPtr++ = (BYTE)pPortInstance->iRadioDeviceAddr;  // "to" transceiver's address
  *pbBufPtr++ = (BYTE)pPortInstance->iRadioMasterAddr;  // "from" controller's address
  if( iCmd >= 0 )
   { *pbBufPtr++ = (BYTE)iCmd;    // e.g. 0x27 = CI-V command "Scope Settings"
   }
  if( iSubCmd >= 0 )
   { *pbBufPtr++ = (BYTE)iSubCmd; // e.g. 0x19 = "Scope Reference level settings" (in an IC-9700)
   }
  if( iDataByte1 >= 0 )
   { *pbBufPtr++ = (BYTE)iDataByte1; // e.g. 0x00 = Main scope, 0x01 = Sub Scope (IC-9700 CI-V Manual page 24)
   }
  if( iDataByte2 >= 0 )
   { *pbBufPtr++ = (BYTE)iDataByte2; // e.g. 10- and 1-dB digit (in the usual BCD format)
   }
  if( iDataByte3 >= 0 )
   { *pbBufPtr++ = (BYTE)iDataByte3; // e.g. 0.1- and 0.01-dB digit (the latter often fixed to ZERO)
   }
  if( iDataByte4 >= 0 )
   { *pbBufPtr++ = (BYTE)iDataByte4; // e.g. 0x00 = POSITIVE,  0x01 = NEGATIVE (quite counter-intuitive..)
   }
  *pbBufPtr++ = 0xFD;  // CI-V code for "End Of Message" (term used by Icom .. NOT "postamble", even though 0xFE is called "preamble" ! )

  iMsgLength = pbBufPtr - b16TxBuffer;
  return RigCtrl_SendAndLogMessage( pRC, iRigCtrlPort, iRigCtrlOrigin, b16TxBuffer, iMsgLength,
           RIGCTRL_MSGTYPE_FLAG_CIV | iMsgType, pszComment );
} // end RigCtrl_SendMidSizeCIVMessage()

//--------------------------------------------------------------------------
BOOL RigCtrl_SendCommandToTurnRigOn_CIV( T_RigCtrlInstance *pRC,
        int iRigCtrlPort,   // [in] RIGCTRL_PORT_RADIO / RIGCTRL_PORT_AUX_COM_1 / 2 ?
        int iRigCtrlOrigin) // [in] usually RIGCTRL_ORIGIN_CONTROLLER, but who knows..
  // Sends a special CI-V message to turn a modern Icom radio on, via CI-V.
  // Returns TRUE when successfully sent, otherwise FALSE.
  //
  // About the TRAFFIC LOG: See RigCtrl_SendAndLogMessage() [called from here].
  //
{
  BYTE b256TxBuffer[256];
  BYTE *pbBufPtr = b256TxBuffer;
  int  i, iMsgLength;
  T_RigCtrl_PortInstance *pPortInstance = &pRC->PortInstance[iRigCtrlPort];


  // From one of the dozens of Icom's "CI-V Reference Guides" ("A7380-7EX"):
  // > When sending the power ON command (18 01), you need
  // > to repeatedly send "FE" before the standard format.
  // > The following is the approximate number of needed repetitions.
  // >  * 115200 bps: 150 "FE"s
  // >  *  57600 bps: 75 "FE"s
  // >  *  38400 bps: 50 "FE"s
  // >  *  19200 bps: 25 "FE"s
  // >  *   9600 bps: 13 "FE"s
  // >  *   4800 bps: 7 "FE"s
  for( i=0; i<150; ++i ) // keep it simple, assume we need the MAXIMUM number of "FE"s
   { *pbBufPtr++ = 0xFE;  // fixed preamble code for CI-V protocol, and obviously also Icom's "wake-up pattern"
   }
  *pbBufPtr++ = (BYTE)pPortInstance->iRadioDeviceAddr;  // "to" transceiver's address
  *pbBufPtr++ = (BYTE)pPortInstance->iRadioMasterAddr;  // "from" controller's address
  *pbBufPtr++ = 0x18; // 0x18 = CI-V command "Turn OFF / ON the transceiver"..
  *pbBufPtr++ = 0x01; // 0x01 = subcommand to turn *ON*, not off
  *pbBufPtr++ = 0xFD; // CI-V code for "End Of Message"
  iMsgLength = pbBufPtr - b256TxBuffer;
  return RigCtrl_SendAndLogMessage( pRC, iRigCtrlPort, iRigCtrlOrigin, b256TxBuffer, iMsgLength,
          RIGCTRL_MSGTYPE_FLAG_CIV | RIGCTRL_MSGTYPE_POWER_ON_OFF, "turn ON" );
} // end RigCtrl_SendCommandToTurnRigOn_CIV()


#if(0) // old stuff ...  the "Radio Port TX FIFO" was replaced by COORDINATING activities
       // between the "Radio Control Port" and the optional "Client Ports" / Virtual Rig Ports / etc,
       // in 2025-08  -  see AuxComPorts.c (for 'serial ports tunnels'),
       //                    RigControl_CIV_Server.c (for 'Virtual Rigs' serving external clients),
       //               and  CwNet.c (for the rig-independent TCP/IP link between "shack" and "remote site").
//--------------------------------------------------------------------------
void RigCtrl_AppendByteToRadioPortTxFifo( BYTE *pbBuffer, int *piHeadIndex, BYTE b )
{
  *piHeadIndex &= (RIGCTRL_RADIO_PORT_TX_FIFO_SIZE-1); // safety first..
  pbBuffer[*piHeadIndex] = b;
  *piHeadIndex = (*piHeadIndex + 1) & (RIGCTRL_RADIO_PORT_TX_FIFO_SIZE-1);
}

//--------------------------------------------------------------------------
void RigCtrl_AppendBlockToRadioPortTxFifo( BYTE *pbBuffer, int *piHeadIndex, BYTE *pbSource, int nBytes )
{
  while( (nBytes--) > 0 )
   { RigCtrl_AppendByteToRadioPortTxFifo( pbBuffer, piHeadIndex, *(pbSource++) );
   }
}

//--------------------------------------------------------------------------
BYTE RigCtrl_ReadByteFromRadioPortTxFifo( BYTE *pbBuffer, int *piTailIndex )
{
  BYTE b;
  *piTailIndex &= (RIGCTRL_RADIO_PORT_TX_FIFO_SIZE-1); // safety first..
  b = pbBuffer[*piTailIndex];
  *piTailIndex = (*piTailIndex + 1) & (RIGCTRL_RADIO_PORT_TX_FIFO_SIZE-1);
  return b;
}

//--------------------------------------------------------------------------
void RigCtrl_ReadBlockFromRadioPortTxFifo( BYTE *pbBuffer, int *piTailIndex, BYTE *pbDest, int nBytes )
{
  while( (nBytes--) > 0 )
   { *(pbDest++) = RigCtrl_ReadByteFromRadioPortTxFifo( pbBuffer, piTailIndex );
   }
}

typedef struct t_RIGCTL_FIFO_MESSAGE_HEADER // contains everything that isn't actually SENT ..
{
  int iMsgLength;   // number of payload bytes following after the T_RIGCTL_FIFO_MESSAGE_HEADER
  int iMsgType;     // bitwise combination like RIGCTRL_MSGTYPE_FREQUENCY_REPORT, etc^10
  int iRigCtrlPort; // e.g. RIGCTRL_PORT_AUX_COM_1/2/3 to identify the "remote client"
  char sz80Comment[84];   // info for e.g. the TRAFFIC LOG, used e.g. when pulling the message from the radio port's TX FIFO and *SENDING* it (really)
} T_RIGCTL_FIFO_MESSAGE_HEADER;


//--------------------------------------------------------------------------
BOOL RigCtrl_AppendMessageToRadioPortTxFifo( T_RigCtrlInstance *pRC,
        int iRigCtrlClientPort, // [in] RIGCTRL_PORT_AUX_COM_1 / 2 ?
        BYTE *pbMessage, int iMsgLength, // [in] message including pre- and postamble
        int iMsgType, char *pszComment ) // [in] info for e.g. the TRAFFIC LOG
  // Called from RigCtrl_ParseCIV() after receiving certain messages
  // from any CLIENT that need to be forwarded to the RADIO PORT,
  //    but cannot be sent immediately because the RADIO (or its port) is already busy
  //    from a previously transmitted command (with response is still pending).
  //
{
  int nBytesOccupied, nBytesFree, iHeadIndex, iTailIndex;
  BYTE *pbBuffer = pRC->bRadioPortTxFifo;
  T_RIGCTL_FIFO_MESSAGE_HEADER hdr;
  BOOL fResult = FALSE;

  if( iMsgLength <= 0 )  // don't trash the TX-FIFO with negative lengths
   { return FALSE;
   }

  // Because ports for REMOTE CLIENTS and THE LOCALLY CONNECTED RADIO
  // are serviced in DIFFERENT THREADS, use a short critical section here:

  RigCtrl_EnterCriticalSection( pRC ); // NO LAZY RETURN after this !

  iHeadIndex = pRC->iRadioPortTxFifoHead;
  iTailIndex = pRC->iRadioPortTxFifoTail;
  pbBuffer   = pRC->bRadioPortTxFifo;
  nBytesOccupied = iHeadIndex - iTailIndex;
  if( nBytesOccupied < 0 ) // circular wrap, e.g. head=0, tail=2047 ->
   {  nBytesOccupied += RIGCTRL_RADIO_PORT_TX_FIFO_SIZE;
   }
  // Remember, a classic circular 2048-byte-FIFO has a maximum USEABLE capacity
  //   of 2047(!) bytes, because HeadIndex==TailIndex means buffer EMPTY (not FULL).
  //   The buffer is completely full when (HeadIndex+1)==TailIndex. Thus:
  nBytesFree = RIGCTRL_RADIO_PORT_TX_FIFO_SIZE - 1 - nBytesOccupied;

  // sizeof( T_RIGCTL_FIFO_MESSAGE_HEADER ) extra bytes (before the message itself)
  // are used to store the MESSAGE LENGTH, MESSGE TYPE, CLIENT PORT NUMBER,
  // and maybe a few other parameters ("info") for the message log later.
  // This way, we can store anything in the FIFO without knowing the structure.
  // It doesn't necessarily have to be CI-V !
  if( nBytesFree >= (iMsgLength+sizeof(hdr)) ) // sufficient FIFO-space for header AND message payload ?
   { fResult = TRUE;

     // Fill out the MESSAGE HEADER and enter it in the FIFO (the HEADER will actually not be sent at all):
     memset( &hdr, sizeof(hdr), 0 );
     hdr.iMsgLength = iMsgLength;
     hdr.iMsgType   = iMsgType,
     hdr.iRigCtrlPort = iRigCtrlClientPort;
     if( pszComment != NULL )
      { SL_strncpy( hdr.sz80Comment, pszComment, 80 );
      }
     RigCtrl_AppendBlockToRadioPortTxFifo( pbBuffer, &iHeadIndex, (BYTE*)&hdr, sizeof(hdr) );
     RigCtrl_AppendBlockToRadioPortTxFifo( pbBuffer, &iHeadIndex, pbMessage, iMsgLength );
     pRC->iRadioPortTxFifoHead = iHeadIndex; // another fragment is now "occupied"
   }

  RigCtrl_LeaveCriticalSection( pRC );

  return fResult;

} // end RigCtrl_AppendMessageToRadioPortTxFifo()

//--------------------------------------------------------------------------
BOOL RigCtrl_SendMessageFromRadioPortTxFifo( T_RigCtrlInstance *pRC,
        T_RigCtrl_MsgFilter *pFilter) // [out] command, subcommand, and possibly accepted message type
  // Transmits the next message (if any) that has previously been
  //           queued up in the *RADIO PORT's* transmit FIFO, TO the radio
  //   - see details in RigCtrl_AppendMessageToRadioPortTxFifo() .
  // To register the expected RESPONSE (from radio, forwarded to the client),
  //    RigCtrl_SendMessageFromRadioPortTxFifo() also prepares a 'filter'
  //    which the caller will pass on to RigCtrl_RegisterCIVReponseForClient(),
  //    just as if the message HAD BEEN immediately forwarded
  //    from client to radio in RigCtrl_ParseCIV() or similar.
  //
  // About the TRAFFIC LOG: See RigCtrl_SendAndLogMessage() [called from here].
  //
{
  int nBytesOccupied, iMsgLength, iClientPort, i;
  int iHeadIndex = pRC->iRadioPortTxFifoHead;
  int iTailIndex = pRC->iRadioPortTxFifoTail;
  BYTE *pbBuffer = pRC->bRadioPortTxFifo;
  T_RigCtrl_PortInstance *pPort = &pRC->PortInstance[ RIGCTRL_PORT_RADIO ];
  BYTE *pbMessage = pPort->sTxMsg.bData; // 'linear' buffer for transmission
  T_RigCtrl_MsgFilter msg_filter;
  DWORD dwCombinedCmdAndSubcode;

  nBytesOccupied = iHeadIndex - iTailIndex;
  if( nBytesOccupied < 0 ) // circular wrap, e.g. head=0, tail=2047 ->
   {  nBytesOccupied += RIGCTRL_RADIO_PORT_TX_FIFO_SIZE;
   }
  if( nBytesOccupied >= 3 ) // need AT LEAST three bytes : 16 bit length + 8 bit client port number
   { // Read the MESSAGE LENGTH and CLIENT PORT in the FIFO (again, low byte first)
     iMsgLength  = RigCtrl_ReadByteFromRadioPortTxFifo( pbBuffer, &iTailIndex );
     iMsgLength |= ((int)RigCtrl_ReadByteFromRadioPortTxFifo( pbBuffer, &iHeadIndex) << 8);
     iClientPort = RigCtrl_ReadByteFromRadioPortTxFifo( pbBuffer, &iTailIndex );
     if( nBytesOccupied >= (3+iMsgLength) )  // still looks good .. queued-up message is COMPLETE
      {
        if( iMsgLength <= RIGCTRL_MAX_MESSAGE_LENGTH ) // must fit inside the "non-circular" TX buffer..
         { for(i=0; i<iMsgLength; ++i)
            { pbMessage[i] = RigCtrl_ReadByteFromRadioPortTxFifo( pbBuffer, &iTailIndex );
            }
           pPort->sTxMsg.iLength = iMsgLength; // here: set in RigCtrl_SendMessageFromRadioPortTxFifo(),
                                               //
         }
        pRC->iRadioPortTxFifoTail = // release transmitted message from the circular FIFO
          ( pRC->iRadioPortTxFifoTail + iMsgLength + 3 ) & (RIGCTRL_RADIO_PORT_TX_FIFO_SIZE-1);

        // The following is similar to the code after
        //      'case RIGCTRL_CLIENT_ACTION_RETRANSMIT' in RigCtrl_ParseCIV(),
        //      incl. registering a "filter" for this message to forward its RESPONSE
        //      to the client (later, when a matching response arrived from the radio):
        msg_filter.iCmd = pbMessage[4];
        // any further bytes in any CI-V message are IGNORED for filtering:
        msg_filter.iSubCmd = msg_filter.iDataB1 = msg_filter.iDataB2 = -1;
        // Also let the filter (for the client) pass reponse "OK" and "NOT OK":
        msg_filter.iMsgTypes = RIGCTRL_MSGTYPE_OK | RIGCTRL_MSGTYPE_NOT_OK;
        dwCombinedCmdAndSubcode = RigCtrl_GetCombinedCmdAndSubcodeForCIV( pbMessage, iMsgLength, iMsgType );
        if( pRC->pRadioPortTxFifo != NULL )
         { RigCtrl_RegisterCIVReponseForClient( pRC, iClientPort, &msg_filter );
           // Mark the "radio port" as occupied, until the response
           //      for the forwarded command (sent below) arrives :
           pPortInstance->dwExpectedResponse_CmdAndSubcode = dwCombinedCmdAndSubcode;
           // ,---------------------'
           // '--> This is mainly important for the CI-V traffic monitor,
           //      to show e.g. > RX 006 FE FE 00 94 FD ; "OK" (for cmd 0x05)
           //      because Icom's response codes 0xFB ("Ok") and 0xFA ("Not Ok")
           //      don't carry any information about WHAT was "Ok" or "Not Ok".
           //      Thus we store this information in yet another state variable.
           //  But do we EXPECT a response from the radio at all ?
           //  How do we know it the CI-V message from the TX FIFO is a
           //      "set" or "read" command, and if it's a "set"-command,
           //      will the radio ALWAYS reply with "Ok" or "Not Ok" ?
           //  Note the confusing multitude of commands to SET THE FREQUENCY:
           //  Some of them (e.g. cmd 0x05) WILL be responded with 0xFB or 0xFA,
           //       others (maybe cmd 0x00) will NOT be confirmed / acknowledged
           //       by the radio.
           return RigCtrl_SendAndLogMessage( pRC, RIGCTRL_PORT_RADIO, RIGCTRL_ORIGIN_CONTROLLER,
                    pbMessage, iMsgLength,
                    RIGCTRL_MSGTYPE_OTHER, "fwd from FIFO" );
         } // end if < pRadioPortTxFifo != NULL > ?
      } // end if( nBytesOccupied >= (3+iMsgLength) )
   }
  // Arrived here ? The TX-FIFO is either completely empty, or the next message
  //        has not been completely assembled by the "writer" in another thread,
  //        or the address of the "transmit function" for this port is invalid.
  return TRUE;

} // end RigCtrl_SendMessageFromRadioPortTxFifo()
#endif // < need RigCtrl_SendMessageFromRadioPortTxFifo() > ?



//--------------------------------------------------------------------------
BOOL RigCtrl_RegisterCIVReponseForClient( T_RigCtrlInstance *pRC,
        int iRigCtrlClientPort,       // [in] RIGCTRL_PORT_AUX_COM_1 / 2 ?
        T_RigCtrl_MsgFilter *pFilter) // [in] command, subcommand, and possibly accepted message type
  // Called from RigCtrl_ParseCIV() when certain messages (commands)
  //        arrive on one of the *CLIENT PORTS* (iRigCtrlClientPort).
  // Because the client "thinks" he's directly talking to the radio,
  //         anything that the radio sends IN RESPONSE must also be
  //         forwarded (not echoed) to the client's port (iRigCtrlClientPort).
  // For CI-V, the response may be an unspecific "Ok" or "Error".
  //         To allow forwarding such responses to the client,
  //         the caller may also specify a CATEGORY of message types, for example
  //         RIGCTRL_MSGTYPE_OK | RIGCTRL_MSGTYPE_NOT_OK (bitwise combineable).
{
  T_RigCtrl_PortInstance *pPortInstance;
  T_RigCtrl_MsgFilter    *pMsgFilter;

  int i;
  // Example: Call after receiving "Power On"-command, 1st frame from "RS-BA1":
  //    pbMessage[0  1  2  3   4   5   6  7 ...
  //         Len Pre.  Addr   cmd sub  d1 d2
  // >    r1 007 FE FE 94 E0   18  01        client to server) -> pFilter->iCmd = 0x18
  // >    TX 007 FE FE 94 E0   18  01        server lab to IC-7300
  // >    RX 007 FE FE 94 E0   18  01        echo directly from IC-7300
  // >    RX 006 FE FE E0 94   FB            IC-7300 to server: "OK", must pass the new filter because it's the "expected response"
  // >    t1 006 FE FE E0 94   FB            response relayed from server to client

  if( (iRigCtrlClientPort<0) || (iRigCtrlClientPort>/*!*/RIGCTRL_MAX_CLIENT_PORTS) )
   { return FALSE;  // not a valid 'port identifier'
   }

  pPortInstance = &pRC->PortInstance[iRigCtrlClientPort];

  // First check if this message (command, subcommand, sub-sub-command) is already registered:
  for( i=0; i<RIGCTRL_MAX_REGISTERED_MESSAGES_PER_CLIENT; ++i )
   { pMsgFilter = &pPortInstance->RegisteredMessages[i];
     if( ( pMsgFilter->iCmd    == pFilter->iCmd )
      && ( pMsgFilter->iSubCmd == pFilter->iSubCmd )
      && ( pMsgFilter->iDataB1 == pFilter->iDataB1 )
      && ( pMsgFilter->iDataB2 == pFilter->iDataB2 ) )
      {
        pMsgFilter->iMsgTypes |= pFilter->iMsgTypes; // e.g. "OK" + "NOT OK"
        return TRUE;
      }
   }

  // Not registered yet, so use the FIRST unused entry :
  for( i=0; i<RIGCTRL_MAX_REGISTERED_MESSAGES_PER_CLIENT; ++i )
   { pMsgFilter = &pPortInstance->RegisteredMessages[i];
     if( (pMsgFilter->iCmd <= 0 ) && (pMsgFilter->iMsgTypes <= 0 ) )
      { *pMsgFilter = *pFilter;
        return TRUE;
      }
   }

  // Running out of space to "register" another message for this client..
  return FALSE;


} // end RigCtrl_RegisterCIVReponseForClient()


//--------------------------------------------------------------------------
void RigCtrl_CheckAndForwardResponseToClients( T_RigCtrlInstance *pRC,
        BYTE *pbMessage, int iMsgLength, // [in] message including pre- and postamble
        T_RigCtrl_MsgFilter *pFilter,    // [in] command, subcommand, and possibly accepted message type
        int iReceivedMessageType,        // [in] class of the received message for the log
        BOOL fIsUnsolicitedMsg,          // [in] TRUE for unsolicited messages, e.g. VFO frequency + Waveform data
        BOOL fIsExpectedResponse,        // [in] RigCtrl_ParseCIV() already tried to(!) find out IF the message is the "expected" response
        char * pszComment )              // [in] optional comment for the log
  // Called from RigCtrl_ParseCIV() after receiving a RESPONSE (not a COMMAND)
  // from pRC->PortInstance[RIGCTRL_PORT_RADIO] .
  //
  // About forwarding messages received from a remote client on a SERVER PORT
  //   to the 'real radio': See RigCtrl_ForwardCommandFromServerPortToRadio(),
  //   and follow the 'thread' (with MULTIPLE SERIAL PORT THREADS involved)
  //              with RigCtrl_iConditionalBreakpointStep 5..9 .
{
  int iRigCtrlClientPort, iFilter, iServerPortInstance;
  int iRigCtrlOrigin = RIGCTRL_ORIGIN_CONTROLLER; // <- may have to modify that one day
  T_RigCtrl_PortInstance *pRadioPortInstance = &pRC->PortInstance[RIGCTRL_PORT_RADIO];
  T_RigCtrl_PortInstance *pServerPortInstance;
  T_RigCtrl_MsgFilter *pMsgFilter;
  DWORD dwCombinedCmdAndSubcode, dwCombinedCmdAndSubcode2;
  BOOL fMatch;


  if( pRadioPortInstance->iWorkingForServerPort > 0 ) // Radio port "has been working" for a particular SERVER PORT ?
   { // (this is the simpler implementation, without all the "filtering" from the
     //  older code / client-port-loop further below)
     iServerPortInstance = pRadioPortInstance->iWorkingForServerPort;
     pServerPortInstance = RigCtrl_IndexToPortInstancePtr( pRC, iServerPortInstance );
     // Only if the SERVER PORT is still waiting for a RESPONSE (for the COMMAND
     //      that has been forwarded from server port to radio port earlier,
     //      in RigCtrl_ForwardCommandFromServerPortToRadio() ) :
     if( pServerPortInstance->iServerState == RIGCTRL_SSTATE_FWD2RADIO_BUSY )
      { // From RigControl.h :
        // > Immediately before switching from RIGCTRL_SSTATE_FWD2RADIO_BUSY to RIGCTRL_SSTATE_FWD2RADIO_DONE,
        // > pServerPortInstance->sRxMsg[1] stores the RESPONSE(!) from the REAL RADIO.
        // That happens HERE ... maybe, if pbMessage[] is the EXPECTED RESPONSE:
        dwCombinedCmdAndSubcode  = RigCtrl_GetCombinedCmdAndSubcodeForCIV( pbMessage, iMsgLength, iReceivedMessageType );
        dwCombinedCmdAndSubcode2 = RigCtrl_GetCombinedCmdAndSubcodeForCIV(
                     pServerPortInstance->sRxMsg[1].bData,
                     pServerPortInstance->sRxMsg[1].iLength,
                     pServerPortInstance->sRxMsg[1].iType );
        // Examples:
        //  (a) Remote CLIENT (WSJT-X) asked for the "Selected VFO Frequency";
        //     >   r2 007 FE FE 94 E0 25 00 FD   (traffic log entry for the SERVER PORT) -> dwCombinedCmdAndSubcode2 = 0x25000000
        //     >   TX 007 FE FE 94 E0 25 00 FD   (message forwarded to the RADIO PORT)
        //    ,-----------------------|___|
        //    '--> COMMAND (0x25) and SUB-COMMAND (0x00) in the COMMAND,
        //         which only RigCtrl_ParseCIV() could identify as READ COMMAND,
        //         thus the necessity of storing that info as part of the 'message type' !
        //     >   RX 00C FE FE E0 94 25 00 30 73 03 07 00 FD  (READ RESPONSE on the RADIO PORT,
        //         ,------------------|___|  identified as iReceivedMessageType = 0x0402 (*)
        //         '--> dwCombinedCmdAndSubcode = 0x2500FFFF (ok after making RigCtrl_GetCombinedCmdAndSubcodeForCIV() aware of cmd 0x25)
        //       (in example a, fIsExpectedResponse was already SET
        //        because RigCtrl_ParseCIV() was aware of what happened on the RADIO PORT.
        //        (*) 0x402 = RIGCTRL_MSGTYPE_FLAG_EXPECTED | RIGCTRL_MSGTYPE_FREQUENCY_REPORT.
        //                    RigCtrl_ParseCIV() had not set  yet.
        //   (b) Same dwCombinedCmdAndSubcode2 = 0x2500FFFF as above (a),
        //        but dwCombinedCmdAndSubcode  = 0x2700FFFF from the follwing:
        //     >   RX 03C FE FE E0 94 27 00 00 08 11 02 00 02 ..
        //         |                  |___|---> "Read the Scope waveform data" (slang from A7508-3EX-4 page 12).
        //         '--> iReceivedMessageType = 0x0004 = RIGCTRL_MSGTYPE_SPECTRUM,
        //              fIsExpectedResponse  = FALSE (ok, wait for the next received message)...
        //   (c)
        //
#      if(SWI_HARDCORE_DEBUGGING)
        if( RigCtrl_iConditionalBreakpointStep == 7 ) // have forwarded a "verbatim copy" of a command like 0x07 = "SelectVFO",
         { //  or any other command that does NOT have a UNIQUE unified PN,
           //  including commands which could not even be identified as a
           //  READ- or WRITE-command, and thus have e.g.
           //  pServerPortInstance->sRxMsg[1].iType, bits 7..0 set to 'RIGCTRL_MSGTYPE_UNKNOWN' .
           // NOT YET: RigCtrl_iConditionalBreakpointStep = 8;
           // Now in RigCtrl_CheckAndForwardResponseToClients(), expecting the reception of
           // a response for the already sent 'verbatim copy' (pServerPortInstance->sRxMsg[1].bData) .
           // 2025-10-20 : Got here when firing up N1MM Logger+, with ...
           //    (first) pbMessage = 0xFE FE E0 94 27 00 00 06 .. (that was NOT the expected response but an UNSOLICITED REPORT, e.g. spectrum data)
           //            iMsgLength= 60 (!)
           //    pServerPortInstance->sParamInfoForPendingCommand.pszToken = "ScopeOnOff" (WRONG !)
           //    pServerPortInstance->sRxMsg[1].bData[0..7] = 0xFE FE 94 E0 27 10 01 FD (WRONG ! See below..),
           //    pServerPortInstance->sRxMsg[1].iLength     = 8 (WRONG ! Should have been 7),
           //
           //    (later)
           //    pServerPortInstance->sRxMsg[1].bData[0..6] = 0xFE FE 94 E0 07 00 FD,
           //    pServerPortInstance->sRxMsg[1].iLength     = 7,
           //    pServerPortInstance->sz255CommentFromParser = "SelectVFO:00",
           //    pServerPortInstance->iMsgTypeFromParser = 0x00030809 = pServerPortInstance->sRxMsg[1].iType .
           //    pServerPortInstance->dwExpectedResponse_CmdAndSubcode = 0xFBFFFFFF
           // EXPECTED something like this in the traffic log :
           //    > r2 007 FE FE 94 E0 07 00 FD ; SelectVFO:00    (original command received on the "Virtual Rig" port)
           //    > TX 007 FE FE 94 E0 07 00 FD ; SelectVFO:00    (command forwarded on the "Real Radio" port)
           //    > RX 006 FE FE E0 94 FB FD    ; SelectVFO : OK  (only at THIS point, we can be sure the original command was a WRITE-COMMAND,
           //    > t2 006 FE FE E0 94 FB FD    ; SelectVFO : OK  ("OK" (=Icom's anonymous 'Write-Response') echoed on the "Virtual Rig" port)
         }
#      endif // SWI_HARDCORE_DEBUGGING ?


        if( ! fIsExpectedResponse )  // take a closer look at the message .. if the COMMAND and (when present) SUB-COMMAND
         { // equals what has been stored in pServerPortInstance->sRxMsg[1] earlier, it's the EXPECTED RESPONSE:
           if( dwCombinedCmdAndSubcode == dwCombinedCmdAndSubcode2 )
            {  fIsExpectedResponse = TRUE;
            }
         }
        if( ! fIsExpectedResponse )  // take a closer look at the message .. maybe not exactly EXPECTED but THE END of a command-respose-transaction
         { // Will get here VERY OFTEN because the radio keeps shelling out e.g. SPECTRUM DATA just as if there was no tomorrow.
           // Because RigCtrl_ParseCIV() doesn't know all those 'special cases'
           // for certain commands and their responses,
           // take a second look at the RECEIVED MESSAGE, and compare it
           // to what has been set in RigCtrl_CIV_Server_OnReadCommand()
           // or RigCtrl_CIV_Server_OnWriteCommand() ... details THERE :
           if( dwCombinedCmdAndSubcode == pServerPortInstance->dwExpectedResponse_CmdAndSubcode )
            { fIsExpectedResponse = TRUE;
            }
         }
        if( ! fIsExpectedResponse )  // take a SECOND look at the message .. maybe it's a "NOT-OK"-response !
         { if( RigCtrl_IsNotOkResponse( pRadioPortInstance->iRadioCtrlProtocol, pbMessage, iMsgLength ) )
            { fIsExpectedResponse = TRUE;  // always "expect" a NOT-OK response from an Icom transceiver
            }
         }

        if( fIsExpectedResponse )
         { memcpy( pServerPortInstance->sRxMsg[1].bData, pbMessage, iMsgLength ); // here: set in RigCtrl_CheckAndForwardResponseToClients(), when "working for a server port" but received a response from the REAL RADIO (!)
           pServerPortInstance->sRxMsg[1].iLength = iMsgLength;
           pServerPortInstance->sRxMsg[1].iType   = iReceivedMessageType;
           RigCtrl_SetServerState( pServerPortInstance, RIGCTRL_SSTATE_FWD2RADIO_DONE ); // here: set in RigCtrl_CheckAndForwardResponseToClients()
           //  ,---------------------------------------------'
           //  '--> The transition from RIGCTRL_SSTATE_FWD2RADIO_DONE to RIGCTRL_SSTATE_PASSIVE
           //       happens in __________(?) .
           pRadioPortInstance->iWorkingForServerPort = 0; // "done" (here: cleared in RigCtrl_CheckAndForwardResponseToClients() )
           pRadioPortInstance->iTrafficMonitorUnpauseCountdown = 0; // here: cleared in RigCtrl_CheckAndForwardResponseToClients(), when it's the "expected" response from the RADIO PORT
           pRadioPortInstance->dwExpectedResponse_CmdAndSubcode = RIGCTRL_NOVALUE_INT; // here: cleared in in RigCtrl_CheckAndForwardResponseToClients()
           //                     '--> actually, this was already cleared in RigCtrl_ParseCIV() .
           pRadioPortInstance->iExpectResponseForUnifiedPN = -1; // here: cleared in RigCtrl_CheckAndForwardResponseToClients()
           // Modify the CI-V "To" and "From"-address for the SERVER PORT ?
           switch( pServerPortInstance->iRadioCtrlProtocol )
            { case RIGCTRL_PROTOCOL_ICOM_CI_V :
                 pServerPortInstance->sRxMsg[1].bData[2] = pServerPortInstance->iRadioMasterAddr;
                 pServerPortInstance->sRxMsg[1].bData[3] = pServerPortInstance->iRadioDeviceAddr;
                 break;
              default:
                 break;
            } // end switch( pServerPortInstance->iRadioCtrlProtocol )

           // Don't waste time by letting some other thread poll for RIGCTRL_SSTATE_FWD2RADIO_DONE.
           // For example, AuxComPorts.c uses a THREAD-SAFE TX FIFO
           //   (in AuxCom_RigControlCallback(RIGCTRL_CBK_SEND_MSG) ) anyway, thus:
           RigCtrl_SendAndLogMessage( pRC, iServerPortInstance, iRigCtrlOrigin,
               pServerPortInstance->sRxMsg[1].bData,
               pServerPortInstance->sRxMsg[1].iLength,
               pServerPortInstance->sRxMsg[1].iType,
               pszComment );
         } // end if < really the EXPECTED RESPONSE (and not just e.g. an UNSOLICITED REPORT, SPECTRUM DATA, etc^10) ?
      }   // end if( pServerPortInstance->iServerState == RIGCTRL_SSTATE_FWD2RADIO_BUSY )
   }     // end if( pRadioPortInstance->iWorkingForServerPort > 0 )


  for( iRigCtrlClientPort = RIGCTRL_PORT_AUX_COM_1;
       iRigCtrlClientPort < (RIGCTRL_PORT_AUX_COM_1+RIGCTRL_MAX_CLIENT_PORTS);
       iRigCtrlClientPort++ )
   { pServerPortInstance = &pRC->PortInstance[iRigCtrlClientPort];

     // Does THIS client want 'unsolicited' messages (e.g. new VFO-frequecy, Scope data) ?
     // A well-written CI-V parser must ignore messages he doesn't understand,
     //    because CI-V has once been designed as a multi-master bus system.
     //    Not so with the mimosa known as "Hamlib" (used per default in WSJT-X).
     if( fIsUnsolicitedMsg && (pServerPortInstance->iClientOptions & RIGCTRL_CLIENT_OPTION_REJECT_UNSOLICITED_MSGS) )
      { // Don't forward this unsolitited message to the client.
        // This stupid Extrawurst was added to prevent WSJT-X to bail out
        // with the oh-so-familar "Ding-Dong ! Rig Control Error" :
        continue;  // continue with the next client . Do NOT forward to the mimosa .
      }

     // Does the received message match any filter registered by THIS client ?
     fMatch = FALSE;
     for( iFilter=0; iFilter<RIGCTRL_MAX_REGISTERED_MESSAGES_PER_CLIENT; ++iFilter )
      { pMsgFilter = &pServerPortInstance->RegisteredMessages[iFilter];
        if( ( ( pMsgFilter->iCmd    == pFilter->iCmd    ) || ( pMsgFilter->iCmd < 0) )
         && ( ( pMsgFilter->iSubCmd == pFilter->iSubCmd ) || ( pMsgFilter->iSubCmd<0) )
         && ( ( pMsgFilter->iDataB1 == pFilter->iDataB1 ) || ( pMsgFilter->iDataB1<0) )
         && ( ( pMsgFilter->iDataB2 == pFilter->iDataB2 ) || ( pMsgFilter->iDataB2<0) ) )
         { fMatch = TRUE;
           break;
         }
        if( pMsgFilter->iMsgTypes == pFilter->iMsgTypes )
         { fMatch = TRUE;
           break;
         }
        // If the message received on the "RADIO"-port was "OK" or "NOT OK",
        //    pMsgFilter->iCmd will not match (e.g. client was asking for 0x18 = "POWER ON"
        //     but radio answered with Cmd=0xFB = "ok"). To pass such "expected" messages,
        //     when expecting "OK" or "NOT OK", ANY of these will match:
        if(  ((pMsgFilter->iMsgTypes & (RIGCTRL_MSGTYPE_OK|RIGCTRL_MSGTYPE_NOT_OK)) != 0)
          && ((pFilter->iMsgTypes    & (RIGCTRL_MSGTYPE_OK|RIGCTRL_MSGTYPE_NOT_OK)) != 0) )
         { fMatch = TRUE;  // "OK" matches "NOT OK", and "NOT OK" also matches "OK" !
           break;
         }
      }

     if( fMatch )
      { RigCtrl_SendAndLogMessage( pRC, iRigCtrlClientPort, iRigCtrlOrigin,
           pbMessage, iMsgLength, iReceivedMessageType, pszComment );
      }
   } // end for < iRigCtrlClientPort >
} // end RigCtrl_CheckAndForwardResponseToClients()


//--------------------------------------------------------------------------
BOOL RigCtrl_AskForRadioID( T_RigCtrlInstance *pRC,
         int iRigCtrlPort ) // [in] e.g. RIGCTRL_PORT_RADIO
{
  BOOL fResult = FALSE;
  T_RigCtrl_PortInstance *pPortInstance = &pRC->PortInstance[iRigCtrlPort];

  switch( pPortInstance->iRadioCtrlProtocol )
   {  // which language does the radio *ON THIS PORT* speak ?
    case RIGCTRL_PROTOCOL_NONE :
    default:
         break;  // unknown protocol, can only send "direct" strings

    case RIGCTRL_PROTOCOL_ICOM_CI_V : // Icom's "CI-V" protocol ...
         fResult = RigCtrl_SendShortCIVMessage( pRC, iRigCtrlPort, RIGCTRL_ORIGIN_CONTROLLER,
                       RIGCTRL_MSGTYPE_RADIO_ID | RIGCTRL_MSGTYPE_FLAG_SUGGEST_NEW_HEADLINE_FOR_TRAFFIC_LOG, // iMsgType for the traffic log ...
                       "get radio ID",     // comment for the traffic log
                       0x19,  // iCmd: CI-V command "Read transceiver ID" (and a few others)
                       0x00,  // iSubCmd: "Read transceiver ID code"
                       -1, -1 );  // additional data bytes : None !
         // > Returns always the default CI-V adress, regardless of the actual
         // > setting of the CI-V adress. This function can be used by software
         // > to determine which radio is connected to the computer.
         // > Despite the functions name this command works also on receivers,
         // > not only transceivers.
         // If all works as planned, the transceiver also responds with cmd 0x1900,
         // and reports his own address. From this moment on,
         // the "To"-address will not be 0x00 = RIGCTRL_DEF_ADDR_AUTO_DETECT = Icom's "BROADCAST" address anymore
         // but, as in the traffic log shown below, the transceiver's real address:
         // Msg# time/ms dir len pre  to fm cmd .. postable (FD)  ; comment
         // -------------------------------------------------------------------
         // >  1 0000939 TX 007 FE FE 00 E0 19 00 FD             ; get radio ID
         // >  3 0000948 RX 008 FE FE E0 94 19 00 94 FD           ; Rig=IC-7300
         // >  5 0000973 TX 006 FE FE 94 E0 03 FD                ; read VFOfreq
         // >  7 0001037 RX 00B FE FE E0 94 03 87 89 09 50 00 FD  ; VFOfreq=...
         // -------------------------------------------------------------------
         pPortInstance->dwExpectedResponse_CmdAndSubcode = RCTL_COMBINE_CMD_AND_SUB(0x19, 0x00);  // only one known subcode, but.. !
         pPortInstance->iResponseCountdown_ms = pRC->iMaxResponseDelay_ms; // if the radio doesn't respond after this time, continue..
         break;  // end case <ICOM CI-V protocol>

#   if( SWI_SUPPORT_YAESU_5_BYTE_CAT )
     case RIGCTRL_PROTOCOL_YAESU_5_BYTE: // Yaesu's OLD UGLY CAT - see Yaesu5Byte.c ..
         pRC->iParameterPollingState = RIGCTRL_POLLSTATE_READ_RADIO_ID;
         Yaesu5Byte_SwitchPollingState( &pRC->Y5B, pRC->iParameterPollingState );  // (for non-Icom-radios, this tries to "identify the radio" by some other means)
         break; // end case <Yaesu binary 5-byte-command protocol>
#   endif // SWI_SUPPORT_YAESU_5_BYTE_CAT ?

   } // end switch( pRC->iRadioCtrlProtocol )


  return fResult;

} // end RigCtrl_AskForRadioID()

//--------------------------------------------------------------------------
BOOL RigCtrl_AskForVFOFrequency( T_RigCtrlInstance *pRC,
         int iRigCtrlPort ) // [in] e.g. RIGCTRL_PORT_RADIO

{
  BOOL fResult = FALSE;
  T_RigCtrl_PortInstance *pPortInstance = &pRC->PortInstance[iRigCtrlPort];
  

  switch( pPortInstance->iRadioCtrlProtocol )
   {  // which language does the radio *ON THIS PORT* speak ?
    case RIGCTRL_PROTOCOL_NONE :
    default:
         break;  // unknown protocol, can only send "direct" strings

    case RIGCTRL_PROTOCOL_ICOM_CI_V : // Icom's "CI-V" protocol ...
         fResult = RigCtrl_SendShortCIVMessage( pRC, RIGCTRL_PORT_RADIO,  RIGCTRL_ORIGIN_CONTROLLER,
                       RIGCTRL_MSGTYPE_FREQUENCY_REPORT, // iMsgType (for traffic log)
                       "read VFO freq", // short comment for the traffic log
                       0x03,  // iCmd: CI-V command "Read operating frequency"
                       -1, -1, -1 );  // iSubCmd and iDataBytes : Omit !
         // If all works as planned, the transceiver responds with 0x03 :
         pPortInstance->dwExpectedResponse_CmdAndSubcode = RCTL_COMBINE_CMD_ONLY(0x03); // CI-V response "Read operating frequency"
         // [IC7610CIV] doesn't explain the exact format of this (and many other)
         // 'simple' queries. But http://www.plicht.de/ekki/civ/civ-p41.html says:
         // > This command is understood by practically all Icom radios.
         // That's why we use THIS simple query, not one of the fancy new
         // multi-byte commands like 0x1A (with its awfully long list of sub-commands).
         pPortInstance->iResponseCountdown_ms = pRC->iMaxResponseDelay_ms; // if radio doesn't respond after this time, continue..
         break;  // end case <ICOM CI-V protocol>

#   if( SWI_SUPPORT_YAESU_5_BYTE_CAT )
     case RIGCTRL_PROTOCOL_YAESU_5_BYTE: // Yaesu's OLD UGLY CAT - see Yaesu5Byte.c ..
         break; // end case <Yaesu binary 5-byte-command protocol>
#   endif // SWI_SUPPORT_YAESU_5_BYTE_CAT ?


   } // end switch( pRC->iRadioCtrlProtocol )

  return fResult;

} // end RigCtrl_AskForVFOFrequency()

//--------------------------------------------------------------------------
BOOL RigCtrl_AskForOperatingMode( T_RigCtrlInstance *pRC,
         int iRigCtrlPort ) // [in] e.g. RIGCTRL_PORT_RADIO
{
  BOOL fResult = FALSE;
  T_RigCtrl_PortInstance *pPortInstance = &pRC->PortInstance[iRigCtrlPort];


  switch( pPortInstance->iRadioCtrlProtocol )
   {  // which language does the radio *ON THIS PORT* speak ?
    case RIGCTRL_PROTOCOL_NONE :
    default:
         break;  // unknown protocol, can only send "direct" strings

    case RIGCTRL_PROTOCOL_ICOM_CI_V : // Icom's "CI-V" protocol ...
         fResult = RigCtrl_SendShortCIVMessage( pRC, RIGCTRL_PORT_RADIO,  RIGCTRL_ORIGIN_CONTROLLER,
                     RIGCTRL_MSGTYPE_OP_MODE, // iMsgType (for traffic log)
                       "read op. mode", // short comment for traffic log
                       0x04,  // iCmd: CI-V command "Read operating mode"
                       -1, -1, -1 );  // iSubCmd and iDataBytes : Omit !
         // If all works as planned, the transceiver responds with 0x04, no sub-command-byte(s) :
         pPortInstance->dwExpectedResponse_CmdAndSubcode = RCTL_COMBINE_CMD_ONLY(0x04); // CI-V response with "operating mode" (command only, no subcode)
         pPortInstance->iResponseCountdown_ms = pRC->iMaxResponseDelay_ms;  // now waiting for the radio to REPORT ITS OPERATING MODE
         break;  // end case <ICOM CI-V protocol>

#   if( SWI_SUPPORT_YAESU_5_BYTE_CAT )
     case RIGCTRL_PROTOCOL_YAESU_5_BYTE: // Yaesu's OLD UGLY CAT - see Yaesu5Byte.c ..
         break; // end case <Yaesu binary 5-byte-command protocol>
#   endif // SWI_SUPPORT_YAESU_5_BYTE_CAT ?


   } // end switch( pRC->iRadioCtrlProtocol )

  return fResult;

} // end RigCtrl_AskForOperatingMode()


//--------------------------------------------------------------------------
void RigCtrl_AskForParameter( T_RigCtrlInstance *pRC, T_RigCtrl_ParamInfo *pPI,
         int iRigCtrlPort ) // [in] e.g. RIGCTRL_PORT_RADIO
  // ... without waiting for a response, as the name should imply ..
  //     Like any other response from the radio, the response will be analysed
  //     later, in RigCtrl_ProcessRxData() -> RigCtrl_ParseCIV()
  // [in] iUnifiedPN : e.g.
  //
  // About the TRAFFIC LOG: See RigCtrl_SendAndLogMessage() [called from here].
  //
{
  BOOL fResult = FALSE;
  T_RigCtrl_PortInstance *pPortInstance = &pRC->PortInstance[iRigCtrlPort];
  BYTE b16TxBuffer[16];
  BYTE *pbBufPtr = b16TxBuffer;
  int  i, iMsgLength;
  int  iMsgType = pPI->iMsgType;
  char sz80Comment[84];
  char *pszDest=sz80Comment, *pszEndstop=sz80Comment+80;

#ifdef __BORLANDC__  // "... is assigned a value that is never used".. oh shut up, we know what we're doing !
  (void)fResult;
#endif

  if( pPI != NULL )
   { if( pPI->pszToken != NULL )
      { SL_AppendPrintf( &pszDest, pszEndstop, "read %s", pPI->pszToken );
      }
     else
      { SL_AppendPrintf( &pszDest, pszEndstop, "read #%d", (int)pPI->iUnifiedPN );
      }
     switch( pPortInstance->iRadioCtrlProtocol )
      { // which language does the radio *ON THIS PORT* speak ?
        case RIGCTRL_PROTOCOL_NONE :
        default:
           break;  // unknown protocol, can only send "direct" strings

        case RIGCTRL_PROTOCOL_ICOM_CI_V : // Icom's "CI-V" protocol ...
           iMsgType |= RIGCTRL_MSGTYPE_FLAG_CIV;
           // If it's NOT one of those dreadful rig-dependent sub-sub-sub-command for e.g. cmd 0x1A sub 0x05,
           // the "parameter info" contains the READ COMMAND, and the LENGTH of the read command:
           if( (pPI->pbCIVReadCmd != NULL )  && (pPI->iCIVReadCmdLength>0) && (pPI->iCIVReadCmdLength<8) )
            { // Assemble a CI-V READ COMMAND based on the T_RigCtrl_ParamInfo :
              *pbBufPtr++ = 0xFE;  // fixed preamble code for CI-V protocol (term used by Icom .. NOT "Begin of Message" ! )
              *pbBufPtr++ = 0xFE;  // fixed preamble code for CI-V protocol
              *pbBufPtr++ = (BYTE)pPortInstance->iRadioDeviceAddr;  // "to" transceiver's address
              *pbBufPtr++ = (BYTE)pPortInstance->iRadioMasterAddr;  // "from" controller's address
              pPortInstance->dwExpectedResponse_CmdAndSubcode = 0; // prepare info for RigCtrl_ParseCIV(), if the response is "NotOK" (0xFA)
              for( i=0; i<pPI->iCIVReadCmdLength; ++i )
               { *pbBufPtr++ = pPI->pbCIVReadCmd[i];  // examples (bytes appended here):
                  // 0x03      = CI-V command "Read operating frequency",
                  // 0x04      = CI-V command "Read operating mode" (modulation),
                  // 0x0F      = CI-V command "Read Split Setting" / "DUP-" / "DUP+" / "RepeaterSimplex" (for IC-9700)
                  // 0x14 0x01 = Read AF level,
                  // 0x14 0x02 = Read RF gain, etc etc
                  pPortInstance->dwExpectedResponse_CmdAndSubcode |= pPI->pbCIVReadCmd[i] << (24-8*i);
                  // ,-----------------------------------'
                  // '--> The result of this bit-fiddling must be complatible
                  //      with RCTL_COMBINE_CMD_AND_SUB() and RCTL_COMBINE_CMD_ONLY().
                  //      The 8-bit "command" will be in bits 31..24,
                  //      the 8-bit "sub-command" in bits 23..16, etc.
                  //      Unused bytes (in the least significant bits) must be 0xFF
                  //      to make this compatible with RigCtrl_GetInfoForCIVCommandAndSubcode().
                  //      With dwExpectedResponse_CmdAndSubcode,
                  //      RigCtrl_ParseCIV() will indicate WHAT was "NotOK"
                  //      as in the example with an IC-9700 refusing to report
                  //      the FILTER BANDWIDTH (and responded with the anonymous "NotOK", 0xFA).
               }
              *pbBufPtr++ = 0xFD;  // CI-V code for "End Of Message" (term used by Icom .. NOT "postamble", even though 0xFE is called "preamble" ! )
              iMsgLength = pbBufPtr - b16TxBuffer;
              fResult = RigCtrl_SendAndLogMessage( pRC, RIGCTRL_PORT_RADIO, RIGCTRL_ORIGIN_CONTROLLER,
                            b16TxBuffer, iMsgLength, iMsgType, sz80Comment );
              // If all works as planned, the transceiver should respond shortly:
              if( fResult ) // successfully sent the READ COMMAND -> now waiting for a response:
               { pPortInstance->iExpectResponseForUnifiedPN = pPI->iUnifiedPN; // <- important for RigCtrl_ParseCIV()
                 pPortInstance->iResponseCountdown_ms = pRC->iMaxResponseDelay_ms; // now waiting for the radio to REPORT THE PARAMETER VALUE
                 // Note: Some rigs don't respond with the READ-OUT VALUE but with "NotOK".
                 // Because the "NotOK"-message does not contain any info about 'what was Not OK',
                 // RigCtrl_ParseCIV(case 0xFA) uses either pPortInstance->iExpectResponseForUnifiedPN
                 //                                      or pPortInstance->dwExpectedResponse_CmdAndSubcode
                 // to show WHAT EXACTLY went wrong in the 'Debug' log. Example from an IC-9700:
                 //
               }
            } // end if( pRC->pbCIVReadCmd != NULL )
           else // the READ COMMAND is so utterly complex,
            { // or the crazy command/subcommand/sub-sub-command depend on the RADIO MODEL (e.g. Icom's commands 0x1A 0x05 NNNN),
              // making it impossible to describe it in our "simple" hard-coded
              // parameter-info-table (array T_RigCtrl_ParamInfo RigCtrl_ParameterInfo[] further below in this file) .
              // For example, to read the current setting of the "Waterfall Speed",
              //  an IC-7300 expects command 0x1A subcommand 0x05 sub-sub-index "0108",
              //  an IC-9700 expects command 0x1A subcommand 0x05 sub-sub-index "0198",
              //  and you can be sure that Icom's NEXT radio will use yet another INCOMPATIBLE sub-sub-index.
              // Fortunately, for SOME of those goddamned "MODEL DEPENDENT special cases",
              // we already have RigCtrl_SendReadCommandForUnifiedPN(), so USE IT:
              if( pPI->iUnifiedPN )  // We have a NON-ICOM-SPECIFIC "unified parameter number", so try this:
               { fResult = RigCtrl_SendReadCommandForUnifiedPN( pRC, pPI->iUnifiedPN ); // <- aware of the dreaful command 0x1A sub 0x05 (and maybe others, too):
                 if( fResult ) // successfully sent the READ COMMAND -> now waiting for a response:
                  { pPortInstance->iExpectResponseForUnifiedPN = pPI->iUnifiedPN; // <- important for RigCtrl_ParseCIV()
                    pPortInstance->iResponseCountdown_ms = pRC->iMaxResponseDelay_ms; // now waiting for the radio to REPORT THE PARAMETER VALUE
                  }
               }
            }
           if( pPortInstance->iExpectResponseForUnifiedPN == RIGCTRL_PN_FILTER_BANDWIDTH ) // place for a 'conditional' breakpoint (2025-08-02):
            { pPortInstance->iExpectResponseForUnifiedPN = pPortInstance->iExpectResponseForUnifiedPN; // <-- set a breakpoint HERE !
            }
           break;  // end case <ICOM CI-V protocol>

#      if( SWI_SUPPORT_YAESU_5_BYTE_CAT )
        case RIGCTRL_PROTOCOL_YAESU_5_BYTE: // Yaesu's OLD UGLY CAT - see Yaesu5Byte.c ..
           break; // end case <Yaesu binary 5-byte-command protocol>
#      endif // SWI_SUPPORT_YAESU_5_BYTE_CAT ?


      } // end switch( pRC->iRadioCtrlProtocol )
   } // end if( pPI != NULL )

} // end RigCtrl_AskForParameter()


//--------------------------------------------------------------------------
BOOL RigCtrl_StartStopSpectrumData( T_RigCtrlInstance *pRC, BOOL fStart,
         int iRigCtrlPort ) // [in] e.g. RIGCTRL_PORT_RADIO
  // Tries to kick the periodic, unsolicited transmission of an IC-7300's
  //          (or IC-7610's, or maybe IC-R8600's) spectrum alive.
  // Icom decided to call this 'Scope waveform data' (command 0x27)
  // but this name is too ambiguous (since a "waveform" sounds like TIME DOMAIN),
  // in fact what we're trying to read here are the data which the LOCAL DISPLAY
  // (TFT screen inside the rig) uses for the spectrum graph + waterfall display.
  //      Actually, this feature was the motivation for the author
  //      to roll his own 'Rig Control' unit for Spectrum Lab,
  //      instead of using some 3rd party library like 'HamLib' or 'OmniRig' .
  // Called from the incredible state machine in RigCtrl_Handler(),
  //      somewhere in state RIGCTRL_POLLSTATE_CONFIGURE_SPECTRUM .
{
  BOOL fResult = FALSE;
  T_RigCtrl_PortInstance *pPortInstance = &pRC->PortInstance[iRigCtrlPort];

  switch( pPortInstance->iRadioCtrlProtocol )
   {  // which language does the radio *ON THIS PORT* speak ?
    case RIGCTRL_PROTOCOL_NONE :
    default:
         break;  // unknown protocol, can only send "direct" strings

    case RIGCTRL_PROTOCOL_ICOM_CI_V : // Icom's "CI-V" protocol ...
         fResult = RigCtrl_SendShortCIVMessage( pRC, RIGCTRL_PORT_RADIO, RIGCTRL_ORIGIN_CONTROLLER,
                     RIGCTRL_MSGTYPE_SPECTRUM_CONFIG, // iMsgType (for traffic log)
                     fStart ? "start spectrum" : "stop spectrum", // comment for traffic log
                       0x27,  // iCmd   : "Scope waveform data"
                       0x11,  // iSubCmd: "Send/read the Scope wave data output setting" [IC7300FM page 19-7]
                      fStart, // 1st data byte : 1 = ON,  0 = off
                       -1 );  // 2nd data byte : omit
         // If all works as planned, the transceiver agrees as shown below:
         // > 12:01:35.5 TX 008 FE FE 94 E0 27 11 01 FD           (scope)  ; start spectrum
         // > 12:01:35.5 RX 006 FE FE E0 94 FB FD                 (OK)
         // (other transceivers may simply start transmitting SPECTRUM DATA,
         //  don't rely on the "OK"-response for the above command)
         pPortInstance->dwExpectedResponse_CmdAndSubcode = RCTL_COMBINE_CMD_ONLY(0xFB); // at least an IC-7300 responded with "OK"
         pPortInstance->iResponseCountdown_ms = pRC->iMaxResponseDelay_ms; // now waiting for the radio to respond with "OK" or "NOT OK"...
         // A few milliseconds later, an IC-7300 began sending "Scope waveform data" periodically,
         //  message format shown in RigCtrl_ParseCIV(), case 0x27 .
         break;  // end case <ICOM CI-V protocol>

#   if( SWI_SUPPORT_YAESU_5_BYTE_CAT )
     case RIGCTRL_PROTOCOL_YAESU_5_BYTE: // Yaesu's OLD UGLY CAT - see Yaesu5Byte.c ..
         // SCOPE DISPLAY for Yaesu's old ugly CAT ? Never !
         break; // end case <Yaesu binary 5-byte-command protocol>
#   endif // SWI_SUPPORT_YAESU_5_BYTE_CAT ?


   } // end switch( pRC->iRadioCtrlProtocol )

  return fResult;
} // end RigCtrl_StartStopSpectrumData()


//--------------------------------------------------------------------------
BOOL RigCtrl_SetScopeSpeed( T_RigCtrlInstance *pRC, int iScopeSpeed, // Icom: "Scope Sweep Speed", cmd 0x27 0x1A
                            int iRigCtrlPort )
{ // (Note that there is also a "Waterfall Speed" in some radios !
  // Details on that ("Waterfall"-, not "Scope"-speed) in RigCtrl_SetWaterfallSpeed(). )
  BOOL fResult = FALSE;
  BYTE bData;
  char *pszComment;
  T_RigCtrl_PortInstance *pPortInstance = &pRC->PortInstance[iRigCtrlPort];

  switch( pPortInstance->iRadioCtrlProtocol )
   {  // which language does the radio *ON THIS PORT* speak ?
    case RIGCTRL_PROTOCOL_NONE :
    default:
         break;  // unknown protocol, can only send "direct" strings

    case RIGCTRL_PROTOCOL_ICOM_CI_V : // Icom's "CI-V" protocol ...
         switch( iScopeSpeed )
          { case RIGCTRL_SCOPE_SPEED_FAST:
            default:
               bData = 0x00;
               pszComment = "scope:FAST";
               break;
            case RIGCTRL_SCOPE_SPEED_MID:
               bData = 0x01;
               pszComment = "scope:MID";
               break;
            case RIGCTRL_SCOPE_SPEED_SLOW:
               bData = 0x02;
               pszComment = "scope:SLOW";
               break;
          }
         fResult = RigCtrl_SendShortCIVMessage( pRC, RIGCTRL_PORT_RADIO, RIGCTRL_ORIGIN_CONTROLLER,
                     RIGCTRL_MSGTYPE_SPECTRUM_CONFIG, // iMsgType (for traffic log)
                       pszComment, // comment for traffic log
                       0x27,  // iCmd   : "Scope waveform data"
                       0x1A,  // iSubCmd: "Send/read the Scope Sweep speed settings"
                       0x00,  // 1st data byte : 0x00 = MAIN scope, 0x01 = SUB scope
                     bData);  // 2nd data byte : "speed setting"
         pPortInstance->dwExpectedResponse_CmdAndSubcode = RCTL_COMBINE_CMD_ONLY(0xFB); // at least an IC-7300 responded with "OK"
         pPortInstance->iResponseCountdown_ms = pRC->iMaxResponseDelay_ms; // now waiting for the radio to respond with "OK" or "NOT OK" after "Set Scope Speed"
         break;  // end case <ICOM CI-V protocol>

#   if( SWI_SUPPORT_YAESU_5_BYTE_CAT )
     case RIGCTRL_PROTOCOL_YAESU_5_BYTE: // Yaesu's OLD UGLY CAT - see Yaesu5Byte.c ..
         // SCOPE DISPLAY for Yaesu's old ugly CAT ? Never !
         break; // end case <Yaesu binary 5-byte-command protocol>
#   endif // SWI_SUPPORT_YAESU_5_BYTE_CAT ?


   } // end switch( pRC->iRadioCtrlProtocol )

  return fResult;
} // end RigCtrl_SetScopeSpeed()

//--------------------------------------------------------------------------
BOOL RigCtrl_SetWaterfallSpeed( T_RigCtrlInstance *pRC, int iWaterfallSpeed, // Icom: "Waterfall Speed", cmd 0x1A 0x05 followed by some MODEL-DEPENDENT 4-digit-BCD "sub-subindex"
                                int iRigCtrlPort )
{
  //       FOR THE IC-9700, "Waterfall Speed" is cmd 0x1A 0x05 "0198".
  //       FOR THE IC-7300, "Waterfall Speed" is cmd 0x1A 0x05 "0108".
  //       FOR THE IC-705,  "Waterfall Speed" will be yet another command.
  //  AAARGH !
  // (Note that this doesn't seem to have anything in common with what Icom calls
  //             "Scope Sweep speed"; cmd 0x27 0x1A ! Even the INTERNAL VALUES
  //             are incompatible: For the "Scope Sweep speed", 0 = FAST.
  //                               For the "Waterfall Speed",   0 = SLOW. )
  BOOL fResult = FALSE;
  BYTE bBCD[2]; // 4-digit BCD in TWO BYTES (to generate the "crazy sub-sub-code" for certain CI-V commands)
  BYTE bData;
  int  iCrazyModelDependingSubSubcode = 0; // hold your breath and read on...
  char *pszComment = "WFspeed";
  T_RigCtrl_PortInstance *pPortInstance = &pRC->PortInstance[iRigCtrlPort];


  switch( pPortInstance->iRadioCtrlProtocol )
   {  // which language does the radio *ON THIS PORT* speak ?
    case RIGCTRL_PROTOCOL_NONE :
    default:
         break;  // unknown protocol, can only send "direct" strings

    case RIGCTRL_PROTOCOL_ICOM_CI_V : // Icom's "CI-V" protocol ...
        iCrazyModelDependingSubSubcode = RigCtrl_GetCrazyModelDependingSubSubcodeForCmd1A_05( // the name says it all...
             pRC->iDefaultAddress,        // [in] e.g. RIGCTRL_DEF_ADDR_IC_7300 (0x94)
             RIGCTRL_PN_WATERFALL_SPEED); // [in] iUnifiedPN, e.g. RIGCTRL_PN_CIV_TRANSCEIVE_MODE, RIGCTRL_PN_WATERFALL_SPEED, etc.
         switch( iWaterfallSpeed )
          { case RIGCTRL_SCOPE_SPEED_FAST: // MEEP-MEEP-MEEP ! "Waterfall Speed" uses a different encoding "on the CI-V bus" THAN RIG CONTROL !
            default:
               bData = 0x02;     // 0x02 is FAST for the WATERFALL, but would be SLOW for the "Scope" (whatever that means) !
               pszComment = "WFspeed:FAST";
               break;
            case RIGCTRL_SCOPE_SPEED_MID:
               bData = 0x01;
               pszComment = "WFspeed:MID";
               break;
            case RIGCTRL_SCOPE_SPEED_SLOW:
               bData = 0x00;
               pszComment = "WFspeed:SLOW";
               break;
          } // end switch( iWaterfallSpeed )

         RigCtrl_IntToBCD_CIV_MSByteFirst( iCrazyModelDependingSubSubcode, bBCD, 4/*nDigits*/ );
             // ,--------------'
             // '--> MOST SIGNIFICANT TWO DIGITS (in one byte) FIRST ! Example:
             //  iCrazyModelDependingSubSubcode = 198 -> bBCD[0] = 0x01,  bBCD[1] = 0x98  .

         fResult = RigCtrl_SendMidSizeCIVMessage( pRC, RIGCTRL_PORT_RADIO, RIGCTRL_ORIGIN_CONTROLLER,
             RIGCTRL_MSGTYPE_SPECTRUM_CONFIG, // message type for traffic log
                       pszComment, // comment for traffic log
                       0x1A,  // iCmd   : "The monster with HUNDREDS of sub- and sub-sub-commands"
                       0x05,  // iSubCmd: "Send/read the Scope Sweep speed settings"
                    bBCD[0],  // iDataByte1 : two upper digits of the 4-digit BCD "sub-sub-command"
                    bBCD[1],  // iDataByte2 : two lower digits of the 4-digit BCD "sub-sub-command"
                      bData,  // iDataByte3 : two-digit BCD DATA
                      -1 );   // iDataByte4 : none (in this special case)

         pPortInstance->dwExpectedResponse_CmdAndSubcode = RCTL_COMBINE_CMD_ONLY(0xFB); // guess for this WRITE-COMMAND, the radio only responds with "OK", not with the command+subcommand
         pPortInstance->iResponseCountdown_ms = pRC->iMaxResponseDelay_ms; // now waiting for the radio to respond with "OK" or "NOT OK" after "Set Scope Speed"
         break;  // end case <ICOM CI-V protocol>

#   if( SWI_SUPPORT_YAESU_5_BYTE_CAT )
     case RIGCTRL_PROTOCOL_YAESU_5_BYTE: // Yaesu's OLD UGLY CAT - see Yaesu5Byte.c ..
         // WATERFALL for Yaesu's old ugly CAT ? Never !
         break; // end case <Yaesu binary 5-byte-command protocol>
#   endif // SWI_SUPPORT_YAESU_5_BYTE_CAT ?


   } // end switch( pRC->iRadioCtrlProtocol )

  return fResult;
} // end RigCtrl_SetWaterfallSpeed()


//--------------------------------------------------------------------------
BOOL RigCtrl_EnableCIVTransceive( T_RigCtrlInstance *pRC, BOOL fTransceive,
                            int iRigCtrlPort )
  // Tries to enable/disable what in Icom's funny language is the "CI-V transceive"
  // option. Basically, when enabled, this causes the radio to send
  // UNSOLICITED REPORTS about changes in frequency ("VFO frequency")
  // and the 'operation mode' (aka modulation, USB, LSB, CW, FM) .
  // This can be configured in the radio's own setup menu, but poorly designed
  // software will DISABLE this wonderful feature without asking, because
  // such software (or their "library") was too dumb to correctly parse
  // anything except the EXPECTED (READ-)RESPONSE.
  // VERY UNFORTUNATELY, Icom uses rig-dependent commands to enable/disable
  // the "CI-V Transceive" option ! For the IC-7300 (in "A7292-4EX-11" page 163),
  // the command is called "Send/read the CI-V transceive setting (00=OFF, 01=ON)"
  //       FOR THE IC-7300, "CI-V transceive" is cmd 0x1A 0x05 "0071".
  //       FOR THE IC-7610, "CI-V transceive" is cmd 0x1A 0x05 "0112".
  //       FOR THE IC-7760, "CI-V transceive" is cmd 0x1A 0x05 "0150".
  //       FOR THE IC-7851, "CI-V transceive" is cmd 0x1A 0x05 "0155".
  //       FOR THE IC-9700, "CI-V transceive" is cmd 0x1A 0x05 "0127".
  //       FOR THE IC-705,  "CI-V transceive" is cmd 0x1A 0x05 "0131".
  //  AAARGH !  Some of this crappy "model-specific" commands (or sub-sub-commands)
  //  can be retrieved by RigCtrl_GetCrazyModelDependingSubSubcodeForCmd1A_05()
  //                                 '--> Nomen est omen !
{ int iCrazyModelDependingSubSubcodeForCmd1A_05
   = RigCtrl_GetCrazyModelDependingSubSubcodeForCmd1A_05( pRC->iDefaultAddress, RIGCTRL_PN_CIV_TRANSCEIVE_MODE );
  BYTE bBCD[2]; // 4-digit BCD in TWO BYTES (to generate the "crazy sub-sub-code" for certain CI-V commands)

  if( iCrazyModelDependingSubSubcodeForCmd1A_05 > 0 )
   {  RigCtrl_IntToBCD_CIV_MSByteFirst( iCrazyModelDependingSubSubcodeForCmd1A_05, bBCD, 4/*nDigits*/ );
      // ,--------------'
      // '--> MOST SIGNIFICANT TWO DIGITS (in one byte) FIRST ! Example:
      //  iCrazyModelDependingSubSubcode = 198 -> bBCD[0] = 0x01,  bBCD[1] = 0x98  .

   } // end if( iCrazyModelDependingSubSubcodeForCmd1A_05 > 0 )
  return FALSE;

} // end RigCtrl_EnableCIVTransceive()

//--------------------------------------------------------------------------
void RigCtrl_EnterState_Done( T_RigCtrlInstance *pRC )
  // Called from e.g. RigCtrl_Handler() .
  // Here, prints an info like the following into the 'Debug' tab:
  //  "RigControl: IC-9700 on COM5, 432250.0 kHz, CWN"
{
  char sz255Msg[256];
  char sz15Buf[16];
  T_RigCtrl_PortInstance *pRadioPortInstance = &pRC->PortInstance[RIGCTRL_PORT_RADIO];

  pRC->fGotBasicParams = TRUE; // "basic" parameters have been read from the radio, may now 'serve' external clients

  if( pRC->iParameterPollingState != RIGCTRL_POLLSTATE_DONE )
   {
     pRC->iParameterPollingState = RIGCTRL_POLLSTATE_DONE;
     //   '--> the GUI may switch from the "Debug" tab to the "TRX" tab now

    snprintf( sz255Msg, 255, "RigControl: Init done, %s on %s, %s, %s",
      RigCtrl_DefaultAddressToString(pRadioPortInstance->iRadioCtrlProtocol, pRC->iDefaultAddress),
      RigCtrl_GetRadioControlPortAsString(pRC), // -> e.g. "COM5", if that's the port "talking to the radio"
      RigCtrl_VfoFrequencyToString( pRC->dblVfoFrequency, sz15Buf ),
      RigCtrl_OperatingModeToString( pRC->iOpMode ) );
    // Emit a single line with 'info' in the error history, with a summary about the remote rig:
    ShowError( ERROR_CLASS_INFO | SHOW_ERROR_TIMESTAMP, sz255Msg );

    // Show the same also in the 'rig control traffic log' ?
    if( RigCtrl_TrafficMonitor.iDisplayOptions & RIGCTRL_TMON_DISPLAY_OPTION_ENABLE )
     { RigCtrl_AddToTrafficLog( pRC, RIGCTRL_PORT_RADIO, RIGCTRL_ORIGIN_RADIO, NULL/*pbMessage*/, 0/*iMsgLength*/,
                                RIGCTRL_MSGTYPE_OK, sz255Msg );
     }
    if( (RigCtrl_TrafficMonitor.iDisplayOptions & RIGCTRL_TMON_DISPLAY_OPTION_PAUSE_ON_INIT_DONE ) // here: for the RADIO CONTROL PORT,
     && (!pRC->PortInstance[RIGCTRL_PORT_RADIO].fTrafficMonitorPausedOnTrigger) )
     { // Before pausing the log, leave a note explaining WHY the traffic monitor is paused:
       sprintf( sz255Msg, "Traffic Log paused on %s after initialisation.",
          RigCtrl_GetDescriptivePortName(pRadioPortInstance) ); // <- delivers e.g. "Radio Control Port", "Virtual Radio Port #2", etc
       // (this shall resemble menu item 'Pause when initialisation done' )
       RigCtrl_AddToTrafficLog( pRC, RIGCTRL_PORT_RADIO, RIGCTRL_ORIGIN_RADIO, NULL/*pbMessage*/, 0/*iMsgLength*/, RIGCTRL_MSGTYPE_INFO, sz255Msg );
       RigCtrl_fUpdateTrafficLog = TRUE; // now "ready for being UPDATED on-screen"
       pRC->PortInstance[RIGCTRL_PORT_RADIO].fTrafficMonitorPausedOnTrigger = TRUE;
     } // end RIGCTRL_TMON_DISPLAY_OPTION_PAUSE_ON_INIT_DONE

   } // end if( pRC->iParameterPollingState != RIGCTRL_POLLSTATE_DONE )


} // end RigCtrl_EnterState_Done()


//--------------------------------------------------------------------------
void RigCtrl_FinishReadingBandStackingRegs( T_RigCtrlInstance *pRC )
  // Called from RigCtrl_Handler() or RigCtrl_ParseCIV() when all
  // 'band stacking registers' have been read from the radio,
  // or are not supported by the radio at all.
  // Details about the UGLY, RIG-SPECIFIC FORMAT in RigCtrl_ParseMemoryContent().
{
  // Notify the application about possibly 'new data' in pRC->BandStackingRegs[] :
  pRC->fBandStackingRegsModified = TRUE; // <- application may POLL and even CLEAR this flag
  pRC->fBandStackingRegsModifiedForCwNet = TRUE; // similar flag for CwNet.c,
          // to start sending band stacking registers from server to clients.

  pRC->iParameterPollingState = RIGCTRL_POLLSTATE_PARAMS_FOR_HAMLIB;
  pRC->iParameterPollingSubState = 0;
} // end RigCtrl_FinishReadingBandStackingRegs()

//--------------------------------------------------------------------------
static void RigCtrl_SetPostponedSetMessageFlag( T_RigCtrlInstance *pRC,
             int iRigctrlMsgType ) // [in] e.g. RIGCTRL_MSGTYPE_SET_FREQUENCY
  // Called e.g. from RigCtrl_SetVFOFrequency() if a 'SET'-message cannot be
  // sent immediately, and must be postponed instead.
  // DOES NOTHING if this Rig-Control-Instance is in Listen-Only Mode.
{ if( (! pRC->fListenOnlyMode ) && (iRigctrlMsgType <= 31) )
   { pRC->iPostponedSetMessages |= (1<<iRigctrlMsgType);
   }
} // end RigCtrl_SetPostponedSetMessageFlag()

//--------------------------------------------------------------------------
static void RigCtrl_ClearPostponedSetMessageFlag( T_RigCtrlInstance *pRC,
             int iRigctrlMsgType ) // [in] e.g. RIGCTRL_MSGTYPE_SET_FREQUENCY
  // Called from various places if a 'postponed SET message' can now be SENT.
{ if( iRigctrlMsgType <= 31 )
   { pRC->iPostponedSetMessages &= ~(1<<iRigctrlMsgType);
   }
} // end RigCtrl_ClearPostponedSetMessageFlag()


//--------------------------------------------------------------------------
BOOL RigCtrl_IsPortBusy( T_RigCtrl_PortInstance *pPortInstance )
{ if( pPortInstance->iResponseCountdown_ms <= 0 ) // not "Busy" anymore because the response-countdown-timer ran off
   { return FALSE;
   }
  if( pPortInstance->dwExpectedResponse_CmdAndSubcode >= 0 ) // BUSY because still waiting for a certain command/subcommand/sub-sub-command, etc
   { return TRUE;
   }
  if( pPortInstance->iExpectResponseForUnifiedPN > RIGCTRL_PN_UNKNOWN ) // BUSY because still waiting for a certain unified PARAMETER NUMBER
   { return TRUE;  // 2025-08-02 : Got HERE when the RADIO PORT was busy from RIGCTRL_PN_FILTER_BANDWIDTH,
     // while one of the SERVER PORTS (with WSJT-X as remote client) was trying to SWITCH THE VFO (A/B).
   }
  if( pPortInstance->iWorkingForServerPort > 0 ) // i.e. RIGCTRL_PORT_AUX_COM_1 in .PortInstance[RIGCTRL_PORT_RADIO] ...
   { return TRUE;
   }
  // Arrived here ? The specified Rig-Control PORT INSTANCE doesn't seem to be busy at the very moment
  return FALSE;
} // end RigCtrl_IsPortBusy()

#if( SWI_RIGCTRL_ACT_AS_SERVER )
//--------------------------------------------------------------------------
void RigCtrl_SetServerState( T_RigCtrl_PortInstance *pServerPortInstance,
        int iNewServerState) // [in] RIGCTRL_SSTATE_PASSIVE, RIGCTRL_SSTATE_PARSING_COMMAND,
          // RIGCTRL_SSTATE_WAIT_RADIO_NBUSY, RIGCTRL_SSTATE_FWD2RADIO_BUSY,
          // RIGCTRL_SSTATE_FWD2RADIO_DONE, RIGCTRL_SSTATE_RESPONSE_READY, ..(?)
{
  T_RigCtrlInstance *pRC = pServerPortInstance->pRC;
  BOOL fValidTransition = FALSE;
  char sz255Msg[256];
  if( pServerPortInstance->iServerState != iNewServerState )
   { pServerPortInstance->iServerState = pServerPortInstance->iServerState; // <- place for a breakpoint,
     // to find out if the SERVER STATE TRANSITIONS occurr in the
     // intended sequence (see SERVER STATE DIAGRAM in RigControl_VIC_Server.c),
     // and which of the threads caused it. Example (CALLERS of RigCtrl_SetServerState() causing a TRANSITION):
     //
     // ---------------  VALID transitions ---------------------------------
     // (0->1) RIGCTRL_SSTATE_PASSIVE  -> RIGCTRL_SSTATE_PARSING_COMMAND :
     //   AuxComThread() -> .. -> RigCtrl_CIV_Server_PrepareParsing()
     //
     // (1->5) RIGCTRL_SSTATE_PARSING_COMMAND -> RIGCTRL_SSTATE_RESPONSE_READY:
     //   AuxComThread() -> .. -> RigCtrl_CIV_Server_OnReadCommand()
     //
     // (1->2) RIGCTRL_SSTATE_PARSING_COMMAND -> RIGCTRL_SSTATE_WAIT_RADIO_NBUSY:
     //
     // (2->3) RIGCTRL_SSTATE_WAIT_RADIO_NBUSY  -> RIGCTRL_SSTATE_FWD2RADIO_BUSY:
     //
     // (3->4) RIGCTRL_SSTATE_FWD2RADIO_BUSY -> RIGCTRL_SSTATE_FWD2RADIO_DONE:
     //
     // (4->5) RIGCTRL_SSTATE_FWD2RADIO_DONE  -> RIGCTRL_SSTATE_PASSIVE:
     //        (almost the same as from RIGCTRL_SSTATE_RESPONSE_READY to ..PASSIVE)
     //
     // (5->0) RIGCTRL_SSTATE_RESPONSE_READY  -> RIGCTRL_SSTATE_PASSIVE:
     //         AuxComThread() -> .. -> RigCtrl_CIV_Server_OnReadCommand()
     //
     //
     // (see INVALID state transitions, and when/why they occurred, further below...)
     switch( pServerPortInstance->iServerState )
      { case RIGCTRL_SSTATE_PASSIVE         :
           fValidTransition = ( iNewServerState == RIGCTRL_SSTATE_PARSING_COMMAND);
           break;
        case RIGCTRL_SSTATE_PARSING_COMMAND :
           fValidTransition = ( iNewServerState == RIGCTRL_SSTATE_RESPONSE_READY)
                           || ( iNewServerState == RIGCTRL_SSTATE_WAIT_RADIO_NBUSY);
           break;
        case RIGCTRL_SSTATE_WAIT_RADIO_NBUSY  :
           fValidTransition = ( iNewServerState == RIGCTRL_SSTATE_FWD2RADIO_BUSY);
           break;
        case RIGCTRL_SSTATE_FWD2RADIO_BUSY :
           fValidTransition = ( iNewServerState == RIGCTRL_SSTATE_FWD2RADIO_DONE);
           break;
        case RIGCTRL_SSTATE_FWD2RADIO_DONE      :
           fValidTransition = ( iNewServerState == RIGCTRL_SSTATE_PASSIVE);
           break;
        case RIGCTRL_SSTATE_RESPONSE_READY  :
           fValidTransition = ( iNewServerState == RIGCTRL_SSTATE_PASSIVE);
           break;
        default:
           break;
      }
     if( ! fValidTransition )
      { if( pServerPortInstance->nServerStateErrors < 32767 )
         { ++pServerPortInstance->nServerStateErrors;
         }
        if( pServerPortInstance->nServerStateErrors < 15 )
         { sprintf( sz255Msg, "Illegal transition from server state %s to %s",
              RigCtrl_ServerStateToString(pServerPortInstance->iServerState),
              RigCtrl_ServerStateToString(iNewServerState) );
           RigCtrl_AddToTrafficLog( pRC,
              RigCtrl_PortInstancePtrToIndex(pServerPortInstance)/*->iRigCtrlPort*/,
              RIGCTRL_ORIGIN_RADIO, NULL/*pbMessage*/, 0/*iMsgLength*/,
              RIGCTRL_MSGTYPE_FLAG_ERROR, sz255Msg );
         }
      } // end if( ! fValidTransition )
     // ---------------  INVALID transitions seen so far: --------------------
     // (5->1) RIGCTRL_SSTATE_RESPONSE_READY  -> RIGCTRL_SSTATE_PARSING_COMMAND:
     //   AuxComThread() -> .. -> RigCtrl_CIV_Server_PrepareParsing()  .
     //   This happened several times when the remote client
     //   didn't wait for our response before sending the NEXT command .



     pServerPortInstance->iServerState = iNewServerState;
   } // end if( pServerPortInstance->iServerState != iNewServerState )
} // end RigCtrl_SetServerState()

//--------------------------------------------------------------------------
void RigCtrl_ForwardCommandFromServerPortToRadio( T_RigCtrlInstance *pRC,
                        T_RigCtrl_PortInstance *pServerPortInstance )
  // Contains the transition from RIGCTRL_SSTATE_WAIT_RADIO_NBUSY to RIGCTRL_SSTATE_FWD2RADIO_BUSY .
  // See the 'big picture' (state diagram) in RigControl_CIV_Server.c !
  // Called from RigCtrl_Handler(?) when the RADIO PORT is NOT BUSY at the moment,
  //        and a SERVER PORT needs to forward a command from an external client
  //        to the 'real radio' .
  // [in] pServerPortInstance->sParamInfoForPendingCommand. ...
  //        |-- iUnifiedPN :
  //        |    Set in RigCtrl_CIV_Server_OnReadCommand() / ..OnWriteCommand()
  //        |    If iUnifiedPN is RIGCTRL_PN_UNKNOWN, other members in
  //        |    pServerPortInstance->sParamInfoForPendingCommand are invalid, too.
  // [in] pServerPortInstance->sRxMsg[1] : used if there is no valid UnifiedPN.
  //            Contains the original command-message, set in
  //            e.g. in RigCtrl_CIV_Server_OnReadCommand()
  //                 or RigCtrl_CIV_Server_OnWriteCommand().
  //            See 'specification' of the role played by sRxMsg[1] in RigControl.h .
  //
  // [in] pServerPortInstance->iMsgTypeFromParser : indicates if the
  //            to-be-forwarded command is a READ- or WRITE-command, which makes
  //            a big difference (in CI-V) for the expected RESPONSE .
  // [in,out] pServerPortInstance->iServerState : RIGCTRL_SSTATE_WAIT_RADIO_NBUSY -> RIGCTRL_SSTATE_FWD2RADIO_BUSY (if all works well)
  //          or -> RIGCTRL_SSTATE_FWD2RADIO_DONE (early abort; with a FAKE ERROR RESPONSE already in
  //
{
  T_RigCtrl_PortInstance *pRadioPortInstance = &pRC->PortInstance[RIGCTRL_PORT_RADIO];
  BOOL fOk = FALSE;
  int iUnifiedPN = pServerPortInstance->sParamInfoForPendingCommand.iUnifiedPN;

  RigCtrl_EnterCriticalSection( pRC ); // here: called from RigCtrl_ForwardCommandFromServerPortToRadio(), which runs OUTSIDE THE RADIO-PORT'S WORKER THREAD

  pRadioPortInstance->iWorkingForServerPort = RigCtrl_PortInstancePtrToIndex(pServerPortInstance);

  if( pServerPortInstance->fTrafficMonitorEnabled && (!pServerPortInstance->fTrafficMonitorPausedOnTrigger) )
   { pRadioPortInstance->iTrafficMonitorUnpauseCountdown = 5; // here: set before forwarding a command from SERVER- to RADIO PORT.
     // Regardless of pRadioPortInstance->fTrafficMonitorEnabled
     //          and  pRadioPortInstance->fTrafficMonitorPausedOnTrigger,
     // allow showing the COMMAND forwarded on the *RADIO PORT*,
     //           and the RESPONSE received on the *RADIO PORT*.
     // pRadioPortInstance->iTrafficMonitorUnpauseCountdown will be CLEARED
     // when the EXPECTED response arrives, or DECREMENTED (down to zero)
     // when any message is received or sent on the same pRadioPortInstance.
     // That way, even if traffic monitor has been stopped 'long ago'
     // for the RADIO PORT, we will see the entire transaction (with SERVER- and the RADIO PORT).
   } // end if < pServerPortInstance->fTrafficMonitorEnabled && .. >

  if( iUnifiedPN == RIGCTRL_PN_ATU_ENABLED) // "conditional breakpoint" [in the line below]
   {  iUnifiedPN = iUnifiedPN;  // 2025-08-19: Got here after WFView asked for the "ATU_enabled" flag,
      // with pServerPortInstance->iMsgTypeFromParser = 0x0001801F = RIGCTRL_MSGTYPE_FLAG_READ_CMD(1<<16) | RIGCTRL_MSGTYPE_OTHER(31),
      //      pServerPortInstance->dwExpectedResponse_CmdAndSubcode= 0x1C01FFFF = cmd 0x15, sub 0x07, no four-digit "sub-sub-index"
      //      pServerPortInstance->sRxMsg[1].iType == pServerPortInstance->iMsgTypeFromParser (ok) .
   }
  if( iUnifiedPN != RIGCTRL_PN_UNKNOWN )
   { // preferred method (works even if Additional COM Port and radio use DIFFERENT PROTOCOLS)
     switch( pServerPortInstance->sRxMsg[1].iType & RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP )
      { case RIGCTRL_MSGTYPE_FLAG_WRITE_CMD :
             fOk = RigCtrl_QueueUpCmdToWriteUnifiedPN( pRC, iUnifiedPN ); // here: on behalf of an EXTERNAL CLIENT !
             break;
        case RIGCTRL_MSGTYPE_FLAG_READ_CMD :
             fOk = RigCtrl_QueueUpCmdToReadUnifiedPN( pRC, iUnifiedPN );  // here: on behalf of an EXTERNAL CLIENT !
             break;
        default: // something's wrong.. got a valid UNIFIED PN but don't know whether to forward a WRITE- or READ-command.
             break;
      }
   } // end if( iUnifiedPN != RIGCTRL_PN_UNKNOWN )
  if( !fOk ) // not just an invalid parameter, but an unknown READ-command,
   {         // or "something that doesn't have a unique UNIFIED PN, like cmd 0x07 = "Select VFO" .
     // 2025-07-27 : Got here immediately after WSJT-X started banging on the door with ...
     //  pServerPortInstance->sRxMsg[1].iLength = 7
     //  pServerPortInstance->sRxMsg[1].iType   = 0x40002 = 'READ' + 'FREQUENCY REPORT'
     //  pServerPortInstance->sRxMsg[1].bData = FE FE 94 E0 25 00 FD
     // ,---------------------------------------------------|___|
     // '--> For an IC-7300, according to "A7292-4EX-11" page 165,
     //      this means "Send/read the selected or unselected VFO frequency".
     //      0x00 is the SUB-COMMAND, specifying "the SELECTED VFO". Omg.
     //  Only the ugly monster (RigCtrl_ParseCIV()) is able to identify this
     //  as a READ COMMAND WITH SUB-COMMAND (not a one-byte-COMMAND with a one-byte-VALUE).
     //  Expect more fun like this, until WSJT-X may eventually work as a REMOTE CLIENT.
     //
     // Commands can only be forwarded to the REAL RADIO if the protocols are equal :
     if( pServerPortInstance->iRadioCtrlProtocol == pRadioPortInstance->iRadioCtrlProtocol ) // ok to forward the 'verbatim copy' to the REAL RADIO ?
      {
#      if(SWI_HARDCORE_DEBUGGING)
        if( RigCtrl_iConditionalBreakpointStep == 6 ) // forwarding a "verbatim copy" of a command like cmd 0x07 = "SelectVFO",
         { // (or any other command that does NOT have a UNIQUE unified PN] !
           RigCtrl_iConditionalBreakpointStep = 7; // now in RigCtrl_ForwardCommandFromServerPortToRadio(), expecting the transmission of
           // the 'verbatim copy' in pServerPortInstance->sRxMsg[1].bData, stored in RigCtrl_CIV_Server_OnWriteCommand().
           // 2025-10-19 : Got here when firing up N1MM Logger+, with ...
           //    iUnifiedPN = 0  (ok)
           //    pServerPortInstance->sParamInfoForPendingCommand.pszToken = "ScopeOnOff" (WRONG !)
           //    pServerPortInstance->sRxMsg[1].bData[0..6] = 0xFE FE 94 E0 07 00 FD,
           //    pServerPortInstance->sRxMsg[1].iLength     = 7,
           //    pServerPortInstance->sRxMsg[1].iType = 0x00030809 = RIGCTRL_MSGTYPE_UNKNOWN (b7..0) | RIGCTRL_MSGTYPE_FLAG_CIV (b11) | RIGCTRL_MSGTYPE_FLAG_WRITE_CMD (3<<16) .. (ok),
           //    pServerPortInstance->sz255CommentFromParser = "SelectVFO:00",
           //    pServerPortInstance->iMsgTypeFromParser = 0x00030809 = pServerPortInstance->sRxMsg[1].iType .
           //        ,---------------------------------------------|/
           //        '--> Forwarding of commands and responses between the
           //             "Virtual Rig"- and the "Real Radio"-port must be
           //             possible even with RIGCTRL_MSGTYPE_UNKNOWN,
           //             because with 'future' Icom radios, commands to read/write
           //             parameters that RCW-Keyer doesn't recognize MUST be
           //             forwarded properly - including Icom's anonymous WRITE RESPONSES,
           //             and even if we don't know if the command was a READ-
           //             or a WRITE-command, or whatever !
           //    pServerPortInstance->dwExpectedResponse_CmdAndSubcode = 0xFBFFFFFF
           //        ,-----------------------------------------------------|/
           //        '--> That "single-byte-command without sub[-sub]-codes
           //             should match the response in the line with "RX" shown below
           //              .. but obviously, it didn't. Search for
           //                 RigCtrl_iConditionalBreakpointStep == 7 for THAT step !
           // Ok so far, matches the line with "TX" in the traffic log .. :
           //    > r2 007 FE FE 94 E0 07 00 FD ; SelectVFO:00
           //    > TX 007 FE FE 94 E0 07 00 FD ; SelectVFO:00
           //    > RX 006 FE FE E0 94 FB FD    ; 'OK'  <- comment indicates a problem. Should have been: "; SelectVFO : OK"
           // .. but MISSING: "t2 006 FE FE E0 94 FB FD ; SelectVFO : OK"  !
         }
#      endif // SWI_HARDCORE_DEBUGGING ?


        // BEFORE calling RigCtrl_SendAndLogMessage(), already set the "response countdown" (= "currently busy"-indicator):
        pRadioPortInstance->iResponseCountdown_ms = pRC->iMaxResponseDelay_ms;
        fOk = RigCtrl_SendAndLogMessage( pRC,  // here: FORWARDING from Server- to the RADIO-port..
                 RIGCTRL_PORT_RADIO, RIGCTRL_ORIGIN_CONTROLLER,
                 pServerPortInstance->sRxMsg[1].bData,   // [in] the 'verbatim copy' from a SERVER / "Virtual Rig" port
                 pServerPortInstance->sRxMsg[1].iLength, // [in] length of the 'verbatim copy' (original CI-V command) in bytes
                 pServerPortInstance->sRxMsg[1].iType,   // [in] CI-V message type and additional flags for the log
                 pServerPortInstance->sz255CommentFromParser ); // [in] e.g. "SelectVFO:00", also for the log
        // Hint for debugging: To catch the RESPONSE from the REAL RADIO,
        //      set a breakpoint in RigCtrl_ParseCIV(), using RigCtrl_iConditionalBreakpointStep !
        if( fOk ) // similar as when called from API functions like RigCtrl_AskForVFOFrequency(),
         { // Set i32ExpectedResponse_CmdAndSubCode (since we have no valid Unified PN).
           // ,-----'
           // '-> Will be cleared later, along with pRadioPortInstance->iWorkingForServerPort,
           //     in RigCtrl_CheckAndForwardResponseToClients() .
           switch( pRadioPortInstance->iRadioCtrlProtocol )
            { case RIGCTRL_PROTOCOL_ICOM_CI_V :
                 switch( pServerPortInstance->iMsgTypeFromParser & RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP )
                  { case RIGCTRL_MSGTYPE_FLAG_READ_CMD  : // the message forwarded from SERVER to RADIO was a CI-V 'READ' command ->
                       // Only in THIS case, the RESPONSE contains at least the COMMAND BYTE, e.g.:
                       //   b[0] [1] [2] [3] [4] [5] [6] [7] [8] [9 [10]
                       //  > FE  FE  94  E0  03  FD                     ("Read operating frequency", iMsgLength=6)
                       //  > FE  FE  E0  94  03  00  00  00  07  00  FD (Read-Response WITH AN UNPREDICTABLE NUMBER OF DATA BYTES)
                       //                    '--------------------------------------,
                       pRadioPortInstance->dwExpectedResponse_CmdAndSubcode = //   |
                         RCTL_COMBINE_CMD_ONLY(pServerPortInstance->sRxMsg[1].bData[4]);
                       break;
                    case RIGCTRL_MSGTYPE_FLAG_WRITE_CMD : // the message forwarded from SERVER to RADIO was a CI-V 'WRITE' command ->
                       pRadioPortInstance->dwExpectedResponse_CmdAndSubcode = RCTL_COMBINE_CMD_ONLY(0xFB); // expect an unspecific "FB" a la CI-V
                       break;
                    default:
                       break;
                  }
                 break; // end case < forwarding a CI-V message from server- to radio-port >
             default: // for other protocols, an awful lot of work would be required here...
                 break; // .. which most likely isn't justified for a REMOTE CW KEYER.
            }
         } // end if < fOk from RigCtrl_SendAndLogMessage() on the RADIO PORT >
      }
     else
      { // leave fOk=FALSE;  will prepare a "Not-OK"-response from server to remote client further below.
      }
   }
  if( fOk )
   {
     RigCtrl_SetServerState( pServerPortInstance, RIGCTRL_SSTATE_FWD2RADIO_BUSY );
     // pRadioPortInstance->iWorkingForServerPort remains nonzero until the response
     // arrives from the REAL RADIO, in RigCtrl_CheckAndForwardResponseToClients(),
     //                 called from RigCtrl_ParseCIV() after receiving a RESPONSE .
   }
  else // could not 'forward' the command from server- to radio-port:
   { pRadioPortInstance->iWorkingForServerPort = 0; // here: aborted in RigCtrl_ForwardCommandFromServerPortToRadio()
     RigCtrl_CIV_Server_PrepareResponse_OkOrNotOk(pServerPortInstance, FALSE); // here: failed in RigCtrl_ForwardCommandFromServerPortToRadio() ..
     RigCtrl_SetServerState(pServerPortInstance, RIGCTRL_SSTATE_RESPONSE_READY); // here: set in RigCtrl_CIV_Server_OnReadCommand(), with the read-value immediately available (without forwarding to the 'real radio')
   }

  RigCtrl_LeaveCriticalSection( pRC ); // here: called when returning from RigCtrl_ForwardCommandFromServerPortToRadio()

  // .. next steps in RigCtrl_CheckAndForwardResponseToClients() ..

} // end RigCtrl_ForwardCommandFromServerPortToRadio()
#endif // SWI_RIGCTRL_ACT_AS_SERVER ?

//--------------------------------------------------------------------------
void RigCtrl_Handler( T_RigCtrlInstance *pRC )   // the incredible state machine ..
  // Periodically called every few ten milliseconds (roughly C_RIGCTRL_HANDLER_CALLING_INTERVAL_MS).
  //
  // Callers in Spectrum Lab :
  //  * If a SERIAL PORT is used : IOaccess.cpp  : IO_RadioCommThdFunc() .
  //  * If an INTERNET SOCKET (e.g. UDP) is used : RigCtrl_UDPThreadFunc().
  //
  // Callers in the Remote CW Keyer (RCW) :
  //  * Keyer_Main.cpp : TKeyerMainForm::Timer1Timer()
  //
  // Received data may be passed in ANYTIME between two calls of this handler,
  //               via RigCtrl_ProcessRxData() [for 'COM' ports, UDP, etc].
  // Potential multithreading issues must be avoided by the CALLER(s).
  // Implemented like a state machine for portability
  //    (with a single-threaded microcontroller firmware in mind) .
  //
  //
{
  char szComment[ RIGCTRL_MAX_COMMENT_LENGTH+4 ], *pszComment = NULL;
  char sz255[256];
  // ex: T_RigCtrl_IcomUdpPortInstance *pUdpPortInst = NULL;
  int i;
  int iPollingState2;  // ... from a sub-module's "periodic" handler
  T_RigCtrl_ParamInfo *pPI;
  BOOL fDone;
  T_RigCtrl_PortInstance *pRadioPortInstance = &pRC->PortInstance[RIGCTRL_PORT_RADIO]; // !
  T_RigCtrl_PortInstance *pServerPortInstance;  // SERVER port on this side, emulating e.g. a "Virtual Rig",
                                                // see RigControl_CIV_Server.c .

#if(0)  // removed: variables to read the 'Fixed Edge' frequencies for Icom's waterfall/spectrum display
  int  iRange, iEdge;
  BYTE b1, b2;
#endif

  if( pRC == NULL ) // oops.. instance deleted but some thread is still calling us !
   { return;
   }
  if( pRC->initState != initState_Opened ) // oops.. didn't call RigCtrl_Init() yet, or called RigCtrl_Close()
   { return;
   }

  if( pRC->iTransmitReqstCountdown_ms > 0 ) // pRC->iTransmitReqst has been cleared only a few milliseconds ago...
   { pRC->iTransmitReqstCountdown_ms -= C_RIGCTRL_HANDLER_CALLING_INTERVAL_MS; // .. so count this DOWN in RigCtrl_Handler()
   }

  // Process the RECEIVED DATA (including "unsolicited reports") on the radio control port:
  RigCtrl_ProcessRxData( pRC, RIGCTRL_PORT_RADIO );

  if( pRC->fListenOnlyMode ) // "passive" mode ? then don't SEND anything to the rig from RigCtrl_Handler() !
   { return;
   }
  if( pRadioPortInstance->iRadioCtrlProtocol == RIGCTRL_PROTOCOL_NONE ) // Neither CI-V nor Hamlib/Rigctld ?
   { return;  // don't try to connect at all, and don't bug me with messages like
              // "Remote spectrum timeout", etc etc pp ...
   }

  if( pRC->PortInstance[RIGCTRL_PORT_RADIO].dwPortErrorFlags & RIGCTRL_PORT_ERROR_FLAG_NOT_OPEN )
   { return;  // dont't try to SEND anything at the moment because the "radio control port" is currently NOT OPEN
   }

# if( SWI_RIGCTRL_ACT_AS_SERVER )
  // If the RADIO PORT is currently not busy from a single SEND_COMMAND / WAIT_FOR_RESPONSE
  // transaction, check if any of the other ports (e.g. used as SERVER for other apps
  // sharing the same radio) need to exchange commands on behalf of external clients:
  for( i=RIGCTRL_PORT_AUX_COM_1; i<=RIGCTRL_PORT_AUX_COM_LAST; ++i)
   { if( RigCtrl_IsPortBusy( pRadioPortInstance ) )
      { break; // polled IN THE LOOP, because THE LOOP ITSELF may make the RADIO CONTROL PORT BUSY again
      }
     pServerPortInstance = &pRC->PortInstance[i];
     switch( pServerPortInstance->iServerState ) // see state diagram in RigControl_CIV_Server.c (!)...
      { case RIGCTRL_SSTATE_WAIT_RADIO_NBUSY : // ok, this server port needs to talk to the radio on behalf of a remote CLIENT ...
           // (see state diagram in RigControl_CIV_Server.c )
#         if(SWI_HARDCORE_DEBUGGING)
           if(RigCtrl_iConditionalBreakpointStep == 1) // previously received a "Set Split Mode" command on an AUX COM PORT ?
            { RigCtrl_iConditionalBreakpointStep = 2;  // just about to call RigCtrl_ForwardCommandFromServerPortToRadio() for the "Split Mode"-command !
            }
           if(RigCtrl_iConditionalBreakpointStep == 5) // previously received a "SelectVFO" command (0x07) from a "Virtual Rig" ?
            { RigCtrl_iConditionalBreakpointStep = 6;  // just about to call RigCtrl_ForwardCommandFromServerPortToRadio() for the "SelectVFO"-command !
              // For cmd 0x07 = "SelectVFO", do NOT expect a call to RigCtrl_SendWriteCommandForUnifiedPN()
              //                                [because that command DOES NOT HAVE A UNIQUE unified PN] !
              // Instead, the entire command/packet received on a "Virtual Rig" port has been stored
              // as a 'verbatim copy' in pServerPortInstance->sRxMsg[1].bData - see RigCtrl_CIV_Server_OnWriteCommand().
              //   ,---------------------|__________________________________|
              //   '--> This 'verbatim copy' will be sent in RigCtrl_ForwardCommandFromServerPortToRadio(),
              //        which is where the next transition of RigCtrl_iConditionalBreakpointStep takes place (6 -> 7).
              // Checked (2025-10-19):
              //  pServerPortInstance->sRxMsg[1].iLength = 7  (ok)
              //  pServerPortInstance->sRxMsg[1].bData[0..6] = 0xFE FE 94 E0 07 00 FD (ok, that's the "SelectVFO"-WRITE-commmand WITH 0x00 = "VFO A")
            }
#         endif // SWI_HARDCORE_DEBUGGING ?
           RigCtrl_ForwardCommandFromServerPortToRadio( pRC, pServerPortInstance );
           //  '--> when successful, pRadioPortInstance->iWorkingForServerPort is nonzero now,
           //       and periodically polling parameters like the S-Meter must pause
           //       until the RADIO PORT finished working on behalf of a remote client.
           break; // end case RIGCTRL_SSTATE_WAIT_RADIO_NBUSY
        case RIGCTRL_SSTATE_FWD2RADIO_BUSY: // COMMAND (on behalf of a remote client) already on the way to the RADIO
           // (nothing to do here; the transition from RIGCTRL_SSTATE_FWD2RADIO_BUSY to RIGCTRL_SSTATE_FWD2RADIO_DONE
           //  happens in RigCtrl_CheckAndForwardResponseToClients(), depending on pRadioPortInstance->iWorkingForServerPort )
           break;
        default: // any other RigControl SERVER state isn't our business (in RigCtrl_Handler() )
           break;

      }   // end switch( pServerPortInstance->iServerState )
   }     // end for < all 'Aux COM ports' (that possibly run as SERVERS for external applications) >
# endif // SWI_RIGCTRL_ACT_AS_SERVER ?

  iPollingState2 = pRC->iParameterPollingState;
# if( SWI_SUPPORT_YAESU_5_BYTE_CAT )
  if( pRadioPortInstance->iRadioCtrlProtocol == RIGCTRL_PROTOCOL_YAESU_5_BYTE )
   { iPollingState2 = Yaesu5Byte_Handler( &pRC->Y5B ); // here: called from RigCtrl_Handler()
   }
# endif // SWI_SUPPORT_YAESU_5_BYTE_CAT ?
  if( iPollingState2 != pRC->iParameterPollingState )
   { // The 'protocol specific handler' (e.g. Yaesu5Byte_Handler) is in
     // a different PARAMETER POLLING STATE than module RigControl.c itself ->
     RigCtrl_OnPollingStateChangeInSubmodule( pRC, iPollingState2 );  // set pRC->iParameterPollingState to the one reported by Yaesu5Byte_Handler() ...
   } // end if < parameter polling state from e.g. Yaesu5Byte.c differs from our own > ?

  if( pRadioPortInstance->iResponseCountdown_ms > 0 ) // "do not send anything at the moment" because we're waiting for a response ->
   { pRadioPortInstance->iResponseCountdown_ms -= C_RIGCTRL_HANDLER_CALLING_INTERVAL_MS; // <- e.g. 50 milliseconds
     if( pRadioPortInstance->iResponseCountdown_ms <= 0 )  // just timed out ...
      { pRadioPortInstance->iResponseCountdown_ms = 0;
        if( (pRadioPortInstance->dwExpectedResponse_CmdAndSubcode != (DWORD)RIGCTRL_NOVALUE_INT )
         && (pRC->iParameterPollingState != RIGCTRL_POLLSTATE_PASSIVE )
         && (pRC->iParameterPollingState != RIGCTRL_POLLSTATE_DONE ) )
         { sprintf( pRC->sz255LastError, "timed out waiting for Cmd 0x%02X, state=%d",
             (int)pRadioPortInstance->dwExpectedResponse_CmdAndSubcode, (int)pRC->iParameterPollingState );
           if( RigCtrl_TrafficMonitor.iDisplayOptions & RIGCTRL_TMON_DISPLAY_OPTION_PAUSE_ON_ERROR )
            { if( ! pRC->PortInstance[RIGCTRL_PORT_RADIO].fTrafficMonitorPausedOnTrigger )
               { sprintf( sz255, "Traffic Log paused on %s after RESPONSE TIMEOUT, cmd=0x%02X",
                   RigCtrl_GetDescriptivePortName(pRadioPortInstance), // <- e.g. "Radio Control Port", or even "COM5:IC-7300"
                   (int)pRadioPortInstance->dwExpectedResponse_CmdAndSubcode );
                 RigCtrl_AddToTrafficLog( pRC, RIGCTRL_PORT_RADIO, RIGCTRL_ORIGIN_CONTROLLER,
                   NULL/*pbMessage*/, 0/*iMsgLength*/,
                   RIGCTRL_MSGTYPE_FLAG_ERROR, sz255 );
               }
              pRC->PortInstance[RIGCTRL_PORT_RADIO].fTrafficMonitorPausedOnTrigger = TRUE;
            } // end if < RIGCTRL_TMON_DISPLAY_OPTION_PAUSE_ON_ERROR .. >
           if( pRC->iParameterPollingState==RIGCTRL_POLLSTATE_READ_RADIO_ID )
            { // This was the FIRST query sent to the radio, and it didn't respond.
              // It may have been turned off right from the start of the program,
              // so instead of wasting more time (polling other parameters),
              // try to turn the radio ON via command:
              if( pRC->iCapabilities & RIGCTRL_CAPS_POWER_ON_OFF_VIA_CAT )
               { // the rig can be turned on and off cia CAT, so try to turn it on:
                 pRC->iParameterPollingState = RIGCTRL_POLLSTATE_TURN_RIG_ON;
               }
            }
           else // not in RIGCTRL_POLLSTATE_READ_RADIO_ID ->
            {
              ++pRadioPortInstance->numResponseTimeouts;
              if( pRadioPortInstance->numResponseTimeouts > 3 ) // too many response timeouts in a row ?
               { // The rig doesn't seem to respond AT ALL. It may be turned off..
                 if( pRC->iCapabilities & RIGCTRL_CAPS_POWER_ON_OFF_VIA_CAT )
                  { // so if we know the rig can be turned on and off cia CAT, try to turn it on again, and start all over
                    pRC->iParameterPollingState = RIGCTRL_POLLSTATE_TURN_RIG_ON; // try to turn the rig on, then start all over
                  }
               } // end if( pRC->numResponseTimeouts > 3 ) ?
            }   // end if < ! RIGCTRL_POLLSTATE_READ_RADIO_ID >
         }     // end if < EXPECTED a response, AND one of the 'active' parameter-polling-states >
        else // pRadioPortInstance->dwExpectedResponse_CmdAndSubcode < 0 .. but we may expect some other response:
        if( pRadioPortInstance->iExpectResponseForUnifiedPN > 0 ) // <- less 'Icom-specific' than dwExpectedResponse_CmdAndSubcode..
         { // timed out waiting waiting for a response after RigCtrl_AskForParameter() :
           pPI = RigCtrl_GetInfoForUnifiedParameterNumber(pRadioPortInstance->iExpectResponseForUnifiedPN);
           if( pPI != NULL )
            { sprintf( pRC->sz255LastError, "timed out waiting for read response for %s, state=%d",
              (char*)pPI->pszToken, (int)pRC->iParameterPollingState );
            }
           if( RigCtrl_TrafficMonitor.iDisplayOptions & RIGCTRL_TMON_DISPLAY_OPTION_PAUSE_ON_ERROR )
            { if( ! pRadioPortInstance->fTrafficMonitorPausedOnTrigger )
               { sprintf( sz255, "Traffic Log paused on %s after RESPONSE TIMEOUT for %s",
                          RigCtrl_GetDescriptivePortName(pRadioPortInstance), // <- e.g. "Radio Control Port", or even "COM5:IC-7300"
                          (pPI != NULL) ? (char*)pPI->pszToken : "(unknown param)"  );
                 RigCtrl_AddToTrafficLog( pRC, RIGCTRL_PORT_RADIO, RIGCTRL_ORIGIN_CONTROLLER,
                   NULL/*pbMessage*/, 0/*iMsgLength*/,
                   RIGCTRL_MSGTYPE_FLAG_ERROR, sz255 );
               }
              pRadioPortInstance->fTrafficMonitorPausedOnTrigger = TRUE;
            } // end if < RIGCTRL_TMON_DISPLAY_OPTION_PAUSE_ON_ERROR .. >
         }   // end if < iExpectResponseForUnifiedPN .. >
      }     // end if < iResponseCountdown_ms just timed out >
   }
  else  // enough time has passed to send the next command..
   { switch( pRC->iParameterPollingState )
      { case RIGCTRL_POLLSTATE_START_RD:  // RigCtrl_StartReading() has been called..
           pRadioPortInstance->numResponseTimeouts = 0;
           // If the protocol is CI-V, start by reading the TRANSCEIVER ID.
           // We'll know if a radio supports a certain 'advanced' feature then,
           // much better than relying on the radio's "default ID" !
           RigCtrl_AskForRadioID( pRC, RIGCTRL_PORT_RADIO );  // (for non-Icom-radios, this starts something equivalent in e.g. Yaesu5Byte.c)
           pRC->iParameterPollingState = RIGCTRL_POLLSTATE_READ_RADIO_ID;
           pRC->iParameterPollingSubState = 0; // here: enumerator for RigCtrl_EnumerateRequiredPNs()
           pRC->iNumBandStackingRegs = 0; // forget old 'Band Stacking Register' entries
           break;
        case RIGCTRL_POLLSTATE_READ_RADIO_ID: // waiting for a response with the radio's "default" ID, as a connection test and to check its type
           pRC->iParameterPollingState = RIGCTRL_POLLSTATE_PARAMS_FOR_RIGCTL;
           pRC->iParameterPollingSubState = 0; // here: enumerator for RigCtrl_EnumerateRequiredPNs()
           pRC->iNumBandStackingRegs = 0; // forget old 'Band Stacking Register' entries
           break;
        case RIGCTRL_POLLSTATE_PARAMS_FOR_RIGCTL: // (try to) read the current VFO frequency, mode, etc
           fDone = TRUE;
           while( (i=RigCtrl_EnumerateRequiredPNs( pRC->iParameterPollingSubState++)) > 0 )
            { pPI = RigCtrl_GetInfoForUnifiedParameterNumber(i/*iUnifiedPN*/);
              if( pPI != NULL )
               { if( !RigCtrl_HaveValidParamValue( pRC, pPI ) )
                  { RigCtrl_AskForParameter( pRC, pPI, RIGCTRL_PORT_RADIO ); // <- sets pRC->iResponseCountdown_ms and pRadioPortInstance->iExpectResponseForUnifiedPN
                    fDone = FALSE;
                    break;
                  }
               }
            }
           if( fDone ) // finished polling all important parameters for RigControl.c itself :
            { pRC->iParameterPollingSubState = pRC->iParameterPollingArrayIndex = 0;
              if( pRC->iNumTxBands > 0 )  // successfully detected the NUMBER OF TRANSMIT BANDS..
               { pRC->iParameterPollingState = RIGCTRL_POLLSTATE_TX_BAND_EDGES;
               }
              else // could not determine pRC->iNumTxBands -> skip RIGCTRL_POLLSTATE_TX_BAND_EDGES
               { pRC->iParameterPollingState = RIGCTRL_POLLSTATE_BAND_STACKING_REGS;
               }
            }
           break; // end case RIGCTRL_POLLSTATE_PARAMS_FOR_RIGCTL
        case RIGCTRL_POLLSTATE_TX_BAND_EDGES :  // Determine the edge frequencies of all TRANSMIT bands:
           // [in] pRC->iNumTxBands : has already been read because RIGCTRL_PN_NUM_TX_BANDS is included in RigCtrl_EnumerateRequiredPNs().
           // [out] pRC->TxBandEdges[ 0 .. pRC->iNumTxBands-1 ] will be read from THE RIG .
           fDone = TRUE;
           if( (pRC->iParameterPollingSubState < pRC->iNumTxBands )
            && (pRC->iParameterPollingSubState < RIGCTRL_NUM_TX_BAND_EDGES ) )
            { // how to read an array of "transmit band edge frequencies" ?
              switch( pRadioPortInstance->iRadioCtrlProtocol )
               { case RIGCTRL_PROTOCOL_ICOM_CI_V : // using Icom's CI-V :
                    // The following is based on Icom's "IC-7300 Full Manual",
                    //  revision 6, document ID "A7292-4EX-6", chapter 19 :
                    // Cmd 0x1E, SubCmd 0x00 = "Read number of available TX frequency band" (what they mean is "bands")
                    //           has already been sent and answered.
                    // Cmd 0x1E, SubCmd 0x01 = "Read TX band edge frequencies" .
                    //   For the READ REQUEST, we supposed to send
                    //   cmd 0x1E followed by SubSmd 0x01 followed by the "Edge Number"
                    //   to retrieve the edge frequencies for the requested "Edge".
                    //   The READ RESPONSE contains an "Edge Number", start frequency
                    //   (delimited by a funny separator, code 0x2D), and the
                    //   end frequency (terminated by the CI-V postamble, 0xFD).
                    //   Tried that on an IC-9700, with the following result:
                    // >  TX 007 FE FE 00 00 1E 00 FD      ; read NumTxBands
                    // >  RX 008 FE FE 00 A2 1E 00 03 FD   ; NumTxBands:3
                    // >  TX 008 FE FE 00 00 1E 01 01 FD   ; tx band edge[0] ?
                    // >  RX 013 FE FE 00 A2 1E 01 01 00 00 00 44 01 2D 00 00 00 46 01 FD ; 144 MHz .. 146 MHz
                    // >  TX 008 FE FE 00 00 1E 01 02 FD   ; tx band edge[1] ?
                    // >  RX 013 FE FE 00 A2 1E 01 02 00 00 00 30 04 2D 00 00 00 40 04 FD ; 430 MHz .. 440 MHz
                    // >  TX 008 FE FE 00 00 1E 01 03 FD   ; tx band edge[2] ?
                    // >  RX 013 FE FE 00 A2 1E 01 03 00 00 00 40 12 2D 00 00 00 00 13 FD ; 1240 MHz .. 1300 MHz
                    //                              '-- "Edge number"
                    //   Tried the same on an IC-7300, modified to transmit on 5.3 and 70 MHz:
                    // >  TX 007 FE FE 00 00 1E 00 FD      ; read NumTxBands
                    // >  RX 008 FE FE 00 94 1E 00 01 FD   ; :1 (surprise surprise, the IC-7300 is a "monoband rig" with only ONE TX BAND now...)
                    // >  TX 008 FE FE 00 00 1E 01 01 FD   ; tx band edge[0] ?
                    // >  RX 013 FE FE 00 94 1E 01 01 00 00 10 00 00 2D 00 00 80 74 00 FD ; 0.1 .. 74.8 MHz
                    //      -> Ok.. technically correct but not really helpful anymore,
                    //         with only a SINGLE BAND from 100 kHz to 74.8 MHz !
                    sprintf(szComment, "tx band edge[%d] ?", (int)pRC->iParameterPollingSubState );
                    if( RigCtrl_SendShortCIVMessage( pRC, RIGCTRL_PORT_RADIO, RIGCTRL_ORIGIN_CONTROLLER,
                       RIGCTRL_MSGTYPE_RADIO_ID, // MsgType for the traffic log ...
                       szComment,  // comment for the traffic log
                       0x1E,  // iCmd: CI-V command "TX bands"
                       0x01,  // iSubCmd: "TX band edge frequencies"
                       RigCtrl_IntToBCD2(++pRC->iParameterPollingSubState), // EDGE NUMBER (Icom starts counting at ONE, not ZERO)
                       -1 ) )  // additional data bytes : None !
                     { // If all works as planned, the transceiver also responds with cmd 0x1E 0x01,
                       // the rest happens in RigCtrl_ParseCIV(), case 0x1E .
                       pRadioPortInstance->iExpectResponseForUnifiedPN = RIGCTRL_PN_TX_BAND_EDGES; // <- for RigCtrl_ParseCIV()
                       pRadioPortInstance->dwExpectedResponse_CmdAndSubcode = RCTL_COMBINE_CMD_AND_SUB(0x1E,0x01);  // <- expected command and subcode
                       pRadioPortInstance->iResponseCountdown_ms = pRC->iMaxResponseDelay_ms; // now waiting for the radio to respond with "OK" or "NOT OK" after "Report another BAND EDGE"
                       fDone = FALSE;
                     }
                    break;
#               if( SWI_SUPPORT_YAESU_5_BYTE_CAT )
                 case RIGCTRL_PROTOCOL_YAESU_5_BYTE: // Yaesu's OLD UGLY CAT - see Yaesu5Byte.c ..
                    // Retrieve BAND EDGES with Yaesu's old ugly CAT ? You got to be kidding !
                    break; // end case <Yaesu binary 5-byte-command protocol>
#               endif // SWI_SUPPORT_YAESU_5_BYTE_CAT ?

                 default: // any other protocol has not been implemented yet
                    break;
               } // end switch( pRC->iRadioCtrlProtocol )
            }   // end if < more TRANSMIT BANDS > ?
           if( fDone ) // finished polling all TRANSMIT BANDS and their edge frequencies :
            { // Notify the application about possibly 'new data'
              // in pRC->TxBandEdges[ 0 .. pRC->iNumTxBands-1 ] :
              pRC->fTxBandsModified = TRUE; // <- application may POLL and even CLEAR this flag
              RigCtrl_InitUserDefinedBandsFromRigInfo(pRC);
              pRC->iParameterPollingState = RIGCTRL_POLLSTATE_BAND_STACKING_REGS;
              pRC->iParameterPollingSubState = 0;
            }
           break; // end case RIGCTRL_POLLSTATE_TX_BAND_EDGES

        case RIGCTRL_POLLSTATE_BAND_STACKING_REGS: // (try to) read out the 'band stacking registers' (special VFO memories).
           fDone = TRUE;
           // [in]  pRC->iParameterPollingSubState   : here, Icom's "Frequency Band Code" for cmd 0x1A 0x01 <BandCode> <RegisterCode>
           // [in]  pRC->iParameterPollingArrayIndex : here, Icom's "Register code" (see details further below)
           // [out] pRC->BandStackingRegs[ pRC->iNumBandStackingRegs++ ] ( in RigCtrl_ParseCIV() )
           //
           // How many "stackable bands" are supported by the radio ?
           // This definitely cannot be the number of TX bands, because for
           // example, an IC-7300 modified for all-band-transmit only reported
           // A SINGLE TRANSMIT-BAND (thus pRC->iNumTxBands == 1).
           // The IC-9700 CI-V manual (document #A7508-3EX-4) shows command
           //  0x1A 0x01 followed by a TWO-BYTE DATA FIELD (with 2 * two-digit BCD)
           //  with a rig-specific "Frequency band code" first (1)
           //               and a "Register code", 01..03 next (2) .
           //               "Register code" 01 is the newest 'stacked entry',
           //                               03 is the oldest 'stacked entry'.
           // "Frequency band codes" for an IC-9700 :
           //              01=2m, 02=70cm, 03=23 cm.  No code for "GENE" at all.
           // "Frequency band codes" for an IC-7300 :
           //              01=160m, 02=80m, 03=40m, 04=30m, 05=20m, 06=17m,
           //              07=15m,  08=12m, 09=10m, 10=6m,  11="GENE" (all the rest).
           // So, to avoid even more 'hard-coded rig-specific wisdom' in this
           // module, simply try to read as many Band stacking registers,
           // beginning at "Frequency band code" 01, until the radio responds
           // with an error for "Register code" 01.
           // As in most array-like items polled here,
           //      pRC->iParameterPollingSubState is the 'iterator',
           //      here: running through all "Frequency band codes" a la Icom.
           // In addition, because each "stackable frequency band"
           //              may have up to THREE STACKED ENTRIES,
           //      pRC->iParameterPollingArrayIndex is used for what Icom calls
           //              "Register code", where code 01 (in BCD) means the
           //              MOST RECENT entry (displayed on the left side in
           //              Icom's own 'BAND STACKING REGISTER' display), etc.
           //              (in an IC-7300, a short touch on the MHz digits
           //               first opens a band list only, a longer touch
           //               immediately shows all three 'stack levels'.
           //               That's an incredibly useful feature, thus our
           //               simple "remote GUI" tries something similar.)
           // The RESPONSE (sent by the radio on CI-V command 0x1A 0x01)
           //      will be parsed in RigCtrl_ParseCIV() [in the long list of
           //      sub-commands after 0x1A].
           if( (pRC->iParameterPollingSubState <= (RIGCTRL_NUM_BAND_STACKING_REGS/3) )
            && (pRC->iNumBandStackingRegs < RIGCTRL_NUM_BAND_STACKING_REGS) )
            { // how to read an array of "transmit band edge frequencies" ?
              switch( pRadioPortInstance->iRadioCtrlProtocol )
               { case RIGCTRL_PROTOCOL_ICOM_CI_V : // using Icom's CI-V :
                    ++pRC->iParameterPollingArrayIndex; // here: 1..3 for Icom's three "stack levels"
                    if( pRC->iParameterPollingArrayIndex > 3 ) // finished all three "Register codes" (stack levels) of a band ?
                     {  pRC->iParameterPollingArrayIndex = 1;
                        ++pRC->iParameterPollingSubState;   // next "Frequency band code"
                     }
                    if( pRC->iParameterPollingSubState < 1 )
                     {  pRC->iParameterPollingSubState = 1; // lowest value for Icom's "Frequency band code"
                     }
                    // The following is based on Icom's "IC-970 CI-V Reference Guide",
                    //  revision 4, document ID "A7508-3EX-4", page 15, "Band stacking register".
                    // READ REQUEST: Cmd 0x1A, SubCmd 0x01, followed by TWO two-digit BCD bytes:
                    //   Data byte "(1)" (circled 'one' seems to be a byte index)
                    //               = "Frequency band code" (rig specific,
                    //                 for the IC-9700, 01=2m, 02=70cm, 03=23cm)
                    //   Date byte "(2)" = "Register code" [01, 02, 03, full stop].
                    //
                    //   The READ RESPONSE is terribly complex and again rig-
                    //   specific; more on that in RigCtrl_ParseMemoryContent() .
                    //   Tried on an IC-9700, with the following result:
                    // >  TX 007 FE FE 00 00 1A 01 01 01   ; give me the the stacked memory for "Band 1", 1st entry (top of the stack
                    // >  RX 013 FE FE 00 A2 1A 01 01 01 ..... ; an INCREDIBLY long response
                    //                "band code"---'  '---"Register code"
                    //                (good for nothing
                    //                 because it's rig-specific.
                    //                 We find out the actual BAND from the
                    //                 FREQUENCY - see RigCtrl_ParseMemoryContent() .)
                    //    Lacking a way to determine the NUMBER of "band codes" for band stacking registers,
                    //    simply increment the "band code" until the radio responds with "Not Ok".
                    //    Here for example an IC-7300, which reported ELEVEN "band codes"
                    //    before responding with "NotOK" when asked for the 12th "band code":
                    // >  TX 009 FE FE 00 00 1A 01 11 03 FD ; BandStackReg[11][3} ?
                    // >  RX 017 FE FE 00 94 1A 01 11 03 20 73 10 70 00 03 02 00 00 08 85 00 08 85 .. ; band stack
                    // >  TX 009 FE FE 00 00 1A 01 12 01 FD ; BandStackReg[12][1] ?
                    // >  RX 006 FE FE 00 94 FA FD          ; NotOK for 0x1A010000
                    //    Because the "NotOK" response (0xFA) does not send back
                    //    the command or even sub-command, RigCtrl_ParseCIV()
                    //    is aware of pRadioInstance->dwExpectedResponse_CmdAndSubcode,
                    //    and -in THAT STATE- uses the "NotOK" reponse as a
                    //    marker for the end of the "stackable bands list".
                    sprintf(szComment, "BandStackReg[%d][%d] ?",
                         (int)pRC->iParameterPollingSubState, (int)pRC->iParameterPollingArrayIndex );
                    if( RigCtrl_SendShortCIVMessage( pRC, RIGCTRL_PORT_RADIO, RIGCTRL_ORIGIN_CONTROLLER,
                       RIGCTRL_MSGTYPE_RADIO_ID, // MsgType for the traffic log ...
                       szComment,  // comment for the traffic log
                       0x1A,  // iCmd: CI-V command for "All Kind Of Stuff"
                       0x01,  // iSubCmd: "Band Stacking Register"
                       RigCtrl_IntToBCD2(pRC->iParameterPollingSubState), // "Frequency Band Code" (01 .. ??=
                       RigCtrl_IntToBCD2(pRC->iParameterPollingArrayIndex) ) )  // additional data bytes : None !
                     { // If all works as planned, the transceiver also responds with cmd 0x1A 0x01 ...,
                       // the rest happens in RigCtrl_ParseCIV(), case 0x1A -> RigCtrl_ParseMemoryContent() .
                       pRadioPortInstance->iExpectResponseForUnifiedPN = RIGCTRL_PN_BAND_STACKING_REGS; // <- for RigCtrl_ParseCIV()
                       pRadioPortInstance->dwExpectedResponse_CmdAndSubcode = RCTL_COMBINE_CMD_AND_SUB(0x1A,0x01);  // <- expected command and subcode
                       pRadioPortInstance->iResponseCountdown_ms = pRC->iMaxResponseDelay_ms; // now waiting for the radio to respond with "OK" or "NOT OK" after "Report another BAND STACKING REGISTER"
                       fDone = FALSE;
                     }
                    break;
#                if( SWI_SUPPORT_YAESU_5_BYTE_CAT )
                  case RIGCTRL_PROTOCOL_YAESU_5_BYTE: // Yaesu's OLD UGLY CAT - see Yaesu5Byte.c ..
                    // Read BAND STACKING REGISTERS with Yaesu's old ugly CAT ? Keep dreaming !
                    break; // end case <Yaesu binary 5-byte-command protocol>
#                endif // SWI_SUPPORT_YAESU_5_BYTE_CAT ?

                 default: // any other protocol has not been implemented yet
                    break;
               } // end switch( pRC->iRadioCtrlProtocol )
            }   // end if < more BAND STACKING REGISTERS > ?
           if( fDone ) // finished polling all 'Band Stacking Registers' :
            { RigCtrl_FinishReadingBandStackingRegs(pRC);
              // '--> next "parameter polling state", etc
            }
           break; // end case RIGCTRL_POLLSTATE_BAND_STACKING_REGS

        case RIGCTRL_POLLSTATE_PARAMS_FOR_HAMLIB: // (try to) read less important parameters that e.g. the Hamlib server may need
           fDone = TRUE;
           while( (i=HLSRV_EnumerateRequiredPNs( pRC->iParameterPollingSubState++)) > 0 )
            { pPI = RigCtrl_GetInfoForUnifiedParameterNumber(i/*iUnifiedPN*/);
              if( pPI != NULL )
               { if( !RigCtrl_HaveValidParamValue( pRC, pPI ) ) // don't "ask" for the same parameter twice (at least not HERE)
                  { RigCtrl_AskForParameter( pRC, pPI, RIGCTRL_PORT_RADIO );
                    fDone = FALSE;
                    break;
                  }
               }
            }
           if( fDone ) // finished polling all important parameters for HamlibServer.c :
            {
              // ex: if( pRC->iCapabilities  & RIGCTRL_CAPS_SPECTRUM_VIA_CAT )
              // Forget it. Don't rely on a hard-coded (built-in) table with
              //            rig-capabilities. It's THE USER'S choice to try, so:
              if( (pRC->dwRigControlFlags & RIGCTRL_FLAG_WANT_SPECTRUM_DATA ) && (pRC->iCapabilities & RIGCTRL_CAPS_SPECTRUM_VIA_CAT ) )
               { pRC->iParameterPollingState = RIGCTRL_POLLSTATE_CONFIGURE_SPECTRUM; // .. to turn spectrum data OFF
                 pRC->iParameterPollingSubState = 0;
               }
              else // the radio most likely will not "understand" CI-V 0x27, so:
               { // ex: pRC->iParameterPollingState = RIGCTRL_POLLSTATE_DONE;
                 RigCtrl_EnterState_Done( pRC );
               }
            }
           break; // end case RIGCTRL_POLLSTATE_PARAMS_FOR_HAMLIB
        case RIGCTRL_POLLSTATE_CONFIGURE_SPECTRUM: // (try to) activate transmission of "spectrum scope waveform data"
           if( (pRC->dwRigControlFlags & RIGCTRL_FLAG_WANT_SPECTRUM_DATA ) && (pRC->iCapabilities & RIGCTRL_CAPS_SPECTRUM_VIA_CAT) )
            { if( pRadioPortInstance->iRadioCtrlProtocol == RIGCTRL_PROTOCOL_ICOM_CI_V ) // only for "CI-V"..
               {
                 switch( pRC->iParameterPollingSubState++ )
                  { case 0  : // find out if the spectrum scope is in CENTER- or FIXED mode:
                       RigCtrl_SendShortCIVMessage( pRC, RIGCTRL_PORT_RADIO, RIGCTRL_ORIGIN_CONTROLLER,
                            RIGCTRL_MSGTYPE_SPECTRUM_CONFIG,
                            "center/fixed ?", // comment for traffic log
                            0x27,  // iCmd   : "Scope waveform data"
                            0x14,  // iSubCmd: "Send/read the Scope Center mode or Fixed mode setting"
                            0x00,  // 1st data byte : 0x00 "Fixed" (what a misnomer.. not the FIXED MODE)
                            -1 );  // 2nd data byte : omit for the REQUEST (only used in the RESPONSE)
                       pRadioPortInstance->dwExpectedResponse_CmdAndSubcode = RCTL_COMBINE_CMD_AND_SUB(0x27,0x14); // at least an IC-7300 responded with 0x27 0x14 ..
                       pRadioPortInstance->iResponseCountdown_ms = pRC->iMaxResponseDelay_ms; // now waiting for the radio to respond after asking for "Spectrum Center/Fixed Mode"
                       break;
                    case 1  : // find out if the spectrum scope's "span" setting
                       RigCtrl_SendShortCIVMessage( pRC, RIGCTRL_PORT_RADIO, RIGCTRL_ORIGIN_CONTROLLER,
                            RIGCTRL_MSGTYPE_SPECTRUM_CONFIG,
                            "scope span ?", // comment for traffic log
                            0x27,  // iCmd   : "Scope waveform data"
                            0x15,  // iSubCmd: "read the span setting in the Center mode Scope"
                            0x00,  // 1st data byte : 0x00 "Fixed" (what a misnomer.. not the FIXED MODE)
                            -1 );  // 2nd data byte : omit for the REQUEST (only used in the RESPONSE)
                       pRadioPortInstance->dwExpectedResponse_CmdAndSubcode = RCTL_COMBINE_CMD_AND_SUB(0x27,0x15); // at least an IC-7300 responded with 0x27 0x15 ..
                       pRadioPortInstance->iResponseCountdown_ms = pRC->iMaxResponseDelay_ms; // now waiting for the radio to respond after asking for "Spectrum Span"
                       break;
                    case 2  : // find out the "Edge number setting in the Fixed mode Scope"
                       RigCtrl_SendShortCIVMessage( pRC, RIGCTRL_PORT_RADIO, RIGCTRL_ORIGIN_CONTROLLER,
                            RIGCTRL_MSGTYPE_SPECTRUM_CONFIG,
                            "edge nr ?", // comment for traffic log
                            0x27,  // iCmd   : "Scope waveform data"
                            0x16,  // iSubCmd: "Edge number setting[s] in the Fixed mode Scope"
                            0x00,  // 1st data byte : 0x00 "Fixed" (what a misnomer.. not the FIXED MODE)
                            -1 );  // 2nd data byte : omit for the REQUEST (only used in the RESPONSE)
                       pRadioPortInstance->dwExpectedResponse_CmdAndSubcode = RCTL_COMBINE_CMD_AND_SUB(0x27,0x16);
                       pRadioPortInstance->iResponseCountdown_ms = pRC->iMaxResponseDelay_ms; // now waiting for the radio to respond after asking for "Spectrum Edge Number"
                       break;
                    // don't need the "Scope hold function status" yet (0x27 0x17) !
                    case 3  : // find out the "Send/read the Scope Attenuator setting" (IC-7851, n/a on IC-7300)
                       RigCtrl_SendShortCIVMessage( pRC, RIGCTRL_PORT_RADIO, RIGCTRL_ORIGIN_CONTROLLER,
                            RIGCTRL_MSGTYPE_SPECTRUM_CONFIG,
                            "scope atten. ?", // comment for traffic log
                            0x27,  // iCmd   : "Scope waveform data"
                            0x18,  // iSubCmd: "Scope Attenuator setting"
                            0x00,  // 1st data byte : 0 = main scope, 1 = sub scope
                            -1 );  // 2nd data byte : omit for the REQUEST (only used in the RESPONSE)
                       pRadioPortInstance->dwExpectedResponse_CmdAndSubcode = RCTL_COMBINE_CMD_AND_SUB(0x27,0x18);
                       pRadioPortInstance->iResponseCountdown_ms = pRC->iMaxResponseDelay_ms; // now waiting for the radio to respond after asking for "Scope Attenuator Setting"
                       // An IC-7300 (which doesn't have an extra attenuator for the SCOPE)
                       // responded with "Not Good" (NG, cmd 0xFA) .
                       break;
                    case 4  : // find out the "Scope Reference level setting"
                       RigCtrl_SendShortCIVMessage( pRC, RIGCTRL_PORT_RADIO, RIGCTRL_ORIGIN_CONTROLLER,
                            RIGCTRL_MSGTYPE_SPECTRUM_CONFIG,
                            "ref lvl ?", // comment for traffic log
                            0x27,  // iCmd   : "Scope waveform data"
                            0x19,  // iSubCmd: "Scope Reference level settings"
                            0x00,  // 1st data byte : 0 = main scope, 1 = sub scope
                            -1 );  // 2nd data byte : omit for the REQUEST (only used in the RESPONSE)
                       pRadioPortInstance->dwExpectedResponse_CmdAndSubcode = RCTL_COMBINE_CMD_AND_SUB(0x27,0x19);
                       pRadioPortInstance->iResponseCountdown_ms = pRC->iMaxResponseDelay_ms; // now waiting for the radio to respond after asking for "Spectrum Reference Level"
                       pRC->iParameterPollingArrayIndex = 0; // prepare reading "Fixed edge frequency settings" ...
                       pRC->iScopeNumFreqRanges        = RIGCTRL_MAX_FIXED_EDGE_SCOPE_FREQ_RANGES;
                       pRC->iScopeNumEdgesPerFreqRange = RIGCTRL_MAX_FIXED_EDGES_PER_FREQ_RANGE;
                       break;
#                 if(0)
                    case 5 :  // read all available "Fixed edge frequency settings"
                       if( pRC->iParameterPollingArrayIndex <
                             (  RIGCTRL_MAX_FIXED_EDGE_SCOPE_FREQ_RANGES
                              * RIGCTRL_MAX_FIXED_EDGES_PER_FREQ_RANGE ) )
                        { --pRC->iParameterPollingSubState; // undo pRC->iParameterPollingSubState++
                          iRange = 1 + (pRC->iParameterPollingArrayIndex / RIGCTRL_MAX_FIXED_EDGES_PER_FREQ_RANGE );
                          iEdge  = 1 + (pRC->iParameterPollingArrayIndex % RIGCTRL_MAX_FIXED_EDGES_PER_FREQ_RANGE );
                          if( (iRange<=pRC->iScopeNumFreqRanges) && (iEdge<=pRC->iScopeNumEdgesPerFreqRange) )
                           { // For example, the IC-R8600 only has ONE "range" and ONE "edge" in that range.
                             // This 'if' will limit the number of "questions" we send to the radio.
                             sprintf(szComment, "edge[%d][%d] ?", iRange, iEdge );
                             RigCtrl_IntToBCD_CIV_LSByteFirst( iRange, &b1, 2/*nDigits*/ );
                             RigCtrl_IntToBCD_CIV_LSByteFirst( iEdge,  &b2, 2/*nDigits*/ );
                             ++pRC->iParameterPollingArrayIndex; // next "Fixed edge frequency setting"
                             RigCtrl_SendShortCIVMessage( pRC, RIGCTRL_PORT_RADIO,
                               RIGCTRL_MSGTYPE_SPECTRUM_CONFIG,
                               szComment, // comment for traffic log
                               0x27,  // iCmd   : "Scope waveform data"
                               0x1E,  // iSubCmd: "read the Fixed edge frequency settings"
                                 b1,  // 1st data byte : "frequency range" (01..13, guess this is BCD, not hex)
                                 b2); // 2nd data byte : "edge number) (01..03, guess this is also meant as BCD)
                             pRadioPortInstance->dwExpectedResponse_CmdAndSubcode = RCTL_COMBINE_CMD_AND_SUB(0x27,0x1E);
                             pRC->iResponseCountdown_ms = pRC->iMaxResponseDelay_ms; // now waiting for the radio to respond after asking for a "Fixed Edge Frequency"
                             // When deliberately exceeding the number of "scope frequency ranges"
                             //  ( 13 for an IC-7300 ), the radio responded with "Not Good" :
                             // > 23:23:00.0 TX 009 FE FE 94 E0 27 1E 14 01 FD (scope)    ; edge[14][1] ?
                             // > 23:23:00.1 RX 006 FE FE E0 94 FA FD          (Not OK)
                             // If this happens, RigCtrl_ParseCIV(case 0x271E) limits
                             //    pRC->iScopeNumFreqRanges and pRC->iScopeNumEdgesPerFreqRange
                             //  to what the radio *really* supports.
                             // This seems to be the only method to avoid hard-coding the number,
                             // because other transceivers (like the IC-7851) have LESS frequency ranges,
                             // and future devices (like HF/VHF/UHF/SHF radios) may have MORE .
                           } // end if < valid frequency-RANGE and -EDGE > ?
                          else
                           { // Will waste about 50 ms until the next attempt. So what ?
                           }
                        }
                       break;
#                 else  // skip reading all "Fixed edge frequency settings" :
                    case 5 :
                       break;
#                 endif // (0,1)
                    case 6  : // try to make the scope FASTER VIA SERIAL PORT
                        // (even when "Waterfall" set to FAST, an IC-7300
                        //  only delivered 4 new spectra per second.
                        //  Note the confusing existance of a ..
                        //   "Scope Sweep speed" (Icom cmd 0x27 0x1A),
                        //   "Waterfall speed"   (Icom cmd 0x1A 0x05 XX XX, the "XX XX" is MODEL DEPENDENT)
                        //   "Scroll Speed"      (IC-9700: cmd 0x1A 0x05 followed by "0166" in BCD) !
                       RigCtrl_SetScopeSpeed( pRC, pRC->iScopeSpeed, RIGCTRL_PORT_RADIO );
                       break;
                    case 7  : // start or stop periodic transmission of spectrum data (which Icom calls "waveform")
                       RigCtrl_StartStopSpectrumData( pRC, TRUE, RIGCTRL_PORT_RADIO );
                       break;
                    default :
                       // ex: pRC->iParameterPollingState = RIGCTRL_POLLSTATE_DONE;
                       RigCtrl_EnterState_Done( pRC );
                       pRC->iParameterPollingSubState = 0;
                       break;
                  } // end switch( pRC->iParameterPollingSubState )
               }
#            if( SWI_SUPPORT_YAESU_5_BYTE_CAT )
              else if( pRadioPortInstance->iRadioCtrlProtocol == RIGCTRL_PROTOCOL_YAESU_5_BYTE ) // Configure spectrum with Yaesu's OLD UGLY CAT ? Never ever.
               { RigCtrl_EnterState_Done( pRC ); // .. try your luck with the next 'pollable item group'
               }
#            endif // SWI_SUPPORT_YAESU_5_BYTE_CAT ?
              else // none of the above ?
               { // reading spectrum data not supported here !
                 // ex: pRC->iParameterPollingState = RIGCTRL_POLLSTATE_DONE;
                 RigCtrl_EnterState_Done( pRC );
               }
            }
           else  // don't WANT spectrum data, or the rig doesn't support that -> turn spectrum data off
            { RigCtrl_StartStopSpectrumData( pRC, FALSE, RIGCTRL_PORT_RADIO );
              // ex: pRC->iParameterPollingState = RIGCTRL_POLLSTATE_DONE;
              RigCtrl_EnterState_Done( pRC );
             }
           break; // end case RIGCTRL_POLLSTATE_CONFIGURE_SPECTRUM
        case RIGCTRL_POLLSTATE_PASSIVE : // nothing else to do.. or not open for business
        case RIGCTRL_POLLSTATE_DONE    : // finished polling "all we need to know" about the radio
           // Is the RADIO CONTROL PORT 'free' to send another command ?
           if( RigCtrl_IsPortBusy( pRadioPortInstance ) )
            { // Cannot send another "postponed" or "queued-up" command now,
              // because the RADIO CONTROL PORT is still busy from something.
            }
           else if( pRC->iPostponedSetMessages & (1<<RIGCTRL_MSGTYPE_SET_FREQUENCY) )
            { RigCtrl_SetVFOFrequency_Internal( pRC, pRC->dblPostponedFrequency );
              pRC->dblVfoFrequency = pRC->dblPostponedFrequency;
              // ex: pRC->iPostponedSetMessages &= ~(1<<RIGCTRL_MSGTYPE_SET_FREQUENCY); // "done"
              RigCtrl_ClearPostponedSetMessageFlag( pRC,RIGCTRL_MSGTYPE_SET_FREQUENCY); // "done"
            }
           else if( pRC->iPostponedSetMessages & (1<<RIGCTRL_MSGTYPE_OP_MODE) )
            { RigCtrl_SetOperatingMode_Internal( pRC, pRC->iPostponedOpMode );
              pRC->iOpMode = pRC->iPostponedOpMode;
              RigCtrl_ClearPostponedSetMessageFlag( pRC,RIGCTRL_MSGTYPE_OP_MODE); // "done"
            }
           else if( pRC->iPostponedSetMessages & (1<<RIGCTRL_MSGTYPE_SPECTRUM_CONFIG) )
            { RigCtrl_StartStopSpectrumData( pRC,
                (pRC->dwRigControlFlags & RIGCTRL_FLAG_WANT_SPECTRUM_DATA ) && (pRC->iCapabilities & RIGCTRL_CAPS_SPECTRUM_VIA_CAT),
                 RIGCTRL_PORT_RADIO );
              RigCtrl_ClearPostponedSetMessageFlag( pRC,RIGCTRL_MSGTYPE_SPECTRUM_CONFIG); // "done"
            }
           else if( pRC->iPostponedSetMessages & (1<<RIGCTRL_MSGTYPE_SCOPE_SPAN) )
            { RigCtrl_SetScopeSpan_Internal( pRC );
              // ex: pRC->iPostponedSetMessages &= ~(1<<RIGCTRL_MSGTYPE_SCOPE_SPAN); // "done"
              RigCtrl_ClearPostponedSetMessageFlag( pRC,RIGCTRL_MSGTYPE_SCOPE_SPAN); // "done"
            }
           else if( pRC->iReadWritePNFifoHead != pRC->iReadWritePNFifoTail )  // other 'read'- or 'write'-command pending ?
            { RigCtrl_SendReadWriteCommandFromFIFO( pRC );
            }
           else if( pRC->iParameterPollingState == RIGCTRL_POLLSTATE_DONE )
            { // nothing else to do ... time for some 'periodic polling'  :
              if( pRC->iTransmitting <= 0 ) // currently NOT TRANSMITTING -> "scope data" should be arriving from a modern rig...
               { if( pRC->iScopeDataCountdown_ms >= 0 )
                  {  pRC->iScopeDataCountdown_ms -= 50; // countdown for 'Scope'-data ...
                  }
                 else
                  { // Reception of scope waveform data has timed out -> kick it alive again.
                    // Since we're in state RIGCTRL_POLLSTATE_DONE here, there's
                    // nothing else to do, so :
                    if( (pRC->dwRigControlFlags & RIGCTRL_FLAG_WANT_SPECTRUM_DATA ) && (pRC->iCapabilities & RIGCTRL_CAPS_SPECTRUM_VIA_CAT) )
                     { // Mr Scope NOT alive but he SHOULD be ...
                       strcpy( szComment, "Remote spectrum timeout, trying to kick it alive again" );
                       strcpy( pRC->sz255LastError, szComment );
                       RigCtrl_AddToTrafficLog( pRC, RIGCTRL_PORT_RADIO, RIGCTRL_ORIGIN_CONTROLLER,
                            NULL/*pbMessage*/, 0/*iMsgLength*/, RIGCTRL_MSGTYPE_FLAG_ERROR, szComment );
                       if( RigCtrl_TrafficMonitor.iDisplayOptions & RIGCTRL_TMON_DISPLAY_OPTION_PAUSE_ON_ERROR )
                        { if( ! pRadioPortInstance->fTrafficMonitorPausedOnTrigger )
                           { char sz255[256];
                             sprintf( sz255, "Traffic Log paused on %s after SPECTRUM TIMEOUT",
                                RigCtrl_GetDescriptivePortName(pRadioPortInstance) ); // <- e.g. "Radio Control Port", or even "COM5:IC-7300"
                             RigCtrl_AddToTrafficLog( pRC, RIGCTRL_PORT_RADIO, RIGCTRL_ORIGIN_CONTROLLER,
                                NULL/*pbMessage*/, 0/*iMsgLength*/, RIGCTRL_MSGTYPE_FLAG_ERROR, sz255 );
                           }
                          pRadioPortInstance->fTrafficMonitorPausedOnTrigger = TRUE;
                        } // end if < RIGCTRL_TMON_DISPLAY_OPTION_PAUSE_ON_ERROR .. >

                       // The trouble started when WSJT-X switched an IC-7300 to TRANSMIT,
                       // causing it to stop the periodic 'Scope Waveform' transmissions:
                       // >  RX 008 FE FE E0 94 1C 00 01 FD (other) ; TRXStatus:1 [WSJT-X to SpecLab]
                       // >  t2 008 FE FE E0 94 1C 00 01 FD (other) ; TRXStatus:1 [SpecLab to IC7300]
                       //   ... other messages, here irrelevant. A second later:
                       // >  Remote spectrum timeout, trying to kick it alive again .
                       // Of course a hopeless case during TX .. and we don't want
                       //    to reconfigure the IC-7300 to have it sending scope data
                       //    during TRANSMISSION (possible, but meaningless, because
                       //    the waterfall would still stop in the preferred FIXED mode).
                       // Cure: The CI-V parser must keep track of all messages
                       //       that may cause the radio to TRANSMIT .
                       //       While the radio TRANSMITS, the scope-data-timeout
                       //       must be IGNORED (no attempt to "kick it alive" again).

                       pRC->iParameterPollingState = RIGCTRL_POLLSTATE_CONFIGURE_SPECTRUM;
                       // Got here, for example, when closing Icom's "RS-BA1"
                       //  because the damned thing sends 27 11 00 (hex) to STOP spectrum data.
                       //  See details in RigCtrl_ParseCIV(), case 0x27 .
                     }
                    pRC->iScopeDataCountdown_ms = RIGCTRL_SPECTRUM_TIMEOUT_MS; // stay calm for a short while, when "trying to kick the spectrum alive again" (*)
                    // Consider this: The "local operator" may have entered a menu in the IC-7300,
                    //  which possibly stops the transmission of spectra when they don't appear on radio's own screen.
                    //  We don't want to intefere immediately by bombarding the radio
                    //  with commands to turn the spectrum/waterfall on again.
                    //  The same applies to reloading pRC->iScopeDataCountdown_ms while TRANSMITTING,
                    //  which (in most cases, because that's also CONFIGURABLE) also stops the IC-7300's spectrum.

                  } // end if < reception of "waveform data" timed out >
               }   // end if < not TRANSMITTING, so spectra should pour in >
              else // pRC->iTransmitting > 0  ->
               { pRC->iScopeDataCountdown_ms = RIGCTRL_SPECTRUM_TIMEOUT_MS; // reload 'Scope waveform' timer (for timeout monitoring) while TRANSMITTING
               }
            }    // end else < no "postponed" messages to send >
           break; // end case RIGCTRL_POLLSTATE_PASSIVE / _DONE
        case RIGCTRL_POLLSTATE_FIND_VFO_FOR_FREQ: // special state for e.g. IC-9700 to "find" the suitable VFO for a given band/frequency
           if( pRadioPortInstance->iResponseCountdown_ms <= 0 ) // not busy sending anything else / not waiting for a response anymore ?
            { // obviously timed out without a response, so let RigCtrl_FindVfoForNewFrequency() try something else:
              RigCtrl_FindVfoForNewFrequency( pRC, -1/*iMsgType*/ ); // here: called from RigCtrl_Handler() after "enough time has passed to send the next command.."
            }
           break;
        case RIGCTRL_POLLSTATE_AFTER_VFO_SWITCH: // read certain parmameters again after e.g. exchanging "VFO A" and "VFO B"
           if( pRadioPortInstance->iResponseCountdown_ms <= 0 ) // not busy sending anything else / not waiting for a response anymore ?
            { RigCtrl_UpdateParamsAfterVfoSwitch( pRC ); // here: called from RigCtrl_Handler() after "enough time has passed to send the next command.."
            }
           break; // end case RIGCTRL_POLLSTATE_AFTER_VFO_SWITCH
        case RIGCTRL_POLLSTATE_TURN_RIG_ON : // (try to) turn the rig on via special command or "wake-up pattern"
           if( pRadioPortInstance->iResponseCountdown_ms <= 0 ) // not busy sending anything else / not waiting for a response anymore ?
            { RigCtrl_TurnPowerOnOrOff( pRC, TRUE/*fPowerOn*/ );
              // '--> sends the 'turn on' command and switches to RIGCTRL_POLLSTATE_WAIT_TURN_ON
              // The message (Icom's command 0x18, sub-command 0x01 = "turn on")
              // is on it's way, and will hopefully be answered with "FB" = "Fine Business":
              // > 19:26:28.3 TX 09B FE FE FE FE FE .. ; turn ON
              // > 19:26:28.3 RX 006 FE FE E0 94 FB FD ; 'OK' for cmd 0x1801
            }
           break; // end case RIGCTRL_POLLSTATE_TURN_RIG_ON
        case RIGCTRL_POLLSTATE_WAIT_TURN_ON: // waiting for the rig to turn on, after RIGCTRL_POLLSTATE_TURN_RIG_ON ..
           if( pRadioPortInstance->iResponseCountdown_ms <= 0 ) // been waiting for long enough (transceiver boot-up time)
            { pRadioPortInstance->dwExpectedResponse_CmdAndSubcode = RIGCTRL_NOVALUE_INT; // not waiting for any response now ..
              pRadioPortInstance->iExpectResponseForUnifiedPN = -1; // .. and not asking for a certain UNIFIED PARAMETER NUMBER yet
              pRadioPortInstance->iResponseCountdown_ms = 500; // .. only WAIT a bit longer before starting to read e.g. the "radio ID" in the next state
              pRC->iParameterPollingState = RIGCTRL_POLLSTATE_START_RD;
            }
           break; // end case RIGCTRL_POLLSTATE_WAIT_TURN_ON
        case RIGCTRL_POLLSTATE_TURN_RIG_OFF: // (try to) turn the rig on via special command (for modern Icom rigs)
           if( pRadioPortInstance->iResponseCountdown_ms <= 0 ) // not busy sending anything else / not waiting for a response anymore ?
            { RigCtrl_TurnPowerOnOrOff( pRC, FALSE/*!fPowerOn*/ );
              // '--> sends the 'turn off' command and switches to RIGCTRL_POLLSTATE_WAIT_TURN_ON
              // The message (Icom's command 0x18, sub-command 0x01 = "turn on")
              // is on it's way, and will hopefully be answered with "FB" = "Fine Business":
              // > 19:26:28.3 TX 09B FE FE FE FE FE .. ; turn ON
              // > 19:26:28.3 RX 006 FE FE E0 94 FB FD ; 'OK' for cmd 0x1801
              // The equivalent YAESU CAT command does NOT send a response !
            }
           break; // end case RIGCTRL_POLLSTATE_TURN_RIG_OFF
        case RIGCTRL_POLLSTATE_WAIT_TURN_OFF: // waiting for the rig to turn off, after RIGCTRL_POLLSTATE_TURN_RIG_OFF ..
           if( pRadioPortInstance->iResponseCountdown_ms <= 0 ) // been waiting for long enough (transceiver turn-off time)
            { pRadioPortInstance->dwExpectedResponse_CmdAndSubcode = RIGCTRL_NOVALUE_INT; // not waiting for a response anymore ..
              pRadioPortInstance->iExpectResponseForUnifiedPN = -1; // .. also not asking for a UNIFIED PARAMETER NUMBER when the radio is OFF ..
              pRadioPortInstance->iResponseCountdown_ms = 500; // .. "do nothing" for at least 500 ms in RIGCTRL_POLLSTATE_TURNED_OFF
              pRC->iParameterPollingState = RIGCTRL_POLLSTATE_TURNED_OFF;
            }
           break; // end case RIGCTRL_POLLSTATE_WAIT_TURN_OFF
        case RIGCTRL_POLLSTATE_TURNED_OFF :  // guess the rig HAS TURNED ITSELF OFF (after RIGCTRL_POLLSTATE_TURN_RIG_OFF)
           break; // end case RIGCTRL_POLLSTATE_TURNED_OFF
        default:
           break;
      } // end switch( pRC->iParameterPollingState )
   }   // end if < enough time passed to send the next command >

  (void)pszComment; // "assigned a value that was never used" .. oh, shut up (it's better than reading a non-initialized variable)



} // end RigCtrl_Handler()


//----------------------------------------------------------------------------
void RigCtrl_OnPollingStateChangeInSubmodule( T_RigCtrlInstance *pRC,
                          int iPollingStateFromSubmodule )
  // Called from RigCtrl_Handler() whenever the 'protocol specific handler'
  // (e.g. Yaesu5Byte_Handler() ) is in a different PARAMETER POLLING STATE
  // than module RigControl.c itself .
  // For example, RigControl.c may be in RIGCTRL_POLLSTATE_PARAMS_FOR_RIGCTL
  //   while Yaesu5Byte.c still chews on RIGCTRL_POLLSTATE_WAIT_TURN_ON, etc.
{



  // Keep it simple; trust whatever the protocol-specific state machine thinks
  //                 is going on at the moment, and reflect that state here:
  pRC->iParameterPollingState = iPollingStateFromSubmodule;


} // end RigCtrl_OnPollingStateChangeInSubmodule()

//----------------------------------------------------------------------------
int RigCtrl_EnumerateRequiredPNs( int iZeroBasedIndex )
  // Called from RigControl.c shortly after 'connecting' the radio,
  // to poll certain parameters that REMOTE CLIENTS (like WSJT-X) often want,
  // like (roughly ordered by appearance by WSJT-X, anno 2025):
  //  RIGCTRL_PN_FREQUENCY,   RIGCTRL_PN_SEL_VFO_FREQUENCY (why the heck ALSO THAT?),
  //  RIGCTRL_PN_UNSEL_VFO_FREQUENCY, RIGCTRL_PN_UNSEL_VFO_OP_MODE,
  //  RIGCTRL_PN_TUNING_STEP (added because WFVIEW wanted to know this),
  //  RIGCTRL_PN_SPLIT_MODE,
  //  RIGCTRL_PN_RIT_FREQ,    RIGCTRL_PN_XIT_FREQ,
  //  RIGCTRL_PN_RIT_ENABLED, RIGCTRL_PN_XIT_ENABLED, end who-knows-what .
  // [in]     : iZeroBasedIndex = the "enumerator" (zero-based index as in C).
  //            Incremented by the caller from zero, stepping by one,
  //            until the RETURN VALUE from HLSRV_EnumerateRequiredPNs() is zero.
  // [return] : "Unified parameter number" for RigControl.c ,
  //            or ZERO when all 'required parameter numbers' are through.
{ switch( iZeroBasedIndex )
   { // Begin with parameters that ANY rig should support...
 //ex:case 0: return RIGCTRL_PN_TRANSCEIVER_ID; // first determine the "Transceiver ID" alias "Radio ID" alias "Default CI-V Address"
     case  0: return RIGCTRL_PN_FREQUENCY;
     case  1: return RIGCTRL_PN_OP_MODE;
     case  2: return RIGCTRL_PN_UNSEL_VFO_FREQUENCY; // <- special service for WSJT-X (and possibly other 'Hamlib'-based remote clients)
     case  3: return RIGCTRL_PN_UNSEL_VFO_OP_MODE;  //   "        "      "   ...
     case  4: return RIGCTRL_PN_SEL_VFO_OP_MODE;    //   "        "      "   ...
     case  5: return RIGCTRL_PN_TUNING_STEP;     // <- special service for WFVIEW (when WFVIEW docks on a RCW Keyer server-port)
     case  6: return RIGCTRL_PN_TRANSMITTING;    //  aka "PTT flag" (0=receive, 1=transmit)
     case  7: return RIGCTRL_PN_TRANSMIT_FREQ;
     case  8: return RIGCTRL_PN_SPLIT_MODE;
     case  9: return RIGCTRL_PN_NUM_TX_BANDS;
     case 10: return RIGCTRL_PN_AUDIO_VOLUME_PERCENT;
     case 11: return RIGCTRL_PN_RF_POWER_SETTING_PERCENT;
     case 12: return RIGCTRL_PN_RF_GAIN_PERCENT;
     case 13: return RIGCTRL_PN_SQUELCH_LEVEL_PERCENT;
     case 14: return RIGCTRL_PN_NOISE_BLANKER_PERCENT;
     case 15: return RIGCTRL_PN_NOISE_REDUCTION_PERCENT;
     case 16: return RIGCTRL_PN_PASSBAND_TUNING_POS1; // one fine day, will show this in the "filter passband" graphics..
     case 17: return RIGCTRL_PN_PASSBAND_TUNING_POS2;
     case 18: return RIGCTRL_PN_CW_PITCH_HZ;
     case 19: return RIGCTRL_PN_MIC_GAIN_PERCENT;
     case 20: return RIGCTRL_PN_KEYER_SPEED_WPM;
     case 21: return RIGCTRL_PN_NOTCH_POS_PERCENT;
     case 22: return RIGCTRL_PN_COMP_SETTING_PERCENT;
     case 23: return RIGCTRL_PN_BREAK_IN_DELAY_PERCENT;
     case 24: return RIGCTRL_PN_MONITOR_GAIN_PERCENT;
     case 25: return RIGCTRL_PN_VOX_GAIN_PERCENT;
     case 26: return RIGCTRL_PN_ANTI_VOX_GAIN_PERCENT;
     case 27: return RIGCTRL_PN_BRIGHTNESS_PERCENT;
     case 28: return RIGCTRL_PN_SCOPE_REF_LEVEL;
     case 29: return RIGCTRL_PN_SCOPE_SPEED;
     case 30: return RIGCTRL_PN_WATERFALL_SPEED;

     // Besides the above "level settings" / "potentiometer-like settings", "speeds", etc,
     // also read out a dozen of "ON" / "OFF" settings .. mostly CI-V command 0x16:
     case 31: return RIGCTRL_PN_PREAMP_SETTING;   // CI-V : 0x16 0x02  0=internal+external OFF, 1=internal, 2=external, 3=BOTH preamps on
     case 32: return RIGCTRL_PN_AGC_SPEED;        // CI-V : 0x16 0x12  1=fast 2=mid 3=slow  . Other radios: 0 = off (MANUAL gain control)
     case 33: return RIGCTRL_PN_NOISE_BLANKER_ON; // CI-V : 0x16 0x22  0=off  1=on
     case 34: return RIGCTRL_PN_NOISE_REDUCTION_ON; // CI-V : 0x16 0x40  0=off  1=on
     case 35: return RIGCTRL_PN_AUTO_NOTCH_ON   ; // CI-V : 0x16 0x41  0=off  1=on
     case 36: return RIGCTRL_PN_RPTR_TONE_ON    ; // CI-V : 0x16 0x42  0=off  1=on
     case 37: return RIGCTRL_PN_TONE_SQUELCH_ON ; // CI-V : 0x16 0x43  0=off  1=on
     case 38: return RIGCTRL_PN_COMP_ON         ; // CI-V : 0x16 0x44  0=off  1=on
     case 39: return RIGCTRL_PN_MONITOR_ON      ; // CI-V : 0x16 0x45  0=off  1=on
     case 40: return RIGCTRL_PN_VOX_ON          ; // CI-V : 0x16 0x46  0=off  1=on
     case 41: return RIGCTRL_PN_BK_IN_MODE      ; // CI-V : 0x16 0x47  0=off 1=semi-BK 2=full-BK
     case 42: return RIGCTRL_PN_MANUAL_NOTCH_ON ; // CI-V : 0x16 0x48  0=off  1=on
     case 43: return RIGCTRL_PN_MANUAL_NOTCH_WIDTH; // CI-V : 0x16 0x57  0=wide 1=mid 2=narrow
     case 44: return RIGCTRL_PN_DIAL_LOCK_ON    ; // CI-V : 0x16 0x50  0=off  1=on
     case 45: return RIGCTRL_PN_DSP_SHARP_SOFT  ; // CI-V : 0x16 0x56  0=sharp 1=soft
     case 46: return RIGCTRL_PN_SSB_TX_BANDWIDTH; // CI-V : 0x16 0x58  0=wide 1=mid 2=narrow (the bandwidths are configurable in an extra commands)
     case 47: return RIGCTRL_PN_SUBBAND_ON;       // CI-V : 0x16 0x59  0=off  1=dual watch ON
     case 48: return RIGCTRL_PN_SATELLITE_MODE;   // CI-V : 0x16 0x5A  0=off  1=on
       // Removed from the above group (CI-V cmd 0x16 ) :
  // case ..: return RIGCTRL_PN_TWIN_PEAK_FILTER_ON; // CI-V : 0x16 0x4F  0=off  1=on (considered too esoteric)

     // Even though the following parameter use the ugly MODEL-SPECIFIC 4-digit sub-subcodes of command 0x1A 0x05,
     // TRY TO(!) read those settings, because they spoiled the operation of the
     // REMOTE CW KEYER more than once -- so try to read them, so we can list,
     // and let the operator CORRECT them if necessary on RCWKeyer's "Debug" tab:
     case 49: return RIGCTRL_PN_REFERENCE_FREQ_OFFSET;   // .iReferenceFrequencyOffset.    IC-7300: 0x1A 0x05 0058
     case 50: return RIGCTRL_PN_AF_IF_OUT_SELECTOR;      // .iAF_IF_OutputSelectOnACC_USB. IC-7300: 0x1A 0x05 0059
     case 51: return RIGCTRL_PN_SIDETONE_ON_USB_AUDIO;   // .iSidetone_on_USB_Audio .      IC-7300: 0x1A 0x05 0062
     case 52: return RIGCTRL_PN_SQUELCH_FOR_AF_ON_USB;   // "Squelch function for the AF signal output to ACC/USB" (heavens, no, TURN THIS OFF when set!)
     case 53: return RIGCTRL_PN_MOD_INPUT_LEVEL_FROM_ACC; // "MOD input level from ACC" (0..255 for 100%)
     case 54: return RIGCTRL_PN_MOD_INPUT_LEVEL_FROM_USB; // "MOD input level from USB" (0..255 for 100%)
     case 55: return RIGCTRL_PN_MOD_IN_CONN_FOR_DATA_OFF; // "MOD input connector during DATA OFF" (0=MIC, 1=ACC, 2=MIC/ACC, 3=USB)
     case 56: return RIGCTRL_PN_MOD_IN_CONN_FOR_DATA;     // "MOD input connector during DATA" (0=MIC, 1=ACC, 2=MIC/ACC, 3=USB)

     default: break;
     // Note: Besides the list of parameters to read in RigCtrl_EnumerateRequiredPNs(),
     //      there's a similar list of parameters to read in HLSRV_EnumerateRequiredPNs().
   }
  return 0;
} // RigCtrl_EnumerateRequiredPNs()

//--------------------------------------------------------------------------
void RigCtrl_SetTransmitRequest( T_RigCtrlInstance *pRC, int iTransmitReqst )
  // Called whenever THIS END ("the keyer or its operator") wants to transmit;
  //          maybe because there is SOMETHING TO SEND (automatic transmit)
  //          or because the MANUAL PTT INPUT (e.g. footswitch) was activated;
  //          or (when operating as SERVER) a remote client with 'TX permission'
  //          has asked this server to start transmitting (Hamlib: "set_ptt 1\n").
  // Warning: This function may be called from a WORKER THREAD,
  //          e.g. KeyerThread.c : KeyerThread(), and thus it interrupts
  //          RigCtrl_Handler().  To make this sufficiently thread safe
  //          without a synchronisation object, and without waiting for anything,
  //          pRC->fTransmitting is ONLY WRITTEN HERE, and READ-ONLY elsewhere.
  // See also / "related functions" : RigCtrl_SetTransmitRequest(), CwNet_SwitchTransmittingClient() .
{
  if( pRC->iTransmitReqst != iTransmitReqst )
   { pRC->iTransmitReqst = iTransmitReqst;  // <- good place for a breakpoint (*)
     pRC->iTransmitReqstCountdown_ms = 500; // <- .. to avoid 'misinterpreting' pRC->iTransmitting vs pRC->iTransmitReqst in ()
     RigCtrl_QueueUpCmdToWriteUnifiedPN( pRC, RIGCTRL_PN_TRANSMIT_REQUEST );
     // What else to do on a STATE TRANSITION of the 'PTT flag' aka 'Transmit Request' ?
     // (*) In the Remote CW Keyer application, the PTT flag was switched by...
     //    - on "manual PTT control", client side, when called from
     //      KeyerThread() when CwKeyer_GetDigitalInput(CwKeyer_Config.iManualPTTInput)
     //      indicated a transition on the 'manual PTT input' (microphone, footswitch, etc)
     //    - when NOT using manual PTT control:
   } // end if( pRC->iTransmitReqst != fTransmitting )

} // end RigCtrl_SetTransmitRequest()


//--------------------------------------------------------------------------
void RigCtrl_TurnPowerOnOrOff( T_RigCtrlInstance *pRC, BOOL fPowerOn )
{
  T_RigCtrl_PortInstance *pPortInstance = &pRC->PortInstance[RIGCTRL_PORT_RADIO];

  switch( pPortInstance->iRadioCtrlProtocol )
   { case RIGCTRL_PROTOCOL_ICOM_CI_V :
        // Turning an Icom radio ON (when it's really turned off) requires
        // a special pattern, with the 0xFE preamble repeated MANY times :
        if( fPowerOn )
         { RigCtrl_SendCommandToTurnRigOn_CIV( pRC, RIGCTRL_PORT_RADIO, RIGCTRL_ORIGIN_CONTROLLER );
           pPortInstance->dwExpectedResponse_CmdAndSubcode = RCTL_COMBINE_CMD_AND_SUB(0x18,0x01); // IC-7300 responded with the 0xFB-"OK" to the 'turn on' command (cmd 0x18 sub 0x01) !
           pPortInstance->iResponseCountdown_ms = 500; // wait for the response ("FB") for the "turn on" command just sent
           pRC->iParameterPollingState = RIGCTRL_POLLSTATE_WAIT_TURN_ON; // now waiting for an "OK"-response after the command to "turn on"
         }
        else // Turning a radio OFF is a "normal CI-V" command (cmd 0x18 + sub-cmd 0x00)
         { RigCtrl_SendShortCIVMessage( pRC, RIGCTRL_PORT_RADIO, RIGCTRL_ORIGIN_CONTROLLER,
              RIGCTRL_MSGTYPE_FLAG_CIV | RIGCTRL_MSGTYPE_POWER_ON_OFF, "turn OFF",
              0x18,  // [in] CI-V main command
              0x00,  // [in] sub-command, here: 0x00 to turn OFF
              -1,    // [in] optional 1st parameter byte. -1 to omit.
              -1);   // [in] optional 2nd parameter byte. -1 to omit.
           pPortInstance->dwExpectedResponse_CmdAndSubcode = RCTL_COMBINE_CMD_AND_SUB(0x18,0x00); // IC-7300 responded with the 0xFB-"OK" to the 'turn off' command (cmd 0x18 sub 0x00) !
           pPortInstance->iResponseCountdown_ms  = 500; // not waiting for a response, but 'busy' for at least 500 ms
           pRC->iParameterPollingState = RIGCTRL_POLLSTATE_TURNED_OFF;
         }
        break;

#  if( SWI_SUPPORT_YAESU_5_BYTE_CAT )
     case RIGCTRL_PROTOCOL_YAESU_5_BYTE: // Turning an old FT8x7 on/off with Yaesu's OLD UGLY CAT .. Meow ! (crash, bang, boom, calibration lost, hi)
        Yaesu5Byte_TurnPowerOnOrOff( &pRC->Y5B, fPowerOn );    // Note: We won't receive a RESPONSE for this !
        pRC->iParameterPollingState = fPowerOn ? RIGCTRL_POLLSTATE_WAIT_TURN_ON
                                               : RIGCTRL_POLLSTATE_WAIT_TURN_OFF;
        pPortInstance->dwExpectedResponse_CmdAndSubcode = RIGCTRL_NOVALUE_INT; // FT-8*7 will NOT respond to any of these commands
        pPortInstance->iResponseCountdown_ms  = 500; // not waiting for a response, but 'busy' for at least 500 ms
        break;
#  endif // SWI_SUPPORT_YAESU_5_BYTE_CAT ?
     default:  // no "Power On/Off"-support for other radios yet
        break;
   } // end switch( pRC->iRadioCtrlProtocol )
} // end RigCtrl_TurnPowerOnOrOff()

//--------------------------------------------------------------------------
static BOOL RigCtrl_SetVFOFrequency_Internal( T_RigCtrlInstance *pRC,
                              double dblFreqHz ) // [in] freq in HERTZ
  // As most functions, returns TRUE when successful, otherwise FALSE.
  // Returns BEFORE the remote rig has actually 'digested' the new setting !
  //
  //
  // About the TRAFFIC LOG: See RigCtrl_SendAndLogMessage() [called from here].
  //
  // 2024-12: After QSYing via the radio's own VFO knob, the RCWK-SERVER
  //          seemed to 'undo' the change in frequency immediately. Got here
  //          with the following call stack :
  //          ServerThread() -> CwNet_OnReceive()
  //           -> CwNet_ExecuteCmd()
  //            -> RigCtrl_OnVfoReportFromNetwork()
  //             -> RigCtrl_SetVFOFrequency_Internal() with the ORIGINAL frequency.
  //          The real culprit wasn't the RCWK-Server but the instance running
  //          as CLIENT - see transmission of CWNET_CMD_FREQ_REPORT in CwNet.c,
  //          which in fact sends a T_RigCtrl_VfoReport from CLIENT to SERVER.
  //          The chaos was even visible on the 'Debug' tab in the client:
  //           > 18:45:26.9 TX0 Set Freq = <some bizarre frequency>
  //          Fixed for the CLIENT SIDE in CwNet.c : CwNet_OnPoll().
  //   Grep for 'Request to "QSY" from this client to the remote server ?' ...
  //
{
  T_RigCtrl_PortInstance *pPortInstance = &pRC->PortInstance[RIGCTRL_PORT_RADIO];
  char sz16FreqAsText[44], szComment[ RIGCTRL_MAX_COMMENT_LENGTH+4 ];
  char *pszLowDigit;
  BYTE b80TxBuffer[84];
  BYTE b;
  int  iTxLength = 0;
  int  i,n;
  int  iRigCtrlPort = RIGCTRL_PORT_RADIO;
  int  iRigCtrlOrigin = RIGCTRL_ORIGIN_CONTROLLER;
  int  iMsgType =  RIGCTRL_MSGTYPE_SET_FREQUENCY | RIGCTRL_MSGTYPE_FLAG_TX;
  BOOL fOk = FALSE;

#ifdef __BORLANDC__  // "... is assigned a value that is never used".. oh shut up, we know what we're doing !
  (void)fOk;
#endif


  // convert frequency in Hertz into a string: 10 digits without fraction,
  // LEADING ZEROES, ok for up to 9 GHz ...
  sprintf(sz16FreqAsText, "%010.0lf", (double)dblFreqHz );
     //    '--> BCB6 crashed shortly after this (*) due to a stack corruption,
     //         when dblFreqHz was 8.1209blablablablabla509E+180 (!).
     // Call stack (cannot be trusted) :
     //   CwNet.c : ServerThread() -> CwNet_OnReceive() -> CwNet_ExecuteCmd()
     //              -> RigCtrl_OnVfoReportFromNetwork() [ok, realistic so far..]
     //               -> RigCtrl_SetVFOFrequency_Internal() .
  pszLowDigit = sz16FreqAsText + (strlen(sz16FreqAsText)-1); // now points to least significant digit

  switch( pPortInstance->iRadioCtrlProtocol )
   { case RIGCTRL_PROTOCOL_NONE :
     default:
         break;  // unknown protocol, can only send "direct" strings

     case RIGCTRL_PROTOCOL_ICOM_CI_V : // use Icom's "CI-V" protocol ...
         b80TxBuffer[iTxLength++] = 0xFE; // fixed preamble code for CI-V protocol  (2025-01: BCB6 crashed HERE with an access violation (*) )
         b80TxBuffer[iTxLength++] = 0xFE; // fixed preamble code for CI-V protocol
         b80TxBuffer[iTxLength++] = (BYTE)pPortInstance->iRadioDeviceAddr;  // "to" transceiver's address (0x48 = IC706)
         b80TxBuffer[iTxLength++] = (BYTE)pPortInstance->iRadioMasterAddr;  // "from" controller's address (often 0xE0=RIGCTRL_CIV_MASTER_ADDR_MIN)
         b80TxBuffer[iTxLength++] = 0x05; // CI-V command "Write operating frequency data"
         // Note: Older CI-V reference manuals [CIV_REF_MAN_V3_2] clearly mention that
         //       command 0x05 "writes operating frequency data into a displayed VFO
         //                     or memory channel", and that the radio responds
         //                     to THIS command with 0xFB = "Ok code"
         //                                  or with 0xFA = "NG code".
         //       Newer CI-V reference manuals don't specify for which commands
         //       the radio will respond with 0xFB / 0xFA, or not respond at all.
         //
         //       An IC-9700 didn't "accept" the new frequency if it was outside
         //       the CURRENTLY ACTIVE VFO'S BAND. Example:
         //          VFO 1 (upper panel): 1298.350 MHz,  "MEMO 29", selected (white)
         //          VFO 2 (lower panel):  144.050 MHz,  "MEMO 2",  not selected (gray)
         //        * trying to send dblFreqHz = 439.250e6 : NO EFFECT !
         //
         // If the frequency is below 100 MHz, may have to use 4 BCD-bytes only,
         // because some old radios like the IC-735 only support 4 bytes = 8 digits.
         // Modern radios (and VHF/UHF rigs) support 5 BCD data bytes = 10 digits.
         // Furthermore, some radios (like the IC-706) NEED these long numbers,
         // because otherwise the 100-MHz-place remains UNCHANGED !
         n = (  pRC->iDefaultAddress==RIGCTRL_DEF_ADDR_IC_725
             || pRC->iDefaultAddress==RIGCTRL_DEF_ADDR_IC_735
             || pRC->iDefaultAddress==RIGCTRL_DEF_ADDR_IC_751A ) ? 4 : 5;
         for(i=0; (i<n) && (pszLowDigit>=sz16FreqAsText); ++i)
          { b =  (BYTE)( (*pszLowDigit--)-'0');     // lowest digit, beginning at 1 Hz
            b |= (BYTE)(((*pszLowDigit--)-'0')<<4); // next digit, beginning at 10 Hz
            b80TxBuffer[iTxLength++] = b;        // append next two digits to BCD data area
          }
         b80TxBuffer[iTxLength++] = 0xFD;        // CI-V code for "End Of Message"
         iMsgType |= RIGCTRL_MSGTYPE_FLAG_CIV; // <-- to decode the above later, in the message log
         break;  // end case <ICOM CI-V protocol>
#   if( SWI_SUPPORT_YAESU_5_BYTE_CAT )
     case RIGCTRL_PROTOCOL_YAESU_5_BYTE: // Yaesu's OLD UGLY CAT - see Yaesu5Byte.c ..
         // Can't use RigCtrl_SendAndLogMessage(); use Yeasu5Byte.c instead,
         // because we may have to leave a GAP between THIS and the PREVIOUS BLOCK (Commands or Responses)
         // for this stoneage protocol, without delimiter bytes (aka "terminators"):
         fOk = Yaesu5Byte_SetVFOFrequency( &pRC->Y5B, dblFreqHz ); // [in] freq in HERTZ
         iTxLength = 0; // don't call RigCtrl_SendAndLogMessage() below, for reasons above :)
         break; // end case <Yaesu binary 5-byte-command protocol>
#   endif // SWI_SUPPORT_YAESU_5_BYTE_CAT ?

   } // end switch( pRC->iRadioCtrlProtocol )

  if( iTxLength>0 )
   {
     sprintf( szComment, "set freq to %.1lf Hz", (double)dblFreqHz );
     fOk = RigCtrl_SendAndLogMessage( pRC, iRigCtrlPort, iRigCtrlOrigin, b80TxBuffer, iTxLength,
              iMsgType/*ex:RIGCTRL_MSGTYPE_SET_FREQUENCY | RIGCTRL_MSGTYPE_FLAG_TX*/, szComment );
     if( fOk )
      { pPortInstance->dwExpectedResponse_CmdAndSubcode = RCTL_COMBINE_CMD_ONLY(0x05); // the radio's CI-V response "Set frequency" (cmd 0x05) doesn't contain command or subcode, so REMEMBER what has been "set" this way
        pPortInstance->iResponseCountdown_ms = pRC->iMaxResponseDelay_ms; // now waiting for the "OK"/"NOT OK" response for "Set frequency" (cmd 0x05)
        // '--> This may fail e.g. in an IC-9700, when the new frequency
        //      for the MAIN VFO (top line) is in the same band as
        //          the SECOND VFO (the one that can only RECEIVE, 2nd line).
        //      Reason: The IC-9700 (and possibly similar radios of that family)
        //              cannot simultaneously receive in TWO FREQUENCIES
        //              in the same band ! (Solution: grep for
        //      In that case, the CI-V traffic on the 'Debug' tab showed:
        // >  TX 00B FE FE 00 E0 05 00 00 05 44 01 FD ; set freq to 144050000.0 Hz
        // >  RX 006 FE FE E0 A2 FA FD                ; NotOK for 0x5000000
        //      At THIS point (in RigCtrl_SetVFOFrequency_Internal),
        //      fOk=TRUE only means the command to 'QSY' has been sent.
        //      The rest (including error handling when the remotely controlled
        //      rig responds with '0xFA' = 'NotOK') happens in RigCtrl_ParseCIV(),
        //      depending on pPortInstance->dwExpectedResponse_CmdAndSubcode
        //         -> RigCtrl_FindVfoForNewFrequency()
      }

   } // end if < something 'sendable' > ?

  return fOk;
} // end RigCtrl_SetVFOFrequency_Internal()


//--------------------------------------------------------------------------
BOOL RigCtrl_SetVFOFrequency( T_RigCtrlInstance *pRC, /* API */
                              double dblFreqHz ) // [in] new radio freq in HERTZ
  // As most functions, returns TRUE when successful, otherwise FALSE.
  // Returns BEFORE the remote rig has actually 'digested' the new setting,
  //         thus TRUE only means "the new setting is on its way" !
  //
  // To retrieve the current 'VFO frequency', simply read pRC->dblVfoFrequency .
  // To convert an 'audio frequency' (in the radio's demodulated output stream)
  //        into a 'radio frequency', see RigCtrl_AudioToRadioFrequency().
  //
  // Shall work on the server side (directly controlling a radio via 'COM' port)
  //        and on the client side (via CwNet.c, which polls the most important
  //              parameters and sends them 'when modified' in CwNet_OnPoll() )
  //
  // Callers in DL4YHF's "Remote CW Keyer" :
  //  (1) TCustomForm::MouseWheelHandler() -> ... -> TKeyerMainForm::FormMouseWheel()
  //       -> Ed_VFOKeyDown() -> RigCtrl_SetVFOFrequency()
  //  (2) CB_BandClick() [retrieves a "Band Stacking Register" from the multi-column combo]
  //       -> RigCtrl_SwitchToFreqMemEntry() -> RigCtrl_SetVFOFrequency() [+ others]
{
  T_RigCtrl_PortInstance *pPortInstance = &pRC->PortInstance[RIGCTRL_PORT_RADIO];
  // This API function may be called by the application at any time, but it may
  // have to be postponed (internally) when still busy from an 'uninterruptable'
  //      sequence of commands in RigCtrl_Handler().
  // In ANY case, immediately copy the new "wanted" VFO frequency :
  pRC->dblVfoFrequency = dblFreqHz;  // here: Modified by RigCtrl_SetVFOFrequency()
  if( pPortInstance->iResponseCountdown_ms > 0 ) // still waiting for a response -> "do not send anything at the moment"..
   {  // ex: pRC->iPostponedSetMessages |= (1<<RIGCTRL_MSGTYPE_SET_FREQUENCY);
      // Must not set any "Send-Postponed-Something"-flag when e.g. in "Liste-Only Mode",
      // thus use a function that is aware of such subtle details:
      RigCtrl_SetPostponedSetMessageFlag( pRC, RIGCTRL_MSGTYPE_SET_FREQUENCY );
      pRC->dblPostponedFrequency = dblFreqHz;
      return TRUE;  // cannot send AT THE MOMENT, but data are (almost) "on the way" !
   }
  else // no need to postpone the "Set Frequency"-command, so send immediately:
   {  // ex: pRC->iPostponedSetMessages &= ~(1<<RIGCTRL_MSGTYPE_SET_FREQUENCY);
      RigCtrl_ClearPostponedSetMessageFlag( pRC, RIGCTRL_MSGTYPE_SET_FREQUENCY );
      return RigCtrl_SetVFOFrequency_Internal( pRC, dblFreqHz );
      // Note: Even though an IC-7300 sent the "OK" (acknowledge) much later,
      //       the VFO frequency could be modified many times per second.
      //       Thus do NOT reload pRC->iResponseCountdown_ms
      //       in this case, even though we expect a response for the CI-V
      //       command that has just been sent. From SL's "CAT traffic" log:
      // > 18:08:39.5: TX 007 FE FE 94 E0 19 00 FD               (radio ID)    ; get transceiver ID
      // > 18:08:40.4: RX 008 FE FE E0 94 19 00 94 FD            (radio ID)    ; rig=IC-7300
      // > 18:08:40.4: TX 006 FE FE 94 E0 03 FD                  (VFO freq)    ; read VFO frequency
      // > 18:08:41.4: RX 00B FE FE E0 94 03 50 00 55 03 00 FD   (VFO freq)    ; 3550050.0 Hz
      // > 18:08:41.5: TX 006 FE FE 94 E0 04 FD                  (op mode)     ; read operating mode
      // > 18:08:42.4: RX 008 FE FE E0 94 04 03 03 FD            (op mode)     ; CW, filter=3
      // > 18:08:59.5: TX 00B FE FE 94 E0 05 60 00 55 03 00 FD   (VFO freq)    ; 3550060.0 Hz
      // > 18:09:00.3: RX 006 FE FE E0 94 FB FD                  (OK)
      // > 18:09:01.7: TX 00B FE FE 94 E0 05 70 00 55 03 00 FD   (VFO freq)    ; 3550070.0 Hz (step #1)
      // > 18:09:01.8: TX 00B FE FE 94 E0 05 80 00 55 03 00 FD   (VFO freq)    ; 3550080.0 Hz (step #2)
      // > 18:09:01.9: TX 00B FE FE 94 E0 05 90 00 55 03 00 FD   (VFO freq)    ; 3550090.0 Hz (step #3)
      // > 18:09:02.4: RX 006 FE FE E0 94 FB FD                  (OK)          (ack for step #1)
      // > 18:09:02.4: RX 006 FE FE E0 94 FB FD                  (OK)          (ack for step #2)
      // > 18:09:02.4: RX 006 FE FE E0 94 FB FD                  (OK)          (ack for step #3)
      // -> At least modern Icom radios seem to have large buffers
      //    for RECEPTION and TRANSMISSION. So for these *simple* commands,
      //     it's unnecessary to wait for a response (which would have wasted
      //     2.4 minus 1.7 = 0.7 seconds in the example shown above) .
   }

} // end RigCtrl_SetVFOFrequency()


//--------------------------------------------------------------------------
BOOL RigCtrl_SendCmd_ExchangeMainAndSubBand( T_RigCtrlInstance *pRC )
  // Internal subroutine, first called from RigCtrl_FindVfoForNewFrequency().
{
  T_RigCtrl_PortInstance *pPortInstance = &pRC->PortInstance[RIGCTRL_PORT_RADIO];
  BOOL fResult = FALSE;
  switch( pPortInstance->iRadioCtrlProtocol )
   { // which language does the radio *ON THIS PORT* speak ?
     case RIGCTRL_PROTOCOL_ICOM_CI_V : // Icom's "CI-V" protocol ...
        fResult = RigCtrl_SendShortCIVMessage( pRC, RIGCTRL_PORT_RADIO, RIGCTRL_ORIGIN_CONTROLLER,
                    RIGCTRL_MSGTYPE_OP_MODE,     // message type for traffic log
                       "Exchange MAIN/SUB band", // comment for traffic log
                       0x07,  // iCmd: CI-V command "Select the VFO mode" (quite misleading..)
                       0xB0,  // iSubCmd: "Exchange MAIN and SUB Bands" (see A7508-3EX-4 page 4)
                         -1,  // iDataByte1 : not used here
                         -1); // iDataByte2 : not used here
        pPortInstance->dwExpectedResponse_CmdAndSubcode = RCTL_COMBINE_CMD_AND_SUB(0x07,0xB0); // REMEMBER what has been "selected" this way
        pPortInstance->iResponseCountdown_ms = pRC->iMaxResponseDelay_ms; // now waiting for the "OK"/"NOT OK" response for "Exchange MAIN and SUB Bands"
        break;
     default: // Good luck implementing something similar in other "CAT languages" !
        break;
   } // end switch( pRC->iRadioCtrlProtocol )
  return fResult;
} // end RigCtrl_SendCmd_ExchangeMainAndSubBand()

//--------------------------------------------------------------------------
void RigCtrl_FindVfoForNewFrequency( T_RigCtrlInstance *pRC,
           int iMsgType ) // [in] : -1 when called from RigCtrl_Handler(),
                          //     or RIGCTRL_MSGTYPE_NOT_OK / RIGCTRL_MSGTYPE_OK
                          //     when called from RigCtrl_ParseCIV() .
  // Repeatedly called to fix the issue when an IC-9700 refused to switch
  // to a certain frequency when ANOTHER VFO
  //        (e.g. the "secondary VFO" which Icom calls "SUB Band")
  // was tuned to the same band. For hardware reasons, the IC-9700 cannot have
  // BOTH VFOs (the primary, transmit-capable VFO aka "MAIN Band" and
  // the secondary, receive-only VFO aka "SUB Band") on the same band.
  //
  //  [in] pRC->dblVfoFrequency : Frequency that WE WANTED to switch to
  //                              (but the 'computer said no', 0xFA..)
  //  [in,out] pRC->iParameterPollingState: here, RIGCTRL_POLLSTATE_FIND_VFO_FOR_FREQ on entry,
  //                               and  RIGCTRL_POLLSTATE_DONE when "done" .
  //  [in,out] pRC->iParameterPollingSubState : simple zero-based "step number"
  //              the value and meaning only plays a role within THIS FUNCTION.
{
  switch( pRC->iParameterPollingSubState )
   { case 0 : // This is the initial sub-state, set in RigCtrl_ParseCIV() when
       // the radio responded as in the following example (refused to set a VFO frequency):
       // > TX 00B FE FE 00 E0 05 00 00 05 44 01 FD    ; set freq to 144050000.0 Hz
       // > RX 006 FE FE E0 A2 FA FD                   ; NotOK for 0x5000000
       // From the 'Command Table' in the IC-9700 CI-V Reference Manual
       //    (A7508-3EX-4 at the time of this writing), page 4 of 28,
       // Cmd 0x07 "Select the VFO mode" with subcommand 0xB0 = "Exchange MAIN and SUB Bands"
       // appeared to be the easiest way (without overwriting other
       // settings in the radio), so first try this:
       pRC->iParameterPollingSubState = 1;
       RigCtrl_SendCmd_ExchangeMainAndSubBand( pRC );
       break; // end case iParameterPollingSubState == 0
     case 1 : // next call, usually from RigCtrl_ParseCIV() after receiving "Ok"/"Not OK" for the command sent above
       if( iMsgType == RIGCTRL_MSGTYPE_OK ) // the rig "agreed" to swap MAIN and SUB bands;
        { // so try again to switch to the desired new VFO frequency (possibly on a 'new band'):
          pRC->iParameterPollingSubState = 2;
          RigCtrl_SetVFOFrequency_Internal( pRC, pRC->dblVfoFrequency );
          // '--> next call of RigCtrl_FindVfoForNewFrequency() from RigCtrl_ParseCIV()
          //      after (hopefully) receiving a message with 0xFB -> iMsgType = RIGCTRL_MSGTYPE_OK.
        }
       else   // timeout ( iMsgType < 0 )   or  "NotOk" (RIGCTRL_MSGTYPE_NOT_OK) :
        { pRC->iParameterPollingSubState = 3; // ToDo: What else to try ?
        }
       break; // end case iParameterPollingSubState == 1
     case 2 : // next call, after swapping MAIN and SUB band, and sending the VFO frequency again:
       if( iMsgType == RIGCTRL_MSGTYPE_OK ) // Bingo.. the rig accepted the VFO frequency on the new "MAIN band"
        { // so leave RIGCTRL_POLLSTATE_FIND_VFO_FOR_FREQ :
          pRC->iParameterPollingState = RIGCTRL_POLLSTATE_DONE; // here: "gave up" in RigCtrl_FindVfoForNewFrequency()
          pRC->iParameterPollingSubState = 0;
          // When leaving this state here, messages on the Debug tab were:
          // > TX 00B FE FE 00 E0 05 10 00 05 44 01 FD ; set freq to 144050010.0 Hz
          // > RX 006 FE FE E0 A2 FA FD        ; refused to set 144050010.0 Hz
          // > TX 007 FE FE 00 E0 07 B0 FD            ; Exchange MAIN/SUB band
          // > RX 006 FE FE E0 A2 FB FD               ; 'OK' for cmd 0x7B00000
          // > TX 00B FE FE 00 E0 05 10 00 05 44 01 FD ; set freq to 144050010.0 Hz
          // > RX 006 FE FE E0 A2 FB FD               ; 'OK' for 'set freq'
          // The command 'set mode and filter' was postponed, and followed after the above:
          // > TX 008 FE FE 00 E0 06 03 02 FD         ; set mode and filter: CWN
          // > RX 006 FE FE E0 A2 FB FD               ; 'OK' for cmd 0x6000000
          // (It's important to keep this sequence, because otherwise,
          //  'mode and filter' would be set for the *old "MAIN Band" in Icom terms).
        }
       else
        { pRC->iParameterPollingSubState = 3; // ToDo: What else to try ?
        }
       break; // end case iParameterPollingSubState == 2


     default: // tried all of the above, but nothing worked : pRC->dblVfoFrequency is invalid, so READ IT from the radio:
       pRC->iParameterPollingSubState = RIGCTRL_POLLSTATE_DONE; // here: "gave up" in RigCtrl_FindVfoForNewFrequency()
       pRC->iParameterPollingSubState = 0;
       RigCtrl_AskForVFOFrequency( pRC, RIGCTRL_PORT_RADIO );
       break; // end default:
   } // end switch( pRC->iParameterPollingSubState ) in RIGCTRL_POLLSTATE_FIND_VFO_FOR_FREQ
} // end RigCtrl_FindVfoForNewFrequency()

//--------------------------------------------------------------------------
void RigCtrl_UpdateParamsAfterVfoSwitch( T_RigCtrlInstance *pRC )
     // [in,out] pRC->iParameterPollingState (RIGCTRL_POLLSTATE_AFTER_VFO_SWITCH),
     //          pRC->iParameterPollingSubState .
  // Another non-blocking state machine, runs shortly after e.g. "Toggle VFO A / B"
  // (which was the only official "VFO control command" supported
  //  by Yaesu FT-817/FT-857/FT-897 without peeking THE EEPROM !).
  // After forwarding such a 'VFO-switching command' from a remote client
  // to the real radio, we know ALMOST NOTHING about the new frequency, op-mode,
  //  filter bandwidth, split mode & offset, etc .... unless we have kept track
  //  of what's going on, WHICH SO FAR ONLY WORKS WITH ICOM TRANSCEIVERS,
  //  which always had a much better command set (in CI-V) than the old Yaesus:
  //  * Icom's command 0x07, "Select the VFO Mode" when used with
  //       SUB-command 0x00 ("Select VFO A") or SUB-command 0x01 ("Select VFO B")
  //       leave no doubt about the NEW VFO NUMBER (stored in pRC->
  //  * We may have stored the 'Selected VFO' parameters as well as the
  //                           'Unselected VFO' parameters when the external client
  //                           has written them (e.g. WSJT-X seems to do this),
  //  * We may have stored a long list of audio filter parameters for each mode,
  //    in T_RigCtrlInstance.sAudioFilterParams[RIGCTRL_NUM_AUDIO_FILTER_PARAMS],
  //    so after writing or reading Icom's "Operating Mode and Filter Number"
  //    (CI-V commands 0x01="Send mode data (transceive)",
  //                   0x04="Read operating mode",
  //                   0x06="Operating mode selection for transceive",
  //     all use the same FOUR-DIGIT-FORMAT to indicate MODE and FILTER-NUMBER),
  //     we can look all we need to know about the FILTER PASSBAND from a table
  //     instead of wasting bandwidth with unnecessary polling,
  //  * etc etc - see details in the implementation.
{
  switch( pRC->iParameterPollingSubState )
   {

     default:
        pRC->iParameterPollingState = RIGCTRL_POLLSTATE_DONE; // here: "last step" in RigCtrl_UpdateParamsAfterVfoSwitch()
        pRC->iParameterPollingSubState = 0;
        RigCtrl_AskForVFOFrequency( pRC, RIGCTRL_PORT_RADIO );
        break;
   }

} // end RigCtrl_UpdateParamsAfterVfoSwitch()

//--------------------------------------------------------------------------
void RigCtrl_OnVfoSwitch( T_RigCtrlInstance *pRC, int iNewVfoIndex )
  // CALLED FROM PARSER FUNCTIONS (e.g. RigCtrl_ParseCIV() ),
  // but isn't an API to SWITCH THE CURRENT VFO !
  // Good transceivers may even send the info about the OPERATOR switching
  // the VFO as an unsolicited report ... but don't bet on that .
  // For CI-V, this function more or less handles command 0x07,
  // with various sub-commands specified in "A7292-4EX-11", page 161.
  // iNewVfoIndex represents the sub-command, but not using Icom-specific codes:
  //   [in] iNewVfoIndex = 0 :  the operator (or whoever) has switched to "VFO A"
  //        iNewVfoIndex = 1 :  the operator (or whoever) has switched to "VFO B"
  //        iNewVfoIndex = VFO_SWITCH_EQUALIZE_A_B:  the operator (or whoever) has 'equalized VFO A and B"
  //        iNewVfoIndex = VFO_SWITCH_EXCHANGE_A_B:  the operator (or whoever) has EXCHANGED (swapped) A and B
  // [in,out] pRC->dblUnselVfoFreq, pRC->iUnselVfoOpMode,
  //          pRC->dblVfoFrequency, pRC->iOpMode,
  //          pRC->dblTxFrequency (?),
  //          pRC->
  //          pRC->iCurrVfoIndex, pRC->iPrevVfoIndex .
{
  BOOL fReadParamsAfterVfoSwitch = FALSE;
  int i;
  if( (iNewVfoIndex>=0) && (iNewVfoIndex<=1) ) // it's a SIMPLE VFO SWITCH:
   { if( iNewVfoIndex != pRC->iCurrVfoIndex )  // .. and it's really a DIFFERENT VFO now..
      { pRC->iPrevVfoIndex = pRC->iCurrVfoIndex;
        pRC->iCurrVfoIndex = iNewVfoIndex;
        fReadParamsAfterVfoSwitch = TRUE;
      }
   }
  else if( iNewVfoIndex < 0 ) // not a real "Swich" but Icom's "Equalize A and B":
   { // [in,out] pRC->dblUnselVfoFreq, pRC->iUnselVfoOpMode,
     //          pRC->dblVfoFrequency, pRC->iOpMode,
     //          pRC->dblTxFrequency (?),
     //          pRC->iCurrVfoIndex, pRC->iPrevVfoIndex .
     if( (pRC->iCurrVfoIndex != RIGCTRL_NOVALUE_INT) || (pRC->iPrevVfoIndex != RIGCTRL_NOVALUE_INT) )
      { i = pRC->iCurrVfoIndex;  // swap these...
        pRC->iCurrVfoIndex = pRC->iPrevVfoIndex;
        pRC->iPrevVfoIndex = i;
      }
     else // "current" and/or "previous" VFO number (zero-based indices) unkown:
      { fReadParamsAfterVfoSwitch = TRUE; // got to read all details about the NEW VFO from the radio !
      }
   } // end if < special case to EXCHANGE (swap) VFOs "A" and "B" >

  if( fReadParamsAfterVfoSwitch
   && ( pRC->iParameterPollingState == RIGCTRL_POLLSTATE_DONE ) )
   { // Don't interfere with other "parameter polling states" !
     // Reacting on a 'VFO switch' is only important AFTER the init-phase.
     pRC->iParameterPollingState = RIGCTRL_POLLSTATE_AFTER_VFO_SWITCH;
     pRC->iParameterPollingSubState = 0;
     // '--> next steps in RigCtrl_UpdateParamsAfterVfoSwitch() ...
   }
} // end RigCtrl_OnVfoSwitch()


//--------------------------------------------------------------------------
CPROT BOOL RigCtrl_SwitchToBand( T_RigCtrlInstance *pRC,
             DWORD dwNewBand ) // [in] RIGCTRL_BAND_xyz (only ONE bit per band)
  // Return value: TRUE = band-switching command was accepted and it on its way
  //                      "to the radio". Some time later, we know the actual
  //                      FREQUENCY (selected by the rig itself, e.g. from
  //                      Icom's "Band Stacking" registers). Then,
  //                      pRC->dblVfoFrequency will change automatically.
  //               FALSE = illegal band (not supported), or rejected by
  //                       local control . pRC->dblVfoFrequency will not change.
  //        Note: There is no extra member in T_RigCtrlInstance for the
  //              currently active band. If necessary, use
  //              RigCtrl_FrequencyToBand() to convert e.g.
  //              pRC->dblVfoFrequency or pRC->dblTxFrequency into one of the
  //              'band bitmasks' (RIGCTRL_BAND_...) .
  //
  // This function is important for radios without 'general coverage', e.g.
  //  IC-9700 (with RIGCTRL_BAND_2M, RIGCTRL_BAND_70CM, and RIGCTRL_BAND_23CM),
  //  where you cannot simply increment the frequency in 1- or 10-MHz steps
  //  because those non-general-coverage radios won't accept frequencies
  //  BETWEEN BANDs.
{

  return FALSE;

} // end  RigCtrl_SwitchToBand()

//--------------------------------------------------------------------------
CPROT DWORD RigCtrl_GetAvailableBands( T_RigCtrlInstance *pRC )
   // Returns a bitwise combination of RIGCTRL_BAND_xyz .
   // Does NOT care about being able to TRANSMIT on a band (RECEIVE is sufficient).
   //
   //  [in] pRC->dwAvailableBands : initialized in RigCtrl_Init()
   //                               but restrived/revised later, depending on
   //                               what the protocol (and THE RADIO) permit.
   //  [in] pRC->TxBandEdges[], pRC->iNumTxBands : read from the rig via CI-V .
   //       (a band on which THE RIG says it can TRANSMIT will be automatically
   //        added to the combination returned by RigCtrl_GetAvailableBands() )
   // See also (related, actually USES RigCtrl_GetAvailableBands() ) :
   //  * RigCtrl_InitUserDefinedBandsFromRigInfo()
   //  * RigCtrl_MayTransmitOnFrequencyAndMode()
{ DWORD dwAvailableBands = pRC->dwAvailableBands;
        // '-> Usually copied from  RigCtrl_RadioInfo_CIV[], e.g.
        //     RIGCTRL_BAND_ALL_SHORTWAVE | RIGCTRL_BAND_6M | RIGCTRL_BAND_4M = 0x35FB .
        //  Note the absence of e.g. the 60-meter-band (etc?) here, because
        //       the IC-7300 required the "diode mod" / "all-band TX" for 60 m.
  int iEdge, iBand;
  DWORD dwBandMask;
  const T_RigCtrlFrequencyRange *pFreqRange;
  double dblFmin_Hz, dblFmax_Hz;

  // If the rig itself (not a "config file") has reported one or more
  // 'TX band edges', iterate through those edges to check for additional bands,
  // that may NOT be covered in RigCtrl_RadioInfo_CIV[] (e.g. 60 m band for IC-7300).
  // A configuration file provided by the server's sysop can only REDUCE the
  // allowed transmit frequency ranges, but never EXTEND them. In the RCW Keyer,
  // this is just a bitwise AND of the value returned by RigCtrl_GetAvailableBands()
  //                               and T_RigCtrlInstance.dwTxBandsAllowedBySysop .
  for(iEdge=0; (iEdge<pRC->iNumTxBands) && (iEdge<RIGCTRL_NUM_TX_BAND_EDGES); ++iEdge)
   { pFreqRange = &pRC->TxBandEdges[iEdge]; // <- this may cover MULTIPLE BANDS !
     // In an IC-7300 with the 60-m-band enabled, a single "TX Band"
     // may even covered ALL BANDS, from 0.1 to 78 MHz ! Thus, to have these
     // extra bands listed, check pFreqRange for *ALL* amateur radio bands:
     for( iBand=0; iBand<=RIGCTRL_BAND_MAX_BITNUM; ++iBand )
      { dwBandMask = 1UL<<iBand;  // convert the bit number into a "bitmask"

        if( (dwBandMask == RIGCTRL_BAND_2200M) // exclude these exotic (and now mostly deserted) bands,
          ||(dwBandMask == RIGCTRL_BAND_600M)) // because they were a side effect of "unlocking" an IC-7300
         { // These low frequencies will saturate the ferrite output transformer
           // unless the operator has modified the PA, or reduced the power to a few mW.
           // Thus, even if THE RIG reports it "can" transmit on these bands,
           // do NOT add the list of TX bands automatically !
         }
        else // looks like a band that is "not expected to do harm", so add it:
         { RigCtrl_BandToFrequencyRange( dwBandMask, &dblFmin_Hz, &dblFmax_Hz );
           // Example: Running through "all bands" with a 60-m-band-modded IC-7300:
           //          pFreqRange->dblFmin_Hz =   100000 ( 100 kHz .. you've got to be kidding, Icom)
           //          pFreqRange->dblFmax_Hz = 74800000 (74.8 MHz .. will never use that in Europe)
           // iBand=0 :  dblFmin_Hz= 1800000, dblFmax_Hz= 2000000 -> 160 m band "ok" (TX-wise..)
           // iBand=1 :  dblFmin_Hz= 3500000, dblFmax_Hz= 3800000 ->  80 m band "ok"
           // iBand=2 :  dblFmin_Hz= 5351500, dblFmax_Hz= 5366500 ->  60 m band "ok" (at least for DL..)
           // iBand=3 :  dblFmin_Hz= 7000000, dblFmax_Hz= 7200000 ->  40 m band "ok"
           //              ....
           // iBand=14:  dblFmin_Hz=14400000, dblFmax_Hz=14600000 ->   2 m band "NOT ok"
           if(  (dblFmax_Hz > dblFmin_Hz )  // is this a valid RANGE at all (not 0..0) ?
             && (pFreqRange->dblFmin_Hz <= dblFmin_Hz )
             && (pFreqRange->dblFmax_Hz >= dblFmax_Hz ) )
            { dwAvailableBands |= dwBandMask;  // add another BIT (bitwise ORed)
            }
         }  // end if < band "not expected to do harm" (TX-wise) >
      }    // end for < all ham radio bands that MAY be contained in Icom's "Tx band edges" after an all-band TX modification >
   }      // end for < all "Tx band edges" reported by the radio itself >


  return dwAvailableBands;
  // In DL4YHF's "Remote CW Keyer", this result (dwAvailableBands) will be
  //  listed in the 'Bands' selection drop-down list on the 'TRX' tab,
  //  possibly along with the edge frequencies for a particular band.
  //  See KeyerGUI_FillComboWithBandList(), which relies on RigCtrl_GetAvailableBands() !
  //

} // end RigCtrl_GetAvailableBands()

//--------------------------------------------------------------------------
static BOOL InRange( double dblVal, double dblMin, double dblMax )
{  return (dblVal>=dblMin) && (dblVal<=dblMax);
}


//--------------------------------------------------------------------------
DWORD RigCtrl_FrequencyToBand( double dblFrequency ) // -> RIGCTRL_BAND_xyz
  // Note the "greater tolerance" of RigCtrl_FrequencyToBand()
  //      for e.g. frequencies in the 60 meter band,
  //      where we have non-overlapping sub-bands in continental EU, the UK,
  //      and whoever has access access to the "WRC-15" band.
  //    Example: 5.2583 MHz ... 5.2640 MHz ("UK band"),
  //             5.3515 MHz ... 5.3540 MHz ("recommended for CW" in continental EU),
  //             etc etc,
  //       will ALL be identified as RIGCTRL_BAND_60M by this function.
  //       When loaded from a USER DEFINED BAND LIST (e.g. RCWKeyer_Bands.txt),
  //       all these "sub-bands" will appear in multiple COLUMNS(!)
  //       in the same ROW(!) of the GUI's "Band / Frequency selection combo",
  //       generated at runtime in Remote_CW_Keyer\Keyer_GUI.cpp : KeyerGUI_FillComboWithBandList() .
  // Because RigCtrl_FrequencyToBand() is not aware of user defined [sub-]bands,
  // the Remote CW Keyer GUI uses iRow from KeyerGUI_BandComboInfoTable[iRow][iColumn]
  // instead of the hard coded 31 "band bits" .

{
  // Similar to modern Icom transceivers (e.g. IC-7300),
  //  which have a bit of 'tolerance' in their BAND STACKING REGISTERS,
  //  be tolerant with the following frequency range checks.
  //  For example, an IC-7300 listed 18.05 MHz in the BAND STACKING REGISTERS
  //  of the 17-meter-band, even though 18.05 MHz isn't an amateur radio frequency.
  //  The GUI (Keyer_GUI.cpp : KeyerGUI_FillComboWithBandList()) will include such entries
  //  in the list of BAND STACKING REGISTERS, but because
  //  RigCtrl_MayTransmitOnFrequencyAndMode() will return FALSE for "out-of-band" frequencies,
  //  will flag such frequencies as "Receive Only".
  if( InRange( dblFrequency, 135000.0, 138000.0 ) ) // LF "2200 meters" ?
   { return RIGCTRL_BAND_2200M;
   }
  if( InRange( dblFrequency, 472000.0, 479000.0 ) ) // MF "630 meters"
   { return RIGCTRL_BAND_600M;
   }
  if( InRange( dblFrequency, 1800000.0, 2000000.0 ) ) // MF "160 meters"
   { return RIGCTRL_BAND_160M;
   }
  if( InRange( dblFrequency, 3500000.0, 4000000.0 ) ) // "80 m"
   { return RIGCTRL_BAND_80M;
   }
  if( InRange( dblFrequency, 5200e3, 5406e3 ) ) // "60 m" (note the very strange and incompatible allocations, worldwide)
   { // ,---------------------'
     // '--> Includes exclusive "UK 60 m bands", e.g. 5.2585 .. 5.2640 MHz,
     //      and the "WRC-15" 60 meter band frequencies, 5351.5 to 5366.5 kHz,
     //      so those sub-bands are accepted in e.g. KeyerGUI_FillComboWithBandList().
     //  The US allocations are even crazier: They have FIVE "widely spaced" channels:
     //   > Channel 1: 5330.5 kHz  (On this channel, the only allowed CW carrier is on 5332.0 kHz)
     //   > Channel 2: 5346.5 kHz  (here, the only allowed CW carrier is on 5348.0 kHz)
     //   > Channel 3: 5357.0 kHz  (here, the only allowed CW carrier is on 5358.5 kHz)
     //   > Channel 4: 5371.5 kHz  (here, the only allowed CW carrier is on 5373.0 kHz)
     //   > Channel 5: 5403.5 kHz  (here, the only allowed CW carrier is on 5405.0 kHz)
     //   > Each channel has an effective bandwidth of 2.8 kHz. (.. but that's not all ..)
     //   > CW operation must take place at the center of your chosen channel.
     //   > This means that your transmitting frequency must be 1.5 kHz
     //   > above the suppressed carrier frequency as specified in the Report and Order
     //   > (see Table 1). Operating at strict channel-center frequencies may come
     //   > as a disappointment to many, but cooperating with the NTIA is key
     //   > to expanded privileges in the future.
     //   >
     return RIGCTRL_BAND_60M;
   }
  if( InRange( dblFrequency,  7.0e6,  7.3e6  ) ) // "40 m"
   { return RIGCTRL_BAND_40M;
   }
  if( InRange( dblFrequency, 10.0e6, 10.15e6 ) ) // "30 m"
   { return RIGCTRL_BAND_30M;
   }
  if( InRange( dblFrequency, 14e6, 14.35e6 ) ) // "20 m"
   { return RIGCTRL_BAND_20M;
   }
  if( InRange( dblFrequency, 18e6,18.2e6 ) ) // "17 m" (with the "tolerance", TX only on 18.068 .. 18.168 MHz; see RigCtrl_MayTransmitOnFrequencyAndMode() )
   { return RIGCTRL_BAND_17M;
   }
  if( InRange( dblFrequency, 21.0e6, 21.45e6 ) )  // "15 m"
   { return RIGCTRL_BAND_15M;
   }
  if( InRange( dblFrequency, 24.89e6, 24.99e6 ) ) // "12 m" (TX only on 24.89 ... 24.99 MHz; see RigCtrl_MayTransmitOnFrequencyAndMode() )
   { return RIGCTRL_BAND_12M;
   }
  if( InRange( dblFrequency, 28e6, 29.7e6 ) ) // "10 m"
   { return RIGCTRL_BAND_10M;
   }
  if( InRange( dblFrequency, 50e6, 54e6 ) ) // "6 m"   in most of ITU Region 1 (for TX ranges, see RigCtrl_MayTransmitOnFrequencyAndMode() )
   { return RIGCTRL_BAND_6M;
   }
  if( InRange( dblFrequency, 70e6, 70.5e6 ) ) // "4 m" in a few lucky countries (not DL)
   { return RIGCTRL_BAND_4M;
   }
  if( InRange( dblFrequency, 144e6, 148e6 ) ) // "2 m" (again, for TX ranges, see RigCtrl_MayTransmitOnFrequencyAndMode() ..)
   { return RIGCTRL_BAND_2M;
   }
  if( InRange( dblFrequency, 430e6, 440e6 ) ) // "70 cm" in ITU Region 1
   { return RIGCTRL_BAND_70CM;
   }
  if( InRange( dblFrequency, 1240e6, 1300e6 ) ) // "23 cm" in ITU Region 1
   { return RIGCTRL_BAND_23CM;
   }
  if( InRange( dblFrequency, 2300e6, 2450e6 ) ) // "13 cm" (note the GAP between 2310 and 2390 MHz)
   { return RIGCTRL_BAND_13CM;
   }
  if( InRange( dblFrequency, 3300e6, 3500e6 ) ) // "9 cm"
   { return RIGCTRL_BAND_9CM;
   }
  if( InRange( dblFrequency, 5650e6, 5850e6 ) ) // "6 cm"
   { return RIGCTRL_BAND_6CM;
   }
  if( InRange( dblFrequency, 10e9, 10.5e9 ) )   // "3 cm"
   { return RIGCTRL_BAND_3CM;
   }

  return 0;

} // end RigCtrl_FrequencyToBand()

//--------------------------------------------------------------------------
static void AddRange( double dblMin, double dblMax,
                      double *pdblCombinedMin, double *pdblCombinedMax )
{ if( (*pdblCombinedMin == RIGCTRL_NOVALUE_DOUBLE ) || ( dblMin < *pdblCombinedMin) )
   {   *pdblCombinedMin = dblMin;
   }
  if( (*pdblCombinedMax == RIGCTRL_NOVALUE_DOUBLE ) || ( dblMax > *pdblCombinedMax) )
   {   *pdblCombinedMax = dblMax;
   }
} // end AddRange()


//--------------------------------------------------------------------------
void RigCtrl_BandToFrequencyRange(  // Function with the "amateur radio band knowledge"...
         DWORD dwBand, // [in] RIGCTRL_BAND_... (usually only ONE BIT set,
                       //                        but combinations are allowed)
         double *pdblFmin_Hz, double *pdblFmax_Hz ) // [out] band edge frequencies
         // [in] RigCtrl_ITU_Region (only to demonstrate the purpose) :
         //   Affects SOME of the band edge frequencies .. for a few bands, incomplete:
         //      ITU region 1 = Europe, Africa, FSU, Mongolia, Middle East, Iraq
         //      ITU region 2 = North- and South America, Greenland, some east pacific islands
         //      ITU region 3 = Far East, non-FSU Asia, Iran, Australia, Oceania

  // If dwBand doesn't contain a valid 'band bit', the result
  // (*pdblFmin_Hz and *pdblFmax_Hz) will be set to RIGCTRL_NOVALUE_DOUBLE.
{
  *pdblFmin_Hz = *pdblFmax_Hz = RIGCTRL_NOVALUE_DOUBLE;
  if( dwBand & RIGCTRL_BAND_2200M )
   { AddRange( 135700.0, 137800.0, pdblFmin_Hz, pdblFmax_Hz );
   }
  if( dwBand & RIGCTRL_BAND_600M )
   { AddRange( 472000.0, 479000.0, pdblFmin_Hz, pdblFmax_Hz );
   }
  if( dwBand & RIGCTRL_BAND_160M )
   { switch( RigCtrl_ITU_Region )
      { case 1: // Europe, Africa, FSU, Mongolia, Middle East, Iraq
        default:
           AddRange( 1810e3, 2000e3, pdblFmin_Hz, pdblFmax_Hz ); // "160 m" in ITU region 1
           break;
        case 2: // N & S America, Greenland, some east pacific islands
           AddRange( 1800e3, 2000e3, pdblFmin_Hz, pdblFmax_Hz ); // "160 m" in ITU region 2
           break;
        case 3: // Far East, non-FSU Asia, Iran, Australia, Oceania
           AddRange( 1800e3, 2000e3, pdblFmin_Hz, pdblFmax_Hz ); // "160 m" in ITU region 3
           break;
      }

   }
  if( dwBand & RIGCTRL_BAND_80M )
   { switch( RigCtrl_ITU_Region )
      { case 1: // Europe, Africa, FSU, Mongolia, Middle East, Iraq
        default:
           AddRange( 3500e3, 3800e3, pdblFmin_Hz, pdblFmax_Hz ); // "80 m" in ITU region 1
           break;
        case 2: // N & S America, Greenland, some east pacific islands
           AddRange( 3500e3, 4000e3, pdblFmin_Hz, pdblFmax_Hz ); // "80 m" in ITU region 2
           break;
        case 3: // Far East, non-FSU Asia, Iran, Australia, Oceania
           AddRange( 3500e3, 3900e3, pdblFmin_Hz, pdblFmax_Hz ); // "80 m" in ITU region 3
           break;
      }
   }
  if( dwBand & RIGCTRL_BAND_60M )
   { AddRange( 5351.5e3, 5366.5e3, pdblFmin_Hz, pdblFmax_Hz ); // "60 m" (from WRC-15, worldwide)
     // Note the special treatment of this band in KeyerGUI_FillComboWithBandList() !
     // If USER-DEFINED BANDS exist, their exact frequency ranges will be used
     // instead of the above "WRC-15 60 m band edges" !
   }
  if( dwBand & RIGCTRL_BAND_40M )
   { switch( RigCtrl_ITU_Region )
      { case 1: // Europe, Africa, FSU, Mongolia, Middle East, Iraq
        default:
           AddRange( 7.0e6,  7.2e6, pdblFmin_Hz, pdblFmax_Hz );  // "40 m" in ITU region 1
           break;
        case 2: // N & S America, Greenland, some east pacific islands
           AddRange( 7.0e6,  7.3e6, pdblFmin_Hz, pdblFmax_Hz );  // "40 m" in ITU region 2
           break;
        case 3: // Far East, non-FSU Asia, Iran, Australia, Oceania
           AddRange( 7.0e6,  7.2e6, pdblFmin_Hz, pdblFmax_Hz );  // "40 m" in ITU region 3
           break;
      }
   }
  if( dwBand & RIGCTRL_BAND_30M )
   { AddRange( 10.1e6, 10.15e6, pdblFmin_Hz, pdblFmax_Hz ); // "30 m" in all ITU regions ?
   }
  if( dwBand & RIGCTRL_BAND_20M )
   { AddRange( 14.0e6, 14.35e6, pdblFmin_Hz, pdblFmax_Hz ); // "20 m" in all ITU regions ?
   }
  if( dwBand & RIGCTRL_BAND_17M )
   { AddRange( 18.068e6,18.168e6, pdblFmin_Hz, pdblFmax_Hz ); // "17 m" in all ITU regions ?
   }
  if( dwBand & RIGCTRL_BAND_15M )
   { AddRange( 21.0e6, 21.45e6, pdblFmin_Hz, pdblFmax_Hz );  // "15 m" in all ITU regions ?
   }
  if( dwBand & RIGCTRL_BAND_12M )
   { AddRange( 24.89e6, 24.99e6, pdblFmin_Hz, pdblFmax_Hz ); // "12 m" in all ITU regions ?
   }
  if( dwBand & RIGCTRL_BAND_10M )
   { AddRange( 28e6, 29.7e6, pdblFmin_Hz, pdblFmax_Hz ); // "10 m" in all ITU regions ?
   }
  if( dwBand & RIGCTRL_BAND_6M )
   { AddRange( 50e6, 54e6, pdblFmin_Hz, pdblFmax_Hz ); // "6 m" in most of ITU Region 1 (except for poor old Germany)
   }
  if( dwBand & RIGCTRL_BAND_4M )
   { AddRange( 70e6, 70.5e6, pdblFmin_Hz, pdblFmax_Hz ); // "4 m" in a few lucky countries (not DL; only "temporarily tolerated")
   }
  if( dwBand & RIGCTRL_BAND_2M )
   { switch( RigCtrl_ITU_Region )
      { case 1: // Europe, Africa, FSU, Mongolia, Middle East, Iraq
        default:
           AddRange( 144e6, 146e6, pdblFmin_Hz, pdblFmax_Hz ); // "2 m"   in ITU Region 1
           break;
        case 2: // N & S America, Greenland, some east pacific islands
           AddRange( 144e6, 148e6, pdblFmin_Hz, pdblFmax_Hz ); // "2 m"   in ITU Region 2
           break;
        case 3: // Far East, non-FSU Asia, Iran, Australia, Oceania
           AddRange( 144e6, 148e6, pdblFmin_Hz, pdblFmax_Hz ); // "2 m"   in ITU Region 3
           break;
      }
   }
  if( dwBand & RIGCTRL_BAND_70CM )
   { AddRange( 430e6, 440e6, pdblFmin_Hz, pdblFmax_Hz ); // "70 cm" in ITU Region 1
   }
  if( dwBand & RIGCTRL_BAND_23CM )
   { AddRange( 1240e6, 1300e6, pdblFmin_Hz, pdblFmax_Hz ); // "23 cm" in ITU Region 1
   }
  if( dwBand & RIGCTRL_BAND_13CM )
   { AddRange( 2300e6, 2450e6, pdblFmin_Hz, pdblFmax_Hz ); // "13 cm" (note the GAP between 2310 and 2390 MHz)
   }
  if( dwBand & RIGCTRL_BAND_9CM )
   { AddRange( 3300e6, 3500e6, pdblFmin_Hz, pdblFmax_Hz ); // "9 cm"
   }
  if( dwBand & RIGCTRL_BAND_6CM )
   { AddRange( 5650e6, 5850e6, pdblFmin_Hz, pdblFmax_Hz ); // "6 cm"
   }
  if( dwBand & RIGCTRL_BAND_3CM )
   { AddRange( 10e9, 10.5e9, pdblFmin_Hz, pdblFmax_Hz );   // "3 cm"
   }

} // end RigCtrl_BandToFrequencyRange()

// The above functions [RigCtrl_GetAvailableBands(), RigCtrl_FrequencyToBand(),
//                      RigCtrl_BandToFrequencyRange()]
// are restrictions from band plans and the radio itself.
// The 'sysop' (who operates the remotely controlled radio) may restrict
// TRANSMISSION to whatever his license, frequency, antenna, tuner, etc
// permits, by the use of a CONFIGURATION FILE. The "IARU bandplan", and the
// rig's self-reported "Tx Edge Frequencies" (Icom slang) are only the DEFAULTS,
// when, at program start, no RCWKeyer_Config.txt can be loaded :

//--------------------------------------------------------------------------
T_RigCtrlUserDefinedBand *RigCtrl_GetUserDefinedBandByFrequencyRange(
             T_RigCtrlInstance *pRC,  // [in] pRC->UserDefinedBands[0..pRC->iNumUserDefinedBands]
             double  dblFmin_Hz, double  dblFmax_Hz )
  // Returns NULL if there is no frequency overlap in ANY of the "User-Defined Bands".
  // Note: When not loaded from a configuration file, these "bands" (frequency ranges)
  //       are filled with meaningful defaults in RigCtrl_InitUserDefinedBandsFromRigInfo().
{
  int iBand;
  T_RigCtrlUserDefinedBand *pBand;

  for(iBand=0; (iBand<RIGCTRL_NUM_USER_DEFINED_BANDS) && (iBand<pRC->iNumUserDefinedBands); ++iBand)
   { pBand = &pRC->UserDefinedBands[iBand];
     // Consider this example: USER-DEFINED BAND as subset of a wider ham radio band.
     //    10.15e6              10.1001e6       10.1000e6           10.130 (end of the "suggested CW range")
     //       |                   |               |                   |
     if( (dblFmax_Hz >= pBand->dblFmin_Hz) && (dblFmin_Hz <= pBand->dblFmax_Hz ) )
      { return pBand;
      }
   }
  return NULL;
} // end RigCtrl_GetUserDefinedBandByFrequencyRange()

//--------------------------------------------------------------------------
T_RigCtrlUserDefinedBand *RigCtrl_GetUserDefinedBandByName(
             T_RigCtrlInstance *pRC,  // [in] pRC->UserDefinedBands[0..pRC->iNumUserDefinedBands]
             const char* pszName )    // [in] e.g. "60 m EU" (when loaded from file..)
  // Returns NULL if there is no matching name in any of the "User-Defined Bands".
{
  int iBand;
  T_RigCtrlUserDefinedBand *pBand;

  for(iBand=0; (iBand<RIGCTRL_NUM_USER_DEFINED_BANDS) && (iBand<pRC->iNumUserDefinedBands); ++iBand)
   { if( strcmp( pBand->sz16Name, pszName ) == 0 )
      { return pBand;
      }
   }
  return NULL;
} // end RigCtrl_GetUserDefinedBandByName()


typedef enum // switch-case friendly enums for the key=value pairs in a "User Defined Band", for RigCtrl_StringToUserDefinedBand() :
{ RIGCTRL_UDEF_BAND_TOKEN_INDEX,    // "i=" -> array index (not a part of the struct, but an extra function argument)
  RIGCTRL_UDEF_BAND_TOKEN_N_ITEMS,  // "n=" -> total number of items in the array
  RIGCTRL_UDEF_BAND_TOKEN_NAME,     // "na=" -> T_RigCtrlUserDefinedBand.sz16Name
  RIGCTRL_UDEF_BAND_TOKEN_FREQ,     // "fr=" -> T_RigCtrlUserDefinedBand.dblFmin_Hz - .dblFmax_Hz
  RIGCTRL_UDEF_BAND_TOKEN_MODES,    // "mo=" -> T_RigCtrlUserDefinedBand.dwOpModes
  RIGCTRL_UDEF_BAND_TOKEN_TX,       // "tx=" -> T_RigCtrlUserDefinedBand.dwPermissions.RIGCTRL_PERMISSION_TRANSMIT
  RIGCTRL_UDEF_BAND_TOKEN_DEF_FREQ, // "df=" -> T_RigCtrlUserDefinedBand.dblFdef_Hz
  RIGCTRL_UDEF_BAND_TOKEN_DEF_MODE  // "dm=" -> T_RigCtrlUserDefinedBand.dwDefOpMode
} RigCtrl_enUserDefdBandTokens;
static const T_SL_TokenList RigCtrl_UserDefinedBandTokens[] =
{ { "i=",    RIGCTRL_UDEF_BAND_TOKEN_INDEX   }, // "i=" -> array index (not a part of the struct, but an extra function argument)
  { "n=",    RIGCTRL_UDEF_BAND_TOKEN_N_ITEMS }, // "n=" -> total number of items in the array
  { "na=",   RIGCTRL_UDEF_BAND_TOKEN_NAME    }, // -> T_RigCtrlUserDefinedBand.sz16Name
  { "fr=",   RIGCTRL_UDEF_BAND_TOKEN_FREQ    }, // -> T_RigCtrlUserDefinedBand.dblFmin_Hz - .dblFmax_Hz
  { "mo=",   RIGCTRL_UDEF_BAND_TOKEN_MODES   }, // -> T_RigCtrlUserDefinedBand.dwOpModes
  { "tx=",   RIGCTRL_UDEF_BAND_TOKEN_TX      }, // -> T_RigCtrlUserDefinedBand.dwPermissions.RIGCTRL_PERMISSION_TRANSMIT
  { "df=",   RIGCTRL_UDEF_BAND_TOKEN_DEF_FREQ}, // -> T_RigCtrlUserDefinedBand.dblFdef_Hz
  { "dm=",   RIGCTRL_UDEF_BAND_TOKEN_DEF_MODE}, // -> T_RigCtrlUserDefinedBand.dwDefOpMode

  { NULL, 0 } // "all zeros" mark the end of the list
};

//---------------------------------------------------------------------------
const char* RigCtrl_GetUserDefdBandToken( int iToken ) // for string builder and parsers;
  // returns e.g. "mo=" for the numeric token code RIGCTRL_UDEF_BAND_TOKEN_MODES
{ return SL_GetStringFromTokenList( RigCtrl_UserDefinedBandTokens, iToken );
}

//---------------------------------------------------------------------------
void RigCtrl_OnTransmitFlagChange( T_RigCtrlInstance *pRC ) // [in] pRC->iTransmitting ( RIGCTRL_PN_TRANSMITTING; switched from RX to TX or back to RX again )
  // Depending on the rig capabilities, this function MAY be called from
  // RigCtrl_SetParamValue_Int() whenever pRC->iTransmitting (a FLAG) changes.
{

  // If the option (checkmark) "Sidetone when keyed ON THE RIG" is set,
  // RCW Keyer may have to reprogram what for the Icom IC-7300, the "Full Manual"
  // (A7292-4EX-11 page 163) calls 'Beep and speech output setting to ACC/USB'.
  // The purpose of doing this is explained in the RCW Keyer manual, see
  // file:///C:/cbproj/Remote_CW_Keyer/manual/Remote_CW_Keyer.htm#Sidetone_when_keyed_ON_THE_RIG
  if( pRC->dwRigControlFlags & RIGCTRL_FLAG_SIDETONE_WHEN_KEYED_ON_RIG )
   {
     if( pRC->iTransmitting > 0 ) // we know FOR SURE the rig is TRANSMITTING now ->
      { // If the rig is transmitting because e.g. RCW KEYER(!) has requested it,
        // RCW KEYER itself(!) will generate a CW sidetone, and we don't want
        // the ICOM RIG (e.g. IC-7300) to add the sidetone to the audio signal
        // on the USB (in the IC-7300, same flag also for "audio on ACC") :
        if( pRC->iTransmitReqst > 0 )  // transmitting because WE (e.g. RCW Keyer) requested the rig to transmit ->
         { // In this particular state, we do NOT want a sidetone added to the USB-Audio-output BY THE RIG,
           // because REMOTE CW KEYER ITSELF generates a sideone on e.g. the "Simple Paddle Adapter"
           // or (with a bit more latency) on the LOCAL AUDIO OUTPUT DEVICE.
           // Thus:
           if( pRC->iSidetone_on_USB > 0 ) // is THE RIG still adding its own CW-sidetone to the USB-Audio-output ?
            {  pRC->iSidetone_on_USB = 0;  // STOP the rig from adding its own sidetone to its USB-audio-output !
               RigCtrl_QueueUpCmdToWriteUnifiedPN( pRC, RIGCTRL_PN_SIDETONE_ON_USB_AUDIO ); // send the new setting of iSidetone_on_USB a.s.a.p.
               // (should get here when starting to send with a Morse key / paddle connected to RCW Keyer.)
               //  2025-09-20 : OK. Got here ONCE when sending was initiated by RCW Keyer.
               //   But then, strange things happened in the DEBUG window / traffic log:
               // > 6789 0110856 TX 007 FE FE 94 E0 1C 00 FD     ; read Transmitting
               // > 6791 0110919 RX 008 FE FE E0 94 1C 00 00 FD  ; Transmitting=0
               // > 6797 0110983 TX 008 FE FE 94 E0 1C 00 01 FD  ; write TransmitReqst
               // > 6798 0111046 RX 006 FE FE E0 94 FB FD        ; Transmitting : OK
               // > 6801 0111046 TX 007 FE FE 94 E0 15 02 FD       ; read SMeterLevel
               // > 6803 0111109 RX 009 FE FE E0 94 15 02 00 00 FD ; SMeterLevel=0 dB
               // > 6805 0111109 TX 007 FE FE 94 E0 1C 00 FD     ; read Transmitting
               // > 6806 0188537 RX 008 FE FE E0 94 1C 00 01 FD  ; Transmitting=1
               // > 6807 0188537 TX 009 FE FE 94 E0 1A 05 00 62 FD    ; write Sidetone_on_USB [WRONG ... the DATA were missing]
               // > 6808 0188597 RX 00A FE FE E0 94 1A 05 00 62 01 FD ; Sidetone_on_USB=1 (*)
               // > 6809 0190451 TX 008 FE FE 94 E0 1C 00 00 FD ; write TransmitReqst
               // > 6810 0190514 RX 006 FE FE E0 94 FB FD       ; Transmitting : OK
               // > 6813 0190514 TX 007 FE FE 94 E0 15 15 FD    ; read SupplyVoltage
               // > 6815 0190577 RX 009 FE FE E0 94 15 15 01 65 FD ; SupplyVoltage=14000 mV
               // > 6818 0190577 TX 007 FE FE 94 E0 15 16 FD    ; read DrainCurrent
               // > 6820 0190640 RX 009 FE FE E0 94 15 16 00 00 FD ; DrainCurrent=0 mA
               // > 6823 0190640 TX 007 FE FE 94 E0 15 11 FD    ; read PowerMeter
               // > 6825 0190704 RX 009 FE FE E0 94 15 11 00 00 FD ; PowerMeter=0 %
               // > 6828 0190704 TX 007 FE FE 94 E0 1C 00 FD    ; read Transmitting
               // > 6830 0190766 RX 008 FE FE E0 94 1C 00 00 FD ; Transmitting=0
               // (*) Message #6808 was coloured CYAN like an unsolicited report !
               //     But in fact it was a correct READ RESPONSE trom the radio,
               //     because the command to WRITE THE NEW SIDETONE SETTING
               //     was lacking the DATA FIELD - which turned the WRITE- into a READ-command !
               //
            } // end if( pRC->iSidetone_on_USB > 0 )
         }
        else // the rig indicates transmitting but not because WE (e.g. RCW Keyer) ask it to to transmit ->
         { // This means, most likely, that the sysop himself uses the rig to send CW (*),
           // with his Morse key connected DIRECTLY to the transmitter.
           // (*) Even if the IC-7300's internal keyer is configured for 'Paddle',
           //     remote keying by others via 'Straight Keying signal' on
           //     the Icom's Virtual COM Port "DTR" is possible. Great !
           if( pRC->iTransmitReqstCountdown_ms <= 0 ) // last time pRC->iTransmitReqst was SET "quite long ago" ->
            {
              // In this case, remote CLIENTS shall hear the sysop's CW signal as quickly as possible, so:
              if( pRC->iSidetone_on_USB <= 0 ) // does the rig NOT add its own CW-sidetone to the USB-Audio-output ?
               {  pRC->iSidetone_on_USB =  1;  // .. then ask the rig to add its own CW-sidetone, and ..
                  RigCtrl_QueueUpCmdToWriteUnifiedPN( pRC, RIGCTRL_PN_SIDETONE_ON_USB_AUDIO ); // ..send the new setting of iSidetone_on_USB a.s.a.p.
                  // (should get here when starting to send with a Morse key / paddle connected DIRECTLY ON THE RADIO.
                  //  2025-09-20 : OK.  Result on the DEBUG TAB :
                  // > 2177 0044309 TX 007 FE FE 94 E0 1C 00 FD     ; read Transmitting
                  // > 2178 0044371 RX 008 FE FE E0 94 1C 00 00 FD  ; Transmitting=0
                  // > 2184 0044433 TX 007 FE FE 94 E0 14 09 FD     ; read CW_Pitch
                  // > 2186 0044496 RX 009 FE FE E0 94 14 09 01 40 FD ; CW_Pitch=630 Hz
                  // > 2192 0044558 TX 007 FE FE 94 E0 15 02 FD     ; read SMeterLevel
                  // > 2194 0044621 RX 009 FE FE E0 94 15 02 00 00 FD ; SMeterLevel=0 dB
                  // > 2200 0044684 TX 007 FE FE 94 E0 1C 00 FD     ; read Transmitting
                  // > 2201 0044747 RX 008 FE FE E0 94 1C 00 01 FD  ; Transmitting=1
                  // > 2203 0044747 TX 00A FE FE 94 E0 1A 05 00 62 01 FD ; write Sidetone_on_USB
                  // > 2204 0044809 RX 006 FE FE E0 94 FB FD ; Sidetone_on_USB : OK

               } // end if( pRC->iSidetone_on_USB <= 0 )
            }   // end if( pRC->iTransmitReqstCountdown_ms <= 0 )
         }     // end else < pRC->iTransmitReqst NOT set, so the rig transmit by itself >
      }       // end if( pRC->iTransmitting > 0 )
     else
      { // If the rig is NOT TRANSMITTING, pRC->iSidetone_on_USB has no effect,
        // so avoid sending unnecessary CI-V commands to the rig until there's
        // reason to switch pRC->iSidetone_on_USB again . This also reduces
        // the problem caused by the low polling-rate for pRC->iTransmitting :
        // Only the sysop's first few ten milliseonds of the FIRST over may be
        // without sent without a sidetone for remote users of the same radio.
        // When transmission starts for SUBSEQUENT overs, pRC->iSidetone_on_USB
        // is already properly set, and remote clients will even hear the first
        // "dit" of the sysop's transmission .
      }
   } // end if( pRC->dwRigControlFlags & RIGCTRL_FLAG_SIDETONE_WHEN_KEYED_ON_RIG )

} // end RigCtrl_OnTransmitFlagChange()


//---------------------------------------------------------------------------
void RigCtrl_ModifyBitInDWORD( DWORD *pdwDest, DWORD dwBitmask, int iNewBitState )
{
  if( iNewBitState != 0 )
   { *pdwDest |= dwBitmask;
   }
  else
   { *pdwDest &= ~dwBitmask;
   }
}

//---------------------------------------------------------------------------
int RigCtrl_StringToUserDefinedBand( // parses a T_RigCtrlUserDefinedBand from a string
        const char **cppSrc,  // [in] line loaded from RCWKeyer_Bands.txt [Bands], or received from remote server,
                // e.g. "i=0 n=10  fr=7037355.0 mo=CWN ..." (space separated key=value pairs, because certain VALUES may be a comma-separated list itself)
                //       |_______| |_|
                //           |      |
                //           |      '--- optional when the line (string) BEGINS with a frequency
                //           '---------- optional, not when 'loading a file'
                // For the [Frequencies] section in RCWKeyer_Bands.txt, the string may be as simple as this:
                //                       // > [Frequencies]  ; new section: "what follows are single frequencies"..
                //      "3.560 mo=CW na=\"CW QRP\" ..."
                // When received from a server connected to a modern Icom radio (especially with Digital Voice),
                //      the to-be-parsed string may contain an awful lot of key=value pairs
                //      that we don't need for remote CW, but keep it in here...
                //      module RigControl.c may be used for very different purposes.
        T_RigCtrlUserDefinedBand *pUserDefdBand, // [out] struct used for a "band" or "sub-band", with frequency range and allowed operating modes, etc
        int *piArrayIndex,    // [out,optional] ARRAY INDEX (NEGATIVE if not specified in the string)
        int *nEntriesInArray) // [out,optional] total number of entries in the array (service for the receiver to detect when all items are through)
  // When successful ("valid syntax"), returns the number of characters
  // parsed from *cppSrc. This allows THE CALLER to continue parsing
  // after the "recognized" part of the source string.
  // Note: Members in T_RigCtrlFreqMemEntry for which no key (name) has
  //       been found in the list of key=value pairs will NOT be modified.
  //       If that's not intended, call RigCtrl_InitFreqMemEntry() before RigCtrl_StringToFreqMemEntry().
  // Callers (not necessarily complete):
  //  * RigCtrl_LoadBandsAndFrequenciesFromFile(), to load RCWKeyer_Bands.txt
  //  * RigCtrl_OnBandStackingRegisterReportFromNetwork(), for client/server communication
  //
  // See also (similar and inverse functions):
  //    RigCtrl_StringToFreqMemEntry(),    RigCtrl_FreqMemEntryToString(),
  //    RigCtrl_StringToUserDefinedBand(), RigCtrl_UserDefinedBandToString() .
{ const char *pszSourceStart = *cppSrc;
  const char *pszKeyStart;
  int iToken, iValue;
  double dblValue;
  BOOL fGotFrequency = FALSE;

  if( piArrayIndex != NULL )
   { *piArrayIndex = -1;   // didn't find "i=.." in the key=value pairs yet
   }
  if( nEntriesInArray != NULL )
   { *nEntriesInArray = -1;   // didn't find "n=.." in the key=value pairs yet
   }
  while( **cppSrc != '\0' ) // continue until the end of the string..
   {                        // .. or until the first unrecognized key (name)
     pszKeyStart = *cppSrc;
     iToken = SL_SkipOneOfNStrings( cppSrc, RigCtrl_UserDefinedBandTokens );
               //   '--> Skips LEADING SPACES, so no need for an extra SL_SkipSpaces()
               // When recognized, iToken may be RIGCTRL_UDEF_BAND_TOKEN_INDEX, .._N_ITEMS,
               //  RIGCTRL_UDEF_BAND_TOKEN_NAME, .._FREQ, .._MODES, .._TX, etc.
     if( iToken < 0 )
      { if( ! fGotFrequency ) // accept the simpler format for USER-DEFINED bands in e.g. RCWKeyer_Bands.txt :
         { // Example: *cppSrc = "3.500-3.570 mo=CWN na=\"80m CW\" " as a shorter alternative,
           //     instead of  "fr=3.500-3.570 mo=CWN na=\"80m CW\" "
           dblValue = SL_ParseFloat( cppSrc, '.'/*cDecimalSeparator*/ ); // band start frequency in MHz
           if( (dblValue > 0 ) && (*cppSrc > pszKeyStart) )
            { pUserDefdBand->dblFmin_Hz = dblValue * 1e6;
              if( SL_SkipChar(cppSrc, '-') ) // here: not "minus" but "to" ..
               { pUserDefdBand->dblFmax_Hz = 1e6 * SL_ParseFloat( cppSrc, '.'/*cDecimalSeparator*/ ); // band end frequency in MHz
               }
              fGotFrequency = TRUE;
              continue; // don't enter the switch( iToken ) list further below but look for the next token
            }
         }
        break; // <- important to avoid getting stuck in the while loop
      }
     switch( iToken )
      { case RIGCTRL_UDEF_BAND_TOKEN_INDEX : // "i=<array index>" -> *piArrayIndex (if non-NULL)
             iValue = SL_ParseInteger( cppSrc );
             if( piArrayIndex != NULL )
              { *piArrayIndex = iValue;
              }
             break;
        case RIGCTRL_UDEF_BAND_TOKEN_N_ITEMS: // "n=<total number of items in the array>" -> *piArrayIndex (if non-NULL)
             iValue = SL_ParseInteger( cppSrc );
             if( nEntriesInArray != NULL )
              { *nEntriesInArray = iValue; // this is a service for the receiver to tell when "all items are through",
                // or to realize a kind of "progress bar" when transferring an awful number of items
              }
             break;
        case RIGCTRL_UDEF_BAND_TOKEN_NAME : // pUserDefdBand->sz16Name (must be double-quoted and "escaped" in the source)
             if( SL_ParseDoubleQuotedString( cppSrc, pUserDefdBand->sz16Name, 16/*iMaxLen*/ ) <= 0 )
              { // obviously not a double-quoted string, so only copy a single word, delimited by space, comma, etc:
                SL_CopyStringUntilDelimiter( cppSrc, pUserDefdBand->sz16Name, 16/*iMaxLen*/, " ,\r\n"/*pszDelimiters*/ );
              }
             break;
        case RIGCTRL_UDEF_BAND_TOKEN_FREQ : // pUserDefdBand->dblFmin_Hz, dblFmax_Hz (separated by "minus", which in this case means "to")
             pUserDefdBand->dblFmin_Hz = 1e6 * SL_ParseFloat( cppSrc, '.'/*cDecimalSeparator*/ );
             if( SL_SkipChar(cppSrc, '-') ) // here: not "minus" but "to" ..
              { pUserDefdBand->dblFmax_Hz = 1e6 * SL_ParseFloat( cppSrc, '.'/*cDecimalSeparator*/ );
              }
             fGotFrequency = TRUE; // don't accept dblFmin_Hz .. dblFmax_Hz without a token now
             break;
        case RIGCTRL_UDEF_BAND_TOKEN_MODES : // e.g. "mo=CW,USB,LSB" -> *piArrayIndex (if non-NULL)
             // inverse to RigCtrl_OperatingModeToString() :
             pUserDefdBand->dwOpModes = RigCtrl_StringToOperatingMode( cppSrc );
             break;
        case RIGCTRL_UDEF_BAND_TOKEN_TX : // "tx=1" : may transmit in this band, "tx=0" : receive-only
             RigCtrl_ModifyBitInDWORD( &pUserDefdBand->dwPermissions, RIGCTRL_PERMISSION_TRANSMIT, SL_ParseInteger( cppSrc ) );
             break;
        case RIGCTRL_UDEF_BAND_TOKEN_DEF_FREQ: // default frequency for the band/frequency selection combo (*)
             pUserDefdBand->dblFdef_Hz = 1e6 * SL_ParseFloat( cppSrc, '.'/*cDecimalSeparator*/ );
             // (*) Used if there are no entries in the [Frequencies] section,
             //     or in the 'Band Stacking Registers' read from certain Icom radios later.
             break;
        case RIGCTRL_UDEF_BAND_TOKEN_DEF_MODE: // default operation mode for the band/frequency selection combo (*)
             pUserDefdBand->dwDefOpMode = (DWORD)SL_SkipOneOfNTokens(cppSrc, RigCtrl_OpModes);
             break;
        default:
             break;
      }
    } // end while( **cppSrc != '\0' )
   return *cppSrc-pszSourceStart; // > .. returns the number of characters parsed ..
} // end RigCtrl_StringToUserDefinedBand()

//---------------------------------------------------------------------------
int RigCtrl_LoadBandsAndFrequenciesFromFile(T_RigCtrlInstance *pRC,
        const char *pszPathAndFilename ) // [in] e.g. "RCWKeyer_Bands.txt" with a full path
  // [out] pRC->UserDefinedBands[], pRC->iNumUserDefinedBands,
  //       and maybe similar "frequency info" in future (thus the name).
  // Returns the NUMBER OF USER-DEFINED FREQUENCY BANDS when successful,
  //         otherwise a negative value (kind of error code).
  //  If NOTHING is loaded from the file, the later-called function
  //         RigCtrl_InitUserDefinedBandsFromRigInfo() will make an
  //         educated guess from the "Tx Band Edges",
  //         which modern Icom radios will report via Ci-V .
  //         That way, the config file (e.g. "RCWKeyer_Bands.txt")
  //         is not mandatory but optional .
{ T_QFile qf;
  int  iCharsPerLine;
  char sz255Src[256];
  const char *cpSrc;
  char *cp;
  const char *ccp;
  enum { SECTION_NONE, SECTION_BANDS, SECTION_FREQUENCIES } enSection;
  int  nBandsLoaded = 0;
  int  nFrequenciesLoaded = 0;
  T_RigCtrlUserDefinedBand userDefdBand;
  T_RigCtrlFreqMemEntry    userDefdFreq;

  enSection = SECTION_NONE;

  if( QFile_Open( &qf, pszPathAndFilename, QFILE_O_RDONLY ) )
   { // Successfully opened the file (e.g. "RCWKeyer_Bands.txt") so process it,
     // line-by-line, to keep things simple ..
     while( (iCharsPerLine=QFile_ReadLine( &qf, sz255Src, 255/*iMaxLen*/ )) >= 0 )
      { if( iCharsPerLine > 0 )  // non-empty line -> parse it !
         { cpSrc        = sz255Src;
           SL_SkipSpaces( &cpSrc );
           if( ( (cpSrc[0] == '/') && (cpSrc[1] == '/') ) // it's a C++ comment so ignore this line
             ||(  cpSrc[0] == ';' ) // it's an Assembler-like comment so also ignore this line
             )
            {
            }
           else // Arrived here ? There MAY be a valid keyword at the begin of this line.
            { // Since there are only a few keywords to recognice, don't use
              // a T_SL_TokenList (from StringLib.h) here, but keep it simple:
              if( SL_SkipToken( &cpSrc, "[Bands]" ) )
               { enSection = SECTION_BANDS;
                 // What follows in the NEXT LINES is a section of BANDS (frequency ranges)
                 // as in the following example:
                 // > [Bands]  ; new section: "what follows is a list of bands"..
                 // > 60m_EU: 5.3515-5.3540 modes=CW tx=1
                 // > 40m: 7.00-7.2 modes=CW,LSB tx=1
                 // > 30m: 10.1-10.13 modes=CW tx=1
               } // end if < line begins with "[Bands]" >
              else if( SL_SkipToken( &cpSrc, "[Frequencies]" ) )
               { enSection = SECTION_FREQUENCIES;
                 // What follows in the NEXT LINES is a list of 'single frequencies'
                 // as in the following example:
                 // > [Frequencies]  ; new section: "what follows are single frequencies"..
                 // > 3.560 mo=CW na="CW QRP"
               } // end if < line begins with "[Frequencies]" >
              else // lines beginning with anything else (no keywords) begin with a user-defined NAME, so:
              switch(enSection) // process the line depending on the last section name :
               { case SECTION_BANDS :
                    if( nBandsLoaded < RIGCTRL_NUM_USER_DEFINED_BANDS )
                     { RigCtrl_InitUserDefinedBand( &userDefdBand );
                       if( RigCtrl_StringToUserDefinedBand( &cpSrc, &userDefdBand, NULL,NULL ) )
                        { pRC->UserDefinedBands[nBandsLoaded++] = userDefdBand;
                          pRC->iNumUserDefinedBands = nBandsLoaded;
                        }
                     }
                    break;
                 case SECTION_FREQUENCIES:
                    if( nFrequenciesLoaded < RIGCTRL_NUM_USER_DEFINED_FREQUENCIES )
                     { RigCtrl_InitFreqMemEntry( &userDefdFreq );
                       if( RigCtrl_StringToFreqMemEntry( &cpSrc, &userDefdFreq, NULL,NULL ) )
                        { pRC->UserDefinedFrequencies[nFrequenciesLoaded++] = userDefdFreq;
                          pRC->iNumUserDefinedFrequencies = nFrequenciesLoaded;
                        }
                     }
                    break;
                 default:
                    break;
               }   // end switch(enSection)
            }     // end else < line may contain keyword or "data" >
         }       // end if < non-empty line >
      }         // end while < more lines read from the text file >
     QFile_Close( &qf );
   } // end if < successfully opened the file >
  return nBandsLoaded;
} // end RigCtrl_LoadBandsAndFrequenciesFromFile()


//--------------------------------------------------------------------------
void RigCtrl_InitUserDefinedBandsFromRigInfo(
             T_RigCtrlInstance *pRC)
             // [in] RigCtrl_ITU_Region
             // [in] pRC->dwAvailableBands, pRC->TxBandEdges[], pRC->iNumTxBands
             // [out] pRC->UserDefinedBands[0..pRC->iNumUserDefinedBands]
  // Called from RigCtrl_Handler() immediately after receiving the rig's own
  //  "Transmit Band Edges", because if the RIG doesn't support a certain band
  //  (or TRANSMIT frequency range), the "User-defined bands" loaded from
  //  configuration file (earlier, by the GUI) will be modified as follows:
  //
  // (1) If pRC->UserDefinedBands[] is EMPTY (i.e. pRC->iNumUserDefinedBands==0),
  //      create new entries in pRC->UserDefinedBands[] based on the
  //      info in pRC->TxBandEdges[] (supplied by the rig itself, at least for Icom).
  //      Note that a single entry in pRC->TxBandEdges[] may create
  //      DOZENS OF NEW ENTRIES in pRC->UserDefinedBands[],
  //      for example if the radio is a IC-7300 modified for TX on the 60 meter band,
  //      which in fact is the "all-band transmit" mod (very unfortunately),
  //      causing the rig to claim being able to transmit from 0.1 to 74.8 MHz
  //      (in that case, with the exclusion of 136 kHz and 472 kHz, all
  //       "known" amateur radio bands will be appended to pRC->UserDefinedBands[]).
  //
  // (2) Check all entries in pRC->UserDefinedBands[x] against the rig's own pRC->TxBandEdges[].
  //      If THE RIG reports not being able to transmit on a band,
  //      clear the flag RIGCTRL_PERMISSION_TRANSMIT in pRC->UserDefinedBands[x].dwPermissions.
  //
  // * See also: RigCtrl_MayTransmitOnFrequencyAndMode(), which relies on the
  //      flag pRC->UserDefinedBands[x].dwPermissions.RIGCTRL_PERMISSION_TRANSMIT.
{
  BOOL fTxPermissionFromRig;
  int iBand, iTxBandEdge;
  double dblFmin_Hz, dblFmax_Hz;
  T_RigCtrlUserDefinedBand *pBand;
  T_RigCtrlFrequencyRange *pTxRange;
  DWORD dwBandMask, dwAvailableBands = RigCtrl_GetAvailableBands( pRC );


  // (1) If pRC->UserDefinedBands[] is EMPTY (i.e. pRC->iNumUserDefinedBands==0),
  //      create new entries in pRC->UserDefinedBands[] based on the
  //      info in pRC->TxBandEdges[] (supplied by the rig itself) :
  if( pRC->iNumUserDefinedBands<=0 ) // no USER-DEFINED BANDS have been imported from a file ?
   { // (if the rig isn't an Icom transceiver, it will probably not have reported anything.
     //  In that case, create a user-defined band for each bit set in dwAvailableBands:
     pRC->iNumUserDefinedBands = 0;
     for( iBand=0; iBand<=RIGCTRL_BAND_MAX_BITNUM; ++iBand )
      { dwBandMask = 1UL<<iBand;  // convert the bit number into a "bitmask"
        RigCtrl_BandToFrequencyRange( dwBandMask, &dblFmin_Hz, &dblFmax_Hz );
        if( (dwBandMask & dwAvailableBands ) // this band is AVAILABLE, and ...
          &&(dblFmin_Hz != RIGCTRL_NOVALUE_DOUBLE)  // .. no objections to use it from RigCtrl_BandToFrequencyRange()
          &&(RigCtrl_GetUserDefinedBandByFrequencyRange(pRC,dblFmin_Hz,dblFmax_Hz)==NULL) // and it's not in pRC->UserDefinedBands[] yet ?
          )
         { if( pRC->iNumUserDefinedBands < RIGCTRL_NUM_USER_DEFINED_BANDS )
            { pBand = &pRC->UserDefinedBands[pRC->iNumUserDefinedBands++]; // occupy a new array entry..
              memset( pBand, 0, sizeof(T_RigCtrlUserDefinedBand) );
              pBand->dblFmin_Hz = dblFmin_Hz;
              pBand->dblFmax_Hz = dblFmax_Hz;
              pBand->dwPermissions = RIGCTRL_PERMISSION_TRANSMIT;
              // ,--------------------'
              // '--> Same as for entries IMPORTED FROM A FILE,
              //      the TX-permission may be withdrawn in the loop below.
              //      But the band remains in the list for RECEPTION.
            }
         }
      }     // end for < all possible amateur radio bands, from HF to SHF >
   }       // end if( pRC->iNumTxBands <= 0 )


  // (2) Check ALREADY EXISTING ENTRIES in pRC->UserDefinedBands[]
  //       against what the RIG has reported in pRC->TxBandEdges[] ...
  for(iBand=0; (iBand<RIGCTRL_NUM_USER_DEFINED_BANDS) && (iBand<pRC->iNumUserDefinedBands); ++iBand)
   { pBand = &pRC->UserDefinedBands[iBand];
     fTxPermissionFromRig = FALSE;
     for( iTxBandEdge=0; (iTxBandEdge<RIGCTRL_NUM_TX_BAND_EDGES) && (iTxBandEdge<pRC->iNumTxBands); ++iTxBandEdge )
      { pTxRange = &pRC->TxBandEdges[iTxBandEdge]; // <- these were reported BY THE RIG,
        if( (pBand->dblFmin_Hz >= pTxRange->dblFmin_Hz) && ( pBand->dblFmax_Hz <= pTxRange->dblFmax_Hz) )
         { // Example (IC-7300 with the "diode mod" to transmit on 60 m):
           //  pRC->iNumTxBands=1, pTxRange->dblFmin_Hz = 100e3, pTxRange->dblFmax_Hz = 74.8e6;
           //  pBand->dblFmin_Hz=1.81e6, pBand->dblFmax_Hz=2.0e6 -> ok, "permission to transmit" !
           //   ,----------------|____|
           //   '--> that's with RigCtrl_ITU_Region=1, see 'wisdom' in RigCtrl_BandToFrequencyRange() !
           fTxPermissionFromRig = TRUE; // "permisstion to transmit" granted..
         }
      }   // end for < all TX BAND EDGES reported earlier by the rig >
     if( !fTxPermissionFromRig ) // no match in any of the rig's own "Tx Band Edges" ->
      { pBand->dwPermissions &= ~RIGCTRL_PERMISSION_TRANSMIT;
      }
   }     // end for < all USER DEFINED BANDS to check against the rig's capabilities >

 }        // end RigCtrl_InitUserDefinedBandsFromRigInfo()


//--------------------------------------------------------------------------
BOOL RigCtrl_MayTransmitOnFrequencyAndMode( T_RigCtrlInstance *pRC,
        double dblOperatingFreq_Hz, // [in] operating frequency, in most cases: CARRIER FREQUENCY
        int    iOpMode ) // [in] RIGCTRL_OPMODE_CW/LSB/USB/.., combineable with _REVERSE, _NARROW, etc.
{
  int iBand;
  T_RigCtrlUserDefinedBand *pBand = RigCtrl_GetUserDefinedBandByFrequencyRange( pRC, dblOperatingFreq_Hz, dblOperatingFreq_Hz );
  // Remember, even when NOT loaded from a file, user-defined bands may exist
  // - see initialisation in RigCtrl_InitUserDefinedBandsFromRigInfo(),
  //   or the transfer of UserDefinedBands[] from server to client via TCP/IP.

#ifdef __BORLANDC__  // "... is assigned a value that is never used".. oh shut up, we know what we're doing !
  (void)pBand;
#endif

  for(iBand=0; (iBand<RIGCTRL_NUM_USER_DEFINED_BANDS) && (iBand<pRC->iNumUserDefinedBands); ++iBand)
   { pBand = &pRC->UserDefinedBands[iBand];

     // ToDo (maybe one fine day): consider the mode-specific bandwidth and UPPER / LOWER sideband.
     if( (dblOperatingFreq_Hz >= pBand->dblFmin_Hz) && (dblOperatingFreq_Hz <= pBand->dblFmax_Hz ) )
      { // Ok, this is the matching frequency range.
        if( pBand->dwPermissions & RIGCTRL_PERMISSION_TRANSMIT )
         { // May TRANSMIT, but also in THIS MODE (iOpMode, CW/LSB/USB/AM/FM) ?
           if( (DWORD)iOpMode & pBand->dwOpModes & RIGCTRL_OPMODE_MASK_TO_STRIP_FLAGS )
            { return TRUE;
            }
         }
        return FALSE; // matching BAND but no permission to TRANSMIT here so bail out
      }
   }

  // Arrived here ? No matching frequency range in the USER DEFINED BANDS.
  // That doesn't mean "must not transmit" - there may no user defined band at all,
  // but just the info from the built-in "rig database", or the rig itself has
  // reported what Icom calls "TX Band Edges" .
  if( pRC->iNumUserDefinedBands > 0 ) // there ARE user-defined bands,
   { // but none of the entries in UserDefinedBands[] gave us the permission
     // to transmit ->
     return FALSE;  // don't transmit on < dblOperatingFreq_Hz >
   }

  // Arrived here ? No user-defined bands at all, so :
  if( RigCtrl_FrequencyToBand(dblOperatingFreq_Hz) & RigCtrl_GetAvailableBands(pRC) )
   { return TRUE;  // frequency is in one of the "available bands" so allow to transmit
   }

  return FALSE;
} // end RigCtrl_MayTransmitOnFrequencyAndMode()


//--------------------------------------------------------------------------
BOOL RigCtrl_SwitchToFreqMemEntry( T_RigCtrlInstance *pRC,
                                   T_RigCtrlFreqMemEntry *pFreqMemEntry )
  // Switches to e.g. a certain "Band Stacking Register" (read out earlier)
  // or a classic 'memory channel'. Both include at least the operating
  // frequency and the operating mode (USB,LSB,CW,..), and maybe an awful
  // lot more (split mode, filter bandwidth, ... ) .
  //
  // Shall work on the server side (directly controlling a radio via 'COM' port)
  //        and on the client side (which sends a few commands via TCP) .
  //
{ BOOL fOk = FALSE;

  // So far, no command has been found in the IC-9700 CI-V manual to
  //         SWITCH TO a certain band stacking register. There only seemed
  //         to be a command to "Send/read band stacking register contents".
  //         So send MULTIPLE COMMANDS to switch frequency, mode, bandwidth,
  //         and whatever else belongs to a band stacking register (future plan).

  if( pFreqMemEntry->RxTx[0].dblOperatingFreq_Hz > 0.0 )
   { fOk = RigCtrl_SetVFOFrequency( pRC, pFreqMemEntry->RxTx[0].dblOperatingFreq_Hz );
   }
  if( pFreqMemEntry->RxTx[0].iOpMode != RIGCTRL_OPMODE_UNKNOWN )
   { fOk |= RigCtrl_SetOperatingMode( pRC, pFreqMemEntry->RxTx[0].iOpMode );
     // ,-----------'
     // '--> Must not immediately send anything yet, because the command
     //      to switch the FREQUENCY must be acknowledged by the remotely
     //      controlled radio first (RIGCTRL_MSGTYPE_OK, in CI-V: unspecific "OK" = cmd 0xFB).
     //
   }

  return fOk;

} // end RigCtrl_SwitchToFreqMemEntry()


//--------------------------------------------------------------------------
BOOL RigCtrl_QueueUpCmdToReadUnifiedPN( T_RigCtrlInstance *pRC, int iUnifiedPN )
  // Non-blocking;  doesn't wait for "arrival" of the response.
  // The return value only indicates if we could *QUEUE UP* the read-command .
  // The rest happens later in RigCtrl_Handler()
  //     -> RigCtrl_SendReadWriteCommandFromFIFO() and RigCtrl_ParseCIV().
  // The read value will LATER be stored in the equivalent member variable,
  // e.g. iUnifiedPN=RIGCTRL_PN_S_METER_LEVEL_DB  ->  pRC->iSMeterLevel_dB .
  //
  // Caller: Usually only the GUI (e.g. TKeyerMainForm::Timer1Timer()),
  //         because the GUI decides which parameters it want to show somewhere.
{
  BOOL fResult = FALSE;

  // Like many other API functions, the actual transmission may have to be
  // postponed (internally) when still busy from an 'uninterruptable'
  //      sequence of commands in RigCtrl_Handler().
  // Since there is a FIFO for this anyway, use it :
  // ex: int iNewFifoHead = (pRC->iReadWritePNFifoHead + 1) % RIGCTRL_READ_WRITE_PN_FIFO_SIZE;
  // ex: if( iNewFifoHead != pRC->iReadWritePNFifoTail )  // FIFO not completely full ?
  if( RigCtrl_GetNumQueuedUpCommandsPending(pRC) < (RIGCTRL_READ_WRITE_PN_FIFO_SIZE-2) )
   { pRC->iReadWritePNFifo[pRC->iReadWritePNFifoHead] = iUnifiedPN & 0x7FFF; // .. with bit 15 (Read/!Write flag) CLEARED
     pRC->iReadWritePNFifoHead = (pRC->iReadWritePNFifoHead + 1) % RIGCTRL_READ_WRITE_PN_FIFO_SIZE;
     fResult = TRUE;  // command to "read something" has been queued up successfully
   }
  else
   { fResult = FALSE; // out of FIFO space to queue up another 'read'-command
     // (got here quite often, because RigCtrl_Handler() was often busy
     //  from receiving spectra from an IC-7300 / IC-9700, and barely had
     //  the chance to ask for new S-meter readings or similar.)
   }

  return fResult;
} // end RigCtrl_QueueUpCmdToReadUnifiedPN()

//--------------------------------------------------------------------------
BOOL RigCtrl_QueueUpCmdToWriteUnifiedPN( T_RigCtrlInstance *pRC, int iUnifiedPN )
  // Similar as RigCtrl_QueueUpCmdToReadUnifiedPN(),
  //  but this function is for delayed WRITING of a parameter identified
  //  by its 'unified' (i.e. non-Icom-specific) parameter number,
  //  e.g. iUnifiedPN = RIGCTRL_PN_TRANSMIT_REQUEST to switch from RX to TX
  //       or vice versa, as soom as time (and the CI-V interface) permits.
  // The return value only indicates if we could *QUEUE UP* the write-command .
  // The rest happens later in RigCtrl_Handler()
  //     -> RigCtrl_SendReadWriteCommandFromFIFO() -> RigCtrl_SendWriteCommandForUnifiedPN().
  // The to-be-written VALUE has already been stored in the equivalent member variable,
  // e.g. iUnifiedPN = RIGCTRL_PN_SCOPE_REF_LEVEL -> pRC->dblScopeRefLevel_dB,
  // so when pulling the to-be-written-parameter-NUMBER from the FIFO later,
  // the to-be-written VALUE can be retrieved from the instance, and translated
  // into a CI-V command .
{
  // Like many other API functions, the actual transmission may have to be
  // postponed (internally) when still busy from an 'uninterruptable'
  //      sequence of commands in RigCtrl_Handler().
  // Since there is a FIFO for this anyway, use it :
  int iNewFifoHead = (pRC->iReadWritePNFifoHead + 1) % RIGCTRL_READ_WRITE_PN_FIFO_SIZE;
  if( iNewFifoHead != pRC->iReadWritePNFifoTail )  // FIFO not completely full ?
   { pRC->iReadWritePNFifo[pRC->iReadWritePNFifoHead] = iUnifiedPN | 0x8000; // .. with bit 15 SET to mark this as a "WRITE-command"
     pRC->iReadWritePNFifoHead = iNewFifoHead;
     return TRUE;  // command to "write this PN" has been queued up successfully so don't queue it up TWICE .
     // Again: The actual transmission happens a few milliseconds later,
     //  in RigCtrl_SendReadWriteCommandFromFIFO() -> RigCtrl_SendWriteCommandForUnifiedPN().
   }
  else
   { return FALSE; // out of FIFO space to queue up another 'read'-command
     // (got here quite often, because RigCtrl_Handler() was often busy
     //  from receiving spectra from an IC-7300 / IC-9700, and barely had
     //  the chance to ask for new S-meter readings or similar.
     //  For PERIODICALLY READ messages that's not a problem, but when this
     //  happened when RCW Keyer tried to switch back from TRANSMIT to RECEIVE,
     //  it was. Fixed by always leaving TWO EMPTY LOCATIONS in pRC->iReadWritePNFifo[],
     //  by calling RigCtrl_GetNumQueuedUpCommandsPending() from RigCtrl_QueueUpCmdToReadUnifiedPN()
     //  instead of letting the FIFO run "completely full" !
   }
} // end RigCtrl_QueueUpCmdToWriteUnifiedPN()

//--------------------------------------------------------------------------
int RigCtrl_GetNumQueuedUpCommandsPending( T_RigCtrlInstance *pRC )
   // '--> applies to RigCtrl_QueueUpCmdToReadUnifiedPN() / ..WriteUnifiedPN().
   //  The Keyer GUI uses this function to decided how "busy" we are at the
   //  moment, to poll stuff like the 'S-Meter', 'Transmit Flag',
   //          TX Power Meter(!), ALC Meter, SWR Meter, Drain Current,
   //          Supply Voltage (monitored especially when TRANSMITTING,
   //                          to check the health of the remote station's
   //                          main power supply),  etc etc.
{
  int iNumQueueEntries = pRC->iReadWritePNFifoHead - pRC->iReadWritePNFifoTail;
  if( iNumQueueEntries < 0 )  // circular buffer wrap ?
   {  iNumQueueEntries += RIGCTRL_READ_WRITE_PN_FIFO_SIZE;
   }
  return iNumQueueEntries;
} // end RigCtrl_GetNumQueuedUpCommandsPending()


//--------------------------------------------------------------------------
int RigCtrl_GetAgeOfBasicParams_ms( T_RigCtrlInstance *pRC )
  // Added 2025-07 for the GUI, because stoneage radios like Yaesu FT8x7
  // don't send unsolitited reports when the operator turns the VFO knob,
  // selects a new mode, etc directly on the radio.
  // This function returns the age (in milliseconds) of the last
  // "basic parameters" which, at the time of this writing, consisted of
  //    RIGCTRL_PN_FREQUENCY and RIGCTRL_PN_OP_MODE .
  // For example, the FT-817 uses Yaesu's "CAT" command 0x03
  //               = "Read Frequency and mode" to read BOTH in a single command.
{
  return TIM_ReadStopwatch_ms( &pRC->sw_LastUpdateOfBasicParams );
} // end RigCtrl_GetAgeOfBasicParams_ms()


//--------------------------------------------------------------------------
BOOL RigCtrl_QueueUpCmdToReadBasicParams( T_RigCtrlInstance *pRC )
  // See RigCtrl_GetAgeOfBasicParams_ms() for the purpose of this function.
  //     RigCtrl_QueueUpCmdToReadBasicParams() does whatever it takes to
  //     READ those "basic parameters" (e.g. Frequency + Mode),
  //     but returns to the caller BEFORE the response has arrived.
  // See also (in RCW Keyer) : Polling of a few 'non-BASIC-parameters'
  //                           for the GUI in TKeyerMainForm::Timer1Timer() .
{
  T_RigCtrl_PortInstance *pRadioPortInstance = &pRC->PortInstance[RIGCTRL_PORT_RADIO];

  TIM_StartStopwatch( &pRC->sw_LastUpdateOfBasicParams );

  // If we REALLY need to cause unnecessary traffic by periodically polling
  // "basic parameters" depends on the protocol. For modern protocols,
  // where radios can send UNSOLICITED REPORTS (aka "event-driven"),
  // we don't need that. For protocols like Yaesu's old "CAT", we must. Meow !
  switch( pRadioPortInstance->iRadioCtrlProtocol )
   {
     case RIGCTRL_PROTOCOL_ICOM_CI_V : // no need to POLL because THE RADIO can send unsolicited reports !
        // (only poorly designed 'radio control software' sometimes turns
        //  UNSOLICITED reports off, because it cannot handle such messages. CRAP ! )
        break;

#   if( SWI_SUPPORT_YAESU_5_BYTE_CAT )
     case RIGCTRL_PROTOCOL_YAESU_5_BYTE :
        if( ! pRC->Y5B.fBusy )
         { return Yaesu5Byte_AskForParameter( &pRC->Y5B, RIGCTRL_PN_FREQUENCY );
         }
        else // Yaesu5Byte.c already "busy" from a command transmission ->
         { return RigCtrl_QueueUpCmdToWriteUnifiedPN( pRC, RIGCTRL_PN_FREQUENCY );
         }
#   endif // SWI_SUPPORT_YAESU_5_BYTE_CAT ?

     default:
        // Arrived here ? The rig doen't seem to have a special command to query
        //                the 'basic parameters', so queue-up commands (plural)
        //                to read them one-by-one:
        return RigCtrl_QueueUpCmdToReadUnifiedPN( pRC, RIGCTRL_PN_FREQUENCY )
            && RigCtrl_QueueUpCmdToReadUnifiedPN( pRC, RIGCTRL_PN_OP_MODE );
   }

  return FALSE;

} // end RigCtrl_QueueUpCmdToReadBasicParams()

//--------------------------------------------------------------------------
static void RigCtrl_SendReadWriteCommandFromFIFO( T_RigCtrlInstance *pRC )
  // [in] pRC->iReadWritePNFifo[pRC->iReadWritePNFifoTail++]
  //
  // Only called from RigCtrl_Handler() when another parameter needs to be
  // polled (e.g. RIGCTRL_PN_S_METER_LEVEL_DB), and the current state
  // ALLOWS sending another frame to the locally connected rig .
  //
  // Note: This function is not suitable for 'unknown'commands received on an
  //       Additional COM Port (they don't have a unified parameter number
  //       for this queue).
  //       Such commands (and ONLY THEY) will be forwarded to the RADIO PORT
  //       via RigCtrl_ForwardCommandFromServerPortToRadio() .
  //
  // The to-be-read "unified parameter" has been requested by the application
  // via RigCtrl_QueueUpCmdToReadUnifiedPN( T_RigCtrlInstance *pRC, int iUnifiedPN );
{ int iUnifiedPN = pRC->iReadWritePNFifo[pRC->iReadWritePNFifoTail];
      // '--> At this point, bit 15 is the "WRITE / !read flag"
  if( pRC->iReadWritePNFifoTail == pRC->iReadWritePNFifoHead ) // oops.. the FIFO for queued-up Read- and Write-commands is EMPTY !
   { return;
   }

  // REMOVE the entry from the FIFO (Queue) .. just a small array of integers:
  pRC->iReadWritePNFifoTail = (pRC->iReadWritePNFifoTail+1) % RIGCTRL_READ_WRITE_PN_FIFO_SIZE;

  if( iUnifiedPN & 0x8000 )
   { RigCtrl_SendWriteCommandForUnifiedPN( pRC, iUnifiedPN & 0x7FFF );
   }
  else
   { RigCtrl_SendReadCommandForUnifiedPN( pRC, iUnifiedPN );
   }

} // end RigCtrl_SendReadWriteCommandFromFIFO()


//--------------------------------------------------------------------------
static BOOL RigCtrl_SendReadCommandForUnifiedPN( T_RigCtrlInstance *pRC, int iUnifiedPN)
  // Called from RigCtrl_Handler() -> RigCtrl_SendReadWriteCommandFromFIFO()
  //        when another parameter shall be polled
  //        (e.g. RIGCTRL_PN_S_METER_LEVEL_DB), and the current state
  // ALLOWS sending another frame to the remotely controlled rig.
  //
  // The to-be-read "unified parameter" has been requested by the application
  // via RigCtrl_QueueUpCmdToReadUnifiedPN( T_RigCtrlInstance *pRC, int iUnifiedPN );
{
  T_RigCtrl_PortInstance *pPortInstance = &pRC->PortInstance[RIGCTRL_PORT_RADIO];
  int iCmd=-1, iSubCmd =-1, iDataByte1=-1, iDataByte2=-1;
  int iCrazyModelDependingSubSubcodeForCmd1A_05 = 0;
  BYTE bBCD[2]; // 4-digit BCD in TWO BYTES (to generate the "crazy sub-sub-code" for certain CI-V commands)
  char sz40Comment[84];
  T_RigCtrl_ParamInfo *pPI = RigCtrl_GetInfoForUnifiedParameterNumber(iUnifiedPN);
  int iMsgType = RIGCTRL_MSGTYPE_UNKNOWN;
  if( pPI != NULL )
   { iMsgType = pPI->iMsgType; // <- avoids a crowded traffic log...
     // by FILTERING in RigCtrl_SendShortCIVMessage(), based on the MESSAGE TYPE (~~category)
     sprintf( sz40Comment, "read %s", pPI->pszToken );
   }
  else
   { strcpy( sz40Comment, "read param" ); // rather meaningless default, but better than nothing
   }

  switch( pPortInstance->iRadioCtrlProtocol )
   {  // which language does the radio speak ?
    case RIGCTRL_PROTOCOL_NONE :
    default:
         break;  // unknown protocol, can only send "direct" strings

    case RIGCTRL_PROTOCOL_ICOM_CI_V : // Icom's "CI-V" protocol ...
         // What's the 'command' and 'subcommand' in CI-V to *READ* this parameter ?
         switch( iUnifiedPN )
          { // Various "meter" readings via CI-V,  command 0x15 --->
            // [IC7300FM] page 19-3 : cmd 0x15 subcode 2 = Read S-meter level, etc
            case RIGCTRL_PN_SQUELCH_STATUS     : iCmd=0x15; iSubCmd=0x01; break;
            case RIGCTRL_PN_S_METER_LEVEL_DB   : iCmd=0x15; iSubCmd=0x02; break;
            case RIGCTRL_PN_POWER_METER_PERCENT: iCmd=0x15; iSubCmd=0x11; break;
            case RIGCTRL_PN_SWR_METER_VALUE    : iCmd=0x15; iSubCmd=0x12; break;
            case RIGCTRL_PN_ALC_METER_LEVEL    : iCmd=0x15; iSubCmd=0x13; break;
            case RIGCTRL_PN_COMP_METER_LEVEL_DB: iCmd=0x15; iSubCmd=0x14; break;
            case RIGCTRL_PN_SUPPLY_VOLTAGE_mV  : iCmd=0x15; iSubCmd=0x15; break;
            case RIGCTRL_PN_DRAIN_CURRENT_mA   : iCmd=0x15; iSubCmd=0x16; break;
            case RIGCTRL_PN_WATERFALL_SPEED : // this is one of those dreadful beasts with a MODEL-DEPENDENT "sub-subcommand" !
            default:  // a few parameters can only be read though the crazy rig-specific cmd 0x1A sub 0x15, followed by a 4-digit 'Sub-Subcode':
               // followed by a CRAZY MODEL-DEPENDING SUB-SUBCODE :
               iCrazyModelDependingSubSubcodeForCmd1A_05 = RigCtrl_GetCrazyModelDependingSubSubcodeForCmd1A_05( pRC->iDefaultAddress, iUnifiedPN );
               if( iCrazyModelDependingSubSubcodeForCmd1A_05 > 0 )
                { iCmd = 0x1A;
                  iSubCmd = 0x05;
                  RigCtrl_IntToBCD_CIV_MSByteFirst( iCrazyModelDependingSubSubcodeForCmd1A_05, bBCD, 4/*nDigits*/ );
                  iDataByte1 = bBCD[0]; // thousands and hundreds of the "crazy model-depending four-digit parameter number" for cmd 0x1A sub 0x05
                  iDataByte2 = bBCD[1]; // tens and ones of the "crazy model-depending four-digit parameter number" for cmd 0x1A sub 0x05
                  // 2025-09-18 : After adding this, RCWKeyer could check if an IC-7300
                  //              sends the locally generated CW SIDETONE through the USB audio interface
                  // (they called it "Send/read beep and speech output setting to ACC/USB" in the IC-7300 manual,
                  //  but renamed it to "SET > Connectors > ACC AF/IF Output > AF Beep/Speech... Output"
                  //  in the IC-9700 CI-V reference manual, just to confuse YOU, dear fellow developer.)
                }
               else // iCrazyModelDependingSubSubcodeForCmd1A_05 <= 0 ->
                { // guess is a simple old CI-V command "understood" by different radios.
                  // In that case, *pPI should contain everything we need to know :
                  if( pPI != NULL )
                   { if( (pPI->pbCIVReadCmd != NULL) && (pPI->iCIVReadCmdLength > 0) )
                      { iCmd = pPI->pbCIVReadCmd[0];  // e.g. a phantastically simple comand like 0x03 = "Read the operating frequency" : ONE byte !
                        if( pPI->iCIVReadCmdLength >= 2 )
                         { iSubCmd = pPI->pbCIVReadCmd[1]; // e.g. a TWO-BYTE-COMMAND like 0x19 0x00 = "Read the transceiver ID"
                         }
                        if( pPI->iCIVReadCmdLength >= 3 )
                         { iDataByte1 = pPI->pbCIVReadCmd[2];
                         }
                        if( pPI->iCIVReadCmdLength >= 4 )
                         { iDataByte2 = pPI->pbCIVReadCmd[3];
                         }
                      }
                   }
                }
               break; // leave iCmd negative to ignore the read-request
          } // end switch( iUnifiedPN )
         if( iCmd>=0 )
          { if( RigCtrl_SendShortCIVMessage( pRC, RIGCTRL_PORT_RADIO, RIGCTRL_ORIGIN_CONTROLLER,
                  iMsgType,       // [in] message type (category) for the traffic log
                  sz40Comment,    // [in] short comment for the traffic log
                  iCmd, iSubCmd,  // e.g. command 0x1A, sub-command 0x05 for the "crazy" stuff
                  iDataByte1,     // optional 1st parameter byte. -1 to omit.
                  iDataByte2 ) )  // optional 2nd parameter byte. -1 to omit.
             { // If all works as planned, the transceiver responds with the same 'cmd' byte:
               pPortInstance->dwExpectedResponse_CmdAndSubcode = RCTL_COMBINE_CMD_AND_SUB(iCmd,iSubCmd); // e.g. cmd=0x15 subcmd=0x12 -> dwExpectedResponse_CmdAndSubcode = 0x1512....
               //              '--> For the "crazy stuff" (cmd 0x1A sub 0x05), this isn't sufficient,
               //                   thus also remember the UNIFIED PARAMETER NUMBER :
               pPortInstance->iExpectResponseForUnifiedPN = iUnifiedPN;
               pPortInstance->iResponseCountdown_ms = pRC->iMaxResponseDelay_ms; // now waiting for the radio to respond after asking for yet another parameter..
               // In SL's CAT traffic monitor, the full 'data exchange' looked like this ...
               // > 20:57:48.4 TX 007 FE FE 94 E0 15 02 FD        (other) ; read param
               // > 20:57:48.5 RX 007 FE FE 94 E0 15 02 FD        (echo)
               // > 20:57:48.5 RX 009 FE FE E0 94 15 02 01 23 FD  (other) ; SMeterLevel:123
               // ... and the data arrived a bit later in RigCtrl_ParseCIV(),
               //     which scaled the result ('123' in BCD) into 'dB over S0'
               //     and updated pRC->iSMeterLevel_dB, from where Spectrum Lab
               //     polled it, and distributed it to OpenWebRX-alike clients.
               return TRUE;
             }
            else // failed to send the "read"-command...
             { // better luck next time !
             }
          }
         break;  // end case <ICOM CI-V protocol>

   } // end switch( pRC->iRadioCtrlProtocol )
  return FALSE;
} // end RigCtrl_SendReadCommandForUnifiedPN()

//--------------------------------------------------------------------------
int RigCtrl_GetSizeOfCIVDataType( int iCIVDataType ) // .. in BYTES (what else) !
{
  switch( iCIVDataType & CIV_DTYPE_MASK_FOR_BASIC_TYPE )
   { case CIV_DTYPE_NONE : return 0;   // dummy for "no data"
     case CIV_DTYPE_BCD1 : // BCD with only ONE digit (in the data byte bits 3..0)
     case CIV_DTYPE_BCD2 : // two-digit BCD (tens in the bytes bits 7..4, ones in bits 3..0)
     case CIV_DTYPE_HEX_1BYTE: // a single byte treated as TWO-DIGIT HEX (e.g. the "Transceiver Default ID", 0xA2 = IC-9700 proves IT'S NOT ALWAYS BCD)
          return 1;
     case CIV_DTYPE_BCD3 : // three-digit BCD (also least significant digit FIRST)
     case CIV_DTYPE_BCD4 : // FOUR-DIGIT BCD .. warning; often bitwise ORed to some of the flags further below !
          return 2;
     case CIV_DTYPE_BCD5 : // etc, etc, ...
     case CIV_DTYPE_BCD6 : // SIX-DIGIT BCD as used for "Duplex Offset frequency setting" in IC-9700
          return 3;
     case CIV_DTYPE_BCD7 :
     case CIV_DTYPE_BCD8 :
          return 4;
     case CIV_DTYPE_BCD9 :
     case CIV_DTYPE_BCD10: // TEN-digit BCD as used for VFO- and "Transmit"-frequencies in IC-7300, IC-9700, etc)
          return 5;
     case CIV_DTYPE_BAND_EDGE_FREQUENCIES: // 2-digit "edge number" + 12..14-digit "Lower edge" + 0x2D as "Separator" + 12..14-digit "Higher edge"
          // (Like the "VFO frequency", for which the type IS NOT ALWAYS BCD10,
          //  we'd need to know the exact radio model.. so for the moment, ignore this)
          return 1/*edgeNr*/ + 7/*LowerEdge*/ + 1/*Sep*/ + 7/*HigherEdge*/;
     default:  // not supported here yet..
          return 0;
   } // end switch( iCIVDataType & CIV_DTYPE_MASK_FOR_BASIC_TYPE )
} // end RigCtrl_GetSizeOfCIVDataType()

//---------------------------------------------------------------------------
int RigCtrl_MapCIVDataToMessage( BYTE *pbDest, int iCIVDataType, long i32Value )
  // Moved into a new function to call it also from RigControl_CIV_Server.c :
  //                             RigCtrl_CIV_Server_MapDataIntoReadResponse()
{ BYTE *pbDestStart = pbDest;
  BOOL fAppendSignAsSuffix=FALSE,  fNegative=FALSE;
  int  iCIVBasicType = iCIVDataType & CIV_DTYPE_MASK_FOR_BASIC_TYPE; // strip extra flags like CIV_DTYPE_FLAG_PERCENT255
  int  nDigits;

  if( iCIVDataType & CIV_DTYPE_SUFFIX_BCD2_SIGN ) // send with Icom's esoteric "sign SUFFIX" (sign as two-digit BCD *after* the value it applies to) ?
   { fAppendSignAsSuffix = TRUE;
     if( i32Value < 0 )
      {  i32Value = -i32Value;
         fNegative = TRUE;
      }
   }
  if( iCIVDataType & CIV_DTYPE_FLAG_PERCENT255 ) // even if the 'basic' type is 'CIV_DTYPE_BCD4', the parameter is a PERCENTAGE (0..100 in the internal value), but for CI-V it must be scaled to 0..255 !
   { // Scale i32Value from 0..100 [PERCENT, as in pRC->iAudioVolume_Percent]
     //           to Icom's 0..255 [0 for "max counterclockwise", 255 for "max clockwise"] :
     i32Value = (255*i32Value+50/*round*/) / 100;  // e.g. for RIGCTRL_PN_AUDIO_VOLUME_PERCENT  -> cmd 0x14 sub 0x01, and MANY MORE(!)
     RigCtrl_LimitInt32( &i32Value, 0, 255 );
   }
  if( (iCIVBasicType >= CIV_DTYPE_BCD1) && (iCIVBasicType <= CIV_DTYPE_BCD10) )
   { nDigits = 1 + (iCIVBasicType - CIV_DTYPE_BCD1);
     // Note the INCOMPATIBLE BCD byte orders in Icom's CI-V ! Examples:
     //  * "0255" (the maximum for e.g. Icom's "AF Level", cmd 0x14 sub 0x01)
     //    is sent with the MOST SIGNIFICANT BYTE FIRST (here: with the thousands and hundreds)
     //            and the LEAST SIGNIFICANT BYTE LAST (here: with the tens and ones)
     //  * FREQUENCIES (like Bandscope edge frequencies, cmd 0x1A sub 0x05 ..)
     //    are usually sent with the LEAST SIGNIFICANT BYTE FIRST (e.g. with the 1 kHz and 100 Hz digit)
     //            and the MOST SIGNIFICANT BYTE LAST (e.g. with the 1 GHz and 100 MHz digit)
     //  * LATITUDES, LONGITUDES, ALTITUDES (e.g. in "Manually entered position data")
     //     were sent with the MOST SIGNIFICANT BYTE FIRST (e.g. tens and ones of a degree)
     //                and the LEAST SIGNIFICANT BYTE LAST (often the two 'fixed to zero' digits)
     //  * RGB COLOUR MIXTURES (as in Icom's cmd 0x1A sub 0x05 0194..)
     //     were sent as four-digit numbers, and again with the
     //                     MOST SIGNIFICANT BYTE FIRST (thousands and hundreds)
     //            and the LEAST SIGNIFICANT BYTE LAST  (tens and ones) .
     //  AAAAAAARGHHH ! Fixed by frying an 'extra sausage' for such parameters below.
     if( iCIVDataType & CIV_DTYPE_FLAG_MSBYTE_FIRST )
      { RigCtrl_IntToBCD_CIV_MSByteFirst( i32Value, pbDest, nDigits );
      }
     else // generate Icom's CIV-BCD-Flavour with the LEAST SIGNIFICANT two digits (in one byte) first:
      { RigCtrl_IntToBCD_CIV_LSByteFirst( i32Value, pbDest, nDigits );
      }
     pbDest += (nDigits+1) / 2; // e.g. for nDigits=1 or 2, RigCtrl_IntToBCD_CIV_LSByteFirst() has appended ONE byte of 'data'
     if( fAppendSignAsSuffix ) // from CIV_DTYPE_SUFFIX_BCD2_SIGN, first seen in Icom's "Scope Reference level" ...
      { // again: Icom treats a number's sign as a TWO-DIGIT SUFFIX, not as a PREFIX !
        if( fNegative )
         { *pbDest++ = (BYTE)0x01;  // "01" (as sign SUFFIX) means NEGATIVE in CI-V
         }
        else
         { *pbDest++ = (BYTE)0x00;  // "00" (as sign SUFFIX) means POSITIVE in CI-V
         }
      }  // end if < append the value's SIGN as a two-digit BCD *SUFFIX* > ?
   }    // end if < some kind of Binary-Coded-Decimal with anything from 1 to 10 digits > ?
  else // some exotic Icom-specific data type ->
  switch( iCIVBasicType )
   { case CIV_DTYPE_BAND_EDGE_FREQUENCIES: // 2-digit "edge number" + 12..14-digit "Lower edge" + 0x2D as "Separator" + 12..14-digit "Higher edge"

     case CIV_DTYPE_HEX_1BYTE : // not 2-digit BCD but a SINGLE HEXADECIMAL BYTE, sent "as such".
        // Added 2025-08-18 for CI-V command 0x19 sub 0x00, which would identify an
        // IC-9700 as single hexadecimal byte-value 0xA2. With other radios, e.g.
        // IC-7300 with "default CI-V address" 0x94, one may guess it's DECIMAL, but it's not !
        *pbDest++ = (BYTE)i32Value;
        break;

     default:
        break;  // not supported here yet ..
   }
  return pbDest - pbDestStart;  // returns THE NUMBER OF BYTES appended (that's NOT the total CI-V message length)
} // end RigCtrl_MapCIVDataToMessage()


//--------------------------------------------------------------------------
static BOOL RigCtrl_SendWriteCommandForUnifiedPN( T_RigCtrlInstance *pRC, int iUnifiedPN )
  // Similar as RigCtrl_SendReadCommandForUnifiedPN(),
  //    but THIS functions sends a WRITE command for the specified parameter number
  //    to the locally (USB-)connected radio, usually via Icom CI-V protocol.
  // [in] iPN : 'unified parameter number'. Identifies WHAT to send (write),
  //                 for example RIGCTRL_PN_TRANSMIT_REQUEST,
  //                             RIGCTRL_PN_SIDETONE_ON_USB_AUDIO, etc^10 .
  // [in] pRC->xyz : contains the TO-BE-WRITTEN VALUE,
  //                 for example pRC->iTransmitReqst .
  //
  // Called from RigCtrl_Handler() -> RigCtrl_SendReadWriteCommandFromFIFO()
  //        when another parameter needs to be written to the 'real' radio
  //        for which we have a unique "unified parameter number".
  //        If that's not the case .. see
  //
  // The to-be-written "unified parameter number" may have been queued up
  //     by API functions like RigCtrl_SetTransmitRequest()
  //                or RigCtrl_QueueUpCmdToWriteUnifiedPN() .
{
  T_RigCtrl_PortInstance *pPortInstance = &pRC->PortInstance[RIGCTRL_PORT_RADIO];
  int iCmd=-1, iSubCmd =-1, iSubSubCmd =-1, iSubSubSubCmd =-1; // aaaargh ! ! !
  int iCrazyModelDependingSubSubcode = 0; // aaaaaaaaargh ! ! ! (thank you, Icom)
  T_RigCtrl_ParamInfo *pPI = RigCtrl_GetInfoForUnifiedParameterNumber(iUnifiedPN);
  long i32Value = RigCtrl_GetParamByPN_Int( pRC, iUnifiedPN );
  double dblValue = (double)i32Value; // don't call RigCtrl_GetParamByPN_Double() unless we really need it (still considering porting this to a microcontroller)
  char sz40Comment[44] = "write param";  // <- just a default, used if we don't know better
  char *pszDest = sz40Comment;
  const char *pszEndstop = sz40Comment + 40;
  BYTE b8BCD[8];
  BYTE b16TxBuffer[16];
  BYTE *pbBufPtr = b16TxBuffer;
  int  nDigits, iMsgLength, iCIVDataType, iMsgType;
  BOOL fOk = FALSE;

  if( pPI == NULL ) // oops.. could not retrieve any info about the 'unified parameter' (by number)
   { return FALSE;  // no parameter-info at all-> bail out ...
   }
#if(SWI_HARDCORE_DEBUGGING)
  if( RigCtrl_iConditionalBreakpointStep == 2 ) // forwarding a "SET Split Mode" command from an AUX COM PORT to the real radio ?
   {  RigCtrl_iConditionalBreakpointStep = 3;   // just entered RigCtrl_SendWriteCommandForUnifiedPN() for the "Set Split Mode"-command !
      // 2025-10-19 : Got here when firing up N1MM Logger+, but the FORWARDED "Set Split Mode"-command
      //              wasn't correctly sent to the real radio (IC-7300) .
   }
#endif // SWI_HARDCORE_DEBUGGING ?

  iMsgType      = pPI->iMsgType | RIGCTRL_MSGTYPE_FLAG_WRITE_CMD | RIGCTRL_MSGTYPE_FLAG_TX | RIGCTRL_MSGTYPE_FLAG_CIV; // message type for the traffic log
  iCIVDataType  = pPI->iCIVDataType;
  if( pPI->pbCIVReadCmd != NULL ) // the "write"- command is usually the "read"-command followed by the value,
   { // .. but this is only an 'initial assumption'; see list of special cases below
     //   (especially the megaton of "crazy model-dependent sub-subcodes for command 0x1A subcode 0x05) !
     // Note that even if a certain parameter is accessable via the 'crazy' set of commands 0x1A subcode 0x05,
     // it MUST have a non-NULL pointer to the 'CI-V read command' (template) in RigCtrl_ParameterInfo[],
     // because otherwise, RigCtrl_SendWriteCommandForUnifiedPN() would not even try
     // to SEND a write(!)-command for that parameter at all.
     // "If we don't know how to READ it, we won't try to WRITE it".
     iCmd = pPI->pbCIVReadCmd[0];
     if( pPI->iCIVReadCmdLength > 1 )
      { iSubCmd = pPI->pbCIVReadCmd[1]; // 'command' followed by this 'sub command'
      }
     if( pPI->iCIVReadCmdLength > 2 )
      { iSubSubCmd = pPI->pbCIVReadCmd[2]; // this "sub-sub-command" is often the VFO- or SCOPE-number.
        // See table entry for RIGCTRL_PN_SCOPE_REF_LEVEL in RigCtrl_ParameterInfo[] :
        //  pPI->iCIVReadCmdLength = 3 (!)  [that's the length without any data]
        //  pPI->pbCIVReadCmd[] = { 0x27,  0x19, 0x00  }
        //    Command ----------------'      |     |
        //    Sub-Command: Scope Ref Level --'     |
        //    Sub-Sub-Command, here: 00 = main scope,
        //                           01 = sub scope (if we had an IC-7610) .
        // MEEP-MEEP-MEEP .. here we go again: Many four-digit "sub-sub-sub-commands"
        // for Icom's cmd 0x1A, sub-cmd 0x05 are MODEL-SPECIFIC ! Thus:
        if(  (iCmd==0x1A) && (iSubCmd==0x05) && (pPortInstance->iRadioCtrlProtocol==RIGCTRL_PROTOCOL_ICOM_CI_V) )
         { iCrazyModelDependingSubSubcode = RigCtrl_GetCrazyModelDependingSubSubcodeForCmd1A_05(
                                              pRC->iDefaultAddress, iUnifiedPN );
           // '--> a DECIMAL, FOUR-DIGIT CODE that you will only find
           //      in YOUR PARTICULAR RADIO'S "Full Manual" or even a separate
           //      "CI-V REFERENCE GUIDE" (e.g. A7508-3EX-4 for the IC-9700) !
           //
           if( iCrazyModelDependingSubSubcode > 0 ) // SUCCESS .. got the "crazy four-digit number" for cmd 0x1A sub 0x05 !
            { RigCtrl_IntToBCD_CIV_MSByteFirst( iCrazyModelDependingSubSubcode, b8BCD, 4/*nDigits*/ );
              // ,--------------'
              // '--> MOST SIGNIFICANT TWO DIGITS (in one byte) FIRST ! Example:
              //  iCrazyModelDependingSubSubcode = 198 -> b8BCD[0] = 0x01, b8BCD[1] = 0x98  .
              iSubSubCmd    = b8BCD[0];
              iSubSubSubCmd = b8BCD[1];
              // Example to write RIGCTRL_PN_SIDETONE_ON_USB_AUDIO in an IC-7300:
              //  iCmd=0x1A, iSubCmd=0x05, iCrazyModelDependingSubSubcode = #0062
              //  -> iSubSubCmd = 0x00, iSubSubSubCmd = 0x62 = #98 .
              // With iCIVDataType = CIV_DTYPE_BCD2 and i32Value = 0 (to turn the rig-generated sidetone "ON USB" off),
              // the CI-V WRITE COMMAND actually sent (and displayed on the DEBUG tab)
              // was
            }
         } // end if < crazy (Icom-)command 0x1A sub 0x05 ? >
      }   // end if( pPI->iCIVReadCmdLength > 2 )
   }     // end if( pPI->pbCIVReadCmd != NULL )
  if( pPI->pszToken != NULL )
   { SL_AppendPrintf( &pszDest, pszEndstop, "write %s", pPI->pszToken );
   }

  switch( pPortInstance->iRadioCtrlProtocol )
   {  // which language does the radio *ON THIS PORT* speak ?
    case RIGCTRL_PROTOCOL_NONE :
    default:
        break;  // unknown protocol, can only send "direct" strings

    case RIGCTRL_PROTOCOL_ICOM_CI_V : // Icom's "CI-V" protocol ...
        // Anything special about the 'command' and 'subcommand' in CI-V to *WRITE* this parameter ?
        //  Beware: Similar special cases must be handled when sending a
        //       WRITE COMMAND on the CIV-CLIENT side [RigCtrl_SendWriteCommandForUnifiedPN()]
        //   and when assembling a
        //       READ RESPONSE on the CIV-SERVER size [RigCtrl_CIV_Server_MapDataIntoReadResponse()] !
        switch( iUnifiedPN )
         { case RIGCTRL_PN_FREQUENCY: // One of the dreadful special cases where the
              // WRITE-command isn't just the READ-command follwed by WRITE-DATA:
              // A7508-3EX-4 page 4 : 0x03 = "READ the operating frequency".
              //                      0x05 = "SET the operating frequency".
              // The parameter info table (RigCtrl_ParameterInfo[]) only contains the code for the READ-command (0x03).
              // Since there was already an internal 'Set'-function use it:
              return RigCtrl_SetVFOFrequency_Internal( pRC, (double)i32Value/*dblFreqHz*/ );
                     // '--> Aware of subtle incompatibilities between
                     //      different Icom radios (number of DIGITS to send).
           case RIGCTRL_PN_OP_MODE : // Another of the dreadful special cases where the WRITE-command isn't just the READ-command follwed by WRITE-DATA...
              // Since there was already an internal 'Set'-function for this, use it:
              return RigCtrl_SetOperatingMode_Internal( pRC, i32Value/*iOpMode*/ );
           case RIGCTRL_PN_TRANSMIT_REQUEST :
              iCmd=0x1C; iSubCmd=0x00; // A7508-3EX-4 page 11 : "Send/read the transceivers status (00=RX, 01=TX)"
              break;
           case RIGCTRL_PN_SCOPE_SPAN :
              return RigCtrl_SetScopeSpan_Internal( pRC );
   // ex:  case RIGCTRL_PN_WATERFALL_SPEED : // this is one of those dreadful beasts with a MODEL-DEPENDENT "sub-subcommand" !
   //          iCmd=0x1A; iSubCmd=0x05;
   //          strcpy( sz40Comment, "set WF-Speed" );
   //          switch( pRC->iDefaultAddress ) // what's the code after 0x1A 0x05 for the "Waterfall Speed" ?
   //           { case RIGCTRL_DEF_ADDR_IC_7300 : // 0x1A 0x05 "0108" for the "Waterfall Speed" in an IC-7300
   //                  iSubSubCmd=0x01; iSubSubSubCmd=0x08; // aaaaargh .. "thank you, Icom" !
   //                  break;
   //             case RIGCTRL_DEF_ADDR_IC_9700 :  // 0x1A 0x05 "0198" for the "Waterfall Speed" in an IC-9700
   //                  iSubSubCmd=0x01; iSubSubSubCmd=0x98;
   //                  break;
   //             default: iCmd=-1; break; // sorry... don't know how to set the "Waterfall Speed" for any other radio yet
   //           } // end switch( pRC->iDefaultAddress )
   //          break; // end case RIGCTRL_PN_WATERFALL_SPEED
   //  RigCtrl_GetCrazyModelDependingSubSubcodeForCmd1A_05() must have taken care of this,
   //  since there are a few other supported radios besides IC-7300 and IC-9700 now !

           case RIGCTRL_PN_AUDIO_VOLUME_PERCENT :  // send a command to SET the radio's own loudspeaker volume..
              // Like MANY other CI-V sequences, trying to reduce an IC-7300's
              // AUDIO VOLUME IN THE BUILT-IN SPEAKER by editing the following
              // line (via 'Settings' .. 'List Rig Control params on the Debug tab')
              // did NOT work right out of the box:
              //   Original setting:     > "PN040: AudioVolume = 13 %"
              //   New (edited) setting: > "PN040: AudioVolume = 3 %"
              // When trying to send the MODIFIED 'AudioVolue',
              // the RCW Keyer's DEBUG TAB showed the following
              //     (after pressing ENTER on the modified line) :
              // > TX 009 FE FE 00 E0 14 01 03 00 FD ; write AudioVolume
              // > RX 006 FE FE E0 94 FA FD ; AudioVolume : NotOK
              // The only (single) occurrence of "Volume" in A7508-3EX-4
              // was on page 4 :
              // > NOTE: Operation of some control dials overrides CI-V commands.
              // >       If a control dial, such as the AF Volume dial that has
              // >       a mark on it, is rotated after sending a CI-V command,
              // >       the command will be overwritten by the operation.
              // A-ha. So what's the correct CI-V COMMAND to *set* the
              //       AF Volume "instead of the dial" ? A7508-3EX-4 page 4 said
              //       about command 0x14, sub command 0x01 (the one that was
              //       rejected as "NotOK" as shown above) :
              // > Send/read the AF level (0000=Minimum to 0255=Maximum) .
              //       Here for comparison the READ COMMAND, and an IC-7300's response:
              // > TX 007 FE FE 00 E0 14 01 FD       ; read AudioVolume
              // > RX 009 FE FE E0 94 14 01 00 34 FD ; AudioVolume:13 %
              //                      |  |  |___|
              //                    CMD SUB  DATA ("0000".."0255", NOT PERCENT).
              // Since there are DOZENS (if not HUNDREDS) of other parameters
              // in these Icom radio's using this "256 steps in FOUR DIGITS",
              // while the T_RigCtrlInstance member (here: pRC->iAudioVolume_Percent)
              // uses -as the name implies- PERCENT, range 0..100,
              // the info about how to SCALE a certain parameter (back and forth)
              // must be part of the parameter info (pPI), besides e.g.
              // CIV_DTYPE_BCD4, those parameters are now decorated with
              // CIV_DTYPE_FLAG_PERCENT255 (bitwise ORed to the basic type like ..BCD4).
              break;

           case RIGCTRL_PN_FILTER_BANDWIDTH: // "FilterBW" (audio filter bandwidth, quite FLEXIBLE for *DSP* rigs)
              // As the name (suffix) implies, T_RigCtrlInstance.iFilterBW_Hz
              // uses the one-and-only SI unit for any kind of frequency: HERTZ.
              // But Icom's cmd 0x1A subcmd 0x03 (*) uses an esoteric two-digit
              // CODE instead of a bandwidth in Hertz - see implementation of
              //      RigCtrl_CIV_AudioFilterBandwidthCodeToFrequency()
              // and  RigCtrl_CIV_FrequencyToAudioFilterBandwidthCode()
              i32Value/* Icom's "Filter Code" */
                = RigCtrl_CIV_FrequencyToAudioFilterBandwidthCode(
                      i32Value ); // [in] bandwidth in Hertz, no stupid "code"
              // (*) Command 0x1A, subcommand 0x03, called
              //     "Send/read the selected IF filter width", actually modifies
              //     the CURRENTLY SELECTED FILTER ("FIL1" to "FIL3"; IC-7300).
              //     The operator may accidentally turn the 'wide filter',
              //     traditionally called "FIL1" by Icom on their touchscreen,
              //     into the NARROWEST filter. Switching from one filter to
              //     another does not bring back the DEFAULT SETTINGS !
              //     An awfully butchered filter can be restored to the
              //     'factory default' setting via softkey 'DEF' in the filter
              //     menu on the local display. Not sure if the REMOTE CW KEYER
              //     will ever allow restoring a filter's default configuration.
              break; // end case RIGCTRL_PN_FILTER_BANDWIDTH

           case RIGCTRL_PN_CW_PITCH_HZ: // here: Convert from HERTZ into the CI-V internal unit:
              // > 0000 = 300 Hz,  0128 = 600 Hz, 0255 = 900 Hz.
              // A7380-7EX-2 page 3 tells a story about "5 Hz steps"
              // but THAT'S NOT THE SCALING STEP. Expect more fun like this !
              // From the inverse function in RigCtrl_ParseCIV() :
              // > dblConvFactor = 600.0/*Hz*/ / 256.0/*steps*/;
              // > dblConvOffset = 300.0/*Hz*/;
              // > dblValue [Hz] = (double)i32Value ["code" a la Icom] * dblConvFactor + dblConvOffset;
              i32Value/*"code" a la Icom*/ = (dblValue/*Hertz*/ - 300.0) * (256.0/600.0);
              RigCtrl_LimitInt32( &i32Value, 0, 255 );
              break; // end case RIGCTRL_PN_CW_PITCH_HZ

           default:  // nothing special, so use the 'parameter info' in RigCtrl_ParameterInfo[],
              // but command/sub-command/sub^3-command from the already set
              //  variables iCmd, iSubCmd (-1 when unused), iSubSubCmd (-1 when unused),
              //       and  i32Value (the TO-BE-WRITTEN value, since we're in RigCtrl_SendWriteCommandForUnifiedPN)
              // to cope with the crazy MODEL-DEPENDENT 4-digit sub-subcodes from cmd 0x1A sub 0x05 .
              break;
         } // end switch( iUnifiedPN )
        if( iCmd>=0 )
         {
           *pbBufPtr++ = 0xFE;  // fixed preamble code for CI-V protocol (term used by Icom .. NOT "Begin of Message" ! )
           *pbBufPtr++ = 0xFE;  // fixed preamble code for CI-V protocol
           *pbBufPtr++ = (BYTE)pPortInstance->iRadioDeviceAddr;  // "to" transceiver's address
           *pbBufPtr++ = (BYTE)pPortInstance->iRadioMasterAddr;  // "from" controller's address
           *pbBufPtr++ = (BYTE)iCmd; // e.g. 0x1C = "various transceiver/transmit related parameters"
           if( iSubCmd>=0 )
            { *pbBufPtr++ = (BYTE)iSubCmd;  // e.g. 0x00 (after 0x1C) to > "Send/read the transceiver's status"
            }
           if( iSubSubCmd>=0 )
            { *pbBufPtr++ = (BYTE)iSubSubCmd;  // sometimes 0x00 for "MAIN", 0x01 for a "SUB"-VFO if the radio was an IC-7610
               // Even for radios without "main" and "sub" (-VFO, -scope), this two-BCD-digit field exists.
               // In the IC-7300 CI-V reference, they usually say "fixed to zero" for this field.
            }
           if( iSubSubSubCmd>=0 )  // e.g. used for those dreadful RIG-SPECIFIC codes after e.g. cmd 0x1A SubCmd 0x05
            { *pbBufPtr++ = (BYTE)iSubSubSubCmd;
            }

           pbBufPtr += RigCtrl_MapCIVDataToMessage( pbBufPtr/*pbDest*/, iCIVDataType, i32Value );
           *pbBufPtr++ = 0xFD;  // final byte in the message: 0xFD = CI-V code for "End Of Message"
           fOk = RigCtrl_SendAndLogMessage( pRC, RIGCTRL_PORT_RADIO, RIGCTRL_ORIGIN_CONTROLLER,
                 b16TxBuffer/*pbMessage*/, pbBufPtr-b16TxBuffer/*iMsgLength*/,
                 iMsgType,      // message type for the traffic log
                 sz40Comment ); // comment for traffic log, e.g. "write SplitMode"
           if( fOk )
            { // If all works as planned, for most/all(?) "Write" / "Set"-commands,
              //    we expect the Icom transceiver to respond with cmd = 0xFB = "Ok".
              pPortInstance->dwExpectedResponse_CmdAndSubcode = RCTL_COMBINE_CMD_AND_SUB(iCmd,iSubCmd); // e.g. cmd=0x15 subcmd=0x12 -> dwExpectedResponse_CmdAndSubcode = 0x1512....
                    // Since the "discovery" of the CRAZY MODEL-DEPENDENT sub-subcodes for CI-V command 0x1A sub-command 0x05,
                    // pPortInstance->dwExpectedResponse_CmdAndSubcode is doomed to fail for those 'crazy' commands.
                    // Thus, since 2025-09, not only store the "expected command and sub-command",
                    // but also the superiour UNIFIED PARAMETER NUMBER, to help the interpretation
                    // of Icom's crazy anonymous WRITE-RESONSE - see "case 0xFB:" in RigCtrl_ParseCIV():
              pPortInstance->iExpectResponseForUnifiedPN = iUnifiedPN; // more "robust" for the 0x1A.0x05 commands, with their crazy MODEL-DEPENDENG 4-digit sub-subcodes !
              pPortInstance->iResponseCountdown_ms = pRC->iMaxResponseDelay_ms; // now waiting for the radio to respond after asking for yet another parameter..
              // 2025-02-27: Failed to switch the operating mode this way. CI-V traffic monitor showed:
              // >  18:37:03.5 TX 008 FE FE 00 E0 04 04 00 FD       ; write OpMode
              // >  18:37:03.6 RX 006 FE FE E0 94 FA FD ; NotOK for cmd 0xFFFF0000
              // Switching the operating mode 'locally' (directly on the server instance):
              // >  18:42:02.8 TX 008 FE FE 00 E0 06 01 01 FD ; set mode and filter: USB
              // >  18:42:02.9 RX 006 FE FE E0 94 FB FD           ; OK for cmd 0x6000000
              // Reason for this bug: Looking up the parameter info for RIGCTRL_PN_OP_MODE
              //    in
              //    only gives the CI-V command for READING the parameter (0x04).
              //    But command 0x04 is one of those special ancient beasts
              //    that only allows READING the parameter, in contrast to the
              //    more modern "Send/read" or "Select/read" commands, where
              //    the command byte(s) without data are the READ command,
              //    and the same command byte(s) followed by data are the WRITE command.
              //  In this case, the counterpart for CI-V "read operating mode" (0x04)
              //                                 is CI-V "set operating mode"  (0x06).
              //
             }
            else // failed to send the "write"-command...
             { // better luck next time !
             }
         }       // end if( iCmd>=0 )
        break;  // end case <ICOM CI-V protocol>
   } // end switch( pRC->iRadioCtrlProtocol )
  (void)dblValue; 
  return fOk;
} // end RigCtrl_SendWriteCommandForUnifiedPN()


//--------------------------------------------------------------------------
void RigCtrl_UpdateBasebandFrequencyConversionParams( T_RigCtrlInstance *pRC )  // -> DemodBasebandFreqOffset, ..Factor
  // Called, for example, from Spectrum Lab when drawing the filter passband
  // into the 'Remote Spectrum Display', received from IC-7300 and similar rigs.
  //    This function must know all the details about the relation between
  //    VFO frequency (aka 'dial frequency' seen on the radio's display)
  //    and the audio frequency.
  //  For USB, this is simple : Audio-frequency = Radio-Frequency - VFO-frequency;
  //                            thus [out] pRC->dblDemodBasebandFreqOffset = 0.0
  //                             and [out] pRC->dblDemodBasebandFreqFactor = +1.0
  //  For CW, it's not,
  //    because (example) : (a) Audio-frequency = Radio-Frequency - VFO-frequency + CW-pitch
  //                   or   (b) Audio-frequency = VFO-frequency - Radio-Frequency + CW-pitch
  //    depending on the radio receiving CW in "narrow USB" or "narrow LSB".
  //
  //
{
  double dblCwPitch_Hz;
  if( (pRC->iCWPitch_Hz != 0) && (pRC->iCWPitch_Hz != RIGCTRL_NOVALUE_INT ) )
   { dblCwPitch_Hz = pRC->iCWPitch_Hz;
   }
  else // CW pitch (audio center frequency + CW sidetone) hasn't been read yet
   { dblCwPitch_Hz = 650.0; // author's favourite (when adjustable..)
   }

  switch( pRC->iOpMode & RIGCTRL_OPMODE_MASK_TO_STRIP_FLAGS )
   {
     case RIGCTRL_OPMODE_CW :
        if( pRC->iOpMode & RIGCTRL_OPMODE_REVERSE )
         { // In most rigs, "CW-REVERSE" is actually USB (no frequency inversion)
           // With the VFO tuned tuned to 3560 kHz, a signal on 3560.1 Hz
           // will be 100 Hz *above* the configurable 'CW pitch' frequency, thus:
           pRC->dblDemodBasebandFreqOffset = dblCwPitch_Hz;
           pRC->dblDemodBasebandFreqFactor = 1.0;
         }
        else // In most rigs, "CW" is actually LSB (with frequency inversion)
         { pRC->dblDemodBasebandFreqOffset = dblCwPitch_Hz;
           pRC->dblDemodBasebandFreqFactor = -1.0;
         }
        break; // end case RIGCTRL_OPMODE_CW
     case RIGCTRL_OPMODE_LSB: // no shifting for the "CW pitch" but sideband inversion:
        pRC->dblDemodBasebandFreqOffset =  0.0;
        pRC->dblDemodBasebandFreqFactor = -1.0;
        break;

     default:  // for any other mode, assume "0 Hz baseband" == "VFO frequency",
               // without sideband inversion :
        pRC->dblDemodBasebandFreqOffset = 0.0;
        pRC->dblDemodBasebandFreqFactor = 1.0;
        break;
   } // end switch( pRC->iOpMode & RIGCTRL_OPMODE_MASK_TO_STRIP_FLAGS )

} // end RigCtrl_UpdateBasebandFrequencyConversionParams()

//--------------------------------------------------------------------------
double RigCtrl_AudioToRadioFrequency( T_RigCtrlInstance *pRC, double dblAudioFreq_Hz )
{ RigCtrl_UpdateBasebandFrequencyConversionParams( pRC );  // <- aware of "USB","LSB","CW","CW reverse", and the "CW pitch"
  return pRC->dblDemodBasebandFreqOffset // e.g. -650 Hz in "USB CW" with a 650 Hz 'CW pitch' !
       + pRC->dblDemodBasebandFreqFactor * dblAudioFreq_Hz // e.g. +1.0 for "USB", -1.0 for "LSB".
       + pRC->dblVfoFrequency; // plus e.g. 3.560 MHz "VFO" frequency aka "dial frequency"
} // end RigCtrl_AudioToRadioFrequency()

//--------------------------------------------------------------------------
double RigCtrl_RadioToAudioFrequency( T_RigCtrlInstance *pRC, double dblRadioFreq_Hz )
{ double dblAF; // return value: "Audio Frequency" in Hertz (should be positive,
                // unless the RX delivers an I/Q output. In that case,
                // the "Audio Frequency" is in fact a "baseband frequency".)
  RigCtrl_UpdateBasebandFrequencyConversionParams( pRC );
  // Inverse to the calculation in RigCtrl_AudioToRadioFrequency() :
  // RF = BasebandFreqOffset + AF * BasebandFreqFactor + VfoFrequency
  // AF * BasebandFreqFactor = RF - BasebandFreqOffset - VfoFrequency
  // AF = (RF - BasebandFreqOffset - VfoFrequency) / BasebandFreqFactor;
  dblAF = dblRadioFreq_Hz - pRC->dblDemodBasebandFreqOffset - pRC->dblVfoFrequency;
  if( pRC->dblDemodBasebandFreqFactor != 0.0 )  // avoid div-by-zero ..
   { dblAF /= pRC->dblDemodBasebandFreqFactor;
   }
  return dblAF;
} // end RigCtrl_RadioToAudioFrequency()

//---------------------------------------------------------------------------
static void RigCtrl_OpModeToIcomModeAndFilterCode(
             int iOpMode,
             int *piMode,
             int *piFilter,
             char **ppszComment, char *pszEndstop ) // [out] comment for the "log"
{
  int iMode   = -1;    // no "Icom-equvalent" for RigControl's "OpMode" yet...
  int iFilter = 0x01;  // Icom's filter codes : 1="wide", 2="medium", 3="narrow"
  // From DF4OR [DF4OR_CIV] :
  // > Due to historical reasons the filter settings are somewhat confusing.
  // > The original CI-V protocol initially had only two settings
  // > for filter width: normal and narrow. When newer rigs came out,
  // > supporting three settings (wide, normal, narrow) the codes
  // > for the filter width had to be extended.
  // > This leads to some confusion about which value sets which filter,
  // > further aggravated by erroneous documentation (e.g. IC-R75).
  // > And when looking at the DSP-filter rigs like IC-756Pro ff.,
  // > where the user can define and arrange each of the three filters
  // > individually, the settings of 'wide' or 'narrow' become meaningless.
  // > Here a filter value of 0x01 corresponds to filter 1,
  // > value 0x02 to filter 2 and 0x03 to filter 3,
  // > regardless of whether filter 1 is acutally wider than filter 2 or 3.
  // > Formerly only 0x01 and 0x02 were used for wide and narrow,
  // > medium/normal was introduced later and the data rearranged.
  // > So it can happen that 'narrow' on older rigs is programmed using 0x02,
  // > on other, newer rigs as 0x03.
  // DL4YHF: In the DEFAULT settings of an IC-7300, "Fil3" is the NARROWEST.
  switch( iOpMode & RIGCTRL_OPMODE_MASK_TO_STRIP_FLAGS )  // translate RigControl's "operating mode" into Icom's
   {
     case RIGCTRL_OPMODE_CW :
         iMode = 0x03;  // CI-V : 3 = "CW". Hopefully on ALL radios.
         SL_AppendString( ppszComment, pszEndstop, "CW" );
         if( iOpMode & RIGCTRL_OPMODE_REVERSE )
          { iMode = 0x07; // CI-V : 0x07 = "CW-R". At least on IC-7300.
            SL_AppendChar( ppszComment, pszEndstop, 'R' );
          }
         if( iOpMode & RIGCTRL_OPMODE_NARROW )
          { iFilter = 0x02;  // some radios: 1="wide", 2="medium", 3="narrow"
            SL_AppendChar( ppszComment, pszEndstop, 'N' );
          }
         if( iOpMode & RIGCTRL_OPMODE_VERY_NARROW )
          { iFilter = 0x03;  // some radios: 1="wide", 2="medium", 3="narrow"
            SL_AppendChar( ppszComment, pszEndstop, 'N' );
          }
         break;
     case RIGCTRL_OPMODE_LSB  :
         iMode = 0x00;  // CI-V : 0 = "LSB"
         SL_AppendString( ppszComment, pszEndstop, "LSB" );
         break;
     case RIGCTRL_OPMODE_USB  :
         iMode = 0x01;  // CI-V : 1 = "USB"
         SL_AppendString( ppszComment, pszEndstop, "USB" );
         break;
     case RIGCTRL_OPMODE_AM   :
         iMode = 0x02;  // CI-V : 2 = "AM"
         SL_AppendString( ppszComment, pszEndstop, "AM" );
         break;
     case RIGCTRL_OPMODE_FM   :   // Fasten seat belt and BE PREPARED FOR TROUBLE !
         iMode = 0x05;  // CI-V : 5 = "FM" (at least for IC-9700)
         SL_AppendString( ppszComment, pszEndstop, "FM" );
         break; // end case RIGCTRL_OPMODE_FM
     case RIGCTRL_OPMODE_FM_WIDE :
         iMode = 0x06;     // CI-V : 6 = "FM wide" (at least for IC-706)
         SL_AppendString( ppszComment, pszEndstop, "FM-Wide" );
         break;
     case RIGCTRL_OPMODE_RTTY :
         iMode = 0x04;    // CI-V : 4 = "RTTY". But on IC-910, 9 is FM. Holy shit.
         // (these annoying subtleties are one reason to use HEX VALUES here,
         //  not macro constants which everyone would have to look up permanently..)
         SL_AppendString( ppszComment, pszEndstop, "RTTY" );
         if( iOpMode & RIGCTRL_OPMODE_REVERSE )
          { iMode = 0x08; // CI-V : 0x08 = "RTTY-R". At least on IC-7300..
            //  not listed for older radios, in Icom's funny-titled
            //  "CI-V Reference Manual -.-.-ver 3.2.-.-. -2002-"
          }
         break;
     case RIGCTRL_OPMODE_PSK  :
         iMode = 0x12;    // CI-V : 0x12 = "PSK". At least on IC-7300..
         SL_AppendString( ppszComment, pszEndstop, "PSK" );
         if( iOpMode & RIGCTRL_OPMODE_REVERSE )
          { iMode = 0x13; // CI-V : 0x13 = "PSK-R". At least on IC-7300
          }
         break;
     default:
         iMode = 0x03;  // CI-V : "CW"
         SL_AppendString( ppszComment, pszEndstop, "CW" );
         break;
    } // end switch( pRC->iOpMode )
  if( piMode != NULL )
   { *piMode = iMode;
   }
  if( piFilter != NULL )
   { *piFilter = iFilter;
   }
} // end RigCtrl_OpModeToIcomModeAndFilterCode()


//--------------------------------------------------------------------------
static BOOL RigCtrl_SetOperatingMode_Internal( T_RigCtrlInstance *pRC, int iOpMode )
  // Sends the new operating mode (e.g. RIGCTRL_OPMODE_USB) to the
  //       locally connected radio, usually via Ci-V, immediately.
  // As most functions, returns TRUE when successful, otherwise FALSE.
  // Returns BEFORE the remote rig has actually 'digested' the new setting !
  // (thus returning TRUE only means "successfully SENT" but not necessarily
  //  "new setting was accepted"). Also applies to many other 'set' functions.
{
  T_RigCtrl_PortInstance *pPortInstance = &pRC->PortInstance[RIGCTRL_PORT_RADIO];
  int  iMode, iFilter;
  BOOL fResult = FALSE;
  char sz80Comment[84];
  char *pszDest=sz80Comment, *pszEndstop=sz80Comment+80;

  SL_AppendString( &pszDest, pszEndstop, "set mode and filter: " );

  switch( pPortInstance->iRadioCtrlProtocol )
   {  // which language does the radio *ON THIS PORT* speak ?
     case RIGCTRL_PROTOCOL_NONE :
     default:
        break;  // unknown protocol, can only send "direct" strings

     case RIGCTRL_PROTOCOL_ICOM_CI_V : // Icom's "CI-V" protocol ...
        // Use CI-V command 0x06 because we want a response ("OK" or "NoGo"), and ..
        // > This command is understood by practically all Icom radios.
        RigCtrl_OpModeToIcomModeAndFilterCode( iOpMode, &iMode, &iFilter,
                                               &pszDest/*comment*/, pszEndstop );
        if( iMode >= 0 )  // successfully translated at least the MODE..
         { fResult = RigCtrl_SendShortCIVMessage( pRC, RIGCTRL_PORT_RADIO, RIGCTRL_ORIGIN_CONTROLLER,
                    RIGCTRL_MSGTYPE_OP_MODE,  // message type for traffic log
                       sz80Comment, // comment for traffic log
                       0x06,  // iCmd: CI-V command "Set mode and filter"
                         -1,  // iSubCmd and iDataByte1 : Omit !
                       iMode, // iDataByte1 : here "mode".
                    iFilter); // iDataByte2 : here "filter" (optionally)
           pPortInstance->dwExpectedResponse_CmdAndSubcode = RCTL_COMBINE_CMD_ONLY(0x06); // the radio's CI-V response "Set operating mode" doesn't contain command or subcode, so REMEMBER what has been "set" this way
           pPortInstance->iResponseCountdown_ms = pRC->iMaxResponseDelay_ms; // now waiting for the "OK"/"NOT OK" response for "Set Operating Mode"
         }
        break;  // end case <ICOM CI-V protocol>

   } // end switch( pRC->iRadioCtrlProtocol )

  return fResult;

} // end RigCtrl_SetOperatingMode_Internal()

//--------------------------------------------------------------------------
BOOL RigCtrl_SetOperatingMode( T_RigCtrlInstance *pRC, int iOpMode ) /* API */
  // Sends the new operating mode (e.g. RIGCTRL_OPMODE_USB) to the radio.
  //  [in] iOpMode : RIGCTRL_OPMODE_CW, RIGCTRL_OPMODE_LSB, RIGCTRL_OPMODE_USB,
  //                 RIGCTRL_OPMODE_AM, RIGCTRL_OPMODE_FM,  RIGCTRL_OPMODE_FM_WIDE
  //                   ("basic modes" below RIGCTRL_OPMODE_MASK_TO_STRIP_FLAGS),
  //                    OPTIONALLY bitwise combined with flags like
  //                 RIGCTRL_OPMODE_REVERSE (for "CW Reverse"),
  //                 RIGCTRL_OPMODE_NARROW, RIGCTRL_OPMODE_VERY_NARROW
  //                 (the latter for compatibility with older rigs,
  //                  which don't support reading the "filter bandwidth").
  //
  // As most functions, returns TRUE when successful, otherwise FALSE.
  // Returns BEFORE the remote rig has actually 'digested' the new setting !
  // (thus returning TRUE only means "successfully SENT" but not necessarily
  //  "new setting was accepted"). Also applies to many other 'set' functions.
  //
  // Like many other 'set' API functions in RigControl.c ,
  //      this one is aware of operating as a CLIENT, and in that case,
  //      passes the new value to the remote server via TCP/IP (CwNet.c).
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // Use RigCtrl_SetOperatingMode() as an example for the flow of information
  //  between the GUI, RigControl, and client/server via TCP/IP using CwNet.c :
  //   1. The operator, client side, modifies something in the GUI (e.g. switches from CW to CWN).
  //      The GUI calls RigCtrl_SetOperatingMode(), regardless of client/server/standalong operation.
  //   2. On the client side, RigCtrl_SetXYZ() compares the new parameter value
  //      to the one in the T_RigCtrlInstance. If the new value is different,
  //      RigCtrl_AddParamToTxQueueForCwNet() adds the PARAMETER NUMBER
  //      (e.g. RIGCTRL_PN_OP_MODE) to a queue of parameter numbers (w/o VALUES).
  //   3. Still on the CLIENT side, but now in the worker thread in CwNet.c :
  //      If the space in the network transmit buffer allows, CwNet_OnPoll()
  //      reads the parameter from the queue, again thread-safe, from entry
  //      pCwNet->RigControl.iTxParameterQueue[pCwNet->RigControl.iTxParameterQueueTailIndex++] .
  //      Depending on the parameter's data type (and maybe more),
  //      CwNet.c : CwNet_OnPoll() sends a block with type CWNET_CMD_PARAM_INTEGER.
  //   4. On the SERVER side, in CwNet.c : ServerThread() -> CwNet_ExecuteCmd()
  //      recognizes the CWNET_CMD_PARAM_INTEGER in the multiplexed TCP/IP stream (which also
  //      contains audio, spectrum data, and who-knows-what), and passes the
  //      "integer parameter report" (with parameter-number and -value) on to
  //      RigCtrl_OnParamReport_Integer() .
  //   5. On the SERVER side, in RigControl.c : RigCtrl_OnParamReport_Integer()
  //      updates the new value in the T_RigCtrlInstance, where RigCtrl_Handler()
  //           -running in a different thread, usually the main thread-
  //      can a few milliseconds later (up to 50 ms, the calling interval of
  //      RigCtrl_Handler() ).

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
{
  T_RigCtrl_PortInstance *pPortInstance = &pRC->PortInstance[RIGCTRL_PORT_RADIO];

  // This function may be called by the application at any time.
  // The actual transmission may have to be postponed when still busy
  // from an 'uninterruptable' sequence of commands in RigCtrl_Handler().
  // In any case, immediately copy the new mode ...
  if( iOpMode != pRC->iOpMode )
   { pRC->iOpMode = iOpMode;
     // Queue this up for a later transmission via network / CwNet.c :
     RigCtrl_AddParamToTxQueueForCwNet(pRC, RIGCTRL_PN_OP_MODE ); // here: called from RigCtrl_SetOperatingMode()
   }
  if( pPortInstance->iResponseCountdown_ms > 0 ) // "do not send anything at the moment" because still waiting for the response for a PREVIOUSLY SENT "Set"-command !
   {  // ex: pRC->iPostponedSetMessages |= (1<<RIGCTRL_MSGTYPE_OP_MODE);
      RigCtrl_SetPostponedSetMessageFlag( pRC, RIGCTRL_MSGTYPE_OP_MODE );
      pRC->iPostponedOpMode = iOpMode;
      return FALSE;  // cannot send AT THE MOMENT (still busy from another transaction)
   }
  else // no need to postpone this "Set"-command :
   {  // ex: pRC->iPostponedSetMessages &= ~(1<<RIGCTRL_MSGTYPE_OP_MODE);
      RigCtrl_ClearPostponedSetMessageFlag( pRC, RIGCTRL_MSGTYPE_OP_MODE );
      return RigCtrl_SetOperatingMode_Internal( pRC, iOpMode );
   }

} // end RigCtrl_SetOperatingMode()

//--------------------------------------------------------------------------
const char* RigCtrl_OperatingModeToString(int iOpMode)
  // First used in Spectrum Lab for interpreter function "rct.mode_string" .
  // Later also used in the Remote CW Keyer / RigCtrl_FreqMemEntryToString().
  // Even later drilled up to support all modes that an IC-9700 supported,
  //      including "DV" (Digital Voice) because that's more neutral than
  //      e.g. "DStar", "DMR", "C4FM", "Tetra", and whatever comes next.
  // If iOpMode has MULTIPLE bitflag for a basic modes, the result may be
  //    a comma-delimited(!) list of symbols like "CW,LSB,USB,AM,FM,RTTY" .
  //    This feature is used in "band definitions", where on a single amateur
  //    radio band, MULTIPLE modes may be used.
  // Inverse functions (string builder and parser) : RigCtrl_OperatingModeToString(),
  //                                                 RigCtrl_StringToOperatingMode().
{ static char sz40Result[44];
  char *pszDest = sz40Result;
  const char *pszEndstop = sz40Result + 40;
  BOOL fAppendComma = FALSE;

  sz40Result[0] = '\0';

  if( (iOpMode & RIGCTRL_OPMODE_MASK_TO_STRIP_FLAGS) == RIGCTRL_OPMODE_ALL )
   { // It's a combination of "all" modes (including those we don't even know yet):
     SL_AppendString( &pszDest, pszEndstop, "ALL" ); // guess what happens when BOTH flags, "NARROW" + "VERY_NARROW" are set
     iOpMode &= ~RIGCTRL_OPMODE_ALL;
     // What may remain in iOpMode are the 'bandwidth bits'.
     // But for "all" modes (including ATV, Amateur Television and "DV"), ignore them.
   }
  else // not RIGCTRL_OPMODE_ALL but a bitwise combination of SOME modes.. which ?
   {
     // If it's a single of one of the common modes, don't use the evil static buffer:
     if( iOpMode & RIGCTRL_OPMODE_CW )
      { SL_AppendString( &pszDest, pszEndstop, "CW" );
        iOpMode &= ~RIGCTRL_OPMODE_CW; // clear this bit to check if others remain (further below)
        if( iOpMode & RIGCTRL_OPMODE_NARROW )
         { SL_AppendString( &pszDest, pszEndstop, "N" );
           iOpMode &= ~RIGCTRL_OPMODE_NARROW;
         }
        if( iOpMode & RIGCTRL_OPMODE_VERY_NARROW )
         { SL_AppendString( &pszDest, pszEndstop, "NN" ); // guess what happens when BOTH flags, "NARROW" + "VERY_NARROW" are set
           iOpMode &= ~RIGCTRL_OPMODE_VERY_NARROW;
         }
        if( iOpMode & RIGCTRL_OPMODE_REVERSE )
         { SL_AppendString( &pszDest, pszEndstop, "R" );
           iOpMode &= ~RIGCTRL_OPMODE_REVERSE;
         }
        fAppendComma = TRUE;
      }
     if( iOpMode & RIGCTRL_OPMODE_LSB )
      { // If this isn't the first "opmode", got to assemble a comma-separated LIST:
        if( fAppendComma )
         { SL_AppendChar( &pszDest, pszEndstop, ',' );
         }
        SL_AppendString( &pszDest, pszEndstop, "LSB" );
        iOpMode &= ~RIGCTRL_OPMODE_LSB;
        fAppendComma = TRUE;
      }
     if( iOpMode & RIGCTRL_OPMODE_USB ) // etc, etc..
      { if( fAppendComma )
         { SL_AppendChar( &pszDest, pszEndstop, ',' );
         }
        SL_AppendString( &pszDest, pszEndstop, "USB" );
        iOpMode &= ~RIGCTRL_OPMODE_USB;
        fAppendComma = TRUE;
      }
     if( iOpMode & RIGCTRL_OPMODE_AM )
      { if( fAppendComma )
         { SL_AppendChar( &pszDest, pszEndstop, ',' );
         }
        SL_AppendString( &pszDest, pszEndstop, "AM" );
        iOpMode &= ~RIGCTRL_OPMODE_AM;
        fAppendComma = TRUE;
      }
     if( iOpMode & RIGCTRL_OPMODE_FM )
      { if( fAppendComma )
         { SL_AppendChar( &pszDest, pszEndstop, ',' );
         }
        SL_AppendString( &pszDest, pszEndstop, "FM" );    // "FM" on amateur radio frequencies means "NBFM". IC-9700: "FIL1" : "15 k" (bandwidth)
        iOpMode &= ~RIGCTRL_OPMODE_FM; // clear this bit to check if others remain:
        if( iOpMode & RIGCTRL_OPMODE_NARROW )
         { SL_AppendString( &pszDest, pszEndstop, "N" );  // "FMN" : The preferred mode; IC-9700: "FIL2" : "10 k" (bandwidth)
           iOpMode &= ~RIGCTRL_OPMODE_NARROW;
         }
        if( iOpMode & RIGCTRL_OPMODE_VERY_NARROW )
         { SL_AppendString( &pszDest, pszEndstop, "NN" ); // "FMNN" : Even narrower FM: IC-9700: "FIL3" : "7 k" (bandwidth)
           iOpMode &= ~RIGCTRL_OPMODE_VERY_NARROW;
         }
        fAppendComma = TRUE;
      }
     if( iOpMode & RIGCTRL_OPMODE_FM_WIDE )
      { if( fAppendComma )
         { SL_AppendChar( &pszDest, pszEndstop, ',' );
         }
        SL_AppendString( &pszDest, pszEndstop, "FMW" );
        iOpMode &= ~RIGCTRL_OPMODE_FM_WIDE;
        fAppendComma = TRUE;
      }
     if( iOpMode & RIGCTRL_OPMODE_RTTY )
      { if( fAppendComma )
         { SL_AppendChar( &pszDest, pszEndstop, ',' );
         }
        SL_AppendString( &pszDest, pszEndstop, "RTTY" );
        iOpMode &= ~RIGCTRL_OPMODE_RTTY;
        fAppendComma = TRUE;
      }
     if( iOpMode & RIGCTRL_OPMODE_PSK )
      { if( fAppendComma )
         { SL_AppendChar( &pszDest, pszEndstop, ',' );
         }
        SL_AppendString( &pszDest, pszEndstop, "PSK" );
        iOpMode &= ~RIGCTRL_OPMODE_PSK;
        fAppendComma = TRUE;
      }
     if( iOpMode & RIGCTRL_OPMODE_DIGITAL_VOICE )
      { if( fAppendComma )
         { SL_AppendChar( &pszDest, pszEndstop, ',' );
         }
        SL_AppendString( &pszDest, pszEndstop, "DV" ); // IC-9700: "DV" = "D-Star"
        iOpMode &= ~RIGCTRL_OPMODE_DIGITAL_VOICE;
        fAppendComma = TRUE;
      }
     if( iOpMode & RIGCTRL_OPMODE_DIGITAL_DATA )
      { if( fAppendComma )
         { SL_AppendChar( &pszDest, pszEndstop, ',' );
         }
        SL_AppendString( &pszDest, pszEndstop, "DD" ); // IC-9700: "DD" = "Digital Data"
        iOpMode &= ~RIGCTRL_OPMODE_DIGITAL_DATA;
        fAppendComma = TRUE;
#      ifdef __BORLANDC__  // "... is assigned a value that is never used".. oh shut up, we know what we're doing !
        (void)fAppendComma;
#      endif
      }

   } // end else < not RIGCTRL_OPMODE_ALL > ?
  if( pszDest==sz40Result ) // Nothing appended above ? As a fallback, emit the DECIMAL code
   { SL_AppendPrintf( &pszDest, pszEndstop, "%d", (int)iOpMode );
   }

  // For a hypothetic rig supporting 'ALMOST everything', expect sz40Result to be..
  //  "CWNNR,LSB,USB,AM,FM,FMW,RTTY,PSK" .. circa 32 chars !
  //    '--> A rig that supports "CW-Very-Narrow-Reverse"
  //         also supports CW, CWN, CWNN, CWR, and CWNR.
  //         Don't confuse the user with an endless list of modes,
  //         including having to specify that in file "RCWKeyer_Bands.txt" .
  // See the inverse function, RigCtrl_StringToOperatingMode(),
  // which makes the same assumption about "CW" modes and bandwidths.
  return sz40Result; // <- may still be EMPTY if none of the "op-mode" bits is set

} // end RigCtrl_OperatingModeToString()

//--------------------------------------------------------------------------
int RigCtrl_StringToOperatingMode(const char **ppszSource)
  //  '--> Returns a BIT COMBINATION of RIGCTRL_OPMODE_CW, RIGCTRL_OPMODE_NARROW, etc.
  // Added 2024-11-21 to exchange e.g. "Band Stacking Registers" (etc) in a
  // non-Icom-specific format between client and server (via TCP/IP).
  // Inverse functions (string builder and parser) : RigCtrl_OperatingModeToString(), RigCtrl_StringToOperatingMode() .
  // Since 2024-12-11, the string may also contain a LIST of operating modes,
  // required by RigCtrl_StringToUserDefinedBand(), e.g. "CW,CWN,CWNN,USB,LSB,AM,FM" .
{ int iOpMode = 0;  // <- bitwise combineable, thus 0 = RIGCTRL_OPMODE_UNKNOWN .
  const char *pszOldSource;

  while(1) // run in a loop if this turns out to be a comma-separated list of MULTIPLE modes..
   { pszOldSource = *ppszSource; // kludge to avoid an endless loop when NOTHING was recognized

     // First check if the pszSource begins with a known "basic mode" like "CW".
     // After that, check for suffixes like "R" (Reverse), "N" (Narrow), "NN" (Very Narrow).
     if( SL_SkipString_AnyCase( ppszSource, "CW" ) ) // SL_SkipString..(), not SL_SkipToken(), to recognize the "CW" in "CWN" or "CWNN"
      { iOpMode |= RIGCTRL_OPMODE_CW;
        // Now for the "bandwidth suffix" (which, when used in a list, only applies to CW but not USB or LSB):
        if( SL_SkipString_AnyCase( ppszSource, "NN" ) )
         { iOpMode |= RIGCTRL_OPMODE_VERY_NARROW;
         }
        else if( SL_SkipString_AnyCase( ppszSource, "N" ) )
         { iOpMode |= RIGCTRL_OPMODE_NARROW;
         }
        if( SL_SkipString_AnyCase( ppszSource, "R" ) )
         { iOpMode |= RIGCTRL_OPMODE_REVERSE;
         }
      }
     else if( SL_SkipString_AnyCase( ppszSource, "LSB" ) )
      { iOpMode |= RIGCTRL_OPMODE_LSB;
        if( SL_SkipString_AnyCase( ppszSource, "N" ) )
         { iOpMode |= RIGCTRL_OPMODE_NARROW;
         }
      }
     else if( SL_SkipString_AnyCase( ppszSource, "USB" ) )
      { iOpMode |= RIGCTRL_OPMODE_USB;
        if( SL_SkipString_AnyCase( ppszSource, "N" ) )
         { iOpMode |= RIGCTRL_OPMODE_NARROW;
         }
      }
     else if( SL_SkipString_AnyCase( ppszSource, "AM" ) )
      { iOpMode |= RIGCTRL_OPMODE_AM;
      }
     else if( SL_SkipString_AnyCase( ppszSource, "FMW" ) )
      { iOpMode |= RIGCTRL_OPMODE_FM_WIDE;
      }
     else if( SL_SkipString_AnyCase( ppszSource, "FM" ) )
      { iOpMode |= RIGCTRL_OPMODE_FM;
      }
     else if( SL_SkipString_AnyCase( ppszSource, "RTTY" ) )
      { iOpMode |= RIGCTRL_OPMODE_RTTY;
      }
     else if( SL_SkipString_AnyCase( ppszSource, "PSK" ) )
      { iOpMode |= RIGCTRL_OPMODE_FM;
      }
     else if( SL_SkipString_AnyCase( ppszSource, "ALL" ) )
      { iOpMode |= RIGCTRL_OPMODE_ALL;
      }
     else if( SL_SkipString_AnyCase( ppszSource, "DV" ) )
      { iOpMode |= RIGCTRL_OPMODE_DIGITAL_VOICE;
      }
     else if( SL_SkipString_AnyCase( ppszSource, "DD" ) )
      { iOpMode |= RIGCTRL_OPMODE_DIGITAL_DATA;
      }
     else if( SL_SkipString_AnyCase( ppszSource, "???" ) ) // <- that's a TOKEN, too !
      { iOpMode |= RIGCTRL_OPMODE_UNKNOWN;
      }
     // Just in case the sender still uses a NUMERIC value:
     if( (*ppszSource==pszOldSource )&& (SL_IsDigit( **ppszSource ) ) )
      { iOpMode = SL_ParseInteger(ppszSource);
      }
     if( pszOldSource == *ppszSource ) // nothing parsed above ?
      { break; // break from the loop !
      }
     if( ! SL_SkipChar( ppszSource, ',' ) )
      { break; // end of the comma-separated "Op-Mode"-list !
      }
   } // end while
  return iOpMode;
} // end RigCtrl_StringToOperatingMode()


//--------------------------------------------------------------------------
BOOL RigCtrl_SetScopeRefLevel_dB( T_RigCtrlInstance *pRC, double dblRefLevel_dB )
  // Sends a new 'spectrum reference level' to a modern Icom radio.
  // This is the parameter that you need to modify whenever switching
  // to a new 'span' (displayed bandwidth) in IC-7300, IC-9700, IC-705, etc,
  // which can be VERY annoying because the transceiver doesn't store
  // the reference levels for each of the supported 'spans'
  // (2.5 kHz, 5 kHz, 10 kHz, 25 kHz, 50 kHz, 100 kHz, 250 kHz, 500 kHz) !
  // In Icom's "IC-9700 CI-V REFERENCE GUIDE", published in June 2019,
  //      (locally saved as IC-9700_ENG_CI-V_1.pdf),
  // this is one of an awful lot of "sub commands" (sub parameters)
  // for command 27, "Send/read scope related parameters" [page 24],
  //  here: sub-command 19 = "Scope Reference level settings",
  //                         ranging from -20.0 to +20.0 dB in 0.5 dB steps.
{
  BOOL fResult;
  BYTE bBCD[2];        // 4-digit BCD in TWO BYTES
  BYTE bSign = 0x00;

  // Icom's "Spectrum REF level" ranges from -20 to +20 dB, so limit to that range:
  RigCtrl_LimitDouble( &dblRefLevel_dB, -20.0, 20.0 );
  if( pRC->dblScopeRefLevel_dB != dblRefLevel_dB ) // when running as NETWORK CLIENT..
   { pRC->dblScopeRefLevel_dB = dblRefLevel_dB;
     // Queue this up for a later transmission from client to the remote NETWORK server (in CwNet.c)
     RigCtrl_AddParamToTxQueueForCwNet(pRC, RIGCTRL_PN_SCOPE_REF_LEVEL );
   }

  if( dblRefLevel_dB < 0 ) // in CI-V cmd 0x27.0x19, the SIGN *follows* after the BCD string :
   { bSign = 0x01;
     dblRefLevel_dB = -dblRefLevel_dB;
   }
  RigCtrl_IntToBCD_CIV_MSByteFirst( (long)(dblRefLevel_dB*100.0/*!*/), bBCD, 4/*nDigits*/ );
  // ,--------------'
  // '--> MOST SIGNIFICANT TWO DIGITS (in one byte) FIRST ! Example:
  //  dblRefLevel_dB = 12.50 -> bBCD[0] = 0x12,  bBCD[1] = 0x50 (!) .

  fResult = RigCtrl_SendMidSizeCIVMessage( pRC, RIGCTRL_PORT_RADIO, RIGCTRL_ORIGIN_CONTROLLER,
             RIGCTRL_MSGTYPE_SPECTRUM_CONFIG, // message type for traffic log
                       "set scope REF level", // comment for traffic log
                       0x27,  // iCmd: CI-V command "Scope settings"
                       0x19,  // iSubCmd: 0x19 = "Scope Reference level settings"
                       0x00,  // iDataByte1 : 0x00 = "main scope", 0x01 = "sub scope"
                    bBCD[0],  // iDataByte2 : here the 10 dB digit and 1 dB digit (BCD)
                    bBCD[1],  // iDataByte3 : here the 0.1 dB digit and 0.01 dB digit (BCD)
                     bSign);  // iDataByte4 : 0x00 for "plus", 0x01 for "minus" (our data-type-flag CIV_DTYPE_SUFFIX_BCD2_SIGN)
  // pPortInstance->dwExpectedResponse_CmdAndSubcode = ?; // expected CI-V response ?
  // Seen in the RCW Keyer's built-in CI-V message display:
  // > 19 21:12:39.4 TX 00B FE FE 00 00 27 19 00 4F 32 32 FD ; set scope REF level
  // > 20 21:12:39.5 RX 006 FE FE 00 A2 FA FD                ; NotOK for ...

  return fResult;


} // end RigCtrl_SetScopeRefLevel_dB()


//--------------------------------------------------------------------------
BOOL RigCtrl_SetScopeCenterFrequency( T_RigCtrlInstance *pRC, double dblFreq_Hz )
  // Sets a new CENTER FREQUENCY (Radio Frequency in Hz), and sends it to the
  // radio (or remote radio control server). Only possible with a modern Icom
  // radio, and only if their "scope" (spectrum / spectrogram display)
  // is in a mode where the center frequeny can be set independently from the
  // VFO frequency, e.g. "Scroll C" ("Scroll Center") mode.
{

  if( pRC->dblSpectrumCenterFreq_Hz != dblFreq_Hz )
   { // Arrived here ? Really need to MODIFY something inside the radio !
     // Implementation details:
     // Many CI-V Reference Manuals mumble somthing about cmd 0x27 0x1C,
     //   > Send/read scope center frequency setting in the Center mode .
     // Sounds as if that gives access to the SCOPE CENTER FREQUENCY,
     //            but nope, another misunderstanding of the Icom mumbo-jumbo.
     // In fact, command 0x27, sub-command 0x1C  is just a TWO-DIGIT PARAMETER
     //    with possible values 0x00="Filter Center", 0x01="Carrier Point Center",
     //                         0x02="Carrier Point Center (Abs. Freq.)". Buaah.
     // Seems there is no way to "scoll" the center frequency in "Scroll Center"
     // mode via CI-V (only the radio itself can do that, internally).
     // The only solution seems to be a "fake Scroll-Center"-mode, abusing one of
     // the few "Fixed Edge Frequency" pairs. Buaah. Maybe something for the future.

     // Queue this up for a later transmission from client to the remote NETWORK server (in CwNet.c)
     RigCtrl_AddParamToTxQueueForCwNet(pRC, RIGCTRL_PN_SCOPE_CENTER_FREQUENCY );

     return FALSE;
   }
  return TRUE;

} // end RigCtrl_SetScopeCenterFrequency()


//--------------------------------------------------------------------------
BOOL RigCtrl_SetScopeSpan_Hz( T_RigCtrlInstance *pRC, double dblScopeSpan_Hz )
  // Sends a new 'spectrum scope span' (diplayed bandwidth in Hertz) to a modern Icom radio.
  // Radio generations like IC-7300 and IC-9700 only supported the following "SPAN" settings:
  //      "+/-2.5 kHz"      "+/-5.0 kHz"    "+/-10.0 kHz"
  //      "+/-25.0 kHz"     "+/-50.0 kHz"   "+/-100.0 kHz"
  //      "+/-250.0 kHz"    "+/-500.0 kHz" (<- that's the highest setting, 500 kHz "Span" = 1 MHz displayed bandwidth.
{
  T_RigCtrl_PortInstance *pPortInstance = &pRC->PortInstance[RIGCTRL_PORT_RADIO];

  // This API function may be called by the application at any time, but it may
  // have to be postponed (internally) when still busy from an 'uninterruptable'
  //      sequence of commands in RigCtrl_Handler().
  // In ANY case, immediately copy the new "wanted" setting :
  if( pRC->dblScopeSpan_Hz != dblScopeSpan_Hz )
   {  pRC->dblScopeSpan_Hz = dblScopeSpan_Hz;
     // Queue this up for a later transmission from client to the remote NETWORK server (in CwNet.c)
     RigCtrl_AddParamToTxQueueForCwNet(pRC, RIGCTRL_PN_SCOPE_SPAN );
   }
  if( pPortInstance->iResponseCountdown_ms > 0 ) // still waiting for a response -> "do not send anything at the moment"..
   {  // ex: pRC->iPostponedSetMessages |= (1<<RIGCTRL_MSGTYPE_SCOPE_SPAN); // remember to send pRC->dblScopeSpan_Hz later
      RigCtrl_SetPostponedSetMessageFlag( pRC, RIGCTRL_MSGTYPE_SCOPE_SPAN );
      return TRUE;  // cannot send AT THE MOMENT, but data are (almost) "on the way"
   }
  else // no need to postpone the "Set SCOPE SPAN"-command :
   {  // ex: pRC->iPostponedSetMessages &= ~(1<<RIGCTRL_MSGTYPE_SCOPE_SPAN);
      RigCtrl_ClearPostponedSetMessageFlag( pRC, RIGCTRL_MSGTYPE_SCOPE_SPAN );
      return RigCtrl_SetScopeSpan_Internal( pRC );
   }
} // end RigCtrl_SetScopeSpan_Hz()

//--------------------------------------------------------------------------
static BOOL RigCtrl_SetScopeSpan_Internal( T_RigCtrlInstance *pRC )
  // Internal subroutine for RigCtrl_SetScopeSpan_Hz() .. details there .
  //  [in] pRC->dblScopeSpan_Hz : not passed in as a function argument,
  //                              but already set by the caller .
{
  T_RigCtrl_PortInstance *pPortInstance = &pRC->PortInstance[RIGCTRL_PORT_RADIO];
  BYTE b16TxBuffer[16];
  BYTE *pbBufPtr = b16TxBuffer;
  int  iMsgLength;
  BOOL fResult = FALSE;

  switch( pPortInstance->iRadioCtrlProtocol )
   {
     case RIGCTRL_PROTOCOL_ICOM_CI_V : // Icom's "CI-V" protocol ...
        // About Icom's "Scope span settings", from the IC-7610 CI-V Reference Guide (A7380-7EX-2) page 14:
        // > Scope span settings (in the Center mode and SCROLL-C mode Scope)
        // > Command: 27 15
        //   Message-Byte[0  1  2  3  4  5  6  7  8  9 10 11 12]
        //               FE FE E0 94 27 15 00 00 25 00 00 00 FD (scope) ; span=2500 Hz
        //                           |  |  |_______________|
        //                           |  |  :: '- 12-digit BCD-encoded frequency in HERZ
        //                        Cmd Sub  ::    (only A FEW values are allowed)
        //                                 ::.... 1-Hz-Digit (!), etc.
        //                                 :
        //                                 10-Hz-Digit in the upper 4 bits of this byte
        *pbBufPtr++ = 0xFE;  // fixed preamble code for CI-V protocol (term used by Icom .. NOT "Begin of Message" ! )
        *pbBufPtr++ = 0xFE;  // fixed preamble code for CI-V protocol
        *pbBufPtr++ = (BYTE)pPortInstance->iRadioDeviceAddr;  // "to" transceiver's address
        *pbBufPtr++ = (BYTE)pPortInstance->iRadioMasterAddr;  // "from" controller's address
        *pbBufPtr++ = 0x27;  // 0x27 = CI-V command "Scope Settings"
        *pbBufPtr++ = 0x15;  // 0x15 = "Scope span settings"
        *pbBufPtr++ = 0x00;  // 0x00 for "MAIN", 0x01 would be "SUB" if the radio was an IC-7610 (not used here)
        RigCtrl_IntToBCD_CIV_LSByteFirst( (long)pRC->dblScopeSpan_Hz, pbBufPtr/*pbDest*/, 10/*nDigits*/ );
        pbBufPtr   += 5;     // RigCtrl_IntToBCD_CIV_LSByteFirst( .., .., 10) has appended 5 bytes for 10 digits
        *pbBufPtr++ = 0xFD;  // 0xFD = CI-V code for "End Of Message"
        fResult = RigCtrl_SendAndLogMessage( pRC, RIGCTRL_PORT_RADIO, RIGCTRL_ORIGIN_CONTROLLER,
                 b16TxBuffer/*pbMessage*/, pbBufPtr-b16TxBuffer/*iMsgLength*/,
                 RIGCTRL_MSGTYPE_SCOPE_SPAN,  // message type for the traffic log
                 "set scope span" ); // comment for traffic log
        pPortInstance->dwExpectedResponse_CmdAndSubcode = RCTL_COMBINE_CMD_AND_SUB(0x27,0x15);
        pPortInstance->iResponseCountdown_ms = pRC->iMaxResponseDelay_ms; // now waiting for the "OK"/"NOT OK" response for "Set Scope Span"
        break;
     default: // command not supported (with the current pRC->iRadioCtrlProtocol)
        break;
   } // end switch( pRC->iRadioCtrlProtocol )

  return fResult;


} // end RigCtrl_SetScopeSpan_Internal()


//--------------------------------------------------------------------------
BYTE RigCtrl_IntToBCD2( int iValue ) // converts an integer between 0 and 99 into TWO-DIGIT BCD (in a single byte)
  // This is often used in Icom's CI-V protol as an additional byte
  // after the command and sub-command, e.g. when reading 'array-like' data .
{
  if( iValue < 0 )
   {  iValue = -iValue;
   }
  return (BYTE)( (((iValue / 10) % 10) << 4) | (iValue % 10) );
  //             |__________tens___________|   |___ones____|
}


//--------------------------------------------------------------------------
void RigCtrl_IntToBCD_CIV_LSByteFirst( long i32Value, BYTE *pbDest, int nDigits )
  // More or less inverse to RigCtrl_ParseBCD_CIV_LSByteFirst_Int32(). Details there.
  // Hopefully the number of digits is always EVEN...
{
  int i, iDigit;
  for(i=0; i<nDigits; ++i) // begin with the LEAST SIGNIFICANT digit at i=0 (*)
   { iDigit = i32Value % 10;
     i32Value /= 10;
     if( !(i & 1) )       // 1st, 3rd, 5th digit ? calculate LOW nibble
      { *pbDest = (BYTE)iDigit;
      }
     else                 // 2nd, 4th, 6th digit ? emit the next byte
      { *pbDest |= ((BYTE)iDigit << 4);
        ++pbDest;
      }
     // (*) Unfortunately, Icom uses a bizarre mix of "most significant digit first"
     //      and "least significant digit first". For example, to set a new
     //      "Scope reference level" (command 0x27 0x19),
     //      the first byte contains TENS and ONES, the second byte 0.1 and 0.01 ...
   }
} // end RigCtrl_IntToBCD_CIV_LSByteFirst()

//--------------------------------------------------------------------------
void RigCtrl_IntToBCD_CIV_MSByteFirst( long i32Value, BYTE *pbDest, int nDigits )
  // This is the SECOND (and incompatible) flavour of "BCD" in Icom's CI-V .
  // Used, for example, to set a new "Scope reference level" (command 0x27 0x19),
  // where for reasons not even the Icom engineers will know,
  // the MOST SIGNIFICANT DIGIT (e.g. TENS) is in bit 7..4 of the first BCD byte,
  // the next digit (e.g. the ONES) is in bit 3..0 of the first BCD byte,
  // and the LEAST SIGNIFICANT DIGIT follows last.   OH MY GOD.
  //
{
  int i, iDigit;
  int ai = (nDigits-1)/2;  // first array offset into pbDest[ai] : 0 for 1 or 2 digits, 1 for 3 or 4 digits, etc
  for(i=0; (ai>=0) && (i<nDigits); ++i) // begin with the LEAST SIGNIFICANT digit,
   { // but emit that digit into the LAST BYTE, lower nibble !
     // Example (FOUR-DIGIT BCD as in the "Scope reference level") :
     // i32Value = 1234, nDigits = 4 -> ai = (4-1) & 0x0E = 3
     // i = 0  :  least significant digit, emitted FIRST in this loop,
     //                                      into pbDest[1] bits 3..0,
     // i = 1  :  medium low  digit, emitted into pbDest[1] bits 7..4,
     // i = 2  :  medium high digit, emitted into pbDest[0] bits 3..0,
     // i = 3  :  most significant digit, "  into pbDest[0] bits 7..4 .
     // Example after the conversion: pbDest[0] = 0x12, pbDest[1] = 0x34 .
     iDigit = i32Value % 10;
     i32Value /= 10;
     if( !(i & 1) )  // i=0,2,4, .. : set lower nibble, clear the upper nibble
      { pbDest[ai] = (BYTE)iDigit;
      }
     else            // i=1,3,5, .. : bitwise OR the upper nibble and switch to the next byte
      { pbDest[ai--] |= ((BYTE)iDigit<<4);
      }
   }
} // end RigCtrl_IntToBCD_CIV_MSByteFirst()


//--------------------------------------------------------------------------
long RigCtrl_ParseBCD_CIV_LSByteFirst_Int32( BYTE *pbData, int nDigits )
  // Parses an integer from a BCD sequency with the *LEAST SIGNIFICANT* byte first,
  //        as OFTEN(!) used in Icom's CI-V protocol .
  // Most may actually be 10-digit fields, but padded with zeros.
  // NOT SUITABLE FOR PARSING THE 4-DIGIT BCD PARAMETER NUMBERS
  //              after command 0x17 0x5 - see RigCtrl_ParseBCD_MSDigitFirst()
  // Example:  cmd 0x28, sub-cmd 0x15,  "Scope span settings" :
  //         pbData[0]  [1]     [2]     [4]     [5]
  //         |<----->|
  //          _______________________________________
  //         |   :   |   :   |   :   |   :   |   :   |
  //         | 0 : 0 | X : X | X : X | 0 : 0 | 0 : 0 |  <-- nDigits = 10 (!)
  //         |___:___|___:___|___:___|___:___|___:___|
  //                  /|\ /|\ /|\ /|\ /|\ /|\ /|\ /|\   .
  //                   |   |   |   |   |   |   |   |
  //         (10   1)  1  100 100  10  10MHz ? 1 GHz ?
  //            Hz    kHz  Hz  kHz kHz     1 MHz ? 100 MHz ?
{
  long sum = 0;
  long factor = 1;
  int  i = 0;
  BYTE b;

  while( (nDigits--) > 0 )
    { b = pbData[i++]; // get next byte (with two digits, BCD)
      sum += factor * (long)(b & 0x0F);
      factor *= 10;
      if( (nDigits--) > 0 )
       {
         sum += factor * (long)((b>>4) & 0x0F);
         factor *= 10;
       }
    }
  return sum;

} // end RigCtrl_ParseBCD_CIV_LSByteFirst_Int32()

//--------------------------------------------------------------------------
double RigCtrl_ParseBCD_CIV_LSByteFirst_Double( BYTE *pbData, int nDigits )
  // Parses a 64-bit 'double' from a BCD sequency with the *LEAST SIGNIFICANT* byte first,
  //        as OFTEN(!) used in Icom's CI-V protocol .
  //  (used instead of RigCtrl_ParseBCD_CIV_LSByteFirst_Int32() in some places,
  //   but 32-bit signed integer won't work for frequencies above the 23 cm band).
  // Most may actually be 10-digit fields, but padded with zeros.
  // NOT SUITABLE FOR PARSING THE 4-DIGIT BCD PARAMETER NUMBERS
  //              after command 0x17 0x5 - see RigCtrl_ParseBCD_MSDigitFirst()
  // Example:  cmd 0x28, sub-cmd 0x15,  "Scope span settings" :
  //         pbData[0]  [1]     [2]     [4]     [5]
  //         |<----->|
  //          _______________________________________
  //         |   :   |   :   |   :   |   :   |   :   |
  //         | 0 : 0 | X : X | X : X | 0 : 0 | 0 : 0 |  <-- nDigits = 10 (!)
  //         |___:___|___:___|___:___|___:___|___:___|
  //                  /|\ /|\ /|\ /|\ /|\ /|\ /|\ /|\   .
  //                   |   |   |   |   |   |   |   |
  //         (10   1)  1  100 100  10  10MHz ? 1 GHz ?
  //            Hz    kHz  Hz  kHz kHz     1 MHz ? 100 MHz ?
{
  double sum = 0.0;
  double factor = 1.0;
  int  i = 0;
  BYTE b;

  while( (nDigits--) > 0 )
    { b = pbData[i++]; // get next byte (with two digits, BCD)
      sum += factor * (double)(b & 0x0F);
      factor *= 10.0;
      if( (nDigits--) > 0 )
       {
         sum += factor * (double)((b>>4) & 0x0F);
         factor *= 10.0;
       }
    }
  return sum;

} // end RigCtrl_ParseBCD_CIV_LSByteFirst_Double()

//--------------------------------------------------------------------------
long RigCtrl_ParseBCD_CIV_MSByteFirst( BYTE *pbData, int nDigits )
  // Parses an integer from a BCD sequency with the *MOST SIGNIFICANT* byte first,
  //        as SOMETIMES(!) used in Icom's CI-V protocol .
  //
  // Example:  cmd 0x27, sub-cmd 0x19,  "Scope reference level settings"
  //         pbData[0]  [1]     [2]     [4]
  //         |<----->|<----value---->|<-sign>| "I'm going slightly mad.." (Queen)
  //          _______ _______________ _______
  //         |   :   |   :   |   :   |       |
  //         | 0 : 0 | X : X | X : X | x : x |
  //         |___:___|___:___|___:___|_______|
  //          00=main /|\ /|\ /|\ /|\    '-----  "SIGN" as another two-digit-BCD !
  //          01=sub   |   |   |   |               00 = + (positive)
  //                  10-  1- 0.1- 0.01-           01 = - (negative)
  //                  dB- db-  dB- dB-
  //                 digit ..      digit (!)
{
  long sum = 0;
  int  i = 0;
  BYTE b;

  while( (nDigits--) > 0 )
    { b = pbData[i++]; // get next byte (with two digits, BCD)
      sum = (10*sum) + (long)((b>>4) & 0x0F);
      if( (nDigits--) > 0 )  // 2nd digit from the same source-byte ?
       { sum = (10*sum) + (long)(b & 0x0F);
       }
    }
  return sum;

} // end RigCtrl_ParseBCD_CIV_MSByteFirst()

//--------------------------------------------------------------------------
int RigCtrl_CountBytesUntilDelimiter( BYTE *pbData, BYTE *pbEndstop,
                BYTE bDelimiterA, // [in] e.g. 0x2D for Icom's "Separator"
                BYTE bDelimiterB) // [in] e.g. 0xFD for Icom's "Postamble"
  // Subroutine for a few CI-V command parsers, e.g. RigCtrl_ParseCIVFreqRange()
{
  BYTE *pbStart=pbData;
  while( (pbData < pbEndstop ) && (*pbData!=bDelimiterA) && (*pbData!=bDelimiterB) )
   { ++pbData;
   }
  return pbData-pbStart; // "C" can be beautifully simple if you use pointers carefully :)
} // end RigCtrl_CountBytesUntilDelimiter()

//--------------------------------------------------------------------------
BOOL RigCtrl_ParseCIVFreqRange( // Used e.g. by Icom's cmd 0x1E, subCmd 0x01 .
      BYTE *pbData, int nBytes, // [in] BCD string a la Icom CI-V .
                                //      The leading "edge number" has already
                                //      been parsed (and skipped) by the caller.
      T_RigCtrlFrequencyRange *pFreqRange ) // [out] pFreqRange->dblFmin_Hz, dblFmax_Hz.
{
  BYTE *pbEndstop = pbData + nBytes;
  // pbData on ENTRY --------------,   (Example from an IC-9700. Don't assume
  //                               |    anything about the NUMBER OF DIGITS..
  //                              \|/   things will be different in an IC-905)
  // pbMessage[0  1  2  3  4  5  6  7  8  9 10 11 12 13 14 15 16 17 18]
  // > RX 013 FE FE 00 A2 1E 01 01 00 00 00 44 01 2D 00 00 00 46 01 FD ; 144 MHz .. 146 MHz
  //       |  |___| |addr |  |  |  |____________| |  |____________|  |
  //       | Preamble     |  | "Edge"   |        Sep.    |          Postamble
  //   iMsgLength=19   cmd sub  |     StartFreq   |    EndFreq       |
  //                               |<--iLen=5-->| |  |<--iLen=5-->|  '----,
  //                                              '---------------,       |
  int iLen = RigCtrl_CountBytesUntilDelimiter( pbData, pbEndstop, 0x2D, 0xFD );
  if( iLen>1 ) // at least ONE byte BEFORE the separator (or postamble):
   { pFreqRange->dblFmin_Hz = RigCtrl_ParseBCD_CIV_LSByteFirst_Double( pbData, 2*iLen/*->nDigits*/ );
     pbData += iLen; // <- should point to the 'Separator' (0x2D) now, not to the postamble (0xFD)
     if( (pbData<pbEndstop) && (*pbData==0x2D) ) // Got the expected separator ?
      { ++pbData;      // skip the separator (0x2D) between "Lower edge" and "Higher edge"
        iLen = RigCtrl_CountBytesUntilDelimiter( pbData, pbEndstop, 0x2D, 0xFD );
        if( iLen>1 ) // at least ONE byte with the "Higher edge" (max frequency) ?
         { pFreqRange->dblFmax_Hz = RigCtrl_ParseBCD_CIV_LSByteFirst_Double( pbData, 2*iLen/*->nDigits*/ );
           return TRUE;   // successfully parsed a frequency range a la CI-V (e.g. "Band edge frequencies")
         }
      }
   }
  return FALSE;
} // end RigCtrl_ParseCIVFreqRange()

//--------------------------------------------------------------------------
int RigCtrl_IcomToRigCtrlOpmode( BYTE bIcomsOpMode )
{ // Don't expect the numeric value for "PSK" and "RTTY" to be the same in all rigs !
  // For module RigControl_CIV_Server.c, there's an inverse function
  //                                     - see RigCtrl_RigCtrlOpmodeToIcom() !
  switch( bIcomsOpMode ) // the following list was checked against A7508-3EX-4 (IC-7300) cmd 0x26
   { case 0x00 : return RIGCTRL_OPMODE_LSB;
     case 0x01 : return RIGCTRL_OPMODE_USB;
     case 0x02 : return RIGCTRL_OPMODE_AM;
     case 0x03 : return RIGCTRL_OPMODE_CW;
     case 0x04 : return RIGCTRL_OPMODE_RTTY;
     case 0x05 : return RIGCTRL_OPMODE_FM;
     case 0x06 : return RIGCTRL_OPMODE_FM_WIDE; // used by IC-706 and possibly other VHF-capable radios
     case 0x07 : return RIGCTRL_OPMODE_CW   | RIGCTRL_OPMODE_REVERSE;
     case 0x08 : return RIGCTRL_OPMODE_RTTY | RIGCTRL_OPMODE_REVERSE; // <- seems to be the highest supported mode in an IC-7300
     case 0x09 : return RIGCTRL_OPMODE_UNKNOWN;
     case 0x12 : return RIGCTRL_OPMODE_PSK; // questionable.. snapped up somewhere
     case 0x13 : return RIGCTRL_OPMODE_PSK  | RIGCTRL_OPMODE_REVERSE; // questinable.. snapped up somewhere
     case 0x17 : return RIGCTRL_OPMODE_DIGITAL_VOICE; // seen in page 13 ("Digital Voice", presumably D-Star, not DMR)
     case 0x22 : return RIGCTRL_OPMODE_DIGITAL_DATA; // seen in page 13 (no idea what "DD" stands for yet.. 128 kbp "Digital Data" on 23 cm ?)
       //   '--> Don't assume this is really TWO DIGIT DECIMAL !
       //        In other parameters, which on first glance look like TWO-DIGIT BCD,
       //        but LATER turn out to be HEXADECIMAL BYTES..
       //        e.g. cmd 0x19 sub 0x00 = "Read the transceiver ID",
       //                                 where value 0xA2 means "IC-9700".
       //
     default: return RIGCTRL_OPMODE_UNKNOWN;
   }
} // end RigCtrl_IcomToRigCtrlOpmode()

//--------------------------------------------------------------------------
int RigCtrl_RigCtrlOpmodeToIcom( int iRigCtrlOpmode )
{ int iIcomsOpMode;
  int iRigCtrlOpmode2;
  for(iIcomsOpMode=0; iIcomsOpMode<0x30/*see RigCtrl_IcomToRigCtrlOpmode()..*/; ++iIcomsOpMode )
   { iRigCtrlOpmode2 = RigCtrl_IcomToRigCtrlOpmode( (BYTE)iIcomsOpMode );
     if( iRigCtrlOpmode2 == iRigCtrlOpmode ) // perfect match ?
      { return iIcomsOpMode;
      }
     if( iRigCtrlOpmode2 == (iRigCtrlOpmode & RIGCTRL_OPMODE_MASK_TO_STRIP_FLAGS ) ) // match when ignoring the FILTER BITS ?
      { return iIcomsOpMode;
      }
   }
  return RIGCTRL_NOVALUE_INT; // Arrived here ? No luck.. let the caller to try something else,
         // or use a "meaningful default" for the current frequency band / section.
} // end RigCtrl_RigCtrlOpmodeToIcom()

//--------------------------------------------------------------------------
int RigCtrl_RigCtrlOpmodeToIcomFilterNumber( int iRigCtrlOpmode )
{
  switch( iRigCtrlOpmode & ( RIGCTRL_OPMODE_NARROW | RIGCTRL_OPMODE_VERY_NARROW ) )
   { case RIGCTRL_OPMODE_NARROW      :
          return 0x02; // Icom: "FIL2" = "narrow, but not VERY narrow"
     case RIGCTRL_OPMODE_VERY_NARROW :
     case RIGCTRL_OPMODE_VERY_NARROW | RIGCTRL_OPMODE_NARROW :
          return 0x03; // Icom: "FIL3" = "VERY narrow"
     default :
          return 0x01; // Icom: "FIL1" = traditionally the WIDE filter
   }
} // end RigCtrl_RigCtrlOpmodeToIcomFilterNumber()


//--------------------------------------------------------------------------
BOOL RigCtrl_ParseMemoryContent( // Used for by Icom's commands
                  //  0x1A sub 0x00 ("memory contents") and
                  //  0x1A sub 0x01 ("band stacking register contents").
      BYTE *pbData, int nBytes, // [in] BCD string a la Icom CI-V .
      //     '---> Begins with the "Operating frequency setting" (5),
      //           because that's the first field common of what
      //           Icom calls "Memory Content",
      //  i.e. the response for commands 0x1A 0x00 and 0x1A 0x01 .
      // The IC-7300 "Full Manual" v6 "Section 19 CONTROL COMMAND" leaves a lot
      // of doubt about the format of the "band stacking register contents".
      // Only the IC-9700 "CI-V REFERENCE GUIDE" v4 (A7508-3EX-4) explained:
      //  > NOTE: When sending the contents, the codes, such as
      //  >       operating frequency and operating mode*, should be
      //  >       added after the frequency band code and the register
      //  >       code, as shown below.
      //  >  * See (5) to (51) on 'Memory content setting.' (p. 14)
      //       "Should be added" ? ? Come on Icom, is this a "to do list"
      //                             for your firmware developers ?
      //       "(5)" to "(51)" (decimal numbers in full circles) seem to be
      //       BYTE OFFSETS in the message, where "(5)" is the operating frequency.
      // Unfortunately, even that is already different between IC-7300 and IC-9700 !
      // Fasten seat belt and read on .. the author had a lot of "fun" with this.
      T_RigCtrlFreqMemEntry *pFreqMem ) // [out] an AWFUL lot of members in Icom's "Memory Content"
{
  BYTE *pbEndstop = pbData + nBytes;
  int  nDigitsForFrequency = 10; // <- at least for IC-9700 and IC-7300 (including a 1-GHz-digit)
  //
  // pbData on ENTRY -----------------,   ( Example from an IC-7300's first
  //                                  |     "Band Stacking Register",
  //                                 \|/    including request and response)
  // pbMessage[0  1  2  3  4  5  6  7  8  9 10 11 12 13 14 15 16 17 18]
  // > TX 009 FE FE 00 00 1A 01 01 01 FD  ; BandStackReg[1][1] ?
  // > RX 017 FE FE 00 94 1A 01 01 01 30 96 83 01 00 00 02 01 00 08 85 00 08 85 .. ; band stack mem
  //       |  |___| |addr |  |  |  |  |____________|  |  |  '--- this is where it gets bizarre and incompatible
  //       | Preamble     |  |  |  |  OperatingFreq   | Filter setting: 01..03="Fil1".."Fil3"
  //   iMsgLength=19   cmd sub  |  |  ||  ......| ||  Operating Mode: 00=LSB, etc... (see RigCtrl_IcomToRigCtrlOpmode() )
  //                            |  |  |1Hz      | |100MHz
  // "Frequency Band Code" (1)--'  |  10Hz   1MHz 1GHz
  // "Register Code"       (2)-----'  (0001839.639 MHz)
  RigCtrl_InitFreqMemEntry( pFreqMem );
  pFreqMem->RxTx[0].dblOperatingFreq_Hz = RigCtrl_ParseBCD_CIV_LSByteFirst_Double( pbData, nDigitsForFrequency );
  pbData += (nDigitsForFrequency+1)/2; // <- should now point to the "Operating Mode" ...
  if( pbData < pbEndstop ) // more bytes following ?  Next: two-BCD-digit "Operating Mode"..
   { pFreqMem->RxTx[0].iOpMode = RigCtrl_IcomToRigCtrlOpmode( *(pbData++) );
   }
  if( pbData < pbEndstop ) // more bytes following ?  Next: two-BCD-digit "Filter Setting"..
   { // (In the IC-9700 "CI-V Reference", this field also belongs to the "Operating Mode".
     //  HERE, we only care for the "Filter Setting" if the (unified) OpMode is CW :
     if( (pFreqMem->RxTx[0].iOpMode & RIGCTRL_OPMODE_MASK_TO_STRIP_FLAGS) == RIGCTRL_OPMODE_CW )
      { pFreqMem->RxTx[0].iOpMode &= ~(RIGCTRL_OPMODE_NARROW | RIGCTRL_OPMODE_VERY_NARROW);
        switch( *pbData )
         { case 0x01:  // Icom's "Filter 1" is usually the WIDE filter -> neither "CWN" nor "CWNN"
              break;
           case 0x02:  // Icom's "Filter 2" is usually the "quite narrow" CW filter -> "CWN"
              pFreqMem->RxTx[0].iOpMode |= RIGCTRL_OPMODE_NARROW;
              break;
           case 0x03:  // Icom's "Filter 3" is usually the "very narrow" CW filter -> "CWNN"
              pFreqMem->RxTx[0].iOpMode |= RIGCTRL_OPMODE_VERY_NARROW;
              break;
           default:    // all other "Filter settings" (term used in A7508-3EX-4) are unknown
              break;
          }
      }
     ++pbData; // skip Icom's "Filter Setting" (which THEY SAY belongs to the "Operating Mode")
   }

  // This is where the compatibility between IC-7300 and IC-9700 ends !
  //     After the "Operating Mode" with "Filter Setting", the next part of a
  //    "Memory Content" in an IC-9700 is the "Data mode setting" ("(12)"),
  //                 but in an IC-7300, it's "Data mode and tone type settings".
  if( pbData < pbEndstop ) // more bytes following ?  Next: two-BCD-digit "Data mode setting"..
   { // .. which, in the IC-7300, they call "Data mode and tone type settings" - buaah !
     switch( *pbData & 0xF0 )
      { case 0x00: // upper nibble (tens) set to ZERO ? IC-7300 : NO "data mode" .
             pFreqMem->RxTx[0].fDataMode = FALSE;
             break;
        case 0x10: // upper nibble (tens) set to ONE ? IC-7300 : "Data Mode" .
             pFreqMem->RxTx[0].fDataMode = TRUE;
             break;
        default:  // oops.. ignore all the rest, we're out of business !
             return TRUE; // .. but at least we've got the FREQUENCY and MODE
      }
     switch( *pbData & 0x0F ) // lower nibble (ones) : "Tone Type Settings" in an IC-7300..
      { case 0: pFreqMem->RxTx[0].iToneType = RIGCTRL_TONE_TYPE_OFF;  break;
        case 1: pFreqMem->RxTx[0].iToneType = RIGCTRL_TONE_TYPE_TONE; break;
        case 2: pFreqMem->RxTx[0].iToneType = RIGCTRL_TONE_TYPE_TSQL; break;
        case 3: pFreqMem->RxTx[0].iToneType = RIGCTRL_TONE_TYPE_DTCS; break;
        default: break;
      }
     ++pbData; // skip the "Data mode setting" (IC-9700) /
               //  "Data mode and tone type settings"  (IC-7300 extravaganza)
#   ifdef __BORLANDC__  // "... is assigned a value that is never used".. oh shut up, we know what we're doing !
     (void)pbData;
#   endif

   }

  // In the IC-9700, the above field was just the "Data mode setting",
  //  but they invented a new field that didn't exist in the IC-7300,
  //  thus breaking compatibility.
  // In the IC-9700, the next field is "(13)", "Duplex and Tone settings".
  // In the IC-7300, the "tone type settings" were part of the "Data mode" byte. Eeek.


  return TRUE;
} // end RigCtrl_ParseMemoryContent()


//---------------------------------------------------------------------------
void RigCtrl_ParseSpectrumData_CIV( T_RigCtrlInstance *pRC,
               BYTE *pbPayload, int iPayloadLength,
               char **ppszCmtDest, char *cpCmtEndstop ) // [out,optional] comment for the traffic log
  // Parses the fragment of a spectrum ("Scope waveform data"),
  //     as sent by IC-7300, IC-7610, IC-7851, IC-9700, IC-705, IC-R8600, etc .
  //  [in] pbPayload[0..iPayloadLength-1] : begins with the
  //                  main scope / sub scope indicator byte,
  //                  and ends at the byte before the CI-V postable.
  //
  //  [out] pRC->SpectrumFifo[pRC->iScopeFifoHeadIndex++]
  //               |_.nBinsUsed, dblBinWidth_Hz, dblFmin_Hz, fltMagnitudes_dB[]
  //
  // NOTE: When the spectrum scope on an IC-7300 showed "Scope Out of Range"
  //       because the VFO frequency was too far away from the fixed edge
  //       frequencies, the radio kept sending unsolicited 0x27 0x00 messages,
  //       but WITHOUT waveform data !
{
  int iMainOrSubScope = pbPayload[0];       // IC-7300 : always 0x00 = MAIN scope
  int iFragment = RigCtrl_ParseBCD_CIV_LSByteFirst_Int32( pbPayload+1, 2/*nDigits*/ );  // IC-7300 : 1..11 (decimal)
  int nFragments= RigCtrl_ParseBCD_CIV_LSByteFirst_Int32( pbPayload+2, 2/*nDigits*/ );  // IC-7300 : 11 fragments
  int nBinsInFragment;
  double dblFcenter_Hz, dblHalfSpan_Hz, dblRangeStart_Hz, dblRangeEnd_Hz;
  BYTE *pbSrcBins;
  T_RigCtrl_Spectrum  *pSpectrum;


  // [IC7300FM] page 19-12 :
  // > The 1st data sends only the wave information ( (1) .. (6) )
  // >     without the waveform data (7).
  // > The 2nd or later data sends the minimum wave information ( (1) .. (3) )
  //       with waveform data (7).
  //
  //  "Command 27 00 : Outputs the waveform data to the controller" .
  //   Example from SL's 'CAT traffic monitor', from an IC-7300 :
  //
  //                                 Cmd   pbPayload ...
  //                 Len Pre.  Addr  |<->| [0  1  2  3  4  5  6  7  8  9 10 11 12 13 14]
  // >             0x016 FE FE E0 94 27 00 00 01 11 00 56 34 12 70 00 00 25 00 00 00 00 FD ("waveform info" in CENTER mode)
  //                    /|\          |  |  |           |070.123456MHz|<--Span=2500Hz->|  |
  //                     |<------------------------- Message, here: 22 bytes ----------->|
  //   pbMessage --------'           |  |  |<------- Payload, here: 15 bytes -------->|  |
  //                                 |  |  |  |  |  |  |  |  |                           |
  //           CI-V main command ----'  |  |  |  |  |  |  |  '- (7)...                   CI-V
  //           CI-V sub command --------'  |  |  |  |  |  '---- (6)                      Postamble
  // (1) 00=main scope, 01=sub scope ------'  |  |  |                                    |
  // (2) Division number (Current, aka NOW) --'  |  | <-  here: iFragment                |
  // (3)       "Division number (Maximum *) -----'  | <-  here: nFragments               |
  // (4)        "Center or Fixed mode data" --------' <-  here: iFixedMode               |
  //                 Len             |  |  |  |  |  | |<----from---->|<---- to ------>|  |
  // >             0x016 FE FE E0 94 27 00 00 01 11 01 00 00 00 70 00 00 00 50 70 00 00 FD ("waveform info" in FIXED mode)
  //                                                   0070.000000MHz  000070.500000MHz
  //
  // Fragments #2 to #10 (full length on IC-7300) and last fragment (#11, shorter on IC-7300):
  //     "minimum wave information":       |<----->|
  //     "waveform data"           :       |       |<-------------------------------------------------------------------------------------------------------------------------------------------------->|
  //                             pbPayload[0   1  2  3  4  5  6  7  8  9 10 11 12 13 14 15 16 17 18 19 20 21 22 23 24 25 26 27 28 29 30 31 32 33 34 35 36 37 38 39 40 41 42 43 44 45 46 47 48 49 50 51 52]
  // >             0x03C FE FE E0 94 27 00 00 02 11 32 34 31 30 2D 0D 10 1D 30 30 1C 1A 1A 1B 31 2C 12 2C 30 1C 11 1D 16 31 34 2D 12 16 11 2B 08 2E 33 2D 1E 12 10 1E 29 07 1D 27 10 1D 1C 29 09 08 19 1C FD (fragments #2..16 of 17)
  // >             0x023 FE FE E0 94 27 00 00 11 11 14 10 0D 08 0C 1A 1E 2F 1B 0A 08 13 15 31 34 1C 1C 0F 07 16 0A 1D 1A 1D 0F FD (LAST fragment, #11 of 11 from IC-7300)                                              |__ CI-V postamble (no payload)
  //                                 |  |  |<------- Payload, last fragment : Only 0x23 - 5 = #30 byte = 30 frequency bins-->|
  //                                 |  |  |  |  |
  //           CI-V main command ----'  |  |  |  |
  //           CI-V sub command --------'  |  |  |
  //                                       |  |  |
  //                                       |  |  |
  // (1) 00=main scope, 01=sub scope ------'  |  |  <- IC-7300: always 0x00 = main scope
  // (2) Division number (Current, aka NOW) --'  |  <-  here: iFragment  = 2..11 (decimal)
  // (3)       "Division number (Maximum *) -----'  <-  here: nFragments = 11 (decimal)
  //     No "Center or Fixed mode data" info in fragments #2..11 (thus "min wave info")
  //
  //  (*) The number of "divisions" (fragments in subsequent messages)
  //      depends on the radio model. This parser shall work on ANY radio.
  // Quoted from IC-7300 "Full Manual" page 19-12:
  //  > When sent through the USB port, the data is
  //  > divided by 11 (*) and sent in sequential order.
  //  > The 1st data sends only the wave information ( (1) ~ (6) )
  //  >                 without the waveform data (7).
  //  > The 2nd or later data sends the minimum wave information
  //  >             ( (1) ~ (3) ) with waveform data (7).
  // (*) "divided by 11" : What they mean is "split into 11 fragments" .
  //     Don't take this number of FRAGMENTS per spectrum for granted:
  //   IC-7300, IC-R8600 : "the data is divided by 11" (over USB).
  //   IC-7610, IC-7851  : "the data is divided by 15" (over USB),
  //         but sometimes "all data is sent together" (over LAN).
  if( iMainOrSubScope != 0 )  // only for the MAIN spectrum scope ...
   { return;  // sub-scope not supported. A donated IC-7851 would change this :)
   }

  // Assemble the received fragment in SpectrumFifo[] ..
  pRC->iScopeFifoHeadIndex %= RIGCTRL_SPECTRUM_FIFO_SIZE; // safety first...
  pSpectrum = &pRC->SpectrumFifo[ pRC->iScopeFifoHeadIndex ];

  if( iFragment == 1 )  // first fragment (Icom: "1st data") ?
   { pRC->iScopeBinCounter = 0; // index into pRC->Spectrum[].
     pSpectrum->dblUnixTime= UTL_GetCurrentUnixDateAndTime_Fast(); // capture timestamp a.s.a.p.
     if( pbPayload[3] == 0 ) // 0 = center mode, 1=fixed mode
      { // (this info is only contained in fragment #1)
        pSpectrum->fFixedEdgeMode = FALSE; // from IC-7300/IC-9700 with spectrum scope in CENTER mode
        dblFcenter_Hz  = RigCtrl_ParseBCD_CIV_LSByteFirst_Double( pbPayload+4, 10/*nDigits*/ );
        dblHalfSpan_Hz = RigCtrl_ParseBCD_CIV_LSByteFirst_Double( pbPayload+9, 10/*nDigits*/ );
        pSpectrum->dblCenterFreq_Hz = dblFcenter_Hz;
        // Unfortunately, before the total number of frequency bins is known,
        // it's impossible to determine pSpectrum->dblBinWidth_Hz. Thus:
        pRC->ScopeFreqRangeFromFirstFragment.dblFmin_Hz = dblFcenter_Hz-dblHalfSpan_Hz; // "radio frequency" stored in the 1st bin
        pRC->ScopeFreqRangeFromFirstFragment.dblFmax_Hz = dblFcenter_Hz+dblHalfSpan_Hz; // "radio frequency" stored in the last bin
            // ... without knowing how many bins the radio will actually send !
        SL_AppendPrintf( ppszCmtDest, cpCmtEndstop, "%ld+-%ld kHz",
             (long)(dblFcenter_Hz*1e-3), (long)(dblHalfSpan_Hz*1e-3) );
      }
     else // not "center" mode but "fixed" mode :
      { pSpectrum->fFixedEdgeMode = TRUE; // from IC-7300/IC-9700 with spectrum scope in FIXED EDGE mode
        dblRangeStart_Hz = RigCtrl_ParseBCD_CIV_LSByteFirst_Double( pbPayload+4, 10/*nDigits*/ );
        dblRangeEnd_Hz   = RigCtrl_ParseBCD_CIV_LSByteFirst_Double( pbPayload+9, 10/*nDigits*/ );
        // No problem even with 32 bit integers on 23 cm (IC-9700):
        //     1298MHz / 2^31  =: 0.604428
        // But with 'Microwave transceivers' in mind, frequencies IN HERTZ won't fit
        // inside a microcontroller-friendly 32-bit integer anymore.
        // Thus: RigCtrl_ParseBCD_CIV_LSByteFirst() returns a 64-bit 'double' instead
        //       of a 32-bit integer now. Farewell, adaption of RigControl.c
        //       on a cheap microcontroller !
        pSpectrum->dblCenterFreq_Hz = 0.5 * ( dblRangeStart_Hz + dblRangeEnd_Hz );
        pRC->ScopeFreqRangeFromFirstFragment.dblFmin_Hz = dblRangeStart_Hz;
        pRC->ScopeFreqRangeFromFirstFragment.dblFmax_Hz = dblRangeEnd_Hz;
        SL_AppendPrintf( ppszCmtDest, cpCmtEndstop, "%ld..%ld kHz",
             (long)(dblRangeStart_Hz*1e-3), (long)(dblRangeEnd_Hz*1e-3) );
      }
     pRC->dblSpectrumCenterFreq_Hz = pSpectrum->dblCenterFreq_Hz;

     // The "first data" (first of N fragments) often contains "wave information",
     //       and usually no "waveform data" (spectrum frequency bins) .
     // BUT : An IC-7610 (which not only has USB, but also a LAN port)
     //       may send EVERYTHING in a single fragment, when using LAN !
     // To support both (USB + LAN), use the PAYLOAD LENGTH to know
     // if (as quoted for the IC-7300 further above) this FIRST FRAGMENT
     // really only contains "wave information" but no "waveform data" :
     nBinsInFragment = iPayloadLength - 15; // -> IC-7300 : iPayloadLength = 15, nBinsInFragment = 0, i.e. NO FREQUENCY BINS in the first fragment.
     pbSrcBins = pbPayload + 15;

   } // end if( iFragment == 1 )
  else if( iFragment>1 && (iFragment<=nFragments) )
   { // the "2nd or later data" (2nd to Nth fragment) sends the minimum
     // wave information, no info about the frequency range :
     nBinsInFragment = iPayloadLength - 3; // IC-7300 : 53 minus 3 bytes "minimum wave information"
         // -> max 50 bins per fragment (except the last, which only contained
         //    28 minus 3 = 25 frequency bins -> total = 9 * 50 + 25 = 475 bins.
     pbSrcBins = pbPayload + 3;
     SL_AppendPrintf( ppszCmtDest, cpCmtEndstop, "fragment %d/%d", iFragment, nFragments );
   } // end if( iFragment>1 && (iFragment<=nFragments) )
  else // iFragment is illegal ->
   { return;
   }

  HERE_I_AM__RIGCTRL();  // RigCtrl_iLastSourceLine=__LINE__  (optional..)

  while( ((nBinsInFragment--) > 0)  // anything to append in THIS fragment ?
     &&  (pRC->iScopeBinCounter>=0)
     &&  (pRC->iScopeBinCounter<RIGCTRL_MAX_FREQ_BINS_PER_SPECTRUM) )
   {
#   if( SWI_HARDCORE_DEBUGGING )
     RigCtrl_pbSource = pbSrcBins;
     RigCtrl_iScopeBin= pRC->iScopeBinCounter;
     RigCtrl_pfltDest = pSpectrum->fltMagnitudes_dB;
#   endif
     pSpectrum->fltMagnitudes_dB[ pRC->iScopeBinCounter++ ] = (float)( *pbSrcBins++ );
     // '-- 2024-01-13 : BCB6 crashed here with an access violation,
     //                  shortly after the SERVER received "set_ptt 1" via CwNet.c !
     //     2024-04-21 : read from address ZERO ? ?  (only after pause/resume via debugger)
     //     2024-08-05 : similar trouble again, this time RigCtrl_pbSource confirmed
     //                  that pbSrcBins was NULL. This happened after switching
     //                  to a new BAND STACKING REGISTER though the
     //                  Remote CW Keyer's "band" combo on the "TRX" tab.
     //     2025-01-17 : Again, BCB6 crashed here with an access violation,
     //                  trying to read something from address 0x00000001 ...
     //                  but the local variable contents shown in the debugger
     //                  were all GARBAGE (as usual).
     //                  Crashing in this loop could not be confirmed
     //                  via HERE_I_AM__RIGCTRL() -> RigCtrl_iLastSourceLine.
   }

  HERE_I_AM__RIGCTRL();  // RigCtrl_iLastSourceLine=__LINE__  (optional..)


  pSpectrum->iAmplitudeUnit = SCALE_UNIT_dB; // pSpectrum->fltMagnitudes_dB[] still uses Icom's "phantasy" unit.
  pSpectrum->fltAmplOffset_dBm = -146.0; // offset to convert fltMagnitudes_dB into dBm, see below
  // The CI-V manual only mentions "data range 0..160" but doesn't explain
  //   the scaling (one step = one decibel ? Relative to WHAT ? Details below.)
  // Design goal : Value RETURNED from here in dBm, comparable with S-meter units.
  // > IARU Region 1 Technical Recommendation R.1 defines S9 for the HF bands
  // >  to be a receiver input power of -73 dBm. This is a level of 50 microvolts
  // >  at the receiver's antenna input assuming the input impedance of the
  // > receiver is 50 ohms.
  // > For VHF bands the recommendation defines S9 to be a receiver input power
  // >  of -93 dBm. This is the equivalent of 5 microvolts in 50 ohms.[6]
  // > The recommendation defines a difference of one S-unit corresponds
  // >   to a difference of 6 decibels (dB), equivalent to a voltage ratio
  // >   of two, or power ratio of four.
  // An IC-7300, fed by a "-20 dBm" signal showed ~~ "S9+43dB" at 3.560 MHz,
  //             (Rigol tracking generator)   and ~~ "S9+22dB" WITH ATTENUATOR.
  //             Scope Bandwidth = 500 kHz .
  // With the generator attenuated until the S-meter JUST showed "S0", "S1", "S2", ..,
  //          the peak bin (in pbSrcBins) delivered  :
  // S-meter   |  "Scope waveform data" (value in pbSrcBins)
  // ----------+---------------------------------------------------
  //   "S0"    :  20 without attenuator,   0 (flat line) with attenuator
  //                 (should be  -127 dBm  =  -22 dBuV in a 50-Ohm system)
  //
  //   "S1"    :  26 without attenuator,   still flat line with attenuator
  //                 (should be  -121 dBm  =  -14 dBuV )
  //     ...
  //   "S5"    :  48 (should be   -97 dBm  =  +10 dBuV )
  //   "S6"    :  54 (should be   -91 dBm  =  +16 dBuV )
  //   "S7"    :  61 (should be   -85 dBm  =  +22 dBuV )
  //   "S8"    :  66 (should be   -79 dBm  =  +28 dBuV )
  //   "S9"    :  73 without attenuator,  35 with attenuator (-> "below S3").
  //                 (should be   -73 dBm  =  +34 dBuV )
  //   "S9+20" :  110 (!?!) without "  ,  73 with attenuator (-> "S9" ).
  // -> Conclusion: pbSrcBins uses neither dBm nor dBuV but some 'phantasy dB",
  //                possibly  "dB relative to the own noise floor".
  //                DL4YHF initially subtracted 73+73 = 146, if (in SpecLab)
  //                the result shall be displayed in dBm ( dB "over" a milliwatt) .
  //                But with a dummyload on the RX antenna input,
  //                the spectrum read '-146 dBm' which is nonsense for an
  //                equivalent receiver bandwidth of 10.5 Hz (IC-7300 spanning
  //                5 kHz of "scope bandwidth" in 475 frequency bins) .
  //     From https://en.wikipedia.org/wiki/Minimum_detectable_signal :
  //        > To calculate the minimum detectable signal we first need to establish
  //        > the noise floor in the receiver by the following equation:
  //        > Noise_floor_dBm = 10*log10( kb * To * 1000 ) + NF + 10*log10(BW),
  //        >  where  kb = Boltzmann constant, 1.38 * 10^-23 Joules per Kelvin
  //        >         To = Temperature of the receiver system in Kelvin, e.g. 290 K
  //        >         NF = Receiver noise figure in dB, let's assume NF = 1.5 [dB]
  //        >         BW = Detection bandwidth in Hz, let's use 84 Hz (Icom spectrum scope).
  //        > Numerical example (modified for the IC-7300's spectrum scope) :
  //        > Noise_floor_dBm = 10*log10( kb * To * 1000 ) + NF + 10*log10(BW)
  //        >                 = 10*log10(1.38E-23 * 290 * 1000) + 1.5 + 10*log10(10.5)
  //        >                 = -162.3 dBm .
  //     Ouch. So we're about 162 - 146 = 16 dB "too high" near the noise floor.
  // The MAXIMUM ("data range 160")
  //                would then be 160 "dB" - 146 dBm = +14 dBm = "overload".
  //     A broadcaster on 7.215 MHz, indicated as circa "S9+15 dB" by the IC-7300
  //       reached '119' (non-scaled peak in pbSrcBins), indicated as
  //       was peaking at (119-146) = -27 dBm in spectrum lab (from 5 kHz wide spectrum scope).
  //       "S9 + 15db" should be -73 dBM + 15 dB = -58 dBm,
  //     so here, we're about 58 - 27 = 31 dB "too low" near the maximum level.
  //     WB, 2020-06-07 : Decided to give up at this point.
  //                      An exact "S9" signal from a generator gave
  //                      "-73 dBm" in SL's own waterfall (with "FFT output unit"
  //                      set to "dBm" in Spectrum Lab) with the preamp OFF,
  //                      so decided to keep pSpectrum->fltAmplOffset_dBm .
  //          Then tried different preamp/attenuator settings in the IC-7300 :
  //             No preamp, no attenuator : "S9" on the S-meter, -73 dBm in SL.
  //             Preamp "1" : "S9 + 5dB" on the s-meter,  -47 dBm in SL (nuts..),
  //             Preamp "2" : "S9 + 10dB" on the s-meter, -39 dBm in SL (nuts..),
  //             ATTENUATOR : "S3" on the Icom's s-meter, -118 dBm in SL (CRAZY!).
  //
  if( (iFragment == nFragments ) && (pRC->iScopeBinCounter>1) ) // all fragments complete -> "emit" the spectrum
   {
     // With an IC-7300, got here with pRC->iScopeBinCounter = 475 .
     pSpectrum->nBinsUsed = pRC->nSpectrumBinsUsed = pRC->iScopeBinCounter;
     pSpectrum->dblFmin_Hz = pRC->ScopeFreqRangeFromFirstFragment.dblFmin_Hz;
     pSpectrum->dblBinWidth_Hz = (pRC->ScopeFreqRangeFromFirstFragment.dblFmax_Hz
                                - pRC->ScopeFreqRangeFromFirstFragment.dblFmin_Hz)
                      / (double)( pRC->iScopeBinCounter - 1 );
     // Allow reading this entry in another thread now :
     pRC->iScopeFifoHeadIndex = (pRC->iScopeFifoHeadIndex + 1) % RIGCTRL_SPECTRUM_FIFO_SIZE;
     if( pRC->iScopeFifoHeadIndex == 0 )
      {  pRC->fScopeFifoFull = TRUE;   // important for RigCtrl_GetSpectrumFromFIFO() with RIGCTRL_GET_LATEST_ENTRY
      }
     pRC->iScopeBinCounter = 0; // index into pRC->Spectrum[].
     if( pRC->dblUnixTimeOfFirstSpectrum <= 0.0 )
      {  pRC->dblUnixTimeOfFirstSpectrum = pSpectrum->dblUnixTime;
      }
     ++pRC->nCompleteSpectraReceived;  // total spectrum counter, added 2018-12
     // to find out if the spectrum-update-rate "seen" in Spectrum Lab
     // was really so much lower than it appeared to be on the radio.
     // Test result: see DebugU1.cpp : TDebugForm::Btn_GetReportOnCommsClick().
   }
} // end RigCtrl_ParseSpectrumData_CIV()


//--------------------------------------------------------------------------
T_RigCtrl_Spectrum *RigCtrl_AppendSpectrumToFIFO( T_RigCtrlInstance *pRC,
      double dblBinWidth_Hz, // [in] 'width' of a single FFT bin in Hertz (=resolution
      double dblFmin_Hz,     // [in] "radio frequency" stored in the first frequency bin
      int    nFrequencyBins) // [in] number of 'frequency bins' .
  // Added 2024-03 to use the Spectrum FIFO also for the CLIENT SIDE
  //       in the 'Remote CW Keyer', which receives spectra via TCP/IP
  //       (see cbproj/Remote_CW_Keyer/CwNet.c : )
  // No whistles and bells here, no averaging, etc.
  // Note the fixed limit of RIGCTRL_MAX_FREQ_BINS_PER_SPECTRUM !
  //
  // Returns NULL if the RigControl instance hasn't been properly initialized.
  //
{
  T_RigCtrl_Spectrum *pSpectrum;

  if( (pRC==NULL) || (nFrequencyBins<1) || (nFrequencyBins>RIGCTRL_MAX_FREQ_BINS_PER_SPECTRUM) )
   { return NULL;
   }

  pRC->iScopeFifoHeadIndex %= RIGCTRL_SPECTRUM_FIFO_SIZE; // safety first...
  pSpectrum = &pRC->SpectrumFifo[ pRC->iScopeFifoHeadIndex++ ];
  pRC->iScopeFifoHeadIndex %= RIGCTRL_SPECTRUM_FIFO_SIZE;
  if( pRC->iScopeFifoHeadIndex == 0 ) // similar as in RigCtrl_ParseSpectrumData_CIV() [which ALSO fills the spectrum-FIFO] ..
   {  pRC->fScopeFifoFull = TRUE;   // important for RigCtrl_GetSpectrumFromFIFO() with RIGCTRL_GET_LATEST_ENTRY
   }



  // Fill out the fixed part of the T_RigCtrl_Spectrum, so the spectrum is
  // at least 'valid' when returning, and the caller only needs to populate the
  // bins ( pSpectrum->fltMagnitudes_dB[0..RIGCTRL_MAX_FREQ_BINS_PER_SPECTRUM-1] )
  pSpectrum->nBinsUsed      = nFrequencyBins;
  pSpectrum->dblBinWidth_Hz = dblBinWidth_Hz;
  pSpectrum->dblFmin_Hz     = dblFmin_Hz;
  pSpectrum->dblCenterFreq_Hz= dblFmin_Hz + 0.5 * dblBinWidth_Hz * nFrequencyBins;
  pSpectrum->fFixedEdgeMode = FALSE; // <- just a GUESS here.. the caller may overwrite this when known
  pSpectrum->dblUnixTime = UTL_GetCurrentUnixDateAndTime_Fast(); // capture timestamp a.s.a.p.
  pSpectrum->iAmplitudeUnit = SCALE_UNIT_dB; // pSpectrum->fltMagnitudes_dB[] still uses Icom's "phantasy" unit.
  pSpectrum->fltAmplOffset_dBm = -146.0; // offset to convert fltMagnitudes_dB into dBm, see below

  ++pRC->nCompleteSpectraReceived;   // total counter, displayed on the 'Debug' tab   

  return pSpectrum;
} // end RigCtrl_AppendSpectrumToFIFO()


//--------------------------------------------------------------------------
T_RigCtrl_Spectrum *RigCtrl_GetSpectrumFromFIFO( T_RigCtrlInstance *pRC, int *piTailIndex )
  // API used by Spectrum Lab to retrieve the address of another spectrum in the FIFO.
  // Avoids unnecessary copying. No whistles and bells here, no averaging, etc.
  //     Any reader (caller) should initialize HIS "tail index" (*piTailIndex)
  //     with *piTailIndex = RIGCTRL_GET_OLDEST_ENTRY to retrieve the oldest available entry in the FIFO,
  //       or *piTailIndex = RIGCTRL_GET_LATEST_ENTRY to retrieve the newest (most recent) entry.
  //
  // Returns NULL if no (new) spectrum is available .
  //
  // In Spectrum Lab, periodically called from void TSpectrumLab::ProcessSpectraFromRemoteRig(),
  //    which may have to 'automatically adjust' the DISPLAYED FREQUENCY RANGE
  //    to the radio-frequency-range contained in the returned T_RigCtrl_Spectrum.
  //    (what makes this incredibly complex is the fact that the Icom radio's
  //     *VFO FREQUENCY* may be outside the *DISPLAYED FREQUENCY RANGE*,
  //     especially with the "Scope" (spectrum display) in "Fix" mode ("fixed edge
  //     without automatic scolling", which has been implemented in more recent
  //     firmware versions).
  //     Solution / details in SpecMain.cpp : TSpectrumLab::ProcessSpectraFromRemoteRig() !
{
  T_RigCtrl_Spectrum *pSpectrum = NULL;

  if( *piTailIndex == RIGCTRL_GET_LATEST_ENTRY )  // since 2020-06, dummy for the MOST RECENT spectrum
   { if( pRC->fScopeFifoFull ) // ALL FIFO entries (except the one at iScopeFifoHeadIndex) are valid...
      { *piTailIndex = pRC->iScopeFifoHeadIndex - 1; // signed index wrap below..
        if( *piTailIndex < 0 )
         {  *piTailIndex = RIGCTRL_SPECTRUM_FIFO_SIZE-1;
         }
      }
     else // index hasn't wrapped around yet, so the oldest available info is at index zero
      { *piTailIndex = pRC->iScopeFifoHeadIndex - 1; // if negative ..
        if( *piTailIndex < 0 ) // .. then there's no valid spectrum at all
         {  *piTailIndex = 0;
         }
      }
   }
  else if( *piTailIndex == RIGCTRL_GET_OLDEST_ENTRY ) // not the MOST RECENT but the OLDEST entry ...
   { if( pRC->fScopeFifoFull ) // ALL FIFO entries (except the one at iScopeFifoHeadIndex) are valid...
      { *piTailIndex = pRC->iScopeFifoHeadIndex + 1; // circular wrap further below..
      }
     else // index hasn't wrapped around yet, so the oldest available entry is at index zero
      { *piTailIndex = 0;
      }
   }
  *piTailIndex %= RIGCTRL_SPECTRUM_FIFO_SIZE;

  if( *piTailIndex != pRC->iScopeFifoHeadIndex ) // something new available for THIS caller !
   { pSpectrum = &pRC->SpectrumFifo[ *piTailIndex ];
     *piTailIndex = (*piTailIndex + 1) % RIGCTRL_SPECTRUM_FIFO_SIZE;
   }

  return pSpectrum;
} // end RigCtrl_GetSpectrumFromFIFO()

//--------------------------------------------------------------------------
CPROT int RigCtrl_GetNumSpectraAvailableInFIFO( T_RigCtrlInstance *pRC, int iTailIndex )
  // Called from e.g. OpenWebRX_Server.c to check IN ADVANCE (before
  //  calling RigCtrl_GetSpectrumFromFIFO() ) HOW MANY spectra (or "waterfall lines")
  //  are 'still waiting to be sent' to a certain client.
{
  int nEntries = 0;
  switch( iTailIndex )
   { case RIGCTRL_GET_LATEST_ENTRY : // dummy for the "MOST RECENT spectrum"
     case RIGCTRL_GET_OLDEST_ENTRY : // dummy for the OLDEST entry ...
          if( pRC->fScopeFifoFull ) // ALL FIFO entries (except the one at iScopeFifoHeadIndex) are valid...
           { nEntries = 1;
           }
          else // "scope FIFO" not full, thus check the *HEAD INDEX* :
           { if( pRC->iScopeFifoHeadIndex > 0 )
              { nEntries = 1;
              }
           }
          break;
     default:  // caller passed in a FIFO TAIL index, so compare against the HEAD index:
          nEntries = (pRC->iScopeFifoHeadIndex - iTailIndex);
          if( nEntries < 0 )   // circular wrap between head and tail ..
           {  nEntries += RIGCTRL_SPECTRUM_FIFO_SIZE;
           }
          break;
   }
  return nEntries;
} // end RigCtrl_GetNumSpectraAvailableInFIFO()


//--------------------------------------------------------------------------
BOOL RigCtrl_GetDisplayableRadioFrequencyRange( T_RigCtrlInstance *pRC,
        double *pdblRadioFreqMin, double *pdblRadioFreqMax, // [out] : RADIO  frequencies [Hz]
        BOOL   *pfFixedEdgeMode )   // [out,optional: TRUE= "fixed edges", FALSE="centered on VFO"
  // Spectrum Lab shall be aware of Icom's "FIXED" / "CENTER" mode setting !
  //    - in FIXED-EDGE mode, the frequency scale (plus spectrum, etc)
  //      must not(!) scroll along with the VFO frequency, because
  //      the VFO frequency may even be OUTSIDE the scope's displayed
  //      frequency range (that's not a bug, but a feature of Icom's
  //      spectrum scope).
  //    - in CENTER mode, the frequency scale (plus spectrum, etc)
  //      must (or at least should) scroll along with the VFO frequency.
  //
  // Return value : TRUE = ok ("we know the frequency range"),
  //                FALSE= no info received from the radio yet, using defaults.
  //
  // That's why RigCtrl_GetDisplayableRadioFrequencyRange() optionally returns
  //      the spectrum scope's current "FIXED" / "CENTER" setting .
{
  T_RigCtrl_Spectrum *pSpectrum;
  int iTailIndex;
  double dblSpan_Hz;

  // When already running, use the most recent spectrum to find the range :
  iTailIndex = RIGCTRL_GET_LATEST_ENTRY;
  pSpectrum  = RigCtrl_GetSpectrumFromFIFO( pRC, &iTailIndex );
  if( pSpectrum != NULL )
   { if( pfFixedEdgeMode != NULL )
      { *pfFixedEdgeMode =  pSpectrum->fFixedEdgeMode; // TRUE=fixed frequencies, FALSE=spectrum centered on VFO
      }
     // Bugfix 2020-06-19 : Span_Hz = (nBinsUse MINUS ONE) * BinWidth_Hz  !
     //        If there are only TWO bins per spectrum, the total span
     //        (from the center of the first  to  the center of the last bin)
     //        is only ONE frequency bin width !
     // ex: dblSpan_Hz = (double)pSpectrum->nBinsUsed * pSpectrum->dblBinWidth_Hz;
     dblSpan_Hz = (double)(pSpectrum->nBinsUsed-1) * pSpectrum->dblBinWidth_Hz;
     //   After this modification, the "sp" edit field in SL's main window
     //   showed the correct span ( 1000.0 kHz, not 1.0021 MHz, for an IC-7300 ).
     if( dblSpan_Hz > 0.0 )
      { *pdblRadioFreqMin = pSpectrum->dblCenterFreq_Hz - 0.5 * dblSpan_Hz;
        *pdblRadioFreqMax = pSpectrum->dblCenterFreq_Hz + 0.5 * dblSpan_Hz;
        return TRUE;
      }
   } // end if( pSpectrum != NULL )

  // Called "early", or spectrum FIFO still empty for any other reason: 2nd guess !
  // If it's an Icom radio, it may have reported its "Span setting in the Center mode Scope"...
  if( pRC->dblScopeSpan_Hz > 0.0 ) // An IC-7300 with a 5 kHz wide display reported "2500 Hz",
   { // .. but that applies to BOTH SIDES of the "center"- (not always the "VFO"-) frequency !
     *pdblRadioFreqMin = pRC->dblVfoFrequency - pRC->dblScopeSpan_Hz;
     *pdblRadioFreqMax = pRC->dblVfoFrequency + pRC->dblScopeSpan_Hz;
     return TRUE;
   }

  // Arrived here ? The spectrum scope's frequency range is completely unknown,
  //  so return the WIDEST possible setting (*) .. if we know the VFO frequency:
  if( pRC->dblVfoFrequency > 0.0 )
   { *pdblRadioFreqMin = pRC->dblVfoFrequency-500e3;  // 1 MHz total span from IC-7300
     *pdblRadioFreqMax = pRC->dblVfoFrequency+500e3;
     // (*) Returning FALSE from here made Spectrum Lab use the last known
     //     frequency range, stored between two sessions in e.g.
     //     UConfig.SpDisplay[0].Chn[0].dblBasebandFreqMin/Max, added to
     //     UConfig.SDR_dblNCOfreq .
   }
  return FALSE;

} // end RigCtrl_GetDisplayableRadioFrequencyRange()


//--------------------------------------------------------------------------
int RigCtrl_SubcodeFrom1A_to_UnifiedParameterNumber( T_RigCtrlInstance *pRC,
           int iSubcode )
{
  switch( pRC->iDefaultAddress )
   { case RIGCTRL_DEF_ADDR_IC_7300 :
        switch( iSubcode ) // IC-7300-specific !!
         {
         //case 0x05 :  // an AWFUL lot of even more rig-specific sub-sub-parameters,
              // identified by a four-digit BCD coded decimal number like e.g.
              //    "0198" for the "Waterfall Speed" in an IC-9700,
              //    "0108" for the "Waterfall Speed" in an IC-7300, etc^10 .
              // A SMALL SUBSET of those parameters is supported
              //   by RigCtrl_SubcodeFrom1A05_to_UnifiedParameterNumber() .
              //
         //case 0x06 :  // "DATA mode with filter width setting" (IC-7300)
              // There's a hard-coded parser for this in RigCtrl_ParseCIV() !
           case 0x07 :  // "IP+ Setting"       (IC-7300)
              return RIGCTRL_PN_IP_PLUS;  // in many rigs, "IP+ ON" == "Preamp OFF".
                 // But there's a different command / subcode for the PREAMP,
                 // so at least in THIS radio, "IP+" may be something more...
           default:
              break;
         }
        break;
     case RIGCTRL_DEF_ADDR_IC_7610 :
        switch( iSubcode ) // IC-7610-specific !!
         {
         //case 0x05 :  // an AWFUL lot of even more rig-specific sub-sub-parameters,
              // identified by a four-digit BCD coded decimal number like e.g.
              //    "0198" for the "Waterfall Speed" in an IC-9700,
              //    "0108" for the "Waterfall Speed" in an IC-7300, etc^10 .
              // A SMALL SUBSET of those parameters is supported
              //   by RigCtrl_SubcodeFrom1A05_to_UnifiedParameterNumber() .
              //
           case 0x07 :  // "NTP server access" (IC-7610)
              return RIGCTRL_PN_NTP_SERVER_ACCESS;
           case 0x08 :  // "NTP access result" (IC-7610)
              return RIGCTRL_PN_NTP_ACCESS_RESULT;
           case 0x09 :  // "AF Mute"           (IC-7610)
              return RIGCTRL_PN_AF_MUTE;
           default:
              break;
         }
        break;
     case RIGCTRL_DEF_ADDR_IC_7760 : // no special support for this expensive radio yet..
        break;

     default:
        break;
   }

  // Arrived here ? Good luck digging out the manual, and finding out
  //  the meaning of the subcode directly after command 0x1A yourself !
  return RIGCTRL_PN_UNKNOWN;


} // end  RigCtrl_SubcodeFrom1A_to_UnifiedParameterNumber()


//--------------------------------------------------------------------------
int RigCtrl_SubcodeFrom1A05_to_UnifiedParameterNumber(
        int iDefaultAddress, // [in] e.g. RIGCTRL_DEF_ADDR_IC_7300 (0x94),
                             //           RIGCTRL_DEF_ADDR_IC_9700 (0xA2),
                             //   etc etc etc etc etc etc etc etc etc etc.
        int iSubcode )
  // Tries to convert A FEW(!) of those CRAZY rig-specific 4-digit subcodes
  //       after Icom's command 0x1A 0x05 into a UNIFIED, and CONSTANT
  //       parameter number that is usable for a switch-case list in "C".
  //       It's a shame that not even an IC-7300 and IC-7610 are compatible
  //       in this context . They could have simply added NEW "subcodes"
  //       for the IC-7610 (for parameters that don't exist in an IC-7300),
  //       but the Icom developers didn't. Grrr.
  //  [in] pRC->iDefaultAddress : e.g. RIGCTRL_DEF_ADDR_IC_7300, .._7610, etc.
  //       Has been read via CI-V *from the radio itself*.
  //  [in] iSubcode = rig-specific number, parsed from a 4-digit BCD coded
  //       value after CI-V command 0x1A 0x05 .
  //  Return value : RIGCTRL_PN_CIV_USB_ECHO, etc
  //       (only those parameter that appeared important are supported here)
  //  See also (inverse to RigCtrl_SubcodeFrom1A05_to_UnifiedParameterNumber() ):
  //                       RigCtrl_GetCrazyModelDependingSubSubcodeForCmd1A_05()
{
  switch( iDefaultAddress )
   { case RIGCTRL_DEF_ADDR_IC_7300 : // here: Sub-Sub-codes from cmd 0x1A sub 0x05 ...
        switch( iSubcode ) // IC-7300-specific ! Consult the IC-7300 "Full Manual" (there's no extra "CI-V REFERENCE GUIDE" for that radio),
                           //                    e.g. "A7292-4EX-11" (Version 11 or later; older versions were very incomplete) !
         { case    1 : return RIGCTRL_PN_SSB_RX_HPF_LPF;
           case    2 : return RIGCTRL_PN_SSB_RX_BASS_LEVEL;
           case    3 : return RIGCTRL_PN_SSB_RX_TREBLE_LEVEL;
           // etc etc, the above were just A FEW examples.. now for the important ones,
           // especially those used by "RS-BA1" and by WFView :
           case   52 : return RIGCTRL_PN_SSB_CW_SYNC_TUNING; // "SSB/CW synchronous tuning". In IC-7300, PN #0052. Used by RS-BA1.
           case   57 : return RIGCTRL_PN_CALIBRATION_MARKER; // "Calibration marker". In IC-7300, PN #0057. Used by RS-BA1.
           case   60 : return RIGCTRL_PN_AF_OUTPUT_LEVEL_TO_USB; // .. or "to ACC or USB". In IC-7300, PN #0060
           case   59 : return RIGCTRL_PN_AF_IF_OUT_SELECTOR; // "Send/read AF/IF signal output to ACC/USB". IC-7300: #0059
           case   61 : return RIGCTRL_PN_SQUELCH_FOR_AF_ON_USB;  // "Squelch function for the AF signal output to ACC/USB"
           case   62 : return RIGCTRL_PN_SIDETONE_ON_USB_AUDIO;  // IC-7300: "Beep and speech output setting to ACC/USB" 0x1A 0x05 #0062
           case   63 : return RIGCTRL_PN_IF_OUTPUT_LEVEL_TO_ACC; // "IF signal output level to ACC/USB" (0..255 for 100%)
           case   64 : return RIGCTRL_PN_MOD_INPUT_LEVEL_FROM_ACC; // "MOD input level from ACC" (0..255 for 100%)
           case   65 : return RIGCTRL_PN_MOD_INPUT_LEVEL_FROM_USB; // "MOD input level from USB" (0..255 for 100%)
           case   66 : return RIGCTRL_PN_MOD_IN_CONN_FOR_DATA_OFF; // "MOD input connector during DATA OFF" (0=MIC, 1=ACC, 2=MIC/ACC, 3=USB)
           case   67 : return RIGCTRL_PN_MOD_IN_CONN_FOR_DATA; // "MOD input connector during DATA" (0=MIC, 1=ACC, 2=MIC/ACC, 3=USB)
              //  '-->  Despite having this in the list, RCW-Keyer didn't understand what WFView wanted, and showed THIS in the log:
              // > Nr time/ms tr len pream to fm CIV-cmd, params..  postamble=FD
              // >  6 0000884 r2 009 FE FE 94 E1 1A 05 00 67 FD ; Cfg[0067] ?
              //   ,----------------------------------------------|_______|
              //   '--> Reason for emitting this 'dummy' in RigCtrl_ParseCIV() :
              //   The remote client (WFView) had asked for this "Mod-input-connector-during-DATA"
              //   long before the rig's DEFAULT CI-V ADDRESS was known,
              //   so RigCtrl_SubcodeFrom1A05_to_UnifiedParameterNumber()
              //   had no chance to find out what 0x15 0x05 0x00 0x67 was.
              // Fixed by ignoring commands received on ports with .fActAsServer=TRUE,
              // until the RADIO PORT has at least finished reading the basic parameters.
           case   71 : return RIGCTRL_PN_CIV_TRANSCEIVE;
           case   72 : return RIGCTRL_PN_CIV_ADDR_USB_TO_REMOTE; // data: hex address
           case   73 : return RIGCTRL_PN_CIV_OUTPUT_FOR_ANT;
           case   74 : return RIGCTRL_PN_CIV_UNLINK_FROM_REMOTE; // data: 0=linked, 1=unlinked
           case   75 : return RIGCTRL_PN_CIV_USB_ECHO;
           case  103 : break; // "spectrum display FILL type" : 0 = fill, 1 = fill + line
           case  108 : // <- IC-7300 : #0108 (A7292-4EX-11 page "19-5" or 163),  IC-9700 : #0198
                       if( iDefaultAddress == RIGCTRL_DEF_ADDR_IC_7300 )
                        { // 0x1A 0x05 "0108" is the "Waterfall Speed" in an IC-7300 (but not in other rigs of the same family)
                          return RIGCTRL_PN_WATERFALL_SPEED;
                        }
                       break;
           case  198 : // <- IC-7300 : #0108 (A7292-4EX-11 page "19-5" or 163),  IC-9700 : #0198
                       if( iDefaultAddress == RIGCTRL_DEF_ADDR_IC_9700 )
                        { // 0x1A 0x05 "0198" for the "Waterfall Speed" in an IC-9700 (but not in other rigs of the same family)
                          return RIGCTRL_PN_WATERFALL_SPEED;
                        }
                       break;
           // If we needed more of this stuff, it would be time to implement something
           // like WFView's "Rig File" (kind of database for each transceiver).
           // But the effort / usefullness ratio appeared too high, in 2025-07 !
           default   : break;
         }
        break; // end case RIGCTRL_DEF_ADDR_IC_7300

     case RIGCTRL_DEF_ADDR_IC_7610 : // here: Sub-Sub-codes from cmd 0x1A sub 0x05 ...
        switch( iSubcode ) // IC-7610-specific !!
         { case    1 : return RIGCTRL_PN_SSB_RX_HPF_LPF;
           case    2 : return RIGCTRL_PN_SSB_RX_BASS_LEVEL;
           case    3 : return RIGCTRL_PN_SSB_RX_TREBLE_LEVEL;
           // etc etc, the above were just A FEW examples.. now for the important ones:
           case  112 : return RIGCTRL_PN_CIV_TRANSCEIVE;
           case  113 : return RIGCTRL_PN_CIV_ADDR_USB_TO_REMOTE; // here: also "LAN to remote"
           case  114 : return RIGCTRL_PN_CIV_OUTPUT_FOR_ANT;
           case  115 : return RIGCTRL_PN_CIV_UNLINK_FROM_REMOTE; // data: 0=linked, 1=unlinked
           case  116 : return RIGCTRL_PN_CIV_USB_ECHO;
           default   : break;
         }
        break; // end case RIGCTRL_DEF_ADDR_IC_7610

     case RIGCTRL_DEF_ADDR_IC_9700 : // here: Sub-Sub-codes from cmd 0x1A sub 0x05 ...
        switch( iSubcode ) // IC-9700-specific ! Consult the IC-9700 "CI-V REFERENCE GUIDE",
                           // e.g. "A7508-3EX-4" (Version 4 or later; older versions were incomplete) !
         { case    1 : return RIGCTRL_PN_SSB_RX_HPF_LPF;     // hey whow.. THE SAME AS FOR THE IC-7300,
           case    2 : return RIGCTRL_PN_SSB_RX_BASS_LEVEL;  // .. but the compatibility ends at some higher-numbered parameter
           case    3 : return RIGCTRL_PN_SSB_RX_TREBLE_LEVEL;
           // etc etc, the above were just A FEW examples.. now for the important ones, especially those used by "RS-BA1":
           case   66 : return RIGCTRL_PN_SSB_CW_SYNC_TUNING; // "SSB/CW synchronous tuning". In IC-7300, PN #0052. Used by RS-BA1.
   // no such thing here: case57 : return RIGCTRL_PN_CALIBRATION_MARKER; // "Calibration marker". In IC-7300, PN #0057. Used by RS-BA1.
      //   case  100 : return RIGCTRL_PN_AF_IF_OUT_SELECTOR;     // IC-9700: "SET > Connectors > ACC AF/IF Output > Output Select (0=AF,1=IF)" : 0x1A 0x05 #0100
      //   case  101 : return RIGCTRL_PN_AF_OUTPUT_LEVEL_TO_USB; // IC-9700: "SET > Connectors > ACC AF/IF Output > AF Output Level": 0x1A 0x05 #0101
      //   case  102 : return RIGCTRL_PN_SQUELCH_FOR_AF_ON_USB;  // IC-9700: "SET > Connectors > ACC AF/IF Output > AF SQL": 0x1A 0x05 #0102
      //   case  103 : return RIGCTRL_PN_SIDETONE_ON_USB_AUDIO;  // IC-9700: "SET > Connectors > ACC AF/IF Output > AF Beep/Speech... Output": 0x1A 0x05 #0103
      //   case  104 : return RIGCTRL_PN_IF_OUTPUT_LEVEL_TO_ACC; // IC-9700: "SET > Connectors > ACC AF/IF Output > ACC IF Output Level": 0x1A 0x05 #0104
           case  105 : return RIGCTRL_PN_AF_IF_OUT_SELECTOR;     // IC-9700: "SET > Connectors > USB AF/IF Output > Output Select (0=AF,1=IF)" : 0x1A 0x05 #0105
           case  106 : return RIGCTRL_PN_AF_OUTPUT_LEVEL_TO_USB; // IC-9700: "SET > Connectors > USB AF/IF Output > AF Output Level": 0x1A 0x05 #0106
           case  107 : return RIGCTRL_PN_SQUELCH_FOR_AF_ON_USB;  // IC-9700: "SET > Connectors > USB AF/IF Output > AF SQL": 0x1A 0x05 #0107
           case  108 : return RIGCTRL_PN_SIDETONE_ON_USB_AUDIO;  // IC-9700: "SET > Connectors > USB AF/IF Output > AF Beep/Speech... Output": 0x1A 0x05 #0108
           case  109 : return RIGCTRL_PN_IF_OUTPUT_LEVEL_TO_ACC; // IC-9700: "SET > Connectors > USB AF/IF Output > ACC IF Output Level": 0x1A 0x05 #0109
      // If one bad day, we need to support AUDIO VIA LAN on the IC-9700, there are even more parameters to care for:
      //   case  110 : return RIGCTRL_PN_AF_IF_OUT_SELECTOR;     // IC-9700: "SET > Connectors > LAN AF/IF Output > Output Select (0=AF,1=IF)" : 0x1A 0x05 #0110
      //   case  111 : return RIGCTRL_PN_AF_OUTPUT_LEVEL_TO_USB; // IC-9700: "SET > Connectors > LAN AF/IF Output > AF Output Level": 0x1A 0x05 #0111
   //      case  112 : return RIGCTRL_PN_MOD_INPUT_LEVEL_FROM_ACC; // IC-9700: "SET > Connectors > MOD Input > ACC MOD Level": 0x1A 0x05 #0112
           case  113 : return RIGCTRL_PN_MOD_INPUT_LEVEL_FROM_USB; // IC-9700: "SET > Connectors > MOD Input > USB MOD Level": 0x1A 0x05 #0113
   //      case  114 : return RIGCTRL_PN_MOD_INPUT_LEVEL_FROM_LAN; // IC-9700: "SET > Connectors > MOD Input > LAN MOD Level": 0x1A 0x05 #0114
           case  115 : return RIGCTRL_PN_MOD_IN_CONN_FOR_DATA_OFF; // IC-9700: "SET > Connectors > MOD Input > DATA OFF MOD" : 0x1A 0x05 #0115
           case  116 : return RIGCTRL_PN_MOD_IN_CONN_FOR_DATA;     // IC-9700: "SET > Connectors > MOD Input > DATA MOD" : 0x1A 0x05 #0116
                       // with these values: 00=MIC, 01=ACC, 02=MIC,ACC, 03=USB, 04=MIC,USB, 05=LAN . A7508-3EX-4 page 7.
   //      case   71 : return RIGCTRL_PN_CIV_TRANSCEIVE;
   //      case   72 : return RIGCTRL_PN_CIV_ADDR_USB_TO_REMOTE; // data: hex address
   //      case   73 : return RIGCTRL_PN_CIV_OUTPUT_FOR_ANT;
           case  129 : return RIGCTRL_PN_CIV_UNLINK_FROM_REMOTE; // data: 0=linked, 1=unlinked . IC-7300 : #0074
           case  130 : return RIGCTRL_PN_CIV_USB_ECHO;    // <- IC-7300 : #0075

           case  198 : return RIGCTRL_PN_WATERFALL_SPEED; // <- IC-7300 : #0108,  IC-9700 : #0198
           default   : break;
         }
        break; // end case RIGCTRL_DEF_ADDR_IC_9700

     case RIGCTRL_DEF_ADDR_IC_7760 : // here: Sub-Sub-codes from cmd 0x1A sub 0x05 ...
        // No real support for this expensive radio yet
        break;

     default:
        break;

   } // end switch( pRC->iDefaultAddress )

  // Arrived here ? Good luck digging out the manual, and finding out
  //  the meaning of the subcode for command 0x1A 0x05 yourself !
  return RIGCTRL_PN_UNKNOWN;
} // end RigCtrl_SubcodeFrom1A05_to_UnifiedParameterNumber()


//--------------------------------------------------------------------------
int RigCtrl_GetCrazyModelDependingSubSubcodeForCmd1A_05( // the name says it all !
        int iDefaultAddress, // [in] e.g. RIGCTRL_DEF_ADDR_IC_7300 (0x94),
                             //           RIGCTRL_DEF_ADDR_IC_9700 (0xA2),
                             //   etc etc etc etc etc etc etc etc etc etc.
        int iUnifiedPN ) // [in] e.g. RIGCTRL_PN_CIV_TRANSCEIVE_MODE, RIGCTRL_PN_WATERFALL_SPEED,
                         //      and maybe a few others in the "crazy" command group 0x1A,
                         //      sub-code 0x05, and ever-changing sub-sub-codes.
  // [return] The crazy model-dependent four-digit decimal sub-sub-code that reads
  //          or writes the value for <iUnifiedPN> for a certain Icom radio.
  //          See HARD-CODED examples in the nested switch/case lists below.
  //          One fine day, we may extract this pesky stuff from WFView's
  //          "rig files".
  //          Note that the return value is a DECIMAL NUMBER (normal integer
  //          like e.g. 321 (int) for "0321" = "Set the NB Level (144 MHz) in an IC-9700.
  //          In a CI-V messages, that value is sent FOUR-DIGIT BCD = TWO BYTES,
  //           e.g. "0321" = 0x03 0x21, followed by the VALUE itself.
  // Returns ZERO if the hard-coded lists (below) don't know the 4-digit code
  // for the requested UNIFIED (i.e. not-even-ICOM-specific) parameter number.
{
  int iSubSubcode; // .. for CI-V command 0x1A, sub-command 0x05 ..

  // Here comes the MODEL-DEPENDENT madness again !
  // For the most common (and frequently used) parameter, use a hard-coded
  // switch/case list. If no match can found there, use an "inverse" search loop
  // further below, calling RigCtrl_SubcodeFrom1A05_to_UnifiedParameterNumber().
  switch( iDefaultAddress )
   {
     case RIGCTRL_DEF_ADDR_IC_705 : // 0xA4 (!)
        switch( iUnifiedPN )
         { case RIGCTRL_PN_CIV_TRANSCEIVE_MODE: return 112; // FOR THE IC-705, "CI-V Transceive" is cmd 0x1A 0x05 "0112".
           case RIGCTRL_PN_WATERFALL_SPEED    : return 184; // FOR THE IC-705, "Waterfall Speed" is cmd 0x1A 0x05 "0185".
           default: break;
         }
        break; // end case RIGCTRL_DEF_ADDR_IC_705
     case RIGCTRL_DEF_ADDR_IC_905 : // 0xAC (?)
        switch( iUnifiedPN )
         { case RIGCTRL_PN_CIV_TRANSCEIVE_MODE: return 142; // FOR THE IC-905, "CI-V Transceive" is cmd 0x1A 0x05 "0142".
           case RIGCTRL_PN_WATERFALL_SPEED    : return 195; // FOR THE IC-905, "Waterfall Speed" is cmd 0x1A 0x05 "0195".
           default: break;
         }
        break; // end case RIGCTRL_DEF_ADDR_IC_905
     case RIGCTRL_DEF_ADDR_IC_7100: // 0x88 (?)
        switch( iUnifiedPN )
         { case RIGCTRL_PN_CIV_TRANSCEIVE_MODE: break; // unknown .. please add support !
           case RIGCTRL_PN_WATERFALL_SPEED    : break; // unknown .. please add support !
           default: break;
         }
        break; // end case RIGCTRL_DEF_ADDR_IC_7100
     case RIGCTRL_DEF_ADDR_IC_7200: // 0x76 (?)
        break; // end case RIGCTRL_DEF_ADDR_IC_7200
     case RIGCTRL_DEF_ADDR_IC_7300: // 0x94 (!)
        switch( iUnifiedPN )
         { case RIGCTRL_PN_CIV_TRANSCEIVE_MODE: return 71;  // FOR THE IC-7300, "CI-V Transceive" is cmd 0x1A 0x05 "0071".
           case RIGCTRL_PN_WATERFALL_SPEED    : return 108; // FOR THE IC-7300, "Waterfall Speed" is cmd 0x1A 0x05 "0108".
           case RIGCTRL_PN_REFERENCE_FREQ_OFFSET: return 58; // IC-7300 : "Reference Frequency Offset" = cmd 0x1A 0x05 "0058"
           case RIGCTRL_PN_AF_IF_OUT_SELECTOR: return 59; // IC-7300: "AF_IF_OutputSelector" = cmd 0x1A 0x05 "0059"
           case RIGCTRL_PN_AF_OUTPUT_LEVEL_TO_USB:  return 60; // IC-7300: "AF_out_level_on_ACC" = cmd 0x1A 0x05 "0060"
           case RIGCTRL_PN_SQUELCH_FOR_AF_ON_USB:   return 61; // IC-7300: "SquelchForAudioOnAcc" = cmd 0x1A 0x05 "0061"
           case RIGCTRL_PN_SIDETONE_ON_USB_AUDIO :  return 62; // IC-7300: "Beep and speech output setting to ACC/USB": 0x1A 0x05 #0062
           case RIGCTRL_PN_IF_OUTPUT_LEVEL_TO_ACC:  return 63; // IC-7300: "IF_out_level_on_ACC" = cmd 0x1A 0x05 "0063"
           case RIGCTRL_PN_MOD_INPUT_LEVEL_FROM_ACC:return 64; // IC-7300: "ModInputLevelFromACC"= cmd 0x1A 0x05 "0064"
           case RIGCTRL_PN_MOD_INPUT_LEVEL_FROM_USB:return 65; // IC-7300: "ModInputLevelFromUSB"= cmd 0x1A 0x05 "0065"
           case RIGCTRL_PN_MOD_IN_CONN_FOR_DATA_OFF:return 66; // IC-7300: "ModInputConnector_NoData" = cmd 0x1A 0x05 "0066"
           case RIGCTRL_PN_MOD_IN_CONN_FOR_DATA  :  return 67; // IC-7300: "ModInputConnector_Data"= cmd 0x1A 0x05 "0067"
           default: break;
         }
        break; // end case RIGCTRL_DEF_ADDR_IC_7300
     case RIGCTRL_DEF_ADDR_IC_9700: // 0xA2 (!)
        switch( iUnifiedPN )
         { case RIGCTRL_PN_CIV_TRANSCEIVE_MODE: return 127; // FOR THE IC-9700, "CI-V Transceive" is cmd 0x1A 0x05 "0127".
           case RIGCTRL_PN_WATERFALL_SPEED    : return 198; // FOR THE IC-9700, "Waterfall Speed" is cmd 0x1A 0x05 "0198".
               // Again: It's a shame that not even an IC-7300 and IC-9700 are compatible
               // in this context . You would spend YEARS adding support
               // for all those crazy "sub-subcodes" for command 0x1A sub 0x05,
               // and for all those INCOMPATIBLE radios (IC-7300, IC-9700,
               //  IC-705, IC-905, IC-7610, IC-7760, to name just a few)
           default: break;
         }
        break; // end case RIGCTRL_DEF_ADDR_IC_9700
     case RIGCTRL_DEF_ADDR_IC_7400: // 0x66 (?)
        break; // end case RIGCTRL_DEF_ADDR_IC_7400
     case RIGCTRL_DEF_ADDR_IC_7600: // 0x7A (?)
        break; // end case RIGCTRL_DEF_ADDR_IC_7600
     case RIGCTRL_DEF_ADDR_IC_7610: // 0x98 (?)
        switch( iUnifiedPN )
         { case RIGCTRL_PN_CIV_TRANSCEIVE_MODE: return 71;  // FOR THE IC-7610, "CI-V Transceive" is cmd 0x1A 0x05 "0112".
           case RIGCTRL_PN_WATERFALL_SPEED    : break;      // FOR THE IC-7610, "Waterfall Speed" has not been looked up yet.
           default: break;
         }
        break; // end case RIGCTRL_DEF_ADDR_IC_7610
     case RIGCTRL_DEF_ADDR_IC_7700: // 0x74 (?)
        break; // end case RIGCTRL_DEF_ADDR_IC_7700
     case RIGCTRL_DEF_ADDR_IC_7760: // 0xB2 (?)
        break; // end case RIGCTRL_DEF_ADDR_IC_7760
     case RIGCTRL_DEF_ADDR_IC_7800: // 0x6A (?)
        break; // end case RIGCTRL_DEF_ADDR_IC_7800
     case RIGCTRL_DEF_ADDR_IC_7851: // 0x8F (?)
        switch( iUnifiedPN )
         { case RIGCTRL_PN_CIV_TRANSCEIVE_MODE: return 155; // FOR THE IC-7851, "CI-V Transceive" is cmd 0x1A 0x05 "0155".
           case RIGCTRL_PN_WATERFALL_SPEED    : break;      // FOR THE IC-7851, "Waterfall Speed" has not been looked up yet.
           default: break;
         }
        break; // end case RIGCTRL_DEF_ADDR_IC_7851
     default:
        break;
   } // end switch( iDefaultAddress ) for the CRAZY command 0x1A, sub 0x05, and hundreds of RIG-SPECIFIC sub-sub-commands


  // Arrived here ? No 'quick lookup' in the switch/case lists above,
  //                so use a SEARCH LOOP with hundreds of calls
  //                to RigCtrl_SubcodeFrom1A05_to_UnifiedParameterNumber()
  //                until we found the right one for the currently used radio.
  //
  // Looking at the page 165 in A7292-4EX-11 ( "IC-7300 Full Manual" ),
  // the already arrived at parameter number "0217" for command 0x1A sub-cmd 0x05.
  // But in A7508-3EX-4 ( "IC-9700 CI-V REFERENCE GUIDE" ), the list is
  // incredibly long, and ends at number "0350" for command 0x1A sub-cmd 0x05.
  // And of course, all those numbers are incompatible between rig models.
  // To "invert" the more frequently used RigCtrl_SubcodeFrom1A05_to_UnifiedParameterNumber(),
  // assume we NEVER support those "higher valued" 4-digit numbers of cmd 0x1A.05, so:
#define MAX_EXPECTED_PARAMETER_NUMBER_FOR_CMD_0x1A_sub0x05 350
  for( iSubSubcode = 1; iSubSubcode <= MAX_EXPECTED_PARAMETER_NUMBER_FOR_CMD_0x1A_sub0x05; ++iSubSubcode )
   { if( iUnifiedPN == RigCtrl_SubcodeFrom1A05_to_UnifiedParameterNumber( iDefaultAddress, iSubSubcode ) )
      { // Bingo -> Now the caller knows the parameter identified by iUnifiedPN
        //          can be accessed via command 0x1A subcode 0x05 [sub-]subcode <iSubcode> [4 BCD digits] !
        return iSubSubcode;
      }
   }

  // Arrived here ? Missing support for the combination of RIG MODEL (by "default CI-V address")
  //                                    and UNIFIED PARAMETER NUMBER !
  // Sooner or later, maybe, this MODEL-DEPENDENT MADNESS will be read
  //   from a file, similar as WFView's "rig files" (*.rig), but the
  //   token names with all those backslashes in them cannot be parsed
  //   through Borland's "TMemIniFile". Example (for an IC-705):
  // > Commands\80\Type=CIV Transceive
  // > Commands\80\String=\\x1a\\x05\\x01\\x12
  // > Commands\80\Min=0
  // > Commands\80\Max=1
  // > Commands\80\Bytes=0
  // > Commands\80\Command29=false
  // > Commands\80\GetCommand=true
  // > Commands\80\SetCommand=true
  // > Commands\80\Admin=false
  //            '--> seems to be just a "running number", or what ?
  //
  return 0;
} // end RigCtrl_GetCrazyModelDependingSubSubcodeForCmd1A_05()


//--------------------------------------------------------------------------
T_RigCtrl_ParamInfo * RigCtrl_GetInfoForUnifiedParameterNumber(int iUnifiedPN)
  // As with other function returning a POINTER, the caller must check
  // for a NULL-result, because returning NULL means "not found" .
{
  T_RigCtrl_ParamInfo *pPI = (T_RigCtrl_ParamInfo*)RigCtrl_ParameterInfo;

  while( pPI->iUnifiedPN != 0 )
   { if( pPI->iUnifiedPN == iUnifiedPN )
      { return pPI;
      }
     ++pPI;
   }

  return NULL;  // no luck ..  maybe
}

//--------------------------------------------------------------------------
T_RigCtrl_ParamInfo *RigCtrl_GetInfoForUnifiedParameterByName(const char **cpSrc)
  // Inspired by RigCtrl_GetInfoForUnifiedParameterNumber(), but searches
  // for the parameter NAME, not for the parameter NUMBER .
  // When successfull, the NAME is skipped in the source-pointer (*cpSrc),
  //                   and the returned pointer (into RigCtrl_ParameterInfo[])
  //                   is non-NULL.
{
  T_RigCtrl_ParamInfo *pPI = (T_RigCtrl_ParamInfo*)RigCtrl_ParameterInfo;

  while( pPI->iUnifiedPN != 0 )
   { if( pPI->pszToken != NULL )
      { if( SL_SkipToken( cpSrc, pPI->pszToken) )
         { return pPI;
         }
      }
     ++pPI;
   }
  return NULL;  // no luck ..

} // end RigCtrl_GetInfoForUnifiedParameterByName()


//--------------------------------------------------------------------------
T_RigCtrl_ParamInfo * RigCtrl_GetInfoForCIVCommandAndSubcode(
          DWORD dwCombinedCmdAndSubcode) // [in] "combined" CI-V command + "sub-command"
               //      and possibly even more (e.g. for 0x1A 0x05 ....)
               //       - see RigCtrl_GetCombinedCmdAndSubcodeForCIV() !
  // Does almost the same as RigCtrl_GetInfoForUnifiedParameterNumber(),
  //  but retrieves the parameter info for a certain "Icom CI-V command + sub-command".
  // As with other function returning a POINTER, the caller must check
  // for a NULL-result, because returning NULL means "not found" .
{
  T_RigCtrl_ParamInfo *pPI = (T_RigCtrl_ParamInfo*)RigCtrl_ParameterInfo;
  BYTE bCmd     = ( dwCombinedCmdAndSubcode >> 24 ) & 0xFF; // e.g. 0x05 when SETTING(!) the VFO frequency (*)
  BYTE bSubCmd  = ( dwCombinedCmdAndSubcode >> 16 ) & 0xFF; // 0xFF (not 0x00!) when UNUSED
  BYTE bSubSub1 = ( dwCombinedCmdAndSubcode >> 8  ) & 0xFF; // Icom doesn't have a name for this..
  BYTE bSubSub2 = ( dwCombinedCmdAndSubcode >> 0  ) & 0xFF; // it's the stuff after e.g. 0x1A 0x05
  BOOL fMatch;

       // (*) 0x05 for "Set VFO Frequency" was chosen to point out the many
       //     cruel details with CI-V : Unlike many other "commands", which
       //     use the same COMMAND BYTE to read or write something,
       //     Icom traditionally uses command 0x05 to "Set the operating frequency"
       //                         but command 0x03 to "Read the operating frequency".
       //     Our parameter-info-look-up-table, T_RigCtrl_ParamInfo RigCtrl_ParameterInfo[],
       //     only has an entry for command 0x03 (READ) but not 0x05 (WRITE),
       //     thus we need yet another ugly 'hard coded special case here'
       //     to avoid turning RigCtrl_ParameterInfo[] into a bloated monster
       //     (with separate entries for the READ- and WRITE-commands) :
  switch( dwCombinedCmdAndSubcode ) // Special case like the one explained above ?
   { case 0x05FFFFFFUL : // cmd = "SET the operating frequency" (no sub-cmd, etc) ?
          bCmd = 0x03; // search for the parameter-info for cmd 0x03 = "READ the operating frequency" instead !
          break;
     default:  // assume ANYTHING ELSE uses the same command-byte for READ AND WRITE !
          break; // (as usually with assumptions, they later turn into BUGS)
   } // end switch( dwCombinedCmdAndSubcode )

  while( pPI->iUnifiedPN != 0 )
   { if( (pPI->pbCIVReadCmd != NULL )  // parameter supported by Icom CI-V at all ?
      && (pPI->iCIVReadCmdLength > 0) )
      { if( bCmd == pPI->pbCIVReadCmd[0] ) // byte-sequence for RigCtrl_SendAndLogMessage() to READ
         { // the "command" (CI-V) matches, but what about the optional SUB-command ?
           fMatch = TRUE;
           // The length of a READ-COMMAND (aka "read request" from host to radio)
           // indicates the presence of an extra byte for what Icom calls "Sub cmd." :
           if( pPI->iCIVReadCmdLength > 1 )  // <- that's the length without preamble, addresses, data, and postamble
            { fMatch &= (bSubCmd == pPI->pbCIVReadCmd[1]);
            }
           if( (pPI->iCIVReadCmdLength > 2 ) && (bSubSub1!=0xFF) )
            { //                                           '--> 3rd byte 0xFF means "unused" / "don't care" (*)
              fMatch &= (bSubSub1 == pPI->pbCIVReadCmd[2]);
              // ,-------|______|
              // '--> e.g. 0x01 for the IC-9700's "SET > Network > Network Control"
              //           (0x1A 0x05 0145 ... we don't care if this is BCD or HEX)
              //             |    |   |  |
              //             |    |   \__|___ up to two extra bytes (four BCD digits?)
              //             |    |__________ the FIRST of up to THREE BYTES
              //             |                that Icom lists in column "Sub cmd."
              //             '--------------- This is what Icom lists as "Cmd." .
              //                 It's more or less the "most significant byte" .
              // (*) bSubSub1 and/or bSubSub2 may be ZERO as a "joker".
              //     For example, when reading the BAND STACKING REGISTERS,
              //     bCmd=0x1A, bSubCmd=0x01, bSubSub1=0="ignore the first array index",
              //                          and bSubSub2=0="ignore the second array index".
            }
           if( (pPI->iCIVReadCmdLength > 3 ) && (bSubSub2!=0xFF) )
            { //                                           '--> 4th byte 0xFF means "unused" / "don't care" (*)
              fMatch &= (bSubSub2 == pPI->pbCIVReadCmd[3]);
              // ,-------|______|
              // '--> e.g. 0x45 for the IC-9700's "SET > Network > Network Control"
              //           (0x1A 0x05 0145)
            }
           if( fMatch )
            { return pPI;
            }
         }    // matching FIRST 'Cmd' byte ?
      }      // end if( (pPI->pbCIVReadCmd != NULL ) && ... )
     ++pPI;  // check the next parameter in RigCtrl_ParameterInfo[]
   }
  return NULL;  // no luck ..
} // end RigCtrl_GetInfoForCIVCommandAndSubcode()


//--------------------------------------------------------------------------
const char *RigCtrl_UnifiedParameterNumberToString(int iUnifiedPN)
{
  static char sz11Unknown[12];
  T_RigCtrl_ParamInfo *pPI = RigCtrl_GetInfoForUnifiedParameterNumber(iUnifiedPN);
  if( pPI != NULL )
   { return pPI->pszToken;
   }

  // The following isn't thread-safe but shouldn't be necessary at all:
  sprintf( sz11Unknown, "PN[%d]",(int)iUnifiedPN);
  return   sz11Unknown;

} // end RigCtrl_UnifiedParameterNumberToString()


//---------------------------------------------------------------------------
int * RigCtrl_GetIntValuePtrForUnifiedParameterNumber( T_RigCtrlInstance *pRC, int iUnifiedPN )
{
  T_RigCtrl_ParamInfo *pPI = RigCtrl_GetInfoForUnifiedParameterNumber(iUnifiedPN);
  if( pPI != NULL )
   { if( (pPI->iByteOffsetIntoRigCtrlInstance > 0 )
      && (pPI->iByteOffsetIntoRigCtrlInstance < sizeof(T_RigCtrlInstance) ) )
      { if( pPI->iRigCtrlDataType == RIGCTRL_DT_INT )
         { return (int*)( (BYTE*)pRC + pPI->iByteOffsetIntoRigCtrlInstance );
         }
      }
   }
  return NULL;  // no luck .. also here, the caller MUST check for NULL-pointers !
} // end RigCtrl_GetIntValuePtrForUnifiedParameterNumber()


//---------------------------------------------------------------------------
double* RigCtrl_GetDblValuePtrForUnifiedParameterNumber( T_RigCtrlInstance *pRC, int iUnifiedPN )
{
  T_RigCtrl_ParamInfo *pPI = RigCtrl_GetInfoForUnifiedParameterNumber(iUnifiedPN);
  if( pPI != NULL )
   { if( (pPI->iByteOffsetIntoRigCtrlInstance > 0 )
      && (pPI->iByteOffsetIntoRigCtrlInstance < sizeof(T_RigCtrlInstance) ) )
      { if( pPI->iRigCtrlDataType == RIGCTRL_DT_DOUBLE )
         { return (double*)( (BYTE*)pRC + pPI->iByteOffsetIntoRigCtrlInstance );
         }
      }
   }
  return NULL;  // no luck .. also here, the caller MUST check for NULL-pointers !
} // end RigCtrl_GetDblValuePtrForUnifiedParameterNumber()


//---------------------------------------------------------------------------
BOOL RigCtrl_SetParamValue_Int( T_RigCtrlInstance *pRC, T_RigCtrl_ParamInfo *pPI, long i32Value )
  // Only stores the new value in the T_RigCtrlInstance,
  //       but does NOT automatically send a CI-V "Set"-command to the radio.
  //       See implementation of CwNet_ExecuteCmd() -> RigCtrl_OnParamReport_Integer() ...
  // Return value :  TRUE when "modified", FALSE = i32Value was still the same.
{
  BOOL fModified = FALSE;
  if( (pRC!=NULL) && (pPI!=NULL) )
   { if( (pPI->iByteOffsetIntoRigCtrlInstance > 0 )
      && (pPI->iByteOffsetIntoRigCtrlInstance < sizeof(T_RigCtrlInstance) ) )
      { switch( pPI->iRigCtrlDataType )
         { case RIGCTRL_DT_INT :
              fModified = (*(int*)( (BYTE*)pRC + pPI->iByteOffsetIntoRigCtrlInstance ) != i32Value);
              *(int*)( (BYTE*)pRC + pPI->iByteOffsetIntoRigCtrlInstance ) = i32Value;
              break;
           case RIGCTRL_DT_DOUBLE :
              fModified = (*(double*)( (BYTE*)pRC + pPI->iByteOffsetIntoRigCtrlInstance ) != (double)i32Value);
              *(double*)( (BYTE*)pRC + pPI->iByteOffsetIntoRigCtrlInstance ) = (double)i32Value;
              break;
           default:
              break;
         }
      }
     // Similar as in RigCtrl_GetParamByPN_Int(), A FEW(!) parmeters require special
     // treatment when *modified* (here: via RigCtrl_SetParamValue_Int() ),
     // regardless of FROM WHERE this function was called (e.g. ) :
     if( fModified )
      { switch( pPI->iUnifiedPN )
         { case RIGCTRL_PN_TRANSMITTING: // THE RIG ITSELF just switched from RX to TX or back..
              RigCtrl_OnTransmitFlagChange( pRC );
              break;
           default: // nothing special when setting any of the hundreds of other parameters
              break;
         } // end switch( pPI->iUnifiedPN )
      }   // end if( fModified )
   }     // end if( (pRC!=NULL) && (pPI!=NULL) )
  else
   { pPI = pPI; // <- set a breakpoint HERE to find callers daring to pass in a NULL pointer :)
   }


  return fModified;
} // end RigCtrl_SetParamValue_Int()

//---------------------------------------------------------------------------
BOOL RigCtrl_SetParamValue_Double( T_RigCtrlInstance *pRC, T_RigCtrl_ParamInfo *pPI, double dblValue )
  // Only stores the new value in the T_RigCtrlInstance,
  //       but does NOT automatically send a CI-V "Set"-command to the locally connected radio.
  //       See implementation of CwNet_ExecuteCmd() -> RigCtrl_OnParamReport_Double() ...
  // Return value :  TRUE when "modified", FALSE = i32Value was still the same.
{
  BOOL fModified = FALSE;
  if( (pRC!=NULL) && (pPI!=NULL) )
   { if( (pPI->iByteOffsetIntoRigCtrlInstance > 0 )
      && (pPI->iByteOffsetIntoRigCtrlInstance < sizeof(T_RigCtrlInstance) ) )
      { switch( pPI->iRigCtrlDataType )
         { case RIGCTRL_DT_INT :
              fModified = (*(int*)( (BYTE*)pRC + pPI->iByteOffsetIntoRigCtrlInstance ) != (int)dblValue);
              *(int*)( (BYTE*)pRC + pPI->iByteOffsetIntoRigCtrlInstance ) = (int)dblValue;
              break;
           case RIGCTRL_DT_DOUBLE :
              fModified = (*(double*)( (BYTE*)pRC + pPI->iByteOffsetIntoRigCtrlInstance ) != dblValue);
              *(double*)( (BYTE*)pRC + pPI->iByteOffsetIntoRigCtrlInstance ) = dblValue;
              break;
           default:
              break;
         }
      }
   }
  else
   { pPI = pPI; // <- set a breakpoint HERE to find callers daring to pass in a NULL pointer :)
   }
  return fModified;
} // end RigCtrl_SetParamValue_Double()

//---------------------------------------------------------------------------
long RigCtrl_GetParamValue_Int( T_RigCtrlInstance *pRC, T_RigCtrl_ParamInfo *pPI )
  // Related to RigCtrl_HaveValidParamValue(), but returns THE VALUE ITSELF,
  //         or RIGCTRL_NOVALUE_INT when there is nothing available.
{
  double dblValue;
  if( (pRC!=NULL) && (pPI!=NULL) )
   { if( (pPI->iByteOffsetIntoRigCtrlInstance > 0 )
      && (pPI->iByteOffsetIntoRigCtrlInstance < sizeof(T_RigCtrlInstance) ) )
      { switch( pPI->iRigCtrlDataType )
         { case RIGCTRL_DT_INT :
              return *(int*)( (BYTE*)pRC + pPI->iByteOffsetIntoRigCtrlInstance );
           case RIGCTRL_DT_DOUBLE :
              dblValue = *(double*)( (BYTE*)pRC + pPI->iByteOffsetIntoRigCtrlInstance );
              if( dblValue != RIGCTRL_NOVALUE_DOUBLE )
               { return (long)dblValue;
               }
           default:
              break;
         }
      }
   }
  return RIGCTRL_NOVALUE_INT; // arrived here ? Bad luck !
} // end RigCtrl_GetParamValue_Int()


//---------------------------------------------------------------------------
BOOL RigCtrl_HaveValidParamValue(T_RigCtrlInstance *pRC, T_RigCtrl_ParamInfo *pPI)
{
  int   iValue;
  double dblValue;
  if( pPI != NULL )
   { switch ( pPI->iRigCtrlDataType )
      { case RIGCTRL_DT_INT  :
           iValue = *(int*)( (BYTE*)pRC + pPI->iByteOffsetIntoRigCtrlInstance );
           return ( iValue!=RIGCTRL_NOVALUE_INT );
        case RIGCTRL_DT_DOUBLE:
           dblValue = *(double*)( (BYTE*)pRC + pPI->iByteOffsetIntoRigCtrlInstance );
           return ( dblValue!=RIGCTRL_NOVALUE_DOUBLE );
        default:  // got a parameter info, but data type not supported yet
           break;
      } // end switch ( pPI->iRigCtrlDataType )
   }
  return FALSE;   // "I don't have a valid value for this parameter yet"
                  //  -> the parameter-enumerating-loop may actively ASK for it,
                  //     or -if the parameter is not so important- keep it invalid.

} // end RigCtrl_HaveValidParamValue()


//---------------------------------------------------------------------------
BOOL RigCtrl_GetParamValueAsString(T_RigCtrlInstance *pRC, T_RigCtrl_ParamInfo *pPI, char *pszDest, int iMaxLen )
  // See also : RigCtrl_SetParamValueFromString() [inverse function / parser]
{
  int   iValue    = RIGCTRL_NOVALUE_INT;
  double dblValue = RIGCTRL_NOVALUE_DOUBLE;
  const char *pszEndstop = pszDest + iMaxLen - 1;
  const char *pszToken;
  T_RigCtrl_PortInstance *pRadioPortInstance = &pRC->PortInstance[RIGCTRL_PORT_RADIO];

  if( pPI != NULL )
   { switch ( pPI->iRigCtrlDataType )
      { case RIGCTRL_DT_INT  :
           iValue = *(int*)( (BYTE*)pRC + pPI->iByteOffsetIntoRigCtrlInstance );
           break;
        case RIGCTRL_DT_DOUBLE:
           dblValue = *(double*)( (BYTE*)pRC + pPI->iByteOffsetIntoRigCtrlInstance );
           break;
        default:
           break;
      } // end switch ( pPI->iRigCtrlDataType )
     if( iValue!=RIGCTRL_NOVALUE_INT )
      { // Got a parameter with a valid value -> look it up in the string table..
        if( pPI->pStringValueTable != NULL ) // .. IF there is a string table ..
         { if( (pszToken=SL_GetStringFromTokenList(pPI->pStringValueTable,iValue)) != NULL )
            { SL_AppendString( &pszDest, pszEndstop, pszToken );
              return TRUE;
            }
         }
        // Without a string table, convert into a decimal string ...
        //    ... unless it's one of the following, "very special" parameters:
        switch( pPI->iUnifiedPN )
         {
           case RIGCTRL_PN_TRANSCEIVER_ID :
              if( (pszToken=RigCtrl_DefaultAddressToString( pRadioPortInstance->iRadioCtrlProtocol, iValue )) != NULL )
               { SL_AppendPrintf( &pszDest, pszEndstop, "0x%02X (%s)",(int)iValue, pszToken );
                 return TRUE;
               }
              break;
           case RIGCTRL_PN_OP_MODE: // current 'operation mode' (USB/LSB/CW..) pRC->iOpMode, sometimes decorated with a "filter code" (e.g. for "CWN" or even "CWNN" instead of just "CW")
           case RIGCTRL_PN_UNSEL_VFO_OP_MODE: // .. same also for the UNSELECTED VFO ..
              // Since this parameter is a BIT COMBINATION,
              //  a code like iValue=8208 = 16 + 8192 = RIGCTRL_OPMODE_FM + RIGCTRL_OPMODE_NARROW
              //  may be ok for a machine, but not for a human, or a human-readable text file:
              SL_AppendString( &pszDest, pszEndstop, RigCtrl_OperatingModeToString(iValue) );
              // '--> the result, when listed on the Debug tab ("List Rig Control parameters")
              //      will be   :    PN003: OpMode = FMN
              //      instead of:    PN003: OpMode = 8208
              return TRUE;

           default:
              break;
         }
        SL_AppendPrintf( &pszDest, pszEndstop, "%d",(int)iValue );
        return TRUE;
      }
     else if( dblValue!=RIGCTRL_NOVALUE_DOUBLE )
      { SL_AppendPrintf( &pszDest, pszEndstop, "%.1lf",(double)dblValue );
        return TRUE;
      }
   }
  else
   { pPI = pPI; // <- set a breakpoint HERE to find callers daring to pass in a NULL pointer :)
   }

  return FALSE;
} // end RigCtrl_GetParamValueAsString()

//---------------------------------------------------------------------------
BOOL RigCtrl_SetParamValueFromString(T_RigCtrlInstance *pRC,
            T_RigCtrl_ParamInfo *pPI,
            const char **cppSrc )  // [in] new value, SKIPPED while parsing
  // Kind of inverse to RigCtrl_GetParamValueAsString() .
  // Returns TRUE only if the specified parameter was really MODIFIED.
  //    (if the caller really wants to send the value to the remotely controlled
  //     RADIO, use RigCtrl_QueueUpCmdToWriteUnifiedPN( pRC, pPI->iUnifiedPN ) )
  // All 'special treatments' implemented THERE should be parsable HERE.
  // First called from Keyer_GUI.cpp, after pressing ENTER on a line
  // like the following (in the 'Debug' tab):
  //     "PN040: AudioVolume = 16 %"
{
  long   i32Value;
  double dblValue;
  const char *pszToken;
  char c;
  BOOL fModified = FALSE;

  if( pPI != NULL )
   {
     SL_SkipSpaces( cppSrc ); // <- returns the NUMBER OF SPACES (not the first non-space character anymore)
     c = SL_PeekChar(cppSrc); // look ahead at the next character (without skipping it yet)
     if( (!SL_IsDigit(c)) && (c!='.') ) // obviously NOT A NUMERIC VALUE.. hmm... another goddamned SPECIAL CASE ?
      { switch( pPI->iUnifiedPN )
         {
           case RIGCTRL_PN_TRANSCEIVER_ID :
              break;
           case RIGCTRL_PN_OP_MODE: // current 'operation mode' (USB/LSB/CW..) pRC->iOpMode, sometimes decorated with a "filter code" (e.g. for "CWN" or even "CWNN" instead of just "CW")
           case RIGCTRL_PN_UNSEL_VFO_OP_MODE: // .. same also for the UNSELECTED VFO ..
              return RigCtrl_SetParamValue_Int( pRC, pPI, RigCtrl_StringToOperatingMode( cppSrc ) );
           default:
              break;
         }
        return FALSE;
      }
     switch ( pPI->iRigCtrlDataType )
      { case RIGCTRL_DT_INT  :
           i32Value = SL_ParseInteger( cppSrc );
           fModified = RigCtrl_SetParamValue_Int( pRC, pPI, i32Value );
           // ,--------|_______________________|
           // '--> Only stores the new value in the T_RigCtrlInstance,
           // but does NOT automatically send a CI-V "Set"-command to the radio.
           // Let THE CALLER decide what to do if fModified==TRUE, for example
           // call RigCtrl_QueueUpCmdToWriteUnifiedPN() to forward the modified
           // parameter to the real radio. The rest then happens LATER (as soon
           // as the radio control port isn't "busy" from other things)
           // in RigCtrl_SendWriteCommandForUnifiedPN() .
           break;
        case RIGCTRL_DT_DOUBLE:
           dblValue = SL_ParseFloat( cppSrc, '.'/*cDecimalSeparator*/ );
           fModified = RigCtrl_SetParamValue_Double( pRC, pPI, dblValue );
           break;
        default:
           break;
      } // end switch ( pPI->iRigCtrlDataType )
   }   // end if( pPI != NULL )
  else
   { pPI = pPI; // <- set a breakpoint HERE to find callers daring to pass in a NULL pointer :)
   }
  return fModified;
} // end RigCtrl_SetParamValueFromString()


//---------------------------------------------------------------------------
long RigCtrl_GetParamByPN_Int( T_RigCtrlInstance *pRC, int iUnifiedPN )
{
  T_RigCtrl_ParamInfo *pPI = RigCtrl_GetInfoForUnifiedParameterNumber(iUnifiedPN);
  long i32Result;
  if( (pRC!=NULL) && (pPI!=NULL) )
   { if( (pPI->iByteOffsetIntoRigCtrlInstance > 0 )
      && (pPI->iByteOffsetIntoRigCtrlInstance < sizeof(T_RigCtrlInstance) ) )
      { if( pPI->iRigCtrlDataType == RIGCTRL_DT_DOUBLE )
         { return (int)(*( (double*)( (BYTE*)pRC + pPI->iByteOffsetIntoRigCtrlInstance ) ) );
         }
        if( pPI->iRigCtrlDataType == RIGCTRL_DT_INT )
         { i32Result = *( (int*)( (BYTE*)pRC + pPI->iByteOffsetIntoRigCtrlInstance ) );
           // A few parameters need special treatment "on read" :
           switch( iUnifiedPN )
            { case RIGCTRL_PN_TRANSMITTING: // -> 0=receiving, 1=transmitting ..
                 // Don't rely on CI-V command "cmd 0x17, subcommand 0x00"
                 // because in older radios, and non-Icom radios, we cannot
                 // polling the PTT (aka 'transmit / receive state') via CAT at all.
                 if( (pRC->iTransmitReqst>0) || (i32Result==RIGCTRL_NOVALUE_INT ) )
                  { // pRC->iTransmitReqst indicates we ASKED THE RIG to TRANSMIT,
                    // or pRC->iTransmitting is not valid (not read/reported BY THE RADIO yet):
                    i32Result = pRC->iTransmitReqst;
                  }
                 break; // end case RIGCTRL_PN_TRANSMITTING (read-access for e.g. the Hamlib server)

              default: // no special treatment for this parameter ON READ :
                 break;
            }
           return i32Result;
         }
      }
   }
  else
   { pPI = pPI; // <- set a breakpoint HERE to find callers daring to pass in a NULL pointer :)
   }
  return 0;
} // end RigCtrl_GetParamByPN_Int()

//---------------------------------------------------------------------------
void RigCtrl_SetParamByPN_Int( T_RigCtrlInstance *pRC, int iUnifiedPN, long i32Value)
{ RigCtrl_SetParamValue_Int( pRC,
      RigCtrl_GetInfoForUnifiedParameterNumber(iUnifiedPN), i32Value );
}


//---------------------------------------------------------------------------
double RigCtrl_GetParamByPN_Double( T_RigCtrlInstance *pRC, int iUnifiedPN )
{
  T_RigCtrl_ParamInfo *pPI = RigCtrl_GetInfoForUnifiedParameterNumber(iUnifiedPN);
  if( (pRC!=NULL) && (pPI!=NULL) )
   { if( (pPI->iByteOffsetIntoRigCtrlInstance > 0 )
      && (pPI->iByteOffsetIntoRigCtrlInstance < sizeof(T_RigCtrlInstance) ) )
      { if( pPI->iRigCtrlDataType == RIGCTRL_DT_DOUBLE )
         { return *( (double*)( (BYTE*)pRC + pPI->iByteOffsetIntoRigCtrlInstance ) );
         }
        if( pPI->iRigCtrlDataType == RIGCTRL_DT_INT )
         { return (double)(*( (int*)( (BYTE*)pRC + pPI->iByteOffsetIntoRigCtrlInstance ) ) );
         }
      }
   }
  return 0.0;
} // end RigCtrl_GetParamByPN_Double()

//---------------------------------------------------------------------------
void RigCtrl_SetParamByPN_Double( T_RigCtrlInstance *pRC, int iUnifiedPN, double dblValue)
{ RigCtrl_SetParamValue_Double( pRC,
    RigCtrl_GetInfoForUnifiedParameterNumber(iUnifiedPN), dblValue );
}

//---------------------------------------------------------------------------
BOOL RigCtrl_AddParamToTxQueueForCwNet(T_RigCtrlInstance *pRC, int iUnifiedPN) // .. for transmission via CwNet.c !
  // Adds a new item to the queue for all 'Unified parameter numbers' that still
  // need to be sent from CLIENT to SERVER, e.g. using Remote_CW_Keyer/CwNet.c .
  //
  // [in] iUnifiedPN : e.g. RIGCTRL_PN_OP_MODE, after the operator (GUI user)
  //                   switched between CW, USB, LSB, or switched the bandwidth
  //                   between CW, CWN (narrow), CWNN (very narrow filter).
  // [out] pRC->iTxParameterQueue[ pRC->iTxParameterQueueHeadIndex++ ]
  //       ,---------------------------------------------'
  //       '--> circular FIFO index, runs from 0 to RIGCTRL_TX_PARAMETER_QUEUE_LENGTH-1
  // [in]  pRC->iTxParameterQueueTailIndex : Used by CwNet.c; incremented when
  //                   a new parameter (including parameter NUMBER and VALUE)
  //                   is on its way to the 'other side' via CwNet-TCP/IP-socket.
  //       As in a classic, lock free circular FIFO, ...
  //         * HeadIndex == TailIndex means the FIFO (here: "Queue") is EMPTY,
  //         * HeadIndex+1 (wrapped) == TailIndex means 'completely full' .
  // Details:
  //     If there is sufficient space in the outbound network buffer, CwNet_OnPoll()
  //     will compare the queue's HEAD index (iTxParameterQueueHeadIndex) with its own,
  //     "per-client" TAIL index, and if they are different, append another "SET xyz"-command,
  //     along with the CURRENT VALUE retrieved via 'unified parameter number' in this queue.
  // As an example for the flow of data between GUI, client, server, and the radio,
  //                            see RigCtrl_SetOperatingMode() .
{ int iNewHeadIndex = ( pRC->iTxParameterQueueHeadIndex+1 ) % RIGCTRL_TX_PARAMETER_QUEUE_LENGTH;
  if( iNewHeadIndex == pRC->iTxParameterQueueTailIndex )
   { return FALSE; // error : the queue with to-be-sent parameter numbers is already full !
   }

  // ToDo: Check if the to-be-sent 'unified parameter number' is already in the queue.
  //       If it is, don't append it to the queue, because CwNet_OnPoll()
  //       can only send THE NEWEST VALUE, e.g. pRC->iOpMode for RIGCTRL_PN_OP_MODE.

  pRC->iTxParameterQueue[pRC->iTxParameterQueueHeadIndex] = iUnifiedPN;

  pRC->iTxParameterQueueHeadIndex = iNewHeadIndex; // make the new item available to "readers in other threads"
       // '--> Compared against the queue's tail index in ANOTHER THREAD,
       //      see Remote_CW_Keyer/CwNet.c : CwNet_OnPoll() .
  return TRUE;

} // end RigCtrl_AddParamToTxQueueForCwNet()

//---------------------------------------------------------------------------
int RigCtrl_GetParamNrFromTxQueueForCwNet(T_RigCtrlInstance *pRC) // <- called from CwNet.c
  // Returns a "unified parameter number" when there are more parameters
  // in the T_RigCtrlInstance to send from ..
  //         (a) client to server
  //         (b) server to clients, except for the client who "asked" the server
  //             to modify something in the remotely controlled radio [avoid playing ping-pong]
  // Return value : RIGCTRL_PN_UNKNOWN (0) when none of the parameters in *pRC
  //                     needs to be sent from client to server (or vice versa),
  //             or RIGCTRL_PN_FREQUENCY, RIGCTRL_PN_OP_MODE, etc etc etc
  //                if one of the 'simple parameters' (with data type int or double)
  //                needs to be sent from THIS instance to others.
  //                When running as a server, CwNet.c keeps track of which remote client
  //                needs to be informed about a modified parameter value itself.
  //                See caller of RigCtrl_GetParamNrFromTxQueueForCwNet() for details.
{
  int iUnifiedPN = RIGCTRL_PN_UNKNOWN;

  if( pRC->iTxParameterQueueHeadIndex != pRC->iTxParameterQueueTailIndex ) // circular FIFO not empty ?
   { iUnifiedPN = pRC->iTxParameterQueue[pRC->iTxParameterQueueTailIndex];
     pRC->iTxParameterQueueTailIndex = (pRC->iTxParameterQueueTailIndex+1) % RIGCTRL_TX_PARAMETER_QUEUE_LENGTH;
   }
  return iUnifiedPN;
} // end RigCtrl_GetParamNrFromTxQueueForCwNet()


//---------------------------------------------------------------------------
int RigCtrl_EnumerateUnifiedParameters( int iPreviousUnifiedPN )
  // To retrieve a list of ALL "unified parameter numbers" (which are NOT contiguous),
  // begin with iPreviousUnifiedPN = 0.  Each new call will then provide
  // a new unified PN. When all PNs are through, returns iUnifiedPN = 0 .
{
  T_RigCtrl_ParamInfo *pPI = (T_RigCtrl_ParamInfo*)RigCtrl_ParameterInfo;
  if( iPreviousUnifiedPN <= 0 )
   {  return pPI->iUnifiedPN;
   }
  while( pPI->iUnifiedPN != 0 )
   { if( pPI->iUnifiedPN == iPreviousUnifiedPN )
      { return pPI[1].iUnifiedPN;
      }
     ++pPI;
   }
  // Arrived here ? Looks like all known "unified parameter numbers" are through.
  return 0;
} // end RigCtrl_EnumerateUnifiedParameters()


//--------------------------------------------------------------------------
DWORD RigCtrl_GetCombinedCmdAndSubcodeForCIV( BYTE *pbMessage, int iMsgLength, int iMsgType )
  // Format of the RETURN VALUE (32-bit unsigned integer, called "DWORD" in the good old days)
  //    Bit 31..24  |  23..16     |  15..0
  //   -------------+-------------+--------------------
  //        "Cmd."  | "Sub Cmd.", | "Sub Cmd.",   <-------------,
  //                |   1st byte  | up to FOUR extra digits     |
  //     ,------------------------------------------------------'
  //     '-> These are the names you will find in Icom' "CI-V REFERENCE GUIDE",
  //         "Command Table". See examples for an IC-9700 quoted from Icom's
  //         document number A7508-3EX-4 further below.
  //  2025-09-24: Older versions had the lower bits CLEARED when UNUSED.
  //              That was fine until "sub commands" with value 0x00 appearead,
  //              for example cmd 0x26 sub 0x00 = "Unselected VFO OpMode+DataMode+Filter".
  //         Solution: unused bits in the 32-bit "combined command and subcode"
  //         are SET ... hoping that 0xFF will never be used as a SUB-COMMAND
  //         by Icom, or in their ugly RIG-SPECIFIC four-digit "sub-subcodes"
  //         after CERTAIN commands in the 0x1A 0x05 - group.
{
  DWORD dwResult = ((long)pbMessage[4] << 24) | 0x00FFFFFF; // result if it's only a "Cmd" without "Sub cmd"
  DWORD dwFourDigitSubindex; // e.g. the stuff after 0x1A 0x05 (four-digit BCD)
  BOOL fIsReadCommand=FALSE;

  if( iMsgLength <= 5 ) // preamble(2), addr (2), postamble(1) : that's FIVE BYTES, no command in here
   { return RIGCTRL_NOVALUE_INT; // will never match anything in pPortInstance->dwExpectedResponse_CmdAndSubcode
   }
  if( iMsgLength == 6 ) // preamble(2), addr (2), command byte (1), postamble(1) but no SUBCODE..
   { return dwResult;
     // In this case, it's obvious that the command has no sub-command (subcode),
     //     e.g. cmd = 0x05 = "get operating frequency", without even knowing
     //     the MESSAGE TYPE (because the message is too short for being
     //     a WRITE COMMAND with one to many DATA BYTES).
   }
  else // iMsgLength > 6 : This is tricky. The byte after the command (pbMessage[4])
   { // may be a sub-command (subcode), but it MAY be a variable ranging from 0 to 255 or even more !
     // To find out if it's a TWO-BYTE command (with "Cmd" and "Sub cmd." in IC-7300 FM page 19-3),
     // branch by the first byte. Not bullet-proof yet, because what Icom decided
     // to call 'Sub cmd.' in the "Command table" of the IC-9700 CI-V manual,
     // isn't always ONE BYTE, but may be MULTIPLE BYTES. Examples from A7508-3EX-4:
     //   "Cmd."  |  "Sub cmd."  | "Data"       | "Description" / comment by DL4YHF
     // ----------+--------------+--------------+-----------------------------------
     //    0x04   |              | See p. 13    | Read the operating mode
     //    0x07   |   (none??)   |              | Select the VFO mode
     //    0x07   | 0x00         |              | Select VFO A (in satellite mode,...)
     //    0x07   | 0x01         |              | Select VFO B (not in satellite mode)
     //    0x07   | 0xA0         |              | Equalize VFO A and VFO B
     //    0x07   | 0xD0         |              | Select the main band
     //    0x07   | 0xD1         |              | Select the sub band
     //    0x1A   | 0x00         | See pp 14,15 | Send/read memory contents
     //    0x1A   | 0x01         | See p. 15    | Send/read band stacking registers
     //    0x1A   | 0x05 0001    | See p. 16    | SET > Tone Control/TBW > RX > ... HPF/LPF
     //    0x1A   | 0x05 0002    | 00  to 10    | SET > Tone Control/TBW > RX > ... Bass ..
     //    0x1A   | 0x05 0003    | 00  to 10    | SET > Tone Control/TBW > RX > ... Treble ..
     //    0x1A   | 0x05 ......  |              | Several HUNDREDS of other parameters
     //    0x1A   | 0x05 0350    | See p. 17    | SCOPE > Fixed Edges > 1200M > No.4
     //             \__________/
     // ,--------------'
     // '--> To find THE LENGTH of the "Sub cmd.", first look at the
     //      "Cmd." and the FIRST BYTE of the "Sub cmd." .
     //      Then, depending on that, decide what else belongs to the
     //      "Sub cmd." a la IC-9700 / document number A7508-3EX-4 .
     //   Knowing the MESSAGE TYPE (especially if it's a WRITE-COMMAND, "with data")
     //   allows telling a "Sub cmd." from a single data byte in a message, thus:
     fIsReadCommand = ( iMsgType & RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP ) == RIGCTRL_MSGTYPE_FLAG_READ_CMD;
     switch( pbMessage[4] ) // check for all known TWO-BYTE commands :
      { case 0x07 : // "Select" something (Subcmd:0=VFO A, 1=VFO B, 0xA0=Equalize A+B, 0xB0=Exchange A+B)
        case 0x0E : // Subcmd: 0=scan stop, 1=scan start, 2=prog scan, 3=F scan, .. 0xB1="set as select channel"
        case 0x10 : // Subcmd: 0=tuning step off, 1="send/read the 100 Hz tuning step" (?!?!)
        case 0x13 : // "Speech" with subcmds 0,1,2
        case 0x14 : // read/set various "Levels" (AF, RF, Squelch, "APF", Noise Reduction, etc^10)
        case 0x15 : // read S-meter, squelch status, power meter, SWR meter, ...
        case 0x16 : // preamp, AGC, noise blanker, noise reduction, ... (mostly ON/OFF switches, not levels)
        case 0x19 : // so far, only ONE subcode but Icom say it HAS a subcode
  //No: case 0x1A : // "Set/read" a horrible number of differently addressed items (WITH and WITHOUT sub-sub-commands)
        case 0x1B : // repeater tone frequency, TSQL tone frequency, ...
        case 0x1C : // transceiver status, ATU, XFC, ...
        case 0x1E : // number of TX frequency bands, TX frequency band edge frequencies, ...
        case 0x21 : // RIT frequency, RIT enable, XIT
        case 0x25 : // "selected" and "unselected" VFO freq. // main/sub VFO freq. (IC-7610)
        case 0x26 : // "selected" and "unselected" operating mode, "data mode"(?!), filter number
        case 0x27 : // scope waveform data, scope settings, level, .....
           dwResult = (dwResult & 0xFF00FFFFUL) | ( (DWORD)pbMessage[5] << 16);
           return dwResult;

        case 0x1A : // "Set/read" a horrible number of differently addressed items (with sub-sub-commands)
           // This horrible monster sometimes has a SINGLE-BYTE "Sub cmd.",
           // but very often (with the first sub-cmd-byte = 0x05)
           // followed by a FOUR-DIGIT "sub-index" (no name given by Icom)
           // as shown in the example quoted from A7508-3EX-4 further above.
           dwResult = (dwResult & 0xFF00FFFFUL) | ( (DWORD)pbMessage[5] << 16);
             // '--> Result e.g. 0x1A04FFFF, which is an "older" command
             //      without the dreadful RIG-SPECIFIC four-digit BCD(?)
             //      that follows after 0x1A 0x05 .. we care for THAT below.

           // Guesswork: 0x1A 0x00 to 0x1A 0x04 are "older" commands,
           // introduced before Icom realized they would run out of "Sub cmd."-values
           // .. thus we end up with the mess that SOME "0x1A"-commands
           //    have a single byte as sub-command, while others have THREE.
           if( ( pbMessage[5] >= 0x05 )
             &&( iMsgLength >= 9 ) ) // at preamble(2), addr (2), command byte (1),
            { // THREE-BYTE "Sub cmd." plus postamble(1) ... that's 2+2+1+3+1 = 9 bytes
              dwResult =  ( dwResult & 0xFFFF0000UL)
                        | ((DWORD)pbMessage[6] << 8)
                        | ((DWORD)pbMessage[7] << 0); // omg !
             // '--> Result e.g. 0x1A050350 = the IC-9700-specific "SCOPE > Fixed Edges > 1200M > No.4" .
            }
           return dwResult;

        case 0x06 : // definitely has NO subcode ("Sub cmd") - not even in an IC-9700 !
           return dwResult;
        default   : // Guess it's a single-byte command WITH DATA .. or do we know better ?
           if( fIsReadCommand ) // it's a READ-COMMAND (not a READ-RESPONSE, and not a WRITE-COMMAND) ->
            { // anything before the POSTAMBLE (at pbMessage[iMsgLength-1] )
              // is a sub-command, sub-sub-command, or sub-sub-sub-command !
              dwResult = ( dwResult & 0xFF00FFFFUL) | ( (DWORD)pbMessage[5] << 16);
                 // '-- .. because iMsgLenghth >= 7 ..
              if( iMsgLength >= 8 )
               {  dwResult = (dwResult & 0xFF00FFFFUL) | ( (unsigned long)pbMessage[6] << 8);
               }
              if( iMsgLength >= 9 )
               {  dwResult |= ( (DWORD)pbMessage[7] << 0);
               }
            } // end if < READ - COMMAND with an unknown FIRST COMMAND-BYTE >
           return dwResult;
      } // end switch( pbMessage[4] )  [ that's the FIRST COMMAND-BYTE in CI-V ]
   }   // end else < iMsgLength >= 7 >
}     // end RigCtrl_GetCombinedCmdAndSubcodeForCIV()

//--------------------------------------------------------------------------
BOOL RigCtrl_IsOkResponse( int iRadioCtrlProtocol, BYTE *pbMessage, int iMsgLength)
  // Returns TRUE if the message as been identified as an "OK"-response (e.g. CI-V: 0xFB)
{
  switch( iRadioCtrlProtocol )
   { case RIGCTRL_PROTOCOL_ICOM_CI_V:
        return (iMsgLength>=5) && (pbMessage[0]==0xFE) && (pbMessage[1]==0xFE) && (pbMessage[4]==0xFB);
     default:
        return FALSE;
   } // end switch( iRadioCtrlProtocol )
} // end RigCtrl_IsOkResponse()

//--------------------------------------------------------------------------
BOOL RigCtrl_IsNotOkResponse( int iRadioCtrlProtocol, BYTE *pbMessage, int iMsgLength)
  // Returns TRUE if the message as been identified as a "NOT OK"-response (e.g. CI-V: 0xFA)
{
  switch( iRadioCtrlProtocol )
   { case RIGCTRL_PROTOCOL_ICOM_CI_V:
        return (iMsgLength>=5) && (pbMessage[0]==0xFE) && (pbMessage[1]==0xFE) && (pbMessage[4]==0xFA);
     default:
        return FALSE;
   } // end switch( iRadioCtrlProtocol )
} // end RigCtrl_IsNotOkResponse()


//--------------------------------------------------------------------------
int RigCtrl_ParseCIV( T_RigCtrlInstance *pRC, // Beware.. this function turned into a MONSTER !
          int iRigCtrlPort,   // [in] RIGCTRL_PORT_RADIO / RIGCTRL_PORT_AUX_COM_1 / 2 / .. ?
              // '--> Index into pRC->PortInstance[ iRigCtrlPort ],
              // but also required for subtle differences in processing
              // for messages from the REAL RADIO (RIGCTRL_PORT_RADIO)
              // and messages from e.g. a SERIAL PORT TUNNEL or a VIRTUAL RIG port !
              //  [in]  pRC->PortInstance[iRigCtrlPort].fActAsServer :
              //          Indicates being called on the SERVER SIDE, using this state machine:
              //  [out] pRC->PortInstance[iRigCtrlPort].iServerState
              //   '---> Details in RigControl_CIV_Server.c,
              //         with a sketch of the interaction between
              //         the "Aux COM"-port's worker thread (LEFT SIDE)
              //         and the RADIO PORT's worker thread (RIGHT SIDE).
              //
          int iRigCtrlOrigin, // [in] e.g. RIGCTRL_ORIGIN_RADIO, RIGCTRL_ORIGIN_CONTROLLER, RIGCTRL_ORIGIN_COM_PORT_RX/TX
              //      (especially important in Listen-Only- and Server-Mode to indicate
              //       if -from the radio's point of view- the message
              //       was SENT or RECEIVED. This 'direction' also appears
              //       in the 'traffic log' - see RigCtrl_AddToTrafficLog() !
          BYTE *pbMessage, int iMsgLength ) // [in] message data, message length in bytes
  // Called upon COMPLETE reception of any CI-V message, via 'COM' (USB) or LAN/WLAN (UDP),
  //    including those messages received on CLIENT PORTS (see "Callers" further below).
  // ALSO CALLED ON RECEPTION OF CI-V MESSAGES ON ONE OF THE "Additional COM Ports" !
  //    Most COMMANDS(!) received on those 'extra' ports will be answered
  //    immediately (if we know the answer, to avoid flooding the radio with
  //    requests from all over the place). Others (especially those we don't
  //    understand) will be passed on from here to the "radio port",
  //    and the response will be relayed back to the originating port.
  //
  // Returns RIGCTRL_MSGTYPE_xyz, e.g. RIGCTRL_MSGTYPE_FREQUENCY_REPORT when successfull,
  //   otherwise RIGCTRL_MSGTYPE_UNKNOWN (i.e. "not recognized") .
  // In Spectrum Lab, anything that arrives HERE MAY be displayed usually
  //    with a CYAN background under 'View/Windows' / 'CAT traffic monitor'.
  //    Certain messages, like error responses, will have a different colour.
  //
  // Callers:
  //   (a) IO_RadioCommThdFunc( for any kind of "COM" port connected to the RADIO)
  //          -> RigCtrl_ProcessRxData(with iRigCtrlPort==RIGCTRL_PORT_RADIO) -> RigCtrl_ParseCIV()
  //   (b) IO_SerialPortDuplicatorThread( for "external clients" via "COM" port, e.g. via "com0com" )
  //          -> RigCtrl_ProcessRxData(with iRigCtrlPort==RIGCTRL_PORT_RADIO) -> RigCtrl_ParseCIV()
{
  int i, iSubcode, iOldSetting, iNewSetting, iRange, iEdge;
  DWORD dwCombinedCmdAndSubcode; // comparable with pPortInstance->dwExpectedResponse_CmdAndSubcode
  int  iUnifiedPN = RIGCTRL_PN_UNKNOWN;
  T_RigCtrl_ParamInfo *pPI=NULL; // "parameter info" from a lookup table ..
  T_RigCtrl_ParamInfo *pParamInfoFromExpectedUnifiedPN = NULL; // " " " after RigCtrl_AskForUnifiedPN()

  double dblConvFactor = 1.0;
  double dblConvOffset = 0.0;
  int iMsgType = RIGCTRL_MSGTYPE_UNKNOWN; // for SL's CAT traffic monitor, to selectively reject certain message types in the list
  BYTE b;
  long   i32Value = 0;
  double dblValue = 0.0;  // for parameter values that MAY not fit inside 32 bit integer (e.g. "Microwave frequencies in Hertz")
  double d, fact;         // for conversion from BCD to floating point
  int   iCIVDataType = CIV_DTYPE_BCD2; // <- added 2025-07 for RigCtrl_CIV_Server_OnReadCommand() to prepare a DUMMY RESPONSE
  T_RigCtrl_PortInstance *pPortInstance = &pRC->PortInstance[ iRigCtrlPort ];
  T_RigCtrl_RadioInfo    *pRadioInfo;
  T_RigCtrl_MsgFilter    msg_filter;
  T_RigCtrlFrequencyRange *pFreqRange;
  T_RigCtrlFreqMemEntry   *pMemEntry;
  char  *pszComment = pPortInstance->sz255CommentFromParser; // DESTINATION for the human-readable traffic log
  char  *pszCmtDest = pszComment; // incremented destination for string building functions
  char  *cpCmtEndstop = pszComment+255; // "endstop" for string building functions
  char  cGetOrSetIndicator = ':'; // separator between parameter name and value in the COMMENT.
         // Formerly always ':'. Since 2025-08 '=' to indicate a "READ RESPONSE" (e.g. SelVFOFreq=703730)
         //                      versus ':' to indicate a  "WRITE COMMAND"/"SET" (e.g. SelVFOFreq:703730)
  char  sz80Temp[84];
  const char *pszToken = "";
  const char *pszUnit  = NULL;
  char  *cpValueAsString = NULL;


  // ex: BOOL fShowInLog = (RigCtrl_TrafficMonitor.iDisplayOptions & RIGCTRL_TMON_DISPLAY_OPTION_ENABLE) != 0;
  BOOL fShowInLog = RigCtrl_IsTrafficMonitorEnabled( pPortInstance ); // <- initial flag (before "filtering") !
  BOOL fIsEcho = FALSE;
  BOOL fIsResponse = FALSE; // .. judging only from the 'CI-V' addresses, thus even an
                            //    unsolicited message from radio to master
                            //    would have fIsResponse *set* . Thus the next flag:
  BOOL fIsExpectedResponse=FALSE; // may be difficult to find out .. but we need it to tell e.g.
                      // a WRITE-COMMAND from a READ-RESPONSE for the optional 'CI-V Server'
  BOOL fIsUnsolicitedMsg = FALSE; // initial guess : it's a REPONSE, not an unsolicited REPORT (new VFO frequ, Spectrum scope data)
  BOOL fUnknown=TRUE; // special service to reject ALL KNOWN COMMANDS (and their "GET"-responses) in the traffic log.
                      // Some commands (like CI-V 0x26 with various subcommands) don't carry a SINGLE parameter,
                      // so even if T_RigCtrl_ParamInfo *pPI is NULL after the long case list below,
                      // if fUnknown has been CLEARED near the end of RigCtrl_ParseCIV(),
                      // the received command (or response) will NOT be added to the traffic log. Phew.

#ifdef __BORLANDC__  // "... is assigned a value that is never used".. shut up ..
  (void)cpValueAsString;
  (void)dblValue;
  (void)pParamInfoFromExpectedUnifiedPN;
#endif

  pszComment[0] = '\0';

  // The TWO FIRST BYTES (0xFE = "Preamble") have already been checked by the
  //  caller (e.g. IO_HandleRcvdDataFromRadio()), so no need to check again, and:
  iMsgType |= RIGCTRL_MSGTYPE_FLAG_CIV; // even if the exact type is still unknown, we know it's CI-V.
              // '--> This flag provides a proper "headline" in the traffic log.
  // Within RigCtrl_ParseCIV(), don't care for pbMessage[2] = transceiver's address (e.g. 0x48 = IC706)
  // and  for pbMessage[3] = controller's address (often 0xE0=RIGCTRL_CIV_MASTER_ADDR_MIN), EXCEPT FOR THIS:
  // If the rig is configured for "CI-V Echo" *ON* (alias "CI-V USB Echo Back" in an IC-7300)
  //  anything we send out (as command) returns as an echo, like this:
  //            pbMessage[0 1  2  3  4  5  6  7  8  9  10]
  // > 01:26:22.2 TX 006 FE FE 94 E0 03 FD                 (VFO freq) ; read VFO freq
  // > 01:26:23.2 RX 006 FE FE 94 E0 03 FD                 (VFO freq) ; echo
  // > 01:26:23.2 RX 00B FE FE E0 94 03 00 70 35 05 00 FD  (VFO freq) ; 5357000.0 Hz
  //   ,-------------'-' |___| |  |  |
  //   |          ,--------'   |  |  |
  //   '--length  '--preamble  |  |  |
  //  [2]: "to address" -------'  |  '--- "cmd" / dwExpectedResponse_CmdAndSubcode
  //  [3]: "from address" --------'
  // Initial guess for this message being a COMMAND or a RESPONSE:
  if( ( pbMessage[2] == pPortInstance->iRadioMasterAddr) // message sent *TO* THE MASTER, or...
    ||((pbMessage[2] >= RIGCTRL_CIV_MASTER_ADDR_MIN)     // message sent *TO* SOME OTHER MASTER
     &&(pbMessage[2] <= RIGCTRL_CIV_MASTER_ADDR_MAX))    // (one any of the 'Additional COM Ports'; WFView prefers 0xE1 as MASTER ADDRESS.. so accept a whole bunch)
    ||( pbMessage[3] == pPortInstance->iRadioDeviceAddr) // message sent *FROM* THE RADIO ?
    )
   { fIsResponse = TRUE; // in any of these 'address cases', this is likely to be a RESPONSE, not a COMMAND
     // Except for commands above 0xF0, it's most likely a READ RESPONSE:
     if( pbMessage[4] < 0xF0 )
      { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_RESP );
        cGetOrSetIndicator = '=';
      }
   }

  if( iRigCtrlPort != RIGCTRL_PORT_RADIO ) // "conditional breakpoint" for debugging the VIRTUAL RIG (emulation on 'Additional COM Ports') :
   { if( fIsResponse )  // conditional breakpoint for a RESPONSE on an 'Additional COM Port' :
      { iRigCtrlPort = iRigCtrlPort; // <- set a breakpoint HERE (for CI-V received from e.g. a REMOTE CLIENT like N1MM Logger+) :
      }
     else               // conditional breakpoint for a COMMAND on an 'Additional COM Port' :
      { iRigCtrlPort = iRigCtrlPort;
      }
      // 2025-09-05: New bug: only RESPONSES, but not COMMANDS, were displayed in the traffic log.
      //  Reason: RigCtrl_IsTrafficMonitorEnabled() had already returned fShowInLog = FALSE
      //          at this point (correct, because the "debug"-option wasn't specified
      //          for the 'Additional COM Port' in CwKeyer_Config.sAuxCom[1].sz40Params),
      //          so why was the RESPONSE ( "t2 00C FE FE E0 94 ..") displayed anyway ?
      //  -> Added similar places for "conditional breakpoints" in RigCtrl_SendAndLogMessage().
   } // end if( iRigCtrlPort != RIGCTRL_PORT_RADIO )


  //
  // Unfortunately, "HamLib" (and thus WSJT-X and FTDX) didn't work
  // without the bandwidth-wasting echo stupidity. HERE (in RigControl.c)
  // don't confuse the ECHO with a RESPONSE or even an EXPECTED RESPONSE :
  if( ( pbMessage[2] == pPortInstance->iRadioDeviceAddr )   // message sent *TO* THE RADIO ?
    &&( pbMessage[3] == pPortInstance->iRadioMasterAddr ) ) // message sent *FROM* THE MASTER ?
   { // Message NOT to/from the expected 'addresses', so this message is an ECHO:
     if( iRigCtrlPort == RIGCTRL_PORT_RADIO )
      { // That's the same as in our "TX"-message (see CAT traffic dump above), so..
        fIsEcho = TRUE;  // .. so it's an ECHO, not the real response message !
      }
     else // guess a 3rd party application (like RS-BA1) is talking to us.
      {   // Example : First message from RS-BA1 = FE FE 94 E0 18 01 FD (7 bytes)
      }
   }

  // Prepare a new filter that will later be passed to RigCtrl_RegisterCIVReponseForClient():
  msg_filter.iCmd = pbMessage[4];
  // per default, any further bytes in any CI-V message are IGNORED for filtering:
  msg_filter.iSubCmd = msg_filter.iDataB1 = msg_filter.iDataB2 = -1;
  // per default, let the filter (for the client) pass reponse "OK" and "NOT OK"
  //   (exceptions will be made in the monster case-list further below) :
  msg_filter.iMsgTypes = RIGCTRL_MSGTYPE_OK | RIGCTRL_MSGTYPE_NOT_OK;

  dwCombinedCmdAndSubcode = RigCtrl_GetCombinedCmdAndSubcodeForCIV( pbMessage, iMsgLength, iMsgType );
  if( dwCombinedCmdAndSubcode == 0x2601FFFF ) // breakpoint for a 'problematic' parameter:
   {  dwCombinedCmdAndSubcode = dwCombinedCmdAndSubcode;
     // e.g. 0x2601FFFF = "Read Unselected VFO OpMode / DataMode / Filter"
   }
  pPI = RigCtrl_GetInfoForCIVCommandAndSubcode( dwCombinedCmdAndSubcode );
  // '--> May be NULL if a response message (CI-V) does not contain all required info, like ...
  //  * no "Cmd." and "Sub cmd." (<- those are the column titles in Icom's
  //       Command tables, e.g. for the IC-9700 in document 'A7508-3EX-4' )
  //  * messages with pbMessage[4] == 0xFA = "No-Good" (Icom slang: "NG") aka "FAilure" aka "NotOK"
  //  * messages with pbMessage[4] == 0xFB = "OK" (success) aka "Fine Business" from the rig
  //
  //  For the above cases, T_RigCtrlInstance must remember what has been asked for,
  //   instead of extracting everything from the received message itself .
  if( pPI != NULL )
   { iUnifiedPN = pPI->iUnifiedPN;
     iCIVDataType = pPI->iCIVDataType;
     if( pPI->pszToken != NULL )
      { pszToken = pPI->pszToken; // -> e.g. "NumTxBands" for cmd=0x1E, subcmd=0x00
        fUnknown = FALSE;
      }
     pszUnit = pPI->pszUnit;
     if( pszUnit != NULL )       // optional physical unit, appended to the value later when NON-NULL
      { if( pszUnit[0] == '\0' )
         {  pszUnit = NULL;
         }
      }
     // Even if there is no 'special treatment' for this parameters in the
     // endless switch/case list (for Icom's "cmd" + "sub-command") further below,
     // try to identify the message as READ-COMMAND, READ-RESPONSE, WRITE-COMMAND, or WRITE-RESPONSE already:
     if( fIsEcho )  // it's an ECHO from what CI-V has been long ago : a BUS, not just a POINT-TO-POINT connection ->
      { iMsgType |= RIGCTRL_MSGTYPE_FLAG_ECHO;
      }
     else if( RigCtrl_IsOkResponse( RIGCTRL_PROTOCOL_ICOM_CI_V, pbMessage, iMsgLength ) )
      { // It's an "OK"-response a la CI-V (cmd=0xFB) -> must be the unspecific WRITE RESPONSE !
        RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_WRITE_RESP );
      }
     else if( RigCtrl_IsNotOkResponse( RIGCTRL_PROTOCOL_ICOM_CI_V, pbMessage, iMsgLength) )
      { // It's a "NOT OK"-response a la CI-V (cmd=0xFA) -> also a WRITE-RESPONSE, but indicating an ERROR ("bad value") !
        RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_WRITE_RESP );
        iMsgType |= RIGCTRL_MSGTYPE_FLAG_ERROR;
      }
     else if( fIsResponse ) // it's a REPONSE, but neither "OK" nor "NOT OK" ->
      { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_RESP );
      }
     else // None of the above -> it may be a READ-COMMAND (w/o data) or a WRITE-COMMAND (with data) ->
      { // That's a dilemma. At this point, we don't know how many message-bytes are COMMAND, SUBCOMMAND,
        //  4-digit "SUB-SUB-COMMAND" (after e.g. 0x1A 0x05), etc.
        //  But RigCtrl_GetCombinedCmdAndSubcodeForCIV() contains some wisdom
        //  about which commands are followed by ZERO, ONE, or up to THREE bytes:
        i = 1;   // initial guess: only ONE COMMAND-BYTE ...
        if( (dwCombinedCmdAndSubcode & 0x00FF0000) != 0x00FF0000 ) // look mum, it has a SUB-COMMAND-byte !
         { ++i;
         }
        if( (dwCombinedCmdAndSubcode & 0x0000FF00) != 0x0000FF00 ) // oh boy, it even has a SUB-SUB-COMMAND-byte !
         { ++i;
         }
        if( (dwCombinedCmdAndSubcode & 0x000000FF) != 0x000000FF ) // whow, the parameter is 'addressed' by YET ANOTHER sub-sub-sub-command-byte !
         { ++i; // (which is the case with SOME, but not ALL, commands beginning with cmd=0x1A, sub=0x05)
         }
        if( iMsgLength > ( 2/*preamble*/ + 2/*addresses*/ + i/*cmd/sub/subsub-cmd*/ + 1/*postamble*/ ) )
         { // CI-V COMMAND(!) seems to have DATA BYTES -> guess(*) it's a WRITE-COMMAND !
           RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_WRITE_CMD );
         }
        else // no data -> guess(*) it's a READ-COMMAND !
         { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_CMD );
         }
        // (*) The horrible switch/case list further below may correct iMsgType
        //     if the above 'guess', based on the MESSAGE LENGTH, is wrong !
      } // end else < READ- or WRITE-COMMAND (not a "response", and not an "echo") >
   } // end if < pPI != NULL >

  if( ( ! fIsEcho ) && (iRigCtrlPort==RIGCTRL_PORT_RADIO) )
   { if ( dwCombinedCmdAndSubcode == pPortInstance->dwExpectedResponse_CmdAndSubcode )
      { // This MAY BE (!) what RigCtrl_Handler() has been "waiting for", but:
        //  Don't confuse the ECHO with the EXPECTED RESPONSE :
        if( ! fIsEcho )  // only if this is NOT an "echo" :
         { fIsExpectedResponse = TRUE; // ok for simple cases without the need for a matching SUB-COMMAND (more later)
           // Example: read S-Meter level;
           //  Byte Index: [0]  [1]  [2]  [3] [4]  [5] [6] [7] [8]
           //  CI-V  :     |_pre__|  To  From cmd  sub  data?  FD=postamble
           //   request  = FE   FE   00   00  15   02   FD
           //   response = FE   FE   00   A2  15   02   00 02  FD
           // ,-------------------------|___|
           // '--> combined into dwExpectedResponse_CmdAndSubcode : 0x1502
         }
        // Note: May have to use dwCombinedCmdAndSubcode' AFTER the monster-case-list
        //       further below, as the NEW "expected response" when a command
        //       retrieved from any CLIENT PORT is forwarded to the radio !
      }
     else // dwCombinedCmdAndSubcode != pPortInstance->dwExpectedResponse_CmdAndSubcode ->
      { // For example, after sending 0x03 ("read operating frequency")
        // an IC-7300 answered with   0xFB ("OK", but without data).
      } // end else < dwCombinedCmdAndSubcode != pPortInstance->dwExpectedResponse_CmdAndSubcode >
        // In other cases, we may be waiting for the response after RigCtrl_AskForUnifiedPN() :
     if( pPortInstance->iExpectResponseForUnifiedPN > 0 ) // <- added 2024-07, more 'universal' replacement for dwExpectedResponse_CmdAndSubcode..
      { pParamInfoFromExpectedUnifiedPN = RigCtrl_GetInfoForUnifiedParameterNumber(
                               pPortInstance->iExpectResponseForUnifiedPN);
        if( pParamInfoFromExpectedUnifiedPN != NULL )
         { iCIVDataType = pParamInfoFromExpectedUnifiedPN->iCIVDataType;
           // If we're lucky, the READ RESPONSE begins with the same "Cmd" and possibly "Sub Cmd"
           // as the READ REQUEST. In that case, finish what RigCtrl_AskForUnifiedPN() had begun:
           if( (pParamInfoFromExpectedUnifiedPN->pbCIVReadCmd != NULL)
            && (pParamInfoFromExpectedUnifiedPN->iCIVReadCmdLength > 0) )
            { if( memcmp(pParamInfoFromExpectedUnifiedPN->pbCIVReadCmd,
                         pbMessage+4 /* "Cmd" and possibly "Sub Cmd" */,
                         pParamInfoFromExpectedUnifiedPN->iCIVReadCmdLength) == 0 )
               { // Bingo; the data that may be parsed further below
                 // are the 'expected response' for RigCtrl_AskForUnifiedPN() :
                 fIsExpectedResponse = TRUE;
               }
            }
         } // if( pParamInfoFromExpectedUnifiedPN != NULL )
      }   // end if < iExpectResponseForUnifiedPN .. >
     if( fIsResponse ) // response (and NOT an "echo") from THE REAL RADIO ?
      { // Store the REAL RADIO's CI-V address for later (e.g. for a "Virtual Radio" on an "Additional COM Port"):
        pPortInstance->iRealRadioAddress = pbMessage[3];  // here: store "FROM"-address in the RESPONSE on the RADIO CONTROL PORT
      }
   } // end if( ( ! fIsEcho ) && (iRigCtrlPort==RIGCTRL_PORT_RADIO) )
  if( ! fIsEcho ) // don't waste time trying to parse CI-V "echoes" !
   { switch( pbMessage[4] )  // [in] what Icom calls 'command'. But it may be a response..
      {

        case 0x05: // Icom CI-V code "frequency control" / "Set operating frequency"
        case 0x00: // Transfer operating freq. data aka "Send frequency data (transceive)"
                   //   (= unsolicited message, e.g. triggered by turning the VFO knob)
        case 0x03: // "Read operating frequency" (response on request, also cmd 0x03)
                   // 0x03 was included in N1MM's and WSJT-X's "polling cycle" via COM port,
                   //      and will be responded to WITHOUT forwarding that read-request to the RADIO PORT.
           fIsUnsolicitedMsg = (pbMessage[4] == 0x00);
           iUnifiedPN = RIGCTRL_PN_FREQUENCY;
           pPI = RigCtrl_GetInfoForUnifiedParameterNumber(iUnifiedPN);
           if( pPI != NULL )
            { pszToken = pPI->pszToken;
              iCIVDataType = pPI->iCIVDataType;
            }
           else
            { pszToken = "Freq";
              iCIVDataType = CIV_DTYPE_BCD10;
            }
           i = 5; // index into pbMessage[] with the first byte of the DATA FIELD, *if* there is a data field...
           if( iMsgLength >= 8 )  // 2 bytes preamble + 2 byte address + 1 byte command + at least a few digits + 1 byte end-of-message
            {
              if( fIsUnsolicitedMsg )
               { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_UNSOLICITED );
                 cGetOrSetIndicator = '=';
               }
              else if( fIsResponse ) // it's a RESPONSE, AND it has DATA FIELD, so this surely is a READ-RESPONSE :
               { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_RESP );
                 // (fIsResponse has been told from the CI-V "To"- and "From"-adddress)
                 cGetOrSetIndicator = '=';
               }
              else // !fIsResponse : // it's not a RESPONSE, but it has a DATA FIELD, so guess it's a WRITE-COMMAND :
               { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_WRITE_CMD );
                 // already set: cGetOrSetIndicator = ':';
               }
              d = 0.0; fact=1.0;
              while(i<(iMsgLength-1))
               { b = pbMessage[i++];
                 d += fact * (double)(b & 0x0F);
                 fact *= 10.0;
                 d += fact * (double)((b>>4) & 0x0F);
                 fact *= 10.0;
               }

              // Info from an IC-905 user (where the REMOTE WATERFALL DISPLAY didn't work) :
              // > I noted in the 905 CI-V programming manual the command 0x00 (Frequency)
              // > now is variable length with 2 extra bytes added when on 10GHz
              // > or higher for the 10's and 100's GHz digit places.
              //    -> Compared (a) C:\datasheets\Rig_Manuals\IC-905\IC-905_CI-V_Reference_Guide.pdf (anno 2023)
              //        against (b) C:\datasheets\Rig_Manuals\IC-705\IC-705_ENG_CI-V_V4b_2021_05.pdf ...
              //   (a) page 16 : 6 bytes with 12 BCD-digits, from 1-Hz to 100-GHz digit (the latter always zero)
              //   (b) page 17 : 5 bytes with 10 BCD-digits, from 1-Hz to 1-GHz digit (the latter always zero)
              SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "%s%c%.1lf Hz", pszToken, cGetOrSetIndicator, (double)d );
              // Since THIS CODE runs in a worker thread,
              // the rest (updating the GUI) happens somewhere else.
              //
              if( ! ( pRC->iPostponedSetMessages & (1<<RIGCTRL_MSGTYPE_SET_FREQUENCY) ) )
               { // Only accept the radio's reported value if we're not going to SEND a NEW frequency soon:
                 if( d != pRC->dblVfoFrequency )
                  { ++pRC->iFrequencyModifiedByRadio_cnt; // <- counter is polled in a GUI-thread somewhere.
                  }
                 pRC->dblVfoFrequency = d; // here: modified by CI-V commands 0x00, 0x03, or 0x05 .
               } // end if < no postponed SET-command for the frequency pending >
              else // the radio has reported its VFO frequency, but a POSTPONED(!)
               {   // request to set a NEW (different) frequency is still pending.
                 // In this case, do NOT modify pRC->dblVfoFrequency now.
                 // This may cause trouble for clients that will read the VFO
                 // frequency through other means (with CI-V, many roads lead to Rome).
               }
            }
           else  // too short for a READ RESPONSE or a WRITE COMMAND, so this is most likely a READ COMMAND (e.g. on a serial port tunnel):
            {
              // (this seemed to be the first message sent by WSJT-X's Hamlib-mimosa)
              if( fIsResponse )
               { // NO-NO: RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_WRITE_RESP );
                 // Icom CI-V does not support WRITE-RESPONSES indicating which command and sub-command they apply to !
                 // All they have is "FB" (Fine Business) or "FA" (Failure),
                 // dating back to the stoneage with 1200 bits/second. Sigh...
                 // .. so we really don't know what this "response without data" is:
                 RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGMASK_FLAGS_RW_UNKNOWN );
               }
              else // !fIsResponse ->
               { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_CMD );
               }
              SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "%s ?", pszToken );
            }
           break;  // end case 0x00, 0x03, 0x05

        case 0x01: // "Send mode data (transceive)" [which is Icom geek speak for an UNSOLICITED report with the operation mode + filter bandwidth]
        case 0x04: // "Read operating mode" (response after RigCtrl_AskForOperatingMode(?) )
                   //   (see also: case 0x26, similar purpose, different syntax)
                   // Cmd 0x04 is included in WSJT-X's "polling cycle",
                   //      but WSJT-X doesn't try to CHANGE the mode.
           RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_BASIC, RIGCTRL_MSGTYPE_OP_MODE ); // ex: iMsgType = RIGCTRL_MSGTYPE_OP_MODE;
           iUnifiedPN = RIGCTRL_PN_OP_MODE;
           pPI = RigCtrl_GetInfoForUnifiedParameterNumber(iUnifiedPN);
           if( pPI != NULL )
            { pszToken = pPI->pszToken;   // -> just "OpMode" for brevity
              iCIVDataType = pPI->iCIVDataType;
            }
           else
            { pszToken = "OpMode";
            }
           iNewSetting = pRC->iOpMode; // -> default, may be -2147483647 = 0x80000001 = RIGCTRL_NOVALUE_INT when not set before !
           // IC-7300, IC-7610 : "Operating Mode" with two data bytes.. IC-7300 used iMsgLength=8
           if( iMsgLength >= 7 )  // 2 bytes preamble + 2 byte address + 1 byte command + at least one 'mode' byte + 1 byte end-of-message
            { iNewSetting = RigCtrl_IcomToRigCtrlOpmode( pbMessage[5] );
              if( fIsResponse )  // .. or, with cmd=0x01, an UNSOLICITED REPORT (same "To"/"From"-addressing)
               { // RESPONSE *with* a DATA FIELD, so this must be a READ-RESPONSE ...
                 RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_RESP );
                 cGetOrSetIndicator = '='; // .. show with '=' between name and value
               }
              else
               { // not a RESPONSE / unsolicited report, but it has a DATA FIELD, so this must be a WRITE-COMMAND :
                 RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_WRITE_CMD );
               }
              SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "%s%c%s", pszToken, cGetOrSetIndicator,
                               RigCtrl_OperatingModeToString(iNewSetting) );
            }
           else // cmd 0x01 or 0x04 with LESS THAN 7 bytes in the message -> READ COMMAND ?
            { // As for many dozens of other CI-V messages, got to find out if this is
              // a READ-COMMAND, WRITE-COMMAND, or READ-RESPONSE (a WRITE-RESPONSE is out of question, because that's just "OK" = 0xFB).
              if( !fIsResponse )
               { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_CMD );
               }
              SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "%s ?", pszToken );
            }
           if( iMsgLength >= 8 )  // "Operating mode" byte followed by optional "Filter setting" ?
            { switch( pbMessage[6] )
               { case 1 : break;  // "FIL1" (normal filter bandwidth)
                 case 2 : iNewSetting |= RIGCTRL_OPMODE_NARROW;      // "FIL2" [IC7610CIV page 10]
                          break;
                 case 3 : iNewSetting |= RIGCTRL_OPMODE_VERY_NARROW; // "FIL3"
                          break;
                 default: break;
               }
              SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, ", filter=%d", pbMessage[6] );
            } // end if < radio also reported the current FILTER SETTING >
           // Only accept the radio's reported value if we're not going to SEND a NEW operating mode:
           if( ! ( pRC->iPostponedSetMessages & (1<<RIGCTRL_MSGTYPE_OP_MODE) ) )
            {
              if( iNewSetting != pRC->iOpMode )
               { pRC->iOpMode = iNewSetting; // .. including the "NARROW"/"VERY_NARROW"-flags
                 // Also here, inform module CwNet.c about the modified parameter.
                 // When running as SERVER, CwNet passes this on to remote clients:
                 RigCtrl_AddParamToTxQueueForCwNet( pRC, iUnifiedPN ); // here: "OpMode and Filter Bandwidth", modified by the radio itself
               }
            }
           break; // end case 0x01 = "Send mode data (transceive)", 0x04 = "Read operating mode"

        case 0x02: // "Read band edge frequencies"
           // Icom's "A7292-4EX-11" page "19-3" (=161) doesn't show a sub-command or parameter for the READ COMMAND,
           //   but it certainly has the "Edge Number" immediately after the COMMAND,
           //   So, to tell a READ COMMAND from a READ RESPONSE, don't rely on the length:
           if( fIsResponse )
            { // It's a response,  and it's CI-V, so it can only be .. :
              RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_RESP );
              cGetOrSetIndicator = '='; // .. and as in dozens of other cases, indicate the READ-OUT VALUE with a '=' between parameter name and value
              // (in this case, using the default SL_AppendPrintf( &pszCmtDest, .. )
              //  after this switch/case-monster )
            }
           else
            { // It's not a RESPONSE, and it's a READ-ONLY parameter, so this must be a READ-COMMAND:
              RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_CMD );
            }
           break;

        case 0x06: // "Set operating mode"  (possibly "transceived", funny name for an unsolicited report,
           // and Icom also used the confusing name "Operating mode selection for transceive" for this)
           // Like many "old commands", this is a SET-CONLY command (for READING they use a DIFFERENT command) !
           if( fIsResponse ) // It's a response,  and it's CI-V, so it can only be .. :
            { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_RESP );
              cGetOrSetIndicator = '=';
            }
           else
            { // It's not a RESPONSE, and it's a SET-ONLY-command-byte, so this must be a WRITE-COMMAND:
              RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_WRITE_CMD );
            }
           break;

        case 0x07: // "Select the VFO 'mode'". Sent by "RS-BA1" but also WSJT-X:
                   //                   '--> Misleading. This SELECTS THE VFO. Not the VFO's "mode" !
                   // pbMessage[0  1  2  3  4  5  6]  (with iMsgLength==7)
                   // > r1 007 FE FE 94 E0 07 00 FD
                   //                         |____ 00 = VFO A, 01 = VFO B, A0 = Equalize A and B, ...
           // [in,out]  pRC->iCurrVfoIndex, pRC->iPrevVfoIndex (to keep track of changes of the 'current VFO')
           // Note: If you expect to receive THIS message as an unsolicited report
           //       after pressing 'A/B' on the IC-7300's front panel,
           //       you will be bitterly disappointed.
           //       What you get instead (when pressing 'A/B' to swap VFOs)
           //       is cmd 0x01 = "Send mode data (transceive)" (! ! !) ...
           //       without any indication that VFO 'A' and 'B' have been swapped.
#         if(SWI_HARDCORE_DEBUGGING)
           if( (iRigCtrlPort >= RIGCTRL_PORT_AUX_COM_1) && (iRigCtrlPort <= RIGCTRL_PORT_AUX_COM_LAST) )
            { // See bug-hunting notes in C:\cbproj\Remote_CW_Keyer\Bugs\2025_08_16_N1MM and RCWK with winkeyer emulation.txt ..
              // in RigControl_RIV_Server.c, pServerPortInstance->iServerState was set to RIGCTRL_SSTATE_WAIT_RADIO_NBUSY (ok),
              // but after that, nothing happened. Grep for the following 'conditional breakpoint steps' (step number PLUS ONE):
              RigCtrl_iConditionalBreakpointStep = 5; // begin of trouble with forwarding of 'Set SelectVFO' from a "Virtual Rig" ?
              // 2025-10-19: Got here a few seconds after launching N1MM+ with pPI = NULL,
              //             iUnifiedPN = 0, fIsResponse = 0, iMsgLength = 7,
            }
#         endif // SWI_HARDCORE_DEBUGGING ?

           fUnknown = FALSE;
           if( fIsResponse ) // It's a response,  and it's CI-V, so it can only be .. :
            { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_RESP );
              cGetOrSetIndicator = '=';
            }
           if( iMsgLength >= 7 )  // "Select VFO mode" followed by at least one DATA byte ?
            {
              if( !fIsResponse )
               { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_WRITE_CMD );
                 // '--> 2025-08-01: This seems to be one of the first WRITE COMMANDS received from WSJT-X :
                 //    > r2 007 FE FE 94 E0 07 00 FD    ; SelVFO:00
                 //  Called way further below, RigCtrl_CIV_Server_OnWriteCommand() will care for this.
               } // end if( !fIsResponse )

              // On the REAL RADIO PORT, keep track of the CURRENTLY SELECTED VFO
              //  for reasons explained in RigCtrl_UpdateParamsAfterVfoSwitch():
              if( iRigCtrlPort==RIGCTRL_PORT_RADIO ) // only care for "VFO Switching" *ON THE RADIO* :
               {
                 switch( pbMessage[5] )
                  { case 0x00:  // "Select VFO A"
                       RigCtrl_OnVfoSwitch( pRC, 0/*iNewVfoIndex*/ );
                       break;
                    case 0x01:  // "Select VFO B"
                       RigCtrl_OnVfoSwitch( pRC, 1/*iNewVfoIndex*/ );
                       break;
                    case 0xA0:  // "Equalize VFO A and VFO B"
                       RigCtrl_OnVfoSwitch( pRC, VFO_SWITCH_EQUALIZE_A_B );
                       break;
                    case 0xB0:  // "Exchange VFO A and VFO B"
                       RigCtrl_OnVfoSwitch( pRC, VFO_SWITCH_EXCHANGE_A_B );
                       break;
                  } // end switch( pbMessage[5] ) for CI-V cmd 0x07 = "Select the VFO mode"
               } // if( iRigCtrlPort==RIGCTRL_PORT_RADIO )
              SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "SelectVFO%c%02X", cGetOrSetIndicator, (int)pbMessage[5] ); // for the log..
            }
           else // something WITHOUT DATA ? If it's not a RESPONSE, it's a READ-COMMAND, so:
            { SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "SelectedVFO ?" );
              if( !fIsResponse )
               { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_CMD );
               }
            }
           break; // end case 0x07 = "Select VFO" (not to be confused with the "Selected VFO mode" = "The currently selected VFO's operating mode")

        case 0x08: // "Select the Memory mode", with sub-commands like "Select the Memory channel". A7292-4EX-11 page "19-3" (=161)
           if( fIsResponse ) // It's a response,  and it's CI-V, so it can only be .. tada! .. a READ RESPONSE:
            { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_RESP );
              cGetOrSetIndicator = '=';
            } // If it's not a RESPONSE but a COMMMAND, finding the type (WRITE_CMD or READ_CMD) is trickier..
           if( iMsgLength >= 9 )  // "Select the Memory mode" (with two sub-cmd-bytes) followed by at least one DATA byte ?
            { if( !fIsResponse )  // if this isn't a RESPONSE, guess it's a WRITE-COMMAND:
               { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_WRITE_CMD );
               }
            }
           else // something WITHOUT DATA ? If it's not a RESPONSE, it's a READ-COMMAND, so:
            { if( !fIsResponse )  // if this isn't a RESPONSE, guess it's a READ-COMMAND:
               { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_CMD );
               }
            }
           break;

        case 0x09: // "Memory write". Not explained in A7292-4EX-11 page "19-3" (=161) at all !
           if( fIsResponse ) // It's a response,  and it's CI-V, so it can only be .. :
            { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_RESP );
              cGetOrSetIndicator = '=';
            }
           else // if this isn't a RESPONSE, guess it's a WRITE-COMMAND:
            { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_WRITE_CMD );
            }
           break;

        case 0x0A: // "Memory copy to VFO". Not explained in A7292-4EX-11 page "19-3" (=161) at all !
           if( fIsResponse ) // It's a response,  and it's CI-V, so it can only be .. :
            { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_RESP );
              cGetOrSetIndicator = '=';
            }
           else // if this isn't a RESPONSE, guess it's a WRITE-COMMAND:
            { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_WRITE_CMD );
            }
           break;

        case 0x0B: // "Memory clear". Not explained in A7292-4EX-11 page "19-3" (=161) at all !
           if( fIsResponse ) // It's a response,  and it's CI-V, so it can only be .. :
            { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_RESP );
              cGetOrSetIndicator = '=';
            }
           else // if this isn't a RESPONSE, guess it's a WRITE-COMMAND:
            { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_WRITE_CMD );
            }
           break;

        case 0x0E: // "Scan"-related commands. Used quite early when launching "RS-BA1":
           //      pbMessage[0  1  2  3  4  5  6]
           //      > r1 007 FE FE 94 E0 0E 00 FD (with iMsgLength=7)
           //           len                |
           //                              |_____ sub command (no data)
           //      pbMessage[0  1  2  3  4  5  6  7]
           //      > r1 008 FE FE 94 E0 0E 00 03 FD (with iMsgLength=8)
           //           len             |  |  |__ optional data, if iMsgLength > 7
           //                          cmd sub
           if( fIsResponse ) // It's a response,  and it's CI-V, so it can only be .. :
            { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_RESP );
              cGetOrSetIndicator = '=';
            }
           if( iMsgLength >= 7 )  // "Scan command" followed by at least one DATA byte ?
            { if( !fIsResponse )
               { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_WRITE_CMD );
               }
            }
           else // "Scan command" WITHOUT DATA ? If it's not a RESPONSE, it's a READ-COMMAND:
            { if( !fIsResponse )
               { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_CMD );
               }
            }
           if( iMsgLength >= 7 ) // valid length (with or without data) ?
            { iSubcode = pbMessage[5]; //
              switch( iSubcode )  // "sub-command" after 0x0E ?
               { case 0x00 : pszToken = "Scan stop";       // all for the log..
                    iUnifiedPN = RIGCTRL_PN_SCAN_MODE;
                    break;
                 case 0x01 : pszToken = "P/M scan start"; break;
                 case 0x02 : pszToken = "P scan start";   break;
                 case 0x03 : pszToken = "F scan start";   break;
                 case 0x12 : pszToken = "FP scan start";  break;
                    // etc, etc, a lot has been omitted here
                 default:
                    sprintf( sz80Temp, "Scan[%02X]", (int)iSubcode );
                    pszToken = sz80Temp;
                    break;
               } // end switch < "sub-command" after 0x0E, *WITHOUT* data >
              pPI = RigCtrl_GetInfoForUnifiedParameterNumber(iUnifiedPN);
              if( pPI != NULL )
               { iCIVDataType = pPI->iCIVDataType;
               }
              if( iMsgLength >= 8 )  // subcode followed by DATA ? Must be a "set"-command..
               { i32Value = RigCtrl_ParseBCD_CIV_MSByteFirst( pbMessage+6, 2*(iMsgLength-7) );
                 SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "%s%c%ld", pszToken, cGetOrSetIndicator, (long)i32Value ); // for the log..
               }
              else // only command and sub-command ..
               { SL_AppendString( &pszCmtDest, cpCmtEndstop, pszToken );
               }
            } // end if( iMsgLength >= ? )
           break; // end case 0x0E = "scan" commands

#  if(1)   // (1)=normal compilation, (0)=TEST to omit a few thousand lines, looking for imbalance curly braces (move THIS LINE up or down until finding the bugged 'case')

        case 0x0F: // Read/Write "Split setting"; Response data for IC-9700: 0x00="Split off", 0x01="Split", 0x11="DUP-", 0x12="DUP+", 0x13="DD Repeater Simplex (RPS)"
           // Received from RS-BA1 : > r1 006 FE FE 94 E0 0F FD      ; Split ?
           // Response from IC-7300: > RX 007 FE FE E0 94 0F 00 FD   ; Split:0
           //                        pbMessage[0  1  2  3  4  5  6]
           // Also included in WSJT-X's "polling cycle", and used by N1MM Logger+ .
           if( (iRigCtrlPort >= RIGCTRL_PORT_AUX_COM_1) && (iRigCtrlPort <= RIGCTRL_PORT_AUX_COM_LAST) )
            { iRigCtrlPort = iRigCtrlPort;  // <- place for a conditional breakpoint when received from a VIRTUAL RADIO port.
              // See bug-hunting notes in C:\cbproj\Remote_CW_Keyer\Bugs\2025_08_16_N1MM and RCWK with winkeyer emulation.txt ..
              // in RigControl_RIV_Server.c, pServerPortInstance->iServerState was set to RIGCTRL_SSTATE_WAIT_RADIO_NBUSY (ok),
              // but after that, nothing happened. Grep for the following 'conditional breakpoint steps' (sequence):
#            if(SWI_HARDCORE_DEBUGGING)
              RigCtrl_iConditionalBreakpointStep = 1;  // received a "Split Mode" command on an AUX COM PORT !
#            endif // SWI_HARDCORE_DEBUGGING ?
            }
           iUnifiedPN = RIGCTRL_PN_SPLIT_MODE;
           pPI = RigCtrl_GetInfoForUnifiedParameterNumber(iUnifiedPN);
           if( fIsResponse ) // It's a response,  and it's CI-V, so it can only be .. tada! .. a READ RESPONSE:
            { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_RESP );
              cGetOrSetIndicator = '=';
            } // If it's not a RESPONSE but a COMMMAND, but is is a WRITE_CMD or READ_CMD ?
           if( iMsgLength >= 6 )  // cmd 0x0F *WITH* data, but what is it.. WRITE-COMMAND, READ-COMMAND, or READ-RESPONSE ?
            { if( pPI != NULL )
               { pszToken = pPI->pszToken;
                 iCIVDataType = pPI->iCIVDataType;
                 fUnknown = FALSE;
               }
              else
               { pszToken = "Split";
               }
              if( iMsgLength >= 7 )  // 0x0F followed by DATA byte(s) -> must be a READ-RESPONSE or WRITE-COMMAND ..
               {
                 if( !fIsResponse ) // it's not a RESPONSE, but it has a DATA FIELD, so guess it's a WRITE-COMMAND :
                  { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_WRITE_CMD );
                  }

                 i32Value = RigCtrl_ParseBCD_CIV_MSByteFirst( pbMessage+5,
                                       2*(iMsgLength/*7*/-6)/*nDigits*/ );
                 // Replace the Icom-specific BCD-values by the 'unified'
                 // symbolic values in RigControl.h :
                 switch( i32Value )
                  { case 0: i32Value = RIGCTRL_SPLIT_MODE_OFF;
                               cpValueAsString = "off";
                               break;
                    case 1: i32Value = RIGCTRL_SPLIT_MODE_ON;
                               cpValueAsString = "ON";
                               break;
                    case 11: i32Value = RIGCTRL_SPLIT_MODE_DUP_NEG;
                               cpValueAsString = "DUP-";
                               break;
                    case 12: i32Value = RIGCTRL_SPLIT_MODE_DUP_POS;
                               cpValueAsString = "DUP+";
                               break;
                    case 13: i32Value = RIGCTRL_SPLIT_MODE_REPEATER_SIMPLEX; // <- IC-9700 extravaganza ?
                               cpValueAsString = "RPTR";
                               break;
                    default: // just leave i32Value as-is
                               cpValueAsString = "???";
                               break;
                  } // end switch < code for 'Split Mode' a la ICOM (IC-9700, and maybe others) >
                 SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "%s%c%s", pszToken, cGetOrSetIndicator, cpValueAsString );
                 if( RigCtrl_SetParamValue_Int( pRC, pPI, i32Value ) ) // <- stores the new value in pRC, but doesn't pass it on to anyone else
                  { //   '--> returns TRUE when "modified", so, also here:
                    // Inform module CwNet.c about the modified parameter.
                    // When running as SERVER, CwNet passes this on to remote clients:
                    RigCtrl_AddParamToTxQueueForCwNet( pRC, iUnifiedPN ); // here: new "Split Mode"
                  }
               } // end if( iMsgLength >= 7 )
              else  // cmd 0x0F without DATA ... guess it's a READ-COMMAND for the 'split mode' :
               { SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "%s ?", pszToken ); // query, no "set" command
                 if( !fIsResponse ) // it's not a RESPONSE, it has NO DATA FIELD, so guess it's a READ-COMMAND :
                  { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_CMD );
                  }
               }
            } // end if < sufficient length for COMMAND > ?
           break;  // end case 0x0F

        case 0x10: // "Send/read tuning step"
           if( fIsResponse ) // It's a response,  and it's CI-V, so it can only be .. tada! .. a READ RESPONSE:
            { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_RESP );
              cGetOrSetIndicator = '=';
            } // If it's not a RESPONSE but a COMMMAND, finding the type (WRITE_CMD or READ_CMD) is trickier..
           if( iMsgLength >= 6 )  // 0x10 with or without data.. (NO subcodes here)
            { pszToken = "TuningStep";
              iUnifiedPN = RIGCTRL_PN_TUNING_STEP; // one of the first parameters that WFVIEW(!) wants to know, immediately after connecting our server
              fUnknown = FALSE;
              pPI = RigCtrl_GetInfoForUnifiedParameterNumber(iUnifiedPN);
              if( pPI != NULL )
               { iCIVDataType = pPI->iCIVDataType;
               }
              if( iMsgLength >= 7 )  // 0x10 followed by DATA byte(s)..
               { // Example from RCW Keyer's traffic log:
                 // Byte[]:  0  1  2  3  4  5
                 // > TX 006 FE FE 94 E0 10 FD               ; read TuningStep
                 // > RX 007 FE FE E0 94 10 00 FD           ; TuningStep=10 Hz
                 // ,-----------------------'
                 // '--> Returning a FREQUENCY IN HERTZ would be too easy. Nnngrrrr.
                 if( !fIsResponse )  // command WITH DATA but NOT a response -> it's a WRITE-COMMAND:
                  { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_WRITE_CMD );
                  }
                 i32Value = RigCtrl_ParseBCD_CIV_MSByteFirst( pbMessage+5,
                                       2*(iMsgLength/*7*/-6)/*nDigits*/ );
                 // 0=off, 1=100 Hz, 2=1kHz, ... 8=25kHz .
                 // As for any other parameter, we don't want a manufacturer-
                 // and in this case even RIG-specific unit in pRC->iTuningStep_Hz !
                 // So convert into the one-and-only valid SI unit for frequencies : HERTZ !
                 i32Value = RigCtrl_CIV_TuningStepCodeToFrequency( pRC->iDefaultAddress, i32Value );
                 SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "%s%c%ld Hz", pszToken, cGetOrSetIndicator, (long)i32Value );
                 if( RigCtrl_SetParamValue_Int( pRC, pPI, i32Value ) ) // <- stores the new value in pRC, but doesn't pass it on to anyone else
                  { //   '--> returns TRUE when "modified", so, also here: Inform CwNet.c about the modified parameter.
                    RigCtrl_AddParamToTxQueueForCwNet( pRC, iUnifiedPN ); // here: new "Split Mode"
                  }
                 if( !fIsResponse ) // it's not a RESPONSE, but it has a DATA FIELD, so guess it's a WRITE-COMMAND :
                  { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_WRITE_CMD );
                  }
               }
              else  // 0x10 followed by subcode but NO DATA ...
               { SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "%s ?", pszToken ); // query, no "set" command
                 if( !fIsResponse )  // command WITHOUT DATA, and NOT a response -> it's a READ-COMMAND:
                  { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_CMD );
                  }
               }
            } // end if < sufficient length for COMMAND > ?
           break; // end case 0x10


        case 0x11: // "Send/read attenuator setting". One of dozens of parameters that "RS-BA1" wanted to know..
           RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_BASIC, RIGCTRL_MSGTYPE_RF_GAIN );
           if( fIsResponse ) // It's a response,  and it's CI-V, so it can only be .. tada! .. a READ RESPONSE:
            { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_RESP );
              cGetOrSetIndicator = '=';
            } // If it's not a RESPONSE but a COMMMAND, finding the type (WRITE_CMD or READ_CMD) is trickier..
           iUnifiedPN = RIGCTRL_PN_ATTENUATOR_DB;
           pPI = RigCtrl_GetInfoForUnifiedParameterNumber(iUnifiedPN);
           if( pPI != NULL )
            { fUnknown = FALSE;
              pszToken = pPI->pszToken;
              iCIVDataType = pPI->iCIVDataType;
              // From RS-BA1 to SpecLab (on a client port) :
              //           [0  1  2  3  4  5]
              //  > r1 006 FE FE 94 E0 11 FD
              // From an IC-7300 back to SpecLab (on the RADIO port):
              //           [0  1  2  3  4  5  6]
              //  > RX 007 FE FE E0 94 11 00 FD
              //                          \|___ data byte ("current setting")
              if( iMsgLength > 6 )  // 0x11 followed by at least ONE data byte..
               { i32Value = RigCtrl_ParseBCD_CIV_MSByteFirst( pbMessage+5,
                                       2*(iMsgLength/*7*/-6)/*nDigits*/ );
                 SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "%s%c%ld", pszToken, cGetOrSetIndicator, (long)i32Value );
                 if( RigCtrl_SetParamValue_Int( pRC, pPI, i32Value ) )
                  { //   '--> returns TRUE when "modified", so, also here:
                    // Inform module CwNet.c about the modified parameter.
                    // When running as SERVER, CwNet passes this on to remote clients:
                    RigCtrl_AddParamToTxQueueForCwNet( pRC, iUnifiedPN ); // here: "Attenuator setting"
                  }
                 if( !fIsResponse ) // it's not a RESPONSE, but it has a DATA FIELD, so guess it's a WRITE-COMMAND :
                  { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_WRITE_CMD );
                  }
               }
              else  // 0x11 followed by subcode but NO DATA ...
               { SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "%s ?", pszToken ); // just a query
                 if( !fIsResponse )  // command WITHOUT DATA, and NOT a response -> it's a READ-COMMAND:
                  { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_CMD );
                  }
               }
            } // end if( pPI != NULL )
           break; // end case 0x11 = "Send/read attenuator setting"

        case 0x12: // "Send/read antenna selection"
           if( fIsResponse ) // It's a response,  and it's CI-V, so it can only be .. tada! .. a READ RESPONSE:
            { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_RESP );
              cGetOrSetIndicator = '=';
            } // If it's not a RESPONSE but a COMMMAND, finding the type (WRITE_CMD or READ_CMD) is trickier..
           break; // end case 0x12

        case 0x13: // "Speech (speak) with voice synthesizer"
           RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_BASIC, RIGCTRL_MSGTYPE_OTHER ); // here: "Speak" (voice synth)
           if( fIsResponse ) // It's a response,  and it's CI-V, so it can only be .. tada! .. a READ RESPONSE:
            { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_RESP );
              cGetOrSetIndicator = '=';
            } // If it's not a RESPONSE but a COMMMAND, finding the type (WRITE_CMD or READ_CMD) is trickier..
           break;

        case 0x14: // "Send/read an awful lot of parameters" (AF level, RF gain, squelch, audio pitch, noise reduction, ....)
           if( fIsResponse ) // It's a response,  and it's CI-V, so it can only be .. tada! .. a READ RESPONSE:
            { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_RESP );
              cGetOrSetIndicator = '=';
            } // If it's not a RESPONSE but a COMMMAND, finding the type (WRITE_CMD or READ_CMD) is trickier..
           // Seen while RS-BA1 tortured the bus with lots of polling traffic :
           //  pbMessage[0  1  2  3  4  5  6  7  8]
           //  > r1 007 FE FE 94 E0 14 0B FD        (query, iMsgLength=7)
           //  > RX 009 FE FE E0 94 14 0B 01 53 FD  (response with data, iMsgLength=9)
           //                       |  |_sub-cmd
           //                      |__ cmd
           if( iMsgLength >= 7 )  // 0x14 followed by subcode, with or without DATA ...
            { iSubcode = pbMessage[5];
              dblConvFactor = 100.0 / 255.0;  // conversion factor for most "percentages", instead of the strange CI-V internal unit,
                  // but BEWARE: CI-V is incredibly counter intuitive !
                  // A few deterring examples:
                  //   * Cmd=0x14, SubCmd=0x01="AF level" : 0000=min=0 %, 0255=max=100 %
                  //   * Cmd=0x15, SubCmd=0x11="PO meter level": 0000=0%, 0143=50%, 0213=100% .. that's NON-LINEAR !
                  //   * Cmd=0x15, SubCmd=0x13=ALC meter level :0000=Min, 0120=Max !
                  //   * Cmd=0x15, SubCmd=0x14=Comp meter level:0000=0 dB, 0130=15 dB, 0241=30 dB.. that's NON-LINEAR !
                  //   * Cmd=0x15, SubCmd=0x15=Supply/PA Voltage: 00=0 V, 0013=10 V, 0241=16 V .. that's incredibly non-linear
                  //   * Cmd=0x15, SubCmd=0x16=Drain Current:   0000=0 A, 0097=10 A, 0146=15 A, 0241=25 A .. almost linear
                  //
              switch( iSubcode )  // decode "sub-command" after 0x14 for the log:
               { case 0x01 : iUnifiedPN = RIGCTRL_PN_AUDIO_VOLUME_PERCENT; break;
                 case 0x02 : iUnifiedPN = RIGCTRL_PN_RF_GAIN_PERCENT;      break;
                 case 0x03 : iUnifiedPN = RIGCTRL_PN_SQUELCH_LEVEL_PERCENT; break;
                 case 0x04 : break;
                 case 0x05 : break; // IC-7610 : "APF position", not used here
                 case 0x06 : iUnifiedPN = RIGCTRL_PN_NOISE_REDUCTION_PERCENT; break;
                 case 0x07 : iUnifiedPN = RIGCTRL_PN_PASSBAND_TUNING_POS1; break;
                 case 0x08 : iUnifiedPN = RIGCTRL_PN_PASSBAND_TUNING_POS2; break;
                 case 0x09 : iUnifiedPN = RIGCTRL_PN_CW_PITCH_HZ; // here: Convert from the CI-V internal unit into HERTZ:
                    // > 0000 = 300 Hz,  0128 = 600 Hz, 0255 = 900 Hz.
                    // A7380-7EX-2 page 3 tells a story about "5 Hz steps"
                    // but THAT'S NOT THE SCALING STEP. Expect more fun like this !
                    dblConvFactor = 600.0/*Hz*/ / 256.0/*steps*/;
                    dblConvOffset = 300.0/*Hz*/;
                    // See also: Inverse function in RigCtrl_SendWriteCommandForUnifiedPN() .
                    break;
                 case 0x0A : iUnifiedPN = RIGCTRL_PN_RF_POWER_SETTING_PERCENT; break;
                 case 0x0B : iUnifiedPN = RIGCTRL_PN_MIC_GAIN_PERCENT;     break;
                 case 0x0C : iUnifiedPN = RIGCTRL_PN_KEYER_SPEED_WPM;
                    // Scale from Icom's internal unit (0..255 ??) to WPM (Words Per Minute) :
                    // From A7380-7EX-2 page 3, about cmd 0x14 sub 0x0C :
                    // > Send/read keying speed (0000=6 wpm ~ 0255=48 wpm)
                    // As often, that was WRONG. When the keyer speed was
                    // set to "48 WPM" in an IC-7300, the raw value (in BCD)
                    // was 250, not 255 as advertised. Thank you Icom.
                    dblConvFactor = (48.0 - 6.0)/*WPM*/ / 250.0;
                    dblConvOffset = 6.0/*WPM*/;
                    break;
                 case 0x0D : iUnifiedPN = RIGCTRL_PN_NOTCH_POS_PERCENT;    break;
                 case 0x0E : iUnifiedPN = RIGCTRL_PN_COMP_SETTING_PERCENT; break;
                 case 0x0F : iUnifiedPN = RIGCTRL_PN_BREAK_IN_DELAY_PERCENT; break;
                 case 0x12 : iUnifiedPN = RIGCTRL_PN_NOISE_BLANKER_PERCENT;  break;
                 case 0x15 : iUnifiedPN = RIGCTRL_PN_MONITOR_GAIN_PERCENT; break;
                 case 0x16 : iUnifiedPN = RIGCTRL_PN_VOX_GAIN_PERCENT;     break;
                 case 0x17 : iUnifiedPN = RIGCTRL_PN_ANTI_VOX_GAIN_PERCENT; break;
                 case 0x18 : break;
                 case 0x19 : iUnifiedPN = RIGCTRL_PN_BRIGHTNESS_PERCENT;   break;
                 default   : break;
               } // end switch < "sub-command" after 0x14 >
              pPI = RigCtrl_GetInfoForUnifiedParameterNumber(iUnifiedPN);
              if( pPI != NULL )
               { fUnknown = FALSE;
                 iCIVDataType = pPI->iCIVDataType;
                 pszToken = pPI->pszToken;
                 pszUnit  = pPI->pszUnit;
                 if( pszUnit != NULL )
                  { if( pszUnit[0] == '\0' )
                     {  pszUnit = NULL;
                     }
                  }
               }
              else
               { sprintf(sz80Temp, "0x14[%02X]", (int)iSubcode );
                 pszToken = sz80Temp;
                 pszUnit = NULL;
               }
              if( iMsgLength >= 9 )  // 0x14+subcode followed by DATA bytes. Example:
               { // pbMessage[0  1  2  3  4  5  6  7  8]
                 // > RX 009 FE FE E0 94 15 12 00 80 FD ; SWRMeterLvl
                 //       |              |  |  |___|-- value in some phantasy unit,
                 //   iMsgLength=9      cmd subcode       usually 4 digit BCD, MSDigit first
                 i32Value = RigCtrl_ParseBCD_CIV_MSByteFirst( pbMessage+6,
                                       2*(iMsgLength/*9*/-7)/*nDigits*/ );
                 dblValue = (double)i32Value * dblConvFactor + dblConvOffset;
                 // Even with the conversion-factor and -offset, certain parameters
                 // need even more "special treatment". For example, the formula
                 // to convert the esoteric "CW Pitch" into an audio frequency in Hertz
                 // will deliver multiples of 2.34 Hz, but for the "display",
                 // the value must be rounded to the nearest multiple of 5 Hz:
                 switch( iUnifiedPN )
                  { case RIGCTRL_PN_CW_PITCH_HZ :
                       dblValue += 2.5; // round to the nearest multiple of 5 Hz
                       dblValue -= fmodl( dblValue, 5.0/*Hz*/ );
                       break;
                    default:
                       break;
                  } // end switch( iUnifiedPN )
                 SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "%s%c%ld", pszToken, cGetOrSetIndicator, (long)dblValue ); // "set" command with the already scaled value
                 if( pszUnit != NULL )
                  { SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, " %s", pszUnit );
                  }
                 if( RigCtrl_SetParamValue_Double( pRC, pPI, dblValue ) )
                  { //   '--> returns TRUE when "modified", so, also here:
                    // Inform module CwNet.c about the modified parameter.
                    // When running as SERVER, CwNet passes this on to remote clients:
                    RigCtrl_AddParamToTxQueueForCwNet( pRC, iUnifiedPN ); // here: "something reported as a sub-command for CI-V command 0x14"
                  }
                 if( !fIsResponse ) // it's not a RESPONSE, but it has a DATA FIELD, so guess it's a WRITE-COMMAND :
                  { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_WRITE_CMD );
                  }
               }
              else  // 0x15 followed by a subcode but NO DATA ...
               { SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "%s ?", pszToken ); // query, no "set" command
                 if( !fIsResponse )  // command WITHOUT DATA, and NOT a response -> it's a READ-COMMAND:
                  { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_CMD );
                  }
               }
            } // end if < sufficient length for COMMAND + SUB-COMMAND > ?
           break; // end case 0x14

        case 0x15: // "Send/read noise or S-meter squelch status", etc etc
           RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_BASIC, RIGCTRL_MSGTYPE_OTHER );
           iMsgType |= RIGCTRL_MSGTYPE_FLAG_PERIODIC_POLL;
           if( fIsResponse ) // It's a response,  and it's CI-V, so it can only be .. tada! .. a READ RESPONSE:
            { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_RESP );
              cGetOrSetIndicator = '=';
            } // If it's not a RESPONSE but a COMMMAND, finding the type (WRITE_CMD or READ_CMD) is trickier..
           //                                     pbMessage[0  1  2  3  4  5  6]
           // Seen when RS-BA1 took over control: > r1 007 FE FE 94 E0 15 01 FD
           //                                                           |  |_sub-cmd
           //                                                           |__ cmd
           // Note: The radio does NOT send any of these messages on its own
           //       - we must ASK FOR them, e.g. the current "S meter" value
           //         as Spectrum Lab periodically does (not very often) by
           //         calling RigCtrl_QueueUpCmdToReadUnifiedPN(RIGCTRL_PN_S_METER_LEVEL_DB)
           //         which, "on the next occasion", would send cmd 0x15 subcmd 0x02.
           iCIVDataType = CIV_DTYPE_BCD4;
           if( iMsgLength >= 7 )  // 0x15 followed by subcode, with or without DATA ...
            { iSubcode = pbMessage[5];
              switch( iSubcode )  // decode "sub-command" after 0x15 for the log:
               { case 0x01 :  // "Read noise or S-meter squelch status"  (IC-7300)
                    iUnifiedPN = RIGCTRL_PN_SQUELCH_STATUS;
                    break;
                 case 0x02 : // S-Meter:  0000="S0" (0 dB), 0120="S9" (9*6 dB), 0241="S9+60dB"
                    iUnifiedPN = RIGCTRL_PN_S_METER_LEVEL_DB;
                    break;
                 case 0x05 : // ex: pszToken = "SquelchStat"; no need to set anything here
                    // because there's an entry in RigCtrl_ParameterInfo[] for this subcode / unified PN:
                    iUnifiedPN = RIGCTRL_PN_SQUELCH_STATUS;
                    break;
                 case 0x07 : // ex: pszToken = "OverflowStat";  (WFView frequently polled for this, in an IC-7300)
                    iUnifiedPN = RIGCTRL_PN_ADC_OVERFLOW_STATUS;
                    break;
                 case 0x11 : // Cmd=0x15, SubCmd=0x11="PO meter level": 0000=0%, 0143=50%, 0213=100% .. that's NON-LINEAR,
                    // so use the special scaling formula further below.
                    iUnifiedPN = RIGCTRL_PN_POWER_METER_PERCENT;
                    break;
                 case 0x12 : iUnifiedPN = RIGCTRL_PN_SWR_METER_VALUE; // dimensionless "SWR" value, range 1.0 ... 3.0
                    break;
                 case 0x13 : iUnifiedPN = RIGCTRL_PN_ALC_METER_LEVEL;
                    // Cmd=0x15, SubCmd=0x13=ALC meter level :0000=Min, 0120=Max !
                    dblConvFactor = (100.0 / 120.0/* strange value for "120 = max", whatever that means*/ );
                    break;
                 case 0x14 : iUnifiedPN = RIGCTRL_PN_COMP_METER_LEVEL_DB;
                    // Cmd=0x15, SubCmd=0x14=Comp meter level:0000=0 dB, 0130=15 dB, 0241=30 dB.. that's NON-LINEAR !
                    // here: convert into decibel, 0=no compression, 30 dB = maximum reading for IC-7300)
                    dblConvFactor = (30.0 / 241.0/* strange value for "30 dB compression" */ );
                    break;
                 case 0x15 : iUnifiedPN = RIGCTRL_PN_SUPPLY_VOLTAGE_mV;
                    // Cmd=0x15, SubCmd=0x15=Supply/PA Voltage: 00=0 V, 0013=10 V, 0241=16 V .. that's incredibly non-linear
                    dblConvFactor = (16000.0/*mV*/ / 241.0/* strange value for "16 Volts" */ );
                    break;
                 case 0x16 : iUnifiedPN = RIGCTRL_PN_DRAIN_CURRENT_mA;
                    // Cmd=0x15, SubCmd=0x16=Drain Current:   0000=0 A, 0097=10 A, 0146=15 A, 0241=25 A .. almost linear
                    // here: convert into milliamps,  not the strange CI-V internal unit (241 = 25 A)
                    // As so often, this didn't work properly. Neither with an IC-7300 nor with an IC-9700.
                    // Also note the ever-changing interpretation of the the four-digit BCD(?) data:
                    // From  A7508-3EX-4 (IC-9700) page 5 :
                    // > Cmd. 15, Sub cmd. 16, Data 0000~0255
                    // >  Read Id meter level  ( 0000=0 A, 0121=10 A, 0241=20 A) .
                    // But when tested on 144 MHz, with "Power Set" = "50 %" -> circa 50 Watt RF,
                    // the bargraph on the IC-9700's "Meters" display showed I_drain = 13 Ampere,
                    // while RCW Keyer's CI-V traffic monitor showed the following;
                    // > TX 007 FE FE A2 E0 15 16 FD        ; read DrainCurrent
                    // > RX 009 FE FE E0 A2 15 16 00 26 FD  ; DrainCurrent=2697 mA
                    //                            |___|-- 4-digit BCD, ranging from 0000 to 0255 ? !?
                    // Tried the same again, now with "Power Set" = "0 %" -> less than 1 Watt RF,
                    // the bargraph on the IC-9700's "Meters" display showed I_drain = 3 Ampere,
                    // > TX 007 FE FE A2 E0 15 16 FD        ; read DrainCurrent
                    // > RX 009 FE FE E0 A2 15 16 00 27 FD  ; DrainCurrent=2800 mA
                    //                            |___|-- almost the same result as above !!?
                    // -> Forget it, it's not a simple "scaling error", it's complete rubbish.
                    // The "Vd" measurement works as advertised, but the "Id" measurement does not.
                    dblConvFactor = (25000.0/*mA*/ / 241.0/* strange value for "25 Ampere" */ );
                    break;
                 default   : sprintf(sz80Temp, "Meter[%02X]", (int)iSubcode );
                    // pszToken = sz80Temp;
                    break;
               } // end switch < "sub-command" after 0x15 >
              pPI = RigCtrl_GetInfoForUnifiedParameterNumber(iUnifiedPN);
              if( pPI != NULL )
               { fUnknown = FALSE;
                 iCIVDataType = pPI->iCIVDataType;
                 pszToken = pPI->pszToken;
                 pszUnit = pPI->pszUnit;
                 if( pszUnit != NULL )
                  { if( pszUnit[0] == '\0' )
                     {  pszUnit = NULL;
                     }
                  }
               }
              else
               { if( pszToken[0]==0 )
                  { sprintf(sz80Temp, "Meter[%02X]", (int)iSubcode );
                    pszToken = sz80Temp;
                  }
               }
              if( iMsgLength >= 8 )  // 0x15+subcode followed by DATA bytes. Examples:
               { // pbMessage[0  1  2  3  4  5  6  7  8]
                 // > RX 009 FE FE E0 94 15 12 00 80 FD ; SWRMeterLvl
                 //       |              |  |  |___|-- value in some phantasy unit,
                 //   iMsgLength=9      cmd subcode       usually 4 digit BCD, MSDigit first
                 // > RX 008 FE FE E1 93 15 07 00 FE    ; READ RESPONSE with OverflowStat=0 (WFView frequently polls this)
                 //                             '-----------------------------------------'
                 i32Value = RigCtrl_ParseBCD_CIV_MSByteFirst( pbMessage+6,
                                       2*(iMsgLength/*9*/-7)/*nDigits*/ );
                 d = (double)i32Value * dblConvFactor;
                 switch( iUnifiedPN ) // extravagant or non-linear conversion formula from RAW VALUE to PHYSICAL UNIT ?
                  {
                    case RIGCTRL_PN_S_METER_LEVEL_DB :  // S-Meter a la IC-7300
                       // > 0000="S0" (0 dB), 0120="S9" (9*6 dB), 0241="S9+60dB"
                       if( i32Value <= 120 ) // lower range.. "S0" (0 dB) to "S9" (54 dB)
                        { d = (double)i32Value * ( 54.0 / (120-0) );
                        }
                       else // higher range.. "S9" (54 dB) to "S9 + 60 dB" = 114 dB
                        { d = 54.0/*dB*/ + (double)(i32Value-120) * ( (114.0-54.0) / (double)(241-120) );
                        }
                       break; // end case RIGCTRL_PN_S_METER_LEVEL_DB

                    case RIGCTRL_PN_POWER_METER_PERCENT: // "PO meter level" a la IC-7300 : not in Watt but "percent of the maximum"
                       // response for Cmd=0x15, SubCmd=0x11=: 0000=0%, 0143=50%, 0213=100% ..
                       // that's NON-LINEAR, thus scale in several sections:
                       if( i32Value <= 143 ) // 1st range.. measured power level = 0 to 50 percent :
                        { d = 0.0 + (double)(i32Value-0 ) * ( (50.0-0.0) / (143-0) );
                          // With an IC-7300, output power set to "5 %" = 5 Watt for QRP in CW,
                          // got here with a MEASURED i32Value =  16 ('raw' value).
                          // Scaled result: 5.59 %  = 5.6 Watt . Ok, close enough.
                          //
                        }
                       else // 2nd range.. measured power level = 50 to 100 percent :
                        { d = 50.0 + (double)(i32Value-143) * ( (100.0-50.0) / (213-143) );
                        }
                       break; // end case RIGCTRL_PN_POWER_METER_PERCENT

                    case RIGCTRL_PN_SWR_METER_VALUE: // SWR meter reading a la IC-7300 (typically "1.0" .. "3.0")
                       // > 0000=SWR1.0, 0048=SWR1.5, 0080=SWR2.0, 0120=SWR3.0
                       // That's a non-linear relation, thus the special treatment:
                       if( i32Value <= 48 ) // 1st range.. SWR = 1.0 to 1.5 :
                        { d = 1.0 + (double)(i32Value-0 ) * ( (1.5-1.0) / (48-0) );
                        }
                       else if( i32Value <= 80 ) // 2nd range.. SWR = 1.5 to 2.0 :
                        { d = 1.5 + (double)(i32Value-48) * ( (2.0-1.5) / (80-48) );
                        }
                       else // 3rd range.. SWR = 2.0 to 3.0,  maybe even more :
                        { d = 2.0 + (double)(i32Value-80) * ( (3.0-2.0) / (120-80) );
                        }
                       break; // end case RIGCTRL_PN_SWR_METER_VALUE

                    case RIGCTRL_PN_COMP_METER_LEVEL_DB: // COMPRESSION meter : neither linear nor logarithmic :
                       // [in] i32Value (RAW)   : 0000    0130   0241
                       // [out] d= value in dB  : 0 dB    15 dB  30 dB
                       if( i32Value <= 130 ) // lower range..
                        { d = (double)i32Value * ( (15.0-0.0) / (double)(130-0) );
                        }
                       else // i32Value > 13  ->  "Comp meter" range 15 .. 30 dB:
                        { d = 15.0/*dB*/ + (double)(i32Value-130) * ( (30.0-15.0) / (double)(241-130) );
                        }
                       break; // end case RIGCTRL_PN_COMP_METER_LEVEL_DB

                    case RIGCTRL_PN_SUPPLY_VOLTAGE_mV: // very non-linear ..
                       // [in] i32Value (RAW)   : 0000    0013   0241
                       // [out] d=PHYSICAL value: 0 Volt  10 V   16 V
                       //                         |       |_________|--> Scaling Factor = 1000*(16-10) / (241-13)
                       //                         |________|--> Scaling Factor = 1000*(10-0) / (13-0)
                       if( i32Value <= 13 ) // lower range..
                        { d = (double)i32Value * ( 1000.0 * (10.0-0.0) / (double)(13-0) );
                        }
                       else // i32Value > 13  ->  Vd range 10 V .. 16 V :
                        { d = 10.0e3/*mV!*/ + (double)(i32Value-13) * ( 1000.0 * (16.0-10.0) / (double)(241-13) );
                          // typically seen here: i32Value = 166 -> d = 14026 [mV],
                          // when the voltage on an IC-7300's DC power cord was 13.94 V .
                        }
                       break; // end case RIGCTRL_PN_SUPPLY_VOLTAGE_mV
                    default: // nothing special -> leave 'd' (physical value) as-is
                       break;
                  }
                 SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "%s%c%ld", pszToken, cGetOrSetIndicator, (long)d ); // "set" command with the user-friendly SCALED value
                 if( pszUnit != NULL )
                  { SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, " %s", pszUnit );
                  }
                 if( RigCtrl_SetParamValue_Double( pRC, pPI, d ) )
                  { //   '--> returns TRUE when "modified", so, also here:
                    // Inform module CwNet.c about the modified parameter.
                    // When running as SERVER, CwNet passes this on to remote clients:
                    RigCtrl_AddParamToTxQueueForCwNet( pRC, iUnifiedPN ); // here: "something reported as a sub-command for CI-V command 0x14"
                  }
                 if( !fIsResponse ) // it's not a RESPONSE, but it has a DATA FIELD, so guess it's a WRITE-COMMAND :
                  { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_WRITE_CMD );
                  }
               }
              else  // 0x15 followed by subcode but NO DATA ...
               { SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "%s ?", pszToken ); // query, no "set" command
                 if( !fIsResponse )  // command and subcode but WITHOUT DATA, and NOT a response -> it's a READ-COMMAND:
                  { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_CMD );
                    // Example: when WFView polled the "Overflow Status" on iRigCtrlPort = 2,
                    //          got HERE with iMsgLength = 7, and
                    // pbMessage[0  1  2  3  4  5  6]  ; READ (ADC-)Overflow Status
                    //          FE FE 94 E1 15 07 FD
                    //  "TO" IC-7300---'  |  |  |  '-- postamble
                    //  "FROM" WFView ----' cmd sub (NO DATA, so this is a READ COMMAND, not a REPONSE)
                    // Because this parameter (RIGCTRL_PN_ADC_OVERFLOW_STATUS) had not been read from the REAL RADIO,
                    // the 'Virtual Radio' (emulator in RigControl_CIV_Server.c)
                    // had to forward it to the 'Real Radio', when calling
                    // RigCtrl_CIV_Server_OnReadCommand() further below.
                  }
               }
            } // end if < sufficient length for COMMAND + SUB-COMMAND > ?
           break; // end case 0x15 = "Send/read noise or S-meter squelch status", etc etc

        case 0x16: // Preamp, AGC speed, Noise blanker, Audio peak filter, Noise reduction, auto notch, etc etc
           if( fIsResponse ) // It's a response,  and it's CI-V, so it can only be .. tada! .. a READ RESPONSE:
            { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_RESP );
              cGetOrSetIndicator = '=';
            } // If it's not a RESPONSE but a COMMMAND, finding the type (WRITE_CMD or READ_CMD) is trickier..
           // Seen when RS-BA1 took over control: > r1 007 FE FE 94 E0 16 43 FD
           //                           but also: > r1 008 FE FE 94 E0 16 46 00 FD
           if( iMsgLength >= 7 )  // 0x1B followed by subcode, with or without DATA ...
            { iSubcode = pbMessage[5];
              switch( iSubcode )  // decode "sub-command" after 0x16 for the log.
               { // We're lucky because the IC-9700 only ADDS a more sub commands
                 // to those already defined for the IC-7300, but it doesn't
                 // GIVE A DIFFERENT MEANING to already existing sub-commands.
                 // DON'T TAKE THIS FOR GRANTED - thing are different for other
                 // CI-V commands, where depending on the radio model,
                 // other COMMANDS (and their sub-sub-commands) have
                 // totally different meanings, depending on the radio model !
                 case 0x02 : iUnifiedPN = RIGCTRL_PN_PREAMP_SETTING;     break;
                 case 0x12 : iUnifiedPN = RIGCTRL_PN_AGC_SPEED;          break;
                 case 0x22 : iUnifiedPN = RIGCTRL_PN_NOISE_BLANKER_ON;   break;
                 case 0x40 : iUnifiedPN = RIGCTRL_PN_NOISE_REDUCTION_ON; break;
                 case 0x41 : iUnifiedPN = RIGCTRL_PN_AUTO_NOTCH_ON;      break;
                 case 0x42 : iUnifiedPN = RIGCTRL_PN_RPTR_TONE_ON;       break;
                 case 0x43 : iUnifiedPN = RIGCTRL_PN_TONE_SQUELCH_ON;    break;
                 case 0x44 : iUnifiedPN = RIGCTRL_PN_COMP_ON;            break;
                 case 0x45 : iUnifiedPN = RIGCTRL_PN_MONITOR_ON;         break;
                 case 0x46 : iUnifiedPN = RIGCTRL_PN_VOX_ON;             break;
                 case 0x47 : iUnifiedPN = RIGCTRL_PN_BK_IN_MODE;         break;
                 case 0x48 : iUnifiedPN = RIGCTRL_PN_MANUAL_NOTCH_ON;    break;
                 case 0x4F : iUnifiedPN = RIGCTRL_PN_TWIN_PEAK_FILTER_ON; break;
                 case 0x50 : iUnifiedPN = RIGCTRL_PN_DIAL_LOCK_ON;       break;
                 case 0x56 : iUnifiedPN = RIGCTRL_PN_DSP_SHARP_SOFT;     break;
                 case 0x57 : iUnifiedPN = RIGCTRL_PN_MANUAL_NOTCH_WIDTH; break;
                 case 0x58 : iUnifiedPN = RIGCTRL_PN_SSB_TX_BANDWIDTH;   break;
                 case 0x59 : iUnifiedPN = RIGCTRL_PN_SUBBAND_ON;         break;
                 case 0x5A : iUnifiedPN = RIGCTRL_PN_SATELLITE_MODE;     break;
                 case 0x65 : iUnifiedPN = RIGCTRL_PN_IMPROVED_IP3;       break; // aka "IP+", aka "improved IP3", maybe just another "gain stage off", or ADC dithering ON. We don't care.
                 default   : break;
               } // end switch < "sub-command" after 0x16 >
              pPI = RigCtrl_GetInfoForUnifiedParameterNumber(iUnifiedPN);
              if( pPI != NULL )
               { // '-- this is also important way further below, to reject
                 //     certain classes of messages from the message log .
                 fUnknown=FALSE;
                 iCIVDataType = pPI->iCIVDataType;
                 pszToken = pPI->pszToken;
                 pszUnit = pPI->pszUnit;
                 if( pszUnit != NULL )
                  { if( pszUnit[0] == '\0' )
                     {  pszUnit = NULL;
#                      ifdef __BORLANDC__  // "... is assigned a value that is never used".. shut up ..
                        (void)pszUnit;
#                      endif
                     }
                  }
               }
              else
               { sprintf(sz80Temp, "0x16[%02X]", (int)iSubcode );
                 pszToken = sz80Temp;
               }
              if( iMsgLength >= 8 )  // 0x16+subcode followed by one or more DATA bytes. Example:
               { // pbMessage[0  1  2  3  4  5  6  7  8]
                 // > RX 009 FE FE E0 94 16 12 00 80 FD ; SWRMeterLvl
                 //       |              |  |  |___|-- value in some phantasy unit,
                 //   iMsgLength=9      cmd subcode       usually 4 digit BCD, MSDigit first
                 i32Value = RigCtrl_ParseBCD_CIV_MSByteFirst( pbMessage+6,
                                       2*(iMsgLength/*9*/-7)/*nDigits*/ );
                 SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "%s%c%ld", pszToken, cGetOrSetIndicator, (long)i32Value ); // "set" command with value in some phantasy unit
                 RigCtrl_SetParamValue_Int( pRC, pPI, i32Value ); // no conversion factor required here
                 // ,----------------------------'
                 // '--> Sufficiently robust to NOT CRASH with pPI=NULL :)
                 if( !fIsResponse ) // it's not a RESPONSE, but it has a DATA FIELD, so guess it's a WRITE-COMMAND :
                  { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_WRITE_CMD );
                  }
               }
              else  // 0x16 followed by subcode but NO DATA ...
               { SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "%s ?", pszToken ); // query, no "set" command
                 if( !fIsResponse )  // command WITHOUT DATA, and NOT a response -> it's a READ-COMMAND:
                  { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_CMD );
                  }
               }
            } // end if < sufficient length for COMMAND + SUB-COMMAND > ?
           break; // end case 0x16

        case 0x17: // Send CW messages. Also seen when "RS-BA1 took over control:
                   // > r1 007 FE FE 94 E0 17 FF FD
                   //                         |____ "FF stops sending CW messages"
           if( fIsResponse ) // It's a response,  and it's CI-V, so it can only be .. tada! .. a READ RESPONSE:
            { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_RESP );
              cGetOrSetIndicator = '=';
            } // If it's not a RESPONSE but a COMMMAND, finding the type (WRITE_CMD or READ_CMD) is trickier..
           else // !fIsResponse : guess it's  WRITE-COMMAND, so:
            { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_WRITE_CMD );
            }
           SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "CWmsg:%02X", (int)pbMessage[5] ); // for the log..
           fUnknown=FALSE;
           break; // end case 0x17

        case 0x18: // Turn transceiver on/off (includes "wake-up via USB")
           RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_BASIC, RIGCTRL_MSGTYPE_POWER_ON_OFF);
           // This was the first message received from RS-BA1 :
           //  pbMessage[0  1  2  3  4  5  6]
           // >  r1 007 FE FE 94 E0 18 01 FD
           if( fIsResponse ) // It's a response,  and it's CI-V, so it can only be .. tada! .. a READ RESPONSE:
            { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_RESP );
              cGetOrSetIndicator = '=';
            } // If it's not a RESPONSE but a COMMMAND, finding the type (WRITE_CMD or READ_CMD) is trickier..
           if( iMsgLength >= 7 )
            { SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "Power:%s", (pbMessage[5]==0) ? "OFF" : "ON" );
              // There is no "Parameter Info" for this command, but it's a KNOWN command, so:
              fUnknown=FALSE;
              if( !fIsResponse )
               { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_WRITE_CMD );
               }
            }
           else // message too short for a DATA field ->
            { if( !fIsResponse ) // if it's not a RESPONSE, it's a READ-COMMAND
               { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_CMD );
               }
            }
           break; // end case 0x18

        case 0x19: // Read the transceiver ID (beware, different meaning depending on length) ?
           //   | Example from the traffic log, when WFView controls an IC-7300 :
           //   '-----------------------------,
           //   Time          Len Pre.  Adddr |     Postamble ; Comment
           // > 18:12:41.2 t1 007 FE FE 00 E1 19 00 FD        ; Rig ID ? (BROADCAST to request the radio ID)
           // > 18:12:41.2 r1 008 FE FE E1 94 19 00 94 FD     ; Rig=IC-7300 (response with the radio's "Default ID")
           //                                 |______|--- 0x19 followed by 0x00 AND a 'data byte' !!
           //
           // [IC7610CIV] page 4 : "Cmd 0x19 : Sub Cmd = 0x00  Data = ??? : "Read the transceiver ID".
           RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_BASIC, RIGCTRL_MSGTYPE_RADIO_ID );
           if( fIsResponse ) // It's a response,  and it's CI-V, so it can only be .. tada! .. a READ RESPONSE:
            { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_RESP );
              cGetOrSetIndicator = '=';
            } // If it's not a RESPONSE but a COMMMAND, finding the type (WRITE_CMD or READ_CMD) is trickier..
           if( iMsgLength >= 8 )  // 0x19 followed by TWO BYTES (!) before the postamble:
            { switch( pbMessage[5] )  // "sub-command" after 0x19 ?
               { case 0x00 :  // Guess what follows is the "DEFAULT CI-V address" alias "Transceiver ID code" ...
                    fUnknown = FALSE;
                    pRC->iDefaultAddress = (int)pbMessage[6]; // <- this may overwrite iDefaultAddress=RIGCTRL_DEF_ADDR_AUTO_DETECT (0)
                      //  '--> e.g. 148 = 0x94 = RIGCTRL_DEF_ADDR_IC_7300 .
                    SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "Rig=%s",
                        RigCtrl_DefaultAddressToString(
                          pPortInstance->iRadioCtrlProtocol, pRC->iDefaultAddress) ); // for the log..
                    pRadioInfo = RigCtrl_GetRadioInfoByDefaultAddress( // -> e.g. from RigCtrl_RadioInfo_CIV[]
                        pPortInstance->iRadioCtrlProtocol, pRC->iDefaultAddress ); // for the 'capabilities'
                    if( pRadioInfo != NULL )
                     {  pRC->iCapabilities    = pRadioInfo->iCapabilities;
                        pRC->dwAvailableBands = pRadioInfo->dwBands; // <- may also change later, after RIGCTRL_POLLSTATE_TX_BAND_EDGES
                        //    '--> This is only ONE of several inputs for RigCtrl_GetAvailableBands() !
                        //         Details there.. the radio may be able to report
                        //         its own 'TX band edge frequencies'. If it does,
                        //         RigCtrl_GetAvailableBands() respects limitations
                        //         or additional bands like 60m and 4m.
                        //
                     }
                    else // radio-info not available .. but be optimistic, it may be some brand-new model !
                     {  pRC->iCapabilities = RIGCTRL_CAPS_ALL;
                     }
                    break;
                 default   :  // anything else is neither documented in [IC7610CIV] nor [DF4OR_CIV] ...
                    break;
               }
              if( !fIsResponse )  // command WITH DATA, and NOT a response -> it's a WRITE-COMMAND:
               { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_WRITE_CMD );
               }
            }
           else if( iMsgLength >= 7 )  // 0x19 followed by ONE BYTE (!) before the postamble:
            { switch( pbMessage[5] )  // "sub-command" after 0x19 ?
               { case 0x00 :  // "READ the radio's CI-V default ID" [often sent to the BROADCAST address to find who's out there]
                    fUnknown = FALSE;
                    SL_AppendString( &pszCmtDest, cpCmtEndstop, "Rig ID ?" ); // for the log..
                    RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_BASIC, RIGCTRL_MSGTYPE_RADIO_ID);
                    if( !fIsResponse )  // NOT a response ? Guess it's a READ-COMMAND:
                     { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_CMD );
                     }
                    break;
                 default   :  // anything else is neither documented in [IC7610CIV] nor [DF4OR_CIV] ...
                    break;
               }
              if( !fIsResponse )  // command WITHOUT DATA, and NOT a response -> it's a READ-COMMAND:
               { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_CMD );
               }
            }
           break; // end case 0x19 (CI-V)

        case 0x1A: // read/write filter bandwidths, memory contents,
                   // band stacking registers, memory keyer contents,
                   // and even a few WATERFALL-DISPLAY-RELATED SETTINGS (omg!) ..
           if( fIsResponse ) // It's a response,  and it's CI-V, so it can only be .. tada! .. a READ RESPONSE:
            { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_RESP );
              cGetOrSetIndicator = '=';
            } // If it's not a RESPONSE but a COMMMAND, finding the type (WRITE_CMD or READ_CMD) is trickier..
           else // because CI-V command 0x1A comes in different flavours :
            { // .. with a simple one-byte SUB-COMMAND (IC-7300 / "A7292-4EX-11" page 162):
              //     sub-commands 0x00 .. 0x04, and MAYBE 0x06 + 0x07, but NOT 0x05 .. Aaargh ! !
              // .. with a sub-command, and TWO MORE sub-sub-command-bytes
              //     (sub-command 0x05)
              // INITIAL GUESS: Since it's not a response, it's a READ-COMMAND:
              RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_CMD );
              if( pbMessage[5] == 0x05 ) // the dreadful monster cmd 0x1A sub 0x05 with its MODEL-DEPENDENT sub-sub-commands :
               { if( iMsgLength > (2+2+ 2+2 +1) ) // preamble, addresses, command+subcmd+subsubcmd[2], and postamble ?
                  { // message is long enough for at least ONE data byte -> guess it's a WRITE COMMAND (with data)
                    RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_WRITE_CMD );
                  }
               }
              else // cmd 0x1A with anything but sub-command 0x05 :
               { if( iMsgLength > (2+2+ 2 +1) )  // preamble, addresses, command+subcmd, and postamble ?
                  { // message is long enough for at least ONE data byte -> guess it's a WRITE COMMAND (with data)
                    RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_WRITE_CMD );
                  }
               }
            }
           if( iMsgLength >= 7 )  // 0x1A followed by at least ONE byte for the sub-command...
            { iSubcode = pbMessage[5];
              switch( iSubcode )  // "sub-command" after 0x1A ?
               { case 0x00 :  // response for "Send / read memory contents" ?
                    SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "mem[%d]", (int)pbMessage[6] ); // for the log..
                    break;
                 case 0x01 :  // response for "Send / read band stacking registers" ?
                    SL_AppendString( &pszCmtDest, cpCmtEndstop, "band stack mem" );
                    if( (iMsgLength > 13) // at least "operating frequency" and "operating mode"..
                     && (pRC->iParameterPollingState==RIGCTRL_POLLSTATE_BAND_STACKING_REGS)
                     && (pRC->iNumBandStackingRegs >=0 )
                     && (pRC->iNumBandStackingRegs < RIGCTRL_NUM_BAND_STACKING_REGS) )
                     { // Try to(!) parse the another 'band stacking register' :
                       // [out] pRC->BandStackingRegs[ pRC->iNumBandStackingRegs++ ]
                       // [in] :  --------------------------,
                       //                                   |
                       //                                  \|/
                       // pbMessage[0  1  2  3  4  5  6  7  8  9 10 11 12 13 14 15 16 17 18]
                       // > RX 017 FE FE 00 94 1A 01 01 01 30 96 83 01 00 00 02 01 00 08 85 00 08 85 .. ; band stack mem
                       //          |PRE| |ADR| CMD |  |  | |____________|  | |__etc, etc, etc, etc, etc, etc, etc ...
                       //                         Sub |  | 10-digit freq  OpMode
                       //         "Frequency band code" "Register code"   (see RigCtrl_IcomToRigCtrlOpmode)
                       //          (ignored here)        (ignored here)
                       pMemEntry = &pRC->BandStackingRegs[pRC->iNumBandStackingRegs];
                       if( RigCtrl_ParseMemoryContent( pbMessage+8, iMsgLength-8-1/*nBytes*/, pMemEntry) )
                        { SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, ": %.1lf kHz, %s",
                            (double)(pMemEntry->RxTx[0].dblOperatingFreq_Hz * 1e-3),
                            RigCtrl_OperatingModeToString( pMemEntry->RxTx[0].iOpMode) );
                          ++pRC->iNumBandStackingRegs;
                        }
                     }
                    break;
                 case 0x02 :  // "Send / read memory keyer contents"
                    SL_AppendString( &pszCmtDest, cpCmtEndstop, "keyer mem" );
                    break;
                 case 0x03 :  // "Send / read selected IF filter"
                    // Surprisingly, Cmd 0x1A sub 0x03 was included in WSJT-X's "polling cycle".
                    // Thus this parameter (converted into Hz) now has its own PN :
                    iUnifiedPN = RIGCTRL_PN_FILTER_BANDWIDTH;
                    pPI = RigCtrl_GetInfoForUnifiedParameterNumber(iUnifiedPN);
                    if( pPI != NULL )
                     { pszToken = pPI->pszToken;
                       iCIVDataType = pPI->iCIVDataType;
                     }
                    else
                     { pszToken = "FilterBW";
                     }
                    // The unit for FILTER BANDWIDTHS used by Icom / IC-7300
                    //             is specified somewhere as quoted below:
                    // > Cmd 0x1A subcmd 0x03: Send/read the selected filter width.
                    // >   ( AM: 00=200 Hz to 49=10 kHz;
                    // >     other than AM modes:
                    // >          00=50 Hz to 31/40=2700 Hz/3600 Hz)
                    // Heavens, no. A CONVERSION FORMULA or simply a SIX-DIGIT BCD
                    // number would be asking too much. Hey Icom, you managed
                    // to send SIX DIGIT DECIMAL NUMBERS elsewhere, so you could
                    // even report the filter bandwidth if the radio (IC-705) was in "Wide FM" !
                    //
                    // Example with an IC-7300 in USB, "FIL2" (seen in each WSJT-X polling cycle):
                    // pbMessage[0  1  2  3  4  5  6  7]
                    // > TX 007 FE FE 94 E0 1A 03 FD    ; FilterBW ?
                    // > RX 008 FE FE E0 94 1A 03 28 FD ; FilterBW = "28" .
                    if( iMsgLength > 7 ) // smells like a SET-command or READ-response for the 'selected IF filter bandwidth' :
                     { i32Value = RigCtrl_ParseBCD_CIV_MSByteFirst( pbMessage+6, 2/*digits*/ );
                       i32Value = RigCtrl_CIV_AudioFilterBandwidthCodeToFrequency( i32Value );
                       RigCtrl_SetParamValue_Int( pRC, pPI, i32Value );
                       SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "%s=%d Hz", pszToken, i32Value );
                       if( !fIsResponse ) // it's not a RESPONSE, but it has a DATA FIELD, so guess it's a WRITE-COMMAND :
                        { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_WRITE_CMD );
                        }
                     }
                    else
                     { SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "%s ?", pszToken );
                       if( !fIsResponse )  // NOT a response ? Guess it's a READ-COMMAND:
                        { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_CMD );
                        }
                     }
                    break;
                 case 0x04 :  // "Send / read selected AGC time constant"
                    SL_AppendString( &pszCmtDest, cpCmtEndstop, "AGC speed" );
                    break;
                 case 0x05 :  // "Send / read an awful lot of RIG-SPECIFIC(!) PARAMETERS"
                    // Typical QUERY (seen after starting RS-BA1 on client port #1):
                    //      pbMessage[0  1  2  3  4  5  6  7  8]
                    //      > r1 009 FE FE 94 E0 1A 05 00 60 FD
                    //           |_|             |  |  |___| '-- postamble
                    //           iMsgLength=9    cmd     '------ 4-digit "subcode"
                    if( iMsgLength >= 9 ) // valid QUERY or SET-command ?
                     { iSubcode = RigCtrl_ParseBCD_CIV_MSByteFirst( pbMessage+6, 4/*digits*/ );
                       sprintf( sz80Temp, "Cfg[%04d]", (int)iSubcode );
                       pszToken = sz80Temp;
                       // Actually, the list of subcodes addressed with this
                       // 4-digit BCD extends over MULTIPLE PAGES in the IC-7610's CI-V manual.
                       // Very unfortunately, the meaning of those 4-digit "subcodes"
                       // is extremely rig specific.
                       // Not even an IC-7300, IC-9700, IC-705 or IC-7610 are compatible with each other !
                       // Thus: Let the following subroutine check it (it's aware of the radio model):
                       iUnifiedPN = RigCtrl_SubcodeFrom1A05_to_UnifiedParameterNumber(pRC->iDefaultAddress, iSubcode );
                       if( iUnifiedPN != RIGCTRL_PN_UNKNOWN )
                        { pPI = RigCtrl_GetInfoForUnifiedParameterNumber( iUnifiedPN );
                          // '--> this is also important to decide if this is a "periodicall polled" parameter,
                          //      optionally rejected from the 'traffic log' way further below.
                          if( pPI != NULL )
                           { iCIVDataType = pPI->iCIVDataType;
                             if( pPI->pszToken != NULL )
                              {  pszToken = pPI->pszToken; // e.g. "WaterfallSpeed" (not to be confused with "Scope Speed"... aaaarggh ! ! ! )
                                 // 2025-07-19: the response with "WaterfallSpeed" was coloured CYAN, not GREEN, in the traffic log.
                                 // Reason: See RigCtrl_ReadTrafficLog() .. RIGCTRL_MSGTYPE_FLAG_EXPECTED was missing in .iMsgType .
                                 // Seen at THIS point:
                                 //    dwCombinedCmdAndSubcode                         = 0x1A 05 01 08
                                 //    pPortInstance->dwExpectedResponse_CmdAndSubcode = 0x1A 05 00 00 (!)
                                 //    pPortInstance->iExpectResponseForUnifiedPN   = 35
                                 //    iUnifiedPN                                       = 35
                                 //    fIsExpectedResponse      =
                                 // Fixed FURTHER BELOW (common treatment for MANY OTHER
                                 //  command / subcommands / subsubcommands / subsubsubcommands)
                                 //  by setting fIsExpectedResponse (and, even later, RIGCTRL_MSGTYPE_FLAG_EXPECTED).
                              }
                           } // end if( pPI != NULL )
                        }
                       else // must be one of the "less important" parameters..
                        {   // .. at least RigControl.c doesn't know what it is.
                        }
                       if( iMsgLength > 9 ) // long enough for a "SET"-command.
                        { i32Value = RigCtrl_ParseBCD_CIV_MSByteFirst( pbMessage+8,
                                       2*(iMsgLength/*11*/-9)/*nDigits*/ );
                          // Typical READ RESPONSE (from an IC-7300, here: subcode 0060
                          //   -> iUnifiedPN = RIGCTRL_PN_AF_OUTPUT_LEVEL_TO_USB =
                          // "AF output level to ACC/USB", iMsgLength=11, 2 byte data = 4 BCD digit
                          //      pbMessage[0  1  2  3  4  5  6  7  8  9 10]
                          //      > RX 00B FE FE E0 94 1A 05 00 60 01 30 FD
                          //      _________________________________|___|
                          //     | nDigits = 2 * (11-9) = 4, i.e. BCD-data with
                          //   a 4-digit "value" (here: volume ranging from 000 to 0255 = 100%)
                          SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "%s=%d", pszToken, (long)i32Value );
                          if( !fIsResponse ) // it's not a RESPONSE, but it has a DATA FIELD, so guess it's a WRITE-COMMAND :
                           { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_WRITE_CMD );
                           }
                          else // guess it's a READ-RESPONSE for the dreadful command 0x1A sub 05 ...
                           { // Example (traffic log when reading settings from an IC-7300) :
                             // > TX 009 FE FE 94 E0 1A 05 00 61 FD           ; read SquelchForAudioOnAcc
                             // > RX 00A FE FE E0 94 1A 05 00 61 00 FD           ; SquelchForAudioOnAcc=0
                             // > TX 009 FE FE 94 E0 1A 05 00 64 FD           ; read ModInputLevelFromACC
                             // > RX 00B FE FE E0 94 1A 05 00 64 01 28 FD      ; ModInputLevelFromACC=128
                             // > TX 009 FE FE 94 E0 1A 05 00 65 FD           ; read ModInputLevelFromUSB
                             // > RX 00B FE FE E0 94 1A 05 00 65 01 28 FD      ; ModInputLevelFromUSB=128
                             // > TX 009 FE FE 94 E0 1A 05 00 66 FD       ; read ModInputConnector_NoData
                             // > RX 00A FE FE E0 94 1A 05 00 66 04 FD       ; ModInputConnector_NoData=4
                             // > TX 009 FE FE 94 E0 1A 05 00 67 FD         ; read ModInputConnector_Data
                             // > RX 00A FE FE E0 94 1A 05 00 67 02 FD         ; ModInputConnector_Data=2
                             // Update the parameter value in the T_RigCtrlInstance:
                             RigCtrl_SetParamValue_Int( pRC, pPI, i32Value ); // here: for the dreadful command 0x1A 0x05 "NNNN" (model-specfic!)
                           }
                        }
                       else // message too short for a "set"-command or a READ-response, so guess it's a QUERY ("get"):
                        { SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "%s ?", pszToken );
                          if( !fIsResponse )  // NOT a response ? Guess it's a READ-COMMAND:
                           { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_CMD );
                           }
                        }
                       // Seen when launching "RS-BA1", connected via SpecLab to an IC-7300:
                       //    Cfg[0052]=0 ( "Function > Memo Pad Quantity" : 5 channels )
                       //    Cfg[0057]=0 ( "Function > Quick RIT/deltaTX Clear" :  OFF )
                       //    Cfg[0071]=1 ( "Connectors > Phones > Level" ? )
                       //    Cfg[0066]=3 ( IC-7300: "MOD input connector during DATA" : 3=USB.
                       //                  Totally different on IC-7610 :
                       //                  0066 would be "Screen Capture Keyboard".. EEEK !!)
                       switch( iUnifiedPN )
                        { case  RIGCTRL_PN_CIV_USB_ECHO : // 0x1A 0x05 Cfg[0075] : "Echo back setting for CI-V operation from USB"
                             // [G3NRW_CIV]
                             // > Note that the description of the "Set echo back setting"
                             // >  in the Command Table in Section 19 of the
                             // >  IC-7300 Full Manual is misleading. The
                             // >  parameter 00 sets the control to OFF
                             // >  and the parameter 01 sets it to ON .
                             // Here, each client may have prefer its own setting.
                             // Most clients will not know about how to configure
                             // the CI-V "echo back setting" via CI-V itself,
                             // thus in Spectrum Lab, the initial setting of
                             //  T_RigCtrlInstance.PortInstance[x].fEmulateCIVEcho
                             // can be configured via checkmark in the setup dialog.
                             break;
                          default: // ignore hundreds(!) of other "subcodes" after 0x1A 0x05
                             break;
                        } // end switch( iUnifiedPN )
                     } // end < message long enough for cmd 0x1A 0x05 >
                    break; // end case "Send / read an awful lot of RIG-SPECIFIC PARAMETERS"
                 case 0x06 :  // (IC-7300) : "Data mode with filter width settings"
                    // Example with an IC-7300 in USB, "FIL2" (seen in each WSJT-X polling cycle):
                    // pbMessage[0  1  2  3  4  5  6  7]
                    // > TX 007 FE FE 94 E0 1A 06 FD          ; Data/Filter ?
                    // > RX RX 009 FE FE E0 94 1A 06 00 00 FD ; Data mode OFF, no filter info.
                    //                                |  |__ only when DATA ON: 01=Fil1, 02=Fil2, 03=Fil3
                    //                                0=Data mode off, 1=Data on.
                    iUnifiedPN = RIGCTRL_PN_DATA_MODE;
                    pPI = RigCtrl_GetInfoForUnifiedParameterNumber(iUnifiedPN);
                    if( pPI != NULL )
                     { iCIVDataType = pPI->iCIVDataType;
                       pszToken = pPI->pszToken;
                     }
                    else
                     { pszToken = "DataMode";
                     }
                    if( iMsgLength >= 8 ) // valid SET-command or GET-response ?
                     { i32Value = RigCtrl_ParseBCD_CIV_MSByteFirst( pbMessage+7, 2/*digits*/ );
                       RigCtrl_SetParamValue_Int( pRC, pPI, i32Value );
                       SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "%s=%d", pszToken, i32Value );
                       if( !fIsResponse ) // it's not a RESPONSE, but it has a DATA FIELD, so guess it's a WRITE-COMMAND :
                        { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_WRITE_CMD );
                        }
                     }
                    else
                     { SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "%s ?", pszToken );
                       if( !fIsResponse )  // NOT a response ? Guess it's a READ-COMMAND:
                        { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_CMD );
                        }
                     }
                    break;
                 case 0x07 :  // "NTP server access" (IC-7610) / "IP+ Setting" (IC-7300)
                 case 0x08 :  // "NTP access result" (IC-7610) / not used in IC-7300
                 case 0x09 :  // "AF Mute"           (IC-7610) / not used in IC-7300
                    iUnifiedPN = RigCtrl_SubcodeFrom1A_to_UnifiedParameterNumber(pRC, iSubcode);
                    pPI = RigCtrl_GetInfoForUnifiedParameterNumber(iUnifiedPN);
                    if( pPI != NULL )  // anything known about this sub-command ? Do we care ?
                     { pszToken = pPI->pszToken;
                       iCIVDataType = pPI->iCIVDataType;
                       if( iMsgLength > 7 ) // valid SET-command or GET-response ?
                        { // Here just one of the RESPONSES from an IC-7300,
                          // after being asked by RS-BA1 for whatever-it-was :
                          //          [0  1  2  3  4  5  6  7]
                          // > RX 008 FE FE E0 94 1A 07 00 FD
                          i32Value = RigCtrl_ParseBCD_CIV_MSByteFirst( pbMessage+6,
                                         2*(iMsgLength/*8*/-7)/*nDigits*/ );
                          RigCtrl_SetParamValue_Int( pRC, pPI, i32Value );
                          SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "%s=%d", pszToken, i32Value );
                          if( !fIsResponse )  // NOT a response ? Guess it's a WRITE-COMMAND:
                           { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_WRITE_CMD );
                           }
                        }
                       else // Just a QUERY .. one of hundreds seen from RS-BA1 :
                        { //          [0  1  2  3  4  5  6]
                          // > r1 007 FE FE 94 E0 1A 07 FD (other)
                          SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "%s ?", pszToken );
                          if( !fIsResponse )  // NOT a response ? Guess it's a READ-COMMAND:
                           { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_CMD );
                           }
                        }
                     }
                    else // don't care what it is .. DL4YHF was sick of hard-coding RIG-SPECIFIC details here
                     {
                       if( iMsgLength > 7 ) // Throw the dice again: valid SET-command or GET-response ?
                        { if( !fIsResponse )  // NOT a response ? Guess it's a WRITE-COMMAND:
                           { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_WRITE_CMD );
                           }
                        }
                       else // Message too short for a DATA field ?Just a QUERY .. one of hundreds seen from RS-BA1 :
                        { if( !fIsResponse )  // NOT a response ? Guess it's a READ-COMMAND:
                           { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_CMD );
                           }
                        }
                     }
                    break;
                 default:
                    break;
               } // end switch < "sub-command" after 0x1A >
            } // end if < sufficient message length for this command ? >
           break; // end case 0x1A

        case 0x1B: // repeater tone frequency, TSQL tone frequency, ...
           if( fIsResponse ) // It's a response,  and it's CI-V, so it can only be .. tada! .. a READ RESPONSE:
            { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_RESP );
              cGetOrSetIndicator = '=';
            } // If it's not a RESPONSE but a COMMMAND, finding the type (WRITE_CMD or READ_CMD) is trickier..
           if( iMsgLength >= 7 )  // 0x1B followed by subcode, with or without DATA ...
            { iSubcode = pbMessage[5];
              switch( iSubcode )  // decode "sub-command" after 0x1B for the log:
               { case 0x00 : iUnifiedPN = RIGCTRL_PN_REPEATER_TONE_FREQ; break;
                 case 0x01 : iUnifiedPN = RIGCTRL_PN_TONE_SQUELCH_FREQ;  break;
                 default   : break;
               } // end switch < "sub-command" after 0x1B >
              pPI = RigCtrl_GetInfoForUnifiedParameterNumber(iUnifiedPN);
              if( pPI != NULL )
               { iCIVDataType = pPI->iCIVDataType;
                 pszToken = pPI->pszToken;
                 pszUnit = pPI->pszUnit;
                 if( pszUnit != NULL )
                  { if( pszUnit[0] == '\0' )
                     {  pszUnit = NULL;
#                      ifdef __BORLANDC__  // "... is assigned a value that is never used".. shut up ..
                        (void)pszUnit;
#                      endif
                     }
                  }
               }
              else
               { sprintf(sz80Temp, "0x1B[%02X]", (int)iSubcode );
                 pszToken = sz80Temp;
               }
              if( iMsgLength >= 10 )  // 0x1B+subcode followed by DATA bytes. Example:
               { // pbMessage[0  1  2  3  4  5  6  7  8  9]
                 // > RX 00A FE FE E0 94 1B 01 00 08 85 FD ; TSQL Tone
                 //       |              |  |  |______|-- value in 0.1 Hz units,
                 //   iMsgLength=10     cmd subcode       6 digit BCD, MSDigit first
                 d = 0.1 * (double)RigCtrl_ParseBCD_CIV_MSByteFirst( pbMessage+6,
                                       2*(iMsgLength/*10*/-7)/*nDigits*/ );
                 SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "%s%c%.1lf Hz", pszToken, cGetOrSetIndicator, (double)d ); // "set" command with value in Hz
                 if( RigCtrl_SetParamValue_Double( pRC, pPI, d ) )
                  { //   '--> returns TRUE when "modified", so, also here:
                    // Inform module CwNet.c about the modified parameter.
                    // When running as SERVER, CwNet passes this on to remote clients:
                    RigCtrl_AddParamToTxQueueForCwNet( pRC, iUnifiedPN ); // here: "something reported as a sub-command for CI-V command 0x1B"
                  }
                 if( !fIsResponse ) // it's not a RESPONSE, but it has a DATA FIELD, so guess it's a WRITE-COMMAND :
                  { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_WRITE_CMD );
                  }
               }
              else  // 0x1B followed by subcode but NO DATA ...
               { // Seen when starting "RS-BA1" : > r1 007 FE FE 94 E0 1B 00 FD
                 SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "%s ?", pszToken ); // query, no "set" command
                 if( !fIsResponse )  // NOT a response ? Guess it's a READ-COMMAND:
                  { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_CMD );
                  }
               }
            } // end if < sufficient length for COMMAND + SUB-COMMAND > ?
           break; // end case 0x1B

        case 0x1C: // transceiver status (RX/TX), antenna tuner, XFC, read TX frequency, etc .
           if( fIsResponse ) // It's a response,  and it's CI-V, so it can only be .. tada! .. a READ RESPONSE:
            { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_RESP );
              cGetOrSetIndicator = '=';
            } // If it's not a RESPONSE but a COMMMAND, finding the type (WRITE_CMD or READ_CMD) is trickier..
           // Received from RS-BA1 : > r1 007 FE FE 94 E0 1C 04 FD  (with iMsgLen=7, only a QUERY)
           //                                                |__ "CIV_ANT" ?
           if( iMsgLength >= 7 )  // 0x1C followed by at least <??> bytes...
            { iSubcode = pbMessage[5];
              switch( iSubcode )  // "sub-command" after 0x1C ?
               { case 0x00 : iUnifiedPN = RIGCTRL_PN_TRANSMITTING;
                    break;
                 case 0x01 : iUnifiedPN = RIGCTRL_PN_ATU_ENABLED;
                    break;
                 case 0x02 : iUnifiedPN = RIGCTRL_PN_XFC_PRESSED;
                    break;
                 case 0x03 : iUnifiedPN = RIGCTRL_PN_TRANSMIT_FREQ;
                    break;
                 case 0x04 : iUnifiedPN = RIGCTRL_PN_CIV_OUTPUT_FOR_ANT; // "CI-V output for ANT"
                    break;
                 default   :
                    break;
               } // end switch < "sub-command" after 0x1C >
              pPI = RigCtrl_GetInfoForUnifiedParameterNumber(iUnifiedPN);
              if( pPI != NULL )
               { pszToken = pPI->pszToken;  // e.g. "Transmitting"
                 iCIVDataType = pPI->iCIVDataType;
                 RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGTYPE_FLAG_PERIODIC_POLL,
                                   pPI->iMsgType & RIGCTRL_MSGTYPE_FLAG_PERIODIC_POLL );
               }
              else
               { sprintf(sz80Temp, "TrxSt[%02X]", (int)iSubcode );
                 pszToken = sz80Temp;
               }
              if( iMsgLength >= 8 )  // 0x1C+subcode followed by one or more DATA bytes. Example:
               { // pbMessage[0  1  2  3  4  5  6  7]
                 // > RX 008 FE FE E0 94 1C 04 00 FD
                 //       |  'pre' to fm |  |  '-- value, here only two digits,
                 //   iMsgLength=8      cmd subcode       guess this is BCD, MSDigit first
                 i32Value = RigCtrl_ParseBCD_CIV_LSByteFirst_Int32( pbMessage+6,
                                       2*(iMsgLength/*8*/-7)/*nDigits*/ );
                   // ,-----------------------------------'
                   // '--> total message length minus SEVEN bytes
                   //      for preamble(2), addresses(2), com+sub(1), postamble(1).
                   // Other example (with "TX_freq" from an IC-9700) :
                   //   LEAST significant byte (with tens and ones) first
                   //    '--------------,           ,----  1-GHz-digit for IC-9700;
                   //                   |           |      thus expect more BYTES for e.g. IC-905 !
                   // FE FE 00 A2 1C 03 10 00 05 44 01 FD
                   //              |  | |____________| |
                   //            cmd sub     |         postamble
                   //                     12-7 = 5 BYTES = TEN BCD-digits,
                   //                decoded into 0144050010 Hz = 144.05001 MHz.
                 SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "%s%c%ld", pszToken, cGetOrSetIndicator, (long)i32Value ); // "set" command with value
                 if( pszUnit != NULL )
                  { SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, " %s", pszUnit );
                  }
                 if( RigCtrl_SetParamValue_Int( pRC, pPI, i32Value ) )
                  { //   '--> returns TRUE when "modified", so, also here:
                    // Inform module CwNet.c about the modified parameter.
                    // When running as SERVER, CwNet passes this on to remote clients:
                    RigCtrl_AddParamToTxQueueForCwNet( pRC, iUnifiedPN ); // here: "something reported as a sub-command for CI-V command 0x1C"
                  }
                 if( !fIsResponse ) // it's not a RESPONSE, but it has a DATA FIELD, so guess it's a WRITE-COMMAND :
                  { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_WRITE_CMD );
                  }
               }
              else  // 0x1C followed by subcode but NO DATA ...
               { SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "%s ?", pszToken ); // query, no "set" command
                 if( !fIsResponse )  // NOT a response ? Guess it's a READ-COMMAND:
                  { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_CMD );
                  }
               }
            } // end if < sufficient message length for this command ? >
           break; // end case 0x1C

        case 0x1E: // read number of TX frequency bands, TX band edge frequencies, etc
           RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_BASIC, RIGCTRL_MSGTYPE_OTHER );
           if( fIsResponse ) // It's a response,  and it's CI-V, so it can only be .. tada! .. a READ RESPONSE:
            { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_RESP );
              cGetOrSetIndicator = '=';
            } // If it's not a RESPONSE but a COMMMAND, finding the type (WRITE_CMD or READ_CMD) is trickier..
           if( iMsgLength >= 7 )  // 0x1E followed by at least <??> bytes...
            {
              if( iMsgLength >= 8 )  // 0x1E+subcode followed by one or more DATA bytes. Example:
               { // pbMessage[0  1  2  3  4  5  6  7  8  9 10 11 12 13 14 15 16 17 18]
                 // > RX 013 FE FE 00 A2 1E 01 01 00 00 00 44 01 2D 00 00 00 46 01 FD ; 144 MHz .. 146 MHz
                 //       |  |___| |addr |  |  |  |____________| |  |____________|
                 //       | Preamble     |  | "Edge"   |        Sep.    |
                 //   iMsgLength=19   cmd sub  |     StartFreq        EndFreq
                 // ,--------------------------'
                 i32Value = RigCtrl_ParseBCD_CIV_MSByteFirst( pbMessage+6, 2/*nDigits*/ );
                 if( iMsgLength <= 9 ) // guess it's not an array-like struct but a simple integer parameter:
                  { SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "%s=%ld", pszToken, (long)i32Value );
                  }
                 else
                  { SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "%s[%ld]=", pszToken, (long)i32Value );
                    // ,-------------------------------------------------'
                    // '--> The FREQUENCY RANGE itself will be appended further below,
                    //      depending on the 'subcode', e.g. 0x01 = "TX band edge frequencies".
                  }
                 if( !fIsResponse ) // it's not a RESPONSE, but it has a DATA FIELD, so guess it's a WRITE-COMMAND :
                  { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_WRITE_CMD );
                  }
               }
              else  // 0x1E followed by a subcode but NO DATA ... possibly a query (?)
               { SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "%s ?", pszToken );
                 if( !fIsResponse )  // NOT a response ? Guess it's a READ-COMMAND:
                  { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_CMD );
                  }
               }
              iSubcode = pbMessage[5];
              switch( iSubcode )  // "sub-command" after 0x1E ?
               { case 0x00 : // "Read number of available TX frequency band"[s??] IC-9700_ENG_CI-V_4.pdf page 13
                    pPI = RigCtrl_GetInfoForUnifiedParameterNumber(RIGCTRL_PN_NUM_TX_BANDS);
                    if( pPI != NULL )
                     { pszToken = pPI->pszToken;
                       iCIVDataType = pPI->iCIVDataType;
                       RigCtrl_SetParamValue_Int( pRC, pPI, i32Value );
                       UTL_LimitInteger( &pRC->iNumTxBands, 0, RIGCTRL_NUM_TX_BAND_EDGES );
                     }
                    break;
                 case 0x01 : // "Read TX band edge frequencies" IC-9700_ENG_CI-V_4.pdf page 13
                    if( (i32Value>=1) && (i32Value<=RIGCTRL_NUM_TX_BAND_EDGES) ) // valid "edge number" ?
                     {
                       // This kind-of-array-index (Icom calls it "Edge Number", 01 .. 30 in BCD)
                       // isn't the sub-command but part of the DATA FIELD !
                       T_RigCtrlFrequencyRange *pFreqRange = &pRC->TxBandEdges[i32Value-1];
                       // From the IC-9700 CI-V Reference Guide Rev. 4, document "A7508-3EX-4", page 13:
                       // > Band edge frequency settings
                       // pbMessage[6       7  8  9 10 11  12  13 14 15 16 17  18]
                       //          Edge#    |___________|  2D  |____________|  FD
                       //          01..30   Lower edge,BCD  |  Upper edge, BCD  |
                       //                   LSByte first    |  LSByte first     |
                       //                            "Separator (fixed)"     Postamble
                       RigCtrl_ParseCIVFreqRange( pbMessage+7, iMsgLength-7-1/*nBytes*/, pFreqRange );
                       // Flag 'fIsExpectedResponse' has already been set above,
                       // so the state machine in RigCtrl_Handler(),
                       //         case RIGCTRL_POLLSTATE_TX_BAND_EDGES,
                       // will read the NEXT 'TX band edge frequencies' very soon.
                       pszToken = "TxBand";
                       SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "%.1lf .. %.1lf MHz", pFreqRange->dblFmin_Hz*1e-6, pFreqRange->dblFmax_Hz*1e-6 );
                     } // end if < parameter-info Non-NULL for RIGCTRL_PN_TX_BAND_EDGES >
                    break;
                 case 0x02 : // "Read number of user-set TX frequency band"[s??]
                    pszToken = "User TX bands";
                    break;
                 case 0x03 : // "Set the user-set TX band edge frequencies"
                    pszToken = "User band edges";
                    // Format as for cmd 1E sub 01 "TX band edge frequencies"
                    break;
                 default   : sprintf(sz80Temp, "TxBands[%02X]", (int)iSubcode );
                    pszToken = sz80Temp;
                    break;
               } // end switch < "sub-command" after 0x1E >
            } // end if < sufficient message length for this command ? >
           break; // end case 0x1E

        case 0x21: // RIT frequency setting, delta TX setting (on/off)
           // N1MM Logger+ seemed to periodically POLL these values; e.g.
           // >  r2 007 FE FE 94 E0 21 00 FD
           // >  r2 007 FE FE 94 E0 21 01 FD
           // 2025-09-05: The TRAFFIC LOG didn't show a command even though
           //             pPI->pszToken = "RIT_freq" (correct, from RigCtrl_ParameterInfo[]).
           RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_BASIC, RIGCTRL_MSGTYPE_RIT_XIT );
           if( fIsResponse ) // It's a response,  and it's CI-V, so it can only be .. tada! .. a READ RESPONSE:
            { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_RESP );
              cGetOrSetIndicator = '=';
            } // If it's not a RESPONSE but a COMMMAND, finding the type (WRITE_CMD or READ_CMD) is trickier..
           break; // end case 0x21

        case 0x25:// "Send/read main or sub-band frequency" (IC-7610)
                   // "Send/read the selected or unselected VFO frequency" (IC-7300)
                   // THANK YOU for using different names for the same thing, Icom !
           RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_BASIC, RIGCTRL_MSGTYPE_FREQUENCY_REPORT );
           //                        pbMessage[0  1  2  3  4  5  6  7  8  9 10 11]
           // Received from RS-BA1 : > r1 007 FE FE 94 E0 25 00 FD   (years later, also received from WSJT-X)
           // Response from IC-7300: > RX 00C FE FE E0 94 25 00 00 30 35 05 00 FD
           //                              |              |   | |____________|-- data, BCD, 10 decimal digits, LEAST significant digits first !
           //                          iMsgLength=12     cmd  0=selected VFO,
           //                                                 1=unselected VFO.
           if( fIsResponse ) // It's a response,  and it's CI-V, so it can only be .. tada! .. a READ RESPONSE:
            { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_RESP );
              cGetOrSetIndicator = '=';
            } // If it's not a RESPONSE but a COMMMAND, finding the type (WRITE_CMD or READ_CMD) is trickier..
           if( iRigCtrlPort == RIGCTRL_PORT_RADIO )
            {  iRigCtrlPort = iRigCtrlPort;  // place for a conditional breakpoint when received from the REAL RADIO
            }
           if( iMsgLength >= 7 )  // 0x25 + subcode ("sel/unsel") with or without data..
            { iSubcode = pbMessage[5];
              switch( iSubcode )  // "sub-command" after 0x25 ?
               { case 0x00 : pszToken = "SelVFOFreq";   // name in the IC-7760: "Main VFO"
                             iUnifiedPN = RIGCTRL_PN_SEL_VFO_FREQUENCY;
                             break;
                 case 0x01 : pszToken = "UnselVFOFreq"; // name in the IC-7760: "Sub VFO"
                             iUnifiedPN = RIGCTRL_PN_UNSEL_VFO_FREQUENCY;
                             break;
                 default   : pszToken = "SomeOtherFreq";
                             break;
               } // end switch < "sub-command" after 0x1B >
              if( iMsgLength >= 8 )  // 0x25+subcode followed by one or more DATA bytes
               { dblValue = RigCtrl_ParseBCD_CIV_LSByteFirst_Double( pbMessage+6,
                                       2*(iMsgLength/*12*/-7)/*nDigits*/ );
                 i32Value = (long)dblValue; // <- this may fail at "microwave" frequencies .. IC-905 anyone ?
                 switch( iSubcode )
                  { case 0x00 : // an Icom radio has just reported its "SELECTED VFO FREQUENCY".
                       //         (or WFView has sent a command to SET the "Selected VFO Frequency")
                       // Don't care about the subtle differences between all those "set/read the frequency"-commands,
                       // just store it for the MAIN VFO DISPLAY ...
                       // but also HERE, only accept the radio's reported value if we're not going to SEND a NEW frequency soon:
                       if( ! ( pRC->iPostponedSetMessages & (1<<RIGCTRL_MSGTYPE_SET_FREQUENCY) ) )
                        {
                          if( dblValue != pRC->dblVfoFrequency )
                           { ++pRC->iFrequencyModifiedByRadio_cnt; // <- counter is polled in a GUI-thread somewhere.
                           }
                          pRC->dblVfoFrequency = dblValue; // here: store the received "Selected VFO frequency" from cmd 0x25 sub 0x00
                        } // end if < no postponed SET-command for the frequency pending >
                       break; // end if < CI-V cmd 0x25, sub 0x00 >
                    case 0x01 : // an Icom radio has just reported its "UNSELECTED VFO FREQUENCY".
                       // Because WFView is also "very interested" in this (polling it frequently), STORE IT, TOO:
                       pRC->dblUnselVfoFreq = dblValue;
                       break; // end if < CI-V cmd 0x25, sub 0x00 >
                    default:
                       break;
                  } // end switch( iSubcode )
                 if( !fIsResponse ) // it's not a RESPONSE, but it has a DATA FIELD, so guess it's a WRITE-COMMAND :
                  { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_WRITE_CMD );
                    // leave cGetOrSetIndicator as preset in the initialisation.
                  }
                 SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "%s%c%ld", pszToken, cGetOrSetIndicator, (long)i32Value ); // "set" command with value
                 // Example (seen VERY frequently when eavesdropping traffic between WFView and an IC-7300):
                 // > 20:33:48.1 r3 007 FE FE 94 E1 25 00 FD                ; SelVFOFreq ?
                 // > 20:33:48.1 r3 00C FE FE E1 94 25 00 00 83 03 07 00 FD ; SelVFOFreq:7038300
                 // WFView also sends "SelVFOFreq" as a COMMAND to switch to a new frequency,
                 // which RigControl.c must interpret correcly (even though WFView immediately 'verifies' by READING):
                 // > 17:59:16.5 t1 00C FE FE 94 E1 25 00 00 60 07 07 00 FD ; SelVFOFreq:7076000 (SET command)
                 // > 17:59:16.5 r1 006 FE FE E1 94 FB FD                   ; "ok"               (response)
                 // > 17:59:16.6 t1 007 FE FE 94 E1 25 00 FD                ; SelVFOFreq ?       (GET command)
                 // > 17:59:16.6 r1 00C FE FE E1 94 25 00 00 60 07 07 00 FD ; SelVFOFreq:7076000 (READ response)
               }
              else  // 0x16 followed by a subcode but NO DATA ...
               { SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "%s ?", pszToken ); // query, no "set" command
                 if( !fIsResponse )  // NOT a response, and NO DATA ? Guess it's a READ-COMMAND:
                  { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_CMD );
                  }
               }
            } // end if < sufficient message length for this command ? >
           break; // end case 0x25

        case 0x26: // Send/read operating mode, "data mode"(?!), and filter setting
           RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_BASIC, RIGCTRL_MSGTYPE_OP_MODE );
           if( fIsResponse ) // It's a response,  and it's CI-V, so it can only be .. tada! .. a READ RESPONSE:
            { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_RESP );
              cGetOrSetIndicator = '=';
            } // If it's not a RESPONSE but a COMMMAND, finding the type (WRITE_CMD or READ_CMD) is trickier..
           //                        pbMessage[0  1  2  3  4  5  6  7  8  9]
           // Received from RS-BA1 : > r1 007 FE FE 94 E0 26 01 FD
           // Response from IC-7300: > RX 00A FE FE E0 94 26 01 03 00 02 FD
           //                              |              |   |  |  |  |
           //                              |              |   |  |  |  '--> Filter (1..3)
           //                              |              |   |  |  '--> DataMode, 0=off, 1=on
           //                              |              |   |  '--> OpMode, 3=CW
           //                          iMsgLength=10     cmd  0=selected VFO,
           //                                                 1=unselected VFO.
           if( iMsgLength >= 7 )  // 0x26 followed by         |
            { // at least ONE byte before the postamble:      |
              iSubcode = pbMessage[5];  // <------------------'
              // pszToken = : already set to "SelVfoOpMode"
              // iUnifiedPN : already set to RIGCTRL_PN_SEL_VFO_OP_MODE
              // fUnknown   : already cleared because we have a valid 'Parameter Info' (pPI!=NULL)
              if( iMsgLength >= 8 ) // 0x26 + VFO-Nr. + at least ONE data byte ? WRITE-COMMAND or READ-RESPONSE ..
               { int iFilterIndex = 0;
                 int iDataMode = 0;
                 iNewSetting = RigCtrl_IcomToRigCtrlOpmode( pbMessage[6] ); // CW,USB,LSB,.. ?
                 if( !fIsResponse )
                  { // If it's not a RESPONSE, and it has a DATA FIELD, it must be a WRITE-COMMAND :
                    RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_WRITE_CMD );
                  }
                 SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "%s%c%s", pszToken, cGetOrSetIndicator, RigCtrl_OperatingModeToString(iNewSetting) );
                 if( iMsgLength >= 10 ) // "Operating mode" byte followed by optional "Data mode" and "Filter setting" ?
                  { iDataMode = pbMessage[7];
                    switch( pbMessage[8] )  // inverse to RigCtrl_CIV_Server_MapDataIntoReadResponse(), case RIGCTRL_PN_[UN]SEL_VFO_OP_MODE ...
                     { case 1 : iFilterIndex = 0;  // "FIL1" (normal filter bandwidth)
                                break;
                       case 2 : iNewSetting |= RIGCTRL_OPMODE_NARROW;      // "FIL2" (usually narrower, e.g. 500 Hz in CW)
                                iFilterIndex = 1;
                                break;
                       case 3 : iNewSetting |= RIGCTRL_OPMODE_VERY_NARROW; // "FIL3" (usually very narrow, e.g. 250 Hz in CW)
                                iFilterIndex = 2;
                                break;
                       default: iFilterIndex = 0;
                                break;
                     }
                    SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, ",FIL%d", (int)iFilterIndex+1 ); // ONE-based "number" for the display, ZERO-based "index" internally
                    SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, ",Data=%d", (int)iDataMode );
                  } // end if < radio also reported the current "Data Mode" and FILTER SETTING >
                 switch( iSubcode )  // the "sub-command" after 0x26 is the ZERO-based VFO index:
                  { case 0x00 : // received the "mode" for the SELECTED VFO ("main VFO")
                       // Only accept the radio's reported value if we're not going to SEND a NEW operating mode:
                       if( ! ( pRC->iPostponedSetMessages & (1<<RIGCTRL_MSGTYPE_OP_MODE) ) )
                        {
                          if( iNewSetting != pRC->iOpMode )
                           { pRC->iOpMode = iNewSetting;  // ToDo: Inform the GUI about the change ?
                           }
                          pRC->iSelVfoDataMode = iDataMode;
                          // Added 2025-08-02: WSJT-X periodically asked for the "Selected VFO Mode",
                          //                   then asked for the "Filter Bandwidth".
                          // RCW Keyer only forwarded ONE of the two messages to the radio,
                          // and responded with a wrong filter bandwidth on a SERVER PORT:
                          // > r2 007 FE FE 94 E0 26 00 FD    ; SelVFOMode ?
                          // > TX 007 FE FE 94 E0 26 00 FD    ; SelVFOMode ?
                          // > RX 00A FE FE E0 94 26 00 01 .. ; SelVFO=USB,FIL1,Data=0
                          // > t2 00A FE FE E0 94 26 00 01 .. ; SelVFO=USB,FIL1,Data=0
                          // > r2 007 FE FE 94 E0 1A 03 FD    ; FilterBW ?
                          // > t2 008 FE FE E0 94 1A 03 09 FD ; FilterBW 500 Hz
                          // ,-------------------------------------------|_|
                          // '--> Wrong answer (not forwarded but taken from pRC->iFilterBW_Hz) .
                          //      Fixed by the following:
                          if( pRC->iFilterBW_Hz != RIGCTRL_NOVALUE_INT )
                           { // If this parameter WAS already valid, THEN update it:
                             pRC->iFilterBW_Hz = RigCtrl_GetAudioFilterBandwidth( pRC, iFilterIndex, pRC->iOpMode );
                             // '--> for example, got here after WSJT-X docked on,
                             //      with pRC->iFilterBW_Hz=3000 BEFORE,
                             //      and  pRC->iFilterBW_Hz=1200 Hz AFTER the above.
                             // (WSJT-X had switched the op-mode to "CW" (BUG in Hamlib?),
                             //  for reasons we will never know.
                             //  Be prepared for more suprises like these.)
                           }
                        }
                       break;
                    case 0x01 : // received the "mode" for the UNSELECTTED VFO ("sub VFO")
                       if( iNewSetting != pRC->iUnselVfoOpMode )
                        { pRC->iUnselVfoOpMode = iNewSetting; // ToDo: Inform the GUI about the change ?
                        }
                       break;
                  } // end switch( iSubcode )  // "sub-command" after 0x26 ?
               }
              else // message too short for a "set"-command, so guess it's "get" (read command):
               { SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "%s ?", pszToken );  // -> "UnselVFOMode",
                 // shelled out very frequently by WFView, maybe for SPLIT mode fans ?
                 if( !fIsResponse )  // NOT a response ? Guess it's a READ-COMMAND:
                  { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_CMD );
                  }
               }
            } // end if < sufficient message length for this command ? >
           break; // end case 0x26

        case 0x27: // Read "Spectrum Scope" waveform data :
           if( iMsgLength >= 7 )  // 0x27 followed by at least ONE byte before the postamble:
            { // (RS-BA1 periodically sent > r1 007 FE FE 94 E0 27 1C FD,
              //       i.e. NO DATA FIELD but a "query". So beware.. )
              //
              if( iRigCtrlPort==RIGCTRL_PORT_RADIO )
               { if( pPortInstance->dwExpectedResponse_CmdAndSubcode == RCTL_COMBINE_CMD_AND_SUB(0x27,pbMessage[5]) )
                  { // RigCtrl_Handler() has been waiting for THIS in case RIGCTRL_POLLSTATE_CONFIGURE_SPECTRUM ...
                    pPortInstance->iResponseCountdown_ms = 0;   // don't wait any longer !
                    pPortInstance->dwExpectedResponse_CmdAndSubcode = RIGCTRL_NOVALUE_INT; // avoid misinterpreting the next unspecific "OK" or "NOT OK".
                    // (only RigCtrl_Handler() decides what to ask NEXT)
                  } // end if < 0x27.. was the EXPECTED response for something >
               } // end if < message received FROM THE RADIO (not from any of the clients) > ?
              RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_BASIC, RIGCTRL_MSGTYPE_SPECTRUM_CONFIG ); // initial guess: spectrum CONTROL (not SPECTRUM DATA)
              iSubcode = pbMessage[5];
              switch( iSubcode )  // "sub-command" after 0x27 ?
               {  case 0x00 :  // Guess what follows is a block of 'Scope waveform data', in this case: part of a SPECTRUM.
                    RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_BASIC, RIGCTRL_MSGTYPE_SPECTRUM ); // not spectrum-CONTROL but -DATA !
                    fUnknown = FALSE;
                    fIsUnsolicitedMsg = TRUE;
                    pRC->iScopeDataCountdown_ms = RIGCTRL_SPECTRUM_TIMEOUT_MS; // reload 'Scope waveform' timer (for timeout monitoring) on SCOPE DATA RECEPTION
                    // CAN we, and do we WANT TO to receive spectra at all (configurable) ?
                    if( (pRC->dwRigControlFlags & RIGCTRL_FLAG_WANT_SPECTRUM_DATA ) && (pRC->iCapabilities & RIGCTRL_CAPS_SPECTRUM_VIA_CAT) )
                     { // Yes... decode the scope data for RCWK's own spectrum display, but here's a catch:
                       //        When e.g. the CI-V stream travels from one SERIAL PORT TUNNEL to another,
                       //        the data must only be parsed (and added to the spectrum buffer)
                       //        on ONE port, not on the other. Fixed in 2025-10-06 as follows:
                       //  Only pass the CI-V message to RigCtrl_ParseSpectrumData_CIV()
                       //  if the message was RECEIVED on this port, not if it was SENT from here:
                       if( (iRigCtrlOrigin==RIGCTRL_ORIGIN_RADIO) || (iRigCtrlOrigin==RIGCTRL_ORIGIN_COM_PORT_RX) )
                        { RigCtrl_ParseSpectrumData_CIV( pRC, pbMessage+6/*payload*/, iMsgLength-7/*payloadLength*/,
                                                         &pszCmtDest, cpCmtEndstop );
                        }
                       // See ASCII graphics in RigCtrl_ParseSpectrumData_CIV()
                       // about what is considered the 'payload' for a SPECTRUM !
                     } // end if < ..ControlFlags & RIGCTRL_FLAG_WANT_SPECTRUM_DATA >
                    else // option RIGCTRL_FLAG_WANT_SPECTRUM_DATA *not* set :
                     { // Turn off the unsolicited transmission of scope data.
                       // This reduces the 'bus load', so other messages
                       // (like VFO frequency control) have a smaller latency.
                       // This also stops flooding the CAT traffic monitor .
                       RigCtrl_SetPostponedSetMessageFlag( pRC, RIGCTRL_MSGTYPE_SPECTRUM_CONFIG);
                       // The rest happens in RigCtrl_Handler(): As soon as
                       // there's an empty slot (in the heavy bus traffic),
                       // it will call RigCtrl_StartStopSpectrumData( pRC, TRUE ).
                     }
                    break;

                 case 0x10:  // "Send/read the Scope ON/OFF status (00=OFF, 01=ON)"
                    // If the scope (on the Icom's own display) is OFF,
                    // the radio also won't send "waveform data" via CI-V !
                    break;

                 case 0x11:  // "Send/read the Scope wave data output setting (00=OFF, 01=ON)"
                    // (should get here after RigCtrl_StartStopSpectrumData() )
                    // When "RS-BA1" connected through client port #1, it sent:
                    // > 21:22:57.3 r1 008 FE FE 94 E0 27 11 00 FD
                    //                                       |__ 0x00 = off, 0x01 = on
                    // No byte to distinguish between "main scope" and "sub scope" here !
                    fUnknown = FALSE;
                    if( iMsgLength < 8 ) // look mum, no DATA .. must be a QUERY .
                     { SL_AppendString( &pszCmtDest, cpCmtEndstop, "Scope data on ?" );
                       if( !fIsResponse )  // NOT a response ? Guess it's a READ-COMMAND:
                        { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_CMD );
                        }
                     }
                    else // message WITH DATA:
                     { if( pbMessage[6] == 0x00 )
                        { SL_AppendString( &pszCmtDest, cpCmtEndstop, "Scope data off" );
                          pRC->iScopeMode = RIGCTRL_SCOPE_MODE_OFF;
                          // When terminating RS-BA1 (connected to an IC-7300 through SL's client port),
                          //       it STOPPED the periodic transmission of scope waveform data :
                          // > 20:08:53.1 r1 008 FE FE 94 E0 27 11 00 FD (scope) ; Scope data off (from client)
                          // > 20:08:53.1 TX 008 FE FE 94 E0 27 11 00 FD (scope) ; Scope data off (forwarding to radio)
                          // > 20:08:53.2 RX 006 FE FE E0 94 FB FD       (OK)    ; response from radio
                          // > 20:08:53.2 t1 006 FE FE E0 94 FB FD       (OK)    ; forwarding response to client
                          // The TIMEOUT MONITOR for spectrum data in RigCtrl_Handler()
                          // will automatically try to kick the scope alive again.
                        }
                       else
                        { SL_AppendString( &pszCmtDest, cpCmtEndstop, "Scope data on" );
                          if( pRC->iScopeMode == RIGCTRL_SCOPE_MODE_OFF )
                           { // cannot be "OFF" not, so assume "fixed frequency" mode:
                             pRC->iScopeMode = RIGCTRL_SCOPE_MODE_FIXED;
                           }
                          // Note: RIGCTRL_SCOPE_MODE_CENTER / FIXED set further below
                        }
                       if( !fIsResponse ) // it's not a RESPONSE, but it has a DATA FIELD, so guess it's a WRITE-COMMAND :
                        { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_WRITE_CMD );
                        }
                     }
                    break;

                 case 0x12:  // "Send/read the Scope MAIN/SUB setting (00=Fixed to MAIN)"
                 case 0x13:  // "Send/read the Scope Single/Dual setting (00=Fixed to Single)"
                    if( iMsgLength < 9 ) // again, no DATA .. must be a QUERY .
                     { if( !fIsResponse )  // NOT a response ? Guess it's a READ-COMMAND:
                        { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_CMD );
                        }
                     }
                    else
                     {
                       if( !fIsResponse ) // it's not a RESPONSE, but it has a DATA FIELD, so guess it's a WRITE-COMMAND :
                        { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_WRITE_CMD );
                        }
                     }
                    break;

                 case 0x14:  // "Send/read the Scope Center mode or Fixed mode setting"
                    // pbMessage[6] : 0x00 = main scope,  0x01 = sub scope (IC-7851) ; IC-7300 : "fixed" 0x00
                    // pbMessage[7] : 0x00 = center mode, 0x01 = fixed mode
                    // (looks like in contrast to Yaesu, Icom cares for compatibility .. in most cases)
                    if( iMsgLength < 9 ) // again, no DATA .. must be a QUERY .
                     { SL_AppendString( &pszCmtDest, cpCmtEndstop, "Scope C/F ?" );
                       if( !fIsResponse )  // NOT a response ? Guess it's a READ-COMMAND:
                        { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_CMD );
                        }
                     }
                    else
                     {
                       if( pbMessage[6] == 0x00 ) // only care for the "main scope" here...
                        { if( pbMessage[7] == 0x00 )
                           { pRC->iScopeMode = RIGCTRL_SCOPE_MODE_CENTER;
                             SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "Scope%cCenter", cGetOrSetIndicator );
                           }
                          else
                           { pRC->iScopeMode = RIGCTRL_SCOPE_MODE_FIXED;
                             SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "Scope%cFixed", cGetOrSetIndicator );
                           }
                        }
                       if( !fIsResponse ) // it's not a RESPONSE, but it has a DATA FIELD, so guess it's a WRITE-COMMAND :
                        { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_WRITE_CMD );
                        }
                     }
                    break;

                 case 0x15:  // "Send/read the span setting in the Center mode Scope"
                    //        pbMesssage[0  1  2  3  4  5  6  7]
                    // RS-BA1: > r1 008 FE FE 94 E0 27 15 00 FD (scope) ; span ?
                    //               |                    |
                    //           iMsgLength               0=main scope, 1=sub scope
                    if( iMsgLength <= 8 ) // no DATA .. must be a QUERY .
                     { SL_AppendString( &pszCmtDest, cpCmtEndstop, "Scope span ?" );
                       if( !fIsResponse )  // NOT a response ? Guess it's a READ-COMMAND:
                        { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_CMD );
                        }
                     }
                    else  // iMsgLength > 8 -> not a query ("get") but a SET-command :
                     { // Response from IC-7300 :
                       //     pbMesssage[0  1  2  3  4  5  6  7  8  9 10 11 12]
                       //      > RX 00D FE FE E0 94 27 15 00 00 25 00 00 00 FD (scope) ; span=2500 Hz
                       //            |                       |____________|
                       //        iMsgLength=13               BCD value, HERE: LEAST significant digits first !
                       if( pbMessage[6] == 0x00 ) // only care for the "main scope" here...
                        { pRC->dblScopeSpan_Hz = RigCtrl_ParseBCD_CIV_LSByteFirst_Double( pbMessage+7,
                               /* number of digits: */ 2*(iMsgLength/*13*/-8) );
                          SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "ScopeSpan:%ld Hz", (long)pRC->dblScopeSpan_Hz );
                        }
                       if( !fIsResponse ) // it's not a RESPONSE, but it has a DATA FIELD, so guess it's a WRITE-COMMAND :
                        { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_WRITE_CMD );
                        }
                     }
                    break;

                 case 0x16:  // "Edge number setting in the Fixed mode Scope"
                    if( iMsgLength <= 8 ) // no DATA .. must be a QUERY .
                     { SL_AppendString( &pszCmtDest, cpCmtEndstop, "Scope edge# ?" );
                       if( !fIsResponse )  // NOT a response ? Guess it's a READ-COMMAND:
                        { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_CMD );
                        }
                     }
                    else // message WITH DATA -> READ-RESPONSE or WRITE-COMMAND :
                     { if( fIsResponse )  // Response WITH DATA ? Guess it's a READ-RESPONSE, so separate name and value by '='
                        { cGetOrSetIndicator = '=';
                          RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_RESP );
                        }
                       else
                        { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_WRITE_CMD );
                        }
                       if( pbMessage[6] == 0x00 ) // only care for the "main scope" here...
                        { pRC->iScopeEdgeNumber = pbMessage[7]; // range : 1..3 (per BAND.. what a strange concept)
                          SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "ScopeEdge%c%d", cGetOrSetIndicator, pRC->iScopeEdgeNumber );
                        }
                     }
                    break;

                 case 0x17:  // "Send/read the Scope hold function status"
                    if( iMsgLength <= 8 ) // no DATA .. must be a QUERY .
                     { SL_AppendString( &pszCmtDest, cpCmtEndstop, "Scope hold ?" );
                       if( !fIsResponse )  // NOT a response ? Guess it's a READ-COMMAND:
                        { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_CMD );
                        }
                     }
                    else // message WITH DATA -> READ-RESPONSE or WRITE-COMMAND :
                     { if( fIsResponse )  // Response WITH DATA ? Guess it's a READ-RESPONSE, so separate name and value by '='
                        { cGetOrSetIndicator = '=';
                          RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_RESP );
                        }
                       else
                        { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_WRITE_CMD );
                        }
                       if( pbMessage[6] == 0x00 ) // only care for the "main scope" here...
                        { pRC->iScopeHoldFlag = pbMessage[7]; // 0=hold OFF, 1=hold ON
                          SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "ScopeHold%c%d",cGetOrSetIndicator, pRC->iScopeHoldFlag );
                        }
                     }
                    break;

                 case 0x18:  // "Send/read the Scope Attenuator setting" (IC-7851, n/a on IC-7300)
                    if( iMsgLength <= 8 ) // no DATA .. must be a QUERY .
                     { SL_AppendString( &pszCmtDest, cpCmtEndstop, "Scope att ?" );
                       if( !fIsResponse )  // NOT a response ? Guess it's a READ-COMMAND:
                        { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_CMD );
                        }
                     }
                    else // message WITH DATA -> READ-RESPONSE or WRITE-COMMAND :
                     { if( fIsResponse )  // Response WITH DATA ? Guess it's a READ-RESPONSE, so separate name and value by '='
                        { cGetOrSetIndicator = '=';
                          RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_RESP );
                        }
                       else
                        { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_WRITE_CMD );
                        }
                       if( pbMessage[6] == 0x00 ) // only care for the "main scope" here...
                        { pRC->dblScopeAttenuator_dB = RigCtrl_ParseBCD_CIV_LSByteFirst_Double( pbMessage+7,
                               /* number of digits: */ 2*(iMsgLength-8) );
                          SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "ScopeAtt%c%.1lf dB",cGetOrSetIndicator, pRC->dblScopeAttenuator_dB );
                        }
                     }
                    break;

                 case 0x19:  // "Send/read the Scope Reference level setting"
                    if( iMsgLength <= 8 ) // no DATA .. must be a QUERY from someone else.
                     { SL_AppendString( &pszCmtDest, cpCmtEndstop, "Scope ref ?" );
                       if( !fIsResponse )  // NOT a response ? Guess it's a READ-COMMAND:
                        { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_CMD );
                        }
                     }
                    else // message WITH DATA -> READ-RESPONSE or WRITE-COMMAND :
                     { if( fIsResponse )  // Response WITH DATA ? Guess it's a READ-RESPONSE, so separate name and value by '='
                        { cGetOrSetIndicator = '=';
                          RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_RESP );
                        }
                       else
                        { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_WRITE_CMD );
                        }
                       // Response seen from an IC-7300 :
                       // pbMessage[0  1  2  3  4  5  6  7  8  9 10]
                       // > RX 00B FE FE E0 94 27 19 00 10 50 00 FD   (scope)    ; ref=+10.5 dB
                       if( pbMessage[6] == 0x00 ) // only care for the "main scope" here...
                        { pRC->dblScopeRefLevel_dB = 0.01f * (float)RigCtrl_ParseBCD_CIV_MSByteFirst( pbMessage+7, 4/*digits*/ );
                          if( pbMessage[9] == 0x01 )  // 0x00 = plus, 0x01 = minus
                           { pRC->dblScopeRefLevel_dB = -pRC->dblScopeRefLevel_dB;
                           }
                          SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "ScopeRef%c%.1lf dB",cGetOrSetIndicator, pRC->dblScopeRefLevel_dB );
                        }
                     }
                    break;

                 case 0x1A: // "Send/read the Sweep speed setting"
                    if( iMsgLength <= 8 ) // no DATA .. must be a QUERY .
                     { SL_AppendString( &pszCmtDest, cpCmtEndstop, "Scope speed ?" );
                       // Seen from RS-BA1 :
                       //  > r1 008 FE FE 94 E0 27 1A 00 FD (scope)    ; Scope speed ?
                       if( !fIsResponse )  // NOT a response ? Guess it's a READ-COMMAND:
                        { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_CMD );
                        }
                     }
                    else // message WITH DATA -> READ-RESPONSE or WRITE-COMMAND :
                     { if( fIsResponse )  // Response WITH DATA ? Guess it's a READ-RESPONSE, so separate name and value by '='
                        { cGetOrSetIndicator = '=';
                          RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_RESP );
                        }
                       else
                        { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_WRITE_CMD );
                        }
                       // Response seen from an IC-7300 :
                       // pbMessage[0  1  2  3  4  5  6  7  8  9 10]
                       //  > RX 009 FE FE E0 94 27 1A 00 00 FD  (scope) ; Speed:FAST
                       if( pbMessage[6] == 0x00 ) // only care for the "main scope" here...
                        { switch( pbMessage[7] )  // 0x00 = fast, 0x01 = mid, 0x02 = slow
                           { case 0x00 : pRC->iScopeSpeed = RIGCTRL_SCOPE_SPEED_FAST;
                                  SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "Scope%cFAST", cGetOrSetIndicator );
                                  break;
                             case 0x01 : pRC->iScopeSpeed = RIGCTRL_SCOPE_SPEED_MID;
                                  SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "Scope%cMID", cGetOrSetIndicator );
                                  break;
                             case 0x02 : pRC->iScopeSpeed = RIGCTRL_SCOPE_SPEED_SLOW;
                                  SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "Scope%cSLOW", cGetOrSetIndicator );
                                  break;
                             default   : // ???
                                  break;
                           }
                        }
                     }
                    break;

                 case 0x1B: // "Send/read the Scope indication during TX in the Center mode" (0=off, 1=on)
                    if( iMsgLength <= 7/*!*/ ) // no DATA .. must be a QUERY .
                     { // Seen from RS-BA1 : > r1 007 FE FE 94 E0 27 1B FD
                       SL_AppendString( &pszCmtDest, cpCmtEndstop, "Scope on TX ?" );
                       if( !fIsResponse )  // NOT a response ? Guess it's a READ-COMMAND:
                        { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_CMD );
                        }
                     }
                    else // message WITH DATA -> READ-RESPONSE or WRITE-COMMAND :
                     { if( fIsResponse )  // Response WITH DATA ? Guess it's a READ-RESPONSE, so separate name and value by '='
                        { cGetOrSetIndicator = '=';
                          RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_RESP );
                        }
                       else
                        { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_WRITE_CMD );
                        }
                       // Response seen from an IC-7300 :
                       // pbMessage[0  1  2  3  4  5  6  7  8  9 10]
                       //  > RX 008 FE FE E0 94 27 1B 00 FD
                       //                             |
                       //                          Here, this isn't the "main/sub"-scope-flag
                       //                          but the OFF/ON status !
                       SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "Scope on TX%c%s", cGetOrSetIndicator, (pbMessage[6]==0) ? "OFF":"ON" );
                     }
                    break;

                 case 0x1C: // "Send/read scope center frequency setting in the Center mode"
                    // 00=Filter center, 01=Carrier point center, 02=Carrier point center (Abs. Freq.))
                    // Seen shortly after launching RS-BA1 :
                    //          [0  1  2  3  4  5  6  7]
                    // > TX 007 FE FE 94 E0 27 1C FD    (scope) ; CenterMode ?
                    // > RX 008 FE FE E0 94 27 1C 01 FD (scope) ; CenterMode=1
                    if( iMsgLength <= 7 ) // no DATA .. must be a QUERY .
                     { SL_AppendString( &pszCmtDest, cpCmtEndstop, "CenterType ?" );
                       if( !fIsResponse )  // NOT a response ? Guess it's a READ-COMMAND:
                        { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_CMD );
                        }
                     }
                    else // message WITH DATA -> READ-RESPONSE or WRITE-COMMAND :
                     { if( fIsResponse )  // Response WITH DATA ? Guess it's a READ-RESPONSE, so separate name and value by '='
                        { cGetOrSetIndicator = '=';
                          RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_RESP );
                        }
                       else
                        { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_WRITE_CMD );
                        }
                       pRC->iScopeCenterType = pbMessage[7]; // 0=RIGCTRL_SCOPE_CM_FILTER_CENTER / 1=CARRIER_POINT / 2=CARRIER_ABS (whatever that means)
                       SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "CenterType=%d", pRC->iScopeCenterType );
                     }
                    break;

                 case 0x1D: // "Send/read the Scope 'Video Band Width' setting" (narrow/wide)
                    // Seen shortly after launching RS-BA1 :
                    //          [0  1  2  3  4  5  6  7  8]
                    // > TX 008 FE FE 94 E0 27 1D 00 FD    (scope) ; CenterMode ?
                    // > RX 009 FE FE E0 94 27 1D 00 00 FD (scope) ; CenterMode=1
                    //                             |  |__ 0=narrow, 1=wide
                    //                             0=main scope, 1=sub scope
                    if( iMsgLength <= 7 ) // no DATA .. must be a QUERY .
                     { SL_AppendString( &pszCmtDest, cpCmtEndstop, "VideoBW ?" );
                       if( !fIsResponse )  // NOT a response ? Guess it's a READ-COMMAND:
                        { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_CMD );
                        }
                     }
                    else // message WITH DATA -> READ-RESPONSE or WRITE-COMMAND :
                     { if( fIsResponse )  // Response WITH DATA ? Guess it's a READ-RESPONSE, so separate name and value by '='
                        { cGetOrSetIndicator = '=';
                          RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_RESP );
                        }
                       else
                        { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_WRITE_CMD );
                        }
                       if( pbMessage[6] == 0x00 ) // only care for the "main scope" here...
                        { pRC->iScopeVideoBW = pbMessage[7]; // 0=RIGCTRL_SCOPE_VBW_NARROR, 1=.._WIDE
                          SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "VideoBW=%d", pRC->iScopeVideoBW );
                        }
                     }
                    break;

                 case 0x1E: // "Send/read the Scope Fixed edge frequencies"
                    // There may be up to 12 selectable frequency ranges.
                    // Each range has THREE "edge numbers" (already in pRC->iScopeEdgeNumber).
                    // The command (read-response) contains RANGE, EDGE NUMBER,
                    //     and the two BCD-encoded edge frequencies (lower, higher).
                    // Which of the 12 frequency ranges is currenty in use, depends on the VFO frequency.
                    // To avoid having to read this repeatedly, only read the
                    // entire "table" ONCE, and store it in an array.
                    // Format :   (1) (2) (3) (4) (5) (6) (7) (8) (9) (10)(11)(12)
                    //           |X:X|0:X|X:X|X:X|X:X|X:X|0:0|X:X|X:X|X:X|X:X|0:0|
                    // range nr ---'   |   |   |   |   |   | | |   |   |   |   |
                    // edge nr (1..3)--' |<--- lower edge -->|<-- higher edge -->|
                    //                     |   |   |   |   | | |   |   |   |   |
                    //                    10+1 .........1GHZ |10+1 ...........1GHz
                    //                     Hz            100M| Hz              100MHz
                    // (expect those ZEROs in the 100 MHz- and 1 GHz to be used
                    //  sooner or later .. thinking about an IC-9700 with V/U/SHF)
                    // Icom's RS-BA1 reads out ALL these frequencies when starting:
                    // > r1 009 FE FE 94 E0 27 1E 01 01 FD  (scope)    ; 16..0 kHz
                    iRange = RigCtrl_ParseBCD_CIV_LSByteFirst_Int32( pbMessage+6, 2/*digits*/ );
                    iEdge  = RigCtrl_ParseBCD_CIV_LSByteFirst_Int32( pbMessage+7, 2/*digits*/ );
                    if( iMsgLength <= 9/*!*/ ) // no DATA, only RANGE- and EDGE-number .. must be a QUERY .
                     { SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "rng[%d],edg[%d] ?",(int)iRange, (int)iEdge );
                       if( !fIsResponse )  // NOT a response ? Guess it's a READ-COMMAND:
                        { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_CMD );
                        }
                     }
                    else // message WITH DATA -> READ-RESPONSE or WRITE-COMMAND :
                     { if( fIsResponse )  // Response WITH DATA ? Guess it's a READ-RESPONSE, so separate name and value by '='
                        { cGetOrSetIndicator = '=';
                          RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_RESP );
                        }
                       else
                        { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_WRITE_CMD );
                        }
                       iRange = RigCtrl_ParseBCD_CIV_LSByteFirst_Int32( pbMessage+6, 2/*digits*/ );
                       iEdge  = RigCtrl_ParseBCD_CIV_LSByteFirst_Int32( pbMessage+7, 2/*digits*/ );
                       if( (iRange>=1) && (iRange<=RIGCTRL_MAX_FIXED_EDGE_SCOPE_FREQ_RANGES)
                         &&(iEdge >=1) && (iEdge <=RIGCTRL_MAX_FIXED_EDGES_PER_FREQ_RANGE) )
                        { pRC->ScopeFreqRange[iRange-1][iEdge-1].dblFmin_Hz = RigCtrl_ParseBCD_CIV_LSByteFirst_Double( pbMessage+8, 10/*digits*/ );
                          pRC->ScopeFreqRange[iRange-1][iEdge-1].dblFmax_Hz = RigCtrl_ParseBCD_CIV_LSByteFirst_Double( pbMessage+13,10/*digits*/ );
                          SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "%ld..%ld kHz",
                             (long)(1e-3 * pRC->ScopeFreqRange[iRange-1][iEdge-1].dblFmin_Hz),
                             (long)(1e-3 * pRC->ScopeFreqRange[iRange-1][iEdge-1].dblFmax_Hz) );
                        }
                     }
                    break; // end case 0x27 0x1E ("Scope Fixed edge frequencies")

                 default : // purpose unknown ... future extensions ?
                    if( iMsgLength < 9 ) // again, no DATA .. must be a QUERY .
                     { if( !fIsResponse )  // NOT a response ? Guess it's a READ-COMMAND:
                        { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_CMD );
                        }
                     }
                    else // message WITH DATA -> READ-RESPONSE or WRITE-COMMAND :
                     { if( fIsResponse )  // Response WITH DATA ? Guess it's a READ-RESPONSE, so separate name and value by '='
                        { cGetOrSetIndicator = '=';
                          RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_READ_RESP );
                        }
                       else
                        { RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP, RIGCTRL_MSGTYPE_FLAG_WRITE_CMD );
                        }
                     }
                    break;
               } // end switch < byte after 0x27 >
            } // end if < valid message length >
           break; // end case 0x27

        case 0x28: // Voice TX Memory (0=stop, 1 to 8 = memory #)
           RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_BASIC, RIGCTRL_MSGTYPE_OTHER );
           break; // end case 0x28

        case 0x29: // "directly specify main or sub band" (IC-7610)
           RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_BASIC, RIGCTRL_MSGTYPE_OTHER );
           break; // end case 0x29

        case 0xFA: // "NG" ("No-Good" or "FAilure" aka "NotOK"), indicates an error
           RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_BASIC, RIGCTRL_MSGTYPE_NOT_OK );
           if( iRigCtrlPort==RIGCTRL_PORT_RADIO ) // only care for "OK" / "Not OK" *FROM THE RADIO*...
            { pPortInstance->iResponseCountdown_ms = 0; // don't wait any longer (there was a response, even though a BAD one)
              // Will usually get here when asking for something not supported by the radio.
              // Example: Asking an IC-7300 for the current SCOPE ATTENUATOR LEVEL :
              // > 21:50:22.7 TX 008 FE FE 94 E0 27 18 00 FD   (scope)  ; atten. ?
              // > 21:50:22.9 RX 006 FE FE E0 94 FA FD         (Not OK)
              if( pPortInstance->dwExpectedResponse_CmdAndSubcode != (DWORD)RIGCTRL_NOVALUE_INT )
               { pPI = RigCtrl_GetInfoForCIVCommandAndSubcode(pPortInstance->dwExpectedResponse_CmdAndSubcode);
                 if( (pPI != NULL) && (pPI->pszToken != NULL) )
                  { SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "%s : NotOK", pPI->pszToken );
                    // 2025-07-19: Not correctly displayed in the TRAFFIC LOG:
                    // >      0011568  SubBandON : NotOK
                    // (no hex dump, just an error message, but this is NOT AN ERROR
                    //  but just a parameter that doesn't exist in the rig, e.g. in an IC-7300)
                    // Fixed by ...  ?
                    //
                  }
                 else
                  { SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "NotOK for cmd 0x%X", (int)pPortInstance->dwExpectedResponse_CmdAndSubcode );
                  }
                 // That's just the default comment. More human-friendly output possibly below. Thus ...
                 pszCmtDest = pszComment; // reset the "comment-building pointer" again
               }
              switch( pPortInstance->dwExpectedResponse_CmdAndSubcode ) // we know the answer ("NG"), but what was the question ?
               { case RCTL_COMBINE_CMD_ONLY(0x05) : // failed to "Set the operating frequency"
                    RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_BASIC, RIGCTRL_MSGTYPE_SET_FREQUENCY );
                    SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "refused to set %.1lf Hz", pRC->dblVfoFrequency );
                    if( pRC->iParameterPollingState != RIGCTRL_POLLSTATE_FIND_VFO_FOR_FREQ )
                     { // Let RigCtrl_FindVfoForNewFrequency() try a few tricks
                       // to accept the WANTED VFO frequency ( still in pRC->dblVfoFrequency )
                       pRC->iParameterPollingState = RIGCTRL_POLLSTATE_FIND_VFO_FOR_FREQ;
                       pRC->iParameterPollingSubState = 0;
                       // '--> the rest happens in RigCtrl_FindVfoForNewFrequency() .
                     }
                    break;
                 case RCTL_COMBINE_CMD_AND_SUB(0x27,0x14) : // have asked for the spectrum scope's "Center/Fixed" setting
                    RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_BASIC, RIGCTRL_MSGTYPE_SPECTRUM_CONFIG );
                    SL_AppendString( &pszCmtDest, cpCmtEndstop, "no scope-C/F" );
                    break;
                 case RCTL_COMBINE_CMD_AND_SUB(0x27,0x15) : // have asked for the spectrum scope's "Frequency Span"
                    RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_BASIC, RIGCTRL_MSGTYPE_SCOPE_SPAN );
                    SL_AppendString( &pszCmtDest, cpCmtEndstop, "no scope-span" );
                    break;
                 case RCTL_COMBINE_CMD_AND_SUB(0x27,0x16) : // have asked for the spectrum scope's "Edge Number"
                    RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_BASIC, RIGCTRL_MSGTYPE_SPECTRUM_CONFIG ); // <- 'some other spectrum-scope-related setting FAILED'
                    SL_AppendString( &pszCmtDest, cpCmtEndstop, "no scope edges" );
                    break;
                 case RCTL_COMBINE_CMD_AND_SUB(0x27,0x17) : // have asked for the "Scope hold function status"
                    RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_BASIC, RIGCTRL_MSGTYPE_SPECTRUM_CONFIG ); // <- 'some other spectrum-scope-related setting FAILED'
                    SL_AppendString( &pszCmtDest, cpCmtEndstop, "no scope-hold" );
                    break;
                 case RCTL_COMBINE_CMD_AND_SUB(0x27,0x18) : // have asked for the spectrum scope's "Attenuator Level",
                    // which for example the IC-7300 doesn't support.
                    RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_BASIC, RIGCTRL_MSGTYPE_SPECTRUM_CONFIG ); // <- 'some other spectrum-scope-related setting FAILED'
                    SL_AppendString( &pszCmtDest, cpCmtEndstop, "no scope-att" );
                    break;
                 case RCTL_COMBINE_CMD_AND_SUB(0x27,0x1E) : // have asked for one of the scope's "frequency ranges":
                    // When deliberately exceeding the number of "scope frequency ranges"
                    //  ( > 13 for an IC-7300 ), the radio responded with "Not Good" :
                    // > 23:23:00.0 TX 009 FE FE 94 E0 27 1E 14 01 FD (scope)    ; edge[14][1] ?
                    // > 23:23:00.1 RX 006 FE FE E0 94 FA FD          (Not OK)
                    // If this happens, limit pRC->iScopeNumFreqRanges
                    //   and/or pRC->iScopeNumEdgesPerFreqRange
                    //   to what the radio *really* supports. But WHAT caused the "Not OK" ?
                    RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_BASIC, RIGCTRL_MSGTYPE_SPECTRUM_CONFIG ); // <- 'some other spectrum-scope-related setting FAILED'
                    break; // end case "Not OK" for a SCOPE FREQUENCY RANGE (0x271E)
                 case RCTL_COMBINE_CMD_AND_SUB(0x1A,0x01) : // have asked for a "Band Stacking Register":
                    RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_BASIC, RIGCTRL_MSGTYPE_OTHER ); // <- here: "NOT OK" for "Band Stacking Register"
                    // Obviously, what Icom calls "Frequency band codes" (rig-specific)
                    // is invalid, so we have exceeded the range of valid "Frequency band codes".
                    // More on that in RigCtrl_Handler(), case RIGCTRL_POLLSTATE_BAND_STACKING_REGS .
                    if( pRC->iParameterPollingState == RIGCTRL_POLLSTATE_BAND_STACKING_REGS )
                     { RigCtrl_FinishReadingBandStackingRegs( pRC );
                       SL_AppendString( &pszCmtDest, cpCmtEndstop, "End of Band Stacking Registers" );
                     }
                    break;
                 case RCTL_COMBINE_CMD_AND_SUB(0x07,0xB0): // tried to "Exchange MAIN and SUB Bands"... but that failed
                    RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_BASIC, RIGCTRL_MSGTYPE_OTHER ); // <- here: "NOT OK" to exchange main/sub bands
                    SL_AppendString( &pszCmtDest, cpCmtEndstop, "failed to exchange MAIN/SUB bands" );
                    if( pRC->iParameterPollingState == RIGCTRL_POLLSTATE_FIND_VFO_FOR_FREQ )
                     { fShowInLog = FALSE; // already emit the message to the traffic log HERE,
                       // because RigCtrl_FindVfoForNewFrequency() may already send the next message:
                       RigCtrl_AddToTrafficLog( pRC, iRigCtrlPort, iRigCtrlOrigin, pbMessage, iMsgLength,
                          iMsgType | RIGCTRL_MSGTYPE_FLAG_RX | RIGCTRL_MSGTYPE_FLAG_EXPECTED, pszComment );
                       RigCtrl_FindVfoForNewFrequency( pRC, RIGCTRL_MSGTYPE_NOT_OK );
                       fIsExpectedResponse = FALSE; // don't enter the 'default handler' further below
                     }
                    break;
                 default: // no special case, so use the comment already prepared above
                    break;
              } // end switch( pPortInstance->dwExpectedResponse_CmdAndSubcode )
              pPortInstance->dwExpectedResponse_CmdAndSubcode = RIGCTRL_NOVALUE_INT; // after "Not OK" from the radio,
              // we're not expecting to receive the "wanted" parameter value or whatever.
              pPortInstance->iExpectResponseForUnifiedPN = -1; // here: cleared after receiving a "Not OK"
            } // end if < "Not OK" received FROM THE RADIO (not on one of the client ports) >
           break; // end case 0xFA < receive "No-Good" / "Not OK">

        case 0xFB: // "OK" (success, "Fine Business") from the rig
           RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_BASIC, RIGCTRL_MSGTYPE_OK );
           // Problem : The received CI-V message doesn't indicate WHAT was ok ("successfully WRITTEN").
           // Solution: Similar as already used for response "No-Good" (above):
           pPortInstance->iResponseCountdown_ms = 0; // don't wait any longer (the 0xFB-"OK" was a response)
           fIsExpectedResponse = TRUE;  // unspecific "OK" treated like an "expected response" !
           // 2025-08-26: WRITE-commands passing through a SERIAL PORT TUNNEL were correctly commented,
           //   but the WRITE RESPONSES (received from radio and passed on to client) were NOT:
           // > 5594 18:06:29.4 r2 00B FE FE 94 E0 05 00 64 03 07 00 FD ; VFOfreq:7036400.0 Hz
           // > 5595 18:06:29.4 t1 00B FE FE 94 E0 05 00 64 03 07 00 FD ; VFOfreq:7036400.0 Hz
           // > 5597 18:06:29.4 r1 006 FE FE E0 94 FB FD     <- missing e.g. ; Set VFOfreq: ok
           // > 5599 18:06:29.4 t2 006 FE FE E0 94 FB FD     <- missing e.g. ; Set VFOfreq: ok
           // For a better display in the TRAFFIC LOG, if the port instance is EXPECTING
           // an "FB" or "FA" reponse, or the "expected" command, subcommand, etc is known,
           // use that info to append a comment in the form "Set <token>: OK",
           // where <token> is retrieved from the CI-V command / subcommand / etc:
           if( pParamInfoFromExpectedUnifiedPN != NULL ) // PREFERRED method..
            { // but only works for WRITE-COMMANDS with a UNIFIED (non-Icom-specific) parameter number:
              if( pParamInfoFromExpectedUnifiedPN->pszToken != NULL )
               { SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "%s : OK",
                   pParamInfoFromExpectedUnifiedPN->pszToken );
               }
              else  // fallback for incomplete entries in RigCtrl_ParameterInfo[] :
               { SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "OK for anonymous parameter, PN=%d",
                   (int)pParamInfoFromExpectedUnifiedPN->iUnifiedPN );
               }
            }
           else if( // older 'fall-back', miserably fails with the crazy sub-subcodes for cmd 0x1A sub 0x05 :
               (pPortInstance->dwExpectedResponse_CmdAndSubcode != (DWORD)RIGCTRL_NOVALUE_INT )
            && (pPortInstance->dwExpectedResponse_CmdAndSubcode != RCTL_COMBINE_CMD_ONLY(0xFB) ) )
            { pPI = RigCtrl_GetInfoForCIVCommandAndSubcode(pPortInstance->dwExpectedResponse_CmdAndSubcode);
              // Example: If the "FB" applies to a previous WRITE COMMAND to set the VFO frequency
              //          (one of the most important commands for REMOTE CW OPERATION),
              //          pPortInstance->dwExpectedResponse_CmdAndSubcode = 0x05FFFFFF
              //   ,---------------------------------------------------------|_______|
              //   '--> "command" 0x05, no "sub command", no sub-sub[-sub] command,
              //
              if( (pPI != NULL) && (pPI->pszToken != NULL) )
               { SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "%s : OK", pPI->pszToken );
               }
              else  // fallback for incomplete entries in RigCtrl_ParameterInfo[] :
               { SL_AppendPrintf( &pszCmtDest, cpCmtEndstop, "OK for cmd 0x%X", (int)pPortInstance->dwExpectedResponse_CmdAndSubcode );
                 // 2024-01: Got here with pPortInstance->dwExpectedResponse_CmdAndSubcode = 0xFB000000,
                 //          which means "expecting 0xFB = OK" but that's not the intended purpose.
                 // Reason: After command 'Start Spectrum' (cmd=0x27 sub=0x11 data=0x01),
                 //         an IC-9700 responded with a SPECTRUM (0x27 0x00..), not with an "OK".
                 //         The "OK" (cmd=0xFB) arrived MUCH LATER, making the interpretation tricky.
               }
            }
           else  // received an "OK" (CI-V:0xFB) but no idea FOR WHAT ..
            { SL_AppendString( &pszCmtDest, cpCmtEndstop, "'OK'" );
            }
           pszCmtDest = pszComment; // reset the "comment-building pointer" again
           if( iRigCtrlPort==RIGCTRL_PORT_RADIO ) // "FB" received on the RADIO CONTROL PORT ?
            { // the "FB" (CI-V "OK") may cause a transition of pRC->iParameterPollingState :
              if( (pRC->iParameterPollingState == RIGCTRL_POLLSTATE_TURN_RIG_ON )
                ||(pRC->iParameterPollingState == RIGCTRL_POLLSTATE_WAIT_TURN_ON ) )
               { fIsExpectedResponse = FALSE; // this WAS the expected response" for "turn on",
                      // '--> but don't use the default processing further below,
                      //      which would CLEAR pPortInstance->iResponseCountdown_ms, etc.
                 pPortInstance->dwExpectedResponse_CmdAndSubcode = RIGCTRL_NOVALUE_INT; // avoid misinterpreting the NEXT unspecific "OK" or "NOT OK".
                 pPortInstance->iExpectResponseForUnifiedPN = -1; // here: cleared in RIGCTRL_POLLSTATE_TURN_RIG_ON, RIGCTRL_POLLSTATE_WAIT_TURN_ON
                 pPortInstance->iResponseCountdown_ms = 5000; // stay in RIGCTRL_POLLSTATE_WAIT_TURN_ON until the radio has booted up completely.
                 // Without this, an IC-7300 ignored the first few read-requests
                 // in RIGCTRL_POLLSTATE_PARAMS_FOR_RIGCTL :
                 // > 19:45:08.2 TX 09B FE FE FE FE FE FE FE .. ; turn ON
                 // > 19:45:08.3 RX 006 FE FE E0 94 FB FD       ; OK, turned on
                 // > 19:45:09.0 TX 007 FE FE 00 E0 19 00 FD    ; read RadioID
                 // > 19:45:11.0 TX 006 FE FE 00 E0 03 FD       ; read VFOfreq
                 // > 19:45:12.9 TX 007 FE FE 00 E0 1C 00 FD ; read Transmitting
                 // > 19:45:13.0 RX 008 FE FE E0 94 1C 00 00 FD ; Transmitting:0
                 //          '--> That's 13-8 = FIVE SECONDS after the "OK, turned on"
                 //               Obviously, the "FB" response for the "turn on"-command
                 //               was not sent by the main CPU, but from some kind
                 //               of 'standby controller' or whatever can communicate
                 //               via CI-V (and USB, or LAN, or WLAN) while the
                 //               main CPU is asleep, or busy from "booting" .
               } // end if( pRC->iParameterPollingState == RIGCTRL_POLLSTATE_WAIT_TURN_ON )
              else // not the special casse for RIGCTRL_POLLSTATE_WAIT_TURN_ON/RIGCTRL_POLLSTATE_WAIT_TURN_ON :
              switch( pPortInstance->dwExpectedResponse_CmdAndSubcode )
               { case RCTL_COMBINE_CMD_ONLY(0x05): // rig has accepted the NEW VFO frequency ->
                    if( pRC->iParameterPollingState == RIGCTRL_POLLSTATE_FIND_VFO_FOR_FREQ )
                     { fShowInLog = FALSE; // already emit the message to the traffic log HERE,
                       // because RigCtrl_FindVfoForNewFrequency() may already send the next message:
                       RigCtrl_AddToTrafficLog( pRC, iRigCtrlPort, iRigCtrlOrigin, pbMessage, iMsgLength,
                          iMsgType | RIGCTRL_MSGTYPE_FLAG_RX | RIGCTRL_MSGTYPE_FLAG_EXPECTED, pszComment );
                       RigCtrl_FindVfoForNewFrequency( pRC, RIGCTRL_MSGTYPE_OK ); // -> exit from RIGCTRL_POLLSTATE_FIND_VFO_FOR_FREQ
                       fIsExpectedResponse = FALSE; // don't enter the 'default handler' further below
                     }
                    break;
                 case RCTL_COMBINE_CMD_AND_SUB(0x07,0xB0): // rig agreed to "Exchange MAIN and SUB Bands" ->
                     { fShowInLog = FALSE; // already emit the message to the traffic log HERE,
                       // because RigCtrl_FindVfoForNewFrequency() may already send the next message:
                       RigCtrl_AddToTrafficLog( pRC, iRigCtrlPort, iRigCtrlOrigin, pbMessage, iMsgLength,
                          iMsgType | RIGCTRL_MSGTYPE_FLAG_RX | RIGCTRL_MSGTYPE_FLAG_EXPECTED, pszComment );
                       RigCtrl_FindVfoForNewFrequency( pRC, RIGCTRL_MSGTYPE_OK ); // -> try to set the VFO frequency again
                       fIsExpectedResponse = FALSE; // don't enter the 'default handler' further below
                     }
                    break;
                 default:
                    fShowInLog = fShowInLog; // <- (place for a breakpoint) .. leave fShowInLog unchanged,
                                             //    so this response can be displayed way further below.
                    break;
               } // end switch( pPortInstance->dwExpectedResponse_CmdAndSubcode ) after receiving 0xFB ("Fine Business"?)
            }      // end if < "FB" received on iRigCtrlPort==RIGCTRL_PORT_RADIO > ?
           break; // end case 0xFB ("Fine Business", "OK")

        case 0xFC: // Old CI-V manuals mention 0xFC ("FCH") as the 'Jammer code',
           // for example Icom's "CI-V Reference Manual ver 3.2" from 2002 [CIV_REF_MAN_V3_2]:
           // > The jammer code, FCH, prevents a message collision among radios and controller.
           // > During message transmission, a radio which is transmitting receives
           // > a transmitted message from itself to detect a message collision.
           // > If a message collision with another radio is detected, the radio
           // > halts message transmission, and checks that no other messages
           // > are transmitted on the CI-V bus line.
           // > When no other message is transmitted, the radio transmits the
           // > jammer code, 5 times as below.
           // >    ,----,----,----,----,----,
           // >    | FC | FC | FC | FC | FC |
           // >    '----'----'----'----'----'
           break; // end case 0xFC

#  endif // TEST to omit a few thousand lines, looking for imbalance curly braces .. AAAAARGH !
           

        default:  // anything else is ignored here (but still can make it to the log,
           // and may be 'answered' if a 'parameter info' was been found (pPI!=NULL).
           RigCtrl_ModifyMsgType( &iMsgType, RIGCTRL_MSGMASK_BASIC, RIGCTRL_MSGTYPE_UNKNOWN ); // this parser couldn't process the message !
           break;
      } // end switch < command code, pbMessage[4] >

#   ifdef __BORLANDC__  // "... is assigned a value that is never used".. oh shut up, we know what we're doing !
     (void)pszToken;
#   endif

     if( (iUnifiedPN != RIGCTRL_PN_UNKNOWN) && (pPI==NULL) ) // if not already set in the endless switch/case above.. (*)
      { pPI = RigCtrl_GetInfoForUnifiedParameterNumber(iUnifiedPN); // .. then retrieve the "parameter info" HERE
        // (*) in most cases, pPI is already nonzero to generate the 'comment' for the traffic log
      }

     if( pPI != NULL ) // common treatment to set fIsExpectedResponse *IF NOT SET ABOVE ALREADY* :
      { if((pPI->iUnifiedPN != RIGCTRL_PN_UNKNOWN )
         &&(pPI->iUnifiedPN == pPortInstance->iExpectResponseForUnifiedPN ) )
         { fIsExpectedResponse = TRUE;  // .. even the response may have been an unspecific "0xFA" / "0xFB" a la CI-V
         }
      }

     if( fIsExpectedResponse ) // "default handler" if this was the "expected response" :
      { // RigCtrl_Handler() (or a remote client) has been waiting for THIS message..
        if( iRigCtrlPort==RIGCTRL_PORT_RADIO )
         { pPortInstance->iResponseCountdown_ms = 0;   // don't wait any longer !
           pPortInstance->dwExpectedResponse_CmdAndSubcode = RIGCTRL_NOVALUE_INT; // avoid misinterpreting the NEXT unspecific "OK" or "NOT OK".
           pPortInstance->iExpectResponseForUnifiedPN = -1; // here: cleared in RigCtrl_ParseCIV(), along with pPortInstance->iResponseCountdown_ms, etc
           // (only RigCtrl_Handler() decides what to ask NEXT)
         }
        iMsgType |= RIGCTRL_MSGTYPE_FLAG_EXPECTED; // .. for the traffic display, but also for RigCtrl_CIV_Server_OnRead/WriteCommand()
        // ,---------'
        // '--> flag for the coloured live CI-V display.
        //      "Expected" responses will be green,
        //      others (like unsolicited messages) have a CYAN-coloured background.
        // 2025-07-27: When receiving a response for cmd 0x25 sub 0x00 with
        //             pPortInstance->iWorkingForServerPort = 2 (WSJT-X asked for it), got here with...
        // iMsgType = 0x0402 = RIGCTRL_MSGTYPE_FLAG_EXPECTED | RIGCTRL_MSGTYPE_FREQUENCY_REPORT (ok),
        // fShowInLog=TRUE, but the message (exchanged on behalf of an external client)
        //                  did NOT appear in the traffic log !
      } // end if < parsed the EXPECTED response >
     else // NOT an echo, and NOT the expected response ->
      {
        if( pPortInstance->iTrafficMonitorUnpauseCountdown > 0 )
         { --pPortInstance->iTrafficMonitorUnpauseCountdown; // "temporarily unpausing" shall stop after some time,
            // for reasons described in RigCtrl_ForwardCommandFromServerPortToRadio() .
         }
      }

   } // end if( ! fIsEcho )
  else // received message is an ECHO, not a response, not a command :
   { iMsgType = RIGCTRL_MSGTYPE_FLAG_ECHO; // for filtering (rejecting 'echos' in the traffic log)
   }

  pPortInstance->iMsgTypeFromParser = iMsgType; // important also to forward messages between server- and radio-port !
  pPortInstance->iUnifiedParameterNumberFromParser = iUnifiedPN;


  //--------------------------------------------------------------------------
  // At this point, finished parsing the RECEIVED message, and extracted some
  //                info (like message type, expected response type, etc).
  // Next: Decide what to do on reception, e.g. forward to another port, etc.
  //--------------------------------------------------------------------------
  if( RigCtrl_IsMsgTypeRejectedForLog( pPortInstance, iMsgType) ) // here: called from RigCtrl_ParseCIV() ..
   { fShowInLog = FALSE; // avoid flooding the CAT traffic log by certain message types, e.g. spectrum scope waveforms
   }

  if( fShowInLog && (pRC->dwMessageFilterForLog & RIGCTRL_MSGFILTER_ANY_KNOWN_COMMAND) )
   { // Reject ANY KNOWN COMMAND (or response) from the log ?
     if( (pPI != NULL) || (!fUnknown) ) // got a 'parameter info' or fUnknown CLEARED above ? Then we could decode it..
      { fShowInLog = FALSE; // <- suppressed for the traffic log.. what remains
        //    are commands that we may consider to support one fine day,
        //    especially when seen frequently on a "client port".
        // For example, when eavesdropping on the traffic betweeen WFView and an
        // IC-7300, occasionally saw this in the traffic log:
        //  >  t1 007 FE FE 94 E1 16 65 FD ; 0x16[65] ?
        // What the heck ?  A7292-4EX-11 (the IC-7300 "full manual") page 19-4 said:
        // Cmd 0x16 sub 0x65 means "Send the IP+ function setting (00=OFF, 01=ON)".
      }
   } // end if < .. dwMessageFilterForLog & RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND > ?

  if( pszComment[0] == '\0' ) // comment (pszComment, points to pPortInstance->sz255CommentFromParser) NOT SET YET ?
   { pszCmtDest = pszComment; // reset the "comment-string building pointer" again
     if( pszToken != NULL)
      { SL_AppendString( &pszCmtDest, cpCmtEndstop, pszToken );
      }
     else if( pPI != NULL )
      { SL_AppendString( &pszCmtDest, cpCmtEndstop, pPI->pszToken );
      }
     // Was it READ-COMMAND, READ-RESPONSE, WRITE-COMMAND, or WRITE-RESPONSE ?
     switch( iMsgType & RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP )
      { case RIGCTRL_MSGTYPE_FLAG_READ_CMD : SL_AppendString( &pszCmtDest, cpCmtEndstop, " ?" ); break;
        case RIGCTRL_MSGTYPE_FLAG_READ_RESP: SL_AppendString( &pszCmtDest, cpCmtEndstop, " = .." ); break;
        case RIGCTRL_MSGTYPE_FLAG_WRITE_CMD: SL_AppendString( &pszCmtDest, cpCmtEndstop, " := .." ); break;
        case RIGCTRL_MSGTYPE_FLAG_WRITE_RESP: SL_AppendString( &pszCmtDest, cpCmtEndstop, " (Wr-Resp)" ); break;
        case RIGCTRL_MSGTYPE_FLAG_OTHER_RESP: SL_AppendString( &pszCmtDest, cpCmtEndstop, " (other resp.)" ); break;
        case RIGCTRL_MSGTYPE_FLAG_UNSOLICITED: SL_AppendString( &pszCmtDest, cpCmtEndstop, " (unsolicited)" ); break;
        default: break;
      }
   } // end if( pszComment[0] = '\0' ) after the huge switch/case list in RigCtrl_ParseCIV() ?


  if( fShowInLog )  // BEFORE transmitting anything, show the *received* message in the log:
   { RigCtrl_AddToTrafficLog( pRC, iRigCtrlPort, iRigCtrlOrigin, pbMessage, iMsgLength,
          iMsgType | RIGCTRL_MSGTYPE_FLAG_RX, // turn 'em green (not yellow)
          pszComment );   // [in] e.g. "SMeterLevel ?" (some remote clients polled this FAR TOO OFTEN..)
   }

  if( pPortInstance->pAppCallback != NULL ) // did THE USER of RigControl.c register a callback ?
   { // Let some other module (e.g. AuxComPorts.c) know we have DECODED another message on this port:
     pPortInstance->pAppCallback( pPortInstance, pRC->iRigCtrlOrigin, RIGCTRL_CBK_DECODED, pbMessage, iMsgLength, pszComment );
     // '--> e.g. AuxComPorts.c : AuxCom_RigControlCallback(), checks when an external client has finished the initialisation, etc .
     //       [in] pPortInstance->iMsgTypeFromParser : has already been set to iMsgType,
     //            with bitgroups/flags like RIGCTRL_MSGTYPE_FLAG_READ_CMD,RIGCTRL_MSGTYPE_FLAG_WRITE_CMD, etc .
   }

  if( pPortInstance->fActAsServer )
   {
     switch( iMsgType & RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP )
      { case RIGCTRL_MSGTYPE_FLAG_READ_CMD :
           RigCtrl_CIV_Server_OnReadCommand( pPortInstance, pPI, pbMessage, iMsgLength, iMsgType, iCIVDataType );
           // '--> Doesn't send anything from CIV-Server to the remote client yet,
           //      but MAY have already placed a complete response in the port-instance,
           //      and set pPortInstance->iServerState to RIGCTRL_SSTATE_RESPONSE_READY .
           //      The actual TRANSMISSION of the response from server to client happens later,
           //      e.g. in AuxComPorts.c : AuxComThread() -> AuxCom_RunVirtualRig() .
           break;
        case RIGCTRL_MSGTYPE_FLAG_WRITE_CMD :
           RigCtrl_CIV_Server_OnWriteCommand( pPortInstance, pPI, pbMessage, iMsgLength, iMsgType, iCIVDataType );
           // '--> Similar as above (RigCtrl_CIV_Server_OnReadCommand), but
           //      a WRITE-COMMAND received from a remote client will be
           //      forwarded to the RADIO PORT in most cases . Thus, as this point,
           //      pPortInstance->iServerState will usually be RIGCTRL_SSTATE_WAIT_RADIO_NBUSY,
           //      and the next step is RigCtrl_ForwardCommandFromServerPortToRadio() .
           break;
        default:
           break;
      } // end switch( iMsgType & RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP )
   }   // end if( pPortInstance->fActAsServer )


  if( iRigCtrlPort == RIGCTRL_PORT_RADIO ) // response received on the "RADIO" port ?
   { // May have to "forward" this message to one or more of the clients,
     //     regardless is this is an EXPECTED RESPONSE or an UNSOLICITED MESSAGE.
     // But some clients (like 'Hamlib' used by WSJT) are irritated by ANYTHING
     //       THEY DO NOT EXPECT (-> WSJT-X : "Ploink... RIG CONTROL ERROR").
     if( ! fIsEcho )
      { RigCtrl_CheckAndForwardResponseToClients( pRC, pbMessage, iMsgLength, // here: called from RigCtrl_ParseCIV() ...
                       &msg_filter, iMsgType, fIsUnsolicitedMsg, fIsExpectedResponse, pszComment );
      }
   }
  else // iRigCtrlPort != RIGCTRL_PORT_RADIO -> command from an EXTERNAL CLIENT ?
   { // "Echo-back" (Icom slang) the RECEIVED message (on the port on which it was received) ?
     //   Some applications developed when CI-V was a simple SINGE-WIRE interface
     //   still expect their transmitted data echoed back, possibly for collision detection .
     if( pPortInstance->fEmulateCIVEcho ) // <- this may have been turned on/off further above via CI-V itself !
      { RigCtrl_SendAndLogMessage( pRC, iRigCtrlPort, iRigCtrlOrigin, pbMessage, iMsgLength,
            RIGCTRL_MSGTYPE_FLAG_ECHO | RIGCTRL_MSGTYPE_FLAG_TX, pszComment );
      } // end if( pPortInstance->fEmulateCIVEcho )
     // Retransmit a client's command to the RADIO port,
     //  or store it in a FIFO (to retransmit it LATER), or ignore it, or... ?
     //  (Removed. Since 2025-08, forwarding commands to the radio
     //            "on be half of a REMOTE CLIENT" happens in
     //            RigCtrl_ForwardCommandFromServerPortToRadio() ! )
   } // end else < command received on one of the CLIENT ports >

  return iMsgType;

} // end RigCtrl_ParseCIV()


//--------------------------------------------------------------------------
void RigCtrl_AutoDetectCIVAddresses( T_RigCtrl_PortInstance *pPortInstance,
                          BYTE bToAddr, BYTE bFromAddr )
  // [out]  may set pPortInstance->iRadioMasterAddr and/or pPortInstance->iRadioDeviceAddr,
  //        when still set to RIGCTRL_DEF_ADDR_AUTO_DETECT,
  //        and if bToAddr and/or bFromAddr are OBVIOUS cases ( 0xE0 or above = MASTER),
  //        below 0xE0 but not 0x00 = DEVICE ).
  // In 'eavesdropping mode' (with pRC->fListenOnlyMode = TRUE),
  // we initially don't know if the message was sent from
  // CONTROLLER (aka MASTER) to RADIO, or VICE VERSA. As a convention used
  // by most amateur radio applications, address 0xE0 = RIGCTRL_CIV_MASTER_ADDR_MIN and above
  // are used for the CONTROLLER, while address 0xDF and below
  // should indicate an Icom radio :
{
  if( (bFromAddr >= RIGCTRL_CIV_MASTER_ADDR_MIN) && (bFromAddr <= RIGCTRL_CIV_MASTER_ADDR_MAX) )  // 0xE0..0xEF ? guess this is a MASTER, aka CONTROLLER address:
   { pPortInstance->iRadioMasterAddr = bFromAddr;
     // if bFromAddr has been identified as CI-V MASTER/CONTROLLER,
     // then bToAddr must be the RADIO's address (unless it's 0x00 = BROADCAST)
     if( bToAddr != 0x00 )
      { pPortInstance->iRadioDeviceAddr = bToAddr;
      }
   }
  // .. and the same with "to"- and "from"-address swapped, because especially
  //    in 'eavesdropping mode' (with pRC->fListenOnlyMode==TRUE)
  //    we cannot be sure which end of the Serial Port Tunnel faces the RADIO,
  //    and which end faces the MASTER/CONTROLLER :
  if( (bToAddr >= RIGCTRL_CIV_MASTER_ADDR_MIN) && (bToAddr <= RIGCTRL_CIV_MASTER_ADDR_MAX) )  // guess this is a MASTER, aka CONTROLLER address:
   { pPortInstance->iRadioMasterAddr = bToAddr;
     // if bToAddr has been identified as CI-V MASTER/CONTROLLER,
     // then bFromAddr must be the RADIO's address (unless it's 0x00 = BROADCAST)
     if( bFromAddr != 0x00 )
      { pPortInstance->iRadioDeviceAddr = bFromAddr;
      }
   }
} // end RigCtrl_AutoDetectCIVAddresses()


//--------------------------------------------------------------------------
int RigCtrl_ProcessData( T_RigCtrlInstance *pRC, // [in,out] 'Rig Control' instance.
        int iRigCtrlPort,   // [in] e.g. RIGCTRL_PORT_RADIO for the 'real' radio,
                            //       or  RIGCTRL_PORT_AUX_COM_1/2/3 ...
                            //             |-- with pRC->fListenOnlyMode=TRUE to "eavesdrop" on a SERIAL PORT TUNNEL,
                            //             |-- with pRC->fListenOnlyMode=FALSE on a VIRTUAL RIG PORT to emulate a radio.
        int iRigCtrlOrigin, // [in] e.g. RIGCTRL_ORIGIN_RADIO, RIGCTRL_ORIGIN_CONTROLLER, RIGCTRL_ORIGIN_COM_PORT_RX/TX
                            // (especially important in Listen-Only mode; see details below).
                            //  The "Rig Control Port number" cannot indicate the DIRECTION of data (RX,TX).
                            // That's why we also pass 'iRigCtrlOrigin' to the protocol-specific decoder,
                            // last not least to have a human friendly output in the traffic log.
        T_RigCtrl_AppCallback pAppCallback, // [in] optional callback for the application (may be NULL)
        BYTE *pbData, int nBytes ) // [in] chunk of received data (not necessarily a complete, nor a SINGLE 'Message')
                            // [out] pRC->bRxBuffer[pRC->nBytesInRxBuffer++],
                            //       until a VALID COMMAND is complete .
                            //  In Listen-Only mode (with pRC->fListenOnlyMode==TRUE),
                            //       pRC->bAltRxBuffer[pRC->nBytesInAltRxBuffer++]
                            //       may be used instead (depends on iRigCtrlOrigin).
  // [return] iMsgType (as returned by e.g. RigCtrl_ParseCIV(), called from here.
  //          Returns -1 when the data block doesn't contain a parseable message.
  //
  // Subroutine for RigCtrl_ProcessRxData(), using a PUSH-, not a PULL model.
  // When used on a 'serial port tunnel', commands and responses may
  //   arrive in both directions. In that case, iOrigin differentiates
  //   data in ONE direction from data travelling in the OTHER direction.
  //   See caller, e.g. AuxComPorts.c : AuxCom_RunSerialPortTunnel() .
  // Callers:
  //   (a) after RECEPTION of data from a 'local' COM port controlling a REAL radio:
  //       RigCtrl_ProcessRxData() -> RigCtrl_ProcessData() -> RigCtrl_ParseCIV() -> ...,
  //       with pRC->fListenOnlyMode = FALSE .
  //   (b) after reception of data from an Additional COM Port, in eavesdropping mode(*)
  //       between an external 'controller' on one end, and 'the rig' somewhere else:
  //       AuxComThread(!) -> AuxComPassRxOrTxDataToProtocolSpecificDecoder()
  //                             with iRigCtrlOrigin = RIGCTRL_ORIGIN_COM_PORT_RX (2)
  //                             and  pRC->fListenOnlyMode = TRUE .
  //        -> RigCtrl_ProcessRxData() -> RigCtrl_ProcessData() -> RigCtrl_ParseCIV() -> ...,
  //       (*) with pRC->fListenOnlyMode = FALSE to avoid interference
  //           if the additional COM port is used as a Serial Port Tunnel.
  //   (c) shortly before TRANSMISSION (overlapped "WriteFile()" complete)
  //                      of data to a 'local' COM port :
  //       AuxComThread(!) -> AuxComPassRxOrTxDataToProtocolSpecificDecoder()
  //                             with iRigCtrlOrigin = RIGCTRL_ORIGIN_COM_PORT_TX (3)
  //                             and  pRC->fListenOnlyMode = TRUE .
{
  T_RigCtrl_PortInstance *pPortInstance;
  T_RigCtrl_MsgBuffer    *pMsgBuf;
  BYTE b, bPrev;
  int iMsgType = -1;

  if( pRC == NULL ) // oops.. instance deleted but some thread is still calling us !
   { return -1;
   }
  if( pRC->initState != initState_Opened ) // oops.. didn't call RigCtrl_Init() yet, or called RigCtrl_Close()
   { return -1;
   }

  if( (iRigCtrlPort>=RIGCTRL_PORT_AUX_COM_1) || (iRigCtrlPort<=RIGCTRL_PORT_AUX_COM_LAST) )
   { // The data were not received from, or sent to, THE RADIO (directly),
     //  but on an 'Additional' (ex: 'Auxiliary') COM PORT in 'serial tunnel' or 'virtual rig' mode.
     pRC->fListenOnlyMode = pRC->fListenOnlyMode; // <- place for a BREAKPOINT :
     //  * If the Additional COM Port is in 'Serial Port Tunnel' mode,
     //    the application must have set pRC->fListenOnlyMode to TRUE,
     //    e.g. in TKeyerMainForm::StartKeyerAndShowInfo() .
     //  * If the Additional COM Port is in 'Virtual Rig' (-emulation) mode,
     //    the application must have set pRC->fListenOnlyMode to FALSE,
     //    e.g. in TKeyerMainForm::StartKeyerAndShowInfo() .
     // At this point, iRigCtrlPort is a vaild ARRAY index into T_RigControl.PortInstance[].
   }
  if( (iRigCtrlPort<0) || (iRigCtrlPort>/*!*/RIGCTRL_MAX_CLIENT_PORTS) )
   { // iRigCtrlPort = 0 = RIGCTRL_PORT_RADIO = communication port that "talks to the radio",
     // uses the same subroutine to process received data !
     return -1;  // not a valid 'port identifier' (not to be confused with a COM-port-number) !
   }

  pPortInstance = &pRC->PortInstance[iRigCtrlPort];
  pPortInstance->pAppCallback = pAppCallback;   // store internally, for dozens of subroutines called from here
  // Messages travelling in ONE direction must not interfere with the OPPOSITE direction !
  // Thus, depending in the LEAST SIGNIFICANT BIT in iRigCtrlOrigin, select DIFFERENT MESSAGE BUFFERS:
  pMsgBuf = &pPortInstance->sRxMsg[iRigCtrlOrigin & 1];
     // ,--------------------------|________________|
     // '--> e.g. for RIGCTRL_ORIGIN_RADIO (0),       use sRxMsg[0];
     //           for RIGCTRL_ORIGIN_CONTROLLER (1),  use sRxMsg[1].
     //   Similar for RIGCTRL_ORIGIN_COM_PORT_RX (2) = "all we know is this was RECEIVED on a local COM port"
     //            vs RIGCTRL_ORIGIN_COM_PORT_TX (3) = "all we know is this was SENT TO a local COM port" .

  // Store the received data from the radio in the NON-CIRCULAR buffer (pMsgBuf),
  // which can easily be parsed as soon as a "message" (e.g. CI-V) is complete:
  bPrev = (pMsgBuf->iLength > 0 ) ? pMsgBuf->bData[pMsgBuf->iLength-1] : 0x00;
  while( (nBytes--) > 0 )
   {
     b = *pbData++;  // get the next BYTE from the input stream
     // Collect all received bytes in an extra buffer until the end of the message is reached:
     if( pMsgBuf->iLength < RIGCTRL_MAX_MESSAGE_LENGTH )
      {  pMsgBuf->bData[pMsgBuf->iLength++] = b;
      }
     else // couldn't append the new byte, count it as "garbage"
      {  ++pPortInstance->dwNumGarbageBytes;
      }

     // Check for end-of-message (depends on the protocol; radio-specific)
     switch( pPortInstance->iRadioCtrlProtocol )
      {
        case RIGCTRL_PROTOCOL_NONE :   // "no" protocol, assume "normal text"..
        default :
             // in this case, "zero" bytes, carriage return, or new line are treated as end-markers:
             if( b==0 || b==10 || b==13 )
              {
                pMsgBuf->iLength = 0;   // begin receiving next frame
              }
             break;
        case RIGCTRL_PROTOCOL_ICOM_CI_V  :
             if( b==0xFD )  // received CI-V code for "End Of Message" aka "postamble" ?
              {
               if( (pMsgBuf->bData[0]==0xFE) && (pMsgBuf->bData[1]==0xFE) )
                { // preamble code ok ? Then the NEXT byte [2] is either Icom's
                  // "Receiver address" or the "Controller address".
                  // May have to GUESS who is who ... :
                  RigCtrl_AutoDetectCIVAddresses( pPortInstance, pMsgBuf->bData[2]/*"TO"*/, pMsgBuf->bData[3]/*FROM*/ );
                     // '--> may set pPortInstance->iRadioMasterAddr and or pPortInstance->iRadioDeviceAddr,
                     //      by checking for addresses 0xE0 .. 0xEF which should be
                     //      used as MASTER (aka CONTROLLER) address only.
                  if(  (pMsgBuf->bData[2]==pPortInstance->iRadioMasterAddr) // <- usually 0xE0, but wfview used 0xE1 (doesn't seem to matter as long as it's different from the RADIO DEVICE ADDRESS)
                     ||(pMsgBuf->bData[2]==pPortInstance->iRadioDeviceAddr) // listen to the "echo" too !
                     ||(pMsgBuf->bData[2]==0x00) // directed to the "broadcast" address ?  (*)
                     || pRC->fListenOnlyMode )  // .. or accept ALL addresses for "eavesdropping" ?
                   { // (*) Messages triggered by turning the dial or by changing
                     //     the mode are sent to this adress 0x00 (provided that "CI-V transceive"
                     //     is set to ON. If OFF, no messages are generated by the rig).
                     ++pPortInstance->dwNumMessagesRcvd;
                     ++RigCtrl_i32MessageNrForTrafficLog; // here: RECEIVED another CI-V message, in RigCtrl_ProcessData()
#                   if( SWI_RIGCTRL_ACT_AS_SERVER )
                     // Special service for CI-V messages received on a VIRTUAL RIG port;
                     //  details only in RigControl_CIV_Server.c [store command+subCmd+subSumCmd+Who-Knows-What]
                     if(pPortInstance->fActAsServer )
                      { RigCtrl_CIV_Server_PrepareParsing( pPortInstance );
                      }
#                   endif // SWI_RIGCTRL_ACT_AS_SERVER ?
                     iMsgType = RigCtrl_ParseCIV( pRC, iRigCtrlPort, iRigCtrlOrigin, pMsgBuf->bData, pMsgBuf->iLength );
#                   if( SWI_RIGCTRL_ACT_AS_SERVER )
                     if(pPortInstance->fActAsServer && (pPortInstance->iServerState==RIGCTRL_SSTATE_RESPONSE_READY))
                      { RigCtrl_SendServerResponse( pPortInstance );
                      }
#                   endif // SWI_RIGCTRL_ACT_AS_SERVER ?
                   } // end if < matching "address" >
                  else // pre- and postamble ok, but something wrong with the addresses
                   { pPortInstance->dwNumGarbageBytes += pMsgBuf->iLength;
                   }
                 } // end if < preamble code ok >
                pMsgBuf->iLength = 0;   // begin receiving next frame
              } // end if < received "end-of-message" code >
             else if( (b==0xFE) && (bPrev==0xFE) )
              { // Detected the CI-V "Preamble code (fixed)" ->
                // Expect the NEXT byte (at offset 2) to be the
                //        "Transceiver's default address"
                // or the "Controller's default address" .
                //      (more on those "default addresses", including a LIST, in RigCtrl_RadioInfo_CIV[] )
                //
                // The IC-7610 CI-V REFERENCE GUIDE, A7380-7EX-1 available from Icom,
                // clearly states that MORE THEN TWO of these 'preamble codes'
                // may be sent. Example:
                // > When sending the power ON command (18 01), you need
                // > to repeatedly send "FE" before the standard format.
                // (at 115200 bits/second, the receiver expects 150 (!) "FE"s)
                // When Icom's "RS-BA1" is at the other end of a com0com connection
                // (thinking it's talking to an Icom radio but in fact talking to us),
                // got here with a huge pile of 0xFE's before the first "real" message.
                if( pMsgBuf->iLength > 3 ) // oops.. something in between that was NOT 0xFE
                 {  pPortInstance->dwNumGarbageBytes += (pMsgBuf->iLength-3);
                 }
                pMsgBuf->iLength = 2; // discard anything before the "Preamble code"
              }

             break; // end case C_RADIO_CTRL_PROT_ICOM_CI_V
      } // end switch( pRC->iRadioCtrlProtocol )
     bPrev = b;
   }  // end for <all bytes from serial data block>

  return iMsgType;
} // end RigCtrl_ProcessData()


//--------------------------------------------------------------------------
static int RigCtrl_ProcessRxData(   // tries to de-packetize a 'serial bytestream'.
        // For CI-V, looks for the 0xFE 0xFE preamble, and the 0xFD postamble, etc.
        // For Yaesu's "5-Byte-command-CAT", lets Yaesu5Byte_ProcessData() do the job.
        T_RigCtrlInstance *pRC,  // [in] 'Rig Control' instance (so far, only one, but who knows..)
                  // [in]  '-- pRC->pRadioPortRxFifo : RX FIFO, most likely filled in ANOTHER THREAD
        int  iRigCtrlPort)       // [in] RIGCTRL_PORT_RADIO / RIGCTRL_PORT_AUX_COM_1 / 2 ?
  // [return] iMsgType (as returned by e.g. RigCtrl_ParseCIV(), called from here.
  //          Returns -1 when the data block doesn't contain a parseable message.
{ // Called from the main thread ("GUI thread") after reception of a block of data.
  //        Initially only for data received from a serial port ("COM"),
  //        later (but not HERE) also used for CI-V embedded in UDP packets for LAN or WLAN.
  int iMsgType = -1;
  int iSource, nBytesReceived = 0;
  int iRigCtrlOrigin = RIGCTRL_ORIGIN_COM_PORT_RX; // iRigCtrlOrigin, e.g. RIGCTRL_ORIGIN_COM_PORT_RX or ..TX.
      //     (in 'eavesdropping mode' for e.g. AuxComPorts.c ,
      //      XYZ_ProcessData() must be called for data
      //      travelling in BOTH directions: TO THE RADIO and FROM THE RADIO.
      //      Thus this extra parameter for the 'direction' of data
      //      for the protocol-specific parsers called further below.
  BYTE bTemp[1024], b;
  T_RigCtrl_PortInstance *pPortInstance;


  if( pRC == NULL ) // oops.. instance deleted but some thread is still calling us !
   { return -1;
   }
  if( pRC->initState != initState_Opened ) // oops.. didn't call RigCtrl_Init() yet, or called RigCtrl_Close()
   { return -1;
   }

  if( (iRigCtrlPort<0) || (iRigCtrlPort>/*!*/RIGCTRL_MAX_CLIENT_PORTS) )
   { // iRigCtrlPort = 0 = RIGCTRL_PORT_RADIO = communication port that "talks to the radio",
     // uses the same subroutine to process received data !
     return -1;  // not a valid 'port identifier' (not to be confused with a COM-port-number) !
   }
  pPortInstance = &pRC->PortInstance[iRigCtrlPort];


  // The keyer thread may have placed received bytes in a thread-safe FIFO.
  // Pass the content of that FIFO to RigControl, with a CI-V parser, etc:
  if( (iRigCtrlPort==RIGCTRL_PORT_RADIO) && (pRC->pRadioPortRxFifo!=NULL) )
   { nBytesReceived = CFIFO_Read( pRC->pRadioPortRxFifo, bTemp, sizeof(bTemp), NULL/*no timestamp*/ );
   }

# if( SWI_SUPPORT_YAESU_5_BYTE_CAT )
  if( pPortInstance->iRadioCtrlProtocol == RIGCTRL_PROTOCOL_YAESU_5_BYTE )
   { if( nBytesReceived > 0 )
      { iMsgType = Yaesu5Byte_ProcessData( &pRC->Y5B, // [in,out] T_Yaesu5ByteControl = a 'Yaesu 5-byte CAT' instance
                RIGCTRL_PORT_RADIO, // [in] iRigCtrlPort,   // [in] e.g. , RIGCTRL_PORT_AUX_COM_1, .. RIGCTRL_PORT_AUX_COM_1, ..
                   iRigCtrlOrigin,  // [in] e.g. RIGCTRL_ORIGIN_COM_PORT_RX
            bTemp, nBytesReceived); // [in] chunk of received data (not necessarily a complete, nor a SINGLE 'Message')
        // Again (important): During a SINGLE CALL of Yaesu5Byte_ProcessData(),
        //     the CALLBACK FUNCTION (here: RigCtrl_Y5B_OnDecodeCallback() )
        //     may be called NOT AT ALL, or ONCE, or even MULTIPLE TIMES !
        //     Since the number of bytes in the CAT's *RESPONSE* is not always
        //     known, or even unpredictable, there's little we can do about it.
        // Keep any "Yeasu-5-byte-CAT-specific" processing out of here !
        // It's already messy enough without support for Yaesu's various CATs.
      } // end if( nBytesReceived > 0 ) ?
     return iMsgType;
   }     // end if( pRC->iRadioCtrlProtocol == RIGCTRL_PROTOCOL_YAESU_5_BYTE )
# endif // SWI_SUPPORT_YAESU_5_BYTE_CAT ?


  return RigCtrl_ProcessData( pRC, iRigCtrlPort, iRigCtrlOrigin,
           pPortInstance->pAppCallback,
           bTemp, nBytesReceived ); // [in] chunk of received data (not necessarily a complete, nor a SINGLE 'Message')

} // end RigCtrl_ProcessRxData()

#if( SWI_RIGCTRL_ACT_AS_SERVER )
//---------------------------------------------------------------------------
static BOOL RigCtrl_SendServerResponse( T_RigCtrl_PortInstance *pServerPortInstance )
  // Called from RigCtrl_ProcessData(), with iServerState == RIGCTRL_SSTATE_RESPONSE_READY .
  // See server state diagram explained in RigControl_CIV_Server.c !
  //
  //  (*) Actually, called IMMEDIATELY AFTER returning from RigCtrl_ParseCIV()
  //      if the RESPONSE could already be filled out without forwarding the
  //      message to the radio.
  //      Beware about the issue with different callers (threads).
{
  BOOL fResult;
  T_RigCtrlInstance *pRC = pServerPortInstance->pRC;
  int iMsgType = pServerPortInstance->iMsgTypeFromParser;
  int iRigCtrlPort = RigCtrl_PortInstancePtrToIndex(pServerPortInstance);
  const char *pszSource;
  char *pszDest, *pszEndstop;

  if( iMsgType & RIGCTRL_MSGTYPE_FLAG_RX ) // message still flagged as "RECEIVED" ?
   { iMsgType &= ~RIGCTRL_MSGTYPE_FLAG_RX;  // this isn't a RECEIVED COMMAND anymore..
     iMsgType |= RIGCTRL_MSGTYPE_FLAG_TX;   // .. but we are going to SEND it (as a RESPONSE) now
   }
  // Similar as for the RX/TX-flag (corrected above), the COMMENT may still be
  // the original one from as set in RigCtrl_ParseCIV(). If it was a READ COMMAND,
  // the comment ends with a '?'. In that case pServerPortInstance->sTxMsg contains
  // a READ RESPONSE (usually with a value), so modify the string as follows:
  pszDest = pServerPortInstance->sz255CommentFromParser;
  pszEndstop = pServerPortInstance->sz255CommentFromParser+255;
  if( SL_SkipToEndOfString( (const char**)&pszDest ) > 0 ) // skipped anything ?
   { if( pszDest[-1] == '?' ) // the comment indeed ended with a '?', so ..
      { --pszDest;
        *pszDest = '\0';      // .. remove the trailing '?', and replace by the following...
        if( (pszDest>pServerPortInstance->sz255CommentFromParser) && (pszDest[-1]==' ') )
         { --pszDest; *pszDest = '\0'; // also remove the space between the parameter name and the '?'
         }
        // Similar as in RigCtrl_ParseCIV(), a ':' indicates a "SET"-command (WRITE something),
        //                                   a '=' indicates a "GET"-response (have READ something).
        // Thanks to the scary RigCtrl_ParseCIV()-monster for 'classifying' the message type:
        switch( iMsgType & RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP )
         { case RIGCTRL_MSGTYPE_FLAG_WRITE_CMD :
                SL_AppendChar( &pszDest, pszEndstop, ':' );
                break;
           default:
                SL_AppendChar( &pszDest, pszEndstop, '=' );
                break;
         } // end switch( iMsgType & RIGCTRL_MSGMASK_FLAGS_RD_WR_RESP )
        if( RigCtrl_GetParamValueAsString( pRC, // current value AS STRING (when available)...
               &pServerPortInstance->sParamInfoForPendingCommand,
               pszDest, (pszEndstop-pszDest-8)/*iMaxLen*/ ) )
         { // If the parameter also has a PHYSICAL UNIT, show it in the log, too:
           if( (pszSource=pServerPortInstance->sParamInfoForPendingCommand.pszUnit) != NULL )
            { if( pszSource[0] != '\0' )  // many PHYSICAL UNITS are empty strings .. ignore them, too
               { SL_SkipToEndOfString( (const char**)&pszDest );
                 SL_AppendPrintf( &pszDest, pszEndstop, " %s", pszSource );
               }
            }
         }
        else
         { SL_AppendString( &pszDest, pszEndstop, "[RD-resp]" ); // worst alternative..
         }
      }
     // The result may be not as nice as parsing the READ-RESPONSE again (with value shown),
     // but better than nothing. Here the first few lines from the begin of WSJT-X session:
     // >   1 0002537 r2 006 FE FE 94 E0 03 FD                        ; VFOfreq ?
     // >   2 0002537 t2 00B FE FE E0 94 03 50 00 03 07 00 FD ; VFOfreq [RD-resp]
     // >   3 0004265 r2 006 FE FE 94 E0 03 FD                        ; VFOfreq ?
     // >   4 0004265 t2 00B FE FE E0 94 03 50 00 03 07 00 FD ; VFOfreq [RD-resp]
     // (Not sure why WSJT-X has unhappy with the the response,
     //   and asked the same ("VFO frequency ?") again after ~2 seconds) ..
     //

   }

  fResult = RigCtrl_SendAndLogMessage( pRC,
        iRigCtrlPort, // [in] here: expecting only RIGCTRL_PORT_AUX_COM_1..3, but not RIGCTRL_PORT_RADIO !
        RIGCTRL_ORIGIN_CONTROLLER,
        pServerPortInstance->sTxMsg.bData,   // [in] message, including pre- and postamble
        pServerPortInstance->sTxMsg.iLength, // [in] number of bytes to send, " " " "
        iMsgType,         // [in] message type category and 'flags', only for the traffic log
        pServerPortInstance->sz255CommentFromParser); // [in] human readable info, also for the traffic log
  return fResult;
} // end RigCtrl_SendServerResponse()
#endif // SWI_RIGCTRL_ACT_AS_SERVER ?




//----------------------------------------------------------------------------
void RigCtrl_AssembleVfoReport( // Periodically called from e.g. CwNet.c : CwNet_OnPoll() to send a 'VFO Report', with frequency, TX status, and S-Meter reading
        T_RigCtrlInstance  *pRC, // [in] Rig Control instance with parameters from the real radio
        T_RigCtrl_VfoReport *pVfoReport ) // [out] "VFO report", bandwidth-saving struct sent from CwNet.c
{
  pVfoReport->bStructSize= sizeof(T_RigCtrl_VfoReport); // allows future extensions at the end of the struct
  pVfoReport->bVfoIndex  = 0;   // future plan for multi-VFO-radios like the IC-7610 : [0]=main VFO, [1]=sub VFO
  pVfoReport->bTrxStatus = 0;   // bitwise combination of flags like RIGCTRL_TRX_STATUS_TRANSMITTING, .._SQUELCH_MUTED, etc
  if( pRC->iTransmitting )
   { pVfoReport->bTrxStatus |= RIGCTRL_TRX_STATUS_TRANSMITTING;
   }
  pVfoReport->i8SMeterLevel_dB = pRC->iSMeterLevel_dB;
  pVfoReport->dblVfoFrequency  = pRC->dblVfoFrequency;
} // end RigCtrl_AssembleVfoReport()


//--------------------------------------------------------------------------
void RigCtrl_OnVfoReportFromNetwork( T_RigCtrlInstance *pRC,
        T_RigCtrl_VfoReport *pVfoReport) // [in] received network packet
  // Called from CwNet.c on reception of a packet with CWNET_CMD_FREQ_REPORT.
  // First used in the 'Remote CW Keyer', CLIENT SIDE (without a real radio),
  // where a 'Rig Control' instance only stores parameters for the display in
  // the GUI, but doesn't control a locally connected radio.
  // [out] pRC->dblVfoFrequency,
  //       pRC->iFrequencyModifiedByRadio_cnt,
  //       and maybe a few other parameters that would NORMALLY be updated
  //       on reception of an unsolicited CI-V message in RigCtrl_ProcessRxData().
  // Note: RigCtrl_OnVfoReportFromNetwork() is used by the Remote CW Keyer
  //       on the CLIENT as well as on the SERVER side. Only the SERVER side
  //       is locally connected to a radio (and passes on the new
  //       frequency).
  //       When running as a CLIENT (not controlling a 'local radio')
  //       the new VFO frequency is only stored in pRC->dblVfoFrequency
  //       (from where the GUI thread can poll it to update the display).
  //   The same note applies to similar handlers like
  //   RigCtrl_OnParamReport_Byte/Double/String(), implemented further below.
{
  if( pVfoReport != NULL )
   { if( pVfoReport->bStructSize >= 8 ) // Struct with the MINIMUM SIZE ?
      { // 2025-01: Got here with pVfoReport->dblVfoFrequency = GARBAGE,
        //          when called from CwNet.c : ServerThread()
        //           -> CwNet_OnReceive() -> CwNet_ExecuteCmd()
        //            -> RigCtrl_OnVfoReportFromNetwork() .
        //          That's the price to pay for space-saving BINARY PACKETS.
        //          The corruption of data seemed to originate from the SENDER
        //          (in this case, the remote client) because this only happened
        //          after an event-driven transmission of 'rigctld'-command
        //          "set_ptt 1\n" from CwNet.c : CwNet_OnPoll() .
        if( (pVfoReport->dblVfoFrequency != pRC->dblVfoFrequency )
          &&(pVfoReport->dblVfoFrequency >  0.0/*Hz*/)
          &&(pVfoReport->dblVfoFrequency < 30e9/*Hz*/)
          )
         { pRC->dblVfoFrequency = pVfoReport->dblVfoFrequency;
           ++pRC->iFrequencyModifiedByRadio_cnt;
           // This counter is polled in a GUI-thread somewhere.
           // If this 'rig control instance' really controls a radio,
           //    pass the requested "new frequency" on to that radio:
           if( (pRC->initState == initState_Opened )
             &&(pRC->iDefaultAddress > RIGCTRL_DEF_ADDR_NO_REMOTE_CTRL) ) // not in "passive" mode ?
            { RigCtrl_SetVFOFrequency_Internal( pRC, pRC->dblVfoFrequency ); // send command to QSY to the real radio
            }
         }
        if( pVfoReport->i8SMeterLevel_dB != -128 )
         { pRC->iSMeterLevel_dB = pVfoReport->i8SMeterLevel_dB;
           // '--> polled and displayed in the GUI-thread
         }
      }
   }
} // end RigCtrl_OnVFOReportFromServer()

//---------------------------------------------------------------------------
void RigCtrl_AssembleMultiFunctionMeterReport(
         T_RigCtrlInstance *pRC,
         T_RigCtrl_MultiFunctionMeterReport *pMeterReport)
{
  memset( pMeterReport, 0x80, sizeof(T_RigCtrl_MultiFunctionMeterReport) ); // set UNUSED fields (bytes) to 'invalid' (-128 %)

  pMeterReport->bStructSize = sizeof(T_RigCtrl_MultiFunctionMeterReport); // allows future extensions at the end of T_CwNet_MultiFunctionMeterReport
  if( pRC->iPowerMeterLevel_pcnt != RIGCTRL_NOVALUE_INT )
   { pMeterReport->i8PowerMeterLevel_pcnt = pRC->iPowerMeterLevel_pcnt; // ICOM-like measured power level IN PERCENT (!)
   }
  if( pRC->dblSWRMeterValue != RIGCTRL_NOVALUE_DOUBLE )
   { pMeterReport->i8SWRMeterValue_x10  = (int)(10.0*pRC->dblSWRMeterValue); // SWR meter reading (dimensionless, typically 1.0 to 3.0 when measured with an IC-7300)
   }
  if( pRC->iALCMeterLevel_pcnt != RIGCTRL_NOVALUE_INT )
   { pMeterReport->i8ALCMeterLevel_pcnt = pRC->iALCMeterLevel_pcnt; // 0 ... 100 % of whatever the MAXIMUM may be
   }
  if( pRC->iCompMeterLevel_dB != RIGCTRL_NOVALUE_INT )
   { pMeterReport->i8CompMeterLevel_dB  = pRC->iCompMeterLevel_dB;  // compression meter level (measured); Icom measures from 0 to 30 dB. We send it "in dB", not "0241 = 30 dB".
   }
  if( pRC->iSupplyVoltage_mV != RIGCTRL_NOVALUE_INT )
   { pMeterReport->u8SupplyVoltage_100mV= pRC->iSupplyVoltage_mV/100; // supply voltage in 100-mV-units
   }
  else
   { pMeterReport->u8SupplyVoltage_100mV= 0xFF; // 'invalid value' for an UNSIGNED 8-bit value
   }
  if( pRC->iDrainCurrent_mA != RIGCTRL_NOVALUE_INT )
   { pMeterReport->u8DrainCurrent_100mA = pRC->iDrainCurrent_mA/100;  // drain current in 100-mA-units
   }
  else
   { pMeterReport->u8DrainCurrent_100mA = 0xFF; // 'invalid value' for an UNSIGNED 8-bit value
   }
  if( pRC->iPATemperature_degC != RIGCTRL_NOVALUE_INT )
   { pMeterReport->u8PATemperature_degC = pRC->iPATemperature_degC;  // power amplifier temperature in C (not Kelvin, not Farenheit!)
                 // Note: Icom seems to have forgotten a CI-V command
                 //       to query the POWER AMPLIFIER TEMPERATURE.
                 //       But keep this struct member, because things may be different for other rigs,
                 //       or when using the planned 'USB I/O controller PIC'
                 //       with a few analog inputs for various purposes.
   }
  else
   { pMeterReport->u8PATemperature_degC = 0xFF; // 'invalid value' for an UNSIGNED 8-bit value
   }

} // end RigCtrl_AssembleMultiFunctionMeterReport()


//--------------------------------------------------------------------------
void RigCtrl_OnMultiFunctionMeterReportFromNetwork(T_RigCtrlInstance *pRC,
               T_RigCtrl_MultiFunctionMeterReport *pMeterReport)
  // Called from CwNet.c on reception of a packet with CWNET_CMD_MULTI_FUNCTION_METER_REPORT.
  // First used in the 'Remote CW Keyer', CLIENT SIDE (without a real radio),
  // where a 'Rig Control' instance only stores parameters for the display in
  // the GUI, but doesn't control a locally connected radio.
  // [out] pRC->iPowerMeterLevel_pcnt, pRC->dblSWRMeterValue,
  //       pRC->iALCMeterLevel_pcnt,   pRC->iCompMeterLevel_dB,
  //       pRC->iSupplyVoltage_mV,     pRC->iDrainCurrent_mA,
  //       pRC->iPATemperature_degC .  ( if, one day, we know how to read this via CI-V)
{
  if( pMeterReport != NULL )
   { if( pMeterReport->bStructSize >= 8 ) // Struct with the MINIMUM SIZE (sizeof(T_RigCtrl_MultiFunctionMeterReport) in its oldest version)
      { if( pMeterReport->i8PowerMeterLevel_pcnt >= 0 )
         { pRC->iPowerMeterLevel_pcnt = pMeterReport->i8PowerMeterLevel_pcnt;
         }
        if( pMeterReport->i8PowerMeterLevel_pcnt >= 0 )
         { pRC->dblSWRMeterValue      = 0.1 * (double)pMeterReport->i8SWRMeterValue_x10;
         }
        if( pMeterReport->i8PowerMeterLevel_pcnt >= 0 )
         {  pRC->iALCMeterLevel_pcnt   = pMeterReport->i8ALCMeterLevel_pcnt;
         }
        if( pMeterReport->i8CompMeterLevel_dB >= 0 )
         {  pRC->iCompMeterLevel_dB    = pMeterReport->i8CompMeterLevel_dB;
         }
        if( pMeterReport->u8SupplyVoltage_100mV < 255 )
         {  pRC->iSupplyVoltage_mV     = pMeterReport->u8SupplyVoltage_100mV * 100;
         }
        if( pMeterReport->u8DrainCurrent_100mA < 255 )
         {  pRC->iDrainCurrent_mA      = pMeterReport->u8DrainCurrent_100mA  * 100;
         }
        if( pMeterReport->u8PATemperature_degC < 255 )
         {  pRC->iPATemperature_degC   = pMeterReport->u8PATemperature_degC;
         }
      }
   }
} // end RigCtrl_OnMultiFunctionMeterReportFromNetwork()

//---------------------------------------------------------------------------
void RigCtrl_AssemblePotiReport(  // less frequently sent than a "Multi Function Meter Report"...
         T_RigCtrlInstance *pRC,         // .. thus a separate struct, exchanged between server and client[s]
         T_RigCtrl_PotiReport *pPotiReport)
{
  memset( pPotiReport, 0x80, sizeof(T_RigCtrl_PotiReport) ); // set UNUSED fields (bytes) to 'invalid' (-128 %)

  pPotiReport->bStructSize = sizeof(T_RigCtrl_PotiReport); // allows future extensions at the end of T_RigCtrl_PotiReport
  if( pRC->iAudioVolume_Percent != RIGCTRL_NOVALUE_INT )
   {  pPotiReport->i8AudioVolume_pcnt = pRC->iAudioVolume_Percent;
   }
  if( pRC->iRFPowerSetting_Percent != RIGCTRL_NOVALUE_INT )
   {  pPotiReport->i8RFPower_pcnt = pRC->iRFPowerSetting_Percent;
   }
  if( pRC->iRFGain_Percent != RIGCTRL_NOVALUE_INT )
   {  pPotiReport->i8RFGain_pcnt = pRC->iRFGain_Percent;
   }
  if( pRC->iSquelchLevel_Percent != RIGCTRL_NOVALUE_INT )
   {  pPotiReport->i8SquelchLevel_pcnt = pRC->iSquelchLevel_Percent;
   }
  if( pRC->iNoiseBlanker_Percent != RIGCTRL_NOVALUE_INT )
   {  pPotiReport->i8NoiseBlanker_pcnt= pRC->iNoiseBlanker_Percent;
   }
  if( pRC->iNoiseReduction_Percent != RIGCTRL_NOVALUE_INT )
   {  pPotiReport->i8NoiseReduction_pcnt= pRC->iNoiseReduction_Percent;
   }
  if( pRC->iPassbandTuningPos1_Percent != RIGCTRL_NOVALUE_INT )
   {  pPotiReport->i8PassbandTuningPos1_pcnt = pRC->iPassbandTuningPos1_Percent;
   }
  if( pRC->iPassbandTuningPos2_Percent != RIGCTRL_NOVALUE_INT )
   {  pPotiReport->i8PassbandTuningPos2_pcnt = pRC->iPassbandTuningPos2_Percent;
   }
  if( (pRC->iCWPitch_Hz > 0) && (pRC->iCWPitch_Hz < (127*25) ) )
   {  pPotiReport->i8CWPitch_25Hz = pRC->iCWPitch_Hz / 25;
   }
  if( (pRC->iKeyerSpeed_WPM > 0) && (pRC->iKeyerSpeed_WPM < 200) )
   {  pPotiReport->i8KeyerSpeed_WPM = pRC->iKeyerSpeed_WPM;
   }
  if( (pRC->iNotchPos_Percent >= 0) && (pRC->iNotchPos_Percent <= 100) )
   {  pPotiReport->i8NotchPos_pcnt = pRC->iNotchPos_Percent;
   }
  if( pRC->iMicGain_Percent != RIGCTRL_NOVALUE_INT )
   {  pPotiReport->i8MicGain_pcnt = pRC->iMicGain_Percent;
   }
  if( pRC->iCompSetting_Percent != RIGCTRL_NOVALUE_INT )
   {  pPotiReport->i8Comp_pcnt = pRC->iCompSetting_Percent;
   }
  if( pRC->iBreakInDelay_Percent != RIGCTRL_NOVALUE_INT )
   {  pPotiReport->i8BreakInDelay_pcnt = pRC->iBreakInDelay_Percent;
   }
  if( pRC->iMonitorGain_Percent != RIGCTRL_NOVALUE_INT )
   {  pPotiReport->i8MonitorGain_pcnt = pRC->iMonitorGain_Percent;
   }
  if( pRC->iVoxGain_Percent != RIGCTRL_NOVALUE_INT )
   {  pPotiReport->i8VoxGain_pcnt = pRC->iVoxGain_Percent;
   }
  if( pRC->iAntiVoxGain_Percent != RIGCTRL_NOVALUE_INT )
   {  pPotiReport->i8AntiVoxGain_pcnt = pRC->iAntiVoxGain_Percent;
   }
  if( pRC->iBrightness_Percent != RIGCTRL_NOVALUE_INT ) // "hey sysop, your remotely operated IC-9700 is burning up its white LEDs"
   {  pPotiReport->i8Brightness_pcnt = pRC->iBrightness_Percent;
   }

} // end RigCtrl_AssemblePotiReport()

//--------------------------------------------------------------------------
void RigCtrl_OnPotiReportFromNetwork(T_RigCtrlInstance *pRC, T_RigCtrl_PotiReport *pPotiReport)
  // Called from CwNet.c on reception of a packet with CWNET_CMD_POTI_REPORT.
  // First used in the 'Remote CW Keyer', CLIENT SIDE (without a real radio),
  // where a 'Rig Control' instance only stores parameters for the display in
  // the GUI, but doesn't control a locally connected radio.
{
  if( (pPotiReport->bStructSize > 1) && (pPotiReport->i8AudioVolume_pcnt>0) )
   { pRC->iAudioVolume_Percent = pPotiReport->i8AudioVolume_pcnt;
   }
  if( (pPotiReport->bStructSize > 2) && (pPotiReport->i8RFPower_pcnt>0) )
   { pRC->iRFPowerSetting_Percent = pPotiReport->i8RFPower_pcnt;
   }
  if( (pPotiReport->bStructSize > 3) && (pPotiReport->i8RFGain_pcnt>0) )
   { pRC->iRFGain_Percent = pPotiReport->i8RFGain_pcnt;
   }
  if( (pPotiReport->bStructSize > 4) && (pPotiReport->i8SquelchLevel_pcnt>0) )
   { pRC->iSquelchLevel_Percent = pPotiReport->i8SquelchLevel_pcnt;
   }
  if( (pPotiReport->bStructSize > 5) && (pPotiReport->i8NoiseBlanker_pcnt>0) )
   { pRC->iNoiseBlanker_Percent = pPotiReport->i8NoiseBlanker_pcnt;
   }
  if( (pPotiReport->bStructSize > 6) && (pPotiReport->i8NoiseReduction_pcnt>0) )
   { pRC->iNoiseReduction_Percent = pPotiReport->i8NoiseReduction_pcnt;
   }
  if( (pPotiReport->bStructSize > 7) && (pPotiReport->i8PassbandTuningPos1_pcnt>0) )
   { pRC->iPassbandTuningPos1_Percent = pPotiReport->i8PassbandTuningPos1_pcnt;
   }
  if( (pPotiReport->bStructSize > 8) && (pPotiReport->i8PassbandTuningPos2_pcnt>0) )
   { pRC->iPassbandTuningPos2_Percent = pPotiReport->i8PassbandTuningPos2_pcnt;
   }
  if( (pPotiReport->bStructSize > 9) && (pPotiReport->i8CWPitch_25Hz>0) )
   { pRC->iCWPitch_Hz = 25 * pPotiReport->i8CWPitch_25Hz;
   }
  if( (pPotiReport->bStructSize > 10) && (pPotiReport->i8MicGain_pcnt>0) )
   { pRC->iMicGain_Percent = pPotiReport->i8MicGain_pcnt;
   }
  if( (pPotiReport->bStructSize > 11) && (pPotiReport->i8KeyerSpeed_WPM>0) )
   { pRC->iKeyerSpeed_WPM = pPotiReport->i8KeyerSpeed_WPM;
   }
  if( (pPotiReport->bStructSize > 12) && (pPotiReport->i8NotchPos_pcnt>0) )
   { pRC->iNotchPos_Percent = pPotiReport->i8NotchPos_pcnt;
   }
  if( (pPotiReport->bStructSize > 13) && (pPotiReport->i8Comp_pcnt>0) )
   { pRC->iCompSetting_Percent = pPotiReport->i8Comp_pcnt;
   }
  if( (pPotiReport->bStructSize > 14) && (pPotiReport->i8BreakInDelay_pcnt>0) )
   { pRC->iBreakInDelay_Percent = pPotiReport->i8BreakInDelay_pcnt;
   }
  if( (pPotiReport->bStructSize > 15) && (pPotiReport->i8MonitorGain_pcnt>0) )
   { pRC->iMonitorGain_Percent = pPotiReport->i8MonitorGain_pcnt;
   }
  if( (pPotiReport->bStructSize > 16) && (pPotiReport->i8VoxGain_pcnt>0) )
   { pRC->iVoxGain_Percent = pPotiReport->i8VoxGain_pcnt;
   }
  if( (pPotiReport->bStructSize > 17) && (pPotiReport->i8AntiVoxGain_pcnt>0) )
   { pRC->iAntiVoxGain_Percent = pPotiReport->i8AntiVoxGain_pcnt;
   }
  if( (pPotiReport->bStructSize > 18) && (pPotiReport->i8Brightness_pcnt>0) )
   { pRC->iBrightness_Percent = pPotiReport->i8Brightness_pcnt;
   }

} // end RigCtrl_OnPotiReportFromNetwork()

//--------------------------------------------------------------------------
void RigCtrl_OnBandStackingRegisterReportFromNetwork( T_RigCtrlInstance *pRC, const char *pszSource )
  // Called from CwNet.c on reception of a packet with CWNET_CMD_BAND_STACKING_REGISTER.
  // First used in the 'Remote CW Keyer', CLIENT SIDE (without a real radio),
  // where a 'Rig Control' instance only stores parameters for the display in
  // the GUI (in this case, the combined "band selection combo" / "band-stacking registers"),
  // but doesn't control a locally connected radio.
  // [out] pRC->BandStackingRegs[RIGCTRL_NUM_BAND_STACKING_REGS],
  //       pRC->iNumBandStackingRegs (if the parsed ARRAY INDEX exceeds this)
  //       pRC->fBandStackingRegsModified  (flag possibly polled by the GUI to redraw itself if necessary)
  //  The REMOTE CW KEYER (GUI application) will show the result
  //  in the 'band list' on the TRX tab. For the instance running
  //  as CLIENT, the band stacking registers (or frequency bands)
  //  reported by the remote server override any local configuration,
  //  because only THE SERVER (and the server's sysop) has control
  //  over the bands supported by the station, both RX- and TX-wise.
  //
  //  The counterpart that SENDS these band-stacking registers
  //  via TCP/IP is in Remote_CW_Keyer/CwNet.c : CwNet_OnPoll() .
  //
  //  Details / musings:
  //   When available, the BAND STACKING REGISTERS are preferred over the
  //        TRANSMIT BAND FREQUENCY EDGES,
  //   because an IC-7300 modified for transmission on 60 meters ("diode mod")
  //   only reports ONE TRANSMIT BAND, covering 100 kHz to 74.8 MHz
  //   - see case RIGCTRL_POLLSTATE_TX_BAND_EDGES /
  //      "surprise surprise, the IC-7300 is a monoband rig now" !
  //
  //
{ T_RigCtrlFreqMemEntry freq_mem_entry;
  int iArrayIndex = 0, nEntriesInArray = -1;
  RigCtrl_InitFreqMemEntry( &freq_mem_entry );
  if( RigCtrl_StringToFreqMemEntry( &pszSource, &freq_mem_entry, &iArrayIndex, &nEntriesInArray ) > 0 )
   { if( (iArrayIndex>=0) && (iArrayIndex<RIGCTRL_NUM_BAND_STACKING_REGS) )
      { if( memcmp( &freq_mem_entry, &pRC->BandStackingRegs[iArrayIndex],
                    sizeof(freq_mem_entry) ) != 0 ) // MODIFIED ? Copy and inform the GUI..
         { pRC->BandStackingRegs[iArrayIndex] = freq_mem_entry;
           if( nEntriesInArray >= 0 ) // the SENDER was so kind to also report the TOTAL number of items..
            { pRC->iNumBandStackingRegs = nEntriesInArray;
              if( (iArrayIndex+1) == nEntriesInArray ) // all items in the zero-based array are through ->
               { pRC->fBandStackingRegsModified = TRUE; // flag for the GUI (running in another thread)
                 // '--> An example for 'consuming' this flag is in
                 // Remote_CW_Keyer/Keyer_Main.cpp : TKeyerMainForm::UpdateTrxDisplay() .
               }
            }
           else // the SENDER seems to use an older software version,
            {   // which didn't report the NUMBER OF ITEMS in the array. So:
              pRC->iNumBandStackingRegs = iArrayIndex+1;
              pRC->fBandStackingRegsModified = TRUE; // flag for the GUI (running in another thread)
            }
         }
      }
   }
} // end RigCtrl_OnBandStackingRegisterReportFromNetwork()

//--------------------------------------------------------------------------
void RigCtrl_OnUserDefinedBandReportFromNetwork( T_RigCtrlInstance *pRC, const char *pszSource )
  // Called from CwNet.c on reception of a packet with CWNET_CMD_USER_DEFINED_BAND.
  // First used in the RCW-Keyer, CLIENT SIDE (without a real radio),
  // where a 'Rig Control' instance only stores parameters for the display in
  // the GUI, but doesn't control a locally connected radio.
  // [out] pRC->UserDefinedBand[RIGCTRL_NUM_USER_DEFINED_BANDS],
  //       pRC->iNumUserDefinedBands (if the parsed ARRAY INDEX exceeds this)
  //       pRC->fUserDefinedBandsModified (flag possibly polled by the GUI to redraw something)
{ T_RigCtrlFreqMemEntry freq_mem_entry;
  int iArrayIndex = 0, nEntriesInArray = -1;
  RigCtrl_InitFreqMemEntry( &freq_mem_entry );
  if( RigCtrl_StringToFreqMemEntry( &pszSource, &freq_mem_entry, &iArrayIndex, &nEntriesInArray ) > 0 )
   { if( (iArrayIndex>=0) && (iArrayIndex<RIGCTRL_NUM_BAND_STACKING_REGS) )
      { if( memcmp( &freq_mem_entry, &pRC->BandStackingRegs[iArrayIndex],
                    sizeof(freq_mem_entry) ) != 0 ) // MODIFIED ? Copy and inform the GUI..
         { pRC->BandStackingRegs[iArrayIndex] = freq_mem_entry;
           if( nEntriesInArray >= 0 ) // the SENDER was so kind to also report the TOTAL number of items..
            { pRC->iNumBandStackingRegs = nEntriesInArray;
              if( (iArrayIndex+1) == nEntriesInArray ) // all items in the zero-based array are through ->
               { pRC->fBandStackingRegsModified = TRUE; // flag for the GUI (running in another thread)
                 // '--> An example for 'consuming' this flag is in
                 // Remote_CW_Keyer/Keyer_Main.cpp : TKeyerMainForm::UpdateTrxDisplay() .
               }
            }
           else // the SENDER seems to use an older software version,
            {   // which didn't report the NUMBER OF ITEMS in the array. So:
              pRC->iNumBandStackingRegs = iArrayIndex+1;
              pRC->fBandStackingRegsModified = TRUE; // flag for the GUI (running in another thread)
            }
         }
      }
   }
} // end RigCtrl_OnUserDefinedBandReportFromNetwork()


//--------------------------------------------------------------------------
void RigCtrl_OnParamReport_Integer( T_RigCtrlInstance *pRC,
        T_RigCtrl_ParamReport_Integer *pReport) // [in] received network packet
  // Called from the **TCP/IP SERVER THREAD** in CwNet.c on reception of a packet
  //             with CWNET_CMD_PARAM_INTEGER. For that reason,
  //     ServerThread() -> CwNet_OnReceive() -> CwNet_ExecuteCmd() -> RigCtrl_OnParamReport_Integer()
  //             must not interfere with the
  //             USB [], or whatever the server uses to control the radio.
  //             We don't want to end up with a mess of "Critical Sections"
  //             in Windows, all waiting for each other .
  //     See example with the "flow of information" between GUI, client,
  //     server, and the radio in RigCtrl_SetOperatingMode().
{ T_RigCtrl_ParamInfo *pPI = RigCtrl_GetInfoForUnifiedParameterNumber(pReport->bRigControlParameterNumber);
  BOOL fModified;
  if( pPI != NULL )
   { fModified = RigCtrl_SetParamValue_Int( pRC, pPI, pReport->i32ParameterValue );
     // ,----------------'
     // '--> Only stores the new value in the T_RigCtrlInstance,
     //       but does NOT automatically send a CI-V "Set"-command
     //       to the locally connected radio. Due to the possibly different
     //       calling threads, use another thread-safe FIFO (Queue) between
     //       RigCtrl_OnParamReport_Integer() and RigCtrl_Handler() :
     if( fModified )
      { RigCtrl_QueueUpCmdToWriteUnifiedPN( pRC, pPI->iUnifiedPN ); // here: on reception of an "Integer Parameter"
      }
   }
} // end RigCtrl_OnParamReport_Integer()


//--------------------------------------------------------------------------
void RigCtrl_OnParamReport_Double( T_RigCtrlInstance *pRC,
        T_RigCtrl_ParamReport_Double *pReport) // [in] received network packet
  // Called from CwNet.c on reception of a packet with CWNET_CMD_PARAM_DOUBLE.
  // Purpose: Same as explained in RigCtrl_OnParamReport_Integer(), but for parameters with type 'double'.
{ T_RigCtrl_ParamInfo *pPI = RigCtrl_GetInfoForUnifiedParameterNumber(pReport->bRigControlParameterNumber);
  BOOL fModified;
  if( pPI != NULL )
   { fModified = RigCtrl_SetParamValue_Double( pRC, pPI, pReport->dblParameterValue );
     if( fModified )
      { RigCtrl_QueueUpCmdToWriteUnifiedPN( pRC, pPI->iUnifiedPN );  // here: on reception of an "Double Parameter"
      }
   }
} // end RigCtrl_OnParamReport_Double()

//--------------------------------------------------------------------------
void RigCtrl_OnParamReport_String( T_RigCtrlInstance *pRC,
        T_RigCtrl_ParamReport_String *pReport) // [in] received network packet
  // Called from CwNet.c on reception of a packet with CWNET_CMD_PARAM_STRING.
{ T_RigCtrl_ParamInfo *pPI = RigCtrl_GetInfoForUnifiedParameterNumber(pReport->bRigControlParameterNumber);
  if( pPI != NULL )
   {
   }
} // end RigCtrl_OnParamReport_String()

//---------------------------------------------------------------------------
void RigCtrl_LimitInt32( long *pi32Value, long i32Min, long i32Max )
{
  if( *pi32Value > i32Max )
   {  *pi32Value = i32Max;
   }
  if( *pi32Value < i32Min )
   {  *pi32Value = i32Min;
   }
} // end RigCtrl_LimitInt32()


//---------------------------------------------------------------------------
int RigCtrl_CIV_AudioFilterBandwidthCodeToFrequency( int iBandwidthCode )
  // For details on the CRAZY ENCODING of the 2-digit BCD value
  // (here: iBandwidthCode), see the IC-9700 CI-V Reference "V4",
  // Icom document number "A7508-3EX-4", page 16.
  //      Read it, try to understand, and CRY OUT LOUD !
  // If you ask the radio for the "current IF filter width setting"
  // in mode FM, the radio will even reply with an ERROR !
  // For AM, the meaning of the 2-digit "Data" value is different
  // than any other mode (SSB/CW/RTTY).
  //  See also (functions inverse to each other) :
  //   * RigCtrl_CIV_AudioFilterBandwidthCodeToFrequency()
  //   * RigCtrl_CIV_FrequencyToAudioFilterBandwidthCode()
{
  int iFrequency_Hz = 0;
  if( iBandwidthCode <= 9 ) // IF filter bandwidth codes 0 .. 9 :
   { iFrequency_Hz = 50/*Hz*/ + 50 * iBandwidthCode; // 50 Hz units for narrow-band SSB/CW/RTTY
   }
  else if( iBandwidthCode <= 40 ) // IF filter bandwidth codes 10 .. 40 (SSB/CW ONLY):
   { iFrequency_Hz = 600/*Hz*/ + 100 * (iBandwidthCode-10); // 100 Hz units for 600 Hz and above
   }
  else // stupid filter bandwidth codes above "40" only occur in AM (?)
   { iFrequency_Hz = 200/*Hz*/ + 200 * iBandwidthCode;  // "200 Hz .. 10 kHz in 200-Hz-steps" (buaaaah)
   }
  return iFrequency_Hz;
} // end RigCtrl_CIV_AudioFilterBandwidthCodeToFrequency()

//---------------------------------------------------------------------------
int RigCtrl_CIV_FrequencyToAudioFilterBandwidthCode( int iFrequency_Hz )
  // Inverse to RigCtrl_CIV_AudioFilterBandwidthCodeToFrequency() .
  //             '--> Details THERE .
{
  int iBandwidthCode = 0;
  if( iFrequency_Hz <= 500 )
   { // IF filter bandwidth codes 0 .. 9 :  Audio filter bandwidths 50 .. 500 Hz
     // iFrequency_Hz = 50/*Hz*/ + 50 * iBandwidthCode; // 50 Hz units for narrow-band SSB/CW/RTTY
     // 50 Hz + 50 Hz * 9 = 500 Hz.
     iBandwidthCode = (iFrequency_Hz-50) / 50;
   }
  else if( iFrequency_Hz <= 3600 )
   { // IF filter bandwidth codes 10 .. 40 (SSB/CW ONLY):
     // iFrequency_Hz = 600/*Hz*/ + 100 * (iBandwidthCode-10); // 100 Hz units for 600 Hz and above
     // 600 Hz + 100 * (40-10) = 3600 Hz .
     iBandwidthCode = 10 + (iFrequency_Hz-600) / 100;
   }
  else // stupid filter bandwidth codes above "40" only occur in AM (?)
   { // iFrequency_Hz = 200/*Hz*/ + 200 * iBandwidthCode;  // "200 Hz .. 10 kHz in 200-Hz-steps" (buaaaah)
     iBandwidthCode = (iFrequency_Hz-200) / 200;
   }
  return iBandwidthCode;
} // end RigCtrl_CIV_FrequencyToAudioFilterBandwidthCode()


//---------------------------------------------------------------------------
T_RigCtrlAudioFilterParams *RigCtrl_GetAudioFilterParams( T_RigCtrlInstance *pRC,
  int iFilterIndex, // [in] 0 for Icom's "FIL1" .. 2 for Icom's "FIL3", or maybe more
  int iOpMode )     // [in] RIGCTRL_OPMODE_CW/LSB/USB, and maybe more
  // Returns NULL if the 'audio filter parameters' for the requested combination
  // of "filter index" (Icom: "FIL1".."FIL3") and OP MODE (Icom: SSB, FM, FM, CW, RTTY)
  // have not been SET yet (e.g. when a curious EXTERNAL CLIENT asked for it,
  //  or even tried to MODIFY an audio filter by its own gusto
  //  - see RigControl_CIV_Server.c : RigCtrl_CIV_Server_OnReadCommand(), etc)
{ int i;
  T_RigCtrlAudioFilterParams *pAFP;
  for(i=0; i<RIGCTRL_NUM_AUDIO_FILTER_PARAMS; ++i)
   { pAFP = &pRC->sAudioFilterParams[i];
     if( (pAFP->iFilterIndex == iFilterIndex) && (pAFP->iOpMode == iOpMode) )
      { return pAFP;
      }
   }
  return NULL;
} // end RigCtrl_GetAudioFilterParams()

//---------------------------------------------------------------------------
T_RigCtrlAudioFilterParams *RigCtrl_AllocAudioFilterParam(T_RigCtrlInstance *pRC)
  // Returns NULL if the pool of 'audio filter parameters' is exhausted !
  // (with RIGCTRL_NUM_AUDIO_FILTER_PARAMS = 16,
  //  and only THREE FILTERS per CW, SSB, and a few exotic modes,
  //  that's unlikely to happen, but be prepared..)
{ int i;
  T_RigCtrlAudioFilterParams *pAFP;
  for(i=0; i<RIGCTRL_NUM_AUDIO_FILTER_PARAMS; ++i)
   { pAFP = &pRC->sAudioFilterParams[i];
     if( (pAFP->iFilterIndex == 0) && (pAFP->iOpMode == 0) ) // here's an UNUSED entry..
      { return pAFP; // .. so "allocate" it from the pool (fixed-size array)
      }
   }
  return NULL;
} // end RigCtrl_AllocAudioFilterParam()


//---------------------------------------------------------------------------
int RigCtrl_GetAudioFilterBandwidth( T_RigCtrlInstance *pRC,
  int iFilterIndex, // [in] 0 for Icom's "FIL1" .. 2 for Icom's "FIL3", or maybe more
  int iOpMode )     // [in] RIGCTRL_OPMODE_CW/LSB/USB, and maybe more
  // Added 2025-08-02 to fix a problem with the FILTER BANDWIDTH
  // when WSJT-X (as remote client on one of RCW Keyer's SERVER PORTS)
  // tried to read the FILTER BANDWIDTH, as obvious in the traffic log:
  // > r2 007 FE FE 94 E0 26 00 FD       ; SelVFOMode ?
  // > TX 007 FE FE 94 E0 26 00 FD       ; SelVFOMode ?
  // > RX 00A FE FE E0 94 26 00 01 .. ; SelVFO=USB,FIL1,Data=0
  // > t2 00A FE FE E0 94 26 00 01 .. ; SelVFO=USB,FIL1,Data=0
  // > r2 007 FE FE 94 E0 1A 03 FD         ; FilterBW ?
  // > t2 008 FE FE E0 94 1A 03 09 FD ; FilterBW 500 Hz
  // ,-------------------------------------------|_|
  // '--> Wrong answer (not forwarded but taken from pRC->iFilterBW_Hz) .
  // How to implement this properly ?
  //  Command 0x1A, sub-cmd 0x03 = "Send/read the selected filter width"
  //         must somehow take the DSP filter configuration into acount,
  //         for what was traditionally called "FIL1"(wide) .. "FIL3"(narrow).
  //         But in a modern Icom DSP transceiver, ALL THESE THREE FILTERS
  //         are configurable, subject to PASSBAND TUNING, subject to the current
  //         TRANSMIT / RECEIVE state ( SSB TBW: "WIDE"/"MID"/"NAR", RX: "SSB RF HPF/LPF", etc).
  //            |          |
  //            |          '--> One would expect an ARRAY for these three DSP filters,
  //            |               with the lower and upper cut-off frequency, but..
  //            |               NOTHING in "A7292-4EX-11", only
  //            |    > "RX HPF/LPF setting for each operating mode",
  //            |    > Command: 1A 05 0001, 0004, 0007, 0010, 0011  ("A7292-4EX-11", IC-9700)
  //            |    > Command: 1A 05 0001, 0004, 0007, 0010, 0013, 0014 ...  ("A7508-3EX-4", IC-9700)
  //            |
  //           \|/
  //         0x1A.05:
  //          0014 = "SSB TX bandwidth for wide"
  //          0015 = "SSB TX bandwidth for mid"
  //          0016 = "SSB TX bandwidth for narrow"
  // -> Conclusion: What a mess. We can only read the filter bandwidth
  //                AFTER selecting "FIL1", "FIL2", or "FIL3". Solution here:
  //    Use an ARRAY for the three "Icom RX Filters" for CW+SSB, fill them with
  //    MEANINGFUL DEFAULTS (dating back to the days of expensive crystal filters)
  //    in the INITIALISATION, and whenever someone (WSJT-X) asks for Cmd 0x1A.03,
  //    and we know WHICH of the three filters, and which modulation type is
  //    currently active, update the "RX HPF/LPF setting for each operating mode"
  //    in our array (T_RigCtrlAudioFilterParams pRC->sAudioFilterParams[]),
  //    and retrieve the value from THERE (instead of pRC->iFilterBW_Hz)
  //    when a remote client tries to read it via 0x1A sub-cmd 0x03 . Phew.
{
  T_RigCtrlAudioFilterParams *pAFP = RigCtrl_GetAudioFilterParams(pRC,iFilterIndex,iOpMode);
  T_RigCtrlAudioFilterParams myFilterParams;
  if( pAFP == NULL )
   { // Audio filter parameters have not been filled out for this Filter-Index / OpMode,
     // so make a guess (purely based on speculation)
     RigCtrl_InitAudioFilterParamsForFilterIndexAndOpMode( &myFilterParams,iFilterIndex,iOpMode);
     pAFP = &myFilterParams;
   }
  return pAFP->iUpperEdge_Hz - pAFP->iLowerEdge_Hz;
} // end RigCtrl_GetAudioFilterBandwidth()


//----------------------------------------------------------------------------
void RigCtrl_InitAudioFilterParamsForFilterIndexAndOpMode( T_RigCtrlAudioFilterParams *pAFP,
  int iFilterIndex, // [in] 0 for Icom's "FIL1" .. 2 for Icom's "FIL3", or maybe more
  int iOpMode )     // [in] RIGCTRL_OPMODE_CW/LSB/USB, and maybe more
  // [out] pAFP->iUpperEdge_Hz, iLowerEdge_Hz, iFilterIndex, iOpMode,
  //       and anything we may need in future that possibly "affects"
  //       an audio filter in a modern Icom transceiver (IC-7300, IC-9700, etc)
{
  int iFilterIndex2 = iFilterIndex;
  pAFP->iFilterIndex = iFilterIndex; // save these two "table keys" for later
  pAFP->iOpMode      = iOpMode;

  // What used to be "CW-Narrow" or "CW-Very-Narrow" in the days of
  // expensive crystal IF filters are now just the freely configurable
  // "FIL1" / "FIL2" / "FIL3" in a modern Icom DSP rig.
  // Traditionally, "FIL1" was the widest filter, and that's what Icom
  // (at least for IC-7300, IC-9700, IC-705) use when you "Reset" a DSP
  // filter via the "DEF"-softkey on the filter configration screen.
  // Thus:
  switch( iOpMode & (RIGCTRL_OPMODE_NARROW | RIGCTRL_OPMODE_VERY_NARROW) )
   { case RIGCTRL_OPMODE_NARROW :
          iFilterIndex2 = 1;
          break;
     case RIGCTRL_OPMODE_VERY_NARROW :
          iFilterIndex2 = 2;
          break;
     default : break; // trust the "filter index" (function argument)
   } // switch( iOpMode & (RIGCTRL_OPMODE_NARROW | RIGCTRL_OPMODE_VERY_NARROW) )
  switch( iOpMode & ~(RIGCTRL_OPMODE_REVERSE | RIGCTRL_OPMODE_NARROW | RIGCTRL_OPMODE_VERY_NARROW ) )
   { case RIGCTRL_OPMODE_CW  :
        switch( iFilterIndex2 )
         { case 0:  // "FIL1" as the traditional WIDE CW FILTER:
           default:
                pAFP->iLowerEdge_Hz = 325;  // seen in an IC-7300 with "CW Pitch" = 630 Hz ..
                pAFP->iUpperEdge_Hz = 1525;
                break;
           case 1:  // "FIL2" as the traditional MEDIUM CW FILTER:
                pAFP->iLowerEdge_Hz = 380;  // seen in an IC-7300 with "CW Pitch" = 630 Hz ..
                pAFP->iUpperEdge_Hz = 880;
                break;
           case 2:  // "FIL3" as the traditional NARROW CW FILTER:
                pAFP->iLowerEdge_Hz = 505;  // seen in an IC-7300 with "CW Pitch" = 630 Hz ..
                pAFP->iUpperEdge_Hz = 755;
                break;
         }
        break; // end case "CW" (including CWN, CWNN, CW-Reverse, etc etc)
     case RIGCTRL_OPMODE_LSB :
     case RIGCTRL_OPMODE_USB :
        switch( iFilterIndex2 )
         { case 0:  // "FIL1" in SSB:
           default:
                pAFP->iLowerEdge_Hz = 0;    // seen in an IC-705 ...
                pAFP->iUpperEdge_Hz = 3000; // 3.0 kHz wide PER "DEF"[ault]
                break;
           case 1:  // "FIL2" in SSB: 2.4 kHz wide PER "DEF"[ault]
                pAFP->iLowerEdge_Hz = 300;
                pAFP->iUpperEdge_Hz = 2700;
                break;
           case 2:  // "FIL3" in SSB: 1.8 kHz wide PER "DEF"[ault]
                pAFP->iLowerEdge_Hz = 600;
                pAFP->iUpperEdge_Hz = 2700;
                break;
         }
        break; // end case "SSB" (ignore differences between USB and LSB passbands)
     case RIGCTRL_OPMODE_AM  :
        switch( iFilterIndex2 )
         { case 0:  // "FIL1" in AM:
           default:
                pAFP->iLowerEdge_Hz = 0;    // seen in an IC-705 ...
                pAFP->iUpperEdge_Hz = 3000; // 3.0 kHz wide PER "DEF"[ault]
                break;
           case 1:  // "FIL2" in AM: 2.4 kHz wide PER "DEF"[ault]
                pAFP->iLowerEdge_Hz = 300;
                pAFP->iUpperEdge_Hz = 2700;
                break;
           case 2:  // "FIL3" in AM: 1.8 kHz wide PER "DEF"[ault]
                pAFP->iLowerEdge_Hz = 600;
                pAFP->iUpperEdge_Hz = 2700;
                break;
         }
        break;
     case RIGCTRL_OPMODE_FM  :
        switch( iFilterIndex2 )
         { case 0:  // "FIL1" in FM  (here not the "IF"-width, -7500 .. +7500 Hz but "the audio" !)
           default:
                pAFP->iLowerEdge_Hz = 100;
                pAFP->iUpperEdge_Hz = 7500; // 15 kHz "wide" on the I/Q signal, which is irrelevant here
                break;
           case 1:  // "FIL2" in FM:
                pAFP->iLowerEdge_Hz = 100;
                pAFP->iUpperEdge_Hz = 5000;
                break;
           case 2:  // "FIL3" in FM:
                pAFP->iLowerEdge_Hz = 100;
                pAFP->iUpperEdge_Hz = 3500;
                break;
         }
        break;
     case RIGCTRL_OPMODE_FM_WIDE: // 200 kHz wide FM-Radio-IF-Filter, which is irrelevant here, for the AUDIO BANDWIDTH:
        pAFP->iLowerEdge_Hz = 50;
        pAFP->iUpperEdge_Hz = 16000;
        break;
     case RIGCTRL_OPMODE_RTTY:
     case RIGCTRL_OPMODE_PSK :
        switch( iFilterIndex2 )
         { case 0:  // "FIL1" in RTTY, seen in an IC-705 :
           default:
                pAFP->iLowerEdge_Hz = 1000;
                pAFP->iUpperEdge_Hz = 3400;
                break;
           case 1:  // "FIL2" in RTTY:  obviously also 500 Hz WIDE (as in CW)
                pAFP->iLowerEdge_Hz = 1960;
                pAFP->iUpperEdge_Hz = 2460;
                break;
           case 2:  // "FIL3" in RTTY:  obviously also 500 Hz WIDE (as in CW)
                pAFP->iLowerEdge_Hz = 2085;
                pAFP->iUpperEdge_Hz = 2335;
                break;
         }
        break;
     case RIGCTRL_OPMODE_DIGITAL_VOICE: // "Digital Voice", presumably D-Star, not DMR
     case RIGCTRL_OPMODE_DIGITAL_DATA : // "DD" seen in the IC-9700 CI-V Reference Guide (it's NOT "digital voice") !
     default: // don't care, nobody will MANUALLY select a filter for these anyway:
        pAFP->iLowerEdge_Hz = 100;
        pAFP->iUpperEdge_Hz = 3500; // again: This is the AUDIO FILTER, not the quadrature digital IF filter !
        break;
   } // end switch( iOpMode & .. )

  if( pAFP->iLowerEdge_Hz >= pAFP->iUpperEdge_Hz ) // fall-back if still not valid:
   {  pAFP->iLowerEdge_Hz = 300;
      pAFP->iUpperEdge_Hz = 2700;
   }

} // end RigCtrl_InitAudioFilterParamsForFilterIndexAndOpMode()


//--------------------------------------------------------------------------
const char *RigCtrl_VfoFrequencyToString( double dblVfoFrequency, char *psz15Buffer )
       // [in] frequency in HERTZ,   [in] e.g. char sz15Temp[15+1],
       // [out] e.g. "144.050 MHz", sufficiently short for the log/debug display
{
  char *pszDest = psz15Buffer; // no evil statics here.. buffer provided by CALLER, thread-safe
  const char *pszEnd = psz15Buffer+15;
  if( dblVfoFrequency == RIGCTRL_NOVALUE_DOUBLE )
   { SL_AppendPrintf( &pszDest, pszEnd, "<invalid freq>" );
   }
  else
   { SL_AppendPrintf( &pszDest, pszEnd, "%.5lf MHz", dblVfoFrequency * 1e-6 );
   }
  return psz15Buffer;
} // end RigCtrl_VfoFrequencyToString()


//--------------------------------------------------------------------------
const char* RigCtrl_ErrorCodeToString(int iRigctrlErrorCode)
{
  static char sz10Result[16]; // If it's one of the known error codes, don't use this evil static buffer !

  if( iRigctrlErrorCode < 0 )
   {  iRigctrlErrorCode = -iRigctrlErrorCode;
   }
  switch( iRigctrlErrorCode )
   { case RIGCTRL_ERROR_OK  : return "ok"; // No error, operation completed sucessfully
     case RIGCTRL_ERROR_INVALID_PARAM   : return "invalid param";
     case RIGCTRL_ERROR_INVALID_CONFIG  : return "invalid config";
     case RIGCTRL_ERROR_OUT_OF_MEMORY   : return "out of memory";
     case RIGCTRL_ERROR_NOT_IMPLEMENTED : return "not implemented";
     case RIGCTRL_ERROR_TIMEOUT         : return "timeout";
     case RIGCTRL_ERROR_INPUT_OUTPUT    : return "I/O error";
     case RIGCTRL_ERROR_INTERNAL        : return "internal error";
     case RIGCTRL_ERROR_PROTOCOL        : return "protocol error";
     case RIGCTRL_ERROR_CMD_REJECTED    : return "command rejected";
     case RIGCTRL_ERROR_TRUNCATED       : return "truncated";
     case RIGCTRL_ERROR_NOT_AVAILABLE   : return "not available";
     case RIGCTRL_ERROR_NOT_TARGETABLE  : return "not targetable"; // "VFO not targetable" ?
     case RIGCTRL_ERROR_BUS             : return "bus error";
     case RIGCTRL_ERROR_RIG_BUS_BUSY    : return "bus busy";
     case RIGCTRL_ERROR_INVALID_POINTER : return "invalid pointer";
     case RIGCTRL_ERROR_INVALID_VFO     : return "invalid VFO";
     case RIGCTRL_ERROR_OUT_OF_RANGE    : return "arg out of range";
     case RIGCTRL_ERROR_CMD_IN_PROGRESS : return "command in progress";
     default: break;
   }
  sprintf( sz10Result, "?%d?", (int)iRigctrlErrorCode );
  return sz10Result;
} // end RigCtrl_ErrorCodeToString()


//--------------------------------------------------------------------------
const char* RigCtrl_MsgTypeToString(int iMsgType)
{ // Mainly used for dislay in Spectrum Lab's / RCW-Keyer's "CAT traffic log".
  // Keep the returned strings short (ideally 7 or 8 characters).
  static char sz10Result[16]; // If it's one of the common types, don't use this evil static buffer !

  if( iMsgType & RIGCTRL_MSGTYPE_FLAG_ECHO )
   { return "echo";
   }

  switch( iMsgType & RIGCTRL_MSGMASK_BASIC )
   {
     case RIGCTRL_MSGTYPE_INFO     :
        return "info";
     case RIGCTRL_MSGTYPE_RADIO_ID :
        return "radio ID";
     case RIGCTRL_MSGTYPE_FREQUENCY_REPORT: // "frequency report" (usually with the VFO frequency)
        return "VFO freq";
     case RIGCTRL_MSGTYPE_SET_FREQUENCY: // "SET frequency" (command from user to radio)
        return "set freq";
     case RIGCTRL_MSGTYPE_OP_MODE  : // "mode data" (often unsolicited, "on change")
        return "op mode";
     case RIGCTRL_MSGTYPE_RIT_XIT  : // RIT or XIT data (often unsolicited, "on change")
        return "RIT/XIT";
     case RIGCTRL_MSGTYPE_RF_GAIN  : // anything that affects the RF gain (preamp, attenuator, "AIP", ..)
        return "RF gain";
     case RIGCTRL_MSGTYPE_SPECTRUM_CONFIG: // "spectrum configuration" (not WAVEFORM DATA)
        return "spec-cfg";
     case RIGCTRL_MSGTYPE_SPECTRUM : // "waveform data" (in fact, for Icom's spectrum scope)
        return "wform";              // short info because the hex field is long
     case RIGCTRL_MSGTYPE_POWER_ON_OFF:
        return "pwr ctl";
     case RIGCTRL_MSGTYPE_OTHER    : // acknowledge for 'set', read "some other parameter", etc pp.
        return "other";
     case RIGCTRL_MSGTYPE_UDP_CONTROL:
        return "UDP Ctrl";
     case RIGCTRL_MSGTYPE_UDP_SERIAL :
        return "UDP Serial";
     case RIGCTRL_MSGTYPE_UDP_AUDIO:
        return "UDP Audio";
     case RIGCTRL_MSGTYPE_UNKNOWN  : // unknown CI-V or in fact UDP (used by Icom radios via LAN or WLAN)
        return "UNKNWN";
     default:
        if( iMsgType & RIGCTRL_MSGTYPE_OK ) // in CI-V : unspecific "OK", cmd 0xFB
         { return "OK";
         }
        else if (iMsgType & RIGCTRL_MSGTYPE_NOT_OK )// in CI-V : unspecific "NG" / "Not Good", cmd 0xFA
         { return "Not OK";
         }
        // Haven't got a clue -> show the entire iMsgType, including FLAGS, in hex:
        sprintf( sz10Result, "?0x%04lX?", (unsigned long)iMsgType );
        return sz10Result; // <- not really thread-safe but ideally NEVER USED at all
   } // end switch( iMsgType )

} // end RigCtrl_MsgTypeToString()

//---------------------------------------------------------------------------
const char* RigCtrl_ServerStateToString(int iServerState)
{ switch( iServerState )
   { case RIGCTRL_SSTATE_PASSIVE         : return "passive";
     case RIGCTRL_SSTATE_PARSING_COMMAND : return "parsing_cmd";
     case RIGCTRL_SSTATE_WAIT_RADIO_NBUSY: return "wait_radio_non_busy";
     case RIGCTRL_SSTATE_FWD2RADIO_BUSY  : return "forward_busy";
     case RIGCTRL_SSTATE_FWD2RADIO_DONE  : return "forward_done";
     case RIGCTRL_SSTATE_RESPONSE_READY  : return "resp_ready";
     default: return "?unknown?";
   }
} // end RigCtrl_ServerStateToString()



//---------------------------------------------------------------------------
T_RigCtrl_RadioInfo *RigCtrl_GetRadioInfoByDefaultAddress( int iRadioCtrlProtocol, int iDefaultAddress )
{
  T_RigCtrl_RadioInfo *pRadioInfo = NULL;

  // For future extensions (Yaesu CAT protocol ?), there may be
  // different tables, not only the one for radios with Icom-compatible CI-V !
  switch( iRadioCtrlProtocol )
   {
     case RIGCTRL_PROTOCOL_NONE      : // if protocol not set yet, assume it's an Icom
     case RIGCTRL_PROTOCOL_ICOM_CI_V :
        pRadioInfo = (T_RigCtrl_RadioInfo *)RigCtrl_RadioInfo_CIV;
        break;
     case RIGCTRL_PROTOCOL_KENWOOD   : // Sorry folks... Kenwood not supported yet
        break;
#   if( SWI_SUPPORT_YAESU_5_BYTE_CAT )
     case RIGCTRL_PROTOCOL_YAESU_5_BYTE : // Yeasu's "Five-Byte-Binary-CAT" is a nightmare.. every radio is incompatible
         // (They don't give a shit for compatibility between Yasu radios :
         // > Unfortunately, there is no "master list" of Yaesu CAT commands.
         // > Most Yeasu models have unique command sets; in some cases these
         // > are similar to those of predecessors, in other cases
         // > they don't even use the same bit ordering.
         // > There are also ROM updates that introduce new commands or change
         // > the behaviour of existing commands. Yeasu's CAT documentation
         // > is rarely comprehensive or error free. (..)
         // > My current theory is that Yaesu engineers learned about computers
         // > and software from reading the backs of cereal boxes.
         //     Looks like Dave, AA6YQ, has also had a lot of fun with Yaesu.
         //     Or, as Simon HB9DRV writes in his 'Command Tester' user guide:
         // > Be aware that there are many errors in the Yaesu
         // > documentation, so the data returned may not agree with the
         // > handbook and the radios current settings.
         // More about the fun with Yaesu's lousy documentation in Yaesu5Byte.c .
         // The only radio that we possibly ever support (with this protocol)
         // is the FT-817/FT-817ND, thanks to the info collected by KA7OEI .
         // Note: In newer Yaesu radios (which the author doesn't have, and WON'T BUY),
         //       what they now call CAT is something COMPLETELY DIFFERENT,
         //       and obviously stolen from or inspired by .. tadaa ... KENWOOD:
         //       They now invented a "terminator that signals the end of the control command",
         //       so those "New CAT Control Commands" are not fixed to FIVE BYTES anymore.
         //       The FT-891 is one of those rigs with that KENWOOD-LIKE protocol (delimiter=semicolon).
         if( iDefaultAddress < RIGCTRL_DEF_ADDR_UNKNOWN_YAESU )
          {  iDefaultAddress = RIGCTRL_DEF_ADDR_UNKNOWN_YAESU;
          }
         pRadioInfo = (T_RigCtrl_RadioInfo *)RigCtrl_RadioInfo_Y5B;
         break; // end case RIGCTRL_PROTOCOL_YAESU_5_BYTE in 'RigCtrl_GetRadioInfoByDefaultAddress()'
#   endif // SWI_SUPPORT_YAESU_5_BYTE_CAT ?

     case RIGCTRL_PROTOCOL_AD9832_I2C  : // not supported HERE but somewhere else (in SpecLab)
         break;
   } // end  switch( iRadioCtrlProtocol )


  if( pRadioInfo != NULL )
   {
     // The table isn't sorted by anything, so use a linear search:
     while( pRadioInfo->pszName != NULL )  // NULL marks the end of the table
      { if( pRadioInfo->iDefaultAddress == iDefaultAddress )
         {  return pRadioInfo; // bingo, found what we're looking for
         }
        ++pRadioInfo; // check next table entry
      }
   }

  // Arrived here ? Out of luck...
  return NULL;
} // end  RigCtrl_GetRadioInfoByDefaultAddress()

//---------------------------------------------------------------------------
const char* RigCtrl_DefaultAddressToString( int iRadioCtrlProtocol, int iDefaultAddress )
{
  static char sz7Result[8];
  T_RigCtrl_RadioInfo *pRadioInfo = RigCtrl_GetRadioInfoByDefaultAddress(
                                          iRadioCtrlProtocol, iDefaultAddress );
  if( pRadioInfo != NULL )
   { return pRadioInfo->pszName;
   }
  else switch( iDefaultAddress )
   { case  RIGCTRL_DEF_ADDR_UNKNOWN_YAESU:
        return "Unknown YAESU";
     default:
        sprintf( sz7Result, "<%02X>", iDefaultAddress );
        return sz7Result;
   }
} // end RigCtrl_DefaultAddressToString()

//---------------------------------------------------------------------------
const char* RigCtrl_RigControlPortNrToString( int iRigCtrlPort )
  // [in] iRigCtrlPort : an index into T_RigControl.PortInstance[1+RIGCTRL_MAX_CLIENT_PORTS]
{ switch( iRigCtrlPort )
   { case RIGCTRL_PORT_RADIO    : return "Radio Control Port";
     case RIGCTRL_PORT_AUX_COM_1: return "Additional Port #1";
     case RIGCTRL_PORT_AUX_COM_2: return "Additional Port #2";
     case RIGCTRL_PORT_AUX_COM_3: return "Additional Port #3";
     default : return "Invalid port";
   }
} // end RigCtrl_RigControlPortNrToString()


//---------------------------------------------------------------------------
const char* RigCtrl_PortUsageToString( int iPortUsage )
{ switch( iPortUsage )
   { case RIGCTRL_PORT_USAGE_NONE             : return "None";
     case RIGCTRL_PORT_USAGE_WINKEYER_HOST    : return "Winkeyer HOST";
     case RIGCTRL_PORT_USAGE_WINKEYER_EMULATOR: return "Winkeyer EMULATOR";
     case RIGCTRL_PORT_USAGE_TEXT_TERMINAL    : return "Text Terminal";
     case RIGCTRL_PORT_USAGE_SERIAL_TUNNEL    : return "Serial Port Tunnel";
     case RIGCTRL_PORT_USAGE_VIRTUAL_RIG      : return "Virtual Rig";
     case RIGCTRL_PORT_USAGE_ECHO_TEST        : return "Echo Test";
     case RIGCTRL_PORT_USAGE_TX_STRESS_TEST   : return "TX Stress Test";
     default: return "??";
   }
} // end RigCtrl_PortUsageToString()

//---------------------------------------------------------------------------
const char* RigCtrl_GetDescriptivePortName(T_RigCtrl_PortInstance *pPortInstance)
  // Returns e.g. "Radio Port" or even "Radio:IC-7300", "Virtual Radio Port #2", etc.
  // Added 2025-08 to have consistent names in certain messages in the log, e.g.:
  // * "Traffic Log paused on Radio Control Port after initialisation."
  // * "Traffic Log paused on Virtual Radio Port #2 after an unknown command."
  // * etc, etc ...
{ int iRigCtrlPort = RigCtrl_PortInstancePtrToIndex( pPortInstance );
  static char sz40Result[44] = "";
  if( iRigCtrlPort==RIGCTRL_PORT_RADIO )
   { if( (pPortInstance->pRC->iDefaultAddress != RIGCTRL_DEF_ADDR_AUTO_DETECT)
       &&(pPortInstance->pRC->iDefaultAddress != RIGCTRL_DEF_ADDR_NO_REMOTE_CTRL) )
      { sprintf( sz40Result, "RadioPort/%s",  // -> e.g. "RadioPort:IC-7300"
           RigCtrl_DefaultAddressToString(pPortInstance->iRadioCtrlProtocol,
                                          pPortInstance->pRC->iDefaultAddress) );
      }
     else
      { sprintf( sz40Result, "Radio Port" );
      }
   }
  else // not the "Radio Control Port" but an "Additional Port" -> what is it used for ?
   { if( pPortInstance->fActAsServer ) // acting as server / emulating a "Virtual Radio"
      { sprintf( sz40Result, "Virtual Radio Port #%d", (int)iRigCtrlPort );
      }
     else // not acting as server (for an external client) ->
      { sprintf( sz40Result, "Additional COM Port #%d", (int)iRigCtrlPort );
      }
   }
  return sz40Result;
} // end RigCtrl_GetDescriptivePortName()


//---------------------------------------------------------------------------
const char *RigCtrl_ControlConnStateToString( T_RigCtrlInstance *pRC ) // more verbose than RigCtrl_ConnStateToString(), but mainly for the "CONTROL"-UDP-port
  // [in] pRC->iConnState    : RIGCTRL_CONNSTATE_OFF/CONNECTING/CONNECTED
  // [in] pRC->iConnSubstate : RIGCTRL_CONN_SUBSTATE_SEND_CONN_REQUEST, etc^10 ..
  // [return] e.g. "Connected to IC-705" (ideally).
  // Called from TDebugForm::Btn_GetReportOnCommsClick(),
  //    and from TSpectrumLab::MI_StartStopClick() to show the current state in SL's MAIN MENU.
{ static char sz80Result[84];
  char *cp = sz80Result, *pszEndstop = sz80Result+80;

  if( pRC==NULL )
   { return "Deleted";
   }
  switch( pRC->initState )
   { case initState_ColdStart : return "NoInit";
     case initState_Closed    : return "Stopped";  // e.g. RigCtrl_Close() called from the main menu, "Start/Stop".."Rig Control".."Stop" (thus this result)
     default : break;   // e.g. initState = 1 = "Opened" ...
   }

  // If a "real" serial port (RS-232 or single-wire CI-V adapter), or a VIRTUAL COM PORT on USB is used,
  // there's no such complex "Connection" state machine as for UDP (with the "three ports").
  // In this case, we consider ourselves "connected" as soon as we RECEIVED
  // the first properly formatted CI-V message from an Icom radio.
  SL_AppendString( &cp, pszEndstop, RigCtrl_ParameterPollingStateToString(pRC->iParameterPollingState) );
  // ,--------------------------------'
  // '--> e.g. "passive" /  "start reading" / "read VFO freq" / "read op-mode" / "start spectrum" /  "running"
  if( pRC->PortInstance[RIGCTRL_PORT_RADIO].dwNumMessagesRcvd > 0 )
   { SL_AppendPrintf( &cp, pszEndstop, ", rcvd %ld msgs",
            (long)pRC->PortInstance[RIGCTRL_PORT_RADIO].dwNumMessagesRcvd );
   }
  else
   { SL_AppendString( &cp, pszEndstop, ", no RX yet" );
   }
  return sz80Result; // end case INET_PROTOCOL_COM_PORT
} // end RigCtrl_ControlConnStateToString()

//---------------------------------------------------------------------------
const char *RigCtrl_ParameterPollingStateToString(int iParameterPollingState)
  // Added 2023-06 when, despite being successfully "connected" to all three
  // UDP ports in an IC-705, none of the essential parameters became valid.
  // Called from, and results displayed in, Spectrum Lab's "CAT traffic monitor".
{ static char sz15Result[16];
  switch( iParameterPollingState )
   { case RIGCTRL_POLLSTATE_PASSIVE             : return "passive";        // initial state: have neither READ nor WRITTEN any setting from/to the remote rig
     case RIGCTRL_POLLSTATE_START_RD            : return "start reading";  // begin reading all we need to know about/from the remote rig
     case RIGCTRL_POLLSTATE_READ_RADIO_ID       : return "read radio ID";  // (try to) read the radio's "default" ID, as a connection test and to check its type
     case RIGCTRL_POLLSTATE_PARAMS_FOR_RIGCTL   : return "read initial params";  // (try to) read the required initial parameters
     case RIGCTRL_POLLSTATE_PARAMS_FOR_HAMLIB   : return "read Hamlib params"; // (try to) read a few more params used by HamlibServer.c
     case RIGCTRL_POLLSTATE_CONFIGURE_SPECTRUM  : return "start spectrum"; // (try to) activate transmission of "spectrum scope waveform data"
     case RIGCTRL_POLLSTATE_DONE                : return "running";        // finished polling "all we need to know" about the radio; periodic polling is "running" (S-meter, VFO, Spectrum, ?)
     case RIGCTRL_POLLSTATE_TURN_RIG_ON         : return "turn rig on";    // (try to) turn the rig on via special command or "wake-up pattern"
     case RIGCTRL_POLLSTATE_WAIT_TURN_ON        : return "wait for turn-on"; // waiting for an "OK"-response after the command to "turn on"
     case RIGCTRL_POLLSTATE_TURN_RIG_OFF        : return "turn rig off";   // (try to) turn the rig on via special command (for modern Icom rigs)
     case RIGCTRL_POLLSTATE_WAIT_TURN_OFF       : return "wait for turn-off";
     case RIGCTRL_POLLSTATE_TURNED_OFF          : return "rig turned off";
     default : sprintf( sz15Result, "?%d?", (int)iParameterPollingState );
               return sz15Result;
   }
} // end RigCtrl_ParameterPollingStateToString()

//---------------------------------------------------------------------------
void RigCtrl_InitFreqMemEntry( T_RigCtrlFreqMemEntry *pFreqMemEntry )
{
  memset( pFreqMemEntry, 0x00, sizeof(T_RigCtrlFreqMemEntry) );
  // Make floating point members valid, because 0x00 0x00 0x00 0x00 isn't necessarily '0.0' :
  pFreqMemEntry->RxTx[0].dblOperatingFreq_Hz = pFreqMemEntry->RxTx[1].dblOperatingFreq_Hz = 0.0;

} // end RigCtrl_InitFreqMemEntry()

//---------------------------------------------------------------------------
void RigCtrl_InitUserDefinedBand( T_RigCtrlUserDefinedBand *pUserDefdBand )
{ memset( pUserDefdBand, 0x00, sizeof(T_RigCtrlUserDefinedBand) );
  pUserDefdBand->dblFmin_Hz = pUserDefdBand->dblFmax_Hz = pUserDefdBand->dblFdef_Hz = 0.0;
} // end RigCtrl_InitUserDefinedBand()

//---------------------------------------------------------------------------
const char* RigCtrl_GetFreqMemToken( int iToken ) // for string builder and parsers;
  // returns e.g. "na=" for the numeric token code RIGCTRL_FREQ_MEM_TOKEN_NAME
{ return SL_GetStringFromTokenList( RigCtrl_FreqMemEntryTokens, iToken );
}

//---------------------------------------------------------------------------
int RigCtrl_FreqMemEntryToString( T_RigCtrlFreqMemEntry *pFreqMemEntry,
       int iArrayIndex,     // [in] zero-based array index for easy parsing (negative for "no array")
       int nEntriesInArray, // [in] total number of entries in the array (service for the receiver to detect when all items are through)
       char *pszDest, int iMaxDestLen ) // [out] key=value pairs, comma separated
  // Returns the length of the generated string (not including the trailing zero).
  //         Format specified in RigCtrl_StringToFreqMemEntry()) .
  // In the remote CW keyer, this function is also used to send
  //    the "band-stacking registers" from server to all clients in a non-Icom-
  //    specific, easily parsable(*), extendable format.
  //    The same, easily parsable(*) format is also used for single frequencies
  //    in the optional FREQUENCIES section in RCWKeyer_Bands.txt .
  // See also (inverse functions): RigCtrl_StringToFreqMemEntry(), RigCtrl_FreqMemEntryToString() .
  // (*) parsable, not parseable, because (English teachers will love this):
  // > "Verbs drop silent final -e except after c and g (e.g. movable but changeable)"
{ char *pszDestStart = pszDest;
  char *pszEndstop = pszDest+iMaxDestLen;
  BOOL fEmitFrequencyWithToken = FALSE;

  if( iArrayIndex >= 0 )  // Emit the two 'common but optional' fields first:
   { SL_AppendPrintf( &pszDest, pszEndstop, "i=%d ", (int)iArrayIndex );
     fEmitFrequencyWithToken = TRUE;
   }
  if( nEntriesInArray >= 0 )
   { SL_AppendPrintf( &pszDest, pszEndstop, "n=%d ", (int)nEntriesInArray );
     fEmitFrequencyWithToken = TRUE;
   }
  // Most important members of the T_RigCtrlFreqMemEntry entry first,
  //  similar as in the file template for "RCWKeyer_Bands.txt", [Frequencies] section:
  //   * operating frequency, operating mode, memory channel name, e.g.:
  //     3.560  mo=CWN  na="80m CW QRP"
  //  (As a special service, the FREQUENCY doesn't need to have "fr=" as a token,
  //   because the parser recognizes it when the frequency is 'first in line'. )
  if( fEmitFrequencyWithToken )
   { SL_AppendPrintf( &pszDest, pszEndstop, "%s",
        RigCtrl_GetFreqMemToken( RIGCTRL_FREQ_MEM_TOKEN_FREQ ) );
   }
  SL_AppendPrintf( &pszDest, pszEndstop, "%lf ",
        pFreqMemEntry->RxTx[0].dblOperatingFreq_Hz * 1e-6 );  // in the STRING/file: radio frequencies in MHz !
  if( (pFreqMemEntry->RxTx[1].dblOperatingFreq_Hz != 0.0)  // DIFFERENT and VALID *transmit* frequency ("split") ?
   && (pFreqMemEntry->RxTx[1].dblOperatingFreq_Hz != pFreqMemEntry->RxTx[0].dblOperatingFreq_Hz ) )
   { SL_AppendPrintf( &pszDest, pszEndstop, "%s%lf ",
        RigCtrl_GetFreqMemToken( RIGCTRL_FREQ_MEM_TOKEN_RX_FREQ ),
        pFreqMemEntry->RxTx[1].dblOperatingFreq_Hz * 1e-6 );
   }
  if( pFreqMemEntry->RxTx[0].iOpMode != RIGCTRL_OPMODE_UNKNOWN )
   { SL_AppendPrintf( &pszDest, pszEndstop, "%s%s ",
        RigCtrl_GetFreqMemToken( RIGCTRL_FREQ_MEM_TOKEN_OP_MODE ),
        RigCtrl_OperatingModeToString(pFreqMemEntry->RxTx[0].iOpMode) );
   }
  if( pFreqMemEntry->sz16Name[0] != '\0' )
   { SL_AppendPrintf( &pszDest, pszEndstop, "%s%s ",
        RigCtrl_GetFreqMemToken( RIGCTRL_FREQ_MEM_TOKEN_NAME ), pFreqMemEntry->sz16Name );
   }
  if( pFreqMemEntry->RxTx[0].fDataMode != 0 )
   { SL_AppendPrintf( &pszDest, pszEndstop, "%s%d ",
        RigCtrl_GetFreqMemToken( RIGCTRL_FREQ_MEM_TOKEN_DATA_MODE ),
        (int)pFreqMemEntry->RxTx[0].fDataMode ); // at least for IC-7300, only 0=FALSE or 1=TRUE, but things may change
   }
  if( pFreqMemEntry->RxTx[0].iDuplexMode != 0 )
   { SL_AppendPrintf( &pszDest, pszEndstop, "%s%d ",
        RigCtrl_GetFreqMemToken( RIGCTRL_FREQ_MEM_TOKEN_DUPL_MODE ),
        pFreqMemEntry->RxTx[0].iDuplexMode );
   }
  if( pFreqMemEntry->RxTx[0].iToneType != 0 )
   { SL_AppendPrintf( &pszDest, pszEndstop, "%s%d ",
        RigCtrl_GetFreqMemToken( RIGCTRL_FREQ_MEM_TOKEN_TONE_TYPE ),
        pFreqMemEntry->RxTx[0].iToneType );
   }
  if( pFreqMemEntry->RxTx[0].iDigitalSquelch != 0 )
   { SL_AppendPrintf( &pszDest, pszEndstop, "%s%d ",
        RigCtrl_GetFreqMemToken( RIGCTRL_FREQ_MEM_TOKEN_DIG_SQLCH ),
        pFreqMemEntry->RxTx[0].iDigitalSquelch );
   }
  if( pFreqMemEntry->RxTx[0].iRepeaterToneFreq_Hz != 0 )
   { SL_AppendPrintf( &pszDest, pszEndstop, "%s%d ",
        RigCtrl_GetFreqMemToken( RIGCTRL_FREQ_MEM_TOKEN_RPTR_TONE ),
        pFreqMemEntry->RxTx[0].iRepeaterToneFreq_Hz );
   }
  if( pFreqMemEntry->RxTx[0].iToneSquelchFreq_Hz != 0 )
   { SL_AppendPrintf( &pszDest, pszEndstop, "%s%d ",
        RigCtrl_GetFreqMemToken( RIGCTRL_FREQ_MEM_TOKEN_TSQL_FREQ ),
        pFreqMemEntry->RxTx[0].iToneSquelchFreq_Hz );
   }
  if( pFreqMemEntry->RxTx[0].iDTCS_Code != 0 ) // Icom slang: "DTCS code settings", whatever that means
   { SL_AppendPrintf( &pszDest, pszEndstop, "%s%d ",
        RigCtrl_GetFreqMemToken( RIGCTRL_FREQ_MEM_TOKEN_DTCS_CODE ),
        pFreqMemEntry->RxTx[0].iDTCS_Code );
   }
  if( pFreqMemEntry->RxTx[0].iDV_DigitalCodeSquelch != 0 ) // Icom slang: "DTCS code settings", whatever that means
   { SL_AppendPrintf( &pszDest, pszEndstop, "%s%d ",
        RigCtrl_GetFreqMemToken( RIGCTRL_FREQ_MEM_TOKEN_D_C_SQLCH ),
        pFreqMemEntry->RxTx[0].iDV_DigitalCodeSquelch );
   }
  if( pFreqMemEntry->RxTx[0].iDuplexOffsetFreq_Hz != 0 )
   { SL_AppendPrintf( &pszDest, pszEndstop, "%s%d ",
        RigCtrl_GetFreqMemToken( RIGCTRL_FREQ_MEM_TOKEN_DUP_OFFS ),
        pFreqMemEntry->RxTx[0].iDuplexOffsetFreq_Hz );
   }
  if( pFreqMemEntry->RxTx[0].c8DestCall[0] != '\0' ) // aka "UR" (destination call sign for digital voice)
   { SL_AppendPrintf( &pszDest, pszEndstop, "%s%s ",
        RigCtrl_GetFreqMemToken( RIGCTRL_FREQ_MEM_TOKEN_DEST_CALL ),
        pFreqMemEntry->RxTx[0].c8DestCall );
   }
  if( pFreqMemEntry->RxTx[0].c8AccessRptrCall[0] != '\0' ) // "R1 (Access repeater) call sign
   { SL_AppendPrintf( &pszDest, pszEndstop, "%s%s ",
        RigCtrl_GetFreqMemToken( RIGCTRL_FREQ_MEM_TOKEN_RPTR_CALL ),
        pFreqMemEntry->RxTx[0].c8AccessRptrCall );
   }
  if( pFreqMemEntry->RxTx[0].c8GatewayRptrCall[0] != '\0') // "R2 (Gateway/Link repeater) call sign
   { SL_AppendPrintf( &pszDest, pszEndstop, "%s%s ",
        RigCtrl_GetFreqMemToken( RIGCTRL_FREQ_MEM_TOKEN_GTWY_CALL ),
        pFreqMemEntry->RxTx[0].c8GatewayRptrCall );
   }

  // Remove the LAST SPACE in the long list of key=value pairs assembled above:
  if( (pszDest > pszDestStart) && (pszDest[-1]==' ') )
   { pszDest[-1] = '\0';
     --pszDest;
   }

  return pszDest - pszDestStart;
} // end RigCtrl_FreqMemEntryToString()

const T_SL_TokenList RigCtrl_FreqMemEntryTokens[] = // .. to convert T_RigCtrlFreqMemEntry into a string and back
{ { "i=",    RIGCTRL_FREQ_MEM_TOKEN_INDEX   }, // "i=" -> array index (not a part of the struct, but an extra function argument)
  { "n=",    RIGCTRL_FREQ_MEM_TOKEN_N_ITEMS }, // "n=" -> total number of items in the array
  { "na=",   RIGCTRL_FREQ_MEM_TOKEN_NAME    }, // "na=" -> pFreqMemEntry->c16MemoryName
  { "fr=",   RIGCTRL_FREQ_MEM_TOKEN_FREQ    }, // "fr=" -> pFreqMemEntry->RxTx[0].dblOperatingFreq_Hz
  { "rx=",   RIGCTRL_FREQ_MEM_TOKEN_RX_FREQ }, // "rx=" -> pFreqMemEntry->RxTx[1].dblOperatingFreq_Hz
  { "mo=",   RIGCTRL_FREQ_MEM_TOKEN_OP_MODE }, // "mo=" -> pFreqMemEntry->RxTx[0].iOpMode
  { "dm=",   RIGCTRL_FREQ_MEM_TOKEN_DATA_MODE}, // "dm=" -> pFreqMemEntry->RxTx[0].iDataMode  ("da" reserved for "date and time")
  { "du=",   RIGCTRL_FREQ_MEM_TOKEN_DUPL_MODE}, // "du=" -> pFreqMemEntry->RxTx[0].iDuplexMode
  { "tt=",   RIGCTRL_FREQ_MEM_TOKEN_TONE_TYPE}, // "tt=" -> pFreqMemEntry->RxTx[0].iToneType
  { "ds=",   RIGCTRL_FREQ_MEM_TOKEN_DIG_SQLCH}, // "ds=" -> pFreqMemEntry->RxTx[0].iDigitalSquelch
  { "rt=",   RIGCTRL_FREQ_MEM_TOKEN_RPTR_TONE}, // "rt=" -> pFreqMemEntry->RxTx[0].iRepeaterToneFreq_Hz
  { "ts=",   RIGCTRL_FREQ_MEM_TOKEN_TSQL_FREQ}, // "ts=" -> pFreqMemEntry->RxTx[0].iToneSquelchFreq_Hz
  { "dt=",   RIGCTRL_FREQ_MEM_TOKEN_DTCS_CODE}, // "dt=" -> pFreqMemEntry->RxTx[0].iDTCS_Code
  { "cs=",   RIGCTRL_FREQ_MEM_TOKEN_D_C_SQLCH}, // "dv=" -> pFreqMemEntry->RxTx[0].iDV_DigitalCodeSquelch
  { "of=",   RIGCTRL_FREQ_MEM_TOKEN_DUP_OFFS }, // "of=" -> pFreqMemEntry->RxTx[0].iDuplexOffsetFreq_Hz
  { "UR=",   RIGCTRL_FREQ_MEM_TOKEN_DEST_CALL}, // "UR=" -> pFreqMemEntry->RxTx[0].c8DestCall        aka "UR" for digital voice
  { "R1=",   RIGCTRL_FREQ_MEM_TOKEN_RPTR_CALL}, // "R1=" -> pFreqMemEntry->RxTx[0].c8AccessRptrCall  aka "R1" for digital voice
  { "R2=",   RIGCTRL_FREQ_MEM_TOKEN_GTWY_CALL}, // "R2=" -> pFreqMemEntry->RxTx[0].c8GatewayRptrCall aka "R2" for digital voice
  { NULL, 0 } // "all zeros" mark the end of the list
};


//---------------------------------------------------------------------------
int RigCtrl_StringToFreqMemEntry( // parses a T_RigCtrlFreqMemEntry from a string
        const char **cppSrc,  // [in] line loaded from RCWKeyer_Bands.txt [Frequencies], or received from remote server,
                // e.g. "i=0 n=33  fr=7037355.0 mo=CWN ..." (space separated key=value pairs, because certain VALUES may be a comma-separated list itself)
                //       |_______| |_|
                //           |      |
                //           |      '--- optional when the line (string) BEGINS with a frequency
                //           '---------- optional, not when 'loading a file'
                // For the [Frequencies] section in RCWKeyer_Bands.txt, the string may be as simple as this:
                //                       // > [Frequencies]  ; new section: "what follows are single frequencies"..
                //      "3.560 mo=CW na=\"CW QRP\" ..."
                // When received from a server connected to a modern Icom radio (especially with Digital Voice),
                //      the to-be-parsed string may contain an awful lot of key=value pairs
                //      that we don't need for remote CW, but keep it in here...
                //      module RigControl.c may be used for very different purposes.
        T_RigCtrlFreqMemEntry *pFreqMemEntry, // [out] struct used for many "single frequencies of interest", including memory channels
        int *piArrayIndex,    // [out,optional] ARRAY INDEX (NEGATIVE if not specified in the string)
        int *nEntriesInArray) // [out,optional] total number of entries in the array (service for the receiver to detect when all items are through)
  // When successful ("valid syntax"), returns the number of characters
  // parsed from *cppSrc. This allows THE CALLER to continue parsing
  // after the "recognized" part of the source string.
  // Note: Members in T_RigCtrlFreqMemEntry for which no key (name) has
  //       been found in the list of key=value pairs will NOT be modified.
  //       If that's not intended, call RigCtrl_InitFreqMemEntry() before RigCtrl_StringToFreqMemEntry().
  // Callers (not necessarily complete):
  //  * RigCtrl_LoadBandsAndFrequenciesFromFile(), to load RCWKeyer_Bands.txt
  //  * RigCtrl_OnBandStackingRegisterReportFromNetwork(), for client/server communication
  //
  // See also (similar and inverse functions):
  //    RigCtrl_StringToFreqMemEntry(),    RigCtrl_FreqMemEntryToString(),
  //    RigCtrl_StringToUserDefinedBand(), RigCtrl_UserDefinedBandToString() .
{ const char *pszSourceStart = *cppSrc;
  const char *pszKeyStart;
  int iToken, iValue;
  double dblValue;
  BOOL fGotFrequency = FALSE;

  if( piArrayIndex != NULL )
   { *piArrayIndex = -1;   // didn't find "i=.." in the key=value pairs yet
   }
  if( nEntriesInArray != NULL )
   { *nEntriesInArray = -1;   // didn't find "n=.." in the key=value pairs yet
   }
  while( **cppSrc != '\0' ) // continue until the end of the string..
   {                        // .. or until the first unrecognized key (name)
     pszKeyStart = *cppSrc;
     iToken = SL_SkipOneOfNStrings( cppSrc, RigCtrl_FreqMemEntryTokens );
               //   '--> Skips LEADING SPACES, so no need for an extra SL_SkipSpaces()
     if( iToken < 0 )
      { if( ! fGotFrequency ) // accept the simpler format for USER-DEFINED frequencies in e.g. RCWKeyer_Bands.txt :
         { // Example: *cppSrc = "3.505 mo=CWN na=\"80m CW DX\" " as a shorter alternative,
           //     instead of  "fr=3.505 mo=CWN na=\"80m CW DX\" "
           dblValue = SL_ParseFloat( cppSrc, '.'/*cDecimalSeparator*/ ); // radio frequency in MHz
           if( (dblValue > 0 ) && (*cppSrc > pszKeyStart) )
            { pFreqMemEntry->RxTx[0].dblOperatingFreq_Hz = dblValue * 1e6;
              fGotFrequency = TRUE;
              continue; // don't enter the switch( iToken ) list further below but look for the next token
            }
         }
        break; // <- important to avoid getting stuck in the while loop
      }
     switch( iToken )
      { case RIGCTRL_FREQ_MEM_TOKEN_INDEX : // "i=<array index>" -> *piArrayIndex (if non-NULL)
             iValue = SL_ParseInteger( cppSrc );
             if( piArrayIndex != NULL )
              { *piArrayIndex = iValue;
              }
             break;
        case RIGCTRL_FREQ_MEM_TOKEN_N_ITEMS: // "n=<total number of items in the array>" -> *piArrayIndex (if non-NULL)
             iValue = SL_ParseInteger( cppSrc );
             if( nEntriesInArray != NULL )
              { *nEntriesInArray = iValue; // this is a service for the receiver to tell when "all items are through",
                // or to realize a kind of "progress bar" when transferring an awful number of items
              }
             break;
        case RIGCTRL_FREQ_MEM_TOKEN_NAME : // pFreqMemEntry->sz16Name (must be double-quoted and "escaped" in the source)
             if( SL_ParseDoubleQuotedString( cppSrc, pFreqMemEntry->sz16Name, 16/*iMaxLen*/ ) <= 0 )
              { // obviously not a double-quoted string, so only copy a single word, delimited by space, comma, etc:
                SL_CopyStringUntilDelimiter( cppSrc, pFreqMemEntry->sz16Name, 16/*iMaxLen*/, " ,\r\n"/*pszDelimiters*/ );
              }
             break;
        case RIGCTRL_FREQ_MEM_TOKEN_FREQ : // pFreqMemEntry->RxTx[0].dblOperatingFreq_Hz
             pFreqMemEntry->RxTx[0].dblOperatingFreq_Hz = 1e6 * SL_ParseFloat( cppSrc, '.'/*cDecimalSeparator*/ );
             fGotFrequency = TRUE;
             break;
        case RIGCTRL_FREQ_MEM_TOKEN_RX_FREQ : // pFreqMemEntry->RxTx[1].dblOperatingFreq_Hz
             pFreqMemEntry->RxTx[1].dblOperatingFreq_Hz = 1e6 * SL_ParseFloat( cppSrc, '.'/*cDecimalSeparator*/ );
             break;
        case RIGCTRL_FREQ_MEM_TOKEN_OP_MODE : // pFreqMemEntry->RxTx[0].iOpMode, but "as a string" to avoid magic numbers
             // inverse to RigCtrl_OperatingModeToString(pFreqMemEntry->RxTx[0].iOpMode) :
             pFreqMemEntry->RxTx[0].iOpMode = RigCtrl_StringToOperatingMode( cppSrc );
             break;
        case RIGCTRL_FREQ_MEM_TOKEN_DATA_MODE : // pFreqMemEntry->RxTx[0].iDataMode (just a boolean, 0=false, 1=true, at least for Icom)
             pFreqMemEntry->RxTx[0].fDataMode = SL_ParseInteger( cppSrc );
             break;
        case RIGCTRL_FREQ_MEM_TOKEN_DUPL_MODE : // pFreqMemEntry->RxTx[0].iDuplexMode
             pFreqMemEntry->RxTx[0].iDuplexMode = SL_ParseInteger( cppSrc );
             break;
        case RIGCTRL_FREQ_MEM_TOKEN_TONE_TYPE : // pFreqMemEntry->RxTx[0].iToneType
             pFreqMemEntry->RxTx[0].iToneType = SL_ParseInteger( cppSrc );
             break;
        case RIGCTRL_FREQ_MEM_TOKEN_DIG_SQLCH : // pFreqMemEntry->RxTx[0].iDigitalSquelch
             pFreqMemEntry->RxTx[0].iDigitalSquelch = SL_ParseInteger( cppSrc );
             break;
        case RIGCTRL_FREQ_MEM_TOKEN_RPTR_TONE : // pFreqMemEntry->RxTx[0].iRepeaterToneFreq_Hz
             pFreqMemEntry->RxTx[0].iRepeaterToneFreq_Hz = SL_ParseInteger( cppSrc );
             break;
        case RIGCTRL_FREQ_MEM_TOKEN_TSQL_FREQ : // pFreqMemEntry->RxTx[0].iToneSquelchFreq_Hz
             pFreqMemEntry->RxTx[0].iToneSquelchFreq_Hz = SL_ParseInteger( cppSrc );
             break;
        case RIGCTRL_FREQ_MEM_TOKEN_DTCS_CODE : // pFreqMemEntry->RxTx[0].iDTCS_Code
             pFreqMemEntry->RxTx[0].iDTCS_Code = SL_ParseInteger( cppSrc );
             break;
        case RIGCTRL_FREQ_MEM_TOKEN_D_C_SQLCH : // pFreqMemEntry->RxTx[0].iDV_DigitalCodeSquelch
             pFreqMemEntry->RxTx[0].iDV_DigitalCodeSquelch = SL_ParseInteger( cppSrc );
             break;
        case RIGCTRL_FREQ_MEM_TOKEN_DUP_OFFS  : // pFreqMemEntry->RxTx[0].iDuplexOffsetFreq_Hz
             pFreqMemEntry->RxTx[0].iDuplexOffsetFreq_Hz = SL_ParseInteger( cppSrc );
             break;
        case RIGCTRL_FREQ_MEM_TOKEN_DEST_CALL : // pFreqMemEntry->RxTx[0].c8DestCall aka "UR" for digital voice
             SL_CopyStringUntilDelimiter( cppSrc, pFreqMemEntry->RxTx[0].c8DestCall, 8/*iMaxLen*/, ",\r\n"/*pszDelimiters*/ );
             break;
        case RIGCTRL_FREQ_MEM_TOKEN_RPTR_CALL : // pFreqMemEntry->RxTx[0].c8AccessRptrCall  aka "R1" for digital voice
             SL_CopyStringUntilDelimiter( cppSrc, pFreqMemEntry->RxTx[0].c8AccessRptrCall, 8/*iMaxLen*/, ",\r\n"/*pszDelimiters*/ );
             break;
        case RIGCTRL_FREQ_MEM_TOKEN_GTWY_CALL : // pFreqMemEntry->RxTx[0].c8GatewayRptrCall aka "R2" for digital voice
             SL_CopyStringUntilDelimiter( cppSrc, pFreqMemEntry->RxTx[0].c8GatewayRptrCall, 8/*iMaxLen*/, ",\r\n"/*pszDelimiters*/ );
             break;
        default:
             break;
      }
    } // end while( **cppSrc != '\0' )
   return *cppSrc-pszSourceStart;
} // end RigCtrl_StringToFreqMemEntry()

//---------------------------------------------------------------------------
const char* RigCtrl_PermissionsToString( // .. as used in T_RigCtrlUserDefinedBand
     DWORD dwPermissions ) // [in] RIGCTRL_PERMISSION_NONE, RIGCTRL_PERMISSION_TRANSMIT, ..
{
  // For the few combinations we have, don't use an evil static string buffer:
  switch( dwPermissions )
   { case RIGCTRL_PERMISSION_NONE    : return "None";
     case RIGCTRL_PERMISSION_TRANSMIT: return "TX";
     default: return "???";
   }
} // end RigCtrl_PermissionsToString()

//---------------------------------------------------------------------------
DWORD RigCtrl_StringToPermissions( // .. as used in T_RigCtrlUserDefinedBand
         const char **ppszSource)
{
  DWORD dwPermissions = RIGCTRL_PERMISSION_NONE;
  const char *pszOldSource;

  while(1) // run in a loop if this turns out to be a comma-separated list of MULTIPLE permissions..
   { pszOldSource = *ppszSource; // kludge to avoid an endless loop when NOTHING was recognized

     // First check if the pszSource begins with a known "basic mode" like "CW".
     // After that, check for suffixes like "R" (Reverse), "N" (Narrow), "NN" (Very Narrow).
     if( SL_SkipString_AnyCase( ppszSource, "None" ) ) // not meaningful in a list, but anyway, keep it..
      { dwPermissions |= RIGCTRL_PERMISSION_NONE; // "code has no effect" (bitwise ORing a zero)
      }
     else if( SL_SkipString_AnyCase( ppszSource, "TX" ) )
      { dwPermissions |= RIGCTRL_PERMISSION_TRANSMIT;
      }
     // Just in case the sender uses a NUMERIC value (not recommended but possible):
     if( (*ppszSource==pszOldSource )&& (SL_IsDigit( **ppszSource ) ) )
      { dwPermissions |= SL_ParseInteger(ppszSource);
      }
     if( pszOldSource == *ppszSource ) // nothing parsed above ?
      { break; // break from the loop !
      }
     if( ! SL_SkipChar( ppszSource, ',' ) )
      { break; // end of the comma-separated "Op-Mode"-list !
      }
   } // end while
  return dwPermissions;
} // end RigCtrl_StringToPermissions()

//---------------------------------------------------------------------------
int RigCtrl_UserDefinedBandToString( // counterpart to RigCtrl_StringToUserDefinedBand()
        T_RigCtrlUserDefinedBand *pUserDefdBand, // [in] struct for a 'user defined band'
        int iArrayIndex,     // [in] zero-based array index for easy parsing (negative for "no array")
        int nEntriesInArray, // [in] total number of entries in the array (service for the receiver to detect when all items are through)
        char *pszDest, int iMaxDestLen ) // [out] key=value pairs, comma separated
  // Returns the length of the generated string (not including the trailing zero).
  // In the remote CW keyer, this function is also used to send
  // the "user defined bands" from server to all clients in an easily parsable, extendable format.
  // Example for the OUTPUT (must be compatible with the parser, RigCtrl_StringToUserDefinedBand):
  //  Format:         BandName: Fmin-Fmax(MHz) key=value key=value key=value...
  // Example: i=0 n=30 60 m EU: 5.3515-5.3540 modes=CW tx=1 DefFreq=5.3525 DefMode=CWN
  //          \______/ \____________________/ \______________________________________/
  //         optional     mandatory part        optional part with key=value pairs,
  //         "trailer"    without key names       see UserDefinedBandTokens[] .
  // "Inverse" functions: RigCtrl_UserDefinedBandToString(), RigCtrl_StringToUserDefinedBand().
  //
{ char *pszDestStart = pszDest;
  char *pszEndstop = pszDest+iMaxDestLen;


  // Optional "trailer" (used when sending the array of user-defined bands
  //                     via TCP/IP, from server to client, line-by-line)
  if( iArrayIndex >= 0 )
   { SL_AppendPrintf( &pszDest, pszEndstop, "i=%d ", (int)iArrayIndex );
   }
  if( nEntriesInArray >= 0 )
   { SL_AppendPrintf( &pszDest, pszEndstop, "n=%d ", (int)nEntriesInArray );
   }


  // Mandatory members of the T_RigCtrlUserDefinedBand entry first:
  //   * band name, start frequency, end frequency.   Full stop. The rest is OPTIONAL.
  SL_AppendPrintf( &pszDest, pszEndstop, "%s: ",pUserDefdBand->sz16Name );
  SL_AppendPrintf( &pszDest, pszEndstop, " %9.4lf-%9.4lf",
         // "width" (or something in that style)--' |
         // "precision" (or whatever it was..)------'
        pUserDefdBand->dblFmin_Hz * 1e-6, pUserDefdBand->dblFmax_Hz * 1e-6 );

  SL_AppendPrintf( &pszDest, pszEndstop, " %s%s",
        RigCtrl_GetUserDefdBandToken( RIGCTRL_UDEF_BAND_TOKEN_MODES ),
        RigCtrl_OperatingModeToString( pUserDefdBand->dwOpModes) ) ; // e.g. "USB", "LSB", "CW", .. may even be a COMMA-SEPARATED LIST
  SL_AppendPrintf( &pszDest, pszEndstop, " %s%d",
        RigCtrl_GetUserDefdBandToken( RIGCTRL_UDEF_BAND_TOKEN_TX ),
        (int)((pUserDefdBand->dwPermissions & RIGCTRL_PERMISSION_TRANSMIT) != 0) );

  if( pUserDefdBand->dblFdef_Hz != 0.0 )
   { SL_AppendPrintf( &pszDest, pszEndstop, " %s%9.4lf",
        RigCtrl_GetUserDefdBandToken( RIGCTRL_UDEF_BAND_TOKEN_DEF_FREQ ),
        pUserDefdBand->dblFdef_Hz * 1e-6 );
   }

  if( pUserDefdBand->dwDefOpMode != 0 ) // "default op-mode", e.g. RIGCTRL_OPMODE_CW ?
   { SL_AppendPrintf( &pszDest, pszEndstop, " %s%s",
        RigCtrl_GetUserDefdBandToken( RIGCTRL_UDEF_BAND_TOKEN_DEF_MODE ),
        RigCtrl_OperatingModeToString( pUserDefdBand->dwDefOpMode) );
   }
  return pszDest - pszDestStart;
} // end RigCtrl_UserDefinedBandToString()

//---------------------------------------------------------------------------
int RigCtrl_MultiFunctionMeterReportToString(
        T_RigCtrl_MultiFunctionMeterReport *pMeterReport, // [in] struct with a 'multi-function meter report'
        char *pszDest, int iMaxDestLen ) // [out] key=value pairs, comma separated
  // Returns the length of the generated string (not including the trailing zero).
{ char *pszDestStart = pszDest;
  char *pszEndstop = pszDest+iMaxDestLen;

  if( iMaxDestLen < 1 )
   { return 0;
   }
  *pszDest = '\0'; // if none of the key=value pairs is emitted below, generate an EMPTY string

  // What makes this C-struct a bit 'special' is that it doesn't necessarily
  // travel "completely" over the network. Instead, the first member
  //  ( .bStructSize ) indicates which of the members are actually contained.
  //        See byte-offsets within the struct in squared brackets .
  // Like RigCtrl_AssembleMultiFunctionMeterReport(), INVALID members are not
  //      emitted to the human-readable string.
  if( (pMeterReport->bStructSize > 1 )  && (pMeterReport->i8PowerMeterLevel_pcnt>=0) )
   { // [1] ICOM-like: They measure output power in PERCENTS, not WATTS !
     SL_AppendPrintf( &pszDest, pszEndstop, "P=%d%% ", (int)pMeterReport->i8PowerMeterLevel_pcnt );
   }
  if( (pMeterReport->bStructSize > 2 ) && (pMeterReport->i8SWRMeterValue_x10 >= 0) )
   { // [2] dimensionless "SWR" value, multiplied by 10 to send it as an 8-bit integer
     SL_AppendPrintf( &pszDest, pszEndstop, "SWR=%d ", (int)pMeterReport->i8SWRMeterValue_x10 );
   }
  if( (pMeterReport->bStructSize > 3 ) && (pMeterReport->i8ALCMeterLevel_pcnt >= 0) )
   { // [3] 0 ... 100 % of whatever the MAXIMUM may be (don't ask me; ask Icom)
     SL_AppendPrintf( &pszDest, pszEndstop, "ALC=%d%% ", (int)pMeterReport->i8ALCMeterLevel_pcnt );
   }
  if( (pMeterReport->bStructSize > 4 ) && (pMeterReport->i8CompMeterLevel_dB >= 0) )
   { // [4] compression meter level (measured); Icom measures from 0 to 30 dB. We send it "in dB", not "0241 = 30 dB".
     SL_AppendPrintf( &pszDest, pszEndstop, "COMP=%ddB ", (int)pMeterReport->i8CompMeterLevel_dB );
   }
  if( (pMeterReport->bStructSize > 5 ) && (pMeterReport->u8SupplyVoltage_100mV != 0xFF) )
   { // [5] measured supply voltage in 100-mV-units
     SL_AppendPrintf( &pszDest, pszEndstop, "U=%.1fV ", (float)pMeterReport->u8SupplyVoltage_100mV * 0.1f );
   }
  if( (pMeterReport->bStructSize > 6 ) && (pMeterReport->u8DrainCurrent_100mA != 0xFF ) )
   { // [6] measured drain current in 100-mA-units
     SL_AppendPrintf( &pszDest, pszEndstop, "I=%.1fA ", (float)pMeterReport->u8DrainCurrent_100mA * 0.1 );
   }
  if( (pMeterReport->bStructSize > 7) && (pMeterReport->u8PATemperature_degC != 0xFF ) )
   { // [7] measured power amplifier temperature in degrees CELSIUS; 0xFF = 255 when invalid .
     // Note: The Icom engineers have forgotten to implement a CI-V command
     //       to query the POWER AMPLIFIER TEMPERATURE.
     //       At least an IC-7300 can only display that important parameter LOCALLY !
     SL_AppendPrintf( &pszDest, pszEndstop, "T=%dC ", (int)pMeterReport->u8PATemperature_degC );
   }
  if( (pszDest>pszDestStart) && (pszDest[-1]==' ') ) // remove the last 'separator' ?
   {  pszDest[-1] = '\0';
      ++pszDest;
   }
  return pszDest - pszDestStart;
} // end RigCtrl_MultiFunctionMeterReportToString()

//---------------------------------------------------------------------------
int RigCtrl_PotiReportToString(
        T_RigCtrl_PotiReport *pPotiReport, // [in] struct with a 'report of multiple poti settings'
        char *pszDest, int iMaxDestLen ) // [out] key=value pairs, comma separated
  // Returns the length of the generated string (not including the trailing zero).
{ char* pszDestStart = pszDest;
  char* pszEndstop = pszDest+iMaxDestLen;

  if( iMaxDestLen < 1 )
   { return 0;
   }
  *pszDest = '\0'; // if none of the key=value pairs is emitted below, generate an EMPTY string

  // Like a few others, this C-struct doesn't necessarily travel "completely"
  // over the network, so check the indicated struct size to decide which members exist.
  // See byte-offsets within the struct definition in squared brackets ..
  if( (pPotiReport->bStructSize > 1) && (pPotiReport->i8AudioVolume_pcnt>0) )
   { // [1] similar for the radio's LOCAL AUDIO VOLUME SETTING (another "poti") .
     //     Via CI-V, this is cmd 0x14 subcmd 0x01 = "Send/read the AF level".
     //     On first glance, doesn't seem to affect the audio on the USB port.
     SL_AppendPrintf( &pszDest, pszEndstop, "AV=%d ", (int)pPotiReport->i8AudioVolume_pcnt );
   }
  if( (pPotiReport->bStructSize > 2) && (pPotiReport->i8RFPower_pcnt>0) )
   { // [1] ICOM-like: They also SET the output power in PERCENTS, not WATTS !
     SL_AppendPrintf( &pszDest, pszEndstop, "RP=%d ", (int)pPotiReport->i8RFPower_pcnt );
   }
  if( (pPotiReport->bStructSize > 3) && (pPotiReport->i8RFGain_pcnt>0) )
   { // [3] RF Gain Setting (in PERCENT of the maximum, whatever that is)
     SL_AppendPrintf( &pszDest, pszEndstop, "RG=%d ", (int)pPotiReport->i8RFGain_pcnt );
   }
  if( (pPotiReport->bStructSize > 4) && (pPotiReport->i8SquelchLevel_pcnt>0) )
   { // [4] Squelch *LEVEL Setting* (not the current READING, which possibly would be the S-meter value)
     SL_AppendPrintf( &pszDest, pszEndstop, "SQ=%d ", (int)pPotiReport->i8SquelchLevel_pcnt );
   }
  if( (pPotiReport->bStructSize > 5) && (pPotiReport->i8NoiseBlanker_pcnt>0) )
   { // [5] Noise Blanker *Setting* (not the current NOISE LEVEL READING)
     SL_AppendPrintf( &pszDest, pszEndstop, "NB=%d ", (int)pPotiReport->i8NoiseBlanker_pcnt );
   }
  if( (pPotiReport->bStructSize > 6) && (pPotiReport->i8NoiseReduction_pcnt>0) )
   { // [6] Noise Reduction *LEVEL Setting* (not the current NOISE LEVEL READING, and not the noise-reduction ON/OFF setting)
     SL_AppendPrintf( &pszDest, pszEndstop, "NR=%d ", (int)pPotiReport->i8NoiseReduction_pcnt );
   }
  if( (pPotiReport->bStructSize > 7) && (pPotiReport->i8PassbandTuningPos1_pcnt>0) )
   { // [7] Passband Tuning Position "1" (aka "the inner ring" on Icom's combined poti knob)
     SL_AppendPrintf( &pszDest, pszEndstop, "P1=%d ", (int)pPotiReport->i8PassbandTuningPos1_pcnt );
   }
  if( (pPotiReport->bStructSize > 8) && (pPotiReport->i8PassbandTuningPos2_pcnt>0) )
   { // [8] Passband Tuning Position "2" (aka "the outer ring" on Icom's combined poti knob)
     SL_AppendPrintf( &pszDest, pszEndstop, "P2=%d ", (int)pPotiReport->i8PassbandTuningPos2_pcnt );
   }
  if( (pPotiReport->bStructSize > 9) && (pPotiReport->i8CWPitch_25Hz>0) )
   { // [9] CW pitch aka 'Sidetone', here in 25 Hz steps :
     SL_AppendPrintf( &pszDest, pszEndstop, "CP=%dHz ", (int)(25 * pPotiReport->i8CWPitch_25Hz) );
   }
  if( (pPotiReport->bStructSize > 10) && (pPotiReport->i8MicGain_pcnt>0) )
   { // [10] Microphone gain, also in Percent :
     SL_AppendPrintf( &pszDest, pszEndstop, "MG=%d ", (int)pPotiReport->i8MicGain_pcnt );
   }
  if( (pPotiReport->bStructSize > 11) && (pPotiReport->i8KeyerSpeed_WPM>0) )
   { // [11] the rig's built-in CW keyer speed, already scaled into WPM :
     SL_AppendPrintf( &pszDest, pszEndstop, "CS=%d ", (int)pPotiReport->i8KeyerSpeed_WPM );
   }
  if( (pPotiReport->bStructSize > 12) && (pPotiReport->i8NotchPos_pcnt>0) )
   { // [12] notch filter position. You already guessed it: Icom doesn't use Hertz, but Percent, possibly relative to the filter passband. Makes sense.
     SL_AppendPrintf( &pszDest, pszEndstop, "NP=%d ", (int)pPotiReport->i8NotchPos_pcnt );
   }
  if( (pPotiReport->bStructSize > 13) && (pPotiReport->i8Comp_pcnt>0) )
   { // [13] audio compressor setting in percent.
     SL_AppendPrintf( &pszDest, pszEndstop, "CO=%d ", (int)pPotiReport->i8Comp_pcnt );
   }
  if( (pPotiReport->bStructSize > 14) && (pPotiReport->i8BreakInDelay_pcnt>0) )
   { // [14] Break-In delay in percent (! - not milliseconds )
     SL_AppendPrintf( &pszDest, pszEndstop, "BK=%d ", (int)pPotiReport->i8BreakInDelay_pcnt );
   }
  if( (pPotiReport->bStructSize > 15) && (pPotiReport->i8MonitorGain_pcnt>0) )
   { // [15] Break-In delay in percent (! - not milliseconds )
     SL_AppendPrintf( &pszDest, pszEndstop, "MO=%d ", (int)pPotiReport->i8MonitorGain_pcnt );
   }
  if( (pPotiReport->bStructSize > 16) && (pPotiReport->i8VoxGain_pcnt>0) )
   { // [16] Vox Gain in percent. In CI-V, cmd 0x14, subcmd 0x19. End of a long list for the IC-7610.
     SL_AppendPrintf( &pszDest, pszEndstop, "VOX=%d ", (int)pPotiReport->i8VoxGain_pcnt );
   }
  if( (pPotiReport->bStructSize > 17) && (pPotiReport->i8AntiVoxGain_pcnt>0) )
   { // [17] Anti-Vox Gain in percent.  Most likely not a real "poti" for this, but anyway..
     SL_AppendPrintf( &pszDest, pszEndstop, "AVOX=%d ", (int)pPotiReport->i8AntiVoxGain_pcnt );
   }
  if( (pPotiReport->bStructSize > 18) && (pPotiReport->i8Brightness_pcnt>0) )
   { // [18] 'Display brightness' in percent.  Most likely not a real "poti" for this, but anyway..
     SL_AppendPrintf( &pszDest, pszEndstop, "BL=%d ", (int)pPotiReport->i8Brightness_pcnt );
   }

  if( (pszDest>pszDestStart) && (pszDest[-1]==' ') ) // remove the last 'separator' ?
   {  pszDest[-1] = '\0';
      ++pszDest;
   }
  return pszDest - pszDestStart;
} // end RigCtrl_PotiReportToString()



//---------------------------------------------------------------------------
// Built-in traffic log (circular FIFO for the display in e.g. DebugU1.cpp)
//    Initially used for CI-V via serial port, but in the meantime,
//    can optionally store the first few bytes in any kind of UDP packets,
//    packetized analog samples received from external A/D converters, etc.
//    (Why that ? Certain Icom radios send / receive audio via UDP too,
//     thus module RigControl.c must be able to exchange audio streams
//     with Spectrum Lab anyway. Thus, to develop microcontroller firmware
//     driving external A/D or D/A converters, the former "CAT Traffic
//     Monitor" shall be able to display other kinds of 'packets', too.
//
//  -> file:///C:/cbproj/SpecLab/html/troubles.htm#CAT_traffic_monitor
//
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
CPROT void RigCtrl_ClearTrafficLog( T_RigCtrlInstance *pRC )
{
  int iTrafficReader;
  pRC->iTrafficLogHeadIndex = 0; // circular FIFO head index (post-incremented, wraps from RIGCTRL_MAX_TRAFFIC_LOG_ENTRIES-1 to zero)
  pRC->fTrafficLogFull  = FALSE; // TRUE when all <RIGCTRL_MAX_TRAFFIC_LOG_ENTRIES> entries are IN USE.
  RigCtrl_i32MessageNrForTrafficLog = 0;  // restart the counter (shown in the traffic log's FIRST COLUMN)
  RigCtrl_TrafficMon_dblUnixTimeOfStart = UTL_GetCurrentUnixDateAndTime_Fast();
  for(iTrafficReader=0; iTrafficReader<RIGCTRL_NUM_TRAFFIC_LOG_READERS; ++iTrafficReader )
   { pRC->iTrafficLogTailIndex[iTrafficReader] = 0;
   }
  // If the traffic monitor was paused AUTOMATICALLY (configurable, e.g. "on full FIFO" or "on Error", un-pause it:
  RigCtrl_UnPauseTrafficLogOnAllPorts( pRC );


} // end RigCtrl_ClearTrafficLog()

//---------------------------------------------------------------------------
void RigCtrl_AddToTrafficLog( T_RigCtrlInstance *pRC, // for 'decoded CI-V' messages (*)
         int  iRigCtrlPort,   // [in] RIGCTRL_PORT_RADIO / RIGCTRL_PORT_AUX_COM_1 / 2 / .. ?
         int  iRigCtrlOrigin, // [in] RIGCTRL_ORIGIN_RADIO / RIGCTRL_ORIGIN_CONTROLLER,
                              //      or just .._RX / _TX if we don't know "who is what".
                              //      Important for the traffic log especially in
                              //      'Listen Only Mode', where both COMMANDS and RESPONSES
                              //      are *received* from Remote CW Keyer's point of view,
                              //      but for the log, shall be shown with 'r*' or 't*',
                              //      where 'r' means "from RADIO to CONTROLLER",
                              //            't' means "from CONTROLLER to RADIO" (would have been 'T'=TRANSMITTED if we *were* the controller),
                              //       and '*' is a single digit indicating the port index
                              //       (not the COM port number).
         BYTE *pbMessage, int iMsgLength, // [in] CI-V message as text, including pre- and postamble,
                            //      or just an 'info' about an important internal state change,
                            //      in complete plain text, with a few parameters.
         int  iMsgType,     // [in] e.g. RIGCTRL_MSGTYPE_RADIO_ID | RIGCTRL_MSGTYPE_NO_OK, etc;
                            //      plus some bitwise combineable flags like RIGCTRL_MSGTYPE_FLAG_UDP .
         const char *pszComment)  // [in] optional comment (*not* a format string)
  // (*) Since 2021-09, RigCtrl_AddToTrafficLog() is also used to show
  //     a few Icom-specific UDP-"control" messages (with UDP payload only).
  //     The COLOUR used for the display (in a Rich Edit control) is selected
  //     later, in RigCtrl_ReadTrafficLog(), depending on "Message Type" flags like
  //       RIGCTRL_MSGTYPE_FLAG_ERROR, RIGCTRL_MSGTYPE_FLAG_TX, RIGCTRL_MSGTYPE_FLAG_RX, ...
  //     Examples (copied and pasted from the RichEdit control into HTML *as text*)
  //     are in the Spectrum Lab manual [SpecLab_CAT_Monitor] .
  //     Since 2025-06, RigCtrl_AddToTrafficLog() is also used to show messages
  //     sent from, and received by, module "Yaesu5Byte.c" .
  //
  // Non-CI-V-entries (but UDP) are flagged by RIGCTRL_MSGTYPE_CONTROL_VIA_UDP in iMsgType,
  // and are formatted slightly different than CI-V in RigCtrl_ReadTrafficLog().
{
  T_RigCtrl_PortInstance *pPortInstance = &pRC->PortInstance[iRigCtrlPort];
  T_RigCtrl_PortInstance *pServerPortInstance;
  T_RigCtrlTrafficLogEntry *pNewEntry;
  double dblUnixTime = UTL_GetCurrentUnixDateAndTime_Fast();
  int iHeadIndex = pRC->iTrafficLogHeadIndex % RIGCTRL_MAX_TRAFFIC_LOG_ENTRIES;
  int i,n;

  if( RigCtrl_fTrafficMonitorPausedByUser )  // <- logging manually paused on ALL PORTS ?
   { return;  // avoid overwriting the "important captured data"
   } // end if < traffic monitor PAUSED BY USER > ?


  if( pPortInstance->fTrafficMonitorPausedOnTrigger ) // <- this works "selectively" for EACH PORT !
   { // If THIS port (iRigCtrlPort) is currently working "on behalf" of a SERVER PORT,
     // and the traffic log FOR THAT OTHER PORT is *not* paused, then let the
     // forwarded messages appear in the log even if the log is principally PAUSED for the RADIO CONTROL PORT:
     if( pPortInstance->iWorkingForServerPort > 0 ) // <- from RigCtrl_ForwardCommandFromServerPortToRadio()...
      { pServerPortInstance = RigCtrl_IndexToPortInstancePtr( pRC, pPortInstance->iWorkingForServerPort );
        if( pServerPortInstance->fTrafficMonitorPausedOnTrigger )
         { return; // bail out because pServerPortInstance->fTrafficMonitorPausedOnTrigger
         }
        else // RigCtrl_AddToTrafficLog() called for a message on the RADIO PORT,
         {   // the RADIO PORT's traffic monitor is already PAUSED,
             // but currently working on behalf of a REMOTE CLIENT (on a SERVER PORT)
             // so let this message appear in the 'principally paused' traffic log.
           // Got HERE, for example, when called from ...
           // * RigCtrl_Handler() -> RigCtrl_SendReadWriteCommandFromFIFO()
           //     -> RigCtrl_SendWriteCommandForUnifiedPN(iUnifiedPN=RIGCTRL_PN_TRANSMITTING)
           //   [after WSJT-X 
           // do NOT bail out ...
         }
      }
     else
      { return;  // bail out because pPortInstance->fTrafficMonitorPausedOnTrigger
      }
   } // end if < traffic monitor PAUSED AUTOMATICALLY, on this port > ?


  if( iMsgType & (RIGCTRL_MSGTYPE_FLAG_TX | RIGCTRL_MSGTYPE_FLAG_RX) )
   { // the caller didn't specify if the message has been SENT or RECEIVED
     // (from this program's point of view) so try to guess from what we have:
     switch( iRigCtrlOrigin )
      { case RIGCTRL_ORIGIN_RADIO : // message originating from a RADIO:
           iMsgType |= RIGCTRL_MSGTYPE_FLAG_RX; // if we WERE the CONTROLLER, we WOULD HAVE received this message
           break;
        case RIGCTRL_ORIGIN_CONTROLLER : // message originating from a CONTROLLER:
           iMsgType |= RIGCTRL_MSGTYPE_FLAG_TX; // if we WERE the CONTROLLER, we WOULD HAVE *sent* this message
           break;
        case RIGCTRL_ORIGIN_COM_PORT_RX: // message has been RECEIVED FROM a local 'COM' port (or from a Serial Port Tunnel)
           iMsgType |= RIGCTRL_MSGTYPE_FLAG_RX;
           break;
        case RIGCTRL_ORIGIN_COM_PORT_TX: // message will be, or has already been, SENT TO a local 'COM' port (or into a Serial Port Tunnel)
           iMsgType |= RIGCTRL_MSGTYPE_FLAG_TX;
           break;
        default: // cannot extract an RX / TX flag from iRigCtrlOrigin, so don't modify iMsgType here
           break;
      } // end switch( iRigCtrlOrigin )
   } // end if < neither RIGCTRL_MSGTYPE_FLAG_TX nor RIGCTRL_MSGTYPE_FLAG_RX specified ? >

  RigCtrl_EnterCriticalSection( pRC ); // no "lazy return" after this point !

  pNewEntry = &pRC->TrafficLog[iHeadIndex];
  memset( pNewEntry, 0, sizeof(T_RigCtrlTrafficLogEntry) );  // provides zero-terminated strings
#if(0) // removed 2025-07-22, because the MESSAGE NUMBER in the first column of the traffic log
       // shall be incremented for messages that don't appear in the log.
       // This was modified when forwarding message for the server ports,
       // to easily see if messages were exchanged with the real radio
       // "on behalf of an EXTERNAL CLIENT" .
       // Since then, RigCtrl_i32MessageNrForTrafficLog is incremented in ...
       //   * RigCtrl_SendAndLogMessage() [regardless of the message being "logged" or not],
       //   *
  if( iMsgLength > 0 )  // Only increment the 'message number' when logging REAL PACKETS,
   { ++RigCtrl_i32MessageNrForTrafficLog; // .. because only 'real packets' are NUMBERED.
     // Info strings like "Opening 'Serial' port: Hostname 'IC-705' resolved into IP 192.168.178.28"
     // are no PACKETS TRAVELLING ON THE NETWORK, and will not have a number in the first column
     // when converted into text in RigCtrl_ReadTrafficLog() .
     // Despite that, if a log entry WITH packet data (i.e. T_RigCtrlTrafficLogEntry.iMsgLength > 0)
     //  is followed by a couple of 'info lines WITHOUT packet data',
     //  those subsequent lines will have the same T_RigCtrlTrafficLogEntry.i32MessageNr .
   }
#endif
  pNewEntry->i32MessageNr = RigCtrl_i32MessageNrForTrafficLog; // unique "message identifier",
    // used in the GUI (DebugU1.cpp) when the operator clicked on a certain line
    // in the RichText control. The "Message Number" (or "No.", as in Wireshark)
    // can be used to retrieve more info than currently visible on the screen .

  pNewEntry->dblUnixTime = dblUnixTime;
  n = iMsgLength; // number of "raw message bytes" that can be stored in a log-entry
  if( n>80 )
   {  n=80;       // truncate the message length for the display
   }
  if( pbMessage != NULL )  // may be NULL for special messages like "Spectrum timeout", etc
   { memcpy( pNewEntry->b80Message, pbMessage, n );  // copy up to 80 bytes of the message data for the hex dump
   }
  pNewEntry->iMsgLength   = iMsgLength;
  pNewEntry->iMsgType     = iMsgType;  // RIGCTRL_MSGTYPE_FLAG_UDP, etc ..
  pNewEntry->iRigCtrlPort = iRigCtrlPort;
  if( pszComment != NULL )
   { SL_strncpy( pNewEntry->szComment, pszComment, RIGCTRL_MAX_COMMENT_LENGTH );
   }


  ++iHeadIndex;
  if( RigCtrl_TrafficMonitor.iDisplayOptions & RIGCTRL_TMON_DISPLAY_OPTION_PAUSE_ON_FULL_FIFO )
   { if( iHeadIndex == (RIGCTRL_MAX_TRAFFIC_LOG_ENTRIES-1) )
      { pPortInstance->fTrafficMonitorPausedOnTrigger = TRUE;
        pRC->fTrafficLogFull = TRUE; // all <RIGCTRL_MAX_TRAFFIC_LOG_ENTRIES> are valid now !
        // In the VERY LAST FIFO entry, leave a note that we stopped because the FIFO is "full":
        pNewEntry = &pRC->TrafficLog[iHeadIndex];
        memset( pNewEntry, 0, sizeof(T_RigCtrlTrafficLogEntry) );  // provides zero-terminated strings
        pNewEntry->i32MessageNr = RigCtrl_i32MessageNrForTrafficLog; // unique "message identifier",
        pNewEntry->dblUnixTime = dblUnixTime;
        SL_strncpy( pNewEntry->szComment, "Traffic Log paused on 'full FIFO' (as configured)", RIGCTRL_MAX_COMMENT_LENGTH );
        pRC->iTrafficLogHeadIndex = 0;
        RigCtrl_fUpdateTrafficLog = TRUE; // now "ready for being UPDATED on-screen"
        RigCtrl_LeaveCriticalSection( pRC );
        return;
      }
   } // RIGCTRL_TMON_DISPLAY_OPTION_PAUSE_ON_FULL_FIFO ?
  if( iHeadIndex >= RIGCTRL_MAX_TRAFFIC_LOG_ENTRIES )
   {  iHeadIndex = 0;
      pRC->fTrafficLogFull = TRUE; // all <RIGCTRL_MAX_TRAFFIC_LOG_ENTRIES> are valid now !
      // Because there may be an unknown number of READERS,
      // T_RigCtrlInstance contains no FIFO *TAIL* INDEX itself !
      // An example for a "traffic log display" (reader) is in
      // C:\cbproj\SpecLab\DebugU1.cpp .
   }
  pRC->iTrafficLogHeadIndex = iHeadIndex;
  RigCtrl_fUpdateTrafficLog = TRUE; // now "ready for being UPDATED on-screen"
  RigCtrl_LeaveCriticalSection( pRC );


} // end RigCtrl_AddToTrafficLog()

//---------------------------------------------------------------------------
void RigCtrl_UnPauseTrafficLogOnAllPorts( T_RigCtrlInstance *pRC ) // -> fTrafficMonitorPausedOnTrigger = FALSE;
{ int i;
  for(i=0; i<RIGCTRL_NUM_PORT_INSTANCES; ++i )
   { pRC->PortInstance[i].fTrafficMonitorPausedOnTrigger = FALSE;
   }
} // end RigCtrl_UnPauseTrafficLogOnAllPorts()


//---------------------------------------------------------------------------
int  RigCtrl_GetNumEntriesInTrafficLog( T_RigCtrlInstance *pRC,
             int iTrafficReader )     // [in] RIGCTRL_TRAFFIC_READER_GUI, RIGCTRL_TRAFFIC_READER_LOGFILE_WRITER, etc
{
  int iHeadIndex = pRC->iTrafficLogHeadIndex % RIGCTRL_MAX_TRAFFIC_LOG_ENTRIES;
  int iTailIndex = pRC->iTrafficLogTailIndex[iTrafficReader]; // not ' % RIGCTRL_MAX_TRAFFIC_LOG_ENTRIES ' here yet !
  int iOldestEntry, nEntries;
  if( pRC->fTrafficLogFull )
   {  // ALL entries, pRC->TrafficLog[0..RIGCTRL_MAX_TRAFFIC_LOG_ENTRIES-1], are valid.
      iOldestEntry = (iHeadIndex+1) % RIGCTRL_MAX_TRAFFIC_LOG_ENTRIES;
   }
  else // only entries in pRC->TrafficLog[0..iHeadIndex-1] are valid
   {  // Reading would BEGIN at index zero .
      // No more entries are available at index iHeadIndex .
      iOldestEntry = 0;
   }
  if( iTailIndex < 0 ) // caller would read the FIRST entry (when available)
   {  iTailIndex = iOldestEntry;
   }
  else
   {  iTailIndex %= RIGCTRL_MAX_TRAFFIC_LOG_ENTRIES;
   }
  nEntries = iHeadIndex - iTailIndex;
  if( nEntries < 0 ) // circular wrap..
   {  nEntries += RIGCTRL_MAX_TRAFFIC_LOG_ENTRIES;
   }
  return nEntries;
} // end RigCtrl_GetNumEntriesInTrafficLog()


//---------------------------------------------------------------------------
BOOL RigCtrl_ReadTrafficLog( // .. as 'human readable text' with colour highlighting
             T_RigCtrlInstance *pRC,   // [in] rig control instance (just in case we support more than one)
             int  iTrafficReader,      // [in] RIGCTRL_TRAFFIC_READER_GUI, RIGCTRL_TRAFFIC_READER_LOGFILE_WRITER, etc
                                       //      (index into pRC->iTrafficLogTailIndex[iTrafficReader])
             char *pszDest,            // [out] human readable text, capacity at least <iWantedCharsPerLine+4> bytes
             int  iWantedCharsPerLine, // [in] typically 80 characters per line, but maybe more (see TDebugForm::UpdateCATTrafficLog() )
             int  *piMsgType,          // [out,optional] RIGCTRL_MSGTYPE_FREQUENCY_REPORT (protocol independent)
             DWORD *pdwColor )         // [out,optional] colour, from MSBIT to LSBIT: BLUE, GREEN, RED (8 bit each)
  // Returns TRUE as long as more entries are available (and the TAIL INDEX is incremented),
  //     or  FALSE when no more entries are available in the CAT message FIFO.
  //
  // Since 2023-07, RigCtrl_ReadTrafficLog(!) may decide HOW TO DISPLAY packets,
  //    depending on RIGCTRL_TMON_DISPLAY_OPTION_SHOW_UDP_PAYLOAD  .
  //    This was easy to achieve because if RIGCTRL_MSGTYPE_FLAG_UDP is set in iMsgType,
  //    pbMessage points to THE FIRST BYTE of the *UDP* payload;
  //    not just the CI-V payload (see RigCtrl_SendCIVDataViaUDP() -> RigCtrl_AddToTrafficLog() ).
  //
{
  int iHeadIndex = pRC->iTrafficLogHeadIndex % RIGCTRL_MAX_TRAFFIC_LOG_ENTRIES;
  int iTailIndex = pRC->iTrafficLogTailIndex[iTrafficReader]; // not ' % RIGCTRL_MAX_TRAFFIC_LOG_ENTRIES ' here yet !
  int iOldestEntry, i, iMsgType, iType2, n, iSlen, nCharsForMsgTypeAndComment;
  T_RigCtrlTrafficLogEntry *pEntry;
  char *cp = pszDest;
  char *pszEndstop = pszDest + iWantedCharsPerLine;
  DWORD dwColor;
  BOOL  fHavePacketData; // TRUE : got a log entry with "packet data";  FALSE : "just an INFO or missing data"
  BYTE  *pbSource;

  *cp = '\0';

  if( pRC->fTrafficLogFull )
   {   // ALL entries, pRC->TrafficLog[0..RIGCTRL_MAX_TRAFFIC_LOG_ENTRIES-1], are valid.
       // Reading must BEGIN at iHeadIndex+1 because that's THE OLDEST entry
       //   (==the next to be overwritten in RigCtrl_AddToTrafficLog() .
       // When the tail index reaches iHeadIndex, no more entries are available.
       iOldestEntry = (iHeadIndex+1) % RIGCTRL_MAX_TRAFFIC_LOG_ENTRIES;
   }
  else // only entries in pRC->TrafficLog[0..iHeadIndex-1] are valid
   {   // Reading must BEGIN at index zero .
       // No more entries are available at index iHeadIndex .
       iOldestEntry = 0;
   }
  if( iTailIndex < 0 ) // caller wants to read the FIRST, aka OLDEST entry (when available)
   {  iTailIndex = iOldestEntry;
   }
  else
   {  iTailIndex %= RIGCTRL_MAX_TRAFFIC_LOG_ENTRIES;
   }
  if( iHeadIndex == iTailIndex ) // no new entry available for THIS "traffic log reader" !
   { return FALSE;
   }
  else
   { pEntry = &pRC->TrafficLog[iTailIndex];
     pRC->iTrafficLogTailIndex[iTrafficReader] = (iTailIndex+1) % RIGCTRL_MAX_TRAFFIC_LOG_ENTRIES;
     iMsgType = pEntry->iMsgType;
     iType2 = iMsgType & RIGCTRL_MSGMASK_BASIC;

     // Which background colour ?
     if( iMsgType & RIGCTRL_MSGTYPE_FLAG_ERROR ) // e.g. a response-reception-TIMEOUT
      { dwColor = 0x7F00FF; // PINK(ish) .. from MSBIT to LSBIT: BLUE, GREEN, RED (8 bit each)
      }
     if( iMsgType & RIGCTRL_MSGTYPE_NOT_OK ) // e.g. in CI-V : unspecific "NG" / "Not Good", cmd 0xFA
      { dwColor = 0xFF00FF; // colour format, from MSBIT to LSBIT: BLUE, GREEN, RED (8 bit each)
      }
     else if( iMsgType & RIGCTRL_MSGTYPE_FLAG_TX ) // this message was SENT (in CI-V speak, from "controller" to "radio")
      { dwColor = 0x00FFFF; // YELLOW .. from MSBIT to LSBIT: BLUE, GREEN, RED (8 bit each)
      }
     else if( iMsgType & RIGCTRL_MSGTYPE_FLAG_RX ) // this message contains 'real DATA' (UDP or CI-V), and it was RECEIVED
      { // RECEIVED message : light cyan background (unless it's a "NoGo"-response, or some other "special case" :
        if( iMsgType & RIGCTRL_MSGTYPE_FLAG_UDP )
         { // Added 2021-09 : not only the normal CI-V payload but UDP-payload :
           // If the message type was recognized, the "meaning" of this message
           // is in pEntry->szComment originating from RigCtrl_ProcessIcomUDP().
           // Show "CI-V less" messages in YELLOW .
           // Otherwise (UDP Control packet from an Icom radio that we DON'T understand),
           // show the message in ORANGE colour. At least, that was the plan...
           if( pEntry->szComment[0] != '\0' ) // ok, RigCtrl_ProcessIcomUDP() had recognized the received type ->
            { dwColor = 0xC0FF7F; // B,G,R for "aqua"(?) = RECEIVED UDP (not yet reduced to "pure CI-V" for the 'Traffic Display")
            }
           else // not recognized ?
            { dwColor = 0x007FFF; // orange = received UDP, but content not recognized
            }
         }   // RIGCTRL_MSGTYPE_CONTROL_VIA_UDP ?
        else // not an entry with UDP "payload" but most likely a CI-V message:
         {
           dwColor = 0xFFFF00; // 100 % BLUE  + 100 % GREEN + 0 % RED (8 bit each) : lightcyan = RECEIVED CiV (not UDP)
           if( iMsgType & RIGCTRL_MSGTYPE_FLAG_EXPECTED )
            { dwColor = 0x7FFF7F; // more green-ish than Cyan : "received the EXPECTED response"
            }
         }
      }
     else // none of the above, no "real DATA packet" (neither RX- nor TX-data):
      { // e.g. info like > "Traffic Log paused on Radio Port after initialisation."
        dwColor = 0xC0C0C0; // B,G,R for "lightgray"
      }

     // Suggest to emit a HEADLINE on the display if this is the first message ?
     if( pEntry->i32MessageNr == 1 )
      { iMsgType |= RIGCTRL_MSGTYPE_FLAG_SUGGEST_NEW_HEADLINE_FOR_TRAFFIC_LOG;
      }

     // Begin each line in the traffic log with the 'Number'
     //   ("No." as in Wireshark, with leading spaces, and RIGHT-ALIGNED.
     //    Four digits should be sufficient, because 'the trouble usually starts early'):
     if( (pEntry->iMsgLength > 0) // traffic log entry with REAL PACKET DATA ?
       &&( ( iType2 != RIGCTRL_MSGTYPE_INFO) || (iMsgType & RIGCTRL_MSGMASK_OK_NOK) )
       )
      { sprintf( cp, "%4ld", (long)( pEntry->i32MessageNr % 10000 ) ); // Show their "No." in the first column.
        fHavePacketData = TRUE;
      }
     else // traffic log entry WITHOUT real packet data -> leave several columns away further below !
      { fHavePacketData = FALSE;
      }
     SL_PadSpaces( &cp, pszEndstop, 5/*iFixedFieldLength*/ );

     // Next (optional) column: "time of day" in in UTC .. also makes sense for RIGCTRL_MSGTYPE_INFO (without PACKET DATA)
     if( RigCtrl_TrafficMonitor.iExtraColumns & RIGCTRL_TMON_EXTRA_COLUMN_TIME )
      { switch( RigCtrl_TrafficMonitor.iTimestampFormat )
         { case RIGCTRL_TMON_TIMESTAMP_MILLISECONDS :
              SL_AppendPrintf( &cp, pszEndstop,"%07ld ", // <- field as long as "time/ms "
                 (long)( 1000.0 * ( pEntry->dblUnixTime - RigCtrl_TrafficMon_dblUnixTimeOfStart) ) );
              break;

           default: // "time of day" in in UTC ..
              UTL_FormatDateAndTime( "hh:mm:ss.s", pEntry->dblUnixTime, cp );
              SL_SkipToEndOfString( (const char**)&cp );
              SL_AppendChar( &cp, pszEndstop, ' ' );  // TEN characters plus one column-separating SPACE
              break;
         }
      }

     if( fHavePacketData ) // most columns that follow are only displayed if there are 'packet data' ...
      {

        // Next (optional) column: show "TX" / "RX" (transmit/receive flag; useful if this cannot be indicated by COLOURS)
        if( RigCtrl_TrafficMonitor.iExtraColumns & RIGCTRL_TMON_EXTRA_COLUMN_TX_RX )
         { switch( pEntry->iRigCtrlPort ) // message on the RADIO PORT or one of the "duplicator" ports ?
            { case RIGCTRL_PORT_RADIO :   // message on the port that connects TO THE RADIO...
                 if( iMsgType & RIGCTRL_MSGTYPE_FLAG_TX )
                  { strcpy( cp, "TX " );
                  }
                 else
                  { strcpy( cp, "RX " );
                  }
                 break;
              default :
                 if( iMsgType & RIGCTRL_MSGTYPE_FLAG_TX )
                  { sprintf( cp, "t%d ", (int)pEntry->iRigCtrlPort );  // "t1".."t4" : transmitted to the 1st .. 4th "CLIENT PORT"
                  }
                 else
                  { sprintf( cp, "r%d ", (int)pEntry->iRigCtrlPort );  // "r1".."r4" : received from the 1st .. 4th "CLIENT PORT"
                  }
                 break;
            } // end switch( iRigCtrlPort )
           SL_PadSpaces( &cp, pszEndstop, 3/*iFixedFieldLength*/ ); // TWO characters plus one column-separating SPACE
         } // end if < .. RIGCTRL_TMON_EXTRA_COLUMN_TX_RX > ?

        // Next (optional) column: show PAYLOAD LENGTH (three hex digits without prefix) ?
        if( RigCtrl_TrafficMonitor.iExtraColumns & RIGCTRL_TMON_EXTRA_COLUMN_PAYLOAD_LENGTH )
         { sprintf( cp, "%03X", (int)pEntry->iMsgLength ); // length (3 digits hex, should be ok for up to 4095 bytes from LAN)
           SL_PadSpaces( &cp, pszEndstop, 4/*iFixedFieldLength*/ ); // THREE hex digits plus one column-separating SPACE
         } // end if < .. RIGCTRL_TMON_EXTRA_COLUMN_PAYLOAD_LENGTH > ?


        // Convert the (mostly binary) entry into human-readable text,
        //   unless the "traffic log entry" contains no data but only a TEXT MESSAGE:
        switch( pEntry->iRigCtrlPort ) // message on the RADIO PORT or one of the "duplicator" ports ?
         { case RIGCTRL_PORT_RADIO :   // message on the port that connects TO THE RADIO...
              // leave dwColor as-is (not "weaker than usual".. see below)
              break;
           default :
              dwColor &= 0xBFBFBF; // make the three colour components a bit (3) darker to indicate a "duplicator client port"
              break;
         } // end switch( iRigCtrlPort )
        n = pEntry->iMsgLength; // number of "raw message bytes" that can be stored in a log-entry
        if( n>80 )
         {  n=80;
         }
        nCharsForMsgTypeAndComment = 0;
        if(  ( pEntry->szComment[0] != '\0' )
          && ( ( RigCtrl_TrafficMonitor.iExtraColumns & RIGCTRL_TMON_EXTRA_COLUMN_COMMENTS ) // show "machine-generated comments, parameter names, etc" ? ...
             ||( ! fHavePacketData ) ) // without "packet data" to show, all we can show is the "comment" or "info line" !
          )
         { nCharsForMsgTypeAndComment += strlen(pEntry->szComment) + 2/*strlen("; ")*/;
         }
        if( nCharsForMsgTypeAndComment == 0 )
         { nCharsForMsgTypeAndComment = 32;  // use the 'old, fixed default'
         }

        for( i=0; (i<n) && (cp<(pszDest+iWantedCharsPerLine-nCharsForMsgTypeAndComment-5)); ++i )
         { sprintf( cp, "%02X ", (int)pEntry->b80Message[i] ); // show data (CI-V, CAT, or UDP) as 2-digit HEX
           cp += strlen(cp);
         }
        if( i<n ) // couldn't dump everything (e.g. with waveform data) ?
         { *cp++ = '.';
           *cp++ = '.'; // marker for omitted hex bytes
           *cp++ = ' ';
         }
        while( cp<(pszDest+iWantedCharsPerLine-nCharsForMsgTypeAndComment) ) // pad spaces for short messages, incl. VFO frequency
         { *cp++ = ' '; // (for rare long messages, columns may be misaligned but that's ok)
         }
        *cp = '\0';
        iSlen = strlen(pEntry->szComment);
        if( (iSlen>0) && ( (cp+iSlen)<(pszDest+iWantedCharsPerLine) ) )
         {
           // if there's enough space, pad up to the next column
           while( (cp<(pszDest+iWantedCharsPerLine-iSlen-2) )
               && (cp<(pszDest+iWantedCharsPerLine-16) )  )
            { *cp++ = ' ';
            }
           sprintf( cp, "; %s", pEntry->szComment );
           cp += strlen(cp);
         } // end if < append the COMMENT > ?
      } // end if( fHavePacketData )
     else // "coloured text" only, no message data:
      { sprintf( cp, " %s", pEntry->szComment );
        cp += strlen(cp);
      } // end else < !fHavePacketData >
     while( cp<(pszDest+iWantedCharsPerLine) )  // for the coloured background, pad even more spaces..
      { *cp++ = ' '; // formerly 80 chars per line, now depends on width of the window
      }
     *cp = '\0';

     if( piMsgType != NULL )
      { *piMsgType = iMsgType; // [out,optional] RIGCTRL_MSGTYPE_xyz (protocol independent)
      }
     if( pdwColor != NULL )
      { *pdwColor = dwColor;
      }
     return TRUE;
   }
} // end RigCtrl_ReadTrafficLog()


//---------------------------------------------------------------------------
void RigCtrl_GetHeadlineForTrafficLog(int iMsgType, char *pszDest, int iWantedCharsPerLine)
  // Generates a 'headline' to simplify interpreting the (hex) message payload
  // in "Icom UDP" packets, any maybe other message types.
  // Occasionally called from TDebugForm::UpdateCATTrafficLog() (!) .
  // The FORMAT must match the strings assembled by RigCtrl_ReadTrafficLog().
  //
  //  [in] iMsgType : bit combination including flags like  ..
  //       RIGCTRL_MSGTYPE_FLAG_UDP : UDP (datagram, usually Icom RS-BA compatible),
  //       RIGCTRL_MSGTYPE_FLAG_CIV : Icom's CI-V protocol, etc.
  //       The protcol-indicator flags must be specified when calling
  //       RigCtrl_SendAndLogMessage() or RigCtrl_AddToTrafficLog(),
  //       because ONLY THE CALLER knows the type / protocol / etc.
  //  [out] pszDest : capactiy should be iWantedCharsPerLine + 4 (!) chars or more.
{
  int byte_index;
  char *cp         = pszDest;
  char *pszEndstop = pszDest + iWantedCharsPerLine; // <- points to the STRING TERMINATOR
  int  nCharsRemaining;


  // First (non-optional) column: "No. " (with FOUR digits plus a column-separating SPACE) ..
  SL_AppendString( &cp, pszEndstop, "; Nr " );

  // Next (optional) column: "timestamp", in various flavours ...
  if( RigCtrl_TrafficMonitor.iExtraColumns & RIGCTRL_TMON_EXTRA_COLUMN_TIME )
   { switch( RigCtrl_TrafficMonitor.iTimestampFormat )
      { case RIGCTRL_TMON_TIMESTAMP_MILLISECONDS :
           SL_AppendString( &cp, pszEndstop, "time/ms " );
           break;

        default: // "time of day" in in UTC ..
           SL_AppendString( &cp, pszEndstop, "hh:mm:ss.s " );
           break;
      }
   }

  // Next (optional) column:  header for the "TX" / "RX" indicator
  if( RigCtrl_TrafficMonitor.iExtraColumns & RIGCTRL_TMON_EXTRA_COLUMN_TX_RX )
   { SL_AppendString( &cp, pszEndstop, "tr " );
   }

  // Next (optional) column: packet length (3 digits hex)
  if( RigCtrl_TrafficMonitor.iExtraColumns & RIGCTRL_TMON_EXTRA_COLUMN_PAYLOAD_LENGTH )
   { SL_AppendString( &cp, pszEndstop, "len " );
   } // end if < .. RIGCTRL_TMON_EXTRA_COLUMN_PAYLOAD_LENGTH > ?

  // Next (NON-OPTIONAL) column : "legend for the hex dump"
  if( iMsgType & RIGCTRL_MSGTYPE_FLAG_UDP ) // line in the traffic log with *UDP payload* :
   {
     // Example (line with UDP hex payload emitted by RigCtrl_ReadTrafficLog() ) :
     //                                 10 00 00 00 03 00 00 00 D7 E9 00 00 00 00 00 00
     SL_AppendString( &cp, pszEndstop, "|-IcomLen-| |typ| |seq| |-sent_id-| |-rcvd_id-|" );
     nCharsRemaining = pszEndstop - cp;
     if( nCharsRemaining > 3 )  // beginning at this character-index,
      { // just add HEXADECIMAL BYTE INDICES, as used in the RS-BA-compatible
        // structure definitions in RigControl.h :
        cp = pszDest + strlen(pszDest);
        byte_index = 0x10;
        while( (cp+2)<pszEndstop )
         { sprintf( cp, " %02X", (int)(byte_index++) );
           SL_SkipToEndOfString( (const char**)&cp );
           // ,------------------|____________|
           // '--> stupid cast .. ?
         }
      }
   }
  else if(iMsgType & RIGCTRL_MSGTYPE_FLAG_CIV ) // only CI-V, without overhead from a "transport layer"
   { //               byte-index----------> 00 01 02 03 04 05 06 07 08 09 0A 0B 0C 0D 0E 0F 10 11 12 13 14 15 16 17 ...
     //               3-digit LENGTH--->010 FE FE A4 E0 27 .. .. .. .. .. .. .. FD
     SL_AppendString( &cp, pszEndstop, "pream to fm CIV-cmd, params..  postamble=FD   " );
   }
  else // neither "UDP carrying some protocol" nor "pure CI-V" -> emit a neutral headline like
   {   //   "; Nr hh:mm:ss.s tr len hex data" :
     SL_AppendString( &cp, pszEndstop, "hex data" );
   }
  // Pad with spaces for the wanted length (just in case the headline has a coloured background) :
  cp = pszDest + SL_strnlen( pszDest, iWantedCharsPerLine );
  while(cp<pszEndstop)
   { *cp++ = ' ';
   }
  *pszEndstop = '\0';  // <- one of the reasons why the capacity of pszDest must exceed iWantedCharsPerLine (by at least ONE BYTE)

} // end RigCtrl_GetHeadlineForTrafficLog()

//---------------------------------------------------------------------------
void RigCtrl_LimitDouble( double *pdblValue, double dblMin, double dblMax )
{ if( *pdblValue < dblMin )
   {  *pdblValue = dblMin;
   }
  if( *pdblValue > dblMax )
   {  *pdblValue = dblMax;
   }
} // end RigCtrl_LimitDouble()

//----------------------------------------------------------------------------
void RigCtrl_Y5B_OnDecodeCallback( // may be invoked ONCE or MULTIPLE TIMES from Yaesu5Byte_ProcessData() ...
        T_Yaesu5ByteControl *pYC, // [in] Y5B-decoder-instance that 'decoded' the message,
                                  //      here: connected to a T_RigControl instance
        int iRigCtrlPort,   // [in] e.g. RIGCTRL_PORT_RADIO, RIGCTRL_PORT_AUX_COM_1, .. RIGCTRL_PORT_AUX_COM_1, ..
        int iRigCtrlOrigin, // [in] e.g. RIGCTRL_ORIGIN_RADIO, RIGCTRL_ORIGIN_CONTROLLER, RIGCTRL_ORIGIN_COM_PORT_RX/TX
        BYTE *pbMessage, int iMsgLength, // [in] A SINGLE COMMAND or RESPONSE, and its length in bytes
        char* pszComment )  // [in] human readable "comment" for the log, generated by Yaesu5Byte.c
  // Called from RigCtrl_ProcessRxData() -> Yaesu5Byte_ProcessData() .
{
  T_RigCtrlInstance *pRC = (T_RigCtrlInstance*)pYC->pvRigControl; // <- this may be NULL !
  T_RigCtrl_PortInstance *pPortInstance;
  int iMsgType = pYC->iCurrentMsgType;

  // Show what happened (in the same format as for TRANSMITTED messages) ?
  // ex: BOOL fShowInLog = (RigCtrl_TrafficMonitor.iDisplayOptions & RIGCTRL_TMON_DISPLAY_OPTION_ENABLE) != 0;
  BOOL fShowInLog = RigCtrl_IsTrafficMonitorEnabled( pPortInstance );


  if( pRC != NULL )
   { pPortInstance = &pRC->PortInstance[ iRigCtrlPort ];

     if( RigCtrl_IsMsgTypeRejectedForLog( pPortInstance, iMsgType) ) // here: called from RigCtrl_Y5B_OnDecodeCallback()
      { fShowInLog = FALSE;
      }
     if( fShowInLog )
      {  RigCtrl_AddToTrafficLog( pRC, iRigCtrlPort, iRigCtrlOrigin, pbMessage, iMsgLength, iMsgType, pszComment );
      }
     ++pRC->PortInstance[iRigCtrlPort].dwNumMessagesRcvd;
     ++RigCtrl_i32MessageNrForTrafficLog; // here: decoded another 'Yaesu-5-Byte'-message, in RigCtrl_Y5B_OnDecodeCallback()

   }
} // end RigCtrl_Y5B_OnDecodeCallback()



//---------------------------------------------------------------------------
// Lookup tables and similar ...
//---------------------------------------------------------------------------

const T_RigCtrl_RadioInfo RigCtrl_RadioInfo_CIV[] = // radios with Icom-CIV :
{ // { iDefaultAddress,            pszName,        iCapabilities (if known),
  //      dwBands (bitwise combined, 32 bit parameter, see RIGCTRL_BAND_xyz)
  // RECENT RADIOS (especially those with four-digit model numbers) FIRST,
  // the old stuff (e.g. monobanders from the past millenium) LATER.
  { RIGCTRL_DEF_ADDR_IC_7000     , "IC-7000"     , 0 /*unknown capabilities*/,
          RIGCTRL_BAND_ALL_SHORTWAVE | RIGCTRL_BAND_6M | RIGCTRL_BAND_2M | RIGCTRL_BAND_70CM },
  { RIGCTRL_DEF_ADDR_IC_7100     , "IC-7100"     , 0 /*unknown capabilities*/,
          RIGCTRL_BAND_ALL_SHORTWAVE | RIGCTRL_BAND_6M | RIGCTRL_BAND_2M | RIGCTRL_BAND_70CM },
  { RIGCTRL_DEF_ADDR_IC_7200     , "IC-7200"     , 0 /*unknown capabilities*/,
          RIGCTRL_BAND_ALL_SHORTWAVE | RIGCTRL_BAND_6M },
  { RIGCTRL_DEF_ADDR_IC_7300     , "IC-7300"  , RIGCTRL_CAPS_SPECTRUM_VIA_CAT
          | RIGCTRL_CAPS_POWER_ON_OFF_VIA_CAT | RIGCTRL_CAPS_HAS_NO_IQ_OUTPUT,
          RIGCTRL_BAND_ALL_SHORTWAVE | RIGCTRL_BAND_6M /* | RIGCTRL_BAND_4M */ },
          // ,----------------------------------------------|_____________|
          // '--> 70 MHz and 5 MHz transmit only after hardware modification.
          //      The "all band TX mod" (removal of diode D422) along with the
          //          "all band RX mod" (removal of diode D416) caused the
          //          radio to report only A SINGLE TRANSMIT BAND (0.1 to 74 MHz).
          //      That will cause RigCtrl_GetAvailableBands() to report
          //      the additional bands like RIGCTRL_BAND_60M.
          //      Similar may apply to other 'wide-band modifyable' radios.
  { RIGCTRL_DEF_ADDR_IC_7400     , "IC-7400"     , 0 /*unknown capabilities*/,
          RIGCTRL_BAND_ALL_SHORTWAVE | RIGCTRL_BAND_6M | RIGCTRL_BAND_2M },
  { RIGCTRL_DEF_ADDR_IC_7600     , "IC-7600"     , 0 /*unknown capabilities*/,
          RIGCTRL_BAND_ALL_SHORTWAVE | RIGCTRL_BAND_6M },
  { RIGCTRL_DEF_ADDR_IC_7610     , "IC-7610"  , RIGCTRL_CAPS_SPECTRUM_VIA_CAT
          | RIGCTRL_CAPS_POWER_ON_OFF_VIA_CAT | RIGCTRL_CAPS_HAS_NO_IQ_OUTPUT,
          RIGCTRL_BAND_ALL_SHORTWAVE | RIGCTRL_BAND_6M }, // not sure about 4m
  { RIGCTRL_DEF_ADDR_IC_7700     , "IC-7700"     , 0 /*unknown capabilities*/,
          RIGCTRL_BAND_ALL_SHORTWAVE | RIGCTRL_BAND_6M },
  { RIGCTRL_DEF_ADDR_IC_7760     , "IC-7760"  , RIGCTRL_CAPS_SPECTRUM_VIA_CAT
          | RIGCTRL_CAPS_POWER_ON_OFF_VIA_CAT,
          RIGCTRL_BAND_ALL_SHORTWAVE | RIGCTRL_BAND_6M }, // not support for 4 m  :(
  { RIGCTRL_DEF_ADDR_IC_7800     , "IC-7800"     , 0 /*unknown capabilities*/,
          RIGCTRL_BAND_ALL_SHORTWAVE | RIGCTRL_BAND_6M },
  { RIGCTRL_DEF_ADDR_IC_705      , "IC-705"   , RIGCTRL_CAPS_SPECTRUM_VIA_CAT
          | RIGCTRL_CAPS_POWER_ON_OFF_VIA_CAT | RIGCTRL_CAPS_HAS_NO_IQ_OUTPUT,
         RIGCTRL_BAND_ALL_SHORTWAVE | RIGCTRL_BAND_2M | RIGCTRL_BAND_70CM }, // a gem !
  { RIGCTRL_DEF_ADDR_IC_9700     , "IC-9700"  , RIGCTRL_CAPS_SPECTRUM_VIA_CAT
          | RIGCTRL_CAPS_POWER_ON_OFF_VIA_CAT | RIGCTRL_CAPS_HAS_NO_IQ_OUTPUT,
          RIGCTRL_BAND_2M | RIGCTRL_BAND_70CM | RIGCTRL_BAND_23CM }, // 23cm always supported
  { RIGCTRL_DEF_ADDR_IC_905      , "IC-905"   , RIGCTRL_CAPS_SPECTRUM_VIA_CAT,
         RIGCTRL_BAND_2M | RIGCTRL_BAND_70CM | RIGCTRL_BAND_23CM
       | RIGCTRL_BAND_13CM   | RIGCTRL_BAND_6CM  | RIGCTRL_BAND_3CM }, // 3cm optional


  { RIGCTRL_DEF_ADDR_IC_271      , "IC-271"      , 0 /*unknown capabilities*/,
          RIGCTRL_BAND_2M },
  { RIGCTRL_DEF_ADDR_IC_275      , "IC-275"      , 0 /*unknown capabilities*/,
          RIGCTRL_BAND_2M },
  { RIGCTRL_DEF_ADDR_IC_471      , "IC-471"      , 0 /*unknown capabilities*/,
          RIGCTRL_BAND_70CM },
  { RIGCTRL_DEF_ADDR_IC_475      , "IC-475"      , 0 /*unknown capabilities*/,
          RIGCTRL_BAND_70CM },
  { RIGCTRL_DEF_ADDR_IC_575      , "IC-575"      , 0 /*unknown capabilities*/,
          RIGCTRL_BAND_10M | RIGCTRL_BAND_6M },
  { RIGCTRL_DEF_ADDR_IC_703      , "IC-703"      , 0 /*unknown capabilities*/,
          RIGCTRL_BAND_ALL_SHORTWAVE | RIGCTRL_BAND_6M },
  { RIGCTRL_DEF_ADDR_IC_706      , "IC-706"      , 0 /*unknown capabilities*/,
          RIGCTRL_BAND_ALL_SHORTWAVE | RIGCTRL_BAND_6M | RIGCTRL_BAND_2M },
  { RIGCTRL_DEF_ADDR_IC_706MkII  , "IC-706MkII"  , 0 /*unknown capabilities*/,
          RIGCTRL_BAND_ALL_SHORTWAVE | RIGCTRL_BAND_6M | RIGCTRL_BAND_2M },
  { RIGCTRL_DEF_ADDR_IC_706MkIIG , "IC-706MkIIG" , 0 /*unknown capabilities*/,
          RIGCTRL_BAND_ALL_SHORTWAVE | RIGCTRL_BAND_6M | RIGCTRL_BAND_2M | RIGCTRL_BAND_70CM },
  { RIGCTRL_DEF_ADDR_IC_707      , "IC-707"      , 0 /*unknown capabilities*/,
          RIGCTRL_BAND_ALL_SHORTWAVE },
  { RIGCTRL_DEF_ADDR_IC_718      , "IC-718"      , 0 /*unknown capabilities*/,
          RIGCTRL_BAND_ALL_SHORTWAVE },
  { RIGCTRL_DEF_ADDR_IC_725      , "IC-725"      , 0 /*unknown capabilities*/,
          RIGCTRL_BAND_ALL_SHORTWAVE },
  { RIGCTRL_DEF_ADDR_IC_726      , "IC-726"      , 0 /*unknown capabilities*/,
          RIGCTRL_BAND_ALL_SHORTWAVE | RIGCTRL_BAND_6M },
  { RIGCTRL_DEF_ADDR_IC_728      , "IC-728"      , 0 /*unknown capabilities*/,
          RIGCTRL_BAND_ALL_SHORTWAVE },
  { RIGCTRL_DEF_ADDR_IC_729      , "IC-729"      , 0 /*unknown capabilities*/,
          RIGCTRL_BAND_ALL_SHORTWAVE | RIGCTRL_BAND_6M },
  { RIGCTRL_DEF_ADDR_IC_735      , "IC-735"      , 0 /*unknown capabilities*/,
          RIGCTRL_BAND_ALL_SHORTWAVE },
  { RIGCTRL_DEF_ADDR_IC_736      , "IC-736"      , 0 /*unknown capabilities*/,
          RIGCTRL_BAND_ALL_SHORTWAVE | RIGCTRL_BAND_6M },
  { RIGCTRL_DEF_ADDR_IC_737      , "IC-737"      , 0 /*unknown capabilities*/,
          RIGCTRL_BAND_ALL_SHORTWAVE },
  { RIGCTRL_DEF_ADDR_IC_738      , "IC-738"      , 0 /*unknown capabilities*/,
          RIGCTRL_BAND_ALL_SHORTWAVE },
  { RIGCTRL_DEF_ADDR_IC_746      , "IC-746"      , 0 /*unknown capabilities*/,
          RIGCTRL_BAND_ALL_SHORTWAVE },
  { RIGCTRL_DEF_ADDR_IC_751A     , "IC-751A"     , 0 /*unknown capabilities*/,
          RIGCTRL_BAND_ALL_SHORTWAVE },
  { RIGCTRL_DEF_ADDR_IC_756      , "IC-756"      , 0 /*unknown capabilities*/,
          RIGCTRL_BAND_ALL_SHORTWAVE | RIGCTRL_BAND_6M },
  { RIGCTRL_DEF_ADDR_IC_756Pro   , "IC-756Pro"   , 0 /*unknown capabilities*/,
          RIGCTRL_BAND_ALL_SHORTWAVE | RIGCTRL_BAND_6M },
  { RIGCTRL_DEF_ADDR_IC_756ProII , "IC-756ProII" , 0 /*unknown capabilities*/,
          RIGCTRL_BAND_ALL_SHORTWAVE | RIGCTRL_BAND_6M },
  { RIGCTRL_DEF_ADDR_IC_756ProIII, "IC-756ProIII", 0 /*unknown capabilities*/,
          RIGCTRL_BAND_ALL_SHORTWAVE | RIGCTRL_BAND_6M },
  { RIGCTRL_DEF_ADDR_IC_761      , "IC-761"      , 0 /*unknown capabilities*/,
          RIGCTRL_BAND_ALL_SHORTWAVE | RIGCTRL_BAND_6M | RIGCTRL_BAND_2M | RIGCTRL_BAND_70CM },
  { RIGCTRL_DEF_ADDR_IC_765      , "IC-765"      , 0 /*unknown capabilities*/,
          RIGCTRL_BAND_ALL_SHORTWAVE },
  { RIGCTRL_DEF_ADDR_IC_775      , "IC-775"      , 0 /*unknown capabilities*/,
          RIGCTRL_BAND_ALL_SHORTWAVE },
  { RIGCTRL_DEF_ADDR_IC_78       , "IC-78"       , 0 /*unknown capabilities*/,
          RIGCTRL_BAND_ALL_SHORTWAVE },
  { RIGCTRL_DEF_ADDR_IC_781      , "IC-781"      , 0 /*unknown capabilities*/,
          RIGCTRL_BAND_ALL_SHORTWAVE },
  { RIGCTRL_DEF_ADDR_IC_820      , "IC-820"      , 0 /*unknown capabilities*/,
          RIGCTRL_BAND_2M | RIGCTRL_BAND_70CM },
  { RIGCTRL_DEF_ADDR_IC_821      , "IC-821"      , 0 /*unknown capabilities*/,
          RIGCTRL_BAND_2M | RIGCTRL_BAND_70CM },
  { RIGCTRL_DEF_ADDR_IC_910      , "IC-910"      , 0 /*unknown capabilities*/,
          RIGCTRL_BAND_2M | RIGCTRL_BAND_70CM | RIGCTRL_BAND_23CM },
  { RIGCTRL_DEF_ADDR_IC_970      , "IC-970"      , 0 /*unknown capabilities*/,
          RIGCTRL_BAND_2M | RIGCTRL_BAND_70CM | RIGCTRL_BAND_23CM }, // 23cm was optional
  { RIGCTRL_DEF_ADDR_IC_1271     , "IC-1271"     , 0 /*unknown capabilities*/,
         RIGCTRL_BAND_23CM },
  { RIGCTRL_DEF_ADDR_IC_1275     , "IC-1275"     , 0 /*unknown capabilities*/,
         RIGCTRL_BAND_23CM },
  { RIGCTRL_DEF_ADDR_IC_R10      , "IC-R10"      , 0 /*unknown capabilities*/,
         RIGCTRL_BAND_ALL_SHORTWAVE | RIGCTRL_BAND_2M | RIGCTRL_BAND_70CM }, // wideband 'scanner'
  { RIGCTRL_DEF_ADDR_IC_R20      , "IC-R20"      , 0 /*unknown capabilities*/,
         RIGCTRL_BAND_ALL_SHORTWAVE | RIGCTRL_BAND_2M | RIGCTRL_BAND_70CM }, // not sure..
  { RIGCTRL_DEF_ADDR_IC_R71      , "IC-R71"      , 0 /*unknown capabilities*/,
         RIGCTRL_BAND_ALL_SHORTWAVE | RIGCTRL_BAND_2M | RIGCTRL_BAND_70CM }, // not sure..
  { RIGCTRL_DEF_ADDR_IC_R72      , "IC-R72"      , 0 /*unknown capabilities*/,
         RIGCTRL_BAND_ALL_SHORTWAVE | RIGCTRL_BAND_2M | RIGCTRL_BAND_70CM }, // not sure..
  { RIGCTRL_DEF_ADDR_IC_R75      , "IC-R75"      , 0 /*unknown capabilities*/,
         RIGCTRL_BAND_ALL_SHORTWAVE | RIGCTRL_BAND_2M | RIGCTRL_BAND_70CM }, // not sure..
  { RIGCTRL_DEF_ADDR_IC_R7000    , "IC-R7000"    , 0 /*unknown capabilities*/,
         RIGCTRL_BAND_ALL_SHORTWAVE | RIGCTRL_BAND_2M | RIGCTRL_BAND_70CM }, // not sure..
  { RIGCTRL_DEF_ADDR_IC_R7100    , "IC-R7100"    , 0 /*unknown capabilities*/,
         RIGCTRL_BAND_ALL_SHORTWAVE | RIGCTRL_BAND_2M | RIGCTRL_BAND_70CM }, // not sure..
  { RIGCTRL_DEF_ADDR_IC_R8500    , "IC-R8500"    , 0 /*unknown capabilities*/,
         RIGCTRL_BAND_ALL_SHORTWAVE | RIGCTRL_BAND_2M | RIGCTRL_BAND_70CM }, // not sure..
  { RIGCTRL_DEF_ADDR_IC_R8600    , "IC-R8600" , RIGCTRL_CAPS_SPECTRUM_VIA_CAT,
         RIGCTRL_BAND_ALL_SHORTWAVE | RIGCTRL_BAND_2M | RIGCTRL_BAND_70CM }, // not sure..
  { RIGCTRL_DEF_ADDR_IC_R9000    , "IC-R9000"    , 0 /*unknown capabilities*/,
         RIGCTRL_BAND_ALL_SHORTWAVE | RIGCTRL_BAND_2M | RIGCTRL_BAND_70CM }, // not sure..
  { RIGCTRL_DEF_ADDR_IC_R9500    , "IC-R9500"    , 0 /*unknown capabilities*/,
         RIGCTRL_BAND_ALL_SHORTWAVE | RIGCTRL_BAND_2M | RIGCTRL_BAND_70CM }, // not sure..
  { RIGCTRL_DEF_ADDR_IC_RX7      , "IC-RX7"      , 0 /*unknown capabilities*/,
         RIGCTRL_BAND_ALL_SHORTWAVE | RIGCTRL_BAND_2M | RIGCTRL_BAND_70CM }, // not sure..
  { 0, NULL, 0, 0 } // <- and entry with "all zeroes" marks the end of the table
}; // end RigCtrl_RadioInfo_CIV[]

#if( SWI_SUPPORT_YAESU_5_BYTE_CAT )
const T_RigCtrl_RadioInfo RigCtrl_RadioInfo_Y5B[] = // radios with stoneage Yaesu "5-Byte-Command" CAT [Meow!]
{ // { iDefaultAddress,            pszName,        iCapabilities (if known),
  //      dwBands (bitwise combined, 32 bit parameter, see RIGCTRL_BAND_xyz)
  { RIGCTRL_DEF_ADDR_UNKNOWN_YAESU, "Unknown YAESU" , RIGCTRL_CAPS_POWER_ON_OFF_VIA_CAT/*capabilities*/,
          RIGCTRL_BAND_ALL_SHORTWAVE | RIGCTRL_BAND_2M | RIGCTRL_BAND_70CM },

  { RIGCTRL_DEF_ADDR_YAESU_FT_817, "FT-817",          RIGCTRL_CAPS_POWER_ON_OFF_VIA_CAT/*capabilities*/,
          RIGCTRL_BAND_ALL_SHORTWAVE | RIGCTRL_BAND_2M | RIGCTRL_BAND_70CM },

  { 0, NULL, 0, 0 } // <- and entry with "all zeroes" marks the end of the table

}; // end RigCtrl_RadioInfo_Y5B[]
#endif // SWI_SUPPORT_YAESU_5_BYTE_CAT ?



const T_SL_TokenList RigCtrl_BandNames[] =
{ // pszKeyword, iTokenValue :
  { "2200 m", RIGCTRL_BAND_2200M}, // note the space between number and SI unit [m]
  { "600 m",  RIGCTRL_BAND_600M },
  { "160 m",  RIGCTRL_BAND_160M },
  { "80 m",   RIGCTRL_BAND_80M  },
  { "60 m",   RIGCTRL_BAND_60M  },
  { "40 m",   RIGCTRL_BAND_40M  },
  { "30 m",   RIGCTRL_BAND_30M  },
  { "20 m",   RIGCTRL_BAND_20M  },
  { "17 m",   RIGCTRL_BAND_17M  },
  { "15 m",   RIGCTRL_BAND_15M  },
  { "12 m",   RIGCTRL_BAND_12M  },
  { "10 m",   RIGCTRL_BAND_10M  },
  { "Generic", RIGCTRL_BAND_GENERIC}, // inspired by Icom .. "GENE" appears in their BAND SELECTOR list
  { "6 m",    RIGCTRL_BAND_6M   },
  { "4 m",    RIGCTRL_BAND_4M   },
  { "2 m",    RIGCTRL_BAND_2M   },
  { "VHF Generic",RIGCTRL_BAND_R_VHF},
  { "70 cm",  RIGCTRL_BAND_70CM },
  { "23 cm",  RIGCTRL_BAND_23CM },
  { "13 cm",  RIGCTRL_BAND_13CM },
  { "9 cm" ,  RIGCTRL_BAND_9CM  },
  { "6 cm" ,  RIGCTRL_BAND_6CM  },
  { "3 cm" ,  RIGCTRL_BAND_3CM  },
  { NULL, 0 } // "all zeros" mark the end of the list
}; // end RigCtrl_BandNames[]


const T_SL_TokenList RigCtrl_OpModes[] =
{ { "CW",  RIGCTRL_OPMODE_CW },
  { "CWN", RIGCTRL_OPMODE_CW | RIGCTRL_OPMODE_NARROW },
  { "CWNN",RIGCTRL_OPMODE_CW | RIGCTRL_OPMODE_VERY_NARROW },
  { "LSB", RIGCTRL_OPMODE_LSB},
  { "USB", RIGCTRL_OPMODE_USB},
  { "AM",  RIGCTRL_OPMODE_AM },
  { "FM",  RIGCTRL_OPMODE_FM }, // IC-9700: "FM" with "FIL1" : '15 k' (bandwidth, only acceptable for 25 kHz channel spacing)
  { "FMN", RIGCTRL_OPMODE_FM | RIGCTRL_OPMODE_NARROW },     // IC-9700: "FM" with "FIL2" : '10 k'
  { "FMNN",RIGCTRL_OPMODE_FM | RIGCTRL_OPMODE_VERY_NARROW}, // IC-9700: "FM" with "FIL3" : '7 k'
  { "WFM", RIGCTRL_OPMODE_FM_WIDE },
  { "RTTY",RIGCTRL_OPMODE_RTTY},

  { "CWR",  RIGCTRL_OPMODE_CW | RIGCTRL_OPMODE_REVERSE },
  { "CWRN", RIGCTRL_OPMODE_CW | RIGCTRL_OPMODE_REVERSE | RIGCTRL_OPMODE_NARROW },

  { NULL, 0 } // "all zeros" mark the end of the list
}; // end RigCtrl_OpModes[]

const T_SL_TokenList RigCtrl_SplitModes[] =
{ // In certain Icom transceivers, "Split Mode" isn't simply ON or OFF !
  // Instead, the response for read-command 0x0F indicates
  // "Split Off" (0x00), "Split ON" (0x01), "DUP-" (0x11), "DUP+" (0x12),
  // or [IC-9700] "DD Repeater Simplex mode (RPS)" (0x13).
  // Most of the strings below are similar to the display in modern Icom rigs.
  { "off",   RIGCTRL_SPLIT_MODE_OFF }, // neither "Split" not "Duplex for Repeater operation"
  { "SPLIT", RIGCTRL_SPLIT_MODE_ON  }, // classic "Split" mode, usually a few kHz offset between RX and TX
  { "DUP-",  RIGCTRL_SPLIT_MODE_DUP_NEG }, // "DUP- operation" (IC-9700)
  { "DUP+",  RIGCTRL_SPLIT_MODE_DUP_POS }, // "DUP+ operation" (IC-9700)
  { "RPS",   RIGCTRL_SPLIT_MODE_REPEATER_SIMPLEX }, // <- just an IC-9700 extravaganza ? "DD Repeater Simplex mode (RPS)"

  { NULL, 0 } // "all zeros" mark the end of the list
}; // end RigCtrl_SplitModes[]




const T_RigCtrl_ParamInfo RigCtrl_ParameterInfo[] = // Originally "only for IC-7300", now a superset of later Icom-rigs, too !
{ // iUnifiedPN,                   pszToken,              pszUnit,  pStringValueTable,
  //  iByteOffsetIntoRigCtrlInstance  , iRigCtrlDataType, iAccessFlags,
  //    pbCIVReadCmd,           iCIVReadCmdLength,     iCIVDataType,
  //   iMsgType :

 { RIGCTRL_PN_TRANSCEIVER_ID,      "RadioID",             "",       NULL,
      // "Transceiver ID" alias "Radio ID" alias "Default CI-C Address" (holy crap..)
      offsetof( T_RigCtrlInstance,iDefaultAddress), RIGCTRL_DT_INT, RIGCTRL_AF_READONLY,
      (const BYTE*)"\x19\x00",     2,   CIV_DTYPE_HEX_1BYTE/* not CIV_DTYPE_BCD2 !*/,
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_FREQUENCY,           "VFOfreq",             "Hz",     NULL,
      offsetof( T_RigCtrlInstance, dblVfoFrequency), RIGCTRL_DT_DOUBLE, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x03"/*"Read the operating frequency"*/, 1,  CIV_DTYPE_BCD10,
      RIGCTRL_MSGTYPE_FREQUENCY_REPORT // <- this typically excludes the message from being displayed in the log !
 },
 { RIGCTRL_PN_OP_MODE,             "OpMode",              "",       RigCtrl_OpModes,
      offsetof( T_RigCtrlInstance, iOpMode),      RIGCTRL_DT_INT,  RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x04",         1,                     CIV_DTYPE_BCD4/*with "Mode" and "Filter"*/,
      RIGCTRL_MSGTYPE_OP_MODE
 },
 { RIGCTRL_PN_FILTER_BANDWIDTH,    "FilterBW",            "Hz",     NULL,
      offsetof( T_RigCtrlInstance, iFilterBW_Hz), RIGCTRL_DT_INT,  RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x1A\x03",     2,                     CIV_DTYPE_BCD2/*lousy CODES, not FREQUENCIES*/,
           // In mode *FM* (Frequency Modulation), an IC-9700 was not happy with "Cmd 1A Subcmd 03":
           // > TX 007 FE FE 00 00 1A 03 FD        ; read FilterBW
           // > RX 006 FE FE 00 A2 FA FD        ; FilterBW : NotOK
           // In mode *CW* (Morse code with "FIL2", an IC-9700 happily responded to "Cmd 1A Subcmd 03":
           // > TX 007 FE FE 00 00 1A 03 FD        ; read FilterBW
           // > RX 008 FE FE 00 A2 1A 03 09 FD   ; FilterBW=815 Hz
           // Expect more "fun" like this (special cases all over the place,
           // even the ENCODING of the stupid TWO-DIGIT BCD VALUE
           // depends on iOpMode (aka modulation type): See IC-9700 CI-V Reference "V4",
           // Icom document number "A7508-3EX-4", page 16. Read, dig it, and CRY OUT LOUD !
           //
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },

  // iUnifiedPN,                   pszToken,              pszUnit,  pStringValueTable,
  //  iByteOffsetIntoRigCtrlInstance  , iRigCtrlDataType, iAccessFlags,
  //  pbCIVReadCmd,           iCIVReadCmdLength,     iCIVDataType,
  //  iMsgType :
 { RIGCTRL_PN_DATA_MODE,           "DataMode",            "",       NULL,
      offsetof( T_RigCtrlInstance, iDataMode),     RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)NULL,           0,                     CIV_DTYPE_BCD2,
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_SEL_VFO_FREQUENCY,   "SelVfoFreq",          "Hz",     NULL, // used by WView and others (Hamlib?)
      offsetof( T_RigCtrlInstance, dblVfoFrequency), RIGCTRL_DT_DOUBLE, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x25\x00",     2,                     CIV_DTYPE_BCD10,
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND | RIGCTRL_MSGTYPE_FLAG_IS_ALIAS // here: "is alias for RIGCTRL_PN_FREQUENCY"
 },
 { RIGCTRL_PN_UNSEL_VFO_FREQUENCY, "UnselVfoFreq",        "Hz",     NULL,
      offsetof( T_RigCtrlInstance, dblUnselVfoFreq),RIGCTRL_DT_DOUBLE, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x25\x01",     2,                     CIV_DTYPE_BCD10,
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_SEL_VFO_OP_MODE,     "SelVfoOpMode",        "",       RigCtrl_OpModes,
      offsetof( T_RigCtrlInstance, iOpMode),        RIGCTRL_DT_INT,  RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x26\x00",     2,  CIV_DTYPE_BCD6/*with "OperatingMode", "DataMode", and "Filter"*/,
      // ,------------------------------------------'
      // '--> This strange combination of "operating mode", "data mode", and bandwidth
      //      requires special treatment to implement the GET- and SET-method,
      //      because multiple struct members of T_RigCtrlInstance are involved,
      //      not just .iOpMode - see RigCtrl_CIV_Server_MapDataIntoReadResponse()
      //                          and RigCtrl_ParseCIV() !
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND | RIGCTRL_MSGTYPE_FLAG_IS_ALIAS // here: "is alias for RIGCTRL_PN_OP_MODE"
 },
 { RIGCTRL_PN_UNSEL_VFO_OP_MODE,   "UnselVfoOpMode",      "",       RigCtrl_OpModes,
      offsetof( T_RigCtrlInstance, iUnselVfoOpMode),RIGCTRL_DT_INT,  RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x26\x01",     2,                 CIV_DTYPE_BCD6/*with "OperatingMode", "DataMode", and "Filter"*/,
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_TRANSMIT_REQUEST,    "TransmitReqst",       "",       NULL,
      offsetof( T_RigCtrlInstance, iTransmitReqst),RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)NULL/*!*/,      2,                     CIV_DTYPE_BCD2,
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_TRANSMITTING,        "Transmitting",        "",       NULL,
      offsetof( T_RigCtrlInstance, iTransmitting), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x1C\x00",     2,                     CIV_DTYPE_BCD2,
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND | RIGCTRL_MSGTYPE_FLAG_PERIODIC_POLL
 },
 { RIGCTRL_PN_ATU_ENABLED,         "ATU_enabled",         "",       NULL,
      offsetof( T_RigCtrlInstance, iATU_enabled ), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)NULL,           0,                     CIV_DTYPE_BCD2,
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_XFC_PRESSED,         "XFC_pressed",         "",       NULL,
      offsetof( T_RigCtrlInstance, iXFC_pressed ), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x1C\x02",     2,                     CIV_DTYPE_BCD2,
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_TRANSMIT_FREQ,       "TX_freq",             "Hz",     NULL,
      offsetof( T_RigCtrlInstance, dblTxFrequency), RIGCTRL_DT_DOUBLE, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x1C\x03",     2,                     CIV_DTYPE_BCD10,
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_REPEATER_TONE_FREQ,  "RptrTone",            "Hz",     NULL,
      offsetof( T_RigCtrlInstance, dblRepeaterTone_Hz),RIGCTRL_DT_DOUBLE, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x1C\x00",     2,                     CIV_DTYPE_BCD6,
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_TONE_SQUELCH_FREQ/*15*/,"ToneSquelch",      "Hz",     NULL,
      offsetof( T_RigCtrlInstance, dblToneSquelch_Hz), RIGCTRL_DT_DOUBLE, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x1C\x01",     2,                     CIV_DTYPE_BCD6,
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },

 //-- "TX Band Edge Frequencies", CI-V cmd 0x1E with various sub-commands -->
 { RIGCTRL_PN_NUM_TX_BANDS/*16*/,  "NumTxBands",           "",       NULL,
      offsetof( T_RigCtrlInstance, iNumTxBands),RIGCTRL_DT_INT,RIGCTRL_AF_READONLY,
      (const BYTE*)"\x1E\x00",     2,                     CIV_DTYPE_BCD2,
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_TX_BAND_EDGES/*17*/,"TxBandEdges",          "",       NULL,
      offsetof( T_RigCtrlInstance, TxBandEdges),RIGCTRL_DT_OTHER,RIGCTRL_AF_READONLY,
      (const BYTE*)"\x1E\x01",     2/*plus X*/, CIV_DTYPE_BAND_EDGE_FREQUENCIES,
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },

 // --- SPECTRUM SCOPE related (with CI-V, exchanged via command 0x27) ------>
 { RIGCTRL_PN_SCOPE_ON_OFF       , "ScopeOnOff",          "",       NULL,
      offsetof( T_RigCtrlInstance, iScopeOnOff),   RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x27\x10",     2,       CIV_DTYPE_BCD2,
      //                    |__|---> Note the ABSENCE of a "Main"/"Sub"-scope-selector here !
      RIGCTRL_MSGTYPE_SPECTRUM_CONFIG
      // For IC-7300: A7292-4EX-11 page 165: 0x27 0x10 0x00 = scope OFF, 0x27 0x10 0x01 = scope ON.
      // For IC-7610: A7380-7EX-2  page 10 : 0x27 0x10 0x00 = scope OFF, 0x27 0x10 0x01 = scope ON.
      // (in this case, no sub-subcode for "main scope" / "sub scope".
      //  In addition to the "scope ON / OFF status" cmd 0x27 0x10,
      //  they also have a "Scope wave data output ON / OFF", cmd 0x27 0x11   in the IC-7610,
      //  and a "Main or Sub scope setting",  cmd 0x27 0x12: 00=MAIN, 01=SUB  in the IC-7610,
      //  plus a "Single/Dual scope setting", cmd 0x27 0x13: 00=Single,01=Dual in the IC-7610.
      //  The IC-9700 understands 0x27 0x10, 0x27 0x11, 0x27 0x12, but not 0x27 0x13.
      //  For the IC-9700, page 12 explains cmd 0x26 0x00 :
      //            > Read the Scope waveform data
      //            > ( Only when "Scope ON/OFF status"
      //            > (Command: 27 10) and "Scope data output"
      //            > (Command: 27 11) are set to "ON," outputs
      //            > the waveform data to the controller.)
      //      In RigControl.c, processing of the SCOPE WAVEFORM DATA (0x27 0x00)
      //      does NOT use a 'unified parameter number', but is hard-coded.
      //  So far, RCWK doesn't support any of those esoteric scope settings listed above)
 },
 { RIGCTRL_PN_SCOPE_MODE         , "ScopeMode",           "",       NULL,
      offsetof( T_RigCtrlInstance, iScopeMode),    RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x27\x14\0x00",  3,       CIV_DTYPE_BCD2,
      // ,-------------------|__|
      // '--> 0x00=MAIN scope, 0x01=SUB scope. Says A7508-3EX-4 page 25 for the IC-9700 !
      // '--> "Fixed to 0x00", says A7292-4EX-11 page 172 for the IC-7300 . Ok, compatible.
      RIGCTRL_MSGTYPE_SPECTRUM_CONFIG
 },
 { RIGCTRL_PN_SCOPE_SPAN         , "ScopeSpan",           "Hz",     NULL,
      offsetof( T_RigCtrlInstance, dblScopeSpan_Hz), RIGCTRL_DT_DOUBLE, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x27\x15\x00"/*main scope*/, 3,        CIV_DTYPE_BCD10,
      RIGCTRL_MSGTYPE_SPECTRUM_CONFIG
 },
 { RIGCTRL_PN_SCOPE_CENTER_FREQUENCY, "ScopeCenterFreq",  "Hz",     NULL,
      offsetof( T_RigCtrlInstance, dblSpectrumCenterFreq_Hz), RIGCTRL_DT_DOUBLE, RIGCTRL_AF_READWRITE,
      NULL/*there's no CI-V command for this ! */,0,       CIV_DTYPE_BCD10,
      // (If the scope center frequency shall be different from the VFO frequency,
      //  Icom's "fixed edge mode" must be used. Of they add a parameter number
      //  for the "scroll offset" in "Scroll Center" or "Scroll Fixed" mode, one fine day.)
      RIGCTRL_MSGTYPE_SPECTRUM_CONFIG
 },
 { RIGCTRL_PN_SCOPE_EDGE_NR      , "ScopeEdge",           "",       NULL,
      offsetof( T_RigCtrlInstance, iScopeEdgeNumber), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x27\x16\x00"/*main scope*/, 3,       CIV_DTYPE_BCD2,
      RIGCTRL_MSGTYPE_SPECTRUM_CONFIG
 },
 { RIGCTRL_PN_SCOPE_HOLD_FLAG    , "ScopeHold",           "",       NULL,
      offsetof( T_RigCtrlInstance, iScopeHoldFlag), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x27\x17\x00"/*main scope*/, 3,       CIV_DTYPE_BCD2,
      RIGCTRL_MSGTYPE_SPECTRUM_CONFIG
 },
 { RIGCTRL_PN_SCOPE_ATTEN        , "ScopeAtt",            "dB",     NULL,
      offsetof( T_RigCtrlInstance, dblScopeAttenuator_dB), RIGCTRL_DT_DOUBLE, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x27\x18\x00"/*main scope*/, 3,       CIV_DTYPE_BCD2,
      RIGCTRL_MSGTYPE_SPECTRUM_CONFIG
 },
 { RIGCTRL_PN_SCOPE_REF_LEVEL    , "ScopeRefLevel",       "dB",     NULL,
      offsetof( T_RigCtrlInstance, dblScopeRefLevel_dB), RIGCTRL_DT_DOUBLE, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x27\x19\x00"/*main scope*/, 3,       CIV_DTYPE_BCD4 | CIV_DTYPE_SUFFIX_BCD2_SIGN,
      RIGCTRL_MSGTYPE_SPECTRUM_CONFIG
 },
 { RIGCTRL_PN_SCOPE_SPEED        , "ScopeSpeed",          "",       NULL,
      offsetof( T_RigCtrlInstance, iScopeSpeed), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x27\x1A\x00"/*main scope*/, 3,       CIV_DTYPE_BCD4,
      //              |       '--> For the IC-9700: 0x00=MAIN, 0x01=SUB .
      //              |            For the IC-7300: "Fixed 0x00" .
      //              '--> That's Icom's "Scope Sweep Speed", not to be confused
      //                   with cmd 0x1A sub 0x05 "0166" = "Scroll Speed";
      // and not to be confused with cmd 0x1A sub 0x05 "0198" = "Waterfall Speed" !
      // Confusing, isn't it ? The IC-7300 "Full Manual" explains :
      //  > Sweep speed (WB: Taking about softkey "Speed" below the WATERFALL)
      //  > Select the sweep speed to change the FFT scope refresh speed
      //  > and the waterfall speed.
      //  >  (info) To change only the waterfall speed, select "Slow,"
      //  >         "Mid," or "Fast" in the Scope set screen. (p. 5-7)
      // Actually, the WATERFALL SCROLL SPEED ("spectrum lines per second")
      // depends on BOTH:
      //  * The "Waterfall Speed" in the "SCOPE SET" menu
      //  * The "Speed" SOFTKEY below the WATERFALL DISPLAY
      RIGCTRL_MSGTYPE_SPECTRUM_CONFIG
 },
 { RIGCTRL_PN_WATERFALL_SPEED    , "WaterfallSpeed",   "",       NULL,
      offsetof( T_RigCtrlInstance, iWaterfallSpeed), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)NULL/*command is MODEL DEPENDENT !*/, 0, CIV_DTYPE_BCD2,
      // '--> One of the dreadful COMMAND with a MODEL-DEPENDENT sub-subcommand !"
      //      .. thus the appropriate read/write command cannot be encoded in this table,
      //         but is a special case in RigCtrl_SendReadCommandForUnifiedPN(),
      //                                  RigCtrl_SendWriteCommandForUnifiedPN(),
      //              and RigCtrl_SubcodeFrom1A05_to_UnifiedParameterNumber() !
      RIGCTRL_MSGTYPE_SPECTRUM_CONFIG
 },
 { RIGCTRL_PN_SCOPE_DURING_TX    , "ScopeDuringTX",       "",       NULL,
      offsetof( T_RigCtrlInstance, iScopeDuringTX), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x27\x1B\x00"/*main scope*/, 3,       CIV_DTYPE_BCD4,
      RIGCTRL_MSGTYPE_SPECTRUM_CONFIG
 },
 { RIGCTRL_PN_SCOPE_CENTER_TYPE,   "ScopeCenterType",     "",       NULL,
      offsetof( T_RigCtrlInstance, iScopeCenterType), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x27\x1C\x00"/*main scope*/, 3,       CIV_DTYPE_BCD4,
      RIGCTRL_MSGTYPE_SPECTRUM_CONFIG
 },
 { RIGCTRL_PN_SCOPE_VBW,           "ScopeVBW",            "",       NULL,
      offsetof( T_RigCtrlInstance, iScopeVideoBW),  RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x27\x1D\x00"/*main scope*/, 3,       CIV_DTYPE_BCD4,
      RIGCTRL_MSGTYPE_SPECTRUM_CONFIG
 },
 { RIGCTRL_PN_SCOPE_FIXED_EDGES,   "ScopeFixedEdges",     "",       NULL,
      offsetof( T_RigCtrlInstance, ScopeFreqRange), RIGCTRL_DT_FREQ_RANGE, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x27\x1E\x00"/*main scope*/, 3,       CIV_DTYPE_SCOPE_EDGE_FREQUENCIES,
      RIGCTRL_MSGTYPE_SPECTRUM_CONFIG
 },
 { RIGCTRL_PN_SCOPE_MARKER_POS_TYPE,"ScopeMarkerPosType", "",       NULL,
      offsetof( T_RigCtrlInstance, iScopeMarkerPosType),  RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x27\x20\x00"/*main scope*/, 3,       CIV_DTYPE_BCD4,
      RIGCTRL_MSGTYPE_SPECTRUM_CONFIG
 },


 //--- Various "poti" settings (with CI-V, exchanged via command 0x14) ------>
 { RIGCTRL_PN_AUDIO_VOLUME_PERCENT, "AudioVolume",        "%",      NULL,
      offsetof( T_RigCtrlInstance, iAudioVolume_Percent), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x14\x01",     2, CIV_DTYPE_BCD4 | CIV_DTYPE_FLAG_MSBYTE_FIRST | CIV_DTYPE_FLAG_PERCENT255,
      RIGCTRL_MSGTYPE_POTI_SETTING
 },
 { RIGCTRL_PN_RF_GAIN_PERCENT    , "RFGain",              "%",      NULL,
      offsetof( T_RigCtrlInstance, iRFGain_Percent), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x14\x02",     2, CIV_DTYPE_BCD4 | CIV_DTYPE_FLAG_MSBYTE_FIRST | CIV_DTYPE_FLAG_PERCENT255,
      RIGCTRL_MSGTYPE_POTI_SETTING
 },
 { RIGCTRL_PN_SQUELCH_LEVEL_PERCENT, "SquelchLevel",      "%",      NULL,
      offsetof( T_RigCtrlInstance, iSquelchLevel_Percent), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x14\x03",     2, CIV_DTYPE_BCD4 | CIV_DTYPE_FLAG_MSBYTE_FIRST | CIV_DTYPE_FLAG_PERCENT255,
      RIGCTRL_MSGTYPE_POTI_SETTING
 },
 { RIGCTRL_PN_NOISE_REDUCTION_PERCENT, "NoiseReduction",  "%",      NULL,
      offsetof( T_RigCtrlInstance, iNoiseReduction_Percent), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x14\x06",     2, CIV_DTYPE_BCD4 | CIV_DTYPE_FLAG_MSBYTE_FIRST | CIV_DTYPE_FLAG_PERCENT255,
      RIGCTRL_MSGTYPE_POTI_SETTING
 },
 { RIGCTRL_PN_PASSBAND_TUNING_POS1, "PassbandT1",         "%",      NULL, // "inner ring" of "TWIN PBT", 50 % = center
      offsetof( T_RigCtrlInstance, iPassbandTuningPos1_Percent), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x14\x07",     2, CIV_DTYPE_BCD4 | CIV_DTYPE_FLAG_MSBYTE_FIRST | CIV_DTYPE_FLAG_PERCENT255,
      RIGCTRL_MSGTYPE_POTI_SETTING
 },
 { RIGCTRL_PN_PASSBAND_TUNING_POS2, "PassbandT2",         "%",      NULL, // "outer ring" of "TWIN PBT", 50 % = center
      offsetof( T_RigCtrlInstance, iPassbandTuningPos2_Percent), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x14\x08",     2, CIV_DTYPE_BCD4 | CIV_DTYPE_FLAG_MSBYTE_FIRST | CIV_DTYPE_FLAG_PERCENT255,
      RIGCTRL_MSGTYPE_POTI_SETTING
 },
 { RIGCTRL_PN_CW_PITCH_HZ        , "CW_Pitch",            "Hz",     NULL,
      offsetof( T_RigCtrlInstance, iCWPitch_Hz), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x14\x09",     2, CIV_DTYPE_BCD4 | CIV_DTYPE_FLAG_MSBYTE_FIRST /* special scaling !!*/,
      RIGCTRL_MSGTYPE_POTI_SETTING
 },
 { RIGCTRL_PN_RF_POWER_SETTING_PERCENT, "PowerSet",       "%",      NULL, // "setting" to avoid confusion with the POWER METER MEASUREMENT !
      offsetof( T_RigCtrlInstance, iRFPowerSetting_Percent), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x14\x0A",     2, CIV_DTYPE_BCD4 | CIV_DTYPE_FLAG_MSBYTE_FIRST | CIV_DTYPE_FLAG_PERCENT255,
      RIGCTRL_MSGTYPE_POTI_SETTING
 },
 { RIGCTRL_PN_MIC_GAIN_PERCENT   , "MicGain",             "%",      NULL,
      offsetof( T_RigCtrlInstance, iMicGain_Percent), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x14\x0B",     2, CIV_DTYPE_BCD4 | CIV_DTYPE_FLAG_MSBYTE_FIRST | CIV_DTYPE_FLAG_PERCENT255,
      RIGCTRL_MSGTYPE_POTI_SETTING
 },
 { RIGCTRL_PN_KEYER_SPEED_WPM,     "KeyerSpeed",          "WPM",    NULL, // no stupid "percents" but WPM (Words Per Minute). Icom's range: "0 %" = 6 WPM ... "100 %" = 48 WPM
      offsetof( T_RigCtrlInstance, iKeyerSpeed_WPM), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x14\x0C",     2, CIV_DTYPE_BCD4 | CIV_DTYPE_FLAG_MSBYTE_FIRST | CIV_DTYPE_FLAG_PERCENT255,
      RIGCTRL_MSGTYPE_POTI_SETTING
 },
 { RIGCTRL_PN_NOTCH_POS_PERCENT  , "NotchPos",            "%",      NULL,
      offsetof( T_RigCtrlInstance, iNotchPos_Percent), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x14\x0D",     2, CIV_DTYPE_BCD4 | CIV_DTYPE_FLAG_MSBYTE_FIRST | CIV_DTYPE_FLAG_PERCENT255,
      RIGCTRL_MSGTYPE_POTI_SETTING
 },
 { RIGCTRL_PN_COMP_SETTING_PERCENT, "CompSet",            "%",      NULL, // audio compressor SETTING (not a "measurement")
      offsetof( T_RigCtrlInstance, iCompSetting_Percent), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x14\x0E",     2, CIV_DTYPE_BCD4 | CIV_DTYPE_FLAG_MSBYTE_FIRST | CIV_DTYPE_FLAG_PERCENT255,
      RIGCTRL_MSGTYPE_POTI_SETTING
 },
 { RIGCTRL_PN_BREAK_IN_DELAY_PERCENT, "BkInDelay",        "%",      NULL,
      offsetof( T_RigCtrlInstance, iBreakInDelay_Percent), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x14\x0F",     2, CIV_DTYPE_BCD4 | CIV_DTYPE_FLAG_MSBYTE_FIRST | CIV_DTYPE_FLAG_PERCENT255,
      RIGCTRL_MSGTYPE_POTI_SETTING
 },
 { RIGCTRL_PN_NOISE_BLANKER_PERCENT, "NoiseBlanker",      "%",      NULL,
      offsetof( T_RigCtrlInstance, iNoiseBlanker_Percent), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x14\x12",     2, CIV_DTYPE_BCD4 | CIV_DTYPE_FLAG_MSBYTE_FIRST | CIV_DTYPE_FLAG_PERCENT255,
      RIGCTRL_MSGTYPE_POTI_SETTING
 },
 { RIGCTRL_PN_MONITOR_GAIN_PERCENT, "MonitorGain",        "%",      NULL,
      offsetof( T_RigCtrlInstance, iMonitorGain_Percent), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x14\x15",     2, CIV_DTYPE_BCD4 | CIV_DTYPE_FLAG_MSBYTE_FIRST | CIV_DTYPE_FLAG_PERCENT255,
      RIGCTRL_MSGTYPE_POTI_SETTING
 },
 { RIGCTRL_PN_VOX_GAIN_PERCENT   , "VoxGain",             "%",      NULL,
      offsetof( T_RigCtrlInstance, iVoxGain_Percent), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x14\x16",     2, CIV_DTYPE_BCD4 | CIV_DTYPE_FLAG_MSBYTE_FIRST | CIV_DTYPE_FLAG_PERCENT255,
      RIGCTRL_MSGTYPE_POTI_SETTING
 },
 { RIGCTRL_PN_ANTI_VOX_GAIN_PERCENT, "AntiVoxGain",       "%",      NULL,
      offsetof( T_RigCtrlInstance, iAntiVoxGain_Percent), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x14\x17",     2, CIV_DTYPE_BCD4 | CIV_DTYPE_FLAG_MSBYTE_FIRST | CIV_DTYPE_FLAG_PERCENT255,
      RIGCTRL_MSGTYPE_POTI_SETTING
 },
 { RIGCTRL_PN_BRIGHTNESS_PERCENT , "Brightness",          "%",      NULL,
      offsetof( T_RigCtrlInstance, iBrightness_Percent), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x14\x19",     2, CIV_DTYPE_BCD4 | CIV_DTYPE_FLAG_MSBYTE_FIRST | CIV_DTYPE_FLAG_PERCENT255,
      RIGCTRL_MSGTYPE_POTI_SETTING
 },


 //------ Various "meter" readings and related statuses --------------------->
 { RIGCTRL_PN_SQUELCH_STATUS     , "SquelchStatus",       "",       NULL,
      offsetof( T_RigCtrlInstance, iSquelchStatus), RIGCTRL_DT_INT, RIGCTRL_AF_READONLY,
      (const BYTE*)"\x15\x01",     2, CIV_DTYPE_BCD2,
      RIGCTRL_MSGTYPE_READ_METER | RIGCTRL_MSGTYPE_FLAG_PERIODIC_POLL
 },
 { RIGCTRL_PN_S_METER_LEVEL_DB   , "SMeterLevel",         "dB",     NULL, // here: "decibel over S0",  not the strange CI-V internal unit
      offsetof( T_RigCtrlInstance, iSMeterLevel_dB), RIGCTRL_DT_INT, RIGCTRL_AF_READONLY,
      (const BYTE*)"\x15\x02",     2, CIV_DTYPE_BCD4 | CIV_DTYPE_FLAG_MSBYTE_FIRST | CIV_DTYPE_FLAG_PERCENT255,
      RIGCTRL_MSGTYPE_READ_METER | RIGCTRL_MSGTYPE_FLAG_PERIODIC_POLL
 },
 { RIGCTRL_PN_ADC_OVERFLOW_STATUS, "OverflowStat",        "",       NULL,
      offsetof( T_RigCtrlInstance, iADCOverflowStatus), RIGCTRL_DT_INT, RIGCTRL_AF_READONLY,
      (const BYTE*)"\x15\x07",     2, CIV_DTYPE_BCD2,
      RIGCTRL_MSGTYPE_READ_METER | RIGCTRL_MSGTYPE_FLAG_PERIODIC_POLL
 },
 { RIGCTRL_PN_POWER_METER_PERCENT, "PowerMeter",          "%",      NULL,
      offsetof( T_RigCtrlInstance, iPowerMeterLevel_pcnt), RIGCTRL_DT_INT, RIGCTRL_AF_READONLY,
      (const BYTE*)"\x15\x11",     2, CIV_DTYPE_BCD4 | CIV_DTYPE_FLAG_MSBYTE_FIRST | CIV_DTYPE_FLAG_PERCENT255,
      RIGCTRL_MSGTYPE_READ_METER | RIGCTRL_MSGTYPE_FLAG_PERIODIC_POLL
 },
 { RIGCTRL_PN_SWR_METER_VALUE    , "SWRMeter",            "",       NULL, // dimensionless, usual value range: 1.0 (perfect) to 3.0 (poor)
      offsetof( T_RigCtrlInstance, dblSWRMeterValue),    RIGCTRL_DT_DOUBLE, RIGCTRL_AF_READONLY,
      (const BYTE*)"\x15\x12",     2, CIV_DTYPE_BCD4 | CIV_DTYPE_FLAG_MSBYTE_FIRST | CIV_DTYPE_FLAG_PERCENT255,
      RIGCTRL_MSGTYPE_READ_METER | RIGCTRL_MSGTYPE_FLAG_PERIODIC_POLL
 },
 { RIGCTRL_PN_ALC_METER_LEVEL    , "ALCMeter",            "%",      NULL,
      offsetof( T_RigCtrlInstance, iALCMeterLevel_pcnt), RIGCTRL_DT_INT, RIGCTRL_AF_READONLY,
      (const BYTE*)"\x15\x13",     2, CIV_DTYPE_BCD4 | CIV_DTYPE_FLAG_MSBYTE_FIRST | CIV_DTYPE_FLAG_PERCENT255,
      RIGCTRL_MSGTYPE_READ_METER | RIGCTRL_MSGTYPE_FLAG_PERIODIC_POLL
 },
 { RIGCTRL_PN_COMP_METER_LEVEL_DB, "CompMeter",           "dB",     NULL, // decibel, 0=no compression, 30 dB = maximum reading for IC-7300)
      offsetof( T_RigCtrlInstance, iCompMeterLevel_dB), RIGCTRL_DT_INT, RIGCTRL_AF_READONLY,
      (const BYTE*)"\x15\x14",     2, CIV_DTYPE_BCD4 | CIV_DTYPE_FLAG_MSBYTE_FIRST | CIV_DTYPE_FLAG_PERCENT255,
      RIGCTRL_MSGTYPE_READ_METER | RIGCTRL_MSGTYPE_FLAG_PERIODIC_POLL
 },
 { RIGCTRL_PN_SUPPLY_VOLTAGE_mV  , "SupplyVoltage",       "mV",     NULL,
      offsetof( T_RigCtrlInstance, iSupplyVoltage_mV), RIGCTRL_DT_INT, RIGCTRL_AF_READONLY,
      (const BYTE*)"\x15\x15", /* "Vd" */  2, CIV_DTYPE_BCD4 | CIV_DTYPE_FLAG_MSBYTE_FIRST | CIV_DTYPE_FLAG_PERCENT255,
      RIGCTRL_MSGTYPE_READ_METER | RIGCTRL_MSGTYPE_FLAG_PERIODIC_POLL
 },
 { RIGCTRL_PN_DRAIN_CURRENT_mA   , "DrainCurrent",        "mA",     NULL,
      offsetof( T_RigCtrlInstance, iDrainCurrent_mA), RIGCTRL_DT_INT, RIGCTRL_AF_READONLY,
      (const BYTE*)"\x15\x16", /* "Id" */  2, CIV_DTYPE_BCD4 | CIV_DTYPE_FLAG_MSBYTE_FIRST | CIV_DTYPE_FLAG_PERCENT255,
      RIGCTRL_MSGTYPE_READ_METER | RIGCTRL_MSGTYPE_FLAG_PERIODIC_POLL
 },
 { RIGCTRL_PN_TEMPERATURE_CELSIUS, "Temperature",         "C",     NULL,
      offsetof( T_RigCtrlInstance, iPATemperature_degC), RIGCTRL_DT_INT, RIGCTRL_AF_READONLY,
      NULL, /* Icom please implement ! */  0,             CIV_DTYPE_NONE/* doesn't exist in CI-V*/,
      RIGCTRL_MSGTYPE_READ_METER | RIGCTRL_MSGTYPE_FLAG_PERIODIC_POLL
 },



 //-- extended Preamp, AGC speed, Noise blanker, Audio peak filter,
 //   noise reduction, auto notch, etc. With CI-V, via command 0x16) -----
 { RIGCTRL_PN_PREAMP_SETTING,      "PreampSet",           "",       NULL,
      offsetof( T_RigCtrlInstance, iPreampSet), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x16\x02",     2,                     CIV_DTYPE_BCD2,
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_AGC_SPEED,           "AGCSpeed",            "",       NULL,
      offsetof( T_RigCtrlInstance, iAGCSpeed), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x16\x12",     2,                     CIV_DTYPE_BCD2,
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_NOISE_BLANKER_ON,    "NoiseBlankerON",      "",       NULL,
      offsetof( T_RigCtrlInstance, iNoiseBlankerON),RIGCTRL_DT_INT,RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x16\x22",     2,                     CIV_DTYPE_BCD2,
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_NOISE_REDUCTION_ON,  "NoiseReductON",       "",       NULL,
      offsetof( T_RigCtrlInstance, iNoiseReductON),RIGCTRL_DT_INT,RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x16\x40",     2,                     CIV_DTYPE_BCD2,
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_AUTO_NOTCH_ON,       "AutoNotchON",         "",       NULL,
      offsetof( T_RigCtrlInstance, iAutoNotchON), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x16\x41",     2,                     CIV_DTYPE_BCD2,
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_RPTR_TONE_ON,        "RptrToneON",          "",       NULL,
      offsetof( T_RigCtrlInstance, iRptrToneON), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x16\x42",     2,                     CIV_DTYPE_BCD2,
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_TONE_SQUELCH_ON,     "ToneSquelchON",       "",       NULL,
      offsetof( T_RigCtrlInstance, iToneSquelchON), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x16\x43",     2,                     CIV_DTYPE_BCD2,
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_COMP_ON,             "CompressON",          "",       NULL,
      offsetof( T_RigCtrlInstance, iCompressON), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x16\x44",     2,                     CIV_DTYPE_BCD2,
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_MONITOR_ON,          "MonitorON",           "",       NULL,
      offsetof( T_RigCtrlInstance, iMonitorON), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x16\x45",     2,                     CIV_DTYPE_BCD2,
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_VOX_ON,              "VoxON",               "",       NULL,
      offsetof( T_RigCtrlInstance, iVoxON), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x16\x46",     2,                     CIV_DTYPE_BCD2,
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_BK_IN_MODE, "BreakInMode", /* not just "on/off"*/ "", NULL,
      offsetof( T_RigCtrlInstance, iBreakInMode), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x16\x47",     2,                     CIV_DTYPE_BCD2,
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_MANUAL_NOTCH_ON,     "ManualNotchON",       "",       NULL,
      offsetof( T_RigCtrlInstance, iManualNotchON), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x16\x48",     2,                     CIV_DTYPE_BCD2,
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_MANUAL_NOTCH_WIDTH,  "ManualNotchWidth",    "",       NULL,
      offsetof( T_RigCtrlInstance, iManualNotchWidth),RIGCTRL_DT_INT,RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x16\x57",     2,                     CIV_DTYPE_BCD2,
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_TWIN_PEAK_FILTER_ON, "TwinPeakFilterON",    "",       NULL,
      offsetof( T_RigCtrlInstance, iTwinPeakFilterON),RIGCTRL_DT_INT,RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x16\x4F",     2,                     CIV_DTYPE_BCD2,
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_DIAL_LOCK_ON,        "DialLockON",          "",       NULL,
      offsetof( T_RigCtrlInstance, iDialLockON), RIGCTRL_DT_INT,RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x16\x50",     2,                     CIV_DTYPE_BCD2,
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_DSP_SHARP_SOFT,      "DSP_SharpSoft",       "",       NULL,
      offsetof( T_RigCtrlInstance, iDSP_SharpSoft),RIGCTRL_DT_INT,RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x16\x56",     2,                     CIV_DTYPE_BCD2,
            RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_SSB_TX_BANDWIDTH,    "SSB_Tx_BW",           "",       NULL,
      offsetof( T_RigCtrlInstance, iSSB_Tx_BW),RIGCTRL_DT_INT,RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x16\x58",     2,                     CIV_DTYPE_BCD2,
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_SUBBAND_ON,          "SubBandON",           "",       NULL,  // aka "Dual Watch"
      offsetof( T_RigCtrlInstance, iSubbandON),RIGCTRL_DT_INT,RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x16\x59",     2,                     CIV_DTYPE_BCD2,
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_SATELLITE_MODE,      "SatelliteMode",       "",       NULL,
      offsetof( T_RigCtrlInstance, iSatelliteMode),RIGCTRL_DT_INT,RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x16\x5A",     2,                     CIV_DTYPE_BCD2,
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_IMPROVED_IP3,       "ImprovedIP3",          "",       NULL,
      offsetof( T_RigCtrlInstance, iImprovedIP3),  RIGCTRL_DT_INT,RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x16\x65",     2,                     CIV_DTYPE_BCD2,
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND | RIGCTRL_MSGTYPE_FLAG_PERIODIC_POLL
      // For the WFView developers, the "Improved IP3" setting was so important
      // that they included it in the periodic polling cycle. Thus it's here, too.
 },

 //-- "Special" settings, audio equalizer, connector settings, etc ---------->
 //    (Many of these use the DREADFUL MODEL-SPECIFIC 4-digit numbers after cmd 0x1A 0x05,
 //     so make sure when adding new entries HERE, also add support for them
 //     in RigCtrl_GetCrazyModelDependingSubSubcodeForCmd1A_05(). Nomen est omen.)
 { RIGCTRL_PN_SSB_RX_HPF_LPF,      "SSB_RX_HPF_LPF",      "",       NULL,
      offsetof( T_RigCtrlInstance, iSSB_RX_HPF_LPF), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x1A\x05\x00\x01",  4/*plus X*/, CIV_DTYPE_BCD4,
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_SSB_RX_BASS_LEVEL,   "SSB_RX_BASS_LEVEL",   "",       NULL,
      offsetof( T_RigCtrlInstance, iSSB_RX_BASS_LEVEL), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)NULL,           0, CIV_DTYPE_NONE,
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_SSB_RX_TREBLE_LEVEL, "SSB_RX_TREBLE_LEVEL", "",       NULL,
      offsetof( T_RigCtrlInstance, iSSB_RX_TREBLE_LEVEL), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)NULL,           0, CIV_DTYPE_NONE,
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_SSB_CW_SYNC_TUNING,  "SSB_CW_sync_tuning",  "",       NULL,
      offsetof( T_RigCtrlInstance, iSSB_CW_sync_tuning), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)NULL,           0, CIV_DTYPE_NONE,
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_CALIBRATION_MARKER,  "calib_marker",        "",       NULL,
      offsetof( T_RigCtrlInstance, iCalibMarker), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)NULL,           0, CIV_DTYPE_NONE,
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_SIDETONE_ON_USB_AUDIO, "Sidetone_on_USB", "",  NULL, // precisely: "CW Sidetone and other 'beeps' on ACC and USB-audio-output" (IC-7300)
      offsetof( T_RigCtrlInstance, iSidetone_on_USB), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x1A\x05\x00\x62",  4, CIV_DTYPE_BCD2/*guesswork, as usual*/,
         // |--> IC-7300: "Beep and speech output setting to ACC/USB" 0x1A 0x05 #0062 . A7292-4EX-11 page 163.
         // '--> Any other Icom radio uses a CRAZY DIFFERENT MODEL-DEPENDENT sub-subcode !
         //      For all those OTHER Icom-radios, the sub-subcode can be retrieved
         //      via RigCtrl_GetCrazyModelDependingSubSubcodeForCmd1A_05(). Nomen est omen.
         RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
      // Notes about this "Sidetone on USB" :
      //  * The IC-7300 has a single command to control what they call "Beeps" on both ACC and USB.
      //    The IC-9700 has separate commands for ACC, USB, and LAN(!).
      //  * RCW Keyer can be configured as explained in
      //    file:///C:/cbproj/Remote_CW_Keyer/manual/Remote_CW_Keyer.htm#Sidetone_when_keyed_ON_THE_RIG
      //    to AUTOMATICALLY switch between pRC->iSidetone_on_USB = 0 (no sidetone added to the USB-Audio-output BY THE RIG)
      //                                and pRC->iSidetone_on_USB = 1 (let the rig add its CW-sidetone to the USB-Audio-output)
      //    That happens in RigControl.c : RigCtrl_OnTransmitFlagChange() .
 },
 { RIGCTRL_PN_REFERENCE_FREQ_OFFSET, "ReferenceFrequencyOffset",  "%",  NULL,
      offsetof( T_RigCtrlInstance, iReferenceFrequencyOffset), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x1A\x05\x00\x58"/*IC7300*/, 4, CIV_DTYPE_BCD4 | CIV_DTYPE_FLAG_MSBYTE_FIRST | CIV_DTYPE_FLAG_PERCENT255,
         // |--> IC-7300: "Send/read reference frequency" 0x1A 0x05 #0058 . A7292-4EX-11 page 162.
         // '--> Any other Icom radio uses a CRAZY DIFFERENT MODEL-DEPENDENT sub-subcode !
         //      For all those OTHER Icom-radios, the sub-subcode can be retrieved
         //      via RigCtrl_GetCrazyModelDependingSubSubcodeForCmd1A_05(). Nomen est omen.
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_AF_IF_OUT_SELECTOR, "AF_IF_OutputSelector",  "",  NULL,
      offsetof( T_RigCtrlInstance, iAF_IF_OutputSelectOnACC_USB), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x1A\x05\x00\x59"/*IC7300*/, 4, CIV_DTYPE_BCD2/*guesswork, as usual*/,
         // |--> IC-7300: "Send/read AF/IF signal output to ACC/USB" 0x1A 0x05 #0059 . A7292-4EX-11 page 162.
         // '--> Any other Icom radio uses a CRAZY DIFFERENT MODEL-DEPENDENT sub-subcode !
         //      For all those OTHER Icom-radios, the sub-subcode can be retrieved
         //      via RigCtrl_GetCrazyModelDependingSubSubcodeForCmd1A_05(). Nomen est omen.
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_SQUELCH_FOR_AF_ON_USB, "SquelchForAudioOnUSB",  "",  NULL,
      offsetof( T_RigCtrlInstance, iSquelch_for_USB), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x1A\x05\x00\x61"/*IC7300*/, 4, CIV_DTYPE_BCD2/*guesswork, as usual*/,
         // |--> IC-7300: "Send/read squelch function for the AF signal output to ACC/USB" 0x1A 0x05 #0061 . A7292-4EX-11 page 163.
         // '--> Any other Icom radio uses a CRAZY DIFFERENT MODEL-DEPENDENT sub-subcode !
         //      For all those OTHER Icom-radios, the sub-subcode can be retrieved
         //      via RigCtrl_GetCrazyModelDependingSubSubcodeForCmd1A_05(). Nomen est omen.
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_AF_OUTPUT_LEVEL_TO_USB, "AF_OutputLevelOnUSB",  "%",  NULL,
      offsetof( T_RigCtrlInstance, iAF_OutputLevelOnACC_USB), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x1A\x05\x00\x60"/*IC7300*/, 4, CIV_DTYPE_BCD4 | CIV_DTYPE_FLAG_MSBYTE_FIRST | CIV_DTYPE_FLAG_PERCENT255,
         // |--> IC-7300: "Send/read AF output level to ACC/USB" 0x1A 0x05 #0061 . A7292-4EX-11 page 162.
         // '--> Any other Icom radio uses a CRAZY DIFFERENT MODEL-DEPENDENT sub-subcode !
         //      For all those OTHER Icom-radios, the sub-subcode can be retrieved
         //      via RigCtrl_GetCrazyModelDependingSubSubcodeForCmd1A_05(). Nomen est omen.
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_IF_OUTPUT_LEVEL_TO_ACC, "IF_OutputLevelOnACC",  "%",  NULL,
      offsetof( T_RigCtrlInstance, iIF_OutputLevelOnACC_USB), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x1A\x05\x00\x63"/*IC7300*/, 4, CIV_DTYPE_BCD4 | CIV_DTYPE_FLAG_MSBYTE_FIRST | CIV_DTYPE_FLAG_PERCENT255,
         // |--> IC-7300: "Send/read IF signal output level to ACC/USB" 0x1A 0x05 #0063 . A7292-4EX-11 page 163.
         // '--> Any other Icom radio uses a CRAZY DIFFERENT MODEL-DEPENDENT sub-subcode !
         //      For all those OTHER Icom-radios, the sub-subcode can be retrieved
         //      via RigCtrl_GetCrazyModelDependingSubSubcodeForCmd1A_05(). Nomen est omen.
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_MOD_INPUT_LEVEL_FROM_ACC, "ModInputLevelFromACC",  "%",      NULL,
      offsetof( T_RigCtrlInstance, iModInputLevelFromACC), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x1A\x05\x00\x64"/*IC7300*/, 4, CIV_DTYPE_BCD4 | CIV_DTYPE_FLAG_MSBYTE_FIRST | CIV_DTYPE_FLAG_PERCENT255,
         // |--> IC-7300: "Send/read MOD input level from ACC" 0x1A 0x05 #0064 . A7292-4EX-11 page 163.
         // '--> Any other Icom radio uses a CRAZY DIFFERENT MODEL-DEPENDENT sub-subcode !
         //      For all those OTHER Icom-radios, the sub-subcode can be retrieved
         //      via RigCtrl_GetCrazyModelDependingSubSubcodeForCmd1A_05(). Nomen est omen.
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_MOD_INPUT_LEVEL_FROM_USB, "ModInputLevelFromUSB",  "%",      NULL,
      offsetof( T_RigCtrlInstance, iModInputLevelFromUSB), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x1A\x05\x00\x65"/*IC7300*/, 4, CIV_DTYPE_BCD4 | CIV_DTYPE_FLAG_MSBYTE_FIRST | CIV_DTYPE_FLAG_PERCENT255,
         // |--> IC-7300: "Send/read MOD input level from USB" 0x1A 0x05 #0065 . A7292-4EX-11 page 163.
         // '--> Any other Icom radio uses a CRAZY DIFFERENT MODEL-DEPENDENT sub-subcode !
         //      For all those OTHER Icom-radios, the sub-subcode can be retrieved
         //      via RigCtrl_GetCrazyModelDependingSubSubcodeForCmd1A_05(). Nomen est omen.
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_MOD_IN_CONN_FOR_DATA_OFF, "ModInputConnector_NoData","",       NULL,
      offsetof( T_RigCtrlInstance, iModInputConnector_NoData), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x1A\x05\x00\x66"/*IC7300*/, 4, CIV_DTYPE_BCD2/*guesswork, as often*/,
         // |--> IC-7300: "Send/read MOD input connector during DATA OFF" 0x1A 0x05 #0066 . A7292-4EX-11 page 163.
         // |  >          ( 00=MIC, 01=ACC, 02=MIC/ACC, 03=USB, 04=MIC/USB )
         // '--> Any other Icom radio uses a CRAZY DIFFERENT MODEL-DEPENDENT sub-subcode !
         //      For all those OTHER Icom-radios, the sub-subcode can be retrieved
         //      via RigCtrl_GetCrazyModelDependingSubSubcodeForCmd1A_05(). Nomen est omen.
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND | RIGCTRL_MSGTYPE_FLAG_PERIODIC_POLL
 },
 { RIGCTRL_PN_MOD_IN_CONN_FOR_DATA, "ModInputConnector_Data",      "",       NULL,
      offsetof( T_RigCtrlInstance, iModInputConnector_Data), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x1A\x05\x00\x67"/*IC7300*/, 4,  CIV_DTYPE_BCD2/*guesswork; alway TWO digits*/,
         // |--> IC-7300: "Send/read MOD input connector during DATA" 0x1A 0x05 #0067 . A7292-4EX-11 page 163.
         // |  >          ( 00=MIC, 01=ACC, 02=MIC/ACC, 03=USB, 04=MIC/USB )
         // '--> Any other Icom radio uses a CRAZY DIFFERENT MODEL-DEPENDENT sub-subcode !
         //      For all those OTHER Icom-radios, the sub-subcode can be retrieved
         //      via RigCtrl_GetCrazyModelDependingSubSubcodeForCmd1A_05(). Nomen est omen.
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND | RIGCTRL_MSGTYPE_FLAG_PERIODIC_POLL
      // WSJT-X periodically polled this parameter, so usually exclude this from the log
 },
 { RIGCTRL_PN_CIV_TRANSCEIVE,      "CIV_transceive",      "",       NULL,
      offsetof( T_RigCtrlInstance, iCIV_transceive), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x1A\x05\x00\x71"/*IC7300*/, 4,  CIV_DTYPE_BCD2/*guesswork; a TWO-digit value*/,
         // |--> IC-7300: "Send/read the CI-V transceive setting" 0x1A 0x05 #0071 . A7292-4EX-11 page 163.
         // |  >          ( funny name for allowing UNSOLICITED REPORTS or not )
         // '--> Any other Icom radio uses a CRAZY DIFFERENT MODEL-DEPENDENT sub-subcode !
         //      For all those OTHER Icom-radios, the sub-subcode can be retrieved
         //      via RigCtrl_GetCrazyModelDependingSubSubcodeForCmd1A_05(). Nomen est omen.
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_CIV_ADDR_USB_TO_REMOTE, "CIV_addr_USB_Rem", "",       NULL,
      offsetof( T_RigCtrlInstance,    iCIV_addr_USB_Rem), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x1A\x05\x00\x72"/*IC7300*/, 4,  CIV_DTYPE_BCD4/*guesswork; a FOUR-digit value*/,
         // |--> IC-7300: "Send/read the transceive CI-V Address for
         // |  >          USB to REMOTE in hexadecimal code" 0x1A 0x05 #0072 . A7292-4EX-11 page 163.
         // |  >          ( possible values are "0000" to "0233"; obviously BCD4 data ).
         // |  WB : Note the limitation of Icom's "CI-V USB Port" to "[REMOTE]" link:
         // |       When ENABLED ("Link to [REMOTE]"), some sources say that
         // |       > In this case, the USB port communication speed is limited to 19,200
         // |       > bps because the [REMOTE] jacks maximum communication speed is 19,200 bps.
         // |  That's why almost any "remote control app for a MODERN Icom radio" says:
         // |       > IMPORTANT: Set to "Unlink to [REMOTE]" when using the
         // |       > spectrum scope function of RS-BA1 with the IC-7300,
         // |       > or IC-7851 through a USB cable. Otherwise RS-BA1 spectrum scope
         // |       > will not function since the data transfer speed of the USB port
         // |       > is insufficient for the amount of data required by the spectrum scope.
         // |  Short form: We will rarely use this "USB-to-[REMOTE]"-stuff,
         // |              including the "CI-V Address for USB to REMOTE" !
         // '--> Any other Icom radio uses a CRAZY DIFFERENT MODEL-DEPENDENT sub-subcode..
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_CIV_OUTPUT_FOR_ANT,  "CIV_for_ANT",         "",       NULL,
      offsetof( T_RigCtrlInstance, iCIV_for_ANT), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x1A\x05\x00\x73"/*IC7300*/, 4,  CIV_DTYPE_BCD2/*guesswork; a TWO-digit value*/,
         // |--> IC-7300: "Send/read the CI-V Output (for ANT) capability" 0x1A 0x05 #0073 . A7292-4EX-11 page 163.
         //     ( possible values are "00" for OFF and "01" for ON, so this is not an ADDRESS )
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_CIV_UNLINK_FROM_REMOTE,"CIV_unlink_Rem",    "",       NULL,
      offsetof( T_RigCtrlInstance, iCIV_unlink_Remote), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x1A\x05\x00\x74"/*IC7300*/, 4,  CIV_DTYPE_BCD2/*guesswork; a TWO-digit value*/,
         // |--> IC-7300: "Send/read the CI-V USB port setting" 0x1A 0x05 #0074 . A7292-4EX-11 page 163.
         //     ( possible values are "00" for "Link to [REMOTE]"
         //                       and "01" for "Unlink to(from?) [REMOTE]" )
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_CIV_USB_ECHO,        "CIV_USB_echo",        "",       NULL,
      offsetof( T_RigCtrlInstance, iCIV_USB_echo), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x1A\x05\x00\x75"/*IC7300*/, 4,  CIV_DTYPE_BCD2/*guesswork; a TWO-digit value*/,
         // |--> IC-7300: "Send/read echo back setting for CI-V operation from USB" 0x1A 0x05 #0075 . A7292-4EX-11 page 163.
         //     ( leave this off because USB is not a BUS but a point-to-point connection,
         //       and "echoing back" only wastes a lot of bandwidth better used for SPECTRUM DATA)
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_CIV_PTT_CONTROL_LINE,  "PTT_Control_Line",        "",       NULL,
      offsetof( T_RigCtrlInstance, iCW_PTTControlLineOnUSB), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x1A\x05\x00\x78"/*IC7300*/, 4,  CIV_DTYPE_BCD2,
         // |--> IC-7300 (from A7292-4EX-11 page 163 ) : 0x1A 0x05 #0078 .
         // |  > Send/read transmission control line setting for USB. (00=OFF, 01= DTR, 02=RTS)
         // |  > Different line must be set from both CW keying and RTTY (FSK)
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_CIV_CW_KEYING_LINE,  "CW_Keying_Line",        "",       NULL,
      offsetof( T_RigCtrlInstance, iCW_KeyingLineOnUSB), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x1A\x05\x00\x79"/*IC7300*/, 4,  CIV_DTYPE_BCD2,
         // |--> IC-7300 (from A7292-4EX-11 page 163 ) : 0x1A 0x05 #0079 .
         // |  > Send/read CW keying line setting for USB. (00=OFF, 01= DTR, 02=RTS)
         // |  > Different line must be set from both transmission control and RTTY (FSK)
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_CIV_RTTY_KEYING_LINE,  "RTTY_Keying_Line",    "",       NULL,
      offsetof( T_RigCtrlInstance, iRTTY_KeyingLineOnUSB), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x1A\x05\x00\x80"/*IC7300*/, 4,  CIV_DTYPE_BCD2,
         // |--> IC-7300 (from A7292-4EX-11 page 163 ) : 0x1A 0x05 #0080 .
         // |  > Send/read RTTY (FSK) line setting for USB. (00=OFF, 01= DTR, 02=RTS)
         // |  > Different line must be set from both CW keying and transmission control"
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },


 { RIGCTRL_PN_IP_PLUS,             "IP_Plus",             "",       NULL,
      offsetof( T_RigCtrlInstance, iIP_Plus),      RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)NULL,           0,                     CIV_DTYPE_NONE,
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_NTP_SERVER_ACCESS,   "NTP_access",          "",       NULL,
      offsetof( T_RigCtrlInstance, iNTP_Server_Access), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)NULL,           0,                     CIV_DTYPE_NONE,
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_NTP_ACCESS_RESULT,   "NTP_result",          "",       NULL,
      offsetof( T_RigCtrlInstance, iNTP_Access_Result), RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)NULL,           0,                     CIV_DTYPE_NONE,
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_AF_MUTE,             "AF_mute",             "",       NULL,
      offsetof( T_RigCtrlInstance, iAF_mute),      RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)NULL,           0,                     CIV_DTYPE_NONE,
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },

 //-- Even more settings (in CI-V, controlled via command 0x0E..0x10) --
 { RIGCTRL_PN_SCAN_MODE,           "ScanMode",            "",       NULL,
      offsetof( T_RigCtrlInstance, iScanMode),     RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE ,
      (const BYTE*)NULL,           0,                     CIV_DTYPE_NONE,
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_SPLIT_MODE,          "SplitMode",           "",       RigCtrl_SplitModes,
      offsetof( T_RigCtrlInstance, iSplitMode),    RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x0F",         1,                     CIV_DTYPE_BCD2,
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_TUNING_STEP,         "TuningStep",          "Hz",     NULL,
      offsetof( T_RigCtrlInstance, iTuningStep_Hz),RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x10",         1,                     CIV_DTYPE_BCD2,
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_ATTENUATOR_DB,       "Attenuator",          "dB",     NULL,
      offsetof( T_RigCtrlInstance, iAttenuator_dB),RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)NULL,           0,                     CIV_DTYPE_BCD2,
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },


 { RIGCTRL_PN_RIT_FREQ, "RIT_freq",  "Hz", NULL, // 'Receiver Incremental Tuning' in Hertz. IC-7300: Cmd 0x21, Sub 0x00
      offsetof( T_RigCtrlInstance, iRIT_freq_Hz),  RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x21\x00",     2, CIV_DTYPE_BCD4 | CIV_DTYPE_SUFFIX_BCD2_SIGN /*LSByte first*/,
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_XIT_FREQ, "XIT_freq",  "Hz", NULL, // 'Transmitter Incremental Tuning' in Hertz. IC-7300: no extra VALUE for "delta TX" (same variable as the "RIT frequency in Hertz") !
      offsetof( T_RigCtrlInstance, iXIT_freq_Hz),  RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)NULL/*no such param in Icom transceivers!*/,  0, CIV_DTYPE_NONE/*no such param!!*/,
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_RIT_ENABLED, "RIT_ON",  "", NULL,  // 'Receiver Incremental Tuning' ON/OFF. IC-7300: Cmd 0x21, Sub 0x01
      offsetof( T_RigCtrlInstance, iRIT_ON ),  RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x21\x01",     2,           CIV_DTYPE_BCD2,
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },
 { RIGCTRL_PN_XIT_ENABLED, "XIT_ON",  "", NULL,  // 'Transmitter Incremental Tuning' ON/OFF. IC-7300: Cmd 0x21, Sub 0x02.
                                           // IC-9700: No such SUB-COMMAND for Cmd 0x21 ! !
      offsetof( T_RigCtrlInstance, iXIT_ON ),  RIGCTRL_DT_INT, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x21\x02",     2,                     CIV_DTYPE_BCD2,
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },

 // Access to "Band Stacking Registers", memory channels, and similar:
 { RIGCTRL_PN_BAND_STACKING_REGS, "BandStack",  "", NULL,  // 'Send/read band stacking register contents'
      offsetof( T_RigCtrlInstance, BandStackingRegs),  RIGCTRL_DT_FREQ_MEM, RIGCTRL_AF_READWRITE,
      (const BYTE*)"\x1A\x01",     2,                     CIV_DTYPE_NONE,
      RIGCTRL_MSGTYPE_ANY_KNOWN_COMMAND
 },


  // Note: If the same 'unified parameter number' appears TWICE in this table,
  //       loops calling RigCtrl_EnumerateUnifiedParameters() will crash.
  //       Or the VCL will crash because the stupid TRichEdit gets overloaded.
  //   This really happened in TKeyerMainForm::MI_ReportRigCtrlParamsClick() !

 // As usual, the end of the table is marked by an entry with "all zeroes" :
  // iUnifiedPN,                   pszToken,              pszUnit,  pStringValueTable,
  //  iByteOffsetIntoRigCtrlInstance  , iRigCtrlDataType, iAccessFlags,
  //       pbCIVReadCmd,           CIVReadCmdLength,        iCIVDataType :
 {   0,                             NULL,                 NULL,     NULL,
       0,                               0,                0,
       (const BYTE*)NULL,          0,                       0 }

}; // end RigCtrl_ParameterInfo[]


/* EOF < C:\cbproj\Remote_CW_Keyer\RigControl.c > */


