//---------------------------------------------------------------------------
/* File:  C:\cbproj\Remote_CW_Keyer\OpenWebRX_Server.h                     */
/* Date:  2019-01-19                                                       */
/* Author: Wolfgang Buescher (DL4YHF)                                      */
/* Purpose:  Special addition for Spectrum Lab's HTTP server, not sure yet */
/*           if this will ever evolve into a 'functional' server for the   */
/*           OpenWebRX 'htdocs' files found in the official distribution.  */
/*           Unlike the original OpenWebRX server, THIS ONE is not written */
/*           in Python but "good old C" .                                  */
/*                                                                         */
/* Revision history :                                                      */
/*   2024 : Dug out the OpenWebRX-based "good old C" sources again,        */
/*          and adapted them for the HTTP server in the 'Remote CW Keyer'. */
/*   2019 : Experiments with Spectrum Lab's web server and OpenWebRX .     */
//---------------------------------------------------------------------------

#ifndef _OPENWEBRX_SERVER_H
#define _OPENWEBRX_SERVER_H

#include "switches.h"


#include "StringLib.h"  // DL4YHF's string library, with stuff like SL_ParseIPv4Address()
#include "Utilities.h"  // stuff like UTL_iWindowsVersion, UTL_iAppInstance, ShowError(), etc
#include "Timers.h"     // high-resolution 'current time'-function, T_TIM_Stopwatch, etc.
#include "Elbug.h"      // the basic 'Elbug' functions (plain C)
#include "SampleRateConv.h"  // convert from 48 to 8 kSamples/sec and back,
                        // for 'audio streaming' between server and client[s].
#include "ALawCompression.h" // simple A-Law compression / decompression
                        // to stream audio between server and client[s].
#include "CwDSP.h"      // 'CW Digitial Signal Processor' (audio I/O and processing)
#include "CwNet.h"      // "CW Network" (not HTTP-based main part)
#include "Inet_Tools.h" // Base64 encoding/decoding, SHA1 calculation, etc..
#include "HttpServer.h" // simplistic HTTP server, used by CwNet.c, too


enum // OpenWebRX-specific tokens. Seen, for example, in index.wrx .
{    // In this "C" implementation, also used as parameter-index
     // when our server asks the "hosting application" (e.g. Spectrum Lab)
     // for meaningful default values - see treatment of keywords in the
     // template file ("index.wrx") in OWRX_OnWebRXTemplateRequest() !
  // ex: WRX_TOKEN_CLIENT_ID = 1, // removed; here HttpServer.c cares for 'session management'
  WRX_TOKEN_WS_URL = 1,
  WRX_TOKEN_RX_TITLE,
  WRX_TOKEN_RX_LOCATION,
  WRX_TOKEN_RX_LOCATOR,
  WRX_TOKEN_RX_HEIGHT_ASL,
  WRX_TOKEN_RX_GPS,
  WRX_TOKEN_RX_PHOTO_TITLE,
  WRX_TOKEN_RX_PHOTO_DESC,
  WRX_TOKEN_RX_PHOTO_HEIGHT,
  WRX_TOKEN_AUDIO_BUFSIZE,
  WRX_TOKEN_MODULATION,        // iModulation (here, no difference between "initial modulation" and "modulation", etc pp)
  WRX_TOKEN_VFO_FREQ,          // i32VFOFreq_Hz
  WRX_TOKEN_FILTER_BW,         // iFilterBW_Hz
  WRX_TOKEN_AUDIO_OFFSET,      // iAudioOffset_Hz
  WRX_TOKEN_SPECTRUM_FCENTER,  // i32SpectrumCenterFreq
  WRX_TOKEN_SPECTRUM_BANDWIDTH,// i32SpectrumBandwidth
  WRX_TOKEN_SPECTRUM_NUM_BINS, // iSpectrumNumFreqBins
  WRX_TOKEN_WATERFALL_COLORS,
  WRX_TOKEN_WATERFALL_MIN_LEVEL,
  WRX_TOKEN_WATERFALL_MAX_LEVEL,
  WRX_TOKEN_WATERFALL_AUTO_LEVEL_MARGIN,
  WRX_TOKEN_DIGIMODES_ENABLE
};



//--------------- Data Types ------------------------------------------------

typedef struct t_OWRX_Client
{
  DWORD dwClientIP;        // four-byte CLIENT IP (IPv4 address) of the current session
  int   iClientIndex;      // this client's array index (into OWRX_Client[], for the log)
  int   iRxChannel;        // receiver channel from the host's point of view, 0 = "default"
  char  sz32ClientID[32+4]; // 32-character CLIENT ID, unique throughout an OpenWebRX session
  double dblUnixGenTime;   // "gen_time", but here measured in Unix seconds
  int    iState;           // one of the following:
#        define OWRX_STATE_PASSIVE        0
#        define OWRX_OPENED_ROOT_DOCUMENT 1  // e.g. loading the expanded "index.wrx"
#        define OWRX_CLOSED_ROOT_DOCUMENT 2  // loaded "index.wrx" (and some others) but didn't open a WebSocket yet
#        define OWRX_OPENED_WEB_SOCKET    3  // opened the web socket but not ready for audio+spectra yet
#        define OWRX_ACTIVE_WEB_SOCKET    4  // web socket ready for audio and spectra (FFTs)
#        define OWRX_CLOSED_WEB_SOCKET    5  // web socket close, but keep the instance alive for a short time
         // When adding new states, also 'support' them in OWRX_StateToString() !
  int  iSpectrumBufferTail;   // FIFO tail index for spectra (delivered by RigControl.c)
  long i32SpectrumCenterFreq; // client's waterfall center frequency, initialized with SL's when client connects
  long i32SpectrumBandwidth;  // client's waterfall bandwidth (display span in Hertz, often limited to 4096)
  int  iSpectrumNumFreqBins;  // client's waterfall number of frequency bins (same as SL's ?)
         // As explained in SL's WebServer_UpdateVFOFreqInRemoteClients(),
         // if the spectrum is actually an IC-7300 / IC-9700's 'Spectrum Scope',
         // the waterfall display in the remote client must be re-initialized
         // when toggling between 'CENTER FREQUENCY' and 'FIX' mode on the Icom,
         // or when selecting another 'EDGE' (funny name for the fixed-frequency range).
         // The following member keeps track of the 'Center'/'Fixed' mode:
  BOOL fFixedEdgeMode;   // TRUE:  Icom's Spectrum Scope in FIXED EDGE mode,
                         // FALSE: Icom's Spectrum Scope centered on VFO-freq.
  long  i32VFOFreq_Hz; // originally exchanged with the Javascript as "offset_freq",
       // which was added to the *spectrum* center frequency. No, thanks.
       // Now we simply let the web interface know our radio's "VFO frequency"
       // aka "dial frequency", and stop messing around with "center_freq"
       // and "offset_freq", which didn't give a correct display in CW anyway.
       // The Javascript (openwebrx.js) will now SIMPLY show whatever our
       // IC-7300 or IC-9700 shows on its display, too .
       // (search for the assignment to 'e("webrx-actual-freq")' in openwebrx.js ).
  char sz80BcastMsg[84]; // no idea what this is for.. will find out one day
  BOOL fClosed;          // > not exactly sure if required (WB: indeed)
  int  iDSP;             // dummy for "dsp = None" (we use the radio's, or Spectrum Lab's)
  // ex: T_ADPCM_Codec sAudioCodec; // ADPCM encoder / decoder for sending AUDIO
  // (in this simplified variant, audio for each remote client is delivered
  //  by CwDSP_GetNumSamplesInFifoForTailIndex(); followed by A-Law compression
  //  via ALawCompression.c : LinearToALawSample() . Also, the sampling rate
  //  is FIXED to 8 kHz. No "audio codec" required.)
  int  iRequestedAction; // from a parsed 'SET' command, one of the following:
#        define OWRX_ACTION_NONE   0
#        define OWRX_ACTION_START  1
#        define OWRX_ACTION_STOP   2
  int  iLowCutoff_Hz;   // lower audio passband cutoff frequency in Hertz, e.g. 300 for SSB
  int  iHighCutoff_Hz;  // upper audio passband cutoff frequency in Hertz, e.g. 3000 for SSB
  int  iAudioOffset_Hz; // aka "CW pitch" (Icom slang), e.g. 650 Hz for CW, 0 for SSB/AM/FM
       // "SPECIFICATION" of iAudioOffset_Hz (don't move or remove, you'll fry in hell):
       // The 'audio offset' (aka CW pitch frequency) is preferably + or -650 Hz,
       //      depending on the receiver receiving CW in "LSB" or "LSB".
       //      (with Icom radios, that info originates from RigControl.c) .
       // An IC-7300 in "CW" actually uses LSB: A signal ABOVE the
       //        "dial frequency" (*) gave an audio note BELOW abs(iAudioOffset_Hz) !
       //        The "zero beat" marker in the yellow passband indicator (vertical line)
       //        must be painted 650 Hz *BELOW* the CW carrier frequency .
       //                     ______
       //                   _/      \_     |  "CW" a la IC-7300 .... is LSB !
       //                    |      |      |
       //                    |      '------------ iHighCutoff_Hz  =  900
       //                    '------------------- iLowCutoff_Hz   =  400
       //                                  '------ iAudioOffset_Hz = +650
       //                        |<- 650 ->|
       //                    CW Carrier    |
       //              & ideal VFO "Dial"  |
       //                      frequency.  |
       //                                  |
       //                                  0 Hz "Audio" frequency,
       //                                        aka zero-beat.
       //
       // An IC-7300 in "CW-R" uses USB internally: A signal above the
       //        "dial frequency" (*) gave an audio note above iAudioOffset_Hz .
       //        The "zero beat" marker in the yellow passband indicator (vertical line)
       //        must be painted 650 Hz *ABOVE* the audio passband center.
       //                     ______
       //              |    _/      \_    "CW-R" a la IC-7300
       //              |     |      |
       //              |     |      '------------ iHighCutoff_Hz  =  900
       //              |     '------------------- iLowCutoff_Hz   =  400
       //              '-- "zero beat" marker --- iAudioOffset_Hz = +650
       //              |<- 650 ->|
       //              |         CW Carrier & ideal VFO "Dial" frequency.
       //              |
       //              0 Hz "Audio" frequency, aka zero-beat.
       //
       // For "USB-CW" / Icom "CW",   iAudioOffset_Hz is POSITIVE (typ. +650 Hz).
       // For "LSB-CW" / Icom "CW-R", iAudioOffset_Hz is NEGATIVE (typ. -650 Hz).
       // For "USB" & "LSB" (voice),  iAudioOffset_Hz must be ZERO .
  int  iRigOpMode; // not sure if we'll EVER support this..

  int  iSMeterLevel_dB;  // S-meter value in "dB above S0"
  BOOL fSendSMeter;      // flag to send iSMeterLevel_dB 'when time permits'

  BOOL fPassbandOrVFOModifiedByClient; // flag set if the audio filter parameters,
       // or the "mode" (CW/LSB/USB/etc) or the radio's "VFO frequency"
       // aka "dial frequency" was modified BY THE CLIENT.
       // Cleared after the server itself has 'reprogrammed' the radio,
       // or the DSP process, according to the new settings.
  BOOL fPassbandOrVFOModifiedByHost;  // flag set is the above params
       // have been modified by the "host" (either Spectrum Lab or the radio
       // controlled via CI-V, e.g. IC-7300 or IC-9700)
  BOOL fSpectrumDisplayModifiedByHost; // flag set if any of the following must be modified IN THE CLIENT:
       // i32SpectrumCenterFreq, i32SpectrumBandwidth, iSpectrumNumFreqBins.
       //  Cleared after the server itself has 'sent' those params to the client.
  T_TIM_Stopwatch stopwatch_for_periodic_msgs_s2c; // for periodic messages
       //  from server to client, e.g. "cpu_usage" and "clients" .
  T_TIM_Stopwatch stopwatch_for_timeouts; // to monitor timeouts, for example
       //  between loading the root document (expanded "index.wrx")
       //  and opening the TCP-socket for the Web Socket .
  long nMillisecondsWaited;  // number of milliseconds, ACCUMULATED because
       // a T_TIM_Stopwatch possibly rolls over after a few ten seconds.
  int  nMinutesWithoutUserEvents; // <- to throw out clients that obviously fell asleep
  T_HttpInstance *pHttpInstance;  // connection to the basic HTTP server.
  T_CwNet           *pCwNet;      // connection to the "CW Network" instance
  T_RigCtrlInstance *pRigControl; // serial or USB link to the "local radio", e.g. IC-7300



  struct t_OWRX_Client *pNext; // pointer to the NEXT client, for a chained list

} T_OWRX_Client;

#define OWRX_MAX_CLIENTS 10 // <- max. number of SIMULTANEOUSLY connected clients.
        // not sure how many are possible, but beware of
        // the "caller IDs" CLI_CALLER_REMOTE_CLIENT_FIRST .. CLI_CALLER_REMOTE_CLIENT_LAST
        // defined in c:\cbproj\SpecLab\Cli_type.h !
extern T_OWRX_Client OWRX_Client[OWRX_MAX_CLIENTS];
extern int OWRX_Server_nClientsConnected; // <- number of clients CURRENTLY connected
extern int OWRX_Server_iCpuUsagePercent;  // <- for the "Server CPU" usage display

#define OWRX_MAX_CLIENT_INFO_ENTRIES 100 // number of clients that we keep a record for
    // (first based on their IP addresses, later, maybe, on their ham radio callsigns)
typedef struct tOWRX_ClientInfo
{
  DWORD dwClientIP; // Four-byte CLIENT IP (IPv4 address) of the current session.
                    // ZERO marks an unused entry in array OWRX_ClientInfo[] .
  char  sz20UserName[24]; // amateur radio callsigns preferred
  double dblUnixTimeOfFirstVisit;
  double dblUnixTimeOfLastActivity;
} T_OWRX_ClientInfo;
extern T_OWRX_ClientInfo OWRX_ClientInfo[OWRX_MAX_CLIENT_INFO_ENTRIES];


// In this microcontroller-friendly, stripped-down variant, there are no
// USER-DEFINEABLE tokens for what used to be the *.wrx templace, but only
// the hard-coded WRX_TOKEN_xyz-values. There are no 'string buffers' for the
// text 'generated' when expanding e.g. "%[RX_TITLE]" (WRX_TOKEN_RX_TITLE),
// "%[MODULATION]" (WRX_TOKEN_MODULATION), "%[VFO_FREQ]" (WRX_TOKEN_VFO_FREQ),
// etc. The following look-up table converts the keywords into integer tokens:
extern const T_SL_TokenList OpenWebRX_Keywords[];
// Depending on the integer token (listed in OpenWebRX_Keywords[],
// OWRX_AppendTextForToken() may a append a few characters for those tokens
// whenever HttpSrv_GenerateHTMLFromTemplate() enconters them.



//--------------- Prototypes ------------------------------------------------
  // "Declaration syntax error" ?
  //   -> look for missing semicolon in prototypeS in all previous headers

#ifndef CPROT  // For peaceful co-existence of "C" and "C++" ....
 // Note on "extern": do not trust the lousy example from Borland !
 //   >extern "c" = wrong    (not recognized by Borland C++ V4.0)
 //   >extern "C" = correct  (though the help system says something different)
# ifdef __cplusplus
#   define CPROT extern "C"
# else
#   define CPROT
# endif  // ! "cplusplus"
#endif  // ndef CPROT

CPROT char *OWRX_ModulationToString( int iModulation );
CPROT BOOL OWRX_IsConnected( DWORD dwClientIP ); // checks if a client with the given IP address is *currently* connected
CPROT T_OWRX_ClientInfo *OWRX_GetClientInfoPtr( T_OWRX_Client *pOWClient );
CPROT T_OWRX_Client *OWRX_GetClientByIP( DWORD dwClientIP ); // may return NULL !
CPROT void OWRX_AppendTextForToken(T_OWRX_Client *pOWClient,char **ppszDest,const char *pszEndstop,int iToken);

// ex: CPROT BOOL OWRX_LoadConfig( char *pszFilename );
// ex: CPROT BOOL OWRX_SaveConfig( char *pszFilename );



//---------------------------------------------------------------------------
// Interface to the HTTP server (here: Remote_CW_Keyer/HttpServer.c)
//---------------------------------------------------------------------------

CPROT int  OWRX_OnWebRXTemplateRequest( T_OWRX_Client *pOWClient );
CPROT int  OWRX_OnSessionTimeout( T_OWRX_Client *pOWClient );
CPROT int  OWRX_OnWebSocketOpened( T_OWRX_Client *pOWClient );
CPROT int  OWRX_OnWebSocketFrameRcvd( T_OWRX_Client *pOWClient, int iWSFrameType, long i32PayloadLength, BYTE * pbPayload );
CPROT void OWRX_OnWebSocketPollForTxData( T_OWRX_Client *pOWClient );
CPROT void OWRX_ParseSETMessage( T_OWRX_Client *pOWClient, const char *cpSrc );
CPROT void OWRX_OnConnectionClosed( T_OWRX_Client *pOWClient );


//---------------------------------------------------------------------------
// Interface to the "application" or "host" (here: Spectrum Lab or an IC-7300)
//---------------------------------------------------------------------------

CPROT long OWRX_GetDefaultParam_Int(int iToken); // defaults for waterfall min level, max level, etc.
#           define OWRX_INVALID_INT32_VALUE (-2147483647L/*-(2^31-1)*/)
CPROT void OWRX_CheckForTimeouts(void); // periodically call this from the main task, every second or so



#endif // ndef _OPENWEBRX_SERVER_H ?


