//---------------------------------------------------------------------------
/* File:  C:\cbproj\Remote_CW_Keyer\OpenWebRX_Server.c                     */
/* Date:  2024-02-17                                                       */
/* Author: Wolfgang Buescher (DL4YHF)                                      */
/* Purpose:  OpenWebRx-alike addition for the Remove CW Keyer's web server,*/
/*           but not compatible with OpenWebRX (by HA7ILM) anymore.        */
/*           Unlike the original OpenWebRX server, THIS ONE is not written */
/*           in Python but "plain old C", for easy integration in a later  */
/*           adaption on a microcontroller (using LwIP or similar).        */
/*                                                                         */
/* Revision history: Further below, the "more important" things first !    */
/*                                                                         */
/* 1. Principle of operation (as implemented here)                         */
/* --------------------------------------------                            */
/*  Whenever a remote client asks for the root document ("\"), the server  */
/*  checks the 'server pages' for the OpenWebRX template file (index.wrx). */
/*  Only is that file exists, a remote client's session is considered an   */
/*  OPEN WEB RX session, and this module (OpenWebRX_Server.c) comes into   */
/*  play (see Step 1 further below).                                       */
/*  Most files that the original OpenWebRX client would read, e.g.         */
/*      sdr.js, openwebrx.js, jquery-3.2.1.min.js,                         */
/*      jquery.nanoscroller.js, etc etc, have been eliminated to keep      */
/*      this implementation sufficienly simple for a microcontroller firm- */
/*      ware, WITHOUT A FILE SYSTEM AT ALL.                                */
/*                                                                         */
/*  1.1 Opening and closing an OpenWebRX-alike session                     */
/*  ---------------------------------------------------                    */
/*  Step 1 : Delivery of the "root document" (formerly: index.wrx)         */
/*           via OWRX_OnWebRXTemplateRequest(). We store the visitor's     */
/*           IP-address at that stage, and without this, we will NOT       */
/*           allow opening the web socket (for audio and spectra) later.   */
/*  Step 2 : After a lot of other ("normal") files have been loaded,       */
/*           the tiny Javascript in the expanded "index.wrx" will call     */
/*           openwebrx_init() [implemented in *.js], which then            */
/*           opens the OWRX Web Socket -> here: OWRX_OnWebSocketOpened().  */
/*    The (TCP-)socket for the OWRX Web Socket is THE ONLY connection      */
/*    that remains open until the remote operator shuts down his client.   */
/*  Step 3 : After a remote OWRX client closed the WEB SOCKET,             */
/*           OWRX_OnConnectionClosed() finally marks the OWRX_Client[]     */
/*           as 'unused' again. (Note: When closing the socket in Step 1,  */
/*           the session data in the same OWRX_Client[] must be kept,      */
/*           even if there will be NO CONNECTION (TCP socket) between      */
/*           client and server, until the Javascript "starts working" ! )  */
/*                                                                         */
/*  1.2 Controlling the local receiver from a remote client                */
/*  -------------------------------------------------------                */
/*    In the *original* OpenWebRX implementation, the waterfall range      */
/*    (and thus the VFO "tuning range") were only set ONCE (on init).      */
/*    After that, the CLIENT controlled the SERVER but not vice versa,     */
/*    using the WebSocket frames parsed in OWRX_ParseSETMessage() .        */
/*    Most important : SET "vfo_freq=<Current 'Dial' Frequency>" .         */
/*                  In contrast to the original OpenWebRX, does NOT depend */
/*                  on the currently used spectrum center frequency .      */
/*            Spectrum Lab periodically checks the client's "QSY request"  */
/*            (fModifiedPassbandOrVFO) and if necessary, reprograms        */
/*            its DSP processing chain, or passes the new settings on to   */
/*            a remotely controlled 'real' radio (e.g. IC-7300 or IC-9700).*/
/*            Since our 'real' radio only has ONE audio- and ONE spectrum, */
/*            only ONE of the OpenWebRX clients can actually control the   */
/*            receiver, while others may only listen and watch to the same */
/*            signal, with the same demodulator parameters, etc.           */
/*            Thus the settings for VFO, audio filter, display spectrum    */
/*            may also have to be passed from SERVER to CLIENTS, and all   */
/*            clients must be able to adapt themselves to new settings     */
/*            at any time (not just in the initialisation phase)           */
/*            - see next chapter.                                          */
/*                                                                         */
/*  1.3 Controlling the remote clients from the server / local receiver    */
/*  -------------------------------------------------------------------    */
/*    The functionality described here did not exist in the original code, */
/*    neither client- nor server side. Typical scenario / data flow :      */
/*       1.)  The operator of the local radio (e.g. IC-7300) changes the   */
/*            frequency, or the 'Spectrum Scope' display range. The main   */
/*            application -Spectrum Lab-  updates all clients by calling   */
/*            WebServer_UpdateVFOFreqInRemoteClients() then .              */
/*                                                                         */
/*  1.4 Controlling the mode (CW, USB, LSB) and the audio passband         */
/*  -------------------------------------------------------------------    */
/*    For comparison, study the behaviour of the few remaining OpenWebRX   */
/*    receivers out there (many were replaced by Kiwis these days).        */
/*    See https://www.receiverbook.de/  . The one used for testing was     */
/*        http://hamnet-bielefeld.de:8073/#freq=14406000,mod=cw,sql=-150 . */
/*    1st observation : You do NOT hear anything when transmitting         */
/*                      a 'CW' carrier on 144.060 MHz if the web radio     */
/*                      was tuned to "144.0600 MHz in mode CW". Too bad !  */
/*    OpenWebRX ("the original") showed this :                             */
/*                  (zero beat marker)  _____  passband                    */
/*                                  | _/     \_                            */
/*                                                                         */
/*           .. | | | | | | | | | | | | | | | | | | | | | ..               */
/*              |                   |                   |                  */
/*          144.05 MHz           144.06 MHz           144.07 MHz           */
/*                                  :        __________________________    */
/*                                  :       | 144.0600 MHz   CW        |   */
/*                                  |       | OpenWebRx Receiver Panel |   */
/*                               CW signal  |__________________________|   */
/*                               was HERE !                                */
/*                               (but not audible despite                  */
/*                                matching "dial frequency")               */
/*   The numeric frequency display on the 'Receiver' panel actually showed */
/*   the frequency of where you WOULD HEAR SOMETHING WITH ZERO HERTZ AUDIO */
/*   (sometimes called 'zero beat', but 'zero beat' is misleading,         */
/*    in CW 'zero beat' often means your SIDETONE = "CW pitch" is exactly  */
/*    the same audio frequency as you hear in the headphone, e.g. 650 Hz). */
/*   In DL4YHF's opinion, a2nd observation : In CW (!), the yellow         */
/*   passband indicator, AND the indicated frequency, were is plain wrong. */
/*   This is what the display SHOULD look like with "upper side band CW":  */
/*                                                                         */
/*                                  ,--------- THIS is what we listen to ! */
/*                                 \|/                                     */
/*            (zero beat marker)  _____  passband                          */
/*                            | _/     \_                                  */
/*                                                                         */
/*           .. | | | | | | | | | | | | | | | | | | | | | ..               */
/*              |                   |                   |                  */
/*          144.05 MHz       /|\  144.06 MHz           144.07 MHz          */
/*                            |     :        __________________________    */
/*                            |     :       | 144.0600 MHz   CW        |   */
/*                            |     |       | OpenWebRx Receiver Panel |   */
/*                            |  CW signal  |__________________________|   */
/*                            |  was HERE                                  */
/*                            |  (and how audible, with                    */
/*                            |   matching "dial frequency")               */
/*                            |                                            */
/*                            this would be our tuning frequency if we     */
/*                            were using USB, but we're using CW. Nnngrr.  */
/* Note: In contrast to OpenWebRX, PA3FWM's WebSDR (e.g. the famous one    */
/*       at http://websdr.ewi.utwente.nl:8901/ ) got this perfectly right: */
/* When TUNED to 3550.00 kHz *IN CW*, it SHOWS you're tuned to 3550.0 kHz, */
/* and if you then TRANSMIT on 3550.0 kHz (in CW, what else), you HEAR it. */
/* It also indicates the "zero Hertz audio" point as a vertical line, and  */
/* you could even tell if the 'demodulator' uses CW in USB or LSB . Great. */
/* This is how our remote display SHOULD also look like, when listening to */
/*               3.550 MHz in CW (RX using "lower-side-band-CW") :         */
/*                           _____________                                 */
/*                         _/             \_     |                         */
/*           .. | | | | | | | | | | | | | | | | | | | | | ..               */
/*              |                   |                   |                  */
/*            3.559 MHz          3.560 MHz            3.561 MHz            */
/*                                               :                         */
/*                                  :        __________________________    */
/*                                  :       | 3.5600 MHz   CW          |   */
/*                                  |       | Twente Web SDR           |   */
/*                               "dial"     |__________________________|   */
/*                               frequency !     :                         */
/*                                  :            :                         */
/*                                  |<- 650 Hz ->|                         */
/*                                             3560.650 kHz ("0 Hz audio") */
/*          ( 3560.650 kHz would be displayed by the original OpenWebRX,   */
/*                     but that's the 'QRG' an IC-7300 and MANY others     */
/*                     would show for LSB   -  but NOT for CW !  )         */
/*   Solution (2020-06-22) :                                               */
/*      Radically killed the dependency of the displayed "VFO frequency"   */
/*      from what used to be "center_freq" in the original OpenWebRX.      */
/*      We now tell the client (GUI) what to show as "vfo_freq", BASTA !   */
/*      When sending the IC-7300's "dial" frequency to all clients,        */
/*   There's no more "SET offset_freq"  (too ambiguous and impractical).   */
/*   When the radio QSY'ed for any reason, we "SET vfo_freq" instead .     */
/*   We also "SET audio_offset" (iAudioOffset_Hz), e.g. to + or -650 Hz    */
/*      when switching the IC-7300 between CW, CW-"Reverse", USB, and LSB. */
/*   We also "SET low_cut=<iLowCutoff_Hz>", "SET high_cut=<iLowCutoff_Hz>",*/
/*      etc, whenever the IC-7300 operator selects a different filter.     */
/*   Note: The IC-7300's CW filters can be adjusted so that the "CW Pitch" */
/*         (=abs(iAudioOffset_Hz)) isn't necessarily in the CENTER of the  */
/*         audio passband (=(abs(iLowCutoff_Hz)+abs(iHighCutoff_Hz))/2) !  */
/*                                                                         */
/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
/*                                                                         */
/* Revision history :                                                      */
/*   2024 : A stripped-down, simplified variant of what used to be the     */
/*          OpenWebRX-server is now used in DL4YHF's 'Remote CW Keyer',    */
/*             see C:\cbproj\Remote_CW_Keyer\OpenWebRx_Server.c (not cpp.) */
/*   2020 : HA7ILM has stopped development of the original OpenWebRX. The  */
/*          last version is still at https://github.com/ha7ilm/openwebrx   */
/*          tagged "This repository has been archived by the owner.        */
/*                  It is now read-only."  (since 2019-12-29) .            */
/*          Locally saved the repo as downloads/OpenWebRX_vs_KiwiSDR/      */
/*                                 openwebrx_final_version_2020_12_29.zip  */
/*          Since OUR SERVER (=the one integrated in Spectrum Lab) doesn't */
/*          depend on Python (neither Python 2.7 nor any other Python),    */
/*          only the contents of the OpenWebRX "htdocs" folder were copied */
/*          into C:\OpenWebRX\htdocs\, keeping a backup of older versions. */
/*          Added some extensions in C:\OpenWebRX\htdocs\openwebrx.js ,    */
/*          to be able to "push" new settings from an IC-7300 (etc) to all */
/*          clients that -like it or not- got to listen to the SAME AUDIO. */
/*          (With an IC-7300 or IC-9700 we can 'play Web SDR' but they are */
/*           no multi-channel Kiwi SDRs, and they don't even have a broad- */
/*           band digital I/Q output unlike SDR-IQ, Perseus, and Co. )     */
/*   2019 : Initial experiments with SpecLab's web server and OpenWebRX .  */
//---------------------------------------------------------------------------


#include "switches.h" // project specific 'compilation switches' like SWI_USE_DSOUND

#ifndef  SWI_USE_HTTP_SERVER   // even if HttpServer.c shall NOT be used in the actual build,
# define SWI_USE_HTTP_SERVER 0 // there's no need to remove this "optional" module from the project..
#endif // ndef SWI_USE_HTTP_SERVER  // .. the presence shall be defined in "switches.h"
#if(SWI_USE_HTTP_SERVER)       // this conditional ends near end-of-file ..

#include <string.h> // not only string functions in here, but also memset()
#include <stdio.h>  // no standard I/O but string functions like sprintf()
#ifndef _WINDOWS    // obviously compiling for a PC, so ..
# include "winsock2.h"  // .. use Microsoft's flavour of the Berkeley Socket API
  // Note: never include winsock2.h *after* windows.h .. you'll fry in hell,
  //       because "windows.h" #includes "winsock.h", not "winsock2.h" .
#endif

#include "OpenWebRX_Server.h" // header for THIS module


// Constants, enums, keyword tables, variables, ...

#define Make3CharToken(c1,c2,c3) (((DWORD)c1)|((DWORD)c2<<8)|((DWORD)c3<<16))

const T_SL_TokenList OpenWebRX_Keywords[] = // OpenWebRX-specific keywords, formerly used in *.wrx templates
{ // The original tokens were too confusing; "RX_LOC" was usused for the CITY,
  // and what was displayed after "Loc:" was read from %[RX_QRA]. Baaah.
  // "QRA" should have been the Station Name or Callsign - a "QRA" has got
  // NOTHING to do with Maidenhead locators.
  // Removed the Q-group abuse by replacing stuff like "RX_QRA" by "RX_LOCATOR",
  // and "RX_LOC" by "RX_LOCATION" (often used to show CITY and COUNTRY), etc.
  // removed: { "CLIENT_ID",        WRX_TOKEN_CLIENT_ID                },
  { "WS_URL",                       WRX_TOKEN_WS_URL                   },
  { "RX_TITLE",                     WRX_TOKEN_RX_TITLE                 },
  { "RX_LOCATION",                  WRX_TOKEN_RX_LOCATION              },
  { "RX_LOCATOR",                   WRX_TOKEN_RX_LOCATOR               },
  { "RX_HEIGHT_ASL",                WRX_TOKEN_RX_HEIGHT_ASL            },
  { "RX_GPS",                       WRX_TOKEN_RX_GPS                   },
  { "RX_PHOTO_TITLE",               WRX_TOKEN_RX_PHOTO_TITLE           },
  { "RX_PHOTO_DESC",                WRX_TOKEN_RX_PHOTO_DESC            },
  { "RX_PHOTO_HEIGHT",              WRX_TOKEN_RX_PHOTO_HEIGHT          },
  { "AUDIO_BUFSIZE",                WRX_TOKEN_AUDIO_BUFSIZE            },
  { "MODULATION",                   WRX_TOKEN_MODULATION               },
  { "VFO_FREQ",                     WRX_TOKEN_VFO_FREQ                 },
  { "FILTER_BW",                    WRX_TOKEN_FILTER_BW                },
  { "AUDIO_OFFSET",                 WRX_TOKEN_AUDIO_OFFSET             },
  { "SPECTRUM_FCENTER",             WRX_TOKEN_SPECTRUM_FCENTER         },
  { "SPECTRUM_BANDWIDTH",           WRX_TOKEN_SPECTRUM_BANDWIDTH       },
  { "SPECTRUM_NUM_BINS",            WRX_TOKEN_SPECTRUM_NUM_BINS        },
  { "WATERFALL_COLORS",             WRX_TOKEN_WATERFALL_COLORS         },
  { "WATERFALL_MIN_LEVEL",          WRX_TOKEN_WATERFALL_MIN_LEVEL      },
  { "WATERFALL_MAX_LEVEL",          WRX_TOKEN_WATERFALL_MAX_LEVEL      },
  { "WATERFALL_AUTO_LEVEL_MARGIN",  WRX_TOKEN_WATERFALL_AUTO_LEVEL_MARGIN },
  { "DIGIMODES_ENABLE",             WRX_TOKEN_DIGIMODES_ENABLE         },
    // Completely removed: "mathbox". No more space-hogging "3D waterfall" .

  { NULL, 0 } // marker for end-of-list
}; // end OpenWebRX_Keywords[]

const T_SL_TokenList OpenWebRX_ModulationsAkaOpModes[] =
{
  { "n/a",     RIGCTRL_OPMODE_UNKNOWN },
  { "CW",      RIGCTRL_OPMODE_CW      },
  { "LSB",     RIGCTRL_OPMODE_LSB     }, // note: there's no "SSB" in this table :)
  { "USB",     RIGCTRL_OPMODE_USB     },
  { "AM",      RIGCTRL_OPMODE_AM      },
  { "FM",      RIGCTRL_OPMODE_FM      },
  { "WFM",     RIGCTRL_OPMODE_FM_WIDE },
  { "RTTY",    RIGCTRL_OPMODE_RTTY    },
  { "PSK",     RIGCTRL_OPMODE_PSK     },  // you will NEVER find "FT8" here :o)
  { "CWN",     RIGCTRL_OPMODE_CW | RIGCTRL_OPMODE_NARROW },
  { "NFM",     RIGCTRL_OPMODE_FM | RIGCTRL_OPMODE_NARROW },

  { NULL, 0 } // marker for end-of-list
}; // end OpenWebRX_ModulationsAkaOpModes[]


//---------------------------------------------------------------------------
// Variables (mostly INTERNAL)
//---------------------------------------------------------------------------

T_OWRX_Client OWRX_Client[OWRX_MAX_CLIENTS];
int OWRX_Server_nClientsConnected = 0; // <- important for audio distribution !
    // (if there's not a single client connected, Spectrum Lab will not waste
    //  time to resample audio for OpenWebRX - see TSoundThread::Execute() )
T_OWRX_ClientInfo OWRX_ClientInfo[OWRX_MAX_CLIENT_INFO_ENTRIES] = { 0 };
    // *The application* may load and save OWRX_ClientInfo[] on disk somewhere.
    // That's why OpenWebRX_Server.cpp doesn't clear OWRX_ClientInfo[] anywhere.
    // There's a crude "lister" for OWRX_ClientInfo[] in SL's HttpServerUI.cpp .

static BOOL fInitialized = FALSE;

int OWRX_Server_iCpuUsagePercent = 0;  // <- for the "Server CPU" usage display


// internal forward references / function prototypes only needed HERE, not in *.h
void  OWRX_UpdateNumClientsConnected(void); // OWRX_Client[].iState -> OWRX_Server_nClientsConnected
const char *OWRX_ModulationAkaOpModeToString( int iRigOpMode );


//---------------------------------------------------------------------------
// Functions (implementation)
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
T_OWRX_Client *OWRX_GetClientByIP( DWORD dwClientIP )
   // May return NULL if a client hasn't been registered properly,
   //     e.g. by trying to open a WebSocket without having entered
   //     the root document ( -> OWRX_OnWebRXTemplateRequest() ) .
{
  int i;

  for(i=0; i<OWRX_MAX_CLIENTS; ++i)
   { if( OWRX_Client[i].dwClientIP == dwClientIP )
      { return &OWRX_Client[i];
      }
   }
  // Arrived here ? Remote client has not been registered yet !
  //                He will not be able to use anything at all.
  return NULL;
} // end OWRX_GetClientByIP()

//---------------------------------------------------------------------------
char *OWRX_StateToString(int iState)
{ switch( iState )
   { case OWRX_STATE_PASSIVE        : return "passive";
     case OWRX_OPENED_ROOT_DOCUMENT : return "opened_root";
     case OWRX_CLOSED_ROOT_DOCUMENT : return "closed_root";
     case OWRX_OPENED_WEB_SOCKET    : return "opened_ws";
     case OWRX_ACTIVE_WEB_SOCKET    : return "active";
     case OWRX_CLOSED_WEB_SOCKET    : return "closed_ws";
     default: return "???";
   }
} // end OWRX_StateToString()

//---------------------------------------------------------------------------
void OWRX_SetState( T_OWRX_Client *pOWClient, // replaces pOWClient->iState = iState
         int iState ) // [in] OWRX_STATE_PASSIVE, OWRX_OPENED_ROOT_DOCUMENT, ..
{
  if( pOWClient->iState != iState ) // if this is a state TRANSITION, log it..
   {
     if( pOWClient->pCwNet->cfg.iDiagnosticFlags & (CWNET_DIAG_FLAGS_VERBOSE | CWNET_DIAG_FLAGS_SHOW_CONN_LOG) )
      { ShowError( ERROR_CLASS_INFO | SHOW_ERROR_TIMESTAMP,
        "OWRX: Client #%d switches from '%s' to '%s'", pOWClient->iClientIndex,
         OWRX_StateToString(pOWClient->iState), OWRX_StateToString(iState) );
      }
   }
  pOWClient->iState = iState;
} // end OWRX_SetState()

//---------------------------------------------------------------------------
void OWRX_InitClient( T_OWRX_Client *pOWClient, T_HttpInstance * pHttpInst )
   // [in]   pHttpInst : The "normal" HTTP server's instance for a single client
   // [out]  pOWClient : Former OpenWebRX-compatible client data
   // Caller : OnHttpFileRequestOrCompletion()
   //           -> OWRX_OnWebRXTemplateRequest()
   //               -> OWRX_InitClient()
{
  T_OWRX_ClientInfo *pClientInfo;
  T_INET_SHA1_Context sha1_context;
  int i;
  BYTE b20Sha1Digest[20];

  memset( pOWClient, 0, sizeof(OWRX_Client) ); // <- harmless due to the absense of dynamically allocated stuff in this struct
  if( pHttpInst != NULL )
   { if( pHttpInst->pCwNet != NULL )
      {  pOWClient->pRigControl = &pHttpInst->pCwNet->RigControl;
      }
   }
  pOWClient->iClientIndex = i;
  pOWClient->pHttpInstance = pHttpInst;

  // Make a few initial guesses (just in case THE CLIENT doesn't "SET" these later):
  pOWClient->iRxChannel = 0;     // maybe one days we'll support MULTI-CHANNEL receivers like KiwiSDR
  pOWClient->iRequestedAction = OWRX_ACTION_NONE;  // do not "start" yet !

  // Default parameters for the SPECTRUM DISPLAY :
  if( pOWClient->pRigControl != NULL )
   { pOWClient->i32SpectrumCenterFreq= pOWClient->pRigControl->dblSpectrumCenterFreq_Hz;
     pOWClient->i32SpectrumBandwidth = pOWClient->pRigControl->dblScopeSpan_Hz;
     pOWClient->iSpectrumNumFreqBins = pOWClient->pRigControl->nSpectrumBinsUsed;
   }
  pOWClient->iSpectrumBufferTail  = RIGCTRL_GET_LATEST_ENTRY; // don't send "old" spectra
} // end OWRX_InitClient()


//---------------------------------------------------------------------------
void OWRX_UpdateNumClientsConnected(void)
  // Sets OWRX_Server_nClientsConnected = f( OWRX_Client[].iState, .dwClientIP).
  // This is important for the audio distribution by Spectrum Lab:
  //      If all OpenWebRX clients are gone, it will not
  //      call OWRX_Server_DistributeAudio() anymore !
{ int i, nClientsConnected=0;
  for(i=0; i<OWRX_MAX_CLIENTS; ++i)
   { if( (OWRX_Client[i].dwClientIP != 0)
      && (OWRX_Client[i].iState     != OWRX_STATE_PASSIVE) )
      { ++nClientsConnected;
        // Note: An OpenWebRX client is still considered "connected"
        //       even if, at the moment(!), there is no opened socket
        //       between client and server. The remote client (web browser)
        //       may have closed the socket (pHttpInstance) that he used
        //       to load the expanded "index.wrx", but not opened the web
        //       socket yet. During that time (several seconds), he will
        //       have to load a bunch of other files (*.js, *.css, *.png, ...)
        //       before opening the web socket.
        //       See also: Treatment of timeouts in  OWRX_CheckForTimeouts() !
      }
   }
  OWRX_Server_nClientsConnected = nClientsConnected;  // <- ONLY set here, in OWRX_UpdateNumClientsConnected() !
} // end OWRX_UpdateNumClientsConnected()

//---------------------------------------------------------------------------
void OWRX_OnConnectionClosed( T_OWRX_Client *pOWClient )  // part of the API !
   // [in]   pHttpInst->dwClientIP (only supports IPv4 so far)
   // [out]  One of the entries in OWRX_Client[], depending on the IP address.
   //
   // Called from OnHttpServerConnClosed() after closing the connection
   //  from ANY side (server, client, timeout), for those connections (TCP
   //  sockets) previously signalled by OWRX_OnWebRXTemplateRequest()
   //  or OWRX_OnWebSocketOpened() .
   // Only if the pHttpInst was the (TCP-)socket for OWRX's WEB SOCKET,
   //      this function actually frees the client's / connection's (*)
   //      instance data in an internal pool.
   // If the pHttpInst was the (TCP-) socket for delivering "index.wrx",
   //      all state stored in the T_OWRX_Client must be KEPT UNCHANGED,
   //      because we still need them (at least some of them) for later !
   //
   // (*) A client initially opens multiple connections to our server,
   //     but only the one with the Web Socket(!) remains open until
   //     the user closes the client appliation in his browser.
   //     Thus OWRX_OnConnectionClosed() will be called MULTIPLE times
   //     for a single client - so beware with the book-keeping
   //     in OWRX_UpdateClientInfoTable() !
{
  if( pOWClient != NULL )
   { // Ok, this client-instance is still valid .. but will it REMAIN valid ?
     //
     // Despite being DISCONNECTED from this socket (and the HTTP server instance),
     // the into stored in the pOWClient when delivering "index.wrx" must be kept
     // there until the (Berkeley-)socket for the Web-Socket(!!) is closed.
     // So what are we closing, the connection (socket) for "index.wrx" ?
     switch( pOWClient->iState )
      { case OWRX_OPENED_ROOT_DOCUMENT :  // seems to work as planned so far..
        case OWRX_CLOSED_ROOT_DOCUMENT :  // (closed two times ? anyway..)
           OWRX_SetState(pOWClient,OWRX_CLOSED_ROOT_DOCUMENT);
           pOWClient->pHttpInstance = NULL; // here: after loading INDEX.WRX (not the WebSocket)
           if( pOWClient->pCwNet->cfg.iDiagnosticFlags & (CWNET_DIAG_FLAGS_VERBOSE | CWNET_DIAG_FLAGS_SHOW_CONN_LOG) )
            { ShowError( ERROR_CLASS_INFO | SHOW_ERROR_TIMESTAMP,
              "OWRX: Keeping client #%d enabled after closing socket, URL=\"%s\"",
               pOWClient->iClientIndex,
               (char*)pOWClient->pHttpInstance->sz255RequestedURLWithoutQuery );
            }
           // Give the remote client client some more time
           // until we expect him to open the WEB SOCKET :
           TIM_StartStopwatch( &pOWClient->stopwatch_for_timeouts );
           pOWClient->nMillisecondsWaited = 0;
           // The timeout is monitored in OWRX_CheckForTimeouts()
           break;
        case OWRX_OPENED_WEB_SOCKET : // we're closing the WebSocket.. FAREWELL, CLIENT INSTANCE DATA !
        case OWRX_ACTIVE_WEB_SOCKET :
           pOWClient->pHttpInstance = NULL; // here: after web socket closed by THE CLIENT
           // no-no : pOWClient->dwClientIP = 0; // mark the client instance as 'unused' again
           // In some cases, a web browser closed and immediately re-opened
           // the web socket again, so don't immediately say "this client is gone" !
           OWRX_SetState(pOWClient,OWRX_CLOSED_WEB_SOCKET);
           TIM_StartStopwatch( &pOWClient->stopwatch_for_timeouts );
           pOWClient->nMillisecondsWaited = 0;
           break;
        case OWRX_CLOSED_WEB_SOCKET : // ? ?
        default :  // the evil of state machines ..
           break;
      }
     OWRX_UpdateNumClientsConnected(); // OWRX_Client[].iState -> OWRX_Server_nClientsConnected
   }
} // end OWRX_OnConnectionClosed()


//---------------------------------------------------------------------------
void OWRX_AppendTextForToken( T_OWRX_Client *pOWClient,
         char **ppszDest,  // [out] string assembled by HttpSrv_GenerateHTMLFromTemplate()
                           //       As for all string builder functions,
                           //       *ppszDest must be incremented by the number
                           //       of characters generated here.
         const char *pszEndstop, // [in]  endstop for the above destination
                           // (made this "const" in 11/2024 for pendantic compilers.
                           //  Here, it's a promise that we DON'T MODIFY the character
                           //  that pszEndstop points to.)
         int iToken )      // [in] one of the WRX_TOKEN_-values def'd in OpenWebRX_Server.h
{
  T_RigCtrlInstance *pRigControl = pOWClient->pRigControl;
  if( pRigControl == NULL )
   { SL_AppendString( ppszDest, pszEndstop, "n/a" );
     return;
   }
  switch( iToken )
   { case WRX_TOKEN_WS_URL    :
           // > %[WS_URL] is the WebSocket base URL, containing the
           // > appropriate port (its use will allow OpenWebRX to be
           // > included into existing sites using a proxy script).
           // In openwebrx.py, replace_dictionary :
           //   "%[WS_URL]" -> "ws://"+cfg.server_hostname+":"+str(cfg.web_port)+"/ws/")
           // Seen in real online receiver / Javascript the expanded index.wrx:
           // > var ws_url="ws://localhost:8888/ws/";
           //               |
           //            "protocol" : WebSocket, not encrypted
           // Since we usually don't have a domain name,
           //   use the PUBLIC IP, retrieved by a service like
           //   http://checkip.dyndns.com/ ,  http://icanhazip.com/ , etc.
           //   Don't want to fool around with this, so copy this
           //   freely adjustable string from SL's config dialog:
           SL_AppendPrintf( ppszDest, pszEndstop, "ws://%s/ws/", HttpSrv_sz80HttpServerHostnameOrPublicIP );
           break;
     case WRX_TOKEN_RX_TITLE  :
           // In openwebrx.py, replace_dictionary :
           //   "%[RX_TITLE]" -> cfg.receiver_name
           SL_AppendPrintf( ppszDest, pszEndstop, "SpectrumLab Web Server" );
           break;
     case WRX_TOKEN_RX_LOCATION:
           // In openwebrx.py, replace_dictionary :
           //   "%[RX_LOCATION]" -> cfg.receiver_location
           SL_AppendPrintf( ppszDest, pszEndstop,  "n/a" );
           break;
     case WRX_TOKEN_RX_LOCATOR :
           // In openwebrx.py, replace_dictionary :
           //   "%[RX_LOCATOR]" -> cfg.receiver_qra (abused 'QRA')
           SL_AppendPrintf( ppszDest, pszEndstop,  "n/a" );
           break;
     case WRX_TOKEN_RX_HEIGHT_ASL    :
           SL_AppendPrintf( ppszDest, pszEndstop, "0" ); // the " m" is in index.wrx . Good or bad ?
           break;
     case WRX_TOKEN_RX_GPS    :
           // In openwebrx.py, replace_dictionary :
           //   "%[RX_GPS]" -> str(cfg.receiver_gps[0])+","+str(cfg.receiver_gps[1]) .
           // The last OpenWebRX version by HA7ILM used this
           // to show the receiver's position on Google Map,
           // resulting in something like
           //  https://www.google.hu/maps/place/@52.1350987,8.493481,14z
           //                                    |________| |______|  |
           //   lat, lon in "decimal degrees" -------'--------'       |
           //      zoom setting, possibly appended by Google ---------'
           //
           SL_AppendPrintf( ppszDest, pszEndstop,  "%08.6lf,%08.6lf",
                    pRigControl->dblGpsLat_deg, pRigControl->dblGpsLon_deg );
           break;
     case WRX_TOKEN_RX_PHOTO_TITLE:
           // In openwebrx.py, replace_dictionary :
           //   "%[RX_PHOTO_TITLE]" -> str(cfg.photo_title) .
           SL_AppendPrintf( ppszDest, pszEndstop,  "Photo Title" );
           break;
     case WRX_TOKEN_RX_PHOTO_DESC  :
           SL_AppendPrintf( ppszDest, pszEndstop,  "Photo Description" );
           break;
     case WRX_TOKEN_RX_PHOTO_HEIGHT:
           // In openwebrx.py, replace_dictionary :
           //   "%[RX_PHOTO_HEIGHT]" -> str(cfg.photo_height) .
           // Seen in real online receiver / Javascript the expanded index.wrx:
           // > var rx_photo_height=1;
           SL_AppendString( ppszDest, pszEndstop, "1" );
           break;
     case WRX_TOKEN_AUDIO_BUFSIZE  :
           // Seen in real online receiver / Javascript the expanded index.wrx:
           // > var audio_buffering_fill_to=5;
           SL_AppendString( ppszDest, pszEndstop, "5" );
           break;
        // Setting all these 'START'-values (initial settings)
        // when loading the expanded index.wrx may seem old school,
        // but it greatly simplified debugging because in ANY browser,
        // you can look at the page source, and see all those
        // machine-generated Javascript var-initialisers there
        // (in a smart browser, enter view-source:http://localhost/ )
        // > var starting_mod="cw";
        // > var starting_spectrum_fcenter = 3562000;
        // > var starting_spectrum_bandwidth= 50000;
        // > var starting_spectrum_num_bins = 475;
        // > var starting_vfo_freq = 3562000;
        // > var starting_low_cut = 400;
        // > var starting_high_cut = 900;
        // > var starting_audio_offset = -650;
        //
     case WRX_TOKEN_MODULATION :
           // Seen in a real online receiver / Javascript the expanded index.wrx:
           // > var starting_mod="nfm";  alternatives seem to be "am", "lsb", "usb", "cw"
           SL_AppendPrintf( ppszDest, pszEndstop,
              OWRX_ModulationAkaOpModeToString( pOWClient->pRigControl->iOpMode ) );
           // Despite informing the client to use "cw" by default
           //   (because that's what the IC-7300 was using),
           // the GUI had the "USB" button marked in yellow (yucc).
           // Wondered why, and how those "initial settings"
           // were supposed to work. In openwebrx.js :
           // > demodulator_analog_replace(starting_mod);
           // Called from there: "demodulator_buttons_update()".
           // Sounds promising. Set a breakpoint there, on
           // > switch(demodulators[0].subtype)
           // When run for the first time,
           //   demodulators[0].subtype = "cw" (a string!)
           // it really got into the block after the 'case "cw":',
           // but : CRAP CRAP CRAP ! ! ! Anything with an audio
           // bandwidth of 300 Hz and more was autmatically marked
           // as "usb" or "lsb" on those difficult-to-see button
           // text colours (weak yellow instead of white) .
           // Fixed that bug (for VHF CW operators) as follows:
           // old:
           // > case "lsb":
           // > case "usb":
           // > case "cw":
           // >   if(demodulators[0].high_cut-demodulators[0].low_cut<300)
           // >          $("#openwebrx-button-cw").addClass("highlighted");
           // >       else
           // >       {
           // >          if(demodulators[0].high_cut<0)
           // >             $("#openwebrx-button-lsb").addClass("highlighted");
           // >          else if(demodulators[0].low_cut>0)
           // >             $("#openwebrx-button-usb").addClass("highlighted");
           // >          else $("#openwebrx-button-lsb, #openwebrx-button-usb").addClass("highlighted");
           // >       }
           // >       break;
           // new:
           // > case "lsb":
           // >       $("#openwebrx-button-lsb").addClass("highlighted");
           // >       break;
           // > case "usb":
           // >       $("#openwebrx-button-usb").addClass("highlighted");
           // >       break;
           // > case "cw":
           // >       $("#openwebrx-button-cw").addClass("highlighted");
           // >       break;
           // (To treat "cw" like "usb" or "lsb" and then decide
           //  later if "usb" or "lsb" is in fact "cw" really sucks,
           //  at least for THIS application.
           //  If our IC-7300 reports mode "CW", not "USB",
           //  then what we want to see in the client GUI is "CW" ! )
           break;
     case WRX_TOKEN_VFO_FREQ: // no funny "offset" anymore, and no distinction between "_START_"- and "CURRENT" value
           SL_AppendPrintf( ppszDest, pszEndstop, "%ld", (long)pRigControl->dblVfoFrequency );
           break;
     case WRX_TOKEN_FILTER_BW:  // let our IC-7300, not OpenWebRX, "decide" !
           SL_AppendPrintf( ppszDest, pszEndstop, "%d", (int)pRigControl->iFilterBW_Hz ); // e.g. 400 Hz (650-500/2)
           break;
     case WRX_TOKEN_AUDIO_OFFSET:
           SL_AppendPrintf( ppszDest, pszEndstop, "%d", (int)pRigControl->iCWPitch_Hz ); // e.g. 650 Hz "CW pitch", 0 Hz for SSB/AM/FM
           break;
     case WRX_TOKEN_SPECTRUM_FCENTER:
           SL_AppendPrintf( ppszDest, pszEndstop, "%ld", (long)pRigControl->dblSpectrumCenterFreq_Hz );
           break;
     case WRX_TOKEN_SPECTRUM_BANDWIDTH:
           SL_AppendPrintf( ppszDest, pszEndstop, "%ld", (long)pRigControl->dblScopeSpan_Hz );
           break;
     case WRX_TOKEN_SPECTRUM_NUM_BINS:
           SL_AppendPrintf( ppszDest, pszEndstop, "%d", (int)pRigControl->nSpectrumBinsUsed );
           break;
     case WRX_TOKEN_WATERFALL_COLORS :
           // Seen in real online receiver / Javascript the expanded index.wrx:
           // > var waterfall_colors=[0x000000ff,0x0000ffff,0x00ffffff,0x00ff00ff,0xffff00ff,0xff0000ff,0xff00ffff,0xffffffff];
           SL_AppendPrintf( ppszDest, pszEndstop, "[0x000000ff,0x0000ffff,0x00ffffff,0x00ff00ff,0xffff00ff,0xff0000ff,0xff00ffff,0xffffffff]" );
           break;
     case WRX_TOKEN_WATERFALL_MIN_LEVEL:
           // Seen in real online receiver / Javascript the expanded index.wrx:
           // > var waterfall_min_level_default=-88;
           SL_AppendPrintf( ppszDest, pszEndstop, "-88" );
           break;
     case WRX_TOKEN_WATERFALL_MAX_LEVEL:
           // Seen in real online receiver / Javascript the expanded index.wrx:
           // > var waterfall_max_level_default=-20;
           SL_AppendPrintf( ppszDest, pszEndstop, "-20" );
           break;
     case WRX_TOKEN_WATERFALL_AUTO_LEVEL_MARGIN:
           // Seen in real online receiver / Javascript the expanded index.wrx:
           // > var waterfall_auto_level_margin=[5,40];
           SL_AppendPrintf( ppszDest, pszEndstop, "[5,40]" );
           break;
     case WRX_TOKEN_DIGIMODES_ENABLE:
           // Seen in real online receiver / Javascript the expanded index.wrx:
           // > var server_enable_digimodes=true;
           SL_AppendPrintf( ppszDest, pszEndstop, "false" );
           break;

     default:
           break;
   } // end  switch( iToken )
} // end void OWRX_AppendTextForToken()

//---------------------------------------------------------------------------
int OWRX_OnSessionTimeout(  T_OWRX_Client *pOWClient )
  // Called from a lower layer ( SpecHttpSrc.c : OnHttpSrvSessionTimeout() )
  //  after an ALREADY ESTABLISHED connection timed out ("TCP Session Timeout").
  // This happens <HTTP_SERVER_SESSION_TIMEOUT_SECONDS> seconds
  //      without any traffic in any direction.
  // As usual, the return value tells the caller what to do. In THIS case:
  //    HTTP_STATUS__OK = "don't consider the Session Timeout an error",
  //    HTTP_STATUS__SRV_TIMEOUT = "send appropriate response, then close" .
  //
{
  return HTTP_STATUS__SRV_TIMEOUT; // send appropriate response, then close and free the resource
} // end OWRX_OnSessionTimeout()


//---------------------------------------------------------------------------
int OWRX_OnWebSocketOpened( T_OWRX_Client *pOWClient )
  // Called as soon as a HTTP server's connection instance
  // has entered the "WebSocket OPENED" state (in geek speak, "after the WebSocket handshake").
  // Intended to send the first "hello" from server to client,
  // e.g. for OpenWebRX: See Wireshark log shown in OWRX_OnWebSocketFrameRcvd(),
  //                  formerly using stuff like "CLIENT DE SERVER openwebrx.py"
  //     (we don't don't pass names of *.py modules because Phython has been
  //      completely eliminated. Hip-hip-hooray; here we come, dear Cortex-M0.)
  // Return value :
  //     HTTP_STATUS__OK if this server agrees to "serve" the web socket.
  //     Other possible valued defined in http_server_sourcecode/http_intf.h .
  // Note: The *TCP*-socket has been opened long before the low-level HTTP
  //       server calls OWRX_OnWebSocketOpened(). Thus the return value cannot
  //       be used to reject the initial GET-response, before "switching protocol".
  //
{
  int iHttpStatus = HTTP_STATUS__OK;
  char sz255Response[256];
  char *cpDst = sz255Response;
  char *cpDstEndstop = sz255Response + sizeof(sz255Response) - 1;
  T_HttpInstance * pHttpInst;

  if( (pOWClient != NULL)                   // valid OpenWebRX-alike session ?
   && ((pHttpInst=pOWClient->pHttpInstance) != NULL) )  // valid instance for the basic HTTP server ?
   { OWRX_SetState(pOWClient,OWRX_OPENED_WEB_SOCKET);
     SL_AppendString( &cpDst, cpDstEndstop, "CLIENT DE SERVER" );
     if( ! HttpSrv_SendWebSocketFrame( pOWClient->pHttpInstance,
              Http_WebSocketFrame_Binary/*!!!*/ | Http_WebSocketFrame_SingleFragment,
              (BYTE*)sz255Response, cpDst-sz255Response ,NULL/*no "mask"*/ ) )
      { if( pOWClient->pCwNet->cfg.iDiagnosticFlags & (CWNET_DIAG_FLAGS_VERBOSE | CWNET_DIAG_FLAGS_SHOW_CONN_LOG) )
         { ShowError( ERROR_CLASS_INFO | SHOW_ERROR_TIMESTAMP,
             "Client #%d: Failed to send WS frame to IP %s .", pOWClient->iClientIndex,
             CwNet_IPv4AddressToString( pOWClient->pHttpInstance->pSession->b4HisIP.b ) );
         }
      }
     else // successfully sent our first frame into the web socket -> state transition...
      {
        if( pOWClient->pCwNet->cfg.iDiagnosticFlags & (CWNET_DIAG_FLAGS_VERBOSE | CWNET_DIAG_FLAGS_SHOW_CONN_LOG) )
         { ShowError( ERROR_CLASS_INFO | SHOW_ERROR_TIMESTAMP,
             "Client #%d: Initial WS frame sent : %s", pOWClient->iClientIndex,
             sz255Response );
         }
      }
     OWRX_UpdateNumClientsConnected(); // OWRX_Client[].iState -> OWRX_Server_nClientsConnected
     TIM_StartStopwatch( &pOWClient->stopwatch_for_periodic_msgs_s2c );
   }
  else  // pOWClient == NULL ... is someone trying to hack or abuse us ?
   { iHttpStatus = HTTP_STATUS__NOCONTENT; // "there's NO CONTENT for you" :o)
     if( pOWClient->pCwNet->cfg.iDiagnosticFlags & (CWNET_DIAG_FLAGS_VERBOSE | CWNET_DIAG_FLAGS_SHOW_CONN_LOG) )
      { ShowError( ERROR_CLASS_ERROR | SHOW_ERROR_TIMESTAMP,
          "OWRX refused to serve WEB SOCKET (client #%d hasn't loaded our Javascript)",
          (int)HttpSrv_GetClientIndex(pHttpInst) );
      }
   }
  return iHttpStatus;
} // end OWRX_OnWebSocketOpened()


//---------------------------------------------------------------------------
int OWRX_OnWebSocketFrameRcvd( T_OWRX_Client *pOWClient,
        int iWSFrameType,      // [in] e.g. Http_WebSocketFrame_Text or Http_WebSocketFrame_Binary + usually with "FIN" (bit 7) SET
        long i32PayloadLength, // [in] number of bytes to process in pbPayload
        BYTE * pbPayload )     // [in] already decoded ("unmasked") payload
  // Called whenever another complete "frame" was received on a WebSocket.
  // Caller:  http_server_sourcecode/httpa.c : HttpSrv_OnWSRcvd()
  //           -> SpecHttpSrv.cpp : OnHttpWebSocketFrameRcvd()
  //            -> OpenWebRX_Server.c : OWRX_OnWebSocketFrameRcvd() .
  // Return value : Hopefully 'HTTP_STATUS__OK' .
{
  T_HttpInstance *pHttpInst;
  char *cpSrc = (char*)pbPayload;
  char *cpSrcEndstop = (char*)pbPayload + i32PayloadLength;
  char *cpDst, *cpDstEndstop, sz1kResponse[1024+4];
  DWORD dw3CharToken;
  cpDst = sz1kResponse;
  cpDstEndstop = sz1kResponse + 1024;

  if( pOWClient == NULL )
   { return HTTP_STATUS__SERVERERROR; // oops.. this is not an OpenWebRX-client-socket !
   }
  // The pHttpInst should have already been tied to our pOWClient ... but play safe:
  if( (pHttpInst=pOWClient->pHttpInstance) == NULL )
   { return HTTP_STATUS__SERVERERROR; // oops.. this is not an OpenWebRX-client-socket !
   }

  TIM_StartStopwatch( &pOWClient->stopwatch_for_timeouts ); // new signs of life from our client
  pOWClient->nMillisecondsWaited = 0;
  // (in case OWRX_ACTIVE_WEB_SOCKET, there was only traffic from the
  //  original OpenWebRX client by HA7LIM if the operator 'turned the dial',
  //  clicked a button, etc. We may use this to throw out visitors that
  //  leave their browser running all day long, without ever QSYing or similar)
  switch( iWSFrameType & Http_WebSocketFrame_Mask4Opcode)
   { case Http_WebSocketFrame_Text   :
        // Example, shortly after being connected by an OpenWebRX client :
        //  iWSFrameType     = 0x81 = Http_WebSocketFrame_Text | Http_WebSocketFrame_SingleFragment
        //  i32PayloadLength = 29
        //  pbPayload        = "SERVER DE CLIENT openwebrx.js" (without double quotes)
        // More commands the server needs to understand can be found in openwebrx.py,
        //   class WebRXHandler(), do_GET(), 'process commands' .
        if( pOWClient->pCwNet->cfg.iDiagnosticFlags & (CWNET_DIAG_FLAGS_VERBOSE | CWNET_DIAG_FLAGS_SHOW_CONN_LOG) )
         { ShowError( ERROR_CLASS_ERROR | SHOW_ERROR_TIMESTAMP,
             "OWRX from Client #%d : %.*s", pOWClient->iClientIndex,
             i32PayloadLength, (char*)pbPayload ); // <- limited length, because
             // pbPayload isn't necessarily a ZERO-TERMINATED "C" string !
         }

        if( i32PayloadLength>=3 )
         { dw3CharToken = Make3CharToken(pbPayload[0],pbPayload[1],pbPayload[2]);
           switch( dw3CharToken )
            { case Make3CharToken( 'S','E','T' ) : // .. followed by a list of KEY=VALUE pairs..
                 cpSrc = (char*)pbPayload + 3;
                 // For example, got here after grabbing the 'passband indicator'
                 //  and sliding it along the frequency scale.
                 OWRX_ParseSETMessage( pOWClient, cpSrc );
                 break; // end case "SET"..
              case Make3CharToken( 'S','E','R' ) : // most likely just the "SERVER DE CLIENT" ..
                 if( strncmp( cpSrc, "SERVER DE CLIENT", 16 ) == 0 ) // yep, he said "hello"...
                  { if( (pOWClient != NULL ) && (pOWClient->pRigControl != NULL ) )
                     { // state transition from OWRX_OPENED_WEB_SOCKET to ..ACTIVE..
                       OWRX_SetState(pOWClient,OWRX_ACTIVE_WEB_SOCKET); // here: because we received "SERVER DE CLIENT", i.e. the first "hello" from him
                       pOWClient->i32SpectrumCenterFreq= pOWClient->pRigControl->dblSpectrumCenterFreq_Hz;
                       pOWClient->i32SpectrumBandwidth = pOWClient->pRigControl->dblScopeSpan_Hz;
                       pOWClient->iSpectrumNumFreqBins = pOWClient->pRigControl->nSpectrumBinsUsed;
                       SL_AppendString( &cpDst, cpDstEndstop, "MSG " );
                       SL_AppendPrintf(&cpDst, cpDstEndstop,
                         "spectrum_center_freq=%ld spectrum_bandwidth=%ld fft_size=%d ",
                         (long)pOWClient->i32SpectrumCenterFreq,
                         (long)pOWClient->i32SpectrumBandwidth,
                          (int)pOWClient->iSpectrumNumFreqBins );
                       SL_AppendPrintf(&cpDst, cpDstEndstop, "max_clients=%d ",
                          (int)OWRX_MAX_CLIENTS ); // required for the bargraph behind "Clients [n]"
                       SL_AppendString( &cpDst, cpDstEndstop, "setup" );
                       // With an IC-7300 (delivering it's "Spectrum Scope" data via USB)
                       // tuned to 7.030 MHz, CW with audio pitch = 600 Hz,  500 Hz wide "Filter 2",
                       //       Spectrum Scope in CENTER mode,
                       //       "CENTER Type Display" = "Carrier Point Center (Abs. Freq)",
                       //       50 kHz wide display,
                       // SL's HTTP server should show the following log entry:
                       // > WS frame to .. : MSG spectrum_center_freq=7030000 spectrum_bandwidth=50000 fft_size=475 (..)
                       // The spectrum display in the OWRX client was ok,
                       // the frequency indicated by the "mouse readout cursor"
                       // was also ok, but the yellow passband indicator was
                       // "off" by the 600 Hz audio pitch:
                       //                  (zero beat marker)  _____  passband
                       //                         |          _/     \_
                       //
                       //  .. | | | | | | | | | | | | | | | | | | | | | ..
                       //     |                   |                   |
                       // 7.029 MHz           7.030 MHz           7.031 MHz
                       //                         :
                       //                         :
                       //     (7300 kHz test signal in the waterfall HERE, ok)
                       //      "7.030,0 MHz" [with the stupid German locale]
                       //        shown in the numeric "frequency" display, ok)
                       // - - - - - - - - - - - - - - - - - - - - - - - - - - -
                       //  As often with network protocols, Wireshark is your friend.
                       //  Enter "websocket" as filter to see what happens !
                       // - - - - - - - - - - - - - - - - - - - - - - - - - - -
                       if( ! HttpSrv_SendWebSocketFrame( pHttpInst,
                                Http_WebSocketFrame_Binary/*!!!*/ | Http_WebSocketFrame_SingleFragment,
                                (BYTE*)sz1kResponse, cpDst-sz1kResponse ,NULL/*no "mask"*/ ) )
                        {
                          if( pOWClient->pCwNet->cfg.iDiagnosticFlags & (CWNET_DIAG_FLAGS_VERBOSE | CWNET_DIAG_FLAGS_SHOW_CONN_LOG) )
                           { ShowError( ERROR_CLASS_ERROR | SHOW_ERROR_TIMESTAMP,
                                "Failed to send WS frame to client #%d .",
                               (int)HttpSrv_GetClientIndex( pHttpInst) );
                           }
                          return HTTP_STATUS__SERVERERROR; // <- like many other errors, causes the "framework" to close the connection !
                        }
                       else
                        {
                          if( pOWClient->pCwNet->cfg.iDiagnosticFlags & (CWNET_DIAG_FLAGS_VERBOSE | CWNET_DIAG_FLAGS_SHOW_CONN_LOG) )
                           { ShowError( ERROR_CLASS_ERROR | SHOW_ERROR_TIMESTAMP,
                               "OWRX to Client #%d : %s",
                               (int)HttpSrv_GetClientIndex( pHttpInst),
                                sz1kResponse );
                           }
                        }
                     } // end if( pOWClient != NULL )
                  } // end if "SERVER DE CLIENT"
                 break; // end case "SER"..

              default : // Must be "something new" we don't understand yet. Just ignore.
                 break;
            } // end switch( dw3CharToken )
         } // end if( i32PayloadLength>=3 )
        break; // end case Http_WebSocketFrame_Text

     case Http_WebSocketFrame_Binary :
        break; // end case Http_WebSocketFrame_Binary

   } // end switch( iWSFrameType )

  (void)cpSrcEndstop; 
  return HTTP_STATUS__OK;

} // end OWRX_OnWebSocketFrameRcvd()

//---------------------------------------------------------------------------
CPROT int OWRX_ParseModulationAkaOpMode( const char **cppSrc )
  // Returns what in RigControl.c (and in many Icom CiV-manuals)
  //         is called the "op-mode".. buaaaah .. e.g. :
  // RIGCTRL_OPMODE_UNKNOWN, RIGCTRL_OPMODE_CW / .._LSB / .._USB / .._AM
  //                         .._FM_WIDE, .._RTTY, etc.
{
  return SL_SkipOneOfNTokens( cppSrc, OpenWebRX_ModulationsAkaOpModes );
} // end OWRX_ParseModulationAkaOpMode()

//---------------------------------------------------------------------------
const char *OWRX_ModulationAkaOpModeToString( int iRigOpMode )
{ return SL_GetStringFromTokenList( OpenWebRX_ModulationsAkaOpModes, iRigOpMode );
} // end OWRX_ModulationAkaOpModeToString()


//---------------------------------------------------------------------------
CPROT void OWRX_ParseSETMessage( T_OWRX_Client *pOWClient, const char * cpSrc )
  // Called from OWRX_OnWebSocketFrameRcvd() (*) on reception of any
  // CLIENT-to-SERVER message beginning with "SET" (in ASCII),
  // followed by a string of key=value pairs as decribed further below.
  //
  // (*) Call stack, seen in Spectrum Lab :
  //     TApplication::Run() -> TApplication::HandleMessage() -> TApplication::ProcessMessage()
  //      -> Classes::StdWndProc()  [all this is Borland VCL stuff, forget it..)
  //       -> TSpectrumLab::MyWindowProc()
  //        -> HttpSrv_HandleAsyncMsg() -> HttpSrv_OnRead()
  //         -> HttpSrv_OnWSRcvd()
  //          -> OnHttpWebSocketFrameRcvd( cpSrc=" ) .
  //
{
  int  iNewValue;
  BOOL fPassbandOrVFOModifiedByClient = FALSE;

  if( ! SL_SkipChar( &cpSrc, ' ' ) )
   { return; // missing the mandatory space between "SET" and the first key (name)
   }
  while( *cpSrc>' ' )
   { if( SL_SkipString( &cpSrc, "action=" ) )
      { // With the last original OpenWebRX client (2019-12-29),
        //  got here with "action=start".
        if( SL_SkipString( &cpSrc, "start" ) )
         { pOWClient->iRequestedAction = OWRX_ACTION_START;
         }
        else if( SL_SkipString( &cpSrc, "stop" ) )
         { pOWClient->iRequestedAction = OWRX_ACTION_STOP;
         }
      }
     else if( SL_SkipString( &cpSrc, "mod=" ) )
      { // ... got here with "mod=ssb" even when the client GUI shows "
        iNewValue = OWRX_ParseModulationAkaOpMode( &cpSrc );
        if( pOWClient->iRigOpMode != iNewValue )
         {
           pOWClient->nMinutesWithoutUserEvents = 0; // signs of human life out there !
         }
        pOWClient->iRigOpMode = iNewValue;
      }
     else if( SL_SkipString( &cpSrc, "low_cut=" ) )
      { // ... got here with "low_cut=300" [Hz, audio],
        // immediately after opening, but also when moving the LOWER EDGE
        // of the yellow passband indicator along the frequency scale.
        // The three keys "low_cut", "high_cut" and "offset_freq" were
        // sent in a single message, but don't bet on that !
        iNewValue = SL_JSON_ParseInt( &cpSrc, 0/*iDefault*/ );
        if( pOWClient->iLowCutoff_Hz != iNewValue )
         { fPassbandOrVFOModifiedByClient = TRUE; // new iLowCutoff_Hz
           pOWClient->nMinutesWithoutUserEvents = 0; // signs of human life ..
         }
        pOWClient->iLowCutoff_Hz = iNewValue;
      }
     else if( SL_SkipString( &cpSrc, "high_cut=" ) )
      { // ... got here with "high_cut=3000" [Hz, audio].
        iNewValue = SL_JSON_ParseInt( &cpSrc, 0/*iDefault*/ );
        if( pOWClient->iHighCutoff_Hz != iNewValue )
         { fPassbandOrVFOModifiedByClient = TRUE; // new iHighCutoff_Hz
           pOWClient->nMinutesWithoutUserEvents = 0;
         }
        pOWClient->iHighCutoff_Hz = iNewValue;
      }
     else if( SL_SkipString( &cpSrc, "audio_offset=" ) ) // aka "CW Pitch"..
      { iNewValue = SL_JSON_ParseInt( &cpSrc, 0/*iDefault*/ );
        if( pOWClient->iAudioOffset_Hz != iNewValue )
         { fPassbandOrVFOModifiedByClient = TRUE; // new iAudioOffset_Hz
           pOWClient->nMinutesWithoutUserEvents = 0;
         }
        pOWClient->iAudioOffset_Hz = iNewValue;
      }
     else if( SL_SkipString( &cpSrc, "vfo_freq=" ) ) // no funny "offset" !
      { // Got here immediately after opening with the default from "%[VFO_FREQ]",
        //  but also later when moving the yellow passband indicator
        //  along the frequency scale (in the original OpenWebRX page),
        //  and much much later also when EDITING the frequency manually.
        iNewValue = SL_JSON_ParseInt( &cpSrc, 0/*iDefault*/ );
        if( pOWClient->i32VFOFreq_Hz != iNewValue )
         { fPassbandOrVFOModifiedByClient = TRUE; // new i32VFOFreq_Hz
           pOWClient->nMinutesWithoutUserEvents = 0;
         }
        pOWClient->i32VFOFreq_Hz = iNewValue;
      }
     else // none of the above, so skip the unknown key and the unknown value further below
      {
      }
     // The "value" may have been skipped or not (further above).
     // Keep it simple and skip all normal characters until, and including,
     // the SPACE that separates two key=value pairs :
     if( ! SL_SkipCharsUntilDelimiter( &cpSrc, " "/*delimiter*/, SL_SKIP_NORMAL ) )
      { break;
      }
   } // end while( *cpSrc>' ' )

  pOWClient->fPassbandOrVFOModifiedByClient |= fPassbandOrVFOModifiedByClient;
} // end OWRX_ParseSETMessage()

//---------------------------------------------------------------------------
int OWRX_Server_GetNextSpectrum(                         // internal (no API)
       T_OWRX_Client *pOWClient,  // [in] Remote client instance data, with...
                                // [in]  pOWClient->i32SpectrumCenterFreq,
                                // [in]  pOWClient->i32SpectrumBandwidth,
                                // [in]  pOWClient->iSpectrumNumFreqBins,
                                //       initially set in OWRX_OnWebSocketFrameRcvd() .
       short *pi16Dest,         // [out] frequency bins (amplitude spectrum) as 16-bit integers,
                                //       *not* compressed here but by the caller !
       int   nMaxFreqBins)      // [in] maximum number of "frequency bins" to be placed in pi16Dest.
                                //      Typically limit to a few hundred points,
                                //      to keep the bandwidth low.
   // Retrieves the 'next' spectrum from Spectrum Lab's "waterfall feeding buffer",
   //  and converts it into signed 16-bit integer samples suitable for compression
   //  (ADPCM compression takes place in the caller, see below).
   // OWRX_Server_GetNextSpectrum() scales the floating-point "dB" values
   //    by 100.0 for reasons explained in OWRX_OnWebSocketPollForTxData().
   // Return value : Number of frequency bins really used,
   //                or ZERO if there's no new spectrum available yet.
   // Caller : OWRX_OnWebSocketPollForTxData(), if there's enough space
   //          in the web-socket's outbound buffer after feeding in *AUDIO*.
   //          (consequence: If the web socket cannot keep up the pace,
   //                        the AUDIO channel has priority over the
   //                        WATERFALL / SPECTRUM display. That's ok.)
{
  int iDestBinIndex, nDestBinsAvailable = 0;
  double dblFreq, dblFminWanted, dblBinWidth_Wanted;
  double dblSrcBinIndex, dblMagnitude, dblPeakMagnitude;
  int    iSourceBinIndex, iPrevSrcBinIndex;

  // OWRX_GetSpectrumFromFIFO() is the 'interface' to Spectrum Lab,
  //     thus not part of OpenWebRxServer.cpp but e.g. in SpecDisp.cpp !
  //     Spectrum Lab itself doesn't give a damn about which frequencies
  //     are displayed in the remote OpenWebRX client; instead it retrieves
  //     a T_RigCtrlSpectrum covering *Spectrum Lab's* current frequency band
  //     for the waterfall (which, in some places, may be the broadband
  //     spectrum read from an IC-7300, IC-9700, or similar). This range
  //     cannot be 'individually' controlled by each client / user !
  T_RigCtrl_Spectrum *pSpectrum = RigCtrl_GetSpectrumFromFIFO(
          &pOWClient->pCwNet->RigControl, &pOWClient->iSpectrumBufferTail );

  if( pSpectrum == NULL )
   { return 0;  // nothing new available yet, "please call me back later"
   }
  // The frequency bins in SL's T_RigCtrlSpectrum structure usually don't match
  // the displayed frequency range in the remote client, we may have to
  // interpolate or resample parts of the spectrum to make things compatible .
  dblFminWanted = (double)pOWClient->i32SpectrumCenterFreq - 0.5*(double)pOWClient->i32SpectrumBandwidth;
  if( nMaxFreqBins < 2 ) // avoid div-by-zero further below
   { return 0;  // cannot draw a reasonable spectrum displays with this anyway
   }
  // If the OpenWebRX client only "wants" TWO frequency bins,
  //    the "wanted" FFT bin width (a frequency in Hertz) would be
  //    equal to pOWClient->i32SpectrumBandwidth, not divided by TWO !
  dblBinWidth_Wanted = (double)pOWClient->i32SpectrumBandwidth
                     / (double)(pOWClient->iSpectrumNumFreqBins-1);
  nDestBinsAvailable = nMaxFreqBins;
  if( nDestBinsAvailable > pOWClient->iSpectrumNumFreqBins )
   {  nDestBinsAvailable = pOWClient->iSpectrumNumFreqBins;
      // 2020-06-13 : Got here with pOWClient->iSpectrumNumFreqBins = 0,
      //              after running without problems for some time.
      //              Of course, with pOWClient->iSpectrumNumFreqBins = 0,
      //              the remote waterfall display STOPPED (out of the blue).
      //
   }
  iPrevSrcBinIndex = -1;
  for( iDestBinIndex=0; iDestBinIndex<nDestBinsAvailable; ++iDestBinIndex)
   { dblFreq = dblFminWanted + (double)iDestBinIndex * dblBinWidth_Wanted;

     // Convert the OpenWebRX client's FFT-bin-center-frequency (dblFreq)
     // into a bin index for the SOURCE spectrum. Only in an ideal case,
     // there's a one-to-one relation between source- and destination index.
     // See C:\cbproj\Remote_CW_Keyer\RigControl.h for the specification of the
     //     scaling parameters (in the T_RigCtrl_Spectrum struct, with stuff like
     //      nBinsUsed, dblBinWidth_Hz, dblFmin_Hz (a "radio frequency")
     //      dblCenterFreq_Hz (which is NOT the same as the radio's "VFO" frequency),
     //      fFixedEdgeMode   (which is a speciality of modern Icom radios of the 2020s),
     //      fltMagnitudes_dB[ RIGCTRL_MAX_FREQ_BINS_PER_SPECTRUM ],
     //      iAmplitudeUnit (e.g. SCALE_UNIT_dB = "Icom's dB scale where ZERO is near the NOISE FLOOR" (below "S0")
     //      fltAmplOffset_dBm (add this to fltMagnitudes_dB[] to convert into dBm = "dB 'over' one milliwatt")
     if( pSpectrum->dblBinWidth_Hz > 0.0 ) // avoid div-by-zero..
      { dblSrcBinIndex = (dblFreq - pSpectrum->dblFmin_Hz) / pSpectrum->dblBinWidth_Hz;
      }
     else
      { dblSrcBinIndex = iDestBinIndex;
      }
     // Note: dblSrcBinIndex is a *double* (floating point) for interpolation,
     //       which may be added here one fine day.
     //       iSourceBinIndex is used for the array access.
     //       iPrevSrcBinIndex is used for peak detection:
     //             We don't want to SKIP any FFT frequency bin,
     //             but pick the peak if the source bins are narrower
     //             than the destination bins.
     iSourceBinIndex = (long)dblSrcBinIndex; // -> ideally 0,1,2,3,...474 (with IC-7300)
     if( (iSourceBinIndex>=0) && (iSourceBinIndex<pSpectrum->nBinsUsed) )
      { if( (iPrevSrcBinIndex < 0) || (iPrevSrcBinIndex>iSourceBinIndex) )
         {  iPrevSrcBinIndex = iSourceBinIndex;
         }
        dblPeakMagnitude = -999.0/*dBm?*/;
        do
         { // Try to receive the magnitude in some kind of "dB" for this 'radio' frequency:
           dblMagnitude = pSpectrum->fltMagnitudes_dB[ iPrevSrcBinIndex++ ];
           if( dblMagnitude > dblPeakMagnitude )
            {  dblPeakMagnitude = dblMagnitude;
            }
         } while( iPrevSrcBinIndex <= iSourceBinIndex );
        pi16Dest[iDestBinIndex] = dblPeakMagnitude * 100.0; /* <- scaling factor for OWRX*/

      }
     else // emit an "invalid" dummy for the OWRX spectrum / waterfall display
      { pi16Dest[iDestBinIndex] = -200.0/*dB*/ * 100.0;
      }
   } // end for < all "wanted" FFT frequency bins >

  return nDestBinsAvailable;   // -> usually returns to OWRX_Server_GetNextSpectrum()

} // end OWRX_Server_GetNextSpectrum()


//---------------------------------------------------------------------------
void OWRX_OnWebSocketPollForTxData( T_OWRX_Client *pOWClient )
  // Called periodically when pHttpInst->fWaitingToContinueWrite==FALSE,
  //  *and* pHttpInst->iTxBufferNumBytesPending == 0 ("nothing to send"),
  //  *and* pHttpInst->nServerState == C_HttpSrvState_WEBSOCKET_OPEN .
{
  int iMaxPayloadSize, nFreeSpaceRemaining;
  BYTE b1kTemp[1024]; // temporary buffer for audio- and waterfall compression
  int i, k, n;
  float fltSample;    // sample as if pours out of the "DSP" (floating point)
  short i16Sample;    // sample in the format we need for the A-Law encoder
  T_HttpInstance *pHttpInst;
  T_CwNet        *pCwNet;
  T_CwNetClient  *pClient;
  // ex: T_ADPCM_Codec *pCodec;        // ADPCM codec for *audio* must be kept "alive" between calls (guesswork, 2019-03)
  // ex: T_ADPCM_Codec sSpectrumCodec; // doesn't need to be kept "alive" between calls
  // 2024-02 : For the simplified 'Remote CW Keyer', we use simple A-Law compression
  //           for the above which is entirely "stateless" - one sample in, one sample out.

  if( (pOWClient!=NULL)
   && ((pHttpInst=pOWClient->pHttpInstance)!=NULL)
   && ((pClient=pHttpInst->pClient)!= NULL)
   && ((pCwNet=pOWClient->pCwNet)  != NULL)
   && (pOWClient->pCwNet->pCwDSP   != NULL)  // ... omg, so many modules and instances involved :o)
    )
   { // Ok, this instance is still valid, from OWRX_InitClient() .
     // Sometimes got here when the pOWClient had FIRST delivered the root document,
     //           then was delivered a web socket already,
     //           then -while still delivering audio & spectra-
     //                was "abused" by an other socket (with the same IP)
     //                to deliver the root document (index.wrx) a SECOND TIME,
     //           then (again for the socket/pHttpInst for the WebSocket)
     //           was asked to deliver more audio & spectra again.
     //  Confusing ? It is. But this server only has ONE instance (pOWClient)
     //              per client-IP-address, so to fix the problem described above,
     //              we simply switch this pOWClient into OWRX_ACTIVE_WEB_SOCKET
     //              again, because if the WebSocket wasn't still open,
     //              HttpSrv_OnPollForTxData() -> OnHttpSrvPollForTxData()
     //              wouldn't have called OWRX_OnWebSocketPollForTxData() :
     if( (pOWClient->iState != OWRX_OPENED_WEB_SOCKET) && (pOWClient->iState != OWRX_ACTIVE_WEB_SOCKET) ) // oops...
      { OWRX_SetState( pOWClient, OWRX_ACTIVE_WEB_SOCKET ); // here: because we're polled for WebSocket-data
        OWRX_UpdateNumClientsConnected(); // OWRX_Client[].iState -> OWRX_Server_nClientsConnected
        // 2020-06-16 : This kludge seemed to fixe the problem of
        //              "no audio after reloading the page in Firefox".
      }

     // Compress and send as many AUDIO SAMPLES as we can...
     iMaxPayloadSize = HttpSrv_GetFreeSpaceInTxBuffer( pOWClient->pHttpInstance ) - 6;
     n = CwDSP_GetNumSamplesInFifoForTailIndex( &pOWClient->pCwNet->pCwDSP->sInputFifo, pClient->iS2CAudioTail );
     if( n > (iMaxPayloadSize - 20/*"framing overheader"*/ ) )
      {  n = (iMaxPayloadSize - 20 );
      }
     if( n > (sizeof(b1kTemp) - 20 ) )
      {  n =  sizeof(b1kTemp) - 20;
      }

     if( n >= 512 )  // avoid sending very small fragments ..
      { // Assemble the next WS frame with COMPRESSED AUDIO .
        // Note: This is NOT compatible with OpenWebRX at all
        //   because we use exactly the same A-Law compressed stream
        //   with <CWNET_STREAM_SAMPLING_RATE> .
        // Counterpart (A-Law-decompressing in *.js : on_ws_recv(evt):
        //  > if(first3Chars=="AUD")
        //  >    {
        //  >       var audio_data; ...
        //  >    }
        // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
        // > When the script receives a message starting with the letters 'AUD',
        // > it initializes the Web Audio API (if it has not been already initialized),
        // > and prepares the payload to be output to the sound card.
        // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
        // Note: The above is only a tiny fraction of the whole story !
        //       Set a (Javascript-)breakpoint on
        //       > audio_node.connect(audio_context.destination);
        //                   in ?.js : "function audio_init()",
        //       and tried to make sense of the funny-named properties
        //       (e.g. "audio_node.context.destination.channelInterpretation" = "speakers"),
        //             "audio_node.context.listener.sampleRate" = 8000 (!) ...
        //       Step into > webrx_set_param("audio_rate",audio_context.sampleRate); ...
        // See also: "Audio output [xyz ksps]" displayed on OpenWebRX'es
        //           status panel, client side implementation in
        //           [C:\OpenWebRX\htdocs]openwebrx.js : debug_audio() :
        //   > var audio_output_value=(audio_buffer_current_count_debug*audio_buffer_size)
        //   >                           /audio_debug_time_taken;
        //   > progressbar_set(e("openwebrx-bar-audio-output"),
        //   >                    audio_output_value/55000,
        //   >                    "Audio output ["+(audio_output_value/1000).toFixed(1)
        //   >                    +" ksps]",audio_output_value>55000||audio_output_value<10000);
        //  Thus, "audio_output_value" (strange name for a sampling rate in Hertz; get rid of it..)
        //  has been CALCULATED by the Javascript.
        //  [in] audio_buffer_current_size_debug  [ever-changing, even when paused via debugger]
        //  [in] kbps_mult  :  8 [bits per sample with ADPCM compression],
        //  [in] audio_debug_time_since_last_call [not sure about the unit, possibly seconds]
        //

        strcpy( (char*)b1kTemp, "AUD " ); // <- token recognized in ?.js :: on_ws_recv()
        k = 4;    // index into b1kTemp[] to append an audio frame (within the websocket frame, after the "AUD ")
        for(i=0; i<n; ++i )
         { CwDSP_ReadFromFifo( &pOWClient->pCwNet->pCwDSP->sInputFifo, &pClient->iS2CAudioTail, &fltSample, 1/*nSamples*/ );
           CwDSP_FloatToShort( &fltSample/*pfltIn*/, &i16Sample/*pi16Out*/, 1/*nSamples*/, 32767.0/*factor*/ );
           b1kTemp[k++] = LinearToALawSample(i16Sample);
         }
        HttpSrv_SendWebSocketFrame( pHttpInst,
              Http_WebSocketFrame_Binary | Http_WebSocketFrame_SingleFragment,
              b1kTemp, k, NULL/*no "mask"*/ );
      } // end while < sufficient number of AUDIO SAMPLES for transmission > ?

     // If our host (e.g. the "Remote CW Keyer") decided to 'push' new parameters
     // to this client, and there is enough buffer space, send those params.
     // Remember, in OpenWebRX, "SET key=value" goes from client to server,
     //                   while "MSG key=value" goes from server to client.
     // These NEW items are parsed in a modified ?.js : on_ws_recv() .
     if(   pOWClient->fPassbandOrVFOModifiedByHost
        && (HttpSrv_GetFreeSpaceInTxBuffer(pHttpInst) > 100)
        && (pOWClient->pRigControl != NULL ) )
      { sprintf( (char*)b1kTemp, "MSG mod=%s low_cut=%d high_cut=%d audio_offs=%d vfo_freq=%ld",
            OWRX_ModulationAkaOpModeToString( pOWClient->pRigControl->iOpMode ),
            pOWClient->iLowCutoff_Hz,
            pOWClient->iHighCutoff_Hz,
            pOWClient->iAudioOffset_Hz,
            pOWClient->i32VFOFreq_Hz); // <- aka "dial frequency", DOES NOT DEPEND on the scope center frequency !
        HttpSrv_SendWebSocketFrame( pHttpInst,
              Http_WebSocketFrame_Binary | Http_WebSocketFrame_SingleFragment,
              b1kTemp, strlen( (const char*)b1kTemp ), NULL/*no "mask"*/ );
        pOWClient->fPassbandOrVFOModifiedByHost = FALSE; // "done".
      } // end if < fPassbandOrVFOModifiedByHost >

     if(   pOWClient->fSpectrumDisplayModifiedByHost // <- originates from WebServer_UpdateVFOFreqInRemoteClients()
       && (HttpSrv_GetFreeSpaceInTxBuffer(pHttpInst) > 100) )
      { sprintf( (char*)b1kTemp, "MSG spectrum_center_freq=%ld spectrum_bandwidth=%ld fft_size=%d",
           pOWClient->i32SpectrumCenterFreq, pOWClient->i32SpectrumBandwidth, pOWClient->iSpectrumNumFreqBins );
        HttpSrv_SendWebSocketFrame( pHttpInst,
              Http_WebSocketFrame_Binary | Http_WebSocketFrame_SingleFragment,
              b1kTemp, strlen( (const char*)b1kTemp ), NULL/*no "mask"*/ );
        pOWClient->fSpectrumDisplayModifiedByHost = FALSE; // "done".
      } // end if < fSpectrumDisplayModifiedByHost >
     if( pOWClient->fSendSMeter
       && (HttpSrv_GetFreeSpaceInTxBuffer(pHttpInst) > 100) )
      { sprintf( (char*)b1kTemp, "MSG smeter=%d", (int)pOWClient->iSMeterLevel_dB );
        HttpSrv_SendWebSocketFrame( pHttpInst,
               Http_WebSocketFrame_Binary | Http_WebSocketFrame_SingleFragment,
               b1kTemp, strlen( (const char*)b1kTemp ), NULL/*no "mask"*/ );
        pOWClient->fSendSMeter = FALSE; // "done".
      } // end if < fSendSMeter >


     // If enough free buffer space remains, also send another SPECTRUM:
     iMaxPayloadSize = HttpSrv_GetFreeSpaceInTxBuffer( pOWClient->pHttpInstance ) - 6;
     n = RigCtrl_GetNumSpectraAvailableInFIFO( &pCwNet->RigControl, pOWClient->iSpectrumBufferTail );
     k = 4;  // (byte-)index into b1kTemp[] to append an FFT
             //   within the websocket frame, after the token "FFT " .
     if( (n>0) && (iMaxPayloadSize > ( pCwNet->RigControl.nSpectrumBinsUsed + k ) ) )
      { T_RigCtrl_Spectrum *pSpectrum = RigCtrl_GetSpectrumFromFIFO(
                &pCwNet->RigControl, &pOWClient->iSpectrumBufferTail );
        if( pSpectrum != NULL )
         { n = pSpectrum->nBinsUsed;
           strcpy( (char*)b1kTemp, "FFT " ); // <- token recognized in ?.js :: on_ws_recv()
           if( n > (iMaxPayloadSize-k) )
            {  n =  iMaxPayloadSize-k;
            }
           for(i=0; i<n; ++i )
            {
              fltSample = pSpectrum->fltMagnitudes_dB[i] + pSpectrum->fltAmplOffset_dBm;
              // At this point, fltSample is in dBm (with "m" = "milliwatt") .
              // Thus, the VALUE RANGE of fltSample may be ..
              //   -127 dBm for the theoretic minimum of "S0",
              //   -121 dBm for an "S1" report, -73 dBm for an "S9" report,
              //   and so on. To avoid sacrificing resolution in our
              //   A-Law compressed EIGHT-BIT sample, scale the "dBm" value into
              //   a 16-bit integer value (i16Sample) as follows:
              fltSample = ( fltSample + 64.0 ) / 64.0;  // normalize to -1.0 .. +1.0,
              //   where -1.0 would be -128/*!*/ dBm ("below the natural noise floor"),
              //     and +1.0 would be   0/*!!*/ dBm ("1 mW frying the receiver's input").
              CwDSP_FloatToShort( &fltSample/*pfltIn*/, &i16Sample/*pi16Out*/, 1/*nSamples*/, 32767.0/*factor*/ );
              b1kTemp[k++] = LinearToALawSample(i16Sample);
            }
           // On the server side, the decompression of these frames
           //    happens in ?.js : on_ws_recv() , somewhere after
           // > if(first3Chars=="FFT"),
           //    set a breakpoint on
           // > var waterfall_i16=fft_codec.decode(new Uint8Array(evt.data,4));
           //      and (when paused there) step over this line and inspect
           //      'waterfall_i16' in the Javascript debugger. The result
           //      should be SIMILAR (but not necessarily EQUAL) to the samples
           //      we stuffed into the encoder above.
           HttpSrv_SendWebSocketFrame( pHttpInst,
               Http_WebSocketFrame_Binary | Http_WebSocketFrame_SingleFragment,
               b1kTemp, k, NULL/*no "mask"*/ );
         } // end if( pSpectrum != NULL )
      }   // end if < enough space in the TX-buffer to send another SPECTRUM > ?

      // Very low priority (even lower than sending spectra) :
      // Send data for the OpenWebRX GUI's "Status" panel ?
      //   (in the original client, there were only
      //    "Server CPU [x %]"  and "Clients [n]" sent over the network.
      //    Indicators for "Audio buffer", "Audio output", "Audio stream [x kbps]",
      //               and "Network usage [56.7 kbps]" were all updated
      //               'locally' in ?.js )
      // Keep it simple, and send these values PERIODICALLY every 2 seconds.
      //      The average bandwidth required by this is neglectable.
      if(  (HttpSrv_GetFreeSpaceInTxBuffer(pHttpInst) > 100)
        && (TIM_ReadStopwatch_ms(&pOWClient->stopwatch_for_periodic_msgs_s2c) >= 2000) )
       { sprintf( (char*)b1kTemp, "MSG cpu_usage=%d clients=%d",
            (int)OWRX_Server_iCpuUsagePercent,
            (int)OWRX_Server_nClientsConnected );
         HttpSrv_SendWebSocketFrame( pHttpInst,
            Http_WebSocketFrame_Binary | Http_WebSocketFrame_SingleFragment,
            b1kTemp, strlen( (const char*)b1kTemp ), NULL/*no "mask"*/ );
         TIM_StartStopwatch(&pOWClient->stopwatch_for_periodic_msgs_s2c);
       } // end if < send "cpu_usage" and "clients" (number of..) >
   }
} // end OWRX_OnWebSocketPollForTxData()


//---------------------------------------------------------------------------
BOOL OWRX_IsConnected( DWORD dwClientIP )
  // Checks if a client with the given IP address is *currently* connected .
  // [in] OWRX_Client[OWRX_MAX_CLIENTS]
{ int iClientIndex;
  T_OWRX_Client *pOWClient;
  for(iClientIndex=0; iClientIndex<OWRX_MAX_CLIENTS; ++iClientIndex)
   { pOWClient = &OWRX_Client[iClientIndex];
     if( (pOWClient->dwClientIP==dwClientIP) && (!pOWClient->fClosed) )
      { return TRUE; // gotcha
      }
   }
  // Arrived here ? The specified IP doesn't seem to be CURRENTLY CONNECTED ->
  return FALSE;
} // end OWRX_IsConnected()

//---------------------------------------------------------------------------
T_OWRX_ClientInfo *OWRX_GetClientInfoPtr( T_OWRX_Client *pOWClient )
  // Retrieves the address of a table entry with 'permanent' info
  //  about a certain client in OWRX_ClientInfo[OWRX_MAX_CLIENT_INFO_ENTRIES] .
  //  This table is displayed in Spectrum Lab's "HTTP Server Client list".
  //  If you have the source, see HttpServerUI.cpp : THttpSrvForm::UpdateListOfClients().
  // [in]  T_OWRX_Client *pOWClient (a "currently connected" or "disconnecting" client instance)
  // [return] pointer OWRX_ClientInfo[0..OWRX_MAX_CLIENT_INFO_ENTRIES-1] .
  // Returns NULL when all available entries are occupied,
  //         which absolutely DOES NOT MATTER for the operation of this server
  //         (except from not being able to remember settings of this client)
{ int iClientInfoIndex;
  T_OWRX_ClientInfo *pInfo;
  double dblTimeNow = UTL_GetCurrentUnixDateAndTime(); // we can already update the "times" here

  if( pOWClient->dwClientIP == 0 ) // oops.. this client doesn't have a valid IP
   { return NULL; // .. refuse to add him to our 'info'-table
   }

  // First check if we already "know" this client by his IP address.
  // If we do, don't add a NEW entry, but update the exising one.
  for(iClientInfoIndex=0; iClientInfoIndex<OWRX_MAX_CLIENT_INFO_ENTRIES; ++iClientInfoIndex)
   { pInfo = &OWRX_ClientInfo[iClientInfoIndex];
     if( pOWClient->dwClientIP==pInfo->dwClientIP )
      { // Bingo .. no need to occupy a new entry because we already know this guy:
        pInfo->dblUnixTimeOfLastActivity = dblTimeNow;
        return pInfo;
      }
   }
  // Arrived here ? The client's current IP is unknown -> occupy a new entry,
  //                and pre-set the "times of his first and last visit".
  for(iClientInfoIndex=0; iClientInfoIndex<OWRX_MAX_CLIENT_INFO_ENTRIES; ++iClientInfoIndex)
   { pInfo = &OWRX_ClientInfo[iClientInfoIndex];
     if( pInfo->dwClientIP==0 )  // here's another UNUSED entry -> occupy it
      { memset( pInfo, 0, sizeof(T_OWRX_ClientInfo) );
        pInfo->dwClientIP = pOWClient->dwClientIP;
        pInfo->dblUnixTimeOfFirstVisit = pInfo->dblUnixTimeOfLastActivity = dblTimeNow;
        return pInfo;
      }
   }

  // Arrived here ? That's impressive, our server had more visitors
  // (at least more different IPs) than OWRX_MAX_CLIENT_INFO_ENTRIES !
  return NULL; // maximum number of entries in OWRX_ClientInfo[] exceeded
} // end OWRX_GetClientInfoPtr()

//---------------------------------------------------------------------------
void OWRX_CheckForTimeouts(void) // periodically called from the main task, to...
  // * check for OpenWebRX clients that have loaded the root file
  //   (expanded index.wrx) but have not (never) tried to open the Web Socket
  //    - i.e. got stuck somewhere in the state when there is NO CONNECTION
  //      between client and server, which is normal (!) for a few seconds.
  // * check for OpenWebRX clients that have exceeded their total usage time
  // * do bookkeeping and timeout monitoring.
  //
{ int iClientIndex, iClientInfoIndex;
  long nMilliseconds;
  T_OWRX_Client     *pOWClient;
  T_OWRX_ClientInfo *pInfo;
  char *cp;
  double dblTimeNow = UTL_GetCurrentUnixDateAndTime();

  for(iClientIndex=0; iClientIndex<OWRX_MAX_CLIENTS; ++iClientIndex)
   { pOWClient = &OWRX_Client[iClientIndex];

     // While a client is alive, update his "time last seen"
     // for the display in the HTTP server's 'Clients' tab .
     // We do this HERE, in OWRX_CheckForTimeouts(), only once in
     // some hundred milliseconds to avoid wasting time in the TCP handlers:
     if( (pOWClient->iState!=OWRX_STATE_PASSIVE)
       &&(pOWClient->iState!=OWRX_CLOSED_WEB_SOCKET) )
      { for(iClientInfoIndex=0; iClientInfoIndex<OWRX_MAX_CLIENT_INFO_ENTRIES; ++iClientInfoIndex)
         { pInfo = &OWRX_ClientInfo[iClientInfoIndex];
           if( pOWClient->dwClientIP==pInfo->dwClientIP )
            { pInfo->dblUnixTimeOfLastActivity = dblTimeNow;
            }
         }
      } // end if < client neither PASSIVE nor CLOSED_WEB_SOCKET >

     // Is the server-instance for this client in one of the states
     // where we really need to check for timeouts, i.e. "active" ?
     switch( pOWClient->iState )
      { case OWRX_OPENED_ROOT_DOCUMENT : // e.g. loading the expanded "index.wrx"
        case OWRX_CLOSED_ROOT_DOCUMENT : // loaded "index.wrx" (and some others) but didn't open a WebSocket yet
        case OWRX_OPENED_WEB_SOCKET    : // opened the web socket but not ready for audio+spectra yet
           // All these states belong to the "initialisation phase".
           // Not ready for business yet. But how long is this going to take ?
           nMilliseconds = TIM_ReadStopwatch_ms( &pOWClient->stopwatch_for_timeouts );
           if( nMilliseconds > 10000 )   // only show this every 10 s, but keep counting
            {  // something seems to has starved ... but remember the incredible
               // page-loading-time on a smartphone via Edge-connection !
               // (it worked but took ages, don't spoil it by using too low
               //  timeouts.
               pOWClient->nMillisecondsWaited += nMilliseconds;
               TIM_StartStopwatch( &pOWClient->stopwatch_for_timeouts );
               switch( pOWClient->iState )
                { case OWRX_OPENED_ROOT_DOCUMENT :
                       cp="loads slowly";
                       break;
                  case OWRX_CLOSED_ROOT_DOCUMENT :
                       cp="doesn't load or run javascript";
                       // example: C:\SpecLab_logfiles\2020_06_16_HttpServerLog_IP45_159_148_253_visited_but_didnt_load_Javascript.txt
                       break;
                  case OWRX_OPENED_WEB_SOCKET    :
                       cp="opened WS but doesn't say hello";
                       break;
                }
               if( pOWClient->pCwNet->cfg.iDiagnosticFlags & (CWNET_DIAG_FLAGS_VERBOSE | CWNET_DIAG_FLAGS_SHOW_CONN_LOG) )
                { ShowError( ERROR_CLASS_INFO | SHOW_ERROR_TIMESTAMP,
                     "OWRX: Client #%d with ip=%s %s (since %d seconds).",
                     pOWClient->iClientIndex,
                     CwNet_IPv4AddressToString( (BYTE*)&pOWClient->dwClientIP ),
                     cp, (int)(pOWClient->nMillisecondsWaited/1000) );
                }
               if( pOWClient->nMillisecondsWaited > 60000 ) // waited "too long"..
                { OWRX_SetState( pOWClient, OWRX_STATE_PASSIVE );
                  if( pOWClient->pHttpInstance != NULL )
                   { HttpSrv_SendErrorAndCloseConnection(
                       pOWClient->pHttpInstance->pCwNet,
                       pOWClient->pHttpInstance->pClient,
                       pOWClient->pHttpInstance->sz255RequestedURLWithoutQuery,
                       HTTP_STATUS__SRV_TIMEOUT );  // not sure if this is the right error code, anyway..
                     // Caution, the "Http Instance" is INVALID after the above call,
                     pOWClient->pHttpInstance = NULL; // so forget this address
                   }
                  OWRX_UpdateNumClientsConnected(); // OWRX_Client[].iState -> OWRX_Server_nClientsConnected
                } // end if < waited "too long" >
            }
           break;
        case OWRX_ACTIVE_WEB_SOCKET    : // web socket sending audio and spectra
           nMilliseconds = TIM_ReadStopwatch_ms( &pOWClient->stopwatch_for_timeouts );
           if( nMilliseconds > 10000 )   // don't assume the "stopwatch" can time-out HOURS..
            { pOWClient->nMillisecondsWaited += nMilliseconds; // .. so accumulate in 10-second-steps
              TIM_StartStopwatch( &pOWClient->stopwatch_for_timeouts );
            }
           if( pOWClient->nMillisecondsWaited >= (60*1000) )
            { ++pOWClient->nMinutesWithoutUserEvents;
              pOWClient->nMillisecondsWaited = 0;
              if( pOWClient->nMinutesWithoutUserEvents >= 60 )
               { // Visitor has enjoyed our receiver for an hour, without
                 //  ever QSY'ing or clicking one of those buttons in the GUI.
                 if( (pOWClient->nMinutesWithoutUserEvents % 10) == 0 )
                  { if( pOWClient->pCwNet->cfg.iDiagnosticFlags & (CWNET_DIAG_FLAGS_VERBOSE | CWNET_DIAG_FLAGS_SHOW_CONN_LOG) )
                     { ShowError( ERROR_CLASS_INFO | SHOW_ERROR_TIMESTAMP,
                         "OpenWebRX client #%d with ip=%s fell asleep (%d minutes online).",
                         pOWClient->iClientIndex, CwNet_IPv4AddressToString( (BYTE*)&pOWClient->dwClientIP ),
                         (int)pOWClient->nMinutesWithoutUserEvents );
                     }
                    // ToDo: Send a text message to this client
                    //       to confirm he's still there.
                  }
                 if( pOWClient->nMinutesWithoutUserEvents > 120 )
                  { // Kick him out .. but don't block him
                    OWRX_SetState( pOWClient, OWRX_STATE_PASSIVE );
                    if( pOWClient->pHttpInstance != NULL )
                     { HttpSrv_SendErrorAndCloseConnection(
                          pOWClient->pHttpInstance->pCwNet,
                          pOWClient->pHttpInstance->pClient,
                          pOWClient->pHttpInstance->sz255RequestedURLWithoutQuery,
                          HTTP_STATUS__SRV_TIMEOUT );
                       // Caution, the "Http Instance" is INVALID after the above call,
                       pOWClient->pHttpInstance = NULL; // so forget this address
                     }
                    OWRX_UpdateNumClientsConnected(); // OWRX_Client[].iState -> OWRX_Server_nClientsConnected
                  } // end if < no signs of life from this user since xxx minutes >
               }   // end if( pOWClient->nMinutesWithoutUserEvents >= 60 ) [time to send warnings periodically]
            }     // end if( pOWClient->nMillisecondsWaited >= (60*1000) )
           break;
        case OWRX_CLOSED_WEB_SOCKET    : // client closed the web socket ...
           if( TIM_ReadStopwatch_ms( &pOWClient->stopwatch_for_timeouts ) > 10000 )
            {  // ... no attempt to re-establish the Web Socket since 10 seconds.
               // Guess this client really "went QRT" and we recycle
               // this instance for someone else, as follows:
               if( pOWClient->pCwNet->cfg.iDiagnosticFlags & (CWNET_DIAG_FLAGS_VERBOSE | CWNET_DIAG_FLAGS_SHOW_CONN_LOG) )
                { ShowError( ERROR_CLASS_INFO | SHOW_ERROR_TIMESTAMP,
                     "OWRX: Client #%d with ip=%s didn't reconnect within 10 seconds.",
                     pOWClient->iClientIndex,
                     CwNet_IPv4AddressToString( (BYTE*)&pOWClient->dwClientIP ) );
                }
               OWRX_SetState(pOWClient,OWRX_STATE_PASSIVE);
               OWRX_UpdateNumClientsConnected(); // OWRX_Client[].iState -> OWRX_Server_nClientsConnected
            }
        default : // no need to check for timeouts in other states
           break;
      } // end switch( pOWClient->iState )
   } // end for < all clients that MAY currently be connected >

} // end OWRX_CheckForTimeouts()


#if(0)
// We will pad the FFT at the beginning, with the first value of the input data, COMPRESS_FFT_PAD_N times.
// No, this is not advanced DSP, just the ADPCM codec produces some gabarge samples at the beginning,
// so we just add data to become garbage and get skipped.
// COMPRESS_FFT_PAD_N should be even.

    if(!strcmp(argv[1],"compress_fft_adpcm_f_u8"))
    {
        if(argc<=2) return badsyntax("need required parameters (fft_size)");
        int fft_size;
        sscanf(argv[2],"%d",&fft_size);
        int real_data_size=fft_size+COMPRESS_FFT_PAD_N;
        if(!getbufsize()) return -2; //dummy
        sendbufsize(real_data_size);
        float* input_buffer_cwa = (float*)malloc(sizeof(float)*real_data_size);
        short* temp_buffer_cwa = (short*)malloc(sizeof(short)*real_data_size);
        unsigned char* output_buffer_cwa = (unsigned char*)malloc(sizeof(unsigned char)*(real_data_size/2));
        ima_adpcm_state_t d;
        d.index=d.previousValue=0;
        for(;;)
        {
            FEOF_CHECK;
            fread(input_buffer_cwa+COMPRESS_FFT_PAD_N, sizeof(float), fft_size, stdin);
            for(int i=0;i<COMPRESS_FFT_PAD_N;i++) input_buffer_cwa[i]=input_buffer_cwa[COMPRESS_FFT_PAD_N]; //do padding
            for(int i=0;i<real_data_size;i++) temp_buffer_cwa[i]=input_buffer_cwa[i]*100; //convert float dB values to short
            encode_ima_adpcm_i16_u8(temp_buffer_cwa, output_buffer_cwa, real_data_size, d); //we always return to original d at any new buffer
            fwrite(output_buffer_cwa, sizeof(unsigned char), real_data_size/2, stdout);
            TRY_YIELD;    // "try yield" ? No, we brake for nobody.
        }
    }
#endif // (0)

#endif // SWI_USE_HTTP_SERVER ?


/* EOF < C:\cbproj\Remote_CW_Keyer\OpenWebRX_Server.c > */


