//---------------------------------------------------------------------------
// File: C:\cbproj\Remote_CW_Keyer\Keyer_Main.cpp
// Date: 2023-11-11
// Author: Wolfgang Buescher (DL4YHF)
// Purpose: Main "form" (window) for an experiment to key certain Icom transceivers.
//          Details about the "keying" itself in KeyerThread.c .
//      THIS module is just the BORing graphic user interface,
//      written in a stoneage version of Borland's C++ Builder (BCB V6),
//      using their easy-to-use VCL (Visual Component Library).
//---------------------------------------------------------------------------

#include "switches.h"   // project specific compiler switches ("options"),
                        // must be included before anything else !
#include "yhf_type.h"   // classic types like BYTE, WORD, DWORD, BOOL, ..
#include <windows.h>    // Must be included BEFORE vcl.h for some strange reason.
                        // Contains stuff like WAVEINCAPS, WAVEOUTCAPS, etc^255 .
#include <vcl.h>        // Borland's stoneage Visual Component Library
#include <buttons.hpp>  // TBitBtn       a la Borland VCL (but why in an extra header?)
#include <ComCtrls.hpp> // TTabSheet     a la Borland VCL ...
#include <CheckLst.hpp> // TCheckListBox a la Borland VCL ....
#include <Grids.hpp>    // TStringGrid   a la Borland VCL .....
#include <string.h>
#include <stdio.h>      // not using "standard I/O" here, but e.g. sprintf()
#include <math.h>
#pragma hdrstop

#include "Utilities.h"   // stuff like UTL_iWindowsVersion, UTL_iAppInstance, ShowError(), etc
#include "YHF_Dialogs.h" // API for a few common dialogs, e.g. YHF_RunStringEditDialog()
#include "YHF_Help.h"    // DL4YHF's HTML-based replacement for the defunct *.hlp system
#include "HelpIDs.h"     // application specific help topic identifiers
#include "Translator.h"  // APPL_TranslateAllForms() [only works with Borland VCL]
#include "Translations.h" // tranlation table (static, built-in string table w/o whistles and bells)

#if( SWI_USE_DSOUND ) // use "Direct Sound" / dsound_wrapper.c ?
# include "dsound_wrapper.h"
#endif // SWI_USE_DSOUND ?

#if( SWI_USE_WAVE_AUDIO || SWI_USE_MIDI )
# include <mmsystem.h> // maybe MIDI is fast enough to start / stope playing notes ?
#endif // SWI_USE_WAVE_AUDIO or SWI_USE_MIDI ?

#include "StringLib.h"
#include "Elbug.h"   // old 'Elbug' functions, converted from PIC-assembler to "C"
#include "CwGen.h"   // CW generator (converts text to Morse code)
#include "CwDSP.h"   // CW-'Digital Signal Processor' / sidetone generator
#include "CwNet.h"   // Socket-based 'Client or Server' for the Remote CW Keyer
#include "CwKeyer.h" // prototypes for the 'worker thread' that stitches all together
#include "Inet_Tools.h" // Base64 encoding/decoding, SHA1 calculation, etc..
#if(SWI_USE_HTTP_SERVER) // build an application with integrated HTTP server ?
# include "HttpServer.h" // formerly simple HTTP server (turned into a monster)
# include "TIconToFavicon.h" // function to convert the application's icon
                             // into "Favicon.ico" for the web server .
#endif // SWI_USE_HTTP_SERVER ?

#include "SpecDisp.h" // SpecDisp_UpdateSpectrum(), ..Waterfall(), FreqScale(),..

#include "Keyer_Main.h"
//---------------------------------------------------------------------------
#pragma package(smart_init)
#pragma resource "*.dfm"
#pragma warn -8004 // ".. assigned a value that is never used" (better than using an un-initialized variable)
TKeyerMainForm *KeyerMainForm;

HINSTANCE APPL_hInstance= 0; // handle to current instance, a Win32 API thing
BOOL APPL_fLaunchKeyerOnStart = FALSE;
BOOL APPL_fPausedDuringSleep  = FALSE;
BOOL APPL_fSaveSettingsOnExit = FALSE;

char g_sz255CommandLine[256]     = "";
char g_sz255PathToDataFiles[256] = "";

// Colours for various GUI components that we'd like to control via SetColourScheme():
// (but convincing Windows and the VCL to actually USE these colours turned into
//  a nightmare; especially for stuff in the 'Non-Client Area', e.g. title and main menu)
TColor g_clMenuBackground, g_clMenuSelBackgnd, g_clMenuForeground;
TColor g_clDecoderOutputForeground, g_clDecoderOutputBackground;
int    g_iNewLanguage = TRANSLATOR_LANGUAGE_ENGLISH;
#define UI_MODE_SIMPLE   0 // only show the basic elements, menu items, etc in the UI
#define UI_MODE_COMPLETE 1 // show everything, including the stuff for debugging, SDR-like rig control, etc
int    g_iUserInterfaceMode = UI_MODE_COMPLETE;

volatile DWORD g_dwDebugMessage,  // kluge for debugging / dummy assignments for breakpoints..
               g_dwDebugWParam, g_dwDebugLParam;
volatile int   g_iDebugDummyIndex;
int g_iTimingScopeBitmapWidth = 0;  // <- only for diagnostic / trouble with "destroyed forms"

static int s_iBandStackingRegIndices[32/*band*/][3/*stack-index*/];
           // '--> filled in FillComboWithBandList(), used in CB_BandClick()


typedef struct t_ErrorFifoEntry
{
  char  sz511Text[512];
  int   iErrorClass;     // "error class" -> background colour in the RichText display
} T_ErrorFifoEntry;
# define C_ERROR_FIFO_SIZE 512

T_CwNet MyCwNet; // <- not a shiny C++ class, but a single instance representing the "CW Network"
DSoundWrapper MyDirectSound; // a "DirectSound wrapper" instance, first used for the SIDETONE OUTPUT

#if( SWI_USE_HAMLIB_SERVER ) // build an application with integrated "Hamlib Net rigctld"-compatible server ?
T_HLSrv MyHamlibServer; // Hamlib-'rigctld'-compatible server (guess the 'd' means 'daemon',
       // but neither demons nor daemons lurking in here. There's NOTHING running 'in the background'.) 
#endif // SWI_USE_HAMLIB_SERVER ?


T_ErrorFifoEntry ErrorHistoryFifo[ C_ERROR_FIFO_SIZE ]; // <- classic lock-free circular FIFO between ShowError() and the GUI
int DEBUG_iErrorHistoryHead = 0;
int DEBUG_iErrorHistoryTail = 0;

typedef struct t_TimingScopeOverlay
{ int x1, y1, x2, y2; // mouse pointer coordinates at "mouse down" and "mouse up"
  int t1_ms;          // relative time in milliseconds (0=left edge) at "mouse down"
  int t2_ms;          // relative time in milliseconds (0=left edge) at "mouse up"
  BOOL visible;
} T_TimingScopeOverlay;
#define N_SCOPE_OVERLAYS 3 // allow showing up to THREE of these overlays simultaneously
T_TimingScopeOverlay TimingScopeOverlay[N_SCOPE_OVERLAYS];
int TimingScope_iCurrentOverlay = 0;
int TimingScope_iUpdateCountOnTestTab, TimingScope_iUpdateCountOnKeyerTab;

T_SpecDispControl g_SpecDispControl = { 0 };

//----------------------------------------------------------------------------
// Global variables for hardcore debugging / post-mortem crash analysis.
//        Useful when the debugger's call stack shows nothing but garbage,
//        e.g. after an exception at the infamous address 0xFEEEFEEE .
//        ( C++Builder's debugger could still inspect SIMPLE GLOBAL
//          VARIABLES after most kinds of exceptions, but of course nothing
//          stack-based because the CPU- or task-stack was usually trashed.)
//----------------------------------------------------------------------------
#if(SWI_HARDCORE_DEBUGGING) // (1) = hardcore-debugging, (0)=normal compilation
 int GUI_iLastSourceLine = 0; // WATCH THIS after crashing with e.g. "0xFEEEFEEE"  ...
# define HERE_I_AM__GUI()  GUI_iLastSourceLine=__LINE__
# if( 0 )
    If certain worker threads of the application don't seem to terminate
    "politely", watch the following (complete list only HERE, in Keyer_Main.cpp):
     * GUI_iLastSourceLine
     * DSW_iLastSourceLine
     * DSP_iLastSourceLine
     * Keyer_iLastSourceLine
     * CwNet_iLastSourceLine
     * HttpSrv_iLastSourceLine
     * RigCtrl_iLastSourceLine
#  endif // (0) .. kludge to inspect the above variables in Borland via mouse-over ...
  extern int DSW_iLastSourceLine, DSP_iLastSourceLine, Keyer_iLastSourceLine,
             CwNet_iLastSourceLine, HttpSrv_iLastSourceLine;
  CPROT void CheckSystemHealth(const char *pszModuleName, int iSourceLine); // <- periodically called via macro CHECK_SYSTEM_HEALTH()
  static  const char *s_ModuleName = "KeyerMain";
# define CHECK_SYSTEM_HEALTH() CheckSystemHealth(s_ModuleName,__LINE__)
#else
# define HERE_I_AM__GUI()      /* rien */
# define CHECK_SYSTEM_HEALTH() /* nada */
#endif // SWI_HARDCORE_DEBUGGING ?

const T_SL_TokenList C_TL_None_Zero[] =
{ { "NONE",  0 },
  { NULL, 0 } // "all zeros" mark the end of the list
};

const T_SL_TokenList SerialPortOutputSignals_Key[] =
{ { "DTR (DE-9 pin 4) on KEYER port",            KEYER_SIGNAL_INDEX_MORSE_KEY_DTR },
  { "DTR (DE-9 pin 4) on KEYER port, INVERTED", -KEYER_SIGNAL_INDEX_MORSE_KEY_DTR },
  { "RTS (DE-9 pin 7) on KEYER port",            KEYER_SIGNAL_INDEX_MORSE_KEY_RTS },
  { "RTS (DE-9 pin 7) on KEYER port, INVERTED", -KEYER_SIGNAL_INDEX_MORSE_KEY_RTS },
  { NULL, 0 } // "all zeros" mark the end of the list
};

const T_SL_TokenList SerialPortOutputSignals_Radio[] =
{ { "DTR (DE-9 pin 4) on RADIO port",            KEYER_SIGNAL_INDEX_RADIO_KEYING_DTR },
  { "DTR (DE-9 pin 4) on RADIO port, INVERTED", -KEYER_SIGNAL_INDEX_RADIO_KEYING_DTR },
  { "RTS (DE-9 pin 7) on RADIO port",            KEYER_SIGNAL_INDEX_RADIO_KEYING_RTS },
  { "RTS (DE-9 pin 7) on RADIO port, INVERTED", -KEYER_SIGNAL_INDEX_RADIO_KEYING_RTS },
  { NULL, 0 } // "all zeros" mark the end of the list
};

const T_SL_TokenList SerialPortInputSignals_Key[] =
{ { "DCD (DE-9 pin 1) on KEYER port",            KEYER_SIGNAL_INDEX_MORSE_KEY_DCD },
  { "DCD (DE-9 pin 1) on KEYER port, INVERTED", -KEYER_SIGNAL_INDEX_MORSE_KEY_DCD },
  { "DSR (DE-9 pin 6) on KEYER port",            KEYER_SIGNAL_INDEX_MORSE_KEY_DSR },
  { "DSR (DE-9 pin 6) on KEYER port, INVERTED", -KEYER_SIGNAL_INDEX_MORSE_KEY_DSR },
  { "CTS (DE-9 pin 8) on KEYER port",            KEYER_SIGNAL_INDEX_MORSE_KEY_CTS },
  { "CTS (DE-9 pin 8) on KEYER port, INVERTED", -KEYER_SIGNAL_INDEX_MORSE_KEY_CTS },
  { "RI  (DE-9 pin 9) on KEYER port",            KEYER_SIGNAL_INDEX_MORSE_KEY_RI  },
  { "RI  (DE-9 pin 9) on KEYER port, INVERTED", -KEYER_SIGNAL_INDEX_MORSE_KEY_RI  },
  { NULL, 0 } // "all zeros" mark the end of the list
};

const T_SL_TokenList SerialPortInputSignals_Radio[] =
{ { "DCD (DE-9 pin 1) on RADIO port",            KEYER_SIGNAL_INDEX_RADIO_KEYING_DCD },
  { "DCD (DE-9 pin 1) on RADIO port, INVERTED", -KEYER_SIGNAL_INDEX_RADIO_KEYING_DCD },
  { "DSR (DE-9 pin 6) on RADIO port",            KEYER_SIGNAL_INDEX_RADIO_KEYING_DSR },
  { "DSR (DE-9 pin 6) on RADIO port, INVERTED", -KEYER_SIGNAL_INDEX_RADIO_KEYING_DSR },
  { "CTS (DE-9 pin 8) on RADIO port",            KEYER_SIGNAL_INDEX_RADIO_KEYING_CTS },
  { "CTS (DE-9 pin 8) on RADIO port, INVERTED", -KEYER_SIGNAL_INDEX_RADIO_KEYING_CTS },
  { "RI  (DE-9 pin 9) on RADIO port",            KEYER_SIGNAL_INDEX_RADIO_KEYING_RI  },
  { "RI  (DE-9 pin 9) on RADIO port, INVERTED", -KEYER_SIGNAL_INDEX_RADIO_KEYING_RI  },
  { NULL, 0 } // "all zeros" mark the end of the list
};

const T_SL_TokenList RigControlMethods[] =
{ { "No rig control, only CW keying (or client mode)", RIGCTRL_PROTOCOL_NONE },
  { "Icom CI-V, using the 'COM port for CW keying'", RIGCTRL_PROTOCOL_ICOM_CI_V },
  { "Hamlib Net rigctl via remote TCP/IP server", RIGCTRL_PROTOCOL_HAMLIB_RIGCTLD  },
  { NULL, 0 } // "all zeros" mark the end of the list
};

const T_SL_TokenList KeyTypes[] =
{ { "None (no CW key)", KEY_TYPE_PASSIVE  },
  { "Straight",         KEY_TYPE_STRAIGHT },
  { "Paddle, IAMBIC MODE B", KEY_TYPE_IAMBIC_B },
  { "Paddle, IAMBIC MODE A", KEY_TYPE_IAMBIC_A },
  { "Basic iambic w/o dash/dot memory", KEY_TYPE_IAMBIC_NO_MEM},
  { NULL, 0 } // "all zeros" mark the end of the list
};

const T_SL_TokenList SidetonesOnTXD[] =  // Sidetone tapped from a serial port's TXD(!) output:
{ { "None",   KEYER_SIDETONE_NONE       },
  { "480 Hz", KEYER_SIDETONE_TXD_480_HZ }, // 4800 "baud" divided by 10 bits (standard baudrate)
  { "960 Hz", KEYER_SIDETONE_TXD_960_HZ }, // also 4800 "baud" but with TWO pulses in the TXD pattern
  { NULL, 0 } // "all zeros" mark the end of the list
};

const T_SL_TokenList SidetonesOnAudioOut[] =  // Sidetones sent to an audio device (with higher latency than "TXD sidetone"):
{ { "None",   0   },
  { "400 Hz", 400 },
  { "450 Hz", 450 }, // ~~ Icom's name "CW Pitch" (but here, it's only the LOCAL KEYER SIDETONE)
  { "500 Hz", 450 },
  { "550 Hz", 550 },
  { "600 Hz", 600 },
  { "650 Hz", 650 },
  { "700 Hz", 700 },
  { "750 Hz", 750 },
  { "800 Hz", 800 },
  { "1000 Hz",1000},
  { NULL, 0 } // "all zeros" mark the end of the list
};

const T_SL_TokenList SidetoneRiseTimes[] =  // "Rise"-, aka "Ramp"-times for sidetone VIA AUDIO OUTPUT
{ { "0 ms (hard-keyed)", 0 },
  { "2 ms",  2 },
  { "4 ms",  4 },
  { "6 ms",  6 },
  { "8 ms",  8 },
  { "20 ms (very soft)",20 }, // huuh-huh, thought Mr Bubo Bubo, and flew away when he heard this
  { NULL, 0 } // "all zeros" mark the end of the list
};


const char *CwKeyer_psz6MemoryDefaults[6] =
{ "Remote CW Keyer. Press F1 to send this message,",
  "Testing remote CW keying. Press F2 to send this message.",
  "Message Nr 3. Click into these fields to edit.",
  "For details, please read the friendly manual.",
  "",
  ""
};


//---------------------------------------------------------------------------
// Helper functions ( no shiny VCL class methods ! )
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
void ParseCommandLine(const char *pszCmdLine)
  // Some examples (copy & paste to CBuilder IDE: Run..Parameters)
{
  const char *cp = pszCmdLine;
  const char *cp2;
  char *cp3;
  char c;
  char sz255Temp[256], *pszDest=sz255Temp, *pszEndstop=sz255Temp+255;

  if( *cp=='"' ) // guess this is the name of the executable -> grab 'our path' from it
   { cp2 = cp+1;
     SL_SkipDoubleQuotedString( &cp );  // <- returns the NUMBER OF CHARS SKIPPED in cp, *including* the double quotes
   }
  else  // Name of the executable NOT in double quotes (thanks for the snag..)
   { cp2 = cp;
     SL_SkipCharsUntilDelimiter( &cp, " ", SL_SKIP_NORMAL );
   }

  // One of the 'nice surprises' from Windoze is that the first parameter
  // on the command line sometimes does NOT contain a full path,
  // To avoid having the configuration files, the "debug run log",
  // and who-knows-what somewhere in the dark where Microsoft
  // wants to have them ( e.g. C:\Users\WerHatsInstalliert\AppData\Local | LocalLow | Roaming | Weiss-der-Teufel ..),
  // extract the full path to the EXECUTABLE (which is where all other files
  // have been unzipped to, without a shiny installer at all) :
  //
  if( ! UTL_GetPathToMyDataFiles( g_sz255PathToDataFiles, 255 ) )
   { // SOS .. unable to find out where "our files" should be for this user,
     //        on this windows version ..
     cp3 = strrchr( g_sz255PathToDataFiles, '\\' );
     if( cp3==NULL )  // quite unlikely that Microsoft uses POSIX one day, but... :
      {  cp3 = strrchr( g_sz255PathToDataFiles, '/' );
      }
     if(cp3!=NULL)  // if there's any kind of slash, TRUNCATE AFTER THAT CHAR..
      { cp3[1] = '\0'; // -> result e.g. g_sz255PathToDataFiles = "C:\\cbproj\Remote_CW_Keyer\\"
      }
   }
  if( ((c=SL_GetLastChar(g_sz255PathToDataFiles,255)) != '\\') && (c!='/') )
   { cp3 = g_sz255PathToDataFiles;
     SL_AppendChar( &cp3, cp2+255, '\\' );
   }

  // After the above alternative for "the first command line argument", parse the rest..
  while( *cp != '\0' ) // repeat until end-of-string
   { SL_SkipSpaces( &cp );
     switch( *cp )
      { case '"' :  // another double-quoted string ? -> throw it away
           if( SL_SkipDoubleQuotedString( &cp ) <= 0 )
            { return;  // something wrong with the syntax -> bail out
            }
           break;
        case '\\' :
        case '/'  : // forward or backward slash : Guess this is an "option"..
           ++cp;
           if( SL_SkipToken( &cp, "debug" ) ) // run this program with a "debug log" :
            { pszDest=sz255Temp;
              SL_AppendString( &pszDest, pszEndstop, g_sz255PathToDataFiles );
              SL_AppendString( &pszDest, pszEndstop, "debug_run_log.txt" );
              UTL_OpenRunLogFile( sz255Temp );
            }
           else if( SL_SkipToken( &cp, "xlate" ) ) // run this program in "translation test mode" ..
            { APPL_iLanguageTestMode = 1; // details in Remote_CW_Keyer\Translator.cpp !
            }
           break;
        default:
           return;  // something wrong with the syntax -> bail out
      }
   } // end while < more characters or substrings in the command line >

} // end ParseCommandLine()

//---------------------------------------------------------------------------
void LoadSettings(void) // .. from an old-fashioned *.ini file, NOT from the bloody registry !
{ // [out] CwKeyer_Config , and maybe A FEW more things for the GUI itself.
  char *pszIniFileName = SWI_APP_EXE_NAME".ini"; // all instances use THE SAME ini file..
        // -> The full path MAY be C:\Windows\RemoteCwKeyer.ini ,
        //    but also seen :
        //  C:\Users\Wolf\AppData\Local\VirtualStore\Windows\RemoteCwKeyer.ini
        //   - seems to depend on FROM WHERE, and with which privileges,
        //     and by which USER the application was launched. Omg.
  char szSec[84];  // "section" (in the ini file, that's the string in squared brackets)
  char szKey[84];  // "key name" (that's the string on the left side of the assignment operator)
  char sz255Value[256];
  TEdit *pEdit;
  int   i;

  CwKeyer_SetDefaultConfig( &CwKeyer_Config );
  CwKeyer_InitTimingScope( &CwKeyer_TimingScope  );
  Elbug_InitInstanceWithDefaults( &CwKeyer_Elbug );
  CwGen_InitInstanceWithDefaults( &CwKeyer_Gen   );
  CwDSP_InitInstanceWithDefaults( &CwKeyer_DSP   );

  CwNet_InitInstanceWithDefaults( &MyCwNet );
  CwKeyer_Config.pCwNet = &MyCwNet; // 'marry' the keyer to the 'CW Network' instance
  MyCwNet.pCwDSP = &CwKeyer_DSP; // 'marry' the audio DSP to the 'CW Network' instance (for audio streaming in both directions)
     // '--> Sometimes, after STEPPING OVER the above instruction, MyCwNet.pCwDSP was still NULL !
     //   Hard to believe but true : Problem fixed by doing a BUILD ALL !
     //   Obviously the ancient IDE (Borland C++ Builder) lost track of header dependencies.

#if( SWI_USE_DSOUND ) // use "DirectSound" / dsound_wrapper.c ?
  CwKeyer_DSP.pDSW = &MyDirectSound; // let the 'DSP' use our DirectSound-wrapper-instance
  // (contains a list of audio-in- and -output devices already 'enumerated')
  if( CwKeyer_DSP.cfg.sz255AudioInputDevice[0] == '\0' )
   { strcpy( CwKeyer_DSP.cfg.sz255AudioInputDevice, "None" );
   }
  if( CwKeyer_DSP.cfg.sz255AudioOutputDevice[0] == '\0' )
   { // Obviously not set yet -> pick an audio output device for 'DirectSound'
     // that usually exists: THE FIRST ONE enumerated by our wrapper !
     if( MyDirectSound.m_nOutputDevices > 0 )
      { strncpy( CwKeyer_DSP.cfg.sz255AudioOutputDevice,
                 MyDirectSound.m_sOutputDevices[0].sz255DevName, 255 );
        // ,-----|_____________________________________________|
        // '--> e.g. "Primary Sound Driver", but ymmv ..
      }
   }
#endif // SWI_USE_DSOUND ?

  sprintf( szSec, "Keyer%d", 1+UTL_iAppInstance ); // e.g. "Keyer1" for the FIRST running instance, etc
  CwKeyer_Config.iComPortNumber_IN = UTL_ReadIntFromIniFile(pszIniFileName,
      szSec, "ComPort_IN"/*pszKeyName*/, CwKeyer_Config.iComPortNumber_IN/*default*/ );
  CwKeyer_Config.iRadioKeyingAndControlPort = UTL_ReadIntFromIniFile(pszIniFileName,
      szSec, "RadioControlAndKeyingPort", CwKeyer_Config.iRadioKeyingAndControlPort );
  CwKeyer_Config.iRadioControlProtocol = UTL_ReadIntFromIniFile(pszIniFileName,
      szSec, "RadioControlProtocol", CwKeyer_Config.iRadioControlProtocol );
  CwKeyer_Config.iRadioControlBaudrate = UTL_ReadIntFromIniFile(pszIniFileName,
      szSec, "RadioControlBaudrate", CwKeyer_Config.iRadioControlBaudrate );
  CwKeyer_Config.iRadioCIVAddress = UTL_ReadIntFromIniFile(pszIniFileName,
      szSec, "RadioCIVAddress", CwKeyer_Config.iRadioCIVAddress );
  UTL_ReadStringFromIniFile( pszIniFileName, szSec, "RemoteRigCtrlServerAddress",
                             CwKeyer_Config.sz80RemoteRigCtrlServerAddress/*Default*/,
                             CwKeyer_Config.sz80RemoteRigCtrlServerAddress, 80 );
#if( SWI_USE_HAMLIB_SERVER ) // not sure if this 'feature' is going to stay...
  CwKeyer_Config.fEnableBuiltInHamlibServer = UTL_ReadIntFromIniFile(pszIniFileName,
      szSec, "EnableBuiltInHamlibServer",CwKeyer_Config.fEnableBuiltInHamlibServer);
  MyHamlibServer.cfg.iServerListeningPort = UTL_ReadIntFromIniFile(pszIniFileName,
      szSec, "BuiltInHamlibServerPort", MyHamlibServer.cfg.iServerListeningPort);
  MyHamlibServer.cfg.iServerOptions = UTL_ReadIntFromIniFile(pszIniFileName,
      szSec, "BuiltInHamlibServerOptions", MyHamlibServer.cfg.iServerOptions);
  MyHamlibServer.cfg.iDiagnosticFlags= UTL_ReadIntFromIniFile(pszIniFileName,
      szSec, "BuiltInHamlibServerDiagnostics", MyHamlibServer.cfg.iDiagnosticFlags);
#endif // SWI_USE_HAMLIB_SERVER ?

  CwKeyer_Config.iMorseKeyType=UTL_ReadIntFromIniFile(pszIniFileName,szSec, "MorseKeyType",
      CwKeyer_Config.iMorseKeyType/* KEY_TYPE_PASSIVE/STRAIGHT/PADDLE */ );
  CwKeyer_Config.iDotInput  = UTL_ReadIntFromIniFile(pszIniFileName,szSec,"DotInputSignal", CwKeyer_Config.iDotInput );
  CwKeyer_Config.iDashInput = UTL_ReadIntFromIniFile(pszIniFileName,szSec,"DashInputSignal",CwKeyer_Config.iDashInput );
  CwKeyer_Config.iManualPTTInput=UTL_ReadIntFromIniFile(pszIniFileName,szSec,"ManualPTTInput",CwKeyer_Config.iManualPTTInput);
  CwKeyer_Config.iTestInput = UTL_ReadIntFromIniFile(pszIniFileName,szSec,"TestInputSignal",CwKeyer_Config.iTestInput );
  CwKeyer_Config.iKeySupply = UTL_ReadIntFromIniFile(pszIniFileName,szSec,"KeySupplySignal",CwKeyer_Config.iKeySupply );
  CwKeyer_Config.iSidetoneOnTXD  = UTL_ReadIntFromIniFile(pszIniFileName,szSec,"SidetoneOnTXD", CwKeyer_Config.iSidetoneOnTXD );
  CwKeyer_Config.iRadioCWKeying  = UTL_ReadIntFromIniFile(pszIniFileName,szSec,"CwKeyingSignal",CwKeyer_Config.iRadioCWKeying );
  CwKeyer_Config.iRadioPTTControl= UTL_ReadIntFromIniFile(pszIniFileName,szSec,"PTTControlSignal",CwKeyer_Config.iRadioPTTControl);
  CwKeyer_Config.iTxDelayTime_ms = UTL_ReadIntFromIniFile(pszIniFileName,szSec,"TxDelayTime_ms",CwKeyer_Config.iTxDelayTime_ms);
  CwKeyer_Config.iTxHangTime_ms  = UTL_ReadIntFromIniFile(pszIniFileName,szSec,"TxHangTime_ms",CwKeyer_Config.iTxHangTime_ms);
  CwKeyer_Config.fDebouncePaddleInputs=UTL_ReadIntFromIniFile(pszIniFileName,szSec,"DebouncePaddleInputs",CwKeyer_Config.fDebouncePaddleInputs);
  CwKeyer_Config.fKeyInputWatchdog=UTL_ReadIntFromIniFile(pszIniFileName,szSec,"KeyInputWatchdog",CwKeyer_Config.fKeyInputWatchdog);
  CwKeyer_Config.fKeyViaShiftAndControl = UTL_ReadIntFromIniFile(pszIniFileName,szSec,"CwViaShiftAndControlKey",
    ( CwKeyer_Config.iComPortNumber_IN < 0) ); // <- per default, only enable this "gadget" if there's no serial port for the Morse Key
  UTL_ReadStringFromIniFile( pszIniFileName, szSec, "AudioInputDevice",
                             CwKeyer_DSP.cfg.sz255AudioInputDevice/*Default*/,
                             CwKeyer_DSP.cfg.sz255AudioInputDevice, 255 );
  UTL_ReadStringFromIniFile( pszIniFileName, szSec, "AudioOutputDevice",
                             CwKeyer_DSP.cfg.sz255AudioOutputDevice/*Default*/,
                             CwKeyer_DSP.cfg.sz255AudioOutputDevice, 255 );
  CwKeyer_DSP.cfg.iSidetoneFreq_Hz = UTL_ReadIntFromIniFile(pszIniFileName,szSec,
         "AudioSidetone_Hz", 650 + 50 * UTL_iAppInstance/*gadget: try running THREE instances..*/ );
  CwKeyer_DSP.cfg.iSidetoneRiseTime_ms = UTL_ReadIntFromIniFile(pszIniFileName,szSec,
         "SidetoneRiseTime_ms", CwKeyer_DSP.cfg.iSidetoneRiseTime_ms );
  CwKeyer_DSP.cfg.iSidetoneGain_dB = UTL_ReadIntFromIniFile(pszIniFileName,szSec,
         "SidetoneGain_dB", CwKeyer_DSP.cfg.iSidetoneGain_dB );
  CwKeyer_DSP.cfg.iAudioInGain_dB = UTL_ReadIntFromIniFile(pszIniFileName,szSec,
         "AudioInGain_dB", CwKeyer_DSP.cfg.iAudioInGain_dB );
  CwKeyer_DSP.cfg.iAudioOutGain_dB = UTL_ReadIntFromIniFile(pszIniFileName,szSec,
         "AudioOutGain_dB", CwKeyer_DSP.cfg.iAudioOutGain_dB );
  CwKeyer_DSP.cfg.iNetworkTonesGain_dB = UTL_ReadIntFromIniFile(pszIniFileName,szSec,
         "NetworkTonesGain_dB", CwKeyer_DSP.cfg.iNetworkTonesGain_dB );

  CwKeyer_DSP.cfg.iAudioFlags = UTL_ReadIntFromIniFile(pszIniFileName, szSec,
          "AudioFlags"/*pszKeyName*/, CwKeyer_DSP.cfg.iAudioFlags/*default*/ );

  CwKeyer_Elbug.cfg.iDotTime_ms = Elbug_WordsPerMinuteToDotTimeInMilliseconds(UTL_ReadIntFromIniFile(pszIniFileName,szSec,"WordsPerMinute",25) );
  CwKeyer_Gen.cfg.iDotTime_ms = CwKeyer_Elbug.cfg.iDotTime_ms; // let 'playback' use the same speed as direct keyed Morse
  APPL_fLaunchKeyerOnStart  = UTL_ReadIntFromIniFile(pszIniFileName,szSec,"LaunchOnStart", APPL_fLaunchKeyerOnStart );
  CwKeyer_TimingScope.cfg.iChannel4Source = UTL_ReadIntFromIniFile(pszIniFileName,szSec,"Channel4Source", CwKeyer_TimingScope.cfg.iChannel4Source );
  CwKeyer_TimingScope.cfg.iVisibleChannels= UTL_ReadIntFromIniFile(pszIniFileName,szSec,"TScopeChannels", CwKeyer_TimingScope.cfg.iVisibleChannels);
  CwKeyer_TimingScope.cfg.iTriggerOptions = UTL_ReadIntFromIniFile(pszIniFileName,szSec,"TScopeTrigger",  CwKeyer_TimingScope.cfg.iTriggerOptions );

  // Still in the ini-file-section "KeyerN", where "N" is the instance index...
  CwKeyer_Config.iTRXOptions = UTL_ReadIntFromIniFile(pszIniFileName,
       szSec, "TRXOptions"/*pszKeyName*/, CwKeyer_Config.iTRXOptions/*default*/ );
       // -> bitwise combineable flag like KEYER_TRX_OPT_KEEP_RUNNING, etc,
       //    all defined for struct T_CwKeyerConfig in Remote_CW_Keyer/CwKeyer.h .

  // Load options for the built-in "Rig Control" module :
  sprintf( szSec, "RigControl%d", 1+UTL_iAppInstance ); // e.g. "RigControl1" for the FIRST running instance, etc
  CwKeyer_Config.dwRejectMessages4Log = (DWORD)UTL_ReadIntFromIniFile(pszIniFileName,szSec,"RejectMessagesInLog",
                             CwKeyer_Config.dwRejectMessages4Log/*Default*/ );
  RigCtrl_iTrafficMonitorDisplayOptions = UTL_ReadIntFromIniFile(pszIniFileName,szSec,"TrafficMonitorOptions",
                             RigCtrl_iTrafficMonitorDisplayOptions );



  // Load the "Network" configuration (for module CwNet.c; simple TCP-client or server):
  sprintf( szSec, "Network%d", 1+UTL_iAppInstance ); // e.g. "Network1" for the FIRST running instance, etc
  MyCwNet.cfg.iFunctionality = UTL_ReadIntFromIniFile(pszIniFileName, szSec, "Functionality",  MyCwNet.cfg.iFunctionality );
  MyCwNet.cfg.iDiagnosticFlags= UTL_ReadIntFromIniFile(pszIniFileName,szSec,"DiagnosticFlags", 0 );
  sprintf( szSec, "Network%dClient", 1+UTL_iAppInstance ); // e.g. "Network1Client" for the FIRST running instance, etc
  UTL_ReadStringFromIniFile( pszIniFileName, szSec, "RemoteServerIP",
                             MyCwNet.cfg.sz80ClientRemoteIP/*Default*/,
                             MyCwNet.cfg.sz80ClientRemoteIP, 80 );
  UTL_ReadStringFromIniFile( pszIniFileName, szSec, "UserName",
                             MyCwNet.cfg.sz80ClientUserName/*Default*/,
                             MyCwNet.cfg.sz80ClientUserName, 80 );
  UTL_ReadStringFromIniFile( pszIniFileName, szSec, "Callsign",
                             MyCwNet.cfg.sz80ClientCallsign/*Default*/,
                             MyCwNet.cfg.sz80ClientCallsign, 80 );

  sprintf( szSec, "Network%dServer", 1+UTL_iAppInstance ); // e.g. "Network1Server" for the FIRST running instance, etc
  MyCwNet.cfg.iServerListeningPort = UTL_ReadIntFromIniFile(pszIniFileName,szSec,
                             "ListeningPort", MyCwNet.cfg.iServerListeningPort );
  UTL_ReadStringFromIniFile( pszIniFileName, szSec, "AcceptUsers",
                             MyCwNet.cfg.sz255AcceptUsers/*Default*/,
                             MyCwNet.cfg.sz255AcceptUsers, 255 );
  UTL_ReadStringFromIniFile( pszIniFileName, szSec, "Pwd",
                             MyCwNet.cfg.sz20ServerAdminPWD/*Default*/,
                             MyCwNet.cfg.sz20ServerAdminPWD, 20 );
  MyCwNet.cfg.iHttpServerOptions = UTL_ReadIntFromIniFile(pszIniFileName,szSec,
                             "HttpOptions", MyCwNet.cfg.iHttpServerOptions );
  MyCwNet.cfg.iNetworkLatency_ms = UTL_ReadIntFromIniFile(pszIniFileName,szSec,
                             "Latency_ms",  MyCwNet.cfg.iNetworkLatency_ms );

  sprintf( szSec, "Keyer%dMainWindow", 1+UTL_iAppInstance ); // e.g. "Keyer1MainWindow" for the FIRST running instance, etc
  CwKeyer_Config.iMainWindowLeft = UTL_ReadIntFromIniFile(pszIniFileName,
       szSec, "Left"/*pszKeyName*/, 50 + 100 * UTL_iAppInstance/*default*/ );
  CwKeyer_Config.iMainWindowTop = UTL_ReadIntFromIniFile(pszIniFileName,
       szSec, "Top"/*pszKeyName*/,  100 + 30 * UTL_iAppInstance/*default*/ );
  CwKeyer_Config.iMainWindowWidth = UTL_ReadIntFromIniFile(pszIniFileName,
       szSec, "Width"/*pszKeyName*/, KeyerMainForm->Width/*default*/ );
  CwKeyer_Config.iMainWindowHeight = UTL_ReadIntFromIniFile(pszIniFileName,
       szSec, "Height"/*pszKeyName*/, KeyerMainForm->Height/*default*/ );
  g_SpecDispControl.iColourScheme = UTL_ReadIntFromIniFile(pszIniFileName,
       szSec, "ColourScheme"/*pszKeyName*/, g_SpecDispControl.iColourScheme/*default*/ );
  g_iNewLanguage       = UTL_ReadIntFromIniFile(pszIniFileName, szSec, "Language", APPL_iLanguage );
  g_iUserInterfaceMode = UTL_ReadIntFromIniFile(pszIniFileName, szSec, "UserInterfaceMode", g_iUserInterfaceMode );
  RigCtrl_ITU_Region   = UTL_ReadIntFromIniFile(pszIniFileName, szSec, "ITU_Region", RigCtrl_ITU_Region );
  sprintf( szSec, "Keyer%d", 1+UTL_iAppInstance ); // e.g. back to "Keyer1" for the FIRST running instance, etc
  for(i=0; i<KEYER_NUM_MESSAGE_MEMORIES; ++i)
   { sprintf( szKey, "Memory%d", (int)(i+1) );
     UTL_ReadStringFromIniFile( pszIniFileName,
        szSec, szKey, (char*)CwKeyer_psz6MemoryDefaults[i],
        CwKeyer_Config.szKeyerMemory[i]/*pszResult*/, KEYER_MAX_CHARS_PER_MEMORY );
   } // end for < all 'keyer message memories' >

} // end LoadSettings()

//---------------------------------------------------------------------------
void SaveSettings(void) // .. in an old-fashioned *.ini file, NOT the bloody registry !
{ // [in] CwKeyer_Config .
  char *pszIniFileName = SWI_APP_EXE_NAME".ini";
  char szSec[84];  // "section" (in the ini file, that's the string in squared brackets)
  char szKey[84];  // "key name" (that's the string on the left side of the assignment operator)
  TEdit *pEdit;
  int   i;

  strcpy( szSec, "About" );
  UTL_WriteStringToIniFile(pszIniFileName,szSec,"1", "Configuration file for the 'Remote CW Keyer'.");
  UTL_WriteStringToIniFile(pszIniFileName,szSec,"2", "If you don't care about sending Morse code via PC,");
  UTL_WriteStringToIniFile(pszIniFileName,szSec,"3", "you can safely delete this file, wherever you found it.");
  UTL_WriteStringToIniFile(pszIniFileName,szSec,"4", " ");

  sprintf( szSec, "Keyer%d", 1+UTL_iAppInstance ); // e.g. "Keyer1" for the FIRST running instance, etc
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "ComPort_IN",  CwKeyer_Config.iComPortNumber_IN );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "RadioControlAndKeyingPort",CwKeyer_Config.iRadioKeyingAndControlPort );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "RadioControlProtocol", CwKeyer_Config.iRadioControlProtocol );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "RadioControlBaudrate", CwKeyer_Config.iRadioControlBaudrate );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "RadioCIVAddress", CwKeyer_Config.iRadioCIVAddress );
  UTL_WriteStringToIniFile(pszIniFileName,szSec,"RemoteRigCtrlServerAddress", CwKeyer_Config.sz80RemoteRigCtrlServerAddress);
#if( SWI_USE_HAMLIB_SERVER ) // not sure if this 'feature' is going to stay...
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "EnableBuiltInHamlibServer",CwKeyer_Config.fEnableBuiltInHamlibServer);
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "BuiltInHamlibServerPort", MyHamlibServer.cfg.iServerListeningPort);
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "BuiltInHamlibServerOptions", MyHamlibServer.cfg.iServerOptions);
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "BuiltInHamlibServerDiagnostics", MyHamlibServer.cfg.iDiagnosticFlags);
#endif // SWI_USE_HAMLIB_SERVER ?

  UTL_WriteIntToIniFile(pszIniFileName, szSec, "MorseKeyType",    CwKeyer_Config.iMorseKeyType );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "DotInputSignal",  CwKeyer_Config.iDotInput );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "DashInputSignal", CwKeyer_Config.iDashInput );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "ManualPTTInput",  CwKeyer_Config.iManualPTTInput);
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "TestInputSignal", CwKeyer_Config.iTestInput );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "KeySupplySignal", CwKeyer_Config.iKeySupply );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "SidetoneOnTXD",   CwKeyer_Config.iSidetoneOnTXD );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "CwKeyingSignal",  CwKeyer_Config.iRadioCWKeying );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "PTTControlSignal",CwKeyer_Config.iRadioPTTControl);
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "TxDelayTime_ms",  CwKeyer_Config.iTxDelayTime_ms);
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "TxHangTime_ms",   CwKeyer_Config.iTxHangTime_ms);
  UTL_WriteIntToIniFile(pszIniFileName,szSec,"DebouncePaddleInputs",CwKeyer_Config.fDebouncePaddleInputs);
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "KeyInputWatchdog",CwKeyer_Config.fKeyInputWatchdog);
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "CwViaShiftAndControlKey", CwKeyer_Config.fKeyViaShiftAndControl);
  UTL_WriteStringToIniFile(pszIniFileName,szSec,"AudioInputDevice", CwKeyer_DSP.cfg.sz255AudioInputDevice);
  UTL_WriteStringToIniFile(pszIniFileName,szSec,"AudioOutputDevice",CwKeyer_DSP.cfg.sz255AudioOutputDevice);
  UTL_WriteIntToIniFile(pszIniFileName, szSec,  "AudioSidetone_Hz", CwKeyer_DSP.cfg.iSidetoneFreq_Hz );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "SidetoneRiseTime_ms",CwKeyer_DSP.cfg.iSidetoneRiseTime_ms );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "SidetoneGain_dB",CwKeyer_DSP.cfg.iSidetoneGain_dB );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "AudioInGain_dB", CwKeyer_DSP.cfg.iAudioInGain_dB );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "AudioOutGain_dB",CwKeyer_DSP.cfg.iAudioOutGain_dB );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "NetworkTonesGain_dB",CwKeyer_DSP.cfg.iNetworkTonesGain_dB );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "AudioFlags",      CwKeyer_DSP.cfg.iAudioFlags );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "WordsPerMinute",Elbug_DotTimeInMillisecondsToWordsPerMinute(CwKeyer_Elbug.cfg.iDotTime_ms) );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "LaunchOnStart", APPL_fLaunchKeyerOnStart );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "Channel4Source",CwKeyer_TimingScope.cfg.iChannel4Source );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "TScopeChannels",CwKeyer_TimingScope.cfg.iVisibleChannels);
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "TScopeTrigger", CwKeyer_TimingScope.cfg.iTriggerOptions );

  // Still in the ini-file-section "KeyerN", where "N" is the instance index...
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "TRXOptions", CwKeyer_Config.iTRXOptions );

  // Save options for the built-in "Rig Control" module :
  sprintf( szSec, "RigControl%d", 1+UTL_iAppInstance ); // e.g. "RigControl1" for the FIRST running instance, etc
  UTL_WriteIntToIniFile(pszIniFileName,szSec,"RejectMessagesInLog",CwKeyer_Config.dwRejectMessages4Log ); // ex: MyCwNet.RigControl.dwRejectMessages4Log);
  UTL_WriteIntToIniFile(pszIniFileName,szSec,"TrafficMonitorOptions",RigCtrl_iTrafficMonitorDisplayOptions);

  // Save the "Network" configuration (from module CwNet.c; simple TCP-client or server):
  sprintf( szSec, "Network%d", 1+UTL_iAppInstance ); // e.g. "Network1" for the FIRST running instance, etc
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "Functionality",  MyCwNet.cfg.iFunctionality );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "DiagnosticFlags",MyCwNet.cfg.iDiagnosticFlags );
  sprintf( szSec, "Network%dClient", 1+UTL_iAppInstance );
  UTL_WriteStringToIniFile( pszIniFileName, szSec, "RemoteServerIP",MyCwNet.cfg.sz80ClientRemoteIP );
  UTL_WriteStringToIniFile( pszIniFileName, szSec, "UserName", MyCwNet.cfg.sz80ClientUserName );
  UTL_WriteStringToIniFile( pszIniFileName, szSec, "Callsign", MyCwNet.cfg.sz80ClientCallsign );

  sprintf( szSec, "Network%dServer", 1+UTL_iAppInstance );
  UTL_WriteIntToIniFile(    pszIniFileName, szSec, "ListeningPort", MyCwNet.cfg.iServerListeningPort );
  UTL_WriteStringToIniFile( pszIniFileName, szSec, "AcceptUsers",MyCwNet.cfg.sz255AcceptUsers );
  UTL_WriteStringToIniFile( pszIniFileName, szSec, "Pwd",MyCwNet.cfg.sz20ServerAdminPWD );
  UTL_WriteIntToIniFile(    pszIniFileName, szSec, "HttpOptions",MyCwNet.cfg.iHttpServerOptions );
  UTL_WriteIntToIniFile(    pszIniFileName, szSec, "Latency_ms", MyCwNet.cfg.iNetworkLatency_ms );

  sprintf( szSec, "Keyer%d", 1+UTL_iAppInstance ); // e.g. back to "Keyer1" for the FIRST running instance, etc
  for(i=0; i<KEYER_NUM_MESSAGE_MEMORIES; ++i)
   { sprintf( szKey, "Memory%d", (int)(i+1) );
     UTL_WriteStringToIniFile( pszIniFileName, szSec, szKey, CwKeyer_Config.szKeyerMemory[i] );
   } // end for < all 'keyer message memories' >
  sprintf( szSec, "Keyer%dMainWindow", 1+UTL_iAppInstance ); // e.g. "Keyer1MainWindow" for the FIRST running instance, etc
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "Left",  CwKeyer_Config.iMainWindowLeft );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "Top",   CwKeyer_Config.iMainWindowTop );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "Width", CwKeyer_Config.iMainWindowWidth );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "Height",CwKeyer_Config.iMainWindowHeight );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "ColourScheme", g_SpecDispControl.iColourScheme );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "Language", APPL_iLanguage );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "UserInterfaceMode", g_iUserInterfaceMode );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "ITU_Region", RigCtrl_ITU_Region );

  APPL_fSaveSettingsOnExit = FALSE;  // "done" (flag cleared in SaveSettings())


} // end SaveSettings()

//---------------------------------------------------------------------------
DWORD CalcHashForSettings(void) // .. to tell if we need to SAVE SETTINGS on exit
{
#ifndef  sizeof_member
# define sizeof_member(type,member) sizeof(((type*)0)->member) /* ugly.. but works */
#endif

  // Actually, the hash function is the 32-bit CRC implemented in Utilities.c :
  DWORD dwHash= UTL_CRC32( 0/*seed*/, (BYTE*)&CwKeyer_Config, sizeof(CwKeyer_Config) );
  dwHash = UTL_CRC32( dwHash/*seed*/, (BYTE*)&CwKeyer_Elbug.cfg, sizeof_member(T_ElbugInstance,cfg) );
  dwHash = UTL_CRC32( dwHash/*seed*/, (BYTE*)&CwKeyer_DSP.cfg, sizeof_member(T_CwDSP,cfg) );
  dwHash = UTL_CRC32( dwHash/*seed*/, (BYTE*)&MyCwNet.cfg, sizeof_member(T_CwNet,cfg) );
  dwHash = UTL_CRC32( dwHash/*seed*/, (BYTE*)&CwKeyer_TimingScope.cfg, sizeof_member(T_KeyerTimingScope,cfg) );
#if( SWI_USE_HAMLIB_SERVER ) // not sure if the 'Hamlib Net rigctld'-emulation is going to stay...
  dwHash = UTL_CRC32( dwHash/*seed*/, (BYTE*)&MyHamlibServer.cfg, sizeof_member(T_HLSrv,cfg) );
#endif // SWI_USE_HAMLIB_SERVER ?

  return dwHash;

} // end CalcHashForSettings()


//---------------------------------------------------------------------------
extern "C" void ShowError( int iErrorClass, const char * pszFormat, ... )
  //                             |   ,------|__________|
  // ,---------------------------'   '--> Not just "char *" anymore, for reasons
  // |                                   explained in ?/YHF_Tools/StringLib.h !
  // '--> ERROR_CLASS_FATAL, ERROR_CLASS_ERROR, ERROR_CLASS_WARNING, ERROR_CLASS_INFO .
  //      If an optional 'debug run log' (file) is supported,
  //      iErrorClass bitmask SHOW_ERROR_IN_RUN_LOG can be used to emit
  //      the same message in "debug_run_log.txt", besides the GUI's "Debug" tab.
  //
{
  va_list parameter;
  T_ErrorFifoEntry *pErrorFifoEntry;
  char *pszDest, *pszEndstop;
  double dblUnixDateTime;

  va_start( parameter, pszFormat );
  // Because ShowError() *may* be called from a worker thread, don't "print"
  // directly into the non-thread-safe "RichEdit" control ! Instead, use a
  // classic lock-free circular FIFO between ShowError() and the GUI itself:
  pErrorFifoEntry = &ErrorHistoryFifo[ DEBUG_iErrorHistoryHead ];
  pszDest    = pErrorFifoEntry->sz511Text;
  pszEndstop = pErrorFifoEntry->sz511Text + 510/*!*/;
  if( iErrorClass & SHOW_ERROR_TIMESTAMP )
   { dblUnixDateTime = UTL_GetCurrentUnixDateAndTime();
     UTL_FormatDateAndTime( "hh:mm:ss.s ", dblUnixDateTime, pszDest );
     SL_SkipToEndOfString( (const char**)&pszDest );
   }
  vsnprintf( pszDest, (pszEndstop-pszDest)/*maxlen*/, pszFormat, parameter );
  pErrorFifoEntry->iErrorClass = iErrorClass & (~SHOW_ERROR_IN_RUN_LOG);
  va_end(parameter);
  // Make the new entry available for the "Debug" window by incrementing the FIFO HEAD index:
  DEBUG_iErrorHistoryHead = (DEBUG_iErrorHistoryHead + 1) % C_ERROR_FIFO_SIZE;

  if( iErrorClass & SHOW_ERROR_IN_RUN_LOG )
   { UTL_WriteRunLogEntry( "%s", pErrorFifoEntry->sz511Text );
     // '--> "Does nothing" if UTL_fRunLogOpened==FALSE (no need to check HERE)
   }

  // ex: SetStatusWindowText(MainWnd_hStatusWindow, STATUS_PART_ERR_MSG, 0, szText );
} // end ShowError()


//---------------------------------------------------------------------------
BOOL FillComboWithSerialPorts( TComboBox *pCombo,
        int iSelectedComPortNr ) // [in] "COM"-port number (1..N) or COMBO_BOX_COM_PORT_NONE
   // Returns TRUE if iSelectedComPortNr is currently valid,
   //         FALSE otherwise (i.e. not found in the devices "enumerated" by the OS)
{
  int  nEntriesFound = 0;
  BOOL fFoundSelectedPort = FALSE;
  int iEnumerator = 0;  // initialize 'enumerator'
  char sz80[84];
  static BOOL firstBug = TRUE;

  // On this occasion, allow the "TComboBox" to get WIDER when DROPPED DOWN :
  SendMessage( pCombo->Handle,CB_SETDROPPEDWIDTH, 280/*MinimumWidthInPixels*/, 0);
  // ,---------------------------------------------'
  // '--> This was even long enough for "COM3 (Silicon Labs CP210x USB to UART Bridge)",
  //      which was the funny "friendly" COM port name of an Icom IC-9700 !

  pCombo->Clear();
  pCombo->ItemIndex = -1; // nothing selected yet

  // First entry : "NONE", which is C_CFG_PORT_NONE = 0 in C:\cbproj\SpecLab\config.h
  pCombo->Items->Add( "NONE" );
  if( iSelectedComPortNr < 0 )
   {
     pCombo->ItemIndex = nEntriesFound; // index ZERO for the FIRST item in the list, etc
     fFoundSelectedPort = TRUE;
   }
  ++nEntriesFound;

  // Let Utilities.c : UTL_EnumerateComPorts() enumerate the COM ports.
  //     If possible (and Windows permitting), 'decorated' with parts of
  //     the 'friendly' name, but AFTER the significant part, in parentheses,
  //     e.g. "COM11 (USB Serial Port)" instead of just "COM11" :
  iEnumerator = 0;  // initialize the 'enumerator' for UTL_EnumerateComPorts()
  UTL_sz255LastError[0] = '\0';
  while( UTL_EnumerateComPorts( &iEnumerator, sz80, 80 ) )
   { // Add to the list of 'COM' ports, and possibly SELECT the item :
     pCombo->Items->Add( sz80 );  // hopefully something like "COM1", "COM2", ..
     // .. and if the "The Mimosa", SetupDiEnumDeviceInfo(), feels like cooperating,
     //    the list of COM ports even contains the 'friendly name' (every now and then).
     if( iSelectedComPortNr == UTL_ParseNumberFromComPortName(sz80) )
      { pCombo->ItemIndex = nEntriesFound; // index ZERO for the FIRST item in the list, etc
        fFoundSelectedPort = TRUE;
      }
     ++nEntriesFound;
   } // end while( UTL_EnumerateComPorts( &iEnumerator, sz80, 80 ) )
  if( (firstBug) && (UTL_sz255LastError[0] != '\0') ) // trouble extracting the "friendly names" again ?
   { ShowError( ERROR_CLASS_WARNING, "%s", UTL_sz255LastError );
     firstBug = FALSE;
     // 2024-01-05 : After being absent for several weeks, the "Bug" returned,
     // and UTL_EnumerateComPorts() diagnosed :
     // > EnumerateComPorts: SetupDiEnumDeviceInfo() failed at device #1 !
   }

  // If the configured COM port is NOT listed in the above loop
  //   (because stupid windows assigned a new COM port number to the device,
  //    or for some reason the COM port driver refuses to enumerate,
  //    as typically happens with COM ports from "VSPE"),
  // add a DUMMY ENTRY at the end of the list, so the pitiful windoze user
  //   can at least see that THERE IS SOMETHING WRONG WITH THE "COM PORT" .
  if( (!fFoundSelectedPort) && (iSelectedComPortNr>0) )
   {
     sprintf( sz80, "COM%d *NOT ENUMERATED*", iSelectedComPortNr );
     pCombo->Items->Add( sz80 );  // hopefully something like "COM1", "COM2", ..
     pCombo->ItemIndex = nEntriesFound; // index ZERO for the FIRST item in the list, etc
   } // end if( ! fFoundSelectedPort ) && ...

  return fFoundSelectedPort;  // -> TRUE=ok, FALSE=something wrong with iSelectedComPortNr
} // end FillComboWithSerialPorts()


//---------------------------------------------------------------------------
BOOL FillComboWithItemsFromTokenLists( TComboBox *pCombo,
        const T_SL_TokenList *pTokens1, // [in] e.g. C_TL_None_Zero (with string = "None", value = zero)
        const T_SL_TokenList *pTokens2, // [in] e.g. SerialPortInputSignals_Key
        const T_SL_TokenList *pTokens3, // [in] e.g. SerialPortInputSignals_Radio, or NULL for only *one* list
        int iSelectedTokenValue  )      // [in] e.g. [-]KEYER_SIGNAL_INDEX_MORSE_KEY_CTS
{
  int   iList;
  int   nEntries = 0;
  const T_SL_TokenList *pTokens;
  pCombo->Clear();
  pCombo->ItemIndex = -1;
  for( iList=1; iList<=3; ++iList)
   { switch(iList)
      { case 1: pTokens = pTokens1; break;
        case 2: pTokens = pTokens2; break;
        case 3: pTokens = pTokens3; break;
        default: pTokens = NULL; break;
      }
     if( pTokens != NULL ) // emit the strings from another list ?
      { while( pTokens->pszKeyword != NULL ) // not the end of the list yet ?
         {
           pCombo->Items->Add( pTokens->pszKeyword );  // hopefully something like "COM1", "COM2", ..
           if( iSelectedTokenValue == pTokens->iTokenValue )
            { pCombo->ItemIndex = nEntries; // index ZERO for the FIRST item in the list, etc
            }
           ++nEntries;
           ++pTokens;
         }
      } // if( pTokens != NULL )
   }
  if( pCombo->ItemIndex >= 0)
   { return TRUE;
   }
  pCombo->ItemIndex = 0; // instead of selecting garbage, select the first ...
  return FALSE;          // ... but give the caller the chance to mark this item in RED
} // end FillComboWithItemsFromTokenLists()


//---------------------------------------------------------------------------
BOOL FillComboWithBandList( TComboBox *pCombo, // [out] combo box with bands or even band-stacking registers
        DWORD dwSupportedBands, // [in] bitwise combination of RIGCTL_BAND_...
        DWORD dwSelectedBand )  // [in] a single bit representing the SELECTED band,
                                //      e.g. RIGCTL_BAND_40M .
{
  int   iBandIndex;
  DWORD dwBitmask, dwStackedBand;
  const char *pszBandName;
  char  sz80ItemText[84];
  char  *pszDest, *cpDestEndstop;
  BOOL  fFoundSelectedBand = FALSE;
  int   nEntriesFound = 0;
  int   nStackingEntriesInCurrentBand, iBandStackingRegIndex, iLen, iWantedPos;
  T_RigCtrlInstance *pRC = &MyCwNet.RigControl;
  T_RigCtrlFreqMemEntry *pBandStackingReg;

  // On this occasion, allow the "TComboBox" to get WIDER when DROPPED DOWN :
  SendMessage( pCombo->Handle,CB_SETDROPPEDWIDTH, 440/*MinimumWidthInPixels*/, 0);
  // ,---------------------------------------------'
  // '--> Should be wide enough for the BAND NAME (like "40 m")
  //      AND three "Band Stacking Registers" that modern Icoms radios like
  //      the IC-7300 has stored automatically, with the "most recent frequencies"
  //      that the operator has visited.
  //      Alternatively (without band-stacking registers), the combo list
  //      may show frequency limits or presets DEFINED BY THE SYSOP (future plan).
  //      With the combo's font set to "Courier New", size EIGHT,
  //      each character had a fixed width of NINE(!) pixels .. but ymmv .
  //  Icom's own 'Band Stacking Register' display on an IC-7300 looked
  //  like this (and this is what we're going to mimick with the TComboBox):
  //      " 50.085 CW    50.095 CW    50.135 USB"
  //      |<-12 chars->|<-12 chars->|<-12 chars->|
  //   In addition, for the NON-DROPPED-DOWN combo list,
  //   the begin of each line shall indicate the BAND NAME, e.g. "80 m: " (6 chars).
  //    ( 6 + 3 * 12 ) characters * 9 pixels_per_character = 396+x, say 440 pixels.

  pCombo->Clear();
  pCombo->ItemIndex = -1; // nothing selected yet

  // To know which rows and columns have been filled with which
  // 'band stacking registers' later in CB_BandClick(),
  // use an extra array :
  memset( s_iBandStackingRegIndices, -1, sizeof(s_iBandStackingRegIndices) );

  for( iBandIndex = 0; iBandIndex<=31; ++iBandIndex ) // try all 32 possible "band-bit" indices
   { dwBitmask = 1UL << iBandIndex; // -> RIGCTRL_BAND_160M (bit 0), etc etc ..
     pszBandName = SL_GetStringFromTokenList( RigCtrl_BandNames, dwBitmask );
     if( (pszBandName!=NULL) && ((dwBitmask & dwSupportedBands)!=0) )
      { // Regardless of what's in pRC->BandStackingRegs[ 0..RIGCTRL_NUM_BAND_STACKING_REGS-1 ],
        // will get here with pszBandName = "160 m", "80 m", "60 m", "40 m" .. .
        // The BAND NAME will always be added to the combo, even if there are
        // no band-stacking-register entries for this band *anywhere*
        // (This is important, *anywhere*, because for example an IC-7300
        //  modified for 'all band transmit' or a few extra bands like 60 m or 4 m
        //  will NOT add extra 'band codes' to the band stacking registers.
        //  Instead, the 'added bands' like 60 m, 4 m, etc will all share a
        //  band that Icom labelled 'GENE' in the IC-7300's band stacking
        //  register display.)
        // EXAMPLE: Content of pRC->BandStackingRegs[], as displayed via
        //   'Settings'..'Report Rig Control parameters on the Debug tab' :
        //      BandStack[00..02]:  1.915  CW,   1.910  CW,   1.915  CW
        //      BandStack[03..05]:  3.573  CW,   3.793 LSB,   3.557  CW
        //      BandStack[06..08]:  7.030  CW,   7.076 LSB,   7.201 LSB
        //      BandStack[09..11]: 10.120  CW,  10.130  CW,  10.140  CW
        //      BandStack[12..14]: 14.244 USB,  14.100  CW,  14.054  CW
        //      BandStack[15..17]: 18.150  CW,  18.150 USB,  18.070  CW
        //      BandStack[18..20]: 21.024  CW,  21.300 USB,  21.050  CW
        //      BandStack[21..23]: 24.898  CW,  24.980 USB,  24.900  CW
        //      BandStack[24..26]: 28.036  CW,  28.565 USB,  29.269  FM
        //      BandStack[27..29]: 50.085  CW,  50.097  CW,  50.095  CW
        //      BandStack[30..32]:  5.353  CW,   0.472  CW,  70.107  CW
        //                          /|\           /|\          /|\   ..
        //      "GENE" band entries--'-------------'------------'
        //
        //    DON'T ASSUME ANYTHING about which frequencies are stored
        //    in which part of the array ... things will be COMPLETELY DIFFERENT
        //    in other transceivers; e.g. IC-9700 and especially IC-705 !
        pszDest = sz80ItemText;
        cpDestEndstop = sz80ItemText + sizeof(sz80ItemText) - 1;
        SL_AppendString( &pszDest, cpDestEndstop, pszBandName );
        // Check all 'Band Stacking Registers' if they are in THIS band (dwBitmask) .
        // If a band stacking register belongs to this band, append it to
        // the string (limited to THREE stack entries per band, as in Icom radios)
        nStackingEntriesInCurrentBand = 0; // -> 0..2
        for( iBandStackingRegIndex=0;
             iBandStackingRegIndex < pRC->iNumBandStackingRegs;
             iBandStackingRegIndex++)
         { pBandStackingReg = &pRC->BandStackingRegs[ iBandStackingRegIndex ];
           dwStackedBand = RigCtrl_FrequencyToBand( pBandStackingReg->RxTx[0].dblOperatingFreq_Hz );
                // '--> RIGCTL_BAND_xyz (a BITMASK, not an index)
           // Only add this entry from pRC->BandStackingRegs[] if it belongs
           // to the current BAND (in a single line of the combo box):
           if( ( (dwStackedBand & dwBitmask) != 0 ) && (nStackingEntriesInCurrentBand<3) )
            { if( nStackingEntriesInCurrentBand == 0 )
               { SL_AppendString( &pszDest, cpDestEndstop, ":" );
               }
              iLen = strlen( sz80ItemText );
#            define BANDLIST_BAND_COLUM_WIDTH  6  // <- also used in CB_BandClick() later..
#            define BANDLIST_FREQ_COLUMN_WIDTH 14 // worst case: " 1296.2345 USB"
              iWantedPos = BANDLIST_BAND_COLUM_WIDTH + BANDLIST_FREQ_COLUMN_WIDTH * nStackingEntriesInCurrentBand;
              if( iLen < iWantedPos )
               { SL_PadSpaces( &pszDest, cpDestEndstop, iWantedPos - iLen );
               }
              // Like Icom in their "BAND STACKING REGISTER" display,
              // show the frequency in MEGAHERTZ (but without a unit),
              // with leading space for a fixed width, and only
              // three decimal places. Umm.. what was the printf format string ?
#            if( 1 ) // format string to have everything vertically aligned:
              SL_AppendPrintf( &pszDest, cpDestEndstop, "%9.4lf %3s",
                 // "width" (or something in that style)--' |
                 // "precision" (or whatever it was..)------'
#            else // format string to have the leftmost digits vertically aligned,
                  // but at least one space as separator (save even for e.g. "1296.200 USB")
              SL_AppendPrintf( &pszDest, cpDestEndstop, " %.3lf",
#            endif
                 (double)(pBandStackingReg->RxTx[0].dblOperatingFreq_Hz * 1e-6),
                  RigCtrl_OperatingModeToString( pBandStackingReg->RxTx[0].iOpMode
                           & RIGCTRL_OPMODE_MASK_TO_STRIP_FLAGS ) );
              s_iBandStackingRegIndices[iBandIndex][nStackingEntriesInCurrentBand]=iBandStackingRegIndex;
              //                        '--> 0..31  '--> 0..2                       |
              //                   ("band index",     "stack levels per band")      |
              //          not the COMBO ITEM INDEX !)                               |
              //    ,---------------------------------------------------------------'
              //    '--> Index into pRC->BandStackingRegs[ iBandStackingRegIndex ]
              // EXAMPLE with BandStackingRegs[3] = "3.5730 CW", in the SECOND LINE
              //         of the combo box:
              //    nEntriesFound = 1 (here used as the zero-based combo item index),
              //    nStackingEntriesInCurrentBand = 0 (i.e. the MOST RECENT entry, aka "top of stack")
              //    dwStackedBand = 2 = (1<<1) = RIGCTRL_BAND_80M
              //    iBandStackingRegIndex = 3
              //  When selecting that entry ("3.5730 CW") in TKeyerMainForm::CB_BandClick() :
              //    (1) from the BEGIN of the item text (sz80ItemText : "80 m: .."),
              //        i32NewBand will be parsed -> result = 2 = (1<<1) = RIGCTRL_BAND_80M .
              //        From that BITMASK, the new "band index" (a zero based array index)
              //        will be: iBandIndex = BitmaskToBitIndex( i32NewBand );
              //    (2) from the HORIZONTAL MOUSE POINTER POSITION,
              //        TKeyerMainForm::CB_BandClick() determines iBandStackIndex .
              //    (3) iBandIndex and iBandStackIndex are the indices into
              //         s_iBandStackingRegIndices[iBandIndex][iBandStackIndex].
              //        That two-dimensional array delivers the array index
              //        (iBandStackingRegIndex) into pRC->BandStackingRegs[iBandStackingRegIndex].
              //    (4) That band-stacking-register should contain everything
              //        module RigControl.c needs to know (frequency, mode, AND MANY MORE)
              //        to switch back to the "old" (stacked) frequency.
              ++nStackingEntriesInCurrentBand;
            } // end if < matching frequency for the current "band" >
         }   // end for( iBandStackingRegIndex .. )

        pCombo->Items->Add( sz80ItemText );
        if( dwBitmask & dwSelectedBand )
         { pCombo->ItemIndex = nEntriesFound; // index ZERO for the FIRST item in the list, etc
           fFoundSelectedBand = TRUE;
         }
        ++nEntriesFound;
      }
   }
  return fFoundSelectedBand;  // -> TRUE=ok, FALSE=something wrong
} // end FillComboWithBandList()


//---------------------------------------------------------------------------
int GetSelectedTokenValueFromComboBox( TComboBox *pCombo,
        const T_SL_TokenList *pTokens1, // [in] e.g. C_TL_None_Zero (with string = "None", value = zero)
        const T_SL_TokenList *pTokens2, // [in] e.g. SerialPortInputSignals_Key
        const T_SL_TokenList *pTokens3) // [in] e.g. SerialPortInputSignals_Radio, or NULL for only *one* list
  // If the selected text in the combo list doesn't match ANY of the
  // strings in the three token lists, this function returns
  // GUI_ERROR_NO_MATCHING_ENTRY to let THE CALLER decide what to use.
# define GUI_ERROR_NO_MATCHING_ENTRY -99999
{
  int   iList;
  const T_SL_TokenList *pTokens;

  for( iList=1; iList<=3; ++iList)
   { switch(iList)
      { case 1: pTokens = pTokens1; break;
        case 2: pTokens = pTokens2; break;
        case 3: pTokens = pTokens3; break;
        default: pTokens = NULL; break;
      }
     if( pTokens != NULL ) // check the strings from another list ?
      { while( pTokens->pszKeyword != NULL ) // not the end of the list yet ?
         {
           if( pCombo->Text == AnsiString( pTokens->pszKeyword ) )
            { return pTokens->iTokenValue;
            }
           ++pTokens;
         }
      }
   }
  return GUI_ERROR_NO_MATCHING_ENTRY;  // No valid selection ? -> let THE CALLER handle this !
} // end GetSelectedTokenValueFromComboBox()

//---------------------------------------------------------------------------
BOOL SelectComboItemByDecimalValue( TComboBox *pCombo, int iValue )
{ int i, n, iSelItem=-1;
  n = pCombo->Items->Count;
  for(i=0; i<n; ++i)
   { if( StrToIntDef( pCombo->Items->Strings[i], 0) == iValue )
      { iSelItem = i;
        break;
      }
   }
  if( iSelItem>=0 )
   { pCombo->ItemIndex = iSelItem;
     return TRUE;
   }
  else
   { return FALSE;
   }
}

//---------------------------------------------------------------------------
BOOL SelectComboItemByHexadecimalValue( TComboBox *pCombo, int iValue )
{ int i, n, iSelItem=-1;
  char sz40Text[44];
  const char *pszSrc;
  n = pCombo->Items->Count;
  for(i=0; i<n; ++i)
   { SL_strncpy( sz40Text, pCombo->Items->Strings[i].c_str(), 40 );
     pszSrc = sz40Text;
     if( SL_SkipString( (const char**)&pszSrc, "0x" ) ) // looks like a HEX number ...
      { if( (int)SL_ParseHex( &pszSrc, 4/*nMaxDigits*/) == iValue )
         { iSelItem = i;
           break;
         }
      }
   }
  if( iSelItem>=0 )
   { pCombo->ItemIndex = iSelItem;
     return TRUE;
   }
  else
   { pCombo->ItemIndex = -1;
     sprintf( sz40Text, "0x%02X", (unsigned int) iValue );
     pCombo->Text = sz40Text; // works with TComboBox.Style=csDropDown, not with csDropDownList !
     return FALSE;
   }
}

//---------------------------------------------------------------------------
int  GetSerialPortSignalIndexFromComboBox( TComboBox *pCombo )
  // Return value : KEYER_SIGNAL_INDEX_... , negative when inverted, 0 = "none"
{
  int iSelectedTokenValue = GetSelectedTokenValueFromComboBox( pCombo,
        C_TL_None_Zero,                // [in] const T_SL_TokenList *pTokens1
        SerialPortInputSignals_Key,    // [in] const T_SL_TokenList *pTokens2
        SerialPortInputSignals_Radio); // [in] const T_SL_TokenList *pTokens3
  if( iSelectedTokenValue == GUI_ERROR_NO_MATCHING_ENTRY )
   { iSelectedTokenValue = GetSelectedTokenValueFromComboBox( pCombo,
        SerialPortOutputSignals_Key,   // [in] const T_SL_TokenList *pTokens1
        SerialPortOutputSignals_Radio, // [in] const T_SL_TokenList *pTokens2
        NULL );                        // [in] const T_SL_TokenList *pTokens3
   }
  return iSelectedTokenValue;
}

//---------------------------------------------------------------------------
int ParseSerialPortNumberFromText( char *pszSource )
   // Returns a non-negative serial port number when successful,
   //         otherwise -1 (also for the entry named "NONE", in combo boxes)
{
  int i;
  if( strncmp( pszSource, "NONE", 4) == 0 )
   { return -1;
   }
  if( pszSource[0] != '\0' )
   { // Does the stupid sscanf() parse the '9' in "COM9 (invalid)" ? [2015-01: yes]
     i =  UTL_ParseNumberFromComPortName( pszSource );
     if( i >= 0 )
      { return i;
      }
   }
  return -1;  // invalid COM port number !

} // end ParseSerialPortNumberFromText()


//---------------------------------------------------------------------------
double GetFrequencyFromEditField( TEdit *pEd, // [in] edit field with frequency
                 double *pdblFreqIncrement ) // [out] 10^N, f(cursor_position)
  // Accepts a string from the VFO frequency input field (or similar fields)
  // in different formats, including 'technical' notations like 12.345 kHz
  //                                                         or 7.0374 MHz .
  // For incrementing / decrementing the frequency via cursor up/down keys
  //      or mouse wheel, the function may pass back the position of the
  //      text cursor (aka 'caret') with in the TEdit, and the value to add/
  //      subtract on each cursor up/down event. Examples

  //      char index:  0123456789ABCDEF
  //      pEd->Text = "   7.012 345 MHz"  (note the spaces, also "in between")
  //                            '------- blinking caret or underscore HERE
  //   -> [out] pdblFreqIncrement = 100 [Hz]
  //
  //
{
  char c, szA[44], *cp;
  int i, slen, nDigits, iExponent;
  int iExponentFromTechUnit=0, nDecimalPlaces=0;
  double dblFreqHz, dblUpperPart, dblLowerPartMult;
  BOOL   fNegative = FALSE;
  BOOL   fGotDecimalPlaces = FALSE;

  SL_strcpy_trunc(szA, sizeof(szA), pEd->Text.c_str(), 39 );

  // Convert string to decimal, ignoring spaces:
  dblFreqHz = dblUpperPart = 0.0;
  dblLowerPartMult = 0.0;  // there's no "lower part" to be added later
  slen = 0;
  cp = szA;
  while( *cp == ' ' )
   { ++cp;
     ++slen;
   }
  if( *cp == '-' )
   { ++cp;
     ++slen;
     fNegative = TRUE;
   }
  while( *cp != '\0' )
   { if( ((c=*cp)>='0') && (c<='9') )
      { dblFreqHz = 10.0*dblFreqHz + (c-'0');
        dblLowerPartMult /= 10.0;
        if( fGotDecimalPlaces )
         { ++nDecimalPlaces;
         }
      }
     else  // a "technical" exponent like 'k', 'M', 'G' ?
      { if( c=='.' || c=='k' || c=='M' || c=='G' || c=='m' || c=='u'  )
         {  // Combine the previous 'upper' and 'lower' part into a single new 'upper' part,
            // and begin a new string->float conversion for the rest (fractional part):
            if( dblLowerPartMult > 0.0 )  // fraction from k(Hz), M(Hz), or G(Hz) ?
             { dblFreqHz *= dblLowerPartMult;
             }
            dblUpperPart += dblFreqHz; // the sum now completely in the "upper part" (incl. fraction)
            dblFreqHz    = 0;        // begin new conversion for the fractional part
            dblLowerPartMult = 1.0;  // factor for trailing digits (will be divided by ten with each new digit)
         }
        switch( c )
         {
           case '.':  // already parsed number was in HERTZ (or whatever), the rest is the fractional part:
              dblFreqHz         = 0;
              dblLowerPartMult  = 1.0;  // factor for trailing digits (will be divided by ten with each new digit)
              nDecimalPlaces    = 0;
              fGotDecimalPlaces = TRUE;
              break;
           case 'k':  // already parsed number was in KILOHERTZ, rest is a fraction:
              iExponentFromTechUnit = 3;  // required later for inc/dec
              dblUpperPart *= 1000.0;
              dblLowerPartMult = 1000.0;  // factor for trailing digits (will be divided by ten with each new digit)
              break;
           case 'M':  // already parsed number was in MEGAHERTZ, rest is a fraction:
              iExponentFromTechUnit = 6;  // required later for inc/dec
              dblUpperPart *= 1E6;
              dblLowerPartMult = 1E6;
              break;
           case 'G':  // already parsed number was in GIGAHERTZ, rest is a fraction:
              iExponentFromTechUnit = 9;  // required later for inc/dec
              dblUpperPart *= 1E9;
              dblLowerPartMult = 1E9;
              break;
           case 'm':  // already parsed number was in MILLIHERTZ, rest a tiny fraction:
              iExponentFromTechUnit = -3;
              dblUpperPart *= 1E-3;
              dblLowerPartMult = 1E-3;
              break;
           case 'u':  // already parsed number was in MICROHERTZ, rest a ridiculous fraction:
              iExponentFromTechUnit = -6;
              dblUpperPart *= 1E-6;
              dblLowerPartMult = 1E-6;
              break;
           default :  // e.g. c='-' if the frequency in Ed_VFO is invalid
              break;
         }
      }
     ++cp;
     ++slen;
   }
  // Examples:
  //  (a) "3M5" -> UpperPart=3000000, LowerPart=5, LowerPartMult=100000 .
  //  (b) "---.----- MHz" ->  fNegative=TRUE, UpperPart=0, dblFreqHz=0, LowerPartMult=0 .
  if( dblLowerPartMult > 0.0 )  // fraction from k(Hz), M(Hz), or G(Hz) ?
   { dblFreqHz *= dblLowerPartMult;
   }
  dblFreqHz += dblUpperPart;
  if( fNegative )
   { dblFreqHz = -dblFreqHz;
   }
  //
  // Convert the current cursor position into a power of ten for inc/dec-editing.
  // EXAMPLE: szA="123", slen=3, i=2 -> iFreqIncrement=1 .
  //          (in this case, there's NO digit to examine, because
  //           the RIGHTMOST digit is selected for inc/dec-editing)
  i = pEd->SelStart+1; // cursor position within the string; 0=first char
  // Because the cursor shall be RIGHT NEXT to the incremented/decremented digit,
  //    a cursor after the 'end' of the string is meaningless, i.e. :
  //    with sLen==1, the only valid cursor position (i) is ZERO, not ONE:
  if( i>slen )  i=slen;
  if( i<0 )     i=0;
  //                      Cursor (left of the digit to increment/decrement,
  //                       |  usually with "SelLength"=1 to highlight digit)
  //                       |
  //  Example:  szA = "   12 345 567"  slen=13, nDigits=8
  //                   |   |       |
  //                   |   |       |
  //               szA[0] i=4  szA[12]
  //  Note that there may be SPACES or other thousand-separators
  //  in the string, so COUNT THE DIGITS here to find the decimal place:
  nDigits = 0; // ... i.e. dblFreqIncrement = 10 ^ iCursorDigitPos
  while( i<slen )
   { if( szA[i]>='0' && szA[i]<='9' )  // only count the DIGITS, nothing else !
      { ++nDigits;
      }
     ++i;
   }
  iExponent = nDigits + iExponentFromTechUnit - nDecimalPlaces;
  if( pdblFreqIncrement != NULL ) // caller wants the power of ten (f(cursor-pos) )
   { *pdblFreqIncrement = pow10( iExponent ); // pow10 takes an INTEGER argument; HERE that's ok
   }

  return dblFreqHz;
} // end GetFrequencyFromEditField()


//---------------------------------------------------------------------------
static void UpdateTimingScope( // Does what the name implies :) ..
              T_KeyerTimingScope *pScope,     // [in] timing scope data and configuration
              Graphics::TBitmap *pDestBitmap) // [out] graphic bitmap (unfortunately, a Borland-specific VCL thing)
      // [in]  TimingScopeOverlay[] : mouse-controlled 'overlay' to measure times
{
  int w = pDestBitmap->Width;  // width in pixels
  int h = pDestBitmap->Height; // height in pixels
  int iSample, x, y, h2, h3, yc, y_min, y_max, val, prev_x, prev_y, th, tw, tx, ty;
  int iSampleAtBeginOfChar;
  float flt;
  WORD  wCwChar;
  DWORD dwPenColour; // R-G-B colour mix (Red in bits 7..0, Green bits 15..8, Blue bits 23..16)
                     // Compatible with the VCL type "TColor". Possibly a 'Windows thing'.
  Graphics::TBitmap *osb;
  TPoint *pPoints;
  int     nPoints;
  // Test result:
  //   Plotting each curve in a single call of TCanvas.Polyline() : 2.5 % CPU load from the "Remote CW keyer"
  //   Plotting curves with MoveTo() / hundreds of LineTo() calls : 7.5 % CPU load  "    "     "     "  "
  //   Not plotting anything (only running the 'keyer thread')    : 1.3 % CPU load  "    "     "     "  "
  // -> Worth the effort (to reduce the CPU load) !
  const char *pszDecoded;
  AnsiString s;

  HERE_I_AM__GUI();

  pPoints = (TPoint*)malloc( TIMING_SCOPE_NUM_SAMPLE_POINTS * sizeof(POINT) );
  if(pPoints==NULL) // oops .. malloc() failed !
   { return;
   }

  osb = new Graphics::TBitmap;
  osb->Width = w;
  osb->Height= h;
  g_iTimingScopeBitmapWidth = w;  // <- only for diagnostic / trouble with "destroyed forms"

  // Erase the background :
  osb->Canvas->Brush->Style = bsSolid;
  osb->Canvas->Brush->Color = clBlack;
  osb->Canvas->FillRect( TRect(0,0,w,h) );  // <- this wasn't the CPU hog !

  // Prepare a few parameters for the "oscilloscope screen layout" :
  h2 = (h-16) / TIMING_SCOPE_N_DIGITAL_CHANNELS; // height for channel, including space between channels
  h3 = (3 * h2) / 4;           // height in pixels for the CURVES, without spaces

  // Draw optional info / warnings 'into the background' ?
  osb->Canvas->Font->Name  = "Arial";
  osb->Canvas->Font->Size  = 12;
  osb->Canvas->Font->Color = clGray;
  th = osb->Canvas->TextHeight( "Q" );
  tx = ty = 16;
  if( pScope->cfg.iVisibleChannels == 0 )   // oops... all scope channels turned off !
   { osb->Canvas->TextOut( tx, ty, "All scope channels are turned off - " );
     ty += th;
     osb->Canvas->TextOut( tx, ty, " Use the RIGHT MOUSE BUTTON" );
     ty += th;
     osb->Canvas->TextOut( tx, ty, " to open the scope's context menu." );
     ty += th;
   }

#if( TIMING_SCOPE_N_ANALOG_CHANNELS > 0 )
  // Draw the ANALOG channels (if any, to have them in the BACKGROUND) :
  for( int iChannel=0; iChannel<TIMING_SCOPE_N_ANALOG_CHANNELS; ++iChannel )
   { // (draw channel by channel, to avoid having to swap colours too often)
     switch( iChannel )  // select an analog-channel-specific "pen colour" :
      { case 0: dwPenColour = 0xFF0080;  // VIOLET
                break;
        case 1: dwPenColour = 0x007FFF;  // ORANGE
                break;
        case 2: dwPenColour = 0xC0C000;  // slightly dimmed CYAN
                break;
        case 3: dwPenColour = 0x007F7F;  // BROWN
                break;
        default: dwPenColour= 0xFF00FF;  // MAGENTA
                break;
      }
     if( ( dwPenColour != 0x000000 )  // only draw the curve if the channel isn't hidden ..
      && ( pScope->cfg.iVisibleChannels & ( 1<<(iChannel+TIMING_SCOPE_FIRST_ANALOG_CHANNEL) ) )  )
      { osb->Canvas->Pen->Color = (TColor)dwPenColour;
        yc = h/2 - 1 + 2*iChannel;
        nPoints = 0;
        for( iSample=0; iSample<TIMING_SCOPE_NUM_SAMPLE_POINTS; ++iSample)
         { g_iDebugDummyIndex = iSample;
           x = (w * iSample) / TIMING_SCOPE_NUM_SAMPLE_POINTS;
           flt = pScope->sSamplePoints[iSample].fltAnalog[iChannel];
           if( flt > 1.0f ) // gotcha..  clipping (no problem for a "float", but for PLOTTING further below)
            {  flt = 1.0f;  // <- good place for a breakpoint ..
               // WATCH the scope's sample-array as described further below.
            }
           if( flt < -1.0f )
            {  flt = -1.0f;
            }
           flt *= (float)(h/2);
           // 2024-01-14: Occasionally got a 'floating point overrun' here ?!
           //      After that, the antique debugger was totally clueless,
           //      and even though PAUSED AT THIS POINT,
           //      could not inspect any local variable or pointer anymore.
           //  To circumvent that (and see what was wrong in pScope->sSamplePoints[]),
           //  WATCH the following expression instead (works because there's only ONE INSTANCE yet,
           //  fortunately, an old-fashioned debugger-friendly global variable):
           //  CwKeyer_TimingScope.sSamplePoints[g_iDebugDummyIndex-2] (with a 'repeat count' of e.g. 4) .
           //
           y = yc - (int)flt;
           if(iSample==0)
            {
              pPoints[nPoints].x   = x;     // add the first point to the polyline
              pPoints[nPoints++].y = y;
            }
           else
            { if( (x != prev_x) || (y != prev_y) ) // avoid unnecessary GDI calls to save CPU time
               {
                 pPoints[nPoints].x   = x;     // append another point to the polyline
                 pPoints[nPoints++].y = y;
               } // end if < different x or y >
            }
           prev_x = x;
           prev_y = y;
         } // end for < all sample of a channel >
        if( nPoints >= 2 )
         { osb->Canvas->Polyline( pPoints, nPoints-1 );
           // ,----------------------------|_______|
           // '--> funny but true: Not the NUMBER OF POINTS,
           //                      but the NUMBER OF POINTS MINUS ONE !
         }
      }  // end if < ANALOG channel visible > ?
   }    // end for < all ANALOG channels >
#endif // TIMING_SCOPE_N_ANALOG_CHANNELS > 0 ?


  // Draw the DIGITAL channels (in the foreground, considered 'more important' than the audio waveforms).
  for( int iChannel=0; iChannel<TIMING_SCOPE_N_DIGITAL_CHANNELS; ++iChannel )
   { // (also here, draw channel by channel, to avoid having to swap colours too often)
     switch( iChannel )  // select a digital-channel-specific "pen colour" :
      { case TIMING_SCOPE_CHANNEL_DASH_INPUT:
             dwPenColour = 0x00FFFF; // YELLOW
             break;
        case TIMING_SCOPE_CHANNEL_DOT_INPUT :
             dwPenColour = 0x3FFF3F; // LIGHTGREEN
             break;
        case TIMING_SCOPE_CHANNEL_CW_OUTPUT :
             dwPenColour = 0x3F3FFF; // LIGHTRED ("we're most likely on air now")
             break;
        case TIMING_SCOPE_CHANNEL_FOUR: // ex: .. _PTT_OUTPUT
             dwPenColour = 0xFF7F7F; // LIGHTBLUE for the user-selectable source
             break;
        default:
             dwPenColour = 0x000000; // black -> HIDDEN !
             break;
      }
     if( (dwPenColour != 0x000000 )  // only draw the curve if the channel isn't hidden ..
       &&((pScope->cfg.iVisibleChannels & (1<<iChannel) ) != 0 ) )
      { osb->Canvas->Pen->Color = (TColor)dwPenColour;
        y_min = iChannel * h2;
        y_max = y_min + (h2-1);
        nPoints = 0;
        for( iSample=0; iSample<TIMING_SCOPE_NUM_SAMPLE_POINTS; ++iSample)
         { x = (w * iSample) / TIMING_SCOPE_NUM_SAMPLE_POINTS;
           val = pScope->sSamplePoints[iSample].iChannel[iChannel];
           y = y_max - (val * h3) / 100/*percent*/;
           if(iSample==0)
            {
              pPoints[nPoints].x   = x;     // add the first point to the polyline
              pPoints[nPoints++].y = y;
            }
           else
            { if( (x != prev_x) || (y != prev_y) ) // avoid unnecessary GDI calls to save CPU time
               {
                 pPoints[nPoints].x   = x;     // append another point to the polyline
                 pPoints[nPoints++].y = y;
               } // end if < different x or y >
            }
           prev_x = x;
           prev_y = y;
         } // end for < all sample of a channel >
         if( nPoints >= 2 )
          { osb->Canvas->Polyline( pPoints, nPoints-1 );
            // ,----------------------------|_______|
            // '--> funny but true: Not the NUMBER OF POINTS,
            //                      but the NUMBER OF POINTS MINUS ONE !
          }
      } // end if < DIGITAL channel visible > ?
   }   // end for < all DIGITAL channels >

  // Draw "decoded text" into the Timing Scope, too :
  osb->Canvas->Font->Name  = "Arial";
  osb->Canvas->Font->Size  = 16;
  osb->Canvas->Font->Color = clWhite;
  osb->Canvas->Pen->Color  = clWhite;
  th = osb->Canvas->TextHeight( "Q" );
  for( iSample=iSampleAtBeginOfChar=0; iSample<TIMING_SCOPE_NUM_SAMPLE_POINTS; ++iSample)
   {
     if( (wCwChar = pScope->sSamplePoints[iSample].wCwChar) != 0)
      { pszDecoded = Elbug_MorseCodePatternToASCII( wCwChar );
        if( wCwChar & ELBUG_RESULT_BEGIN_NEW_CHAR ) // nothing "decoded" at this sample index,
         { // but a marker for the BEGIN of a character (decoded and printed LATER):
           iSampleAtBeginOfChar = iSample;
         }
        else
         { // It's a decoded character, so print it, horizontally centered...
           x = (w * iSample) / TIMING_SCOPE_NUM_SAMPLE_POINTS;
           y = (TIMING_SCOPE_N_DIGITAL_CHANNELS-1) * h2 + th/4; // near the "CW output" channel
           prev_x = (w * iSampleAtBeginOfChar) / TIMING_SCOPE_NUM_SAMPLE_POINTS;
           // Horizontally center the text under the "CW Signal" that belongs to it:
           //      ___   _   ___     ___   _
           //  ___|   |_| |_|   |___|   |_| |___ <- "CW Signal" (keyer output)
           //
           //     '-------K -------''--- N ---'  <- CW decoder output
           //     |       |        |
           // prev_x      tx       x
           //  ,----------------|__|
           //  '--> This TWO-DOT-GAP belongs to the character,
           //       because without it, the decoder cannot decode it.
           //       If this 'Timing Scope' convinces anyone to stop "smearing
           //       characters together", this little gadget has already paid off :o)
           //
           tw = osb->Canvas->TextWidth( AnsiString(pszDecoded) );
           tx = prev_x + (x-prev_x-tw) / 2;
           SetBkMode( osb->Canvas->Handle, TRANSPARENT ); // <- old Windows 'GDI' function
           osb->Canvas->TextOut( tx, y, AnsiString(pszDecoded) );
           if( x != prev_x )
            { osb->Canvas->MoveTo( prev_x,  y      );
              osb->Canvas->LineTo( prev_x,  y+th/2 );
              osb->Canvas->LineTo( tx,      y+th/2 );
              osb->Canvas->MoveTo( tx+tw,   y+th/2 );
              osb->Canvas->LineTo( x,       y+th/2 );
              osb->Canvas->LineTo( x,       y      );
            }
         }
      } // end if < "decoded character" at index iSample > ?
   } // end for < all sample-indices, looking for DECODED CHARACTERS to print >

  // Draw a primitive, relative timescale; stepwidth = ONE DOT INTERVAL..
  osb->Canvas->Pen->Color = clWhite;
  iSample=0;
  while( iSample<TIMING_SCOPE_NUM_SAMPLE_POINTS )
   { x = (w * iSample) / TIMING_SCOPE_NUM_SAMPLE_POINTS;
     osb->Canvas->MoveTo( x,  0   );
     osb->Canvas->LineTo( x,  3   );
     osb->Canvas->MoveTo( x,  h-4 );
     osb->Canvas->LineTo( x,  h-1 );

     // Assume the CW-keyer's worker thread really samples the inputs
     //        at <pScope->iMillisecondsPerSample> ...
     if(  (pScope->iMillisecondsPerSample >= 1 )
        &&(CwKeyer_Elbug.cfg.iDotTime_ms >= pScope->iMillisecondsPerSample ) )
      { iSample += (CwKeyer_Elbug.cfg.iDotTime_ms / pScope->iMillisecondsPerSample);
      }
     else // something wrong with the Elbug or Timing Scope parameters..
      { // draw a tick every TEN samples (this may be something like 20 ms per tick)
        iSample += 10;
      }
   }

  // Draw the overlays for 'mouse cursor readout' (especially time differences)
  for( int i=0; i<N_SCOPE_OVERLAYS; ++i )
   { T_TimingScopeOverlay *pOverlay = &TimingScopeOverlay[i];
     if( pOverlay->visible )
      { osb->Canvas->Pen->Color = clWhite;
        osb->Canvas->Font->Size  = 10;
        osb->Canvas->MoveTo( pOverlay->x1, pOverlay->y1 );
        osb->Canvas->LineTo( pOverlay->x2, pOverlay->y2 );
        SetBkMode( osb->Canvas->Handle, OPAQUE ); // <- old Windows 'GDI' function
        s = "dt="+IntToStr( pOverlay->t2_ms - pOverlay->t1_ms )+" ms";
        th = osb->Canvas->TextHeight( s );
        tw = osb->Canvas->TextWidth( s );
        tx = (pOverlay->x1 + pOverlay->x2) / 2 - tw/2;
        ty = (pOverlay->y1 + pOverlay->y2) / 2 + th/4;
        osb->Canvas->TextOut( tx, ty, s );
        if( CwKeyer_Elbug.cfg.iDotTime_ms >= 1 )
         { s = "="+FormatFloat( "#0.0",
                     (double)(pOverlay->t2_ms - pOverlay->t1_ms)
                   / (double)CwKeyer_Elbug.cfg.iDotTime_ms) + " dots";
           osb->Canvas->TextOut( tx, ty+th, s );
         }

      }
   }


  // Draw the "windscreen wiper" to show the current position on the
  // non-scrolling, Radar-like sweeping display:
  x = (w * pScope->iLatestSampleIndex) / TIMING_SCOPE_NUM_SAMPLE_POINTS;
  if( x>=0 && x<w )
   { osb->Canvas->Pen->Color = clWhite;
     osb->Canvas->MoveTo( x, 0 );
     osb->Canvas->LineTo( x, h-15 );
   }

  pDestBitmap->Canvas->Draw( 0,0, osb );

  HERE_I_AM__GUI();

  delete osb; // clean up: delete the temporary "off-screen bitmap"

  free(pPoints);
  HERE_I_AM__GUI();

} // end UpdateTimingScope()

//---------------------------------------------------------------------------
static void HandleMouseEventInTimingScope( Graphics::TBitmap *pBitmap,
           int iEvent, int x, int y )
#        define EVT_MOUSE_DOWN 1
#        define EVT_MOUSE_MOVE 2
#        define EVT_MOUSE_UP   3
      // [out]  TimingScopeOverlay[] : mouse-controlled 'overlay' to measure times,
      //                               displayed in UpdateTimingScope()
{
  int w = pBitmap->Width;  // width in pixels
  int h = pBitmap->Height; // height in pixels
  int iSample, t_ms;
  T_TimingScopeOverlay *pOverlay;

  if( iEvent==EVT_MOUSE_DOWN ) // switch to the next overlay ?
   { TimingScope_iCurrentOverlay = (TimingScope_iCurrentOverlay+1) % N_SCOPE_OVERLAYS;
   }
  pOverlay = &TimingScopeOverlay[ TimingScope_iCurrentOverlay ];

  // Convert 'x' into a sample index, inverse to the following :
  //   x = (w * iSample) / TIMING_SCOPE_NUM_SAMPLE_POINTS;
  if( (w > 0 ) && (CwKeyer_TimingScope.iMillisecondsPerSample > 0 ) )
   { iSample = (x * TIMING_SCOPE_NUM_SAMPLE_POINTS) / w;
     // Convert the (relative) SAMPLE INDEX into an offset in milliseconds :
     t_ms = iSample * CwKeyer_TimingScope.iMillisecondsPerSample;
     if( iEvent == EVT_MOUSE_DOWN )
      { pOverlay->x1 = x;
        pOverlay->y1 = y;
        pOverlay->t1_ms = t_ms;
      }
     else //  EVT_MOUSE_MOVE or EVT_MOUSE_UP ->
      { pOverlay->x2 = x;
        pOverlay->y2 = y;
        pOverlay->t2_ms = t_ms;
        pOverlay->visible = TRUE;
      }
     TimingScope_iUpdateCountOnTestTab = -1; // kludge to redraw the timing scope..
     TimingScope_iUpdateCountOnKeyerTab= -1; // ... even when 'paused' .
   }
  (void)h;
} // end HandleMouseEventInTimingScope()

//---------------------------------------------------------------------------
static void ClearScopeOverlays(void)
{
  for(int i=0; i<N_SCOPE_OVERLAYS; ++i)
   { TimingScopeOverlay[i].visible = FALSE;
   }
  TimingScope_iUpdateCountOnTestTab = -1; // redraw a.s.a.p. ...
  TimingScope_iUpdateCountOnKeyerTab= -1; // ... even when 'paused' .
} // end ClearScopeOverlays()

//---------------------------------------------------------------------------
static void SetColourScheme( int iColourScheme )
{ // The venerable old VCL (Borland C++ Builder V6) does not support
  // "Schemes", "Themes", "Skins" or whatever. But for standard controls like
  // menus, tabsheets, edit fields, combo boxes and the like, Borland's
  // form designer / "object inspector" often uses colour names like e.g.
  //    "clBtnFace"    (funny name, often used as the entire form's
  //                    BACKGROUND COLOUR, for example in 'KeyerMainForm'),
  //    "clWindowText" (typically used for a TTabsheet's "Font.Color" property),
  //
  //    "clWindow"     (funny name for the WHITE background in combo boxes,
  //                    edit fields, and a couple of others).
  // The VCL documentation lists the above as ..
  //  > colors that are defined in the Windows Control panel.
  // Ok, but we don't want to fool around with the 'Windows Control panel'
  // (whatever / wherever that may be these days) - instead, THIS APPLICATION
  // shall of course only modify ITS OWN COLOURS .
  // There is no 'TStyleManager'. Didn't want to hook into Win32 "GetSysColor()".
  //   Maybe the Win32 "SetSysColors()" function will do the trick ?
  //   No, forget it, SetSysColors() affects the colours for ALL windows.

  // Thus: Dumbly iterate through all controls on all forms, find their TYPES,
  //       and depending on that, replace some of their colours:
  int iComp, iForm, nForms = Screen->FormCount;  // how many FORMS do we have ? [at least one]
  TForm      *pForm;
  TComponent *pComp;
  T_SpecDispControl *pDispCtrl = &g_SpecDispControl;
  DWORD dwErrorCode;
  char  sz255[256], *pszMsg;

  HERE_I_AM__GUI();
  pDispCtrl->clButtonFace = (DWORD)clBtnFace;    // form's BACKGROUND COLOUR - don't ask why
  pDispCtrl->clWindowText = (DWORD)clWindowText;
  pDispCtrl->clWindowBackground = (DWORD)clWindow;     // per default, WHITE background in e.g. combos and edit fields
  pDispCtrl->clActiveCaption  = (DWORD)clActiveCaption;   // > "color of the active window's title bar"
  pDispCtrl->clInactiveCaption= (DWORD)clInactiveCaption; // > "color of inactive windows' title bar"
  pDispCtrl->clButtonFont = pDispCtrl->clWindowText;

  switch( iColourScheme )
   { case COLOUR_SCHEME_DEFAULT : // back to e.g. "clBtnFace", etc etc etc etc
     default:
        // (most already set in the variable initialisers)
        g_clMenuBackground = TColor( 0xF0F0F0 ); // light-gray background
        g_clMenuSelBackgnd = TColor( 0xC08080 ); // light grayish-blue when selected
        g_clMenuForeground = TColor( 0x000000 ); // black text (regardless of selection)
        g_clDecoderOutputForeground = TColor( 0x000000 ); // black text
        g_clDecoderOutputBackground = TColor( 0xFFC0C0 ); // light-blue background
        break;
     case COLOUR_SCHEME_DARK : // for night owls and low-band CW ops
        pDispCtrl->clButtonFace    = TColor( 0x202020 ); // dark gray
        pDispCtrl->clWindowText = clWhite;
        pDispCtrl->clWindowBackground     = clBlack;
        pDispCtrl->clActiveCaption   = TColor( 0xC04040 ); // dark grayish-blue
        pDispCtrl->clInactiveCaption = TColor( 0x404040 ); // even darker, almost dead
        // As long as we cannot change a button's BACKGROUND COLOUR (a light gray)
        // it's a bad idea to set the button's FONT COLOR to ALMOST WHITE. Thus:
        pDispCtrl->clButtonFont = TColor( 0x804040 ); // dark grayish-blue
        g_clMenuBackground = TColor( 0x404040 );   // dark background
        g_clMenuSelBackgnd = TColor( 0xC04040 );   // dark grayish-blue when selected
        g_clMenuForeground = TColor( 0xE0E0E0 );   // white text
        g_clDecoderOutputForeground = TColor( 0xE0E0E0 ); // white text
        g_clDecoderOutputBackground = TColor( 0x800000 ); // dark blue background

        break;
   } // end switch( iColourScheme )

  for(iForm=0; iForm<nForms; ++iForm)
   {
     pForm = Screen->Forms[iForm];  // glorious VCL allows accessing them all 'like an array'
     pForm->Color = (TColor)pDispCtrl->clButtonFace;
     pForm->Font->Color = (TColor)pDispCtrl->clWindowText;
     for( iComp=0; (iComp<pForm->ComponentCount); iComp++ )
      { HERE_I_AM__GUI();
        pComp = pForm->Components[iComp];
        if (pComp->ClassNameIs("TMainMenu") )
         { // ex: TMainMenu *menu = (TMainMenu*)pComp;
           // Adapting the stupid main menu's back- and foreground colours
           // via e.g. "SetMenuInfo()" is absolutely hopeless. The crappy API
           // can modifiy the background of the DROPPED-DOWN items but not the
           // text colour, and it cannot modify the colours in the menu title.
         }
        else if (pComp->ClassNameIs("TMenuItem") )
         { // Unfortunately, the VCL designers at Borland forgot to add a
           // 'Color'/'Colour' property for this beast as well,
           // so without digging in AWFULLY DEEP into Win32 programming,
           // we cannot adapt the background of the title bar and main menu
           // to our 'Dark' colour scheme.
         }
        else if (pComp->ClassNameIs("TPopupMenu") )
         {
         }
        else if (pComp->ClassNameIs("TLabel") )
         { TLabel *label = (TLabel*)pComp;
           // Replace the TLabel's back- and foreground colour:
           label->Color = (TColor)pDispCtrl->clButtonFace;
           if( label->OnClick != NULL )
            { // "TLabel" acting like a 'link' (clickable) : BLUE
              if( iColourScheme == COLOUR_SCHEME_DARK )
               { label->Font->Color = clSkyBlue; // there's no clLightBlue :o(
               }
              else // white background -> dark blue text
               { label->Font->Color = clBlue;
               }
            }
           else  // not clickable -> no "link" but normal static text
            { label->Font->Color = (TColor)pDispCtrl->clWindowText;
            }
         } // end if <component is a TLabel>
        else if (pComp->ClassNameIs("TButton") )
         { TButton *btn = (TButton *)pComp;
           btn->Font->Color = (TColor)pDispCtrl->clWindowText;
         } // end if <component is a TButton>
        else if (pComp->ClassNameIs("TBitBtn") )
         { TBitBtn *btn = (TBitBtn *)pComp;
           // ex: btn->Font->Color = pDispCtrl->clWindowText;
           btn->Font->Color = (TColor)pDispCtrl->clButtonFont;
         } // end if <component is a TBitBtn>
        else if (pComp->ClassNameIs("TRadioButton") )
         { TRadioButton *btn = (TRadioButton *)pComp;
           btn->Font->Color = (TColor)pDispCtrl->clWindowText;
         } // end if <component is a TRadioButton>
        else if (pComp->ClassNameIs("TSpeedButton") )
         { // ex: TSpeedButton *btn = (TSpeedButton *)pComp;
           // A "speed button" neither has a foreground nor a background colour
           // that we can modify here. Modifying the FONT colour is nonsense
           // because a "speed button" is usually a square thing without any text,
           // but just a "glyph" (microscopic bitmap) that cannot be re-coloured.
         } // end if <component is a TSpeedButton>
        else if (pComp->ClassNameIs("TPanel") )
         { TPanel *pnl = (TPanel *)pComp;
           pnl->Color  = (TColor)pDispCtrl->clWindowBackground;  // !
           pnl->Font->Color = (TColor)pDispCtrl->clWindowText;
         }
        else if (pComp->ClassNameIs("TTabSheet") )
         { TTabSheet *ts = (TTabSheet *)pComp;
           ts->Font->Color = (TColor)pDispCtrl->clWindowText;
           // For obscure reasons, even the form's "Refresh()" didn't
           // cause the VCL TTabSheet to redraw its BACKGROUND . Nnngrrr.
           // ts->Refresh();  // <- tried this.. no effect
         } // end if <component is a TTabSheet>
        else if (pComp->ClassNameIs("TCheckBox") )
         { TCheckBox *chk = (TCheckBox *)pComp;
           chk->Color = (TColor)pDispCtrl->clButtonFace;  // !
           chk->Font->Color = (TColor)pDispCtrl->clWindowText;
         } // end if <component is a TCheckBox>
        else if (pComp->ClassNameIs("TGroupBox") )
         { TGroupBox *gb = (TGroupBox *)pComp;
           gb->Color       = (TColor)pDispCtrl->clWindowBackground;  // !
           gb->Font->Color = (TColor)pDispCtrl->clWindowText;
         }
        else if (pComp->ClassNameIs("TListBox") )
         { TListBox *lb = (TListBox *)pComp;
           lb->Color       = (TColor)pDispCtrl->clWindowBackground;
           lb->Font->Color = (TColor)pDispCtrl->clWindowText;
         } // end if <component is a TListBox>
        else if (pComp->ClassNameIs("TComboBox") )
         { TComboBox *cb = (TComboBox *)pComp;
           cb->Color       = (TColor)pDispCtrl->clWindowBackground;
           cb->Font->Color = (TColor)pDispCtrl->clWindowText;
         } // end if <component is a TComboBox>
        else if (pComp->ClassNameIs("TRadioGroup") )
         { TRadioGroup *rg = (TRadioGroup *)pComp;
           rg->Color       = (TColor)pDispCtrl->clWindowBackground;
           rg->Font->Color = (TColor)pDispCtrl->clWindowText;
         } // end if <component is a TRadioGroup>
        else if (pComp->ClassNameIs("TCheckListBox") )
         { TCheckListBox *cl = (TCheckListBox *)pComp;
           cl->Color       = (TColor)pDispCtrl->clWindowBackground;
           cl->Font->Color = (TColor)pDispCtrl->clWindowText;
         } // end if <component is a TCheckListBox>
        else if (pComp->ClassNameIs("TStringGrid") )
         { TStringGrid *sgr = (TStringGrid *)pComp;
           (void)sgr;
         } // end if <component is a TStringGrid>
        else if (pComp->ClassNameIs("TEdit") )
         { TEdit *ed = (TEdit *)pComp;
           ed->Color       = (TColor)pDispCtrl->clWindowBackground;
           ed->Font->Color = (TColor)pDispCtrl->clWindowText;
         } // end if <component is a TEdit>
        else if (pComp->ClassNameIs("TRichEdit") )
         { TRichEdit *ed = (TRichEdit *)pComp;
           ed->Color       = (TColor)pDispCtrl->clWindowBackground;
           ed->Font->Color = (TColor)pDispCtrl->clWindowText;
           ed->Clear();
         } // end if <component is a TRichEdit>
        else if (pComp->ClassNameIs("TScrollBar") )
         { TScrollBar *sb = (TScrollBar *)pComp;
           // Like many others, this poor VCL fellow also doen't have
           // an accessable 'Color' property...
           (void)sb;
         }
        else if (pComp->ClassNameIs("TTrackBar") )
         { TTrackBar *tb = (TTrackBar *)pComp;
           // This poor fellow doesn't even have an accessable 'Color' property !
           // C compiler said "Access to TControl::Color is impossible". Nnngrrr.
           // tb->Color = (TColor)pDispCtrl->clWindowBackground;
           (void)tb;
         } // end if <component is a TTrackBar>
        else if (pComp->ClassNameIs("TPageControl") )
         { // no attempt made to 'colourize' this yet ..
         }
        else if (pComp->ClassNameIs("TScrollBox") )
         { // no attempt made to 'colourize' this yet ..
         }
        else if (pComp->ClassNameIs("TImage") )
         { // no attempt made to 'colourize' this yet ..
         }
        else if (pComp->ClassNameIs("TBevel") )
         { // no attempt made to 'colourize' this yet ..
         }
        else if (pComp->ClassNameIs("TStaticText") )
         { // no attempt made to 'colourize' this yet ..
         }
        else if (pComp->ClassNameIs("TTimer") )
         { // no need to ever 'colourize' this, because a "TTimer" is never visible (except in the form designer)
         }
        else // kludge for development .. what's missing in the above list ?
         { AnsiString sClassName = pComp->ClassName();
           ShowError( ERROR_CLASS_ERROR, "SetColourScheme: Unknown class name %s",
                                         sClassName.c_str() );
         }
        HERE_I_AM__GUI();
      } // end for( iComp=0; (iComp<pForm->ComponentCount); iComp++ )
   } // end for<all forms of the application>
  HERE_I_AM__GUI();
  pDispCtrl->fUpdateSpectrum = pDispCtrl->fUpdateFreqScale = TRUE;

} // SetColourScheme()

//---------------------------------------------------------------------------
void UpdateSMeter(
        TImage *pImg,  // [in,out] Borland-VCL-style "graphic image", with a "canvas" to paint into
        double dblSMeterLevel_dB ) // [in] "decibel over S0"
        // Keep it simple, begin at "S zero" :
#define C_SMETER_LEVEL_S0   0 /* "S0" on HF:       -20 dBuV */
#define C_SMETER_LEVEL_S9  54 /* "S9" on HF:       +34 dBuV */
#define C_SMETER_LEVEL_MAX 99 /* "S9+45 dB"                 */
{
  int h, w, w2, th, tw, dB, x1, x, xt, x_S9, x_Lev;
  AnsiString sLabel;
  BOOL fShowLabel;

  // Borland's/Embercadero's VCL-style "TImage" contains a "TPicture".
  //    A "TPicture" is a "TGraphic" container; here it's a "TBitmap".
  //    The "TBitmap" has a "TCanvas", which is the thing we can actually paint on.
  TCanvas *pCanvas;

  // Make sure the "TBitmap" has the same pixel dimensions as the "TImage":
  pImg->Picture->Bitmap->Height = h = pImg->Height;
  pImg->Picture->Bitmap->Width  = w = pImg->Width;
  pCanvas = pImg->Picture->Bitmap->Canvas;

  // Prepare a sufficiently SMALL font for the S-Meter labels:
  pCanvas->Font->Name  = "Arial";
  pCanvas->Font->Size  = 6;
  pCanvas->Font->Color = (TColor)g_SpecDispControl.clWindowText; // <- this may be WHITE or BLACK .. good contrast against the bargraph colours and clWindowBackground
  th = pCanvas->TextHeight( "S" );

  x1 = pCanvas->TextWidth( "S" ); // wide enough to draw e.g. "S0" for the leftmost tick
  w2 = w - x1 - 4;

  // From the width (w, in pixels), calculate the end of the coloured bargraph:
  x_Lev = x1 + (int)((double)(dblSMeterLevel_dB  - C_SMETER_LEVEL_S0) * (double)w2
                   / (double)(C_SMETER_LEVEL_MAX - C_SMETER_LEVEL_S0) );
  if( x_Lev<0 ) x_Lev=0;
  if( x_Lev>=w) x_Lev=w-1;
  // Calculate the pixel position for "S9"
  // (where Icom's S-meter changes from lightblue to red)
  x_S9 = x1 + (int)((double)(C_SMETER_LEVEL_S9  - C_SMETER_LEVEL_S0) * (double)w2
                  / (double)(C_SMETER_LEVEL_MAX - C_SMETER_LEVEL_S0) );

  // Draw the "bargraph" into the BACKGROUND ..
  x = x_Lev;
  if( x > x_S9 )
   {  x = x_S9;
   }
  pCanvas->Brush->Color = clSkyBlue;    // colours inspired by Icom IC-7300 ..
  pCanvas->FillRect( TRect(x1,0,x,h) ); // below S9 = light blue
  if( x_Lev > x_S9 )
   { pCanvas->Brush->Color = clRed;    // above S9 = red
     pCanvas->FillRect( TRect(x_S9,0, x_Lev,h) );
   }
  pCanvas->Brush->Color = (TColor)g_SpecDispControl.clWindowBackground; // <- this may be BLACK or WHITE !
  pCanvas->FillRect( TRect(0,0, x1,h) );
  pCanvas->FillRect( TRect(x_Lev,0, w,h) );


  // Draw a crude "S-meter scale" into the foreground
  SetBkMode( pCanvas->Handle, TRANSPARENT ); // <- old Windows 'GDI' function .. what's the VCL equivalent ?
  pCanvas->Pen->Color = pCanvas->Font->Color;
  for( dB=C_SMETER_LEVEL_S0; dB<=C_SMETER_LEVEL_S9; dB+=6 )
   { x = x1 + (int)((double)( dB - C_SMETER_LEVEL_S0) * (double)w2
                  / (double)(C_SMETER_LEVEL_MAX - C_SMETER_LEVEL_S0) );
     pCanvas->MoveTo( x, 0 );
     pCanvas->LineTo( x, h-th );
     if(dB==C_SMETER_LEVEL_S0)
      { sLabel = "S0";
      }
     else
      { sLabel = IntToStr( (dB-C_SMETER_LEVEL_S0) / 6 );
      }
     tw = pCanvas->TextWidth( sLabel );
     xt = x-tw/2;
     pCanvas->TextOut( xt, h-th, sLabel );
   }
  // Above "S9" : only show "+x" every TWENTY dB (like Icom)
  for( dB=C_SMETER_LEVEL_S9+10; dB<=C_SMETER_LEVEL_MAX; dB+=10 )
   { x = x1 + (int)((double)( dB - C_SMETER_LEVEL_S0) * (double)w2
                  / (double)(C_SMETER_LEVEL_MAX - C_SMETER_LEVEL_S0) );
     pCanvas->MoveTo( x, 0 );
     pCanvas->LineTo( x, h-th );
     sLabel = AnsiString("+") + IntToStr( dB-C_SMETER_LEVEL_S9 );
     tw = pCanvas->TextWidth( sLabel );
     xt = x-tw/2;
     pCanvas->TextOut( xt, h-th, sLabel );
   }


} // end UpdateSMeter()

//---------------------------------------------------------------------------
void UpdateMultiFunctionMeter( // renders any of the horizontal bargraphs on the "Mult-funtion meter" (panel)
        TImage *pImg,  // [in,out] Borland-VCL-style "graphic image", with a "canvas" to paint into
        int x1, int y1, int x2, int y2, // [in] graphic area for one of the six(?) "meters" within pImg
        double dblValue, // [in] value to be displayed (percent, "dB", SWR, current, voltage, power, temperature..)
        double dblMinValue, double dblMaxValue, double dblStepwidth, // [in] physical value range (also used for the labelled scale)
        char *pszParamName, // [in] name like "Power", "ALC", "COMP", "SWR", "Id", "Vd", "Temp"
        char *pszPhysUnit,  // [in] physical unit (shown at the end of the tick scale), e.g. "%", "dB", "V", "A", "W", "C"
        double dblRedStartValue, double dblRedEndValue ) // e.g. 10.0 .. 12.0 [Volts for the "red line" indicating UNDERVOLTAGE]

  // Result (approximately..)
  //
  //    ,----------------------------------------,   -  y1
  //    | Power ##|##|##|##|##|##|##|##|##|##| % |
  //    |       0 10 20 30 40 50 60 70 80 90 100 |
  //    '----------------------------------------'   -  y2
  //    |       |                           |    |
  //    x1     x1i ("inner bargraph area"  x2i   x2

{
  int x1i, y1i, x2i, y2i, h, w, w2, th, tw, dB, x, xt, yt;
  int x_Lev, x_RedStart, x_RedEnd;
  double d;
  AnsiString sLabel;
  BOOL fShowLabel;
  BOOL fEraseAll = FALSE;

#define L_COLOUR_TEST 0 // (0)=normal compilation, (1)=TEST using fixed colours

  // Borland's/Embercadero's VCL-style "TImage" contains a "TPicture".
  //    A "TPicture" is a "TGraphic" container; here it's a "TBitmap".
  //    The "TBitmap" has a "TCanvas", which is the thing we can actually paint on.
  TCanvas *pCanvas;


  // Make sure the "TBitmap" has the same pixel dimensions as the "TImage":
  h = pImg->Height;
  w = pImg->Width;
  if( (pImg->Picture->Bitmap->Height != h) || (pImg->Picture->Bitmap->Width != w) )
   { fEraseAll = TRUE;
     pImg->Picture->Bitmap->Height = h;
     pImg->Picture->Bitmap->Width  = w;
   }
  pCanvas = pImg->Picture->Bitmap->Canvas;

  if( fEraseAll )
   {
#   if( L_COLOUR_TEST ) // Fill the entire bitmap with a painful colour ?
     pCanvas->Brush->Color = clRed;
#   else  // no eye-cancer please ..
     pCanvas->Brush->Color = (TColor)g_SpecDispControl.clWindowBackground; // <- this may be BLACK or WHITE !
#   endif
     pCanvas->Brush->Style = bsSolid;
     pCanvas->FillRect( TRect(0,0,w,h) );
   }

  // return;  // what caused the WHITE-filled area ? [not the stuff below..]

  // Prepare a sufficiently SMALL font for the S-Meter labels:
  pCanvas->Font->Name  = "Arial";
  pCanvas->Font->Size  = 6;
#if( L_COLOUR_TEST )
  pCanvas->Font->Color = clYellow;
#else
  pCanvas->Font->Color = (TColor)g_SpecDispControl.clWindowText; // <- this may be WHITE or BLACK .. good contrast against the bargraph colours and clWindowBackground
#endif // L_COLOUR_TEST ?
  th = pCanvas->TextHeight( "S" );
  tw = pCanvas->TextWidth( "S" ); // wide enough to draw e.g. "S0" for the leftmost tick

  // "inner width" for the bargraph, without margins for the labels and :
  x1i = x1 + 6*tw;
  x2i = x2 - 2*tw;
  y1i = y1 + 2;
  y2i = y2 - 2;
  w2  = x2i - x1i;

  // From the "inner width" (w2 in pixels), calculate the end of the coloured bargraph:
  x_Lev = x1i + (int)((double)(dblValue    - dblMinValue) * (double)w2
                    / (double)(dblMaxValue - dblMinValue) );
  UTL_LimitInteger( &x_Lev, x1, x2 );
  // Calculate the pixel position for "S9"
  // (where Icom's S-meter changes from lightblue to red)
  x_RedStart = x1i + (int)((double)(dblRedStartValue - dblMinValue) * (double)w2
                         / (double)(dblMaxValue      - dblMinValue) );
  x_RedEnd = x1i + (int)((double)(dblRedEndValue - dblMinValue) * (double)w2
                       / (double)(dblMaxValue    - dblMinValue) );
  UTL_LimitInteger( &x_RedStart, x1i, x2i );
  UTL_LimitInteger( &x_RedEnd,   x1i, x2i );
  // Note: For SOME "meters", the RED AREA is at the start (e.g. undervoltage).
  //       For OTHER "meters", the RED AREA is at the end (e.g. PA temperature).
  //       The "red area" is indicated as a line below the bargraph.
  //       Unlike the S-meter, the colour of the bargraph itself doesn't seem
  //       to change depending on the displayed value (?)

  // Clear the entire background (no problem with flicker; we use "double buffering")
#if( L_COLOUR_TEST )
  pCanvas->Brush->Color = clBlue;
#else
  pCanvas->Brush->Color = (TColor)g_SpecDispControl.clWindowBackground; // <- this may be BLACK or WHITE !
#endif // L_COLOUR_TEST ?
  pCanvas->Brush->Style = bsSolid;
  pCanvas->FillRect( TRect(x1,y1,x2, y2) );
    // ,-------------'
    // '--> "TRect represents the dimensions of a rectangle. The coordinates
    //       are specified as either four separate integers representing the
    //       left, top, right, and bottom sides, or as two points (...) "

#if( L_COLOUR_TEST )
  pCanvas->Brush->Color = clGreen;
#else
  pCanvas->Brush->Color = (TColor)g_SpecDispControl.clWindowText; // <- this may be BLACK or WHITE !
#endif // L_COLOUR_TEST ?

  pCanvas->FrameRect( TRect(x1+2,y1+2,x2-2,y2-2) ); // quite counter-intuitive, "FrameRect()" doesn't use the "Pen" but the "Brush" to draw a 1-pixel-wide border..

#if( L_COLOUR_TEST )
  pCanvas->Font->Color = (TColor)0x00C0FF; // "clOrange", but that doesn't exist in the VCL
#else
  pCanvas->Font->Color = (TColor)g_SpecDispControl.clWindowText;
#endif // L_COLOUR_TEST ?
  SetBkMode( pCanvas->Handle, TRANSPARENT ); // <- Windows 'GDI' .. what's the VCL equivalent ?
  if( (pszPhysUnit != NULL ) && (pszPhysUnit[0] != '\0') )
   { yt = (y1+y2)/2-th;  // vertical position for the parameter name ..
   }
  else // no "physical unit" so vertically align the PARAMETER NAME:
   { yt = (y1+y2)/2-th/2; // <- for example, SWR and ALC were dimensionless
   }
  if( pszParamName != NULL )
   { pCanvas->TextOut( x1 + 4, yt, pszParamName );
     yt += th;
   }
  if( (pszPhysUnit != NULL ) && (pszPhysUnit[0] != '\0') )
   { pCanvas->TextOut( x1 + 8, yt, "/ " + AnsiString(pszPhysUnit) );
     yt += th;
   }
  SetBkMode( pCanvas->Handle, OPAQUE );

#if( L_COLOUR_TEST )
  pCanvas->Brush->Color = clGray;
#else
  pCanvas->Brush->Color = (TColor)g_SpecDispControl.clWindowBackground; // <- this may be BLACK or WHITE !
#endif // L_COLOUR_TEST ?


  // Draw the "bargraph" into the BACKGROUND ..
  x = x_Lev;
  if( x>x1i )
   { pCanvas->Brush->Color = clSkyBlue;         // colours inspired by Icom IC-7300 ..
     pCanvas->FillRect( TRect(x1i,y1i,x,y2i) ); // light blue (for MOST bargraphs)
   }

  // Draw a crude "labelled scale" into the foreground ?
  if( (dblStepwidth>0.0) && (dblMaxValue>dblMinValue) )
   {
#   if( L_COLOUR_TEST )
     pCanvas->Font->Color = (TColor)0x00C0FF; // "clOrange", but that doesn't exist in the VCL
#   else
     pCanvas->Font->Color = (TColor)g_SpecDispControl.clWindowText;
#   endif // L_COLOUR_TEST ?
     pCanvas->Pen->Color = pCanvas->Font->Color;
     SetBkMode( pCanvas->Handle, TRANSPARENT ); // <- old Windows 'GDI' function .. what's the VCL equivalent ?
     for( d=dblMinValue; d<=dblMaxValue; d += dblStepwidth )
      { x = x1i + (int)((double)( d          - dblMinValue) * (double)w2
                      / (double)(dblMaxValue - dblMinValue) );
        pCanvas->MoveTo( x, y1i );
        pCanvas->LineTo( x, y2i-th );
        if( dblStepwidth >= 1.0 )
         { sLabel = IntToStr( (int)d );
         }
        else
         { sLabel = FormatFloat( "#.0", d );
         }
        tw = pCanvas->TextWidth( sLabel );
        xt = x-tw/2;
        pCanvas->TextOut( xt, y2i-th, sLabel );
      }
     SetBkMode( pCanvas->Handle, OPAQUE );

   }

} // end UpdateMultiFunctionMeter()

//---------------------------------------------------------------------------
const char* RigCtrl_GetRadioControlPortAsString(void) // -> e.g. "COM5", if that's the port "talking to the radio" .
  // Called from withing RigControl.c to emit 'info messages' like
  //     "RigControl: IC-9700 on COM5, 432250.0 kHz, CWN"  .
  // Since RighControl.c doesn't know anything about serial ports,
  // RigCtrl_GetRadioControlPortAsString() must be provided by the application.
{
  static char sz15Result[16];

  // If the "COM port to 'key' the radio" is ONLY used for keying,
  //             but not to CONTROL the radio, the Remote CW Keyer
  //             can alternatively control the radio via Hamlib's "rigctld"
  //             ('Rig Control Daemon' which is basically 'Hamlib control
  //               commands in ASCII sent over TCP/IP; details at
  //               github.com/Hamlib/Hamlib/wiki/Network-Device-Control ).
  //             Other applications (like wfview) can also EMULATE
  //             Hamlib's "rigctld", with a TCP server listening on port 4532.
  switch( CwKeyer_Config.iRadioControlProtocol )
   { case RIGCTRL_PROTOCOL_NONE : return "NoPort";
     case RIGCTRL_PROTOCOL_ICOM_CI_V: // supported since the early beginnings..
        // HERE, to keep things simple in the Remote CW Keyer,
        // Icom's CI-V protocol is only supported via "the same local COM port
        // that also KEYS the rig (provides the Morse code modulation):
        sprintf( sz15Result, "COM%d", (int)CwKeyer_Config.iRadioKeyingAndControlPort );
        return sz15Result;
     // case RIGCTRL_PROTOCOL_KENWOOD, RIGCTRL_PROTOCOL_YAESU_CAT, RIGCTRL_PROTOCOL_ELECRAFT, ...
     // NONE of these will ever be supported natively .. so instead,
     //   use use something Hamlib/Rigctld-compatible that runs a server on a
     //   TCP/IP port, e.g wfview's built-in "rigctld emulation" !
     //
     case RIGCTRL_PROTOCOL_HAMLIB_RIGCTLD : // added 2024-06 in the Remote CW Keyer...
     // ex: case RIGCTRL_PROTOCOL_FLRIG_XMLRPC   :
        // Doesn't use the built-in CI-V engine in DL4YHF's RigControl.c,
        // but the ASCII-based 'Hamlib' compatible commands via TCP/IP:
        return CwKeyer_Config.sz80RemoteRigCtrlServerAddress;

     default:
        sprintf( sz15Result, "??%d??", (int)CwKeyer_Config.iRadioControlProtocol );
        return sz15Result;

   } // end switch( CwKeyer_Config.iRadioControlProtocol )

} // end RigCtrl_GetRadioControlPortAsString()


//---------------------------------------------------------------------------
// Above : Functions used by the 'main form', and maybe other modules.
// Below : Implementation of the 'main form' itself (a Borland VCL thing).
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
__fastcall TKeyerMainForm::TKeyerMainForm(TComponent* Owner) : TForm(Owner)
{
  m_fInitialising = TRUE;
  m_iUpdating   = 1; // <- cleared when returning from TKeyerMainForm::FormCreate()
  m_iCurrentMainTab = MAIN_TAB_UNKNOWN; // also contains one of the MAIN_TAB_.. constants, set via SwitchMainTab()
  m_fAutomaticTabSwitching = TRUE; // allow *automatic* switching of the main tab until the 'rig initialisation' is finished.

  APPL_hInstance = HInstance; // handle to current instance, a Win32 thing.
     // > "HInstance contains the instance handle of the application
     // >  or library as provided by the Windows environment."
     // Tolle Beschreibung, gell ? Jetzt wissen wir auch GANZ GENAU,
     // was dieses verdammte "HInstance"-Dingsbums ist, und wer es wozu braucht.
     // This "HInstance"-thingy, provided by the VCL, is required by certain
     // Win32 API function calls like CreateWindow(). We store it in an
     // old-fashioned global variable to make it accessable for NON-VCL modules,
     // e.g. YHF_Dialogs.cpp .
} // end TKeyerMainForm::TKeyerMainForm()

//---------------------------------------------------------------------------
void TKeyerMainForm::UpdateSettingsTab(void) // later, "I/O Config" and "Keyer Settings"..
{
  int i;

  ++m_iUpdating; // prevent interfering in certain "OnClick()" and "OnChange()" handlers
  HERE_I_AM__GUI();
  FillComboWithSerialPorts( CB_MorseKeyPort, CwKeyer_Config.iComPortNumber_IN );
  FillComboWithSerialPorts( CB_RadioKeyingPort, CwKeyer_Config.iRadioKeyingAndControlPort );
  FillComboWithItemsFromTokenLists( CB_DotKeyInput,
        C_TL_None_Zero,                // [in] const T_SL_TokenList *pTokens1 : "None" -> 0
        SerialPortInputSignals_Key,    // [in] const T_SL_TokenList *pTokens2
        SerialPortInputSignals_Radio,  // [in] const T_SL_TokenList *pTokens3
        CwKeyer_Config.iDotInput );    // [in] int iSelectedTokenValue
  FillComboWithItemsFromTokenLists( CB_DashKeyInput,
        C_TL_None_Zero,                // [in] const T_SL_TokenList *pTokens1 : "None" -> 0
        SerialPortInputSignals_Key,    // [in] const T_SL_TokenList *pTokens2
        SerialPortInputSignals_Radio,  // [in] const T_SL_TokenList *pTokens3
        CwKeyer_Config.iDashInput );
  FillComboWithItemsFromTokenLists( CB_PTT_Input,
        C_TL_None_Zero,                // [in] const T_SL_TokenList *pTokens1 : "None" -> 0
        SerialPortInputSignals_Key,    // [in] const T_SL_TokenList *pTokens2
        SerialPortInputSignals_Radio,  // [in] const T_SL_TokenList *pTokens3
        CwKeyer_Config.iManualPTTInput );
  FillComboWithItemsFromTokenLists( CB_TestInput,
        C_TL_None_Zero,                // [in] const T_SL_TokenList *pTokens1 : "None" -> 0
        SerialPortInputSignals_Key,    // [in] const T_SL_TokenList *pTokens2
        SerialPortInputSignals_Radio,  // [in] const T_SL_TokenList *pTokens3
        CwKeyer_Config.iTestInput );
  FillComboWithItemsFromTokenLists( CB_InputKeySupply,
        C_TL_None_Zero,                // [in] const T_SL_TokenList *pTokens1 : "None" -> 0
        SerialPortOutputSignals_Key,   // [in] const T_SL_TokenList *pTokens2
        SerialPortOutputSignals_Radio, // [in] const T_SL_TokenList *pTokens3
        CwKeyer_Config.iKeySupply );
  FillComboWithItemsFromTokenLists( CB_RadioCWKeying,
        C_TL_None_Zero,                // [in] const T_SL_TokenList *pTokens1 : "None" -> 0
        SerialPortOutputSignals_Radio, // [in] const T_SL_TokenList *pTokens2
        SerialPortOutputSignals_Key,   // [in] const T_SL_TokenList *pTokens3
        CwKeyer_Config.iRadioCWKeying );
  FillComboWithItemsFromTokenLists( CB_RadioPTTControl,
        C_TL_None_Zero,                // [in] const T_SL_TokenList *pTokens1 : "None" -> 0
        SerialPortOutputSignals_Radio, // [in] const T_SL_TokenList *pTokens2
        SerialPortOutputSignals_Key,   // [in] const T_SL_TokenList *pTokens3
        CwKeyer_Config.iRadioPTTControl);

  // The following controls were moved to an extra sub-tabsheet ("Remote Control")
  //   when the space on the 'I/O Config' tab got too small:
  FillComboWithItemsFromTokenLists( CB_RemoteControlMethod,
        RigControlMethods, // [in] const T_SL_TokenList *pTokens1 (RIGCTRL_PROTOCOL_NONE/RIGCTRL_PROTOCOL_ICOM_CI_V/RIGCTRL_PROTOCOL_HAMLIB_RIGCTLD,..(?) )
        NULL,              // [in] const T_SL_TokenList *pTokens2
        NULL,              // [in] const T_SL_TokenList *pTokens3
        CwKeyer_Config.iRadioControlProtocol); // .. aka "Remote Control Method" for RigControl.c
  SelectComboItemByDecimalValue( CB_RadioCtrlBaudrate, CwKeyer_Config.iRadioControlBaudrate );
  if(CwKeyer_Config.iRadioCIVAddress <= RIGCTRL_DEF_ADDR_NO_REMOTE_CTRL)  // negative "radio address" : no CI-V-controllable radio at all
   { CB_CIV_Address->ItemIndex = 0; // ItemIndex 0 = "Rig Control OFF"
   }
  else if( CwKeyer_Config.iRadioCIVAddress == RIGCTRL_DEF_ADDR_AUTO_DETECT ) // "radio address" ZERO : AUTO-DETECT (in RigControl.h :
   { CB_CIV_Address->ItemIndex = 1; // ItemIndex 1 = "AutoDetect" ..
   }
  else
   { SelectComboItemByHexadecimalValue( CB_CIV_Address, CwKeyer_Config.iRadioCIVAddress );
   }
#if( SWI_USE_HAMLIB_SERVER ) // not sure if this 'feature' is going to stay...
  Ed_BuiltInHamlibServerPort->Text = IntToStr( MyHamlibServer.cfg.iServerListeningPort );
  Chk_EnableBuiltInHamlibServer->Checked = CwKeyer_Config.fEnableBuiltInHamlibServer;
#endif // SWI_USE_HAMLIB_SERVER ?
  UpdateRigControlMethodDependingFields();

  // Moved to a new tabsheet titled "Keyer Settings" :
  HERE_I_AM__GUI();
  FillComboWithItemsFromTokenLists( CB_KeyType, KeyTypes, NULL, NULL, CwKeyer_Config.iMorseKeyType );
  // The INTERNAL unit for the elbug timing is MILLISECONDS.
  // Only for the "GUI", the speed is displayed in "Words per Minute" .
  // With a thread loop time of 2 milliseconds, speeds above 48 WPM are possible,
  // but not ALL. For example, 59 WPM = 1200 ms / 59 = 20.33 ms;
  //                           60 WPM = 1200 ms / 60 = 20.00 ms.
  i = Elbug_DotTimeInMillisecondsToWordsPerMinute( CwKeyer_Elbug.cfg.iDotTime_ms );
  TrackBar_WPM->Position = i;   // Old-school "track bar" (a la Borland VCL), with up/down via keyboard
  Ed_WPM->Text = IntToStr( i ); // NUMERIC edit field for the speed in WPM
  Ed_TxDelayTime->Text = IntToStr( CwKeyer_Config.iTxDelayTime_ms );
  Ed_TxHangTime->Text  = IntToStr( CwKeyer_Config.iTxHangTime_ms );

  FillComboWithItemsFromTokenLists( CB_SidetoneOnTXD, SidetonesOnTXD, NULL, NULL, CwKeyer_Config.iSidetoneOnTXD );
  if( ! FillComboWithItemsFromTokenLists( CB_SidetoneOnAudioOut, SidetonesOnAudioOut, NULL, NULL, CwKeyer_DSP.cfg.iSidetoneFreq_Hz) )
   { CB_SidetoneOnAudioOut->Text = IntToStr(CwKeyer_DSP.cfg.iSidetoneFreq_Hz) + " Hz";
     // (this only works because in the form designer, this Combo-Box has the
     //  'csDropDown' style, not just 'csDropDownList' )
   }
  if( ! FillComboWithItemsFromTokenLists( CB_SidetoneRiseTime, SidetoneRiseTimes, NULL, NULL, CwKeyer_DSP.cfg.iSidetoneRiseTime_ms) )
   { CB_SidetoneRiseTime->Text = IntToStr(CwKeyer_DSP.cfg.iSidetoneRiseTime_ms) + " ms";
     // Technically, this isn't just a "rise" time but a "ramp" time.
     // But Icom coined the name (it's in their 'CW-KEY SET' menu).
   }
  Chk_Audio_AllowNetwork->Checked = (CwKeyer_DSP.cfg.iAudioFlags & DSP_AUDIO_FLAGS_ALLOW_NETWORK_AUDIO) != 0;
  UTL_LimitInteger( &CwKeyer_DSP.cfg.iAudioInGain_dB, -20, 20 );
  SB_AudioInGain_dB->Position = -CwKeyer_DSP.cfg.iAudioInGain_dB;
  UTL_LimitInteger( &CwKeyer_DSP.cfg.iAudioOutGain_dB, -20, 20 );
  SB_AudioOutGain_dB->Position = -CwKeyer_DSP.cfg.iAudioOutGain_dB;
  UTL_LimitInteger( &CwKeyer_DSP.cfg.iSidetoneGain_dB, -20, 20 );
  SB_SidetoneGain_dB->Position = -CwKeyer_DSP.cfg.iSidetoneGain_dB; // invert, because "min" and "max" were reversed in the "TScrollBar"
  UTL_LimitInteger( &CwKeyer_DSP.cfg.iNetworkTonesGain_dB, -20, 20 );
  SB_NetworkTonesGain_dB->Position = -CwKeyer_DSP.cfg.iNetworkTonesGain_dB;

  --m_iUpdating;
  HERE_I_AM__GUI();

} // end TKeyerMainForm::UpdateSettingsTab()

//---------------------------------------------------------------------------
void TKeyerMainForm::ApplySettingsTab(void)
{
  const char *pszSrc;
  CwKeyer_Config.iComPortNumber_IN = ParseSerialPortNumberFromText( CB_MorseKeyPort->Text.c_str() );
  CwKeyer_Config.iRadioKeyingAndControlPort= ParseSerialPortNumberFromText( CB_RadioKeyingPort->Text.c_str() );
  CwKeyer_Config.iRadioControlProtocol = SL_FindStringInTable( RigControlMethods, CB_RemoteControlMethod->Text.c_str() );
  CwKeyer_Config.iRadioControlBaudrate = StrToIntDef(CB_RadioCtrlBaudrate->Text, CwKeyer_Config.iRadioControlBaudrate );
  if( CB_CIV_Address->ItemIndex == 0 )  // 0 = "Rig Control OFF", 1 = "AutoDetect" ..
   { CwKeyer_Config.iRadioCIVAddress = RIGCTRL_DEF_ADDR_NO_REMOTE_CTRL; // CwKeyer_Config.iRadioCIVAddress = -1 = "OFF (No RigControl)"
   }
  else if( CB_CIV_Address->ItemIndex == 1 )
   { CwKeyer_Config.iRadioCIVAddress = RIGCTRL_DEF_ADDR_AUTO_DETECT;  // CwKeyer_Config.iRadioCIVAddress = 0 = "Auto-Detect"
   }
  else // expecting a HEXADECIMAL CI-V address (usually with two digits) ->
   { pszSrc = CB_CIV_Address->Text.c_str();
     if( SL_SkipString( &pszSrc, "0x" ) ) // looks like a HEX number ...
      { CwKeyer_Config.iRadioCIVAddress= SL_ParseHex( &pszSrc, 4/*nMaxDigits*/ );
      }
     else
      { CwKeyer_Config.iRadioCIVAddress= SL_ParseInteger( &pszSrc );
      }
   }
  strncpy( CwKeyer_Config.sz80RemoteRigCtrlServerAddress, Ed_RemoteRigctrlServerAddress->Text.c_str(), 80 );

#if( SWI_USE_HAMLIB_SERVER ) // not sure if this 'feature' is going to stay...
  MyHamlibServer.cfg.iServerListeningPort = StrToIntDef( Ed_BuiltInHamlibServerPort->Text, HLSRV_DEFAULT_SERVER_PORT );
  CwKeyer_Config.fEnableBuiltInHamlibServer= Chk_EnableBuiltInHamlibServer->Checked;
#endif // SWI_USE_HAMLIB_SERVER ?

  CwKeyer_Config.iDotInput       = GetSerialPortSignalIndexFromComboBox( CB_DotKeyInput    );
  CwKeyer_Config.iDashInput      = GetSerialPortSignalIndexFromComboBox( CB_DashKeyInput   );
  CwKeyer_Config.iManualPTTInput = GetSerialPortSignalIndexFromComboBox( CB_PTT_Input      );
  CwKeyer_Config.iTestInput      = GetSerialPortSignalIndexFromComboBox( CB_TestInput      );
  CwKeyer_Config.iKeySupply      = GetSerialPortSignalIndexFromComboBox( CB_InputKeySupply );
  CwKeyer_Config.iRadioCWKeying  = GetSerialPortSignalIndexFromComboBox( CB_RadioCWKeying  );
  CwKeyer_Config.iRadioPTTControl= GetSerialPortSignalIndexFromComboBox( CB_RadioPTTControl);
} // end TKeyerMainForm::ApplySettingsTab()

//---------------------------------------------------------------------------
void TKeyerMainForm::UpdateRigControlMethodDependingFields(void)
{ int iNewRigControlMethod = SL_FindStringInTable( RigControlMethods, CB_RemoteControlMethod->Text.c_str() );
  switch( iNewRigControlMethod )
   { case RIGCTRL_PROTOCOL_NONE:
        Lab_RigCtrl_CIV_Addr->Enabled       = FALSE;
        Lab_RemoteRigControlServer->Enabled = FALSE;
        break;

     case RIGCTRL_PROTOCOL_ICOM_CI_V:
        Lab_RigCtrl_CIV_Addr->Enabled       = TRUE;
        Lab_RemoteRigControlServer->Enabled = FALSE;
        break;

     case RIGCTRL_PROTOCOL_HAMLIB_RIGCTLD:
     // ex: case RIGCTRL_PROTOCOL_FLRIG_XMLRPC: // removed, far too complex
        Lab_RigCtrl_CIV_Addr->Enabled       = FALSE;
        Lab_RemoteRigControlServer->Enabled = TRUE;
        break;
   }
} // end TKeyerMainForm::UpdateRigControlMethodDependingFields()

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::CB_RemoteControlMethodClick(TObject *Sender)
{ // Depending on the Rig-Control "Method / Protocol",
  // several visual controls on the 'Config' / 'Rig Control' tab
  // may be disabled of enabled. For example, if the rig is controlled
  // via serial port ("COM port") and CI-V protocol, the 'CI-V Address'
  // is important.
  // If the rig is controlled via an external application
  // that provides a Hamlib/Rigctld (d for 'daemon' ?) compatible server,
  // the server's URL with optional numeric IP address and remote TCP port
  // is important.
  if( m_iUpdating==0 )
   { UpdateRigControlMethodDependingFields();
   }  // end if( m_iUpdating==0 )
}
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
void TKeyerMainForm::UpdateMultiFunctionMeters(void) // [out] Img_TRX_Bottom
{
  TImage *pImg = Img_TRX_Bottom;
  T_CwNet *pCwNet = &MyCwNet;   // [in] MyCwNet.RigControl.iPowerMeterLevel_pcnt, etc

  int iSingleMeterWidth = pImg->Width / 3;
  int iSingleMeterHeight= pImg->Height / 2;

  // Take "snapshots" of the various "meter" readings displayed in the GUI:
  m_iDisplayedPowerMeterLevel_pcnt = pCwNet->RigControl.iPowerMeterLevel_pcnt;
  m_dblDisplayedSWRMeterValue      = pCwNet->RigControl.dblSWRMeterValue;  // dimensionless; usually ranging from 1.0 to 3.0 (from IC-7300)
  m_iDisplayedALCMeterLevel_pcnt   = pCwNet->RigControl.iALCMeterLevel_pcnt;
  m_iDisplayedCompMeterLevel_dB    = pCwNet->RigControl.iCompMeterLevel_dB;
  m_iDisplayedSupplyVoltage_mV     = pCwNet->RigControl.iSupplyVoltage_mV;
  m_iDisplayedDrainCurrent_mA      = pCwNet->RigControl.iDrainCurrent_mA;
  m_iDisplayediPATemperature_degC  = pCwNet->RigControl.iPATemperature_degC; // <- one fine day we know how to read this via CI-V ..

  // First meter (top left) : TX POWER. Not in WATTS but in "percent of the maximum" (thanks Icom):
  UpdateMultiFunctionMeter( pImg,
     0/*x1*/, 0/*y1*/, iSingleMeterWidth/*x2*/, iSingleMeterHeight/*y2*/, // <- graphic area for one of the six(?) "meters" within pImg
     (double)m_iDisplayedPowerMeterLevel_pcnt, // [in] value to be displayed
     0.0/*dblMinValue*/, 100.0/*dblMaxValue*/, 10.0/*dblStepwidth*/, // [in] physical value range (also used for the labelled scale)
     "Power",  // [in] char *pszParamName, e.g. "Power", "ALC", "COMP", "SWR", "Id", "Vd", "Temp"
     "%",      // [in] char *pszPhysUnit, physical unit (shown at the end of the tick scale), e.g. "%", "dB", "V", "A", "W", "C"
     0.0/*dblRedStartValue*/, 0.0/*dblRedEndValue*/ ); // e.g. 10.0 .. 12.0 [range for an optional "red line" indicating a critical range]

  // Second meter (top middle) : SWR meter. "Right next to the power meter, that's where it belongs !"
  UpdateMultiFunctionMeter( pImg,
     iSingleMeterWidth/*x1*/, 0/*y1*/, 2*iSingleMeterWidth/*x2*/, iSingleMeterHeight/*y2*/, // <- graphic area for one of the six(?) "meters" within pImg
     m_dblDisplayedSWRMeterValue, // [in] value to be displayed (here: dimensionless SWR, typical range 1.0 .. 3.0)
     1.0/*dblMinValue*/, 4.0/*dblMaxValue*/, 0.5/*dblStepwidth*/, // [in] physical value range (also used for the labelled scale)
     "SWR",  // [in] char *pszParamName, e.g. "Power", "ALC", "COMP", "SWR", "Id", "Vd", "Temp"
     "",     // [in] char *pszPhysUnit, physical unit (shown at the end of the tick scale), e.g. "%", "dB", "V", "A", "W", "C"
     3.0/*dblRedStartValue*/, 4.0/*dblRedEndValue*/ ); // range for an optional "red line" indicating a critical range]

  // Third meter (top right) : ALC meter.
  UpdateMultiFunctionMeter( pImg,
     2*iSingleMeterWidth/*x1*/, 0/*y1*/, pImg->Width/*x2*/, iSingleMeterHeight/*y2*/, // <- graphic area for one of the six(?) "meters" within pImg
     (double)m_iDisplayedALCMeterLevel_pcnt, // [in] value to be displayed
     0.0/*dblMinValue*/, 100.0/*dblMaxValue*/, 10.0/*dblStepwidth*/, // [in] physical value range (also used for the labelled scale)
     "ALC",  // [in] char *pszParamName, e.g. "Power", "ALC", "COMP", "SWR", "Id", "Vd", "Temp"
     "",     // [in] char *pszPhysUnit, physical unit (shown at the end of the tick scale), e.g. "%", "dB", "V", "A", "W", "C"
     0.0/*dblRedStartValue*/, 50.0/*dblRedEndValue*/ ); // range for an optional "red line" indicating WHAT ?

  // 4th meter (bottom left) : Drain current. To make it more interesting, Icom uses AMPERES here, not PERCENT. IC-7300: Range 0 .. 25 A.
  UpdateMultiFunctionMeter( pImg,
     0/*x1*/, iSingleMeterHeight/*y1*/, iSingleMeterWidth/*x2*/, 2*iSingleMeterHeight/*y2*/, // <- graphic area for one of the six(?) "meters" within pImg
     (double)m_iDisplayedDrainCurrent_mA * 1e-3, // [in] value to be displayed
     0.0/*dblMinValue*/, 25.0/*dblMaxValue*/, 5.0/*dblStepwidth*/, // [in] physical value range (also used for the labelled scale)
     "Id",  // [in] char *pszParamName, e.g. "Power", "ALC", "COMP", "SWR", "Id", "Vd", "Temp"
     "A",   // [in] char *pszPhysUnit, physical unit (shown at the end of the tick scale), e.g. "%", "dB", "V", "A", "W", "C"
     0.0/*dblRedStartValue*/, 0.0/*dblRedEndValue*/ ); // e.g. 10.0 .. 12.0 [range for an optional "red line" indicating a critical range]

  // 5th meter (bottom middle) : DC Drain voltage ~~ Supply voltage. Icom uses VOLTS here, no stupid PERCENT. IC-7300: Range 10 .. 16 V.
  UpdateMultiFunctionMeter( pImg,
     iSingleMeterWidth/*x1*/, iSingleMeterHeight/*y1*/, 2*iSingleMeterWidth/*x2*/, 2*iSingleMeterHeight/*y2*/, // <- graphic area for one of the six(?) "meters" within pImg
     (double)m_iDisplayedSupplyVoltage_mV * 1e-3, // [in] value to be displayed
     10.0/*dblMinValue*/, 16.0/*dblMaxValue*/, 1.0/*dblStepwidth*/, // [in] physical value range (also used for the labelled scale)
     "Vd",  // [in] char *pszParamName, e.g. "Power", "ALC", "COMP", "SWR", "Id", "Vd", "Temp"
     "V",   // [in] char *pszPhysUnit, physical unit (shown at the end of the tick scale), e.g. "%", "dB", "V", "A", "W", "C"
     10.0/*dblRedStartValue*/, 12.0/*dblRedEndValue*/ ); // e.g. 10.0 .. 12.0 [range for an optional "red line" indicating a critical range]

  // 6th meter (bottom right) : Power Amplifier Temperature. Unfortunately, Icom forgot to make this accessable via CI-V, or forgot to document it.
  UpdateMultiFunctionMeter( pImg,
     2*iSingleMeterWidth/*x1*/, iSingleMeterHeight/*y1*/, pImg->Width/*x2*/, 2*iSingleMeterHeight/*y2*/, // <- graphic area for one of the six(?) "meters" within pImg
     (double)m_iDisplayediPATemperature_degC, // [in] value to be displayed
     0.0/*dblMinValue*/, 100.0/*dblMaxValue*/, 20.0/*dblStepwidth*/, // [in] physical value range (also used for the labelled scale)
     "Temp", // [in] char *pszParamName, e.g. "Power", "ALC", "COMP", "SWR", "Id", "Vd", "Temp"
     "C",   // [in] char *pszPhysUnit, physical unit (shown at the end of the tick scale)
     80.0/*dblRedStartValue*/, 100.0/*dblRedEndValue*/ ); // [in] range for an optional "red line" indicating a critical range


}

//---------------------------------------------------------------------------
void TKeyerMainForm::UpdateTrxDisplay(void) // ..IF MODIFIED (checked internally)
  // Periodically called from TKeyerMainForm::Timer1Timer() .
{ double d;
  char sz80Temp[84];
  DWORD dwSelectedBand, dwAvailableBands;
  BOOL  fTxBandsModified = MyCwNet.RigControl.fTxBandsModified;
        // '--> TRUE when there is an updated list of available transmit bands,
        //      and their edge frequencies in MyCwNet.RigControl.TxBandEdges[].
  BOOL  fBandStackingRegsModified = MyCwNet.RigControl.fBandStackingRegsModified;

  if( fTxBandsModified )
   { MyCwNet.RigControl.fTxBandsModified = FALSE; // acknowledge "new TX bands" for RigControl.c
   }
  if( fBandStackingRegsModified )
   { MyCwNet.RigControl.fBandStackingRegsModified = FALSE; // acknowledge "new Band-Stacking Registers" for RigControl.c
     
   }


  // Periodically update the VFO frequency (displayed in Ed_VFO),
  //  or leave that edit-field untouched because the local operator
  //  is currently TYPING into that field ?
  if( (MyCwNet.RigControl.dblVfoFrequency != RIGCTL_NOVALUE_DOUBLE ) // only if the VFO frequency is KNOWN..
   && (g_SpecDispControl.iVFOEditTimer_ms <= 0 ) // avoid interference while the user TYPES into the VFO edit field ..
   && (!g_SpecDispControl.fDraggingVfoFrequency) // avoid interference while the user 'drags' the VFO frequency via mouse
    )
   { d = g_SpecDispControl.dblDisplayedVfoFreq_Hz - MyCwNet.RigControl.dblVfoFrequency;
     if( (d < -1.0) || (d > 1.0) )  // significant change since the last update (1 Hz) ?
      { UpdateVFODisplay( MyCwNet.RigControl.dblVfoFrequency );
      }
   }
  // Indicate if the VFO display is currently in "editing" state
  //   (in which it possibly shows an outdated value if the frequency
  //    has been modified by another user, or the 'local operator')
  // or in "normal display" state (when it always shows the current value):
  if( (g_SpecDispControl.iVFOEditTimer_ms <= 0 ) // NOT editing the VFO frequency ..
   && (!g_SpecDispControl.fDraggingVfoFrequency) // .. and NOT dragging the VFO frequency via mouse ?
    )
   { switch( g_SpecDispControl.iColourScheme )
      { case COLOUR_SCHEME_DARK: // white digits on black background
           Ed_VFO->Color = clBlack;
           break;
        default: // COLOUR_SCHEME_DEFAULT : usually black digits on white background
           Ed_VFO->Color = clWindow;
           break;
      }
   }
  else // VFO frequency is currently being EDITED (typing or via up/down on a certain digit):
   { switch( g_SpecDispControl.iColourScheme )
      { case COLOUR_SCHEME_DARK: // white digits on black background
           Ed_VFO->Color = (TColor)RGB(0x00,0x60,0x00); // very dark green
           break;
        default: // COLOUR_SCHEME_DEFAULT : usually black digits on white background
           Ed_VFO->Color = (TColor)RGB(0xD0,0xFF,0xD0); // very light green
           break;
      }
   }


  if( m_iDisplayedRigControlOpMode != MyCwNet.RigControl.iOpMode )
   { m_iDisplayedRigControlOpMode  = MyCwNet.RigControl.iOpMode;
     ++m_iUpdating;
     FillComboWithItemsFromTokenLists( CB_Mod, RigCtrl_OpModes, NULL, NULL,
                                        m_iDisplayedRigControlOpMode );
     --m_iUpdating;
   }

  dwSelectedBand  = RigCtrl_FrequencyToBand( MyCwNet.RigControl.dblVfoFrequency );
  dwAvailableBands= RigCtrl_GetAvailableBands( &MyCwNet.RigControl );
  if(  (m_dwDisplayedSelectedBand   != dwSelectedBand )
    || (m_dwDisplayedAvailableBands != dwAvailableBands)
    ||  fTxBandsModified || fBandStackingRegsModified )
   { m_dwDisplayedSelectedBand   = dwSelectedBand;
     m_dwDisplayedAvailableBands = RigCtrl_GetAvailableBands( &MyCwNet.RigControl );
     ++m_iUpdating;
     FillComboWithBandList( CB_Band, m_dwDisplayedAvailableBands, m_dwDisplayedSelectedBand );
     // '--> When available, the "band list" also shows "Band Stacking Registers" !
     //      Thus this list must be updated also when a new 'band stacking register'
     //      value has poured in (in RigControl.c), which may happen anytime,
     //      because the GUI will never wait for data pouring in from the radio.
     //      Surprisingly, Borland's VCL will even properly update the list
     //      when ALREADY DROPPED DOWN in on the "TRX" tab !
     //      Without band-stacking register values, the combo will only show
     //      a boring list of bands (usually wavelengths in meters).
     //      WITH band-stacking register value, the combo will also show
     //      the most recent three frequencies per band.
     // See also: Managing THREE (or even FOUR) different columns to click on
     //           in TKeyerMainForm::CB_BandClick() .
     --m_iUpdating;
   }

  // Missing the periodic update of the spectrum/spectrogram/frequency scale ?
  // SpecDisp_UpdateWaterfall(), SpecDisp_UpdateSpectrum(), SpecDisp_UpdateFreqScale()
  // are called directly from TKeyerMainForm::Timer1Timer() !


} // end TKeyerMainForm::UpdateTrxDisplay()

//-----------------------------------------------------------------------------
void TKeyerMainForm::UpdateVFODisplay( double dblVfoFrequency_Hz ) // -> Ed_VFO
  // Called from UpdateTrxDisplay() if the frequency was modified externally,
  //     or from Ed_VFOKeyDown() if the frequency was tuned LOCALLY (up/down editing).
{
  char sz80Temp[84];
  int iOldSelStart   = Ed_VFO->SelStart;
  int iOldTextLength = Ed_VFO->Text.Length();
  int iNewTextLength;

  g_SpecDispControl.dblDisplayedVfoFreq_Hz = dblVfoFrequency_Hz;
  // The same old question since the haydays of the "printf format string" :
  // Which format string to use ? Tested the following with DL4YHF's CalcEd.exe:
  // > @format("'%10.5lf' MHz")
  // > 12345.67891234 =: '12345.67891' MHz (oh well; too wide by ONE character..)
  // > 1234.567891234 =: '1234.56789' MHz
  // > 123.4567891234 =: ' 123.45679' MHz  (... but ok from VLF to UHF ...)
  // > 12.34567891234 =: '  12.34568' MHz
  // > 1.234567891234 =: '   1.23457' MHz
  // > 0.123456789123 =: '   0.12346' MHz
  // > 0.001234567891 =: '   0.00123' MHz
  // > 0.000012345678 =: '   0.00001' MHz  (display resolution = 10 Hz; ok for CW)
  // > 0.000001234567 =: '   0.00000' MHz
  sprintf( sz80Temp, "%10.5lf MHz", (double)g_SpecDispControl.dblDisplayedVfoFreq_Hz * 1e-6 );
  iNewTextLength = strlen(sz80Temp);
  ++m_iUpdating;
  Ed_VFO->Text = sz80Temp;  // <- the ONLY place where Ed_VFO shall me modified,
                            //    because assigning new Text spoils manual editing
  // Assigning a new string as the edit field's "Text" set the cursor (caret)
  // to the leftmost position (this is what Borland/Delphi/VCL called "SelStart").
  // Not really clever for up/down editing via cursor keys or mouse wheel, thus:
  Ed_VFO->SelStart = iOldSelStart;
  if( iNewTextLength != iOldTextLength ) // trouble with the new decimal place ?
   {
   }

  --m_iUpdating;
} // end UpdateVFODisplay()

//-----------------------------------------------------------------------------
extern "C" void SpecDisp_UpdateVFODisplay( double dblVfoFrequency_Hz ) // -> Ed_VFO (in the GUI)
  // Just a wrapper for UpdateVFODisplay() without any Borland-VCL-dependency.
  // Called from SpecDisp.cpp (which is is "VCL free" module, just plain C).
{
  if( KeyerMainForm != NULL )
   {  KeyerMainForm->UpdateVFODisplay( dblVfoFrequency_Hz );
   }
} // end SpecDisp_UpdateVFODisplay()


//---------------------------------------------------------------------------
void TKeyerMainForm::ApplyNewVFOFreq(void) // Ed_VFO -> MyCwNet.RigControl.dblVfoFrequency
{
  double dblFreq_Hz;

  ++m_iUpdating; // prevent interference from the VCL while updating certain GUI elements..

  dblFreq_Hz = GetFrequencyFromEditField( Ed_VFO, NULL );

  // Apply the new VFO frequency immediately ?
  if( dblFreq_Hz > 1e3 )
   { RigCtrl_SetVFOFrequency( &MyCwNet.RigControl, dblFreq_Hz );
     // ex: CLI_fRedrawFreqScale = TRUE;  // here: after modifying the radio's "VFO" via edit field
     // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
     // The new setting in MyCwNet.RigControl [here: RigControl.dblVfoFrequency]
     // will be compared with the 'last values' sent to a remote server/clients,
     // and if the value was MODIFIED, CwNet.c : CwNet_OnPoll() will send the
     // new setting also via the Remote CW Keyer's own TCP/IP connection.
     // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
   }

  --m_iUpdating;
} // end TKeyerMainForm::ApplyNewVFOFreq()

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::CB_ModClick(TObject *Sender)
{
  int iNewOpMode;
  if( m_iUpdating==0 )
   {
     iNewOpMode = SL_FindStringInTable( RigCtrl_OpModes, CB_Mod->Text.c_str() );
     if( iNewOpMode > 0 )
      { RigCtrl_SetOperatingMode( &MyCwNet.RigControl, iNewOpMode );
      }
   }  // end if( m_iUpdating==0 )
}    // end TKeyerMainForm::CB_ModClick()

//---------------------------------------------------------------------------
int BitmaskToBitIndex(long i32Bitmask )
 // Returns the zero-based index of the least significant, NONZERO bit in i32Bitmask.
{ int iBitIndex = 0;
  while( (i32Bitmask != 0) && ( (i32Bitmask & 1) == 0 ) )
   { ++iBitIndex;
     i32Bitmask >>= 1;
   }
  return iBitIndex;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::CB_BandClick(TObject *Sender)
  // Mouse event handler for the 'Band list'. Originally a simple combo list,
  // later extended to show up to three 'Band Stacking Registers' per band,
  // read out via CI-V from certain Icom radios, arranged in THREE COLUMNS.
  // Selecting a certain BAND STACKING REGISTER not only switches band / frequency,
  // but also the operating mode (and, maybe, even more).
{
  long i32NewBand;
  TPoint tpScreen, tpClient;
  int  i, n, iCharWidth, iCharIndex, iComboItemIndex, iBandIndex, iBandStackIndex, iBandStackingRegIndex;
  char sz80ItemText[84];
  const char* pszBandName;
  T_RigCtrlInstance *pRC = &MyCwNet.RigControl;
  T_RigCtrlFreqMemEntry *pBandStackingReg;
  BOOL fOk = FALSE;

  if( m_iUpdating==0 )
   {
     // The bands listed in this combo box may be limited to what THE SYSOP allows,
     //  or what the remotely controlled radio permits.
     //  When available, the frequency on the new(?) band will be
     //  one of up to three 'Band Stacking Registers', at least for Icom radios.
     // For that purpose, a single entry in the TComboBox (a standard VCL control)
     //  may be divided into THREE PARTS ("left", "center", "right"),
     //  resembling Icom's own 'BAND STACKING REGISTER' display (they used to
     //  write all in upper case).
     strncpy( sz80ItemText, CB_Band->Text.c_str(), 80 );
     sz80ItemText[80] = '\0'; // strncpy() sucks.. it doesn't always terminate the destination
       // '--> e.g. "40 m:  7.005 CW   7.030 CW   7.033 CW" (see FillComboWithBandList() ).
       //  or just  "40 m"  if there are no 'Band Stacking Registers" for this band.


     // Because a Borland VCL TComboBox doesn't expose the coordinate of the
     // mouse pointer when clicking the box, try to determine the 'column' (!)
     // of the mouse pointer within the combo as follows:
     tpScreen = TPoint( Mouse->CursorPos.x, Mouse->CursorPos.y ); // obfuscated C++ stuff...
     tpClient = CB_Band->ScreenToClient(tpScreen);
     // Test results at THIS POINT (sic) when clicking into the 1st item,
     // first character of the item text: tpClient.x = 4,  tpClient.y = 34 .
     // Clicked on the last character: tpClient.x = 254.
     // Tried to convert the horizontal pixel offset into a character index,
     // and -from that character index- determine the 'band stack index' (0..2):
     iCharWidth = CB_Band->Canvas->TextWidth( "w" ); // with "Courier New size 8" : 9 pixels / character
     iCharIndex = tpClient.x / iCharWidth;
     if( iCharIndex < (BANDLIST_BAND_COLUM_WIDTH+BANDLIST_FREQ_COLUMN_WIDTH) )
      { // If there are 'band stacking' frequencies at all, it's the first
        // (that's the one with Icom's MOST RECENT band stacking register)
        iBandStackIndex = 0; // -> use s_iBandStackingRegIndices[?][0] further below
      }
     else if( iCharIndex < (BANDLIST_BAND_COLUM_WIDTH+2*BANDLIST_FREQ_COLUMN_WIDTH) )
      { // the SECOND "band stacking column" If there are 'band stacking' frequencies at all, it's the
        iBandStackIndex = 1; // -> use s_iBandStackingRegIndices[?][1] further below
      }
     else
      { iBandStackIndex = 2; // -> use s_iBandStackingRegIndices[?][2] further below
      }

     // ex: i32NewBand = SL_FindStringInTable( RigCtrl_BandNames, sz80ItemText );
     // The above doesn't work anymore since the addition of Band Stacking frequencies
     // in the in the item text from the combo box. So, first extract the
     // BAND NAME (e.g. "40 m") from sz80ItemText :
     i32NewBand = -1;
     for( i=0; (pszBandName=RigCtrl_BandNames[i].pszKeyword) != NULL; ++i )
      { n = strlen(pszBandName);
        if( strncmp( sz80ItemText, pszBandName, n ) == 0 )
         { i32NewBand = RigCtrl_BandNames[i].iTokenValue;
           // Found the BAND (e.g. iNewBand = RIGCTRL_BAND_40M = 8) to switch to,
           // but maybe the mouse points over a frequency from a BAND STACKING REGISTER.
           // If it does, don't just switch to the particular BAND but to the FREQUENCY,
           // and the mode, stored in the Band Stacking Register.
           break;
         }
      } // end for < all entries in RigCtrl_BandNames[i] >

     if( i32NewBand > 0 )
      { // Is there a valid BAND STACKING REGISTER for the new selected band ?
        // [in]  T_RigCtrlInstance *pRC = &MyCwNet.RigControl; especially:
        //       pRC->BandStackingRegs[]
        iBandIndex/*0..31*/ = BitmaskToBitIndex( i32NewBand/*RIGCTRL_BAND_xyz*/ );
        iBandStackingRegIndex = s_iBandStackingRegIndices[iBandIndex][iBandStackIndex];
           // '--> filled in FillComboWithBandList(), used in CB_BandClick() .
        if( (iBandStackingRegIndex >= 0 ) && (iBandStackingRegIndex<RIGCTRL_NUM_BAND_STACKING_REGS) )
         { pBandStackingReg = &pRC->BandStackingRegs[iBandStackingRegIndex];
           // When controlling e.g. an IC-7300, the behaviour shall be exactly
           // as when opening the "Band Stacking Register" :
           // which kind-of remembers the last frequency used on THAT BAND.
           if( pBandStackingReg->RxTx[0].dblOperatingFreq_Hz > 0.0 )
            { fOk = RigCtrl_SwitchToFreqMemEntry( pRC, pBandStackingReg ); // switch frequency, mode, and maybe even more ("split")
              // Abort a pending frequency change from the VFO edit field,
              // instead ALWAYS overwrite the edit field (Ed_VFO) with the new frequency:
              g_SpecDispControl.iVFOEditTimer_ms = 0; // not editing Ed_VFO anymore
              g_SpecDispControl.fDraggingVfoFrequency = FALSE; // not dragging the VFO frequency anymore
              UpdateVFODisplay( pRC->dblVfoFrequency );
              // If RigCtrl_SwitchToFreqMemEntry() also modified
              //   MyCwNet.RigControl.iOpMode, it will be different from
              //   m_iDisplayedRigControlOpMode until UpdateTrxDisplay()
              //   takes care of that.
            }
         }
        if( !fOk ) // no valid "band stacking register" but we know THE BAND to switch to:
         { RigCtrl_SwitchToBand( pRC, i32NewBand );
         } // end if < switch to a certain band, but without a "band stacking register" >
      }   // end if( i32NewBand > 0 )
   }     // end if( m_iUpdating==0 )
}       // end TKeyerMainForm::CB_BandClick()

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Btn_HelpBandSwitchClick(TObject *Sender)
{
  YHF_HELP_ShowHelpContext( HELPID_BAND_SELECTION );
}


//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Chk_Audio_AllowNetworkClick(TObject *Sender)
{
  // This control shall have an 'immediate effect' when clicking,
  //      e.g. disable sending audio to, or receiving audio from the network:
  if( m_iUpdating <= 0 ) // prevent interference from "OnClick"
   { //  (it sometimes fires when MODIFYING 'Checked' programmatically)
     if( Chk_Audio_AllowNetwork->Checked )
      {  CwKeyer_DSP.cfg.iAudioFlags |= DSP_AUDIO_FLAGS_ALLOW_NETWORK_AUDIO;
      }
     else
      {  CwKeyer_DSP.cfg.iAudioFlags &= ~DSP_AUDIO_FLAGS_ALLOW_NETWORK_AUDIO;
      }
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::ApplyKeyerSettings(TObject *Sender)
{ // No need to click "Apply" - called from various combo-boxes "OnClick"-handlers:
  AnsiString s;
  if( m_iUpdating <= 0 ) // prevent interference from certain "OnClick" events..
   { CwKeyer_Config.iMorseKeyType = GetSelectedTokenValueFromComboBox( CB_KeyType, KeyTypes, NULL, NULL );
     CwKeyer_Config.iSidetoneOnTXD= GetSelectedTokenValueFromComboBox( CB_SidetoneOnTXD, SidetonesOnTXD, NULL, NULL );
     // Formerly on the "Keyer" tab, now on an extra "Audio" tab.. anyway, keep them here:
     CwKeyer_DSP.cfg.iSidetoneFreq_Hz = SL_atoi( CB_SidetoneOnAudioOut->Text.c_str() );
     CwKeyer_DSP.cfg.iSidetoneRiseTime_ms = SL_atoi( CB_SidetoneRiseTime->Text.c_str() );

     s = CB_AudioInputDevice->Text;
     if( s != "" )
      { if( s != AnsiString(CwKeyer_DSP.cfg.sz255AudioInputDevice) )
         { // MODIFIED 'audio input device' so copy to config, and re-init...
           strncpy( CwKeyer_DSP.cfg.sz255AudioInputDevice, s.c_str(), 255 );
           m_fRestartKeyerThread = TRUE;  // ... because a basic part of the config was modified
         }
      }
     s = CB_AudioOutputDevice->Text;
     if( s != "" )
      { if( s != AnsiString(CwKeyer_DSP.cfg.sz255AudioOutputDevice) )
         { strncpy( CwKeyer_DSP.cfg.sz255AudioOutputDevice, s.c_str(), 255 );
           m_fRestartKeyerThread = TRUE; // here: because a new AUDIO OUTPUT device was selected
         }
      }
     CwKeyer_Config.iTxDelayTime_ms = StrToIntDef( Ed_TxDelayTime->Text, 25);
     CwKeyer_Config.iTxHangTime_ms  = StrToIntDef( Ed_TxHangTime->Text, 500);
   } // end if( m_iUpdating <= 0 )
} // end TKeyerMainForm::ApplyKeyerSettings()


//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::CB_SidetoneOnAudioOutChange(TObject *Sender)
{ // Special service to modify the sidetone (via AUDIO OUTPUT) "while typing"
  int iNewSidetoneFreq;
  if( CB_SidetoneOnAudioOut->Text == "None" )
   {  iNewSidetoneFreq = 0;  // 0 = off
   }
  else // not OFF, so parse from the combo's EDIT FIELD :
   { iNewSidetoneFreq = SL_atoi( CB_SidetoneOnAudioOut->Text.c_str() );
     UTL_LimitInteger( &iNewSidetoneFreq, 50/*Hz*/, 3000/*Hz*/ );
     // (who knows .. "audio keying a transmitter in USB" should use a
     //  high enough frequency, to that 2 * f, 3 * f, etc aren't in the passband.
     //  Thus if the operator TYPES into this edit field, accept up to 3 kHz.)
   }
  if( iNewSidetoneFreq != CwKeyer_DSP.cfg.iSidetoneFreq_Hz )
   { CwKeyer_DSP.cfg.iSidetoneFreq_Hz = iNewSidetoneFreq;
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::ApplyVolumeAdjustments(TObject *Sender)
{ // The 'Volume Adjustment' sliders on the 'Audio' tab shall also have
  // an IMMEDIATE effect on the digital signal processing (in CwDSP.c) :
  if( m_iUpdating <= 0 ) // don't react of "OnChange" fired because THE PROGRAM changed a value:
   { CwKeyer_DSP.cfg.iSidetoneGain_dB = -SB_SidetoneGain_dB->Position;
     CwKeyer_DSP.cfg.iAudioInGain_dB  = -SB_AudioInGain_dB->Position;
     CwKeyer_DSP.cfg.iAudioOutGain_dB = -SB_AudioOutGain_dB->Position;
     CwKeyer_DSP.cfg.iNetworkTonesGain_dB = -SB_NetworkTonesGain_dB->Position;
   }
} // end TKeyerMainForm::ApplyVolumeAdjustments(

//---------------------------------------------------------------------------
void TKeyerMainForm::UpdateClientServerLabels(void)
  // Grays out those labels on the "Network" tab that, depending on the
  // currently selected "Network functionality", have NO FUNCTION .
{
  Lab_RemoteURL->Enabled  = Rbtn_Client->Checked;
  Lab_MyUserName->Enabled = Rbtn_Client->Checked;
  Lab_UserCallsign->Enabled=Rbtn_Client->Checked;
  Lab_ServerListeningPort->Enabled = Rbtn_Server->Checked;
  Lab_AcceptedUsers->Enabled = Rbtn_Server->Checked;
} // end TKeyerMainForm::UpdateClientServerLabels()


//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::ApplyNetworkSetup(TObject *Sender)
{
  int  iNewValue;
  BOOL fModified = FALSE;
  if( m_iUpdating <= 0 ) // prevent interference from 'UpdateNetworkSetup' ..
   {
     if( Rbtn_Client->Checked )
      { fModified |= (MyCwNet.cfg.iFunctionality != CWNET_FUNC_CLIENT);
        MyCwNet.cfg.iFunctionality = CWNET_FUNC_CLIENT;
      }
     else if(Rbtn_Server->Checked)
      { fModified |= (MyCwNet.cfg.iFunctionality != CWNET_FUNC_SERVER);
        MyCwNet.cfg.iFunctionality = CWNET_FUNC_SERVER;
      }
     else
      { fModified |= (MyCwNet.cfg.iFunctionality != CWNET_FUNC_OFF);
        MyCwNet.cfg.iFunctionality = CWNET_FUNC_OFF;
      }
     strncpy( MyCwNet.cfg.sz80ClientRemoteIP, Ed_ClientRemoteIP->Text.c_str(), 80 );
     strncpy( MyCwNet.cfg.sz80ClientUserName, Ed_ClientUserName->Text.c_str(), 80 );
     strncpy( MyCwNet.cfg.sz80ClientCallsign, Ed_ClientCallsign->Text.c_str(), 80 );
     iNewValue = StrToIntDef( Ed_ServerListeningPort->Text, 7355 );
     fModified |= (MyCwNet.cfg.iServerListeningPort != iNewValue);
     MyCwNet.cfg.iServerListeningPort = iNewValue;
     strncpy( MyCwNet.cfg.sz255AcceptUsers, Ed_ServerAcceptUsers->Text.c_str(), 255 );
     strncpy( MyCwNet.cfg.sz20ServerAdminPWD, Ed_ServerAdminPWD->Text.c_str(), 20 );
#   if( SWI_USE_HTTP_SERVER )  // a built-in HTTP server is optional ..
     if( Chk_EnableHTTP->Checked )
      { MyCwNet.cfg.iHttpServerOptions |= HTTP_SERVER_OPTIONS_ENABLE;
      }
     else
      { MyCwNet.cfg.iHttpServerOptions &= ~HTTP_SERVER_OPTIONS_ENABLE;
      }
#   endif // SWI_USE_HTTP_SERVER ?

     UpdateClientServerLabels();
     if( fModified )  // modified so that CwNet.c needs a "restart" ?
      {
        HERE_I_AM__GUI();
        CwNet_Stop( &MyCwNet );
        // For safety, also stop and optionally restart the Hamlib server:
        HLSRV_Stop( &MyHamlibServer );
        if( MyCwNet.cfg.iFunctionality != CWNET_FUNC_OFF )
         { ShowError( ERROR_CLASS_INFO, "Starting Network functionality .." );
           CwNet_Start( &MyCwNet );
         }
#      if( SWI_USE_HAMLIB_SERVER ) // not sure if this 'feature' is going to stay...
        if(  CwKeyer_Config.fEnableBuiltInHamlibServer
         && (MyHamlibServer.cfg.iServerListeningPort > 0 ) )
         { HLSRV_Start( &MyHamlibServer );
         }
#      endif // SWI_USE_HAMLIB_SERVER ?
        HERE_I_AM__GUI();
      }
     UpdateWindowCaption(); // -> e.g. "Remote CW Keyer [1]  (Server)/(Client)"
   } // end if( m_iUpdating <= 0 )
  HERE_I_AM__GUI();

} // TKeyerMainForm::ApplyNetworkSetup()

//---------------------------------------------------------------------------
void TKeyerMainForm::UpdateNetworkSetup(void)
{
  char sz80[84];
  struct hostent *pHE;
  DWORD  dwIPv4Address;

  HERE_I_AM__GUI();

  ++m_iUpdating;
  Rbtn_NetworkOff->Checked = MyCwNet.cfg.iFunctionality == CWNET_FUNC_OFF;
  Rbtn_Client->Checked     = MyCwNet.cfg.iFunctionality == CWNET_FUNC_CLIENT;
  Rbtn_Server->Checked     = MyCwNet.cfg.iFunctionality == CWNET_FUNC_SERVER;
  Ed_ClientRemoteIP->Text  = MyCwNet.cfg.sz80ClientRemoteIP;
  Ed_ClientUserName->Text  = MyCwNet.cfg.sz80ClientUserName;
  Ed_ClientCallsign->Text  = MyCwNet.cfg.sz80ClientCallsign;
  Ed_ServerListeningPort->Text = IntToStr( MyCwNet.cfg.iServerListeningPort );
  Ed_ServerAcceptUsers->Text = MyCwNet.cfg.sz255AcceptUsers;
#if( SWI_USE_HTTP_SERVER )  // use a built-in HTTP server ?
  Chk_EnableHTTP->Checked = (MyCwNet.cfg.iHttpServerOptions & HTTP_SERVER_OPTIONS_ENABLE) != 0;
  Ed_ServerAdminPWD->Text = MyCwNet.cfg.sz20ServerAdminPWD;
#else // ! SWI_USE_HTTP_SERVER -> disable all GUI controls for the HTTP server:
  Chk_EnableHTTP->Checked    = FALSE;
  Chk_EnableHTTP->Enabled    = FALSE;
  Lab_AdminPWD->Enabled      = FALSE;
  Ed_ServerAdminPWD->Enabled = FALSE;
#endif // SWI_USE_HTTP_SERVER ?

  // The following LABELS are "just for info", and/or for troubleshooting:
  if( gethostname( sz80,80) == SOCKET_ERROR )
   { Lab_LocalHost->Caption = "*FAIL*"; // here we are again, in Winsock heaven..
     Lab_LocalIP->Caption   = "*FAIL*";
   }
  else  // successfully retrieve this PC's LOCAL NAME in the network ("hostname")..
   { Lab_LocalHost->Caption = sz80;
     pHE = gethostbyname(sz80); // resolves this machine's NAME into one of its NUMERIC IP ADDRESSES ..
     strcpy( sz80, "???" );
     if( pHE != NULL )
      { // Note: the 'struct hostent pointer') isn't a single item,
        // but a WHOLE LIST of items. For example, if the PC has multiple
        // 'network interfaces' (say two Ethernet LAN ports, and one WLAN),
        // then there may be many roads leading to Rome. We only show THE FIRST..
        if( ( pHE->h_addrtype == AF_INET )  // sounds good, the "address family" is "internet"...
         && ( pHE->h_length == 4 )       // .. and the list of addresses seem to be "IPv4"...
         && ( pHE->h_addr_list[0] != NULL) ) // .. and there's at least ONE entry in it ...
         { dwIPv4Address = *(DWORD*)pHE->h_addr_list[0]; // read FOUR BYTES in "network byte order"
           sprintf( sz80, "%d.%d.%d.%d",
               (int)(BYTE)(dwIPv4Address>>0),  (int)(BYTE)(dwIPv4Address>>8),
               (int)(BYTE)(dwIPv4Address>>16), (int)(BYTE)(dwIPv4Address>>24) );
         }
      }
     Lab_LocalIP->Caption = sz80;
   }
  UpdateClientServerLabels();
  --m_iUpdating;

  HERE_I_AM__GUI();

} // TKeyerMainForm::ApplyNetworkSetup()

//---------------------------------------------------------------------------
void TKeyerMainForm::SwitchMainTab(int iNewTab) // [in] one of the following:
  //  MAIN_TAB_CONFIG,  MAIN_TAB_KEYER_SETTINGS, MAIN_TAB_AUDIO,
  //  MAIN_TAB_NETWORK, MAIN_TAB_DEBUG,          MAIN_TAB_MEMORY,
  //  MAIN_TAB_TEST,    MAIN_TAB_TRX,            ... (?)
  // Called from the main thread ("GUI thread") whenever THE PROGRAM
  // needs to switch the main window's main tabsheet, PageControl1 .
  // Callers:  1. FormCreate(), shows the "I/O Config" tab initially,
  //              because this is where the user MUST provide some info
  //              (COM port for the Morse key adapter and remotely controlled rig)
  //           2. StartKeyerAndShowInfo(), switches to the "Debug" tab
  //              while reading the most important settings FROM the radio
  //           3. Timer1Timer() when module RigControl enters RIGCTRL_POLLSTATE_DONE,
  //              to switch from the "Debug" tab (with a live CAT traffic display)
  //              to the "TRX" tab (Transceiver control with VFO and spectrum).
  //
{
  switch( iNewTab )
   { case MAIN_TAB_CONFIG:  PageControl1->ActivePage = TS_Config;  break;
     case MAIN_TAB_KEYER_SETTINGS: PageControl1->ActivePage = TS_KeyerSettings; break;
     case MAIN_TAB_AUDIO:   PageControl1->ActivePage = TS_Audio;   break;
     case MAIN_TAB_NETWORK: PageControl1->ActivePage = TS_Network; break;
     case MAIN_TAB_DEBUG:   PageControl1->ActivePage = TS_Debug;   break;
     case MAIN_TAB_MEMORY:  PageControl1->ActivePage = TS_Memory;  break;
     case MAIN_TAB_TEST:    PageControl1->ActivePage = TS_Test;    break;
     case MAIN_TAB_TRX:     PageControl1->ActivePage = TS_TRX;     break;
     default: break;
   } // end switch( iNewTab )
  m_iCurrentMainTab = iNewTab;  // save this to avoid having to compare PageControl1->ActivePage (a Borland VCL thing)
} // end TKeyerMainForm::SwitchMainTab()

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::PageControl1Change(TObject *Sender)
  // About the VCL's TPageControl.OnChange :
  // > Occurs after a new tab is selected.
  // > Write an OnChange event handler to take specific action
  // > immediately after the selected tab changes. Use the
  // > TabIndex property to determine which tab is now selected.
  // > This is the opportunity to make any changes to the control
  // > that reflect the new state implied by the selected tab.
  // > Before the value of TabIndex changes, an OnChanging event occurs.
  // > Note: The event is not called if you change the active page in code,
  // >       for example, by setting the value of ActivePage.
{
  int iNewTab = m_iCurrentMainTab;
  // If we recognize the NEW value of 'PageControl1->ActivePage',
  // modify m_iCurrentMainTab accordingly
  if( PageControl1->ActivePage == TS_Config )
   { iNewTab = MAIN_TAB_CONFIG;
   }
  else if( PageControl1->ActivePage == TS_KeyerSettings )
   { iNewTab = MAIN_TAB_KEYER_SETTINGS;
   }
  else if( PageControl1->ActivePage == TS_Audio )
   { iNewTab = MAIN_TAB_AUDIO;
   }
  else if( PageControl1->ActivePage == TS_Network )
   { iNewTab = MAIN_TAB_NETWORK;
   }
  else if( PageControl1->ActivePage == TS_Debug )
   { iNewTab = MAIN_TAB_DEBUG;
   }
  else if( PageControl1->ActivePage == TS_Memory )
   { iNewTab = MAIN_TAB_MEMORY;
   }
  else if( PageControl1->ActivePage == TS_Test )
   { iNewTab = MAIN_TAB_TEST;
   }
  else if( PageControl1->ActivePage == TS_TRX )
   { iNewTab = MAIN_TAB_TRX;
   }
  else
   { // oops.. new tabsheet ?
   }
  if( iNewTab != m_iCurrentMainTab )  // tabsheet obviously switched by the USER ->
   { m_iCurrentMainTab = iNewTab;
     m_fAutomaticTabSwitching = FALSE;
   }
} // end TKeyerMainForm::PageControl1Change()


//---------------------------------------------------------------------------
void TKeyerMainForm::StartKeyerAndShowInfo(void)
{
  HERE_I_AM__GUI();
  CwDSP_Start( &CwKeyer_DSP ); // start our 'audio DSP' before the keyer..
  HERE_I_AM__GUI();
  if( CwKeyer_Start() )  // try to start the CW keyer.  [in] CwKeyer_Config ...
    { // Successfully opened the two serial ports and launched the worker thread
      // -> automatically switch from the "Settings"- to the "Debug"-tab,
      //    where the 'Remote CW Keyer test' will report what's going on.
      HERE_I_AM__GUI();
      SwitchMainTab( MAIN_TAB_DEBUG ); // ex: PageControl1->ActivePage = TS_Debug;
      ShowError(ERROR_CLASS_INFO|SHOW_ERROR_TIMESTAMP, "Keyer thread running, %d WPM.",
       (int)Elbug_DotTimeInMillisecondsToWordsPerMinute(CwKeyer_Elbug.cfg.iDotTime_ms) );
      // Even after CwKeyer_Start() returned TRUE, it may have reported
      // a 'minor problems' via CwKeyer_sz255LastError, so check this:
      if( CwKeyer_sz255LastError[0] != '\0' ) // <- set in e.g. KeyerThread.c : OpenAndConfigureSerialPort()
       { ShowError(ERROR_CLASS_INFO|SHOW_ERROR_TIMESTAMP, CwKeyer_sz255LastError );
         CwKeyer_sz255LastError[0] = '\0';  // "done" (reported this; may be set in OTHER THREADS again)
       }
      // If remote operation via NETWORK is also configured,
      // also start the 'CW Network' client or server (only ONE of them) :
      if( MyCwNet.cfg.iFunctionality != CWNET_FUNC_OFF )
       { if( CwNet_Start( &MyCwNet ) )
          { ShowError(ERROR_CLASS_INFO | SHOW_ERROR_TIMESTAMP,
                      "%s", CwNet_GetCurrentStatusAsString(&MyCwNet) );
            // Result e.g. "CW-Server: Running, ..."
          }
         else // "network" (client or server) didn't start .. anyway, keep going
          { ShowError(ERROR_CLASS_WARNING | SHOW_ERROR_TIMESTAMP,
                      "Network Error: %s", CwNet_GetLastErrorAsString(&MyCwNet) );
            // Result e.g. "CW-Server: can't listen on port 7355 (..)"
          }
       } // end if < start with any of the 'network' functionalities active > ?
    }
   else // Failed to open a port, or failed to launch the worker thread :
    { // -> ALSO switch from the "Settings"- to the "Debug"-tab, and show
      //    what went wrong :
      SwitchMainTab( MAIN_TAB_DEBUG ); // ex: PageControl1->ActivePage = TS_Debug;
      ShowError(ERROR_CLASS_FATAL|SHOW_ERROR_TIMESTAMP, CwKeyer_sz255LastError );
      // ,----------------------------------------------'
      // '--> e.g. "Could not open serial port XYZ", set in KeyerThread.c : OpenAndConfigureSerialPort()
    }
  m_fRestartKeyerThread = FALSE;  // "done" (restarted the keyer)

# if( SWI_USE_HAMLIB_SERVER )
  // For safety, stop, reconfigure, and optionally restart the Hamlib server:
  HLSRV_Stop( &MyHamlibServer );
  if(  CwKeyer_Config.fEnableBuiltInHamlibServer
   && (MyHamlibServer.cfg.iServerListeningPort > 0 ) )
   { HERE_I_AM__GUI();
     if( HLSRV_Start( &MyHamlibServer ) )
      { ShowError(ERROR_CLASS_INFO | SHOW_ERROR_TIMESTAMP, "%s",
          HLSRV_GetCurrentStatusAsString(&MyHamlibServer) );
      }
     else // built-in "Hamlib server" didn't start .. anyway, keep going, it's not essential
      { ShowError(ERROR_CLASS_WARNING | SHOW_ERROR_TIMESTAMP,
          "HL-Server Error: %s", HLSRV_GetLastErrorAsString(&MyHamlibServer) );
      }
   }
  HERE_I_AM__GUI();
# endif // SWI_USE_HAMLIB_SERVER ?

   UpdateWindowCaption(); // -> e.g. "Remote CW Keyer [1]  (Server)/(Client)"
   HERE_I_AM__GUI();
} // end TKeyerMainForm::StartKeyerAndShowInfo()

//---------------------------------------------------------------------------
void TKeyerMainForm::UpdateStatusIndicatorText(void) // -> m_sz80StatusIndicatorText
  // [in]   m_iStatusIndicatorUsage, m_iStatusIndicatorMemIdx .
  // [out]  m_sz80StatusIndicatorText
  // The actual 'graphic rendering' of the status indicator happens
  // later, when processing yet another funny windows message (WM_NCPAINT) .
{
  switch( m_iStatusIndicatorUsage )  // <- set in UpdateStatusIndicator() ...
   { case STATUS_INDICATOR_OFF :
           strcpy( m_sz80StatusIndicatorText, "[ off ]" );
           break;
     case STATUS_INDICATOR_IDLE:
           strcpy( m_sz80StatusIndicatorText, "[ idle ]" );
           break;
     case STATUS_INDICATOR_ON_AIR:
           strcpy( m_sz80StatusIndicatorText, "[ ON AIR ]" );
           break;
     case STATUS_INDICATOR_TX_MEM:
           sprintf( m_sz80StatusIndicatorText, "[ TX MEM #%d ]",
                    (int)m_iStatusIndicatorMemIdx+1 );
           break;
     case STATUS_INDICATOR_CW_TEST:
           strcpy( m_sz80StatusIndicatorText, "[ Off-air CW ]" );
           break;
     case STATUS_INDICATOR_PAUSED:
           strcpy( m_sz80StatusIndicatorText, "[ paused ]" );
           break;
     default:
           strcpy( m_sz80StatusIndicatorText, "[ ? ]" );
           break;
   }

} // end UpdateStatusIndicatorText()

//---------------------------------------------------------------------------
void TKeyerMainForm::UpdateStatusIndicator(void) // .. somewhere in the "non-client area"
  // Periodically called from Timer1Timer() .
{ int iStatusIndicatorUsage = 0;

  if( CwKeyer_iThreadStatus != KEYER_THREAD_STATUS_RUNNING )
   { iStatusIndicatorUsage = STATUS_INDICATOR_OFF;
   }
  else // the keyer thread is running, but what is it doing at the moment ?
   {   // See KeyerThread.c : KeyerThread() ...
     if( CwKeyer_Elbug.fPauseTransmitter )
      { iStatusIndicatorUsage = STATUS_INDICATOR_PAUSED;
      }
     else if( CwKeyer_Gen.iState != CW_GEN_OFF )
      { iStatusIndicatorUsage = STATUS_INDICATOR_TX_MEM;
      }
     else if( MyCwNet.RigControl.fLocalPTTFlag )
      { iStatusIndicatorUsage = STATUS_INDICATOR_ON_AIR;
      }
     else if( (CwKeyer_Config.iManualPTTInput != KEYER_SIGNAL_INDEX_NONE) && (CwKeyer_swMorseActivityTimer!=0) ) // MANUAL PTT control, and Morse-Output WITHOUT active PTT ?
      { iStatusIndicatorUsage = STATUS_INDICATOR_CW_TEST;
      }
     else
      { iStatusIndicatorUsage = STATUS_INDICATOR_IDLE;
      }
   } // end switch( CwKeyer_iThreadStatus )

  if( (iStatusIndicatorUsage != m_iStatusIndicatorUsage ) // need to update the "GUI" ?
    ||(m_iCurrentMemoryIndex != m_iStatusIndicatorMemIdx) )
   { m_iStatusIndicatorUsage  = iStatusIndicatorUsage;
     m_iStatusIndicatorMemIdx = m_iCurrentMemoryIndex;
     UpdateStatusIndicatorText();

     // Abuse one of the items in the MAIN MENU(!) as the 'status indicator' :
     MI_StatusIndicator->Caption = m_sz80StatusIndicatorText;

   } // end if < m_iStatusIndicatorUsage modified since the last call > ?
} // end TKeyerMainForm::UpdateStatusIndicator()

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_StatusIndicatorClick(TObject *Sender)
{
  // Depending on what the 'status indicator' currently indicates,
  //        clicking the indicator may ...
  switch( m_iStatusIndicatorUsage )
   { case STATUS_INDICATOR_PAUSED:  // ... "unpause" the keyer (unlock the "Operator Panic"-button)
        // the smoke disappeared, the dust has settled, the fuse was replaced,
        // the PA or the TRX cooled down, the antenna has been connected,
        // and the correct 'COM port' for the paddle was found.. so try again:
        if( CwKeyer_Elbug.fPauseTransmitter ) // "pause transmission" / "operator panic"
         { ShowError(ERROR_CLASS_INFO, "Resuming normal operation after being 'paused'." );
           CwKeyer_Elbug.fPauseTransmitter = FALSE;  // allow the keyer to key the TX again
           // '--> The CW keyer task should also poll this flag while sending
           //          'from memory', using CwGen.c instead of Elbug.c !
         }
        break;
     default:
        break;
   }
} // end TKeyerMainForm::MI_StatusIndicatorClick()


//---------------------------------------------------------------------------
void TKeyerMainForm::PauseTransmission(void) // Called on e.g. pressing ESCAPE.
  // Intended as an 'operator panic key' to stop (or at least pause) transmission
  // when e.g. a wrong COM port has been selected for the local key adapter,
  // and the paddle input states indicate 'send a dash' or 'send a dot'.
{
  if( ! CwKeyer_Elbug.fPauseTransmitter )
   { CwKeyer_Elbug.fPauseTransmitter = TRUE; // "pause transmission" / "operator panic"
     ShowError(ERROR_CLASS_INFO, "TX paused - click on 'paused' indicator to resume normal operation." );
     CwKeyer_fUpdateAllOutputs = TRUE; // on "Pause Transmission", send all outputs at least ONCE
   }


} // end TKeyerMainForm::PauseTransmission()


//---------------------------------------------------------------------------
void TKeyerMainForm::AddAudioInputDevice( AnsiString sItemText )
  // Appends an enumerated audio input device to a combo (CB_AudioInputDevice),
  // and if that's the one currently selected (CwKeyer_DSP.sz255AudioInputDevice)
{ CB_AudioInputDevice->Items->Add( sItemText );
  if( sItemText == AnsiString(CwKeyer_DSP.cfg.sz255AudioInputDevice) )
   { CB_AudioInputDevice->ItemIndex = CB_AudioInputDevice->Items->Count - 1;
   }
}

//---------------------------------------------------------------------------
void TKeyerMainForm::AddAudioOutputDevice( AnsiString sItemText )
  // Appends an enumerated audio input device to a combo (CB_AudioOutputDevice),
  // and if that's the one currently selected (CwKeyer_DSP.sz255AudioOutputDevice)
{ CB_AudioOutputDevice->Items->Add( sItemText );
  if( sItemText == AnsiString(CwKeyer_DSP.cfg.sz255AudioOutputDevice) )
   { CB_AudioOutputDevice->ItemIndex = CB_AudioOutputDevice->Items->Count - 1;
   }
}


//---------------------------------------------------------------------------
void TKeyerMainForm::UpdateAudioDeviceCombos(void)
{ // "Borrowed" from Spectrum Lab, but stripped down for 'normal audio devices'..
  int i, iAudioDeviceID, iAudioDeviceType, n_good_devs;
  char *cp, *pszDeviceNameWithoutPrefix, sz80Temp[84];
#if( SWI_USE_WAVE_AUDIO ) // use the ancient "Windows Multimedia Extensions" / "wave audio" ?
  WAVEINCAPS  my_waveincaps;
  WAVEOUTCAPS my_waveoutcaps;
  int n_devs;
#endif // SWI_USE_WAVE_AUDIO ?

  // int  nEntriesFound = 0;
  AnsiString s;

  HERE_I_AM__GUI();
  ++m_iUpdating;

  // Try to increase the DROPPED-DOWN width (the VCL doesn't allow this,
  // but Windows may support it, since "Vista" ... )
  // > Setting the drop down width is as simple as sending the
  // > CB_SETDROPPEDWIDTH message to the controls handle:
  SendMessage( CB_AudioInputDevice->Handle, CB_SETDROPPEDWIDTH, 280/*MinimumWidthInPixels*/, 0);
  SendMessage( CB_AudioOutputDevice->Handle,CB_SETDROPPEDWIDTH, 280/*MinimumWidthInPixels*/, 0);

  //--------- build a list of available AUDIO INPUT DEVICES ------------------
  CB_AudioInputDevice->Items->Clear();
  CB_AudioInputDevice->ItemIndex = -1; // hopefully set to a valid index further below..
  AddAudioInputDevice( "None" );
#if( SWI_USE_DSOUND ) // use the mature "DirectSound" / dsound_wrapper.c ?
  for(i=0; i<MyDirectSound.m_nInputDevices; ++i) // in Microsoft terms, these are "capture devices" ..
   { AddAudioInputDevice( AnsiString(MyDirectSound.m_sInputDevices[i].sz255DevName) );
   }
#elif( SWI_USE_WAVE_AUDIO ) // use the ancient "Windows Multimedia Extensions" / "wave audio" ?
  n_devs = waveInGetNumDevs();
  AddAudioInputDevice( "Default WAVE input" );
  for(i=0; i<n_devs; ++i)
   {
    if( waveInGetDevCaps(     i, // UINT uDeviceID
                 &my_waveincaps, // LPWAVEINCAPS pwic,
             sizeof(WAVEINCAPS)) // UINT cbwic
       == MMSYSERR_NOERROR)
     { // The WAVEINCAPS structure describes the capabilities
       // of a waveform-audio input device.
       // Drivers for the standard multi-media system are identified by NUMBERS.
      if( my_waveincaps.wChannels > 0 )  // "accept" the device if it has more than ZERO CHANNELS
       {
        AddAudioInputDevice( AnsiString(my_waveincaps.szPname) );
        // Note: Names are already UGLILY TRUNCATED by waveInGetDevCaps(),
        //       e.g. "Onboard-Microphone (Realtek Hig" - that's all !
       }
     }
   }
#endif // SWI_USE_DSOUND / SWI_USE_WAVE_AUDIO / .. ?
  if( CB_AudioInputDevice->ItemIndex < 0 ) // no match for CwKeyer_DSP.sz255AudioInputDevice ?
   {  CB_AudioInputDevice->ItemIndex = 0;  // item index for "None"
   }

  //- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // Above: enumerate INPUT audio devices  (bare Win32 "wave audio" API) .
  // Below: enumerate OUTPUT audio devices ...
  //- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  //--------- Build a list of available AUDIO OUTPUT DEVICES -----------------
  CB_AudioOutputDevice->Items->Clear();
  CB_AudioOutputDevice->ItemIndex = -1; // hopefully set to a valid index below
  AddAudioOutputDevice( "None" ); // .. and SELECT this item if it matches CwKeyer_DSP.sz255AudioOutputDevice
#if( SWI_USE_DSOUND ) // use the mature "DirectSound" / dsound_wrapper.c ?
  for(i=0; i<MyDirectSound.m_nOutputDevices; ++i)
   { AddAudioOutputDevice( AnsiString(MyDirectSound.m_sOutputDevices[i].sz255DevName) );
   }
#elif( SWI_USE_WAVE_AUDIO ) // use the ancient "Windows Multimedia Extensions" / "wave audio" ?
  AddAudioOutputDevice( "Default WAVE output" );
  n_devs = waveOutGetNumDevs();
  for( i=0; i<n_devs; ++i)
   {
    if( waveOutGetDevCaps(    i, // UINT uDeviceID
                &my_waveoutcaps, // LPWAVEOUTCAPS pwoc,
            sizeof(WAVEOUTCAPS)) // UINT cbwoc
       == MMSYSERR_NOERROR)
     { // The WAVEOUTCAPS structure describes the capabilities
       // of a waveform-audio input device.
       //
      if( my_waveincaps.wChannels > 0 )  // "accept" the device if it has more than ZERO CHANNELS
       {
         AddAudioOutputDevice( AnsiString(my_waveoutcaps.szPname) );
         // Examples: s = "SoundMAX HD Audio"
       }
     }
   }
#endif // SWI_USE_DSOUND / SWI_USE_WAVE_AUDIO / .. ?
  if( CB_AudioOutputDevice->ItemIndex < 0 ) // no match for CwKeyer_DSP.sz255AudioOutputDevice ?
   {  CB_AudioOutputDevice->ItemIndex = 0;  // item index for "None"
   }
  --m_iUpdating;
  HERE_I_AM__GUI();

} // end TKeyerMainForm::UpdateAudioDeviceCombos()

//---------------------------------------------------------------------------
void TKeyerMainForm::StartVFOEditTimer( int nMilliseconds )
{
  if( g_SpecDispControl.iVFOEditTimer_ms <= 0 )  // change the VFO editor's background colour...
   { switch( g_SpecDispControl.iColourScheme ) // .. depending on the colour scheme..
      { case COLOUR_SCHEME_DEFAULT: // e.g. white background in windows
        default:
           Ed_VFO->Color = (TColor)0xC0FFC0; // light green (0xBBGGRR)
           break;
        case COLOUR_SCHEME_DARK:
           Ed_VFO->Color = (TColor)0x008000; // dark green
           break;
      }
   }
  g_SpecDispControl.iVFOEditTimer_ms = nMilliseconds; // suppress assigning new text to Ed_VFO "while typing" !
} // end TKeyerMainForm::StartVFOEditTimer()

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Ed_VFOChange(TObject *Sender) // USER edits Ed_VFO..
{
  if(m_iUpdating) return;  // don't react on "OnChange" from PROGRAM, not USER
  ApplyNewVFOFreq(); // Ed_VFO -> MyCwNet.RigControl.dblVfoFrequency
  StartVFOEditTimer(10000/*ms*/);
} // end TKeyerMainForm::Ed_VFOChange()

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Ed_VFOKeyDown(TObject *Sender, WORD &Key, TShiftState Shift)
{
  // Intercept certain key codes for the "VFO-like" frequency editor:
  TEdit *pEd = dynamic_cast<TEdit *>(Sender);
  char c, szA[44], *cp, cThouSep;
  int i, slen;
  long   iResult;
  double dblFreqHz, dblFreqIncrement;
  BOOL fSetNewFreq = FALSE;
  BOOL fUpdateEditField = FALSE;

  if( (pEd!=NULL) && (m_iUpdating==0) )  // only on USER INPUT...
   {
     ++m_iUpdating;  // disable certain event handlers

     // Get the current frequency from the visible control:
     dblFreqHz = GetFrequencyFromEditField( pEd, &dblFreqIncrement );
     // Modify it with the cursor keys by adding/subtracting a power of ten:
     switch( Key )
      { case VK_UP:
           dblFreqHz += dblFreqIncrement;
           fSetNewFreq = fUpdateEditField = TRUE;
           Key = 0;
           break;
        case VK_DOWN:
           dblFreqHz -= dblFreqIncrement;
           fSetNewFreq = fUpdateEditField = TRUE;
           Key = 0;
           break;
        case VK_RETURN:   // "Finish" the input; apply and convert to standard form
           // ( for example, turn "3M5" into  "3 500 000 Hz" )
           fSetNewFreq = TRUE;
           Key = 0;       // Note: Why does the stupid windows say "Clunk" when pressing the ENTER-key ?!
           break;
        default:    // all other keys will be passed on to the edit control
           break;
      } // end switch( Key )
     if( fSetNewFreq )
      { RigCtrl_SetVFOFrequency( &MyCwNet.RigControl, dblFreqHz );
        // Even if the new frequency hasn't arrived at the remote rig yet,
        // already update the LOCAL frequency display, to have an immediate feedback:
        if( fUpdateEditField )
         { UpdateVFODisplay( dblFreqHz );
         }
        g_SpecDispControl.iVFOEditTimer_ms = 20000; // suppresses assigning new text to Ed_VFO "while typing" !
      } // end if( fSetNewFreq )
     --m_iUpdating;
   } // end if < sender is a TEdit control >
} // end TKeyerMainForm::Ed_VFOKeyDown()
//---------------------------------------------------------------------------

void __fastcall TKeyerMainForm::EditSuppressEnterKey(TObject *Sender, char &Key)
{
  // Just another kludge for stupid windows behaviour.
  // Without this, Windows goes "Clunk" or "DongDongDong" whenever pressing ENTER
  // (aka "Return") when the focus is on this input field.
  // Copy the name of this method (EditSuppressEnterKey) into the
  // "OnKeyPress"-event of all TEdit controls which shall NOT go "Clunk"/"Dong"!
  if( Key==VK_RETURN )
   {  Key = 0;
   }
}
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
int TKeyerMainForm::IdentifyClientCoord( int iClientX, int iClientY,
         float *pfltXrel, // optional output: relative 'X' coord, 0...1 within the area
         float *pfltYrel) // optional output: relative 'X' coord, 0...1 within the area
  // Identifies the 'area' within Grp_Spectra for a given client coordinate.
  // Return values:  SCREEN_AREA_NOT_DEFINED  (i.e. "not identified")
  //                 SCREEN_AREA_SPECTRUM_GRAPH,
  //                 SCREEN_AREA_FREQUENCY_SCALE,
  //                 SCREEN_AREA_SPECTROGRAM .
{
  int x,y;
  int iSpecDispArea = SCREEN_AREA_NOT_DEFINED;
  float fltXrel, fltYrel;
  fltXrel = fltYrel = 0.0;

  // Check the different 'areas' within the main window, from top to bottom.
  x = iClientX - Img_Spectrum->Left;
  y = iClientY - Img_Spectrum->Top;
  if( x>=0 && x<Img_Spectrum->Width && y>=0 && y<Img_Spectrum->Height )
   {  iSpecDispArea = SCREEN_AREA_SPECTRUM_GRAPH;
      fltXrel = (float)x / (float)Img_Spectrum->Width;
      fltYrel = (float)y / (float)Img_Spectrum->Height;
   }
  if( pfltXrel != NULL )
   { *pfltXrel = fltXrel;
   }
  if( pfltYrel != NULL )
   { *pfltYrel = fltYrel;
   }
  return iSpecDispArea;
} // end TKeyerMainForm::IdentifyClientCoord()




//---------------------------------------------------------------------------
// Machine-generated functions for the GUI (from Borland's form designer),
//   "manually populated" of course to do something meaningful..
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::FormCreate(TObject *Sender)
{
  char sz255Temp[256], *cp;
  WSADATA wsa;
  int i,n;
  TEdit *pEdit;


  HERE_I_AM__GUI();

  // Due to the use of hardware resources like "COM ports",
  // this application must only run in a SINGLE INSTANCE. Make sure it does:
  UTL_Init();  // <- MAY contain built-in self tests; result in UTL_sz255LastError
  if( UTL_iAppInstance >= 3 ) // oops... too many instances of this program running already !
   {
     ::MessageBox( Handle, "Sorry, only up to THREE instances of this program\n"
                           "may run simultaneously on the same machine (PC)\n"
                           " - which they already do.",
                           "Remote CW Keyer", MB_OK );
     UTL_Exit();
     Application->Terminate(); // > " By calling Terminate rather than freeing
                      // > the application object, you allow the application
                      // > to shut down in an orderly fashion."  - sounds good.
   }

  // There may be a VCL-things to retrieve the command line,
  //  but we don't use it .. being fed-up with the growing bulk.
  memset( g_sz255CommandLine, 0, sizeof(g_sz255CommandLine) );
  HERE_I_AM__GUI();
  cp = ::GetCommandLine(); // <- this is ancient Windows, not Borland/Embercadero
  HERE_I_AM__GUI();
  if( cp != NULL )
   { SL_strncpy( g_sz255CommandLine, cp, 255 );
     // Note: The command line contains the name of the executable being launched.
     //       In other words, it is NEVER EMPTY, even without arguments !
     //       e.g. g_sz255CommandLine = "\"C:\cproj\Remote_CW_Keyer\Remote_CW_Keyer.exe\"\0"
     //       (note the double quote characters INSIDE THE STRING ITSELF,
     //        to tell space characters in file- and directory names
     //        from space characters separating ARGUMENTS in the command line).
     ParseCommandLine(g_sz255CommandLine);
   }
  HERE_I_AM__GUI();
  UpdateWindowCaption(); // -> e.g. "Remote CW Keyer [1]  (SERVER)/(Client)"

  TIM_Init();  // initialize e.g. the "high-resolution timestamps"

  // Because even "WindowProc" didn't intercept certain keys tried this:
  Application->OnMessage = AppMessage;
  DecimalSeparator = '.'; // a VCL thing controlling "FormatFloat()"..
  // don't want to see the rotten ',' when the entire GUI is in english !
  // Without the following, some crappy VCL mechanism (or Windows)
  // kept changing the stupid "DecimalSeparator" back to the stupid COMMA
  // at irregular intervals (reason further below) :
  Application->UpdateFormatSettings = FALSE;
  // Found this by accident in the LAZARUS (free Pascal) forum,
  // about the TRIPLE STUPID "DecimalSeparator" / "decimalSeparator" :
  // > I gather this variable is now deprecated and we should use
  // > formatsettings.defaultseperator.
  // > Now in the scientific world, decimalSeparator is
  // > application specific, not locale specific.
  // > So I need to make sure that all my apps only use '.' wherever
  // > they are used in the world.
  // > The problem with delphi's decimalSeparator variable was that
  // > it didn't obey what I told it to do and it would periodically
  // > set itself back to the locale setting.
  //    ( A-ha ! Can somebody please shoot the guy who implemented it that way ? )
  // > Does the Lazarus settings do this, can I set it only once
  // > at the start of the app or do I have to keep checking it
  // > every time I do a float conversion?
  //      (...)
  // > Until recent, Lazarus did not update Formatsettings when Windows
  // > sent a message that they were changed, but nowadays it does (like Delphi).
  // > > You can prevent this by adding this code [not applicable in BCB]
  // > >     // prevent unwanted re-setting of DefaultFormatsettings
  // > >     Application.UpdateFormatSettings := False;

#if( SWI_USE_HTTP_SERVER )  // support the optional, built-in HTTP server ?
  // The VCL's "Application"-thingy had an "Icon" property (a "TIcon",
  // which can be IMPORTED VIA OBJECT INSPECTOR IN THE MAIN FORM, "Icon"..).
  // Can we use this directly as our web server's "favicon.ico" ? Of course not,
  // because an icon is not an icon - there are myriads of "icon file types",
  // so convert, using TIconToFaviocon.cpp (unfortunately depends on the VCL):
  if( Application->Icon != NULL )
   { HttpSrv_iSizeofFavicon = TIconToFavicon( KeyerMainForm->Icon, // ex: Application->Icon,
             HttpSrv_bFavicon/*out*/, sizeof(HttpSrv_bFavicon)/*maxlen*/ );
     // (for some reason, "Application->Icon" retrieved a 40*40-pixel-icon,
     //  while  "KeyerMainForm->Icon" retrieves the original 32*32-pixel-icon.)
   }
#endif // SWI_USE_HTTP_SERVER ?

  // Fire up the Windows Socket Services (only ONCE.. thus FROM HERE) :
  HERE_I_AM__GUI();
  WSAStartup(MAKEWORD(2,2),&wsa); // already initialized in the main module
    // ,---------------------'
    // '-> "data structure that is to receive details of the Windows Sockets implementation"


  m_fDontSaveOnExit = FALSE;
  m_iStatusIndicatorUsage = -1;
  m_iStatusIndicatorMemIdx= 0;
#if( SWI_USE_DSOUND ) // use "DirectSound" / dsound_wrapper.c ?
  HERE_I_AM__GUI();
  DSW_Init( &MyDirectSound ); // init our "DirectSound wrapper" instance, first used for the SIDETONE OUTPUT
  DSW_EnumerateDevices( &MyDirectSound ); // .. for the two audio device combos ..
#endif // SWI_USE_DSOUND ?
#if( SWI_USE_HAMLIB_SERVER ) // build an application with integrated "Hamlib Net rigctld"-compatible server ?
  HLSRV_InitInstanceWithDefaults( &MyHamlibServer ); // <- must be called BEFORE loading the 'config' from a file
#endif // SWI_USE_HAMLIB_SERVER ?
  HERE_I_AM__GUI();
  SpecDisp_InitControl( &g_SpecDispControl, 32/*iFreqScaleHeight*/ );
  LoadSettings(); // <- may use the FIRST ENUMERATED DirectSound device as default,
                  //  thus enumerate the devices before loading the configuration.
  HERE_I_AM__GUI();
  ++m_iUpdating;  // prevent "interference" from certain VCL event handlers
                  // The anonymous-looking stuff like "Left","Top", etc are
                  // properties of Borland's 'TForm' (or something like that).
  Left  = CwKeyer_Config.iMainWindowLeft;
  Top   = CwKeyer_Config.iMainWindowTop;
  Width = CwKeyer_Config.iMainWindowWidth;
  Height= CwKeyer_Config.iMainWindowHeight;
  for(i=0; i<KEYER_NUM_MESSAGE_MEMORIES; ++i)
   { pEdit = GetEditorForMemoryIndex( i/*iMemoryIndex*/ );
     if( pEdit != NULL ) // only if there really is an "editor" for this message memory..
      { pEdit->MaxLength = KEYER_MAX_CHARS_PER_MEMORY;
        pEdit->Text = CwKeyer_Config.szKeyerMemory[i];
      }
   } // end for < all 'keyer message memories' >
  Ed_ErrorHistory->WordWrap = (MyCwNet.cfg.iDiagnosticFlags & CWNET_DIAG_FLAGS_WORD_WRAP) != 0;

  --m_iUpdating; // allow dozens of VCL-"controls" (GUI elements) to react again

  HERE_I_AM__GUI();
  m_fAdaptImageSizesOnTRXTab = TRUE; // call AdaptImageSizesOnTRXTab() if that tab ever gets visible
  SetColourScheme(g_SpecDispControl.iColourScheme); // <- sounds easy but turned into a nightmare
  g_SpecDispControl.fClearWaterfall = TRUE; // fill the (empty) waterfall with the scheme-depending background colour (ONCE)
  HERE_I_AM__GUI();
  ShowError( ERROR_CLASS_INFO | SHOW_ERROR_IN_RUN_LOG, "Remote CW Keyer compiled %s", __DATE__ );
  if( UTL_sz255LastError[0] != '\0' )
   { ShowError( ERROR_CLASS_INFO | SHOW_ERROR_IN_RUN_LOG,
       "Self test in UTL_Init() failed: %s", UTL_sz255LastError );
   }
  HERE_I_AM__GUI();
  if( UTL_GetPathAndNameOfExecutable( sz255Temp, 230/*maxlen!*/ ) )
   { // '--> e.g. "C:\\cbproj\\Remote_CW_Keyer\\Remote_CW_Keyer.exe" .
     // Replace the name by "Translations.c" (one of the SOURCECODE MODULES) :
     cp = strrchr( sz255Temp, '\\' );
     if( cp != NULL )
      { strcpy( cp, "\\Translations.c" );
        // '--> e.g. sz255Temp = "C:\\cbproj\\Remote_CW_Keyer\\Translations.c"
        //      (As noted in 'Translations.c', placing A COPY of that file
        //       in the same directory as the executable (*.exe)
        //       will REPLACE THE BUILT-IN ("hardcoded") string table.
        //       This simplifies the work of volunteers, and allows them
        //       to test a new language WITHOUT the need to
        //       re-compile / re-build the application. )
        n = APPL_LoadTranslationsFromFile( sz255Temp/*pszPathAndFilename*/ );
        if( n > 0 )
         { ShowError( ERROR_CLASS_INFO | SHOW_ERROR_IN_RUN_LOG,
              "Loaded %d lines of translations from %s .", (int)n, sz255Temp );
           // ,--------'
           // '--> Service for volunteers to TEST TRANSLATIONS without rebuilding the project.
         } // end if < APPL_LoadTranslationsFromFile() successful >
        else
         { ShowError( ERROR_CLASS_INFO | SHOW_ERROR_IN_RUN_LOG,
              "Using the built-in translation table (Translations.c) with %d lines.",
              (int)APPL_GetNumEntriesInTranslationTable(TranslationTable) );
         }
      }
   } // end if < UTL_GetPathAndNameOfExecutable() successful >
  if( g_iNewLanguage != APPL_iLanguage )
   { SetNewLanguage( g_iNewLanguage );  // e.g. switch from ENGLISH to GERMAN
   }
  i = INET_Init(); // returns a NEGATIVE error code when the built-in self-test failed..
  HERE_I_AM__GUI();
  if( i < 0 )
   { ShowError( ERROR_CLASS_INFO | SHOW_ERROR_IN_RUN_LOG,
       "Self test in INET_Init() failed with error code %d", (int)i );
   }

  memset( s_iBandStackingRegIndices, -1, sizeof(s_iBandStackingRegIndices) );

  RigCtrl_Init( &MyCwNet.RigControl, // initialize everything for a "single instance"
     CwKeyer_Config.iRadioControlProtocol, // [in] e.g. RIGCTRL_PROTOCOL_ICOM_CI_V,
     CwKeyer_Config.iRadioCIVAddress,      // [in] e.g.
     &Keyer_RadioControlPortRxFifo.fifo, &Keyer_RadioControlPortTxFifo.fifo,
     0,      // [in] 0..RIGCTRL_MAX_CLIENT_PORTS
     NULL ); // [in] array, one per client
     // Loaded later from a config file: RigCtrl_iTrafficMonitorDisplayOptions ..
  MyCwNet.RigControl.iRigControlFlags |= RIGCTRL_FLAG_WANT_SPECTRUM_DATA;
  MyCwNet.RigControl.dwRejectMessages4Log = CwKeyer_Config.dwRejectMessages4Log;
#if( SWI_USE_HAMLIB_SERVER ) // build an application with integrated "Hamlib Net rigctld"-compatible server ?
  HLSRV_LinkToRigControl( &MyHamlibServer, &MyCwNet.RigControl ); // <- must be called AFTER RigCtrl_Init() !
#endif // SWI_USE_HAMLIB_SERVER ?

  // When first launched with the DEFAULT CONFIGURATION, switch to the
  // "I/O Config" tab, because without the proper setting there the keyer is useless:
  SwitchMainTab( MAIN_TAB_CONFIG ); // ex: PageControl1->ActivePage = TS_Config;
  // ,-----------------------'
  // '--> Formerly "I/O Config", now with serveral sub-tabs like "I/O" and "Rig Control":
  PC_Config->ActivePage = TS_IO;
  UpdateSettingsTab();
  UpdateAudioDeviceCombos();
  UpdateNetworkSetup();

  // The same old annoying story with "flickering screens" in Windows (or the VCL)...
  Img_TimingScopeOnKeyerTab->ControlStyle = Img_TimingScopeOnKeyerTab->ControlStyle << csOpaque; // tried this to reduce flicker...
  Img_TimingScopeOnTestTab->ControlStyle = Img_TimingScopeOnTestTab->ControlStyle << csOpaque;
  Img_Spectrum->ControlStyle  = Img_Spectrum->ControlStyle << csOpaque;
  Img_Waterfall->ControlStyle = Img_Waterfall->ControlStyle << csOpaque;
  Img_FreqScale->ControlStyle = Img_FreqScale->ControlStyle << csOpaque;
  Img_SMeter->ControlStyle    = Img_SMeter->ControlStyle << csOpaque;
  Img_TRX_Bottom->ControlStyle= Img_TRX_Bottom->ControlStyle << csOpaque;

  // If, as happened hundreds of times before, the above (alone) doesn't fix
  // the annoying flicker, it sometimes helped to set 'csOpaque' also for the
  // image's parent, which in this particular case are TTabSheets :
  TS_KeyerSettings->ControlStyle = TS_KeyerSettings->ControlStyle << csOpaque;
  TS_Test->ControlStyle = TS_Test->ControlStyle << csOpaque;
  TS_TRX->ControlStyle  = TS_TRX->ControlStyle << csOpaque;

  // Sometimes a flickering window can drive you mad (as in this case) !
  // Because setting the control style csOpaque above
  // (which should prevent sending those useless WM_ERASEBKGND messages),
  // did *not* cure the flicker (it only fixed it "to some degree"),
  // so added the following ..
  HERE_I_AM__GUI();
  DoubleBuffered=TRUE;  // .. because "csOpaque" alone didn't cure the flicker !

  // Prepare the built-in "Help system" ...
  //  YHF_HELP_Init() expects a full path to the HTML help files.
  HERE_I_AM__GUI();
  if( ! GetModuleFileName(
                  NULL, // handle to module to find filename for
             sz255Temp, // pointer to buffer for module path
                  200 ) )  // size of buffer, in characters
   { // GetModuleFileName() failed  [shit happens; it's Windows] ..
     sz255Temp[0] = 0;
   }
  else  // got a full path to this "Module", e.g. sz255Temp="C:\\cbproj\\Remote_CW_Keyer\\Remote_CW_Keyer.exe" .
   { // Remove the name of the executable, and replace that with the name of the
     // subdirectory with the "help files" (in this case, just "manual") :
     cp = strrchr( sz255Temp, '\\' );
     if( cp==NULL ) // oops, there no backslash in this thing.. someone's got a life
      {             // and uses FORWARD slashes in paths ? ;o)
        cp = strrchr( sz255Temp, '/' );
      }
     if ( cp != NULL ) // ok, got it..
      { strcpy( cp+1, "manual" );
      }
   }
  HERE_I_AM__GUI();
  YHF_HELP_Init(
       Handle,    //  handle of the application's main window (a "HWND"; this "Handle" is a VCL property of the form)
       sz255Temp, //  [in] char *html_file_path, path to html files, e.g. "C:\\cbproj\\Remote_CW_Keyer\\manual" .
       "http://www.qsl.net/dl4yhf/Remote_CW_Keyer/", // alternative WEB LOCATION of the html file(s)
                  // (used if the 'local' html_file_path is invalid, for example
                  //  because only the executable has been unpacked but nothing else)
      MyHelpMap,  //  [in] T_YHF_HelpMapEntry *pHelpMap (translation table for Help Topic IDs)
             1);  //  iLanguageCode, usually the international dialing code

  m_dwHashForSettings = CalcHashForSettings();

  m_iUpdating = 0;
  HERE_I_AM__GUI();

}   // end TKeyerMainForm::FormCreate()
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
void TKeyerMainForm::UpdateWindowCaption(void) // -> e.g. "Remote CW Keyer [1]  (SERVER)/(Client)"
{ AnsiString s = "Remote CW Keyer [" + IntToStr(UTL_iAppInstance+1) + "]";
  switch( MyCwNet.cfg.iFunctionality )
   { case CWNET_FUNC_CLIENT :
        s = s + "   (Client)";
        break;
     case CWNET_FUNC_SERVER :
        s = s + "   (Server)";
        break;
     default:  // nothing special to write home about .. just a "computer controlled keyer"
        break;
   }
  Caption = s;
}

//---------------------------------------------------------------------------
void TKeyerMainForm::ApplyAndStart(void)
{
  HERE_I_AM__GUI();
  CwKeyer_Stop();  // stop the keyer (with the old config, if it was running at all)
  HERE_I_AM__GUI();
  CwDSP_Stop( &CwKeyer_DSP ); // stop the 'audio DSP', too
  HERE_I_AM__GUI();
  ApplySettingsTab();
  UpdateSettingsTab();  // kludge to re-enumerate the serial port(s) in the combo lists,
   // for example after unplugging/reconnecting the RADIO's USB-port
   // to find out which of the foolishly-named "COM Ports" is currently connected
   // to the IC-9700, which to the IC-7300, and which to the IC-705.
   // All of what Microsoft calls "friendly names" don't offer any clues,
   // for example:
   //   * a "Digitus" USB-to-serial-port adapter appeared as "COM12 (USB Serial Port)"
   //   * any if Icom's IC-9700 / IC-7300 / IC-705 appeared as
   //       "COMx (Silicon Labs CP210x USB to UART Bridge)" .
   // Nnnngrrr !
   // To find out if the selected COM port *really* leads to the intended radio,
   //   * select it in the combo list, e.g. "COM5 (Silicon Labs CP210x USB to UART Bridge)"
   //   * UNPLUG the USB cable on the radio's side
   //   * Click on the "Apply / Start" button
   //   * If the unplugged radio was really "COM5 (Silicon Labs CP210x USB to UART Bridge)",
   //     the combo box will now show "COM5 *INVALID*" .
  StartKeyerAndShowInfo();
  HERE_I_AM__GUI();

} // end TKeyerMainForm::ApplyAndStart()
//---------------------------------------------------------------------------


void __fastcall TKeyerMainForm::Btn_ApplyAndStartClick(TObject *Sender)
{
  ApplyAndStart();
}
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::TrackBar_WPMChange(TObject *Sender)
  // Called from Borland's VCL when modifying the CW speed via 'trackbar'
{
  int iNewPosition_WPM = TrackBar_WPM->Position;
  int iNewDotTime_ms   = Elbug_WordsPerMinuteToDotTimeInMilliseconds( iNewPosition_WPM );
  if( (m_iUpdating==0) && (iNewDotTime_ms != CwKeyer_Elbug.cfg.iDotTime_ms) )
   { CwKeyer_Elbug.cfg.iDotTime_ms = CwKeyer_Gen.cfg.iDotTime_ms
       = CwKeyer_StraightKeyDecoder.iDotTime_ms = iNewDotTime_ms;
     ++m_iUpdating;
     Ed_WPM->Text = IntToStr( iNewPosition_WPM );
     --m_iUpdating;
     // Redraw the timing scope a.s.a.p., because a "tick" on the timescale
     // shall represent one dot-time :
     TimingScope_iUpdateCountOnTestTab = -1;  // <- polled in "Timer1Timer()" ..
     TimingScope_iUpdateCountOnKeyerTab= -1;
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Ed_WPMChange(TObject *Sender)
  // Called from Borland's VCL when modifying the CW speed via EDIT FIELD.
{
  int iNewPosition_WPM = StrToIntDef( Ed_WPM->Text, 0 );
  int iNewDotTime_ms   = Elbug_WordsPerMinuteToDotTimeInMilliseconds( iNewPosition_WPM );
  if( (m_iUpdating==0) && (iNewPosition_WPM != 0) && (iNewDotTime_ms != CwKeyer_Elbug.cfg.iDotTime_ms) )
   { CwKeyer_Elbug.cfg.iDotTime_ms = CwKeyer_Gen.cfg.iDotTime_ms
       = CwKeyer_StraightKeyDecoder.iDotTime_ms = iNewDotTime_ms;
     ++m_iUpdating;
     TrackBar_WPM->Position = iNewPosition_WPM;
     // Note: Setting the trackbar position 'programmatically' here
     //       causes Borland's VCL to invoke the trackbar's "OnChange"-handler, too.
     //       Setting m_iUpdating here prevents interference from that handler.
     --m_iUpdating;
     // Also here: Redraw the timing scope a.s.a.p., because a "tick"
     //            on the timescale shall represent one dot-time :
     TimingScope_iUpdateCountOnTestTab = -1;  // <- polled in "Timer1Timer()" ..
     TimingScope_iUpdateCountOnKeyerTab= -1;

   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Exit1Click(TObject *Sender)
{
  HERE_I_AM__GUI();
  Close();
  HERE_I_AM__GUI();
}
//---------------------------------------------------------------------------

void __fastcall TKeyerMainForm::MI_ExitNoSaveClick(TObject *Sender)
{
  HERE_I_AM__GUI();
  m_fDontSaveOnExit = TRUE;
  Close(); // -> TKeyerMainForm::FormClose(), etc etc
  HERE_I_AM__GUI();
}
//---------------------------------------------------------------------------

void __fastcall TKeyerMainForm::FormClose(TObject *Sender, TCloseAction &Action)
{
  int i;
  TEdit *pEdit;

  if( UTL_iAppInstance >= 0 )     // no "premature termination" from FormCreate() ?
   { APPL_fLaunchKeyerOnStart = (CwKeyer_iThreadStatus==KEYER_THREAD_STATUS_RUNNING);
     UTL_WriteRunLogEntry( "Beginning to close the main form, instance #%d ..", (int)UTL_iAppInstance+1 );

     // Put the main form's window position and -size into the old-fashioned
     // configuration structure (because THAT's what SaveSettings() will save,
     // if the 32-bit CRC for the entire CwKeyer_Config is different now..)
     CwKeyer_Config.iMainWindowLeft  = Left;
     CwKeyer_Config.iMainWindowTop   = Top;
     CwKeyer_Config.iMainWindowWidth = Width;
     CwKeyer_Config.iMainWindowHeight= Height;
     CwKeyer_Config.dwRejectMessages4Log = MyCwNet.RigControl.dwRejectMessages4Log;     
     for(i=0; i<KEYER_NUM_MESSAGE_MEMORIES; ++i)
      { pEdit = KeyerMainForm->GetEditorForMemoryIndex( i/*iMemoryIndex*/ );
        if( pEdit != NULL ) // only if there really is an "editor" for this message memory..
         { SL_strncpy( CwKeyer_Config.szKeyerMemory[i], pEdit->Text.c_str(), KEYER_MAX_CHARS_PER_MEMORY );
         }
      } // end for < all 'keyer message memories' >
     HERE_I_AM__GUI();
     CwNet_Stop( &MyCwNet );      // stop network activities (if any)
     HERE_I_AM__GUI();
     UTL_WriteRunLogEntry( "Stopping the keyer thread." );
     CwKeyer_Stop();              // stop the keyer / keyer thread / etc before terminating
     HERE_I_AM__GUI();            // 2024-01-06 : Crashed with the infamous "0xFEEEFEEE" shortly AFTER this point..
     UTL_WriteRunLogEntry( "Stopping the DSP thread." );
     CwDSP_Stop( &CwKeyer_DSP );  // stop the 'audio DSP', too (BEFORE "terminating DirectSound")
     HERE_I_AM__GUI();            // 2024-01-06 : Crashed with the infamous "0xFEEEFEEE" shortly BEFORE this point.
#   if( SWI_USE_DSOUND ) // use "DirectSound" / dsound_wrapper.c ?
     UTL_WriteRunLogEntry( "Stopping DirectSound." );
     DSW_Term( &MyDirectSound );  // terminate the D[irect]S[ound]W[rapper]
     HERE_I_AM__GUI();
#   endif // SWI_USE_DSOUND ?
     if( (!m_fDontSaveOnExit )
      && ((m_dwHashForSettings != CalcHashForSettings()) || (APPL_fSaveSettingsOnExit) ) )
      { HERE_I_AM__GUI();
        UTL_WriteRunLogEntry( "Saving modified settings." );
        SaveSettings();
        HERE_I_AM__GUI();
      }
     UTL_WriteRunLogEntry( "Calling WSACleanup()." );
     WSACleanup(); // we're out of business, Mr. Winsock !
     HERE_I_AM__GUI();
     APPL_UnloadTranslationsFromFile();
     UTL_WriteRunLogEntry( "Last step of closing the main form, also closing the logfile." );
     UTL_Exit();   // we're out of business in Utilities.c !
   } // end if( UTL_iAppInstance >= 0 )
  else
   { UTL_WriteRunLogEntry( "Beginning to close the main form, NO VALID INSTANCE INDEX." );
   }
  HERE_I_AM__GUI(); // arrived in this line ? Survived TKeyerMainForm::FormClose() !
} // end TKeyerMainForm::FormClose()
//---------------------------------------------------------------------------

void __fastcall TKeyerMainForm::MI_RestoreDefaultSettingsClick(TObject *Sender)
{
  BOOL fWasRunning = (CwKeyer_iThreadStatus==KEYER_THREAD_STATUS_RUNNING);
  char *pszInfo;
  HERE_I_AM__GUI();
  switch( APPL_iLanguage )
   { case 49:
        pszInfo = "Mchten Sie wirklich alle Einstellungen dieses Programms,\n"
                  "inklusive der I/O- und Netzwerk-Konfiguration,\n"
                  "wieder auf die Defaultwerte setzen ?";
        break;
     case 1:
     default:
        pszInfo = "Do you really want to set this program's settings,\n"
                  "including the I/O- and network configuration,\n"
                  "back to the defaults as after a fresh install ?";
        break;
   }
  if( ::MessageBox( Handle, pszInfo, "Remote CW Keyer", MB_ICONQUESTION | MB_YESNO) != ID_YES )
   { return;
   }

  CwKeyer_Stop();       // stop the keyer (with the old config, if it was running at all)
  CwDSP_Stop( &CwKeyer_DSP ); // stop the DSP (with the old config, " " " " )
  HERE_I_AM__GUI();
  CwKeyer_SetDefaultConfig( &CwKeyer_Config );
  Elbug_InitInstanceWithDefaults( &CwKeyer_Elbug );
  UpdateSettingsTab();  // kludge to re-enumerate the serial port(s) in the combo lists,
  HERE_I_AM__GUI();
  if( fWasRunning )
   { HERE_I_AM__GUI();
     CwDSP_Start( &CwKeyer_DSP ); // start the 'audio DSP', possibly with NEW settings
     HERE_I_AM__GUI();
     CwKeyer_Start();  // try to start the CW keyer, now with the NEW settings
     HERE_I_AM__GUI();
     m_fRestartKeyerThread = FALSE;  // "done" (restarted the keyer)
   }
} // end TKeyerMainForm::MI_RestoreDefaultSettingsClick()
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
BOOL TKeyerMainForm::DumpErrorHistoryToDebugTab(void)
  // Drains messages for the "Debug" tab from the thread-safe message FIFO
  //        and emits them to a VCL RichText edit control ("Ed_ErrorHistory").
{
  T_ErrorFifoEntry *pErrorFifoEntry;
  DWORD dwTextColor  = g_SpecDispControl.clWindowText; // default, depending on the colour scheme ..
  DWORD dwBkgndColor = g_SpecDispControl.clWindowBackground;
  BOOL fAppendedNewText = FALSE;


  DEBUG_iErrorHistoryTail %= C_ERROR_FIFO_SIZE;
  while( DEBUG_iErrorHistoryHead != DEBUG_iErrorHistoryTail )
   {
     pErrorFifoEntry = &ErrorHistoryFifo[ DEBUG_iErrorHistoryTail ];
     DEBUG_iErrorHistoryTail = (DEBUG_iErrorHistoryTail+1) % C_ERROR_FIFO_SIZE;
     // Just having stupid different BACKGROUND COLOURS for individual lines
     // in Borland's "TRichEdit" control is incredibly complex. Principle:
     // 1.) Save the RECENT cursor position before adding the text
     // 2.) Add the new text (line)
     // 3.) Using the stored position from step one, turn the new text into a SELECTION
     // 4.) Perform some voodoo magic on the selection to achieve the desired effect.
     // Note: The stupid 'index' of characters in Borland's "AnsiString" type
     //       begin at ONE (not ZERO as a C programmer would expect) .
     // >>> This information is duplicated in various other modules.  <<<
     // >>> The 'master' is in C:\cbproj\SpecLab\DebugU1.cpp .        <<<
     // >>> Everything else is a possibly outdated *COPY* .           <<<
     Ed_ErrorHistory->SelStart = Ed_ErrorHistory->Text.Length();
     Ed_ErrorHistory->SelLength= 1;
     // utterly useless: Ed_ErrorHistory->SelAttributes->NoStupidBackgroundColor = ?
     // (these brainless "SelAttributes" don't support background colours,
     //  even though the native windows "RichEdit" control supports it since ancient times)
     switch( pErrorFifoEntry->iErrorClass & ERROR_CLASS_MASK ) // which background colour ?
      {
        case ERROR_CLASS_FATAL:
              dwBkgndColor = 0xFF00FF; // colour format, from MSBIT to LSBIT: BLUE, GREEN, RED (8 bit each)
              dwTextColor  = 0x000000;
              break;
        case ERROR_CLASS_ERROR:
              dwBkgndColor = 0x0000FF; // colour format, from MSBIT to LSBIT: BLUE, GREEN, RED (8 bit each)
              dwTextColor  = 0xFFFFFF;
              break;
        case ERROR_CLASS_WARNING:
              dwBkgndColor = 0x00FFFF; // colour format, from MSBIT to LSBIT: BLUE, GREEN, RED (8 bit each)
              dwTextColor  = 0x000000;
              break;
        case ERROR_CLASS_RX_TRAFFIC: /* received "network traffic"    (TCP/IP, HTTP) */
              dwBkgndColor = 0xC0FFC0; // colour format, from MSBIT to LSBIT: BLUE, GREEN, RED (8 bit each)
              dwTextColor  = 0x000000;
              break;
        case ERROR_CLASS_TX_TRAFFIC: /* transmitted "network traffic" (TCP/IP, HTTP) */
              dwBkgndColor = 0xC0FFFF; // colour format, from MSBIT to LSBIT: BLUE, GREEN, RED (8 bit each)
              dwTextColor  = 0x000000;
              break;
        case ERROR_CLASS_INFO:
        case ERROR_CLASS_ALL:
        default:
              switch( g_SpecDispControl.iColourScheme )
               { case COLOUR_SCHEME_DARK:
                    dwBkgndColor = 0x202020;
                    dwTextColor  = 0xFFFFFF;
                    break;
                 default:
                    dwBkgndColor = 0xFFFFFF;
                    dwTextColor  = 0x000000;
                    break;
               }
              break;
      } // end switch( <iErrorClass> )
     // Change the INDIVIDUAL CHARACTER BACKGROUND COLOUR for the "RichEdit"
     // (but not with Borland's VCL.. it takes more than the VCL to get a job done !) :
     // ex: RichEdit_SetSelBgColor( Ed_ErrorHistory->Handle, dwColor );
     RichEdit_SetSelColors(Ed_ErrorHistory->Handle,dwTextColor,dwBkgndColor);
     Ed_ErrorHistory->SelText = AnsiString( pErrorFifoEntry->sz511Text) + "\r\n";
     // NOW, after changing the background colour, de-select the text...
     Ed_ErrorHistory->SelLength = 0;
     // .. and set the text cursor (which in Borland's geek speak is "SelStart")
     //    to the end of the NEW text :
     // ex: Ed_ErrorHistory->SelStart = Ed_ErrorHistory->Text.Length();
     //      '--> This now happens in TKeyerMainForm::Timer1Timer(),
     //           after OTHER KINDS OF TEXT LINES have been appended,
     //           to avoid having to determine the text length
     //           over and over again. Borland's VCL and Microsoft's RichEdit-
     //           control are not spectacularly fast at this !
     // Unfortunately, during the above stupid RichText-editor-operations,
     // the stupid RichText editor's VERTICAL SCROLLBAR began to switch
     // "up and down" like crazy during each update (this only happened when
     // 'a lot of text' was in the editor). Even worse: After adding the last line,
     // and despite the 'text cursor' (aka caret) being at THE END OF THE TEXT,
     // the vertical scrollbar was often AT THE TOP. To cure this, THE CALLER
     // will "scroll the text to the caret" when finihed, conrolled be the
     // following flag:
     fAppendedNewText = TRUE;
   } // end while( DEBUG_iErrorHistoryHead != DEBUG_iErrorHistoryTail )
  return fAppendedNewText;
} // end TKeyerMainForm::DumpErrorHistoryToDebugTab()


//---------------------------------------------------------------------------
BOOL TKeyerMainForm::DumpRigCtrlTrafficToDebugTab(int iWantedCharsPerLine)
{ char szTextLine[512], szHeadLine[512];
  DWORD dwTextColor  = g_SpecDispControl.clWindowText; // default, depending on the colour scheme ..
  DWORD dwBkgndColor = g_SpecDispControl.clWindowBackground;
  int   iMsgType, iMsgType2, total_length;
  BOOL fAppendedNewText = FALSE;

  while( RigCtrl_ReadTrafficLog( &MyCwNet.RigControl, &m_iRigControlTrafficTailIndex,
                                szTextLine, iWantedCharsPerLine, &iMsgType, &dwBkgndColor ) )
   { iMsgType2 = iMsgType & RIGCTRL_MSGTYPE_STRIP_FLAGS;
     dwTextColor = 0x000000;  // .. because dwBkgndColor was for the default "black on white" scheme
     if((((iMsgType & RIGCTRL_MSGTYPE_FLAG_UDP)
         != (m_iPrevMsgTypeInTrafficLog & RIGCTRL_MSGTYPE_FLAG_UDP) )
          ||(m_iPrevMsgTypeInTrafficLog < 0 )
          ||(iMsgType & RIGCTRL_MSGTYPE_FLAG_SUGGEST_NEW_HEADLINE_FOR_TRAFFIC_LOG)
          )
       &&(!(iMsgType  &  RIGCTRL_MSGTYPE_ERROR ) )
       &&(  iMsgType2 != RIGCTRL_MSGTYPE_INFO)
        )
      { // message type changed between "UDP" and "pure CI-V" ?
        m_iPrevMsgTypeInTrafficLog = iMsgType;
        RigCtrl_GetHeadlineForTrafficLog(iMsgType, szHeadLine, iWantedCharsPerLine );  // <- this causes the "extra lines" in the RichText which are NOT in the
           // Default (white) background for COMMENT LINES emitted into the dreadful TRichEdit.
           // (explained further below, for BACKGROUND colours)
        total_length = Ed_ErrorHistory->Text.Length();
        Ed_ErrorHistory->SelStart = total_length;
        Ed_ErrorHistory->SelLength= 1;
        // ex: RichEdit_SetSelBgColor( Ed_ErrorHistory->Handle, 0xFFFFFF/*dwColor:white*/ );
        RichEdit_SetSelColors(Ed_ErrorHistory->Handle,dwTextColor,dwBkgndColor);
        Ed_ErrorHistory->SelText = AnsiString( szHeadLine ) + "\r\n";
        Ed_ErrorHistory->SelLength = 0;
      }
     // Just having different BACKGROUND COLOURS for individual lines
     //      in Borland's "TRichEdit" control is incredibly complex. Principle:
     // 1.) Save the RECENT cursor position before adding the text
     // 2.) Add the new text (line)
     // 3.) Using the stored position from step one, turn the new text into a SELECTION
     // 4.) Perform some voodoo magic on the selection to achieve the desired effect.
     // Note: The stupid 'index' of characters in Borland's "AnsiString" type
     //       begin at ONE (not ZERO as a C programmer would expect) .
     total_length = Ed_ErrorHistory->Text.Length();
     Ed_ErrorHistory->SelStart = total_length;
     Ed_ErrorHistory->SelLength= 1;
     // Change the INDIVIDUAL CHARACTER BACKGROUND COLOUR for the "RichEdit"
     // (but not with Borland's VCL.. it takes more than VCL to get a job done !) :
     // ex: RichEdit_SetSelBgColor( Ed_ErrorHistory->Handle, dwBkgndColor );
     RichEdit_SetSelColors(Ed_ErrorHistory->Handle,dwTextColor,dwBkgndColor);
     Ed_ErrorHistory->SelText = AnsiString( szTextLine ) + "\r\n";
     // NOW, after changing the background colour, de-select the text...
     Ed_ErrorHistory->SelLength = 0;

     // ex: m_fShowingCATTraffic = TRUE; // remember "there is CAT traffic in RichEd_Comms now"
   } // end while < read and display entries from the "CAT traffic monitor" >
  return fAppendedNewText;
} // end TKeyerMainForm::DumpRigCtrlTrafficToDebugTab()


//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Timer1Timer(TObject *Sender)
{
  T_ErrorFifoEntry *pErrorFifoEntry;
  DWORD dwColor;  // colour as a simple R-G-B mix - not just another fancy data type
  AnsiString s;
  WORD  wCwChar;  // 'Cw character pattern in Morse code' (not decoded into ASCII yet)
  const char *pszDecoded;
  char  sz255[256], *pszDest, *pszEndstop;
  int   i;
  static int iMeterPollingCountdown   = 0;
  static int iMeterPollingMultiplexer = 0;

  if( m_fInitialising )  // first call ?
   { HERE_I_AM__GUI();
     UpdateTestDisplay();
     if( APPL_fLaunchKeyerOnStart ) // start the keyer IMMEDIATELY (without klicking "Apply") ?
      {  StartKeyerAndShowInfo();   // <- clears m_fRestartKeyerThread when successful
      } // end if( APPL_fLaunchKeyerOnStart )
     m_fInitialising = FALSE;
   }
  else // NOT the first call .. all modules should be INITIALIZED now ...
   { CHECK_SYSTEM_HEALTH();
   }

  if( m_fRestartKeyerThread )  // Some control element in the GUI wants a 'keyer thread restart' ->
   { HERE_I_AM__GUI();
     CwKeyer_Stop();  // stop the keyer (if it was running at all; doesn't hurt if it didn't)
     CwDSP_Stop( &CwKeyer_DSP );    // stop the 'audio DSP'
     CwDSP_Start( &CwKeyer_DSP );   // restart the 'audio DSP'
     CwKeyer_Start(); // start the keyer (~worker thread) with the NEW settings in CwKeyer_Config & Co
     m_fRestartKeyerThread = FALSE; // clear this flag even if CwKeyer_Start() was NOT successful
     CHECK_SYSTEM_HEALTH();
     HERE_I_AM__GUI();
   }

  // If the 'Radio Keying Port' is valid and ALSO used as a 'Radio Control Port',
  //    periodically call RigControl.c from here :
  if( CwKeyer_Config.iRadioKeyingAndControlPort > 0 )
   { if( MyCwNet.RigControl.iParameterPollingState == RIGCTRL_POLLSTATE_PASSIVE )
      { RigCtrl_StartReading( &MyCwNet.RigControl ); // -> e.g send command "Read the transceiver ID" (0x19 0x00), etc^10 ...
      }
     if( MyCwNet.RigControl.iParameterPollingState == RIGCTRL_POLLSTATE_DONE ) // Time to switch from the "Debug" tab to the "TRX" tab ?
      { if(m_fAutomaticTabSwitching) // allow switching from one tabsheet to another (on the main form) ?
         { if( m_iCurrentMainTab == MAIN_TAB_DEBUG )
            { SwitchMainTab( MAIN_TAB_TRX );
            }
         }
      }

     if( (iMeterPollingCountdown--) <= 0 ) // update several "meter"-readings, not only for the "TRX"-tab but also for the SERVER
      { iMeterPollingCountdown = 200/*milliseconds*/ / Timer1->Interval;
        iMeterPollingMultiplexer = (iMeterPollingMultiplexer+1) & 255;
        // Some of the many "meters" in an IC-7300 (or similar) are polled very frequently, others less frequently:
        if( iMeterPollingMultiplexer & 1 ) // every 2nd time the program gets here, poll the S-METER:
         { RigCtrl_SendCmdToReadUnifiedPN( &MyCwNet.RigControl, RIGCTRL_PN_S_METER_LEVEL_DB);
         }
        else // time to poll ONE of the less-frequently-polled "meter" readings:
         { switch( iMeterPollingMultiplexer / 2 )
            { case 0: RigCtrl_SendCmdToReadUnifiedPN( &MyCwNet.RigControl, RIGCTRL_PN_POWER_METER_PERCENT ); break;
              case 1: RigCtrl_SendCmdToReadUnifiedPN( &MyCwNet.RigControl, RIGCTRL_PN_SWR_METER_VALUE     ); break;
              case 2: RigCtrl_SendCmdToReadUnifiedPN( &MyCwNet.RigControl, RIGCTRL_PN_ALC_METER_LEVEL     ); break;
              case 3: RigCtrl_SendCmdToReadUnifiedPN( &MyCwNet.RigControl, RIGCTRL_PN_COMP_METER_LEVEL_DB ); break;
              case 4: RigCtrl_SendCmdToReadUnifiedPN( &MyCwNet.RigControl, RIGCTRL_PN_SUPPLY_VOLTAGE_mV   ); break;
              case 5: RigCtrl_SendCmdToReadUnifiedPN( &MyCwNet.RigControl, RIGCTRL_PN_DRAIN_CURRENT_mA    ); break;
              default: RigCtrl_SendCmdToReadUnifiedPN(&MyCwNet.RigControl, RIGCTRL_PN_POWER_METER_PERCENT );
                  iMeterPollingMultiplexer = 0;
                  break;
            } // end switch( iMeterPollingMultiplexer )
         }
      } // end if( (iMeterPollingCountdown--) <= 0 )

     RigCtrl_Handler( &MyCwNet.RigControl ); // periodic transmissions, timeout monitoring, etc..
   }
  if(g_SpecDispControl.iVFOEditTimer_ms > 0 ) // local operator still EDITING the value in Ed_VFO !
   { g_SpecDispControl.iVFOEditTimer_ms -= Timer1->Interval; // subtract another GUI-timer-calling interval [ms]
     if(g_SpecDispControl.iVFOEditTimer_ms <= 0 )
      { g_SpecDispControl.iVFOEditTimer_ms = 0;  // time has come to EVALUATE the input from Ed_VFO ..
        // (like pressing the ENTER, which almost nobody seems to use anymore)
        Ed_VFO->Color = (TColor)g_SpecDispControl.clWindowBackground;
        ApplyNewVFOFreq();  // Ed_VFO -> MyCwNet.RigControl.dblVfoFrequency
      }
   }


  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // Update the 'Rich Text' edit control for the Error History (and, maybe,
  //             the output from the keyer's built-in Morse decoder) .
  //        Fasten seat belts; it's getting messy with the old Borland VCL !
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  if( ! m_fDebugOutputPaused )
   { BOOL fAppendedNewText = FALSE;
     // Before we start emitting all kinds of 'text lines' to the ancient "RichEdit"-thing:
     // What's the optimum NUMBER OF CHARACTERS PER LINE (w/o horizontal scrolling) ?
     // Fasten seat belts for another joyful ride into of Win32 / GDI programming...
     //
     HDC hdc;          // Windows GDI rules..  some things are impossible with Borland's VCL !
     HGDIOBJ hOldFont;
     SIZE tsize;  // TEXTMETRIC tm;
     int   iWidthInPixel, iWantedCharsPerLine = 80;
     // Width of "Courier New, Size 10" = HOW MANY PIXELS ? ? ?
     hdc = GetDC( Ed_ErrorHistory->Handle );  // This stupid function doesn't do what you EXPECT from it !
      // It doesn't retrieve the "current" DC selected into the TRichEdit-thingy;
      // instead it -kind of- creates(?!) a new DC with some stupid defaults
      // which don't have anything to do with the TRichEdit's currently used font !
      // All this goddamned Win32-low-level-stuff is only necessary because
      // in Borland C++Builder, a TRichEdit doesn't expose its TCanvas which
      // is necessary to 'measure' the width+height of a string in PIXELS .
     hOldFont = SelectObject( hdc, Ed_ErrorHistory->Font->Handle );
     SetMapMode( hdc, MM_TEXT );
     // Only support MM_TEXT here, and forget about the dirty dozen others:
     // > Each logical unit is mapped to one device pixel.
     // > Positive x is to the right; positive y is down.
     // Despite that, GetTextExtentPoint32 returned complete garbage.
     // The garbage was fixed by the "SelectObject"-stuff further above.
     GetTextExtentPoint32( hdc, "ABC4567890", 10/*nStringLength*/, &tsize );
     // > The GetTextExtentPoint32 function computes the width and height
     // >     of the specified string of text.
     // Seen at this point: tsize.cx=100,  tsize.cy=20 .
     iWidthInPixel = tsize.cx;
     SelectObject( hdc, hOldFont );
     ReleaseDC( Ed_ErrorHistory->Handle, hdc);
     // Don't assume this obfuscated sequence of Windows GDI calls never fails !
     // Only use the "measured width for ten characters in pixels" when plausible:
     if( iWidthInPixel >= 60 )
      {  iWantedCharsPerLine = ((Ed_ErrorHistory->Width-30/*pixels for the scrollbar*/) * 10/*chars*/) / iWidthInPixel;
      }

     // Next, try to avoid the old problem with VCL applications crashing
     //            if the text in an editor gets too long:
     while( Ed_ErrorHistory->Lines->Count > 500 ) // 500 lines is an INCREDIBLE amount of text :o(
      { HERE_I_AM__GUI();
        Ed_ErrorHistory->Lines->Delete(0);  // delete the 1st line (a "TAnsiString")
        fAppendedNewText = TRUE;
      }
     fAppendedNewText |= DumpErrorHistoryToDebugTab();
     fAppendedNewText |= DumpRigCtrlTrafficToDebugTab(iWantedCharsPerLine);

     // Similar as for the thread-safe, non-VCL 'Error History', also dump
     // the CW-keyer's "decoded output" (text) into the same RichEdit control:
     while( (wCwChar=CwKeyer_ReadFromDecoderFifo(&CwKeyer_DecoderFifo, &m_iDecoderFifoTail)) != 0)
      {
        pszDecoded = Elbug_MorseCodePatternToASCII(wCwChar);
        if( MyCwNet.cfg.iDiagnosticFlags & CWNET_DIAG_FLAGS_SHOW_DECODER_OUTPUT )
         { // Again this obfuscated way to append text to a RichEdit control
           //       with a custom background colour :
           Ed_ErrorHistory->SelStart = Ed_ErrorHistory->Text.Length();
           Ed_ErrorHistory->SelLength= 1;
           RichEdit_SetSelColors( Ed_ErrorHistory->Handle,
               g_clDecoderOutputForeground, g_clDecoderOutputBackground );
           Ed_ErrorHistory->SelText = AnsiString( pszDecoded );
           Ed_ErrorHistory->SelLength = 0;
           // ex: Ed_ErrorHistory->SelStart = Ed_ErrorHistory->Text.Length();
           fAppendedNewText = TRUE;
         } // end if < .. CWNET_DIAG_FLAGS_SHOW_DECODER_OUTPUT ? >
        // ToDo: Record the decoded 'Morse traffic' in a logfile ?
      } // end while < more decoded characters or Morse code 'prosigns' >

     // Only for development, optionally dump the CW-NET 'Morse keying bytestream'
     //  (received via network, not locally from the keyer) on the 'Debug' tab, too:
     if( MyCwNet.cfg.iDiagnosticFlags & CWNET_DIAG_FLAGS_SHOW_KEYING_BYTESTREAM )
      { pszDest    = sz255;
        pszEndstop = sz255 + 250;
        if( MyCwNet.cfg.iFunctionality == CWNET_FUNC_CLIENT )
         { while( m_iMorseFIFOTailIndex2 != MyCwNet.MorseTxFifo.iHeadIndex )
            { SL_AppendHex( &pszDest, pszEndstop,
                      MyCwNet.MorseTxFifo.elem[m_iMorseFIFOTailIndex2++].bCmd,
                      2/* hex digits */ );
              SL_AppendChar( &pszDest, pszEndstop, ' ' );
              m_iMorseFIFOTailIndex2 &= (CW_KEYING_FIFO_SIZE-1);
            }
         }
        else if( MyCwNet.cfg.iFunctionality == CWNET_FUNC_SERVER )
         { while( m_iMorseFIFOTailIndex2 != MyCwNet.MorseRxFifo.iHeadIndex )
            { SL_AppendHex( &pszDest, pszEndstop,
                      MyCwNet.MorseRxFifo.elem[m_iMorseFIFOTailIndex2++].bCmd,
                      2/* hex digits */ );
              SL_AppendChar( &pszDest, pszEndstop, ' ' );
              m_iMorseFIFOTailIndex2 &= (CW_KEYING_FIFO_SIZE-1);
            }
         }
        if( pszDest > sz255 ) // anything to append (as "coloured hex dump") ?
         { // And again, the obscure way to append text to a RichEdit control
           //     with a custom background colour (half Borland VCL, half Win32 API)
           Ed_ErrorHistory->SelStart  = Ed_ErrorHistory->Text.Length();
           Ed_ErrorHistory->SelLength = 1;
           RichEdit_SetSelColors(Ed_ErrorHistory->Handle,
                    0x000000/*dwTextColor*/, 0xC0C0FF/*dwBkgndColor:light red*/);
           Ed_ErrorHistory->SelText   = AnsiString( sz255 );
           Ed_ErrorHistory->SelLength = 0;
           // ex: Ed_ErrorHistory->SelStart  = Ed_ErrorHistory->Text.Length();
           fAppendedNewText = TRUE;
         }
      } // end if( MyCwNet.iDiagnosticFlags & CWNET_DIAG_FLAGS_SHOW_KEYING_BYTESTREAM )

     // After deleting lines AT THE TOP (to limit the number of lines in the editor),
     // the RichEditor's VERTICAL SCROLLBAR ran havoc:
     // During the manipulation of the text in the editor the scrollbar
     // flipped around like crazy, and after the update, despite the
     // "text caret" / "text curor" / "TRichEdit.SelStart" being AT THE END
     // of the text, the goddamned vertical scrollbar hopped back to the
     // BEGIN of the text - and in between, it caused an awful flicker.
     // Tried to fix this similar as in Spectrum Lab's "debug form" :
     // To make the new caret position visible, ...
     // > The EM_SCROLLCARET message scrolls the caret into view in an edit control.
     // > You can send this message to either an edit control or a rich edit control.
     if( fAppendedNewText )
      { Ed_ErrorHistory->SelLength = 0;
        Ed_ErrorHistory->SelStart  = Ed_ErrorHistory->Text.Length();
        SendMessage( Ed_ErrorHistory->Handle, // handle of destination window
                      EM_SCROLLCARET, // message to send : "scroll into view"
                (WPARAM)0,(LPARAM)0); // both params must be zero
      }

     HERE_I_AM__GUI();
   }   // end if( ! m_fDebugOutputPaused )



  // Update various indicators for the digital in- and outputs
  //        on the 'keyer'- and 'radio' ports.
  //  Note: CwKeyer_GetDigitalInput() can also 'read back' the states
  //        of DIGITAL OUTPUTS (from the application's point of view).
  if( CwKeyer_iTestMode == KEYER_TEST_MODE_OFF )
   { Chk_MorseKeyDTR->Checked = CwKeyer_GetDigitalInput( KEYER_SIGNAL_INDEX_MORSE_KEY_DTR );
     Chk_MorseKeyRTS->Checked = CwKeyer_GetDigitalInput( KEYER_SIGNAL_INDEX_MORSE_KEY_RTS );
   }
  ST_MorseKeyDCD->Color = CwKeyer_GetDigitalInput( KEYER_SIGNAL_INDEX_MORSE_KEY_DCD ) ? clRed : clBtnFace;
  ST_MorseKeyDSR->Color = CwKeyer_GetDigitalInput( KEYER_SIGNAL_INDEX_MORSE_KEY_DSR ) ? clRed : clBtnFace;
  ST_MorseKeyCTS->Color = CwKeyer_GetDigitalInput( KEYER_SIGNAL_INDEX_MORSE_KEY_CTS ) ? clRed : clBtnFace;
  ST_MorseKeyRI->Color  = CwKeyer_GetDigitalInput( KEYER_SIGNAL_INDEX_MORSE_KEY_RI  ) ? clRed : clBtnFace;
  if( CwKeyer_iTestMode == KEYER_TEST_MODE_OFF )
   { Chk_RadioKeyingDTR->Checked = CwKeyer_GetDigitalInput( KEYER_SIGNAL_INDEX_RADIO_KEYING_DTR );
     Chk_RadioKeyingRTS->Checked = CwKeyer_GetDigitalInput( KEYER_SIGNAL_INDEX_RADIO_KEYING_RTS );
   }
  ST_RadioKeyingDCD->Color = CwKeyer_GetDigitalInput( KEYER_SIGNAL_INDEX_RADIO_KEYING_DCD ) ? clRed : clBtnFace;
  ST_RadioKeyingDSR->Color = CwKeyer_GetDigitalInput( KEYER_SIGNAL_INDEX_RADIO_KEYING_DSR ) ? clRed : clBtnFace;
  ST_RadioKeyingCTS->Color = CwKeyer_GetDigitalInput( KEYER_SIGNAL_INDEX_RADIO_KEYING_CTS ) ? clRed : clBtnFace;
  ST_RadioKeyingRI->Color  = CwKeyer_GetDigitalInput( KEYER_SIGNAL_INDEX_RADIO_KEYING_RI  ) ? clRed : clBtnFace;

  // The time-consuming periodic update of the CW-'Timing Scope'
  //  only happens when the scope is currently visible (on two of the tabs):
  if(m_iCurrentMainTab==MAIN_TAB_TEST ) // ex: if (PageControl1->ActivePage == TS_Test )
   { // Currently on the "Test" tab with the 'CW timing' scope display:
     // Because updating the 'CW timing scope' display consumed 5 % of the
     // available CPU time (while the 'keyer worker thread' consumed less than 0.2 %),
     // only redraw the display is the SAMPLE DATA have been modified.
     // As long as there is no TRIGGER for the scope, samples remain unmodified,
     // and the timing scope is NOT redrawn.
     if( CwKeyer_TimingScope.iUpdateCount != TimingScope_iUpdateCountOnTestTab )
      { HERE_I_AM__GUI();
        TimingScope_iUpdateCountOnTestTab = CwKeyer_TimingScope.iUpdateCount;
        Img_TimingScopeOnTestTab->Picture->Bitmap->Height = Img_TimingScopeOnTestTab->Height;
        Img_TimingScopeOnTestTab->Picture->Bitmap->Width  = Img_TimingScopeOnTestTab->Width;
        UpdateTimingScope( &CwKeyer_TimingScope, Img_TimingScopeOnTestTab->Picture->Bitmap );
      }
   }
  if( PageControl1->ActivePage == TS_KeyerSettings )
   { // Currently on the "Keyer Settings" tab, where we'd also like to see the TIMING:
     if( CwKeyer_TimingScope.iUpdateCount != TimingScope_iUpdateCountOnKeyerTab )
      { HERE_I_AM__GUI();
        TimingScope_iUpdateCountOnKeyerTab = CwKeyer_TimingScope.iUpdateCount;
        Img_TimingScopeOnKeyerTab->Picture->Bitmap->Height = Img_TimingScopeOnKeyerTab->Height;
        Img_TimingScopeOnKeyerTab->Picture->Bitmap->Width  = Img_TimingScopeOnKeyerTab->Width;
        UpdateTimingScope( &CwKeyer_TimingScope, Img_TimingScopeOnKeyerTab->Picture->Bitmap);
      }
   }
  if( PageControl1->ActivePage == TS_Network )
   { HERE_I_AM__GUI();
     Ed_NetworkStatus->Text = CwNet_GetCurrentStatusAsString( &MyCwNet );
     i = MyCwNet.Client[CWNET_LOCAL_CLIENT_INDEX].iAnnouncedTxClient;
     if(m_iTransmittingClient != i) // show who's "on the key" at the moment
      { m_iTransmittingClient = i;
        if( MyCwNet.cfg.iFunctionality == CWNET_FUNC_SERVER )
         { Ed_OnTheKeyNow->Text = CwNet_GetClientCallOrInfo( &MyCwNet, m_iTransmittingClient );
           // The above only works on the SERVER SIDE. None of the CLIENTS
           // has a list of callsigns (or whatever) currently logged into the server...
         }
        else
         { // ... so instead, for any kind of 'announced info', use this string buffer:
           Ed_OnTheKeyNow->Text = MyCwNet.Client[CWNET_LOCAL_CLIENT_INDEX].sz80TxInfo;
         }
      }
   } // end if( PageControl1->ActivePage == TS_Network )
  if( (PageControl1->ActivePage == TS_TRX )
    ||((CwKeyer_Config.iTRXOptions & KEYER_TRX_OPT_KEEP_RUNNING) != 0) )
   { // Currently on the "Transceiver" tab (TRX), with spectrum / spectrogram;
     // or the 'TRX tab' (with spectrum/spectrogram) shall be updated even when NOT visible...

     if( m_fAdaptImageSizesOnTRXTab ) // call AdaptImageSizesOnTRXTab() .. now that the tab is visible ?
      { AdaptImageSizesOnTRXTab(); // -> Img_Spectrum, Img_Waterfall, Img_FreqScale (and their BITMAPS)
        g_SpecDispControl.fUpdateSpectrum = g_SpecDispControl.fUpdateWaterfall = g_SpecDispControl.fUpdateFreqScale = TRUE;
      }
     UpdateTrxDisplay(); // e.g. VFO frequency and "op mode" (modulation type) ... IF MODIFIED

     if( g_SpecDispControl.fUpdateWaterfall || g_SpecDispControl.fClearWaterfall
      || (MyCwNet.RigControl.iScopeFifoHeadIndex != m_iLastScopeDisplayHeadIndex) )
      { int nNewSpectra = MyCwNet.RigControl.iScopeFifoHeadIndex - m_iLastScopeDisplayHeadIndex;
        if( nNewSpectra < 0 )
         {  nNewSpectra += RIGCTRL_SPECTRUM_FIFO_SIZE;
            // ,-----------|________________________|
            // '--> cannot plot more than
            //    T_RigCtrl_Spectrum SpectrumFifo[ RIGCTRL_SPECTRUM_FIFO_SIZE ]
            //    permits !
         }
        if( nNewSpectra > RIGCTRL_SPECTRUM_FIFO_SIZE )
         {  nNewSpectra = RIGCTRL_SPECTRUM_FIFO_SIZE;
         }
        HERE_I_AM__GUI();  // fasten seat belt .. NEW spectra waiting to be drawn !
        SpecDisp_UpdateWaterfall( Img_Waterfall->Picture->Bitmap, // [out] TBitmap a la VCL
           &g_SpecDispControl,   // [in] layout, etc
           &MyCwNet.RigControl,  // [in] RigControl instance with spectrum FIFO
           m_iLastScopeDisplayHeadIndex,
           nNewSpectra );
        m_iLastScopeDisplayHeadIndex = MyCwNet.RigControl.iScopeFifoHeadIndex;
        // If there were new data to redraw the waterfall,
        // also redraw the most recent SPECTRUM (as a curve):
        g_SpecDispControl.fUpdateSpectrum = TRUE;
      }
     if( g_SpecDispControl.fUpdateSpectrum )
      { SpecDisp_UpdateSpectrum( Img_Spectrum->Picture->Bitmap,
           &g_SpecDispControl,   // [in] layout, etc
           &MyCwNet.RigControl); // [in] RigControl instance with spectrum FIFO
      }
     if(( g_SpecDispControl.fmin_from_RigCtrl != g_SpecDispControl.fmin_on_FreqScale )
      ||( g_SpecDispControl.fmax_from_RigCtrl != g_SpecDispControl.fmax_on_FreqScale ) )
      { g_SpecDispControl.fUpdateFreqScale = TRUE;
        g_SpecDispControl.fmin_on_FreqScale = g_SpecDispControl.fmin_from_RigCtrl; // avoid unnecessary updates
        g_SpecDispControl.fmax_on_FreqScale = g_SpecDispControl.fmax_from_RigCtrl;
        // Note: With only 475 frequency bins from e.g. an IC-7300,
        //       it's pointless to ZOOM INTO the spectrum here locally.
        // Thus, the DISPLAYED frequency range is the same as
        //       what RigControl.c has delivered from the radio.
        // ( fmin_from_RigCtrl, fmax_from_RigCtrl updated when updating either
        //   the SPECTRUM, WATERFALL, or the FREQUENCY SCALE ).
        // This is utterly different than the original code from Spectrum Lab,
        // where FFTs with over a million frequency "bins" was calculated LOCALLY.
        // Short: The Remote CW Keyer shows the same spectrum frequency range
        //        as the radio, and "scrolls along" in what Icom calls 'CENTER' mode,
        //        or "hops from range to range" in 'SCROLL-C' mode.
      }
     if( g_SpecDispControl.fUpdateFreqScale )
      { SpecDisp_UpdateFreqScale( Img_FreqScale->Picture->Bitmap, // [out] TBitmap a la VCL
           &g_SpecDispControl,   // [in] layout, etc
           &MyCwNet.RigControl); // [in] RigControl instance with VFO frequency, displayable frequency range, etc
      }
     if( MyCwNet.RigControl.iSMeterLevel_dB != m_dblDisplayedSMeterLevel_dB )
      { m_dblDisplayedSMeterLevel_dB = MyCwNet.RigControl.iSMeterLevel_dB;
        UpdateSMeter( Img_SMeter, m_dblDisplayedSMeterLevel_dB );
      }
     // The graphics in the BOTTOM panel on the "TRX" tab can be selected
     // in an extra combo box. First implemented: A "multi-function meter",
     // inspired by ..you guessed it.. an IC-7300:
     switch( CB_TRX_Bottom->ItemIndex )
      { case CB_TRX_BOTTOM_SHOW_MULTI_FUNCTION_METER :
           UpdateMultiFunctionMeters();
           break;
        default:
           break;
      } // end switch( CB_TRX_Bottom->ItemIndex )

   } // end if < update elements on the "TRX" tab > ?

  HERE_I_AM__GUI();
  UpdateStatusIndicator(); // .. in an 'abused' item of the MAIN MENU ..
  HERE_I_AM__GUI();


} // end TKeyerMainForm::Timer1Timer()
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::AppMessage(tagMSG &Msg, bool &Handled)
  // Note: This handler may be called HUNDREDS OF TIMES PER SECOND !
  //       There can only be ONE of these handlers, because it's registered
  //       in Borland's 'VCL' ecosystem by setting Application->OnMessage = AppMessage .
{
  char buffer[81];
  WORD key = 0;

  /* See help on TApplication::OnMessage.
   *  Description of tagMSG cannot be found in the CBuilder help system,
   *             but in WINUSER.H:
   *
   * typedef struct tagMSG {
   *    HWND        hwnd;        // a window handle, dont care about this.
   *    UINT        message;     // a WM_xyz constant (message type)
   *    WPARAM      wParam;      // data depending on the message type
   *    LPARAM      lParam;
   *    DWORD       time;
   *    POINT       pt;
   * } MSG, *PMSG, NEAR *NPMSG, FAR *LPMSG;
   *
   *  Information on where the keyboard codes are located in this message
   *  can be found in the help on WM_KEYDOWN
   *  Taken from Microsofts Win32 Programmers Reference:
   *   Parameters of WM_KEYDOWN:
   *     nVirtKey
   *       Value of wParam. Specifies the virtual-key code of the nonsystem key.
   *
   *     lKeyData
   *       Value of lParam. Specifies the repeat count, scan code,
   *       extended-key flag, context code, previous key-state flag,
   *       and transition-state flag, as shown in the following table:
   *       ("Value" used here seems to be a "Bit Number" [WB] )..
   *
   *      Value Description
   *       0-15  Specifies the repeat count. The value is the number of times
   *             the keystroke is repeated as a result of the user holding down
   *             the key.
   *       16-23 Specifies the scan code. The value depends on the original
   *             equipment manufacturer (OEM).
   *       24    Specifies whether the key is an extended key, such as the
   *             right-hand ALT and CTRL keys that appear on an enhanced
   *             101- or 102-key keyboard. The value is 1 if it is an extended
   *             key; otherwise, it is 0.
   *       25-28 Reserved; do not use.
   *       29    Specifies the context code. The value is always 0 for a
   *             WM_KEYDOWN message.
   *       30    Specifies the previous key state. The value is 1 if the key
   *             is down before the message is sent, or it is 0 if the key is up.
   *       31    Specifies the transition state. The value is always 0 for a
   *             WM_KEYDOWN message.
   *  Return Values
   *       An application should return zero if it processes this message.
   *       "Return something" .. how ? Borlands's VCL / "AppMessage()"
   *       doesn't return anything ('void') result, but we're supposed
   *       to set the boolean 'Handled' to TRUE if we processed the message.
   */
  switch(Msg.message)
   { case WM_KEYDOWN :
        if(Active)   /* If Active is true , this form has the focus */
         {
           key = (WORD)Msg.wParam;
           switch( key )
            { case VK_ESCAPE : // ESCAPE (#27) : Immediately stop transmitting ("panic key")
                 PauseTransmission();
                 Handled = TRUE;
                 break;
              case VK_F1 : // function key 'F1' (#112) : start sending from memory #1
              case VK_F2 :
              case VK_F3 :
              case VK_F4 :
              case VK_F5 :
              case VK_F6 :
                 StartPlaying( key-VK_F1 );
                 Handled = TRUE;
                 break;
              case VK_F12 :
                 CwGen_StopReplay( &CwKeyer_Gen );
                 Handled = TRUE;
                 break;
              default:  // none of the "special" keys above..
                 break;
            } // end switch( key )
         } /* end if(Active) */
        else
         { // What they call 'Active' here is what they call 'Focused' somewhere else...
         } // end if < not "Active" aka not "Focused" >
      break; // end case WM_KEYDOWN

     default:
        // for all other messages, Handled remains False
        // so that other message handlers can respond !
        break;
   }
} // end TKeyerMainForm::AppMessage()


//---------------------------------------------------------------------------
void TKeyerMainForm::UpdateTestDisplay(void)
{
  if( CwKeyer_iTestMode == KEYER_TEST_MODE_MANUAL )
   { Chk_ManualControl->Checked  = TRUE;
     Chk_MorseKeyDTR->Enabled    = TRUE;
     Chk_MorseKeyRTS->Enabled    = TRUE;
     Chk_RadioKeyingDTR->Enabled = TRUE;
     Chk_RadioKeyingRTS->Enabled = TRUE;
   }
  else // *NO* manual control of the digital outputs (DTR, RTS) :
   { Chk_MorseKeyDTR->Enabled    = FALSE;
     Chk_MorseKeyRTS->Enabled    = FALSE;
     Chk_RadioKeyingDTR->Enabled = FALSE;
     Chk_RadioKeyingRTS->Enabled = FALSE;
   }

} // end TKeyerMainForm::UpdateTestDisplay()

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::ApplyManualTestSignals(TObject *Sender)
  // Called as "OnClick" method (handler) from Chk_MorseKeyDTR,  Chk_MorseKeyRTS,
  //                                      Chk_RadioKeyingDTR, Chk_RadioKeyingRTS.
{
  if( Chk_ManualControl->Checked  )
   {  CwKeyer_iTestMode = KEYER_TEST_MODE_MANUAL;
      CwKeyer_SetDigitalOutput( KEYER_SIGNAL_INDEX_MORSE_KEY_DTR, Chk_MorseKeyDTR->Checked );
      // Why does the IDE (BCB V6) crash when hitting a breakpoint HERE ?
      CwKeyer_SetDigitalOutput( KEYER_SIGNAL_INDEX_MORSE_KEY_RTS, Chk_MorseKeyRTS->Checked );
      CwKeyer_SetDigitalOutput( KEYER_SIGNAL_INDEX_RADIO_KEYING_DTR, Chk_RadioKeyingDTR->Checked );
      CwKeyer_SetDigitalOutput( KEYER_SIGNAL_INDEX_RADIO_KEYING_RTS, Chk_RadioKeyingRTS->Checked );
      // Note: The actual switching of the above 'digital outputs'
      //       happens in the "keyer thread", using some funny-named Serial Port
      //       API functions that we don't want to invoke from here.
   }
  else
   {  CwKeyer_iTestMode = KEYER_TEST_MODE_OFF;
      // Do *not* call CwKeyer_SetDigitalOutput() in this mode !
   }
  UpdateTestDisplay();
}

//---------------------------------------------------------------------------
void TKeyerMainForm::UpdateItemsInSettingsMenu(void)
{
  int i;
  AnsiString s;
  MI_Ch4_PTT->Checked = ( CwKeyer_TimingScope.cfg.iChannel4Source == TIMING_SCOPE_CHANNEL_SOURCE_PTT_OUTPUT);
  MI_Ch4_TestInput->Checked = (CwKeyer_TimingScope.cfg.iChannel4Source == TIMING_SCOPE_CHANNEL_SOURCE_TEST_INPUT);
  MI_Ch4_NetworkTrouble->Checked = (CwKeyer_TimingScope.cfg.iChannel4Source==TIMING_SCOPE_CHANNEL_SOURCE_NETWORK_TROUBLE);
  MI_Ch4_KeyingFifoUsage->Checked= (CwKeyer_TimingScope.cfg.iChannel4Source==TIMING_SCOPE_CHANNEL_SOURCE_KEYING_FIFO_USAGE);
  MI_Ch4_ElbugKeyerFlags->Checked= (CwKeyer_TimingScope.cfg.iChannel4Source==TIMING_SCOPE_CHANNEL_SOURCE_ELBUG_KEYER_FLAGS);

  MI_TScopeShowAudioIn->Checked  = (CwKeyer_TimingScope.cfg.iVisibleChannels & (1<<TIMING_SCOPE_CHANNEL_AUDIO_INPUT) ) != 0;
  MI_TScopeShowAudioOut->Checked = (CwKeyer_TimingScope.cfg.iVisibleChannels & (1<<TIMING_SCOPE_CHANNEL_AUDIO_OUTPUT) ) != 0;

  MI_ColourScheme_Default->Checked = ( g_SpecDispControl.iColourScheme == COLOUR_SCHEME_DEFAULT);
  MI_ColourScheme_Dark->Checked    = ( g_SpecDispControl.iColourScheme == COLOUR_SCHEME_DARK);
  MI_Language_English->Checked     = ( APPL_iLanguage==TRANSLATOR_LANGUAGE_ENGLISH );
  MI_Language_German->Checked      = ( APPL_iLanguage==TRANSLATOR_LANGUAGE_GERMAN  );
  MI_Language_French->Checked      = ( APPL_iLanguage==TRANSLATOR_LANGUAGE_FRENCH  );
  MI_Language_Dutch->Checked       = ( APPL_iLanguage==TRANSLATOR_LANGUAGE_DUTCH   );
  MI_Language_Italian->Checked     = ( APPL_iLanguage==TRANSLATOR_LANGUAGE_ITALIAN );
  switch( g_iUserInterfaceMode )
   { case UI_MODE_SIMPLE :
        MI_Settings_UserInterface->Caption = AnsiString(TE("&User Interface: Simple"));
        MI_UI_Simple->Checked   = TRUE;
        MI_UI_Complete->Checked = FALSE;
        break;
     case UI_MODE_COMPLETE:
        MI_Settings_UserInterface->Caption = AnsiString(TE("&User Interface: Complete"));
        MI_UI_Simple->Checked   = FALSE;
        MI_UI_Complete->Checked = TRUE;
        break;
     default: // ?!
        break;
   }
  MI_ITU_Region->Visible = (g_iUserInterfaceMode==UI_MODE_COMPLETE);
  MI_ITU_Region_1->Checked = ( RigCtrl_ITU_Region == 1 );
  MI_ITU_Region_2->Checked = ( RigCtrl_ITU_Region == 2 );
  MI_ITU_Region_3->Checked = ( RigCtrl_ITU_Region == 3 );

  // Hide almost everything in the "Settings" menu that may distract an unexperienced user:
  MI_ConnectScopeCh4->Visible    = (g_iUserInterfaceMode==UI_MODE_COMPLETE);
  MI_ShowAudioWaveforms->Visible = (g_iUserInterfaceMode==UI_MODE_COMPLETE);
  MI_MoreKeyerSettings->Visible  = (g_iUserInterfaceMode==UI_MODE_COMPLETE);
  MI_Settings_SpectrumDisplay->Visible = (g_iUserInterfaceMode==UI_MODE_COMPLETE);
  MI_SeparatorBelowSpectrumDisplaySettings->Visible = (g_iUserInterfaceMode==UI_MODE_COMPLETE);
  MI_ClearScopeOverlays->Visible = (g_iUserInterfaceMode==UI_MODE_COMPLETE);
  MI_ClearDebugAndDecoderOutput->Visible = (g_iUserInterfaceMode==UI_MODE_COMPLETE);
  MI_Settings_ShowMorseDecoderOutput->Visible = (g_iUserInterfaceMode==UI_MODE_COMPLETE);
  MI_ReportTestResults->Visible  = (g_iUserInterfaceMode==UI_MODE_COMPLETE);
  MI_ReportRigCtrlParams->Visible = (g_iUserInterfaceMode==UI_MODE_COMPLETE);
  MI_ShowStraightKeyDecoderHistory->Visible = (g_iUserInterfaceMode==UI_MODE_COMPLETE);
  MI_PauseDebugOutput->Visible   = (g_iUserInterfaceMode==UI_MODE_COMPLETE);
  MI_SeparatorBelowPauseDebugOutput->Visible = (g_iUserInterfaceMode==UI_MODE_COMPLETE);
  MI_NetworkServerOptions->Visible = (g_iUserInterfaceMode==UI_MODE_COMPLETE);
  MI_NetworkLatency->Visible     = (g_iUserInterfaceMode==UI_MODE_COMPLETE);
  MI_NetworkDiagnostics->Visible = (g_iUserInterfaceMode==UI_MODE_COMPLETE);
  MI_Settings_RigControlOptions->Visible = (g_iUserInterfaceMode==UI_MODE_COMPLETE);
  MI_HamlibServerOptions->Visible  = (g_iUserInterfaceMode==UI_MODE_COMPLETE);
  MI_Settings_TRX_Options->Visible = (g_iUserInterfaceMode==UI_MODE_COMPLETE);
    // '--> that was the last item in the "Settings" menu; only visible in the "COMPLETE UI"
    // SUB MENU ITEMS (below) don't need to be hidden when their parents are already invisible.


  MI_PauseDebugOutput->Checked     = m_fDebugOutputPaused;
  MI_Server_EnableHTTP->Checked    = (MyCwNet.cfg.iHttpServerOptions & HTTP_SERVER_OPTIONS_ENABLE) != 0;
  MI_Server_RestrictAccess->Checked= (MyCwNet.cfg.iHttpServerOptions & HTTP_SERVER_OPTIONS_RESTRICT) != 0;
  MI_Server_AddTestTone->Checked   = (CwKeyer_DSP.cfg.iAudioFlags & DSP_AUDIO_FLAGS_SEND_NETWORK_TEST_TONE) != 0;

  s = AnsiString(TE("Network &Latency")) + " : ";
  if( MyCwNet.cfg.iNetworkLatency_ms <= 0 )
   { i = CwNet_GetIndexOfCurrentlyActiveClient( &MyCwNet );
     s = s + AnsiString(TE("Auto-detect, currently measured: ") )
           + IntToStr( CwNet_GetLatencyForRemoteClient_ms(&MyCwNet, i) )
           + " ms";
   }
  else
   { s = s + IntToStr(MyCwNet.cfg.iNetworkLatency_ms) + " ms";
   }
  MI_NetworkLatency->Caption = s;
  MI_Network_DiagnosticMode->Checked = (MyCwNet.cfg.iDiagnosticFlags & CWNET_DIAG_FLAGS_VERBOSE) != 0;
  MI_Network_SimulateBadConnection->Checked = (MyCwNet.cfg.iDiagnosticFlags & CWNET_DIAG_FLAGS_SIMULATE_BAD_CONN) != 0;
  MI_Network_ShowConnectionLog->Checked = (MyCwNet.cfg.iDiagnosticFlags & CWNET_DIAG_FLAGS_SHOW_CONN_LOG) != 0;
  MI_Network_ShowNetworkTraffic->Checked= (MyCwNet.cfg.iDiagnosticFlags & CWNET_DIAG_FLAGS_SHOW_NETWORK_TRAFFIC) != 0;
  MI_Network_ShowRigControlTraffic->Checked=(RigCtrl_iTrafficMonitorDisplayOptions & RIGCTRL_TRAFFIC_DISPLAY_OPTION_ENABLE) != 0;

  MI_Network_ShowRcvdKeyingBytestream->Checked = (MyCwNet.cfg.iDiagnosticFlags & CWNET_DIAG_FLAGS_SHOW_KEYING_BYTESTREAM) != 0;
  MI_Settings_ShowMorseDecoderOutput->Checked  = (MyCwNet.cfg.iDiagnosticFlags & CWNET_DIAG_FLAGS_SHOW_DECODER_OUTPUT) != 0;
  MI_KeyViaShiftAndControl->Checked = CwKeyer_Config.fKeyViaShiftAndControl;
  MI_KeyInputWatchdog->Checked      = CwKeyer_Config.fKeyInputWatchdog;
  MI_DebouncePaddleInputs->Checked  = CwKeyer_Config.fDebouncePaddleInputs;

  MI_RigCtrl_RejectEchosInLog->Checked = (MyCwNet.RigControl.dwRejectMessages4Log & (1<<RIGCTRL_MSGTYPE_ECHO)) != 0;
  MI_RigCtrl_RejectPeriodicCmds->Checked = (MyCwNet.RigControl.dwRejectMessages4Log & RIGCTRL_MSGTYPE_FLAG_PERIODIC_POLL) != 0;
  MI_RigCtrl_RejectSpectrumInLog->Checked = (MyCwNet.RigControl.dwRejectMessages4Log & (1<<RIGCTRL_MSGTYPE_SPECTRUM)) != 0;
  MI_RigCtrl_RejectFrequencyReports->Checked = (MyCwNet.RigControl.dwRejectMessages4Log & (1<<RIGCTRL_MSGTYPE_FREQUENCY_REPORT)) != 0;

  // Above: Menu items for the important 'Rig Control' .
  // Below: Menu items for the optional 'Hamlib Net rigctld'-compatible SERVER:
  s = AnsiString(TE("Built-in &Hamlib 'Net' Server Options")) + " (";
  if( CwKeyer_Config.fEnableBuiltInHamlibServer )
   { s = s + AnsiString(TE("enabled")) + ", port " + IntToStr(MyHamlibServer.cfg.iServerListeningPort);
   }
  else
   { s = s + AnsiString(TE("disabled"));
   }
  MI_HamlibServerOptions->Caption = s + ")";
  MI_HLServer_Enable->Checked = CwKeyer_Config.fEnableBuiltInHamlibServer;
  MI_HLServer_ShowTraffic->Checked = (MyHamlibServer.cfg.iDiagnosticFlags & HLSRV_DIAG_FLAGS_SHOW_NETWORK_TRAFFIC) != 0;
  MI_HLServer_VerboseOutput->Checked = (MyHamlibServer.cfg.iDiagnosticFlags & HLSRV_DIAG_FLAGS_VERBOSE_OUTPUT) != 0;
} // end TKeyerMainForm::UpdateItemsInSettingsMenu()

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Menu_SettingsClick(TObject *Sender)
{ // Called from the VCL when opening the "Settings" menu.
  UpdateItemsInSettingsMenu();
}

void __fastcall TKeyerMainForm::MI_HelpClick(TObject *Sender)
{ // Called from the VCL when opening the "Help" menu.
  // Similar as in the "Settings" menu, certain items are hidden
  // when the "User Interface" is set to "Simple" instead of "Complete":
  // MI_HelpIndex->Visible = (g_iUserInterfaceMode==UI_MODE_COMPLETE); // oh c'mon, keep this visible, even in the "Simple" UI ...
  MI_Help_MainMenu->Visible = (g_iUserInterfaceMode==UI_MODE_COMPLETE);
  // Show help about the "TRX" tab either in the "Complete UI", or when at least remote VFO control or remote spectrum display are possible:
  MI_Help_TransceiverControl->Visible = (g_iUserInterfaceMode==UI_MODE_COMPLETE) || (TS_TRX->Visible);
  MI_Help_Hardware->Visible = (g_iUserInterfaceMode==UI_MODE_COMPLETE);
  MI_Help_DebugTab->Visible = (g_iUserInterfaceMode==UI_MODE_COMPLETE) || (TS_Debug->Visible);

} // end TKeyerMainForm::MI_HelpClick()
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Ch4_PTTClick(TObject *Sender)
{ CwKeyer_TimingScope.cfg.iChannel4Source = TIMING_SCOPE_CHANNEL_SOURCE_PTT_OUTPUT;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Ch4_TestInputClick(TObject *Sender)
{ CwKeyer_TimingScope.cfg.iChannel4Source = TIMING_SCOPE_CHANNEL_SOURCE_TEST_INPUT;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Ch4_NetworkTroubleClick(TObject *Sender)
{ CwKeyer_TimingScope.cfg.iChannel4Source = TIMING_SCOPE_CHANNEL_SOURCE_NETWORK_TROUBLE;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Ch4_KeyingFifoUsageClick(TObject *Sender)
{ CwKeyer_TimingScope.cfg.iChannel4Source = TIMING_SCOPE_CHANNEL_SOURCE_KEYING_FIFO_USAGE;
}
//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Ch4_ElbugKeyerFlagsClick(TObject *Sender)
{ CwKeyer_TimingScope.cfg.iChannel4Source = TIMING_SCOPE_CHANNEL_SOURCE_ELBUG_KEYER_FLAGS;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_TScopeShowAudioInClick(TObject *Sender)
{ CwKeyer_TimingScope.cfg.iVisibleChannels ^= (1 << TIMING_SCOPE_CHANNEL_AUDIO_INPUT);
  TimingScope_iUpdateCountOnTestTab = TimingScope_iUpdateCountOnKeyerTab = -1; // redraw a.s.a.p.
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_TScopeShowAudioOutClick(TObject *Sender)
{ CwKeyer_TimingScope.cfg.iVisibleChannels ^= (1 << TIMING_SCOPE_CHANNEL_AUDIO_OUTPUT);
  TimingScope_iUpdateCountOnTestTab = TimingScope_iUpdateCountOnKeyerTab = -1; // redraw a.s.a.p.
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_KeyInputWatchdogClick(TObject *Sender)
{ CwKeyer_Config.fKeyInputWatchdog = !CwKeyer_Config.fKeyInputWatchdog;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_DebouncePaddleInputsClick(TObject *Sender)
{ CwKeyer_Config.fDebouncePaddleInputs = !CwKeyer_Config.fDebouncePaddleInputs;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Hotkey_EscapeClick(TObject *Sender)
{ PauseTransmission();  // <- same as when pressing the REAL "ESCAPE" key
}


//---------------------------------------------------------------------------
TEdit *TKeyerMainForm::GetEditorForMemoryIndex( int iMemoryIndex )
{ switch( iMemoryIndex )
   { case 0 : return Ed_Mem1; // kludge because VCL forms don't support arrays..
     case 1 : return Ed_Mem2;
     case 2 : return Ed_Mem3;
     case 3 : return Ed_Mem4;
     case 4 : return Ed_Mem5;
     case 5 : return Ed_Mem6;
     default: return NULL;
   }
}

//---------------------------------------------------------------------------
void TKeyerMainForm::StartPlaying( int iMemoryIndex )
{ TEdit *pEdit = GetEditorForMemoryIndex(iMemoryIndex);
  if( pEdit != NULL )
   { CwGen_StartReplay( &CwKeyer_Gen, pEdit->Text.c_str() );
     m_iCurrentMemoryIndex = iMemoryIndex;
     m_iCurrentCharIndex   = 0;
     // The rest ("colouring" of certain GUI elements or a range of characters)
     // happens later, in Timer1Timer(), by monitoring the variables
     // set in KeyerThread() -> CwGen_Handler() .
   }
}
//---------------------------------------------------------------------------

void __fastcall TKeyerMainForm::Btn_Mem1Click(TObject *Sender)
{
  StartPlaying(0);
}
//---------------------------------------------------------------------------

void __fastcall TKeyerMainForm::Btn_Mem2Click(TObject *Sender)
{
  StartPlaying(1);
}
//---------------------------------------------------------------------------

void __fastcall TKeyerMainForm::Btn_Mem3Click(TObject *Sender)
{
  StartPlaying(2);
}
//---------------------------------------------------------------------------

void __fastcall TKeyerMainForm::Btn_Mem4Click(TObject *Sender)
{
  StartPlaying(3);
}
//---------------------------------------------------------------------------

void __fastcall TKeyerMainForm::Btn_Mem5Click(TObject *Sender)
{
  StartPlaying(4);
}
//---------------------------------------------------------------------------

void __fastcall TKeyerMainForm::Btn_Mem6Click(TObject *Sender)
{
  StartPlaying(5);
}
//---------------------------------------------------------------------------


void __fastcall TKeyerMainForm::Btn_StopSendingFromMemoryClick(TObject *Sender)
{
  CwGen_StopReplay( &CwKeyer_Gen );
}
//---------------------------------------------------------------------------

void __fastcall TKeyerMainForm::Ed_MemChange(TObject *Sender)
{
  int i;
  TEdit *pEdit;
  if( m_iUpdating==0 )
   { for(i=0; i<KEYER_NUM_MESSAGE_MEMORIES; ++i)
      { pEdit = KeyerMainForm->GetEditorForMemoryIndex( i/*iMemoryIndex*/ );
        if( pEdit != NULL ) // only if there really is an "editor" for this message memory..
         { SL_strncpy( CwKeyer_Config.szKeyerMemory[i], pEdit->Text.c_str(), KEYER_MAX_CHARS_PER_MEMORY );
         }
      } // end for < all 'keyer message memories' >
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_ColourScheme_DefaultClick(TObject *Sender)
{
  g_SpecDispControl.iColourScheme = COLOUR_SCHEME_DEFAULT;
  SetColourScheme( g_SpecDispControl.iColourScheme );
  m_dblDisplayedSMeterLevel_dB = -1;  // kludge to redraw the S-Meter
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_ColourScheme_DarkClick(TObject *Sender)
{
  g_SpecDispControl.iColourScheme = COLOUR_SCHEME_DARK;
  SetColourScheme( g_SpecDispControl.iColourScheme );
  m_dblDisplayedSMeterLevel_dB = -1;  // kludge to redraw the S-Meter
}

//---------------------------------------------------------------------------
void TKeyerMainForm::AdaptImageSizesOnTRXTab(void)
{ // It's important to do this AT LEAST ONCE, to initialize these three bitmaps
  // on the 'TRX' tab. These three images (TImage a la VCL) are placed on a
  // TPanel element (abused as their parent container) :
  int iTotalHeight, iSpectrumHeight, iFreqScaleHeight;
  int iOldMainTab;
  // Without the following kludge, Img_Spectrum->Height / Width still had
  // the "designed", but not the "actual" values when the 'TRX' tab was hidden,
  // but the SPECTROGRAM should be periodically updated due to
  // CwKeyer_Config.iTRXOptions & KEYER_TRX_OPT_KEEP_RUNNING :
  if( (iOldMainTab=m_iCurrentMainTab) != MAIN_TAB_TRX )
   { SwitchMainTab( MAIN_TAB_TRX ); // ex: PageControl1->ActivePage = TS_TRX;
     // Making the tabsheet visible (even if only for a few milliseconds)
     // convinced the VCL to properly adjust the size of the TABSHEET (TS_TRX),
     // and the various PANELS (with different alignments) :
     //  Pnl_TRX_Top: TPanel with Align=alTop,  FIXED ("designed") height .
     //    '--> children: Ed_VFO, CB_Mod, ...
     //  Pnl_TRX_Bottom: TPanel with Align = alBottom, FIXED ("designed") height.
     //    '--> children: sliders for contrast and brightness, etc
     //  Pnl_TRX_Center: TPanel with Align = alClient : Uses the full "remaining" height.
     //    '--> children: Img_Spectrum, Img_Waterfall, Img_FreqScale .
     //                     |    __________|___________    |
     //                     '---|Heights assigned BELOW|---'
     //                         |______________________|
   }


  iTotalHeight = Pnl_TRX_Center->Height; // sum of heights for the following:
  iFreqScaleHeight = iTotalHeight / 4;
  UTL_LimitInteger( &iFreqScaleHeight, 16, 32 );
  iSpectrumHeight = (iTotalHeight-iFreqScaleHeight) / 3;
  UTL_LimitInteger( &iSpectrumHeight, 16, 200 );

  // From top to bottom (as on Icom screens):
  //  Spectrum (graph, "curve"), Spectrogram (waterfall), Frequency Scale .
  Img_Spectrum->Picture->Bitmap->PixelFormat = pf32bit; // use 32 bit/pixel for SpecDisp.cpp / T_RGBColor !
  Img_Spectrum->Height = iSpectrumHeight;
  Img_Spectrum->Picture->Bitmap->Height = Img_Spectrum->Height;
  Img_Spectrum->Picture->Bitmap->Width  = Img_Spectrum->Width;
  Img_FreqScale->Picture->Bitmap->PixelFormat = pf32bit;
  Img_FreqScale->Picture->Bitmap->Height= Img_FreqScale->Height;
  Img_FreqScale->Width = iFreqScaleHeight;
  Img_FreqScale->Picture->Bitmap->Width = Img_FreqScale->Width;
  // Controlled by the VCL's "Aligment" (alClient), the WATERFALL automatically
  // occupies the remaining height between spectrum (alTop) and f-scale (alBottom).
  // The form's size constraints make sure what remains never gets negative or zero.
  Img_Waterfall->Picture->Bitmap->PixelFormat = pf32bit;
  Img_Waterfall->Picture->Bitmap->Height= Img_Waterfall->Height;
  Img_Waterfall->Picture->Bitmap->Width = Img_Waterfall->Width;
  m_fAdaptImageSizesOnTRXTab = FALSE;  // "done" (until the next 'Resze' or whatever)

  if( m_iCurrentMainTab != iOldMainTab )
   { SwitchMainTab( iOldMainTab ); // ex: PageControl1->ActivePage = TS_TRX;
   }
} // end AdaptImageSizesOnTRXTab()

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::FormResize(TObject *Sender)
{
  // Unfortunately, this event handler is also called through a long and winding
  // road when the VCL invoked "Controls::TControl::SetVisible" !
  // Also, there's a bug in the VCL that ON SOME MACHINES causes the timing scope
  // (and other 'visual controls') to occupy only about THREE QUARTERS of the
  // available 'client area' (on its tabsheet, or the tabsheet / page control
  // is broken itself).
  if( !m_fInitialising )
   {  ClearScopeOverlays();
      TimingScope_iUpdateCountOnTestTab = -1; // kludge to redraw the timing scope..
      TimingScope_iUpdateCountOnKeyerTab= -1; // ... even when 'paused' .
      m_fAdaptImageSizesOnTRXTab = TRUE;  // call AdaptImageSizesOnTRXTab() it that tab ever gets visible
   }
}
//---------------------------------------------------------------------------




void __fastcall TKeyerMainForm::Img_TimingScopeOnTestTabMouseDown(
      TObject *Sender, TMouseButton Button, TShiftState Shift, int X, int Y)
{ if( Shift.Contains(ssLeft) ) // only WHILE THE LEFT MOUSE BUTTON is pressed..
   { HandleMouseEventInTimingScope( Img_TimingScopeOnTestTab->Picture->Bitmap, EVT_MOUSE_DOWN, X, Y );
   }
}
//---------------------------------------------------------------------------

void __fastcall TKeyerMainForm::Img_TimingScopeOnTestTabMouseMove(
      TObject *Sender, TShiftState Shift, int X, int Y)
{ if( Shift.Contains(ssLeft) ) // only WHILE THE LEFT MOUSE BUTTON is pressed..
   { HandleMouseEventInTimingScope( Img_TimingScopeOnTestTab->Picture->Bitmap, EVT_MOUSE_MOVE, X, Y );
   }
}
//---------------------------------------------------------------------------

void __fastcall TKeyerMainForm::Img_TimingScopeOnTestTabMouseUp(
      TObject *Sender, TMouseButton Button, TShiftState Shift, int X, int Y)
{ if( Button == mbLeft )
   { HandleMouseEventInTimingScope( Img_TimingScopeOnTestTab->Picture->Bitmap, EVT_MOUSE_UP, X, Y );
   }
  else
   { PM_TimingScope->Popup(Mouse->CursorPos.x, Mouse->CursorPos.y);
   }
}
//---------------------------------------------------------------------------

void __fastcall TKeyerMainForm::Img_TimingScopeOnKeyerTabMouseDown(
      TObject *Sender, TMouseButton Button, TShiftState Shift, int X, int Y)
{ HandleMouseEventInTimingScope( Img_TimingScopeOnKeyerTab->Picture->Bitmap, EVT_MOUSE_DOWN, X, Y );
}
//---------------------------------------------------------------------------

void __fastcall TKeyerMainForm::Img_TimingScopeOnKeyerTabMouseMove(
      TObject *Sender, TShiftState Shift, int X, int Y)
{ if( Shift.Contains(ssLeft) ) // only WHILE THE LEFT MOUSE BUTTON is pressed..
   { HandleMouseEventInTimingScope( Img_TimingScopeOnKeyerTab->Picture->Bitmap, EVT_MOUSE_MOVE, X, Y );
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Img_TimingScopeOnKeyerTabMouseUp(
      TObject *Sender, TMouseButton Button, TShiftState Shift, int X, int Y)
{ if( Button == mbLeft )
   { HandleMouseEventInTimingScope( Img_TimingScopeOnKeyerTab->Picture->Bitmap, EVT_MOUSE_UP, X, Y );
   }
  else
   { PM_TimingScope->Popup(Mouse->CursorPos.x, Mouse->CursorPos.y);
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::PM_TimingScopePopup(TObject *Sender)
{
  PM_TScope_ShowDashInput->Checked= (CwKeyer_TimingScope.cfg.iVisibleChannels & (1<<TIMING_SCOPE_CHANNEL_DASH_INPUT) ) != 0;
  PM_TScope_ShowDotInput->Checked = (CwKeyer_TimingScope.cfg.iVisibleChannels & (1<<TIMING_SCOPE_CHANNEL_DOT_INPUT ) ) != 0;
  PM_TScope_ShowKeyingOutput->Checked=(CwKeyer_TimingScope.cfg.iVisibleChannels&(1<<TIMING_SCOPE_CHANNEL_CW_OUTPUT ) ) !=0;
  PM_TScope_ShowPttOutput->Checked= (CwKeyer_TimingScope.cfg.iVisibleChannels & (1<<TIMING_SCOPE_CHANNEL_FOUR      ) ) != 0;
  PM_TScope_ShowAudioIn->Checked  = (CwKeyer_TimingScope.cfg.iVisibleChannels & (1<<TIMING_SCOPE_CHANNEL_AUDIO_INPUT) ) != 0;
  PM_TScope_ShowAudioOut->Checked = (CwKeyer_TimingScope.cfg.iVisibleChannels & (1<<TIMING_SCOPE_CHANNEL_AUDIO_OUTPUT)) != 0;
  PM_TScope_FreeRun->Checked = (CwKeyer_TimingScope.cfg.iTriggerOptions & TIMING_SCOPE_TRIGGER_FREE_RUN) != 0;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::PM_TScope_ShowDashInputClick(TObject *Sender)
{ CwKeyer_TimingScope.cfg.iVisibleChannels ^= (1<<TIMING_SCOPE_CHANNEL_DASH_INPUT );
  TimingScope_iUpdateCountOnTestTab = TimingScope_iUpdateCountOnKeyerTab = -1; // redraw a.s.a.p.
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::PM_TScope_ShowDotInputClick(TObject *Sender)
{ CwKeyer_TimingScope.cfg.iVisibleChannels ^= (1<<TIMING_SCOPE_CHANNEL_DOT_INPUT );
  TimingScope_iUpdateCountOnTestTab = TimingScope_iUpdateCountOnKeyerTab = -1;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::PM_TScope_ShowKeyingOutputClick(TObject *Sender)
{ CwKeyer_TimingScope.cfg.iVisibleChannels ^= (1<<TIMING_SCOPE_CHANNEL_CW_OUTPUT );
  TimingScope_iUpdateCountOnTestTab = TimingScope_iUpdateCountOnKeyerTab = -1;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::PM_TScope_ShowPttOutputClick(TObject *Sender) // .. or "Channel Four" ..
{ CwKeyer_TimingScope.cfg.iVisibleChannels ^= (1<<TIMING_SCOPE_CHANNEL_FOUR );
  TimingScope_iUpdateCountOnTestTab = TimingScope_iUpdateCountOnKeyerTab = -1;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::PM_TScope_ShowAudioInClick(TObject *Sender)
{ CwKeyer_TimingScope.cfg.iVisibleChannels ^= (1<<TIMING_SCOPE_CHANNEL_AUDIO_INPUT);
  TimingScope_iUpdateCountOnTestTab = TimingScope_iUpdateCountOnKeyerTab = -1;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::PM_TScope_ShowAudioOutClick(TObject *Sender)
{ CwKeyer_TimingScope.cfg.iVisibleChannels ^= (1<<TIMING_SCOPE_CHANNEL_AUDIO_OUTPUT);
  TimingScope_iUpdateCountOnTestTab = TimingScope_iUpdateCountOnKeyerTab = -1;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::PM_TScope_FreeRunClick(TObject *Sender)
{ CwKeyer_TimingScope.cfg.iTriggerOptions ^= TIMING_SCOPE_TRIGGER_FREE_RUN;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::PM_TScope_ClearOverlaysClick(TObject *Sender)
{ ClearScopeOverlays();
}

void __fastcall TKeyerMainForm::PM_TScope_HelpClick(TObject *Sender)
{
  YHF_HELP_ShowHelpContext(HELPID_TIMING_SCOPE);
}
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_ClearScopeOverlaysClick(TObject *Sender)
{ ClearScopeOverlays();
}

void __fastcall TKeyerMainForm::MI_ClearDebugAndDecoderOutputClick(TObject *Sender)
{ Ed_ErrorHistory->Clear();
}
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_ReportTestResultsClick(TObject *Sender)
{
  int i, n, iClientWidth, iPageControlWidth, iChildWidth;
  char sz255[256], *pszDest, *pszEndstop = sz255+255;
  int iErrorClass = ERROR_CLASS_INFO | SHOW_ERROR_IN_RUN_LOG;

  Ed_ErrorHistory->Clear();
  m_fAutomaticTabSwitching = FALSE; // stop AUTOMATIC tab-switching, e.g. from DEBUG- to TRX tab
  SwitchMainTab( MAIN_TAB_DEBUG ); // ex: PageControl1->ActivePage = TS_Debug;
  ShowError( iErrorClass, "KEYER THREAD.." );
  ShowError( iErrorClass, " Poll inputs   : av= %d us, pk= %d us",
            (int)CwKeyer_GetSpeedTestAverage_us(KEYER_SPEEDTEST_POLL_INPUTS),
            (int)CwKeyer_iSpeedTestPeaks_us[KEYER_SPEEDTEST_POLL_INPUTS] );
  ShowError( iErrorClass, " Drive outputs : av= %d us, pk= %d us",
            (int)CwKeyer_GetSpeedTestAverage_us(KEYER_SPEEDTEST_SET_OUTPUTS),
            (int)CwKeyer_iSpeedTestPeaks_us[KEYER_SPEEDTEST_SET_OUTPUTS] );
  if( CwKeyer_Config.fKeyViaShiftAndControl )
   { ShowError(iErrorClass," Poll keyboard: av= %d us, pk= %d us",
            (int)CwKeyer_GetSpeedTestAverage_us(KEYER_SPEEDTEST_POLL_KEYBRD),
            (int)CwKeyer_iSpeedTestPeaks_us[KEYER_SPEEDTEST_POLL_KEYBRD] );
   }
  pszDest = sz255; // Convert the 'CW Keyer-Thread loop intervals' into a string:
  for(i=0; i<8; ++i)
   { SL_AppendInt( &pszDest, pszEndstop, CwKeyer_dw8ThreadIntervals_us[i] );
     SL_AppendChar(&pszDest, pszEndstop, ' ' );
   }
  ShowError( iErrorClass, " Loop intervals: %sus", sz255 );
     // (the above may sooner of later replace the display in 'Ed_ThreadInfo',
     //  because the user will now be interested in all this stuff.. )
  CwKeyer_ResetSpeedTestResults(); // begin a new "peak- and average detection"
  if( ! Keyer_fDotInputWasPassive )
   { ShowError( ERROR_CLASS_INFO | SHOW_ERROR_IN_RUN_LOG," Error: DOT-input seems to be stuck 'active' !" );
   }
  if( ! Keyer_fDashInputWasPassive )
   { ShowError( ERROR_CLASS_INFO | SHOW_ERROR_IN_RUN_LOG," Error: DASH-input seems to be stuck 'active' !" );
   }
#if(SWI_HARDCORE_DEBUGGING) // (1) = hardcore-debugging, (0)=normal compilation
  ShowError( iErrorClass, " Source line nr: %d", (int)Keyer_iLastSourceLine );
#endif // SWI_HARDCORE_DEBUGGING ?

  ShowError(iErrorClass, "DSP THREAD.." );
  ShowError(iErrorClass," Audio input   : av= %d us, pk= %d us",
            (int)CwDSP_GetSpeedTestAverage_us( &CwKeyer_DSP, DSP_SPEEDTEST_READ_AUDIO_INPUT),
            (int)CwKeyer_DSP.iSpeedTestPeaks_us[DSP_SPEEDTEST_READ_AUDIO_INPUT] );
  ShowError(iErrorClass," Audio output  : av= %d us, pk= %d us",
            (int)CwDSP_GetSpeedTestAverage_us( &CwKeyer_DSP, DSP_SPEEDTEST_READ_AUDIO_OUTPUT),
            (int)CwKeyer_DSP.iSpeedTestPeaks_us[DSP_SPEEDTEST_READ_AUDIO_OUTPUT] );
  ShowError(iErrorClass," Downsampling  : av= %d us, pk= %d us",
            (int)CwDSP_GetSpeedTestAverage_us( &CwKeyer_DSP, DSP_SPEEDTEST_DOWNSAMLE_INPUT),
            (int)CwKeyer_DSP.iSpeedTestPeaks_us[DSP_SPEEDTEST_DOWNSAMLE_INPUT] );
  ShowError(iErrorClass," Upsampling    : av= %d us, pk= %d us",
            (int)CwDSP_GetSpeedTestAverage_us( &CwKeyer_DSP, DSP_SPEEDTEST_UPSAMPLE_OUTPUT),
            (int)CwKeyer_DSP.iSpeedTestPeaks_us[DSP_SPEEDTEST_UPSAMPLE_OUTPUT] );
  pszDest = sz255; // Convert the 'DSP Keyer-Thread loop intervals' into a string:
  for(i=0; i<8; ++i)
   { SL_AppendInt( &pszDest, pszEndstop, CwKeyer_DSP.dw8ThreadIntervals_us[i] );
     SL_AppendChar(&pszDest, pszEndstop, ' ' );
   }
  ShowError(iErrorClass," Loop intervals: %sus", sz255 );
  CwDSP_ResetSpeedTestResults( &CwKeyer_DSP ); // begin a new "peak- and average detection"
  pszDest = sz255; // Convert the 'Sidetone audio samples per Keyer-Thread-loop' into a string:
  for(i=0; i<16; ++i) // (used to test the proper function of CwDSP_UpdateSidetone() )
   { SL_AppendInt( &pszDest, pszEndstop, CwKeyer_DSP.i16SidetoneSamplesPerUpdate[i] );
     SL_AppendChar(&pszDest, pszEndstop, ' ' );
   }
  ShowError(iErrorClass," SidetoneSamples: %s", sz255 );
  if( CwKeyer_DSP.pDSW != NULL )
   { ShowError(iErrorClass, "Audio I/O ('DirectSound Wrapper').." );
     ShowError(iErrorClass," Input from '%s' :", CwKeyer_DSP.pDSW->sz255InputDeviceName );
     ShowError(iErrorClass,"    %lu kSamples read, %ld kS captured, %d Overflows",
              (unsigned long)(CwKeyer_DSP.pDSW->dblSamplesReadFromInput/1024.0),
              (unsigned long)(CwKeyer_DSP.pDSW->dblSamplesCapturedFromInput/1024.0),
              (int)CwKeyer_DSP.pDSW->nInputOverflows );
     ShowError(iErrorClass,"    Last Error: %s", CwKeyer_DSP.pDSW->sz255LastInputError );
     ShowError(iErrorClass," Output to '%s' :", CwKeyer_DSP.pDSW->sz255OutputDeviceName );
       // ,-------------------------------------------------------|___________________|
       // '--> 2024-02-25 : Contained a garbage characters when there was
       //                   NO OUTPUT DEVICE SELECTED at all !
       //  The same garbage also in MyDirectSound.sz255OutputDeviceName .
       //  On first sight, memory trashed somewhere during the call
       //  to VorbisStream_Init() . Tried to examine this in CheckSystemHealth().
       //
     ShowError(iErrorClass,"    %lu kSamples written, %lu kS played, %d Underflows",
              (unsigned long)(CwKeyer_DSP.pDSW->dblSamplesWrittenToOutput/1024.0),
              (unsigned long)(CwKeyer_DSP.pDSW->dblSamplesPlayedToOutput/1024.0),
              (int)CwKeyer_DSP.pDSW->nOutputUnderflows );
     ShowError(iErrorClass,"    Last Error: %s", CwKeyer_DSP.pDSW->sz255LastOutputError );
   } // end if( CwKeyer_DSP.pDSW != NULL )

  // Show what's going on in the 'RIG CONTROL' module ?
  if( CwKeyer_Config.iRadioCIVAddress >= 0 ) // Rig-Control SHOULD be enabled and active..
   { T_RigCtrlInstance *pRigCtrl = &MyCwNet.RigControl;
     ShowError(iErrorClass, "RIG CONTROL.." );
     ShowError(iErrorClass, " %ld spectra, %d bins, %ld VFO reports, %.5lf MHz",
       (long)pRigCtrl->nCompleteSpectraReceived,
        (int)pRigCtrl->nSpectrumBinsUsed,
       (long)pRigCtrl->iFrequencyModifiedByRadio_cnt,
       (double)(pRigCtrl->dblVfoFrequency * 1e-6) );
     ShowError(iErrorClass, " %ld messages sent, %ld received on the 'Radio' port",
       (long)pRigCtrl->PortInstance[RIGCTRL_PORT_RADIO].dwNumMessagesSent,
       (long)pRigCtrl->PortInstance[RIGCTRL_PORT_RADIO].dwNumMessagesRcvd );
     if( pRigCtrl->PortInstance[RIGCTRL_PORT_RADIO].dwNumGarbageBytes > 0 )
      { ShowError(iErrorClass, " %ld bytes of unrecognized garbage on the 'Radio' port",
          (long)pRigCtrl->PortInstance[RIGCTRL_PORT_RADIO].dwNumGarbageBytes );
      }
     ShowError(iErrorClass, " Parameter polling state : %s, SubState=%d",
        RigCtrl_ParameterPollingStateToString(pRigCtrl->iParameterPollingState),
        (int)pRigCtrl->iParameterPollingSubState );
     ShowError(iErrorClass, " PTTFlag=%d", (int)pRigCtrl->fLocalPTTFlag );
   } // end if( MyCwNet.RigControl.initState == initState_Opened )


  // Show what's going on in the 'CW NETWORK' module:
  ShowError(iErrorClass, "NETWORK THREAD.." );
  ShowError(iErrorClass, " Status: %s", CwNet_GetCurrentStatusAsString( &MyCwNet ) );
#if( SWI_HARDCORE_DEBUGGING )  // show line number of a potentially 'crashed worker thread':
  ShowError(iErrorClass, " Source lines  : CwNet=%d, HttpSrv=%d",
                 (int)CwNet_iLastSourceLine, (int)HttpSrv_iLastSourceLine );
    // Murphy will make sure the program NEVER crashes under debugger control !
#endif // SWI_HARDCORE_DEBUGGING ?
  ShowError(iErrorClass, " Last Error    : %s",CwNet_GetLastErrorAsString( &MyCwNet ) );
  // ex: ShowError(iErrorClass, " Bytes RX,TX   : %ld, %ld", (long)MyCwNet.dwNumBytesRcvd, (long)MyCwNet.dwNumBytesSent);
  ShowError(iErrorClass, " Network Thread: %ld loops, %ld errors, RxPeak=%d bytes",
                (long)MyCwNet.dwThreadLoops,
                (long)MyCwNet.dwThreadErrors,
                 (int)MyCwNet.iPeakRxBufferUsage);
  pszDest = sz255; // Convert the 'Network Thread loop intervals' into a string:
  for(i=0; i<8; ++i)
   { SL_AppendInt( &pszDest, pszEndstop, MyCwNet.dw8ThreadIntervals_us[i] );
     SL_AppendChar(&pszDest, pszEndstop, ' ' );
   }
  ShowError(iErrorClass, " Loop intervals: %s us", sz255 );
  n = MyCwNet.nRemoteClientsConnected;
  ShowError(iErrorClass, "Currently active connection(s) :" );
  for( i=0; i<=/*!*/CWNET_MAX_CLIENTS; ++i )
   { // i=0 is the "local client" (when operating AS CLIENT),
     // i=1..CWNET_MAX_CLIENTS are the "remote clients" (when operating AS SERVER).
     // Because REMOTE clients may 'come and go', array indices aren't predictable.
     T_CwNetClient *pClient = &MyCwNet.Client[i];
     if( pClient->iClientState != CWNET_CLIENT_STATE_DISCONN )
      { T_HttpInstance *pHttpInst;
#      if( SWI_USE_HTTP_SERVER )  // support the optional, built-in HTTP server ?

        if( pClient->fUsesHTTP && ((pHttpInst=pClient->pHttpInst)!=NULL) )
         { ShowError(iErrorClass, " #%d: %s on %s:%d, %s%s, rx=%lu, tx=%lu, err='%s'",
             (int)i, // --------------'   |    |____|  |        '----,   |   |
              pHttpInst->pSession->sz40UserName,// |   '-------,     |   |   |
              CwNet_IPv4AddressToString(pClient->b4HisIP.b),// |     |   |   |
              (int)pClient->iHisPort,                     //  \|/   \|/ \|/ \|/ ...
              HttpSrv_MethodToString(pHttpInst->nMethod), // GET, POST, PUT, or what the heck..
              pClient->pHttpInst->sz255RequestedURLWithoutQuery,
              (unsigned long)pHttpInst->i64TotalBytesRcvd,
              (unsigned long)pHttpInst->i64TotalBytesSent,
              pHttpInst->sz80ErrorMessage );
            // Listing ALL CONNECTIONS (~~sockets) like this revealed why
            // something "seemed to be missing" sometimes, when following a single
            // HTTP stream with Wireshark: The browser bombarded the poor little
            // web server with MULTIPLE connections, sometimes even requesting
            // THE SAME RESOURCE :
            // > Currently active connection(s) :
            // >  #1: Moritz on 127.0.0.1:53780, GET /LiveData.htm, rx=63550, tx=45725, err=''
            // >  #2: Moritz on 127.0.0.1:53785, GET /LiveData.htm, rx=62730, tx=45135, err=''
            // ,-------------------------------------------------------'
            // '--> These byte counters confirmed that BOTH socket were 'alive' .
            //      The RECEIVE counter increasing faster than the TRANSMIT counter
            //      was caused by the chatty HTTP REQUEST HEADERS (with soo many lines);
            //      they were larger than the CONTENT (delivered to the Javascript
            //      from the dynamically generated HTML fragment 'LiveData.htm').
         }
        else // not using HTTP (using a web browser) but our simple binary TCP/IP based protocol:
#      endif // SWI_USE_HTTP_SERVER ?
         { ShowError(iErrorClass, " #%d: IP=%s, %s, latency=%d ms", (int)i,
              CwNet_IPv4AddressToString( pClient->b4HisIP.b ),
              CwNet_ClientStateToString( pClient->iClientState ),
              (int)CwNet_GetLatencyForRemoteClient_ms( &MyCwNet, i ) );
         }
      }
   } // end for < all clients >

# if( SWI_USE_VORBIS_STREAM ) // allow streaming audio via Ogg/Vorbis ?
  if( MyCwNet.sVorbis.dwInitMagicE == VORBIS_MAGIC_E ) // successfully initialized ?
   { ShowError(iErrorClass, "Ogg/Vorbis encoder: %lu kSamples encoded, %d kS/s, %lu Ogg pages,",
               (unsigned long)(MyCwNet.sVorbis.dwNumSamplesWritten/1000),
                         (int)(MyCwNet.sVorbis.fltSampleRate*1e-3),
               (unsigned long)MyCwNet.sVorbis.dwPageIndex );
     pszDest = sz255; // show the most recent Ogg page sizes:
     for( i=0; i<20; ++i )
      { int iPageBuf = (int)MyCwNet.sVorbis.iBufferPagesHead - 1 - i;
        if( iPageBuf < 0 )
         {  iPageBuf += VORBIS_STREAM_BUFFER_MAX_PAGES;
         }
        if( (iPageBuf>=0) && (iPageBuf<VORBIS_STREAM_BUFFER_MAX_PAGES) )
         { SL_AppendPrintf( &pszDest, pszEndstop, "%d ",
               (int)MyCwNet.sVorbis.BufferPages[iPageBuf].iSize_Byte );
         }
      }
     ShowError(iErrorClass, "Ogg pages: %s bytes", sz255 );
   } // end if < Ogg/Vorbis encoder initialized ("magic E set") ? >
# endif // SWI_USE_VORBIS_STREAM

# if( SWI_USE_HAMLIB_SERVER ) // not sure if this 'feature' is going to stay...
  if( CwKeyer_Config.fEnableBuiltInHamlibServer )
   { ShowError(iErrorClass, "HAMLIB SERVER.." );
     ShowError(iErrorClass, " Status: %s", HLSRV_GetCurrentStatusAsString( &MyHamlibServer ) );
   } // end if( CwKeyer_Config.fEnableBuiltInHamlibServer )
# endif // SWI_USE_HAMLIB_SERVER ?

  // Dump the thread-safe FIFO (filled by ShowError()) into the thread-unsafe 'RichEdit' control,
  // even if the output on the 'Debug' tab is currently PAUSED :
  DumpErrorHistoryToDebugTab();  // here: to report 'test results' in the GUI

  (void)n;

} // end < ..ReportTestResults().. >

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_ReportRigCtrlParamsClick(TObject *Sender)
  // Added 2024-07 to find out which parameters required by HamlibServer.c
  //               work properly, and which don't.
  // This report can be invoked through the main menu via
  //  'Settings'..'Report Rig Control parameters on the Debug tab' .
{
  char sz255[256], *pszDest, *pszEndstop = sz255+255;
  char sz80Value[84];
  int iErrorClass = ERROR_CLASS_INFO | SHOW_ERROR_IN_RUN_LOG;
  T_RigCtrlInstance   *pRC = &MyCwNet.RigControl;
  T_RigCtrl_ParamInfo *pPI;
  T_RigCtrl_RadioInfo *pRadioInfo;
  T_RigCtrlFreqMemEntry *pBandStackingReg;
  int    iUnifiedPN, nValidParameters, iValue, i, i1, i2, n;
  double dblValue;

  Ed_ErrorHistory->Clear();
  m_fAutomaticTabSwitching = FALSE; // stop AUTOMATIC tab-switching, e.g. from DEBUG- to TRX tab
  SwitchMainTab( MAIN_TAB_DEBUG ); // ex: PageControl1->ActivePage = TS_Debug;
  ShowError( iErrorClass, "Current Rig Control parameters:" );

  pRadioInfo = RigCtrl_GetRadioInfoByDefaultAddress(
                  pRC->iRadioCtrlProtocol, pRC->iDefaultAddress );

  nValidParameters = iUnifiedPN = 0;
  while( (iUnifiedPN=RigCtrl_EnumerateUnifiedParameters(iUnifiedPN/*iPreviousUnifiedPN*/)) != 0 )
   { if( (pPI = RigCtrl_GetInfoForUnifiedParameterNumber(iUnifiedPN)) != NULL )
      { if( RigCtrl_GetParamValueAString( pRC, pPI, sz80Value, 80/*iMaxLen*/ ) )
         { pszDest = sz255;
           SL_AppendPrintf( &pszDest, pszEndstop, "PN%03d: %s = %s", (int)pPI->iUnifiedPN, pPI->pszToken, sz80Value );
           if( pPI->pszUnit != NULL )
            { SL_AppendPrintf( &pszDest, pszEndstop, " %s",(char*)pPI->pszUnit );
            }
           ShowError(iErrorClass, sz255);
           ++nValidParameters;
         }
      }
     ++pPI;
   } // end while < all "unified parameters" >

  // When in use, show Icom's 'Band Stacking Registers'
  //      (special memory channels automatically updated by the rig,
  //       filled with the most recent THREE frequencies+modes per band),
  // or the 'emulated' band stacking registers from RigControl.c :
  n = pRC->iNumBandStackingRegs;
  if( n > 0 )
   {
     for(i=0; i<n; i+=3 )
      { pszDest = sz255;
        i1 = i;
        i2 = i+2;
        if( i2>=n )
         {  i2=n-1;
         }
        SL_AppendPrintf( &pszDest, pszEndstop, "BandStack[%02d..%02d]:",(int)i1, (int)i2 );
        while( i1<=i2 )
         { pBandStackingReg = &pRC->BandStackingRegs[ i1 ];
           SL_AppendPrintf( &pszDest, pszEndstop, " %8.3lf %3s",
                 (double)(pBandStackingReg->RxTx[0].dblOperatingFreq_Hz * 1e-6),
                 RigCtrl_OperatingModeToString( pBandStackingReg->RxTx[0].iOpMode
                           & RIGCTRL_OPMODE_MASK_TO_STRIP_FLAGS ) );
           if( i1<i2 ) // more "stacked entries" for this band (or "GENE" for all the rest) ?
            { SL_AppendString( &pszDest, pszEndstop, ", " );
            }
           ++i1;
         }
        ShowError(iErrorClass, sz255);
      }
   } // end if < pRC->iNumBandStackingRegs > 0 >

  // Final 'summary' :
  pszDest = sz255;
  SL_AppendPrintf( &pszDest, pszEndstop, "Rig Control: Got %d parameter(s)",
     (int)nValidParameters );
  if( pRC->iNumBandStackingRegs > 0 )
   { SL_AppendPrintf( &pszDest, pszEndstop, " and %d band stacking entries",
     (int)pRC->iNumBandStackingRegs );
   }
  if( pRadioInfo != NULL ) // only if RigControl.c could also detect the RIG MODEL:
   { SL_AppendPrintf( &pszDest, pszEndstop, " from %s on %s .",
      pRadioInfo->pszName, RigCtrl_GetRadioControlPortAsString() );
   }
  ShowError(iErrorClass, sz255);


  // Dump the thread-safe FIFO (filled by ShowError()) into the thread-unsafe 'RichEdit' control,
  // even if the output on the 'Debug' tab is currently PAUSED :
  DumpErrorHistoryToDebugTab();  // here: to report 'test results' in the GUI


} // end < ..ReportRigCtrlParams().. >


//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_ShowStraightKeyDecoderHistoryClick(TObject *Sender)
{
  int i, n, iClientWidth, iPageControlWidth, iChildWidth;
  char sz255[256], *pszDest, *pszEndstop = sz255+255;
  int iErrorClass = ERROR_CLASS_INFO | SHOW_ERROR_IN_RUN_LOG;


  Ed_ErrorHistory->Clear();
  m_fAutomaticTabSwitching = FALSE; // stop AUTOMATIC tab-switching, e.g. from DEBUG- to TRX tab
  SwitchMainTab( MAIN_TAB_DEBUG ); // ex: PageControl1->ActivePage = TS_Debug;

  // If the 'Straight Key Decoder' has new entries in its history, show them:
  if( CwKeyer_StraightKeyDecoder.iHistoryHeadIndex != CwKeyer_StraightKeyDecoder.iHistoryTailIndex )
   { ShowError(iErrorClass, "'Straight Key' decoder: t_dot=%d ms; history:",
             (int)CwKeyer_StraightKeyDecoder.iDotTime_ms ); // <- this helped a lot to "interpret" the following output..
     while( CwKeyer_StraightKeyDecoder.iHistoryTailIndex != CwKeyer_StraightKeyDecoder.iHistoryHeadIndex )
      { T_StraightKeyDecoderHistoryEntry *pHE = &CwKeyer_StraightKeyDecoder.sHistory[CwKeyer_StraightKeyDecoder.iHistoryTailIndex];
        CwKeyer_StraightKeyDecoder.iHistoryTailIndex = (CwKeyer_StraightKeyDecoder.iHistoryTailIndex+1) % SKD_HISTORY_N_ENTRIES;
        pszDest = sz255; // Convert the 'Network Thread loop intervals' into a string:
        SL_AppendPrintf( &pszDest, pszEndstop,
              " #%d: %s after %ld us : ShiftReg=0x%04X, new State=%s",
             (int)CwKeyer_StraightKeyDecoder.iHistoryTailIndex,
             (char*)(pHE->fKeyDown ? "Key Down" : "Key Up"),
             (long)pHE->nMicrosecondsBeforeNewState,
              (int)pHE->wShiftReg,
             (char*)StraightKeyDecoder_StateToString(pHE->iState) );
        if( pHE->wEmittedCode != 0 )
         { SL_AppendPrintf( &pszDest, pszEndstop, ", decoded: \"%s\"",
                 Elbug_MorseCodePatternToASCII( pHE->wEmittedCode ) );
         }
        ShowError(iErrorClass, sz255);
      }
   } // end if < new entries in the Straight Key Decoder's "history" > ?
  else
   { ShowError(iErrorClass, "'Straight Key' decoder: No new entries in the history." );
   }

  DumpErrorHistoryToDebugTab();  // here: to report 'test results' in the GUI
} // end < ..ShowStraightKeyDecoderHistory.. >


//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Server_EnableHTTPClick(TObject *Sender)
{ if( m_iUpdating==0 )
   { MyCwNet.cfg.iHttpServerOptions ^= HTTP_SERVER_OPTIONS_ENABLE;
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Server_RestrictAccessClick(TObject *Sender)
{ if( m_iUpdating==0 )
   { MyCwNet.cfg.iHttpServerOptions ^= HTTP_SERVER_OPTIONS_RESTRICT; // .. to visitors specifying a VALID USER NAME
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Server_AddTestToneClick(TObject *Sender)
{ if( m_iUpdating==0 )
   { CwKeyer_DSP.cfg.iAudioFlags ^= DSP_AUDIO_FLAGS_SEND_NETWORK_TEST_TONE;
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Network_DiagnosticModeClick(TObject *Sender)
{ if( m_iUpdating==0 )
   { MyCwNet.cfg.iDiagnosticFlags ^= CWNET_DIAG_FLAGS_VERBOSE;
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Network_SimulateBadConnectionClick(TObject *Sender)
{ if( m_iUpdating==0 )
   { MyCwNet.cfg.iDiagnosticFlags ^= CWNET_DIAG_FLAGS_SIMULATE_BAD_CONN;
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Network_ShowConnectionLogClick(TObject *Sender)
{ if( m_iUpdating==0 )
   { MyCwNet.cfg.iDiagnosticFlags ^= CWNET_DIAG_FLAGS_SHOW_CONN_LOG;
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Network_ShowNetworkTrafficClick(TObject *Sender)
{ if( m_iUpdating==0 )
   { MyCwNet.cfg.iDiagnosticFlags ^= CWNET_DIAG_FLAGS_SHOW_NETWORK_TRAFFIC;
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Network_ShowRigControlTrafficClick(TObject *Sender)
{ RigCtrl_iTrafficMonitorDisplayOptions ^= RIGCTRL_TRAFFIC_DISPLAY_OPTION_ENABLE;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_RigCtrl_RejectEchosInLogClick(TObject *Sender)
{ MyCwNet.RigControl.dwRejectMessages4Log ^= (1<<RIGCTRL_MSGTYPE_ECHO);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_RigCtrl_RejectPeriodicCmdsClick(TObject *Sender)
{
  MyCwNet.RigControl.dwRejectMessages4Log ^= RIGCTRL_MSGTYPE_FLAG_PERIODIC_POLL;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_RigCtrl_RejectSpectrumInLogClick(TObject *Sender)
{ MyCwNet.RigControl.dwRejectMessages4Log ^= (1<<RIGCTRL_MSGTYPE_SPECTRUM);
}

void __fastcall TKeyerMainForm::MI_RigCtrl_RejectFrequencyReportsClick(TObject *Sender)
{ MyCwNet.RigControl.dwRejectMessages4Log ^= (1<<RIGCTRL_MSGTYPE_FREQUENCY_REPORT);
}
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Network_ShowRcvdKeyingBytestreamClick(TObject *Sender)
{ if( m_iUpdating==0 )
   { MyCwNet.cfg.iDiagnosticFlags ^= CWNET_DIAG_FLAGS_SHOW_KEYING_BYTESTREAM;
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_HLServer_OpenControlTabClick(TObject *Sender)
{
  if( (PageControl1->ActivePage!=TS_Config) || (PC_Config->ActivePage!=TS_RigControl) )
   { UpdateSettingsTab();
     SwitchMainTab( MAIN_TAB_DEBUG ); // ex: PageControl1->ActivePage = TS_Config;
     PC_Config->ActivePage = TS_RigControl;
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_HLServer_EnableClick(TObject *Sender)
{
  if( CwKeyer_Config.fEnableBuiltInHamlibServer ) // DISABLE now :
   {  CwKeyer_Config.fEnableBuiltInHamlibServer = FALSE;
      HLSRV_Stop( &MyHamlibServer );
   }
  else // ENABLE now :
   {  CwKeyer_Config.fEnableBuiltInHamlibServer = TRUE;
      if( MyHamlibServer.cfg.iServerListeningPort > 0 )
       { HLSRV_Start( &MyHamlibServer );
       }
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_HLServer_ShowTrafficClick(TObject *Sender)
{ MyHamlibServer.cfg.iDiagnosticFlags ^= HLSRV_DIAG_FLAGS_SHOW_NETWORK_TRAFFIC;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_HLServer_VerboseOutputClick(TObject *Sender)
{ MyHamlibServer.cfg.iDiagnosticFlags ^= HLSRV_DIAG_FLAGS_VERBOSE_OUTPUT;
}


//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_KeyViaShiftAndControlClick(TObject *Sender)
{ CwKeyer_Config.fKeyViaShiftAndControl = !CwKeyer_Config.fKeyViaShiftAndControl;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Network_OpenServerPageInBrowserClick(TObject *Sender)
{
  char sz255URLAndQueryString[256];
  char *pszDest    = sz255URLAndQueryString;
  char *pszEndstop = sz255URLAndQueryString + 255;
  BOOL fOk;

  sz255URLAndQueryString[0] = '\0';

  // Whether to open the 'Server Page' via LOCALHOST or an 'external' URL
  // depends on the current setting on the 'Network' tab :
  if( Rbtn_Client->Checked )  // this instance operates as CLIENT so use the REMOTE SERVER's URL:
   { SL_AppendString( &pszDest, pszEndstop, "http://" );
     SL_AppendString( &pszDest, pszEndstop, Ed_ClientRemoteIP->Text.c_str() );
     if( SL_GetLastChar(sz255URLAndQueryString, 80) != '/' )
      { SL_AppendString( &pszDest, pszEndstop, "/" );
      }
     SL_AppendString( &pszDest, pszEndstop, "?user=" );
     INET_PlainTextToUrlEncoding(
        Ed_ClientUserName->Text.c_str(), // [in] char * pszPlainTextSource,
        &pszDest, pszEndstop,            // [out] URL encoded result
        "\0"/*pszDelimiters*/ );
     SL_AppendString( &pszDest, pszEndstop, "&call=" );
     INET_PlainTextToUrlEncoding(
        Ed_ClientCallsign->Text.c_str(),
        &pszDest, pszEndstop,            // [out] URL encoded result
        "\0"/*pszDelimiters*/ );
   } // end if < instance operates as CLIENT >
  else if( Rbtn_Server->Checked ) // this instance operates as SERVER so use the LOCALHOST:
   { SL_AppendString( &pszDest, pszEndstop, "http://127.0.0.1:" );
     SL_AppendString( &pszDest, pszEndstop, Ed_ServerListeningPort->Text.c_str());
     SL_AppendString( &pszDest, pszEndstop, "/" );
     // There are MULTIPLE USERS listed under "Accepted Users and Permissions".
     // To 'simulate' a visit of ANY OF THEM, don't simply pick the FIRST user,
     // but use the freedly editable "User" and "Call" from the edit fields
     // that are otherwise used if this instance would operate as CLIENT :
     SL_AppendString( &pszDest, pszEndstop, "?user=" );
     INET_PlainTextToUrlEncoding(
        Ed_ClientUserName->Text.c_str(), // [in] char * pszPlainTextSource,
        &pszDest, pszEndstop,            // [out] URL encoded result
        "\0"/*pszDelimiters*/ );
     SL_AppendString( &pszDest, pszEndstop, "&call=" );
     INET_PlainTextToUrlEncoding(
        Ed_ClientCallsign->Text.c_str(),
        &pszDest, pszEndstop,            // [out] URL encoded result
        "\0"/*pszDelimiters*/ );
   } // end if < instance operates as SERVER >

  if( sz255URLAndQueryString[0] != '\0' ) // got a URL + Query-string to play with ?
   { fOk = HELP_ShowAnyFileOrURL( sz255URLAndQueryString, ""/* no HTML anchor*/,
                                  FALSE/*fTranslateFilenameToURL*/ );
     if( ! fOk )
      { ShowError( ERROR_CLASS_ERROR, "Failed to open '%s'", sz255URLAndQueryString );
      }
   }
} // end MI_Network_OpenServerPageInBrowserClick()


//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_HelpIndexClick(TObject *Sender)
{ YHF_HELP_ShowHelpContext(HELPID_MAIN_INDEX);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Help_MainMenuClick(TObject *Sender)
{ YHF_HELP_ShowHelpContext(HELPID_MAIN_MENU);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Hotkeysummary1Click(TObject *Sender)
{ YHF_HELP_ShowHelpContext(HELPID_HOTKEYS);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Help_IOConfigClick(TObject *Sender)
{ YHF_HELP_ShowHelpContext(HELPID_IO_CONFIG);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Help_KeyerSettingsClick(TObject *Sender)
{ YHF_HELP_ShowHelpContext(HELPID_KEYER_SETTINGS);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Help_AudioSettingsClick(TObject *Sender)
{ YHF_HELP_ShowHelpContext(HELPID_AUDIO_SETTINGS);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Btn_Help_HamlibServerClick(TObject *Sender)
{ YHF_HELP_ShowHelpContext(HELPID_HAMLIB_SERVER);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Help_NetworkSettingsClick(TObject *Sender)
{ YHF_HELP_ShowHelpContext(HELPID_NETWORK_SETTINGS);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Lab_NetworkFunctionalityClick(TObject *Sender)
{ YHF_HELP_ShowHelpContext(HELPID_NETWORK_FUNCS);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Lab_NetStatusClick(TObject *Sender)
{ YHF_HELP_ShowHelpContext(HELPID_NETWORK_STATUS);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Lab_LocalIPHelpClick(TObject *Sender)
{ YHF_HELP_ShowHelpContext(HELPID_LOCAL_IP_ADDRESS);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Lab_ServerListeningPortClick(TObject *Sender)
{ YHF_HELP_ShowHelpContext(HELPID_LISTENING_PORT);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Lab_AcceptedUsersClick(TObject *Sender)
{ YHF_HELP_ShowHelpContext(HELPID_NETWORK_USERS);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Ed_ServerAcceptUsersDblClick(TObject *Sender)
{ // As the "Hint" (a Borland VCL thing) for this edit field promised:
  // "Double-click this field for help". Here: manual/Remote_CW_Keyer.htm#network_users
  YHF_HELP_ShowHelpContext(HELPID_NETWORK_USERS); // e.g. 'bitmasks for permissions'
  // If you've got a lot of time to spend on improving the GUI (WB didn't),
  // use e.g. a scrolling table with a 'checklist' of permissions per user.
  // For a start, this single-line text edit field ("Accepted Users and Permissions")
  // will do.
}


//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Lab_RemoteURLClick(TObject *Sender)
{ YHF_HELP_ShowHelpContext(HELPID_REMOTE_URL_AND_PORT);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Lab_MyUserNameClick(TObject *Sender)
{ YHF_HELP_ShowHelpContext(HELPID_USER_NAME);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Lab_UserCallsignClick(TObject *Sender)
{ YHF_HELP_ShowHelpContext(HELPID_USER_CALLSIGN);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Help_TimingScopeClick(TObject *Sender)
{ YHF_HELP_ShowHelpContext(HELPID_TIMING_SCOPE);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Help_TransceiverControlClick(TObject *Sender)
{ YHF_HELP_ShowHelpContext(HELPID_TRX_TAB);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Help_HardwareClick(TObject *Sender)
{ YHF_HELP_ShowHelpContext(HELPID_HARDWARE);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Help_DebugTabClick(TObject *Sender)
{ YHF_HELP_ShowHelpContext(HELPID_DEBUG_TAB);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_ClearErrorHistoryClick(TObject *Sender)
{ Ed_ErrorHistory->Clear();
}

void __fastcall TKeyerMainForm::MI_CopyHistoryToClipboardClick( TObject *Sender)
{ Ed_ErrorHistory->CopyToClipboard();
}
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Debug_ToggleWordWrap(TObject *Sender)
{
  MyCwNet.cfg.iDiagnosticFlags ^= CWNET_DIAG_FLAGS_WORD_WRAP;
  Ed_ErrorHistory->WordWrap = (MyCwNet.cfg.iDiagnosticFlags & CWNET_DIAG_FLAGS_WORD_WRAP) != 0;
  // "WordWrap" (automatically performed by the RichEdit-thingy) is ON
  // by default, but to see exactly ONE LINE per TCP/IP fragment helped a lot
  // with troubleshooting the embedded web server (HttpServer.c) .
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Debug_VerboseDiagnosticsClick(TObject *Sender)
{
  if( m_iUpdating==0 )
   { MyCwNet.cfg.iDiagnosticFlags ^= CWNET_DIAG_FLAGS_VERBOSE;
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Debug_ShowKeyingBytestreamClick(TObject *Sender)
{
  if( m_iUpdating==0 )
   { MyCwNet.cfg.iDiagnosticFlags ^= CWNET_DIAG_FLAGS_SHOW_KEYING_BYTESTREAM;
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Debug_ShowMorseDecoderOutputClick(TObject *Sender)
{ // Similar as below, but THIS menu item is in the Rich Text editor's context menu
  //  on the 'Debug' tab itself (where advanced users would expect to find it)
  if( m_iUpdating==0 )
   { MyCwNet.cfg.iDiagnosticFlags ^= CWNET_DIAG_FLAGS_SHOW_DECODER_OUTPUT;
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Debug_ShowNetworkConnectionLogClick(TObject *Sender)
{ if( m_iUpdating==0 )
   { MyCwNet.cfg.iDiagnosticFlags ^= CWNET_DIAG_FLAGS_SHOW_CONN_LOG;
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Debug_ShowNetworkTrafficClick(TObject *Sender)
{
  if( m_iUpdating==0 )
   { MyCwNet.cfg.iDiagnosticFlags ^= CWNET_DIAG_FLAGS_SHOW_NETWORK_TRAFFIC;
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Debug_ShowRigCtrlTrafficClick(TObject *Sender)
{
  RigCtrl_iTrafficMonitorDisplayOptions ^= RIGCTRL_TRAFFIC_DISPLAY_OPTION_ENABLE;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Settings_ShowMorseDecoderOutputClick(TObject *Sender)
{ // Similar as above, but THIS menu item is in the old-school 'main menu',
  // under 'Settings' .  Per default, the display of decoded Morse characters
  // is on; but when also displaying the 'encoded morse code stream' in hex,
  // having a mix of all those funny background-coloured characters in the
  // same editor was confusing.
  if( m_iUpdating==0 )
   { MyCwNet.cfg.iDiagnosticFlags ^= CWNET_DIAG_FLAGS_SHOW_DECODER_OUTPUT;
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Debug_PauseDisplayClick(TObject *Sender)
{  // Pauses / resumes the output from the thread-safe 'ShowError'-FIFO
   // into the non-thread-safe 'RichEdit' control on the 'Debug' tab.
   m_fDebugOutputPaused = !m_fDebugOutputPaused;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_PauseDebugOutputClick(TObject *Sender)
{
   m_fDebugOutputPaused = !m_fDebugOutputPaused;
   // Almost the same as in the RichText editor's "context menu",
   // but THIS ONE ("MI_PauseDebugOutput") is in the main menu under "Settings".
}
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::PM_ErrorHistoryPopup(TObject *Sender)
{
  MI_Debug_WordWrap->Checked = (MyCwNet.cfg.iDiagnosticFlags & CWNET_DIAG_FLAGS_WORD_WRAP) != 0;
  MI_Debug_VerboseDiagnostics->Checked = (MyCwNet.cfg.iDiagnosticFlags & CWNET_DIAG_FLAGS_VERBOSE) != 0;
  MI_Debug_ShowKeyingBytestream->Checked=(MyCwNet.cfg.iDiagnosticFlags & CWNET_DIAG_FLAGS_SHOW_KEYING_BYTESTREAM) != 0;
  MI_Debug_ShowMorseDecoderOutput->Checked=(MyCwNet.cfg.iDiagnosticFlags & CWNET_DIAG_FLAGS_SHOW_DECODER_OUTPUT) != 0;
  MI_Debug_ShowNetworkTraffic->Checked = (MyCwNet.cfg.iDiagnosticFlags & CWNET_DIAG_FLAGS_SHOW_NETWORK_TRAFFIC) != 0;
  MI_Debug_ShowNetworkConnectionLog->Checked = (MyCwNet.cfg.iDiagnosticFlags & CWNET_DIAG_FLAGS_SHOW_CONN_LOG) != 0;
  MI_Debug_ShowRigCtrlTraffic->Checked = (RigCtrl_iTrafficMonitorDisplayOptions & RIGCTRL_TRAFFIC_DISPLAY_OPTION_ENABLE) != 0;
  MI_Debug_PauseDisplay->Checked = m_fDebugOutputPaused;
}
//---------------------------------------------------------------------------

void __fastcall TKeyerMainForm::MyDrawMenuItem(TObject *Sender,
                   TCanvas *ACanvas, TRect &ARect, bool Selected)
{
  AnsiString s;
  int i;
  ACanvas->Brush->Color = Selected ? g_clMenuSelBackgnd : g_clMenuBackground;
  ACanvas->Font->Color  = g_clMenuForeground;
  ACanvas->FillRect(ARect);
  s = ((TMenuItem*)Sender)->Caption;
  // Even if you don't want it, some voodo priest inserts a '&' before
  // some character to indicate a hotkey. Eliminate that:
  if( (i=s.AnsiPos("&") ) > 0 ) // character indices in an AnsiString are ONE-based. Yucc.
   { s = s.Delete( i,1 );
   }
  ACanvas->TextOut(ARect.Left + 2, ARect.Top, s);
}
//---------------------------------------------------------------------------

void __fastcall TKeyerMainForm::DrawStatusIndicatorInMenu(TObject *Sender,
      TCanvas *ACanvas, TRect &ARect, bool Selected)
{
  AnsiString s;
  int i, tw;
  if( m_iStatusIndicatorUsage == STATUS_INDICATOR_ON_AIR )
   { ACanvas->Brush->Color = clRed;
   }
  else
   { ACanvas->Brush->Color = g_clMenuBackground;
   }
  ARect.Right = ARect.Left + 100; // attempt to give this field a fixed width
  ACanvas->Font->Color  = g_clMenuForeground;
  ACanvas->FillRect(ARect);
  s = ((TMenuItem*)Sender)->Caption;
  // Even if you don't want it, some voodo magic inserts a '&' somewhere. Eliminate that:
  if( (i=s.AnsiPos("&") ) > 0 ) // character indices in an AnsiString are ONE-based. Yucc.
   { s = s.Delete( i,1 );
   }
  tw = ACanvas->TextWidth( s );
  ACanvas->TextOut(ARect.Left + ( ARect.Right - ARect.Left - tw) / 2, ARect.Top, s);
}
//---------------------------------------------------------------------------



void __fastcall TKeyerMainForm::MI_Help_AboutClick(TObject *Sender)
{
  char sz2k[2048], sz255[256];
  char *pszDest = sz2k;
  char *pszEndstop = sz2k + sizeof(sz2k) - 1;
  int iVersionInfoSize;
  int iVersionInfo[4];
  TVSFixedFileInfo *pFileInfo;
  unsigned int iLength;
  char  cBuffer[4000];
  DWORD dwHandle;

  // Obscure stuff to retrieve the version-info that Borland's linker has placed
  // IN THE *.EXE :
  memset( iVersionInfo, 0, sizeof(iVersionInfo) );
  iVersionInfoSize = GetFileVersionInfoSize( ParamStr(0).c_str(), &dwHandle);
  // '--> "if size is zero, then there is no version info in the exe".
  //      (be prepared for this, and similar nice surprise..)
  // WB, 2023-10 : Got here with a suprising iVersionInfoSize of 1524 bytes !
  if( (iVersionInfoSize>0) && (iVersionInfoSize<sizeof(cBuffer)) )
   {
    try  // another example of the monstrosities of C++ and Win32 :
     { GetFileVersionInfo(ParamStr(0).c_str(), 0, iVersionInfoSize, cBuffer);
       VerQueryValue( cBuffer, "\\"/*lpSubBlock:"root block"*/, (void**)&pFileInfo, &iLength);
       // WB, 2023-10 : Got here with iLength = 52, whatever that means.
       iVersionInfo[0] = pFileInfo->dwFileVersionMS >> 16;
       iVersionInfo[1] = pFileInfo->dwFileVersionMS & 0xffff;
       iVersionInfo[2] = pFileInfo->dwFileVersionLS >> 16;
       iVersionInfo[3] = pFileInfo->dwFileVersionLS & 0xffff;
       // WB, 2023-10 : Got here with ....
       //  iVersionInfo[0] = Borland's "Hauptversion", (major release ?)
       //  iVersionInfo[1] = Borland's "Nebenversion", (minor release ?)
       //  iVersionInfo[2] = Borland's "Ausgabe",      (revision number)
       //  iVersionInfo[3] = Borland's "Compilierung"  ("build number"?)
       // When examining the properties of the *.exe file via file manager,
       // the four numbers in iVersionInfo[0..3] were what Microsoft calls
       //  "File version", not "Product version" !
     }
    catch (...)
     {
     }
   }

  SL_AppendPrintf( &pszDest, pszEndstop, "Remote CW Keyer V%d.%d.%d.%d, compiled %s .\n",
    iVersionInfo[0], iVersionInfo[1], iVersionInfo[2], iVersionInfo[3],  __DATE__ );
  // Note: The version info displayed above should be copied into the
  //       INNO SETUP SCRIPT, in the following line:
  //       > #define MyAppVersion "1.0.2.1"
  SL_AppendPrintf( &pszDest, pszEndstop, "Please check for updates at\n"
    "  www.qsl.net/dl4yhf/Remote_CW_Keyer/Remote_CW_Keyer.htm !\n" );
  memset( sz255, 0, sizeof(sz255) );  // make sure there's a TRAILING ZERO !
  ::GetModuleFileName(0, sz255, 255); // <- this is WINDOWS, not BORLAND/VCL
  SL_AppendPrintf( &pszDest, pszEndstop, "Executable file : %s\n", sz255 );
  SL_AppendPrintf( &pszDest, pszEndstop, "Data file path  : %s\n", g_sz255PathToDataFiles );
  SL_AppendPrintf( &pszDest, pszEndstop, "Command Line: %s\n", g_sz255CommandLine );
  SL_AppendPrintf( &pszDest, pszEndstop, "Last browser commmand:\n %s\n", YHF_HELP_sz1023LastBrowserCommandLine );
  ::MessageBox( Handle, sz2k, "Remote CW Keyer", MB_OK );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::FormMouseWheel(TObject *Sender,
      TShiftState Shift, int WheelDelta, TPoint &MousePos, bool &Handled)
{ // Notes:
  //  - For a single step of the mouse wheel, this event seemed to be called
  //    more than once !  Setting "Handled=TRUE" cured this problem.
  //  - This event handler must be aware of "who" has the focus, because
  //    some visible controls have (and need!) their own mouse wheel handler !
  //  - When, for example, the TEdit for the VFO frequency had the focus,
  //    Borland's "FormMouseWheel()" fired with 'WheelDelta' = 120 (or -120).
  //    Thus ignore the 'WheelDelta' value (only care for the sign)
  //    
  WORD wKey;
  TEdit *pEd = NULL;
  BOOL fHandled=FALSE;
  TPoint pt = Img_Spectrum->ScreenToClient( MousePos );
  int   iSpecDispArea, iSplitScaleIndex, iChannelOfAnalyser;
  float fltXrel, fltYrel;
  double fc, d1, d2, dblBasebandFmin, dblBasebandFmax, dblNewBasebandFmin;


  // Identify the mouse screen area... if it's over the frequency scale,
  // it may be used to zoom or scroll there (similar as in DL4YHF's modified WSQ2 waterfall)
  if( pt.x>=0 && pt.y>=0 && pt.x<Img_Spectrum->Width && pt.y<Img_Spectrum->Height )
   { // Inside the spectrum/spectrogram/frequency scale area, but where exactly ?
     iSpecDispArea = IdentifyClientCoord(pt.x, pt.y, &fltXrel, &fltYrel );
#   if(0)
     switch( iSpecDispArea )
      { case SCREEN_AREA_FREQUENCY_SCALE :
           // Zoom in/out, with the center "under" the mouse pointer :
              fc = FreqScale_MousePosToBasebandFreq( pt.x, pt.y, &iSplitScaleIndex, NULL );
              iChannelOfAnalyser = CLI_SplitScaleIndexToAnalyserChannel( iSplitScaleIndex );
              SpecDisp_GetDisplayableBasebandFreqRange(CFG_SD_MAIN_ANALYSER, iSplitScaleIndex,
                     &dblBasebandFmin, &dblBasebandFmax, SD_GET_FREQ_RANGE_WITHOUT_REPROGRAMMING_LO );
              d1 = fc - pDisplaySettings->Chn[iSplitScaleIndex].dblBasebandFreqMin;
              d2 = pDisplaySettings->Chn[iSplitScaleIndex].dblBasebandFreqMax - fc;
              if( WheelDelta > 0 )
               { d1 *= 0.95;    // zoom IN
                 d2 *= 0.95;
               }
              else if( WheelDelta < 0 )
               { d1 /= 0.95;    // zoom OUT
                 d2 /= 0.95;
               }
              ChangeDisplayedBasebandFreqRange( iSplitScaleIndex, fc-d1, fc+d2, LIMIT_F_RANGE, FALSE/*fInitialCall*/ );
              fHandled = TRUE;
            }
           break;
        default:
           break;
      }
#   endif
   } // end if < inside the spectrogram/frequency scale/spectrum area >


  if( !fHandled )
   { // FALSE = default "new style", i.e. react on input immediately,
     //        and allow increment/decrement of digits via cursor or mousewheel:
     if( Ed_VFO->Focused() )     // VFO frequency (for SDR or external hardware / ExtIO-DLL)
      {  pEd = Ed_VFO;
      }
     if( pEd != NULL )
      { fHandled = TRUE;
        if( WheelDelta > 0 )
         { wKey = VK_UP;
           pEd->OnKeyDown( pEd, wKey, Shift ); // -> Ed_VFOKeyDown()
           Handled = TRUE;
         }
        if( WheelDelta < 0 )
         { wKey = VK_DOWN;
           pEd->OnKeyDown( pEd, wKey, Shift ); // -> Ed_VFOKeyDown()
           Handled = TRUE;
         }
      } // end if < VFO or similar "frequency" edit field has the focus >
   } // end if( ! UConfig.fPlainFreqInputFields )

}
//---------------------------------------------------------------------------

#if( SWI_HARDCORE_DEBUGGING )
//---------------------------------------------------------------------------
void CheckSystemHealth(const char *pszModuleName, int iSourceLine) // checks for 'memory corruption'...
  // Periodically called from a few places via macro CHECK_SYSTEM_HEALTH() .
  // Added 2024-02-25 when suspecting a problem with the hyper-complicated,
  // memory-hoggig Ogg/Vorbis encoder (that later turned out to be utterly
  // non-suited for low-latency audio, at least when used for the <audio>
  // element in HTML 5 without megatons of Javascript..)
  //
  // Because CheckSystemHealth() may be called from WORKER THREADS, hundreds
  // of times per second, keep the "checks" performed in this function to the
  // ultimate minimum (e.g. check a few 'magic numbers' at the end of FIFOs,
  // etc).
{
  BOOL fGotcha = FALSE;
  static BOOL firstBug = TRUE;
  //  2024-02-25: Out of the blue, there was garbage in MyDirectSound.sz255OutputDeviceName :
  if(  (MyDirectSound.sz255OutputDeviceName[0] >  0x00)
     &&(MyDirectSound.sz255OutputDeviceName[0] <= 0x20) )
   { fGotcha = TRUE;
   }

  if( fGotcha && firstBug )
   { ShowError( ERROR_CLASS_ERROR, "System health check failed, caller = %s, line = %d",
                      pszModuleName, (int)iSourceLine );
   }

} // end CheckSystemHealth()


#endif // SWI_HARDCORE_DEBUGGING ?

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::PM_TRX_TabPopup(TObject *Sender)
{
  MI_TRX_KeepRunning->Checked = (CwKeyer_Config.iTRXOptions & KEYER_TRX_OPT_KEEP_RUNNING) != 0;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_TRX_KeepRunningClick(TObject *Sender)
{
  CwKeyer_Config.iTRXOptions ^= KEYER_TRX_OPT_KEEP_RUNNING;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Settings_TRX_OptionsClick(TObject *Sender)
{ // Kludge to open the CONTEXT MENU for the "TRX" tab also from the MAIN menu.
  // (didn't find a way in the VCL to use a "TPopupMenu" as submenu
  //  in a "TMainMenu", as a REAL sub-menu for MainMenu1.MI_Settings_TRX_Options
  //  in Borland C++Builder 6's "form designer" / "object inspector").
  SwitchMainTab( MAIN_TAB_TRX ); // ex: PageControl1->ActivePage = TS_TRX; // switch to the "TRX" tab, with PM_TRX_Tab
  PM_TRX_Tab->Popup(Mouse->CursorPos.x, Mouse->CursorPos.y);
}

//---------------------------------------------------------------------------

void __fastcall TKeyerMainForm::MI_NetworkLatency_AutoDetectClick(TObject *Sender)
{ MyCwNet.cfg.iNetworkLatency_ms = 0; // fixed network latency in milliseconds (default: 0 = auto-detect)
}
//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_NetworkLatency_100msClick(TObject *Sender)
{ MyCwNet.cfg.iNetworkLatency_ms = 100;
}
//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_NetworkLatency_250msClick(TObject *Sender)
{ MyCwNet.cfg.iNetworkLatency_ms = 250;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_NetworkLatency_500msClick(TObject *Sender)
{ MyCwNet.cfg.iNetworkLatency_ms = 500;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_NetworkLatency_1000msClick(TObject *Sender)
{ MyCwNet.cfg.iNetworkLatency_ms = 1000;
}

//---------------------------------------------------------------------------
int MouseButtonToGuiKeyFlags(TMouseButton Button)
  // Abstraction of Borland's "TMouseButton"-thingy,
  //   Unlike "TMouseButton" (which uses ZERO as the code for LEFT mouse button),
  //   our bit-combination of flags like
  //      GUI_KEY_FLAGS_LEFT_MOUSE_BUTTON, GUI_KEY_FLAGS_RIGHT_MOUSE_BUTTON,
  //      GUI_KEY_FLAGS_SHIFT, GUI_KEY_FLAGS_CONTROL, ..
  //   can even encode "NO mouse button pressed" (guess what the value is .. ZERO).
  //   Returns a simple integer, no fancy "set" or whatever.
{ int iKeyFlags = 0;

  // Borland's "TMouseButton" doesn't seem to be a cryptic "set" but an enum.
  // Obviously they don't support detection of BOTH (or even more) buttons
  // pressed simultaneously :
  if( Button==mbLeft )
   { iKeyFlags |= GUI_KEY_FLAGS_LEFT_MOUSE_BUTTON;
   }
  if( Button==mbRight )
   { iKeyFlags |= GUI_KEY_FLAGS_RIGHT_MOUSE_BUTTON;
   }
  return iKeyFlags;
} // end MouseButtonToGuiKeyFlags()

//---------------------------------------------------------------------------
int ShiftStateToGuiKeyFlags( TShiftState Shift)
  // Simplistic abstraction of Borland-specific "TShiftState".
  //     Returns a simple integer, no fancy "set" or whatever.
{ int iKeyFlags = 0;

  // Borland's "TShiftState" is neither an enum nor a simple combination of bits
  //  but a "set of flags". We convert it into simple bitflags in an integer
  //  to get away from the counter-intuitive syntax..
  //  > A 'Set' is a C++ template class designed to emulate the Delphi Set type.
  //  No, thanks. Why use a VCL-specific C++ gizmo when a simple 'int' will do ?
  if( Shift.Contains(ssShift) )
   { iKeyFlags |= GUI_KEY_FLAGS_SHIFT;
   }
  if( Shift.Contains(ssCtrl) )
   { iKeyFlags |= GUI_KEY_FLAGS_CONTROL;
   }
  return iKeyFlags;
} // end ShiftStateToGuiKeyFlags()


//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Img_SpectrumMouseDown(TObject *Sender,
      TMouseButton Button, TShiftState Shift, int X, int Y)
  // Called from Borland's VCL when just pressing the mouse
  // on the specified position ("client area coordinate") in the SPECTRUM.
  // The same handler is used for the "OnMouseDown", "OnMouseMove", "OnMouseUp"
  // events in multiple window parts on the "TRX" tab (at least spectrum,
  // spectrogram, and frequency scale) - see SpecDisp.cpp : SpecDisp_HandleMouseEvent() !
{
  SpecDisp_HandleMouseEvent( &g_SpecDispControl, &MyCwNet.RigControl, GUI_EVENT_MOUSE_DOWN, GUI_CONTROL_SPECTRUM,
      MouseButtonToGuiKeyFlags(Button) | ShiftStateToGuiKeyFlags(Shift), X, Y );
  // Save this for Borland's "OnMouseMove()" handler (which doesn't pass in a "TMouseButton"):
  m_iMouseButtonAsGuiKeyFlags = MouseButtonToGuiKeyFlags( Button );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Img_WaterfallMouseDown(TObject *Sender,
      TMouseButton Button, TShiftState Shift, int X, int Y)
{ // Similar as Img_SpectrumMouseDown(), but this is for the spectrogram aka waterfall,
  // which not only has a FREQUENCY AXIS but also a TIME AXIS:
  SpecDisp_HandleMouseEvent( &g_SpecDispControl, &MyCwNet.RigControl, GUI_EVENT_MOUSE_DOWN, GUI_CONTROL_SPECTROGRAM,
      MouseButtonToGuiKeyFlags(Button) | ShiftStateToGuiKeyFlags(Shift), X, Y );
  // Save this for Borland's "OnMouseMove()" handler (which doesn't pass in a "TMouseButton"):
  m_iMouseButtonAsGuiKeyFlags = MouseButtonToGuiKeyFlags( Button );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Img_SpectrumMouseMove(TObject *Sender, TShiftState Shift, int X, int Y)
{
  SpecDisp_HandleMouseEvent( &g_SpecDispControl, &MyCwNet.RigControl, GUI_EVENT_MOUSE_MOVE, GUI_CONTROL_SPECTRUM,
      m_iMouseButtonAsGuiKeyFlags | ShiftStateToGuiKeyFlags(Shift), X, Y );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Img_WaterfallMouseMove(TObject *Sender, TShiftState Shift, int X, int Y)
{
  SpecDisp_HandleMouseEvent( &g_SpecDispControl, &MyCwNet.RigControl, GUI_EVENT_MOUSE_MOVE, GUI_CONTROL_SPECTROGRAM,
      m_iMouseButtonAsGuiKeyFlags | ShiftStateToGuiKeyFlags(Shift), X, Y );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Img_SpectrumMouseUp(TObject *Sender,
      TMouseButton Button, TShiftState Shift, int X, int Y)
{
  BOOL fHandledEvent = SpecDisp_HandleMouseEvent( &g_SpecDispControl,
       &MyCwNet.RigControl, GUI_EVENT_MOUSE_UP, GUI_CONTROL_SPECTRUM,
       MouseButtonToGuiKeyFlags(Button) | ShiftStateToGuiKeyFlags(Shift), X, Y );
  m_iMouseButtonAsGuiKeyFlags = 0;     //

  if( ! fHandledEvent )
   { PM_Spectrum->Popup(Mouse->CursorPos.x, Mouse->CursorPos.y); // [in] g_SpecDispControl.dblClickedVfoFreq_Hz, etc (?)
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Img_WaterfallMouseUp(TObject *Sender,
      TMouseButton Button, TShiftState Shift, int X, int Y)
{ // Similar as Img_SpectrumMouseUp(), but for the WATERFALL aka Spectrogram,
  // and with subtle differences because the spectrogram also has a TIME AXIS.
  BOOL fHandledEvent = SpecDisp_HandleMouseEvent( &g_SpecDispControl,
       &MyCwNet.RigControl, GUI_EVENT_MOUSE_UP, GUI_CONTROL_SPECTROGRAM,
       MouseButtonToGuiKeyFlags(Button) | ShiftStateToGuiKeyFlags(Shift), X, Y );
  m_iMouseButtonAsGuiKeyFlags = 0;     //

  if( ! fHandledEvent )
   { PM_Waterfall->Popup(Mouse->CursorPos.x, Mouse->CursorPos.y);
     //            '--> [in] g_SpecDispControl.dblClickedVfoFreq_Hz, etc (?)
   }
}


//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::PM_SpectrumPopup(TObject *Sender)
{
  MI_Spectrum_SetVFO->Caption = AnsiString( TE("Set &VFO frequency to") )
    + FormatFloat( " #0.0000#", g_SpecDispControl.dblClickedVfoFreq_Hz*1e-6) + " MHz";
  MI_Spectrum_RefLevel->Caption = AnsiString( TE("Spectrum &Reference Level") )
    + " : " + FormatFloat( "#0.0", MyCwNet.RigControl.dblScopeRefLevel_dB) + " dB";
  MI_Spectrum_Span->Caption = "Spectrum Span in \"Center\" modes : "
    + FormatFloat( "#0.0", MyCwNet.RigControl.dblScopeSpan_Hz*1e-3) + " kHz";
  MI_SpectrumSpan_2500->Checked   = (MyCwNet.RigControl.dblScopeSpan_Hz == 2.5e3);
  MI_SpectrumSpan_5kHz->Checked   = (MyCwNet.RigControl.dblScopeSpan_Hz == 5e3);
  MI_SpectrumSpan_10kHz->Checked  = (MyCwNet.RigControl.dblScopeSpan_Hz == 10e3);
  MI_SpectrumSpan_25kHz->Checked  = (MyCwNet.RigControl.dblScopeSpan_Hz == 25e3);
  MI_SpectrumSpan_50kHz->Checked  = (MyCwNet.RigControl.dblScopeSpan_Hz == 50e3);
  MI_SpectrumSpan_100kHz->Checked = (MyCwNet.RigControl.dblScopeSpan_Hz == 100e3);
  MI_SpectrumSpan_250kHz->Checked = (MyCwNet.RigControl.dblScopeSpan_Hz == 250e3);
  MI_SpectrumSpan_500kHz->Checked = (MyCwNet.RigControl.dblScopeSpan_Hz == 500e3);

}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::PM_WaterfallPopup(TObject *Sender)
{
  MI_WFall_SetVFO->Caption = AnsiString( TE("Set &VFO frequency to") )
    + FormatFloat( " #0.0000#", g_SpecDispControl.dblClickedVfoFreq_Hz*1e-6) + " MHz";
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Spectrum_SetVFOClick(TObject *Sender)
{
  RigCtrl_SetVFOFrequency( &MyCwNet.RigControl, g_SpecDispControl.dblClickedVfoFreq_Hz );
  UpdateVFODisplay( g_SpecDispControl.dblClickedVfoFreq_Hz );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_WFall_SetVFOClick(TObject *Sender)
{ // Simular as MI_Spectrum_SetVFOClick(), but here for the WATERFALL,
  // possibly with subtle differences one fine day..
  RigCtrl_SetVFOFrequency( &MyCwNet.RigControl, g_SpecDispControl.dblClickedVfoFreq_Hz );
  UpdateVFODisplay( g_SpecDispControl.dblClickedVfoFreq_Hz );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Spectrum_SetPermanentMarkerClick(TObject *Sender)
{
  if( YHF_RunStringEditDialog(
       Handle,  // [in] handle to the window that will be blocked by the dialog
       (char*)TE("Define permanent frequency marker"), // [in] char *pszTitle,
       NULL,                     // [in] char *pszLabel1,
       NULL,                     // [in] char *pszLabel2,
       m_sz80LastMarkerText,     // [in] char *pszDefaultValue,
       YHF_EDIT_NORMAL,          // [in] int iEditOptions,
       (char*)m_sz80LastMarkerText, // [out] char *pszDestination,
       80,                       // [in] int iMaxLength,
       HELPID_FREQUENCY_MARKER)  // [in] int  iHelpID
     == YHF_DLG_BTN_OK ) // operator clicked "OK" or confirmed with "Enter" ->
   {
   }
}
//---------------------------------------------------------------------------

void __fastcall TKeyerMainForm::MI_Spectrum_RefLevelClick(TObject *Sender)
{
  double dblValue = MyCwNet.RigControl.dblScopeRefLevel_dB;
  char   sz80Label1[84], sz80Label2[84], *pszDest, *pszEndstop;
  pszDest    = sz80Label1;
  pszEndstop = sz80Label1+80;
  SL_AppendPrintf( &pszDest, pszEndstop, "%s: -20 .. +20 dB", TE("Valid range") );
  pszDest    = sz80Label2;
  pszEndstop = sz80Label2+80;
  SL_AppendPrintf( &pszDest, pszEndstop, "%s: %.1lf dB", TE("Current value"), dblValue );
  if( YHF_RunFloatInputDialog(
       Handle,  // [in] handle to the window that will be blocked by the dialog
       (char*)TE("Set Spectrum Reference Level"), // [in] char *pszTitle,
       sz80Label1, sz80Label2,   // [in] labels (info strings) above the edit field
       "%.1lf",                  // [in] char *pszFormatString,
       &dblValue,                // [in,out] double *pdblValue
       YHF_EDIT_NORMAL,          // [in] int iEditOptions,
       HELPID_SPECTRUM_REF_LEVEL)// [in] int  iHelpID
     == YHF_DLG_BTN_OK ) // operator clicked "OK" or confirmed with "Enter" ->
   { // Send the new 'spectrum reference level' to the radio (*)
     RigCtrl_SetScopeRefLevel_dB( &MyCwNet.RigControl, dblValue );
     // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
     // The new setting in MyCwNet.RigControl [here: RigControl.dblScopeRefLevel_dB]
     // will be compared with the 'last values' sent to a remote server/clients,
     // and if the value was MODIFIED, CwNet.c : CwNet_OnPoll() will send the
     // new setting also via the Remote CW Keyer's own TCP/IP connection.
     // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
   }
}
//---------------------------------------------------------------------------


void __fastcall TKeyerMainForm::Btn_MoreKeyerSettingsClick(TObject *Sender)
{
  UpdateItemsInSettingsMenu(); // before opening a submenu of the "Settings",
                               // update checkmarks, values, etc in the menu items.
  // Because users hardly saw the "More Keyer Settings" in the old-fashioned
  // main menu, try to open that particular sub-menu programmatically:
  // MI_MoreKeyerSettings->Click();   // <- of course THIS did not work (the first attempt)
  // Googled around, and found this at stackoverflow.com :
  // > As part of a built in tutorial system for my software I would like to
  // > programatically open the window menus of the software and show the user
  // > where to find certain functions. I've searched the Win32 API
  // > and can't find what I'm looking for. Is this possible via the API
  // > or do I need to provide some kind of workaround?
  // Answer:
  // > The submenu items of menubar or any other submenu (popu menu)
  // > can be displayed with TrackPopupMenu(Ex) API function. This function
  // > needs the menu handle of the submenu and coordinates at which to display it.
  // > For instance in order to display the File menu of menubar
  // > the following actions should be taken:
  // >    Get File submenu handle with GetSubMenu(0);
  // >    Get File menu item rectangle using GetMenuItemRect;
  // >    calculate the desired coordinates using the rectangle;
  // >    TrackPopupMenu(Ex) to display the menu.
  // "TrackPopupMenu()". A-ha. Who would have thought of that ? MS say:
  // > Displays a shortcut menu at the specified location and tracks
  // > the selection of items on the menu. The shortcut menu can appear
  // > anywhere on the screen.
  // A-ha. But "GetSubMenu()" expects a handle to the MENU (here: the entire main menu)
  //       and a "menu position", which seem to be a zero-based index into something.
  // What we have here (in a Borland VCL application) is the TMainMenu (and its handle)
  // and a "TMenuItem" (e.g. MI_MoreKeyerSettings) deeply buried in the TMainMenu.
  TrackPopupMenu( // <- funny-named Win32 API function to "open a certain menu item"
          // [in] HMENU hMenu : "A handle to the shortcut menu to be displayed" :
          GetSubMenu( Menu_Settings->Handle, Menu_Settings->IndexOf(MI_MoreKeyerSettings) ),
          //   ,-----------------------------|__________________________________________|
          //   '--> iMenuPos = "The zero-based relative position in the specified menu
          //                    of an item that activates a drop-down menu or submenu."
          // .. phew, fortunately Borland's VCL came to the rescue,
          //    and to the author's big surprise, this worked straight out of the box !
          TPM_LEFTALIGN, // [in] UINT uFlags : "Use zero of more of these flags to specify function options":
                         // TPM_CENTERALIGN, TPM_LEFTALIGN, TPM_RIGHTALIGN .
          Mouse->CursorPos.x, // [in] int x,
          Mouse->CursorPos.y, // [in] int y,
          0,       // [in] int nReserved,
          Handle,  // [in] HWND hWnd : "A handle to the window that owns the shortcut menu."
          NULL);   // [in, optional] const RECT *prcRect : "Ignored" (!).
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Btn_MoreHLServerOptionsClick(TObject *Sender)
{
  UpdateItemsInSettingsMenu(); // before opening a submenu of the "Settings",
                               // update checkmarks, values, etc in the menu items.
  TrackPopupMenu( // <- funny-named Win32 API function to "open a certain menu item"
     // [in] HMENU hMenu : "A handle to the shortcut menu to be displayed" :
     GetSubMenu( Menu_Settings->Handle, Menu_Settings->IndexOf(MI_HamlibServerOptions) ),
     // ,-------------------------------|____________________________________________|
     // '--> iMenuPos = "The zero-based relative position in the specified menu
     //                  of an item that activates a drop-down menu or submenu."
     TPM_LEFTALIGN, // [in] UINT uFlags : TPM_CENTERALIGN, TPM_LEFTALIGN, TPM_RIGHTALIGN .
     Mouse->CursorPos.x, Mouse->CursorPos.y, // [in] int x, y
     0,       // [in] int nReserved
     Handle,  // [in] HWND hWnd : "A handle to the window that owns the shortcut menu."
     NULL);   // [in, optional] const RECT *prcRect : "Ignored" (!).
}


//---------------------------------------------------------------------------
void TKeyerMainForm::SetNewLanguage( int iNewLanguage )
{
  if( iNewLanguage != APPL_iLanguage )
   { APPL_iPrevLanguage = APPL_iLanguage;
     APPL_iLanguage = iNewLanguage;
     APPL_TranslateAllForms(); // [in] APPL_iLanguage, APPL_iPrevLanguage
     APPL_fSaveSettingsOnExit = TRUE; // flag to call SaveSettings() when terminating
   }
} // end TKeyerMainForm::SetNewLanguage()

void __fastcall TKeyerMainForm::MI_Language_EnglishClick(TObject *Sender)
{ SetNewLanguage( TRANSLATOR_LANGUAGE_ENGLISH );
}
void __fastcall TKeyerMainForm::MI_Language_GermanClick(TObject *Sender)
{ SetNewLanguage( TRANSLATOR_LANGUAGE_GERMAN );
}
void __fastcall TKeyerMainForm::MI_Language_FrenchClick(TObject *Sender)
{ SetNewLanguage( TRANSLATOR_LANGUAGE_FRENCH );
}
void __fastcall TKeyerMainForm::MI_Language_DutchClick(TObject *Sender)
{ SetNewLanguage( TRANSLATOR_LANGUAGE_DUTCH );
}
void __fastcall TKeyerMainForm::MI_Language_ItalianClick(TObject *Sender)
{ SetNewLanguage( TRANSLATOR_LANGUAGE_ITALIAN );
}
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_UI_SimpleClick(TObject *Sender)
{ g_iUserInterfaceMode = UI_MODE_SIMPLE;
}
void __fastcall TKeyerMainForm::MI_UI_CompleteClick(TObject *Sender)
{ g_iUserInterfaceMode = UI_MODE_COMPLETE;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_ITU_Region_1Click(TObject *Sender)
{ RigCtrl_ITU_Region = 1;
}
void __fastcall TKeyerMainForm::MI_ITU_Region_2Click(TObject *Sender)
{ RigCtrl_ITU_Region = 2;
}
void __fastcall TKeyerMainForm::MI_ITU_Region_3Click(TObject *Sender)
{ RigCtrl_ITU_Region = 3;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Funcs_TrxPowerOffClick(TObject *Sender)
{ // Send a command to turn the radio off (as soon as possible),
  //  but don't interfere with a possibly pending command/response :
  MyCwNet.RigControl.iParameterPollingState = RIGCTRL_POLLSTATE_TURN_RIG_OFF;
     // '--> the rest happens in a state machine in RigCtrl_Handler() .
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Funcs_TrxPowerOnClick(TObject *Sender)
{ // Send a command to turn the radio ON (as soon as possible) ... :
  MyCwNet.RigControl.iParameterPollingState = RIGCTRL_POLLSTATE_TURN_RIG_ON;
}


//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_SpectrumSpan_2500Click(TObject *Sender)
{ RigCtrl_SetScopeSpan_Hz( &MyCwNet.RigControl, 2.5e3 );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_SpectrumSpan_5kHzClick(TObject *Sender)
{ RigCtrl_SetScopeSpan_Hz( &MyCwNet.RigControl, 5e3 );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_SpectrumSpan_10kHzClick(TObject *Sender)
{ RigCtrl_SetScopeSpan_Hz( &MyCwNet.RigControl, 10e3 );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_SpectrumSpan_25kHzClick(TObject *Sender)
{ RigCtrl_SetScopeSpan_Hz( &MyCwNet.RigControl, 25e3 );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_SpectrumSpan_50kHzClick( TObject *Sender)
{ RigCtrl_SetScopeSpan_Hz( &MyCwNet.RigControl, 50e3 );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_SpectrumSpan_100kHzClick(TObject *Sender)
{ RigCtrl_SetScopeSpan_Hz( &MyCwNet.RigControl, 100e3 );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_SpectrumSpan_250kHzClick(TObject *Sender)
{ RigCtrl_SetScopeSpan_Hz( &MyCwNet.RigControl, 250e3 );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_SpectrumSpan_500kHzClick(TObject *Sender)
{ RigCtrl_SetScopeSpan_Hz( &MyCwNet.RigControl, 500e3 );
}

//---------------------------------------------------------------------------

void __fastcall TKeyerMainForm::MI_Settings_SpectrumDisplayClick(TObject *Sender)
{
  // When invoked from the MAIN MENU, there is no 'clicked frequency' for PM_Spectrum,
  // so use this instead:
  g_SpecDispControl.dblClickedVfoFreq_Hz = g_SpecDispControl.dblDisplayedVfoFreq_Hz;
  PM_Spectrum->Popup(Mouse->CursorPos.x, Mouse->CursorPos.y); // [in] g_SpecDispControl.dblClickedVfoFreq_Hz, etc (?)
}
//---------------------------------------------------------------------------




