//---------------------------------------------------------------------------
// File: C:\cbproj\Remote_CW_Keyer\Keyer_Main.cpp  (original location for C++Builder 6),
//       C:\cbproj\Remote_CW_Keyer\BCB12_Project\Keyer_Main.cpp (copy for C++Builder 12,
//             with the main form (*.dfm) automatically modified by the new IDE,
//             making it incompatible with C++Builder V6...)
// Date: 2024-11-25
// Author: Wolfgang Buescher (DL4YHF)
// Purpose: Main "form" (window) for an experiment to key certain Icom transceivers.
//          Details about the "keying" itself in KeyerThread.c .
//      THIS module is just the BORing graphic user interface,
//      originally written in a stoneage version of Borland's C++ Builder (BCB V6),
//      using their easy-to-use VCL (Visual Component Library).
//
// Most recent modifications: See C:\cbproj\Remote_CW_Keyer\Keyer_GUI.cpp .
//
//---------------------------------------------------------------------------

#include "switches.h"   // project specific compiler switches ("options"),
                        // must be included before anything else !
#include "yhf_type.h"   // classic types like BYTE, WORD, DWORD, BOOL, ..
#include <windows.h>    // Must be included BEFORE vcl.h for some strange reason.
                        // Contains stuff like WAVEINCAPS, WAVEOUTCAPS, etc^255 .
#include <vcl.h>        // Borland's stoneage Visual Component Library
#include <buttons.hpp>  // TBitBtn       a la Borland VCL (but why in an extra header?)
#include <ComCtrls.hpp> // TTabSheet     a la Borland VCL ...
#include <CheckLst.hpp> // TCheckListBox a la Borland VCL ....
#include <Grids.hpp>    // TStringGrid   a la Borland VCL .....
#include <string.h>
#include <stdio.h>      // not using "standard I/O" here, but e.g. sprintf()
#include <math.h>
#pragma hdrstop

#include "Utilities.h"    // stuff like UTL_iWindowsVersion, UTL_iAppInstance, ShowError(), etc
#include "YHF_Dialogs.h"  // API for a few common dialogs, e.g. YHF_RunStringEditDialog()
#include "YHF_Help.h"     // DL4YHF's HTML-based replacement for the defunct *.hlp system
#include "YHF_WinStuff.h" // Win32 API functions, helpers for 'Rich Edit' controls, etc
// #include "RTF_Builder.h"  // DL4YHF's VCL-free "Rich Text Builder" (abandoned 2025-09-02, interferes with Borland's TRichEdit.Text)
#include "HelpIDs.h"      // application specific help topic identifiers
#include "Translator.h"   // APPL_TranslateAllForms() [only works with Borland VCL]
#include "Translations.h" // tranlation table (static, built-in string table w/o whistles and bells)

#if( SWI_USE_DSOUND ) // use "Direct Sound" / dsound_wrapper.c ?
# include "dsound_wrapper.h"
#endif // SWI_USE_DSOUND ?

#if( SWI_USE_WAVE_AUDIO || SWI_USE_MIDI )
# include <mmsystem.h> // without this, some stupid header didn't know what a 'LPCWAVEFORMATEX' is
#endif // SWI_USE_WAVE_AUDIO or SWI_USE_MIDI ?

#include "StringLib.h" // DL4YHF's "string library", uses C STRINGS ONLY, even runs on Microcontrollers
#include "Elbug.h"     // old 'Elbug' functions, converted from PIC-assembler to "C"
#include "SoundTab.h"  // cosine lookup table, filter coefficients, T_Float, etc.
#include "CwGen.h"     // CW generator (converts text to Morse code)
#include "CwDSP.h"     // CW-'Digital Signal Processor' / sidetone generator
#include "CwNet.h"     // Socket-based 'Client or Server' for the Remote CW Keyer
#include "CwKeyer.h"   // prototypes for the 'worker thread' that stitches all together
#include "Chatbox.h"   // simple 'text chat' feature to coordinate the use of the remote radio between users and sysop
#include "Inet_Tools.h" // Base64 encoding/decoding, SHA1 calculation, etc..
#if(SWI_USE_HTTP_SERVER) // build an application with integrated HTTP server ?
# include "HttpServer.h" // formerly simple HTTP server (turned into a monster)
# include "TIconToFavicon.h" // function to convert the application's icon
                             // into "Favicon.ico" for the web server .
#endif // SWI_USE_HTTP_SERVER ?

#include "SpecDisp.h"  // SpecDisp_UpdateSpectrum(), ..Waterfall(), FreqScale(),..
#include "FreqList.h"  // import 'frequencies of interest' from e.g. the EiBi frequency list

#if( SWI_NUM_AUX_COM_PORTS > 0 ) // compile with support for 'Auxiliary / Additional COM ports' / Winkeyer emulation ?
# include "AuxComPorts.h" // structs and API functions for the "Auxiliary" (later: "Additional") COM ports
# include "AuxComPortDetails_Dialog.h" // <- dialog to configure the 'Additional COM POrts' (in CwKeyer_Config)
#endif // SWI_NUM_AUX_COM_PORTS ?

#include "Keyer_GUI.h" // subroutines formerly in Keyer_Main.cpp (now only CALLED from there)

#include "Keyer_Main.h"
//---------------------------------------------------------------------------
#pragma package(smart_init)
#pragma resource "*.dfm"
#pragma warn -8004 // ".. assigned a value that is never used" (better than using an un-initialized variable)
TKeyerMainForm *KeyerMainForm;


// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
// Many variables and functions were moved from here (Keyer_Main.cpp / *.h)
// into Keyer_GUI.cpp / *.h, to avoid duplicating them in the *.cpp modules
//  of the 'main form' (TKeyerMainForm *KeyerMainForm), with one variant for
// Borland C++Builder V6, and another one for Embarcadero C++Builder V12 !
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


TForm *KeyerGUI_GetMainForm(void) // provides access to the application's MAIN FORM (without exposing its name)
{ return (TForm*)KeyerMainForm;
}


//---------------------------------------------------------------------------
// Implementation of the 'main form' itself (a Borland VCL thing).
//---------------------------------------------------------------------------

#define SYSMENUITEM_HIDE_MAIN_MENU 16  // menu items for "system menu" alias "control menu" alias "window menu".
#define SYSMENUITEM_SHOW_MAIN_MENU 32  // DON'T USE THE LOWER FOUR BITS HERE because windows is stupid
#define SYSMENUITEM_SETTINGS       48
#define SYSMENUITEM_FUNCTIONS      64
#define SYSMENUITEM_HELP           80


//---------------------------------------------------------------------------
__fastcall TKeyerMainForm::TKeyerMainForm(TComponent* Owner) : TForm(Owner)
{
  m_fInitialising = TRUE;
  KeyerGUI_iUpdating   = 1; // <- cleared when returning from TKeyerMainForm::FormCreate()
  KeyerGUI_iCurrentMainTab = KEYER_GUI_MAIN_TAB_UNKNOWN; // also contains one of the KEYER_GUI_MAIN_TAB_.. constants, set via SwitchMainTab()
  KeyerGUI_fAutomaticTabSwitching = TRUE; // allow *automatic* switching of the main tab until the 'rig initialisation' is finished.
  RichEd_ChatboxOnTrxTab = NULL;   // the "Chatbox" on the 'TRX' tab will be created dynamically "when needed".

  APPL_hInstance = HInstance; // handle to current instance, a Win32 thing.
     // > "HInstance contains the instance handle of the application
     // >  or library as provided by the Windows environment."
     // This "HInstance"-thingy, provided by the VCL, is required by certain
     // Win32 API function calls like CreateWindow(). We store it in an
     // old-fashioned global variable to make it accessable for NON-VCL modules,
     // e.g. YHF_Dialogs.cpp .
} // end TKeyerMainForm::TKeyerMainForm()

//---------------------------------------------------------------------------
int TKeyerMainForm::UpdateSettingsTab(void) // later, "I/O Config" and "Keyer Settings"..
  // Returns ZERO if there are no conflicting settings,
  //   otherwise a bitwise combination of KEYER_GUI_CONFLICT_ .. flags.
  // When clicking the 'Apply'-button, that result is used to decide
  // whether to switch away from the 'Settings'-tab or not.
{
  int i;
  int iResult = 0;
  BOOL fConflict;
  const char *pszHintForOtherSignalsOnTheSamePin = TE("Conflict: Other signals use the same pin on this port !");

  ++KeyerGUI_iUpdating; // prevent interfering in certain "OnClick()" and "OnChange()" handlers
  HERE_I_AM__GUI();
  FillComboWithSerialPorts( CB_MorseKeyPort, CwKeyer_Config.iComPortNumber_IN,
                            CheckForSignalsOnMorseKeyPort(), &fConflict,
                            TE( "KEYER port, see manual for 'Simple Paddle Adapter'" ) );
  if( fConflict )
   { iResult |= KEYER_GUI_CONFLICT_MORSE_KEY_PORT;
   }
  FillComboWithSerialPorts( CB_RadioKeyingPort, CwKeyer_Config.iRadioKeyingAndControlPort,
                            CheckForSignalsOnRadioKeyingPort(), &fConflict,
                            TE( "Radio Keying / Radio Control Port (serial)" ) );
  if( fConflict )
   { iResult |= KEYER_GUI_CONFLICT_RADIO_CONTROL_PORT;
   }
  FillComboWithItemsFromTokenLists( CB_DotKeyInput,
        C_TL_None_Zero,                // [in] const T_SL_TokenList *pTokens1 : "None" -> 0
        SerialPortInputSignals_Key,    // [in] const T_SL_TokenList *pTokens2
        SerialPortInputSignals_Radio,  // [in] const T_SL_TokenList *pTokens3
        CwKeyer_Config.iDotInput );    // [in] int iSelectedTokenValue
  if( CheckForSignalsOnSamePin( &CwKeyer_Config.iDotInput ) )
   { CB_DotKeyInput->Color = clRed;
     CB_DotKeyInput->Hint = pszHintForOtherSignalsOnTheSamePin;
     iResult |= KEYER_GUI_CONFLICT_DOT_INPUT;
   }
  else
   { CB_DotKeyInput->Color = (TColor)g_SpecDispControl.clWindowBackground;
     CB_DotKeyInput->Hint = TE( "e.g. for the 'Simple Paddle Adapter' : CTS, inverted" );
   }
  FillComboWithItemsFromTokenLists( CB_DashKeyInput,
        C_TL_None_Zero,                // [in] const T_SL_TokenList *pTokens1 : "None" -> 0
        SerialPortInputSignals_Key,    // [in] const T_SL_TokenList *pTokens2
        SerialPortInputSignals_Radio,  // [in] const T_SL_TokenList *pTokens3
        CwKeyer_Config.iDashInput );
  if( CheckForSignalsOnSamePin( &CwKeyer_Config.iDashInput ) )
   { CB_DashKeyInput->Color = clRed;
     CB_DashKeyInput->Hint = pszHintForOtherSignalsOnTheSamePin;
     iResult |= KEYER_GUI_CONFLICT_DASH_INPUT;
   }
  else
   { CB_DashKeyInput->Color = (TColor)g_SpecDispControl.clWindowBackground;
     CB_DashKeyInput->Hint = TE( "e.g. for the 'Simple Paddle Adapter' : DSR, inverted" );
   }
  FillComboWithItemsFromTokenLists( CB_PTT_Input,
        C_TL_None_Zero,                // [in] const T_SL_TokenList *pTokens1 : "None" -> 0
        SerialPortInputSignals_Key,    // [in] const T_SL_TokenList *pTokens2
        SerialPortInputSignals_Radio,  // [in] const T_SL_TokenList *pTokens3
        CwKeyer_Config.iManualPTTInput );
  if( CheckForSignalsOnSamePin( &CwKeyer_Config.iManualPTTInput ) )
   { CB_PTT_Input->Color = clRed;
     CB_PTT_Input->Hint  = pszHintForOtherSignalsOnTheSamePin;
     iResult |= KEYER_GUI_CONFLICT_PTT_INPUT;
   }
  else
   { CB_PTT_Input->Color = (TColor)g_SpecDispControl.clWindowBackground;
     CB_PTT_Input->Hint = TE( "typically 'NONE' to use the program's own Semi-BK feature in CW" );
   }
  FillComboWithItemsFromTokenLists( CB_TestInput,
        C_TL_None_Zero,                // [in] const T_SL_TokenList *pTokens1 : "None" -> 0
        SerialPortInputSignals_Key,    // [in] const T_SL_TokenList *pTokens2
        SerialPortInputSignals_Radio,  // [in] const T_SL_TokenList *pTokens3
        CwKeyer_Config.iTestInput );
  if( CheckForSignalsOnSamePin( &CwKeyer_Config.iTestInput ) )
   { CB_TestInput->Color = clRed;
     CB_TestInput->Hint  = pszHintForOtherSignalsOnTheSamePin;
     iResult |= KEYER_GUI_CONFLICT_TEST_INPUT;
   }
  else
   { CB_TestInput->Color = (TColor)g_SpecDispControl.clWindowBackground;
     CB_TestInput->Hint  = "";
   }
  FillComboWithItemsFromTokenLists( CB_InputKeySupply,
        C_TL_None_Zero,                // [in] const T_SL_TokenList *pTokens1 : "None" -> 0
        SerialPortOutputSignals_Key,   // [in] const T_SL_TokenList *pTokens2
        SerialPortOutputSignals_Radio, // [in] const T_SL_TokenList *pTokens3
        CwKeyer_Config.iKeySupply );
  if( CheckForSignalsOnSamePin( &CwKeyer_Config.iKeySupply ) )
   { CB_InputKeySupply->Color = clRed;
     CB_InputKeySupply->Hint  = pszHintForOtherSignalsOnTheSamePin;
     iResult |= KEYER_GUI_CONFLICT_KEY_INTERFACE_SUPPLY;
   }
  else
   { CB_InputKeySupply->Color = (TColor)g_SpecDispControl.clWindowBackground;
     CB_InputKeySupply->Hint  = TE( "Supply voltage for the 'Simple Paddle Adapter' : DTR" );
   }
  FillComboWithItemsFromTokenLists( CB_RadioInterfaceSupply,
        C_TL_None_Zero,                // [in] const T_SL_TokenList *pTokens1 : "None" -> 0
        SerialPortOutputSignals_Key,   // [in] const T_SL_TokenList *pTokens2
        SerialPortOutputSignals_Radio, // [in] const T_SL_TokenList *pTokens3
        CwKeyer_Config.iRadioSupply );
  if( CheckForSignalsOnSamePin( &CwKeyer_Config.iRadioSupply ) )
   { CB_RadioInterfaceSupply->Color = clRed;
     CB_RadioInterfaceSupply->Hint  = pszHintForOtherSignalsOnTheSamePin;
     iResult |= KEYER_GUI_CONFLICT_RADIO_INTERFACE_SUPPLY;
   }
  else
   { CB_RadioInterfaceSupply->Color = (TColor)g_SpecDispControl.clWindowBackground;
     CB_RadioInterfaceSupply->Hint  = TE( "Supply voltage for old CI-V interfaces : DTR or RTS");
   }
  FillComboWithItemsFromTokenLists( CB_RadioCWKeying,
        C_TL_None_Zero,                // [in] const T_SL_TokenList *pTokens1 : "None" -> 0
        SerialPortOutputSignals_Radio, // [in] const T_SL_TokenList *pTokens2
        SerialPortOutputSignals_Key,   // [in] const T_SL_TokenList *pTokens3
        CwKeyer_Config.iRadioCWKeying );
  if( CheckForSignalsOnSamePin( &CwKeyer_Config.iRadioCWKeying ) )
   { CB_RadioCWKeying->Color = clRed;
     CB_RadioCWKeying->Hint  = pszHintForOtherSignalsOnTheSamePin;
     iResult |= KEYER_GUI_CONFLICT_RADIO_CW_KEYING_OUTPUT;
   }
  else
   { CB_RadioCWKeying->Color = (TColor)g_SpecDispControl.clWindowBackground;
     CB_RadioCWKeying->Hint  = TE( "for Icom radios, typically DTR" );
   }

  FillComboWithItemsFromTokenLists( CB_RadioPTTControl,
        C_TL_None_Zero,                // [in] const T_SL_TokenList *pTokens1 : "None" -> 0
        SerialPortOutputSignals_Radio, // [in] const T_SL_TokenList *pTokens2
        SerialPortOutputSignals_Key,   // [in] const T_SL_TokenList *pTokens3
        CwKeyer_Config.iRadioPTTControl);
  if( CheckForSignalsOnSamePin( &CwKeyer_Config.iRadioPTTControl ) )
   { CB_RadioPTTControl->Color = clRed;
     CB_RadioPTTControl->Hint  = pszHintForOtherSignalsOnTheSamePin;
     iResult |= KEYER_GUI_CONFLICT_RADIO_PTT_OUTPUT;
   }
  else
   { CB_RadioPTTControl->Color = (TColor)g_SpecDispControl.clWindowBackground;
     CB_RadioPTTControl->Hint  = TE( "for modern radios, NONE to use the built-in Semi-BK feature" );
   }


  // The following controls were moved to an extra sub-tabsheet ("Remote Control")
  //   when the space on the 'I/O Config' tab got too small:
  FillComboWithItemsFromTokenLists( CB_RemoteControlMethod,
        RigControlMethods, // [in] const T_SL_TokenList *pTokens1 (RIGCTRL_PROTOCOL_NONE/RIGCTRL_PROTOCOL_ICOM_CI_V/RIGCTRL_PROTOCOL_HAMLIB_RIGCTLD,..(?) )
        NULL,              // [in] const T_SL_TokenList *pTokens2
        NULL,              // [in] const T_SL_TokenList *pTokens3
        CwKeyer_Config.iRadioControlProtocol); // .. aka "Remote Control Method" for RigControl.c
  // ex: SelectComboItemByDecimalValue( CB_RadioCtrlBaudrate, CwKeyer_Config.iRadioControlBaudrate );
  // For stoneage radios like FT-817 (controlled via "5-byte-command CAT"), we need 4800/9600/38400 baud, this:
  FillComboWithItemsFromTokenLists( CB_RadioCtrlBaudrate, SerialPortBaudrates, NULL, NULL, CwKeyer_Config.iRadioControlBaudrate );

  // For historic reasons (Icom), different radios are identified by their "default CI-V Address".
  // For non-Icom-radio, the same combo-box may contain dummy-addresses ABOVE 0xFF .
  // Only THREE ITEM INDICES are fixed, and hard-coded at the BEGIN of the list:
  //   [0] : None ("passive" control)
  //   [1] : Auto-Detect
  //   [2] : Non-Icom Radio
  AppendRadiosFromRigControlToCombo( CB_CIV_Address );
     // ,----------------------------------'
     // '--> Originally contained 'hard coded' items entered via "Object Inspector".
     //      More CI-V "default addresses" from RigControl.c : RigCtrl_RadioInfo_CIV[]
     //      may be added HERE.
  if(CwKeyer_Config.iRadioCIVAddress <= RIGCTRL_DEF_ADDR_NO_REMOTE_CTRL)  // negative "radio address" : no CI-V-controllable radio at all
   { CB_CIV_Address->ItemIndex = 0; // ItemIndex 0 = "Rig Control OFF"
   }
  else if( CwKeyer_Config.iRadioCIVAddress == RIGCTRL_DEF_ADDR_AUTO_DETECT ) // "radio address" ZERO : AUTO-DETECT (in RigControl.h :
   { CB_CIV_Address->ItemIndex = 1; // ItemIndex 1 = "AutoDetect" ..
   }
  else if( CwKeyer_Config.iRadioCIVAddress == RIGCTRL_DEF_ADDR_UNKNOWN_YAESU )
   { CB_CIV_Address->ItemIndex = 2; // ItemIndex 2 = "Non-Icom Radio" .. should also DISABLE 'Lab_RigCtrl_CIV_Addr'
     // (besides the "Unknown YAESU", there are a few dummy-'CI-V'-addresses like e.g. RIGCTRL_DEF_ADDR_YAESU_FT_817 .
     //  THEY appear with hexadecimal codes ABOVE 0xFF in the hard-coded list in CB_CIV_Address)
   }
  else
   { SelectComboItemByHexadecimalValue( CB_CIV_Address, CwKeyer_Config.iRadioCIVAddress );
     // '--> If this doesn't find a match with any existing item,
     //      it will simply SET THE COMBO BOX TEXT to the hexadecimal string
     //      (but not add an item to the DROP-DOWN LIST).
     // If all works as planned, a CI-V address like '0xB2' (entered manually
     // in a previous session) will be displayed as "0xB2 IC-7760" after adding
     // support it in e.g. RigControl.c : 
   }
#if( SWI_USE_HAMLIB_SERVER ) // not sure if this 'feature' is going to stay...
  Ed_BuiltInHamlibServerPort->Text = IntToStr( MyHamlibServer.cfg.iServerListeningPort );
  Chk_EnableBuiltInHamlibServer->Checked = CwKeyer_Config.fEnableBuiltInHamlibServer;
#endif // SWI_USE_HAMLIB_SERVER ?


  // Update yet another "sub-tabsheet" under "I/O Config":
  //  "Auxiliary COM Ports" tab, first intend for a Winkey emulation or -host..
  UpdateAuxComPortSettings();

  UpdateRigControlMethodDependingFields();

  // Moved to a new tabsheet titled "Keyer Settings" :
  HERE_I_AM__GUI();
  FillComboWithItemsFromTokenLists( CB_KeyType, KeyTypes, NULL, NULL, CwKeyer_Config.iMorseKeyType );
  UpdateWPMIndicatorFromCwKeyerConfig(); // [in] CwKeyer_Elbug.cfg.iDotTime_ms, [out] VCL controls

  Ed_TxDelayTime->Text = IntToStr( CwKeyer_Config.iTxDelayTime_ms );
  Ed_TxHangTime->Text  = IntToStr( CwKeyer_Config.iTxHangTime_ms );

  FillComboWithItemsFromTokenLists( CB_SidetoneOnTXD, SidetonesOnTXD, NULL, NULL, CwKeyer_Config.iSidetoneOnTXD );
  if( ! FillComboWithItemsFromTokenLists( CB_SidetoneOnAudioOut, SidetonesOnAudioOut, NULL, NULL, CwKeyer_DSP.cfg.iSidetoneFreq_Hz) )
   { CB_SidetoneOnAudioOut->Text = IntToStr(CwKeyer_DSP.cfg.iSidetoneFreq_Hz) + " Hz";
     // (this only works because in the form designer, this Combo-Box has the
     //  'csDropDown' style, not just 'csDropDownList' )
   }
  if( ! FillComboWithItemsFromTokenLists( CB_SidetoneRiseTime, SidetoneRiseTimes, NULL, NULL, CwKeyer_DSP.cfg.iSidetoneRiseTime_ms) )
   { CB_SidetoneRiseTime->Text = IntToStr(CwKeyer_DSP.cfg.iSidetoneRiseTime_ms) + " ms";
     // Technically, this isn't just a "rise" time but a "ramp" time.
     // But Icom coined the name (it's in their 'CW-KEY SET' menu).
   }
  Chk_Audio_AllowNetwork->Checked = (CwKeyer_DSP.cfg.iAudioFlags & DSP_AUDIO_FLAGS_ALLOW_NETWORK_AUDIO) != 0;
  Chk_Audio_DecodeCW->Checked     = (CwKeyer_DSP.cfg.iAudioFlags & DSP_AUDIO_FLAGS_DECODE_CW) != 0;
  Chk_Audio_SidetoneWhenKeyedOnTheRig->Checked = (CwKeyer_Config.dwAudioOptions & KEYER_AUDIO_OPTION_SIDETONE_WHEN_KEYED_ON_RIG) != 0;

  UTL_LimitInteger( &CwKeyer_DSP.cfg.iAudioInGain_dB, -20, 20 );
  SB_AudioInGain_dB->Position = -CwKeyer_DSP.cfg.iAudioInGain_dB;
  UTL_LimitInteger( &CwKeyer_DSP.cfg.iAudioOutGain_dB, -20, 20 );
  SB_AudioOutGain_dB->Position = -CwKeyer_DSP.cfg.iAudioOutGain_dB;
  UTL_LimitInteger( &CwKeyer_DSP.cfg.iSidetoneGain_dB, -20, 20 );
  SB_SidetoneGain_dB->Position = -CwKeyer_DSP.cfg.iSidetoneGain_dB; // invert, because "min" and "max" were reversed in the "TScrollBar"
  UTL_LimitInteger( &CwKeyer_DSP.cfg.iNetworkTonesGain_dB, -20, 20 );
  SB_NetworkTonesGain_dB->Position = -CwKeyer_DSP.cfg.iNetworkTonesGain_dB;

  --KeyerGUI_iUpdating;
  HERE_I_AM__GUI();

  return iResult;
} // end TKeyerMainForm::UpdateSettingsTab()


//---------------------------------------------------------------------------
void TKeyerMainForm::ApplySettingsTab(void)
{
  const char *pszSrc;
  CwKeyer_Config.iComPortNumber_IN = ParseSerialPortNumberFromText( AnsiString(CB_MorseKeyPort->Text).c_str() );
  CwKeyer_Config.iRadioKeyingAndControlPort= ParseSerialPortNumberFromText( AnsiString(CB_RadioKeyingPort->Text).c_str() );
  CwKeyer_Config.iRadioControlProtocol = SL_FindStringInTable( RigControlMethods, AnsiString(CB_RemoteControlMethod->Text).c_str() );
  CwKeyer_Config.iRadioControlBaudrate = StrToIntDef(CB_RadioCtrlBaudrate->Text, CwKeyer_Config.iRadioControlBaudrate );
  CwKeyer_Config.iRadioCIVAddress = GetCIVAddressFromComboBox( CB_CIV_Address );
  strncpy( CwKeyer_Config.sz80RemoteRigCtrlServerAddress, (const char*)Ed_RemoteRigctrlServerAddress->Text.c_str(), 80 );

#if( SWI_USE_HAMLIB_SERVER ) // not sure if this 'feature' is going to stay...
  MyHamlibServer.cfg.iServerListeningPort = StrToIntDef( Ed_BuiltInHamlibServerPort->Text, HLSRV_DEFAULT_SERVER_PORT );
  CwKeyer_Config.fEnableBuiltInHamlibServer= Chk_EnableBuiltInHamlibServer->Checked;
#endif // SWI_USE_HAMLIB_SERVER ?

  CwKeyer_Config.iDotInput       = GetSerialPortSignalIndexFromComboBox( CB_DotKeyInput    );
  CwKeyer_Config.iDashInput      = GetSerialPortSignalIndexFromComboBox( CB_DashKeyInput   );
  CwKeyer_Config.iManualPTTInput = GetSerialPortSignalIndexFromComboBox( CB_PTT_Input      );
  CwKeyer_Config.iTestInput      = GetSerialPortSignalIndexFromComboBox( CB_TestInput      );
  CwKeyer_Config.iKeySupply      = GetSerialPortSignalIndexFromComboBox( CB_InputKeySupply );
  CwKeyer_Config.iRadioSupply    = GetSerialPortSignalIndexFromComboBox( CB_RadioInterfaceSupply );
  CwKeyer_Config.iRadioCWKeying  = GetSerialPortSignalIndexFromComboBox( CB_RadioCWKeying  );
  CwKeyer_Config.iRadioPTTControl= GetSerialPortSignalIndexFromComboBox( CB_RadioPTTControl);

  // Apply settings from yet another "sub-tabsheet" under "I/O Config":
  //  "Auxiliary COM Ports" tab, first intend for a Winkey emulation or -host..
  ApplyAuxComPortSettings();

  // Some of those new settings in CwKeyer_Config must be passed on to other modules:
  RigCtrl_SetRadioControlProtocol( &MyCwNet.RigControl, CwKeyer_Config.iRadioControlProtocol );

} // end TKeyerMainForm::ApplySettingsTab()

//---------------------------------------------------------------------------
void TKeyerMainForm::UpdateWPMIndicatorFromCwKeyerConfig(void)
  // [in]  CwKeyer_Elbug.cfg.iDotTime_ms, [out] various(!) VCL controls .
  // [out] KeyerGUI_fUpdateWPMIndicator : will be CLEARED here when 'done'.
  // Since 2025-06-14, this may be called anytime when the "CW Speed"
  // has been modified by an external application,
  //  e.g. N1MM Logger+ via the 'Winkeyer EMULATOR' on an 'Additional COM Port' !
{
  // The INTERNAL unit for the elbug timing is MILLISECONDS PER DOT.
  // Only for the "GUI", the speed is displayed in "Words per Minute" .
  // With a thread loop time of 2 milliseconds, speeds above 48 WPM are possible,
  // but not ALL. For example, 59 WPM = 1200 ms / 59 = 20.33 ms;
  //                           60 WPM = 1200 ms / 60 = 20.00 ms.
  int i = Elbug_DotTimeInMillisecondsToWordsPerMinute( CwKeyer_Elbug.cfg.iDotTime_ms );
  TrackBar_WPM->Position = i;   // Old-school "track bar" (a la Borland VCL), with up/down via keyboard
  Ed_WPM->Text = IntToStr( i ); // NUMERIC edit field for the speed in WPM
  KeyerGUI_fUpdateWPMIndicator = FALSE; // "done" (request for update not pending anymore)
} // end TKeyerMainForm::UpdateWPMIndicatorFromCwKeyerConfig()

//---------------------------------------------------------------------------
int TKeyerMainForm::UpdateAuxComPortSettings(void)
  // Returns ZERO if there are no conflicting settings,
  //   otherwise a bitwise combination of KEYER_GUI_CONFLICT_ .. flags.
{
  char sz80[84];
  BOOL fConflict;
  int  iResult = 0;
#if( SWI_NUM_AUX_COM_PORTS >= 1 )
  FillComboWithSerialPorts( CB_AuxComPort1,  CwKeyer_Config.sAuxCom[0].iPortNumber,
        (CwKeyer_Config.sAuxCom[0].iPortUsage != RIGCTRL_PORT_USAGE_NONE), // fNeedValidPort ?
        &fConflict,
        TE("Optional serial port for multiple purposes (purpose selected below)") );
  if( fConflict )
   { iResult |= KEYER_GUI_CONFLICT_AUX_COM_PORT_1;
   }
  FillComboWithItemsFromTokenLists( CB_Aux1Baud, SerialPortBaudrates, NULL, NULL, CwKeyer_Config.sAuxCom[0].iBitsPerSecond );
  FillComboWithItemsFromTokenLists( CB_AuxComPort1Usage,
        AuxComPortUsages, // [in] const T_SL_TokenList *pTokens1 (RIGCTRL_PROTOCOL_NONE/RIGCTRL_PROTOCOL_ICOM_CI_V/RIGCTRL_PROTOCOL_HAMLIB_RIGCTLD,..(?) )
        NULL,              // [in] const T_SL_TokenList *pTokens2
        NULL,              // [in] const T_SL_TokenList *pTokens3
        CwKeyer_Config.sAuxCom[0].iPortUsage );
  Lab_AuxCom1Info->Caption = AuxCom_GetCurrentStatusAsString( &AuxComPorts[0] );
#endif // SWI_NUM_AUX_COM_PORTS >= .. ?
#if( SWI_NUM_AUX_COM_PORTS >= 2 )
  FillComboWithSerialPorts( CB_AuxComPort2,  CwKeyer_Config.sAuxCom[1].iPortNumber,
        (CwKeyer_Config.sAuxCom[1].iPortUsage != RIGCTRL_PORT_USAGE_NONE), // fNeedValidPort ?
        &fConflict,
        TE("Optional serial port for multiple purposes (purpose selected below)") );
  if( fConflict )
   { iResult |= KEYER_GUI_CONFLICT_AUX_COM_PORT_2;
   }
  FillComboWithItemsFromTokenLists( CB_Aux2Baud, SerialPortBaudrates, NULL, NULL, CwKeyer_Config.sAuxCom[1].iBitsPerSecond );
  FillComboWithItemsFromTokenLists( CB_AuxComPort2Usage,
        AuxComPortUsages, // [in] const T_SL_TokenList *pTokens1 (RIGCTRL_PROTOCOL_NONE/RIGCTRL_PROTOCOL_ICOM_CI_V/RIGCTRL_PROTOCOL_HAMLIB_RIGCTLD,..(?) )
        NULL,              // [in] const T_SL_TokenList *pTokens2
        NULL,              // [in] const T_SL_TokenList *pTokens3
        CwKeyer_Config.sAuxCom[1].iPortUsage );
  Lab_AuxCom2Info->Caption = AuxCom_GetCurrentStatusAsString( &AuxComPorts[1] );
#endif // SWI_NUM_AUX_COM_PORTS >= .. ?
#if( SWI_NUM_AUX_COM_PORTS >= 3 )
  FillComboWithSerialPorts( CB_AuxComPort3,  CwKeyer_Config.sAuxCom[2].iPortNumber,
        (CwKeyer_Config.sAuxCom[2].iPortUsage != RIGCTRL_PORT_USAGE_NONE), // fNeedValidPort ?
        &fConflict,
        TE("Optional serial port for multiple purposes (purpose selected below)") );
  if( fConflict )
   { iResult |= KEYER_GUI_CONFLICT_AUX_COM_PORT_3;
   }
  FillComboWithItemsFromTokenLists( CB_Aux3Baud, SerialPortBaudrates, NULL, NULL, CwKeyer_Config.sAuxCom[2].iBitsPerSecond );
  FillComboWithItemsFromTokenLists( CB_AuxComPort3Usage,
        AuxComPortUsages, // [in] const T_SL_TokenList *pTokens1 (RIGCTRL_PROTOCOL_NONE/RIGCTRL_PROTOCOL_ICOM_CI_V/RIGCTRL_PROTOCOL_HAMLIB_RIGCTLD,..(?) )
        NULL,              // [in] const T_SL_TokenList *pTokens2
        NULL,              // [in] const T_SL_TokenList *pTokens3
        CwKeyer_Config.sAuxCom[2].iPortUsage );
  Lab_AuxCom3Info->Caption = AuxCom_GetCurrentStatusAsString( &AuxComPorts[2] );
#endif // SWI_NUM_AUX_COM_PORTS >= .. ?
  return iResult;
} // end UpdateAuxComPortSettings()

//---------------------------------------------------------------------------
void TKeyerMainForm::ApplyAuxComPortSettings(void)
{
#if( SWI_NUM_AUX_COM_PORTS >= 1 )
  CwKeyer_Config.sAuxCom[0].iPortNumber = ParseSerialPortNumberFromText( AnsiString(CB_AuxComPort1->Text).c_str() );
  CwKeyer_Config.sAuxCom[0].iBitsPerSecond = StrToIntDef(CB_Aux1Baud->Text, CwKeyer_Config.sAuxCom[0].iBitsPerSecond );
  CwKeyer_Config.sAuxCom[0].iPortUsage = SL_FindStringInTable( AuxComPortUsages, AnsiString(CB_AuxComPort1Usage->Text).c_str() );
#endif // SWI_NUM_AUX_COM_PORTS >= .. ?
#if( SWI_NUM_AUX_COM_PORTS >= 2 )
  CwKeyer_Config.sAuxCom[1].iPortNumber = ParseSerialPortNumberFromText( AnsiString(CB_AuxComPort2->Text).c_str() );
  CwKeyer_Config.sAuxCom[1].iBitsPerSecond = StrToIntDef(CB_Aux2Baud->Text, CwKeyer_Config.sAuxCom[1].iBitsPerSecond );
  CwKeyer_Config.sAuxCom[1].iPortUsage = SL_FindStringInTable( AuxComPortUsages, AnsiString(CB_AuxComPort2Usage->Text).c_str() );
#endif // SWI_NUM_AUX_COM_PORTS >= .. ?
#if( SWI_NUM_AUX_COM_PORTS >= 3 )
  CwKeyer_Config.sAuxCom[2].iPortNumber = ParseSerialPortNumberFromText( AnsiString(CB_AuxComPort3->Text).c_str() );
  CwKeyer_Config.sAuxCom[2].iBitsPerSecond = StrToIntDef(CB_Aux3Baud->Text, CwKeyer_Config.sAuxCom[2].iBitsPerSecond );
  CwKeyer_Config.sAuxCom[2].iPortUsage = SL_FindStringInTable( AuxComPortUsages, AnsiString(CB_AuxComPort3Usage->Text).c_str() );
#endif // SWI_NUM_AUX_COM_PORTS >= .. ?
} // end ApplyAuxComPortSettings()

//---------------------------------------------------------------------------
void TKeyerMainForm::UpdateRigControlMethodDependingFields(void)
{ AnsiString s = CB_RemoteControlMethod->Text;
  // In BCB V12 "Athens", almost anything is a "UnicodeString", and that beast's
  //  c_str() method ALWAYS returns a wchar_t, not a 'char *' !
  int iNewRigControlMethod = SL_FindStringInTable( RigControlMethods, s.c_str() );
  int iNewCIVAddress = GetCIVAddressFromComboBox( CB_CIV_Address );

  TColor clCIVAddressSelection = (TColor)g_SpecDispControl.clWindowBackground; // ex: clWindow;
  switch( iNewRigControlMethod )
   { case RIGCTRL_PROTOCOL_NONE:
        Lab_RigCtrl_CIV_Addr->Enabled       = FALSE;
        Lab_RemoteRigControlServer->Enabled = FALSE;
        break;

     case RIGCTRL_PROTOCOL_ICOM_CI_V:
        Lab_RigCtrl_CIV_Addr->Enabled       = TRUE;
        Lab_RemoteRigControlServer->Enabled = FALSE;
        if( (iNewCIVAddress != RIGCTRL_DEF_ADDR_AUTO_DETECT)
         &&((iNewCIVAddress < RIGCTRL_DEF_ADDR_ICOM_LOWEST) || (iNewCIVAddress > RIGCTRL_DEF_ADDR_ICOM_HIGHEST) )
          )
         {  clCIVAddressSelection = clRed;
         }
        break;

     case RIGCTRL_PROTOCOL_HAMLIB_RIGCTLD:
     // ex: case RIGCTRL_PROTOCOL_FLRIG_XMLRPC: // removed, far too complex
        Lab_RigCtrl_CIV_Addr->Enabled       = FALSE;
        Lab_RemoteRigControlServer->Enabled = TRUE;
        break;

     case RIGCTRL_PROTOCOL_YAESU_5_BYTE :
        Lab_RigCtrl_CIV_Addr->Enabled       = FALSE;
        Lab_RemoteRigControlServer->Enabled = TRUE;
        if( (iNewCIVAddress != RIGCTRL_DEF_ADDR_AUTO_DETECT)
         &&((iNewCIVAddress < RIGCTRL_DEF_ADDR_YAESU5B_LOWEST) || (iNewCIVAddress > RIGCTRL_DEF_ADDR_YAESU5B_HIGHEST) )
          )
         {  clCIVAddressSelection = clRed;
         }
        break;
   }
  CB_CIV_Address->Color = clCIVAddressSelection;
} // end TKeyerMainForm::UpdateRigControlMethodDependingFields()

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::CB_RemoteControlMethodClick(TObject *Sender)
{ // Depending on the Rig-Control "Method / Protocol",
  // several visual controls on the 'Config' / 'Rig Control' tab
  // may be disabled of enabled. For example, if the rig is controlled
  // via serial port ("COM port") and CI-V protocol, the 'CI-V Address'
  // is important.
  // If the rig is controlled via an external application
  // that provides a Hamlib/Rigctld (d for 'daemon' ?) compatible server,
  // the server's URL with optional numeric IP address and remote TCP port
  // is important.
  if( KeyerGUI_iUpdating==0 )
   { UpdateRigControlMethodDependingFields();
   }  // end if( KeyerGUI_iUpdating==0 )
}
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
void TKeyerMainForm::UpdateMultiFunctionMeters(void) // [out] Img_TRX_Bottom
{
  TImage *pImg = Img_TRX_Bottom;
  T_CwNet *pCwNet = &MyCwNet;   // [in] MyCwNet.RigControl.iPowerMeterLevel_pcnt, etc

  int iSingleMeterWidth = pImg->Width / 3;
  int iSingleMeterHeight= pImg->Height / 2;

  if( RichEd_ChatboxOnTrxTab != NULL )
   { RichEd_ChatboxOnTrxTab->Visible = false;
     Img_TRX_Bottom->Visible = true;
   }

  // Take "snapshots" of the various "meter" readings displayed in the GUI:
  m_iDisplayedPowerMeterLevel_pcnt = pCwNet->RigControl.iPowerMeterLevel_pcnt;
  m_dblDisplayedSWRMeterValue      = pCwNet->RigControl.dblSWRMeterValue;  // dimensionless; usually ranging from 1.0 to 3.0 (from IC-7300)
  m_iDisplayedALCMeterLevel_pcnt   = pCwNet->RigControl.iALCMeterLevel_pcnt;
  m_iDisplayedCompMeterLevel_dB    = pCwNet->RigControl.iCompMeterLevel_dB;
  m_iDisplayedSupplyVoltage_mV     = pCwNet->RigControl.iSupplyVoltage_mV;
  m_iDisplayedDrainCurrent_mA      = pCwNet->RigControl.iDrainCurrent_mA;
  if( pCwNet->RigControl.iPATemperature_degC != RIGCTRL_NOVALUE_INT ) // <- one fine day we know how to read this via CI-V ..
   { m_iDisplayediPATemperature_degC = pCwNet->RigControl.iPATemperature_degC;
   }

  // First meter (top left) : TX POWER. Not in WATTS but in "percent of the maximum" (thanks Icom):
  KeyerGUI_UpdateMultiFunctionMeter( pImg,
     0/*x1*/, 0/*y1*/, iSingleMeterWidth/*x2*/, iSingleMeterHeight/*y2*/, // <- graphic area for one of the six(?) "meters" within pImg
     (double)m_iDisplayedPowerMeterLevel_pcnt, // [in] value to be displayed
     0.0/*dblMinValue*/, 100.0/*dblMaxValue*/, 10.0/*dblStepwidth*/, // [in] physical value range (also used for the labelled scale)
     "Power",  // [in] char *pszParamName, e.g. "Power", "ALC", "COMP", "SWR", "Id", "Vd", "Temp"
     "%",      // [in] char *pszPhysUnit, physical unit (shown at the end of the tick scale), e.g. "%", "dB", "V", "A", "W", "C"
     0.0/*dblRedStartValue*/, 0.0/*dblRedEndValue*/ ); // e.g. 10.0 .. 12.0 [range for an optional "red line" indicating a critical range]

  // Second meter (top middle) : SWR meter. "Right next to the power meter, that's where it belongs !"
  KeyerGUI_UpdateMultiFunctionMeter( pImg,
     iSingleMeterWidth/*x1*/, 0/*y1*/, 2*iSingleMeterWidth/*x2*/, iSingleMeterHeight/*y2*/, // <- graphic area for one of the six(?) "meters" within pImg
     m_dblDisplayedSWRMeterValue, // [in] value to be displayed (here: dimensionless SWR, typical range 1.0 .. 3.0)
     1.0/*dblMinValue*/, 4.0/*dblMaxValue*/, 0.5/*dblStepwidth*/, // [in] physical value range (also used for the labelled scale)
     "SWR",  // [in] char *pszParamName, e.g. "Power", "ALC", "COMP", "SWR", "Id", "Vd", "Temp"
     "",     // [in] char *pszPhysUnit, physical unit (shown at the end of the tick scale), e.g. "%", "dB", "V", "A", "W", "C"
     3.0/*dblRedStartValue*/, 4.0/*dblRedEndValue*/ ); // range for an optional "red line" indicating a critical range]

  // Third meter (top right) : ALC meter.
  KeyerGUI_UpdateMultiFunctionMeter( pImg,
     2*iSingleMeterWidth/*x1*/, 0/*y1*/, pImg->Width/*x2*/, iSingleMeterHeight/*y2*/, // <- graphic area for one of the six(?) "meters" within pImg
     (double)m_iDisplayedALCMeterLevel_pcnt, // [in] value to be displayed
     0.0/*dblMinValue*/, 100.0/*dblMaxValue*/, 10.0/*dblStepwidth*/, // [in] physical value range (also used for the labelled scale)
     "ALC",  // [in] char *pszParamName, e.g. "Power", "ALC", "COMP", "SWR", "Id", "Vd", "Temp"
     "",     // [in] char *pszPhysUnit, physical unit (shown at the end of the tick scale), e.g. "%", "dB", "V", "A", "W", "C"
     0.0/*dblRedStartValue*/, 50.0/*dblRedEndValue*/ ); // range for an optional "red line" indicating WHAT ?

  // 4th meter (bottom left) : Drain current. To make it more interesting, Icom uses AMPERES here, not PERCENT. IC-7300: Range 0 .. 25 A.
  KeyerGUI_UpdateMultiFunctionMeter( pImg,
     0/*x1*/, iSingleMeterHeight/*y1*/, iSingleMeterWidth/*x2*/, 2*iSingleMeterHeight/*y2*/, // <- graphic area for one of the six(?) "meters" within pImg
     (double)m_iDisplayedDrainCurrent_mA * 1e-3, // [in] value to be displayed
     0.0/*dblMinValue*/, 25.0/*dblMaxValue*/, 5.0/*dblStepwidth*/, // [in] physical value range (also used for the labelled scale)
     "Id",  // [in] char *pszParamName, e.g. "Power", "ALC", "COMP", "SWR", "Id", "Vd", "Temp"
     "A",   // [in] char *pszPhysUnit, physical unit (shown at the end of the tick scale), e.g. "%", "dB", "V", "A", "W", "C"
     0.0/*dblRedStartValue*/, 0.0/*dblRedEndValue*/ ); // e.g. 10.0 .. 12.0 [range for an optional "red line" indicating a critical range]

  // 5th meter (bottom middle) : DC Drain voltage ~~ Supply voltage. Icom uses VOLTS here, no stupid PERCENT. IC-7300: Range 10 .. 16 V.
  KeyerGUI_UpdateMultiFunctionMeter( pImg,
     iSingleMeterWidth/*x1*/, iSingleMeterHeight/*y1*/, 2*iSingleMeterWidth/*x2*/, 2*iSingleMeterHeight/*y2*/, // <- graphic area for one of the six(?) "meters" within pImg
     (double)m_iDisplayedSupplyVoltage_mV * 1e-3, // [in] value to be displayed
     10.0/*dblMinValue*/, 16.0/*dblMaxValue*/, 1.0/*dblStepwidth*/, // [in] physical value range (also used for the labelled scale)
     "Vd",  // [in] char *pszParamName, e.g. "Power", "ALC", "COMP", "SWR", "Id", "Vd", "Temp"
     "V",   // [in] char *pszPhysUnit, physical unit (shown at the end of the tick scale), e.g. "%", "dB", "V", "A", "W", "C"
     10.0/*dblRedStartValue*/, 12.0/*dblRedEndValue*/ ); // e.g. 10.0 .. 12.0 [range for an optional "red line" indicating a critical range]

  // 6th meter (bottom right) : Power Amplifier Temperature.
  // Unfortunately, Icom forgot to make this accessable via CI-V, or forgot to document it.
  // So far, this important parameter (for remote TX) is only available "locally", on the radio itself.
  KeyerGUI_UpdateMultiFunctionMeter( pImg,
     2*iSingleMeterWidth/*x1*/, iSingleMeterHeight/*y1*/, pImg->Width/*x2*/, 2*iSingleMeterHeight/*y2*/, // <- graphic area for one of the six(?) "meters" within pImg
     (double)m_iDisplayediPATemperature_degC, // [in] value to be displayed
     0.0/*dblMinValue*/, 100.0/*dblMaxValue*/, 20.0/*dblStepwidth*/, // [in] physical value range (also used for the labelled scale)
     "Temp", // [in] char *pszParamName, e.g. "Power", "ALC", "COMP", "SWR", "Id", "Vd", "Temp"
     "C",   // [in] char *pszPhysUnit, physical unit (shown at the end of the tick scale)
     80.0/*dblRedStartValue*/, 100.0/*dblRedEndValue*/ ); // [in] range for an optional "red line" indicating a critical range

} // end TKeyerMainForm::UpdateMultiFunctionMeters()

//---------------------------------------------------------------------------
void TKeyerMainForm::UpdateFrequencyInfo(void) // [out] Img_TRX_Bottom
  // (same bitmap as used by UpdateMultiFunctionMeters(), but different content:
  //  Here, shows additional information about bands (frequency ranges),
  //  different "frequencies of interest", and maybe even a web-SDR-like
  //  passband display)
{

  // For the NCDXF/IARU "International Beacon Project", the FREQUENCY INFO
  // may have to be updated every 10 seconds, shortly AFTER the begin of a
  // new 10-second interval, synchronized to the Unix time (and thus UTC).
  // See implementation of SpecDisp_ReplaceStationNameByNCDXFBeaconSchedule() :
  double dblUnixDateAndTime = UTL_GetCurrentUnixDateAndTime(); // .. in SECONDS, because that's the one-and-only SI unit for time
  int iTimeSlot1 = (int)fmodl( dblUnixDateAndTime * 0.1, 18.0 ); // -> 0..17 (per 10-second interval)
  int iTimeSlot2 = (int)fmodl( g_SpecDispControl.dblUnixDateAndTimeOfFreqInfoUpdate * 0.1, 18.0 );
  if(  g_SpecDispControl.fUpdateFrequencyInfo || (iTimeSlot1 != iTimeSlot2)
   || (g_SpecDispControl.fmin_on_FreqInfo != g_SpecDispControl.fmin_on_FreqScale)
   || (g_SpecDispControl.fmax_on_FreqInfo != g_SpecDispControl.fmax_on_FreqScale)
    )
   { g_SpecDispControl.dblUnixDateAndTimeOfFreqInfoUpdate = dblUnixDateAndTime;
     if( RichEd_ChatboxOnTrxTab != NULL )
      { RichEd_ChatboxOnTrxTab->Visible = false;
        Img_TRX_Bottom->Visible = true;
      }
     SpecDisp_UpdateFrequencyInfo(
        Img_TRX_Bottom->Picture->Bitmap, // [out] TBitmap a la Borland VCL
        &g_SpecDispControl, &MyCwNet.RigControl,
        0,0,Img_TRX_Bottom->Width-1, Img_TRX_Bottom->Height-1 ); // [in] x1, y1, x2, y2
   }
} // end TKeyerMainForm::UpdateFrequencyInfo()

//---------------------------------------------------------------------------
void TKeyerMainForm::CreateChatboxOnTrxTab(void) // Purpose: See UpdateChatboxOnTrxTab() ...
{
  // Implementation details: The "Chat Box" is another Rich Text edit control (RichEd_ChatboxOnTrxTab),
  //      but -unlike other VCL objects here- it's not contained in THE FORM ITSELF,
  //      but created 'dynamically' as CHILD 'Pnl_TRX_Bottom', overlaying 'Img_TRX_Bottom'
  //      in the original VCL FORM displayed AS TEXT below:
  // >       object Pnl_TRX_Bottom: TPanel // <- hard to see in the FORM DESIGNER at all !
  // >        Left = 0
  // >        Top = 301
  // >        Width = 574
  // >        Height = 77
  // >        Align = alBottom
  // >        TabOrder = 1
  // >        DesignSize = (
  // >          574
  // >          77)
  // >        object Img_TRX_Bottom: TImage
  // >          Left = 0
  // >          Top = 0
  // >          Width = 574
  // >          Height = 77
  // >          Anchors = [akLeft, akTop, akRight, akBottom]
  // >          OnMouseDown = Img_TRX_BottomMouseDown
  // >          OnMouseMove = Img_TRX_BottomMouseMove
  // >          OnMouseUp = Img_TRX_BottomMouseUp
  // >        end // Img_TRX_Bottom
  // >      end // Pnl_TRX_Bottom

  if( RichEd_ChatboxOnTrxTab == NULL )
   {  RichEd_ChatboxOnTrxTab = new TRichEdit(this);
   }
  if( RichEd_ChatboxOnTrxTab != NULL )
   { RichEd_ChatboxOnTrxTab->Parent = Pnl_TRX_Bottom;
     RichEd_ChatboxOnTrxTab->Name   = "RichEd_ChatboxOnTrxTab";
     // So far, only the dynamically created RichEdit's PARENT (aka "AOwner"), Pnl_TRX_Bottom,
     // has been set. Many of the other *VISUAL* Component Object's properties
     // are set below, to match those of the statically created Img_TRX_Bottom shown above:
     RichEd_ChatboxOnTrxTab->Left   = 0;
     RichEd_ChatboxOnTrxTab->Top    = 0;
     RichEd_ChatboxOnTrxTab->Width  = Pnl_TRX_Bottom->Width;
     RichEd_ChatboxOnTrxTab->Height = Pnl_TRX_Bottom->Height;
#   if(0)
     RichEd_ChatboxOnTrxTab->Anchors = [akLeft, akTop, akRight, akBottom];
     // We're using C++, which is an obfuscated language. Not the clear syntax
     // of Borland's "Form Designer" / Text DFM. For example, TRichEdit.Anchors
     // is a "Set" in the C++ meaning, so:
#   else
     RichEd_ChatboxOnTrxTab->Anchors << akLeft << akTop << akRight << akBottom;
#   endif
     Img_TRX_Bottom->Visible = FALSE;
     RichEd_ChatboxOnTrxTab->Visible = TRUE;
     RichEd_ChatboxOnTrxTab->Align   = alClient;
     RichEd_ChatboxOnTrxTab->Alignment = taLeftJustify;
     RichEd_ChatboxOnTrxTab->Color   = (TColor)g_SpecDispControl.clWindowBackground;
     RichEd_ChatboxOnTrxTab->Cursor  = crDefault;
     RichEd_ChatboxOnTrxTab->DragCursor= crDrag;
     RichEd_ChatboxOnTrxTab->DragKind= dkDrag;
     RichEd_ChatboxOnTrxTab->DragMode= dmManual;
     RichEd_ChatboxOnTrxTab->Enabled = true;
     RichEd_ChatboxOnTrxTab->Font->Color= (TColor)g_SpecDispControl.clWindowText;
     RichEd_ChatboxOnTrxTab->Font->Name = "Courier New";
     RichEd_ChatboxOnTrxTab->Font->Size = 12;
     RichEd_ChatboxOnTrxTab->HideScrollBars = true;
     RichEd_ChatboxOnTrxTab->HideSelection  = true;
     RichEd_ChatboxOnTrxTab->PopupMenu  = PM_ChatBox;
     RichEd_ChatboxOnTrxTab->ReadOnly   = false;
     RichEd_ChatboxOnTrxTab->ScrollBars = ssBoth;
     RichEd_ChatboxOnTrxTab->WantReturns= true;  // <- same as for Ed_ErrorHistory:
     // > An Enter keystroke inserts a caret return character into an
     // > open document at the caret position. Set the WantReturns property
     // > to False to allow the application form to process Enter keystrokes
     // > rather than the Rich Edit control. An end user can press the
     // > Ctrl+Enter key combination instead of the Enter key to insert
     // > a caret return character in this mode.
     RichEd_ChatboxOnTrxTab->OnKeyDown = RichEd_Chatbox_KeyDown;  // .. for NON-ASCII keys
     RichEd_ChatboxOnTrxTab->OnKeyPress= RichEd_Chatbox_KeyPress; // .. for ASCII keys including ENTER ('\r' = #13)
     RichEd_ChatboxOnTrxTab->Text = ""; // without this, the TEXT contained the control's NAME !


   } // end if( RichEd_ChatboxOnTrxTab != NULL )

} // TKeyerMainForm::CreateChatboxOnTrxTab()

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::RichEd_Chatbox_KeyDown(TObject *Sender, WORD &Key, TShiftState Shift)
  // This is the 'OnKeyDown' method (a la Borland VCL), in reality a handler
  // for WM_KEYDOWN for the Rich Text edit control on the Remote CW Keyer's DEBUG tab.
  // For details about ..KeyDown() versus ..KeyPress(), see Ed_ErrorHistoryKeyDown() !
{

  switch( Key/*a "WORD"*/ ) // "suppress" this key (by setting Key=0), or let it pass through to the editor ?
   { case VK_RETURN: // .. in RichEdit_Chatbox : do NOT 'enter' this into the text,
        // because pressing ENTER alias RETURN shall SEND the line that has just
        // been typed by the user (including the sysop) .
        // Beware: Because this key 'generates a character', Windows has
        //         already translated the WM_KEYDOWN into a WM_CHAR message,
        // which the "OnKeyDown"-handler cannot intercept. Only "OnKeyPress" can.
        Key = 0;
        break;
     default:
        break;
   } // end switch( Key ) in the "OnKeyDown"-handler (!)

} // end TKeyerMainForm::Ed_ErrorHistoryKeyDown()


//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::RichEd_Chatbox_KeyPress(TObject *Sender, char &Key)
{
  if( RichEd_ChatboxOnTrxTab != NULL ) // <- a "dynamically" created (VCL-) object !
   {
     // "suppress" certain keys (by setting Key=0), because as explained in
     // Ed_ErrorHistoryKeyDown(), Borland's "OnKeyDown" (handler for WM_KEYDOWN)
     // cannot intercept keys with ASCII equivalent, e.g. '\r' (0x0D) for VK_RETURN !
     // So handle those keyboard events ("with ASCII") here, and if a certain key
     // (with an ASCII equivalent like '\r' for ENTER alias RETURN) has been handled
     // by Chatbox_OnKey(), CLEAR the ASCII-'Key' here:
     if( Chatbox_OnKey( RichEd_ChatboxOnTrxTab, Key ) )
      { Key = 0;  // <- Borland's way to indicate 'we have processed this keystroke,
        //              and don't want to pass this to the default handler'.
      }
   }
} // end TKeyerMainForm::RichEd_Chatbox_KeyPress()


//---------------------------------------------------------------------------
void TKeyerMainForm::UpdateChatboxOnTrxTab(void) // Updates a small text box primarily intended for messages from the SYSOP to other users
  // (But since the SYSOP will usually not be on the REMOTE SITE,
  //  he may have to type text about the current station's state, e.g.
  //  the currently connected antenna, maximum allowed power, etc into this
  //  box himself, which would then be relays from HIS CLIENT INSTANCE to the SERVER.
  //  Thus the name 'chatbox', to give it a short name.
  //  Also, OTHER REGISTERED USERS allowed to TALK to others may send short lines
  //  of text TO THE SYSOP via the chat box, e.g. "there's a thunderstorm on the way".
{
  if( RichEd_ChatboxOnTrxTab == NULL )
   { CreateChatboxOnTrxTab();
   }
  if( RichEd_ChatboxOnTrxTab != NULL )
   {
   }
} // end TKeyerMainForm::UpdateChatboxOnTrxTab()


//---------------------------------------------------------------------------
void TKeyerMainForm::UpdateTrxDisplay(void) // ..IF MODIFIED (checked internally)
  // Periodically called from TKeyerMainForm::Timer1Timer() .
{ double d;
  char sz80Temp[84];
  int   iSelectedBandComboRow;
  DWORD dwAvailableBands;
  BOOL  fTxBandsModified = MyCwNet.RigControl.fTxBandsModified;
        // '--> TRUE when there is an updated list of available transmit bands,
        //      and their edge frequencies in MyCwNet.RigControl.TxBandEdges[].
  BOOL  fBandStackingRegsModified = MyCwNet.RigControl.fBandStackingRegsModified;

  if( fTxBandsModified )
   { MyCwNet.RigControl.fTxBandsModified = FALSE; // acknowledge "new TX bands" for RigControl.c
   }
  if( fBandStackingRegsModified )
   { MyCwNet.RigControl.fBandStackingRegsModified = FALSE; // acknowledge "new Band-Stacking Registers" for RigControl.c

   }

  // Periodically update the VFO frequency (displayed in Ed_VFO),
  //  or leave that edit-field untouched because the local operator
  //  is currently TYPING into that field ?
  if( (MyCwNet.RigControl.dblVfoFrequency != RIGCTRL_NOVALUE_DOUBLE ) // only if the VFO frequency is KNOWN..
   && (g_SpecDispControl.iVFOEditTimer_ms <= 0 ) // avoid interference while the user TYPES into the VFO edit field ..
   && (!g_SpecDispControl.fDraggingVfoFrequency) // avoid interference while the user 'drags' the VFO frequency via mouse
    )
   { d = g_SpecDispControl.dblDisplayedVfoFreq_Hz - MyCwNet.RigControl.dblVfoFrequency;
     if( (d < -1.0) || (d > 1.0) )  // significant change since the last update (1 Hz) ?
      { KeyerGUI_UpdateVFODisplay( Ed_VFO, MyCwNet.RigControl.dblVfoFrequency );
      }
   }
  // Indicate if the VFO display is currently in "editing" state
  //   (in which it possibly shows an outdated value if the frequency
  //    has been modified by another user, or the 'local operator')
  // or in "normal display" state (when it always shows the current value):
  if( (g_SpecDispControl.iVFOEditTimer_ms <= 0 ) // NOT editing the VFO frequency ..
   && (!g_SpecDispControl.fDraggingVfoFrequency) // .. and NOT dragging the VFO frequency via mouse ?
    )
   { switch( g_SpecDispControl.iColourScheme )
      { case COLOUR_SCHEME_DARK: // white digits on black background
           Ed_VFO->Color = clBlack;
           break;
        default: // COLOUR_SCHEME_DEFAULT : usually black digits on white background
           Ed_VFO->Color = clWindow;
           break;
      }
   }
  else // VFO frequency is currently being EDITED (typing or via up/down on a certain digit):
   { switch( g_SpecDispControl.iColourScheme )
      { case COLOUR_SCHEME_DARK: // white digits on black background
           Ed_VFO->Color = (TColor)RGB(0x00,0x60,0x00); // very dark green
           break;
        default: // COLOUR_SCHEME_DEFAULT : usually black digits on white background
           Ed_VFO->Color = (TColor)RGB(0xD0,0xFF,0xD0); // very light green
           break;
      }
   }


  if( m_iDisplayedRigControlOpMode != MyCwNet.RigControl.iOpMode )
   { m_iDisplayedRigControlOpMode  = MyCwNet.RigControl.iOpMode;
     ++KeyerGUI_iUpdating;
     FillComboWithItemsFromTokenLists( CB_OpMode, RigCtrl_OpModes, NULL, NULL,
                                        m_iDisplayedRigControlOpMode );
     --KeyerGUI_iUpdating;
   }

  // ex: dwSelectedBand = RigCtrl_FrequencyToBand( MyCwNet.RigControl.dblVfoFrequency );
  iSelectedBandComboRow = KeyerGUI_FrequencyToBandComboRow( MyCwNet.RigControl.dblVfoFrequency );
       // '---> iRow in KeyerGUI_BandComboInfoTable[iRow][iColumn]
  dwAvailableBands= RigCtrl_GetAvailableBands( &MyCwNet.RigControl );
  if(  (m_iSelectedBandComboRow     != iSelectedBandComboRow )
    || (m_dwDisplayedAvailableBands != dwAvailableBands)
    ||  fTxBandsModified || fBandStackingRegsModified )
   { m_iSelectedBandComboRow     = iSelectedBandComboRow;
     m_dwDisplayedAvailableBands = RigCtrl_GetAvailableBands( &MyCwNet.RigControl );
     ++KeyerGUI_iUpdating;
     KeyerGUI_FillComboWithBandList( &MyCwNet.RigControl, CB_Band );
     // '--> When available, the "band list" also shows "Band Stacking Registers",
     //      and then loaded from a file, also shows "User Defined Band".
     // See also: Managing THREE (or even more) different columns to click on
     //           in TKeyerMainForm::CB_BandClick() .
     --KeyerGUI_iUpdating;
   }

  // Missing the periodic update of the spectrum/spectrogram/frequency scale ?
  // SpecDisp_UpdateWaterfall(), SpecDisp_UpdateSpectrum(), SpecDisp_UpdateFreqScale()
  // are called directly from TKeyerMainForm::Timer1Timer() !


} // end TKeyerMainForm::UpdateTrxDisplay()


//-----------------------------------------------------------------------------
extern "C" void SpecDisp_UpdateVFODisplay( double dblVfoFrequency_Hz ) // -> Ed_VFO (in the GUI)
  // Just a wrapper for KeyerGUI_UpdateVFODisplay() without any Borland-VCL-dependency.
  // Called from SpecDisp.cpp (which is is "VCL free" module, just plain C).
{
  if( KeyerMainForm != NULL )
   {  KeyerGUI_UpdateVFODisplay( KeyerMainForm->Ed_VFO, dblVfoFrequency_Hz );
   }
} // end SpecDisp_UpdateVFODisplay()


//---------------------------------------------------------------------------
void TKeyerMainForm::ApplyNewVFOFreq(void) // Ed_VFO -> MyCwNet.RigControl.dblVfoFrequency
{
  double dblFreq_Hz;

  ++KeyerGUI_iUpdating; // prevent interference from the VCL while updating certain GUI elements..

  dblFreq_Hz = KeyerGUI_GetFrequencyFromEditField( Ed_VFO, Ed_VFO->SelStart, NULL );

  // Apply the new VFO frequency immediately ?
  if( dblFreq_Hz > 1e3 )
   { RigCtrl_SetVFOFrequency( &MyCwNet.RigControl, dblFreq_Hz );
     // ex: CLI_fRedrawFreqScale = TRUE;  // here: after modifying the radio's "VFO" via edit field
     // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
     // The new setting in MyCwNet.RigControl [here: RigControl.dblVfoFrequency]
     // will be compared with the 'last values' sent to a remote server/clients,
     // and if the value was MODIFIED, CwNet.c : CwNet_OnPoll() will send the
     // new setting also via the Remote CW Keyer's own TCP/IP connection.
     // Grep for 'Request to "QSY" from this client to the remote server ?' ...
     // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
   }

  --KeyerGUI_iUpdating;
} // end TKeyerMainForm::ApplyNewVFOFreq()

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::CB_OpModeClick(TObject *Sender)
{
  int iNewOpMode;
  AnsiString s = CB_OpMode->Text; // for BCB V12, converts Unicode to ANSI.
                                  // for BCB V6, just copies.
    // Later, s.c_str() MUST return a "char *", not a "wchar_t *" !
  if( KeyerGUI_iUpdating==0 )
   {
     iNewOpMode = SL_FindStringInTable( RigCtrl_OpModes, s.c_str() );
     if( iNewOpMode > 0 )
      { RigCtrl_SetOperatingMode( &MyCwNet.RigControl, iNewOpMode );
        // '--> Like many other 'set' API functions in RigControl.c ,
        //      this one is aware of operating as a CLIENT, and in that case,
        //      passes the new value to the remote server via TCP/IP (CwNet.c).
        //      See implementation of RigCtrl_SetOperatingMode() for details
        //      (used as an example for exchanging parameters between
        //       GUI, RigControl, and client/server via TCP/IP using CwNet.c ).
      }
   }  // end if( KeyerGUI_iUpdating==0 )
  KeyerGUI_iFocusSwitchCountdownTimer_ms = 3000; // don't AUTOMATICALLY switch the focus away for a few seconds, here: when clicking into the "Op-Mode" / "Modulation and Filter bandwidth" combo
}   // end TKeyerMainForm::CB_ModClick()

//---------------------------------------------------------------------------
int BitmaskToBitIndex(long i32Bitmask )
 // Returns the zero-based index of the least significant, NONZERO bit in i32Bitmask.
{ int iBitIndex = 0;
  while( (i32Bitmask != 0) && ( (i32Bitmask & 1) == 0 ) )
   { ++iBitIndex;
     i32Bitmask >>= 1;
   }
  return iBitIndex;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::CB_BandClick(TObject *Sender)
  // Mouse event handler for the 'Band list'. Originally a simple combo list,
  // later extended to show up to three 'Band Stacking Registers' per band,
  // read out via CI-V from certain Icom radios, arranged in THREE COLUMNS.
  // Selecting a certain BAND STACKING REGISTER not only switches band / frequency,
  // but also the operating mode (and, maybe, even more).
{
  long i32NewBand;
  TPoint tpScreen, tpClient;
  int  i, n, iCharWidth, iCharIndex, iComboItemIndex, iBandIndex, iRow, iColumn, iBandStackingRegIndex;
  char sz80ItemText[84];
  char sz80SelectedBandName[84];
  const char* pszBandName, *pszSource;
  T_RigCtrlInstance *pRC = &MyCwNet.RigControl;
  T_RigCtrlFreqMemEntry     *pBandStackingReg;
  T_RigCtrlUserDefinedBand  *pUserDefdBand;
  T_BandComboInfoTableEntry *pInfoTableEntry;
  BOOL fOk = FALSE;

  iRow = CB_Band->ItemIndex;

  if( (KeyerGUI_iUpdating==0) && (iRow>=0) && (iRow<KEYER_GUI_MAX_BANDS) )
   {                   //  '-> valid index into KeyerGUI_BandComboInfoTable[iRow] ?
     // The bands listed in this combo box may be limited to what THE SYSOP allows,
     //  or what the remotely controlled radio permits.
     //  When available, the frequency on the new(?) band will be
     //  one of up to three 'Band Stacking Registers', at least for Icom radios.
     // For that purpose, a single entry in the TComboBox (a standard VCL control)
     //  may be divided into THREE PARTS ("left", "center", "right"),
     //  resembling Icom's own 'BAND STACKING REGISTER' display (they used to
     //  write all in upper case).
     strncpy( sz80ItemText, AnsiString(CB_Band->Text).c_str(), 80 );
     sz80ItemText[80] = '\0'; // strncpy() sucks.. it doesn't always terminate the destination
       // '--> e.g. "40 m:  7.005 CW   7.030 CW   7.033 CW" (see KeyerGUI_FillComboWithBandList() ),
       //  or even  "60 m UK: 5.2600 CW" (loaded from e.g. RCWKeyer_Bands.txt into pRC->UserDefinedBands[]),
       //    versus "60 m EU: 5.3525 CW" (different "sub-band", also in pRC->UserDefinedBands[]),
       //  or just  "40 m"  if there are no 'Band Stacking Registers" for this band.
     pszSource = sz80ItemText;
     SL_CopyStringUntilDelimiter( &pszSource, sz80SelectedBandName/*dest*/, 16/*iMaxLen*/, ":;\r\n"/*delimiters*/ );
       // ,------------------------------------'
       // '--> Again: This may have to be USER-DEFINED, because the rig itself
       //      is not aware of national regulations or even BAND PLANS !
       //      So drilled up in 2024-12-22 for operation on the 60 meter band,
       //      with its crazy different and incompatible regulations everywhere.


     // Because a Borland VCL TComboBox doesn't expose the coordinate of the
     // mouse pointer when clicking the box, try to determine the 'column' (!)
     // of the mouse pointer within the combo as follows:
     tpScreen = TPoint( Mouse->CursorPos.x, Mouse->CursorPos.y ); // obfuscated C++ stuff...
     tpClient = CB_Band->ScreenToClient(tpScreen);
     // Test results at THIS POINT (sic) when clicking into the 1st item,
     // first character of the item text: tpClient.x = 4,  tpClient.y = 34 .
     // Clicked on the last character: tpClient.x = 254.
     // Tried to convert the horizontal pixel offset into a character index,
     // and -from that character index- determine the 'band stack index' (0..2):
     iCharWidth = CB_Band->Canvas->TextWidth( "w" ); // with "Courier New size 8" : 9 pixels / character
     iCharIndex = tpClient.x / iCharWidth;
     if( iCharIndex < (BANDLIST_BAND_COLUM_WIDTH+BANDLIST_FREQ_COLUMN_WIDTH) )
      { // If there are 'band stacking' frequencies at all, it's the first
        // (that's the one with Icom's MOST RECENT band stacking register)
        iColumn = 0; // -> use KeyerGUI_BandComboInfoTable[?][0] further below
      }
     else if( iCharIndex < (BANDLIST_BAND_COLUM_WIDTH+2*BANDLIST_FREQ_COLUMN_WIDTH) )
      { // the SECOND "band stacking column" If there are 'band stacking' frequencies at all, it's the
        iColumn = 1; // -> use KeyerGUI_BandComboInfoTable[?][1] further below
      }
     else
      { iColumn = 2; // -> use KeyerGUI_BandComboInfoTable[?][2] further below
      }
     pInfoTableEntry = &KeyerGUI_BandComboInfoTable[iRow][iColumn];
     i32NewBand = SL_FindStringInTable( RigCtrl_BandNames, sz80SelectedBandName );
     // The above may fail since the addition of Band Stacking frequencies
     // and 'user defined bands' in the in the item text from the combo box.
     // Only pInfoTableEntry gives a clue about what the clicked cell contains:
     if(   ((i=pInfoTableEntry->iUserDefinedBandIndex) >= 0 )
        && ( i < RIGCTRL_NUM_USER_DEFINED_BANDS ) )
      { // user clicked on cell with a reference to a "user defined band",
        // so switch to the band's "default frequency" when valid :
        pUserDefdBand = &pRC->UserDefinedBands[i];
        if( pUserDefdBand->dblFdef_Hz > 0.0 )
         { fOk = RigCtrl_SetVFOFrequency( pRC, pUserDefdBand->dblFdef_Hz );
         }
        if( pUserDefdBand->dwDefOpMode != 0 )
         { RigCtrl_SetOperatingMode( pRC, pUserDefdBand->dwDefOpMode );
         }
      } // end if < cell with reference to a USER-DEFINED BAND >
     else if( ((i=pInfoTableEntry->iBandStackingRegIndex) >= 0 )
           && ( i < RIGCTRL_NUM_BAND_STACKING_REGS ) )
      { // user clicked on cell with a reference to a "band-stacking register",
        // so switch to whatever such a band-stacking register may contain:
        pBandStackingReg = &pRC->BandStackingRegs[i];
           // When controlling e.g. an IC-7300, the behaviour shall be exactly
           // as when opening the "Band Stacking Register" :
           // which kind-of remembers the last frequency used on THAT BAND.
        if( pBandStackingReg->RxTx[0].dblOperatingFreq_Hz > 0.0 )
         { fOk = RigCtrl_SwitchToFreqMemEntry( pRC, pBandStackingReg ); // switch frequency, mode, and maybe even more ("split")
         }
      }
     else // neither switch to a USER-DEFINED BAND nor a BAND-STACKING REGISTER .. what else ?
      { if( i32NewBand >= 0 ) // no valid "band stacking register" but we know THE BAND to switch to:
         { fOk = RigCtrl_SwitchToBand( pRC, i32NewBand );
         }
      }
     if( fOk )
      { // Abort a pending frequency change from the VFO edit field,
        // instead ALWAYS overwrite the edit field (Ed_VFO) with the new frequency:
        g_SpecDispControl.iVFOEditTimer_ms = 0; // not editing Ed_VFO anymore
        g_SpecDispControl.fDraggingVfoFrequency = FALSE; // not dragging the VFO frequency anymore
        KeyerGUI_UpdateVFODisplay( Ed_VFO, pRC->dblVfoFrequency );
              // If RigCtrl_SwitchToFreqMemEntry() also modified
              //   MyCwNet.RigControl.iOpMode, it will be different from
              //   m_iDisplayedRigControlOpMode until UpdateTrxDisplay()
              //   takes care of that.
        m_iSelectedBandComboRow = KeyerGUI_FrequencyToBandComboRow( pRC->dblVfoFrequency );
      }   // end if( fOk )
   }     // end if( KeyerGUI_iUpdating==0 )
  KeyerGUI_iFocusSwitchCountdownTimer_ms = 3000; // don't AUTOMATICALLY switch the focus away for a few seconds, here: when clicking into the "Band List" / "Band Stacking Registers"
}       // end TKeyerMainForm::CB_BandClick()

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Btn_HelpBandSwitchClick(TObject *Sender)
{
  YHF_HELP_ShowHelpContext( HELPID_BAND_SELECTION );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_HelpOnIOConfigClick(TObject *Sender)
{
  YHF_HELP_ShowHelpContext( HELPID_IO_CONFIG );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Chk_Audio_AllowNetworkClick(TObject *Sender)
{
  // This control shall have an 'immediate effect' when clicking,
  //      e.g. disable sending audio to, or receiving audio from the network:
  if( KeyerGUI_iUpdating <= 0 ) // prevent interference from "OnClick"
   { //  (it sometimes fires when MODIFYING 'Checked' programmatically)
     if( Chk_Audio_AllowNetwork->Checked )
      {  CwKeyer_DSP.cfg.iAudioFlags |= DSP_AUDIO_FLAGS_ALLOW_NETWORK_AUDIO;
      }
     else
      {  CwKeyer_DSP.cfg.iAudioFlags &= ~DSP_AUDIO_FLAGS_ALLOW_NETWORK_AUDIO;
      }
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Chk_Audio_DecodeCWClick(TObject *Sender)
{
  // This control shall have an 'immediate effect' when clicking,
  //  because it's mostly annoying to let a MACHINE decode Morse Code.
  //  But sometimes it helps if the other OP stubbornly ignores "pse QRS" ;o)
  if( KeyerGUI_iUpdating <= 0 ) // prevent interference from "OnClick"
   { //  (it sometimes fires when MODIFYING 'Checked' programmatically)
     if( Chk_Audio_DecodeCW->Checked )
      {  CwKeyer_DSP.cfg.iAudioFlags |= DSP_AUDIO_FLAGS_DECODE_CW;
      }
     else
      {  CwKeyer_DSP.cfg.iAudioFlags &= ~DSP_AUDIO_FLAGS_DECODE_CW;
      }
   }
}


//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Chk_Audio_SidetoneWhenKeyedOnTheRigClick(TObject *Sender)
{ // Added 2025-09-18. When enabled, and the 'sysop' operates his radio
  //   LOCALLY (morse key connected DIRECTLY to the transceiver, e.g. IC-7300),
  //   remote clients can HEAR the sidetone because with this setting,
  //   the IC-7300's "ACC / USB" audio output has the sidetone ADDED.
  //   In practice, this is even more complex, because when REMOTE USERS
  //   key the radio in CW, they do NOT want to hear the severly delayed sidetone
  //   as long as THEY transmit. Thus Rigctrl.c must now find out IF and WHY
  //   the radio is transmitting, and depending on that, quickly reconfigure
  //   what the IC-7300's "Full Manual" A7292-4EX-11 page 163 calls
  // "Send/read beep and speech output setting to ACC/USB (when AF signal output is set)",
  //   via CI-V command 0x1A, subcommand 0x05, sub-sub-command "0062" .
  // Implementation / ugly internal details only in RigControl.c !
  if( KeyerGUI_iUpdating <= 0 ) // prevent interference from "OnClick"
   {
     RigCtrl_ModifyBitInDWORD( &CwKeyer_Config.dwAudioOptions, KEYER_AUDIO_OPTION_SIDETONE_WHEN_KEYED_ON_RIG,
                               Chk_Audio_SidetoneWhenKeyedOnTheRig->Checked );
     // Immediately "apply" the new setting for the Rig-Control-Module, too
     // (it may have change what Icom calls 'Beep and speech output setting to ACC/USB'):
     RigCtrl_ModifyBitInDWORD( &MyCwNet.RigControl.dwRigControlFlags, RIGCTRL_FLAG_SIDETONE_WHEN_KEYED_ON_RIG,
                                CwKeyer_Config.dwAudioOptions & KEYER_AUDIO_OPTION_SIDETONE_WHEN_KEYED_ON_RIG );

   }
} // end TKeyerMainForm::Chk_Audio_SidetoneWhenKeyedOnTheRig()


//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::ApplyKeyerSettings(TObject *Sender)
{ // No need to click "Apply" - called from various combo-boxes "OnClick"-handlers:
  AnsiString s;
  if( KeyerGUI_iUpdating <= 0 ) // prevent interference from certain "OnClick" events..
   { CwKeyer_Config.iMorseKeyType = GetSelectedTokenValueFromComboBox( CB_KeyType, KeyTypes, NULL, NULL );
     CwKeyer_Config.iSidetoneOnTXD= GetSelectedTokenValueFromComboBox( CB_SidetoneOnTXD, SidetonesOnTXD, NULL, NULL );
     // Formerly on the "Keyer" tab, now on an extra "Audio" tab.. anyway, keep them here:
     CwKeyer_DSP.cfg.iSidetoneFreq_Hz = SL_atoi( AnsiString(CB_SidetoneOnAudioOut->Text).c_str() );
     CwKeyer_DSP.cfg.iSidetoneRiseTime_ms = SL_atoi( AnsiString(CB_SidetoneRiseTime->Text).c_str() );

     s = CB_AudioInputDevice->Text;
     if( s != "" )
      { if( s != AnsiString(CwKeyer_DSP.cfg.sz255AudioInputDevice) )
         { // MODIFIED 'audio input device' so copy to config, and re-init...
           strncpy( CwKeyer_DSP.cfg.sz255AudioInputDevice, s.c_str(), 255 );
           m_fRestartKeyerThread = TRUE;  // ... because a basic part of the config was modified
         }
      }
     s = CB_AudioOutputDevice->Text;
     if( s != "" )
      { if( s != AnsiString(CwKeyer_DSP.cfg.sz255AudioOutputDevice) )
         { strncpy( CwKeyer_DSP.cfg.sz255AudioOutputDevice, s.c_str(), 255 );
           m_fRestartKeyerThread = TRUE; // here: because a new AUDIO OUTPUT device was selected
         }
      }
     CwKeyer_Config.iTxDelayTime_ms = StrToIntDef( Ed_TxDelayTime->Text, 25);
     CwKeyer_Config.iTxHangTime_ms  = StrToIntDef( Ed_TxHangTime->Text, 500);
   } // end if( KeyerGUI_iUpdating <= 0 )
} // end TKeyerMainForm::ApplyKeyerSettings()


//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::CB_SidetoneOnAudioOutChange(TObject *Sender)
{ // Special service to modify the sidetone (via AUDIO OUTPUT) "while typing"
  int iNewSidetoneFreq;
  if( CB_SidetoneOnAudioOut->Text == "None" )
   {  iNewSidetoneFreq = 0;  // 0 = off
   }
  else // not OFF, so parse from the combo's EDIT FIELD :
   { iNewSidetoneFreq = SL_atoi( AnsiString(CB_SidetoneOnAudioOut->Text).c_str() );
     UTL_LimitInteger( &iNewSidetoneFreq, 50/*Hz*/, 3000/*Hz*/ );
     // (who knows .. "audio keying a transmitter in USB" should use a
     //  high enough frequency, to that 2 * f, 3 * f, etc aren't in the passband.
     //  Thus if the operator TYPES into this edit field, accept up to 3 kHz.)
   }
  if( iNewSidetoneFreq != CwKeyer_DSP.cfg.iSidetoneFreq_Hz )
   { CwKeyer_DSP.cfg.iSidetoneFreq_Hz = iNewSidetoneFreq;
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::ApplyVolumeAdjustments(TObject *Sender)
{ // The 'Volume Adjustment' sliders on the 'Audio' tab shall also have
  // an IMMEDIATE effect on the digital signal processing (in CwDSP.c) :
  if( KeyerGUI_iUpdating <= 0 ) // don't react of "OnChange" fired because THE PROGRAM changed a value:
   { CwKeyer_DSP.cfg.iSidetoneGain_dB = -SB_SidetoneGain_dB->Position;
     CwKeyer_DSP.cfg.iAudioInGain_dB  = -SB_AudioInGain_dB->Position;
     CwKeyer_DSP.cfg.iAudioOutGain_dB = -SB_AudioOutGain_dB->Position;
     CwKeyer_DSP.cfg.iNetworkTonesGain_dB = -SB_NetworkTonesGain_dB->Position;
   }
} // end TKeyerMainForm::ApplyVolumeAdjustments(

//---------------------------------------------------------------------------
void TKeyerMainForm::UpdateClientServerLabels(void)
  // Grays out those labels on the "Network" tab that, depending on the
  // currently selected "Network functionality", have NO FUNCTION .
{
  Lab_RemoteURL->Enabled  = Rbtn_NetworkClient->Checked;
  Lab_MyUserName->Enabled = Rbtn_NetworkClient->Checked;
  Lab_UserCallsign->Enabled=Rbtn_NetworkClient->Checked;
  Lab_ServerListeningPort->Enabled = Rbtn_NetworkServer->Checked;
  Lab_AcceptedUsers->Enabled = Rbtn_NetworkServer->Checked;
} // end TKeyerMainForm::UpdateClientServerLabels()


//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::ApplyNetworkSetup(TObject *Sender)
{
  int  iNewValue;
  BOOL fModified = FALSE;
  if( KeyerGUI_iUpdating <= 0 ) // prevent interference from 'UpdateNetworkSetup' ..
   {
     if( Rbtn_NetworkClient->Checked )
      { fModified |= (MyCwNet.cfg.iFunctionality != CWNET_FUNC_CLIENT);
        MyCwNet.cfg.iFunctionality = CWNET_FUNC_CLIENT;
      }
     else if(Rbtn_NetworkServer->Checked)
      { fModified |= (MyCwNet.cfg.iFunctionality != CWNET_FUNC_SERVER);
        MyCwNet.cfg.iFunctionality = CWNET_FUNC_SERVER;
      }
     else
      { fModified |= (MyCwNet.cfg.iFunctionality != CWNET_FUNC_OFF);
        MyCwNet.cfg.iFunctionality = CWNET_FUNC_OFF;
      }
     strncpy( MyCwNet.cfg.sz80ClientRemoteIP, AnsiString(Ed_ClientRemoteIP->Text).c_str(), 80 );
     strncpy( MyCwNet.cfg.sz80ClientUserName, AnsiString(Ed_ClientUserName->Text).c_str(), 80 );
     strncpy( MyCwNet.cfg.sz80ClientCallsign, AnsiString(Ed_ClientCallsign->Text).c_str(), 80 );
     iNewValue = StrToIntDef( Ed_ServerListeningPort->Text, 7355 );
     fModified |= (MyCwNet.cfg.iServerListeningPort != iNewValue);
     MyCwNet.cfg.iServerListeningPort = iNewValue;
     strncpy( MyCwNet.cfg.sz255AcceptUsers,   AnsiString(Ed_ServerAcceptUsers->Text).c_str(), 255 );
     strncpy( MyCwNet.cfg.sz20ServerAdminPWD, AnsiString(Ed_ServerAdminPWD->Text).c_str(), 20 );
#   if( SWI_USE_HTTP_SERVER )  // a built-in HTTP server is optional ..
     if( Chk_EnableHTTP->Checked )
      { MyCwNet.cfg.iHttpServerOptions |= HTTP_SERVER_OPTIONS_ENABLE;
      }
     else
      { MyCwNet.cfg.iHttpServerOptions &= ~HTTP_SERVER_OPTIONS_ENABLE;
      }
#   endif // SWI_USE_HTTP_SERVER ?

     UpdateClientServerLabels();
     if( fModified )  // modified so that CwNet.c needs a "restart" ?
      {
        HERE_I_AM__GUI();
        CwNet_Stop( &MyCwNet );
        // For safety, also stop and optionally restart the Hamlib server:
        HLSRV_Stop( &MyHamlibServer );
        if( MyCwNet.cfg.iFunctionality != CWNET_FUNC_OFF )
         { ShowError( ERROR_CLASS_INFO, "Starting Network functionality .." );
           CwNet_Start( &MyCwNet );
         }
#      if( SWI_USE_HAMLIB_SERVER ) // not sure if this 'feature' is going to stay...
        if(  CwKeyer_Config.fEnableBuiltInHamlibServer
         && (MyHamlibServer.cfg.iServerListeningPort > 0 ) )
         { HLSRV_Start( &MyHamlibServer );
         }
#      endif // SWI_USE_HAMLIB_SERVER ?
        HERE_I_AM__GUI();
      }
     UpdateWindowCaption(); // -> e.g. "Remote CW Keyer [1]  (Server)/(Client)"
   } // end if( KeyerGUI_iUpdating <= 0 )
  HERE_I_AM__GUI();

} // TKeyerMainForm::ApplyNetworkSetup()

//---------------------------------------------------------------------------
void TKeyerMainForm::UpdateNetworkSetup(void)
{
  char sz80[84];
  struct hostent *pHE;
  DWORD  dwIPv4Address;

  HERE_I_AM__GUI();

  ++KeyerGUI_iUpdating;
  Rbtn_NetworkOff->Checked = MyCwNet.cfg.iFunctionality == CWNET_FUNC_OFF;
  Rbtn_NetworkClient->Checked = MyCwNet.cfg.iFunctionality == CWNET_FUNC_CLIENT;
  Rbtn_NetworkServer->Checked = MyCwNet.cfg.iFunctionality == CWNET_FUNC_SERVER;
  Ed_ClientRemoteIP->Text  = MyCwNet.cfg.sz80ClientRemoteIP;
  Ed_ClientUserName->Text  = MyCwNet.cfg.sz80ClientUserName;
  Ed_ClientCallsign->Text  = MyCwNet.cfg.sz80ClientCallsign;
  Ed_ServerListeningPort->Text = IntToStr( MyCwNet.cfg.iServerListeningPort );
  Ed_ServerAcceptUsers->Text = MyCwNet.cfg.sz255AcceptUsers;
#if( SWI_USE_HTTP_SERVER )  // use a built-in HTTP server ?
  Chk_EnableHTTP->Checked = (MyCwNet.cfg.iHttpServerOptions & HTTP_SERVER_OPTIONS_ENABLE) != 0;
  Ed_ServerAdminPWD->Text = MyCwNet.cfg.sz20ServerAdminPWD;
#else // ! SWI_USE_HTTP_SERVER -> disable all GUI controls for the HTTP server:
  Chk_EnableHTTP->Checked    = FALSE;
  Chk_EnableHTTP->Enabled    = FALSE;
  Lab_AdminPWD->Enabled      = FALSE;
  Ed_ServerAdminPWD->Enabled = FALSE;
#endif // SWI_USE_HTTP_SERVER ?

  // The following LABELS are "just for info", and/or for troubleshooting:
  if( gethostname( sz80,80) == SOCKET_ERROR )
   { Lab_LocalHost->Caption = "*FAIL*"; // here we are again, in Winsock heaven..
     Lab_LocalIP->Caption   = "*FAIL*";
   }
  else  // successfully retrieve this PC's LOCAL NAME in the network ("hostname")..
   { Lab_LocalHost->Caption = sz80;
     pHE = gethostbyname(sz80); // resolves this machine's NAME into one of its NUMERIC IP ADDRESSES ..
     strcpy( sz80, "???" );
     if( pHE != NULL )
      { // Note: the 'struct hostent pointer') isn't a single item,
        // but a WHOLE LIST of items. For example, if the PC has multiple
        // 'network interfaces' (say two Ethernet LAN ports, and one WLAN),
        // then there may be many roads leading to Rome. We only show THE FIRST..
        if( ( pHE->h_addrtype == AF_INET )  // sounds good, the "address family" is "internet"...
         && ( pHE->h_length == 4 )       // .. and the list of addresses seem to be "IPv4"...
         && ( pHE->h_addr_list[0] != NULL) ) // .. and there's at least ONE entry in it ...
         { dwIPv4Address = *(DWORD*)pHE->h_addr_list[0]; // read FOUR BYTES in "network byte order"
           sprintf( sz80, "%d.%d.%d.%d",
               (int)(BYTE)(dwIPv4Address>>0),  (int)(BYTE)(dwIPv4Address>>8),
               (int)(BYTE)(dwIPv4Address>>16), (int)(BYTE)(dwIPv4Address>>24) );
         }
      }
     Lab_LocalIP->Caption = sz80;
   }
  UpdateClientServerLabels();
  --KeyerGUI_iUpdating;

  HERE_I_AM__GUI();

} // TKeyerMainForm::ApplyNetworkSetup()

//---------------------------------------------------------------------------
void TKeyerMainForm::SwitchMainTab(int iNewTab) // [in] one of the following:
  //  KEYER_GUI_MAIN_TAB_CONFIG,  KEYER_GUI_MAIN_TAB_KEYER_SETTINGS, KEYER_GUI_MAIN_TAB_AUDIO,
  //  KEYER_GUI_MAIN_TAB_NETWORK, KEYER_GUI_MAIN_TAB_DEBUG,          KEYER_GUI_MAIN_TAB_MEMORY,
  //  KEYER_GUI_MAIN_TAB_TEST,    KEYER_GUI_MAIN_TAB_TRX,            ... (?)
  // Called from the main thread ("GUI thread") whenever THE PROGRAM
  // needs to switch the main window's main tabsheet, PageControl1 .
  // Callers:  1. FormCreate(), shows the "I/O Config" tab initially,
  //              because this is where the user MUST provide some info
  //              (COM port for the Morse key adapter and remotely controlled rig)
  //           2. StartKeyerAndShowInfo(), switches to the "Debug" tab
  //              while reading the most important settings FROM the radio
  //           3. Timer1Timer() when module RigControl enters RIGCTRL_POLLSTATE_DONE,
  //              to switch from the "Debug" tab (with a live CAT traffic display)
  //              to the "TRX" tab (Transceiver control with VFO and spectrum).
  //
{
  KeyerGUI_fMaySwitchToVfoEditField = (iNewTab==KEYER_GUI_MAIN_TAB_TRX);

  switch( iNewTab )
   { case KEYER_GUI_MAIN_TAB_CONFIG:  PageControl1->ActivePage = TS_Config;  break;
     case KEYER_GUI_MAIN_TAB_KEYER_SETTINGS: PageControl1->ActivePage = TS_KeyerSettings; break;
     case KEYER_GUI_MAIN_TAB_AUDIO:   PageControl1->ActivePage = TS_Audio;   break;
     case KEYER_GUI_MAIN_TAB_NETWORK: PageControl1->ActivePage = TS_Network; break;
     case KEYER_GUI_MAIN_TAB_DEBUG:   PageControl1->ActivePage = TS_Debug;   break;
     case KEYER_GUI_MAIN_TAB_MEMORY:  PageControl1->ActivePage = TS_Memory;  break;
     case KEYER_GUI_MAIN_TAB_TEST:    PageControl1->ActivePage = TS_Test;    break;
     case KEYER_GUI_MAIN_TAB_TRX:     PageControl1->ActivePage = TS_TRX;     break;
     default: break;
   } // end switch( iNewTab )
  KeyerGUI_iCurrentMainTab = iNewTab;  // save this to avoid having to compare PageControl1->ActivePage (a Borland VCL thing)
} // end TKeyerMainForm::SwitchMainTab()

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::PageControl1Change(TObject *Sender)
  // About the VCL's TPageControl.OnChange :
  // > Occurs after a new tab is selected.
  // > Write an OnChange event handler to take specific action
  // > immediately after the selected tab changes. Use the
  // > TabIndex property to determine which tab is now selected.
  // > This is the opportunity to make any changes to the control
  // > that reflect the new state implied by the selected tab.
  // > Before the value of TabIndex changes, an OnChanging event occurs.
  // > Note: The event is not called if you change the active page in code,
  // >       for example, by setting the value of ActivePage.
{
  int iNewTab = KeyerGUI_iCurrentMainTab;
  // If we recognize the NEW value of 'PageControl1->ActivePage',
  // modify KeyerGUI_iCurrentMainTab accordingly
  if( PageControl1->ActivePage == TS_Config )
   { iNewTab = KEYER_GUI_MAIN_TAB_CONFIG;
   }
  else if( PageControl1->ActivePage == TS_KeyerSettings )
   { iNewTab = KEYER_GUI_MAIN_TAB_KEYER_SETTINGS;
   }
  else if( PageControl1->ActivePage == TS_Audio )
   { iNewTab = KEYER_GUI_MAIN_TAB_AUDIO;
   }
  else if( PageControl1->ActivePage == TS_Network )
   { iNewTab = KEYER_GUI_MAIN_TAB_NETWORK;
   }
  else if( PageControl1->ActivePage == TS_Debug )
   { iNewTab = KEYER_GUI_MAIN_TAB_DEBUG;
   }
  else if( PageControl1->ActivePage == TS_Memory )
   { iNewTab = KEYER_GUI_MAIN_TAB_MEMORY;
   }
  else if( PageControl1->ActivePage == TS_Test )
   { iNewTab = KEYER_GUI_MAIN_TAB_TEST;
   }
  else if( PageControl1->ActivePage == TS_TRX )
   { iNewTab = KEYER_GUI_MAIN_TAB_TRX;
   }
  else
   { // oops.. new tabsheet ?
   }
  KeyerGUI_fMaySwitchToVfoEditField = (iNewTab==KEYER_GUI_MAIN_TAB_TRX);
  if( iNewTab != KeyerGUI_iCurrentMainTab )  // tabsheet obviously switched by the USER ->
   { KeyerGUI_iCurrentMainTab = iNewTab;
     KeyerGUI_fAutomaticTabSwitching = FALSE;
   }
} // end TKeyerMainForm::PageControl1Change()

//---------------------------------------------------------------------------
void TKeyerMainForm::StartKeyerAndShowInfo(void)
  // Since 2025-06, includes starting the "additional/auxiliary ports" and their worker threads !
  // Depending on CwKeyer_Config(), certain worker threads may also be STOPPED here.
{
  int i;
  T_RigCtrlInstance *pRC = &MyCwNet.RigControl;
#if( SWI_NUM_AUX_COM_PORTS > 0 )
  T_AuxComPortInstance *pAuxComPort;
  T_AuxComConfig *pAuxComCfg;
  T_RigCtrl_PortInstance *pRigctrlPort;
#endif

  HERE_I_AM__GUI();

  // Just in case various threads are still "peeking" at the CW keyer's
  //  current states of digital in- and outputs, make sure they don't
  //  get confused by e.g. flags like "Paddle input active", "PTT active", etc:
  Keyer_dwCurrentSignalStates = 0; // <- default state as in the var-declaration

  HERE_I_AM__GUI();
  CwDSP_Start( &CwKeyer_DSP ); // start our 'audio DSP' before the keyer..
  HERE_I_AM__GUI();            // 2025-08-22: Crashed shortly after this line
  pRC->fListenOnlyMode = FALSE;  // normal operation mode: RCW Keyer may *actively* control the radio, not just eavesdrop traffic
#if( SWI_NUM_AUX_COM_PORTS > 0 )
  for( int iAuxCom=0; iAuxCom<SWI_NUM_AUX_COM_PORTS; ++iAuxCom )
   { pAuxComPort = &AuxComPorts[iAuxCom];
     pAuxComCfg  = &CwKeyer_Config.sAuxCom[iAuxCom];
     // Exotic parameters for the serial port are not passed to AuxCom_Start()
     // as parameter, but must be set in the struct members of T_AuxComPortInstance
     // between AuxCom_InitStruct() and AuxCom_Start(), which happens HERE:
     KeyerGUI_ParseAuxComParams( pAuxComCfg->sz40Params, pAuxComPort );
     pAuxComPort->iNumDatabits = pAuxComCfg->iNumDatabits;
     pAuxComPort->iParity      = pAuxComCfg->iParity;
     pAuxComPort->iNumStopbits = pAuxComCfg->iNumStopbits;
     pAuxComPort->iTunnelIndex = pAuxComCfg->iTunnelIndex;
     pRigctrlPort = &pRC->PortInstance[ iAuxCom/*0..n*/ + RIGCTRL_PORT_AUX_COM_1];
     pRigctrlPort->fActAsServer = FALSE; // assume THIS PORT does NOT act as server (may be revised below)
     pRigctrlPort->fTrafficMonitorEnabled = ( pAuxComPort->dwParams & AUX_COM_PARAMS_DEBUG ) != 0;
     if(  (pAuxComCfg->iPortNumber >= 0 ) && (pAuxComCfg->iPortUsage != RIGCTRL_PORT_USAGE_NONE) )
      { // Shortly before AuxCom_Start().. do we need the help of RigControl.c to decode
        //         the traffic on thus Aux COM Port ?
        if( pAuxComCfg->iPortUsage == RIGCTRL_PORT_USAGE_SERIAL_TUNNEL )
         {
           switch( pAuxComPort->dwParams & AUX_COM_PARAMS_MASK_PROTOCOL )
            { case AUX_COM_PARAMS_PROTOCOL_ICOM :
                 pRigctrlPort->iRadioCtrlProtocol = RIGCTRL_PROTOCOL_ICOM_CI_V;
                 pRigctrlPort->iRadioMasterAddr = RIGCTRL_DEF_ADDR_AUTO_DETECT; // <- THIS IS ONLY FOR "RIGCTRL_PORT_USAGE_SERIAL_TUNNEL" !
                 pRigctrlPort->iRadioDeviceAddr = RIGCTRL_DEF_ADDR_AUTO_DETECT; // <- THIS IS ONLY FOR "RIGCTRL_PORT_USAGE_SERIAL_TUNNEL" !
                 break;
              case AUX_COM_PARAMS_PROTOCOL_KENWOOD     :
              case AUX_COM_PARAMS_PROTOCOL_YAESU_ASCII :
                 pRigctrlPort->iRadioCtrlProtocol = RIGCTRL_PROTOCOL_KENWOOD;
                 break;
              case AUX_COM_PARAMS_PROTOCOL_YAESU_5_BYTE:
                 pRigctrlPort->iRadioCtrlProtocol = RIGCTRL_PROTOCOL_YAESU_5_BYTE;
                 break;
              default:  // no particular PROTOCOL specified for this port ?
                 // leave pRC->fListenOnlyMode unchanged. It may have already been
                 // set to TRUE for another tunnel that DOES decode rig-control-traffic !
                 break;
            } // end switch( pAuxComPort->dwParams & AUX_COM_PARAMS_MASK_PROTOCOL )
         }   // end if( pAuxComCfg->iPortUsage == RIGCTRL_PORT_USAGE_SERIAL_TUNNEL )
        else if( pAuxComCfg->iPortUsage == RIGCTRL_PORT_USAGE_VIRTUAL_RIG )
         { // In this case, RigControl.c must NOT operate in "Listen-Only Mode",
           // but ACTIVELY control the radio (on the RADIO CONTROL port).
           // On the AUX COM PORT, RigControl shall not act as CLIENT (controlling a radio)
           //    but as a SERVER (emulating a radio, controlled by an external client):
           pRC->fListenOnlyMode = FALSE; // RigControl shall ACTIVELY control the radio..
           pRigctrlPort->fActAsServer = TRUE; // .. but THIS PORT shall EMULATE a radio, and act as a server for an external client ("controller")
           switch( pAuxComPort->dwParams & AUX_COM_PARAMS_MASK_PROTOCOL )
            { case AUX_COM_PARAMS_PROTOCOL_ICOM :
                 pRigctrlPort->iRadioCtrlProtocol = RIGCTRL_PROTOCOL_ICOM_CI_V;
                 pRigctrlPort->iRadioMasterAddr = RIGCTRL_DEF_ADDR_AUTO_DETECT; // <- THIS IS ONLY FOR RIGCTRL_PORT_USAGE_VIRTUAL_RIG ! ! !
                 pRigctrlPort->iRadioDeviceAddr = RIGCTRL_DEF_ADDR_AUTO_DETECT; // <- THIS IS ONLY FOR RIGCTRL_PORT_USAGE_VIRTUAL_RIG ! ! !
                 break;
              case AUX_COM_PARAMS_PROTOCOL_KENWOOD     :
              case AUX_COM_PARAMS_PROTOCOL_YAESU_ASCII :
                 pRigctrlPort->iRadioCtrlProtocol = RIGCTRL_PROTOCOL_KENWOOD;
                 break;
              case AUX_COM_PARAMS_PROTOCOL_YAESU_5_BYTE:
                 pRigctrlPort->iRadioCtrlProtocol = RIGCTRL_PROTOCOL_YAESU_5_BYTE;
                 break;
              default:  // no explicit PROTOCOL specified for this port ? Use the same as for the 'real radio' :
                 pRigctrlPort->iRadioCtrlProtocol= CwKeyer_Config.iRadioControlProtocol;
                 pRigctrlPort->iRadioMasterAddr = RIGCTRL_CIV_MASTER_ADDR_MIN; // <- usually 0xE0, but wfview used 0xE1 (doesn't seem to matter as long as it's different from the RADIO DEVICE ADDRESS)
                 pRigctrlPort->iRadioDeviceAddr = CwKeyer_Config.iRadioCIVAddress;
                 pAuxComPort->dwParams &= ~AUX_COM_PARAMS_MASK_PROTOCOL;
                 // Also let the T_AuxComPortInstance know the RIG CONTROL PROTOCOL.
                 // It needs this for a proper function of AuxCom_GetCurrentStatusAsString():
                 switch( CwKeyer_Config.iRadioControlProtocol )
                  { case RIGCTRL_PROTOCOL_ICOM_CI_V: pAuxComPort->dwParams |= AUX_COM_PARAMS_PROTOCOL_ICOM; break;
                    case RIGCTRL_PROTOCOL_KENWOOD  : pAuxComPort->dwParams |= AUX_COM_PARAMS_PROTOCOL_KENWOOD; break;
                    case RIGCTRL_PROTOCOL_YAESU_5_BYTE: pAuxComPort->dwParams |= AUX_COM_PARAMS_PROTOCOL_YAESU_5_BYTE; break;
                    case RIGCTRL_PROTOCOL_YAESU_ASCII: pAuxComPort->dwParams |= AUX_COM_PARAMS_PROTOCOL_YAESU_ASCII; break;
                    default: break;
                  }
                 break;
            } // end switch( pAuxComPort->dwParams & AUX_COM_PARAMS_MASK_PROTOCOL )
         }   // end if( pAuxComCfg->iPortUsage == RIGCTRL_PORT_USAGE_VIRTUAL_RIG )

        HERE_I_AM__GUI();
        if( ! AuxCom_Start( pAuxComPort, // <- actually 'OPEN PORT and START WORKER THREAD' ...
          pAuxComCfg->iPortNumber,  // [in] "COM port number", e.g. 8 for "COM8"
          pAuxComCfg->iPortUsage,    // [in] function assigned to this port, e.g. RIGCTRL_PORT_USAGE_WINKEYER_EMULATOR
          pAuxComCfg->iBitsPerSecond // [in] iBitsPerSecond (also stored in, and loaded from, RCWK's config file)
          ))
         { // Something went wrong .. if details are known (in AuxComPorts.c), shown them in the GUI:
           HERE_I_AM__GUI();
           if( pAuxComPort->sz255LastError[0] != '\0' ) // <- set in e.g. KeyerThread.c : OpenAndConfigureSerialPort()
            { ShowError(ERROR_CLASS_INFO|SHOW_ERROR_TIMESTAMP, pAuxComPort->sz255LastError );
              // For example, when tested with a deliberately wrong COM port number,
              // AuxComPorts[iAuxCom].sz255LastError was "Could not open serial port 'COM235'" .
              pAuxComPort->sz255LastError[0] = '\0';  // "done" (reported this; don't report this again)
            }
         }
        else
         { HERE_I_AM__GUI();
           ShowError( ERROR_CLASS_INFO | SHOW_ERROR_IN_RUN_LOG | SHOW_ERROR_TIMESTAMP,
              "Opened COM%d, used as %s.",  // similar "info format" as in KeyerThread.c : CwKeyer_Start()
              (int)CwKeyer_Config.sAuxCom[iAuxCom].iPortNumber,
              RigCtrl_PortUsageToString(CwKeyer_Config.sAuxCom[iAuxCom].iPortUsage) );
         }
      }
     else // ! (  (pAuxComCfg->iPortNumber >= 0 ) && (pAuxComCfg->iPortUsage != RIGCTRL_PORT_USAGE_NONE) ) ->
      { HERE_I_AM__GUI();
        AuxCom_Stop( &AuxComPorts[iAuxCom] );
        HERE_I_AM__GUI();
      }
   }
#endif // SWI_NUM_AUX_COM_PORTS > 0 ?

  HERE_I_AM__GUI();  // 2025-08-22: Crashed very shortly after this line
  if( CwKeyer_Start() )  // try to start the CW keyer.  [in] CwKeyer_Config ...
    { //  includes serial port settings like CwKeyer_Config.iComPortNumber_IN .
      // Successfully opened the two serial ports and launched the worker thread
      // -> automatically switch from the "Settings"- to the "Debug"-tab,
      //    where the 'Remote CW Keyer test' will report what's going on.
      HERE_I_AM__GUI();
      if( KeyerGUI_iConflictsFromSettingsTab != 0 )  // conflicts e.g. on the "I/O Config" tab ?
       { // The CW keyer may work despite that (e.g. if a less important input cannot be polled),
         // but to indicate the problem, switch to the tabsheet with the first
         // visual control highlighted with a RED BACKGROUND :
         SwitchMainTab( KEYER_GUI_MAIN_TAB_CONFIG );

         // Show WHY we switched to this tab, in the upper status line (RichEdit_RxTxInfo) :
         KeyerGUI_ShowInfoInStatusLine( RichEdit_RxTxInfo, KEYER_GUI_RXTXINFO_ERROR_MSG,
            TE( "Please fix conflicts on the 'Config' tab (highlighted red)" ) );

         // If possible, also switch to the SUB-TAB with the first 'conflict':
         if( KeyerGUI_iConflictsFromSettingsTab &
            (   KEYER_GUI_CONFLICT_MORSE_KEY_PORT     // something wrong with the COM port(s) again....
              | KEYER_GUI_CONFLICT_RADIO_CONTROL_PORT //  (plugged the damed USB gadget into a different connector today ?)
              | KEYER_GUI_CONFLICT_DOT_INPUT    // .. or a permanent error, like
              | KEYER_GUI_CONFLICT_DASH_INPUT   //    assigning TWO different functions
              | KEYER_GUI_CONFLICT_PTT_INPUT    //    to the same pin of the same serial port ?
              | KEYER_GUI_CONFLICT_TEST_INPUT
              | KEYER_GUI_CONFLICT_KEY_INTERFACE_SUPPLY
              | KEYER_GUI_CONFLICT_RADIO_INTERFACE_SUPPLY
              | KEYER_GUI_CONFLICT_RADIO_CW_KEYING_OUTPUT
              | KEYER_GUI_CONFLICT_RADIO_PTT_OUTPUT ) )
          { PC_Config->ActivePage = TS_IO;
          }
         else if( KeyerGUI_iConflictsFromSettingsTab &
            (   KEYER_GUI_CONFLICT_AUX_COM_PORT_1  // something wrong on the 'Additional COM Ports' tab ?
              | KEYER_GUI_CONFLICT_AUX_COM_PORT_2
              | KEYER_GUI_CONFLICT_AUX_COM_PORT_2   ) )
          { PC_Config->ActivePage = TS_AuxPorts;
          }
         else if( KeyerGUI_iConflictsFromSettingsTab & KEYER_GUI_CONFLICT_RIG_CONTROL )
          { // assume the reason is on the "Rig Control" tab
            PC_Config->ActivePage = TS_RigControl;
          }
       } // end if( KeyerGUI_iConflictsFromSettingsTab != 0 )
      else
      if( (CwKeyer_Config.iRadioKeyingAndControlPort > 0 ) // does this instance control a "locally connected" radio ?
        ||(MyCwNet.cfg.iFunctionality == CWNET_FUNC_CLIENT ) ) // does this instance control a radio REMOTELY ?
       { SwitchMainTab( KEYER_GUI_MAIN_TAB_DEBUG ); // here: after successful call of CwKeyer_Start()
         // ,-----------'
         // '--> Will automatically switch back from the DEBUG tab to the TRX tab as soon as
         //      MyCwNet.RigControl.iParameterPollingState enters RIGCTRL_POLLSTATE_DONE,
         //      or as soon as MyCwNet.Client[0].iClientState changes to CWNET_CLIENT_STATE_LOGIN_CFMD .
         //                            |_______|---> that's THIS INSTANCE, when acting as a client.
       }
      else // neither controlling a radio LOCALLY nor REMOTELY (as client), so:
       { SwitchMainTab( KEYER_GUI_MAIN_TAB_TEST ); // without SDR-like TRX control, switch to the "Test" tab with the 'CW timing' scope
       }
      ShowError(ERROR_CLASS_INFO|SHOW_ERROR_TIMESTAMP, "Keyer thread running, %d WPM.",
       (int)Elbug_DotTimeInMillisecondsToWordsPerMinute(CwKeyer_Elbug.cfg.iDotTime_ms) );
      // Even after CwKeyer_Start() returned TRUE, it may have reported
      // a 'minor problems' via CwKeyer_sz255LastError, so check this:
      if( CwKeyer_sz255LastError[0] != '\0' ) // <- set in e.g. KeyerThread.c : OpenAndConfigureSerialPort()
       { ShowError(ERROR_CLASS_INFO|SHOW_ERROR_TIMESTAMP, CwKeyer_sz255LastError );
         CwKeyer_sz255LastError[0] = '\0';  // "done" (reported this; may be set in OTHER THREADS again)
       }

      // If remote operation via NETWORK is also configured,
      // also start the 'CW Network' client or server (only ONE of them) :
      if( MyCwNet.cfg.iFunctionality != CWNET_FUNC_OFF )
       { if( CwNet_Start( &MyCwNet ) )
          { ShowError(ERROR_CLASS_INFO | SHOW_ERROR_TIMESTAMP,
                      "%s", CwNet_GetCurrentStatusAsString(&MyCwNet) );
            // Result e.g. "CW-Server: Running, ..."
          }
         else // "network" (client or server) didn't start .. anyway, keep going
          { ShowError(ERROR_CLASS_WARNING | SHOW_ERROR_TIMESTAMP,
                      "Network Error: %s", CwNet_GetLastErrorAsString(&MyCwNet) );
            // Result e.g. "CW-Server: can't listen on port 7355 (..)"
          }
       } // end if < start with any of the 'network' functionalities active > ?
    }
   else // Failed to open a port, or failed to launch the worker thread :
    { // -> ALSO switch from the "Settings"- to the "Debug"-tab, and show
      //    what went wrong :
      SwitchMainTab( KEYER_GUI_MAIN_TAB_DEBUG ); // here: after CwKeyer_Start() indicated a problem
      ShowError(ERROR_CLASS_FATAL|SHOW_ERROR_TIMESTAMP, CwKeyer_sz255LastError );
      // ,----------------------------------------------'
      // '--> e.g. "Could not open serial port XYZ", set in KeyerThread.c : OpenAndConfigureSerialPort()
    }
  m_fRestartKeyerThread = FALSE;  // "done" (restarted the keyer)

# if( SWI_USE_HAMLIB_SERVER )
  // For safety, stop, reconfigure, and optionally restart the Hamlib server:
  HERE_I_AM__GUI();
  HLSRV_Stop( &MyHamlibServer );
  if(  CwKeyer_Config.fEnableBuiltInHamlibServer
   && (MyHamlibServer.cfg.iServerListeningPort > 0 ) )
   { HERE_I_AM__GUI();
     if( HLSRV_Start( &MyHamlibServer ) )
      { ShowError(ERROR_CLASS_INFO | SHOW_ERROR_TIMESTAMP, "%s",
          HLSRV_GetCurrentStatusAsString(&MyHamlibServer) );
      }
     else // built-in "Hamlib server" didn't start .. anyway, keep going, it's not essential
      { ShowError(ERROR_CLASS_WARNING | SHOW_ERROR_TIMESTAMP,
          "HL-Server Error: %s", HLSRV_GetLastErrorAsString(&MyHamlibServer) );
      }
   }
  HERE_I_AM__GUI();
# endif // SWI_USE_HAMLIB_SERVER ?

} // end TKeyerMainForm::StartKeyerAndShowInfo()

//---------------------------------------------------------------------------
void TKeyerMainForm::UpdateStatusIndicator(void)
  // Periodically called from Timer1Timer() .
{ static int iStatusIndicatorUsage = 0;
  static int iStatusIndicatorMemIdx = -1;

  KeyerGUI_UpdateStatusIndicatorText(); // -> KeyerGUI_sz80StatusIndicatorText, etc..

  if( (iStatusIndicatorUsage  != KeyerGUI_iStatusIndicatorUsage) // need to update the "GUI" ?
    ||(iStatusIndicatorMemIdx != KeyerGUI_iStatusIndicatorMemIdx) )
   { iStatusIndicatorUsage  = KeyerGUI_iStatusIndicatorUsage;
     iStatusIndicatorMemIdx = KeyerGUI_iStatusIndicatorMemIdx;

     Pnl_StatusIndicator->Caption = KeyerGUI_sz80StatusIndicatorText;
     if( (KeyerGUI_iStatusIndicatorUsage & STATUS_INDICATOR_ON_AIR )
       ||(MyCwNet.RigControl.iTransmitReqst>0) ) // <- set via RigCtrl_SetTransmitRequest() when "we" asked the rig-controller to transmit
      { Pnl_StatusIndicator->Color = clRed;
        // The VCL's stoneage "TPanel" is one of the few components that allows
        // assigning a colour for a background. A crazy "TButton" cannot do this!
      }
     else if((KeyerGUI_iStatusIndicatorUsage & STATUS_INDICATOR_SOMEONE_ELSE_ON_AIR ) // "someone else connected to the rig is on air"
          || (KeyerGUI_iStatusIndicatorUsage & STATUS_INDICATOR_ON_AIR_SIM)
          || (MyCwNet.RigControl.iTransmitting > 0 ) ) // <- set when THE RIG ITSELF transmits, see spec in RigControl.h
      { Pnl_StatusIndicator->Color = (TColor)0x00C0FF; // Borland have forgotten to define clOrange (between clYellow and clRed)
      }
     else
      { Pnl_StatusIndicator->Color = g_clMenuBackground;
      }
   } // end if < KeyerGUI_iStatusIndicatorUsage modified since the last call > ?

} // end TKeyerMainForm::UpdateStatusIndicator()

//---------------------------------------------------------------------------
void TKeyerMainForm::PauseTransmissionAndClearTxBuffers(void) // Called on e.g. pressing ESCAPE.
  // Intended as an 'operator panic key' to stop (or at least pause) transmission
  // when e.g. a wrong COM port has been selected for the local key adapter,
  // and the paddle input states indicate 'send a dash' or 'send a dot'.
  // Since 2025-09, also clears the TX buffer(s), including the TRichEdit that feeds them.
{
  if( ! CwKeyer_Elbug.fPauseTransmitter )
   { CwKeyer_Elbug.fPauseTransmitter = TRUE; // "pause transmission" / "operator panic"
     ShowError(ERROR_CLASS_INFO, "TX paused - click on 'paused' indicator to resume normal operation." );
     CwKeyer_fUpdateAllOutputs = TRUE; // on "Pause Transmission", send all outputs at least ONCE
     // Clear the MORSE CODE TRANSMIT BUFFERS ...
     CwGen_ClearReplayBuffer( &CwKeyer_Gen );
     // .. and the 'type-ahead' EDIT FIELD that usually feeds it (RichEdit_RxTxInfo,
     //    via KeyerGUI_TransferCharsFromEditFieldToCwGenerator() ) :
     RichEdit_RxTxInfo->Clear();
     KeyerGUI_iTxEditorCharIndex_CwGenerator= 0; // index of the next character to move from the type-ahead buffer into the CW Generator
     KeyerGUI_iTxEditorCharIndex_StartOfMsg = 0; // index of the FIRST character to send from the type-ahead input field
   }


} // end TKeyerMainForm::PauseTransmissionAndClearTxBuffers()

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Pnl_StatusIndicatorClick(TObject *Sender)
{
  // Depending on what the 'status indicator' currently indicates,
  //        clicking the indicator may ...
  switch( KeyerGUI_iStatusIndicatorUsage )
   { case STATUS_INDICATOR_PAUSED:  // ... "unpause" the keyer (unlock the "Operator Panic"-button)
        // the smoke disappeared, the dust has settled, the fuse was replaced,
        // the PA or the TRX cooled down, the antenna has been connected,
        // and the correct 'COM port' for the paddle was found.. so try again:
        if( CwKeyer_Elbug.fPauseTransmitter ) // "pause transmission" / "operator panic"
         { ShowError(ERROR_CLASS_INFO, "Resuming normal operation after being 'paused'." );
           CwKeyer_Elbug.fPauseTransmitter = FALSE;  // allow the keyer to key the TX again
           // '--> The CW keyer task should also poll this flag while sending
           //          'from memory', using CwGen.c instead of Elbug.c !
         }
        break;
     default:
        break;
   }
}


//---------------------------------------------------------------------------
void TKeyerMainForm::AddAudioInputDevice( AnsiString sItemText )
  // Appends an enumerated audio input device to a combo (CB_AudioInputDevice),
  // and if that's the one currently selected (CwKeyer_DSP.sz255AudioInputDevice)
{ CB_AudioInputDevice->Items->Add( sItemText );
  if( sItemText == AnsiString(CwKeyer_DSP.cfg.sz255AudioInputDevice) )
   { CB_AudioInputDevice->ItemIndex = CB_AudioInputDevice->Items->Count - 1;
   }
}

//---------------------------------------------------------------------------
void TKeyerMainForm::AddAudioOutputDevice( AnsiString sItemText )
  // Appends an enumerated audio input device to a combo (CB_AudioOutputDevice),
  // and if that's the one currently selected (CwKeyer_DSP.sz255AudioOutputDevice)
{ CB_AudioOutputDevice->Items->Add( sItemText );
  if( sItemText == AnsiString(CwKeyer_DSP.cfg.sz255AudioOutputDevice) )
   { CB_AudioOutputDevice->ItemIndex = CB_AudioOutputDevice->Items->Count - 1;
   }
}


//---------------------------------------------------------------------------
void TKeyerMainForm::UpdateAudioDeviceCombos(void)
{ // "Borrowed" from Spectrum Lab, but stripped down for 'normal audio devices'..
  int i, iAudioDeviceID, iAudioDeviceType, n_good_devs;
  char *cp, *pszDeviceNameWithoutPrefix, sz80Temp[84];
#if( SWI_USE_WAVE_AUDIO ) // use the ancient "Windows Multimedia Extensions" / "wave audio" ?
  WAVEINCAPS  my_waveincaps;
  WAVEOUTCAPS my_waveoutcaps;
  int n_devs;
#endif // SWI_USE_WAVE_AUDIO ?

  // int  nEntriesFound = 0;
  AnsiString s;

  HERE_I_AM__GUI();
  ++KeyerGUI_iUpdating;

  // Try to increase the DROPPED-DOWN width (the VCL doesn't allow this,
  // but Windows may support it, since "Vista" ... )
  // > Setting the drop down width is as simple as sending the
  // > CB_SETDROPPEDWIDTH message to the controls handle:
  SendMessage( CB_AudioInputDevice->Handle, CB_SETDROPPEDWIDTH, 280/*MinimumWidthInPixels*/, 0);
  SendMessage( CB_AudioOutputDevice->Handle,CB_SETDROPPEDWIDTH, 280/*MinimumWidthInPixels*/, 0);

  //--------- build a list of available AUDIO INPUT DEVICES ------------------
  CB_AudioInputDevice->Items->Clear();
  CB_AudioInputDevice->ItemIndex = -1; // hopefully set to a valid index further below..
  AddAudioInputDevice( "None" );
#if( SWI_USE_DSOUND ) // use the mature "DirectSound" / dsound_wrapper.c ?
  for(i=0; i<MyDirectSound.m_nInputDevices; ++i) // in Microsoft terms, these are "capture devices" ..
   { AddAudioInputDevice( AnsiString(MyDirectSound.m_sInputDevices[i].sz255DevName) );
   }
#elif( SWI_USE_WAVE_AUDIO ) // use the ancient "Windows Multimedia Extensions" / "wave audio" ?
  n_devs = waveInGetNumDevs();
  AddAudioInputDevice( "Default WAVE input" );
  for(i=0; i<n_devs; ++i)
   {
    if( waveInGetDevCaps(     i, // UINT uDeviceID
                 &my_waveincaps, // LPWAVEINCAPS pwic,
             sizeof(WAVEINCAPS)) // UINT cbwic
       == MMSYSERR_NOERROR)
     { // The WAVEINCAPS structure describes the capabilities
       // of a waveform-audio input device.
       // Drivers for the standard multi-media system are identified by NUMBERS.
      if( my_waveincaps.wChannels > 0 )  // "accept" the device if it has more than ZERO CHANNELS
       {
        AddAudioInputDevice( AnsiString(my_waveincaps.szPname) );
        // Note: Names are already UGLILY TRUNCATED by waveInGetDevCaps(),
        //       e.g. "Onboard-Microphone (Realtek Hig" - that's all !
       }
     }
   }
#endif // SWI_USE_DSOUND / SWI_USE_WAVE_AUDIO / .. ?
  if( CB_AudioInputDevice->ItemIndex < 0 ) // no match for CwKeyer_DSP.sz255AudioInputDevice ?
   {  CB_AudioInputDevice->ItemIndex = 0;  // item index for "None"
   }

  //- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // Above: enumerate INPUT audio devices  (bare Win32 "wave audio" API) .
  // Below: enumerate OUTPUT audio devices ...
  //- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  //--------- Build a list of available AUDIO OUTPUT DEVICES -----------------
  CB_AudioOutputDevice->Items->Clear();
  CB_AudioOutputDevice->ItemIndex = -1; // hopefully set to a valid index below
  AddAudioOutputDevice( "None" ); // .. and SELECT this item if it matches CwKeyer_DSP.sz255AudioOutputDevice
#if( SWI_USE_DSOUND ) // use the mature "DirectSound" / dsound_wrapper.c ?
  for(i=0; i<MyDirectSound.m_nOutputDevices; ++i)
   { AddAudioOutputDevice( AnsiString(MyDirectSound.m_sOutputDevices[i].sz255DevName) );
   }
#elif( SWI_USE_WAVE_AUDIO ) // use the ancient "Windows Multimedia Extensions" / "wave audio" ?
  AddAudioOutputDevice( "Default WAVE output" );
  n_devs = waveOutGetNumDevs();
  for( i=0; i<n_devs; ++i)
   {
    if( waveOutGetDevCaps(    i, // UINT uDeviceID
                &my_waveoutcaps, // LPWAVEOUTCAPS pwoc,
            sizeof(WAVEOUTCAPS)) // UINT cbwoc
       == MMSYSERR_NOERROR)
     { // The WAVEOUTCAPS structure describes the capabilities
       // of a waveform-audio input device.
       //
      if( my_waveincaps.wChannels > 0 )  // "accept" the device if it has more than ZERO CHANNELS
       {
         AddAudioOutputDevice( AnsiString(my_waveoutcaps.szPname) );
         // Examples: s = "SoundMAX HD Audio"
       }
     }
   }
#endif // SWI_USE_DSOUND / SWI_USE_WAVE_AUDIO / .. ?
  if( CB_AudioOutputDevice->ItemIndex < 0 ) // no match for CwKeyer_DSP.sz255AudioOutputDevice ?
   {  CB_AudioOutputDevice->ItemIndex = 0;  // item index for "None"
   }
  --KeyerGUI_iUpdating;
  HERE_I_AM__GUI();

} // end TKeyerMainForm::UpdateAudioDeviceCombos()

//---------------------------------------------------------------------------
void TKeyerMainForm::StartVFOEditTimer( int nMilliseconds )
{
  if( g_SpecDispControl.iVFOEditTimer_ms <= 0 )  // change the VFO editor's background colour...
   { switch( g_SpecDispControl.iColourScheme ) // .. depending on the colour scheme..
      { case COLOUR_SCHEME_DEFAULT: // e.g. white background in windows
        default:
           Ed_VFO->Color = (TColor)0xC0FFC0; // light green (0xBBGGRR)
           break;
        case COLOUR_SCHEME_DARK:
           Ed_VFO->Color = (TColor)0x008000; // dark green
           break;
      }
   }
  g_SpecDispControl.iVFOEditTimer_ms = nMilliseconds; // suppress assigning new text to Ed_VFO "while typing" !
} // end TKeyerMainForm::StartVFOEditTimer()

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Ed_VFOChange(TObject *Sender) // USER edits Ed_VFO..
{
  if(KeyerGUI_iUpdating) return;  // don't react on "OnChange" from PROGRAM, not USER
  ApplyNewVFOFreq(); // Ed_VFO -> MyCwNet.RigControl.dblVfoFrequency
  StartVFOEditTimer(10000/*ms*/);
} // end TKeyerMainForm::Ed_VFOChange()

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Ed_VFOKeyDown(TObject *Sender, WORD &Key, TShiftState Shift)
{
  // Intercept certain key codes for the "VFO-like" frequency editor:
  TEdit *pEd = dynamic_cast<TEdit *>(Sender);
  char c, szA[44], *cp, cThouSep;
  int i, slen;
  long   iResult;
  double dblFreqHz, dblFreqIncrement;
  BOOL fSetNewFreq = FALSE;
  BOOL fUpdateEditField = FALSE;

  if( (pEd!=NULL) && (KeyerGUI_iUpdating==0) )  // only on USER INPUT...
   {
     ++KeyerGUI_iUpdating;  // disable certain event handlers

     // Get the current frequency from the visible control:
     dblFreqHz = KeyerGUI_GetFrequencyFromEditField( pEd, pEd->SelStart, &dblFreqIncrement );
     // Modify it with the cursor keys by adding/subtracting a power of ten:
     switch( Key )
      { case VK_UP:
           dblFreqHz += dblFreqIncrement;
           fSetNewFreq = fUpdateEditField = TRUE;
           Key = 0;
           break;
        case VK_DOWN:
           dblFreqHz -= dblFreqIncrement;
           fSetNewFreq = fUpdateEditField = TRUE;
           Key = 0;
           break;
        case VK_RETURN:   // "Finish" the VFO frequency input; apply and convert to standard form
           // ( for example, turn "3M5" into  "3 500 000 Hz" )
           fSetNewFreq = TRUE;
           Key = 0;       // Note: Why does the stupid windows say "Clunk" when pressing the ENTER-key ?!
           break;
        default:    // all other keys will be passed on to the edit control
           break;
      } // end switch( Key )
     if( fSetNewFreq )
      { RigCtrl_SetVFOFrequency( &MyCwNet.RigControl, dblFreqHz );
        // Even if the new frequency hasn't arrived at the remote rig yet,
        // already update the LOCAL frequency display, to have an immediate feedback.
        // On this occasion: Don't show dblFreqHz, but what has possibly been
        //                   LIMITED by RigControl.c :
        if( fUpdateEditField )
         { KeyerGUI_UpdateVFODisplay( Ed_VFO, MyCwNet.RigControl.dblVfoFrequency );
         }
        g_SpecDispControl.iVFOEditTimer_ms = 20000; // suppresses assigning new text to Ed_VFO "while typing" !
      } // end if( fSetNewFreq )
     --KeyerGUI_iUpdating;
   } // end if < sender is a TEdit control >
} // end TKeyerMainForm::Ed_VFOKeyDown()
//---------------------------------------------------------------------------

void __fastcall TKeyerMainForm::EditSuppressEnterKey(TObject *Sender, char &Key)
{
  // Just another kludge for stupid windows behaviour.
  // Without this, Windows goes "Clunk" or "DongDongDong" whenever pressing ENTER
  // (aka "Return") when the focus is on this input field.
  // Copy the name of this method (EditSuppressEnterKey) into the
  // "OnKeyPress"-event of all TEdit controls which shall NOT go "Clunk"/"Dong"!
  if( Key==VK_RETURN )
   {  Key = 0;
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Ed_VFOMouseMove(TObject *Sender,
                                TShiftState Shift, int X, int Y)
{
  // Note: To stop this TEdit-thingy from switching the MOUSE POINTER
  // from the usual 'arrow' into an 'IBeam' when hovering over the field,
  // modify the TEdit's "Cursor" property from "crDefault" to "crArrow"
  // in Borland's 'Object Inspector' !
  // "crDefault" doesn't mean "keep the default mouse cursor shape"
  //  but "let Windows change the pointer to what IT thinks it should look like
  //  when hovering over any kind of EDIT CONTROL (TEdit, TRichEdit)" !
  // The "IBeam" cursor is utterly unsuited for clicking into the upper third,
  // middle third, or lower third of the single-line edit field, which is
  // what we use in KeyerGUI_HandleMouseEventInVfoFreqEditor() to increment/
  // decrement the VFO frequency in a similar way as in WFView .
  KeyerGUI_HandleMouseEventInVfoFreqEditor( Ed_VFO, EVT_MOUSE_MOVE, X, Y );

} // end TKeyerMainForm::Ed_VFOMouseMove()


//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Ed_VFOMouseDown( TObject *Sender,
                  TMouseButton Button, TShiftState Shift, int X, int Y)
  // Inspired by WFView, and possibly more user friendly for touchscreen users:
  // a mouse click (or tap) into the UPPER HALF of a digit INCREMENTS it,
  // a mouse click (or tap) into the LOWER HALF of a digit DECREMENTS it.
  // But, since *our* VFO control is an 'Edit Control' (albeit "single-line"),
  //      this is where it gets really ugly, because we must not interfere
  //      with the default mouse handling (e.g. click into the editor to simply
  //      set the 'text caret' alias 'cursor', until someone decided that the
  //      'cursor' is now a funny name for the MOUSE POINTER..)
{
  // Did the "underlying Windows control" already switch the caret
  // to the clicked position when we receive this event handler call ?  ...
  m_Ed_VFO_iSelStartBeforeDoubleClick = Ed_VFO->SelStart; // .. obviously YES.

  if( ! KeyerGUI_HandleMouseEventInVfoFreqEditor( Ed_VFO, EVT_MOUSE_DOWN, X, Y) )
   { // not a valid "decimal place UP / DOWN click", e.g. clicked into the
     // MIDDLE THIRD -> let the VCL set the text cursor (actually, that ALREADY HAPPENED).
   }
} // end TKeyerMainForm::Ed_VFOMouseDown()

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Ed_VFOMouseUp(TObject *Sender,
                  TMouseButton Button, TShiftState Shift, int X, int Y)
  // Last part of the story that began in Ed_VFOMouseDown() ..
{
  KeyerGUI_HandleMouseEventInVfoFreqEditor( Ed_VFO, EVT_MOUSE_UP, X, Y );
} // end TKeyerMainForm::Ed_VFOMouseUp()

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Ed_VFODblClick(TObject *Sender)
{
   Ed_VFO->SelLength = 0;  // undo the unwanted "select word" :
    // this kind of worked, but the caret was now on the BEGIN OF THE WORD
    // instead of being on the clicked character. Aaargh !
   Ed_VFO->SelStart = m_Ed_VFO_iSelStartBeforeDoubleClick;
}


//---------------------------------------------------------------------------
void KeyerGUI_SetDefaultTextCaretPosInVFOInputField( void )
{ // Called from Keyer_GUI.cpp after "programatically" setting the keyboard
  // focus back to the VFO (frequency) input field.
  // Here, at least in CW mode, the default position is on the 100-Hz-digit.
  // Example: Text in the VFO input field = "_ _7.02900 MHz"
  //                                         |   |   |
  //                                        [0] [4] [8]
  KeyerMainForm->Ed_VFO->SelLength = 0;
  KeyerMainForm->Ed_VFO->SelStart  = 8;
} // end KeyerGUI_SetDefaultTextCaretPosInVFOInputField()


//---------------------------------------------------------------------------
int TKeyerMainForm::IdentifyClientCoord( int iClientX, int iClientY,
         float *pfltXrel, // optional output: relative 'X' coord, 0...1 within the area
         float *pfltYrel) // optional output: relative 'X' coord, 0...1 within the area
  // Identifies the 'area' within Grp_Spectra for a given client coordinate.
  // Return values:  SCREEN_AREA_NOT_DEFINED  (i.e. "not identified")
  //                 SCREEN_AREA_SPECTRUM_GRAPH,
  //                 SCREEN_AREA_FREQUENCY_SCALE,
  //                 SCREEN_AREA_SPECTROGRAM .
{
  int x,y;
  int iSpecDispArea = SCREEN_AREA_NOT_DEFINED;
  float fltXrel, fltYrel;
  fltXrel = fltYrel = 0.0;

  // Check the different 'areas' within the main window, from top to bottom.
  x = iClientX - Img_Spectrum->Left;
  y = iClientY - Img_Spectrum->Top;
  if( x>=0 && x<Img_Spectrum->Width && y>=0 && y<Img_Spectrum->Height )
   {  iSpecDispArea = SCREEN_AREA_SPECTRUM_GRAPH;
      fltXrel = (float)x / (float)Img_Spectrum->Width;
      fltYrel = (float)y / (float)Img_Spectrum->Height;
   }
  if( pfltXrel != NULL )
   { *pfltXrel = fltXrel;
   }
  if( pfltYrel != NULL )
   { *pfltYrel = fltYrel;
   }
  return iSpecDispArea;
} // end TKeyerMainForm::IdentifyClientCoord()


//---------------------------------------------------------------------------
static void KeyerMain_InitRigControl(void) // called from TKeyerMainForm::FormCreate() and after loading a new configuration
{
  int i;
  char sz255Temp[256];

  RigCtrl_Init( &MyCwNet.RigControl, // initialize everything for a "single instance"
     CwKeyer_Config.iRadioControlProtocol, // [in] e.g. RIGCTRL_PROTOCOL_ICOM_CI_V,
     CwKeyer_Config.iRadioCIVAddress,      // [in] e.g.  RIGCTRL_DEF_ADDR_IC_7300, ... RIGCTRL_DEF_ADDR_UNKNOWN_YAESU (?)
     &Keyer_RadioControlPortRxFifo.fifo, &Keyer_RadioControlPortTxFifo.fifo,
     0,      // [in] 0..RIGCTRL_MAX_CLIENT_PORTS
     NULL ); // [in] array, one per client
     // Loaded later from a config file: RigCtrl_TrafficMonitor.iDisplayOptions ..
  MyCwNet.RigControl.dwRigControlFlags |= RIGCTRL_FLAG_WANT_SPECTRUM_DATA;
  MyCwNet.RigControl.dwMessageFilterForLog = CwKeyer_Config.dwMessageFilterForLog;

  RigCtrl_ModifyBitInDWORD( &MyCwNet.RigControl.dwRigControlFlags, RIGCTRL_FLAG_SIDETONE_WHEN_KEYED_ON_RIG,
                             CwKeyer_Config.dwAudioOptions & KEYER_AUDIO_OPTION_SIDETONE_WHEN_KEYED_ON_RIG );

  SL_strncpy( sz255Temp, g_sz255PathToDataFiles, 240 );
  SL_strncat( sz255Temp, "RCWKeyer_Bands.txt", sizeof(sz255Temp)-1 );
  i = RigCtrl_LoadBandsAndFrequenciesFromFile( &MyCwNet.RigControl, sz255Temp );
  // ,--------------------------------------------------------------'
  // '--> e.g. "C:\Ham Radio\Remote CW Keyer\RCWKeyer_Bands.txt"
  //      when installed into the recommended directory (where this application,
  //      and hopefully the user's TEXT EDITOR, will have full R/W access).
  ShowError( ERROR_CLASS_INFO | SHOW_ERROR_IN_RUN_LOG,
       "Loaded %d user-defined bands from %s .", (int)i, sz255Temp );

  // Load 'frequencies of interest' also from the Eibi database,
  //  downloaded by the user from http://www.eibispace.de/ (sorted BY FREQUENCY),
  //  renamed to "Eibi_Frequencies.txt", and placed in the program's own folder:
  SL_strncpy( sz255Temp, g_sz255PathToDataFiles, 240 );
  SL_strncat( sz255Temp, "Eibi_Frequencies.txt", sizeof(sz255Temp)-1 );
  i = FreqList_Load(sz255Temp); // Try to load the Eibi frequency database ...
  if( i>0 )
   { ShowError( ERROR_CLASS_INFO | SHOW_ERROR_IN_RUN_LOG,
       "Loaded %d entries from %s .", (int)i, sz255Temp );
   }
  else
   { ShowError( ERROR_CLASS_INFO | SHOW_ERROR_IN_RUN_LOG,
       "No valid entries in %s .", sz255Temp );
   }


#if( SWI_USE_HAMLIB_SERVER ) // build an application with integrated "Hamlib Net rigctld"-compatible server ?
  HLSRV_LinkToRigControl( &MyHamlibServer, &MyCwNet.RigControl ); // <- must be called AFTER RigCtrl_Init() !
#endif // SWI_USE_HAMLIB_SERVER ?
} // end KeyerMain_InitRigControl()




//---------------------------------------------------------------------------
// Machine-generated functions for the GUI (from Borland's form designer),
//   "manually populated" of course to do something meaningful..
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::FormCreate(TObject *Sender)
{
  char sz255Temp[256], *cp;
  WSADATA wsa;
  int i,n;

  HERE_I_AM__GUI();

  // Due to the use of hardware resources like "COM ports",
  // this application must only run in a SINGLE INSTANCE. Make sure it does:
  UTL_Init();  // <- MAY contain built-in self tests; result in UTL_sz255LastError
  if( UTL_iAppInstance >= 3 ) // oops... too many instances of this program running already !
   {
     ::MessageBox( Handle, "Sorry, only up to THREE instances of this program\n"
                           "may run simultaneously on the same machine (PC)\n"
                           " - which they already do.",
                           "Remote CW Keyer", MB_OK );
     UTL_Exit();
     Application->Terminate(); // > " By calling Terminate rather than freeing
                      // > the application object, you allow the application
                      // > to shut down in an orderly fashion."  - sounds good.
   }

  // There may be a VCL-thing to retrieve the command line,
  //  but we don't use it .. being fed-up with the growing bulk.
  memset( g_sz255CommandLine, 0, sizeof(g_sz255CommandLine) );
  HERE_I_AM__GUI();
  cp = ::GetCommandLineA(); // <- this is ancient Windows, not Borland/Embarcadero
  HERE_I_AM__GUI();
  if( cp != NULL )
   { SL_strncpy( g_sz255CommandLine, cp, 255 );
     // Note: The command line contains the name of the executable being launched.
     //       In other words, it is NEVER EMPTY, even without arguments !
     //       e.g. g_sz255CommandLine = "\"C:\cproj\Remote_CW_Keyer\Remote_CW_Keyer.exe\"\0"
     //       (note the double quote characters INSIDE THE STRING ITSELF,
     //        to tell space characters in file- and directory names
     //        from space characters separating ARGUMENTS in the command line).
     ParseCommandLine(g_sz255CommandLine);
   }
  HERE_I_AM__GUI();
  TIM_Init();  // initialize e.g. the "high-resolution timestamps"

  // Because even "WindowProc" didn't intercept certain keys tried this:
  Application->OnMessage = AppMessage;
#if(defined __clang__) // guess we're compiling with C++Builder V12 or later ->
  // The old global variable(?) 'DecimalSeparator' was unknown. Someone explained:
  //  "the old global variables for format settings have been removed".. so:
  FormatSettings.DecimalSeparator = '.'; // <- still a global-ish proprietary VCL thing
#else // guess we're compiling with old Borland C++ V6 :
  DecimalSeparator = '.'; // a lousy old VCL thing (at least in BCB V6) controlling "FormatFloat()"..
  // don't want to see the rotten ',' when the entire GUI is in english !
  // Without the following, some crappy VCL mechanism (or Windows)
  // kept changing the stupid "DecimalSeparator" back to the stupid COMMA
  // at irregular intervals (reason further below) :
  Application->UpdateFormatSettings = FALSE;
  // Found this by accident in the LAZARUS (free Pascal) forum,
  // about the TRIPLE STUPID "DecimalSeparator" / "decimalSeparator" :
  // > I gather this variable is now deprecated and we should use
  // > formatsettings.defaultseperator.
  // > Now in the scientific world, decimalSeparator is
  // > application specific, not locale specific.
  // > So I need to make sure that all my apps only use '.' wherever
  // > they are used in the world.
  // > The problem with delphi's decimalSeparator variable was that
  // > it didn't obey what I told it to do and it would periodically
  // > set itself back to the locale setting.
  //    ( A-ha ! Can somebody please shoot the guy who implemented it that way ? )
  // > Does the Lazarus settings do this, can I set it only once
  // > at the start of the app or do I have to keep checking it
  // > every time I do a float conversion?
  //      (...)
  // > Until recent, Lazarus did not update Formatsettings when Windows
  // > sent a message that they were changed, but nowadays it does (like Delphi).
  // > > You can prevent this by adding this code [not applicable in BCB]
  // > >     // prevent unwanted re-setting of DefaultFormatsettings
  // > >     Application.UpdateFormatSettings := False;
#endif // not __clang__ ?


#if( SWI_USE_HTTP_SERVER )  // support the optional, built-in HTTP server ?
  // The VCL's "Application"-thingy had an "Icon" property (a "TIcon",
  // which can be IMPORTED VIA OBJECT INSPECTOR IN THE MAIN FORM, "Icon"..).
  // Can we use this directly as our web server's "favicon.ico" ? Of course not,
  // because an icon is not an icon - there are myriads of "icon file types",
  // so convert, using TIconToFaviocon.cpp (unfortunately depends on the VCL):
  if( Application->Icon != NULL )
   { HttpSrv_iSizeofFavicon = TIconToFavicon( KeyerMainForm->Icon, // ex: Application->Icon,
             HttpSrv_bFavicon/*out*/, sizeof(HttpSrv_bFavicon)/*maxlen*/ );
     // (for some reason, "Application->Icon" retrieved a 40*40-pixel-icon,
     //  while  "KeyerMainForm->Icon" retrieves the original 32*32-pixel-icon.)
   }
#endif // SWI_USE_HTTP_SERVER ?

  // Fire up the Windows Socket Services (only ONCE.. thus FROM HERE) :
  HERE_I_AM__GUI();
  WSAStartup(MAKEWORD(2,2),&wsa); // already initialized in the main module
    // ,---------------------'
    // '-> "data structure that is to receive details of the Windows Sockets implementation"

  // To receive certain events for the goddamned "system menu" / "window menu",
  //    for example WM_SYSCOMMAND / SC_MOUSEMENU, had to add the following:
  WindowProc = MyWindowProc; // .. because "AppMessage()" alone was not sufficient

  m_fDontSaveOnExit = FALSE;
  KeyerGUI_iStatusIndicatorUsage = -1;
  KeyerGUI_iStatusIndicatorMemIdx= KEYER_MEMORY_INDEX_NONE;
  TIM_StopStopwatch( &KeyerGUI_swTxFromKeyboard );
  SoundTab_InitCosTable(); // build a cosine lookup table for the DSP -> SoundTab_fltCosTable[SOUND_COS_TABLE_LEN]
#if( SWI_USE_DSOUND ) // use "DirectSound" / dsound_wrapper.c ?
  HERE_I_AM__GUI();
  DSW_Init( &MyDirectSound ); // init our "DirectSound wrapper" instance, first used for the SIDETONE OUTPUT
  DSW_EnumerateDevices( &MyDirectSound ); // .. for the two audio device combos ..
#endif // SWI_USE_DSOUND ?
#if( SWI_USE_HAMLIB_SERVER ) // build an application with integrated "Hamlib Net rigctld"-compatible server ?
  HLSRV_InitInstanceWithDefaults( &MyHamlibServer ); // <- must be called BEFORE loading the 'config' from a file
#endif // SWI_USE_HAMLIB_SERVER ?
  HERE_I_AM__GUI();
  SpecDisp_InitControl( &g_SpecDispControl, 32/*iFreqScaleHeight*/ );
  CwKeyer_SetDefaultConfig( &CwKeyer_Config );
  CwKeyer_InitTimingScope( &CwKeyer_TimingScope  );
  Elbug_InitInstanceWithDefaults( &CwKeyer_Elbug );
  CwGen_InitInstanceWithDefaults( &CwKeyer_Gen   );
  CwDSP_InitInstanceWithDefaults( &CwKeyer_DSP   );
  CwNet_InitInstanceWithDefaults( &MyCwNet );
  CwKeyer_Config.pCwNet = &MyCwNet; // 'marry' the keyer with the 'CW Network'
  MyCwNet.pCwDSP = &CwKeyer_DSP; // 'marry' the audio DSP to the 'CW Network' (for audio streaming)
     // '--> Sometimes, after STEPPING OVER the above instruction, MyCwNet.pCwDSP was still NULL !
     //   Hard to believe but true : Problem fixed by doing a BUILD ALL !
     //   Obviously the ancient IDE (Borland C++ Builder) lost track of header dependencies.

#if( SWI_USE_DSOUND ) // use "DirectSound" / dsound_wrapper.c ?
  CwKeyer_DSP.pDSW = &MyDirectSound; // let the 'DSP' use our DirectSound-wrapper-instance
  // (contains a list of audio-in- and -output devices already 'enumerated')
  if( CwKeyer_DSP.cfg.sz255AudioInputDevice[0] == '\0' )
   { strcpy( CwKeyer_DSP.cfg.sz255AudioInputDevice, "None" );
   }
  if( CwKeyer_DSP.cfg.sz255AudioOutputDevice[0] == '\0' )
   { // Obviously not set yet -> pick an audio output device for 'DirectSound'
     // that usually exists: THE FIRST ONE enumerated by our wrapper !
     if( MyDirectSound.m_nOutputDevices > 0 )
      { strncpy( CwKeyer_DSP.cfg.sz255AudioOutputDevice,
                 MyDirectSound.m_sOutputDevices[0].sz255DevName, 255 );
        // ,-----|_____________________________________________|
        // '--> e.g. "Primary Sound Driver", but ymmv ..
      }
   }
#endif // SWI_USE_DSOUND ?
  ClearRecentFiles();
  LoadSettings( SWI_APP_EXE_NAME".ini", CONFIG_FILE_OPTION_INCLUDE_ALL );
  HERE_I_AM__GUI();
  ++KeyerGUI_iUpdating;  // prevent "interference" from certain VCL event handlers
                  // The anonymous-looking stuff like "Left","Top", etc are
                  // properties of Borland's 'TForm' (or something like that).
  Left  = CwKeyer_Config.iMainWindowLeft;
  Top   = CwKeyer_Config.iMainWindowTop;
  Width = CwKeyer_Config.iMainWindowWidth;
  Height= CwKeyer_Config.iMainWindowHeight;

  // Append a few lines to the "window menu" alias "system menu" alias "control menu"
  //   or whatever some madman decides to call it next (it's the thing that opens
  //   when clicking on the program's own icon in the window title bar,
  //   in our case the 'Morse Paddle' icon).
  HANDLE hMenu = GetSystemMenu( Handle, TRUE/*revert*/ );
  hMenu = GetSystemMenu( Handle, FALSE/*revert*/ );
   // GetSystemMenu(hWnd, bRevert) :
   //    hWnd Identifies the window that will own a copy of the window menu.
   // bRevert Specifies the action to be taken. If this parameter is FALSE,
   //         GetSystemMenu returns the handle of the copy of the window menu
   //         currently in use. The copy is initially identical
   //         to the window menu, but it can be modified.
   //         If this parameter is TRUE, GetSystemMenu resets the window menu
   //         back to the Windows default state. The previous window menu,
   //         if any, is destroyed.
  AppendMenu( hMenu, MF_SEPARATOR, 0, NULL);
  AppendMenu(
          hMenu,              // handle to menu to be changed
          MF_STRING,          // UINT uFlags, menu-item flags
          SYSMENUITEM_HIDE_MAIN_MENU, // menu-item identifier or handle of drop-down menu or submenu
          TE("Hide the MAIN MENU") );   // lpNewItem,  menu-item content
  AppendMenu( hMenu, MF_STRING, SYSMENUITEM_SHOW_MAIN_MENU, TE("Show the MAIN MENU") );
   // '--> Events from this "system menu" aka "window menu" are signalled
   //      via WM_SYSCOMMAND .. they don't have ANYTHING to do with the normal "main menu".
  // ex: AppendMenu( hMenu, MF_STRING, SYSMENUITEM_SETTINGS, TE("&Settings") );
  // ex: AppendMenu( hMenu, MF_STRING, SYSMENUITEM_FUNCTIONS, TE("&Functions") );
  //      '--> would be nice to have, but there doesn't seem a way to open
  //           TMenuItems (part of TMainMenu) like a TPopupMenu .. curse the VCL !
  AppendMenu( hMenu, MF_STRING, SYSMENUITEM_HELP, TE("&Help") );

  m_iMainMenuCountdownTimer = 5000/*ms*/ / Timer1->Interval;
  HideOrShowMainMenu( FALSE/* do *not* hide the main menu during the first few seconds*/ );

  --KeyerGUI_iUpdating; // allow dozens of VCL-"controls" (GUI elements) to react again

  ShowError( ERROR_CLASS_INFO | SHOW_ERROR_IN_RUN_LOG, "Remote CW Keyer compiled %s", __DATE__ );
  if( UTL_sz255LastError[0] != '\0' )
   { ShowError( ERROR_CLASS_INFO | SHOW_ERROR_IN_RUN_LOG,
       "Self test in UTL_Init() failed: %s", UTL_sz255LastError );
   }
  HERE_I_AM__GUI();
  if( UTL_GetPathAndNameOfExecutable( sz255Temp, 230/*maxlen!*/ ) )
   { // '--> e.g. "C:\\cbproj\\Remote_CW_Keyer\\Remote_CW_Keyer.exe" .
     // Replace the name by "Translations.c" (one of the SOURCECODE MODULES) :
     cp = strrchr( sz255Temp, '\\' );
     if( cp != NULL )
      { strcpy( cp, "\\Translations.c" );
        // '--> e.g. sz255Temp = "C:\\cbproj\\Remote_CW_Keyer\\Translations.c"
        //      (As noted in 'Translations.c', placing A COPY of that file
        //       in the same directory as the executable (*.exe)
        //       will REPLACE THE BUILT-IN ("hardcoded") string table.
        //       This simplifies the work of volunteers, and allows them
        //       to test a new language WITHOUT the need to
        //       re-compile / re-build the application. )
        n = APPL_LoadTranslationsFromFile( sz255Temp/*pszPathAndFilename*/ );
        if( n > 0 )
         { ShowError( ERROR_CLASS_INFO | SHOW_ERROR_IN_RUN_LOG,
              "Loaded %d lines of translations from %s .", (int)n, sz255Temp );
           // ,--------'
           // '--> Service for volunteers to TEST TRANSLATIONS without rebuilding the project.
         } // end if < APPL_LoadTranslationsFromFile() successful >
        else
         { ShowError( ERROR_CLASS_INFO | SHOW_ERROR_IN_RUN_LOG,
              "Using the built-in translation table (Translations.c) with %d lines.",
              (int)APPL_GetNumEntriesInTranslationTable(TranslationTable) );
         }
      }
   } // end if < UTL_GetPathAndNameOfExecutable() successful >
  if( g_iNewLanguage != APPL_iLanguage )
   { SetNewLanguage( g_iNewLanguage );  // e.g. switch from ENGLISH to GERMAN
   }
  i = INET_Init(); // returns a NEGATIVE error code when the built-in self-test failed..
  HERE_I_AM__GUI();
  if( i < 0 )
   { ShowError( ERROR_CLASS_INFO | SHOW_ERROR_IN_RUN_LOG,
       "Self test in INET_Init() failed with error code %d", (int)i );
   }

  memset( KeyerGUI_BandComboInfoTable, -1, sizeof(KeyerGUI_BandComboInfoTable) );

#if( SWI_NUM_AUX_COM_PORTS > 0 )
  for(i=0; i<SWI_NUM_AUX_COM_PORTS; ++i )
   { AuxCom_InitStruct( &AuxComPorts[i], // <- doesn't OPEN or START anything yet !
                        &MyCwNet.RigControl,                 // [in] 
                        &MyCwNet.RigControl.PortInstance[i], // [in]
                        &CwKeyer_Gen );      // associate the "CW generator" for the Winkeyer emulation
   }
#endif // SWI_NUM_AUX_COM_PORTS > 0 ?

  KeyerMain_InitRigControl(); // -> RigCtrl_Init(), etc (anything related with &MyCwNet.RigControl)

  // The same annoying story with "flickering screens" in Windows (or the VCL)...
  Img_TimingScopeOnKeyerTab->ControlStyle = Img_TimingScopeOnKeyerTab->ControlStyle << csOpaque; // tried this to reduce flicker...
  Img_TimingScopeOnTestTab->ControlStyle = Img_TimingScopeOnTestTab->ControlStyle << csOpaque;
  Img_Spectrum->ControlStyle  = Img_Spectrum->ControlStyle << csOpaque;
  Img_Waterfall->ControlStyle = Img_Waterfall->ControlStyle << csOpaque;
  Img_FreqScale->ControlStyle = Img_FreqScale->ControlStyle << csOpaque;
  Img_SMeter->ControlStyle    = Img_SMeter->ControlStyle << csOpaque;
  Img_TRX_Bottom->ControlStyle= Img_TRX_Bottom->ControlStyle << csOpaque;

  // If, as happened hundreds of times before, the above (alone) doesn't fix
  // the annoying flicker, it sometimes helped to set 'csOpaque' also for the
  // image's parent, which in this particular case are TTabSheets :
  TS_KeyerSettings->ControlStyle = TS_KeyerSettings->ControlStyle << csOpaque;
  TS_Test->ControlStyle = TS_Test->ControlStyle << csOpaque;
  TS_TRX->ControlStyle  = TS_TRX->ControlStyle << csOpaque;

  // Sometimes a flickering window can drive you mad (as in this case) !
  // Because setting the control style csOpaque above
  // (which should prevent sending those useless WM_ERASEBKGND messages),
  // did *not* cure the flicker (it only fixed it "to some degree"),
  // so added the following ..
  HERE_I_AM__GUI();
  DoubleBuffered=TRUE;  // .. because "csOpaque" alone didn't cure the flicker !
    // '--> One of these ugly C++ inventions.. "DoubleBuffered" looks like a
    //      global variable, but it's a PROPERTY of Borland's "TWinControl" :
    //  > When DoubleBuffered is false, the windowed control paints itself
    //  > directly to the window. When DoubleBuffered is true, the windowed
    //  > control paints itself to an in-memory bitmap that is then used to
    //  > paint the window. Double buffering reduces the amount of flicker
    //  > when the control repaints, but is more memory intensive.
    // Not sure if CHILD CONTROLS (like the bitmap graphics on the "TRX" tab)
    // automatically inherit the "DoubleBuffered" flag from their parents ..


  // Prepare the built-in "Help system" ...
  //  YHF_HELP_Init() expects a full path to the HTML help files.
  // (*) What used to be GetModuleFileName() didn't work with BCB V12:
  //     > [bcc32c Error]  no matching function for call to 'GetModuleFileNameW'
  //     > libloaderapi.h(220): candidate function not viable:
  //     > no known conversion from 'char [256]' to 'LPWSTR' (aka 'wchar_t *')
  //     > for 2nd argument .
  // Eeek.. the old annoyance of "wide characters" versus UTF-8-capable 8-bit C-strings.
  //  > _TCHAR mapping in C++Builder is intended to make it easier to write
  //  > source code that can be used with either wide or narrow strings.
  //  > You can map _TCHAR to either wchar_t or char so that your code uses
  //  > the correct string type required by RAD Studio frameworks and libraries
  //  > or the Windows API, and so that RTL functions float to the correct definition
  //  > (wide or narrow, respectively). To set the _TCHAR maps to option,
  //  > use the Project > Options > C++ (Shared Options) dialog box.
  // WB: We want _TCHAR to be 'char', not 'wchar_t', so:
  //     " _TCHAR maps to :         char"  (way down the scrollable list).
  HERE_I_AM__GUI();
  if( ! GetModuleFileName(  // BCB6: Ok. BCB12: "no matching function for call.." (*)
                  NULL, // handle to module to find filename for
             sz255Temp, // pointer to buffer for module path
                  200 ) )  // size of buffer, in characters
   { // GetModuleFileName() failed  [shit happens; it's Windows] ..
     sz255Temp[0] = 0;
   }
  else  // got a full path to this "Module", e.g. sz255Temp="C:\\cbproj\\Remote_CW_Keyer\\Remote_CW_Keyer.exe" .
   { // Remove the name of the executable, and replace that with the name of the
     // subdirectory with the "help files" (in this case, just "manual") :
     cp = strrchr( sz255Temp, '\\' );
     if( cp==NULL ) // oops, there no backslash in this thing.. someone's got a life
      {             // and uses FORWARD slashes in paths ? ;o)
        cp = strrchr( sz255Temp, '/' );
      }
     if ( cp != NULL ) // ok, got it..
      { strcpy( cp+1, "manual" );
      }
   }
  HERE_I_AM__GUI();
  YHF_HELP_Init(
       Handle,    //  handle of the application's main window (a "HWND"; this "Handle" is a VCL property of the form)
       sz255Temp, //  [in] char *html_file_path, path to html files, e.g. "C:\\cbproj\\Remote_CW_Keyer\\manual" .
       "http://www.qsl.net/dl4yhf/Remote_CW_Keyer/", // alternative WEB LOCATION of the html file(s)
                  // (used if the 'local' html_file_path is invalid, for example
                  //  because only the executable has been unpacked but nothing else)
      MyHelpMap,  //  [in] T_YHF_HelpMapEntry *pHelpMap (translation table for Help Topic IDs)
             1);  //  iLanguageCode, usually the international dialing code

  m_dwHashForSettings = CalcHashForSettings();

  // When first launched with the DEFAULT CONFIGURATION, switch to the
  // "I/O Config" tab, because without the proper setting there the keyer is useless:
  SwitchMainTab( KEYER_GUI_MAIN_TAB_CONFIG ); // here: called from TKeyerMainForm::FormCreate(), show "Config" to let the user select his hardware
  // ,-----------------------'
  // '--> Formerly "I/O Config", now with serveral sub-tabs like "I/O" and "Rig Control":
  PC_Config->ActivePage = TS_IO;

  UpdateAllControlsFromConfig();

  KeyerGUI_iUpdating = 0;
  HERE_I_AM__GUI();

}   // end TKeyerMainForm::FormCreate()

//---------------------------------------------------------------------------
void TKeyerMainForm::UpdateWindowCaption(void) // -> e.g. "Remote CW Keyer [1]  (SERVER)/(Client)  [TX DISABLED]..."
{ AnsiString s = "Remote CW Keyer [" + IntToStr(UTL_iAppInstance+1) + "]";

  switch( m_iFunctionalityInCaption )
   { case CWNET_FUNC_CLIENT :
        s = s + "   (Client)";
        break;
     case CWNET_FUNC_SERVER :
        s = s + "   (Server)";
        break;
     default:  // nothing special to write home about .. just a "computer controlled keyer"
        break;
   }
  // At the risk of cluttering the 'title bar', show if user intervention is required:
  if( m_fMustApplyInfoInCaption )
   { // I/O configuration was modified but "Apply new settings and start" hasn't been clicked yet:
     s = s + "  " + AnsiString(TE("Click 'Apply and Start' when the configuration is finished !") );
   }
  else if( m_fTxDisbledInfoInCaption )
   { // TRANSMISSION currently disabled ( configured by the sysop, or diabled for "practicing" ):
     s = s + "  " + AnsiString(TE("TRANSMIT DISABLED (RX only)") );
   }
  Caption = s;


} // end TKeyerMainForm::UpdateWindowCaption()

//---------------------------------------------------------------------------
void TKeyerMainForm::UpdateAllControlsFromConfig(void) // .. after LoadSettings() ..
{
  int i;
  TEdit *pEdit;
  
  ++KeyerGUI_iUpdating;  // prevent "interference" from certain VCL event handlers
                  // The anonymous-looking stuff like "Left","Top", etc are
                  // properties of Borland's 'TForm' (or something like that).

  for(i=0; i<KEYER_NUM_MESSAGE_MEMORIES; ++i)
   { pEdit = GetEditorForMemoryIndex( i+KEYER_MEMORY_INDEX_1 ) ;
     if( pEdit != NULL ) // only if there really is an "editor" for this message memory..
      { pEdit->MaxLength = KEYER_MAX_CHARS_PER_MEMORY;
        pEdit->Text = CwKeyer_Config.szKeyerMemory[i];
      }
   } // end for < all 'keyer message memories' >
  Ed_MyCall->Text = CwKeyer_Config.sz15MyCall;
  Ed_ErrorHistory->WordWrap = (MyCwNet.cfg.iDiagnosticFlags & CWNET_DIAG_FLAGS_WORD_WRAP) != 0;

  m_fAdaptImageSizesOnTRXTab = TRUE; // call AdaptImageSizesOnTRXTab() if that tab ever gets visible
  KeyerGUI_SetColourScheme(g_SpecDispControl.iColourScheme); // <- sounds easy but turned into a nightmare
  g_SpecDispControl.fClearWaterfall = TRUE; // fill the (empty) waterfall with the scheme-depending background colour (ONCE)

  UpdateSettingsTab();
  UpdateAudioDeviceCombos();
  UpdateNetworkSetup();

  --KeyerGUI_iUpdating; // allow dozens of VCL-"controls" (GUI elements) to react again

} // end TKeyerMainForm::UpdateAllControlsFromConfig()


//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::OnChangeOfAnyRigControlParam(TObject *Sender)
{ // Note: This VCL-"method" is also called "on change" VIA APPLICATION CODE,
  //       e.g. when the program ITSELF *updates* the visual controls !
  if( KeyerGUI_iUpdating == 0 ) // only if the "OnChange"-event was fired on a USER ACTION:
   { UpdateRigControlMethodDependingFields();
     KeyerGUI_fMustApplyConfigAndStart = TRUE;
   }
}

//---------------------------------------------------------------------------
void TKeyerMainForm::StopAndRestart(void)
{
  int iNewConflictsFromSettingsTab;
  HERE_I_AM__GUI();
  CwKeyer_Stop();  // stop the keyer (with the old config, if it was running at all)
  HERE_I_AM__GUI();
  CwDSP_Stop( &CwKeyer_DSP ); // stop the 'audio DSP', too
  HERE_I_AM__GUI();
  RigCtrl_Close( &MyCwNet.RigControl ); // stop controlling the rig (not in RIGCTRL_POLLSTATE_DONE anymore)
  HERE_I_AM__GUI();

  ApplySettingsTab();
  iNewConflictsFromSettingsTab = UpdateSettingsTab(); // kludge to re-enumerate the serial port(s) in the combo lists,
   // for example after unplugging/reconnecting the RADIO's USB-port
   // to find out which of the foolishly-named "COM Ports" is currently connected
   // to the IC-9700, which to the IC-7300, and which to the IC-705.
   // All of what Microsoft calls "friendly names" didn't offer any clue,
   // for example:
   //   * a "Digitus" USB-to-serial-port adapter appeared as "COM12 (USB Serial Port)"
   //   * any if Icom's IC-9700 / IC-7300 / IC-705 appeared as
   //       "COMx (Silicon Labs CP210x USB to UART Bridge)" .
   // Nnnngrrr !
   // To find out if the selected COM port *really* leads to the intended radio,
   //   * select it in the combo list, e.g. "COM5 (Silicon Labs CP210x USB to UART Bridge)"
   //   * UNPLUG the USB cable on the radio's side
   //   * Click on the "Apply / Start" button
   //   * If the unplugged radio was really "COM5 (Silicon Labs CP210x USB to UART Bridge)",
   //     the combo box will now show "COM5 *INVALID*" .
  if( (KeyerGUI_iConflictsFromSettingsTab != 0 ) && (iNewConflictsFromSettingsTab==0) )
   { KeyerGUI_ShowInfoInStatusLine( RichEdit_RxTxInfo, KEYER_GUI_RXTXINFO_PROG_INFO,
         TE( "Conflicts on the 'Config' tab have been solved." ) );
   }
  KeyerGUI_iConflictsFromSettingsTab = iNewConflictsFromSettingsTab;


  KeyerGUI_fMustApplyConfigAndStart = FALSE; // "done" (clear the flag that lets the "Apply new settings and start"-button blink)

  RigCtrl_ClearPortErrorFlags( &MyCwNet.RigControl, RIGCTRL_PORT_RADIO, RIGCTRL_PORT_ERROR_FLAGS_ALL );
  KeyerMain_InitRigControl();
  StartKeyerAndShowInfo();  // <- since 2025-06, includes starting the "additional/auxiliary ports" and their worker threads !
  HERE_I_AM__GUI();

} // end TKeyerMainForm::StopAndRestart()
//---------------------------------------------------------------------------


void __fastcall TKeyerMainForm::Btn_ApplyAndStartClick(TObject *Sender)
{
  ApplySettingsTab();
  StopAndRestart();  // <- also clears KeyerGUI_fMustApplyConfigAndStart to stop 'flashing'
}
//---------------------------------------------------------------------------

void __fastcall TKeyerMainForm::Btn_ConfigTabMenuClick(TObject *Sender)
{
  PM_IO_Config->Popup(Mouse->CursorPos.x, Mouse->CursorPos.y);
}
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::TrackBar_WPMChange(TObject *Sender)
  // Called from Borland's VCL when modifying the CW speed via 'trackbar'
  // (besides that, the speed may be modified by direct input in an EDIT FIELD
  //                - see TKeyerMainForm::Ed_WPMChange() )

{
  int iNewPosition_WPM = TrackBar_WPM->Position;
  int iNewDotTime_ms   = Elbug_WordsPerMinuteToDotTimeInMilliseconds( iNewPosition_WPM );
  if( (KeyerGUI_iUpdating==0) && (iNewDotTime_ms != CwKeyer_Elbug.cfg.iDotTime_ms) )
   { CwKeyer_Elbug.cfg.iDotTime_ms = CwKeyer_Gen.cfg.iDotTime_ms
       = CwKeyer_StraightKeyDecoder.iDotTime_ms = iNewDotTime_ms;
     CwDSP_SetDotTimeForAudioCWDecoder( &CwKeyer_DSP, iNewDotTime_ms );
     ++KeyerGUI_iUpdating;
     Ed_WPM->Text = IntToStr( iNewPosition_WPM );
     --KeyerGUI_iUpdating;
     // Redraw the timing scope a.s.a.p., because a "tick" on the timescale
     // shall represent one dot-time :
     TimingScope_iUpdateCountOnTestTab = -1;  // <- polled in "Timer1Timer()" ..
     TimingScope_iUpdateCountOnKeyerTab= -1;
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Ed_WPMChange(TObject *Sender)
  // Called from Borland's VCL when modifying the CW speed via EDIT FIELD.
  // (besides that, the speed may be modified through a "slider"
  //                - see TKeyerMainForm::TrackBar_WPMChange() )
{
  int iNewPosition_WPM = StrToIntDef( Ed_WPM->Text, 0 );
  int iNewDotTime_ms   = Elbug_WordsPerMinuteToDotTimeInMilliseconds( iNewPosition_WPM );
  if( (KeyerGUI_iUpdating==0) && (iNewPosition_WPM != 0) && (iNewDotTime_ms != CwKeyer_Elbug.cfg.iDotTime_ms) )
   { CwKeyer_Elbug.cfg.iDotTime_ms = CwKeyer_Gen.cfg.iDotTime_ms
       = CwKeyer_StraightKeyDecoder.iDotTime_ms = iNewDotTime_ms;
     CwDSP_SetDotTimeForAudioCWDecoder( &CwKeyer_DSP, iNewDotTime_ms );
     ++KeyerGUI_iUpdating;
     TrackBar_WPM->Position = iNewPosition_WPM;
     // Note: Setting the trackbar position 'programmatically' here
     //       causes Borland's VCL to invoke the trackbar's "OnChange"-handler, too.
     //       Setting KeyerGUI_iUpdating here prevents interference from that handler.
     --KeyerGUI_iUpdating;
     // Also here: Redraw the timing scope a.s.a.p., because a "tick"
     //            on the timescale shall represent one dot-time :
     TimingScope_iUpdateCountOnTestTab = -1;  // <- polled in "Timer1Timer()" ..
     TimingScope_iUpdateCountOnKeyerTab= -1;

   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Exit1Click(TObject *Sender)
{
  HERE_I_AM__GUI();
  Close();
  HERE_I_AM__GUI();
}
//---------------------------------------------------------------------------

void __fastcall TKeyerMainForm::MI_ExitNoSaveClick(TObject *Sender)
{
  HERE_I_AM__GUI();
  m_fDontSaveOnExit = TRUE;
  Close(); // -> TKeyerMainForm::FormClose(), etc etc
  HERE_I_AM__GUI();
}
//---------------------------------------------------------------------------

void __fastcall TKeyerMainForm::FormClose(TObject *Sender, TCloseAction &Action)
{
  int i;
  TEdit *pEdit;
  BOOL fWaitForRigControl = FALSE;

  if( UTL_iAppInstance >= 0 )     // no "premature termination" from FormCreate() ?
   { APPL_fLaunchKeyerOnStart = (CwKeyer_iThreadStatus==KEYER_THREAD_STATUS_RUNNING);
     UTL_WriteRunLogEntry( "Beginning to close the main form, instance #%d ..", (int)UTL_iAppInstance+1 );

     if(   CwKeyer_Config.fTurnRigOffWhenClosing
       && (CwKeyer_Config.iRadioKeyingAndControlPort > 0 ) ) // does this instance control a "locally connected" radio ?
      { // It's UGLY, but in this particular case, the GUI will have to WAIT
        // for a few hundred milliseconds until all those worker threads
        // (especially the KEYER THREAD that also 'talks to the radio')
        // have FINISHED what we only INITIATE below:
        RigCtrl_SwitchPollingState( &MyCwNet.RigControl, RIGCTRL_POLLSTATE_TURN_RIG_OFF );
        // '--> the rest happens in a state machine in RigCtrl_Handler(), or wherever..
        fWaitForRigControl = TRUE;
      } // if( CwKeyer_Config.fTurnRigOffWhenClosing )

     if( fWaitForRigControl )
      { for( i=0; i<10; ++i ) // here the reason why this is UTTERLY UGLY .. WAIT in a Windows Message Handler; oh, oh...
         { RigCtrl_Handler( &MyCwNet.RigControl ); // do whatever it takes to TURN OFF THE RADIO
           Sleep(50);  // imitate 'periodic calls' from TKeyerMainForm::Timer1Timer() for APPROXIMATELY 500 ms
           // (Ugly, but this greatly simplified TESTING, especially how to
           //  reliably TURN THE RADIO ON AGAIN after a new RCW Keyer start.)
         }
      }

     // Put the main form's window position and -size into the old-fashioned
     // configuration structure (because THAT's what SaveSettings() will save,
     // if the 32-bit CRC for the entire CwKeyer_Config is different now..)
     CwKeyer_Config.iMainWindowLeft  = Left;
     CwKeyer_Config.iMainWindowTop   = Top;
     CwKeyer_Config.iMainWindowWidth = Width;
     CwKeyer_Config.iMainWindowHeight= Height;
     CwKeyer_Config.dwMessageFilterForLog = MyCwNet.RigControl.dwMessageFilterForLog;
     for(i=0; i<KEYER_NUM_MESSAGE_MEMORIES; ++i)
      { pEdit = KeyerMainForm->GetEditorForMemoryIndex( i+KEYER_MEMORY_INDEX_1 );
        if( pEdit != NULL ) // only if there really is an "editor" for this message memory..
         { SL_strncpy( CwKeyer_Config.szKeyerMemory[i], AnsiString(pEdit->Text).c_str(), KEYER_MAX_CHARS_PER_MEMORY );
         }
      } // end for < all 'keyer message memories' >
     HERE_I_AM__GUI();
     CwNet_Stop( &MyCwNet );      // stop network activities (if any)
     HERE_I_AM__GUI();
     UTL_WriteRunLogEntry( "Stopping the keyer thread." );
     CwKeyer_Stop();              // stop the keyer / keyer thread / etc before terminating
     HERE_I_AM__GUI();            // 2024-01-06 : Crashed with the infamous "0xFEEEFEEE" shortly AFTER this point..
     UTL_WriteRunLogEntry( "Stopping the DSP thread." );
     CwDSP_Stop( &CwKeyer_DSP );  // stop the 'audio DSP', too (BEFORE "terminating DirectSound")
     HERE_I_AM__GUI();            // 2024-01-06 : Crashed with the infamous "0xFEEEFEEE" shortly BEFORE this point.
#   if( SWI_USE_DSOUND ) // use "DirectSound" / dsound_wrapper.c ?
     UTL_WriteRunLogEntry( "Stopping DirectSound." );
     DSW_Term( &MyDirectSound );  // terminate the D[irect]S[ound]W[rapper]
     HERE_I_AM__GUI();
#   endif // SWI_USE_DSOUND ?
#   if( SWI_NUM_AUX_COM_PORTS > 0 )
     for(i=0; i<SWI_NUM_AUX_COM_PORTS; ++i )
      { AuxCom_Stop( &AuxComPorts[i] );
      }
#   endif // SWI_NUM_AUX_COM_PORTS > 0 ?

     CwKeyer_Config.dwMessageFilterForLog = MyCwNet.RigControl.dwMessageFilterForLog;
     // '--> CwKey_Config is included in CalcHashForSettings(); MyCwNet.RigControl is not.
     if( (!m_fDontSaveOnExit )
      && ((m_dwHashForSettings != CalcHashForSettings()) || (APPL_fSaveSettingsOnExit) ) )
      { HERE_I_AM__GUI();
        UTL_WriteRunLogEntry( "Saving modified settings." );
        SaveSettings( SWI_APP_EXE_NAME".ini", CONFIG_FILE_OPTION_INCLUDE_ALL );
        APPL_fSaveSettingsOnExit = FALSE;  // "done" (flag cleared after SaveSettings())
        HERE_I_AM__GUI();
      }
     UTL_WriteRunLogEntry( "Unloading frequency database." );
     FreqList_Exit();
     UTL_WriteRunLogEntry( "Calling WSACleanup()." );
     WSACleanup(); // we're out of business, Mr. Winsock !
     HERE_I_AM__GUI();
     APPL_UnloadTranslationsFromFile();
     UTL_WriteRunLogEntry( "Last step of closing the main form, also closing the logfile." );
     UTL_Exit();   // we're out of business in Utilities.c !
   } // end if( UTL_iAppInstance >= 0 )
  else
   { UTL_WriteRunLogEntry( "Beginning to close the main form, NO VALID INSTANCE INDEX." );
   }
  HERE_I_AM__GUI(); // arrived in this line ? Survived TKeyerMainForm::FormClose() !
} // end TKeyerMainForm::FormClose()
//---------------------------------------------------------------------------

void __fastcall TKeyerMainForm::MI_RestoreDefaultSettingsClick(TObject *Sender)
{
  BOOL fWasRunning = (CwKeyer_iThreadStatus==KEYER_THREAD_STATUS_RUNNING);
  const char *pszInfo;
  HERE_I_AM__GUI();
  switch( APPL_iLanguage )
   { case 49:
        pszInfo = "Mchten Sie wirklich alle Einstellungen dieses Programms,\n"
                  "inklusive der I/O- und Netzwerk-Konfiguration,\n"
                  "wieder auf die Defaultwerte setzen ?";
        break;
     case 1:
     default:
        pszInfo = "Do you really want to set this program's settings,\n"
                  "including the I/O- and network configuration,\n"
                  "back to the defaults as after a fresh install ?";
        break;
   }
  if( ::MessageBox( Handle, pszInfo, "Remote CW Keyer", MB_ICONQUESTION | MB_YESNO) != ID_YES )
   { return;
   }

  CwKeyer_Stop();       // stop the keyer (with the old config, if it was running at all)
  CwDSP_Stop( &CwKeyer_DSP ); // stop the DSP (with the old config, " " " " )
  HERE_I_AM__GUI();
  CwKeyer_SetDefaultConfig( &CwKeyer_Config );
  Elbug_InitInstanceWithDefaults( &CwKeyer_Elbug );
  UpdateSettingsTab();  // kludge to re-enumerate the serial port(s) in the combo lists,
  HERE_I_AM__GUI();
  if( fWasRunning )
   { HERE_I_AM__GUI();
     CwDSP_Start( &CwKeyer_DSP ); // start the 'audio DSP', possibly with NEW settings
     HERE_I_AM__GUI();
     CwKeyer_Start();  // try to start the CW keyer, now with the NEW settings
     HERE_I_AM__GUI();
     m_fRestartKeyerThread = FALSE;  // "done" (restarted the keyer)
   }
} // end TKeyerMainForm::MI_RestoreDefaultSettingsClick()

//---------------------------------------------------------------------------
void TKeyerMainForm::RecallPrefinedSetting( int iSetting ) // [in] PREDEF_SETTING_....
# define PREDEF_SETTING_NEITHER_PADDLE_NOR_RADIO_KEYING_ADAPTER             0
# define PREDEF_SETTING_SIMPLE_PADDLE_ADAPTER_WITH_SEPARATE_PORT_FOR_KEYING 1
# define PREDEF_SETTING_SIMPLE_PADDLE_ADAPTER_WITH_KEYING_OUTPUT_ON_RTS     2
# define PREDEF_SETTING_N1MM_FOOTSWITCH_COMPATIBLE_KEYING_ADAPTER           3
{
  BOOL fWasRunning = (CwKeyer_iThreadStatus==KEYER_THREAD_STATUS_RUNNING);
  const char *pszInfo;
  T_CwKeyerConfig *pConfig = &CwKeyer_Config;
  DWORD dwOldHashForSettings = CalcHashForSettings();

  HERE_I_AM__GUI();
  switch( APPL_iLanguage )
   { case 49:
        pszInfo = "Durch den Abruf der 'vordefinierten Eintellung'\n"
                  "werden die Einstellungen auf der Registerkarte\n"
                  "'I/O-Config berschrieben.\n"
                  "Fortfahren ?";
        break;
     case 1:
     default:
        pszInfo = "By recalling a 'predefined Configuration',\n"
                  "any changes you made earlier on the 'I/O Config' tab\n"
                  "will be overwritten.\n"
                  "Continue ?";
        break;
   }
  if( ::MessageBox( Handle, pszInfo, "Remote CW Keyer", MB_ICONQUESTION | MB_YESNO) != ID_YES )
   { return;
   }

  CwKeyer_Stop();       // stop the keyer (with the old config, if it was running at all)
  CwDSP_Stop( &CwKeyer_DSP ); // stop the DSP (with the old config, " " " " )
  HERE_I_AM__GUI();
  switch( iSetting ) // [in] PREDEF_SETTING_SIMPLE_PADDLE_ADAPTER, etc
   { case PREDEF_SETTING_NEITHER_PADDLE_NOR_RADIO_KEYING_ADAPTER:
        pConfig->iMorseKeyType   = KEY_TYPE_IAMBIC_B;
        pConfig->iDotInput       = KEYER_SIGNAL_INDEX_NONE;
        pConfig->iDashInput      = KEYER_SIGNAL_INDEX_NONE;
        pConfig->iManualPTTInput = KEYER_SIGNAL_INDEX_NONE;
        pConfig->iTestInput      = KEYER_SIGNAL_INDEX_NONE;
        pConfig->iKeySupply      = KEYER_SIGNAL_INDEX_NONE;
        pConfig->iRadioSupply    = KEYER_SIGNAL_INDEX_NONE;
        pConfig->iSidetoneOnTXD  = KEYER_SIDETONE_NONE;
        pConfig->iRadioCWKeying  = KEYER_SIGNAL_INDEX_NONE;
        pConfig->iRadioPTTControl= KEYER_SIGNAL_INDEX_NONE;
        pConfig->iTxDelayTime_ms = 25;
        pConfig->iTxHangTime_ms  = 500;
        pConfig->fDebouncePaddleInputs = FALSE;
        pConfig->fKeyInputWatchdog     = TRUE;
        pConfig->fKeyViaShiftAndControl= FALSE; // ex: = ( pConfig->iComPortNumber_IN < 0 );
        pConfig->fDisableTx = FALSE;
        // With the above setting, both COM PORT SELECTION COMBOS
        // shall not turn RED because they are not used for anything.
        // This limits the features of the REMOTE CW KEYER application
        // to remotely controlling the radio, and maybe listen to what others send.
        break; // end case PREDEF_SETTING_NEITHER_PADDLE_NOR_RADIO_KEYING_ADAPTER

     case PREDEF_SETTING_SIMPLE_PADDLE_ADAPTER_WITH_SEPARATE_PORT_FOR_KEYING:
     case PREDEF_SETTING_SIMPLE_PADDLE_ADAPTER_WITH_KEYING_OUTPUT_ON_RTS:
        pConfig->iMorseKeyType   = KEY_TYPE_IAMBIC_B;
        pConfig->iDotInput       = -KEYER_SIGNAL_INDEX_MORSE_KEY_CTS;
        pConfig->iDashInput      = -KEYER_SIGNAL_INDEX_MORSE_KEY_DSR;
        pConfig->iManualPTTInput = KEYER_SIGNAL_INDEX_NONE; // INPUT from the CW-keyer's side: Per default, 'automatic PTT control' !
        pConfig->iTestInput      = -KEYER_SIGNAL_INDEX_MORSE_KEY_DCD;
        pConfig->iKeySupply      = KEYER_SIGNAL_INDEX_MORSE_KEY_DTR;
        pConfig->iRadioSupply    = KEYER_SIGNAL_INDEX_NONE;
        pConfig->iSidetoneOnTXD  = KEYER_SIDETONE_TXD_480_HZ;
        if( iSetting==PREDEF_SETTING_SIMPLE_PADDLE_ADAPTER_WITH_SEPARATE_PORT_FOR_KEYING )
         { pConfig->iRadioCWKeying  = KEYER_SIGNAL_INDEX_RADIO_KEYING_DTR;
           pConfig->iRadioPTTControl= KEYER_SIGNAL_INDEX_RADIO_KEYING_RTS; // <- default for DL4YHF's "simple keying adapter"
         }
        else // case PREDEF_SETTING_SIMPLE_PADDLE_ADAPTER_WITH_KEYING_OUTPUT_ON_RTS :
         {   // no separate serial port for KEYING, so use RTS on the 'Morse Key' port as on-off keying output:
           pConfig->iRadioCWKeying  = KEYER_SIGNAL_INDEX_MORSE_KEY_RTS;
           pConfig->iRadioPTTControl= KEYER_SIGNAL_INDEX_NONE;
         }
        pConfig->iTxDelayTime_ms = 25;
        pConfig->iTxHangTime_ms  = 500;
        pConfig->fDebouncePaddleInputs = FALSE;
        pConfig->fKeyInputWatchdog     = TRUE;
        pConfig->fKeyViaShiftAndControl= FALSE;
        pConfig->fDisableTx = FALSE;
        break; // end case PREDEF_SETTING_SIMPLE_PADDLE_ADAPTER

     case PREDEF_SETTING_N1MM_FOOTSWITCH_COMPATIBLE_KEYING_ADAPTER:
        // From Brian, N3OC, 2025-09-04 :
        // > I also moved my positive supply voltage to RTS for dot, dash and PTT
        // > because that is the documented source for N1MM footswitch.
        // > I wanted to stay compatible with their wiring. That worked fine too.
        pConfig->iMorseKeyType   = KEY_TYPE_IAMBIC_B;
        pConfig->iDotInput       = -KEYER_SIGNAL_INDEX_MORSE_KEY_DCD;
        pConfig->iDashInput      = -KEYER_SIGNAL_INDEX_MORSE_KEY_CTS;
        pConfig->iManualPTTInput = -KEYER_SIGNAL_INDEX_MORSE_KEY_DSR;
        pConfig->iTestInput      = KEYER_SIGNAL_INDEX_NONE;
        pConfig->iKeySupply      = KEYER_SIGNAL_INDEX_MORSE_KEY_RTS;
        pConfig->iRadioSupply    = KEYER_SIGNAL_INDEX_NONE;
        pConfig->iSidetoneOnTXD  = KEYER_SIDETONE_NONE;
        pConfig->iRadioCWKeying  = KEYER_SIGNAL_INDEX_NONE;  // DTR (output) could be used for this purpose
        pConfig->iRadioPTTControl= KEYER_SIGNAL_INDEX_NONE;
        pConfig->iTxDelayTime_ms = 25;
        pConfig->iTxHangTime_ms  = 500;
        pConfig->fDebouncePaddleInputs = FALSE;
        pConfig->fKeyInputWatchdog     = TRUE;
        pConfig->fKeyViaShiftAndControl= FALSE; // ex: = ( pConfig->iComPortNumber_IN < 0 );
        pConfig->fDisableTx = FALSE;
        break; // end case PREDEF_SETTING_N1MM_FOOTSWITCH_COMPATIBLE_KEYING_ADAPTER

     default:
        break;
   }


  // not called from HERE: Elbug_InitInstanceWithDefaults( &CwKeyer_Elbug );
  UpdateSettingsTab();  // kludge to re-enumerate the serial port(s) in the combo lists
  if( dwOldHashForSettings != CalcHashForSettings() )
   { KeyerGUI_fMustApplyConfigAndStart = TRUE;
   }

  HERE_I_AM__GUI();
  if( fWasRunning )
   { HERE_I_AM__GUI();
     CwDSP_Start( &CwKeyer_DSP ); // start the 'audio DSP', possibly with NEW settings
     HERE_I_AM__GUI();
     CwKeyer_Start();  // try to start the CW keyer, now with the NEW settings
     HERE_I_AM__GUI();
     m_fRestartKeyerThread = FALSE;  // "done" (restarted the keyer)
   }

} // end TKeyerMainForm::RecallPrefinedSetting()

//---------------------------------------------------------------------------
BOOL TKeyerMainForm::DumpRigCtrlTrafficToDebugTab(int iWantedCharsPerLine)
{ char szTextLine[512], szHeadLine[512];
  DWORD dwTextColor  = g_SpecDispControl.clWindowText; // default, depending on the colour scheme ..
  DWORD dwBkgndColor = g_SpecDispControl.clWindowBackground;
  int   iMsgType, iMsgType2, total_length;
  BOOL fAppendedNewText = FALSE;
  // ex: BOOL fBeganUpdate = FALSE;

  while( RigCtrl_ReadTrafficLog( &MyCwNet.RigControl, RIGCTRL_TRAFFIC_READER_GUI,
                                szTextLine, iWantedCharsPerLine, &iMsgType, &dwBkgndColor ) )
   {
  // if( ! fBeganUpdate )
  //  { Ed_ErrorHistory->Lines->BeginUpdate();  // pause screen-update while adding new lines
  //    fBeganUpdate = TRUE;  // flag to only do this ONCE (not for EVERY LINE added)
  //    // (As so many times before with Windows, this did NOT cure the flickering display.
  //    //  Instead, BeginUpdate() / EndUpdate() seemed to INCREASE the CPU load
  //    //  shown in Sysinternals "Process Explorer" :
  //    //  7.7 % CPU load from "Remote_CW_Keyer.exe" WITHOUT BeginUpdate() / EndUpdate(),
  //    //  9.5 % CPU load from "Remote_CW_Keyer.exe" WITH BeginUpdate() / EndUpdate() here,
  //    //  3.0 % CPU load from "Remote_CW_Keyer.exe" after PAUSING the live traffic log display.
  //  }
     iMsgType2 = iMsgType & RIGCTRL_MSGMASK_BASIC;
     dwTextColor = 0x000000;  // .. because dwBkgndColor was for the default "black on white" scheme
     if((((iMsgType & RIGCTRL_MSGTYPE_FLAG_UDP)
         != (m_iPrevMsgTypeInTrafficLog & RIGCTRL_MSGTYPE_FLAG_UDP) )
          ||(m_iPrevMsgTypeInTrafficLog < 0 )
          ||(iMsgType & RIGCTRL_MSGTYPE_FLAG_SUGGEST_NEW_HEADLINE_FOR_TRAFFIC_LOG)
          )
       &&(!(iMsgType  &  RIGCTRL_MSGTYPE_FLAG_ERROR ) )
       &&(  iMsgType2 != RIGCTRL_MSGTYPE_INFO)
        )
      { // Emit a 'headline' before the next entry from the TRAFFIC LOG :
        m_iPrevMsgTypeInTrafficLog = iMsgType;
        // Note: iMsgType should also indicate the PROTOCOL, e.g. via RIGCTRL_MSGTYPE_FLAG_CIV .
        RigCtrl_GetHeadlineForTrafficLog(iMsgType, szHeadLine, iWantedCharsPerLine );  // <- this causes the "extra lines" in the RichText which are NOT in the
           // Default (white) background for COMMENT LINES emitted into the dreadful TRichEdit.
           // (explained further below, for BACKGROUND colours)
        total_length = Ed_ErrorHistory->Text.Length();
        Ed_ErrorHistory->SelStart = total_length;
        Ed_ErrorHistory->SelLength= 1;
        // ex: RichEdit_SetSelBgColor( Ed_ErrorHistory->Handle, 0xFFFFFF/*dwColor:white*/ );
        RichEdit_SetSelColors(Ed_ErrorHistory->Handle,dwTextColor,dwBkgndColor);
        Ed_ErrorHistory->SelText = AnsiString( szHeadLine ) + "\r\n";
        Ed_ErrorHistory->SelLength = 0;
      }
     // Just having different BACKGROUND COLOURS for individual lines
     //      in Borland's "TRichEdit" control is incredibly complex. Principle:
     // 1.) Save the RECENT cursor position before adding the text
     // 2.) Add the new text (line)
     // 3.) Using the stored position from step one, turn the new text into a SELECTION
     // 4.) Perform some voodoo magic on the selection to achieve the desired effect.
     // Note: The stupid 'index' of characters in Borland's "AnsiString" type
     //       begin at ONE (not ZERO as a C programmer would expect) .
     total_length = Ed_ErrorHistory->Text.Length();
     Ed_ErrorHistory->SelStart = total_length;
     Ed_ErrorHistory->SelLength= 1;
     // Change the INDIVIDUAL CHARACTER BACKGROUND COLOUR for the "RichEdit"
     // (but not with Borland's VCL.. it takes more than VCL to get a job done !) :
     // ex: RichEdit_SetSelBgColor( Ed_ErrorHistory->Handle, dwBkgndColor );
     RichEdit_SetSelColors(Ed_ErrorHistory->Handle,dwTextColor,dwBkgndColor);
     Ed_ErrorHistory->SelText = AnsiString( szTextLine ) + "\r\n";
     // NOW, after changing the background colour, de-select the text...
     Ed_ErrorHistory->SelLength = 0;

     // ex: m_fShowingCATTraffic = TRUE; // remember "there is CAT traffic in RichEd_Comms now"
   } // end while < read and display entries from the "CAT traffic monitor" >
  // if( fBeganUpdate )
  //  { Ed_ErrorHistory->Lines->EndUpdate();  // NOW the new lines become visible
  //    fBeganUpdate = TRUE;  // flag to only do this ONCE (not for EVERY LINE)
  //  }

  return fAppendedNewText;
} // end TKeyerMainForm::DumpRigCtrlTrafficToDebugTab()

//---------------------------------------------------------------------------
static int Keyer_GetNextParameterNumberForSlowPollingCycle(void)
  // Periodically called from TKeyerMainForm::Timer1Timer() .. details THERE !
{ static int iItem = 0;
  switch( iItem++ )
   { // Inspired by other software like WSJT-X, WFView, N1MM Logger+,
     // there are a few (!) parameters that we want to poll in a SLOW cycle.
     // More frequently polled parameters like the "S-meter" value are NOT part of this list:
     case 0: return RIGCTRL_PN_AUDIO_VOLUME_PERCENT;     // pRC->iAudioVolume_Percent
     case 1: return RIGCTRL_PN_RF_POWER_SETTING_PERCENT; // pRC->iRFPowerSetting_Percent;  "setting" to avoid confusion with the (on the IC-7300 non-functional) POWER METER MEASUREMENT !
     case 2: return RIGCTRL_PN_RF_GAIN_PERCENT         ; // pRC->iRFGain_Percent
     case 3: return RIGCTRL_PN_SQUELCH_LEVEL_PERCENT   ; // pRC->iSquelchLevel_Percent
     case 4: return RIGCTRL_PN_NOISE_BLANKER_PERCENT   ; // pRC->iNoiseBlanker_Percent;
     case 5: return RIGCTRL_PN_NOISE_REDUCTION_PERCENT ; // pRC->iNoiseReduction_Percent
     case 6: return RIGCTRL_PN_PASSBAND_TUNING_POS1    ; // pRC->iPassbandTuningPos1_Percent; "inner ring" of "TWIN PBT", 50 % = center
     case 7: return RIGCTRL_PN_PASSBAND_TUNING_POS2    ; // pRC->iPassbandTuningPos2_Percent; "outer ring" of "TWIN PBT", 50 % = center
     case 8: return RIGCTRL_PN_CW_PITCH_HZ             ; // pRC->iCWPitch_Hz; here: in Hertz, not the strange CI-V internal unit
     case 9: return RIGCTRL_PN_MIC_GAIN_PERCENT        ; // pRC->iMicGain_Percent
     default: iItem = 0; // end of the "slow polling cycle" ..
             return RIGCTRL_PN_KEYER_SPEED_WPM        ; // pRC->iKeyerSpeed_WPM;  applies to the rig's BUILT-IN keyer
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Timer1Timer(TObject *Sender)
{
  T_ErrorFifoEntry *pErrorFifoEntry;
  DWORD dwColor;  // colour as a simple R-G-B mix - not just another fancy data type
  AnsiString s;
  WORD  wCwChar;  // 'Cw character pattern in Morse code' (not decoded into ASCII yet)
  const char *pszDecoded;
  char  sz255[256], *pszDest, *pszEndstop;
  int   i,n;
  int   iBottomPanelMode = g_SpecDispControl.iBottomPanelMode;
  static int iParamPollingCountdown_ms = 0;
  static int iParamPollingMultiplexer = 0;
  double dblUnixDateAndTime = UTL_GetCurrentUnixDateAndTime(); // .. in UTC SECONDS
  static double dblPrevDateAndTime = 0.0;
  static int iPrevNumCharsToPlay = 0;

  if( floor(dblUnixDateAndTime) != floor(dblPrevDateAndTime) ) // new second -> update clock display
   { UTL_FormatDateAndTime( "YYYY-MM-DD hh:mm:ss", dblUnixDateAndTime, sz255 );
     Lab_DateAndTime->Caption = sz255;
     dblPrevDateAndTime = dblUnixDateAndTime;
   }

  if( m_iMainMenuCountdownTimer > 0 )
   { --m_iMainMenuCountdownTimer;
     if( m_iMainMenuCountdownTimer==0 ) // time for action ?
      { if( g_fHideMenu ) // main menu shall be HIDDEN when not in use for some time ?
         { HideOrShowMainMenu( TRUE/* HIDE the menu after some time of inactivity */ );
         }
      }
   } // end if( m_iMainMenuCountdownTimer > 0 ) ?

  if( m_fInitialising )  // first call ?
   { HERE_I_AM__GUI();
     UpdateTestDisplay();
     if( APPL_fLaunchKeyerOnStart ) // start the keyer IMMEDIATELY (without clicking "Apply new settings and Start", because in the LAST SESSION, it was running properly) ?
      {  StartKeyerAndShowInfo();   // <- clears m_fRestartKeyerThread when successful
      } // end if( APPL_fLaunchKeyerOnStart )
     else // didn't finish the configuration, didn't click "Apply new settings and Start" yet ->
      {  ShowError( ERROR_CLASS_INFO | SHOW_ERROR_TIMESTAMP, "Configuration not finished yet, switching to 'Config' tab." );
         SwitchMainTab( KEYER_GUI_MAIN_TAB_CONFIG ); // go back to the tab with that big button ("Apply new settings and Start") !
      }
     m_fInitialising = FALSE;
   }
  else
   { i = CwNet_SanityCheck( &MyCwNet ); // <- built-in self-text, should return ZERO for "no errors"
     if( i != 0 )
      { ShowError( ERROR_CLASS_FATAL | SHOW_ERROR_TIMESTAMP, "Sanity check failed (code %d)", (int) i );
      }
   }

  if( m_fRestartKeyerThread )  // Some control element in the GUI wants a 'keyer thread restart' ->
   { HERE_I_AM__GUI();
     CwKeyer_Stop();  // stop the keyer (if it was running at all; doesn't hurt if it didn't)
     CwDSP_Stop( &CwKeyer_DSP );    // stop the 'audio DSP'
     CwDSP_Start( &CwKeyer_DSP );   // restart the 'audio DSP'
     CwKeyer_Start(); // start the keyer (~worker thread) with the NEW settings in CwKeyer_Config & Co
     m_fRestartKeyerThread = FALSE; // clear this flag even if CwKeyer_Start() was NOT successful
   }

  // If the 'Radio Keying Port' is valid and ALSO used as a 'Radio Control Port',
  //    periodically call RigControl.c from here :
  if( CwKeyer_Config.iRadioKeyingAndControlPort > 0 ) // does this instance control a "locally connected" radio ?
   { // The user may have plugged his IC-7300 into a different USB port today than where it was yesterday.
     // So the port may be NOT OPEN ANYMORE, or not open right from the start.
     // In that case, inform RigControl.c (!) about the problem:
     if( Keyer_hComPortRadioKeyingAndControl == INVALID_HANDLE_VALUE )
      { // Oops.. the "radio keying- and control-port" should be open but isn't.
        // This may happen after any of the serial port API functions
        // called from KeyerThread.c failed, for example because the
        // USB cable was pulled out, or there was strong RFI on USB, etc.
        // ToDo: Wait for a few of seconds, then try to reopen it ?
        RigCtrl_SetPortErrorFlags( &MyCwNet.RigControl, RIGCTRL_PORT_RADIO, RIGCTRL_PORT_ERROR_FLAG_NOT_OPEN );
                // '--> this stops RigControl.c from writing into the serial port's transmit buffer.
      } // end if( Keyer_hComPortRadioKeyingAndControl == INVALID_HANDLE_VALUE )
     if( MyCwNet.RigControl.iParameterPollingState == RIGCTRL_POLLSTATE_PASSIVE )
      { RigCtrl_StartReading( &MyCwNet.RigControl ); // -> e.g send command "Read the transceiver ID" (0x19 0x00), etc^10 ...
      }
     if( MyCwNet.RigControl.iParameterPollingState == RIGCTRL_POLLSTATE_DONE ) // Time to switch from the "Debug" tab to the "TRX" tab ?
      { if(KeyerGUI_fAutomaticTabSwitching) // allow switching from one tabsheet to another (on the main form) ?
         { if( KeyerGUI_iCurrentMainTab == KEYER_GUI_MAIN_TAB_DEBUG )
            { SwitchMainTab( KEYER_GUI_MAIN_TAB_TRX ); // here: automatically switch to the 'TRX' tab after the Rig Control module indicates 'open for business' (necessary parameters have been successfully polled from the radio)
            }
         }
      }

     if( iParamPollingCountdown_ms > 0 )
      {  iParamPollingCountdown_ms -= Timer1->Interval;
      }
     if( (RigCtrl_GetNumQueuedUpCommandsPending(&MyCwNet.RigControl) < 3 )
       &&(iParamPollingCountdown_ms <= 0 ) )
      { iParamPollingCountdown_ms = 100;
        // Less than 3 commands to READ or WRITE some parameter from/to the rig pending ->
        // Time to poll a few parameter, update several "meter"-readings, etc...
        // WHICH of them ? Depends on the TRANSMIT / RECEIVE state !
        if( MyCwNet.RigControl.iTransmitting > 0 )
         { // Currently TRANSMITTING so we're more interested in the SWR-meter and TX-power than in the "S-meter",
           // but also if we're STILL TRANSMITTING (thus also poll RIGCTRL_PN_TRANSMITTING *while* transmitting):
           switch( iParamPollingMultiplexer++ )
            { case 0: RigCtrl_QueueUpCmdToReadUnifiedPN( &MyCwNet.RigControl, RIGCTRL_PN_POWER_METER_PERCENT ); break;
              case 1: RigCtrl_QueueUpCmdToReadUnifiedPN( &MyCwNet.RigControl, RIGCTRL_PN_SWR_METER_VALUE     ); break;
              case 2: RigCtrl_QueueUpCmdToReadUnifiedPN( &MyCwNet.RigControl, RIGCTRL_PN_ALC_METER_LEVEL     ); break;
              case 3: RigCtrl_QueueUpCmdToReadUnifiedPN( &MyCwNet.RigControl, RIGCTRL_PN_COMP_METER_LEVEL_DB ); break;
              case 4: RigCtrl_QueueUpCmdToReadUnifiedPN( &MyCwNet.RigControl, RIGCTRL_PN_SUPPLY_VOLTAGE_mV   ); break;
              case 5: RigCtrl_QueueUpCmdToReadUnifiedPN( &MyCwNet.RigControl, RIGCTRL_PN_DRAIN_CURRENT_mA    ); break;
              case 6: RigCtrl_QueueUpCmdToReadUnifiedPN( &MyCwNet.RigControl, RIGCTRL_PN_POWER_METER_PERCENT ); break;
              case 7: RigCtrl_QueueUpCmdToReadUnifiedPN( &MyCwNet.RigControl, RIGCTRL_PN_TRANSMITTING        ); break;
              default:
                  RigCtrl_QueueUpCmdToReadUnifiedPN( &MyCwNet.RigControl, Keyer_GetNextParameterNumberForSlowPollingCycle() );
                  // got all parameters we're interested in, so begin a new polling cycle:
                  iParamPollingMultiplexer = 0;
                  break;
            } // end switch( iParamPollingMultiplexer ) during TX
         }
        else // currently not TRANSMITTING but RECEIVING : more frequent updates of the S-Meter ..
         { switch( iParamPollingMultiplexer++ )
            { case 0: RigCtrl_QueueUpCmdToReadUnifiedPN( &MyCwNet.RigControl, RIGCTRL_PN_S_METER_LEVEL_DB ); break;
              case 1: RigCtrl_QueueUpCmdToReadUnifiedPN( &MyCwNet.RigControl, RIGCTRL_PN_TRANSMITTING     ); break;
              case 2: RigCtrl_QueueUpCmdToReadUnifiedPN( &MyCwNet.RigControl, RIGCTRL_PN_S_METER_LEVEL_DB ); break;
              case 3: RigCtrl_QueueUpCmdToReadUnifiedPN( &MyCwNet.RigControl, RIGCTRL_PN_TRANSMITTING     ); break;
              default:
                  RigCtrl_QueueUpCmdToReadUnifiedPN( &MyCwNet.RigControl, Keyer_GetNextParameterNumberForSlowPollingCycle() );
                  // got all parameters we're interested in, so begin a new polling cycle:
                  iParamPollingMultiplexer = 0;
                  break;
            } // end switch( iParamPollingMultiplexer ) during RX
         }
        // Because some poor rigs (like old Yaesu radios) don't send unsoliticted reports like the VFO frequency...
        if( RigCtrl_GetAgeOfBasicParams_ms( &MyCwNet.RigControl ) > 500 ) // make sure at least VFO FREQUENCY and MODE are up-to-date
         { RigCtrl_QueueUpCmdToReadBasicParams( &MyCwNet.RigControl );
         }
      } // end if < time to poll certain parameters from the rig again ? >

     RigCtrl_Handler( &MyCwNet.RigControl ); // periodic transmissions, timeout monitoring, etc..
   }
  else // ( CwKeyer_Config.iRadioKeyingAndControlPort <= 0 ) -> not controlling a "locally connected" radio..
   { if( MyCwNet.cfg.iFunctionality == CWNET_FUNC_CLIENT ) // does this instance act as a CLIENT, and control a radio REMOTELY ?
      { if( MyCwNet.Client[0].iClientState == CWNET_CLIENT_STATE_LOGIN_CFMD ) // Has the remote server has confirmed our log-in ?
         { //       |_______|---> that's THIS INSTANCE, when acting as a client.
           if(KeyerGUI_fAutomaticTabSwitching) // allow switching from one tabsheet to another (on the main form) ?
            { if( KeyerGUI_iCurrentMainTab == KEYER_GUI_MAIN_TAB_DEBUG )
               { SwitchMainTab( KEYER_GUI_MAIN_TAB_TRX ); // here: automatically switch to the 'TRX' tab after successful log-in into a remote server
               }
            }
         }
      } // end if( MyCwNet.cfg.iFunctionality == CWNET_FUNC_CLIENT )
   } // end if ( CwKeyer_Config.iRadioKeyingAndControlPort > 0 ) ?

  KeyerGUI_On50msTimer();  // let the VCL-independent part of the GUI do whatever it needs to (see Keyer_GUI.cpp)

  // Update VCL-dependent part of the GUI (some of that controlled by simple "C"-variables controlled by KeyerGUI_On50msTimer) :
  if(g_SpecDispControl.iVFOEditTimer_ms > 0 ) // local operator still EDITING the value in Ed_VFO !
   { g_SpecDispControl.iVFOEditTimer_ms -= Timer1->Interval; // subtract another GUI-timer-calling interval [ms]
     if(g_SpecDispControl.iVFOEditTimer_ms <= 0 ) // Time to EVALUATE the input from Ed_VFO now ? -> ApplyNewVFOFreq() -> RigCtrl_SetVFOFrequency() !
      { g_SpecDispControl.iVFOEditTimer_ms = 0;
        // (like pressing the ENTER, which almost nobody seems to use anymore)
        Ed_VFO->Color = (TColor)g_SpecDispControl.clWindowBackground;
        ApplyNewVFOFreq();  // Ed_VFO -> MyCwNet.RigControl.dblVfoFrequency
      }
   }


  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // Update the multi-line 'Rich Text' editor for the Error History (and, maybe,
  //             the output from the keyer's built-in Morse decoder, etc, etc).
  //        Fasten seat belts; it's getting messy with the old Borland VCL !
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  if( ! m_fDebugOutputPaused )
   { BOOL fAppendedNewText = FALSE;
     // Before we start emitting all kinds of 'text lines' to the ancient "RichEdit"-thing:
     // What's the optimum NUMBER OF CHARACTERS PER LINE (w/o horizontal scrolling) ?
     int iWantedCharsPerLine = 80;
     int iMaxLinesInHistory = (MyCwNet.cfg.iDiagnosticFlags & CWNET_DIAG_FLAGS_SHORT_HISTORY) ? 100 : 500;
     int nLinesInHistory    = Ed_ErrorHistory->Lines->Count;
     int iCharWidthInPixel = KeyerGUI_GetCharWidthInPixelsForRichEditControl( Ed_ErrorHistory );
     if( iCharWidthInPixel >= 6 )  // seen with "Courier New, Size 12" : 12 pixels per character (pure coindicence)
      { iWantedCharsPerLine = ((Ed_ErrorHistory->Width-30/*pixels for the scrollbar*/) ) / iCharWidthInPixel;
      }
     // Next, try to avoid the old problem with VCL applications crashing
     //            if the text in an editor gets too long:

     while( nLinesInHistory > iMaxLinesInHistory ) // tooo many lines (a rapidly SCROLLING 'RichEdit' is a CPU hog) ?
      { HERE_I_AM__GUI();
        Ed_ErrorHistory->Lines->Delete(0);  // delete the 1st line (a "TAnsiString")
        --nLinesInHistory;  // don't call Ed_ErrorHistory->Lines->Count again.. who knows what it does internally - waste even more CPU time ?
        fAppendedNewText = TRUE;
      }
     fAppendedNewText |= KeyerGUI_DumpErrorHistoryToRichText(Ed_ErrorHistory);
     fAppendedNewText |= DumpRigCtrlTrafficToDebugTab(iWantedCharsPerLine);
     fAppendedNewText |= KeyerGUI_DumpCharsFromTerminalToRichText(Ed_ErrorHistory);

     // Similar as for the thread-safe, non-VCL 'Error History', also dump
     // the CW-keyer's "decoded output" (text) into the same RichEdit control:
     while( (wCwChar=CwKeyer_ReadFromDecoderFifo(&CwKeyer_DecoderFifo, &m_iDecoderFifoTail)) != 0)
      {
        pszDecoded = Elbug_MorseCodePatternToASCII(wCwChar);
        if( MyCwNet.cfg.iDiagnosticFlags & CWNET_DIAG_FLAGS_SHOW_DECODER_OUTPUT )
         { // Again this obfuscated way to append text to a RichEdit control
           //       with a custom background colour :
           Ed_ErrorHistory->SelStart = Ed_ErrorHistory->Text.Length();
           Ed_ErrorHistory->SelLength= 1;
           RichEdit_SetSelColors( Ed_ErrorHistory->Handle,
               g_clDecoderOutputForeground, g_clDecoderOutputBackground );
           Ed_ErrorHistory->SelText = AnsiString( pszDecoded );
           Ed_ErrorHistory->SelLength = 0;
           // ex: Ed_ErrorHistory->SelStart = Ed_ErrorHistory->Text.Length();
           fAppendedNewText = TRUE;
         } // end if < .. CWNET_DIAG_FLAGS_SHOW_DECODER_OUTPUT ? >
        // Regardless of CWNET_DIAG_FLAGS_SHOW_DECODER_OUTPUT (which applies to the DEBUG tab),
        // output drained from CwKeyer_DecoderFifo may be displayed
        // in the combined "Info / RX data / TX data" edit field (RichEdit_RxTxInfo),
        // because that field is ALWAS visible above the tabbed display pages.
        // But avoid interfering with the user TYPING commands or TX-data :
        if( KeyerGUI_iStatusIndicatorMemIdx != KEYER_MEMORY_INDEX_TX_EDITOR )
         { KeyerGUI_AppendDataToRichEditWithRxTxInfo( RichEdit_RxTxInfo, pszDecoded, KEYER_GUI_RXTXINFO_SENT_DATA );
           // Note: KeyerGUI_iStatusIndicatorMemIdx switches from KEYER_MEMORY_INDEX_TX_EDITOR
           //       to KEYER_MEMORY_INDEX_NONE if the transmission of Morse code
           //       from RichEdit_RxTxInfo *ENDS* .. see polling of CW_GEN_FINISHED_SENDING .
           //  Besides that, KeyerGUI_iStatusIndicatorMemIdx may also be modified
           //  from WORKER THREADS, like AuxComThread() when EMULATING A WINKEYER.
         }

        // ToDo: Record the decoded 'Morse traffic' in a logfile ?
      } // end while < more decoded characters or Morse code 'prosigns' >

     // Similar as above for the "locally connected key", also emit
     // characters decoded from the radio receiver's AUDIO SIGNAL:
     while( (wCwChar=CwDSP_ReadFromCwPatternDecoderFifo( &CwKeyer_DSP.AudioCwDecoder[0], &m_iAudioCwDecoderFifoTail)) != 0)
      {
        pszDecoded = Elbug_MorseCodePatternToASCII(wCwChar);
        if( MyCwNet.cfg.iDiagnosticFlags & CWNET_DIAG_FLAGS_SHOW_DECODER_OUTPUT )
         { // Again this obfuscated way to append text to a RichEdit control
           //       with a custom background colour :
           Ed_ErrorHistory->SelStart = Ed_ErrorHistory->Text.Length();
           Ed_ErrorHistory->SelLength= 1;
           RichEdit_SetSelColors( Ed_ErrorHistory->Handle,
               g_clDecoderOutputForeground, g_clDecoderOutputBackground );
           Ed_ErrorHistory->SelText = AnsiString( pszDecoded );
           Ed_ErrorHistory->SelLength = 0;
           // ex: Ed_ErrorHistory->SelStart = Ed_ErrorHistory->Text.Length();
           fAppendedNewText = TRUE;
         } // end if < .. CWNET_DIAG_FLAGS_SHOW_DECODER_OUTPUT ? >
        // Regardless of CWNET_DIAG_FLAGS_SHOW_DECODER_OUTPUT (which applies to the DEBUG tab),
        // output drained from CwKeyer_DecoderFifo may be displayed
        // in the combined "Info / RX data / TX data" edit field (RichEdit_RxTxInfo),
        // because that field is ALWAS visible above the tabbed display pages.
        // But avoid interfering with the user TYPING commands or TX-data :
        if( KeyerGUI_iStatusIndicatorMemIdx != KEYER_MEMORY_INDEX_TX_EDITOR )
         { KeyerGUI_AppendDataToRichEditWithRxTxInfo( RichEdit_RxTxInfo, pszDecoded, KEYER_GUI_RXTXINFO_RXDATA );
         }
        // ToDo: Record the decoded 'Morse traffic' in a logfile ?
      } // end while < more decoded characters from the AUDIO CW DECODER / DSP >



     // Optional hex-dump of the 'Morse keying bytestream' on the 'Debug' tab:
     //  (server-side: RECEIVED bytestream; client-side: SENT bytestream)
     if( MyCwNet.cfg.iDiagnosticFlags & CWNET_DIAG_FLAGS_SHOW_KEYING_BYTESTREAM )
      { pszDest    = sz255;
        pszEndstop = sz255 + 250;
        if( MyCwNet.cfg.iFunctionality == CWNET_FUNC_CLIENT )
         { while( m_iMorseFIFOTailIndex2 != MyCwNet.MorseTxFifo.iHeadIndex )
            { SL_AppendHex( &pszDest, pszEndstop,
                      MyCwNet.MorseTxFifo.elem[m_iMorseFIFOTailIndex2++].bCmd,
                      2/* hex digits */ );
              SL_AppendChar( &pszDest, pszEndstop, ' ' );
              m_iMorseFIFOTailIndex2 &= (CW_KEYING_FIFO_SIZE-1);
            }
         }
        else if( MyCwNet.cfg.iFunctionality == CWNET_FUNC_SERVER )
         { while( m_iMorseFIFOTailIndex2 != MyCwNet.MorseRxFifo.iHeadIndex )
            { SL_AppendHex( &pszDest, pszEndstop,
                      MyCwNet.MorseRxFifo.elem[m_iMorseFIFOTailIndex2++].bCmd,
                      2/* hex digits */ );
              SL_AppendChar( &pszDest, pszEndstop, ' ' );
              m_iMorseFIFOTailIndex2 &= (CW_KEYING_FIFO_SIZE-1);
            }
         }
        if( pszDest > sz255 ) // anything to append (as "coloured hex dump") ?
         { // And again, the obscure way to append text to a RichEdit control
           //     with a custom background colour (half Borland VCL, half Win32 API)
           Ed_ErrorHistory->SelStart  = Ed_ErrorHistory->Text.Length();
           Ed_ErrorHistory->SelLength = 1;
           RichEdit_SetSelColors(Ed_ErrorHistory->Handle,
                    0x000000/*dwTextColor*/, 0xC0C0FF/*dwBkgndColor:light red*/);
           Ed_ErrorHistory->SelText   = AnsiString( sz255 );
           Ed_ErrorHistory->SelLength = 0;
           // ex: Ed_ErrorHistory->SelStart  = Ed_ErrorHistory->Text.Length();
           fAppendedNewText = TRUE;
         }
      } // end if( MyCwNet.iDiagnosticFlags & CWNET_DIAG_FLAGS_SHOW_KEYING_BYTESTREAM )

     // After deleting lines AT THE TOP (to limit the number of lines in the editor),
     // the RichEditor's VERTICAL SCROLLBAR ran havoc:
     // During the manipulation of the text in the editor the scrollbar
     // flipped around like crazy, and after the update, despite the
     // "text caret" / "text curor" / "TRichEdit.SelStart" being AT THE END
     // of the text, the goddamned vertical scrollbar hopped back to the
     // BEGIN of the text - and in between, it caused an awful flicker.
     // Tried to fix this similar as in Spectrum Lab's "debug form" :
     // To make the new caret position visible, ...
     // > The EM_SCROLLCARET message scrolls the caret into view in an edit control.
     // > You can send this message to either an edit control or a rich edit control.
     if( fAppendedNewText )
      { Ed_ErrorHistory->SelLength = 0;
        Ed_ErrorHistory->SelStart  = Ed_ErrorHistory->Text.Length();
        SendMessage( Ed_ErrorHistory->Handle, // handle of destination window
                      EM_SCROLLCARET, // message to send : "scroll into view"
                (WPARAM)0,(LPARAM)0); // both params must be zero
      }

     HERE_I_AM__GUI();
   }   // end if( ! m_fDebugOutputPaused )



  // Update various indicators for the digital in- and outputs
  //        on the 'keyer'- and 'radio' ports.
  //  Note: CwKeyer_GetDigitalInput() can also 'read back' the states
  //        of DIGITAL OUTPUTS (from the application's point of view).
  if( CwKeyer_iTestMode == KEYER_TEST_MODE_OFF )
   { Chk_MorseKeyDTR->Checked = CwKeyer_GetDigitalInput( KEYER_SIGNAL_INDEX_MORSE_KEY_DTR );
     Chk_MorseKeyRTS->Checked = CwKeyer_GetDigitalInput( KEYER_SIGNAL_INDEX_MORSE_KEY_RTS );
   }
  ST_MorseKeyDCD->Color = CwKeyer_GetDigitalInput( KEYER_SIGNAL_INDEX_MORSE_KEY_DCD ) ? clRed : clBtnFace;
  ST_MorseKeyDSR->Color = CwKeyer_GetDigitalInput( KEYER_SIGNAL_INDEX_MORSE_KEY_DSR ) ? clRed : clBtnFace;
  ST_MorseKeyCTS->Color = CwKeyer_GetDigitalInput( KEYER_SIGNAL_INDEX_MORSE_KEY_CTS ) ? clRed : clBtnFace;
  ST_MorseKeyRI->Color  = CwKeyer_GetDigitalInput( KEYER_SIGNAL_INDEX_MORSE_KEY_RI  ) ? clRed : clBtnFace;
  if( CwKeyer_iTestMode == KEYER_TEST_MODE_OFF )
   { Chk_RadioKeyingDTR->Checked = CwKeyer_GetDigitalInput( KEYER_SIGNAL_INDEX_RADIO_KEYING_DTR );
     Chk_RadioKeyingRTS->Checked = CwKeyer_GetDigitalInput( KEYER_SIGNAL_INDEX_RADIO_KEYING_RTS );
   }
  ST_RadioKeyingDCD->Color = CwKeyer_GetDigitalInput( KEYER_SIGNAL_INDEX_RADIO_KEYING_DCD ) ? clRed : clBtnFace;
  ST_RadioKeyingDSR->Color = CwKeyer_GetDigitalInput( KEYER_SIGNAL_INDEX_RADIO_KEYING_DSR ) ? clRed : clBtnFace;
  ST_RadioKeyingCTS->Color = CwKeyer_GetDigitalInput( KEYER_SIGNAL_INDEX_RADIO_KEYING_CTS ) ? clRed : clBtnFace;
  ST_RadioKeyingRI->Color  = CwKeyer_GetDigitalInput( KEYER_SIGNAL_INDEX_RADIO_KEYING_RI  ) ? clRed : clBtnFace;

  // Let the "Apply new settings and start" button on the 'I/O Config' tab flash ?
  if(KeyerGUI_iCurrentMainTab==KEYER_GUI_MAIN_TAB_CONFIG ) // ex: if (PageControl1->ActivePage == TS_Config )
   {
     if( KeyerGUI_fMustApplyConfigAndStart && (KeyerGUI_dwTimerTicks_50ms & 16 ) )
      { // Btn_ApplyAndStart->Font->Style = fsBold; // this would be easy and intuitive, but doesn't compile.
        Btn_ApplyAndStart->Font->Style = TFontStyles() << fsBold; // baaah ...
        // (first "construct" an empty set of "TFontStyles", only to add ONE, for BOLD characters,
        //  because the BACKGROUND COLOUR (aka "button's face") cannot be modified in a VCL-"TButton".
      }
     else
      { Btn_ApplyAndStart->Font->Style = TFontStyles(); // remove the "bold" attribute
      }
     // Other visual indicators on that tab (or sub-tab like "Additional COM Ports"):
     Lab_AuxCom1Info->Caption = AuxCom_GetCurrentStatusAsString( &AuxComPorts[0] );
     Lab_AuxCom2Info->Caption = AuxCom_GetCurrentStatusAsString( &AuxComPorts[1] );
     Lab_AuxCom3Info->Caption = AuxCom_GetCurrentStatusAsString( &AuxComPorts[2] );
   }

  // Also update the GUI's window title (VCL: 'Caption') ?
  if( ( m_iFunctionalityInCaption != MyCwNet.cfg.iFunctionality )
    ||( m_fMustApplyInfoInCaption != KeyerGUI_fMustApplyConfigAndStart )
    ||( m_fTxDisbledInfoInCaption != ( CwKeyer_Config.fDisableTx/*locally*/ || MyCwNet.fDisableTx/*remotely*/ ) )
    )
   {    m_iFunctionalityInCaption = MyCwNet.cfg.iFunctionality;
        m_fMustApplyInfoInCaption = KeyerGUI_fMustApplyConfigAndStart;
        m_fTxDisbledInfoInCaption = CwKeyer_Config.fDisableTx || MyCwNet.fDisableTx;
        UpdateWindowCaption(); // -> e.g. "Remote CW Keyer [1]  (SERVER)/(Client)"
   }


  // The time-consuming periodic update of the CW-'Timing Scope'
  //  only happens when the scope is currently visible (on two of the tabs):
  if(KeyerGUI_iCurrentMainTab==KEYER_GUI_MAIN_TAB_TEST ) // ex: if (PageControl1->ActivePage == TS_Test )
   { // Currently on the "Test" tab with the 'CW timing' scope display:
     // Because updating the 'CW timing scope' display consumed 5 % of the
     // available CPU time (while the 'keyer worker thread' consumed less than 0.2 %),
     // only redraw the display is the SAMPLE DATA have been modified.
     // As long as there is no TRIGGER for the scope, samples remain unmodified,
     // and the timing scope is NOT redrawn.
     if( CwKeyer_TimingScope.iUpdateCount != TimingScope_iUpdateCountOnTestTab )
      { HERE_I_AM__GUI();
        TimingScope_iUpdateCountOnTestTab = CwKeyer_TimingScope.iUpdateCount;
        Img_TimingScopeOnTestTab->Picture->Bitmap->Height = Img_TimingScopeOnTestTab->Height;
        Img_TimingScopeOnTestTab->Picture->Bitmap->Width  = Img_TimingScopeOnTestTab->Width;
        KeyerGUI_UpdateTimingScope( &CwKeyer_TimingScope, Img_TimingScopeOnTestTab->Picture->Bitmap );
      }
     Lab_AuxCom1Signals->Caption = "Aux1:" + AnsiString(AuxCom_SignalStatesToString( &AuxComPorts[0], sz255, 40 ) );
     Lab_AuxCom2Signals->Caption = "Aux2:" + AnsiString(AuxCom_SignalStatesToString( &AuxComPorts[1], sz255, 40 ) );
     Lab_AuxCom3Signals->Caption = "Aux3:" + AnsiString(AuxCom_SignalStatesToString( &AuxComPorts[2], sz255, 40 ) );
   }
  if( PageControl1->ActivePage == TS_KeyerSettings )
   { // Currently on the "Keyer Settings" tab, where we'd also like to see the TIMING:
     if( CwKeyer_TimingScope.iUpdateCount != TimingScope_iUpdateCountOnKeyerTab )
      { HERE_I_AM__GUI();
        TimingScope_iUpdateCountOnKeyerTab = CwKeyer_TimingScope.iUpdateCount;
        Img_TimingScopeOnKeyerTab->Picture->Bitmap->Height = Img_TimingScopeOnKeyerTab->Height;
        Img_TimingScopeOnKeyerTab->Picture->Bitmap->Width  = Img_TimingScopeOnKeyerTab->Width;
        KeyerGUI_UpdateTimingScope( &CwKeyer_TimingScope, Img_TimingScopeOnKeyerTab->Picture->Bitmap);
      }

     // Since 2025-06-14, the keyer's CW Speed may be modified anytime
     //  by an external application,
     //  e.g. from N1MM Logger+ via the 'Winkeyer EMULATOR' on an 'Additional COM Port' !
     // Because that happens in a WORKER THREAD and the VCL isn't thread-safe,
     // added an extra flag to "update the WPM indicator in the GUI",
     // which is POLLED and CLEARED here (in the VCL-safe "OnTimer" method):
     if( KeyerGUI_fUpdateWPMIndicator ) // <- declared as BOOL in "Keyer_GUI_no_VCL.h", not "Keyer_GUI.h" !
      { UpdateWPMIndicatorFromCwKeyerConfig(); // [in] CwKeyer_Elbug.cfg.iDotTime_ms, [out] VCL controls
        KeyerGUI_fUpdateWPMIndicator = FALSE;  // "done"
      }


   } // if( PageControl1->ActivePage == TS_KeyerSettings )
  if( PageControl1->ActivePage == TS_Network )
   { HERE_I_AM__GUI();
     Ed_NetworkStatus->Text = CwNet_GetCurrentStatusAsString( &MyCwNet );
     i = MyCwNet.Client[CWNET_LOCAL_CLIENT_INDEX].iAnnouncedTxClient;
     if(m_iTransmittingClient != i) // show who's "on the key" at the moment
      { m_iTransmittingClient = i;
        if( MyCwNet.cfg.iFunctionality == CWNET_FUNC_SERVER )
         { Ed_OnTheKeyNow->Text = CwNet_GetClientCallOrInfo( &MyCwNet, m_iTransmittingClient );
           // The above only works on the SERVER SIDE. None of the CLIENTS
           // has a list of callsigns (or whatever) currently logged into the server...
         }
        else
         { // ... so instead, for any kind of 'announced info', use this string buffer:
           Ed_OnTheKeyNow->Text = MyCwNet.Client[CWNET_LOCAL_CLIENT_INDEX].sz80TxInfo;
         }
      }
   } // end if( PageControl1->ActivePage == TS_Network )
  if( (PageControl1->ActivePage == TS_TRX )
    ||((CwKeyer_Config.iTRXOptions & KEYER_TRX_OPT_KEEP_RUNNING) != 0) )
   { // Currently on the "Transceiver" tab (TRX), with spectrum / spectrogram;
     // or the 'TRX tab' (with spectrum/spectrogram) shall be updated even when NOT visible...

     if( m_fAdaptImageSizesOnTRXTab ) // call AdaptImageSizesOnTRXTab() .. now that the tab is visible ?
      { AdaptImageSizesOnTRXTab(); // -> Img_Spectrum, Img_Waterfall, Img_FreqScale (and their BITMAPS)
        g_SpecDispControl.fUpdateSpectrum = g_SpecDispControl.fUpdateWaterfall = g_SpecDispControl.fUpdateFreqScale = g_SpecDispControl.fUpdateFrequencyInfo = TRUE;
      }
     if( ( g_SpecDispControl.iDisplayOptions & SPEC_DISP_OPTIONS_CW_DECODER_SPECTRUM )
      && ( g_SpecDispControl.i32AudioSpectrumUpdateCounter != CwKeyer_DSP.i32AudioSpectrumUpdateCounter)
       ) // Even without a new spectrum from the radio, update the spectrum display
      {  // when the AUDIO SPECTRUM is visible *and* was modified :
         g_SpecDispControl.i32AudioSpectrumUpdateCounter = CwKeyer_DSP.i32AudioSpectrumUpdateCounter;
         g_SpecDispControl.fUpdateSpectrum = TRUE;
      }

     UpdateTrxDisplay(); // e.g. VFO frequency and "op mode" (modulation type) ... IF MODIFIED

     if( g_SpecDispControl.fUpdateWaterfall || g_SpecDispControl.fClearWaterfall
      || (MyCwNet.RigControl.iScopeFifoHeadIndex != m_iLastScopeDisplayHeadIndex) )
      { int nNewSpectra = MyCwNet.RigControl.iScopeFifoHeadIndex - m_iLastScopeDisplayHeadIndex;
        if( nNewSpectra < 0 )
         {  nNewSpectra += RIGCTRL_SPECTRUM_FIFO_SIZE;
            // ,-----------|________________________|
            // '--> cannot plot more than
            //    T_RigCtrl_Spectrum SpectrumFifo[ RIGCTRL_SPECTRUM_FIFO_SIZE ]
            //    permits !
         }
        if( nNewSpectra > RIGCTRL_SPECTRUM_FIFO_SIZE )
         {  nNewSpectra = RIGCTRL_SPECTRUM_FIFO_SIZE;
         }
        HERE_I_AM__GUI();  // fasten seat belt .. NEW spectra waiting to be drawn !
        SpecDisp_UpdateWaterfall( Img_Waterfall->Picture->Bitmap, // [out] TBitmap a la VCL
           &g_SpecDispControl,   // [in] layout, etc
           &MyCwNet.RigControl,  // [in] RigControl instance with spectrum FIFO
           m_iLastScopeDisplayHeadIndex,
           nNewSpectra );
        m_iLastScopeDisplayHeadIndex = MyCwNet.RigControl.iScopeFifoHeadIndex;
        // If there were new data to redraw the waterfall,
        // also redraw the most recent SPECTRUM (as a curve):
        g_SpecDispControl.fUpdateSpectrum = TRUE;
      }
     if( g_SpecDispControl.fUpdateSpectrum )
      { SpecDisp_UpdateSpectrum( Img_Spectrum->Picture->Bitmap,
           &g_SpecDispControl,   // [in] layout, etc
           &MyCwNet.RigControl); // [in] RigControl instance with spectrum FIFO
      }
     if(( g_SpecDispControl.fmin_from_RigCtrl != g_SpecDispControl.fmin_on_FreqScale )
      ||( g_SpecDispControl.fmax_from_RigCtrl != g_SpecDispControl.fmax_on_FreqScale ) )
      { g_SpecDispControl.fUpdateFreqScale = g_SpecDispControl.fUpdateFrequencyInfo = TRUE;
        g_SpecDispControl.fmin_on_FreqScale = g_SpecDispControl.fmin_from_RigCtrl; // avoid unnecessary updates
        g_SpecDispControl.fmax_on_FreqScale = g_SpecDispControl.fmax_from_RigCtrl;
        // Note: With only 475 frequency bins from e.g. an IC-7300,
        //       it's pointless to ZOOM INTO the spectrum here locally.
        // Thus, the DISPLAYED frequency range is the same as
        //       what RigControl.c has delivered from the radio.
        // ( fmin_from_RigCtrl, fmax_from_RigCtrl updated when updating either
        //   the SPECTRUM, WATERFALL, or the FREQUENCY SCALE ).
        // This is utterly different than the original code from Spectrum Lab,
        // where FFTs with over a million frequency "bins" was calculated LOCALLY.
        // Short: The Remote CW Keyer shows the same spectrum frequency range
        //        as the radio, and "scrolls along" in what Icom calls 'CENTER' mode,
        //        or "hops from range to range" in 'SCROLL-C' mode.
      }
     if( g_SpecDispControl.fUpdateFreqScale )
      { SpecDisp_UpdateFreqScale( Img_FreqScale->Picture->Bitmap, // [out] TBitmap a la Borland VCL
           &g_SpecDispControl,   // [in] layout, etc
           &MyCwNet.RigControl); // [in] RigControl instance with VFO frequency, displayable frequency range, etc
      }
     if( MyCwNet.RigControl.iSMeterLevel_dB != m_dblDisplayedSMeterLevel_dB )
      { m_dblDisplayedSMeterLevel_dB = MyCwNet.RigControl.iSMeterLevel_dB;
        KeyerGUI_UpdateSMeter( Img_SMeter, m_dblDisplayedSMeterLevel_dB );
      }
     // The graphics in the BOTTOM panel on the "TRX" tab can be selected by the user:
     iBottomPanelMode = g_SpecDispControl.iBottomPanelMode;
     if( iBottomPanelMode==SPEC_DISP_PANEL_MODE_AUTOMATIC ) // automatically show e.g. "meters" during TX and "frequency info" during RX ?
      {
        if( (MyCwNet.RigControl.iTransmitReqst>0) // <- set via RigCtrl_SetTransmitRequest() when "we" asked the rig to transmit
          ||(MyCwNet.RigControl.iTransmitting >0) // <- set when THE RIG ITSELF transmits, see spec in RigControl.h
          )
         { iBottomPanelMode = SPEC_DISP_PANEL_MODE_MULTI_FUNCTION_METER;
         }
        else // currently not transmitting, so ..
         { iBottomPanelMode = SPEC_DISP_PANEL_MODE_FREQUENCY_INFO;
         }
      }
     switch( iBottomPanelMode )
      { case SPEC_DISP_PANEL_MODE_MULTI_FUNCTION_METER  : // S-Meter/Power, ALC, Compressor, SWR, Drain current, Supply Voltage, PA Temperature
           UpdateMultiFunctionMeters();
           break;
        case SPEC_DISP_PANEL_MODE_FREQUENCY_INFO : // indicator for band edge frequencies and 'special frequencies of interest'
           UpdateFrequencyInfo(); // .. depending on g_SpecDispControl.fUpdateFrequencyInfo ..
           break;
        case SPEC_DISP_PANEL_MODE_CHATBOX : // small text box primarily intended for messages from the SYSOP to other users
           UpdateChatboxOnTrxTab();
           break;
        default:
           break;
      } // end switch( CB_TRX_Bottom->ItemIndex )

   } // end if < update elements on the "TRX" tab > ?

  HERE_I_AM__GUI();
  UpdateStatusIndicator(); // -> coloured PANEL in the always-visible status line ABOVE the tabbed pages
  HERE_I_AM__GUI();

  // Transfer a few more characters from the 'type-ahead buffer' (RichEdit control)
  // to the CW generator for transmission ?
  if( (KeyerGUI_iRxTxInfoUsage==KEYER_GUI_RXTXINFO_TYPING_TXD )  // Typing TRANSMIT DATA into RichEdit_RxTxInfo ?
    &&(KeyerGUI_iStatusIndicatorMemIdx==KEYER_MEMORY_INDEX_TX_EDITOR) // Sending directly from RichEdit_RxTxInfo ?
    )
   { // If the user operator is typing 'transmittable text' into the edit field,
     // and the CW generator is ON, feed the generator with more characters:
     if( CwKeyer_Gen.iState != CW_GEN_OFF ) // transmission STARTED (e.g. by pressing ENTER in RichEdit_RxTxInfo) ?
      {  //  '--> this includes CW_GEN_FINISHED_SENDING, to allow
         //       "automatic restart" of the CW generator as soon as
         //       the operator entered MORE TEXT, without having to press
         //       ENTER repeatedly .
        if( KeyerGUI_TransferCharsFromEditFieldToCwGenerator(
          RichEdit_RxTxInfo/* in: TRichEdit used as "type-ahead buffer" */,
          &CwKeyer_Gen/* out: CW generator's own, thread-safe TX buffer */ ) > 0 )
         { // Transferred more characters from the edit field to the CW generator
           // -> Restart the stopwatch that monitors 'direct keyboard input from the operator'
           TIM_StartStopwatch( &KeyerGUI_swTxFromKeyboard );
         }

        // Call KeyerGUI_UpdateColourOfSentCharsInRichEdit() ?
        // (that's the "multi-purpose" edit- and display field in the status line,
        //  where individually coloured character cells show e.g. received data,
        //  data already transmitted, characters yet-to-send, etc)
        if( CwKeyer_Gen.nCharsRemainingToPlay != iPrevNumCharsToPlay ) // different number of chars remaining to send ?
         { iPrevNumCharsToPlay = CwKeyer_Gen.nCharsRemainingToPlay;
           KeyerGUI_UpdateColourOfSentCharsInRichEdit( RichEdit_RxTxInfo );
           // '--> [in] KeyerGUI_sz80TextForRxTxInfo, KeyerGUI_iRxTxInfoUsage, ..
         }
      } // end if( CwKeyer_Gen.iState != CW_GEN_OFF )
   } // end if( (KeyerGUI_iRxTxInfoUsage==KEYER_GUI_RXTXINFO_TYPING_TXD) && ...

  if( KeyerGUI_iStatusIndicatorMemIdx==KEYER_MEMORY_INDEX_TX_EDITOR ) // terminate "direct keyboard entry mode" after X seconds of inactivity ?
   { if( (CwKeyer_Gen.iState == CW_GEN_OFF) || (CwKeyer_Gen.iState == CW_GEN_FINISHED_SENDING) )
      { if( TIM_ReadStopwatch_ms( &KeyerGUI_swTxFromKeyboard ) > 5000/*ms*/ )
         {  TIM_StopStopwatch( &KeyerGUI_swTxFromKeyboard );
            KeyerGUI_iStatusIndicatorMemIdx=KEYER_MEMORY_INDEX_NONE; // here: terminated "type-ahead keyboard mode" because of inactivity
            if( KeyerGUI_iStatusIndicatorUsage == STATUS_INDICATOR_TX_MEM ) // WAS sending from the "type-ahead buffer" ?
             {  KeyerGUI_iStatusIndicatorUsage = STATUS_INDICATOR_IDLE; // back from "transmit from memory / type-ahead buffer" to "idle"
             }
         } // end if < transmission from keyboard ended X seconds ago >
      } // end if < CW generator "off" or "finished sending" >
   } // end if( KeyerGUI_iStatusIndicatorMemIdx==KEYER_MEMORY_INDEX_TX_EDITOR )

  if( KeyerGUI_iStatusIndicatorUsage & (STATUS_INDICATOR_ON_AIR | STATUS_INDICATOR_TX_MEM | STATUS_INDICATOR_CW_TEST ) )
   { // Possibly "sending from memory"  ... from WHICH memory ?
   }   // end if( KeyerGUI_iStatusIndicatorUsage & .. )

  HERE_I_AM__GUI();


} // end TKeyerMainForm::Timer1Timer()
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::AppMessage(tagMSG &Msg, bool &Handled)
  // Note: This handler may be called HUNDREDS OF TIMES PER SECOND !
  //       There can only be ONE of these handlers, because it's registered
  //       in Borland's 'VCL' ecosystem by setting Application->OnMessage = AppMessage .
{
  char buffer[81];
  WORD key = 0;

  /* See help on TApplication::OnMessage.
   *  Description of tagMSG cannot be found in the CBuilder help system,
   *             but in WINUSER.H:
   *
   * typedef struct tagMSG {
   *    HWND        hwnd;        // a window handle, dont care about this.
   *    UINT        message;     // a WM_xyz constant (message type)
   *    WPARAM      wParam;      // data depending on the message type
   *    LPARAM      lParam;
   *    DWORD       time;
   *    POINT       pt;
   * } MSG, *PMSG, NEAR *NPMSG, FAR *LPMSG;
   *
   *  Information on where the keyboard codes are located in this message
   *  can be found in the help on WM_KEYDOWN
   *  Taken from Microsofts Win32 Programmers Reference:
   *   Parameters of WM_KEYDOWN:
   *     nVirtKey
   *       Value of wParam. Specifies the virtual-key code of the nonsystem key.
   *
   *     lKeyData
   *       Value of lParam. Specifies the repeat count, scan code,
   *       extended-key flag, context code, previous key-state flag,
   *       and transition-state flag, as shown in the following table:
   *       ("Value" used here seems to be a "Bit Number" [WB] )..
   *
   *      Value Description
   *       0-15  Specifies the repeat count. The value is the number of times
   *             the keystroke is repeated as a result of the user holding down
   *             the key.
   *       16-23 Specifies the scan code. The value depends on the original
   *             equipment manufacturer (OEM).
   *       24    Specifies whether the key is an extended key, such as the
   *             right-hand ALT and CTRL keys that appear on an enhanced
   *             101- or 102-key keyboard. The value is 1 if it is an extended
   *             key; otherwise, it is 0.
   *       25-28 Reserved; do not use.
   *       29    Specifies the context code. The value is always 0 for a
   *             WM_KEYDOWN message.
   *       30    Specifies the previous key state. The value is 1 if the key
   *             is down before the message is sent, or it is 0 if the key is up.
   *       31    Specifies the transition state. The value is always 0 for a
   *             WM_KEYDOWN message.
   *  Return Values
   *       An application should return zero if it processes this message.
   *       "Return something" .. how ? Borlands's VCL / "AppMessage()"
   *       doesn't return anything ('void') result, but we're supposed
   *       to set the boolean 'Handled' to TRUE if we processed the message.
   */
  switch(Msg.message)
   { case WM_KEYDOWN :
        if(Active)   /* If Active is true , this form has the focus */
         { KeyerGUI_iFocusSwitchCountdownTimer_ms = 3000; // here: reloaded when receiving WM_KEYDOWN, with the focus *anywhere* in the RCW Keyer's main form
           key = (WORD)Msg.wParam;
           switch( key )
            { case VK_ESCAPE : // ESCAPE (#27) : Immediately stop transmitting ("panic key") and, since 2025-09, clear TX buffer
                 PauseTransmissionAndClearTxBuffers();
                 Handled = TRUE;
                 break;
              case VK_F1 : // function key 'F1' (#112) : start sending from memory #1
              case VK_F2 :
              case VK_F3 :
              case VK_F4 :
              case VK_F5 :
              case VK_F6 :
                 StartPlaying( KEYER_MEMORY_INDEX_1 + (key-VK_F1) );
                 Handled = TRUE;
                 break;
              case VK_F12 :
                 CwGen_StopReplay( &CwKeyer_Gen );
                 Handled = TRUE;
                 break;

              default:  // none of the "special" keys above..
                 break;
            } // end switch( key )
         } /* end if(Active) */
        else
         { // What they call 'Active' here is what they call 'Focused' somewhere else...
         } // end if < not "Active" aka not "Focused" >
        break; // end case WM_KEYDOWN

     case WM_SYSKEYDOWN :
        if(Active)   /* If Active is true , this form has the focus */
         { KeyerGUI_iFocusSwitchCountdownTimer_ms = 3000; // here: reloaded when receiving WM_SYSKEYDOWN, with the focus *anywhere* in the RCW Keyer's main form
           key = (WORD)Msg.wParam;
           switch( key )
            { case VK_MENU : // just 'M' or 'Alt-M' (Alternate)
                 if( g_fHideMenu ) // main menu was hidden so bring it back temporarily
                  { m_iMainMenuCountdownTimer = 5000/*ms*/ / Timer1->Interval;
                    HideOrShowMainMenu( FALSE );
                  }
                 break;
              default:  // none of the "system" keys above..
                 break;
            } // end switch( key )
         } /* end if(Active) */
        break; // end case WM_SYSKEYDOWN


     // case WM_SYSCOMMAND:  didn't work HERE, so moved it to MyWindowProc() !

     default:
        // for all other messages, Handled remains False
        // so that other message handlers can respond !
        break;
   }
} // end TKeyerMainForm::AppMessage()

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MyWindowProc(Messages::TMessage &Message)
  // Low level message handler (but still a VCL-, and not a WINDOWS-thing) .
  // Set through some VCL-thingy by assigning it to 'TControl::WindowProc' :
  // > Description
  // > Use the WindowProc property to temporarily replace or subclass
  // > the window procedure of the control. Component writers that are
  // > customizing the window procedure for a descendent class should
  // > override the WndProc method instead.
  // > Before assigning a new value to WindowProc, store the original value.
  // > WindowProc is initially set to the WndProc method, so if the value
  // > has not been changed since then, the original value need not be stored.
  // > Within the procedure that is used as the new value for WindowProc,
  // > pass any unhandled messages to the original procedure that was the value
  // > of WindowProc. After completing any specialized message handling,
  // > restore the value of WindowProc to the original procedure.
{
 BOOL  fMessageHasBeenHandled = FALSE;
 HANDLE hMenu;

  switch( Message.Msg )
   {
     case WM_SYSCOMMAND:  // something happened in this window's "system menu" :
           // Looking at the spec of WM_SYSCOMMAND on first glance, one would expect
           // to get here with Message.WParam == SC_MOUSEMENU after clicking
           // the window's "system icon" (=the thing in the upper LEFT corner).
           // BUT (yet another windows stupidity, at least THIS ONE is documented):
           //   When clicking the window's "system icon",
           //   got here with Message.WParam=0xF093, not 0xF090.  Wossat ?
           //  > In WM_SYSCOMMAND messages, the four low-order bits of the
           //  > uCmdType parameter are used internally by Windows.
           //  >  To obtain the correct result when testing the value of uCmdType,
           //  > an application must combine the value 0xFFF0 with the uCmdType
           //  > value by using the bitwise AND operator.
           // AAAARGHHH  !  Yet another stupid 'magic value' ...
           switch(Message.WParam & 0xFFF0/*stupid but necessary*/ )
            {
              case SC_MOUSEMENU/*0xF090*/: // "Retrieves the window menu as a result of a mouse click."
                { // Guess this message should be sent when the "system menu"
                  //  is about to be opened, but (as usual) the spec isn't precise.
                  // In fact, there was no WM_SYSCOMMAND here at all when the
                  // "system menu" aka "window menu" opened, when clicking
                  // the program's icon in the upper left corner of the window.
                  // SC_MOUSEMENU can possibly NOT be intercepted via AppMessage(),
                  // but other obscure stuff like Borland's "TControl::WindowProc".
                  HMENU hMenu = GetSystemMenu( Handle, FALSE/*revert*/ );
                  if( g_fHideMenu )
                   { CheckMenuItem( hMenu, SYSMENUITEM_HIDE_MAIN_MENU, MF_CHECKED );
                     CheckMenuItem( hMenu, SYSMENUITEM_SHOW_MAIN_MENU, MF_UNCHECKED );
                   }
                  else
                   { CheckMenuItem( hMenu, SYSMENUITEM_HIDE_MAIN_MENU, MF_UNCHECKED );
                     CheckMenuItem( hMenu, SYSMENUITEM_SHOW_MAIN_MENU, MF_CHECKED );
                   }
                } break; // end case "WM_SYSCOMMAND", subcode "SC_MOUSEMENU"
              case SYSMENUITEM_HIDE_MAIN_MENU:
                 g_fHideMenu = TRUE;
                 m_iMainMenuCountdownTimer = 0;
                 HideOrShowMainMenu(g_fHideMenu); // take effect IMMEDIATELY
                 break;
              case SYSMENUITEM_SHOW_MAIN_MENU:
                 g_fHideMenu = FALSE;
                 m_iMainMenuCountdownTimer = 0;
                 HideOrShowMainMenu(g_fHideMenu);
                 break;
           // case SYSMENUITEM_SETTINGS: // impossible, because items in a TMainMenu cannot be treated like a TPopupMenu.. aaargh !
           //    Menu_Settings->Popup(Mouse->CursorPos.x, Mouse->CursorPos.y);
           //    break;
           // case SYSMENUITEM_FUNCTIONS:
           //    Menu_Functions->Popup(Mouse->CursorPos.x, Mouse->CursorPos.y);
           //    break;
              case SYSMENUITEM_HELP:
                 YHF_HELP_ShowHelpContext(HELPID_MAIN_INDEX);
                 break;
              default:
                 break;
            }   // end switch (Msg.wParam) for our own additions to the system menu
        break; // end case WM_SYSCOMMAND


     default:
        // EXTREM WICHTIGER "default"-ZWEIG, WIRD
        //  **** VIELE TAUSEND MAL PRO SEKUNDE **** AUSGEFUEHRT !
        break;

   } // end switch( Message.Msg )

 if( ! fMessageHasBeenHandled )
  { WndProc(Message); // WndProc ist die "Original-Prozedur" fuers Message-Handling
  }

} // end TKeyerMainForm::MyWindowProc()
//---------------------------------------------------------------------------



//---------------------------------------------------------------------------
void TKeyerMainForm::UpdateTestDisplay(void)
{
  if( CwKeyer_iTestMode == KEYER_TEST_MODE_MANUAL )
   { Chk_ManualControl->Checked  = TRUE;
     Chk_MorseKeyDTR->Enabled    = TRUE;
     Chk_MorseKeyRTS->Enabled    = TRUE;
     Chk_RadioKeyingDTR->Enabled = TRUE;
     Chk_RadioKeyingRTS->Enabled = TRUE;
   }
  else // *NO* manual control of the digital outputs (DTR, RTS) :
   { Chk_MorseKeyDTR->Enabled    = FALSE;
     Chk_MorseKeyRTS->Enabled    = FALSE;
     Chk_RadioKeyingDTR->Enabled = FALSE;
     Chk_RadioKeyingRTS->Enabled = FALSE;
   }

} // end TKeyerMainForm::UpdateTestDisplay()

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::ApplyManualTestSignals(TObject *Sender)
  // Called as "OnClick" method (handler) from Chk_MorseKeyDTR,  Chk_MorseKeyRTS,
  //                                      Chk_RadioKeyingDTR, Chk_RadioKeyingRTS.
{
  if( Chk_ManualControl->Checked  )
   {  CwKeyer_iTestMode = KEYER_TEST_MODE_MANUAL;
      CwKeyer_SetDigitalOutput( KEYER_SIGNAL_INDEX_MORSE_KEY_DTR, Chk_MorseKeyDTR->Checked );
      // Why does the IDE (BCB V6) crash when hitting a breakpoint HERE ?
      CwKeyer_SetDigitalOutput( KEYER_SIGNAL_INDEX_MORSE_KEY_RTS, Chk_MorseKeyRTS->Checked );
      CwKeyer_SetDigitalOutput( KEYER_SIGNAL_INDEX_RADIO_KEYING_DTR, Chk_RadioKeyingDTR->Checked );
      CwKeyer_SetDigitalOutput( KEYER_SIGNAL_INDEX_RADIO_KEYING_RTS, Chk_RadioKeyingRTS->Checked );
      // Note: The actual switching of the above 'digital outputs'
      //       happens in the "keyer thread", using some funny-named Serial Port
      //       API functions that we don't want to invoke from here.
   }
  else
   {  CwKeyer_iTestMode = KEYER_TEST_MODE_OFF;
      // Do *not* call CwKeyer_SetDigitalOutput() in this mode !
   }
  UpdateTestDisplay();
}

//---------------------------------------------------------------------------
void TKeyerMainForm::UpdateItemsInFileMenu(void)
{
  MI_File_ExportSettingsAs->Visible   = ( g_iUserInterfaceMode==UI_MODE_COMPLETE );
  MI_File_ImportSettingsFrom->Visible = ( g_iUserInterfaceMode==UI_MODE_COMPLETE );
  MI_File_Separator1->Visible         = ( g_iUserInterfaceMode==UI_MODE_COMPLETE );
  MI_RecentFile0->Caption = GetRecentFileName(0);
  MI_RecentFile1->Caption = GetRecentFileName(1);
  MI_RecentFile2->Caption = GetRecentFileName(2);
  MI_RecentFile3->Caption = GetRecentFileName(3);
#if( GUI_RECENT_FILES_HISTORY_LENGTH > 4 )
# error "Add more items to this submenu ;-)"
#endif

} // end TKeyerMainForm::UpdateItemsInFileMenu()

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::File1Click(TObject *Sender)
{
  KeyerGUI_OnSetFocus( KEYER_GUI_ITEM_MAIN_MENU_FILE/*iNewFocusedItem*/ );
  UpdateItemsInFileMenu();
}


//---------------------------------------------------------------------------
void TKeyerMainForm::UpdateItemsInSettingsMenu(void)
{
  int i;
  AnsiString s;
  MI_Ch4_PTT->Checked = ( CwKeyer_TimingScope.cfg.iChannel4Source == TIMING_SCOPE_CHANNEL_SOURCE_PTT_OUTPUT);
  MI_Ch4_TestInput->Checked = (CwKeyer_TimingScope.cfg.iChannel4Source == TIMING_SCOPE_CHANNEL_SOURCE_TEST_INPUT);
  MI_Ch4_NetworkTrouble->Checked = (CwKeyer_TimingScope.cfg.iChannel4Source==TIMING_SCOPE_CHANNEL_SOURCE_NETWORK_TROUBLE);
  MI_Ch4_KeyingFifoUsage->Checked= (CwKeyer_TimingScope.cfg.iChannel4Source==TIMING_SCOPE_CHANNEL_SOURCE_KEYING_FIFO_USAGE);
  MI_Ch4_ElbugKeyerFlags->Checked= (CwKeyer_TimingScope.cfg.iChannel4Source==TIMING_SCOPE_CHANNEL_SOURCE_ELBUG_KEYER_FLAGS);

  MI_TScopeShowAudioIn->Checked  = (CwKeyer_TimingScope.cfg.iVisibleChannels & (1<<TIMING_SCOPE_CHANNEL_AUDIO_INPUT) ) != 0;
  MI_TScopeShowAudioOut->Checked = (CwKeyer_TimingScope.cfg.iVisibleChannels & (1<<TIMING_SCOPE_CHANNEL_AUDIO_OUTPUT) ) != 0;
  MI_TScopeShowCwDecoderAudio->Checked = (CwKeyer_TimingScope.cfg.iVisibleChannels & (1<<TIMING_SCOPE_CHANNEL_AUDIO_CW_DEC))!=0;

  MI_ColourScheme_Default->Checked = ( g_SpecDispControl.iColourScheme == COLOUR_SCHEME_DEFAULT);
  MI_ColourScheme_Dark->Checked    = ( g_SpecDispControl.iColourScheme == COLOUR_SCHEME_DARK);
  MI_Language_English->Checked     = ( APPL_iLanguage==TRANSLATOR_LANGUAGE_ENGLISH );
  MI_Language_German->Checked      = ( APPL_iLanguage==TRANSLATOR_LANGUAGE_GERMAN  );
  MI_Language_French->Checked      = ( APPL_iLanguage==TRANSLATOR_LANGUAGE_FRENCH  );
  MI_Language_Dutch->Checked       = ( APPL_iLanguage==TRANSLATOR_LANGUAGE_DUTCH   );
  MI_Language_Italian->Checked     = ( APPL_iLanguage==TRANSLATOR_LANGUAGE_ITALIAN );
  switch( g_iUserInterfaceMode )
   { case UI_MODE_SIMPLE :
        s = AnsiString(TE("&User Interface: Simple"));
        MI_UI_Simple->Checked   = TRUE;
        MI_UI_Complete->Checked = FALSE;
        break;
     case UI_MODE_COMPLETE:
        s = AnsiString(TE("&User Interface: Complete"));
        MI_UI_Simple->Checked   = FALSE;
        MI_UI_Complete->Checked = TRUE;
        break;
     default: // ?!
        break;
   }
  if( g_fHideMenu )
   { s = s + ", " + AnsiString(TE("menu opened via ALT key"));
   }
  else
   { s = s + ", " + AnsiString(TE("menu always visible"));
   }
  if( KeyerGUI_fAutomaticFocusSwitching )
   { s = s + ", " + AnsiString(TE("focus switch"));
   }
  MI_Settings_UserInterface->Caption = s;
  MI_HideMenu->Checked = g_fHideMenu;
  MI_AutoFocusSwitch->Checked = KeyerGUI_fAutomaticFocusSwitching;

  MI_ITU_Region->Visible = (g_iUserInterfaceMode==UI_MODE_COMPLETE);
  MI_ITU_Region_1->Checked = ( RigCtrl_ITU_Region == 1 );
  MI_ITU_Region_2->Checked = ( RigCtrl_ITU_Region == 2 );
  MI_ITU_Region_3->Checked = ( RigCtrl_ITU_Region == 3 );

  s = AnsiString(TE("More '&Keyer Settings'")); // 'dynamic content' for KeyerMainForm.MI_MoreKeyerSettings ?
  if( CwKeyer_Config.fDisableTx )
   { s = s + " : " + AnsiString(TE("Disable TX") );
   }
  MI_MoreKeyerSettings->Caption = s;


  // Hide almost everything in the "Settings" menu that may distract an unexperienced user:
  MI_ConnectScopeCh4->Visible    = (g_iUserInterfaceMode==UI_MODE_COMPLETE);
  MI_ShowAudioWaveforms->Visible = (g_iUserInterfaceMode==UI_MODE_COMPLETE);
  MI_MoreKeyerSettings->Visible  = (g_iUserInterfaceMode==UI_MODE_COMPLETE);
  MI_Settings_SpectrumDisplay->Visible = (g_iUserInterfaceMode==UI_MODE_COMPLETE);
  MI_SeparatorBelowSpectrumDisplaySettings->Visible = (g_iUserInterfaceMode==UI_MODE_COMPLETE);
  MI_ClearScopeOverlays->Visible = (g_iUserInterfaceMode==UI_MODE_COMPLETE);
  MI_ClearDebugAndDecoderOutput->Visible = (g_iUserInterfaceMode==UI_MODE_COMPLETE);
  MI_Settings_ShowMorseDecoderOutput->Visible = (g_iUserInterfaceMode==UI_MODE_COMPLETE);
  MI_ReportTestResults->Visible  = (g_iUserInterfaceMode==UI_MODE_COMPLETE);
  MI_ListAndEditRigParams->Visible = (g_iUserInterfaceMode==UI_MODE_COMPLETE);
  MI_ListUserDefinedBands->Visible = (g_iUserInterfaceMode==UI_MODE_COMPLETE);
  MI_ListUserDefinedFrequencies->Visible = (g_iUserInterfaceMode==UI_MODE_COMPLETE);
  MI_ShowStraightKeyDecoderHistory->Visible = (g_iUserInterfaceMode==UI_MODE_COMPLETE);
  MI_PauseDebugOutput->Visible   = (g_iUserInterfaceMode==UI_MODE_COMPLETE);
  MI_SeparatorBelowPauseDebugOutput->Visible = (g_iUserInterfaceMode==UI_MODE_COMPLETE);
  MI_NetworkServerOptions->Visible = (g_iUserInterfaceMode==UI_MODE_COMPLETE);
  MI_NetworkLatency->Visible     = (g_iUserInterfaceMode==UI_MODE_COMPLETE);
  MI_NetworkDiagnostics->Visible = (g_iUserInterfaceMode==UI_MODE_COMPLETE);
  MI_Settings_RigControlOptions->Visible = (g_iUserInterfaceMode==UI_MODE_COMPLETE);
  MI_HamlibServerOptions->Visible  = (g_iUserInterfaceMode==UI_MODE_COMPLETE);
    // '--> this is the last item in the "Settings" menu; only visible in the "COMPLETE UI"
    // SUB MENU ITEMS (below) don't need to be hidden when their parents are already invisible.


  MI_PauseDebugOutput->Checked     = m_fDebugOutputPaused;
  MI_Server_EnableHTTP->Checked    = (MyCwNet.cfg.iHttpServerOptions & HTTP_SERVER_OPTIONS_ENABLE) != 0;
  MI_Server_RestrictAccess->Checked= (MyCwNet.cfg.iHttpServerOptions & HTTP_SERVER_OPTIONS_RESTRICT) != 0;
  MI_Server_AddTestTone->Checked   = (CwKeyer_DSP.cfg.iAudioFlags & DSP_AUDIO_FLAGS_SEND_NETWORK_TEST_TONE) != 0;

  s = AnsiString(TE("Network &Latency")) + " : ";
  if( MyCwNet.cfg.iNetworkLatency_ms <= 0 )
   { i = CwNet_GetIndexOfCurrentlyActiveClient( &MyCwNet );
     s = s + AnsiString(TE("Auto-detect, currently measured: ") )
           + IntToStr( CwNet_GetLatencyForRemoteClient_ms(&MyCwNet, i) )
           + " ms";
   }
  else
   { s = s + IntToStr(MyCwNet.cfg.iNetworkLatency_ms) + " ms";
   }
  MI_NetworkLatency->Caption = s;
  MI_Network_DiagnosticMode->Checked = (MyCwNet.cfg.iDiagnosticFlags & CWNET_DIAG_FLAGS_VERBOSE) != 0;
  MI_Network_SimulateBadConnection->Checked = (MyCwNet.cfg.iDiagnosticFlags & CWNET_DIAG_FLAGS_SIMULATE_BAD_CONN) != 0;
  MI_Network_ShowConnectionLog->Checked = (MyCwNet.cfg.iDiagnosticFlags & CWNET_DIAG_FLAGS_SHOW_CONN_LOG) != 0;
  MI_Network_ShowNetworkTraffic->Checked= (MyCwNet.cfg.iDiagnosticFlags & CWNET_DIAG_FLAGS_SHOW_NETWORK_TRAFFIC) != 0;
  MI_Network_ShowRigControlTraffic->Checked=(RigCtrl_TrafficMonitor.iDisplayOptions & RIGCTRL_TMON_DISPLAY_OPTION_ENABLE) != 0;
  MI_Network_ShowRcvdKeyingBytestream->Checked = (MyCwNet.cfg.iDiagnosticFlags & CWNET_DIAG_FLAGS_SHOW_KEYING_BYTESTREAM) != 0;
  MI_Settings_ShowMorseDecoderOutput->Checked  = (MyCwNet.cfg.iDiagnosticFlags & CWNET_DIAG_FLAGS_SHOW_DECODER_OUTPUT) != 0;
  MI_Network_TrafficLog_RejectPeriodicMsgs->Checked = (MyCwNet.cfg.iDiagnosticFlags & CWNET_DIAG_FLAGS_REJECT_PERIODIC_MSGS) != 0;
  MI_Test_ShowGuiEvents->Checked = (MyCwNet.cfg.iDiagnosticFlags & CWNET_DIAG_FLAGS_SHOW_GUI_EVENTS) != 0;
  

  MI_KeyViaShiftAndControl->Checked = CwKeyer_Config.fKeyViaShiftAndControl;
  MI_KeyInputWatchdog->Checked      = CwKeyer_Config.fKeyInputWatchdog;
  MI_DebouncePaddleInputs->Checked  = CwKeyer_Config.fDebouncePaddleInputs;
  MI_MoreKeyerSettings_DisableTx->Checked = CwKeyer_Config.fDisableTx;

  MI_RigCtrl_RejectEchosInLog->Checked = (MyCwNet.RigControl.dwMessageFilterForLog & RIGCTRL_MSGFILTER_ECHO) != 0;
  MI_RigCtrl_RejectPeriodicCmds->Checked = (MyCwNet.RigControl.dwMessageFilterForLog & RIGCTRL_MSGFILTER_PERIODIC_POLL) != 0;
  MI_RigCtrl_RejectSpectrumInLog->Checked = (MyCwNet.RigControl.dwMessageFilterForLog & RIGCTRL_MSGFILTER_SPECTRUM) != 0;
  MI_RigCtrl_RejectFrequencyReports->Checked = (MyCwNet.RigControl.dwMessageFilterForLog & RIGCTRL_MSGFILTER_FREQUENCY_REPORT) != 0;
  MI_RigCtrl_RejectAnyKnownCommandFromLog->Checked = (MyCwNet.RigControl.dwMessageFilterForLog & RIGCTRL_MSGFILTER_ANY_KNOWN_COMMAND) != 0;

  MI_TMColumns_All->Checked = (RigCtrl_TrafficMonitor.iExtraColumns == RIGCTRL_TMON_EXTRA_COLUMNS_ALL);
  MI_TMColumns_Time->Checked = (RigCtrl_TrafficMonitor.iExtraColumns & RIGCTRL_TMON_EXTRA_COLUMN_TIME) != 0;
  MI_TMColumns_TX_RX->Checked= (RigCtrl_TrafficMonitor.iExtraColumns & RIGCTRL_TMON_EXTRA_COLUMN_TX_RX) != 0;
  MI_TMColumns_PayloadLength->Checked=(RigCtrl_TrafficMonitor.iExtraColumns & RIGCTRL_TMON_EXTRA_COLUMN_PAYLOAD_LENGTH) != 0;
  MI_TMColumns_Comment->Checked=(RigCtrl_TrafficMonitor.iExtraColumns & RIGCTRL_TMON_EXTRA_COLUMN_COMMENTS) != 0;
  MI_TMon_Timestamp_UTC->Checked=(RigCtrl_TrafficMonitor.iTimestampFormat==RIGCTRL_TMON_TIMESTAMP_TIME_OF_DAY);
  MI_TMon_Timestamp_Milliseconds->Checked=(RigCtrl_TrafficMonitor.iTimestampFormat==RIGCTRL_TMON_TIMESTAMP_MILLISECONDS);
  MI_TrafficMon_PauseOnFullFIFO->Checked =(RigCtrl_TrafficMonitor.iDisplayOptions & RIGCTRL_TMON_DISPLAY_OPTION_PAUSE_ON_FULL_FIFO) != 0;
  MI_TrafficMonitor_PauseOnError->Checked=(RigCtrl_TrafficMonitor.iDisplayOptions & RIGCTRL_TMON_DISPLAY_OPTION_PAUSE_ON_ERROR) != 0;
  MI_TrafficMonitor_PauseOnInitDone->Checked=(RigCtrl_TrafficMonitor.iDisplayOptions & RIGCTRL_TMON_DISPLAY_OPTION_PAUSE_ON_INIT_DONE) != 0;
    // '-- 'Pause when initialisation done' (but that's difficult to tell when an EXTERNAL CLIENT like WFView controls the rig !)


  // Above: Menu items for the important 'Rig Control' and its 'Traffic Monitor'.
  // Below: Menu items for the optional 'Hamlib Net rigctld'-compatible SERVER:
  s = AnsiString(TE("Built-in &Hamlib 'Net' Server Options")) + " (";
  if( CwKeyer_Config.fEnableBuiltInHamlibServer )
   { s = s + AnsiString(TE("enabled")) + ", port " + IntToStr(MyHamlibServer.cfg.iServerListeningPort);
   }
  else
   { s = s + AnsiString(TE("disabled"));
   }
  MI_HamlibServerOptions->Caption = s + ")";
  MI_HLServer_Enable->Checked = CwKeyer_Config.fEnableBuiltInHamlibServer;
  MI_HLServer_ShowTraffic->Checked = (MyHamlibServer.cfg.iDiagnosticFlags & HLSRV_DIAG_FLAGS_SHOW_NETWORK_TRAFFIC) != 0;
  MI_HLServer_VerboseOutput->Checked = (MyHamlibServer.cfg.iDiagnosticFlags & HLSRV_DIAG_FLAGS_VERBOSE_OUTPUT) != 0;
} // end TKeyerMainForm::UpdateItemsInSettingsMenu()

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Menu_SettingsClick(TObject *Sender)
{ // Called from the VCL when opening the "Settings" menu.
  KeyerGUI_OnSetFocus( KEYER_GUI_ITEM_MAIN_MENU_SETTINGS/*iNewFocusedItem*/ );
  UpdateItemsInSettingsMenu();
}

//---------------------------------------------------------------------------
void TKeyerMainForm::UpdateItemsInFunctionsMenu(void)
{
  MI_Funcs_PowerOffWhenClosing->Checked = CwKeyer_Config.fTurnRigOffWhenClosing;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Menu_FunctionsClick(TObject *Sender)
{ // Called from the VCL when opening the "Functions" menu.
  KeyerGUI_OnSetFocus( KEYER_GUI_ITEM_MAIN_MENU_FUNCTIONS/*iNewFocusedItem*/ );
  UpdateItemsInFunctionsMenu();
}
//---------------------------------------------------------------------------


void __fastcall TKeyerMainForm::MI_HelpClick(TObject *Sender)
{ // Called from the VCL when opening the "Help" menu.
  KeyerGUI_OnSetFocus( KEYER_GUI_ITEM_MAIN_MENU_HELP/*iNewFocusedItem*/ );

  // Similar as in the "Settings" menu, certain items are hidden
  // when the "User Interface" is set to "Simple" instead of "Complete":
  // MI_HelpIndex->Visible = (g_iUserInterfaceMode==UI_MODE_COMPLETE); // oh c'mon, keep this visible, even in the "Simple" UI ...
  MI_Help_MainMenu->Visible = (g_iUserInterfaceMode==UI_MODE_COMPLETE);
  // Show help about the "TRX" tab either in the "Complete UI", or when at least remote VFO control or remote spectrum display are possible:
  MI_Help_TransceiverControl->Visible = (g_iUserInterfaceMode==UI_MODE_COMPLETE) || (TS_TRX->Visible);
  MI_Help_Hardware->Visible = (g_iUserInterfaceMode==UI_MODE_COMPLETE);
  MI_Help_DebugTab->Visible = (g_iUserInterfaceMode==UI_MODE_COMPLETE) || (TS_Debug->Visible);

} // end TKeyerMainForm::MI_HelpClick()
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Ch4_PTTClick(TObject *Sender)
{ CwKeyer_TimingScope.cfg.iChannel4Source = TIMING_SCOPE_CHANNEL_SOURCE_PTT_OUTPUT;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Ch4_TestInputClick(TObject *Sender)
{ CwKeyer_TimingScope.cfg.iChannel4Source = TIMING_SCOPE_CHANNEL_SOURCE_TEST_INPUT;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Ch4_NetworkTroubleClick(TObject *Sender)
{ CwKeyer_TimingScope.cfg.iChannel4Source = TIMING_SCOPE_CHANNEL_SOURCE_NETWORK_TROUBLE;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Ch4_KeyingFifoUsageClick(TObject *Sender)
{ CwKeyer_TimingScope.cfg.iChannel4Source = TIMING_SCOPE_CHANNEL_SOURCE_KEYING_FIFO_USAGE;
}
//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Ch4_ElbugKeyerFlagsClick(TObject *Sender)
{ CwKeyer_TimingScope.cfg.iChannel4Source = TIMING_SCOPE_CHANNEL_SOURCE_ELBUG_KEYER_FLAGS;
}
//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_TScopeShowAudioInClick(TObject *Sender)
{ CwKeyer_TimingScope.cfg.iVisibleChannels ^= (1 << TIMING_SCOPE_CHANNEL_AUDIO_INPUT);
  TimingScope_iUpdateCountOnTestTab = TimingScope_iUpdateCountOnKeyerTab = -1; // redraw a.s.a.p.
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_TScopeShowAudioOutClick(TObject *Sender)
{ CwKeyer_TimingScope.cfg.iVisibleChannels ^= (1 << TIMING_SCOPE_CHANNEL_AUDIO_OUTPUT);
  TimingScope_iUpdateCountOnTestTab = TimingScope_iUpdateCountOnKeyerTab = -1; // redraw a.s.a.p.
}

void __fastcall TKeyerMainForm::MI_TScopeShowCwDecoderAudioClick(TObject *Sender)
{ CwKeyer_TimingScope.cfg.iVisibleChannels ^= (1 << TIMING_SCOPE_CHANNEL_AUDIO_CW_DEC);
  TimingScope_iUpdateCountOnTestTab = TimingScope_iUpdateCountOnKeyerTab = -1; // redraw a.s.a.p.
}
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_KeyInputWatchdogClick(TObject *Sender)
{ CwKeyer_Config.fKeyInputWatchdog = !CwKeyer_Config.fKeyInputWatchdog;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_DebouncePaddleInputsClick(TObject *Sender)
{ CwKeyer_Config.fDebouncePaddleInputs = !CwKeyer_Config.fDebouncePaddleInputs;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Hotkey_EscapeClick(TObject *Sender)
{ PauseTransmissionAndClearTxBuffers();  // <- same as when pressing the REAL "ESCAPE" key
}


//---------------------------------------------------------------------------
TEdit *TKeyerMainForm::GetEditorForMemoryIndex(
          int iKeyerMemoryIndex ) // [in] KEYER_MEMORY_INDEX_1..6
{ switch( iKeyerMemoryIndex )
   { case KEYER_MEMORY_INDEX_1 : return Ed_Mem1; // kludge because VCL forms don't support arrays..
     case KEYER_MEMORY_INDEX_2 : return Ed_Mem2;
     case KEYER_MEMORY_INDEX_3 : return Ed_Mem3;
     case KEYER_MEMORY_INDEX_4 : return Ed_Mem4;
     case KEYER_MEMORY_INDEX_5 : return Ed_Mem5;
     case KEYER_MEMORY_INDEX_6 : return Ed_Mem6;
     default: return NULL;
     // Note: KEYER_MEMORY_INDEX_TX_EDITOR cannot be supported here,
     //       because (a) that's a RICH TEXT control
     //           and (b) because RichEdit_RxTxInfo contains much more than just a "string to send"
   }
}

//---------------------------------------------------------------------------
void TKeyerMainForm::StartPlaying(  // .. using CwKeyer_Gen / CwGen_StartReplay()
        int iKeyerMemoryIndex )  // [in] here: only KEYER_MEMORY_INDEX_1..6
{ TEdit *pEdit = GetEditorForMemoryIndex(iKeyerMemoryIndex);
  char szOriginalText[KEYER_MAX_CHARS_PER_MEMORY+4], szExpandedText[KEYER_MAX_CHARS_PER_MEMORY+4];
  const char *pszSrc;
  char  *pszDest = szExpandedText;
  char  *pszEndstop = szExpandedText+KEYER_MAX_CHARS_PER_MEMORY;
  int   nCharsExpanded;
  if( pEdit != NULL )
   {
     SL_strncpy( szOriginalText, AnsiString(pEdit->Text).c_str(), KEYER_MAX_CHARS_PER_MEMORY );
     // Expand certain 'macros' that CwGen_StartReplay() isn't aware of :
     pszSrc = szOriginalText;
     nCharsExpanded = KeyerGUI_ExpandTextToSend( &pszSrc, KEYER_MAX_CHARS_PER_MEMORY/*nCharsWanted*/, pszDest, pszEndstop );
     // Note: When playing from memory '1' to '6', characters are dumped
     //       into the multi-purpose edit field in the status line
     //       IN THE MOMENT THEY ARE ACTUALLY SENT. Thus, in contrast to
     //       direct text input in that field ("type-ahead mode"),
     //       the CW generator's own transmit buffer can be filled up
     //       ENTIRELY here. Thus nCharsWanted = KEYER_MAX_CHARS_PER_MEMORY,
     //       not just 2 or 3 as in KeyerGUI_OnKeyInTransmitTextEditor() when pressing ENTER.
     if( nCharsExpanded > 0 )
      { CwGen_StartReplay( &CwKeyer_Gen, szExpandedText );  // no problem with "speed-switching macros" here, because the ENTIRE STRING is available for CwGen.c now
        KeyerGUI_iStatusIndicatorMemIdx = iKeyerMemoryIndex;
        // The rest ("colouring" of certain GUI elements or a range of characters)
        // happens later, in Timer1Timer(), by monitoring the variables
        // set in KeyerThread() -> CwGen_Handler() .
      }
   }
}
//---------------------------------------------------------------------------

void __fastcall TKeyerMainForm::Btn_Mem1Click(TObject *Sender)
{
  StartPlaying(KEYER_MEMORY_INDEX_1);
}
//---------------------------------------------------------------------------

void __fastcall TKeyerMainForm::Btn_Mem2Click(TObject *Sender)
{
  StartPlaying(KEYER_MEMORY_INDEX_2);
}
//---------------------------------------------------------------------------

void __fastcall TKeyerMainForm::Btn_Mem3Click(TObject *Sender)
{
  StartPlaying(KEYER_MEMORY_INDEX_3);
}
//---------------------------------------------------------------------------

void __fastcall TKeyerMainForm::Btn_Mem4Click(TObject *Sender)
{
  StartPlaying(KEYER_MEMORY_INDEX_4);
}
//---------------------------------------------------------------------------

void __fastcall TKeyerMainForm::Btn_Mem5Click(TObject *Sender)
{
  StartPlaying(KEYER_MEMORY_INDEX_5);
}
//---------------------------------------------------------------------------

void __fastcall TKeyerMainForm::Btn_Mem6Click(TObject *Sender)
{
  StartPlaying(KEYER_MEMORY_INDEX_6);
}
//---------------------------------------------------------------------------


void __fastcall TKeyerMainForm::Btn_StopSendingFromMemoryClick(TObject *Sender)
{
  CwGen_StopReplay( &CwKeyer_Gen );
}
//---------------------------------------------------------------------------

void __fastcall TKeyerMainForm::Ed_MemChange(TObject *Sender)
{
  int i;
  TEdit *pEdit;
  if( KeyerGUI_iUpdating==0 )
   { for(i=0; i<KEYER_NUM_MESSAGE_MEMORIES; ++i)
      { pEdit = GetEditorForMemoryIndex( i+KEYER_MEMORY_INDEX_1 );
        if( pEdit != NULL ) // only if there really is an "editor" for this message memory..
         { SL_strncpy( CwKeyer_Config.szKeyerMemory[i], AnsiString(pEdit->Text).c_str(), KEYER_MAX_CHARS_PER_MEMORY );
         }
      } // end for < all 'keyer message memories' >
     SL_strncpy( CwKeyer_Config.sz15MyCall, AnsiString(Ed_MyCall->Text).c_str(), KEYER_MAX_CHARS_PER_MEMORY );
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_ColourScheme_DefaultClick(TObject *Sender)
{
  g_SpecDispControl.iColourScheme = COLOUR_SCHEME_DEFAULT;
  KeyerGUI_SetColourScheme( g_SpecDispControl.iColourScheme );
  m_dblDisplayedSMeterLevel_dB = -1;  // kludge to redraw the S-Meter
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_ColourScheme_DarkClick(TObject *Sender)
{
  g_SpecDispControl.iColourScheme = COLOUR_SCHEME_DARK;
  KeyerGUI_SetColourScheme( g_SpecDispControl.iColourScheme );
  m_dblDisplayedSMeterLevel_dB = -1;  // kludge to redraw the S-Meter
}

//---------------------------------------------------------------------------
void TKeyerMainForm::AdaptImageSizesOnTRXTab(void)
{ // It's important to do this AT LEAST ONCE, to initialize these three bitmaps
  // on the 'TRX' tab. These three images (TImage a la VCL) are placed on a
  // TPanel element (abused as their parent container) :
  int iTotalHeight, iSpectrumHeight, iFreqScaleHeight;
  int iOldMainTab;
  // Without the following kludge, Img_Spectrum->Height / Width still had
  // the "designed", but not the "actual" values when the 'TRX' tab was hidden,
  // but the SPECTROGRAM should be periodically updated due to
  // CwKeyer_Config.iTRXOptions & KEYER_TRX_OPT_KEEP_RUNNING :
  if( (iOldMainTab=KeyerGUI_iCurrentMainTab) != KEYER_GUI_MAIN_TAB_TRX )
   { SwitchMainTab( KEYER_GUI_MAIN_TAB_TRX ); // here: called from AdaptImageSizesOnTRXTab() ..
     // Making the tabsheet visible (even if only for a few milliseconds)
     // convinced the VCL to properly adjust the size of the TABSHEET (TS_TRX),
     // and the various PANELS (with different alignments) :
     //  Pnl_TRX_Top: TPanel with Align=alTop,  FIXED ("designed") height .
     //    '--> children: Ed_VFO, CB_OpMode, ...
     //  Pnl_TRX_Bottom: TPanel with Align = alBottom, FIXED ("designed") height.
     //    '--> children: sliders for contrast and brightness, etc
     //  Pnl_TRX_Center: TPanel with Align = alClient : Uses the full "remaining" height.
     //    '--> children: Img_Spectrum, Img_Waterfall, Img_FreqScale .
     //                     |    __________|___________    |
     //                     '---|Heights assigned BELOW|---'
     //                         |______________________|
   }

  Pnl_TRX_Bottom->Visible = ( g_SpecDispControl.iBottomPanelMode != SPEC_DISP_PANEL_MODE_OFF);
    // '--> Panel with "Align = alBottom" (a Borland VCL thing).
    //      Pnl_TRX_Center has "Align" set to "alClient", so when
    //      Pnl_TRX_Bottom is INVISIBLE, Pnl_TRX_Center will 'automatically'
    //      be made higher by some black voodoo magic in the VCL .
    //  What is SHOWN INSIDE 'Pnl_TRX_Bottom' (Img_TRX_Bottom or RichEd_ChatboxOnTrxTab)
    //  depends on g_SpecDispControl.iBottomPanelMode, too !

  iTotalHeight = Pnl_TRX_Center->Height; // sum of heights for the following:
  iFreqScaleHeight = iTotalHeight / 4;
  UTL_LimitInteger( &iFreqScaleHeight, 16, 32 );
  iSpectrumHeight = (iTotalHeight-iFreqScaleHeight) / 3;
  UTL_LimitInteger( &iSpectrumHeight, 16, 200 );

  // From top to bottom (as on Icom screens):
  //  Spectrum (graph, "curve"), Spectrogram (waterfall), Frequency Scale .
  Img_Spectrum->Picture->Bitmap->PixelFormat = pf32bit; // use 32 bit/pixel for SpecDisp.cpp / T_RGBColor !
  Img_Spectrum->Height = iSpectrumHeight;
  Img_Spectrum->Picture->Bitmap->Height = Img_Spectrum->Height;
  Img_Spectrum->Picture->Bitmap->Width  = Img_Spectrum->Width;
  Img_FreqScale->Picture->Bitmap->PixelFormat = pf32bit;
  Img_FreqScale->Picture->Bitmap->Height= Img_FreqScale->Height;
  Img_FreqScale->Width = iFreqScaleHeight;
  Img_FreqScale->Picture->Bitmap->Width = Img_FreqScale->Width;
  // Controlled by the VCL's "Aligment" (alClient), the WATERFALL automatically
  // occupies the remaining height between spectrum (alTop) and f-scale (alBottom).
  // The form's size constraints make sure what remains never gets negative or zero.
  Img_Waterfall->Picture->Bitmap->PixelFormat = pf32bit;
  Img_Waterfall->Picture->Bitmap->Height= Img_Waterfall->Height;
  Img_Waterfall->Picture->Bitmap->Width = Img_Waterfall->Width;

  if( Pnl_TRX_Bottom->Visible )  // another kludge; this one is vital for SpecDisp_UpdateFrequencyInfo() ..
   { Img_TRX_Bottom->Picture->Bitmap->PixelFormat = pf32bit;
     Img_TRX_Bottom->Picture->Bitmap->Height= Img_TRX_Bottom->Height;
     Img_TRX_Bottom->Picture->Bitmap->Width = Img_TRX_Bottom->Width;
   }

  m_fAdaptImageSizesOnTRXTab = FALSE;  // "done" (until the next 'Resze' or whatever)

  if( KeyerGUI_iCurrentMainTab != iOldMainTab )
   { SwitchMainTab( iOldMainTab ); // here: called from AdaptImageSizesOnTRXTab() to switch back to the previous tab
   }
} // end AdaptImageSizesOnTRXTab()

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::FormResize(TObject *Sender)
{
  // Unfortunately, this event handler is also called through a long and winding
  // road when the VCL invoked "Controls::TControl::SetVisible" !
  // Also, there's a bug in the VCL that ON SOME MACHINES causes the timing scope
  // (and other 'visual controls') to occupy only about THREE QUARTERS of the
  // available 'client area' (on its tabsheet, or the tabsheet / page control
  // is broken itself).
  if( !m_fInitialising )
   {  KeyerGUI_ClearScopeOverlays();
      TimingScope_iUpdateCountOnTestTab = -1; // kludge to redraw the timing scope..
      TimingScope_iUpdateCountOnKeyerTab= -1; // ... even when 'paused' .
      m_fAdaptImageSizesOnTRXTab = TRUE;  // call AdaptImageSizesOnTRXTab() it that tab ever gets visible
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Img_TimingScopeOnTestTab_MouseDown(
      TObject *Sender, TMouseButton Button, TShiftState Shift, int X, int Y)
{ if( Shift.Contains(ssLeft) ) // only WHILE THE LEFT MOUSE BUTTON is pressed..
   { KeyerGUI_HandleMouseEventInTimingScope( Img_TimingScopeOnTestTab->Picture->Bitmap, EVT_MOUSE_DOWN, X, Y );
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Img_TimingScopeOnTestTab_MouseMove(
      TObject *Sender, TShiftState Shift, int X, int Y)
{ if( Shift.Contains(ssLeft) ) // only when moved WHILE THE LEFT MOUSE BUTTON is held down..
   { KeyerGUI_HandleMouseEventInTimingScope( Img_TimingScopeOnTestTab->Picture->Bitmap, EVT_MOUSE_MOVE, X, Y );
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Img_TimingScopeOnTestTab_MouseUp(
      TObject *Sender, TMouseButton Button, TShiftState Shift, int X, int Y)
{ if( Button == mbLeft )
   { if( ! KeyerGUI_HandleMouseEventInTimingScope( Img_TimingScopeOnTestTab->Picture->Bitmap, EVT_MOUSE_UP, X, Y ) )
      { PM_TimingScope->Popup(Mouse->CursorPos.x, Mouse->CursorPos.y);
      }
   }
  else
   { PM_TimingScope->Popup(Mouse->CursorPos.x, Mouse->CursorPos.y);
   }
}
//---------------------------------------------------------------------------

void __fastcall TKeyerMainForm::Img_TimingScopeOnKeyerTab_MouseDown(
      TObject *Sender, TMouseButton Button, TShiftState Shift, int X, int Y)
{ KeyerGUI_HandleMouseEventInTimingScope( Img_TimingScopeOnKeyerTab->Picture->Bitmap, EVT_MOUSE_DOWN, X, Y );
}
//---------------------------------------------------------------------------

void __fastcall TKeyerMainForm::Img_TimingScopeOnKeyerTab_MouseMove(
      TObject *Sender, TShiftState Shift, int X, int Y)
{ if( Shift.Contains(ssLeft) ) // only WHILE THE LEFT MOUSE BUTTON is pressed..
   { KeyerGUI_HandleMouseEventInTimingScope( Img_TimingScopeOnKeyerTab->Picture->Bitmap, EVT_MOUSE_MOVE, X, Y );
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Img_TimingScopeOnKeyerTab_MouseUp(
      TObject *Sender, TMouseButton Button, TShiftState Shift, int X, int Y)
{ if( Button == mbLeft )
   { KeyerGUI_HandleMouseEventInTimingScope( Img_TimingScopeOnKeyerTab->Picture->Bitmap, EVT_MOUSE_UP, X, Y );
   }
  else
   { PM_TimingScope->Popup(Mouse->CursorPos.x, Mouse->CursorPos.y);
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::PM_TimingScopePopup(TObject *Sender)
{
  PM_TScope_ShowDashInput->Checked= (CwKeyer_TimingScope.cfg.iVisibleChannels & (1<<TIMING_SCOPE_CHANNEL_DASH_INPUT) ) != 0;
  PM_TScope_ShowDotInput->Checked = (CwKeyer_TimingScope.cfg.iVisibleChannels & (1<<TIMING_SCOPE_CHANNEL_DOT_INPUT ) ) != 0;
  PM_TScope_ShowKeyingOutput->Checked=(CwKeyer_TimingScope.cfg.iVisibleChannels&(1<<TIMING_SCOPE_CHANNEL_CW_OUTPUT ) ) !=0;
  PM_TScope_ShowPttOutput->Checked= (CwKeyer_TimingScope.cfg.iVisibleChannels & (1<<TIMING_SCOPE_CHANNEL_FOUR      ) ) != 0;
  PM_TScope_ShowAudioIn->Checked  = (CwKeyer_TimingScope.cfg.iVisibleChannels & (1<<TIMING_SCOPE_CHANNEL_AUDIO_INPUT) ) != 0;
  PM_TScope_ShowAudioOut->Checked = (CwKeyer_TimingScope.cfg.iVisibleChannels & (1<<TIMING_SCOPE_CHANNEL_AUDIO_OUTPUT)) != 0;
  PM_TScope_ShowAudioSpectrumAndCwDecoder->Checked = (CwKeyer_TimingScope.cfg.iVisibleChannels & (1<<TIMING_SCOPE_CHANNEL_AUDIO_CW_DEC)) != 0;
  PM_TScope_Timebase->Caption = AnsiString(TE("&Timebase")) + " : " + IntToStr(CwKeyer_TimingScope.cfg.iMillisecondsPerSample) + " ms / sample";
  MI_TimingScopeTB_2ms->Checked  = (CwKeyer_TimingScope.cfg.iMillisecondsPerSample==2);
  MI_TimingScopeTB_5ms->Checked  = (CwKeyer_TimingScope.cfg.iMillisecondsPerSample==5);
  MI_TimingScopeTB_10ms->Checked = (CwKeyer_TimingScope.cfg.iMillisecondsPerSample==10);
  MI_TimingScopeTB_20ms->Checked = (CwKeyer_TimingScope.cfg.iMillisecondsPerSample==20);
  PM_TScope_FreeRun->Checked = (CwKeyer_TimingScope.cfg.iTriggerOptions & TIMING_SCOPE_TRIGGER_FREE_RUN) != 0;
  PM_TScope_Paused->Checked = CwKeyer_TimingScope.fPaused;
  MI_AudioSpectrumRefLevel_m20->Checked = (CwKeyer_TimingScope.cfg.iAudioSpectrumRefLevel_dB == -20);
  MI_AudioSpectrumRefLevel_m10->Checked = (CwKeyer_TimingScope.cfg.iAudioSpectrumRefLevel_dB == -10);
  MI_AudioSpectrumRefLevel_0->Checked   = (CwKeyer_TimingScope.cfg.iAudioSpectrumRefLevel_dB == 0);
  MI_AudioSpectrumRefLevel_10->Checked  = (CwKeyer_TimingScope.cfg.iAudioSpectrumRefLevel_dB == 10);
  MI_AudioSpectrumRefLevel_20->Checked  = (CwKeyer_TimingScope.cfg.iAudioSpectrumRefLevel_dB == 20);
  MI_TScope_AudioSpectrumAmplRange_10->Checked = (CwKeyer_TimingScope.cfg.iAudioSpectrumAmplRange_dB==10);
  MI_TScope_AudioSpectrumAmplRange_20->Checked = (CwKeyer_TimingScope.cfg.iAudioSpectrumAmplRange_dB==20);
  MI_TScope_AudioSpectrumAmplRange_30->Checked = (CwKeyer_TimingScope.cfg.iAudioSpectrumAmplRange_dB==30);
  MI_TScope_AudioSpectrumAmplRange_50->Checked = (CwKeyer_TimingScope.cfg.iAudioSpectrumAmplRange_dB==50);
  MI_TScope_AudioSpectrumAmplRange_70->Checked = (CwKeyer_TimingScope.cfg.iAudioSpectrumAmplRange_dB==70);
  PM_TScope_AudioSpectrumRefLevel->Caption = "Audio Spectrum Reference Level : " + IntToStr(CwKeyer_TimingScope.cfg.iAudioSpectrumRefLevel_dB) + " dB";
  PM_TScope_AudioSpectrumAmplitudeRange->Caption="Audio Spectrum Amplitude Range : " + IntToStr(CwKeyer_TimingScope.cfg.iAudioSpectrumAmplRange_dB) + " dB";
  PM_TScope_AudioSpectrumOptions->Caption = "Audio Spectrum options (Ref="
      +IntToStr(CwKeyer_TimingScope.cfg.iAudioSpectrumRefLevel_dB) + "dB, Range="
      +IntToStr(CwKeyer_TimingScope.cfg.iAudioSpectrumAmplRange_dB)+ "dB)";
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::PM_TScope_ShowDashInputClick(TObject *Sender)
{ CwKeyer_TimingScope.cfg.iVisibleChannels ^= (1<<TIMING_SCOPE_CHANNEL_DASH_INPUT );
  TimingScope_iUpdateCountOnTestTab = TimingScope_iUpdateCountOnKeyerTab = -1; // redraw a.s.a.p.
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::PM_TScope_ShowDotInputClick(TObject *Sender)
{ CwKeyer_TimingScope.cfg.iVisibleChannels ^= (1<<TIMING_SCOPE_CHANNEL_DOT_INPUT );
  TimingScope_iUpdateCountOnTestTab = TimingScope_iUpdateCountOnKeyerTab = -1;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::PM_TScope_ShowKeyingOutputClick(TObject *Sender)
{ CwKeyer_TimingScope.cfg.iVisibleChannels ^= (1<<TIMING_SCOPE_CHANNEL_CW_OUTPUT );
  TimingScope_iUpdateCountOnTestTab = TimingScope_iUpdateCountOnKeyerTab = -1;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::PM_TScope_ShowPttOutputClick(TObject *Sender) // .. or "Channel Four" ..
{ CwKeyer_TimingScope.cfg.iVisibleChannels ^= (1<<TIMING_SCOPE_CHANNEL_FOUR );
  TimingScope_iUpdateCountOnTestTab = TimingScope_iUpdateCountOnKeyerTab = -1;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::PM_TScope_ShowAudioInClick(TObject *Sender)
{ CwKeyer_TimingScope.cfg.iVisibleChannels ^= (1<<TIMING_SCOPE_CHANNEL_AUDIO_INPUT);
  TimingScope_iUpdateCountOnTestTab = TimingScope_iUpdateCountOnKeyerTab = -1;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::PM_TScope_ShowAudioOutClick(TObject *Sender)
{ CwKeyer_TimingScope.cfg.iVisibleChannels ^= (1<<TIMING_SCOPE_CHANNEL_AUDIO_OUTPUT);
  TimingScope_iUpdateCountOnTestTab = TimingScope_iUpdateCountOnKeyerTab = -1;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::PM_TScope_ShowAudioSpectrumAndCwDecoderClick(TObject *Sender)
{ CwKeyer_TimingScope.cfg.iVisibleChannels ^= (1<<TIMING_SCOPE_CHANNEL_AUDIO_CW_DEC);
  // When turning on the AUDIO SPECTRUM and AUDIO CW DECODER *display* here,
  // also turn on the audio CW decoder *in the DSP* :
  if( CwKeyer_TimingScope.cfg.iVisibleChannels & (1<<TIMING_SCOPE_CHANNEL_AUDIO_CW_DEC) )
   { CwKeyer_DSP.cfg.iAudioFlags |= DSP_AUDIO_FLAGS_DECODE_CW;
   }
  // The opposite isn't true: Turn off the AUDIO SPECTRUM and audio decoder *display*
  //     in the "timing scope" doesn't turn off the audio CW decoder itself !
  TimingScope_iUpdateCountOnTestTab = TimingScope_iUpdateCountOnKeyerTab = -1;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_TimingScopeTB_2msClick(TObject *Sender)
{ CwKeyer_TimingScope.cfg.iMillisecondsPerSample = 2;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_TimingScopeTB_5msClick(TObject *Sender)
{ CwKeyer_TimingScope.cfg.iMillisecondsPerSample = 5;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_TimingScopeTB_10msClick(TObject *Sender)
{ CwKeyer_TimingScope.cfg.iMillisecondsPerSample = 10;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_TimingScopeTB_20msClick(TObject *Sender)
{ CwKeyer_TimingScope.cfg.iMillisecondsPerSample = 20;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::PM_TScope_FreeRunClick(TObject *Sender)
{ CwKeyer_TimingScope.cfg.iTriggerOptions ^= TIMING_SCOPE_TRIGGER_FREE_RUN;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::PM_TScope_PausedClick(TObject *Sender)
{ CwKeyer_TimingScope.fPaused = !CwKeyer_TimingScope.fPaused;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::PM_TScope_ClearOverlaysClick(TObject *Sender)
{ KeyerGUI_ClearScopeOverlays();
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::PM_TScope_HelpClick(TObject *Sender)
{ YHF_HELP_ShowHelpContext(HELPID_TIMING_SCOPE);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::PM_ErrorHistory_HelpClick(TObject *Sender)
{ // So far, the "debug tab" only contains the "error history".
  // But since 2025, the "error history" can even be used to LIST and EDIT certain
  // settings in the remote rig, explained in the manual - thus this 'Help' link:
  YHF_HELP_ShowHelpContext(HELPID_DEBUG_TAB);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_ClearScopeOverlaysClick(TObject *Sender)
{ KeyerGUI_ClearScopeOverlays();
}

void __fastcall TKeyerMainForm::MI_ClearDebugAndDecoderOutputClick(TObject *Sender)
{ Ed_ErrorHistory->Clear();
}
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_ReportTestResultsClick(TObject *Sender)
{
  int i, n, iSum, iClientWidth, iPageControlWidth, iChildWidth;
  char sz255[256], *pszDest, *pszEndstop = sz255+255;
  char sz15Buf[16];
  int iErrorClass = ERROR_CLASS_INFO | SHOW_ERROR_IN_RUN_LOG;
  T_DSW_InputCaptureHistoryEntry *pCH;
  double d;
  BOOL first;
  int iRigCtrlPort;
  T_RigCtrl_PortInstance *pRigctrlPort;
#if( SWI_NUM_AUX_COM_PORTS > 0 )
  T_AuxComPortInstance *pAuxCom;
#endif // SWI_NUM_AUX_COM_PORTS > 0 ?


  Ed_ErrorHistory->Clear();
  KeyerGUI_fAutomaticTabSwitching = FALSE; // stop AUTOMATIC tab-switching, e.g. from DEBUG- to TRX tab
  SwitchMainTab( KEYER_GUI_MAIN_TAB_DEBUG ); // here: called to "report TEST RESULTS in the debug tab"
  ShowError( iErrorClass, "KEYER THREAD.." );
  ShowError( iErrorClass, " Poll inputs   : av= %d us, pk= %d us",
            (int)CwKeyer_GetSpeedTestAverage_us(KEYER_SPEEDTEST_POLL_INPUTS),
            (int)CwKeyer_iSpeedTestPeaks_us[KEYER_SPEEDTEST_POLL_INPUTS] );
  ShowError( iErrorClass, " Drive outputs : av= %d us, pk= %d us",
            (int)CwKeyer_GetSpeedTestAverage_us(KEYER_SPEEDTEST_SET_OUTPUTS),
            (int)CwKeyer_iSpeedTestPeaks_us[KEYER_SPEEDTEST_SET_OUTPUTS] );
  if( CwKeyer_Config.fKeyViaShiftAndControl )
   { ShowError(iErrorClass," Poll keyboard: av= %d us, pk= %d us",
            (int)CwKeyer_GetSpeedTestAverage_us(KEYER_SPEEDTEST_POLL_KEYBRD),
            (int)CwKeyer_iSpeedTestPeaks_us[KEYER_SPEEDTEST_POLL_KEYBRD] );
   }
  pszDest = sz255; // Convert the 'CW Keyer-Thread loop intervals' into a string:
  for(i=0; i<8; ++i)
   { SL_AppendInt( &pszDest, pszEndstop, CwKeyer_dw8ThreadIntervals_us[i] );
     SL_AppendChar(&pszDest, pszEndstop, ' ' );
   }
  ShowError( iErrorClass, " Loop intervals: %sus", sz255 );
     // (the above may sooner of later replace the display in 'Ed_ThreadInfo',
     //  because the user will now be interested in all this stuff.. )
  CwKeyer_ResetSpeedTestResults(); // begin a new "peak- and average detection"
  if( ! Keyer_fDotInputWasPassive )
   { ShowError( ERROR_CLASS_INFO | SHOW_ERROR_IN_RUN_LOG," Error: DOT-input seems to be stuck 'active' !" );
   }
  if( ! Keyer_fDashInputWasPassive )
   { ShowError( ERROR_CLASS_INFO | SHOW_ERROR_IN_RUN_LOG," Error: DASH-input seems to be stuck 'active' !" );
   }
  if( (Keyer_RadioControlPortRxFifo.fifo.dwTotalBytesWritten + Keyer_RadioControlPortTxFifo.fifo.dwTotalBytesWritten) != 0 )
   { ShowError( iErrorClass, " Radio Ctl Port: %lu bytes rcvd, %lu bytes sent",
          (unsigned long)Keyer_RadioControlPortRxFifo.fifo.dwTotalBytesWritten,
          (unsigned long)Keyer_RadioControlPortTxFifo.fifo.dwTotalBytesWritten );
   }

#if(SWI_HARDCORE_DEBUGGING) // (1) = hardcore-debugging, (0)=normal compilation
  ShowError( iErrorClass, " Source line nr: %d", (int)Keyer_iLastSourceLine );
#endif // SWI_HARDCORE_DEBUGGING ?

  ShowError(iErrorClass, "DSP THREAD.." );
  ShowError(iErrorClass," Audio input   : av= %d us, pk= %d us",
            (int)CwDSP_GetSpeedTestAverage_us( &CwKeyer_DSP, DSP_SPEEDTEST_READ_AUDIO_INPUT),
            (int)CwKeyer_DSP.iSpeedTestPeaks_us[DSP_SPEEDTEST_READ_AUDIO_INPUT] );
  ShowError(iErrorClass," Audio output  : av= %d us, pk= %d us",
            (int)CwDSP_GetSpeedTestAverage_us( &CwKeyer_DSP, DSP_SPEEDTEST_READ_AUDIO_OUTPUT),
            (int)CwKeyer_DSP.iSpeedTestPeaks_us[DSP_SPEEDTEST_READ_AUDIO_OUTPUT] );
  ShowError(iErrorClass," Downsampling  : av= %d us, pk= %d us",
            (int)CwDSP_GetSpeedTestAverage_us( &CwKeyer_DSP, DSP_SPEEDTEST_DOWNSAMLE_INPUT),
            (int)CwKeyer_DSP.iSpeedTestPeaks_us[DSP_SPEEDTEST_DOWNSAMLE_INPUT] );
  ShowError(iErrorClass," Upsampling    : av= %d us, pk= %d us",
            (int)CwDSP_GetSpeedTestAverage_us( &CwKeyer_DSP, DSP_SPEEDTEST_UPSAMPLE_OUTPUT),
            (int)CwKeyer_DSP.iSpeedTestPeaks_us[DSP_SPEEDTEST_UPSAMPLE_OUTPUT] );
  ShowError(iErrorClass," Audio Spectrum: av= %d us, pk= %d us", // .. for the AUDIO CW DECODER
            (int)CwDSP_GetSpeedTestAverage_us( &CwKeyer_DSP, DSP_SPEEDTEST_DECODER_SPECTRUM),
            (int)CwKeyer_DSP.iSpeedTestPeaks_us[DSP_SPEEDTEST_DECODER_SPECTRUM] );

  pszDest = sz255; // Convert the 'DSP Keyer-Thread loop intervals' into a string:
  for(i=0; i<8; ++i)
   { SL_AppendInt( &pszDest, pszEndstop, CwKeyer_DSP.dw8ThreadIntervals_us[i] );
     SL_AppendChar(&pszDest, pszEndstop, ' ' );
   }
  ShowError(iErrorClass," Loop intervals: %sus", sz255 );
  CwDSP_ResetSpeedTestResults( &CwKeyer_DSP ); // begin a new "peak- and average detection"
  pszDest = sz255; // Convert the 'Sidetone audio samples per Keyer-Thread-loop' into a string:
  for(i=0; i<16; ++i) // (used to test the proper function of CwDSP_UpdateSidetone() )
   { SL_AppendInt( &pszDest, pszEndstop, CwKeyer_DSP.i16SidetoneSamplesPerUpdate[i] );
     SL_AppendChar(&pszDest, pszEndstop, ' ' );
   }
  ShowError(iErrorClass," SidetoneSamples: %s", sz255 );
  if( CwKeyer_DSP.pDSW != NULL )
   { ShowError(iErrorClass, "Audio I/O ('DirectSound Wrapper').." );
     ShowError(iErrorClass," Input from '%s' :", CwKeyer_DSP.pDSW->sz255InputDeviceName );
     ShowError(iErrorClass,"    %lu kSamples read, %ld kS captured, %d Overflows, SR error=%+.1lf ppm",
              (unsigned long)(CwKeyer_DSP.pDSW->dblSamplesReadFromInput/1024.0),
              (unsigned long)(CwKeyer_DSP.pDSW->dblSamplesCapturedFromInput/1024.0),
              (int)CwKeyer_DSP.pDSW->nInputOverflows,
              (double)CwKeyer_DSP.pDSW->dblInputSRCalibError_ppm );
     // Show the most recent Direct Sound 'Read Positions' and their TIMESTAMPS, from dsound_wrapper.c :
     pszDest = sz255;
     d = CwKeyer_DSP.pDSW->sCaptureHistory[DSW_CAPTURE_HISTORY_LENGTH-1].dblTimestamp_s;
     for(i=0; i<8; ++i)
      { pCH = &CwKeyer_DSP.pDSW->sCaptureHistory[i];
        SL_AppendPrintf( &pszDest, pszEndstop, "%ld,%.1lf ", (long)pCH->dwReadPos,(double)((pCH->dblTimestamp_s-d) * 1e3) );
        d = pCH->dblTimestamp_s;
      }
     ShowError(iErrorClass,"    RdPos+dT/ms: %s", sz255 );

     // From the timestamps and and audio sample counter in dsound_wrapper.c,
     // show a few subsequent readings of the MEASURED audio input sample rate:
     pszDest = sz255;
     for(i=0; i<=5; ++i)
      { SL_AppendPrintf( &pszDest, pszEndstop, "%.1lf ",
         (double)DSW_GetInputSampleRateFromHistory( CwKeyer_DSP.pDSW, i ) );
      }
     SL_AppendPrintf( &pszDest, pszEndstop, " avrg=%.1lf ",
         (double)DSW_GetMeasuredInputSampleRate( CwKeyer_DSP.pDSW ) );
     ShowError(iErrorClass,"    Measured SR: %sHz", sz255 );


     // Show the most recent 'timestamp jitter' measured in dsound_wrapper.c :
     pszDest = sz255;
     for(i=iSum=0; i<10; ++i)
      { pCH = &CwKeyer_DSP.pDSW->sCaptureHistory[i];
        SL_AppendInt( &pszDest, pszEndstop, pCH->iTimestampJitter_ms );
        SL_AppendChar(&pszDest, pszEndstop, ' ' );
        iSum += pCH->iTimestampJitter_ms;
        // '--> The SUM of timestamp jitter values should be ZERO in the long run.
        //      Anything else indicates a problem with the timestamped audio input
        //      in C:\cbproj\Remote_CW_Keyer\dsound_wrapper.c .
      }
     SL_AppendPrintf( &pszDest, pszEndstop, "ms, sum=%d ms", (int)iSum );
     ShowError(iErrorClass,"    TS Jitter : %s", sz255 );
     ShowError(iErrorClass,"    Last Error: %s", CwKeyer_DSP.pDSW->sz255LastInputError );
     ShowError(iErrorClass," Output to '%s' :", CwKeyer_DSP.pDSW->sz255OutputDeviceName );
       // ,-------------------------------------------------------|___________________|
       // '--> 2024-02-25 : Contained a garbage characters when there was
       //                   NO OUTPUT DEVICE SELECTED at all !
       //  The same garbage also in MyDirectSound.sz255OutputDeviceName .
       //  On first sight, memory trashed somewhere during the call
       //  to VorbisStream_Init() . Tried to examine this in CheckSystemHealth().
       //
     ShowError(iErrorClass,"    %lu kSamples written, %lu kS played, %d Underflows",
              (unsigned long)(CwKeyer_DSP.pDSW->dblSamplesWrittenToOutput/1024.0),
              (unsigned long)(CwKeyer_DSP.pDSW->dblSamplesPlayedToOutput/1024.0),
              (int)CwKeyer_DSP.pDSW->nOutputUnderflows );
     ShowError(iErrorClass,"    Last Error: %s", CwKeyer_DSP.pDSW->sz255LastOutputError );
   } // end if( CwKeyer_DSP.pDSW != NULL )

  // Show what's happening in the RIG CONTROL module (when IN USE) ?
  if( CwKeyer_Config.iRadioCIVAddress >= 0 ) // Rig-Control enabled (actively controlling, or eavesdropping on a radio) ?
   { T_RigCtrlInstance *pRC = &MyCwNet.RigControl;
     pszDest = sz255;
     SL_AppendString( &pszDest, pszEndstop, "RIG CONTROL (");
     if( pRC->fListenOnlyMode )
      { SL_AppendString( &pszDest, pszEndstop, "in Listen Only Mode"); // <- term inspired by CAN (Controller Area Network)
      }
     else
      { SL_AppendPrintf( &pszDest, pszEndstop, "on %s", KeyerGUI_PortNumberToString(CwKeyer_Config.iRadioKeyingAndControlPort) );
      }
     SL_AppendString( &pszDest, pszEndstop, ")..");
     ShowError(iErrorClass, sz255 );
     ShowError(iErrorClass, " %ld spectra, %d bins, %ld VFO reports, %s",
       (long)pRC->nCompleteSpectraReceived,
        (int)pRC->nSpectrumBinsUsed,
       (long)pRC->iFrequencyModifiedByRadio_cnt,
       RigCtrl_VfoFrequencyToString( pRC->dblVfoFrequency, sz15Buf) );

     // Show what's happening on the RIG CONTROL PORTS (when IN USE) ?
     for( iRigCtrlPort=0; iRigCtrlPort<RIGCTRL_NUM_PORT_INSTANCES; ++iRigCtrlPort )
      { // '--> index into pRC->PortInstance[] :
        //       0 = RIGCTRL_PORT_RADIO, 1 = RIGCTRL_PORT_AUX_COM_1, ... !
        pRigctrlPort = &pRC->PortInstance[iRigCtrlPort];
        // Only show the number of messages (or the number of 'garbage bytes') on this port when active (nonzero counts):
        if( (pRigctrlPort->dwNumMessagesSent + pRigctrlPort->dwNumMessagesRcvd + pRigctrlPort->dwNumGarbageBytes) != 0 )
         { ShowError(iErrorClass, " %ld messages sent, %ld received on '%s'",
             (long)pRigctrlPort->dwNumMessagesSent, (long)pRigctrlPort->dwNumMessagesRcvd,
             RigCtrl_RigControlPortNrToString(iRigCtrlPort) );
           if( pRigctrlPort->dwNumGarbageBytes > 0 )
            { ShowError(iErrorClass, " %ld bytes of unrecognized garbage on '%s'",
              (long)pRigctrlPort->dwNumGarbageBytes, RigCtrl_RigControlPortNrToString(iRigCtrlPort) );
            }
         }
      } // end for < iRigCtrlPort >
     ShowError(iErrorClass, " Parameter polling state : %s, SubState=%d",
        RigCtrl_ParameterPollingStateToString(pRC->iParameterPollingState),
        (int)pRC->iParameterPollingSubState );
     ShowError(iErrorClass, " TxRqst=%d, Txing=%d", (int)pRC->iTransmitReqst, (int)pRC->iTransmitting );
   } // end if( MyCwNet.RigControl.initState == initState_Opened )


  // Show what's going on in the 'CW NETWORK' module:
  ShowError(iErrorClass, "NETWORK THREAD.." );
  ShowError(iErrorClass, " Status: %s", CwNet_GetCurrentStatusAsString( &MyCwNet ) );
#if( SWI_HARDCORE_DEBUGGING )  // show line number of a potentially 'crashed worker thread':
  ShowError(iErrorClass, " Source lines  : CwNet=%d, HttpSrv=%d",
                 (int)CwNet_iLastSourceLine, (int)HttpSrv_iLastSourceLine );
    // Murphy will make sure the program NEVER crashes under debugger control !
#endif // SWI_HARDCORE_DEBUGGING ?
  ShowError(iErrorClass, " Last Error    : %s",CwNet_GetLastErrorAsString( &MyCwNet ) );
  // ex: ShowError(iErrorClass, " Bytes RX,TX   : %ld, %ld", (long)MyCwNet.dwNumBytesRcvd, (long)MyCwNet.dwNumBytesSent);
  ShowError(iErrorClass, " Network Thread: %ld loops, %ld errors, RxPeak=%d bytes",
                (long)MyCwNet.dwThreadLoops,
                (long)MyCwNet.dwThreadErrors,
                 (int)MyCwNet.iPeakRxBufferUsage);
  pszDest = sz255; // Convert the 'Network Thread loop intervals' into a string:
  for(i=0; i<8; ++i)
   { SL_AppendInt( &pszDest, pszEndstop, MyCwNet.dw8ThreadIntervals_us[i] );
     SL_AppendChar(&pszDest, pszEndstop, ' ' );
   }
  ShowError(iErrorClass, " Loop intervals: %s us", sz255 );
  ShowError(iErrorClass, " Network Audio : %ld kSamples sent, %ld kSamples rcvd",
      MyCwNet.i32AudioSamplesSent_k, MyCwNet.i32AudioSamplesRcvd_k );
  n = MyCwNet.nRemoteClientsConnected;
  for( first=FALSE, i=0; i<=/*!*/CWNET_MAX_CLIENTS; ++i )
   { // i=0 is the "local client" (when operating AS CLIENT),
     // i=1..CWNET_MAX_CLIENTS are the "remote clients" (when operating AS SERVER).
     // Because REMOTE clients may 'come and go', array indices aren't predictable.
     T_CwNetClient *pClient = &MyCwNet.Client[i];
     if( pClient->iClientState != CWNET_CLIENT_STATE_DISCONN )
      { T_HttpInstance *pHttpInst;
        if( first )
         { first = FALSE;
           ShowError(iErrorClass, "Currently active connection(s) :" );
         }
#      if( SWI_USE_HTTP_SERVER )  // support the optional, built-in HTTP server ?

        if( pClient->fUsesHTTP && ((pHttpInst=pClient->pHttpInst)!=NULL) )
         { ShowError(iErrorClass, " #%d: %s on %s:%d, %s%s, rx=%lu, tx=%lu, err='%s'",
             (int)i, // --------------'   |    |____|  |        '----,   |   |
              pHttpInst->pSession->sz40UserName,// |   '-------,     |   |   |
              CwNet_IPv4AddressToString(pClient->b4HisIP.b),// |     |   |   |
              (int)pClient->iHisPort,                     //  \|/   \|/ \|/ \|/ ...
              HttpSrv_MethodToString(pHttpInst->nMethod), // GET, POST, PUT, or what the heck..
              pClient->pHttpInst->sz255RequestedURLWithoutQuery,
              (unsigned long)pHttpInst->i64TotalBytesRcvd,
              (unsigned long)pHttpInst->i64TotalBytesSent,
              pHttpInst->sz80ErrorMessage );
            // Listing ALL CONNECTIONS (~~sockets) like this revealed why
            // something "seemed to be missing" sometimes, when following a single
            // HTTP stream with Wireshark: The browser bombarded the poor little
            // web server with MULTIPLE connections, sometimes even requesting
            // THE SAME RESOURCE :
            // > Currently active connection(s) :
            // >  #1: Moritz on 127.0.0.1:53780, GET /LiveData.htm, rx=63550, tx=45725, err=''
            // >  #2: Moritz on 127.0.0.1:53785, GET /LiveData.htm, rx=62730, tx=45135, err=''
            // ,-------------------------------------------------------'
            // '--> These byte counters confirmed that BOTH socket were 'alive' .
            //      The RECEIVE counter increasing faster than the TRANSMIT counter
            //      was caused by the chatty HTTP REQUEST HEADERS (with soo many lines);
            //      they were larger than the CONTENT (delivered to the Javascript
            //      from the dynamically generated HTML fragment 'LiveData.htm').
         }
        else // not using HTTP (using a web browser) but our simple binary TCP/IP based protocol:
#      endif // SWI_USE_HTTP_SERVER ?
         { ShowError(iErrorClass, " #%d: IP=%s, %s, latency=%d ms", (int)i,
              CwNet_IPv4AddressToString( pClient->b4HisIP.b ),
              CwNet_ClientStateToString( pClient->iClientState ),
              (int)CwNet_GetLatencyForRemoteClient_ms( &MyCwNet, i ) );
         }
      }
   } // end for < all NETWORK clients, including the server's "local" client >

# if( SWI_USE_VORBIS_STREAM ) // allow streaming audio via Ogg/Vorbis ?
  if( MyCwNet.sVorbis.dwInitMagicE == VORBIS_MAGIC_E ) // successfully initialized ?
   { ShowError(iErrorClass, "Ogg/Vorbis encoder: %lu kSamples encoded, %d kS/s, %lu Ogg pages,",
               (unsigned long)(MyCwNet.sVorbis.dwNumSamplesWritten/1000),
                         (int)(MyCwNet.sVorbis.fltSampleRate*1e-3),
               (unsigned long)MyCwNet.sVorbis.dwPageIndex );
     pszDest = sz255; // show the most recent Ogg page sizes:
     for( i=0; i<20; ++i )
      { int iPageBuf = (int)MyCwNet.sVorbis.iBufferPagesHead - 1 - i;
        if( iPageBuf < 0 )
         {  iPageBuf += VORBIS_STREAM_BUFFER_MAX_PAGES;
         }
        if( (iPageBuf>=0) && (iPageBuf<VORBIS_STREAM_BUFFER_MAX_PAGES) )
         { SL_AppendPrintf( &pszDest, pszEndstop, "%d ",
               (int)MyCwNet.sVorbis.BufferPages[iPageBuf].iSize_Byte );
         }
      }
     ShowError(iErrorClass, " Ogg pages: %s bytes", sz255 );
   } // end if < Ogg/Vorbis encoder initialized ("magic E set") ? >
# endif // SWI_USE_VORBIS_STREAM

# if( SWI_USE_HAMLIB_SERVER ) // not sure if this 'feature' is going to stay...
  if( CwKeyer_Config.fEnableBuiltInHamlibServer )
   { ShowError(iErrorClass, "HAMLIB SERVER.." );
     ShowError(iErrorClass, " Status: %s", HLSRV_GetCurrentStatusAsString( &MyHamlibServer ) );
   } // end if( CwKeyer_Config.fEnableBuiltInHamlibServer )
# endif // SWI_USE_HAMLIB_SERVER ?

  // Show what's going on with the 'Auxiliary / Additional Serial Ports' ?
#if( SWI_NUM_AUX_COM_PORTS > 0 ) // compile with support for 'Auxiliary / Additional COM ports' ?
  for(int iPort=0; iPort<SWI_NUM_AUX_COM_PORTS; ++iPort )
   { pAuxCom = &AuxComPorts[iPort];
     if( AuxCom_IsInUse( pAuxCom ) )
      { ShowError(iErrorClass, "ADDITIONAL COM PORT #%d (on %s)..",
            (int)iPort+1, KeyerGUI_PortNumberToString(pAuxCom->iComPortNumber) );
        ShowError(iErrorClass, " Status: %s", AuxCom_GetCurrentStatusAsString( pAuxCom ) );
        ShowError( iErrorClass," Ser. I/O times: av= %d us, pk= %d us",
            (int)AuxCom_GetSpeedTestAverage_us(pAuxCom, AUXCOM_SPEEDTEST_SERIAL_IO),
            (int)pAuxCom->iSpeedTestPeaks_us[AUXCOM_SPEEDTEST_SERIAL_IO] );
        ShowError( iErrorClass," Signal States : %s", AuxCom_SignalStatesToString( pAuxCom, sz255, 40 ) );
        pszDest = sz255; // Convert the 'Aux COM Port Thread loop intervals' into a string:
        for(i=0; i<8; ++i)
         { SL_AppendInt( &pszDest, pszEndstop, pAuxCom->dw8ThreadIntervals_us[i] );
           SL_AppendChar(&pszDest, pszEndstop, ' ' );
         }
        ShowError(iErrorClass, " Loop intervals: %s us", sz255 );
      }  // end if( AuxCom_IsInUse( pAuxCom ) )
     AuxCom_ResetSpeedTestResults( pAuxCom );
   }    // end for < all SWI_NUM_AUX_COM_PORTS > 
#endif    // SWI_NUM_AUX_COM_PORTS > 0 ?


  // Dump the thread-safe FIFO (filled by ShowError()) into the thread-unsafe 'RichEdit' control,
  // even if the output on the 'Debug' tab is currently PAUSED :
  KeyerGUI_DumpErrorHistoryToRichText(Ed_ErrorHistory);  // here: to report 'test results' in the GUI

  (void)n;

} // end < ..ReportTestResults().. >

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_ListAndEditRigParamsClick(TObject *Sender)
  // Added 2024-07 to find out which parameters required by HamlibServer.c
  //               work properly, and which don't.
  // This report can be invoked through the main menu via
  //  "Settings".."List / Edit Rig Parameters ans Settings on the 'Debug' tab",
  // and through the context menu of the 'Debug'-tab's Rich Edit control itself.
{
  KeyerGUI_fAutomaticTabSwitching = FALSE; // stop AUTOMATIC tab-switching, e.g. from DEBUG- to TRX tab
  SwitchMainTab( KEYER_GUI_MAIN_TAB_DEBUG ); // here: to report the current set of 'Rig parameters' on the debug tab
  RigCtrl_fTrafficMonitorPausedByUser = m_fDebugOutputPaused = TRUE;
  KeyerGUI_ListRigParamsOnDebugTab( &MyCwNet.RigControl/*in*/, Ed_ErrorHistory/*out*/ );
} // end < ..ReportRigCtrlParams().. >

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Debug_ListAndEditRigParamsClick(TObject *Sender)
  // Almost the same as above, but invoked from the CONTEXT MENU
  //  on 'Debug tab' / 'Error History' .
{ RigCtrl_fTrafficMonitorPausedByUser = m_fDebugOutputPaused = TRUE;
  KeyerGUI_ListRigParamsOnDebugTab( &MyCwNet.RigControl/*in*/, Ed_ErrorHistory/*out*/ );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Test_EnumerateAudioDevicesClick(TObject *Sender)
{
  char sz255[256], *pszDest, *pszEndstop = sz255+255;
  char sz80Value[84];
  int i;
  int iErrorClass = ERROR_CLASS_INFO | SHOW_ERROR_IN_RUN_LOG;

  Ed_ErrorHistory->Clear();
  KeyerGUI_fAutomaticTabSwitching = FALSE; // stop AUTOMATIC tab-switching, e.g. from DEBUG- to TRX tab
  SwitchMainTab( KEYER_GUI_MAIN_TAB_DEBUG ); // here: to report the current set of 'Rig parameters' on the debug tab


  // This is basically the same as when filling CB_AudioInputDevice and CB_AudioOutputDevice .
  // It happened more than once (under certain Windows versions) that
  // a previously present audio device / modern radio like IC-9700 suddenly
  // DISAPPEARED from the list, or even from the audio devices enumerated by Windows itself !
#if( SWI_USE_DSOUND ) // use the mature "DirectSound" / dsound_wrapper.c ?
  // As a service for users trying to find out the stupid AUDIO DEVICE NAME
  //    of their brand-new gadget with built-in USB audio device,
  //    RE-ENUMERATE the device here even though they have been enumerated
  //    already, when launching the Remote CW Keyer application. This allows
  //    find out which devices disappear from the list when unplugging a certain
  //    USB cable on the PC side:
  DSW_EnumerateDevices( &MyDirectSound ); // RE-enumerate, here: for the list of audio devices on the 'Debug' tab
  ShowError( iErrorClass, "'Direct Sound' compatible audio input Devices:" );
  for(i=0; i<MyDirectSound.m_nInputDevices; ++i) // in Microsoft terms, these are "capture devices" ..
   { ShowError(iErrorClass, "  (%i) %s", (int)(i+1), MyDirectSound.m_sInputDevices[i].sz255DevName );
   }
  ShowError( iErrorClass, "'Direct Sound' compatible audio output Devices:" );
  for(i=0; i<MyDirectSound.m_nOutputDevices; ++i) // in Microsoft terms, these are "rendering devices"
   { ShowError(iErrorClass, "  (%i) %s", (int)(i+1), MyDirectSound.m_sOutputDevices[i].sz255DevName );
   }
#endif // support for the ancient "Windows Multimedia Extensions" / "wave audio" has been removed here


} // end TKeyerMainForm::MI_Test_EnumerateAudioDevicesClick()


//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_ListUserDefinedBandsClick(TObject *Sender)
  // Added 2024-12 to check if the optional "User Defined Bands",
  //       have been properly loaded from a file via RigCtrl_LoadBandsAndFrequenciesFromFile(),
  //       or, if this instance acts as a CLIENT, have been properly
  //       transferred from the remote SERVER).
  // This report can be invoked through the main menu via
  //       'Settings'..'List User Defined Bands on the 'Debug' tab' .
{
  char sz255[256], *pszDest, *pszEndstop = sz255+255;
  int iErrorClass = ERROR_CLASS_INFO | SHOW_ERROR_IN_RUN_LOG;
  T_RigCtrlInstance   *pRC = &MyCwNet.RigControl;
  T_RigCtrlUserDefinedBand *pUserDefdBand;
  T_RigCtrlFreqMemEntry    *pUserDefdFreq;
  int    iValue, i, i1, i2, n;
  double dblValue;

  Ed_ErrorHistory->Clear();
  KeyerGUI_fAutomaticTabSwitching = FALSE; // stop AUTOMATIC tab-switching, e.g. from DEBUG- to TRX tab
  SwitchMainTab( KEYER_GUI_MAIN_TAB_DEBUG ); // here: to report the current set of 'Rig parameters' on the debug tab
  ShowError( iErrorClass, "User Defined Bands (or bands received from the server): %d",
                           (int)pRC->iNumUserDefinedBands );

  for(i=0; i<pRC->iNumUserDefinedBands; ++i)
   { pUserDefdBand = &pRC->UserDefinedBands[i];
     pszDest = sz255;
     // The display format is SIMILAR to what is used in e.g. RCWKeyer_Bands.txt,
     // but not guaranteed to be parsable by RigCtrl_StringToUserDefinedBand().
     // (a tribute to user friendliness and vertically aligned "pretty printing")
     SL_AppendPrintf( &pszDest, pszEndstop, "%8.4lf .. %8.4lf MHz, modes=%s, perm=%s",
         // ,-----------------------------------'
         // '--> FOUR fractional digits because e.g. the 60-meter "WRC-15" band
         //      begins at 5.3515 MHz, not 5.351 MHz . Thus: "Accurate to 100 Hz".
         //      Fixed field widths at least for the mandatory part
         //      to have a pretty-printed output on the debug tab. Example below.
        (double)(pUserDefdBand->dblFmin_Hz * 1e-6),
        (double)(pUserDefdBand->dblFmax_Hz * 1e-6),
        RigCtrl_OperatingModeToString( pUserDefdBand->dwOpModes ),  // e.g. "USB", "LSB", "CW", ...
        RigCtrl_PermissionsToString( pUserDefdBand->dwPermissions)); // e.g. "none" or "TX"
     if( pUserDefdBand->sz16Name[0] != '\0' )
      { SL_AppendString( &pszDest, pszEndstop, ", na=" );
        SL_AppendDoubleQuotedString( &pszDest, pszEndstop, pUserDefdBand->sz16Name );
      }
     if( pUserDefdBand->dblFdef_Hz != 0.0 )
      { SL_AppendPrintf( &pszDest, pszEndstop, ", df=%lg MHz",
          (double)(pUserDefdBand->dblFdef_Hz * 1e-6) );
      }
     if( pUserDefdBand->dwDefOpMode != 0 )
      { SL_AppendPrintf( &pszDest, pszEndstop, ", DefMode=%s",
          RigCtrl_OperatingModeToString(pUserDefdBand->dwDefOpMode) );
      }
     ShowError( iErrorClass, sz255 );
   } // end for  < all "user-defined bands" >
  // Result from the above loop (example):
  // > User Defined Bands (or bands received from the server): 3
  // > 160m      :   1.8100 ..   1.8380 MHz, modes=CW, permissions=TX
  // > 80m       :   3.5000 ..   3.5700 MHz, modes=CW, permissions=TX
  // > 60m_EU    :   5.3515 ..   5.3540 MHz, modes=CW, permissions=TX

  // Dump the thread-safe FIFO (filled by ShowError()) into the thread-unsafe 'RichEdit' control,
  // even if the output on the 'Debug' tab is currently PAUSED :
  KeyerGUI_DumpErrorHistoryToRichText(Ed_ErrorHistory);  // here: to report 'test results' in the GUI
} // end ..ListUserDefinedBands..()

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_ListUserDefinedFrequenciesClick(TObject *Sender)
  // This report can be invoked through the main menu via
  //       'Settings'..'List User Defined Frequencies on the 'Debug' tab' .
{
  char sz255[256], *pszDest, *pszEndstop = sz255+255;
  int iErrorClass = ERROR_CLASS_INFO | SHOW_ERROR_IN_RUN_LOG;
  T_RigCtrlInstance   *pRC = &MyCwNet.RigControl;
  T_RigCtrlUserDefinedBand *pUserDefdBand;
  T_RigCtrlFreqMemEntry    *pUserDefdFreq;
  int    iValue, i, i1, i2, n;
  double dblValue;

  Ed_ErrorHistory->Clear();
  KeyerGUI_fAutomaticTabSwitching = FALSE; // stop AUTOMATIC tab-switching, e.g. from DEBUG- to TRX tab
  SwitchMainTab( KEYER_GUI_MAIN_TAB_DEBUG ); // here: to report the current set of 'Rig parameters' on the debug tab

  ShowError( iErrorClass, "User Defined Frequencies: %d", (int)pRC->iNumUserDefinedFrequencies );
  for(i=0; i<pRC->iNumUserDefinedFrequencies; ++i)
   { pUserDefdFreq = &pRC->UserDefinedFrequencies[i];
     pszDest = sz255;
     SL_AppendPrintf( &pszDest, pszEndstop, "%8.4lf : %s%s",
      // ,-----------------------------------------'   | |
      // '--> FOUR fractional digits because e.g.      | '-> Operating mode like
      //      the 60-meter "WRC-15" band               |     "CW", "USB", "LSB"
      //      begins at 5.3515 MHz, not 5.351 MHz .    '-> Token like "mo=" taken
      //      Thus: "Accurate to 100 Hz".                  from RigCtrl_FreqMemEntryTokens[]
       (double)(pUserDefdFreq->RxTx[0].dblOperatingFreq_Hz * 1e-6),
       (char*)SL_GetStringFromTokenList( RigCtrl_FreqMemEntryTokens, RIGCTRL_FREQ_MEM_TOKEN_OP_MODE ),
       (char*)RigCtrl_OperatingModeToString( pUserDefdFreq->RxTx[0].iOpMode ) );
     if( pUserDefdFreq->sz16Name[0] != '\0' )
      { // Append e.g.  na="<double-quoted string>"
        SL_AppendPrintf( &pszDest, pszEndstop, " %s",
           (char*)SL_GetStringFromTokenList( RigCtrl_FreqMemEntryTokens, RIGCTRL_FREQ_MEM_TOKEN_NAME ) );
        SL_AppendDoubleQuotedString( &pszDest, pszEndstop, pUserDefdFreq->sz16Name );
      }
     ShowError( iErrorClass, sz255 );
   }

  // Dump the thread-safe FIFO (filled by ShowError()) into the thread-unsafe 'RichEdit' control,
  // even if the output on the 'Debug' tab is currently PAUSED :
  KeyerGUI_DumpErrorHistoryToRichText(Ed_ErrorHistory);  // here: to report 'test results' in the GUI
} // end ..ListUserDefinedFrequencies..()


//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_ShowStraightKeyDecoderHistoryClick(TObject *Sender)
{
  int i, n, iClientWidth, iPageControlWidth, iChildWidth;
  char sz255[256], *pszDest, *pszEndstop = sz255+255;
  int iErrorClass = ERROR_CLASS_INFO | SHOW_ERROR_IN_RUN_LOG;


  Ed_ErrorHistory->Clear();
  KeyerGUI_fAutomaticTabSwitching = FALSE; // stop AUTOMATIC tab-switching, e.g. from DEBUG- to TRX tab
  SwitchMainTab( KEYER_GUI_MAIN_TAB_DEBUG ); // here: called to show activity of the 'Straight Key Decoder'

  // If the 'Straight Key Decoder' has new entries in its history, show them:
  if( CwKeyer_StraightKeyDecoder.iHistoryHeadIndex != CwKeyer_StraightKeyDecoder.iHistoryTailIndex )
   { ShowError(iErrorClass, "'Straight Key' decoder: t_dot=%d ms; history:",
             (int)CwKeyer_StraightKeyDecoder.iDotTime_ms ); // <- this helped a lot to "interpret" the following output..
     while( CwKeyer_StraightKeyDecoder.iHistoryTailIndex != CwKeyer_StraightKeyDecoder.iHistoryHeadIndex )
      { T_StraightKeyDecoderHistoryEntry *pHE = &CwKeyer_StraightKeyDecoder.sHistory[CwKeyer_StraightKeyDecoder.iHistoryTailIndex];
        CwKeyer_StraightKeyDecoder.iHistoryTailIndex = (CwKeyer_StraightKeyDecoder.iHistoryTailIndex+1) % SKD_HISTORY_N_ENTRIES;
        pszDest = sz255; // Convert the 'Network Thread loop intervals' into a string:
        SL_AppendPrintf( &pszDest, pszEndstop,
              " #%d: %s after %ld us : ShiftReg=0x%04X, new State=%s",
             (int)CwKeyer_StraightKeyDecoder.iHistoryTailIndex,
             (char*)(pHE->fKeyDown ? "Key Down" : "Key Up"),
             (long)pHE->nMicrosecondsBeforeNewState,
              (int)pHE->wShiftReg,
             (char*)StraightKeyDecoder_StateToString(pHE->iState) );
        if( pHE->wEmittedCode != 0 )
         { SL_AppendPrintf( &pszDest, pszEndstop, ", decoded: \"%s\"",
                 Elbug_MorseCodePatternToASCII( pHE->wEmittedCode ) );
         }
        ShowError(iErrorClass, sz255);
      }
   } // end if < new entries in the Straight Key Decoder's "history" > ?
  else
   { ShowError(iErrorClass, "'Straight Key' decoder: No new entries in the history." );
   }

  KeyerGUI_DumpErrorHistoryToRichText(Ed_ErrorHistory);  // here: to report 'test results' in the GUI
} // end < ..ShowStraightKeyDecoderHistory.. >


//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Server_EnableHTTPClick(TObject *Sender)
{ if( KeyerGUI_iUpdating==0 )
   { MyCwNet.cfg.iHttpServerOptions ^= HTTP_SERVER_OPTIONS_ENABLE;
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Server_RestrictAccessClick(TObject *Sender)
{ if( KeyerGUI_iUpdating==0 )
   { MyCwNet.cfg.iHttpServerOptions ^= HTTP_SERVER_OPTIONS_RESTRICT; // .. to visitors specifying a VALID USER NAME
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Server_AddTestToneClick(TObject *Sender)
{ if( KeyerGUI_iUpdating==0 )
   { CwKeyer_DSP.cfg.iAudioFlags ^= DSP_AUDIO_FLAGS_SEND_NETWORK_TEST_TONE;
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Network_DiagnosticModeClick(TObject *Sender)
{ if( KeyerGUI_iUpdating==0 )
   { MyCwNet.cfg.iDiagnosticFlags ^= CWNET_DIAG_FLAGS_VERBOSE;
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Network_SimulateBadConnectionClick(TObject *Sender)
{ if( KeyerGUI_iUpdating==0 )
   { MyCwNet.cfg.iDiagnosticFlags ^= CWNET_DIAG_FLAGS_SIMULATE_BAD_CONN;
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Network_ShowConnectionLogClick(TObject *Sender)
{ if( KeyerGUI_iUpdating==0 )
   { MyCwNet.cfg.iDiagnosticFlags ^= CWNET_DIAG_FLAGS_SHOW_CONN_LOG;
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Network_ShowNetworkTrafficClick(TObject *Sender)
{ if( KeyerGUI_iUpdating==0 )
   { MyCwNet.cfg.iDiagnosticFlags ^= CWNET_DIAG_FLAGS_SHOW_NETWORK_TRAFFIC;
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Network_TrafficLog_RejectPeriodicMsgsClick(TObject *Sender)
{ if( KeyerGUI_iUpdating==0 )
   { MyCwNet.cfg.iDiagnosticFlags ^= CWNET_DIAG_FLAGS_REJECT_PERIODIC_MSGS;
     // '--> reject e.g. CWNET_CMD_PING from the traffic log ('live' on the Debug tab)
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Test_ShowGuiEventsClick(TObject *Sender)
{ if( KeyerGUI_iUpdating==0 )
   { MyCwNet.cfg.iDiagnosticFlags ^= CWNET_DIAG_FLAGS_SHOW_GUI_EVENTS; // ... not necessarily limited to the "CW network", anyway ...
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Network_ShowRigControlTrafficClick(TObject *Sender)
{ RigCtrl_TrafficMonitor.iDisplayOptions ^= RIGCTRL_TMON_DISPLAY_OPTION_ENABLE;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_RigCtrl_RejectEchosInLogClick(TObject *Sender)
{ MyCwNet.RigControl.dwMessageFilterForLog ^= RIGCTRL_MSGFILTER_ECHO;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_RigCtrl_RejectPeriodicCmdsClick(TObject *Sender)
{
  MyCwNet.RigControl.dwMessageFilterForLog ^= RIGCTRL_MSGFILTER_PERIODIC_POLL;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_RigCtrl_RejectSpectrumInLogClick(TObject *Sender)
{ MyCwNet.RigControl.dwMessageFilterForLog ^= RIGCTRL_MSGFILTER_SPECTRUM;
}

void __fastcall TKeyerMainForm::MI_RigCtrl_RejectFrequencyReportsClick(TObject *Sender)
{ MyCwNet.RigControl.dwMessageFilterForLog ^= RIGCTRL_MSGFILTER_FREQUENCY_REPORT;
}
//---------------------------------------------------------------------------

void __fastcall TKeyerMainForm::MI_RigCtrl_RejectAnyKnownCommandFromLogClick(TObject *Sender)
{ MyCwNet.RigControl.dwMessageFilterForLog ^= RIGCTRL_MSGFILTER_ANY_KNOWN_COMMAND;
}
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Network_ShowRcvdKeyingBytestreamClick(TObject *Sender)
{ if( KeyerGUI_iUpdating==0 )
   { MyCwNet.cfg.iDiagnosticFlags ^= CWNET_DIAG_FLAGS_SHOW_KEYING_BYTESTREAM;
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_HLServer_OpenControlTabClick(TObject *Sender)
{
  if( (PageControl1->ActivePage!=TS_Config) || (PC_Config->ActivePage!=TS_RigControl) )
   { UpdateSettingsTab();
     SwitchMainTab( KEYER_GUI_MAIN_TAB_DEBUG ); // here: called from the main menu, "Hamlib Server" -> "Switch to the control tab"
     PC_Config->ActivePage = TS_RigControl;
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_HLServer_EnableClick(TObject *Sender)
{
  if( CwKeyer_Config.fEnableBuiltInHamlibServer ) // DISABLE now :
   {  CwKeyer_Config.fEnableBuiltInHamlibServer = FALSE;
      HLSRV_Stop( &MyHamlibServer );
   }
  else // ENABLE now :
   {  CwKeyer_Config.fEnableBuiltInHamlibServer = TRUE;
      if( MyHamlibServer.cfg.iServerListeningPort > 0 )
       { HLSRV_Start( &MyHamlibServer );
       }
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_HLServer_ShowTrafficClick(TObject *Sender)
{ MyHamlibServer.cfg.iDiagnosticFlags ^= HLSRV_DIAG_FLAGS_SHOW_NETWORK_TRAFFIC;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_HLServer_VerboseOutputClick(TObject *Sender)
{ MyHamlibServer.cfg.iDiagnosticFlags ^= HLSRV_DIAG_FLAGS_VERBOSE_OUTPUT;
}


//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_KeyViaShiftAndControlClick(TObject *Sender)
{ CwKeyer_Config.fKeyViaShiftAndControl = !CwKeyer_Config.fKeyViaShiftAndControl;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_MoreKeyerSettings_DisableTxClick(TObject *Sender)
{ CwKeyer_Config.fDisableTx = !CwKeyer_Config.fDisableTx;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Network_OpenServerPageInBrowserClick(TObject *Sender)
{
  char sz255URLAndQueryString[256];
  char *pszDest    = sz255URLAndQueryString;
  char *pszEndstop = sz255URLAndQueryString + 255;
  BOOL fOk;

  sz255URLAndQueryString[0] = '\0';

  // Whether to open the 'Server Page' via LOCALHOST or an 'external' URL
  // depends on the current setting on the 'Network' tab :
  if( Rbtn_NetworkClient->Checked )  // this instance operates as CLIENT so use the REMOTE SERVER's URL:
   { SL_AppendString( &pszDest, pszEndstop, "http://" );
     SL_AppendString( &pszDest, pszEndstop, AnsiString(Ed_ClientRemoteIP->Text).c_str() );
     if( SL_GetLastChar(sz255URLAndQueryString, 80) != '/' )
      { SL_AppendString( &pszDest, pszEndstop, "/" );
      }
     SL_AppendString( &pszDest, pszEndstop, "?user=" );
     INET_PlainTextToUrlEncoding(
        AnsiString(Ed_ClientUserName->Text).c_str(), // [in] char * pszPlainTextSource,
        &pszDest, pszEndstop,            // [out] URL encoded result
        "\0"/*pszDelimiters*/ );
     SL_AppendString( &pszDest, pszEndstop, "&call=" );
     INET_PlainTextToUrlEncoding(
        AnsiString(Ed_ClientCallsign->Text).c_str(),
        &pszDest, pszEndstop,            // [out] URL encoded result
        "\0"/*pszDelimiters*/ );
   } // end if < instance operates as CLIENT >
  else if( Rbtn_NetworkServer->Checked ) // this instance operates as SERVER so use the LOCALHOST:
   { SL_AppendString( &pszDest, pszEndstop, "http://127.0.0.1:" );
     SL_AppendString( &pszDest, pszEndstop, AnsiString(Ed_ServerListeningPort->Text).c_str());
     SL_AppendString( &pszDest, pszEndstop, "/" );
     // There are MULTIPLE USERS listed under "Accepted Users and Permissions".
     // To 'simulate' a visit of ANY OF THEM, don't simply pick the FIRST user,
     // but use the freedly editable "User" and "Call" from the edit fields
     // that are otherwise used if this instance would operate as CLIENT :
     SL_AppendString( &pszDest, pszEndstop, "?user=" );
     INET_PlainTextToUrlEncoding(
        AnsiString(Ed_ClientUserName->Text).c_str(), // [in] char * pszPlainTextSource,
        &pszDest, pszEndstop,            // [out] URL encoded result
        "\0"/*pszDelimiters*/ );
     SL_AppendString( &pszDest, pszEndstop, "&call=" );
     INET_PlainTextToUrlEncoding(
        AnsiString(Ed_ClientCallsign->Text).c_str(),
        &pszDest, pszEndstop,            // [out] URL encoded result
        "\0"/*pszDelimiters*/ );
   } // end if < instance operates as SERVER >

  if( sz255URLAndQueryString[0] != '\0' ) // got a URL + Query-string to play with ?
   { fOk = HELP_ShowAnyFileOrURL( sz255URLAndQueryString, ""/* no HTML anchor*/,
                                  FALSE/*fTranslateFilenameToURL*/ );
     if( ! fOk )
      { ShowError( ERROR_CLASS_ERROR, "Failed to open '%s'", sz255URLAndQueryString );
      }
   }
} // end MI_Network_OpenServerPageInBrowserClick()


//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_HelpIndexClick(TObject *Sender)
{ YHF_HELP_ShowHelpContext(HELPID_MAIN_INDEX);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Help_MainMenuClick(TObject *Sender)
{ YHF_HELP_ShowHelpContext(HELPID_MAIN_MENU);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Hotkeysummary1Click(TObject *Sender)
{ YHF_HELP_ShowHelpContext(HELPID_HOTKEYS);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Help_IOConfigClick(TObject *Sender)
{ YHF_HELP_ShowHelpContext(HELPID_IO_CONFIG);
}

void __fastcall TKeyerMainForm::Btn_Help_AdditionalComPortsClick(TObject *Sender)
{ YHF_HELP_ShowHelpContext(HELPID_AUX_COM_PORTS);
}
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Help_KeyerSettingsClick(TObject *Sender)
{ YHF_HELP_ShowHelpContext(HELPID_KEYER_SETTINGS);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Help_AudioSettingsClick(TObject *Sender)
{ YHF_HELP_ShowHelpContext(HELPID_AUDIO_SETTINGS);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::HI_Help_AudioCWDecoderClick(TObject *Sender)
{ YHF_HELP_ShowHelpContext(HELPID_AUDIO_CW_DECODER);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Help_CWSidetoneClick( TObject *Sender)
{ YHF_HELP_ShowHelpContext(HELPID_CW_SIDETONE);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Btn_Help_HamlibServerClick(TObject *Sender)
{ YHF_HELP_ShowHelpContext(HELPID_HAMLIB_SERVER);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Help_NetworkSettingsClick(TObject *Sender)
{ YHF_HELP_ShowHelpContext(HELPID_NETWORK_SETTINGS);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Lab_NetworkFunctionalityClick(TObject *Sender)
{ YHF_HELP_ShowHelpContext(HELPID_NETWORK_FUNCS);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Lab_NetStatusClick(TObject *Sender)
{ YHF_HELP_ShowHelpContext(HELPID_NETWORK_STATUS);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Lab_LocalIPHelpClick(TObject *Sender)
{ YHF_HELP_ShowHelpContext(HELPID_LOCAL_IP_ADDRESS);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Lab_ServerListeningPortClick(TObject *Sender)
{ YHF_HELP_ShowHelpContext(HELPID_LISTENING_PORT);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Lab_AcceptedUsersClick(TObject *Sender)
{ YHF_HELP_ShowHelpContext(HELPID_NETWORK_USERS);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Ed_ServerAcceptUsersDblClick(TObject *Sender)
{ // As the "Hint" (a Borland VCL thing) for this edit field promised:
  // "Double-click this field for help". Here: manual/Remote_CW_Keyer.htm#network_users
  YHF_HELP_ShowHelpContext(HELPID_NETWORK_USERS); // e.g. 'bitmasks for permissions'
  // If you've got a lot of time to spend on improving the GUI (WB didn't),
  // use e.g. a scrolling table with a 'checklist' of permissions per user.
  // For a start, this single-line text edit field ("Accepted Users and Permissions")
  // will do.
}


//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Lab_RemoteURLClick(TObject *Sender)
{ YHF_HELP_ShowHelpContext(HELPID_REMOTE_URL_AND_PORT);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Lab_MyUserNameClick(TObject *Sender)
{ YHF_HELP_ShowHelpContext(HELPID_USER_NAME);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Lab_UserCallsignClick(TObject *Sender)
{ YHF_HELP_ShowHelpContext(HELPID_USER_CALLSIGN);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Help_TimingScopeClick(TObject *Sender)
{ YHF_HELP_ShowHelpContext(HELPID_TIMING_SCOPE);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Help_TransceiverControlClick(TObject *Sender)
{ YHF_HELP_ShowHelpContext(HELPID_TRX_TAB);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Help_HardwareClick(TObject *Sender)
{ YHF_HELP_ShowHelpContext(HELPID_HARDWARE);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Help_DebugTabClick(TObject *Sender)
{ YHF_HELP_ShowHelpContext(HELPID_DEBUG_TAB);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_ClearErrorHistoryClick(TObject *Sender)
{ Ed_ErrorHistory->Clear();

  // Because the poor little TRichEdit had auch a hard time to show the content
  // of the 'traffic log' after e.g. being paused, als clear that:
  RigCtrl_ClearTrafficLog( &MyCwNet.RigControl );

  // Also clear the thread-safe circular FIFO filled by e.g. ShowError() :
  DEBUG_iErrorHistoryTail = DEBUG_iErrorHistoryHead;
}

void __fastcall TKeyerMainForm::MI_CopyHistoryToClipboardClick( TObject *Sender)
{ Ed_ErrorHistory->CopyToClipboard();
}
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Debug_ToggleWordWrap(TObject *Sender)
{
  MyCwNet.cfg.iDiagnosticFlags ^= CWNET_DIAG_FLAGS_WORD_WRAP;
  Ed_ErrorHistory->WordWrap = (MyCwNet.cfg.iDiagnosticFlags & CWNET_DIAG_FLAGS_WORD_WRAP) != 0;
  // "WordWrap" (automatically performed by the RichEdit-thingy) is ON
  // by default, but to see exactly ONE LINE per TCP/IP fragment helped a lot
  // with troubleshooting the embedded web server (HttpServer.c) .
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Debug_LimitHistoryLengthClick(TObject *Sender)
{
  MyCwNet.cfg.iDiagnosticFlags ^= CWNET_DIAG_FLAGS_SHORT_HISTORY;
}


//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Debug_VerboseDiagnosticsClick(TObject *Sender)
{
  if( KeyerGUI_iUpdating==0 )
   { MyCwNet.cfg.iDiagnosticFlags ^= CWNET_DIAG_FLAGS_VERBOSE;
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Debug_ShowKeyingBytestreamClick(TObject *Sender)
{
  if( KeyerGUI_iUpdating==0 )
   { MyCwNet.cfg.iDiagnosticFlags ^= CWNET_DIAG_FLAGS_SHOW_KEYING_BYTESTREAM;
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Debug_ShowMorseDecoderOutputClick(TObject *Sender)
{ // Similar as below, but THIS menu item is in the Rich Text editor's context menu
  //  on the 'Debug' tab itself (where advanced users would expect to find it)
  if( KeyerGUI_iUpdating==0 )
   { MyCwNet.cfg.iDiagnosticFlags ^= CWNET_DIAG_FLAGS_SHOW_DECODER_OUTPUT;
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Debug_ShowNetworkConnectionLogClick(TObject *Sender)
{ if( KeyerGUI_iUpdating==0 )
   { MyCwNet.cfg.iDiagnosticFlags ^= CWNET_DIAG_FLAGS_SHOW_CONN_LOG;
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Debug_ShowNetworkTrafficClick(TObject *Sender)
{
  if( KeyerGUI_iUpdating==0 )
   { MyCwNet.cfg.iDiagnosticFlags ^= CWNET_DIAG_FLAGS_SHOW_NETWORK_TRAFFIC;
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Debug_ShowRigCtrlTrafficClick(TObject *Sender)
{
  RigCtrl_TrafficMonitor.iDisplayOptions ^= RIGCTRL_TMON_DISPLAY_OPTION_ENABLE;
  // '---> included in CalcHashForSettings() / LoadSettings() / SaveSettings()
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Debug_ShowWinkeyerTrafficClick(TObject *Sender)
{
  if( KeyerGUI_iUpdating==0 )
   { AuxCom_iDiagnosticFlags ^= AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC;
     // '---> also included in CalcHashForSettings() / LoadSettings() / SaveSettings()
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Debug_ShowAuxComTrafficClick(TObject *Sender)
{
  if( KeyerGUI_iUpdating==0 )
   { AuxCom_iDiagnosticFlags ^= AUX_COM_DIAG_FLAGS_SHOW_OTHER_TRAFFIC;
     // '---> also included in CalcHashForSettings() / LoadSettings() / SaveSettings()
   }
}


//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Debug_SerialTextTerminalClick(TObject *Sender)
{
  if( KeyerGUI_iUpdating==0 )
   { KeyerGUI_fSerialTerminalActive = !KeyerGUI_fSerialTerminalActive;
   }
}


//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Settings_ShowMorseDecoderOutputClick(TObject *Sender)
{ // Similar as above, but THIS menu item is in the old-school 'main menu',
  // under 'Settings' .  Per default, the display of decoded Morse characters
  // is on; but when also displaying the 'encoded morse code stream' in hex,
  // having a mix of all those funny background-coloured characters in the
  // same editor was confusing.
  if( KeyerGUI_iUpdating==0 )
   { MyCwNet.cfg.iDiagnosticFlags ^= CWNET_DIAG_FLAGS_SHOW_DECODER_OUTPUT;
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Debug_PauseDisplayClick(TObject *Sender)
{  // Pauses / resumes the output from the thread-safe 'ShowError'-FIFO
   // into the non-thread-safe 'RichEdit' control on the 'Debug' tab.
   m_fDebugOutputPaused = !m_fDebugOutputPaused;
   RigCtrl_fTrafficMonitorPausedByUser = m_fDebugOutputPaused;
   if( !m_fDebugOutputPaused )
    { RigCtrl_UnPauseTrafficLogOnAllPorts( &MyCwNet.RigControl ); // -> fTrafficMonitorPausedOnTrigger = FALSE;
    }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_PauseDebugOutputClick(TObject *Sender)
{
   m_fDebugOutputPaused = !m_fDebugOutputPaused;
   RigCtrl_fTrafficMonitorPausedByUser = m_fDebugOutputPaused;
   if( !m_fDebugOutputPaused )
    { RigCtrl_UnPauseTrafficLogOnAllPorts( &MyCwNet.RigControl ); // -> fTrafficMonitorPausedOnTrigger = FALSE;
    }
   // Almost the same as in the RichText editor's "context menu",
   // but THIS ONE ("MI_PauseDebugOutput") is in the main menu under "Settings".
}
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::PM_ErrorHistoryPopup(TObject *Sender)
{
  AnsiString s;

  s = AnsiString(TE("&Pause Display"));
  if( MyCwNet.RigControl.PortInstance[RIGCTRL_PORT_RADIO].fTrafficMonitorPausedOnTrigger )
   { s = s + " (traffic log now paused on trigger)";
   }
  else if( RigCtrl_fTrafficMonitorPausedByUser)
   { s = s + " (traffic log now paused by user)";
   }
  MI_Debug_PauseDisplay->Caption = s;
  MI_Debug_WordWrap->Checked = (MyCwNet.cfg.iDiagnosticFlags & CWNET_DIAG_FLAGS_WORD_WRAP) != 0;
  MI_Debug_LimitHistoryLength->Checked = (MyCwNet.cfg.iDiagnosticFlags & CWNET_DIAG_FLAGS_SHORT_HISTORY) != 0;
  MI_Debug_VerboseDiagnostics->Checked = (MyCwNet.cfg.iDiagnosticFlags & CWNET_DIAG_FLAGS_VERBOSE) != 0;
  MI_Debug_ShowKeyingBytestream->Checked=(MyCwNet.cfg.iDiagnosticFlags & CWNET_DIAG_FLAGS_SHOW_KEYING_BYTESTREAM) != 0;
  MI_Debug_ShowMorseDecoderOutput->Checked=(MyCwNet.cfg.iDiagnosticFlags & CWNET_DIAG_FLAGS_SHOW_DECODER_OUTPUT) != 0;
  MI_Debug_ShowNetworkTraffic->Checked = (MyCwNet.cfg.iDiagnosticFlags & CWNET_DIAG_FLAGS_SHOW_NETWORK_TRAFFIC) != 0;
  MI_Debug_ShowNetworkConnectionLog->Checked = (MyCwNet.cfg.iDiagnosticFlags & CWNET_DIAG_FLAGS_SHOW_CONN_LOG) != 0;
  MI_Debug_ShowRigCtrlTraffic->Checked = (RigCtrl_TrafficMonitor.iDisplayOptions & RIGCTRL_TMON_DISPLAY_OPTION_ENABLE) != 0;
  MI_Debug_ShowWinkeyerTraffic->Checked= (AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC) != 0;
  MI_Debug_ShowAuxComTraffic->Checked= (AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_OTHER_TRAFFIC) != 0;
  MI_Debug_SerialTextTerminal->Checked = KeyerGUI_fSerialTerminalActive;
  MI_Debug_PauseDisplay->Checked = m_fDebugOutputPaused;
}
//---------------------------------------------------------------------------

void __fastcall TKeyerMainForm::MyDrawMenuItem(TObject *Sender,
                   TCanvas *ACanvas, TRect &ARect, bool Selected)
{
  AnsiString s;
  int i;
  ACanvas->Brush->Color = Selected ? g_clMenuSelBackgnd : g_clMenuBackground;
  ACanvas->Font->Color  = g_clMenuForeground;
  ACanvas->FillRect(ARect);
  s = ((TMenuItem*)Sender)->Caption;
  // Even if you don't want it, some voodo priest inserts a '&' before
  // some character to indicate a hotkey. Eliminate that:
  if( (i=s.AnsiPos("&") ) > 0 ) // character indices in an AnsiString are ONE-based. Yucc.
   { s = s.Delete( i,1 );
   }
  ACanvas->TextOut(ARect.Left + 2, ARect.Top, s);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Help_AboutClick(TObject *Sender)
{
  char sz2k[2048], sz255[256];
  char *pszDest = sz2k;
  char *pszEndstop = sz2k + sizeof(sz2k) - 1;
  int iVersionInfoSize;
  int iVersionInfo[4];
  TVSFixedFileInfo *pFileInfo;
  unsigned int iLength;
  char  cBuffer[4000];
  DWORD dwHandle;

  // Obscure stuff to retrieve the version-info that Borland's linker has placed
  // IN THE *.EXE :
  memset( iVersionInfo, 0, sizeof(iVersionInfo) );
  iVersionInfoSize = GetFileVersionInfoSize( AnsiString(ParamStr(0)).c_str(), &dwHandle);
  // |      ,--------------------------------|__________|
  // |      '--> Important when compiling with e.g. C++Builder V12 "Athens",
  // |           where regardless of the mapping of _TCHAR to wchar_t or char,
  // |           many of these VCL functions don't return an AnsiString anymore
  // |           but a UnicodeString. The "c_str()" method of the latter
  // |           always returns a pointer to a 16-bit "wide character" string,
  // |           not the nice old UTF-8 compatible 8-bit "C string" that
  // |           every microcontroller developer loves so dearly ;o)
  // |       Without this conversion to AnsiString before the c_str(),
  // |       Embarcadero C++Builder V12 "Athens" complained:
  // |       "no matching function for call to 'GetFileVersionInfoA' ",
  // |       because Microsoft's ..A() functions (A for "ANSI") expects an
  // |       'LPCSTR' as argument 'lptstrFilename', which is -tada- a
  // |       'const TCHAR *', and in our case 'TCHAR' is 8-bit char.
  // |       If you try to fix such errors by brainless cast operators,
  // |       the problem will reappear later at RUNTIME (which is even worse).
  // '--> "if size is zero, then there is no version info in the exe".
  //      (be prepared for this, and similar nice surprise..)
  // WB, 2023-10 : Got here with a suprising iVersionInfoSize of 1524 bytes !
  if( (iVersionInfoSize>0) && (iVersionInfoSize<sizeof(cBuffer)) )
   {
    try  // another example of the monstrosities of C++ and Win32 :
     { GetFileVersionInfo( AnsiString(ParamStr(0)).c_str(), 0, iVersionInfoSize, cBuffer);
       VerQueryValue( cBuffer, "\\"/*lpSubBlock:"root block"*/, (void**)&pFileInfo, &iLength);
       // WB, 2023-10 : Got here with iLength = 52, whatever that means.
       iVersionInfo[0] = pFileInfo->dwFileVersionMS >> 16;
       iVersionInfo[1] = pFileInfo->dwFileVersionMS & 0xffff;
       iVersionInfo[2] = pFileInfo->dwFileVersionLS >> 16;
       iVersionInfo[3] = pFileInfo->dwFileVersionLS & 0xffff;
       // WB, 2023-10 : Got here with ....
       //  iVersionInfo[0] = Borland's "Hauptversion", (major release ?)
       //  iVersionInfo[1] = Borland's "Nebenversion", (minor release ?)
       //  iVersionInfo[2] = Borland's "Ausgabe",      (revision number)
       //  iVersionInfo[3] = Borland's "Compilierung"  ("build number"?)
       // When examining the properties of the *.exe file via file manager,
       // the four numbers in iVersionInfo[0..3] were what Microsoft calls
       //  "File version", not "Product version" !
       // Remember to keep these version numbers in sync, in the project file
       // (*.bpr?) and in the installer script (Inno_Setup_Script_for_RCW.iss) !

     }
    catch (...)
     { // Windows API function call failed (why am I not suprised ?) -> 
     }
   }

  SL_AppendPrintf( &pszDest, pszEndstop, "Remote CW Keyer V%d.%d.%d.%d, compiled %s .\n",
    iVersionInfo[0], iVersionInfo[1], iVersionInfo[2], iVersionInfo[3],  __DATE__ );
  // Note: The version info displayed above should be copied into the
  //       INNO SETUP SCRIPT, in the following line:
  //       > #define MyAppVersion "1.0.2.1"
  SL_AppendPrintf( &pszDest, pszEndstop, "Please check for updates at\n"
    "  www.qsl.net/dl4yhf/Remote_CW_Keyer/Remote_CW_Keyer.htm !\n" );
  memset( sz255, 0, sizeof(sz255) );  // make sure there's a TRAILING ZERO !
  ::GetModuleFileName(0, sz255, 255); // <- this is WINDOWS, not BORLAND/VCL
  SL_AppendPrintf( &pszDest, pszEndstop, "Executable file : %s\n", sz255 );
  SL_AppendPrintf( &pszDest, pszEndstop, "Data file path  : %s\n", g_sz255PathToDataFiles );
  SL_AppendPrintf( &pszDest, pszEndstop, "Command Line: %s\n", g_sz255CommandLine );
  SL_AppendPrintf( &pszDest, pszEndstop, "Last browser commmand:\n %s\n", YHF_HELP_sz1023LastBrowserCommandLine );
  ::MessageBox( Handle, sz2k, "Remote CW Keyer", MB_OK );
     // '--> Did you know that the good old Windows "MessageBox"
     //      can be copied AS TEXT into the Windows clipboard ?
     //      Try it.. press CTRL-C to copy while the message box is open,
     //          then press CTRL-V in the plain old 'notepad' or similar.
     // Result (example) :
     // > ---------------------------
     // > Remote CW Keyer
     // > ---------------------------
     // > Remote CW Keyer V1.2.2.3, compiled Sep 14 2025 .
     // > Please check for updates at
     // >   www.qsl.net/dl4yhf/Remote_CW_Keyer/Remote_CW_Keyer.htm !
     // > Executable file : C:\CwKeyerTest\Remote_CW_Keyer.exe
     // > Data file path  : C:\CwKeyerTest\                           .
     // > Command Line: "C:\CwKeyerTest\Remote_CW_Keyer.exe"
     // > Last browser commmand:
     // >
     // >
     // > ---------------------------
     // > OK
     // > ---------------------------


} // end TKeyerMainForm::MI_Help_AboutClick()

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::FormMouseWheel(TObject *Sender,
      TShiftState Shift, int WheelDelta, TPoint &MousePos, bool &Handled)
{ // Notes:
  //  - For a single step of the mouse wheel, this event seemed to be called
  //    more than once !  Setting "Handled=TRUE" cured this problem.
  //  - This event handler must be aware of "who" has the focus, because
  //    some visible controls have (and need!) their own mouse wheel handler !
  //  - When, for example, the TEdit for the VFO frequency had the focus,
  //    Borland's "FormMouseWheel()" fired with 'WheelDelta' = 120 (or -120).
  //    Thus ignore the 'WheelDelta' value (only care for the sign)
  //    
  WORD wKey;
  TEdit *pEd = NULL;
  BOOL fHandled=FALSE;
  TPoint pt = Img_Spectrum->ScreenToClient( MousePos );
  int   iSpecDispArea, iSplitScaleIndex, iChannelOfAnalyser;
  float fltXrel, fltYrel;
  double fc, d1, d2, dblBasebandFmin, dblBasebandFmax, dblNewBasebandFmin;


  // Identify the mouse screen area... if it's over the frequency scale,
  // it may be used to zoom or scroll there (similar as in DL4YHF's modified WSQ2 waterfall)
  if( pt.x>=0 && pt.y>=0 && pt.x<Img_Spectrum->Width && pt.y<Img_Spectrum->Height )
   { // Inside the spectrum/spectrogram/frequency scale area, but where exactly ?
     iSpecDispArea = IdentifyClientCoord(pt.x, pt.y, &fltXrel, &fltYrel );
#   if(0)
     switch( iSpecDispArea )
      { case SCREEN_AREA_FREQUENCY_SCALE :
           // Zoom in/out, with the center "under" the mouse pointer :
              fc = FreqScale_MousePosToBasebandFreq( pt.x, pt.y, &iSplitScaleIndex, NULL );
              iChannelOfAnalyser = CLI_SplitScaleIndexToAnalyserChannel( iSplitScaleIndex );
              SpecDisp_GetDisplayableBasebandFreqRange(CFG_SD_MAIN_ANALYSER, iSplitScaleIndex,
                     &dblBasebandFmin, &dblBasebandFmax, SD_GET_FREQ_RANGE_WITHOUT_REPROGRAMMING_LO );
              d1 = fc - pDisplaySettings->Chn[iSplitScaleIndex].dblBasebandFreqMin;
              d2 = pDisplaySettings->Chn[iSplitScaleIndex].dblBasebandFreqMax - fc;
              if( WheelDelta > 0 )
               { d1 *= 0.95;    // zoom IN
                 d2 *= 0.95;
               }
              else if( WheelDelta < 0 )
               { d1 /= 0.95;    // zoom OUT
                 d2 /= 0.95;
               }
              ChangeDisplayedBasebandFreqRange( iSplitScaleIndex, fc-d1, fc+d2, LIMIT_F_RANGE, FALSE/*fInitialCall*/ );
              fHandled = TRUE;
            }
           break;
        default:
           break;
      }
#   endif
   } // end if < inside the spectrogram/frequency scale/spectrum area >


  if( !fHandled )
   { // FALSE = default "new style", i.e. react on input immediately,
     //        and allow increment/decrement of digits via cursor or mousewheel:
     if( Ed_VFO->Focused() )     // VFO frequency (for SDR or external hardware / ExtIO-DLL)
      {  pEd = Ed_VFO;
      }
     if( pEd != NULL )
      { fHandled = TRUE;
        if( WheelDelta > 0 )
         { wKey = VK_UP;
           pEd->OnKeyDown( pEd, wKey, Shift ); // -> Ed_VFOKeyDown()
           Handled = TRUE;
         }
        if( WheelDelta < 0 )
         { wKey = VK_DOWN;
           pEd->OnKeyDown( pEd, wKey, Shift ); // -> Ed_VFOKeyDown()
           Handled = TRUE;
         }
      } // end if < VFO or similar "frequency" edit field has the focus >
   } // end if( ! UConfig.fPlainFreqInputFields )
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_TRX_KeepRunningClick(TObject *Sender)
{
  CwKeyer_Config.iTRXOptions ^= KEYER_TRX_OPT_KEEP_RUNNING;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Settings_TRX_OptionsClick(TObject *Sender)
{ // Kludge to open the CONTEXT MENU for the "TRX" tab also from the MAIN menu.
  // (didn't find a way in the VCL to use a "TPopupMenu" as submenu
  //  in a "TMainMenu", as a REAL sub-menu for MainMenu1.MI_Settings_TRX_Options
  //  in Borland C++Builder 6's "form designer" / "object inspector").
  SwitchMainTab( KEYER_GUI_MAIN_TAB_TRX ); // here: called from the main menu, "Settings" -> "TRX Options"
  PM_Spectrum->Popup(Mouse->CursorPos.x, Mouse->CursorPos.y);
}

//---------------------------------------------------------------------------

void __fastcall TKeyerMainForm::MI_NetworkLatency_AutoDetectClick(TObject *Sender)
{ MyCwNet.cfg.iNetworkLatency_ms = 0; // fixed network latency in milliseconds (default: 0 = auto-detect)
}
//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_NetworkLatency_100msClick(TObject *Sender)
{ MyCwNet.cfg.iNetworkLatency_ms = 100;
}
//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_NetworkLatency_250msClick(TObject *Sender)
{ MyCwNet.cfg.iNetworkLatency_ms = 250;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_NetworkLatency_500msClick(TObject *Sender)
{ MyCwNet.cfg.iNetworkLatency_ms = 500;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_NetworkLatency_1000msClick(TObject *Sender)
{ MyCwNet.cfg.iNetworkLatency_ms = 1000;
}

//---------------------------------------------------------------------------
int MouseButtonToGuiKeyFlags(TMouseButton Button)
  // Abstraction of Borland's "TMouseButton"-thingy,
  //   Unlike "TMouseButton" (which uses ZERO as the code for LEFT mouse button),
  //   our bit-combination of flags like
  //      GUI_KEY_FLAGS_LEFT_MOUSE_BUTTON, GUI_KEY_FLAGS_RIGHT_MOUSE_BUTTON,
  //      GUI_KEY_FLAGS_SHIFT, GUI_KEY_FLAGS_CONTROL, ..
  //   can even encode "NO mouse button pressed" (guess what the value is .. ZERO).
  //   Returns a simple integer, no fancy "set" or whatever.
{ int iKeyFlags = 0;

  // Borland's "TMouseButton" doesn't seem to be a cryptic "set" but an enum.
  // Obviously they don't support detection of BOTH (or even more) buttons
  // pressed simultaneously :
  if( Button==mbLeft )
   { iKeyFlags |= GUI_KEY_FLAGS_LEFT_MOUSE_BUTTON;
   }
  if( Button==mbRight )
   { iKeyFlags |= GUI_KEY_FLAGS_RIGHT_MOUSE_BUTTON;
   }
  return iKeyFlags;
} // end MouseButtonToGuiKeyFlags()

//---------------------------------------------------------------------------
int ShiftStateToGuiKeyFlags( TShiftState Shift)
  // Simplistic abstraction of Borland-specific "TShiftState".
  //     Returns a simple integer, no fancy "set" or whatever.
{ int iKeyFlags = 0;

  // Borland's "TShiftState" is neither an enum nor a simple combination of bits
  //  but a "set of flags". We convert it into simple bitflags in an integer
  //  to get away from the counter-intuitive syntax..
  //  > A 'Set' is a C++ template class designed to emulate the Delphi Set type.
  //  No, thanks. Why use a VCL-specific C++ gizmo when a simple 'int' will do ?
  if( Shift.Contains(ssShift) )
   { iKeyFlags |= GUI_KEY_FLAGS_SHIFT;
   }
  if( Shift.Contains(ssCtrl) )
   { iKeyFlags |= GUI_KEY_FLAGS_CONTROL;
   }
  return iKeyFlags;
} // end ShiftStateToGuiKeyFlags()


//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Img_SpectrumMouseDown(TObject *Sender,
      TMouseButton Button, TShiftState Shift, int X, int Y)
  // Called from Borland's VCL when just pressing the mouse
  // on the specified position ("client area coordinate") in the SPECTRUM.
  // The same handler is used for the "OnMouseDown", "OnMouseMove", "OnMouseUp"
  // events in multiple window parts on the "TRX" tab (at least spectrum,
  // spectrogram, and frequency scale) - see SpecDisp.cpp : SpecDisp_HandleMouseEvent() !
{
  SpecDisp_HandleMouseEvent( &g_SpecDispControl, &MyCwNet.RigControl, GUI_EVENT_MOUSE_DOWN, GUI_CONTROL_SPECTRUM,
      MouseButtonToGuiKeyFlags(Button) | ShiftStateToGuiKeyFlags(Shift), X, Y );
  // Save this for Borland's "OnMouseMove()" handler (which doesn't pass in a "TMouseButton"):
  m_iMouseButtonAsGuiKeyFlags = MouseButtonToGuiKeyFlags( Button );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Img_WaterfallMouseDown(TObject *Sender,
      TMouseButton Button, TShiftState Shift, int X, int Y)
{ // Similar as Img_SpectrumMouseDown(), but this is for the spectrogram aka waterfall,
  // which not only has a FREQUENCY AXIS but also a TIME AXIS:
  SpecDisp_HandleMouseEvent( &g_SpecDispControl, &MyCwNet.RigControl, GUI_EVENT_MOUSE_DOWN, GUI_CONTROL_SPECTROGRAM,
      MouseButtonToGuiKeyFlags(Button) | ShiftStateToGuiKeyFlags(Shift), X, Y );
  // Save this for Borland's "OnMouseMove()" handler (which doesn't pass in a "TMouseButton"):
  m_iMouseButtonAsGuiKeyFlags = MouseButtonToGuiKeyFlags( Button );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Img_SpectrumMouseMove(TObject *Sender, TShiftState Shift, int X, int Y)
{
  SpecDisp_HandleMouseEvent( &g_SpecDispControl, &MyCwNet.RigControl, GUI_EVENT_MOUSE_MOVE, GUI_CONTROL_SPECTRUM,
      m_iMouseButtonAsGuiKeyFlags | ShiftStateToGuiKeyFlags(Shift), X, Y );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Img_WaterfallMouseMove(TObject *Sender, TShiftState Shift, int X, int Y)
{
  SpecDisp_HandleMouseEvent( &g_SpecDispControl, &MyCwNet.RigControl, GUI_EVENT_MOUSE_MOVE, GUI_CONTROL_SPECTROGRAM,
      m_iMouseButtonAsGuiKeyFlags | ShiftStateToGuiKeyFlags(Shift), X, Y );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Img_SpectrumMouseUp(TObject *Sender,
      TMouseButton Button, TShiftState Shift, int X, int Y)
{
  BOOL fHandledEvent = SpecDisp_HandleMouseEvent( &g_SpecDispControl, // here: in the SPECTRUM
       &MyCwNet.RigControl, GUI_EVENT_MOUSE_UP, GUI_CONTROL_SPECTRUM,
       MouseButtonToGuiKeyFlags(Button) | ShiftStateToGuiKeyFlags(Shift), X, Y );
  m_iMouseButtonAsGuiKeyFlags = 0;     //

  if( ! fHandledEvent )
   { PM_Spectrum->Popup(Mouse->CursorPos.x, Mouse->CursorPos.y); // [in] g_SpecDispControl.dblClickedVfoFreq_Hz, etc (?)
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Img_WaterfallMouseUp(TObject *Sender,
      TMouseButton Button, TShiftState Shift, int X, int Y)
{ // Similar as Img_SpectrumMouseUp(), but for the WATERFALL aka Spectrogram,
  // and with subtle differences because the spectrogram also has a TIME AXIS.
  BOOL fHandledEvent = SpecDisp_HandleMouseEvent( &g_SpecDispControl, // here: in the SPECTROGRAM
       &MyCwNet.RigControl, GUI_EVENT_MOUSE_UP, GUI_CONTROL_SPECTROGRAM,
       MouseButtonToGuiKeyFlags(Button) | ShiftStateToGuiKeyFlags(Shift), X, Y );
  m_iMouseButtonAsGuiKeyFlags = 0;     //

  if( ! fHandledEvent )
   { PM_Spectrum->Popup(Mouse->CursorPos.x, Mouse->CursorPos.y);
     //            '--> [in] g_SpecDispControl.dblClickedVfoFreq_Hz, etc (?)
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Img_FreqScaleMouseDown(TObject *Sender,
      TMouseButton Button, TShiftState Shift, int X, int Y)
{ // Again, similar as above.. but here, especially for the FREQUENCY SCALE
  SpecDisp_HandleMouseEvent( &g_SpecDispControl, &MyCwNet.RigControl, GUI_EVENT_MOUSE_DOWN, GUI_CONTROL_FREQ_SCALE,
      MouseButtonToGuiKeyFlags(Button) | ShiftStateToGuiKeyFlags(Shift), X, Y );
  m_iMouseButtonAsGuiKeyFlags = MouseButtonToGuiKeyFlags( Button );
}
//---------------------------------------------------------------------------

void __fastcall TKeyerMainForm::Img_FreqScaleMouseMove(TObject *Sender,
      TShiftState Shift, int X, int Y)
{ // Again, similar as above.. but here, especially for the FREQUENCY SCALE
  SpecDisp_HandleMouseEvent( &g_SpecDispControl, &MyCwNet.RigControl, GUI_EVENT_MOUSE_MOVE, GUI_CONTROL_FREQ_SCALE,
      m_iMouseButtonAsGuiKeyFlags | ShiftStateToGuiKeyFlags(Shift), X, Y );
}
//---------------------------------------------------------------------------

void __fastcall TKeyerMainForm::Img_FreqScaleMouseUp(TObject *Sender,
      TMouseButton Button, TShiftState Shift, int X, int Y)
{ // Again, similar as above.. but here, especially for the FREQUENCY SCALE
  BOOL fHandledEvent = SpecDisp_HandleMouseEvent( &g_SpecDispControl, // here: in the FREQUENCY SCALE
       &MyCwNet.RigControl, GUI_EVENT_MOUSE_UP, GUI_CONTROL_FREQ_SCALE,
       MouseButtonToGuiKeyFlags(Button) | ShiftStateToGuiKeyFlags(Shift), X, Y );
  m_iMouseButtonAsGuiKeyFlags = 0;
  if( ! fHandledEvent )
   { PM_Spectrum->Popup(Mouse->CursorPos.x, Mouse->CursorPos.y);
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Img_TRX_BottomMouseDown(TObject *Sender,
      TMouseButton Button, TShiftState Shift, int X, int Y)
{ // And again, similar as above.. but not the same
  m_iMouseButtonAsGuiKeyFlags = MouseButtonToGuiKeyFlags( Button );
  SpecDisp_HandleMouseEvent( &g_SpecDispControl, &MyCwNet.RigControl, // here: in the FREQUENCY INFO below the frequency scale
           GUI_EVENT_MOUSE_DOWN, GUI_CONTROL_FREQ_INFO,
           MouseButtonToGuiKeyFlags(Button) | ShiftStateToGuiKeyFlags(Shift), X, Y );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Img_TRX_BottomMouseMove(TObject *Sender,
      TShiftState Shift, int X, int Y)
{ SpecDisp_HandleMouseEvent( &g_SpecDispControl, &MyCwNet.RigControl,
      GUI_EVENT_MOUSE_MOVE, GUI_CONTROL_FREQ_INFO,
      m_iMouseButtonAsGuiKeyFlags | ShiftStateToGuiKeyFlags(Shift), X, Y );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Img_TRX_BottomMouseUp(TObject *Sender,
      TMouseButton Button, TShiftState Shift, int X, int Y)
{ BOOL fHandledEvent;
  T_SpecDispMarker *pClickableMarker;
  T_FreqListEntry  *pFLE;
  char sz2k[2048];
  char *pszDest = sz2k;
  char *pszEndstop = sz2k + sizeof(sz2k) - 1;

  m_iMouseButtonAsGuiKeyFlags = MouseButtonToGuiKeyFlags(Button) | ShiftStateToGuiKeyFlags(Shift);
  fHandledEvent = SpecDisp_HandleMouseEvent( &g_SpecDispControl, // here: released mouse button while over the FREQUENCY INFO below the frequency scale
       &MyCwNet.RigControl, GUI_EVENT_MOUSE_UP, GUI_CONTROL_FREQ_INFO,
       m_iMouseButtonAsGuiKeyFlags, X, Y );
  if( ! fHandledEvent )
   { // Maybe it was a mouse-click on one of the hundreds of "clickable frequency labels" ?
     if( m_iMouseButtonAsGuiKeyFlags == GUI_KEY_FLAGS_LEFT_MOUSE_BUTTON ) // only with the LEFT mouse button (the RIGHT button shall open the context menu) ..
      { pClickableMarker = SpecDisp_CheckCoordForClickableMarker( &g_SpecDispControl,
                             X/*iClientX*/, Y/*iClientY*/, GUI_CONTROL_FREQ_INFO );
        if( pClickableMarker != NULL )
         { fHandledEvent = TRUE;  // don't open the context menu further below
           if( (pFLE=(T_FreqListEntry*)pClickableMarker->pvObject) != NULL )
            { // See type definition in C:\cbproj\Remote_CW_Keyer\FreqList.h ...
              FreqList_EntryToMultiLineText( pFLE, pszDest, pszEndstop );
              ::MessageBox( Handle, sz2k, "Frequency / Station info", MB_OK );
              // Would be great to have a message box from where the user can
              // select and copy text (with something like Borland/VCL's "Static Text"
              // instead of a simple "Label" element). But:
              // SUPRISE ! When pressing CTRL-C while MessageBox()
              // has the focus, and then pressing CTRL-V in e.g. "Notepad",
              // generated a PLAIN TEXT like the following:
              //  ---------------------------
              //  Frequency / Station info
              //  ---------------------------
              //  7.039 MHz : "OK0EPB Pendulum"
              //  ITU country: CZE
              //  Program language:   -TS
              //  Target area: Eu
              //  Remarks   :    p
              //  Database item index: 3414
              //
              //  ---------------------------
              //  OK
              //  --------------------------
            } // end if < "clickable marker" with a pointer to a T_FreqListEntry > ?
         }   // end if < found a "clickable marker" for the mouse coordinate > ?
      }     // end if < clicked with the LEFT mouse button > ?
   }       // end if < mouse event NOT handled by SpecDisp_HandleMouseEvent() > ?
  m_iMouseButtonAsGuiKeyFlags = 0;
  if( ! fHandledEvent )
   { PM_Spectrum->Popup(Mouse->CursorPos.x, Mouse->CursorPos.y);
   }
}

//---------------------------------------------------------------------------
AnsiString GetMenuItemTextForBottomPanelMode(void)
{
  switch( g_SpecDispControl.iBottomPanelMode )
   { case SPEC_DISP_PANEL_MODE_OFF :
        return AnsiString(TE("&Bottom panel (below the frequency scale): Off") );
     case SPEC_DISP_PANEL_MODE_MULTI_FUNCTION_METER:
        return AnsiString(TE( "&Bottom panel: Show Multi-Function Meters" ) );
     case SPEC_DISP_PANEL_MODE_FREQUENCY_INFO:
        return AnsiString(TE( "&Bottom panel: Show Band- and Frequency Info") );
     case SPEC_DISP_PANEL_MODE_CHATBOX:
        return AnsiString(TE( "&Bottom panel: Show 'chat box' for messages from the sysop, etc") );
     case SPEC_DISP_PANEL_MODE_AUTOMATIC:
        return AnsiString(TE( "&Bottom panel: Automatic (show 'meters' or freq-info)") );
     default:
        return AnsiString(TE("&Bottom panel: mode=")) + IntToStr(g_SpecDispControl.iBottomPanelMode) + " ?? [bug]" ;
   }
}


//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::PM_SpectrumPopup(TObject *Sender)
{
  if( g_SpecDispControl.dblClickedVfoFreq_Hz <= 0.0 )
   {  // Provide a valid frequency "to do something with" in the following menu items
      g_SpecDispControl.dblClickedVfoFreq_Hz = MyCwNet.RigControl.dblVfoFrequency;
   }
  MI_Spectrum_SetVFO->Caption = AnsiString( TE("Set &VFO frequency to") )
   + FormatFloat( " #0.0000#", g_SpecDispControl.dblClickedVfoFreq_Hz*1e-6) + " MHz";
  MI_Spectrum_RefLevel->Caption = AnsiString( TE("Spectrum &Reference Level") )
    + " : " + FormatFloat( "#0.0", MyCwNet.RigControl.dblScopeRefLevel_dB) + " dB";
  MI_Spectrum_Span->Caption = "Spectrum Span in \"Center\" modes : "
    + FormatFloat( "#0.0", MyCwNet.RigControl.dblScopeSpan_Hz*1e-3) + " kHz";
  MI_SpectrumSpan_2500->Checked   = (MyCwNet.RigControl.dblScopeSpan_Hz == 2.5e3);
  MI_SpectrumSpan_5kHz->Checked   = (MyCwNet.RigControl.dblScopeSpan_Hz == 5e3);
  MI_SpectrumSpan_10kHz->Checked  = (MyCwNet.RigControl.dblScopeSpan_Hz == 10e3);
  MI_SpectrumSpan_25kHz->Checked  = (MyCwNet.RigControl.dblScopeSpan_Hz == 25e3);
  MI_SpectrumSpan_50kHz->Checked  = (MyCwNet.RigControl.dblScopeSpan_Hz == 50e3);
  MI_SpectrumSpan_100kHz->Checked = (MyCwNet.RigControl.dblScopeSpan_Hz == 100e3);
  MI_SpectrumSpan_250kHz->Checked = (MyCwNet.RigControl.dblScopeSpan_Hz == 250e3);
  MI_SpectrumSpan_500kHz->Checked = (MyCwNet.RigControl.dblScopeSpan_Hz == 500e3);
  MI_WFColourPalette_Sunrise->Checked=(g_SpecDispControl.iWFColorPalette == WF_PALETTE_SUNRISE);
  MI_WFColourPalette_Linrad->Checked= (g_SpecDispControl.iWFColorPalette == WF_PALETTE_LINRAD);
  MI_WFColourPalette_Red->Checked   = (g_SpecDispControl.iWFColorPalette == WF_PALETTE_RED);
  MI_WFColourPalette_Green->Checked = (g_SpecDispControl.iWFColorPalette == WF_PALETTE_GREEN);
  MI_WFColourPalette_Blue->Checked  = (g_SpecDispControl.iWFColorPalette == WF_PALETTE_BLUE);
  MI_TRX_Tab_ShowTimeMarkers->Checked = ( g_SpecDispControl.iDisplayOptions & SPEC_DISP_OPTIONS_TIME_MARKERS) != 0;
  MI_TRX_Tab_ShowCWDecoderSpectrum->Checked = ( g_SpecDispControl.iDisplayOptions & SPEC_DISP_OPTIONS_CW_DECODER_SPECTRUM) != 0;
  MI_TRX_Tab_KeepRunning->Checked = (CwKeyer_Config.iTRXOptions & KEYER_TRX_OPT_KEEP_RUNNING) != 0;

  MI_TRX_Tab_BPMode_Meters->Checked = ( g_SpecDispControl.iBottomPanelMode==SPEC_DISP_PANEL_MODE_MULTI_FUNCTION_METER);
  MI_TRX_Tab_BPMode_FreqInfo->Checked=( g_SpecDispControl.iBottomPanelMode==SPEC_DISP_PANEL_MODE_FREQUENCY_INFO);
  MI_TRX_Tab_BPMode_ChatBox->Checked= ( g_SpecDispControl.iBottomPanelMode==SPEC_DISP_PANEL_MODE_CHATBOX );
  MI_TRX_Tab_BPMode_Auto->Checked   = ( g_SpecDispControl.iBottomPanelMode==SPEC_DISP_PANEL_MODE_AUTOMATIC);

  MI_TRX_Tab_BottomPanelMode->Caption = GetMenuItemTextForBottomPanelMode();
} // end TKeyerMainForm::PM_SpectrumPopup()


//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::PM_ChatBoxPopup(TObject *Sender)
{
  MI_ChatBox_BottomPanelMode->Caption = GetMenuItemTextForBottomPanelMode();
  MI_ChatBox_BPMode_Meters->Checked = ( g_SpecDispControl.iBottomPanelMode==SPEC_DISP_PANEL_MODE_MULTI_FUNCTION_METER);
  MI_ChatBox_BPMode_FreqInfo->Checked=( g_SpecDispControl.iBottomPanelMode==SPEC_DISP_PANEL_MODE_FREQUENCY_INFO);
  MI_ChatBox_BPMode_ChatBox->Checked= ( g_SpecDispControl.iBottomPanelMode==SPEC_DISP_PANEL_MODE_CHATBOX );
  MI_ChatBox_BPMode_Auto->Checked   = ( g_SpecDispControl.iBottomPanelMode==SPEC_DISP_PANEL_MODE_AUTOMATIC);
} // end TKeyerMainForm::PM_ChatBoxPopup()


//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_TRX_Tab_BPMode_OFFClick(TObject *Sender)
{ g_SpecDispControl.iBottomPanelMode = SPEC_DISP_PANEL_MODE_OFF;
  m_fAdaptImageSizesOnTRXTab = TRUE; // call AdaptImageSizesOnTRXTab() a.s.a.p.
}


//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_TRX_Tab_BPMode_FreqInfoClick(TObject *Sender)
{ g_SpecDispControl.iBottomPanelMode = SPEC_DISP_PANEL_MODE_FREQUENCY_INFO;
  m_fAdaptImageSizesOnTRXTab = TRUE; // call AdaptImageSizesOnTRXTab() a.s.a.p.
}

void __fastcall TKeyerMainForm::MI_TRX_Tab_BPMode_ChatBoxClick(TObject *Sender)
{ g_SpecDispControl.iBottomPanelMode = SPEC_DISP_PANEL_MODE_CHATBOX;
  m_fAdaptImageSizesOnTRXTab = TRUE; // call AdaptImageSizesOnTRXTab() a.s.a.p.
}
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_TRX_Tab_BPMode_MetersClick(TObject *Sender)
{ g_SpecDispControl.iBottomPanelMode = SPEC_DISP_PANEL_MODE_MULTI_FUNCTION_METER;
  m_fAdaptImageSizesOnTRXTab = TRUE; // call AdaptImageSizesOnTRXTab() a.s.a.p.
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_TRX_Tab_BPMode_AutoClick(TObject *Sender)
{ g_SpecDispControl.iBottomPanelMode = SPEC_DISP_PANEL_MODE_AUTOMATIC;
  m_fAdaptImageSizesOnTRXTab = TRUE; // call AdaptImageSizesOnTRXTab() a.s.a.p.
}


//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_TRX_Tab_ShowTimeMarkersClick(TObject *Sender)
{ g_SpecDispControl.iDisplayOptions ^= SPEC_DISP_OPTIONS_TIME_MARKERS;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_TRX_Tab_ShowCWDecoderSpectrumClick(TObject *Sender)
{ g_SpecDispControl.iDisplayOptions ^= SPEC_DISP_OPTIONS_CW_DECODER_SPECTRUM;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_TRX_Tab_KeepRunningClick(TObject *Sender)
{ CwKeyer_Config.iTRXOptions ^= KEYER_TRX_OPT_KEEP_RUNNING;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Spectrum_SetVFOClick(TObject *Sender)
{
  RigCtrl_SetVFOFrequency( &MyCwNet.RigControl, g_SpecDispControl.dblClickedVfoFreq_Hz );
  KeyerGUI_UpdateVFODisplay( Ed_VFO, g_SpecDispControl.dblClickedVfoFreq_Hz );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_WFall_SetVFOClick(TObject *Sender)
{ // Simular as MI_Spectrum_SetVFOClick(), but here for the WATERFALL,
  // possibly with subtle differences one fine day..
  RigCtrl_SetVFOFrequency( &MyCwNet.RigControl, g_SpecDispControl.dblClickedVfoFreq_Hz );
  KeyerGUI_UpdateVFODisplay( Ed_VFO, g_SpecDispControl.dblClickedVfoFreq_Hz );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Spectrum_SetPermanentMarkerClick(TObject *Sender)
{
  if( YHF_RunStringEditDialog(
       Handle,  // [in] handle to the window that will be blocked by the dialog
       (char*)TE("Define permanent frequency marker"), // [in] char *pszTitle,
       NULL,                     // [in] char *pszLabel1,
       NULL,                     // [in] char *pszLabel2,
       m_sz80LastMarkerText,     // [in] char *pszDefaultValue,
       YHF_EDIT_NORMAL,          // [in] int iEditOptions,
       (char*)m_sz80LastMarkerText, // [out] char *pszDestination,
       80,                       // [in] int iMaxLength,
       HELPID_FREQUENCY_MARKER)  // [in] int  iHelpID
     == YHF_DLG_BTN_OK ) // operator clicked "OK" or confirmed with "Enter" ->
   {
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_CenterDisplayOnClickedFrequencyClick(TObject *Sender)
{
  RigCtrl_SetScopeCenterFrequency( &MyCwNet.RigControl, g_SpecDispControl.dblClickedVfoFreq_Hz );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_SpectrumRefLevel_EditClick(TObject *Sender)
{
  double dblValue = MyCwNet.RigControl.dblScopeRefLevel_dB;
  char   sz80Label1[84], sz80Label2[84], *pszDest, *pszEndstop;
  pszDest    = sz80Label1;
  pszEndstop = sz80Label1+80;
  SL_AppendPrintf( &pszDest, pszEndstop, "%s: -20 .. +20 dB", TE("Valid range") );
  pszDest    = sz80Label2;
  pszEndstop = sz80Label2+80;
  SL_AppendPrintf( &pszDest, pszEndstop, "%s: %.1lf dB", TE("Current value"), dblValue );
  if( YHF_RunFloatInputDialog(
       Handle,  // [in] handle to the window that will be blocked by the dialog
       (char*)TE("Set Spectrum Reference Level"), // [in] char *pszTitle,
       sz80Label1, sz80Label2,   // [in] labels (info strings) above the edit field
       "%.1lf",                  // [in] const char *pszFormatString,
       &dblValue,                // [in,out] double *pdblValue
       YHF_EDIT_NORMAL,          // [in] int iEditOptions,
       HELPID_SPECTRUM_REF_LEVEL)// [in] int  iHelpID
     == YHF_DLG_BTN_OK ) // operator clicked "OK" or confirmed with "Enter" ->
   { // Send the new 'spectrum reference level' to the radio (*)
     RigCtrl_SetScopeRefLevel_dB( &MyCwNet.RigControl, dblValue );
     // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
     // The new setting in MyCwNet.RigControl [here: RigControl.dblScopeRefLevel_dB]
     // will be compared with the 'last values' sent to a remote server/clients,
     // and if the value was MODIFIED, CwNet.c : CwNet_OnPoll() will send the
     // new setting also via the Remote CW Keyer's own TCP/IP connection.
     // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_SpectrumRefLevel_m20Click(TObject *Sender)
{ RigCtrl_SetScopeRefLevel_dB( &MyCwNet.RigControl, -20.0/*dB*/ );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_SpectrumRefLevel_m10Click(TObject *Sender)
{ RigCtrl_SetScopeRefLevel_dB( &MyCwNet.RigControl, -10.0/*dB*/ );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_SpectrumRefLevel_0Click(TObject *Sender)
{ RigCtrl_SetScopeRefLevel_dB( &MyCwNet.RigControl, 0.0/*dB*/ );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_SpectrumRefLevel_10Click(TObject *Sender)
{ RigCtrl_SetScopeRefLevel_dB( &MyCwNet.RigControl, 10.0/*dB*/ );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_SpectrumRefLevel_20Click(TObject *Sender)
{ RigCtrl_SetScopeRefLevel_dB( &MyCwNet.RigControl, 20.0/*dB*/ );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Btn_MoreKeyerSettingsClick(TObject *Sender)
{
  UpdateItemsInSettingsMenu(); // before opening a submenu of the "Settings",
                               // update checkmarks, values, etc in the menu items.
  // Because users hardly saw the "More Keyer Settings" in the old-fashioned
  // main menu, try to open that particular sub-menu programmatically:
  // MI_MoreKeyerSettings->Click();   // <- of course THIS did not work (the first attempt)
  // Googled around, and found this at stackoverflow.com :
  // > As part of a built in tutorial system for my software I would like to
  // > programatically open the window menus of the software and show the user
  // > where to find certain functions. I've searched the Win32 API
  // > and can't find what I'm looking for. Is this possible via the API
  // > or do I need to provide some kind of workaround?
  // Answer:
  // > The submenu items of menubar or any other submenu (popu menu)
  // > can be displayed with TrackPopupMenu(Ex) API function. This function
  // > needs the menu handle of the submenu and coordinates at which to display it.
  // > For instance in order to display the File menu of menubar
  // > the following actions should be taken:
  // >    Get File submenu handle with GetSubMenu(0);
  // >    Get File menu item rectangle using GetMenuItemRect;
  // >    calculate the desired coordinates using the rectangle;
  // >    TrackPopupMenu(Ex) to display the menu.
  // "TrackPopupMenu()". A-ha. Who would have thought of that ? MS say:
  // > Displays a shortcut menu at the specified location and tracks
  // > the selection of items on the menu. The shortcut menu can appear
  // > anywhere on the screen.
  // A-ha. But "GetSubMenu()" expects a handle to the MENU (here: the entire main menu)
  //       and a "menu position", which seem to be a zero-based index into something.
  // What we have here (in a Borland VCL application) is the TMainMenu (and its handle)
  // and a "TMenuItem" (e.g. MI_MoreKeyerSettings) deeply buried in the TMainMenu.
  TrackPopupMenu( // <- funny-named Win32 API function to "open a certain menu item"
          // [in] HMENU hMenu : "A handle to the shortcut menu to be displayed" :
          GetSubMenu( Menu_Settings->Handle, Menu_Settings->IndexOf(MI_MoreKeyerSettings) ),
          //   ,-----------------------------|__________________________________________|
          //   '--> iMenuPos = "The zero-based relative position in the specified menu
          //                    of an item that activates a drop-down menu or submenu."
          // .. phew, fortunately Borland's VCL came to the rescue with "IndexOf()",
          //    and to the author's big surprise, this worked straight out of the box !
          TPM_LEFTALIGN, // [in] UINT uFlags : "Use zero of more of these flags to specify function options":
                         // TPM_CENTERALIGN, TPM_LEFTALIGN, TPM_RIGHTALIGN .
          Mouse->CursorPos.x, // [in] int x,
          Mouse->CursorPos.y, // [in] int y,
          0,       // [in] int nReserved,
          Handle,  // [in] HWND hWnd : "A handle to the window that owns the shortcut menu."
          NULL);   // [in, optional] const RECT *prcRect : "Ignored" (!).
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Btn_MoreHLServerOptionsClick(TObject *Sender)
{
  UpdateItemsInSettingsMenu(); // before opening a submenu of the "Settings",
                               // update checkmarks, values, etc in the menu items.
  TrackPopupMenu( // <- funny-named Win32 API function to "open a certain menu item"
     // [in] HMENU hMenu : "A handle to the shortcut menu to be displayed" :
     GetSubMenu( Menu_Settings->Handle, Menu_Settings->IndexOf(MI_HamlibServerOptions) ),
     // ,-------------------------------|____________________________________________|
     // '--> iMenuPos = "The zero-based relative position in the specified menu
     //                  of an item that activates a drop-down menu or submenu."
     TPM_LEFTALIGN, // [in] UINT uFlags : TPM_CENTERALIGN, TPM_LEFTALIGN, TPM_RIGHTALIGN .
     Mouse->CursorPos.x, Mouse->CursorPos.y, // [in] int x, y
     0,       // [in] int nReserved
     Handle,  // [in] HWND hWnd : "A handle to the window that owns the shortcut menu."
     NULL);   // [in, optional] const RECT *prcRect : "Ignored" (!).
}


//---------------------------------------------------------------------------
void TKeyerMainForm::SetNewLanguage( int iNewLanguage )
{
  if( iNewLanguage != APPL_iLanguage )
   { APPL_iPrevLanguage = APPL_iLanguage;
     APPL_iLanguage = iNewLanguage;
     APPL_TranslateAllForms(); // [in] APPL_iLanguage, APPL_iPrevLanguage
     APPL_fSaveSettingsOnExit = TRUE; // flag to call SaveSettings() when terminating
   }
} // end TKeyerMainForm::SetNewLanguage()

void __fastcall TKeyerMainForm::MI_Language_EnglishClick(TObject *Sender)
{ SetNewLanguage( TRANSLATOR_LANGUAGE_ENGLISH );
}
void __fastcall TKeyerMainForm::MI_Language_GermanClick(TObject *Sender)
{ SetNewLanguage( TRANSLATOR_LANGUAGE_GERMAN );
}
void __fastcall TKeyerMainForm::MI_Language_FrenchClick(TObject *Sender)
{ SetNewLanguage( TRANSLATOR_LANGUAGE_FRENCH );
}
void __fastcall TKeyerMainForm::MI_Language_DutchClick(TObject *Sender)
{ SetNewLanguage( TRANSLATOR_LANGUAGE_DUTCH );
}
void __fastcall TKeyerMainForm::MI_Language_ItalianClick(TObject *Sender)
{ SetNewLanguage( TRANSLATOR_LANGUAGE_ITALIAN );
}
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_UI_SimpleClick(TObject *Sender)
{ g_iUserInterfaceMode = UI_MODE_SIMPLE;
}
void __fastcall TKeyerMainForm::MI_UI_CompleteClick(TObject *Sender)
{ g_iUserInterfaceMode = UI_MODE_COMPLETE;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_AutoFocusSwitchClick(TObject *Sender)
{ KeyerGUI_fAutomaticFocusSwitching = !KeyerGUI_fAutomaticFocusSwitching;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_HideMenuClick(TObject *Sender)
{
  g_fHideMenu = !g_fHideMenu;
  m_iMainMenuCountdownTimer = 0;
  HideOrShowMainMenu(g_fHideMenu); // turning off the menu in the menu shall happen IMMEDIATELY :)
}

//---------------------------------------------------------------------------
void TKeyerMainForm::HideOrShowMainMenu(BOOL fHideMenu)
{
  // Why hide the main menu ? Because it's ugly, and doesn't give a shit
  //                          about this application's "colour scheme" (e.g. dark).
  //                    Also, it occupies precious space in the client area ..
  if( fHideMenu )
   { MainMenu1->Items->Visible = FALSE; // this does NOT make the menu itself invisble (but it cannot hurt)
     // MainMenu1->Visible = FALSE;     // this simply does not exist (at least not in the VCL)
     Menu = NULL;
     // '--> A property / member of this particular TKeyerMainForm .
     //      Unfortunately, taking "MainMenu1" away from this form
     //      didn't impress the VCL. Still, hovering over the place
     //      WHERE THE MENU USED TO BE (before it was hidden)
     //      made the items in the main menu line gradually reappear,
     //      just as if some VCL-internal still "remembered" the menu was there.
     //
     // As so often, it Borland's VCL cannot do it, Windows (plain old Win32 API) can.
     // Suggested by Raymond Chen(!) on Stack Overflow:
     // > The SetMenu function lets you add/remove the menu from the window.
     // > It does not destroy the menu.
     // > Note that most applications which have the dynamic menu hide/show behavior
     // > are not really showing a menu. They're showing a custom control that looks like a menu.
     // Directly from Richmond :
     // > SetMenu(hWnd, hMenu) assigns a new menu to the specified window.
     // > If the parameter hMenu is NULL, the window's current menu is removed.
     ::SetMenu( Handle/*that's the MAIN WINDOW'S handle*/, NULL/*that's the "hMenu"*/ );
     // To the author's surprise, even the client area resized itself properly
     //    when hiding or unhiding the main menu.  BUT:
     // Borland's VCL thought the menu was still visible, and hovering
     // with the mouse pointer over the area "where the menu used to be"
     // caused a 'transparent ghost image' there, and the blue-coloured
     // main menu items gradually destroyed the labels in the top panel
     // (Pnl_Top). Tried to fix this with the obfuscated stuff further above
   }
  else // make the menu VISIBLE again (in its own 'line' above the client area):
   { Menu = MainMenu1;  // restore this property(?) of TKeyerMainForm .
     ::SetMenu( Handle/*that's the MAIN WINDOW*/, MainMenu1->Handle/*that's the "hMenu"*/ );
     MainMenu1->Items->Visible = TRUE; // let some hidden beast in the VCL know "the menu items are back"
   }
} // end TKeyerMainForm::HideOrShowMainMenu()


//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_ITU_Region_1Click(TObject *Sender)
{ RigCtrl_ITU_Region = 1;
}
void __fastcall TKeyerMainForm::MI_ITU_Region_2Click(TObject *Sender)
{ RigCtrl_ITU_Region = 2;
}
void __fastcall TKeyerMainForm::MI_ITU_Region_3Click(TObject *Sender)
{ RigCtrl_ITU_Region = 3;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Funcs_TrxPowerOffClick(TObject *Sender)
{ // Send a command to turn the radio off (as soon as possible),
  //  but don't interfere with a possibly pending command/response :
  RigCtrl_SwitchPollingState( &MyCwNet.RigControl, RIGCTRL_POLLSTATE_TURN_RIG_OFF );
     // '--> the rest happens in a state machine in RigCtrl_Handler(), or wherever..
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Funcs_TrxPowerOnClick(TObject *Sender)
{ // Send a command to turn the radio ON (as soon as possible) ... :
  RigCtrl_SwitchPollingState( &MyCwNet.RigControl, RIGCTRL_POLLSTATE_TURN_RIG_ON );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Funcs_PowerOffWhenClosingClick(TObject *Sender)
{
  CwKeyer_Config.fTurnRigOffWhenClosing = !CwKeyer_Config.fTurnRigOffWhenClosing;
}


//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_SpectrumSpan_2500Click(TObject *Sender)
{ RigCtrl_SetScopeSpan_Hz( &MyCwNet.RigControl, 2.5e3 );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_SpectrumSpan_5kHzClick(TObject *Sender)
{ RigCtrl_SetScopeSpan_Hz( &MyCwNet.RigControl, 5e3 );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_SpectrumSpan_10kHzClick(TObject *Sender)
{ RigCtrl_SetScopeSpan_Hz( &MyCwNet.RigControl, 10e3 );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_SpectrumSpan_25kHzClick(TObject *Sender)
{ RigCtrl_SetScopeSpan_Hz( &MyCwNet.RigControl, 25e3 );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_SpectrumSpan_50kHzClick( TObject *Sender)
{ RigCtrl_SetScopeSpan_Hz( &MyCwNet.RigControl, 50e3 );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_SpectrumSpan_100kHzClick(TObject *Sender)
{ RigCtrl_SetScopeSpan_Hz( &MyCwNet.RigControl, 100e3 );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_SpectrumSpan_250kHzClick(TObject *Sender)
{ RigCtrl_SetScopeSpan_Hz( &MyCwNet.RigControl, 250e3 );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_SpectrumSpan_500kHzClick(TObject *Sender)
{ RigCtrl_SetScopeSpan_Hz( &MyCwNet.RigControl, 500e3 );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_WFColourPalette_SunriseClick(TObject *Sender)
{ g_SpecDispControl.iWFColorPalette = WF_PALETTE_SUNRISE;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_WFColourPalette_LinradClick(TObject *Sender)
{ g_SpecDispControl.iWFColorPalette = WF_PALETTE_LINRAD;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_WFColourPalette_RedClick(TObject *Sender)
{ g_SpecDispControl.iWFColorPalette = WF_PALETTE_RED;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_WFColourPalette_GreenClick(TObject *Sender)
{ g_SpecDispControl.iWFColorPalette = WF_PALETTE_GREEN;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_WFColourPalette_BlueClick(TObject *Sender)
{ g_SpecDispControl.iWFColorPalette = WF_PALETTE_BLUE;
}
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Settings_SpectrumDisplayClick(TObject *Sender)
{
  // When invoked from the MAIN MENU, there is no 'clicked frequency' for PM_Spectrum,
  // so use this instead:
  g_SpecDispControl.dblClickedVfoFreq_Hz = g_SpecDispControl.dblDisplayedVfoFreq_Hz;
  PM_Spectrum->Popup(Mouse->CursorPos.x, Mouse->CursorPos.y); // [in] g_SpecDispControl.dblClickedVfoFreq_Hz, etc (?)
}


//---------------------------------------------------------------------------
// Events from the RichEdit control for "Info, RX-data, or input for TX"
//        on the panel ABOVE the tabbed pages (thus ALWAYS IN SIGHT) :
//  * "OnKeyDown", "OnKeyPress", and "OnKeyUp" handlers for RichEdit_RxTxInfo
//  * Functions to append individually coloured characters to the text in the editor
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::RichEdit_RxTxInfoKeyDown(TObject *Sender,
      WORD &Key, TShiftState Shift) // READ THE DETAILS BELOW..
           // there a lot of pitfalls with 'Key', and we need
           // MORE KEYBOARD EVENT HANDLERS than this to suppress certain keystrokes !
{ // "OnKeyDown" handler for the RichText control with RX / TX data or "Info",
  //             located in the always-visible status line above the tabbed pages.
  // This is only ONE of THREE keyboard-event-handlers in Windows (and the VCL);
  // actually it's Borland's abstraction of WM_KEYDOWN.
  // Since the help system in C++Builder V6 is now completely defunct,
  // snapped up the following info in a forum:
  // > The OnKeyDown handler can respond to keyboard keys, including
  // > function keys and keys combined with the SHIFT, ALT, and CTRL keys,
  // > and pressed mouse buttons.
  // > An application gets Windows WM_KEYDOWN messages for all keys when the user
  // > presses a key. These messages indirectly fire the OnKeyDown event.
  // > Setting the Key parameter to #0 prevents any further processing of this message.
  // > But for keys that generate characters Windows also produces WM_CHAR.
  // > At the time your OnKeyDown event fires, the WM_CHAR message for the key
  // > will already be in the message queue. Setting Key to #0 does not stop it
  // > from being delivered, so it fires the OnKeyPress event.
  // > If you set the Key to #0, OnKeyPress will be prevented from being fired
  // > only for keys that do not have chars. For keys that represent characters,
  // > OnKeyPress will continue to be fired.
  // > This method of organizing key processing has advantages. Code that only
  // > deals with characters, including control characters like #13 for carriage return,
  // > #3 for CTRL-C, and so on, should go into the OnKeyPress event.
  // > Code that deals with keys that do not generate characters should be put
  // > into the OnKeyDown event (..)
  // (similar also for other Edit- and 'Rich Edit' controls, so see also:
  //            * RichEdit_RxTxInfoKeyDown(), RichEdit_RxTxInfoKeyPress();
  //            * Ed_ErrorHistoryKeyDown(),   Ed_ErrorHistoryKeyPress() .
  switch( Key/*a "WORD"*/ ) // "suppress" this key (by setting Key=0), or let it pass through to the editor ?
   { case VK_RETURN: // .. in RichEdit_RxTxInfo : do NOT 'enter' this into the text, but pass it to KeyerGUI_OnKeyInTransmitTextEditor()
        Key = 0;  // <- Borland's way to indicate 'we have processed this keystroke,
        //                        and don't want to pass this to the default handler'.
        // Beware: Because this key 'generates a character', Windows has
        //         already translated the WM_KEYDOWN into a WM_CHAR message,
        // which the "OnKeyDown"-handler cannot intercept. Only "OnKeyPress" can.
        // With breakpoints in both RichEdit_RxTxInfoKeyDown() and RichEdit_RxTxInfoKeyPress(),
        // the one in RichEdit_RxTxInfoKeyDown() fired FIRST, so do the actual
        // 'processing' here (e.g. switch from receive to transmit, and start
        // sending whatever has been entered in the TRichEdit control so far):
        KeyerGUI_OnKeyInTransmitTextEditor( RichEdit_RxTxInfo, '\r' );
        //   '--> [out] KeyerGUI_iTxEditorCharIndex_StartOfMsg, KeyerGUI_iTxEditorCharIndex_CwGenerator, ...
        break;
     default:
        break;
   } // end switch( Key ) in the "OnKeyDown"-handler (!)

} // end RichEdit_RxTxInfoKeyDown()


//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::RichEdit_RxTxInfoKeyPress(TObject *Sender,
      char &Key) // See details about "OnKeyPress" in the summary about
                 // keyboard handling in RichEdit_RxTxInfoKeyDown() !
                 // We need MORE THAN A SINGLE KEYBOARD EVENT HANDLER
                 // to suppress certain keystrokes !
                 // > "For keys that represent characters, OnKeyPress
                 // >  will continue to be fired."
  // Thus, for characters that cannot be sent in Morse code later,
  // this "OnKeyPress" handler (windows:WM_CHAR) had to be implemented BESIDES
  //  the "OnKeyDown" handler (windows:WM_KEYDOWN, further above) ... buaaaah !
{ // "OnKeyPress" handler for the RichText control with RX / TX data or "Info",
  //             located in the always-visible status line above the tabbed pages.
  // Since the help system in C++Builder V6 is now completely defunct,
  // snapped up the following info in a forum:
  // > The Key parameter in the OnKeyPress event handler is of type Char;
  // > therefore, the OnKeyPress event registers the ASCII character
  // > of the key pressed. Keys that do not correspond to an ASCII Char value
  // > (SHIFT or F1, for example) do not generate an OnKeyPress event.
  // > Key combinations (such as SHIFT+A) generate only one OnKeyPress event
  // > (for this example, SHIFT+A results in a Key value of "A" if Caps Lock is off).
  // > To respond to non-ASCII keys or key combinations, use the OnKeyDown
  // > or OnKeyUp event handler.
  BOOL fOkToSend;

  switch( Key/*a "char"*/ ) // "suppress" this ASCII character (by setting Key=0), or let it pass through to the editor ?
   {
     case '\r':  // Carriage Return (translated from the ENTER key) ?
        Key = 0; // we don't want this to insert a NEW LINE in the single-line edit text,
                 // because if it would, the characters still waiting to be sent
                 // (in Morse code) would be scrolled out.
                 // (BCB6 VCL's "TRichEdit" doesn't have a 'single line' option)
        // Note: Most of the work on pressing ENTER has already been performed
        //       in RichEdit_RxTxInfoKeyDown().
        // Intercepting "Enter" alias "Return" here also caused the CURSOR
        // to stay where it was. That's intended behaviour - the operator may
        // keep editing the text WHILE IT IS BEING SENT, as long as the to-be-
        // replaced character hasn't been transmitted (indicated by colour).
        break;
     default:
        fOkToSend = KeyerGUI_OnKeyInTransmitTextEditor(RichEdit_RxTxInfo, Key);
        if( !fOkToSend ) // the CW generator will not be able to send this character later ->
         { Key = 0; // .. so don't let Mr. Rich Edit append it to the text
         }
        // Note: The decision if, when, and now many characters to transfer
        //       from RichEdit_RxTxInfo (when used as a "type-ahead buffer"
        //       for transmissions in real time) isn't made HERE,
        //       but periodically in the GUI's "OnTimer"-handler.
        //       That involves checking CwKeyer_Gen.nCharsRemainingToPlay
        //       to keep the number of characters handed over from the
        //       editor (RichEdit_RxTxInfo) to the CW generator thread
        //       as low as possible, because:
        //  * Characters transferred from RichEdit_RxTxInfo to the
        //    CW generator cannot be DELETED (backspace) or EDITED anymore.
        //  * Characters transferred from RichEdit_RxTxInfo to the
        //    CW generator will get a different BACKGROUND COLOUR
        //    on that occasion (with KeyerGUI_iRxTxInfoUsage==KEYER_GUI_RXTXINFO_TYPING_TXD).
        break; // end default, i.e. "any other, potentially SENDABLE character"
   } // end switch( Key ) in the "OnKeyPress"-handler (!)
}   // end RichEdit_RxTxInfoKeyPress()

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Btn_MoreAudioCwDecoderOptionsClick(TObject *Sender)
{ PM_AudioCwDecoderOptions->Popup(Mouse->CursorPos.x, Mouse->CursorPos.y);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::PM_AudioCwDecoderOptionsPopup(TObject *Sender)
{
  AnsiString s;
  ++KeyerGUI_iUpdating; // prevent interfering in certain "OnClick()" and "OnChange()" handlers
  MI_ACwDecoder_Enabled->Checked = (CwKeyer_DSP.cfg.iAudioFlags & DSP_AUDIO_FLAGS_DECODE_CW) != 0;
  MI_PeakHoldingAudioSpectrum->Checked = (CwKeyer_DSP.cfg.iAudioFlags & DSP_AUDIO_FLAGS_PEAK_HOLDING_SPECTRUM) != 0;

  --KeyerGUI_iUpdating;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_ACwDecoder_EnabledClick(TObject *Sender)
{
  CwKeyer_DSP.cfg.iAudioFlags ^= DSP_AUDIO_FLAGS_DECODE_CW;
  Chk_Audio_DecodeCW->Checked = (CwKeyer_DSP.cfg.iAudioFlags & DSP_AUDIO_FLAGS_DECODE_CW) != 0;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_PeakHoldingAudioSpectrumClick(TObject *Sender)
{
  CwKeyer_DSP.cfg.iAudioFlags ^= DSP_AUDIO_FLAGS_PEAK_HOLDING_SPECTRUM;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_TRX_Tab_AudioCWDecoderOptionsClick(TObject *Sender)
{ PM_AudioCwDecoderOptions->Popup(Mouse->CursorPos.x, Mouse->CursorPos.y);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::PM_TScope_MoreAudioSpectrumSettingsClick(TObject *Sender)
{ // Same submenu as on the "TRX" tab, with AUDIO CW decoder / Audio Spectrum related settings:
  PM_AudioCwDecoderOptions->Popup(Mouse->CursorPos.x, Mouse->CursorPos.y);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_AudioSpectrumRefLevel_EditClick(TObject *Sender)
{
  int    iValue = CwKeyer_TimingScope.cfg.iAudioSpectrumRefLevel_dB;
  char   sz80Label1[84], sz80Label2[84], *pszDest, *pszEndstop;
  pszDest    = sz80Label1;
  pszEndstop = sz80Label1+80;
  SL_AppendPrintf( &pszDest, pszEndstop, "%s: -50 .. +50 dB", TE("Valid range") );
  pszDest    = sz80Label2;
  pszEndstop = sz80Label2+80;
  SL_AppendPrintf( &pszDest, pszEndstop, "%s: %d dB", TE("Current value"), iValue );
  if( YHF_RunIntegerInputDialog(
       Handle,  // [in] handle to the window that will be blocked by the 'modal' dialog
       (char*)TE("Audio Spectrum Reference Level"), // [in] char *pszTitle,
       sz80Label1, sz80Label2,   // [in] labels (info strings) above the edit field
       &iValue,                  // [in,out] int *piValue
       YHF_EDIT_NORMAL,          // [in] int iEditOptions,
       HELPID_SPECTRUM_REF_LEVEL)// [in] int  iHelpID
     == YHF_DLG_BTN_OK ) // operator clicked "OK" or confirmed with "Enter" ->
   { CwKeyer_TimingScope.cfg.iAudioSpectrumRefLevel_dB = iValue;
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_AudioSpectrumRefLevel_m20Click(TObject *Sender)
{ // For lazy people and Tablet PC users, allow picking a few 'typical' values from a submenu:
  CwKeyer_TimingScope.cfg.iAudioSpectrumRefLevel_dB = -20;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_AudioSpectrumRefLevel_m10Click(TObject *Sender)
{ CwKeyer_TimingScope.cfg.iAudioSpectrumRefLevel_dB = -10;
}
//---------------------------------------------------------------------------

void __fastcall TKeyerMainForm::MI_AudioSpectrumRefLevel_0Click(TObject *Sender)
{ CwKeyer_TimingScope.cfg.iAudioSpectrumRefLevel_dB = 0;
}
//---------------------------------------------------------------------------

void __fastcall TKeyerMainForm::MI_AudioSpectrumRefLevel_10Click(TObject *Sender)
{ CwKeyer_TimingScope.cfg.iAudioSpectrumRefLevel_dB = 10;
}
//---------------------------------------------------------------------------

void __fastcall TKeyerMainForm::MI_AudioSpectrumRefLevel_20Click(TObject *Sender)
{ CwKeyer_TimingScope.cfg.iAudioSpectrumRefLevel_dB = 20;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_TScope_AudioSpectrumAmplRange_10Click(TObject *Sender)
{ CwKeyer_TimingScope.cfg.iAudioSpectrumAmplRange_dB = 10;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_TScope_AudioSpectrumAmplRange_20Click(TObject *Sender)
{ CwKeyer_TimingScope.cfg.iAudioSpectrumAmplRange_dB = 20;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_TScope_AudioSpectrumAmplRange_30Click(TObject *Sender)
{ CwKeyer_TimingScope.cfg.iAudioSpectrumAmplRange_dB = 30;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_TScope_AudioSpectrumAmplRange_50Click(TObject *Sender)
{ CwKeyer_TimingScope.cfg.iAudioSpectrumAmplRange_dB = 50;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_TScope_AudioSpectrumAmplRange_70Click(TObject *Sender)
{ CwKeyer_TimingScope.cfg.iAudioSpectrumAmplRange_dB = 70;
  // '--> 70 dB dynamic range is too much for an audio stream with 8 bits/sample,
  //      but nice-to-have for a real SDR, or high-end receiver with digital IF output
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Ed_ErrorHistoryKeyDown(TObject *Sender, WORD &Key, TShiftState Shift)
  // This is the 'OnKeyDown' method (a la Borland VCL), in reality a handler
  // for WM_KEYDOWN for the Rich Text edit control on the Remote CW Keyer's DEBUG tab.
  // Normally, that visual control object is only used for a (colourful)
  // console-like DISPLAY. But in THIS application, it may be used for EDITING
  // lines of text, and sending them as a COMMAND back to a simplistic
  // command line parser. For example, the user may have listed the
  //      'Rig Control Parameters' as text in the debug tab,
  //      resulting in a couple of lines like the following:
  // > Current Rig Control parameters:
  // > PN001: RadioID = 0x94 (IC-7300)
  // > PN002: VFOfreq = 7035022.3 Hz
  //          (... etc, etc ...)
  // > PN040: AudioVolume = 16 %
  // > PN042: RFGain = 100 %
  // > PN043: SquelchLevel = 14 %
  // > PN045: NoiseReduction = 15 %
  // > PN046: PassbandT1 = 50 %
  // > PN047: PassbandT2 = 50 %
  // > PN048: CW_Pitch = 630 Hz
  // > PN041: PowerSet = 5 %
  // > PN049: MicGain = 35 %
  // > PN050: KeyerSpeed = 22 WPM   (that's the speed of the radio's BUILT IN CW keyer)
  // > PN051: NotchPos = 49 %
  // > PN052: CompSet = 68 %
  //          (... etc, etc ...)
  // Instead of plastering the GUI with hundreds of edit fields, the operator can
  // click onto the numeric parameter in the editor, overwrite it, and finally
  // press ENTER to 'send back' the modified value to the remotely controlled radio.
  //
  // This is only ONE of THREE keyboard-event-handlers in Windows (and the VCL);
  // actually it's Borland's abstraction of WM_KEYDOWN.
  // Since the help system in C++Builder V6 is now completely defunct,
  // snapped up the following info in a forum:
  // > The OnKeyDown handler can respond to keyboard keys, including
  // > function keys and keys combined with the SHIFT, ALT, and CTRL keys,
  // > and pressed mouse buttons.
  // > An application gets Windows WM_KEYDOWN messages for all keys when the user
  // > presses a key. These messages indirectly fire the OnKeyDown event.
  // > Setting the Key parameter to #0 prevents any further processing of this message.
  // > But for keys that generate characters Windows also produces WM_CHAR.
  // > At the time your OnKeyDown event fires, the WM_CHAR message for the key
  // > will already be in the message queue. Setting Key to #0 does not stop it
  // > from being delivered, so it fires the OnKeyPress event.
  // > If you set the Key to #0, OnKeyPress will be prevented from being fired
  // > only for keys that do not have chars. For keys that represent characters,
  // > OnKeyPress will continue to be fired.
  // > This method of organizing key processing has advantages. Code that only
  // > deals with characters, including control characters like #13 for carriage return,
  // > #3 for CTRL-C, and so on, should go into the OnKeyPress event.
  // > Code that deals with keys that do not generate characters should be put
  // > into the OnKeyDown event (..)
  // (similar also for other Edit- and 'Rich Edit' controls, so see also:
  //            * RichEdit_RxTxInfoKeyDown(), RichEdit_RxTxInfoKeyPress();
  //            * Ed_ErrorHistoryKeyDown(),   Ed_ErrorHistoryKeyPress();
  //            * Ed_Chatbox_KeyDown(),       Ed_Chatbox_KeyPress() .
{
  switch( Key/*a "WORD"*/ ) // "suppress" this key (by setting Key=0), or let it pass through to the editor ?
   { case VK_RETURN: // .. in RichEdit_ErrorHistory : do NOT 'enter' this into the text, because KeyerGUI_OnKeyInErrorHistory() has handled it.
        // Beware: Because this key 'generates a character', Windows has
        //         already translated the WM_KEYDOWN into a WM_CHAR message,
        // which the "OnKeyDown"-handler cannot intercept. Only "OnKeyPress" can.
        // Thus, for any key with an ASCII equivalent (including CONTROL CODES),
        // let
        Key = 0;
        break;
     default:
        break;
   } // end switch( Key ) in the "OnKeyDown"-handler (!)

} // end TKeyerMainForm::Ed_ErrorHistoryKeyDown()

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Ed_ErrorHistoryKeyPress(TObject *Sender,char &Key)
{
  // "suppress" certain keys (by setting Key=0), because as explained in
  // Ed_ErrorHistoryKeyDown(), Borland's "OnKeyDown" (handler for WM_KEYDOWN)
  // cannot intercept keys with ASCII equivalent, e.g. '\r' (0x0D) for VK_RETURN !
  // So handle those keyboard events ("with ASCII") here, and let
  // KeyerGUI_OnKeyInErrorHistory() decide what to do (modify text in the editor,
  // send in 'text terminal mode', etc):
  if( KeyerGUI_OnKeyInErrorHistory( Ed_ErrorHistory, Key ) )
   { Key = 0;  // <- Borland's way to indicate 'we have processed this keystroke,
     //              and don't want to pass this to the default handler'.
     // ( Unfortunately, in "OnKeyDown" this has no effect on keys
     //   with an ASCII equivalent - including CONTROL CODES like '\r'
     //   for reasons quoted above .. but here again, madness takes its toll:
     // > If you set the Key to #0, OnKeyPress will be prevented from being fired
     // > only for keys that do not have chars. For keys that represent characters,
     // > OnKeyPress will continue to be fired.
     //   Fixed by implementing an OnKeyPress handler besides this OnKeyDown handler.
     // )
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Btn_AuxCom1DetailsClick(TObject *Sender)
{
   ConfigureAuxComDetails(0);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Btn_AuxCom2DetailsClick(TObject *Sender)
{
   ConfigureAuxComDetails(1);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Btn_AuxCom3DetailsClick(TObject *Sender)
{
   ConfigureAuxComDetails(2);
}

//---------------------------------------------------------------------------
void TKeyerMainForm::ConfigureAuxComDetails(int iAuxComIndex )
{
  DWORD dwHash = UTL_CRC32( 0/*seed*/, (BYTE*)&CwKeyer_Config.sAuxCom[iAuxComIndex], sizeof(CwKeyer_Config.sAuxCom[0]) );

  ApplyAuxComPortSettings(); // copy from controls on the "Additional COM Ports" tab into CwKeyer_Config.xyz,
     // because AuxComPortDetails_RunModal() only operates on CwKeyer_Config
     // but not on the VCL stuff (which it doesn't "see" anyway)
  if( AuxComPortDetails_RunModal( iAuxComIndex ) ) // <- in AuxComPortDetails_Dialog.cpp !
   { // here when the user clicked "OK", so CwKeyer_Config may have been modified:
     if( dwHash != UTL_CRC32( 0, (BYTE*)&CwKeyer_Config.sAuxCom[iAuxComIndex], sizeof(CwKeyer_Config.sAuxCom[0]) ) )
      { KeyerGUI_fMustApplyConfigAndStart = TRUE; // let the button "Apply new settings and start" BLINK
      }
     UpdateAuxComPortSettings();
   }
} // end TKeyerMainForm::ConfigureAuxComDetails()


//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_File_ImportSettingsFromClick(TObject *Sender)
{
  AnsiString s, sUserFile;
  int iLoadWhat;
  int iAnswer;
  (void)Sender;

  /* see help on TOpenDialog .. unfortunately the ancient help system is defunct */
  OpenDialog->Title = "Import RCW Keyer Settings from file";
  // OpenDialog->InitialDir = ?;  // leave if to the VCL, or to Windows, to chose an "initial directory"
  OpenDialog->DefaultExt = "ini";
  OpenDialog->FileName = "*.ini";
  OpenDialog->Filter = "INI Files (*.ini)|*.INI";
  OpenDialog->Options.Clear();
  OpenDialog->Options << ofFileMustExist << ofHideReadOnly << ofNoChangeDir << ofShowHelp;
  OpenDialog->HelpContext = HELPID_EXPORT_SETTINGS;
  if (OpenDialog->Execute())
   { m_sLastExportedOrImportedFile = ExpandFileName( OpenDialog->FileName );
     if( LoadSettings( m_sLastExportedOrImportedFile.c_str(), CONFIG_FILE_OPTION_ALL_BUT_RECENT_FILES ) )
      {  AddToRecentFiles( m_sLastExportedOrImportedFile.c_str() );
         ShowError( ERROR_CLASS_INFO | SHOW_ERROR_TIMESTAMP, "Loaded settings from %s",
                           m_sLastExportedOrImportedFile.c_str() );

         KeyerGUI_fAutomaticTabSwitching = TRUE; // allow switching between tabsheets again, here: after LOADING A NEW CONFIGURATION.
         //  The rest happens in Timer1Timer() when module RigControl enters
         //  RIGCTRL_POLLSTATE_DONE. ONLY THEN, the GUI switches from the
         //  "Debug" tab (with a live CAT traffic display)
         //  to the "TRX" tab (Transceiver control with VFO and spectrum.
         UpdateAllControlsFromConfig();
         StopAndRestart();
      }
   } // end if (OpenDialog->Execute())
} // end TKeyerMainForm::MI_File_ImportSettingsFromClick()

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_File_ExportSettingsAsClick(TObject *Sender)
{
  (void)Sender;
  // SetCurrentDirectory( ? );  // use 'our data file' directory with read- AND WRITE permission
  SaveDialog->Title = "Export RCW Keyer Settings as file";
  // SaveDialog->InitialDir = ?
  // The stupid windows fileselector box often DOESN'T GIVE A DAMN about
  //   this 'InitialDir'-thing, even though InitialDir was valid (existed) .
  //   Setting the current directory to a *parent directory* seemed to help .
  SaveDialog->DefaultExt= "ini";
  SaveDialog->FileName = m_sLastExportedOrImportedFile;
  SaveDialog->Filter = "INI Files (*.ini)|*.INI";
  SaveDialog->Options.Clear();
  SaveDialog->Options << ofFileMustExist << ofHideReadOnly << ofNoChangeDir << ofShowHelp;
  SaveDialog->HelpContext = HELPID_EXPORT_SETTINGS;
  if (SaveDialog->Execute())
   { m_sLastExportedOrImportedFile = SaveDialog->FileName;
     SaveSettings( m_sLastExportedOrImportedFile.c_str(), CONFIG_FILE_OPTION_ALL_BUT_RECENT_FILES );
     AddToRecentFiles( m_sLastExportedOrImportedFile.c_str() );
   }
  // ex: SetCurrentDirectory( APPL_sz255CurrentDirectory );  // set CWD back to where it should be

} // end TKeyerMainForm::MI_File_ExportSettingsAsClick()
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_ClearCecentFilesClick(TObject *Sender)
{
  ClearRecentFiles();
}

//---------------------------------------------------------------------------
void TKeyerMainForm::ImportRecentFile( int iZeroBasedIndex )
{ const char *pszFilename = GetRecentFileName(iZeroBasedIndex);
  if( pszFilename[0] != '\0' ) // not an empty entry in the list -> try to LOAD it
   { if( LoadSettings( pszFilename, CONFIG_FILE_OPTION_ALL_BUT_RECENT_FILES ) )
      { m_sLastExportedOrImportedFile = AnsiString( pszFilename );
        UpdateAllControlsFromConfig();
        StopAndRestart();  // <- also clears KeyerGUI_fMustApplyConfigAndStart to stop 'flashing'
      }
   }
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_RecentFile0Click(TObject *Sender)
{
  ImportRecentFile(0);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_RecentFile1Click(TObject *Sender)
{
  ImportRecentFile(1);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_RecentFile2Click(TObject *Sender)
{
  ImportRecentFile(2);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_RecentFile3Click(TObject *Sender)
{
  ImportRecentFile(3);
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_TMColumns_AllClick(TObject *Sender)
{ RigCtrl_TrafficMonitor.iExtraColumns = RIGCTRL_TMON_EXTRA_COLUMNS_ALL;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_TMColumns_TimeClick(TObject *Sender)
{ RigCtrl_TrafficMonitor.iExtraColumns ^= RIGCTRL_TMON_EXTRA_COLUMN_TIME;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_TMColumns_TX_RXClick(TObject *Sender)
{ RigCtrl_TrafficMonitor.iExtraColumns ^= RIGCTRL_TMON_EXTRA_COLUMN_TX_RX;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_TMColumns_PayloadLengthClick(TObject *Sender)
{ RigCtrl_TrafficMonitor.iExtraColumns ^= RIGCTRL_TMON_EXTRA_COLUMN_PAYLOAD_LENGTH;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_TMColumns_CommentClick(TObject *Sender)
{ RigCtrl_TrafficMonitor.iExtraColumns ^= RIGCTRL_TMON_EXTRA_COLUMN_COMMENTS;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_TMon_Timestamp_UTCClick(TObject *Sender)
{ RigCtrl_TrafficMonitor.iTimestampFormat = RIGCTRL_TMON_TIMESTAMP_TIME_OF_DAY;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_TMon_Timestamp_MillisecondsClick(TObject *Sender)
{ RigCtrl_TrafficMonitor.iTimestampFormat = RIGCTRL_TMON_TIMESTAMP_MILLISECONDS;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_TrafficMon_PauseOnFullFIFOClick(TObject *Sender)
{ RigCtrl_TrafficMonitor.iDisplayOptions ^= RIGCTRL_TMON_DISPLAY_OPTION_PAUSE_ON_FULL_FIFO;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_TrafficMonitor_PauseOnErrorClick(TObject *Sender)
{ RigCtrl_TrafficMonitor.iDisplayOptions ^= RIGCTRL_TMON_DISPLAY_OPTION_PAUSE_ON_ERROR;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_TrafficMonitor_PauseOnInitDoneClick(TObject *Sender)
{ RigCtrl_TrafficMonitor.iDisplayOptions ^= RIGCTRL_TMON_DISPLAY_OPTION_PAUSE_ON_INIT_DONE;
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Predef_IO_Simple_Paddle_AdapterClick(TObject *Sender)
{
  RecallPrefinedSetting( PREDEF_SETTING_SIMPLE_PADDLE_ADAPTER_WITH_SEPARATE_PORT_FOR_KEYING );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Predef_IO_Simple_Paddle_Adapter_with_Keying_on_RTSClick(TObject *Sender)
{
  RecallPrefinedSetting( PREDEF_SETTING_SIMPLE_PADDLE_ADAPTER_WITH_KEYING_OUTPUT_ON_RTS );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Predef_IO_N1MM_Footswitch_CompatibleClick(TObject *Sender)
{
  RecallPrefinedSetting( PREDEF_SETTING_N1MM_FOOTSWITCH_COMPATIBLE_KEYING_ADAPTER );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::MI_Predef_IO_Neither_Paddle_Adapter_nor_Radio_Keying_PortClick(TObject *Sender)
{
  RecallPrefinedSetting( PREDEF_SETTING_NEITHER_PADDLE_NOR_RADIO_KEYING_ADAPTER );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Ed_VFOEnter(TObject *Sender) // "OnEnter"-event for the MAIN VFO (edit field)
{ // Since Microsoft's *.hlp is defunct, C++Builder 6's integrated help
  // doesn't help anywhere. So here's the purpose of "OnEnter" / "OnExit" :
  // > When the application focus moves from one to another control
  // > or the application itself to a control, the OnEnter event is fired.
  // > When the control looses focus, an OnExit event is fired.
  KeyerGUI_OnSetFocus( KEYER_GUI_ITEM_EDIT_MAIN_VFO/*iNewFocusedItem*/ );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::CB_OpModeEnter(TObject *Sender)
{ KeyerGUI_OnSetFocus( KEYER_GUI_ITEM_COMBO_MAIN_OP_MODE/*iNewFocusedItem*/ );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::CB_BandEnter(TObject *Sender)
{ KeyerGUI_OnSetFocus( KEYER_GUI_ITEM_COMBO_BAND_SWITCH/*iNewFocusedItem*/ );
  // '--> reloads KeyerGUI_iFocusSwitchCountdownTimer_ms for a few seconds.
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Btn_ApplyAndStartEnter(TObject *Sender)
{ KeyerGUI_OnSetFocus( KEYER_GUI_ITEM_BTN_APPLY_AND_START/*iNewFocusedItem*/ );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::TrackBar_WPMEnter(TObject *Sender)
{ KeyerGUI_OnSetFocus( KEYER_GUI_ITEM_SCR_KEYER_SPEED/*iNewFocusedItem*/ );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Ed_WPMEnter(TObject *Sender)
{ KeyerGUI_OnSetFocus( KEYER_GUI_ITEM_EDIT_KEYER_SPEED/*iNewFocusedItem*/ );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::SB_AudioInGain_dBEnter(TObject *Sender)
{ KeyerGUI_OnSetFocus( KEYER_GUI_ITEM_SCR_AUDIO_IN_VOLUME/*iNewFocusedItem*/ );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::SB_AudioOutGain_dBEnter(TObject *Sender)
{ KeyerGUI_OnSetFocus( KEYER_GUI_ITEM_SCR_AUDIO_OUT_VOLUME/*iNewFocusedItem*/ );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::SB_SidetoneGain_dBEnter(TObject *Sender)
{ KeyerGUI_OnSetFocus( KEYER_GUI_ITEM_SCR_SIDETONE_VOLUME/*iNewFocusedItem*/ );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::SB_NetworkTonesGain_dBEnter(TObject *Sender)
{ KeyerGUI_OnSetFocus( KEYER_GUI_ITEM_SCR_NETWORK_IN_VOLUME/*iNewFocusedItem*/ );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Rbtn_NetworkOffEnter(TObject *Sender)
{ KeyerGUI_OnSetFocus( KEYER_GUI_ITEM_RBTN_NETWORK_OFF/*iNewFocusedItem*/ );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Rbtn_NetworkClientEnter(TObject *Sender)
{ KeyerGUI_OnSetFocus( KEYER_GUI_ITEM_RBTN_NETWORK_CLIENT/*iNewFocusedItem*/ );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Rbtn_NetworkServerEnter(TObject *Sender)
{ KeyerGUI_OnSetFocus( KEYER_GUI_ITEM_RBTN_NETWORK_SERVER/*iNewFocusedItem*/ );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Ed_NetworkStatusEnter(TObject *Sender)
{ KeyerGUI_OnSetFocus( KEYER_GUI_ITEM_EDIT_NETWORK_STATUS/*iNewFocusedItem*/ );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Ed_OnTheKeyNowEnter(TObject *Sender)
{ KeyerGUI_OnSetFocus( KEYER_GUI_ITEM_EDIT_NETWORK_ON_KEY/*iNewFocusedItem*/ );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Ed_ErrorHistoryEnter(TObject *Sender)
{ KeyerGUI_OnSetFocus( KEYER_GUI_ITEM_EDIT_ERROR_HISTORY/*iNewFocusedItem*/ );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Ed_Mem1Enter(TObject *Sender)
{ KeyerGUI_OnSetFocus( KEYER_GUI_ITEM_EDIT_KEYER_MEMORY_1/*iNewFocusedItem*/ );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Ed_Mem2Enter(TObject *Sender)
{ KeyerGUI_OnSetFocus( KEYER_GUI_ITEM_EDIT_KEYER_MEMORY_1/*iNewFocusedItem*/ );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Ed_Mem3Enter(TObject *Sender)
{ KeyerGUI_OnSetFocus( KEYER_GUI_ITEM_EDIT_KEYER_MEMORY_1/*iNewFocusedItem*/ );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Ed_Mem4Enter(TObject *Sender)
{ KeyerGUI_OnSetFocus( KEYER_GUI_ITEM_EDIT_KEYER_MEMORY_1/*iNewFocusedItem*/ );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Ed_Mem5Enter(TObject *Sender)
{ KeyerGUI_OnSetFocus( KEYER_GUI_ITEM_EDIT_KEYER_MEMORY_1/*iNewFocusedItem*/ );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Ed_Mem6Enter(TObject *Sender)
{ KeyerGUI_OnSetFocus( KEYER_GUI_ITEM_EDIT_KEYER_MEMORY_1/*iNewFocusedItem*/ );
}

//---------------------------------------------------------------------------
void __fastcall TKeyerMainForm::Ed_MyCallEnter(TObject *Sender)
{ KeyerGUI_OnSetFocus( KEYER_GUI_ITEM_EDIT_MYCALL/*iNewFocusedItem*/ );
}

//---------------------------------------------------------------------------



