//---------------------------------------------------------------------------
// File: C:\cbproj\Remote_CW_Keyer\Keyer_GUI.h
// Date: 2024-11-29
// Author: Wolfgang Buescher (DL4YHF)
// Purpose and revision history: See implementation, Keyer_GUI.cpp .
//---------------------------------------------------------------------------

#ifndef _KEYER_GUI_NO_VCL_H_
# include "Keyer_GUI_no_VCL.h" // .. formerly part of Keyer_GUI.h, but moved
           // into that extra header, includeable from nice old "C" modules WITHOUT VCL DEPENDENCIES
#endif // ndef _KEYER_GUI_NO_VCL_H_


// Constants (as #defines) ..

#define EVT_MOUSE_DOWN 1 // .. for parameter iEvent in e.g. KeyerGUI_HandleMouseEventInTimingScope()
#define EVT_MOUSE_MOVE 2
#define EVT_MOUSE_UP   3


#define BANDLIST_BAND_COLUM_WIDTH  6  // .. measured in CHARACTERS, for the multi-column "frequency band list"
#define BANDLIST_FREQ_COLUMN_WIDTH 14 // worst case: " 1296.2345 USB"


// Variables .. (besides the VCL-free stuff in Keyer_GUI_no_VCL.h) :

extern HINSTANCE APPL_hInstance; // handle to current instance, a Win32 API thing

// Colours for various GUI components that we'd like to control via KeyerGUI_SetColourScheme():
// (but convincing Windows and the VCL to actually USE these colours turned into
//  a nightmare; especially for stuff in the 'Non-Client Area', e.g. title and main menu)
extern TColor g_clMenuBackground, g_clMenuSelBackgnd, g_clMenuForeground;
extern TColor g_clDecoderOutputForeground, g_clDecoderOutputBackground;
extern int    g_iNewLanguage;
#define UI_MODE_SIMPLE   0 // only show the basic elements, menu items, etc in the UI
#define UI_MODE_COMPLETE 1 // show everything, including the stuff for debugging, SDR-like rig control, etc
extern int    g_iUserInterfaceMode;
extern BOOL   g_fHideMenu; // automatically hide the old main menu ? FALSE=no, TRUE=yes

extern volatile DWORD g_dwDebugMessage,  // kluge for debugging / dummy assignments for breakpoints..
               g_dwDebugWParam, g_dwDebugLParam;
extern volatile int   g_iDebugDummyIndex;

typedef struct t_BandComboInfoTableEntry // .. for KeyerGUI_BandComboInfoTable[row][column] :
{
  double dblFmin_Hz, dblFmax_Hz; // only for colum 0 : radio frequency range of this band or sub-band
                             // Added 2024-12 to automatically switch the selected item (row)
                             // when "tuning around the band", and leaving / entering bands.

  int iBandStackingRegIndex; // 0..2 : 'this entry references a BAND STACKING REGISTER',
                             //        so the new frequency AND MODE are in pRC->BandStackingRegs[].
                             //  < 0 when the entry is NOT from a "band stacking register" (a la Icom).
      // A "band stacking register" (a la Icom, IC-7300, IC-9700 and the like)
      // contains MUCH, MUCH MORE than you can imagine in your wildest dreams.
      // So we only REFERENCE it from the Keyer GUI's 2-dimensional "Band selection combo",
      // and leave the rest to the 'RigControl' module.  See definition of
      //    T_RigCtrlFreqMemEntry BandStackingRegs[RIGCTRL_NUM_BAND_STACKING_REGS]
      // in C:\cbproj\Remote_CW_Keyer\RigControl.h .. and even THAT MONSTER
      // doesn't contain everything that e.g. an IC-9700 had to offer (for D-Star).
  int iUserDefinedBandIndex; // if 0..RIGCTRL_NUM_USER_DEFINED_BANDS-1,
              // this entry in the two-dimensional "Band" combo
              //  references one of the USER DEFINED BANDS
              //  in T_RigCtrlInstance.UserDefinedBands[iUserDefinedBandIndex].
} T_BandComboInfoTableEntry; // -> KeyerGUI_BandComboInfoTable[KEYER_GUI_MAX_BANDS][KEYER_GUI_MAX_COLUMNS_PER_BAND]
#define KEYER_GUI_MAX_BANDS            32 // .. in KeyerGUI_BandComboInfoTable[][] ..
#define KEYER_GUI_MAX_COLUMNS_PER_BAND 3  // .. in KeyerGUI_BandComboInfoTable[][] ..
extern T_BandComboInfoTableEntry KeyerGUI_BandComboInfoTable[KEYER_GUI_MAX_BANDS][KEYER_GUI_MAX_COLUMNS_PER_BAND];
           // '--> filled in KeyerGUI_FillComboWithBandList(), used in CB_BandClick()

extern int KeyerGUI_iUpdating;  // recursive flag to avoid certain event handlers in the GUI from interfering,
                                // if the 'changed' is caused by PROGRAMMATICALLY updating anything in the GUI.

extern T_TIM_Stopwatch KeyerGUI_swTxFromKeyboard; // stopwatch to switch back from KEYER_MEMORY_INDEX_TX_EDITOR to KEYER_MEMORY_INDEX_NONE after some time of inactivity

extern char KeyerGUI_sz80TextForRxTxInfo[84]; // text to display in the edit field for 'Info or Rx / Tx Data'

extern char KeyerGUI_sz80StatusIndicatorText[84];


typedef struct t_ErrorFifoEntry // T_ErrorFifoEntry is used for a lock-free FIFO between ShowError() and the GUI (main thread)
{
  char  sz511Text[512];
  int   iErrorClass;     // "error class" -> background colour in the RichText display
} T_ErrorFifoEntry;
# define C_ERROR_FIFO_SIZE 512

//----------------------------------------------------------------------------
// Global variables for hardcore debugging / post-mortem crash analysis.
//        Useful when the debugger's call stack shows nothing but garbage,
//        e.g. after an exception at the infamous address 0xFEEEFEEE .
//        ( C++Builder's debugger could still inspect SIMPLE GLOBAL
//          VARIABLES after most kinds of exceptions, but of course nothing
//          stack-based because the CPU- or task-stack was usually trashed.)
//----------------------------------------------------------------------------
#if(SWI_HARDCORE_DEBUGGING) // (1) = hardcore-debugging, (0)=normal compilation
extern int GUI_iLastSourceLine; // WATCH THIS after crashing with e.g. "0xFEEEFEEE"  ...
# define HERE_I_AM__GUI()  GUI_iLastSourceLine=__LINE__
# if( 0 )
    If certain worker threads of the application don't seem to terminate
    "politely", watch the following (complete list only HERE, in Keyer_Main.cpp):
      GUI_iLastSourceLine     DSW_iLastSourceLine      DSP_iLastSourceLine
      Keyer_iLastSourceLine   CwNet_iLastSourceLine    HttpSrv_iLastSourceLine
      RigCtrl_iLastSourceLine ..(?)
#  endif // (0) .. kludge to inspect the above variables in Borland via mouse-over ...
  extern int DSW_iLastSourceLine, DSP_iLastSourceLine, Keyer_iLastSourceLine,
             CwNet_iLastSourceLine, HttpSrv_iLastSourceLine;
  CPROT void CheckSystemHealth(const char *pszModuleName, int iSourceLine); // <- periodically called via macro CHECK_SYSTEM_HEALTH()
#else
# define HERE_I_AM__GUI()      /* rien */
#endif // SWI_HARDCORE_DEBUGGING ?


extern T_CwNet MyCwNet; // <- not a shiny C++ class, but a single instance representing the "CW Network"
extern DSoundWrapper MyDirectSound; // a "DirectSound wrapper" instance, first used for the SIDETONE OUTPUT

#if( SWI_USE_HAMLIB_SERVER ) // build an application with integrated "Hamlib Net rigctld"-compatible server ?
extern T_HLSrv MyHamlibServer; // Hamlib-'rigctld'-compatible server (guess the 'd' means 'daemon',
       // but neither demons nor daemons lurking in here. There's NOTHING running 'in the background'.)
#endif // SWI_USE_HAMLIB_SERVER ?


extern T_ErrorFifoEntry ErrorHistoryFifo[ C_ERROR_FIFO_SIZE ]; // <- classic lock-free circular FIFO between ShowError() and the GUI
extern int DEBUG_iErrorHistoryHead;
extern int DEBUG_iErrorHistoryTail;

typedef struct t_TimingScopeOverlay
{ int x1, y1, x2, y2; // mouse pointer coordinates at "mouse down" and "mouse up"
  int t1_ms;          // relative time in milliseconds (0=left edge) at "mouse down"
  int t2_ms;          // relative time in milliseconds (0=left edge) at "mouse up"
  BOOL visible;
} T_TimingScopeOverlay;
#define KEYER_GUI_N_SCOPE_OVERLAYS 3 // allow showing up to THREE of these overlays simultaneously
extern T_TimingScopeOverlay TimingScopeOverlay[KEYER_GUI_N_SCOPE_OVERLAYS];
extern int TimingScope_iCurrentOverlay;
extern int TimingScope_iUpdateCountOnTestTab, TimingScope_iUpdateCountOnKeyerTab;
           // '-- No prefix "KeyerGUI_" here because the "Timing Scope" was
           //     recycled from another project, and it should be obvious that
           //     a "Timing Oscilloscope" is part of the graphic user interface

extern T_SpecDispControl g_SpecDispControl;

extern const T_SL_TokenList C_TL_None_Zero[]; // list with only one item: "NONE".
extern const T_SL_TokenList SerialPortOutputSignals_Key[]; // "DTR", "RTS", optionally inverted
extern const T_SL_TokenList SerialPortOutputSignals_Radio[]; // similar as above, but on the "RADIO port"
extern const T_SL_TokenList SerialPortInputSignals_Key[]; // "DCD", "DSR", "CTS" and "RI"
extern const T_SL_TokenList SerialPortInputSignals_Radio[]; // similar as above, but on the "RADIO port"
extern const T_SL_TokenList SerialPortBaudrates[]; // .. in bit/second, 1200 .. 115200
extern const T_SL_TokenList RigControlMethods[]; // "No rig control", "CI-V", and who-knows-what
extern const T_SL_TokenList KeyTypes[]; // "None", "Straight", "Paddle", .. 
extern const T_SL_TokenList SidetonesOnTXD[]; // Sidetone frequencies available on a serial port's TXD(!) output
extern const T_SL_TokenList SidetonesOnAudioOut[]; // Sidetones sent to an audio device (with higher latency than "TXD sidetone"):
extern const T_SL_TokenList SidetoneRiseTimes[]; // "Rise"-, aka "Ramp"-times for sidetone VIA AUDIO OUTPUT
extern const T_SL_TokenList AuxComPortUsages[];        // .. for CwKeyer_Config.iAuxComPort1Usage, etc
extern const T_SL_TokenList AuxComPortTunnelNumbers[]; // .. for CwKeyer_Config.iAuxComPortTunnelIndex[]

extern const char *CwKeyer_psz6MemoryDefaults[6];


//---------------------------------------------------------------------------
// Helper functions ( no shiny VCL class methods ! )
//    Details only in the implementation (Keyer_GUI.cpp) .
//---------------------------------------------------------------------------

void ParseCommandLine(const char *pszCmdLine);
BOOL LoadSettings( // .. from an old-fashioned *.ini file, NOT from the bloody registry !
           const char *pszIniFileName, int iConfigFileOptions );
void SaveSettings( // .. in an old-fashioned *.ini file, NOT in the bloody registry !
           const char *pszIniFileName, int iConfigFileOptions );
  // Bitwise combineable flags for 'iConfigFileOptions' for LoadSettings() + SaveSettings():
#define CONFIG_FILE_OPTION_USER_SETTINGS        0x0001
#define CONFIG_FILE_OPTION_MACHINE_SETTINGS     0x0002 // e.g. COM port numbers, audio device names, etc
#define CONFIG_FILE_OPTION_RECENT_FILES         0x0004 // history of 'recently exported/imported configurations'
#define CONFIG_FILE_OPTION_INCLUDE_ALL          0xFFFF
#define CONFIG_FILE_OPTION_ALL_BUT_RECENT_FILES (~CONFIG_FILE_OPTION_RECENT_FILES)
void AddToRecentFiles(const char *pszConfigFileName);
void ClearRecentFiles(void);
const char *GetRecentFileName(int iZeroBasedIndex);


DWORD CalcHashForSettings(void); // .. to tell if we need to SAVE SETTINGS on exit
extern "C" void ShowError( int iErrorClass, const char * pszFormat, ... );
  // ,---------------------------'
  // '--> ERROR_CLASS_FATAL, ERROR_CLASS_ERROR, ERROR_CLASS_WARNING, ERROR_CLASS_INFO .
  //      If an optional 'debug run log' (file) is supported,
  //      iErrorClass bitmask SHOW_ERROR_IN_RUN_LOG can be used to emit
  //      the same message in "debug_run_log.txt", besides the GUI's "Debug" tab.
  //

//---------------------------------------------------------------------------
// Functions doomed to use Borland-specific data types like "TComboBox", etc:
//---------------------------------------------------------------------------

BOOL FillComboWithSerialPorts( TComboBox *pCombo, int iSelectedComPortNr,
           BOOL fNeedValidPort, BOOL *pfConflict, const char *pszDefaultHint);
BOOL CheckForSignalsOnMorseKeyPort(void);
BOOL CheckForSignalsOnRadioKeyingPort(void);
BOOL CheckForSignalsOnSamePin( int *piSignalIndex ); // details only in the implementation (Keyer_GUI.cpp)
BOOL FillComboWithItemsFromTokenLists( TComboBox *pCombo,
        const T_SL_TokenList *pTokens1, // [in] e.g. C_TL_None_Zero (with string = "None", value = zero)
        const T_SL_TokenList *pTokens2, // [in] e.g. SerialPortInputSignals_Key
        const T_SL_TokenList *pTokens3, // [in] e.g. SerialPortInputSignals_Radio, or NULL for only *one* list
        int iSelectedTokenValue  );     // [in] e.g. [-]KEYER_SIGNAL_INDEX_MORSE_KEY_CTS


int AppendRadiosFromRigControlToCombo( TComboBox *pCombo ); // --, kind of..
int GetCIVAddressFromComboBox( TComboBox *pCombo );         // --' "inverse functions" !

int GetSelectedTokenValueFromComboBox( TComboBox *pCombo,
        const T_SL_TokenList *pTokens1, // [in] e.g. C_TL_None_Zero (with string = "None", value = zero)
        const T_SL_TokenList *pTokens2, // [in] e.g. SerialPortInputSignals_Key
        const T_SL_TokenList *pTokens3); // [in] e.g. SerialPortInputSignals_Radio, or NULL for only *one* list
  // If the selected text in the combo list doesn't match ANY of the
  // strings in the three token lists, this function returns
  // GUI_ERROR_NO_MATCHING_ENTRY to let THE CALLER decide what to use.
BOOL SelectComboItemByDecimalValue( TComboBox *pCombo, int iValue );
BOOL SelectComboItemByHexadecimalValue( TComboBox *pCombo, int iValue );
int  GetSerialPortSignalIndexFromComboBox( TComboBox *pCombo );
int  ParseSerialPortNumberFromText( const char *pszSource );

void KeyerGUI_FillComboWithBandList( T_RigCtrlInstance *pRC, TComboBox *pCombo );
int  KeyerGUI_FrequencyToBandComboRow( double dblFreq_Hz ); // -> iRow in KeyerGUI_BandComboInfoTable[iRow][iColumn]
double KeyerGUI_GetFrequencyFromEditField( TEdit *pEd, int iSelectedCharIndex, double *pdblFreqIncrement );
void KeyerGUI_UpdateTimingScope( T_KeyerTimingScope *pScope, Graphics::TBitmap *pDestBitmap);
BOOL KeyerGUI_HandleMouseEventInTimingScope( Graphics::TBitmap *pBitmap, int iEvent, int x, int y );
void KeyerGUI_ClearScopeOverlays(void);
void KeyerGUI_SetColourScheme( int iColourScheme );
int  KeyerGUI_ExpandTextToSend( const char **ppszOriginalText, int nCharsWanted, char *pszDest, const char *pszEndstop );
void KeyerGUI_UpdateStatusIndicatorText(void); // -> KeyerGUI_sz80StatusIndicatorText, etc
int  KeyerGUI_GetCharWidthInPixelsForRichEditControl( TRichEdit *pRichEdit );
void KeyerGUI_AppendDataToRichEditWithRxTxInfo( TRichEdit *pRichEdit, const char *pszText, int iRxTxInfoUsage);
int  KeyerGUI_TransferCharsFromEditFieldToCwGenerator( TRichEdit *pRichEdit, T_CwGen *pGenerator );
void KeyerGUI_UpdateColourOfSentCharsInRichEdit(TRichEdit *pRichEdit);
void KeyerGUI_GetColoursForRichEditWithRxTxInfo(int iRxTxInfoUsage,
                      DWORD *pdwTextColour, DWORD *pdwBkgndColour );
void KeyerGUI_ModifyColoursInRichEdit( TRichEdit *pRichEdit,
                      int iFromCharIndex, int nCharsToModify,
                      DWORD dwTextColor,  DWORD dwBkgndColor );
void KeyerGUI_ShowInfoInStatusLine( TRichEdit *pRichEdit, int iRxTxInfoUsage, const char *pszInfo );
BOOL KeyerGUI_OnKeyInTransmitTextEditor(TRichEdit *pRichEdit, char cKey );
BOOL KeyerGUI_OnKeyInErrorHistory( TRichEdit *pRichEdit, char cKey );

int  KeyerGUI_ExecuteCommand( const char *pszCommand, char *pszResponse, int iMaxResponseLength );
BOOL KeyerGUI_DumpErrorHistoryToRichText(TRichEdit *pRichEdit);
BOOL KeyerGUI_DumpCharsFromTerminalToRichText(TRichEdit *pRichEdit);
void KeyerGUI_ListRigParamsOnDebugTab(T_RigCtrlInstance *pRC, TRichEdit *pRichEdit );

void KeyerGUI_UpdateVFODisplay( TEdit *pED, double dblVfoFrequency_Hz );
BOOL KeyerGUI_HandleMouseEventInVfoFreqEditor( TEdit *pED, int iEvent, int x, int y );

void KeyerGUI_UpdateSMeter( TImage *pImg, double dblSMeterLevel_dB );
      // Keep it simple, begin at "S zero" :
#     define C_SMETER_LEVEL_S0   0 /* "S0" on HF:       -20 dBuV */
#     define C_SMETER_LEVEL_S9  54 /* "S9" on HF:       +34 dBuV */
#     define C_SMETER_LEVEL_MAX 99 /* "S9+45 dB"                 */
void KeyerGUI_UpdateMultiFunctionMeter( // renders any of the horizontal bargraphs on the "Mult-funtion meter" (panel)
        TImage *pImg,  // [in,out] Borland-VCL-style "graphic image", with a "canvas" to paint into
        int x1, int y1, int x2, int y2, // [in] graphic area for one of the six(?) "meters" within pImg
        double dblValue, // [in] value to be displayed (percent, "dB", SWR, current, voltage, power, temperature..)
        double dblMinValue, double dblMaxValue, double dblStepwidth, // [in] physical value range (also used for the labelled scale)
        const char *pszParamName, // [in] name like "Power", "ALC", "COMP", "SWR", "Id", "Vd", "Temp"
        const char *pszPhysUnit,  // [in] physical unit (shown at the end of the tick scale), e.g. "%", "dB", "V", "A", "W", "C"
        double dblRedStartValue, double dblRedEndValue ); // e.g. 10.0 .. 12.0 [Volts for the "red line" indicating UNDERVOLTAGE]

const char* RigCtrl_GetRadioControlPortAsString(void); // -> e.g. "COM5", if that's the port "talking to the radio" .
  // Called from within RigControl.c to emit 'info messages' like
  //     "RigControl: IC-9700 on COM5, 432250.0 kHz, CWN"  .
  // Since RighControl.c doesn't know anything about serial ports,
  // RigCtrl_GetRadioControlPortAsString() must be provided by the application.

#ifdef __BORLANDC__  // here the specific stuff for Borland / C++Builder / VCL :
TForm *KeyerGUI_GetMainForm(void);  // provides access to the application's MAIN FORM (without exposing its name)
#endif // __BORLANDC__ ?

/* EOF <Keyer_GUI.h> */