//---------------------------------------------------------------------------
// File: C:\cbproj\Remote_CW_Keyer\Keyer_GUI.cpp
// Date: 2025-09-16
// Author: Wolfgang Buescher (DL4YHF)
// Contains a part(!) of the Remote CW Keyer's graphic user interface,
//          with as few dependencies on the VCL-based 'main form' itself.
// Function overview:
//  * LoadSettings(), SaveSettings(), ShowError(),
//  * Dozens of functions to fill COMBO BOXES and SUBMENUS with items, e.g.:
//      KeyerGUI_FillComboWithBandList(),
//  * Functions to control the VFO on the GUI:
//      KeyerGUI_UpdateVFODisplay(), KeyerGUI_GetFrequencyFromEditField(),
//      KeyerGUI_HandleMouseEventInVfoFreqEditor(), ...
//  * KeyerGUI_UpdateTimingScope()
//        |-> plots analog and digital channels, decoded text, etc
//        |-> ScopeDisplay_DrawAudioSpectrumAndDecoderInfo()
//  * Functions related with the RichEdit control on the 'Debug' tab,
//      with a simplistic parser for a 'console'-like control .
//
// Many variables and functions were moved from Keyer_Main.cpp / *.h
// into Keyer_GUI.cpp / *.h, to avoid duplicating them in the *.cpp modules
//  of the 'main form' (TKeyerMainForm *KeyerMainForm), with one variant for
// Borland C++Builder V6, and another one for Embarcadero C++Builder V12 !
//
// Most recent modifications: See C:\cbproj\Remote_CW_Keyer\Keyer_GUI.cpp .
//  2024-11-25: First steps to migrate from Borland C++Builder V6
//              to Embarcadero C++Builder V12 "Athens", free Community Edition,
//              using the "Clang enhanced" compiler "bcc32c" (with "B"="Borland"?).
//              Details about migrating from C++Builder V6 to V12 are in file
//               Migrating_from_CppBuilder6_to_CppBuilder12.htm (blog style).
//              Abandoned this, because the compilers are just too incompatible,
//              and a REAL cross-platform "Embarcadero C++Builder V12"
//              (with targets at least Windows and Linux) was obscenely overpriced
//            - over 10000 Euros including tax for a new "Enterprise" edition
//              including one(!) whopping year of maintenance - no, thanks.
//              Go for Qt instead ?  But porting a VCL application to Qt
//              is an AWFUL lot of effort, and more or less means re-writing the GUI.
//

#include "switches.h"   // project specific compiler switches ("options"),
                        // must be included before anything else !
#include "yhf_type.h"   // classic types like BYTE, WORD, DWORD, BOOL, ..
#include <windows.h>    // Must be included BEFORE vcl.h for some strange reason.
                        // Contains stuff like WAVEINCAPS, WAVEOUTCAPS, etc^255 .
#include <vcl.h>        // Borland's stoneage Visual Component Library
#include <buttons.hpp>  // TBitBtn       a la Borland VCL (but why in an extra header?)
#include <ComCtrls.hpp> // TTabSheet     a la Borland VCL ...
#include <CheckLst.hpp> // TCheckListBox a la Borland VCL ....
#include <Grids.hpp>    // TStringGrid   a la Borland VCL .....
#include <string.h>
#include <stdio.h>      // not using "standard I/O" here, but e.g. sprintf()
#include <math.h>
#pragma hdrstop

#include "Utilities.h"    // stuff like UTL_iWindowsVersion, UTL_iAppInstance, ShowError(), etc
#include "YHF_Dialogs.h"  // API for a few common dialogs, e.g. YHF_RunStringEditDialog()
#include "YHF_Help.h"     // DL4YHF's HTML-based replacement for the defunct *.hlp system
#include "YHF_WinStuff.h" // Win32 API functions, helpers for 'Rich Edit' controls, etc
#include "HelpIDs.h"      // application specific help topic identifiers
#include "Translator.h"   // APPL_TranslateAllForms() [only works with Borland VCL]
#include "Translations.h" // tranlation table (static, built-in string table w/o whistles and bells)

#if( SWI_USE_DSOUND ) // use "Direct Sound" / dsound_wrapper.c ?
# include "dsound_wrapper.h"
#endif // SWI_USE_DSOUND ?

#if( SWI_USE_WAVE_AUDIO || SWI_USE_MIDI )
# include <mmsystem.h> // maybe MIDI is fast enough to start / stope playing notes ?
#endif // SWI_USE_WAVE_AUDIO or SWI_USE_MIDI ?

#include "StringLib.h"
#include "Elbug.h"   // old 'Elbug' functions, converted from PIC-assembler to "C"
#include "CwGen.h"   // CW generator (converts text to Morse code)
#include "CwDSP.h"   // CW-'Digital Signal Processor' / sidetone generator
#include "CwNet.h"   // Socket-based 'Client or Server' for the Remote CW Keyer. Tied to a "Rig Control" instance.
#include "CwKeyer.h" // prototypes for the 'worker thread' that stitches all together
#include "Inet_Tools.h" // Base64 encoding/decoding, SHA1 calculation, etc..
#if(SWI_USE_HTTP_SERVER) // build an application with integrated HTTP server ?
# include "HttpServer.h" // formerly simple HTTP server (turned into a monster)
# include "TIconToFavicon.h" // function to convert the application's icon
                             // into "Favicon.ico" for the web server .
#endif // SWI_USE_HTTP_SERVER ?

#include "SpecDisp.h"  // SpecDisp_UpdateSpectrum(), ..Waterfall(), FreqScale(),..
#include "Keyer_GUI.h" // subroutines formerly in Keyer_Main.cpp (now only CALLED from there)
#include "FreqList.h"  // import 'frequencies of interest' from e.g. the EiBi frequency list

#if( SWI_NUM_AUX_COM_PORTS > 0 ) // compile with support for 'Auxiliary / Additional COM ports' / Winkeyer emulation ?
# include "AuxComPorts.h" // structs and API functions for the "Auxiliary" (later: "Additional") COM ports
#endif // SWI_NUM_AUX_COM_PORTS ?

// Internal data types
typedef struct t_ScopeDisplay // structure for drawing the "timing scope" display
{
  T_RGBColor dwPenColour; // R-G-B colour mix (Red in bits 7..0, Green bits 15..8, Blue bits 23..16)
                     // *currently selected* into the "Graphic Device Context" (or whatever comes next).
                     // Compatible with the VCL type "TColor". Possibly a 'Windows thing'.
                     // Qt will roll its own thing ("QColor"), of course incompatible
                     // with each new major version ["setNamedColor() is deprecated
                     //  since 6.6. We strongly advise against using it in new code.]
#ifdef __BORLANDC__  // here the implementation for Borland/Embarcadero C++Builder/VCL :
  Graphics::TBitmap *pBitmap; // off-screen bitmap for flicker-free drawing
  TPoint *pPoints;   // an array for up to <TIMING_SCOPE_NUM_SAMPLE_POINTS> "points" for drawing e.g. curves as polygons
  int     nPoints;   // number of points acutually USED (valid) in the above array
  // Speed comparison plotting curves with different Win32 GDI functions (or their VCL wrappers):
  //   Plotting each curve in a single call of TCanvas.Polyline() : 2.5 % CPU load from the "Remote CW keyer"
  //   Plotting curves with MoveTo() / hundreds of LineTo() calls : 7.5 % CPU load  "    "     "     "  "
  //   Not plotting anything (only running the 'keyer thread')    : 1.3 % CPU load  "    "     "     "  "
  // -> Worth the effort (to reduce the CPU load) !
#else // ndef __BORLANDC__, ndef who-knows-what-else ...
# error "You will have a lot of fun porting this to e.g. Qt !"
#endif // __BORLANDC__ or what will be the cool stuff a few years from now ?
  T_RGBColor **ppdwPixelLines; // an array of 'pixel line pointers' (into the bitmap), with one DWORD per pixel
  int     nPixelLines; // number of entries in the above array (same as the height of the bitmap in pixels)

  // ex: T_RGBColor dwChannelColours[TIMING_SCOPE_N_DIGITAL_CHANNELS+TIMING_SCOPE_N_ANALOG_CHANNELS];
  //      '--> now part of T_KeyerTimingScope.cfg ("public"), along with
  //           a lot of layout parameters prepared in KeyerGUI_UpdateTimingScope(),
  //           required by the MOUSE EVENT HANDLER in KeyerGUI_HandleMouseEventInTimingScope() .

} T_ScopeDisplay;



HINSTANCE APPL_hInstance= 0; // handle to current instance, a Win32 API thing
BOOL APPL_fLaunchKeyerOnStart = FALSE;
BOOL APPL_fPausedDuringSleep  = FALSE;
BOOL APPL_fSaveSettingsOnExit = FALSE;

char g_sz255CommandLine[256]     = "";
char g_sz255PathToDataFiles[256] = "";
char g_sz255RecentFiles[GUI_RECENT_FILES_HISTORY_LENGTH][256];

// Colours for various GUI components that we'd like to control via KeyerGUI_SetColourScheme():
// (but convincing Windows and the VCL to actually USE these colours turned into
//  a nightmare; especially for stuff in the 'Non-Client Area', e.g. title and main menu)
TColor g_clMenuBackground, g_clMenuSelBackgnd, g_clMenuForeground;
TColor g_clDecoderOutputForeground, g_clDecoderOutputBackground;
int    g_iNewLanguage = TRANSLATOR_LANGUAGE_ENGLISH;
int    g_iUserInterfaceMode = UI_MODE_COMPLETE;
BOOL   g_fHideMenu; // automatically hide the old main menu ? FALSE=no, TRUE=yes

volatile DWORD g_dwDebugMessage,  // kluge for debugging / dummy assignments for breakpoints..
               g_dwDebugWParam, g_dwDebugLParam;
volatile int   g_iDebugDummyIndex;

int KeyerGUI_iUpdating = 0; // recursive flag to avoid certain event handlers in the GUI from interfering,
                            // if the 'changed' is caused by PROGRAMMATICALLY updating anything in the GUI.

DWORD KeyerGUI_dwTimerTicks_50ms = 0;  // <- Incremented every 50 milliseonds. Individual bits are used as e.g. 'blink flag' anywhere in the GUI
BOOL  KeyerGUI_fMustApplyConfigAndStart = FALSE; // flag that lets the "Apply new settings and start"-button on any of the CONFIG tabs *blink*
BOOL  KeyerGUI_fMaySwitchToVfoEditField = FALSE; // flag to allow switching BACK to the VFO edit field (within the 'TRX' tab only)
int   KeyerGUI_iFocusSwitchCountdownTimer_ms = 0; // countdown timer, in milliseconds, for 'automatic switching of the keyboard focus' back to the VFO FIELD
int   KeyerGUI_iCurrentlyFocusedItem = KEYER_GUI_ITEM_NONE; // used in combination with m_iFocusSwitchCountdownTimer. Contains KEYER_GUI_ITEM_...
        // Some keyboard-focus-switching actions were implemented here without VCL/Qt/whatever-dependencies.
        // See Keyer_GUI.cpp : KeyerGUI_OnSetFocus( iNewFocusedItem = KEYER_GUI_ITEM_EDIT_MAIN_VFO, etc ).
int   KeyerGUI_iCurrentMainTab = KEYER_GUI_MAIN_TAB_UNKNOWN;
BOOL  KeyerGUI_fAutomaticTabSwitching = FALSE; // normally FALSE = "tabs switched by the USER";  TRUE to allow e.g. switching from KEYER_GUI_MAIN_TAB_DEBUG to KEYER_GUI_MAIN_TAB_TRX when the 'rig initialisation' is finished.
BOOL  KeyerGUI_fAutomaticFocusSwitching= TRUE; // normally TRUE = "allow automatic switching of the input focus" (e.g. back to the VFO edit field)


T_BandComboInfoTableEntry KeyerGUI_BandComboInfoTable[KEYER_GUI_MAX_BANDS][KEYER_GUI_MAX_COLUMNS_PER_BAND];
           // '--> filled in KeyerGUI_FillComboWithBandList(), used in CB_BandClick(),
           //      and queried via KeyerGUI_FrequencyToBandComboRow() .
int  KeyerGUI_nRowsInBandComboInfoTable = 0; // 0 .. KEYER_GUI_MAX_BANDS-1, set in KeyerGUI_FillComboWithBandList()
int  KeyerGUI_iStatusIndicatorUsage  = STATUS_INDICATOR_OFF;
int  KeyerGUI_iStatusIndicatorMemIdx = 0; // also for the 'Status Indicator'; one of the following constants:
           //  KEYER_MEMORY_INDEX_1         : "sending from Memory #1 (the one activated by function key F1)"
           //  KEYER_MEMORY_INDEX_2         : "sending from Memory #2 (the one activated by function key F2)"
           //  KEYER_MEMORY_INDEX_3         : "sending from Memory #3 (the one activated by function key F3)"
           //  KEYER_MEMORY_INDEX_4         : "sending from Memory #4 (the one activated by function key F4)"
           //  KEYER_MEMORY_INDEX_5         : "sending from Memory #5 (the one activated by function key F5)"
           //  KEYER_MEMORY_INDEX_6         : "sending from Memory #6 (the one activated by function key F6)"
           //  KEYER_MEMORY_INDEX_TX_EDITOR : "sending directly from RichEdit_RxTxInfo"
           //  KEYER_MEMORY_INDEX_WINKEYER_EMU : "sending CW from the Winkeyer-EMULATOR" (possibly fooling an external application like N1MM Logger)
           //  KEYER_MEMORY_INDEX_WINKEYER_HOST: "sending CW from the Winkeyer-HOST" (and an EXTERNAL Winkeyer-chip on an 'Additional COM Port'
           //  KEYER_MEMORY_INDEX_NONE      : "neither sending from TX memory 1..6 nor from the 'direct text input field' in the status line"
           //  Note: KeyerGUI_iStatusIndicatorMemIdx may also be modified
           //        from WORKER THREADS, like AuxComThread() when EMULATING A WINKEYER !
           //        Whoever sets is LAST, wins...

T_TIM_Stopwatch KeyerGUI_swTxFromKeyboard;  // stopwatch to switch back from KEYER_MEMORY_INDEX_TX_EDITOR to KEYER_MEMORY_INDEX_NONE after some time of inactivity

char KeyerGUI_sz80StatusIndicatorText[84];  // text to display on the 'Status' panel
int  KeyerGUI_iConflictsFromSettingsTab = 0; // bitwise combination, e.g. KEYER_GUI_CONFLICT_MORSE_KEY_PORT | KEYER_GUI_CONFLICT_RADIO_CONTROL_PORT
int  KeyerGUI_iRxTxInfoUsage = KEYER_GUI_RXTXINFO_OFF; // <- applies to e.g. RichEdit_RxTxInfo, values:  KEYER_GUI_RXTXINFO_OFF, KEYER_GUI_RXTXINFO_TYPING, KEYER_GUI_RXTXINFO_RXDATA, KEYER_GUI_RXTXINFO_SENT_DATA, etc...
long KeyerGUI_i32CountdownForRxTxInfo_ms = 0; // "countdown timer", in milliseconds, for e.g. RichEdit_RxTxInfo (the "multi-purpose" edit- and display field on the 'status panel')
           // As long as KeyerGUI_i32CountdownForRxTxInfo_ms has not been decremented to zero,
           // the GUI won't switch from a higher- to a lower priority display
           // (that's the purpose of KeyerGUI_iRxTxInfoUsage - see KeyerGUI_SetInfoText() )
char KeyerGUI_sz80TextForRxTxInfo[84]; // text to display in the edit field for 'Info or Rx / Tx Data'

     // Zero-based character indices for the Transmit-Text-Editor (aka 'type-ahead buffer').
     // Decremented when characters are REMOVED on the begin of the 'scrolling' text,
     // and incremented e.g. when characters are passed from the edit control
     // to the CW generator (in that moment, they cannot be edited anymore, etc).
     // Most of these KeyerGUI_iTxEditorCharIndex_..variables are also used
     // for changing the background colour of characters in the TRichEdit .
     // Details in KeyerGUI_UpdateColourOfSentCharsInRichEdit() :
int  KeyerGUI_iTxEditorCharIndex_CwGenerator= 0; // index of the next character to move from the type-ahead buffer into the CW Generator
int  KeyerGUI_iTxEditorCharIndex_StartOfMsg = 0; // index of the FIRST character to send from the type-ahead input field

BOOL KeyerGUI_fUpdateWPMIndicator = FALSE; // May be set by ANY THREAD after modifying CwKeyer_Elbug.cfg.iDotTime_ms .
            // Requests a call of UpdateWPMIndicatorFromCwKeyerConfig() from where it's safe to call the VCL :o)

BOOL KeyerGUI_fSerialTerminalActive=FALSE; // global, def'd in Keyer Keyer_GUI_no_VCL.h: TRUE=active, FALSE="display only"


T_CwNet MyCwNet; // <- not a shiny C++ class, but a single instance representing the "CW Network". Owns a "Rig Control" instance, too.
DSoundWrapper MyDirectSound; // a "DirectSound wrapper" instance, first used for the SIDETONE OUTPUT

#if( SWI_USE_HAMLIB_SERVER ) // build an application with integrated "Hamlib Net rigctld"-compatible server ?
T_HLSrv MyHamlibServer; // Hamlib-'rigctld'-compatible server (guess the 'd' means 'daemon',
       // but neither demons nor daemons lurking in here. There's NOTHING running 'in the background'.)
#endif // SWI_USE_HAMLIB_SERVER ?


T_ErrorFifoEntry ErrorHistoryFifo[ C_ERROR_FIFO_SIZE ]; // <- classic lock-free circular FIFO between ShowError() and the GUI
int DEBUG_iErrorHistoryHead = 0;
int DEBUG_iErrorHistoryTail = 0;
T_TimingScopeOverlay TimingScopeOverlay[KEYER_GUI_N_SCOPE_OVERLAYS];
int TimingScope_iCurrentOverlay = 0;
int TimingScope_iUpdateCountOnTestTab, TimingScope_iUpdateCountOnKeyerTab;

T_SpecDispControl g_SpecDispControl = { 0 };

// Internal function prototypes (no shiny C++ class methods!)
BOOL ScopeDisplay_Init(T_ScopeDisplay *pScopeDisplay, int iWidth, int iHeight);
void ScopeDisplay_Free(T_ScopeDisplay *pScopeDisplay); // Cleans up after ScopeDisplay_Init()


//----------------------------------------------------------------------------
// Global variables for hardcore debugging / post-mortem crash analysis.
//        Useful when the debugger's call stack shows nothing but garbage,
//        e.g. after an exception at the infamous address 0xFEEEFEEE .
//        ( C++Builder's debugger could still inspect SIMPLE GLOBAL
//          VARIABLES after most kinds of exceptions, but of course nothing
//          stack-based because the CPU- or task-stack was usually trashed.)
//----------------------------------------------------------------------------
#if(SWI_HARDCORE_DEBUGGING) // (1) = hardcore-debugging, (0)=normal compilation
 int GUI_iLastSourceLine = 0; // WATCH THIS after crashing with e.g. "0xFEEEFEEE"  ...
# define HERE_I_AM__GUI()  GUI_iLastSourceLine=__LINE__
# if( 0 )
    If certain worker threads of the application don't seem to terminate
    "politely", watch the following (complete list only HERE, in Keyer_Main.cpp):
     * GUI_iLastSourceLine
     * DSW_iLastSourceLine
     * DSP_iLastSourceLine
     * Keyer_iLastSourceLine
     * CwNet_iLastSourceLine
     * HttpSrv_iLastSourceLine
     * RigCtrl_iLastSourceLine
     * AuxCom_iLastSourceLine
     * Chatbox_iLastSourceLine
#  endif // (0) .. kludge to inspect the above variables in Borland via mouse-over ...
  extern int DSW_iLastSourceLine, DSP_iLastSourceLine, Keyer_iLastSourceLine,
             CwNet_iLastSourceLine, HttpSrv_iLastSourceLine;
  CPROT void CheckSystemHealth(const char *pszModuleName, int iSourceLine); // <- periodically called via macro CHECK_SYSTEM_HEALTH()
  static  const char *s_ModuleName = "KeyerMain";
# define CHECK_SYSTEM_HEALTH() CheckSystemHealth(s_ModuleName,__LINE__)
#else
# define HERE_I_AM__GUI()      /* rien */
# define CHECK_SYSTEM_HEALTH() /* nada */
#endif // SWI_HARDCORE_DEBUGGING ?

const T_SL_TokenList C_TL_None_Zero[] =
{ { "NONE",  0 },
  { NULL, 0 } // "all zeros" mark the end of the list
};

const T_SL_TokenList SerialPortOutputSignals_Key[] =
{ { "DTR (DE-9 pin 4) on KEYER port",            KEYER_SIGNAL_INDEX_MORSE_KEY_DTR },
  { "DTR (DE-9 pin 4) on KEYER port, INVERTED", -KEYER_SIGNAL_INDEX_MORSE_KEY_DTR },
  { "RTS (DE-9 pin 7) on KEYER port",            KEYER_SIGNAL_INDEX_MORSE_KEY_RTS },
  { "RTS (DE-9 pin 7) on KEYER port, INVERTED", -KEYER_SIGNAL_INDEX_MORSE_KEY_RTS },
  { NULL, 0 } // "all zeros" mark the end of the list
};

const T_SL_TokenList SerialPortOutputSignals_Radio[] =
{ { "DTR (DE-9 pin 4) on RADIO port",            KEYER_SIGNAL_INDEX_RADIO_KEYING_DTR },
  { "DTR (DE-9 pin 4) on RADIO port, INVERTED", -KEYER_SIGNAL_INDEX_RADIO_KEYING_DTR },
  { "RTS (DE-9 pin 7) on RADIO port",            KEYER_SIGNAL_INDEX_RADIO_KEYING_RTS },
  { "RTS (DE-9 pin 7) on RADIO port, INVERTED", -KEYER_SIGNAL_INDEX_RADIO_KEYING_RTS },
  { NULL, 0 } // "all zeros" mark the end of the list
};

const T_SL_TokenList SerialPortInputSignals_Key[] =
{ { "DCD (DE-9 pin 1) on KEYER port",            KEYER_SIGNAL_INDEX_MORSE_KEY_DCD },
  { "DCD (DE-9 pin 1) on KEYER port, INVERTED", -KEYER_SIGNAL_INDEX_MORSE_KEY_DCD },
  { "DSR (DE-9 pin 6) on KEYER port",            KEYER_SIGNAL_INDEX_MORSE_KEY_DSR },
  { "DSR (DE-9 pin 6) on KEYER port, INVERTED", -KEYER_SIGNAL_INDEX_MORSE_KEY_DSR },
  { "CTS (DE-9 pin 8) on KEYER port",            KEYER_SIGNAL_INDEX_MORSE_KEY_CTS },
  { "CTS (DE-9 pin 8) on KEYER port, INVERTED", -KEYER_SIGNAL_INDEX_MORSE_KEY_CTS },
  { "RI  (DE-9 pin 9) on KEYER port",            KEYER_SIGNAL_INDEX_MORSE_KEY_RI  },
  { "RI  (DE-9 pin 9) on KEYER port, INVERTED", -KEYER_SIGNAL_INDEX_MORSE_KEY_RI  },
  { NULL, 0 } // "all zeros" mark the end of the list
};

const T_SL_TokenList SerialPortInputSignals_Radio[] =
{ { "DCD (DE-9 pin 1) on RADIO port",            KEYER_SIGNAL_INDEX_RADIO_KEYING_DCD },
  { "DCD (DE-9 pin 1) on RADIO port, INVERTED", -KEYER_SIGNAL_INDEX_RADIO_KEYING_DCD },
  { "DSR (DE-9 pin 6) on RADIO port",            KEYER_SIGNAL_INDEX_RADIO_KEYING_DSR },
  { "DSR (DE-9 pin 6) on RADIO port, INVERTED", -KEYER_SIGNAL_INDEX_RADIO_KEYING_DSR },
  { "CTS (DE-9 pin 8) on RADIO port",            KEYER_SIGNAL_INDEX_RADIO_KEYING_CTS },
  { "CTS (DE-9 pin 8) on RADIO port, INVERTED", -KEYER_SIGNAL_INDEX_RADIO_KEYING_CTS },
  { "RI  (DE-9 pin 9) on RADIO port",            KEYER_SIGNAL_INDEX_RADIO_KEYING_RI  },
  { "RI  (DE-9 pin 9) on RADIO port, INVERTED", -KEYER_SIGNAL_INDEX_RADIO_KEYING_RI  },
  { NULL, 0 } // "all zeros" mark the end of the list
};


const T_SL_TokenList SerialPortBaudrates[] =
{ { "1200",    1200 }, // <- important for Winkeyer[2,USB] !
  { "2400",    2400 },
  { "4800",    4800 },
  { "9600",    9600 },
  { "19200",  19200 },
  { "38400",  38400 },
  { "115200",115200 }, // <- important for modern Icom radios (with USB) !
  { NULL, 0 } // "all zeros" mark the end of the list
};

const T_SL_TokenList RigControlMethods[] =
{ { "No active rig control, only CW keying (or client)", RIGCTRL_PROTOCOL_NONE },
  { "Icom CI-V, using the 'COM port to key the radio'", RIGCTRL_PROTOCOL_ICOM_CI_V },
#if(SWI_SUPPORT_YAESU_5_BYTE_CAT) // Support the OLD UGLY "YAESU 5-BYTE-CAT" (see KA7OEI, "FT817 Meow") ?
  { "Yaesu FT8x7-CAT, via 'COM port to key the radio'", RIGCTRL_PROTOCOL_YAESU_5_BYTE },
#endif
  { "Hamlib Net rigctl via remote TCP/IP server", RIGCTRL_PROTOCOL_HAMLIB_RIGCTLD  },
  { NULL, 0 } // "all zeros" mark the end of the list
};

const T_SL_TokenList AuxComPortUsages[] = // .. for CwKeyer_Config.iAuxComPort1Usage[]
{ { "No function assigned",   RIGCTRL_PORT_USAGE_NONE                 },
  { "Winkeyer HOST",          RIGCTRL_PORT_USAGE_WINKEYER_HOST        },
  { "Winkeyer EMULATOR",      RIGCTRL_PORT_USAGE_WINKEYER_EMULATOR    },
  { "Text Terminal",          RIGCTRL_PORT_USAGE_TEXT_TERMINAL        },
  { "Serial Port Tunnel",     RIGCTRL_PORT_USAGE_SERIAL_TUNNEL        },
  { "Virtual Rig",            RIGCTRL_PORT_USAGE_VIRTUAL_RIG          },
  { "Echo Test",              RIGCTRL_PORT_USAGE_ECHO_TEST            },
  { "TX Stress Test",         RIGCTRL_PORT_USAGE_TX_STRESS_TEST       },

  { NULL, 0 } // "all zeros" mark the end of the list
};

const T_SL_TokenList AuxComPortTunnelNumbers[] = // .. for CwKeyer_Config.iAuxComPortTunnelIndex[]
{ { "ANY (connect to ANY Client/Server tunnel)",   -1 },
  { "1 (only connect to ports on tunnel #1)",  0 }, // note the ZERO-BASED array indices (internally)
  { "2 (only connect to ports on tunnel #2)",  1 },
  { "3 (only connect to ports on tunnel #3)",  2 },
  { NULL, 0 } // "all zeros" mark the end of the list
};


const T_SL_TokenList KeyTypes[] =
{ { "None (no CW key)", KEY_TYPE_PASSIVE  },
  { "Straight",         KEY_TYPE_STRAIGHT },
  { "Paddle, IAMBIC MODE B", KEY_TYPE_IAMBIC_B },
  { "Paddle, IAMBIC MODE A", KEY_TYPE_IAMBIC_A },
  { "Basic iambic w/o dash/dot memory", KEY_TYPE_IAMBIC_NO_MEM},
  { NULL, 0 } // "all zeros" mark the end of the list
};

const T_SL_TokenList SidetonesOnTXD[] =  // Sidetone tapped from a serial port's TXD(!) output:
{ { "None",   KEYER_SIDETONE_NONE       },
  { "480 Hz", KEYER_SIDETONE_TXD_480_HZ }, // 4800 "baud" divided by 10 bits (standard baudrate)
  { "960 Hz", KEYER_SIDETONE_TXD_960_HZ }, // also 4800 "baud" but with TWO pulses in the TXD pattern
  { NULL, 0 } // "all zeros" mark the end of the list
};

const T_SL_TokenList SidetonesOnAudioOut[] =  // Sidetones sent to an audio device (with higher latency than "TXD sidetone"):
{ { "None",   0   },
  { "400 Hz", 400 },
  { "450 Hz", 450 }, // ~~ Icom's name "CW Pitch" (but here, it's only the LOCAL KEYER SIDETONE)
  { "500 Hz", 450 },
  { "550 Hz", 550 },
  { "600 Hz", 600 },
  { "650 Hz", 650 },
  { "700 Hz", 700 },
  { "750 Hz", 750 },
  { "800 Hz", 800 },
  { "1000 Hz",1000},
  { NULL, 0 } // "all zeros" mark the end of the list
};

const T_SL_TokenList SidetoneRiseTimes[] =  // "Rise"-, aka "Ramp"-times for sidetone VIA AUDIO OUTPUT
{ { "0 ms (hard-keyed)", 0 },
  { "2 ms",  2 },
  { "4 ms",  4 },
  { "6 ms",  6 },
  { "8 ms",  8 },
  { "20 ms (very soft)",20 }, // huuh-huh, thought Mr Bubo Bubo, and flew away when he heard this
  { NULL, 0 } // "all zeros" mark the end of the list
};


const char *CwKeyer_psz6MemoryDefaults[6] =
{ "cq cq cq de <mycall> <mycall> <mycall> ^ar pse k",
  "qrz ? pse agn de <mycall> k",
  "<mycall>",
  "<mycall> testing Remote CW Keyer, sending from memory. Operator possibly not on the key.",
  "",
  ""
};


const T_SL_TokenList KeyerGUI_VisualComponentNames[] =
{ { "None",              KEYER_GUI_ITEM_NONE                 },
  { "Menu_File",         KEYER_GUI_ITEM_MAIN_MENU_FILE       },
  { "Menu_Settings",     KEYER_GUI_ITEM_MAIN_MENU_SETTINGS   },
  { "Menu_Functions",    KEYER_GUI_ITEM_MAIN_MENU_FUNCTIONS  },
  { "Menu_Help",         KEYER_GUI_ITEM_MAIN_MENU_HELP       },
  { "Btn_ApplyAndStart", KEYER_GUI_ITEM_BTN_APPLY_AND_START  },
  { "TrackBar_WPM",      KEYER_GUI_ITEM_SCR_KEYER_SPEED      },
  { "Ed_WPM",            KEYER_GUI_ITEM_EDIT_KEYER_SPEED     },
  { "SB_AudioInGain_dB", KEYER_GUI_ITEM_SCR_AUDIO_IN_VOLUME  },
  { "SB_AudioOutGain_dB",KEYER_GUI_ITEM_SCR_AUDIO_OUT_VOLUME },
  { "SB_SidetoneGain_dB",KEYER_GUI_ITEM_SCR_SIDETONE_VOLUME  },
  { "SB_NetworkTonesGain_dB", KEYER_GUI_ITEM_SCR_NETWORK_IN_VOLUME },
  { "Rbtn_NetworkOff",   KEYER_GUI_ITEM_RBTN_NETWORK_OFF     },
  { "Rbtn_NetworkClient",KEYER_GUI_ITEM_RBTN_NETWORK_CLIENT  },
  { "Rbtn_NetworkServer",KEYER_GUI_ITEM_RBTN_NETWORK_SERVER  },
  { "Ed_NetworkStatus",  KEYER_GUI_ITEM_EDIT_NETWORK_STATUS  },
  { "Ed_OnTheKeyNow",    KEYER_GUI_ITEM_EDIT_NETWORK_ON_KEY  },
  { "Ed_ErrorHistory",   KEYER_GUI_ITEM_EDIT_ERROR_HISTORY   },
  { "Ed_Mem1",           KEYER_GUI_ITEM_EDIT_KEYER_MEMORY_1  },
  { "Ed_Mem2",           KEYER_GUI_ITEM_EDIT_KEYER_MEMORY_2  },
  { "Ed_Mem3",           KEYER_GUI_ITEM_EDIT_KEYER_MEMORY_3  },
  { "Ed_Mem4",           KEYER_GUI_ITEM_EDIT_KEYER_MEMORY_4  },
  { "Ed_Mem5",           KEYER_GUI_ITEM_EDIT_KEYER_MEMORY_5  },
  { "Ed_Mem6",           KEYER_GUI_ITEM_EDIT_KEYER_MEMORY_6  },
  { "Ed_MyCall",         KEYER_GUI_ITEM_EDIT_MYCALL          },
  { "Ed_VFO",            KEYER_GUI_ITEM_EDIT_MAIN_VFO        },
  { "CB_OpMode",         KEYER_GUI_ITEM_COMBO_MAIN_OP_MODE   },
  { "CB_Band",           KEYER_GUI_ITEM_COMBO_BAND_SWITCH    },
  //  '---> Many of these names are compatible with the "Visual Components"
  //        in C:\cbproj\Remote_CW_Keyer\Keyer_Main.dfm .. but don't take that for granted.
  { NULL, 0 } // "all zeros" mark the end of the list
}; // end KeyerGUI_VisualComponentNames[]



//---------------------------------------------------------------------------
// Helper functions ( no shiny VCL class methods ! )
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
void ParseCommandLine(const char *pszCmdLine)
  // Some examples (copy & paste to C++Builder IDE: Run..Parameters)
{
  const char *cp = pszCmdLine;
  const char *cp2;
  char *cp3;
  char c;
  char sz255Temp[256], *pszDest=sz255Temp, *pszEndstop=sz255Temp+255;

  if( *cp=='"' ) // guess this is the name of the executable -> grab 'our path' from it
   { cp2 = cp+1;
     SL_SkipDoubleQuotedString( &cp );  // <- returns the NUMBER OF CHARS SKIPPED in cp, *including* the double quotes
   }
  else  // Name of the executable NOT in double quotes (thanks for the snag..)
   { cp2 = cp;
     SL_SkipCharsUntilDelimiter( &cp, " ", SL_SKIP_NORMAL );
   }

  // One of the 'nice surprises' from Windoze is that the first parameter
  // on the command line sometimes does NOT contain a full path,
  // To avoid having the configuration files, the "debug run log",
  // and who-knows-what somewhere in the dark where Microsoft
  // wants to have them ( e.g. C:\Users\WerHatsInstalliert\AppData\Local | LocalLow | Roaming | Weiss-der-Teufel ..),
  // extract the full path to the EXECUTABLE (which is where all other files
  // have been unzipped to, without a shiny installer at all) :
  //
  if( ! UTL_GetPathToMyDataFiles( g_sz255PathToDataFiles, 255 ) )
   { // SOS .. unable to find out where "our files" should be for this user,
     //        on this windows version ..
     cp3 = strrchr( g_sz255PathToDataFiles, '\\' );
     if( cp3==NULL )  // quite unlikely that Microsoft uses POSIX one day, but... :
      {  cp3 = strrchr( g_sz255PathToDataFiles, '/' );
      }
     if(cp3!=NULL)  // if there's any kind of slash, TRUNCATE AFTER THAT CHAR..
      { cp3[1] = '\0'; // -> result e.g. g_sz255PathToDataFiles = "C:\\cbproj\Remote_CW_Keyer\\"
      }
   }
  if( ((c=SL_GetLastChar(g_sz255PathToDataFiles,255)) != '\\') && (c!='/') )
   { cp3 = g_sz255PathToDataFiles;
     SL_AppendChar( &cp3, cp2+255, '\\' );
   }

  // After the "first command line argument" (name of the executable), parse the rest:
  while( *cp != '\0' ) // repeat until end-of-string
   { SL_SkipSpaces( &cp );
     switch( *cp )
      { case '"' :  // another double-quoted string ? -> throw it away
           if( SL_SkipDoubleQuotedString( &cp ) <= 0 )
            { return;  // something wrong with the syntax -> bail out
            }
           break;
        case '\\' :
        case '/'  : // forward or backward slash : Guess this is an "option"..
           ++cp;
           if( SL_SkipToken( &cp, "debug" ) ) // run this program with a "debug log" :
            { pszDest=sz255Temp;
              SL_AppendString( &pszDest, pszEndstop, g_sz255PathToDataFiles );
              SL_AppendString( &pszDest, pszEndstop, "debug_run_log.txt" );
              UTL_OpenRunLogFile( sz255Temp );
            }
           else if( SL_SkipToken( &cp, "xlate" ) ) // run this program in "translation test mode" ..
            { APPL_iLanguageTestMode = 1; // details in Remote_CW_Keyer\Translator.cpp !
            }
           break;
        default:
           return;  // something wrong with the syntax -> bail out
      }
   } // end while < more characters or substrings in the command line >

} // end ParseCommandLine()


//---------------------------------------------------------------------------
BOOL LoadSettings(const char *pszIniFileName, // .. from an old-fashioned *.ini file, NOT from the bloody registry !
                  int iConfigFileOptions ) // [in] bitwise combination of CONFIG_FILE_OPTION_... def'd in Keyer_GUI.h
  // [out] CwKeyer_Config , and maybe A FEW more things for the GUI itself.
  // [in]  const char *pszIniFileName : e.g. SWI_APP_EXE_NAME".ini" = "RemoteCwKeyer.ini" .
  //   The full path MAY be C:\Windows\RemoteCwKeyer.ini ,
  //       but also seen :
  //   C:\Users\Wolf\AppData\Local\VirtualStore\Windows\RemoteCwKeyer.ini
  //   - seems to depend on FROM WHERE, and with which privileges,
  //     and by which USER the application was launched. Omg.
  //       Thus, when called on program start, pszIniFileName is only a
  //       FILENAME WITHOUT PATH, to avoid trouble with the Windows UAC or whatever.
  // [return] TRUE if the specified file seems to be a valid config file
  //          for this application. The result is used in the GUI:
  //          When called from "Import configuration", the filename is only
  //          added to the 'Recent Files' history when some content is recognized.
{
  char szSec[84];  // "section" (in the ini file, that's the string in squared brackets)
  char szKey[84];  // "key name" (that's the string on the left side of the assignment operator)
  char sz255Value[256];
  const char *pszSrc;
  TEdit *pEdit;
  int   i;
#if( SWI_NUM_AUX_COM_PORTS > 0 )
  T_AuxComConfig *pAuxComCfg;
#endif

  // To find out if this *.ini file is "valid", check the [About] section:
  UTL_ReadStringFromIniFile( pszIniFileName, "About"/*section*/, "1"/*key*/,
                             ""/*default*/, sz255Value/*dest*/, 255/*maxlen*/);
  pszSrc = sz255Value;  // parse the VALUE after [About] .. "1=Configuration .."
  if( ! SL_SkipToken( &pszSrc,"Configuration") ) // ..file for the 'Remote CW Keyer'.
   { // this is not a valid "Configuration" file, so bail out here:
     return FALSE;
   }

  sprintf( szSec, "Keyer%d", 1+UTL_iAppInstance ); // e.g. "Keyer1" for the FIRST running instance, etc
  CwKeyer_Config.iComPortNumber_IN = UTL_ReadIntFromIniFile(pszIniFileName,
      szSec, "ComPort_IN"/*pszKeyName*/, CwKeyer_Config.iComPortNumber_IN/*default*/ );
  CwKeyer_Config.iRadioKeyingAndControlPort = UTL_ReadIntFromIniFile(pszIniFileName,
      szSec, "RadioControlAndKeyingPort", CwKeyer_Config.iRadioKeyingAndControlPort );
  CwKeyer_Config.iRadioControlProtocol = UTL_ReadIntFromIniFile(pszIniFileName,
      szSec, "RadioControlProtocol", CwKeyer_Config.iRadioControlProtocol );
  CwKeyer_Config.iRadioControlBaudrate = UTL_ReadIntFromIniFile(pszIniFileName,
      szSec, "RadioControlBaudrate", CwKeyer_Config.iRadioControlBaudrate );
  CwKeyer_Config.iRadioCIVAddress = UTL_ReadIntFromIniFile(pszIniFileName,
      szSec, "RadioCIVAddress", CwKeyer_Config.iRadioCIVAddress );
  UTL_ReadStringFromIniFile( pszIniFileName, szSec, "RemoteRigCtrlServerAddress",
                             CwKeyer_Config.sz80RemoteRigCtrlServerAddress/*Default*/,
                             CwKeyer_Config.sz80RemoteRigCtrlServerAddress/*Dest*/,
                             80/*max length*/ );
#if( SWI_USE_HAMLIB_SERVER ) // not sure if this 'feature' is going to stay...
  CwKeyer_Config.fEnableBuiltInHamlibServer = UTL_ReadIntFromIniFile(pszIniFileName,
      szSec, "EnableBuiltInHamlibServer",CwKeyer_Config.fEnableBuiltInHamlibServer);
  MyHamlibServer.cfg.iServerListeningPort = UTL_ReadIntFromIniFile(pszIniFileName,
      szSec, "BuiltInHamlibServerPort", MyHamlibServer.cfg.iServerListeningPort);
  MyHamlibServer.cfg.iServerOptions = UTL_ReadIntFromIniFile(pszIniFileName,
      szSec, "BuiltInHamlibServerOptions", MyHamlibServer.cfg.iServerOptions);
  MyHamlibServer.cfg.iDiagnosticFlags= UTL_ReadIntFromIniFile(pszIniFileName,
      szSec, "BuiltInHamlibServerDiagnostics", MyHamlibServer.cfg.iDiagnosticFlags);
#endif // SWI_USE_HAMLIB_SERVER ?

  CwKeyer_Config.iMorseKeyType=UTL_ReadIntFromIniFile(pszIniFileName,szSec, "MorseKeyType",
      CwKeyer_Config.iMorseKeyType/* KEY_TYPE_PASSIVE/STRAIGHT/PADDLE */ );
  CwKeyer_Config.iDotInput  = UTL_ReadIntFromIniFile(pszIniFileName,szSec,"DotInputSignal", CwKeyer_Config.iDotInput );
  CwKeyer_Config.iDashInput = UTL_ReadIntFromIniFile(pszIniFileName,szSec,"DashInputSignal",CwKeyer_Config.iDashInput );
  CwKeyer_Config.iManualPTTInput=UTL_ReadIntFromIniFile(pszIniFileName,szSec,"ManualPTTInput",CwKeyer_Config.iManualPTTInput);
  CwKeyer_Config.iTestInput = UTL_ReadIntFromIniFile(pszIniFileName,szSec,"TestInputSignal",CwKeyer_Config.iTestInput );
  CwKeyer_Config.iKeySupply = UTL_ReadIntFromIniFile(pszIniFileName,szSec,"KeySupplySignal",CwKeyer_Config.iKeySupply );
  CwKeyer_Config.iRadioSupply=UTL_ReadIntFromIniFile(pszIniFileName,szSec,"RadioSupplySignal",CwKeyer_Config.iRadioSupply );
  CwKeyer_Config.iSidetoneOnTXD  = UTL_ReadIntFromIniFile(pszIniFileName,szSec,"SidetoneOnTXD", CwKeyer_Config.iSidetoneOnTXD );
  CwKeyer_Config.iRadioCWKeying  = UTL_ReadIntFromIniFile(pszIniFileName,szSec,"CwKeyingSignal",CwKeyer_Config.iRadioCWKeying );
  CwKeyer_Config.iRadioPTTControl= UTL_ReadIntFromIniFile(pszIniFileName,szSec,"PTTControlSignal",CwKeyer_Config.iRadioPTTControl);
  CwKeyer_Config.iTxDelayTime_ms = UTL_ReadIntFromIniFile(pszIniFileName,szSec,"TxDelayTime_ms",CwKeyer_Config.iTxDelayTime_ms);
  CwKeyer_Config.iTxHangTime_ms  = UTL_ReadIntFromIniFile(pszIniFileName,szSec,"TxHangTime_ms",CwKeyer_Config.iTxHangTime_ms);
  CwKeyer_Config.fDebouncePaddleInputs=UTL_ReadIntFromIniFile(pszIniFileName,szSec,"DebouncePaddleInputs",CwKeyer_Config.fDebouncePaddleInputs);
  CwKeyer_Config.fKeyInputWatchdog=UTL_ReadIntFromIniFile(pszIniFileName,szSec,"KeyInputWatchdog",CwKeyer_Config.fKeyInputWatchdog);
  CwKeyer_Config.fKeyViaShiftAndControl = UTL_ReadIntFromIniFile(pszIniFileName,szSec,"CwViaShiftAndControlKey",
    ( CwKeyer_Config.iComPortNumber_IN < 0) ); // <- per default, only enable this "gadget" if there's no serial port for the Morse Key
  CwKeyer_Config.fDisableTx = UTL_ReadIntFromIniFile(pszIniFileName,szSec,"DisableRealTransmit", FALSE );
  CwKeyer_Config.dwAudioOptions=(DWORD)UTL_ReadIntFromIniFile(pszIniFileName,szSec, "AudioOptions",(long)CwKeyer_Config.dwAudioOptions);

#if( SWI_NUM_AUX_COM_PORTS > 0 )
  for(i=0; i<SWI_NUM_AUX_COM_PORTS; ++i)
   { pAuxComCfg = &CwKeyer_Config.sAuxCom[i];
     sprintf( szKey, "AuxComPort%dNumber", (int)(i+1) );
     pAuxComCfg->iPortNumber = UTL_ReadIntFromIniFile(pszIniFileName,szSec,szKey, -1 );
     sprintf( szKey, "AuxComPort%dUsage",  (int)(i+1) );
     pAuxComCfg->iPortUsage = UTL_ReadIntFromIniFile(pszIniFileName,szSec,szKey, RIGCTRL_PORT_USAGE_NONE );
     sprintf( szKey, "AuxComPort%dTunnel", (int)(i+1) );
     pAuxComCfg->iTunnelIndex = UTL_ReadIntFromIniFile(pszIniFileName,szSec,szKey, -1 );
     sprintf( szKey, "AuxCom%dBitsPerSecond", (int)(i+1) );
     pAuxComCfg->iBitsPerSecond = UTL_ReadIntFromIniFile(pszIniFileName,szSec,szKey, 1200 );
     sprintf( szKey, "AuxCom%dNumDatabits", (int)(i+1) );
     pAuxComCfg->iNumDatabits = UTL_ReadIntFromIniFile(pszIniFileName,szSec,szKey, 8 );
     sprintf( szKey, "AuxCom%dParity", (int)(i+1) );
     pAuxComCfg->iParity = UTL_ReadIntFromIniFile(pszIniFileName,szSec,szKey, 0 );
     sprintf( szKey, "AuxCom%dNumStopbits", (int)(i+1) );
     pAuxComCfg->iNumStopbits = UTL_ReadIntFromIniFile(pszIniFileName,szSec,szKey, 1 );
     sprintf( szKey, "AuxCom%dUsageDTR", (int)(i+1) ); // freely programmable DIGITAL OUTPUT
     UTL_ReadStringFromIniFile( pszIniFileName,szSec,szKey,"", pAuxComCfg->sz40DTR,40 );
     sprintf( szKey, "AuxCom%dUsageRTS", (int)(i+1) ); // freely programmable DIGITAL OUTPUT
     UTL_ReadStringFromIniFile( pszIniFileName,szSec,szKey,"", pAuxComCfg->sz40RTS,40 );
     sprintf( szKey, "AuxCom%dUsageDCD", (int)(i+1) ); // freely programmable DIGITAL INPUT
     UTL_ReadStringFromIniFile( pszIniFileName,szSec,szKey,"", pAuxComCfg->sz40DCD,40 );
     sprintf( szKey, "AuxCom%dUsageDSR", (int)(i+1) ); // freely programmable DIGITAL INPUT
     UTL_ReadStringFromIniFile( pszIniFileName,szSec,szKey,"", pAuxComCfg->sz40DSR,40 );
     sprintf( szKey, "AuxCom%dUsageCTS", (int)(i+1) ); // freely programmable DIGITAL INPUT
     UTL_ReadStringFromIniFile( pszIniFileName,szSec,szKey,"", pAuxComCfg->sz40CTS,40 );
     sprintf( szKey, "AuxCom%dUsageRI",  (int)(i+1) ); // freely programmable DIGITAL INPUT
     UTL_ReadStringFromIniFile( pszIniFileName,szSec,szKey,"", pAuxComCfg->sz40RI,40 );
     sprintf( szKey, "AuxCom%dParams",(int)(i+1) ); // e.g. hint to decode received traffic, CAT-protocol-specific parameters, etc
     UTL_ReadStringFromIniFile( pszIniFileName,szSec,szKey,"", pAuxComCfg->sz40Params,40 );
   }
  AuxCom_iDiagnosticFlags = UTL_ReadIntFromIniFile(pszIniFileName,szSec,"AuxComDiagnosticFlags",
                             AuxCom_iDiagnosticFlags );
#endif // SWI_NUM_AUX_COM_PORTS > 0 ?

  UTL_ReadStringFromIniFile( pszIniFileName, szSec, "AudioInputDevice",
                             CwKeyer_DSP.cfg.sz255AudioInputDevice/*Default*/,
                             CwKeyer_DSP.cfg.sz255AudioInputDevice, 255 );
  UTL_ReadStringFromIniFile( pszIniFileName, szSec, "AudioOutputDevice",
                             CwKeyer_DSP.cfg.sz255AudioOutputDevice/*Default*/,
                             CwKeyer_DSP.cfg.sz255AudioOutputDevice, 255 );
  CwKeyer_DSP.cfg.iSidetoneFreq_Hz = UTL_ReadIntFromIniFile(pszIniFileName,szSec,
         "AudioSidetone_Hz", 650 + 50 * UTL_iAppInstance/*gadget: try running THREE instances..*/ );
  CwKeyer_DSP.cfg.iSidetoneRiseTime_ms = UTL_ReadIntFromIniFile(pszIniFileName,szSec,
         "SidetoneRiseTime_ms", CwKeyer_DSP.cfg.iSidetoneRiseTime_ms );
  CwKeyer_DSP.cfg.iSidetoneGain_dB = UTL_ReadIntFromIniFile(pszIniFileName,szSec,
         "SidetoneGain_dB", CwKeyer_DSP.cfg.iSidetoneGain_dB );
  CwKeyer_DSP.cfg.iAudioInGain_dB = UTL_ReadIntFromIniFile(pszIniFileName,szSec,
         "AudioInGain_dB", CwKeyer_DSP.cfg.iAudioInGain_dB );
  CwKeyer_DSP.cfg.iAudioOutGain_dB = UTL_ReadIntFromIniFile(pszIniFileName,szSec,
         "AudioOutGain_dB", CwKeyer_DSP.cfg.iAudioOutGain_dB );
  CwKeyer_DSP.cfg.iNetworkTonesGain_dB = UTL_ReadIntFromIniFile(pszIniFileName,szSec,
         "NetworkTonesGain_dB", CwKeyer_DSP.cfg.iNetworkTonesGain_dB );

  CwKeyer_DSP.cfg.iAudioFlags = UTL_ReadIntFromIniFile(pszIniFileName, szSec,
          "AudioFlags"/*pszKeyName*/, CwKeyer_DSP.cfg.iAudioFlags/*default*/ );

  CwKeyer_Elbug.cfg.iDotTime_ms = Elbug_WordsPerMinuteToDotTimeInMilliseconds(UTL_ReadIntFromIniFile(pszIniFileName,szSec,"WordsPerMinute",25) );
  CwKeyer_Gen.cfg.iDotTime_ms = CwKeyer_Elbug.cfg.iDotTime_ms; // let 'playback' use the same speed as direct keyed Morse
  CwKeyer_DSP.cfg.iCwDecoderDotTime_ms = CwKeyer_Elbug.cfg.iDotTime_ms; // same also as DEFAULT for the Audio-CW-Decoder in CwDSP.c

  APPL_fLaunchKeyerOnStart  = UTL_ReadIntFromIniFile(pszIniFileName,szSec,"LaunchOnStart", APPL_fLaunchKeyerOnStart );
  CwKeyer_TimingScope.cfg.iChannel4Source = UTL_ReadIntFromIniFile(pszIniFileName,szSec,"Channel4Source", CwKeyer_TimingScope.cfg.iChannel4Source );
  CwKeyer_TimingScope.cfg.iVisibleChannels= UTL_ReadIntFromIniFile(pszIniFileName,szSec,"TScopeChannels", CwKeyer_TimingScope.cfg.iVisibleChannels);
  CwKeyer_TimingScope.cfg.iMillisecondsPerSample=UTL_ReadIntFromIniFile(pszIniFileName,szSec,"TScopeMsPerSample", CwKeyer_TimingScope.cfg.iMillisecondsPerSample);
  CwKeyer_TimingScope.cfg.iTriggerOptions = UTL_ReadIntFromIniFile(pszIniFileName,szSec,"TScopeTrigger",  CwKeyer_TimingScope.cfg.iTriggerOptions );
  CwKeyer_TimingScope.cfg.iAudioSpectrumRefLevel_dB  = UTL_ReadIntFromIniFile(pszIniFileName,szSec,"AudioSpectrumRefLevel_dB", CwKeyer_TimingScope.cfg.iAudioSpectrumRefLevel_dB );
  CwKeyer_TimingScope.cfg.iAudioSpectrumAmplRange_dB = UTL_ReadIntFromIniFile(pszIniFileName,szSec,"AudioSpectrumAmpleRange_dB", CwKeyer_TimingScope.cfg.iAudioSpectrumAmplRange_dB );
  for(i=0; i<(TIMING_SCOPE_N_DIGITAL_CHANNELS + TIMING_SCOPE_N_ANALOG_CHANNELS); ++i)
   { sprintf( szKey, "TScopeVertChannelPos%d", (int)(i+1) );
     CwKeyer_TimingScope.cfg.iChannelVerticalPos_pcnt[i] =
       UTL_ReadIntFromIniFile(pszIniFileName,szSec,szKey,
       10/*percent*/ + i * 90 / (TIMING_SCOPE_N_DIGITAL_CHANNELS + TIMING_SCOPE_N_ANALOG_CHANNELS) );
   }

  // Still in the ini-file-section "KeyerN", where "N" is the instance index...
  CwKeyer_Config.iTRXOptions = UTL_ReadIntFromIniFile(pszIniFileName,
       szSec, "TRXOptions"/*pszKeyName*/, CwKeyer_Config.iTRXOptions/*default*/ );
       // -> bitwise combineable flag like KEYER_TRX_OPT_KEEP_RUNNING, etc,
       //    all defined for struct T_CwKeyerConfig in Remote_CW_Keyer/CwKeyer.h .

  // Load options for the built-in "Rig Control" module :
  sprintf( szSec, "RigControl%d", 1+UTL_iAppInstance ); // e.g. "RigControl1" for the FIRST running instance, etc
  CwKeyer_Config.dwMessageFilterForLog = (DWORD)UTL_ReadIntFromIniFile(pszIniFileName,szSec,"RejectMessagesInLog",
                             CwKeyer_Config.dwMessageFilterForLog/*Default*/ );
  CwKeyer_Config.fTurnRigOffWhenClosing = UTL_ReadIntFromIniFile(pszIniFileName,szSec,"TurnRigOffWhenClosing", FALSE );
  RigCtrl_TrafficMonitor.iDisplayOptions = UTL_ReadIntFromIniFile(pszIniFileName,szSec,
          "TrafficMonitorOptions", RigCtrl_TrafficMonitor.iDisplayOptions );
  RigCtrl_TrafficMonitor.iTimestampFormat= UTL_ReadIntFromIniFile(pszIniFileName,szSec,
          "TrafficMonitorTimestampFormat", RigCtrl_TrafficMonitor.iTimestampFormat );
  RigCtrl_TrafficMonitor.iExtraColumns = UTL_ReadIntFromIniFile(pszIniFileName,szSec,
          "TrafficMonitorExtraColumns", RigCtrl_TrafficMonitor.iExtraColumns );



  // Load the "Network" configuration (for module CwNet.c; simple TCP-client or server):
  sprintf( szSec, "Network%d", 1+UTL_iAppInstance ); // e.g. "Network1" for the FIRST running instance, etc
  MyCwNet.cfg.iFunctionality = UTL_ReadIntFromIniFile(pszIniFileName, szSec, "Functionality",  MyCwNet.cfg.iFunctionality );
  MyCwNet.cfg.iDiagnosticFlags= UTL_ReadIntFromIniFile(pszIniFileName,szSec,"DiagnosticFlags", 0 );
  sprintf( szSec, "Network%dClient", 1+UTL_iAppInstance ); // e.g. "Network1Client" for the FIRST running instance, etc
  UTL_ReadStringFromIniFile( pszIniFileName, szSec, "RemoteServerIP",
                             MyCwNet.cfg.sz80ClientRemoteIP/*Default*/,
                             MyCwNet.cfg.sz80ClientRemoteIP, 80 );
  UTL_ReadStringFromIniFile( pszIniFileName, szSec, "UserName",
                             MyCwNet.cfg.sz80ClientUserName/*Default*/,
                             MyCwNet.cfg.sz80ClientUserName, 80 );
  UTL_ReadStringFromIniFile( pszIniFileName, szSec, "Callsign",
                             MyCwNet.cfg.sz80ClientCallsign/*Default*/,
                             MyCwNet.cfg.sz80ClientCallsign, 80 );

  sprintf( szSec, "Network%dServer", 1+UTL_iAppInstance ); // e.g. "Network1Server" for the FIRST running instance, etc
  MyCwNet.cfg.iServerListeningPort = UTL_ReadIntFromIniFile(pszIniFileName,szSec,
                             "ListeningPort", MyCwNet.cfg.iServerListeningPort );
  UTL_ReadStringFromIniFile( pszIniFileName, szSec, "AcceptUsers",
                             MyCwNet.cfg.sz255AcceptUsers/*Default*/,
                             MyCwNet.cfg.sz255AcceptUsers, 255 );
  UTL_ReadStringFromIniFile( pszIniFileName, szSec, "Pwd",
                             MyCwNet.cfg.sz20ServerAdminPWD/*Default*/,
                             MyCwNet.cfg.sz20ServerAdminPWD, 20 );
  MyCwNet.cfg.iHttpServerOptions = UTL_ReadIntFromIniFile(pszIniFileName,szSec,
                             "HttpOptions", MyCwNet.cfg.iHttpServerOptions );
  MyCwNet.cfg.iNetworkLatency_ms = UTL_ReadIntFromIniFile(pszIniFileName,szSec,
                             "Latency_ms",  MyCwNet.cfg.iNetworkLatency_ms );

  sprintf( szSec, "Keyer%dMainWindow", 1+UTL_iAppInstance ); // e.g. "Keyer1MainWindow" for the FIRST running instance, etc
  CwKeyer_Config.iMainWindowLeft = UTL_ReadIntFromIniFile(pszIniFileName,
       szSec, "Left"/*pszKeyName*/, 50 + 100 * UTL_iAppInstance/*default*/ );
  CwKeyer_Config.iMainWindowTop = UTL_ReadIntFromIniFile(pszIniFileName,
       szSec, "Top"/*pszKeyName*/,  100 + 30 * UTL_iAppInstance/*default*/ );
  CwKeyer_Config.iMainWindowWidth = UTL_ReadIntFromIniFile(pszIniFileName, szSec, "Width", 600 );
  CwKeyer_Config.iMainWindowHeight= UTL_ReadIntFromIniFile(pszIniFileName, szSec, "Height",500 );
  g_SpecDispControl.iWFColorPalette=UTL_ReadIntFromIniFile(pszIniFileName, szSec, "WFColourPalette", WF_PALETTE_SUNRISE);
  g_SpecDispControl.iColourScheme = UTL_ReadIntFromIniFile(pszIniFileName, szSec, "ColourScheme", COLOUR_SCHEME_DARK );
  g_SpecDispControl.iDisplayOptions = UTL_ReadIntFromIniFile(pszIniFileName, szSec, "SpecDispOptions",SPEC_DISP_OPTIONS_TIME_MARKERS );
  g_SpecDispControl.iBottomPanelMode= UTL_ReadIntFromIniFile(pszIniFileName, szSec, "BottomPanelMode",SPEC_DISP_PANEL_MODE_OFF );
  g_iNewLanguage       = UTL_ReadIntFromIniFile(pszIniFileName, szSec, "Language", APPL_iLanguage );
  g_iUserInterfaceMode = UTL_ReadIntFromIniFile(pszIniFileName, szSec, "UserInterfaceMode", g_iUserInterfaceMode );
  g_fHideMenu          = UTL_ReadIntFromIniFile(pszIniFileName, szSec, "HideMainMenu", g_fHideMenu );
  KeyerGUI_fAutomaticFocusSwitching= UTL_ReadIntFromIniFile(pszIniFileName, szSec, "AutomaticFocusSwitch", TRUE );

  RigCtrl_ITU_Region   = UTL_ReadIntFromIniFile(pszIniFileName, szSec, "ITU_Region", RigCtrl_ITU_Region );
  sprintf( szSec, "Keyer%d", 1+UTL_iAppInstance ); // e.g. back to "Keyer1" for the FIRST running instance, etc
  for(i=0; i<KEYER_NUM_MESSAGE_MEMORIES; ++i)
   { sprintf( szKey, "Memory%d", (int)(i+1) );
     UTL_ReadStringFromIniFile( pszIniFileName,
        szSec, szKey, (char*)CwKeyer_psz6MemoryDefaults[i],
        CwKeyer_Config.szKeyerMemory[i]/*pszResult*/, KEYER_MAX_CHARS_PER_MEMORY );
   } // end for < all 'keyer message memories' >
  UTL_ReadStringFromIniFile( pszIniFileName,
        szSec, "MyCall", (char*)CwKeyer_Config.sz15MyCall, CwKeyer_Config.sz15MyCall/*default*/, 15 );

  // Optionally load the 'history of recent configuration files' from THIS file:
  if( iConfigFileOptions & CONFIG_FILE_OPTION_RECENT_FILES )
   { strcpy( szSec, "RecentConfigFiles" );
     for(i=0; i<GUI_RECENT_FILES_HISTORY_LENGTH; ++i)
      { sprintf( szKey, "File%d", (int)(i+1) );
        UTL_ReadStringFromIniFile( pszIniFileName, szSec, szKey,
            ""/*default*/, sz255Value/*dest*/, 255/*max len*/ );
        pszSrc = sz255Value;
        // Note: the CRAPPY WINDOWS API (GetPrivateProfileString) seems to
        //       remove the double quotes from the string by its own gusto.
        // For example, an INI file contained this ...
        // > [RecentConfigFiles]
        // > File1="C:\CwKeyerTest\ft817_local.INI"
        // > File2="C:\CwKeyerTest\ic7300_local.INI"
        // > File3=""
        // > File4=""
        // .... but despite having written those strings WITH double quotes,
        //           Microsoft's crazy API returned them WITHOUT.
        if( pszSrc[0]=='"' )
         { SL_ParseDoubleQuotedString( &pszSrc, g_sz255RecentFiles[i], 255 );
         }
        else // "thanks" to GetPrivateProfileString() for killing our double-quoted string !
         { SL_CopyStringUntilEndOfLine( &pszSrc, g_sz255RecentFiles[i], 255 );
         }
      } // end for < all 'Recent Files' (history) >
   } // end if < CONFIG_FILE_OPTION_RECENT_FILES > ?


  return TRUE;

} // end LoadSettings()

//---------------------------------------------------------------------------
void SaveSettings( const char *pszIniFileName, // .. in an old-fashioned *.ini file, NOT the bloody registry !
                   int iConfigFileOptions ) // [in] bitwise combination of CONFIG_FILE_OPTION_... def'd in Keyer_GUI.h
{ // [in] CwKeyer_Config .
  char szSec[84];  // "section" (in the ini file, that's the string in squared brackets)
  char szKey[84];  // "key name" (that's the string on the left side of the assignment operator)
  char sz255Value[256];
  char *pszDst, *pszEnd;
  TEdit *pEdit;
  int   i;
#if( SWI_NUM_AUX_COM_PORTS > 0 )
  T_AuxComConfig *pAuxComCfg;
#endif

  strcpy( szSec, "About" );
  UTL_WriteStringToIniFile(pszIniFileName,szSec,"1", "Configuration file for the 'Remote CW Keyer'.");
  UTL_WriteStringToIniFile(pszIniFileName,szSec,"2", "If you don't care about sending Morse code via PC,");
  UTL_WriteStringToIniFile(pszIniFileName,szSec,"3", "you can safely delete this file, wherever you found it.");
  UTL_WriteStringToIniFile(pszIniFileName,szSec,"4", " ");

  sprintf( szSec, "Keyer%d", 1+UTL_iAppInstance ); // e.g. "Keyer1" for the FIRST running instance, etc
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "ComPort_IN",  CwKeyer_Config.iComPortNumber_IN );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "RadioControlAndKeyingPort",CwKeyer_Config.iRadioKeyingAndControlPort );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "RadioControlProtocol", CwKeyer_Config.iRadioControlProtocol );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "RadioControlBaudrate", CwKeyer_Config.iRadioControlBaudrate );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "RadioCIVAddress", CwKeyer_Config.iRadioCIVAddress );
  UTL_WriteStringToIniFile(pszIniFileName,szSec,"RemoteRigCtrlServerAddress", CwKeyer_Config.sz80RemoteRigCtrlServerAddress);
#if( SWI_USE_HAMLIB_SERVER ) // not sure if this 'feature' is going to stay...
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "EnableBuiltInHamlibServer",CwKeyer_Config.fEnableBuiltInHamlibServer);
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "BuiltInHamlibServerPort", MyHamlibServer.cfg.iServerListeningPort);
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "BuiltInHamlibServerOptions", MyHamlibServer.cfg.iServerOptions);
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "BuiltInHamlibServerDiagnostics", MyHamlibServer.cfg.iDiagnosticFlags);
#endif // SWI_USE_HAMLIB_SERVER ?

  UTL_WriteIntToIniFile(pszIniFileName, szSec, "MorseKeyType",    CwKeyer_Config.iMorseKeyType );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "DotInputSignal",  CwKeyer_Config.iDotInput );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "DashInputSignal", CwKeyer_Config.iDashInput );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "ManualPTTInput",  CwKeyer_Config.iManualPTTInput);
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "TestInputSignal", CwKeyer_Config.iTestInput );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "KeySupplySignal", CwKeyer_Config.iKeySupply );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "RadioSupplySignal",CwKeyer_Config.iRadioSupply );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "SidetoneOnTXD",   CwKeyer_Config.iSidetoneOnTXD );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "CwKeyingSignal",  CwKeyer_Config.iRadioCWKeying );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "PTTControlSignal",CwKeyer_Config.iRadioPTTControl);
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "TxDelayTime_ms",  CwKeyer_Config.iTxDelayTime_ms);
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "TxHangTime_ms",   CwKeyer_Config.iTxHangTime_ms);
  UTL_WriteIntToIniFile(pszIniFileName,szSec,"DebouncePaddleInputs",CwKeyer_Config.fDebouncePaddleInputs);
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "KeyInputWatchdog",CwKeyer_Config.fKeyInputWatchdog);
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "CwViaShiftAndControlKey", CwKeyer_Config.fKeyViaShiftAndControl);
  UTL_WriteIntToIniFile(pszIniFileName, szSec,"DisableRealTransmit",CwKeyer_Config.fDisableTx );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "AudioOptions",(long)CwKeyer_Config.dwAudioOptions );

#if( SWI_NUM_AUX_COM_PORTS > 0 )
  for(i=0; i<SWI_NUM_AUX_COM_PORTS; ++i)
   { pAuxComCfg = &CwKeyer_Config.sAuxCom[i];
     sprintf( szKey, "AuxComPort%dNumber", (int)(i+1) );
     UTL_WriteIntToIniFile(pszIniFileName,szSec,szKey, pAuxComCfg->iPortNumber );
     sprintf( szKey, "AuxComPort%dUsage",  (int)(i+1) );
     UTL_WriteIntToIniFile(pszIniFileName,szSec,szKey, pAuxComCfg->iPortUsage );
     sprintf( szKey, "AuxComPort%dTunnel", (int)(i+1) );
     UTL_WriteIntToIniFile(pszIniFileName,szSec,szKey, pAuxComCfg->iTunnelIndex );
     sprintf( szKey, "AuxCom%dBitsPerSecond", (int)(i+1) );
     UTL_WriteIntToIniFile(pszIniFileName,szSec,szKey, pAuxComCfg->iBitsPerSecond );
     sprintf( szKey, "AuxCom%dNumDatabits", (int)(i+1) );
     UTL_WriteIntToIniFile(pszIniFileName,szSec,szKey, pAuxComCfg->iNumDatabits );
     sprintf( szKey, "AuxCom%dParity", (int)(i+1) );
     UTL_WriteIntToIniFile(pszIniFileName,szSec,szKey, pAuxComCfg->iParity );
     sprintf( szKey, "AuxCom%dNumStopbits", (int)(i+1) );
     UTL_WriteIntToIniFile(pszIniFileName,szSec,szKey, pAuxComCfg->iNumStopbits );
     sprintf( szKey, "AuxCom%dUsageDTR", (int)(i+1) ); // freely programmable DIGITAL OUTPUT
     UTL_WriteStringToIniFile( pszIniFileName,szSec,szKey,pAuxComCfg->sz40DTR );
     sprintf( szKey, "AuxCom%dUsageRTS", (int)(i+1) ); // freely programmable DIGITAL OUTPUT
     UTL_WriteStringToIniFile( pszIniFileName,szSec,szKey,pAuxComCfg->sz40RTS );
     sprintf( szKey, "AuxCom%dUsageDCD", (int)(i+1) ); // freely programmable DIGITAL INPUT
     UTL_WriteStringToIniFile( pszIniFileName,szSec,szKey,pAuxComCfg->sz40DCD );
     sprintf( szKey, "AuxCom%dUsageDSR", (int)(i+1) ); // freely programmable DIGITAL INPUT
     UTL_WriteStringToIniFile( pszIniFileName,szSec,szKey,pAuxComCfg->sz40DSR );
     sprintf( szKey, "AuxCom%dUsageCTS", (int)(i+1) ); // freely programmable DIGITAL INPUT
     UTL_WriteStringToIniFile( pszIniFileName,szSec,szKey,pAuxComCfg->sz40CTS );
     sprintf( szKey, "AuxCom%dUsageRI",  (int)(i+1) ); // freely programmable DIGITAL INPUT
     UTL_WriteStringToIniFile( pszIniFileName,szSec,szKey,pAuxComCfg->sz40RI );
     sprintf( szKey, "AuxCom%dParams",(int)(i+1) ); // e.g. hint to decode received traffic, CAT-protocol-specific parameters, etc
     UTL_WriteStringToIniFile( pszIniFileName,szSec,szKey,pAuxComCfg->sz40Params );
   }
  UTL_WriteIntToIniFile(pszIniFileName,szSec,"AuxComDiagnosticFlags",AuxCom_iDiagnosticFlags );
#endif // SWI_NUM_AUX_COM_PORTS > 0 ?
  UTL_WriteStringToIniFile(pszIniFileName,szSec,"AudioInputDevice", CwKeyer_DSP.cfg.sz255AudioInputDevice);
  UTL_WriteStringToIniFile(pszIniFileName,szSec,"AudioOutputDevice",CwKeyer_DSP.cfg.sz255AudioOutputDevice);
  UTL_WriteIntToIniFile(pszIniFileName, szSec,  "AudioSidetone_Hz", CwKeyer_DSP.cfg.iSidetoneFreq_Hz );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "SidetoneRiseTime_ms",CwKeyer_DSP.cfg.iSidetoneRiseTime_ms );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "SidetoneGain_dB",CwKeyer_DSP.cfg.iSidetoneGain_dB );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "AudioInGain_dB", CwKeyer_DSP.cfg.iAudioInGain_dB );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "AudioOutGain_dB",CwKeyer_DSP.cfg.iAudioOutGain_dB );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "NetworkTonesGain_dB",CwKeyer_DSP.cfg.iNetworkTonesGain_dB );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "AudioFlags",      CwKeyer_DSP.cfg.iAudioFlags );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "WordsPerMinute",Elbug_DotTimeInMillisecondsToWordsPerMinute(CwKeyer_Elbug.cfg.iDotTime_ms) );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "LaunchOnStart", APPL_fLaunchKeyerOnStart );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "Channel4Source",CwKeyer_TimingScope.cfg.iChannel4Source );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "TScopeChannels",CwKeyer_TimingScope.cfg.iVisibleChannels);
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "TScopeMsPerSample",CwKeyer_TimingScope.cfg.iMillisecondsPerSample);
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "TScopeTrigger", CwKeyer_TimingScope.cfg.iTriggerOptions );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "AudioSpectrumRefLevel_dB", CwKeyer_TimingScope.cfg.iAudioSpectrumRefLevel_dB );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "AudioSpectrumAmpleRange_dB",CwKeyer_TimingScope.cfg.iAudioSpectrumAmplRange_dB );
  for(i=0; i<(TIMING_SCOPE_N_DIGITAL_CHANNELS + TIMING_SCOPE_N_ANALOG_CHANNELS); ++i)
   { sprintf( szKey, "TScopeVertChannelPos%d", (int)(i+1) );
     UTL_WriteIntToIniFile(pszIniFileName,szSec,szKey, CwKeyer_TimingScope.cfg.iChannelVerticalPos_pcnt[i] );
   }

  // Still in the ini-file-section "KeyerN", where "N" is the instance index...
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "TRXOptions", CwKeyer_Config.iTRXOptions );

  // Save options for the built-in "Rig Control" module :
  sprintf( szSec, "RigControl%d", 1+UTL_iAppInstance ); // e.g. "RigControl1" for the FIRST running instance, etc
  UTL_WriteIntToIniFile(pszIniFileName,szSec,"RejectMessagesInLog",CwKeyer_Config.dwMessageFilterForLog ); // ex: MyCwNet.RigControl.dwMessageFilterForLog);
  UTL_WriteIntToIniFile(pszIniFileName,szSec,"TurnRigOffWhenClosing",CwKeyer_Config.fTurnRigOffWhenClosing);
  UTL_WriteIntToIniFile(pszIniFileName,szSec,"TrafficMonitorOptions",RigCtrl_TrafficMonitor.iDisplayOptions);
  UTL_WriteIntToIniFile(pszIniFileName,szSec,"TrafficMonitorTimestampFormat", RigCtrl_TrafficMonitor.iTimestampFormat );
  UTL_WriteIntToIniFile(pszIniFileName,szSec,"TrafficMonitorExtraColumns", RigCtrl_TrafficMonitor.iExtraColumns );


  // Save the "Network" configuration (from module CwNet.c; simple TCP-client or server):
  sprintf( szSec, "Network%d", 1+UTL_iAppInstance ); // e.g. "Network1" for the FIRST running instance, etc
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "Functionality",  MyCwNet.cfg.iFunctionality );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "DiagnosticFlags",MyCwNet.cfg.iDiagnosticFlags );
  sprintf( szSec, "Network%dClient", 1+UTL_iAppInstance );
  UTL_WriteStringToIniFile( pszIniFileName, szSec, "RemoteServerIP",MyCwNet.cfg.sz80ClientRemoteIP );
  UTL_WriteStringToIniFile( pszIniFileName, szSec, "UserName", MyCwNet.cfg.sz80ClientUserName );
  UTL_WriteStringToIniFile( pszIniFileName, szSec, "Callsign", MyCwNet.cfg.sz80ClientCallsign );

  sprintf( szSec, "Network%dServer", 1+UTL_iAppInstance );
  UTL_WriteIntToIniFile(    pszIniFileName, szSec, "ListeningPort", MyCwNet.cfg.iServerListeningPort );
  UTL_WriteStringToIniFile( pszIniFileName, szSec, "AcceptUsers",MyCwNet.cfg.sz255AcceptUsers );
  UTL_WriteStringToIniFile( pszIniFileName, szSec, "Pwd",MyCwNet.cfg.sz20ServerAdminPWD );
  UTL_WriteIntToIniFile(    pszIniFileName, szSec, "HttpOptions",MyCwNet.cfg.iHttpServerOptions );
  UTL_WriteIntToIniFile(    pszIniFileName, szSec, "Latency_ms", MyCwNet.cfg.iNetworkLatency_ms );

  sprintf( szSec, "Keyer%d", 1+UTL_iAppInstance ); // e.g. back to "Keyer1" for the FIRST running instance, etc
  for(i=0; i<KEYER_NUM_MESSAGE_MEMORIES; ++i)
   { sprintf( szKey, "Memory%d", (int)(i+1) );
     UTL_WriteStringToIniFile( pszIniFileName, szSec, szKey, CwKeyer_Config.szKeyerMemory[i] );
   } // end for < all 'keyer message memories' >
  UTL_WriteStringToIniFile( pszIniFileName, szSec, "MyCall", CwKeyer_Config.sz15MyCall );

  sprintf( szSec, "Keyer%dMainWindow", 1+UTL_iAppInstance ); // e.g. "Keyer1MainWindow" for the FIRST running instance, etc
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "Left",  CwKeyer_Config.iMainWindowLeft );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "Top",   CwKeyer_Config.iMainWindowTop );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "Width", CwKeyer_Config.iMainWindowWidth );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "Height",CwKeyer_Config.iMainWindowHeight );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "ColourScheme", g_SpecDispControl.iColourScheme );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "WFColourPalette",g_SpecDispControl.iWFColorPalette);
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "SpecDispOptions",g_SpecDispControl.iDisplayOptions );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "BottomPanelMode",g_SpecDispControl.iBottomPanelMode );

  UTL_WriteIntToIniFile(pszIniFileName, szSec, "Language", APPL_iLanguage );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "UserInterfaceMode", g_iUserInterfaceMode );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "HideMainMenu", g_fHideMenu );
  UTL_WriteIntToIniFile(pszIniFileName, szSec, "AutomaticFocusSwitch", KeyerGUI_fAutomaticFocusSwitching );


  UTL_WriteIntToIniFile(pszIniFileName, szSec, "ITU_Region", RigCtrl_ITU_Region );

  // Optionally save the 'history of recent configuration files' in THIS file:
  if( iConfigFileOptions & CONFIG_FILE_OPTION_RECENT_FILES )
   { strcpy( szSec, "RecentConfigFiles" );
     for(i=0; i<GUI_RECENT_FILES_HISTORY_LENGTH; ++i)
      { sprintf( szKey, "File%d", (int)(i+1) );
        pszDst = sz255Value;
        pszEnd = sz255Value + 255;
        SL_AppendDoubleQuotedString( &pszDst, pszEnd, GetRecentFileName(i) );
        UTL_WriteStringToIniFile( pszIniFileName, szSec, szKey, sz255Value );
      } // end for < all 'Recent Files' (history) >
   } // end if < CONFIG_FILE_OPTION_RECENT_FILES > ?


} // end SaveSettings()

//---------------------------------------------------------------------------
DWORD CalcHashForSettings(void) // .. to tell if we need to SAVE SETTINGS on exit
{
#ifndef  sizeof_member
# define sizeof_member(type,member) sizeof(((type*)0)->member) /* ugly.. but works */
#endif

  // Actually, the hash function is the 32-bit CRC implemented in Utilities.c :
  DWORD dwHash= UTL_CRC32( 0/*seed*/, (BYTE*)&CwKeyer_Config, sizeof(CwKeyer_Config) );
  dwHash = UTL_CRC32( dwHash/*seed*/, (BYTE*)&CwKeyer_Elbug.cfg, sizeof_member(T_ElbugInstance,cfg) );
  dwHash = UTL_CRC32( dwHash/*seed*/, (BYTE*)&CwKeyer_DSP.cfg, sizeof_member(T_CwDSP,cfg) );
  dwHash = UTL_CRC32( dwHash/*seed*/, (BYTE*)&MyCwNet.cfg, sizeof_member(T_CwNet,cfg) );
  dwHash = UTL_CRC32( dwHash/*seed*/, (BYTE*)&CwKeyer_TimingScope.cfg, sizeof_member(T_KeyerTimingScope,cfg) );
#if( SWI_USE_HAMLIB_SERVER ) // not sure if the 'Hamlib Net rigctld'-emulation is going to stay...
  dwHash = UTL_CRC32( dwHash/*seed*/, (BYTE*)&MyHamlibServer.cfg, sizeof_member(T_HLSrv,cfg) );
#endif // SWI_USE_HAMLIB_SERVER ?
#if( SWI_NUM_AUX_COM_PORTS > 0 )
  dwHash = UTL_CRC32( dwHash/*seed*/, (BYTE*)&AuxCom_iDiagnosticFlags,sizeof(int) );
#endif // SWI_NUM_AUX_COM_PORTS ?
  dwHash = UTL_CRC32( dwHash/*seed*/, (BYTE*)&RigCtrl_TrafficMonitor/*config*/,sizeof(RigCtrl_TrafficMonitor) );
  dwHash = UTL_CRC32( dwHash/*seed*/, (BYTE*)&g_SpecDispControl.iColourScheme,   sizeof(int) );
  dwHash = UTL_CRC32( dwHash/*seed*/, (BYTE*)&g_SpecDispControl.iWFColorPalette, sizeof(int) );
  dwHash = UTL_CRC32( dwHash/*seed*/, (BYTE*)&g_SpecDispControl.iDisplayOptions, sizeof(int) );
  dwHash = UTL_CRC32( dwHash/*seed*/, (BYTE*)&g_SpecDispControl.iBottomPanelMode,sizeof(int) );
  dwHash = UTL_CRC32( dwHash/*seed*/, (BYTE*)&g_sz255RecentFiles, sizeof(g_sz255RecentFiles) );

  return dwHash;

} // end CalcHashForSettings()

//---------------------------------------------------------------------------
void AddToRecentFiles( const char *pszConfigFileName )
{ int i;
  int iUnusedEntry = -1;

  if( pszConfigFileName == NULL ) // avoid garbage in the 'Recent Files' history
   { return;
   }
  if( SL_strnlen( pszConfigFileName, 256 ) >= 254 ) // hey, has Windows gone completely mad now ?
   { return;  // expect "fun" like the following (in pszConfigFileName) :
     // "
   }

  // First check it the file isn't already in the history.
  // If it is, don't append it. On this occation, already find the index
  // of the next UNUSED entry in g_sz255RecentFiles[] :
  for(i=0; i<GUI_RECENT_FILES_HISTORY_LENGTH; ++i)
   { if( g_sz255RecentFiles[i][0] == '\0' )
      { if( iUnusedEntry < 0 )
         {  iUnusedEntry = i;
         }
      }
     else // g_sz255RecentFiles[i] is already occupied with a filename ->
      { if( SL_strncmp( g_sz255RecentFiles[i], pszConfigFileName, 255) == 0 )
         { return;  // already in the list; no need to append it or even "scroll" the list
         }
      }
   }
  if( iUnusedEntry < 0 ) // no UNUSED entry, so scroll up the EXISTING entries:
   { for(i=0; i<(GUI_RECENT_FILES_HISTORY_LENGTH-1); ++i)
      { SL_strncpy( g_sz255RecentFiles[i], g_sz255RecentFiles[i+1], 255);
      }
     iUnusedEntry = GUI_RECENT_FILES_HISTORY_LENGTH-1;
   }
  // At this point, iUnusedEntry is always valid, so store the new entry HERE:
  SL_strncpy( g_sz255RecentFiles[iUnusedEntry], pszConfigFileName, 255 );

} // end AddToRecentFiles()

//---------------------------------------------------------------------------
const char* GetRecentFileName(int iZeroBasedIndex)
{ if((iZeroBasedIndex>=0) && (iZeroBasedIndex<GUI_RECENT_FILES_HISTORY_LENGTH))
   { return g_sz255RecentFiles[iZeroBasedIndex];
   }
  else
   { return ""; // caller doesn't even need to check for NULL pointers.
                // (he must check for EMPTY STRINGS, though)
   }
} // end GetRecentFileName()


//---------------------------------------------------------------------------
void ClearRecentFiles(void)
{
  memset( g_sz255RecentFiles, 0, sizeof(g_sz255RecentFiles) );
} // end ClearRecentFiles()


//---------------------------------------------------------------------------
extern "C" void ShowError( int iErrorClass, const char * pszFormat, ... )  // must be thread-safe ! Thus no VCL...
  // ,---------------------------'
  // '--> ERROR_CLASS_FATAL, ERROR_CLASS_ERROR, ERROR_CLASS_WARNING, ERROR_CLASS_INFO .
  //      If an optional 'debug run log' (file) is supported,
  //      iErrorClass bitmask SHOW_ERROR_IN_RUN_LOG can be used to emit
  //      the same message in "debug_run_log.txt", besides the GUI's "Debug" tab.
  //
{
  va_list parameter;
  T_ErrorFifoEntry *pErrorFifoEntry;
  char *pszDest, *pszEndstop;
  double dblUnixDateTime;
  int iHeadIndex = DEBUG_iErrorHistoryHead; // take a snapshot of the FIFO HEAD INDEX (this is "atomic" on a 32-bit machine)
  int iNewHeadIndex = ( iHeadIndex + 1 ) % C_ERROR_FIFO_SIZE;

  va_start( parameter, pszFormat );
  // Because ShowError() *may* be called from a worker thread, don't "print"
  // directly into the non-thread-safe "RichEdit" control ! Instead, use a
  // classic lock-free circular FIFO between ShowError() and the GUI itself.
  // As usual, if the INCREMENTED HEAD INDEX equals the TAIL, the FIFO is already full,
  // which means the sluggish IDE (VCL-based) could not keep up the pace:
  if( iNewHeadIndex != DEBUG_iErrorHistoryTail ) // circular FIFO not FULL yet ?
   {
     pErrorFifoEntry = &ErrorHistoryFifo[ DEBUG_iErrorHistoryHead % C_ERROR_FIFO_SIZE ];
     pszDest    = pErrorFifoEntry->sz511Text;
     pszEndstop = pErrorFifoEntry->sz511Text + 510/*!*/;
     pszDest[0] = '\0'; // safety first .. provide a trailing zero already

     if( iErrorClass & SHOW_ERROR_TIMESTAMP )
      { dblUnixDateTime = UTL_GetCurrentUnixDateAndTime_Fast();
        UTL_FormatDateAndTime( "hh:mm:ss.s ", dblUnixDateTime, pszDest );
        SL_SkipToEndOfString( (const char**)&pszDest );
      }
     vsnprintf( pszDest, (pszEndstop-pszDest)/*maxlen*/, pszFormat, parameter );
     // 2024-12-23 : vsnprintf() crashed somewhere in __org_vsnprintf() -> __vprinter()
     //              when called from CwServer_OnConnect() due to mismatch between
     //              the format string and the actual parameter on the stack.
     //              Fixed by adding typecasts, to leave no doubt about the type.
     pErrorFifoEntry->iErrorClass = iErrorClass & (~SHOW_ERROR_IN_RUN_LOG);

     // Make the new entry available for the "Debug" window.
     //  The GUI/VCL will drain it later in the main task/thread.
     DEBUG_iErrorHistoryHead = iNewHeadIndex;

     if( iErrorClass & SHOW_ERROR_IN_RUN_LOG )
      { UTL_WriteRunLogEntry( "%s", pErrorFifoEntry->sz511Text );
        // '--> "Does nothing" if UTL_fRunLogOpened==FALSE (no need to check HERE)
      }

   } // end if( iNewHeadIndex != DEBUG_iErrorHistoryTail )

  va_end(parameter); // we have called "va_start()", so we also call "va_end()" !


} // end ShowError()


//---------------------------------------------------------------------------
BOOL FillComboWithSerialPorts( TComboBox *pCombo,
        int iSelectedComPortNr, // [in] "COM"-port number (1..N) or NEGATIVE for "NONE"
        BOOL fNeedValidPort,    // [in] TRUE: "need a valid port" (e.g. because any of the
                                //           KEYER SIGNALS is on the port)
                                //    -> colour the combo RED when iSelectedComPortNr is INVALID.
                                //      FALSE: "no signal on this port selected yet"
                                //   (so do NOT colour the combo RED when invalid)
        BOOL *pfConflict,       // [out] TRUE: "there's a problem with this port", FALSE: no problem.
        const char *pszDefaultHint) // [in] e.g. "KEYER port, see manual for 'Simple Paddle Adapter'"
   // Returns TRUE if iSelectedComPortNr is currently valid,
   //         FALSE otherwise (i.e. not found in the devices "enumerated" by the OS)
   //
   // IMPORTANT: Any "TComboBox" used to select a serial port must have the
   //            'Style' set to 'csDropDown', not 'csDropDownList'. Reasons later..
   // From the VCL documentation:
   //  > csDropDown :
   //  >    Defines a drop-down list with an edit box for manually entered text.
   //  >
   //  > csDropDownList :
   //  >    Defines a drop-down list with no edit box;
   //  >    the user cannot enter text manually.
   // Reason: Users reported that on certain Windows versions, "virtual serial
   //         ports" or "virtual serial port emulators" like com0com and VSPE
   //         did NEITHER appear in this application's SELECTION LIST
   //            (because UTL_EnumerateComPorts() was unable to find them),
   //         NOR did they appear in the 'Windows Device Manager',
   //         'System Control', or wherever you find COM ports in the tree
   //         of devices like "Ports (COM & LPT)" !
   //      What a piece of crap but hey, it's Windows !
   //      Thus allow the user to TYPE IN anything he wants into the edit field,
   //      and ONLY evaluate TComboBox.Text (not the "item index") later,
   //      when APPLYING the new setting via
   //       ParseSerialPortNumberFromText() -> UTL_ParseNumberFromComPortName() !
{
  int  nEntriesFound = 0;
  BOOL fFoundSelectedPort = FALSE;
  int iEnumerator = 0;  // initialize 'enumerator'
  char sz80[84];
  static BOOL firstBug = TRUE;

  // On this occasion, allow the "TComboBox" to get WIDER when DROPPED DOWN :
  SendMessage( pCombo->Handle,CB_SETDROPPEDWIDTH, 280/*MinimumWidthInPixels*/, 0);
  // ,---------------------------------------------'
  // '--> This was even long enough for "COM3 (Silicon Labs CP210x USB to UART Bridge)",
  //      which was the funny "friendly" COM port name of an Icom IC-9700 !

  pCombo->Clear();
  pCombo->ItemIndex = -1; // nothing selected yet

  // First entry : "NONE", which is C_CFG_PORT_NONE = 0 in C:\cbproj\SpecLab\config.h
  pCombo->Items->Add( "NONE" );
  if( iSelectedComPortNr < 0 )
   {
     pCombo->ItemIndex = nEntriesFound; // index ZERO for the FIRST item in the list, etc
     fFoundSelectedPort = TRUE;
   }
  ++nEntriesFound;

  // Let Utilities.c : UTL_EnumerateComPorts() enumerate the COM ports.
  //     If possible (and Windows permitting), 'decorated' with parts of
  //     the 'friendly' name, but AFTER the significant part, in parentheses,
  //     e.g. "COM11 (USB Serial Port)" instead of just "COM11" :
  iEnumerator = 0;  // initialize the 'enumerator' for UTL_EnumerateComPorts()
  UTL_sz255LastError[0] = '\0';
  while( UTL_EnumerateComPorts( &iEnumerator, sz80, 80 ) )
   { // Add to the list of 'COM' ports, and possibly SELECT the item :
     pCombo->Items->Add( sz80 );  // hopefully something like "COM1", "COM2", ..
     // .. and if the "The Mimosa", SetupDiEnumDeviceInfo(), feels like cooperating,
     //    the list of COM ports even contains the 'friendly name' (every now and then).
     if( iSelectedComPortNr == UTL_ParseNumberFromComPortName(sz80) )
      { pCombo->ItemIndex = nEntriesFound; // index ZERO for the FIRST item in the list, etc
        fFoundSelectedPort = TRUE;
      }
     ++nEntriesFound;
   } // end while( UTL_EnumerateComPorts( &iEnumerator, sz80, 80 ) )
  if( (firstBug) && (UTL_sz255LastError[0] != '\0') ) // trouble extracting the "friendly names" again ?
   { ShowError( ERROR_CLASS_WARNING, "%s", UTL_sz255LastError );
     firstBug = FALSE;
     // 2024-01-05 : After being absent for several weeks, the "Bug" returned,
     // and UTL_EnumerateComPorts() diagnosed :
     // > EnumerateComPorts: SetupDiEnumDeviceInfo() failed at device #1 !
   }

  // If the configured COM port is NOT listed in the above loop
  //   (because stupid windows assigned a new COM port number to the device,
  //    or for some reason the COM port driver refuses to enumerate,
  //    as typically happens with COM ports from "VSPE"),
  // add a DUMMY ENTRY at the end of the list, so the pitiful windoze user
  //   can at least see that THERE IS SOMETHING WRONG WITH THE "COM PORT" .
  if( (!fFoundSelectedPort) && (iSelectedComPortNr>0) )
   {
     sprintf( sz80, "COM%d *NOT ENUMERATED*", iSelectedComPortNr );
     pCombo->Items->Add( sz80 );  // hopefully something like "COM1", "COM2", ..
     pCombo->ItemIndex = nEntriesFound; // index ZERO for the FIRST item in the list, etc
   } // end if( ! fFoundSelectedPort ) && ...

  pCombo->Font->Color = (TColor)g_SpecDispControl.clWindowText; // ex: clBlack   
  if( fNeedValidPort && ( (!fFoundSelectedPort) || (iSelectedComPortNr<0) ) )
   { pCombo->Color = clRed;
     // Modifiy the 'hint' to explain WHY this combo box is coloured red:
     pCombo->Hint = (AnsiString)TE("Signals on this port are used, but the port is currently invalid");
   }
  else
   { pCombo->Color = (TColor)g_SpecDispControl.clWindowBackground; // ex: clWindow
     pCombo->Hint = (AnsiString)TE(pszDefaultHint); // hint like "KEYER port, see manual for 'Simple Paddle Adapter'"
   }

  return fFoundSelectedPort;  // -> TRUE=ok, FALSE=something wrong with iSelectedComPortNr
} // end FillComboWithSerialPorts()


//---------------------------------------------------------------------------
static BOOL IsSignalOnMorseKeyPort(int iSignalIndex) // [in] KEYER_SIGNAL_INDEX... (negative when inverted)
{
  if( iSignalIndex < 0 )
   {  iSignalIndex = -iSignalIndex;
   }
  switch( iSignalIndex )
   { case KEYER_SIGNAL_INDEX_MORSE_KEY_DTR:
     case KEYER_SIGNAL_INDEX_MORSE_KEY_RTS:
     case KEYER_SIGNAL_INDEX_MORSE_KEY_DCD:
     case KEYER_SIGNAL_INDEX_MORSE_KEY_DSR:
     case KEYER_SIGNAL_INDEX_MORSE_KEY_CTS:
     case KEYER_SIGNAL_INDEX_MORSE_KEY_RI :
          return TRUE;
     default: return FALSE;  // the signal is NOT on the "Morse Key" port
   }

} // end IsSignalOnMorseKeyPort()


//---------------------------------------------------------------------------
static BOOL IsSignalOnRadioKeyingPort(int iSignalIndex) // [in] KEYER_SIGNAL_INDEX... (negative when inverted)
{
  if( iSignalIndex < 0 )
   {  iSignalIndex = -iSignalIndex;
   }
  switch( iSignalIndex )
   { case KEYER_SIGNAL_INDEX_RADIO_KEYING_DTR:
     case KEYER_SIGNAL_INDEX_RADIO_KEYING_RTS:
     case KEYER_SIGNAL_INDEX_RADIO_KEYING_DCD:
     case KEYER_SIGNAL_INDEX_RADIO_KEYING_DSR:
     case KEYER_SIGNAL_INDEX_RADIO_KEYING_CTS:
     case KEYER_SIGNAL_INDEX_RADIO_KEYING_RI :
          return TRUE;
     default: return FALSE;  // the signal is NOT on the "Radio Keying" port
   }
} // end IsSignalOnRadioKeyingPort()

//---------------------------------------------------------------------------
static BOOL IterateSignalsInCwKeyerConfig( int iterator, int **ppiSignalIndex )
{
  switch( iterator )
   { case 0: *ppiSignalIndex = &CwKeyer_Config.iDotInput;        break;
     case 1: *ppiSignalIndex = &CwKeyer_Config.iDashInput;       break;
     case 2: *ppiSignalIndex = &CwKeyer_Config.iTestInput;       break;
     case 3: *ppiSignalIndex = &CwKeyer_Config.iManualPTTInput;  break;
     case 4: *ppiSignalIndex = &CwKeyer_Config.iKeySupply;       break;
     case 5: *ppiSignalIndex = &CwKeyer_Config.iRadioSupply;     break;
     case 6: *ppiSignalIndex = &CwKeyer_Config.iRadioCWKeying;   break;
     case 7: *ppiSignalIndex = &CwKeyer_Config.iRadioPTTControl; break;
     default: return FALSE;
   }
  return TRUE;

} // end IterateSignalsInCwKeyerConfig()

//---------------------------------------------------------------------------
BOOL CheckForSignalsOnMorseKeyPort(void) // .. to find out if the MORSE KEY port is used at all.
  // Called from TKeyerMainForm::UpdateSettingsTab() to colour the
  //   SERIAL PORT SELECTION LIST red, when e.g. port set to "NONE"
  //   but any of the configurable I/O signals uses that port.
{
  BOOL fResult = FALSE;
  int  *piSignalIndex;
  int  iterator=0;

  while( IterateSignalsInCwKeyerConfig( iterator++, &piSignalIndex ) )
   { fResult |= IsSignalOnMorseKeyPort( *piSignalIndex );
   }
  return fResult;
} // end CheckForSignalsOnMorseKeyPort()


//---------------------------------------------------------------------------
BOOL CheckForSignalsOnRadioKeyingPort(void) // .. to find out if the RADIO KEYING/CONTROL port is used for digital I/O.
  // Called from TKeyerMainForm::UpdateSettingsTab() to colour the
  //   SERIAL PORT SELECTION LIST red, when e.g. port set to "NONE"
  //   but any of the configurable I/O signals uses that port.
{
  BOOL fResult = FALSE;
  int  *piSignalIndex;
  int  iterator=0;

  while( IterateSignalsInCwKeyerConfig( iterator++, &piSignalIndex ) )
   { fResult |= IsSignalOnRadioKeyingPort( *piSignalIndex );
   }
  return fResult;
} // end CheckForSignalsOnRadioKeyingPort()

//---------------------------------------------------------------------------
BOOL CheckForSignalsOnSamePin( int *piSignalIndex ) // .. to check for conflicts / duplicates in
  // CwKeyer_Config.iDotInput, .iDashInput, .iTestInput, .iManualPTTInput,
  //       .iKeySupply, ,iRadioSupply, .iRadioCWKeying, .iRadioPTTControl,
  // and whatever may be added in future (known by IterateSignalsInCwKeyerConfig() ).
  //  [in] piSignalIndex : points to one of the struct members listed above.
  //                       That's the signal TO CHECK against ALL OTHERS .
  // Called from TKeyerMainForm::UpdateSettingsTab() to colour the
  //   SIGNAL ASSIGNMENT COMBOS red, when for example
  //   CwKeyer_Config.iDotInput and CwKeyer_Config.iDashInput
  //   use the same "physical signal" on the same serial port.
  // Returns TRUE if *piSignalIndex (may point to CwKeyer_Config.iDotInput)
  //              contains e.g. KEYER_SIGNAL_INDEX_MORSE_KEY_CTS,
  //     and ANY OTHER of the config members listed above
  //     ALSO contains  +KEYER_SIGNAL_INDEX_MORSE_KEY_CTS
  //            or      -KEYER_SIGNAL_INDEX_MORSE_KEY_CTS (inverted, but the same physical signal).
{
  BOOL fConflict = FALSE;
  int  *piSignalIndex2;
  int  iterator=0;

  while( IterateSignalsInCwKeyerConfig( iterator++, &piSignalIndex2 ) )
   { if( piSignalIndex != piSignalIndex2 ) // a config member isn't a duplicate of itself ..
      { if( *piSignalIndex2 != KEYER_SIGNAL_INDEX_NONE ) // a signal configured to 'NONE' isn't a duplicate of other 'NONEs'
         { if( (*piSignalIndex2 == *piSignalIndex ) || (*piSignalIndex2 == -*piSignalIndex ) )
            { fConflict = TRUE;
            }
         }
      }
   }
  return fConflict; // -> TRUE = "it's a dupe",  FALSE = "no problem, it's NOT a dupe"
} // end CheckForSignalsOnSamePin()

//---------------------------------------------------------------------------
int AppendRadiosFromRigControlToCombo( TComboBox *pCombo )
  // ,----------------------------------------------'
  // '--> Originally contained 'hard coded' items entered via "Object Inspector".
  //      Other CI-V "default addresses" from RigControl.c : RigCtrl_RadioInfo_CIV[]
  //      for a few newer radios may be added HERE. To avoid duplicating entries,
  //      strings in the 'RigControl' database that do NOT exist in
  //      the original 'TComboBox' CB_CIV_Address. Since that, except for the
  //      'unknown Icom radio' dummy, all specific models have been moved
  //        from  Keyer_Main.dfm : TComboBox CB_CIV_Address
  //       into  RigControl.c   : T_RigCtrl_RadioInfo RigCtrl_RadioInfo_CIV[] .
  // Returns the NUMBER OF ITEMS ADDED HERE (not the total number of items in the "box").
{
  int  iItem, nItems;
  BOOL fExists;
  int  nItemsAdded = 0;
  char sz80Entry[84];
  T_RigCtrl_RadioInfo *pRadioInfo = (T_RigCtrl_RadioInfo *)RigCtrl_RadioInfo_CIV;

  nItems = pCombo->Items->Count;
  while( pRadioInfo->pszName != NULL )
   { fExists = FALSE;
     // To remain compatible with SelectComboItemByHexadecimalValue(),
     // the entries in the COMBO LIST must begin with the hex address, e.g.:
     // "0x94 IC-7300", "0xA2 IC-9700", etc, etc, etc:
     sprintf( sz80Entry, "0x%02X %s",
            (unsigned int)pRadioInfo->iDefaultAddress, pRadioInfo->pszName );
     for( iItem=0; iItem<nItems; ++iItem )
      { if( pCombo->Items->Strings[iItem] == AnsiString(sz80Entry) )
         { fExists = TRUE;
           break;
         }
      }
     if( ! fExists )
      { pCombo->Items->Add( sz80Entry );
        ++nItemsAdded;
      }
     ++pRadioInfo;
   }

  return nItemsAdded;
} // end AppendRadiosFromRigControlToCombo()

//---------------------------------------------------------------------------
int GetCIVAddressFromComboBox( TComboBox *pCombo )
  // ,--------------------------------------'
  // '--> Contains a few 'hard coded' items entered via "Object Inspector",
  //      and MANY MORE appended by AppendRadiosFromRigControlToCombo() .
  // Returns the 'Default CI-V Address' that originally identified a certain
  //         ICOM-radio, but later also a few YAESU- and KENWOOD radios
  //         with "default CI-V Addresses" above 0x00FF (which can't be ICOM).
{
  const char *pszSrc;
  T_RigCtrl_RadioInfo *pRadioInfo = (T_RigCtrl_RadioInfo *)RigCtrl_RadioInfo_CIV;

  if( pCombo->ItemIndex == 0 )  // 0 = "Rig Control OFF", 1 = "AutoDetect" ..
   { return RIGCTRL_DEF_ADDR_NO_REMOTE_CTRL; // CwKeyer_Config.iRadioCIVAddress = -1 = "OFF (No RigControl)"
   }
  else if( pCombo->ItemIndex == 1 )
   { return RIGCTRL_DEF_ADDR_AUTO_DETECT;  // CwKeyer_Config.iRadioCIVAddress = 0 = "Auto-Detect"
   }
  else if( pCombo->ItemIndex == 2 )
   { return RIGCTRL_DEF_ADDR_UNKNOWN_YAESU; // dummy for YAESU-radios that "don't speak CI-V"
   }
  else // expecting a HEXADECIMAL CI-V address (usually with two digits) ->
   { pszSrc = (const char*)pCombo->Text.c_str();
     if( SL_SkipString( &pszSrc, "0x" ) ) // looks like a HEX number ...
      { return SL_ParseHex( &pszSrc, 4/*nMaxDigits*/ );
      }
     else // may be an INTEGER number, or one of the strings (names)
      {   // from  RigCtrl_RadioInfo_CIV[] (and whatever may be added in future).
        while( pRadioInfo->pszName != NULL )
         { if( SL_stricmp( pszSrc, pRadioInfo->pszName ) == 0 )
            { return pRadioInfo->iDefaultAddress;
            }
         }
        // Arrived here ? Not HEX, not a name from the database in RigControl.c,
        // so guess someone has entered a DECIMAL VALUE manually:
        return SL_ParseInteger( &pszSrc );
      }
   }
} // end GetCIVAddressFromComboBox()


//---------------------------------------------------------------------------
BOOL FillComboWithItemsFromTokenLists( TComboBox *pCombo,
        const T_SL_TokenList *pTokens1, // [in] e.g. C_TL_None_Zero (with string = "None", value = zero)
        const T_SL_TokenList *pTokens2, // [in] e.g. SerialPortInputSignals_Key
        const T_SL_TokenList *pTokens3, // [in] e.g. SerialPortInputSignals_Radio, or NULL for only *one* list
        int iSelectedTokenValue  )      // [in] e.g. [-]KEYER_SIGNAL_INDEX_MORSE_KEY_CTS
{
  int   iList;
  int   nEntries = 0;
  const T_SL_TokenList *pTokens;
  pCombo->Clear();
  pCombo->ItemIndex = -1;
  for( iList=1; iList<=3; ++iList)
   { switch(iList)
      { case 1: pTokens = pTokens1; break;
        case 2: pTokens = pTokens2; break;
        case 3: pTokens = pTokens3; break;
        default: pTokens = NULL; break;
      }
     if( pTokens != NULL ) // emit the strings from another list ?
      { while( pTokens->pszKeyword != NULL ) // not the end of the list yet ?
         {
           pCombo->Items->Add( pTokens->pszKeyword );  // hopefully something like "COM1", "COM2", ..
           if( iSelectedTokenValue == pTokens->iTokenValue )
            { pCombo->ItemIndex = nEntries; // index ZERO for the FIRST item in the list, etc
            }
           ++nEntries;
           ++pTokens;
         }
      } // if( pTokens != NULL )
   }
  if( pCombo->ItemIndex >= 0)
   { return TRUE;
   }
  pCombo->ItemIndex = 0; // instead of selecting garbage, select the first ...
  return FALSE;          // ... but give the caller the chance to mark this item in RED
} // end FillComboWithItemsFromTokenLists()

enum
{ COM_PARAMS_TOKEN_DEBUG=0,
  COM_PARAMS_TOKEN_DELAY, // "delay=<value in ms>" to test 'impatient' remote clients / throttle down THEIR traffic
  COM_PARAMS_TOKEN_TERMINAL,
  COM_PARAMS_TOKEN_PROTOCOL, COM_PARAMS_TOKEN_ICOM, COM_PARAMS_TOKEN_KENWOOD,
  COM_PARAMS_TOKEN_YAESU_5_BYTE, COM_PARAMS_TOKEN_YAESU_ASCII
};


const T_SL_TokenList KeyerGUI_AuxComParams[] = // so far, only used in KeyerGUI_ParseAuxComParams()
{ // pszKeyword, iTokenValue :
 { "debug",    COM_PARAMS_TOKEN_DEBUG    }, // an EXPLANATION of these bitflags ..
 { "delay",    COM_PARAMS_TOKEN_DELAY    }, // "delay=<value in ms>" to test 'impatient' remote clients / throttle down THEIR traffic
 { "terminal", COM_PARAMS_TOKEN_TERMINAL }, //  .. is in AuxComPorts.h : T_AuxComPortInstance.dwParams !
 { "protocol", COM_PARAMS_TOKEN_PROTOCOL }, // "protocol=" + one of the following:
 { "icom",     COM_PARAMS_TOKEN_ICOM     },     // \  only ONE of these "CAT protocols"..
 { "kenwood",  COM_PARAMS_TOKEN_KENWOOD  },     //  | .. may be specified !
 { "yaesu_5byte", COM_PARAMS_TOKEN_YAESU_5_BYTE }, // / (being IMPLEMENTED is another thing..)
 { "yaesu_ascii", COM_PARAMS_TOKEN_YAESU_ASCII},
 { NULL, 0 } // "all zeros" mark the end of the list
}; // end RigCtrl_BandNames[]


//---------------------------------------------------------------------------
void KeyerGUI_ParseAuxComParams(
        const char *psz40AuxComParams,     // [in] e.g. "terminal, protocol=Icom, debug, delay=200"
        T_AuxComPortInstance *pAuxComPort) // [out] Rig-Control 'port instance'
  // [out] pAuxComPort->dwParams: bitwise combination of flags like AUX_COM_PARAMS_DEBUG,
  //          and a bitgroup (mask AUX_COM_PARAMS_MASK_PROTOCOL) for the protcol,
  //     e.g. AUX_COM_PARAMS_PROTOCOL_ICOM, AUX_COM_PARAMS_PROTOCOL_YAESU_5_BYTE, etc.
  // [out] pAuxComPort->iExtraDelay_ms: set e.g. via 'delay=200' to 200 ms .
  //
  // (Keywords / tokens are checked CASE-INSENSITIVE, thus "Icom"="icom"="ICOM")
  // The implementation of the varous 'Additional COM Port' functions
  // do not PARSE the optional string in CwKeyer_Config.sAuxCom[].sz40Params.
  // Instead (last not least for the sake of SPEED), they used a BIT COMBINATION
  // in T_AuxComPortInstance.dwParams. THIS function parses the "Params" string,
  // expecting the format specified in
  //  file:///C:/cbproj/Remote_CW_Keyer/manual/Remote_CW_Keyer.htm#Additional_COM_Params
  //
{
  int iToken;
  const char *pszSrc = psz40AuxComParams;
  const char *cpEnd = psz40AuxComParams + 40;  // "endstop" for the primitive parser below
  const char *pszPrevSrc;
  DWORD dwAuxComParams = AUX_COM_PARAMS_DEFAULT; // (0) if the "Params"-field in the GUI is EMPTY, NONE of the bits shall be set

  while( (pszSrc < cpEnd) && (*pszSrc != '\0') )
   { pszPrevSrc = pszSrc;   // <- kludge to prevent getting stuck in this loop
     SL_SkipSpaces( &pszSrc );
     iToken = SL_SkipOneOfNStrings_AnyCase( &pszSrc, KeyerGUI_AuxComParams );
     SL_SkipSpaces( &pszSrc );
     switch( iToken )
      { case COM_PARAMS_TOKEN_DEBUG  :
           dwAuxComParams |= AUX_COM_PARAMS_DEBUG; // -> Allow display in the traffic monitor; see RigCtrl_IsMsgTypeRejectedForLog() ..
           break;
        case COM_PARAMS_TOKEN_DELAY  : // "delay=<value in ms>" to test 'impatient' remote clients / throttle down THEIR traffic
           if( SL_SkipChar( &pszSrc, '=' ) )
            { pAuxComPort->iExtraDelay_ms = SL_ParseInteger( &pszSrc );
            }
           break;
        case COM_PARAMS_TOKEN_TERMINAL:
           dwAuxComParams |= AUX_COM_PARAMS_DUMP_TO_TERMINAL;
           break;
        case COM_PARAMS_TOKEN_PROTOCOL: // more intuitive syntax, eg "protocol=Icom"
           SL_SkipCharsUntilDelimiter( &pszSrc, "=: "/*delimiters*/, SL_SKIP_NORMAL ); // "until, AND INCLUDING.."
           iToken = SL_SkipOneOfNStrings_AnyCase( &pszSrc, KeyerGUI_AuxComParams );
           break;
        // not here: case COM_PARAMS_TOKEN_ICOM/KENWOOD/... :
        default:
           break;
      } // end switch( iToken )
     switch( iToken ) // .. with our without the already skipped "protocol=" :
      {
        case COM_PARAMS_TOKEN_ICOM   :
           // Only ONE of the "Protocol Decoders" may be selected,
           // .. so clear other bits and set only ONE :
           dwAuxComParams = (dwAuxComParams & ~AUX_COM_PARAMS_MASK_PROTOCOL) | AUX_COM_PARAMS_PROTOCOL_ICOM;
           break;
        case COM_PARAMS_TOKEN_KENWOOD:
           dwAuxComParams = (dwAuxComParams & ~AUX_COM_PARAMS_MASK_PROTOCOL) | AUX_COM_PARAMS_PROTOCOL_KENWOOD;
           break;
        case COM_PARAMS_TOKEN_YAESU_5_BYTE:    // the dreadful old stuff used in e.g. FT-817, FT-897
           dwAuxComParams = (dwAuxComParams & ~AUX_COM_PARAMS_MASK_PROTOCOL) | AUX_COM_PARAMS_PROTOCOL_YAESU_5_BYTE;
           break;
        case COM_PARAMS_TOKEN_YAESU_ASCII:     // the semicolon-delimited stuff with TWO-ASCII-CHARACTER-COMMANDS, used in e.g. FT-891
           dwAuxComParams = (dwAuxComParams & ~AUX_COM_PARAMS_MASK_PROTOCOL) | AUX_COM_PARAMS_PROTOCOL_YAESU_ASCII;
           break;

        default:
           break;
      } // end switch( iToken )
     if( iToken < 0 ) // unknown token (single word as a string) -> skip it
      { if( SL_SkipCharsUntilDelimiter( &pszSrc, " ,"/*delimiters*/, SL_SKIP_NORMAL ) < 0 )
         {  // '--> Returns the NUMBER OF CHARACTERS actually skipped, INCLUDING the delimiter.
           break;
         }
      }
     if( pszPrevSrc == pszSrc ) // oops.. not a single character was skipped in this loop ?
      { break;
      }
   } // end while( pszSrc < cpEnd )

  pAuxComPort->dwParams = dwAuxComParams;


} // end KeyerGUI_ParseAuxComParams()


//---------------------------------------------------------------------------
BOOL KeyerGUI_ParseSerialFormat(
         const char *pszSrc,  // [in] e.g. "8-N-1", or maybe "8N1", etc
         int  *piNumDatabits, // [out] what the name implies..
         int  *piParity, int *piNumStopbits )
  // See also (inverse function) : KeyerGUI_SerialFormatToString()
{ BOOL fResult = TRUE;  // -> TRUE=successfully parse;  FALSE=garbage in, "8-N-1" out :)
  int  iNumDatabits, iParity, iNumStopbits;
  char c;

  iNumDatabits = SL_ParseInteger( &pszSrc );
  SL_SkipChar( &pszSrc, '-' );  // the separating '-' is optional (8N1 gives the same result as 8-N-1)
  switch( SL_ToLower( SL_SkipAnyChar(&pszSrc) ) )
   { case 'n': iParity = SER_PORT_PARITY_NONE;  break;
     case 'o': iParity = SER_PORT_PARITY_ODD ;  break;
     case 'e': iParity = SER_PORT_PARITY_EVEN;  break;
     case 'm': iParity = SER_PORT_PARITY_MARK;  break;
     case 's': iParity = SER_PORT_PARITY_SPACE; break;
     default:  iParity = SER_PORT_PARITY_NONE;
          fResult = FALSE;  // have 'corrected' something, PLEASE CHECK !
          break;
   }
  SL_SkipChar( &pszSrc, '-' );  // another optional separator..
  if( SL_SkipToken( &pszSrc, "1.5" ) )
   { iNumStopbits = SER_PORT_STOPBITS_1_5;  // VERY rare, but who knows..
   }
  else if( SL_SkipToken( &pszSrc, "1" ) )
   { iNumStopbits = SER_PORT_STOPBITS_1;
   }
  else if( SL_SkipToken( &pszSrc, "2" ) )
   { iNumStopbits = SER_PORT_STOPBITS_2;
   }
  else
   { iNumStopbits = SER_PORT_STOPBITS_1;
     fResult = FALSE;  // have 'corrected' something, PLEASE CHECK ("colour me RED", dear string input-field)
   }
  if( piNumDatabits != NULL )       // all these OUTPUTS are OPTIONAL..
   { *piNumDatabits = iNumDatabits; 
   }
  if( piParity != NULL )
   { *piParity = iParity;
   }
  if( piNumStopbits != NULL )
   { *piNumStopbits = iNumStopbits;
   }
  return fResult;
} // end KeyerGUI_ParseSerialFormat()

//---------------------------------------------------------------------------
void KeyerGUI_SerialFormatToString( // inverse to KeyerGUI_ParseSerialFormat()
         int  iNumDatabits, int iParity, int iNumStopbits, // [in] what the name implies..
         char *psz7Dest ) // [out] C-string with at least 7 characters capacity + trailing zero
{ char *cpEndstop = psz7Dest+7;
  char cParity;
  char *pszStopbits;
  if( (iNumDatabits < 5) || (iNumDatabits > 9 ) ) // remember 5-bit Baudot.. still possible today
   { iNumDatabits = 8; // if the config-struct had been "memset zeroed", the format will be "8-N-1"
   }
  switch( iParity )
   { case SER_PORT_PARITY_NONE  : cParity = 'N'; break; // None (N) means that no parity bit is sent and the transmission is shortened.
     case SER_PORT_PARITY_ODD   : cParity = 'O'; break; // Odd (O) means that the parity bit is set so that the number of 1 bits is odd.
     case SER_PORT_PARITY_EVEN  : cParity = 'E'; break; // Even (E) means that the parity bit is set so that the number of 1 bits is even.
     case SER_PORT_PARITY_MARK  : cParity = 'M'; break; // Mark (M) parity means that the parity bit is always set to the mark signal condition (1 bit value).
     case SER_PORT_PARITY_SPACE : cParity = 'S'; break; // Space (S) parity always sends the parity bit in the space signal condition (0 bit value).
     default: cParity = 'N'; break; // garbage in, but no garbage out .. Assume NO parity.
   }
  switch( iNumStopbits )
   { case SER_PORT_STOPBITS_1:
     default:
          pszStopbits = "1";
          break;
     case SER_PORT_STOPBITS_2:
          pszStopbits = "2";
          break;
     case SER_PORT_STOPBITS_1_5:  // VERY rare, but who knows..
          pszStopbits = "1.5";
          break;
   }
  SL_AppendPrintf( &psz7Dest, cpEndstop, "%d-%c-%s", iNumDatabits, cParity, pszStopbits );

} // end KeyerGUI_SerialFormatToString()

//---------------------------------------------------------------------------
const char* KeyerGUI_PortNumberToString( int iPortNumber )
{ static char sz15Result[16];
  if( iPortNumber < 0 )
   { return "<NoPort>";
   }
  sprintf(sz15Result,"COM%d",(int)iPortNumber );
  return sz15Result;
} // end KeyerGUI_PortNumberToString()

//---------------------------------------------------------------------------
void KeyerGUI_AppendFreqRangeToBandCombo(
        T_RigCtrlInstance *pRC,  // [in] 'Rig Control' instance, with e.g. the current operating frequency for the "current band" indicator
        TComboBox      *pCombo,  // [out] VCL combo box with bands or even band-stacking registers
        double dblFmin_Hz, double dblFmax_Hz, // [in] frequency range of band or sub-band
        const char* pszBandName) // [in] name of band (e.g. "60 m") or SUB-BAND (e.g. "60 m EU")
  // [out] (in global variables, only used by the GUI but not by the 'rig control' module):
  //       KeyerGUI_BandComboInfoTable[row][column].dblFmin_Hz, .dblFmax_Hz,
  //          .iUserDefinedBandIndex, .iBandStackingRegIndex (for Icom), ..(?)
{
  int   iBandIndex, iUserDefdBand, nColumns;
  int   iRow = pCombo->Items->Count; // <- may run from zero to KEYER_GUI_MAX_BANDS-1
  int   iColumn = 0;                 // <- may run from zero to KEYER_GUI_MAX_COLUMNS_PER_BAND-1
  char  sz7OperatingMode[8]; // result from RigCtrl_OperatingModeToString(), but "lower-cased" for RECEIVE-ONLY entries
  char  sz80FreqAndMode[84];
  char  sz80ItemText[84];
  char  *pszSource, *pszDest, *cpDestEndstop;
  BOOL  fOkToAddFrequency;
  int   iBandStackingRegIndex, iUserDefinedBand, iLen, iWantedPos, nExtraChars;
  T_RigCtrlFreqMemEntry     *pBandStackingReg;
  T_RigCtrlUserDefinedBand  *pUserDefdBand;
  T_BandComboInfoTableEntry *pInfoTableEntry = NULL;

#ifdef __BORLANDC__  // "... is assigned a value that is never used".. shut up ..
  (void)pInfoTableEntry;  // better a NULL pointer than a DULL (uninitialized) pointer !
#endif


  if( (iRow<0) || (iRow>=KEYER_GUI_MAX_BANDS) ) // oops.. illegal array index into KeyerGUI_BandComboInfoTable[iRow] !
   { return;
   }

  pInfoTableEntry = &KeyerGUI_BandComboInfoTable[iRow][0/*iColumn*/];
  pInfoTableEntry->dblFmin_Hz = dblFmin_Hz;
  pInfoTableEntry->dblFmax_Hz = dblFmax_Hz;

  // Regardless of pRC->BandStackingRegs[ 0..RIGCTRL_NUM_BAND_STACKING_REGS-1 ],
  //            or in pRC->UserDefinedBands[ 0..RIGCTRL_NUM_USER_DEFINED_BANDS-1 ],
  // pszBandName = "160 m", "80 m", "60 m", "40 m",
  //       or a user-defined band name like "60 m EU", "60 m UK", etc.
  // The BAND NAME will always be added to the combo, even if there are
  // no band-stacking-register entries (etc) for this band *anywhere*
  // (This is important, because for example an IC-7300
  //  modified for 'all band transmit' or a few extra bands like 60 m or 4 m
  //  will NOT add extra 'band codes' to the band stacking registers.
  //  Instead, the 'added bands' like 60 m, 4 m, etc will all share a
  //  band that Icom shows as 'GENE' in the IC-7300's band stacking
  //  register display.)
  // EXAMPLE: Content of pRC->BandStackingRegs[], as displayed via
  //   'Settings'..'Report Rig Control parameters on the Debug tab' :
  //      BandStack[00..02]:  1.915  CW,   1.910  CW,   1.915  CW
  //      BandStack[03..05]:  3.573  CW,   3.793 LSB,   3.557  CW
  //      BandStack[06..08]:  7.030  CW,   7.076 LSB,   7.201 LSB
  //      BandStack[09..11]: 10.120  CW,  10.130  CW,  10.140  CW
  //      BandStack[12..14]: 14.244 USB,  14.100  CW,  14.054  CW
  //      BandStack[15..17]: 18.150  CW,  18.150 USB,  18.070  CW
  //      BandStack[18..20]: 21.024  CW,  21.300 USB,  21.050  CW
  //      BandStack[21..23]: 24.898  CW,  24.980 USB,  24.900  CW
  //      BandStack[24..26]: 28.036  CW,  28.565 USB,  29.269  FM
  //      BandStack[27..29]: 50.085  CW,  50.097  CW,  50.095  CW
  //      BandStack[30..32]:  5.353  CW,   0.472  CW,  70.107  CW
  //                          /|\           /|\          /|\   ..
  //      "GENE" band entries--'-------------'------------'
  //
  //    DON'T ASSUME ANYTHING about which frequencies are stored
  //    in which part of the array ... things will be COMPLETELY DIFFERENT
  //    in other transceivers; e.g. IC-9700, IC-705, etc etc etc !
  pszDest = sz80ItemText;
  cpDestEndstop = sz80ItemText + sizeof(sz80ItemText) - 1;
  SL_AppendString( &pszDest, cpDestEndstop, pszBandName );

  // Check all 'Band Stacking Registers' if they are in THIS band (dwBitmask) .
  // If a band stacking register belongs to this band, append it to
  // the string (limited to THREE stack entries per band, as in Icom radios).
  //
  // As long as the maximum number of entries PER BAND (=visible columns per row)
  // has not been reached yet, append entries from Icom's BAND STACKING REGISTERS:
  for( iBandStackingRegIndex=0;
      (iBandStackingRegIndex < pRC->iNumBandStackingRegs)
                 && (iColumn < KEYER_GUI_MAX_COLUMNS_PER_BAND);
       iBandStackingRegIndex++)
   { pBandStackingReg = &pRC->BandStackingRegs[ iBandStackingRegIndex ];
     // Only add this entry from pRC->BandStackingRegs[] if the frequency
     // is inside the current BAND or sub-band:
     if( (pBandStackingReg->RxTx[0].dblOperatingFreq_Hz >= dblFmin_Hz)
       &&(pBandStackingReg->RxTx[0].dblOperatingFreq_Hz <= dblFmax_Hz) )
      { if( iColumn == 0 )
         { SL_AppendString( &pszDest, cpDestEndstop, ":" );
         }
        SL_strncpy( sz7OperatingMode,
          RigCtrl_OperatingModeToString( pBandStackingReg->RxTx[0].iOpMode & RIGCTRL_OPMODE_MASK_TO_STRIP_FLAGS ),
          sizeof(sz7OperatingMode) );
        if( ! RigCtrl_MayTransmitOnFrequencyAndMode( pRC,
                pBandStackingReg->RxTx[0].dblOperatingFreq_Hz,
                pBandStackingReg->RxTx[0].iOpMode ) )
         { SL_ToLower_String( sz7OperatingMode ); // -> e.g. "cwn" instead of "CWN" when the listed frequency is "RECEIVE ONLY"
         }

        // As in Icom's own "BAND STACKING REGISTER" display,
        // show the frequency in MEGAHERTZ (but without a unit),
        // with leading space for a fixed width, and only
        // three decimal places. Umm.. what was the printf format string ?
        // Here: Use a format string to have everything vertically aligned:
        snprintf( sz80FreqAndMode, 40, "%9.4lf %3s",
           // "width" (?) ---------------' |
           // "precision" (or whatever)----'
           (double)(pBandStackingReg->RxTx[0].dblOperatingFreq_Hz * 1e-6),
            sz7OperatingMode/*CW,CWN,CWNN, etc; lower-case for "RX only"*/ );
        pszSource = sz80FreqAndMode;
        iLen = strlen( sz80ItemText );
        iWantedPos = BANDLIST_BAND_COLUM_WIDTH + BANDLIST_FREQ_COLUMN_WIDTH * iColumn;
        if( iLen < iWantedPos )
         { SL_PadSpaces( &pszDest, cpDestEndstop, iWantedPos - iLen );
         }
        else // something in the previous column was too long ->
         { // if the FREQUENCY doesn't need FOUR integer digits,
           // make it shorter by removing LEADING SPACES :
           nExtraChars = iLen - iWantedPos;
           while( (nExtraChars>0) && (pszSource[0]==' ') && (pszSource[1]==' ') )
            { ++pszSource;
              --nExtraChars;
            }
         }
        SL_AppendString( &pszDest, cpDestEndstop, pszSource );
        pInfoTableEntry = &KeyerGUI_BandComboInfoTable[iRow][iColumn++];
        pInfoTableEntry->iBandStackingRegIndex = iBandStackingRegIndex;
      } // end if < matching frequency for the current "band" >
   }   // end for( iBandStackingRegIndex .. )

  // As long as the maximum number of entries PER BAND (=visible columns per row)
  // has not been reached yet, append the START- or DEFAULT frequencies
  // of the sysop-defined USER-DEFINED BANDS:
  for( iUserDefinedBand=0;
      (iUserDefinedBand < pRC->iNumUserDefinedBands)
            && (iColumn < KEYER_GUI_MAX_COLUMNS_PER_BAND);
       iUserDefinedBand++)
   { pUserDefdBand = &pRC->UserDefinedBands[ iUserDefinedBand ];
     // '--> char   sz15Name[16];
     //      double dblFmin_Hz, dblFmax_Hz, dblFdef_Hz;
     //      DWORD  dwOpModes; Bitwise combination of RIGCTRL_OPMODE_CW, ..
     //      DWORD  dwPermissions : RIGCTRL_PERMISSION_NONE, RIGCTRL_PERMISSION_TRANSMIT, ..
     // Only add this entry from pRC->UserDefinedBands[] if it is
     //  "compatible with" / "contained in" the current BAND, dwBitmask :
     if(  (   (pUserDefdBand->dblFdef_Hz >= dblFmin_Hz)
           && (pUserDefdBand->dblFdef_Hz <= dblFmax_Hz) )
        ||(   (pUserDefdBand->dblFmin_Hz >= dblFmin_Hz)
           && (pUserDefdBand->dblFmax_Hz <= dblFmax_Hz) )
       )
      { // At least there is some OVERLAP with this band (or sub-band) ->
        if( iColumn == 0 )
         { SL_AppendString( &pszDest, cpDestEndstop, ":" );
         }
        iLen = strlen( sz80ItemText );
        iWantedPos = BANDLIST_BAND_COLUM_WIDTH + BANDLIST_FREQ_COLUMN_WIDTH * iColumn;
        if( iLen < iWantedPos )
         { SL_PadSpaces( &pszDest, cpDestEndstop, iWantedPos - iLen );
         }
        // If the user-defined band has a valid DEFAULT FREQUENCY, show that.
        // Otherwise, if the user-defined band has a NAME (e.g. "60 m EU"), show the NAME.
        //  (Since the FREQUENCY SCALE on the "TRX" tab indicates the
        //   USER-DEFINED BANDs and their names, it's easy to see
        //   in which band or even "sub-band" the current VFO frequency is.
        //   Thus no need to show the "sub-band name" if there's a DEFAULT FREQUENCY.)
        // If neither a DEFAULT FREQUENCY or a BAND NAME exists,
        //   show the (sub-)band's START FREQUENCY (worst alternative).
        // Note (again) : On the GUI, radio frequencies are shown in Megahertz !
        SL_strncpy( sz7OperatingMode,
           RigCtrl_OperatingModeToString( pUserDefdBand->dwDefOpMode & RIGCTRL_OPMODE_MASK_TO_STRIP_FLAGS ),
           sizeof(sz7OperatingMode) );
        if( ! (pUserDefdBand->dwPermissions & RIGCTRL_PERMISSION_TRANSMIT) )
         { // Ooops .. a "receive-only" band !
           SL_ToLower_String( sz7OperatingMode ); // -> e.g. "cwn" instead of "CWN" when the listed frequency is "RECEIVE ONLY"
         }
        if( (pUserDefdBand->dblFdef_Hz >= dblFmin_Hz)
         && (pUserDefdBand->dblFdef_Hz <= dblFmax_Hz) )
         { snprintf( sz80FreqAndMode, 40,  "%9.4lf %3s",
                pUserDefdBand->dblFdef_Hz * 1e-6, sz7OperatingMode );
         }
        else
         { snprintf( sz80FreqAndMode, 40, "%9.4lf %3s",
                pUserDefdBand->dblFmin_Hz * 1e-6, sz7OperatingMode );
         }
        pszSource = sz80FreqAndMode;
        iLen = strlen( sz80ItemText );
        iWantedPos = BANDLIST_BAND_COLUM_WIDTH + BANDLIST_FREQ_COLUMN_WIDTH * iColumn;
        if( iLen < iWantedPos )
         { SL_PadSpaces( &pszDest, cpDestEndstop, iWantedPos - iLen );
         }
        else // something in the previous column was too long ->
         { // if the FREQUENCY doesn't need FOUR integer digits,
           // make it shorter by removing LEADING SPACES :
           nExtraChars = iLen - iWantedPos;
           while( (nExtraChars>0) && (pszSource[0]==' ') && (pszSource[1]==' ') )
            { ++pszSource;
              --nExtraChars;
            }
         }
        SL_AppendString( &pszDest, cpDestEndstop, pszSource );
        pInfoTableEntry = &KeyerGUI_BandComboInfoTable[iRow][iColumn++];
        pInfoTableEntry->iUserDefinedBandIndex = iUserDefinedBand;
        // '--> required in TKeyerMainForm::CB_BandClick(),
        //      because the text displayed in the "cell" doesn't contain
        //      everything we need to switch to the band/frequency
        //      (the text doesn't even contain the accurate frequency) !
      } // end if < matching BAND (dwBitmask) >
   } // end for ( iUserDefinedBand .. )


  pCombo->Items->Add( sz80ItemText ); // <- increments pCombo->Items->Count
  // Select this item in the combo box if it's the "currently selected band".
  // ex: if( dwBitmask & dwSelectedBand ) ...
  // Since the addition of USER-DEFINED BANDS (or sub-bands), this isn't
  // as trivial as it used to be. For example, if the current VFO frequency
  // is in one of the "UK 60 m band fragments", show e.g. "60 m UK",
  // if that frequency range was loaded from RCWKeyer_Bands.txt .
  // Thus, what matters for pCombo->ItemIndex is the CURRENT FREQUENCY,
  // but not a matching bits in the hard-coded, worldwide amateur radio bands:
  if( (pCombo->ItemIndex < 0 ) && (iRow < pCombo->Items->Count) )
   { if( (pRC->dblVfoFrequency >= dblFmin_Hz) && (pRC->dblVfoFrequency <= dblFmax_Hz) )
      { pCombo->ItemIndex = iRow; // index ZERO for the FIRST item in the list, etc
      }
   }

} // end KeyerGUI_AppendFreqRangeToBandCombo()

//---------------------------------------------------------------------------
void KeyerGUI_FillComboWithBandList(
        T_RigCtrlInstance *pRC, // [in] 'Rig Control' instance, with e.g. the current operating frequency for the "current band" indicator
        TComboBox *pCombo )     // [out] VCL combo box with bands or even band-stacking registers
  // Unlike the list of band stacking registers displayed via main menu,
  // "Settings" .. "List rig control parameters on Debug tab",
  // the list generated by KeyerGUI_FillComboWithBandList() may contain entries from...
  //   * pRC->UserDefinedBands[] (for example, "sub-bands" that the RIG isn't aware of,
  //          loaded from RCWKeyer_Bands.txt via RigCtrl_LoadBandsAndFrequenciesFromFile() )
  //   * pRC->BandStackingRegs[], with the most recent frequencies that
  //          *THE RIG* has been on (at least modern Icom rigs).
{
  int   iBandIndex, iUserDefdBand, iRow, iColumn;
  DWORD dwBitmask, dwStackedBand;
  const char *pszBandName;
  double dblFmin_Hz, dblFmax_Hz;
  T_RigCtrlUserDefinedBand  *pUserDefdBand;
  T_BandComboInfoTableEntry *pInfoTableEntry;


  DWORD dwAvailableBands = RigCtrl_GetAvailableBands(pRC);
        // '--> bit combination of amateur radio bands,  e.g. for IC-9700:
        // 0x34000 = RIGCTRL_BAND_2M   (bit 14)
        //         | RIGCTRL_BAND_70CM (bit 16)
        //         | RIGCTRL_BAND_23CM (bit 17)
        // These up to 32 "amateur radio band bit numbers" don't have anything
        // in common with the array index for "user defined bands or sub-bands".


  // On this occasion, allow the "TComboBox" to get WIDER when DROPPED DOWN :
  SendMessage( pCombo->Handle,CB_SETDROPPEDWIDTH, 440/*MinimumWidthInPixels*/, 0);
  // ,---------------------------------------------'
  // '--> Should be wide enough for the BAND NAME (like "40 m")
  //      AND three "Band Stacking Registers" that modern Icoms radios like
  //      the IC-7300 store automatically, with the "most recent frequencies"
  //      that the operator has visited. (At the time of this writing, it wasn't
  //                                      clear WHEN EXACTLY the radio does this).
  //      Alternatively (without band-stacking registers), the combo list
  //      may show frequency limits or presets DEFINED BY THE SYSOP.
  //      With the combo's font set to "Courier New", size EIGHT,
  //      each character had a fixed width of NINE(!) pixels .. but ymmv .
  //  Icom's own 'Band Stacking Register' display on an IC-7300 looked
  //  like this (and this is what we're going to mimick with the TComboBox):
  //      " 50.085 CW    50.095 CW    50.135 USB"
  //      |<-12 chars->|<-12 chars->|<-12 chars->|
  //   In addition, for the NON-DROPPED-DOWN combo list,
  //   the begin of each line shall indicate the BAND NAME, e.g. "80 m: " (6 chars).
  //    ( 6 + 3 * 12 ) characters * 9 pixels_per_character = 396+x, say 440 pixels.

  pCombo->Clear();
  pCombo->ItemIndex = -1; // nothing selected yet

  // To know which rows and columns have been filled with which
  // 'band stacking registers', 'user-defined band', etc later in CB_BandClick(),
  // use an extra array :
  for( iRow=0; iRow<KEYER_GUI_MAX_BANDS; ++iRow )
   { for( iColumn=0; iColumn<KEYER_GUI_MAX_COLUMNS_PER_BAND ; ++iColumn )
      { pInfoTableEntry = &KeyerGUI_BandComboInfoTable[iRow][iColumn];
        pInfoTableEntry->iBandStackingRegIndex = -1; // it's not a reference to a BAND STACKING REGISTER (a la Icom) yet
        pInfoTableEntry->iUserDefinedBandIndex = -1; // it's not a reference to a simpler 'user-defined band' entry as well.
      }
   } // end < loop to clear the entire KeyerGUI_BandComboInfoTable[][] >
  KeyerGUI_nRowsInBandComboInfoTable = 0; // no valid entries in KeyerGUI_BandComboInfoTable[][] yet


  // Try all 32 possible "band-bit" indices, but also respect the maximum
  //  of <KEYER_GUI_MAX_BANDS> rows in KeyerGUI_BandComboInfoTable[iRow][iColumn] :
  for( iBandIndex = 0; iBandIndex<=31; ++iBandIndex )
   { // In each of these "outer" loops, MULTIPLE entries may be added to the
     // combo list ! See "inner" loop, iterating through pRC->iNumUserDefinedBands further below.
     dwBitmask = 1UL << iBandIndex; // -> RIGCTRL_BAND_160M (bit 0), etc etc ..
     RigCtrl_BandToFrequencyRange( dwBitmask, &dblFmin_Hz, &dblFmax_Hz );  // <- this may be "wider" than the user defined band and the rig capabilities !
     pszBandName = SL_GetStringFromTokenList( RigCtrl_BandNames, dwBitmask );
     if( (dwBitmask & dwAvailableBands)!=0 ) // only if THE RIG supports this band..
      {
        // If there are NO user-defined bands, accept whatever RigCtrl_GetAvailableBands() permits,
        // and restict operation to whatever RigCtrl_BandToFrequencyRange() has delivered:
        if( pRC->iNumUserDefinedBands <= 0 )
         { KeyerGUI_AppendFreqRangeToBandCombo( pRC, pCombo,
                              dblFmin_Hz, dblFmax_Hz, pszBandName );
         } // end if < no USER-DEFINED bands ? >
        else // pRC->iNumUserDefinedBands > 0 ->
         {
           // If there *are* user-defined bands, on CERTAIN BANDS, accept a wider
           // range of frequencies than RigCtrl_GetAvailableBands() permits:
           if( (dblFmin_Hz >= 5e6) && (dblFmin_Hz <= 5.7e6) ) // somewhere "near" a 60 meter band ?
            {   dblFmin_Hz =  5e6;    // this is an ugly kludge of course, but...
                dblFmax_Hz =  5.7e6;  // "wide enough" for all those crazy tiny fractions, or even "channels" on 60 m
            }

           // Inner loop to possibly add MULTIPLE SUB-BANDS within dblFmin_Hz and dblFmax_Hz
           //      to KeyerGUI_BandComboInfoTable[iRow++] :
           for( iUserDefdBand=0; iUserDefdBand<pRC->iNumUserDefinedBands; ++iUserDefdBand )
            { pUserDefdBand = &pRC->UserDefinedBands[iUserDefdBand];
              if(   (pUserDefdBand->dblFmin_Hz >= dblFmin_Hz )    // user defined (sub-)band contained in the current band ?
                 && (pUserDefdBand->dblFmax_Hz <= dblFmax_Hz ) )
               { // (*) If an entry in pRC->UserDefinedBands[] doesn't have a NAME,
                 //     the BAND NAME from RigCtrl_BandNames[] will be used instead.
                 // Only add band-stacking entries and "individual frequencies" in THE CURRENT SUB-BAND :
                 KeyerGUI_AppendFreqRangeToBandCombo( pRC, pCombo,
                    pUserDefdBand->dblFmin_Hz,
                    pUserDefdBand->dblFmax_Hz,
                    pUserDefdBand->sz16Name[0] ? pUserDefdBand->sz16Name : pszBandName );
               } // end if < user-defined band/sub-band with a NAME >
            }   // end for( iUserDefdBand ..
         }     // end else < pRC->iNumUserDefinedBands ? >
      }       // end if( (dwBitmask & dwAvailableBands)!=0 )
   }         // end for < all 32 possible bits in dwAvailableBands >
  KeyerGUI_nRowsInBandComboInfoTable = pCombo->Items->Count;
} // end KeyerGUI_FillComboWithBandList()

//---------------------------------------------------------------------------
int KeyerGUI_FrequencyToBandComboRow( double dblFreq_Hz ) // -> iRow in KeyerGUI_BandComboInfoTable[iRow][iColumn]
  // [in] dblFreq_Hz : "VFO" frequency in Hertz
  // [in] KeyerGUI_BandComboInfoTable[iRow:0..KeyerGUI_nRowsInBandComboInfoTable-1][iColumn=0]
  // [return] -1 if the frequency is not contained in any of the selectable bands,
  //          0 .. KEYER_GUI_MAX_BANDS-1 when successful (return = iRow).
  //          In that case, KeyerGUI_BandComboInfoTable[iRow][0] can be used
  //          to retrieve the entire frequency range.
  // Caller: TKeyerMainForm::Timer1Timer() -> UpdateTrxDisplay(),
  //           which occasionally checks if the "VFO frequency" hasn't slipped
  //           from one band to another via local VFO knob, or remotely, etc.

{ int iRow;
  T_BandComboInfoTableEntry *pInfoTableEntry;

  for( iRow=0; (iRow<KEYER_GUI_MAX_BANDS) && (iRow<KeyerGUI_nRowsInBandComboInfoTable); ++iRow )
   { pInfoTableEntry = &KeyerGUI_BandComboInfoTable[iRow][0];
     if( (dblFreq_Hz >= pInfoTableEntry->dblFmin_Hz) && (dblFreq_Hz <= pInfoTableEntry->dblFmax_Hz) )
      { return iRow;
      }
   }
  return -1; // arrived here ? Frequency not found in any of the 'band selection' table rows !
} // end KeyerGUI_FrequencyToBandComboRow()

//---------------------------------------------------------------------------
int GetSelectedTokenValueFromComboBox( TComboBox *pCombo,
        const T_SL_TokenList *pTokens1, // [in] e.g. C_TL_None_Zero (with string = "None", value = zero)
        const T_SL_TokenList *pTokens2, // [in] e.g. SerialPortInputSignals_Key
        const T_SL_TokenList *pTokens3) // [in] e.g. SerialPortInputSignals_Radio, or NULL for only *one* list
  // If the selected text in the combo list doesn't match ANY of the
  // strings in the three token lists, this function returns
  // GUI_ERROR_NO_MATCHING_ENTRY to let THE CALLER decide what to use.
# define GUI_ERROR_NO_MATCHING_ENTRY -99999
{
  int   iList;
  const T_SL_TokenList *pTokens;

  for( iList=1; iList<=3; ++iList)
   { switch(iList)
      { case 1: pTokens = pTokens1; break;
        case 2: pTokens = pTokens2; break;
        case 3: pTokens = pTokens3; break;
        default: pTokens = NULL; break;
      }
     if( pTokens != NULL ) // check the strings from another list ?
      { while( pTokens->pszKeyword != NULL ) // not the end of the list yet ?
         {
           if( pCombo->Text == AnsiString( pTokens->pszKeyword ) )
            { return pTokens->iTokenValue;
            }
           ++pTokens;
         }
      }
   }
  return GUI_ERROR_NO_MATCHING_ENTRY;  // No valid selection ? -> let THE CALLER handle this !
} // end GetSelectedTokenValueFromComboBox()

//---------------------------------------------------------------------------
BOOL SelectComboItemByDecimalValue( TComboBox *pCombo, int iValue )
{ int i, n, iSelItem=-1;
  n = pCombo->Items->Count;
  for(i=0; i<n; ++i)
   { if( StrToIntDef( pCombo->Items->Strings[i], 0) == iValue )
      { iSelItem = i;
        break;
      }
   }
  if( iSelItem>=0 )
   { pCombo->ItemIndex = iSelItem;
     return TRUE;
   }
  else
   { return FALSE;
   }
}

//---------------------------------------------------------------------------
BOOL SelectComboItemByHexadecimalValue( TComboBox *pCombo, int iValue )
{ int i, n, iSelItem=-1;
  char sz40Text[44];
  const char *pszSrc;
  AnsiString s;  // in BCB V12, only an "AnsiString" returns a char * from c_str() !
                 // in BCV V6, there was no UnicodeString at all ..
  n = pCombo->Items->Count;
  for(i=0; i<n; ++i)
   { s = pCombo->Items->Strings[i];
     SL_strncpy( sz40Text, s.c_str(), 40 );
     // -> BCB V12 "Athens" : [bcc32c Error] no matching function for call to 'SL_strncpy'
     // >  StringLib.h(194): candidate function not viable:
     // > no known conversion from 'System::WideChar *' (aka 'wchar_t *')
     // > to 'const char *' for 2nd argument
     pszSrc = sz40Text;
     if( SL_SkipString( (const char**)&pszSrc, "0x" ) ) // looks like a HEX number ...
      { if( (int)SL_ParseHex( &pszSrc, 4/*nMaxDigits*/) == iValue )
         { iSelItem = i;
           break;
         }
      }
   }
  if( iSelItem>=0 )
   { pCombo->ItemIndex = iSelItem;
     return TRUE;
   }
  else
   { pCombo->ItemIndex = -1;
     sprintf( sz40Text, "0x%02X", (unsigned int) iValue );
     pCombo->Text = sz40Text; // works with TComboBox.Style=csDropDown, not with csDropDownList !
     return FALSE;
   }
}

//---------------------------------------------------------------------------
int  GetSerialPortSignalIndexFromComboBox( TComboBox *pCombo )
  // Return value : KEYER_SIGNAL_INDEX_... , negative when inverted, 0 = "none"
{
  int iSelectedTokenValue = GetSelectedTokenValueFromComboBox( pCombo,
        C_TL_None_Zero,                // [in] const T_SL_TokenList *pTokens1
        SerialPortInputSignals_Key,    // [in] const T_SL_TokenList *pTokens2
        SerialPortInputSignals_Radio); // [in] const T_SL_TokenList *pTokens3
  if( iSelectedTokenValue == GUI_ERROR_NO_MATCHING_ENTRY )
   { iSelectedTokenValue = GetSelectedTokenValueFromComboBox( pCombo,
        SerialPortOutputSignals_Key,   // [in] const T_SL_TokenList *pTokens1
        SerialPortOutputSignals_Radio, // [in] const T_SL_TokenList *pTokens2
        NULL );                        // [in] const T_SL_TokenList *pTokens3
   }
  return iSelectedTokenValue;
}

//---------------------------------------------------------------------------
int ParseSerialPortNumberFromText( const char *pszSource )
   // Returns a non-negative serial port number when successful,
   //         otherwise -1 (also for the entry named "NONE", in combo boxes)
{
  int i;
  if( strncmp( pszSource, "NONE", 4) == 0 )
   { return -1;
   }
  if( pszSource[0] != '\0' )
   { // Does the stupid sscanf() parse the '9' in "COM9 (invalid)" ? [2015-01: yes]
     i =  UTL_ParseNumberFromComPortName( pszSource );
     if( i >= 0 )
      { return i;
      }
   }
  return -1;  // invalid COM port number !

} // end ParseSerialPortNumberFromText()


//---------------------------------------------------------------------------
double KeyerGUI_GetFrequencyFromEditField( TEdit *pEd, // [in] edit field with frequency
                 int iSelectedCharIndex,     // [in] cursor position within the string; 0=first char; ex: pEd->SelStart+1
                 double *pdblFreqIncrement ) // [out] 10^N, f(cursor_position)
  // Accepts a string from the VFO frequency input field (or similar fields)
  // in different formats, including 'technical' notations like 12.345 kHz
  //                                                         or 7.0374 MHz .
  // For incrementing / decrementing the frequency via cursor up/down keys
  //      or mouse wheel, the function may pass back the position of the
  //      text cursor (aka 'caret') with in the TEdit, and the value to add/
  //      subtract on each cursor up/down event. Examples :
  //      char index:  0123456789ABCDEF
  //      pEd->Text = "   7.012 345 MHz"  (note the spaces, also "in between")
  //                            '------- blinking caret or underscore HERE
  //   -> [out] pdblFreqIncrement = 100 [Hz]
  //
  //
{
  char c, szA[44], *cp;
  int i, slen, nDigits, iExponent;
  int iExponentFromTechUnit=0, nDecimalPlaces=0;
  double dblFreqHz, dblUpperPart, dblLowerPartMult;
  BOOL   fNegative = FALSE;
  BOOL   fGotDecimalPlaces = FALSE;
  AnsiString s;

  // ex: SL_strcpy_trunc(szA, sizeof(szA), pEd->Text.c_str(), 39 );
  // When compiled with BCB V12 "Athens" :
  // > [bcc32c Error] no matching function for call to 'SL_strcpy_trunc'
  // > StringLib.h(196): candidate function not viable: no known conversion
  // > from 'System::WideChar *' (aka 'wchar_t *') to 'const char *' for 3rd argument
  s = pEd->Text;  // only an AnsiString's "c_str()" returns a "char *" !
  SL_strcpy_trunc(szA, sizeof(szA), s.c_str(), 39 );

  // Convert string to decimal, ignoring spaces:
  dblFreqHz = dblUpperPart = 0.0;
  dblLowerPartMult = 0.0;  // there's no "lower part" to be added later
  slen = 0;
  cp = szA;
  while( *cp == ' ' )
   { ++cp;
     ++slen;
   }
  if( *cp == '-' )
   { ++cp;
     ++slen;
     fNegative = TRUE;
   }
  while( *cp != '\0' )
   { if( ((c=*cp)>='0') && (c<='9') )
      { dblFreqHz = 10.0*dblFreqHz + (c-'0');
        dblLowerPartMult /= 10.0;
        if( fGotDecimalPlaces )
         { ++nDecimalPlaces;
         }
      }
     else  // a "technical" exponent like 'k', 'M', 'G' ?
      { if( c=='.' || c=='k' || c=='M' || c=='G' || c=='m' || c=='u'  )
         { // Combine the previous 'upper' and 'lower' part into a single new 'upper' part,
           // and begin a new string->float conversion for the rest (fractional part):
           if( dblLowerPartMult > 0.0 )  // fraction from k(Hz), M(Hz), or G(Hz) ?
            { dblFreqHz *= dblLowerPartMult;
            }
           dblUpperPart += dblFreqHz; // the sum now completely in the "upper part" (incl. fraction)
           dblFreqHz    = 0;        // begin new conversion for the fractional part
           dblLowerPartMult = 1.0;  // factor for trailing digits (will be divided by ten with each new digit)
         }
        switch( c )
         {
           case '.':  // already parsed number was in HERTZ (or whatever), the rest is the fractional part:
              dblFreqHz         = 0;
              dblLowerPartMult  = 1.0;  // factor for trailing digits (will be divided by ten with each new digit)
              nDecimalPlaces    = 0;
              fGotDecimalPlaces = TRUE;
              break;
           case 'k':  // already parsed number was in KILOHERTZ, rest is a fraction:
              iExponentFromTechUnit = 3;  // required later for inc/dec
              dblUpperPart *= 1000.0;
              dblLowerPartMult = 1000.0;  // factor for trailing digits (will be divided by ten with each new digit)
              break;
           case 'M':  // already parsed number was in MEGAHERTZ, rest is a fraction:
              iExponentFromTechUnit = 6;  // required later for inc/dec
              dblUpperPart *= 1E6;
              dblLowerPartMult = 1E6;
              break;
           case 'G':  // already parsed number was in GIGAHERTZ, rest is a fraction:
              iExponentFromTechUnit = 9;  // required later for inc/dec
              dblUpperPart *= 1E9;
              dblLowerPartMult = 1E9;
              break;
           case 'm':  // already parsed number was in MILLIHERTZ, rest a tiny fraction:
              iExponentFromTechUnit = -3;
              dblUpperPart *= 1E-3;
              dblLowerPartMult = 1E-3;
              break;
           case 'u':  // already parsed number was in MICROHERTZ, rest a ridiculous fraction:
              iExponentFromTechUnit = -6;
              dblUpperPart *= 1E-6;
              dblLowerPartMult = 1E-6;
              break;
           default :  // e.g. c='-' if the frequency in Ed_VFO is invalid
              break;
         }
      }
     ++cp;
     ++slen;
   }
  // Examples:
  //  (a) "3M5" -> UpperPart=3000000, LowerPart=5, LowerPartMult=100000 .
  //  (b) "---.----- MHz" ->  fNegative=TRUE, UpperPart=0, dblFreqHz=0, LowerPartMult=0 .
  if( dblLowerPartMult > 0.0 )  // fraction from k(Hz), M(Hz), or G(Hz) ?
   { dblFreqHz *= dblLowerPartMult;
   }
  dblFreqHz += dblUpperPart;
  if( fNegative )
   { dblFreqHz = -dblFreqHz;
   }
  //
  // Convert the current cursor position into a power of ten for inc/dec-editing.
  // EXAMPLE: szA="123", slen=3, i=2 -> iFreqIncrement=1 .
  //          (in this case, there's NO digit to examine, because
  //           the RIGHTMOST digit is selected for inc/dec-editing)
  i = iSelectedCharIndex+1; // cursor position within the string; 0=first char
  // Because the cursor shall be RIGHT NEXT to the incremented/decremented digit,
  //    a cursor after the 'end' of the string is meaningless, i.e. :
  //    with sLen==1, the only valid cursor position (i) is ZERO, not ONE:
  if( i>slen )  i=slen;
  if( i<0 )     i=0;
  //                      Cursor (left of the digit to increment/decrement,
  //                       |  usually with "SelLength"=1 to highlight digit)
  //                       |
  //  Example:  szA = "   12 345 567"  slen=13, nDigits=8
  //                   |   |       |
  //                   |   |       |
  //               szA[0] i=4  szA[12]
  //  Note that there may be SPACES or other thousand-separators
  //  in the string, so COUNT THE DIGITS here to find the decimal place:
  nDigits = 0; // ... i.e. dblFreqIncrement = 10 ^ iCursorDigitPos
  while( i<slen )
   { if( szA[i]>='0' && szA[i]<='9' )  // only count the DIGITS, nothing else !
      { ++nDigits;
      }
     ++i;
   }
  iExponent = nDigits + iExponentFromTechUnit - nDecimalPlaces;
  if( pdblFreqIncrement != NULL ) // caller wants the power of ten (f(cursor-pos) )
   { *pdblFreqIncrement = pow10( iExponent ); // pow10 takes an INTEGER argument; HERE that's ok
   }

  return dblFreqHz;
} // end KeyerGUI_GetFrequencyFromEditField()

//---------------------------------------------------------------------------
const char *KeyerGUI_GuiItemCodeToString( // .. for imporant Visual Components in the RCW Keyer GUI ..
        int iItem ) // [in]     e.g. KEYER_GUI_ITEM_EDIT_MAIN_VFO
                    // [return] e.g. "Ed_VFO" (at least that was the name when RCW-Keyer was written in Borland C++Builder 6)
{
  const char *pszResult = SL_GetStringFromTokenList( KeyerGUI_VisualComponentNames, iItem );
  static char sz15Unknown[16];

  if( pszResult != NULL )
   {  return pszResult;
   }
  // Arrived here ? That shouldn't happen, but anyway, return the NUMERIC item-ID as a string:
  sprintf( sz15Unknown, "?%d?", (int)iItem );
  return sz15Unknown;
} // end KeyerGUI_GuiItemCodeToString()

//---------------------------------------------------------------------------
void KeyerGUI_OnSetFocus( // implements some keyboard-focus-switching actions without VCL/Qt/whatever-dependencies.
        int iNewFocusedItem ) // [in] e.g. KEYER_GUI_ITEM_EDIT_MAIN_VFO, etc (defined in Keyer_GUI_no_VCL.h)
        // [in,out] KeyerGUI_iFocusSwitchCountdownTimer_ms, KeyerGUI_iCurrentlyFocusedItem
  // No need to pass in any Borland-specific stuff here (like the "main form"-instance)
  // because one instance of the Remote CW Keyer has exactly ONE main window/"form"/etc.
  // In VCL, Vcl.Forms.TScreen.OnActiveControlChange could be used for that purpose.
  //                   '--> Type of GLOBAL VARIABLE "Screen", not a part of the main form !
  //                        WB decided NOT to use that, but call KeyerGUI_OnSetFocus()
  //                        from individual 'Visual Components' when they RECEIVE the focus.
{
  int iPrevFocusedItem = KeyerGUI_iCurrentlyFocusedItem;
  if( iNewFocusedItem == iPrevFocusedItem ) // no CHANGE of the keyboard focus,
   { // but most likely called on a USER ACTION like pressing another key,
     // selecting a different item in the currently focused combo box (w/o focus switch),
     // typing a new digit into the VFO frequency field, etc :
     KeyerGUI_iFocusSwitchCountdownTimer_ms = 2000; // don't AUTOMATICALLY switch the focus away for another XXXX milliseconds !
   }
  else // a USER ACTION really switches the focus to a DIFFERENT "Visual Component"/"Widget"/etc :
   {
     KeyerGUI_iCurrentlyFocusedItem = iNewFocusedItem;
     KeyerGUI_iFocusSwitchCountdownTimer_ms = 3000; // don't AUTOMATICALLY switch the focus away for another YYYY milliseconds !

     if( MyCwNet.cfg.iDiagnosticFlags & CWNET_DIAG_FLAGS_SHOW_GUI_EVENTS ) // show what's going on ?
      { ShowError( ERROR_CLASS_INFO | SHOW_ERROR_TIMESTAMP, "Switched focus from %s to %s",
           KeyerGUI_GuiItemCodeToString(iPrevFocusedItem), KeyerGUI_GuiItemCodeToString(iNewFocusedItem) );
      }


     // Anything special to do when the PREVIOUSLY focused item loses it ?
     switch( iPrevFocusedItem )
      { case KEYER_GUI_ITEM_MAIN_MENU_FILE      : // keyboard focus now in the MAIN MENU.. any of these items:
        case KEYER_GUI_ITEM_MAIN_MENU_SETTINGS  :
        case KEYER_GUI_ITEM_MAIN_MENU_FUNCTIONS :
        case KEYER_GUI_ITEM_MAIN_MENU_HELP      :
        case KEYER_GUI_ITEM_BTN_APPLY_AND_START : // switching AWAY from the "Apply and Start"-button on the "Config" tab
        case KEYER_GUI_ITEM_SCR_KEYER_SPEED     : // switching AWAY from the "scroller" / "trackbar" / whatever for the KEYER SPEED IN WPM
        case KEYER_GUI_ITEM_EDIT_KEYER_SPEED    : // switching AWAY from the numeric edit field for the KEYER SPEED IN WPM
        case KEYER_GUI_ITEM_SCR_AUDIO_IN_VOLUME : // "scrollbar" for the audio INPUT volume
        case KEYER_GUI_ITEM_SCR_AUDIO_OUT_VOLUME: // "scrollbar" for the audio OUTUT volume
        case KEYER_GUI_ITEM_SCR_SIDETONE_VOLUME : // "scrollbar" for the SIDETONE volume (signal added to the audio output)
        case KEYER_GUI_ITEM_SCR_NETWORK_IN_VOLUME:// "scrollbar" for the NETWORK AUDIO volume (signal added to the audio output)
        case KEYER_GUI_ITEM_RBTN_NETWORK_OFF    : // "radio buttons"(!) for 'Network functionality' ..
        case KEYER_GUI_ITEM_RBTN_NETWORK_CLIENT : // switching AWAY FROM this ..
        case KEYER_GUI_ITEM_RBTN_NETWORK_SERVER :
        case KEYER_GUI_ITEM_EDIT_NETWORK_STATUS :
        case KEYER_GUI_ITEM_EDIT_NETWORK_ON_KEY :
        case KEYER_GUI_ITEM_EDIT_ERROR_HISTORY  :
        case KEYER_GUI_ITEM_EDIT_MYCALL         :
        case KEYER_GUI_ITEM_EDIT_MAIN_VFO       : // switching AWAY FROM the edit field for the main VFO frequency, on the "TRX" tab
        case KEYER_GUI_ITEM_COMBO_MAIN_OP_MODE  : // switching AWAY FROM the combo box(?) for the 'opmode' (CW, CWN, USB, LSB), on the "TRX" tab
        case KEYER_GUI_ITEM_COMBO_BAND_SWITCH   : // switching AWAY FROM the combo box(?) to switch to a different BAND or BAND STACKING REGISTER

        default:  // switching AWAY from an array-like visible control (e.g. edit field for a keyer memory) ?
           if( (iPrevFocusedItem>=KEYER_GUI_ITEM_EDIT_KEYER_MEMORY_1) && (iPrevFocusedItem<=KEYER_GUI_ITEM_EDIT_KEYER_MEMORY_LAST) )
            {
            }
           else // e.g. switching the focus AWAY FROM 'KEYER_GUI_ITEM_NONE' ->
            { // do NOTHING
            }
           break;
      } // end switch( iPrevFocusedItem )
     // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
     // Above: Special actions when switching AWAY FROM < iPrevFocusedItem >
     // Below: Special actions when switching TO        < iNewFocusedItem > ?
     // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


   } // end else < iNewFocusedItem != iPrevFocusedItem >

} // end KeyerGUI_OnSetFocus()


//---------------------------------------------------------------------------
BOOL KeyerGUI_SwitchFocus(  // VCL-free API, but VCL-dependent implementation !
        int iNewFocusedItem ) // [in] e.g. KEYER_GUI_ITEM_EDIT_MAIN_VFO, etc (defined in Keyer_GUI_no_VCL.h)

{
  BOOL  fResult = FALSE;   // return value : TRUE=success, FALSE=failed
  const char *pszComponentName = KeyerGUI_GuiItemCodeToString(iNewFocusedItem);
  // int   iPrevFocusedItem = KeyerGUI_iCurrentlyFocusedItem;
  int   i;

#ifdef __BORLANDC__  // here the specific stuff for Borland / C++Builder / VCL :
  TForm      *pForm = KeyerGUI_GetMainForm();
  TComponent *pComp; // pointer to ONE of the Visual Components on the above "form" (GUI window)
  AnsiString sCompName;
#endif // __BORLANDC__ ?

  if( (pszComponentName[0] != '\0') && (pszComponentName[0] != '?') )
   { // Succesfully retrieved the NAME of the visual component,
     // so try to find it in the MAIN FORM. This is where the VCL-dependency starts:
#   ifdef __BORLANDC__  // here the specific stuff for Borland / C++Builder / VCL :
     sCompName = AnsiString( pszComponentName );
     if( pForm != NULL )
      { for( i=0; (i<pForm->ComponentCount); i++ )
         { pComp = pForm->Components[i];
           // ,-----------'
           // '--> Use Components to access any of the components owned by this
           //    > component, such as the components owned by a form.
           if( pComp->Name == sCompName )
            {  ((TWinControl*)pComp)->SetFocus();
               fResult = TRUE;
            }
         } // end for( iComp=0; (iComp<pForm->ComponentCount); iComp++ )
      }
#   endif // __BORLANDC__ ?

   } // end if < retrieved the NAME of the visual component that shall receive the keyboard focus >

#if(0)  // unnecessary because the VCL calls the "OnFocus" event handler also when calling TWinControl.SetFocus() 
  if( fResult && (MyCwNet.cfg.iDiagnosticFlags & CWNET_DIAG_FLAGS_SHOW_GUI_EVENTS) ) // show what's going on ?
   { ShowError( ERROR_CLASS_INFO | SHOW_ERROR_TIMESTAMP, "Program switched focus from %s to %s",
           KeyerGUI_GuiItemCodeToString(iPrevFocusedItem), KeyerGUI_GuiItemCodeToString(iNewFocusedItem) );
   }
#endif


  return fResult;
} // end KeyerGUI_SwitchFocus()


//---------------------------------------------------------------------------
void KeyerGUI_On50msTimer(void) // VCL-independent part of 'periodic updates' of the GUI:
  // Periodically called from the main task / main thread, approximately every 50 milliseconds.
{
  ++KeyerGUI_dwTimerTicks_50ms;   // <- Incremented every 50 milliseonds. Individual bits are used as e.g. 'blink flag' anywhere in the GUI

  // Automatically switch the keyboard focus back to the VFO edit field ?
  if( KeyerGUI_iFocusSwitchCountdownTimer_ms > 0 )
   { KeyerGUI_iFocusSwitchCountdownTimer_ms -= 50;
     if( KeyerGUI_iFocusSwitchCountdownTimer_ms <= 0 ) // countdown timer for the automatic keyboard-focus-switch JUST expired ?
      { if( KeyerGUI_fMaySwitchToVfoEditField )   // allow switching BACK to the VFO edit field ?
         { switch( KeyerGUI_iCurrentlyFocusedItem )
            { case KEYER_GUI_ITEM_COMBO_BAND_SWITCH  :
              case KEYER_GUI_ITEM_COMBO_MAIN_OP_MODE :
                 // Switch the focus from any of the controls above to the
                 // edit field for the VFO frequency, to receive e.g. mousewheel events
                 // and cursor keys (instead of accidentally scrolling to A DIFFERENT BAND):
                 KeyerGUI_SwitchFocus( KEYER_GUI_ITEM_EDIT_MAIN_VFO );
                 // Because the above focus-switched doesn't involve setting the
                 // the TEXT CARET to the "clicked digit" : ...
                 KeyerGUI_SetDefaultTextCaretPosInVFOInputField();
                 break;
              default:  // keyboard focus anywhere else ->
                 break; // do NOTHING (at least not here)
            } // end switch( KeyerGUI_iCurrentlyFocusedItem )
         }   // end if( KeyerGUI_fMaySwitchToVfoEditField )
      }     // end if( KeyerGUI_iFocusSwitchCountdownTimer <= 0 )
   }       // end if( KeyerGUI_iFocusSwitchCountdownTimer > 0 )
}         // end KeyerGUI_On50msTimer()


//---------------------------------------------------------------------------
BOOL ScopeDisplay_Init(T_ScopeDisplay *pScopeDisplay, int iWidth, int iHeight)
{
  int i;

  memset(pScopeDisplay,0,sizeof(T_ScopeDisplay) );

  pScopeDisplay->pPoints = (TPoint*)malloc( TIMING_SCOPE_NUM_SAMPLE_POINTS * sizeof(TPoint) );
  if(pScopeDisplay->pPoints==NULL) // oops .. malloc() failed !
   { return FALSE;
   }
  pScopeDisplay->nPixelLines = iHeight;
  pScopeDisplay->ppdwPixelLines = (T_RGBColor**)malloc( pScopeDisplay->nPixelLines * sizeof(T_RGBColor*) );
  if(pScopeDisplay->ppdwPixelLines==NULL) // oops .. another malloc() failure
   { ScopeDisplay_Free( pScopeDisplay ); // bail out without a memory leak
     return FALSE;
   }
  pScopeDisplay->pBitmap = new Graphics::TBitmap;
  pScopeDisplay->pBitmap->PixelFormat = pf32bit; // we use 32 bit/pixel / T_RGBColor
  pScopeDisplay->pBitmap->Width = iWidth;
  pScopeDisplay->pBitmap->Height= iHeight;
  for(i=0; i<pScopeDisplay->nPixelLines; ++i) // determine the address of "ScanLines" (pixel addresses for each line in the bitmap)
   { pScopeDisplay->ppdwPixelLines[i] = (T_RGBColor*)pScopeDisplay->pBitmap->ScanLine[i];
     // ,-------------------------------|___________|
     // '--> Without this, Borland C++ complained "cannot convert void* to 'T_RGBColor*'
   }

  return TRUE;
} // end ScopeDisplay_Init()

//---------------------------------------------------------------------------
void ScopeDisplay_Free(T_ScopeDisplay *pScopeDisplay) // Cleans up after ScopeDisplay_Init()
{
  pScopeDisplay->nPixelLines = pScopeDisplay->nPoints = 0;
  if(pScopeDisplay->ppdwPixelLines!=NULL)
   { free( pScopeDisplay->ppdwPixelLines );
     pScopeDisplay->ppdwPixelLines = NULL;  // replace INVALID POINTER with a NULL POINTER
   }
  if(pScopeDisplay->pPoints != NULL )
   { free( pScopeDisplay->pPoints );
     pScopeDisplay->pPoints = NULL;
   }
  if(pScopeDisplay->pBitmap != NULL )
   { delete pScopeDisplay->pBitmap; // this is a C++/"VCL" object, not a simple memory block !
     pScopeDisplay->pBitmap = NULL;
   }

} // end ScopeDisplay_Free()


//---------------------------------------------------------------------------
void ScopeDisplay_AppendPoint(T_ScopeDisplay *pScopeDisplay, int x, int y)
{
  if( pScopeDisplay->nPoints < TIMING_SCOPE_NUM_SAMPLE_POINTS )
   {  pScopeDisplay->pPoints[pScopeDisplay->nPoints].x   = x;
      pScopeDisplay->pPoints[pScopeDisplay->nPoints++].y = y;
   }
} // end ScopeDisplay_AppendPoint()

//---------------------------------------------------------------------------
void ScopeDisplay_DrawAudioSpectrumAndDecoderInfo(
        T_ScopeDisplay *pScopeDisplay, // [in,out] everything required to render the graphics (off-screen bitmap, etc)
        T_CwDSP_PlotterSample *pAudioCwDecoderSample, // [in] audio spectrum and other audio CW decoder parameters.
                                                      //      This pointer may be NULL if there is no valid entry
                                                      //      retrieved via CwDSP_ReadFromPlotterFifo() .
        int x1, int x2, int y1, int y2 ) // [in] drawing area within the target bitmap
        // [in] CwKeyer_TimingScope.cfg.iAudioSpectrumRefLevel_dB, iAudioSpectrumAmplRange_dB
{
  int   h,x,y,iBinIndex,iBrightness;
  int   iAudioSpectrumRefLevel_dB = CwKeyer_TimingScope.cfg.iAudioSpectrumRefLevel_dB;
  int   iAudioSpectrumAmplRange_dB= CwKeyer_TimingScope.cfg.iAudioSpectrumAmplRange_dB;
  float flt, fltBinIndex, fltBinsPerPixel;
        // 'pixels per bin' is ideally a small integer like 1 or 2, but don't bet on that
  T_RGBColor dwColor, *pdwPixels;

  UTL_LimitInteger( &y1, 0, pScopeDisplay->nPixelLines-1 );
  UTL_LimitInteger( &y2, 0, pScopeDisplay->nPixelLines-1 );
  h = 1+y2-y1;
  if( h<=1 )
   { return; // bail out (instead of dividing by zero further below)
   }
  UTL_LimitInteger( &iAudioSpectrumRefLevel_dB, -50, 50 );
  UTL_LimitInteger( &iAudioSpectrumAmplRange_dB, 6, 100 ); // avoid div-by-zero !

  fltBinsPerPixel = (float)CWDSP_AUDIO_SPECTRUM_NUM_FREQUENCY_BINS / (float)h;
  fltBinIndex = (float)(CWDSP_AUDIO_SPECTRUM_NUM_FREQUENCY_BINS-1); // <- decremented by fltBinsPerPixel in the loop below
             // (decremented because drawing begins on top, lowest 'y', but highest audio frequency)
  for(y=y1; y<=y2; ++y)
   { pdwPixels = pScopeDisplay->ppdwPixelLines[y];
     if( pdwPixels != NULL )
      { pdwPixels += x1;
        iBinIndex = (int)fltBinIndex;
        if( (pAudioCwDecoderSample == NULL ) || (iBinIndex>=CWDSP_AUDIO_SPECTRUM_NUM_FREQUENCY_BINS) )
         { // oops.. no valid data to plot !
           dwColor.dw = 0x7F7F7F; // gray output = "no data"
         }
        else if( pAudioCwDecoderSample->dblTimestamp_s <= 0.0 )
         { // oops.. invalid timestamp, invalid data !
           dwColor.dw = 0x7F7F7F; // gray output = "no data"
         }
        else // pAudioCwDecoderSample != NULL -> convert spectrum sample into a colour..
         { // using the same subroutine as for the 'radio spectrogram' in SpecDisp.cpp
           flt = pAudioCwDecoderSample->fltAudioPowerSpectrum[iBinIndex]; // power density, normalized to 0.0 .. 1.0 (not logarithmized)
           flt = SpecDisp_Power_to_dBfs( flt/* fltNormalizedPower, 0..1 */ );
           // A typical amplitude (magnitude) range for the audio CW decoder
           // is -50 to 0 dB "over" full scale,
           // and the defaults are iAudioSpectrumRefLevel_dB = 0, iAudioSpectrumAmplRange_dB = 50.
           // So ADD 50 dB to the "dB over full scale" value, plus the user-defineable
           // offset (kind of "brightness" control). Similar as Icom's "scope ref level",
           // adding a "ref level" of +20 dB makes the spectrogram very sensitive,
           //                   and -20 dB makes the spectrogram very sensitive.
           //        A "ref level" of 0 dB should give a reasonable display.
           flt = flt/*dBfs*/ + 50/*dB*/ + iAudioSpectrumRefLevel_dB/* "extra gain*/;
           iBrightness/*0..255*/ = (int)( flt * (255.0/iAudioSpectrumAmplRange_dB) );
           SpecDisp_BrightnessToWaterfallColour( g_SpecDispControl.iWFColorPalette, iBrightness, &dwColor );
         }
        for(x=x1; x<=x2; ++x)  // beautifully simple method to write directly into a bitmap's pixel matrix:
         { (pdwPixels++)->dw = dwColor.dw;
         }
      }
     fltBinIndex -= fltBinsPerPixel;
   }

} // end ScopeDisplay_DrawAudioSpectrumAndDecoderInfo()

//---------------------------------------------------------------------------
void ScopeDisplay_DrawChannelMarker(  // Draws something as sketched below:
        //    ______                                                      .
        //   |      \                                                     .
        //   |  /|   \ _________  y (begin of the "curve" with value=0)   .
        //   |   |   /                                                    .
        //   |______/                                                     .
        //   .                                                            .
        //   .                                                            .
        //   x  (horizontal position, also measured in PIXELS)            .
        //
        T_ScopeDisplay *pScopeDisplay, // [in,out] everything required to render the graphics (off-screen bitmap, etc)
        int x, int y,                  // [in] pixel coordinate (see sketch above)
        T_RGBColor dwColor,            // [in] channel-specific colour
        char cCaption )                // [in] single-character "caption", usually '1'..'9' or 'T' for 'Trigger Level'
   // Note: A sufficiently small FONT has already been set by the caller,
   //       only ONCE before drawing multiple "channel markers" in a loop !
{
#ifdef __BORLANDC__  // here the implementation for Borland/Embarcadero C++Builder/VCL :
  Graphics::TCanvas *pCanvas; // A TCanvas can be painted on. A TBitmap is only the "pixel storage". Welcome to the funny world of Borland's VCL.
  TColor oldPenColor, oldBrushColor;
  TPoint points[5];
  int    th;  // text height in pixels
  int    thh; // half text height in pixels
  char   sz4Label[4];
  if( pScopeDisplay->pBitmap != NULL ) // only if there's an off-screen bitmap for flicker-free drawing:
   { pCanvas = pScopeDisplay->pBitmap->Canvas;
     th = pCanvas->TextHeight( sz4Label ); // "Arial" with pCanvas->Font->Size = 10 gave th=19 [pixels]. Far too much.
            // "Arial" with pCanvas->Font->Size = 6  gave th=13 [pixels]. Still awful.
            // Surrendered to the ever-changing, unpredictable text-height in pixels,
            // and now using the TEXT HEIGHT also as the CHANNEL MARKER HEIGHT. Eeek.
     thh = th/2;
     points[0] = TPoint( x,    y-thh );  // upper left corner (see ASCII sketch above)
     points[1] = TPoint( x+th, y-thh );
     points[2] = TPoint( x+th+thh, y );  // arrow's head pointing right
     points[3] = TPoint( x+th, y+thh );
     points[4] = TPoint( x,    y+thh );  // lower part (remember, "y=0 is on TOP of the bitmap")
     oldPenColor   = pCanvas->Pen->Color;    // the "pen" draws the outline (thin lines)
     oldBrushColor = pCanvas->Brush->Color;  // the "brush" fills the polygon's interior
     pCanvas->Pen->Color = TColor( dwColor.dw );
     pCanvas->Brush->Color = pCanvas->Pen->Color;
     pCanvas->Polygon( points, 5-1 );
     //  ,---------------------|_|
     //  '--> Like the crazy Windows GDI function that it's based upon, Borland's
     //       Borland's VCL-Polygon() also expects the number of polygon points
     //       (in the array) MINUS ONE !
     // From the WIN32 'GDI' (not 'GDI+') documentation:
     // > The Polygon function draws a polygon consisting of two or more vertices
     // > connected by straight lines. The polygon is outlined by using the current
     // > pen and filled by using the current brush and polygon fill mode.
     // > Remarks
     // >  The polygon is closed automatically by drawing a line
     // >  from the last vertex to the first.
     // Note (by WB): Windows defines a "POINT" struct, with two members: "LONG x; LONG y;".
     //               Borland's VCL uses a "TPoint" instead, most likely compatible.
     //               Would you expect Qt to use a "QPoint" for this ? Of course not.
     sz4Label[0] = cCaption;
     sz4Label[1] = '\0';
     pCanvas->TextOutA(x+thh/2, y-thh, sz4Label );
     pCanvas->Brush->Color = oldBrushColor;
     pCanvas->Pen->Color   = oldPenColor;
   }
#else
#  error "Sorry, this module hasn't been ported to other targets yet."
#endif



} // end ScopeDisplay_DrawChannelMarker()

//---------------------------------------------------------------------------
void ScopeDisplay_AppendSamplesFromDSP( // Does what the name implies .. the "DSP" is mainly the AUDIO DSP / CW DECODER FOR AUDIO SIGNALS.
              T_KeyerTimingScope *pScope,  // [in,out] timing scope data and configuration
                                // [out] pScope->sSamplePoints[pScope->iSampleIndexFromDSP++].sDspPlotterSample
              T_CwDSP *pCwDSP ) // [in] source with audio spectra and other 'plottable values' from module CwDSP.c
{
  T_CwDSP_PlotterSample *pDspPlotterSample;
  double dblTimestamp_s, dblSecondsPerSample;
  double dblNewestTimestamp_s, dblOldestTimestamp_s;
#if( SWI_CHECK_TIMESTAMPS )
  int  i;
  long i32FftCounterStep;
#endif


  // Unless the timing scope is paused, copy as many samples
  //   as available in the audio DSP's own 'plotter FIFO',
  //   and until        pScope->iSampleIndexFromDSP
  //   catches up with  pScope->iLatestSampleIndex .
  if( !pScope->fPaused )
   {
     dblNewestTimestamp_s = CwDSP_GetTimestampOfNewestSampleInPlotterFifo(pCwDSP);
     dblOldestTimestamp_s = CwDSP_GetTimestampOfOldestSampleInPlotterFifo(pCwDSP);

     // To properly align samples from the DSP (including the audio spectrum)
     // with other channels contained in pScope->sSamplePoints[], determine
     // the TIMESTAMP of the sample at pScope->iSampleIndexFromDSP :
     dblSecondsPerSample = 1e-3 * pScope->cfg.iMillisecondsPerSample;
     dblTimestamp_s = pScope->dblTimestampAtSampleIndexZero
                    + dblSecondsPerSample * pScope->iSampleIndexFromDSP;
     // If all works as planned, dblOldestTimestamp_s <= dblTimestamp_s <= dblNewestTimestamp_s .
     // If dblTimestamp_s exceeds dblNewestTimestamp_s by a few ten milliseconds (which it often does),
     //    pScope->iLatestSampleIndex must not be modified in the loop below.
     // Instead, in a subsequent periodic call of ScopeDisplay_AppendSamplesFromDSP(),
     //    a sample with the 'wanted' timestamp (dblTimestamp_s)
     //    will have arrived from the CW audio DSP (CwDSP.c) .
     //
     while( pScope->iSampleIndexFromDSP != pScope->iLatestSampleIndex )
      { pDspPlotterSample = CwDSP_ReadFromPlotterFifo( pCwDSP, dblTimestamp_s );
        if( pDspPlotterSample == NULL ) // oops.. the requested sample (referenced by TIMESTAMP) isn't available YET, or not available ANYMORE !
         { if( dblTimestamp_s >= dblNewestTimestamp_s )
            { // sample not available YET (e.g. due to processing delay in CwDSP.c) ->
              break; // don't append it to pScope->sSamplePoints[pScope->iSampleIndexFromDSP++] now, but LATER
            }
           else
            { // sample not available because it's TOO OLD ->
              // don't brake the loop, because it doesn't make sense to try again later.
              // Instead, set the TIMESTAMP in pScope->sSamplePoints[]
              // to make this (old) displayed as "not available" in ScopeDisplay_DrawAudioSpectrumAndDecoderInfo():
              pScope->sSamplePoints[pScope->iSampleIndexFromDSP++].sDspPlotterSample.dblTimestamp_s = 0.0;
            }
         }
        else // ok, got a "plotter sample" from the DSP module with matching timestamp, so copy it:
         { pScope->sSamplePoints[pScope->iSampleIndexFromDSP++].sDspPlotterSample = *pDspPlotterSample;
#         if( SWI_CHECK_TIMESTAMPS )
           // If all works as planned, the 'FFT counter' from the audio DSP
           // should increment here, but never step back:
           if( (i=pScope->iSampleIndexFromDSP)>=2 )
            { i32FftCounterStep = pScope->sSamplePoints[i-1].sDspPlotterSample.dwFFTCounter
                                - pScope->sSamplePoints[i-2].sDspPlotterSample.dwFFTCounter;
              if( (i32FftCounterStep < 0 ) || (i32FftCounterStep > 1 ) )
               { i32FftCounterStep = i32FftCounterStep; // <-- set a breakpoint HERE
                 // 2025-04-05 : Got here with i32FftCounterStep=2
                 //                    or even i32FftCounterStep=-1 ,
                 //   when the timing scope was configured for 2 ms/sample,
                 //   and the audio-DSP provided
                 //   CwKeyer_DSP.dblPlotterSamplingInterval_s = 0.016 (16 ms) !
                 // Reason: A lot of jitter on the timestamps for the audio input
                 //   from Direct Sound (which doesn't provide timestamps itself)
                 //   in Remote_CW_Keyer\dsound_wrapper.c : DSW_QueryInputFilled().
                 //
               }
            }
#         endif // SWI_CHECK_TIMESTAMPS ?
         }
        pScope->iSampleIndexFromDSP &= (TIMING_SCOPE_NUM_SAMPLE_POINTS-1);
        dblTimestamp_s += dblSecondsPerSample;
      }
   }
  (void)dblOldestTimestamp_s; // ... assigned a value that is never used ..

} // end ScopeDisplay_AppendSamplesFromDSP()

//---------------------------------------------------------------------------
void KeyerGUI_UpdateTimingScope( // Does what the name implies :) ..
              T_KeyerTimingScope *pScope,     // [in,out] timing scope data and configuration
                                 // [out] pScope->pScope->sChannelInfo[] : Used later in KeyerGUI_HandleMouseEventInTimingScope()
              Graphics::TBitmap *pDestBitmap) // [out] graphic bitmap
              // (unfortunately, a Borland-specific VCL thing. You'll have lots of fun
              //  trying to port this to e.g. Qt one fine day.)
      // [in]  pScope->sSamplePoints[iSample].fltAnalog[iChannel] : from CwKeyer_CollectDataForTimingScope()
      // [in]  TimingScopeOverlay[] : mouse-controlled 'overlay' to measure times
      // [out] pScope->sChannelInfo[].iVerticalPos_pixels, .iCurveHeight_pixels
{
  int iTotalWidth  = pDestBitmap->Width;  // width in pixels
  int iTotalHeight = pDestBitmap->Height; // height in pixels
  int iCurveAreaHeight, iSpectrumHeight;
  int i, iChannel, iSample, x, y, h2, curve_height, yc, y_min, y_max, val, prev_x, prev_y, th, tw, tx, ty, x_wiper;
  int iSampleAtBeginOfChar;
  int iMilliseconds;
  float flt;
  double dblTimestamp_s; // timestamp to retrieve data (sample points) from the Audio CW decoder's "plotter FIFO"
  double dblEndTime_s, diff, dblSecondsPerStep;
  WORD  wCwChar;
  T_CwDSP *pCwDSP = &CwKeyer_DSP;
  T_CwDSP_PlotterSample *pAudioCwDecoderSample; // .. with a short-time audio power spectrum, etc
  T_ScopeDisplay scope_display; // wrapper for the offscreen bitmap, plus helpful stuff
                                // like an array of 'pixel line pointers' (into the bitmap)
                                // to reduce or even eliminate the number of GDI calls
                                // when drawing e.g. spectrogram lines directly into the bitmap
  const char *pszDecoded;
  AnsiString s;
# define TSCOPE_TOP_TIME_SCALE_HEIGHT    8
# define TSCOPE_BOTTOM_TIME_SCALE_HEIGHT 8
# define TSCOPE_CHANNEL_SEPARATOR_HEIGHT 16
  T_RGBColor dwColour; // R-G-B colour mix as a union (Red in bits 7..0, Green bits 15..8, Blue bits 23..16)


  HERE_I_AM__GUI();
  ScopeDisplay_AppendSamplesFromDSP( pScope, pCwDSP ); // .. to pScope->sSamplePoints[pScope->iSampleIndexFromDSP++].sDspPlotterSample

  if( ! ScopeDisplay_Init( &scope_display, iTotalWidth, iTotalHeight ) )
   { return; // oops.. something failed, most likely with memory allocation
   }

  if( pScope->cfg.iMillisecondsPerSample < 1 )
   {  pScope->cfg.iMillisecondsPerSample = 1;  // prevent endless loops below
   }

  // Already calculate the horizontal position of the "windscreen wiper",
  // which shows the current position on the non-scrolling, Radar-like sweeping display:
  x_wiper = (int)(((long)iTotalWidth * (long)pScope->iLatestSampleIndex) / (long)TIMING_SCOPE_NUM_SAMPLE_POINTS);


  // Erase the background :
  scope_display.pBitmap->Canvas->Brush->Style = bsSolid;
  scope_display.pBitmap->Canvas->Brush->Color = clBlack;
  scope_display.pBitmap->Canvas->FillRect( TRect(0,0,iTotalWidth,iTotalHeight) );
      // '--> erasing the entire bitmap was NOT the CPU hog !

  // Prepare a few parameters for the oscilloscope's screen layout :
  if( CwKeyer_TimingScope.cfg.iVisibleChannels & (1<<TIMING_SCOPE_CHANNEL_AUDIO_CW_DEC) )
   { iSpectrumHeight = CWDSP_AUDIO_SPECTRUM_NUM_FREQUENCY_BINS;
     if( iTotalHeight > (200+2*CWDSP_AUDIO_SPECTRUM_NUM_FREQUENCY_BINS) )
      { iSpectrumHeight = 2*CWDSP_AUDIO_SPECTRUM_NUM_FREQUENCY_BINS;
      }
     else
      { iSpectrumHeight = CWDSP_AUDIO_SPECTRUM_NUM_FREQUENCY_BINS;
      }
   }
  else // no audio spectrum / audio CW decoder display in the 'timing scope' ->
   { iSpectrumHeight = 0;
   }
  iCurveAreaHeight = iTotalHeight; // remaining height for the "curve area", above the audio spectrum / audio CW decoder area ..
  if( iSpectrumHeight > 0 )
   { iCurveAreaHeight -= ( iSpectrumHeight + TSCOPE_BOTTOM_TIME_SCALE_HEIGHT );
   }
  h2 = (iCurveAreaHeight-16) / TIMING_SCOPE_N_DIGITAL_CHANNELS; // height for channel, including space between channels

  // Draw optional info / warnings 'into the background' ?
  scope_display.pBitmap->Canvas->Font->Name  = "Arial";
  scope_display.pBitmap->Canvas->Font->Size  = 12;
  scope_display.pBitmap->Canvas->Font->Color = clGray;
  th = scope_display.pBitmap->Canvas->TextHeight( "Q" ); // text height in pixels
  tx = ty = 16;
  if( pScope->cfg.iVisibleChannels == 0 )   // oops... all scope channels turned off !
   { scope_display.pBitmap->Canvas->TextOut( tx, ty, "All scope channels are turned off - " );
     ty += th;
     scope_display.pBitmap->Canvas->TextOut( tx, ty, " Use the RIGHT MOUSE BUTTON" );
     ty += th;
     scope_display.pBitmap->Canvas->TextOut( tx, ty, " to open the scope's context menu." );
     ty += th;
   }

  // Prepare the CHANNEL LAYOUT :
  // [in]  pScope->cfg.iVisibleChannels : bitwise combination (bit 0 for the first channel, etc)
  //       pScope->cfg.iChannelVerticalPos_pcnt[]
  // [out] pScope->sChannelInfo[].iVerticalPos_pixels
  //       pScope->sChannelInfo[].iCurveHeight_pixels
  for( iChannel=0; iChannel<(TIMING_SCOPE_N_DIGITAL_CHANNELS+TIMING_SCOPE_N_ANALOG_CHANNELS); ++iChannel )
   {
     yc = (iCurveAreaHeight * pScope->cfg.iChannelVerticalPos_pcnt[iChannel]) / 100;
     pScope->sChannelInfo[iChannel].iVerticalPos_pixels = yc;
     if( iChannel > 0 )  // not the FIRST channel : Height = difference between the baseline of THIS and the previous channel 
      { h2 = yc - pScope->sChannelInfo[iChannel-1].iVerticalPos_pixels - TSCOPE_CHANNEL_SEPARATOR_HEIGHT;
      }
     else
      { h2 = yc - TSCOPE_CHANNEL_SEPARATOR_HEIGHT/2;
      }
     if( pScope->cfg.iVisibleChannels & (1<<iChannel) ) // if the channel isn't hidden ..
      { pScope->sChannelInfo[iChannel].iCurveHeight_pixels = h2;
      }
     else // channel is HIDDEN :
      { pScope->sChannelInfo[iChannel].iCurveHeight_pixels = 0;
      }
   }

#if( TIMING_SCOPE_N_ANALOG_CHANNELS > 0 )
  // Draw the ANALOG channels (if any, to have them in the BACKGROUND) :
  for( iChannel=0; iChannel<TIMING_SCOPE_N_ANALOG_CHANNELS; ++iChannel )
   { // (draw channel by channel, to avoid having to swap colours too often)
     scope_display.dwPenColour.dw = pScope->cfg.dwChannelColours[iChannel+TIMING_SCOPE_FIRST_ANALOG_CHANNEL].dw;
     if( pScope->cfg.iVisibleChannels & ( 1<<(iChannel+TIMING_SCOPE_FIRST_ANALOG_CHANNEL) )  )
      { scope_display.pBitmap->Canvas->Pen->Color = (TColor)scope_display.dwPenColour.dw;
        // ex: yc = iCurveAreaHeight/2 - 1 + 2*iChannel;
        curve_height = pScope->sChannelInfo[iChannel].iCurveHeight_pixels;
        yc           = pScope->sChannelInfo[iChannel].iVerticalPos_pixels;
        UTL_LimitInteger( &yc, 10/*pixels*/, iCurveAreaHeight-10 );
        y_max = yc - curve_height / 2;
        y_min = yc + curve_height / 2;
        // Because the VERTICAL POSITIONS are user-adjustable,
        // we need a kind of 'handle' where the channel can be grabbed with the mouse,
        // and moved up / down by the user. Those 'handles' (inspired by a Rigol MSO5104)
        // are drawn LATER, *AFTER* the curves, to appear in the front.
        scope_display.nPoints = 0;
        for( iSample=0; iSample<TIMING_SCOPE_NUM_SAMPLE_POINTS; ++iSample)
         { g_iDebugDummyIndex = iSample;
           x = (iTotalWidth * iSample) / TIMING_SCOPE_NUM_SAMPLE_POINTS;
           switch( iChannel )  // is there a SPECIAL SOURCE for this "analog channel" ?
            {
              case (TIMING_SCOPE_CHANNEL_AUDIO_CW_DEC-TIMING_SCOPE_FIRST_ANALOG_CHANNEL)/*2*/:
                 flt = pScope->sSamplePoints[iSample].sDspPlotterSample.fltCwDecoderKeyingSignal;
                 break;

              default: // no 'SPECIAL SOURCE' for this analog channel ->
                 flt = pScope->sSamplePoints[iSample].fltAnalog[iChannel];
                 break;
            } // end switch( iChannel )
           if( flt > 1.0f ) // clipping (analog samples range from -1 to +1)
            {  flt = 1.0f;
            }
           if( flt < -1.0f )
            {  flt = -1.0f;
            }
           flt = (flt+1.0) * (float)(iCurveAreaHeight/2);
           y = yc - (int)flt;
           if(iSample==0)
            { ScopeDisplay_AppendPoint( &scope_display, x,y); // add the first point to the polyline
            }
           else
            { if( (x != prev_x) || (y != prev_y) ) // avoid unnecessary GDI calls to save CPU time
               { ScopeDisplay_AppendPoint( &scope_display, x,y); // append another point to the polyline
               } // end if < different x or y >
            }
           prev_x = x;
           prev_y = y;
         } // end for < all samples of a channel >
        if( scope_display.nPoints >= 2 )
         { scope_display.pBitmap->Canvas->Polyline( scope_display.pPoints, scope_display.nPoints-1/*!*/ );
           // ,------------------------------------------------------------|_____________________|
           // '--> Not the NUMBER OF POINTS but the NUMBER OF POINTS MINUS ONE !
         }
      }     // end if < ANALOG channel visible > ?
   }       // end for < all ANALOG channels >
#endif    // TIMING_SCOPE_N_ANALOG_CHANNELS > 0 ?

  if( 1 && (iSpectrumHeight > 0) )  // also draw the AUDIO SPECTRUM (by-product from the audio CW decoder) ?
   { // Due to the buffering of audio samples, FFT windowing,
     // the latency of the AUDIO SPECTRUM and the variables of the
     // audio CW decoder is much larger than the latency of e.g.
     // the digital inputs from the Morse key (TIMING_SCOPE_CHANNEL_DASH_INPUT, etc).
#   if(1) // (1) = use the DSP samples (including spectra) in pScope->sSamplePoints[]
     for( iSample=0; iSample<TIMING_SCOPE_NUM_SAMPLE_POINTS; ++iSample)
      { int x1,x2;
        g_iDebugDummyIndex = iSample;
        x1 = (iTotalWidth * iSample) / TIMING_SCOPE_NUM_SAMPLE_POINTS;
        x2 = (iTotalWidth * (iSample+1)) / TIMING_SCOPE_NUM_SAMPLE_POINTS;
        if( x2 > x1 )
         { --x2;  // no overlap with the NEXT plotted DSP sample
         }
        ScopeDisplay_DrawAudioSpectrumAndDecoderInfo( &scope_display,
           &pScope->sSamplePoints[iSample].sDspPlotterSample,
           x1, x2, iCurveAreaHeight+1/*y1*/, iCurveAreaHeight+iSpectrumHeight/*y2*/ );
      }
#   else  // older code, plotted the audio spectrum directly from the audio DSP :
     //  HERE, for the AUDIO SPECTROGRAM / Audio CW Decoder 'keying signal',
     //  use a loop that runs along all horizontal PIXELS, calculate the
     //  timestamp(!) for that pixel position, and ask module CwDSP.c
     //  for an entry in its thread-safe 'plotter FIFO' :
     // > Note: One horizontal pixel is NOT ONE SAMPLE of the timing scope !
     // >      (it would be, if iTotalWidth == TIMING_SCOPE_NUM_SAMPLE_POINTS)
     dblSecondsPerStep = (double)pScope->cfg.iMillisecondsPerSample * 1e-3
                       * (double)TIMING_SCOPE_NUM_SAMPLE_POINTS / (double)iTotalWidth;
     // Example: 2 ms / SAMPLE, iTotalWidth = 1432, TIMING_SCOPE_NUM_SAMPLE_POINTS = 1024
     //          -> dblSecondsPerStep = 0.002 [seconds/sample] * 1024 [samples] / 1432 [pixels]
     //                               = 1.43017e-3 seconds per pixel (horizontally).
     // ex: dblEndTime_s = DSW_ReadHighResTimestamp_s();  // <- timestamp related to "x_wiper" (close to the NEWEST entry)
     // Avoid jitter when retrieving channel data via TIMESTAMP by sampling the timestamp only once per sweep:
     dblEndTime_s = pScope->dblTimestampAtSampleIndexZero  + (double)x_wiper * dblSecondsPerStep;
     dblTimestamp_s = dblEndTime_s - dblSecondsPerStep * iTotalWidth;
     x = x_wiper+1;
     while( dblTimestamp_s < dblEndTime_s )
      { pAudioCwDecoderSample = CwDSP_ReadFromPlotterFifo( pCwDSP, dblTimestamp_s );
        if( pAudioCwDecoderSample != NULL )
         {
           diff = pAudioCwDecoderSample->dblTimestamp_s - dblTimestamp_s;
           // Typical values seen here for the first sample processed in the loop:
           // dblEndTime_s   = 374.1379 ("current time" in seconds since start)
           // dblTimestamp_s = 371.2739 ("looking back" for one horizontal sweep; here with iTotalWidth=1432 pixel * 2 ms per step)
           // pAudioCwDecoderSample->dblTimestamp_s = 371.2768 (hmm.. that's more than 16 milliseconds off.. "timestamp jitter" ! )
           (void)diff;
         }
        if( x>=iTotalWidth )
         {  x=0;
         }
        ScopeDisplay_DrawAudioSpectrumAndDecoderInfo( &scope_display,
           pAudioCwDecoderSample, x/*x1*/, x/*x2*/,
           iCurveAreaHeight+1/*y1*/, iCurveAreaHeight+iSpectrumHeight/*y2*/ );
        dblTimestamp_s += dblSecondsPerStep;
        ++x;
      } // end for( x.. ) (loop to run along all horizontal pixels for the CW decoder's AUDIO SPECTROGRAM)
#   endif // < new / old method to retrieve the samples for the AUDIO SPECTROGRAM in the timing scope >
   } // end if( iSpectrumHeight > 0 )


  // Draw the DIGITAL channels (in the foreground, considered 'more important' than the audio waveforms).
  for( iChannel=0; iChannel<TIMING_SCOPE_N_DIGITAL_CHANNELS; ++iChannel )
   { // (also here, draw channel by channel, to avoid having to swap colours too often)
     scope_display.dwPenColour.dw = pScope->cfg.dwChannelColours[iChannel].dw;
     if( pScope->cfg.iVisibleChannels & ( 1<<(iChannel+TIMING_SCOPE_FIRST_DIGITAL_CHANNEL)) ) // only draw the curve if the channel isn't hidden ..
      { scope_display.pBitmap->Canvas->Pen->Color = (TColor)scope_display.dwPenColour.dw;
        y_max = pScope->sChannelInfo[iChannel].iVerticalPos_pixels;
        curve_height = pScope->sChannelInfo[iChannel].iCurveHeight_pixels;
        y_min = y_max - curve_height;
        scope_display.nPoints = 0;
        for( iSample=0; iSample<TIMING_SCOPE_NUM_SAMPLE_POINTS; ++iSample)
         { x = (iTotalWidth * iSample) / TIMING_SCOPE_NUM_SAMPLE_POINTS;
           val = pScope->sSamplePoints[iSample].iChannel[iChannel];
           y = y_max - (val * curve_height) / 100/*percent*/;
           if(iSample==0)
            { ScopeDisplay_AppendPoint( &scope_display, x, y); // add the first point to the polyline
            }
           else
            { if( (x != prev_x) || (y != prev_y) ) // avoid unnecessary GDI calls to save CPU time
               { ScopeDisplay_AppendPoint( &scope_display, x, y); // append another point to the polyline
               } // end if < different x or y >
            }
           prev_x = x;
           prev_y = y;
         } // end for < all sample of a channel >
         if( scope_display.nPoints >= 2 )
          { scope_display.pBitmap->Canvas->Polyline( scope_display.pPoints, scope_display.nPoints-1 );
            // ,------------------------------------------------------------|_____________________|
            // '--> funny but true: Not the NUMBER OF POINTS,
            //                      but the NUMBER OF POINTS MINUS ONE !
          }
      } // end if < DIGITAL channel visible > ?
   }   // end for < all DIGITAL channels >

  // Draw the CHANNEL MARKERS (inspired by Rigol MSO5104) :
        //    ______                                                      .
        //   |      \                                                     .
        //   |  /|   \ _________  y (begin of the "curve" with value=0)   .
        //   |   |   /                                                    .
        //   |______/                                                     .
        //                                                                .
  scope_display.pBitmap->Canvas->Font->Name  = "Arial";
  scope_display.pBitmap->Canvas->Font->Size  = 6;
  scope_display.pBitmap->Canvas->Font->Color = clBlack;
  SetBkMode( scope_display.pBitmap->Canvas->Handle, OPAQUE ); // <- old Windows 'GDI' function ("OPAQUE" is what the VCL expects when it calls the GDI internally)
  for( iChannel=0; iChannel<(TIMING_SCOPE_N_DIGITAL_CHANNELS+TIMING_SCOPE_N_ANALOG_CHANNELS); ++iChannel )
   { if( pScope->cfg.iVisibleChannels & ( 1<<iChannel)  ) // only draw the MARKER if the channel isn't hidden ..
      { yc = (iCurveAreaHeight * pScope->cfg.iChannelVerticalPos_pcnt[iChannel] ) / 100;
        ScopeDisplay_DrawChannelMarker( &scope_display, 0/*x*/, yc,
           pScope->cfg.dwChannelColours[iChannel], // [in] colour as a T_RGBColor (24-bit RGB mix in a 32-bit 'DWORD')
           (char)('1' + iChannel) ); // [in] single-character "caption"
      }
   }

  // Draw "decoded text" into the Timing Scope, too :
  scope_display.pBitmap->Canvas->Font->Name  = "Arial";
  scope_display.pBitmap->Canvas->Font->Size  = 16;
  scope_display.pBitmap->Canvas->Font->Color = clWhite;
  scope_display.pBitmap->Canvas->Pen->Color  = clWhite;
  th = scope_display.pBitmap->Canvas->TextHeight( "Q" );
  for( iSample=iSampleAtBeginOfChar=0; iSample<TIMING_SCOPE_NUM_SAMPLE_POINTS; ++iSample)
   {
     if( (wCwChar = pScope->sSamplePoints[iSample].wCwChar) != 0)
      { pszDecoded = Elbug_MorseCodePatternToASCII( wCwChar );
        if( wCwChar & ELBUG_RESULT_BEGIN_NEW_CHAR ) // nothing "decoded" at this sample index,
         { // but a marker for the BEGIN of a character (decoded and printed LATER):
           iSampleAtBeginOfChar = iSample;
         }
        else
         { // It's a decoded character, so print it, horizontally centered...
           x = (iTotalWidth * iSample) / TIMING_SCOPE_NUM_SAMPLE_POINTS;
           y = (TIMING_SCOPE_N_DIGITAL_CHANNELS-1) * h2 + th/4; // near the "CW output" channel
           prev_x = (iTotalWidth * iSampleAtBeginOfChar) / TIMING_SCOPE_NUM_SAMPLE_POINTS;
           // Horizontally center the text under the "CW Signal" that belongs to it:
           //      ___   _   ___     ___   _
           //  ___|   |_| |_|   |___|   |_| |___ <- "CW Signal" (keyer output)
           //
           //     '-------K -------''--- N ---'  <- CW decoder output
           //     |       |        |
           // prev_x      tx       x
           //  ,----------------|__|
           //  '--> This TWO-DOT-GAP belongs to the character,
           //       because without it, the decoder cannot decode it.
           //       If this 'Timing Scope' convinces anyone to stop "smearing
           //       characters together", this little gadget has already paid off :o)
           //
           tw = scope_display.pBitmap->Canvas->TextWidth( AnsiString(pszDecoded) );
           tx = prev_x + (x-prev_x-tw) / 2;
           SetBkMode( scope_display.pBitmap->Canvas->Handle, TRANSPARENT ); // <- old Windows 'GDI' function
           scope_display.pBitmap->Canvas->TextOut( tx, y, AnsiString(pszDecoded) );
           if( x != prev_x )
            { scope_display.pBitmap->Canvas->MoveTo( prev_x,  y      );
              scope_display.pBitmap->Canvas->LineTo( prev_x,  y+th/2 );
              scope_display.pBitmap->Canvas->LineTo( tx,      y+th/2 );
              scope_display.pBitmap->Canvas->MoveTo( tx+tw,   y+th/2 );
              scope_display.pBitmap->Canvas->LineTo( x,       y+th/2 );
              scope_display.pBitmap->Canvas->LineTo( x,       y      );
            }
         }
      } // end if < "decoded character" at index iSample > ?
   } // end for < all sample-indices, looking for DECODED CHARACTERS to print >

  // Draw a primitive, relative timescale; stepwidth = ONE DOT INTERVAL..
  scope_display.pBitmap->Canvas->Pen->Color = clWhite;
  iSample=0;
  while( iSample<TIMING_SCOPE_NUM_SAMPLE_POINTS )
   { x = (iTotalWidth * iSample) / TIMING_SCOPE_NUM_SAMPLE_POINTS;
     scope_display.pBitmap->Canvas->MoveTo( x,  0   );
     scope_display.pBitmap->Canvas->LineTo( x,  3   );
     scope_display.pBitmap->Canvas->MoveTo( x,  iTotalHeight-4 );
     scope_display.pBitmap->Canvas->LineTo( x,  iTotalHeight-1 );

     // Assume the CW-keyer's worker thread really samples the inputs
     //        at <pScope->cfg.iMillisecondsPerSample> ...
     if( CwKeyer_Elbug.cfg.iDotTime_ms >= pScope->cfg.iMillisecondsPerSample )
      { iSample += (CwKeyer_Elbug.cfg.iDotTime_ms / pScope->cfg.iMillisecondsPerSample);
      }
     else // something wrong with the Elbug or Timing Scope parameters..
      { // draw a tick every TEN samples (this may be something like 20 ms per tick)
        iSample += 10;
      }
   }

  // Draw the overlays for 'mouse cursor readout' (especially time differences)
  for( int i=0; i<KEYER_GUI_N_SCOPE_OVERLAYS; ++i )
   { T_TimingScopeOverlay *pOverlay = &TimingScopeOverlay[i];
     if( pOverlay->visible )
      { scope_display.pBitmap->Canvas->Pen->Color = clWhite;
        scope_display.pBitmap->Canvas->Font->Size  = 10;
        scope_display.pBitmap->Canvas->MoveTo( pOverlay->x1, pOverlay->y1 );
        scope_display.pBitmap->Canvas->LineTo( pOverlay->x2, pOverlay->y2 );
        SetBkMode( scope_display.pBitmap->Canvas->Handle, OPAQUE ); // <- old Windows 'GDI' function
        s = "dt="+IntToStr( pOverlay->t2_ms - pOverlay->t1_ms )+" ms";
        th = scope_display.pBitmap->Canvas->TextHeight( s );
        tw = scope_display.pBitmap->Canvas->TextWidth( s );
        tx = (pOverlay->x1 + pOverlay->x2) / 2 - tw/2;
        ty = (pOverlay->y1 + pOverlay->y2) / 2 + th/4;
        scope_display.pBitmap->Canvas->TextOut( tx, ty, s );
        if( CwKeyer_Elbug.cfg.iDotTime_ms >= 1 )
         { s = "="+FormatFloat( "#0.0",
                     (double)(pOverlay->t2_ms - pOverlay->t1_ms)
                   / (double)CwKeyer_Elbug.cfg.iDotTime_ms) + " dots";
           scope_display.pBitmap->Canvas->TextOut( tx, ty+th, s );
         }
      }
   }    // end for( int i=0; i<KEYER_GUI_N_SCOPE_OVERLAYS; ++i )


  // Draw the "windscreen wiper" to show the current position on the
  // non-scrolling, Radar-like sweeping display:
  if( x_wiper>=0 && x_wiper<iTotalWidth )
   { scope_display.pBitmap->Canvas->Pen->Color = clWhite;
     scope_display.pBitmap->Canvas->MoveTo( x_wiper, TSCOPE_TOP_TIME_SCALE_HEIGHT );
     scope_display.pBitmap->Canvas->LineTo( x_wiper, iTotalHeight-TSCOPE_BOTTOM_TIME_SCALE_HEIGHT );
   }

  // Note that up to this point, all the 'drawing' used the off-screen bitmap in scope_display !
  pDestBitmap->Canvas->Draw( 0,0, scope_display.pBitmap );

  HERE_I_AM__GUI();

  ScopeDisplay_Free( &scope_display ); // free whatever ScopeDisplay_Init() has allocated or even 'constructed' (the off-screen bitmap)

  HERE_I_AM__GUI();

  (void)y_min; // ... assigned a value that is never used ..


} // end KeyerGUI_UpdateTimingScope()

//---------------------------------------------------------------------------
BOOL KeyerGUI_HandleMouseEventInTimingScope( Graphics::TBitmap *pBitmap,
           int iEvent,    // [in] EVT_MOUSE_DOWN / EVT_MOUSE_MOVE / UP
           int x, int y )
      // [out]  TimingScopeOverlay[] : mouse-controlled 'overlay' to measure times,
      //                               displayed in KeyerGUI_UpdateTimingScope()
      // [return] TRUE if the mouse event was handled HERE,
      //          FALSE otherwise (the caller may decide what to do then,
      //                           for example open the CONTEXT MENU
      //                           when clicking into the o'scope's with the LEFT button)
{
  int iTotalWidth  = pBitmap->Width;  // width in pixels
  int iTotalHeight = pBitmap->Height; // height in pixels
  int iSample, t_ms;
  T_TimingScopeOverlay *pOverlay;
  BOOL fHandledEvent = TRUE;

  pOverlay = &TimingScopeOverlay[ TimingScope_iCurrentOverlay ];

  // Convert 'x' into a sample index, inverse to the following :
  //   x = (iTotalWidth * iSample) / TIMING_SCOPE_NUM_SAMPLE_POINTS;
  if( (iTotalWidth > 0 ) && (CwKeyer_TimingScope.cfg.iMillisecondsPerSample > 0 ) )
   { iSample = (x * TIMING_SCOPE_NUM_SAMPLE_POINTS) / iTotalWidth;
     // Convert the (relative) SAMPLE INDEX into an offset in milliseconds .
     // Note: One horizontal pixel is NOT ONE SAMPLE of the timing scope
     //       (it would be, if iTotalWidth == TIMING_SCOPE_NUM_SAMPLE_POINTS) !
     t_ms = iSample * CwKeyer_TimingScope.cfg.iMillisecondsPerSample;
     if( iEvent == EVT_MOUSE_DOWN )
      { pOverlay->x1 = x;
        pOverlay->y1 = y;
        pOverlay->t1_ms = t_ms;
      }
     else //  EVT_MOUSE_MOVE or EVT_MOUSE_UP ->
      { pOverlay->x2 = x;
        pOverlay->y2 = y;
        pOverlay->t2_ms = t_ms;
        pOverlay->visible = TRUE;
      }
     TimingScope_iUpdateCountOnTestTab = -1; // kludge to redraw the timing scope..
     TimingScope_iUpdateCountOnKeyerTab= -1; // ... even when 'paused' .
   }

  if( iEvent==EVT_MOUSE_UP ) // switch to the next overlay ? (only if there was a movement between EVT_MOUSE_DOWN and EVT_MOUSE_UP)
   {
     if( (pOverlay->x1 != pOverlay->x2) || (pOverlay->y1 != pOverlay->y2) )
      { TimingScope_iCurrentOverlay = (TimingScope_iCurrentOverlay+1) % KEYER_GUI_N_SCOPE_OVERLAYS;
      }
     else
      { pOverlay->visible = FALSE;
        fHandledEvent = FALSE;
      }
   }

  (void)iTotalHeight;
  return fHandledEvent;
} // end KeyerGUI_HandleMouseEventInTimingScope()

//---------------------------------------------------------------------------
void KeyerGUI_ClearScopeOverlays(void)
{
  for(int i=0; i<KEYER_GUI_N_SCOPE_OVERLAYS; ++i)
   { TimingScopeOverlay[i].visible = FALSE;
   }
  TimingScope_iUpdateCountOnTestTab = -1; // redraw a.s.a.p. ...
  TimingScope_iUpdateCountOnKeyerTab= -1; // ... even when 'paused' .
} // end KeyerGUI_ClearScopeOverlays()

//-----------------------------------------------------------------------------
void KeyerGUI_UpdateVFODisplay( TEdit *pED, double dblVfoFrequency_Hz )
  // Called from UpdateTrxDisplay() if the frequency was modified externally,
  //     or from Ed_VFOKeyDown() if the frequency was tuned LOCALLY (up/down editing).
{
  char sz80Temp[84];
  int iOldSelStart   = pED->SelStart;
  int iOldTextLength = pED->Text.Length();
  int iNewTextLength;

  g_SpecDispControl.dblDisplayedVfoFreq_Hz = dblVfoFrequency_Hz;
  // The same old question since the haydays of the "printf format string" :
  // Which format string to use ? Tested the following with DL4YHF's CalcEd.exe:
  // > @format("'%10.5lf' MHz")
  // > 12345.67891234 =: '12345.67891' MHz (oh well; too wide by ONE character..)
  // > 1234.567891234 =: '1234.56789' MHz
  // > 123.4567891234 =: ' 123.45679' MHz  (... but ok from VLF to UHF ...)
  // > 12.34567891234 =: '  12.34568' MHz
  // > 1.234567891234 =: '   1.23457' MHz
  // > 0.123456789123 =: '   0.12346' MHz
  // > 0.001234567891 =: '   0.00123' MHz
  // > 0.000012345678 =: '   0.00001' MHz  (display resolution = 10 Hz; ok for CW)
  // > 0.000001234567 =: '   0.00000' MHz
  sprintf( sz80Temp, "%10.5lf MHz", (double)g_SpecDispControl.dblDisplayedVfoFreq_Hz * 1e-6 );
  iNewTextLength = strlen(sz80Temp);
  ++KeyerGUI_iUpdating;
  pED->Text = sz80Temp;
  // Assigning a new string as the edit field's "Text" set the cursor (caret)
  // to the leftmost position (this is what Borland/Delphi/VCL called "SelStart").
  // Not really clever for up/down editing via cursor keys or mouse wheel, thus:
  pED->SelStart = iOldSelStart;
  if( iNewTextLength != iOldTextLength ) // trouble with the new decimal place ?
   {
   }

  --KeyerGUI_iUpdating;
} // end KeyerGUI_UpdateVFODisplay()


//---------------------------------------------------------------------------
BOOL KeyerGUI_HandleMouseEventInVfoFreqEditor( TEdit *pED,
           int iEvent,    // [in] EVT_MOUSE_DOWN / EVT_MOUSE_MOVE / UP
           int x, int y ) // [in] pixel position in CLIENT coordinates (for pED)
  // Inspired by WFView, and possibly more user friendly for touchscreen users:
  // a mouse click (or tap) into the UPPER HALF of a digit INCREMENTS it,
  // a mouse click (or tap) into the LOWER HALF of a digit DECREMENTS it.
  // But, since *our* VFO control is an 'Edit Control' (albeit "single-line"),
  //      this is where it gets really ugly, because we must not interfere
  //      with the default mouse handling (e.g. click into the editor to simply
  //      set the 'text caret' alias 'cursor', until someone decided that the
  //      'cursor' is now a funny name for the MOUSE POINTER..)
{
  int iTextLine, iTextColumn, iRelativeVertPos;
  double dblOldVfoFreq, dblFreqIncrement;
  TPoint tpScreen = pED->ClientToScreen( TPoint(x,y) );
  int iHeight = pED->Height;

  if( (iHeight<8)  )
   { // Something's wrong -> bail out
     return FALSE;
   }

  iRelativeVertPos = (y * 100) / iHeight; // relative vertical position in PERCENT of the VFO field's height
  // (a click into the upper third increments, a click into the lower third decrements,
  //  and a click into the vertical center only switches the text cursor without incrementing/decrementing)

  if( PoorEdit_ScreenCoordToTextLineAndColumn( pED->Handle/*hwndRichEdit*/,
        tpScreen.x,  tpScreen.y,   // [in] absolute pixel coordinate ("screen", not "client")
        &iTextLine, &iTextColumn)) // [out] text line and -column
   { // Ok; got the CHARACTER INDEX and the TEXT COLUMN (the latter should be zero).
     // If the COLUMN equals the poor edit control's "SelStart" start,
     // incrementing/decrementing the frequency here will not interfere
     // with a click into the edit field to SWITCH TO DIFFERENT DIGIT.
     if( iTextColumn == pED->SelStart )
      { // Next: From the CHARACTER INDEX, find the
        //       This is SIMILAR as in TKeyerMainForm::FormMouseWheel(),
        //       but here we cannot simply send a VK_UP or VK_DOWN (virtual key)
        //       to Mr. TRichEdit.. but a part of that could be recycled here:
        dblOldVfoFreq = KeyerGUI_GetFrequencyFromEditField( pED,
                       iTextColumn,         // [in] cursor position within the string; 0=first char; ex: pEd->SelStart+1
                       &dblFreqIncrement ); // [out] 10^N, f(cursor_position)
        if( iRelativeVertPos < 33 )       // click into the upper third ?
         { // increment : just leave dblFreqIncrement as-is (POSITIVE)
         }
        else if( iRelativeVertPos > 60 )  // click into the lower third ?
         { dblFreqIncrement = -dblFreqIncrement; // decrement (further below)
         }
        else                              // click into the middle third ?
         { return FALSE;                  // do nothing
         }
        switch( iEvent ) // EVT_MOUSE_DOWN / EVT_MOUSE_MOVE / UP ?
         { case EVT_MOUSE_DOWN :
              RigCtrl_SetVFOFrequency( &MyCwNet.RigControl, dblOldVfoFreq + dblFreqIncrement );
              // Even if the new frequency hasn't arrived at the remote rig yet,
              // already update the LOCAL frequency display, to have an immediate feedback.
              // On this occasion: Show the value POSSIBLY LIMITED by RigControl.c :
              KeyerGUI_UpdateVFODisplay( pED, MyCwNet.RigControl.dblVfoFrequency );
              pED->SelLength = 0; // without this, the VCL sometimes MARKED the text.
              // Default behaviour to "mark a word" on double click ? Someone wrote:
              // > A double click will select the word underneath the mouse.
              // > This is all standard behaviour implemented by the underlying
              // > Windows control.
              // So how to convince the "underlying Windows control" from STOPPING THIS ?
              //  * Setting 'AutoSelect' to FALSE in the 'Object Inspector' didn't help,
              //  * what helped is implement an on "OnDoubleClick"-handler for the TEdit,
              //    which "undoes" the WORD-SELECTION as quickly as possible. CRAP !
              // 
              g_SpecDispControl.iVFOEditTimer_ms = 300; // let the entire field flash up,
              // same effect as when changing the frequency by other means.
              // This also suppresses assigning new text to Ed_VFO "while clicking up/down" !
              break;
           case EVT_MOUSE_MOVE :
              break;
           case EVT_MOUSE_UP   :
              break;
           default:
              break;
         }
        return TRUE;
      } // end if( iTextColum == Ed_VFO->SelStart )
   }   // end if < successfully converted a PIXEL- into a TEXT coordinate >
  return FALSE;
} // end KeyerGUI_HandleMouseEventInVfoFreqEditor()


//---------------------------------------------------------------------------
void KeyerGUI_SetColourScheme( int iColourScheme )
{
#if( ! defined __clang__ ) // guess it's "an older VCL" NOT supporting 'Styles' ..
  int iComp, iForm, nForms = Screen->FormCount;  // how many FORMS do we have ? [at least one]
  TForm      *pForm;
  TComponent *pComp;
#endif // old VCL without support for 'Styles' ?
  T_SpecDispControl *pDispCtrl = &g_SpecDispControl;
  DWORD dwErrorCode;
  char  sz255[256], *pszMsg;

#ifdef __BORLANDC__  // "... is assigned a value that is never used".. shut up ..
  (void)nForms;
#endif

  HERE_I_AM__GUI();
  pDispCtrl->clButtonFace = (DWORD)clBtnFace;    // form's BACKGROUND COLOUR - don't ask why they chose this stupid name for a COLOUR
  pDispCtrl->clWindowText = (DWORD)clWindowText;
  pDispCtrl->clWindowBackground=(DWORD)clWindow; // per default, WHITE background in e.g. combos and edit fields (but not for buttons, panels, tabsheets, and the like)
  pDispCtrl->clActiveCaption  = (DWORD)clActiveCaption;   // > "color of the active window's title bar"
  pDispCtrl->clInactiveCaption= (DWORD)clInactiveCaption; // > "color of inactive windows' title bar"
  pDispCtrl->clButtonFont = pDispCtrl->clWindowText;

  switch( iColourScheme )
   { case COLOUR_SCHEME_DEFAULT : // back to e.g. "clBtnFace", etc etc etc etc
     default:
        // (most already set in the variable initialisers)
        g_clMenuBackground = TColor( 0xF0F0F0 ); // light-gray background
        g_clMenuSelBackgnd = TColor( 0xC08080 ); // light grayish-blue when selected
        g_clMenuForeground = TColor( 0x000000 ); // black text (regardless of selection)
        g_clDecoderOutputForeground = TColor( 0x000000 ); // black text
        g_clDecoderOutputBackground = TColor( 0xFFC0C0 ); // light-blue background
        break;
     case COLOUR_SCHEME_DARK :
        pDispCtrl->clButtonFace = TColor( 0x202020 ); // dark gray
        pDispCtrl->clWindowText = clWhite;
        pDispCtrl->clWindowBackground= clBlack;
        pDispCtrl->clActiveCaption   = TColor( 0xC04040 ); // dark grayish-blue
        pDispCtrl->clInactiveCaption = TColor( 0x404040 ); // even darker, almost black
        // As long as we cannot change a button's BACKGROUND COLOUR (a light gray)
        // it's a bad idea to set the button's FONT COLOR to ALMOST WHITE. Thus:
        pDispCtrl->clButtonFont = TColor( 0x804040 ); // dark grayish-blue
        g_clMenuBackground = TColor( 0x404040 );   // dark background
        g_clMenuSelBackgnd = TColor( 0xC04040 );   // dark grayish-blue when selected
        g_clMenuForeground = TColor( 0xE0E0E0 );   // white text
        g_clDecoderOutputForeground = TColor( 0xE0E0E0 ); // white text
        g_clDecoderOutputBackground = TColor( 0x800000 ); // dark blue background

        break;
   } // end switch( iColourScheme )

#if( SWI_USE_VCL_STYLES ) // guess it's "a newer VCL" supporting 'Styles' ..
  // The C++Builder 12  "RAD Studio Topics", chapter "Application Appearance"
  // mentions a "TStyleManager.TrySetStyle("name"), where "name" seems to be
  // what can be selected in the 'Project' options under "Application"..
  // "Appearance".."Custom Styles", e.g.
  //  * "Charcoal Dark Slate" (resembles iColourScheme=COLOUR_SCHEME_DARK)
  //  * "Light"               (resembles iColourScheme=COLOUR_SCHEME_DEFAULT)
  // Surprisingly (and a bit scary), a VCL form's current "Style" is not
  // a property of the form itself, but set through the following :
  bool ok;
  switch( iColourScheme )
   { case COLOUR_SCHEME_DARK:
        ok = Vcl::Themes::TStyleManager::TrySetStyle("Charcoal Dark Slate", false/*ShowErrorDialog?NO!*/);
        // ,------------------------|_________|
        // '-> Sets the style specified by name as the active style,
        //   > without raising an exception.
        //   > TrySetStyle returns True if the style was successfully set
        //   > as the active style, and False otherwise.
        // When first tried, ok was TRUE, but the entire form still used the
        // default colours as shown in the FORM DESIGNER . Why ?
        // Possibly because of this (found in an outdated online resource):
        // > You Must Define Control Ownership
        // > In styled VCL applications, it is specially important that all
        // > controls are created with a valid owner, such as the form
        // > where they belong, so that the VCL can identify them as part
        // > of the styled application and apply the configured style
        // > to them accordingly.
        //
        break;
     default:
        ok = Vcl::Themes::TStyleManager::TrySetStyle("Light", false/*ShowErrorDialog?NO!*/);
        break;
   }
#else // ! SWI_USE_VCL_STYLES -> guess it's "the old VCL" without 'Styles'

  // The venerable old VCL (from Borland C++ Builder V6) did not support
  // "Schemes", "Themes", "Styles", "Skins" or whatever they call it now.
  // But for standard controls like menus, tabsheets, edit fields,
  // combo boxes and the like, Borland's form designer / "object inspector"
  // often uses colour names like e.g.
  //    "clBtnFace"    (funny name, often used as the entire form's
  //                    BACKGROUND COLOUR, for example in 'KeyerMainForm'),
  //    "clWindowText" (typically used for a TTabsheet's "Font.Color" property),
  //
  //    "clWindow"     (funny name for the WHITE background in combo boxes,
  //                    edit fields, and a couple of others).
  // The VCL documentation lists the above as ..
  //  > colors that are defined in the Windows Control panel.
  // Ok, but we don't want to fool around with the 'Windows Control panel'
  // (whatever / wherever that may be these days) - instead, THIS APPLICATION
  // shall of course only modify ITS OWN COLOURS .
  // There is no 'TStyleManager'. Didn't want to hook into Win32 "GetSysColor()".
  //   Maybe the Win32 "SetSysColors()" function will do the trick ?
  //   No, forget it, SetSysColors() affects the colours for ALL windows.

  // Thus: Iterate through all controls on all forms, find their TYPES,
  //       and depending on that, replace some of their colours:
  nForms = Screen->FormCount;  // how many FORMS do we have ? [at least one]

  for(iForm=0; iForm<nForms; ++iForm)
   { // 2024-11-30: Controlling BACKGROUND COLOURS this way was ok when compiled with BCB V6,
     //             but failed when compiled with Embarcadero C++Builder V12 "Athens".
     // For the "old VCL" (from C++Builder V6), TTabSheets like e.g. "TS_IO"
     //     were coloured
     pForm = Screen->Forms[iForm];  // glorious VCL allows accessing them all 'like an array'
     pForm->Color = (TColor)pDispCtrl->clButtonFace;
     // |--> With the 'old VCL' (from BCB V6), setting this TForm.Color also affected
     // |    the colour on all TABSHEETS of the form; regardless of nesting. Fine.
     // '--> With the 'new VCL' (in C++Builder V12) ...
     //         * all those TTabSheets did NOT change their background colour,
     //         * TLabels on those TTabSheets did NOT change their background colour,
     //           but their FOREGROUND colour,
     //         *
     //
     pForm->Font->Color = (TColor)pDispCtrl->clWindowText;
     for( iComp=0; (iComp<pForm->ComponentCount); iComp++ )
      { HERE_I_AM__GUI();
        pComp = pForm->Components[iComp];
        if (pComp->ClassNameIs("TMainMenu") )
         { // ex: TMainMenu *menu = (TMainMenu*)pComp;
           // Adapting the stupid main menu's back- and foreground colours
           // via e.g. "SetMenuInfo()" is absolutely hopeless. The crappy API
           // can modifiy the background of the DROPPED-DOWN items but not the
           // text colour, and it cannot modify the colours in the menu title.
         }
        else if (pComp->ClassNameIs("TMenuItem") )
         { // Unfortunately, the VCL designers at Borland forgot to add a
           // 'Color'/'Colour' property for this beast as well,
           // so without digging in AWFULLY DEEP into Win32 programming,
           // we cannot adapt the background of the title bar and main menu
           // to our 'Dark' colour scheme.
         }
        else if (pComp->ClassNameIs("TPopupMenu") )
         {
         }
        else if (pComp->ClassNameIs("TLabel") )
         { TLabel *label = (TLabel*)pComp;
           // Replace the TLabel's back- and foreground colour:
           label->Color = (TColor)pDispCtrl->clButtonFace;
           if( label->OnClick != NULL )
            { // "TLabel" acting like a 'link' (clickable) : BLUE
              if( iColourScheme == COLOUR_SCHEME_DARK )
               { label->Font->Color = clSkyBlue; // there's no clLightBlue :o(
               }
              else // white background -> dark blue text
               { label->Font->Color = clBlue;
               }
            }
           else  // not clickable -> no "link" but normal static text
            { label->Font->Color = (TColor)pDispCtrl->clWindowText;
            }
         } // end if <component is a TLabel>
        else if (pComp->ClassNameIs("TButton") )
         { TButton *btn = (TButton *)pComp;
           btn->Font->Color = (TColor)pDispCtrl->clWindowText;
         } // end if <component is a TButton>
        else if (pComp->ClassNameIs("TBitBtn") )
         { TBitBtn *btn = (TBitBtn *)pComp;
           // ex: btn->Font->Color = pDispCtrl->clWindowText;
           btn->Font->Color = (TColor)pDispCtrl->clButtonFont;
         } // end if <component is a TBitBtn>
        else if (pComp->ClassNameIs("TRadioButton") )
         { TRadioButton *btn = (TRadioButton *)pComp;
           btn->Font->Color = (TColor)pDispCtrl->clWindowText;
         } // end if <component is a TRadioButton>
        else if (pComp->ClassNameIs("TSpeedButton") )
         { // ex: TSpeedButton *btn = (TSpeedButton *)pComp;
           // A "speed button" neither has a foreground nor a background colour
           // that we can modify here. Modifying the FONT colour is nonsense
           // because a "speed button" is usually a square thing without any text,
           // but just a "glyph" (microscopic bitmap) that cannot be re-coloured.
         } // end if <component is a TSpeedButton>
        else if (pComp->ClassNameIs("TPanel") )
         { TPanel *pnl = (TPanel *)pComp;
           pnl->Color  = (TColor)pDispCtrl->clButtonFace; // ex: clWindowBackground
           pnl->Font->Color = (TColor)pDispCtrl->clWindowText;
         }
        else if (pComp->ClassNameIs("TTabSheet") )
         { TTabSheet *ts = (TTabSheet *)pComp;
           ts->Font->Color = (TColor)pDispCtrl->clWindowText;
           // For obscure reasons, even the form's "Refresh()" didn't
           // cause the VCL TTabSheet to redraw its BACKGROUND . Nnngrrr.
           // ts->Refresh();  // <- tried this.. no effect
         } // end if <component is a TTabSheet>
        else if (pComp->ClassNameIs("TCheckBox") )
         { TCheckBox *chk = (TCheckBox *)pComp;
           chk->Color = (TColor)pDispCtrl->clButtonFace;  // !
           chk->Font->Color = (TColor)pDispCtrl->clWindowText;
         } // end if <component is a TCheckBox>
        else if (pComp->ClassNameIs("TGroupBox") )
         { TGroupBox *gb = (TGroupBox *)pComp;
           gb->Color       = (TColor)pDispCtrl->clWindowBackground;  // !
           gb->Font->Color = (TColor)pDispCtrl->clWindowText;
         }
        else if (pComp->ClassNameIs("TListBox") )
         { TListBox *lb = (TListBox *)pComp;
           lb->Color       = (TColor)pDispCtrl->clWindowBackground;
           lb->Font->Color = (TColor)pDispCtrl->clWindowText;
         } // end if <component is a TListBox>
        else if (pComp->ClassNameIs("TComboBox") )
         { TComboBox *cb = (TComboBox *)pComp;
           cb->Color       = (TColor)pDispCtrl->clWindowBackground;
           cb->Font->Color = (TColor)pDispCtrl->clWindowText;
         } // end if <component is a TComboBox>
        else if (pComp->ClassNameIs("TRadioGroup") )
         { TRadioGroup *rg = (TRadioGroup *)pComp;
           rg->Color       = (TColor)pDispCtrl->clWindowBackground;
           rg->Font->Color = (TColor)pDispCtrl->clWindowText;
         } // end if <component is a TRadioGroup>
        else if (pComp->ClassNameIs("TCheckListBox") )
         { TCheckListBox *cl = (TCheckListBox *)pComp;
           cl->Color       = (TColor)pDispCtrl->clWindowBackground;
           cl->Font->Color = (TColor)pDispCtrl->clWindowText;
         } // end if <component is a TCheckListBox>
        else if (pComp->ClassNameIs("TStringGrid") )
         { TStringGrid *sgr = (TStringGrid *)pComp;
           (void)sgr;
         } // end if <component is a TStringGrid>
        else if (pComp->ClassNameIs("TEdit") )
         { TEdit *ed = (TEdit *)pComp;
           ed->Color       = (TColor)pDispCtrl->clWindowBackground;
           ed->Font->Color = (TColor)pDispCtrl->clWindowText;
         } // end if <component is a TEdit>
        else if (pComp->ClassNameIs("TRichEdit") )
         { TRichEdit *ed = (TRichEdit *)pComp;
           ed->Color       = (TColor)pDispCtrl->clWindowBackground;
           ed->Font->Color = (TColor)pDispCtrl->clWindowText;
           ed->Clear();
         } // end if <component is a TRichEdit>
        else if (pComp->ClassNameIs("TScrollBar") )
         { TScrollBar *sb = (TScrollBar *)pComp;
           // Like many others, this poor VCL fellow also doen't have
           // an accessable 'Color' property...
           (void)sb;
         }
        else if (pComp->ClassNameIs("TTrackBar") )
         { TTrackBar *tb = (TTrackBar *)pComp;
           // This poor fellow doesn't even have an accessable 'Color' property !
           // C compiler said "Access to TControl::Color is impossible". Nnngrrr.
           // tb->Color = (TColor)pDispCtrl->clWindowBackground;
           (void)tb;
         } // end if <component is a TTrackBar>
        else if (pComp->ClassNameIs("TPageControl") )
         { // no attempt made to 'colourize' this yet ..
         }
        else if (pComp->ClassNameIs("TScrollBox") )
         { // no attempt made to 'colourize' this yet ..
         }
        else if (pComp->ClassNameIs("TImage") )
         { // no attempt made to 'colourize' this yet ..
         }
        else if (pComp->ClassNameIs("TBevel") )
         { // no attempt made to 'colourize' this yet ..
         }
        else if (pComp->ClassNameIs("TStaticText") )
         { // no attempt made to 'colourize' this yet ..
         }
        else if (pComp->ClassNameIs("TTimer") )
         { // no need to ever 'colourize' this, because a "TTimer" is never visible (except in the form designer)
         }
        else if (pComp->ClassNameIs("TOpenDialog") )
         { // cannot 'colourize' this, it's just a VCL wrapper for a Windows thing
         }
        else if (pComp->ClassNameIs("TSaveDialog") )
         { // cannot 'colourize' this, it's just a VCL wrapper for a Windows thing
         }
        else // kludge for development .. what's missing in the above list ?
         { AnsiString sClassName = pComp->ClassName();
           ShowError( ERROR_CLASS_ERROR, "SetColourScheme: Unknown class name %s",
                                         sClassName.c_str() );
         }
        HERE_I_AM__GUI();
      } // end for( iComp=0; (iComp<pForm->ComponentCount); iComp++ )
   } // end for<all forms of the application>
  HERE_I_AM__GUI();
  pDispCtrl->fUpdateSpectrum = pDispCtrl->fUpdateFreqScale = TRUE;
#endif // ! SWI_USE_VCL_STYLES ?
} // KeyerGUI_SetColourScheme()

//---------------------------------------------------------------------------
void KeyerGUI_UpdateStatusIndicatorText(void) // -> KeyerGUI_sz80StatusIndicatorText, KeyerGUI_iStatusIndicatorUsage(!) .
  // Periodically called from the GUI thread (e.g. from  "timer").
  // This function decides "what to show" in the main status indicator,
  //      visible in the upper left corner of the main window's client area.
  //      (.. which kind of resembles Icom's red "TX" indicator, but
  //          OUR status indicator only turns read when.. guess what..)
  // [in]   Various thread-, "Rig Control"-, and network statuses
  // [out]  KeyerGUI_iStatusIndicatorUsage : STATUS_INDICATOR_IDLE, ..ON_AIR, etc.
  //        Possibly used in the VCL-specific part to give the indicator
  //        a certain colour. Those colours are applied in ___() .
  // [out]  KeyerGUI_sz80StatusIndicatorText (will be displayed on an
  //        on an approximately 90 pixel wide 'TPanel', text automatically centered,
  //        so the result looks best with 5 to 10 chars in the string.
{

  if( CwKeyer_iThreadStatus != KEYER_THREAD_STATUS_RUNNING )
   { KeyerGUI_iStatusIndicatorUsage = STATUS_INDICATOR_OFF;
   }
  else // the keyer thread is running, but what is it doing at the moment ?
   {   // See KeyerThread.c : KeyerThread() ...
     if( CwKeyer_Elbug.fPauseTransmitter )
      { KeyerGUI_iStatusIndicatorUsage = STATUS_INDICATOR_PAUSED;
      }
     else if( (CwKeyer_Gen.iState != CW_GEN_OFF) && (CwKeyer_Gen.iState != CW_GEN_FINISHED_SENDING) )
      { KeyerGUI_iStatusIndicatorUsage = STATUS_INDICATOR_TX_MEM;
      }
     else if( MyCwNet.RigControl.iTransmitReqst > 0 )
      { KeyerGUI_iStatusIndicatorUsage = STATUS_INDICATOR_ON_AIR;
      }
     else if( MyCwNet.RigControl.iTransmitting > 0 )
      { KeyerGUI_iStatusIndicatorUsage = STATUS_INDICATOR_SOMEONE_ELSE_ON_AIR;
      }
     else if( CwKeyer_Config.iManualPTTInput != KEYER_SIGNAL_INDEX_NONE ) // MANUAL PTT control, and  ... ?
      { if( CwKeyer_swMorseActivityTimer !=0 ) // Morse-Output WITHOUT active PTT ?
         { KeyerGUI_iStatusIndicatorUsage = STATUS_INDICATOR_CW_TEST;  // here: "something" is generating Morse-Output WITHOUT active PTT. Possible reasons:
           // (a) On the "Keyer" tab, "More"-button, in the menu with rarely used special functions,
           //     the checkmark [v] "Disable Transmission, 'SIM TX' (only CW sidetone output)" is set .
           // (b) A MANUAL PTT switchin INPUT(!) is selected on the 'I/O' tab,
           //     and CW output is started (e.g. via locally connected key on the SERVER,
           //     or sending a text from memory) WITHOUT activating the PTT,
           // (c) A user without the permission to TRANSMIT operates his locally connected key.
         }
        if( CwKeyer_Config.fDisableTx && CwKeyer_GetDigitalInput( CwKeyer_Config.iManualPTTInput ) )
         { // Note: Keyer_dwCurrentSignalStates.KEYER_SIGNAL_INDEX_PTT is NOT DRIVEN when CwKeyer_Config.fDisableTx is set,
           //       thus poll the MANUAL PTT INPUT as KeyerThread.c : KeyerThread() would do .
           KeyerGUI_iStatusIndicatorUsage = STATUS_INDICATOR_ON_AIR_SIM;
         }
        else // MANUAL PTT control, and NONE of the above current states ->
         { KeyerGUI_iStatusIndicatorUsage = STATUS_INDICATOR_IDLE;
         }
      }
     else // AUTOMATIC PTT control, but no reason to transmit at the moment ->
      { KeyerGUI_iStatusIndicatorUsage = STATUS_INDICATOR_IDLE;
      }
   } // end switch( CwKeyer_iThreadStatus )

  switch( KeyerGUI_iStatusIndicatorUsage )  // <- set in UpdateStatusIndicator() ...
   { case STATUS_INDICATOR_OFF :
           strcpy( KeyerGUI_sz80StatusIndicatorText, "off" );
           break;
     case STATUS_INDICATOR_IDLE:
           strcpy( KeyerGUI_sz80StatusIndicatorText, "idle" );
           break;
     case STATUS_INDICATOR_ON_AIR:
     case STATUS_INDICATOR_SOMEONE_ELSE_ON_AIR:
           if( CwKeyer_Config.fDisableTx )
            { strcpy( KeyerGUI_sz80StatusIndicatorText, "OnAir,SIM" );
            }
           else
            { strcpy( KeyerGUI_sz80StatusIndicatorText, "ON AIR" ); // .. usually because MyCwNet.RigControl.iTransmitReqst set, not necessarily with "something to send" yet
            }
           break;
     case STATUS_INDICATOR_ON_AIR_SIM:
           strcpy( KeyerGUI_sz80StatusIndicatorText, "OnAir,SIM" );
           break;
     case STATUS_INDICATOR_TX_MEM:  // "transmitting from MEMORY ?"
           if( (KeyerGUI_iStatusIndicatorMemIdx >= KEYER_MEMORY_INDEX_1) // sending from keyer memory ?
            && (KeyerGUI_iStatusIndicatorMemIdx <= KEYER_MEMORY_INDEX_MAX) ) // (KeyerGUI_iStatusIndicatorMemIdx was set in TKeyerMainForm::StartPlaying() )
            { if( CwKeyer_Config.fDisableTx )
               { sprintf( KeyerGUI_sz80StatusIndicatorText, "SIM TX #%d",
                    (int)(KeyerGUI_iStatusIndicatorMemIdx+1-KEYER_MEMORY_INDEX_1) );
               }
              else
               { sprintf( KeyerGUI_sz80StatusIndicatorText, "TX MEM #%d",
                    (int)(KeyerGUI_iStatusIndicatorMemIdx+1-KEYER_MEMORY_INDEX_1) );
               }
            }
           else  // not transmitting from one of the six "memories", but from something special...
           switch(KeyerGUI_iStatusIndicatorMemIdx) //  KeyerGUI_iStatusIndicatorUsage==STATUS_INDICATOR_TX_MEM but
            { // "sending directly from the multi-purpose edit field",
              // or from the Winkeyer-EMULATOR (e.g. for a Contest Logger),
              // or from the Winkeyer-HOST (e.g. with a real Winkeyer connected):
              case KEYER_MEMORY_INDEX_WINKEYER_EMU: // "sending CW from the Winkeyer-EMULATOR" (possibly fooling an external application like N1MM Logger)
                 strcpy( KeyerGUI_sz80StatusIndicatorText, "TX: WK-EMU" );
                 break;
              case KEYER_MEMORY_INDEX_WINKEYER_HOST: // "sending CW from the Winkeyer-HOST" (and an EXTERNAL Winkeyer-chip on an 'Additional COM Port'
                 strcpy( KeyerGUI_sz80StatusIndicatorText, "TX: WinKyr" );
                 break;
              default:
                 if( CwKeyer_Config.fDisableTx )
                  { strcpy( KeyerGUI_sz80StatusIndicatorText, "SIM TX" );
                  }
                 else
                  { strcpy( KeyerGUI_sz80StatusIndicatorText, "TX text" );
                  }
                 break;
            }     // end switch(KeyerGUI_iStatusIndicatorMemIdx)
           break; // end case < KeyerGUI_iStatusIndicatorUsage == STATUS_INDICATOR_TX_MEM >
     case STATUS_INDICATOR_CW_TEST:
           strcpy( KeyerGUI_sz80StatusIndicatorText, "Off-air CW" ); // what's this ? :
           // TEST to 'generate CW', but only LOCALLY (sidetone without keying the TX).
           // Possible reasons for this further above in KeyerGUI_UpdateStatusIndicatorText().
           break;
     case STATUS_INDICATOR_PAUSED:
           strcpy( KeyerGUI_sz80StatusIndicatorText, "paused" );
           break;
     default:
           sprintf( KeyerGUI_sz80StatusIndicatorText, "? %d ?",
                    (int)KeyerGUI_iStatusIndicatorUsage );
           break;
   }
} // end KeyerGUI_UpdateStatusIndicatorText()


//---------------------------------------------------------------------------
int KeyerGUI_ExpandTextToSend(  // evaluates macros like <mycall> but not e.g. <s40> (*)
       const char **ppszOriginalText, int nCharsWanted,
       char *pszDest, const char *pszEndstop )
  // In many cases, simply copies the original text into the destination (string).
  // Expands certain 'macros' that CwGen_StartReplay() isn't aware of, e.g. "<mycall>" .
  // (*) Other macros, like "<sNN>" (NN=two-digit speed in WPM) and "<s>"
  //     must be passed on UNMODIFIED to the CW generator, because only the
  //     generator can interpret them at the right time - see CwGen.c :
  //   CwGen_PrepareSendingNextChar() -> Elbug_ParseMorseCodePatternFromASCII(); ...
  // [in] nCharsWanted : Special feature when sending text from the 'type-ahead buffer',
  //          To allow editing as many characters as possible, even when already
  //          transmitting, only a few characters (e.g. TWO) are handed over
  //          from the edit field to the CW generator in each call of
  //          KeyerGUI_UpdateColourOfSentCharsInRichEdit().
  //          This is only a "wish" though, because macros like <mycall>
  //          or prosigns like "^KA" are copied COMPLETELY.
  // [in,out] ppszOriginalText : Will be incremented by the number of characters
  //          parsed from the INPUT (e.g. the length of a parsed macro, prosign, etc).
  // [return] number of characters appended to pszDest.
  //          THIS MAY BE MORE than 'nCharsWanted', for reasons explained above.
{
  const char *pszSrc = *ppszOriginalText;
  char *pszDest2;
  int nCharsExpanded;
  if( pszDest < pszEndstop ) // make sure the result is always properly terminated,
   { *pszDest = '\0';        // even if there is no 'original text' to expand at all.
   }
# if(0) // this is the SIMPLE variant, without caring for 'nCharsWanted' :
  pszDest2 = pszDest;
  while( (*pszSrc != '\0') && (pszDest < pszEndstop) )
   { if( SL_SkipString_AnyCase( &pszSrc, "<mycall>" ) )
      {  SL_AppendString( &pszDest, pszEndstop, CwKeyer_Config.sz15MyCall );
      }
     else
      { SL_AppendChar( &pszDest, pszEndstop, *pszSrc++ );
      }
   }
  nCharsExpanded = pszDest - pszDest2;
# else  // advanced variant, aware of 'nCharsWanted', macros, and CW PROSIGNS:
  nCharsExpanded = 0;
  while( (*pszSrc != '\0') && (pszDest < pszEndstop) && (nCharsExpanded < nCharsWanted) )
   { pszDest2 = pszDest;
     if( SL_SkipString_AnyCase( &pszSrc, "<mycall>" ) )
      {  SL_AppendString( &pszDest, pszEndstop, CwKeyer_Config.sz15MyCall );
         // Like it or not, MACROS are either expanded completely
         //  or not expanded at all, so :
         nCharsExpanded += (pszDest-pszDest2); // <- may increment by strlen(callsign) :)
      }
     else if( pszSrc[0] == '^' ) // begin of a PROSIGN ? Move to destination only when COMPLETE !
      { if( (pszSrc[1] != '\0') && (pszSrc[2] != '\0') )
         { SL_AppendChar( &pszDest, pszEndstop, pszSrc[0] );
           SL_AppendChar( &pszDest, pszEndstop, pszSrc[1] );
           SL_AppendChar( &pszDest, pszEndstop, pszSrc[2] );
           // (the CW generator itself will parse this prosign, see Elbug_ParseMorseCodePatternFromASCII() )
           nCharsExpanded += 3;
           pszSrc += 3;
         }
        else // not enough characters after the '^', so try again later (with more text)
         { break;
         }
      }
     else // neither a macro nor a prosign, so copy and count a single character:
      { SL_AppendChar( &pszDest, pszEndstop, *pszSrc++ );
        ++nCharsExpanded;
      }
     if( pszDest2 == pszDest ) // nothing processed in this loop ?
      { break; // terminate loop, for example because a macro in the type-ahead buffer isn't complete yet
      }
   }
# endif // SIMPLE or adcanced variant, with or without caring for 'nCharsWanted' ?

  *ppszOriginalText = pszSrc; // pass back the INCREMENTED(!) source pointer

  return nCharsExpanded;

} // end KeyerGUI_ExpandTextToSend()

//---------------------------------------------------------------------------
int KeyerGUI_GetCharWidthInPixelsForRichEditControl( TRichEdit *pRichEdit )
{ // Fasten seat belts for another joyful ride into of Win32 / GDI programming...
  HDC  hdc;          // Windows GDI rules..  some things are impossible with Borland's VCL !
  HGDIOBJ hOldFont;
  SIZE tsize;  // TEXTMETRIC tm;
  int  iWidthInPixel;

  // Width of "Courier New, Size 10" = HOW MANY PIXELS ? ? ?
  hdc = GetDC( pRichEdit->Handle );  // This stupid function doesn't do what you EXPECT from it !
      // It doesn't retrieve the "current" DC selected into the TRichEdit-thingy;
      // instead it -kind of- creates(?!) a new DC with some stupid defaults
      // which don't have anything to do with the TRichEdit's currently used font !
      // All this goddamned Win32-low-level-stuff is only necessary because
      // in Borland C++Builder, a TRichEdit doesn't expose its TCanvas (which
      // would be nice to 'measure' the width+height of a string in PIXELS).
  hOldFont = SelectObject( hdc, pRichEdit->Font->Handle );
  SetMapMode( hdc, MM_TEXT );
     // Only support MM_TEXT here, and forget about the dirty dozen others:
     // > Each logical unit is mapped to one device pixel.
     // > Positive x is to the right; positive y is down.
     // Despite that, GetTextExtentPoint32 returned complete garbage.
     // The garbage was fixed by the "SelectObject"-stuff further above.
  GetTextExtentPoint32( hdc, "ABC4567890", 10/*nStringLength*/, &tsize );
     // > The GetTextExtentPoint32 function computes the width and height
     // >     of the specified string of text.
     // Seen at this point: tsize.cx=100,  tsize.cy=20 .
  iWidthInPixel = tsize.cx;
  SelectObject( hdc, hOldFont );
  ReleaseDC( pRichEdit->Handle, hdc);
  // Don't assume this obfuscated sequence of Windows GDI calls never fails !
  // Only use the "measured width for ten characters in pixels" when plausible:
  if( iWidthInPixel >= 60 )
   { return  ( 5 + iWidthInPixel ) / 10/*chars in the 'test string'*/;
   }
  else
   { return 0;  // something went wrong; let the caller try something else !
   }
} // end KeyerGUI_GetCharWidthInPixelsForRichEditControl()

//---------------------------------------------------------------------------
void KeyerGUI_GetColoursForRichEditWithRxTxInfo(
        int iRxTxInfoUsage,    // [in] KEYER_GUI_RXTXINFO_OFF, KEYER_GUI_RXTXINFO_TYPING_..,
                               // [in] g_SpecDispControl.iColourScheme : COLOR_SCHEME_DEFAULT (light), COLOR_SCHEME_DARK .
        DWORD *pdwTextColour,  // [out] foreground- aka "font" colour for the text, 0x00BBGGRR (*)
        DWORD *pdwBkgndColour) // [out] background colour suitable for a crazy TRichEdit control (**)
  // Returns the BACKGROUND colour for the Rich Edit control in the status line,
  // depending on what is currently being displayed or typed in that field.
  // (*) Borland's VCL uses yet another superfluent type for these, "TColor",
  //     which is just an RGB mixture as a 32-bit unsigned, in hex: 0x00BBGGRR .
  // (**) In contrast to the FOREGROUND colour, a RichEdit control's choice of
  //      BACKGROUND COLOUR may be very limited .. at least it was.
  //      Details in Utilities1.c : RichEdit_SetSelColors() !
  //
  // The colours shall be similar to those used on the 'Debug' tab
  // for RECEIVED and TRANSMITTED text :
  //    * blue-ish(*) for text from the CW KEYER's
  //      own decoder (which means text that is about to be sent
  //      from the operator's own key or keyboard),
  //    * green-ish(*) for text decoded from a RECEIVED SIGNAL,
  //    * yellow-ish(*) for text that is obviously a COMMAND (e.g. after '#').
  //  (*) "colour-ish" because fore- and background colour shall be aware of
  //      the currently selected colour scheme, g_SpecDispControl.iColourScheme .
{
  BOOL fDark = (g_SpecDispControl.iColourScheme == COLOUR_SCHEME_DARK);

  switch( iRxTxInfoUsage )
   {
     case KEYER_GUI_RXTXINFO_OFF: // the edit field on status panel shows nothing yet,
     default:                     // and the user hasn't typed into it yet
        *pdwTextColour  = (DWORD)g_clDecoderOutputForeground;
        *pdwBkgndColour = (DWORD)g_clDecoderOutputBackground;
        break;
     case KEYER_GUI_RXTXINFO_PROG_INFO :
        if( fDark )
         { *pdwTextColour  = 0xE0E0E0; // white text
           *pdwBkgndColour = 0x804040; // dark blue-ish background
         }
        else
         { *pdwTextColour  = 0x000000; // black text
           *pdwBkgndColour = 0xFFC0C0; // light blue-ish background
         }
        break;
     case KEYER_GUI_RXTXINFO_TYPING_TXD: // the operator is typing 'transmittable text' into the edit field on status panel
        if( fDark )
         { *pdwTextColour  = 0xE0E0E0; // light gray text
           *pdwBkgndColour = 0x800000; // dark blue background (24-bit colour values in Windows: 0x00BBGGRR)
         }
        else
         { *pdwTextColour  = 0x000000; // black text
           *pdwBkgndColour = 0xFFC0C0; // light blue-ish background
         }
        break;
     case KEYER_GUI_RXTXINFO_TYPING_CMD: // the operator is typing a COMMAND into the edit field on status panel
        if( fDark )
         { *pdwTextColour  = 0xE0E0E0; // light gray text
           *pdwBkgndColour = 0x0080C0; // dark brown (not "dark yellow") background
         }
        else
         { *pdwTextColour  = 0x000000; // black text
           *pdwBkgndColour = 0x00FFFF; // yellow background
         }
        break;
     case KEYER_GUI_RXTXINFO_RXDATA: // the edit field on status panel shows RECEIVED TEXT
        if( fDark )
         { *pdwTextColour  = 0xE0E0E0; // white text
           *pdwBkgndColour = 0x008000; // dark green background (received/decoded text)
         }
        else
         { *pdwTextColour  = 0x000000; // black text
           *pdwBkgndColour = 0x80FF80; // light green background (received/decoded text)
         }
        break;
     case KEYER_GUI_RXTXINFO_SENT_DATA: // the edit field on status panel shows ALREADY TRANSMITTED TEXT
        if( fDark )
         { *pdwTextColour  = 0xE0E0E0; // white text
           *pdwBkgndColour = 0x808000; // stronger blue background (stronger than characters TYPED but not SENT yet)
           // Note: Due to the crazy way Microsoft implemented their RichText control,
           //       BACKGROUND COLOURS for character cells are very limited.
           //       When the background colour of 'sent text' was 0xFF0000,
           //        and the background colour of 'typed-but-not-sent' text was 0x800000,
           //        there was no visible difference between the background colours !
           //        Thus tried "dark cyan" (0x808000 as a 0xBBGGRR mix) instead.
         }
        else
         { *pdwTextColour  = 0x000000; // black text
           *pdwBkgndColour = 0xFF8080; // stronger blue-ish background (stronger than characters TYPED but not SENT yet)
         }
        break;
     case KEYER_GUI_RXTXINFO_ERROR_MSG:
        if( fDark )
         { *pdwTextColour  = 0xE0E0E0; // white text
           *pdwBkgndColour = 0x0000C0; // red background (as in the highlighted combo boxes on the 'I/O Config' tab)
         }
        else
         { *pdwTextColour  = 0x000000; // black text
           *pdwBkgndColour = 0x8080FF; // light red background
         }
        break; // end case KEYER_GUI_RXTXINFO_ERROR_MSG
   }

} // end KeyerGUI_GetColoursForRichEditWithRxTxInfo()

//---------------------------------------------------------------------------
void KeyerGUI_ShowInfoInStatusLine( TRichEdit *pRichEdit, int iRxTxInfoUsage, const char *pszInfo )
{ DWORD dwTextColor, dwBkgndColor;
  KeyerGUI_iRxTxInfoUsage = iRxTxInfoUsage; // e.g. KEYER_GUI_RXTXINFO_ERROR_MSG
  pRichEdit->Text = pszInfo;
  KeyerGUI_GetColoursForRichEditWithRxTxInfo( KeyerGUI_iRxTxInfoUsage, &dwTextColor, &dwBkgndColor );
  KeyerGUI_ModifyColoursInRichEdit( pRichEdit,
             0/*iFromCharIndex*/, strlen(pszInfo)/*nCharsToModify*/,
             dwTextColor, dwBkgndColor );
} // end KeyerGUI_ShowInfoInStatusLine()

//---------------------------------------------------------------------------
void KeyerGUI_AppendDataToRichEditWithRxTxInfo( TRichEdit *pRichEdit,
        const char *pszText, // [in] e.g. text BEING SENT, drained from CwKeyer_DecoderFifo, or a command / response
        int iRxTxInfoUsage)  // [in] KEYER_GUI_RXTXINFO_RXDATA, KEYER_GUI_RXTXINFO_SENT_DATA, etc (?)
   // If the text in the 'Rx-Text' / 'Tx-Text' / 'Scrolling Info' text editor
   // gets TOO LONG (depending on the window width), this function may not only
   //          APPEND new text (e.g. DECODED characters) on the right,
   // but also REMOVE old text on the left !
   //     This eliminates the use of a vertical scrollbar by keeping
   //     the number of characters in the edit field low, so "all is visible".
   // Thus, a few GLOBAL VARIABLES may also be involved here:
   //  * KeyerGUI_iTxEditorCharIndex_CwGenerator may be decremented
   //    by the number of characters REMOVED on left side ("oldest characters"),
   //  * ?
{
  char szCleanedUpText[256];
  const char *cpSrc;
  char c, cPrev;
  int n, iNewTextLength = 0;
  int iOldTextLength = pRichEdit->Text.Length();
  int iCharWidthInPixel = KeyerGUI_GetCharWidthInPixelsForRichEditControl( pRichEdit );
      // '--> got e.g. 10 pixels for a RichEdit control with "Courier New, BOLD, size=10"
  int iMaxCharsPerLine = 32;
  DWORD dwTextColor, dwBkgndColor;
  if( iCharWidthInPixel >= 6 )
   {  iMaxCharsPerLine = pRichEdit->Width / iCharWidthInPixel;
      // '--> with a 333-pixel wide edit field, expect something around 32 characters per line.
      //      That's not sufficient for a typical 'over' in CW, but enough
      //      to read what's going on for the sysop, or CW newbie.
   }

  cpSrc = pszText;
  cPrev = '\0';
  while( ((c=*cpSrc++)!=0x00) && ((iNewTextLength+1)<sizeof(szCleanedUpText)) )
   { // Replace ASCII control characters that would 'spoil' the single-line text editor:
     if( (unsigned char)c < 0x20 ) // exclude "ASCII control characters" like '\n', '\r', '\t', etc:
      { if( cPrev != ' ' )
         { szCleanedUpText[iNewTextLength++] = ' ';
           cPrev = ' ';
         }
        else // previous emitted character was a SPACE: don't append a 'replacement' for e.g. '\n'
         {
         }
      }
     else // 'c' seem to be "harmless" for a single-line text editor, so emit it:
      { szCleanedUpText[iNewTextLength++] = c;
        cPrev = c;
      }
   }
  szCleanedUpText[iNewTextLength] = '\0'; // always provide a trailing zero for C strings

  n = iOldTextLength + iNewTextLength + 2/*"reserve"*/- iMaxCharsPerLine;
  if( n > 0 ) // the resulting text would exceed <iMaxCharsPerLine>,
   { // so delete the "oldest" characters at the begin of pRichEdit->Text :
     pRichEdit->SelStart = 0;
     pRichEdit->SelLength= n;
     pRichEdit->SelText = "";  // replace those 'n' excessive chars with an empty string ("AnsiString" a la Borland VCL)
     iOldTextLength = pRichEdit->Text.Length(); // revise the 'old text length'
          // because we need it to APPEND TEXT as a "selection" further below .
     // Because <n> characters have been deleted from the begin of the string,
     // KeyerGUI_iTxEditorCharIndex_CwGenerator may have to be adjusted
     // so pRichEdit->Text.c_str[KeyerGUI_iTxEditorCharIndex_CwGenerator]
     // remains the same as BEFORE the call of KeyerGUI_AppendDataToRichEditWithRxTxInfo():
     KeyerGUI_iTxEditorCharIndex_CwGenerator  -= n;
     UTL_LimitInteger( &KeyerGUI_iTxEditorCharIndex_CwGenerator, 0, iOldTextLength );
     KeyerGUI_iTxEditorCharIndex_StartOfMsg -= n;
     UTL_LimitInteger( &KeyerGUI_iTxEditorCharIndex_StartOfMsg, 0, iOldTextLength );
   } // end if < delete <n> characters at the begin of the edit field >

  pRichEdit->SelStart = iOldTextLength;
  pRichEdit->SelLength= 1;
  KeyerGUI_GetColoursForRichEditWithRxTxInfo( iRxTxInfoUsage, &dwTextColor, &dwBkgndColor );
  RichEdit_SetSelColors( pRichEdit->Handle, dwTextColor, dwBkgndColor ); // <- Win32, not Borland VCL !
  // ex: pRichEdit->SelText = AnsiString( pszText );
  // the above played havoc with out SINGLE-LINE edit control,
  // when the original pszText contained e.g. "\n" from a Hamlib command/response.
  // Thus: Converted pszText into  szCleanedUpText  further above, and add THIS to the text in the edit field:
  pRichEdit->SelText = AnsiString( szCleanedUpText );
  pRichEdit->SelLength = 0;
  // ex: pRichEdit->SelStart = pRichEdit->Text.Length();

} // end KeyerGUI_AppendDataToRichEditWithRxTxInfo()


//---------------------------------------------------------------------------
void KeyerGUI_ModifyColoursInRichEdit( TRichEdit *pRichEdit,
             int iFromCharIndex, int nCharsToModify,
             DWORD dwTextColor,  DWORD dwBkgndColor )
{
  int iOldSelStart  = pRichEdit->SelStart;
  int iOldSelLength = pRichEdit->SelLength;
  pRichEdit->SelStart  = iFromCharIndex;
  pRichEdit->SelLength = nCharsToModify;
  RichEdit_SetSelColors( pRichEdit->Handle, dwTextColor, dwBkgndColor ); // <- Win32, not Borland VCL !
  pRichEdit->SelLength = 0; // don't select anything when modifying "SelStart" ..
  pRichEdit->SelStart  = iOldSelStart; // this "SelStart"-thingy is the same as the EDIT CURSOR POSITION, so restore it !
  pRichEdit->SelLength = iOldSelLength;
} // end KeyerGUI_ModifyColoursInRichEdit()

//---------------------------------------------------------------------------
int KeyerGUI_TransferCharsFromEditFieldToCwGenerator(
        TRichEdit *pRichEdit, // [in] TRichEdit used as "type-ahead buffer"
        T_CwGen *pGenerator ) // [out] CW generator with its own, thread-safe TX buffer
  // [in,out] KeyerGUI_iTxEditorCharIndex_CwGenerator : set to the index of
  //                   the first character "to send" in KeyerGUI_OnKeyInTransmitTextEditor().
  //          Incremented with each character handed over from pRichEdit
  //          to the CW generator (single global instance CwKeyer_Gen) .
  // [in] KeyerGUI_iTxEditorCharIndex_StartOfMsg : index of the first character
  //          in pRichEdit of the next message; "operator started typing HERE".
  //          Used for changing the colour of characters ALREADY SENT
  //          (from KeyerGUI_iTxEditorCharIndex_StartOfMsg to .._CwGenerator) .
  // [return] number of characters transferred from pRichEdit,
  //          beginning at KeyerGUI_iTxEditorCharIndex_CwGenerator(++) .
  //
{
  int i,n = 0;
  int nCharsWanted = 2; // .. unless the CW GENERATOR is in state CW_GEN_WAITING_FOR_MACRO ..
  char szOriginalText[KEYER_MAX_CHARS_PER_MEMORY+4], szExpandedText[KEYER_MAX_CHARS_PER_MEMORY+4];
  const char *pszSrc, *pszSrc2;
  char  *pszDest = szExpandedText, *pszEndstop = szExpandedText + KEYER_MAX_CHARS_PER_MEMORY;
  DWORD dwTextColor, dwBkgndColor;

#ifdef __BORLANDC__  // "... is assigned a value that is never used".. shut up ..
  (void)pszDest;
#endif

  if( pGenerator->iState==CW_GEN_WAITING_FOR_MACRO ) // waiting for more text before a MACRO like "<sWPM>" can be parsed ?
   { nCharsWanted = 8; // typical macros like "<s40>" are shorter than this .. anyway.
     // But beware of the CW generator's own (thread-safe) transmit buffer !
   }
  i = CwGen_GetFreeSpaceInReplayBuffer( pGenerator );
  if( nCharsWanted > i )
   {  nCharsWanted = i;
   }

  if( pGenerator->iState != CW_GEN_OFF )
   { n = pGenerator->nCharsRemainingToPlay;
     if( n <= nCharsWanted ) // less characters available to the generator than it currently "wants" ?
      { // Time to pass a few more characters from pRichEdit to CwKeyer_Gen :
        // [in] KeyerGUI_iTxEditorCharIndex_CwGenerator : index of the NEXT character to send
        SL_strncpy( szOriginalText, AnsiString(pRichEdit->Text).c_str(), KEYER_MAX_CHARS_PER_MEMORY );
        // Avoid cutting macros and CW prosigns into pieces:
        i = strlen( szOriginalText );
        if( i > KeyerGUI_iTxEditorCharIndex_CwGenerator )  // ANYTHING left to send (even a single character) ?
         { pszSrc  = pszSrc2 = szOriginalText + KeyerGUI_iTxEditorCharIndex_CwGenerator; // skip characters ALREADY SENT !
           pszDest = szExpandedText;
           n = KeyerGUI_ExpandTextToSend( &pszSrc, nCharsWanted, pszDest, pszEndstop );
           if( n > 0 )
            { CwGen_AppendForReplay( pGenerator, szExpandedText );
              // '--> the string in pGenerator->szTxMemory should have grown now,
              //      but if the buffered length reaches CW_GEN_TX_MEMORY_SIZE,
              //      the oldest ("already sent" characters are scrolled out.
              KeyerGUI_iTxEditorCharIndex_CwGenerator += ( pszSrc - pszSrc2 );
              // |  ,------------------------------------|__________________|
              // |  '--> number of characters 'consumed' from the edit field
              // Index of the next character to move from the type-ahead buffer into the CW Generator,
              //       in a future call of KeyerGUI_UpdateColourOfSentCharsInRichEdit() .
              // Example: szOriginalText = "abcdefghijklmnopqrstuvwxyz"
              //  pGenerator->stTxMemory = "abcd" (after CwGet_AppendForReplay() appended "cd" to the already present "ab")
              //  -> KeyerGUI_iTxEditorCharIndex_CwGenerator = 4 (zero-based source index for the next call, but also for colouring below)
            }
         }  // end if < Length of the text in the editor exceeds KeyerGUI_iTxEditorCharIndex_CwGenerator ? >
      }    // end if < time to move more characters from the input field to the CW generator ? >
   }      // if( pGenerator->iState != CW_GEN_OFF )
  return n;
}       // end KeyerGUI_TransferCharsFromEditFieldToCwGenerator()


//---------------------------------------------------------------------------
void KeyerGUI_UpdateColourOfSentCharsInRichEdit( TRichEdit *pRichEdit ) // periodically called from the GUI thread
  // [in] KeyerGUI_iRxTxInfoUsage : KEYER_GUI_RXTXINFO_OFF, KEYER_GUI_RXTXINFO_TYPING_..,
  //                  KEYER_GUI_RXTXINFO_RXDATA, KEYER_GUI_RXTXINFO_SENT_DATA .
  // [in] KeyerGUI_sz80TextForRxTxInfo[] : characters to displayed in the field,
  //                  unless ..
  //        * the operator is currently TYPING text-to-send (KEYER_GUI_RXTXINFO_TYPING_TXD),
  //          or a COMMAND into the Rich Edit control (KEYER_GUI_RXTXINFO_TYPING_CMD)
  //        * the edit field shows RECEIVED characters (KEYER_GUI_RXTXINFO_RXDATA)
  //        * the edit field shows TRANSMITTED characters (KEYER_GUI_RXTXINFO_SENT_DATA)
  // [in,out] pRichEdit: the single-line edit field on status panel shows TRANSMITTED TEXT
  //                  (if, one fine day, the GUI isn't based on the VCL anymore,
  //                   the type will not be TRichEdit anymore but something similar,
  //                   a TEXT EDIT / DISPLAY FILED with individual back- and
  //                   foreground colour control for each character) .
  // [in]  Colours for characters provided by KeyerGUI_GetColoursForRichEditWithRxTxInfo() .
  // [in,out] KeyerGUI_iTxEditorCharIndex_CwGenerator : set to the index of
  //                   the first character "to send" in KeyerGUI_OnKeyInTransmitTextEditor().
  //          Incremented with each character handed over from pRichEdit
  //          to the CW generator (single global instance CwKeyer_Gen) .
  // [in] KeyerGUI_iTxEditorCharIndex_StartOfMsg : index of the first character
  //          in pRichEdit of the next message; "operator started typing HERE".
  //          Used for changing the colour of characters ALREADY SENT
  //          (from KeyerGUI_iTxEditorCharIndex_StartOfMsg to .._CwGenerator) .
  //
  //   Depending on KeyerGUI_iRxTxInfoUsage, KeyerGUI_UpdateColourOfSentCharsInRichEdit()
  //   may "feed" the CW-generator with characters from RichEdit_RxTxInfo,
  //   as a kind of 'type-ahead buffer'. This allows EDITING the typed text
  //   as long as possible, shortly before the generator runs out of gas,
  //   and the next few characters are transferred from the editor into the
  //   CW-generator's internal transmit buffer.
  //
{
  int   nCharsToModify;
  DWORD dwTextColor, dwBkgndColor;


  nCharsToModify = KeyerGUI_iTxEditorCharIndex_CwGenerator
    - KeyerGUI_iTxEditorCharIndex_StartOfMsg
    - CwKeyer_Gen.nCharsRemainingToPlay; // <- number of characters waiting to be played in CwKeyer_Gen.szTxMemory[]
    // ex: - CwKeyer_Gen.iNumCharsPending; // <- number of characters (usually 0 or 1) still pending somewhere else
  // Most simple example: Only ONE character typed into pRichEdit, previously empty:
  //    KeyerGUI_iTxEditorCharIndex_CwGenerator will stop at ONE (like the text cursor, at char index ONE)
  //    KeyerGUI_iTxEditorCharIndex_StartOfMsg = 0  (because in most cases, the editor is cleared when typing new TX-text)
  //    CwKeyer_Gen.nCharsRemainingToPlay = 0, CwKeyer_Gen.iNumCharsPending = 0 (text buffer and 'shift register' are empty)
  // -> nCharsToModify = 1  (number of characters to switch to the colour for SENT DATA)
  if( nCharsToModify>0 )
   { // Applies to characters between KeyerGUI_iTxEditorCharIndex_StartOfMsg and KeyerGUI_iTxEditorCharIndex_CwGenerator..
     KeyerGUI_GetColoursForRichEditWithRxTxInfo( KEYER_GUI_RXTXINFO_SENT_DATA, &dwTextColor, &dwBkgndColor );
     KeyerGUI_ModifyColoursInRichEdit( pRichEdit, KeyerGUI_iTxEditorCharIndex_StartOfMsg, nCharsToModify, dwTextColor, dwBkgndColor );
   }

} // end KeyerGUI_UpdateColourOfSentCharsInRichEdit()

//---------------------------------------------------------------------------
BOOL KeyerGUI_OnKeyInTransmitTextEditor(
        TRichEdit *pRichEdit,  // [in] Rich Edit control (here: in the flavour of Borland's VCL.
                               //      In Qt, an equivalent may be QTextEditor.)
        char cKey )            // [in] 8-bit ASCII, including control codes like '\r'
                               // [in,out] KeyerGUI_iRxTxInfoUsage
                               // [in,out] CwKeyer_Gen (may be started from here)
  // As the name shall imply, it's called from the KeyerGUI (e.g. Keyer_Main.cpp)
  // when the operator presses a key with an ASCII equivalent (including ENTER)
  // with the keyboard focus on the multi-purpose Rich Edit control,
  //          located in the always-visible status line above the tabbed pages.
  // Return value:  TRUE when it's "ok" to insert this character in the edit text,
  //                FALSE otherwise (e.g. because it's a character not supported
  //                                 in Morse code)
{
  BOOL fOkToInsert = (Elbug_SingleCharToMorseCodePattern(cKey) != 0)
                || (cKey==' ') || (cKey=='^')  // '^' begins a pro-sign
                || (cKey=='<') || (cKey=='>'); // begin or end of a MACRO (may also be typed into pRichEdit)
  char szOriginalText[KEYER_MAX_CHARS_PER_MEMORY+4], szExpandedText[KEYER_MAX_CHARS_PER_MEMORY+4];
  const char *pszSrc = szOriginalText;
  char  *pszDest    = szExpandedText;
  char  *pszEndstop = szExpandedText+KEYER_MAX_CHARS_PER_MEMORY;
  int   iTextLength  = pRichEdit->Text.Length();
  BOOL  fCursorAtEndOfText = pRichEdit->SelStart >= iTextLength;
  int   nCharsExpanded;
  DWORD dwTextColor, dwBkgndColor;
  int   iRigctrlErrorCode;

#ifdef __BORLANDC__  // "... is assigned a value that is never used".. shut up ..
  (void)pszDest;
#endif

  switch( cKey )
   { case '\r' :  // ENTER key (here: 'carriage return') may start transmitting,
        // or at least playing the "local" sidetone:
        switch( KeyerGUI_iRxTxInfoUsage ) // here: to decide what happens on pressing ENTER in the Info/Command/Rx/Tx-Text-Editor..
         { case KEYER_GUI_RXTXINFO_TYPING_TXD : // user typed TEXT TO TRANSMIT into the editor -> START SENDING !
              // [in] KeyerGUI_iTxEditorCharIndex_StartOfMsg :
              //      Index of the FIRST character to send from the type-ahead input field.
              //      Already set when switching from to KeyerGUI_iRxTxInfoUsage = KEYER_GUI_RXTXINFO_TYPING_TXD .
              // [out] KeyerGUI_iTxEditorCharIndex_CwGenerator:
              //       Index of the next character to transfer from the type-ahead input field.
              KeyerGUI_iTxEditorCharIndex_CwGenerator = KeyerGUI_iTxEditorCharIndex_StartOfMsg;
              SL_strncpy( szOriginalText, AnsiString(pRichEdit->Text).c_str(), KEYER_MAX_CHARS_PER_MEMORY );
              // The edit field now acts as the "type-ahead" buffer.
              // But by pressing 'Enter', the operator confirmed that what has been
              // typed so far is "ok to send" - but for the PROGRESS INDICATOR,
              // only the first few characters are handed over from the editor
              // to the CW generator. More characters follow in future calls
              // of TKeyerMainForm::Timer1Timer() .
              KeyerGUI_iStatusIndicatorMemIdx = KEYER_MEMORY_INDEX_TX_EDITOR; // not sending from a "keyer memory" but from the "direct input field"
                       // ,------------------------'
                       // '--> This also suppresses appending the TRANSMITTED TEXT
                       //      back into the same edit field that the USER typed it into,
                       //      like an unwanted 'echo'.
              // Expand certain 'macros' that CwGen_StartReplay() isn't aware of.
              // This also ensures that e.g. PROSIGNS and MACROS are not "cut into pieces".
              pszSrc  = szOriginalText;
              pszDest = szExpandedText;
              nCharsExpanded = KeyerGUI_ExpandTextToSend( &pszSrc, 2/*nCharsWanted*/, pszDest, pszEndstop );
              CwGen_StartReplay( &CwKeyer_Gen, szExpandedText );
              if( nCharsExpanded > 0 )
               { KeyerGUI_iTxEditorCharIndex_CwGenerator += ( pszSrc-szOriginalText ); // here: increment by the number of characters passed to the CW generator when pressing ENTER to START SENDING
                 // Example: Type "<mycall>" into the edit field, THEN press ENTER to start transmitting:
                 //          szOriginalText = "<mycall>" (EIGHT-character-token)
                 //          szExpandedText = "dl4yhf", nCharsExpanded = 6
                 //  -> KeyerGUI_iTxEditorCharIndex_CwGenerator incremented by EIGHT (not 6) !
               }
              KeyerGUI_GetColoursForRichEditWithRxTxInfo( KEYER_GUI_RXTXINFO_TYPING_TXD, &dwTextColor, &dwBkgndColor );
              KeyerGUI_ModifyColoursInRichEdit( pRichEdit, 0/*iFromCharIndex*/,
                                      strlen(szOriginalText)/*nCharsToModify*/,
                                      dwTextColor, dwBkgndColor );
              // Later, in Timer1Timer() -> KeyerGUI_UpdateColourOfSentCharsInRichEdit(),
              //  the character in the edit field are re-coloured again,
              //  to indicate which of them were SENT (instead of being TYPED IN FOR TRANSMISSION).
              break; // end case KEYER_GUI_RXTXINFO_TYPING_TXD
           case KEYER_GUI_RXTXINFO_TYPING_CMD:
              SL_strncpy( szOriginalText, AnsiString(pRichEdit->Text).c_str(), KEYER_MAX_CHARS_PER_MEMORY );
              // ,---------'
              // '--> Begins with the '#' to tell a "command" from "text to send".
              //      This will NOT be passed on to the built-in command interpreter.
              // In contrast to pressing ENTER after entering "text to send",
              // pressing ENTER after entering a COMMAND, clear the field,
              // and allow KeyerGUI_UpdateColourOfSentCharsInRichEdit() to print
              // whatever it wants to:
              KeyerGUI_iRxTxInfoUsage = KEYER_GUI_RXTXINFO_OFF;
              pRichEdit->Clear();
              KeyerGUI_iTxEditorCharIndex_CwGenerator = KeyerGUI_iTxEditorCharIndex_StartOfMsg = 0; // here: cleared along with the text in the "type-ahead" buffer (RichText control)
              iRigctrlErrorCode = KeyerGUI_ExecuteCommand( szOriginalText,
                    szExpandedText/*here: "command response"*/, sizeof(szExpandedText)-1 );
              // In some cases, KeyerGUI_ExecuteCommand() cannot wait for a response,
              // because the "command" is actually addressed to the RADIO (RigControl.c)
              // or even to the REMOTE SERVER (via module CwNet.c) .
              // The GUI thread (at least the one using Borland's VCL) must not
              // wait for anything longer than a few dozen millisecond.
              // Thus the RESPONSE for the command issued above may arrive
              // later (depending on the internet connection, SECONDS later) !
              // Note: Hamlib / "rigctl[d]" uses an "\n" at the end
              //       of the response. We don't want NEW LINE or CARRIAGE RETURN
              //       characters in a single-line edit field. Thus,
              //       KeyerGUI_AppendDataToRichEditWithRxTxInfo() will replace
              //       'control characters' (codes below 0x20) by SPACES.
              // Examples:   command  response
              //             #f       7037333  [Hamlib-compatible "get frequency", result in Hertz]
              //
              //
              if( szExpandedText[0] != '\0' ) // got a response for the command ?
               { // Append it to the command that has just been entered,
                 // but -you guessed it- using a slightly different colour:
                 KeyerGUI_AppendDataToRichEditWithRxTxInfo( pRichEdit, szOriginalText, KEYER_GUI_RXTXINFO_TYPING_CMD );
                 KeyerGUI_AppendDataToRichEditWithRxTxInfo( pRichEdit, " =: ", KEYER_GUI_RXTXINFO_CMD_RESPONSE );
                 KeyerGUI_AppendDataToRichEditWithRxTxInfo( pRichEdit, szExpandedText, KEYER_GUI_RXTXINFO_CMD_RESPONSE );
               }
              else // no response text, but possibly an ERROR CODE ?
              if( iRigctrlErrorCode != RIGCTRL_ERROR_OK )
               { KeyerGUI_AppendDataToRichEditWithRxTxInfo( pRichEdit, szOriginalText, KEYER_GUI_RXTXINFO_ERROR_MSG );
                 sprintf( szExpandedText, " : %s", RigCtrl_ErrorCodeToString(iRigctrlErrorCode) );
                 KeyerGUI_AppendDataToRichEditWithRxTxInfo( pRichEdit, szExpandedText, KEYER_GUI_RXTXINFO_ERROR_MSG );
               }
              KeyerGUI_iRxTxInfoUsage = KEYER_GUI_RXTXINFO_CMD_RESPONSE;
              break; // end case KEYER_GUI_RXTXINFO_TYPING_CMD
         } // end switch( KeyerGUI_iRxTxInfoUsage ) when pressing ENTER in the Info/Command/Rx/Tx-Text-Editor

        fOkToInsert = FALSE; // we don't want the "Enter" aka "Return" inserted in the text
        break;
     case '#' : // hash character may begin a COMMAND INPUT.
        // What it is ("text to send" or "command to execute" will be
        // decided later. But already switch the colour to indicate a COMMAND:
        fOkToInsert = TRUE;
        KeyerGUI_iRxTxInfoUsage = KEYER_GUI_RXTXINFO_TYPING_CMD;
        break;
     default: // any other key.. when ALREADY "Ok to insert", and not in COMMAND ENTRY MODE,
        // then it's a character we can send in Morse code later,
        // or at least something that can be EXPANDED into transmittable morse code,
        // including prosigns like "^KA" or macros like "<mycall>" .
        if( KeyerGUI_iRxTxInfoUsage == KEYER_GUI_RXTXINFO_TYPING_CMD )
         { fOkToInsert = TRUE;
         }
        else if( fOkToInsert ) // not in COMMAND ENTRY mode, so guess it's TEXT TO SEND
         { // Clear the old content of the entire edit field before inserting the new character ?
           if( KeyerGUI_iRxTxInfoUsage != KEYER_GUI_RXTXINFO_TYPING_TXD )
            { pRichEdit->Clear(); // allow using the entire field as 'type-ahead buffer',
              // for example after sending a text from the "constant" keyer memory [1..6].
              // Because this clearing of the edit field happens immediately
              // *before* appending the new character to the edit field (a few lines below),
              // the new character (in cKey) will not be lost.
              // When STARTING to send from here (on cKey='\r', above),
              // outputting of TRANSMITTED CHARACTERS back into the editor
              // is suppressed by setting KeyerGUI_iStatusIndicatorMemIdx := KEYER_MEMORY_INDEX_TX_EDITOR .
              KeyerGUI_iTxEditorCharIndex_CwGenerator = KeyerGUI_iTxEditorCharIndex_StartOfMsg = 0; // here: cleared along with the text in the "type-ahead" editor
            }
           KeyerGUI_iRxTxInfoUsage = KEYER_GUI_RXTXINFO_TYPING_TXD;  // now "typing transmittable data"
         } // end if( fOkToInsert
        break; // end default (for  most 'transmittable characters')
   } // end switch( cKey )

  if( fOkToInsert )  // "ok to insert" (usually APPEND) the new character ... but which COLOUR ?
   {
     if( fCursorAtEndOfText )
      { DWORD dwTextColor, dwBkgndColor;
        KeyerGUI_GetColoursForRichEditWithRxTxInfo( KeyerGUI_iRxTxInfoUsage, &dwTextColor, &dwBkgndColor );
        RichEdit_SetSelColors( pRichEdit->Handle, dwTextColor, dwBkgndColor ); // <- Win32, not Borland VCL !
        // Fortunately, this message handler is invoked BEFORE the VCL
        // inserts the character into the RichEdit control. So we can
        // modify the 'selection attributes' (including the two colours)
        // microseconds BEFORE the VCL passes the event (keystroke) to the editor.
      }
   } // end if( fOkToInsert )

  return fOkToInsert;

} // end KeyerGUI_OnKeyInTransmitTextEditor()



//---------------------------------------------------------------------------
BOOL KeyerGUI_OnKeyInErrorHistory( // based on KeyerGUI_OnKeyInTransmitTextEditor() ...
        TRichEdit *pRichEdit,  // [in] Rich Edit control (here: in the flavour of Borland's VCL.
                               //      In Qt, an equivalent may be QTextEditor.)
        char cKey )            // [in] 8-bit ASCII, including control codes like '\r'
                               // [in,out] KeyerGUI_iRxTxInfoUsage
                               // [in,out] CwKeyer_Gen (may be started from here)
  // As the name shall imply, it's called from the KeyerGUI (e.g. Keyer_Main.cpp)
  // when the operator presses a key with an ASCII equivalent (including ENTER)
  // with the keyboard focus on the Rich Edit control in the 'Debug' tab  .
  // Implemented 04/2025 to allow editing numeric values in a possibly long list
  // of RIG CONTROL PARAMETERS and their current values.
  // Later also used as a simple 'text terminal' to control equipment through
  // an ASCII like command interface.
  //
  // Return value:  TRUE  when the key has been completely processed HERE,
  //                      and we don't want it to be inserted into the editor text
  //                      (for example, to prevent ENTER / '\r' to insert a new line),
  //                FALSE otherwise (e.g. because it's a character not supported
  //                                 in Morse code)
{
  HWND  hWndEditor = pRichEdit->Handle;  // for Win32 API functions, we need this WINDOW HANDLE
  DWORD dwLine,dwSelStart,dwSelEnd;
  int   iLength, iUnifiedPN;
  char  sz255Temp[256];
  const char *cpSrc;
  BOOL  fResult = FALSE;     // assume we don't intercept the key here
  BOOL  fModified;
  T_RigCtrlInstance   *pRC = &MyCwNet.RigControl;
  T_RigCtrl_ParamInfo *pPI;

  if( KeyerGUI_fSerialTerminalActive )
   { return AuxCom_OnKeystrokeInTextTerminal( cKey );
   } // end switch( KeyerGUI_iSerialTerminalMode )

  switch( cKey )
   { case '\r': // carriage return (ENTER key)
        // Determine the SELECTION START and the SELECTION END, using our wrapper for the Win32 API:
        RichEdit_GetTextSelection( hWndEditor, &dwSelStart, &dwSelEnd );
                   // Note: dwSelStart equals dwSelEnd when there is nothing SELECTED
                   //       but just the usual TEXT CURSOR !

        // Convert the SELECTION START (which is a CHARACTER index)
        //  into the TEXT LINE NUMBER from the current 'selection' :
        dwLine = RichEdit_GetTextLineIndexFromCharIndex( hWndEditor, dwSelStart );

        // To "interpret" the content of a line of text, we need it as a plain old C-string:
        iLength = RichEdit_GetTextLineAsCString( hWndEditor, dwLine, sz255Temp, sizeof(sz255Temp)-1);

        // Take a look at the begin of the text line (here: plain text, no RTF tokens):
        cpSrc = sz255Temp;
        if( SL_SkipString( &cpSrc, "PN") ) // line possibly begins with a RigControl Parameter Number ("PN" ...)
         { // e.g. something like "PN040: AudioVolume = 16 %", etc ...
           fResult = TRUE;  // don't break this line into pieces by pressing ENTER,
           // because the user/operator may want to MODIFY and SEND IT AGAIN.
           iUnifiedPN = SL_ParseInteger( &cpSrc );
           if( SL_SkipChar( &cpSrc, ':') ) // ok, obviously a label-like parameter number followed by colon
            { SL_SkipSpaces( &cpSrc ); // skip ANY number of spaces ("pretty-printed input")
              // Next: A symbolic token for the parameter itself, e.g. "AudioVolume".
              //       Should match the PARAMETER NUMBER in most if not all cases:
              if( (pPI = RigCtrl_GetInfoForUnifiedParameterByName( &cpSrc )) != NULL )
               { if( pPI->iUnifiedPN == iUnifiedPN ) // ok, the "PN"-number matches the NAME,
                  { // so dare to modify the value, if the syntax (parsed below) is ok:
                    SL_SkipSpaces( &cpSrc );
                    if( SL_SkipChar( &cpSrc, '=' ) )
                     { SL_SkipSpaces( &cpSrc );
                       // Now for the toughest part: the assigned VALUE.
                       // In some cases, it's an integer, in others, it's a DOUBLE (float),
                       // or it may even be a STRING, or token from a list
                       // (e.g. the modulation type, decorated with suffix 'N' for NARROW BAND, etc).
                       fModified = RigCtrl_SetParamValueFromString( pRC, pPI, &cpSrc );
                       if( fModified )
                        { // Ok, RigControl.c has successfully parsed the VALUE and SET it,
                          // and it was MODIFIED. But RigControl.c didn't automatically
                          // send e.g. a CI-V "write" command. Thus the following:
                          RigCtrl_QueueUpCmdToWriteUnifiedPN( pRC, pPI->iUnifiedPN ); // here: after editing/modifying a value in the parameter list on the 'Debug' tab (!)
                          // '--> For example, will send a CI-V command to set
                          //      the radio's own loudspeaker volume to 3 %
                          // after editing the line with
                          //    e.g.    > "PN040: AudioVolume = 13 %"
                          //    to e.g. > "PN040: AudioVolume = 3 %"
                          // and pressing the ENTER key with the cursor still in that line.
                          // At least, that was the plan in April 2025.
                          // But like MANY other CI-V sequences, this didn't work
                          // right out of the box, and the DEBUG TAB showed the
                          // following (which is actually very nice for testing):
                          // >  139 21:50:27.0 TX 009 FE FE 00 E0 14 01 03 00 FD ; write AudioVolume
                          // >  140 21:50:27.1 RX 006 FE FE E0 94 FA FD ; AudioVolume : NotOK
                          // What went wrong here ? That's outside the scope of Keyer_GUI.cpp.
                          // For details, see C:\cbproj\Remote_CW_Keyer\RigControl.c,
                          // RigCtrl_SendWriteCommandForUnifiedPN() / case RIGCTRL_PN_AUDIO_VOLUME_PERCENT !
                        }
                     } // end if < got the '=' as assignment operator between parameter NAME and VALUE >
                  } // end if( pPI->iUnifiedPN == iUnifiedPN )
               }   // end if < recognized the NAME of a 'unified parameter' a la DL4YHF's RigControl.c
            }     // end if < "PNnnn" followed by a colon (looking like a 'label')
         }       // end if < "PN.." (line generated by 'List Rig Control params on the Debug tab') >
        break;  // end case < key = carriage return (ENTER key) >
     default:  // allow typing any other (ASCII-) key into the editor
        break;
   } // end switch( cKey )

  (void)iLength; // ... assigned a value that is never used ..

  return fResult;
} // end KeyerGUI_OnKeyInErrorHistory()


//---------------------------------------------------------------------------
int KeyerGUI_ExecuteCommand( // called e.g. from KeyerGUI_OnKeyInTransmitTextEditor() when pressing ENTER, after typing a *command*
        const char* pszCommand,  // [in] command entered in the multi-purpose input field in the status line
        char *pszResponse,       // [out] optional response, appended with a different colour in the above field
        int iMaxResponseLength ) // [in] maximum string capacity of pszResponse
  // Return value: hopefully RIGCTRL_ERROR_OK, or one of the error codes in RigControl.h .
{
  int n, iCwNetCmdType;
  pszResponse[0] = '\0';

  SL_SkipChar( &pszCommand, '#' );  // '#' tells a COMMAND from TEXT TO SEND in in the edit field in the status line

  // First check for the RCWK-specific commands like "t"/"talk", etc...
  if( SL_SkipToken( &pszCommand, "t" ) ) // "talk" (to all users currently logged in with a callsign)
   {
     return RIGCTRL_ERROR_OK;
   } // end if < "talk" command > ?
  else if( SL_SkipToken( &pszCommand, "rigctl" ) ) // threat whatever follows like a "rigctl"-compatible command
   { // Treat this command as if it was exececuted on behalf of the "local client" (not remote) = Client[0] :
     n = CwNet_ExecuteRigctldCmd( &MyCwNet, &MyCwNet.Client[0],
                 pszCommand, // [in] Rigctrld/Hamlib compatible command string
                 pszResponse, iMaxResponseLength ); // [out] string that CwNet.c would "send as a response"
                 // (but, in this case, the caller will SHOW IN THE GUI somewhere)
     if( n>=0 ) // ok ..
      { return RIGCTRL_ERROR_OK;
      }
     else // not ok (CwNet_ExecuteRigctldCmd() returned a "negative-made" error code)
      { return -n; // KeyerGUI_ExecuteCommand() returns the original, POSITIVE, error code (not a string length)
      }
   } // end if < "rigctl " > ?
  else if(((iCwNetCmdType=CwNet_CheckCommandType( pszCommand )) == CWNET_CMDTYPE_RIGCTRL_SET)
        || (iCwNetCmdType==CWNET_CMDTYPE_RIGCTRL_GET) )
   { // It's a "rigctl[d]"-compatible command line, "short" or "long" form,
     // so process it similar as above (without the extra "rigctl ") :
     n = CwNet_ExecuteRigctldCmd( &MyCwNet, &MyCwNet.Client[0],
                 pszCommand, pszResponse, iMaxResponseLength );
     if( n>=0 ) // ok ..
      { return RIGCTRL_ERROR_OK;
      }
     else // not ok (CwNet_ExecuteRigctldCmd() returned a "negative-made" error code)
      { return -n; // KeyerGUI_ExecuteCommand() returns the original, POSITIVE, error code (not a string length)
      }
   }


  return RIGCTRL_ERROR_NOT_IMPLEMENTED;
} // end KeyerGUI_ExecuteCommand()

//---------------------------------------------------------------------------
BOOL KeyerGUI_DumpErrorHistoryToRichText(TRichEdit *pRichEdit)
  // Drains messages for the "Debug" tab from the thread-safe message FIFO
  //        and emits them to a VCL RichText edit control (pRichEdit).
{
  T_ErrorFifoEntry *pErrorFifoEntry;
  DWORD dwTextColor  = g_SpecDispControl.clWindowText; // default, depending on the colour scheme ..
  DWORD dwBkgndColor = g_SpecDispControl.clWindowBackground;
  BOOL fAppendedNewText = FALSE;


  DEBUG_iErrorHistoryTail %= C_ERROR_FIFO_SIZE;
  while( DEBUG_iErrorHistoryHead != DEBUG_iErrorHistoryTail )
   {
     pErrorFifoEntry = &ErrorHistoryFifo[ DEBUG_iErrorHistoryTail ];
     DEBUG_iErrorHistoryTail = (DEBUG_iErrorHistoryTail+1) % C_ERROR_FIFO_SIZE;
     // Just having stupid different BACKGROUND COLOURS for individual lines
     // in Borland's "TRichEdit" control is incredibly complex. Principle:
     // 1.) Save the RECENT cursor position before adding the text
     // 2.) Add the new text (line)
     // 3.) Using the stored position from step one, turn the new text into a SELECTION
     // 4.) Perform some voodoo magic (RichEdit_SetSelColors) on the selection
     //     to achieve the desired effect.
     // 5.) Borland's TRichEdit.SelText may also be modified (e.g. replaced)
     //     immediately AFTER RichEdit_SetSelColors() has set the new
     //     back- and foreground colour.
     // 6.) After the above, set TRichEdit.SelLength to ZERO because otherwise,
     //     if the operator presses a key, the selection will be replaced
     //     by the 'appended' character (which almost certainly is NOT intended).
     // Note: The stupid 'index' of characters in Borland's "AnsiString" type
     //       begins at ONE (not ZERO as a C programmer would expect) .
     // >>> This information is duplicated in various other modules.  <<<
     // >>> The 'master' is in C:\cbproj\SpecLab\DebugU1.cpp .        <<<
     // >>> Everything else is a possibly outdated *COPY* .           <<<
     pRichEdit->SelStart = pRichEdit->Text.Length();
     pRichEdit->SelLength= 1;
     // utterly useless: pRichEdit->SelAttributes->NoStupidBackgroundColor = ?
     // (these brainless "SelAttributes" don't support background colours,
     //  even though the native windows "RichEdit" control supports it since ancient times)
     switch( pErrorFifoEntry->iErrorClass & ERROR_CLASS_MASK ) // which background colour ?
      {
        case ERROR_CLASS_FATAL:
              dwBkgndColor = 0xFF00FF; // colour format, from MSBIT to LSBIT: BLUE, GREEN, RED (8 bit each)
              dwTextColor  = 0x000000;
              break;
        case ERROR_CLASS_ERROR:
              dwBkgndColor = 0x0000FF; // colour format, from MSBIT to LSBIT: BLUE, GREEN, RED (8 bit each)
              dwTextColor  = 0xFFFFFF;
              break;
        case ERROR_CLASS_WARNING:
              dwBkgndColor = 0x00FFFF; // colour format, from MSBIT to LSBIT: BLUE, GREEN, RED (8 bit each)
              dwTextColor  = 0x000000;
              break;
        case ERROR_CLASS_RX_TRAFFIC: /* received "network traffic"    (TCP/IP, HTTP, SERIAL, WINKEYER, ..) */
              dwBkgndColor = 0xC0FFC0; // colour format, from MSBIT to LSBIT: BLUE, GREEN, RED (8 bit each)
              dwTextColor  = 0x000000;
              break;
        case ERROR_CLASS_TX_TRAFFIC: /* transmitted "network traffic" (TCP/IP, HTTP) */
              dwBkgndColor = 0xC0FFFF; // colour format, from MSBIT to LSBIT: BLUE, GREEN, RED (8 bit each)
              dwTextColor  = 0x000000;
              break;
        case ERROR_CLASS_INFO:
        case ERROR_CLASS_ALL:
        default:
              switch( g_SpecDispControl.iColourScheme )
               { case COLOUR_SCHEME_DARK:
                    dwBkgndColor = 0x202020;
                    dwTextColor  = 0xFFFFFF;
                    break;
                 default:
                    dwBkgndColor = 0xFFFFFF;
                    dwTextColor  = 0x000000;
                    break;
               }
              break;
      } // end switch( <iErrorClass> )
     // Change the INDIVIDUAL CHARACTER BACKGROUND COLOUR for the "RichEdit"
     // (but not with Borland's VCL.. it takes more than the VCL to get a job done !) :
     // ex: RichEdit_SetSelBgColor( pRichEdit->Handle, dwColor );
     RichEdit_SetSelColors(pRichEdit->Handle,dwTextColor,dwBkgndColor);
     pRichEdit->SelText = AnsiString( pErrorFifoEntry->sz511Text) + "\r\n";
     // NOW, after changing the background colour, de-select the text...
     pRichEdit->SelLength = 0;
     // .. and set the text cursor (which in Borland's geek speak is "SelStart")
     //    to the end of the NEW text :
     // ex: pRichEdit->SelStart = pRichEdit->Text.Length();
     //      '--> This now happens in TKeyerMainForm::Timer1Timer(),
     //           after OTHER KINDS OF TEXT LINES have been appended,
     //           to avoid having to determine the text length
     //           over and over again. Borland's VCL and Microsoft's RichEdit-
     //           control are not spectacularly fast at this !
     // Unfortunately, during the above stupid RichText-editor-operations,
     // the stupid RichText editor's VERTICAL SCROLLBAR began to switch
     // "up and down" like crazy during each update (this only happened when
     // 'a lot of text' was in the editor). Even worse: After adding the last line,
     // and despite the 'text cursor' (aka caret) being at THE END OF THE TEXT,
     // the vertical scrollbar was often AT THE TOP. To cure this, THE CALLER
     // will "scroll the text to the caret" when finihed, conrolled be the
     // following flag:
     fAppendedNewText = TRUE;
   } // end while( DEBUG_iErrorHistoryHead != DEBUG_iErrorHistoryTail )
  return fAppendedNewText;
} // end KeyerGUI_DumpErrorHistoryToRichText()


//---------------------------------------------------------------------------
BOOL KeyerGUI_DumpCharsFromTerminalToRichText(TRichEdit *pRichEdit)
  // Drains the receive-buffers of all 'Auxiliary COM Ports' configured as
  // 'Text Terminals' from the thread-safe message FIFOs in AuxComPort.c,
  // and emits the received text to a VCL RichText edit control (pRichEdit).
{

#if( SWI_NUM_AUX_COM_PORTS > 0 )
  T_AuxComPortInstance *pAuxCom;

  char sz255[256];
  char *pszDest    = sz255;
  char *pszEndstop = sz255 + 250;
  int  iAuxComPortIndex;
  BYTE bRcvdChar;

  DWORD dwBkgndColor = 0xC0FFC0; // colour format, from MSBIT to LSBIT: BLUE, GREEN, RED (8 bit each)
  DWORD dwTextColor  = 0x000000;


  for( iAuxComPortIndex=0; iAuxComPortIndex<SWI_NUM_AUX_COM_PORTS; ++iAuxComPortIndex )
   { pAuxCom = &AuxComPorts[iAuxComPortIndex];
     switch( pAuxCom->pRigctrlPort->iPortUsage )
      {
        case RIGCTRL_PORT_USAGE_WINKEYER_HOST     : // be a HOST using K1EL's "CW Keyer IC for Windows", aka "Winkeyer"/"Winkeyer2"
           break; // end case RIGCTRL_PORT_USAGE_WINKEYER_HOST
        case RIGCTRL_PORT_USAGE_WINKEYER_EMULATOR : // try to EMULATE  K1EL's "CW Keyer IC for Windows", used e.g. to fool the N1MM Logger
           break; // end case RIGCTRL_PORT_USAGE_WINKEYER_EMULATOR
        case RIGCTRL_PORT_USAGE_TEXT_TERMINAL : // control other 'station equipment' via commands entered on the 'Debug' tab
           while( (pszDest<pszEndstop) && (CFIFO_Read( &pAuxCom->sRxFifo.fifo, &bRcvdChar, sizeof(bRcvdChar), NULL/*no timestamp*/ ) > 0 ) )
            { // If the character is in the "harmless 7-bit ASCII" range, emit it as such:
              if( (bRcvdChar>=32) && (bRcvdChar<=127) )
               { SL_AppendChar( &pszDest, pszEndstop, (char)bRcvdChar );
               }
              else switch(bRcvdChar) // something special ->
               { case '\r' :  // CARRIAGE RETURN .. here it comes again, the old "text line ending" chaos !
                    // some (windows) always used "\r\n", others (Unix) only "\n", old MACs only "\r". Bleah.
                    // TeraTerm, possibly with the DEFAULT configuration, only sent "CR" so this is a real mess.
                    if( pAuxCom->bPrevRcvdChar != '\n' )
                     { SL_AppendString( &pszDest, pszEndstop, "\r\n" ); // make Mr "Rich Edit" happy, and append a "\r\n"
                     }
                    break;
                 case '\n' :  // NEW LINE alias LINEFEED ..
                    if( pAuxCom->bPrevRcvdChar != '\r' )
                     { SL_AppendString( &pszDest, pszEndstop, "\r\n" ); // make Mr "Rich Edit" happy, and append a "\r\n"
                     }
                    break;
                 case '\b' :  // BACKSPACE. Seen when pressing that KEY in PuTTY and TeraTerm.
                    SL_AppendString( &pszDest, pszEndstop, "<BS>" );
                    break;
                 case '': case '': case '':
                 case '': case '': case '': // not ASCII but harmless (ANSI)
                 case '':    // (emit german umlauts and sharp 's' as such)
                    SL_AppendChar( &pszDest, pszEndstop, (char)bRcvdChar );
                    break;
                 default:     // silently ignore anything else
                    break;
               }
              pAuxCom->bPrevPrevRcvdChar = pAuxCom->bPrevRcvdChar;
              pAuxCom->bPrevRcvdChar = bRcvdChar;
            } // end while < more characters in THIS 'auxiliary COM port's RX buffer >
           break;
        case RIGCTRL_PORT_USAGE_SERIAL_TUNNEL:
           // Nothing to do here, because AuxComThread() communicates directly
           // with either CwNet.c : ServerThread()  or  CwNet.c : ClientThread() .
           break;
        case RIGCTRL_PORT_USAGE_ECHO_TEST :
           // Nothing to do here, because AuxComThread() -> AuxCom_RunEchoTest()
           // will move anything from the RX FIFO into the TX FIFO .
           break;
        case RIGCTRL_PORT_USAGE_TX_STRESS_TEST :
           // Nothing to do here, because AuxComThread() -> AuxCom_RunTxStressTest()
           // will fill the TX FIFO up to the maximum (and we want to see
           // a GAP-LESS stream of bytes with an O'scope on the TXD pin)
           break;

        default: // case RIGCTRL_PORT_USAGE_NONE
           // also don't do anything here.
           break;
      } // end switch( pInstance->pRigctrlPort->iPortUsage ) [here: in AuxComThread()]
   }
  if( pszDest > sz255 ) // Ok, got some text to append ...
   { // See explanation for the following in KeyerGUI_DumpErrorHistoryToRichText() ...
     pRichEdit->SelStart = pRichEdit->Text.Length();
     pRichEdit->SelLength= 1;
     // Change the INDIVIDUAL CHARACTER BACKGROUND COLOUR for the "RichEdit"
     // (but not with Borland's VCL.. it takes more than the VCL to get a job done !) :
     // ex: RichEdit_SetSelBgColor( pRichEdit->Handle, dwColor );
     RichEdit_SetSelColors(pRichEdit->Handle,dwTextColor,dwBkgndColor);
     pRichEdit->SelText = AnsiString( sz255 ); // no Carriage Return / New Line here !
     // After inserting the new coloured text into the "selection", de-select the text:
     pRichEdit->SelLength = 0;
     return TRUE;  // "have appended something" to Mr. Rich Text Edit Control
   }

#endif // SWI_NUM_AUX_COM_PORTS > 0 ?

  // Arrived here ? Have not appended anything to Mr. Rich Text in this call !
  return FALSE;

} // end KeyerGUI_DumpErrorHistoryToRichText()


//---------------------------------------------------------------------------
void KeyerGUI_ListRigParamsOnDebugTab(
        T_RigCtrlInstance *pRC, // [in] parameters and settings of the 'Rig'
        TRichEdit *pRichEdit )  // [out] 'Rich Text' edit control, e.g Ed_ErrorHistory
  // Lists as many parameters and settings from module RigControl.c
  //       as we could read from the radio, one parameter or setting
  //       per line of text. A few of those parameters can even be
  //       EDITED, and sent back to the radio as a 'write'-command
  //       (more on that in KeyerGUI_OnKeyInErrorHistory() ) .
  // This report can be invoked through the main menu via
  //  "Settings".."List / Edit Rig Parameters ans Settings on the 'Debug' tab",
  // and through the context menu of the 'Debug'-tab's Rich Edit control itself.
{
  char sz255[256], *pszDest, *pszEndstop = sz255+255;
  char sz80Value[84];
  int iErrorClass = ERROR_CLASS_INFO | SHOW_ERROR_IN_RUN_LOG;
  T_RigCtrl_ParamInfo *pPI;
  T_RigCtrl_RadioInfo *pRadioInfo;
  T_RigCtrlFreqMemEntry *pBandStackingReg;
  int    iUnifiedPN, nValidParameters, iValue, i, i1, i2, n;
  int    iEmergencyBrake; // <- the name says it all.. it's just an "emergency brake" for certain loops
  double dblValue;

  pRichEdit->Clear();
  ShowError( iErrorClass, "Rig Parameters and Settings (some may be editable; see help system):" );

  pRadioInfo = RigCtrl_GetRadioInfoByDefaultAddress(
                  pRC->PortInstance[0].iRadioCtrlProtocol, pRC->iDefaultAddress );

  nValidParameters = iUnifiedPN = iEmergencyBrake = 0;
  while( (iUnifiedPN=RigCtrl_EnumerateUnifiedParameters(iUnifiedPN/*iPreviousUnifiedPN*/)) != 0 )
   { if( (pPI = RigCtrl_GetInfoForUnifiedParameterNumber(iUnifiedPN)) != NULL )
      { // Skip parameter numbers that are just ALIASES for others,
        // accessing the same parameter but using a different command
        // (for example the "Selected VFO Frequency", RIGCTRL_PN_SEL_VFO_FREQUENCY,
        //         versus the simple "VFO Frequency", RIGCTRL_PN_FREQUENCY ):
        if( (pPI->iMsgType & RIGCTRL_MSGTYPE_FLAG_IS_ALIAS) == 0 )
        if( RigCtrl_GetParamValueAsString( pRC, pPI, sz80Value, 80/*iMaxLen*/ ) )
         { pszDest = sz255;
           SL_AppendPrintf( &pszDest, pszEndstop, "PN%03d: %s = %s", (int)pPI->iUnifiedPN, pPI->pszToken, sz80Value );
           if( pPI->pszUnit != NULL )
            { SL_AppendPrintf( &pszDest, pszEndstop, " %s",(char*)pPI->pszUnit );
            }
           ShowError(iErrorClass, sz255);
           ++nValidParameters;
         }
      }
     ++pPI;
     if( (iEmergencyBrake++) > 1000 )
      { ShowError( ERROR_CLASS_FATAL, "Bug in ..EnumerateParameters (stuck at pn #%d) !", (int)iUnifiedPN );
        // The reason for this bug was usually a duplicate entry in RigControl.c : RigCtrl_ParameterInfo[],
        // or THE SAME DECIMAL VALUE for two different RIGCTRL_PN_ constants in RigControl.h .
        break;
      }
   } // end while < all "unified parameters" >

  // When in use, show Icom's 'Band Stacking Registers'
  //      (special memory channels automatically updated by the rig,
  //       filled with the most recent THREE frequencies+modes per band),
  // or the 'emulated' band stacking registers from RigControl.c :
  n = pRC->iNumBandStackingRegs;
  if( n > 0 )
   {
     for(i=0; i<n; i+=3 )
      { pszDest = sz255;
        i1 = i;
        i2 = i+2;
        if( i2>=n )
         {  i2=n-1;
         }
        SL_AppendPrintf( &pszDest, pszEndstop, "BandStack[%02d..%02d]:",(int)i1, (int)i2 );
        while( i1<=i2 )
         { pBandStackingReg = &pRC->BandStackingRegs[ i1 ];
           SL_AppendPrintf( &pszDest, pszEndstop, " %8.3lf %3s",
                 (double)(pBandStackingReg->RxTx[0].dblOperatingFreq_Hz * 1e-6),
                 RigCtrl_OperatingModeToString( pBandStackingReg->RxTx[0].iOpMode
                           & RIGCTRL_OPMODE_MASK_TO_STRIP_FLAGS ) );
           if( i1<i2 ) // more "stacked entries" for this band (or "GENE" for all the rest) ?
            { SL_AppendString( &pszDest, pszEndstop, ", " );
            }
           ++i1;
         }
        ShowError(iErrorClass, sz255);
      }
   } // end if < pRC->iNumBandStackingRegs > 0 >

  // Final 'summary' :
  pszDest = sz255;
  SL_AppendPrintf( &pszDest, pszEndstop, "Rig Control: Got %d parameter(s)",
     (int)nValidParameters );
  if( pRC->iNumBandStackingRegs > 0 )
   { SL_AppendPrintf( &pszDest, pszEndstop, " and %d band stacking entries",
     (int)pRC->iNumBandStackingRegs );
   }
  if( pRadioInfo != NULL ) // only if RigControl.c could also detect the RIG MODEL:
   { SL_AppendPrintf( &pszDest, pszEndstop, " from %s on %s .",
      pRadioInfo->pszName, RigCtrl_GetRadioControlPortAsString(pRC) );
   }
  ShowError(iErrorClass, sz255);


  // Dump the thread-safe FIFO (filled by ShowError()) into the thread-unsafe 'RichEdit' control,
  // even if the output on the 'Debug' tab is currently PAUSED :
  KeyerGUI_DumpErrorHistoryToRichText(pRichEdit);  // here: to report 'test results' in the GUI

} // end KeyerGUI_ListRigParamsOnDebugTab()


//---------------------------------------------------------------------------
void KeyerGUI_UpdateSMeter(
        TImage *pImg,  // [in,out] Borland-VCL-style "graphic image", with a "canvas" to paint into
        double dblSMeterLevel_dB ) // [in] "decibel over S0"
        // Keep it simple, begin at "S zero" :
#define C_SMETER_LEVEL_S0   0 /* "S0" on HF:       -20 dBuV */
#define C_SMETER_LEVEL_S9  54 /* "S9" on HF:       +34 dBuV */
#define C_SMETER_LEVEL_MAX 99 /* "S9+45 dB"                 */
{
  int h, w, w2, th, tw, dB, x1, x, xt, x_S9, x_Lev;
  AnsiString sLabel;
  BOOL fShowLabel;

  // Borland's/Embarcadero's VCL-style "TImage" contains a "TPicture".
  //    A "TPicture" is a "TGraphic" container; here it's a "TBitmap".
  //    The "TBitmap" has a "TCanvas", which is the thing we can actually paint on.
  TCanvas *pCanvas;

  // Make sure the "TBitmap" has the same pixel dimensions as the "TImage":
  pImg->Picture->Bitmap->Height = h = pImg->Height;
  pImg->Picture->Bitmap->Width  = w = pImg->Width;
  pCanvas = pImg->Picture->Bitmap->Canvas;

  // Prepare a sufficiently SMALL font for the S-Meter labels:
  pCanvas->Font->Name  = "Arial";
  pCanvas->Font->Size  = 6;
  pCanvas->Font->Color = (TColor)g_SpecDispControl.clWindowText; // <- this may be WHITE or BLACK .. good contrast against the bargraph colours and clWindowBackground
  th = pCanvas->TextHeight( "S" );

  x1 = pCanvas->TextWidth( "S" ); // wide enough to draw e.g. "S0" for the leftmost tick
  w2 = w - x1 - 4;

  // From the width (w, in pixels), calculate the end of the coloured bargraph:
  x_Lev = x1 + (int)((double)(dblSMeterLevel_dB  - C_SMETER_LEVEL_S0) * (double)w2
                   / (double)(C_SMETER_LEVEL_MAX - C_SMETER_LEVEL_S0) );
  if( x_Lev<0 ) x_Lev=0;
  if( x_Lev>=w) x_Lev=w-1;
  // Calculate the pixel position for "S9"
  // (where Icom's S-meter changes from lightblue to red)
  x_S9 = x1 + (int)((double)(C_SMETER_LEVEL_S9  - C_SMETER_LEVEL_S0) * (double)w2
                  / (double)(C_SMETER_LEVEL_MAX - C_SMETER_LEVEL_S0) );

  // Draw the "bargraph" into the BACKGROUND ..
  x = x_Lev;
  if( x > x_S9 )
   {  x = x_S9;
   }
  pCanvas->Brush->Color = clSkyBlue;    // colours inspired by Icom IC-7300 ..
  pCanvas->FillRect( TRect(x1,0,x,h) ); // below S9 = light blue
  if( x_Lev > x_S9 )
   { pCanvas->Brush->Color = clRed;    // above S9 = red
     pCanvas->FillRect( TRect(x_S9,0, x_Lev,h) );
   }
  pCanvas->Brush->Color = (TColor)g_SpecDispControl.clWindowBackground; // <- this may be BLACK or WHITE !
  pCanvas->FillRect( TRect(0,0, x1,h) );
  pCanvas->FillRect( TRect(x_Lev,0, w,h) );


  // Draw a crude "S-meter scale" into the foreground
  SetBkMode( pCanvas->Handle, TRANSPARENT ); // <- old Windows 'GDI' function .. what's the VCL equivalent ?
  pCanvas->Pen->Color = pCanvas->Font->Color;
  for( dB=C_SMETER_LEVEL_S0; dB<=C_SMETER_LEVEL_S9; dB+=6 )
   { x = x1 + (int)((double)( dB - C_SMETER_LEVEL_S0) * (double)w2
                  / (double)(C_SMETER_LEVEL_MAX - C_SMETER_LEVEL_S0) );
     pCanvas->MoveTo( x, 0 );
     pCanvas->LineTo( x, h-th );
     if(dB==C_SMETER_LEVEL_S0)
      { sLabel = "S0";
      }
     else
      { sLabel = IntToStr( (dB-C_SMETER_LEVEL_S0) / 6 );
      }
     tw = pCanvas->TextWidth( sLabel );
     xt = x-tw/2;
     pCanvas->TextOut( xt, h-th, sLabel );
   }
  // Above "S9" : only show "+x" every TWENTY dB (like Icom)
  for( dB=C_SMETER_LEVEL_S9+10; dB<=C_SMETER_LEVEL_MAX; dB+=10 )
   { x = x1 + (int)((double)( dB - C_SMETER_LEVEL_S0) * (double)w2
                  / (double)(C_SMETER_LEVEL_MAX - C_SMETER_LEVEL_S0) );
     pCanvas->MoveTo( x, 0 );
     pCanvas->LineTo( x, h-th );
     sLabel = AnsiString("+") + IntToStr( dB-C_SMETER_LEVEL_S9 );
     tw = pCanvas->TextWidth( sLabel );
     xt = x-tw/2;
     pCanvas->TextOut( xt, h-th, sLabel );
   }


} // end KeyerGUI_UpdateSMeter()

//---------------------------------------------------------------------------
void KeyerGUI_UpdateMultiFunctionMeter( // renders any of the horizontal bargraphs on the "Mult-funtion meter" (panel)
        TImage *pImg,  // [in,out] Borland-VCL-style "graphic image", with a "canvas" to paint into
        int x1, int y1, int x2, int y2, // [in] graphic area for one of the six(?) "meters" within pImg
        double dblValue, // [in] value to be displayed (percent, "dB", SWR, current, voltage, power, temperature..)
        double dblMinValue, double dblMaxValue, double dblStepwidth, // [in] physical value range (also used for the labelled scale)
        const char *pszParamName, // [in] name like "Power", "ALC", "COMP", "SWR", "Id", "Vd", "Temp"
        const char *pszPhysUnit,  // [in] physical unit (shown at the end of the tick scale), e.g. "%", "dB", "V", "A", "W", "°C"
        double dblRedStartValue, double dblRedEndValue ) // e.g. 10.0 .. 12.0 [Volts for the "red line" indicating UNDERVOLTAGE]
  // Result (approximately..)
  //
  //    ,----------------------------------------,   -  y1
  //    | Power ##|##|##|##|##|##|##|##|##|##| % |
  //    |       0 10 20 30 40 50 60 70 80 90 100 |
  //    '----------------------------------------'   -  y2
  //    |       |                            |   |
  //    x1     x1i ("inner bargraph area"  x2i  x2

{
  int x1i, y1i, x2i, y2i, h, w, w2, th, tw, dB, x, xt, yt;
  int x_Lev, x_RedStart, x_RedEnd;
  double d;
  AnsiString sLabel;
  BOOL fShowLabel;
  BOOL fEraseAll = FALSE;

#define L_COLOUR_TEST 0 // (0)=normal compilation, (1)=TEST using fixed colours

  // Borland's/Embarcadero's VCL-style "TImage" contains a "TPicture".
  //    A "TPicture" is a "TGraphic" container; here it's a "TBitmap".
  //    The "TBitmap" has a "TCanvas", which is the thing we can actually paint on.
  TCanvas *pCanvas;


  // Make sure the "TBitmap" has the same pixel dimensions as the "TImage":
  h = pImg->Height;
  w = pImg->Width;
  if( (pImg->Picture->Bitmap->Height != h) || (pImg->Picture->Bitmap->Width != w) )
   { fEraseAll = TRUE;
     pImg->Picture->Bitmap->Height = h;
     pImg->Picture->Bitmap->Width  = w;
   }
  pCanvas = pImg->Picture->Bitmap->Canvas;

  if( fEraseAll )
   {
#   if( L_COLOUR_TEST ) // Fill the entire bitmap with a painful colour ?
     pCanvas->Brush->Color = clRed;
#   else  // no eye-cancer please ..
     pCanvas->Brush->Color = (TColor)g_SpecDispControl.clWindowBackground; // <- this may be BLACK or WHITE !
#   endif
     pCanvas->Brush->Style = bsSolid;
     pCanvas->FillRect( TRect(0,0,w,h) );
   }

  // return;  // what caused the WHITE-filled area ? [not the stuff below..]

  // Prepare a sufficiently SMALL font for the S-Meter labels:
  pCanvas->Font->Name  = "Arial";
  pCanvas->Font->Size  = 6;
#if( L_COLOUR_TEST )
  pCanvas->Font->Color = clYellow;
#else
  pCanvas->Font->Color = (TColor)g_SpecDispControl.clWindowText; // <- this may be WHITE or BLACK .. good contrast against the bargraph colours and clWindowBackground
#endif // L_COLOUR_TEST ?
  th = pCanvas->TextHeight( "S" );
  tw = pCanvas->TextWidth( "S" ); // wide enough to draw e.g. "S0" for the leftmost tick

  // "inner width" for the bargraph, without margins for the labels and :
  x1i = x1 + 6*tw;
  x2i = x2 - 2*tw;
  y1i = y1 + 2;
  y2i = y2 - 2;
  w2  = x2i - x1i;

  // From the "inner width" (w2 in pixels), calculate the end of the coloured bargraph:
  x_Lev = x1i + (int)((double)(dblValue    - dblMinValue) * (double)w2
                    / (double)(dblMaxValue - dblMinValue) );
  UTL_LimitInteger( &x_Lev, x1, x2 );
  // Calculate the pixel position for "S9"
  // (where Icom's S-meter changes from lightblue to red)
  x_RedStart = x1i + (int)((double)(dblRedStartValue - dblMinValue) * (double)w2
                         / (double)(dblMaxValue      - dblMinValue) );
  x_RedEnd = x1i + (int)((double)(dblRedEndValue - dblMinValue) * (double)w2
                       / (double)(dblMaxValue    - dblMinValue) );
  UTL_LimitInteger( &x_RedStart, x1i, x2i );
  UTL_LimitInteger( &x_RedEnd,   x1i, x2i );
  // Note: For SOME "meters", the RED AREA is at the start (e.g. undervoltage).
  //       For OTHER "meters", the RED AREA is at the end (e.g. PA temperature).
  //       The "red area" is indicated as a line below the bargraph.
  //       Unlike the S-meter, the colour of the bargraph itself doesn't seem
  //       to change depending on the displayed value (?)

  // Clear the entire background (no problem with flicker; we use "double buffering")
#if( L_COLOUR_TEST )
  pCanvas->Brush->Color = clBlue;
#else
  pCanvas->Brush->Color = (TColor)g_SpecDispControl.clWindowBackground; // <- this may be BLACK or WHITE !
#endif // L_COLOUR_TEST ?
  pCanvas->Brush->Style = bsSolid;
  pCanvas->FillRect( TRect(x1,y1,x2, y2) );
    // ,-------------'
    // '--> "TRect represents the dimensions of a rectangle. The coordinates
    //       are specified as either four separate integers representing the
    //       left, top, right, and bottom sides, or as two points (...) "

#if( L_COLOUR_TEST )
  pCanvas->Brush->Color = clGreen;
#else
  pCanvas->Brush->Color = (TColor)g_SpecDispControl.clWindowText; // <- this may be BLACK or WHITE !
#endif // L_COLOUR_TEST ?

  pCanvas->FrameRect( TRect(x1+2,y1+2,x2-2,y2-2) ); // quite counter-intuitive, "FrameRect()" doesn't use the "Pen" but the "Brush" to draw a 1-pixel-wide border..

#if( L_COLOUR_TEST )
  pCanvas->Font->Color = (TColor)0x00C0FF; // "clOrange", but that doesn't exist in the VCL
#else
  pCanvas->Font->Color = (TColor)g_SpecDispControl.clWindowText;
#endif // L_COLOUR_TEST ?
  SetBkMode( pCanvas->Handle, TRANSPARENT ); // <- Windows 'GDI' .. what's the VCL equivalent ?
  if( (pszPhysUnit != NULL ) && (pszPhysUnit[0] != '\0') )
   { yt = (y1+y2)/2-th;  // vertical position for the parameter name ..
   }
  else // no "physical unit" so vertically align the PARAMETER NAME:
   { yt = (y1+y2)/2-th/2; // <- for example, SWR and ALC were dimensionless
   }
  if( pszParamName != NULL )
   { pCanvas->TextOut( x1 + 4, yt, pszParamName );
     yt += th;
   }
  if( (pszPhysUnit != NULL ) && (pszPhysUnit[0] != '\0') )
   { pCanvas->TextOut( x1 + 8, yt, "/ " + AnsiString(pszPhysUnit) );
     yt += th;
   }
  SetBkMode( pCanvas->Handle, OPAQUE ); // <- old Windows 'GDI' function ("OPAQUE" is what the VCL expects when it calls the GDI internally)

#ifdef __BORLANDC__  // "... is assigned a value that is never used".. shut up ..
  (void)yt;
#endif


#if( L_COLOUR_TEST )
  pCanvas->Brush->Color = clGray;
#else
  pCanvas->Brush->Color = (TColor)g_SpecDispControl.clWindowBackground; // <- this may be BLACK or WHITE !
#endif // L_COLOUR_TEST ?


  // Draw the "bargraph" into the BACKGROUND ..
  x = x_Lev;
  if( x>x1i )
   { pCanvas->Brush->Color = clSkyBlue;         // colours inspired by Icom IC-7300 ..
     pCanvas->FillRect( TRect(x1i,y1i,x,y2i) ); // light blue (for MOST bargraphs)
   }

  // Draw a crude "labelled scale" into the foreground ?
  if( (dblStepwidth>0.0) && (dblMaxValue>dblMinValue) )
   {
#   if( L_COLOUR_TEST )
     pCanvas->Font->Color = (TColor)0x00C0FF; // "clOrange", but that doesn't exist in the VCL
#   else
     pCanvas->Font->Color = (TColor)g_SpecDispControl.clWindowText;
#   endif // L_COLOUR_TEST ?
     pCanvas->Pen->Color = pCanvas->Font->Color;
     SetBkMode( pCanvas->Handle, TRANSPARENT ); // <- old Windows 'GDI' function .. what's the VCL equivalent ?
     for( d=dblMinValue; d<=dblMaxValue; d += dblStepwidth )
      { x = x1i + (int)((double)( d          - dblMinValue) * (double)w2
                      / (double)(dblMaxValue - dblMinValue) );
        pCanvas->MoveTo( x, y1i );
        pCanvas->LineTo( x, y2i-th );
        if( dblStepwidth >= 1.0 )
         { sLabel = IntToStr( (int)d );
         }
        else
         { sLabel = FormatFloat( "#.0", d );
         }
        tw = pCanvas->TextWidth( sLabel );
        xt = x-tw/2;
        pCanvas->TextOut( xt, y2i-th, sLabel );
      }
     SetBkMode( pCanvas->Handle, OPAQUE );

   }

} // end KeyerGUI_UpdateMultiFunctionMeter()


#if( SWI_HARDCORE_DEBUGGING )
//---------------------------------------------------------------------------
void CheckSystemHealth(const char *pszModuleName, int iSourceLine) // checks for 'memory corruption'...
  // Periodically called from a few places via macro CHECK_SYSTEM_HEALTH() .
  // Added 2024-02-25 when suspecting a problem with the hyper-complicated,
  // memory-hoggig Ogg/Vorbis encoder (that later turned out to be utterly
  // non-suited for low-latency audio, at least when used for the <audio>
  // element in HTML 5 without megatons of Javascript..)
  //
  // Because CheckSystemHealth() may be called from WORKER THREADS, hundreds
  // of times per second, keep the "checks" performed in this function to the
  // ultimate minimum (e.g. check a few 'magic numbers' at the end of FIFOs,
  // etc).
{
  BOOL fGotcha = FALSE;
  static BOOL firstBug = TRUE;
  //  2024-02-25: Out of the blue, there was garbage in MyDirectSound.sz255OutputDeviceName :
  if(  (MyDirectSound.sz255OutputDeviceName[0] >  0x00)
     &&(MyDirectSound.sz255OutputDeviceName[0] <= 0x20) )
   { fGotcha = TRUE;
   }

  if( fGotcha && firstBug )
   { ShowError( ERROR_CLASS_ERROR, "System health check failed, caller = %s, line = %d",
                      pszModuleName, (int)iSourceLine );
   }

} // end CheckSystemHealth()
#endif // SWI_HARDCORE_DEBUGGING ?


//---------------------------------------------------------------------------
const char* RigCtrl_GetRadioControlPortAsString(T_RigCtrlInstance *pRC) // -> e.g. "COM5", if that's the port "talking to the radio" .
  // Called from within RigControl.c to emit 'info messages' like
  //     "RigControl: IC-9700 on COM5, 432250.0 kHz, CWN"  .
  // Since RighControl.c doesn't know anything about serial ports,
  // RigCtrl_GetRadioControlPortAsString() must be provided by the application.
{
  static char sz15Result[16];
  int iRigCtrlPort;
  T_RigCtrl_PortInstance *pRigCtrlPort;

  switch( CwKeyer_Config.iRadioControlProtocol )
   { case RIGCTRL_PROTOCOL_NONE : // Even if the dedicated RADIO CONTROL PORT is off,
          // the rig-control instance may be actively communicating with a radio,
          // or eavesdrop on the communication with a 3rd party application and the radio
          // via 'Additional COM Port' (ex: "auxiliary COM port'). Thus, similar as
          // in the "Test Report" / RIG CONTROL statistics :
          for( iRigCtrlPort=0; iRigCtrlPort<<RIGCTRL_NUM_PORT_INSTANCES; ++iRigCtrlPort )
           { // iRigCtrlPort is an index into PortInstance[] :
             //   0 = RIGCTRL_PORT_RADIO, 1 = RIGCTRL_PORT_AUX_COM_1, ... !
             pRigCtrlPort = &pRC->PortInstance[iRigCtrlPort];
             // Only if DECODED MESSAGES arrived on this port, we know it's really active:
             if( (pRigCtrlPort->dwNumMessagesSent + pRigCtrlPort->dwNumMessagesRcvd) != 0 )
              { return RigCtrl_RigControlPortNrToString(iRigCtrlPort);
              }
           }
          return "NoPort";
     case RIGCTRL_PROTOCOL_ICOM_CI_V:
     case RIGCTRL_PROTOCOL_YAESU_5_BYTE:
        // HERE, to keep things simple in the Remote CW Keyer, these protocols
        // are only supported via "the same local COM port that also KEYS the rig"
        //  (provides the Morse code modulation):
        sprintf( sz15Result, "COM%d", (int)CwKeyer_Config.iRadioKeyingAndControlPort );
        return sz15Result;
     // case RIGCTRL_PROTOCOL_KENWOOD, RIGCTRL_PROTOCOL_YAESU_CAT, RIGCTRL_PROTOCOL_ELECRAFT, ...
     // NONE of these will ever be supported natively .. so instead,
     //   use use something Hamlib/Rigctld-compatible that runs a server on a
     //   TCP/IP port, e.g wfview's built-in "rigctld emulation" !
     //
     case RIGCTRL_PROTOCOL_HAMLIB_RIGCTLD : // added 2024-06 in the Remote CW Keyer...
     // ex: case RIGCTRL_PROTOCOL_FLRIG_XMLRPC   :
        // Doesn't use the built-in CI-V engine in DL4YHF's RigControl.c,
        // but the ASCII-based 'Hamlib' compatible commands via TCP/IP:
        return CwKeyer_Config.sz80RemoteRigCtrlServerAddress;

     default:
        sprintf( sz15Result, "??%d??", (int)CwKeyer_Config.iRadioControlProtocol );
        return sz15Result;

   } // end switch( CwKeyer_Config.iRadioControlProtocol )

} // end RigCtrl_GetRadioControlPortAsString()


