//---------------------------------------------------------------------------
// File: C:\cbproj\Remote_CW_Keyer\KeyerThread.c
// Date: 2023-12-20
// Author: Wolfgang Buescher (DL4YHF)
// Purpose: Experiment to key certain Icom transceivers (IC-7300, IC-9700, IC-705)
//          via USB, using the rig's built-in virtual 'COM' port,
//          as explained for the IC-7300 in the "Full Manual", chapter 12,
//          "SET MODE" / "USB Keying (CW)" .
//          Later grown into the 'almost-real-time core' for a network-capable
//          Morse Code keyer (not just a "keyer for the serial port" anymore).
//---------------------------------------------------------------------------

#include "switches.h"  // project specific compiler switches ("options"),
                       // must be included before anything else !
#include "yhf_type.h"  // classic types like BYTE, WORD, DWORD, BOOL, ..

#include <windows.h>  // try NOT to use any Borland-specific stuff here (no VCL)
#include <string.h>
#include <stdio.h>    // no "standard I/O" but e.g. sprintf() used here

#if( SWI_USE_DSOUND ) // use "Direct Sound" / dsound_wrapper.c ?
# include "dsound_wrapper.h"
#endif // SWI_USE_DSOUND ?

#if( SWI_USE_WAVE_AUDIO || SWI_USE_MIDI )
# include <mmsystem.h> // maybe MIDI is fast enough to start / stop playing notes ? [No, it's not. At least not the crappy MIDI emulator in Windows.]
#endif // SWI_USE_WAVE_AUDIO or SWI_USE_MIDI ?

#include "Utilities.h" // stuff like UTL_iWindowsVersion, UTL_iAppInstance, ShowError(), etc
#include "Timers.h"    // high-resolution 'current time'-function, T_TIM_Stopwatch(), etc.
#include "Debouncer.h" // 'debouncer' for paddle inputs (T_Debouncer in T_ElbugInstance)
#include "Elbug.h"     // old 'Elbug' functions, converted from PIC-assembler to "C"
#include "CwGen.h"     // CW generator (converts text to Morse code)
#include "CwDSP.h"     // CW-'Digital Signal Processor' / sidetone generator
#include "CwNet.h"     // Socket-based 'Client or Server' for the Remote CW Keyer
#include "CwKeyer.h"   // CW-keyer ("from paddle contacts to Morse code")
#include "StraightKeyDecoder.h" // Morse decoder for input from a STRAIGHT KEY
#if( SWI_NUM_AUX_COM_PORTS > 0 ) // compile with support for 'Auxiliary / Additional COM ports' / Winkeyer emulation ?
# include "AuxComPorts.h" // structs and API functions for the "Auxiliary" (later: "Additional") COM ports
#endif // SWI_NUM_AUX_COM_PORTS ?


//---------------------------------------------------------------------------
// The following incarnation of the "serial port keyer thread" uses a windows
//    "multimedia"-timer for precise timing of the 'keyer thread'.
//
//    As often, Microsoft says the "multimedia timer" is deprecated,
//    and just a "legacy feature", but the author had more important things to do
//    than adapting C code for each new API they come up with, only to find out
//    it does NOT (!) work on 'legacy hardware' or older windows version anymore !
// > Microsoft suggests to rewrite existing code that used the legacy APIs,
// > to use "the new APIs" (whatever those "new APIs" may be) if possible.
//   [again: "no, thanks".  The same applies to 'Wave Audio' or 'DirectSound']
//---------------------------------------------------------------------------

T_CwKeyerConfig CwKeyer_Config;   // configuration for a single instance
T_ElbugInstance CwKeyer_Elbug;    // 'Elbug' instance, with its own configuration
T_StraightKeyDecoder CwKeyer_StraightKeyDecoder; // Morse decoder for input from a STRAIGHT KEY
T_KeyerTimingScope   CwKeyer_TimingScope; // 'timing scope' for in- and output signals
T_KeyerDecoderFifo   CwKeyer_DecoderFifo; // FIFO written by worker thread,
                                          // drained by multiple readers (e.g. the GUI)
T_KeyerSerialPortFifo Keyer_RadioControlPortRxFifo, Keyer_RadioControlPortTxFifo;

T_CwGen CwKeyer_Gen; // 'CW Generator' instance, to convert text to Morse code
T_CwDSP CwKeyer_DSP; // 'CW Digital Signal Processor', with sidetone generator, CW filter/decoder, etc
// Note: A T_CwNet instance ('MyCwNet') is married with the Keyer in Keyer_Main.cpp !

#if( SWI_USE_MIDI )  // ever tried CW on a 'Recorder' (deutsch: Blockflte) ?
  // that's what MIDI sounded like: huuuh hu huuuh hu, huuuh huuuh hu huuuh .
int       CwKeyer_iMidiDeviceId = 0; // ex: MIDI_MAPPER;
HMIDIOUT  CwKeyer_hMidiDevice   = INVALID_HANDLE_VALUE;
typedef union t_ShortMidiMsg { DWORD dwData; BYTE bData[4]; } T_ShortMidiMsg;
int       CwKeyer_iMidiInstrument = 0x51;
        // Which "musical instrument" to use for the sidetone ?
        // de.wikipedia.org/wiki/General_MIDI listed a few "simple" sounds:
        // > 0x01 = acoustic grand piano
        // > 0x48 = piccolo (flute) : far too slow attack
        // > 0x50 = "Lead 1" (what a stupid misleadig name for square wave)
        // > 0x51 = "Lead 2" (stupid misleadig name for sawtooth)
        //
        // All of the instruments had unsuited Attack-Decay-Sustain-Release parameters.
        // A Wikipedia article mentioned in *General MIDI Level 2*,
        // Attack- and  Release time can be tailored via "CC" (Control Change)
        // message. But this had no effect with the Windoze 'Midi Mapper'.
int       CwKeyer_iMidiNote = 0x3C;   // 0x3C = "Middle-C", says the MIDI spec
BOOL      CwKeyer_fMidiToneOn = FALSE;
#endif // SWI_USE_MIDI ?

char CwKeyer_sz255LastError[256]; // "last error message" as plain text
const char C_EmptyString[] = "\0";

#define L_USE_MULTIMEDIA_TIMER_CALLBACK 1 // 1=use a CALLBACK, 0=use an EVENT

int CwKeyer_iThreadStatus = KEYER_THREAD_STATUS_NOT_CREATED; // may contain one of the following:
#   define KEYER_THREAD_STATUS_NOT_CREATED 0
#   define KEYER_THREAD_STATUS_LAUNCHED    1
#   define KEYER_THREAD_STATUS_RUNNING     2 // <- only set ONCE(!), immediately when entering the thread function
#   define KEYER_THREAD_STATUS_TERMINATE   3 // <- just a KLUDGE to "request POLITE termination", but takes a few dozen milliseconds to actually TERMINATE !
#   define KEYER_THREAD_STATUS_TERMINATED -1 // <- set by the worker thread if it TERMINATED ITSELF

DWORD Keyer_dwThreadLoops = 0; // number of thread loops, counted in KeyerThread()
DWORD Keyer_dwThreadErrors= 0; // number of errors (mostly TIMING ERRORS) counted in KeyerThread()
DWORD Keyer_dwSerialPortErrors=0; // number of errors related with SERIAL PORTS counted in KeyerThread()


HANDLE Keyer_hComPort_IN  = INVALID_HANDLE_VALUE; // the "input" COM port is the one
       // with the PADDLE / Morse straight key attached .
HANDLE Keyer_hComPortRadioKeyingAndControl = INVALID_HANDLE_VALUE; // the "output" COM port drives the transceiver's
       // CW-input (via DTR), and maybe the PTT ("Push-To-Transmit", "Request-To-Send", via RTS) .
       // Note: The RCWKeyer GUI occasionally checks for INVALID_HANDLE_VALUE,
       //       because when some worker thread decides to CLOSE the port
       //       after serious errors, and -after closing a port- sets its handle
       //       to INVALID_HANDLE_VALUE again.
       // Beware: For THE ABOVE HANDLES, some lunatic in Richmond(?) decided
       //         that zero/NULL is a perfectly valid 'handle value'.
       //         Thus the need for INVALID_HANDLE_VALUE for THESE KINDS OF HANDLES.
       //         The "invalid handle value" for OTHER kinds of handles
       //         isn't INVALID_HANDLE_VALUE but zero (or NULL).
HANDLE Keyer_hThread = NULL; // also a HANDLE, but for a THREAD, and HERE, "NULL" means invalid. Yucc.
DWORD  Keyer_dwThreadId = 0; // the joys of Win32 API programming.. not just a
                             //  "thread handle" but also a "thread ID"...
                             //  we may need it for polite thread termination,
                             //  or for waking up the thread via event one day.

BYTE   Keyer_bRadioControlPortRxFifo[KEYER_SERIAL_PORT_FIFO_SIZE];
DWORD  Keyer_dwRadioControlPortRxFifoHead, Keyer_dwRadioControlPortRxFifoTail;



MMRESULT Keyer_MultimediaTimerId; // value returned by timeSetEvent() (that's the counter-intuitive function for their MULTIMEDIA-timer)
HANDLE Keyer_hEventFromMMTimer = NULL;  // may be NULL when "not in use". Must be public, because a CALLBACK cannot be a C++ class method.
       // ,-----------------------------|__|
       // '--> Thanks to some luncatic who designed the Win32 API,
       //      for SOME kinds of 'HANDLE' the handle value ZERO (alias NULL)
       //      is perfectly valid, while for other kinds of a 'HANDLE' variable,
       //      the "invalid handle value" is INVALID_HANDLE_VALUE, not NULL !
T_TIM_Stopwatch Keyer_swMMTimerToThread;  // .. to measure the latency between SetEvent() and returning from WaitForSingleObject() ..
long   Keyer_i32LatencyMMTimerToThread;   // measured latency between SetEvent() and returning from WaitForSingleObject(), in microseconds
long   Keyer_i32TimeSpentInSendEvent_us;  // number of microseconds spent in ::SendEvent() itself .

DWORD  CwKeyer_dw8ThreadIntervals_us[8]; // microseconds per thread loop (for testing, displayed in the experimental GUI)

DWORD  Keyer_dwCurrentSignalStates = 0;   // bitwise combination of 1 << KEYER_SIGNAL_INDEX_xyz
BOOL   Keyer_fDotInputWasPassive, Keyer_fDashInputWasPassive; // flags to prevent ENDLESS keying; TRUE when ok (saw a PASSIVE input in the previous FIVE SECONDS)

BOOL   CwKeyer_fUpdateAllOutputs = FALSE; // kludge to 'send all outputs' even if they didn't change
BOOL   CwKeyer_fLocalMorseOutput = FALSE; // kludge to plot the Elbug-output even when NOT driving the CW modulation output.
                // '--> input for CwKeyer_CollectDataForTimingScope(), when using MANUAL PTT CONTROL but PTT currently off.
T_TIM_Stopwatch CwKeyer_swMorseActivityTimer; // timer related to CwKeyer_fLocalMorseOutput,
                // '--> used to avoid flickering display in the GUI - see STATUS_INDICATOR_CW_TEST.

int    CwKeyer_iTestMode = KEYER_TEST_MODE_OFF; // "normal operation" (keyer controls digital outputs)

int    CwKeyer_iSpeedTestPeaks_us[8]; // microseconds spent in each thread loop for the following operations:
       // [KEYER_SPEEDTEST_POLL_DIGITAL_INPUTS] : time spent in KeyerThread() loop to poll MODEM STATUS LINES (as 'digital inputs')
       // [KEYER_SPEEDTEST_SET_DIGITAL_OUTPUTS] : time spent in KeyerThread() loop to drive MODEM CONTROL LINES (as 'digital outputs')
       // [KEYER_SPEEDTEST_POLL_KEYBOARD] : time spent in each KeyerThread() loop to poll SHIFT and CONTROL on the PC's keyboard
long   CwKeyer_i32SpeedTestSums_us[8];   // SUM of microseconds used for the above operations
long   CwKeyer_i32SpeedTestCounts_us[8]; // counters to convert the above SUMS into AVERAGES, in CwKeyer_GetSpeedTestResults()

//----------------------------------------------------------------------------
// Global variables for hardcore debugging / post-mortem crash analysis.
//        Useful when the debugger's call stack shows nothing but garbage,
//        e.g. after an exception at the infamous address 0xFEEEFEEE .
//        ( C++Builder's debugger could still inspect SIMPLE GLOBAL
//          VARIABLES after most kinds of exceptions, but of course nothing
//          stack-based because the CPU- or task-stack was usually trashed.)
//----------------------------------------------------------------------------
#if(SWI_HARDCORE_DEBUGGING) // (1) = hardcore-debugging, (0)=normal compilation
 int Keyer_iLastSourceLine = 0; // WATCH THIS after crashing with e.g. "0xFEEEFEEE"  ...
# define HERE_I_AM__KEYER()  Keyer_iLastSourceLine=__LINE__
     // (see complete list of other XYZ_iLastSourceLine variables to watch
     //  in C:\cbproj\Remote_CW_Keyer\Keyer_Main.cpp, near GUI_iLastSourceLine)
#else
# define HERE_I_AM__KEYER()
#endif // SWI_HARDCORE_DEBUGGING ?


//----------------------------------------------------------------------------
// "Internal function prototypes" (functions called before their implementation)
//----------------------------------------------------------------------------

DWORD WINAPI KeyerThread( LPVOID lpParam );
static BOOL OpenAndConfigureSerialPort(int iComPortNumber,long i32BitsPerSecond,
              int iOptions, int iDtrControl, int iRtsControl, HANDLE *pHandle );
# define SP_OPTION_NONE 0 // bitwise combineable values for the 'Serial Port Options':
# define SP_OPTION_USE_SMALL_TX_BUFFER 0x0001 // .. to have the SIDETONE stop quickly

static BOOL Keyer_StartMMTimer( int iTxInterval_ms );
static void Keyer_StopMMTimer( void );

static void Keyer_UpdateSpeedTestResult( int iTestItem, int nMicroseconds );
static BOOL Keyer_GetMorseOutputFromRxFifo( T_CwNet *pCwNet );


//----------------------------------------------------------------------------
// "Application Interface" (functions called e.g. from the test application)
//----------------------------------------------------------------------------

//----------------------------------------------------------------------------
void CwKeyer_SetDefaultConfig( T_CwKeyerConfig *pConfig )
  // Called from the main application (or "GUI") before loading an actual
  // configuration FROM A FILE. The result (in *pConfig) are the "defaults".
{
  int i;
#if( SWI_NUM_AUX_COM_PORTS > 0 )
  T_AuxComConfig *pAuxComCfg;
#endif

  // ex: memset( pConfig, 0, sizeof(T_CwKeyerConfig) );
  if( pConfig->iComPortNumber_IN == 0 )
   {  pConfig->iComPortNumber_IN  = -1;  // "COM"-port for the Morse key / paddle contacts : negative = "NONE"
      // The "COM port numbers" shall NOT be affected when selecting
      // "Settings" .. "Restore Default" -> CwKeyer_SetDefaultConfig() !
   }
  if( pConfig->iRadioKeyingAndControlPort == 0 )
   {  pConfig->iRadioKeyingAndControlPort = -1;  // "COM"-port for keying the radio : negative = "NONE"
   }
  pConfig->iRadioControlProtocol = RIGCTRL_PROTOCOL_ICOM_CI_V;
  pConfig->iRadioControlBaudrate = 115200; // [bits/second] to use Spectrum data from IC-7300, IC-9700 & Co
  pConfig->iRadioCIVAddress = 0x00; // e.g. 0x00 for AUTO-DETECT, 0xA2 for IC-9700, 0x94 for IC-7300, 0xA4 for IC-705, ...
  pConfig->fEnableBuiltInHamlibServer = FALSE;
  pConfig->dwMessageFilterForLog = RIGCTRL_MSGFILTER_ECHO | RIGCTRL_MSGFILTER_FREQUENCY_REPORT | RIGCTRL_MSGFILTER_SPECTRUM;
  pConfig->iMorseKeyType   = KEY_TYPE_IAMBIC_B;
  pConfig->iDotInput       = -KEYER_SIGNAL_INDEX_MORSE_KEY_CTS;
  pConfig->iDashInput      = -KEYER_SIGNAL_INDEX_MORSE_KEY_DSR;
  pConfig->iManualPTTInput = KEYER_SIGNAL_INDEX_NONE; // INPUT from the CW-keyer's side: Per default, 'automatic PTT control' !
  pConfig->iTestInput      = -KEYER_SIGNAL_INDEX_MORSE_KEY_DCD;
  pConfig->iKeySupply      = KEYER_SIGNAL_INDEX_MORSE_KEY_DTR;
  pConfig->iRadioSupply    = KEYER_SIGNAL_INDEX_NONE;
  pConfig->iSidetoneOnTXD  = KEYER_SIDETONE_TXD_480_HZ;
  pConfig->iRadioCWKeying  = KEYER_SIGNAL_INDEX_RADIO_KEYING_DTR;
  pConfig->iRadioPTTControl= KEYER_SIGNAL_INDEX_RADIO_KEYING_RTS; // <- default for DL4YHF's "simple keying adapter"
  pConfig->iTxDelayTime_ms = 25;
  pConfig->iTxHangTime_ms  = 500;
  pConfig->fDebouncePaddleInputs = FALSE; // OFF per default, because this degraded 'QRQ' performance
  pConfig->fKeyInputWatchdog     = TRUE;
  pConfig->fKeyViaShiftAndControl= FALSE; // ex: = ( pConfig->iComPortNumber_IN < 0 );
  pConfig->fDisableTx = FALSE; // allow real transmission (not just testing/practicing, with CW sidetone output only)
  // See also: Keyer_Main.cpp : TKeyerMainForm::RecallPrefinedSetting() .

#if( SWI_NUM_AUX_COM_PORTS > 0 )
  for(i=0; i<SWI_NUM_AUX_COM_PORTS; ++i )
   { // Again: The "COM port numbers" shall NOT be affected when selecting
     //        "Settings" .. "Restore Default" -> CwKeyer_SetDefaultConfig() !
     pAuxComCfg = &pConfig->sAuxCom[i];

     if( pAuxComCfg->iPortNumber <= 0 )  // no COM PORT selected for this instance : set "usage" to "none", etc
      {  pAuxComCfg->iPortUsage = RIGCTRL_PORT_USAGE_NONE;
      }
     if( pAuxComCfg->iBitsPerSecond <= 300 ) // serial baudrate hasn't been properly set yet -> make a guess:
      { switch( pAuxComCfg->iPortUsage )
         {
           case RIGCTRL_PORT_USAGE_NONE:
           case RIGCTRL_PORT_USAGE_WINKEYER_EMULATOR:
           case RIGCTRL_PORT_USAGE_WINKEYER_HOST:
              pAuxComCfg->iBitsPerSecond = 1200; // <- Winkeyer / Winkeyer USB : "1200 Baud Serial Rx/Tx Interface"
              break;
           case RIGCTRL_PORT_USAGE_TEXT_TERMINAL: // .. e.g. to control other 'station equipment' via ASCII commands:
              pAuxComCfg->iBitsPerSecond = 1200; // <- Winkeyer / Winkeyer USB : "1200 Baud Serial Rx/Tx Interface"
              break;
           case RIGCTRL_PORT_USAGE_ECHO_TEST: // send ANYTHING received on this port back AS FAST AS POSSIBLE
           case RIGCTRL_PORT_USAGE_TX_STRESS_TEST: // trying to flood TXD with data (256-step 'ramp up') ..
              pAuxComCfg->iBitsPerSecond = 115200; // <- minimize time for a "single byte reception" to 10 bits/115k2 = 87 microseconds (but ECHO RESPONSE TIME was much more)
              break;
           case RIGCTRL_PORT_USAGE_SERIAL_TUNNEL:
           case RIGCTRL_PORT_USAGE_VIRTUAL_RIG:
           default:
              pAuxComCfg->iBitsPerSecond = pConfig->iRadioControlBaudrate;
              break;
         }
      }
   }
#endif // SWI_NUM_AUX_COM_PORTS > 0 ?

} // end CwKeyer_SetDefaultConfig()


//----------------------------------------------------------------------------
void CwKeyer_InitTimingScope( T_KeyerTimingScope *pScope )
  // Called BEFORE loading the configuration from this application's config-file !
{ int i,j;
  T_RGBColor dwColour;
  memset( pScope, 0, sizeof(T_KeyerTimingScope) );
  pScope->cfg.iMillisecondsPerSample = 2;
  pScope->cfg.iVisibleChannels = (1<<TIMING_SCOPE_CHANNEL_DASH_INPUT)
                           | (1<<TIMING_SCOPE_CHANNEL_DOT_INPUT )
                           | (1<<TIMING_SCOPE_CHANNEL_CW_OUTPUT )
                           | (1<<TIMING_SCOPE_CHANNEL_FOUR      )
                           | (1<<TIMING_SCOPE_CHANNEL_AUDIO_INPUT)
                           | (1<<TIMING_SCOPE_CHANNEL_AUDIO_OUTPUT);
  pScope->cfg.iAudioSpectrumRefLevel_dB  = 0;
  pScope->cfg.iAudioSpectrumAmplRange_dB = 30;

  // Prepare colours for each scope channel (analog and digital) :
  for( i=0; i<(TIMING_SCOPE_N_DIGITAL_CHANNELS+TIMING_SCOPE_N_ANALOG_CHANNELS); ++i )
   {
     switch( i )
      { case TIMING_SCOPE_CHANNEL_DASH_INPUT/*0?*/ :
             dwColour.dw = 0x00FFFF; // YELLOW
             break;
        case TIMING_SCOPE_CHANNEL_DOT_INPUT /*1?*/ :
             dwColour.dw = 0x3FFF3F; // LIGHTGREEN
             break;
        case TIMING_SCOPE_CHANNEL_CW_OUTPUT /*2?*/ :
             dwColour.dw = 0x3F3FFF; // LIGHTRED
             break;
        case TIMING_SCOPE_CHANNEL_FOUR      /*3?*/ :
             dwColour.dw = 0xFF7F7F; // LIGHTBLUE for the user-selectable source
             break;
        case TIMING_SCOPE_CHANNEL_AUDIO_INPUT  /*4?*/:
             dwColour.dw = 0xFF0080;  // VIOLET (e.g. AUDIO INPUT)
             break;
        case TIMING_SCOPE_CHANNEL_AUDIO_OUTPUT /*5?*/:
             dwColour.dw = 0x007FFF;  // ORANGE (e.g. AUDIO OUTPUT)
             break;
        case TIMING_SCOPE_CHANNEL_AUDIO_CW_DEC /*6?*/:
             dwColour.dw = 0xC0C000;  // CYAN (e.g. for the
             // "filtered and rectified input" for the AUDIO CW DECODER,
             // or whatever appeared important for developing CwDSP.c .
             // See CwDSP_ReadFromPlotterFifo(),  called from
             // ScopeDisplay_AppendSamplesFromDSP(), which places those samples
             // in pScope->sSamplePoints[pScope->iSampleIndexFromDSP++].sDspPlotterSample .
             // See also: ScopeDisplay_DrawAudioSpectrumAndDecoderInfo() ...
             break;
        case TIMING_SCOPE_CHANNEL_AUDIO_CW_DEC+1 /*7?*/: dwColour.dw = 0x007F7F; break; // BROWN (future reserve)
        case TIMING_SCOPE_CHANNEL_AUDIO_CW_DEC+2 /*8?*/: dwColour.dw = 0xFF00FF; break; // MAGENTA
        default: dwColour.dw = 0x1F1F1F * (DWORD)i; break;
      }
     pScope->cfg.dwChannelColours[i].dw = dwColour.dw;
   }


  // Make sure FLOATING POINT NUMBERS in the scope's sample array are valid:
  for(i=0; i<TIMING_SCOPE_NUM_SAMPLE_POINTS; ++i )
   { for(j=0; j<TIMING_SCOPE_N_ANALOG_CHANNELS; ++j )
      { pScope->sSamplePoints[i].fltAnalog[j] = 0.0f;
      }
   }
} // end CwKeyer_InitTimingScope()

//----------------------------------------------------------------------------
void CwKeyer_CollectDataForTimingScope( T_KeyerTimingScope *pScope, WORD wCwChar )
  // Called roughly every two milliseconds from KeyerThread() to append another
  // multi-channel sample point for 'plotting' in the "CW Timing Scope" .
  // Depending on the scope's "trigger mode", a signal transition in any of the
  // input channels listed below may trigger a new sweep. In addtion, a sweep
  // may also be triggered via CwKeyer_TriggerTimingScope(), WITHOUT appending samples.
  //
  // [in,out] pScope  : Timing Scope instance data
  //            |--[in,out] : pScope->dblSampleCollectingTimestamp_s : controls emitting samples at <pScope->cfg.iMillisecondsPerSample>)
  //            |--[in,out] : pScope->iLatestSampleIndex (incremented here)
  //            |--[in,out] : pScope->iAnalogSourceSampleTimeAccu_ms[] (for 'analog' sources providing less than one sample in 2 milliseconds)
  //            '--[out]    : pScope->sSamplePoints[] (with DIGITAL and ANALOG channels, but without the audio spectrum from the DSP)
  // [in]     wCwChar : zero most of the time,
  //                    nonzero when the 'Elbug' emulator just detected
  //                    the completion of a complete character in Morse code.
  //                    Will be stored along with the samples in an array.
  // [in]     Keyer_dwCurrentSignalStates (with a few "boolean" in- and outputs)
  // [in]     CwKeyer_DSP : A few 'analog inputs' for the timing scope
  //                        may be tapped from the AUDIO DSP,
  //                        controlled by pScope->iAudioDisplay .
  //
{
  T_TimingScopeSample sSamplePoint; // <- "sample point" with integer data for up to 8 channels,
                     //    and a short array of floats for a few ANALOG channels.
  int iNewSampleIndex = pScope->iLatestSampleIndex + 1;
  T_CwDSP *pCwDSP = &CwKeyer_DSP;  // [in] source for ANALOG inputs, especially:
      //   pCwDSP->fltInputFifo[CW_DSP_PROCESS_FIFO_NSAMPLES],  pCwDSP->iInputFifoHead,
      //   pCwDSP->fltOutputFifo[CW_DSP_PROCESS_FIFO_NSAMPLES], pCwDSP->iOutputFifoHead.
  float *pfltAudioFifo, fltSample, fltMin, fltMax, fltSampleRate;
  int   nSamplesToCollect, i, iFifoHead, iFifoTail, iFifoSize, iAudioSample, nAudioSamples, n;
  BOOL  fTrigger;
  double diff, dblSampingInterval_s, dblTimestamp_s = DSW_ReadHighResTimestamp_s();
  static WORD s_wCwCharLatch = 0;
  T_CwNet *pCwNet = CwKeyer_Config.pCwNet; // optional instance for CW keying "via network"

  // Time to emit another sample (at an interval of ideally pScope->cfg.iMillisecondsPerSample) ?
  dblSampingInterval_s = 1e-3 * pScope->cfg.iMillisecondsPerSample;
  if( dblSampingInterval_s < 2e-3 )  // sampling interval must not be less than the keyer-thread-loop time
   {  dblSampingInterval_s = 2e-3;   // this also avoids div-by-zero further below
   }
  diff = dblTimestamp_s - pScope->dblSampleCollectingTimestamp_s;
  if( (diff < (-2.0* dblSampingInterval_s) ) // looks like the timestamp generator rolled over to ZERO -> "restart" ...
   || (diff > (5.0 * dblSampingInterval_s) ) ) // calling thread was asleep for too long -> also "restart" at the CURRENT timestamp
   { pScope->dblSampleCollectingTimestamp_s = dblTimestamp_s;
     nSamplesToCollect = 1;   // "collect" the FIRST sample in this call
   }
  else if( diff >= dblSampingInterval_s ) // just arrived at the right time for another sample ->
   { nSamplesToCollect = (int)(0.5 + diff / dblSampingInterval_s );
   }
  else // not time for the next sample yet -> return to "wait" for the right time
   { nSamplesToCollect = 0;
     // If there's NOTHING to collect (because it's not time for the NEXT sample)
     //    decoded Morse characters passed in as function argument 'wCwChar'
     //    must be stored in a latch, to be stored in pScope->sSamplePoints[]
     //    in a FUTURE call:
     if( wCwChar != 0 )
      { s_wCwCharLatch = wCwChar;
      }
   }

  while( (nSamplesToCollect--) > 0 )
   { // [in,out] pScope->dblSampleCollectingTimestamp_s : Incremented by dblSampingInterval_s
     //          at the end of this "sample-collecting loop"

     // Assemble the next "sample point" .. first, DIGITAL or INTEGER channels:
     for(i=0; i<TIMING_SCOPE_N_DIGITAL_CHANNELS; ++i)
      { sSamplePoint.iChannel[i] = 0;
      }
     if( Keyer_dwCurrentSignalStates & (1 << KEYER_SIGNAL_INDEX_DASH_INPUT) )
      { sSamplePoint.iChannel[TIMING_SCOPE_CHANNEL_DASH_INPUT] = 100/*percent*/;
      }
     if( Keyer_dwCurrentSignalStates & (1 << KEYER_SIGNAL_INDEX_DOT_INPUT ) )
      { sSamplePoint.iChannel[TIMING_SCOPE_CHANNEL_DOT_INPUT] = 100/*percent*/;
      }
     if( CwKeyer_GetDigitalInput( CwKeyer_Config.iRadioCWKeying ) )
      { sSamplePoint.iChannel[TIMING_SCOPE_CHANNEL_CW_OUTPUT] = 100/*percent*/;
      }
     else // not driving the ELECTRIC OUTPUT with the On/Off-keying signal,
      { // but the scope shall still indicate the output from Mr. Elbug
        // (in TIMING_SCOPE_CHANNEL_CW_OUTPUT), but with a lower amplitude
        // so the operator sees "this signal is not really being transmitted":
        if( CwKeyer_fLocalMorseOutput )
         { sSamplePoint.iChannel[TIMING_SCOPE_CHANNEL_CW_OUTPUT] = 25/*percent*/;
         }
      }

     // The FOURTH scope channel may be connected to a SELECTABLE source.
     // Since 2024-05-08, "channel 4" isn't necessarily an ON/OFF value but integer.
     switch( pScope->cfg.iChannel4Source )
      { case TIMING_SCOPE_CHANNEL_SOURCE_PTT_OUTPUT :
           if( CwKeyer_GetDigitalInput( CwKeyer_Config.iRadioPTTControl ) )
            { sSamplePoint.iChannel[TIMING_SCOPE_CHANNEL_FOUR] = 100;
            }
           break;
        case TIMING_SCOPE_CHANNEL_SOURCE_TEST_INPUT :
           if( CwKeyer_GetDigitalInput( CwKeyer_Config.iTestInput ) )
            { sSamplePoint.iChannel[TIMING_SCOPE_CHANNEL_FOUR] = 100;
            }
           break;
        case TIMING_SCOPE_CHANNEL_SOURCE_NETWORK_TROUBLE :
           if( pCwNet != NULL )
            { if( pCwNet->fSimulateBadConnNow ) // "deliberate network trouble" at the moment ?
               { sSamplePoint.iChannel[TIMING_SCOPE_CHANNEL_FOUR] = 100;
               }
            }
           break;
        case TIMING_SCOPE_CHANNEL_SOURCE_KEYING_FIFO_USAGE:
           if( pCwNet != NULL )
            { n = pCwNet->MorseRxFifo.iHeadIndex - pCwNet->MorseRxFifo.iTailIndex;
              if( n<0 )
               {  n += CW_KEYING_FIFO_SIZE;
               }
              sSamplePoint.iChannel[TIMING_SCOPE_CHANNEL_FOUR] = (100 * n) / CW_KEYING_FIFO_SIZE; // scale into percent
            }
           break;
        case TIMING_SCOPE_CHANNEL_SOURCE_ELBUG_KEYER_FLAGS:
           sSamplePoint.iChannel[TIMING_SCOPE_CHANNEL_FOUR] =
             ( (CwKeyer_Elbug.bFlags & (ELBUG_FLAG_STORED_DOT | ELBUG_FLAG_STORED_DASH))
                   * 100/*percent*/) / (ELBUG_FLAG_STORED_DOT | ELBUG_FLAG_STORED_DASH);
           break;

        default:
           break;
      } // end switch( pScope->cfg.iChannel4Source )

     // The short-time audio spectrum is not available here yet (in CwKeyer_CollectDataForTimingScope).
     // Invalidate the "DSP plotter sample", which includes the spectrum.
     // ScopeDisplay_AppendSamplesFromDSP() will merge the output from the DSP's "plotting FIFO"
     // to the other sample point members in pScope->sSamplePoints[] .
     sSamplePoint.sDspPlotterSample.dblTimestamp_s = 0.0;

     // Some ANALOG inputs are tapped from the DSP's sample buffers.
     // Those buffers are CIRCULAR FIFOs. The scope has its own FIFO TAIL INDICES
     // for each of those buffers. Because the DSP worker thread may fills those FIFOs
     // in larger bursts (dictated e.g. by the audio sampling rate and blocksize),
     // only consume as many samples as the DSP thread would produce in TWO milliseconds:
     for( i=0; i<TIMING_SCOPE_N_ANALOG_CHANNELS; ++i )
      { switch( i + TIMING_SCOPE_FIRST_ANALOG_CHANNEL )
         { case TIMING_SCOPE_CHANNEL_AUDIO_INPUT : // get the next 2ms from the AUDIO INPUT ...
           default:           // source: Remote_CW_Keyer/CwDSP.c : 
              pfltAudioFifo = pCwDSP->sInputFifo.fltQueue;
              iFifoHead   = pCwDSP->sInputFifo.head.index;
              iFifoSize   = CWDSP_PROCESS_FIFO_NSAMPLES;
              fltSampleRate = CWDSP_INPUT_FIFO_SAMPLING_RATE; // <- usually 8000 samples/second
              break;
           case TIMING_SCOPE_CHANNEL_AUDIO_OUTPUT: // get the next 2ms from the AUDIO OUTPUT
              pfltAudioFifo = pCwDSP->sOutputFifo.fltQueue;
              iFifoHead   = pCwDSP->sOutputFifo.head.index;
              iFifoSize   = CWDSP_PROCESS_FIFO_NSAMPLES;
              fltSampleRate = CWDSP_OUTPUT_FIFO_SAMPLING_RATE;
              break;
#         if(0) // replaced by T_CwDSP_PlotterSample / CwDSP_ReadFromPlotterFifo(), used by the GUI for signal-latency-compensated plotting of the audio spectrum, along with the 'rectified' audio keying signal
           case TIMING_SCOPE_CHANNEL_AUDIO_CW_DEC: // get the next 2ms from the "filtered and rectified input" for the AUDIO CW DECODER, treated like an analog input
              pfltAudioFifo = pCwDSP->AudioCwDecoder[0].sKeyingSignalFifo.fltQueue;
              iFifoHead   = pCwDSP->AudioCwDecoder[0].sKeyingSignalFifo.iHead;
              iFifoSize   = CWDSP_PROCESS_FIFO_NSAMPLES;
              fltSampleRate = pCwDSP->fltAudioSpectrumFrameRate; // e.g. 1 / 8ms = 125 Hz
               // Note: The audio FFT window length may be 16 ms, but new audio
               //       spectra are calculated every 8 ms due to the 50 % FFT overlap.
               // Limit the LATENCY of samples plotted here to n = 10 samples a 8 milliseconds ?
              n = 10;  // <- maximum acceptable "latency" measured in SAMPLES
              iFifoTail = pScope->iAnalogSourceFifoTail[i];
              nAudioSamples = iFifoHead - iFifoTail; // how many samples are REALLY in the FIFO ?
              if( nAudioSamples < 0 )
               {  nAudioSamples += iFifoSize;
               }
              if( nAudioSamples > n )  // adjust the FIFO tail for this channel to avoid samples getting "too old"
               { n = iFifoHead - n;
                 if( n < 0 )
                  {  n += iFifoSize;
                  }
                 pScope->iAnalogSourceFifoTail[i] = n;
               }
              break; // end case TIMING_SCOPE_CHANNEL_AUDIO_CW_DEC
#          endif // (old stuff, used before the TIMESTAMP-BASED retrieving of plot-data, INCLUDING THE AUDIO SPECTRUM, via CwDSP_ReadFromPlotterFifo )
         } // end switch( iChannel )
        iFifoHead %= iFifoSize;
        iFifoTail = pScope->iAnalogSourceFifoTail[i];
        iFifoTail %= iFifoSize;

        nAudioSamples = iFifoHead - iFifoTail;
        if( nAudioSamples < 0 )
         {  nAudioSamples += iFifoSize;  // -> number of samples CURRENTLY AVAILABLE
         }
        // Note: The element at pfltAudioFifo[iFifoHead] does NOT contain a valid sample yet,
        //       thus if iFifoHead == iFifoTail,  NO SAMPLES ARE AVAILABLE YET !
        //
        if( fltSampleRate < 1.0f )
         {  fltSampleRate = 1.0f;  // avoid div-by-zero further below
         }
        // Ideally, we'd consume <dblSampingInterval_s> seconds of samples from the DSP's FIFO.
        //        If there's backlash building up, consume a few samples more...
        n = (int)(0.5f + fltSampleRate * dblSampingInterval_s);
         // '-> number of audio samples per scope sampling interval,
         //   e.g. 8000 Hz from the source * 2 ms for the scope -> 8 * 2 = 16 samples to process per call
        if( nAudioSamples > (iFifoSize/2) )  // Source-FIFO more than half full ->
         { // ex: nAudioSamples = (iSampleRate+443)/444; // consume SLIGHTLY MORE than 2ms of samples
           nAudioSamples = n+1;
         }
        else if( n == 0 ) // SOURCE provides less than one sample per pScope->iMillisecondsPerSample ?
         { // The original code further failed miserably for the "rectified CW signal"
           //     from the short-time audio FFT, with iSampleRate = 125 Hz.
           // If the source provides less than one sample per in 2 milliseconds,
           // then use this principle :
           //  * If iAnalogSourceSampleTimeAccu_ms[<analog channel>] is <= 0,
           //         it's time to emit a NEW sample from pfltAudioFifo[iFifoTail++] .
           //  * When emitting a NEW sample, add the SOURCE'S sampling interval (in milliseconds)
           //         to iAnalogSourceSampleTimeAccu_ms[<analog channel>]
           //  * Otherwise (not emitting a NEW sample but duplicating the previous one),
           //         subtract the SCOPE'S sampling interval (iMillisecondsPerSample)
           //         from iAnalogSourceSampleTimeAccu_ms[<analog channel>]
           if( pScope->iAnalogSourceSampleTimeAccu_ms[i] <= 0 )
            { // time to pull ONE new sample from pfltAudioFifo[ iFifoTail++ ] further below:
              nAudioSamples = 1;  // <- a breakpoint HERE often made C++Builder itself crash ! ! !
              pScope->iAnalogSourceSampleTimeAccu_ms[i] += (int)(0.5f + 1000.0f / fltSampleRate); // e.g. 1000ms / 125 [Hz] = 8 ms for the AUDIO-SPECTRUM-BASED "keying signal"
            }
           else
            { nAudioSamples = 0;  // let the code way further below DUPLICATE the last sample
            }
           pScope->iAnalogSourceSampleTimeAccu_ms[i] -= pScope->cfg.iMillisecondsPerSample; // e.g. SUBTRACT 2 ms from the "time accu" for each DUPLICATED sample
         }
        else if( nAudioSamples >= (2*n) ) // between 4 and 10 ms of samples available -> consume and process exactly 2 ms of samples
         { nAudioSamples = n;
         }
        else // less than 4 ms of samples available -> ramp down to avoid running COMPLETELY out of samples
         { nAudioSamples = n/2/*=1ms*/ + nAudioSamples/16;
           // The effect can be checked with SWI_AUDIO_INPUT_DUMMY=1 (slow sawtooth, displayed WITHOUT "staircase effect")
         }

        if( nAudioSamples > 0 )
         { for( iAudioSample=0; iAudioSample<nAudioSamples; ++iAudioSample )
            { fltSample = pfltAudioFifo[ iFifoTail++ ];
              if( iFifoTail >= iFifoSize )
               {  iFifoTail =  0; // circular index wrap (iFifoSize is not necessarily a power of two here)
               }
              if( fltSample > 1.0f ) // gotcha..  clipping (no problem for a "float", but for PLOTTING later)
               {  fltSample = 1.0f;
               }
              if( fltSample < -1.0f )
               {  fltSample = -1.0f;
               }
              if( iAudioSample == 0 )
               { fltMin = fltMax = fltSample;
               }
              else
               { if( fltSample < fltMin )
                  {  fltMin = fltSample;
                  }
                 if( fltSample > fltMax )
                  {  fltMax = fltSample;
                  }
               }
            }
           // Write back the INCREMENTED FIFO tail index for the next call :
           pScope->iAnalogSourceFifoTail[i] = iFifoTail;

           // A horizontal pixel position on the 'timing scope' represents
           // an awful lot of AUDIO SAMPLES - thus the min/max detection above.
           // To avoid having to store BOTH min and may for the scope,
           // alternatively store fltMin in every EVEN iNewSampleIndex,
           //                 and fltMax in every ODD  iNewSampleIndex.
           // When converting this into a polyline for plotting later,
           // a noisy audio signal will appear like a broad "band" instead of
           // appearing like a 'DC signal' .
           if( iNewSampleIndex & 1 ) // ODD array index into pScope->sSamplePoints[] ?
            { sSamplePoint.fltAnalog[i] = fltMax;
            }
           else
            { sSamplePoint.fltAnalog[i] = fltMin;
            }
         } // end if( nAudioSamples > 0 )
        else // Completely ran out of "analog input" for this scope channel,
         { // possibly because the SOURCE receives fresh input very infrequently
           // -> cheat by simply repeating the PREVIOUS sample value
           //    for THIS analog channel .
           sSamplePoint.fltAnalog[i] = pScope->sSamplePoints[pScope->iLatestSampleIndex].fltAnalog[i];
         } // end if ( nAudioSamples > 0 ) ?

      } // end for < all "audio" channels on the timing scope >

     if( iNewSampleIndex >= TIMING_SCOPE_NUM_SAMPLE_POINTS )
      { // Sample array "completely filled" -> waiting for a trigger event...
        fTrigger = FALSE;
        // Do NOT overwrite the current content with a new 'sweep'
        // if nothing (no transition) happens on any of the plotted signals.
        // This way, if the operator sees "something unusual" on the scope
        // and immediately stops operating the paddle, the 'timing scope'
        // display will pause (like a digital scope in "Normal", not "Auto" mode).
        for( i=0; i<TIMING_SCOPE_N_DIGITAL_CHANNELS; ++i)
         { if( sSamplePoint.iChannel[i] != pScope->sPreTriggerPoints[TIMING_SCOPE_PRETRIGGER_POINTS-1].iChannel[i] )
            {  fTrigger = TRUE;
            }
         }
        if( pScope->cfg.iTriggerOptions & TIMING_SCOPE_TRIGGER_FREE_RUN )
         {  fTrigger = TRUE;
         }
        if( fTrigger )
         { // Trigger new sweep : Move the "pre-trigger" samples(*) into the displayed pScope->sSamplePoints[]...
           pScope->dblTimestampAtSampleIndexZero = pScope->dblTimestampAtPretriggerIndexZero;
           for( iNewSampleIndex=0; iNewSampleIndex<TIMING_SCOPE_PRETRIGGER_POINTS; ++iNewSampleIndex )
            { pScope->sSamplePoints[iNewSampleIndex] = pScope->sPreTriggerPoints[iNewSampleIndex];
            }
           iNewSampleIndex = TIMING_SCOPE_PRETRIGGER_POINTS;
             // '--> stored in pScope->iLatestSampleIndex further below,
             //       after appending the *new* sample (sSamplePoint) .
           // (*) These "pre-trigger" samples do NOT contain data from the AUDIO DSP (CwDSP.c).
           //     Members in pScope->sSamplePoints[] from the audio CW DSP will be
           //     merged in Keyer_GUI.cpp :: ScopeDisplay_AppendSamplesFromDSP(),
           //     using pScope->iSampleIndexFromDSP, not pScope->iLatestSampleIndex :
           pScope->iSampleIndexFromDSP = 0; // here: Cleared in CwKeyer_CollectDataForTimingScope(), when triggered for a new sweep
         }
      }

     // Also store the 'Morse code pattern' in the timing scope's array of sample points ?
     if( (s_wCwCharLatch != 0)  && ( ! (s_wCwCharLatch & CW_CHR_SPACE)) )
      { sSamplePoint.wCwChar = s_wCwCharLatch;
        s_wCwCharLatch = 0;
      }
     else if( ! (wCwChar & CW_CHR_SPACE) )
      { sSamplePoint.wCwChar = wCwChar;
      }
     else  // no extra marking for 'inter-word spaces' in the timing scope...
      { sSamplePoint.wCwChar = 0;
      }

     // Append the new sample point to the buffer (unless "waiting for trigger"):
     if( iNewSampleIndex < TIMING_SCOPE_NUM_SAMPLE_POINTS )
      { pScope->sSamplePoints[iNewSampleIndex] = sSamplePoint;
        if( iNewSampleIndex == 0 )
         { pScope->dblTimestampAtSampleIndexZero = dblTimestamp_s;
         }
        pScope->iLatestSampleIndex = iNewSampleIndex;  // here: updated in CwKeyer_CollectDataForTimingScope()
        ++pScope->iUpdateCount;
      }
     // Update the pre-trigger history :
     memmove( pScope->sPreTriggerPoints/*dst*/,
              pScope->sPreTriggerPoints+1/*src*/,
              (TIMING_SCOPE_PRETRIGGER_POINTS-1) * sizeof(T_TimingScopeSample) );
     pScope->sPreTriggerPoints[TIMING_SCOPE_PRETRIGGER_POINTS-1] = sSamplePoint;
     pScope->dblTimestampAtPretriggerIndexZero = dblTimestamp_s
       - ((double)TIMING_SCOPE_PRETRIGGER_POINTS * 1e-3 * (double)pScope->cfg.iMillisecondsPerSample);


     pScope->dblSampleCollectingTimestamp_s += dblSampingInterval_s;
  } // end while( (nSamplesToCollect--) > 0 )
} // end CwKeyer_CollectDataForTimingScope()

//----------------------------------------------------------------------------
void CwKeyer_TriggerTimingScope( T_KeyerTimingScope *pScope )
  // Triggers a new sweep for the timing scope, WITHOUT "collecting data",
  //                                and WITHOUT the "pre-trigger samples".
  // Added 2024-05-20 to examine the timing / buffering with a MANUALLY
  // selected 'Network Latency'. When set to e.g. 250 ms, those 250 ms should
  // appear between the left edge of the scope trace, and the first dot or dash.
  //
  // [in,out] pScope  : Timing Scope instance data, including "pre-trigger"
  //                    and "post-trigger" samples.
  //
{ if( pScope->iLatestSampleIndex >= (TIMING_SCOPE_NUM_SAMPLE_POINTS-1) )
   { // Sample array "completely filled" -> indeed waiting for a trigger event...
     //  -> trigger new sweep, but in this case WITHOUT the pre-trigger points,
     //     because the event that "triggered" the scope isn't visible at all.
     pScope->iLatestSampleIndex = 0;
     ++pScope->iUpdateCount;
   } // end if < really "waiting for a trigger" ? >
} // end CwKeyer_TriggerTimingScope()


//----------------------------------------------------------------------------
void CwKeyer_WriteToDecoderFifo( T_KeyerDecoderFifo *pFifo, WORD wCwChar )
{
  int iHeadIndex = pFifo->iFifoHead % KEYER_DECODER_FIFO_SIZE; // <- safety first..
  // Keep it simple.. don't care if the READER (GUI) can keep up the pace;
  //      just append whatever Mr. Elbug has 'decoded' (character or prosign)
  //      to the FIFO. Since 2024-05-09, this function is also called from
  //      a 'straight keying' decoder; last not least to check what has been
  //      sent on the SERVER SIDE, where the keying signal arrives like a
  //      'straight key' signal (no separate inputs for DASH and DOT).
  //      Thus, the CALLERS may now be ..
  // (a) KeyerThread(), after Elbug_Handler() returned with a completed character
  // (b) KeyerThread(),
  pFifo->wCwChar[ iHeadIndex++ ] = wCwChar;
  // Writing the INCREMENTED head index back to the struct makes the new entry
  // available for another thread (e.g. the GUI thread, occasionally polling the head index):
  pFifo->iFifoHead = iHeadIndex % KEYER_DECODER_FIFO_SIZE;
  //
  // Note: This "FIFO writer" doesn't care about the reader's TAIL INDEX,
  //       last not least to keep it simple, and because there may be
  //       MULTIPLE READERS (and each of them has its own tail index).
  //
} // end CwKeyer_WriteToDecoderFifo()


//----------------------------------------------------------------------------
WORD CwKeyer_ReadFromDecoderFifo( T_KeyerDecoderFifo *pFifo, int *piTailIndex )
  // Returns ZERO if nothing is currently available for THIS reader,
  // otherwise an old-school, zero-terminated C string with a single character
  // or a short string with a Morse code 'prosign' (see format in Elbug.h) .
{
  int iTailIndex;
  WORD wCwChar = 0;
  if( (pFifo!=NULL) && (piTailIndex!=NULL) )
   { iTailIndex = *piTailIndex % KEYER_DECODER_FIFO_SIZE;
     if( pFifo->iFifoHead != iTailIndex ) // "FIFO not empty" for THIS reader ->
      { wCwChar = pFifo->wCwChar[ iTailIndex++ ];
        *piTailIndex = iTailIndex % KEYER_DECODER_FIFO_SIZE;
      }
   }
  return wCwChar;
} // end CwKeyer_ReadFromDecoderFifo()


//----------------------------------------------------------------------------
BOOL CwKeyer_Start( void )  // Starts the CW keyer (but not the DSP) .
  // [in] CwKeyer_Config .
  // Returns TRUE when successful,
  //      or FALSE when something went wrong. In that case,
  //               CwKeyer_sz255LastError[] contains the 'reason' for failing.
{
  int i32BitsPerSecond, iDtrControl, iRtsControl;
  char sz80PortUsedFor[88];

  if( Keyer_hThread!=NULL ) // oops .. there is still an OLD "thread handle" ?!
   { // Maybe the previous thread instance has terminated itself in between.
     // In that case, we can safely "CloseHandle()" on it, to clean up,
     // before creating a new one a few lines further below:
     if( CwKeyer_iThreadStatus==KEYER_THREAD_STATUS_TERMINATED )
      { CloseHandle( Keyer_hThread );
        Keyer_hThread = NULL;
      }
   }

  // Initialize the 'contact debouncers' for dash/dot/strait inputs:
  Debouncer_Init( &CwKeyer_Elbug.dashDebouncer, 3/*iMinStableIntervals*/ );
  Debouncer_Init( &CwKeyer_Elbug.dotDebouncer,  3/*iMinStableIntervals*/ );
    // ,----------------------------------------'
    // '-- 3 calls of Debounce() * 2 milliseconds per thread loop
    //      -> 3 * 2ms additional latency caused by the 'debouncer' stage.
    //  6 milliseconds is "a half dot" at 100 WPM ( 1200 ms / 100 WPM = 12 ms)
    //   -> Tolerable, even at very high speeds. If not, the debouncer may be
    //      turned off through the remote keyer's main menu / "Settings".

  // Initialize the 'straight key decoder' (used on the client side if a STRAIGHT KEY
  //   is connected instead of an elbug, or on the server side to show what is
  //   being sent to the remotely controlled radio). The default dot-time
  //   is the same as configured for the ELBUG, but the decoder itself
  //   may adapt itself for a not-so-ideal timing.
  StraightKeyDecoder_Init( &CwKeyer_StraightKeyDecoder, CwKeyer_Elbug.cfg.iDotTime_ms );


  // Initialize the Decoder-Output-FIFO 'between worker thread and GUI' :
  memset( &CwKeyer_DecoderFifo, 0, sizeof(CwKeyer_DecoderFifo) );

  // Initialize lock-free, thread-safe FIFO between KeyerThread() and RigControl.c :
  CFIFO_Init( &Keyer_RadioControlPortRxFifo.fifo, KEYER_SERIAL_PORT_FIFO_SIZE, sizeof(BYTE), 0.0/*dblSecondsPerSample*/ );
  CFIFO_Init( &Keyer_RadioControlPortTxFifo.fifo, KEYER_SERIAL_PORT_FIFO_SIZE, sizeof(BYTE), 0.0/*dblSecondsPerSample*/ );

  CwKeyer_Gen.iState = CW_GEN_OFF;

  // Depending on the assignment of 'functions' to 'pins'
  //  and their signal polarities, already CLEAR or SET various bits in
  //  Keyer_dwCurrentSignalStates . We use this in OpenAndConfigureSerialPort()
  //  to set the two available DIGITAL OUTPUTS (in each COM port adapter)
  //  to the configurable *passive* state. Without this, an Icom radio
  //  produced a short 'dit' (sidetone) before KeyerThread() put things right,
  //  or the "positive voltage supply" bypass capacitor (tapped from e.g. DTR)
  //  wasn't sufficiently charged (which, to the Software-Elbug, looked like
  //  the paddle contact pulled to GROUND = active). Again, THIS fixed it:
  CwKeyer_SetDigitalOutput( CwKeyer_Config.iKeySupply, TRUE/* "supply voltage" ON */ );
  CwKeyer_SetDigitalOutput( CwKeyer_Config.iRadioSupply, TRUE/* "supply voltage" ON */ );
  CwKeyer_SetDigitalOutput( CwKeyer_Config.iRadioCWKeying,   FALSE/*passive*/ );
  CwKeyer_SetDigitalOutput( CwKeyer_Config.iRadioPTTControl, FALSE/*passive*/ );
      //  '--> depending on SIGNAL POLARITY, clears OR SETS bits in Keyer_dwCurrentSignalStates .


  // Initial states for DTR and RTS when opening the "Paddle Port" :
  iDtrControl = ( Keyer_dwCurrentSignalStates & (1UL << KEYER_SIGNAL_INDEX_MORSE_KEY_DTR) )
              ? DTR_CONTROL_ENABLE : DTR_CONTROL_DISABLE;
  iRtsControl = ( Keyer_dwCurrentSignalStates & (1UL << KEYER_SIGNAL_INDEX_MORSE_KEY_RTS) )
              ? RTS_CONTROL_ENABLE : RTS_CONTROL_DISABLE;

  // TRY TO (!) open the two serial ports (one for the Morse key,
  //                                       and one for the keyed radio):
  if( CwKeyer_Config.iComPortNumber_IN >= 0 )
   {
     HERE_I_AM__KEYER();
     if( ! OpenAndConfigureSerialPort(  // <- 2025-08-22: Crashed somewhere HERE
          CwKeyer_Config.iComPortNumber_IN, 4800/*-> ca 480 bytes/second*/,
          SP_OPTION_USE_SMALL_TX_BUFFER, // <- to have the SIDETONE stop quickly
          iDtrControl, iRtsControl, &Keyer_hComPort_IN ) )
      { HERE_I_AM__KEYER();
        // If OpenAndConfigureSerialPort fails, it sets CwKeyer_sz255LastError, so:
        ShowError( ERROR_CLASS_ERROR | SHOW_ERROR_IN_RUN_LOG | SHOW_ERROR_TIMESTAMP,
           "Failed to open the 'Morse key' port: %s", CwKeyer_sz255LastError );
        return FALSE;
      }
     else
      { HERE_I_AM__KEYER();
        ShowError( ERROR_CLASS_INFO | SHOW_ERROR_IN_RUN_LOG | SHOW_ERROR_TIMESTAMP,
           "Opened COM%d, used as the 'Morse key' port.",
           (int)CwKeyer_Config.iComPortNumber_IN );
      }
   }


  // If the 'Radio Keying / Radio Control Port' is valid, try to open it, unless...
  if( CwKeyer_Config.iRadioKeyingAndControlPort >= 0 )
   { // .. unless the 'Radio Keying..' port is THE SAME PHYSICAL PORT as the "Morse key" port:
     if( CwKeyer_Config.iRadioKeyingAndControlPort == CwKeyer_Config.iComPortNumber_IN )
      { ShowError( ERROR_CLASS_INFO | SHOW_ERROR_IN_RUN_LOG | SHOW_ERROR_TIMESTAMP,
           "Both 'Morse key' and 'Rig control/keying' on same port (COM%d) - this may cause problems !",
           (int)CwKeyer_Config.iRadioKeyingAndControlPort );
        // This WILL be a problem - at least under Windows, it's impossible to open the same port twice.
        // And accessing the same port from different threads, using the "COM port API",
        // is asking for trouble. Also, the KEYER THREAD cannot afford to wait
        // for the RADIO CONTROL / RIG CONTROL / HAMLIB SERVER THREAD !
      }
     else
      { // Similar as above for the 'Morse key' port, here for the RIG KEYING / RIG CONTROL port ...
        // Initial states for DTR and RTS when opening the "Radio KEYING Port" :
        iDtrControl = ( Keyer_dwCurrentSignalStates & (1UL << KEYER_SIGNAL_INDEX_RADIO_KEYING_DTR) )
              ? DTR_CONTROL_ENABLE : DTR_CONTROL_DISABLE;
        iRtsControl = ( Keyer_dwCurrentSignalStates & (1UL << KEYER_SIGNAL_INDEX_RADIO_KEYING_RTS) )
              ? RTS_CONTROL_ENABLE : RTS_CONTROL_DISABLE;
        // 2025-10-04: Suspected a problem with the above. From Brian, N3OC:
        // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
        // > I forgot to say I am using a K3, so all interfaces are legacy.
        // > DTR/CW and RTS/PTT on physical com port at server end.
        // > Both paddle and N1MM are working on client end at first run,
        // > and N1MM initializes it OK on first run.
        // > Only bug I see now is on Server end. When I start the program,
        // > RTS/PTT is stuck on until I key something from Client once,
        // > then it clears.
        // > Oddly enough, if I start the Server as Run as Administrator,
        // > it does not do this and works fine.
        // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
        HERE_I_AM__KEYER();
        if( ! OpenAndConfigureSerialPort(
          CwKeyer_Config.iRadioKeyingAndControlPort,
          CwKeyer_Config.iRadioControlBaudrate/*Bits/Second*/,
          SP_OPTION_NONE, iDtrControl, iRtsControl, &Keyer_hComPortRadioKeyingAndControl ) )
         { HERE_I_AM__KEYER();
           CwKeyer_Stop(); // here: called to close the already opened "Morse key" port
           return FALSE;
         }
        else
         { HERE_I_AM__KEYER();
           // Is this port really used for BOTH, "radio control" AND "radio keying" ?
           // The author stumbled over this: "keying port" successfully opened,
           //  but NO RADIO CONTROL, no frequency display, not spectrum data,
           //  because on the "Rig Control" tab, the "Method / Protocol"
           //  was set to "No rig control, only CW keying (or client mode)".
           //  Modified as follows, to give a better clue on the 'Debug' tab:
           sz80PortUsedFor[0] = '\0'; // further below: .. used as the 'Radio keying' port (without CAT/CI-V control)
           switch( CwKeyer_Config.iRadioControlProtocol )
            { case RIGCTRL_PROTOCOL_NONE : // < CwKeyer_Config.iRadioKeyingAndControlPort > opened but not for RADIO CONTROL..
                 break;  // leave pszSomething as already set ("Radio keying")
              case RIGCTRL_PROTOCOL_ICOM_CI_V : // "Icom CI-V, using the 'COM port to key the radio'"
                 strcpy( sz80PortUsedFor, "Icom CI-V Radio control" );
                 break;
              case RIGCTRL_PROTOCOL_KENWOOD : // "Kenwood PC Control" (ASCII with semicolon delimiter), using the 'COM port to key the radio'"
                 strcpy( sz80PortUsedFor, "Kenwood 'PC' control (ASCII)" );
                 break;
              case RIGCTRL_PROTOCOL_YAESU_5_BYTE : // "Yaesu 5-Byte CAT, using the 'COM port to key the radio'"
                 strcpy( sz80PortUsedFor, "Yaesu '5-Byte' CAT control" );
                 break;
              case RIGCTRL_PROTOCOL_HAMLIB_RIGCTLD:
                 strcpy( sz80PortUsedFor, "Hamlib rig control" );
                 break;
              default:
                 break;
            }
           if( sz80PortUsedFor[0] != '\0' )
            { strcat( sz80PortUsedFor, " and " );
            }
           strcat( sz80PortUsedFor, "CW Keying" );
           ShowError( ERROR_CLASS_INFO | SHOW_ERROR_IN_RUN_LOG | SHOW_ERROR_TIMESTAMP,
              "Opened COM%d, used for %s.", (int)CwKeyer_Config.iRadioKeyingAndControlPort, sz80PortUsedFor );
         }
      } // end if < CwKeyer_Config.iRadioKeyingAndControlPort != CwKeyer_Config.iComPortNumber_IN >
   } // end if( CwKeyer_Config.iRadioKeyingAndControlPort > 0 )

#if( SWI_USE_MIDI )
  // If not done yet, open a MIDI device for output:
  CwKeyer_fMidiToneOn = FALSE;
  if( CwKeyer_hMidiDevice == INVALID_HANDLE_VALUE )
   { if( midiOutOpen(
           &CwKeyer_hMidiDevice,  // [out] LPHMIDIOUT phmo .. "This location is filled with a handle identifying the opened MIDI output device."
           CwKeyer_iMidiDeviceId, // [in] UINT       uDeviceID
           NULL,                  // [in] DWORD_PTR  dwCallback
           NULL,                  // [in] DWORD_PTR  dwInstance : "User instance data passed to the callback"
           CALLBACK_NULL          // [in] DWORD      fdwOpen : "There is no callback mechanism. This value is the default setting."
                    ) != MMSYSERR_NOERROR )
      { // Out of luck, no MIDI ..
        CwKeyer_hMidiDevice = INVALID_HANDLE_VALUE; // make sure midiOutOpen() didn't leave garbage here
      }
     else // success .. try to select a suitable "musical instrument" for the sidetone:
      { T_ShortMidiMsg msg;
        msg.dwData = 0;      // clean the entire 4-byte field
        msg.bData[0] = 0xC0; // "status byte" : "Program Change", aka "Change Instrument"
        // ,--------------'
        // '--> lower 4 bits are the synthesizer's MIDI channel, #0..15
        msg.bData[1] = (BYTE)CwKeyer_iMidiInstrument; // "instrument", e.g. 0x51 = "Lead 2  (sawtooth)"
        midiOutShortMsg( CwKeyer_hMidiDevice, msg.dwData );
        // Try to modify the SUSTAIN LEVEL, using a "MIDI CC" (Control Change):
        // > In MIDI, adjustable parameters for each of the 16 possible MIDI channels
        // > may be set with the Control Change (CC) message, which has a
        // > Control Number parameter and a Control Value parameter
        // > (expressed in a range from 0 to 127). GM also specifies
        // > which operations should be performed by multiple Control Numbers.
        msg.bData[0] = 0xB0; // "status byte" : "Control Change Message"
        // ,--------------'
        // '--> lower 4 bits are the synthesizer's MIDI channel, #0..15
        msg.bData[1] = 79; // "CC" number .. Wikipedia on "General Midi":
        //    #1 = Modulation wheel, .., #79 = 0x4F = Sustain Level.
        // http://personal.kent.edu/~sbirch/Music_Production/MP-II/MIDI/midi_control_change_messages.htm :
        //  0x01 = Modulation wheel, 0x46 .. 0x4F = "Sound Controllers 1-10",
        //                     where "Controller 3" = "Release Time": 0=shortest,
        //                           "Controller 4" = "Attack Time" : 0=shortest,
        msg.bData[2] = 127; // maximum sustain level ?
        midiOutShortMsg( CwKeyer_hMidiDevice, msg.dwData );

      } // end if <  midiOutOpen() successful > ?
   } // end if( CwKeyer_hMidiDevice == INVALID_HANDLE_VALUE )
#endif // SWI_USE_MIDI ?



  //
  // Create the CW keyer's worker thread (using WIN32 API functions only,
  //                           no "dot net" framework, no VCL, no Qt, etc)
  //
  // From "Microsoft Win32 Programmer's Reference" ("Creating Threads").
  // > The CreateThread function creates a new thread for a process.
  // > The creating thread must specify the starting address of the code
  // > that the new thread is to execute. Typically, the starting address
  // > is the name of a function defined in the program code.
  // > This function takes a single parameter and returns a DWORD value.
  // > A process can have multiple threads simultaneously
  // > executing the same function.
  // > The following example demonstrates how to create a new thread
  // > that executes the locally defined function, ThreadFunc.
  if( Keyer_hThread==NULL )
   {  CwKeyer_iThreadStatus = KEYER_THREAD_STATUS_LAUNCHED;
      HERE_I_AM__KEYER();
      Keyer_hThread = CreateThread(
         NULL,    // LPSECURITY_ATTRIBUTES lpThreadAttributes = pointer to thread security attributes
         65536,   // DWORD dwStackSize  = initial thread stack size, in bytes
         KeyerThread, // LPTHREAD_START_ROUTINE lpStartAddress = pointer to thread function
         NULL,    // LPVOID lpParameter = argument for new thread
         0,       // DWORD dwCreationFlags = creation flags
                  // zero -> the thread runs immediately after creation
         &Keyer_dwThreadId // LPDWORD lpThreadId = pointer to returned thread id
       );
      // > The thread object remains in the system until the thread has terminated
      // > and all handles to it have been closed through a call to CloseHandle.
      HERE_I_AM__KEYER();
   }
  if( Keyer_hThread==NULL ) // Check the return value for success.
   {
     strcpy( CwKeyer_sz255LastError, "CreateThread failed." );
     CwKeyer_iThreadStatus = KEYER_THREAD_STATUS_NOT_CREATED;
     HERE_I_AM__KEYER();
     CwKeyer_Stop(); // here: called to close the already opened serial ports
     return FALSE;
   }

  // > Define the Thread's priority as required.
  HERE_I_AM__KEYER();
  SetThreadPriority( Keyer_hThread, // handle to the thread
                     THREAD_PRIORITY_NORMAL); // thread priority level

  // Next: Start the windows-"Multimedia"-timer to provide a reliable
  //       thread loop time (how reliable ? See test results further below..)
  // With the speed of sound in air around 331 meters/second, and a distance
  // of 1 meter between sidetone output and the operator's ear, the 'acoustic'
  // delay would already be 3 ms, so it's useless to waste CPU time for a
  // one-millisecond thread loop. Thus the decision to use 2ms per thread loop.
  // How to generate a LOCAL SIDETONE with such short delays is a different question
  //  - see Remote_CW_Keyer/manual/Remote_CW_Keyer.htm#Sidetone_generation .
  if( ! Keyer_StartMMTimer( 2/*iTxInterval_ms*/ ) )
   { strcpy( CwKeyer_sz255LastError, "Failed to start the 'multimedia' timer." );
     HERE_I_AM__KEYER();
     CwKeyer_Stop(); // here: called to stop the thread and close the already opened serial ports,
                     // because Mr. "Multimedia Timer" refused to cooperate.
     HERE_I_AM__KEYER();
     return FALSE;
   }

  CwKeyer_fUpdateAllOutputs = TRUE; // send all outputs (with dozens of EscapeCommFunction()-calls) at least ONCE
  HERE_I_AM__KEYER();
  return TRUE;   // even when returning TRUE, 'minor problems' may be reported via CwKeyer_sz255LastError !

} // end CwKeyer_Start()


//----------------------------------------------------------------------------
void CwKeyer_Stop( void )  // Stops the keyer, and frees resources used by it.
{
  int i;

  if( ( CwKeyer_iThreadStatus == KEYER_THREAD_STATUS_LAUNCHED )
    ||( CwKeyer_iThreadStatus == KEYER_THREAD_STATUS_RUNNING  ) )
   { CwKeyer_iThreadStatus = KEYER_THREAD_STATUS_TERMINATE;
     HERE_I_AM__KEYER();
     for(i=0; i<20; ++i )
      { Sleep(10);
        if( CwKeyer_iThreadStatus == KEYER_THREAD_STATUS_TERMINATED ) // bingo..
         { break;  // .. the thread has terminated itself, so no need to kill it
         }
      }
     HERE_I_AM__KEYER();
   }
  if( Keyer_hThread != NULL )
   { HERE_I_AM__KEYER();
     CloseHandle( Keyer_hThread );
     Keyer_hThread  =  NULL;
     HERE_I_AM__KEYER();
   }
  CwKeyer_iThreadStatus = KEYER_THREAD_STATUS_NOT_CREATED;

  // AFTER stopping the keyer thread, also stop the "multimedia"-timer that sychronized it :
  Keyer_StopMMTimer();  // <- "won't crash if not running at all"
  HERE_I_AM__KEYER();

  // After stopping the keyer thread, also close the COM ports that it may have used:
  if( Keyer_hComPort_IN != INVALID_HANDLE_VALUE )
   { HERE_I_AM__KEYER();
     CloseHandle( Keyer_hComPort_IN );
     Keyer_hComPort_IN = INVALID_HANDLE_VALUE;
   }

  if( Keyer_hComPortRadioKeyingAndControl != INVALID_HANDLE_VALUE )
   { HERE_I_AM__KEYER();
     CloseHandle( Keyer_hComPortRadioKeyingAndControl );
     Keyer_hComPortRadioKeyingAndControl = INVALID_HANDLE_VALUE;
   }

  // If there are other threads still "peeking" at the CW keyer's
  //  current states of digital in- and outputs, make sure they don't
  //  get confused by e.g. flags like "Paddle input active", "PTT active", etc:
  Keyer_dwCurrentSignalStates = 0; // <- default state as in the var-declaration
  HERE_I_AM__KEYER();

} // end CwKeyer_Stop()

//---------------------------------------------------------------------------
void CwKeyer_SetDigitalOutput( int iSignalIndex, int iNewState )
  // [in] iSignalIndex : KEYER_SIGNAL_INDEX_xyz, defined in CwKeyer.h
  //          for T_CwKeyerConfig.iRadioCWKeying, .iRadioPTTControl, etc.
  //          If the *SIGNAL INDEX* value is NEGATIVE, the signal's polarity
  //          will be INVERTED in this function.
  //          A call with iSignalIndex = 0 = KEYER_SIGNAL_INDEX_NONE will be IGNORED.
  // [in] iNewState : TRUE  = "set" the signal (unless inverted),
  //                  FALSE = "clear" the signal (unless inverted).
  // [out] Keyer_dwCurrentSignalStates .
  // Modifies the current state for any of the application's "digital outputs"
  //          in a latch (Keyer_dwCurrentSignalStates) .
  //          Does NOT involve any time-consuming OS function call !
  // Note: The command to actually switch a digital output
  //       will be invoked from KeyerThread(), up to 2 ms later.
  //       When the new 'logic state' appears at the output
  //       is a different matter .. and depends on the USB transfer type,
  //       which is completely outside our control (on a Windows PC) !
{
  if( iSignalIndex == KEYER_SIGNAL_INDEX_NONE )
   { return;
   }
  if( iSignalIndex < 0 ) // invert the signal's polarity ?
   {  iSignalIndex = -iSignalIndex;
      iNewState    = !iNewState;
   }
  if( iNewState != 0 )
   { Keyer_dwCurrentSignalStates |= (1UL << iSignalIndex);
   }
  else
   { Keyer_dwCurrentSignalStates &= ~(1UL << iSignalIndex);
   }
  // The rest (call of some funny-named Windows API function)
  // happens in KeyerThread() !
} // end CwKeyer_SetDigitalOutput()

//---------------------------------------------------------------------------
BOOL CwKeyer_IsInputSignalAvailable(
       int iSignalIndex ) // [in]    KEYER_SIGNAL_INDEX_MORSE_KEY_..
                          //      or KEYER_SIGNAL_INDEX_RADIO_KEYING_..,
                          // defined in Remote_CW_Keyer/CwKeyer.h
  // Very frequently called from KeyerThread() -> CwKeyer_GetDigitalInput(),
  //                 to check if the "source" for a certain function
  //                 (i.e. inputs from the paddle, manual PTT switch, etc)
  //                 is available at all. When NOT available, neither
  //                 the INVERTED or NON-INVERTED function of an input
  //                 must be active !
{
  if( iSignalIndex < 0 )             // for the 'availability' of a signal,...
   {  iSignalIndex = -iSignalIndex;  // it's *POLARITY* doesn't matter.
   }
  switch( iSignalIndex )
   { case KEYER_SIGNAL_INDEX_MORSE_KEY_DTR: // first, all signals on the MORSE KEY port..
     case KEYER_SIGNAL_INDEX_MORSE_KEY_RTS:
     case KEYER_SIGNAL_INDEX_MORSE_KEY_DCD:
     case KEYER_SIGNAL_INDEX_MORSE_KEY_DSR:
     case KEYER_SIGNAL_INDEX_MORSE_KEY_CTS:
     case KEYER_SIGNAL_INDEX_MORSE_KEY_RI:
          return ( Keyer_hComPort_IN != INVALID_HANDLE_VALUE );

     case KEYER_SIGNAL_INDEX_RADIO_KEYING_DTR: // next, all signals on the RADIO KEYING port..
     case KEYER_SIGNAL_INDEX_RADIO_KEYING_RTS:
     case KEYER_SIGNAL_INDEX_RADIO_KEYING_DCD:
     case KEYER_SIGNAL_INDEX_RADIO_KEYING_DSR:
     case KEYER_SIGNAL_INDEX_RADIO_KEYING_CTS:
     case KEYER_SIGNAL_INDEX_RADIO_KEYING_RI:
          return ( Keyer_hComPortRadioKeyingAndControl != INVALID_HANDLE_VALUE );

     case KEYER_SIGNAL_INDEX_DASH_INPUT: // combined 'dash' input, possibly ORed with the state of the PC keyboard's SHIFT key
     case KEYER_SIGNAL_INDEX_DOT_INPUT : // combined 'dash' input, possibly ORed with the state of the PC keyboard's CONTROL key
          return ( Keyer_hComPort_IN != INVALID_HANDLE_VALUE )
              || ( CwKeyer_Config.fKeyViaShiftAndControl );

     default: // Anything else (that you may have added to the list in CwKeyer.h)
              // can NOT be polled as a valid 'input signal' yet .. missing support *above* 
        return FALSE;
   } // end switch( iSignalIndex )
} // end CwKeyer_IsInputSignalAvailable()


//---------------------------------------------------------------------------
BOOL CwKeyer_GetDigitalInput( int iSignalIndex )
  // Retrieves the current state of any of the application's "digital inputs"
  //           from a latch (Keyer_dwCurrentSignalStates) .
  //           Also cares for the input POLARITY (configurable, may be inverted).
  //           Does NOT involve any time-consuming OS function call !
  // [in] iSignalIndex : KEYER_SIGNAL_INDEX_xyz, defined in CwKeyer.h
  //                     for T_CwKeyerConfig.iDotInput, .iDashInput, etc.
  //                     May be ZERO = KEYER_SIGNAL_INDEX_NONE,
  //                     if there is no connection for a certain signal.
  //                     For example, T_CwKeyerConfig.iRadioPTTControl is often
  //                     "not connected" -> KEYER_SIGNAL_INDEX_NONE .
  // [in] Keyer_dwCurrentSignalStates : current states of all digital in- and
  //                     outputs, on both the MORSE KEY port
  //                                *and* the RADIO KEYING port .
  // Return values: FALSE = "logic low" (zero or even a NEGATIVE voltage),
  //                TRUE  = "logic high" (positive voltage) .
{ BOOL fInverted = FALSE;
  if( iSignalIndex == KEYER_SIGNAL_INDEX_NONE )
   { return FALSE;
   }
  // Even if a signal selection is VALID, e.g. iSignalIndex = 3 = KEYER_SIGNAL_INDEX_MORSE_KEY_DCD,
  //  the serial port providing that signal may NOT BE AVAILABLE (because stupid windows
  //  decided to assign a new COM port to it, after the user plugged the RS-232 adapter
  //  into a different port..) ! In some occasions (e.g. "manual PTT input on
  //  the keyer port's DCD), this caused unexpected activation of the 'PTT flag',
  //  which in turn the RIG CONTROL module took for granted and switched to TRANSMIT).
  // Too bad. Fixed as follows (2024-11-20) :
  if( ! CwKeyer_IsInputSignalAvailable( iSignalIndex ) )
   { return FALSE;
   }

  if( iSignalIndex < 0 ) // invert the signal's polarity ?
   {  iSignalIndex = -iSignalIndex;
      fInverted  = TRUE;
   }
  return ( (Keyer_dwCurrentSignalStates & (1UL << (DWORD)iSignalIndex) ) != 0) ^ fInverted;
  //  ,-----|_________________________|
  //  '--> Digital inputs in this bitwise combineable state variable
  //       have been updated in KeyerThread() -> GetCommModemStatus() .
} // end CwKeyer_GetDigitalInput()

//---------------------------------------------------------------------------
void CwKeyer_InitDigitalInputState( // .. to the "passive" state
      DWORD *pdwSignalStates, int iSignalIndex )
  // Initializes a certain bit in *pdwCurrentSignalStates to its "passive"
  //  state, so that CwKeyer_GetDigitalInput(iSignalIndex) returns ZERO later.
  //  This is especially important if, for example, the serial port for the
  //  PADDLE INPUTS is set to NONE. By initialising all important bits to
  //  their PASSIVE states (depending on the polarity = sign of iSignalIndex),
  //  the worker thread doesn't need to care about those 'missing inputs',
  //  and it won't erroneously start keying if a certain input hasn't been
  //  read from the hardware yet.
{
  if( iSignalIndex < 0 ) // signal with an INVERTED polarity ->
   {  iSignalIndex = -iSignalIndex;
      *pdwSignalStates |= (1UL << (DWORD)iSignalIndex);
   }
  else // non-inverted signal : passive bit in Keyer_dwCurrentSignalStates is ZERO.
   {  *pdwSignalStates &= ~(1UL << (DWORD)iSignalIndex);
   }
} // end CwKeyer_InitDigitalInputState()


//----------------------------------------------------------------------------
// Megatons of bulk / junk to access/control SERIAL PORTS under Windows ..
//----------------------------------------------------------------------------


//----------------------------------------------------------------------------
static BOOL OpenAndConfigureSerialPort(
                int iComPortNumber,    // [in] "COM port number". May be -1 for "NONE".
                long i32BitsPerSecond, // [in] "baudrate" measured in BITS PER SECOND
                int iOptions, // [in] SP_OPTION_NONE,
                              //      SP_OPTION_USE_SMALL_TX_BUFFER .
             int iDtrControl, // [in] DTR_CONTROL_ENABLE or DTR_CONTROL_DISABLE
             int iRtsControl, // [in] RTS_CONTROL_ENABLE or RTS_CONTROL_DISABLE
             HANDLE *pHandle) // [out] handle to the serial port
{
  char szPort[80];
  HANDLE hComPort;
  COMMTIMEOUTS timeouts;
  DCB dcb;       // this funny-named fellow is a "Device Contol Block".
                 // The only type of device that it can control is a SERIAL PORT.
  COMMPROP props; // this funny-named fellow may(!) tell us a bit more about
                 // the serial port driver's current configuration. We are
                 // especially interested about prop.dwMaxTxQueue, the
                 //  > "size of the driver's internal output buffer, in bytes"

  if( iComPortNumber < 0 )  // COM port number configured to NONE ? Don't try to open !
   { *pHandle = INVALID_HANDLE_VALUE;
     return TRUE;  // COM port not open, but not an error when set to "NONE"
     // (the 'keyer thread' may run with less than two COM ports,
     //  especially when used as REMOTE KEYER CLIENT or -SERVER,
     //  with a network connection between them. Then, one instance is connected
     //  connected to the SIMPLE KEYER ADAPTER, the other to the KEYING ADAPTER)
   }

  wsprintf( szPort, "\\\\.\\COM%d", iComPortNumber ); // funny decive name for windows, with THREE(!) backslashes, escaped in "C"
  HERE_I_AM__KEYER();   // 2025-08-22: suspected crashing in CreateFile() .. AND INDEED, IT CRASHED IN THE CALL BELOW, for "COM8" = the MORSE KEY PORT ! (confirmed by Keyer_iLastSourceLine in THIS line)
      // (could only be solved by unplugging and re-plugging the USB-RS232-adapter's USB cable.. very bad for true remote operation)
  hComPort = CreateFile( szPort, // [in] pointer to name of the file
         GENERIC_READ | GENERIC_WRITE, // [in] access (read-write) mode
                                    0, // [in] share mode
                                 NULL, // [in] pointer to security attributes
                        OPEN_EXISTING, // [in] "how to create"
                 FILE_FLAG_OVERLAPPED, // [in] file attribute "overlapped" (geek speak for non-blocking)
                                 NULL  // [in] handle to file with attributes to copy
                             );
  HERE_I_AM__KEYER();
  if( hComPort != INVALID_HANDLE_VALUE ) // successfully "created" (opened) the "file" (serial port)
   { // Next step (using bare Win32 API to configure the port) :
     // Configure serial data format and baudrate.
     dcb.DCBlength = sizeof( DCB );
     GetCommState( hComPort, &dcb );
     dcb.BaudRate = i32BitsPerSecond; // override the current baud rate
     dcb.fBinary  = TRUE;             // binary mode, no EOF check
     dcb.fOutxCtsFlow = FALSE;        // no CTS output flow control
     dcb.fOutxDsrFlow = FALSE;        // no DSR output flow control
     // Many "simple" level shifting interfaces draw the positive supply from the
     // DTR line (pin 4 of 9pins), so :
     dcb.fDtrControl = iDtrControl;   // DTR [flow] control type :
     // ,--------------'
     // '--> Possible values and their meanings:
     // > DTR_CONTROL_DISABLE : Disables the DTR line when the device is opened
     // >                       and leaves it disabled.
     // > DTR_CONTROL_ENABLE  : Enables the DTR line when the device is opened
     // >                       and leaves it on.
     // > DTR_CONTROL_HANDSHAKE: Enables DTR handshaking. If handshaking is enabled,
     // >                       it is an error for the application to adjust
     // >                       the line by using the EscapeCommFunction function.
     // Wikipedia said:
     //  > When a serial connection is made between two computers using a
     //  > null modem adapter, the DTR and the Data Carrier Detect (DCD)
     //  > lines are typically paired. This allows both ends of the
     //  > connection to sense when the connection is active.
     //  > On many operating systems, including Windows, the DTR line is held
     //  > low while the serial port is unused and not being controlled
     //  > by any applications.
     //       Thus, via 'DTR_CONTROL_ENABLE', we let the other guy know
     //       that WE HAVE OPENED THE INTERFACE from our side.
     dcb.fDsrSensitivity = FALSE;   // DSR sensitivity (FALSE=ignore state of DSR)
     dcb.fTXContinueOnXoff = FALSE; // XOFF continues Tx ?
     dcb.fOutX      = FALSE; // XON/XOFF out flow control
     dcb.fInX       = FALSE; // XON/XOFF in flow control
     dcb.fErrorChar = FALSE; // enable error replacement ? (no)
     dcb.fNull      = FALSE; // enable null stripping ? (no)
     dcb.fRtsControl= iRtsControl; // RTS [flow] control :
     // ,--------------'
     // '--> RTS_CONTROL_DISABLE: Disables the RTS line when the device is opened
     //    >                      and leaves it disabled.
     //    > RTS_CONTROL_ENABLE : Enables the RTS line when the device is opened
     //    >                      and leaves it on.
     //    > RTS_CONTROL_HANDSHAKE: Enables RTS handshaking. The driver raises
     //    >                      the RTS line when the "type-ahead" (input) buffer
     //    >                      is less than one-half full and lowers the RTS line
     //    >                      when the buffer is more than three-quarters full.
     //    >                      If handshaking is enabled, it is an error for the
     //    >                      application to adjust the line by using the
     //    >                      EscapeCommFunction function.
     //    > RTS_CONTROL_TOGGLE:  Specifies that the RTS line will be high if bytes
     //    >                      are available for transmission. After all buffered
     //    >                      bytes have been sent, the RTS line will be low.
     dcb.fAbortOnError = FALSE; // abort reads/writes on error
     dcb.ByteSize = (BYTE)8; // number of bits/byte, always 8
     dcb.fParity  = FALSE;   // disable parity checking ON RECEIVE
     dcb.Parity   = 0;       // no parity on TRANSMIT
     dcb.StopBits = (BYTE)0; // ONE(!) stop bit
     // dcb.XonChar = 17;    // Tx and Rx XON character
     // dcb.XoffChar= 19;    // Tx and Rx XOFF character
     // dcb.ErrorChar= 0;    // error replacement character
     // dcb.EofChar = 0;     // end of input character
     // dcb.EvtChar = 0;     // received event character
     HERE_I_AM__KEYER();
     SetCommState( hComPort, &dcb );
     HERE_I_AM__KEYER();

     // Next step (optional, and not "guaranteed to work" [*] ) :
     // Try to reduce the TRANSMIT BUFFER (output buffer), but first,
     // find out the COM port driver's current setting:
     memset( &props, 0, sizeof(props) );
     props.wPacketLength = sizeof(props); // "PacketLength" is a funny name for 'size of this struct' !
     props.dwProvSpec1 = COMMPROP_INITIALIZED;
     HERE_I_AM__KEYER();
     if( GetCommProperties( hComPort, &props ) )
      { // > "If the function succeeds, the return value is nonzero" ....
        // With an USB<->RS232 adapter with FTDI chip, got here with
        // props.wPacketVersion = 2
        // props.dwMaxBaud      = 0x10000000 = "BAUD_USER". A-ha.
        // props.dwMaxTxQueue = 0 = "no maximum value is imposed by the serial provider"
        // props.dwMaxRxQueue = 0 = "no maximum value is imposed by the serial provider"
        // props.dwCurrentTxQueue = 0 = "the value is unavailable" (Nnngrrr...)
        if( iOptions & SP_OPTION_USE_SMALL_TX_BUFFER )
         { if( ! SetupComm( hComPort, // <- this funny-named fellow may be able to configure BUFFER SIZES (or not..)
                      256, // [in] dwInQueue  (number of bytes)
                       1)) // [in] dwOutQueue (number of bytes)  : 2 ms * 4800 baud = 9.6 bits per thread loop
            { // [*] about SetupComm(), from what used to be MSDN:
              // > The device driver receives the recommended buffer sizes,
              // > but is free to use any input and output (I/O) buffering scheme.
              // Arrived here: SetupComm() FAILED. Not a big surprise. Just carry on.
              // strcpy( CwKeyer_sz255LastError, "SetupComm() could not set the TX-buffer-size" );
            }
         } // end if( iOptions & SP_OPTION_USE_SMALL_TX_BUFFER )
      } // end if <GetCommProperties() successful >

     // Next step (using bare Win32 API to configure the port) :
     // Configure the COM port "timeouts", so that ReadFile() / WriteFile()
     //  won't block for more than a few dozen milliseconds:
     HERE_I_AM__KEYER();
     GetCommTimeouts( hComPort, &timeouts );
     timeouts.ReadIntervalTimeout        = 0; // (1)
     // About COMMTIMEOUTS.ReadIntervalTimeout   (1) :
     // > "maximum time, in ms, allowed to elapse between the arrival of two characters" . (..)
     // > A value of MAXDWORD, combined with zero values for both the
     // > ReadTotalTimeoutConstant and ReadTotalTimeoutMultiplier members,
     // > specifies that the read operation is to return immediately
     // > with the characters that have already been received, even if
     // > no characters have been received.
     // The above WOULD BE used to make "ReadFile" non-blocking.
     //
     timeouts.ReadTotalTimeoutMultiplier = 0; // (2)
     // About   *ReadTotalTimeoutMultiplier*     (2) :
     // > The multiplier used to calculate the total time-out period
     // > for read operations, in milliseconds. For each read operation,
     // > this value (2) is multiplied by the requested number of bytes to be read.
     // (In this application, ReadFile() is always called with the MAXIMUM
     //  size of our own receive-buffer to drain as many bytes bytes as
     //  possible in a single call, and of course we never wait
     //  until the COMPLETE buffer is filled.
     //  Thus, a 'real' ReadTotalTimeoutMultiplier is useless for this purpose. )
     //
     timeouts.ReadTotalTimeoutConstant = 20;  // (3)
     // About * ReadTotalTimeoutConstant *       (3)  :
     // > A constant used to calculate the total time-out period for read operations,
     // > in milliseconds. For each read operation, this value (3) is added
     // > to the product of the ReadTotalTimeoutMultiplier member (2)
     // > and the requested number of bytes.
     // > Remarks
     // > If an application sets ReadIntervalTimeout (1) to MAXDWORD
     // >         **and** ReadTotalTimeoutMultiplier (2) to MAXDWORD
     // >      **and** sets ReadTotalTimeoutConstant (3) to a value greater
     // > than zero and less than MAXDWORD, one of the following occurs
     // > when the ReadFile function is called:
     // >   * If there are any bytes in the input buffer,
     // >      ReadFile returns immediately with the bytes in the buffer.
     // >   * If there are no bytes in the input buffer, ReadFile waits until
     // >      a byte arrives and then returns immediately.
     // >   * If no bytes arrive within the time specified by
     // >      ReadTotalTimeoutConstant, ReadFile times out.
     //
     timeouts.WriteTotalTimeoutMultiplier= 0;  // (4)
     // About * WriteTotalTimeoutMultiplier *  // (4) :
     // > The multiplier used to calculate the total time-out period
     // > for write operations, in milliseconds. For each write operation,
     // > this value is multiplied by the number of bytes to be written.
     timeouts.WriteTotalTimeoutConstant  = 0;  // (5)
     // About * WriteTotalTimeoutConstant   *  // (5) :
     // > A constant used to calculate the total time-out period for
     // > write operations, in milliseconds. For each write operation,
     // > this value is added to the product of the
     // > WriteTotalTimeoutMultiplier member and the number of bytes
     // > to be written.
     // >     A value of zero for both the WriteTotalTimeoutMultiplier
     // >     **and** WriteTotalTimeoutConstant members indicates that
     // >     total time-outs are not used for write operations.
     //    (the latter appeared ok for overlapped I/O :
     //         ReadIntervalTimeout        = MAXDWORD,
     //         ReadTotalTimeoutMultiplier = MAXDWORD,
     //         ReadTotalTimeoutConstant   = 5 .. 10 [ms, but see TESTS further above]
     HERE_I_AM__KEYER();
     SetCommTimeouts( hComPort, &timeouts );

     *pHandle = hComPort;    // store the serial port's handle to access it
     HERE_I_AM__KEYER();
     return TRUE;
   }
  else // could't open the serial port, so don't try to CONFIGURE it :
   { snprintf( CwKeyer_sz255LastError, 255, "Could not open serial port 'COM%d'",(int)iComPortNumber );
     //         '--> Displayed on the 'Debug' tab in TKeyerMainForm::StartKeyerAndShowInfo() !
     HERE_I_AM__KEYER();
     return FALSE;
   }
} // end OpenAndConfigureSerialPort()


#if(L_USE_MULTIMEDIA_TIMER_CALLBACK)
//-------------------------------------------------------------------------
void CALLBACK Keyer_MMTimerCallback(UINT wTimerID, UINT msg, DWORD dwUser, DWORD dw1, DWORD dw2)
  // Callback function for the multimedia timer. Periodically wakes up
  //                       the KEYER THREAD (every 2 milliseconds) .
{
  // BEWARE ! DANGER ! Only a few functions may be called from here,
  //                   last not least because Mr. "MultiMedia-Timer"
  //                   runs in its own thread, almost like an interrupt handler
  //                   (which of course it is NOT). But that's possibly the
  //                   reason why Microsoft wants to let Mr. MultiMediaTimer die,
  //                   without offering a suitable replacement
  //                   that runs on older hardware, too .
  //   So DON'T EVEN THING ABOUT calling any shiny C++ class methods from here,
  //   don't use the VCL (which is said to be thread-unsafe),
  //   don't do this, don't do that, etc .  Even the Win32 API "SetEvent()"
  //   is a questionable case, but we use it to measure the latency here.
  if( Keyer_hEventFromMMTimer != NULL ) // may be NULL when "not in use"
   { TIM_StartStopwatch( &Keyer_swMMTimerToThread ); // what's the latency between SetEvent() and returning from WaitForSingleObject() ?
        // '-- considered "harmless" even when called from the MMTimer-thread;
        //     only 'Queries the Performance Counter', and that's it ...
     SetEvent( Keyer_hEventFromMMTimer ); // CQ worker thread !
        // '-- not really sure if THIS CALL is allowed. In old forum threads,
        //     someone speculated it's not. We just "dare to send an event"
        //     from here. For our own peace of mind, measure the time
        //     spent in SetEvent() itself (further below).
     // In Microsoft geek speak:
     // > Sets the specified event object to the signaled state.
     //
     // Here, to keep the number of Win32 API calls to the minimum,
     // Keyer_hEventFromMMTimer is "self-resetting" which means,
     // as soon as the thread waiting for it (in e.g. WaitForSingleObject)
     // has been woken up, Keyer_hEventFromMMTimer is RESET,
     // without the need for ResetEvent(). See CreateEventForPeriodicTransmissions().
     // If Mr. Multimedia-Timer "sets" the event more often than the
     // event receiver - e.g. CSound_SerialPortWriterThread() -
     // can handle, the 'event' simply remains signaled, but there is no
     // fancy queue or anything that "counts". Just signaled or not signaled.
     Keyer_i32TimeSpentInSendEvent_us = TIM_ReadStopwatch_us( &Keyer_swMMTimerToThread );
        //  '-- Test result: SetEvent() consumed  1 us here;
        //        WaitForSingleObject() returned  a few dozen microseconds later.
        // Not reliable, but MUCH BETTER than trying to achieve a half-
        // way constant SAMPLE- or PACKET-TRANSMIT-INTERVAL by other means.
   }

#if( SWI_NUM_AUX_COM_PORTS > 0 ) // compile with support for 'Auxiliary / Additional COM ports' / Winkeyer emulation ?
  // Since 2025-06-07, module Remote_CW_Keyer/AuxComPorts.c may also have got
  //  something to do 'every two milliseconds'. To avoid having to open register
  //  another 'Multi-Media-Timer-Callback' for that, call the
  //  'Auxiliary / Additional COM port' module from here. The following function
  //  will also just set an event (via SetEvent()) to wake up some worker thread
  //  *IF NECESSARY* :
  AuxComPorts_MultiMediaTimerCallback_2ms();  // <- must be "robust enough" for being called FROM ANYWHERE, AT ANY TIME !
#endif // SWI_NUM_AUX_COM_PORTS > 0 ?

} // end CALLBACK Keyer_MMTimerCallback()
#endif // (L_USE_MULTIMEDIA_TIMER_CALLBACK) ?


//-------------------------------------------------------------------------
BOOL Keyer_StartMMTimer( int iTxInterval_ms )
  // Creates a WIN32 MULTIMEDIA TIMER to produce a 'precisely timed' events
  //  - as far as this non-real-time-operating-system permits.
{
  TIMECAPS tc;  // yet another funny windows-specific stuff

  // If timer already created, kick it out..
  if( Keyer_MultimediaTimerId != NULL )
   { timeKillEvent( Keyer_MultimediaTimerId );
     // > Remarks
     // > Each call to timeSetEvent for periodic timer events requires
     // > a corresponding call to the timeKillEvent function.
     Keyer_MultimediaTimerId = 0;
   }


  if( Keyer_hEventFromMMTimer == NULL ) // no "event" object yet ? Create...
   { Keyer_hEventFromMMTimer = CreateEvent( NULL, FALSE,  FALSE,  NULL );
     // Pointer to security attributes -------'    |       |       |
     // Flag for manual-reset event.  <------------'       |       |
     //    If FALSE, Windows automatically resets          |       |
     //    the state to nonsignaled after a                |       |
     //    single waiting thread has been released.        |       |
     // Flag for initial state: FALSE = not signalled <----'       |
     // Pointer to event-object name  <----------------------------'
     // Note: If "CreateEvent" fails, it returns NULL (not INVALID_HANDLE_VALUE).
     //       But the return value is type HANDLE. Thank you, Microsoft.
   }

  timeGetDevCaps( &tc, sizeof(tc) ); // > "queries the timer device to determine its resolution." (holy crap..)
  timeBeginPeriod( tc.wPeriodMin );  // > "requests a minimum resolution for periodic timers." (au hauer ha...)
      // > uPeriod ------'
      // > Minimum timer resolution, in milliseconds, for the application
      // > or device driver. A lower value specifies a higher (more accurate) resolution.
      // (On the author's "HP Envy" with Intel i7 and Windows 10, the above
      //  funny-named functions promised a 'resolution' of one millisecond. )

  // "Create" (start?) the now-deprecated Multi-Media timer (the only thing we had
  //                   for such "incredibly short intervals" like ONE MILLISECOND) :
  //
  // > The timeSetEvent function starts a specified timer event.
  // > The multimedia timer runs in its own thread. After the event is activated,
  // > it calls the specified callback function or sets or pulses
  // > the specified event object.
  //
  //   MMRESULT timeSetEvent(
  //   UINT           uDelay,
  //   UINT           uResolution,
  //   LPTIMECALLBACK lpTimeProc,
  //   DWORD_PTR      dwUser,
  //   UINT           fuEvent
  //   );
  //
  //   uDelay:
  //    Event delay, in milliseconds
  //
  //   uResolution:
  //    Resolution of the timer event, in milliseconds.
  //    A resolution of 0 indicates periodic events should occur with the
  //    greatest possible accuracy.
  //    You should use the use the maximum value appropriate to reduce system overhead.
  //
  //   lpTimeProc:
  //    if fuEvent specifies the TIME_CALLBACK_EVENT_SET or TIME_CALLBACK_EVENT_PULSE flag,
  //    then the lpTimeProc parameter is interpreted as a handle to an event object.
  //    The event will be set or pulsed upon completion of a single event
  //    or periodically upon completion of periodic events.
  //
  //   fuEvent:
  //    TIME_ONESHOT Event occurs once, after uDelay milliseconds.
  //    TIME_PERIODIC Event occurs every uDelay milliseconds.
  //    TIME_CALLBACK_EVENT_SET: When the timer expires, Windows calls the
  //         SetEvent function to set the event pointed to by the
  //         lpTimeProc parameter. The dwUser parameter is ignored.
  //
  // Return Values
  //   Returns an identifier for the timer event if successful or
  //   an error otherwise. This function returns NULL if it fails
  //   and the timer event was not created. (This identifier is also passed
  //   to the callback function.)
  //
  // Remarks
  //   Each call to timeSetEvent for periodic timer events must be matched
  //   with a call to the timeKillEvent function.
  Keyer_MultimediaTimerId = timeSetEvent(
         iTxInterval_ms, // UINT uDelay : period in milliseconds
         0 ,             // UINT uResolution :  a resolution of 0 indicates periodic
                         // events should occur with the greatest possible accuracy
#      if(L_USE_MULTIMEDIA_TIMER_CALLBACK)
        Keyer_MMTimerCallback, //   LPTIMECALLBACK lpTimeProc: here, a "C" function
        (DWORD_PTR)0,      // DWORD_PTR dwUser: not used yet, maybe a class instance one day
        TIME_PERIODIC);    // UINT fuEvent
#      else  // don't use a CALLBACK FUNCTION but let Mr. MMTimer send EVENTS:
        LPTIMECALLBACK(Keyer_hEventFromMMTimer), // LPTIMECALLBACK lpTimeProc: here, an EVENT HANDLE (!)
        (DWORD_PTR)0,      // DWORD_PTR dwUser: Here: IGNORED (when sending EVENTS, not INVOKING A CALLBACK)
        TIME_PERIODIC | TIME_CALLBACK_EVENT_SET );  //   UINT fuEvent
#      endif // ! L_USE_MULTIMEDIA_TIMER_CALLBACK ?
  return Keyer_MultimediaTimerId != 0;  // -> TRUE=success, FALSE=error

} // end Keyer_StartMMTimer()

//-------------------------------------------------------------------------
void Keyer_StopMMTimer( void ) // <- won't crash if not running at all, but
              // without the MM timer, KeyerThread() doesn't run properly !
              // Only called from CwKeyer_Stop() .
{
  if( Keyer_MultimediaTimerId != NULL )
   { timeKillEvent( Keyer_MultimediaTimerId );
     // > Remarks (by Microsoft) :
     // >  Each call to timeSetEvent for periodic timer events must be matched
     // >  with a call to the timeKillEvent function.
     Keyer_MultimediaTimerId = 0;
   }
} // end Keyer_StopMMTimer()

#if( SWI_USE_MIDI )
//---------------------------------------------------------------------------
void CwKeyer_StartMidiSidetone(void)
{
  T_ShortMidiMsg msg;
  if( CwKeyer_hMidiDevice != INVALID_HANDLE_VALUE )
   {
     msg.bData[0] = 0x90;                  // "status byte" : "Note on"
     // ,--------------'
     // '--> lower 4 bits are the synthesizer's MIDI channel, #0..15
     msg.bData[1] = CwKeyer_iMidiNote;     // note # / 'synthesizer key'
     msg.bData[2] = 0x40; // velocity with which key was pressed,
                          //  1 = very slow .. 0x40 = default .. 0x7F = fast
     // (0x40 already caused an annoying 'overshoot', but didn't fix
     //       the annoying LATENCY of the MIDI output)
     msg.bData[3] = 0;    // unused, but don't send garbage
     midiOutShortMsg( CwKeyer_hMidiDevice, msg.dwData );
   } // end if( CwKeyer_hMidiDevice == INVALID_HANDLE_VALUE )
}

//---------------------------------------------------------------------------
void CwKeyer_StopMidiSidetone(void)
{
  T_ShortMidiMsg msg;
  if( CwKeyer_hMidiDevice != INVALID_HANDLE_VALUE )
   {
     msg.bData[0] = 0x80;                  // "status byte" : "Note off"
     // ,--------------'
     // '--> lower 4 bits are the synthesizer's MIDI channel, #0..15
     msg.bData[1] = CwKeyer_iMidiNote;     // note # / 'synthesizer key'
     msg.bData[2] = 100;   // velocity with which key was released,
                           //  0=min,  0x40=default, 0x7F=max .
     msg.bData[3] = 0;     // unused, but don't send garbage
     midiOutShortMsg( CwKeyer_hMidiDevice, msg.dwData );
     // > "The interpretation of this message is up to the receiving MIDI device"
   } // end if( CwKeyer_hMidiDevice == INVALID_HANDLE_VALUE )
}

#endif // SWI_USE_MIDI ?

//---------------------------------------------------------------------------
static void KeyerThread_CheckLastErrorAfterTroubleWithSerialPort(
             HANDLE *pComPortHandle, // [in,out] handle, possibly closed and set to INVALID_HANDLE_VALUE
             char *pszTriedWhat )    // [in] : e.g. "ReadFile"
  // If the check of the 'Last Error Code' indicates a real error,
  // the serial port's handle is closed, and the handle value set to "illegal".
  // The MAIN THREAD / task may decide to re-open the port a few times,
  //     after waiting for a considerable time to allow the OPERATOR
  //     to re-insert the USB device or similar.
{
  DWORD dwLastError = GetLastError(); // <- MS claims this is 'thread aware' but.. :
    // > You should call the GetLastError function immediately when a function's
    // > return value indicates that such a call will return useful data.
    // > That is because some functions call SetLastError with a zero when
    // > they succeed, wiping out the error code set by the most recently
    // > failed function.
  char sz127[128];
  BOOL fCloseSerialPort = FALSE;

  // Is it a REAL error (not one of those 'false alerts' caused by overlapped I/O) ?
  switch( dwLastError )
   { case ERROR_IO_PENDING   : // the infamous '997' : "Overlapped I/O operation is in progress."
     case ERROR_IO_INCOMPLETE: // the infamous '996' : "Overlapped I/O event is not in a signaled state."
        break;
     case ERROR_OPERATION_ABORTED: // '995' : "The I/O operation has been aborted because of either a thread exit or an application request."
        fCloseSerialPort = TRUE;
        if( Keyer_dwSerialPortErrors < 10 )   // Cry for help in the 'error history' ..
         { ++Keyer_dwSerialPortErrors;
           // "The I/O operation has been aborted because of either a thread exit or an application request."
           // Whow. Now go and explain that to your customer. Good luck asking for mercy.
           ShowError( ERROR_CLASS_ERROR, "KeyerThread: I/O on the RADIO PORT failed (%s)", pszTriedWhat );
         }
        break;
     default: // guess it's another REAL error, related with a "COM" port :
        fCloseSerialPort = TRUE;
        if( Keyer_dwSerialPortErrors < 10 )   // Cry for help in the 'error history' ..
         { ++Keyer_dwSerialPortErrors;
           // When pulling out the USB cable connecting the keyer to an IC-7300,
           // the result (shown in the 'Debug' tab) sometimes was:
           //  "Access is denied".
           //    '--> OH MY GOD.
           // In other cases, the result in sz127 was a long, chatty story like this (#995) :
           // > "The I/O operation has been aborted because of either a thread exit or an application request."
           // Whow. Now go and explain that to your customer. Good luck asking for mercy.
           // > "The communication with the remotely controlled radio stopped because your cat has pulled out the USB cable."
           // Well, maybe, after Microsoft forced the introduction of AI in Windows.
           UTL_LastWindowsErrorCodeToString( dwLastError, sz127, 127/*iMaxDestLen*/ );
           ShowError( ERROR_CLASS_ERROR, "KeyerThread: RADIO PORT failed (%s -> %s)", pszTriedWhat, sz127 );
         }
        break; // end default < dwLastError considered to be a REAL error with the "COM" / "Comm" port API >
   } // end switch( dwLastError )
  if( fCloseSerialPort && (pComPortHandle!=NULL) )
   { if( *pComPortHandle != INVALID_HANDLE_VALUE )
      { CloseHandle( *pComPortHandle );
        *pComPortHandle = INVALID_HANDLE_VALUE;  // <- TKeyerMainForm::Timer1Timer() may take notice and try to re-open it AFTER SOME TIME
      }
   }
} // end KeyerThread_CheckLastErrorAfterTroubleWithSerialPort()


//---------------------------------------------------------------------------
DWORD WINAPI KeyerThread( LPVOID lpParam )
  // 'Keyer thread' : Polls the paddle- or straight key contacts,
  //   and TRIES TO (CW-)"key" the radio via USB / Virtual COM port.
  //   Later versions could also play text from memory, using CwGen.c .
  //   Even later versions also produced a sidetone on a selectable
  //                'audio' device (speaker, soundcard) using CwDSP.c .
  //   Also, this thread may interact with the 'CW Network',  CwNet.c .
  //
  // Like applications in a PLC (Programmable Logic Controller) ...
  //   * this thread ideally runs with a constant loop time (2 ms),
  //   * samples all inputs for the 'process image' (PLC term) only ONCE per loop,
  //   * and drives all outputs from the 'process image' only ONCE per loop,
  //   * must not call any subroutine that consumes more than say ONE LOOP TIME.
  //
{

  #define L_TX_BUFFER_SIZE 8192 // write to SERIAL PORT in small chunks (no problem here, in this "fast-running thread loop")
    // At 9216000 "baud", expect having to send 8192 bytes in 88 ms !
  BYTE  bMyTxBuffer[L_TX_BUFFER_SIZE];
  #define L_THREAD_MAX_MS_PER_TX_LOOP 5
    // If it wasn't WINDOWS, we'd only wait for 5 ms per thread loop
    // to keep the 'bursts' small but sent as frequently as possible.
    // But "expect the worst" (WaitForSingleObject() waiting for DOZENS OF MILLISECONDS,
    //                         or even worse, Sleep() ),
    // so make L_TX_BUFFER_SIZE much larger so the next WriteFile()
    // can send larger blocks to 'catch up' with the buffer-filling-rate .
  DWORD dwNumBytesToWrite, dwNumBytesWritten, dwNumBytesRead;
  DWORD dwModemStatus;
  DCB sCommState[2]; // these funny-named fellows are "Device Contol Blocks".
       // The only type of device that they can control are SERIAL PORTS.
       // sCommState[0] is for the serial port that polls the MORSE KEY,
       // sCommState[1] is for the serial port that controls ("keys") the RADIO.
  DWORD dwLastSentSignalStates = 0; // snapshot of last Keyer_dwCurrentSignalStates, when updating DIGITAL OUTPUTS,
           // to minimize the number of calls to SetCommState() or EscapeCommFunctions().
  BOOL  fDigitalInputState, fMorseOutput, fMorseOutput_sent, fResult;
  T_CwNet *pCwNet = CwKeyer_Config.pCwNet; // optional instance for CW keying "via network"


  // ex: BOOL  fWaitedSomewhere;  // Useless. We MEASURE the time spent in a loop,
  // to find out if Windows has "waited" in any of the API functions called below.
  T_TIM_Stopwatch sw_BeginOfThreadLoop, sw_SpeedTest, sw_MorseTxFifo, sw_TxBusy,
                  sw_DotInputWatchdog, sw_DashInputWatchdog;
  long nMicrosecondsSinceLastCall;
  int  t_us, t_ms, iResult;
  WORD wCwChar;
  char *cp;
  char sz127[128];


 // INT i,iMax,iChn;
 // INT error_code;

  // Also here (similar as for CSound_SerialPortReaderThread) : See Microsoft's
  //     "Serial Communications in Win32" by Allen Denver,
  // saved on WB's workhorse as file:///C:/downloads/Windows_Programming/msdn_serial.htm
  // or (slightly vamped up) C:\downloads\Windows_Programming\Serial Communications in Win32.pdf .
  int   iThreadExitCode = 1;
         // Note: The code STILL_ACTIVE must not be used as parameter for
         // ExitThread(), but what else ? See Win32 API help !
         // in winbase.h:
         //    #define STILL_ACTIVE STATUS_PENDING
        // in winnt.h:
        //    #define STATUS_PENDING ((DWORD   )0x00000103L)

  DWORD dwWaitResult, dwRes, dwNewSignalStates;
  BOOL  fAltDashInput, fAltDotInput;
  OVERLAPPED osWriteForMorseKeyPort = { 0 };
  OVERLAPPED osWriteForRadioPort    = { 0 };
  OVERLAPPED osReadForRadioPort     = { 0 };
  BOOL  fReadFromRadioPortPending = FALSE;
  BOOL  fWriteToRadioPortPending  = FALSE;
  DWORD dwNumBytesWrittenToRadioPort;
  char  sz4Temp[4];
  BYTE  b256RxBuffer[256], b256TxBuffer[256];

  CwKeyer_iThreadStatus = KEYER_THREAD_STATUS_RUNNING;
  Keyer_dwThreadLoops = Keyer_dwThreadErrors = Keyer_dwSerialPortErrors = 0;
  pCwNet->fCurrentRemotelyReceivedKeyDownState = pCwNet->fNextRemotelyReceivedKeyDownState = FALSE;


  // Create the overlapped event for READING (only used within this function, thus LOCAL).
  // Must be closed before exiting to avoid a handle leak.
  osWriteForMorseKeyPort.hEvent = CreateEvent(NULL, FALSE, FALSE,  NULL);
      // Pointer to security attributes ---------'    |     |       |
      // Flag for manual-reset event.  <--------------'     |       |
      //    If FALSE, Windows automatically resets          |       |
      //    the state to nonsignaled after a                |       |
      //    single waiting thread has been released.        |       |
      // Flag for initial state: FALSE = "not signalled" <--'       |
      // Pointer to event-object name  <----------------------------'
  // Similar as above for the RADIO KEYING / RADIO CONTROL port,
  // over which we MAY have to send or receive serial data, with e.g. CI-V protocol.
  // Because we opened the serial ports with FILE_FLAG_OVERLAPPED, this is mandatory:
  osWriteForRadioPort.hEvent = CreateEvent(NULL, FALSE, FALSE,  NULL);
  osReadForRadioPort.hEvent  = CreateEvent(NULL, FALSE, FALSE,  NULL);

  if( (osWriteForMorseKeyPort.hEvent == NULL)
    ||(osWriteForRadioPort.hEvent == NULL)
    ||(osReadForRadioPort.hEvent  == NULL) )
   { CwKeyer_iThreadStatus = KEYER_THREAD_STATUS_TERMINATED;
     return -1;  // Error creating overlapped event; abort.
     // (if NOT ALL THREE 'creations' failed, there's a leak now. Oh well.)
   }

  CwKeyer_fUpdateAllOutputs = TRUE; // send all outputs (with dozens of EscapeCommFunction()-calls) at least ONCE


  //- - - - - - - - - No 'lazy return' after this point ! - - - - - - - - - - -

  TIM_StartStopwatch( &sw_BeginOfThreadLoop ); // don't leave this un-initialized..
  TIM_StartStopwatch( &sw_MorseTxFifo );
  TIM_StopStopwatch( &sw_TxBusy );  // not TRANSMITTING yet so keep THIS stopwatch stopped :o)
  TIM_StopStopwatch( &CwKeyer_swMorseActivityTimer );
  TIM_StartStopwatch( &sw_DotInputWatchdog );
  TIM_StartStopwatch( &sw_DashInputWatchdog );

  fMorseOutput_sent = FALSE;
  Keyer_fDotInputWasPassive = Keyer_fDashInputWasPassive = FALSE;

  // Almost endless KEYER thread loop begins HERE.............................
  while( CwKeyer_iThreadStatus == KEYER_THREAD_STATUS_RUNNING )
   {
     ++Keyer_dwThreadLoops;

     // WAIT for an event from the multimedia-timer to avoid busy-spinning loops:
     if( Keyer_hEventFromMMTimer != NULL ) // shouldn't be NULL, but PLAY SAFE..
      { HERE_I_AM__KEYER(); // -> Keyer_iLastSourceLine here ? Died in 'WaitForSingleObject()' !
        if( (dwWaitResult=WaitForSingleObject( Keyer_hEventFromMMTimer, 10/*ms plus a LOT (*) */ ) )
             == WAIT_OBJECT_0 ) // the event is in a "signaled" state so RESET it (for the next time)
         { Keyer_i32LatencyMMTimerToThread = TIM_ReadStopwatch_us( &Keyer_swMMTimerToThread );
           // Measured latency between SetEvent()       [in the MultiMedia timer thread]
           // and returning from WaitForSingleObject()  [in the "Serial Port Writer" thread]:
           //   61 us,  2 us (!?), 2 us (!?),  6 us (!?), sometimes even ZERO !
           HERE_I_AM__KEYER();
         }
        else
         { HERE_I_AM__KEYER(); // -> Keyer_iLastSourceLine here ? failed to wait for the "Multi-Media-Timer" !
           // After resuming from the PC's "sleep mode" (?), WaitForSingleObject()
           // didn't wait for circa 2000 us anymore, but for only about 10 us
           // per loop .. in other words, KeyerThread() turned into a CPU-hogging
           // monster (busy-spinning loop). Tried to find out why.
           // From the documentation about WaitForSingleObject() :
           // > If the function succeeds, the return value indicates the event
           // > that caused the function to return. (...)
           // > WAIT_OBJECT_0  : This is what we EXPECT when all works as planned.
           // > WAIT_ABANDONED :
           // >   The specified object is a mutex object that was not released
           // >   by the thread that owned the mutex object
           // >   before the owning thread terminated.
           // > WAIT_TIMEOUT : The time-out interval elapsed, and the object's
           // >   state is nonsignaled.
           //    (shouldn't happen unless the MultiMedia-Timer dies while the
           //     PC "sleeps" ... wouldn't be the first 'nice suprise')
           // > WAIT_FAILED : The function has failed. To get extended error
           // >   information, call GetLastError.
           switch( dwWaitResult )
            { case WAIT_ABANDONED :
                   ++Keyer_dwThreadErrors;
                   break;
              case WAIT_TIMEOUT   :  // hopefully just a 'temporary hiccup' from Mr. Multimedia-Timer ..
                   ++Keyer_dwThreadErrors;
                   break;
              case WAIT_FAILED    :
                   ++Keyer_dwThreadErrors;
                   if( Keyer_dwThreadErrors < 10 )  // Cry for help in the 'error history' ?
                    { UTL_LastWindowsErrorCodeToString( GetLastError(), sz127, 127/*iMaxDestLen*/ );
                      ShowError( ERROR_CLASS_ERROR, "KeyerThread: Failed to wait for MMTimer (%s)", sz127 );
                    }
                   break;
            }
         } // end < WaitForSingleObject() returned something 'unusual' >
      }
     else // it there's no "event" (handle) to wait for, Sleep() instead ...
      { HERE_I_AM__KEYER();
        Sleep(5); // <- This completely sucks for periodic events,
           // because windows will let this thread "sleep"
           // for MUCH LONGER than 5 milliseconds (often 16 ms) .
           // But SLEEPING here (when there's no "multimedia timer")
           // is better than burning CPU time in a busy-spinning loop.
      }
     HERE_I_AM__KEYER();
     // Measure the number of microseconds elapsed since getting HERE in the previous thread loop.
     // This is also used for the internal timing of a few functions periodically called from here:
     CwKeyer_dw8ThreadIntervals_us[Keyer_dwThreadLoops&7]
       = nMicrosecondsSinceLastCall = TIM_ReadAndRestartStopwatch_us( &sw_BeginOfThreadLoop );
     // If anything slows down this thread occasionally (e.g. task switching,
     // sluggish API function to drive modem control lines, etc etc etc),
     // the "Elbug handler" needs to know this, to keep the duration of
     // dots, dashes, and pauses as precise as the thread loop times permit:


     // Poll the current "digital inputs" (Modem status lines)
     // by calling GetCommModemStatus() as a BLOCKING CALL, because ..
     // (from Microsoft about "Communications Events")  :
     // > When monitoring an event that occurs when a signal (CTS, DSR, and so on)
     // > changes state, WaitCommEvent reports the change, but not the current state.
     // > To query the current state of the CTS (clear-to-send), DSR (data-set-ready),
     // > RLSD (receive-line-signal-detect), and ring indicator signals,
     // > a process can use the GetCommModemStatus function.
#   if(0) // old stuff, doesn't work with the "manual output control" in Keyer_Main.cpp !
     dwNewSignalStates = 0;
#   else  // don't touch the states of digital OUTPUTs in "new signal states" :
     dwNewSignalStates = Keyer_dwCurrentSignalStates
                     & ( (1<<KEYER_SIGNAL_INDEX_MORSE_KEY_DTR) // leave these digital OUTPUTS unchanged...
                        |(1<<KEYER_SIGNAL_INDEX_MORSE_KEY_RTS) // .. and only clear the INPUT-bits in "new signal states"
                        |(1<<KEYER_SIGNAL_INDEX_RADIO_KEYING_DTR)
                        |(1<<KEYER_SIGNAL_INDEX_RADIO_KEYING_RTS) );
#   endif
     // Set all "input"-bits in the 'process image' (PLC slang) to their PASSIVE state.
     // If we cannot poll them further below via GetCommModemStatus() further below,
     // we want those INPUTS to indicate 'button NOT pressed', 'paddle NOT operated', etc !
     CwKeyer_InitDigitalInputState( &dwNewSignalStates, CwKeyer_Config.iDashInput );
     CwKeyer_InitDigitalInputState( &dwNewSignalStates, CwKeyer_Config.iDotInput  );
     CwKeyer_InitDigitalInputState( &dwNewSignalStates, CwKeyer_Config.iManualPTTInput);
     if(  (CwKeyer_Config.iTestInput != CwKeyer_Config.iDashInput      )
       && (CwKeyer_Config.iTestInput != CwKeyer_Config.iDotInput       )
       && (CwKeyer_Config.iTestInput != CwKeyer_Config.iManualPTTInput ) )
      { // only if the "initial state" for THIS input hasn't been set above:
        CwKeyer_InitDigitalInputState( &dwNewSignalStates, CwKeyer_Config.iTestInput );
      }

     TIM_StartStopwatch( &sw_SpeedTest );  // measure time spent in the following calls:
     if( Keyer_hComPort_IN != INVALID_HANDLE_VALUE ) // is the "Morse Key" INPUT port open ?
      { // > The GetCommModemStatus function retrieves modem control-register values.
        // > If the function succeeds, the return value is nonzero.
        // > If the function fails, the return value is zero.
        if( GetCommModemStatus( Keyer_hComPort_IN, &dwModemStatus ) )
         { if( dwModemStatus & MS_CTS_ON )
            { // > "The CTS (clear-to-send) signal is on." ->
              dwNewSignalStates |= (1 << KEYER_SIGNAL_INDEX_MORSE_KEY_CTS);
            }
           else
            { dwNewSignalStates &= ~(1 << KEYER_SIGNAL_INDEX_MORSE_KEY_CTS);
            }
           if( dwModemStatus & MS_DSR_ON )
            { // > "The DSR (data-set-ready) signal is on." ->
              dwNewSignalStates |= (1 << KEYER_SIGNAL_INDEX_MORSE_KEY_DSR);
            }
           else
            { dwNewSignalStates &= ~(1 << KEYER_SIGNAL_INDEX_MORSE_KEY_DSR);
            }
           if( dwModemStatus & MS_RING_ON )
            { // > "The ring indicator signal is on." ->
              dwNewSignalStates |= (1 << KEYER_SIGNAL_INDEX_MORSE_KEY_RI);
            }
           else
            { dwNewSignalStates &= ~(1 << KEYER_SIGNAL_INDEX_MORSE_KEY_RI);
            }
           if( dwModemStatus & MS_RLSD_ON )
            { // > "The RLSD (receive-line-signal-detect) signal is on."
              // (a more common name for this funny thing is "DCD" = "Data Carrier Detect")
              dwNewSignalStates |= (1 << KEYER_SIGNAL_INDEX_MORSE_KEY_DCD);
            }
           else
            { dwNewSignalStates &= ~(1 << KEYER_SIGNAL_INDEX_MORSE_KEY_DCD);
            }
         } // end if < GetCommModemStatus() successful > ?
      }   // end if( Keyer_hComPort_IN != INVALID_HANDLE_VALUE ) ?
     // Almost the same as above also for the RADIO-KEYING-AND-CONTROL-port :
     if( Keyer_hComPortRadioKeyingAndControl != INVALID_HANDLE_VALUE ) // is the "Radio Keying/Control Port" open ?
      { if( GetCommModemStatus( Keyer_hComPortRadioKeyingAndControl, &dwModemStatus ) )
         { if( dwModemStatus & MS_CTS_ON )
            { // > "The CTS (clear-to-send) signal is on." ->
              dwNewSignalStates |= (1 << KEYER_SIGNAL_INDEX_RADIO_KEYING_CTS);
            }
           else
            { dwNewSignalStates &= ~(1 << KEYER_SIGNAL_INDEX_RADIO_KEYING_CTS);
            }
           if( dwModemStatus & MS_DSR_ON )
            { // > "The DSR (data-set-ready) signal is on." ->
              dwNewSignalStates |= (1 << KEYER_SIGNAL_INDEX_RADIO_KEYING_DSR);
            }
           else
            { dwNewSignalStates &= ~(1 << KEYER_SIGNAL_INDEX_RADIO_KEYING_DSR);
            }
           if( dwModemStatus & MS_RING_ON )
            { // > "The ring indicator signal is on." ->
              dwNewSignalStates |= (1 << KEYER_SIGNAL_INDEX_RADIO_KEYING_RI);
            }
           else
            { dwNewSignalStates &= ~(1 << KEYER_SIGNAL_INDEX_RADIO_KEYING_RI);
            }
           if( dwModemStatus & MS_RLSD_ON )
            { // > "The RLSD (receive-line-signal-detect) signal is on."
              // (a more common name for this funny thing is "DCD" = "Data Carrier Detect")
              dwNewSignalStates |= (1 << KEYER_SIGNAL_INDEX_RADIO_KEYING_DCD);
            }
           else
            { dwNewSignalStates &= ~(1 << KEYER_SIGNAL_INDEX_RADIO_KEYING_DCD);
            }
         } // end if < GetCommModemStatus() successful > ?
      }   // end if( Keyer_hComPortRadioKeyingAndControl != INVALID_HANDLE_VALUE ) ?
     Keyer_UpdateSpeedTestResult( KEYER_SPEEDTEST_POLL_INPUTS, TIM_ReadStopwatch_us( &sw_SpeedTest ) );

     // Use the PC keyboard's SHIFT and CONTROL key as replacement for a paddle ?
     if( CwKeyer_Config.fKeyViaShiftAndControl )
      {
        TIM_StartStopwatch( &sw_SpeedTest );  // measure time spent in the following calls:
        if( GetAsyncKeyState( VK_SHIFT ) & ( 1 << 15) )
         { dwNewSignalStates |= (1 << KEYER_SIGNAL_INDEX_DASH_INPUT); // alternative input, here from the PC keyboard's SHIFT key
         }
        if( GetAsyncKeyState( VK_CONTROL ) & ( 1 << 15) )
         { dwNewSignalStates |= (1 << KEYER_SIGNAL_INDEX_DOT_INPUT); // alternative input, here from the PC keyboard's CONTROL key
         }
        Keyer_UpdateSpeedTestResult( KEYER_SPEEDTEST_POLL_KEYBRD, TIM_ReadStopwatch_us( &sw_SpeedTest ) );
      } // end if( CwKeyer_Config.fKeyViaShiftAndControl ) ?
     Keyer_dwCurrentSignalStates = dwNewSignalStates;
     HERE_I_AM__KEYER();

     // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
     // A bit of 'digital logic' for the keyer's CONFIGURABLE inputs ..
     // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
     fDigitalInputState = CwKeyer_GetDigitalInput( CwKeyer_Config.iDotInput );
#   if( SWI_NUM_AUX_COM_PORTS >= 3 ) // Any secondary "DOT" input from 'Additional COM Ports' set ?
     if( (AuxComPorts[0].dwDigitalSignalStates // ugly but thread-safe and fast, no thread-syncing object needed..
        | AuxComPorts[1].dwDigitalSignalStates
        | AuxComPorts[2].dwDigitalSignalStates ) & AUX_COM_DIG_SIGNAL_DOT )
      { fDigitalInputState = TRUE;
      }
#   endif // SWI_NUM_AUX_COM_PORTS ?
     if( CwKeyer_Config.fDebouncePaddleInputs )
      { fDigitalInputState = Debounce( &CwKeyer_Elbug.dotDebouncer, fDigitalInputState );
      }
     if( fDigitalInputState ) // digital input from the DOT contact active (pressed):
      { // To avoid sending e.g. an endless stream of ".-.-.-.-.-.-.-"
        // when the 'Simple Paddle Adapter' with its PULL-UP RESISTORS
        // isn't connected to the serial port, don't allow the
        // DOT-INPUT to become ACTIVE as long as we have not seen
        // a PASSIVE dot-input-signal :
        if( TIM_ReadStopwatch_ms( &sw_DotInputWatchdog ) > 10000 )
         { Keyer_fDotInputWasPassive = FALSE; // DOT-input seems to be stuck 'active' !
         }
        if( Keyer_fDotInputWasPassive // ok, the Paddle Adapter's DOT-input seems to work...
           || (!CwKeyer_Config.fKeyInputWatchdog) ) // .. or if the 'Key-input watchdog' if OFF..
         { Keyer_dwCurrentSignalStates |= (1 << KEYER_SIGNAL_INDEX_DOT_INPUT);
         }
      }
     else
      { Keyer_fDotInputWasPassive = TRUE;
        TIM_StartStopwatch( &sw_DotInputWatchdog );
      }

     // Similar 'polling from multiple inputs' also for the DASH input :
     fDigitalInputState = CwKeyer_GetDigitalInput( CwKeyer_Config.iDashInput );
#   if( SWI_NUM_AUX_COM_PORTS >= 3 ) // Any secondary "DASH" input from 'Additional COM Ports' set ?
     if( (AuxComPorts[0].dwDigitalSignalStates
        | AuxComPorts[1].dwDigitalSignalStates
        | AuxComPorts[2].dwDigitalSignalStates ) & AUX_COM_DIG_SIGNAL_DASH )
      { fDigitalInputState = TRUE;
      }
#   endif // SWI_NUM_AUX_COM_PORTS ?
     if( CwKeyer_Config.fDebouncePaddleInputs )
      { fDigitalInputState = Debounce( &CwKeyer_Elbug.dashDebouncer, fDigitalInputState );
      }
     if( fDigitalInputState ) // digital input from the DASH contact active (pressed):
      { if( TIM_ReadStopwatch_ms( &sw_DashInputWatchdog ) > 10000 )
         { Keyer_fDashInputWasPassive = FALSE; // DASH-input seems to be stuck 'active' !
         }
        if( Keyer_fDashInputWasPassive // ok, the Paddle Adapter's DASH-input seems to work...
           || (!CwKeyer_Config.fKeyInputWatchdog) ) // .. or the 'Key-input watchdog' is OFF..
         { Keyer_dwCurrentSignalStates |= (1 << KEYER_SIGNAL_INDEX_DASH_INPUT);
         }
      }
     else
      { Keyer_fDashInputWasPassive = TRUE;
        TIM_StartStopwatch( &sw_DashInputWatchdog );
      }

     HERE_I_AM__KEYER();

     // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
     // Implementation of the 'CW keyer' functionality :
     //    [in,out]  Keyer_dwCurrentSignalStates .
     // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
     wCwChar = 0;  // not the end of a character (Morse code pattern) yet
     fMorseOutput = FALSE;  // CW 'keying signal' currently passive, unless set below...
     if( CwKeyer_Gen.iState != CW_GEN_OFF )  // "playing from memory" (or from the 'Winkeyer EMULATOR') ?
      { // Playing text "from memory" is possible even without a Morse key or paddle.
        // It can be interrupted at any time by hitting any paddle contact:
        if(  ( Keyer_dwCurrentSignalStates & (1 << KEYER_SIGNAL_INDEX_DASH_INPUT) )
          || ( Keyer_dwCurrentSignalStates & (1 << KEYER_SIGNAL_INDEX_DOT_INPUT) ) )
         { // Enough of that boring "Morse from memory", the operator is back..
           CwKeyer_Gen.iState = CW_GEN_OFF;
         }
        else // neither dash- nor dot contact closed -> keep "playing" from memory
         { wCwChar = CwGen_Handler( &CwKeyer_Gen, nMicrosecondsSinceLastCall );
           // ,-----------------------------'
           // '--> "modulation output" in CwKeyer_Gen.fMorseOutput
           //  (drives the local sidetone, and UNDER SOME CONDITIONS keys the transmitter)
           if( wCwChar != 0 )
            { CwKeyer_WriteToDecoderFifo( &CwKeyer_DecoderFifo, wCwChar ); // here: TX-text emitted WHILE SENDING text *from memory* (shall also appear on the 'Debug' panel or in the single-line 'Rx/Tx/Info' edit field, even when not really "decoded")
            }
           fMorseOutput = CwKeyer_Gen.fMorseOutput;
         }
      }
     else // not "playing from memory" ..  maybe sending from a locally connected Morse key ?
     switch( CwKeyer_Config.iMorseKeyType )
      { case KEY_TYPE_PASSIVE : // no keying of the output(s), but e.g. "manual" control via CwKeyer_SetDigitalOutput()
           break;
        case KEY_TYPE_STRAIGHT: // straight key for INPUT (only one digital input)
           fMorseOutput = ( Keyer_dwCurrentSignalStates & (1<<KEYER_SIGNAL_INDEX_DOT_INPUT) )!=0;
           if( ! pCwNet->fSendingFromRxFifo )
            { iResult = StraightKeyDecoder( &CwKeyer_StraightKeyDecoder, // here: for the 'locally connected' straight key
                             fMorseOutput, nMicrosecondsSinceLastCall );
              wCwChar = (WORD)iResult;
              if( ( (iResult>0) && (iResult<=0xFF) ) || (iResult & CW_CHR_SPACE) )
               { // Append the 'morse code pattern' to a thread-safe FIFO,
                 // from where the application can retrieve it:
                 CwKeyer_WriteToDecoderFifo( &CwKeyer_DecoderFifo, wCwChar ); // here: TX-text decoded WHILE SENDING with a straight key
                 // Note: Further below, wCwChar will also be passed to
                 //       CwKeyer_CollectDataForTimingScope() !
               }
            }
           break;
        default : // external paddle, two digital inputs, "Iambic B", "Iambic A", and who-knows-what..
           CwKeyer_Elbug.cfg.iMorseKeyType = CwKeyer_Config.iMorseKeyType; // pass on the key type (e.g. "Iambic mode X") to Elbug.c
           iResult = Elbug_Handler( &CwKeyer_Elbug,
             (Keyer_dwCurrentSignalStates&(1<<KEYER_SIGNAL_INDEX_DOT_INPUT))!=0,  // [in] BOOL fDotInput
             (Keyer_dwCurrentSignalStates&(1<<KEYER_SIGNAL_INDEX_DASH_INPUT))!=0, // [in] BOOL fDashInput
             nMicrosecondsSinceLastCall ); // [in] int  nMicrosecondsSinceLastCall
           fMorseOutput = CwKeyer_Elbug.fMorseOutput; // "Morse"-carrier ON/OFF (here: from Elbug_Handler())
           // At the end of a character or on detection of an "inter-word space",
           // Elbug_Handler() returns the transmitted "Morse code pattern"
           // as specified in the definition of the CW_CHAR_..-constants
           // in Elbug.h (with a 'startbit' to indicate the number of dashes
           // and dots in the rest of the BYTE).
           wCwChar = (WORD)iResult;
           if( ( (iResult>0) && (iResult<=0xFF) ) || (iResult & CW_CHR_SPACE) )
            { // Append the 'morse code pattern' to a thread-safe FIFO,
              // from where the application can retrieve it:
              CwKeyer_WriteToDecoderFifo( &CwKeyer_DecoderFifo, wCwChar ); // here: TX-text decoded WHILE SENDING with the built-in Elbug
              // Note: Further below, wCwChar will also be passed to
              //       CwKeyer_CollectDataForTimingScope() !
            }
           break;
      } // end switch( CwKeyer_Config.iMorseKeyType )
     if( fMorseOutput ) // currently "Key Down", LOCALLY generated ?
      { if( pCwNet->RigControl.iTransmitReqst > 0 ) // ... and PTT activated from this end (manually or via Semi-BK) ...
         { CwNet_SwitchTransmittingClient( pCwNet, CWNET_LOCAL_CLIENT_INDEX ); // here: "key occupied" by the server's sysop !
         }
      }
     HERE_I_AM__KEYER();

     // Last not least, when operating AS SERVER, the on/off keying signal,
     //   fMorseOutput, may originate from a REMOTE CLIENT (the "CW network") :
     if( pCwNet->cfg.iFunctionality == CWNET_FUNC_SERVER ) // Currently operating AS SERVER ?
      { fMorseOutput |= Keyer_GetMorseOutputFromRxFifo( pCwNet );
        // Decode characters from this "remote keying input" only
        // if the TX isn't keyed LOCALLY, to avoid clashing
        // with other callers of StraightKeyDecoder() :
        //
        if( (CwKeyer_Config.iMorseKeyType==KEY_TYPE_PASSIVE) || (pCwNet->fSendingFromRxFifo) )
         { iResult = StraightKeyDecoder( &CwKeyer_StraightKeyDecoder, // here: for straight keying from a remote client
                             fMorseOutput, nMicrosecondsSinceLastCall );
           wCwChar = (WORD)iResult;
           // Also in THIS case ("straight keying signal" actually received via network),
           // allow the application (GUI) to dump the decoder output as plain text
           // on the "Debug" tab. This allows checking the CW signal by the sysop
           // even if the speed is beyond the sysop's personal limit :)
           if( ( (iResult>0) && (iResult<=0xFF) ) || (iResult & CW_CHR_SPACE) )
            { // Append the 'morse code pattern' to a thread-safe FIFO,
              // from where the application can retrieve it:
              CwKeyer_WriteToDecoderFifo( &CwKeyer_DecoderFifo, wCwChar ); // here: TX-text decoded WHILE SENDING text "on behalf of a remote client" (keying pattern decoded on the server side with the 'straight key' decoder)
            }
         } // end if < ok to pass the 'morse output' from a remote client to the STRAIGHT KEY DECODER > ?
      } // end if < pCwNet->iFunctionality == CWNET_FUNC_SERVER > ?
     HERE_I_AM__KEYER();

     if( CwKeyer_iTestMode==KEYER_TEST_MODE_OFF) // "normal operation" (keyer controls digital outputs) ?
      {
        if( CwKeyer_Elbug.fPauseTransmitter || CwKeyer_Config.fDisableTx || ( pCwNet->RigControl.iTransmitReqst <= 0 ) )
         { CwKeyer_SetDigitalOutput( CwKeyer_Config.iRadioCWKeying, FALSE );
         }
        else
         { CwKeyer_SetDigitalOutput( CwKeyer_Config.iRadioCWKeying, fMorseOutput );
         }
      }
     // As a replacement for the sidetone, the ON-OFF-KEYERING SIGNAL can be
     // routed to any OUTPUT on any "Additional COM Port", not affected by
     // flags like CwKeyer_Config.fDisableTx (etc?) :
     if( fMorseOutput ) // also store this flag in the "Process Image", where e.g. the PLC in AuxComThread() can read it:
      { Keyer_dwCurrentSignalStates |= (1<<KEYER_SIGNAL_INDEX_CW);
      }
     else
      { Keyer_dwCurrentSignalStates &= ~(1<<KEYER_SIGNAL_INDEX_CW);
      }
     HERE_I_AM__KEYER();

     // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
     // Receive / Transmit / PTT control (configurable, and more complex than you think) :
     // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
     //  * The adjustable "Tx Delay Time" shall protect the transceiver's / PA's
     //        receive / transmit switching relays, by FIRST activating the PTT,
     //        the "waiting" for the relay to switch, before emitting the
     //        'modulation'. This of course is only important for the instance
     //        directly connected to the radio.
     //        To avoid losing "the first dash or dot" when BEGINNING a
     //        transmit-over, the KEYING SIGNAL (like the voice modulation for
     //        an SSB transmitter) must pass through a delay line (future plan).
     //  * The local sidetone is NOT subject to the "TX delay" !
     //  * The adjustable "Tx Hang Time", also known as "Semi-Break-in-time".
     //    During the hang time, 'the key' still belongs to the
     //    currently transmitting station (e.g. a particular remote client).
     //  * Since 2024-09, the PTT can also be controlled "manually" (feature
     //    requested in the Remote CW Keyer user's group, e.g. to operate the
     //    PTT manually or via footswitch. Or, if the Remote CW Keyer ever
     //    supports VOICE TRANSMISSIONS e.g. for CW lessons via internet,
     //    the 'manual PTT switching input' may be connected to a microphone's
     //    PTT switch.
     //  * With CwKeyer_Config.fDisableTx = TRUE (e.g. QRM-free testing,
     //    speed adjustment while listening to the station before replying,
     //    code practicing, or just "playing with the program" to get familiar
     //    with a few features),   the manual PTT input remains active
     //    but the PTT OUTPUT to the transceiver, or to the network, is DISABLED.
     if(  (CwKeyer_Config.iManualPTTInput != KEYER_SIGNAL_INDEX_NONE ) // MANUAL PTT control (via digital *input* from e.g. a footswitch) ?
       || (pCwNet->iSetPTTFromNetwork  > 0 ) ) // MANUAL PTT control from the 'hamlib-like' "set_ptt" in CwNet.c  ?
      { if( CwKeyer_Config.iManualPTTInput != KEYER_SIGNAL_INDEX_NONE )
         { fDigitalInputState = CwKeyer_GetDigitalInput( CwKeyer_Config.iManualPTTInput );
         }
        else
         { fDigitalInputState = FALSE; // if there's no "local manual PTT control input", PTT is passive.. but that's not all:
         }
        if( pCwNet->iSetPTTFromNetwork > 0)
         { fDigitalInputState = TRUE;
         }
        if( fDigitalInputState/*here: PTT input*/ && (!CwKeyer_Config.fDisableTx) )
         { TIM_StartStopwatch( &sw_TxBusy ); // here: "TX busy" due to MANUAL PTT control
           CwKeyer_SetDigitalOutput( CwKeyer_Config.iRadioPTTControl, TRUE ); // PTT "on" (here: passed through from "manual PTT control")
           // Also set the PTT-flag in the "Process Image", which the PLC in AuxComThread() can READ:
           Keyer_dwCurrentSignalStates |= (1<<KEYER_SIGNAL_INDEX_PTT);
           // Also pass the PTT-flag on to the "Rig Control" unit (which may send it as a CAT/CI-V command):
           RigCtrl_SetTransmitRequest( &pCwNet->RigControl, TRUE ); // now TRANSMITTING from the Rig Control's point of view, here: controlled by MANUAL PTT INPUT
                 //  '--> THIS may manually control the radio's PTT via CI-V,
                 //       when there is no classic PTT control via e.g. 'RTS on the RADIO PORT'.
           if( CwKeyer_DSP.iOutputState != DSP_OUTPUT_STATE_GENERATE_SIDETONE )
            { CwDSP_SwitchOutputState( &CwKeyer_DSP, DSP_OUTPUT_STATE_GENERATE_SIDETONE );
            }
           // The "manual PTT control state" must also be passed on
           // from client to server over the network. But that doesn't happen
           // HERE (in KeyerThread()) but in CwNet.c : CwNet_OnPoll(),
           // when acting as a CLIENT, if pCwNet->RigControl.iTransmitReqst (set/cleared via RigCtrl_SetTransmitRequest)
           // differs from what has been sent from client to remote server.
           //
           // Regardless of the manual PTT-INPUT(!), the CW keyer keeps running,
           // and WILL produce a sidetone just like any modern transceiver will do
           // for 'code practice', or to adjust the speed to the calling station
           // without causing interference on air.
         }
        else // *manual PTT INPUT* currently passive, e.g. not wanting to transmit but wanting to receive ->
         { // To protect the power amplitfier and who-knows-what,
           // don't clear the 'local PTT flag' immediately. Instead,
           // wait for the configurable transmitter 'hang time'. During this time,
           // the transmitter itself will 'ramp down' the RF envelope. Only then
           // without RF from the transmitter to the PA / RF switching relays,
           // it's safe to switch the PTT-*output* back from TX to RX :
           if( TIM_ReadStopwatch_ms( &sw_TxBusy ) > CwKeyer_Config.iTxHangTime_ms )
            { // "manual PTT input" passive, TX hang time expired -> switch back from TX to RX ..
              if( CwKeyer_iTestMode==KEYER_TEST_MODE_OFF) // "normal operation" (keyer controls digital outputs) ?
               {  CwKeyer_SetDigitalOutput( CwKeyer_Config.iRadioPTTControl, FALSE ); // PTT "off" (back to receive)
                  // Also clear the PTT-flag in the "Process Image", which the PLC in AuxComThread() can READ:
                  Keyer_dwCurrentSignalStates &= ~(1<<KEYER_SIGNAL_INDEX_PTT);
               }
              if( CwKeyer_DSP.iOutputState == DSP_OUTPUT_STATE_GENERATE_SIDETONE )
               { // Let the DSP stop "producing a sidetone" on the audio output,
                 // instead switch back to "play back the RECEIVED AUDIO" :
                 CwDSP_SwitchOutputState( &CwKeyer_DSP, DSP_OUTPUT_STATE_RECEIVER_AUDIO );
               }
              RigCtrl_SetTransmitRequest( &pCwNet->RigControl, FALSE ); // stop transmitting after the TX-"hang time" has elapsed w/o activity on the "manual PTT input"
              TIM_StopStopwatch( &sw_TxBusy ); // stop the "transmit-busy" timer because we're not transmitting anymore
            } // end if < TX "hang time" (Sender-Nachlaufzeit) expired, with MANUAL PTT control >
         } // end else < manual PTT, and PTT-input currently PASSIVE >
      }
     else // pConfig->iManualPTTInput == KEYER_SIGNAL_INDEX_NONE ->
      { // "automatic" PTT, as supported by almost any of today's radios ("Break-in")
        if( fMorseOutput )  // "key down" from any of the sources listed above ?
         { TIM_StartStopwatch( &sw_TxBusy ); // here: "TX busy" due to AUTOMATIC PTT control
           if( !CwKeyer_Config.fDisableTx )  // <- that's the flag controlled e.g. on the "Keyer"-tab, "More"-button, menu item "Disable TX / only SIDETONE output".
            { CwKeyer_SetDigitalOutput( CwKeyer_Config.iRadioPTTControl, TRUE ); // PTT "on" (transmit, here: with AUTOMATIC PTT control)
              // Also set the PTT-flag in the "Process Image", which the PLC in AuxComThread() can READ:
              Keyer_dwCurrentSignalStates |= (1<<KEYER_SIGNAL_INDEX_PTT);
              // Also pass the PTT-flag on to the "Rig Control" unit (which may send it as a CAT/CI-V command):
              RigCtrl_SetTransmitRequest( &pCwNet->RigControl, TRUE ); // now TRANSMITTING from the Rig Control's point of view, here: automatic PTT on fMorseOutput ("Morse key down")
              // '--> This allows to transmit even if CwKeyer_Config.iRadioPTTControl is set to KEYER_SIGNAL_INDEX_NONE !
            }
           else // CwKeyer_Config.fDisableTx -> do NOT activate the PTT, but allow generating the SIDETONE.
            { // (the KEYER GUI will shown something like "SIM TX" in this state; see KeyerGUI_UpdateStatusIndicatorText()
            }
           if( CwKeyer_DSP.iOutputState != DSP_OUTPUT_STATE_GENERATE_SIDETONE )
            { CwDSP_SwitchOutputState( &CwKeyer_DSP, DSP_OUTPUT_STATE_GENERATE_SIDETONE );  // SIDETONE output *not* affected by CwKeyer_Config.fDisableTx, for "practicing".
            }
         }
        else // no Morse output ("key down") at the moment..
         {
           if( TIM_ReadStopwatch_ms( &sw_TxBusy ) > CwKeyer_Config.iTxHangTime_ms )
            { // Transmitter's "hang time" expired; time to switch back to RX ..
              if( CwKeyer_iTestMode==KEYER_TEST_MODE_OFF) // "normal operation" (keyer controls digital outputs) ?
               {  CwKeyer_SetDigitalOutput( CwKeyer_Config.iRadioPTTControl, FALSE ); // PTT "off" (receive)
                  // Also clear the PTT-flag in the "Process Image", which the PLC in AuxComThread() can READ:
                  Keyer_dwCurrentSignalStates &= ~(1<<KEYER_SIGNAL_INDEX_PTT);
               }
              if( CwKeyer_DSP.iOutputState == DSP_OUTPUT_STATE_GENERATE_SIDETONE )
               { // Let the DSP stop "producing a sidetone" on the audio output,
                 // instead switch back to "play back the RECEIVED AUDIO" :
                 CwDSP_SwitchOutputState( &CwKeyer_DSP, DSP_OUTPUT_STATE_RECEIVER_AUDIO );
               }
              RigCtrl_SetTransmitRequest( &pCwNet->RigControl, FALSE ); // stop transmitting after the TX-"hang time" has elapsed w/o a 'key-down' signal (fMorseOutput=FALSE, *automatic* PTT control)
              TIM_StopStopwatch( &sw_TxBusy ); // stop the "transmit-busy" timer because we're not transmitting anymore
            } // end if < TX "hang time" (Sender-Nachlaufzeit) expired, with AUTOMATIC PTT control >
         } // end else < ! fMorseOutput >, with AUTOMATIC PTT control (aka Semi-BK)
      } // end else pConfig->iManualPTTInput == KEYER_SIGNAL_INDEX_NONE, i.e. AUTOMATIC PTT control (Semi-BK)

     if( CwKeyer_DSP.pDSW != NULL ) // audio I/O available (DirectSound wrapper) ?
      { HERE_I_AM__KEYER();
        CwDSP_UpdateSidetone( &CwKeyer_DSP, (!CwKeyer_Elbug.fPauseTransmitter) && fMorseOutput/*fKeyDown?*/ );
        // Note: The SIDETONE remains active regardless of CwKeyer_Config.fDisableTx .
        HERE_I_AM__KEYER();
      }

     // The "CW Timing Scope" formerly showed the state of the CW-keying-output.
     // To allow plotting the Morse code even when not 'transmitted'.. :
     CwKeyer_fLocalMorseOutput = fMorseOutput; // <- input for CwKeyer_CollectDataForTimingScope()
     if( fMorseOutput )
      { TIM_StartStopwatch( &CwKeyer_swMorseActivityTimer ); // -> polled by the GUI, which may indicate "[ Off-air CW ]" now
      }
     else
      { if( TIM_ReadStopwatch_ms( &CwKeyer_swMorseActivityTimer ) > 2000 )
         { // the "CW practicing without PTT" seems to have ended..
           TIM_StopStopwatch( &CwKeyer_swMorseActivityTimer );
         }
      }


     // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
     // Common 'drive' for outputs like the key-supply-voltage (pull-up resistors),
     //        sidetone, delay line for the radio's CW keying, and PTT control:
     // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
     if( (CwKeyer_Config.iKeySupply != KEYER_SIGNAL_INDEX_NONE ) // allow driving this output at all ?
       &&(CwKeyer_iTestMode==KEYER_TEST_MODE_OFF) ) // "normal operation" (keyer controls digital outputs) ?
      { CwKeyer_SetDigitalOutput( CwKeyer_Config.iKeySupply, TRUE ); // emit positive voltage on e.g. DTR
      }
     if( (CwKeyer_Config.iRadioSupply != KEYER_SIGNAL_INDEX_NONE ) // allow driving this output at all ?
       &&(CwKeyer_iTestMode==KEYER_TEST_MODE_OFF) ) // "normal operation" (keyer controls digital outputs) ?
      { CwKeyer_SetDigitalOutput( CwKeyer_Config.iRadioSupply, TRUE ); // emit positive voltage on e.g. DTR
      }


     // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
     // Switch any of the DIGITAL OUTPUTS on various ports ?
     //    [in]  Keyer_dwCurrentSignalStates, dwLastSentSignalStates .
     // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
     if( CwKeyer_fUpdateAllOutputs )
      {  CwKeyer_fUpdateAllOutputs = FALSE;
         // Kludge to invoke the galore of EscapeCommFunction()-calls further below,
         // even if a certain signal has NOT changed since the last thread loop:
         dwLastSentSignalStates = ~Keyer_dwCurrentSignalStates;
      }
     TIM_StartStopwatch( &sw_SpeedTest );
     //
     // One may be tempted to use the bulky "SetCommState()" for this purpose, but:
     // > The EscapeCommFunction function directs a specified
     // > communications device to perform an extended function.
     // > BOOL EscapeCommFunction(  HANDLE hFile,  DWORD dwFunc )
     // > hFile  [in] Handle to the communications device.
     // >             The CreateFile function returns this handle.
     // > dwFunc [in] Extended function to be performed.
     // >             This parameter can be one of the following values.
     // > CLRBREAK Restores character transmission and places
     // >   the transmission line in a nonbreak state. The CLRBREAK
     // >   extended function code is identical to the ClearCommBreak function.
     // > SETBREAK Suspends character transmission
     // >        and places the transmission line in a break state
     // >        until the ClearCommBreak function is called
     // >        (or EscapeCommFunction is called with the CLRBREAK extended
     // >        function code). The SETBREAK extended function code
     // >        is identical to the SetCommBreak function.
     // >        Note that this extended function does not flush data
     // >        that has not been transmitted.
     //  WB: BREAK on  = continuous high pegel on TXD
     //      BREAK off = TXD controlled by serial output, or low pegel (negative voltage = IDLE STATE)
     // > CLRDTR Clears the DTR (data-terminal-ready) signal.
     // > SETDTR  Sends the DTR (data-terminal-ready) signal.
     // > CLRRTS Clears the RTS (request-to-send) signal.
     // > SETRTS  Sends the RTS (request-to-send) signal.
     // > SETXOFF Causes transmission to act as if an XOFF character has been received.
     // > SETXON  Causes transmission to act as if an XON character has been received.
     // Note: With certain crappy USB<->RS-232 adapters, or crappy drivers,
     //       the "EscapeCommFunction" can be INCREDIBLY SLOW .
     if( Keyer_hComPort_IN != INVALID_HANDLE_VALUE ) // is the "Morse Key" INPUT port open ?
      {
        fResult = TRUE;  // so far, no error from EscapeCommFunction() ..
        if(  ( Keyer_dwCurrentSignalStates & (1UL << KEYER_SIGNAL_INDEX_MORSE_KEY_DTR) )
           > ( dwLastSentSignalStates      & (1UL << KEYER_SIGNAL_INDEX_MORSE_KEY_DTR) ) )
         { // must turn DTR on this serial port ON ("enable") :
           fResult &= EscapeCommFunction( Keyer_hComPort_IN, SETDTR );  // DTR high (positive voltage)
           // > "If the function succeeds, the return value is nonzero."
           dwLastSentSignalStates |= (1UL << KEYER_SIGNAL_INDEX_MORSE_KEY_DTR);
         }
        if(  ( Keyer_dwCurrentSignalStates & (1UL << KEYER_SIGNAL_INDEX_MORSE_KEY_DTR) )
           < ( dwLastSentSignalStates      & (1UL << KEYER_SIGNAL_INDEX_MORSE_KEY_DTR) ) )
         { // must turn DTR on this serial port OFF ("disable") :
           fResult &= EscapeCommFunction( Keyer_hComPort_IN, CLRDTR );  // DTR low (negative voltage)
           dwLastSentSignalStates &= ~(1UL << KEYER_SIGNAL_INDEX_MORSE_KEY_DTR);
         }
        if(  ( Keyer_dwCurrentSignalStates & (1UL << KEYER_SIGNAL_INDEX_MORSE_KEY_RTS) )
           > ( dwLastSentSignalStates      & (1UL << KEYER_SIGNAL_INDEX_MORSE_KEY_RTS) ) )
         { // must turn RTS on this serial port ON ("enable") :
           fResult &= EscapeCommFunction( Keyer_hComPort_IN, SETRTS );  // RTS high (positive voltage)
           // > "If the function succeeds, the return value is nonzero."
           dwLastSentSignalStates |= (1UL << KEYER_SIGNAL_INDEX_MORSE_KEY_RTS);
         }
        if(  ( Keyer_dwCurrentSignalStates & (1UL << KEYER_SIGNAL_INDEX_MORSE_KEY_RTS) )
           < ( dwLastSentSignalStates      & (1UL << KEYER_SIGNAL_INDEX_MORSE_KEY_RTS) ) )
         { // must turn RTS on this serial port OFF ("disable") :
           fResult &= EscapeCommFunction( Keyer_hComPort_IN, CLRRTS );  // RTS low (negative voltage)
           dwLastSentSignalStates &= ~(1UL << KEYER_SIGNAL_INDEX_MORSE_KEY_RTS);
         }
        if( (!fResult) && ( CwKeyer_sz255LastError[0] == '\0' ) )
         { sprintf(CwKeyer_sz255LastError, "EscapeCommFunction() failed on COM%d !",
                      (int)CwKeyer_Config.iComPortNumber_IN );
         }
        // To generate a low-latency sidetone on the TxD line (square wave),
        // keep the transmit-shift-register filled with a PWM pattern .
        // To avoid a lot of overhead and TX-buffer-tests, use a NON-BLOCKING
        // write operation I/O for Keyer_hComPort_IN, and the smallest
        // 'transmit buffer' size that the 'COM port API' permits.
        if( (!CwKeyer_Elbug.fPauseTransmitter) && fMorseOutput ) // sidetone "on" ?
         { HERE_I_AM__KEYER();
           if(  (CwKeyer_Config.iSidetoneOnTXD >= KEYER_SIDETONE_TXD_480_HZ)
             && (CwKeyer_Config.iSidetoneOnTXD <= KEYER_SIDETONE_TXD_960_HZ) )
            { // 'feed' the UART's transmit shift register with this pattern
              //        for a 'pulse width modulated' output on 'TXD' .
              // Notes about the above crude 'sidetone generation':
              //  * With only EIGHT data bits, and a "fixed" start- and stopbit,
              //    there's not a lot of PWM level steps between
              //    the weakest and the loudest tone (TXD pattern 0xF0) .
              //  * The RS232 "start bit" always has a POSITIVE voltage.
              //  * The "start bit" is followed by the LEAST SIGNIFICANT digit.
              //  * A logic "high" data bit emits a NEGATIVE voltage,
              //    a logic "low" data bit emits a POSITIVE voltage on TXD.
              //  * The RS232 "stop bit" has the same voltage as an "idle line":
              //      NEGATIVE voltage .. which would be a logic "high" data bit.
              switch( CwKeyer_Config.iSidetoneOnTXD )
               { case KEYER_SIDETONE_TXD_960_HZ : // 4800 baud / 5 bits = 960 Hz sidetone ->
                      sz4Temp[0] = 0xCE;
                      //                _____          _____
                      // TXD (voltage) |     |________|     |________
                      //                 2*H    3*L     2*H    3*L
                      //               :<----- TEN bit times ---->:
                      //               :  |  |  |  |  |  |  |  |  |  |
                      //             START b0 b1 b2 b3 b4 b5 b6 b7 STOP
                      break;
                 default:  // 4800 baud / 10 bits = 480 Hz sidetone ->
                      sz4Temp[0] = 0xF0;
                      //                ______________
                      // TXD (voltage) |              |______________
                      //                  5 * H            5 * L
                      //               :<----- TEN bit times ---->:
                      //               :  |  |  |  |  |  |  |  |  |  |
                      //             START b0 b1 b2 b3 b4 b5 b6 b7 STOP
                      break;
               }
              WriteFile( Keyer_hComPort_IN, // handle to "file" to write to
                      sz4Temp,   // pointer to data to write to file
                      1,         // number of bytes to write (ONLY ONE without buffering)
               &dwNumBytesWritten,  // pointer to number of bytes written
               &osWriteForMorseKeyPort );  // pointer to the windows 'overlapped' thing
               // Because Keyer_hComPort_IN has been opened with FILE_FLAG_OVERLAPPED,
               //  and with CommTimeouts.WriteTotalTimeoutMultiplier=0,
               //        +  CommTimeouts.WriteTotalTimeoutConstant = 0,
               // the above WriteFile() will NEVER block, and if a previous
               // 'Write' is still spending, it will simply FAIL instead of
               // filling an internal transmit buffer.
               // With this (and the software 'Debouncer' off',
               //   the sidetone on TXD started 5 ms after pressing the STRAIGHT KEY,
               //                  and ended 3..5 ms after releasing it.
               // Note: Calling WriteFile() with TWO bytes caused the serial
               //       port's internall buffer to fill, so that when releasing
               //       the key, the tone was TOO LONG due to the 'accumulated'
               //       bytes in the buffer. With 10 bits / 4800 bits/sec = 2.08333 ms / byte,
               //       but the keyer thread filling in a new byte every 2.0 ms (+/- jitter),
               //       the effect of a backlash in the buffer is neglectable.
               //       In a microcontroller firmware, we could simple check
               //       how many bytes are still in the pipe (between application and the UART).
               //       Not possible with Windows ! See also:
               //       Experiments with serial ports, overlapped I/O, "Comm-Timeouts",
               //       and CONFIGURABLE SERIAL PORT LATENCIES (managed by Windows)
               //       in AuxComPorts.c : AuxComThread() !
               //       when writing the bytes one by one .
            }
           HERE_I_AM__KEYER();
         }  // end if < fMorseOutput >
      } // end if( Keyer_hComPort_IN != INVALID_HANDLE_VALUE)
     if( Keyer_hComPortRadioKeyingAndControl != INVALID_HANDLE_VALUE ) // is the "Radio Keying" / OUTPUT port open ?
      { // Similar as above for the OTHER (optional) serial port ...
        // .. but in most cases, there's nothing to POLL (read) on this port,
        //    thus don't waste precious time calling GetCommModemStatus() from here
        //    if none of the "digital inputs" (DCD, DSR, CTS, RI) are used !
        //
        fResult = TRUE;  // so far, no error from EscapeCommFunction() ..
        HERE_I_AM__KEYER();
        if(  ( Keyer_dwCurrentSignalStates & (1UL << KEYER_SIGNAL_INDEX_RADIO_KEYING_DTR) )
           > ( dwLastSentSignalStates      & (1UL << KEYER_SIGNAL_INDEX_RADIO_KEYING_DTR) ) )
         { // must turn DTR on this serial port ON ("enable") :
           if( EscapeCommFunction( Keyer_hComPortRadioKeyingAndControl, SETDTR ) ) // DTR high (positive voltage)
            { // > "If the function succeeds, the return value is nonzero."
              dwLastSentSignalStates |= (1UL << KEYER_SIGNAL_INDEX_RADIO_KEYING_DTR);
            }
           else // EscapeCommFunction() failed ... WHY ?
            { fResult = FALSE;
              // In 2025-01, "EscapeCommFunction()" is used as an indicator for
              //             TROUBLE WITH THE SERIAL PORT, for example like
              //             a playful cat pulling the USB cable out of the plug.
              // The MS documentation said about EscapeCommFunction() :
              //   > If the function succeeds, the return value is nonzero.
              //   > If the function fails, the return value is zero.
              //   > To get extended error information, call GetLastError.
              // Ah. Here we are again. Instead of returning an e.g. negative ERROR CODE,
              // EscapeCommFunction() returns a stupid BOOL, and we need to call another funny
              // function that by some black voodo magic 'remembers' what has gone wrong.
              KeyerThread_CheckLastErrorAfterTroubleWithSerialPort(
                  &Keyer_hComPortRadioKeyingAndControl, "set DTR" );
            }
         }
        if(  ( Keyer_dwCurrentSignalStates & (1UL << KEYER_SIGNAL_INDEX_RADIO_KEYING_DTR) )
           < ( dwLastSentSignalStates      & (1UL << KEYER_SIGNAL_INDEX_RADIO_KEYING_DTR) ) )
         { // must turn DTR on this serial port OFF ("disable") :
           if( EscapeCommFunction( Keyer_hComPortRadioKeyingAndControl, CLRDTR ) ) // DTR low (negative voltage)
            { dwLastSentSignalStates &= ~(1UL << KEYER_SIGNAL_INDEX_RADIO_KEYING_DTR);
            }
           else // EscapeCommFunction() failed ... again, WHY ?
            { fResult = FALSE;
              KeyerThread_CheckLastErrorAfterTroubleWithSerialPort(
                  &Keyer_hComPortRadioKeyingAndControl, "clear DTR" );
            }
         }
        if(  ( Keyer_dwCurrentSignalStates & (1UL << KEYER_SIGNAL_INDEX_RADIO_KEYING_RTS) )
           > ( dwLastSentSignalStates      & (1UL << KEYER_SIGNAL_INDEX_RADIO_KEYING_RTS) ) )
         { // must turn RTS on this serial port ON ("enable") :
           if( EscapeCommFunction( Keyer_hComPortRadioKeyingAndControl, SETRTS ) ) // RTS high (positive voltage)
            { // > "If the function succeeds, the return value is nonzero."
              dwLastSentSignalStates |= (1UL << KEYER_SIGNAL_INDEX_RADIO_KEYING_RTS);
            }
           else // EscapeCommFunction() failed ... again, WHY ?
            { fResult = FALSE;
              KeyerThread_CheckLastErrorAfterTroubleWithSerialPort(
                  &Keyer_hComPortRadioKeyingAndControl, "set RTS" );
            }
         }
        if(  ( Keyer_dwCurrentSignalStates & (1UL << KEYER_SIGNAL_INDEX_RADIO_KEYING_RTS) )
           < ( dwLastSentSignalStates      & (1UL << KEYER_SIGNAL_INDEX_RADIO_KEYING_RTS) ) )
         { // must turn RTS on this serial port OFF ("disable") :
           if( EscapeCommFunction( Keyer_hComPortRadioKeyingAndControl, CLRRTS ) ) // RTS low (negative voltage)
            { dwLastSentSignalStates &= ~(1UL << KEYER_SIGNAL_INDEX_RADIO_KEYING_RTS);
            }
           else // EscapeCommFunction() failed ... again, WHY ?
            { fResult = FALSE;
              KeyerThread_CheckLastErrorAfterTroubleWithSerialPort(
                  &Keyer_hComPortRadioKeyingAndControl, "clear RTS" );
            }
         }
        if( (!fResult) && ( CwKeyer_sz255LastError[0] == '\0' ) )
         { sprintf(CwKeyer_sz255LastError, "EscapeCommFunction() failed on COM%d !",
                      (int)CwKeyer_Config.iRadioKeyingAndControlPort );
         }
        HERE_I_AM__KEYER();
        // Regardless if the "Radio Keying / Radio Control Port" is really used
        // to CONTROL the radio (e.g. via CI-V protocol) or not, drain the port's
        // RECEIVE buffer here, and place the result in a thread-safe, lock-free,
        // circular FIFO:
        dwNumBytesRead = 0;
        if( ! fReadFromRadioPortPending )
         { if( /*BOOL*/ReadFile( Keyer_hComPortRadioKeyingAndControl,
                                 b256RxBuffer, sizeof(b256RxBuffer), &dwNumBytesRead,
                                 &osReadForRadioPort ) )
           { // > If hFile is opened with FILE_FLAG_OVERLAPPED, the lpOverlapped
             // > parameter must point to a valid and unique OVERLAPPED structure,
             // > otherwise the function can incorrectly report that the read operation is complete.
             // Arrived here ? Surprise surprise; despite the OVERLAPPED thingy,
             // ReadFile seems to have completed IMMEDIATELY,
             // so take dwNumBytesRead for granted, and PROCESS those bytes
             // further below (after also checking for an overlapped I/O completion)
           }
          else // ReadFile() returned FALSE : We're optimistic and guess the operation
           { // is PENDING .. which is not an error.
             fReadFromRadioPortPending = TRUE;
             KeyerThread_CheckLastErrorAfterTroubleWithSerialPort(
                 &Keyer_hComPortRadioKeyingAndControl, "ReadFile" );
           }
         }
        else // a ReadFile() on the Radio-Keying / Radio-Control port is ALREADY PENDING ->
         { if( /*BOOL*/GetOverlappedResult( Keyer_hComPortRadioKeyingAndControl,
                &osReadForRadioPort, &dwNumBytesRead, FALSE/* do NOT wait*/ ) )
            { // GetOverlappedResult() claimed success for the previous ReadFile(),
              // which doesn't necessarily mean we really RECEIVED something !
              fReadFromRadioPortPending = FALSE; // all we know it ReadFile() isn't PENDING anymore !
              // Regardless of dwNumBytesRead was set HERE or immediately
              // in ReadFile(), whatever was received will be processed further below.
              // Due to the "commtimeouts", often got here with the overlapped I/O
              // "completed", but NOTHING received. That's a feature, not a bug.
              // Guess what happened with ReadFile() / GetOverlappedResult()
              //            after pulling out the USB cable ?
              // NOTHING !
            }
           else // GetOverlappedResult( Keyer_hComPortRadioKeyingAndControl ..) failed...
            { // Maybe THIS is a way to reliably detect a broken-down USB connection ?
              KeyerThread_CheckLastErrorAfterTroubleWithSerialPort(
                  &Keyer_hComPortRadioKeyingAndControl, "WaitForRead" );
            }
         } // end if( fReadFromRadioPortPending ) ?
        if( dwNumBytesRead > 0 ) // received something on the "Radio-Keying / Radio-Control Port" ?
         { // For example, when turning an IC-7300's VFO knob, got here with
           // dwNumBytesRead = 11 and an "unsolicited CI-V message" in b256RxBuffer:
           //  0xFE 0xFE 0x00 0x94 0x00 0x50 0x77 0x03 0x07 0x00 0xFD
           //  |_______|                                         |__|
           //   preamble  ... see RigControl.c for details ...  postamble
           CFIFO_Write( &Keyer_RadioControlPortRxFifo.fifo, b256RxBuffer, (int)dwNumBytesRead, DSW_ReadHighResTimestamp_s() );
           // ,----------|__________________________|
           // '--> Drained in the main thread, and fed into RigControl.c, along with the timestamp.
         }
        // Use ANOTHER thread-safe, circular FIFO also to TRANSMIT DATA
        // from RigControl.c to the 'Radio Keying- and Control Port' :
        if( ! fWriteToRadioPortPending )
         { double dblTimestampFromFIO_s;
           int nBytesToSend = CFIFO_GetNumBytesReadable( &Keyer_RadioControlPortTxFifo.fifo );
           if( nBytesToSend > sizeof(b256TxBuffer) )
            {  nBytesToSend = sizeof(b256TxBuffer);
            }
           if( nBytesToSend > 0 )
            {  nBytesToSend = CFIFO_Read( &Keyer_RadioControlPortTxFifo.fifo, b256TxBuffer, nBytesToSend, &dblTimestampFromFIO_s );
            } // end if( nBytesToSend > 0 )
           if( nBytesToSend > 0 )  // ok, REALLY got something to send on the 'radio control port' ..
            { if( WriteFile( Keyer_hComPortRadioKeyingAndControl,
                       b256TxBuffer, // pointer to data to write to file
                       nBytesToSend, // number of bytes to write
                 &dwNumBytesWrittenToRadioPort, // pointer to number of bytes written
                 &osWriteForRadioPort) ) // pointer to the windows 'overlapped' thing
               { // Arrived here ? Surprise surprise; despite the OVERLAPPED thingy,
                 // WriteFile seems to have completed IMMEDIATELY !
               }
              else // no "immediate success", so guess the WriteFile() is pending..
               { fWriteToRadioPortPending = TRUE;
               }
            }
         }
        else // fWriteToRadioPortPending -> is the above WriteFile() STILL pending ?
         { if( /*BOOL*/GetOverlappedResult( Keyer_hComPortRadioKeyingAndControl,
                &osWriteForRadioPort, &dwNumBytesWrittenToRadioPort, FALSE/* do NOT wait*/ ) )
            { // GetOverlappedResult() claimed success for the previous WriteFile(),
              // which means we can fill in more data if we have (in the next thread loop).
              fWriteToRadioPortPending = FALSE;
            }
         } // end if < previous call of WriteFile() still pending > ?
      } // end if( Keyer_hComPortRadioKeyingAndControl != INVALID_HANDLE_VALUE)
     Keyer_UpdateSpeedTestResult( KEYER_SPEEDTEST_SET_OUTPUTS, TIM_ReadStopwatch_us( &sw_SpeedTest ) );

#   if( SWI_USE_MIDI ) // try to generate a sidetone via MIDI synthesizer ?  (forget it, crappy latency..)
     if( CwKeyer_hMidiDevice != INVALID_HANDLE_VALUE )
      { if( (!CwKeyer_Elbug.fPauseTransmitter) && fMorseOutput ) // sidetone "on" ?
         { if( ! CwKeyer_fMidiToneOn )   // avoid flooding Mr MIDI with commands..
            { CwKeyer_StartMidiSidetone();
              CwKeyer_fMidiToneOn = TRUE;
            }
         }
        else // MIDI-generated sidetone SHOULD be off .. so turn it off ?
         { if( CwKeyer_fMidiToneOn )   // avoid flooding Mr MIDI with commands..
            { CwKeyer_StopMidiSidetone();
              CwKeyer_fMidiToneOn = FALSE;
            }
         }
      } // end if < handle to a MIDI device (synthesizer) valid > ?
#   endif // SWI_USE_MIDI ?

     // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
     // Send this client's 'Morse Output' as a low-bandwidth KEYING BYTESTREAM
     //                                   to the remote server ?
     //        (may be displayed in hex on the 'Debug' screen; lightred)
     //
     if( pCwNet->cfg.iFunctionality == CWNET_FUNC_CLIENT )
      { // Convert the current state of fMorseOutput into a timestamped stream
        // of bytes that CwNet.c will MULTIPLEX/merge into the TCP stream .
        // The rest happens in CwNet.c : ClientThread() -> CwNet_OnPoll(),
        // where a couple of bytes from pCwNet->MorseTxFifo will be 'multiplexed'
        // into a TCP segment, along with whatever else to send. That happens
        // every TWENTY MILLISECONDS (or so, depending on the loop time of the
        // 'network thread'), thus the need for yet another thread-safe FIFO.
        long t_us;
        int  nMillisecondsEncoded; // "rounded" to whatever can be encoded in 7 bits
        if( fMorseOutput != fMorseOutput_sent ) // here's another TRANSITION, so encode..
         { // Measure the time elapsed since the PREVIOUS transition on fMorseOutput:
           t_us = TIM_ReadStopwatch_us( &sw_MorseTxFifo );
           if( !pCwNet->fFillingTxFifo ) // not filling the Morse-TX-FIFO (and stopwatch not running yet) ?
            { t_us = 0;   // no "previous time" to encode
              TIM_StartStopwatch( &sw_MorseTxFifo );
              pCwNet->fFillingTxFifo = TRUE;
            }
           HERE_I_AM__KEYER();
           nMillisecondsEncoded = CwStream_EncodeKeyUpDownEvent( &pCwNet->MorseTxFifo, fMorseOutput,
              (t_us + 500) / 1000); // [in] : nMillisecondsToWait, rounded, not truncated
           // Avoid 'accumulated' timing errors between sender and receiver:
           // Ajust the stopwatch (sw_MorseTxFifo) by the number of milliseconds
           // (an integer) that the receiver will THEORETICALLY wait.
           // For example, intervals above 31 milliseconds can only be encoded
           // in the seven-bit time with a resolution of 4 milliseconds
           //  (more on that in the "specification" in CwStreamEnc.c).
           // To 'compensate' the slight timing errors in the NEXT element,
           // the stopwatch (sw_MorseTxFifo) will not be "adjusted" by
           // the exact value of t_us, but by nMillisecondsEncoded .
           // That way, if an sequence of dots was to be sent at crazy speed
           // of 300 WPM, each dot and gap should be
           TIM_AdjustStopwatch_ms( &sw_MorseTxFifo, nMillisecondsEncoded );
           HERE_I_AM__KEYER();
           fMorseOutput_sent = fMorseOutput;
         } // end if < signal transition on fMorseOutput > ?
        else // Even WITHOUT a signal transition on fMorseOutput, may have to send
         { // the Morse-key-state to let the remote server know the END of an over,
           // and 'hand over' the key to someone else:
           t_us = TIM_ReadStopwatch_us( &sw_MorseTxFifo );
           if( (!fMorseOutput) && (t_us>(14000L*CwKeyer_Elbug.cfg.iDotTime_ms)) )
            { // No activity since more than two "word spaces" ?
              if( pCwNet->fFillingTxFifo ) // still actively filling MorseTxFifo ? Stop now !
               { // To let the receiver know the END of an over,
                 // the CW-sending client emits a second "key up"-command (bit 7 cleared),
                 // as soon as ten dot-times(?) or 500 ms have elapsed:
                 nMillisecondsEncoded = CwStream_EncodeKeyUpDownEvent( &pCwNet->MorseTxFifo, fMorseOutput,
                   (t_us + 500) / 1000); // [in] : nMillisecondsToWait, rounded, not truncated
                 TIM_AdjustStopwatch_ms( &sw_MorseTxFifo, nMillisecondsEncoded );
                 pCwNet->fFillingTxFifo = FALSE; // signalled the END of an over ->
                 // no need to actively fill the FIFO anymore,
                 // until the next transition on fMorseOutput.
               } // end if( pCwNet->fFillingTxFifo )
            }   // end if < no transition on fMorseOutput for a "long time" > ?
         }     // end of < no transition on fMorseOutput > ?
      }       // end if( pCwNet->iFunctionality == CWNET_FUNC_CLIENT )

     // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
     // At the end of the processing loop:
     //    Every 2 milliseconds, take a snapshot of the most important
     //    'process data' for the timing scope
     // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
     if( (! CwKeyer_Elbug.fPauseTransmitter) && (!CwKeyer_TimingScope.fPaused) )
      { HERE_I_AM__KEYER();
        CwKeyer_CollectDataForTimingScope( &CwKeyer_TimingScope, wCwChar );
        HERE_I_AM__KEYER();
      }

   } // end while < thread loop >

  HERE_I_AM__KEYER();

  // Close the overlapped event to avoid handle leaks.
  CloseHandle(osWriteForMorseKeyPort.hEvent);
  CloseHandle(osWriteForRadioPort.hEvent);
  CloseHandle(osReadForRadioPort.hEvent);


  // If the program ever gets here, the thread has "politely" terminated itself,
  //                    or someone has pulled the emergency brake .
  CwKeyer_iThreadStatus = KEYER_THREAD_STATUS_TERMINATED;
  HERE_I_AM__KEYER();
  ExitThread( iThreadExitCode ); // exit code for this thread
  HERE_I_AM__KEYER();
  return iThreadExitCode; // will this ever be reached after "ExitThread" ?
} // end KeyerThread()
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
void Keyer_UpdateSpeedTestResult( // .. for development and software test ..
             int iTestItem,     // [in] e.g. KEYER_SPEEDTEST_POLL_KEYBOARD
             int nMicroseconds ) // [in] new test result in microseconds
  // Note: Similar _UpdateSpeedTestResult() incarnations exist in
  //        AuxComPorts.c, CwText.c, and KeyerThread.c. The latter is even
  //        instantiatable for an ARRAY of 'worker threads' using serial ports.
{
  if( nMicroseconds >= CwKeyer_iSpeedTestPeaks_us[iTestItem] )
   {  CwKeyer_iSpeedTestPeaks_us[iTestItem] = nMicroseconds;
      // Note: A new "peak- and average detection" begins
      //       when clicking "Report Test Results (on the 'debug' tab)" in the GUI
      //  -> CwKeyer_ResetSpeedTestResults()
   }
  CwKeyer_i32SpeedTestSums_us[iTestItem] += nMicroseconds;
  ++CwKeyer_i32SpeedTestCounts_us[iTestItem];

} // end Keyer_UpdateSpeedTestResult()

//---------------------------------------------------------------------------
int  CwKeyer_GetSpeedTestAverage_us(
              int iTestItem ) // [in] e.g. KEYER_SPEEDTEST_POLL_KEYBOARD
{ long i32Divisor = CwKeyer_i32SpeedTestCounts_us[ iTestItem ];
  if( i32Divisor > 0 )
   { return (int)( CwKeyer_i32SpeedTestSums_us[iTestItem] / i32Divisor );
   }
  else // no valid data for this 'item' -> say the average time spent for it was ZERO
   { return 0;
   }
} // end CwKeyer_GetSpeedTestAverage_us()

//---------------------------------------------------------------------------
void CwKeyer_ResetSpeedTestResults(void) // begins a new peak detection
{
  memset( (void*)CwKeyer_iSpeedTestPeaks_us,   0, sizeof(CwKeyer_iSpeedTestPeaks_us) );
  memset( (void*)CwKeyer_i32SpeedTestSums_us,  0, sizeof(CwKeyer_i32SpeedTestSums_us)  );
  memset( (void*)CwKeyer_i32SpeedTestCounts_us,0, sizeof(CwKeyer_i32SpeedTestCounts_us));
} // end CwKeyer_ResetSpeedTestResults()

//---------------------------------------------------------------------------
BOOL Keyer_GetMorseOutputFromRxFifo(  // -> TRUE = "key down",  FALSE = "key up"
        T_CwNet *pCwNet ) // [in] pCwNet->MorseRxFifo (drained here, filled in CwNet.c)
  // Periodically called from KeyerThread() to drain the RX-FIFO for
  //   remote CW keying, and convert that back into the on-off-keyed signal.
  // Sounds trivial but isn't, to avoid running out of 'transmit data'
  //        from pCwNet->MorseRxFifo if the network latency suddenly
  //        increases (aka lots of jitter on the LATENCY), as detailed below.
{
  int  t_ms, n_ms_available, n_ms_to_wait;
  T_CW_KEYING_FIFO_ELEMENT *pElem; // <- contains bCmd (a single byte) but also .i32TimeOfReception_ms (for latency compensation)
  int iLatency_ms = CwNet_GetLatencyForRemoteClient_ms( pCwNet, pCwNet->iTransmittingClient );
  long i32HighResTime_ms = TIM_ReadHighResTimer_ms();

  // The 'automatic detection' of the network latency was sometimes unreliable,
  // so the WORST-CASE EXPECTED network latency (including jitter) can be con-
  // figured in the 'Settings' menu / 'Network Latency'. Use whatever is LARGER:
  if( iLatency_ms < pCwNet->cfg.iNetworkLatency_ms ) // don't use the MEASURED latency..
   {  iLatency_ms = pCwNet->cfg.iNetworkLatency_ms;  // but the CONFIGURED value
   }
  //
  if( iLatency_ms < 50 ) // even for tests with client and server running on the same "Local Host",
   {  iLatency_ms = 50;  // use this MINUMUM latency (50 ms?)
   }


  // 2024-04-25 : Principle to deal with JITTER on the network latency:
  //
  // [in] pCwNet->MorseRxFifo : One entry (byte) per "key-up"/"key-down" event,
  //   ________________         filled in CwNet_OnReceive() .
  //  |     _____   _  |        Each entry contains the time-to-wait BEFORE
  //  | ___|     |_| |_|        emitting the new key state (bit 7 = H = "key down"),
  //  |________________|        but also the timestamp of RECEPTION .
  //  :
  //  :<Latency_ms>|         (1) <- Keying starts when either
  //  :                           CwStream_GetNumMillisecondsBufferedInFifo()
  //  :                           indicates at least <iLatency_ms> ms of data
  //  :                           are available, or
  //  :                       (2) when the Morse-RX-FIFO already contains
  //  :......>......              the "End-Of-Transmission" code .
  //               :_____________________________
  //               |     _____   _   ____   _    |
  //               | ___|     |_| |_|    |_| |___|  "delayed" emission
  //               |_____________________________|
  //                                             :
  //     ........................................:
  //     :                          \____________/
  //     :                           "new" data appended to MorseRxFifo
  //     :                           while already keying TX from FIFO.
  //     :
  //    (3) Once started, keying won't stop before MorseRxFifo is completely empty.
  //
  if( ( !pCwNet->fSendingFromRxFifo ) // NOT started yet, but RX-FIFO not empty ?
    &&(  pCwNet->MorseRxFifo.iHeadIndex != pCwNet->MorseRxFifo.iTailIndex) )
   { pElem = &pCwNet->MorseRxFifo.elem[ pCwNet->MorseRxFifo.iTailIndex ];
     pCwNet->MorseRxFifo.iTailIndex = (pCwNet->MorseRxFifo.iTailIndex+1) & (CW_KEYING_FIFO_SIZE-1);  // <- new, circular wrapped TAIL index
     pCwNet->fSendingFromRxFifo = TRUE;  // START KEYING from the RX-FIFO
     pCwNet->fCurrentRemotelyReceivedKeyDownState = FALSE; // CW carrier still off.. must wait until enough patterns have been received depending on the network latency
     TIM_StartStopwatch( &pCwNet->sw_MorseRxFifo );

     if( CwKeyer_TimingScope.cfg.iChannel4Source == TIMING_SCOPE_CHANNEL_SOURCE_KEYING_FIFO_USAGE )
      { // the first entry in the "MorseRxFifo" may disappear from there so quicky
        // that CwKeyer_CollectDataForTimingScope() won't see the short 'pulse'
        // on the "Morse Keying FIFO Usage".  So, if that value is shown on 'Channel 4',
        // trigger a new scope sweep when pCwNet->fSendingFromRxFifo changes from FALSE to TRUE:
        CwKeyer_TriggerTimingScope( &CwKeyer_TimingScope );
      }

     // Got THE FIRST 'key up/down element' in the RX FIFO -> decode it:
     CwStream_DecodeKeyUpDownEvent( pElem, // [in] element in the 'Morse Rx Fifo'
        &pCwNet->fNextRemotelyReceivedKeyDownState, // [out] next "key down" / "key up" state to emit
        &n_ms_to_wait); // [out] number of milliseconds to wait BEFORE(!) emitting the above state.
        // > ZERO means "this is the begin of a new OVER, or at least a new WORD" .
     if( n_ms_to_wait <= 0 ) // do NOT wait before emitting the new key up/down state ?
      {  n_ms_to_wait = 0;
      }
     n_ms_to_wait += iLatency_ms; // add the NETWORK LATENCY to the "number of milliseconds BEFORE emitting the new key-up/down-state
     TIM_AdjustStopwatch_ms( &pCwNet->sw_MorseRxFifo, n_ms_to_wait );
              // '--> Adjusts an already running stopwatch (with TIM_StartStopwatch)
              //      as if it was started "<nMilliseconds> LATER" than it really was,
              //      without stopping and restarting it.
   } // end if <keying "from RX FIFO" not started yet, but FIFO non-empty, and "time to start" >
  if( pCwNet->fSendingFromRxFifo )
   { // Time to pull the next "key-up" / "key-down" command from pCwNet->MorseRxFifo,
     //         along with the element's length in milliseconds ?
     t_ms = TIM_ReadStopwatch_ms( &pCwNet->sw_MorseRxFifo );
     // '-- As long as the time for the currently sent element hasn't expired,
     //     t_ms will be negative, due to the adjustment via TIM_AdjustStopwatch_ms().
     //  When t_ms gets ZERO or even POSITIVE, it's time to pull the next
     //     key-up/key-down command from pCwNet->MorseRxFifo :
     if( t_ms >= 0 )  // time for action ..
      { pCwNet->fCurrentRemotelyReceivedKeyDownState = pCwNet->fNextRemotelyReceivedKeyDownState;
        if( pCwNet->MorseRxFifo.iHeadIndex != pCwNet->MorseRxFifo.iTailIndex )
         { // Morse-RX-FIFO not empty, so remote keying continues ..
           if( t_ms > 2 ) // oops.. keyer thread badly "out of sync" (missed over 2 milliseconds) ?
            { TIM_StartStopwatch( &pCwNet->sw_MorseRxFifo ); // re-synchronize as if keying just started
            }

           // Got another 'key up/down element' in the RX FIFO -> decode it:
           pElem = &pCwNet->MorseRxFifo.elem[ pCwNet->MorseRxFifo.iTailIndex ];
           pCwNet->MorseRxFifo.iTailIndex = (pCwNet->MorseRxFifo.iTailIndex+1) & (CW_KEYING_FIFO_SIZE-1);  // <- new, circular wrapped TAIL index
           // (at this point, the Fifo-WRITER-thread may already fill in another item)
           CwStream_DecodeKeyUpDownEvent( pElem, // [in] element in the 'Morse Rx Fifo'
              &pCwNet->fNextRemotelyReceivedKeyDownState, // [out] next "key down" / "key up" state to emit
              &n_ms_to_wait); // [out] number of milliseconds to wait BEFORE emitting the above state.
              // > ZERO means "this is the begin of a new OVER, or at least a new WORD" .
           if( n_ms_to_wait <= 0 ) // do NOT wait before emitting the new key up/down state ?
            { // (e.g. the begin of a new 'over', when there is no PREVIOUS element length to wait for,
              //  with T_CW_KEYING_FIFO_ELEMENT.bCmd = 0x80 = "key DOWN immediately",
              //  followed by the NEXT bCmd = 0x23 to send a DOT at 25 WPM. )
              TIM_StartStopwatch( &pCwNet->sw_MorseRxFifo );
              if( pCwNet->fCurrentRemotelyReceivedKeyDownState==FALSE )
               { // Key still UP, BEGIN of a new "over" : In this case, adjust
                 // the delay (time BEFORE emitting fNextRemotelyReceivedKeyDownState)
                 // to begin the next element after <iLatency_ms> :
                 t_ms = (int)( i32HighResTime_ms - pElem->i32TimeOfReception_ms );
                 // '--> Current "age" of the pending command in milliseconds.
                 if( (t_ms<0) || (t_ms>iLatency_ms) ) // reject garbage, avoid waiting endlessly..
                  {   t_ms=0;
                  }
                 n_ms_to_wait = iLatency_ms - t_ms;
               }
              else // pCwNet->fCurrentRemotelyReceivedKeyDownState==TRUE ->
               { n_ms_to_wait = 0;
               }
            }
           // Prepare emitting the next Morse code element (dash,dot,gap)
           // exactly <n_ms_to_wait> after the PREVIOUS element has ended (no "slip"):
           TIM_AdjustStopwatch_ms( &pCwNet->sw_MorseRxFifo, n_ms_to_wait );
              // '--> Adjusts an already running stopwatch (with TIM_StartStopwatch)
              //      as if it was started "<nMilliseconds> LATER" than it really was,
              //      without stopping and restarting it.
         } // end if < Morse-RX-FIFO not empty >
        else // nothing in pCwNet->MorseRxFifo this time -> end of an "over" ?
         { // If nothing arrives for e.g. a second, stop keying :
           if( t_ms > CwKeyer_Config.iTxHangTime_ms ) // transmitter's "hang time" expired ?
            { pCwNet->fCurrentRemotelyReceivedKeyDownState = pCwNet->fNextRemotelyReceivedKeyDownState = FALSE;
              pCwNet->fSendingFromRxFifo = FALSE;
              TIM_StartStopwatch( &pCwNet->sw_MorseRxFifo );
            }
         }
      } // end if < time to "consume" the NEXT on/off switching command > ?
   }   // end if( pCwNet->fSendingFromRxFifo )
  return pCwNet->fCurrentRemotelyReceivedKeyDownState; // -> TRUE = "key down",  FALSE = "key up"
} // end Keyer_GetMorseOutputFromRxFifo()


