// File:  C:\cbproj\Remote_CW_Keyer\JS_WebAudio.c
// Date:  2024-02-16
// Author:  Wolfgang Buescher, DL4YHF .
// Purpose: Experimental audio stream player using HTML5 and its 'Web Audio API'.
//          Format: Javascript embedded in a C string,
//                  to DOCUMENT / COMMENT what we're doing without the need
//                  to send those COMMENTS along with the Javascript itself.
// Based on an example from
// developer.mozilla.org/en-US/docs/Web/API/Web_Audio_API/Using_Web_Audio_API
//
// Introduction (from the above):
// > The Web Audio API handles audio operations inside an audio context, 
// > and has been designed to allow modular routing. Basic audio operations 
// > are performed with audio nodes, which are linked together to form an 
// > audio routing graph. You have input nodes, which are the source of the 
// > sounds you are manipulating, modification nodes that change those sounds
// > as desired, and output nodes (destinations), which allow you to save 
// > or hear those sounds.
// >
// 
const char JS_WebAudio[] = // Javascript to play 'Web Audio' ..
  // > Audio context
  // > To be able to do anything with the Web Audio API, we need to create 
  // > an instance of the audio context. This then gives us access to all 
  // > the features and functionality of the API.
  // > // for legacy browsers ...
"const AudioContext = window.AudioContext || window.webkitAudioContext;\r\n"
"const audioContext = new AudioContext();\r\n"
  // > So what's going on when we do this? A BaseAudioContext is created for us
  // > automatically and extended to an online audio context.
  // > We'll want this because we're looking to play live sound.
  //
  // > Loading sound
  // > Now, the audio context we've created needs some sound to play through it.
  // > There are a few ways to do this with the API. Let's begin with a
  // > simple method - We'll expose the song on the page using an <audio> element.
  // >    HTML: <audio src="myCoolTrack.mp3"></audio>
  //  (WB: Needs to replaced with a Javascript-based 'audio stream receiver' later,
  //       using Web-Sockets and all that ugly stuff.
  //       Directly letting the <audio> element receive and decode
  //       e.g. "/LiveAudio.ogg" resulted in the same incredibly large
  //       latency (~~ 20 seconds) as without the Web Audio API at all.)
  // > To use all the nice things we get with the Web Audio API, we need to grab
  // > the source from this element and pipe it into the context we have created.
  // > Lucky for us there's a method that allows us to do just that -
  // >   AudioContext.createMediaElementSource:
  // get the audio element
"const audioElement = document.querySelector(\"audio\");\r\n"
  // pass it into the audio context
"const track = audioContext.createMediaElementSource(audioElement);\r\n"
  // > Controlling sound
  // > When playing sound on the web, it's important to allow the user to
  // > control it. Depending on the use case, there's a myriad of options(..).
  // > Controlling sound programmatically from JavaScript code is covered by
  // > browsers' autoplay support policies, as such is likely to be blocked
  // > without permission being granted by the user (or an allowlist).
  // > Autoplay policies typically require either explicit permission
  // > or a user engagement with the page before scripts can trigger
  // > audio to play.
  // > These special requirements are in place essentially because unexpected
  // > sounds can be annoying and intrusive, and can cause accessibility problems.
  // > You can learn more about this in our article Autoplay guide for media
  // > and Web Audio APIs. ( developer.mozilla.org/en-US/docs/Web/Media/Autoplay_guide )
  // > Since our scripts are playing audio in response to a user input event
  // > (a click on a play button, for instance), we're in good shape and should
  // > have no problems from autoplay blocking. So, let's start by taking
  // > a look at our play and pause functionality.
  // > We have a play button that changes to a pause button when the track is playing:
  // HTML:
  // <button data-playing="false" role="switch" aria-checked="false">
  //    <span>Play/Pause</span>
  // </button>
  // > Before we can play our track we need to connect our audio graph from
  // > the audio source/input node to the destination.
  // > We've already created an input node by passing our audio element into
  // > the API. For the most part, you don't need to create an output node,
  // > you can just connect your other nodes to BaseAudioContext.destination,
  // > which handles the situation for you:
"track.connect(audioContext.destination);\r\n"
  // > A good way to visualize these nodes is by drawing an audio graph
  // > so you can visualize it. This is what our current audio graph looks like:
  // >   ______________                   _____________________
  // >  |              |                 |                     |
  // >  | Source       |________________\| Destination         |
  // >  | mediaElement |                /| context.destination |
  // >  |______________|                 |_____________________|
  // >
  // > Now we can add the play and pause functionality.
  // Select our play button
"const playButton = document.querySelector(\"button\");\r\n"
"\r\n"
"playButton.addEventListener(\r\n"
"  \"click\",\r\n"
"  () => {\r\n"
    // Check if context is in suspended state (autoplay policy)
"   if (audioContext.state === \"suspended\") {\r\n"
"      audioContext.resume();\r\n"
"    }\r\n"
"\r\n"
    // Play or pause track depending on state
"   if (playButton.dataset.playing === \"false\") {\r\n"
"     audioElement.play();\r\n"
"     playButton.dataset.playing = \"true\";\r\n"
"   } else if (playButton.dataset.playing === \"true\") {\r\n"
"     audioElement.pause();\r\n"
"     playButton.dataset.playing = \"false\";\r\n"
"   }\r\n"
" },\r\n"
" false,\r\n"
");\r\n"
  // > We also need to take into account what to do when the track finishes playing.
  // > Our HTMLMediaElement fires an ended event once it's finished playing,
  // > so we can listen for that and run code accordingly:
"audioElement.addEventListener(\r\n"
"  \"ended\",\r\n"
"  () => {\r\n"
"    playButton.dataset.playing = \"false\";\r\n"
"  },\r\n"
"  false,\r\n"
");\r\n"
  // > Modifying sound
  // > Let's delve into some basic modification nodes, to change the sound that
  // > we have. This is where the Web Audio API really starts to come in handy.
  // > First of all, let's change the volume. This can be done using a GainNode,
  // > (...)
  // > There are two ways you can create nodes with the Web Audio API.
  // > You can use the factory method on the context itself
  // > (e.g. audioContext.createGain()) or via a constructor of the node
  // > (e.g. new GainNode()). We'll use the factory method in our code:
"const gainNode = audioContext.createGain();\r\n"
  // > Now we have to update our audio graph from before, so the input is
  // > connected to the gain, then the gain node is connected to the destination:
"track.connect(gainNode).connect(audioContext.destination);\r\n"
  // > This will make our audio graph look like this:
  // >   ______________     _____________      _____________________
  // >  |              |   |             |    |                     |
  // >  | Source       |__\| Modification|___\| Destination         |
  // >  | mediaElement |  /|  gainNode   |   /| context.destination |
  // >  |______________|   |_____________|    |_____________________|
  // >
  // > Let's give the user control to do this - we'll use a range input:
  // HTML:
  //   <input type="range" id="volume" min="0" max="2" value="1" step="0.01" />
  // > Note: Range inputs are a really handy input type for updating values
  // >       on audio nodes. You can specify a range's values and use them
  // >       directly with the audio node's parameters.
  // > So let's grab this input's value and update the gain value when the
  // > input node has its value changed by the user:
"const volumeControl = document.querySelector(\"#volume\");\r\n"
"\r\n"
"volumeControl.addEventListener(\r\n"
"  \"input\",\r\n"
"  () => {\r\n"
"    gainNode.gain.value = volumeControl.value;\r\n"
"  },\r\n"
"  false,\r\n"
");\r\n"
  // > Note: The values of node objects (e.g. GainNode.gain) are not simple values;
  // >       they are actually objects of type AudioParam - these called parameters.
  // >       This is why we have to set GainNode.gain's value property,
  // >       rather than just setting the value on gain directly.
  // >       This enables them to be much more flexible, allowing for passing
  // >       the parameter a specific set of values to change
  // >       between over a set period of time, for example.
  // > Great, now the user can update the track's volume! The gain node is the
  // > perfect node to use if you want to add mute functionality.

"\r\n" ;  // end JS_WebAudio[]


const char HTML_WebAudio[] = // HTML template to run the above Javascript (with 'Web Audio') ..
"<!doctype html>\r\n"
"<html>\r\n"
"<head>\r\n"
"  <title>Remote CW Web Server - Main Page</title>\r\n"
"  <style type=\"text/css\">\r\n"
// "<!--#include file=\"head_styles.css\" -->\r\n"
"  </style>\r\n"
"<script type=\"text/javascript\" src=\"LiveData.js\" defer></script>\r\n"
"<script type=\"text/javascript\" src=\"WebAudio.js\" defer></script>\r\n"
 "</head>\r\n"
"<body>\r\n"
"<h2>Remote CW Web Server - Web Audio Test</h2>\r\n"
"<p>\r\n"
"This site is hosted by the <a href=\"https://www.qsl.net/dl4yhf/Remote_CW_Keyer/Remote_CW_Keyer.htm\">Remote CW Keyer</a>.<BR>\r\n"
// "<p>\r\n"
// "<img alt=\"Framebuffer\" src=\"screen1.bmp\" height=\"$SCH$\" width=\"$SCW$\">\r\n"
// "<p>\r\n"
"<audio src=\"/LiveAudio.ogg\" type=\"audio/ogg\"></audio>\r\n" // Note the absence of "controls" here !
"<button data-playing=\"false\" role=\"switch\" aria-checked=\"false\">\r\n"
"   <span>Play/Pause</span>\r\n"
"</button>\r\n"
  // > "What the Heck is ARIA ?"
  // > "ARIA is shorthand for Accessible Rich Internet Applications."
  // > The aria-checked attribute indicates whether the element is checked (true),
  // > unchecked (false), or if the checked status is indeterminate (mixed),
  // > meaning it is neither checked nor unchecked.
  // A-ha. Another 'language' plagued with counter-intuitive names and abbrvs.
"<input type=\"range\" id=\"volume\" min=\"0\" max=\"2\" value=\"1\" step=\"0.01\" />\r\n"
"\r\n"
"<div id=\"LiveData\">\r\n"
" No 'live data', waiting for Javascript.<br><br>\r\n"
"</div>\r\n"
"See also: <a href=\"links.htm\">Links</a>, <a href=\"WebAudio.htm\">Web Audio test</a>.\r\n"
"\r\n"
"</body></html>\r\n"
"\r\n" ;  // end HTML_WebAudio[]

