/*------------------------------------------------------------------*/
/* File: C:\cbproj\Remote_CW_Keyer\Inet_Tools.c                     */
/*                                                                  */
/* 'Internet tools', mainly use by the HTTP server.                 */
/* Support a few of those bazillions of encodings (like "Base64")   */
/*   and checksums / hashes (like "SHA1")                           */
/*   required for various web services, from HTTP to "WebSockets".  */
/*                                                                  */
/*                                                                  */
/* Autor: Wolfgang Buescher (DL4YHF), 2011 ... 2023                 */
/*  Used in:                                                        */
/*    *  Spectrum Lab, to retrieve the contents of a remote *.M3U,  */
/*    *     in SL's "OggVorbis"-DLL, to read Ogg/Vorbis streams,    */
/*    *     to "serve" HTTP, "WebAudio" and "WebSockets",           */
/*    *  in C:\cbproj\Remote_CW_Keyer\HttpServer.c  .               */
/*    *  and whatever comes next, to make this even more bizarre .  */
/*                                                                  */
/* Revision history (YYYY-MM-DD):                                   */
/*                                                                  */
/*  2024-01-26 : Copied a few fragments (w/o 'winsock' dependence)  */
/*                from   C:\cbproj\YHF_Tools\YHF_Inet.c             */
/*                into   C:\cbproj\Remote_CW_Keyer\Inet_Tools.c .   */
/*                                                                  */
/*  2019-01-20 :                                                    */
/*   Added a few utility functions required for WebSockets, like    */
/*      Base64 encoding and decoding (for the "Sec-WebSocket-Key"), */
/*      SHA-1 calculation (also for the WebSocket-key), etc .       */
/*                                                                  */
/*------------------------------------------------------------------*/


#include "switches.h" // project specific 'compilation switches'


/* NO VCL IN THIS UNIT !!!!! (or other BORing stuff) */
#include <string.h>
#include <windows.h> // Can we eliminate this ? Nope, at least not yet .
#include <stdarg.h>
#include <stdio.h>   // not really 'standard I/O' but vsprintf used here
#pragma hdrstop

#include "StringLib.h"  // DL4YHF's string library, with stuff like SL_ParseIPv4Address()
#include "Utilities.h"  // stuff like UTL_iWindowsVersion, UTL_iAppInstance, ShowError(), etc

#include "Inet_Tools.h" // Header file for THIS module



//------------ internal defines ------------------------------------------

   // ... none so far ...



//------------ internal data types ------------------------------------------



//------------ Test data for some encoders and hash generators --------------
   // From de.wikipedia.org/wiki/Secure_Hash_Algorithm#Beispiel-Hashes :
static const char szFranz[] = "Franz jagt im komplett verwahrlosten Taxi quer durch Bayern";
static const char szFranzBase64[] =
   "RnJhbnogamFndCBpbSBrb21wbGV0dCB2ZXJ3YWhybG9zdGVuIFRheGkgcXVlciBkdXJjaCBCYXllcm4=";
static const BYTE b20FranzSHA1[] = { 0x68, 0xac, 0x90, 0x64,
       /* 160 bit / 8bit_per_byte*/  0x95, 0x48, 0x0a, 0x34,
                                     0x04, 0xbe, 0xee, 0x48,
                                     0x74, 0xed, 0x85, 0x3a,
                                     0x03, 0x7a, 0x7a, 0x8f };


//------------ Functions ----------------------------------------------------


//---------------------------------------------------------------------------
int INET_SplitURL(
          char * pszFilenameOrURL,   // [in] filename or URL (will be DESTROYED / "chopped into pieces" !)
          T_INET_UrlParts *parts )   // [out] pointers to the 'parts'
  // Breaks up a filename (also for local files), or a URL, into pieces .
  // Loosely based on 'HTParse.c' from www.w3.org/Library/src/HTParse.c .
  //
  // [in] pszFilenameOrURL : usually a URL, with the following format :
  //               scheme://domain:port/path?query_string#fragment_id
  //      Example: http://67.207.143.181/vlf6.m3u
  // [out] T_INET_UrlParts *parts : Simple zero-terminated C-strings, example:
  //         parts->access   =
  //         parts->host     = hostname like "qsl.net" or IP string like "192.168.0.123", WITHOUT PORT NUMBER
  //         parts->absolute =
  //         parts->relative =
  //         parts->fragment =
  //         parts->iProtocol: may contain 0=unknown, or INET_PROTOCOL_HTTP, etc .
  //         parts->iPort    : port NUMBER (0=not explicitly set in INET_SplitURL)
  // Return value: ?
{
  int  iResult = 0;
  char * p;
  char * after_access = pszFilenameOrURL;
  memset(parts, 0, sizeof(T_INET_UrlParts) );

  /* Look for fragment identifier */
  if ((p = strchr(pszFilenameOrURL, '#')) != NULL)
   {
     *p++ = '\0';
     parts->pszFragment = p;
   }


  if ((p = strchr(pszFilenameOrURL, ' ')) != NULL)
   { *p++ = '\0';
   }

#ifdef __BORLANDC__
  (void)p;
#endif

  for(p=pszFilenameOrURL; *p; p++)
   {

     // Look for any whitespace. This is very bad for pipelining
     // as it makes the request invalid
     if (isspace((int) *p))
      {
        char *orig=p, *dest=p+1;
        while ((*orig++ = *dest++) != 0);  // scroll up by one character, delete the space character
        p = p-1;
      }
     if (*p=='/' || *p=='#' || *p=='?')
      { break;
      }
     if (*p==':')
      {
        *p = 0;
        parts->pszAccess = after_access; /* Scheme has been specified */

        // The combination of gcc, the "-O" flag and the HP platform is
        // unhealthy. The following three lines is a quick & dirty fix, but is
        // not recommended. Rather, turn off "-O".
        /*              after_access = p;*/
        /*              while (*after_access == 0)*/
        /*                  after_access++;*/

        after_access = p+1;

        if (0==stricmp("URL", parts->pszAccess))
         {
           parts->pszAccess = NULL;  /* Ignore IETF's URL: pre-prefix */
         }
        else
         { break;
         }
      }
   }

  // Added by DL4YHF: If there's this 'access'-thingy, examine it further:
  parts->iProtocol = INET_PROTOCOL_UNKNOWN; // may contain 0=unknown, or INET_PROTOCOL_HTTP, etc .
  if( parts->pszAccess != NULL )
   { if( stricmp(parts->pszAccess, "http" ) == 0 )
      { parts->iProtocol = INET_PROTOCOL_HTTP;
      }
     else if( stricmp(parts->pszAccess, "https" ) == 0 )
      { parts->iProtocol = INET_PROTOCOL_HTTPS;
      }
     else if( stricmp(parts->pszAccess, "file" ) == 0 )
      { parts->iProtocol = INET_PROTOCOL_FILE;  // it's a LOCAL FILE !
      }
     else if( stricmp(parts->pszAccess, "ftp" ) == 0 )
      { parts->iProtocol = INET_PROTOCOL_FTP;
      }
     else if( stricmp(parts->pszAccess, "mailto" ) == 0 )
      { parts->iProtocol = INET_PROTOCOL_FTP;
      }
     else if( stricmp(parts->pszAccess, "rawtcp" ) == 0 )
      { parts->iProtocol = INET_PROTOCOL_RAW;
      }
     else if( stricmp(parts->pszAccess, "raw" ) == 0 )
      { parts->iProtocol = INET_PROTOCOL_RAW;
      }
     else if( stricmp(parts->pszAccess, "tcp" ) == 0 )
      { parts->iProtocol = INET_PROTOCOL_TCP;
      }
   }

  p = after_access;
  if (*p=='/')
   {
     if (p[1]=='/')
      {
        parts->pszHost = p+2;   // host has been specified
        *p=0;                   // Terminate access
        p=strchr(parts->pszHost,'/');   // look for end of host name if any
        if(p)
         {
           *p=0;                                // Terminate host
           parts->pszAbsolute = p+1;   // Root has been found
         }
        // Added by WoBu: If there's a port number after the host name,
        //                separate it, and convert it into integer :
        p=strchr(parts->pszHost,':'); // look for separator between host and port
        if(p)
         { *p=0;             // Terminate host again (now before the port)
           parts->iPort = atoi(p+1);
         }

      }
     else
      {
        parts->pszAbsolute = p+1;               // Root found but no host
      }
   }
  else
   {
     parts->pszRelative = (*after_access) ? after_access : NULL; /* NULL for "" */
   }
  return iResult;
} // end INET_SplitURL()



//---------------------------------------------------------------------------
BOOL INET_IsInternetResource( const char * pszFilenameOrURL )
   // Returns TRUE when pszFilenameOrURL looks like an 'internet' resource;
   //      or FALSE when it's most likely the path+name of a 'local' file .
   // For Spectrum Lab, this function must also recognize stuff beginning with
   //     "raw://", "udp://", "tcp://" as 'internet resources' (like "http://")
   //     even though, strictly/technically, they are NOT ("raw://" isn't
   //     even a known internet protocol, but a dummy for a 'raw' audio stream
   //     on a certain (usually numeric) IP address) .
   // Since 2021-08 (first experiments to connect directly to an IC-705
   // via WLAN instead of USB), INET_IsInternetResource() is also
   // used to check for special names in Spectrum Lab's  ...
   //    * "Serial port nr" (!) for Remote Rig Control :
   //        c:\cbproj\SpecLab\CfgDlgU.cpp : CB_RadioPortNr->Text
   //    *
{
  char *cpFirstColon = strchr(pszFilenameOrURL,':');

  if(  (cpFirstColon != NULL) && (cpFirstColon > (pszFilenameOrURL+2) )
    && (cpFirstColon[1]=='/') && (cpFirstColon[2]=='/')  )
   { // something like http:// ... doesn't look like a file ...
     return TRUE;   // there's colon, but not something like "C:\blabla" ... guess it's an internet stream
   }
  else
   { return FALSE;  // not an (internet-)STREAM but a (local) FILE
   }
} // end INET_IsInternetResource()


//---------------------------------------------------------------------------
char *INET_InetProtocolToString( int iInetProtocol )
{
  switch( iInetProtocol )
   { case INET_PROTOCOL_OFF     : return "OFF";
     case INET_PROTOCOL_HTTP    : return "HTTP";
     case INET_PROTOCOL_HTTPS   : return "HTTPS";
     case INET_PROTOCOL_FTP     : return "FTP";
     case INET_PROTOCOL_MAILTO  : return "MAILTO";
     case INET_PROTOCOL_RAW     : return "RAW";  // <- often the same as TCP..
     case INET_PROTOCOL_TCP     : return "TCP";
     case INET_PROTOCOL_UDP     : return "UDP";
     case INET_PROTOCOL_COM_PORT: return "COM_PORT";
     case INET_PROTOCOL_FILE    : return "FILE";
     default: return "???";
   }
} // end INET_InetProtocolToString()


//---------------------------------------------------------------------------
int INET_ParseProtocolHostnameAndPort( // Added 2021-08 to support IC-705 controlled via LAN/WLAN
    const char* pszFilenameOrURL, // [in] e.g. "udp://IC-705:50001"
         int  *piProtocol,        // [out] INET_PROTOCOL_HTTP/TCP/UDP/RAW, , but may also be INET_PROTOCOL_COM_PORT,  etc etc etc !
         char *psz80HostName,     // [out] hostname, e.g. "IC-705". Expect up to 80 chars. May be "numeric" (dotted decimal string).
         int  *piPortNumber,      // [out] e.g. 50001 (for the example shown above)
         int  iParserOptions)     // [in] 0 = "normal" (only accept "complete URLs",
                                  //      INET_PARSER_OPTION_ALLOW_INCOMPLETE_URL, etc ..
  //
  // For comparison (to the "Pseudo-URLs" parsed here) : "URI" syntax diagram:
  //
  //    ,------,                                                         ,----,
  // >>-|scheme|--:------------------------------------------------------|path|-- .....
  //    '------'    |  ,--,                       ,----,               | '----' |
  //                '--|//|-----------------------|host|---------------'        '- ? ..
  //                   '--'  |  ,--------,  ,-, | '----' | ,-, ,----, |
  //                         '--|userinfo|--|@|-'        '-|:|-|port|-'
  //                            '--------'  '-'            '-' '----'
  // (from Wikipedia on Uniform_Resource_Identifier)
  //      * "A non-empty scheme followed by a colon", e.g.
  //           http, https, ftp, .. blah.. IN OUR PSEUDO-URL: udp, tcp, ..
  //      * "A host subcomponent, consisting of either a registered name
  //         (including but not limited to a hostname) or an IP address"...
  //      * "An optional port subcomponent preceded by a colon (:),
  //          consisting of decimal digits .."
  //
  //      * "A path is always defined for a URI, though it may be empty (zero length)
  //
  // INET_ParseProtocolHostnameAndPort() returns THE NUMBER OF CHARACTERS PARSED(*)
  //      if pszFilenameOrURL was 'recognized' as a valid 'Pseudo-URL' here,
  //      even it it was not a 'network resource' but (for example)
  //      a simple stoneage 'COM port' name like "COM21" .
  // Not as 'destructive' for the source-string as the old INET_SplitURL().
  // Note that this function does NOT use a DNS lookup to resolve the 'host name'
  // into a numeric IP address. Instead, the host name MAY BE a numeric IP !
  //
  //   (*) In older versions, the return value was just a BOOLEAN (TRUE, FALSE).
  //       Since 2023-08, it's an integer ("number of characters parsed")
  //       to simplify parsing a COMMA-SEPARATED LIST of e.g. hostnames and ports,
  //       as in C:\cbproj\SoundUtl\Sound.cpp, to send the same UDP payload
  //       to different 'receivers', e.g. :
  //       udp://localhost:6999,shack-pc:6999,7000,7001
  //       |____complete URL__|||_incomplete_| ...  ...
  //                           |  '--> only "accepted" with
  //                           |       INET_PARSER_OPTION_ALLOW_INCOMPLETE_URL.
  //                           '--------------------------,
  //    INET_ParseProtocolHostnameAndPort() stops parsing HERE !
  //    (it's the caller's "intelligence" to handle the "incomplete pseudo-URLs"
  //     that follow. Example in Sound.cpp : PrepareSocketSendersAsOutputDevices(). )
  //
  // See also ("similar parsers used in Spectrum Lab"):
  //  * C:\cbproj\SoundUtl\Sound.cpp   : Sound_GetAudioDeviceNameWithoutPrefix()
  //  * C:\cbproj\YHF_Tools\YHF_Inet.c : INET_ParseProtocolHostnameAndPort()
  //  * C:\cbproj\SpecLab\Config.cpp   : Config_GetAudioDeviceType()
  //
{ const char *cp = pszFilenameOrURL;
  char *cpDst= psz80HostName;
  const char *cpDstEndstop = psz80HostName+80;
  int iProtocol, iPortNumber, nDigitsParsed;
  long i32;
  BOOL fSyntaxOk = TRUE;
  const char *cp2 = cp;
  if( SL_SkipString_AnyCase( &cp, "http://" ) )
   { iProtocol = INET_PROTOCOL_HTTP;
   }
  else if( SL_SkipString_AnyCase( &cp, "tcp://" ) )
   { iProtocol = INET_PROTOCOL_TCP;
   }
  else if( SL_SkipString_AnyCase( &cp, "raw://" ) )
   { iProtocol = INET_PROTOCOL_RAW;
   }
  else if( SL_SkipString_AnyCase( &cp, "udp://" ) )
   { iProtocol = INET_PROTOCOL_UDP;
   }
  else if( (SL_SkipString( &cp2, "COM") ) && ( (iPortNumber=SL_ParseInteger(&cp2))>0) && (*cp2=='\0')  ) // <- CASE SENSITIVE !
   { iProtocol = INET_PROTOCOL_COM_PORT; // not "the internet" but "a COM port" !
     // NOTHING ELSE after e.g. "COM8" -> it's REALLY a "COM port", so we're through:
     if( piProtocol != NULL )    // pass back optional results ..
      { *piProtocol = iProtocol; // -> e.g. 7 = INET_PROTOCOL_UDP
      }
     if( piPortNumber != NULL )
      { *piPortNumber = iPortNumber; // -> e.g. 21 after parsing "COM21"
      }
     return TRUE;  // leave it to the caller to decide if he can "handle" SERIAL PORTS (aka "COM" ports under windows) !
   }
  else if( SL_SkipString( &cp2, "COM-LOG:") ) // <- almost the same as "COM" (serial port), but in fact a LOGFILE - details in Sound_IP.cpp !
   { iProtocol = INET_PROTOCOL_COM_PORT; // treat e.g. "COM-LOG:C:\cbproj\SpecLab\logfiles\raw_input_from_COM3.bin" like a COM port !
     if( piProtocol != NULL )    // pass back optional results ..
      { *piProtocol = iProtocol; // -> e.g. 7 = INET_PROTOCOL_UDP
      }
     if( piPortNumber != NULL )
      { *piPortNumber = iPortNumber; // -> e.g. 21 after parsing "COM21"
      }
     return TRUE;  // the real work (playing back the "COM-LOG") takes place in e.g. Sound_IP.cpp:CSound_SerialPortReaderThread()
   }
  else // none of the transport- or higher-layer protocols ..
   { iProtocol = INET_PROTOCOL_UNKNOWN;
     fSyntaxOk = FALSE;   // <- added 2023-07
   }
  // At this point, cp points to the first character of the "hostname",
  //    or the first digit of the first decimal "byte" in an IPv4 address.
  //    Copy either of those into psz80HostName. Don't try to resolve anything here.
  //    Also don't try to interpret special names like "localhost" here,
  //    because that happens later, e.g. in SL_ParseIPv4Address() .
  while( (*cp!=':') && (*cp!='\0') )
   { if( cpDst < cpDstEndstop )
      { *cpDst++ = *cp++;
      }
   }
  if( cpDst < cpDstEndstop )
   { *cpDst = '\0';
   }
  iPortNumber = -1; // dummy for "unknown port" / "any port" / "port not specified"
  if( *cp==':' ) // "hostname" seems to be followed by a PORT NUMBER : Parse it..
   { ++cp;
     nDigitsParsed = SL_str2int( cp, &i32 );
     if(nDigitsParsed>0)
      { iPortNumber = i32;
        cp += nDigitsParsed;
      }
   }
  if( piProtocol != NULL )    // pass back optional results ..
   { *piProtocol = iProtocol; // -> e.g. 7 = INET_PROTOCOL_UDP
   }
  if( piPortNumber != NULL )
   { *piPortNumber = iPortNumber; // -> e.g. 50001 for Icom's UDP-"Control"-port
   }

  if( fSyntaxOk )
   { return cp - pszFilenameOrURL; // > "returns THE NUMBER OF CHARACTERS PARSED" when successful (!)
   }
  else
   { return 0;
   }
} // INET_ParseProtocolHostnameAndPort()


//---------------------------------------------------------------------------
int INET_ReplaceCharInPath( char * pszFilenameOrURL, int iMaxLen, char cOriginal, char cReplacement )
   // Used, for example, to replace '/' in a URL (-path) by '\' for a windoze path,
   //       or vice versa. The old "Unix/Posix versus DOS/Windows" dilemma.
   // Not aware of multi-byte characters !
{
  int nCharsReplaced = 0;
  char *cpEndstop = pszFilenameOrURL + iMaxLen;
  while( (*pszFilenameOrURL != '\0') && (pszFilenameOrURL<cpEndstop) )
   { if(  *pszFilenameOrURL == cOriginal )
      {   *pszFilenameOrURL =  cReplacement;
          ++nCharsReplaced;
      }
     ++pszFilenameOrURL;
   }
  return nCharsReplaced;
} // end INET_ReplaceCharInPath()


//---------------------------------------------------------------------------
// Low-level stuff to parse / assemble "binary" stuff, as often seen in
// protocols like [m]DNS, etc.  But also used by module 'RigControl.cpp',
// because in Icom's UDP-based remote control protocols, they use an
// absolutely awful mix of LITTLE and BIG ENDIAN field of different widths.
// The following functions operate on BYTE ARRARYS, but in contrast to Qt,
// only use standard data types that are even present on simple controllers.
//  The destination or source pointer is incremented by the size of the
//  data type, e.g. INET_GetU16FromBE() increments the source pointer 
//  by two bytes.
//  THE CALLER is responsible for not exceeding the array limits,
//  which can be easily achieved by comparing the pointer against
//  an 'endstop' pointer, as in C:\cbproj\SpecLab\RigControl\RigControl.c .
//
// Note: The term "network byte order" is avoided here,
//       because NOT ALL NETWORKS, and NOT ALL 'NETWORK PROTOCOLS',
//       use the same annoying BIG ENDIAN format !
//       For an especially dreadful example, see Icom's UDP packets
//       sent/received via LAN or WLAN to/from radios like IC-9700 
//       or IC-705 : Even WITHIN THE UDP PAYLOAD, for some fields
//   they use LITTLE ENDIAN (low byte first, aka "Intel" byte order),
//   for other BIG ENDIAN (high byte first, aka "Motorola" byte order).
//   Because even reading LITTLE ENDIAN format from odd addresses may be
//   problematic for some CPUs, there are "LE" = LITTLE ENDIAN twins below.
//---------------------------------------------------------------------------
WORD INET_GetU16FromBE( BYTE **ppbSource )     // "BE" = BIG ENDIAN ...
{ WORD wResult = ((WORD)((*ppbSource)[0]) << 8 )  // most significant byte first,
               | ((WORD)((*ppbSource)[1]) << 0 ); // least significant byte last
  (*ppbSource) += 2;  // skip two bytes because we're read 16 bits
  return wResult;
} 

DWORD INET_GetU32FromBE( BYTE **ppbSource )     // "BE" = BIG ENDIAN ...
{ DWORD dwResult=((DWORD)((*ppbSource)[0]) << 24)  // most significant byte first,
               | ((DWORD)((*ppbSource)[1]) << 16)  // ...
               | ((DWORD)((*ppbSource)[2]) << 8 )  // ...
               | ((DWORD)((*ppbSource)[3]) << 0 ); // least significant byte last
  (*ppbSource) += 4;  // skip four bytes because we're read 32 bits
  return dwResult;
} 

WORD INET_GetU16FromLE( BYTE **ppbSource )     // "LE" = LITTLE ENDIAN ...
{ WORD wResult = ((WORD)((*ppbSource)[0]) << 0 )  // least significant byte first,
               | ((WORD)((*ppbSource)[1]) << 8 ); // most significant byte last
  (*ppbSource) += 2;  // skip two bytes because we're read 16 bits
  return wResult;
} 

DWORD INET_GetU32FromLE( BYTE **ppbSource )     // "LE" = LITTLE ENDIAN ...
{ DWORD dwResult=((DWORD)((*ppbSource)[0]) << 0 )  // least significant byte first,
               | ((DWORD)((*ppbSource)[1]) << 8 )  // ...
               | ((DWORD)((*ppbSource)[2]) << 16)  // ...
               | ((DWORD)((*ppbSource)[3]) << 24); // most significant byte last
  (*ppbSource) += 4;  // skip four bytes because we're read 32 bits
  return dwResult;
} 

void INET_AppendU16AsBE( BYTE **ppbDest, WORD wData ) // append 16 bit as "BE" = BIG ENDIAN, and increment dest pointer by two bytes
{ (*ppbDest)[0] = (BYTE)( wData >> 8 );   // most significant byte first ...
  (*ppbDest)[1] = (BYTE)( wData >> 0 );   // least significant byte last.
  (*ppbDest) += 2;  // increment pointer by two bytes because we're appended 16 bits
}

void INET_AppendU32AsBE( BYTE **ppbDest, DWORD dwData ) // append 32 bit as "BE" = BIG ENDIAN, and increment dest pointer by four bytes
{ (*ppbDest)[0] = (BYTE)( dwData >> 24 );  // most significant byte first ...
  (*ppbDest)[1] = (BYTE)( dwData >> 16 );
  (*ppbDest)[2] = (BYTE)( dwData >> 8 );
  (*ppbDest)[3] = (BYTE)( dwData >> 0 );   // least significant byte last.
  (*ppbDest) += 2;  // increment pointer by two bytes because we're appended 16 bits
}

void INET_AppendU16AsLE( BYTE **ppbDest, WORD wData ) // append 16 bit as "LE" = LITTLE ENDIAN, and increment dest pointer by two bytes
{ (*ppbDest)[0] = (BYTE)( wData >> 0 );   // least significant byte first ...
  (*ppbDest)[1] = (BYTE)( wData >> 8 );   // most significant byte last.
  (*ppbDest) += 2;  // increment pointer by two bytes because we're appended 16 bits
}

void INET_AppendU32AsLE( BYTE **ppbDest, DWORD dwData) // append 32 bit as "LE" = LITTLE ENDIAN, and increment dest pointer by four bytes
{ (*ppbDest)[0] = (BYTE)( dwData >> 0 );   // least significant byte first ...
  (*ppbDest)[1] = (BYTE)( dwData >> 8 );
  (*ppbDest)[2] = (BYTE)( dwData >> 16);
  (*ppbDest)[4] = (BYTE)( dwData >> 24);   // most significant byte last.
  (*ppbDest) += 4;  // increment pointer by four bytes because we're appended 32 bits
}

//---------------------------------------------------------------------------
void INET_AppendByte( BYTE **ppbDest, BYTE *pbEndstop, BYTE bValue )
{ if( *ppbDest < pbEndstop )
   { *(*ppbDest)++ = bValue;
   }
} // end INET_AppendByte()

//---------------------------------------------------------------------------
void INET_AppendBlock( BYTE **ppbDest, BYTE *pbEndstop, BYTE *pbSource, int nBytes )
{
  while( nBytes-- )
   { INET_AppendByte( ppbDest, pbEndstop, *(pbSource++) );
   }
} // end INET_AppendBlock()


//---------------------------------------------------------------------------
void INET_DumpTraffic_HexOrASCII( char *pszDest, int iMaxLen, BYTE *pbData, int nBytesToDump )
  // Converts any kind of 'traffic' (text or binary) into a string
  // that can 'safely' be printed into a visible control, logfile, or similar.
{ char *pszEndstop = pszDest + iMaxLen - 1;
  BYTE b;
  BOOL fHexMode = FALSE;

  SL_AppendPrintf( &pszDest, pszEndstop, "[%04X] ", (unsigned int)nBytesToDump );
  while( ( (nBytesToDump--) > 0 ) && ((pszEndstop-pszDest) > 8 ) )
   { b = *pbData++;
     if( (b>=32) && (b<=127) )   // "safely printable" into the RichEdit-thingy
      { if(fHexMode)
         { SL_AppendChar( &pszDest, pszEndstop, '>' );
           fHexMode = FALSE;
         }
        SL_AppendChar( &pszDest, pszEndstop, (char)b );
      }
     else
     switch( b )
      { case '\r':
           if(fHexMode)
            { SL_AppendChar( &pszDest, pszEndstop, '>' );
              fHexMode = FALSE;
            }
           SL_AppendString( &pszDest, pszEndstop, "\\r" );
           break;
        case '\n':
           if(fHexMode)
            { SL_AppendChar( &pszDest, pszEndstop, '>' );
              fHexMode = FALSE;
            }
           SL_AppendString( &pszDest, pszEndstop, "\\n" );
           break;
        default:
           if(!fHexMode)
            { SL_AppendChar( &pszDest, pszEndstop, '<' );
              fHexMode = TRUE;
            }
           SL_AppendPrintf( &pszDest, pszEndstop, "%02X", (unsigned int)b );
           break;
      }
   }
  if(fHexMode) // .. this is why we left a few character as 'spare' in the loop ..
   { SL_AppendChar( &pszDest, pszEndstop, '>' );
   }
  if( nBytesToDump>0 )  // didn't manage to "dump" the entire traffic fragment ->
   { SL_AppendString( &pszDest, pszEndstop, ".." ); // to dots = "truncated here"
   }
} // end INET_DumpTraffic_HexOrASCII()


//---------------------------------------------------------------------------
// Various code snippets added to "play" with WebSockets .. 2019-01 :
//---------------------------------------------------------------------------

static const BYTE base64_table[65] =
        "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";

//---------------------------------------------------------------------------
int INET_EncodeBase64( BYTE *pbSrc, int iSrcLen,
                       BYTE *pbDst, int iMaxDstLen,
                       int  iMaxCharsPerLine ) // [in] for optional "limited line lengths". 0 = all in ONE line of text.
  // Base64 encode
  // [in]  pbSrc[0..iSrcLen-1]    : Data (a block of bytes) to be encoded
  // [out] pbDst[0..iMaxDstLen-1] : Encoded output (base64), array with limited capacity !
  //           To be safe, use a destination with at least
  //           < iSrcLen * 4 / 3 + 4 + iSrcLen / 70 > bytes.
  //           If iMaxDstLen permits, pbDst will be terminated with 0x00.
  // Returns the length of the ENCODED "payload",  measured in bytes,
  // including '\n' characters for the line breaks, and a trailing zero-byte.
  // Returns the number of bytes (and thus CHARACTERS) placed in the output buffer,
  //  or ZERO if the destination block is too short.
  // UNLIKE THE ORIGINAL IMPLEMENTATION (found somewhere on the web years ago),
  //        this one does not append a stupid '\n' character if the output
  //        fits in a single line, because in almost any case, the CALLER
  //        had to remove that stupid character again (e.g. WebSocket) .
{
  BYTE *pbDstPos;
  BYTE *pbSrcEndstop, *pbSrcPos;
  int out_len, line_len;

  out_len = iSrcLen * 4 / 3 + 4; /* 3-byte blocks to 4-byte */
  if( iMaxCharsPerLine > 0 )
   { out_len += out_len / iMaxCharsPerLine; /* line feeds */
   }
  out_len++; // add one byte for the nul termination (to treat the result as a "C"-string)
  if( out_len > iMaxDstLen )
   { return 0; // > returns ZERO if the destination block is too short
   }

  pbSrcEndstop = pbSrc + iSrcLen;
  pbSrcPos = pbSrc;
  pbDstPos = pbDst;
  line_len = 0;
  while( (pbSrcEndstop - pbSrcPos) >= 3)
   { *pbDstPos++ = base64_table[pbSrcPos[0] >> 2];
     *pbDstPos++ = base64_table[((pbSrcPos[0] & 0x03) << 4) | (pbSrcPos[1] >> 4)];
     *pbDstPos++ = base64_table[((pbSrcPos[1] & 0x0f) << 2) | (pbSrcPos[2] >> 6)];
     *pbDstPos++ = base64_table[  pbSrcPos[2] & 0x3f];
     pbSrcPos += 3;
     line_len += 4;
     if( iMaxCharsPerLine > 0 ) // e.g. 72 ? Or 76 for MIME ? Nnngrrr !
      { if (line_len >= 72) // fortunately, the PARSER is smart enough to skip any '\n'
         {
           *pbDstPos++ = '\n';
           line_len = 0;
         }
      }
   }

  if( pbSrcPos < pbSrcEndstop ) // some remaining input, less than 3 bytes ..
   {
     *pbDstPos++ = base64_table[pbSrcPos[0] >> 2];
     if( (pbSrcEndstop - pbSrcPos) == 1)
      { *pbDstPos++ = base64_table[(pbSrcPos[0] & 0x03) << 4];
        *pbDstPos++ = '=';
      }
     else
      { *pbDstPos++ = base64_table[((pbSrcPos[0] & 0x03) << 4)
                                  | (pbSrcPos[1] >> 4)];
        *pbDstPos++ = base64_table[ (pbSrcPos[1] & 0x0f) << 2];
      }
     *pbDstPos++ = '=';
     line_len += 4;
   }

  if( iMaxCharsPerLine > 0 )
   { if (line_len) // for "nice-looking" output, line break after the last fragment
      { *pbDstPos++ = '\n';
      }
   }

  *pbDstPos = '\0';
  return pbDstPos - pbDst;
} // end INET_EncodeBase64()


//---------------------------------------------------------------------------
int INET_DecodeBase64( BYTE *pbSrc, int iSrcLen,
                       BYTE *pbDst, int iMaxDstLen )
  // Base64 decoder. Not the fastest, not elegant, but not as bulky as others,
  //                 and not poisioned by anything that isn't plain old "C".
  // [in]  pbSrc[0..iSrcLen-1]    : Data (base64) to be decoded
  // [out] pbDst[0..iMaxDstLen-1] : Decoded output, array with limited capacity !
  //           If iMaxDstLen permits, pbDst will be terminated with 0x00.
  // Returns the length of the DECODED "payload", measured in bytes.
  // Unlike many implementations floating around the planet,
  // this one does NOT use dynamically allocated memory blocks.
  // It's the caller's decision to use whatever he wants to.
{
  BYTE dtable[256], *pbDstPos, block[4], tmp;
  int  i, count; // , out_len;  // who needs a "size_t" ? We don't.
  int pad = 0;
  int nCharsDecoded;

  memset(dtable, 0x80, sizeof(dtable) );
  for (i = 0; i < sizeof(base64_table) - 1; i++)
   { dtable[base64_table[i]] = (unsigned char) i;
   }
  dtable['='] = 0;

  count = 0;
  for (i = 0; i < iSrcLen; i++)
   { if (dtable[pbSrc[i]] != 0x80)
      { count++;
      }
   }

  if (count == 0 || count % 4)
   { return 0;  // something wrong with the base64-encoded input (not N*4 "valid" characters)
   }

  // out_len = count / 4 * 3;
  pbDstPos = pbDst;

  count = 0;
  for (i = 0; i < iSrcLen; i++)
   {
     tmp = dtable[pbSrc[i]];
     if (tmp == 0x80)  // not a "valid" Base64 char ? Skip it, including '\n'
      { continue;
      }

     if (pbSrc[i] == '=')
      { pad++;
      }
     block[count] = tmp;
     count++;
     if (count == 4)
      { *pbDstPos++ = (block[0] << 2) | (block[1] >> 4);
        *pbDstPos++ = (block[1] << 4) | (block[2] >> 2);
        *pbDstPos++ = (block[2] << 6) | block[3];
        count = 0;
        switch(pad)
         { case 0 : // not padding right now
              break;
           case 1 :
              pbDstPos--;
              break;
           case 2 :
              pbDstPos -= 2;
              break;
           default: // invalid padding
              return 0;
         }
      } // end if( count==4 )
   }   // end for i ..

  nCharsDecoded = pbDstPos - pbDst;
  if( nCharsDecoded < iMaxDstLen ) // if possible, provide a trailing zero...
   { pbDst[nCharsDecoded] = 0x00;
   }

  return nCharsDecoded; // returns the length of the DECODED payload in bytes
} // end INET_DecodeBase64()


//---------------------------------------------------------------------------

int INET_UrlEncodingToPlainText( const char * pszUrlEncodedSource,
        char **ppszDestPlainText,
        const char *pszDestEndstop, // 'const' to promise we won't modify the char at this pointer
        const char *pszDelimiters )
  // Converts an URI / "URL" - encoded string into plain text
  // for any further processing .
  // Returns the number of characters processed FROM THE SOURCE STRING .
  // Copying stops *BEFORE* any of the delimiters is reached, in other words
  // the delimiter will *NOT* be contained in the destination string.
  // CAUTION: AFTER DECODING, IT MAY BE IMPOSSIBLE
  //          TO SEPARATE THE 'PARTS' FROM EACH OTHER !
  //       ( consider 'values' the query string containing hex-encoded '&' ) .
  //
  // Taken from Wikipedia on "Query string":
  //
  // > Some characters cannot be part of a URL (for example, the space)
  // > and some other characters have a special meaning in a URL:
  // > for example, the character # is used to locate a point within a page;
  // > the character = is used to separate a name from a value.
  // > A query string may need to be converted to satisfy these constraints.
  // > This can be done using a schema known as URL encoding.
  // > In particular, encoding the query string uses the following rules:
  // >  - [a-z A-Z 0-9] | '.' | '-' | '*' | '_' are left as-is
  // >  - SPACE is encoded as '+' (in fact, more recently they encode SPACE as "%20")
  // >  - All other characters are encoded as %FF hex representation
  // >    with any non-ASCII characters first encoded as UTF-8
  // >    (or other specified encoding)
{ char c;
  int  iCode;
  int  iNumCharsReadFromSource = 0;
  char* pszDestPlainText = *ppszDestPlainText;
  const char *pszDelim;
  BOOL gotDelim;

  while( (*pszUrlEncodedSource != '\0') && (pszDestPlainText<(pszDestEndstop-1)) )
   { c = *pszUrlEncodedSource++;
     pszDelim = pszDelimiters; // check for this LIST OF delimiter characters (in the form of a nice old C string)
     gotDelim = FALSE;
     while( !gotDelim )
      { if( c==*pszDelim ) // <- this includes "the ultimate delimiter" (0x00)
         { gotDelim = TRUE;
         }
        else if( *pszDelim=='\0' ) // through with all delimiters, none of them matched
         { break;  // break from the inner loop
         }
        ++pszDelim;
      }
     if( gotDelim )
        break;     // break from the outer loop
     ++iNumCharsReadFromSource;
     switch( c )
      {
        case '+':
          *pszDestPlainText++ = ' ';
          break;
        case '%':
          iCode = SL_ParseHex( &pszUrlEncodedSource, 2/*nMaxDigits*/ );
          if( iCode>0 )
           { iNumCharsReadFromSource += 2;
             *pszDestPlainText++ = (char)iCode;
           }
          break;
        default:
          *pszDestPlainText++ = c;
          break;
      } // end switch( c )
   } // end while()
  if( pszDestPlainText < pszDestEndstop )
   { *pszDestPlainText = '\0'; // terminate destination string if enough space left
   }
  *ppszDestPlainText = pszDestPlainText; // pointer to the NEXT character for other 'string builder' functions
  return iNumCharsReadFromSource;
} // end INET_UrlEncodingToPlainText()

//---------------------------------------------------------------------------
int INET_PlainTextToUrlEncoding( const char * pszPlainTextSource,
        char **ppszDestUrlEncoded, const char *pszDestEndstop,
        const char *pszDelimiters )
  // Converts plain text into an "URL"-encoded string (aka "percent encoding").
  // Returns the number of characters processed FROM THE SOURCE STRING .
  // For details about 'URL encoding', see INET_UrlEncodingToPlainText().
{ char c;
  int  iCode;
  int  iNumCharsReadFromSource = 0;
  char* pszDestUrlEncoded = *ppszDestUrlEncoded;
  const char *pszDelim;
  BOOL gotDelim;


  while( (*pszPlainTextSource != '\0') && (pszDestUrlEncoded<(pszDestEndstop-3)) )
   { c = *pszPlainTextSource++;
     pszDelim = pszDelimiters; // check for this LIST OF delimiter characters (in the form of a nice old C string)
     gotDelim = FALSE;
     while( !gotDelim )
      { if( c==*pszDelim ) // <- this includes "the ultimate delimiter" (0x00)
         { gotDelim = TRUE;
         }
        else if( *pszDelim=='\0' ) // through with all delimiters, none of them matched
         { break;  // break from the inner loop
         }
        ++pszDelim;
      }
     if( gotDelim )
        break;     // break from the outer loop
     ++iNumCharsReadFromSource;
     switch( c )
      { case '+':
        case '%': // used for two-digit hex encoded so must be encoded itself..
        case ' ':
        case '#': // "The character "#" is excluded because it is used to delimit a URI from a fragment identifier."
        case '<': // "disallowed within the URI syntax" ..
        case '>':
        case '|': // "unwise characters" (in a URL/URI) ..
        case '^':
        case '[':
        case ']':
        case '{':
        case '}':
        case ';': // "reserved within a query component and/or have special meaning within a URI/URL" ...
        case '/':
        case '?':
        case ':':
        case '@':
        case '=':
        case '$':
        case ',':
        case '': // german Umlauts (ae, oe, ue, Ae, Oe, Ue) ... not 7-bit ASCII but often seen in USER NAMES
        case '':
        case '':
        case '':
        case '':
        case '':
           SL_AppendString( &pszDestUrlEncoded, pszDestEndstop, "%" );
           SL_AppendHex( &pszDestUrlEncoded, pszDestEndstop, (DWORD)c, 2/*Digits*/ );
           break;
        default:
           *pszDestUrlEncoded++ = c;
           break;
      } // end switch( c )
   } // end while()
  if( pszDestUrlEncoded < pszDestEndstop )
   { *pszDestUrlEncoded = '\0'; // terminate destination string if enough space left
   }
  *ppszDestUrlEncoded = pszDestUrlEncoded; // pointer to the NEXT character for other 'string builder' functions
  return iNumCharsReadFromSource;
} // end INET_PlainTextToUrlEncoding()


//---------------------------------------------------------------------------

BOOL INET_ParseKeyAndValueFromQueryString(

        const char **ppszSrc, // [in] pointer to the next character in the QUERY STRING

        char *pszKey, int iMaxKeyLen, // [out] key name (as zero-terminated C string with length limit)

        char *pszVal, int iMaxValLen) // [out] "value" (as zero-terminated 'plain' C string with length limit)

  // Note: Even though rarely seen yet, both KEY and VALUE *may* be URL-encoded.

  //       What MUST NOT be URL-encoded are the '?' at the begin,

  //       the '=' between key and value, the '&' between two key=value pairs,

  //       and whatever may be the DELIMITER for the entire query string,

  //           MUST NOT be URL-encoded. Thus, the caller MUST NOT

  //           run the entire query string through INET_UrlEncodingToPlainText()

  //           before passing it to e.g. INET_ParseKeyAndValueFromQueryString().

{ const char *pszSrc = *ppszSrc;

  char       *pszDest;

  const char *pszEndstop;

  int nCharsSkipped;


  // Example: pszSrc = "?user=Moritz&call=DL4YHF%20testing"  ("URL-encoded")



  if( (*pszSrc=='?') || (*pszSrc=='&') )

   { ++pszSrc;  // skip the leading '?' or the separating '&'

     pszDest    = pszKey;

     pszEndstop = pszKey + iMaxKeyLen - 1;

     nCharsSkipped = INET_UrlEncodingToPlainText( pszSrc, &pszDest, pszEndstop,

                           "?&= :;#+\r\n\0" /*delimiters*/ );

     if( nCharsSkipped > 0 )

      { pszSrc += nCharsSkipped;

      }

     else // zero-length KEY (name) ?

      { pszKey[0] = '\0';

      }

     // Similar as above for the VALUE (also an URL-encoded string) :

     if( *pszSrc=='=' )

      { ++pszSrc;  // skip the '=' that separates key and value

        pszDest    = pszVal;

        pszEndstop = pszVal + iMaxValLen - 1;

        nCharsSkipped = INET_UrlEncodingToPlainText( pszSrc, &pszDest, pszEndstop,

                           "?&= :;#+\r\n\0" /*delimiters*/ );

        if( nCharsSkipped > 0 )     // similar as for the KEY, also skip the VALUE,

         { pszSrc += nCharsSkipped; // so the caller can call us again with the

         }                          // INCREMENTED pszSrc (example: with "&call=...")


      }

     else // zero-length VALUE (name) ?

      { pszVal[0] = '\0';

      }

     *ppszSrc = pszSrc; // pass back the incremented source-pointer (for the next call)

     return TRUE;  // got something, even if key and/or value may be EMPTY STRINGS

   }

  else // 'source pointer' neither points to '?' nor to '&' ->

   { return FALSE; // wherever it points to isn't the prefix for another key=value pair

   }

} // end INET_ParseKeyAndValueFromQueryString()



//---------------------------------------------------------------------------

// SHA-1 in C. By Steve Reid <steve@edmweb.com>.  100% Public Domain.

//    Not such a "Secure Hash Algorithm" as expected, but even "WebSockets"

//   (as specified in 2013) and TLS still used it in 2019.

//---------------------------------------------------------------------------


  // Test Vectors (from FIPS PUB 180-1)
  // "abc"
  //   A9993E36 4706816A BA3E2571 7850C26C 9CD0D89D
  // "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq"
  //   84983E44 1C3BD26E BAAE4AA1 F95129E5 E54670F1
  // A million repetitions of "a"
  //   34AA973C D4C4DAA4 F61EEB2B DBAD2731 6534016F
  // From https://de.wikipedia.org/wiki/Secure_Hash_Algorithm :
  //   "Franz jagt im komplett verwahrlosten Taxi quer durch Bayern"
  //   68ac9064 95480a34 04beee48 74ed853a 037a7a8f

#define LITTLE_ENDIAN /* This should be #define'd if true.   */
#define SHA1HANDSOFF  /* Copies data before messing with it. */

void SHA1Transform(unsigned long state[5], unsigned char buffer[64]);
void INET_SHA1_Init(  T_INET_SHA1_Context* context);
void INET_SHA1_Update(T_INET_SHA1_Context* context, unsigned char* data, unsigned int len);
void INET_SHA1_Final( T_INET_SHA1_Context* context, unsigned char digest[20]);

#define rol(value, bits) (((value) << (bits)) | ((value) >> (32 - (bits))))

/* blk0() and blk() perform the initial expand. */
/* I got the idea of expanding during the round function from SSLeay */
#ifdef LITTLE_ENDIAN
#define blk0(i) (block->l[i] = (rol(block->l[i],24)&0xFF00FF00) \
    |(rol(block->l[i],8)&0x00FF00FF))
#else
#define blk0(i) block->l[i]
#endif
#define blk(i) (block->l[i&15] = rol(block->l[(i+13)&15]^block->l[(i+8)&15] \
    ^block->l[(i+2)&15]^block->l[i&15],1))

/* (R0+R1), R2, R3, R4 are the different operations used in SHA1 */
#define R0(v,w,x,y,z,i) z+=((w&(x^y))^y)+blk0(i)+0x5A827999+rol(v,5);w=rol(w,30);
#define R1(v,w,x,y,z,i) z+=((w&(x^y))^y)+blk(i)+0x5A827999+rol(v,5);w=rol(w,30);
#define R2(v,w,x,y,z,i) z+=(w^x^y)+blk(i)+0x6ED9EBA1+rol(v,5);w=rol(w,30);
#define R3(v,w,x,y,z,i) z+=(((w|x)&y)|(w&x))+blk(i)+0x8F1BBCDC+rol(v,5);w=rol(w,30);
#define R4(v,w,x,y,z,i) z+=(w^x^y)+blk(i)+0xCA62C1D6+rol(v,5);w=rol(w,30);


/* Hash a single 512-bit block. This is the core of the algorithm. */

void SHA1Transform(unsigned long state[5], unsigned char buffer[64])
{
  unsigned long a, b, c, d, e;
  typedef union {
    unsigned char c[64];
    unsigned long l[16];
  } CHAR64LONG16;
  CHAR64LONG16* block;
#ifdef SHA1HANDSOFF
  static unsigned char workspace[64];
    block = (CHAR64LONG16*)workspace;
    memcpy(block, buffer, 64);
#else
    block = (CHAR64LONG16*)buffer;
#endif
    /* Copy context->state[] to working vars */
    a = state[0];
    b = state[1];
    c = state[2];
    d = state[3];
    e = state[4];
    /* 4 rounds of 20 operations each. Loop unrolled. */
    R0(a,b,c,d,e, 0); R0(e,a,b,c,d, 1); R0(d,e,a,b,c, 2); R0(c,d,e,a,b, 3);
    R0(b,c,d,e,a, 4); R0(a,b,c,d,e, 5); R0(e,a,b,c,d, 6); R0(d,e,a,b,c, 7);
    R0(c,d,e,a,b, 8); R0(b,c,d,e,a, 9); R0(a,b,c,d,e,10); R0(e,a,b,c,d,11);
    R0(d,e,a,b,c,12); R0(c,d,e,a,b,13); R0(b,c,d,e,a,14); R0(a,b,c,d,e,15);
    R1(e,a,b,c,d,16); R1(d,e,a,b,c,17); R1(c,d,e,a,b,18); R1(b,c,d,e,a,19);
    R2(a,b,c,d,e,20); R2(e,a,b,c,d,21); R2(d,e,a,b,c,22); R2(c,d,e,a,b,23);
    R2(b,c,d,e,a,24); R2(a,b,c,d,e,25); R2(e,a,b,c,d,26); R2(d,e,a,b,c,27);
    R2(c,d,e,a,b,28); R2(b,c,d,e,a,29); R2(a,b,c,d,e,30); R2(e,a,b,c,d,31);
    R2(d,e,a,b,c,32); R2(c,d,e,a,b,33); R2(b,c,d,e,a,34); R2(a,b,c,d,e,35);
    R2(e,a,b,c,d,36); R2(d,e,a,b,c,37); R2(c,d,e,a,b,38); R2(b,c,d,e,a,39);
    R3(a,b,c,d,e,40); R3(e,a,b,c,d,41); R3(d,e,a,b,c,42); R3(c,d,e,a,b,43);
    R3(b,c,d,e,a,44); R3(a,b,c,d,e,45); R3(e,a,b,c,d,46); R3(d,e,a,b,c,47);
    R3(c,d,e,a,b,48); R3(b,c,d,e,a,49); R3(a,b,c,d,e,50); R3(e,a,b,c,d,51);
    R3(d,e,a,b,c,52); R3(c,d,e,a,b,53); R3(b,c,d,e,a,54); R3(a,b,c,d,e,55);
    R3(e,a,b,c,d,56); R3(d,e,a,b,c,57); R3(c,d,e,a,b,58); R3(b,c,d,e,a,59);
    R4(a,b,c,d,e,60); R4(e,a,b,c,d,61); R4(d,e,a,b,c,62); R4(c,d,e,a,b,63);
    R4(b,c,d,e,a,64); R4(a,b,c,d,e,65); R4(e,a,b,c,d,66); R4(d,e,a,b,c,67);
    R4(c,d,e,a,b,68); R4(b,c,d,e,a,69); R4(a,b,c,d,e,70); R4(e,a,b,c,d,71);
    R4(d,e,a,b,c,72); R4(c,d,e,a,b,73); R4(b,c,d,e,a,74); R4(a,b,c,d,e,75);
    R4(e,a,b,c,d,76); R4(d,e,a,b,c,77); R4(c,d,e,a,b,78); R4(b,c,d,e,a,79);
    /* Add the working vars back into context.state[] */
    state[0] += a;
    state[1] += b;
    state[2] += c;
    state[3] += d;
    state[4] += e;
    /* Wipe variables */ // (but modern compilers tell you this is "never used"..)
    a = b = c = d = e = 0;
    (void)a;   // shut up Borland, we don't want to leave traces on the stack !
}


/* INET_SHA1_Init - Initialize new context */

void INET_SHA1_Init(T_INET_SHA1_Context* context)
{
    /* SHA1 initialization constants */
    context->state[0] = 0x67452301;
    context->state[1] = 0xEFCDAB89;
    context->state[2] = 0x98BADCFE;
    context->state[3] = 0x10325476;
    context->state[4] = 0xC3D2E1F0;
    context->count[0] = context->count[1] = 0;
}


/* Run your data through this. */

void INET_SHA1_Update(T_INET_SHA1_Context* context, unsigned char* data, unsigned int len)
{
  unsigned int i, j;

  j = (context->count[0] >> 3) & 63;
  if ((context->count[0] += len << 3) < (len << 3)) context->count[1]++;
  context->count[1] += (len >> 29);
  if ((j + len) > 63)
   { memcpy(&context->buffer[j], data, (i = 64-j));
     SHA1Transform(context->state, context->buffer);
     for ( ; i + 63 < len; i += 64)
      { SHA1Transform(context->state, &data[i]);
      }
     j = 0;
   }
  else
   { i = 0;
   }
  memcpy(&context->buffer[j], &data[i], len - i);
} // end INET_SHA1_Update()


/* Add padding and return the message digest. */

void INET_SHA1_Final(T_INET_SHA1_Context* context, unsigned char digest[20])
{
  unsigned long i, j;
  unsigned char finalcount[8];

  for (i = 0; i < 8; i++)
   { finalcount[i] = (unsigned char)((context->count[(i >= 4 ? 0 : 1)]
                      >> ((3-(i & 3)) * 8) ) & 255);  /* Endian independent */
   }
  INET_SHA1_Update(context, (unsigned char *)"\200", 1);
  while ((context->count[0] & 504) != 448)
   { INET_SHA1_Update(context, (unsigned char *)"\0", 1);
   }
  INET_SHA1_Update(context, finalcount, 8);  /* Should cause a SHA1Transform() */
  for (i = 0; i < 20; i++)
   { digest[i] = (unsigned char)
         ((context->state[i>>2] >> ((3-(i & 3)) * 8) ) & 255);
   }
  /* Wipe variables */
  i = j = 0;
  (void)i; // defeat warning from Borland C : "blah was assigned a value that is never used".
           // Guess Steve's intention for 'wiping variables' is NOT to leave a trace
           // of what has been calculated on the stack .
  (void)j; // again: shut up, Borland (blubb assigned a value that is never used)
  memset(context->buffer, 0, 64);
  memset(context->state, 0, 20);
  memset(context->count, 0, 8);
  memset(&finalcount, 0, 8);
#ifdef SHA1HANDSOFF  /* make SHA1Transform overwrite it's own static vars */
  SHA1Transform(context->state, context->buffer);
#endif
} // end INET_SHA1_Final()


//---------------------------------------------------------------------------
BOOL INET_WebSecSocketKeyToWebSocketAcceptValue(
         char* pszSecWebSocketKey,       // [in]  value from field "Sec-Websocket-Key"
         char* cpDestSecWebSocketAccept, // [out] value for "Sec-Websocket-Accept"
         char* cpDestEndstop )           // [in] endstop for the destination (max. position of the string's trailing zero)
{ char sz255[256];
  T_INET_SHA1_Context sha1_context; // temporary instance to calculate SHA1..
  BYTE b20Digest[20];  // .. as "digest" to generate the "Sec-WebSocket-Accept"-value
  int  iEncodedLength;
  SL_strncpy( sz255, pszSecWebSocketKey, sizeof(sz255)-40/*max dest length*/ );
  SL_strncat( sz255, "258EAFA5-E914-47DA-95CA-C5AB0DC85B11", sizeof(sz255)-1 );
  INET_SHA1_Init(&sha1_context);  // Prepare SHA-1 calculation
  INET_SHA1_Update(&sha1_context, (BYTE*)sz255, SL_strnlen(sz255,255) ); // add to SHA-1 hash ("digest")
  INET_SHA1_Final(&sha1_context, b20Digest ); // Finish SHA-1 calculation
  iEncodedLength = INET_EncodeBase64( b20Digest/*pbSrc*/, 20/*iSrcLen*/,
                      (BYTE*)cpDestSecWebSocketAccept/*pbDst*/,
                      cpDestEndstop - cpDestSecWebSocketAccept/*iMaxDstLen*/,
                      0/*all in ONE line*/ );
  if( iEncodedLength < 20*4/3 ) // oops.. this cannot be correct for a 20-byte "digest" !
   { return FALSE;
   }                    
  // Example : sz255="qZUNEqiPTUQhPTem1RFaUA==258EAFA5-E914-47DA-95CA-C5AB0DC85B11"
  //       b20Digest= 4c e7 07 a1 70 45 cf 13 e2 15 af 33 18 b3 e9 99 23 8f 82 6c
  //                  (ok, according to www.sha1generator.de/ )
  //       cpDestSecWebSocketAccept = "TOcHoXBFzxPiFa8zGLPpmSOPgmw="
  //                  (calculated with the "Hexadecimal -> base64 string encoder" at tomeko.net/online_tools/hex_to_base64.php)
  if( (cpDestSecWebSocketAccept+iEncodedLength) < cpDestEndstop )
   { cpDestSecWebSocketAccept[ iEncodedLength ] = '\0'; // ok (sufficient capacity)
     return TRUE;
   }
  else
   { *cpDestEndstop = '\0'; // error, truncated, this won't be a valid "Sec-WebSocket-Accept" - value
     return FALSE;
   }
} // end INET_WebSecSocketKeyToWebSocketAcceptValue()

//---------------------------------------------------------------------------
char* INET_WinsockErrorCodeToString( int iWSAErrorCode )
 // Turns error codes returned by "WSAGetLastError()" into text .
{
  static char sz80OtherError[84];  // <<< avoid using this static var wherever possible
                                   //  (it may cause problems with multi-threading one day)
  if( iWSAErrorCode < 0 )  // oops.. must have been a Winsock-error "made negative"..
   {  iWSAErrorCode = -iWSAErrorCode;  // .. to distinguish it from a "successful return value"
   }
  switch( iWSAErrorCode )
   {
    case WSAE_NO_ERROR     : return "ok";  // <- WB's addition to the Windows Socket Application Error codes
    case WSAEINTR          : return "Interrupted function call";
    case WSAEBADF          : return "Invalid socket descriptor";
    case WSAEACCES         : return "Permission denied";
    case WSAEFAULT         : return "Bad address";
    case WSAEINVAL         : return "Invalid argument";
    case WSAEMFILE         : return "Too many open files or sockets";
    case WSAEWOULDBLOCK    : return "Call of a WINSOCK function would block";
    case WSAEINPROGRESS    : return "Operation now in progress";
    case WSAEALREADY       : return "Operation already in progress";
    case WSAENOTSOCK       : return "Socket operation on non-socket";
    case WSAEDESTADDRREQ   : return "Destination address required";
    case WSAEMSGSIZE       : return "Message too long";
    case WSAEPROTOTYPE     : return "Protocol wrong type for socket";
    case WSAENOPROTOOPT    : return "Bad protocol option";
    case WSAEPROTONOSUPPORT: return "Socket type not supported";
    case WSAESOCKTNOSUPPORT: return "Socket not supported";
    case WSAEOPNOTSUPP     : return "Operation not supported";
    case WSAEPFNOSUPPORT   : return "Protocol family not supported";
    case WSAEAFNOSUPPORT   : return "Address family not supported by protocol family";
    case WSAEADDRINUSE     : return "Address already in use";
    case WSAEADDRNOTAVAIL  : return "Cannot assign requested address";
    case WSAENETDOWN       : return "Network is down";
    case WSAENETUNREACH    : return "Network is unreachable";
    case WSAENETRESET      : return "Network dropped connection on reset";
    case WSAECONNABORTED   : return "Software caused connection abort";
    case WSAECONNRESET     : return "Connection reset by peer";
    case WSAENOBUFS        : return "No buffer space available";
    case WSAEISCONN        : return "socket already connected";
    case WSAENOTCONN       : return "Socket is not connected";
    case WSAESHUTDOWN      : return "Cannot send after socket shutdown";
    case WSAETOOMANYREFS   : return "too many refs";
    case WSAETIMEDOUT      : return "Connection timed out";
    case WSAECONNREFUSED   : return "Connection refused";
    case WSAELOOP          : return "loop detected";
    case WSAENAMETOOLONG   : return "name too long";
    case WSAEHOSTDOWN      : return "Host is down";
    case WSAEHOSTUNREACH   : return "No route to host";
    case WSAENOTEMPTY      : return "not empty";
    case WSAEPROCLIM       : return "Too many processes";
    case WSAEUSERS         : return "Ran out of user quota";
    case WSAEDQUOT         : return "Ran out of disk quota";
    case WSAESTALE         : return "Stale handle reference";
    case WSAEREMOTE        : return "Item is remote";
    case WSAEDISCON        : return "Graceful shutdown in progress";
    case WSASYSNOTREADY    : return "Network subsystem is unavailable";
    case WSAVERNOTSUPPORTED: return "WINSOCK.DLL version out of range";
    case WSANOTINITIALISED : return "WSAStartup not yet performed";
    case WSAHOST_NOT_FOUND : return "Host not found";
    case WSATRY_AGAIN      : return "Non-authoritative host not found";
    case WSANO_RECOVERY    : return "non-recoverable error";
    case WSANO_DATA        : return "valid name but no data";
    default                :
         sprintf(sz80OtherError, "Other WINSOCK error, code %d",(int)iWSAErrorCode);
         return sz80OtherError; // utterly thread-unsafe, but ideally NEVER USED
   } // end switch( iWSAErrorCode )

} // end INET_WinsockErrorCodeToString()



//---------------------------------------------------------------------------
void INET_BlockIPv4Address( T_InetBlacklist *pBlacklist, BYTE *pbIPv4Address, char *pszWhy )
  // Called e.g. from the HTTP server (or the Hamlib server) if a visitor
  // tries to GET or POST funny stuff - see examples received from a botnet in HttpServer.c .
{ int i, iFirstUnusedEntry = -1;
  unsigned char c;
  DWORD dwIP;
  T_InetBlacklistEntry *pBE;

  memcpy( &dwIP, pbIPv4Address, 4 ); // who knows if pbIPv4Address is DWORD-aligned ?

  for( i=0; i<INET_MAX_BLOCKED_IP_ADDRESSES; ++i )
   { pBE = &pBlacklist->sBlockedIPs[i];
     if( pBE->b4HisIP.dw == 0 )  // here is a so-far UNUSED entry ..
      { if(iFirstUnusedEntry<0)
         { iFirstUnusedEntry = i;
         }
      }
     if( pBE->b4HisIP.dw == dwIP ) // .. but here we see the bad guy is already KNOWN !
      { return; // no need to search any further, and no need to block him TWICE
      }
   }
  // Arrived here ? It's the bad guy's first visit, so Welcome to our blacklist:
  if( iFirstUnusedEntry >= 0 )
   { pBE = &pBlacklist->sBlockedIPs[iFirstUnusedEntry];
     memset( pBE, 0, sizeof(T_InetBlacklistEntry) );
     pBE->b4HisIP.dw = dwIP;
     if( pszWhy != NULL )
      { for(i=0; i<59; ++i)
         { c = (unsigned char)*pszWhy++; // beware of 'poisoned' strings ...
           if( (c!=0) && ( (c<32) || (c>127) ) ) // so replace non-ASCII crap by underscores
            { c = '_';
            }
           pBE->sz59Reason[i] = c; // "only for the log"
         }
      }
     else
      { strcpy( pBE->sz59Reason, "?reason?" );
      }
   }
} // end INET_BlockIPv4Address()

//---------------------------------------------------------------------------
BOOL INET_IsBlockedIPv4Address( T_InetBlacklist *pBlacklist, BYTE *pbIPv4Address )
  // Returns TRUE if the specified IPv4 address has previously been blocked
  // via INET_BlockIPv4Address() . Sorry for users that 'recycled' the blocked
  // IP address, because they will not receive a single byte of info from
  // our server.. until the program is restarted with an EMPTY blacklist.
{ int i;
  DWORD dwIP;
  memcpy( &dwIP, pbIPv4Address, 4 ); // who knows if pbIPv4Address is DWORD-aligned ?
  for( i=0; i<INET_MAX_BLOCKED_IP_ADDRESSES; ++i )
   { if( pBlacklist->sBlockedIPs[i].b4HisIP.dw == dwIP )
      { return TRUE;  // gotcha .. already blacklisted -> farewell
      }
   }
  return FALSE; // no info about this guy, so don't block him (for a start)
} // end INET_IsBlockedIPv4Address()


//----------------------------------------------------------------------------
CPROT long INET_GetSocketLocalPort( SOCKET s ) // aka 'ephemeral' port number,
   // as long as BindSocketToLocalPort() was not called for the socket.
   // Returns a positive value when successful,
   //      or a ZERO when the 'socket services' don't have
   //           allocated an 'ephemeral' port for this socket yet,
   //           or the socket hasn't been BOUND to a certain port yet,
   //      or a NEGATIVE-MADE "WSAE" error code .
{
   // > So you thought you could simply call getsockname() after creating
   // > a socket, and that funny-named function would deliver the "name"
   // > (IPv4 address) plus the ephemeral port ? Wrong.
   // > Welcome to the wonderful world of Winsock-, and possibly
   // > Berkeley socket programming. Be brave, and READ ON.
   //
   struct sockaddr_in local_addr; // <- this beast is required to find out OUR "port number"
#ifndef   socklen_t    // some socket APIs have this, others don't. Crap.
#  define socklen_t int
#endif
   socklen_t sicklen;  // yes, this is really sick. Winsick.
   int iWinsockError;

   memset( &local_addr, 0, sizeof(local_addr) ); // not sure why, but others do this, too
   sicklen/*16*/ = sizeof(local_addr); // getsockname needs a POINTER to the length
     //                     (not just the length itself; that'd be too simple)
     // > The getsockname function retrieves the local name for a socket.
     // > The getsockname function retrieves the current name for the
     // > specified socket descriptor in name (??). It is used on the bound
     // > or connected socket specified by the s parameter.
     // > The local association is returned. This call is especially useful
     // > when a connect call has been made without doing a bind first;
     // > the getsockname function provides the only way to determine
     // > the local association that has been set by the system.
     // "name" .. "local association" .. not mentioning "port number" at all. Mumbo-Jumbo.
  iWinsockError = getsockname( s, (struct sockaddr*)&local_addr, &sicklen );
  if( iWinsockError == SOCKET_ERROR ) // simply returning an ERROR CODE from getsockname() would be too easy...
   { // Will often get here with iWinsockError = -1 .. see lengthy explanation above.
     return -WSAGetLastError();
   }
  else  // > If no error occurs, getsockname returns zero ...
   { // whow, getsockname() finally surrendered and returned what we wanted.. REALLY ?
     return ntohs( local_addr.sin_port ); // -> without the dumy-bind(), the result was always ZERO !
     // If the result is ZERO, how can the caller convince Windows (Winsock)
     // to allocate an EPHEMERAL PORT (for example, we needed to know that
     // before sending the first datagram (UDP) to an IC-705 *IN THE UDP PAYLOAD*):
     // > MSDN's documentation (later moved to "learn.microsoft") for sendto() said:
     // > Note If a socket is opened, a setsockopt() call is made, and then a
     // > sendto() call is made, Windows Sockets performs an implicit bind() call.
     // > If the socket is unbound, unique values are assigned to the local
     // > association by the system, and the socket is then marked as bound.
     // > If the socket is connected, the getsockname function can be used
     // > to determine the local IP address and port associated with the socket.
     // > If the socket is using a connectionless protocol (WB: UDP), the address
     // > may not be available until I/O occurs on the socket.
     //     A-ha. Very practical. UDP is one of those 'connectionless protocols'.
     //     Thus, in other words, we're supposed to "do I/O"
     //     on this unbound, unconnected socket before we know our own port number.
     //     This is triple-stupid, and there must be a way to convince the bloody
     //     misleading 'getsockname()' to spit out our local, 'ephemeral' UDP port
     //     number somehow ! [the solution later turned out the dummy-bind() ]
     // The Linux manual gives some advice - maybe this works with the
     // windoze socket services, too ? :
     // > An ephemeral port is allocated to a socket in the following circumstances:
     // >  * the port number in a socket address is specified as 0 when calling bind(2);
     // >  * listen(2) is called on a stream socket that was not previously bound;
     // >  * connect(2) was called on a socket that was not previously bound;
     // >  * sendto(2) is called on a datagram socket that was not previously bound.
     //             '--- What's this geek stuff in parentheses ?
     //                  It cannot be the number of arguments. Anyway.
   }
} // end INET_GetSocketLocalPort()

//---------------------------------------------------------------------------
BOOL INET_LookupDomainOrParseIP( const char *pszUrlOrNumericIP,    // API
             BYTE *pbIPv4Address, // [out] 4-byte (IPv4) 'address'
             int  *piPort)        // [out] 16-bit port number, in HOST byte order
{
  BOOL fResult = FALSE;
  const char *cp = pszUrlOrNumericIP;  // [in] e.g. "MyRemoteRadio" or maybe "192.168.0.123", etc.
  struct hostent *hp; // omg... sooo many obfuscated structs in winsock.h ...
  struct sockaddr_in my_addr; // .. here's one more; this beast is required to find out OUR "port number"
#ifndef  socklen_t    // some socket APIs have this / need this, others don't. Crap.
# define socklen_t int
#endif
  DWORD dwIPv4Address;
  const char *cpLastColon = strrchr( pszUrlOrNumericIP, ':' ); // check if there's a colon near the end

  while(*cp>='0' && *cp<='9') ++cp;
  if( *cp=='.' && cp>pszUrlOrNumericIP ) // something like '192.' .. must be a numeric IP address !
   { cp = pszUrlOrNumericIP;
     SL_ParseIPv4Address( &cp, pbIPv4Address );
     // FORGET ABOUT "inet_addr()" ! That obfuscated piece of crap will
     // treat leading zeros as indicator for OCTAL VALUES. Also, it doesn't
     // clearly specify if the "first byte" in our dotted *decimal* notation
     // will land in bits 31..24 or bits 7..0 of the returned "doubleword".
     fResult = TRUE; // be optimistic ..
   }
  else // everything else can hopefully be resolved by a DNS- or "localhosts"-lookup:
   {
     // if there's a colon near the end, then carefully REMOVE IT, TEMPORARILY..
     if( cpLastColon != NULL ) // .. because who knows what gethostbyname() would do..
      { *(char*)cpLastColon = '\0';   // when encountering something that doesn't belong to the HOST NAME !
      }
     hp = gethostbyname(pszUrlOrNumericIP);    // e.g. sz40RemoteHostName = "IC-705" ...
     if( cpLastColon != NULL )     // 'repair' the original string (omg.. )
      { *(char*)cpLastColon = ':'; // .. we'll care for what follows after the colon LATER
      }
     if (hp != NULL) // gethostbyname() sucessful ?
      { // As usual in the obfuscated 'Socket Services', the result
        // (a crazy 'struct hostent pointer') isn't a single item,
        // but a WHOLE LIST of items. For example, if the PC has multiple
        // 'network interfaces' (say two Ethernet LAN ports, and one WLAN),
        // then there may be many roads leading to Rome (or the remote server):
        // > typedef struct hostent
        // >   {  char  *h_name;  // The official name of the host (PC). If using the DNS or similar resolution system, it is the Fully Qualified Domain Name (FQDN) that caused the server to return a reply. If using a local hosts file, it is the first entry after the IPv4 address.
        // >      char  **h_aliases; // A NULL-terminated array of alternate names.
        // >      short h_addrtype;  // The type of address being returned.
        // >      short h_length;    // The length, in bytes, of each address.
        // >      char  **h_addr_list; // A NULL-terminated list of addresses for the host. Addresses are returned in network byte order.
        // >   }
        if( ( hp->h_addrtype == AF_INET )   // sounds good, the "address family" is "internet"...
         && ( hp->h_length == 4 )           // .. and the list of addresses all seem to be "IPv4"...
         && ( hp->h_addr_list[0] != NULL) ) // .. and there's at least ONE entry in the ...
         { dwIPv4Address = *(DWORD*)hp->h_addr_list[0]; // read FOUR BYTES in "network byte order"
           pbIPv4Address[0] = (BYTE)(dwIPv4Address>>0);  // .. and convert into a SIMPLE,
           pbIPv4Address[1] = (BYTE)(dwIPv4Address>>8);  // non-esoteric data type,
           pbIPv4Address[2] = (BYTE)(dwIPv4Address>>16); // that neither depends on Qt nor Winsick !
           pbIPv4Address[3] = (BYTE)(dwIPv4Address>>24);
           fResult = TRUE; // be optimistic ...
         }
      } // end if < gethostbyname() successful >
   }   // end else < not a "dotted numeric IP address" but really a "hostname" >

  if( piPort != NULL )     // Caller wants to parse a PORT number, too,
   { // e.g. pszUrlOrNumericIP = "192.168.0.50:7533" -> *piPort = 7355
     if( cpLastColon != NULL ) // Houston, we seem to have a PORT NUMBER !
      { ++cpLastColon;
        *piPort = SL_ParseInteger( &cpLastColon );
      }
     else // look mum, no port number !
      { *piPort = -1; // ex: CWNET_DEFAULT_SERVER_PORT
      }
   }
  return fResult;

} // end INET_LookupDomainOrParseIP()


//---------------------------------------------------------------------------
CPROT int INET_ConnectSocketToRemoteAddressAndPort( SOCKET s,
             BYTE *b4RemoteAddress,    // [in] 4-byte (IPv4) 'remote' address
             unsigned int iRemotePort) // [in] 16-bit 'remote' port to connect
  // On failure, returns one of the 'winsock' error codes, but "made negative".
  // When successful, returns 0 = WSAE_NO_ERRORS .
  // When used on a NON-BLOCKING socket, may return ..WOULDBLOCK
  //           (which means the operation is STILL PENDING) .
  //
  // This is more or less a wrapper for windsock[2]'s "connect()" .
  //      Typically used for TCP on the *client* side .
{ struct sockaddr_in remote_addr; // Winsock-thing for "connecting" a socket to a certain remote address (IPv4) and port

  // Set up a SOCKADDR_IN structure for bind() :
  memset( &remote_addr, 0, sizeof(remote_addr) ); // not sure why, but others do this, too
  remote_addr.sin_family = AF_INET;
  remote_addr.sin_addr.S_un.S_un_b.s_b1 = b4RemoteAddress[0]; // e.g. 224 ...
  remote_addr.sin_addr.S_un.S_un_b.s_b2 = b4RemoteAddress[1]; // e.g.   0 ...
  remote_addr.sin_addr.S_un.S_un_b.s_b3 = b4RemoteAddress[2]; // e.g.   2 ("AD-HOC block 1")
  remote_addr.sin_addr.S_un.S_un_b.s_b4 = b4RemoteAddress[3]; // e.g. 111 (avoids the risk of colliding)
  remote_addr.sin_port = htons( iRemotePort );
  // Establish a connection to a specified socket using connect()
  //   [warning: This may take considerable time, and may fail on a
  //             non-blocking socket, returning the old familiar "..WOULDBLOCK"]
  if( connect( s, (SOCKADDR *)&remote_addr, sizeof(remote_addr)) == SOCKET_ERROR)
   { return -WSAGetLastError();
   }
  return WSAE_NO_ERROR;
} // end INET_ConnectSocketToRemoteAddressAndPort()



#if(0)  // TEST application for the above SHA-1 implementation, by Steve Reid
        // Of course we don't want to have a "main" in this module !
        // A *very short* test for the SHA-1 is contained in INET_Init() .

int main(int argc, char** argv)
{
  int i, j;
  T_INET_SHA1_Context context;
  unsigned char digest[20], buffer[16384];
  FILE* file;

  if (argc > 2) {
        puts("Public domain SHA-1 implementation - by Steve Reid <steve@edmweb.com>");
        puts("Produces the SHA-1 hash of a file, or stdin if no file is specified.");
        exit(0);
    }
    if (argc < 2) {
        file = stdin;
    }
    else {
        if (!(file = fopen(argv[1], "rb"))) {
            fputs("Unable to open file.", stderr);
            exit(-1);
        }
    }
    INET_SHA1_Init(&context);
    while (!feof(file)) {  /* note: what if ferror(file) */
        i = fread(buffer, 1, 16384, file);
        INET_SHA1_Update(&context, buffer, i);
    }
    INET_SHA1_Final(digest, &context);
    fclose(file);
    for (i = 0; i < 5; i++) {
        for (j = 0; j < 4; j++) {
            printf("%02X", digest[i*4+j]);
        }
        putchar(' ');
    }
    putchar('\n');
    exit(0);
}

#endif // (0)

//---------------------------------------------------------------------------
int INET_Init(void) // -> return value : >= 0 when ok, < 0 indicates an error.
  // Must be called ONCE before calling anything else
  // in this module (YHF_Inet.c). Also retrieves the FUNCTION POINTERS of
  // a few crazy functions that cannot be called 'directly' in Winsock2,
  // for example the 'overlapped capable' ConnectEx().
  //
  // Returns a NEGATIVE error code if any of the built-in self tests failed.
{
  BYTE b256[256], b20Digest[20];
  char sz255[256];
  int  i, iEncodedLength, iDecodedLength;
  T_INET_SHA1_Context sha1_context;
  DWORD dwBytes;
  int iResult;


  // Perform a few(!) internal self tests, save headaches later when stuff gets REALLY complex !
  iEncodedLength = INET_EncodeBase64( (BYTE*)szFranz, strlen(szFranz), b256, sizeof(b256), 0 );
  if( iEncodedLength != (int)strlen( szFranzBase64 ) )  // 2019-01-21 : 80 bytes
   { return -1;  // not the expected length from the Base64 encoder
   }
  if( memcmp( b256, szFranzBase64, iEncodedLength) != 0 ) // 2019-01-21: "RnJh.."
   { return -2;  // not the expected data from the Base64 encoder
   }
  iDecodedLength = INET_DecodeBase64( (BYTE*)szFranzBase64, strlen(szFranzBase64), b256, sizeof(b256) );
  if( iDecodedLength != (int)strlen( szFranz ) )
   { return -3;  // not the expected length from the Base64 decoder
   }
  if( memcmp( b256, szFranz, iDecodedLength) != 0 )
   { return -4;  // not the expected data from the Base64 decoder
   }
  iEncodedLength = INET_EncodeBase64( (BYTE*)"a", 1, b256, sizeof(b256), 0/*no line-breaks*/ );
  if( iEncodedLength != 4 )
   { return -5;  // not the expected length from the Base64 decoder
   }
  if( memcmp( b256, "YQ==", 4) != 0 )
   { return -6;  // not the expected data from the Base64 encoder for a SINGLE character
   }
  iDecodedLength = INET_DecodeBase64( b256, iEncodedLength, (BYTE*)sz255,255 ); // decode back to "a" again ?
  if( iDecodedLength != 1 )
   { return -7;  // not the expected length from the Base64 decoder for a single byte
   }
  if( memcmp( sz255, "a", iDecodedLength) != 0 )
   { return -8;  // not the expected data from the Base64 decoder
   }
  iEncodedLength = INET_EncodeBase64( (BYTE*)"ab", 2, b256, sizeof(b256), 0/*no line-breaks*/ );
  if( iEncodedLength != 4 )
   { return -9;  // not the expected length from the Base64 decoder
   }
  if( memcmp( b256, "YWI=", 4) != 0 )
   { return -10;  // not the expected data from the Base64 encoder for a SINGLE character
   }
  iDecodedLength = INET_DecodeBase64( b256, iEncodedLength, (BYTE*)sz255,255 ); // decode back to "ab" again ?
  if( iDecodedLength != 2 )
   { return -11;  // not the expected length from the Base64 decoder for a two-byte string
   }
  if( memcmp( sz255, "ab", iDecodedLength) != 0 )
   { return -12;  // not the expected data from the Base64 decoder
   }
  iEncodedLength = INET_EncodeBase64( (BYTE*)"abc", 3, b256, sizeof(b256), 0/*no line-breaks*/ );
  if( iEncodedLength != 4 )
   { return -13;  // not the expected length from the Base64 decoder
   }
  if( memcmp( b256, "YWJj", 4) != 0 )
   { return -14;  // not the expected data from the Base64 encoder for a SINGLE character
   }
  iDecodedLength = INET_DecodeBase64( b256, iEncodedLength, (BYTE*)sz255,255 ); // decode back to "abc" again ?
  if( iDecodedLength != 3 )
   { return -15;  // not the expected length from the Base64 decoder for a two-byte string
   }
  if( memcmp( sz255, "abc", iDecodedLength) != 0 )
   { return -16;  // not the expected data from the Base64 decoder
   }

  INET_SHA1_Init(&sha1_context);  // Prepare SHA-1 calculation
  INET_SHA1_Update(&sha1_context, (BYTE*)szFranz, strlen(szFranz) ); // add to SHA-1 hash ("digest")
  INET_SHA1_Final(&sha1_context, b20Digest ); // Finish SHA-1 calculation
  for( i=0; i<20; ++i )   // 2019-01-21 : Test passed (all 20 bytes of the "digest")
   { if( b20Digest[i] != b20FranzSHA1[i] )
      { return -17;  // something screwed up with the SHA-1 calculation
      }
   }

  // Test "Sec-WebSocket-Key" -> "Sec-WebSocket-Accept",
  // using data from an OpenWebSDR sessession (with Chrome's "Network" tools)
  if( ! INET_WebSecSocketKeyToWebSocketAcceptValue( // generate value for "Sec-Websocket-Accept"
         "qZUNEqiPTUQhPTem1RFaUA==",  // [in]  value from field "Sec-Websocket-Key"
         sz255,      // [out] cpDestSecWebSocketAccept, value for "Sec-Websocket-Accept"
         sz255+255)) // [in] cpDestEndstop, endstop for the destination (max. position of the string's trailing zero)
   { return -18;  // INET_WebSecSocketKeyToWebSocketAcceptValue() noticed an error itself
   }
  if( strcmp(sz255, "TOcHoXBFzxPiFa8zGLPpmSOPgmw=") != 0 )
   { return -19;  // INET_WebSecSocketKeyToWebSocketAcceptValue() produced a wrong "Sec-WebSocket-Accept" value (string)
   }

  return 1;  // successfully performed all built-in module tests

} // end INET_Init()


/* EOF < Inet_Tools.c > */






