// File:  C:\cbproj\Remote_CW_Keyer\HttpServer.h
// Date:  2024-01-20
// Author:  Wolfgang Buescher (DL4YHF)
// Purpose: Socket-based 'Client and Server for the Remote CW Keyer'.
//          As usual, details only in the implementation (HttpServer.c) .
//
// Most recent modifications: see implementation, Remote_CW_Keyer/HttpServer.c .
//

#ifndef SWI_USE_HTTP_SERVER
# include "switches.h"
#endif
#ifndef SWI_USE_HTTP_SERVER
# error "Missing definition of SWI_USE_HTTP_SERVER in 'switches.h' !"
#endif
#if( SWI_USE_HTTP_SERVER ) // this conditional ends near end-of-file ..


#ifndef  _HTTP_SERVER_H  // prevent multiple header inclusion ..
# define _HTTP_SERVER_H  //  .. and let other modules know we're been included

# ifndef _CW_NET_H       // this tiny HTTP-Server is *used by* CwNet.c ...
#  include "CwNet.h"     // .. see the chicken-and-egg problem, A uses B, B uses A
# endif



//---------------------------------------------------------------------------
// Data Types
//---------------------------------------------------------------------------


typedef enum  // tokens for items in the web-based SETUP MENU (setup.htm) and similar pages :
{ C_HttpSrv_ConfigItem_Unknown = 0,
  C_HttpSrv_ConfigItem_TopMenu,        // $cfg$top_menu = HTML header and body with a "top menu"
  C_HttpSrv_ConfigItem_Password1,         C_HttpSrv_ConfigItem_Password2,
  C_HttpSrv_ConfigItem_Password1_BgColor, C_HttpSrv_ConfigItem_Password2_BgColor,
  C_HttpSrv_ConfigItem_LoginBox,       // sub-form with either password+"Login"  or "Logout"
  C_HttpSrv_ConfigItem_LoginButton,    // machine-generated button on the above form
  C_HttpSrv_ConfigItem_ErrorMessage,   // not a CONFIGURABLE ITEM, but a response after submitting a form
  C_HttpSrv_ConfigItem_Setup,          // for buttons on the "setup" pages (identified by value="Submit", etc)
  C_HttpSrv_ConfigItem_Trace,          // for buttons on the "trace" page (identified by value="Update","Clear",etc)
  C_HttpSrv_ConfigItem_UploadedFiles,  // list of all files which have been "POSTED" (submitted)
  C_HttpSrv_ConfigItem_PeriodicUpdate, // for the checkbox "periodic update" on various pages

  // Display- and Audio Setup Items ...
  C_HttpSrv_ConfigItem_LCD_Brightness_Norm, C_HttpSrv_ConfigItem_LCD_Brightness_Low,
  C_HttpSrv_ConfigItem_Bootup_Display, C_HttpSrv_ConfigItem_Visible_Gestures,
  C_HttpSrv_ConfigItem_Click_Volume,   C_HttpSrv_ConfigItem_Speaker_Volume,
  C_HttpSrv_ConfigItem_Warning_Volume, C_HttpSrv_ConfigItem_Mic_Gain,
  C_HttpSrv_ConfigItem_RTC_Date,       C_HttpSrv_ConfigItem_RTC_Time,

  // Network Setup Items ...
  C_HttpSrv_ConfigItem_MAC_Address,    C_HttpSrv_ConfigItem_Hostname,
  C_HttpSrv_ConfigItem_IP_Address,     C_HttpSrv_ConfigItem_DHCP_Enable,
  C_HttpSrv_ConfigItem_IP_Subnet,      C_HttpSrv_ConfigItem_IP_Gateway,
  C_HttpSrv_ConfigItem_LocalHttpPort,  C_HttpSrv_ConfigItem_LocalTelnetPort,
  C_HttpSrv_ConfigItem_LocalFtpPort,

} T_HttpSrv_ConfigItem;


typedef enum  // HTTP server states :
{ C_HttpSrvState_PASSIVE             = 0,
  C_HttpSrvState_RECEIVING_REQUEST   = 1,
  C_HttpSrvState_RECEIVED_REQUEST,
  C_HttpSrvState_RECEIVING_MULTIPART, // waiting for more data from a "Multipart message"
  C_HttpSrvState_WAIT_LONG_POLL,      // waiting in a 'long poll', e.g. for a "Server Event"
  C_HttpSrvState_SENDING_HEADER,
  C_HttpSrvState_SENT_HEADER,
  C_HttpSrvState_SENDING_DATA,        // after this, with "keep-alive" : next state = C_HttpSrvState_RECEIVING_REQUEST or maybe .._ASSEMBLE_NEXT_CHUNK, etc
  C_HttpSrvState_SENDING_ERROR,
  C_HttpSrvState_WEBSOCKET_OPEN,      // a TCP-like bidirectional "WebSocket" has been established, handshake finished, waiting for some data to send or receive
  C_HttpSrvState_ASSEMBLE_NEXT_CHUNK, // waiting for THE APPLICATION to assemble the next "chunk", for Transfer-Encoding = Chunked, e.g. to send endless streams, not size-limited files.
  C_HttpSrvState_CLOSE_WHEN_SENT,
  C_HttpSrvState_TRIED_TO_CLOSE_BUT_TCP_CLOSE_FAILED /* a VERY annoying "feature" of LwIP ! */
  // ( TCP/IP is so utterly complex.. instead of simply/immediately "closing" a connection
  //   from either side, the connection must be kept 'dangling' just in case the other side
  //   asks for a retransmission, etc etc etc etc etc etc etc etc etc etc etc .
  //   That's why LwIP doesn't allow really 'closing' a connection immediately,
  //   and often runs out of PCBs (Protocol Control Blocks, not Printed Circuit Boards) ... )
} T_HttpServerState; // used as .nServerState in T_HttpServer


// Possible values parsed from header-line: "Content-Type: ", stored in variables named 'nContentType' .
// Used in string table HttpSrv_ContentTypes[], access via HttpSrv_GetTokenFromTable() / HttpSrv_GetStringFromTable() .
// Later also used as BITWISE COMBINATION for the "Accept"-field in an HTTP request header .
#define T_HttpSrv_ContentType  unsigned long
#define C_HttpSrv_ContentType_Unknown      0
#define C_HttpSrv_ContentType_All          0x00FFFFFF
#define C_HttpSrv_ContentType_Text         0x00000001
#define C_HttpSrv_ContentType_Text_Plain   0x00000002 // -> "Content-Type:text/plain" (also used to POST FORMS!)
#define C_HttpSrv_ContentType_Text_HTML_ISO_8859_1 0x00000004 // -> "Content-Type:text/html; charset=ISO-8859-1"
#define C_HttpSrv_ContentType_Text_HTML    0x00000008 // "text/html" without a 'charset'-specification
#define C_HttpSrv_ContentType_Image        0x00000010
#define C_HttpSrv_ContentType_Image_BMP    0x00000020 // "image/bmp", used by the UPT web server to retrieve the framebuffer
#define C_HttpSrv_ContentType_Image_X_Icon 0x00000040 // "image/x-icon", typically used by a web server's "favicon.ico"
#define C_HttpSrv_ContentType_Video        0x00000080
#define C_HttpSrv_ContentType_Audio        0x00000100
#define C_HttpSrv_ContentType_Audio_X_Wav  0x00000200  // "audio/x-wav"
#define C_HttpSrv_ContentType_Application              0x00000800
#define C_HttpSrv_ContentType_Application_URLEncoded   0x00001000
#define C_HttpSrv_ContentType_Application_JSON         0x00002000 // "application/json", not "text/json", used by openABK
#define C_HttpSrv_ContentType_Application_Octet_Stream 0x00004000 // "application/octet-stream"
#define C_HttpSrv_ContentType_Application_XHTML        0x00008000 // "application/xhtml" (often used by browsers)
#define C_HttpSrv_ContentType_Multipart_FormData 0x00010000
#define C_HttpSrv_ContentType_Multipart_Other    0x00020000 /* not supported ! */
#define C_HttpSrv_ContentType_Message      0x00040000
#define C_HttpSrv_ContentType_XML          0x00100000
#define C_HttpSrv_ContentType_XML_UTF8     0x00200000 // -> "Content-Type:text/xml; charset=utf-8"  (used when returning anything.xml)

typedef enum // supported "Transfer-Encoding" types :
{ C_HttpSrv_XferEncoding_None = 0,  // no "Transfer-Encoding" header at all
  C_HttpSrv_XferEncoding_Chunked    // "Transfer-Encoding: chunked" (e.g. for STREAMS with unlimited length)
} T_HttpSrv_XferEncoding;

typedef struct t_HttpMultipartProcessor
{
  enum  // MIME Multipart decoder states (internal) :
   { C_HttpMPState_Off = 0,
     C_HttpMPState_ParsingHeaders    ,   // still waiting for "Content-Type" *IN THE HEADER*
     C_HttpMPState_WaitingForBoundary,   // waiting for the next boundary (inside or outside FILE DATA)
     C_HttpMPState_ParsingContentHeader, // parsing the "CONTENT-HEADER" (*after* a boundary)
   } nState;

  BYTE bParserInputBuf[1024]; // a FIFO-like buffer (but NOT CIRCULAR) which feeds the parser
  int  iParserInputBufUsage;  // number of characters (bytes) currently in bParserInputBuf[]

  char sz80Boundary[84];    // multipart boundary, copied from "Content-Type".. "boundary=" .
    // Must be long enough for stuff like "\r\n---------------------------265001916915724" [used by Firefox]
  int  iBoundaryLength;     // number of characters (or bytes) used as the current boundary pattern,
                            // INCLUDING the leading "\r\n--" (4 bytes) .

  // Note: the DWORD members below are BITWISE COMBINEABLE flags. 'Enums' are utterly useless here.
  DWORD dwContentTypeFromHeader;
  DWORD dwContentTypeOfCurrentPart;
  DWORD dwDispositionType;  // usually C_HttpSrv_DispositionType_FormData (?)

  long i32TotalContentLength;   // from "Content-Length" for the entire MESSAGE BODY (not a netto filesize!)
  long i32TotalContentByteCnt;  // counts from 0 to i32TotalContentLength-1 (theoretically)

} T_HttpMultipartProcessor;

typedef struct t_HttpSession
{
  char sz40UserName[44]; // user name extracted from query string; input for the later session ID
  char sz40UserCall[44]; // user's callsign from query string; input for the later session ID
  char sz40Password[44]; // optional password, also from a query string; also affects the session ID
  union
   { BYTE  b[4]; // "his" (the remote peer's) IPv4 address; also used for the session ID
     DWORD dw;   // THE SAME, accessable as a 32-bit doubleword
   } b4HisIP;
  int iPermissions; // .. delivered by CwNet_CheckUserAndGetPermissions() for the current user.
                    // Contains a BITWISE COMBINATION of flags defined in CwNet.h,
                    // e.g. CWNET_PERMISSION_NONE / TALK / TRANSMIT / CTRL_RIG .
  char sz8SessionID[12]; // 'session ID' (purpose explained in HttpSrv_UpdateSessionID() )
} T_HttpSession;


typedef struct t_HttpInstance // instance to serve ONE REMOTE CLIENT ..
{ // Only required to deliver 'larger blocks' or dynamic content,
  //      or even an audio stream (if HTML5 wouldn't make everything so
  //      terribly complex.. like Web Sockets to stream audio to Javascript).
  // Error response like the infamous '404', or attempty to connect the
  // server without a valid user name in the query string, etc,
  // will be immediately rejected WITHOUT allocating a T_HttpInstance at all.
  //
  // Note the kludgy solution for the 'chicken and egg problem'
  //  in CwNet.h, where the 'T_CwNetClient' struct contain A POINTER
  //  to a 'struct t_HttpInstance', even though the struct itself is
  //  unknown at that point. This way, "CwNet.h" can be included
  //  BEFORE actually defining the T_HttpInstance. Downside:
  //  The 'struct t_HttpInstance POINTER' in T_CwNetClient can only be
  //  a POINTER, not a 'sub-structure'. So HttpServer.c will ALLOCATE
  //  memory for a T_HttpInstance only when required, via HttpSrv_CreateInstance().
  T_CwNet       *pCwNet;  // address of a 'CW Network' SERVER INSTANCE
  T_CwNetClient *pClient; // address of a 'CW Network Client Instance'
        // (which performs the basic network I/O, usually WITHOUT HTTP)
  struct t_OWRX_Client *pOWClient; // address of an 'OpenWebRX-alike' client

  int iServerOptions; // bitwise combineable HTTP_SERVER_OPTIONS :
#     define HTTP_SERVER_OPTIONS_ENABLE   0x0001 // general 'enable HTTP server' flag
#     define HTTP_SERVER_OPTIONS_RESTRICT 0x0002 // restrict access to visitors providing a valid user name (already in the very first 'GET' request, via Query-String) ?
      // Note: In this version adapted for the 'CW Network',
      //       T_HttpServer.iServerOptions is copied from
      //       T_CwNet.iHttpServerOptions whenever a new T_HttpServer 'client instance'
      //       is created - see implementation of HttpSrv_CreateInstance().
  T_HttpSession *pSession; // client IP address, user name, callsign, password, permissions, and session-ID as string
  BOOL fIsNewSession; // Flag set in HttpSrv_CreateInstance() if connection made with an "unknown IP".
                      // In that case, the query-string MUST contain a valid user name, etc.
  BOOL fIsNewConnection; // Flag set in HttpSrv_CreateInstance() when an ALREADY REGISTERED IP
                      // (remote client; web browser) opened a new connection/socket/etc .
  char sz255RequestedURLWithoutQuery[256]; // originally requested URL (URIs are for snobs),
      // already converted from "URL encoding" into "plain text" .
      // "A URL is a specific type of URI". And that's all we need here. Basta.
      // "The part that makes something a URL is the combination of the name and an access method."
      //
      // Example:
      //       http://de.wikipedia.org/wiki/Uniform_Resource_Locator .....
      //       |__|| ||______________||____________________________| |____|
      //       |   | |    host             path with segments        truncated here:
      //       |   | |                     separated by slashes      query, fragment
      //  Scheme/  | |
      //  protocol | '- omitted here / not supported: "userinfo", ending with an '@'
      //           |
      //  For nit-pickers: "the colon after the scheme doesn't belong to the scheme"
      //  See also: YHF_Tools/YHF_Inet.c :: INET_SplitURL() .

  char sz255QueryString[256]; // current query string, including the leading "?" (if any)

  int nMethod; // possible values : HTTP_METHOD_GET, POST, PUT, DELETE, etc (?)
  DWORD dwAcceptedContentTypes; // "Content-Types that are acceptable for the response." From a requests "Accept"-field. Contains a BITWISE COMBINATION of C_HttpSrv_ContentType_xyz ...
  T_HttpSrv_ContentType nContentType; // HTTPSRV_CONTENT_TYPE_TEXT_HTML, .., HTTPSRV_CONTENT_TYPE_APPL_JSON,...
  T_HttpSrv_XferEncoding nTransferEncoding; // so far, only C_HttpSrv_XferEncoding_None or C_HttpSrv_XferEncoding_Chunked

  long i32ReqContentLength;  // from the GET REQUEST header, "Content-Length: "
  long i32RespContentLength; // for the RESPONSE header, initially 0 as long as there's no "response body",
                             // meaningless for "endless" streams (audio, etc)
  T_HttpServerState nServerState; // C_HttpSrvState_PASSIVE, C_HttpSrvState_RECEIVING_REQUEST, C_HttpSrvState_RECEIVED_REQUEST,
      // C_HttpSrvState_SENDING_HEADER, C_HttpSrvState_SENT_HEADER, C_HttpSrvState_SENDING_DATA, C_HttpSrvState_SENDING_ERROR, etc...
  DWORD dwConnectionPersistance;  // bitwise combination parsed from field "Connection: keep-alive, Upgrade, ..." . Required for WebSockets.
  DWORD dwUpgradeRequested;       // bitwise combination parsed from field "Upgrade: ", e.g. C_HttpSrv_Upgrade_WebSocket
  BOOL isMobile; // FALSE : guess the visitor is NOT a mobile phone / smartphone / tablet
                 // TRUE  : guess the visitor is a mobile phone / smartphone / tablet
        // (guesswork based on a few keywords in the 'User-Agent' like "Android", "Mobile",..)

  // Buffer for the received request header (from client to server, ending with "\r\n\r\n") :
  long i32ReqHeaderLength;  // number of characters in the buffer, used by HttpSrv_GetValueFromRequestHeader()
  #define C_HTTP_SRV_MAX_SIZE_OF_REQUEST_HEADER 2048
  char szRequestHeader[C_HTTP_SRV_MAX_SIZE_OF_REQUEST_HEADER+4];  // contains all HTTP (-GET) request HEADER LINES (no posted data or "files")
       // From  en.wikipedia.org/wiki/List_of_HTTP_header_fields :
       // > The standard imposes no limits to the size of each header field name or value,
       // > or to the number of fields. However, most servers, clients,
       // > and proxy software impose some limits for practical and security reasons.
       // > For example, the Apache 2.3 server by default limits the size of each field
       // > to 8190 bytes, and there can be at most 100 header fields in a single request.
       // (DL4YHF: We don't play that game here. A total of 2048 characters for
       //          THE ENTIRE header (with all "fields", misnomer: "headers")
       //          should be sufficient for *this* appplication. )

  BOOL fReadingFileFromDisk;    // see UPT_HttpSrv.c::HttpSrv_AccessPseudoFileSystem() .
  long i32NettoFileSize;        // aka "Content-Length" for transmission via HTTP
  BOOL fIsDynamicPage;          // TRUE=may insert dynamic values; FALSE=don't
        // Note: In the older mickey-mouse HTTP server, the 'dynamic values'
        //  were inserted *AFTER* packetizing. This was totally unacceptable.
        //  Since 2009-03-06, the values are inserted in the HTML pages
        //  *in a single block which can grow*.
  char sz15Password1[16];        // password (ASCII string) set on certain forms
  BOOL logged_in_with_password;  // TRUE=yes, FALSE=no
  char sz80ErrorMessage[84];     // Server Error Message displayed as plain text somewhere

  BOOL finished_receiving_request; // set to TRUE when any Request-parser has detected the end of the ENTIRE POSTED MESSAGE.
  BOOL parsing_posted_response; // FALSE when processing a GET-request, TRUE when preparing the response for a POST-request.

  /* path+names of all files uploaded (posted) to the server, stored in a RAMDISK: */
  // ex: # define C_HTTP_SRV_MAX_UPLOADED_FILES 20
  // ex: char szFileList[C_HTTP_SRV_MAX_UPLOADED_FILES][C_HTTP_SRV_MAX_PATH+1];
  // ex: int  i32FileSizes[C_HTTP_SRV_MAX_UPLOADED_FILES];
  // ex: int  nPostedFiles; // number of files saved; also used as index into szFileList[]
  // ex: BOOL fSaveFile;   // internal flag: TRUE when the current file shall be saved
                           // when an EMPTY LINE after the CONTENT HEADER was found
  // ex: BOOL fSavingFile; // internal flag: TRUE when T_HttpRequestInstance.pfs
         // is currently being used to SAVE a file during an HTTP-multipart-POST
  // ex: int  i32BytesWrittenToFile;
  // ex: int  iApplicationSpecific1;  // see SpecHttpSrv.cpp (meaning doesn't matter here)
  // ex: void *pvApplicationSpecific; // initially NULL, may later point to custom data

  #define C_HttpSrv_RX_BUFFER_SIZE 2048 /* must be large enough even for "chatty" clients ! */
                                        /* (at least large enough for one Ethernet frame)   */
  BYTE bRxBuffer[C_HttpSrv_RX_BUFFER_SIZE+4]; // "small" buffer for received bytes for WebSockets, etc
        //  ,-----------------------------'
        //  '--> ".. plus 4" to safely add a trailing zero for our 'string parsers'.
        //  This is NOT the buffer in which a received file/firmware/application
        //  will be temporarily stored ! To receive 'very large objects',
        //  sent from web browser to the embedded device via HTTP POST,
        //  there was only ONE GLOBAL BUFFER. See HttpSrv_AllocLargeRxBuffer() !
        //  (we possibly never need this in the 'downsized' server for the "Cw Network")
  int  iRxBufferNumBytesPending; // number of bytes in bRxBuffer[] not "digested"
                  // by the application yet. This is NOT a circular FIFO index !


  // Buffer for assembling the response HEADER (ending with "\r\n\r\n") :
  #define C_HttpSrv_RESP_BUFFER_SIZE 1024
  char szResponseHeader[C_HttpSrv_RESP_BUFFER_SIZE];  // contains the HTTP response HEADER
  int iRespHeaderLength, iRespHeaderTxIndex;
      // (an extra buffer for the reponse HEADER(s) is necessary because
      //  in many cases, the RESPONSE BODY must be assembled EARLIER to find
      //  the actual "Content-Length"; the latter affects the size of the
      //  total length of the RESPONSE HEADER (at least one of the many text lines).

  // Buffer for assembling the response BODY (HTML, binary stuff like "favicon.ico",
  //                                          chunked audio, etc):
  #define C_HttpSrv_TX_BUFFER_SIZE 8192
  BYTE bTxBuffer[C_HttpSrv_TX_BUFFER_SIZE]; // buffer for transmission and to assemble TX-data
        // HttpSrv_FlushTxBuffer() will try to send whatever remains in bTxBuffer.
        // If the (TCP- / Berkeley) "Socket" is currently UNABLE to send data,
        // for example after send() returned WSAEWOULDBLOCK, but Mr Winsick
        // hasn't sent the FD_WRITE message yet ...
  int  iTxBufferLength, iTxBufferTxIndex;
        // To determine the FREE space in bTxBuffer[], use HttpSrv_GetFreeSpaceInTxBuffer().
        //
  BOOL fWebSocketSendPong; // TRUE = "transmission of a Web-Socket-'Pong' still pending"
  double dblUnixTimeOfStart;        // <- set immediately after accepting, for bandwidth-calculations etc
  double dblUnixTimeOfLastActivity; // <- used to prevent "inactive connection", especially for WebSockets
  double dblCurrentUnixTime;        // <- periodically updated, avoids excessive calls
                                    //    of UTL_GetCurrentUnixDateAndTime() .

  // To receive large 'requests' via HTTP POST, with files in MIME "multipart" form-data,
  // the following parser was added (2012-08-03) to process POSTed multipart messages:
  T_HttpMultipartProcessor MultipartProcForPostData;

  int   iStreamType; // 0="not an ENDLESS stream but a file", or one of the following:
# define C_HttpSrv_StreamType_None        0
# define C_HttpSrv_StreamType_VorbisAudio 1
# define C_HttpSrv_StreamType_WebSocket   2

  BYTE *pbBufferInMemory;  // NULL when not allocated (and a real DISK FILE shall be used)
  DWORD dwNumAllocdBytesInMemory;   // number of bytes allocated for 'pbBufferInMemory'
  DWORD dwBufInMemHead;    // 'head' index into pbBufferInMemory[] where the next byte will be WRITTEN .
  DWORD dwBufInMemTail;    // 'tail' index into pbBufferInMemory[] from where the next byte will be READ .
     // Replaces a "file pointer", when using pbBufferInMemory instead of sending a REAL FILE.
     // Used when 'reading' the "file". Runs from zero to dwBufInMemHead-1 .
     //
     //                                              dwBufInMemHead (head index)
     //                                                   |
     //                     _____________________________\|/______________
     //   pbBufferInMemory:|  already sent |not sent yet | unused         |
     //                    |_______________|_____________|________________|
     //                    |                |                             |
     //                   [0]              \|/           dwNumAllocdBytesInMemory
     //                                   dwBufInMemTail (tail index)
     //
     //
     //
     // To send an endless STREAM (rather than a FILE with limited size),
     // the HTTP server will periodically invoke the following callback function:
  BOOL (*StreamOutCallback)( struct t_HttpInstance *pHttpInst );
        // If the application has 'more to send' (for example, from an audio stream)
        // it may call HttpSrv_AppendToDataBlock() from StreamOutCallback().
        // Return value from StreamOutCallback:
        //  TRUE:  Keep the connection open, even if there is nothing in the buffer now;
        //  FALSE: Close the connection, there will not be anything else to send.

  double   dblUnixTimeOfConnection;

  BOOL  fFileRequestedFromApplication;  // balances HTTP_MSG_FILE_REQUEST + HTTP_MSG_FILE_FINISHED
  BOOL  fWaitingToContinueWrite; // added 2008-01-02, see long story in OnWrite() !
  BOOL  fCallOnConnClose;  // TRUE : must still call OnHttpServerConnClosed()
  BOOL  fUsedByOpenWebRX ; // TRUE when we need to call OWRX_OnConnectionClosed() on closing

  DWORD    i64TotalBytesRcvd;  // total number of bytes received on this socket (header(s) PLUS payload)
  DWORD    dwPayloadBytesSent; // number of PAYLOAD bytes sent to this socket from the CURRENT FILE (without HEADER[s])
  LONGLONG i64TotalBytesSent;  // number of bytes send including HEADER(s) and PAYLOAD .
           // Incremented while still running, and added to the server's TOTAL byte counter
           // when the connection is finally CLOSED - see
  int     nPostedFiles;
#   define C_HTTP_SRV_MAX_UPLOADED_FILES 1            
  // ex: HFILE hFile;         // set to INVALID_HANDLE_VALUE if invalid or not opened (micro$tuff)
  HANDLE  hFile;              // used by CreateFile, GetFileSize, etc etc
  DWORD dwFilePtr;            // not really a pointer, but the current file position (offset)


} T_HttpInstance;


//---------------------------------------------------------------------------
// Constants
//---------------------------------------------------------------------------
#define HTTP_METHOD_UNKNOWN 0
#define HTTP_METHOD_GET     1
#define HTTP_METHOD_POST    2
#define HTTP_METHOD_PUT     3
#define HTTP_METHOD_DELETE  4
#define HTTP_METHOD_OPTIONS 5
#define HTTP_METHOD_HEAD    6


extern const T_SL_TokenList HttpMethods[]; // tokens like "GET ", "POST ", "PUT "


// HTTP "Result codes", actually array indices into HttpSrv_szStatMsgs[] .
// Borland : "Redefinition von 'HTTP_STATUS_OK' ist nicht identisch".
//           Leider gibt der elende Compiler nicht an, in welchem Header
//           dieses Symbol SCHON VORHER definiert war.
//   Nach langer Suche fand sich eine (selbstverstndlich INKOMPATIBLE)
//   Defintion in C:\CBuilder6\Include\Mcf\afxisapi.h . Was fr ein dmlicher Name !
//   Noch eine (ebenfalls inkompatible) Definition stand in 'wininet.h' .
//   Sollen "winninet", "willinet" und "derfinet" bleiben wo der Pfeffer wchst.
//   Zhneknirschend musste WB die folgenden Definitionen (in HttpSrv.h) umbenennen:
//   Alt: #define HTTP_STATUS_OK 0    Neu: #define HTTP_STATUS__OK 0
#define HTTP_STATUS__OK            200 /* -> in HttpSrv_szStatMsgs[HTTP_STATUS__CREATED],     "200 OK"                */
#define HTTP_STATUS__CREATED       201 /* -> in HttpSrv_szStatMsgs[HTTP_STATUS__CREATED],     "201 Created"           */
#define HTTP_STATUS__ACCEPTED      202 /* -> in HttpSrv_szStatMsgs[HTTP_STATUS__ACCEPTED],    "202 Accepted"          */
#define HTTP_STATUS__NOCONTENT     204 /* -> in HttpSrv_szStatMsgs[HTTP_STATUS__NOCONTENT],   "204 No Content"        */
#define HTTP_STATUS__MOVEDPERM     301 /* -> in HttpSrv_szStatMsgs[HTTP_STATUS__MOVEDPERM],   "301 Moved Permanently" */
#define HTTP_STATUS__MOVEDTEMP     302 /* -> in HttpSrv_szStatMsgs[HTTP_STATUS__MOVEDTEMP],   "302 Moved Temporarily" */
#define HTTP_STATUS__NOTMODIFIED   304 /* -> in HttpSrv_szStatMsgs[HTTP_STATUS__NOTMODIFIED], "304 Not Modified"      */
#define HTTP_STATUS__BADREQUEST    400 /* -> in HttpSrv_szStatMsgs[HTTP_STATUS__BADREQUEST],  "400 Bad Request"       */
#define HTTP_STATUS__UNAUTHORIZED  401 /* -> in HttpSrv_szStatMsgs[HTTP_STATUS__UNAUTHORIZED],"401 Unauthorized"      */
#define HTTP_STATUS__FORBIDDEN     403 /* -> in HttpSrv_szStatMsgs[HTTP_STATUS__FORBIDDEN],   "403 Forbidden"         */
#define HTTP_STATUS__NOTFOUND      404 /* -> in HttpSrv_szStatMsgs[HTTP_STATUS__NOTFOUND],    "404 Not Found"         */
#define HTTP_STATUS__SERVERERROR   500 /* -> in HttpSrv_szStatMsgs[HTTP_STATUS__SERVERERROR], "500 Internal Server Error" */
#define HTTP_STATUS__NOTIMPLEMENTED 501 /* -> in HttpSrv_szStatMsgs[HTTP_STATUS__NOTIMPLEMENTED],"501 Not Implemented" */
#define HTTP_STATUS__BADGATEWAY    502 /* -> in HttpSrv_szStatMsgs[HTTP_STATUS__BADGATEWAY],  "502 Bad Gateway"       */
#define HTTP_STATUS__UNAVAILABLE   503 /* -> in HttpSrv_szStatMsgs[HTTP_STATUS__UNAVAILABLE], "503 Service Unavailable" */
#define HTTP_STATUS__SRV_TIMEOUT   504 /* -> in HttpSrv_szStatMsgs[HTTP_STATUS__SRV_TIMEOUT], "504 Server or Gateway Timeout" */
#define HTTP_STATUS__SWITCHING_PROTOCOLS 101 /* actually "HTTP/1.1 101" , in HttpSrv_szStatMsgs "101 Switching Protocols" */
#define HTTP_STATUS__INSUFF_STORAGE 506 /* -> in HttpSrv_szStatMsgs[HTTP_STATUS__INSUFF_STORAGE],"507 Insufficient Storage" */
#define HTTP_STATUS__LONG_POLL      600 /* * NOCH KEINE RESPONSE SENDEN ... warten auf 'Long Poll', noch keine sendefhige Response fertig */
#define HTTP_STATUS__SUSPICIOUS_REQ 601 /* "wenn Du noch einmal Bldsinn ausprobierst, fliegst Du fr immer 'raus" */
#define HTTP_STATUS__DONT_RESPOND   602 /* don't respond but immediately close the connection, without leaking out any info */
        // (the "visitor" seems to be a bad guy, see examples in Hwill be added to our blacklist, see HttpSrv_ParseRequest() )
#define HTTP_STATUS__NOT_MY_BUSINESS 603  // this value may be returned by
        // specialized server modules if THEY don't want to process the request,
        // but let the default HTTP server's file-request-handler
        //   (e.g. in SpecHttpSrv.cpp : OnHttpFileRequestOrCompletion() )
        // "serve" an ordinary file, loaded from the 'server pages' folder.
extern const T_SL_TokenList HttpStatusCodes[];


typedef enum  // Tokens for keywords in HTTP requests, mostly in the HEADER lines:
{  C_HttpSrv_Kwd_Unknown = 0,
   C_HttpSrv_Kwd_Host,
   C_HttpSrv_Kwd_User_Agent,
   C_HttpSrv_Kwd_Accept,
   C_HttpSrv_Kwd_Accept_Language,
   C_HttpSrv_Kwd_Accept_Encoding,
   C_HttpSrv_Kwd_DNT,
   C_HttpSrv_Kwd_Connection,
   C_HttpSrv_Kwd_Referer,
   C_HttpSrv_Kwd_Content_Type,
   C_HttpSrv_Kwd_Content_Length,
   C_HttpSrv_Kwd_Content_Disposition,
   C_HttpSrv_Kwd_Access_Control_Allow_Origin, // added 2016, important for CORS
   C_HttpSrv_Kwd_Upgrade, // added 2019, important for WebSockets
   C_HttpSrv_Kwd_future
} T_HttpSrv_KeywordToken;

// Possible values parsed from header-line: "Content-Type: ", stored in variables named 'nContentType' .
// Used in string table HttpSrv_ContentTypes[], access via HttpSrv_GetTokenFromTable() / HttpSrv_GetStringFromTable() .
// Later also used as BITWISE COMBINATION for the "Accept"-field in an HTTP request header .
#define T_HttpSrv_ContentType  unsigned long
#define C_HttpSrv_ContentType_Unknown      0
#define C_HttpSrv_ContentType_All          0x00FFFFFF
#define C_HttpSrv_ContentType_Text         0x00000001
#define C_HttpSrv_ContentType_Text_Plain   0x00000002 // -> "Content-Type:text/plain" (also used to POST FORMS!)
#define C_HttpSrv_ContentType_Text_HTML_ISO_8859_1 0x00000004 // -> "Content-Type:text/html; charset=ISO-8859-1"
#define C_HttpSrv_ContentType_Text_HTML    0x00000008 // "text/html" without a 'charset'-specification
#define C_HttpSrv_ContentType_Image        0x00000010
#define C_HttpSrv_ContentType_Image_BMP    0x00000020 // "image/bmp", used by an embedded web server to retrieve the framebuffer
#define C_HttpSrv_ContentType_Image_X_Icon 0x00000040 // "image/x-icon", typically used by a web server's "favicon.ico"
#define C_HttpSrv_ContentType_Video        0x00000080
#define C_HttpSrv_ContentType_Audio        0x00000100
#define C_HttpSrv_ContentType_Audio_X_Wav  0x00000200  // "audio/x-wav"
#define C_HttpSrv_ContentType_Application              0x00000800
#define C_HttpSrv_ContentType_Application_URLEncoded   0x00001000
#define C_HttpSrv_ContentType_Application_JSON         0x00002000 // "application/json", not "text/json" ..
#define C_HttpSrv_ContentType_Application_Octet_Stream 0x00004000 // "application/octet-stream"
#define C_HttpSrv_ContentType_Application_XHTML        0x00008000 // "application/xhtml" (often used by browsers)
#define C_HttpSrv_ContentType_Multipart_FormData 0x00010000
#define C_HttpSrv_ContentType_Multipart_Other    0x00020000 /* not supported ! */
#define C_HttpSrv_ContentType_Message      0x00040000
#define C_HttpSrv_ContentType_XML          0x00100000
#define C_HttpSrv_ContentType_XML_UTF8     0x00200000 // -> "Content-Type:text/xml; charset=utf-8"  (used when returning anything.xml)

// Bitwise combineable flags parsed from header-line "Connection: ",
//  used in string-table HttpSrv_ConnectionTypes[], and for dwConnectionPersistance :
#define C_HttpSrv_ConnectionPersistance_Unknown    0
#define C_HttpSrv_ConnectionPersistance_KeepAlive  1
#define C_HttpSrv_ConnectionPersistance_Close      2
#define C_HttpSrv_ConnectionPersistance_Upgrade    4 /* inevitable for "WebSockets" */

// bitwise combineable flags for keywords in the header-line "Upgrade:" -> dwUpgradeRequested :
#define C_HttpSrv_DontUpgrade       0
#define C_HttpSrv_Upgrade_WebSocket 1
#define C_HttpSrv_Upgrade_TLS       2
#define C_HttpSrv_Upgrade_HTTP2     4

// Bitwise combineable flags parsed from header-line "Disposition-Type" / HttpSrv_DispositionTypes[] :
#define C_HttpSrv_DispositionType_Unknown          0
#define C_HttpSrv_DispositionType_FormData         1
#define C_HttpSrv_DispositionType_Attachment       2

// "Opcodes" in a WebSocket frame (fragment). Bitwise combineable with the
//  "FIN"-flag, from RFC6455, anno 2011.
// Not limited to the SERVER side, thus prefix "Http_", not "HttpSrv_" .
// Used as parameter 'iWSFrameType' in OnHttpWebSocketFrameRcvd() .
#define Http_WebSocketFrame_FinalFragment 0x80 // marks the "final" frame (SET when *not* fragmented. Doesn't have anything to do with a request to close the socket.)
#define Http_WebSocketFrame_SingleFragment 0x80 // <- Leaves less doubt than just "FIN" (from RFC6455). USUALLY SET, NOT CLEARED !
#define Http_WebSocketFrame_Mask4Opcode   0x0F // mask for the "opcodes" (lower 4 bits) defined below
#define Http_WebSocketFrame_Continuation  0x00
#define Http_WebSocketFrame_Text          0x01
#define Http_WebSocketFrame_Binary        0x02
#define Http_WebSocketFrame_Rsvd_03       0x03 // "reserved for further non-control frames"...
#define Http_WebSocketFrame_Rsvd_04       0x04
#define Http_WebSocketFrame_Rsvd_05       0x05
#define Http_WebSocketFrame_Rsvd_06       0x06
#define Http_WebSocketFrame_Rsvd_07       0x07
#define Http_WebSocketFrame_CloseConn     0x08 // "denotes a connection close"
#define Http_WebSocketFrame_Ping          0x09
#define Http_WebSocketFrame_Pong          0x0A // not kidding. We can play ping pong with this.
#define Http_WebSocketFrame_Rsvd_0B       0x0B
#define Http_WebSocketFrame_Rsvd_0C       0x0C
#define Http_WebSocketFrame_Rsvd_0D       0x0D
#define Http_WebSocketFrame_Rsvd_0E       0x0E
#define Http_WebSocketFrame_Rsvd_0F       0x0F


//---------------------------------------------------------------------------
// Data Types (in addition to those in CwNet.h) ..
//---------------------------------------------------------------------------



//---------------------------------------------------------------------------
// Global variables  (?)
//---------------------------------------------------------------------------

  // (as few as possible. Most "per-client" data are in T_CwNetClient )
#undef EXTERN
#ifdef _I_AM_HTTP_SERVER_  /* for single-source-variables in this header */
  #define EXTERN
#else
  #define EXTERN extern
#endif

EXTERN char HttpSrv_sz80HttpServerHostnameOrPublicIP[84]; // in the original OpenWebRX,
                // this was used as the "WebSocket base URL", to avoid problems
                // with proxies. Expanded in the HTML template from "%[WS_URL]" :
                //  > Since we usually don't have a domain name,
                //  > use the PUBLIC IP, retrieved by a service like
                //  > http://checkip.dyndns.com/ ,  http://icanhazip.com/ , etc.


EXTERN int  HttpSrv_iSizeofFavicon; // 0 when invalid, > 0 if we have a valid "favicon.ico"
EXTERN BYTE HttpSrv_bFavicon[8192]; // filled somewhere at RUN-TIME ..
                // the Remote CW Keyer's program icon symbol was 40 * 40 pixels;
                // 40 * 40 * 4 bytes_per_pixel plus some overhead is > 6400 !


//---------------------------------------------------------------------------
// Function Prototypes
//---------------------------------------------------------------------------

#ifndef CPROT   // for peaceful co-existence of C and C++ ...
#ifdef __cplusplus
 #define CPROT extern "C"
#else
 #define CPROT
#endif  // not "cplusplus" ?
#endif // ndef CPROT ?

CPROT BOOL HttpSrv_CreateInstance( T_CwNet *pCwNet, T_CwNetClient *pClient );
CPROT void HttpSrv_DeleteInstance( T_CwNet *pCwNet, T_CwNetClient *pClient );
CPROT int  HttpSrv_GetClientIndex(T_HttpInstance *pHttpInst); // -> 0..CWNET_MAX_CLIENTS(!)

CPROT void HttpSrv_OnReceive( T_CwNet *pCwNet, // [in] single SERVER INSTANCE
              T_CwNetClient *pClient,          // [in] instance data for a CLIENT
            BYTE *pbRcvdData, int nBytesRcvd); // [in] received stream segment

CPROT void HttpSrv_OnPoll( T_HttpInstance *pHttpInst );

CPROT const char *HttpSrv_StatusCodeToString( int iHttpStatusCode );
CPROT const char *HttpSrv_MethodToString( int iHttpMethod );

CPROT int  HttpSrv_GetFreeSpaceInTxBuffer( T_HttpInstance *pHttpInst );

CPROT void HttpSrv_SendErrorAndCloseConnection( T_CwNet *pCwNet,
              T_CwNetClient *pClient, const char *pszBadURL, int iHttpStatusCode );


// Fasten seat belts .. Web Sockets sound a bit like "TCP/IP sockets in a browser"
//   but they are TERRIBLY COMPLICATED stuff, full of counter-intuitive details
//   like "socket mask" (only allowed -and even mandatory- in ONE direction), etc.
//   Note: A "WebSocket frame" requires SIX additional bytes in the transmit-
//         buffer, so SUBTRACT SIX from the value returned by
//         HttpSrv_GetFreeSpaceInTxBuffer() to find the MAXIMUM PAYLOAD SIZE
//         that can be sent in the next call of HttpSrv_SendWebSocketFrame().
CPROT int HttpSrv_AssembleWebSocketFrameHeader( BYTE *pbFrameHeader,
            int iWSFrameType, int iPayloadLength, const BYTE *pb4WebSocketMask);
CPROT BOOL HttpSrv_SendWebSocketFrame(T_HttpInstance *pHttpInst, int iWSFrameType,
            const BYTE *pbSource, int iPayloadLength, const BYTE *pb4WebSocketMask);


#endif // ndef  _HTTP_SERVER_H ?

#endif // SWI_USE_HTTP_SERVER ?

