// File:  C:\cbproj\Remote_CW_Keyer\HttpServer.c
// Date:  2024-01-20
// Author:  Wolfgang Buescher (DL4YHF)
// Purpose: Simple Socket-based HTTP server (~dummy) for the
//          'Remote CW Keyer'. So far, only good to test connections
//          (and port forwarding, dynamic DNS, and all that stuff)
//          using a web browser to replace a 'Remote CW CLIENT'.
//      This is by no means a "universal, standalone" HTTP server,
//      but infact only called from module CwNet.c if the remote
//      client's initial data (from client to server)
//      "look like they MAY be HTTP" (hypertext transfer protocol,
//      in this case a TCP segment beginning with a "method"
//      like "GET ", "POST ", or even "PUT " and ending with an empty line.
//
// Most recent modifications:
//   2024-11-25: Replaced HUNDREDS of "char pointers" by "const char *"
//               because 'modern compilers' (e.g. "ISO C++11") are
//               very pedantic about char pointers, and refuse to
//               use STRING LITERALS as a "char *" !
//    After that, most of the non-VCL-modules could be compiled
//    with the old Borland C++Builder V6
//    and Embarcadero C++Builder V12 "Athens", "Community Edition".
//               Details about the migration process from BCB V6 to V12
//               in DL4YHF's C:\cbproj\YHF_Tools\StringLib.h .
//


#include "switches.h" // project specific 'compilation switches' like SWI_USE_DSOUND

#ifndef  SWI_USE_HTTP_SERVER   // even if HttpServer.c shall NOT be used in the actual build,
# define SWI_USE_HTTP_SERVER 0 // there's no need to remove this "optional" module from the project..
#endif // ndef SWI_USE_HTTP_SERVER  // .. the presence shall be defined in "switches.h"
#if(SWI_USE_HTTP_SERVER)       // this conditional ends near end-of-file ..

#include <string.h> // not only string functions in here, but also memset()
#include <stdio.h>  // no standard I/O but string functions like sprintf()
#ifndef _WINDOWS    // obviously compiling for a PC, so ..
# include "winsock2.h"  // .. use Microsoft's flavour of the Berkeley Socket API
  // Note: never include winsock2.h *after* windows.h .. you'll fry in hell,
  //       because "windows.h" #includes "winsock.h", not "winsock2.h" .
#endif

#include "StringLib.h"  // DL4YHF's string library, with stuff like SL_ParseIPv4Address()
#include "Utilities.h"  // stuff like UTL_iWindowsVersion, UTL_iAppInstance, ShowError(), etc
#include "Timers.h"     // high-resolution 'current time'-function, T_TIM_Stopwatch, etc.
#include "Elbug.h"      // the basic 'Elbug' functions (plain C)
#include "SampleRateConv.h"  // convert from 48 to 8 kSamples/sec and back,
                        // for 'audio streaming' between server and client[s].
#include "ALawCompression.h" // simple A-Law compression / decompression
                        // to stream audio between server and client[s].
#include "CwDSP.h"      // 'CW Digitial Signal Processor' (audio I/O and processing)
#include "CwNet.h"      // "CW Network" (not HTTP-based main part)
#include "Inet_Tools.h" // Base64 encoding/decoding, SHA1 calculation, etc..
#define _I_AM_HTTP_SERVER_ 1 // for single-source-variables in the following header:
#include "HttpServer.h" // simplistic HTTP server, used by CwNet.c, too
#include "OpenWebRX_Server.h"

//----------------------------------------------------------------------------
// "Internal" constants, macros, lookup tables, and similar
//----------------------------------------------------------------------------

#define CWNET_SOCKET_POLLING_INTERVAL_MS 20 // interval, in milliseconds,
                       // at which all of the active client <-> server sockets
                       // are PERIODICALLY POLLED for transmission, even if
                       // nothing has been received for a while.

#ifndef  sizeof_member
# define sizeof_member(type,member) sizeof(((type*)0)->member) /* omg .. but works */
#endif

const T_SL_TokenList HttpMethods[] =
{ { "GET ",    HTTP_METHOD_GET    },
  { "POST ",   HTTP_METHOD_POST   },
  { "PUT ",    HTTP_METHOD_PUT    },
  { "DELETE ", HTTP_METHOD_DELETE },
  { "OPTIONS ",HTTP_METHOD_OPTIONS}, // <-- added 2015-12-08 to make a paranoid web browser happy
  { "HEAD ",   HTTP_METHOD_HEAD   }, // "HEAD" is like "GET" without a body in the response:
    // > The HTTP HEAD method requests the headers that would be returned if
    // > the HEAD request's URL was instead requested with the HTTP GET method.
    // > For example, if a URL might produce a large download, a HEAD request
    // > could read its Content-Length header to check the filesize
    // > without actually downloading the file.
    // Here, in Spectrum Lab's HTTP server, "HEAD" makes almost no sense because
    // most 'documents' delivered by "GET" don't have a known size in advance !
  { NULL, 0 } // "all zeros" mark the end of the list
};

const T_SL_TokenList HttpStatusCodes[] =
{ { "OK",                       200  },
  { "Created",                  201  },
  { "Accepted",                 202  },
  { "No Content",               204  },
  { "Moved Permanently",        301  },
  { "Moved Temporarily",        302  },
  { "Not Modified",             304  },
  { "Bad Request",              400  },
  { "Unauthorized",             401  },
  { "Forbidden",                403  },
  { "Not Found",                404  },
  { "Internal Server Error",    500  },
  { "Not Implemented",          501  },
  { "Bad Gateway",              502  },
  { "Service Unavailable",      503  },
  { "Server Timeout",           504  },
  { "Switching Protocols",      101  },
  { "Insufficient Storage",     507  },
  { "Unavailable/Suspicious",   451  },
  { "Forbidden",                403  },
  { "Conflict",                 409  },
  { NULL, 0 } // "all zeros" mark the end of the list
}; // end HttpStatusCodes[]


const T_SL_TokenList HttpSrv_Keywords[] = // stuff like "\r\nUser-Agent: " -> C_HttpSrv_Kwd_User_Agent, etc
{ // From yet another RFC (guess it was RFC 2616) :
  // > Each header field consists of a name followed by a colon (":")
  // > and the field value. Field names are case-insensitive.
  // > The field value MAY be preceded by any amount of LWS, though a single SP is preferred.
  // In other words: the space after the colon is "preferred", but OPTIONAL. Don't assume anything !
  // Also DON'T ASSUME a client will use the typical CamelCasing seen in almost any HTTP example.
  //   ( All but a few use "Content-Length" not "content-length". But both MUST be accepted,
  //     because HTTP header field names are case INSENSITIVE (as per RFC 2616).
  //     But beware: OTHER elements in this table, like the "HOST"-stuff, and the HTTP-"VERBS"/"methods",
  //     are CASE-SENSITIVE ! !  What a mess, this HTTP (or is it Http / http ? ) !
  //  -> Fixed by implementing SL_SkipOneOfNTokens(), which is case-INSENSITIVE .
  //     Be extremely careful when deciding WHICH of the above to use !
  //   )
  //
  { "HOST:",                 C_HttpSrv_Kwd_Host                 },
  { "User-Agent:",           C_HttpSrv_Kwd_User_Agent           },
  { "Accept:",               C_HttpSrv_Kwd_Accept               },
  { "Accept-Language:",      C_HttpSrv_Kwd_Accept_Language      },
  { "Accept-Encoding:",      C_HttpSrv_Kwd_Accept_Encoding      },
  { "DNT:",                  C_HttpSrv_Kwd_DNT                  },
  { "Connection:",           C_HttpSrv_Kwd_Connection           }, // followed by stuff like "keep-alive", etc
  { "Referer:",              C_HttpSrv_Kwd_Referer              },
  { "Content-Type:",         C_HttpSrv_Kwd_Content_Type         },
  { "Content-Length:",       C_HttpSrv_Kwd_Content_Length       },
  { "Content-Disposition:",  C_HttpSrv_Kwd_Content_Disposition  },
  { "Access-Control-Allow-Origin:", C_HttpSrv_Kwd_Access_Control_Allow_Origin }, // added 2016, important for CORS
  { "Upgrade:",              C_HttpSrv_Kwd_Upgrade              },
  { NULL, 0 } // marker for end-of-list
}; // end HttpSrv_Keywords[]


// A list of all supported ("allowed") header field names, reported by HttpSrv_ListAllowedHeaders() .
// Sent by a server after the client made a CORS request. Search for "Access-Control-Allow-Headers" !
const T_SL_TokenList HttpSrv_AllowedHeaderFieldNames[] =
{
  { "HOST",                 C_HttpSrv_Kwd_Host                 },
  { "User-Agent",           C_HttpSrv_Kwd_User_Agent           },
  { "Accept",               C_HttpSrv_Kwd_Accept               },
  { "Accept-Language",      C_HttpSrv_Kwd_Accept_Language      },
  { "Accept-Encoding",      C_HttpSrv_Kwd_Accept_Encoding      },
  { "DNT",                  C_HttpSrv_Kwd_DNT                  },
  { "Connection",           C_HttpSrv_Kwd_Connection           },
  { "Referer",              C_HttpSrv_Kwd_Referer              },
  { "Content-Type",         C_HttpSrv_Kwd_Content_Type         },
  { "Content-Length",       C_HttpSrv_Kwd_Content_Length       },
  { "Content-Disposition",  C_HttpSrv_Kwd_Content_Disposition  },
  { NULL, 0 } // marker for end-of-list
  // Note that there are zillions of other 'field names' which we
  // do NOT support here (at least not yet, but who knows..) .
  // Here is a list of the stuff which OTHERS support :
  //   "Accept, Accept-Charset, Accept-Encoding, Accept-Language,
  //    Access-Control-Allow-Credentials, Access-Control-Allow-Headers,
  //    Access-Control-Allow-Methods, Access-Control-Allow-Origin,
  //    Access-Control-Expose-Headers, Access-Control-Max-Age,
  //    Access-Control-Request-Headers, Access-Control-Request-Method,
  //    Cache-Control, Connection,
  //    Content-Encoding, Content-Length, Content-Type, Cookie,
  //    DNT, Date, Expires, HTTP_CLIENT_IP, HTTP_COMING_FROM, HTTP_VIA, Host,
  //    If-Modified-Since, Keep-Alive, Origin, Pragma, REMOTE_ADDR,
  //    Referer, Server, Set-Cookie, Srv, Transfer-Encoding, User-Agent, Vary,
  //    X-Content-Type-Options, X-CustomHeader, X-DNS-Prefetch-Control,
  //    X-Forwarded-For, X-Forwarded-Host, X-Forwarded-Server, X-Frame-Options,
  //    X-Modified, X-OTHER, X-Originating-IP, X-Output, X-PING, X-PINGOTHER,
  //    X-Powered-By, X-Real-IP, X-Redirect, X-Requested-With, X-Robots-Tag,
  //    X-XSS-Protection, X-Xss-Protection" .
  // (found at http://icompile.eladkarako.com/right-so-access-control-expose-headers-does-not-support-wildcard/ ,
  //  and the list certainly keeps growing at a deadly pace ;o)
  //
}; // end HttpSrv_AllowedHeaderFieldNames[]


// "Internet Media Types" / "HTTP Content Types" (just a few; we don't support all)
const T_SL_TokenList HttpSrv_ContentTypes[] = // stuff like "text" -> C_HttpSrv_ContentType_Text, etc
{   // Access via HttpSrv_GetTokenFromTable() / HttpSrv_GetStringFromTable() !
    // LONGER STRINGS FIRST to simplify the string-comparison in HttpSrv_GetTokenFromTable() .
  { "text/html; charset=ISO-8859-1", C_HttpSrv_ContentType_Text_HTML_ISO_8859_1 },
  { "text/html",                     C_HttpSrv_ContentType_Text_HTML    },
  { "text/plain",                    C_HttpSrv_ContentType_Text_Plain   },
  { "text",                          C_HttpSrv_ContentType_Text         },
  { "image/x-icon",                  C_HttpSrv_ContentType_Image_X_Icon },
  { "image",                         C_HttpSrv_ContentType_Image        },
  { "video",                         C_HttpSrv_ContentType_Video        },
  { "audio/x-wav",                   C_HttpSrv_ContentType_Audio_X_Wav  },
  { "audio",                         C_HttpSrv_ContentType_Audio        },
  { "application/x-www-form-urlencoded", C_HttpSrv_ContentType_Application_URLEncoded}, // unfortunately the default when POSTing "forms" !
  { "application/json",              C_HttpSrv_ContentType_Application_JSON },
  { "application/octet-stream",      C_HttpSrv_ContentType_Application_Octet_Stream },
  { "application/xhtml",             C_HttpSrv_ContentType_Application_XHTML},
  { "application/xml",               C_HttpSrv_ContentType_XML          },
  { "application",                   C_HttpSrv_ContentType_Application  },
  { "multipart/form-data",           C_HttpSrv_ContentType_Multipart_FormData },
  { "multipart",                     C_HttpSrv_ContentType_Multipart_Other},
  { "message",                       C_HttpSrv_ContentType_Message      },
  { "text/xml; charset=utf-8",       C_HttpSrv_ContentType_XML_UTF8     }, // (beware the spaces)
  { "text/xml",                      C_HttpSrv_ContentType_XML          },

  { NULL, 0 } // marker for end-of-list
}; // end HttpSrv_ContentTypes[]

// "Connection Persistance" types. The only string known so far is "keep-alive" .
//  (in RFC2616, you won't find any of the possible values for the "Connection" header field !)
const T_SL_TokenList HttpSrv_ConnectionTypes[] = // stuff like "keep-alive" (after "\r\nConnection: ")
{  { "keep-alive",          C_HttpSrv_ConnectionPersistance_KeepAlive },
   { "Close",               C_HttpSrv_ConnectionPersistance_Close     },
   { "Upgrade",             C_HttpSrv_ConnectionPersistance_Upgrade   },
   { NULL, 0 } // marker for end-of-list
}; // end HttpSrv_ConnectionTypes[]

// "Disposition Types" from the "Content-Disposition" line in multipart messages:
const T_SL_TokenList HttpSrv_DispositionTypes[] =
{
  { "form-data",            C_HttpSrv_DispositionType_FormData },
  { "attachment",           C_HttpSrv_DispositionType_Attachment},
  { NULL, 0 } // marker for end-of-list
}; // end HttpSrv_DispositionTypes[]

  // Diverse HTTP-Header ("Antworten vom HTTP-Server an den Client").
  // Aus  http://de.wikipedia.org/wiki/Hypertext_Transfer_Protocol :
  // > Sobald der Header mit einer Leerzeile abgeschlossen wird,
  // > sendet der Computer, der einen Web-Server (an Port 80) betreibt,
  // > seinerseits eine HTTP-Antwort zurck. Diese besteht aus
  // >  * den Header-Informationen des Servers,
  // >  * einer Leerzeile
  // >  * und dem tatschlichen Inhalt der Nachricht,
  // >      also dem Dateiinhalt der infotext.html-Datei.
  // >
  // > Derzeit werden zwei Protokollversionen, HTTP/1.0 und HTTP/1.1, verwendet.
  // > Bei HTTP/1.0 wird vor jeder Anfrage eine neue TCP-Verbindung aufgebaut
  // > und nach bertragung der Antwort standardmig vom Server wieder geschlossen.
  // > Sind in ein HTML-Dokument beispielsweise zehn Bilder eingebettet,
  // > so werden insgesamt elf TCP-Verbindungen bentigt, um die Seite auf
  // > einem grafikfhigen Browser aufzubauen.
  //
  // In many cases (for example openABK) the TCP connection must be kept open
  // (alias "keep-alive" in geek speak). In that case, we need a
  // "Content-Length" already in the header, which adds some complexity
  // because the entire content would have to be assembled 'in memory'
  // instead of generating it 'on the fly' (while already sending the first data).

const char HttpSrv_szGetResponseHttpOkCrNl[] = "HTTP/1.1 200 OK\r\n";

const char HttpSrv_szGetResponseTextHtml_ISO[] = // 1st thing our server sends to a client in response to "GET"
{
  "HTTP/1.0 200 OK\r\n"           // protocol ver 1.0, code 200, reason OK
  // No-No: "HTTP/1.1 200 OK\r\n"  // protocol ver 1.1 WOULD REQUIRE A "Content-Length" field (-> must be built in RAM, cannot use headers from ROM) !!
  "Content-Type:text/html; charset=ISO-8859-1\r\n"   // type of data we want to send
  "\r\n"                          // indicate end of HTTP-header
}; // end HttpSrv_szGetResponseTextHtml_ISO[]

#if(0)
const char HttpSrv_szGetResponseTextHtml_ISO_NoCache[] = // similar, but prevents caching for 'dynamic' content
{
  "HTTP/1.0 200 OK\r\n"           // protocol ver 1.0, code 200, reason OK
  // No-No: "HTTP/1.1 200 OK\r\n"  // protocol ver 1.1 WOULD REQUIRE A "Content-Length" field (-> must be built in RAM, cannot use headers from ROM) !!
  "Content-Type:text/html; charset=ISO-8859-1\r\n"   // type of data we want to send
  "Cache-Control: no-cache, no-store\r\n" // Extrawurst fr den dmmsten Browser der Welt
  // Leider hat die obige 'No-Cache', 'No-Store'-Anweisung keinen Effekt bei IE8.
  // Das elende gottverdammte Drecksding sieht es weiterhin nicht ein,
  // die nach diesem Response-Header gesendete "Datei" (z.B. source.htm) neu zu laden.
  // Stattdessem zog das elende gottverdammte Drecksding die "Datei", auch beim
  // Laden per Javascript, jedesmal aus seinem Scheiss-Browser-Cache.
  // Knnte daran liegen, das das oberschlaue Drecksding die Cache-Control-Anweisung
  // ignoriert, weil dies in HTTP/1.0 noch nicht vorgesehen war.
  // Leider 'kann' der embedded Web-Server bislang kein HTTP/1.1 .....
  // Bei vernnftigen Browsern (wie z.B. Iron, Firefox) hatte 'No-Cache'
  // den unangenehmen Nebeneffekt, dass der 'Back'-Button nicht mehr den
  // vom Bediener gewohnten Effekt hatte; darum 'weg damit' !
  // WB gab an dieser Stelle (zunchst?) auf, und wandte sich wichtigeren Dingen zu
  // als weitere Extrawrste fr den 'dmlichsten Browser unter der Sonne' zu braten.
  // Verwenden sie einen VERNNFTIGEN Browser, aber keinesfalls IE (Internet Explorer) !
  //   (gute Erfahrungen wurden anno 2013 z.B. mit Firefox und 'SRWare Iron',
  //    einem Chrome-Clone angeblich ohne Google-Schnffelfunktionen, gemacht)
  "\r\n"                          // indicate end of HTTP-header
}; // end HttpSrv_szGetResponseTextHtml_ISO_NoCache[]
#endif // (0)

const char HttpSrv_szGetResponseImageBmp[] = // HTTP response to send bmp-encoded image
{
  "HTTP/1.0 200 OK\r\n"          // protocol ver 1.0, code 200, reason OK
  // No-No: "HTTP/1.1 200 OK\r\n"  // protocol ver 1.1 WOULD REQUIRE A "Content-Length" field (-> must be built in RAM, cannot use headers from ROM) !!
  "Content-Type:image/bmp\r\n"   // type of data we want to send
  "\r\n"                         // indicate end of HTTP-header
  // See   http://www.w3schools.com/media/media_mimeref.asp   for details !
};

const char HttpSrv_szGetResponseImageXIcon[] = // HTTP response to send 'ico' images
{
  "HTTP/1.0 200 OK\r\n"           // protocol ver 1.0, code 200, reason OK
  // No-No: "HTTP/1.1 200 OK\r\n"  // protocol ver 1.1 WOULD REQUIRE A "Content-Length" field (-> must be built in RAM, cannot use headers from ROM) !!
  "Content-Type:image/x-icon\r\n" // type of data, used for 'favicon.ico'
  "\r\n"                          // indicate end of HTTP-header
};

const char HttpSrv_szPostResponseTextHtml[] = // response after receiving a 'POST'-request
{
  "HTTP/1.0 200 OK\r\n"           // protocol ver 1.0, code 200, reason OK
  // No-No: "HTTP/1.1 200 OK\r\n"  // protocol ver 1.1 WOULD REQUIRE A "Content-Length" field (-> must be built in RAM, cannot use headers from ROM) !!
  "Content-Type:text/html; charset=ISO-8859-1\r\n"   // type of data we want to send
  "\r\n"                          // indicate end of HTTP-header
};

const char HttpSrv_szGetResponseTextPlain[] = // response after receiving a 'GET'-request for "*.txt"
{
  "HTTP/1.0 200 OK\r\n"           // protocol ver 1.0, code 200, reason OK
  // No-No: "HTTP/1.1 200 OK\r\n"  // protocol ver 1.1 WOULD REQUIRE A "Content-Length" field (-> must be built in RAM, cannot use headers from ROM) !!
  "Content-Type:text/plain\r\n"   // type of data we want to send (encoding type unknown)
  "\r\n"                          // indicate end of HTTP-header
};


const char HttpSrv_szGetResponseTextXml[] = // response header for XMLHttpRequest
{
  "HTTP/1.0 200 OK\r\n"           // protocol ver 1.0, code 200, reason OK
  // No-No: "HTTP/1.1 200 OK\r\n"  // protocol ver 1.1 WOULD REQUIRE A "Content-Length" field (-> must be built in RAM, cannot use headers from ROM) !!
  // ex:  "Content-Type: text/html\r\n"   // type of data we want to send
  // Irgendeine oberschlaue Fehlerkonsole in irgendeinem oberschlauen Browser maulte:
  // > Fehler: Die Zeichenkodierung des HTML-Dokuments wurde nicht deklariert.
  // Daraufhin wurde folgende Modifikation getestet, um in der verdammten Konsole
  // nur noch die WIRKLICH WICHTIGEN Fehler zu sehen:
  "Content-Type:text/xml; charset=utf-8\r\n"   // type of data we want to send .
           // Note: "ANSI" strings must be re-encoded into UTF-8 before sending !
  // Even more fun with THE MOST STUPID WEB BROWSER UNDER THE SUN (Internet Explorer):
  // > By default, IE will cache results from AJAX requests. (..)
  // From www.itworld.com/development/303295/ajax-requests-not-executing-or-updating-internet-explorer-solution
  // > The Symptoms
  // > * Requests work just fine the first time you try
  // > * As data is modified, you realize that youre still seeing old results
  // > * Everything appears to work correctly in other browsers
  // > * You're slowly going insane
  // > One of the solutions:
  // > You can also prevent caching by sending additional headers along with your response.
  // > By specifying the "Cache-Control" header with a value of "no-cache,no-store"
  // > and returning it with the web service response you can instruct the browser 
  // > not to cache the result.
  // WB: Ok, let's try. We already had so much fun with IE6,7,8,9 ... :
  // "Cache-Control: no-cache, no-store\r\n" // Extrawurst fr den dmmsten Browser der Welt
  "\r\n"                          // indicate end of HTTP-header (XML FOLLOWS)
  // If the content type is "text/xml", the response *MUST* contain a valid piece of XML,
  // even if there's nothing between '<response>' and '</response>' !
  // Otherwise, some browsers (Firefox, Iron) indicated the XMLHttpRequest as 'pending'
  // for an awfully long time (over 10 seconds), even though the request was delivered
  // (from browser to the embedded device) within a few milliseconds (2013-10-15) !
};// end HttpSrv_szGetResponseTextXml[]
  // AFTER this 'Response Header' (with content-type text/xml),
  // most browsers expect WELL-FORMATTED XML, as in the following example
  // from from http://www.the-art-of-web.com/javascript/ajax :
  // > <?xml version="1.0" encoding="UTF-8" standalone="yes"?>
  // > <response>
  // >   <command method="setcontent">
  // >     <target>example2</target>
  // >     <content>hello world!</content>
  // >   </command>
  // > </response> 
const char HttpSrv_szBeginOfXmlResponse[] = // ..part of the RESPONSE for an XMLHttpRequest..
{ "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\r\n"
  " <response>\r\n"
};// end HttpSrv_szBeginOfXmlResponse[]
const char HttpSrv_szEndOfXmlResponse[] = // an 'empty' response for an XMLHttpRequest
{ " </response>\r\n\r\n"
};// end HttpSrv_szEndOfXmlResponse[]

const char HttpSrv_szGetResponseJSON[] = // HTTP response header for JSON
{
  "HTTP/1.0 200 OK\r\n"           // protocol ver 1.0, code 200, reason OK
  // No-No: "HTTP/1.1 200 OK\r\n"  // protocol ver 1.1 WOULD REQUIRE A "Content-Length" field (-> must be built in RAM, cannot use headers from ROM) !!
  // ex:  "Content-Type: text/json\r\n" // type of data we want to send
  "Content-Type:application/json\r\n"   // type of data we want to send
  // "Cache-Control: no-cache, no-store\r\n" // Extrawurst fr den dmmsten Browser der Welt (hatte leider keinen Effekt)
  "\r\n"                        // indicate end of HTTP-header (JSON data FOLLOWS)
  // Note: There was a long fruitless debate about the 'correct' MIME type
  //       in a JSON response header. Only stupid browsers don't understand
  //       the one-and-only official type, which is "application/json".
  //       We don't support those antique stupid browsers
  //       in this memory-constrained environment (embedded web server) !
};// end HttpSrv_szGetResponseJSON[]


const char HttpSrv_szAccessControlAllowOrigin_All[] = // added 2016-01-07 for CORS ("Cross-Origin Resource Sharing")
{ "Access-Control-Allow-Origin: *\r\n"
}; // end HttpSrv_szAccessControlAllowOrigin_All[]


// Valid SSI directives. Only a small subset of 'Server Side Includes' supported here.
//  (some call this 'directives', Apache calls these 'elements'. We don't.)
typedef enum  // tokens for SSI directives:
{ C_HttpSrv_SSI_Directive_Unknown = 0,
  C_HttpSrv_SSI_Directive_Include,  // includes a fragment of text
  C_HttpSrv_SSI_Directive_Exec,     // executes a program, script, or shell command on the server
  C_HttpSrv_SSI_Directive_Echo      // displays content of an HTTP environment variable
} T_HttpSrv_SSI_Directive;
const T_SL_TokenList HttpSrv_SSI_Directives[] =
{ { "include",   C_HttpSrv_SSI_Directive_Include },
  { "exec",      C_HttpSrv_SSI_Directive_Exec    },
  { "echo",      C_HttpSrv_SSI_Directive_Echo    },
  { NULL, 0 } // marker for end-of-list

}; // end HttpSrv_SSI_Directives[]

// Known SSI parameters. At the moment, only some 'include' directives are really supported !
//  (some call these 'parameters', Apache calls them 'attributes'. We don't.)
typedef enum  // tokens for SSI parameters:
{ C_HttpSrv_SSI_Param_Unknown = 0,
  C_HttpSrv_SSI_Param_File,    // include a file, "relative to the directory of the current file"
  C_HttpSrv_SSI_Param_Virtual, // include a file, "relative to the domain root"
  C_HttpSrv_SSI_Param_CGI, C_HttpSrv_SSI_Param_CMD // <!-- all these are NOT supported -->
} T_HttpSrv_SSI_Parameter;
const T_SL_TokenList HttpSrv_SSI_Parameters[] =
{ { "file",     C_HttpSrv_SSI_Param_File    },
  { "virtual",  C_HttpSrv_SSI_Param_Virtual },
  { "cgi",      C_HttpSrv_SSI_Param_CGI     }, // "path to a CGI script" (not supported here)
  { "cmd",      C_HttpSrv_SSI_Param_CMD     }, // "path to a shell script" (not supported)
  { NULL, 0 } // marker for end-of-list
}; // end HttpSrv_SSI_Parameters[]

// Recognized keywords in the line (also called "header") beginning with "Upgrade:"
const T_SL_TokenList HttpSrv_UpgradeTypes[] =
{ { "websocket", C_HttpSrv_Upgrade_WebSocket },  // HTTP/2 ? Hamwa nich, kennwa aber :)
  { "TLS",       C_HttpSrv_Upgrade_TLS    },  // TLS ist zwar bekannt, wird aber nicht untersttzt. Basta.
  { "h2c",       C_HttpSrv_Upgrade_HTTP2  },  // HTTP/2 ? Hamwa nich, kennwa aber :)
  { NULL, 0 } // marker for end-of-list
}; // end HttpSrv_UpgradeTypes[]


// Recognized keys (short constant names) in this server's Query String :
#define C_HttpSrvKey_USER 1
#define C_HttpSrvKey_CALL 2
#define C_HttpSrvKey_PWD  3  // short for "password"
#define C_HttpSrvKey_SID  4  // short for "session ID", carried along from one PAGE to another
const T_SL_TokenList HttpSrv_QueryStringKeys[] =
{ { "user",    C_HttpSrvKey_USER },
  { "call",    C_HttpSrvKey_CALL },
  { "pwd",     C_HttpSrvKey_PWD  },
  { "sid",     C_HttpSrvKey_SID  },
  { NULL, 0 } // marker for end-of-list
}; // end HttpSrv_QueryStringKeys[]


#define C_HttpSrvPage_NotFound      0
#define C_HttpSrvPage_HeadStyles    1  // HeadStyles_css[] (hard coded)
#define C_HttpSrvPage_Root          2
#define C_HttpSrvPage_Statistics    3
#define C_HttpSrvPage_LiveData_js   4  // javascript fragment HTML_LiveData_js
#define C_HttpSrvPage_LiveData_htm  5  // HTML fragment loaded into <div id="LiveData"> by LiveData.js
#define C_HttpSrvFile_LiveAudio_ogg 6  // start STREAMING "LiveAudio.ogg" (for <audio> element without any Javascript)
#define C_HttpSrvPage_WebAudio_js   7  // javascript for LOW LATENCY audio output, using the browser's Web Audio API
#define C_HttpSrvPage_WebAudio_htm  8  // first test page in which the above JS was intended to run
#define C_HttpSrvFile_Favicon_ico   9  // the application's own "icon" (symbol) in an ancient ".ico" format

const T_SL_TokenList HttpSrv_BuiltInPages[] =
{ { "/",                 C_HttpSrvPage_Root         },
  { "/statistics/",      C_HttpSrvPage_Statistics   },
  { "/LiveData.js",      C_HttpSrvPage_LiveData_js  },
  { "/LiveData.htm",     C_HttpSrvPage_LiveData_htm },
  { "/LiveAudio.ogg",    C_HttpSrvFile_LiveAudio_ogg},
  { "/WebAudio.js",      C_HttpSrvPage_WebAudio_js  },
  { "/WebAudio.htm",     C_HttpSrvPage_WebAudio_htm },
  { "/HeadStyles.css",   C_HttpSrvPage_HeadStyles   },
  { "/favicon.ico",      C_HttpSrvFile_Favicon_ico  },

  { NULL, 0 } // marker for end-of-list
}; // end HttpSrv_BuiltInPages[]

extern const char JS_WebAudio[];   // advanced javascript in Remote_CW_Keyer/JS_WebAudio.c
extern const char HTML_WebAudio[]; // HTML document (template) in which the above JS shall run.

static const char JS_LiveData[] = // simplistic javascript for periodically refreshed 'Live Data'
"function getData()\r\n"
"{ var xhr = new XMLHttpRequest(); // Create XMLHttpRequest object\r\n"
"  xhr.onload = function() {       // When response has loaded..\r\n"
   // The following conditional check will not work locally - only on a server
"    if(xhr.status === 200) {      // .. and server status was ok ..\r\n"
"      document.getElementById('LiveData').innerHTML = xhr.responseText; // Update\r\n"
"    }\r\n"
"  };\r\n"
"\r\n"
// ex: "  xhr.open('GET', 'LiveData.htm?user=$USER$', true); // Prepare the request\r\n"
   // ,-------------------------|___________|
   // '--> Repeatedly sending the USER NAME kept the 'live data' running
   //      in the browser even after terminating and restarting the server,
   //      WITHOUT manually reloading the entire page ("/") .
   //      But repeatedly sending out the user name every few hundred milliseconds
   //      (over an unsecure HTTP connection) was considered not worth the effort.
   //      Thus back to the original Javascript:
"  xhr.open('GET', 'LiveData.htm', true); // Prepare the request\r\n"
"  xhr.send(null);           // Send the request\r\n"
"} // end getData()\r\n"
"\r\n"
"setInterval(getData, 1000); // start PERIODIC updates\r\n"
"\r\n"
"\r\n"; // end HtmlInclude_jsLiveData

static const char HTML_LiveData[] = // just an INCLUDED FRAGMENT ..
"<pre><code>\r\n"
"User Name       : $USER$\r\n"
"User Callsign   : $CALL$\r\n"
"Permissions     : $PERM$\r\n"
"Radio tuned to  : $FREQ$ in $MODE$\r\n"
"Date and time   : $DATE$ $TIME$\r\n"
"</code></pre>"; // end HTML_LiveData[]

static const char HTML_Root[] = // Original 'main page' with simple <audio> element ..
"<!doctype html>\r\n"
"<html>\r\n"
"<head>\r\n"
 // ex: "  <meta http-equiv=\"refresh\" content=\"2\">\r\n" // flickers like hell
"  <title>Remote CW Web Server - Main Page</title>\r\n"
"  <style type=\"text/css\" src=\"HeadStyles.css\">\r\n"
"  </style>\r\n"
"<script type=\"text/javascript\" src=\"LiveData.js\" defer></script>\r\n"
 "</head>\r\n"
"<body>\r\n"
"<h2>Remote CW Web Server - Main Page</h2>\r\n"
"<p>\r\n"
"This site is hosted by the <a href=\"https://www.qsl.net/dl4yhf/Remote_CW_Keyer/Remote_CW_Keyer.htm\">Remote CW Keyer</a>.<BR>\r\n"
// "<p>\r\n"
// "<img alt=\"Framebuffer\" src=\"screen1.bmp\" height=\"$SCH$\" width=\"$SCW$\">\r\n"
// "<p>\r\n"
"<audio controls preload=\"none\">\r\n"
"  <source src=\"/LiveAudio.ogg\" type=\"audio/ogg\">\r\n"
"</audio>\r\n"
"<div id=\"LiveData\">\r\n"
" No 'live data', waiting for Javascript.<br><br>\r\n"
"</div>\r\n"
 // In Firefox ON ANDROID / smartphone, without the css crap, the fonts in the <pre><code>
 // was MUCH TOO SMALL, while the font in the following line (with "See also:")
 // was MUCH TOO LARGE. No suprise to see these megatons of font-sizes
 // specified in PIXELS all around. What a pile of junk.
"<p>\r\n"
"See also: <a href=\"links.htm\">Links</a>, <a href=\"WebAudio.htm\">Web Audio test</a>.\r\n"
"\r\n"
"</body></html>\r\n"
"\r\n" ;  // end HTML_Root[]

static const char HTML_Statistics[] =
 "";
 // end HTML_Statistics[]


static const char HeadStyles_css[] =
  // The following tries to fix the bugged rendering of <pre><code> in Firefox on Android.
  // See stackoverflow.com/questions/36524272/why-does-firefox-not-honor-css-font-size-for-code-tags-wrapped-in-pre !
  // Unfortunately, the funny "monospace, monospace" hack doesn't work anymore.
"pre { font-family: monospace, monospace; }\r\n"
"code { font-family: monospace, monospace; font-size: 22px; }\r\n"
"\r\n";



//----------------------------------------------------------------------------
// Global (or at least static) variables ...
//----------------------------------------------------------------------------
#if(SWI_HARDCORE_DEBUGGING) // (1) = hardcore-debugging, (0)=normal compilation
 int HttpSrv_iLastSourceLine = 0;  // WATCH THIS after crashing with e.g. "0xFEEEFEEE" !
# define HERE_I_AM__HTTPSRV() HttpSrv_iLastSourceLine=__LINE__
     // (see complete list of other XYZ_iLastSourceLine variables to watch
     //  in C:\cbproj\Remote_CW_Keyer\Keyer_Main.cpp, near GUI_iLastSourceLine)
#else
# define HERE_I_AM__HTTPSRV()
#endif // SWI_HARDCORE_DEBUGGING ?

#define C_HttpSrv_MaxSessions 20
T_HttpSession HttpSrv_Sessions[C_HttpSrv_MaxSessions] = { 0 };


//----------------------------------------------------------------------------
// "Internal function prototypes" (functions called before their implementation)
//----------------------------------------------------------------------------
static int  HttpSrv_OnGET( T_HttpInstance *pHttpInst );
static void HttpSrv_CheckUserNameAndUpdatePermissions( T_HttpInstance *pHttpInst, char *pszUserName );
static void HttpSrv_CheckUserCallAndUpdatePermissions( T_HttpInstance *pHttpInst, char *pszUserCall );

static void HttpSrv_CheckPasswordAndUpdatePermissions( T_HttpInstance *pHttpInst, char *pszPassword );
static void HttpSrv_CheckSessionIDAndUpdatePermissions(T_HttpInstance *pHttpInst, char *pszSessionID );
static int  HttpSrv_FinishTransferChunk( BYTE *pbDest, int iNettoChunkSize ); // .. im RAM !
static void HttpSrv_GenerateSessionID( T_HttpSession *pSession, char *psz8SessionID );
static int  HttpSrv_StartStreamingLiveAudio( T_HttpInstance *pHttpInst );
static void HttpSrv_ContinueStreamingLiveAudio( T_HttpInstance *pHttpInst );
static int  HttpSrv_SendBinaryBlock( T_HttpInstance *pHttpInst, char *pszMimeType,
                                     BYTE *pbSource, int iContentLength );

// implementation of functions:

//---------------------------------------------------------------------------
int HttpSrv_CopyQuotedStringWithoutQuotes(
        const char **ppszSource,   // [in,out] sourcecode pointer (will be incremented)
        char *pszDest, int maxLen) // [out] string [in] maximum number of characters
  // Used, for example, to parse the (usually double-quoted) FILENAME in "content-disposition".
  // Terminator is either the last doublequote character (IF the string is double-quoted);
  // otherwise any SPACE-LIKE character acts as delimiter.
  // Return value: Length of the string copied into pszDest .
  // Note: HttpSrv_CopyQuotedStringWithoutQuotes() does NOT convert "percent-encoded" characters
  //       into normal, human-readable text, because that would remove some information
  //       which may be relevant for the caller (depending on the context) !
  //       To convert "URL-encoded" aka "Percent-encoded" text into plain ASCII,
  //       use INET_UrlEncodingToPlainText() .
{
  const char *cp = *ppszSource;
  int len = 0;

  // Look ahead for the first non-space character:
  while(*cp==' ')
   { ++cp;
   }
  if( *cp=='"' )  // it REALLY is a double-quoted string (usually a filename WITH SPACES):
   { ++cp;
     while( *cp!='"' && *cp!='\r' && *cp!='\n' && *cp!='\0' )
      { if( len<maxLen )
         { *pszDest++ = *cp;
           ++len;
         }
        ++cp;
      }
     if(*cp=='"')
      { ++cp;
      }
   }
  else // it's NOT a double-quoted string, so copy ONLY ONE "WORD" (without spaces)
   {
     while( *cp!=' ' && *cp!='\r' && *cp!='\n' && *cp!='\0' )
      { if( len<maxLen )
         { *pszDest++ = *cp;
           ++len;
         }
        ++cp;
      }
   }
  if( len<maxLen )
   { *pszDest = '\0';  // append a trailing zero only if the dest-buffer is long enough
   }
  *ppszSource = cp;
  return len;
} // end HttpSrv_CopyQuotedStringWithoutQuotes()


//---------------------------------------------------------------------------
const char *HttpSrv_StatusCodeToString( int iHttpStatusCode )
{ const char *pszStatusText = SL_GetStringFromTokenList( HttpStatusCodes, iHttpStatusCode );
  if( pszStatusText != NULL ) // oops.. HTTP status CODE for which there is no TEXT yet !
   {  return pszStatusText;
   }
  else
   {  return "?status?";
   }
}

//---------------------------------------------------------------------------
const char *HttpSrv_MethodToString( int iHttpMethod )
{ const char *pszMethod = SL_GetStringFromTokenList( HttpMethods, iHttpMethod );
  if( pszMethod != NULL ) // oops..
   {  return pszMethod;
   }
  else
   {  return "?method?";
   }
}

//---------------------------------------------------------------------------
BOOL HttpSrv_SkipCharsInHeaderFieldIncludingNextComma(
        const char **ppszSource ) // [in,out] sourcecode pointer (string and ONE comma will be skipped here if found)
  // Added 2015-12 to parse chatty HTTP header fields with 'unknown' values .
  // First used for "Accept:" in HttpSrv_ParseAcceptedContentTypes() .
  //
  // But beware (about the non-triviality of HTTP, seen somewhere):
  //  > "essentially each header has it's own Grammar"
  // (in SOME, but as usual NOT ALL "headers" the RFCs mention "OWS",
  //  typic obfuscating geek speak for "Optional WhiteSpace",
  //  in SOME OTHERS they use "RWS" ("Required WhiteSpace"), etc.
  //  Pfui Deibel. Frequently seen (but almost impossible to find in the RFC):
  //  Normal space characters AFTER a comma. We'll silently skip them here,
  //  without giving a damn about which f***g "header" (header-line)
  //  is currently being parsed.
  //

{ const char *pszSource = *ppszSource;
  while( (BYTE)*pszSource > 13 )
   { if( *pszSource == ',' )
      {  ++pszSource;        // skip the comma...
         while( *pszSource == ' ' ) // .. and normal space(s) AFTER(!) the comma
          { ++pszSource;  // (seen, for example, in the "Upgrade: "-header-line)
          }
         *ppszSource = pszSource;
         return TRUE;
      }
     ++pszSource;           // skip anything which isn't CR, NL, or COMMA
   }
  return FALSE;
} // end HttpSrv_SkipCharsInHeaderFieldIncludingNextComma()


//---------------------------------------------------------------------------
DWORD HttpSrv_ParseBitCombinationOfCommaSeparatedKeywords(
          const char *pszSource, // [in] chatty stuff in a comma-separated list of keywords, until end-of-line
          const T_SL_TokenList *pKeywordTable ) // [in] table with a few possible keywords
                // that may appear in the comma-separated list of values,
                // e.g. HttpSrv_ConnectionTypes[] or HttpSrv_UpgradeTypes[].
  // Returns a bitwise combination of the tokens (thus, powers of two)
  //         defined in the keyword table, e.g. HttpSrv_ConnectionTypes[] .
  // The string comparison (for the keywords) accepts ANY CASE, which is ok
  // for MOST but unfortunately not ALL HTTP "header lines" .
  //
{
  DWORD dwBitCombination = 0;
  DWORD dwTokenFromTable;

  while( (BYTE)*pszSource > 13 )
   { if( (dwTokenFromTable = SL_SkipOneOfNStrings_AnyCase( &pszSource, pKeywordTable )) > 0 )
      { dwBitCombination |= dwTokenFromTable;
      }
     else // no idea what the current 'word' at pszSource is; just skip it ..
      {   // ... there will be hundreds of keywords in HTTP & Co that we neither know nor need.
      }
     if( ! HttpSrv_SkipCharsInHeaderFieldIncludingNextComma( &pszSource ) )
      { break;  // no comma, guess this is the end of a single header-line, e.g. after "Accept:".
      }
   }
  return dwBitCombination;
} // end HttpSrv_ParseBitCombinationOfCommaSeparatedKeywords()


//---------------------------------------------------------------------------
DWORD HttpSrv_ParseAcceptedContentTypes(
          const char *pszSource ) // [in] chatty stuff like "text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8"
  // Returns a bitwise combination of the C_HttpSrv_ContentType_-constants
  // defined in UPT_HttpSrv.h, for example C_HttpSrv_ContentType_Unknown or
  //            C_HttpSrv_ContentType_Text_Plain | C_HttpSrv_ContentType_Application_JSON , etc etc etc etc.
  //
{
  DWORD dwAcceptedContentTypes = C_HttpSrv_ContentType_Unknown;
  DWORD dwContentType;

  // Keine Ahnung, wozu man diesen elenden Zirkus braucht, ... wer weiss das schon ?
  //  Vielleicht mchte der Client uns auf diesem Weg ja mitteilen, dass er einen bestimmten "Content-Type" *NICHT* untersttzt.
  //  Ein Browser ist i.A., je nach Art des Requests, manchmal "offen fr alles",
  //  und teilt uns dies mit dem folgendem, extrem schnatterhaften Header-Feld mit:
  // > Accept: text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8
  //  (was ungefhr bedeutet: "Am liebsten text/html, oder, oder, und; notfalls aber auch ALLES ANDERE (*/* = wildcard).
  // Kryptische Syntax (mit allen "Feinheiten", die so ein "Accept"-Feld im schlimmsten Fall enthalten kann), aus RFC2616 :
  // >       Accept         = "Accept" ":"
  // >                        #( media-range [ accept-params ] )
  // >
  // >       media-range    = ( "*/*"
  // >                        | ( type "/" "*" )
  // >                        | ( type "/" subtype )
  // >                        ) *( ";" parameter )
  // >       accept-params  = ";" "q" "=" qvalue *( accept-extension )
  // >       accept-extension = ";" token [ "=" ( token | quoted-string ) ]
  // >
  // (Anhand der oben zitierten kryptischen Syntax wrde man keine Komma-Aufzhlung erwarten;
  //  wer viel Zeit und Spass an brain teasern hat, kann ja mal versuchen,
  //  das 'versteckte Komma' fr die Aufzhlung zu finden. Ist halt "typisch RFC".. )
  while( (BYTE)*pszSource > 13 )
   { SL_SkipSpaces( &pszSource );  // <- returns the NUMBER OF SPACES (not the first non-space character anymore)
     if( *pszSource == '*' )
      { ++pszSource;  // skip the "*"
        // "*" or even "*/*" acts like a "wildcard", or, in the geek words of RFC2616 :
        // > The asterisk "*" character is used to group media types into ranges,
        // >  with "*/*" indicating all media types and "type/*" indicating
        // > all subtypes of that type. The media-range MAY include media type parameters
        // > that are applicable to that range.
        // HERE: Forget about subtypes, all we care about is "*/*" = "all media types" :
        dwAcceptedContentTypes |= C_HttpSrv_ContentType_All;
      }
     else if( (dwContentType = SL_SkipOneOfNTokens( &pszSource, HttpSrv_ContentTypes )) > 0 )
      { dwAcceptedContentTypes |= dwContentType; // turn the bulky monster into a single 32-bit DWORD ("bitwise combined")
      }
     else // no idea what the current 'word' at pszSource is; just skip it ..
      {   // ... further below, where the comma after the KNOWN "Content-Type"-tokens is skipped, too
      }
     if( ! HttpSrv_SkipCharsInHeaderFieldIncludingNextComma( &pszSource ) )
      { break;  // no comma, guess this is the end of the header field for "Accept" .
      }
   }
  return dwAcceptedContentTypes;
} // end HttpSrv_ParseAcceptedContentTypes()


//---------------------------------------------------------------------------
void HttpSrv_PrintStupidCheckboxState( char *pszDest, int iChecked )
  // Helper function to preset those stupid 'checkbox' controls in HTML.
  // Prints the "value" of an HTML "checkbox" into a string.
{
  // Purpose: In HTML, checkboxes ("input type="checkbox") REALLY suck.
  //       They are not capable of POSTing simple TRUE / FALSE,
  //       depending on their checked/unchecked stat to the server
  //       when submitting a form. Instead, they POST their stupid "value"
  //       when checked; and they don't post ANYTHING when unchecked.
  //       This is REALLY, REALLY stupid for most if not call cases.
  //       CHECKBOXES (in HTML) ARE EVIL !  As someone put it:
  // > This is just a rant.
  // > I am so sick of validating forms. I do all that I can to make 
  // > it easy and whatnot, but it still comes back to spite me !
  // > Here are two examples of things that are dumb:
  // > 
  // >     Checkboxes
  // > 
  // > So html checkboxes are SO DUMB. 
  // > If they are checked, the value is set to "on." 
  // > That's annoying alone, but if the checkbox is not set 
  // > it doesn't even get submitted ! Anyway, that's pretty annoying. (...)
  //
  // See also http://planetozh.com/blog/2008/09/posting-unchecked-checkboxes-in-html-forms/ :
  //
  // > I needed to have a value for each checkbox in the $_POST array, 
  // > either blank/off/empty or checked/on/whatever.
  // > The solution I found was simple: before each checkbox,
  // > add a hidden field with the same name:
  // > HTML:
  // >  <form method="post" action="">
  // >       <input type="hidden" name="checkbox" value="">
  // >       <input type="checkbox" name="checkbox">
  // >  </form>
  // > This way, there is always a value for field checkbox
  // > in the $_POST array: either "" if the checkbox was unchecked,
  // > or "on" if it was checked. I've uploaded a simple HTML form 
  // > if you're curious to try: regular way, without the "special hidden" fields, 
  // > or without the duplicate hidden fields. Submit the form with all fields 
  // > left to empty and see the resulting $_POST array.
  // WB: This doesn't work too nice, at least not with Firefox.
  //      Indeed when the checkbox had been CHECKED BY THE USER (not checked BY DEFAULT),
  //      the POST content contained two name/value pairs, for example:
  //      bootup_display=0       [this was the stuff from the "hidden" input control]
  //      bootup_display=1       [this was the value POSTED from the damned checkbox]  .
  //   Better avoid those brain-damaged checkboxes completely, and use something like
  //   'yes/no dropdowns' instead ? No thanks; dropdown boxes ("dropdown menus") are ugly.
  //   Solution:  
  //   An initially UN-SELECTED checkbox  in stupid chatty HTML :
  //    <input type="hidden" name="bootup_display" value="0">
  //    <input type="checkbox" name="bootup_display" value="1" checked="checked"> Bootup Display
  //
  //   The same checkbox  in stupid HTML, but initially SELECTED :
  //    <input type="hidden" name="bootup_display" value="0">
  //    <input type="checkbox" name="bootup_display" value="1"> Bootup Display
  // 
  //   Note: The stupid checkbox still is NOT ABLE TO CHANGE THE VALUE.
  //         It can still either POST the value, or don't post the value at all.
  //         Simply posting the STATE (checked/unchecked) seems to be impossible
  //         without a stupid bunch of obfuscated Javascript (which wouldn't work
  //         on Browsers with Javascript disabled; and there are good reasons
  //         to disable Javascript by default .. ) .
  //   The EMBEDDED WEB SERVER cares for the problem (double-posting the "value")
  //    by simply using the LAST value received through the POSTed content.
  //    For example, when receiving (via POST) the content ..
  //         bootup_display=0
  //         bootup_display=1
  //    .. the web server uses the LAST value specified, and ignores the dummy sent before.
  //       This way, if a webpage (like setup.htm) CONTAINS A CONTROL FOR THE "bootup_display",
  //       we will always receive a state. Unfortunately, due to the incredible stupidity
  //       of the 'checkbox' element, there's no way to tell if the state (checked/unchecked)
  //       has been modified by the operator or not.
  //
  // When GENERATING the 'setup' page in the server, some of those stupid fucking checkboxes
  // shall be properly PRE-SET (checked!) depending on the current state.
  // When POSTING the modified 'setup' back to the server, those stupid fucking checkboxes
  // should ideally simply post their names (WHEN THE STATE HAS BEEN CHANGED BY THE USER,
  // as most other, not-as-sick-as-a-checkbox controls do). 
  // With the "hidden" dummy control as shown above, TWO DIFFERENT TAG COMBINATIONS
  // are produced here; depending on the stupid "checked" / "unchecked" state
  // (if those brain-damaged checkboxes supported something like "checked=0" or "checked=1"
  //  it would all be fine, but HTML, like many cases of "web programming", are NOT LOGICAL) .
  // In the HTML-"template" for setup.htm, one of those dreadful checkboxes is defined as:
  //
  //      <input type="checkbox" name="dhcp_enable" $CFG$dhcp_enable> Enable DHCP <br>
  //
  // Thus, for one of those stupid fucking checkboxes,  $CFG$dhcp_enable must expand 
  //       into the following ugly bunch :
  //      
  if( iChecked )  // print a fragment of a "checkbox" in HTML code which is INITIALLY CHECKED:
   { strcpy( pszDest, "value=\"1\" checked=\"checked\"" );  
     // > According to HTML5 drafts, the checked attribute is a "boolean attribute",
     // >  and "The presence of a boolean attribute on an element represents the true value, 
     // >  and the absence of the attribute represents the false value."    (heavens, no! HTML SUCKS !!)
     // > It is the name of the attribute that matters, and suffices.
     // > Thus, to make a checkbox initially checked, you use
     // >       <input type=checkbox checked>
     // > By default, in the absence of the checked attribute, a checkbox is initially unchecked:
     // >       <input type=checkbox>
     // > Keeping things this way keeps them simple, but if you need to 
     // > conform to XML syntax (i.e. to use HTML5 in XHTML linearization), 
     // > you cannot use an attribute name alone. Then the allowed (as per HTML5 drafts) 
     // > values are the empty string and the string checked, case insensitively. Example:
     // >       <input type="checkbox" checked="checked"
   }  
  else
   { strcpy( pszDest, "value=\"1\""  );  // when NOT CHECED, the stupid fucking "checkbox" won't post this value !
   }
} // end HttpSrv_PrintStupidCheckboxState()


//---------------------------------------------------------------------------
void HttpSrv_AppendCheckbox( char **ppszDest, const char * pszEndstop,
            const char *pszName,    // [in] "id" and/or "name" of the control (for antique browsers, because THOSE MADMEN KEEP CHANGING THE RULES)
            const char *pszCaption, // [in] text that shall appear right next to the checkbox  
                  BOOL fChecked)    // [in] TRUE=checked, FALSE=unchecked
  // Called from a few 'page generators' within an HTML-'form' to append a checkbox
  //    which reflects the current state of 'fChecked' .
  // When clicked, the checkbox submits / POSTS its new state 
  // identified by its 'name' or 'id' (both are emitted here). 
  // Details in the function body.
{
  char *cp = *ppszDest;
  SL_AppendPrintf( &cp, pszEndstop, "<input type=\"hidden\" name=\"%s\" value=\"0\">\r\n", pszName );
     // Details about the above hidden dummy below - it's important to send (post) also the UNCHECKED state !
  SL_AppendPrintf( &cp, pszEndstop, "<input type=\"checkbox\" id=\"%s\" name=\"%s\" onchange=\"submit();\" ",
                                                                       pszName,     pszName );
  HttpSrv_PrintStupidCheckboxState( cp, fChecked ); // YES, HTML CHECKBOXES ARE STUPID !
  SL_SkipToEndOfString( (const char**)&cp );
     // ,---------------|____________|
     // '--> Why do we need this bloody cast, Mr. Pedantic Compiler ?
     //      "cp" points to a char, so "&cp" is a pointer to a pointer to a char.
     //      SL_SkipToEndOfString() expects a "const char **", which means
     //      it promises not to modify the CHAR(s), so what's so "suspicious"
     //      about the original code ( SL_SkipToEndOfString( &cp ); ) ?
  SL_AppendPrintf( &cp, pszEndstop, "> %s ", pszCaption );
     // The stupid HTML checkbox created this way will be:
     //   <input type="checkbox" id="PeriodicUpdate" name="periodic_update" 
     //          onchange="submit();" value="1" checked="checked"> Periodic Update<br>  
     // OR (with HttpSrv_fPeriodicUpdate=FALSE) :
     //          onchange="submit();" value="1"> Periodic Update<br>
     // When clicking the stupid HTML 'checkbox' to toggle its state, 
     // the control's "onchange" fires, which in this case calls the FORM's "submit"-method,
     // which in turn causes the browser to send an HTTP "POST" with the following content:
     //     "periodic_update=0&periodic_update=1"  (the first key=value pair from the hidden dummy)
     // OR (with HttpSrv_fPeriodicUpdate=FALSE) :
     //     "periodic_update=0"   (the stupid HTML checkbox doesn't send ANYTHING when not checked)
     // which some time later will be parsed in HttpSrv_OnPost() .
     // We're doomed to do it this way because otherwise, the simple 'window.location.reload(true)'
     // would always switch the goddamned stupid HTML 'checkbox' back to the original state !
     // By storing the state HERE, ON THE SERVER SIDE (in HttpSrv_fPeriodicUpdate),
     // the same user selection can also be re-cycled on OTHER server pages.
     // Fortunately, at least IRON (a Chromium clone) didn't change the vertical
     // scrolling position during a 'page update' by Javascript 'window.location.reload(true)'.
     // More recent versions of Firefox were a show stopper:
     // When setting the checkmark 'Periodic Update' on various pages,
     // Firefox starts bugging the user with an alert like this :
     //  > "Besttigen
     //  >  Um diese Seite anzuzeigen, mssen die von Firefox gesammelten Daten erneut gesendet werden,
     //  >  wodurch alle zuvor durchgefhrten Aktionen wiederholt werden (wie eine Suche oder eine
     //  >  Bestellungsaufgabe). "
     // Na toll, schlauer Fuchs. Wir bestellen hier nichts, sondern wollen EINFACH NUR
     // die Seite neu vom Server laden.
     // Scheiss-Javascript, Scheiss-HTML, warum mssen einfache Dinge immer so ausarten ?!
  *ppszDest = cp;
} // end HttpSrv_AppendCheckbox()

//---------------------------------------------------------------------------
const char * HttpSrv_FindBuiltInPage( const char *pszFilename ) // ... or fragment for SSI ..
  // Find a "page", or the fragment of an HTML document, formerly in Flash/ROM,
  //                now (in the Remote CW Keyer's simplified HTTP server)
  //                in one of of the "built-in pages" as a const C string.
  // These "pages" are also used by the Server-Side-Include !
  // [in] pszFilename: something like "index.htm", "remctrl.htm", "setup.htm", etc.
{
  int iBuiltInPage = SL_SkipOneOfNTokens( &pszFilename, HttpSrv_BuiltInPages );
  switch( iBuiltInPage )
   {
     case C_HttpSrvPage_HeadStyles   : return HeadStyles_css;
     case C_HttpSrvPage_Root         : return HTML_Root;
     case C_HttpSrvPage_Statistics   : return HTML_Statistics;
     case C_HttpSrvPage_LiveData_js  : return JS_LiveData;
     case C_HttpSrvPage_LiveData_htm : return HTML_LiveData;
     case C_HttpSrvPage_WebAudio_js  : return JS_WebAudio;
     case C_HttpSrvPage_WebAudio_htm : return HTML_WebAudio;
     // No-No: case C_HttpSrvFile_LiveAudio_ogg: ... // ENDLESS "stream", cannot use strlen() on this !
     // No-No: case C_HttpSrvFile_Favicon_ico  : ... // BINARY "file", cannot use strlen() on this !
     case C_HttpSrvPage_NotFound     :
     default :
          break;
   } // end switch( iBuiltInPage )
  return NULL;
} // end HttpSrv_FindBuiltInPage()

//---------------------------------------------------------------------------
void HttpSrv_PrepareBuildingResponseHeaders( T_HttpInstance *pHttpInst,
            char **ppszDest, const char **ppszEndstop ) // [out] pointers for string builder functions
  // Prepares two a nice plain old portable 'char pointers'
  // to assemble the 'response' in pHttpInst->szResponseHeader[] .
{
  pHttpInst->iRespHeaderLength = pHttpInst->iRespHeaderTxIndex = 0;
  *ppszDest = (char*)pHttpInst->szResponseHeader; // start address of a string buffer for the response-header-lines
  *ppszEndstop = (const char*)pHttpInst->szResponseHeader + C_HttpSrv_RESP_BUFFER_SIZE - 1;
} // end HttpSrv_PrepareBuildingResponseHeaders()

//---------------------------------------------------------------------------
int HttpSrv_FinishBuildingResponseHeaders( T_HttpInstance *pHttpInst,
        char *cpEnd ) // [in] char pointer incremented by the 'string builder functions';
                      //      originally *ppszDest from HttpSrv_PrepareBuildingResponseHeaders()
{ int iLen = cpEnd - pHttpInst->szResponseHeader;
       // ,---'  ,---------------'
       // |      '--> INITIAL "string building pointer" from HttpSrv_PrepareBuildingResponseHeaders()
       // '--> incremented "string building pointer"
  if( (iLen>0) && (iLen<=C_HttpSrv_RESP_BUFFER_SIZE) )
   { pHttpInst->iRespHeaderLength = iLen;
   }
  else
   { pHttpInst->iRespHeaderLength = 0;
   }
  pHttpInst->iRespHeaderTxIndex = 0;
  return pHttpInst->iRespHeaderLength;  // returns the "response HEADER length" (multiple lines, including the final EMPTY line)

} // end HttpSrv_FinishBuildingResponseHeaders()

//---------------------------------------------------------------------------
void HttpSrv_PrepareBuildingResponseBody( T_HttpInstance *pHttpInst,
            char **ppszDest, const char **ppszEndstop ) // [out] pointers for string builder functions
  // Prepares two a nice plain old portable 'char pointers'
  // to assemble the 'response' in pHttpInst->bTxBuffer[] .
{
  if( pHttpInst->iTxBufferTxIndex < pHttpInst->iTxBufferLength ) // oops... STILL IN USE !
   { ShowError( ERROR_CLASS_INFO | SHOW_ERROR_TIMESTAMP, "PrepareResponseBody but previous response still BEING SENT !");
   }
  pHttpInst->iTxBufferLength = pHttpInst->iTxBufferTxIndex = 0;
  *ppszDest = (char*)pHttpInst->bTxBuffer; // start address of a string buffer for the response-header-lines
  *ppszEndstop = (const char*)pHttpInst->bTxBuffer + C_HttpSrv_TX_BUFFER_SIZE - 1;
} // end HttpSrv_PrepareBuildingResponseBody()

//---------------------------------------------------------------------------
int HttpSrv_FinishBuildingResponseBody( T_HttpInstance *pHttpInst,
        char *cpEnd ) // [in] char pointer incremented by the 'string builder functions';
                      //      originally *ppszDest from HttpSrv_PrepareBuildingResponseBody()
{ int iLen = cpEnd - (char*)pHttpInst->bTxBuffer;
       // ,---'  ,---------------------'
       // |      '--> INITIAL "string building pointer" from HttpSrv_PrepareBuildingResponseBody()
       // '--> incremented "string building pointer" (and possibly by binary stuff
       //                                       emitted by the Ogg/Vorbis encoder)
  if( (iLen>0) && (iLen<=C_HttpSrv_TX_BUFFER_SIZE) )
   { pHttpInst->iTxBufferLength = iLen;
   }
  else
   { pHttpInst->iTxBufferLength = 0;
   }
  pHttpInst->iTxBufferTxIndex = 0;
  return pHttpInst->iTxBufferLength;  // returns the "body length" (including everything)
} // end HttpSrv_FinishBuildingResponseBody()




//---------------------------------------------------------------------------
BOOL HttpSrv_GetValueFromRequestHeader( T_HttpInstance *pHttpInst,
        const char* pszFieldName,      // [in] e.g. "Sec-Websocket-Key"
        char* pszValue, int iMaxValueLength ) // [out] value, see details below.
  // Returns TRUE if a header-line beginning with the requested key exists.
  //         The VALUE [out: pszValue] may be an empty string.
  // Returns FALSE if the key (e.g. "Sec-Websocket-Key") doesn't exist
  //         in any of the "header-lines" in the previous HTTP request at all .
  //  [in] pHttpInst->szRequestHeader[0 .. pHttpInst->i32ReqHeaderLength-1 ] .
  // Note : An HTTP-header-field is *NOT* an "HTTP-header". The HTTP-header
  //        contains much more than just a couple of "header fields" !
  //        Welcome to the confusing terminology of the HTTP, TCP/IP and Co.
{
  char *cp = pHttpInst->szRequestHeader + 1; // not a "const char *" because we abuse the network buffer by inserting ZERO-BYTES to treat fragments like C strings ! ! !
  char *cpValue;
  const char *pszEndstop = pHttpInst->szRequestHeader + pHttpInst->i32ReqHeaderLength;
  int  nCharsRemaining, iValueLength;

  while( (nCharsRemaining = (pszEndstop-cp)) > 3 )
   { // From the bazillionth "RFC" :
     //  > Each header field consists of a name followed by a colon (":")
     //  > and the field value. Field names are case-insensitive.
     //  > This is in contrast to HTTP method names (GET, POST, etc.),
     //  > which are case-sensitive.
     cp = (char*)SL_strnstr( cp/*cpHaystack*/, pszFieldName/*pszNeedle*/,
                nCharsRemaining/*iHaystackLength*/, SL_COMPARE_OPTION_IGNORE_CASE );
     if( cp==NULL ) // no further occurrence
      { return FALSE;
      }
     else // found a name, but is it really a *FIELD NAME* ?
      { if( (cp[-1]=='\r' || cp[-1]=='\n' ) ) // field name at be the BEGIN of a line ?
         { cp += strlen( pszFieldName );
           if( *cp==':' ) // field name followed by a colon, ok...
            { // Arrived here: Got the field name, and a colon, but where does the VALUE begin ?
              ++cp;   // skip the colon, too
              // For simplicity, skip any whitespace here, and consider anything
              // after that (up to, but not including, the end-of-line "\r\n") as "value":
              cpValue = cp;
              cp = (char*)SL_strnstr( cpValue, "\r\n", (pszEndstop-cpValue), SL_COMPARE_OPTION_NORMAL );
              if( cp != NULL )
               { *cp = '\0';  // now cpValue isn't just a 'char pointer' but a C-string
               }
              if( pszValue != NULL )
               { SL_strncpy( pszValue, cpValue, iMaxValueLength );
               }
              return TRUE;
            } // end if < field name followed by a colon >
         } // end if < name at the BEGIN of a line, and name followed by a colon >
      } // end if < found the field name >
     cp += strlen( pszFieldName );  // skip whatever it was.. NOT a field name
   } // end while < not the end of the entire HTTP-header >
  return FALSE;  // field name not found in the multi-line "request header"
} // end HttpSrv_GetValueFromRequestHeader()


//---------------------------------------------------------------------------
void HttpSrv_DoServerSideInclude( // .. for HTML "files" with content generated on-the-fly.
        T_HttpInstance *pHttpInst,     // [in] instance data for a particular client
        const char **ppszSource,       // [in,out] sourcecode pointer with NAME of the include (will be skipped here if found)
        char **ppszDest, const char *pszEndstop) // [out] result ("value as string")
  // Kind of "poor man's SSI" (a microscopic subset of Server Side Include) .
  // Inspired by http://en.wikipedia.org/wiki/Server_Side_Includes .
  // The output (as a C-string) will be terminated with a zero byte .
  // Like all other HTML "documents", the included fragments were originally
  // converted from HTML into C tables, to embed them in a microcontroller's
  // Flash memory. To save previous space in the uC's Flash/ROM, tons of red tape
  // that are found in almost any HTML document (especially the headers)
  // were "included" by this complex string builder function, e.g.:
  //  - CSS styles
  //  - Javascript fragments
  //  - HTML fragments (like menus, 'headers' in the HTML document)
  //  - ... and an awful lot of red tape that was REMOVED HERE for simplicity.
  // All those 'SSI'-documents were retrieved from ROM via HttpSrv_FindBuiltInPage() .
{
  const char *cp = *ppszSource;     // our local 'source pointer'
  char *pszDest = *ppszDest;  // in the loop below, use a 'char pointer', not a 'char pointer pointer'
  char *cpOriginalDest = pszDest;
  char sz80ValueWithoutQuotes[84];
  const char *pszBuiltInPage;
  BOOL fContinue = TRUE;
  int iPar;
  T_HttpSrv_SSI_Directive dir; // SSI "directive" aka "element" ..
  T_HttpSrv_SSI_Parameter par; // SSI "parameter" aka "attribute", go figure !

  if( SL_SkipToken( &cp, "<!--#" ) ) // ok, this MAY be some SSI-thingy..
   { // but only when follwed by one of the known SSI "directives" alias "elements" alias "commands".
     // Note that the SSI-block containes ONE "directive", but possibly more
     // than one "parameter" for that directive (include multile files, echo multiple strings, etc).
     dir = (T_HttpSrv_SSI_Directive)SL_SkipOneOfNTokens( &cp, HttpSrv_SSI_Directives ); // "include","exec","echo" ?
     // Also get the first(!) parameter, if there's one:
     sz80ValueWithoutQuotes[0] = '\0';
     SL_SkipSpaces( &cp );       // e.g. separator between "include" and "file"
     iPar = SL_SkipOneOfNTokens( &cp, HttpSrv_SSI_Parameters ); // "file","virtual","cgi","cmd" ?
      // '--> may be NEGATIVE when none of the tokens (strings) matches.
     par  = (T_HttpSrv_SSI_Parameter)iPar;
      // '--> may be an UNSIGNED ENUM, causing BCB V12 to panic in "if (par>=0)" ...
     if( (iPar>=0) && (par != C_HttpSrv_SSI_Param_Unknown) )
      { if( *cp=='=' )
         { ++cp;  // separator between parameter name and value
           HttpSrv_CopyQuotedStringWithoutQuotes( &cp/*src*/,
              sz80ValueWithoutQuotes/*dest*/, 80/*maxlen*/ );
         }
      }
     switch( dir )
      {
        case C_HttpSrv_SSI_Directive_Include:
           // e.g. <!--#include file="header_styles.htm" -->
           while( fContinue && (iPar>=0) && (par != C_HttpSrv_SSI_Param_Unknown) )
            { switch( par )
               {
                 case C_HttpSrv_SSI_Param_File:
                 case C_HttpSrv_SSI_Param_Virtual:
                    // Here, we don't care for the subtle difference between "file" and "virtual":
                    // First try to find the requested document in ROM (index.htm, etc etc) :
                    pszBuiltInPage = HttpSrv_FindBuiltInPage( sz80ValueWithoutQuotes );
                    if( pszBuiltInPage != NULL )  // found a "page in ROM" ?
                     { // Do NOT call HttpSrv_DoServerSideInclude() recursively !
                       // The entire HTTP server may runs in a real-time task
                       // on a deeply embedded system (Cortex-M) with limited
                       // stack space, and no 'heap' at all, thus NO RECURSION
                       // allowed here.
                       // Instead, copy the "page" (or included file, whatever it is) unchanged.
                       SL_AppendString( &pszDest, pszEndstop, (char*)pszBuiltInPage );
                     } // end if < "HTML page in ROM" >
                    break; // end case < "#include file" + "#include virtual" >
                 default: // any other "parameter" here causes abort of the command !
                    fContinue = FALSE;
                    break;
               } // end switch( par )
              // Try to get the next parameter for the next loop:
              iPar = SL_SkipOneOfNTokens( &cp, HttpSrv_SSI_Parameters );
              par = (T_HttpSrv_SSI_Parameter)iPar;
              if( (iPar>=0) && (par != C_HttpSrv_SSI_Param_Unknown) )
               { if( *cp=='=' ) ++cp;  // separator between parameter name and value
                 else par = C_HttpSrv_SSI_Param_Unknown;
               }
            } // end while < more PARAMETERS for the SSI include-command >
           break;
        case C_HttpSrv_SSI_Directive_Exec   : // not implemented yet !
           break;
        case C_HttpSrv_SSI_Directive_Echo   : // not implemented yet !
           break;
        default: // everything else ignored, and the line will be emitted unchanged
           break;
      } // end switch(dir)

     // After the last parameter, we expect this thing. If not, it's not valid SSI ..
     SL_SkipSpaces( &cp );          // skip space(s?) before the delimiter "-->"
     if( ! SL_SkipToken( &cp, "-->" ) )
      { pszDest = cpOriginalDest;    // damnit, discard the whole stuff !
      }

   } // end if < "<!--#" >


  if( pszDest > cpOriginalDest ) // succesfull ?
   { *ppszSource = cp;    // skip the SSI-tag ( <!--#directive parameter=value parameter2=value2 --> )
     *ppszDest = pszDest; // pass back the INCREMENTED source-pointer
   }
} // end HttpSrv_DoServerSideInclude()

//---------------------------------------------------------------------------
static BOOL IsFourLetterToken( const char *pszSource )
{ int i;
  if( !((*pszSource>='A') && (*pszSource<='Z')) ) // The first character MUST be an upper-case LETTER
   { return FALSE;
   }
  ++pszSource;
  for(i=0; i<=2; ++i) // the 2nd, 3rd, and 4th character may be upper case LETTERS or digits...
   { if( !(((*pszSource>='A') && (*pszSource<='Z')) || ((*pszSource>='0') && (*pszSource<='9'))) )
       return FALSE;
     ++pszSource;
   }
  // the FIFTH character (which doesn't belong to the token) must *NOT* be an upper-case letter or digit !
  if( ((*pszSource>='A') && (*pszSource<='Z')) || ((*pszSource>='0') && (*pszSource<='9')) )
       return FALSE;
  return TRUE;
} // end IsFourLetterToken()



//---------------------------------------------------------------------------
int HttpSrv_GenerateHTMLFromTemplate( // .. with "dynamic content" inserted HERE
        T_HttpInstance *pHttpInst,    // [in] instance data for a particular client
        const char **ppszSource,      // [in,out] sourcecode pointer with NAME of the HTML 'template'
        char **ppszDest, const char *pszEndstop) // [out] result ("value as string")
  // The 'insertion of dynamic values' is now done in a single over,
  //   to avoid Easyweb's annoying hassle with frame boundaries, etc .
  //   The OUTPUT doesn't need to have exactly the same length
  //   as the INPUT anymore.  Hip-hip-hooray !
  //
  // Returns THE NUMBER OF CHARACTERS APPENDED TO THE OUTPUT (*ppszDest++) .
  //
{
  const char *pszSrc  = *ppszSource;
  const char *pszSrcPrev;  // "previous" source pointer
  char *pszDest = *ppszDest;  // in the loop below, use a 'char pointer', not a 'char pointer pointer'
  char *pszOldDest = pszDest;
  char *cp, c;
  int  i, nDigitsParsed;
  long i32;
  int  nBytesRemaining = pszEndstop - pszDest;
  if( nBytesRemaining < 256 ) // refuse to deliver anything if the dest-buffer is too small
   { return 0;  // there can't be any special string
   }

  while( ((pszDest+8) < pszEndstop) && ((c=*pszSrc)!='\0') )
   { pszSrcPrev = pszSrc;  // "previous" source pointer (to detect movement)
     *pszDest = '\0';      // important for later (to check if token was replaced)

     // In the interest of speed, first look at the CURRENT CHARACTER (c) :
     switch(c) // character with POSSIBLY a special meaning ?
      { case '<' : // Begin of a HTML tag, or 'SSI'-compatible includes, beginning with "<!--#" :
           if( pszSrc[1]=='!' && strncmp(pszSrc,"<!--#", 5)==0 )
            { // In this case, we'll let HttpSrv_DoServerSideInclude() parse and replace
              // the whole thing. It may decide NOT to replace the field, though !
              HERE_I_AM__HTTPSRV();
              HttpSrv_DoServerSideInclude( pHttpInst,
                 &pszSrc,            // [in] sourcecode, will be skipped when recognized
                 &pszDest, pszEndstop); // [out] result ("included text") with size limit
              HERE_I_AM__HTTPSRV();
            }
           break; // end case '<'
        case '$' : // Watch for the old-style 'dynamic' values, format "$ABCD$" (four letters) :
           if( IsFourLetterToken( pszSrc+1 ) && pszSrc[5]=='$' )
            {
#             define FOURCHARS2DW(a,b,c,d) (((DWORD)a)|((DWORD)b<<8)|((DWORD)c<<16)|((DWORD)d<<24))
              switch( FOURCHARS2DW(pszSrc[1],pszSrc[2],pszSrc[3],pszSrc[4]) )
               {
                 case FOURCHARS2DW('C','A','L','L') :  // "$CALL$" = the client's own callsign
                    SL_AppendString( &pszDest, pszEndstop, pHttpInst->pSession->sz40UserCall );
                    break; // end case "$CALL$"
                 case FOURCHARS2DW('C','L','I','P') :  // "$CLIP$" = the client's "dotted IP address"
                    SL_AppendPrintf( &pszDest, pszEndstop, "%s", CwNet_IPv4AddressToString(pHttpInst->pSession->b4HisIP.b) );
                    break; // end case "$CLIP$"
                 case FOURCHARS2DW('D','A','T','E') :  // "$DATE$" = current date in glorious ISO 8601 format
                    if( (pszEndstop-pszDest)  > 10 )
                     { UTL_FormatDateAndTime("YYYY-MM-DD", UTL_GetCurrentUnixDateAndTime(), pszDest );
                       SL_SkipToEndOfString( (const char**)&pszDest );
                       // ,------------------|____________|
                       // '--> Why do we need this bloody cast, Mr. Stoneage-Compiler ?
                       //      "cp" points to a char, so "&cp" is a pointer to a pointer to a char.
                       //      SL_SkipToEndOfString() expects a "const char **", which means
                       //      it promises not to modify the CHAR(s), so what's so "suspicious"
                       //      about the original code ( SL_SkipToEndOfString( &cp ); ) ?
                     }
                    break; // end case "$DATE$"
                 case FOURCHARS2DW('F','R','E','Q') :  // "$FREQ$" = VFO frequency as a string, with unit
                    if( pHttpInst->pCwNet != NULL )  // only if there's a "Rig Control" instance..
                     { SL_AppendPrintf( &pszDest, pszEndstop, "%.5lf MHz",
                         (double)(pHttpInst->pCwNet->RigControl.dblVfoFrequency * 1e-6) );
                     }
                    else // cannot tell you the "FREQ" ..
                     { SL_AppendPrintf( &pszDest, pszEndstop, "-FREQ-" );
                     }
                    break; // end case "$FREQ$"
                 case FOURCHARS2DW('M','O','D','E') :  // "$MODE$" = the rig's "operation mode" (CW,LSB,USB,etc)
                    if( pHttpInst->pCwNet != NULL )  // only if there's a "Rig Control" instance..
                     { SL_AppendString( &pszDest, pszEndstop,
                        RigCtrl_OperatingModeToString( pHttpInst->pCwNet->RigControl.iOpMode ) );
                        // e.g. "USB", "LSB", "CW"
                     }
                    else // cannot tell you the "MODE" ..
                     { SL_AppendPrintf( &pszDest, pszEndstop, "-MODE-" );
                     }
                    break; // end case "$MODE$"
                 case FOURCHARS2DW('P','E','R','M') :  // "$PERM$" = permissions granted to this user
                    SL_AppendPrintf( &pszDest, pszEndstop, "%d", (int)pHttpInst->pSession->iPermissions );
                    break; // end case "$CALL$"
                 case FOURCHARS2DW('S','E','S','S') :  // "$SESS$" = the client's current 'Session ID'
                    SL_AppendString( &pszDest, pszEndstop, pHttpInst->pSession->sz8SessionID );
                    break; // end case "$SESS$"
                 case FOURCHARS2DW('T','I','M','E') :  // "$TIME$" = time of day (in UTC)
                    if( (pszEndstop-pszDest)  > 10 )
                     { UTL_FormatDateAndTime("hh:mm:ss", UTL_GetCurrentUnixDateAndTime(), pszDest );
                       SL_SkipToEndOfString( (const char**)&pszDest );
                       // ,------------------|____________|
                       // '--> Again: Why do we need this bloody cast, Mr. Stoneage-Compiler ?
                     }
                    break; // end case "$TIME$"
                 case FOURCHARS2DW('U','S','E','R') :  // "$USER$" = the client's own user name
                    SL_AppendString( &pszDest, pszEndstop, pHttpInst->pSession->sz40UserName );
                    break; // end case "$USER$"
                 default:
                    break;
                } // end switch < four-character token enclosed by '$' >
              pszSrc += 6; // skip the "$ABCD$"-token ( 1 + 4 + 1 characters )
            } // end if < $ABCD$ >
           break; // end case '$'
        case '%': // Possibly the begin of an OpenWebRX-like token, "%[<keyword>]" ..
           // Note : The token %[<keyword>] must even be replaced INSIDE STRINGS !
           if( pszSrc[1]=='[' ) // it *may* be one of the following special tokens...
            { // Search and skip stuff like "%[CLIENT_ID]", "%[WS_URL]", "%[RX_PHOTO_HEIGHT]", etc.
              // In the original OpenWebRX server, these tokens were explaced
              // by meaningful data in openwebrx.py, using 'replace_dictionary'
              // and the Python expressions quoted further below :
              const char *pszKeyword = pszSrc+2;   // -> e.g. now "[CLIENT_ID]"
              int iToken = SL_SkipOneOfNTokens( &pszKeyword, OpenWebRX_Keywords );
              if( (iToken>0) && (*pszKeyword==']') // well-formatted "OpenWebRX-like" macro ?
                &&(pHttpInst->pOWClient != NULL) ) // accepted "OpebWebRX-like session" ?
               { OWRX_AppendTextForToken( pHttpInst->pOWClient, &pszDest, pszEndstop, iToken );
                 pszSrc = pszKeyword + 1;  // skip the "%[<token>]"
               }
            }
           break; // end case '%'
        default: // none of the above 'special' characters : simply COPY, char-by-char...
           break;
      } // end switch(c)
     if( pszSrc == pszSrcPrev )
      { SL_AppendChar( &pszDest, pszEndstop, *pszSrc++ );
      }
   } // end while < more characters >
  // Many of the HTML "templates" (formerly in Flash/ROM) do NOT end with an
  // empty line; some of them didn't even end with a single "\r\n".
  // Even though perfectly legal, this didn't look good in Wireshark when
  // "following a HTTP Stream" with periodic updates via the same "keep-alive"-
  // connection. For example, subsequent "GET /LiveData.htm HTTP/1.1"-requests
  // appeared IN THE SAME LINE as the last BODY. Because "\r\n" doesn't have
  // any special meaning in HTML (except from making the loaded document
  // more friendly to analyse FOR HUMANS), simply append "\r\n" here
  // AS PART OF THE RESPONSE BODY (four bytes included in the "Content-Length"):
  if( pszDest > pszOldDest ) // non-empty CONTENT ?
   { if( pszDest[-1] != '\n' )  // no end-of-line in content generated IN THE LOOP above ?
      { SL_AppendString( &pszDest, pszEndstop, "\r\n\r\n" ); // append at least one empty line
      }
   }


  *pszDest  = '\0';    // provide a trailing zero (the result shall be a C-string)
  *ppszDest = pszDest; // pass back the INCREMENTED destination pointer
  *ppszSource= pszSrc; // pass back the INCREMENTED source pointer


  return pszDest - pszOldDest; // Returns THE NUMBER OF CHARACTERS APPENDED TO THE OUTPUT

} // end HttpSrv_GenerateHTMLFromTemplate()


//---------------------------------------------------------------------------
int HttpSrv_OnWebSocketRequest( T_HttpInstance * pHttpInst )
  // Called on reception of the "initial" request to open a web-socket,
  //        with "Upgrade: websocket\r\n"
  //        and  "Connection: upgrade\r\n"  .
  // Return value : As shown on https://de.wikipedia.org/wiki/WebSocket .
  //     (if the server agrees to play the "websocket" game,
  //      he must respond with something like "HTTP/1.1 101 Switching Protocols",
  //      in THIS case : iHttpStatus not 'HTTP_STATUS__OK'
  //                                 but 'HTTP_STATUS__SWITCHING_PROTOCOLS' .
{
  int iHttpStatus = HTTP_STATUS__SWITCHING_PROTOCOLS; // optimistic return value
  char *pszDest;
  const char *pszEndstop;
  char sz255SecWebSocketKey[256], sz255SecWebSocketAcceptValue[256];
  int  iEncodedLength;

  pHttpInst->iStreamType = C_HttpSrv_StreamType_WebSocket;

  // Assemble the HTTP response in memory (will be sent in a single block later):
  // An OpenWebRXb server with HTTP/1.1 responded with only 147 byte "response header lines".
  //    (what Firefoxy's "Network"-tab usually doesn't show is the FIRST response line,
  //     so better switch the display to 'UNFORMATTED header lines' - or use Chrome instead. )
  // > HTTP/1.1 101 Switching Protocols
  // > Upgrade: websocket
  // > Connection: Upgrade
  // > Sec-WebSocket-Accept: kFT0F2o+2SOlayu/pLIcaXHRFQQ=   (details on this below)
  // > CQ-CQ-de: HA5KFU
  // Besides the above, a well-behaving WebSocket server (provider) may send:
  // > Sec-WebSocket-Protocol: ..... (etc, as usual, a very chatty protocol)
  // Similar as for (principally "endless") streams, no "Content-Length" can be given.
  HttpSrv_PrepareBuildingResponseHeaders( pHttpInst, &pszDest, &pszEndstop ); // prepare assembling the response-header-lines..
  // Note: To keep things simple, C_HTTP_SRV_MAX_SIZE_OF_RESPONSE_HEADERS
  //   is large enough to build the entire response in memory before sending.
  SL_AppendString( &pszDest, pszEndstop, "HTTP/1.1 101 Switching Protocols\r\nUpgrade: websocket\r\nConnection: Upgrade\r\n" );
  // Next header line : "Sec-Websocket-Accept". In Python with a lot of libs, this was a two-liner:
  // > ws_key=h_value("sec-websocket-key")
  // > ws_key_toreturn=base64.b64encode(sha.new(ws_key+"258EAFA5-E914-47DA-95CA-C5AB0DC85B11").digest())
  if( ! HttpSrv_GetValueFromRequestHeader( pHttpInst, "Sec-Websocket-Key", sz255SecWebSocketKey, 255 ) )
   { return HTTP_STATUS__BADREQUEST; // error : missing "Sec-Websocket-Key" !
   }
  // 2019-02-02 :  Got here with sz255SecWebSocketKey = "nBZDFblablablabla=" .
  // From Wikipedia, de.wikipedia.org/wiki/WebSocket :
  // > Anfrage des Clients
  // >
  // >  GET /chat HTTP/1.1
  // >  Host: server.example.com
  // >  Upgrade: websocket
  // >  Connection: Upgrade
  // >  Sec-WebSocket-Key: x3JJHMbDL1EzLkh9GBhXDw==
# if( 0 ) // (0): normal compilation, (1): try the *EN* Wikipedia example (beware, as so often, the german one sucks) ?
  strcpy( sz255SecWebSocketKey, "x3JJHMbDL1EzLkh9GBhXDw==" );
# endif
  // >  Origin: http://example.com
  // >  Sec-WebSocket-Protocol: chat, superchat
  // >  Sec-WebSocket-Version: 13
  // > Wie auch im HTTP-Protokoll gibt der Client an, auf welche Ressource (hier: /chat)
  // > und auf welchen Host (hier: server.example.com) er zugreifen mchte.
  // > Auerdem fordert der Client ein Upgrade auf das Websocket-Protokoll.
  // > Der zufllig generierte "Sec-WebSocket-Key" dient zur berprfung,
  // > ob der Server die Anfrage tatschlich gelesen und verstanden hat (s.U.) ....
  // > Unter "Sec-WebSocket-Protocol" hat der Client die Mglichkeit, auf das
  // > Websocket-Protokoll aufbauende Protokolle anzugeben, die die Clientanwendung
  // > untersttzt (hier: ein Chat-Protokoll). Selbsterklrend sollte unter
  // > "Sec-WebSocket-Version" die verwendete Protokollversion angegeben werden.
  //
  // > Antwort des Servers
  // >
  // >  HTTP/1.1 101 Switching Protocols
  // >  Upgrade: websocket
  // >  Connection: Upgrade
  // >  Sec-WebSocket-Accept: HSmrc0sMlYUkAGmm5OPpG2HaGWk=   (2019-02-02: The example on the EN wikipedia article was ok. Got the same result below.)
  // >  Sec-WebSocket-Protocol: chat
  // > Der zurckgesendete Schlssel unter "Sec-WebSocket-Accept" dient der Verifikation,
  // > dass der Server die Anfrage des Clients gelesen hat. Er wird wie folgt
  // > erstellt: An den oben erwhnten, Base64-kodierten String, den der Client
  // > sendet ("Sec-WebSocket-Key") wird der Globally Unique Identifier
  // > 258EAFA5-E914-47DA-95CA-C5AB0DC85B11 angehngt. Daraufhin wird ein
  // > SHA1-Hash des entstandenen Schlssels erstellt und Base64-kodiert.
  // > Anzumerken ist hierbei, dass der ursprnglich empfangene Schlssel
  // > zwar Base64-kodiert ist, jedoch zu keinem Zeitpunkt dekodiert wird.
  //   Hinweis zu Firefox: Dessen Anzeigen unter "Entwicklerwerkzeuge" - "Netzwerkanalyse"
  //                       ist (oder zumindest war) sehr unpraktisch.
  //                       Das dmliche Teil zeigte unter "Kopfzeilen"
  //                       nur die ZWEITE und FOLGENDE Zeilen des Response-"Headers" an,
  //               aber nicht die ERSTE Zeile mit "HTTP/1.1 101 Switching Protocols".
  //               Elender Schwachsinn. Wer kommt bloss auf solchen Unfug ?
  //      (a)      Nur in der "Konsole"(!) konnte nach Markieren der Zeile mit "GET ... /ws/..." (WebSocket)
  //               auf den unscheinbaren Button-hnlichen Text "Kopfzeilen (unformatiert)"
  //               eine halbwegs brauchbare Darstellung (inklusive der ERSTEN Header-Zeile)
  //               erzielt werden.
  //      (b)      Die AKTIVITT eines WebSockets, oder gar deren bertragenen Inhalt
  //               zu untersuchen, ging selbst mit dem gehypten "Firefox Quantum" nicht.
  //               Zhneknirschend wurde daraufhin Google Chrome eine neue Change gegeben.
  //               Et voila - saubere "DevTools", wesentlich intuitiver zu bedienen. Sorry, Firefox.
  //      (c)      Um die CLIENT-seitige Verarbeitung der per Web-Socket empfangenen
  //               Datenblcke in OpenWebSDR zu testen, eignet sich ein Breakpoint
  //               in openwebrx.js :: function on_ws_recv(evt). In Chrome gut zu erkennen:
  //               "first3Chars" = "CLI" / "AUD" / "FFT" / "DAT" / "MSG" .
  //                                 |       |       |       |       |
  //                   "client de server"    |       |       |       |       '--- various "messages"
  //                                        Audio Waterfall Data for "secondary demodulator"
  //               (this "multiplexed use" explains why there is only ONE WebSocket per client,
  //                not multiple. Also, there is NO EXTRA "Web-Audio" socket in OpenWebRX !
  //      (d)      In early 2019, Chrome had a MUCH better 'error diagnostics' than Firefox.
  //           In Chrome "Console" :
  //                > openwebrx.js:1702 WebSocket connection to 'ws://yhf6/ws/1B7E5955C26B51DEBEB5E08AE0566492' failed:
  //                > Error during WebSocket handshake: Incorrect 'Sec-WebSocket-Accept' header value .
  //                Aaah ! Nice diagnostics, thank you.
  //           In Firefox "Konsole" :
  //                > Firefox kann keine Verbindung zu dem Server unter ws://yhf6/ws/1B7E5955C26B51DEBEB5E08AE0566492 aufbauen.
  //                > openwebrx.js:1702:6  (that's also on "ws = new WebSocket(ws_url+client_id);", but not a clue WHY it failed).
  //                Baaah ! Crap !
  //
  INET_WebSecSocketKeyToWebSocketAcceptValue( sz255SecWebSocketKey, sz255SecWebSocketAcceptValue, sz255SecWebSocketAcceptValue+255 );
  SL_AppendPrintf( &pszDest, pszEndstop, "Sec-Websocket-Accept: %s\r\n", sz255SecWebSocketAcceptValue );

  SL_AppendString( &pszDest, pszEndstop, "CQ-CQ-de: HA5KFU\r\n" ); // just a gimmick or really required for OpenWebRX ?
  // (with or without this gimmick, always got a "WebSocket has closed unexpectedly. Please reload the page.")

  // Empty line to separate the HTTP header and the next part of the "WebSocket handshake":
  SL_AppendString( &pszDest, pszEndstop, "\r\n" );

  HttpSrv_FinishBuildingResponseHeaders( pHttpInst, pszDest );

     // WATCH pHttpInst->szResponseHeader at this point. For OpenWebRX, it contained...
     // the "response line" (first line of the header) :
     //                 "HTTP/1.1 101 Switching Protocols\r\n"
     // + header-lines  "Upgrade: ..\r\n",
     //                 "Connection: Upgrade\r\n",
     //                 "Sec-Websocket-Accept: <base64-encoded crap>\r\n",
     // +               "\r\n" (empty HEADER-LINE, aka "header-field", marking the end).
     // The HTTP server framework must KEEP THE CONNECTION ("socket") ALIVE,
     // regardless of the HTTP protocol version, because we said
     // "Connection: Upgrade", not "Close" :
  pHttpInst->dwConnectionPersistance |= C_HttpSrv_ConnectionPersistance_KeepAlive | C_HttpSrv_ConnectionPersistance_Upgrade;
  // For WEB SOCKETS, the following message for the 'Event Log'
  //  replaces the one for 'normal files' in http_intf.c : HttpSrv_FileRequested():
#if(0)  // "server log" ? Not in this stripped-down variant !
  HttpSrv_LogEvent(  HTTP_HIGHLIGHT_NORMAL | HTTP_EFLAG_REQUEST,
     "Connection #%d is WebSocket, %s . Sending resp-hdr[%d] on socket %d to %s",
         (int)pHttpInst->iConnIndex,
         pHttpInst->szRequestedURLWithoutQuery,
         (int)pHttpInst->i32RespHeaderLength,
         (int)pHttpInst->Socket,
         HttpSrv_IPAddressToString(pHttpInst->dwClientIP) );
#endif // (0)
  return iHttpStatus;

} // end HttpSrv_OnWebSocketRequest()

//---------------------------------------------------------------------------
int HttpSrv_OnWSRcvd( T_HttpInstance *pHttpInst, // WS = WebSocket ...
     BYTE *bRxBuffer,       // [in] address of the OLDEST byte to be consumed
     int nBytesInRxBuffer ) // [in] max. number of bytes "consumeable" from the buffer
  // Called from HttpSrv_OnRead() if the socket has already been turned
  //  into a "WebSocket", i.e. with pHttpInst->nServerState == C_HttpSrvState_WEBSOCKET_OPEN .
  //  Implements the RX-side of the WebSocket "Base Framing Protocol". Details below.
  // Return value : Number of bytes acutally "consumed" from bRxBuffer[] in THIS call.
  // Note: The input buffer is NOT a circular FIFO to simplify the task of
  //       "de-framing" data directly in bRxBuffer[].
  //    If HttpSrv_OnWSRcvd() cannot "consume" the next packet of data
  //    received from a "web socket", it's perfectly ok to return ZERO from here.
  //    In that case, in the NEXT call, bRxBuffer[0] will still contain the same,
  //    but MORE bytes appended, up to nBytesInRxBuffer = C_HttpSrv_RX_BUFFER_SIZE.
  //    If you expect LONGER PACKETS that need to be "decoded" in a single call,
  //    increase C_HttpSrv_RX_BUFFER_SIZE (in http_intf.h) to the max. chunk size.
{ int  nBytesRemaining;
  int  iWSFrameType;    // aka "Opcode", but that's too misleading. Usually Http_WebSocketFrame_Text or Http_WebSocketFrame_Binary.
  long i, i32PayloadLength;
  BYTE *pbSrc         = bRxBuffer;  // don't increment before "consuming" a complete FRAME
  BYTE *pbSrcEndstop  = bRxBuffer + nBytesInRxBuffer; // pbSrc must not reach this !
  BYTE *pbMaskAndPayload;
  BYTE b4Mask[4];
  char cSavedForTrailingZero;

  while( (nBytesRemaining = (pbSrcEndstop-pbSrc)) > 2 ) // may have to REPEAT the following,
   { // if Mr. Winsock or Mr. LwIP delivered MORE THAN ONE "WebSocket frame" !
     //   (don't assume anything about the size of the chunks that the TCP layer
     //    will stuff in .. it will certainly NOT be one "frame" per "chunk").
     //
     // How to de-packetize the data received from a WebSocket, e.g. for OpenWebRX ?
     // The starting point was the client-side Javascript of OpenWebRX .
     // Due to the lack of support for WebSockets in Firefox Quantum (anno 2019 you
     // could not even watch the 'ws' payload sent and received on the "Network" tab
     // because none of the old developer plugins for WebSockets seemed to work properly),
     // so use Chrome instead, and see what happens in openwebrx.js.
     // The first chunk of data we'd expect to see here is "SERVER DE CLIENT ..",
     //  as displayed in Chrome's "DevTools" - "Network" - "WS" (Web Sockets),
     //  with   Data = "SERVER DE CLIENT openwebrx.js", Length = 29 = strlen(Data).
     //  (sent from 'function on_ws_opened()', approximately line 1310 :
     //    > ws.send("SERVER DE CLIENT openwebrx.js");
     // But a breakpoint HERE fired with..
     //  nBytesInRxBuffer = 35, and
     //  bRxBuffer[0..34] = { 0x81, 0x9D, 0xE3, 0x6D, 0x24, 0xA8, 0xB0, 0x28, 0x76, 0xFE, ... }
     //                       |__________"WebSocket"___________|  |___ "Masked ______ _ _ _ _
     //                                   |___"Masking-Key"____|  |     Payload"  (29 bytes)
     //
     // As many times before, only Wireshark came to the rescue, and decoded
     // the fields as shown above. It could even DECODE the "Masked Payload"
     // into "Payload" (in this case, plain text "SERVER DE CLIENT openwebrx.js").
     // The  "Payload" itself was titled "Line-based text data (1 lines)",
     // and what Chrome's "DevTools" - "Network" - "WS" showed as "Length = 29"
     // was NOT the number of bytes in the TCP stream, but the length of the
     // WebSocket frame's "payload" (here: 29 characters).
     // The 6-byte "WebSocket" overhead is part of what RFC6455 describes
     //            as follows in chapter 5, "Data Framing" :
     // > In the WebSocket Protocol, data is transmitted using a sequence of frames.
     // > To avoid confusing network intermediaries (such as
     // > intercepting proxies) and for security reasons that are further
     // > discussed in Section 10.3, a client MUST mask all frames that it
     // > sends to the server (see Section 5.3 for further details).  (Note
     // > that masking is done whether or not the WebSocket Protocol is running
     // > over TLS.)  The server MUST close the connection upon receiving a
     // > frame that is not masked.  In this case, a server MAY send a Close
     // > frame with a status code of 1002 (protocol error) as defined in
     // > Section 7.4.1.  A server MUST NOT mask any frames that it sends to
     // > the client.  A client MUST close a connection if it detects a masked
     // > frame.  In this case, it MAY use the status code 1002 (protocol
     // > error) as defined in Section 7.4.1.  (These rules might be relaxed in
     // > a future specification.)
     //    (on this occation: The crazy RFC numbers bits beginning at the MOST
     //     significant bit. For example, the first byte in a frame, bit "0"
     //     actually the MOST SIGNIFICANT BIT, is the "Fin"-bit.
     //     We don't follow that nonsense here. The MOST SIGNIFICANT BIT in a byte
     //     is 7 -SEVEN-, and will always be. So forget about the BIT NUMBERS
     //     in RFC6455, chapter 5.2. "Base Framing Protocol" .
     // Because nobody can guarantee that the bytes in the receive-buffer
     // are "nicely aligned", we cannot simply cast the first six(?) bytes
     // into a nice C-struct-pointer (this wouldn't work because the number of bytes
     // for the WebSocket's "framing"-overhead IS NOT CONSTANT.)
     // Again, the BEST way to see and understand the following bit-fiddling
     //        is Wireshark's built-in decoder for "WebSocket"-packets !
     iWSFrameType    = pbSrc[0]; // 4 bits "Opcode" + "FIN" in bit 7
     i32PayloadLength= pbSrc[1] & 0x7F; // 7 bits "Payload len" (with the "MASK"-flag masked out, hi)
     pbMaskAndPayload= pbSrc + 2;       // "Masking-key" if there's no "Extended payload length" .
     switch( i32PayloadLength ) // is there an "Extended payload length" ?
      { case 126 :  // yes, TWO bytes "extended payload length" !
           if( nBytesRemaining < 4 )  // not enough bytes for a 4-byte WebSocket frame-header !
            { // "call me back later when there are more bytes available"
              return (pbSrc-bRxBuffer); // return the number of bytes acutally "consumed" (stepping frame-by-frame)
            }
           // RFC6455 : "Multibyte length quantities are expressed in network byte order."
           //  Typical geek speak. "Network byte order" means Big Endian, and Big Endian sucks.
           //  To make this code independent of the "host byte order"
           //     (which in almost any case is Little Endian, aka "Intel"),
           //  and to avoid those obfuscated socket-macros (these data aren't
           //  aligned to 16- or 64-bit boundaries anyway), use bit-fiddling :
           i32PayloadLength = ((long)pbSrc[2] << 8) // 1st "length"-byte in "Network" byte order : Bits 15..8
                             | (long)pbSrc[3];      // 2nd "length"-byte in "Network" byte order : Bits  7..0
           pbMaskAndPayload += 2;      // "Masking key" (if any) two more bytes later
           break;
        case 127 :  // this is crazy: "extended payload length" is a SIXTYFOUR-bit unsigned integer !!
           if( nBytesRemaining < 8 )  // not enough bytes for a 6-byte WebSocket frame-header !
            { // "call me back later when there are more bytes available"
              return (pbSrc-bRxBuffer);  // "call me back later" (with more data)
            }
           // Ignore this 64-bit madness because our tiny "RX buffer" can only
           //  store "frames" with a few kBytes anyway. So bail out if the frame
           //  is longer than we can store in a 32-bit integer:
           if( ( pbSrc[2] // 1st "length"-byte "Network" byte order : Bits 63..56
               | pbSrc[3] // 2nd "length"-byte "Network" byte order : Bits 55..48
               | pbSrc[4] // 3rd "length"-byte "Network" byte order : Bits 47..40
               | pbSrc[5])// 4th "length"-byte "Network" byte order : Bits 39..32
             > 0 )
            { i32PayloadLength = 0x7FFFFFFF; // whatever it is.. it's too much
            }
           else
            { i32PayloadLength = ( (long)pbSrc[6] << 24) // 5th "length"-byte : Bits 31..24
                               | ( (long)pbSrc[7] << 16) // 6th "length"-byte : Bits 23..16
                               | ( (long)pbSrc[8] << 8 ) // 7th "length"-byte : Bits 15..8
                               |   (long)pbSrc[9];       // 8th "length"-byte : Bits  7..0
              pbMaskAndPayload += 8;      // "Masking key" (if any) eight more bytes later
            }
           break;
        default  :  // no "extended" paylength, iPayloadLength = 1...127 bytes
           break;
      } // end switch( iPayloadLength )
     if( pbSrc[1] & 0x80 ) // flag indicating the existence of a "Masking Key" ...
      { // > Masking-key:  0 or 4 bytes
        // >  All frames sent from the client to the server are masked by a
        // > 32-bit value that is contained within the frame.  This field is
        // > present if the mask bit is set to 1 and is absent if the mask bit
        // > is set to 0.  See Section 5.3 for further information on client-
        // > to-server masking.
        // What RFC6455 calls "masking" is in fact a simple EXOR combination:
        // The first payload byte is EXORED with b4Mask[0], the 2nd by b4Mask[1],
        // the 3rd by b4Mask[2], the 4th by b4Mask[3], the 5th again by b4Mask[0], etc.
        b4Mask[0] = *pbMaskAndPayload++;
        b4Mask[1] = *pbMaskAndPayload++;
        b4Mask[2] = *pbMaskAndPayload++;
        b4Mask[3] = *pbMaskAndPayload++;
      } // end if < "Masking-Key" present >

     // At THIS point, pbMaskAndPayload points to the PAYLOAD ("mask" already skipped) !

     if( i32PayloadLength > C_HttpSrv_RX_BUFFER_SIZE-8 )
      {
        HttpSrv_SendErrorAndCloseConnection( pHttpInst->pCwNet, pHttpInst->pClient,
                         pHttpInst->sz255RequestedURLWithoutQuery, // [in] pszBadURL
                         HTTP_STATUS__INSUFF_STORAGE );
        return 0;
      }
     if( (pbMaskAndPayload+i32PayloadLength) > pbSrcEndstop ) // WebSocket-"frame" not completely received yet
      { return (pbSrc-bRxBuffer);  // "call me back later" (with more data)
      }

     // Arrived here: Another "frame" received from the WebSocket is complete.
     //  Since the source-buffer is ours, DECODE ("unmask") the data directly
     //  in the rx-buffer, before passing the next frame to the application :
     if( pbSrc[1] & 0x80 )  // "MASK"-bit set in the frame header -> "unmask" (decode) the payload ..
      { for( i=0; i<i32PayloadLength; ++i )
         { pbMaskAndPayload[i] ^= b4Mask[i&3]; // tear down the "mask"
         }
      }
     pbSrc = pbMaskAndPayload + i32PayloadLength; // skip this frame in the RX buffer
                   // (important because we may have to process MORE THAN ONE FRAME
                   //  in each call of HttpSrv_OnWSRcvd(). Without incrementing
                   //  pbSrc here, we may end up in an endless loop...)
     cSavedForTrailingZero = pbMaskAndPayload[i32PayloadLength]; // <- reason for the "plus 4" in bRxBuffer[C_HttpSrv_RX_BUFFER_SIZE+4]
     switch( iWSFrameType & Http_WebSocketFrame_Mask4Opcode ) // don't bother the APPLICATION with "control frames" - process them HERE:
      {
        case Http_WebSocketFrame_CloseConn: // "denotes a connection close"
           break;
        case Http_WebSocketFrame_Ping     :
           // > When you get a ping, send back a pong with the exact same
           // > Payload Data as the ping (for pings and pongs, the max payload
           // > length is 125). You might also get a pong without ever sending
           // > a ping; ignore this if it happens.
           // > If you have gotten more than one ping before you get the chance
           // > to send a pong, you only send one pong.
           if( i32PayloadLength < 125 )
            { // Immediately respond with a "pong" only if the tx-buffer is COMPLETELY empty :
              if( pHttpInst->iRxBufferNumBytesPending == 0 )
               { pHttpInst->fWebSocketSendPong = TRUE;
                 if( HttpSrv_SendWebSocketFrame( pHttpInst, Http_WebSocketFrame_Pong | Http_WebSocketFrame_SingleFragment,
                      pbMaskAndPayload/*here: unmasked data from "Ping"*/, i32PayloadLength, NULL/*no mask*/ ) )
                  { pHttpInst->fWebSocketSendPong = FALSE; // no transmission of "Pong" pending anymore
                  }
               }
            }
           break; // end case "ping"
        case Http_WebSocketFrame_Pong     :
           break; // end case "pong"
        default :
           pbMaskAndPayload[i32PayloadLength] = 0x00; // TEMPORARILY provide a trailing zero, to simplify string parsing
           if( pHttpInst->pOWClient != NULL )
            { OWRX_OnWebSocketFrameRcvd( pHttpInst->pOWClient,
                   iWSFrameType, i32PayloadLength, pbMaskAndPayload );
            }
           pbMaskAndPayload[i32PayloadLength] = cSavedForTrailingZero; // "repair" the buffer again, because there may be more frames following
           break;
      } // end switch < "opcode" >
   } // end while( (nBytesRemaining = (pbSrcEndstop-pbSrc)) > 2 )

  return (pbSrc-bRxBuffer); // consumed THIS number of bytes from bRxBuffer in the above loop (if any..)

} // end HttpSrv_OnWSRcvd()


//---------------------------------------------------------------------------
int HttpSrv_AssembleWebSocketFrameHeader( // does what the name says..
       BYTE  *pbFrameHeader,     // [out] WebSocket "frame" header, 6 bytes for worst case
       int   iWSFrameType,       // [in] Http_WebSocketFrame_Text, .._Binary, etc.
                                 //      Almost always combined with Http_WebSocketFrame_SingleFragment BY THE CALLER.
       int   iPayloadLength,     // [in] payload length (measured in bytes)
 const BYTE  *pb4WebSocketMask)  // [in] optional "mask" for the data (NULL=none) (*)
  // Returns the size of the FRAME HEADER, also measured in bytes.
  //         Depending on the payload length and the presence/absence of a "Mask",
  //         expect 2 to 6 bytes (thus use a destination buffer with at least 6 bytes).
  // (*) About "mask" (exclusive-or for the payload in a WebSocket frame):
  //     Don't miss the note about the strange requirements in RFC6455 :
  //     > A client MUST mask all frames (...)
  //     > A server MUST NOT mask any frames that it sends (...)
  //     In the author's opinion this is really braindead
  //      (and will possibly be changed in future versions),
  //     but for "Sec-WebSocket-Version: 13", it's a fact.
  //     A client MUST "mask" frames, a server MUST NOT (thus dwWebSocketMask=0) !
{
  BYTE *pbDst = pbFrameHeader;
  BYTE *pbOpcode = pbDst;

  *pbDst++ = iWSFrameType; // 4 bits "Opcode" (almost always with the "FIN"-bit, alias Http_WebSocketFrame_SingleFragment, SET ! )
  if( iPayloadLength < 126 )           // very short payload ? Only 7-bit length indicator
   { *pbDst++ = (BYTE)iPayloadLength;
     // pbDst now points to the optional "mask" or the first payload byte
   }
  else if( iPayloadLength <= 65535 )   // reasonably short payload ? 16-bit length indicator
   { *pbDst++ = 126;  // indicator for "two byte payload length"
     *pbDst++ = (BYTE)(iPayloadLength >> 8);  // bits 15..8 first (ugly Big Endian format)
     *pbDst++ = (BYTE)(iPayloadLength & 255); // bits  7..0 last
   }
  else if( iPayloadLength <= 65535 )   // reasonably short payload ? 16-bit length indicator
   { *pbDst++ = 127;  // indicator for "EIGHT(!!!) byte payload length" (insane)
     *pbDst++ = 0x00; // 1st "length"-byte in ugly Big Endian byte order : Bits 63..56
     *pbDst++ = 0x00; // 2nd "length"-byte in ugly Big Endian byte order : Bits 55..48
     *pbDst++ = 0x00; // 3rd "length"-byte in ugly Big Endian byte order : Bits 47..40
     *pbDst++ = 0x00; // 4th "length"-byte in ugly Big Endian byte order : Bits 39..32
     *pbDst++ = (BYTE)(iPayloadLength >> 24);  // bits 31..24
     *pbDst++ = (BYTE)(iPayloadLength >> 16);  // bits 23..16
     *pbDst++ = (BYTE)(iPayloadLength >> 8);   // bits 15..8
     *pbDst++ = (BYTE)(iPayloadLength & 255);  // bits  7..0 last
   }
  if( pb4WebSocketMask != NULL ) // header for a "masked" frame (*) ?
   { pbOpcode[1] |= 0x80; // "mask"-indicator-bit in bit 7 of the *LENGTH*(!)-byte
     // (*) What RFC6455 calls "masking" is in fact a simple EXOR combination .
     *pbDst++ = pb4WebSocketMask[0];           // 1st "mask"-byte
     *pbDst++ = pb4WebSocketMask[1];           // 2nd "mask"-byte
     *pbDst++ = pb4WebSocketMask[2];           // 3rd "mask"-byte
     *pbDst++ = pb4WebSocketMask[3];           // 4th "mask"-byte
   }
  return pbDst - pbOpcode; // returns the number of bytes used for the WebSocket "frame"-header

} // end HttpSrv_AssembleWebSocketFrameHeader()

//---------------------------------------------------------------------------
BOOL HttpSrv_SendWebSocketFrame( // Assembles a WebSocket frame and queues it up for transmission
       T_HttpInstance *pHttpInst, // [in] HTTP server connection instance,
       int   iWSFrameType,       // [in] Http_WebSocketFrame_Text, .._Binary, etc
       const BYTE *pbSource,     // [in] address of the payload (1st byte)
       int   iPayloadLength,     // [in] length of the payload in bytes, w/o frame header
       const BYTE *pb4WebSocketMask)  // [in] optional "mask" for the data (NULL=none) (*)
  // Should only be called in state C_HttpSrvState_WEBSOCKET_OPEN,
  //                     and if pHttpInst->iTxBufferNumBytesPending == 0 !
  // [out] pHttpInst->bTxBuffer[ 0..pHttpInst->iTxBufferNumBytesPending ] :
  //       buffer used for transmission
  // (*) About "mask" (exclusive-or for the payload in a WebSocket frame):
  //     Beware of the strange requirements in RFC6455 :
  //     > A client MUST mask all frames (...)
  //     > A server MUST NOT mask any frames that it sends (...)
{
  int i;
  int nFreeSpaceRemaining; // .. number of bytes
  BYTE *pbDest, *pbEndstop;

  if( pbSource==NULL )
   { return FALSE;
   }
  if( iPayloadLength < 1 ) // don't try to send frames without payload
   { return FALSE;
   }

  // Don't assume pHttpInst->bTxBuffer[] is completely empty !
  //  In fact, it's empty when pHttpInst->iTxBufferTxIndex reached (or even exceeded)
  //                           pHttpInst->iTxBufferLength .
  //  To allow buffering MULTIPLE "web socket frames" for a single TCP/IP-"send()",
  //  shift out any data that have already been sent to give as much space as
  //  possible, in our simple microcontroller-firmware-friendly transmit buffer:
  //
  //  pHttpInst->bTxBuffer[ 0 ... C_HttpSrv_TX_BUFFER_SIZE-1 ] :
  //    ______________________________________________________________
  //   |                |                            |                |:
  //   | <already sent> | <waiting for transmission> | <unused space> |:
  //   |________________|____________________________|________________|:
  //    |                |                            |                |
  //   [0]         [iTxBufferTxIndex]   [iTxBufferLength]    [TX_BUFFER_SIZE]
  //         Note: [TX_BUFFER_SIZE] does *not* point to a valid array element !
  //
  if( pHttpInst->iTxBufferTxIndex > 0 ) // got some "already sent" bytes ?
   { if( (pHttpInst->iTxBufferLength > pHttpInst->iTxBufferTxIndex )
      && (pHttpInst->iTxBufferLength <= C_HttpSrv_TX_BUFFER_SIZE ) )
      { // got a few bytes "waiting for transmission" ?
        memmove( pHttpInst->bTxBuffer/*destination*/,
                 pHttpInst->bTxBuffer+pHttpInst->iTxBufferTxIndex/*source*/,
                 pHttpInst->iTxBufferLength-pHttpInst->iTxBufferTxIndex/*nBytes*/ );
        pHttpInst->iTxBufferLength  = pHttpInst->iTxBufferTxIndex;
        pHttpInst->iTxBufferTxIndex = 0;  // <- easy to understand with the "ASCII art" above !
      }
     else // nothing "waiting for transmission" so fill bTxBuffer[] from the start:
      { pHttpInst->iTxBufferTxIndex = pHttpInst->iTxBufferLength = 0;
      }
   } // end if( pHttpInst->iTxBufferTxIndex > 0 )
  nFreeSpaceRemaining = C_HttpSrv_TX_BUFFER_SIZE-pHttpInst->iTxBufferLength;
  // For worst case, the "WebSocket frame header" requires SIX bytes
  //  (more on that in HttpSrv_AssembleWebSocketFrameHeader(), so :
  if( (iPayloadLength + 6 ) > nFreeSpaceRemaining )
   { return FALSE;  // payload length exceeds available buffer space !
   }
  pbDest    = pHttpInst->bTxBuffer + pHttpInst->iTxBufferLength;
  pbEndstop = pHttpInst->bTxBuffer + C_HttpSrv_TX_BUFFER_SIZE;

  // Assemble another WebSocket "frame"-header (details about the format in HttpSrv_OnWSRcvd() )
  pbDest += HttpSrv_AssembleWebSocketFrameHeader( pbDest/*pbFrameHeader*/,
              iWSFrameType, // [in] Http_WebSocketFrame_Text, .._Binary, etc
              iPayloadLength, pb4WebSocketMask );  // [in] payload length in bytes, optional "mask"
  if( pb4WebSocketMask != NULL )
   { for(i=0; (i<iPayloadLength) && (pbDest<pbEndstop); ++i)
      { *pbDest++ = pbSource[i] ^ pb4WebSocketMask[i & 3];
      }
   }
  else // no "mask" (what they meant was bitwise EXOR, not an 'AND-mask') :
   { for(i=0; (i<iPayloadLength) && (pbDest<pbEndstop); ++i)
      { *pbDest++ = pbSource[i];
      }
   }
  pHttpInst->iTxBufferLength = pbDest-pHttpInst->bTxBuffer;
  // ex: return HttpSrv_FlushTxBuffer( pHttpInst ) ; // returns TRUE when the "frame" is on its way
  //     (cannot "flush the transmit buffer" from here, because
  //      to make Mr. Nagle's Algorithm happy, only call "send()" once
  //      after receiving another fragment/segment/whatever-the-name-was)
  return TRUE;

} // end HttpSrv_SendWebSocketFrame()

//---------------------------------------------------------------------------
int HttpSrv_GetFreeSpaceInTxBuffer( // Called from OpenWebRX_Server()
       // BEFORE deciding whether or not to send another 'WebSocket Frame'
       // (again: A "WebSocket" doesn't have anything to do with a TCP/IP-
       //         socket. A "WebSocket" is an HTTP thing, and not really simple).
       T_HttpInstance *pHttpInst ) // [in] HTTP server connection instance
       //      on which the caller "plans" to send a WebSocket frame,
       //      using HttpSrv_SendWebSocketFrame() .
{ int nFreeSpaceRemaining = C_HttpSrv_TX_BUFFER_SIZE-pHttpInst->iTxBufferLength;
      // '--> See the ASCII art in HttpSrv_SendWebSocketFrame() !
      //      This may be a bit pessistic, depending on the number of bytes
      //      ALREADY SENT.
      // As already mentioned, the "WebSocket frame header" may requires SIX bytes
      //  (more on that in HttpSrv_AssembleWebSocketFrameHeader(), so :
   return nFreeSpaceRemaining;
} // end HttpSrv_GetFreeSpaceInTxBuffer()


//---------------------------------------------------------------------------
BOOL HttpSrv_CreateInstance(                                           // API
        T_CwNet *pCwNet,        // [in] single "Cw Network" SERVER INSTANCE
        T_CwNetClient *pClient) // [in,out] instance data for a CLIENT :
                                // [in] pClient->b4HisIP (important to
                                //       identify ALL CONNECTION-INSTANCES
                                //       initiatated by this client's web browser)
     // [out] pClient->pHttpInst, especially :
     //       pClient->pHttpInst->pSession (REFERENCE to the same T_HttpSession,
     //         with the same USER NAME, CALLSIGN, and possibly PASSWORD .
     //         We only allow ONE SESSION for an individual remote IP address,
     //         but accept MULTIPLE CONNECTIONS from that addresss).
     // [out] pClient->pHttpInst->fIsNewSession : Set to TRUE for an "unknown IP".
     //       In that case, the query-string MUST contain a valid user name, etc,
     //       which will be tested in HttpSrv_OnReceive() .

{
  T_HttpInstance *pHttpInst;
  T_HttpSession  *pSession = NULL;
  BOOL fIsNewSession = FALSE;
  int i;
  for(i=0; i<C_HttpSrv_MaxSessions; ++i)
   { if( HttpSrv_Sessions[i].b4HisIP.dw == pClient->b4HisIP.dw )
      { pSession = &HttpSrv_Sessions[i]; // a SESSION already exists for this IP
        break;
      }
   }
  if( pSession == NULL )  // no SESSION with a matching IP -> get a "new" one from the pool
   { for(i=0; i<C_HttpSrv_MaxSessions; ++i)
      { if( HttpSrv_Sessions[i].b4HisIP.dw == 0 ) // UNUSED entry (marked by IP address ZERO) ?
         { pSession = &HttpSrv_Sessions[i]; // use THIS, at the moment unused SESSION instance
           memset( pSession, 0, sizeof(T_HttpSession) ); // clear 'garbage' from previous use
           fIsNewSession = TRUE;
           break;
         }
      }
   }
  if( pClient->pHttpInst == NULL ) // only if NOT CREATED YET ...
   { if( pSession == NULL )  // error : running out of SESSION instances !
      { return FALSE;
      }
     pHttpInst = (T_HttpInstance*)malloc( sizeof(T_HttpInstance) );
     if( pHttpInst != NULL )
      { memset( pHttpInst, 0, sizeof(T_HttpInstance) );
        pHttpInst->pSession = pSession;   // tie the same SESSION to possibly multiple CONNECTIONS (e.g. sockets)
        pClient->pHttpInst  = pHttpInst;
      }
   }
  if( (pHttpInst=pClient->pHttpInst) != NULL ) // 'clean up for recycling' ..
   { pHttpInst->pCwNet  = pCwNet;  // <- reference to the entire "CW Network server"
     pHttpInst->pClient = pClient; // <- reference to a remote client in the "CW Network"
     pHttpInst->pSession->b4HisIP.dw = pClient->b4HisIP.dw;
     pHttpInst->fIsNewSession = fIsNewSession;
     pHttpInst->fIsNewConnection = TRUE;
     pHttpInst->dblUnixTimeOfLastActivity = pHttpInst->dblCurrentUnixTime = UTL_GetCurrentUnixDateAndTime();
     pHttpInst->iServerOptions = pCwNet->cfg.iHttpServerOptions; // <- HTTP_SERVER_OPTIONS_ENABLE / _RESTRICT / ..
     pHttpInst->nMethod = HTTP_METHOD_UNKNOWN;
     pHttpInst->nServerState = C_HttpSrvState_RECEIVED_REQUEST;
     pHttpInst->dwAcceptedContentTypes = 0;
     pHttpInst->nContentType = 0;
     pHttpInst->i32ReqContentLength = 0;
     pHttpInst->i32RespContentLength= 0;
     pHttpInst->dwConnectionPersistance = C_HttpSrv_ConnectionPersistance_Unknown;  // Required for WebSockets .. maybe later !
     pHttpInst->dwUpgradeRequested = 0; // .. maybe later, if we steal support for "WebSockets" from Spectrum Lab
     pHttpInst->isMobile = FALSE; // guess the visitor is NOT a mobile phone / smartphone / tablet
     pHttpInst->i32ReqHeaderLength = 0;
     pHttpInst->szRequestHeader[0] = '\0';
     pHttpInst->fReadingFileFromDisk = FALSE;
     pHttpInst->i32NettoFileSize = 0;
     pHttpInst->fIsDynamicPage = FALSE; // TRUE=may insert dynamic values; FALSE=don't
     // Note: In the older mickey-mouse HTTP server, the 'dynamic values'
     //  were inserted *AFTER* packetizing. This was totally unacceptable.
     //  Since 2009-03-06, the values are inserted in the HTML pages
     //  *in a single block that may grow*.
     pHttpInst->sz15Password1[0] = '\0';
     pHttpInst->sz80ErrorMessage[0] = '\0';
     pHttpInst->finished_receiving_request = TRUE;
     pHttpInst->parsing_posted_response = FALSE; // FALSE when processing a GET-request, TRUE when preparing the response for a POST-request.
     pHttpInst->iRxBufferNumBytesPending = 0;
     pHttpInst->iRespHeaderLength = pHttpInst->iRespHeaderTxIndex = 0;
     pHttpInst->iTxBufferLength   = pHttpInst->iTxBufferTxIndex = 0;
     pHttpInst->fWebSocketSendPong = FALSE;
     pHttpInst->iStreamType = C_HttpSrv_StreamType_None;
     pHttpInst->dwBufInMemHead = pHttpInst->dwBufInMemTail = 0;
     pHttpInst->StreamOutCallback = NULL;
     pHttpInst->fFileRequestedFromApplication = FALSE;
     pHttpInst->fWaitingToContinueWrite = FALSE;
     pHttpInst->fCallOnConnClose  = FALSE;
     pHttpInst->fUsedByOpenWebRX  = FALSE;
     pHttpInst->i64TotalBytesRcvd = 0;
     pHttpInst->dwPayloadBytesSent= 0;
     pHttpInst->i64TotalBytesSent = 0;
     pHttpInst->dwFilePtr = 0;
   }
  return pClient->pHttpInst != NULL; // TRUE = "successfully created a HTTP server instance (serving ONE CLIENT/socket)
} // end HttpSrv_CreateInstance()

//---------------------------------------------------------------------------
void HttpSrv_DeleteInstance(                                          // API
        T_CwNet *pCwNet,        // [in] single "Cw Network" SERVER INSTANCE
        T_CwNetClient *pClient) // [in,out] instance data for a CLIENT
{ T_HttpInstance *pHttpInst;
  if( pClient != NULL )
   { pClient->fWantVorbis = FALSE; // at least THIS client doesn't need the Ogg/Vorbis stream anymore
     pHttpInst = pClient->pHttpInst;
     if( pHttpInst != NULL )
      { // Let module CwNet.c know we don't exist anymore:
        pClient->pHttpInst = NULL;
        // Perform more clean-up work here ?  [not yet, but who knows..]
        free( pHttpInst );
      }
   }
} // end HttpSrv_DeleteInstance()

//---------------------------------------------------------------------------
void HttpSrv_SendString(                                  // internal function
        T_CwNet *pCwNet,        // [in] single SERVER INSTANCE
        T_CwNetClient *pClient, // [in] instance data for a CLIENT
        char *pszString )       // [in] string to append to the TX buffer
{

} // end HttpSrv_SendString()

//---------------------------------------------------------------------------
void HttpSrv_SendError(                                  // internal function
        T_CwNet *pCwNet,        // [in] single SERVER INSTANCE
        T_CwNetClient *pClient, // [in] instance data for a CLIENT
        const char *pszBadURL,  // [in] URL of the non-existing 'resource'
        int iHttpStatusCode )   // [in] HTTP_STATUS__xxxx = array index into HttpSrv_szStatMsgs[].
  // Note: Errors as HTML documents are only sent to well-behaving clients.
  //       Bad guys will not receive any information,
  //       their sockets will immediately be closed with ZERO bytes sent.
  //
{
  int iLen;
  char sz1k[1024];
  BYTE *pbTxBuffer;
  const char *pszStatusText = HttpSrv_StatusCodeToString( iHttpStatusCode );

  // As the minimum HTTP response, send the mandatory "status line"
  //  (that's the thing beginning with "HTTP/1.1", a "http status code", and "status text"),
  // and optionally -for a nicer display in a web browser- also some HTTP HEADERS,
  // or even (as the last, very optional part in an HTTP response) a "body" :
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // > HTTP Responses
  // >   Status line
  // >        The start line of an HTTP response, called the status line, contains the following information:
  // >            The protocol version, usually HTTP/1.1.
  // >            A status code, indicating success or failure of the request.
  // >            Common status codes are 200, 404, or 302.
  // >            A status text. A brief, purely informational, textual description
  // >            of the status code to help a human understand the HTTP message.
  // >        A typical status line looks like: HTTP/1.1 404 Not Found.
  // >
  // >   Headers
  // >        HTTP headers for responses follow the same structure as any other header:
  // >            a case-insensitive string followed by a colon (':')
  // >            and a value whose structure depends upon the type of the header.
  // >        The whole header, including its value, presents as a single line.
  // >
  // >   Body
  // >        The last part of a response is the body. Not all responses have one:
  // >        responses with a status code that sufficiently answers the request
  // >        without the need for corresponding payload
  // >        (like 201 Created or 204 No Content) usually don't.
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  //
  // Not many of those 'tutorials' managed to present a full example.
  // Here is one that did (from tutorialspoint) :
  //     > HTTP/1.1 200 OK                                  (that's the "status line")
  //     > Date: Mon, 27 Jul 2009 12:28:53 GMT              (that's one of many RESPONSE HEADERS(plural!) )
  //     > Server: Apache/2.2.14 (Win32)
  //     > Last-Modified: Wed, 22 Jul 2009 19:15:56 GMT
  //     > Content-Length: 88
  //     > Content-Type: text/html
  //     > Connection: Closed
  //     >                         <- EMPTY LINE between "headers" and "body"
  //     > <html>
  //     > <body>
  //     > <h1>Hello, World!</h1>
  //     > </body>
  //     > </html>                 <- that's the end of the "body". Do we need another EMPTY LINE after ths ?
  //
  sprintf( sz1k, "HTTP/1.1 %d %s\r\n"
                  "Content-Type: text/html\r\n"
                  "Connection: Closed\r\n"
                  "\r\n"         // <- EMPTY LINE between "headers" and "body"
                  "<html>\r\n"   // Firefox whined about "missing DOCTYPE" and all kind of crap.
                  "<body>\r\n"   //
                  "<h1>Awfully sorry, but I'm not the server you're looking for.</h1>\r\n"
                  "Your current IPv4 address is %s .<br>\r\n"
                  "Your ephemeral TCP port number is %d .<br>\r\n"
                  "You tried to access \"%s\".<br>\r\n"
                  "Connection closed with status %d = '%s' .<br>\r\n"
                  "</body>\r\n"
                  "</html>\r\n", // <- that's the end of the "REPONSE body" (not the "HTML document body")
              (int)iHttpStatusCode, pszStatusText,
               CwNet_IPv4AddressToString( pClient->b4HisIP.b ),
              (int)pClient->iHisPort,
               pszBadURL,
              (int)iHttpStatusCode, pszStatusText );
  iLen = SL_strnlen( sz1k, 1024 );
  if( (pbTxBuffer = CwNet_AllocBytesInTxBuffer( pCwNet, CWNET_CMD_NONE/*no "binary command*/, iLen )) != NULL )
   { memcpy( pbTxBuffer, sz1k, iLen );
   }
} // end HttpSrv_SendError()


//---------------------------------------------------------------------------
void HttpSrv_SendErrorAndCloseConnection(                        // internal
        T_CwNet *pCwNet,        // [in] single SERVER INSTANCE
        T_CwNetClient *pClient, // [in] instance data for a CLIENT
        const char *pszBadURL,  // [in] URL of the non-existing 'resource'
        int iHttpStatusCode )   // [in] HTTP_STATUS__xxxx = array index into HttpSrv_szStatMsgs[].
  // Note: This function must work without a valid T_HttpInstance,
  //       because the reason for calling it may be running out of pool entries.
{ HttpSrv_SendError( pCwNet, pClient, pszBadURL, iHttpStatusCode );
  if( pCwNet->cfg.iDiagnosticFlags & (CWNET_DIAG_FLAGS_VERBOSE | CWNET_DIAG_FLAGS_SHOW_CONN_LOG) )
   { ShowError( ERROR_CLASS_INFO | SHOW_ERROR_TIMESTAMP, "HTTP: Close %s:%d (%s \"%s\")",
          CwNet_IPv4AddressToString( pClient->b4HisIP.b ), (int)pClient->iHisPort,
          HttpSrv_StatusCodeToString( iHttpStatusCode ), pszBadURL );
     // Let this run for a few hours. Most of this 'garbage traffic' originates
     // from port scanners or maybe botnets. Some of them even wait for a few
     // seconds before "going away" from their side, e.g.:
     // > 19:19:58.5 HTTP: Close 152.32.235.85:16682 (Unauthorized "/")
     // > 19:23:43.2 Server: client #2 on 167.94.138.125:54842 disconn, 0 bytes sent, 0 received.
     // > 19:23:43.2 Server: connected by #2 on 167.94.138.125:54842 (binary)
     // > 19:23:46.3 HTTP: Close 167.94.138.125:47856 (Unauthorized "/")
     // > 19:23:51.4 HTTP: Close 167.94.138.125:38170 (Unauthorized "/")
     // > 19:23:52.0 Server: connected by #2 on 167.94.138.125:34160 (binary)
     // > 19:23:57.0 Disconn #2 on 167.94.138.125:34160 (timeout)
     // > 19:23:57.0 Server: client #2 on 167.94.138.125:34160 disconn, 0 bytes sent, 0 received.
     //   ,-------------------------------|____________|
     //   '--> whois said : "Censys, Inc. (CENSY)" [this guy kept banging on the door,
     //                                    until INET_BlockIPv4Address() stopped him]
     // > 19:30:21.4 HTTP: Close 106.75.177.81:32378 (Unauthorized "/")
     // > 19:30:22.1 HTTP: Close 106.75.177.81:32508 (Unauthorized "/favicon.ico")  [obviously a BROWSER..]
     //   ,----------------------|___________|
     //   '--> whois said : "Shanghai UCloud Information Technology Company Limited"
   }
  pClient->fDisconnect = TRUE; // flag for e.g. the 'Server Thread' to close this socket
} // end HttpSrv_SendErrorAndCloseConnection()

//---------------------------------------------------------------------------
int HttpSrv_GetClientIndex(T_HttpInstance *pHttpInst) // -> 0..CWNET_MAX_CLIENTS
  // Return values: 0 = CWNET_LOCAL_CLIENT_INDEX (not a 'real', remote client),
  //                1 = CWNET_FIRST_REMOTE_CLIENT_INDEX,
  //            ...     CWNET_MAX_CLIENTS (maximum, see Remote_CW_Keyer/CwNet.h)
{
  int iClientIndex = 0;
  if( (pHttpInst->pClient!=NULL) && (pHttpInst->pCwNet!=NULL) )
   { iClientIndex = (int)(pHttpInst->pClient - &pHttpInst->pCwNet->Client[0]); // the "pointer trick" again..
     // '--> 1..CWNET_MAX_CLIENTS = 'remote' clients .
     if( (iClientIndex<0) || (iClientIndex>/*!*/CWNET_MAX_CLIENTS) )
      { iClientIndex = 0;  // index ZERO =
      }
   }
  return iClientIndex;
} // end HttpSrv_GetClientIndex()


//---------------------------------------------------------------------------
void HttpSrv_OnPoll( // Tries to send as much as we can ...
       T_HttpInstance *pHttpInst ) // [in] HTTP server connection instance,
                    // [in] pHttpInst->bTxBuffer[ 0..iTxBufferNumBytesPending ]
  // Called from CwNet_OnPoll() if the peer uses HTTP, not raw TCP/IP.
{
  int n, nBytesRemaining, iClientIndex;
  BYTE *pbTxBuffer;
  if(pHttpInst==NULL) // no T_HttpInstance allocated for this connection yet
   { return;          // -> there cannot be anything to 'poll for transmission'
   }
  iClientIndex = HttpSrv_GetClientIndex(pHttpInst);

  // FIRST send the REPONSE HEADER(s). Then, if the network buffer space permits,
  //  also send the first part of the "response BODY" in HTTP.
  nBytesRemaining = pHttpInst->iRespHeaderLength - pHttpInst->iRespHeaderTxIndex;
  if( (nBytesRemaining>0) // something left to send from the REPONSE HEADER LINES ?
   && (pHttpInst->nServerState == C_HttpSrvState_SENDING_HEADER) )
   { // ex: nRet = send(pHttpInst->Socket, pHttpInst->bTxBuffer, pHttpInst->iTxBufferNumBytesPending, 0/*flags*/ );
     // For good reasons (with a microcontroller / LwIP in mind), don't directly
     // use the SOCKET layer here. Instead, pass on anything we have
     // to the transmit buffer in CwNet.c (!), and let that module do the rest.
     n = CwNet_GetFreeSpaceInTxBuffer( pHttpInst->pCwNet );
     if( n > nBytesRemaining )
      {  n = nBytesRemaining;
      }
     if(  (n > 0)
       && (pbTxBuffer = CwNet_AllocBytesInTxBuffer( pHttpInst->pCwNet, CWNET_CMD_NONE/*no "binary command*/, n )) != NULL )
      { memcpy( pbTxBuffer, pHttpInst->szResponseHeader + pHttpInst->iRespHeaderTxIndex, n );
        pHttpInst->dblUnixTimeOfLastActivity = pHttpInst->dblCurrentUnixTime; // at least "some progress".. here: sent something
        pHttpInst->i64TotalBytesSent += n;  // "sent".. here, not really, but placed in some kind of 'network buffer'
        pHttpInst->iRespHeaderTxIndex+= n;
        if( pHttpInst->iRespHeaderTxIndex >= pHttpInst->iRespHeaderLength ) // finished sending the response HEADER line(s) ->
         { pHttpInst->nServerState = C_HttpSrvState_SENT_HEADER;
         }
      }
   }
  // If the outbound network buffer permits, also send a part (or the complete) response *BODY* now:
  nBytesRemaining = pHttpInst->iTxBufferLength - pHttpInst->iTxBufferTxIndex;
  if( nBytesRemaining<=0) // NOTHING left to send now, but maybe the source is a STREAM ..
   { if( pHttpInst->nTransferEncoding == C_HttpSrv_XferEncoding_Chunked )
      { // may send another "chunk" (binary stuff like AUDIO, WATERFALL DATA, etc) ..
        // so what STREAM TYPE to send from here ?
        switch( pHttpInst->iStreamType )
         {
#         if( SWI_USE_VORBIS_STREAM )
           case C_HttpSrv_StreamType_VorbisAudio :
              HttpSrv_ContinueStreamingLiveAudio( pHttpInst );
              break;
#         endif // SWI_USE_VORBIS_STREAM ?
           case C_HttpSrv_StreamType_WebSocket   :
              break;
           default:
              break;
         } // end switch( pHttpInst->iStreamType )
      }   // end if < currently sending "chunked" blocks of data >
   }     // end if < pHttpInst->bTxBuffer is COMPLETELY empty > ?
  // Check again if there is something to send ..
  nBytesRemaining = pHttpInst->iTxBufferLength - pHttpInst->iTxBufferTxIndex;
  if( (nBytesRemaining>0) // something left to send from the REPONSE HEADER LINES ?
   && (  (pHttpInst->nServerState == C_HttpSrvState_SENT_HEADER)
       ||(pHttpInst->nServerState == C_HttpSrvState_SENDING_DATA) )
    ) // similar as above, but now for the "response BODY", not for the "response HEADER(s)"..
   { n = CwNet_GetFreeSpaceInTxBuffer( pHttpInst->pCwNet );
     if( n > nBytesRemaining )
      {  n = nBytesRemaining;
      }
     if(  (n > 0)
       && (pbTxBuffer = CwNet_AllocBytesInTxBuffer( pHttpInst->pCwNet, CWNET_CMD_NONE/*no "binary command*/, n )) != NULL )
      { memcpy( pbTxBuffer, pHttpInst->bTxBuffer, n );
        pHttpInst->dblUnixTimeOfLastActivity = pHttpInst->dblCurrentUnixTime; // at least "some progress".. here: sent something
        pHttpInst->i64TotalBytesSent += n;  // "sent".. not yet, but placed in some kind of 'network buffer'
        pHttpInst->iTxBufferTxIndex  += n;
        if( pHttpInst->iTxBufferTxIndex >= pHttpInst->iTxBufferLength ) // finished sending the response BODY (or the current "transfer chunk") ->
         { if( pHttpInst->nTransferEncoding == C_HttpSrv_XferEncoding_Chunked)
            { pHttpInst->nServerState = C_HttpSrvState_ASSEMBLE_NEXT_CHUNK;
              // 2024-02-10: Wireshark didn't show the "Chunks" with Vorbis audio
              //             in Ogg containers ("pages"). Set a breakpoint HERE,
              //             then check what happens on the socket after returning
              //             from HttpSrv_OnPoll() to CwNet_OnPoll() and
              //             Step over 'send()' there, THEN stop Wireshark.
              if( pHttpInst->pCwNet->cfg.iDiagnosticFlags & CWNET_DIAG_FLAGS_VERBOSE )
               { ShowError( ERROR_CLASS_INFO | SHOW_ERROR_TIMESTAMP, "Server%d: New %s for %s (%d bytes, next OggPage=%ld)",
                     (int)iClientIndex,
                     (pHttpInst->iStreamType==C_HttpSrv_StreamType_VorbisAudio) ? "audio chunk" : "chunk",
                     CwNet_IPv4AddressToString( pHttpInst->pSession->b4HisIP.b ),
                     (int)pHttpInst->iTxBufferLength,
#                   if( SWI_USE_VORBIS_STREAM )
                     (long)pHttpInst->pClient->dwNextVorbisPage );
#                   else  // ! SWI_USE_VORBIS_STREAM )
                     (long)-1 );
#                   endif //   SWI_USE_VORBIS_STREAM ?

               }
            }
           else
            { pHttpInst->nServerState = C_HttpSrvState_RECEIVING_REQUEST;
            }
         }
        else // NOT finished yet ->
         { pHttpInst->nServerState = C_HttpSrvState_SENDING_DATA;
         }
      }
   }


} // end HttpSrv_OnPoll()


//---------------------------------------------------------------------------
static void HttpSrv_ExtractSessionParamsFromQueryString(
                T_HttpSession *pSession, const char *psz255QueryString )
{
  char sz40Key[44], sz40Val[44];
  while( INET_ParseKeyAndValueFromQueryString( &psz255QueryString/*incremented*/ ,
            sz40Key,40,  // [out] char *pszKey, with limited length

            sz40Val,40)) // [out] char *pszKey, with limited length

   { switch( SL_FindStringInTable( HttpSrv_QueryStringKeys, sz40Key ) )

      { case C_HttpSrvKey_USER :

             SL_strncpy( pSession->sz40UserName, sz40Val, 40 );

             break;

        case C_HttpSrvKey_CALL :
             SL_strncpy( pSession->sz40UserCall, sz40Val, 40 );
             break;
        case C_HttpSrvKey_PWD  :
             SL_strncpy( pSession->sz40Password, sz40Val, 40 );
             break;
        case C_HttpSrvKey_SID  : // visitor thinks he already has a valid "Session ID" ?
             if( SL_strncmp( pSession->sz8SessionID, sz40Val, 8 ) != 0 )
              { // wrong session ID ?
                pSession->iPermissions = -1; // withdraw old permissions
              }
             break;
      }
   }  // end while < more key=value pairs in the Query String >
}    // end HttpSrv_ExtractSessionParamsFromQueryString()


//----------------------------------------------------------------------------
void HttpSrv_OnReceive(                                                // API
        T_CwNet *pCwNet,                  // [in] single SERVER INSTANCE
        T_CwNetClient *pClient,           // [in] instance data for a CLIENT
        BYTE *pbRcvdData, int nBytesRcvd) // [in] received stream segment
  // Called from CwNet_OnReceive() when the peer (remote client) obviously
  // tries to "talk HTTP" with us, in the very first TCP segment after connecting.
{
  const char *cpBeginOfFirstLine, *cpEndOfLine, *cpEndOfHeader;
  int nCharsRemaining, iMethod, iHttpStatus = HTTP_STATUS__DONT_RESPOND;
  char sz255RequestedURL[256];
  const char *pszQueryString = NULL;
  char *pszDest, *pszEndOfHeader, *cp;
  const char *pszSrc, *pszEndstop;
  int  iBuiltInPage, iLen, iLen2, iRequestLength;
  char sz80Temp[84], sz40Key[44], sz40Val[44];
  T_HttpInstance *pHttpInst;
  int iClientIndex = (int)(pClient - &pCwNet->Client[0]); // perfectly legal "pointer trick":
      // '--> 0 = CWNET_LOCAL_CLIENT_INDEX; 1..CWNET_MAX_CLIENTS = 'remote' clients .
  (void)pszQueryString; // "assigned a value that is never used" .. oh, shut up, Mr Pedantic !
  (void)iHttpStatus;


  // Keep this simple for a start .. expect to receive the ENTIRE REQUEST
  // in a single block.
  // In HTTP (1.0 or 1.1), the block ends with an empty like (one "\r\n"
  // for the last non-empty line, followed by another "\r\n" for the EMPTY line).
  //
  // Quoted from somewhere, about the "empty line" (CRLF after CRLF) :
  // >  In the interest of robustness, servers SHOULD ignore any empty line(s)
  // >  received where a Request-Line is expected.
  // >  In other words, if the server is reading the protocol stream at the
  // >  beginning of a message and receives a CRLF first, it should ignore the CRLF.
  // >
  // >  Certain buggy HTTP/1.0 client implementations generate extra CRLF's
  // >  after a POST request. To restate what is explicitly forbidden by the BNF,
  // >  an HTTP/1.1 client MUST NOT preface or follow a request with an extra CRLF.
  // >
  // WB: At this point, cp points (sic) to what we think/hope is the 'Request-Line' .
  //     So skip the junk from those 'certain buggy HTTP/1.0 clients' HERE :
  HERE_I_AM__HTTPSRV();
  pszSrc = (char*)pbRcvdData; // example: "GET /HTTP/1.1\r\nHost: MyTestServer.dynv6.net:7355\r\n".. blablabla
                              //      or  "POST /upload.htm HTTP/1.1\r\n" ..blablabla, blabla, blabla.. "password1=admin" CR NL "login=Login" .
  pszEndstop = pszSrc + nBytesRcvd;
  while( (*pszSrc=='\r' || *pszSrc=='\n') && (pszSrc<pszEndstop) )
   { ++pszSrc; // > "In the interest of robustness, servers SHOULD ignore any empty line(s)"... (details above)
   }
  cpBeginOfFirstLine = pszSrc; // <- save cpBeginOfFirstLine (e.g. with the "GET /HTTP/1.1 .." for later,
                       // to store the entire request in pHttpInst->szRequestHeader[] further below.
  nCharsRemaining = pszEndstop - pszSrc;
  cpEndOfLine   = SL_strnstr( pszSrc, "\r\n",     nCharsRemaining, SL_COMPARE_OPTION_NORMAL );
  cpEndOfHeader = SL_strnstr( pszSrc, "\r\n\r\n", nCharsRemaining, SL_COMPARE_OPTION_NORMAL );
  // Example:  cpEndOfHeader = "\r\n\r\npwd=test\r\n" ...
  //                                    '--> that's already the BODY after e.g. "POST"
  if( cpEndOfHeader == NULL ) // illegal HTTP request ?
   { HERE_I_AM__HTTPSRV();
     pClient->fDisconnect = TRUE; // flag for e.g. the 'Server Thread' to close this socket
     pszDest    = sz80Temp;
     pszEndstop = sz80Temp + sizeof(sz80Temp) - 1;
     SL_AppendString( &pszDest, pszEndstop, "Malformed request (" );
     INET_DumpTraffic_HexOrASCII( pszDest, pszEndstop-pszDest, pbRcvdData, nBytesRcvd );
     SL_AppendString( &pszDest, pszEndstop, ") !" );
     INET_BlockIPv4Address( &pCwNet->sBlacklist, pClient->b4HisIP.b, sz80Temp );
     // ,----'
     // '--> depending on pCwNet->cfg.iDiagnosticFlags (CWNET_DIAG_FLAGS_SHOW_CONN_LOG),
     //      emits messages like "Blacklisted 185.224.128.10 (Malformed HTTP request)".
     //      See examples of blacklisted requests are in T_CwNetBlacklistEntry.
     HERE_I_AM__HTTPSRV();
     return; // don't even send a single byte to the potential "bad guy"
     // (the transition of pClient->iClientState to CWNET_CLIENT_STATE_DISCONN
     //  happens shortly later in CwNet.c, after really closing the socket, etc)
   }
  iRequestLength = 4/*don't strip the"\r\n\r\n" */ + cpEndOfHeader - cpBeginOfFirstLine;
  //                                                   |               |
  //    ,----------------------------------------------'               |
  //   points to the "\r\n\r\n"                          points to the "GET .."
  iMethod = SL_SkipOneOfNStrings( &pszSrc, HttpMethods ); // "GET", "POST" (to name just a few)

  // What many of those simplistic 'tutorials' fail to mention is the complete
  // syntax of a "Request-Line" in HTTP. Here's what www.w3.org specified:
  // > 5.1 Request-Line
  // > The Request-Line begins with a method token, followed by the Request-URI
  // > and the protocol version, and ending with CRLF. The elements are
  // > separated by SP characters.
  // > No CR or LF is allowed except in the final CRLF sequence.
  // > Request-Line = Method SP Request-URI SP HTTP-Version CRLF
  //    ,-------------|_____|   |_________|    |__________|
  //    '--> e.g. "GET" +  space   |              |
  //                           May be just "/"    e.g. "HTTP/1.0" or "HTTP/1.0"
  // Get the file name (or whatever the URI may be; with POST it's usually NOT a filename).
  // The URI(/URL?) ends with a space (usually), and is NOT double quoted.
  // In such cases, HttpSrv_CopyQuotedStringWithoutQuotes() copies everyhting
  // until the next non-percent-encoded SPACE or CARRIAGE RETURN or NEW LINE :
  HttpSrv_CopyQuotedStringWithoutQuotes( &pszSrc/*in*/, sz255RequestedURL/*out*/, 255 );
     //                                ,---------------------'
     //                                '--> Here still with the optional QUERY STRING .
     // Note: If only the HOSTNAME is entered in a web browser,
     //  the resulting "Path-and-Name" is just "/" - not a blank string !
     //  It's up to the application to chose a default "root file" then .
     //  See example in HttpSrv_OnGET() .
     //  The URI/URL(?) may contain a "Query String" *after* the "file":
     // > The question mark is used as a separator, and is not part of the query string.
     //  If the QUERY STRING contains multiple key=value pairs, separated by ampersand.
     //  For a start, ignore the fact that the entire string (sz255RequestedURL
     //  shall be "URL encoded" (to replace e.g. '#', SPACE, the '+' (because it MAY
     //  replace SPACE), etc). Only 'A'-'Z', 'a'-'z', '0'-'9', and '~', '-', '.' and '_'
     //  are left as-is. Buaah.
  // Keep some of those bastards out, BEFORE actually creating a per-client instance:
  cp = strchr( sz255RequestedURL, '.' );
  if( cp!=NULL )  // anyone asking for .php is definitely NOT our user, etc ..
   { if( SL_strnicmp( cp, ".php", 4) == 0 )
      { HERE_I_AM__HTTPSRV();
        pClient->fDisconnect = TRUE; // flag for e.g. the 'Server Thread' to close this socket
        INET_BlockIPv4Address( &pCwNet->sBlacklist, pClient->b4HisIP.b, "Asking for .php" );
        return; // don't even send a single byte to the potential "bad guy"
      }
   }

  // Allocate or 'recycle' a T_HttpInstance for this connection:
  if( pClient->pHttpInst == NULL ) // have NOT created an instance for this connection yet ?
   { HERE_I_AM__HTTPSRV();
     HttpSrv_CreateInstance( pCwNet, pClient ); // .. or at least "clean up for recycling" ..
     HERE_I_AM__HTTPSRV();
   } // end if( pClient->pHttpInst == NULL )
  if( (pHttpInst=pClient->pHttpInst) == NULL ) // HttpSrv_CreateInstance() failed ->
   { HERE_I_AM__HTTPSRV();
     HttpSrv_SendErrorAndCloseConnection( pCwNet, pClient, sz255RequestedURL, HTTP_STATUS__UNAVAILABLE );
     // ,----'
     // '--> depending on pCwNet->cfg.iDiagnosticFlags (CWNET_DIAG_FLAGS_SHOW_CONN_LOG),
     //      emits messages like HTTP: Close 185.224.128.10:46040 (Unauthorized "/")
     HttpSrv_DeleteInstance( pCwNet, pClient );
     return;
   }

  // Arrived here: pClient->pHttpInst is NON-NULL, so use it and update some members:
  HERE_I_AM__HTTPSRV();
  pHttpInst->iServerOptions = pCwNet->cfg.iHttpServerOptions; // <- HTTP_SERVER_OPTIONS_ENABLE / _RESTRICT / ..
  pHttpInst->i64TotalBytesRcvd += nBytesRcvd;
  pHttpInst->nMethod = iMethod; // e.g. HTTP_METHOD_GET
  pHttpInst->nTransferEncoding = C_HttpSrv_XferEncoding_None; // no "Transfer-Encoding" yet
  pHttpInst->nServerState = C_HttpSrvState_RECEIVED_REQUEST;
  pHttpInst->i32ReqHeaderLength = iRequestLength;  // number of characters in the buffer, used by HttpSrv_GetValueFromRequestHeader()
  if( pHttpInst->i32ReqHeaderLength >= C_HTTP_SRV_MAX_SIZE_OF_REQUEST_HEADER )
   {  pHttpInst->i32ReqHeaderLength = C_HTTP_SRV_MAX_SIZE_OF_REQUEST_HEADER-1;
   }
  if( pHttpInst->i32ReqHeaderLength > 0 )
   { SL_strncpy( pHttpInst->szRequestHeader, cpBeginOfFirstLine, pHttpInst->i32ReqHeaderLength+1/*!*/ );
   }
  pHttpInst->szRequestHeader[ pHttpInst->i32ReqHeaderLength ] = '\0'; // NOW it's an easily parsable "C"-string !
  memset( pHttpInst->sz255RequestedURLWithoutQuery, 0, 256 );
  pszDest    = pHttpInst->sz255RequestedURLWithoutQuery;
  pszEndstop = pszDest + 255;
  INET_UrlEncodingToPlainText( sz255RequestedURL, // [in] pszUrlEncodedSource,
                           &pszDest, pszEndstop, // [out] Plain Text ("decoded")
                           "?&"/*pszDelimiters*/ );
  pszQueryString = strrchr( sz255RequestedURL, '?' );
  if( pszQueryString != NULL )
   { SL_strncpy( pHttpInst->sz255QueryString, pszQueryString, 255 );
   }
  else // No query string ...
   { pHttpInst->sz255QueryString[0] = '\0';
   }

  if( pHttpInst->pSession->sz40UserName[0] == '\0' ) // USER NAME not set yet ?
   { // To reduce the problems resulting from NOT passing the QUERY STRING
     //    in LINKS WITHIN OUR "WEBPAGE", as an alternative to the query string,
     //    many browsers (at least Firefox) pass those info in the chatty
     //    request header lines : "Referer:". For example, if Javascript tries to
     //    periodically update the 'LiveData.htm' fragment via "XMLHttpRequest()",
     //    the fox kept sending (in each "GET .. LiveData.htm") :
     //     "Referer: http://127.0.0.1:7355/?user=Moritz&call=DL4YHF%20testing" .
     // So look for "\r\nReferer: " in the REQUEST HEADER LINES :
     pszSrc = SL_strnstr_skip( pHttpInst->szRequestHeader/*haystack*/, "\r\nReferer: "/*needle*/,
                          pHttpInst->i32ReqHeaderLength, SL_COMPARE_OPTION_NORMAL );
     pszEndstop = pHttpInst->szRequestHeader + pHttpInst->i32ReqHeaderLength;
     if( pszSrc != NULL ) // the text after "Referer: " is an URL-encoded URL,
      { // e.g. pszSrc = "http://127.0.0.1:7355/?user=Moritz&call=DL4YHF%20testing\r\n" ...
        // ,------->>---------"endstop" for the referer's query-string HERE --->>-'
        pszEndstop = SL_strnstr(pszSrc,"\r\n",pszEndstop-pszSrc,SL_COMPARE_OPTION_NORMAL );
        if( pszEndstop != NULL )
         { pszQueryString = SL_strnstr(pszSrc,"?",pszEndstop-pszSrc,SL_COMPARE_OPTION_NORMAL );
           if( pszQueryString != NULL )
            { HERE_I_AM__HTTPSRV();
              HttpSrv_ExtractSessionParamsFromQueryString( pHttpInst->pSession, pszQueryString );
              HERE_I_AM__HTTPSRV();
              // After this, pHttpInst->pSession->sz40UserName, ..->sz40UserCall, etc,
              // may have been from the "Referer:" . But if they are ALSO specified
              // in the 'real' query string, they will be overwritten a few lines
              // further below (because the info from the QUERY STRING is considered
              // more up-to-date than anything the browser said in the "Referer:".
            }
         }
      }
   } // end if < USER NAME not set yet -> search for it in the "Referer: " header line >


  // Run through the REAL Query String to extract the USER NAME, CALLSIGN, etc .
  // Note: This is done for ANY kind of request, i.e. "GET","POST", .
  pszSrc = pHttpInst->sz255QueryString; // look for more KEYS in the query string..
  pszEndstop = pszSrc + 255;
  (void)pszEndstop; // "assigned a value that is never used" .. oh, shut up, Mr Pedantic !

  HERE_I_AM__HTTPSRV();
  HttpSrv_ExtractSessionParamsFromQueryString( pHttpInst->pSession, pHttpInst->sz255QueryString );
  HERE_I_AM__HTTPSRV();
  HttpSrv_GenerateSessionID( pHttpInst->pSession, pHttpInst->pSession->sz8SessionID );
  if( pHttpInst->pSession->sz40UserName[0] != '\0' )
   { pHttpInst->pSession->iPermissions = CwNet_CheckUserAndGetPermissions( pCwNet,
         pHttpInst->pSession->sz40UserName, pHttpInst->pSession->sz40UserCall );
     if( pHttpInst->pSession->iPermissions >= 0 )
      { pClient->iClientState = CWNET_CLIENT_STATE_LOGIN_HTTP;
      }
   }
  else // NOT specifying a user name is only ok if the visitor's IP address
   {   // is already being used by another 'healthy' connection / session.
       // In that case, pHttpInst->pSession->sz40UserName WOULD BE NON-EMPTY.
       // Thus, getting here means 'unauthorized' ..
     HttpSrv_SendErrorAndCloseConnection( pCwNet, pClient, sz255RequestedURL/*pszBadURL*/, HTTP_STATUS__UNAUTHORIZED );
     // ,----'
     // '--> depending on pCwNet->cfg.iDiagnosticFlags (CWNET_DIAG_FLAGS_SHOW_CONN_LOG),
     //      emits messages like HTTP: Close 185.224.128.10:46040 (Unauthorized "/")
     HttpSrv_DeleteInstance( pCwNet, pClient );
     return;
   } // no USER NAME set (via query-string, in the same SESSION)

  if( pHttpInst->fIsNewSession || pHttpInst->fIsNewConnection ) // "new session" or "new connection" ? Show it...
   { // This replaces the connection-log-entry emitted by CwNet.c : CwNet_OnReceive() .
     if( pCwNet->cfg.iDiagnosticFlags & (CWNET_DIAG_FLAGS_VERBOSE | CWNET_DIAG_FLAGS_SHOW_CONN_LOG) )
      {
        ShowError( ERROR_CLASS_INFO | SHOW_ERROR_TIMESTAMP, "Server%d: New %s for %s on %s:%d: %s%s",
                (int)iClientIndex,  // 0 would be the 'local client', 1..
                pHttpInst->fIsNewSession ? "session" : "connection",
                pHttpInst->pSession->sz40UserName,
                CwNet_IPv4AddressToString( pClient->b4HisIP.b ), (int)pClient->iHisPort,
                HttpSrv_MethodToString(iMethod)/*contains trailing space*/, sz255RequestedURL );
      }
   }
  pHttpInst->fIsNewSession = pHttpInst->fIsNewConnection = FALSE; // not a "new" anymore (for the connection log)


  switch( iMethod )
   { case HTTP_METHOD_GET : // don't care about "content types" or the "requested resource" yet ...
        HERE_I_AM__HTTPSRV();
        iHttpStatus = HttpSrv_OnGET( pHttpInst );
        // 2024-02: Got iHttpStatus = 602 = HTTP_STATUS__DONT_RESPOND when a
        //  REGISTERED USER (with pHttpInst->pSession->iPermissions = 7)
        //  tried to retrieve
        //     pHttpInst->sz255RequestedURLWithoutQuery = "\links.htm" ,
        //  and was promptly blacklisted further below .
        //  Fixed in HttpSrv_OnGET() by returning HTTP_STATUS__NOTFOUND
        //  instead of HTTP_STATUS__DONT_RESPOND, if the user has a valid SESSION already.
        break; // end case HTTP_METHOD_GET
     default:  // unsupported "method" -> send error and exit .
        iHttpStatus = HTTP_STATUS__NOTIMPLEMENTED;
        break;
   }

  if( iHttpStatus == HTTP_STATUS__DONT_RESPOND )
   { // Don't send anything in response - see examples of "bad guys"
     // (attempted attacks from e.g. botnets) in HttpServer.c .
     // On his next attempt to connect (and try the next "funny request"),
     // this IP will already be rejected immediately after accept()
     // [unfortunately there is no easy way to NOT EVEN ACCEPT() him]
     HERE_I_AM__HTTPSRV();
     pszDest    = sz80Temp;
     pszEndstop = sz80Temp + sizeof(sz80Temp) - 1;
     SL_AppendString( &pszDest, pszEndstop, "(tried to '" );
     INET_DumpTraffic_HexOrASCII( pszDest, pszEndstop-pszDest, pbRcvdData, nBytesRcvd );
     SL_AppendString( &pszDest, pszEndstop, "')" );

     INET_BlockIPv4Address( &pCwNet->sBlacklist, pClient->b4HisIP.b, sz80Temp );
     // ,----'
     // '--> depending on pCwNet->cfg.iDiagnosticFlags (CWNET_DIAG_FLAGS_SHOW_CONN_LOG),
     //      emits messages like "Blacklisted 185.224.128.10 (Rejected by HTTP server)".
     HttpSrv_DeleteInstance( pCwNet, pClient );
   }
  else if( (iHttpStatus != 0) && (iHttpStatus != HTTP_STATUS__OK) )
   { HttpSrv_SendErrorAndCloseConnection( pCwNet, pClient, sz255RequestedURL/*pszBadURL*/, iHttpStatus );
     // ,----'
     // '--> depending on pCwNet->cfg.iDiagnosticFlags (CWNET_DIAG_FLAGS_SHOW_CONN_LOG),
     //      emits messages like HTTP: Close 185.224.128.10:46040 (Unauthorized "/")
     HttpSrv_DeleteInstance( pCwNet, pClient );
   }
  (void)cpEndOfLine; // POSSIBLY unused .. so what ?

} // end HttpSrv_OnReceive()

//----------------------------------------------------------------------------
int HttpSrv_OnGET( T_HttpInstance *pHttpInst )
  // [in] (most important): pHttpInst->sz255RequestedURLWithoutQuery[],
  //                        pHttpInst->sz255QueryString[] .
  // [out](most important): pHttpInst->szResponseHeader[], ->iRespHeaderLength;
  //                        pHttpInst->bTxBuffer[], ->iTxBufferLength;
  //                        pHttpInst->nServerState (e.g. C_HttpSrvState_SENDING_HEADER) .
  // Return value : Any of the familiar "HTTP Status Codes", hopefully HTTP_STATUS__OK,
  //                but expect HTTP_STATUS__NOTFOUND, etc etc.
{
  char *pszQueryString = NULL;
  const char *pszSrc, *pszEndstop, *pszEndOfHeader;
  char *pszDest;
  int  iBuiltInPage, iLen, iLen2;

  // Parse the initial header lines (aka "fields") inside the GET request .
  // The rest of the HTTP REQUEST is usually ignored,
  //   and there is an AWFUL LOT MORE in it !
  //   This is what pszString may point to at the moment, for GET-, but also for POST-requests:
  // - The HTTP Version : "HTTP/1.1"
  // - The "Host" (which may be just the client's IP)
  // - The "User-Agent" (which was "Mozilla/5.0 (Windows; U; Windows NT 5.1; de; blabla)
  // - a list of all "acceptable" stuff, whatever this means :
  //   "Accept: text/xml,application/xml,application/xhtml+xml,text/html;q=0.9 ..blabla)
  //   "Accept-Encoding: gzip,deflate"
  //   "Accept-Charset: ISO-8859-1,utf-8;q=0.7,*;q=0.7"
  // - other stuff which the author knew nothing about:
  //   "Keep-Alive: 300"
  //   "Connection: keep-alive"
  //   "Cache-Control: max-age=0"
  // Inspired by 'http_post_request()' from LwIP-Contrib/.../httpd.c :
  pszSrc     = pHttpInst->szRequestHeader;
  pszEndstop = pszSrc + pHttpInst->i32ReqHeaderLength;
  pszEndOfHeader = SL_strnstr( pszSrc/*haystack*/, "\r\n\r\n"/*needle*/,
        pHttpInst->i32ReqHeaderLength/*maxlen*/, SL_COMPARE_OPTION_NORMAL );
  if( pszEndOfHeader != NULL )  // ok, found the empty line (0x0D 0x0A 0x0D 0x0A)
   { pHttpInst->dwAcceptedContentTypes = C_HttpSrv_ContentType_All; // when NOT specified (as "Accept:"), guess *ALL* content-types are allowed (for the response)
     HERE_I_AM__HTTPSRV();
     while( (*pszSrc!=0) && (pszSrc<pszEndOfHeader) ) // parse what we need to know about "GET" ...
      { // search for stuff like "Content-Type:", "User-Agent:", etc
        //     (HTTP is an extremely 'chatty' protocol.. full of stuff that we're not interested in)
        const char *cp2;
        const char* cpEndOfLine = SL_strnstr( pszSrc, "\r\n", (pszEndstop-pszSrc)/*maxlen*/,SL_COMPARE_OPTION_NORMAL);
        T_HttpSrv_KeywordToken kwd = (T_HttpSrv_KeywordToken)SL_SkipOneOfNStrings_AnyCase( &pszSrc, HttpSrv_Keywords );
        // -> kwd==0 when NONE of the keywords was found !
        // HttpSrv_Keywords contains stuff like "\r\nContent-Type:", etc .
        SL_SkipSpaces( &pszSrc ); // skip be OPTIONAL(!!) space(s) after the colon (colon already skipped as part of the keyword)
        switch( kwd )
         { case C_HttpSrv_Kwd_Content_Type:  // found and skipped "Content-Type: "
                // Example for a posted FORM (from script.htm, posted by IRON):
                //  cp  = "application/x-www-form-urlencoded faselblah..."
                //        -> nContentType=C_HttpSrv_ContentType_Application_URLEncoded
                pHttpInst->nContentType = (T_HttpSrv_ContentType)SL_SkipOneOfNStrings_AnyCase(&pszSrc, HttpSrv_ContentTypes );
                break; // end case C_HttpSrv_Kwd_Content_Type
           case C_HttpSrv_Kwd_Content_Length:  // found and skipped "\r\nContent-Length: "
                pHttpInst->i32ReqContentLength = SL_atoi( pszSrc );
                // (this also applies to the COMPLETE multi-part message;
                //  which may be several hundred kilobytes
                //  - MUCH MORE than any buffer in RAM permits [in an embedded system] ! )
                break; // end case C_HttpSrv_Kwd_Content_Length

           case C_HttpSrv_Kwd_User_Agent:     // found and skipped "\r\nUser-Agent: "
                // Try to detect mobile phones. As usual with HTTP & HTML, this turned out
                // into a nightmare because everything is so non-standardized (aka bastardized).
                // Don't blame us if we don't properly detect YOUR mobile device.
                // Like many other elements in the HTTP request, the "User-Agent" (string) ends
                // with carriage return + new line:
                cp2 = SL_strnstr( pszSrc/*Haystack*/, "Android"/*Needle*/, (cpEndOfLine-pszSrc)/*HaystackLength*/, 0/*iOptions*/ );
                if( cp2 != NULL )
                 { pHttpInst->isMobile = TRUE; // guess the visitor is a mobile phone / smartphone / tablet .
                   // Got here when "visiting" our embedded device with a chatty smartphone (Samsung S4 mini) :
                   //  cp = "Mozilla/5.0 (Linux; Android 4.4; de-de; SAMSUNG GT-I9195 Build/KOT49H)" ...
                   //
                   // Note: To check the webpage's appearance "for mobile devices without a mobile device",
                   //       the "User-Agent" sent by Firefox could be modified as follows:
                   //  - enter the dreadful "about:config" page in Firefox
                   //  - find your way through huge piles of junk, look for "general.useragent.override" .
                   //    If that entry doesn't exist, create it as explained on
                   //      http://www.howtogeek.com/113439/how-to-change-your-browsers-user-agent-without-installing-any-extensions
                   //  - The TYPE of that entry is 'string'. Set the value to whatever you like, e.g.
                   //      Mozilla/5.0 (Mobile; MeinTollesSmartphoneWelchesSieNichtKennen_Harr_Harr)  [<- google refuses to work with this !!]
                   //      Mozilla/5.0 (Mobile; what else do you need to know)       [<- google doesn't recognize this as a mobile device]
                   //      Mozilla/5.0 (Android 4.4; what else do you need to know)  [<- this is enough for google to recognize a MOBILE device]
                 }
                cp2 = SL_strnstr( pszSrc/*Haystack*/, "BlackBerry"/*Needle*/, (cpEndOfLine-pszSrc)/*HaystackLength*/, 0/*iOptions*/ );
                if( cp2 != NULL )
                 { pHttpInst->isMobile = TRUE; // guess the visitor is a mobile phone / smartphone / tablet
                 }
                cp2 = SL_strnstr( pszSrc/*Haystack*/, "Mobile"/*Needle*/, (cpEndOfLine-pszSrc)/*HaystackLength*/, 0/*iOptions*/ );
                if( cp2 != NULL )
                 { pHttpInst->isMobile = TRUE; // guess the visitor is a mobile phone / smartphone / tablet
                   // Got here with cp = "Mozilla/5.0 (iPhone; bla bla bla bla bla Gecko) Mobile/11A465 Safari/9537.53"
                 }
                // Don't trust the User Agent ! Only a few scanners really tell who they are, e.g.:
                //  "User-Agent: masscan/1.0 (https://github.com/robertdavidgraham/masscan)"
                // Some joker (viktor?) even sent: "User-Agent: polaris botnet" .
                break; // end case C_HttpSrv_Kwd_User_Agent

           case C_HttpSrv_Kwd_Connection:     // found and skipped "\r\nConnection: ", followed by stuff like "keep-alive"..
                pHttpInst->dwConnectionPersistance = HttpSrv_ParseBitCombinationOfCommaSeparatedKeywords( pszSrc, HttpSrv_ConnectionTypes );
                break; // end case C_HttpSrv_Kwd_Connection

           case C_HttpSrv_Kwd_Accept:        // found and skipped "\r\nAccept: ", followed by stuff like "text/plain", etc etc etc etc etc etc etc etc etc etc etc etc
                //  Vielleicht mchte der Client uns auf diesem Weg mitteilen,
                //  dass er einen bestimmten "Content-Type" *NICHT* untersttzt.
                //  Ein Browser ist i.A., je nach Art des Requests, manchmal "offen fr alles",
                //  und teilt uns dies mit dem folgendem, extrem schnatterhaften Header-Feld mit:
                // > Accept: text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8
                //  (was ungefhr bedeutet: "Am liebsten text/html, oder, oder, und; notfalls aber auch ALLES ANDERE (*/* = wildcard).
                pHttpInst->dwAcceptedContentTypes = HttpSrv_ParseAcceptedContentTypes( pszSrc );  // -> bitwise combination of C_HttpSrv_ContentType_ ...
                break;

           case C_HttpSrv_Kwd_Upgrade: // found and skipped "\r\nUpgrade: ", typically followed by stuff like "websocket".
                // (to confuse readers, the Wikipedia article uses CamelCasing like "WebSocket",
                //  but as MANY -not all- things in HTTP, these keywords seem to be
                //  case-insensitive).
                // In that case, pHttpInst->szRequestedURLWithoutQuery
                // should begin with something like "ws/" . But HERE, we don't care.
                // > The Upgrade header field is an HTTP header field introduced in HTTP/1.1.
                // > In the exchange, the client begins by making a cleartext request,
                // > which is later upgraded to a newer HTTP protocol version
                // > or switched to a different protocol. Connection upgrade must be
                // > requested by the client; if the server wants to enforce an upgrade
                // > it may send a 426 Upgrade Required response. The client can then
                // > send a new request with the appropriate upgrade headers
                // > while keeping the connection open.
                //   .....  plus, for "websockets" alias "WebSockets" :
                // > First, a client requests a WebSocket connection by using the
                // > Upgrade: WebSocket
                // >      and
                // > Connection: Upgrade
                // >      headers,
                // > along with a few protocol-specific headers to establish
                // > the version being used and set up a handshake. The server,
                // > if it supports the protocol, replies with the same
                // > Upgrade: WebSocket and Connection: Upgrade headers
                // > and completes the handshake. (.......)
                // What the example failed to mention is that like many other
                //  "header lines" in HTTP, the "Upgrade: "-thingy may be followed
                //  by a list of COMMA-SEPARATED keywords, thus the flags
                //  stored in pHttpInst->dwUpgradeRequested are bitwise combineable.
                //  Furthermore, from tools.ietf.org/html/rfc6455#section-1.3 :
                // >  The method of the request MUST be GET, and the HTTP version MUST
                // >  be at least 1.1.
                // >  For example, if the WebSocket URI is "ws://example.com/chat",
                // >  the first line sent should be "GET /chat HTTP/1.1".
                //
                // The really tough stuff (server for *certain* "websockets")
                // is in c:\cbproj\SpecLab\http_server_sourcecode\OpenWebRX_Server.c .
                pHttpInst->dwUpgradeRequested = HttpSrv_ParseBitCombinationOfCommaSeparatedKeywords( pszSrc, HttpSrv_UpgradeTypes );
                // The WebSocket "Handshake"-thing between client and server
                //  is neither implemented HERE nor in SpecHttpSrv.cpp
                //  but in OpenWebRX_Server.c ...
                break;

           default:  // everything else is IGNORED in HttpSrv_ParseRequest(),
                // the 'unknown line' will be skipped further below.
                break;
          } // end switch( kwd )
        // Skip the rest of this line (until, but not including, "\r\n") .
        // It's important to skip all 'unknown' header lines
        //   because HTTP is an extremely chatty protocol  !
        pszSrc = cpEndOfLine+2;
        if( pszSrc>(pszEndOfHeader-2) )
         { break;  // no more HEADER LINES to parse here !
         }
      } // end while < more characters in the HEADER lines >
     HERE_I_AM__HTTPSRV();
     // After wading through the swamp of "header lines",
     // and knowing the 'resource' requested via GET,
     // send WHAT to the 'known user' ?
     pszSrc = (char*)HttpSrv_FindBuiltInPage( pHttpInst->sz255RequestedURLWithoutQuery );
     if( pszSrc == NULL ) // Not a "simple, built-in page" (or fragment, javascript, etc) ->
      { pszSrc = pHttpInst->sz255RequestedURLWithoutQuery;
        iBuiltInPage = SL_SkipOneOfNTokens( &pszSrc, HttpSrv_BuiltInPages );
        // Some of these "built-in PAGES" are in fact BINARY FILES :
        switch( iBuiltInPage )
         { case C_HttpSrvFile_LiveAudio_ogg: // ENDLESS "stream", cannot use strlen() on this !
              return HttpSrv_StartStreamingLiveAudio( pHttpInst );
                  // '--> sends the HTTP response header with proper MIME type,
                  //      followed by a few initial "Ogg pages" in the
                  //      BINARY response body, then returns HTTP_STATUS__OK .

           case C_HttpSrvFile_Favicon_ico : // BINARY "file", cannot use strlen() on this !
              return HttpSrv_SendBinaryBlock( pHttpInst, "image/x-icon"/*MIME-type*/,
                         HttpSrv_bFavicon, HttpSrv_iSizeofFavicon );
           default: break;
          }
        // Arrived here ? Out of luck ... politely say "Not found" or kick out a bad guy ?
        //   See common treatment near the end of HttpSrv_OnGET() .
      }
     else // found a TEMPLATE for the requested page, so 'expand' it in RAM
      { // using T_HttpInstance.bTxBuffer[C_HttpSrv_TX_BUFFER_SIZE].
        // The bunch of RESPONSE HEADERS will be created afterwards,
        // in T_HttpInstance.szResponseHeader[C_HTTP_SRV_MAX_SIZE_OF_RESPONSE_HEADERS],
        // when we know THE SIZE of the "response body" .
        HttpSrv_PrepareBuildingResponseBody( pHttpInst, &pszDest, &pszEndstop );
        HERE_I_AM__HTTPSRV();
        iLen  = HttpSrv_GenerateHTMLFromTemplate( pHttpInst, &pszSrc, &pszDest, pszEndstop );
        iLen2 = HttpSrv_FinishBuildingResponseBody( pHttpInst, pszDest );
        if( iLen2 != iLen )
         { ShowError( ERROR_CLASS_INFO | SHOW_ERROR_TIMESTAMP, "HTTP: Tx-Buffer to small - need %d, got %d bytes",
               (int)iLen, (int)iLen2 );
         }
        HERE_I_AM__HTTPSRV();
        if( iLen2 > 0 )
         { // Now, since we know the 'content length', assemble the response-header-lines :
           HttpSrv_PrepareBuildingResponseHeaders( pHttpInst, &pszDest/*out*/, &pszEndstop/*out*/ );
           // Build the chatty 'GET reponse' in RAM (including all required "headers").
           SL_AppendPrintf( &pszDest, pszEndstop, // here: "GET" response ..
               "HTTP/1.1 200 OK\r\n" // protocol ver 1.1 REQUIRES A "Content-Length" field !
               "Content-Type:text/html; charset=utf-8\r\n"
               "Keep-Alive: timeout=5\r\n"   // timeout in seconds (no need to specify a "max" here)
               "Connection: Keep-Alive\r\n"
               "Content-Length:%d\r\n"
               "\r\n",   // <- empty line indicates the end of the HTTP-header
              (int)iLen2 );
           HttpSrv_FinishBuildingResponseHeaders( pHttpInst, pszDest );
           pHttpInst->nServerState = C_HttpSrvState_SENDING_HEADER;
           HERE_I_AM__HTTPSRV();
           return HTTP_STATUS__OK;
           // Next steps:
           //  * The response ("HEADER" and "BODY") will be
           //     sent in the next call to HttpSrv_OnPoll().
           //  * Because we want to recycle the same connection for
           //     OTHER requests, pHttpInst->nServerState will
           //     switch from    C_HttpSrvState_SENDING_HEADER
           //              to    C_HttpSrvState_SENDING_DATA
           //     or directly to C_HttpSrvState_RECEIVING_REQUEST again.
         } // end if < HttpSrv_GenerateHTMLFromTemplate() successful >
      }   // end if < found a 'template' (formerly in a uC's ROM / Flash memory) >
   }     // end if < found the empty line between "header lines" and "content" >

  if(  (strcmp( pHttpInst->sz255RequestedURLWithoutQuery, "/") == 0 )
    || (strcmp( pHttpInst->sz255RequestedURLWithoutQuery, "/favicon.ico") == 0 ) // <- the game that FIREFOX usually plays..
    )
   { // A more-or-less well behaving visitor, trying to read the main page,
     // but without a valid user name :
     HERE_I_AM__HTTPSRV();
     return HTTP_STATUS__NOTFOUND; // send the infamous "404" and say farewell
     // (this is possible WITHOUT using allocating a T_HttpInstance and its buffers)
   }
  if( pHttpInst->pSession != NULL )
   { if( pHttpInst->pSession->iPermissions > 0 ) // a visitor with a valid SESSION (and permissions)..
      { // has hit a broken link in our 'server pages'. Don't blacklist him but say "404" :
        HERE_I_AM__HTTPSRV();
        return HTTP_STATUS__NOTFOUND; // send the infamous "404" and say farewell
      }
   }


  // Arrived here ? Whatever the 'accidental visitor' (or bad guy) tried
  // to GET is not what we expect, so kick him out. A few examples:
  //  * GET "/solr/admin/info/system?wt=json" (from 83.97.73.245)
  //  * GET "/?XDEBUG_SESSION_START=phpstorm" (from 83.97.73.245)
  //  * GET "/console/"                       (from 83.97.73.245)
  //  * GET "/_ignition/execute-solution"     (from 83.97.73.245)
  //  * GET "/actuator/gateway/routes"        (from 83.97.73.245)
  //  * GET "http://141.98.7.179/a.php"       (from 141.98.7.179)
  //  * POST "/boaform/admin/formLogin"       (from 89.190.156.234)
  //  * GET "/index.php?s=/index/<09>hink<07>pp/invokefunction&function=call_user_func_array ..."
  // -> Put those 'accidental visitors' (e.g. attacking bots) on an automatic
  //    blacklist of 'ill-behaving IPs' to stop them from stealing OUR BANDWIDTH.
  if( pHttpInst->pCwNet->cfg.iDiagnosticFlags & CWNET_DIAG_FLAGS_VERBOSE )
   {
     ShowError( ERROR_CLASS_INFO | SHOW_ERROR_TIMESTAMP, "HTTP: blacklisted %s for requesting %s",
           CwNet_IPv4AddressToString( pHttpInst->pSession->b4HisIP.b ),
           pHttpInst->sz255RequestedURLWithoutQuery );
   }
  HERE_I_AM__HTTPSRV();
  (void)pszQueryString; // POSSIBLY unused - so what (better than possibly non-initialized..)

  return HTTP_STATUS__DONT_RESPOND; // "blacklist him, and say farewell"

} // end HttpSrv_OnGET()

#if(0)
//---------------------------------------------------------------------------
void HttpSrv_ParseRequest( T_HttpInstance *pHttpInst, BYTE *pbRxBuffer, int iRcvdLength)
  // Called from HttpSrv_OnGET() [etc] after COMPLETION of the reception
  //         of a request like "GET".
  // In the former (more complete) HTTP server, this told us WHAT to read
  // (a file name, etc), sometimes with a QUERY STRING appended (after the first
  // question mark),  and "how exactly" the peer (a web browser?) wanted to
  // get those data delivered, whether to keep the socket or close it, etc.
{

  // Build the full path to a file on the local disk storage ?
  // In many cases, this file is only a 'template' for what the application
  // will actually 'serve' to the remote client (later, in another module):
  SL_strncpy( pHttpInst->szPathAndName, szWebRoot/*e.g. "C:\Somewhere\Something\"*/, C_HTTP_SRV_MAX_PATH-20 );
  // Append the requested 'path' (part of the URL) to the web server's path.
  // Caution, if the path ends with a backslash, and the filename begins
  // with a slash (or, maybe, another backslash), skip the first character
  // from the filename. Otherwise we may end up with crap like this:
  //   c:\cbproj\SpecLab\server_pages\/index.html  .
  // But caution, the result is still a crude mix of DOS- and UNIX file paths,
  // for example c:\cbproj\SpecLab\server_pages/index.html (backslash+slash!) .
  cp = pHttpInst->szPathAndName;
  iLen = SL_strnlen( cp, C_HTTP_SRV_MAX_PATH-20 );
  if(iLen>0)
   { cp += (iLen-1); // cp now points to the last character in what used to be the "web root" folder
   }
  if( *cp=='/' )  // oops.. whatever this was supposed to be, it won't work UNDER WINDOWS
   { *cp = '\0';  // truncate the copied "web root" folder, eliminating the last slash
   }
  else if( *cp=='\\' ) // trailing backslash was meant to be a "folder separator", but we'll append our own further below
   { *cp = '\0';  // truncate the copied "web root" folder, eliminating the last backslash
   }
  cp2 = cp+strlen(cp); // pointer where a part of the "path" (from the URL) may be appended
  if (strlen(pHttpInst->szRequestedURLWithoutQuery) <= 1) // the "path" (part of the URL) seems to be "/" or nothing at all..
   {  // For experiments with HA7ILM's OpenWebRX, the caller may try "index.wrx" instead:
      // > The file index.wrx is the HTML layout for the web GUI of OpenWebRX.
      // > It contains some special tags like %[WS_URL] that the web server replaces
      // > with actual values on every request. Images, style sheet and
      // > script files are referenced from within index.wrx.
      cp = cp2;
      SL_AppendString( &cp/*ppszDest*/, pHttpInst->szPathAndName+C_HTTP_SRV_MAX_PATH/*pszEndstop*/, "\\index.html" );
      if( YHF_FILE_DoesDirectoryOrFileExist( pHttpInst->szPathAndName ) == YHF_F_BOTH_DIRECTORY_AND_FILE_EXIST )
       { // good guess, use this "file" :
       }
      else
       { cp = cp2;
         SL_AppendString( &cp/*ppszDest*/, pHttpInst->szPathAndName+C_HTTP_SRV_MAX_PATH/*pszEndstop*/, "\\index.wrx" );
         if( YHF_FILE_DoesDirectoryOrFileExist( pHttpInst->szPathAndName ) == YHF_F_BOTH_DIRECTORY_AND_FILE_EXIST )
          { // good guess, use this file (cpToken)
          }
         else // what else to try besides "index.html" and "index.wrx" .. "index.htm" ?
          { cp = cp2;
            SL_AppendString( &cp/*ppszDest*/, pHttpInst->szPathAndName+C_HTTP_SRV_MAX_PATH/*pszEndstop*/, "\\index.htm" );
          }
       }
   } // end if < "root" without path/filename > ?
  else // request ( szRequestedURLWithoutQuery ) seems to contain a non-empty "path" ->
   { // Combine the local "web root" folder with whatever the remote client was asking for,
     // e.g. :
     //   [in] szTempName = "C:\OpenWebRX\htdocs"
     //   [in] pHttpInst->szRequestedURLWithoutQuery = "/nanoscroller.css"
     //  [out] pHttpInst->szPathAndName = "C:\OpenWebRX\htdocs\nanoscroller.css"
     //                   Note the BACKSLASH for windoze ! -->|
     cp = cp2;
     SL_AppendString( &cp/*ppszDest*/, pHttpInst->szPathAndName+C_HTTP_SRV_MAX_PATH/*pszEndstop*/,
           pHttpInst->szRequestedURLWithoutQuery ); // <- e.g. "/nanoscroller.css"
   }
  // Defeat the Posix-incompatibility of older windows versions :
  INET_ReplaceCharInPath( pHttpInst->szPathAndName, C_HTTP_SRV_MAX_PATH,
                           '/'/*cOriginal*/, '\\'/*cReplacement*/ );


  // Preset the HTTP RESPONSE HEADER LINES with some hopefully meaningful
  // defaults. The application may modify these fields if necessary:
  cp = strrchr(pHttpInst->szPathAndName, '.' );
  pHttpInst->nContentType = HttpSrv_GetContentTypeFromFileExtension( cp );
     // Since the KiwiSDR-folks broke the above, by renaming OpenWebRX's "index.wrx"
     // into "index.html" (yucc..), we must fry an extra sausage to support both:
     //
  // ex: pHttpInst->i32ContentLength = 0; // 0 as long as the 'Content-Length' FOR THE RESPONSE is unknown
  //     Leave pHttpInst->i32ContentLength unchanged (should be -1 = "NOT SET" in a GET-REQUEST),
  //     to let the application -

  // Before sending the requested file, inform the application (i.e. SpecLab).
  //   Reason: The file must possibly be generated on-the-fly !
  //   Since 2013-05-16, the "file" may optionally be just be a BLOCK OF RAM,
  //         or an HTML template (like *.wrx for OpenWebRX, or index.html
  //         for "KiwiSDR" who for strange reasons renamed the file),
  //     In any case (*.wrx or Kiwi's "index.html"), many tokens
  //         must be replaced by fragments of text. In SpecLab, that happens
  //         in OpenWebRX_Server.c :: OWRX_OnWebRXTemplateRequest() .
  // Everything the application needs to know  is packed inside a
  // T_HttpInstance structure. A pointer to that struct will be passed
  // to the application in the LPARAM value of a HTTP_MSG_FILE_REQUEST message.
  // The SendMessage function sends the specified message to a window.
  // The function calls the window procedure for the specified window
  // and does not return until the window procedure has processed the message.
  // Important !  PostMessage(), which returns immedately, wouldn't work here.
  //              SendMessage() blocks the caller (that's us) until the
  //              to-be-sent "file" has been prepared, either as a REAL file,
  //              or just some block of data in memory (RAM).
  //              In Spectrum Lab, all that happens in
  //                 c:\cbproj\SpecLab\SpecHttpSrv.cpp :
  //                  TSpectrumLab::OnHttpFileRequestOrCompletion(),
  //                 if pHttpInst->nContentType == HTTPSRV_CONTENT_TYPE_TEXT_WRX.
  iResult = HttpSrv_SendMsgToApplication( HTTP_MSG_FILE_REQUEST, (long)pHttpInst );
  // At this point, the "application" has produced the FILE to send in the response,
  //                or a block of data in RAM with "dynamic content", or whatever...
  switch( iResult )
   { case HTTP_STATUS__OK :
     case HTTP_STATUS__CREATED :
        // The application said "ok, I will send the requested file" (which may be just a block of RAM, since 2013)
        // Note: The name of the file actually sent MAY HAVE BEEN MODIFIED by the application,
        // for example if the application decided to place the response in a RAMDISK, etc.
        HttpSrv_BeginToSendFile( pHttpInst ); // .. send file or data from a block in RAM
        break;

     case HTTP_STATUS__SWITCHING_PROTOCOLS :
        // the application indicated (via iResult) to "switch protocols".
        // All required header lines must have been appended to the response
        // by the application, for example in SpecHttpSrv.cpp : OnHttpWebSocketRequest() .
        // Send as much of the response (-header) as we can :
        HttpSrv_SendResponseHeaderAndFileContents( pHttpInst );
        break;

     case HTTP_STATUS__DONT_RESPOND : // blacklist visitor, show his request in the log, and close without a response.
        // Visitor was identified as malicious, wannabe-hacker, or similar:
        if( HttpSrv_BlacklistClientIP( pHttpInst->dwClientIP, NULL ) )
         { HttpSrv_LogEvent( HTTP_HIGHLIGHT_PURPLE | HTTP_EFLAG_ERROR,
              "Auto-blackisted IP %s (see 'Statistics')",
               HttpSrv_IPAddressToString( pHttpInst->dwClientIP ) );
         }
        // For the sysop's curiosity, show the full request header
        //     using similar syntax as in http_intf.c:HttpSrv_FileRequested() :
        HttpSrv_LogEvent( HTTP_HIGHLIGHT_NORMAL | HTTP_EFLAG_VERBOSE,
          "Connection #%d, from %s (now blacklisted) : hdr[%d]=\"%s\"",
          (int)pHttpInst->iConnIndex,
          (char*)HttpSrv_IPAddressToString(pHttpInst->dwClientIP), // <- possibly a hacker ?
          (int)pHttpInst->i32ReqHeaderLength,  // <- header length in bytes
          (char*)pHttpInst->szRequestHeader ); // <- beware, this request-header may be ILL-FORMATTED !
        // No-No: HttpSrv_SendError( pHttpInst, iResult );
        // Don't leak out any info but pull the plug from this end :
        HttpSrv_CloseConnection(pHttpInst); // caution, pHttpInst is invalid after this !
        break;

     default :
        // The application isn't happy about the request, so :
        //     Send the error code to the remote HTTP client .
        HttpSrv_SendError( pHttpInst, iResult );
        HttpSrv_LogEvent( HTTP_HIGHLIGHT_RED | HTTP_EFLAG_ERROR,
                 "Application refused to send %s : %s",
                 cpToken,  HttpSrv_HttpStatusCodeToString( iResult ) );
        HttpSrv_CloseConnection(pHttpInst); // caution, pHttpInst is invalid after this !
   } // end switch( iResult )

} // end HttpSrv_ParseRequest()
#endif // (0)

//--------------------------------------------------------------------------
void HttpSrv_CheckUserNameAndUpdatePermissions( T_HttpInstance *pHttpInst, char *pszUserName )
{
} // end HttpSrv_CheckUserNameAndUpdatePermissions()


//--------------------------------------------------------------------------

void HttpSrv_CheckUserCallAndUpdatePermissions( T_HttpInstance *pHttpInst, char *pszUserCall )

{
} // end HttpSrv_CheckUserCallAndUpdatePermissions()


//--------------------------------------------------------------------------

void HttpSrv_CheckPasswordAndUpdatePermissions( T_HttpInstance *pHttpInst, char *pszPassword )
{
} // end HttpSrv_CheckPasswordAndUpdatePermissions()


//--------------------------------------------------------------------------

void HttpSrv_CheckSessionIDAndUpdatePermissions(T_HttpInstance *pHttpInst, char *pszSessionID )
{
} // end HttpSrv_CheckSessionIDAndUpdatePermissions()


//--------------------------------------------------------------------------

void HttpSrv_GenerateSessionID( T_HttpSession *pSession, char *psz8SessionID )
  // Inspired by a Wikipedia article on 'Query_string', chapter "Tracking" :
  // > A program receiving a query string can ignore part or all of it.
  // > If the requested URL corresponds to a file and not to a program,
  // > the whole query string is ignored.  (blah, .. server log files, irrelevant)
  // > These facts allow query strings to be used to track users in a manner
  // > similar to that provided by HTTP cookies. For this to work, every time
  // > the user downloads a page, a unique identifier must be chosen and added
  // > as a query string to the URLs of all links the page contains. As soon as
  // > the user follows one of these links, the corresponding URL is requested
  // > to the server. This way, the download of this page is linked with the
  // > previous one.
  //
  // WB: We don't use it for TRACKING USER ACTIVITY but to manage a single
  //     "Session", since modern browsers open a multitude of SIMULTANEOUS
  //     connections (e.g. Sockets, or LwIP's equivalent..). All of those
  //     MULTIPLE connections between HTML browser and HTTP server shall use
  //     the same "session ID" (in our server, "sid=NNNNNNNN", where NN is hexadecimal).
  //     What we want is to prevent script kiddies from playing around:
  //  * The first visit (when there is no valid "session ID") shall always have
  //     at least the visitor's user name in the query string ("name=Moritz"),
  //        plus optionally the visitor's amateur radio callsign ("call=SWL123"),
  //        or even a password ("pwd=DontCallMeAdmin") .
  //  * From the above, plus maybe some kind of unique counter,
  //     this server generates a SESSION ID - see HttpSrv_GenerateNewSessionID().
  //  * In pages LINKED within this server, neither USER NAME, CALLSIGN, nor PASSWORD
  //     will be emitted but the session ID, to keep track the visitor's
  //     permissions, callsign, and whatever may be necessary when filling out
  //     dynamically created content - see HttpSrv_GenerateHTMLFromTemplate() .
  //  * With a 32-bit session ID, there's a change of one in 4294967295 for a
  //     collision between two logged-in users -> No need for an extra check...
  //
{
  // Actually, the 'session ID' uses the 32-bit CRC implemented in Utilities.c :
  DWORD dw = UTL_CRC32(0/*seed*/, pSession->b4HisIP.b, 4/*bytes*/ );
  dw = UTL_CRC32( dw/*seed*/, (BYTE*)pSession->sz40UserName, strlen(pSession->sz40UserName) );
  dw = UTL_CRC32( dw/*seed*/, (BYTE*)pSession->sz40UserCall, strlen(pSession->sz40UserCall) );
  dw = UTL_CRC32( dw/*seed*/, (BYTE*)pSession->sz40Password, strlen(pSession->sz40Password) );
  sprintf( psz8SessionID, "%08lx", dw ); // <- appended as a query string when linking to sub-pages
} // end HttpSrv_UpdateSessionID()


//--------------------------------------------------------------------------

int  HttpSrv_SendBinaryBlock( T_HttpInstance *pHttpInst, char *pszMimeType,

                               BYTE *pbSource, int iContentLength )
  // Called from HttpSrv_OnGET() to deliver "short binary content",
  //  that already exists in a simple BYTE-array somewhere,
  //  short enough to fit inside <C_HttpSrv_TX_BUFFER_SIZE> bytes .
{
  char *pszDest, *pszResponsePayload;
  const char *pszEndstop;
  int  nBytesAvailable;

  HERE_I_AM__HTTPSRV();

  pHttpInst->iStreamType        = C_HttpSrv_StreamType_None;
  pHttpInst->nTransferEncoding  = C_HttpSrv_XferEncoding_None;

  if( (iContentLength<=0 ) || (pbSource==NULL) )
   { return HTTP_STATUS__NOTFOUND;
   }
  if( iContentLength > (C_HttpSrv_TX_BUFFER_SIZE-16) )
   { return HTTP_STATUS__INSUFF_STORAGE;  // here: "too much; increase C_HttpSrv_TX_BUFFER_SIZE"
   }

  HttpSrv_PrepareBuildingResponseBody( pHttpInst, &pszDest/*out*/, &pszEndstop/*out*/ );
  // ,--------------------------------------------'
  // '--> points into pHttpInst->bTxBuffer now
  nBytesAvailable = pszEndstop-pszDest;
  if( (iContentLength+16) >= nBytesAvailable )
   { return HTTP_STATUS__INSUFF_STORAGE;  // too much for a deeply embedded web server without dynamic memory allocation..
   }
  memcpy( pszDest, pbSource, iContentLength );
  pszDest += iContentLength;
  // Note: Appending a "\r\n" after BINARY DATA would be really, really foolish !
  HttpSrv_FinishBuildingResponseBody( pHttpInst, pszDest/*in*/ );

  HttpSrv_PrepareBuildingResponseHeaders( pHttpInst, &pszDest/*out*/, &pszEndstop/*out*/ );
  // ,------------------------------------------------'
  // '--> points into pHttpInst->szResponseHeader, as for any GET or POST response-"headers"
  SL_AppendPrintf( &pszDest, pszEndstop, // here: "GET" response for AUDIO with "chunked encoding"
     "HTTP/1.1 200 OK\r\n" // protocol ver 1.1 REQUIRES A "Content-Length" field !
     "Content-Type: %s\r\n"
     "Server: Remote CW Keyer\r\n"
     "Keep-Alive: timeout=5, max=100\r\n" // timeout in seconds (do we really need the "max" here ?)
     "Connection: Keep-Alive\r\n"
     "Content-Length: %d\r\n"
     "\r\n", // <- empty line indicates the end of the HTTP-header
    pszMimeType, (int)iContentLength );
  HttpSrv_FinishBuildingResponseHeaders( pHttpInst, pszDest );

  pHttpInst->nServerState = C_HttpSrvState_SENDING_HEADER;  // .. here: and the FIRST PART of the "body"
  HERE_I_AM__HTTPSRV();
  return HTTP_STATUS__OK;

} // end HttpSrv_SendBinaryBlock()



//---------------------------------------------------------------------------

int HttpSrv_FinishTransferChunk( BYTE *pbDest, int iNettoChunkSize ) // .. im RAM !
  // Moves the already existing "payload" (at pbDest) a few bytes up in memory,
  // then inserts a HEXADECIMAL ASCII size indicator (for the "netto size"),
  //      and appends another "\r\n" *after* the payload - see examples below.
  // ON ENTRY, pbDest[0...iNettoChunkSize] already contains the PAYLOAD
  //           *WITHOUT* the size indicator, and *WITHOUT* the trailing "\r\n".
  // ON EXIT,  the first few bytes in pbDest contain the HEXADECIMAL SIZE INDICATOR,
  //           followed by the payload (that was moved up in memory by a few bytes),
  //           and (after the payload) a trailing "\r\n" as explained below.
  // Return value : Number of bytes to send, including ...
  //                 * the hexadecimal chunk size indicator plus "\r\n"
  //                 * the netto chunk size
  //                 * two bytes for the trailing "\r\n" *after* the chunk.
  //
  // About "Transfer-Encoding: chunked", from developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Transfer-Encoding:
  // > chunked
  // > -------
  // > Data is sent in a series of chunks. The Content-Length header is omitted
  // > in this case and at the beginning of each chunk you need to add the length
  // > of the current chunk in hexadecimal format, followed by '\r\n' and then
  // > the chunk itself, followed by another '\r\n'. The terminating chunk is a
  // > regular chunk, with the exception that its length is zero. It is followed
  // > by the trailer, which consists of a (possibly empty) sequence of header fields.
  // > A chunked response looks like this:
  // > HTTP/1.1 200 OK
  // > Content-Type: text/plain
  // > Transfer-Encoding: chunked     <- instead of the usual "Content-Length"
  // >                                <- empty line, end of HTTP RESPONSE HEADER
  // > 7\r\n                 <- length of the first chunk in ASCII/HEX
  // > Mozilla\r\n           <- chunk (7 bytes) followed by another "\r\n"
  // > 11\r\n                <- length of the second chunk in ASCII/HEX
  // > Developer Network\r\n <- chunk (17 bytes) followed by another "\r\n"
  // > 0\r\n                 <- length of the third chunk = ZERO -> LAST !
  // > \r\n                  <- "trailer", here: "empty sequence of header fields"
  //
{
  char sz15ChunkSizeInHex[16];
  int  iSizeIndicatorLength, nTotalBytesToSend;
  sprintf( sz15ChunkSizeInHex, "%X\r\n", (unsigned int)iNettoChunkSize );
  iSizeIndicatorLength = strlen( sz15ChunkSizeInHex );
  // Now that we know the INDICATOR length (usually five or six bytes including the "\r\n"),
  //     move the already existing PAYLOAD a few bytes up in memory
  //     to "give way" for <iSizeIndicatorLength> bytes :
  memmove( pbDest+iSizeIndicatorLength/*DEST*/, pbDest/*SOURCE*/, iNettoChunkSize );
  // Insert the hexadecimal length indicator + "\r\n" into the free space,
  // WITHOUT the string's trailing zero (thus memmove or memcpy, not strcpy) :
  memmove( pbDest/*DEST*/, sz15ChunkSizeInHex/*SOURCE*/, iSizeIndicatorLength/*..w/o trailing zero-byte*/ );
  nTotalBytesToSend = iSizeIndicatorLength + iNettoChunkSize; // <- this is what we have so far, but it's not all yet..
  pbDest[ nTotalBytesToSend++ ] = '\r'; // APPEND the marker for the end of a chunk..
  pbDest[ nTotalBytesToSend++ ] = '\n';
  return  nTotalBytesToSend;
} // end HttpSrv_FinishTransferChunk()




//--------------------------------------------------------------------------

int HttpSrv_StartStreamingLiveAudio( T_HttpInstance *pHttpInst )
  // Called from HttpSrv_OnGET() after allocating a T_HttpInstance,
  // with the requested URL e.g. "/LiveAudio.ogg" .
  // Appends the proper HTTP response header with MIME type for an
  //   ENDLESS STREAM (not a simple, limited-size Ogg/Vorbis FILE),
  //   possibly already followed by a few initial "Ogg pages" in the
  //   BINARY response body. Returns HTTP_STATUS__OK when successful .
  // EXAMPLE (with a similar Ogg/Vorbis stream played via brower /
  //   linked <audio> element):
  // > GET /vlf/live-stream.php?stream=vlf1 HTTP/1.1  (request from client, decoded by Wireshark)
  // > Host: abelian.org
  // > User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64; rv:122.0) Gecko/20100101 Firefox/122.0
  // > Accept: audio/webm,audio/ogg,audio/wav,audio/*;q=0.9,application/ogg;q=0.7,video/*;q=0.6,*/*;q=0.5
  // > Accept-Language: de,en-US;q=0.7,en;q=0.3
  // > Referer: http://abelian.org/vlf/
  // > Range: bytes=0-
  // > DNT: 1
  // > Connection: keep-alive
  // > Sec-GPC: 1
  // > Accept-Encoding: identity
  // >
  // > HTTP/1.1 200 OK                      (response from the server at abelian.org)
  // > Date: Tue, 06 Feb 2024 19:58:15 GMT
  // > Server: Apache/2.4.57 (Debian)
  // > Cache-Control: no-store, no-cache, must-revalidate, max-age=0
  // > Expires: 0
  // > Pragma: no-cache
  // > Keep-Alive: timeout=5, max=100       (do we really need to limit the maximum number of requests here?)
  // > Connection: Keep-Alive
  // > Transfer-Encoding: chunked           (see explanation of "chunked" encoding below)
  // > Content-Type: audio/ogg              (note the ABSENCE of a "Content-Length")
  // >
  // > 2dbf    <- that's really ASCII, "2dbf\r\n" = length of the current chunk IN HEXADECIMAL FORMAT followed by CR+NL
  // > OggS.............        (that's already the BINARY stuff; begin of Ogg containers .. )
  //
  // For an example and details about "Transfer-Encoding: chunked",
  //     see HttpSrv_FinishTransferChunk() .
{
  // As with 'normal' received GET requests, pHttpInst->bTxBuffer is EMPTY
  // on entry, so we can use the entire capacity (C_HttpSrv_TX_BUFFER_SIZE)
  // for the RESPONSE HEADER :
  char *pszDest, *pszResponsePayload;
  const char *pszEndstop;
  int  iChunkSize;
# if( SWI_USE_VORBIS_STREAM )
  int  iOggVorbisSize = VorbisStream_GetLengthOfInitialHeadersFromEncoder( &pHttpInst->pCwNet->sVorbis );
# endif // SWI_USE_VORBIS_STREAM ?

# if( SWI_USE_VORBIS_STREAM )
  if( (iOggVorbisSize<=0) || (iOggVorbisSize>(C_HttpSrv_TX_BUFFER_SIZE-16)) )
   { // Ogg/Vorbis encoded audio not available, or the 'initial headers'
     //  are too large for the static 'transmit buffer' !
     return HTTP_STATUS__INSUFF_STORAGE;  // here: "too much; increase C_HttpSrv_TX_BUFFER_SIZE"
     // (For example, to send the "three initial Ogg pages" prepared by VorbisStream.c,
     //  iOggVorbisSize was 2658 bytes at this point.)
   }

  pHttpInst->pClient->fWantVorbis = TRUE; // flag for CwNet.c to provide an Ogg/Vorbis stream
  pHttpInst->iStreamType = C_HttpSrv_StreamType_VorbisAudio;
# else    // ! SWI_USE_VORBIS_STREAM -> what else to use for audio played via stupid WEB BROWSER ?
  return HTTP_STATUS__INSUFF_STORAGE;  // here: "too much; increase C_HttpSrv_TX_BUFFER_SIZE"
# endif   //   SWI_USE_VORBIS_STREAM ?

  pHttpInst->nTransferEncoding  = C_HttpSrv_XferEncoding_Chunked;

  HttpSrv_PrepareBuildingResponseBody( pHttpInst, &pszDest/*out*/, &pszEndstop/*out*/ );
  // ,--------------------------------------------'
  // '--> points into pHttpInst->bTxBuffer now
  pszResponsePayload = pszDest;  // <- kludge to inspect the result via debugger later..

# if( SWI_USE_VORBIS_STREAM )
  SL_AppendPrintf( &pszDest, pszEndstop,
     "%X\r\n", // <- hexadecimal 'chunk size' for the FIRST "chunk", with the "initial Ogg headers"
     (unsigned int)iOggVorbisSize );
  VorbisStream_GetInitialHeadersFromEncoder(
     &pHttpInst->pCwNet->sVorbis,
     (BYTE*)pszDest, (int)(pszEndstop-pszDest), // [out] network buffer with limited length
     &pHttpInst->pClient->dwNextVorbisPage); // [out] indicator for the NEXT CALL
                                             // of VorbisStream_GetNextPagesFromEncoder()
  pszDest += iOggVorbisSize;
  SL_AppendString( &pszDest, pszEndstop, "\r\n" ); // read the text quoted from Mozilla.org again..
  // there's a CARRIAGE RETURN + NEW LINE after "the chunk itself"  !
  HttpSrv_FinishBuildingResponseBody( pHttpInst, pszDest/*in*/ );
     // 2024-02-10 : Tried to "see" these 'initial headers from the Vorbis encoder'
     // in Firefox' "Network Analysis" (under 'Extras' aka 'Tools'
     //    .. 'Browser Tools' / 'Web Developer Tools' or whatever they call it now):
     //     .. 'Network' .. (start the audio player, then PAUSE the network display
     //                      as soon as you see the "GET LiveAudio.ogg" somewhere)
     //      .. select the line with "GET <Domain> LiveAudio.gg .
     //       .. check the REQUEST headers (those under "GET http://blah/LiveAudio.ogg")
     //       .. check the RESPONSE HEADERS (those with "Content-Type: audio/ogg")
     //       .. check the RESPONSE ITSELF (it's on an extra tab, further right) .
     //  During the first test, the "Size" of the first chunk(?) was indicated
     //  as "2.66 kB" in Firefox, but the "Response" itself was only displayed
     //  as a single, loong line of ASCII text under "Response payload"
     //               (no way to switch the damned thing to HEXADECIMAL) :
     //  > "T2dnUwACAAAAAAAAAAAAAAAAAAAAAC7NDBgBHgF2b3Jia".... WHAT THE HECK IS THIS ?
     //  Theoretically, at THIS point, pszResponsePayload[0..pHttpInst->iTxBufferLength-1]
     //  should show something similar (plus the funny hexadecimal chunk size followed by "\r\n"),
     //  but it didn't. AS EXPECTED, pszResponsePayload was e.g. "0a62\r\nOggS" .
     //  Tried the equivalent in Chrome, but Chrome said (on the "Response" tab):
     //  > "Failed to load response data: No data found for reponse with given identifier"
     //     .. whatever that means .. WHAT identifier ? ?


  HttpSrv_PrepareBuildingResponseHeaders( pHttpInst, &pszDest/*out*/, &pszEndstop/*out*/ );
  // ,------------------------------------------------'
  // '--> points into pHttpInst->szResponseHeader, as for any GET or POST response-"headers"
  SL_AppendString( &pszDest, pszEndstop, // here: "GET" response for AUDIO with "chunked encoding"
     "HTTP/1.1 200 OK\r\n" // protocol ver 1.1 REQUIRES A "Content-Length" field !
     "Content-Type: audio/ogg\r\n"
     "Server: Remote CW Keyer\r\n"
     "Cache-Control: no-store, no-cache, must-revalidate, max-age=0\r\n"
     "Expires: 0\r\n"
     "Pragma: no-cache\r\n"
     "Keep-Alive: timeout=5, max=100\r\n" // timeout in seconds (do we really need the "max" here ?)
     "Connection: Keep-Alive\r\n"
     "Transfer-Encoding: chunked\r\n"
     // again: Note the ABSENCE of a "Content-Length".. it's UNLIMITED here !
     "\r\n" ); // <- empty line indicates the end of the HTTP-header
# endif   //   SWI_USE_VORBIS_STREAM ?

  HttpSrv_FinishBuildingResponseHeaders( pHttpInst, pszDest );

  pHttpInst->nServerState = C_HttpSrvState_SENDING_HEADER;  // .. here: and the FIRST PART of the "body"

  (void)pszResponsePayload; // "assigned a value that is never used"
    //  .. well, maybe, but better than using an uninitialized pointer.

  HERE_I_AM__HTTPSRV();
  return HTTP_STATUS__OK;

  // To examine ONLY the Ogg/Vorbis encoded audio stream with Wireshark,
  //   use this filter:  _ws.col.info == "GET /LiveAudio.ogg HTTP/1.1 "
  //   then "follow stream". If the "Transfer-Encoding: chunked" works as planned,
  //   the next call of HttpSrv_OnPoll() will send the above HTTP-REPONSE-HEADER,
  //   immediately followed by the three "initial Ogg pages" (as BODY in the first CHUNK).
  // pHttpInst->nServerState will then transit to C_HttpSrvState_ASSEMBLE_NEXT_CHUNK,
  //


} // end HttpSrv_StartStreamingLiveAudio()



//--------------------------------------------------------------------------

void HttpSrv_ContinueStreamingLiveAudio( T_HttpInstance *pHttpInst )
   // Called from HttpSrv_OnPoll() whenever pHttpInst->bTxBuffer is EMPTY,
   //        the HTTP "transfer encoding type" is "Chunked",
   //        and pHttpInst->iStreamType is C_HttpSrv_StreamType_VorbisAudio .
{
  char *pszDest;

  const char *pszEndstop;  // the char to which this pointer points shall NEVER be modified.. thus "const"

  int  iChunkSize, iMaxChunkSize, nTotalBytesToSend;
  char sz15ChunkSizeInHex[16];


  HttpSrv_PrepareBuildingResponseBody( pHttpInst, &pszDest/*out*/, &pszEndstop/*out*/ );

  // ,--------------------------------------------'
  // '--> points into pHttpInst->bTxBuffer now .
  iMaxChunkSize = pszEndstop - pszDest;

# if( SWI_USE_VORBIS_STREAM )
  // Keep it simple : Ask for as many bytes as fit in our buffer from the Vorbis encoder,
  //  but leave a few empty bytes for the later HEXADECIMAL ASCII CHUNK SIZE indicator:
  iChunkSize = VorbisStream_GetNextPagesFromEncoder( &pHttpInst->pCwNet->sVorbis,
        (BYTE*)pszDest, iMaxChunkSize-8/*iMaxDestLength : reserve 8 bytes for the "chunk overhead"*/,
        &pHttpInst->pClient->dwNextVorbisPage); // [in,out] Ogg/Vorbis PAGE INDEX for THIS and the NEXT call
  if( iChunkSize > 0 )  // got at least one Ogg page to send now ? Send it..
   { nTotalBytesToSend = HttpSrv_FinishTransferChunk( (BYTE*)pszDest, iChunkSize );
     HttpSrv_FinishBuildingResponseBody( pHttpInst, pszDest+nTotalBytesToSend );
     // '--> here: not the "entire body", but just the next CHUNK including the "chunk" overhead
     pHttpInst->nServerState = C_HttpSrvState_SENDING_DATA;  // ..ASSEMBLE_NEXT_CHUNK -> .._SENDING_DATA (endlessly..)
     // 2024-02-14 : Streaming audio from the Remote CW Keyer into e.g. Firefox V122
     //   kind of worked, but with (a) a long initial delay until audio began to play;
     //                            (b) with an EVER-GROWING AUDIO LATENCY ... wtf ?!
   }
# endif //   SWI_USE_VORBIS_STREAM ?

} // end HttpSrv_ContinueStreamingLiveAudio()

#endif // SWI_USE_HTTP_SERVER ?

/* EOF < Remote_CW_Keyer/HttpServer.c > */






