// File:  C:\cbproj\Remote_CW_Keyer\HamlibServer.h
// Date:  2024-01-01
// Author:  Wolfgang Buescher (DL4YHF)
// Purpose: Socket-based, "Hamlib NET rigctrl"-compatible Server for the 
//          'Remote CW Keyer'. Listens on a TCP/IP port (typically 4532)
//          to accept multiple remote clients that would otherwise try
//          to communicate directly with the radio via [virtual] COM port.
//          As usual, details only in the implementation (HamlibServer.c).

#ifndef  _HAMLIB_SERVER_H  // prevent multiple header inclusion ..
# define _HAMLIB_SERVER_H  //  .. and let other modules know we're been included

#include "switches.h" // <- all project specific 'compilation switches'

#include "Timers.h"     // <- T_TIM_Stopwatch, T_TIM_Stopwatch(), etc...

#ifndef _INET_TOOLS_H
# include "Inet_Tools.h"
#endif

#include "RigControl.h" // .. delivers info ABOUT the radio, waterfall data FROM the radio, etc..
          // (so far, there's exactly ONE "rig control" per "CW Network Server",
          //  thus a T_RigCtrlInstance-reference is a member of our T_HLSrv struct)


//---------------------------------------------------------------------------
// Constants
//---------------------------------------------------------------------------
#ifndef  WSAE_NO_ERROR // our own addition to the (windows-)Socket Error Codes..
# define WSAE_NO_ERROR 0  // ... used by a few wrapper functions, not Winsock itself
#endif

#define HLSRV_MAX_CLIENTS 3

// Single-byte binary command codes used in the TCP streams between client and server.
// To reduce the protocol overhead, the presence of a BLOCK LENGTH, immediately
// in one or two byte AFTER the "CMD_CMD_xyz" is indicated by BITS SEVEN and SIX
// in the command:
#define HLSRV_CMD_MASK_BLOCKLEN    0xC0 // Mask for bitwise AND-ing a received
          // single-byte command, for checking the two-bit combinations below:
#define HLSRV_CMD_MASK_NO_BLOCK    0x00 // "no BLOCK LENGTH INDICATOR after the command byte"
#define HLSRV_CMD_MASK_SHORT_BLOCK 0x40 // "short block" (ONE-BYTE length, i.e. max 255 byte payload)
#define HLSRV_CMD_MASK_LONG_BLOCK  0x80 // "long block" (TWO-BYTE length, i.e. max 65535 byte payload)
#define HLSRV_CMD_MASK_RESERVE     0xC0 // reserved for future use / special command 0xFF
// This allows encoding / embedding multiple commands, morse code fragments,
// audio sample blocks, Icom's CI-V messages, and who-knows-what-else in a
// single TCP segment. On this occasion: The receiving side need a STREAMING
// PARSER, because the commands listed below don't necessary arrive in a
// single 'block' returned by the Socket API ("recv()") !
#define HLSRV_CMD_MASK_COMMAND     0x3F // mask to strip 'the received command itself'
#define HLSRV_CMD_NONE     0x00 // dummy command to send HTTP instead of our binary protocol
#define HLSRV_CMD_CONNECT  0x01 // "Connect" (first command from client to server)
                                //  or "Connected" (first response from server to client)
#define HLSRV_CMD_DISCONN  0x02 // "Disconnect" (at any time, in any direction)
#define HLSRV_CMD_PING     0x03 // "Ping Test" (to check the connection's speed)
#define HLSRV_CMD_PRINT    0x04 // "Text that the receiver should print somewhere",
                                //   for example the 'Welcome'-text, or even 'chat'
#define HLSRV_CMD_TX_INFO  0x05 // Announcement to all connected users about
                                //   'how has the key now' (or the microphone).
                                // In the experimental "Remove CW Keyer" GUI,
                                // this was displayed on the 'Network' tab.
#define HLSRV_CMD_MORSE    0x10 // Hand-keyed or machine-generated MORSE CODE
#define HLSRV_CMD_AUDIO    0x11 // Audio samples, 8-bit A-Law compressed, multiplexed into the TCP stream
#define HLSRV_CMD_VORBIS   0x12 // future plan: Ogg/Vorbis compressed audio stream, optional, not suited for simple microcontrollers !
#define HLSRV_CMD_RSV13    0x13 // reserved for some other audio stream format
#define HLSRV_CMD_CI_V     0x14 // A single CI-V packet (passed to/from Icom transceivers)
#define HLSRV_CMD_SPECTRUM 0x15 // A spectrum for waterfall/spectrum display, incl. T_HLSrv_SpectrumHeader
#define HLSRV_CMD_FREQ_REPORT 0x16 // Transceiver's frequency, mode, etc (T_RigCtrl_VfoReport)
#define HLSRV_CMD_PARAM_INTEGER 0x18 // a single parameter, data type 'integer', identified by RIGCTRL_PN_xyz, in a  T_RigCtrl_ParamReport_Integer
#define HLSRV_CMD_PARAM_DOUBLE  0x19 // a single parameter, data type 'double', identified by RIGCTRL_PN_xyz, in a  T_RigCtrl_ParamReport_Double
#define HLSRV_CMD_PARAM_STRING  0x1A // a single parameter, data type 'string', identified by RIGCTRL_PN_xyz, in a  T_RigCtrl_ParamReport_String
#define HLSRV_CMD_QUERY_PARAM   0x1B // request to send a certain parameter, identified by RIGCTRL_PN_xyz
#define HLSRV_CMD_RESERVE  0xFF // "special command", reserved for future use

// Fixed payload sizes for SOME of the above "commands" :
#define HLSRV_PAYLOAD_SIZE_PING 16  // REQ/RESP, ID, two alignment dummy bytes, and THREE 32-bit timestamps
                                    // as explained in CwNet.c : HLSRV_ExecuteCmd() / case HLSRV_CMD_PING .

// Fixed audio sample rate used in all NETWORK AUDIO STREAMS :
#define HLSRV_STREAM_SAMPLING_RATE 8000  // 8 kSamples/second is sufficient for CW,USB,LSB (and even NBFM)


// Integer values returned by 'command handlers', invoked via function pointers,
// telling the caller (in HamlibServer.c) "what to send in response" :
#ifndef HAMLIB_RESULT_NO_ERROR
# include "HamlibResultCodes.h" // Integer values returned by 'rigctld'-compatible command handlers,
                                // NEGATIVE values indicate various ERRORS,  ZERO means "No Error".
#endif // ndef HAMLIB_RESULT_NO_ERROR ?


//---------------------------------------------------------------------------
// Data Types
//---------------------------------------------------------------------------



typedef struct t_HLSrvClient // struct for an ARRAY OF CLIENTS, managed by our SERVER;
{                            // only ONE of these is used for our LOCAL CLIENT.
  int iClientState;  // HLSRV_CLIENT_STATE_DISCONN, etc:
#     define HLSRV_CLIENT_STATE_DISCONN    0
#     define HLSRV_CLIENT_STATE_ACCEPTED   1  // for the SERVER SIDE: socket "accepted" a REMOTE client, but not successfully "logged in" yet
#     define HLSRV_CLIENT_STATE_CONNECTED  1  // same value, similar meaning, but for the LOCAL CLIENT side
#     define HLSRV_CLIENT_STATE_CONFIRMED  2  // "this client really talks the EXPECTED protocol, no watchdog/timeout monitoring for more requests from him"
  union
   { BYTE b[4];        // "his" (the remote peer's) IPv4 address ...
     DWORD dw;         // .. also accessable as a 32-bit double word for easy comparison
   } b4HisIP;
  int  iHisPort;       // <- for the CLIENT SIDE, the remote server's "published" port number;
                       //    for the SERVER SIDE, the remote client's "ephemeral" port number.
  DWORD dwNumBytesSent, dwNumBytesRcvd; // .. total, since THIS remote client was 'accepted' by our server
  DWORD dwNumSegmentsRcvd; // .. to avoid flooding the 'debug log' ..
  BOOL  fDisconnect;   // flag polled in ServerThread() to disconnect a remote client,
                       // may be set by any 'command handler' invoked from the
                       // streaming parser, when detecting an ill-behaving client.
  T_TIM_Stopwatch swActivityWatchdog;  // watchdog for this connection's activity
#     define HLSRV_ACTIVITY_TIMEOUT_MS 5000 // max nr. of milliseconds w/o activity
                       // (also used for HTTP; frees a T_HLSrvClient
                       //  AND the associated T_HttpInstance after a few seconds
                       //  of "no activity" )
  double dblUnixTimeOfStart; // <- set immediately after accepting, for bandwidth-calculations etc

  // Buffer for a single received line (for Hamlib NET, always ending with '\n' )
  int iCurrentCommand;   // <- important for Hamlib's "Extended Response Protocol",
                         //     with multiple lines, until ..
          // > The last record of each block is the string "RPRT x\n" where x
          // > is the numeric return value of the Hamlib backend function
          // > that was called by the command.
# define HLSRV_RCVD_LINE_MAX_LENGTH 2048 // <- long enough for spectra, audio sample blocks, etc ..
  BYTE bRcvdLine[HLSRV_RCVD_LINE_MAX_LENGTH];
  int iRcvdLineLength;

# define HLSRV_RESPONSE_MAX_LENGTH 16384 // <- long enough for spectra, audio sample blocks,
          // the bizarre (and poorly documented) "dump_state"-thingy, etc ..
  BYTE bResponse[HLSRV_RESPONSE_MAX_LENGTH];
  int iResponseLength; // <- only required if the "protocol" isn't purely ASCII-based.
      // For the original Hamlib protocol, bResponse[] contains
      // a C string terminated with a zero-byte by any command-handler function,
      // so the number of bytes to send can be determined by strlen() .


} T_HLSrvClient; // -> T_HLSrv.Client[HLSRV_MAX_CLIENTS+1] .

typedef struct t_HLSRV_FreqRange
{
  double dblStartFreq_Hz; // Start frequency in Hertz
  double dblEndFreq_Hz;   // End frequency in Hertz
  DWORD dwModes; // the lower 32 bits of Hamlib's SIXTYFOUR-BIT-field of "rig modes"
                 //  (one bit per basic mode, H=supported, L=not supported) :
# define HLSRV_MODE_NONE    0x00000000 // '' -- None */
# define HLSRV_MODE_AM      0x00000001 // AM -- Amplitude Modulation
# define HLSRV_MODE_CW      0x00000002 // CW -- CW "normal" sideband
# define HLSRV_MODE_USB     0x00000004 // USB -- Upper Side Band
# define HLSRV_MODE_LSB     0x00000008 // LSB -- Lower Side Band
# define HLSRV_MODE_RTTY    0x00000010 // RTTY -- Radio Teletype
# define HLSRV_MODE_FM      0x00000020 // FM -- "narrow" band FM */
# define HLSRV_MODE_WFM     0x00000040 // WFM -- broadcast wide FM */
# define HLSRV_MODE_CWR     0x00000080 // CWR -- CW "reverse" sideband */
# define HLSRV_MODE_RTTYR   0x00000100 // RTTYR -- RTTY "reverse" sideband (who the heck needs THIS ? ?)
# define HLSRV_MODE_AMS     0x00000200 // AMS -- Amplitude Modulation Synchronous
# define HLSRV_MODE_PKTLSB  0x00000400 // PKTLSB -- Packet/Digital LSB mode (dedicated port)
# define HLSRV_MODE_PKTUSB  0x00000800 // PKTUSB -- Packet/Digital USB mode (dedicated port)
# define HLSRV_MODE_PKTFM   0x00001000 // PKTFM -- Packet/Digital FM mode (dedicated port)
# define HLSRV_MODE_ECSSUSB 0x00002000 // ECSSUSB -- Exalted Carrier Single Sideband USB
# define HLSRV_MODE_ECSSLSB 0x00004000 // ECSSLSB -- Exalted Carrier Single Sideband LSB
# define HLSRV_MODE_FAX     0x00008000 // FAX -- Facsimile Mode (omg.. "FM fax" or "SSB fax" ?)
# define HLSRV_MODE_SAM     0x00010000 // SAM -- Synchronous AM double sideband
# define HLSRV_MODE_SAL     0x00020000 // SAL -- Synchronous AM lower sideband
# define HLSRV_MODE_SAH     0x00040000 // SAH -- Synchronous AM upper (higher) sideband
# define HLSRV_MODE_DSB     0x00080000 // DSB -- Double sideband suppressed carrier
# define HLSRV_MODE_BIT20   0x00100000 // Note: Someone has killed bit 20 without documenting it
# define HLSRV_MODE_FMN     0x00200000 // FMN   -- FM Narrow Kenwood ts990s
# define HLSRV_MODE_PKTAM   0x00400000 // PKTAM -- Packet/Digital AM mode e.g. IC7300
# define HLSRV_MODE_P25     0x00800000 // P25   -- APCO/P25 VHF,UHF digital mode IC-R8600
# define HLSRV_MODE_DSTAR   0x01000000 // D-Star -- VHF,UHF digital mode IC-R8600
# define HLSRV_MODE_DPMR    0x02000000 // dPMR  -- digital PMR, VHF,UHF digital mode IC-R8600
# define HLSRV_MODE_NXDNVN  0x04000000 // NXDN-VN -- VHF,UHF digital mode IC-R8600
# define HLSRV_MODE_NXDN_N  0x08000000 // NXDN-N  -- VHF,UHF digital mode IC-R8600
# define HLSRV_MODE_DCR     0x10000000 // DCR  -- VHF,UHF digital mode IC-R8600
# define HLSRV_MODE_AMN     0x20000000 // AM-N -- Narrow band AM mode IC-R30
# define HLSRV_MODE_PSK     0x40000000 // PSK  -- Kenwood PSK and others
# define HLSRV_MODE_PSKR    0x80000000 // (bit 31) PSKR -- Kenwood PSKR and others
# define HLSRV_MODE_ALL     0xFFFFFFFF // "any mode" constant for dwModes
  DWORD dwExtModes; // only the rarely used upper 32 bits of Hamlib's SIXTYFOUR-BIT-field of "rig modes"
                    //  (one bit per exotic mode, H=supported, L=not supported)
# define HLSRV_EMODE_NONE   0x00000000 // None of the "esoteric" / "extended" modes in dwExtModes
# define HLSRV_EMODE_DD     0x00000001 // (formerly bit 32) DD Mode IC-9700
# define HLSRV_EMODE_C4FM   0x00000002 // Yaesu C4FM mode
# define HLSRV_EMODE_PKTFMN 0x00000004 // Yaesu DATA-FM-N
# define HLSRV_EMODE_SPEC   0x00000008 // Unfiltered as in PowerSDR
# define HLSRV_EMODE_CWN    0x00000010 // CWN -- Narrow band CW (FT-736R)
# define HLSRV_EMODE_IQ     0x00000020 // IQ mode for a couple of kit rigs
# define HLSRV_EMODE_ISBUSB 0x00000040 // ISB mode monitoring USB
# define HLSRV_EMODE_ISBLSB 0x00000080 // ISB mode monitoring LSB
# define HLSRV_EMODE_ALL    0xFFFFFFFF // "any mode" constant for dwExtModes

  long i32LowPower_mW;  // Lower RF power in mW, -1 for no power (ie. rx list)
  long i32HighPower_mW; // Higher RF power in mW, -1 for no power (ie. rx list)
  DWORD dwVFO; // VFO list equipped with this range. Possible values: HLSRV_VFO_A, HLSRV_VFO_B, HLSRV_VFO_CURR:
# define HLSRV_VFO_NONE 0 // vfo unknown
# define HLSRV_VFO_A       (1<<0)
# define HLSRV_VFO_B       (1<<1)
# define HLSRV_VFO_SUB_A   (1<<21) // alias for SUB_A
# define HLSRV_VFO_SUB_B   (1<<22) // alias for SUB_B
# define HLSRV_VFO_MAIN_A  (1<<23)
# define HLSRV_VFO_MAIN_B  (1<<24)
# define HLSRV_VFO_OTHER   (1<<5)
# define HLSRV_VFO_SUB     (1<<25)
# define HLSRV_VFO_MAIN    (1<<26) // bizarre, isn't it ?
# define HLSRV_VFO_VFO     (1<<27)
# define HLSRV_VFO_MEM     (1<<28) "VFO MEM" means Memory mode, to be used with set_vfo
# define HLSRV_VFO_CURRENT (1<<29) // current "tunable channel"/VFO
# define HLSRV_VFO_TX_FLAG (1<<30) // Flag to set if VFO can transmit
# define HLSRV_VFO_ALL    (1<<31) //  Flag to set all VFOS

  DWORD dwAnt; // Antenna list (bitwise combination) for this range: 0 = "none" = all (?!),
               // or HLSRV_ANT_CURR means dedicated to certain bands
               //   and automatically switches, no set_ant command .
# define HLSRV_ANT_NONE 0
# define HLSRV_ANT_1 0x0001
# define HLSRV_ANT_2 0x0002
# define HLSRV_ANT_3 0x0004
# define HLSRV_ANT_4 0x0008
# define HLSRV_ANT_UNKNOWN 0x40000000
# define HLSRV_ANT_CURRENT 0x80000000 // using the "current antenna", whatever that is


  char *label; // "Label for this range that explains why.  e.g. Icom rigs USA, EUR, ITR, TPE, KOR"
} T_HLSRV_FreqRange; // -> T_HLSrv.RxRangeList[], T_HLSrv.TxRangeList[] .


typedef struct t_HLSrv // data for a multi-client 'Hamlib Server' instance:
{

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // "Configuration parameters" (must be filled out be the application
  //          before start, usually from 'visual components' in the GUI
  //          with similar names as the struct members used here)
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  struct { // .cfg = "everything that needs to be SAVED between sessions" (and included in a hash value):
     char sz80ClientRemoteIP[84]; // remote server IP:PORT (or URL:PORT) for the client
     char sz80ClientUserName[84]; // user name for the "log-in" (into a remote client)
     char sz80ClientCallsign[84]; // user's modifiable callsign (e.g. "DL4YHF", "DL4YHF/p", "DL4YHF/r", "DL4YHF testing")
     int  iServerListeningPort;   // server's listening port (to listen for
                                  // incomining connection requests, similar
                                  // as a multi-client HTTP server .. but simpler)
#      define HLSRV_DEFAULT_SERVER_PORT 4532 // <- if your office internet router blocks this, try to abuse port 80 
     char sz255AcceptUsers[256];  // comma-separated list of user names that the
                                  // server shall 'accept', and their permissions
     char sz20ServerAdminPWD[24]; // HTTP server administrator password,
                                  // allows reading 'server log', 'error log', etc,
                                  // which may contain user's IP addresses and similar
                                  // info that must not leak out to normal visitors.
     int iServerOptions; // bitwise combination of the HLSRV_OPTION_.. flags below:
#      define HLSRV_OPTIONS_READ_ONLY 0x0000 /* "only allow READING parameters, but don't let 3rd party applications MODIFY anything" */
#      define HLSRV_OPTIONS_ALLOW_QSY 0x0001 /* allow 3rd party applications to CHANGE THE FREQUENCY */
#      define HLSRV_OPTIONS_ALLOW_SET_MODE 0x0002 /* allow 3rd party applications to CHANGE THE MODE */
#      define HLSRV_OPTIONS_ALLOW_ALL 0x00FF

     int iDiagnosticFlags; // bitwise combination of the HLSRV_DIAG_FLAGS_.. below:
#      define HLSRV_DIAG_FLAGS_NONE                 0x0000 /* "dont show anything of the stuff listed below" ..     */
#      define HLSRV_DIAG_FLAGS_SHOW_NETWORK_TRAFFIC 0x0001 /* show 'network traffic' (HEX or ASCII) on the 'Debug' tab*/
#      define HLSRV_DIAG_FLAGS_VERBOSE_OUTPUT       0x0002 /* produce verbose output via ShowError() (awful, only for debugging) */
       // Note: Module RigControl.c has it's own "logging control flags",
       // see RigControl.c: RigCtrl_TrafficMonitor.iDisplayOptions.RIGCTRL_TMON_DISPLAY_OPTION_ENABLED, etc .
   } cfg; // <- end of the part that needs to be stored permanently (between sessions)


  char  *pszLastError;  // NULL when there's nothing to report (in the error log)
  char  sz255LastError[256]; // buffer for an "sprintf'ed" error message

  HANDLE hThread; // also a HANDLE, but for a THREAD, not a file,
                  // thus "NULL" means invalid (not INVALID_HANDLE_VALUE). Bleah.
  DWORD  dwThreadId; // a (windows-) thread-ID, possibly required to send events to our thread.
  int iThreadStatus; // what's our client- or server thread doing ? one of these:
#   define HLSRV_THREAD_STATUS_NOT_CREATED 0
#   define HLSRV_THREAD_STATUS_LAUNCHED    1
#   define HLSRV_THREAD_STATUS_RUNNING     2 // <- only set ONCE(!), immediately when entering the thread function
#   define HLSRV_THREAD_STATUS_TERMINATE   3 // <- just a KLUDGE to "request POLITE termination", but takes a few dozen milliseconds to actually TERMINATE !
#   define HLSRV_THREAD_STATUS_TERMINATED -1 // <- set by the worker thread if it TERMINATED ITSELF

  DWORD dwThreadLoops;       // number of thread loops, counted in ServerThread()
  DWORD dwThreadErrors;      // number of errors (mostly TIMING ERRORS) counted the above thread
  DWORD dw8ThreadIntervals_us[8]; // number of MICROSECONDS per loop in ServerThread()
  DWORD dwNumBytesSent, dwNumBytesRcvd;  // .. total, since the last HLSRV_Start()


#   define HLSRV_SOCKET_RX_BUFFER_SIZE 16384 // size of our receive buffer; ex: 1024 bytes
  BYTE  bRxBuffer[HLSRV_SOCKET_RX_BUFFER_SIZE];
  int   nBytesInRxBuffer, iPeakRxBufferUsage;
#   define HLSRV_SOCKET_TX_BUFFER_SIZE 16384 // size of our transmit buffer; ex: 1024 bytes
  BYTE  bTxBuffer[HLSRV_SOCKET_TX_BUFFER_SIZE];
  int   nBytesInTxBuffer, iPeakTxBufferUsage;

  T_HLSrvClient Client[HLSRV_MAX_CLIENTS+1];  // <- an ARRAY OF REMOTE CLIENTS, managed by our SERVER

  T_InetBlacklist sBlacklist; // automatically filled blacklist of 'bad guys'

  int nRemoteClientsConnected; // set in ServerThread(); read-only for anyone else
  int iTransmittingClient;     // index into Client[*] if a particular client
     // currently 'has the key' to TRANSMIT, along with HLSRV_PERMISSION_TRANSMIT.
     // Only THIS client may fill T_HLSrv.MorseRxFifo in HLSRV_OnReceive() !
     // (*) iTransmittingClient = 0 = HLSRV_LOCAL_CLIENT_INDEX means
     //                               THE SERVER'S OPERATOR "has the key".
     //     iTransmittingClient = HLSRV_FIRST_REMOTE_CLIENT_INDEX .. HLSRV_MAX_CLIENTS
     //                               means a REMOTE CLIENT currently "has the key".
     //     iTransmittingClient < 0 means "nobody has the key", and "anyone with
     //                               HLSRV_PERMISSION_TRANSMIT may take it" .
  T_TIM_Stopwatch swTransmittingClient; // .. to detect when the 'transmitting client'
                               // drops the key, allowing others to take it.


  T_RigCtrlInstance *pRigControl; // so far, ONE "RigControl" instance will be tied
                                  //      to ONE "Hamlib Server" instance.

# define HLSRV_NUM_RX_RANGES 16 // one entry per ham radio band, at least from LF to UHF ..
# define HLSRV_NUM_TX_RANGES 16 // same for frequencies the radio can TRANSMIT on .
    // Note: In the original Hamlib rig.h, #define HAMLIB_FRQRANGESIZ 30
    //       applies to both the "RX" and the "TX" ranges.
    //       For the REMOTE CW KEYER, we will possibly load the following
    //       arrays from a user-editable configuration file, so that the
    //       "Hamlib NET rigctld"-compatible server will only allow tuning
    //       or transmitting in what the SYSOP (or the station owner's license)
    //       allows (instead of what some ITU-region-X- bandplan allows) !
  T_HLSRV_FreqRange RxRangeList[HLSRV_NUM_RX_RANGES]; // reported in the "dump_state"-response..
  T_HLSRV_FreqRange TxRangeList[HLSRV_NUM_TX_RANGES]; // see HLSRV_OnDumpState()


} T_HLSrv;


//---------------------------------------------------------------------------
// Global variables  (?)
//---------------------------------------------------------------------------

  // (none so far. A single T_HLSrv instance exists in Keyer_Main.cpp)


//---------------------------------------------------------------------------
// Function Prototypes
//---------------------------------------------------------------------------

#ifndef CPROT   // for peaceful co-existence of C and C++ ...
#ifdef __cplusplus
 #define CPROT extern "C"
#else
 #define CPROT
#endif  // not "cplusplus" ?
#endif // ndef CPROT ?


CPROT void  HLSRV_InitInstanceWithDefaults( T_HLSrv *pServer );
CPROT void  HLSRV_InitFreqRange( T_HLSRV_FreqRange *pFreqRange, double dblStartFreq_Hz, double dblEndFreq_Hz);

CPROT int   HLSRV_EnumerateRequiredPNs( int iZeroBasedIndex ); // <- used by RigControl.c to poll all that Hamlib(!) may need
CPROT void  HLSRV_LinkToRigControl( T_HLSrv *pServer, T_RigCtrlInstance *pRigControl );
CPROT BOOL  HLSRV_Start( T_HLSrv *pServer );
CPROT void  HLSRV_Stop( T_HLSrv *pServer );
CPROT char* HLSRV_GetCurrentStatusAsString( T_HLSrv *pServer );
CPROT char* HLSRV_ClientStateToString( int iClientState );
CPROT char* HLSRV_GetLastErrorAsString( T_HLSrv *pServer );
CPROT char* HLSRV_IPv4AddressToString( BYTE *pbIP ); // -> "dotted decimal IP address"
CPROT int   HLSRV_CheckUserAndGetPermissions( T_HLSrv *pServer, char *pszUserName, char *pszCallsign );
CPROT int   HLSRV_GetUserPermissionsByRecentIP(T_HLSrv *pServer, BYTE *pbIPv4Address );
CPROT void  HLSRV_RefreshUserPermissionsByIP(T_HLSrv *pServer,BYTE *pbIPv4Address,int iPermissions);

CPROT int   HLSRV_GetFreeSpaceInTxBuffer( T_HLSrv *pServer );
CPROT BYTE* HLSRV_AllocBytesInTxBuffer( T_HLSrv *pServer, int iPayloadSize);

#endif // ndef _HAMLIB_SERVER_H



