// File:  C:\cbproj\Remote_CW_Keyer\HamlibServer.c
// Date:  2024-01-01
// Author:  Wolfgang Buescher (DL4YHF)
// Purpose: Socket-based, "Hamlib NET rigctrl"-compatible Server for the
//          'Remote CW Keyer'. Listens on a TCP/IP port (typically 4532)
//          to accept multiple remote clients that would otherwise try
//          to communicate directly with the radio via [virtual] COM port.
//          A similar functionality (Hamlib-compatible SERVER) is also
//          implemented in wfview, configurable there under
//          'Settings' .. 'External Control' .. [v] Enable RigCtld
// Module prefix:  "HLSRV" = HamLib-compatible Server .
//
//  The SERVER part  [ServerThread()] is loosely based on two articles about
//      "Asynchronous socket programming", aka "select()-based event handling",
//      locally saved as "Socket-based async multi-client tcp server.txt" .
//  Most of the application-specific processing happens in subroutines called
//      from the server's single thread, like:
//  * HLSRV_OnReceive()  : Called whenever data arrived on a socket.
//                             Implements a simple 'streaming parser'
//                             to demultiplex the received bytestream.
//  * HLSRV_ExecuteCmd() : Called from HLSRV_OnReceive() when a
//                             received command is "ready for processing".
//  * HLSRV_OnPoll()     : Called every ~~20 ms for any open socket,
//                             even if there was NOTHING received, to allow
//                             assembling all that the client or server needs
//                             to send in a SINGLE call to the send() function.
//                             Also performs some housekeeping, e.g. handing over
//                             'the key' (or maybe the microphone) from one
//                             user to the other.
//  * HLSRV_OnConnect(), HLSRV_OnDisconnect(), etc: Speak for themselves.
//
// Relevant info or a detailed(!) specification of the Hamlib commands
// and responses was difficult to find (especially about "dump_state"), so in some
// cases, it was necessary to study the original Hamlib sources. Despite that,
// THIS implementation is written from scratch, and NOT based on Hamlib itself.
//  For example, www.mankier.com/1/rigctl explained "dump_state" as follows:
//               '--> saved as Hamlib_rigctl_commands__Man_Pages_ManKier.txt
//   > dump_state
//   >    Return certain state information about the radio backend .
//  Hmm. A link to the details would have been incredibly helpful,
//       but doesn't exist.
//
#include "switches.h" // project specific 'compilation switches'

#include <string.h> // not only string functions in here, but also memset()
#include <stdio.h>  // no standard I/O but string functions like sprintf()
#ifdef _WINDOWS     // obviously compiling for a PC, so ..
# include "winsock2.h"  // .. use Microsoft's flavour of the Berkeley Socket API
  // Note: never include winsock2.h *after* windows.h .. you'll fry in hell,
  //       because "windows.h" #includes "winsock.h", not "winsock2.h" .
#endif

#include "StringLib.h" // DL4YHF's string library, with stuff like SL_ParseIPv4Address()
#include "Utilities.h" // stuff like UTL_iWindowsVersion, UTL_iAppInstance, ShowError(), etc
#include "Inet_Tools.h" // stuff like INET_DumpTraffic_HexOrASCII(), etc
#include "Timers.h"    // high-resolution 'current time'-function, T_TIM_Stopwatch, etc.
#include "CwNet.h"     // some TCP/IP related functions are implemented in CwNet.c
#include "HamlibServer.h"  // header for THIS module ("Hamlib Net rigctld"-compatible TCP server)
                           //  ,------------------------------|_|
                           //  '--> some write this in upper case, but it's simply "Network".

//----------------------------------------------------------------------------
// "Internal" constants, macros, lookup tables, and similar
//----------------------------------------------------------------------------

#define HLSRV_SOCKET_POLLING_INTERVAL_MS 20 // interval, in milliseconds,
                       // at which all of the active client <-> server sockets
                       // are PERIODICALLY POLLED for transmission, even if
                       // nothing has been received for a while.

#ifndef  sizeof_member
# define sizeof_member(type,member) sizeof(((type*)0)->member) /* ugly.. but works */
#endif

// Function pointer type used in HLSRV_Commands[] :
typedef int(*HLSrvCommandHandler)(T_HLSrv *pServer, T_HLSrvClient *pClient,
               struct t_HLSRV_CommandTableEntry *pCmdInfo, const char *cpCmdArgs );
  // BCB V12 (but not BCB V6) complained about the above line:
  // > [bcc32c Warning] : declaration of 'struct t_HLSRV_CommandTableEntry'
  // > will not be visible outside of this function
  // (thanks for NOT showing a 3- or 4-digit warning code so we could temporarily
  //  turn this stupidity off. It's the classic chicken-and-egg problem:
  //  We need the HLSrvCommandHandler in struct t_HLSRV_CommandTableEntry,
  //      and the struct t_HLSRV_CommandTableEntry in HLSrvCommandHandler.)

typedef struct t_HLSRV_CommandTableEntry
{ char cShortToken;    // e.g. "F",        "f",        ... (except non-ASCII hexadecimal garbage here)
  const char *pszLongToken;  // e.g. "set_freq", "get_freq", ... . NULL marks the end of the table.
                       //      An EMPTY STRING means "only the single-byte form" (cShortToken) exists.
       // 2024-11-26: Had to replace "char *" by "const char *" to make Mr. Pedantic happy,
       // without a myriad of stupid pointer casts before string literals all over the place.
  int  iAccess;        // HLSRV_ACCESS_GET or HLSRV_ACCESS_SET ..
#      define HLSRV_ACCESS_GET 0
#      define HLSRV_ACCESS_SET 1
  HLSrvCommandHandler pHandler;
} T_HLSRV_CmdTabEntry; // -> HLSRV_Commands[] (table with all commands; with short and long tokens)

// Internal forward declaractions for the functions in HLSRV_Commands[] .
// SEQUENCE as listed at hamlib.sourceforge.net/html/rigctld.1.html .
static int HLSRV_OnGetFreq(T_HLSrv *pServer, T_HLSrvClient *pClient, T_HLSRV_CmdTabEntry *pCmdInfo, const char *cpCmdArgs );
static int HLSRV_OnSetFreq(T_HLSrv *pServer, T_HLSrvClient *pClient, T_HLSRV_CmdTabEntry *pCmdInfo, const char *cpCmdArgs );
static int HLSRV_OnGetMode(T_HLSrv *pServer, T_HLSrvClient *pClient, T_HLSRV_CmdTabEntry *pCmdInfo, const char *cpCmdArgs );
static int HLSRV_OnSetMode(T_HLSrv *pServer, T_HLSrvClient *pClient, T_HLSRV_CmdTabEntry *pCmdInfo, const char *cpCmdArgs );
static int HLSRV_OnGetVFO( T_HLSrv *pServer, T_HLSrvClient *pClient, T_HLSRV_CmdTabEntry *pCmdInfo, const char *cpCmdArgs );
static int HLSRV_OnSetVFO( T_HLSrv *pServer, T_HLSrvClient *pClient, T_HLSRV_CmdTabEntry *pCmdInfo, const char *cpCmdArgs );
static int HLSRV_OnGetRIT( T_HLSrv *pServer, T_HLSrvClient *pClient, T_HLSRV_CmdTabEntry *pCmdInfo, const char *cpCmdArgs );
static int HLSRV_OnSetRIT( T_HLSrv *pServer, T_HLSrvClient *pClient, T_HLSRV_CmdTabEntry *pCmdInfo, const char *cpCmdArgs );
static int HLSRV_OnGetXIT( T_HLSrv *pServer, T_HLSrvClient *pClient, T_HLSRV_CmdTabEntry *pCmdInfo, const char *cpCmdArgs );
static int HLSRV_OnSetXIT( T_HLSrv *pServer, T_HLSrvClient *pClient, T_HLSRV_CmdTabEntry *pCmdInfo, const char *cpCmdArgs );
static int HLSRV_OnGetPTT( T_HLSrv *pServer, T_HLSrvClient *pClient, T_HLSRV_CmdTabEntry *pCmdInfo, const char *cpCmdArgs );
static int HLSRV_OnSetPTT( T_HLSrv *pServer, T_HLSrvClient *pClient, T_HLSRV_CmdTabEntry *pCmdInfo, const char *cpCmdArgs );
static int HLSRV_OnGetSplitVFO(T_HLSrv *pServer,T_HLSrvClient *pClient,T_HLSRV_CmdTabEntry *pCmdInfo,const char *cpCmdArgs);
static int HLSRV_OnSetSplitVFO(T_HLSrv *pServer,T_HLSrvClient *pClient,T_HLSRV_CmdTabEntry *pCmdInfo,const char *cpCmdArgs);

static int HLSRV_OnDumpState(T_HLSrv *pServer, T_HLSrvClient *pClient, T_HLSRV_CmdTabEntry *pCmdInfo, const char *cpCmdArgs );
static int HLSRV_OnCheckVFO( T_HLSrv *pServer, T_HLSrvClient *pClient, T_HLSRV_CmdTabEntry *pCmdInfo, const char *cpCmdArgs );

static int HLSRV_OnQuit( T_HLSrv *pServer, T_HLSrvClient *pClient, T_HLSRV_CmdTabEntry *pCmdInfo, const char *cpCmdArgs );


// Table with all tokens (long and short) and their handlers (C functions) :
//   [Without modifications, Mr. Pedantic BCB V12 complained:
//     > [bcc32c Warning] : incompatible pointer types initializing
//     > 'HLSrvCommandHandler'
//     > (aka 'int (*)(struct t_HLSrv *, struct t_HLSrvClient *,
//     >        struct t_HLSRV_CommandTableEntry *, const char *)')
//     > with an expression of type 'int (T_HLSrv *, T_HLSrvClient *, -,
//     >        T_HLSRV_CmdTabEntry *, const char *)'                  |
//     > (aka 'int (struct t_HLSrv *, struct t_HLSrvClient *,          |
//     >        struct t_HLSRV_CommandTableEntry *, const char *)')    |
//     Fixed by inserting the SUPERFLUENT CASTS in the table below.]   |
static const T_HLSRV_CmdTabEntry HLSRV_Commands[] =  //                |
{                                                    //  ,-------------'
  // ShortToken LongToken Access             Handler     |
  { 'f',     "get_freq",  HLSRV_ACCESS_GET,  (HLSrvCommandHandler)HLSRV_OnGetFreq },
  { 'F',     "set_freq",  HLSRV_ACCESS_SET,  (HLSrvCommandHandler)HLSRV_OnSetFreq },

  { 'm',     "get_mode",  HLSRV_ACCESS_GET,  (HLSrvCommandHandler)HLSRV_OnGetMode },
  { 'M',     "set_mode",  HLSRV_ACCESS_SET,  (HLSrvCommandHandler)HLSRV_OnSetMode },

  { 'q',     "exit rigctl",HLSRV_ACCESS_GET, (HLSrvCommandHandler)HLSRV_OnQuit },
  { 'Q',     "exit_rigctl",HLSRV_ACCESS_GET, (HLSrvCommandHandler)HLSRV_OnQuit },

  { 's',  "get_split_vfo",HLSRV_ACCESS_GET,  (HLSrvCommandHandler)HLSRV_OnGetSplitVFO },
  { 'S',  "set_split_vfo",HLSRV_ACCESS_SET,  (HLSrvCommandHandler)HLSRV_OnSetSplitVFO },

  { 't',     "get_ptt",   HLSRV_ACCESS_SET,  (HLSrvCommandHandler)HLSRV_OnGetPTT }, // from WSJT-X: "t VFOA\n"
  { 'T',     "set_ptt",   HLSRV_ACCESS_SET,  (HLSrvCommandHandler)HLSRV_OnSetPTT }, // from WSJT-X: "T VFOA 0\n" (0=RX, 1=TX, 2=TX_mic, 3=TX_data)

  { 'v',     "get_vfo",   HLSRV_ACCESS_GET,  (HLSrvCommandHandler)HLSRV_OnGetVFO },
  { 'V',     "set_vfo",   HLSRV_ACCESS_SET,  (HLSrvCommandHandler)HLSRV_OnSetVFO },

  { 0x00,    "dump_state",HLSRV_ACCESS_GET,  (HLSrvCommandHandler)HLSRV_OnDumpState},

  { 0x00,    "chk_vfo",   HLSRV_ACCESS_GET,  (HLSrvCommandHandler)HLSRV_OnCheckVFO },


  // End of the table marked by 'all zeros' (and NULL pointers):
  { 0x00,    NULL,        0,                 NULL }

}; // end HLSRV_Commands[]

static const long HLSRV_i32TuningSteps_Hz[] = // tuning steps for the "dump_state" reponse
{ 1, 10, 100, 1000, 2500, 5000, 6125, 8333, 1000, 12500, 25000,
  100000, 250000, 1000000
}; // end HLSRV_i32TuningSteps_Hz[]

//----------------------------------------------------------------------------
// Global variables for hardcore debugging / post-mortem crash analysis,
//        when the debugger's call stack shows nothing but garbage,
//        e.g. after an exception at the infamous address 0xFEEEFEEE .
//        ( C++Builder's debugger could still inspect SIMPLE GLOBAL
//          VARIABLES after most kinds of exceptions, but of course nothing
//          stack-based because the CPU- or task-stack was usually trashed.)
//----------------------------------------------------------------------------
#if(SWI_HARDCORE_DEBUGGING) // (1) = hardcore-debugging, (0)=normal compilation
 int HLSRV_iLastSourceLine = 0; // WATCH THIS after crashing with e.g. "0xFEEEFEEE" !
# define HERE_I_AM__HLSRV() HLSRV_iLastSourceLine=__LINE__
     // (see complete list of other XYZ_iLastSourceLine varibles to watch
     //  in C:\cbproj\Remote_CW_Keyer\Keyer_Main.cpp, near GUI_iLastSourceLine)
#else
# define HERE_I_AM__HLSRV()
#endif // SWI_HARDCORE_DEBUGGING ?


//----------------------------------------------------------------------------
// "Internal function prototypes" (functions called before their implementation)
//----------------------------------------------------------------------------

static DWORD WINAPI ServerThread( LPVOID lpParam );

// implementation of functions:

//----------------------------------------------------------------------------
void HLSRV_InitInstanceWithDefaults(T_HLSrv *pServer)
  // Called from the main application (or "GUI") before loading an actual
  // configuration FROM A FILE. The result (in *pServer) are the "defaults".
{
  int i;

  HERE_I_AM__HLSRV(); // -> HLSRV_iLastSourceLine (optional, for hardcore debugging)

  memset( pServer, 0, sizeof(T_HLSrv) ); // <- this also clears various FIFOs
  HERE_I_AM__HLSRV();

  // Provide a few 'meaningful defaults'. When first launching the application,
  // these will be displayed on the 'Network' tab, because Keyer_Main.cpp
  // (or whatever the GUI is written in) will use these values as DEFAULTS
  // in a function like LoadSettings() :
  pServer->cfg.iServerOptions = HLSRV_OPTIONS_ALLOW_ALL; // per default, allow all (including "QSY" and "change mode")
  pServer->cfg.iDiagnosticFlags = HLSRV_DIAG_FLAGS_NONE ; // per default, no diagnostic output on the 'Debug' tab
  pServer->cfg.iServerListeningPort = HLSRV_DEFAULT_SERVER_PORT;
  HERE_I_AM__HLSRV();

  // Initialize dummy frequency ranges. The 'real' ranges will be set later
  // after detecting the type of the ratio via e.g. CI-V in RigControl.c !
  HLSRV_InitFreqRange( &pServer->RxRangeList[0],  135700.0, 137800.0 ); // LF "2200 meters"
  HLSRV_InitFreqRange( &pServer->RxRangeList[1],  472000.0, 479000.0 ); // MF "630 meters"
  HLSRV_InitFreqRange( &pServer->RxRangeList[2], 1800000.0, 2000000.0 ); // MF "160 meters"
  HLSRV_InitFreqRange( &pServer->RxRangeList[3], 3500000.0, 4000000.0 ); // "80 m"
  HLSRV_InitFreqRange( &pServer->RxRangeList[4], 7000000.0, 7300000.0 ); // "40 m"
  HLSRV_InitFreqRange( &pServer->RxRangeList[5],10000000.0,10150000.0 ); // "30 m"
  HLSRV_InitFreqRange( &pServer->RxRangeList[6],14000000.0,14350000.0 ); // "20 m"
  HLSRV_InitFreqRange( &pServer->RxRangeList[7],18068000.0,18168000.0 ); // "17 m"
  HLSRV_InitFreqRange( &pServer->RxRangeList[8],21000000.0,21450000.0 ); // "15 m"
  HLSRV_InitFreqRange( &pServer->RxRangeList[9],24890000.0,24990000.0 ); // "12 m"
  HLSRV_InitFreqRange( &pServer->RxRangeList[10],      28e6,    29.7e6 ); // "10 m"
  HLSRV_InitFreqRange( &pServer->RxRangeList[11],      50e6,      52e6 ); // "6 m"   in most of ITU Region 1
  HLSRV_InitFreqRange( &pServer->RxRangeList[12],      70e6,    70.5e6 ); // "4 m"   in a few lucky countries (not DL)
  HLSRV_InitFreqRange( &pServer->RxRangeList[13],     144e6,     146e6 ); // "2 m"   in ITU Region 1
  HLSRV_InitFreqRange( &pServer->RxRangeList[14],     430e6,     440e6 ); // "70 cm" in ITU Region 1
  HLSRV_InitFreqRange( &pServer->RxRangeList[15],    1240e6,    1300e6 ); // "23 cm" in ITU Region 1
#if( HLSRV_NUM_RX_RANGES < 16) || ( HLSRV_NUM_RX_RANGES < 16 )
# error "Chaos !"
#endif
  for(i=0; i<((HLSRV_NUM_RX_RANGES<HLSRV_NUM_TX_RANGES)?HLSRV_NUM_RX_RANGES:HLSRV_NUM_TX_RANGES); ++i)
   { pServer->TxRangeList[i] = pServer->RxRangeList[i];
     pServer->RxRangeList[i].i32LowPower_mW  = 0;
     pServer->RxRangeList[i].i32HighPower_mW = 0;
   }
} // end HLSRV_InitInstanceWithDefaults()

//----------------------------------------------------------------------------
void HLSRV_InitFreqRange( T_HLSRV_FreqRange *pFreqRange, double dblStartFreq_Hz, double dblEndFreq_Hz)
{
  memset( pFreqRange, 0, sizeof(T_HLSRV_FreqRange) );
  pFreqRange->dblStartFreq_Hz = dblStartFreq_Hz; // Start frequency in Hertz
  pFreqRange->dblEndFreq_Hz   = dblEndFreq_Hz;   // End frequency in Hertz
  // Assume it's an "allmode" rig.. without those esoteric modess like "AMS", "FAX", "SAM", "SAL", SAH"..
  pFreqRange->dwModes = HLSRV_MODE_AM | HLSRV_MODE_CW | HLSRV_MODE_USB | HLSRV_MODE_LSB
                    | HLSRV_MODE_RTTY | HLSRV_MODE_FM | HLSRV_MODE_WFM | HLSRV_MODE_CWR;
  pFreqRange->dwExtModes = HLSRV_EMODE_NONE;
  pFreqRange->i32LowPower_mW = 1000;   // Lower RF power in mW, -1 for no power (ie. rx list)
  pFreqRange->i32HighPower_mW= 100000; // Higher RF power in mW, -1 for no power (ie. rx list)
  pFreqRange->dwVFO = HLSRV_VFO_CURRENT;
  pFreqRange->dwAnt = HLSRV_ANT_CURRENT;
  pFreqRange->label = ""; // "Label for this range that explains why.  e.g. Icom rigs USA, EUR, ITR, TPE, KOR"
} // end HLSRV_InitFreqRange()

//----------------------------------------------------------------------------
int HLSRV_EnumerateRequiredPNs( int iZeroBasedIndex )
  // Called from RigControl.c shortly after 'connecting' the radio,
  // to poll certain parameters that a REMOTE HAMLIB CLIENT may want, like...
  //  RIGCTRL_PN_RIT_FREQ,    RIGCTRL_PN_XIT_FREQ,
  //  RIGCTRL_PN_RIT_ENABLED, RIGCTRL_PN_XIT_ENABLED, end who-knows-what .
  // [in]     : iZeroBasedIndex = the "enumerator" (zero-based index as in C).
  //            Incremented by the caller from zero, stepping by one,
  //            until the RETURN VALUE from HLSRV_EnumerateRequiredPNs() is zero.
  // [return] : "Unified parameter number" for RigControl.c ,
  //            or ZERO when all 'required parameter numbers' are through.
{ switch( iZeroBasedIndex )
   { // Begin with parameters that ANY rig should support...
     // Note: Module RigControl.c is smart enough to NOT READ parameters
     //       that have already been successfully read from the controlled rig.
     case 0: return RIGCTRL_PN_FREQUENCY;      // .. of the CURRENTLY SELECTED VFO..
     case 1: return RIGCTRL_PN_TRANSMIT_FREQ;
     case 2: return RIGCTRL_PN_OP_MODE;
     case 3: return RIGCTRL_PN_FILTER_BANDWIDTH;
     case 4: return RIGCTRL_PN_SPLIT_MODE;
     case 5: return RIGCTRL_PN_AUDIO_VOLUME_PERCENT;
     case 6: return RIGCTRL_PN_RF_GAIN_PERCENT;
     case 7: return RIGCTRL_PN_SQUELCH_LEVEL_PERCENT;
     case 8: return RIGCTRL_PN_RF_POWER_SETTING_PERCENT;
     default: break;
     // Note: Not all of the parameters that a typical Hamlib-client
     //       like WSJT-X would 'like to know' will be available later.
     //       To check which of the parameters have really been retrieved
     //       from the connected rig (radio), select    "Settings" ..
     //           "Report Rig Control parameters on the Debug tab" .
     //       This spitted out a list like the one below, when some of the
     //       RigControl-parameter-numbers listed above were not supported yet:
     // > Current Rig Control parameters:
     // > PN001: VFOfreq = 144050000.000000 Hz
     // > PN002: OpMode = 33
     // > PN013: TX_freq = 54401 Hz
     // > PN031: RFGain = 100 %
     // > PN032: SquelchLevel = 28 %
     // > PN111: SplitMode = 0
     // > Rig Control: Got 6 valid parameter(s) so far from IC-9700

   }
  return 0;
} // HLSRV_EnumerateRequiredPNs()

//----------------------------------------------------------------------------
void HLSRV_LinkToRigControl( T_HLSrv *pServer, T_RigCtrlInstance *pRigControl )
{
  pServer->pRigControl = pRigControl;
  if( pRigControl != NULL )  // the "link" works both ways ..
   {  pRigControl->pvHamlibServer = pServer; // "link backwards" from the T_RigCtrlInstance to this Hamlib-Server
   }
} // end HLSRV_LinkToRigControl()

//----------------------------------------------------------------------------
BOOL HLSRV_Start( T_HLSrv *pServer )                                    // API
  // ,---------------------'
  // '- [in] usually the single instance from Keyer_Main.cpp ( &HamlibServer )
{
  int i;

  HERE_I_AM__HLSRV();
  HLSRV_Stop( pServer );  // <- doesn't do harm when NOT running at all
  pServer->dwThreadLoops  = pServer->dwThreadErrors = 0;
  pServer->dwNumBytesSent = pServer->dwNumBytesRcvd = 0;

  // Reset the states of all 'client instance':
  for( i=0; i<HLSRV_MAX_CLIENTS; ++i )
   { pServer->Client[i].iClientState = HLSRV_CLIENT_STATE_DISCONN;
   }

   // Create a worker thread for the TCP server, the Win32 way:
   if( pServer->hThread==NULL )
    {  pServer->iThreadStatus = HLSRV_THREAD_STATUS_LAUNCHED; // "just launched, soon RUNNING"
       HERE_I_AM__HLSRV();
       pServer->hThread = CreateThread(
          NULL,    // LPSECURITY_ATTRIBUTES lpThreadAttributes = pointer to thread security attributes
          2*65536, // DWORD dwStackSize  = initial thread stack size, in bytes
           ServerThread, // LPTHREAD_START_ROUTINE lpStartAddress = pointer to thread function ..
          (LPVOID)pServer, // LPVOID lpParameter = argument for new thread
          0,       // DWORD dwCreationFlags = creation flags
                   // zero -> the thread runs immediately after creation
          &pServer->dwThreadId // LPDWORD lpThreadId = pointer to returned thread id
          // > If CreateThread() fails, the return value is NULL (not INVALID_HANDLE_VALUE) .
          );
      HERE_I_AM__HLSRV();
      // > The thread object remains in the system until the thread has terminated
      // > and all handles to it have been closed through a call to CloseHandle.
   } // end if < CLIENT or SERVER enabled >  ?
  if( pServer->hThread==NULL ) // Check the return value for success.
   {
     pServer->pszLastError  = "CreateThread failed.";
     pServer->iThreadStatus = HLSRV_THREAD_STATUS_NOT_CREATED;
     HLSRV_Stop(pServer); // here: set various struct members back to defaults
     HERE_I_AM__HLSRV();
     return FALSE;
   }

  // > Define the Thread's priority as required.
  SetThreadPriority( pServer->hThread, // handle to the thread
             THREAD_PRIORITY_NORMAL); // thread priority level
  HERE_I_AM__HLSRV();
  return TRUE;



} // end HLSRV_Start()

//----------------------------------------------------------------------------
void HLSRV_Stop( T_HLSrv *pServer )                                      // API
{
  int i;
  if(  (pServer->iThreadStatus == HLSRV_THREAD_STATUS_LAUNCHED )
    || (pServer->iThreadStatus == HLSRV_THREAD_STATUS_RUNNING  ) )
   { pServer->iThreadStatus = HLSRV_THREAD_STATUS_TERMINATE; // politely ask the thread to terminate itself
     for(i=0; i<20; ++i )
      { Sleep(10);
        if( pServer->iThreadStatus == HLSRV_THREAD_STATUS_TERMINATED ) // bingo..
         { break;  // .. the thread has terminated itself, so no need to kill it
         }
      }
   }
  if( pServer->hThread != NULL )
   { CloseHandle( pServer->hThread );
     pServer->hThread  =  NULL;
   }
  pServer->iThreadStatus = HLSRV_THREAD_STATUS_NOT_CREATED;

} // end HLSRV_Stop()

//---------------------------------------------------------------------------
char* HLSRV_ThreadStatusToString( int iThreadStatus )
{ switch( iThreadStatus )
   { case HLSRV_THREAD_STATUS_NOT_CREATED: return "not created";
     case HLSRV_THREAD_STATUS_LAUNCHED   : return "launched";
     case HLSRV_THREAD_STATUS_RUNNING    : return "running";
     case HLSRV_THREAD_STATUS_TERMINATE  : return "waiting for termination";
     case HLSRV_THREAD_STATUS_TERMINATED : return "terminated";
     default : return "invalid thread status";
   }
} // end HLSRV_ThreadStatusToString()

//---------------------------------------------------------------------------
char* HLSRV_IPv4AddressToString( BYTE *pbIP ) // API (for GUI and 'error log')
{ static char szResult[20];
  sprintf( szResult, "%d.%d.%d.%d",  // oh well, there's some winsock thing for this...
           (int)pbIP[0], (int)pbIP[1],   // ..anyway, use this one,
           (int)pbIP[2], (int)pbIP[3]);  // without so many esoteric tyes
  return szResult;
} // end HLSRV_IPv4AddressToString()


//---------------------------------------------------------------------------
char* HLSRV_GetCurrentStatusAsString( T_HLSrv *pServer )   // API (for the GUI)
{
  static char sz127Result[128]; // only called by the GUI, no need to be thread safe,
                                // so a simple static buffer is ok here
  char *cp         = sz127Result;
  char *pszEndstop = sz127Result + 120;

  SL_AppendString( &cp, pszEndstop, "HL-Server: " );
  SL_AppendString( &cp, pszEndstop, HLSRV_ThreadStatusToString(pServer->iThreadStatus) );
  if( pServer->nRemoteClientsConnected > 0 )
   { SL_AppendPrintf( &cp, pszEndstop, ", %d client", (int)pServer->nRemoteClientsConnected );
     if( pServer->nRemoteClientsConnected > 1 )
      { SL_AppendChar( &cp, pszEndstop, 's' ); // clients, not client :)
      }
   }
  else
   { SL_AppendPrintf( &cp, pszEndstop, ", no client connected" );
   }
  SL_AppendPrintf( &cp, pszEndstop, ", tx=%ld, rx=%d bytes",
                   (long)pServer->dwNumBytesSent,
                   (long)pServer->dwNumBytesRcvd );
  return sz127Result;
} // end HLSRV_GetCurrentStatusAsString()

//---------------------------------------------------------------------------
char* HLSRV_ClientStateToString( int iClientState )
{ switch( iClientState )
   { case HLSRV_CLIENT_STATE_DISCONN : return "disc";
     case HLSRV_CLIENT_STATE_ACCEPTED: return "accepted";
     //   HLSRV_CLIENT_STATE_CONNECTED: // same value, similar meaning, but for the LOCAL CLIENT side
     case HLSRV_CLIENT_STATE_CONFIRMED: return "confirmed"; // requested "\chk_vfo", "\dump_state", etc(?), without Hamlib protocol errors
     default : return "???";
   }
} // end HLSRV_ClientStateToString()

//---------------------------------------------------------------------------
char* HLSRV_GetLastErrorAsString( T_HLSrv *pServer )      // API (for the GUI)
{ char *cpResult = pServer->pszLastError; // sufficiently thread-safe ..
  if( cpResult != NULL )
   { return cpResult;
   }
  else
   { return "no error";
   }
} // end HLSRV_GetLastErrorAsString()


//---------------------------------------------------------------------------
int HLSRV_GetFreeSpaceInTxBuffer( T_HLSrv *pServer )
  // Called from various places BEFORE HLSRV_AllocBytesInTxBuffer(),
  //     to check in advance if enough bytes can be appended for transmission.
{
  if( pServer == NULL ) // oops.. HttpServer.c doesn't have a valid reference anymore ?
   { return 0;  // avoid crashing with a NULL-pointer reference below
     // (the problem was actually fixed in HttpSrv_CreateInstance() ... )
   }
  return HLSRV_SOCKET_TX_BUFFER_SIZE - pServer->nBytesInTxBuffer;
} // end HLSRV_GetFreeSpaceInTxBuffer()


//---------------------------------------------------------------------------
BYTE* HLSRV_AllocBytesInTxBuffer( // API - but see restrictions below (*)
                 T_HLSrv *pServer,
                 int iPayloadSize)
  // Allocates a block of bytes pServer->bTxBuffer[0], and fills in the
  // 'Command' and (depending on iPayloadSize) one or two extra header bytes
  // with the BLOCK SIZE. Does NOT fill in the payload itself (but returns
  // a simple byte-pointer to the first PAYLOAD BYTE).
  //   This avoids block-copying by assembling the to-be-transmitted data block
  //   DIRECTLY in the TX buffer (aka 'outbound network buffer') .
  // Returns the address of the first byte in the **PAYLOAD**
  //         when allocation was successful, and if there's a payload to fill at all.
  // If there is no payload to fill, HLSRV_AllocBytesInTxBuffer() returns NULL .
  // (*) Because there is no PER-CLIENT transmit buffer, this function
  //     may only be called from the context of the Server- or Client-Thread,
  //     for example from ...
  //       * HLSRV_OnReceive() -> HLSRV_ExecuteCmd() -> ... -> HLSRV_AllocBytesInTxBuffer()
  //       * HLSRV_OnPoll()    -> HLSRV_AllocBytesInTxBuffer(), etc .
{
  BYTE *pb;
  int iOldBytesInTxBuffer = pServer->nBytesInTxBuffer;
  if( iPayloadSize <= ( HLSRV_SOCKET_TX_BUFFER_SIZE - iOldBytesInTxBuffer ) )
   { // Ok.. enough space left in the "transmit buffer" so ALLOCATE the block,
     //      and already fill out the 1..3 "header"-bytes :
     pServer->nBytesInTxBuffer = iOldBytesInTxBuffer + iPayloadSize; // NOW the space is occupied ..
     pb = pServer->bTxBuffer + iOldBytesInTxBuffer; // new block begins here..
     memset( pb, 0, iPayloadSize);  // clean old garbage in the "short payload"
     return pb;    // caller shall fill in the payload HERE
   } // end if < number of bytes to "allocate" fits inside the "network transmit buffer" >
  return NULL;
} // end HLSRV_AllocBytesInTxBuffer()


//---------------------------------------------------------------------------
void HLSRV_FeedActivityWatchdog( T_HLSrvClient *pClient )
{
  TIM_StartStopwatch( &pClient->swActivityWatchdog ); // feed the dog,
  // so it won't bark or bite for the next <HLSRV_ACTIVITY_TIMEOUT_MS> milliseconds.
} // end HLSRV_FeedActivityWatchdog()

//---------------------------------------------------------------------------
static void HLSRV_OnConnect( T_HLSrv *pServer, T_HLSrvClient *pClient )
  // Called from ServerThread() after a new REMOTE CLIENT has joined in.
  // We didn't receive anything from the guy on the other side ("peer") yet,
  // but we may already return a few bytes as the 'greeting message' from here.
  // [out, optional] pServer->bTxBuffer[ pServer->nBytesInTxBuffer++ ]
  //     = "anything to send", limited to
  //       HLSRV_SOCKET_TX_BUFFER_SIZE - pServer->nBytesInTxBuffer (on entry) .
{
  HLSRV_FeedActivityWatchdog( pClient ); // feed the dog, so it won't bite ...
                       // for the next <HLSRV_ACTIVITY_TIMEOUT_MS> milliseconds.
  pClient->dblUnixTimeOfStart = UTL_GetCurrentUnixDateAndTime(); // <- here: set immediately in HLSRV_OnConnect()
  if( pServer->cfg.iDiagnosticFlags & HLSRV_DIAG_FLAGS_VERBOSE_OUTPUT )
   {
     ShowError( ERROR_CLASS_INFO | SHOW_ERROR_TIMESTAMP, "HamlibSrv: Accepted connection from %s:%d (no data yet)",
         HLSRV_IPv4AddressToString( pClient->b4HisIP.b ), (int)pClient->iHisPort );
   } // end if < show 'accepted connections'> ?

  // On the 'CW Server' side, don't send anything immediately .
  // Let the PEER (remote client) send his first few bytes (and check what
  // 'he' is) . As long we we haven't verified he's a REMOTE CW CLIENT,
  // he won't receive a single byte from this server. See implementation
  // of HLSRV_OnPoll( ) .

} // end HLSRV_OnConnect()

//---------------------------------------------------------------------------
static void HLSRV_OnDisconnect( T_HLSrv *pServer, T_HLSrvClient *pClient )
  // Called from ServerThread() after a REMOTE CLIENT disconnected
  //                            (thus it's too late to say 'goodbye' here).
  // Note: Besides 'graceful' TCP disconnect, there's also the
  //               'unexpect connection reset'. To keep things simple,
  //  we don't care WHY and HOW a connection was "cut"/"closed"/"reset".
{
  if( (pServer->cfg.iDiagnosticFlags & HLSRV_DIAG_FLAGS_VERBOSE_OUTPUT )
    &&( ! pClient->fDisconnect ) // only show this if NOT DISCONNECTED YET :
    )
   { // Will typically get here when CLOSING THE REMOTE WEB BROWSER ...
     ShowError( ERROR_CLASS_INFO | SHOW_ERROR_TIMESTAMP, "HamlibSrv: Disconnected by %s:%d",
                HLSRV_IPv4AddressToString(pClient->b4HisIP.b),
                (int)pClient->iHisPort ); // (*)
     if( (pClient->dwNumBytesRcvd + pClient->dwNumBytesSent) > 0 )
      { ShowError( ERROR_CLASS_INFO, "        %ld bytes rvcd, %ld sent.",
         (long)pClient->dwNumBytesRcvd, (long)pClient->dwNumBytesSent );
      }
   } // end if( pServer->cfg.iDiagnosticFlags & HLSRV_DIAG_FLAGS_VERBOSE )

} // end HLSRV_OnDisconnect()

//---------------------------------------------------------------------------
static void HLSRV_PrintToResponse( T_HLSrv *pServer, T_HLSrvClient *pClient, const char *pszFormat, ... )
{
  va_list parameter;   // argument list for VA_... macros
  int nCharsRemaining; // .. in our fixed-sized string buffer for the 'response'
  char *pszStart   = (char*)pClient->bResponse; // 2024-11 : (char*) cast added for pedantic BCB V12.
                      // '-->  Of course these stupid pointer casts don't improve the code,
                      //       but with Microcontroller firmware in mind,
                      //       we want to "build a string (response) in an array of BYTES"
                      //       without unnecessary block-copying etc.
  char *pszDest    = pszStart;
  char *pszEndstop = pszStart + HLSRV_RESPONSE_MAX_LENGTH - 1;
  va_start( parameter, pszFormat );

  SL_SkipToEndOrEndstop( &pszDest, pszEndstop );  // <- APPEND, don't OVERWRITE ..
  nCharsRemaining = pszEndstop - pszDest;
  if( nCharsRemaining > 1 )
   { vsnprintf( pszDest, nCharsRemaining, pszFormat, parameter );
   }
  va_end( parameter );  // WE called va_start(), so WE call va_end() [because vsnprintf() does NOT do that]
  SL_SkipToEndOrEndstop( &pszDest, pszEndstop );
  pClient->iResponseLength = (int)(pszDest - pszStart); // -> number of bytes to send,
                        // unless the caller appends more.
  // 2024-07-10: pszDest was 0x00744631, pClient->bResponse was 0x00744628,
  //   but BCB thought pClient->iResponseLength was ZERO at this point ? ! ?
  //   Fixed, as many times before, by doing a BUILD ALL (bloody precompiled headers).
} // end HLSRV_PrintToResponse()


//---------------------------------------------------------------------------
static void HLSRV_AppendErrorCodeToResponse( T_HLSrv *pServer, T_HLSrvClient *pClient,
                            int iHamlibErrorCode )
  //    ,------------------------'
  //    '--> What's this ? From hamlib.sourceforge.net/html/rigctld.1.html :
  // > A one line response will be sent as a reply to set commands,
  // > "RPRT x\n" where x is the Hamlib error code with '0' indicating success
  // > of the command.   (...)
  // > The protocol is simple, commands are sent to rigctld on one line and
  // > rigctld responds to get commands with the requested values,
  // > one per line, when successful, otherwise, it responds with one line
  // > "RPRT x", where x is a negative number indicating the error code.
  // > Commands that do not return values respond with the line RPRT x,
  // > where x is 0 when successful, otherwise is a regative number (A-HA!)
  // > indicating the error code. (...)
  //     In THIS implementation, error codes ARE ALREADY NEGATIVE
  //     *in the definition* in HamlibServer.h, except for
  //     result code 0 = HLSRV_RESULT_NO_ERROR .
  //     Not sure what the token 'RPRT' was meant to be; possibly 'Report' ?
{
  char *pszDest    = (char*)pClient->bResponse;
  char *pszEndstop = (char*)pClient->bResponse + HLSRV_RESPONSE_MAX_LENGTH - 1;

  SL_SkipToEndOrEndstop( &pszDest, pszEndstop );  // <- APPEND, don't OVERWRITE ..
  SL_AppendPrintf( &pszDest, pszEndstop, "RPRT %d\n", (int)iHamlibErrorCode );
  pClient->iResponseLength = pszDest - (char*)pClient->bResponse; // -> number of bytes to send
   // ,--------------------------------|_____|
   // '--> Yet another Extrawurst for Mr. Pedantic, who complained:
   //  > [bcc32c Error] : 'char *' and 'unsigned char *' are not pointers to compatible types

} // end HLSRV_AppendErrorCodeToResponse()


//---------------------------------------------------------------------------
static void HLSRV_ExecuteCmd( T_HLSrv *pServer,
              T_HLSrvClient *pClient ) // [in] instance data for a CLIENT,
  // Called from HLSRV_OnReceive() when a received command is "ready for processing",
  //        including all bytes belonging to the command line.
  //   From a "manpage" about the original Hamlib "rigctld":
  // > Commands can be sent over the TCP socket either as a single char,
  // > or as a long command  name plus the value(s) space separated on one
  // > '\n' terminated line. See PROTOCOL.
  // >
  // > Since most of the Hamlib operations have a set and a get method,
  // > an upper case letter will be used for set methods
  // > whereas the corresponding lower case letter refers to the get method.
  // > Each operation also has a long name; prepend a backslash to send a
  // > long command name.
  //
  // [in]  pClient->bRcvdLine[ 0 .. pClient->iRcvdLineLength-1]
  // [out] pClient->bResponse[]: For the Hamlib protocol, a zero-byte-terminated
  //                             C string that possibly contains MULTIPLE lines.
  //                             Each line in a multi-line response ends with a '\n'.
  //
{
  int  iClientIndex = (int)(pClient - &pServer->Client[0]); // again the "pointer trick" to get an array index (in C)
  int  i,n;
  int  iPingLatency_ms;
  BYTE *pb;
  BOOL found = FALSE;
  char sz255[256], *cp, *cpSrc, *cpSrcEndstop, *cpDest, *cpDstEndstop;
  int  iResult;
  T_HLSRV_CmdTabEntry *pCmd = (T_HLSRV_CmdTabEntry*)HLSRV_Commands;

  cpSrc        = (char*)pClient->bRcvdLine;
  cpSrcEndstop = (char*)pClient->bRcvdLine + pClient->iRcvdLineLength;
  if( *cpSrc == '\\' ) // seems to be a "long command".. how long is THE TOKEN ?
   { ++cpSrc;         // skip the leading backslash
     cp = cpSrc;
     while( (cp<cpSrcEndstop) && SL_IsLetterOrUnderscore(*cp) )
      { ++cp;
      }
     n = cp - cpSrc;  // length of the "long command" for string comparison,
                      // e.g. bRcvdLine = "\\chk_vfo\n" -> n = 7
     while( (!found ) && (pCmd->pszLongToken != NULL) ) // repeat until found or end-of-table
      { if(  (SL_strncmp( cpSrc, pCmd->pszLongToken, n ) == 0 )
           &&(pCmd->pszLongToken[n] == '\0') ) // token length also ok ?
         { found = TRUE;
           cpSrc += n;
           break;
         }
        else
         { ++pCmd;    // check the next entry in HLSRV_Commands[]
         }
      }
   }
  else // not a "long command" but a single byte (not necessary a plain text character) :
   { while( (!found ) && (pCmd->pszLongToken != NULL) ) // repeat until found or end-of-table
      { if( *cpSrc == pCmd->cShortToken )
         { found = TRUE;
           ++cpSrc;
           break;
         }
        else
         { ++pCmd;    // check the next entry in HLSRV_Commands[]
         }
      }
   }

  if( found )  // found the Hamlib command in the lookup table ?
   { memset( pClient->bResponse, 0, HLSRV_RESPONSE_MAX_LENGTH ); // clear the response buffer
     pClient->iResponseLength = 0; // <- if not set by the handler function,
                                   //  the number of bytes to send in response
                                   //  is determined by strlen() further below.

     // If there's a "handler" for this get-/set-/check-command, invoke it:
     if( pCmd->pHandler != NULL )
      {
#      ifndef __CODEGEARC__
        iResult = pCmd->pHandler( pServer, pClient, pCmd, cpSrc/*arguments*/ );
        // ,------------|______|          ,---------------|___|
        // '-> e.g. HLSRV_OnCheckVFO()    '-> e.g. "\n" if there are NO arguments in the line.
        // The above function pointer call was ok for BCB V6, no warnings.
        // But C++Builder V12 (using the "Clang-enhanced compiler") complained:
        // > [bcc32c Warning] : incompatible pointer types passing
        // >     'T_HLSRV_CmdTabEntry *' (aka 'struct t_HLSRV_CommandTableEntry *')
        // >  to parameter of type
        // >     'struct t_HLSRV_CommandTableEntry *' .
        // What the heck ? Both are THE SAME STRUCTURE, once as "struct Blah",
        //                                          and once as "type Blah",
        //  which was used this way because of the classic 'chicken and egg' problem
        //  where a struct contains a function pointer, and the function itself
        //  expects the address of the struct itself (kind of "this"-pointer w/o C++).
#      else //  __CODEGEARC__ defined -> try to eliminate the umpteenth warning..
        iResult = pCmd->pHandler( pServer, pClient,
          (struct t_HLSRV_CommandTableEntry*)pCmd,  // <- OMG. What a stupid kludge !
                                  cpSrc/*arguments*/ );
          // With the above cast, the WARNING from "bcc32x" (C++Builder V12)
          // was even more ridiculous :
          // > [bcc32c Warning] HamlibServer.c(716):
          // >    incompatible pointer types passing
          // >      'struct t_HLSRV_CommandTableEntry *' to parameter of type
          // >      'struct t_HLSRV_CommandTableEntry *' .
          // Where is the difference between these two pointer types,
          //       Mr. Pedantic ? Why don't you show us a NUMERIC WARNING CODE
          //       to turn off this silly warning temporarily, via #pragma warn ?
#      endif // Extrawurst fr 'Embarcadero' / 'Codegear' C++Builder V12 / "bcc32c" ?
        switch( iResult ) // anything 'special' to do here ?
         { case HLSRV_RESULT_NO_ERROR: // all ok; just send what's in pClient->bResponse[]
              // If the remote client sent a well-formed request supported by this server,
              // turn off the 'activity watchdog' (timeout monitor) because
              // for example, WSJT-X stops asking questions after a while
              // but must keep the TCP/IP connection open as long as it runs. Thus:
              if( pClient->iResponseLength > 0 ) // remote client asked for e.g. "\chk_vfo" and gets a non-empty response ?
               { pClient->iClientState = HLSRV_CLIENT_STATE_CONFIRMED; // don't kick him out if he doesn't send any other requests
               }
              // Detect the response length if the handler didn't explicitly set it:
              if((pClient->iResponseLength == 0) && (!pClient->fDisconnect) )
               {  pClient->iResponseLength = SL_strnlen( (char*)pClient->bResponse, HLSRV_RESPONSE_MAX_LENGTH );
                  if( pClient->iResponseLength == 0 ) // still nothing to send ?
                   { HLSRV_AppendErrorCodeToResponse(pServer,pClient, iResult);
                     // '--> e.g. pClient->bResponse[] = "RPRT 0\n" = "no error".
                   }
               }
              break;
           default: // the handler (command-specific C function) bailed out with an error code.
              // So unless the function already emitted it into pClient->bResponse[],
              // append the "RPRT x" (where 'x' shall be a NEGATIVE error code) here:
              if( iResult > 0 )
               {  iResult = -iResult;
               }
              if( pClient->bResponse[0] == 0x00 )
               { HLSRV_AppendErrorCodeToResponse(pServer,pClient, iResult);
                 // '--> e.g. pClient->bResponse[] = "RPRT -9\n" = "Command rejected by the rig".
                 // HLSRV_AppendErrorCodeToResponse() also updates pClient->iResponseLength .
               }
              break;
         } // end switch( iResult )
        // If there's a response (or an 'error code' to send, append it
        // to the server's outbound network buffer:
        if( pClient->iResponseLength > 0 )
         { pb = HLSRV_AllocBytesInTxBuffer( pServer, pClient->iResponseLength );
           if( pb != NULL )
            { memcpy( pb/*dest*/, pClient->bResponse, pClient->iResponseLength );
            }
           pClient->iResponseLength = 0; // nothing more to send - it's all in the "network buffer" now
         }
      } // end if < got a function pointer with the "command handler" >
   } // end if( found )
} // end HLSRV_ExecuteCmd()

//---------------------------------------------------------------------------
static void HLSRV_OnPoll( T_HLSrv *pServer, // allows sending "unsolicited data"
              T_HLSrvClient *pClient) // [in] instance data for a CLIENT
  // [out, optional] pServer->bTxBuffer[ pServer->nBytesInTxBuffer++ ]
  //      = "anything to send", limited to
  //         HLSRV_SOCKET_TX_BUFFER_SIZE - pServer->nBytesInTxBuffer (on entry)
  // Periodically called from either client- or server thread to poll for
  // transmission, even if there was NOTHING RECEIVED on a socket during the
  // "polling interval" of approximately <HLSRV_SOCKET_POLLING_INTERVAL_MS> .
{
  int  iClientIndex = (int)(pClient - &pServer->Client[0]); // again the "pointer trick" to get an array index (in C)
  int  i, iLen, iLen2, iHeadIndex, iTailIndex, nSamples, nSpectra;
  int  iTime_ms;
  float fltSample;
  short i16Sample;

  // Since 2024-07-12, the "activity watchdog" shall not bark up the wrong tree.
  //  For example, after a few initial quieries (via our Hamlib server port),
  //  WSJT-X doesn't ask any other questions, but wants to keep the TCP port open.
  //  Thus:
  if( pClient->iClientState != HLSRV_CLIENT_STATE_CONFIRMED ) // still "subject to timeout checking" ?
   { iTime_ms = TIM_ReadStopwatch_ms( &pClient->swActivityWatchdog );
     if( iTime_ms > HLSRV_ACTIVITY_TIMEOUT_MS )
      { if( ! pClient->fDisconnect ) // only show this if NOT DISCONNECTED YET :
         { ShowError( ERROR_CLASS_INFO | SHOW_ERROR_TIMESTAMP, "HamlibSrv%d: Disconnected %s:%d (timeout after %d ms), rx=%ld, tx=%ld",
                (int)iClientIndex, HLSRV_IPv4AddressToString( pClient->b4HisIP.b ),
                (int)pClient->iHisPort,
                (int)iTime_ms, // <- seeing the number of MILLISECONDS helped to
                               //    track down missing calls of
                (long)pClient->dwNumBytesRcvd,
                (long)pClient->dwNumBytesSent );
           // Sometimes, a certain web browser simultaneously opened THREE connections
           // (sockets), but only used ONE of them. See example in HLSRV_OnConnect().
         }
        pClient->fDisconnect = TRUE;
      } // end if < "activity timeout" >
     // Note the problem with this: If a remote client 'falls asleep',
     // he may theoretically occupy one of our precious TCP/IP sockets forever.
     // But for example WSJT-X sent NOTHING MORE after a few initial queries
     //  - see Remote_CW_Keyer/Hamlib_traffic_example.txt (commented Wireshark dump).
   } // end if( pClient->iClientState != HLSRV_CLIENT_STATE_CONFIRMED )


} // end HLSRV_OnPoll()

//---------------------------------------------------------------------------
static void HLSRV_OnReceive( T_HLSrv *pServer, // used by BOTH, Client- *AND* Server-side (*)
              T_HLSrvClient *pClient,         // [in] instance data for a CLIENT
              BYTE *pbRcvdData, int nBytesRcvd ) // [in] received stream segment
              // ,------------------'
              // '-> may be zero or even negative if a connection "broke down" .
  // [out, optional] pServer->bTxBuffer[ pServer->nBytesInTxBuffer++ ]
  //      = "anything to send", limited to
  //         HLSRV_SOCKET_TX_BUFFER_SIZE - pServer->nBytesInTxBuffer (on entry)
  // (*) HLSRV_OnReceive() contains the 'bytestream parser',
  //     used for ANYTHING received on any socket (in both directions).
  //
{
  BYTE b;
  char *pszInfo, sz40Info[44], sz1kDump[1024];
  // For certain commands (or bytestream types), we need the CLIENT INDEX,
  // for example to tell if the remote client (here, represented by pClient)
  // 'currently has the key' (may key the transmitter) :
  int iClientIndex = (int)(pClient - &pServer->Client[0]); // perfectly legal "pointer trick":
      // iClientIndex = 0..HLSRV_MAX_CLIENTS-1 means pClients points into the
      //                above array (Client[]), and may be compared
      //                against pServer->iTransmittingClient further below.
  if( (pClient==NULL) || (pServer==NULL) || (pbRcvdData==NULL) )
   { return;
   }

  if( nBytesRcvd > 0 )
   { HLSRV_FeedActivityWatchdog( pClient ); // feed the dog .. here: after RECEPTION
   }

  //  If this is the FIRST segment received from this client, check if the
  //  other guy isn't a WEB BROWSER (or some hacker thing trying to attack
  //  a web browser).
  //  To avoid having to check for all this stuff (that we don't support anyway),
  //  check in advance if the remote client is an "ordinary web browser"
  //  who tries to talk HTTP with us. If he does, DON'T LET ANYTHING ELSE FROM HIM
  //  pass to our beautifully simple 'bytestream parser' further below.
  pszInfo = "";
  if( (pClient->dwNumBytesRcvd==0) && (pClient->dwNumSegmentsRcvd==0) // very first call ?
    && (nBytesRcvd > 16) ) // Enough bytes to check for "GET"/"POST"/etc and the trailing "\r\n\r\n" ?
   { HERE_I_AM__HLSRV();
     if( SL_strncmp( (char*)pbRcvdData + nBytesRcvd - 4, "\r\n\r\n", 4 ) == 0 ) // looks like HTTP ending with an empty LINE..
      { pszInfo = "(HTTP)";
      } // end if < may be HTTP because 'looks like text and ends with an EMPTY LINE' >
   } // end if < first reception on a new connection with what MAY BE a request from a WEB BROWSER >

  HERE_I_AM__HLSRV();

  if( (pServer->cfg.iDiagnosticFlags & HLSRV_DIAG_FLAGS_VERBOSE_OUTPUT )
    &&(pClient->dwNumSegmentsRcvd == 0) ) // don't flooding the 'debug log' ..
   { HERE_I_AM__HLSRV();
     ShowError( ERROR_CLASS_INFO | SHOW_ERROR_TIMESTAMP, "HamlibSrv%d: connected by %s:%d %s",
                (int)iClientIndex, HLSRV_IPv4AddressToString( pClient->b4HisIP.b ),
                (int)pClient->iHisPort, pszInfo );
   }

  ++pClient->dwNumSegmentsRcvd;
  if( nBytesRcvd > 0 )
   { pClient->dwNumBytesRcvd += nBytesRcvd; // here: nunber of bytes PER CONNECTION (socket).
   }

  if( pServer->cfg.iDiagnosticFlags & HLSRV_DIAG_FLAGS_SHOW_NETWORK_TRAFFIC )
   { HERE_I_AM__HLSRV();
     INET_DumpTraffic_HexOrASCII( sz1kDump,500, pbRcvdData, nBytesRcvd );
     ShowError( ERROR_CLASS_RX_TRAFFIC | SHOW_ERROR_TIMESTAMP, "HLSrv%d %s",
                 (int)iClientIndex, sz1kDump );
     // Example (seen in the 'Debug' tab when WSJT-X tried to control the rig):
     // > 16:53:32.9 HLSrv0 [0009] \chk_vfo\n
     // > 16:53:42.9 HLSrv0 [000C] \dump_state\n
   }

  // Pass everything through the simplistic 'Hamlib / rigctld protocol parser':
  // From hamlib.sourceforge.net/html/rigctld.1.html :
  // > PROTOCOL
  // >  There are two protocols in use by rigctld, the Default Protocol
  // >  and the Extended Response Protocol.
  // >
  // >  The Default Protocol is intended primarily for the communication
  // >  between Hamlib library functions and rigctld
  // >  ("NET rigctl", available using radio model 2).
  // >
  // >  The Extended Response Protocol is intended to be used with scripts
  // >  or other programs interacting directly with rigctld
  // >  as consistent feedback is provided.
  // >
  // > Default Protocol
  // > ----------------
  // > The Default Protocol is intentionally simple. Commands are entered on a
  // > single line with any needed values. In practice, reliable results are
  // > obtained by terminating each command string with a newline character, \n.
  // >
  // > Example set frequency and mode commands :
  // >     "F 14250000\n"
  // >     "\\set_mode LSB 2400\n"  (leading single backslash, trailing newline)
  // >
  // > A one line response will be sent as a reply to set commands,
  // > "RPRT x\n" where x is the Hamlib error code with 0 indicating success of the command.
  // >
  // > Responses from rigctld get commands are text values and match the same
  // > tokens used in the set commands. Each value is returned on its own line.
  // > On error the string RPRT x\n is returned where x is the Hamlib error code.
  // >
  // > Example get frequency :
  // >       "f\n"            (DL4YHF: that's the 'get' request)
  // >       "14250000\n"     (DL4YHF: that's the response)
  // >
  // > Most get functions return one to three values. A notable exception
  // > is the dump_caps command which returns many lines of key:value pairs.
  // >
  // > This protocol is primarily used by the "NET rigctl" (rigctl model 2)
  // > backend which allows applications already written for Hamlibs C API
  // > to take advantage of rigctld without the need of rewriting application code.
  // > An applications user can select rotator model 2 (NET rigctl) and then
  // > set rig_pathname to localhost:4532 or other network host:port
  // > (set by the -T/-t options, respectively, above).
  // >
  // > Extended Response Protocol
  // > --------------------------
  // > The Extended Response protocol adds several rules to the strings
  // > returned by rigctld and adds a rule for the command syntax.
  // >
  // > 1. The command received by rigctld is echoed with its long command name
  // >    followed by the value(s) (if any) received from the client
  // >    terminated by the specified response separator
  // >    as the first record of the response.
  // >
  // > 2. The last record of each block is the string RPRT x\n where x
  // >    is the numeric return value of the Hamlib backend function
  // >    that was called by the command.
  // >
  // > 3. Any records consisting of data values returned by the radio backend
  // >    are prepended by a string immediately followed by a colon
  // >    then a space and then the value terminated by the response separator.
  // >   e.g. "Frequency: 14250000\n" when the command was prepended by '+'.
  // >
  // > 4. All commands received will be acknowledged by rigctld
  // >    with records from rules 1 and 2. Records from rule 3 are only returned
  // >    when data values must be returned to the client.
  // >
  // > An example response to a set_mode command sent from the shell prompt
  // >   (note the prepended +):
  // >       $ echo "+M USB 2400" | nc -w 1 localhost 4532
  // >          set_mode: USB 2400
  // >          RPRT 0
  // >    In this case the long command name and values are returned on the
  // >    first line and the second line contains the end of block marker
  // >    and the numeric radio backend return value indicating success.
  // >
  // > An example response to a get_mode query:
  // >       $ echo "+\get_mode" | nc -w 1 localhost 4532
  // >          get_mode:
  // >          Mode: USB
  // >          Passband: 2400
  // >          RPRT 0
  // > Note: The '\' is still required for the long command name
  // >       even with the ERP character.
  // > In this case, as no value is passed to rigctld, the first line consists
  // > only of the long command name. The final line shows that the command
  // > was processed successfully by the radio backend.
  // >
  // > Invoking the Extended Response Protocol requires prepending a command
  // > with a punctuation character. As shown in the examples above,
  // > prepending a + character to the command results in the responses
  // > being separated by a newline character (\n).
  // > Any other punctuation character recognized by the C ispunct() function
  // > except \, ?, or _ will cause that character to become
  // > the response separator and the entire response will be on one line.
  // >
  // > Separator character summary:
  // > +   Each record of the response is appended with a newline (\n).
  // > ;, |, or, ,  Each record of the response is appended by the
  // >       given character resulting in entire response on one line.
  // >       These are common record separators for text representations
  // >       of spreadsheet data, etc.
  // > ?   Reserved for help in rigctl.
  // > '_   Reserved for get_info short command
  // > #   Reserved for comments when reading a command file script.
  // >       Note: Other punctuation characters have not been tested!
  // >             Use at your own risk.
  // >
  // > For example, invoking a get_mode query with a leading ; returns:
  // >       get_mode:;Mode: USB;Passband: 2400;RPRT 0
  // >
  // > Or, using the pipe character | returns:
  // >       get_mode:|Mode: USB|Passband: 2400|RPRT 0
  // >
  // > And a set_mode command prepended with a | returns:
  // >       set_mode: USB 2400|RPRT 0
  // > Such a format will allow reading a response as a single event
  // > using a preferred response separator.
  // > Other punctuation characters have not been tested!
  // >
  // > The following commands have been tested with the Extended Response protocol (..):
  // > set_freq, get_freq, set_split_freq, get_split_freq, set_mode, get_mode,
  // > set_split_mode, get_split_mode, set_vfo, get_vfo,   set_split_vfo,
  // > get_split_vfo, set_rit, get_rit, set_xit, get_xit, set_ptt, get_ptt,
  // > power2mW, mW2power, dump_caps.

  HERE_I_AM__HLSRV();
  while( (nBytesRcvd--) > 0 )
   { b = *pbRcvdData++;
     if( pClient->iRcvdLineLength < HLSRV_RCVD_LINE_MAX_LENGTH )
      {  pClient->bRcvdLine[pClient->iRcvdLineLength++] = b;
      }
     if( b=='\n' ) // end of a line in any of Hamlib's protocols ->
      { // Note: The newline character ('\n') is in pClient->bRcvdLine[] .
        //       To ease parsing that line with standard C string functions,
        //       append a TRAILING ZERO here (*not* counted in iRcvdLineLength):
        if( pClient->iRcvdLineLength < HLSRV_RCVD_LINE_MAX_LENGTH )
         {  pClient->bRcvdLine[pClient->iRcvdLineLength] = 0x00; // now the "line" is a C string
         }
        HLSRV_ExecuteCmd( pServer, pClient );
        // '--> invokes a command handler function, and (in most cases)
        //      appends a few bytes to the outbound 'network buffer' .

        pClient->iRcvdLineLength = 0; // prepare receiving the next line of text

      } // end if < finished receiving a line, ending with 'newline'
   } // end while( nBytesRcvd-- )
  HERE_I_AM__HLSRV();
} // end HLSRV_OnReceive()


//---------------------------------------------------------------------------
static DWORD WINAPI ServerThread( LPVOID lpParam )
  // SERVER THREAD .. principle explained in
  //                  "Socket-based async multi-client tcp server.txt" .

{
  T_HLSrv *pServer = (T_HLSrv*)lpParam; // address of our instance data (struct)
     // ,--'
     // '--> expect this to be &HamlibServer = the instance prepared in Keyer_Main.cpp .
  T_HLSrvClient *pClient;
  SOCKET master, client_socket[HLSRV_MAX_CLIENTS], s;
    // '---> In winsock2, a SOCKET is a "UINT_PTR" - NOT AN INTEGER !
    //       But don't assume it's a POINTER - thus when invalid,
    //       any of these variables must not be NULL or 0 but INVALID_SOCKET !
  struct sockaddr_in server, address;
  int activity, addrlen, iClientIndex, n, valread;
  TIMEVAL timeout; // funny thing with SECONDS and MICROSECONDS, for select()

  // set of socket descriptors
  fd_set readfds;
  int max_sd;   // "highest file descriptor number" for select() [Berkeley only, ignored by winsock]
  BOOL fOk, fUpdateStatistics = FALSE;

  // Debugging stuff .. may kick this out one fine day:
  T_TIM_Stopwatch swLoopTime;
  char sz1kDump[1024];

  HERE_I_AM__HLSRV(); // <- for post-mortem analysis, or thread-crash-analysis in the GUI

  pServer->iThreadStatus = HLSRV_THREAD_STATUS_RUNNING; // cq cq, this is ServerThread() calling and standing by ..

  for(iClientIndex=0; iClientIndex<HLSRV_MAX_CLIENTS; iClientIndex++)
   { client_socket[iClientIndex] = INVALID_SOCKET;  // INVALID_SOCKET, not zero
     // (who nows if 'zero' isn't a VALID socket - don't assume anything)
   }

  // Create the server's "listening" socket (on the 'published' port number)
  if((master = socket(AF_INET , SOCK_STREAM , 0 )) == INVALID_SOCKET)
   { //  ,---------------------------|____|
     //  '-- "STREAM" is the usual geek speak for a TCP/IP socket .
     // shit happens .. try to report this somewhere :
     int error_code = WSAGetLastError();  // "quickly" get the error code, before calling any other 'WSA'-thing.. (*)
     // (*) Somehow, WSAGetLastError() must be aware of the 'caller' (thread ID?)..
     sprintf( pServer->sz255LastError, "HamlibSrv: could not create socket (%s)",
                                       INET_WinsockErrorCodeToString( error_code ) );
     pServer->pszLastError = pServer->sz255LastError;
     pServer->iThreadStatus = HLSRV_THREAD_STATUS_TERMINATED; // flag for the application
     if( pServer->cfg.iDiagnosticFlags & HLSRV_DIAG_FLAGS_VERBOSE_OUTPUT )
      { ShowError( ERROR_CLASS_ERROR, pServer->sz255LastError );
      }
     HERE_I_AM__HLSRV();
     return -1; // rest in peace, ServerThred() ... running without a LISTENING socket is useless
   }

  // Prepare the sockaddr_in structure
  server.sin_family = AF_INET;
  server.sin_addr.s_addr = INADDR_ANY;
  server.sin_port = htons( pServer->cfg.iServerListeningPort );

  // Bind the server's "listening socket" to our well-known (published) TCP port:
  if( bind(master ,(struct sockaddr *)&server , sizeof(server)) == SOCKET_ERROR)
   { // shit happens ...
     sprintf( pServer->sz255LastError, "HamlibSrv: could not bind socket (%s)",
              INET_WinsockErrorCodeToString( WSAGetLastError() ) );
     pServer->pszLastError = pServer->sz255LastError;
     pServer->iThreadStatus = HLSRV_THREAD_STATUS_TERMINATED; // flag for the application
     if( pServer->cfg.iDiagnosticFlags & HLSRV_DIAG_FLAGS_VERBOSE_OUTPUT )
      { ShowError( ERROR_CLASS_ERROR, pServer->sz255LastError );
      }
     HERE_I_AM__HLSRV();
     return -1; // farewell, ServerThred() ...
   }

  // Begin listening to incoming connections
  if( listen(master , 3/*"backlog"*/ ) == SOCKET_ERROR)
   { // shit ..
     snprintf( pServer->sz255LastError, 200, "HamlibSrv: can't listen on port %d (%s)",
              (int)pServer->cfg.iServerListeningPort,
              INET_WinsockErrorCodeToString( WSAGetLastError() ) );
     pServer->pszLastError = pServer->sz255LastError;
     pServer->iThreadStatus = HLSRV_THREAD_STATUS_TERMINATED; // flag for the application
     if( pServer->cfg.iDiagnosticFlags & HLSRV_DIAG_FLAGS_VERBOSE_OUTPUT )
      { ShowError( ERROR_CLASS_ERROR, pServer->sz255LastError );
      }
     HERE_I_AM__HLSRV();
     return -1; // farewell, ServerThred() ...
   }
  // 2024-06-24 : See troubleshooting with "netstat -ab -p tcp" described in
  //     Remote_CW_Keyer/manual/Remote_CW_Keyer.htm#network_troubleshooting !

  if( pServer->cfg.iDiagnosticFlags & HLSRV_DIAG_FLAGS_VERBOSE_OUTPUT )
   { ShowError( ERROR_CLASS_INFO | SHOW_ERROR_TIMESTAMP, "HamlibSrv: thread running.." );
   }

  // Accept incoming connections, and 'serve' remote client in this loop ...
  while( pServer->iThreadStatus == HLSRV_THREAD_STATUS_RUNNING )
   {
     HERE_I_AM__HLSRV();
     ++pServer->dwThreadLoops;  // Does this thread "sleep" most of the time, or is it a CPU hog ?
     pServer->dw8ThreadIntervals_us[ pServer->dwThreadLoops & 7 ] = TIM_ReadStopwatch_us( &swLoopTime );
     TIM_StartStopwatch( &swLoopTime );



     // Forget old flags BEFORE EVERY CALL OF SELECT, at least for the 'set'
     // of sockets on which we want to wait. Seen somewhere on the net:
     // > It's important that you re-enable the file descriptors that were cleared
     // > prior to starting another select, otherwise, you will no longer be
     // > waiting on those file descriptors.
     FD_ZERO(&readfds);

     // add master socket to fd set
     FD_SET(master, &readfds);
     // > FD_SET places sockets into a "set" for various purposes, such as
     // > testing a given socket for readability using the readfds parameter
     // > of the select function.

     // add child sockets to fd set
     for( max_sd=iClientIndex=0 ; iClientIndex < HLSRV_MAX_CLIENTS; iClientIndex++)
      {
        s = client_socket[iClientIndex];
        if(s != INVALID_SOCKET)
         { FD_SET( s , &readfds);
         }
        // highest file descriptor number; Berkeley/Linux needs this for select()
        if((int)s > max_sd)
         { max_sd = (int)s;  // indeed the maximum "SOCKET" value, not the array index 'i' !
           // (guess in Berkeley, a SOCKET is indeed an array index,
           //  not a 'handle' / 'pointer'-like thingy that is possibly is in winsock)
         }
      } // end for < add child sockets to fd set >

     // Prepare waiting for an activity on any of the sockets.
     // Unlike the simple server example, our timeout for select()
     // isn't NULL, because we do NOT want to wait indefinitely ('endlessly'):
     timeout.tv_sec  = 0;     // Pfui Deibel.. warum einfach, wenn's auch kompliziert geht !
     timeout.tv_usec = HLSRV_SOCKET_POLLING_INTERVAL_MS * 1000; // max time to wait in MICROSECONDS !
     HERE_I_AM__HLSRV();
     activity = select( max_sd+1, &readfds , NULL , NULL , &timeout);
     //           ,-----'           |          |      |
     //           |   ,-------------'          |      |
     //           |   | ,----------------------'      |
     //           |   | | ,---------------------------'
     //           |   | | |
     //           |   | | An optional pointer to a set of sockets to be checked for errors.
     //           |   | An optional pointer to a set of sockets to be checked for writability.
     //           |   An optional pointer to a set of sockets to be checked for readability.
     //           |
     //           Ignored by Winsock ("nfds" is a UNIX thing, something like
     //           "the highest dont-know-what (SOCKET? HANDLE? INDEX?) PLUS ONE")
     HERE_I_AM__HLSRV();
     // > The select function returns the total number of socket handles
     // > that are ready and contained in the fd_set structures,
     // > zero if the time limit expired, or SOCKET_ERROR
     // > if an error occurred. If the return value is SOCKET_ERROR,
     // > WSAGetLastError can be used to retrieve a specific error code.
     // > A set of macros is provided for manipulating an fd_set structure.
     // > These macros are compatible with those used in the Berkeley software,
     // > but the underlying representation is completely different.
     // >
     // > The parameter readfds identifies the sockets that are to be checked
     // > for readability. If the socket is currently in the listen state,
     // > it will be marked as readable if an incoming connection request
     // > has been received such that an accept is guaranteed to complete
     // > without blocking. For other sockets, readability means that queued
     // > data is available for reading such that a call to recv, WSARecv,
     // > WSARecvFrom, or recvfrom is guaranteed not to block.
     if( activity == 0 )  // > "zero if the time limit expired"...
      { // .. which means NOTHING HAPPENED on any socket;
        //    so don't waste time checking FD_ISSET() below,
        //    just loop around and sleep for another 20 ms in select() .
        //    (only in THIS particular case, don't forget to POLL all
        //     currently opened sockets; maybe some of them have something
        //     TO SEND; even without having received anything.
        //     Mr. Nagle will possibly be not amused; but there's not much
        //     we can do (without digging in deeply) to make him cheer up) :
        HERE_I_AM__HLSRV();
        for( iClientIndex=0; iClientIndex<HLSRV_MAX_CLIENTS; iClientIndex++)
         {
           s = client_socket[iClientIndex];
           // With 'activity==0', we know NONE of the sockets is readable,
           //      so don't check for RECEPTION - only "poll" for TRANSMISSION:
           if( s != INVALID_SOCKET )
            { // here's a VALID socket, so poll the application for TRANSMISSION :

              pClient = &pServer->Client[iClientIndex];
              pServer->nBytesInTxBuffer = 0; // nothing appended to pServer->bTxBuffer[ pServer->nBytesInTxBuffer++ ] yet ..
                  // (Note: There is only ONE COMMON pServer->bTxBuffer,
                  //        used for MULTIPLE remote clients in this loop !
                  //        Thus, anything that HLSRV_OnPoll() may allocate
                  //        and assemble for transmission via HLSRV_AllocBytesInTxBuffer()
                  //        MUST be sent() further below.
              HLSRV_OnPoll( pServer, pClient );
              if( pServer->nBytesInTxBuffer > 0 ) // IF there's anything to send..
               { // send whatever HLSRV_OnPoll() may have
                 // appended to pServer->bTxBuffer[ pServer->nBytesInTxBuffer++ ],
                 // with a maximum of <HLSRV_SOCKET_TX_BUFFER_SIZE> :
                 if( send(s, (char*)pServer->bTxBuffer, pServer->nBytesInTxBuffer, 0)
                                             != pServer->nBytesInTxBuffer )
                  { pServer->pszLastError = "HLServerThread failed to 'send()' after 'poll'";
                    ShowError( ERROR_CLASS_ERROR | SHOW_ERROR_TIMESTAMP, pServer->pszLastError );
                  }
                 else // successfully SENT something, so COUNT this (for statistics):
                  { pServer->dwNumBytesSent  += pServer->nBytesInTxBuffer; // .. for ALL clients (entire server)
                    pClient->dwNumBytesSent += pServer->nBytesInTxBuffer; // .. and PER CONNECTION (socket)
                    HLSRV_FeedActivityWatchdog( pClient ); // feed the dog .. here: after successful send()
                    if( pServer->cfg.iDiagnosticFlags & HLSRV_DIAG_FLAGS_SHOW_NETWORK_TRAFFIC ) // beware, a "CPU hog"..
                     { INET_DumpTraffic_HexOrASCII( sz1kDump,500, pServer->bTxBuffer, pServer->nBytesInTxBuffer );
                       ShowError( ERROR_CLASS_TX_TRAFFIC | SHOW_ERROR_TIMESTAMP, "TX%d %s",
                                   (int)iClientIndex, sz1kDump );
                     }
                    pServer->nBytesInRxBuffer = 0;
                  }
               } // end if < HLSRV_OnPoll() deposited something in the TX-buffer >
            }   // end if < valid socket for POLLING *without* previous reception >
         }     // end for < iClientIndex .. >
      } // end if < activity==0, which means NO ERROR but NO RECEPTION on any socket >
     else if (activity < 0) // > "SOCKET_ERROR (-1) if an error occurred"...
      { // select() failed.  "This is a permanent error" ... close the socket, give up.
        pServer->pszLastError = "ServerThread terminated after 'select()' failed";
        pServer->iThreadStatus = HLSRV_THREAD_STATUS_TERMINATED; // flag for the application
        if( pServer->cfg.iDiagnosticFlags & HLSRV_DIAG_FLAGS_VERBOSE_OUTPUT )
         { ShowError( ERROR_CLASS_ERROR | SHOW_ERROR_TIMESTAMP, pServer->pszLastError );
         }
        HERE_I_AM__HLSRV();
        return -1; // farewell, ServerThred() ...
      }
     else // neither "timeout" in select() or a "complete fail" -> SOMETHING must have happened !
      { // If something happened on the master socket , then its an incoming connection
        if (FD_ISSET(master , &readfds))
         { SOCKET new_socket;
           // Even a WEB BROWSER can be used to test this "locally":
           //  In Firefox, enter "localhost:4532" in the address field.
           //  The Fox tries to be smart and replaces this by something like
           //                    "http://localhost:4532/" - oh well...
           //  (we don't speak HTTP but 'raw TCP', but anyway, good for a test)
           addrlen = sizeof(struct sockaddr_in);
           HERE_I_AM__HLSRV();
           new_socket = accept(master , (struct sockaddr *)&address, (int *)&addrlen);
           HERE_I_AM__HLSRV();
           if( (new_socket != INVALID_SOCKET )  // Only "let him in" if he's not a bad guy ..
             &&(!INET_IsBlockedIPv4Address( &pServer->sBlacklist, &address.sin_addr.S_un.S_un_b.s_b1 ) )
             )
            { int i;
              // accept() successful ... we've got a NEW, "accepted" socket ->
              // > inform user of socket number - used in send and receive commands
              // ex: printf("New connection , socket fd is %d , ip is : %s , port : %d \n" , new_socket , inet_ntoa(address.sin_addr) , ntohs(address.sin_port));

              // add new socket to array of sockets ..
              iClientIndex = -1;
              for( i=0; i<HLSRV_MAX_CLIENTS; i++)
               {
                 if( (client_socket[i] == INVALID_SOCKET) && (iClientIndex<0) )
                  { // Bingo, found the first UNUSED entry, so use it for the new client:
                    client_socket[i] = new_socket;
                    iClientIndex = i;
                    new_socket = INVALID_SOCKET; // "moved" the socket handle into client_socket[] so forget 'new_socket'
                    break;
                  }
               } // end for < add new socket to array of sockets >
              fUpdateStatistics = TRUE; // update pServer->nRemoteClientsConnected a.s.a.p.
              if( iClientIndex >= 0 ) // Retrieve some info about the new peer, then say hello...
               { pClient = &pServer->Client[iClientIndex];
                 // Extract what we need to know about the remote client
                 //   in less exotic structs than "struct sockaddr_in", etc,
                 //   and (even more important) in THE HOST'S byte order :
                 memset( (void*)pClient, 0, sizeof(T_HLSrvClient) ); // clear counter, state, etc
                 pClient->b4HisIP.b[0] = address.sin_addr.S_un.S_un_b.s_b1; // "first byte" of HIS IP-v4-address
                 pClient->b4HisIP.b[1] = address.sin_addr.S_un.S_un_b.s_b2; // ..
                 pClient->b4HisIP.b[2] = address.sin_addr.S_un.S_un_b.s_b3;
                 pClient->b4HisIP.b[3] = address.sin_addr.S_un.S_un_b.s_b4; // "fourth byte" of HIS IP-v4-address
                 pClient->iHisPort   = ntohs(address.sin_port); // in this case, "his" port is an EPHEMERAL port number
                 HERE_I_AM__HLSRV();
                 HLSRV_OnConnect( pServer, pClient ); // WE (server) have been connected by a remote client..
                 HERE_I_AM__HLSRV();
#               if(0) // do NOT start sending from this end .. the other guy may be a WEB BROWSER or a bad guy
                 if( pServer->nBytesInTxBuffer > 0 ) // *IF* there's a response to send..
                  { // ... send the 'initial greeting' from the application-specific "OnConnect"-method:
                    if( send(new_socket, pServer->bTxBuffer, pServer->nBytesInTxBuffer, 0)
                                                         != pServer->nBytesInTxBuffer )
                     { pServer->pszLastError = "ServerThread failed to 'greet' a new client";
                     }
                    else // successfully SENT something, so COUNT this (for statistics):
                     { pServer->dwNumBytesSent  += pServer->nBytesInTxBuffer;
                       pClient->dwNumBytesSent += pServer->nBytesInTxBuffer;
                       HLSRV_FeedActivityWatchdog( pClient ); // feed the dog .. here: after successful send()
                       if( pServer->cfg.iDiagnosticFlags & HLSRV_DIAG_FLAGS_SHOW_NETWORK_TRAFFIC ) // beware, a "CPU hog"..
                        { INET_DumpTraffic_HexOrASCII( sz1kDump,500, pServer->bTxBuffer, pServer->nBytesInTxBuffer );
                          ShowError( ERROR_CLASS_TX_TRAFFIC | SHOW_ERROR_TIMESTAMP, "TX%d %s",
                                      (int)iClientIndex, sz1kDump );
                        }
                       pServer->nBytesInTxBuffer = 0;
                     }
                  }
#                endif // send something "immediately" after being connected by a client ? No.
               } // end if( iNewClientIndex >= 0 )
            }   // end if < new (accepted) socket VALID > ?
           if( new_socket != INVALID_SOCKET ) // not moved into the array or closed yet ?
            { closesocket( new_socket );
            }
           // end of the existance of 'SOCKET new_socket;'
         } // end if (FD_ISSET(master , &readfds)) ..

        // ..else its some IO operation on some other socket :)
        for( iClientIndex=0; iClientIndex<HLSRV_MAX_CLIENTS; iClientIndex++)
         {
           s = client_socket[iClientIndex];
           pClient = &pServer->Client[iClientIndex];

           pServer->nBytesInTxBuffer = 0; // nothing appended to pServer->bTxBuffer[ pServer->nBytesInTxBuffer++ ]
           // for THIS remote client in THIS server-thread-loop yet ...

           // If the client is present in the 'readable sockets',  READ (call "recv()") !
           if( (s != INVALID_SOCKET) &&  (FD_ISSET( s , &readfds)) )
            {
              // get details of the client (geek name = "peer" = "the guy at the other end of the socket")
              addrlen = sizeof(struct sockaddr_in);
              getpeername(s , (struct sockaddr*)&address , (int*)&addrlen);

              // Check if it was for closing, and also read the incoming message
              HERE_I_AM__HLSRV();
              valread = recv( s, (char*)pServer->bRxBuffer, HLSRV_SOCKET_RX_BUFFER_SIZE, 0);
              // ,---------------------------------------------------------------'
              // '--> From the Winsock specification (about the last argument):
              // > A set of flags that influences the behavior of this function.
              // Zero if we don't want to "peek" (read without removing from the
              // network buffer), don't want "Out Of Band"-data, etc etc.
              if( valread == SOCKET_ERROR)
               {
                 int error_code = WSAGetLastError();
                 if(error_code == WSAECONNRESET)
                  { HERE_I_AM__HLSRV();
                    // Somebody disconnected , get his details and print
                    // ex: printf("Host disconnected unexpectedly , ip %s , port %d \n" , inet_ntoa(address.sin_addr) , ntohs(address.sin_port));
                    HLSRV_OnDisconnect( pServer, pClient );

                    // Close the socket and mark as 0 in list for reuse
                    closesocket( s );
                    client_socket[iClientIndex] = INVALID_SOCKET;
                    fUpdateStatistics = TRUE; // update pServer->nRemoteClientsConnected a.s.a.p.
                  }
                 else // "SOCKET_ERROR" but not "WSAECONNRESET" ?
                  { HERE_I_AM__HLSRV();
                    if( pServer->cfg.iDiagnosticFlags & HLSRV_DIAG_FLAGS_VERBOSE_OUTPUT )
                     { ShowError( ERROR_CLASS_INFO | SHOW_ERROR_TIMESTAMP, "Recv failed, IP %s (%s)",
                        HLSRV_IPv4AddressToString( (BYTE*)&address.sin_addr.S_un.S_un_b.s_b1 ),
                        INET_WinsockErrorCodeToString( error_code ) );
                     }
                  }
               } // end if recv() -> SOCKET_ERROR ?
              if ( valread == 0)
               {
                 // Somebody disconnected , get his details and print
                 // ex: printf("Host disconnected , ip %s , port %d \n" , inet_ntoa(address.sin_addr) , ntohs(address.sin_port));
                 HERE_I_AM__HLSRV();
                 HLSRV_OnDisconnect( pServer, pClient );

                 // Close the socket and mark it's handle as INVALID in list, for reuse
                 closesocket( s );
                 client_socket[iClientIndex] = INVALID_SOCKET;
                 fUpdateStatistics = TRUE; // update pServer->nRemoteClientsConnected a.s.a.p.
               }
              else // something received on this client socket ->  pass it on...
               { pServer->nBytesInRxBuffer = valread;
                 if( pServer->nBytesInRxBuffer > pServer->iPeakRxBufferUsage )
                  {  pServer->iPeakRxBufferUsage = pServer->nBytesInRxBuffer;
                  }
                 if( valread > 0 )
                  { pServer->dwNumBytesRcvd += valread; // here: inc'd in ServerThread()
                  }
                 // When tested with a WEB BROWSER (to replace the not-yet-functional
                 // "Remote CW"-client), got here with valread above 400 bytes
                 //      (!  ... getting more with each browser update ... ),
                 //   and pServer->bRxBuffer = "GET / HTTP/1.1\r\n" ...
                 //
                 // To make Nagle's Algorithm happy, it's important
                 // to call send() only ONCE(!) after a successfull recv() .
                 // So here's THE ONLY PLACE where we pass the received segment
                 // to the application, and let some subroutine decide if,
                 // and what to send "as a response" (from server to client):
                 // add null character, if you want to use with printf/puts or other string handling functions
                 HERE_I_AM__HLSRV();
                 HLSRV_OnReceive( pServer, pClient, pServer->bRxBuffer, pServer->nBytesInRxBuffer );
                 HERE_I_AM__HLSRV();
               }
            } // end if (FD_ISSET( s , &readfds) )

           // Regardless if something was received on this socket or not,
           //  poll for TRANSMISSION (here on the SERVER SIDE, possibly for MULTIPLE sockets):
           if( s != INVALID_SOCKET )  // remember, we're still in the "for-all-remote-clients"-loop..
            { HERE_I_AM__HLSRV();
              HLSRV_OnPoll( pServer, pClient );
              HERE_I_AM__HLSRV();
              if( pServer->nBytesInTxBuffer > 0 ) // IF there's anything to send..
               { // send whatever all those functions called above may have
                 // appended to pServer->bTxBuffer[ pServer->nBytesInTxBuffer++ ],
                 // with a maximum of <HLSRV_SOCKET_TX_BUFFER_SIZE> in bytes.
                 if( send( s, (const char*)pServer->bTxBuffer, pServer->nBytesInTxBuffer, 0)
                                              != pServer->nBytesInTxBuffer )
                  { pServer->pszLastError = "ServerThread failed to 'send()'";
                  }
                 else // successfully SENT something, so COUNT this (for statistics):
                  { pServer->dwNumBytesSent  += pServer->nBytesInTxBuffer;
                    pClient->dwNumBytesSent += pServer->nBytesInTxBuffer;
                    HLSRV_FeedActivityWatchdog( pClient ); // feed the dog .. here: after successful send()
                    if( pServer->cfg.iDiagnosticFlags & HLSRV_DIAG_FLAGS_SHOW_NETWORK_TRAFFIC ) // beware, a "CPU hog"..
                     { INET_DumpTraffic_HexOrASCII( sz1kDump,500, pServer->bTxBuffer, pServer->nBytesInTxBuffer );
                       ShowError( ERROR_CLASS_TX_TRAFFIC | SHOW_ERROR_TIMESTAMP, "TX%d %s",
                                   (int)iClientIndex, sz1kDump );
                     }
                    pServer->nBytesInTxBuffer = 0;
                  }
               } // end if < something to send after HLSRV_OnPoll() >
              if( pClient->fDisconnect ) // anything called above decided to DISCONNECT :
               { // (Note: Without this, an HTTP response with an error was NOT displayed
                 //        in a web browser, as if NOTHING had been sent from here at all.
                 //  The myths of HTTP and "modern browsers" ... see HttpServer.c )
                 s = client_socket[iClientIndex];  // socket handle still valid ?
                 if( s != INVALID_SOCKET )
                  { // Close the socket and mark as 0 in list for reuse
                    closesocket( s );
                    client_socket[iClientIndex] = INVALID_SOCKET;
                    fUpdateStatistics = TRUE; // update pServer->nRemoteClientsConnected a.s.a.p.
                  }
                 pClient->fDisconnect = FALSE; // "done" (have disconnected and closed the socket)
                 pClient->iClientState= HLSRV_CLIENT_STATE_DISCONN;
               }
            } // end if( pClient->fDisconnect )
         }   // end for < iClientIndex .. >
      }     // end else <neither "timeout"  or a "complete fail" in select() >

     if( fUpdateStatistics ) // update "statistics" (mostly for the GUI) ?
      { for( n=iClientIndex=0; iClientIndex<HLSRV_MAX_CLIENTS; iClientIndex++)
         { if( client_socket[iClientIndex] != INVALID_SOCKET )
            { ++n;
            }
         }
        pServer->nRemoteClientsConnected = n; // <- for "diagnostics", read in another thread
        fUpdateStatistics = FALSE; // "done"
      } // end if( fUpdateStatistics )
   }   // end while < server-thread shall run >

  HERE_I_AM__HLSRV();
  for(iClientIndex=0; iClientIndex<HLSRV_MAX_CLIENTS; iClientIndex++)
   { if( client_socket[iClientIndex] != INVALID_SOCKET )
      { closesocket( client_socket[iClientIndex] );
      }
   }

  if( master != INVALID_SOCKET )
   { closesocket( master );
   }
  if( pServer->cfg.iDiagnosticFlags & HLSRV_DIAG_FLAGS_VERBOSE_OUTPUT )
   { ShowError( ERROR_CLASS_INFO | SHOW_ERROR_TIMESTAMP, "HamlibSrv: Disconnected and thread terminated" );
   }

  HERE_I_AM__HLSRV();
  return 0;
} // end ServerThread()

//---------------------------------------------------------------------------
// Functions invoked via FUNCTION POINTER after recognizing a Hamlib 'command'
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
static int HLSRV_OnGetFreq(T_HLSrv *pServer, T_HLSrvClient *pClient,
                           T_HLSRV_CmdTabEntry *pCmdInfo, const char *cpCmdArgs )
{
  // From www.mankier.com/1/rigctl (same at hamlib.sourceforge.net/html/rigctld.1.html) :
  // > f, get_freq
  // >  Get 'Frequency', in Hz.
  // >  Returns an integer value and the VFO hamlib thinks is active.
  // >  Note that some rigs (e.g. all Icoms) cannot track current VFO
  // >  so hamlib can get out of sync with the rig if the user presses
  // >  rig buttons like the VFO.
  // Seen in the Wireshark dump (between WSJT-X and wfview's rigctld-emulation):
  //   Command:   "f\n"          // short form of "what's the frequency" ?
  //   Response:  "144300000\n"  // here is our frequency in Hertz
  if( pServer->pRigControl != NULL )
   { HLSRV_PrintToResponse( pServer, pClient, "%ld\n", (long)pServer->pRigControl->dblVfoFrequency );
   }
  else // The real rig control module (RigControl.c, not "Hamlib") isn't available ->
   { HLSRV_PrintToResponse( pServer, pClient, "0\n" );
   }
  return HLSRV_RESULT_NO_ERROR;
} // end HLSRV_OnGetFreq()

//---------------------------------------------------------------------------
static int HLSRV_OnSetFreq(T_HLSrv *pServer, T_HLSrvClient *pClient,
                           T_HLSRV_CmdTabEntry *pCmdInfo, const char *cpCmdArgs )
{
  return HLSRV_RESULT_NO_ERROR;
} // end HLSRV_OnSetFreq()

//---------------------------------------------------------------------------
static int HLSRV_OnGetMode(T_HLSrv *pServer, T_HLSrvClient *pClient,
                           T_HLSRV_CmdTabEntry *pCmdInfo, const char *cpCmdArgs )
{
  return HLSRV_RESULT_NO_ERROR;
} // end HLSRV_OnGetMode()

//---------------------------------------------------------------------------
static int HLSRV_OnSetMode(T_HLSrv *pServer, T_HLSrvClient *pClient,
                           T_HLSRV_CmdTabEntry *pCmdInfo, const char *cpCmdArgs )
{
  return HLSRV_RESULT_NO_ERROR;
} // end HLSRV_OnSetMode()

//---------------------------------------------------------------------------
static int HLSRV_OnGetVFO( T_HLSrv *pServer, T_HLSrvClient *pClient,
                           T_HLSRV_CmdTabEntry *pCmdInfo, const char *cpCmdArgs )
{
  // From www.mankier.com/1/rigctl (same at hamlib.sourceforge.net/html/rigctld.1.html) :
  // > v, get_vfo
  // >  Get current 'VFO'.
  // >  Returns VFO as a token as in set_vfo above.
  // Seen in the Wireshark dump (between WSJT-X and wfview's rigctld-emulation):
  //   Command:   "v\n"   (short form of "get_vfo")
  //   Response:  "VFOA"  ("the current VFO is VFO A" or something in that style)
  HLSRV_PrintToResponse( pServer, pClient, "VFOA\n" );
  return HLSRV_RESULT_NO_ERROR;
} // end HLSRV_OnGetVFO()

//---------------------------------------------------------------------------
static int HLSRV_OnSetVFO( T_HLSrv *pServer, T_HLSrvClient *pClient,
                           T_HLSRV_CmdTabEntry *pCmdInfo, const char *cpCmdArgs )
{
  return HLSRV_RESULT_NO_ERROR;
} // end HLSRV_OnSetVFO()


//---------------------------------------------------------------------------
static int HLSRV_OnGetRIT( T_HLSrv *pServer, T_HLSrvClient *pClient,
                           T_HLSRV_CmdTabEntry *pCmdInfo, const char *cpCmdArgs )
{
  return HLSRV_RESULT_NO_ERROR;
} // end HLSRV_OnGetRIT()

//---------------------------------------------------------------------------
static int HLSRV_OnSetRIT( T_HLSrv *pServer, T_HLSrvClient *pClient,
                           T_HLSRV_CmdTabEntry *pCmdInfo, const char *cpCmdArgs )
{
  return HLSRV_RESULT_NO_ERROR;
} // end HLSRV_OnSetRIT()

//---------------------------------------------------------------------------
static int HLSRV_OnGetSplitVFO(T_HLSrv *pServer,T_HLSrvClient *pClient,
                           T_HLSRV_CmdTabEntry *pCmdInfo, const char *cpCmdArgs )
{
  // > s, get_split_vfo
  // >  Get 'Split' mode.
  // >  Split is either 0 = Normal   or   1 = Split.  (an IC-9700 knows other split modes; see RigControl.c : cmd = 0x0F ! )
  // >  Get 'TX VFO'.
  // >  TX VFO is a token as in set_split_vfo .
  // Seen in the Wireshark dump (between WSJT-X and wfview's rigctld-emulation):
  //   Command:   "s\n" (short form of "get_split_vfo")
  //   Response:  "0"   ("the current VFO is NOT in split mode" or something in that style)
  //   (2nd line) "VFOB" (if the current VFO was using split mode, the TX VFO would be "VFOB")
  //
  long i32 = RigCtrl_GetParamByPN_Int( pServer->pRigControl, RIGCTRL_PN_SPLIT_MODE);
    // 2024-06-30 : Got here with i32 = 0x80000001 = -2147483647 = RIGCTL_NOVALUE_INT,
    //              because 'split mode' wasn't one of the parameters polled
    //              from the radio after the initial connect. The same problem
    //              would occur with other 'exotic' parameters that only
    //              the hamlib clients in some 3rd party software would use.
    //  Fixed by reading (or at least TRYING TO read) those parameters at least
    //  once, in Remote_CW_Keyer/RigControl.c : RigCtrl_Handler() /
  HLSRV_PrintToResponse( pServer, pClient, "%d\nVFOB\n", (int)i32 );

  return HLSRV_RESULT_NO_ERROR;
} // end HLSRV_OnGetSplitVFO()

//---------------------------------------------------------------------------
static int HLSRV_OnSetSplitVFO(T_HLSrv *pServer,T_HLSrvClient *pClient,
                           T_HLSRV_CmdTabEntry *pCmdInfo, const char *cpCmdArgs )
{ // > S, set_split_vfo 'Split' 'TX VFO'
  // >  Set 'Split' mode.
  // >  Split is either 0 = Normal or 1 = Split.
  // >
  // >  Set 'TX VFO'.
  // >  TX VFO is a token: VFOA, VFOB, VFOC, currVFO,
  // >                     VFO, MEM, Main, Sub, TX, RX .
  int iSplitMode = SL_ParseInteger( &cpCmdArgs );
  RigCtrl_SetParamByPN_Int( pServer->pRigControl, RIGCTRL_PN_SPLIT_MODE, iSplitMode );
  // The 2nd argument, "TX VFO", is ignored here.
  return HLSRV_RESULT_NO_ERROR;

} // end HLSRV_OnSetSplitVFO()

//---------------------------------------------------------------------------
static int HLSRV_OnGetXIT( T_HLSrv *pServer, T_HLSrvClient *pClient,
                           T_HLSRV_CmdTabEntry *pCmdInfo, const char *cpCmdArgs )
{
  return HLSRV_RESULT_NO_ERROR;
} // end HLSRV_OnGetXIT()

//---------------------------------------------------------------------------
static int HLSRV_OnSetXIT( T_HLSrv *pServer, T_HLSrvClient *pClient,
                           T_HLSRV_CmdTabEntry *pCmdInfo, const char *cpCmdArgs )
{
  return HLSRV_RESULT_NO_ERROR;
} // end HLSRV_OnSetXIT()

//---------------------------------------------------------------------------
static int HLSRV_OnGetPTT( T_HLSrv *pServer, T_HLSrvClient *pClient,
                           T_HLSRV_CmdTabEntry *pCmdInfo, const char *cpCmdArgs )
{
  // > "t, get_ptt"        (from www.mankier.com/1/rigctl anno 2024)
  // >   Get 'PTT' status.
  // >   Returns PTT as a value in set_ptt .
  long i32 = RigCtrl_GetParamByPN_Int( pServer->pRigControl, RIGCTRL_PN_TRANSMITTING);
    // '-->  0=receiving, 1=transmitting, or RIGCTL_NOVALUE_INT when NOT AVAILABLE
  if( i32==RIGCTL_NOVALUE_INT ) // if PTT status isn't *available*, indicate RECEPTION
   {  i32 = 0;
   }
  HLSRV_PrintToResponse( pServer, pClient, "%d\n", (int)i32 );
  // Note: In Hamlib, the 'PTT' status value may be
  //       0 = RX,  1 =  TX,  2 = "TX mic", or 3 = "TX data".
  //       HERE, we don't care if we're 'transmitting a microphone',
  //                                    'transmitting data',
  //                                    or transmitting Morse code.

  return HLSRV_RESULT_NO_ERROR;
} // end HLSRV_OnGetPTT()

//---------------------------------------------------------------------------
static int HLSRV_OnSetPTT( T_HLSrv *pServer, T_HLSrvClient *pClient,
                           T_HLSRV_CmdTabEntry *pCmdInfo, const char *cpCmdArgs )
{
  // > "T, set_ptt 'PTT'"  (from www.mankier.com/1/rigctl anno 2024)
  // >   Set 'PTT'.
  // >   PTT is a value: '0' (RX), '1' (TX), '2' (TX mic), or '3' (TX data).
  return HLSRV_RESULT_NO_ERROR;
} // end HLSRV_OnSetPTT()

//---------------------------------------------------------------------------
static int HLSRV_OnDumpState(T_HLSrv *pServer, T_HLSrvClient *pClient,
                           T_HLSRV_CmdTabEntry *pCmdInfo, const char *cpCmdArgs )
  // Implements Hamlib's "dump_state" command (assembles a response).
  // From hamlib.sourceforge.net/html/rigctld.1.html :
  // > Return certain state information about the radio backend.
  // A-ha. But what's the format we're expected to send ? DOCUMENTATION ? ? ?
  // We can only GUESS the meanings of all lines in the response from e.g.
  //     Hamlib-4.5.5/tests/rigctl_parse.c, but it's too obfuscated.
{
  T_RigCtrl_RadioInfo *pRadioInfo = NULL;
  T_HLSRV_FreqRange *pFR;
  int i, j, n;

  if( pServer->pRigControl != NULL )
   { pRadioInfo = RigCtrl_GetRadioInfoByDefaultAddress(
                    pServer->pRigControl->iRadioCtrlProtocol,
                    pServer->pRigControl->iDefaultAddress );
   }
  HLSRV_PrintToResponse( pServer, pClient, "1\n" ); // (1) rigctld protocol version (?)
  HLSRV_PrintToResponse( pServer, pClient, "2\n" ); // (2) rigctld "rig_model", but as a DECIMAL CODE, not a proper name like e.g. "IC-7300" !
          // Not sure if "rig_model" is the same as the "radio number".
          // If it is, then "radio model 2" would indicate a "NET rigtcl".
          // But when talking to an IC-9700, wfview reported '3081' which seems
          // to be Hamlib's 'rigctrModel' for an IC-9700
          // (see commented Wireshark dump in Remote_CW_Keyer/Hamlib_traffic_example.txt)
  HLSRV_PrintToResponse( pServer, pClient, "0\n" ); // (3) rigctld "rig state itu region" (forget it, just a dummy)

  // Emitted next:
  //  (4) RX frequency range lists with start- and end frequencies, modes, powers, "vfo"(?), "ant"(?), ...
  //  (5) a string with "0 0 0 0 0 0 0\n" that possibly marks the end of the above list
  //  (6) TX frequency range lists with start- and end frequencies, modes, powers, "vfo"(?), "ant"(?), ...
  //  (7) another unexplained string with "0 0 0 0 0 0 0\n"
  //      ( to understand what's in these what Hamlib calls "range_list" items,
  //        see hamlib.../src/rig.c : rig_init(), grep for "rx_range_list1" and "tx_range_list1" .
  //        All these "range lists" are an array of 'freq_range_t' types,
  //        with the struct defined in K:\Hamlib\SOURCE\include\hamlib\rig.h .
  //        Our equivalent is T_HLSRV_FreqRange in HamlibServer.h )
  //  (8) a list of tuning steps with "modes" and "ts"
  //  (9) an unexplained string with "0 0\n" which probably marks the end of the above "ts" list
  //  (10) a list of filters(?) with "modes" and "width"
  //  (11) another unexplained string with "0 0\n" which probably marks the end of the above "flt" list
  //  (12) a list of "channels" that seems to have been removed in newer versions of Hamlib
  //  (13) a line with "max_rit" (without a token, just a decimal number, bleah..)
  //  (14) a line with "max_xit"
  //  (15) a line with "max_ifshift"
  //  (16) a line with "announces"    (whatever that is, no token, just a decimal number)
  //  (17) a line with "preamp[]"     (array, also without a token, only decimal stuff)
  //  (18) a line with "attenuator[]" (array, also without a token, only decimal stuff)
  //  (19..24) lines with decimal values for "has_get_func", "has_set_func",
  //          "has_get_level", "has_set_level", "has_get_parm", "has_set_parm"
  //  (25) plus much more parser-friendly stuff with key=value pairs ... hooray
  //  (26) plus a final line with "done\n", as seen in Hamlib_traffic_example.txt (Wireshark TCP dump).
  //       Seems to be generated in rigctl_parse.c(!), "declare_proto_rig(dump_state)" .
  //
  // Unfortuntely, without the above stuff, a remote Hamlib client was unhappy,
  // and whined (in this case, WSJT-X) :
  // > Hamlib error: 0000    31 0a                                               1.
  // > read_string_generic called, rxmax=1024 direct=1, expected_len=1
  // > read_string_generic(): RX 2 characters, direct=1
  // > 0000    32 0a                                               2.
  // > read_string_generic called, rxmax=1024 direct=1, expected_len=1
  // > read_string_generic(): RX 2 characters, direct=1
  // > 0000    30 0a                                               0.
  // > read_string_generic called, rxmax=1024 direct=1, expected_len=1
  // > read_string_generic(): RX 7 characters, direct=1
  // > 0000    52 50 52 54 20 30 0a                                RPRT 0.
  // >   2:netrigctl.c(350):netrigctl_open returning(-8) Protocol error
  // Tried to understand this mumbo-jumbo. From our server's point of view,
  // the 'Hamlib compatible traffic' was displayed as follows:
  // > 09:13:24.6 HamlibSrv: Accepted connection from 127.0.0.1:58321 (no data yet)
  // > 09:13:24.6 HamlibSrv0: connected by 127.0.0.1:58321
  // > 09:13:24.6 HLSrv0 [0009] \chk_vfo\n           [coloured green = received]
  // > 09:13:24.6 TX0 [0009] CHKVFO 0\n              [coloured yellow = sent]
  // > 09:13:24.6 HLSrv0 [000C] \dump_state\n        [coloured green = received]
  // > 09:13:24.6 TX0 [000D] 1\n2\n0\nRPRT 0\n       [coloured yellow = sent]
  // > 09:13:24.6 HamlibSrv: Disconnected by 127.0.0.1:58321  [obviously WSJT-X was unhappy..]
  // >         21 bytes rvcd, 22 sent.
  for(j=0; j<=1; ++j) // (4) RX frequency range list, (6) RX frequency range list (same format?)
   { for(i=0; i<HLSRV_NUM_RX_RANGES; ++i) //
      { if(j==0) pFR = &pServer->RxRangeList[i];
        else     pFR = &pServer->TxRangeList[i];
        if( pFR->dblEndFreq_Hz > pFR->dblStartFreq_Hz )
         { HLSRV_PrintToResponse( pServer, pClient, "%lf %lf 0x%lx%08lx %ld %ld 0x%lx 0x%lx\n",
            (double)pFR->dblStartFreq_Hz, // <-----'   |        |   |    |   |     |     |
            (double)pFR->dblEndFreq_Hz,   // <---------'        |   |    |   |     |     |
            (unsigned long)pFR->dwExtModes, // <-(upper 32 bit)-'   |    |   |     |     |
            (unsigned long)pFR->dwModes,    // <-(lower 32 bit)-----'    |   |     |     |
            (long)pFR->i32LowPower_mW,      // <-------------------------'   |     |     |
            (long)pFR->i32HighPower_mW,     // <-----------------------------'     |     |
            (unsigned long)pFR->dwVFO,      // <--(another 32-bit-combination)....-'     |
            (unsigned long)pFR->dwAnt );    // <--(another 32-bit-combination)-----------'
         }
        else // seems to be the END of this list-type (5, 7) :
         {
           break;
         }
      }
     HLSRV_PrintToResponse( pServer, pClient, "0 0 0 0 0 0 0\n" ); // (5),(7) SEVEN ZEROs to mark the end of the above list      
   }
  n = sizeof(HLSRV_i32TuningSteps_Hz) / sizeof(HLSRV_i32TuningSteps_Hz[0]);
  for(i=0; i<n; ++i)
   { pFR = &pServer->RxRangeList[ 0/*!*/ ];
     HLSRV_PrintToResponse( pServer, pClient, "0x%lx%08lx %ld\n",  // (8) "tuning steps"...
       (unsigned long)pFR->dwExtModes, // <-------'   |    |
       (unsigned long)pFR->dwModes,    // <-----------'    |
       (long)HLSRV_i32TuningSteps_Hz[i] ); // <------------'
   }
  HLSRV_PrintToResponse( pServer, pClient, "0 0\n" ); // (9) TWO ZEROs to mark the end of the above list

  // Next: Emit a list of "possible bandwidths" for each mode or group of modes.
  // Again, see example from a Wireshark session in Hamlib_traffic_example.txt !
  HLSRV_PrintToResponse( pServer, pClient, "0x%lx %ld\n",  // (10) "bandwidths" for USB,LSB (in the original Hamlib: "FLTLST")
       (unsigned long)( HLSRV_MODE_USB | HLSRV_MODE_LSB ), (long)3000 );
  HLSRV_PrintToResponse( pServer, pClient, "0x%lx %ld\n",
       (unsigned long)( HLSRV_MODE_USB | HLSRV_MODE_LSB ), (long)2400 );
  HLSRV_PrintToResponse( pServer, pClient, "0x%lx %ld\n",
       (unsigned long)( HLSRV_MODE_AM | HLSRV_MODE_AMS | HLSRV_MODE_SAM | HLSRV_MODE_PKTAM | HLSRV_MODE_AMN ),
       (long)9000 );  // 9 kHz filter for any kind of "AM"
  HLSRV_PrintToResponse( pServer, pClient, "0x%lx %ld\n",
       (unsigned long)( HLSRV_MODE_AM | HLSRV_MODE_AMS | HLSRV_MODE_SAM | HLSRV_MODE_PKTAM | HLSRV_MODE_AMN ),
       (long)6000 );  // 6 kHz filter for any kind of "AM"
  HLSRV_PrintToResponse( pServer, pClient, "0x%lx %ld\n",
       (unsigned long)( HLSRV_MODE_AM | HLSRV_MODE_AMS | HLSRV_MODE_SAM | HLSRV_MODE_PKTAM | HLSRV_MODE_AMN ),
       (long)9000 );  // 3 kHz filter for any kind of "AM"
  HLSRV_PrintToResponse( pServer, pClient, "0x%lx %ld\n",
       (unsigned long)( HLSRV_MODE_CW | HLSRV_MODE_CWR ), (long)1200 ); // 1200 Hz filter for any kind of CW
  HLSRV_PrintToResponse( pServer, pClient, "0x%lx %ld\n",
       (unsigned long)( HLSRV_MODE_CW | HLSRV_MODE_CWR ), (long)500 ); // 500 Hz filter for any kind of CW
  HLSRV_PrintToResponse( pServer, pClient, "0x%lx %ld\n",
       (unsigned long)( HLSRV_MODE_CW | HLSRV_MODE_CWR ), (long)200 ); // 200 Hz filter for any kind of CW
       // (modern Icom rigs offer much more than what can be encoded here,
       //  and the default "CWNN" filter = "Filter 3" is 250 Hz in an IC-7300, IC-9700, and many more)
  HLSRV_PrintToResponse( pServer, pClient, "0x%lx %ld\n",
       (unsigned long)( HLSRV_MODE_AM | HLSRV_MODE_AMS | HLSRV_MODE_SAM | HLSRV_MODE_PKTAM | HLSRV_MODE_AMN ),
       (long)9000 );  // 9 kHz filter for any kind of "AM"
  HLSRV_PrintToResponse( pServer, pClient, "0x%lx %ld\n",
       (unsigned long)( HLSRV_MODE_FM | HLSRV_MODE_WFM | HLSRV_MODE_FMN ),
       (long)15000 ); // 15 kHz AUDIO (not "IF filter") bandwidth for any kind of FM
  HLSRV_PrintToResponse( pServer, pClient, "0x%lx %ld\n",
       (unsigned long)( HLSRV_MODE_FM | HLSRV_MODE_WFM | HLSRV_MODE_FMN ),
       (long)10000 ); // 10 kHz bandwidth for narrow-ish FM
  HLSRV_PrintToResponse( pServer, pClient, "0x%lx %ld\n",
       (unsigned long)( HLSRV_MODE_FM | HLSRV_MODE_FMN ),
       (long)7000 ); // 7 kHz bandwidth for narrow FM (but now "Wide FM")
  HLSRV_PrintToResponse( pServer, pClient, "0x%lx %ld\n",
       (unsigned long)( HLSRV_MODE_RTTY | HLSRV_MODE_RTTYR ),
       (long)2400 ); // 2400 Hz bandwidth for don't-know-who-uses-this-wide RTTY
  HLSRV_PrintToResponse( pServer, pClient, "0x%lx %ld\n",
       (unsigned long)( HLSRV_MODE_RTTY | HLSRV_MODE_RTTYR ),
       (long)500 );  // 500 Hz bandwidth for any kind of RTTY
  HLSRV_PrintToResponse( pServer, pClient, "0x%lx %ld\n",
       (unsigned long)( HLSRV_MODE_RTTY | HLSRV_MODE_RTTYR ),
       (long)250 );  // 250 Hz bandwidth for any kind of RTTY
  HLSRV_PrintToResponse( pServer, pClient, "0 0\n" ); // (11) marker for the end of the above "filter bandwidths" list

  // Ex: Emit a list of "channels" (12) ... seems to have been removed from the "dump_state" response
  HLSRV_PrintToResponse( pServer, pClient, "%d\n", (int)9900 ); // (13) "max_rit" ?
  HLSRV_PrintToResponse( pServer, pClient, "%d\n", (int)9900 ); // (14) "max_xit" ?
  HLSRV_PrintToResponse( pServer, pClient, "%d\n", (int)10000); // (15) "max_ifshift" ?
  HLSRV_PrintToResponse( pServer, pClient, "%d\n", (int)0  ); // (16) "announces" (what the heck is that) ?
  HLSRV_PrintToResponse( pServer, pClient, "%d\n", (int)10 ); // (17) "preamp" ?
  HLSRV_PrintToResponse( pServer, pClient, "%d\n", (int)10 ); // (18) "attenuator" ?

  // Next: (19)..(24) = "has_get_func" .. "has_set_pam", with 64-bit-dummy-values because we really don't care
  for(i=19; i<=24; ++i)
   { HLSRV_PrintToResponse( pServer, pClient, "0xffffffffffffffff\n" );
   }
  // The rest (item types 25 to whatever) are glorious key=value pairs, not anonymous values):
  HLSRV_PrintToResponse( pServer, pClient,
      "vfo_ops=0xff\n"
      "ptt_type=0x1\n"
      "has_set_vfo=0x1\n"  // Actually, all these lines ...
      "has_get_vfo=0x1\n"  // have been brainlessly copied from a
      "has_set_freq=0x1\n" // Wireshark log ("follow TCP/IP stream") ..
      "has_get_freq=0x1\n" // - see Hamlib_traffic_example.txt !
      "has_set_conf=0x1\n"
      "has_get_conf=0x1\n"
      "has_power2mW=0x1\n"
      "has_mW2power=0x1\n"
      "timeout=0x3e8\n" // there's more than this, but WE DON'T CARE about e.g. "dcs_lists" and all that stuff
      "done\n" );       // <- seems to be a keyword (not specified anywhere), but Hamlib emits "done", too.

  // For a hypothetical rig with ALL AMATEUR RADIO BANDS from longwave to UHF,
  // the number of bytes "printed into the response" above (pClient->iResponseLength)
  // was ____, thus the buffer size (HLSRV_RESPONSE_MAX_LENGTH)
  // was increased to

  return HLSRV_RESULT_NO_ERROR;
} // end HLSRV_OnDumpState()

//---------------------------------------------------------------------------
static int HLSRV_OnCheckVFO( T_HLSrv *pServer, T_HLSrvClient *pClient,
                           T_HLSRV_CmdTabEntry *pCmdInfo, const char *cpCmdArgs )
  // When tested with WSJT-X as remote client, "chk_vfo" was the first received
  // command. Purpose ( from hamlib.sourceforge.net/html/rigctld.1.html ) :
  // > Returns "CHKVFO 1\n" (single line only) if rigctld was invoked
  // > with the -o/--vfo option and "CHKVFO 0\n" if not.
  // > When in VFO mode the client will need to pass 'VFO' as the first parameter
  // > to set or get commands. VFO is one of the strings defined in set_vfo above.
{
  HLSRV_PrintToResponse( pServer, pClient, "CHKVFO %d\n", (int)0 );
  return HLSRV_RESULT_NO_ERROR;
} // end HLSRV_OnCheckVFO()

//---------------------------------------------------------------------------
static int HLSRV_OnQuit( T_HLSrv *pServer, T_HLSrvClient *pClient,
                         T_HLSRV_CmdTabEntry *pCmdInfo, const char *cpCmdArgs )
  // Not sure if the long command form is really "exit rigctl"
  //     as listed at www.mankier.com/1/rigctl under "rigctl Command".
  //     No other command uses a space character (e.g. "set_freq", with underscore).
  // NOT listed as a valid command at hamlib.sourceforge.net/html/rigctld.1.html
  //     (under "COMMANDS") at all.
  // WSJT-X sent this command in the short form, so we probably need it.
  // > Q|q, exit rigctl
  // >  Exit rigctl in interactive mode.
  // >  When rigctl is controlling the rig directly, will close
  // >  the rig backend and port.  When rigctl is connected to rigctld
  // >  (radio model 2), the TCP/IP connection to rigctld is closed and
  // >  rigctld remains running, available for another TCP/IP network connection.
{
  pClient->fDisconnect = TRUE;
  return HLSRV_RESULT_NO_ERROR;
} // end HLSRV_OnQuit()


#if(0) // implement a Hamlib NET *client*, too ?
//---------------------------------------------------------------------------
static void HLSRV_CloseClientSocketAndStartWaitingForReconnect(
   T_HLSrv *pServer, SOCKET *pSocket ) // exclusively called from ClientThread() !
{
  if( *pSocket != INVALID_SOCKET )
   { closesocket( *pSocket );
     *pSocket = INVALID_SOCKET;
   }
  TIM_StartStopwatch( &pServer->swClientReconnTimer );
}

//---------------------------------------------------------------------------
static DWORD WINAPI ClientThread( LPVOID lpParam )
  // Our CLIENT THREAD .. more or less the 'remote counterpart' for ServerThread(),
  //                      but simpler because here's only ONE socket to manage.
{
  T_HLSrv *pServer = (T_HLSrv*)lpParam; // address of our instance data (struct)
  SOCKET  s = INVALID_SOCKET; // no NULL, not zero, but INVALID_SOCKET !
  int    iWSAerror;
  struct sockaddr_in server, address;
  int activity, addrlen, i, valread;
  TIMEVAL timeout; // funny thing with SECONDS and MICROSECONDS, for select()

  // set of socket descriptors
  fd_set readfds;
  int max_sd;   // "highest file descriptor number" for select() [Berkeley only, ignored by winsock]
  BOOL fOk;

  // Debugging stuff .. may kick this out one fine day:
  T_TIM_Stopwatch swLoopTime;
  char sz1kDump[1024];


  pServer->iThreadStatus = HLSRV_THREAD_STATUS_RUNNING; // cq cq, this is ServerThread() calling and standing by ..

  TIM_StartStopwatch( &pServer->swClientReconnTimer ); // .. for re-connect, etc

  // Loop to create a socket, connect, send and receive data ... here: CLIENT side
  while( pServer->iThreadStatus == HLSRV_THREAD_STATUS_RUNNING )
   {
     ++pServer->dwThreadLoops;  // Does this thread "sleep" most of the time, or is it a CPU hog ?
     pServer->dw8ThreadIntervals_us[ pServer->dwThreadLoops & 7 ] = TIM_ReadStopwatch_us( &swLoopTime );
     TIM_StartStopwatch( &swLoopTime );

     pServer->nBytesInTxBuffer = 0; // nothing appended to pServer->bTxBuffer[ pServer->nBytesInTxBuffer++ ]
                                   // IN THIS THREAD LOOP yet ...

     // Unlike many simple TCP CLIENT examples, we even try the "connect()"
     //  IN THE LOOP, so the client (-thread) has the chance to repeat his
     //  attempt to connect the remote server - for example, if the client
     //  application (instance) was launched BEFORE the server.
     //  Thus, here's another STATE MACHINE :
     switch( pServer->iLocalClientConnState )  // what's the next step ?
      { case HLSRV_CONN_STATE_OFF :        //  .. nothing yet, just WAIT ..
           break;
        case HLSRV_CONN_STATE_WAIT_RECONN: // waiting before trying to connect() again
           if( TIM_ReadStopwatch_ms( &pServer->swClientReconnTimer) > HLSRV_RECONN_INTERVAL_MS )
            { pServer->iLocalClientConnState = HLSRV_CONN_STATE_TRY_CONNECT;
            }
           break;
        default:
           break;
        case HLSRV_CONN_STATE_TRY_CONNECT:
           // Parse whatever we have now (domain name or numeric IP, plus port number):
           INET_LookupDomainOrParseIP( pServer->cfg.sz80ClientRemoteIP,
                 pServer->Client[0].b4HisIP.b, &pServer->Client[0].iHisPort );

           // Prepare the sockaddr_in structure ..
           server.sin_family = AF_INET;
           server.sin_addr.S_un.S_un_b.s_b1 = pServer->Client[0].b4HisIP.b[0];
           server.sin_addr.S_un.S_un_b.s_b2 = pServer->Client[0].b4HisIP.b[1];
           server.sin_addr.S_un.S_un_b.s_b3 = pServer->Client[0].b4HisIP.b[2];
           server.sin_addr.S_un.S_un_b.s_b4 = pServer->Client[0].b4HisIP.b[3];
           server.sin_port = htons( pServer->Client[0].iHisPort );

           // Create a socket if not done yet (a TCP client only ever needs ONE):
           if( s == INVALID_SOCKET )
            {  s = socket(AF_INET , SOCK_STREAM , 0 );
            }
           if( s == INVALID_SOCKET) // oops.. a hopeless case; bail out
            { // report this 'unrecoverable error' somewhere:
              sprintf( pServer->sz255LastError, "HLClientThread: could not create socket (%s)",
                        INET_WinsockErrorCodeToString( WSAGetLastError() ) );
              if( pServer->cfg.iDiagnosticFlags & HLSRV_DIAG_FLAGS_VERBOSE_OUTPUT )
               { ShowError( ERROR_CLASS_ERROR | SHOW_ERROR_TIMESTAMP, "%s", pServer->sz255LastError );
               }
              pServer->pszLastError = pServer->sz255LastError;
              pServer->iThreadStatus = HLSRV_THREAD_STATUS_TERMINATED; // flag for the application
              return -1; // farewell, ServerThred() ...
            }
           // Try to connect to the remote server:
           if( pServer->cfg.iDiagnosticFlags & HLSRV_DIAG_FLAGS_VERBOSE_OUTPUT )
            { ShowError( ERROR_CLASS_INFO | SHOW_ERROR_TIMESTAMP, "HamlibClient: Trying to connect %s:%d ...",
                         HLSRV_IPv4AddressToString(pServer->Client[0].b4HisIP.b),
                         (int)pServer->Client[0].iHisPort );
            }
           if( connect( s, (struct sockaddr *)&server, sizeof(server)) < 0 )
            { // failed to connect() our remote "CW" server ->
              iWSAerror = WSAGetLastError();  // bizarre API .. why not directly return an error code ?
              if( pServer->cfg.iDiagnosticFlags & HLSRV_DIAG_FLAGS_VERBOSE_OUTPUT )
               { ShowError( ERROR_CLASS_ERROR | SHOW_ERROR_TIMESTAMP, " .. failed: %s",
                            INET_WinsockErrorCodeToString( iWSAerror ) );
               }
              sprintf( pServer->sz255LastError, "ClientThread: could not connect (%s)",
                       INET_WinsockErrorCodeToString( iWSAerror ) );
              pServer->pszLastError = pServer->sz255LastError;
              // In THIS case, keep the thread running. Not being able to
              // connect() may be just a temporary hiccup. Try again a few
              // seconds later; maybe the network is back 'on-line' then.
              // Not sure if after a failed connect(), a socket can be recycled.
              // Assume it CANNOT, so:
              HLSRV_CloseClientSocketAndStartWaitingForReconnect( pServer, &s );
              //     '--> HLSRV_CONN_STATE_WAIT_RECONN
            }
           else // ok, connect() was successful, but that doesn't mean we're "logged in" ..
            {
              if( pServer->cfg.iDiagnosticFlags & HLSRV_DIAG_FLAGS_VERBOSE_OUTPUT )
               { ShowError( ERROR_CLASS_INFO | SHOW_ERROR_TIMESTAMP, " ..successfully connected to %s:%d",
                         HLSRV_IPv4AddressToString(pServer->Client[0].b4HisIP.b),
                         (int)pServer->Client[0].iHisPort );
               }
              // In the next state(s), we may actively "communicate",
              //  e.g. repeatedly call select() and possibly recv() further below.
              pServer->iLocalClientConnState = HLSRV_CONN_STATE_CONNECTED;
              // for the higher-level "client state machine" (with log-in, etc):
              pServer->Client[0].iClientState = HLSRV_CLIENT_STATE_CONNECTED;
              //  ,-------------------------------------------------'
              //  '--> "connected but not yet logged in" !
            }
           break;
        case HLSRV_CONN_STATE_CONNECTED :
           break;
      } // end switch( pServer->iLocalClientConnState )


     // 'Actively communicate' [send(), recv()] via the socket ?
     if(  ( s != INVALID_SOCKET )
        &&( pServer->iLocalClientConnState == HLSRV_CONN_STATE_CONNECTED )
       )
      {
        FD_ZERO(&readfds);  // similar as in the SERVER thread, prepare to select()..
        FD_SET(s, &readfds);
        // > FD_SET places sockets into a "set" for various purposes, such as
        // > testing a given socket for readability using the readfds parameter
        // > of the select function.

        // Prepare waiting for an activity on any of the sockets.
        // Also here (in the CLIENT thread), we don't want to wait endlessly
        // for RECEPTION, because we may have other plans, too ..
        timeout.tv_sec  = 0; // Pfui Deibel.. warum einfach, wenn's auch kompliziert geht !
        timeout.tv_usec = HLSRV_SOCKET_POLLING_INTERVAL_MS * 1000; // max time to wait in MICROSECONDS !
        activity = select( max_sd+1, &readfds , NULL , NULL , &timeout);
        // > The select function returns the total number of socket handles
        // > that are ready and contained in the fd_set structures,
        // > zero if the time limit expired, or SOCKET_ERROR
        // > if an error occurred. If the return value is SOCKET_ERROR,
        // > WSAGetLastError can be used to retrieve a specific error code. ...
        if( activity == 0 )  // > "zero if the time limit expired"...
         { // .. which means NOTHING HAPPENED on the socket,
           //    so don't waste time checking FD_ISSET(),
           //    just loop around and sleep for another 20 ms in select(),
           //    unless this client needs to *SEND* unsolicited data:
           HLSRV_OnPoll( pServer, &pServer->Client[HLSRV_LOCAL_CLIENT_INDEX] );
              //  '--> May deposit 'unsolicited' data in pServer->bTxBuffer !
           if( pServer->nBytesInTxBuffer > 0 ) // send "unsolicited data" (not "Nagle friendly") ?
            { if( send( s, pServer->bTxBuffer, pServer->nBytesInTxBuffer, 0)
                                           != pServer->nBytesInTxBuffer )
               { pServer->pszLastError = "ClientThread failed to send unsolicited data";
               }
              else // successfully SENT something, so COUNT this (for statistics):
               { pServer->dwNumBytesSent  += pServer->nBytesInTxBuffer;
                 pServer->Client[HLSRV_LOCAL_CLIENT_INDEX].dwNumBytesSent += pServer->nBytesInTxBuffer;
                 // HLSRV_FeedActivityWatchdog( &pServer->Client[HLSRV_LOCAL_CLIENT_INDEX] ); // feed the dog ?
                 if( pServer->cfg.iDiagnosticFlags & HLSRV_DIAG_FLAGS_SHOW_NETWORK_TRAFFIC ) // beware, a "CPU hog"..
                  { INET_DumpTraffic_HexOrASCII( sz1kDump,500, pServer->bTxBuffer, pServer->nBytesInTxBuffer );
                    ShowError( ERROR_CLASS_TX_TRAFFIC | SHOW_ERROR_TIMESTAMP, "TX %s", sz1kDump );
                  }
                 pServer->nBytesInTxBuffer = 0;
               }
            }
         } // end if( activity == 0 ), i.e. "nothing received but no error"
        else if (activity < 0) // > "SOCKET_ERROR (-1) if an error occurred"...
         { // select() failed.  "This is a permanent error" ... close the socket, try again later
           ShowError( ERROR_CLASS_INFO | SHOW_ERROR_TIMESTAMP, "HamlibClient: select() failed; closing socket, try later." );
           HLSRV_CloseClientSocketAndStartWaitingForReconnect( pServer, &s );
              //     '--> HLSRV_CONN_STATE_WAIT_RECONN
         }
        else // neither "timeout" in select() or a "complete fail" -> SOMETHING must have happened !
        if (FD_ISSET( s, &readfds)) // Mr. select() promised that Mr. recv() won't block...
         {
           valread = recv( s, pServer->bRxBuffer, HLSRV_SOCKET_RX_BUFFER_SIZE, 0);
           if( valread == SOCKET_ERROR)
            {
              int error_code = WSAGetLastError();
              if( pServer->cfg.iDiagnosticFlags & HLSRV_DIAG_FLAGS_VERBOSE_OUTPUT )
               { ShowError( ERROR_CLASS_ERROR | SHOW_ERROR_TIMESTAMP, "HamlibClient: recv() failed (%s)",
                            INET_WinsockErrorCodeToString( error_code ) );
               }
              if(error_code == WSAECONNRESET)
               { // Someone pulled the plug, not necessarily the REMOTE SERVER ..
                 // Close the socket, restart the "waiting timer" ..
                 HLSRV_CloseClientSocketAndStartWaitingForReconnect( pServer, &s );
                 //     '--> HLSRV_CONN_STATE_WAIT_RECONN
               }
              else // revc() failed for some other reason
               {   // also pull the plug, wait, then try to plug it in again ?
                 HLSRV_CloseClientSocketAndStartWaitingForReconnect( pServer, &s );
               }
            } // end if < recv() -> SOCKET_ERROR > ?
           if ( valread == 0)
            {
              // Somebody disconnected ...
              if( pServer->cfg.iDiagnosticFlags & HLSRV_DIAG_FLAGS_VERBOSE_OUTPUT )
               { ShowError( ERROR_CLASS_ERROR | SHOW_ERROR_TIMESTAMP, "HamlibClient: recv() returned 0 (disconnected?)" );
               }
              HLSRV_CloseClientSocketAndStartWaitingForReconnect( pServer, &s );
            }
           else // something received ->
            { pServer->nBytesInRxBuffer = valread;
              if( pServer->nBytesInRxBuffer > pServer->iPeakRxBufferUsage )
               {  pServer->iPeakRxBufferUsage = pServer->nBytesInRxBuffer;
               }
              pServer->dwNumBytesRcvd += pServer->nBytesInRxBuffer; // here: inc'd in ClientThread()
              HLSRV_OnReceive( pServer, &pServer->Client[HLSRV_LOCAL_CLIENT_INDEX], pServer->bRxBuffer, pServer->nBytesInRxBuffer );
              //  '--> May already deposit some data in pServer->bTxBuffer !
              if( ! pServer->fSimulateBadConnNow )
               { HLSRV_OnPoll( pServer,    &pServer->Client[HLSRV_LOCAL_CLIENT_INDEX] );
                 //  '--> May deposit MORE data in pServer->bTxBuffer !
               }
              if( pServer->nBytesInTxBuffer > 0 ) // send a RESPONSE (here: "Nagle friendly") ?
               { if( send( s, pServer->bTxBuffer, pServer->nBytesInTxBuffer, 0)
                                              != pServer->nBytesInTxBuffer )
                  { pServer->pszLastError = "ClientThread failed to send response";
                  }
                 else
                  { pServer->dwNumBytesSent += pServer->nBytesInTxBuffer;
                    if( pServer->cfg.iDiagnosticFlags & HLSRV_DIAG_FLAGS_SHOW_NETWORK_TRAFFIC ) // beware, a "CPU hog"..
                     { INET_DumpTraffic_HexOrASCII( sz1kDump,500, pServer->bTxBuffer, pServer->nBytesInTxBuffer );
                       ShowError( ERROR_CLASS_TX_TRAFFIC | SHOW_ERROR_TIMESTAMP, "TX %s", sz1kDump );
                     }
                    pServer->nBytesInTxBuffer = 0;
                  }
               }
            }
         } // end if (FD_ISSET( s , &readfds))
      } // end if < 'Actively communicate' > ?
     else // Because there's no CONNECTED socket, or fSimulateBadConnNow is SET,
      {   //         can't "sleep" in select(),
          // so give the CPU to anyone else by "sleeping" in Sleep() .
          // This is an ugly hack and there are more elegant methods,
          // but any attempt to "wait for connect() using select()" failed
          // (WB didn't want to mess around with "jocktellsocket FIONBIO" & Co)
          Sleep( HLSRV_SOCKET_POLLING_INTERVAL_MS );  // holy crap ... but KISS
      }

   }   // end while < client-thread shall run >

  if( s != INVALID_SOCKET )
   { closesocket(s);
   }
  if( pServer->cfg.iDiagnosticFlags & HLSRV_DIAG_FLAGS_VERBOSE_OUTPUT )
   { ShowError( ERROR_CLASS_INFO | SHOW_ERROR_TIMESTAMP, "HamlibClient: Disconnected and thread terminated" );
   }
  return 0;
} // end ClientThread()
#endif // < implement a Hamlib NET *client* > ?


  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // SOCKET related helper functions : All in CwNet.c (which existed FIRST)
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -



/* EOF < HamlibServer.c > */














