//---------------------------------------------------------------------------
// File  :  C:\CBproj\SoundUtl\Goertzel.cpp
// Date  :  2004-01-29   (ISO 8601, YYYY-MM-DD)
// Author:  Wolfgang Buescher  (DL4YHF)
//
// Description:
//     Implementation of class for a SINGLE Goertzel (DFT-like) instance;
//     and an array-like class for a few 'Goertzels' working in parallel .
//
// Revision history :  See *.cpp
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
#ifndef _Goertzel_H_
#define _Goertzel_H_

#ifdef BUILDING_THE_DLL  /* defined somewhere under 'Project..Options'  */
# define DLLIMPEX __declspec(dllexport)  /* here: DLLIMPEX = DLL EXPORT */
#else
# define DLLIMPEX __declspec(dllimport)  /* here: DLLIMPEX = DLL IMPORT */
#endif

#ifndef SoundTabH       // only if not included yet :
# include "SoundTab.h"  // cosine lookup table, filter coefficients, T_Float, etc.
#endif

//----------------------- Constants ---------------------------------------

#if(0) // (1)=just a test to see the full spectrum of a squared MSK baseband signal
#define C_GOERTZEL_MAX_FREQUENCY_BINS 4096
#else  // normal compilation:
#define C_GOERTZEL_MAX_FREQUENCY_BINS 128
   // Note: If you really need more frequency bins than this,
   //       consider using an FFT ;-)
#endif

// Other 'frequently needed' stuff..
#ifndef  C_PI
 #define C_PI  (3.1415926535897932384626)
#endif


#ifndef CPROT   // for peaceful co-existence of C and C++ ...
#ifdef __cplusplus
 #define CPROT extern "C"
#else
 #define CPROT
#endif  // not "cplusplus" ?
#endif // ndef CPROT ?

//----------------------- Data Types --------------------------------------

#define T_GoertzelFloat double
  //   ,----------------'
  //   '--> You don't want this to be "double" when compiling for a
  //        16- or even 8-bit microcontroller !
  // Speed test on an ancient 'Windows' PC :
  //  100 Goertzel filters (frequency bins) in parallel, fed with 37793 I/Q samples / sec:
  //             ~ 12 % CPU load   with T_GoertzelFloat = float ;
  //             ~ 16 % CPU load   with T_GoertzelFloat = double .

typedef struct tGoertzelFilter
{ // Structure with all required data for a single "frequency bin";
  // used as an array inside CGoertzel .
  union
   {struct
     { T_GoertzelFloat coeff; // coefficient (used when processing a new sample)
       T_GoertzelFloat q1;
       T_GoertzelFloat q2;    // queue (delay line with two previous samples, real part)
       T_GoertzelFloat q1_im; // imaginary part, only used if the INPUT is complex
       T_GoertzelFloat q2_im;
     }v;
    T_GoertzelFloat a[5];     // for an attempted speed-optimized loop (failed)
       # define GOERTZEL_COEFF 0
       # define GOERTZEL_Q1R   1
       # define GOERTZEL_Q2R   2
       # define GOERTZEL_Q1I   3
       # define GOERTZEL_Q2I   4
   }u;
  T_GoertzelFloat c_sin, c_cos; // sin(omega) and cos(omega). Required when a block
                       // is complete, and a DFT shall be calculated .
                       // (not required for the 'recursive part')
  // ex: double dblFcenter;   // center frequency of this bin in Hertz (!)
  int k;  // k = (int) (0.5 + BlockLen * dblCenterFrequency / dblInputSampleRate );
    // k can be used to retrieve the frequency bin's CENTER FREQUENCY precisely;
    // that's why it is stored in this structure (instead of dblFcenter) .
} T_GoertzelFilter;

#ifdef __cplusplus  // for peaceful coexistence between C an C++ ...
extern "C" {
#endif

// Function pointer prototype for a CALLBACK FUNCTION when <N> samples are through .
typedef void (*T_GoertzelCallback)( void *pvUserData, T_Complex *pComplexDFT, int iNumBins );

typedef struct tGoertzelArray
{ // Structure with an ARRAY of Goertzel filters, for MULTIPLE "frequency bins".
  // For some of the following struct members, TWO instance are required
  //          to handle the two "overlapped blocks" !
  T_GoertzelFilter Filters[2][C_GOERTZEL_MAX_FREQUENCY_BINS]; // q1,q2,coeff, ...
  T_Complex     ComplexDFT[2][C_GOERTZEL_MAX_FREQUENCY_BINS]; // last "finished" complex DFT

  long   i32BlockLength;
  long   i32SampleIndexInBlock[2];
  int    iNumFrequencyBins;
  int    iOptions;  // GOERTZEL_OPT_NO_WINDOW, GOERTZEL_OPT_HANN_WINDOW, ...
  int    iInputSampleRate;
  int    iFirstCenterFrequency;
  int    iFrequencyStepwidth;

} T_GoertzelArray;



//------------------------ Global Variables -------------------------------



//---------------------------------------------------------------------------
// Functions  (don't use C++ classes when a simple function does the job.
//       For the intended implemenation on a Microcontroller, use C, not C++.)
//---------------------------------------------------------------------------


CPROT T_GoertzelFloat Goertzel_CalcWindowFunction( int iOptions, float fltNormalizedIndex/*0..1*/ );


BOOL GoertzelArray_Init( // Initializes a T_GoertzelArray and sets ALL parameters...
       T_GoertzelArray *pGA,  // [in,out] formerly a microcontroller-unfriendly C++ class
       long   i32BlockLength,     // number of samples fed into the Goertzel filters
                                  // before finishing (and reading the result)
       int    iNumFrequencyBins,  // number of frequency bins, 1..C_GOERTZEL_MAX_FREQUENCY_BINS
       int    iOptions,           // bit combination of the following flags:
#define GOERTZEL_OPT_NO_WINDOW    0  // aka rectangular window (best noise bandwidth, B=1)
#define GOERTZEL_OPT_HANN_WINDOW  1  // aka raised cosine window (better roll-off, but B=1.5)
#define GOERTZEL_OPT_TUKEY_WINDOW 2  // 50% flat center, raised cos at the edges, B=1.22
#define GOERTZEL_OPT_OVERLAP      4  // 50% overlap (use together with cos^2 - window)
       int iInputSampleRate, // often 11025, 22050, 44100, or 48000 samples/second (the latter only if "int" is 32 bit)
       int iFirstCenterFrequency, // center frequency of the FIRST "frequency bin" [Hertz]
       int iFrequencyStepwidth);  // distance between two frequency bins [Hertz]

int    GoertzelArray_GetSampleRate( T_GoertzelArray *pGA ); // -> initial ("constant") input sampling rate
void   GoertzelArray_SetSampleRate( T_GoertzelArray *pGA, int iSampleRate);
int    GoertzelArray_GetCenterFrequency( T_GoertzelArray *pGA, int iBinIndex);  // -> center frequency for a frequency bin,
         // under the assumption that the input sampling rate is CONSTANT,
         // i.e. still the same value as specified in GoertzelArray_Init() .
BOOL   GoertzelArray_SetCenterFrequency( T_GoertzelArray *pGA, int iBinIndex, int iCenterFrequency);
int    GoertzelArray_GetInputBlockLength( T_GoertzelArray *pGA );
int    GoertzelArray_GetNumSamplesInInputBlock( T_GoertzelArray *pGA );
int    GoertzelArray_GetNumSamplesMissingForNextBlock( T_GoertzelArray *pGA );
int    GoertzelArray_GetNumFrequencyBins( T_GoertzelArray *pGA );

BOOL   GoertzelArray_ProcessSamples( T_GoertzelArray *pGA,
       T_Float *pfltSource,  // pointer to SOURCE samples (real signal or "I" component)
       T_Float *pfltSourceQ, // optional Q-component, or NULL for real input
       int  iNrSamplePoints,
       T_GoertzelCallback pcbkBlockComplete, // "blocks complete" callback (optional)
       void    *pvUserData); // user-defined callback data

#ifdef __cplusplus
}; // end extern "C" (when #included from a C++ module)
#endif


#endif // _Goertzel_H_