//---------------------------------------------------------------------------
// File:   C:\cbproj\Remote_CW_Keyer\FreqList.h         
//         (based on \cbproj\SpecLab\FreqList.h)                          
// Date:   2009-07-09  (ISO 8601, YYYY-MM-DD)                              
// Author: Wolfgang Buescher (DL4YHF)                                      
//   - C implementation by DL4YHF 2009 to 2012 for Spectrum Lab,
//   - Stripped-down variant in \cbproj\Remote_CW_Keyer\FreqList.c         
//   - Using as little BORLAND-specific stuff as possible in here !
//   - First supported file format was DL4YHF's own text file format,      
//       for example SpecLab/frequencies/default.txt .                     
//   - Second supported file format was EiBi's *.txt or *.csv format,      
//       SORTED BY FREQUENCY,  see   http://www.eibispace.de/  .           
//---------------------------------------------------------------------------

#include <windows.h>  // must be included BEFORE some other files, for bizarre reasons
#include <string.h>
#include "QFile.h"     // "Quick File Access Functions" (to read text files line by line)
#include "StringLib.h" // replace strncpy() by SL_strncpy(), because strncpy() sucks !
#include "Utilities.h" // stuff like UTL_iWindowsVersion, UTL_iAppInstance, ShowError(), etc

#include "FreqList.h"  // header for THIS module

//------------ Internal types ------------------------------------------------
// To avoid allocating unnecessarily small memory blocks (with overhead),
//    100 entries (with ~100 bytes per entry) are packed in one "block" .
//    The linked-list search functions are not affected by this !
#define FREQLIST_NODES_PER_BLOCK 100
typedef struct t_FreqListBlock
{
   int nNodesUsed;
   T_FreqListEntry node[FREQLIST_NODES_PER_BLOCK];
   struct t_FreqListBlock *pNext; // pointer to next allocated block
} T_FreqListBlock;

#define FreqList_NUM_COLUMNS 30  // Note: HFCC tables have over 20 different columns..
  // Standard columns for EiBi's database format:
#define FreqList_Col_Freq       0     /* normal column index for the frequenc ("kHz") */
#define FreqList_Col_TimesOfDay 1     /* normal column index for the "Time(UTC)" [TWO times] */
#define FreqList_Col_Days       2     /* normal column index for the "Days"      */
#define FreqList_Col_ITU_Country 3    /* normal column index for the "ITU" (country) */
#define FreqList_Col_Station    4     /* normal column index for the "Station" (full name, sometimes City/QTH) */
#define FreqList_Col_Language   5     /* normal column index for the "Lang."     */
#define FreqList_Col_Target     6     /* normal column index for the "Target" (continent) */
#define FreqList_Col_Remarks    7     /* normal column index for the "Remarks"   */
 // Additional columns for the DL4YHF's own file format; header :
 //  kHz    Call   ITU   Act Station                 Locator Lang. Remarks
#define FreqList_Col_Call       8     /* column identifier for the "Callsign" */
#define FreqList_Col_Active     9     /* column identifier for the active/passive status */
#define FreqList_Col_Locator   10     /* column identifier for the Maidenhead Locator */
 // Additional columns for the HFCC "OPERATIONAL SCHEDULE" format :
#define FreqList_Col_StartTimeOfDay 11
#define FreqList_Col_StopTimeOfDay  12


typedef struct t_ColumnInfo
{
  char cColumnSeparator;         // ' ' for *.txt;  ';' for *.csv; sometimes '\t'
  double dblFrequencyScalingFactor;  // 1=Hz, 1e3=kHz, 1e6=MHz, etc etc
  int iPos[FreqList_NUM_COLUMNS];
  int iWidth[FreqList_NUM_COLUMNS];
  int iContent[FreqList_NUM_COLUMNS];
} T_ColumInfo;



//------------ Internal variables --------------------------------------------
static T_FreqListBlock * FreqList_pBlocks = NULL;  // NULL: nothing allocated
// Because the above list of ALLOCATION BLOCKS is not sorted by frequency,
// there's another pointer for the entry with the LOWEST FREQUENCY:
static T_FreqListEntry  * FreqList_pnLowestFreq = NULL;


//------------ Implementation ------------------------------------------------


//---------------------------------------------------------------------------
void FreqList_Exit(void)
   // Unloads the list and cleans up resources .
   // Must be called on exit .
{
  T_FreqListBlock *pBlock = FreqList_pBlocks;
  T_FreqListBlock *pNext;
  FreqList_pBlocks = NULL;  // already mark the entire chain as "invalid"
  FreqList_pnLowestFreq = NULL;
  while( pBlock )
   { pNext = pBlock->pNext;
     free( (void*) pBlock );
     pBlock = pNext;
   }

} // FreqList_Exit()


//---------------------------------------------------------------------------
T_FreqListEntry *FreqList_GetFirstNode( void )
{
  if( FreqList_pBlocks != NULL )
   { if( FreqList_pBlocks->nNodesUsed > 0 )
      { if( FreqList_pnLowestFreq != NULL )
         { return FreqList_pnLowestFreq;  // return the node with the LOWEST FREQUENCY
         }
        else
         { return &FreqList_pBlocks->node[0];
         }
      }
   }
  return NULL;  // no entry loaded
} // end FreqList_GetFirstNode()


//---------------------------------------------------------------------------
static T_FreqListEntry *FreqList_NewNode( T_FreqListEntry *pSourceData )
  // "allocates" (well, not really)  a new node,
  //   keeps the list sorted by frequency !
{
  T_FreqListBlock *pBlock,   *pPrevBlock;
  T_FreqListEntry  *pNewNode, *pPrevLastNode;
  T_FreqListEntry  *pNextNodeByFrequency;  // since 2012, to keep list sorted by frequency
  int iNodeOfBlock;

  if( FreqList_pBlocks == NULL )
   { // Allocate the very first 'block' (with 100 nodes)...
     pBlock = FreqList_pBlocks = (T_FreqListBlock*)malloc( sizeof(T_FreqListBlock) );
     if( pBlock )
      { pBlock->nNodesUsed = 0;  // nothing used in this block yet
        pBlock->pNext = NULL;    // this is the last block now
      }
   }
  // Search the preceding node BY FREQUENCY.
  // The list is always sorted by increasing frequency.
  //   'pNextNodeByFrequency' will be the node *BEFORE* which the new entry
  //   is inserted into the linked list .
  //   If pNextNodeByFrequency is NULL, the new entry will be APPENDED
  //   to the END of the already existing list.
  pNextNodeByFrequency = FreqList_GetFirstNode(); // this pointer may be NULL !
  while( pNextNodeByFrequency )
   { if( pNextNodeByFrequency->dblFrequency_Hz > pSourceData->dblFrequency_Hz )
      { break; // ok; the new entry shall be inserted BEFORE this item
      }
     pNextNodeByFrequency = pNextNodeByFrequency->pNext; // may be NULL !
   }
  // Run through all ALLOCATION BLOCKS (with 100 nodes each) until we find one
  //  which has at least one unused entry...
  pBlock = FreqList_pBlocks;
  pPrevBlock = NULL;
  while( pBlock )
   {
     if( (iNodeOfBlock=pBlock->nNodesUsed) < FREQLIST_NODES_PER_BLOCK )
      { // here's an unused entry; no need to call 'malloc' or similar yet again..
        pNewNode = &pBlock->node[iNodeOfBlock];
        pPrevLastNode = NULL;
        if( iNodeOfBlock>0 )
         { pPrevLastNode = &pBlock->node[iNodeOfBlock-1];
         }
        else
        if( pPrevBlock != NULL )
         { pPrevLastNode = &pPrevBlock->node[FREQLIST_NODES_PER_BLOCK-1];
         }
        ++pBlock->nNodesUsed;
        // Initialize and append the new node to the double-linked chain of NODES:
        *pNewNode = *pSourceData;
        pNewNode->index = -1; // index still unknown (set by caller) !
        if( FreqList_pnLowestFreq == NULL )
         { FreqList_pnLowestFreq = pNewNode;
         }
        if( pNextNodeByFrequency == NULL ) // append the new entry to the END of the list:
         {
           pNewNode->pNext = NULL;  // this node is the new "last" node now
           pNewNode->pPrev = pPrevLastNode;  // may be NULL if it's the first entry
           if( pPrevLastNode != NULL )
            {  if( pPrevLastNode->pNext != NULL )
                {  pPrevLastNode->pNext = pPrevLastNode->pNext; // should never happen, set breakpoint HERE <<<
                }
               pPrevLastNode->pNext = pNewNode;
            }
         }
        else // Insert the new node in the double-linked list *BEFORE* pNextNodeByFrequency,
         {   //   to keep the list SORTED BY FREQUENCY (much faster to do it this way
             //   than sorting it AFTER all entries have been loaded) :
           pNewNode->pNext = pNextNodeByFrequency;         // link from NEW to NEXT node
           pNewNode->pPrev = pNextNodeByFrequency->pPrev;  // link from NEW to PREVIOUS node
           if( pNextNodeByFrequency->pPrev != NULL )
            { pNextNodeByFrequency->pPrev->pNext=pNewNode; // link from PREVIOUS to NEW node
            }
           pNextNodeByFrequency->pPrev = pNewNode;         // link from NEXT to NEW node
           // FOUR(*) node-pointers have been adjusted now:
           //      ______             ______            ______
           //     |PREV *|------>    | NEW *|------->  | NEXT |
           //     |______|   <-------|*_____|  <-------|*_____|
           //
           // The new entry may be the one with the LOWEST frequency of all:
           if( pNewNode->pPrev == NULL ) // this node is the new "first" node:
            { FreqList_pnLowestFreq = pNewNode;
              // 2012-10: Got here when loading a "VLF" list,
              //    with pNewNode->data.dblFrequency_Hz = 11904.7 Hz
              //    AFTER loading a medium wave frequency list. Ok.
            }
         } // end else < insert new node *BEFORE* pNextNodeByFrequency >

        return pNewNode;
      }
     if( pBlock->pNext == NULL )
      { // This is the LAST allocated "block", and it's completely full .
        // Allocate another one (with ~100 nodes per block) :
        pBlock->pNext = (T_FreqListBlock*)malloc( sizeof(T_FreqListBlock) );
        if( pBlock->pNext )
         { pBlock->pNext->nNodesUsed = 0;  // nothing used in this block yet
           pBlock->pNext->pNext = NULL;    // this is the last block now
         }
      }
     pPrevBlock = pBlock;
     pBlock = pBlock->pNext;
   } // end while( pBlock )
  return NULL;  // out of memory ?!
} // end FreqList_NewNode()

//---------------------------------------------------------------------------
long FreqList_ParseInt(char **ppcSource, int maxdigits)
{
  return QFile_ParseInteger( ppcSource, maxdigits, 10/*radix*/, 0/*deflt*/ );
} // end FreqList_ParseInt()

//---------------------------------------------------------------------------
void FreqList_SkipSpaces(char **ppcSource)
{
  char *cp = *ppcSource;
  while(*cp==' ') ++cp;
  *ppcSource = cp;
} // end FreqList_SkipSpaces()

//---------------------------------------------------------------------------
void FreqList_SkipAllExcept(char **ppcSource, char c)
{
  char *cp = *ppcSource;
  while( (*cp!=c) && (*cp!='\0') )
   { ++cp;
   }
  *ppcSource = cp;
} // end FreqList_SkipAllExcept()


//---------------------------------------------------------------------------
BOOL FreqList_SkipChar(char **ppcSource, char c)
{
  if( c!=' ' )
   { FreqList_SkipSpaces(ppcSource);
   }
  if( **ppcSource==c )
   { ++*ppcSource;
     return TRUE;
   }
  else
   { return FALSE;
   }
} // end FreqList_SkipChar()

//---------------------------------------------------------------------------
BOOL FreqList_SkipString(char **ppcSource, char *pszString)
{ return QFile_SkipString( ppcSource, pszString );
} // end FreqList_SkipString()


//---------------------------------------------------------------------------
BOOL FreqList_SkipCharOrAdvanceToColumn( char *pszLineStart, char **ppcSource,
                                      char cColumnSeparator, int iColumnNr)
{
  BOOL skipped_something = FALSE;
  char *cp;
  if( cColumnSeparator!=' ' )
   { return FreqList_SkipChar(ppcSource,cColumnSeparator);
   }
  // If the "column separator" is a SPACE (which isn't really a separator),
  // the source pointer should be INCREMENTED to the specified column number.
  //  (but ONLY incremented; NEVER decremented) .
  cp = *ppcSource;
  while( ((cp-pszLineStart)<iColumnNr) && (*cp!='\0') )
   { ++cp;
     skipped_something = TRUE;
   }

  return skipped_something;
} // end FreqList_SkipCharOrAdvanceToColumn()

//---------------------------------------------------------------------------
int FreqList_GetStringFromColumn(
        char **ppcSource,         // [in] ASCII line (read from file)
        char *pszDest,            // [out] string with data for <iColumn>
        int iMaxDestLen,          // [in]  max length of pszDest (# chars)
        T_ColumInfo *pColumnInfo, // [in]  info about the data line format
        int iColumn )             // [in]  column index for which to retrieve data
  // Sounds simple but isn't, because of the damned "space separated" fields
  // in some hand-written lists (*.txt) .
{
  char *cp = *ppcSource;
  char *cp2 = cp;
  char *cpDst = pszDest;
  char c;
  int  iColumnWidth = pColumnInfo->iWidth[iColumn];
  if( pColumnInfo->cColumnSeparator==' ' )
   { // If a space is the separator between two fields,
     //   all fields must have EXACTLY the correct size (as in the 'header line').
     while( (iMaxDestLen>0) && ((c=*cp)!='\0') && ((cp-cp2)<iColumnWidth)  )
      { *cpDst++ = c;  // note that this field may be A FEW characters longer than it should be
        ++cp;
        --iMaxDestLen;
      }
     *cpDst = '\0';  // terminate the destination string
     // Walk backwards through the copied string and remove TRAILING SPACES:
     while( cpDst>pszDest && (cpDst[-1]==' ') )
      { --cpDst;
        *cpDst = '\0';  // remove another ugly trailing space
      }
     // Skip remaining spaces (if the field is larger than the destination string)
     while( (*cp==' ') && ((cp-cp2)<iColumnWidth) )
      { ++cp;
      }
   }
  else // hooray... the column separator is NOT the stupid space..
   { // Here we don't need to care about the "approximate field length" !
     FreqList_SkipSpaces( &cp );   // remove leading spaces (usually there aren't any)
     while( (iMaxDestLen>0) && ((c=*cp)!=pColumnInfo->cColumnSeparator) && (c!='\0') )
      { *cpDst++ = c;
        ++cp;
        --iMaxDestLen;
      }
     *cpDst = '\0';  // terminate the destination string
   }
  *ppcSource = cp;
  return cp-cp2; // -> number of characters skipped (from the source)
} // end FreqList_GetStringFromColumn()

//---------------------------------------------------------------------------
long FreqList_ParseIntegerWithThousandSeparators(
        char **ppcSource,
        int maxdigits,
        char cThousandsSeparator,  // ignored 'thousands' separator (locale annoyance)
        long deflt)
  // String handling routine: Parses a decimal integer number from any sourcecode.
  // If the sourcecode doesn't contain a valid number,
  // the source pointer will not be moved, and the default value will be returned .
{
 long ret=0;
 int  neg=0;
 BOOL valid=FALSE;
 BYTE *bp = (BYTE*)*ppcSource;
 BYTE c;
  while(*bp==' ' || *bp=='\t')   // skip SPACES and TABS (for reading "text data files")
    { ++bp;
    }
  if(*bp=='-')
    { ++bp; neg=1; }
  else
  if(*bp=='+')
   { ++bp;
   }
  if( cThousandsSeparator != 0 )
   {
     while( ((c=*bp)!=0) && (maxdigits>0) )
      {
        if( c==cThousandsSeparator )
         { ++bp;
           --maxdigits;
         }
        else if( c>='0' && c<='9' )
         { ++bp;
           --maxdigits;
           valid = TRUE;
           ret = 10*ret + (c-'0');
         }
        else
         { break;
         }
      }
   }
  else // no stupid 'thousands separator' to be skipped :
   {   // use a simpler, and slightly faster loop
     while( (c=*bp)>='0' && (c<='9') && (maxdigits>0) )
      { ++bp;
        --maxdigits;
        valid = TRUE;
        ret = 10*ret + (c-'0');
      }
   } // end else < no 'thousands separator'
  *ppcSource = (char*)bp;
  if( valid )
       return neg ? -ret : ret;
  else return deflt;
} // end FreqList_ParseIntegerWithThousandSeparators()


//---------------------------------------------------------------------------
double FreqList_ParseFloat(char **ppcSource)
  // Modified 2010-06-21 to parse IK4HDQ's frequency list, too .
  // IK4HDQ uses a COMMA as separator before the fractional part;
  //         and a DOT after the thousands to increase the 'readability'.
  //   Some test cases :
  //      123.567 = 123.567 [DL4YHF]
  //        11,90 =  11.90  [IK4HDQ]
  //     3.622,50 = 3622.50 [IK4HDQ]
  // To avoid having to specify the decimal separators, FreqList_ParseFloat()
  //    now tries to find out what the '.' and ',' characters mean,
  //    using the following principle:
  //  Before really parsing the decimal number, analyse the string
  //    from left to right, ending at the first character which is not
  //    '0..9' , ',' , or '.' .  THEN DECIDE:
  // *  '.' was found but not a ','   ->  the '.' must be the start of the fractional part
  // *  ',' was found but not a '.'   ->  the ',' must be the start of the fractional part
  // *  '.' was found before the ','  ->  '.' = thousand separator (ignored), ',' = fractional part
  // *  ',' was found before the '.'  ->  ',' = thousand separator (ignored), '.' = fractional part
{
  char *cp = *ppcSource;
  char c, *cp2;
  char cThousandsSeparator = 0;
  char cDecimalSeparator   = 0;  // initial guess for the separator between integer and fractional part
  char cSep1, cSep2;
  double mantissa,fract;
  BOOL   negative = FALSE;
  FreqList_SkipSpaces( &cp );
  if( FreqList_SkipChar( &cp, '-' ) )
   { negative=TRUE;
   }
  cp2 = cp;    // Try to find out the thousands- and decimal separator character..
  cSep1 = cSep2 = 0;
  while( ((c=*cp2++)=='.') || (c==',') || (c>='0' && c<='9') ) // analyse separators
   { if( c=='.' || c==',' )
      { if( cSep1==0 )
         { cSep1=c;  // found the 1st separator (which *MAY* be the thousands separator)
         }
        else if( cSep2==0 && c!=cSep1 )
         { cSep2=c;  // found a 2nd (different) separator which must be the "decimal point"
         }
        if( cSep1!=0 && cSep2!=0 )
         { break;    // found two different separators, break from pre-analysis loop
         }
      }
   } // end while < pre-analysis to auto-detect the separator characters >
  // Now decide, which is which ...
  // *  '.' was found but not a ','   ->  the '.' must be the start of the fractional part
  // *  ',' was found but not a '.'   ->  the ',' must be the start of the fractional part
  // *  '.' was found before the ','  ->  '.' = thousand separator (ignored), ',' = fractional part
  // *  ',' was found before the '.'  ->  ',' = thousand separator (ignored), '.' = fractional part
  if( cSep1!=0 && cSep2==0 )    // only one 'separator character' found: should be the decimal separator
   { cDecimalSeparator = cSep1;
   }
  else
  if( cSep1!=0 && cSep2!=0 )    // two 'separator character' found: 1st=thousands, 2nd=decimal "point"
   { cThousandsSeparator = cSep1;
     cDecimalSeparator   = cSep2;
   }
  if( cDecimalSeparator == 0 )
   {  cDecimalSeparator = '.';  // initial guess for the separator between integer and fractional part
   }
  mantissa = FreqList_ParseIntegerWithThousandSeparators( &cp, 8/*maxdigits*/, cThousandsSeparator, 0/*deflt*/ );
  if( *cp==cDecimalSeparator ) // fractional part ?
   { ++cp; cp2=cp;
     fract = QFile_ParseInteger( &cp, 8/*maxdigits*/, 10/*radix*/, 0/*deflt*/ );
     // divide the fraction by TEN for each digit after the point... (warning, tricky)
     while(cp2<cp)
      { fract *= 0.1;
        ++cp2;
      }
     mantissa += fract;
   }
  *ppcSource=cp;
  return negative ? -mantissa : mantissa;
} // end FreqList_ParseFloat()


//---------------------------------------------------------------------------
void FreqList_InitColumnInfo(T_ColumInfo *pColumnInfo, char *pszFilename )
{ int iColumn;
  char *cp;

  memset( pColumnInfo, 0, sizeof(T_ColumInfo) );
  pColumnInfo->cColumnSeparator = ';';  // assume it's a CSV file
  cp = strrchr( pszFilename, '.' );
  if( cp )
   { if ( stricmp(cp,".csv")==0 )
      { pColumnInfo->cColumnSeparator = ';';
      }
     else if ( stricmp(cp,".txt")==0 )
      { pColumnInfo->cColumnSeparator = ' ';
      }
   }

  // Set the DEFAULT column positions (if we can't parse them...)
  for( iColumn=0; iColumn<FreqList_NUM_COLUMNS; ++iColumn )
   { // Ignore all other columns later (if they are not "defined" in the header line)
     pColumnInfo->iPos[iColumn]   = -1;
     pColumnInfo->iWidth[iColumn] = 0;
     pColumnInfo->iContent[iColumn] = -1;
   }
} // end FreqList_InitColumnInfo()


//---------------------------------------------------------------------------
int FreqList_ParseHeaderLine(char **ppcSource, T_ColumInfo *pColumnInfo)
  // Called after reading the FIRST (or the first few) line(s) from a file.
  // Tries to identify the file format by looking for common column names
  // like "kHz", "MHz", "Station", "Call", "Location:", etc etc pp. .
  // Note: *pColumnInfo must have been initialized by THE CALLER already,
  //       for reasons explained in FreqList_Load() .
{
  int iColumn, iRecognizedTokenLength;
  int nRecognizedTokens = 0;
  char *cp = *ppcSource;
  char *cp2;

  iColumn = 0;
  cp2 = cp;    // save 'start'-pointer
  while( (*cp!='\0') && (iColumn<FreqList_NUM_COLUMNS) )
   { FreqList_SkipSpaces( &cp );  // spaces may be skipped here, ALSO FOR CSV or tab-separated files !
     if( (iColumn>0) && (pColumnInfo->cColumnSeparator!=' ') )
      { if( ! FreqList_SkipChar( &cp, pColumnInfo->cColumnSeparator) )
         { break;   // no more column separators; we're through
         }
      }
     pColumnInfo->iPos[iColumn] = cp-cp2;
     if( iColumn>0 )
      { pColumnInfo->iWidth[iColumn-1] = pColumnInfo->iPos[iColumn] - pColumnInfo->iPos[iColumn-1];
      }
     iRecognizedTokenLength = 0;
     if( strncmp(cp,"kHz",3)==0 )   // used in EiBi table (*.txt or *.csv) and many others
      { pColumnInfo->iContent[iColumn] = FreqList_Col_Freq;
        pColumnInfo->dblFrequencyScalingFactor = 1e3;  // definitely "kilohertz"
        iRecognizedTokenLength = 3;
      }
     else if( strncmp(cp,"MHz",3)==0 )       // some other table (with frequencies in MEGAHERTZ)
      { pColumnInfo->iContent[iColumn] = FreqList_Col_Freq;
        pColumnInfo->dblFrequencyScalingFactor = 1e6;  // scale into "MHz"
        iRecognizedTokenLength = 3;
      }
     else if( strncmp(cp,"Freq/Hz",7)==0 )   // and yet another table (with frequencies in HERTZ)
      { pColumnInfo->iContent[iColumn] = FreqList_Col_Freq;
        pColumnInfo->dblFrequencyScalingFactor = 1.0;
        iRecognizedTokenLength = 7;
      }
     else if( strncmp(cp,"Hz ",3)==0)  // for convenience, and because "Hz" is the SI unit
      { pColumnInfo->iContent[iColumn] = FreqList_Col_Freq;
        pColumnInfo->dblFrequencyScalingFactor = 1.0;
        iRecognizedTokenLength = 3;
      }
     else if( strncmp(cp,";FREQ ",6)==0 )   // Frequency column header, HFCC Operational Schedule
      { pColumnInfo->iContent[iColumn] = FreqList_Col_Freq;
        pColumnInfo->dblFrequencyScalingFactor = 1e3; // not explicitly said, but HFCC uses kHz, too
        iRecognizedTokenLength = 6;
      }
     else if( strncmp(cp,"Time(UTC)",9)==0 ) // looks like an EiBi table ..
      { pColumnInfo->iContent[iColumn] = FreqList_Col_TimesOfDay; // these are TWO times (unlike HFCC)
        iRecognizedTokenLength = 9;
      }
     else if( strncmp(cp,"Days",4)==0 )
      { pColumnInfo->iContent[iColumn] = FreqList_Col_Days;
        iRecognizedTokenLength = 4;
      }
     else if( strncmp(cp,"ITU",3)==0 )   // ITU country code of what.. the TX site or the owner ?
      { pColumnInfo->iContent[iColumn] = FreqList_Col_ITU_Country;
        iRecognizedTokenLength = 3;
      }
     else if( strncmp(cp,"Station",7)==0 )  // seen in Eu_NDB_list.txt and many others..
      { pColumnInfo->iContent[iColumn] = FreqList_Col_Station;
        iRecognizedTokenLength = 7;
      }
     else if( strncmp(cp,"Lang.",5)==0 )   // EiBi *.txt,   Eu_NDB_list.txt
      { pColumnInfo->iContent[iColumn] = FreqList_Col_Language;
        iRecognizedTokenLength = 5;
      }
     else if( strncmp(cp,"Lng",3)==0 )     // EiBi *.csv
      { pColumnInfo->iContent[iColumn] = FreqList_Col_Language;
        iRecognizedTokenLength = 3;
      }
     else if( strncmp(cp,"Target",6)==0 )  // EiBi *.txt + *.csv
      { pColumnInfo->iContent[iColumn] = FreqList_Col_Target;
        iRecognizedTokenLength = 6;
      }
     else if( strncmp(cp,"Remarks",7)==0 ) // Eu_NDB_list.txt
      { pColumnInfo->iContent[iColumn] = FreqList_Col_Remarks;
        iRecognizedTokenLength = 7;
      }
     // Additional tokens/columns for the DL4YHF's own file format:
     else if( strncmp(cp,"Call",4)==0 )
      { pColumnInfo->iContent[iColumn] = FreqList_Col_Call;
        iRecognizedTokenLength = 4;
        // e.g. in Eu_NDB_list.txt : "kHz    Call    ITU  Act Station      Locator  Lang. Remarks"
      }
     else if( strncmp(cp,"Locator",7)==0 )  // seen in Eu_NDB_list.txt
      { pColumnInfo->iContent[iColumn] = FreqList_Col_Locator;
        iRecognizedTokenLength = 7;
      }
     // Additional tokens/columns for EiBi's CSV format
     // (not clear what these entries are for; they are not documented
     //  in the README file).  CSV Example (manually space padded) :
     // kHz:75;Time(UTC):93;Days:59;ITU:49;Station:201;Lng:49;Target:62;Remarks:135;P:35;Start:60;Stop:60;U9050:40
     // 198   ;0600-0630   ;       ;POL   ;Polskie R..;  D   ; Eu      ;           ;6   ;2903    ;   3108
     // 198   ;0630-0700   ;       ;POL   ;Polskie R..;  HB  ; Eu      ;           ;6   ;2903    ;   3108
     else if( strncmp(cp,"P:",2)==0 )
      { // pColumnInfo->iContent[iColumn] = ??;
        // ++nRecognizedTokens;
      }
     else if( strncmp(cp,"Start:",6)==0 )
      { // pColumnInfo->iContent[iColumn] = ??;
        // ++nRecognizedTokens;
      }
     else if( strncmp(cp,"Stop:",5)==0 )
      { // pColumnInfo->iContent[iColumn] = ??;
        // ++nRecognizedTokens;
      }
     // Additional tokens/columns/"special cases" for HFCC Operational Schedule; example:
     // ;----+----+----+------------------------------+---+----+-------+-------+-------+------+------+-+-----+----------+---+---+---+-----+-+-----+-----+-----+-------
     // ;FREQ STRT STOP CIRAF ZONES                    LOC POWR AZIMUTH         DAYS    FDATE  TDATE MOD      LANGUAGE   ADM BRC FMO REQ# OLD ALT1 ALT2  ALT3  NOTES
     // ;----+----+----+------------------------------+---+----+-------+-------+-------+------+------+-+-----+----------+---+---+---+-----+-+-----+-----+-----+-------
     //  3185 0000 1200 2-4,23,24                      WRB  100 340             1234567 290309 251009 D                  USA WRB FCC  5604
     else if( strncmp(cp,"STRT ",5)==0 )         // "Start Time Of Day" (HFCC way)
      { pColumnInfo->iContent[iColumn] = FreqList_Col_StartTimeOfDay;
        iRecognizedTokenLength = 5;
      }
     else if( strncmp(cp,"STOP ",5)==0 )         // "Stop  Time Of Day" (HFCC way)
      { pColumnInfo->iContent[iColumn] = FreqList_Col_StopTimeOfDay;
        iRecognizedTokenLength = 5;
      }
     else if( strncmp(cp,"CIRAF ZONES",11)==0 )
      {
        // (This "CIRAF ZONES" thing is a special case, because there's a SPACE in it,
        //  which would cause the regular parsing to fail without the following
        //  special treatment.  Bleah.) What's this CIRAF thing at all ? -->
        // > Following are CIRAF ZONES, used by the IFRB (International Frequency
        // > Registration Board) and a number of international broadcasters
        // > to specify intended audience areas for broadcasts.
        // > These are the zones quoted, for example, in the BBC schedules
        // > and on the FCC HF Bureau Web pages.
        cp += 11;  // skip "CIRAF ZONES" (only one column !)
        pColumnInfo->iContent[iColumn] = FreqList_Col_Target;
        iRecognizedTokenLength = 11;
      }
     else if( strncmp(cp,"LOC ",4)==0 )         // code for the "transmitter's location" ?! (HFCC way)
      { pColumnInfo->iContent[iColumn] = -1; // no idea what to do with this
        // (in fact, to make sense out of this "LOC"-thing, one would have to load
        //  YET ANOTHER file... in this case "site.txt" . We don't support this yet)
        // > WRB Morrison, TN                   USA 35N37 086W01
        // > BLN Berlin (Deutschlandradio)      D   52N30 013E20
        iRecognizedTokenLength = 4;
      }
     else if( strncmp(cp,"DAYS ",5)==0 )     // days-of-week, HFCC, totally different coding than EiBi !
      { pColumnInfo->iContent[iColumn] = FreqList_Col_Days;
        iRecognizedTokenLength = 5;
      }
     else if( strncmp(cp,"LANGUAGE ",9)==0 ) // program language, HFCC, totally different coding...
      { pColumnInfo->iContent[iColumn] = FreqList_Col_Language;
        iRecognizedTokenLength = 9;
      }
     else if( strncmp(cp,"ADM ",4)==0 ) // no idea what "ADM" stands for, it looks like some country code
      { pColumnInfo->iContent[iColumn] = FreqList_Col_ITU_Country; // just another bloody GUESS
        iRecognizedTokenLength = 4;
      }
     else if( strncmp(cp,"BRC ",4)==0 ) // "BRC" seems to be the broadcaster's 3-letter ID, like "BBC"
      { pColumnInfo->iContent[iColumn] = FreqList_Col_Station;
        iRecognizedTokenLength = 4;
        // Note: to turn this "BRC"-thing into a meaningful string, the file
        //       "broadcas.txt" (contained in the zip archive from HFCC)
        //       would have to be parsed. But since that file contains entries like
        //  > DTK Deutsche Telekom"
        //       ... so it doesn't seen worth the effort. Who wants to know the
        //       name of the company which runs a certain transmitter ?
      }
     else if( strncmp(cp,"NOTES",5)==0 ) // these "Notes" in the HFCC database are VERY CRYPTIC, anyway..
      { pColumnInfo->iContent[iColumn] = FreqList_Col_Remarks;
        iRecognizedTokenLength = 5;
      }
     // Keywords added 2012-07-14 to parse G4UCJ's Eu_NDB_list.txt :
     else if( strncmp(cp, "Call:",5)==0 )
      { pColumnInfo->iContent[iColumn] = FreqList_Col_Call;
        iRecognizedTokenLength = 5;
      }
     else if( strncmp(cp, "Location:",9)==0 )
      { // In G4UCJ's Eu_NDB_list.txt, this is actually the CITY, not the locator.
        // We put this into the REMARKS component so SL will display it, too:
        pColumnInfo->iContent[iColumn] = FreqList_Col_Remarks; // here: Station's "QTH" (City)
        iRecognizedTokenLength = 9;
      }
     else if( strncmp(cp, "Country:",8)==0 )
      { pColumnInfo->iContent[iColumn] = FreqList_Col_ITU_Country; // here: COUNTRY, no ITU abbrev.
        iRecognizedTokenLength = 8;
      }
     // SHORT TOKENS AT THE END OF THE LIST ! :
     else if( strncmp(cp,"Loc",3)==0 )
      { pColumnInfo->iContent[iColumn] = FreqList_Col_Locator;
        iRecognizedTokenLength = 3;
      }
     else if( strncmp(cp,"Act",3)==0 )  // seen in Eu_NDB_list.txt ..
      { pColumnInfo->iContent[iColumn] = FreqList_Col_Active;
        iRecognizedTokenLength = 3;
      }


     if( iRecognizedTokenLength > 0 )
      { cp += iRecognizedTokenLength;
        ++nRecognizedTokens;
        // Added 2012-07-14 to parse G4UCJ's Eu_NDB_list.txt,
        //  which seemed to use TABS instead of SPACES as column separator
        // (which is unknown at this point):
        if( cp[0]==':' && (cp[1]==' ' || cp[1]=='\t') )
         { ++cp;  // Skip the colon *AFTER* a token, but only when followed by a separator
         }
        if( cp[0]=='\t' && pColumnInfo->cColumnSeparator==' ')
         { pColumnInfo->cColumnSeparator = '\t'; // column separator not SPACE but TAB
         }
      }

     FreqList_SkipAllExcept( &cp, pColumnInfo->cColumnSeparator );
     ++iColumn;
   }
  // Arrived here, the width of the LAST column [iColumn-1] is still unknown !
  if((iColumn>0) && (iColumn<=FreqList_NUM_COLUMNS) )
   { pColumnInfo->iWidth[iColumn-1] = 40;  // another GUESS
     // (since it's impossible to tell the length of the LAST COLUMN from the headline,
     //  because the last item is usually not padded with spaces )
   }

  return nRecognizedTokens;
} // end FreqList_ParseHeaderLine()


//---------------------------------------------------------------------------
int  FreqList_Load(char * pszFilenames)
   // Loads a frequency list from one or more files (*.txt or *.csv, various formats) .
   // Returns the number of items loaded, 0 = nothing found .
{
  T_QFile qf;
  T_FreqListEntry *pNode;
  T_FreqListEntry new_entry;
  char sz255Filename[256], *cpNextFilename;
  char sz255[256], c, *cp;
  char sz255Value[256], *pszValue;
  T_ColumInfo column_info, column_info_2;
  BOOL header_found = FALSE;
  int iColumn, iNumLinesRead=0, iNumItemsLoaded=0;
  int nRecognizedTokens, nBestRecognizedTokens=0;



#ifdef __BORLANDC__  // "... is assigned a value that is never used".. oh shut up, we know what we're doing !
  (void)iNumLinesRead;
  (void)header_found;
  (void)nBestRecognizedTokens;
#endif

  if( FreqList_pBlocks != NULL )
   { FreqList_Exit();  // delete old list, and free resources
   }
  cpNextFilename = pszFilenames;
  while(1)  // repeat until there are no more filenames in the LIST OF FILES..
   {
     // Copy the next filename from the source, which may be a COMMA-DELIMITED list
     //  of MULTIPLE files (since 2012-10-11) :
     cp = sz255Filename;
     while( *cpNextFilename==' ' )
      { ++cpNextFilename;  // skip LEADING spaces (but don't skip spaces in stupid paths,
                           // which is another annoying windows-specific craze)
      }
     while( cp<(sz255Filename+255) && (c=*cpNextFilename)!='\0' )
      { if( c==',' ) // end of the next filename, and MORE TO FOLLOW
         { ++cpNextFilename;  // skip the comma in the source list
           break;             // try to load this file now
         }
        else
         { *cp++ = c; // copy everything else (unchanged)
           ++cpNextFilename;  // skip the character in the source list; whatever it was
         }
      }
     *cp = '\0';  // provide a string end marker for the next name
     if( cp == sz255Filename ) // seems to be the end of the list....
      { break;
      }
     if( (sz255Filename[0] != '\0') &&  QFile_Open(&qf, sz255Filename, QFILE_O_RDONLY ) )
      {
        iNumLinesRead = 0; // ... from the CURRENT file !
        header_found = FALSE;
        nBestRecognizedTokens=0;

        // Note: Since 2010-06-21, QFile_ReadLine() may throw out the stupid
        //       UTF-8 'BOM' thingy, when the ASCII file is really an ASCII file
        //       but some stupid editor (windoze notepad) decided to convert it
        //       from plain ASCII into this unnecessary UTF-8 bullshit .
        //       See details in QFile.c .
        while( (QFile_ReadLine( &qf, sz255, 255 )>=0)
               && (iNumLinesRead < 65535) )
         { ++iNumLinesRead;
           cp = sz255;  // 2020-07: Got here with sz255 = "y with diaresis" when trying to load "schumann.txt".
                        // The file was encoded as
           while(*cp==' ') ++cp;
           if( (c=*cp) != 0 )  // only if NOT a "blank" line:
            { // Check if this is a HEADER line (must begin in character index zero) :
              if( (strncmp(sz255,"kHz ",4)==0)  // in Eu_NDB_list.txt : "kHz    Call    ITU  Act Station      Locator  Lang."
                ||(strncmp(sz255,"KHz ",4)==0)  // found 'KHz' somewhere, should be 'kHz' though...
                ||(strncmp(sz255,"kHz:",4)==0)  // the same for EiBi's CSV..
                ||(strncmp(sz255,"kHz;",4)==0)  // other CSV without field lengths
                ||(strncmp(sz255,";FREQ",5)==0) // looks like HFCC "Operational Schedule"
                ||(strncmp(sz255,"MHz ",4)==0)
                ||(strncmp(sz255,"Hz  ",4)==0)  // for convenience, and because "Hz" is the SI unit
                ||(strncmp(sz255,"Freq/Hz ",8)==0) )
               { // this must be the "headline" of a *.txt file (EiBi's format)..
                 // Carefully analyse this line, and store the column positions.
                 cp = sz255;  // don't skip spaces before trying to analyse a HEADER line..
                 // Beware: We may get here MULTIPLE times to 'probe' for the header line !
                 FreqList_InitColumnInfo( &column_info_2, sz255Filename );
                 nRecognizedTokens = FreqList_ParseHeaderLine( &cp, &column_info_2 );
                 // Use the "best" header line we can find .
                 // For example, got here with ...
                 //     sz255Filename = "frequencies/default.txt"
                 //     sz255 = "kHz    Call    ITU  Act Station      Locator  Lang. Remarks"
                 //             -> nRecognizedTokens =  8
                 //  - - - - - - - - - - - - - - - - - - - - - - - - -
                 //     sz255Filename = "frequencies/Eu_NDB_list.txt"
                 //     sz255 = "kHz:\tCall:\tLocation:\tCounty:"
                 //             -> nRecognizedTokens =  4 (yes, only FOUR valid columns!)
                 if( nRecognizedTokens >= 3 )
                  { if( nRecognizedTokens > nBestRecognizedTokens )
                     { column_info = column_info_2;
                       nBestRecognizedTokens = nRecognizedTokens;
                       header_found = TRUE;
                     }
                  }
               }
              else if( c>='0' && c<='9' ) // begins with a digit, must be a DATA LINE,
               {                     // at least for EiBi's TEXT- and CSV-format...
                 // To avoid problems with RIGHT-ALIGNED NUMBERS in FIXED COLUMN LENGTH files
                 // (like HFCC), begin parsing at the first character again :
                 cp = sz255;               // important for HFCC (with leading spaces)
                 // pNode = FreqList_NewNode();  // allocate a new "node" (list item)
                 memset( &new_entry, 0, sizeof(new_entry) );
                 // First load the new entry from the file. The frequency must
                 // be known before inserting it into the list of nodes,
                 // because the linked list shall be SORTED BY FREQUENCY :
                 for(iColumn=0 ; (*cp!='\0') && (iColumn<FreqList_NUM_COLUMNS) ; ++iColumn )
                  { // Retrieve the "value" (as a string) for the current column.
                    // Note: the "max length" is important if the separator is
                    //       the SPACE character, because in that case,
                    //       the columns have a fixed witdth which (*.txt) .
                    //   *.csv (here: "semicolon separated values") is much better.
                    sz255Value[0] = '\0';
                    FreqList_GetStringFromColumn( &cp, sz255Value,255/*iMaxDestLen*/,
                              &column_info, iColumn );
                    pszValue = sz255Value;
                    switch( column_info.iContent[iColumn] )
                     {
                          case FreqList_Col_Freq:      /* column with the frequency ("kHz") */
                             new_entry.dblFrequency_Hz = FreqList_ParseFloat( &pszValue )
                                      * column_info.dblFrequencyScalingFactor;
                             break;
                          case FreqList_Col_TimesOfDay: /* column with "Time(UTC)", EiBi special */
                             // Parse the "Time (UTC)" (second data column, for example 0600-0630)
                             new_entry.iStartTime = 60 * FreqList_ParseInt( &pszValue, 2/*digits*/ );
                             new_entry.iStartTime +=     FreqList_ParseInt( &pszValue, 2/*digits*/ );
                             FreqList_SkipChar(&pszValue,'-');
                             new_entry.iEndTime   = 60 * FreqList_ParseInt( &pszValue, 2/*digits*/ );
                             new_entry.iEndTime   +=     FreqList_ParseInt( &pszValue, 2/*digits*/ );
                             break;
                          case FreqList_Col_StartTimeOfDay: /* column with the "START" time of day (HFCC) */
                             new_entry.iStartTime = 60 * FreqList_ParseInt( &pszValue, 2/*digits*/ );
                             new_entry.iStartTime +=     FreqList_ParseInt( &pszValue, 2/*digits*/ );
                             break;
                          case FreqList_Col_StopTimeOfDay: /* column with the "START" time of day (HFCC) */
                             new_entry.iEndTime = 60 * FreqList_ParseInt( &pszValue, 2/*digits*/ );
                             new_entry.iEndTime +=     FreqList_ParseInt( &pszValue, 2/*digits*/ );
                             break;
                          case FreqList_Col_Days     : /* column with the  "Days"      */
                             // Copy the "Days" (3rd column, for example "Mi-Tu")
                             SL_strncpy( new_entry.sz7DaysActive, pszValue, 7/*MaxDestLength*/ );
                             break;
                          case FreqList_Col_ITU_Country: /* column with the  "ITU" (country) */
                             SL_strncpy( new_entry.sz7ITUcountry, pszValue, 7 );
                             break;
                          case FreqList_Col_Station  : /* column with the  "Station" (full name) */
                             SL_strncpy( new_entry.sz31StationName, pszValue, 31 );
                             break;
                          case FreqList_Col_Language : /* column with the  "Language"     */
                             SL_strncpy( new_entry.sz7Language, pszValue, 7 );
                             break;
                          case FreqList_Col_Target   : /* column with the  "Target" (continent) */
                             SL_strncpy( new_entry.sz7Target,   pszValue, 7 );
                             break;
                          case FreqList_Col_Remarks  : /* column with the  "Remarks" , sometimes the nearest CITY  */
                             SL_strncpy( new_entry.sz31Remarks, pszValue, 31 );
                             break;
                          // Additional tokens/columns for the DL4YHF's own file format:
                          case FreqList_Col_Call     : /* column with the "Callsign" */
                             SL_strncpy( new_entry.sz7Callsign, pszValue, 7 );
                             break;
                          case FreqList_Col_Active   : /* column with the active/passive status */
                             SL_strncpy( new_entry.sz7ActiveStatus, pszValue, 7 );
                             break;
                          case FreqList_Col_Locator  : /* column with the Maidenhead Locator */
                             SL_strncpy( new_entry.sz7Locator, pszValue, 7 );
                             break;
                          default :  // add the destination for new columns here
                             break;
                     } // end switch( column_info.iContent[iColumn] )
                    if( column_info.cColumnSeparator!=' ' )
                     { if( ! FreqList_SkipChar( &cp, column_info.cColumnSeparator) )
                        { break;   // no more column separators; we're through
                        }
                     }
                  } // end for < all possible columns >

                 pNode = FreqList_NewNode( &new_entry );  // allocate a new "node" (list item)
                 if( pNode )
                  { pNode->index = iNumItemsLoaded;
                    ++iNumItemsLoaded;
                  } // end < FreqList_NewNode() successfull >
               } // end if < line beginning with a NUMBER, here: a FREQUENCY >
            } // end if < not a blank line >
         } // end while < more lines from the input file>
        QFile_Close(&qf);

        if( ! header_found ) // Something very wrong with the to-be-loaded file
         { ShowError( ERROR_CLASS_INFO | SHOW_ERROR_IN_RUN_LOG,
             "No headline found in %s, %d lines parsed !",
             sz255Filename, (int)iNumLinesRead );
           // 2020: Found no headline in 'Eu_NDB_list.txt' .
         }
      } // end if < file exists >
   } // end while < more files in the COMMA-delimited list >

  return iNumItemsLoaded;
} // end FreqList_Load()


//---------------------------------------------------------------------------
BOOL FreqList_CheckEntryByDateAndTime( T_FreqListEntry *pEntry,
                                       double dblUnixDateAndTime )
  // Returns TRUE if the specified "station" is theoretically transmitting
  //              at the specified date (day of week) and time (of day),
  //  otherwise FALSE.
  // [in] pEntry->iStartTime, iEndTime : "Time (UTC)" (time of day in MINUTES)
  // [in] pEntry->sz7DaysActive[8]     : "Days" (of week, with activity) .
  //      MUST BE TAKEN WITH A KILOGRAM OF SALT (in the EiBi database),
  //      because for example SAQ (Grimeton on 17.2 kHz) is listed to transmit
  //      only on December 24th .. so ignore that entry in most cases,
  //      or only trust it in clear cases like ...
  //          pEntry->sz7DaysActive = "Mo-Fr", "SaSu", "Mo-Th" (seen quite often).
  //          In case of doubt, SHOW an frequency marker, but don't hide it .
{
  int iDayOfWeek;
  if( dblUnixDateAndTime == 0.0 )   // dummy for 'accept entry regardless of time of day / day of week
   { return TRUE;
   }


  return TRUE; 

} // end FreqList_CheckEntryByDateAndTime()


//---------------------------------------------------------------------------
T_FreqListEntry *FreqList_GetEntryByIndex( int index )
   // Returns a pointer to an item in the frequency list (table) .
   // Note that we're using "C", so we start counting array indices at ZERO .
   // A return value of NULL means "we've reached the end of the list" .
{
  T_FreqListEntry *pNode = FreqList_GetFirstNode(); // this pointer may be NULL !
  while( pNode )
   { if( index==pNode->index )
      { return pNode;
      }
     pNode = pNode->pNext;
   }

  return NULL;  // nothing found
} // end FreqList_GetEntryByIndex()

//---------------------------------------------------------------------------
T_FreqListEntry *FreqList_GetNextEntry( T_FreqListEntry *pEntry,
                                        double dblUnixDateAndTime )
  // See FreqList_GetEntryByFrequency() for details on the 2nd argument !
{
  if( pEntry!=NULL )
   { return pEntry->pNext;
   }
  else
   { return NULL;
   }
} // end FreqList_GetNextEntry()

//---------------------------------------------------------------------------
T_FreqListEntry *FreqList_GetEntryByFrequency( double dblFreq_Hz,
                                               double dblUnixDateAndTime )
  // Finds the first station for the specified, or "closest higher" frequency.
  //  For example, searching for a station on 197 kHz may return a station
  //               on 197 or 198 kHz, but not a station on 196 kHz .
  //  Optionally (especially for broadcast listening using the EiBi database),
  //       the result may be 'filtered' for a certain time of day (in UTC).
  //       For NO filtering by date (e.g. day-of-week) and time (of day),
  //       pass in dblUnixDateAndTime = 0.0 (Unix birthdate, long ago).
  //  To retrieve more entries for -possibly- the same frequency,
  //       use FreqList_GetNextEntry() .
{
  T_FreqListEntry *pNode = FreqList_GetFirstNode(); // this pointer may be NULL !
  while( pNode )
   { if( pNode->dblFrequency_Hz >= dblFreq_Hz )
      { if( FreqList_CheckEntryByDateAndTime( pNode, dblUnixDateAndTime ) )
         { return pNode;
         }
      }
     pNode = pNode->pNext;
   }

  return NULL;  // nothing found

} // end FreqList_GetEntryByFrequency()


//----------------------------------------------------------------------------
int FreqList_EntryToMultiLineText( T_FreqListEntry *pFLE, char *pszDest, const char *pszEndstop )
  // Converts a frequency database entry into human readable, multi-line text.
  // Used in the GUI when the operator clicked on a 'frequency marker'
  // like e.g. "OK0EPB Pendulum", imported from the EiBi database .
{
  char *pszOldDest = pszDest;
  SL_AppendPrintf( &pszDest, pszEndstop, "%.3lf MHz : \"%s\"\n",
     (double)pFLE->dblFrequency_Hz * 1e-6, (char*)pFLE->sz31StationName );
  if( (pFLE->iStartTime != 0 ) || (pFLE->iEndTime != (24*60) ) )
   { SL_AppendString( &pszDest, pszEndstop, "Start- and end time: " );
     UTL_FormatDateAndTime("hh:mm", (double)(pFLE->iStartTime * 60), pszDest );
     SL_SkipToEndOfString( (const char**)&pszDest );
     SL_AppendString( &pszDest, pszEndstop, " - " );
     UTL_FormatDateAndTime("hh:mm", (double)(pFLE->iEndTime * 60), pszDest );
     SL_SkipToEndOfString( (const char**)&pszDest );
     SL_AppendString( &pszDest, pszEndstop, " UTC\n" );
   }
  if(pFLE->sz7Callsign[0] != '\0' ) // unfortunately, callsigns are very rare in the EiBi database
   { SL_AppendPrintf( &pszDest, pszEndstop, "Callsign  : %s\n", (char*)pFLE->sz7Callsign );
   }
  if(pFLE->sz7ITUcountry[0] != '\0' )
   { SL_AppendPrintf( &pszDest, pszEndstop, "ITU country: %s\n", (char*)pFLE->sz7ITUcountry );
   }
  if(pFLE->sz7Language[0] != '\0' )
   { SL_AppendPrintf( &pszDest, pszEndstop, "Program language: %s\n", (char*)pFLE->sz7Language );
   }
  if(pFLE->sz7Target[0] != '\0' )
   { SL_AppendPrintf( &pszDest, pszEndstop, "Target area: %s\n", (char*)pFLE->sz7Target );
   }
  if(pFLE->sz31Remarks[0] != '\0' )
   { SL_AppendPrintf( &pszDest, pszEndstop, "Remarks   : %s\n", (char*)pFLE->sz31Remarks );
   }
  if(pFLE->index > 0 )
   { SL_AppendPrintf( &pszDest, pszEndstop, "Database item index: %d\n", (int)pFLE->index );
   }
  return pszDest-pszOldDest;
} // end FreqList_EntryToMultiLineText()


/* EOF < FreqList.c >  */
