//---------------------------------------------------------------------------
// File  :  C:\cbproj\Remote_CW_Keyer\FFT_API.c
// Date  :  2025-02-16 (ISO 8601,  YYYY-MM-DD)
// Author:  Wolfgang Buescher  (DL4YHF)
//
// Revisions, Description and Literature :  See FFT_API.h !
//---------------------------------------------------------------------------

#ifndef _FFT_API_H_
# define _FFT_API_H_

#ifndef  C_PI
 #define C_PI  (3.1415926535897932384626)
#endif

// Because Borland's "C" header files (not C++) lacked the definition
// of a 'quiet NAN' (aka 'non-signalling NAN'), roll our own,
// to replace sick stuff like
//    "std::numeric_limits<double>::quiet_NaN()"  (only usable in C++)
// or "std::numeric_limits<float>::quiet_NaN()" .
// It's amazing to find such a simple thing is NOT STANDARDIZED, see :
//  http://stackoverflow.com/questions/1923837/how-to-use-nan-and-inf-in-c  !
#ifndef  C_QUIET_NAN_FLOAT
# ifdef  NAN
#    define C_QUIET_NAN_FLOAT NAN
# else  // no NAN available in math.h ?  Shame on you, Borland...
#  if(1) // method A without pointer-acrobatics at runtime:
#    define C_QUIET_NAN_FLOAT (0.0/0.0)
#  else // method B uses an ugly, and possibly non-portable pointer cast:
     extern long FFT_i32NAN;
#    define C_QUIET_NAN_FLOAT (*(float*)&FFT_i32NAN)
#  endif
# endif // ndef NAN
#endif // ndef C_QUIET_NAN_FLOAT

#ifndef  SWI_FLOAT_PRECISION   // should be defined under Project..Options..Conditionals
 #define SWI_FLOAT_PRECISION 1 // 1=single precision, 2=double precision
#endif

#ifndef  SWI_USE_OOURAS_FFT   // Use Takuya Ooura's FFT (based on fftsg_h.c) ?
 #define SWI_USE_OOURAS_FFT 1 // 1=use Ooura's FFT,  0=use a non-optimized textbook FFT
 // (T. Ooura's complex FFT is much faster than the one from DSP-Guide;
 //  for example 1.7 GHz Centrino Duo (only one core used),  262144 points/FFT :
 //    441 ms per complex FFT for the textbook-algorithm;
 //     91 ms per FFT using the algorithm from Ooura's "fftsg_h.c" ! )
#endif


// names of FFT windowing functions, used by FFT_BuildWindowTable() 
#define FFT_WINDOW_RECTANGLE 0
#define FFT_WINDOW_HAMMING   1
#define FFT_WINDOW_HANN      2   /* author's choice, aka cos^2. Often called "Hanning" (wrong) */
#define FFT_WINDOW_GAUSS     3
#define FFT_WINDOW_NUTTALL4B 4   /* for special applications.. */
#define FFT_WINDOW_FLATTOP5F 5   /* "Fast decaying 5-term flat top window" (from [HFTWIN] D.1.3, "SFT5F") */
#define FFT_WINDOW_FLATTOP5M 6   /* "Minimum sidelobe 5-term flat top window" (from [HFTWIN] D.1.6, "SFT5M") */
#define FFT_WINDOW_HFT95     7   /* "Heinzel Flat-Top -95 dB sidelobe" (from [HFTWIN] D.3.2, "HFT95")    */
#define FFT_WINDOW_HFT144D   8   /* "Heinzel Flat-Top -144 dB sidelobe" (from [HFTWIN] D.3.5, "HFT144D") */
#define FFT_WINDOW_HFT196D   9   /* "Heinzel Flat-Top -196 dB sidelobe" (from [HFTWIN] D.3.7, "HFT196D") */
#define FFT_WINDOW_HFT248D  10   /* "Heinzel Flat-Top -248 dB sidelobe" (from [HFTWIN] D.3.9, "HFT248D") */



extern long FFT_i32MathErrorCounter; // number of errors in FFT_API.c : _matherr, for the debug-log



//**************************************************************************
//    Prototypes (no class methods, no C++, just good old "C")
//**************************************************************************

#ifndef CPROT
// Note on "extern": do not trust the lousy example from Borland !!
//   >extern "c" = wrong (not recognized by Borland C++ V4.0)
//   >extern "C" = right (though the help system says something different)
#ifdef __cplusplus
 #define CPROT extern "C"
#else
 #define CPROT
#endif  /* ! "cplusplus" */
#endif


//---------------------------------------------------------------------------
CPROT void FFT_Init( void );
  // Should be called ONCE on init. In the older (bloated) variants, tried to
  // dynamically load a DLL with a faster FFT (feature removed many years ago).
  // If only the FFT Takuya Ooura or the 'textbook FFT' from www.DSPguide.com
  // is in used, FFT_Init() doesn't do anything at all - but anyway, CALL IT.
CPROT void FFT_Exit( void );
  // Should be called ONCE when the application exits. Possibly frees resources,
  // and formerly unloaded the 'pretty fast FFT' DLL (or whatever the name was).


//---------------------------------------------------------------------------
CPROT void FFT_ClearFloatArray(float *pfltArray, int iLength );
  // Clears an array with floating point values (sets all cells to 0.0) .

//---------------------------------------------------------------------------
CPROT void FFT_CopyFloatArray(float *pfltSource, float *pfltDest, int iLength );


//---------------------------------------------------------------------------
CPROT void FFT_ResampleFloatArray(float *pfltArray, int iSourceLength, int iDestLength );
#if(SWI_FFT_NEED_DOUBLE_PRECISION)
CPROT void FFT_ResampleDoubleArray(double *pfltArray, int iSourceLength, int iDestLength );
#endif
  // Stretches or shrinks an array.  Originally used for the FFT-based filter,
  // to adapt the frequency response curve when changing the FFT size .
  // Neither iSourceLength nor iDestLength may be zero or negative !


//---------------------------------------------------------------------------
CPROT double FFT_CalculateAngle(double re, double im);
  // Precise four-quadrant conversion of a complex pair ("I/Q")
  //   into an phase value (in radians, but explained in degrees here).
  // A positive real value gives an angle of zero, etc.
  // Returned value range is -180 .. +180 =  -pi .. pi .
  // If both real and imaginary part are zero, the returned value
  // is zero.


//---------------------------------------------------------------------------
CPROT float FFT_CalculateAngleFast(float x, float y);
  // Fast atan2 calculation with self normalization.
  // Returned value range is  -pi..pi =  -180 .. +180 .
  // Detailed explanation and discussion of accuracy in *.cpp !

//---------------------------------------------------------------------------
CPROT void  FFT_MultiplyHannWindow( float *pfltArray, int iLength ); // slow
CPROT float FFT_BuildWindowTable( float *pfltWindowTbl, int iFftSize, int iWindowFunction);
CPROT void  FFT_MultiplyWindow_Real( float *pfltInputSamples, float *pfltWindow,
                                     float *pfltDestSamples,  int iLength );



//---------------------------------------------------------------------------
CPROT void FFT_CalcComplexFft(
          int iNrOfPoints,    // N =  number of points in the DFT *AND* in the time domain
          float *pfltRe,      // REX[] = real parts of input and output
          float *pfltIm );    // IMX[] = imaginary parts of input and output
#if(SWI_FFT_NEED_DOUBLE_PRECISION)
CPROT void FFT_CalcComplexFft_Double(
          int iNrOfPoints,    // N =  number of points in the DFT *AND* in the time domain
          double *pfltRe,     // REX[] = real parts of input and output
          double *pfltIm );   // IMX[] = imaginary parts of input and output
#endif // SWI_FFT_NEED_DOUBLE_PRECISION ?
 //  THE FAST FOURIER TRANSFORM - without any whistles and bells .
 //  Upon entry, N contains the number of points in the DFT, REX[ ] and
 //   IMX[ ] contain the real and imaginary parts of the input.
 //   All signals run from 0 to N-1.
 //  Upon return, REX[ ] & IMX[ ] contain the DFT output. More info in *.cpp !

//---------------------------------------------------------------------------
CPROT void FFT_CalcComplexInverseFft(
       int iNrOfPoints, // N  number of points in the IDFT (?) .. IN THE TIME DOMAIN
       float *pfltRe,   // REX[] = input: real parts of frequency domain, result: re(time domain)
       float *pfltIm ); // IMX[] = input: imag. part of frequency domain, result: im(time domain)
#if(SWI_FFT_NEED_DOUBLE_PRECISION)
CPROT void FFT_CalcComplexInverseFft_Double(
       int iNrOfPoints, // N  number of points in the IDFT (?) .. IN THE TIME DOMAIN
       double *pfltRe,  // REX[] = input: real parts of frequency domain, result: re(time domain)
       double *pfltIm); // IMX[] = input: imag. part of frequency domain, result: im(time domain)
#endif // SWI_FFT_NEED_DOUBLE_PRECISION ?
 //  INVERSE FFT FOR COMPLEX SIGNALS  - inspired by [SGDSP] TABLE 12-5 .
 //  Upon entry, N contains the number of points in the IDFT, REX[ ] & IMX[]
 //  contain the real & imaginary parts of the complex frequency domain.
 //   Upon return, REX[ ] and IMX[ ] contain the complex time domain.
 //   All signals run from 0 to N-1.


//---------------------------------------------------------------------------
CPROT void FFT_SortComplexFftForIncreasingFreqBins(
          int iNrOfPoints,    // N =  number of points in the DFT *AND* in the time domain
          float *pfltRe,      // REX[] = real parts of input and output
          float *pfltIm );    // IMX[] = imaginary parts of input and output
#if(SWI_FFT_NEED_DOUBLE_PRECISION)
CPROT void FFT_SortComplexFftForIncreasingFreqBins_Double( // similar for double precision
          int iNrOfPoints, double *pfltRe, double *pfltIm );
#endif // SWI_FFT_NEED_DOUBLE_PRECISION ?


//---------------------------------------------------------------------------
CPROT void FFT_CalcRealFft( // REAL samples in the time domain -> COMPLEX frequency bins
          int iNrOfPoints,    // N  number of points in the DFT (details further below!)
          float *pfltRe,      // REX[] = the real input signal, also used as result
          float *pfltIm );    // IMX[] = output, imaginary part
#if(SWI_FFT_NEED_DOUBLE_PRECISION)
CPROT void FFT_CalcRealFft_Double(
          int iNrOfPoints,    // N  number of points in the DFT
          double *pfltRe,     // REX[] = the real input signal, also used as result
          double *pfltIm );   // IMX[] = output, imaginary part
#endif // SWI_FFT_NEED_DOUBLE_PRECISION ?
 //  FFT FOR REAL SIGNALS  - inspired by [SGDSP] TABLE 12-7 .
 //  Upon entry, iNrOfPoints contains the number of points in the DFT,
 //              REX[ ] contains the real input signal,
 //              while values in IMX[ ] are ignored.
 //              These signals run from 0 to iNrOfPoints-1  .
 // Upon return, REX[ ] & IMX[ ] contain the DFT output.
 // ( YHF: The output signals run from  0...iNrOfPoints/2 !
 //   A "1024 point REAL FFT" produces 513(!) POINTS in pfltRe[]
 //                                and 513(!) POINTS in pfltIm[] . )


//---------------------------------------------------------------------------
CPROT void FFT_CalcRealInverseFft(
          int iNrOfPoints, // N  number of points in the IDFT (?!?) .. IN THE TIME DOMAIN !
          float *pfltRe,   // REX[] = real parts of frequency domain, AND result
          float *pfltIm ); // IMX[] = imaginary parts of frequency domain
#if(SWI_FFT_NEED_DOUBLE_PRECISION)
CPROT void FFT_CalcRealInverseFft_Double(
          int iNrOfPoints, // N  number of points in the IDFT (?!?) .. IN THE TIME DOMAIN !
          double *pfltRe,  // REX[] = real parts of frequency domain, AND result
          double *pfltIm); // IMX[] = imaginary parts of frequency domain
#endif // SWI_FFT_NEED_DOUBLE_PRECISION ?
 //  INVERSE FFT FOR REAL SIGNALS  - inspired by [SGDSP] TABLE 12-6 .
 //  Upon entry, N contains the number of points in the IDFT,
 //  REX[ ] and IMX[ ] contain the real & imaginary parts of the frequency domain
 //  running from index 0 to N%/2.  The remaining samples in REX[] and IMX[]
 //  are ignored.
 //  Upon return, REX[0..N-1] contains the real time domain,
 //               IMX[ ] contains zeros.

//---------------------------------------------------------------------------
CPROT float FFT_BuildWindowTable(
           float *pfltWindowTbl, // [out] window table, single or double precision, depending on SWI_FLOAT_PRECISION
           int   iFftSize,         // [in] number of points (usually 2^n)
           int   iWindowFunction); // [in] FFT_WINDOW_HANN, etc etc
  // Builds a table with one of the usual FFT windowing functions,
  //  and returns the AVERAGE of that window (which is usually ~ 0.5) .


#endif // ndef _FFT_API_H_

/* EOF < SoundMaths.h > */
