//------------------------------------------------------------------------
//  C:\cbproj\Icom_CW_Keyer\Errors.c
//
//  This file contains conversion functions for all the error codes
//       that can be caused by Windows itself,
//       and by a few functions of the application itself.
// Note that the error codes in Errors.h are all POSITIVE !
//------------------------------------------------------------------------

#include <windows.h>     // some error codes are defined HERE !

#pragma hdrstop          // no precompiled headers after this point

#include "StringLib.h"   // replace strncpy() by SL_strncpy(), because strncpy() sucks !
#include "Errors.h"      // "my" error codes are defined here..


//---------------------------------------------------------------------------
char *ErrorCodeToString(int error_code )
   // Most error codes are defined in ErrorCodes.h,
   //      some are application specific, some originate from Win32 API calls.
   //      Some may even be compatible with those returned by M$'s "GetLastError()".
   //      (but don't bet on that; for errors used by the "operating system",
   //       ErrorCodeToString() internally uses the "FormatMessage()" function,
   //       but without the triple-stupid '\n' or '\r' in the result.
   //
{
 static char unknown_msg[128];  // evil static, hopefully rarely necessary

  if( error_code < 0 )
   { // caller forgot to make a NEGATIVE function return value positive
     // to turn the NEGATIVE ERROR RESULT into a POSITIVE ERROR CODE :
     error_code = -error_code;  // .. bear with him .. :)
   }

  switch(error_code)
   {
     case NO_ERROR            :   return("no error");

     // general "wave" errors.  See "ErrorCodes.h" and <MMSYSTEM.H>:
     case WAVERR_BADFORMAT    :   return("unsupported wave format");
     case WAVERR_STILLPLAYING :   return("still something playing");
     case WAVERR_UNPREPARED   :   return("header not prepared");
     case WAVERR_SYNC         :   return("device is synchronous");

     // input wave file error codes
     case WAVIN_ERR_OPEN     :    return("can't open wave file or COM port for input");
     case WAVIN_ERR_NOTWAVE  :    return("file is not a RIFF wave type");
     case WAVIN_ERR_INVALID  :    return("invalid wave file");
     case WAVIN_ERR_NODATA   :    return("no data in file");
     case WAVIN_ERR_NOTSUPPORTED: return("not a supported data type");
     case WAVIN_ERR_READING :     return("error reading data from file");
     case WAVIN_ERR_NOTOPEN :     return("tried to read but file is not open");

     // output wave file error codes
     case WAVOUT_ERR_OPEN   :     return("can't open wave file for output");
     case WAVOUT_ERR_WRITING:     return("error writing to wave file");
     case WAVOUT_ERR_NOTOPEN:     return("tried to write and file is not open");

     // Soundcard input error codes
     case SOUNDIN_ERR_NOTOPEN:    return("tried to read but input device is not open");
     case SOUNDIN_ERR_OVERFLOW:   return("input buffers overflowed");
     case SOUNDIN_ERR_HIGH_WATER: return("input buffer critically full");
     case SOUNDIN_ERR_TIMEOUT :   return("timed out waiting for input buffers");  // 2015-01-28 : Shit happens. (in Sound.cpp:CSound::WaitForInputData)

     // Soundcard output error codes
     case SOUNDOUT_ERR_NOTOPEN:   return("tried to write but output device is not open");
     case SOUNDOUT_ERR_UNDERFLOW: return("output buffers underflowed");
     case SOUNDOUT_ERR_OVERFLOW:  return("output buffers overflowed"); // can only occur with NON-BLOCKING audio output
     case SOUNDOUT_ERR_LOW_WATER: return("output buffer critically empty");
     case SOUNDOUT_ERR_TIMEOUT:   return("timed out waiting for output buffers");

     // Less "specific" file I/O errors (details only AS STRINGS in the error log):
     case ERROR_COULD_NOT_CREATE: return "failed to 'create file' or open device";
     case ERROR_GUESS_PORT_IS_OCCUPIED: return "failed to open port; maybe OCCUPIED ?";
     case ERROR_COULD_NOT_OPEN:   return "failed to open 'file' or similar";
     case ERROR_COULD_NOT_READ:   return "failed to read 'file' or similar";
     case ERROR_COULD_NOT_WRITE:  return "failed to write 'file' or similar";

     // general multimedia error values as defined in mmsystem.h
     case MMSYSERR_ERROR:         return("unspecified MMSYS error");
          // 2015-10-20: Shit happens, over and over again. Got this "unspecified MMSYS error"
          //             on a Win8.1 system after the audio OUTPUT (to device 'MAGIC LCD 190 (Intel(R) Display')
          //             had been running for hours. SpecLab did't manage to kick the damned thing alive
          //             without yet another modification in Sound.cpp .
          //   The error seemed to happen quite often after being paused
          //   via debugger !
     case MMSYSERR_BADDEVICEID:   return("device ID out of range");
     case MMSYSERR_NOTENABLED:    return("driver failed enable");
     case MMSYSERR_ALLOCATED :    return("device already allocated");
     case MMSYSERR_INVALHANDLE:   return("device handle is invalid");
     case MMSYSERR_NODRIVER   :   return("no device driver present");
     case MMSYSERR_NOMEM      :   return("memory allocation error");
     case MMSYSERR_NOTSUPPORTED:  return("function isn't supported");
     case MMSYSERR_BADERRNUM   :  return("error value out of range");
     case MMSYSERR_INVALFLAG   :  return("invalid flag passed");
     case MMSYSERR_INVALPARAM  :  return("invalid parameter passed");
     case MMSYSERR_HANDLEBUSY  :  return("handle being used");
     case MMSYSERR_NODRIVERCB  :  return("driver does not call DriverCallback");

     // Other errors...
     case ERROR_FROM_AUDIO_IO_DLL  : return ("error from audio-I/O-DLL");
     case ERROR_DULL_POINTER_ASSIGNMENT: return "dull pointer assignment"; // null pointer, dull pointer (pointer into 'nirvana'), illegal address, bad alignment .. whatever you can imagine !
     case ERROR_WORD_AT_ODD_ADDRESS: return ("word at odd address");
     case ERROR_THREAD_FAILED:       return ("worker thread failed");
     case ERROR_WITH_ASIO_DRIVER:    return ("error with ASIO driver");
     case ERROR_UNKNOWN_CMD_ARGUMENT:  return ("unknown command argument");
     case ERROR_ILLEGAL_CMD_PARAMETER: return ("illegal parameter value");
     case ERROR_CMD_COMBI_UNSUPPORTED:    return ("command combination not supported");
     case ERROR_DATA_TYPE_NOT_SUPPORTED:  return ("data type not supported");
     case ERROR_OUTPUT_FILE_NOT_CONSUMED: return ("output file not consumed");
     case ERROR_OPEN_OR_CREATE_FAILED: return ("Open or Create failed");
     case ERROR_FILE_DOESNT_EXIST:     return ("File doesnt exist");
     case ERROR_FILE_WRITE_INCOMPLETE: return ("File write incomplete"); // .. for example, when trying to SEND to a serial port via FileWrite(), WROTE less than expected


     // "Internet"-related errors (added 2011-12-04, first used in YHF_Init.c) :
     case INET_ERROR_SOCKET_NOT_OPEN: return "Socket not open";
     case INET_ERROR_COULDNT_CONNECT: return "Couldn't connect";
     case INET_ERROR_BAD_URL_SYNTAX : return "Bad URL syntax";
     case INET_ERROR_CONNECTION_RESET: return "connection reset";
     case INET_ERROR_CONNECTION_ABORT: return "connection aborted";
     case INET_ERROR_CANNOT_SEND    : return "cannot send";
     case INET_ERROR_WOULD_BLOCK    : return "would block";
     case INET_ERROR_UNREACHABLE    : return "unreachable";
     case INET_ERROR_BIND_FAILED    : return "bind() failed";
     case INET_ERROR_ADDRESS_IN_USE : return "Address already in use";
     case INET_ERROR_SOCKET_FAILED  : return "Other Winsock error";
     case INET_ERROR_NOT_AVAILABLE  : return "Socket function n/a";
     case INET_ERROR_INVALID_ARGUMENT: return "Invalid argument";

     case MEMORY_ERROR         :     return("Memory error");

     default:  // unknown MMSYSTEM error code (or any other "os-specific" error):
      {    // try to make the best of it, maybe the OS knows what it means..
        LPVOID lpMsgBuf;
        DWORD  dwLength;

        wsprintf(unknown_msg, "unknown error #%d", error_code);
        if( error_code < MYERR_BASE ) // It's NOT a self-defined error code,
         {                            // so maybe Windows can convert it into a string:
           /* Convert the GetLastError()-code into a readable string: */
           if( FormatMessage(FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM,
               NULL,
               error_code,   // 'int' has 32 bit, here: DWORD expected
               0,
              (LPTSTR)&lpMsgBuf,
               0,
               NULL) == 0)
            { /* failed to format the message */
               wsprintf(unknown_msg, "unknown error #%d", error_code);
            }
           else // FormatMessage() successfull ..
            {
              // Convert the string and free the message's buffer.
              SL_strncpy(unknown_msg, (char*)lpMsgBuf, sizeof(unknown_msg)-1);

              // Free the buffer which has been allocated somehow in FormatMessage(!)
              LocalFree( lpMsgBuf );

              // If the result string contains this foolish CR and/or NL characters at the end,
              // trow them out.. they are UTTERLY USELESS in 99.9999 % of all cases !
              dwLength = strlen(unknown_msg);
              if( (dwLength>0) && (unknown_msg[dwLength-1] == '\r') )  // crazy1
               { --dwLength;
                 unknown_msg[dwLength] =  '\0';
               }
              if( (dwLength>0) && (unknown_msg[dwLength-1] == '\n') )  // crazy2
               { --dwLength;
                 unknown_msg[dwLength] =  '\0';
               }
              if( (dwLength>0) && (unknown_msg[dwLength-1] == '\r') )  // crazy3
               { --dwLength;
                 unknown_msg[dwLength] =  '\0';
               }
            }
         } // end if( error_code < MYERR_BASE )
        return unknown_msg;
      }
   }
} // end ErrorCodeToString()
