// File:  C:\cbproj\Remote_CW_Keyer\Elbug.h
// Date:  2023-11-22
// Author:  Wolfgang Buescher (DL4YHF)
// Purpose: Emulation of an electronic Morse keyer aka 'Elbug' on a PC .
//          First used as a proof-of-concept in the author's
//          "Remote CW Keyer" project (using SERIAL PORTS to poll the
//          paddle contacts, and to drive the remotely controlled radio's
//          CW-keying input).
// Try to keep ANY DEPENDENCY on 'CwKeyer.h', or even the 'Remote Keyer GUI',
// or any proprietary stuff like "dot net", VCL, and Qt out of this module !

#ifndef  ELBUG_H_INCLUDED
# define ELBUG_H_INCLUDED 1

#ifndef _DATA_TYPES_H_
# include "yhf_type.h"  // old fashioned stuff like BYTE, WORD, DWORD, etc^10
#endif

#ifndef DEBOUNCER_H_INCLUDED
# include "Debouncer.h" // 'debouncer' for paddle inputs (T_Debouncer in T_ElbugInstance)
#endif


//---------------------------------------------------------------------------
// Constants
//---------------------------------------------------------------------------


#ifndef TRUE
# define TRUE  1
# define FALSE 0
#endif


  //
  // Storage format of "spaces", "control characters" and "printable character"
  //         (in Morse code, NOT converted to ASCII)
  // ---------------------------------------------------------------------------------
  //  The most significant bits define the code type.
  //     Bit 15: 1 = "this is a SPACE (=pause) or CONTROL character", in this case:
  //        Bit 14, Bit 13 = control character type:
  //        0x = SPACE (=pause,     bits5..0 contain the length in DOTS (max.63)
  //        10 = extra long "dash", bits4..0 contain the length in DOTS (max.31)
  //        11 = future reserve,    bits4..0 contain 31 possible codes
  //     Bit 15: 0 = "this is a transmittable character (not SPACE or CONTROL)".
  //        Bits 14..0 contain dashes or dots,
  //             with leading 1="Startbit" before the "dash/dot-matrix".
  //             Bit values the dash/dot-matrix : 0=dot 1=dash .
  //             Bit 0 always contains the LAST TRANSMITTED dash/dot.
  // See definitions below for some examples of CW 'patterns' stored in memory.
  // Originally written in binary format (PIC assembler), converted to hex
  // so even the ancient Borland C compiler can handle them.
#define CW_CHR_0      0x003F // b'00111111'  ; "0"
#define CW_CHR_1      0x002F // b'00101111'  ; "1"
#define CW_CHR_2      0x0027 // b'00100111'  ; "2"
#define CW_CHR_3      0x0023 // b'00100011'  ; "3"
#define CW_CHR_4      0x0021 // b'00100001'  ; "4"
#define CW_CHR_5      0x0020 // b'00100000'  ; "5"
#define CW_CHR_6      0x0030 // b'00110000'  ; "6"
#define CW_CHR_7      0x0038 // b'00111000'  ; "7"
#define CW_CHR_8      0x003C // b'00111100'  ; "8"
#define CW_CHR_9      0x003E // b'00111110'  ; "9"

#define CW_CHR_A      0x0005 // b'00000101'  ; 'A'
#define CW_CHR_B      0x0018 // b'00011000'  ; 'B'
#define CW_CHR_C      0x001A // b'00011010'  ; 'C'
#define CW_CHR_D      0x000C // b'00001100'  ; 'D'
#define CW_CHR_E      0x0002 // b'00000010'  ; 'E'
#define CW_CHR_F      0x0012 // b'00010010'  ; 'F'
#define CW_CHR_G      0x000E // b'00001110'  ; 'G'
#define CW_CHR_H      0x0010 // b'00010000'  ; 'H'
#define CW_CHR_I      0x0004 // b'00000100'  ; 'I'
#define CW_CHR_J      0x0017 // b'00010111'  ; 'J'
#define CW_CHR_K      0x000D // b'00001101'  ; 'K'
#define CW_CHR_L      0x0014 // b'00010100'  ; 'L'
#define CW_CHR_M      0x0007 // b'00000111'  ; 'M'
#define CW_CHR_N      0x0006 // b'00000110'  ; 'N'
#define CW_CHR_O      0x000F // b'00001111'  ; 'O'
#define CW_CHR_P      0x0016 // b'00010110'  ; 'P'
#define CW_CHR_Q      0x001D // b'00011101'  ; 'Q'
#define CW_CHR_R      0x000A // b'00001010'  ; 'R'
#define CW_CHR_S      0x0008 // b'00001000'  ; 'S'
#define CW_CHR_T      0x0003 // b'00000011'  ; 'T'
#define CW_CHR_U      0x0009 // b'00001001'  ; 'U'
#define CW_CHR_V      0x0011 // b'00010001'  ; 'V'
#define CW_CHR_W      0x000B // b'00001011'  ; 'W'
#define CW_CHR_X      0x0019 // b'00011001'  ; 'X'
#define CW_CHR_Y      0x001B // b'00011011'  ; 'Y'
#define CW_CHR_Z      0x001C // b'00011100'  ; 'Z'

#define CW_CHR_EQUAL  0x0031 // b'00110001'  ; '='  (-...-)  often used as "separator"
#define CW_CHR_HYPHEN 0x0061 // b'01100001'  ; '-'  (-....-)
#define CW_CHR_POINT  0x0055 // b'01010101'  ; '.'  (-.-.-.)
#define CW_CHR_SLASH  0x0032 // b'00110010'  ; '/'  (-..-.)
#define CW_CHR_QSTN   0x004C // b'01001100'   '?'  (..--..)
#define CW_CHR_COMMA  0x0073 // b'01110011'   ','  (--..--)
#define CW_CHR_COLON  0x0078 // b'01111000'   ':'  (---...)
#define CW_CHR_OPEN_B  0x006D // b'01101101'  '('  (-.--.-) opening bracket, rarely ever seen
#define CW_CHR_CLOSE_B 0x0036 // b'00110110'  ')'  (-.--.) opening bracket, rarely ever seen
#define CW_CHR_AT      0x005A // b'01011010'  '@'  (.--.-.) "facilitates sending e-mail addresses by Morse code"
#define CW_CHR_APOSTROPHE 0x005E // b'01011110'  (.----.) Apostrophe

        // "non-Latin extensions" like German Umlauts =AE, =OE, =UE :
#define CW_CHR_GERMAN_AE 0x0015 // b'00010101' (.-.-) German Umlaut  (AE)
#define CW_CHR_GERMAN_OE 0x001E // b'00011110' (---.) German Umlaut  (OE)
#define CW_CHR_GERMAN_UE 0x0013 // b'00010011' (..--) German Umlaut  (UE)


        // Special Morse code sequences ('prosigns') sent as strings
        //               of characters with no intercharacter space:
#define CW_CHR_AR     0x002A // b'00101010'  ; 'AR' (.-.-.)
#define CW_CHR_BK     0x00C5 // b'11000101'  ; 'BK' (-...-.-) rarely used, everyone understands "bk" as two characters
#define CW_CHR_SK     0x0045 // b'01000101'  ; 'SK' (...-.-)
#define CW_CHR_KA     0x0035 // b'00110101'  ; 'KA' (-.-.-)
#define CW_CHR_KN     0x0036 // b'00110110'  ; 'KN' (-.--.)
#define CW_CHR_EOM    0x005F // b'01011111'  ; 'EOM' (.-----) used for "partitions" of msg in DL4YHF's "QRP Keyer"
#define CW_CHR_NNN    0x006A // b'01101010'  ; 'NNN' (-.-.-.) replaced by serial number in DL4YHF's "QRP Keyer"
//ex:#define CW_CHR_ANN 0x005A // b'01011010'  ; 'ANN' (.--.-.) advance to next number  in DL4YHF's "QRP Keyer"

#define CW_CHR_SPACE  0x8000 // ' ' (pause length in "dots" may be added,
      // at least when recording into memory for playback with 'individual spacing')


//---------------------------------------------------------------------------
// Data Types
//---------------------------------------------------------------------------


typedef struct t_ElbugInstance // everthing we need for a single 'Elbug' instance:
{

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // "configuration parameters" (must be filled out be the application before start)
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  struct { // .cfg = "everything that needs to be SAVED between sessions" (and included in a hash value):
     int iDotTime_ms;   // duration of a dot, measured in MILLISECONDS (!)
           // (use Elbug_WordsPerMinuteToDotTimeInMilliseconds() to convert..)
     int iMorseKeyType; // may contain one of the following:
#    define ELBUG_TYPE_PASSIVE  0 // no keying of the output(s), but e.g. "manual" control via CwKeyer_SetDigitalOutput()
#    define ELBUG_TYPE_STRAIGHT 1 // straight key for INPUT (only one digital input)
#    define ELBUG_TYPE_IAMBIC_B 2 // external paddle, "Iambic mode B" emulated by software, mostly LEVEL-TRIGGERED flip-flops (*)
#    define ELBUG_TYPE_IAMBIC_A 3 // external paddle, "Iambic mode A" emulated by software, mostly EDGE-TRIGGERED flip-flops (*)
#    define ELBUG_TYPE_IAMBIC_NO_MEM 4 // "Basic Iambic Keyer without dot/dash memory" as described by DJ5IL (*)
           // Note: Compatible definitions may also be in CwKeyer.h,
           //       but none of those header files shall depend on the other.
           // (*) See literature quoted in Elbug.c, mostly from DJ5IL's
           //     article "All about Squeeze-Keying", which explains
           //     the subtle differences between the "Iambic" modes.
   } cfg; // <- end of the part that needs to be stored permanently (between sessions)


  int iState;  // State for the keyer state machine (simplified) :
  // Former PIC keyer states for WAITING and switching into SLEEP mode :
# define ES_WAITING           0  // waiting for anything to happen...
# define ES_SEND_DOT          1  // sending a dot
# define ES_SEND_DASH         2  // sending a dash
# define ES_SEND_GAP          3  // sending a gap of at least ONE DOT LENGTH
# define ES_SEND_GAP_2        4  // sending a gap without a minimum duration
                                 // (while "measuring the gap" to detect the end
                                 //  of a character, or even an inter-word SPACE)
# define ES_WAIT_FOR_SPACE    5  // wait for the end of a "minimum gap" to emit a SPACE character

  // Elbug_Handler() doesn't call any functions to "drive the digital output"
  // for the transmitter's PTT and Morse key input. Instead, it only sets
  // the following flags, and lets the caller of Elbug_Handler() do the
  // actual 'output port switching', or whatever:
  BOOL fMorseOutput;  // "Morse output" driving the side tone, modulate the TX,
                      // and (possibly delayed by the RX/TX switching time)
                      // also the TRANSMITTER'S keying input.
  BOOL fPauseTransmitter; // global flag to "pause transmission", set on 'operator panic' in the GUI (ESCAPE key)

  T_Debouncer dashDebouncer, dotDebouncer;
  long i32CountdownForDashOrDot_us; // "countdown timer", in microseconds, for the internal timing
  WORD wShiftReg;  // shift register to decode the character currently being sent.
  BYTE bFlags;     // BITWISE COMBINATION of the following:
# define ELBUG_FLAG_STORED_DASH 0x01 // "dash memory" for squeezing / Iambic B (many decades ago, a flip-flop)
# define ELBUG_FLAG_STORED_DOT  0x02 // "dot memory"  for squeezing / ...
# define ELBUG_FLAG_LAST_SENT_DASH 0x04 // "the last element actually sent was a DASH"
     // (so if both ELBUG_FLAG_STORED_DASH *and* ELBUG_FLAG_STORED_DOT are set,
     //  the next element to send is a DOT; otherwise, it's a DASH for "iambic" keying)
# define ELBUG_FLAG_EMITTED_SPACE  0x08 // flag to avoid emitting multiple SPACE CHARACTERS for the GUI
  BOOL fCurrDashSignal, fCurrDotSignal; // "current states" of the paddle contacts, updated in Elbug_Handler()
  BOOL fPrevDashSignal, fPrevDotSignal; // "previous states" of the paddle contacts to detect edges

  int  iHandlerResult; // bitwise ORed to the value returned by Elbug_Handler()

} T_ElbugInstance;


//---------------------------------------------------------------------------
// Function prototypes
//---------------------------------------------------------------------------
#ifndef CPROT   // for peaceful co-existence of C and C++ ...
#ifdef __cplusplus
 #define CPROT extern "C"
#else
 #define CPROT
#endif  // not "cplusplus" ?
#endif // ndef CPROT ?


CPROT int  Elbug_WordsPerMinuteToDotTimeInMilliseconds( int iWPM );
CPROT int  Elbug_DotTimeInMillisecondsToWordsPerMinute( int iDotTime_ms ); // <- needed for the "GUI"
CPROT void Elbug_InitInstanceWithDefaults( T_ElbugInstance *pElbug );
CPROT int  Elbug_Handler( T_ElbugInstance *pElbug, BOOL fDotInput, BOOL fDashInput, long nMicrosecondsSinceLastCall );
  // Values returned by Elbug_Handler() : One of the CW_CHR-codes (More code pattern
  // emitted at the end of a single character or space, ranging from 1 to 255),
  // or one of the values defined below:
# define ELBUG_RESULT_NOTHING_DECODED  0x0000
# define ELBUG_RESULT_BEGIN_NEW_CHAR   0x4000 // flag "BEGIN of a new character"
  // (bitwise ORed, for storage in the 'Timing Scope' display buffer)

CPROT WORD Elbug_SingleCharToMorseCodePattern( char c ); // single character (ASCII) -> CW_CHR_xxx
CPROT const char *Elbug_MorseCodePatternToASCII( WORD bCwChr ); // CW_CHR_xxx -> ASCII
CPROT WORD Elbug_ParseMorseCodePatternFromASCII(const char **ppszSource); // ASCII string -> CW_CHR_xxx

#endif // ndef ELBUG_H_INCLUDED ?
