// File:  C:\cbproj\Remote_CW_Keyer\Elbug.c
// Date:  2023-11-22
// Author:  Wolfgang Buescher (DL4YHF)
// Purpose: Emulation of an electronic Morse keyer aka 'Elbug' on a PC .
//          First used as a proof-of-concept in the author's
//          "Remote CW Keyer" project (using SERIAL PORTS to poll the
//          paddle contacts, and to drive the remotely controlled radio's
//          CW-keying input).
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
// Literature:
//  "All about Squeeze-Keying" by DJ5IL. Important parts from that:
//  [Lit1] > A correctly programmed iambic keyer with dot/dash-memory
//         > checks for paddle inputs during each dot or dash in type "A"
//         > as well as in type "B". But in type "A" (Curtis-keyer) it checks
//         > for the transient from unpressed to pressed
//         > whereas in type "B" (Accu-keyer) it checks just for the state pressed.
//  [Lit2] > And even in basic iambic mode without dot/dash-memory it does not
//         > check after the end of each dot or dash, but after the
//         > end of each dot- or dash- element which contains
//         > the following space.
//
//  [Lit3] > The behaviour of the Curtis-keyer can be described by the basic
//         > iambic set of instructions together with the following
//         > dot-memory rule: if anytime during generation of a dash-element
//         > the dot-lever changed its state from unpressed to pressed,
//         > generate an extra dot-element.
//
//  [Lit4] > To test a keyer for dot/dash-memory and its iambic type,
//         > set the speed as low as any possible and key an "N" as fast
//         > as possible - both levers must be released before the dash-element
//         > is completed !
//         > With dot-memory the dash is always followed by a dot and you get the
//         > "N", without dot-memory the dot is lost and you get a "T" instead.
//  [Lit5] > Now key an "A" as fast as possible - again, both levers must be
//         > released before the dot-element is completed ! With dash-memory
//         > the dot is always followed by a dash and you get the "A",
//         > without dash-memory the dash is lost and you get an "E" instead.
//         >
//  [Lit6] > Finally squeeze a "K" and release both levers while the second dash
//         > is heard. If you get the "K" the iambic type depends on the
//         > previous dot/dash-memory test: without dot- and dash-memory
//         > the keyer works in plain iambic mode,
//         > with dot- and dash-memory in iambic type "A" (Curtis-keyer).
//  [Lit7] > If you get a "C" instead, the keyer works in iambic type "B"
//         > (Accu-keyer).
//  [Lit8] When to CLEAR dash/dot memories ? This implementation deviates a bit from the above:
//          * At the BEGIN OF ANY DASH OR DOT, *both* dot- and dash memories
//            are cleared (in Elbug_StartDash() or Elbug_StartDot() ).
//          * At the END of a DASH (not at the end of the one-dot-GAP that follows),
//            only the dash-memory (here: ELBUG_FLAG_STORED_DASH) is cleared.
//          * At the END of a DOT (not at the end of the one-dot-GAP that follows),
//            only the dot-memory (here: ELBUG_FLAG_STORED_DOT) is cleared.
//  [Lit9] When to SET dash/dot memories ?
//   |       * During the one-dot-gap after dash or dot, the two memories (flags)
//   |         may be set again: "rising-edge-triggered" in iambic A,
//   |                           "level-triggered" in iambic B.
//   |       * In iambic "B", while sending a DASH (or the one-dot-gap that follows),
//   |          an active dot input STATE sets the dot-memory .
//   |       * In iambic "B", while sending a DOT (or the one-dot-gap that follows),
//   |          an active dash input STATE sets the dash-memory .
//   |__\ Some of these 'Literature' labels in squared brackets are referenced
//      / in the implementation below further, so don't renumber/rename them !
//
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

#include <string.h> // no string functions used here, but memset()
#include "StringLib.h" // "Old school C string library" for embedded systems, etc
#include "Debouncer.h" // 'debouncer' for paddle inputs (T_Debouncer in T_ElbugInstance)
#include "Elbug.h"  // header for THIS module (electronik keyer for an emulated 'elbug')


//----------------------------------------------------------------------------
int Elbug_WordsPerMinuteToDotTimeInMilliseconds( int iWPM ) // only for the GUI / Configuration..
{ // Someone wrote:
  // > In the case of 1 WPM, the duration of a dot is 60 seconds / 50 dots per minute
  // >  = 1.2 seconds per dot.
  // (50 dot-times in the word "PARIS ", including the word-space with 7 dots)
  // Thus:
  if( iWPM > 0 )
   { return (1200 + iWPM/2) / iWPM;
     // ,-----------|_____|
     // '--> ROUND, don't truncate
   }
  else
   { return 1200;
   }
} // end Elbug_WordsPerMinuteToDotTimeInMilliseconds()

//----------------------------------------------------------------------------
int Elbug_DotTimeInMillisecondsToWordsPerMinute( int iDotTime_ms )
  // Inverse to Elbug_WordsPerMinuteToDotTimeInMilliseconds(). Needed for the "GUI".
{
  if( iDotTime_ms > 0 )
   { return (1200 + iDotTime_ms/2) / iDotTime_ms;
     // ,-----------|____________|
     // '--> ROUND, don't truncate
     // Example: [in] iDotTime_ms = 48  -> (1200 + 48/2) / 48 = 25 [WPM]
   }
  else
   { return 1;  // 1 WORD PER MINUTE is very slow, but ok for QRPP beacons
   }
} // end Elbug_DotTimeInMillisecondsToWordsPerMinute()


//----------------------------------------------------------------------------
void Elbug_InitInstanceWithDefaults( T_ElbugInstance *pElbug )
  // Called from the main application (or "GUI") before loading an actual
  // configuration FROM A FILE. The result (in *pConfig) are the "defaults".
{
  memset( pElbug, 0, sizeof(T_ElbugInstance) );
  pElbug->cfg.iDotTime_ms = Elbug_WordsPerMinuteToDotTimeInMilliseconds( 25 );

} // end Elbug_InitInstanceWithDefaults()

//---------------------------------------------------------------------------
static void Elbug_ReloadCountdownForDashOrDot( T_ElbugInstance *pElbug, int nDotTimes )
{
  // Due to the calling interval of Elbug_Handler() with approximately 2 ms,
  // simply setting
  //   pElbug->i32CountdownForDashOrDot_us = pElbug->cfg.iDotTime_ms * 1000 * nDotTimes
  // would cause slipping of timestamps. As in CwGen_ReloadCountdownForDashOrDot(),
  // fixed by ADDING the new element duration to whatever remained
  // in pElbug->i32CountdownForDashOrDot_us (up to 3000 microseconds) :
  if( (pElbug->i32CountdownForDashOrDot_us < -3000) || (pElbug->i32CountdownForDashOrDot_us > 3000) )
   { pElbug->i32CountdownForDashOrDot_us = 0;
   }
  pElbug->i32CountdownForDashOrDot_us += pElbug->cfg.iDotTime_ms * 1000 * nDotTimes;

} // end Elbug_ReloadCountdownForDashOrDot()


//---------------------------------------------------------------------------
static void Elbug_StartDash( T_ElbugInstance *pElbug ) // Starts sending a DASH,
  // and clears the "stored elements" [Lit8]. Only called from Elbug_Handler() .
{
  pElbug->fMorseOutput = TRUE;
  Elbug_ReloadCountdownForDashOrDot( pElbug, 3 );
  pElbug->bFlags &= ~(ELBUG_FLAG_STORED_DASH | ELBUG_FLAG_STORED_DOT); // [Lit8]
  // Note: ELBUG_FLAG_STORED_DASH is also cleared at THE END of a dash !
  if( pElbug->wShiftReg==1 ) // "start bit" still in its initial position ?
   { // Let the caller of Elbug_Handler() know we just STARTED a new character:
     pElbug->iHandlerResult |= ELBUG_RESULT_BEGIN_NEW_CHAR; // indicate "BEGIN of a new character"
                                                            // in the 'Timing Scope'.
   }
  pElbug->wShiftReg = (pElbug->wShiftReg << 1) | 1; // for the "decoder"..
  // ,---------------------------------------------'
  // '--> 0 = shift in a "dot", 1 = shift a "dash" into the decoder's shift register
  pElbug->iState = ES_SEND_DASH;  // new state : "sending a dash"
}

//---------------------------------------------------------------------------
static void Elbug_StartDot( T_ElbugInstance *pElbug ) // Starts sending a DOT,
  // and clears the "stored elements" [Lit8]. Only called from Elbug_Handler() .
{
  pElbug->fMorseOutput = TRUE;
  Elbug_ReloadCountdownForDashOrDot( pElbug, 1 );
  pElbug->bFlags &= ~(ELBUG_FLAG_STORED_DASH | ELBUG_FLAG_STORED_DOT); // [Lit8]
  if( pElbug->wShiftReg==1 ) // "start bit" still in its initial position ?
   { // Let the caller of Elbug_Handler() know we just STARTED a new character:
     pElbug->iHandlerResult |= ELBUG_RESULT_BEGIN_NEW_CHAR; // indicate "BEGIN of a new character"
                                                            // for the 'Timing Scope'.
   }
  pElbug->wShiftReg = (pElbug->wShiftReg << 1) | 0; // for the "decoder"..
  // ,---------------------------------------------'
  // '--> 0 = shift in a "dot", 1 = shift a "dash" into the decoder's shift register
  pElbug->iState = ES_SEND_DOT;  // new state : "sending a dot"
}

//---------------------------------------------------------------------------
static void Elbug_StartGap(T_ElbugInstance *pElbug) // start a gap, with duration of AT LEAST one "dot"
  // Only called from Elbug_Handler() .
{
  pElbug->fMorseOutput = FALSE;
  Elbug_ReloadCountdownForDashOrDot( pElbug, 1 );
  pElbug->iState = ES_SEND_GAP;  // new state : "sending a gap"
}

//---------------------------------------------------------------------------
static BOOL Elbug_StartDashOrDot(T_ElbugInstance *pElbug) // called from Elbug_Handler()
  // in certain states to start sending a dash or dot.
  // [in] pElbug->fCurrDashSignal, pElbug->fCurrDotSignal :
  //         important for DJ5IL's "basic iambic mode without dot/dash memory".
  // [in] pElbug->bFlags : ELBUG_FLAG_STORED_DASH, ELBUG_FLAG_STORED_DOT,
  //                       ELBUG_FLAG_LAST_SENT_DASH .
  //                       ,----------------'
  //                       '-> flag to alternate between DASHES and DOTS
  //                           while BOTH paddle contacts are closed ("squeeze") .
  // [out] pElbug->iState: May switch to ES_SEND_DASH or ES_SEND_DOT here.
  // [return value] : TRUE when a dash or dot was started,
  //                  FALSE when there was neither a dash nor dot to send.
  //                        In this case, it's THE CALLER's DUTY
  //                        to switch to the next appropriate state.
{

  // What to send next depends on the "dash and dot memory".
  // If *BOTH* (dash *AND* dot) have been stored, send the complementary
  // element. Result: the Iambic "didahdidadiah.." when BOTH contacts are closed.
  if( pElbug->bFlags & ELBUG_FLAG_LAST_SENT_DASH )
   { // Last sent a DASH, so priority to send a DOT (for any kind of IAMBIC keying) ->
     if( (pElbug->bFlags & ELBUG_FLAG_STORED_DOT) || pElbug->fCurrDotSignal )
      { Elbug_StartDot(pElbug); // start sending a dot (and clear the "stored elements")
        return TRUE;
      }
     else // lower priority : after a dash, send another dash ?
     if( (pElbug->bFlags & ELBUG_FLAG_STORED_DASH) || pElbug->fCurrDashSignal )
      { Elbug_StartDash(pElbug); // start sending a dash (and clear the "stored elements")
        return TRUE;
      }
   }
  else // ! ELBUG_FLAG_LAST_SENT_DASH :
   { // Last sent a DOT, so priority to send a DASH (..for any kind of IAMBIC keying) ->
     if( (pElbug->bFlags & ELBUG_FLAG_STORED_DASH) || pElbug->fCurrDashSignal )
      { Elbug_StartDash(pElbug); // start sending another dash (and clear the "stored elements")
        return TRUE;
      }
     else // lower priority : after a dot, send another dot ?
     if( (pElbug->bFlags & ELBUG_FLAG_STORED_DOT) || pElbug->fCurrDotSignal )
      { Elbug_StartDot(pElbug); // start sending another dot (and clear the "stored elements")
        return TRUE;
      }
   }
  // Arrived here -> Neither dot or dash to send -> begin of a "longer gap" ...
  // depending on the current state, the caller calls e.g.
  //  Elbug_StartGap2(pElbug) to start a "subsequent" gap, etc.
  return FALSE; // neither a dash nor a dot "waiting to be sent" !

} // end Elbug_StartDashOrDot()

//---------------------------------------------------------------------------
static void Elbug_StartGap2(T_ElbugInstance *pElbug) // start a subsequent gap, WITHOUT a minimum duration
  // Only called from Elbug_Handler() .
{
  pElbug->fMorseOutput = FALSE;
  Elbug_ReloadCountdownForDashOrDot( pElbug, 1 ); // <- if this SECOND dot-time expires, it's time to emit the character
  pElbug->iState = ES_SEND_GAP_2;  // new state : "sending a gap WITHOUT a minimum duration"
} // end Elbug_StartGap2()

//---------------------------------------------------------------------------
static void Elbug_OnEndOfChar(T_ElbugInstance *pElbug) // called from Elbug_Handler()
  // when a gap of TWO(!) dot-lengths has been detected, after the end
  // of the last transmitted dash or dot -> end of character .
{
  pElbug->fMorseOutput = FALSE;
  // TWO dot-times have already expired, so if TWO or THREE MORE dot times expire,
  // it's time to emit a SPACE character. Someone specified (or at least explained):
  //   > The space between letters is 3 time units.
  //   > The space between words is 7 time units.
  // In this implementation, a "time unit" is pElbug->cfg.iDotTime_ms [N milliseconds] .
  // To allow "tolerance" for operators using a slightly shorter inter-WORD-space,
  // a SPACE CHARACTER will be emitted by the keyer (when sending) after FIVE time-units.
  // An advanced keyer firmware may extend the 'word space' to those SEVEN time-units.
  Elbug_ReloadCountdownForDashOrDot( pElbug, 3 ); // two dot-times already expired, so wait for at least THREE additional ones
  pElbug->iState = ES_WAIT_FOR_SPACE;  // new state : "wait for the minimum gap to emit a SPACE character"
  pElbug->wShiftReg=1; // prepare decoding of the next transmitted character
                 //  '--> that's the "start bit" in its initial position,
                 //       bitwise shifted LEFT later for each new dash or dot.
} // end Elbug_OnEndOfChar()

//---------------------------------------------------------------------------
static void Elbug_EmitSpace(T_ElbugInstance *pElbug) // called from Elbug_Handler()
  // when a total gap of at least FIVE(?) dot-lengths has been detected, after the end
  // of the last transmitted dash or dot -> emit a SPACE character .
{
  pElbug->fMorseOutput = FALSE;
  pElbug->iState = ES_WAITING;  // new state : "wait for anything to happen"
                     // (begin of the next character, i.e. first dash or dot)
  if( !(pElbug->bFlags & ELBUG_FLAG_EMITTED_SPACE) )  // only emit ONE "decoded" space character in a row
   { pElbug->bFlags |= ELBUG_FLAG_EMITTED_SPACE;
     pElbug->iHandlerResult = CW_CHR_SPACE;
   }
} // end Elbug_EmitSpace()

//---------------------------------------------------------------------------
static void Elbug_StoreDash_EdgeTriggered(T_ElbugInstance *pElbug)
  // Checks for TRANSIENT on DASH from unpressed to pressed (positive edge-triggered)
  //  [Lit1] > .. in type "A" (Curtis-keyer) it checks
  //         > for the transient from unpressed to pressed
  //         > whereas in type "B" (Accu-keyer) it checks just for the state pressed.
{ if( pElbug->fCurrDashSignal && (!pElbug->fPrevDashSignal) ) // active edge on the dash-contact (JUST been closed) ?
   { pElbug->bFlags |= ELBUG_FLAG_STORED_DASH; // similar as the ETM-5C's "dash memory" (flipflop)
   }
} // end Elbug_StoreDash_EdgeTriggered()

//---------------------------------------------------------------------------
static void Elbug_StoreDot_EdgeTriggered(T_ElbugInstance *pElbug)
  // Checks for TRANSIENT on DOT from unpressed to pressed (positive edge-triggered)
{ if( pElbug->fCurrDotSignal && (!pElbug->fPrevDotSignal) )   // active edge on the dot-contact (JUST been closed) ?
   { pElbug->bFlags |= ELBUG_FLAG_STORED_DOT;  // similar as the ETM-5C's "dot memory" (flipflop)
   }
} // end Elbug_StoreDot_EdgeTriggered()


//---------------------------------------------------------------------------
static void Elbug_StoreDash_LevelTriggered(T_ElbugInstance *pElbug) // called from Elbug_Handler()
  // In contrast to mode "A"'s the EDGE- (transient) detection,
  // Iambic "B" (Accu-keyer) sets the 'stored dash' flag
  // depending on the paddle input STATE (level-triggered), not TRANSITION:
{ if( pElbug->fCurrDashSignal ) // DASH-contact currently closed (active) ?
   { pElbug->bFlags |= ELBUG_FLAG_STORED_DASH; // similar as the ETM-5C's "dash memory" (flipflop)
   }
} // end Elbug_StoreDash_LevelTriggered()

//---------------------------------------------------------------------------
static void Elbug_StoreDot_LevelTriggered(T_ElbugInstance *pElbug) // called from Elbug_Handler()
  // In contrast to mode "A"'s the EDGE- (transient) detection,
  // Iambic "B" (Accu-keyer) sets the 'stored dot' flag
  // depending on the paddle input STATE (level-triggered), not TRANSITION:
{ if( pElbug->fCurrDotSignal )  // DOT-contact currently closed (active) ?
   { pElbug->bFlags |= ELBUG_FLAG_STORED_DOT;  // similar as the ETM-5C's "dot memory" (flipflop)
   }
} // end Elbug_StoreDot_LevelTriggered()

//---------------------------------------------------------------------------
int Elbug_Handler( T_ElbugInstance *pElbug,
       BOOL fDotSignal,   // [in] TRUE = dot-contact currently closed, FALSE = open
       BOOL fDashSignal,  // [in] TRUE = dash-contact currently closed, FALSE = open
       long nMicrosecondsSinceLastCall ) // [in] ideally 2000 us, if windoze permits !
  // Values returned by Elbug_Handler() : One of the CW_CHR-codes (More code
  // pattern emitted at the end of a single character or space),
  // or one of the ELBUG_RESULT_.. values defined in Elbug.h, for example
  //               ELBUG_RESULT_NOTHING_DECODED  ( == 0 ) .
  // Called periodically from a time-critical 'worker thread'.
{
  pElbug->iHandlerResult = ELBUG_RESULT_NOTHING_DECODED;
  pElbug->fCurrDashSignal= fDashSignal;
  pElbug->fCurrDotSignal = fDotSignal;

  if( pElbug->i32CountdownForDashOrDot_us > 0 )  // software timer still running ?
   {  pElbug->i32CountdownForDashOrDot_us -= nMicrosecondsSinceLastCall; // let it COUNT DOWN,
      // until it stops at zero or with a "slightly negative value".
   }
  switch( pElbug->iState )
   {
     case ES_WAITING :    // WAITING for anything to happen...
        pElbug->wShiftReg=1; // prepare detection of the next transmitted character,
             // with the "start bit" in the first possible position
             // (see specification of the "Storage format",
             //  and some CW_CHR_-constants defined in Elbug.h)
        // Regardless of the "Iambic" mode (with or without memories),
        // in THIS state, the LEVELS on the paddle inputs may ALWAYS start the next element.
        // For that purpose, Elbug_StartDashOrDot() also checks
        //   pElbug->fCurrDashSignal and pElbug->fCurrDotSignal,
        //   to also implement DJ5IL's "basic Iambic mode without memories" .
        if( ! Elbug_StartDashOrDot(pElbug) ) // start keying the next dash or dot ?
         { // nothing to send now -> the original keyer firmware switched to SLEEP MODE here
           pElbug->fMorseOutput = FALSE;
         }
        break; // end case ES_WAITING

     case ES_SEND_DOT:  // sending a dot, waiting for the end
        switch( pElbug->cfg.iMorseKeyType ) // LEVEL- or EDGE-triggered dash/dot memories ?
         { case ELBUG_TYPE_IAMBIC_B : // LEVEL-triggered [Lit9], [Lit1] :
              //  > A correctly programmed iambic keyer with dot/dash-memory
              //  > checks for paddle inputs during each dot or dash in type "A"
              //  > as well as in type "B". But in type "A" (Curtis-keyer) it checks
              //  > for the TRANSIENT from unpressed to pressed
              //  > whereas in type "B" (Accu-keyer) it checks just for the STATE pressed.
              Elbug_StoreDash_LevelTriggered(pElbug); // sample the "opposite" element, here: LEVEL-triggered
              break;
           case ELBUG_TYPE_IAMBIC_A : // EDGE-triggered DASH-memory (why not.. in contrast to the original Curtis keyer) ?
              Elbug_StoreDash_EdgeTriggered(pElbug);
              break;
           default:  // e.g. ELBUG_TYPE_IAMBIC_NO_MEM = "Basic Iambic Keyer without dot/dash memory"
              // No need to sample anything into dash/dot memory here,
              // because Elbug_StartDashOrDot() also checks the 'direct' inputs.
              break;
         } // end switch( pElbug->cfg.iMorseKeyType )

        if( pElbug->i32CountdownForDashOrDot_us <= 0 ) // finished sending a DOT .. what next ?
         { pElbug->bFlags &= ~ELBUG_FLAG_LAST_SENT_DASH; // remember WHAT HAS BEEN SENT (not a dash)
           // to toggle between dashes and dots later, as long as BOTH padddle contacts are closed.
           pElbug->bFlags &= ~ELBUG_FLAG_STORED_DOT; // clear the "dot flipflop" at the END of a dot [Lit8]
           Elbug_StartGap(pElbug);  // start a gap, with duration of AT LEAST one "dot"
         }
        break; // end case ES_SEND_DOT

     case ES_SEND_DASH:  // sending a dash, waiting for the end
        switch( pElbug->cfg.iMorseKeyType ) // LEVEL- or EDGE-triggered dash/dot memories ?
         { case ELBUG_TYPE_IAMBIC_B : // LEVEL-triggered [Lit9], [Lit1] :
              Elbug_StoreDot_LevelTriggered(pElbug); // sample the "opposite" element, here: LEVEL-triggered
              // '-- Without this, the keyer wouldn't pass DJ5IL's test [Lit6],[Lit7] :
              // > Finally squeeze a "K" and release both levers while the second dash
              // > is heard. If you get the "K" the iambic type depends on the
              // > previous dot/dash-memory test: without dot- and dash-memory
              // > the keyer works in plain iambic mode,
              // > with dot- and dash-memory in iambic type "A" (Curtis-keyer).
              // > If you get a "C" instead, the keyer works in iambic type "B"
              // > (Accu-keyer).
              break;
           case ELBUG_TYPE_IAMBIC_A : // EDGE-triggered [Lit1], [Lit3] :
              // > If anytime during generation of a dash-element
              // > the dot-lever changed its state from unpressed to pressed,
              // > generate an extra dot-element.
              Elbug_StoreDot_EdgeTriggered(pElbug); // sample the "opposite" element, here: EDGE-triggered
              break;
           default:  // e.g. ELBUG_TYPE_IAMBIC_NO_MEM = "Basic Iambic Keyer without dot/dash memory"
              // No need to sample anything into dash/dot memory here,
              // because Elbug_StartDashOrDot() also checks the 'direct' inputs.
              break;
         } // end switch( pElbug->cfg.iMorseKeyType )
        if( pElbug->i32CountdownForDashOrDot_us <= 0 ) // finished sending a DASH .. what next ?
         { pElbug->bFlags |= ELBUG_FLAG_LAST_SENT_DASH; // remember WHAT HAS BEEN SENT (a dash)
           pElbug->bFlags &= ~ELBUG_FLAG_STORED_DASH;   // clear the "dash flipflop" at the END of a dash [Lit8]
           Elbug_StartGap(pElbug);  // start a gap, with duration of AT LEAST one "dot"
         }
        break; // end case ES_SEND_DASH

     case ES_SEND_GAP: // sending a GAP with the duration of a single dot, immediately after a dot or dash.
        // Which of the two (dot-length gap following a dash or dot)
        //  is indicated by pElbug->bFlags.ELBUG_FLAG_LAST_SENT_DASH .
        if( pElbug->cfg.iMorseKeyType == ELBUG_TYPE_IAMBIC_B ) // allow LEVEL-triggered dash/dot memories ?
         { Elbug_StoreDash_LevelTriggered(pElbug);  // [Lit1], [Lit2] !
           // ex: Elbug_StoreDot_LevelTriggered(pElbug);
           // Modified 2024-05-25 : Having to release the DOT level while the
           //     dot is being sent (not during the dot-long-gap that follows immediately)
           //     is almost impossible at higher speeds, thus removed the above.
           //  The NEW timing to send an "E" in *our* variant of "Iambic B" is now:
           //                   ______
           //  Dot contact ____|      |____ ( contact MAY be closed for almost TWO DOT TIMES to send an "E")
           //                   ___   :
           //  CW output   ____|   |__:____
           //                         :
           //                      transition from ES_SEND_GAP to ES_SEND_GAP2
         }

        if( pElbug->i32CountdownForDashOrDot_us <= 0 ) // finished sending a one-dot-GAP .. what next ?
         { //  [Lit2] > Even in basic iambic mode without dot/dash-memory it does not
           //         > check after the end of each dot or dash, but after the
           //         > end of each dot- or dash- element which contains
           //         > the following space.
           //  (WB: that's exactly where we are at THIS point..)
           if( ! Elbug_StartDashOrDot(pElbug) ) // start keying the next dash or dot ?
            { // nothing to send now -> start a "subsequent" gap, before deciding if it's the end of a CHARACTER or even an inter-word-SPACE
              Elbug_StartGap2(pElbug);
            }
          }
         break; // end case ES_SEND_GAP (one-dot-gap immediately after dash or dot)

     case ES_SEND_GAP_2:  // WAIT for the end of a gap, after the initial single-dot-time.
        // Regardless of the "Iambic" mode (with or without memories),
        // in THIS state, fDashSignal or fDotSignal may IMMEDIATELY start the next element
        if( Elbug_StartDashOrDot(pElbug) ) // start keying the next dash or dot AT ANY TIME ?
         { // .. ok, looks like another dash or dot WITHIN A CHARACTER.
           // Just keep on "keying", don't emit a decoded character yet.
         }
        else // neither a dash or dot to send yet...
         { // A new character is detected after a TOTAL pause of 2 dot-times has expired.
           // (the norm is a gap of three dot-times, so TWO dot times are the threshold).
           // The old character continues if time not expired and any paddle pressed.
           // Note: A gap of one "dot-time" is already over
           //       since the end of last DASH or DOT!
           if( pElbug->i32CountdownForDashOrDot_us <= 0 )
            { // Detected the end of a "new transmitted character",
              // after a gap of two dot times since the end of the last DOT or DASH.
              pElbug->iHandlerResult = (int)pElbug->wShiftReg; // -> CW_CHR_... defined in Elbug.h
              pElbug->bFlags &= ~ELBUG_FLAG_EMITTED_SPACE; // allow emitting a single SPACE CHARACTER again
              Elbug_OnEndOfChar(pElbug);
            }
         }
        break; // end case ES_SEND_GAP_2

     case ES_WAIT_FOR_SPACE: // Waiting for a "longer gap" before emitting a SPACE:
        if( ! Elbug_StartDashOrDot(pElbug) ) // start keying the next dash or dot AT ANY TIME ?
         { if( pElbug->i32CountdownForDashOrDot_us <= 0 )
            { // Detected the a sufficiently long gap to emit a SPACE character:
              Elbug_EmitSpace(pElbug);
            }
         }
        break;

        // ;======= end of CW DECODER states ===============================
   } // end switch( ElbugState )

  // Update "previous states" of the paddle contacts to detect edges
  //  in the next call of Elbug_Handler() :
  pElbug->fPrevDashSignal = fDashSignal;
  pElbug->fPrevDotSignal  = fDotSignal;

  return pElbug->iHandlerResult;

} // end Elbug_Handler()

//--------------------------------------------------------------------------
const char *Elbug_MorseCodePatternToASCII( WORD wCwChr ) // CW_CHR_xxx -> ASCII
  // Instead of creating a nice lookup table, just a simple switch/case list
  // to look up the "Morse code pattern" (with startbit and up to fourteen
  // trailing bits, where 0=dot and 1=dash) :
{
  switch( wCwChr )
   { // If we would sort the following codes, the C compiler might use a
     // 'computed jump' for this .. but on a Windows PC, speed doesn't matter !
     case CW_CHR_0      : return "0";
     case CW_CHR_1      : return "1";
     case CW_CHR_2      : return "2";
     case CW_CHR_3      : return "3";
     case CW_CHR_4      : return "4";
     case CW_CHR_5      : return "5";
     case CW_CHR_6      : return "6";
     case CW_CHR_7      : return "7";
     case CW_CHR_8      : return "8";
     case CW_CHR_9      : return "9";
     case CW_CHR_A      : return "A";
     case CW_CHR_B      : return "B";
     case CW_CHR_C      : return "C";
     case CW_CHR_D      : return "D";
     case CW_CHR_E      : return "E";
     case CW_CHR_F      : return "F";
     case CW_CHR_G      : return "G";
     case CW_CHR_H      : return "H";
     case CW_CHR_I      : return "I";
     case CW_CHR_J      : return "J";
     case CW_CHR_K      : return "K";
     case CW_CHR_L      : return "L";
     case CW_CHR_M      : return "M";
     case CW_CHR_N      : return "N";
     case CW_CHR_O      : return "O";
     case CW_CHR_P      : return "P";
     case CW_CHR_Q      : return "Q";
     case CW_CHR_R      : return "R";
     case CW_CHR_S      : return "S";
     case CW_CHR_T      : return "T";
     case CW_CHR_U      : return "U";
     case CW_CHR_V      : return "V";
     case CW_CHR_W      : return "W";
     case CW_CHR_X      : return "X";
     case CW_CHR_Y      : return "Y";
     case CW_CHR_Z      : return "Z";

     case CW_CHR_EQUAL  : return "=";   // (-...-)
     case CW_CHR_HYPHEN : return "-";   // (-....-)
     case CW_CHR_POINT  : return ".";   // (-.-.-.)
     case CW_CHR_SLASH  : return "/";   // (-..-.)
     case CW_CHR_QSTN   : return "?";   // (..--..)
     case CW_CHR_COMMA  : return ",";   // (--..--)
     case CW_CHR_COLON  : return ":";   // (---...)
     case CW_CHR_OPEN_B : return "(";   // (-.--.-) opening bracket, rarely ever seen
  // case CW_CHR_CLOSE_B: return ")";   // (-.--.) opening bracket, collides with prosign ^KN
     // Not implemented here (because ".-.-." is prosign ^AR = END OF MESSAGE) : "+"
     case CW_CHR_AT     : return "@";   // '@' (.--.-.) "facilitates sending e-mail addresses by Morse code"
     case CW_CHR_APOSTROPHE: return "'"; // '  (.----.) Apostrophe



     case CW_CHR_GERMAN_AE: return ""; // German Umlaut  (AE)
     case CW_CHR_GERMAN_OE: return ""; // German Umlaut  (OE)
     case CW_CHR_GERMAN_UE: return ""; // German Umlaut  (UE)

     // Special Morse code sequences ('prosigns') use the same display format
     //         as used in Icom radios with built-in "text to Morse" converters.
     //         From the IC-7300 "Full Manual", "Operating CW", "Keyer memory edit menu":
     // > About the symbols
     // > * Enter "^" to send a string of characters with no intercharacter space.
     // > * Put "^" before a text string such as ^AR, and the string "ar" is sent
     // >   with no space
     case CW_CHR_AR     : return "^AR ";  // (.-.-.)
     case CW_CHR_BK     : return "^BK ";  // (-...-.-) rarely used, everyone understands "bk" as two characters
     case CW_CHR_SK     : return "^SK ";  // (...-.-)
     case CW_CHR_KA     : return "^KA ";  // (-.-.-)
     case CW_CHR_KN     : return "^KN ";  // (-.--.)
     case CW_CHR_EOM    : return "^EOM "; // (.-----) used for "partitions" of msg in DL4YHF's "QRP Keyer"
     case CW_CHR_NNN    : return "^NNN "; // (-.-.-.) replaced by serial number in DL4YHF's "QRP Keyer"
     // ex: case CW_CHR_ANN: return "^ANN "; // (.--.-.) advance to next number in DL4YHF's "QRP Keyer"
     default:
        if( wCwChr & CW_CHR_SPACE ) // bit SEVEN set -> "SPACE (=pause) or CONTROL character"
         { return " ";
         }
        break;
   }
  // Arrived here ? None of the 'known' Morse character patterns ..
  return "";
} // end Elbug_MorseCodePatternToASCII()

//--------------------------------------------------------------------------
WORD Elbug_SingleCharToMorseCodePattern( char c ) // single character (ASCII) -> CW_CHR_xxx
{
  if( (c>='a') && (c<='z') )
   {   c = c - 'a' + 'A'; // convert to UPPER CASE for the switch-case list
   }
  switch( c )
   { case '0' : return CW_CHR_0;  // b'00111111'  ; "0"
     case '1' : return CW_CHR_1;  // b'00101111'  ; "1"
     case '2' : return CW_CHR_2;  // b'00100111'  ; "2"
     case '3' : return CW_CHR_3;  // b'00100011'  ; "3"
     case '4' : return CW_CHR_4;  // b'00100001'  ; "4"
     case '5' : return CW_CHR_5;  // b'00100000'  ; "5"
     case '6' : return CW_CHR_6;  // b'00110000'  ; "6"
     case '7' : return CW_CHR_7;  // b'00111000'  ; "7"
     case '8' : return CW_CHR_8;  // b'00111100'  ; "8"
     case '9' : return CW_CHR_9;  // b'00111110'  ; "9"
     case 'A' : return CW_CHR_A;  // b'00000101'  ; 'A'
     case 'B' : return CW_CHR_B;  // b'00011000'  ; 'B'
     case 'C' : return CW_CHR_C;  // b'00011010'  ; 'C'
     case 'D' : return CW_CHR_D;  // b'00001100'  ; 'D'
     case 'E' : return CW_CHR_E;  // b'00000010'  ; 'E'
     case 'F' : return CW_CHR_F;  // b'00010010'  ; 'F'
     case 'G' : return CW_CHR_G;  // b'00001110'  ; 'G'
     case 'H' : return CW_CHR_H;  // b'00010000'  ; 'H'
     case 'I' : return CW_CHR_I;  // b'00000100'  ; 'I'
     case 'J' : return CW_CHR_J;  // b'00010111'  ; 'J'
     case 'K' : return CW_CHR_K;  // b'00001101'  ; 'K'
     case 'L' : return CW_CHR_L;  // b'00010100'  ; 'L'
     case 'M' : return CW_CHR_M;  // b'00000111'  ; 'M'
     case 'N' : return CW_CHR_N;  // b'00000110'  ; 'N'
     case 'O' : return CW_CHR_O;  // b'00001111'  ; 'O'
     case 'P' : return CW_CHR_P;  // b'00010110'  ; 'P'
     case 'Q' : return CW_CHR_Q;  // b'00011101'  ; 'Q'
     case 'R' : return CW_CHR_R;  // b'00001010'  ; 'R'
     case 'S' : return CW_CHR_S;  // b'00001000'  ; 'S'
     case 'T' : return CW_CHR_T;  // b'00000011'  ; 'T'
     case 'U' : return CW_CHR_U;  // b'00001001'  ; 'U'
     case 'V' : return CW_CHR_V;  // b'00010001'  ; 'V'
     case 'W' : return CW_CHR_W;  // b'00001011'  ; 'W'
     case 'X' : return CW_CHR_X;  // b'00011001'  ; 'X'
     case 'Y' : return CW_CHR_Y;  // b'00011011'  ; 'Y'
     case 'Z' : return CW_CHR_Z;  // b'00011100'  ; 'Z'
     case '=' : return CW_CHR_EQUAL;   // (-...-)
     case '-' : return CW_CHR_HYPHEN;  // (-....-)
     case '.' : return CW_CHR_POINT;   // (-.-.-.)
     case '/' : return CW_CHR_SLASH;   // (-..-.)
     case '?' : return CW_CHR_QSTN ;   // (..--..)
     case ',' : return CW_CHR_COMMA;   // (--..--)
     case ':' : return CW_CHR_COLON;   // (---...)
     case '(' : return CW_CHR_OPEN_B;  // (-.--.-)
     case ')' : return CW_CHR_CLOSE_B; // (-.--.)
     case '@' : return CW_CHR_AT;      // '@' (.--.-.) "facilitates sending e-mail addresses by Morse code"
     case '\'': return CW_CHR_APOSTROPHE; // '  (.----.) Apostrophe

     case ' ' : return CW_CHR_SPACE;   // ' '
     case '' : case '': return CW_CHR_GERMAN_AE; // (.-.-) German Umlaut  (AE)
     case '' : case '': return CW_CHR_GERMAN_OE; // (---.) German Umlaut  (OE)
     case '' : case '': return CW_CHR_GERMAN_UE; // (..--) German Umlaut  (UE)

     default  : return 0;
   } // end switch( c )
} // end Elbug_SingleCharToMorseCodePattern()


//--------------------------------------------------------------------------
WORD Elbug_ParseMorseCodePatternFromASCII( const char **ppszSource ) // ASCII -> CW_CHR_xxx
  // More or less the inverse to Elbug_MorseCodePatternToASCII().
  // The source pointer will be incremented by the number of ASCII characters
  // parsed (one character for normal letters and digits, more than one
  // for prosigns like "^AR ^SK ^SOS" ) .
  // The return value uses the format specified in Elbug.h for the
  // 'normal' CW characters like CW_CHR_0 ... CW_CHR_Z, with the
  // FIRST transmitted element (0=dot, 1=dash) in bit zero, and the
  // NUMBER of elements indicated by a 'startbit' .
  // Returns 0x0000 if the source pointer already points to the trailing zero
  //         in a C string, or to the begin of a macro like <sWPM>, <s>, etc .
  //         In that case, *ppszSource is not modified, and it's up to the
  //         caller (e.g. CwGen.c : CwGen_PrepareSendingNextChar() to parse it.
{ const char *cpSrc = *ppszSource;
  char c1;
  WORD wResult = 0;
  if( cpSrc != NULL )
   { c1 = *(cpSrc++);   // get the first character  (may be '^' to begin a 'prosign')
     wResult = Elbug_SingleCharToMorseCodePattern(c1);
     if( wResult == 0 ) // oops.. c1 is NOT "simple, single character",
      { if( c1=='^' )   //        but the begin of a 'prosign' like "^AR" ->
         { // Only implement a few hard-coded 'prosigns' here,
           // at least those also recognized in the reverse direction
           //    - see Elbug_MorseCodePatternToASCII() :
           if( SL_SkipString_AnyCase( &cpSrc, "ar" ) )
            { wResult = CW_CHR_AR; // "^AR" (.-.-.)  end-of-transmission / end-of-message (but not end of a QSO)
            }
           else if( SL_SkipString_AnyCase( &cpSrc, "bk" ) )
            { wResult = CW_CHR_BK; // "^BK " (...-.-) "break" / "back to you for a quick reply"
            }
           else if( SL_SkipString_AnyCase( &cpSrc, "sk" ) )
            { wResult = CW_CHR_SK; // "^SK " (...-.-) silent key /  end of contact (QSO) / frequency no longer in use from my side
            }
           else if( SL_SkipString_AnyCase( &cpSrc, "ka" ) )
            { wResult = CW_CHR_KA; // "^KA " (-.-.-)  message begins / start of work / new message
            }
           else if( SL_SkipString_AnyCase( &cpSrc, "kn" ) )
            { wResult = CW_CHR_KN; // "^KN " (-.--.)  Go ahead, specific named station
              // (it's terrible to hear stations brainlessly using this ALWAYS, even at the end of a CQ call)
            }
           else if( SL_SkipString_AnyCase( &cpSrc, "eom" ) )
            { wResult = CW_CHR_EOM; // "^EOM " (.-----) used for "partitions" of msg in DL4YHF's "QRP Keyer"
            }
           else if( SL_SkipString_AnyCase( &cpSrc, "nnn" ) )
            { wResult = CW_CHR_NNN; // "^NNN " (-.-.-.) replaced by serial number in DL4YHF's "QRP Keyer"
            }
         }
        else  // nothing we can parse as 'something transmittable', so:
         { return 0; // return WITHOUT incrementing the caller's source pointer
           // CwGen.c may try CwGen_ParseMacro() now, with e.g. *ppszSource = "<s40>"
         }
      }
     *ppszSource = cpSrc;  // pass back the INCREMENTED "source pointer"
     return wResult;
   } // end if < valid dereferenced source pointer >
  return wResult;
} // end Elbug_ParseMorseCodePatternFromASCII()
