//---------------------------------------------------------------------------
// File: C:\cbproj\Remote_CW_Keyer\Dialogs.c
// Purpose:  Wrapper for some "common dialogs" in the Windows API .
//               by  Wolfgang Buescher (DL4YHF) .
//               NO APPLICATION-SPECIFIC STUFF IN HERE !
//               This unit is used in a number of projects (see history).
//
// Revision History:
//  2024-11-25: Replaced HUNDREDS of "char pointers" by "const char *"
//              because 'modern compilers' (e.g. "ISO C++11") are
//              very pedantic about char pointers, and refuse to
//              use STRING LITERALS as a "char *" !
//           After that, most of the non-VCL-modules could be compiled
//           with Borland C++Builder V6 and Embarcadero C++Builder V12.
//           Details about those modifications in YHF_Tools/StringLib.h .
//  2005-10-24:  Added "YHF_BrowseForFolder" (for WinPic) .
//  2004-07-13:  Created for DL4YHF's "Spectrum Lab".
//               Also used -the same day- for DL4YHF's "WinPic".
//
//---------------------------------------------------------------------------

#include "switches.h" // WB's "project specific switches" (include FIRST).
                      // Located in a PROJECT SPECIFIC subdirectory.
                      // In Borland C++ V6 projects, defined under
                      //  "Project".."Options".."Directories(..)".."Include Path".

// NO VCL IN HERE !!!!
#include <stdio.h>   // no "I/O" but sprintf
#include <math.h>    // using HUGE_VAL for error checking here
#include <windows.h>
#include <shlobj.h>


#pragma hdrstop   // Borland Stuff. No precompiled headers after this.

#include "StringLib.h"   // replace strncpy() by SL_strncpy(), because strncpy() sucks !
#include "YHF_Dialogs.h" // header file for this module

#ifndef  SWI_YHF_HELP_INCLUDED    // if not def'd in SWITCHES.H ...
# define SWI_YHF_HELP_INCLUDED 0  //  ...then assume 'YHF_Help' (replacement for the defunct *.hlp system) isn't available
#endif
#if( SWI_YHF_HELP_INCLUDED )
   // When compiled with SWI_YHF_HELP_INCLUDED=1 (in switches.h),
   // YHF_Dialogs.cpp will call YHF_HELP_ShowHelpContext()
   // when clicking the "Help" button with the help-ID passed
   // as parameter 'iHelpID', e.g. in YHF_RunStringEditDialog().
#  include "YHF_Help.h"  // replacement for the defunct Windows "*.HLP" system
   // (What THEY called "help context" is our "help ID" - it's just an integer)
#endif // SWI_YHF_HELP_INCLUDED ?



#include "QFile.h"  // sometimes we need "QFile_LastErrorCodeToString",
                    // to find those f****g error codes from GetLastError(),
                    // which you will never find in WINNT.H !


char YHFdlg_sz255ColorDlgCaption[256];
UINT APIENTRY YHFdlg_ColorDialogHook(
    HWND hdlg,         // handle to the dialog box window
    UINT uiMsg,        // message identifier
    WPARAM wParam,     // message parameter
    LPARAM lParam )    // message parameter
{
 // See Win32 Programmer's Reference on ChooseColor(),  CCHookProc  .
 // Return Values
 // If the hook procedure returns zero, the default dialog box procedure processes the message.
 // If the hook procedure returns a nonzero value, the default dialog box procedure ignores the message.

 switch( uiMsg )
  {
   case WM_INITDIALOG:
        SendMessage( hdlg, WM_SETTEXT, 0, (LPARAM) YHFdlg_sz255ColorDlgCaption);
        // 2004-05-05 : Bingo, this works. Now the window has a much more "informal" title.
        return 1;
   default:
        break;
  }

 return 0;  // all other messages shall be handled by the default dialog box procedure !

} // end YHFdlg_ColorDialogHook()



//--------------------------------------------------------------------------
BOOL YHF_ChooseColor( HWND hwndOwner, const char *pszCaption, DWORD *pdwRGBColor )
  // Color Selection Dialog.
  // Works a bit like Borland's TColorDialog
  //  but (and this is only one BIG PRO) allows to define the
  //  title ("caption") of the colour selection window .
  // Furthermore, it does not depend on Borland's bulky VCL .
{
  // > The ChooseColor function creates a Color common dialog box
  // > that enables the user to select a color.
  // Doesn't sound too complicated, huh ? Let's try:
  // First fill a CHOOSECOLOR structure and then see what happens !
 CHOOSECOLOR ColorDlgInfo;
 static COLORREF MyColours[16];
  memset(&ColorDlgInfo, 0, sizeof(ColorDlgInfo) );
  memset(&MyColours,    0, sizeof(MyColours) );
  ColorDlgInfo.lStructSize = sizeof(CHOOSECOLOR);
  ColorDlgInfo.hwndOwner = hwndOwner; // a WINDOW handle
  ColorDlgInfo.hInstance = NULL;      // ummm... do I need this ?
  ColorDlgInfo.rgbResult = (COLORREF)*pdwRGBColor; // a COLORREF = 32-bit value used to specify an RGB color
  MyColours[0] = ColorDlgInfo.rgbResult;
  ColorDlgInfo.lpCustColors = MyColours;  // a COLORREF pointer to a user-defined palette
  ColorDlgInfo.Flags     = CC_RGBINIT | CC_ENABLEHOOK;
  ColorDlgInfo.lCustData = (long)YHFdlg_sz255ColorDlgCaption;
  ColorDlgInfo.lpfnHook  = YHFdlg_ColorDialogHook;
  strcpy(YHFdlg_sz255ColorDlgCaption, pszCaption );
  if( ChooseColor( &ColorDlgInfo ) )
   {  // user clicked "Ok" in the colour selection dialog, result in ColorDlgInfo :
     *pdwRGBColor = (DWORD)ColorDlgInfo.rgbResult;
     return TRUE;
   }
  return FALSE;
} // end YHF_ChooseColor()


//--------------------------------------------------------------------------
int CALLBACK YHF_BFFCallback(HWND hwnd,UINT uMsg,LPARAM lp, LPARAM pData)
{
  char szPath[MAX_PATH];

  switch(uMsg)
   {
     case BFFM_INITIALIZED:
        SendMessage(hwnd, BFFM_SETSELECTION, TRUE, pData);
        break;

     case BFFM_SELCHANGED:
        if (SHGetPathFromIDList((LPITEMIDLIST) lp ,szPath))
         {
           SendMessage(hwnd, BFFM_SETSTATUSTEXT,0,(LPARAM)szPath);
         }
        break;
   }
  return 0;
} // end YHF_BFFCallback ( Browse-For-Folder-callback )

//--------------------------------------------------------------------------
BOOL YHF_BrowseFolder(HWND hwndOwner, const char *pszCaption,
                      char *pszFolder, int iMaxLen )
  // Folder Selection Dialog.
  //  The current folder (pszFolder) will be selected in the dialog,
  //  so it must be VALID (or contain an empty string) before the call .
{
  // Inspired by www.nirsoft.net/vc/brfolder.html .
  // (would take DAYS to write this, if you only have the M$-documentation ;)
  BROWSEINFO bi;
  char szPath[MAX_PATH + 1];
  LPITEMIDLIST pidl;
  BOOL bResult = FALSE;
  LPMALLOC pMalloc;

  if( SHGetMalloc(&pMalloc) == NOERROR ) 
   {
     bi.hwndOwner = hwndOwner;
     bi.pidlRoot = NULL;
     bi.pszDisplayName = NULL;
     bi.lpszTitle = pszCaption;
     bi.ulFlags = BIF_STATUSTEXT; //BIF_EDITBOX
     bi.lpfn   = YHF_BFFCallback;
     bi.lParam = (LPARAM)pszFolder;

     pidl = SHBrowseForFolder(&bi);
     if (pidl)
      {
        if (SHGetPathFromIDList(pidl,szPath))
         {
           bResult = TRUE;
           SL_strncpy(pszFolder, szPath, iMaxLen );
         }
#      if(defined __CODEGEARC__)
        // Because the original code (further below) didn't compile, tried this
        // (suggested in a forum, where the problem was "C" versus "C++".
        //  The "C" interface expects an "IMalloc * This" as 1st argument,
        //  The "C++" interface of course does NOT pass a "This"-pointer.)
        //
        //  Maybe C++Builder V12 (which, unlike V6, #defined "__CODEGEARC__")
        //  compiles everything as if it was a *.cpp file, including *.c ? ?)
        pMalloc->lpVtbl->Free(pMalloc, pidl);  // <- this is the "C" interface..
        pMalloc->lpVtbl->Release(pMalloc);     // .. for the obfuscated "IMalloc" thingy
#      else // ! defined __CODEGEARC__ : the following was ok for Borland C++ Builder V6,
        // where "interface IMalloc" was in C:\CBuilder6\Include\objidl.h :
        pMalloc->Free(pidl);
        pMalloc->Release();
        // '--> C++Builder V12 didn't understand, and threw these errors:
        // > [bcc32c Error] : no member named 'Free' in 'struct IMalloc'
        // > [bcc32c Error] : no member named 'Release' in 'struct IMalloc'
#      endif // def'd __CODEGEARC__

      }
   }

  return bResult;

} // end YHF_BrowseFolder()

//---------------------------------------------------------------------------
void YHF_LeftAlignedTextOut( HDC dc, int x, int y, char *pszText )
{
  if( dc!=NULL && pszText!=NULL)
   { SetTextAlign( dc, TA_LEFT|TA_TOP);
     TextOut( dc, x, y, pszText, strlen(pszText) );
   }
}

//---------------------------------------------------------------------------
// internal variables for the "simple dialog window" (input dialog w/o resources)
//---------------------------------------------------------------------------
static int   YhfDlgs_iEditOptions = 0;
static int   YhfDlgs_iHelpID  = 0;
static char *YhfDlgs_pszTitle = "NoTitle";
static char *YhfDlgs_pszLabel1= "NoLabel1";
static char *YhfDlgs_pszLabel2= "NoLabel2";
static char  YhfDlgs_sz80EditedString[84];
static HWND  YhfDlgs_hwndModalDialog;
static HWND  YhfDlgs_hwndEditField;
static long  YhfDlgs_i32BackgndColor = 0x00F0E0E0; /*light bluish gray by default*/
static BOOL  YhfDlgs_fDialogWndclassRegistered = FALSE;
static BOOL  YhfDlgs_fDialogWindowClosed;
static int   YhfDlgs_iModalResult;

//------------------------------------------------------------------------
LONG YhfDlg1_OnWmPaint( HWND hWnd )
{ // reaction to a WM_PAINT message
  HDC hDC;
  // HPEN hPen,hOldPen;
  HBRUSH hBrush,hOldBrush;
  PAINTSTRUCT ps;
  RECT rctClientArea;
  int  iScreenWidth,iScreenHeight;

  hDC = BeginPaint(hWnd,&ps);

  // Center a message in the "parameter display area"..
  GetClientRect(hWnd,&rctClientArea);
  iScreenHeight = rctClientArea.bottom-rctClientArea.top;
  iScreenWidth  = rctClientArea.right-rctClientArea.left;
  if(iScreenWidth>50 && iScreenHeight>50)
   {

    // Erase the entire background (that's why we don't want WM_ERASEBKGND)
    hBrush  = CreateSolidBrush( YhfDlgs_i32BackgndColor );
    hOldBrush = SelectObject( hDC, hBrush );
    SetBkColor( hDC, YhfDlgs_i32BackgndColor );
    FillRect( hDC, &rctClientArea, hBrush );
    if( YhfDlgs_pszLabel1 != NULL )
     { YHF_LeftAlignedTextOut( hDC, 16/*x*/, 8/*y*/, YhfDlgs_pszLabel1 );
     }
    if( YhfDlgs_pszLabel2 != NULL )
     { YHF_LeftAlignedTextOut( hDC, 16/*x*/, 32/*y*/, YhfDlgs_pszLabel2 );
     }
    SelectObject( hDC, hOldBrush );
    DeleteObject( hBrush );
   }

  EndPaint(hWnd,&ps);


  // An application should return zero if it processes this message...
  return 0L;
} // end YhfDlg1_OnWmPaint()


//----------------------------------------------------------------------------
static LRESULT CALLBACK YhfDlgs_WndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
  // "window procedure" (message handler) for the simple "dialog"(*) window
  // (*) it's modal, but in reality it's NOT a dialog window !
{
  switch(message)
   {
     // case WM_INITDIALOG:  // not handled here ! we look like a dialog but aren't !
     case WM_COMMAND:
        switch(LOWORD(wParam))
         {
           case YHF_DLG_BTN_OK:
                if( YhfDlgs_hwndEditField != NULL )
                 { memset( YhfDlgs_sz80EditedString, 0, sizeof(YhfDlgs_sz80EditedString) );
                   GetWindowText( YhfDlgs_hwndEditField, YhfDlgs_sz80EditedString, 80 );
                 }
                YhfDlgs_iModalResult = YHF_DLG_BTN_OK;
                SendMessage( hWnd, WM_CLOSE, 0, 0 ); // CloseWindow( hWnd ) didn't work
                return 0;
           case YHF_DLG_BTN_CANCEL:
                YhfDlgs_iModalResult = YHF_DLG_BTN_CANCEL;
                SendMessage( hWnd, WM_CLOSE, 0, 0 );
                return 0;
#         if( SWI_YHF_HELP_INCLUDED )  // support "Help via HTML browser" / YHF_Help.cpp ?
           case YHF_DLG_BTN_HELP:
                YHF_HELP_ShowHelpContext( YhfDlgs_iHelpID );  // Here: Application-specific help via "Help"-button in a modal dialog window
                return 0;
#         endif  // SWI_YHF_HELP_INCLUDED
         }      // end switch(LOWORD(wParam))
        break; // end case WM_COMMAND (notifications from child windows like Buttons, etc etc)

     case WM_ERASEBKGND: // sent when the window background must be erased
        // > If the application returns zero, the window will remain marked for erasing.
        return 0;

     case WM_PAINT:
        return YhfDlg1_OnWmPaint( hWnd );

     case WM_CLOSE:  // someone closed the dialog window, somehow..
        // > The WM_CLOSE message is sent as a signal that a window
        // > (or an application, which is not the case here) should terminate.
        // > An application can prompt the user for confirmation, prior to
        // > destroying a window, by processing the WM_CLOSE message and calling
        // > the DestroyWindow function only if the user confirms the choice.
        // (a-ha. sounds like we should call DestroyWindow() in the window procedure)
        YhfDlgs_fDialogWindowClosed = TRUE; // the window hasn't closed ITSELF yet
        DestroyWindow( hWnd );  // sends WM_DESTROY and cleans up..
        // > (WM_QUIT) If an application processes this message, it should return zero.
        // (not the stupid info from someone talking about Win32 API programming..
        //    > "in general you return FALSE for messages you don't process," NO-NO-NO !
        return 0;  // return ZERO because we DID process this message .
     case WM_DESTROY: // we're being destroyed;
        YhfDlgs_fDialogWindowClosed = TRUE; // should already have been set in WM_CLOSE, anyway..
        break;     // let DefWindowProc() look at this, too

     default:
        break;  // let DefWindowProc() do whatever it needs to ... see below
        // DON'T ASSUME ANYTHING, ESPECIALLY NOT WHETER TO RETURN "TRUE" OR "FALSE" HERE !
   } // end switch(Message)

  // By order of the prophet (Micro$oft, in the Win32 programmer's ref or the MSDN):
  // > An application-defined window procedure should pass any messages
  // > that it does not process to the DefWindowProc function for default
  // > processing.
  // Note that -for good reason- they don't say anything about
  //      *THE VALUE* to return .
  // You'll sometimes find advice on the net like this:
  //    > in general you return FALSE for messages you don't process,
  //    > and TRUE for messages you do process
  //  THIS MAY BE TRUE "IN MANY CASES" BUT TAKEN STRICTLY, IT'S WRONG .
  //  Do you know which messages M$ will come up with in windoze 7,8,9 ?  ;o)
  return DefWindowProc (hWnd, message, wParam, lParam) ;
} // end YhfDlgs_WndProc()

//----------------------------------------------------------------------------
static int YhfDlgs_RunModalDialog( HWND hwndOwner, // INTERNAL function !
              int iWidth, int iHeight )
   // For historic reasons (and to avoid breaking the API for future
   // extensions), more parameters are passed in through the following
   // variables : YhfDlgs_iEditOptions,
   //             YhfDlgs_iHelpID (shows a "Help"-button when nonzero),
   //             YhfDlgs_pszTitle, YhfDlgs_pszLabel1, YhfDlgs_pszLabel2,
   //             YhfDlgs_sz80EditedString .
{
  int  x,x2,x3, y, screen_width, screen_height, button_width, num_buttons = 2;
  WNDCLASS wndclass ;
  HWND hwndOldFocus;
  MSG  Msg;
  static BOOL already_here = FALSE;

  if( APPL_hInstance==NULL || hwndOwner==NULL )
   { return YHF_DLG_BTN_CANCEL;  // say "not open for business"
   }
  if( already_here )   // avoid recursion ! this dialog mustn't call itself
   { return YHF_DLG_BTN_CANCEL;  // say "not open for business"
   }
  already_here = TRUE;
  if( YhfDlgs_iHelpID != 0 )
   { ++num_buttons;
   }

  // Forget about "CreateDialog" and similar - they all need RESOURCE FILES,
  // or at least RESOURCE TEMPLATES .  Googling around for a while showed this:
  // > I've been reading about that DialogBox function and tried to understand,
  // > why the window which is displayed using that function becomes modal,
  // > I was actually trying to read between the lines, there had to be
  // > something which triggered that effect, that's what I've been looking for.
  // > As I went deeper, I found out that dialog boxes simply disable
  // > the owner window , so what I actually wanted to hear in the reply,
  // > was this one line (two actually):
  // >  Code:
  // > // add this line before calling the ShowWindow to open a modal window
  // > wasEnabled = EnableWindow(hwndOwner, FALSE); // disable the owner window
  // > // before calling function to close the child (modal) window, add this:
  // > EnableWindow(hwndOwner, wasEnabled);
  // But that's not the full story. The application's endless message loop
  // must still be kept alive, so we cannot simply run in our own little loop
  // here (or can we .. until the OK- or CANCEL- button was clicked ? ? )
  // The catch is that a function like RunStringEditDialog() should not return
  // to the caller until the input is "finished". Everything else would make
  // things utterly complex (and one would be better off with a non-modal,
  //  i.e. a non-blocking window, which is not a "dialog"-window) .

  // set up window class (if not already registered in a previous call)
  if( ! YhfDlgs_fDialogWndclassRegistered )
   { wndclass.style         = CS_HREDRAW | CS_VREDRAW ;
     wndclass.lpfnWndProc   = YhfDlgs_WndProc ;
     wndclass.cbClsExtra    = 0 ;
     wndclass.cbWndExtra    = 0 ;
     wndclass.hInstance     = APPL_hInstance;
     wndclass.hIcon         = NULL;
     wndclass.hCursor       = LoadCursor(NULL, IDC_ARROW);
     wndclass.hbrBackground = (HBRUSH)COLOR_WINDOW;
     wndclass.lpszMenuName  = 0 ;   // no menu !
     wndclass.lpszClassName = "YhfDlgWindow"; // window name
     RegisterClass (&wndclass) ;
   }

  // Set up some internal (static) variables for the window procedure:
  YhfDlgs_fDialogWindowClosed = FALSE; // the window hasn't closed ITSELF yet
  YhfDlgs_iModalResult = YHF_DLG_BTN_NONE;
  screen_width  = GetSystemMetrics(SM_CXSCREEN);
  screen_height = GetSystemMetrics(SM_CYSCREEN);
  hwndOldFocus  = GetFocus();

  // now create the dialog window (which is a "normal" window so far..)
  x = (screen_width-iWidth)   / 2;
  y = (screen_height-iHeight) / 2;
  YhfDlgs_hwndModalDialog = CreateWindow(
    "YhfDlgWindow",   // lpClassName,    pointer to registered class name
    YhfDlgs_pszTitle, // "lpWindowName", pointer to window name, effectively the TITLE
    // DWORD dwStyle,  window style. Tried a lot ...
    // WS_POPUPWINDOW | WS_CAPTION | WS_SYSMENU | WS_MINIMIZEBOX,  // non-sizeable window
    // WS_VSCROLL | WS_HSCROLL |  // scroll bars ? heavens, no !
       WS_OVERLAPPEDWINDOW | WS_VISIBLE, // quite normal window
    // WS_POPUP | WS_DLGFRAME ,   // window with no title, quite unusual as main window
    // WS_POPUPWINDOW | WS_CAPTION | WS_SYSMENU | WS_MINIMIZEBOX | WS_SIZEBOX, // sizeable
    // WS_CAPTION,   // with title, but no icon and no system menu nor close symbol
    x, y, iWidth, iHeight,
    hwndOwner,      // HWND hWndParent, handle to parent or owner(!) window
    NULL,           // HMENU hMenu,     handle to menu or child-window identifier
    APPL_hInstance, // HANDLE hInstance, handle to application instance
    NULL);          // LPVOID lpParam,   pointer to window-creation data
  if( YhfDlgs_hwndModalDialog != NULL )
   { // CreateWindow (for the "dialog") was successfull...
     EnableWindow(hwndOwner, FALSE); // disable the owner window
     button_width = 64;
     x = (iWidth-(num_buttons+1)*button_width) / 2;  // width of LEFT and RIGHT margin for "buttons"
     CreateWindow( "BUTTON"/*ClassName*/,  "Ok"/*window name*/,
                  WS_VISIBLE | WS_CHILD | WS_TABSTOP | BS_DEFPUSHBUTTON	 | BS_TEXT /*Style*/,
                  x, iHeight-80/*y*/,
                  button_width, 24/*height*/,
                  YhfDlgs_hwndModalDialog/*hWndParent*/,
                  (HMENU)YHF_DLG_BTN_OK, /*hMenu abused for the CONTROL ID - gg "Assigning a control id to a win32 button" */
                  APPL_hInstance, NULL);
     x2 = x3 = iWidth-button_width-x/*x*/;  // x3 = horz position of RIGHTMOST button
     if( YhfDlgs_iHelpID != 0 )  // got to move the "Cancel" button (x2) INTO THE CENTER ?
      { x2 = (iWidth - button_width) / 2;
      }
     CreateWindow( "BUTTON"/*ClassName*/,  "Cancel"/*window name*/,
                  WS_VISIBLE | WS_CHILD | WS_TABSTOP | BS_PUSHBUTTON | BS_TEXT /*Style*/,
                  x2, iHeight-80/*y*/,
                  button_width, 24/*height*/,
                  YhfDlgs_hwndModalDialog/*hWndParent*/,
                  (HMENU)YHF_DLG_BTN_CANCEL, APPL_hInstance, NULL);
     if( YhfDlgs_iHelpID != 0 )
      { CreateWindow( "BUTTON"/*ClassName*/,  "Help"/*window name*/,
                  WS_VISIBLE | WS_CHILD | WS_TABSTOP | BS_PUSHBUTTON | BS_TEXT /*Style*/,
                  x3/*x*/, iHeight-80/*y*/,
                  button_width, 24/*height*/,
                  YhfDlgs_hwndModalDialog/*hWndParent*/,
                  (HMENU)YHF_DLG_BTN_HELP, APPL_hInstance, NULL);
      }
     YhfDlgs_hwndEditField = CreateWindow( "EDIT"/*ClassName*/,
                  YhfDlgs_sz80EditedString, /* "window name" (here: initial text) */
                  WS_VISIBLE | WS_CHILD | WS_TABSTOP | WS_BORDER | ES_LEFT  /*Style...*/
                     | ( (YhfDlgs_iEditOptions & YHF_EDIT_DECIMAL_ONLY )  // allow ONLY NUMERIC input ?
                          ? ES_NUMBER	: 0 )  ,
                  16/*x*/, iHeight-80-32/*y*/,
                  iWidth-40, 24/*height*/,
                  YhfDlgs_hwndModalDialog/*hWndParent*/,
                  (HMENU)YHF_DLG_ID_EDIT1, APPL_hInstance, NULL);
     // Set the focus (with the blinking cursor) to the EDIT FIELD (not to the "ok"-button) :
     SetFocus( YhfDlgs_hwndEditField );

     // Dialog loop: process all window messages (not just "our own"! ),
     //      SIMILAR to what you'll find in most win32 API applications .
     // But: If GetMessage() receives the WM_QUIT message, the return value is zero.
     //      This doesn't help us to return from the modal "dialog" here, because
     //      the WM_QUIT message indicates a request to terminate an *application*
     //      and is generated when the application calls the PostQuitMessage
     //      function. It causes the GetMessage function to return zero.
     // Fix: If YhfDlgs_WndProc() closes its window, it kindly sets a flag
     //      which we stupidly poll in the message loop below .
     while(GetMessage(&Msg, NULL, 0, 0))
      {
        // From MSDN (remember this if the TAB KEY doesn't switch the active control..)
        // > Although the IsDialogMessage function is intended for modeless dialog boxes,
        // > you can use it with any window that contains controls,
        // > enabling the windows to provide the same keyboard selection
        // > as is used in a dialog box.
        // > When IsDialogMessage processes a message, it checks for keyboard messages
        // > and converts them into selections for the corresponding dialog box.
        // > For example, the TAB key, when pressed, selects the next control
        // > or group of controls, and the DOWN ARROW key, when pressed,
        // > selects the next control in a group.
        // > Because the IsDialogMessage function performs all necessary translating
        // > and dispatching of messages, a message processed by IsDialogMessage
        // > must *not* be passed to the TranslateMessage or DispatchMessage function.
        //
        if (!IsDialogMessage( YhfDlgs_hwndModalDialog, &Msg) ) // important for TAB-nav.
          {
            TranslateMessage(&Msg);
            DispatchMessage(&Msg);
          }
        if( YhfDlgs_fDialogWindowClosed )
           break;  // workaround for the missing WM_QUIT in the dialog
      }

     EnableWindow(hwndOwner, TRUE );
     SetFocus( hwndOldFocus );  // ex: SetFocus(hwndOwner);
   } // end if < creation of the dialog window successful >

  already_here = FALSE;  // accept next call
  return YhfDlgs_iModalResult; // YHF_DLG_BTN_OK, YHF_DLG_BTN_CANCEL; maybe more

} // end YhfDlgs_RunModalDialog()

//----------------------------------------------------------------------------
int YHF_RunIntegerInputDialog(  // MODAL dialog; returns YHF_DLG_BTN_...
  HWND hwndOwner,  // handle to the window which will be blocked by the dialog
  const char *pszTitle, const char *pszLabel1,  const char *pszLabel2,
  int  *piValue,  int iEditOptions, int  iHelpID )
  // return value:  one of the YHF_DLG_BTN_ codes defined in YHF_Dialogs.h .
{
   int iButtonCode;

   YhfDlgs_iEditOptions = iEditOptions;
   YhfDlgs_iHelpID      = iHelpID;
   if( (iEditOptions & YHF_EDIT_HEXADECIMAL) == 0 )
    { YhfDlgs_iEditOptions |= YHF_EDIT_DECIMAL_ONLY; // allow ONLY NUMERIC input
    }
   YhfDlgs_pszTitle = pszTitle;
   if(pszLabel1)
           YhfDlgs_pszLabel1 = pszLabel1;
     else  YhfDlgs_pszLabel1 = "";
   if(pszLabel2)
           YhfDlgs_pszLabel2 = pszLabel2;
     else  YhfDlgs_pszLabel2 = "";
   if(iEditOptions & YHF_EDIT_HEXADECIMAL)
           sprintf(YhfDlgs_sz80EditedString,"$%04X",*piValue);
     else  sprintf(YhfDlgs_sz80EditedString, "%d",  *piValue);
   iButtonCode = YhfDlgs_RunModalDialog(hwndOwner, 500/*w:ex400*/, 200/*h*/ );
   if( iButtonCode==YHF_DLG_BTN_OK )
    {
      if(YhfDlgs_sz80EditedString[0]=='$')
        { sscanf(YhfDlgs_sz80EditedString+1,"%X", piValue );
          // beware of lousy error handling in some scanf implementations !
        }
      else
        { sscanf(YhfDlgs_sz80EditedString,"%d", piValue );
        }
    }
   return iButtonCode;
} // end YHF_RunIntegerInputDialog()

//----------------------------------------------------------------------------
int YHF_RunFloatInputDialog(  // MODAL dialog; returns YHF_DLG_BTN_...
  HWND hwndOwner,  // handle to the window which will be blocked by the dialog
  const char *pszTitle, const char *pszLabel1, const char *pszLabel2,
  const char *pszFormatString,  // [in] your choice.. ex: hard-coded to "%.3lf" (until 2023-01, for DL4YHF's "Curve Editor")
  double *pdblValue,  int iEditOptions, int iHelpID )
  // return value:  one of the YHF_DLG_BTN_ codes defined in YHF_Dialogs.h .
{
  double dblTemp;
  int  iButtonCode;

   YhfDlgs_iEditOptions = iEditOptions;
   YhfDlgs_iHelpID      = iHelpID;
   YhfDlgs_pszTitle = pszTitle;
   if(pszLabel1)
           YhfDlgs_pszLabel1 = pszLabel1;
     else  YhfDlgs_pszLabel1 = "";
   if(pszLabel2)
           YhfDlgs_pszLabel2 = pszLabel2;
     else  YhfDlgs_pszLabel2 = "";
   sprintf(YhfDlgs_sz80EditedString, pszFormatString, *pdblValue);
   iButtonCode = YhfDlgs_RunModalDialog(hwndOwner, 500/*w:ex400*/, 200/*h*/ );
   if( iButtonCode==YHF_DLG_BTN_OK )
    {
      if(YhfDlgs_sz80EditedString[0]>0)
       { dblTemp = strtod( YhfDlgs_sz80EditedString, NULL);
         if( dblTemp!=HUGE_VAL && dblTemp!=-HUGE_VAL )
          { *pdblValue = dblTemp;
            return YHF_DLG_BTN_OK;
          }
       }
    }
   return iButtonCode;
} // end YHF_RunIntegerInputDialog()



//----------------------------------------------------------------------------
int YHF_RunStringEditDialog( // MODAL dialog; returns YHF_DLG_BTN_...
  HWND hwndOwner,  // handle to the window which will be blocked by the dialog
  const char *pszTitle,
  const char *pszLabel1, const char *pszLabel2,
  const char *pszDefaultValue, int iEditOptions,
  char *pszDestination,  int iMaxLength,
  int  iHelpID )
  // return value:  one of the YHF_DLG_BTN_ codes defined in YHF_Dialogs.h .
{
  int  iButtonCode;

   YhfDlgs_iEditOptions = iEditOptions;
   YhfDlgs_iHelpID      = iHelpID;
   YhfDlgs_pszTitle     = pszTitle;
   if(pszLabel1)
           YhfDlgs_pszLabel1 = pszLabel1;
     else  YhfDlgs_pszLabel1 = "";
   if(pszLabel2)
           YhfDlgs_pszLabel2 = pszLabel2;
     else  YhfDlgs_pszLabel2 = "";
   if( pszDefaultValue != NULL )
    { SL_strncpy(YhfDlgs_sz80EditedString, pszDefaultValue, 80);
    }
   else
    { YhfDlgs_sz80EditedString[0] = '\0';
    }
   iButtonCode = YhfDlgs_RunModalDialog(hwndOwner, 500/*w:ex400*/, 200/*h*/ );
   if( iButtonCode==YHF_DLG_BTN_OK )
    {
      SL_strncpy( pszDestination, YhfDlgs_sz80EditedString, iMaxLength);
    }
   return iButtonCode;
} // end YHF_RunStringEditDialog()



//---------------------------------------------------------------------------
BOOL YHF_GetMouseCursorPos( int *piScreenX, int *piScreenY )
  //  Replacement for Borland's 'Mouse' object .
  //  Retrieves the mouse cursor position (SCREEN coordinate),
  //  without the bugs in the VCL ...
  //    from http://qc.codegear.com/wc/qcmain.aspx?d=9344
  //  > When Mouse.GetCursorPos is called while Windows is switching
  //  > the active desktop it fails and returns:
  //  >  "System error.  Code: 6 (sometimes 5). The handle is invalid / "access denied"
  //  >  Switching the desktop can happen when the screensaver is activated
  //  > or when the computer is being locked using Ctrl+Alt+Delete.
  //  > THintWindow.ActivateHint uses Mouse.GetCursorPos to determine
  //  > the mouseposition, so this can lead to the problem described.
{
  POINT pt;
  BOOL  fOkToGetCursorPos;
  HDESK hDesk;  // added 2009-09-09 to cure the "handle leak"
 // DWORD dwLastError;
 // char  sz80[84];
  OSVERSIONINFO windoze_version_info;

  *piScreenX = 0; // return dummy coord if the call fails..
  *piScreenY = 0;


  // First check if Desktop is not LOCKED ...
  if(  (hDesk=OpenInputDesktop( // 2009-09-09: Calling this function EATS ONE MORE HANDLE !
     0 ,// DWORD dwFlags, flags to control interaction with other applications
     FALSE, // BOOL fInherit, specifies whether returned handle is inheritable
     DESKTOP_READOBJECTS)) // DWORD dwDesiredAccess, specifies access of returned handle
      != NULL )
   { // the desktop doesn't seem to be "locked",
     // it should be ok to trying to retrieve the mouse position:
     fOkToGetCursorPos = TRUE;
     // 2009-09-09 : Just a "feeling", even though the Win32 API reference doesn't mention this,
     //              a function beginning with "Open".. should have an associated "Close".
     //    This seemed to be missing in the (otherwise nice) example
     //    at http://qc.codegear.com/wc/qcmain.aspx?d=9344 ! !
     CloseDesktop( hDesk );  // close this desktop handle thing, whatever it is exactly.
     // At least it releases the handle (but doesn't close the visible desktop, sigh..)
   } // end if < Desktop NOT LOCKED >
  else // OpenInputDesktop() failed. There may be different reasons for this:
   { // - the desktop may be LOCKED (under Windows XP)
     // - the PC is running windows 98 (yes, some people still do !)
     //     [not clear if OpenInputDesktop() is available under Win98,
     //      since the lousy microsoft documentation never mentions it anymore]
     fOkToGetCursorPos = FALSE;
  // dwLastError = GetLastError();  // meaningless result under win98, removed
  // // Got here with dwLastError=120=0x00078 under Windows 98, when the desktop
  // //     was definitely "valid".
  // QFile_LastErrorCodeToString(dwLastError/*in*/, sz80/*out*/, 80 );
  // //  Win89 -> "Die Funktion ist nur im Win32-Modus gltig" ?!?!
  // //   (forget about this, it's utterly useless, don't rely on that junk)
  // //   Second attempt (2009-02-19) :
     // Which version of windoze are we running,  win98 or xp ?
     windoze_version_info.dwOSVersionInfoSize = sizeof( OSVERSIONINFO );
     if( GetVersionEx( &windoze_version_info ) )
      { if( windoze_version_info.dwMajorVersion >= 5 )
         { // Win2000, Win XP, etc..  , but definitely NOT win98:
           // Must not ignore the error from OpenInputDesktop(),
           //  otherwise the program would -most likely- crash !
         }
        else // Got here with windoze_version_info.dwMajorVersion=4 under Win98.
         { // -> We're running on a stoneage computer, using a stoneage OS,
           // most likely the unsuccessful call of OpenInputDesktop()
           // must be IGNORED in this case :
           fOkToGetCursorPos = TRUE; // Call GetCursorPos() even though OpenInputDesktop() failed !
         }
      }
   }
  if( fOkToGetCursorPos )
   { if( GetCursorPos( &pt ) )
      { *piScreenX = pt.x;
        *piScreenY = pt.y;
        return TRUE;
      }
   } // end if < Ok To GetCursorPos >


  // Arrived here ?
  // Bad luck; windoze doesn't allow reading the mouse-coord at the moment.
  return FALSE;
} // end YHF_GetMouseCursorPos()
