// File:  C:\cbproj\Remote_CW_Keyer\Debouncer.h
// Date:  2024-05-26
// Author:  Wolfgang Buescher (DL4YHF)
// Purpose: "Instantiatable" Debouncer for digital inputs.
//          First used in the author's "Remote CW Keyer" project,
//          to debounce the "dash" and "dot" signals from a
//          Morse key, before processing in the 'Elbug' emulator.
//
// Revision History (latest entry FIRST):
//   2024-05-26: Created this module for the "Remote CW Keyer"
//               when in Iambic Mode "A" (where, according to DJ5IL,
//               dash- and dot-memory flags should only be set on a
//               TRANSITION of the key from UNPRESSED to PRESSED),
//               keying sometimes did not work as expected, caused
//               by the morse keys's contact bounce (mechanical vibration,
//               especially when a Kent double-level squeeze key was
//               "kicked deliberately hard". This was NOT caused by
//               the usual 'dust between contacts):
//                       pressed       _____
// input (key contact)           ___|||     |____
//                     unpressed 
//                               -->|-|<-- Bouncing for a few milliseconds,
//                               even visible on the Keyer's "timing scope".
//

#include "Debouncer.h" // header file for THIS module ("digital input debouncer")

//---------------------------------------------------------------------------
void Debouncer_Init( T_Debouncer *pDebouncer, int iMinStableIntervals )
{
  pDebouncer->iMinStableIntervals = iMinStableIntervals;
  pDebouncer->fPrevInput = pDebouncer->fCurrOutput = FALSE;
  pDebouncer->iBounceCountdown = 0;
} // end Debouncer_Init()

//---------------------------------------------------------------------------
BOOL Debounce( T_Debouncer *pDebouncer, BOOL fNewInput )
  // Debounces a single digital input (instantiatable).
  // Intended to be called in REGULAR INTERVALS, to eliminate the need
  // for a fancy "timer API" called from here.
{
  if( fNewInput != pDebouncer->fPrevInput )
   { pDebouncer->iBounceCountdown = pDebouncer->iMinStableIntervals;
     pDebouncer->fPrevInput = fNewInput;
   }
  if( pDebouncer->iBounceCountdown > 0 ) // contact still bouncing ?
   { pDebouncer->iBounceCountdown--; // wait for the next call..
   }
  else // input signal "stable" for a sufficient time (or "debouncer is OFF"):
   { pDebouncer->fCurrOutput = fNewInput; // accept the "new input" as "current output"
   }
  return pDebouncer->fCurrOutput;
} // end Debounce()


/* EOF < Debouncer.c > */

