// File:  C:\cbproj\Remote_CW_Keyer\CwStreamEnc.h
// Date:  2024-01-01
// Author:  Wolfgang Buescher (DL4YHF)
// Purpose: 'CW Stream Encoder / Decoder' .
//          Converts an simple on/off keyed Morse code signal
//          into a series of bytes in the format described
//          in the *implementation* (module CwStreamEnc.c) .


#ifndef  _CW_STREAM_ENCODER_H_  // protect against multiple header inclusion
# define _CW_STREAM_ENCODER_H_

//---------------------------------------------------------------------------
// Data Types
//---------------------------------------------------------------------------

#ifndef  BYTE
# define BYTE unsigned char /* who needs a "uint8_t" ? We don't. */
#endif

typedef struct t_CW_KEYING_FIFO_ELEMENT
{
  BYTE bCmd; // one byte with a key-up/down bit + timestamp.
       // '--> Specification of each byte in bFifo[] :
       //        bit 7 :  H="key down" (CW carrier ON)
       //                 L="key up"   (CW carrier OFF)
       //    bits 6..0 : SEVEN-BIT time to wait before emitting the new state,
       //                 encoded by CwStreamEnc_MillisecondsTo7BitTimestamp() as follows:
       //        0 .. 31   :  direct value MILLISECONDS (1 ms resolution)
       //     0x20 .. 0x3F :  32 +  4 * (value-0x20) milliseconds (4 ms resolution)
       //     0x40 .. 0x7F : 157 + 16 * (value-0x40) milliseconds (16 ms resolution)
       // Again: The 7-bit timestamp is the time to wait between the previous
       //        "keying command" and the emission of the NEW state (from bit 7),
       //        in other words: The time to wait BEFORE sending bit 7 .
       //      * At the BEGIN of a new "over" (beginning with key-down),
       //        the SEVEN-BIT timestamp will be ZERO because there is no
       //        PREVIOUS symbol to wait for (-> only wait for the LATENCY).
       //      * To let the receiver know the END of an over,
       //        the CW-sending client(*) emits a second "key up"-command (bit 7 cleared),
       //        as soon as ten dot-times(?) or 500 ms have elapsed.
       //        (*) See KeyerThread() -> CwStream_EncodeKeyUpDownEvent() .

  long i32TimeOfReception_ms; // value returned by TIM_ReadHighResTimer_ms() *on reception* of bCmd       


} T_CW_KEYING_FIFO_ELEMENT;  // -> a single element in MorseTxFifo, MorseRxFifo


typedef struct t_CW_KEYING_FIFO
{
#define CW_KEYING_FIFO_SIZE 128 // Expect up to 128 high-to-low and low-to-high
  //               transitions in the SIGNAL during the worst-case network latency
  //               (CwNet.h : MyCwNet.cfg.iNetworkLatency_ms : up to 1000 ms),
  //               and a hypothetical high speed of 60 WPM  ~~ 20 ms per dot .
  //  -> 1000 ms / 20 ms = 50 dot times that need to be buffered (worst case)
  //  -> Add some headroom, and use a POWER OF TWO for the FIFO size :
  //               128 bytes per "CW stream FIFO" should be more than enough.
  T_CW_KEYING_FIFO_ELEMENT elem[ CW_KEYING_FIFO_SIZE ]; // one byte with a key-up/down bit + timestamp.
  int iHeadIndex;
  int iTailIndex;
} T_CW_KEYING_FIFO;  // -> MorseTxFifo, MorseRxFifo


//---------------------------------------------------------------------------
// Function Prototypes
//---------------------------------------------------------------------------

#ifndef CPROT   // for peaceful co-existence of C and C++ ...
#ifdef __cplusplus
 #define CPROT extern "C"
#else
 #define CPROT
#endif  // not "cplusplus" ?
#endif // ndef CPROT ?


//--------------------------------------------------------------------------
CPROT BYTE CwStreamEnc_MillisecondsTo7BitTimestamp( int nMilliseconds );
  // Converts a time in milliseconds (range : 0 .. 1040 ms)
  // into a SEVEN-BIT timestamp as explained in the introduction
  // in Remote_CW_Keyer\CwStreamEnc.c .
  // Used by the "CW Stream ENCODER" .

//--------------------------------------------------------------------------
CPROT int CwStreamEnc_7BitTimestampToMilliseconds( BYTE b7BitTimestamp );
  // Inverse to CwStreamEnc_MillisecondsTo7BitTimestamp() .
  // Used by the "CW Stream DECODER" .

//--------------------------------------------------------------------------
CPROT int  CwStream_EncodeKeyUpDownEvent( T_CW_KEYING_FIFO *pTxFifo, BOOL fKeyDown, int nMillisecondsToWait );
CPROT void CwStream_DecodeKeyUpDownEvent( T_CW_KEYING_FIFO_ELEMENT *pElem, BOOL *pfKeyDown, int *pnMillisecondsToWait );
CPROT int  CwStream_GetNumMillisecondsBufferedInFifo( T_CW_KEYING_FIFO *pRxFifo );
CPROT BOOL CwStream_CheckForAnyEndOfTransmissionInFifo(T_CW_KEYING_FIFO *pRxFifo);

#endif // ndef _CW_STREAM_ENCODER_H_ ?


/* EOF < CwStreamEnc.h > */

