// File:  C:\cbproj\Remote_CW_Keyer\CwStreamEnc.c
// Date:  2024-01-01
// Author:  Wolfgang Buescher (DL4YHF)
// Purpose: Low-bandwidth 'CW Stream Encoder / Decoder' .
//          Converts an simple on/off keyed Morse code signal
//          into a series of bytes in the format described below.
//          This format allows to reproduce even the worst
//          "hand-sent" Morse character timing .
//  Principle: Each byte sent as part of a low-bandwidth stream
//             contains the "key up / key down" - state in
//             BIT SEVEN .   Bits SIX TO ZERO in the same byte
//             inform the receiver about "how long to wait"
//             ***before*** emitting BIT SEVEN to the transmitter's
//             Morse-keying input (configured as STRAIGHT KEY input).
//   Note:  * The timestamp in bits 6..0 applies to the time
//            to wait BEFORE outputting the signal from bit 7.
//          * After a long gap, we IMMEDIATELY want to send
//            a "key down" command to reduce latencies.
//            In this case, the length of the gap is irrelevent,
//            so the first byte to send is 0x80 ("key down IMMEDIATELY, don't wait").
//          * After a dash or dot, we know the pulse duration,
//            and can send it along with the "key up" command.
//          * For longer delays, and slow CW, the time resolution
//            doesn't need to be as good as for fast CW,
//            thus the SEVEN BIT TIMESTAMP is encoded non-linear,
//            as specified further below (along with a look-up
//            table to easily convert from the seven-bit timestamp
//            into milliseconds and back).
//          * In reality, if e.g. the network latency exceeds the duration
//            of a 'dot', things get a bit more complicated, but this is
//            beyond the scope of module CwStreamEnc.c .
//
//      Example (with a "not-so-perfect" Morse code signal,
//               and a timing diagram when the encoder sends what):
//
//      key "down"        _____      ___________         ___
//   (Carrier ON)        |     |15ms|           |  31ms |   |
// [in]                  |<--->|<-->|<--------->|<----->|   |
//      key "up"    _____| 20ms|____|    40 ms  |_______|   |_______
//   (Carrier OFF)
//                       |     |    |           |       |   |
//                      \|/   \|/  \|/         \|/     \|/ \|/
//      Bytes sent:     0x80  0x14 0x8F        0x22    0x9F ....
//                        |     |
//                        |     Most significant bit CLEARED:
//                        |     "Turn the CARRIER OFF"
//                        |     (here, after waiting for 0x14 = 20 milliseconds)
//                        Most significant bit SET:
//                        "Turn the transmitter's CARRIER ON"
//                          (or start beeping in the speaker,
//                           when used for the 'CW Network'
//                           without any real radio involved)
//                        Here, with the lower 7 bits CLEARED,
//                        do that IMMEDIATELY (don't wait) .
//
// Encoding of seven-bit timestamps, "suitable for high- and low speed CW":
//  * t = 0  to 31  milliseconds are encoded as such (in milliseconds)
//  * t = 32 to 156 milliseconds are encoded as 32 + (t-32)/4
//           (resulting in encoded values from 0x20=32 to 0x3F=63)
//  * t = 157 to 1165 milliseconds are encoded as 64 + (t-157)/16
//           (resulting in encoded values from 0x40=64 to 0x7F=127)
//  * pulses or gaps longer than 1165 milliseconds may be sent as subsequent
//           bytes with the same "key down" / "key up" state in the MSBit.


#include <string.h> // no string functions used here, but memset()

#include "Elbug.h" // header for the basic 'Elbug' functions (plain C)
#include "CwStreamEnc.h" // header for THIS module (Morse code "bytestream encoder")

//--------------------------------------------------------------------------
BYTE CwStreamEnc_MillisecondsTo7BitTimestamp( int nMilliseconds )
  // Converts a time in milliseconds (range : 0 .. 1040 ms)
  // into a SEVEN-BIT timestamp as explained in the introduction
  // in Remote_CW_Keyer\CwStreamEnc.c .
  // Used by the "CW Stream ENCODER" .
{ if( nMilliseconds < 0 )
   {  return 0x00;
   }
  if( nMilliseconds <= 31 )  // we're in the high-speed range ..
   {  return (BYTE)nMilliseconds; // timestamp resolution 1 ms
   }
  if( nMilliseconds <= 156 ) // we're in the moderate-speed range ..
   {  return (BYTE)(0x20+(nMilliseconds-32)/4); // timestamp resolution 4 ms
      // 0x20 + ( 156 - 32) / 4   = 63, q.e.d.  
   }
  if( nMilliseconds <= 1165) // we're in the slow-speed range ..
   {  return (BYTE)(0x40+(nMilliseconds-157)/16); // timestamp resolution 16 ms
      // 0x40 + (1165 - 157) / 16 = 127, q.e.d.
   }
  return (BYTE)127;  // if THE CALLER doesn't handle this properly,
      // use "the longest time we can encode in 7 bits" = 1165 ms .
} // end CwStreamEnc_MillisecondsTo7BitTimestamp()


//--------------------------------------------------------------------------
int CwStreamEnc_7BitTimestampToMilliseconds( BYTE b7BitTimestamp )
  // Inverse to CwStreamEnc_MillisecondsTo7BitTimestamp() .
  // Used by the "CW Stream DECODER" .
{
  b7BitTimestamp &= 0x7F; // just in case the caller didn't strip the "key up"/"key down" bit

  if( b7BitTimestamp <= 0x1F )     // 0..31 [ms] ...
   { return (int)b7BitTimestamp;   // .. > are encoded as such
   }
  if( b7BitTimestamp <= 0x3F )     // 0x20 .. 0x3F -> 32 to 156 [ms] ...
   { return 32 + 4 * (int)(b7BitTimestamp-0x20); // .. in 4-ms-steps
   }
  // else 0x40 .. 0x7F -> 157 to 1165 [ms] ... in 16-ms-steps
  return 157 +  16 * (int)(b7BitTimestamp-0x40);
} // end CwStreamEnc_7BitTimestampToMilliseconds()



//---------------------------------------------------------------------------
int CwStream_EncodeKeyUpDownEvent( // ... into a short FIFO, e.g. for sending over a network
        T_CW_KEYING_FIFO *pTxFifo, // [out] short FIFO for the transmitted low-bandwidth keying bytestream
        BOOL fKeyDown,             // [in] TRUE = "key down" = RF carrier "on", FALSE = RF carrier "off"
        int  nMillisecondsToWait ) // [in] number of milliseconds since the PREVIOUS signal edge .
              // > nMillisecondsToWait = 0 means "this is the begin
              // >                       of a new OVER, or at least a new WORD".
  // Called for example from KeyerThread() when the 'local operator' sends Morse code
  //                    via network (from his paddle adapter, TX memory, etc).
  // Return value: number of milliseconds that COULD ACTUALLY BE ENCODED
  //          in the seven least significant bits of the key up/down command.
  //          Since 2024-06-09, the CW Keyer Thread uses this information
  //          to avoid "slipping" and inaccuracies at high speed,
  //          when a dot time isn't an integer number of milliseconds.
{
  BYTE b, bNewHeadIndex;
  int  t2_ms; // duration of the PREVIOUS symbol, limited to what we can encode PER BYTE
  int  nMillisecondsEncoded = 0;


  do // repeat this if the duration is TOO LONG to be encoded in seven bits (1165 ms) :
   { t2_ms = ( nMillisecondsToWait > 1165 ) ? 1165 : nMillisecondsToWait;
     b = CwStreamEnc_MillisecondsTo7BitTimestamp( t2_ms ); // encoded timestamp in the lower seven bits
     if( fKeyDown )
      {  b |= 0x80;  // "key up" / "key down" flag in the most significant bit
      }
     bNewHeadIndex = pTxFifo->iHeadIndex & (CW_KEYING_FIFO_SIZE-1);  // safety first ..
     pTxFifo->elem[ bNewHeadIndex++ ].bCmd = b;
     pTxFifo->iHeadIndex = bNewHeadIndex & (CW_KEYING_FIFO_SIZE-1);  // <- new, circular wrapped HEAD index
     nMillisecondsToWait -= t2_ms;
     nMillisecondsEncoded += CwStreamEnc_7BitTimestampToMilliseconds( b & 0x7F );
   } while( nMillisecondsToWait > 0 );
  return nMillisecondsEncoded;
} // end CwStream_EncodeKeyUpDownEvent()

//---------------------------------------------------------------------------
void CwStream_DecodeKeyUpDownEvent( // ... into a short FIFO, e.g. for sending over a network
        T_CW_KEYING_FIFO_ELEMENT *pElem, // [in] FIFO element for the received Morse keying bytestream
        BOOL *pfKeyDown,            // [out] TRUE = "key down" = RF carrier "on", FALSE = RF carrier "off"
        int  *pnMillisecondsToWait) // [out] number of milliseconds TO WAIT
              //       **before** switching the 'CW keying output'
              //                  to the new state (*pfKeyDown).
              //  nMillisecondsToWait = 0 means "this is the begin
              //                        of a new OVER, or at least a new WORD".
              //
  // More or less the counterpart to CwStream_EncodeKeyUpDownEvent(),
  //         but trickier to use because, depending on the network latency,
  //         the caller may have to 'postpone' the begin of his 'CW playback'
  //         to avoid running out of data in the middle of a CW character.
{
  BYTE b = pElem->bCmd; // read the next 'CW keying event'..
     // Byte 'b' contains the "key up" / "key down" flag in BIT SEVEN,
     // and the encoded number of milliseconds to wait BEFORE emitting the new
     // state in bits SIX TO ZERO. More about the time-encoding
     // in CwStreamEnc_7BitTimestampToMilliseconds() .
  *pfKeyDown = (b & 0x80) != 0;  // -> TRUE = "key down" = RF carrier "on"; FALSE = "key up, carrier OFF"
  *pnMillisecondsToWait = CwStreamEnc_7BitTimestampToMilliseconds( b & 0x7F );
} // end CwStream_DecodeKeyUpDownEvent()


//---------------------------------------------------------------------------
int CwStream_GetNumMillisecondsBufferedInFifo( // ... for the 'intelligent latency control' in KeyerThread.c ...
        T_CW_KEYING_FIFO *pRxFifo)  // [in] short FIFO for the received low-bandwidth keying bytestream
    // Returns the NUMBER OF MILLISECONDS of "keying patterns" currently available
    // in the 'CW keying FIFO' (with one byte per "key up" / "key down" event),
    // by summing up the seven-bit timestamps in each entry between the
    // FIFO head and tail .
{
  int nMillisecondsBuffered = 0;
  BYTE b;
  int  index = pRxFifo->iTailIndex & (CW_KEYING_FIFO_SIZE-1);  // safety first ..
  while( index != pRxFifo->iHeadIndex )   // repeat for all VALID entries in the FIFO
   { b = pRxFifo->elem[ index ].bCmd;  // read the next 'CW keying event'..
     // Byte 'b' contains the "key up" / "key down" flag in BIT SEVEN,
     // and the encoded number of milliseconds to wait BEFORE emitting the new
     // state in bits SIX TO ZERO. More about the time-encoding
     // in CwStreamEnc_7BitTimestampToMilliseconds() .
     index = (index+1) & (CW_KEYING_FIFO_SIZE-1);  // <- new, circular wrapped TAIL index
     nMillisecondsBuffered += CwStreamEnc_7BitTimestampToMilliseconds( b & 0x7F );
   }
  return nMillisecondsBuffered;
}

//---------------------------------------------------------------------------
BOOL CwStream_CheckForAnyEndOfTransmissionInFifo( // ... also for the 'intelligent latency control' in KeyerThread.c ...
        T_CW_KEYING_FIFO *pRxFifo)  // [in] short FIFO for the received low-bandwidth keying bytestream
    // Returns TRUE if the specified 'Morse-keying FIFO' is non-empty,
    // and already contains the special 'End-Of-Transmission'-flag,
    //     which usually is appended by the remote sender when detecting
    //     an 'inter-word gap' .
{
  BYTE b, bPrev=0;
  int  index = pRxFifo->iTailIndex & (CW_KEYING_FIFO_SIZE-1);  // safety first ..
  BOOL first_entry = TRUE;
  while( index != pRxFifo->iHeadIndex ) // repeat for all VALID entries in the FIFO
   { b = pRxFifo->elem[ index ].bCmd;   // read the next 'CW keying event'..
     // Byte 'b' contains the "key up" / "key down" flag in BIT SEVEN,
     // and the encoded number of milliseconds to wait BEFORE emitting the new
     // state in bits SIX TO ZERO. More about the time-encoding
     // in CwStreamEnc_7BitTimestampToMilliseconds() .
     // Regardless of the seven-bit-encoded time, TWO CONSECUTIVE "key up"-
     // commands (aka 'CW carrier OFF') indicate END-OF-TRANSMISSION.
     index = (index+1) & (CW_KEYING_FIFO_SIZE-1);  // <- new, circular wrapped TAIL index
     if( !first_entry )
      { if( ((b & 0x80)==0) && ((bPrev & 0x80)==0) )
         { return TRUE;  // saw the "End-Of-Transmission" (EOT) !
         }
      }
     bPrev = b;
   }
  return FALSE; // did NOT see any "End-Of-Transmission" (EOT) in the filled part of the FIFO
} // end CwStream_CheckForAnyEndOfTransmissionInFifo()


/* EOF < CwStreamEnc.c > */

