// File:  C:\cbproj\Remote_CW_Keyer\CwNet.h
// Date:  2024-01-01
// Author:  Wolfgang Buescher (DL4YHF)
// Purpose: Socket-based 'Client and Server for the Remote CW Keyer'.
//          As usual, details only in the implementation (CwNet.c) .

#ifndef  _CW_NET_H  // prevent multiple header inclusion ..
# define _CW_NET_H  //  .. and let other modules know we're been included

#ifndef SWI_USE_VORBIS_STREAM // still NOT defined ?
# error "Missing definition of SWI_USE_VORBIS_STREAM in 'switches.h' !"
#endif

#include "Timers.h"      // <- T_TIM_Stopwatch, T_TIM_Stopwatch(), etc...

#include "CwStreamEnc.h" // <- Low-bandwidth 'CW Stream Encoder / Decoder'

#if( SWI_USE_VORBIS_STREAM ) // allow streaming audio via Ogg/Vorbis ?
# include "VorbisStream.h"   // .. use this wrapper for "libvorbis" and "libogg"
#endif // SWI_USE_VORBIS_STREAM ?

#include "RigControl.h" // .. delivers info ABOUT the radio, and waterfall data FROM the radio, etc..
                        // (so far, there's exactly ONE "rig control" per "CW Network Server",
                        //  thus a T_RigCtrlInstance is a member of our T_CwNet struct)
#if( SWI_USE_HAMLIB_SERVER ) // build an application with integrated "Hamlib Net rigctld"-compatible server ?
# include "HamlibServer.h"  // "Hamlib Net rigctld"-compatible TCP server
#endif // SWI_USE_HAMLIB_SERVER ?

#include "CwDSP.h"     // 'CW Digitial Signal Processor' (audio I/O and processing)

//---------------------------------------------------------------------------
// Constants
//---------------------------------------------------------------------------

#define CWNET_MAX_CLIENTS 3 // maxium number of SIMULTANEOUSLY connected clients.
        // Each of them must be fed with AUDIO and SPECTRUM DATA,
        // which will quickly max out on a 6-Megabit domestic ADSL connection !
        // Guesstimate: 8 kSamples/second * 8 bits for the AUDIO signal,
        //         plus 115 kBit/second for Icom's SPECTRUM VIA Virtual COM Port,
        //         plus a neglectable bandwidth for the Morse code keying signal
        // -> 3 clients * ( 64 + 115 kBit/second) = 537 kBit / second
        //         ON THE DSL UPLINK !  Expect a maximum of 768 kbit/sec .

// Single-byte binary command codes used in the TCP streams between client and server.
// To reduce the protocol overhead, the presence of a BLOCK LENGTH, immediately
// in one or two byte AFTER the "CMD_CMD_xyz" is indicated by BITS SEVEN and SIX
// in the command:
#define CWNET_CMD_MASK_BLOCKLEN    0xC0 // Mask for bitwise AND-ing a received
          // single-byte command, for checking the two-bit combinations below:
#define CWNET_CMD_MASK_NO_BLOCK    0x00 // "no BLOCK LENGTH INDICATOR after the command byte"
#define CWNET_CMD_MASK_SHORT_BLOCK 0x40 // "short block" (ONE-BYTE length, i.e. max 255 byte payload)
#define CWNET_CMD_MASK_LONG_BLOCK  0x80 // "long block" (TWO-BYTE length, i.e. max 65535 byte payload)
#define CWNET_CMD_MASK_RESERVE     0xC0 // reserved for future use / special command 0xFF
// This allows encoding / embedding multiple commands, morse code fragments,
// audio sample blocks, Icom's CI-V messages, and who-knows-what-else in a
// single TCP segment. On this occasion: The receiving side needs a STREAMING
// PARSER, because the commands listed below don't necessary arrive in a
// single 'block' returned by the Socket API ("recv()") !
#define CWNET_CMD_MASK_COMMAND     0x3F // mask to strip 'the received command itself'
#define CWNET_CMD_NONE     0x00 // dummy command to send HTTP instead of our binary protocol
#define CWNET_CMD_CONNECT  0x01 // "Connect" (first command, actually a struct, from client to server)
                                //  or "Connected" (first response, same struct, from server to client)
#define CWNET_CMD_DISCONN  0x02 // "Disconnect" (at any time, in any direction)
#define CWNET_CMD_PING     0x03 // "Ping Test" (to check the connection's speed)
#define CWNET_CMD_PRINT    0x04 // "Text that the receiver should print somewhere",
                                //   for example the 'Welcome'-text, or even 'chat'
#define CWNET_CMD_TX_INFO  0x05 // Announcement to all connected users about
                                //   'how has the key now' (or the microphone).
                                // In the experimental "Remove CW Keyer" GUI,
                                // this was displayed on the 'Network' tab.
#define CWNET_CMD_RIGCTLD  0x06 // 'rigctld'-compatible ASCII string (plus a trailing ZERO) - see www.mankier.com/1/rigctld
#define CWNET_CMD_MORSE    0x10 // Hand-keyed or machine-generated MORSE CODE
#define CWNET_CMD_AUDIO    0x11 // Audio samples, 8-bit A-Law compressed, multiplexed into the TCP stream
#define CWNET_CMD_VORBIS   0x12 // future plan: Ogg/Vorbis compressed audio stream, optional, not suited for simple microcontrollers !
#define CWNET_CMD_RSV13    0x13 // reserved for some other audio stream format
#define CWNET_CMD_CI_V     0x14 // A single CI-V packet (passed to/from Icom transceivers)
#define CWNET_CMD_SPECTRUM 0x15 // A spectrum for waterfall/spectrum display, incl. T_CwNet_SpectrumHeader
#define CWNET_CMD_FREQ_REPORT 0x16 // Transceiver's frequency, mode, etc (T_RigCtrl_VfoReport)
#define CWNET_CMD_PARAM_INTEGER 0x18 // a single parameter, data type 'int', identified by RIGCTRL_PN_xyz, in a  T_RigCtrl_ParamReport_Integer
#define CWNET_CMD_PARAM_DOUBLE  0x19 // a single parameter, data type 'double', identified by RIGCTRL_PN_xyz, in a  T_RigCtrl_ParamReport_Double
#define CWNET_CMD_PARAM_STRING  0x1A // a single parameter, data type 'string', identified by RIGCTRL_PN_xyz, in a  T_RigCtrl_ParamReport_Double
#define CWNET_CMD_QUERY_PARAM   0x1B // request to send a certain parameter, identified by RIGCTRL_PN_xyz

#define CWNET_CMD_MULTI_FUNCTION_METER_REPORT 0x20 // a group of "meter readings" to emulate Icom's "Multi-Function Meter"
#define CWNET_CMD_POTI_REPORT                 0x21 // a group of "poti settings", exchanged less frequently than the "meter report"
#define CWNET_CMD_BAND_STACKING_REGISTER      0x22 // a "Band Stacking Register", converted into a string of key=value pairs by RigCtrl_FreqMemEntryToString()
#define CWNET_CMD_USER_DEFINED_BAND           0x23 // a "User-defined band", converted into a string of key=value pairs by RigCtrl_UserDefinedBandToString()


#define CWNET_CMD_SERIAL_PORT_TUNNEL_ANY 0x30 // "Serial Port Tunnel" configured to RECEIVE FROM ANY TUNNEL", connection between TCP/IP and one or more 'Additional COM Ports' in mode RIGCTRL_PORT_USAGE_SERIAL_TUNNEL
#define CWNET_CMD_SERIAL_PORT_TUNNEL_1   0x31 // first "Serial Port Tunnel", connection between TCP/IP and one or more 'Additional COM Ports' in mode RIGCTRL_PORT_USAGE_SERIAL_TUNNEL
#define CWNET_CMD_SERIAL_PORT_TUNNEL_2   0x32 // second "Serial Port Tunnel", routable between other 'Additional COM Ports' with the same configurable 'TunnelIndex' .
#define CWNET_CMD_SERIAL_PORT_TUNNEL_3   0x33 // ( see details in AuxComPort.s : AuxCom_OnReceptionFromNetworkPortTunnel() )


#define CWNET_CMD_RESERVE  0xFF // "special command", reserved for future use

// Fixed payload sizes for SOME of the above "commands" :
#define CWNET_PAYLOAD_SIZE_PING 16  // REQ/RESP, ID, two alignment dummy bytes, and THREE 32-bit timestamps
                                    // as explained in CwNet.c : CwNet_ExecuteCmd() / case CWNET_CMD_PING .

// Fixed audio sample rate used in all NETWORK AUDIO STREAMS :
#define CWNET_STREAM_SAMPLING_RATE 8000  // 8 kSamples/second is sufficient for CW,USB,LSB (and even NBFM)

//---------------------------------------------------------------------------
// Data Types
//---------------------------------------------------------------------------

#define CWNET_CHAT_FIFO_SIZE 1024
typedef struct t_CwNetChatFifo // .. to exchange SHORT text messages between sysop and users,
{ T_CFIFO fifo; // using the 'Chatbox' optionally visible on the bottom of the 'TRX' tab.
# if( CWNET_CHAT_FIFO_SIZE > SWI_CIRCULAR_FIFO_SIZE )
  BYTE bExtraBytes[ CWNET_CHAT_FIFO_SIZE - SWI_CIRCULAR_FIFO_SIZE ];
  //                 '--> e.g. 512 bytes     '--> e.g. 256 bytes
# endif
} T_CwNetChatFifo; // -> T_CwNet.Client[0..CWNET_MAX_CLIENTS].sChatRxFifo, sChatTxFifo



typedef struct t_CwNetClient // struct for an ARRAY OF CLIENTS, managed by our SERVER;
{ // only ONE of these [index 0] is used for our LOCAL CLIENT (when "running as a client, not server").

# define CWNET_CMD_DATA_BUFFER_SIZE 2048 // <- a POWER OF TWO, and long enough for spectra, audio sample blocks, 'rigctld' strings, etc ..
  BYTE bCmdDataBuffer[CWNET_CMD_DATA_BUFFER_SIZE]; // <- a BYTE ARRAY, but on an 8-byte-aligned offset within this struct
  DWORD dwMagicPI;  // 0x31415926 (magic number gets distroyed on array index violation in the preceding array)
# define CWNET_MAGIC_PI 0x31415926

  int iClientState;  // CWNET_CLIENT_STATE_DISCONN, etc:
#     define CWNET_CLIENT_STATE_DISCONN    0
#     define CWNET_CLIENT_STATE_ACCEPTED   1  // for the SERVER SIDE: socket "accepted" a REMOTE client, but not successfully "logged in" yet
#     define CWNET_CLIENT_STATE_CONNECTED  1  // same value, similar meaning, but for the LOCAL CLIENT side
#     define CWNET_CLIENT_STATE_LOGIN_SENT 2  // "log-in sent, but not confirmed yet"
#     define CWNET_CLIENT_STATE_LOGIN_CFMD 3  // "log-in confirmed, identified a 'known user'"
#     define CWNET_CLIENT_STATE_LOGIN_HTTP 4  // "log-in confirmed, a 'known user', but HE USES HTTP (a web browser)
  union
   { BYTE b[4];        // "his" (the remote peer's) IPv4 address ...
     DWORD dw;         // .. also accessable as a 32-bit double word for easy comparison
   } b4HisIP;
  int  iHisPort;       // <- for the CLIENT SIDE, the remote server's "published" port number;
                       //    for the SERVER SIDE, the remote client's "ephemeral" port number.
  DWORD dwNumBytesSent, dwNumBytesRcvd; // .. total, since THIS remote client was 'accepted' by our server
  DWORD dwNumSegmentsRcvd; // .. to avoid flooding the 'debug log' ..
  BOOL  fUsesHTTP;     // Flag set in the first call of CwNet_OnReceive()
                       // when 'this guy' obviously tried to "HTTP" with us
                       // (he's either a bad guy, or tries to check a connection
                       //  using a web browser, mobile phone, or whatever) .
  BOOL  fWantVorbis;   // Flag set if this client wants Vorbis audio in Ogg containers
  BOOL  fDisconnect;   // flag polled in ServerThread() to disconnect a remote client,
                       // may be set by any 'command handler' invoked from the
                       // streaming parser, when detecting an ill-behaving client.
                       // Or (for use in ClientThread()), to disconnect from server.
  T_TIM_Stopwatch swActivityWatchdog;  // watchdog for this connection's activity
#     define CWNET_ACTIVITY_TIMEOUT_MS 5000 // max nr. of milliseconds w/o activity
                       // (also used for HTTP; frees a T_CwNetClient
                       //  AND the associated T_HttpInstance after a few seconds
                       //  of "no activity" )
  double dblUnixTimeOfStart; // <- set immediately after accepting, for bandwidth-calculations etc

  char sz80UserName[84];  // name that this current user has logged in with, NOT VISIBLE TO OTHER USERS on the server
                          // (only the server's sysop shall know the user names, and of course the user himself)
  char sz80Callsign[84];  // user's callsign, self-modifiable, VISIBLE TO OTHER USERS logged into the server
  char sz80TxInfo[84];    // Text exchanged with the last CWNET_CMD_TX_INFO .
                          // Originally only showed "how has the key at the moment".
                          // Modified along with T_CwNetClient.iAnnouncedTxClient .

  int iPermissions;    // ZERO or a bitwise combination of the following:
#     define CWNET_PERMISSION_NONE     0x00 // "this client may only RECEIVE but neither 'talk' or 'send in CW'
#     define CWNET_PERMISSION_TALK     0x01 // "this client may TALK to other users" (no RADIO TRANSMISSION)
#     define CWNET_PERMISSION_TRANSMIT 0x02 // "this client may TRANSMIT" (in CW, or -maybe one day- voice)
#     define CWNET_PERMISSION_CTRL_RIG 0x04 // "this client may control the remote rig"
#     define CWNET_PERMISSION_ADMIN    0x08 // "this client is actually the HTTP SERVER'S ADMINISTRATOR"
                     // (only admins may read the connection log via HTTP, etc)


  // Struct members for the simplistic bytestream parser:
  int iRxParserState;  // expected "meaning" of the NEXT RECEIVED (parsed) byte:
#     define CWNET_PSTATE_RX_CMD       0 // "the next byte we expect to receive is a COMMAND"
#     define CWNET_PSTATE_RX_SHORT_LEN 1 // "the next byte we expect is the SHORT BLOCK LENGTH"
#     define CWNET_PSTATE_RX_LENGTH_LO 2 // "the next byte we expect is the LSB of a 16-bit BLOCK LENGTH"
#     define CWNET_PSTATE_RX_LENGTH_HI 3 // "the next byte we expect is the MSB of a 16-bit BLOCK LENGTH"
#     define CWNET_PSTATE_RX_BLOCK     4 // "the next byte(s) we expect is/are DATA BYTES (in the block)"
  int iCurrentCommand;   // CWNET_CMD_xyz, applies to ALL BYTES received in the current 'block'
  int iCmdDataLength;    // block length belonging to iCurrentCommand
  int iCmdDataByteIndex; // 0..iCmdDataLength, incremented in state CWNET_PS_RX_BLOCK
  int nGarbageBlocksReceived;

  int  iPingTxCountdown_ms; // "countdown timer" (in milliseconds) until the next "ping test".
  long i32PingTimestamp_ms[3]; // THREE "Ping timestamps" - details in CwNet_ExecuteCmd(), case CWNET_CMD_PING .
  int  iPingLatency_ms;     // "Ping" trip time in milliseconds. This is important
  T_TIM_Stopwatch swPing;   // for emitting received Morse code patterns:
                            // Keep approximately <iPingLatency_ms> milliseconds of
                            // 'Morse code patterns' buffered somewhere
                            // to avoid running out of data when sending Morse code
                            // on behalf of this remote client.

  int  iAnnouncedTxClient;  // Last value of T_CwNet.iTransmittingClient
                            // that has been 'announced' via CWNET_CMD_TX_INFO
                            // to a client (along with a sz80TxInfo, not just this index) .
                            // When running as server, this field is also valid
                            // for the 'local display' (server admin info) in
                            // T_CwNet.Client[CWNET_LOCAL_CLIENT_INDEX] .
  int  iSentTransmitReqst;  // Last state of TRigControl.iTransmitReqst (possibly MANUALLY CONTROLLED PTT),
                            //    *sent* from the local client to remote server
                            //    either via CI-V tunnel (cmd 0x1C subcmd 0x00 data 0x00/0x01)
                            //    or in a non-Icom-specific ASCII command "set_ptt 0" / "set_ptt 1".
                            // ,------------------------------------------|_________|
                            // '--> Here: "rigctld"-compatible string wrapped in a
                            //      CWNET_CMD_RIGCTLD block, assembled in CwNet_OnPoll() .
  double dblReportedVfoFrequency; // last VFO frequency (etc) exchanged between client and server
                                 // as part of a "VFO report" (which may be a request to "QSY")
  T_TIM_Stopwatch sw_VfoReport;  // 'stopwatch' for the last transmission of sVfoReport
  DWORD dwHashFromLastVfoReport; // UTL_CRC32() calculated on the last sVfoReport sent to this client

  T_TIM_Stopwatch sw_MultiFunctionMeterReport; // stopwatch to limit the rate of "Multi-Function Meter" reports
  DWORD dwHashFromLastMeterReport; // UTL_CRC32() calculated on the last sMultiFunctionMeterReport sent to this client
          // (a T_RigCtrl_MultiFunctionMeterReport contains a dozen of parameters,
          //  including a few 'potentiometer settings' in Icom transceivers,
          //  so a couple of 'unified parameter numbers' (defined in RigControl.h)
          //  don't need extra copies of the 'last sent values' in this T_CwNetClient:
  T_TIM_Stopwatch sw_PotiReport;  // stopwatch to limit the rate of "Poti" reports
  DWORD dwHashFromLastPotiReport; // UTL_CRC32() calculated on the last "Poti" report sent to this client

# define CWNET_TX_PARAMETER_QUEUE_LENGTH 16
  int  iTxParameterQueue[CWNET_TX_PARAMETER_QUEUE_LENGTH];
  int  iTxParameterQueueHeadIndex, iTxParameterQueueTailIndex;

  int  iS2CAudioTail; // audio sample TAIL index into a circular FIFO of the DSP,
          // to send A-Law compressed audio from Server To Client (thus "S2C") .
  int  iSpectrumBufferTail; // Spectrum buffer (FIFO) tail index, used when
                            // sending spectra (Icom slang: "Waveform Data")
                            // retrieved via RigCtrl_GetSpectrumFromFIFO() .
  double dblScopeRefLevel_dB;    // last value of the "Scope REF level" sent to this client
  double dblScopeSpan_Hz;        // last value of the "Scope SPAN" sent to this client
  int  iBandStackingRegsTxIndex; // negative when NOT sending band-stacking registers (or a simple list of supported bands),
       // otherwise an index into T_CwNet.RigControl.BandStackingRegs[RIGCTRL_NUM_BAND_STACKING_REGS].
       // While sending the band-stacking-registers from server to a certain client,
       // iBandStackingRegsTxIndex runs from 0 to T_CwNet.RigControl.iNumBandStackingRegs .

   T_CwNetChatFifo sChatRxFifo, sChatTxFifo; // receive- and transmit-FIFOs for 'text chat'.

# if( SWI_USE_VORBIS_STREAM ) // support streaming audio via Ogg/Vorbis ?
   // There is only ONE instance of a T_VorbisStream to encode the
   // audio signal for MULTPLE clients connected to our server,
   // because we only want to Vorbis-compress and Ogg-package the audio
   // ONCE, regardless of the number of clients currently connected
   // via CwNet's beautifully simple binary protocol, or the bulky chatty HTTP.
   // For that reason, the T_VorbisStream is neither part of T_CwNetClient
   //          nor the HTTP server's T_HttpInstance in HttpServer.h,
   //          but part of the single-instance T_CwNet.
   // The only PER-CLIENT info for streaming Vorbis-compressed audio
   // is the 'page buffer tail index', as explained in VorbisStream.h .
   // See "Short overview / most important functions for AUDIO STREAMING" .
  DWORD dwNextVorbisPage; // per-client "tail index" for VorbisStream_GetNextPagesForStreamOutputServer()
# endif // SWI_USE_VORBIS_STREAM

# if( SWI_NUM_AUX_COM_PORTS > 0 )
  int iTxBufferTailForAuxComPort[SWI_NUM_AUX_COM_PORTS];
# endif // SWI_NUM_AUX_COM_PORTS > 0 ?



  struct t_HttpInstance *pHttpInst; // additional instance data for a remote client
                            // using HTTP (e.g. with a browser). So far, no real
                            // use besides a PORT FORWARDING TEST; for example to
                            // find out if the company's restrictive IP infra-
                            // structure would allow accessing a remote HTTP server.
                            // Details (and the actual typedef T_HttpInstance)
                            // in HttpServer.c / HttpServer.h .
  int iTestTonePhaseAccu; // phase accumulator for the optional AUDIO TEST TONE,
                          // generated on-the-fly, and ADDED to the 'streamed-out'
                          // signal in e.g. CwNet_OnPoll()
  int nPeriodicMessagesInTrafficLog; // .. to show the first few 'periodic' messages in the traffic log,
                          // despite T_CwNet.cfg.iDiagnosticFlags.CWNET_DIAG_FLAGS_REJECT_PERIODIC_MSGS.
  int nUnknownMessagesInTrafficLog;  // similar for messages that POSSIBLE are not handled
                          // in a long switch/case statement of the streaming parser
  int nMeterReportsInTrafficLog; // etc, etc (similar purpose; limit the number of messages in the traffic log)

} T_CwNetClient; // -> T_CwNet.Client[CWNET_MAX_CLIENTS+1] .

typedef struct t_CwNetRecentClientEntry
{ union
   { DWORD dw;
     BYTE  b[4];
   } b4HisIP;
  int   iPermissions;
  T_TIM_Stopwatch swLastSeen;
} T_CwNetRecentClientEntry;


typedef struct t_CwNet // data for a single 'CW Network Client or Server' instance:
{

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // "Configuration parameters" (must be filled out be the application
  //          before start, usually from 'visual components' in the GUI
  //          with similar names as the struct members used here)
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  struct { // .cfg = "everything that needs to be SAVED between sessions" (and included in a hash value):
     int  iFunctionality; // CWNET_FUNC_OFF / .._CLIENT / .._SERVER
#      define CWNET_FUNC_OFF    0
#      define CWNET_FUNC_CLIENT 1
#      define CWNET_FUNC_SERVER 2
     char sz80ClientRemoteIP[84]; // remote server IP:PORT (or URL:PORT) for the client
     char sz80ClientUserName[84]; // user name for the "log-in" (into a remote client)
     char sz80ClientCallsign[84]; // user's modifiable callsign (e.g. "DL4YHF", "DL4YHF/p", "DL4YHF/r", "DL4YHF testing")
     int  iServerListeningPort;   // server's listening port (to listen for
                                  // incomining connection requests, similar
                                  // as a multi-client HTTP server .. but simpler)
#      define CWNET_DEFAULT_SERVER_PORT 7355 // <- CW amateurs will know ..
                               // but your office internet router will block it!
     char sz255AcceptUsers[256];  // comma-separated list of user names that the
                                  // server shall 'accept', and their permissions
     char sz20ServerAdminPWD[24]; // HTTP server administrator password,
                                  // allows reading 'server log', 'error log', etc,
                                  // which may contain user's IP addresses and similar
                                  // info that must not leak out to normal visitors.
     int iHttpServerOptions; // bitwise combination of the HTTP_SERVER_OPTIONS
                             // defined in e.g. Remote_CW_Keyer/HttpServer.h .
     int iNetworkLatency_ms; // fixed network latency in milliseconds (0 = auto-detect)

     int iDiagnosticFlags;  // bitwise combination of the following (no extra enum, just hex bitmasks) :
#      define CWNET_DIAG_FLAGS_NONE                   0x0000 /* "dont show anything of the stuff listed below" ..     */
#      define CWNET_DIAG_FLAGS_SHOW_DECODER_OUTPUT    0x0001 /* show output from the Morse decoder on the 'Debug' tab */
#      define CWNET_DIAG_FLAGS_SHOW_KEYING_BYTESTREAM 0x0002 /* show transmitted or received 'keying bytesstream', e.g. on the 'Debug' tab */
#      define CWNET_DIAG_FLAGS_SHOW_CONN_LOG          0x0004 /* show 'connection log' (remote clients connecting / disconnecting) " " " */
#      define CWNET_DIAG_FLAGS_SHOW_NETWORK_TRAFFIC   0x0008 /* show 'network traffic' (HEX or ASCII) on the 'Debug' tab*/
#      define CWNET_DIAG_FLAGS_VERBOSE                0x0010 /* produce verbose output via ShowError() (awful, only for debugging) */
#      define CWNET_DIAG_FLAGS_SIMULATE_BAD_CONN      0x0020 /* .. with high latency, and large jitter on the latency */
#      define CWNET_DIAG_FLAGS_REJECT_PERIODIC_MSGS   0x0040 /* .. in the "Network Traffic" log (not the "Rig Control" log) */
#      define CWNET_DIAG_FLAGS_WORD_WRAP              0x0100 /* word-wrapped display on the 'Debug' tab */
#      define CWNET_DIAG_FLAGS_SHORT_HISTORY          0x0200 /* "Limit the history to 100 lines ( avoids being a CPU hog :)" */
#      define CWNET_DIAG_FLAGS_SHOW_GUI_EVENTS        0x0400 /* ... not necessarily limited to the "CW network", anyway ...  */
       // Note: Module RigControl.c has it's own "logging control flags",
       // see RigControl.c: RigCtrl_TrafficMonitor.iDisplayOptions.RIGCTRL_TMON_DISPLAY_OPTION_ENABLED, etc .
   } cfg; // <- end of the part that needs to be stored permanently (between sessions)


  // Classic, circular, lock-free FIFOs with the 'keying signal' ..
  // (a) for TX (transmit): Filled in KeyerThread.c : KeyerThread(),
  //                     drained in  CwNet.c : ClientThread() -> CwNet_OnPoll():
  T_CW_KEYING_FIFO MorseTxFifo; // transmit FIFO for CW keying
  DWORD dwMagicPI_1; // 0x31415926 (magic number gets distroyed on array index violation in the preceding object)
#     define CWNET_MAGIC_PI 0x31415926

  BOOL  fFillingTxFifo;  // TRUE when actively filling MorseTxFifo,
                         // FALSE when passive.
  //
  // (b) for RX (receive): Filled in CwNet.c : ServerThread() -> CwNet_OnReceive(),
  //                       if the remote client 'has the key' (may transmit) .
  T_CW_KEYING_FIFO MorseRxFifo; // receive FIFO for CW keying, also acts as a
                                // 'delay line' that shall not run empty even
                                // if there is JITTER(!) on the network latency
                                //  - see KeyerThread.c : KeyerThread().
  BOOL  fSendingFromRxFifo; // TRUE when actively draining MorseTxFifo.
                         //  set only AFTER seeing 'enough milliseconds buffered'
                         //  for the configurable (or measured) network latency.
  T_TIM_Stopwatch sw_MorseRxFifo; // accurate 'stopwatch' to send from MorseRxFifo
  BOOL fCurrentRemotelyReceivedKeyDownState, fNextRemotelyReceivedKeyDownState;


  char  *pszLastError;  // NULL when there's nothing to report (in the error log)
  char  sz255LastError[256]; // buffer for an "sprintf'ed" error message
  DWORD dwMagicPI_2;  // also should contain CWNET_MAGIC_PI, sanity check for the struct


  HANDLE hThread; // also a HANDLE, but for a THREAD, not a file,
                  // thus "NULL" means invalid (not INVALID_HANDLE_VALUE). Bleah.
         // Note: ServerThread() and ClientThread() never run SIMULTANEOUSLY
         //   in a single T_CwNet instance, thus only ONE thread-handle, etc.
  DWORD  dwThreadId; // a (windows-) thread-ID, possibly required to send events to our thread.
  int iThreadStatus; // what's our client- or server thread doing ? one of these:
#   define CWNET_THREAD_STATUS_NOT_CREATED 0
#   define CWNET_THREAD_STATUS_LAUNCHED    1
#   define CWNET_THREAD_STATUS_RUNNING     2 // <- only set ONCE(!), immediately when entering the thread function
#   define CWNET_THREAD_STATUS_TERMINATE   3 // <- just a KLUDGE to "request POLITE termination", but takes a few dozen milliseconds to actually TERMINATE !
#   define CWNET_THREAD_STATUS_TERMINATED -1 // <- set by the worker thread if it TERMINATED ITSELF

  DWORD dwThreadLoops;       // number of thread loops, counted in ClientThread() or ServerThread()
  DWORD dwThreadErrors;      // number of errors (mostly TIMING ERRORS) counted the above thread
  DWORD dw8ThreadIntervals_us[8]; // number of MICROSECONDS per loop in ClientThread() or ServerThread()
  DWORD dwNumBytesSent, dwNumBytesRcvd;  // .. total, since the last CwNet_Start()


#   define CWNET_SOCKET_RX_BUFFER_SIZE 16384 // size of our receive buffer; ex: 1024 bytes (*)
  BYTE  bRxBuffer[CWNET_SOCKET_RX_BUFFER_SIZE];
  DWORD dwMagicPI_3;  // also should contain CWNET_MAGIC_PI, sanity check for the struct
  int   nBytesInRxBuffer, iPeakRxBufferUsage;
        // only used in either ServerThread() or ClientThread(),
        // but we don't want a 'stack variable' for this object
        // so that pointers into the above buffer never get invalid
        // as long as our T_CwNet instance exists.
#   define CWNET_SOCKET_TX_BUFFER_SIZE 16384 // size of our transmit buffer; ex: 1024 bytes (*)
  BYTE  bTxBuffer[CWNET_SOCKET_TX_BUFFER_SIZE];
  DWORD dwMagicPI_4;  // also should contain CWNET_MAGIC_PI, sanity check for the struct
  int   nBytesInTxBuffer, iPeakTxBufferUsage;
        // (*) bRxBuffer[] and bTxBuffer[] are used by ALL REMOTE CLIENTS
        //     that are possibly simultaneously connected to our server,
        //     or by the single local client connected to a REMOTE SERVER.
        //     For AUDIO STREAMING with 8 kSamples / second, 8-bit A-Law
        //     compression, and a worst-case transmission interval of 40 ms,
        //     at least 40 ms * 8 kHz = 320 bytes must be sent in each call
        //     of CwNet_OnPoll() .  But after the horrible results with
        //     'mobile internet' (Android smartphone as 'Mobile Hotspot'
        //     for a Windows PC on one side, and another Windows PC with
        //     16 MBit ADSL connection on the other side), expect a backlash
        //     of SEVERAL HUNDRED MILLISECONDS building up occasionally .
        //      - and we want to drain the network buffer rapidly, by offering
        //        recv() a much larger destination buffer than USUALLY necessary.
        //

  BOOL fDisableTx;          // Flag from server / sysop to ALL CLIENTS : TRUE when TRANSMIT DISABLED ("RX only" or "off-air code practicing")        
  BOOL fSimulateBadConnNow; // Flag only set when running with CWNET_DIAG_FLAGS_SIMULATE_BAD_CONN .
        // When SET, certain transmissions (unsolicited or responses) will be DISABLED
        // until the flag is CLEARED again. This simulated a very bad connection,
        // not only with a LARGE latency, but also a lot of JITTER on the latency.

  T_CwNetClient Client[CWNET_MAX_CLIENTS+1];  // <- an ARRAY OF REMOTE CLIENTS, managed by our SERVER
        // ,-----------------------------'
        // '--> "plus one" because array index ZERO is used for the LOCAL CLIENT,
        //       while array indices 1 ... CWNET_MAX_CLIENTS are REMOTE clients.
#   define CWNET_LOCAL_CLIENT_INDEX        0
#   define CWNET_FIRST_REMOTE_CLIENT_INDEX 1
  DWORD dwMagicPI_5;  // also should contain CWNET_MAGIC_PI, sanity check for the struct


  T_InetBlacklist sBlacklist; // automatically filled blacklist of 'bad guys'

#   define CWNET_MAX_RECENT_CLIENTS 16
  T_CwNetRecentClientEntry sRecentClients[CWNET_MAX_RECENT_CLIENTS];

  int nRemoteClientsConnected; // set in ServerThread(); read-only for anyone else
  int iTransmittingClient;     // index into Client[*] if a particular client
     // currently 'has the key' to TRANSMIT, along with CWNET_PERMISSION_TRANSMIT.
     // Only THIS client may fill T_CwNet.MorseRxFifo in CwNet_OnReceive() !
     // (*) iTransmittingClient = 0 = CWNET_LOCAL_CLIENT_INDEX means
     //                               THE SERVER'S OPERATOR "has the key".
     //     iTransmittingClient = CWNET_FIRST_REMOTE_CLIENT_INDEX .. CWNET_MAX_CLIENTS
     //                               means a REMOTE CLIENT currently "has the key".
     //     iTransmittingClient < 0 means "nobody has the key", and "anyone with
     //                               CWNET_PERMISSION_TRANSMIT may take it" .
  T_TIM_Stopwatch swTransmittingClient; // .. to detect when the 'transmitting client'
                               // drops the key, allowing others to take it.
  int  iSetPTTFromNetwork;  // State for manual PTT control, *received* on the server side
                            // from a "rigctld"-compatible "set_ptt" command.
                            // Contains RIGCTRL_NOVALUE_INT if the remote client
                            // does NOT use 'manual PTT control'. Set in CwNet_Rigctld_OnSetPTT().

  int iLocalClientConnState;  // <- used in ClientThread(); read-only for anyone else
#   define CWNET_CONN_STATE_OFF         0
#   define CWNET_CONN_STATE_TRY_CONNECT 1
#   define CWNET_CONN_STATE_CONNECTED   2
#   define CWNET_CONN_STATE_WAIT_RECONN 3 // "waiting some time BEFORE switching back to CWNET_CLIENT_STATE_TRY_CONNECT"
  T_TIM_Stopwatch swClientReconnTimer;
#   define CWNET_RECONN_INTERVAL_MS 5000

  T_CwDSP *pCwDSP; // Reference to a DSP instance, used for audio streaming.
  T_RigCtrlInstance RigControl; // so far, ONE "RigControl" instance is tied
                                //      to ONE "CW Network Server" instance.
   // Note: If the application is built with SWI_USE_HAMLIB_SERVER=1,
   //       the address of a "Hamlib Net rigctld"-compatible server instance
   //       is in RigControl.pvHamlibServer !


  long i32AudioSamplesSent_k, i32AudioSamplesRcvd_k; // audio sample counters, in 1000 sample units, for diagnostics
          // (helps troubleshooting when there is "no audio on the client side":
          //  There's a checkbox [ ] allow 'Network Audio' / bitflag DSP_AUDIO_FLAGS_ALLOW_NETWORK_AUDIO,
          //  polled somewhere in CwNet_OnPoll() [nomen est omen] .
  int  nAudioSamplesSent_Mod1k, nAudioSamplesRcvd_Mod1k; // lower part, counts individial samples
# if( SWI_USE_VORBIS_STREAM ) // allow streaming audio via Ogg/Vorbis ?
  T_VorbisStream sVorbis;  // wrapper for "libvorbis" and "libogg"
   // Note: There is only ONE instance of a T_VorbisStream to encode the
   //       audio signal for MULTPLE clients connected to our server.
   //       For that reason, the T_VorbisStream isn't part of the HTTP server's
   //       T_HttpInstance (in HttpServer.h) but part of T_CwNet .
  int iVorbisAudioTail; // one of the many "reader TAIL indices" for audio samples
                        // in pCwDSP->sInputFifo
# endif // SWI_USE_VORBIS_STREAM

  DWORD dwMagicPI_6;  // also should contain CWNET_MAGIC_PI, sanity check at THE END of the struct

} T_CwNet;  // <- in RCW-Keyer, there's ONE instance of this ('MyCwNet', also used by the 'Chatbox') !

typedef struct tCwNet_SpectrumHeader // <- sent as a BINARY OBJECT via network,
{ // thus only use simple types here that can easily be 'decoded' even in
  // Javascript ... just in case we use the same type in a "Websocket" one day.
  // MDN (developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/DataView)
  // mentioned a "DataView.prototype.getFloat32()", and if we're lucky, this
  // also exists in other browsers without having to pull in brutally large JS libraries.
  // Short form: Use 32 bit floating points; and remember, our byte order
  //             would be "littleEndian" for the Javascript .
  unsigned short u16NumBinsUsed; // number of frequency bins that follow after the T_CwNet_SpectrumHeader
  unsigned short u16Reserved; // reserved for future use / 32-bit alignment of the following:
  float flt32BinWidth_Hz; // 'width' of a single frequency bin in Hertz
  float flt32FreqMin_Hz;  // "radio frequency" stored in the first frequency bin
} T_CwNet_SpectrumHeader;

typedef struct t_CwNet_ConnectData // <- sent as a BINARY OBJECT via network, so use SIMPLE DATA TYPES only !
{ char sz40UserName[44];
  char sz40Callsign[44];
  DWORD dwPermissions;  // .. granted from SERVER to CLIENT only !
} T_CwNet_ConnectData;

// Function pointer type used in CwNet_Rigctld_Commands[] :
typedef int(*CwNetCommandHandler)(T_CwNet *pCwNet, T_CwNetClient *pClient,
               struct t_CwNet_CommandTableEntry *pCmdInfo, const char *pszCmdArgs,
               char *pszResp, int iMaxRespLength );
  // BCB V12 (but not BCB V6) complained about the above line:
  // > [bcc32c Warning] : declaration of 'struct t_CwNet_CommandTableEntry'
  // > will not be visible outside of this function
  // (thanks for NOT showing a 3- or 4-digit warning code so we could temporarily
  //  turn this stupidity off. It's the classic chicken-and-egg problem:
  //  We need the CwNetCommandHandler in struct t_CwNet_CommandTableEntry,
  //      and the struct t_CwNet_CommandTableEntry in CwNetCommandHandler.)

typedef struct t_CwNet_CommandTableEntry  // <- borrowed from DL4YHF's HamlibServer.c ..
{ char cShortToken;    // e.g. "F",        "f",  ...
  const char *pszLongToken;  // e.g. "set_freq", "get_freq", ... . NULL marks the end of the table.
                       // An EMPTY STRING means "only the single-byte form" (cShortToken) exists.
  int  iAccess;        // HLSRV_ACCESS_GET or HLSRV_ACCESS_SET ..
#      define HLSRV_ACCESS_GET 0
#      define HLSRV_ACCESS_SET 1
  CwNetCommandHandler pHandler;
} T_CwNet_CmdTabEntry; // -> CwNet_Rigctld_Commands[] (table with all commands; with short and long tokens)


//---------------------------------------------------------------------------
// Global variables  (?)
//---------------------------------------------------------------------------

  // (none so far. A single T_CwNet instance exists in Keyer_Main.cpp)


//---------------------------------------------------------------------------
// Function Prototypes
//---------------------------------------------------------------------------

#ifndef CPROT   // for peaceful co-existence of C and C++ ...
#ifdef __cplusplus
 #define CPROT extern "C"
#else
 #define CPROT
#endif  // not "cplusplus" ?
#endif // ndef CPROT ?


CPROT void  CwNet_InitInstanceWithDefaults( T_CwNet *pCwNet );
CPROT int   CwNet_SanityCheck( T_CwNet *pCwNet ); // <- for built-in hardcore debugging
CPROT BOOL  CwNet_Start( T_CwNet *pCwNet );
CPROT void  CwNet_Stop( T_CwNet *pCwNet );
CPROT char* CwNet_GetCurrentStatusAsString( T_CwNet *pCwNet );
CPROT char* CwNet_ClientStateToString( int iClientState );
CPROT char* CwNet_GetLastErrorAsString( T_CwNet *pCwNet );
CPROT char* CwNet_IPv4AddressToString( BYTE *pbIP ); // -> "dotted decimal IP address"
CPROT int   CwNet_CheckUserAndGetPermissions( T_CwNet *pCwNet, char *pszUserName, char *pszCallsign );
CPROT int   CwNet_GetUserPermissionsByRecentIP(T_CwNet *pCwNet, BYTE *pbIPv4Address );
CPROT void  CwNet_RefreshUserPermissionsByIP(T_CwNet *pCwNet,BYTE *pbIPv4Address,int iPermissions);

CPROT int   CwNet_GetFreeSpaceInTxBuffer( T_CwNet *pCwNet );
CPROT BYTE* CwNet_AllocBytesInTxBuffer( T_CwNet *pCwNet, BYTE bCommand, int iPayloadSize);
CPROT BOOL  CwNet_SwitchTransmittingClient( T_CwNet *pCwNet, int iNewTransmittingClient );
CPROT int   CwNet_GetIndexOfCurrentlyActiveClient( T_CwNet *pCwNet );
CPROT int   CwNet_GetLatencyForRemoteClient_ms( T_CwNet *pCwNet, int iTransmittingClient );
CPROT char *CwNet_GetClientCallOrInfo( T_CwNet *pCwNet, int iClientIndex ); // .. for the GUI

CPROT int   CwNet_CheckCommandType( const char *pszCmd );
   // '-- Return values:
#  define CWNET_CMDTYPE_NONE 0 // "not recognized as a rigctl[d] / hamlib command
#  define CWNET_CMDTYPE_RIGCTRL_GET 1 // it's a rigctl-like "get" command
#  define CWNET_CMDTYPE_RIGCTRL_SET 2 // it's a rigctl-like "set" command

CPROT int   CwNet_ExecuteRigctldCmd( T_CwNet *pCwNet, T_CwNetClient *pClient, const char *pszCmd, char *pszResp, int iMaxRespLength );

CPROT BOOL  CwNet_WriteStringToChatTxFifo( T_CwNet *pCwNet, T_CwNetClient *pClient, const char *pszTxData );
CPROT int   CwNet_ReadStringFromChatRxFifo( T_CwNet *pCwNet, T_CwNetClient *pClient, char *pszRxData, int iMaxLen );


#endif // ndef _CW_NET_H




