//---------------------------------------------------------------------------
// File: C:\cbproj\Remote_CW_Keyer\CwKeyer.h
// Date: 2023-11-11
// Author: Wolfgang Buescher (DL4YHF)
// Purpose: GUI-independent part of the 'Remote CW Keyer' experiment.
//          Details about the "keying" itself in KeyerThread.c .
//          In contrast to the IMPLEMENTATION (which is doomed to use a lot
//          of Windows-specific garbage), try to keep this header file CLEAN.
//    Especially, DO NOT use any Windows-/Linux-/Borland-/Microsoft-/Qt-specific
//    stuff in the INTERFACE !
//---------------------------------------------------------------------------

#ifndef  CIRCULAR_FIFO_INCLUDED
# include "CircularFifo.h"
#endif
#ifndef  STRAIGHT_KEY_DECODER_H_INCLUDED
# include "StraightKeyDecoder.h"
#endif

#ifndef  _DATA_TYPES_H_  // prevent multiple header inclusion ..
# include "yhf_type.h"   // T_RGBColor (used in struct T_KeyerTimingScope)
#endif


//---------------------------------------------------------------------------
// Constants
//---------------------------------------------------------------------------



//---------------------------------------------------------------------------
// Data types and variables
//---------------------------------------------------------------------------

typedef struct t_AuxComConfig // configuration for A SINGLE 'Additional COM Port' (we have <SWI_NUM_AUX_COM_PORTS> of them)
{

  int  iPortNumber;   // "COM port number"; negative = 'don't try to open'
  int  iBitsPerSecond; // serial "baudrate" in bits per second, e.g. 1200 for Winkeyer, 115200 for Icom radios
  int  iPortUsage;    // function assigned to the "Auxiliary COM Port #1":
        // contains one of the AUX_COM_PORT_USAGE_ constants defined in AuxComPorts.h .
  int  iTunnelIndex; // "Serial Port Tunnel Index" for routing between ports.
                 // In T_AuxComPortInstance, this is .iSerialTunnelIndex .
                 // ZERO-BASED, but iTunnelIndex = -1 means
                 // "THIS additional COM port is connected to ANY OTHER
                 // additional COM port with Usage=RCW_COM_PORT_USAGE_CLIENT_SERVER_TUNNEL .
  int  iNumDatabits; // any decent UART should allow 5 to 9 bits. We almost exclusively use EIGHT.
  int  iParity;  // may be one of the following:
#      define SER_PORT_PARITY_NONE  0 // None (N) means that no parity bit is sent and the transmission is shortened.
#      define SER_PORT_PARITY_ODD   1 // Odd (O) means that the parity bit is set so that the number of 1 bits is odd.
#      define SER_PORT_PARITY_EVEN  2 // Even (E) means that the parity bit is set so that the number of 1 bits is even.
#      define SER_PORT_PARITY_MARK  3 // Mark (M) parity means that the parity bit is always set to the mark signal condition (1 bit value).
#      define SER_PORT_PARITY_SPACE 4 // Space (S) parity always sends the parity bit in the space signal condition (0 bit value).
  int  iNumStopbits; // may be one of the following:
#      define SER_PORT_STOPBITS_1   1 // almost exclusively used: ONE stopbit (but some old amateur radio rigs don't work properly with this)
#      define SER_PORT_STOPBITS_2   2 // rare but an FT-817 need THIS for "Yeasu's old CAT Control with 5-byte-blocks" !
#      define SER_PORT_STOPBITS_1_5 3 // one-and-a-half stopbit .. very rare

  char sz40DTR[44]; // DE-9 pin 4: freely programmable; periodically evaluated to drive "DTR". Leave empty for the default setting.
  char sz40RTS[44]; // DE-9 pin 7: freely programmable; periodically evaluated to drive "RTS". Leave empty for the default setting.
  char sz40DCD[44]; // DE-9 pin 1: freely programmable; application-specific use of "DCD" (inputs). Leave empty for the default setting.
  char sz40DSR[44]; // DE-9 pin 6: freely programmable; application-specific use of "DSR" (inputs). Leave empty for the default setting.
  char sz40CTS[44]; // DE-9 pin 8: freely programmable; application-specific use of "CTS" (inputs). Leave empty for the default setting.
  char sz40RI [44]; // DE-9 pin 9: freely programmable; application-specific use of "RI" (inputs). Leave empty for the default setting.
  char sz40Params[44]; // e.g. hint to decode received traffic, CAT-protocol-specific parameters, etc

} T_AuxComConfig; // -> CwKeyer_Config.sAuxCom[SWI_NUM_AUX_COM_PORTS] (global variable, no fancy C++ thing)

typedef struct t_CwKeyerConfig
{
  int iComPortNumber_IN;  // serial port for the MORSE KEY (Modem status lines abused as digital INPUTS)
  int iRadioKeyingAndControlPort; // serial port to key the RADIO  (Modem control lines abused as digital OUTPUTS)
                                  // PLUS Radio CONTROL Port using CI-V (optional, for Icom radios)
  int iRadioControlProtocol; // RIGCTRL_PROTOCOL_NONE / RIGCTRL_PROTOCOL_ICOM_CI_V / RIGCTRL_PROTOCOL_HAMLIB_RIGCTLD (via TCP/IP)
  int iRadioControlBaudrate; // usually 115200 [bits/second] to use Spectrum data from IC-7300, IC-9700 & Co
  int iRadioCIVAddress;      // e.g. 0xA2 for IC-9700, 0x94 for IC-7300, 0x9A for IC-7610,
      //      0xA4 for IC-705,  0xAC for IC-905,
      //      0x47 for IC-706,  0x4E for IC-706MkII, 0x58 for IC-706MkII-G,
      //      etc etc (consult the radio's manual)
      // iRadioCIVAddress = -1 (RIGCTRL_DEF_ADDR_NO_REMOTE_CTRL) : "Rig Control OFF";
      // iRadioCIVAddress = 0  (RIGCTRL_DEF_ADDR_AUTO_DETECT)    : "try to detect the CiV-address AUTOMATICALLY"

  int iMorseKeyType; // may contain one of the following:
#   define KEY_TYPE_PASSIVE  0 // no keying of the output(s), but e.g. "manual" control via CwKeyer_SetDigitalOutput()
#   define KEY_TYPE_STRAIGHT 1 // straight key for INPUT (only one digital input)
#   define KEY_TYPE_IAMBIC_B 2 // external paddle, "Iambic mode B" emulated by software
#   define KEY_TYPE_IAMBIC_A 3 // external paddle, "Iambic mode A" emulated by software
#   define KEY_TYPE_IAMBIC_NO_MEM 4 // "Basic Iambic Keyer without dot/dash memory" as described by DJ5IL
           // Note: Compatible definitions are also in Elbug.h,
           //       but none of those header files shall depend on the other.

  int  iDotInput;   // one of the KEYER_SIGNAL_INDEX_ constants, positive=non-inverted, negative=inverted, zero=none.
#  define KEYER_SIGNAL_INDEX_NONE 0
#  define KEYER_SIGNAL_INDEX_MORSE_KEY_DTR     1  // digital OUTPUT on the "COM port for Morse key"
#  define KEYER_SIGNAL_INDEX_MORSE_KEY_RTS     2  // digital OUTPUT on the "COM port for Morse key"
#  define KEYER_SIGNAL_INDEX_MORSE_KEY_DCD     3  // digital INPUT  on the "COM port for Morse key"
#  define KEYER_SIGNAL_INDEX_MORSE_KEY_DSR     4  // digital INPUT  on the "COM port for Morse key"
#  define KEYER_SIGNAL_INDEX_MORSE_KEY_CTS     5  // digital INPUT  on the "COM port for Morse key"
#  define KEYER_SIGNAL_INDEX_MORSE_KEY_RI      6  // digital INPUT  on the "COM port for Morse key"
#  define KEYER_SIGNAL_INDEX_RADIO_KEYING_DTR  8  // digital OUTPUT on the "COM port to 'key' the radio"
#  define KEYER_SIGNAL_INDEX_RADIO_KEYING_RTS  9  // digital OUTPUT on the "COM port to 'key' the radio"
#  define KEYER_SIGNAL_INDEX_RADIO_KEYING_DCD 10  // digital INPUT  on the "COM port to 'key' the radio"
#  define KEYER_SIGNAL_INDEX_RADIO_KEYING_DSR 11  // digital INPUT  on the "COM port to 'key' the radio"
#  define KEYER_SIGNAL_INDEX_RADIO_KEYING_CTS 12  // digital INPUT  on the "COM port to 'key' the radio"
#  define KEYER_SIGNAL_INDEX_RADIO_KEYING_RI  13  // digital INPUT  on the "COM port to 'key' the radio"
#  define KEYER_SIGNAL_INDEX_DASH_INPUT       14  // combined 'dash' input, possibly ORed with the state of the PC keyboard's SHIFT key
#  define KEYER_SIGNAL_INDEX_DOT_INPUT        15  // combined 'dash' input, possibly ORed with the state of the PC keyboard's CONTROL key
   // Bits numbers 16 and above only apply to Keyer_dwCurrentSignalStates .. see notes below !
#  define KEYER_SIGNAL_INDEX_CW  16 // bit Keyer_dwCurrentSignalStates, with the current state of the "CW" output,
                                    //  driven by elbug, CW generator, or whatever runs in KeyerThread() .
                                    //  In contast to the TRANSMITTER KEYING SIGNAL (
                                    //  Keyer_dwCurrentSignalStates.KEYER_SIGNAL_INDEX_CW
                                    //  (like the SIDETONE output, because that's the intention
                                    //   of being able to route KEYER_SIGNAL_INDEX_CW as token "CW"
                                    //   to any of the digital outputs (DTR or RTS).
#  define KEYER_SIGNAL_INDEX_PTT 17 // bit Keyer_dwCurrentSignalStates, with the current state of the "PTT" output, also driven ONLY inside KeyerThread()
   // Notes:
   //  * When adding new input- or output signals to the above list,
   //       also add support for them in ...
   //         * KeyerThread.c : CwKeyer_IsInputSignalAvailable() [switch-case]
   //
   //  * The above "signal indices" are also BIT NUMBERS
   //       for the Keyer_dwCurrentSignalStates, frequently updated in KeyerThread() .
   //       The 'Remote CW Keyer' test application / GUI uses them
   //       to indicate the CURRENT STATES of some of these signals
   //       on the 'Test' tab. The 'Additional COM Port' worker threads
   //       also may READ all those flags, and additionally WRITE some of those
   //       flags (with bit numbers 16 and above).
   //
   //  * Those flags with bit numbers 16 and above have a FIXED MEANING for the
   //       PLC-like "combinatorial logic" for ADDITIONAL COM PORTS.
   //       Note: Any instance of AuxComThread() MUST NEVER WRITE
   //       into Keyer_dwCurrentSignalStates ! The "PLC"(s) in those threads
   //       only WRITE into their own process images, which is
   //           AuxComPorts[0..SWI_NUM_AUX_COM_PORTS-1].dwDigitalSignalStates,
   //       with a BITWISE COMBINATION of flags like
   //           AUX_COM_DIG_SIGNAL_DOT / DASH / SKEY / PTT .
   //
   //  * What Microsoft calls "RLSD (Receive-Line-Signal-Detect) signal"
   //       is an 'alternative name' for what the rest of the world calls
   //       "DCD" (Data Carrier Detect), or just "CD" (Carrier Detect).
   //       We don't follow this madness in *our* API, and use the name "DCD".
   //
   //  * The default assigments between RS-232 / "Modem" SIGNALS and their
   //       configurable FUNCTIONS is preset in CwKeyer_SetDefaultConfig(),
   //       but all these assignments are flexible, and can be configured
   //       in the GUI (at least in the author's VCL-based Keyer_Main.cpp).
   //
  int iDashInput;  // one of the KEYER_SIGNAL_INDEX_ constants, positive=non-inverted, negative=inverted, zero=none.
  int iTestInput;  // similar as above, selectable as source for "scope channel 4". No fixed funtion here.
  int iManualPTTInput; // rarely used feature for manual PTT control, "footswitch", or similar.
                   // Usually iManualPTTInput = KEYER_SIGNAL_INDEX_NONE for 'automatic PTT control'.
  int iKeySupply;  // one of the KEYER_SIGNAL_INDEX_ constants, sign=polarity..
  int iRadioSupply; // similar as above, but to provide power for old CI-V interfaces, or to turn the RADIO POWER SUPPLY ON.
  int iSidetoneOnTXD; // optional sidetone ouput on the MORSE KEY PORT / TXD (iComPortNumber_IN):
#     define KEYER_SIDETONE_NONE 0 // .. see implementation details in KeyerThread.c
#     define KEYER_SIDETONE_TXD_480_HZ  480 // 4800 "baud" divided by 10 bits (incl. start and stop)
#     define KEYER_SIDETONE_TXD_960_HZ  960 // also 4800 "baud" but TWO PULSES in the data pattern
  int iTxDelayTime_ms;
  int iTxHangTime_ms;
  BOOL fDebouncePaddleInputs;  // TRUE : Debounce paddle inputs
  BOOL fKeyInputWatchdog;      // TRUE : Stop keying if any input from the Morse key (dash,dot,straight)
                               //        appears 'stuck in the ACTIVE state' for 10 (?) seconds
  BOOL fKeyViaShiftAndControl; // TRUE : Poll the PC keyboard's SHIFT- and CONTROL keys
                   //   as replacement for a real Morse key or paddle .
                   //   (this gadged may consume too much CPU time
                   //    when polled 500 times per second .. )
                   // FALSE: Off, "no interference from the SHIFT- and CONTROL keys"
  BOOL fDisableTx; // FALSE : Normal operation, allow real transmission, with PTT control and Morse keying signal sent to the radio
                   // TRUE  : Only allow CW output as 'sidetone', for testing/practicing/speed adjustment w/o causing QRM
  DWORD dwAudioOptions; // bitwise combination of the following:
# define KEYER_AUDIO_OPTION_SIDETONE_WHEN_KEYED_REMOTELY (1<<0)
# define KEYER_AUDIO_OPTION_SIDETONE_WHEN_KEYED_ON_RIG   (1<<1)


# if( SWI_NUM_AUX_COM_PORTS > 0 ) // Who knows how many of these we may use one day ? So use an ARRAY right from the start:
  T_AuxComConfig sAuxCom[SWI_NUM_AUX_COM_PORTS];
# endif // SWI_NUM_AUX_COM_PORTS > 0 ?

# define KEYER_NUM_MESSAGE_MEMORIES 6
# define KEYER_MAX_CHARS_PER_MEMORY 1023  // fixed array sizes as in a microcontroller firmware .. YESS ! KISS ! :)
  char szKeyerMemory[KEYER_NUM_MESSAGE_MEMORIES][KEYER_MAX_CHARS_PER_MEMORY+1];
  char sz15MyCall[16];  // text inserted instead of <mycall> in szKeyerMemory[]

  int iRadioCWKeying;   // one of the KEYER_SIGNAL_INDEX_ constants, sign=polarity..
  int iRadioPTTControl; // one of the KEYER_SIGNAL_INDEX_ constants, sign=polarity..

  char sz80RemoteRigCtrlServerAddress[84]; // e.g. 127.0.0.1:4532 or just "4532" for "rigctld",
                    // or "12345" for the incompatible "flrig" with "xmlrpc".
                    // The numeric IP (before the colon) is only required if the
                    // server is "really remote" (running on a different PC).
  BOOL fEnableBuiltInHamlibServer;

  T_CwNet *pCwNet;  // address of a 'T_CwNet' instance for real remote operation (not just USB or COM port)

  // Only *a few* properties of the user interface (Borland VCL: "main form")
  // shall be saved between settings. To have them included in the 32-bit CRC
  // calculated from CwKeyer_Config, make them part of the T_CwKeyerConfig:
  int iMainWindowLeft, iMainWindowTop, iMainWindowWidth, iMainWindowHeight;
  int iTRXOptions; // bitwise combineable options for the 'TRX' control tab:
#     define KEYER_TRX_OPT_DEFAULT      0
#     define KEYER_TRX_OPT_KEEP_RUNNING 0x0001 // keep e.g. the spectrum running even if the tab isn't visible
           // (costs a few percents of the available CPU time, thus OPTIONAL)
  BOOL  fTurnRigOffWhenClosing;
  DWORD dwMessageFilterForLog; // later in RigControl.h : T_RigCtrlInstance,
           // but THIS ONE (in CwKeyer_Config) is loaded from the config file
           // long before RigControl_Init() is called .

} T_CwKeyerConfig; // -> CwKeyer_Config (global variable, no fancy C++ thing)


extern T_CwKeyerConfig CwKeyer_Config; // configuration for a single instance
extern T_ElbugInstance CwKeyer_Elbug;  // 'Elbug' instance, with its own configuration
extern T_StraightKeyDecoder CwKeyer_StraightKeyDecoder; // Morse decoder for input from a STRAIGHT KEY
extern T_CwGen         CwKeyer_Gen;    // 'CW Generator' instance
extern T_CwDSP         CwKeyer_DSP;    // 'CW Digital Signal Processor', initially just a sidetone generator
extern T_CwNet         CwKeyer_ClientOrServer; // simple network client or server, see CwNet.c

extern BOOL CwKeyer_fPauseTransmitter; // global "pause transmit"-flag, set on 'operator panic'
extern char CwKeyer_sz255LastError[256];

extern DWORD CwKeyer_dw8ThreadIntervals_us[8]; // microseconds per thread loop (for testing, displayed in the experimental GUI)
extern int   CwKeyer_iSpeedTestPeaks_us[8]; // microseconds spent in each thread loop for the following operations:
#define KEYER_SPEEDTEST_POLL_INPUTS 0 // time spent in KeyerThread() loop to poll MODEM STATUS LINES (as 'digital inputs')
#define KEYER_SPEEDTEST_SET_OUTPUTS 1 // time spent in KeyerThread() loop to drive MODEM CONTROL LINES (as 'digital outputs')
#define KEYER_SPEEDTEST_POLL_KEYBRD 2 // time spent in each KeyerThread() loop to poll SHIFT and CONTROL on the PC's keyboard

extern int CwKeyer_iThreadStatus; // may contain one of the following:
#   define KEYER_THREAD_STATUS_NOT_CREATED 0
#   define KEYER_THREAD_STATUS_LAUNCHED    1
#   define KEYER_THREAD_STATUS_RUNNING     2 // <- only set ONCE(!), immediately when entering the thread function
#   define KEYER_THREAD_STATUS_TERMINATE   3 // <- just a KLUDGE to "request POLITE termination", but takes a few dozen milliseconds to actually TERMINATE !
#   define KEYER_THREAD_STATUS_TERMINATED -1 // <- set by the worker thread if it TERMINATED ITSELF

extern DWORD Keyer_dwCurrentSignalStates; // bitwise combination of 1 << KEYER_SIGNAL_INDEX_xyz .
       // Exposed as a global variable since 2025-06 to allow the "programmable logic" in AuxComThread()
       // to READ (but not MODIFY) the above flags in Keyer_dwCurrentSignalStates .

extern HANDLE Keyer_hComPort_IN; // goddamned 'handle' to the serial port for INPUT from the Morse key.
                                 // May be INVALID_HANDLE_VALUE (which is NOT ZERO) thanks to the stupid Windows API.
extern HANDLE Keyer_hComPortRadioKeyingAndControl; // goddamned 'handle' to the "output" COM port drives the transceiver's
       // CW-input (via DTR), and maybe the PTT ("Push-To-Transmit", "Request-To-Send", via RTS),
       // or even CONTROLS the radio via CI-V.
       // Also here: It's a stupid Windows thing, here an invalid 'HANDLE'
       //            isn't zero or NULL but INVALID_HANDLE_VALUE (!) .


extern int CwKeyer_iTestMode; // KEYER_TEST_MODE_OFF/.._MANUAL/.. :
#   define KEYER_TEST_MODE_OFF    0 // "normal operation" (keyer controls digital outputs)
#   define KEYER_TEST_MODE_MANUAL 1 // "manual control" of the keyer's digital outputs,
                                    // the APPLICATION may invoke CwKeyer_SetDigitalOutput().
extern BOOL CwKeyer_fLocalMorseOutput; // kludge to plot the Elbug-output even when NOT driving the CW modulation output.
             // '--> input for CwKeyer_CollectDataForTimingScope(), when using MANUAL PTT CONTROL but PTT currently off.
extern T_TIM_Stopwatch CwKeyer_swMorseActivityTimer; // timer related to CwKeyer_fLocalMorseOutput,
             // '--> used to avoid flickering display in the GUI - see STATUS_INDICATOR_CW_TEST.

extern BOOL Keyer_fDotInputWasPassive, Keyer_fDashInputWasPassive; // flags to prevent ENDLESS keying:
             // TRUE when ok (saw a PASSIVE input in the previous FIVE SECONDS)

#define TIMING_SCOPE_N_DIGITAL_CHANNELS 4
#define TIMING_SCOPE_FIRST_DIGITAL_CHANNEL 0
#define TIMING_SCOPE_CHANNEL_DASH_INPUT   0 // bit 0 in bSamplePoints[] (and iVisibleChannels) ..
#define TIMING_SCOPE_CHANNEL_DOT_INPUT    1
#define TIMING_SCOPE_CHANNEL_CW_OUTPUT    2
#define TIMING_SCOPE_CHANNEL_FOUR         3 // .. bit 3 in bSamplePoints[] .
        // Unlike the first 3 digital channels, "Channel Four" can be connected to
        //  a selectable source, see CwKeyer_TimingScope.iChannel4Source .
#define TIMING_SCOPE_FIRST_ANALOG_CHANNEL 4
#define TIMING_SCOPE_CHANNEL_AUDIO_INPUT  4
#define TIMING_SCOPE_CHANNEL_AUDIO_OUTPUT 5
#define TIMING_SCOPE_CHANNEL_AUDIO_CW_DEC 6  // "filtered and rectified input" for the AUDIO CW DECODER, treated like an analog input, with an extra FIFO
#define TIMING_SCOPE_N_ANALOG_CHANNELS  3

typedef struct tTimingScopeSample   // struct for a SINGLE SAMPLE POINT (with multiple channels) for the keyer's "timing scope"
{ int iChannel[TIMING_SCOPE_N_DIGITAL_CHANNELS]; // four 'integer' or 'digital' channels
                     // (initially only used for the scope's "channel 4"
                     // to plot internal parameters like e.g. the KEYING FIFO USAGE;
                     // but also used for the THREE FIXED SCOPE CHANNELS :
                     //  [0] = TIMING_SCOPE_CHANNEL_DASH_INPUT,
                     //  [1] = TIMING_SCOPE_CHANNEL_DOT_INPUT,
                     //  [2] = TIMING_SCOPE_CHANNEL_CW_OUTPUT,
                     //  [3] = TIMING_SCOPE_CHANNEL_FOUR .
                     // Values scaled into PERCENT (0..100) for simplicity.
  WORD wCwChar;  // 'Morse code pattern', also visible in the timing scope,
                 // emitted at the time of <T_KeyerTimingScope.iSampleIndex> .

  T_CwDSP_PlotterSample sDspPlotterSample; // <- short-time audio spectrum and a few 'readings from the audio CW decoder
                 // Because the audio spectra arrive much later than the other
                 // members in a T_TimingScopeSample, they are not filled in
                 // in the keyer thread (CwKeyer_CollectDataForTimingScope)
                 // but in the main thread (Keyer_GUI.cpp : KeyerGUI_UpdateTimingScope).
# if( TIMING_SCOPE_N_ANALOG_CHANNELS > 0 )
  float fltAnalog[ TIMING_SCOPE_N_ANALOG_CHANNELS ]; // <- floating point samples for a few ANALOG channels.
# endif // TIMING_SCOPE_N_ANALOG_CHANNELS > 0 ?
} T_TimingScopeSample; // -> used in T_KeyerTimingScope.sSamplePoints[TIMING_SCOPE_NUM_SAMPLE_POINTS]
                       //        and T_KeyerTimingScope.sPreTriggerPoints[TIMING_SCOPE_PRETRIGGER_POINTS] .


#define TIMING_SCOPE_NUM_SAMPLE_POINTS 1024
#define TIMING_SCOPE_PRETRIGGER_POINTS 8
typedef struct tKeyerTimingScope // "instance data" for a very primitive scope display..
{
  struct { // .cfg = "everything that needs to be SAVED between sessions" (and included in a hash value):
     // Since the addition of 'audio I/O' and ANALOG channels for the scope,
     //  *all* channels can be selectively turned on and off (via popup menu).
     //  The following struct member is a bitwise combination;
     //  each of the "zero-based" channels (TIMING_SCOPE_CHANNEL_xyz) has its own bit:
     int iVisibleChannels; // bitwise combination of e.g. (1 << TIMING_SCOPE_CHANNEL_DASH_INPUT)
     int iMillisecondsPerSample; // "wanted" sampling interval for the display in the timing scope, originally fixed to 2 [ms per sample]
     // Note: One horizontal PIXEL is NOT necessarily ONE SAMPLE of the timing scope !
     // CwKeyer_TimingScope.iMillisecondsPerSample applies to the samples
     // in arrays with <TIMING_SCOPE_NUM_SAMPLE_POINTS> - *not* to the x-coordinate.


     int iTriggerOptions;  // bitwise combination of the following:
#    define TIMING_SCOPE_TRIGGER_NORMAL   0 // new sweep on any transition on any DIGITAL input
#    define TIMING_SCOPE_TRIGGER_FREE_RUN 1 // begin new sweep immediately after finishing the previous


     // The FOURTH scope channel shows the state of the PTT output per
     // default, but it may be connected to a few other sources, e.g. to the
     // 'Test input'. The 'Test input' may be one of the few unused digital
     // inputs on a serial port (Modem Status lines), selectable in one of the
     // combo boxes in the GUI.
     int iChannel4Source; // "Source for channel FOUR": One of the following few values:
#    define TIMING_SCOPE_CHANNEL_SOURCE_PTT_OUTPUT        0
#    define TIMING_SCOPE_CHANNEL_SOURCE_TEST_INPUT        1
#    define TIMING_SCOPE_CHANNEL_SOURCE_NETWORK_TROUBLE   2
#    define TIMING_SCOPE_CHANNEL_SOURCE_KEYING_FIFO_USAGE 3
#    define TIMING_SCOPE_CHANNEL_SOURCE_ELBUG_KEYER_FLAGS 4

     int iChannelVerticalPos_pcnt[TIMING_SCOPE_N_DIGITAL_CHANNELS + TIMING_SCOPE_N_ANALOG_CHANNELS];
     T_RGBColor dwChannelColours[TIMING_SCOPE_N_DIGITAL_CHANNELS + TIMING_SCOPE_N_ANALOG_CHANNELS];

     int iAudioSpectrumRefLevel_dB;  // .. e.g. for the display in the "timing scope" (!)
     int iAudioSpectrumAmplRange_dB;

   } cfg; // <- end of the part that needs to be stored permanently (between sessions)

  BOOL fPaused;  // <- not related to "trigger" / "free running" mode, but allows MANUALLY pausing the display by the operator

  T_TimingScopeSample sSamplePoints[TIMING_SCOPE_NUM_SAMPLE_POINTS]; // "digital" and "analog" in- and outputs
  double dblSampleCollectingTimestamp_s; // used in KeyerThread.c : CwKeyer_CollectDataForTimingScope()
  int  iAnalogSourceFifoTail[TIMING_SCOPE_N_ANALOG_CHANNELS];
  int  iAnalogSourceSampleTimeAccu_ms[TIMING_SCOPE_N_ANALOG_CHANNELS];

  int iLatestSampleIndex; // 0...<TIMING_SCOPE_NUM_SAMPLE_POINTS-1>. Incremented in CwKeyer_CollectDataForTimingScope().
      // iLatestSampleIndex also controls the position of the "windscreen wiper",
      //  indicates "waiting for trigger" - see CwKeyer_CollectDataForTimingScope() .
  int iSampleIndexFromDSP; // similar like iLatestSampleIndex, but used
      // by ScopeDisplay_AppendDspPlotterSamples() to merge samples for plotting
      // from module CwDSP.c into sSamplePoints[ iSampleIndexFromDSP++ ].
  double dblTimestampAtSampleIndexZero; // used in to avoid 'jitter' in KeyerGUI_UpdateTimingScope() when retrieving channel data via TIMESTAMP
  double dblTimestampAtPretriggerIndexZero;
  T_TimingScopeSample sPreTriggerPoints[TIMING_SCOPE_PRETRIGGER_POINTS];
      // ,----------------'
      // '--> [0] = oldest pre-trigger sample,
      //      [TIMING_SCOPE_PRETRIGGER_POINTS-1] = newest pre-trigger sample
  int iUpdateCount; // incremented whenever the data in bSamplePoints[] were modified
  // Note: The KEYER SPEED (CwKeyer_Elbug.iDotTime_ms) dictates the
  //       SAMPLING RATE for filling the above array.
  //       With TIMING_SCOPE_NUM_SAMPLE_POINTS = 1024,
  //       and TEN samples per "dot time", the timing scope
  //       will show approximately 1024 / 10 = 100 dot-times .
  // See also: Rendering of the "scope display" using Borland's VCL
  //           in KeyerGUI_UpdateTimingScope() .

  // Layout parameters: Updated and used in KeyerGUI_UpdateTimingScope(),
  //  also used in the MOUSE EVENT HANDLER, KeyerGUI_HandlerMouseEventInTimingScope():
  struct
   { int iVerticalPos_pixels; // vertical position in pixels; "tip of the arrow pointing right"
     int iCurveHeight_pixels; // <- nonzero for channels with (1<<iChannel) SET in cfg.iVisibleChannels
   } sChannelInfo[TIMING_SCOPE_N_DIGITAL_CHANNELS + TIMING_SCOPE_N_ANALOG_CHANNELS];



} T_KeyerTimingScope; // -> CwKeyer_TimingScope
extern T_KeyerTimingScope CwKeyer_TimingScope;


typedef struct tKeyerDecoderFifo // thread-safe, lock-free circular FIFO
{ // Filled by the CW-keyer-thread whenever Mr Elbug has 'decoded' whatever has
  // been sent to the radio (this is by no means a Morse decoder for RECEPTION).
  // Drained by the GUI, after dumping the decoded text into an edit control.
# define KEYER_DECODER_FIFO_SIZE 8
  WORD wCwChar[KEYER_DECODER_FIFO_SIZE]; // 'Morse code pattern' (not decoded or converted into a string yet), emitted at the time of <iSampleIndex> .

  int iFifoHead; // index into wCwChar[], runs from 0 to KEYER_DECODER_FIFO_SIZE-1,
       // then wraps around to zero, etc (classic lock-free circular FIFO) .
       // iFifoHead is only WRITTEN by the worker thread; read-only for anyone else.
  // ex: int iFifoTail; // tail index: only WRITTEN TO by the GUI (not by the worker thread).
          // iFifoHead == iFifoTail means EMPTY FIFO (for a certain reader, in this case the GUI)
          // 2023-12: struct component iFifoTail removed, because there may
          //          be MULTIPLE READERS for a particular FIFO instance,
          //          and each of them owns his own 'tail index' now.
} T_KeyerDecoderFifo;
extern T_KeyerDecoderFifo CwKeyer_DecoderFifo;

#define KEYER_SERIAL_PORT_FIFO_SIZE 1024
typedef struct
{ T_CFIFO fifo;
  BYTE bExtraBytes[ KEYER_SERIAL_PORT_FIFO_SIZE - SWI_CIRCULAR_FIFO_SIZE ];
} T_KeyerSerialPortFifo;
extern T_KeyerSerialPortFifo Keyer_RadioControlPortRxFifo, Keyer_RadioControlPortTxFifo;

#if( SWI_USE_DSOUND ) // use "DirectSound" / dsound_wrapper.c ?
extern DSoundWrapper CwKeyer_Dsound; // a "DirectSound wrapper" instance, first used for the SIDETONE OUTPUT
#endif // SWI_USE_DSOUND ?

//---------------------------------------------------------------------------
// Function prototypes  (details only in the implementation, *.c)
//---------------------------------------------------------------------------

#ifndef CPROT   // for peaceful co-existance of C and C++ ...
#ifdef __cplusplus
 #define CPROT extern "C"
#else
 #define CPROT
#endif  // not "cplusplus" ?
#endif // ndef CPROT ?

CPROT void CwKeyer_SetDefaultConfig( T_CwKeyerConfig *pConfig );
CPROT void CwKeyer_InitTimingScope( T_KeyerTimingScope *pScope );
CPROT BOOL CwKeyer_Start( void );
CPROT void CwKeyer_Stop( void );
CPROT void CwKeyer_StartReplay( char *pszTextToSend ); // .. from one of the GUI's "memories"

CPROT void  CwKeyer_SetDigitalOutput( int iSignalIndex, int iNewState );
CPROT BOOL  CwKeyer_GetDigitalInput( int iSignalIndex );
CPROT void  CwKeyer_InitDigitalInput(int iSignalIndex); // set to the "passive" state
extern BOOL CwKeyer_fUpdateAllOutputs; // kludge to 'send all outputs' even if they didn't change


CPROT void CwKeyer_WriteToDecoderFifo( T_KeyerDecoderFifo *pFifo, WORD wCwChar );
CPROT WORD CwKeyer_ReadFromDecoderFifo( T_KeyerDecoderFifo *pFifo, int *piTailIndex );

      // Functions for debugging / software testing :
CPROT int  CwKeyer_GetSpeedTestAverage_us(int iTestItem); // [in] e.g. KEYER_SPEEDTEST_POLL_KEYBOARD
CPROT void CwKeyer_ResetSpeedTestResults(void); // begins a new "peak detection"

