// File:  C:\cbproj\Remote_CW_Keyer\CwGen.h
// Date:  2023-12-06
// Author:  Wolfgang Buescher (DL4YHF)
// Purpose: State-machine-driven 'CW Generator' .
//          Generates Morse code from text (plain "C" strings).
//          Periodically called from the 'keyer thread' (or task,
//          or timer interrupt, or whatever on a microcontroller).
//          Depends on the 'Elbug' module, at least for the functions
//          to convert text ("ASCII") into the shift register values
//          with the dash- and dot pattern for each character.


//---------------------------------------------------------------------------
// Constants
//---------------------------------------------------------------------------


#ifndef TRUE
# define TRUE  1
# define FALSE 0
#endif

#ifndef CW_CHR_0  // Morse code patterns for common "CW characters" not defined yet ?
# include "Elbug.h" // .. should be defined in Elbug.h
#endif



//---------------------------------------------------------------------------
// Data Types
//---------------------------------------------------------------------------


typedef struct t_CwGen // everthing we need for a single 'CW generator' instance:
{

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // "configuration parameters" (must be filled out be the application before start)
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  struct { // .cfg = "everything that needs to be SAVED between sessions" (and included in a hash value):
     int iDotTime_ms;   // duration of a dot, measured in MILLISECONDS (!)
          // (use Elbug_WordsPerMinuteToDotTimeInMilliseconds() to convert..)
   } cfg; // <- end of the part that needs to be stored permanently (between sessions)


  // Input variables for the CW generator:
# define CW_GEN_TX_MEMORY_SIZE 1023
  char szTxMemory[CW_GEN_TX_MEMORY_SIZE+1];
  int  iTxMemoryCharIndex; // index into szTxMemory[]. Incremented WHILE playing,
       // as soon as a character has been 'consumed' by CwGen_PrepareSendingNextChar()
       //   (this doesn't mean the character has been SENT; see iTxMemoryNumCharsSent).
       // The Remote CW Keyer GUI monitors iTxMemoryCharIndex for the
       //    'progress indicator' in the status line / RichEdit control
       //    for direct input as text, like a type-ahead buffer .
       // In the combined "Info / RX data / TX data" edit field (RichEdit_RxTxInfo),
       // characters ALREADY SENT will have a slightly different background colour
       // than characters that hat NOT BEEN SENT YET. To allow appending a
       // prinicipally unlimited number of characters to the string (T_CwGen.szTxMemory[])
       // while already sending from it, ...
       //   * CwGen_StartReplay() will clear (zero-fill) the ENTIRE szTxMemory[]
       //       before copying the initial string of characters into it;
       //   * CwGen_AppendForReplay() will append new data,
       //       and -if iTxMemoryCharIndex approaches CW_GEN_TX_MEMORY_SIZE-
       //       "scroll" the already sent characters out of szTxMemory[].
       //       When that happens ("scolling" of the buffer to create space
       //       for the new characters), i32TxMemoryNumCharsRemoved will be
       //       incremented, and iTxMemoryCharIndex decremented, by the number
       //       of characters that 'fell out of the buffer' when scrolling.
       //   * Thus, the sum of iTxMemoryCharIndex + i32TxMemoryNumCharsRemoved
       //       is the number of characters sent since calling CwGen_StartReplay(),
       //       which may greatly exceed CW_GEN_TX_MEMORY_SIZE .
  long i32TxMemoryNumCharsRemoved; // see explanation above (add value to iTxMemoryCharIndex)
  int  nCharsRemainingToPlay; // Number of characters still waiting to be played in szTxMemory[]

  int iState;  // State for the CW generator state machine :
  // Former PIC keyer states for WAITING and switching into SLEEP mode :
# define CW_GEN_OFF               0 // CwGen_Handler() won't do anything
# define CW_GEN_START             1 // CwGen_Handler() will START if there's something to send
# define CW_GEN_SEND_DASH_OR_DOT  2 // sending a dash or dot
# define CW_GEN_SEND_GAP          3 // sending a gap with ONE to SEVEN dot lengths
# define CW_GEN_FINISHED_SENDING  4 // reached the end of text in szTxMemory[], "nothing more to send", application may decide WHAT ELSE to send.
# define CW_GEN_WAITING_FOR_MACRO 5 // waiting for more text fed into szTxMemory[] before a MACRO like "<sWPM>" can be parsed

  int iTempSpeedDotTime_ms; //  > 0 when currently controlled by macro <sNN> and <s>,
                            // else use cfg.iDotTime_ms .


  // Like Elbug_Handler() before, CwGen_Handler() doesn't call any functions
  // to "drive the digital output" for the transmitter's PTT and Morse key input.
  // Instead, it only sets the following flag, and lets the caller do the
  // actual 'output port switching', or whatever:
  BOOL fMorseOutput;  // "Morse output" driving the side tone, modulate the TX,
                      // and (possibly delayed by the RX/TX switching time)
                      // also the TRANSMITTER'S keying input.

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // "internal" stuff
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  long i32CountdownForDashOrDot_us; // "countdown timer", in microseconds, for sending dashes or dots
  WORD wShiftReg;  // latch, or kind of shift register, for the character currently being sent.
                   // The next dash(1) or dot(0) to send is always in BIT FIFTEEN.
  int  nDashesAndDotsInShiftReg;
  WORD wCurrentCwChr; // currently sent MORSE CODE pattern (emitted at the end
                      // of a character transmission for the timing scope,
                      // returned by CwGen_Handler() just like Elbug_Handler()

} T_CwGen;


//---------------------------------------------------------------------------
// Function prototypes
//---------------------------------------------------------------------------
#ifndef CPROT   // for peaceful co-existence of C and C++ ...
#ifdef __cplusplus
 #define CPROT extern "C"
#else
 #define CPROT
#endif  // not "cplusplus" ?
#endif // ndef CPROT ?


CPROT void CwGen_InitInstanceWithDefaults( T_CwGen *pCwGen );
CPROT int  CwGen_Handler( T_CwGen *pCwGen, long nMicrosecondsSinceLastCall );
  // Values returned by CwGen_Handler() : One of the CW_CHR-codes (More code pattern
  // emitted at the end of a single character or space, ranging from 1 to 255),
  // or one of the ELBUG_RESULT_ - bitflags defined in Elbug.h .

//---------------------------------------------------------------------------
CPROT void CwGen_StartReplay( T_CwGen *pCwGen, char* pszTextToSend );
  // Called from the GUI when the operator wants to send text
  //   from one of the "Memories".
  // [in] pszTextToSend : text from the to-be-played 'keyer memory'.
  // The 'real work' happens later, when the keyer thread (or task on a uC)
  // calls CwGen_Handler() every two milliseconds or so...

//---------------------------------------------------------------------------
CPROT int  CwGen_GetFreeSpaceInReplayBuffer( T_CwGen *pCwGen );
  // Called from the GUI before preparing "text" to pass via CwGen_AppendForReplay()

//---------------------------------------------------------------------------
CPROT void CwGen_AppendForReplay( T_CwGen *pCwGen, char* pszMoreTextToSend );

//---------------------------------------------------------------------------
CPROT long CwGen_GetNumCharsPlayedSinceStart( T_CwGen *pCwGen );

//---------------------------------------------------------------------------
CPROT void CwGen_ClearReplayBuffer( T_CwGen *pCwGen );

//---------------------------------------------------------------------------
CPROT void CwGen_StopReplay( T_CwGen *pCwGen );

//---------------------------------------------------------------------------
CPROT BOOL CwGen_IsTxBusy( T_CwGen *pCwGen ); // -> TRUE=yes=busy, FALSE=not busy

