// File:  C:\cbproj\Remote_CW_Keyer\CwGen.c
// Date:  2023-11-22
// Author:  Wolfgang Buescher (DL4YHF)
// Purpose: State-machine-driven 'CW Generator' .
//          Generates Morse code from text (plain "C" strings).
//          Periodically called from the 'keyer thread' (or task,
//          or timer interrupt, or whatever on a microcontroller).
//          Depends on the 'Elbug' module, at least for the functions
//          to convert text ("ASCII") into the shift register values
//          with the dash- and dot pattern for each character.
//

#include <string.h> // no old string functions used here, but memset()
#include "StringLib.h" // DL4YHF's string library, with stuff like SL_strnlen(), SL_strncpy()
#include "Elbug.h" // header for the basic 'Elbug' functions (plain C)
#include "CwGen.h" // header for THIS module (Morse code "generator")


//----------------------------------------------------------------------------
// "Internal" constants, macros, lookup tables, prototypes, and similar
//----------------------------------------------------------------------------

#ifndef  sizeof_member
# define sizeof_member(type,member) sizeof(((type*)0)->member) /* ugly.. but works */
#endif

BOOL CwGen_ParseMacro( T_CwGen *pCwGen, const char **cppSrc, const char *cpEndstop );

//----------------------------------------------------------------------------
void CwGen_InitInstanceWithDefaults( T_CwGen *pCwGen )
  // Called from the main application (or "GUI") before loading an actual
  // configuration FROM A FILE. The results (in *pConfig) are the "defaults".
{
  memset( pCwGen, 0, sizeof(T_CwGen) );
  pCwGen->cfg.iDotTime_ms = Elbug_WordsPerMinuteToDotTimeInMilliseconds( 25 );

} // end CwGen_InitInstanceWithDefaults()

//---------------------------------------------------------------------------
static void CwGen_ReloadCountdownForDashOrDot( T_CwGen *pCwGen, int nDotTimes )
{
  // Due to the calling interval of CwGen_Handler() with approximately 2 ms,
  // simply setting
  //   pCwGen->i32CountdownForDashOrDot_us = pCwGen->cfg.iDotTime_ms * 1000 * nDotTimes
  // would cause slipping of timestamps. For example, when sent at extreme speeds,
  // the word "PARIS" took 44 .. 45 "dot times", which should have been 43:
  //     P  A  R  I  S
  //  .--. .- .-. .. ...
  //   11 + 5+ 7 + 3 + 5 (length of characters in dot times) \  total =
  //      3 +3  +3 +3 (gaps between characters)              /   43 dots
  // Fixed this by ADDING the new element duration to whatever remained
  // in pCwGen->i32CountdownForDashOrDot_us (up to 3000 microseconds) :
  int iDotTime_ms = pCwGen->cfg.iDotTime_ms;
  if( pCwGen->iTempSpeedDotTime_ms > 0 )   // currently controlled by macro <sNN> and <s> ?  (see CwGen_ParseMacro)
   { iDotTime_ms = pCwGen->iTempSpeedDotTime_ms;
   }
  if( (pCwGen->i32CountdownForDashOrDot_us < -3000) || (pCwGen->i32CountdownForDashOrDot_us > 3000) )
   { pCwGen->i32CountdownForDashOrDot_us = 0;
   }

  pCwGen->i32CountdownForDashOrDot_us += iDotTime_ms * 1000 * nDotTimes;

} // end CwGen_ReloadCountdownForDashOrDot()


//---------------------------------------------------------------------------
static void CwGen_StartDashOrDot( T_CwGen *pCwGen )
  // Does what the name implies. Only called from CwGen_Handler() .
{
  pCwGen->fMorseOutput = TRUE;
  // pCwGen->wShiftReg still contains <pCwGen->nDashesAndDotsInShiftReg>
  // elements to send, with the NEXT (or first) element to send in the
  // most significant bit (the "start bit", which is used to encode the
  // length in A SINGLE BIT of the Morse code pattern) has already been
  // shifted out *to the left*, when counting nDashesAndDotsInShiftReg.
  // Thus:
  if( pCwGen->wShiftReg & 0x8000 ) // next element to send is a DASH :
   { CwGen_ReloadCountdownForDashOrDot( pCwGen, 3 );
   }
  else // next element to send is a DOT :
   { CwGen_ReloadCountdownForDashOrDot( pCwGen, 1 );
   }
  pCwGen->wShiftReg <<= 1;
  --pCwGen->nDashesAndDotsInShiftReg;
  pCwGen->iState = CW_GEN_SEND_DASH_OR_DOT;
}

//---------------------------------------------------------------------------
static void CwGen_StartGap( // start a gap, with duration of 1..7 dot times
              T_CwGen *pCwGen,
              int nDotTimes )
  // Only called from CwGen_Handler() .
{
  pCwGen->fMorseOutput = FALSE;
  CwGen_ReloadCountdownForDashOrDot( pCwGen, nDotTimes );
  pCwGen->iState = CW_GEN_SEND_GAP;  // new state : "sending a gap"
}


//---------------------------------------------------------------------------
static void CwGen_OnEndOfChar(T_CwGen *pCwGen) // called from CwGen_Handler()
  // at the end of a character's last dash or dot -> "end of character" .
{
  pCwGen->fMorseOutput = FALSE;
  // TWO dot-times have already expired, so if TWO or THREE MORE dot times expire,
  // it's time to emit a SPACE character. Someone specified (or at least explained):
  //   > The space between letters is 3 time units.
  //   > The space between words is 7 time units.
  // In this implementation, a "time unit" is pCwGen->cfg.iDotTime_ms [N milliseconds] .
  // To allow "tolerance" for operators using a slightly shorter inter-WORD-space,
  // a SPACE CHARACTER will be emitted by the keyer (when sending) after FIVE time-units.
  // An advanced keyer firmware may extend the 'word space' to those SEVEN time-units.
  CwGen_ReloadCountdownForDashOrDot( pCwGen, 3 ); // two time-units already expired, so wait for at least THREE additional ones
  pCwGen->iState = ES_WAIT_FOR_SPACE;  // new state : "wait for the minimum gap to emit a SPACE character"
} // end CwGen_OnEndOfChar()


//---------------------------------------------------------------------------
static BOOL CwGen_PrepareSendingNextChar(T_CwGen *pCwGen)
  // NO API FUNCTION ! Only called internally, from CwGen_Handler() .
  // [in]  pCwGen->szTxMemory[pCwGen->iTxMemoryCharIndex++]
  // [out] pCwGen->wShiftReg, wCurrentCwChr, nDashesAndDotsInShiftReg, iState,
  //       pCwGen->nCharsRemainingToPlay .
  // [return] TRUE = "ok, there's more to send" (or "wait for more data before we can parse a MACRO")
  //          FALSE= "nothing more to send in the TX buffer"
{
  const char *cpSrc;
  const char *cpEndstop = pCwGen->szTxMemory + CW_GEN_TX_MEMORY_SIZE-1; // endstop for the primitive "macro parser"
  WORD wCwChar;
  int  nCharsLeftToSend;
  while( (pCwGen->iTxMemoryCharIndex >= 0 )
      && (pCwGen->iTxMemoryCharIndex < CW_GEN_TX_MEMORY_SIZE )
      && (pCwGen->szTxMemory[ pCwGen->iTxMemoryCharIndex ] != '\0') )
   { // At least one character to send .. convert from ASCII to Morse code,
     // count the number of dashes and dots to send,
     // then turn the generator on:
     cpSrc = pCwGen->szTxMemory + pCwGen->iTxMemoryCharIndex;
     // '--> Pointer to the next normal character in ASCII,
     //       or prosign in the 'Icom-inspired' notation like "^ar" or "^AR",
     //       or short macro like <sWPM>, <s> (change CW speed on the fly).
     wCwChar = Elbug_ParseMorseCodePatternFromASCII( &cpSrc );
     if( wCwChar != 0 )  // successfully retrieved the next 'Morse code pattern'
      { // Elbug_ParseMorseCodePatternFromASCII() has incremented the "char pointer"
        // by the number of characters actually parsed (usually only ONE character,
        // but it may have been more than that for 'prosigns' like "^AR ^SK" .
        //   ,------------------------------------------------------|_|
        //   '--> cpSrc would have been incremented by 3 in this example.
        pCwGen->iTxMemoryCharIndex = (int)(cpSrc - pCwGen->szTxMemory);
        pCwGen->nCharsRemainingToPlay = strlen(cpSrc);
        pCwGen->wCurrentCwChr = wCwChar; // remember this for the timing scope
        // Next state: CW_GEN_SEND_DASH_OR_DOT (for non-space characters)
        //             or CW_GEN_SEND_GAP (for space characters ~ "inter-word gap")
        if( wCwChar & CW_CHR_SPACE )  // bit 15 set -> it's a SPACE ...
         { // most likely, an "inter-word space", SEVEN dots long,
           // but since the previously sent character already included
           // a THREE-dot "inter-character space", the additional time
           // to wait (for the gap) are only FOUR dot-times :
           CwGen_StartGap( pCwGen, 4/*nDotTimes*/ );
         }
        else // not a SPACE but a character, beginning with a dash or dot.
         { // First determine the number of dashes and dots to send,
           // by bitwise shifting wCwChar to THE LEFT, until the "start bit"
           // arrives in bit 15. If the startbit ALREADY WAS in bit 15,
           // there would be FIFTEEN dashes or dots to send.. but that's
           // impossible, because bit 15 in wCwChar is CW_CHR_SPACE (0x8000).
           // Example: CW_CHR_V = 0x0011 = 0b00000000 00010001' ( "V" = ...- )
           // ,------------------------------------------'||||
           // '--> "Start bit"            three dots -----'''|
           //                               one dash --------'
           pCwGen->nDashesAndDotsInShiftReg = 15;
           while( pCwGen->nDashesAndDotsInShiftReg > 0 )
            { if( wCwChar & 0x8000 ) // "start bit" arrived in bit 15 ?
               { break;
               }
              wCwChar <<= 1;
              --pCwGen->nDashesAndDotsInShiftReg;
            }
           pCwGen->wShiftReg = wCwChar << 1; // startbit "falls out"
             // (from bit 15 into the non-existing bit 16 in a 16-bit "WORD")
           // Example ("V") : pCwGen->wShiftReg = 4096 = 0b0001 0000 0000 0000;
           //                                              ||||
           //                                             "...-"
           //                 pCwGen->nDashesAndDotsInShiftReg = 4 .
           CwGen_StartDashOrDot( pCwGen );
         }  // end else < not a SPACE >
        return TRUE;
      }    // end if( wCwChar != 0 )
     else // Elbug_ParseMorseCodePatternFromASCII() failed to parse cpSrc, i.e.
      {   // pCwGen->szTxMemory[] at pCwGen->iTxMemoryCharIndex, so STOP here ?
        if ( *cpSrc == '<' ) // don't STOP but evaluate a MACRO, e.g. "<s40>" :
         { pCwGen->wCurrentCwChr = 0;  // not sending anything from this macro yet
           // The following call may have to be repeated several times
           // until the macro (string) is complete, i.e. the closing '>' arrived.
           if( CwGen_ParseMacro( pCwGen, &cpSrc, cpEndstop ) )
            { // Parsed AND SKIPPED the macro, so continue in the while-loop
              // until Elbug_ParseMorseCodePatternFromASCII() has something TO SEND:
              if( pCwGen->iState == CW_GEN_WAITING_FOR_MACRO )
               {  pCwGen->iState = CW_GEN_START;
               }
              pCwGen->iTxMemoryCharIndex = (int)(cpSrc - pCwGen->szTxMemory);
              pCwGen->nCharsRemainingToPlay = SL_strnlen(cpSrc, CW_GEN_TX_MEMORY_SIZE );
              // Note: pCwGen->wCurrentCwChr may have been set in CwGen_ParseMacro()
            }
           else // looks like the begin of a macro, but not enough characters to parse it yet:
            { // Keep on waiting until the macro is complete, in yet another state of the CW generator:
              pCwGen->iState = CW_GEN_WAITING_FOR_MACRO;
              return TRUE;
            }
         }
        else // next character to send is not a '<' (not a macro), so guess it's END-OF-STRING
         { pCwGen->iState = CW_GEN_OFF;
           return FALSE;
         }
      }
   }

  // Arrived here ? Nothing more to send in pCwGen->szTxMemory[ pCwGen->iTxMemoryCharIndex ] !
  pCwGen->iState = CW_GEN_FINISHED_SENDING; // ex: CW_GEN_OFF
  // ,--------------'
  // '--> Polled by e.g. the 'Remote CW Keyer GUI', which may decide
  //      to append more text from the type-ahead buffer, etc etc.
  return FALSE;


} // end CwGen_PrepareSendingNextChar()


//---------------------------------------------------------------------------
int CwGen_Handler( T_CwGen *pCwGen, // API for the *worker thread*
       long nMicrosecondsSinceLastCall ) // [in] ideally 2000 us, if windoze permits !
  // Values returned by CwGen_Handler() : One of the CW_CHR-codes (More code
  // patterns are emitted on each new 'character' (or prosign) from here,
  //     including CW_CHR_SPACE,
  // or one of the ELBUG_RESULT_.. values defined in Elbug.h, for example
  //               ELBUG_RESULT_NOTHING_DECODED  ( == 0 ) .
  // Called periodically from a time-critical worker thread / task,
  //        or (in a microcontroller firmware) from a timer interrupt handler .
{
  char c;
  int  iResult = ELBUG_RESULT_NOTHING_DECODED;

  if( pCwGen->i32CountdownForDashOrDot_us > 0 )  // software timer still running ?
   {  pCwGen->i32CountdownForDashOrDot_us -= nMicrosecondsSinceLastCall; // let it COUNT DOWN,
      // until it stops at zero or with a "slightly negative value".
   }
  switch( pCwGen->iState )
   {
     case CW_GEN_OFF     : // CwGen_Handler() won't do anything
        pCwGen->fMorseOutput = FALSE;
        break;
     case CW_GEN_START   : // Let CwGen_Handler() START as soon as there is somthing to send
        pCwGen->fMorseOutput = FALSE;
        CwGen_PrepareSendingNextChar( pCwGen ); // -> wShiftReg, nDashesAndDotsInShiftReg
           // '--> next state: CW_GEN_SEND_DASH_OR_DOT (for non-space characters)
           //               or CW_GEN_SEND_GAP (for space characters ~ "inter-word gap")
        break; // end case CW_GEN_WAITING

     case CW_GEN_SEND_DASH_OR_DOT:  // sending a dot, waiting for the end, register "dash" for sequeezing
        if( pCwGen->i32CountdownForDashOrDot_us <= 0 ) // finished sending the element .. what next ?
         { pCwGen->fMorseOutput = FALSE;
           if( pCwGen->nDashesAndDotsInShiftReg > 0 ) // if more elements (dashes and/or dots) to send,
            { CwGen_StartGap( pCwGen, 1/*nDotTimes*/ );
            }
           else // just completed sending a character, so what next ?
            { CwGen_StartGap( pCwGen, 3/*nDotTimes*/ );
              if( ! (pCwGen->wCurrentCwChr & CW_CHR_SPACE ) )
               { iResult = pCwGen->wCurrentCwChr; // emit this "info" for the timing scope
               }
            }
         }
        break; // end case CW_GEN_SEND_DASH_OR_DOT

     case CW_GEN_SEND_GAP: // sending a GAP with the duration of N dots
        if( pCwGen->i32CountdownForDashOrDot_us <= 0 ) // finished sending a GAP .. what next ?
         {
           if( pCwGen->nDashesAndDotsInShiftReg > 0 ) // if more elements (dashes and/or dots) to send,
            { CwGen_StartDashOrDot( pCwGen );         // start sending the next ..
            }
           else
            { CwGen_PrepareSendingNextChar( pCwGen );
              // '--> next state: CW_GEN_SEND_DASH_OR_DOT (for non-space characters)
              //               or CW_GEN_SEND_GAP (for space characters ~ "inter-word gap")
              //               or CW_GEN_FINISHED_SENDING (when there's nothing more to send).
              if( (pCwGen->wCurrentCwChr & CW_CHR_SPACE) || (pCwGen->iState==CW_GEN_OFF) )
               { iResult = CW_CHR_SPACE; // <- added 2025-01 for the 'QSO transcript'
               }
              iResult |= ELBUG_RESULT_BEGIN_NEW_CHAR;
            }
         }
        break; // end case CW_GEN_SEND_GAP

     case CW_GEN_FINISHED_SENDING :
        pCwGen->fMorseOutput = FALSE;
        break; // end case CW_GEN_FINISHED_SENDING

     case CW_GEN_WAITING_FOR_MACRO: // waiting for more text fed into szTxMemory[] before we can parse the end of a MACRO like "<sWPM>", etc .
        // (from the CW generator's point of view, this is like CW_GEN_SEND_GAP
        //  but without a "time limit", because the human operator at the keyboard may be a slow typer)
        // The EXIT from this state happens in CwGen_PrepareSendingNextChar(),
        // when the '>' (=marker for the end of the macro) is available. 
        break; // end case CW_GEN_WAITING_FOR_MACRO

     default:  // oops...
        pCwGen->iState = CW_GEN_OFF;
        break;

   } // end switch( pCwGen->iState )

  return iResult;

} // end CwGen_Handler()

//---------------------------------------------------------------------------
BOOL CwGen_ParseMacro( T_CwGen *pCwGen,
      const char **cppSrc, // [in] source pointer to the BEGIN of the macro,
                           //      e.g. *cppSrc = "<s40>"...  
                           // [out] pointer to the next character AFTER the macro
    const char *cpEndstop) // [in] "endstop", just in case *cppSrc isn't
                           //      a properly terminated C-string .
  // Returns TRUE when the end of the macro has been found, and the macro
  //              was SKIPPED, and the caller may CONTINUE SENDING TEXT
  //              (at the *incremented* *cppSrc) .
  // Returns FALSE if the marker for any macro's end ('>') isn't available yet.
  // Periodically called from CwGen_PrepareSendingNextChar(), so don't waste
  // time HERE if the macro isn't "ready for being interpreted".
{
  const char *cpSrc = *cppSrc;
  const char *cpSrc2;
  int  iEndPos;
  BOOL fEndingTag = FALSE;  // ... inspired by tags in HTML, e.g. "</s>" = END of the previous "<s..>"
  int  iParamValue;
  // Look ahead until the "endstop" (last character available in a buffer).
  // All macros BEGIN with a '<', and end with a '>', so it's easy to tell
  // if we have all the required characters BEFORE parsing the command itself.
  if( *cpSrc != '<' ) // oops.. this is NOT the begin of a macro !
   { return TRUE;     // let the caller CONTINUE SENDING, at the character that isn't the begin of a macro
   }
  iEndPos = SL_FindCharInArray( cpSrc, cpEndstop, '>' );
            //  '--> e.g. with cpSrc="<s40>Blah", iEndPos=4 (ZERO-BASED array index),
            //    - but the value of iEndPos doesn't matter here as long as it's POSITIVE.
  if( iEndPos > 0 )  // ok, found the '>', so go ahead and "interpret" the macro
   { cpSrc2  = *cppSrc + 1; // *cppSrc points to the '<', cpSrc2 to the NEXT character (= expected "comand")
     if( SL_SkipChar( &cpSrc2, '/' ) )
      { fEndingTag = TRUE;
      }
     switch( *cpSrc2 )
      { case 's' :    // change CW speed temporarily, or switch back to the speed configured in the GUI.
           // Example: "remote cw keyer test <s40>hello contest gun pse <s20>qrs<s> pse again my report"
           ++cpSrc2;  // skip the 's', which MAY be followed by a decimal number:
           iParamValue = SL_ParseInteger( &cpSrc2 );
           if( iParamValue > 0 ) // obviously not the "ending tag" (</s> or just <s> )
            { pCwGen->iTempSpeedDotTime_ms = Elbug_WordsPerMinuteToDotTimeInMilliseconds( iParamValue );
              // '--> e.g. 30 milliseconds when parsing "<s40>" for the "contester's brrrt-dadit-dadit"
            }
           else // switch back to the "normal" speed :
            { pCwGen->iTempSpeedDotTime_ms = 0;  // use pCwGen->cfg.iDotTime_ms again 
            }
           *cppSrc = cpSrc2+1; // skip the macro, including the final '>', FOR THE CALLER
           break;     // end case "<sNN>" or just "<s>"

        default :     // got something between angle brackets, but not a recognized macro !
           // keep it simple; let the caller send the macro as if it was ordinary text.
           *cppSrc = cpSrc+1; // throw away the '<' to send the rest of the macro as ordinary text
           break;     // (return TRUE further below)
      } // end switch < character immediately after the '<' >
     return TRUE;     // let the caller CONTINUE SENDING, at the next character AFTER the macro
   }
  (void)fEndingTag; // ... assigned a value that is never used ..
                    // shut up, better than using a non-initialized value

  return FALSE;  // "need more characters in *cppSrc, please call me again.."
                 //  '--> Generator state switches to CW_GEN_WAITING_FOR_MACRO;
                 //       GUI will quickly fill up the TX-buffer if possible,
                 //       because it's aware of that special state .
                 //   See KeyerGUI_TransferCharsFromEditFieldToCwGenerator() !

} // end CwGen_ParseMacro()


//---------------------------------------------------------------------------
void CwGen_StartReplay( T_CwGen *pCwGen, char* pszTextToSend )
  // Usually called from the GUI when the operator wants to send text
  //         from one of the "Memories".
  // [in]  pszTextToSend   : text from the to-be-played 'keyer memory'.
  // [out] pCwGen->iTxMemoryCharIndex, will be incremented WHILE playing.
  // The 'real work' happens later, when the keyer thread (or task on a uC)
  // calls CwGen_Handler() every two milliseconds or so... including the
  // RX/TX-switch-sequencing.
{
  int iLength = SL_strnlen( pszTextToSend, CW_GEN_TX_MEMORY_SIZE );
  memset( pCwGen->szTxMemory, 0, sizeof_member(T_CwGen,szTxMemory)  );
  strncpy( pCwGen->szTxMemory, pszTextToSend, iLength );
  pCwGen->iTxMemoryCharIndex = pCwGen->i32TxMemoryNumCharsRemoved = 0;
  pCwGen->nCharsRemainingToPlay = iLength;
  pCwGen->iState = CW_GEN_START;
  // Note: Sending "from memory" automatically stops on the first
  //       zero-byte in pCwGen->szTxMemory, or when
  //       pCwGen->iTxMemoryCharIndex exceeds CW_GEN_TX_MEMORY_SIZE .
  //       Dont't call CwGen_PrepareSendingNextChar() from here,
  //       because due to the complete (and intended) lack of thread synchro-
  //       nisation ('critital sections', etc), functions that are called from a
  //       worker thread (like CwGen_Handler() -> CwGen_PrepareSendingNextChar())
  //       shall not be called from API functions (intended to be called from
  //       the GUI), like CwGen_StartReplay() .
} // end CwGen_StartReplay()

//---------------------------------------------------------------------------
int CwGen_GetFreeSpaceInReplayBuffer( T_CwGen *pCwGen )
  // Called from the GUI before preparing "text" to pass via CwGen_AppendForReplay().
  // Returns the number of characters that CwGen_AppendForReplay() can accept at the moment.
{
  return CW_GEN_TX_MEMORY_SIZE - pCwGen->iTxMemoryCharIndex; // simple because this isn't a CIRCULAR FIFO :)
} // end CwGen_GetFreeSpaceInReplayBuffer()


//---------------------------------------------------------------------------
void CwGen_ClearReplayBuffer( T_CwGen *pCwGen )
{
  memset( pCwGen->szTxMemory, 0, sizeof_member(T_CwGen,szTxMemory)  );
  pCwGen->iTxMemoryCharIndex = pCwGen->i32TxMemoryNumCharsRemoved = pCwGen->nCharsRemainingToPlay = 0;
  if( pCwGen->iState == CW_GEN_FINISHED_SENDING ) // here: automatically continue sending, in CwGen_AppendForReplay()
   {  pCwGen->iState = CW_GEN_START;
   }
} // end CwGen_ClearReplayBuffer()


//---------------------------------------------------------------------------
void CwGen_AppendForReplay( T_CwGen *pCwGen, char* pszMoreTextToSend )
  // CwGen_AppendForReplay() appends new data, in addition to those already
  //              present in pCwGen->szTxMemory[] from CwGen_StartReplay().
  // [in]  pszMoreTextToSend : text appended to internal 'tx memory'.
  // [out] pCwGen->iTxMemoryCharIndex : Will be incremented WHILE playing.
  // [out] pCwGen->i32TxMemoryNumCharsRemoved : Will be incremented HERE,
  //              if characters must be "scrolled out" of pCwGen->szTxMemory[]
  //              when strlen( pCwGen->szTxMemory[] )
  //                 + strlen( pszMoreTextToSend )
  //              exceeds CW_GEN_TX_MEMORY_SIZE (e.g. 1023 characters).
  //       This is intended for the GUI, for the coloured backgrounds
  //       in the 'type-ahead buffer' :
  //       The sum of iTxMemoryCharIndex + i32TxMemoryNumCharsRemoved
  //       is the number of characters sent since calling CwGen_StartReplay(),
  //       which may greatly exceed CW_GEN_TX_MEMORY_SIZE .
{
  int iOldTxIndex  = pCwGen->iTxMemoryCharIndex;
  int iOldLength   = SL_strnlen( pCwGen->szTxMemory, CW_GEN_TX_MEMORY_SIZE );
  int iAddedLength = SL_strnlen( pszMoreTextToSend, CW_GEN_TX_MEMORY_SIZE );
  int nCharsToRemove = iOldLength + iAddedLength - CW_GEN_TX_MEMORY_SIZE;
  if( nCharsToRemove > 0 ) // got to REMOVE some characters from szTxMemory[] ?
   { // May be problematic, because this requrires modifying iTxMemoryCharIndex
     // while another thread (the "Keyer Thread" ?) reads szTxMemory[iTxMemoryCharIndex++] .
     // pCwGen->iTxMemoryCharIndex is the number of characters that
     // CwGen_Handler() -> CwGen_PrepareSendingNextChar() doesn't need any longer.
     if( nCharsToRemove >= CW_GEN_TX_MEMORY_SIZE ) // remove EVERYTHING in szTxMemory !
      { CwGen_ClearReplayBuffer( pCwGen );
        SL_strncpy( pCwGen->szTxMemory, pszMoreTextToSend, iAddedLength );
        return;
      }
     if( iOldTxIndex < iOldLength ) // some old text must remain in szTxMemory :
      { // "Scroll up" szTxMemory[] .   Example: BEFORE the call..
        //  i32TxMemoryNumCharsRemoved = 10000 (from tousands of earlier calls)
        //              ,------------------------------------------------------,
        //  szTxMemory: | Text already sent. Old text not sent.                |
        //              '------------------------------------------------------'
        //               |                  |
        //               0              iOldTxIndex=19
        //               |<------------ iOldLength=38 ------->|
        //
        //  .. and after the call :
        //              ,------------------------------------------------------,
        //  szTxMemory: | Old text not sent. New Text appended in this call.   |
        //              '------------------------------------------------------'
        //               |
        //              iTxMemoryCharIndex=0
        //
        //  i32TxMemoryNumCharsRemoved = 10019 (iOldTxIndex = 19 chars  were removed in THIS call)
        //
        pCwGen->i32TxMemoryNumCharsRemoved += iOldTxIndex;
        iOldLength -= iOldTxIndex; // now this is the length of the 'Old text not sent', REMAINING after the "move" below
        // Source and destination overlap, so use memmove(), not memcpy() :
        memmove( pCwGen->szTxMemory/*dst*/, pCwGen->szTxMemory+iOldTxIndex/*src*/, iOldLength );
        if( (iOldLength > 0 ) && (iOldLength<CW_GEN_TX_MEMORY_SIZE) )
         { // Clear the "new empty part" at the END of pCwGen->szTxMemory[CW_GEN_TX_MEMORY_SIZE] :
           memset( pCwGen->szTxMemory+iOldLength, 0, (CW_GEN_TX_MEMORY_SIZE-iOldLength) );
           // (We want the string-terminating zeros in this as quickly as possible,
           //  so CwGen_PrepareSendingNextChar() doesn't see 'old stuff' twice.)
         }
        if( iAddedLength > (CW_GEN_TX_MEMORY_SIZE-iOldLength) ) // oops.. too much 'New Text appended in this call' !
         {  iAddedLength =  CW_GEN_TX_MEMORY_SIZE-iOldLength;
         }
        pCwGen->iTxMemoryCharIndex = iOldTxIndex = 0;
        pCwGen->nCharsRemainingToPlay = iOldLength - pCwGen->iTxMemoryCharIndex;
      }
   } // end if( nCharsToRemove > 0 )

  if( (iAddedLength > 0 ) && ((iOldLength+iAddedLength) < CW_GEN_TX_MEMORY_SIZE) )
   { SL_strncpy( pCwGen->szTxMemory+iOldLength, pszMoreTextToSend, CW_GEN_TX_MEMORY_SIZE-1-iOldLength );
     pCwGen->nCharsRemainingToPlay = iOldLength + iAddedLength - pCwGen->iTxMemoryCharIndex;
     if( pCwGen->iState == CW_GEN_FINISHED_SENDING ) // here: automatically continue sending, in CwGen_AppendForReplay()
      {  pCwGen->iState = CW_GEN_START;
      }
   }

} // end CwGen_AppendForReplay()


//---------------------------------------------------------------------------
long CwGen_GetNumCharsPlayedSinceStart( T_CwGen *pCwGen )
{
  return pCwGen->iTxMemoryCharIndex + pCwGen->i32TxMemoryNumCharsRemoved;
} // end CwGen_GetNumCharsPlayedSinceStart()


//---------------------------------------------------------------------------
void CwGen_StopReplay( T_CwGen *pCwGen )
{
  pCwGen->iState = CW_GEN_OFF; // too trivial for an extra subroutine,
                               // but this may change in future versions
} // end CwGen_StopReplay()


//---------------------------------------------------------------------------
CPROT BOOL CwGen_IsTxBusy( T_CwGen *pCwGen ) // -> TRUE=yes=busy, FALSE=not busy
  // Implemented 2025-06-07 for module AuxCom_Winkeyer.c; see details THERE.
  // If this function indicates "CW generator NOT busy from transmission",
  // the Winkeyer-emulation may e.g. change the keying speed between two
  // characters, without spoiling the CW timing.
{
  if( pCwGen->nDashesAndDotsInShiftReg > 0 )
   { return TRUE;  // transmit-shift-register not empty -> TX BUSY
   }
  if( pCwGen->nCharsRemainingToPlay > 0 ) // some characters are still waiting to be played in szTxMemory[]
   { return TRUE;  // -> TX also STILL BUSY
   }

  return FALSE;    // CW-generator NOT busy from transmission anymore (or not yet)
} // end CwGen_IsTxBusy()

/* EOF < CwGen.c > */

