// File:  C:\cbproj\Remote_CW_Keyer\CwDSP.h
// Date:  2023-12-06
// Author:  Wolfgang Buescher (DL4YHF)
// Purpose: 'CW Digital Signal Processor' - initially just a
//          sine wave generator with adjustable frequency,
//          amplitude, and envelope shaping.
//          Details in the implementation, Remote_CW_Keyer/CwDSP.c .

#ifndef  _CW_DSP_H  // prevent multiple header inclusion ..
# define _CW_DSP_H  //  .. and let other modules know we're been included


#ifndef  STRAIGHT_KEY_DECODER_H_INCLUDED
# include "StraightKeyDecoder.h" // Morse decoder for input from a STRAIGHT KEY
#endif

//---------------------------------------------------------------------------
// Constants
//---------------------------------------------------------------------------


#ifndef TRUE
# define TRUE  1
# define FALSE 0
#endif

#ifndef CW_CHR_0  // Morse code patterns for common "CW characters" not defined yet ?
# include "Elbug.h" // .. should be defined in Elbug.h
#endif

#include "SoundTab.h"       // cosine lookup table, filter coefficients, T_Float, etc.
//ex: #include "Goertzel.h" // DFT-like array of 'Goertzel' filters for the CW decoder
#include "SampleRateConv.h" // sample rate converter from 48 to 8 kS/sec and back
#include "Timers.h"         // T_TIM_Stopwatch, etc (high-res, portable "timers")

#if( SWI_USE_DSOUND && (!defined __DSOUND_WRAPPER_H) ) // use "DirectSound" / dsound_wrapper.c ?
# include "dsound_wrapper.h" // "Simplified DirectSound interface" in plain C
#endif // SWI_USE_DSOUND ?


//---------------------------------------------------------------------------
// Constants
//---------------------------------------------------------------------------
#if(0)  // tried a lot of combinations for the sampling rate and audio buffer size..
# define CWDSP_SAMPLING_RATE          12000
# define CWDSP_SIDETONE_BUFFER_NSAMPLES 256 // a buffer with 256 sample points at 12 kSamples/sec was ok,
   // .. but when completely filled, means an audio latency of 21 milliseconds !
   // Maybe Mr. DirectSound is happy with lower powers of two (for the buffer size) ?
#elif(0)
# define CWDSP_SAMPLING_RATE          12000
# define CWDSP_SIDETONE_BUFFER_NSAMPLES 128 // a 128-sample buffer at 12 kSamples/sec made a horrible sound
#elif(1)
   // Maybe Mr. DirectSound is more happy with 48 kHz (which is the default
   // setting in more recent windows versions) / less "unexplainable latency" ?
# define CWDSP_SAMPLING_RATE          48000
# define CWDSP_SIDETONE_BUFFER_NSAMPLES 1024 // a buffer with 1024 sample points at 48 kSamples/sec
   // also means 21.3 ms of "audio" in the buffer ..
   // surprisingly, this was a bit better than a 128-sample buffer at 12 kSamples/second,
   // but it could not beat the low-latency "audio via TXD on the COM port".
#else
# define CWDSP_SAMPLING_RATE          48000
# define CWDSP_SIDETONE_BUFFER_NSAMPLES 512 // 512 sample points / 48 kHz = 10.66 ms ... nogo
#endif // which combination of SAMPLING_RATE and OUTPUT BUFFER SIZE ?
   // Looking for the audio sample rate used in all NETWORK AUDIO STREAMS ?
   // It's not defined HERE (in CwDSP.h) but in CwNet.h : CWNET_STREAM_SAMPLING_RATE (8000) !
   //

// For the audio INPUT (usually from the remote radio on the SERVER side),
// latency isn't as critical as for the SIDETONE OUTPUT, last not least
// because the SERVER THREAD only sends a new TCP segment every 20 milliseconds.
#define CWDSP_INPUT_BUFFER_NSAMPLES  4096 // ex: 42.6 ms * 48 kHz = 2048 samples; parameter for DSW_InitInputBuffer() (*)
                                          // (DirectSound's "dwReadPos" made coarse steps of 3000 (bytes) at f_sample=48 kHz,
                                          //  thus decided to increase the INPUT buffer from 2048 to 4096 samples in 2025-04)
#define CWDSP_OUTPUT_BUFFER_NSAMPLES 2048 // 42.6 ms * 48 kHz = 2048 samples; parameter for DSW_InitOutputBuffer()
   // (*) Even with CWDSP_INPUT_BUFFER_NSAMPLES = 2048 = ~~42 ms, when playing
   //     the almost-latency-free SIDETONE FROM THE COM-PORT'S TXD OUTPUT
   //     back into the PC's onboard microphone, and plotting the result in the
   //     timing scope, resulted in a total latency of originally 138 ms
   //     - see screenshot in
   //       Remote_CW_Keyer/manual/MIC_input_latency_with_DirectSound.png !
   // The 'DSP process FIFO', running at the internal, DECIMATED sampling rate
   // of 8 kHz (just enough for 3400 Hz for SSB) is also used to buffer samples
   // received from a network connection (TCP/IP), it should be large enough
   // to store at least 300 ms of audio (see latency / jitter tests in CwNet.c).
   // 300 ms * 8 kHz would be 2400 samples, so let's use the next larger POWER OF TWO:
#define CWDSP_PROCESS_FIFO_NSAMPLES 4096 // > 300 ms * 8 kHz = 2400 -> 4096 samples
     // (MUST be a power of two for OUR circular FIFO, also in THIS case)
     // With e.g. CWDSP_PROCESS_FIFO_NSAMPLES = 4096 / fs = 8 kHz,
     // 512 milliseconds of samples CAN be buffered in T_CwDSP.sNetworkRxFifo .
     // This allows CwNet_OnReceive() -> CwDSP_ProcessSampleFromReceivedAudioStream()
     //  to PUSH received samples into this thread-safe FIFO, even if for some
     //  reason TCP segments only arrive every 500 milliseconds.
     // (.. which acutally happened when streaming audio via MOBILE internet
     //     on one side to a poor ADSL connection on the other side.
     //     See periodic "PING"-latency test results via TCP/IP in CwNet.c . )

   // FFT length (number of samples entering each short-time FFT in the TIME DOMAIN):
#define CWDSP_AUDIO_SPECTRUM_FFT_LENGTH 256  // initially 128, but more gave "better copy" of slow weak signals
   // ,----------------------------------'
   // '--> Must be short enough for 'separability' of CW dots in the time domain,
   //      must be long enough for 'separability' between CW signals in the frequency domain.
   //      We're aiming for say 35 WPM = only 34 milliseconds per dot !
   //      For the REAL FFT (real-valued input), N REAL samples in the time domain
   //      gives N/2 COMPLEX frequency bins (spanning 0 Hz .. f_Samples/2),
   //  so initially tried CWDSP_AUDIO_SPECTRUM_FFT_LENGTH = 128 ->
   //      Frequency bin width = CWDSP_INPUT_FIFO_SAMPLING_RATE / CWDSP_AUDIO_SPECTRUM_FFT_LENGTH
   //                          = 8000 Hz / ( 128 time_domain_samples_per_FFT ) = 62.5 Hz .
#define CWDSP_AUDIO_SPECTRUM_NUM_FREQUENCY_BINS (1+CWDSP_AUDIO_SPECTRUM_FFT_LENGTH/2)
   //      Due to the 50 percent overlap between subsequent FFTs,
   //      new audio spectra were originally analysed every
   //            (CWDSP_AUDIO_SPECTRUM_FFT_LENGTH / 2 ) / ( 8000 Hz )
   //         =  (128 samples / 2 ) / 8000 samples_per_second = 8 ms.
   //      To reliably catch an audio spectrum with "no signal" for reference,
   //      the "peak power history" (from the FFT bin with the tracked carrier)
   //      must be as long the longest GAP : 5 dot times between two words,
   //      lowest practical CW speed = 5 WPM, thus :
   //      5 dots * ( 1200 ms / 5 WPM ) = 1200 ms ->
   //      MINIMUM History length = 1200 ms / 8 ms = 150 elements = ~~20000/128
#define CWDSP_DECODER_POWER_HISTORY_LENGTH (20000/CWDSP_AUDIO_SPECTRUM_FFT_LENGTH)

//ex: #define  CWDSP_COS_TABLE_SIZE 4096  // <- size must be a power of two !
//ex: extern float CwDSP_fltCosTable[CWDSP_COS_TABLE_SIZE]; // lookup table with a single sinewave period
//     '--> Back to the original cosine lookup table in SoundTab.cpp,
//          which is also used by OTHER DSP modules like Goertzel.cpp .
//          Use           SoundTab_fltCosTable[SOUND_COS_TABLE_LEN] (larger!)
//             instead of CwDSP_fltCosTable[CWDSP_COS_TABLE_SIZE] .
//


//---------------------------------------------------------------------------
// Data Types
//---------------------------------------------------------------------------

typedef struct t_CwDspFifoHead // replacement for T_CwDSP_Fifo.iHeadIndex : T_CwDSP_Fifo.head.index (along with a timestamp)
{
  int    index;           // Replacement for e.g. T_CFIFO.iHeadIndex : T_CFIFO.head.index .
                          // The FIFO-"writer" updates this field FIRST, before all other members.
  int    index2;          // Copy of 'index'. Must contain the same as 'index' if the struct is consistent.
                          // The FIFO-"writer" updates this field LAST, after all other members.
  double dblTimestamp_s;  // precise timestamp in seconds, comparable with e.g. dsound_wrapper.h : DSW_ReadHighResTimestamp_s()
} T_CWDSP_FifoHead;


typedef struct t_CwDSP_Fifo // Classic, lock-free, circular FIFO
{ // for digital signal processing. Using 'float' samples, normalized to -1.0 .. +1.0 .
  // Also used for buffering input FROM THE NETWORK .
  T_CWDSP_FifoHead head;
  int    iTail;
  double dblSecondsPerSample;  // sampling interval IN SECONDS PER SAMPLE, for timestamp calculations
  float  fltQueue[CWDSP_PROCESS_FIFO_NSAMPLES];
} T_CwDSP_Fifo;

typedef struct t_CwDSP_PlotterSample // a single element for plotting data from CW DECODER in the keyer's "Timing Scope",
{ // including the short-time audio power spectrum, the "rectified keying signal(s)",
  // the tracked peak frequency, and the Morse code pattern decoded by the main audio decoder instance .
  double dblTimestamp_s;  // timestamp in seconds, comparable with DSW_ReadHighResTimestamp_s().
         // (this timestamp is used to retrieve a certain element in the plotter,
         //  because due to the FFT-block-processing, there's some latency,
         //  up to T_CwDSP.dblPlotterSamplingInterval_s [measured in seconds]
         //  Details in CwDSP_ReadFromPlotterFifo().
         //  dblTimestamp_s = 0 means 'no valid data' in
         //  T_KeyerTimingScope.sSamplePoints[TIMING_SCOPE_NUM_SAMPLE_POINTS],
         //  see CwKeyer_CollectDataForTimingScope() .
  DWORD dwFFTCounter; // Counts FFTs (short-time power spectra). Wraps around
                   // from 0xFFFFFFFF to zero after ~~ 16 ms * 2^32 = 2 years.
                   // May be an alternative to the timestamp, for plotting.

  float fltCwDecoderCenterFrequency;
  float fltCwDecoderKeyingSignal;
  float fltAudioPowerSpectrum[CWDSP_AUDIO_SPECTRUM_NUM_FREQUENCY_BINS];
  WORD  wCwPattern;
  WORD  wReserve;

} T_CwDSP_PlotterSample; // -> T_CwDSP.sPlotterSampleFifo[CWDSP_PLOTTER_SAMPLE_FIFO_SIZE]


typedef struct tAudioCwDecoder // everthing we need for a single 'Audio CW Decoder' instance
{ // (There may be more than one instance, but all fed from THE SAME 'audio spectrum'.
  //  Thus there is already an ARRAY of T_AudioCwDecoder in T_CwDSP,
  //  despite the stupid amputated "digital IF output" of the IC-7300,
  //  which offered exactly the same narrow bandwidth as the rig's own CW filter.
  //  No chance to track multiple CW transmissions in a 'CW Skimmer' fashion yet...)
  int iFirstFrequencyBin, iLastFrequencyBin; // FFT frequency bin range for THIS decoder instance.
  int iCwDecoderState;       // One of the following:
#define CWDSP_DECODER_STATE_OFF       0
#define CWDSP_DECODER_STATE_SQUELCHED 1 // active but no valid signal (also no short gap AFTER a signal)
#define CWDSP_DECODER_STATE_ACTIVE    2 // found a signal with sufficient SNR for the demodulator/decoder
#define CWDSP_DECODER_STATE_INIT_ERROR -1 // illegal configuration or errors trying to initialize the demodulator/decoder
  float fltCwDecoderCenterFrequency; // momentary audio carrier center frequency, subject to AFC (automatic frequency control)
  float fltCwDecoderSignalPower; // momentary "signal power" near fltCwDecoderCenterFrequency, displayed as the "ball riding the waves" of the audio spectrum.
                                 // Like the "noise power", neither logarithmized nor normalized.
  float fltCwDecoderKeyingSignal; // "rectified" keying signal, still "analog",
                                  // ranging from 0.0 = "at the minimum in the power history"
                                  //           to 1.0 = "at the maximum in the power history" .
  float fltPowerHistory[CWDSP_DECODER_POWER_HISTORY_LENGTH];
  int   iPowerHistoryIndex, iPowerHistoryLength;
  float fltCwDecoderNoisePower;  // momentary "noise power" (density), based on the "power history"

  T_CwDSP_Fifo sKeyingSignalFifo; // FIFO with the most recent values of fltCwDecoderKeyingSignal, also used for PLOTTING - see data acquisition in KeyerThread.c

  T_StraightKeyDecoder MorseDecoder;

# define CWDSP_DECODER_FIFO_SIZE 16
  WORD wCwPatternFifo[CWDSP_DECODER_FIFO_SIZE]; // <- filled in DspThread(), drained in the GUI-thread
  int  iCwPatternFifoHeadIndex; // index into wCwPatternFifo[] for the next decoded CW-pattern
  int  iCwPatternFifoUsage;     // number of elements in the FIFO (maxes out at CWDSP_DECODER_FIFO_SIZE)


} T_AudioCwDecoder; // -> in T_CwDSP: T_AudioCwDecoder AudioCwDecoder[CWDSP_NUM_AUDIO_CW_DECODERS]


typedef struct t_CwDSP // everthing we need for a single 'CW DSP' instance:
{

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // "configuration parameters" (must be filled out be the application before start)
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  struct { // .cfg = "everything that needs to be SAVED between sessions" (and included in a hash value):
     char sz255AudioInputDevice[256];  // anyone to 'whistle CW' via microphone into the remote transmitter ?
     char sz255AudioOutputDevice[256]; // alternative sidetone-, or even 'receive audio' output.
     int iAudioFlags;      // bitwise combination of the following, mostly from the "Audio" tab:
#     define DSP_AUDIO_FLAGS_NONE                   0x0000 // only "local audio pass-through", no network audio
#     define DSP_AUDIO_FLAGS_ALLOW_NETWORK_AUDIO    0x0001
#     define DSP_AUDIO_FLAGS_DECODE_CW              0x0002 // try to decode CW in the RECEIVED audio signal
#     define DSP_AUDIO_FLAGS_PEAK_HOLDING_SPECTRUM  0x0004 // provide a PEAK-HOLDING audio power spectrum (on the display)
#     define DSP_AUDIO_FLAGS_SEND_NETWORK_TEST_TONE 0x0010

     int iSidetoneFreq_Hz; // 0 = no CW sidetone to add to the output, else frequency in Hertz
      // ( preferrably 0 = "off", in favour of the 'Sidetone on TXD' with a lower latency )

     int iSidetoneRiseTime_ms;  // rise- and fall time for the CW sidetone, in milliseconds
     int iSidetoneGain_dB; // 0 dB = normal volume (not at the CLIPPING POINT but some dB down); +/- 20 dB to amplify or attenuate
     int iAudioInGain_dB;  // 0 dB = "read from audio-input without EXTRA gain", range maybe +/- 20 dB (limited by sliders in the GUI)
     int iAudioOutGain_dB; // 0 dB = "read from audio-input without EXTRA gain"; else amplify/attenuate it by up to 20 dB (?)
     int iNetworkTonesGain_dB; // 0 dB = "emit SIGNALLING TONES from the network as they came in"; +/- 20 dB to amplify or attenuate

     int iCwDecoderDotTime_ms;
   } cfg; // <- end of the part that needs to be stored permanently (between sessions)

  int iOutputState;     // what is the AUDIO OUTPUT currently being used for ?
#     define DSP_OUTPUT_STATE_OFF 0 // output currently 'sending silence' (or not opened at all)
#     define DSP_OUTPUT_STATE_GENERATE_SIDETONE 1 // output currently occupied for the LOW-LATENCY SIDETONE
#     define DSP_OUTPUT_STATE_RECEIVER_AUDIO    2 // output currently occupied to play back the LOCAL RECEIVER'S audio signal
#     define DSP_OUTPUT_STATE_NETWORK_AUDIO     3 // output currently occupied to play received NETWORK AUDIO (e.g. from remote server)


#if( SWI_USE_DSOUND ) // use "DirectSound" / dsound_wrapper.c ?
  DSoundWrapper *pDSW; // Address of the application's "DirectSound" wrapper instance.
                       // NULL if the user decided not to use 'audio output' at all.
#endif // SWI_USE_DSOUND ?


  BOOL  fKeyDown;    // "key down" signal for the synthesized sidetone.
                     // Set or cleared via CwDSP_UpdateSidetone().

  float fltSidetoneRamp;  // multiplier for the 'sidetone sinewave'.
           // normalized from 0 (="off") to 1.0 (="full volume"),
           // incremented or decremented after each audio sample
           // by an increment, depending on iSidetoneRiseTime_ms .
  int   iSidetonePhaseAccu, iSidetonePhaseInc;
  int   i16SidetoneSamplesPerUpdate[ 16 ];
  DWORD dwSidetoneUpdates;


  char  *pszLastError;  // NULL when there's nothing to report (in the error log),
           // but often pointed to a string on a fixed address
           // in the DirectSound-wrapper's error-message-string-buffer.
  char  sz255LastError[256]; // buffer for 'sprintf'-generated error messages
  T_TIM_Stopwatch sw_ThreadWatchdog;

  T_CwDSP_Fifo sInputFifo;   // input from the 'local audio device', already DOWNSAMPLED and normalized (sample value range: -1.0 .. +1.0)
#define CWDSP_INPUT_FIFO_SAMPLING_RATE (CWDSP_SAMPLING_RATE/6) // sampling rate for sInputFifo (e.g. 8000 samples/second, also for most of the internal processing)
        // ( When SENDING AUDIO to a network, CwNet.c pulls the TO-BE-TRANSMITTED
        //   samples from sInputFifo .. NOT from sOutputFifo, because other users
        //   may prefer their own audio filter settings.
        //   See audio flow sketched in CwDSP.c : DspThread() .
        // )

  T_CwDSP_Fifo sOutputFifo;  // output to the 'local audio device', before UPSAMPLING.
#define CWDSP_OUTPUT_FIFO_SAMPLING_RATE (CWDSP_SAMPLING_RATE/6) // sampling rate for sOutputFifo


  T_CwDSP_Fifo sNetworkRxFifo; // audio stream RECEIVED FROM a network (TCP/IP), also DOWNSAMPLED,
                               // added to other 'tones' sent to the local audio output device.
                               // (due to large jitter in the network latencies,
                               //  this buffer must be large enough for at least 300 ms of audio.
                               //  Thus the rather large value of CWDSP_PROCESS_FIFO_NSAMPLES..)
#define CWDSP_NETWORK_FIFO_SAMPLING_RATE (CWDSP_SAMPLING_RATE/6) // sampling rate for sNetworkRxFifo


  // Stuff to process the AUDIO INPUT (for streaming, from remote receiver to clients):
  T_SRCONV_DECIMATOR Decimators[2];    // decimator chain to convert audio input from 48 to 8 kSamples/sec
  T_SRCONV_DECIMATOR Interpolators[2]; // interpolator chain to convert back from 8 to 48 kSamples for output
         // (the "internal DSP sampling rate" of 8 kHz not only reduces the
         //  network bandwidth for streaming, but also the internal CPU load,
         //  and with f_sample = 8 kHz, we have a theoretic audio bandwidth
         //  of 4 kHz if the decimators and interpolators were 'ideal'.
         //  They are NOT ideal (with their finite FIR kernel lenghts),
         //  but this is adequate for the ~~ 2.7 kHz bandwidth in SSB or CW.
  float fltFFTRe[CWDSP_AUDIO_SPECTRUM_FFT_LENGTH]; // real in- and output for FFT_CalcRealFft()
  float fltFFTIm[CWDSP_AUDIO_SPECTRUM_FFT_LENGTH]; // imaginary part of the output from FFT_CalcRealFft()
  float fltFFTWindow[CWDSP_AUDIO_SPECTRUM_FFT_LENGTH]; // real-valued FFT window
  float fltFFTInputQueue[CWDSP_AUDIO_SPECTRUM_FFT_LENGTH]; // non-windowed input queue 'feeding' the overlapped FFT.
        // NOT WINDOWED, because every sample must be multiplied with the FFT window TWICE.
        // See ASCII sketch in DspThread(): When iFFTInputQueueIndex++ reaches CWDSP_AUDIO_SPECTRUM_FFT_LENGTH,
        // it's time for another OVERLAPPED FFT; after that, HALF (the oldest part) of
        // in fltFFTInputQueue[] is scrolled out.
  DWORD dwFFTCounter; // Counts FFTs (short-time power spectra). Wraps around
                   // from 0xFFFFFFFF to zero after ~~ 16 ms * 2^32 = 2 years.
                   // May be an alternative to the timestamp, for plotting.
  int  iFFTInputQueueHeadIndex;  // index for the next sample, entered in fltFFTInputQueue[iFFTInputQueueHeadIndex++].
  T_Complex CwDec_ComplexAudioFrequencyBins[CWDSP_AUDIO_SPECTRUM_NUM_FREQUENCY_BINS];
  float     fltAudioPowerSpectrum[CWDSP_AUDIO_SPECTRUM_NUM_FREQUENCY_BINS];  // power densities, not complex, range 0.0 .. 1.0, not logarithmized
  float     fltAudioSpectrumBinWidth_Hz;
  float     fltAudioSpectrumFrameRate;  // "sample rate", for plotting and for timing in the decoder itself
  long      i32AudioSpectrumUpdateCounter;  // <- polled by the GUI to check for a screen update
        // The above audio-spectrum-related members only exist ONCE.
        // The Audio-CW-Decoder itself is "instantiable",
        // just in case Icom is clever enough to add a *real* broadband IF output
        // to their new models, not the bandwidth-amputated "11 kHz" IF output
        // of an IC-7300 that off the same -narrow- bandwidth as the currently
        // selected *CW FILTER IN THE RIG ITSELF*.
        // Until then, there's only a single instance of the T_AudioCwDecoder :
# define CWDSP_NUM_AUDIO_CW_DECODERS 1
  T_AudioCwDecoder AudioCwDecoder[CWDSP_NUM_AUDIO_CW_DECODERS];
                   // '--> instance data with e.g. iCwDecoderState,
                   //      fltCwDecoderCenterFrequency, fltCwDecoderNoisePower,
                   //      fltCwDecoderSignalPower, fltCwDecoderKeyingSignal, ..

# define CWDSP_PLOTTER_SAMPLE_FIFO_SIZE 512 // <- a power of two, to use bitwise AND instead of MODULO for the circular index wrap / uC firmware
  // Rough guesstimate of the required CWDSP_PLOTTER_SAMPLE_FIFO_SIZE :
  //  * "timing scope" in the RCW keyer GUI uses TWO MILLISECONDS per horizontal step (pixel)
  //  * "audio spectra" are emitted every *HALF* FFT window length, i.e.
  //     0.5 * CWDSP_AUDIO_SPECTRUM_FFT_LENGTH / 8 kHz = 0.5 * 256 / 8 kHz = 16 milliseconds
  //     -> need at least ( 2 / 16 ) * maximum_scope_width_in_pixels = (2/16)*4096 = 512 sample points.
  T_CwDSP_PlotterSample sPlotterSampleFifo[CWDSP_PLOTTER_SAMPLE_FIFO_SIZE];
  int iPlotterSampleFifoHeadIndex; // index of the MOST RECENT ("newest") entry
  int iPlotterSampleFifoUsage;     // number of elements in the sPlotterSampleFifo[] (maxes out at CWDSP_PLOTTER_SAMPLE_FIFO_SIZE)
  double dblPlotterSamplingInterval_s; // interval between two samples in sPlotterSampleFifo[], IN SECONDS PER SAMPLE, for timestamp calculations


  int iCwDecoderSourceFifoTail;  // FIFO tail index for this extra "reader" of one of the 8-kHz-sample-FIFOs
  int  nSpectraForSpeedTest;


  // Stuff for debugging / benchmarking / discovery of performance bottlenecks:
  int   iSpeedTestPeaks_us[8]; // microseconds spent in each thread loop for the following operations:
#  define DSP_SPEEDTEST_READ_AUDIO_INPUT  0 // time spent reading a block of samples from the INPUT audio device
#  define DSP_SPEEDTEST_READ_AUDIO_OUTPUT 1 // time spent writing a block of samples to the OUTPUT audio device
#  define DSP_SPEEDTEST_DOWNSAMLE_INPUT   2 // time spent to decimate / downsample the INPUT (from e.g. 48 to 8 kS/sec)
#  define DSP_SPEEDTEST_UPSAMPLE_OUTPUT   3 // time spent to interpolate / upsample the OUTPUT (from e.g. 8 to 48 kS/sec)
#  define DSP_SPEEDTEST_AUDIO_COMPRESS    4 // time to COMPRESS a block of sample (for ouput to the network)
#  define DSP_SPEEDTEST_AUDIO_DECOMPRESS  5 // time to DECOMPRESS a block of sample (for input from the network)
#  define DSP_SPEEDTEST_DECODER_SPECTRUM  6 // time to calculate an AUDIO SPECTRUM for the CW decoder or display
  long  i32SpeedTestSums_us[8];   // SUM of microseconds used for the above operations
  long  i32SpeedTestCounts_us[8]; // counters to convert the above SUMS into AVERAGES, in CwKeyer_GetSpeedTestResults()

  // Windows-specific stuff ...
  HANDLE hThread;    // Not a FILE- but a THREAD-HANDLE, and in THIS case, "NULL" means invalid.
  DWORD  dwThreadId; // the joys of Win32 API programming.. not just a "thread handle" but also a "thread ID"
  int iThreadStatus; // not really windows-specific but one of the following:
#   define DSP_THREAD_STATUS_NOT_CREATED 0
#   define DSP_THREAD_STATUS_LAUNCHED    1
#   define DSP_THREAD_STATUS_RUNNING     2 // <- only set ONCE(!), immediately when entering the thread function
#   define DSP_THREAD_STATUS_TERMINATE   3 // <- just a KLUDGE to "request POLITE termination", but takes a few dozen milliseconds to actually TERMINATE !
#   define DSP_THREAD_STATUS_TERMINATED -1 // <- set by the worker thread if it TERMINATED ITSELF
  DWORD dwThreadLoops, dwThreadErrors, dw8ThreadIntervals_us[8];

# if( SWI_USE_CRITICAL_SECTIONS_FOR_AUDIO_IO )
  CRITICAL_SECTION csAudioIO; // Win32 object for mutual exclusion, here:
                     // to protect the possibly NOT-THREAD-SAFE audio driver
                     // from being called from e.g. thread B
                     //     while still working for thread A ;)
                     // Because not only DSPThread() may occupy the device
                     // but also KeyerThread() -> CwDSP_UpdateSidetone() ,
                     // the time between CwDSP_EnterCriticalSection_AudioIO()
                     //  and CwDSP_LeaveCriticalSection_AudioIO() must be kept
                     //  as low as possible !
# endif
  DWORD dwInitMagic; // == DSP_INIT_MAGIC when initialized
                     // (don't assume the content of T_CwDSP is nicely filled
                     //  with zeros when the application calls CwDSP_InitInstanceWithDefaults(),
                     //  and don't assume the application will call that function only ONCE)
#  define DSP_INIT_MAGIC 0x31415926  // not magic but .. a PI !

} T_CwDSP; // -> in the remote CW KEYER, there is ONE instance of this: CwKeyer_DSP .


//---------------------------------------------------------------------------
// Function prototypes  (API)
//---------------------------------------------------------------------------
#ifndef CPROT   // for peaceful co-existence of C and C++ ...
#ifdef __cplusplus
 #define CPROT extern "C"
#else
 #define CPROT
#endif  // not "cplusplus" ?
#endif // ndef CPROT ?

CPROT void CwDSP_InitDecimatorChain( void );

CPROT void CwDSP_InitInstanceWithDefaults( T_CwDSP *pCwDSP );
CPROT void CwDSP_Start( T_CwDSP *pCwDSP );
CPROT void CwDSP_Stop( T_CwDSP *pCwDSP );

CPROT void CwDSP_SwitchOutputState( T_CwDSP *pCwDSP, int iNewOutputState );
CPROT void CwDSP_UpdateSidetone( T_CwDSP *pCwDSP, BOOL fKeyDown );

CPROT void CwDSP_InitSampleFifo( T_CwDSP_Fifo *pFifo, double dblSecondsPerSample );
CPROT int  CwDSP_GetNumSamplesInFifoForTailIndex( T_CwDSP_Fifo *pFifo, int iTail );
CPROT void CwDSP_ReadFromFifo( T_CwDSP_Fifo *pFifo, int *piTail, float *pfltDestBuffer, int nSamples, double *pdblTimestamp_s );
CPROT void CwDSP_WriteToFifo( T_CwDSP_Fifo *pFifo, float *pfltSourceBuffer, int nSamples, double dblTimestamp_s );
CPROT void CwDSP_ProcessSampleFromReceivedAudioStream( T_CwDSP *pCwDSP, float fltSample, double dblTimestamp_s );

CPROT void CwDSP_ShortToFloat( short *pi16In, float *pfltOut, int nSamples, float fltFactor );
CPROT void CwDSP_FloatToShort( float *pfltIn, short *pi16Out, int nSamples, float fltFactor );

CPROT void CwDSP_UpdateSpeedTestResult(  T_CwDSP *pCwDSP, int iTestItem, int nMicroseconds );
CPROT int  CwDSP_GetSpeedTestAverage_us( T_CwDSP *pCwDSP, int iTestItem );
CPROT void CwDSP_ResetSpeedTestResults(  T_CwDSP *pCwDSP );

CPROT void CwDSP_StartAudioCwDecoder( T_CwDSP *pCwDSP );
CPROT void CwDSP_SetDotTimeForAudioCWDecoder( T_CwDSP *pCwDSP, int iNewDotTime_ms );
CPROT void CwDSP_WriteToCwPatternDecoderFifo( T_AudioCwDecoder *pDecoder, WORD wCwPattern );
CPROT WORD CwDSP_ReadFromCwPatternDecoderFifo( T_AudioCwDecoder *pDecoder, int *piTailIndex );
CPROT void CwDSP_WriteToPlotterFifo( T_CwDSP *pCwDSP, double dblTimestamp_s, WORD wCwPattern );
CPROT T_CwDSP_PlotterSample* CwDSP_ReadFromPlotterFifo( T_CwDSP *pCwDSP, double dblTimestamp_s );
CPROT double CwDSP_GetTimestampOfNewestSampleInPlotterFifo( T_CwDSP *pCwDSP );
CPROT double CwDSP_GetTimestampOfOldestSampleInPlotterFifo( T_CwDSP *pCwDSP );


#endif // ndef  _CW_DSP_H ?

