//---------------------------------------------------------------------------
// File: C:\cbproj\Remote_CW_Keyer\CircularFifo.h
// Date: 2024-02-20
// Author: Wolfgang Buescher (DL4YHF)
// Purpose: Simple, classic, lock-free, thread-safe CIRCULAR FIFO
//          with ONE WRITER (in thread A)
//          and MULTIPLE READERS (in threads B,C,D,..)
//          Used in various firmware projects, but since 2024
//          also in the 'Remote CW Keyer' (initially a windows
//          application), to pass data received from, or sent to,
//          a serial port from one thread to another
//          WITHOUT THE NEED FOR THREAD SYNCHRONISATION.
//---------------------------------------------------------------------------

#ifndef  CIRCULAR_FIFO_INCLUDED
# define CIRCULAR_FIFO_INCLUDED

#include "switches.h"  // project specific compiler switches ("options"),
                       // must be included before anything else !
#include "yhf_type.h"  // classic types like BYTE, WORD, DWORD, BOOL, ..

#ifndef  SWI_CIRCULAR_FIFO_SIZE      // if not defined in "switches.h" ..
# define SWI_CIRCULAR_FIFO_SIZE 256  // .. define this configuration parameter HERE
#endif // ndef SWI_CIRCULAR_FIFO_SIZE

typedef struct t_CircularFifoHead // replacement for T_CFIFO.iHeadIndex : T_CFIFO.head.index (along with a timestamp)
{
  int    index;           // Replacement for e.g. T_CFIFO.iHeadIndex : T_CFIFO.head.index .
                          // The FIFO-"writer" updates this field FIRST, before all other members.
  int    index2;          // Copy of 'index'. Must contain the same as 'index' if the struct is consistent.
                          // The FIFO-"writer" updates this field LAST, after all other members.
  double dblTimestamp_s;  // precise timestamp in seconds, comparable with e.g. dsound_wrapper.h : DSW_ReadHighResTimestamp_s()

} T_CFIFO_Head;


typedef struct t_CircularFifo
{
  int iTailIndex, iSize;   // head- and tail index step by BYTES, not SAMPLES ..
  T_CFIFO_Head head; // Head-index (instead of T_FIFO.iHeadIndex use T_CFIFO.head.index)
                     //  and timestamp for that head-index.
                     // Can be retrieved from other threads in an almost "undisturbed" way,
                     // if the writer manages to set the struct members within a few microseconds
                     //  - without the need for costly thread synchronisations, critical sections, semaphores, etc.
  double dblSecondsPerSample; // sampling interval in seconds(!) for timestamp adjustment in CFIFO_Read()
  int    nBytesPerSample;     // <- also required only for the timestamp adjustment in CFIFO_Read()
  int    nOverflows;          // <- should always remain zero; maxes out at 32767. Mainly for DEBUGGING.
  DWORD  dwTotalBytesWritten; // <- also just a service for software tests (shown e.g. on the 'Debug' tab)
                     // (counts the number of bytes that someone TRIED TO write into the FIFO,
                     //  including bytes that could not be 'accepted' because the FIFO was full)

  BYTE b[SWI_CIRCULAR_FIFO_SIZE]; // the buffer itself must be at the
             // end of the T_CFIFO struct for reasons explained somewhere else.


} T_CFIFO;

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
// If, inside the same application, a few FIFOs with larger sizes
// are required, wrap the T_CFIFO in a larger struct as follows:
// > typedef struct
// > { T_CFIFO fifo;
// > # define LARGE_FIFO_SIZE 16384
// > BYTE bExtraBytes[ LARGER_FIFO_SIZE - SWI_CIRCULAR_FIFO_SIZE ];
// > } T_LARGE_CFIFO
// T_LARGE_CFIFO large_fifo;
//   ...
// CFIFO_Init( &large_fifo.fifo );
// large_fifo.fifo.iSize = LARGE_FIFO_SIZE;
//   ...
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


//----------------------------------------------------------------------------
// Prototypes (callable from C and C++)
//----------------------------------------------------------------------------

#ifndef CPROT  // For peaceful co-existence of "C" and "C++" ....
# ifdef __cplusplus
#   define CPROT extern "C"
# else
#   define CPROT
# endif  // ! "cplusplus"
#endif  // ndef CPROT


CPROT void CFIFO_Init( T_CFIFO *pFifo, int iQueueSizeInBytes, int nBytesPerSample, double dblSecondsPerSample );
CPROT int  CFIFO_Write(T_CFIFO *pFifo, BYTE *pbSource, int nBytes, double dblTimestamp_s );
CPROT void CFIFO_WriteForMultipleReaders( T_CFIFO *pFifo, BYTE *pbSource, int nBytes );
CPROT int  CFIFO_Read( T_CFIFO *pFifo, BYTE *pbDest, int nBytesMax, double *pdblTimestamp_s );
CPROT void CFIFO_ReadHeadIndexAndTimestamp( T_CFIFO *pFifo, T_CFIFO_Head *pHead);
CPROT int  CFIFO_GetNumBytesReadable( T_CFIFO *pFifo );
CPROT int  CFIFO_GetNumSamplesReadable( T_CFIFO *pFifo );
CPROT int  CFIFO_GetNumBytesWriteable( T_CFIFO *pFifo );

  // Functions for "additional readers" of a T_FIFO (they cannot use pFifo->iTailIndex) :
CPROT int  CFIFO_GetNumBytesReadableForTailIndex( T_CFIFO *pFifo, int iTailIndex );
CPROT int  CFIFO_ReadBytesForTailIndex( T_CFIFO *pFifo, BYTE *pbDest, int nBytesMax, int *piTailIndex );


#endif // ndef CIRCULAR_FIFO_INCLUDED ?


/* EOF < CircularFifo.h > */

