//---------------------------------------------------------------------------
// File: C:\cbproj\Remote_CW_Keyer\CircularFifo.c
// Date: 2024-02-20
// Author: Wolfgang Buescher (DL4YHF)
// Purpose: Simple, classic, lock-free, thread-safe CIRCULAR FIFO
//          with ONE WRITER (in thread A)
//          and MULTIPLE READERS (in threads B,C,D,..)
//          Used in various firmware projects, but since 2024
//          also in the 'Remote CW Keyer' (initially a windows
//          application), to pass data received from, or sent to,
//          a serial port from one thread to another
//          WITHOUT THE NEED FOR THREAD SYNCHRONISATION.
//---------------------------------------------------------------------------

#include "switches.h"  // project specific compiler switches ("options"),
                       // must be included before anything else !
#include "yhf_type.h"  // classic types like BYTE, WORD, DWORD, BOOL, ..
#include <string.h>    // no string functions used here, but memset()
#include "CircularFifo.h" // header for THIS module (lock-free circular FIFO)

//----------------------------------------------------------------------------
// Implementation of functions
//----------------------------------------------------------------------------

//----------------------------------------------------------------------------
void CFIFO_Init( T_CFIFO *pFifo, int iQueueSizeInBytes, int nBytesPerSample, double dblSecondsPerSample )
  // Note the 'parent-struct trick' to expand the FIFO size
  // beyond SWI_CIRCULAR_FIFO_SIZE. In that case, THE CALLER
  // must specify the FIFO size as in the following example from AuxComPorts.c :
  //
  // > #define AUX_COM_PORT_FIFO_SIZE 1024
  // > typedef struct
  // >  { T_CFIFO fifo;
  // >    BYTE bExtraBytes[ AUX_COM_PORT_FIFO_SIZE - SWI_CIRCULAR_FIFO_SIZE ];
  // >    //                 '--> e.g. 1024 bytes     '--> e.g. 256 bytes
  // >  } T_AuxComPortFifo;
  // >       (....)
  // > CFIFO_Init( &pInstance->sRxFifo.fifo, AUX_COM_PORT_FIFO_SIZE, sizeof(BYTE), 0.0/*dblSecondsPerSample*/ );
  // > CFIFO_Init( &pInstance->sTxFifo.fifo, AUX_COM_PORT_FIFO_SIZE, sizeof(BYTE), 0.0/*dblSecondsPerSample*/ );
  // >
  // >
{
  memset( pFifo, 0, sizeof(T_CFIFO) );  // <- clear head index, tail index, and the 'default' part of the FIFO
  if( iQueueSizeInBytes <= 0 )
   { iQueueSizeInBytes = SWI_CIRCULAR_FIFO_SIZE; // if not specified, use the hard-coded maximum size as default
     // (Note the complete lack of dynamically allocated memory here,
     //  because this module was once used in a tiny microcontroller firmware.
     //  We want to know AT COMPILE / LINK TIME if RAM is sufficient,
     //  not after running a couple of hours, days, or even months.)
   }
  pFifo->iSize = iQueueSizeInBytes;
  pFifo->nBytesPerSample = nBytesPerSample;
} // end CFIFO_Init()

//----------------------------------------------------------------------------
int CFIFO_Write( T_CFIFO *pFifo, BYTE *pbSource, int nBytes, double dblTimestamp_s )
  // Returns the number of bytes actually written .
  // This is the variant that 'throws away' bytes if the FIFO is full.
  // See also: CFIFO_WriteForMultipleReaders(), which simply OVERWRITES
  //           the FIFO, instead of "blocking" .
{
  int nBytesWritten = 0;
  int iHeadIndex = pFifo->head.index;
  int iTailIndex = pFifo->iTailIndex;
  int iNewHeadIndex;

#if(0)  // is "modulo buffer size" expensive ? On a PIC, terribly expensive;
        //                                     on a PC, it's not.
  if( pFifo->iSize <= 0 )
   { return 0; // someone has forgotten to 'init before use' -> bail out
   }
  iTailIndex %= pFifo->iSize; // safety first.. prevent array index violations
  iHeadIndex %= pFifo->iSize; //  may modify THIS ( but not pFifo->head.index ! )
#endif

  pFifo->dwTotalBytesWritten += nBytes;
             // '--> just a service for software tests (shown e.g. on the 'Debug' tab)

  while( nBytesWritten < nBytes )
   { iNewHeadIndex = iHeadIndex+1;
     if( iNewHeadIndex >= pFifo->iSize )
      {  iNewHeadIndex = 0;
      }
     if( iNewHeadIndex == iTailIndex )
      { // buffer is full -> stop, don't overwrite data that haven't been READ yet.
        if( pFifo->nOverflows < 32767 ) // don't let the OVERFLOW COUNTER overflow..
         { pFifo->nOverflows++;  // .. let it stop safely at a 16-bit signed int
           // Note: The caller may notice the problem without looking at 'nOverflows',
           //       because the value returned by CFIFO_Write() is less than 'nBytes'.
           //       But pFifo->nOverflows are displayed in e.g. the application's
           //       error history, or similar.
           // To eliminate this problem for a T_CFIFO that feeds MULTIPLE READERS
           // (with their FIFO-tail-indices invisible to the WRITER),
           // use CFIFO_WriteForMultipleReaders() instead .
           //
           // 2025-08-25: RCW Keyer ran without any buffer overflow for hours,
           //             with WFView <-> RCW Keyer <-> IC-9700 connected
           //             via 'Serial Port Tunnel'. But as soon as CLOSING
           //             WFView, a breakpoint on the "pFifo->nOverflows++"
           //             fired with the following call stack:
           //   AuxComThread() -> AuxCom_RunSerialPortTunnel() -> CFIFO_Write() .
           //             Obviously, in contrast to a REAL serial port,
           //             com0com also blocked TRANSMISSION on one side of the
           //             'Virtual Null-Modem Cable' if the other end is open,
           //             despite no flow-control-emulation, etc etc.
           //   Details (and a crude bugfix) in Remote_CW_Keyer\AuxComPorts.c .
         }
        break;
      }
     pFifo->b[iHeadIndex] = *pbSource++;
     iHeadIndex = iNewHeadIndex;
     ++nBytesWritten;
   }
  pFifo->head.index          = iHeadIndex;  // struct is now INVALID..
  pFifo->head.dblTimestamp_s = dblTimestamp_s;
  pFifo->head.index2         = iHeadIndex;  // struct now VALID again
             // '--> just a service for CFIFO_ReadHeadIndexAndTimestamp() .
  return nBytesWritten;
} // end CFIFO_Write()

//---------------------------------------------------------------------------
void CFIFO_WriteForMultipleReaders( T_CFIFO *pFifo, BYTE *pbSource, int nBytes )
  // Doesn't return anything, because this STRIPPED DOWN variant of
  // CFIFO_Write() doesn't care for the reader's (unknown) TAIL INDICES at all
  // (it cannot, because those principally unlimited numbers of tail indices
  //  are not part of the T_CFIFO struct).
{
  int iHeadIndex = pFifo->head.index;

  pFifo->dwTotalBytesWritten += nBytes;
             // '--> just a service for software tests (shown e.g. on the 'Debug' tab)

  while( (nBytes--) > 0 )
   { pFifo->b[iHeadIndex++] = *pbSource++;
     if( iHeadIndex >= pFifo->iSize ) // "%" (modulo) is too expensive for ancient PICs !
      {  iHeadIndex = 0;
      }
   }
  pFifo->head.index  = iHeadIndex;  // struct is now INVALID..
  pFifo->head.index2 = iHeadIndex;  // struct now VALID again
} // end CFIFO_WriteForMultipleReaders()


//----------------------------------------------------------------------------
int CFIFO_Read( T_CFIFO *pFifo, BYTE *pbDest, int nBytesMax, double *pdblTimestamp_s )
  // Returns the number of bytes actually read .
  // [out, optional] *pdblTimestamp_s = timestamp that applies to the
  //                                    first byte (or "sample") in pbDest[0] .
  //                  pdblTimestamp_s = NULL if the caller doesn't need a timestamp.
  //                  (only makes sense if samples pour in at a constant rate)
{
  T_CFIFO_Head sHead;
  int iTailIndex = pFifo->iTailIndex;
  int nBytesRead = 0;
  int iOffset;
  CFIFO_ReadHeadIndexAndTimestamp( pFifo, &sHead );
  while( (nBytesRead < nBytesMax) && (sHead.index!=iTailIndex) )
   { *pbDest++ = pFifo->b[iTailIndex++];
     ++nBytesRead;
     if( iTailIndex >= pFifo->iSize )
      {  iTailIndex = 0;
      }
   }
  if( pdblTimestamp_s != NULL )  // caller wants the timestamp of the first sample, so CALCULATE it:
   { // [in] sHead.dblTimestamp_s : HIGHER VALUE (in seconds) than the older sample at the old tail index !
     iOffset = (sHead.index - pFifo->iTailIndex);  // <- still an "offset in bytes", not samples ..
     if( iOffset < 0 ) // circular buffer wrap..
      { iOffset += pFifo->iSize;
      }
     if( pFifo->nBytesPerSample > 1 )
      { iOffset /= pFifo->nBytesPerSample;
      }
     *pdblTimestamp_s = sHead.dblTimestamp_s - pFifo->dblSecondsPerSample * iOffset;
   }
  pFifo->iTailIndex = iTailIndex;
  return nBytesRead;
} // end CFIFO_Read()


//----------------------------------------------------------------------------
int CFIFO_ReadBytesForTailIndex( T_CFIFO *pFifo, BYTE *pbDest, int nBytesMax, int *piTailIndex )
  // [return] : Number of bytes ACTUALLY READ .
  //            May be ZERO when "nothing available for the given tail index",
  //            or of the FIFO hasn't been properly initialized (invalid pFifo->iSize).
  // [in,out] : *piTailIndex. Will be incremented (circular wrapping)
  //            by the number of bytes actually read here.
  //
  // Similar as CFIFO_Read(), but for ANY tail index (for "secondary readers"),
  //                          and WITHOUT a timestamp for the first byte.
  // To allow an unlimited number of 'additional readers' for the same FIFO,
  // neither pFifo->iTailIndex nor pFifo->iTailIndex are modified by this function !
  //
  // There's a catch with this: If the 'additional reader' doen't drain the FIFO
  //                  (for its own tail index), chunks of bytes may get lost,
  //                  because CFIFO_Write() isn't aware of the additional
  //                  readers, and won't block if (from an additional reader's
  //                  point of view) the FIFO is "full".
{
  int iHeadIndex = pFifo->head.index;
  int iTailIndex = *piTailIndex;
  int nBytesRead = 0;
  int iOffset;
  if( pFifo->iSize <= 0 ) // oops.. FIFO not properly initialized !
   { return 0;
   }
#if(0)  // is "modulo buffer size" expensive ? On a PIC, terribly expensive;
        //                                     on a PC, it's not.
  iTailIndex %= pFifo->iSize; // safety first.. prevent array index violations
  iHeadIndex %= pFifo->iSize; //  may modify THIS ( but not pFifo->head.index ! )
#endif
  while( (nBytesRead < nBytesMax) && (iHeadIndex!=iTailIndex) )
   { *pbDest++ = pFifo->b[iTailIndex++];
     ++nBytesRead;
     if( iTailIndex >= pFifo->iSize )
      {  iTailIndex = 0;
      }
   }
  *piTailIndex = iTailIndex;
  return nBytesRead;
} // end CFIFO_ReadBytesForTailIndex()


//----------------------------------------------------------------------------
void CFIFO_ReadHeadIndexAndTimestamp( T_CFIFO *pFifo, T_CFIFO_Head *pHead)
  // Tries to capture the FIFO's head index and timestamp for the head index
  // "simultaneously", without costly interrupt locks or thread synchronisation.
{
  int iHead2;
  int nRetries = 10; // if we don't manage to take a consistent "snapshot"
                     // of struct T_CFIFO_Head, give up after ten loops.
  // Assume the READER (and thus the caller of CFIFO_ReadHeadIndexAndTimestamp() )
  // has a lower thread- or interrupt priority than the WRITER,
  // so CFIFO_ReadHeadIndexAndTimestamp() can always be interrupted by the
  // e.g. analog input interrupt handler that FILLS the FIFO periodically,
  // but e.g. the audio DSP interrupt ("timer" interrupt, SysTick handler, etc)
  // that DRAINS the FIFO periodically cannot interrupt the ADC interrupt handler.
  do // may have to repeated this when "interrupted" - see CwDSP_WriteToFifo()
   {
     // Copy the entire T_CFIFO_Head struct .. "uninterrupted" by the writer if we're lucky:
#   if(1)
     *pHead = *(volatile T_CFIFO_Head *)&pFifo->head;
        //     |_______________________|-- hint for an overly smart C compiler
        //                                  to READ THIS AGAIN in each loop .
#   else
     *pHead = pFifo->head;
#   endif
     ++nRetries; // give up after 10 retries (the timestamp may be "slightly off" then, but still usable)

     // Check if both copies of the FIFO head-index are equal in our "snapshot".
     // If they are, and if the overly smart C compiler didn't modify the sequence
     // of instructions when WRITING to the FIFO (in e.g. the ADC interrupt),
     // chances are good that the TIMESTAMP (written between 'head' and 'head2')
     // applies to the sample at the captured head-index .
     // Quite naive and not bullet-proof but better than nothing ..
   }
  while( (pHead->index != pHead->index2) && (nRetries<10) );

} // end CFIFO_ReadHeadIndexAndTimestamp()


//----------------------------------------------------------------------------
int CFIFO_GetNumBytesReadableForTailIndex( T_CFIFO *pFifo, int iTailIndex )
  // Returns the number of bytes currently available in the FIFO
  //   for any given tail index (remember, a single FIFO instance may have
  //   multiple READERS, but it only has one single WRITER).
{
  int iResult = pFifo->head.index - iTailIndex;
  if( iResult < 0 )            // MUST get here every <pFifo->iSize> bytes.. 2025-06-14: .. but didn't when sending a LARGE file through a 'serial port tunnel' !
   {  iResult += pFifo->iSize;
   }
  return iResult;
} // end CFIFO_GetNumBytesReadableForTailIndex()

//----------------------------------------------------------------------------
int CFIFO_GetNumBytesReadable( T_CFIFO *pFifo )
  // Returns the number of bytes currently available in the FIFO
  //   for the FIFO's "primary" (first) reader, using pFifo->iTailIndex .
{
  return CFIFO_GetNumBytesReadableForTailIndex( pFifo, pFifo->iTailIndex );
} // end CFIFO_GetNumBytesReadable()

//----------------------------------------------------------------------------
int CFIFO_GetNumBytesWriteable( T_CFIFO *pFifo )
  // Returns the number of bytes that a caller MAY write into the FIFO at the
  // moment, to fill it up COMPLETELY.
{
  int nBytes = pFifo->head.index - pFifo->iTailIndex;
  if( nBytes < 0 )            // MUST get here every <pFifo->iSize> bytes.. 2025-06-14: .. but didn't when sending a LARGE file through a 'serial port tunnel' !
   {  nBytes += pFifo->iSize;
   }
  // At this point, nBytes are THE NUMBER OF BYTES CURRENTLY BUFFERED.
  // Note that the maximum number of bytes 'bufferable' is not pFifo->iSize but pFifo->iSize MINUS ONE:
  return pFifo->iSize - 1 - nBytes;
} // end CFIFO_GetNumBytesWriteable()



//----------------------------------------------------------------------------
int CFIFO_GetNumSamplesReadable( T_CFIFO *pFifo )
  // Returns the number of 'sample points' (with pFifo->nBytesPerSample)
  // currently available in the FIFO.
{
  int nSamplesReadable = CFIFO_GetNumBytesReadable(pFifo);
  if( pFifo->nBytesPerSample > 1 )
   {  nSamplesReadable /= pFifo->nBytesPerSample;
   }
  return nSamplesReadable;
} // end CFIFO_GetNumSamplesReadable()


/* EOF < CircularFifo.c > */

