//---------------------------------------------------------------------------
// File: C:\cbproj\Remote_CW_Keyer\Chatbox.cpp
// Date: 2025-09-16
// Author: Wolfgang Buescher (DL4YHF)
// Implements a simple 'text chat' feature to coordinate the use
//            of the remote radio between users and sysop .
//
// Contains a part(!) of the Remote CW Keyer's graphic user interface,
//          with as few dependencies on the VCL-based 'main form' as possible.
//
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
//   Routing principle (between RCWK Server and remote clients)
//   ------------------------------------------------------------
//
//  ,----------------------------------------------------,
//  | Remote CW Keyer (RCWK) Server :                    |
//  | MyCwNet.Client[] (array of type T_CwNetClient)     |
//  |  |- [0] = CWNET_LOCAL_CLIENT                       |
//  |  |- [1] = CWNET_FIRST_REMOTE_CLIENT, with ...      |
//  |  |         |- sz80UserName (only for log-in)       |
//  |  |         |- sz80Callsign (also "for routing") <->|<--------,
//  |  |- [2] = second remotely connected client         |         |
//  |  |         |- sz80UserName                         |         |
//  |  |         |- sz80Callsign <---------------------->|<---,    |
//  |  '- [CWNET_MAX_CLIENTS] (fixed maximum in CwNet.h) |    |    |
//  |            |- sz80UserName                         |    |    |
//  |            |- sz80Callsign                         |    |    |
//  '----------------------------------------------------'    |    |
//         /|\    ,---------------,                           |    |
//          '---->| "Chat screen" | on the remote radio site  |    |
//                '---------------'                           |    |
//                                                            |    |
//   ,------------------,                                     |    |
//   |  RCWK Client "x" |   Single TCP/IP connection "x"      |    |
//   |  Call = "DL4YHF" |<------------------------------------'    |
//   '------------------'  for audio, keying, rig control, chat.   |
//         /|\    ,---------------,                                |
//          '---->| "Chat screen" | in DL4YHF's shack              |
//                '---------------'                                |
//                                                                 |
//   ,------------------,                                          |
//   |  RCWK Client "y" |   Another single TCP/IP connection "y"   |
//   |  Call = "DL4YDU" |<-----------------------------------------'
//   '------------------'  for audio, keying, rig control, chat.
//         /|\    ,---------------,
//          '---->| "Chat screen" | in DL4YDU's shack
//                '---------------'
//
//  Example: DL4YHF (client "x") sends a message to DL4YDU (client "y") :
//    1. In DL4YHF's RCWK client, the operator enters the following
//       in the 'chatbox' (text editor) :  "@DL4YDU: Hallo OM",
//       then presses ENTER .
//       Chatbox_OnKey() recognizes the ENTER KEY, classifies the string
//       as a "text to send" (not a "command line"), and passes the
//       entire line to Chatbox_SendStringToReceivers() .
//       The CLIENT doesn't care about the '@'+receiver's callsign, because
//       only the SERVER knows how to read that client.
//       Chatbox_SendStringToReceivers( pszText = "@DL4YDU: Hallo OM" )
//        -> CwNet_WriteStringToChatTxFifo( .., pszTxData="@DL4YDU: Hallo OM" ) .
//       A few hundred milliseconds, a TCP fragment with that message
//       arrives at the remote server.
//    2. On the RCWK Server (2) :
//       Chatbox_ProcessDataInCharRxFifos() drains ALL 'Chat receive FIFOs',
//       type T_CwNetChatFifo, including the one in
//       MyCwNet.Client[x] with the matching SENDER'S callsign,  and
//       MyCwNet.Client[y] with the matching RECEIVER'S callsign .
//       So the SERVER(!) retransmits the packet received on connection "x"
//                        on connection "y".
//       The SERVER(!) also displays what's going on on its own screen:
//        > DL4YHF -> DL4YDU: Hello OM
//    3. A few hundred milliseconds later, the message arrives on client "y".
//       On RCWK Client "y", Chatbox_ProcessDataInCharRxFifos() drains
//       its own 'Chat receive FIFO' (only ONE, because it's a CLIENT, not a SERVER),
//       and also displays the message including sender- and receiver callsign:
//        > DL4YHF -> DL4YDU: Hello OM
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


#include "switches.h"   // project specific compiler switches ("options"),
                        // must be included before anything else !
#include "yhf_type.h"   // classic types like BYTE, WORD, DWORD, BOOL, ..
#include <windows.h>    // Must be included BEFORE vcl.h for some strange reason.
                        // Contains stuff like WAVEINCAPS, WAVEOUTCAPS, etc^255 .
#include <vcl.h>        // Borland's stoneage Visual Component Library
#include <buttons.hpp>  // TBitBtn       a la Borland VCL (but why in an extra header?)
#include <ComCtrls.hpp> // TTabSheet     a la Borland VCL ...
#include <CheckLst.hpp> // TCheckListBox a la Borland VCL ....
#include <Grids.hpp>    // TStringGrid   a la Borland VCL .....
#include <string.h>
#include <stdio.h>      // not using "standard I/O" here, but e.g. sprintf()
#include <math.h>
#pragma hdrstop

#include "Utilities.h"    // stuff like UTL_iWindowsVersion, UTL_iAppInstance, ShowError(), etc
#include "YHF_Dialogs.h"  // API for a few common dialogs, e.g. YHF_RunStringEditDialog()
#include "YHF_Help.h"     // DL4YHF's HTML-based replacement for the defunct *.hlp system
#include "YHF_WinStuff.h" // Win32 API functions, helpers for 'Rich Edit' controls, etc
#include "HelpIDs.h"      // application specific help topic identifiers
#include "Translator.h"   // APPL_TranslateAllForms() [only works with Borland VCL]
#include "Translations.h" // tranlation table (static, built-in string table w/o whistles and bells)

#if( SWI_USE_DSOUND ) // use "Direct Sound" / dsound_wrapper.c ?
# include "dsound_wrapper.h"
#endif // SWI_USE_DSOUND ?

#if( SWI_USE_WAVE_AUDIO || SWI_USE_MIDI )
# include <mmsystem.h> // maybe MIDI is fast enough to start / stope playing notes ?
#endif // SWI_USE_WAVE_AUDIO or SWI_USE_MIDI ?

#include "StringLib.h"
#include "Elbug.h"   // old 'Elbug' functions, converted from PIC-assembler to "C"
#include "CwGen.h"   // CW generator (converts text to Morse code)
#include "CwDSP.h"   // CW-'Digital Signal Processor' / sidetone generator
#include "CwNet.h"   // Socket-based 'Client or Server' for the Remote CW Keyer. Tied to a "Rig Control" instance.
#include "CwKeyer.h" // prototypes for the 'worker thread' that stitches all together
#include "Inet_Tools.h" // Base64 encoding/decoding, SHA1 calculation, etc..
#if(SWI_USE_HTTP_SERVER) // build an application with integrated HTTP server ?
# include "HttpServer.h" // formerly simple HTTP server (turned into a monster)
#endif // SWI_USE_HTTP_SERVER ?

#include "SpecDisp.h"  // SpecDisp_UpdateSpectrum(), ..Waterfall(), FreqScale(),..
#include "Keyer_GUI.h" // subroutines formerly in Keyer_Main.cpp (now only CALLED from there)
#include "FreqList.h"  // import 'frequencies of interest' from e.g. the EiBi frequency list

#if( SWI_NUM_AUX_COM_PORTS > 0 ) // compile with support for 'Auxiliary / Additional COM ports' / Winkeyer emulation ?
# include "AuxComPorts.h" // structs and API functions for the "Auxiliary" (later: "Additional") COM ports
#endif // SWI_NUM_AUX_COM_PORTS ?


extern T_CwNet MyCwNet; // <- not a shiny C++ class, but a single instance representing the "CW Network". Owns a "Rig Control" instance, too.


//----------------------------------------------------------------------------
// Global variables for hardcore debugging / post-mortem crash analysis.
//        Useful when the debugger's call stack shows nothing but garbage,
//        e.g. after an exception at the infamous address 0xFEEEFEEE .
//        ( C++Builder's debugger could still inspect SIMPLE GLOBAL
//          VARIABLES after most kinds of exceptions, but of course nothing
//          stack-based because the CPU- or task-stack was usually trashed.)
//----------------------------------------------------------------------------
#if(SWI_HARDCORE_DEBUGGING) // (1) = hardcore-debugging, (0)=normal compilation
 int Chatbox_iLastSourceLine = 0; // WATCH THIS after crashing with e.g. "0xFEEEFEEE"  ...
# define HERE_I_AM()  Chatbox_iLastSourceLine=__LINE__
#else
# define HERE_I_AM()      /* rien */
#endif // SWI_HARDCORE_DEBUGGING ?


//---------------------------------------------------------------------------
// Functions ( no shiny VCL class methods ! )
//---------------------------------------------------------------------------



//---------------------------------------------------------------------------
BOOL Chatbox_SendStringToReceivers( char *pszText )
{
  BOOL fResult = FALSE;
  int iClient, iLen;
  T_CwNetClient *pClient = &MyCwNet.Client[CWNET_LOCAL_CLIENT_INDEX];
  // The above (pClient) is used when THIS INSTANCE runs as CLIENT.
  // If this RCW-Keyer-instance current run as SERVER, the *PLURAL* form
  // deserves its name, because the same message text may have to be sent
  // to MULTIPLE remote client :
  if( MyCwNet.cfg.iFunctionality == CWNET_FUNC_SERVER )
   { for( iClient=CWNET_FIRST_REMOTE_CLIENT_INDEX; iClient<=/*!*/CWNET_MAX_CLIENTS; ++iClient )
      { pClient = &MyCwNet.Client[iClient];
        if( (pClient->iClientState == CWNET_CLIENT_STATE_LOGIN_CFMD)  // "log-in confirmed, identified a 'known user'"
          ||(pClient->iClientState == CWNET_CLIENT_STATE_LOGIN_HTTP)) // "log-in confirmed, a 'known user', but HE USES HTTP (a web browser)
         { // Send the message to THIS remote client if his callsign matches
           // the string after the '@', or if there is no '@' at all (="send to ALL"):
           iLen = strlen( pClient->sz80Callsign );
           if(  (pszText[0] != '@') || ( SL_strnicmp( pszText+1, pClient->sz80Callsign, iLen )==0) )
            { fResult |= CwNet_WriteStringToChatTxFifo( &MyCwNet, pClient, pszText );
            }
         }
      }
   }
  else  // this RCW-Keyer-instance runs as a CLIENT, not as SERVER:
   { fResult = CwNet_WriteStringToChatTxFifo( &MyCwNet, pClient, pszText );
   }

  return fResult;
} // end Chatbox_SendStringToReceivers()


//---------------------------------------------------------------------------
BOOL Chatbox_OnKey( TRichEdit *pRichEdit, char cKey )
  // Called from the KeyerGUI (e.g. Keyer_Main.cpp)
  // when the operator presses a key with an ASCII equivalent (including ENTER)
  // with the keyboard focus on the Rich Edit control for the 'Chatbox' on the 'TRX'
  // tab.
  //
  // Return value:  TRUE  when the key has been completely processed HERE,
  //                      and we don't want it to be inserted into the editor text
  //                      (for example, to prevent ENTER / '\r' to insert a new line),
  //                FALSE otherwise .
{
  HWND  hWndEditor = pRichEdit->Handle;  // for Win32 API functions, we need this WINDOW HANDLE
  DWORD dwLine,dwSelStart,dwSelEnd;
  int   iLength, iUnifiedPN;
  char  sz255Temp[256];
  const char *cpSrc;
  BOOL  fResult = FALSE;     // assume we don't intercept the key here
  BOOL  fModified;
  T_RigCtrlInstance   *pRC = &MyCwNet.RigControl;
  T_RigCtrl_ParamInfo *pPI;

  switch( cKey )
   { case '\r': // carriage return (ENTER key)
        // Determine the SELECTION START and the SELECTION END, using our wrapper for the Win32 API:
        RichEdit_GetTextSelection( hWndEditor, &dwSelStart, &dwSelEnd );
                   // Note: dwSelStart equals dwSelEnd when there is nothing SELECTED
                   //       but just the usual TEXT CURSOR !

        // Convert the SELECTION START (which is a CHARACTER index)
        //  into the TEXT LINE NUMBER from the current 'selection' :
        dwLine = RichEdit_GetTextLineIndexFromCharIndex( hWndEditor, dwSelStart );

        // To "interpret" the content of a line of text, we need it as a plain old C-string:
        iLength = RichEdit_GetTextLineAsCString( hWndEditor, dwLine, sz255Temp, sizeof(sz255Temp)-1);

        // Check the begin of the text line for special tokens:
        cpSrc = sz255Temp;
        if( SL_SkipString( &cpSrc, "@set ") ) // line is a COMMAND to set something,
         { // e.g. something like "@set AudioVolume = 16 %", etc ...
           fResult = TRUE;  // don't break this line into pieces by pressing ENTER,
         } // end if < "@set " >
        else if( SL_SkipString( &cpSrc, "@ann ") ) // ANNOUNCEMENT to all users, stored permanently in the REMOTE SERVER INSTANCE,
         { // and shown to all currently connected users, but also to future connections(!)
           fResult = TRUE;
         } // end if < "@set " >
        else // none of the above tokens at the begin of a line, so ...
         { // guess the text line shall just be passed on to other users:
           Chatbox_SendStringToReceivers( sz255Temp );
           fResult = TRUE;
         }
        break;  // end case < key = carriage return (ENTER key) >
     default:  // allow typing any other (ASCII-) key into the editor
        break;
   } // end switch( cKey )

  (void)iLength; // ... assigned a value that is never used .. so what ?
  (void)pRC;

  return fResult;
} // end Chatbox_OnKey()



/* EOF < C:\cbproj\Remote_CW_Keyer\Chatbox.cpp > */


