//---------------------------------------------------------------------------
// File: C:\cbproj\Remote_CW_Keyer\AuxCom_Winkeyer.c
// Date: 2023-12-20
// Author: Wolfgang Buescher (DL4YHF)
// Purpose: Implements a 'Winkeyer Emulator' and a 'Winkeyer Host'
//          residing in DL4YHF's 'Remote CW Keyer', short : 'RCW',
//          communicating with the outside world through
//          any of the "Additional (ex:Auxiliary) COM Ports" .
//  Not to be confused with RCW's ELECTRONIC KEYER THREAD, which uses a
//  *different* port (also a "COM port") - see Remote_CW_Keyer\KeyerThread.c .
//  THIS MODULE (Remote_CW_Keyer/AuxCom_Winkeyer.c) mainly implements
//  the 'Winkeyer' protocol, as far as required for e.g. N1MM Logger+ .
//---------------------------------------------------------------------------

#include "switches.h"  // project specific compiler switches ("options"),
                       // must be included before anything else !
#include "yhf_type.h"  // classic types like BYTE, WORD, DWORD, BOOL, ..

#include <windows.h>  // try NOT to use any Borland-specific stuff here (no VCL)
#include <string.h>
#include <stdio.h>    // no "standard I/O" but e.g. sprintf() used here

#if( SWI_USE_DSOUND ) // use "Direct Sound" / dsound_wrapper.c ?
# include "dsound_wrapper.h"
#endif // SWI_USE_DSOUND ?


#include "Utilities.h" // stuff like UTL_iWindowsVersion, UTL_iAppInstance, ShowError(), etc
#include "Timers.h"    // high-resolution 'current time'-function, T_TIM_Stopwatch(), etc.
#include "Debouncer.h" // 'debouncer' for paddle inputs (T_Debouncer in T_ElbugInstance)
#include "Elbug.h"     // old 'Elbug' functions, converted from PIC-assembler to "C"
#include "CwGen.h"     // CW generator (converts text to Morse code)
#include "CwDSP.h"     // CW-'Digital Signal Processor' / sidetone generator
#include "CwNet.h"     // Socket-based 'Client or Server' for the Remote CW Keyer
#include "StraightKeyDecoder.h" // Morse decoder for input from a STRAIGHT KEY
#include "CwKeyer.h"   // CW-keyer ("from paddle contacts to Morse code") .
#include "AuxComPorts.h" // structs and API functions for the "Auxiliary" (later: "Additional") COM ports
#include "Keyer_GUI_no_VCL.h" // <- formerly Keyer_GUI.h, but THIS part has no VCL dependencies, thus "OK for pure C"


/* DETAILS ...
   This implementation is mainly based on the publicly available manual
   for the original K1EL 'CW Keyer IC for Windows', Winkeyer2 v22,
   locally saved at
   file:///C:/datasheets/Rig_Manuals/Morsetasten_und_Keyer/WinkeyUSBman.pdf .
   Quoted from there (to reference parts of this text further down in the C code):


Theory of Operation
-------------------
This section will describe how the Winkeyer2 works. As shown in Figure 1,
the host PC is connected to Winkeyer2 over a serial COM port, which can be
a USB port supporting virtual COM. Winkeyer2 is a slave to the PC in that it
receives commands and data from the PC and acts upon them. The PC can send
commands while Winkeyer2 is sending Morse allowing dynamic configuration changes.
Winkeyer2 will communicate back to the host for four reasons:
1) Inform the host of a status change in Winkeyer2.
2) Inform the host of a speed pot or pushbutton change (WK2 mode only).
3) Respond to a request for information from the host.
4) Echo back morse in ASCII as its being sent from either the serial port or the paddles.

There are two types of serial input from the host to Winkeyer2: Command and Data.
Commands modify Winkeyer2s operation in some way, for example changing
operating speed, pausing transmission, or asking for status.
Data can be letters, numbers, or prosigns that are to be sent in Morse.
Commands and data are processed differently in Winkeyer2.
Data is put into a serial buffer that allows the host to send data ahead of
the Morse being sent. The size of this buffer is 128 characters and is a FIFO
which is an acronym for First In First Out. This means that characters
are taken out in the order they were put in. Since there can be a considerable
delay from host input to Morse output, commands bypass the input FIFO
and are acted upon immediately.
This allows changes to be made while sending is underway.

                   ,--------------, Data  ,---------------,
   Serial Input -->| Input Parser |--->---| 128 byte FIFO |--*--> Processing
                   '--------------'       '---------------' /|\     ..
                          |         Command Bypass           |
                          '----------------------------------'

    Figure 3  Data and Command Flow inside Winkeyer2

Since there are times when you don't want commands to take effect immediately,
Winkeyer2 allows commands to be buffered. This means that the command is placed
in the serial buffer and wont be acted on until it comes out of the buffer.
An example of the use of a buffered command would be to send two words at
two different speeds, the first at 15 WPM and the second at 20 WPM.
By placing a buffered speed command between the words the speed will not be
changed until the first word is completely sent. Not all, but many of
the immediate commands can be entered as buffered commands.
      [Note, WB: The "Change Speed Buffered" must only occupy a SINGLE BYTE
                 in the 128 byte FIFO because otherwise the "buffer pointer
                 commands" will fail ! Only for the INPUT PARSER, the command
                 consists of TWO BYTES, 0x1C <NN> .  See musings in
                 Bugs/2025_08_16_N1MM and RCWK with winkeyer emulation.txt,
                 and the actual implementation of WK_CMD_SET_MORSE_SPEED_BUFF .
      ]

   (....)

Host Mode Command Descriptions
------------------------------
This section documents the commands that are sent from the host to Winkeyer2
over the serial interface.
Commands are special hex codes that are sent to Winkeyer2. These codes range
from 0x01 through 0x1F.
In this document a hex value will be presented in angle brackets, for example <02>.
Some commands have one or more parameters sent immediately after the
command code is sent, this will be documented as the command code
followed by a value: <02><nn> where nn is a single byte binary value.
The notation [c] represents a single ASCII character sent to Winkeyer2
as a single serial byte.

Immediate Commands
============================

These commands are processed as soon as they are received, they bypass the input buffer.

Immediate Admin Commands
----------------------------

 * Admin <00><nn> nn is a value from 0 to 8
After power-up the host interface is closed, serial status, echo, or pot change
data will be not be sent to the host.
The only commands WK2 will accept are Admin commands. Admin commands are received,
processed and any return status or data will be sent back immediately.
Admin commands calibrate the interface, reset WK, obtain debug information
and open the interface. With the exception of the Amin:Close command,
all Admin commands should only be issued while the host interface is closed.
Following are descriptions of the Admin commands:

0: Calibrate
   This is an historical command preserved for WK1 compatibility.
   It is no longer required for the more accurate PIC the WK2 is implemented on.
   You can issue the command and it will not cause ill effects,
   but it is not processed by WK2. The command syntax is:
      <00><00> pause 100 mSec <FF>

1: Reset
   Resets the Winkeyer2 processor to the power up state.
   Do not send this as part of the initialization sequence.
   Only send this if you want to do a cold reboot of WK2.

2: Host Open
   Upon power-up, Winkeyer2 initializes with the host mode turned off.
   To enable host mode, the PC host must issue the admin:open command.
   Upon open, Winkeyer2 will respond by sending the revision code back to the host.
   The host must wait for this return code before any other commands or data
   can be sent to Winkeyer2. Upon open, WK1 mode is set.

3: Host Close
   Use this command to turn off the host interface.
   Winkeyer2 will return to standby mode after this command is issued.
   Standby settings will be restored.

4: Echo Test
   Used to test the serial interface. The next character sent to Winkeyer2
   after this command will be echoed back to the host.

5: Paddle A2D
   Historical command not supported in WK2, always returns 0.

6: Speed A2D
   Historical command not supported in WK2, always returns 0.

7: Get Values Returns all of the internal setup parameters.
   They are sent back in the same order as issued by the Load Defaults command.
   Again, this command is a diagnostic aid. Only issue this command
   when host interface is closed.

8: Reserved
   K1EL Debug use only
9: Get Cal
   Historical command not supported in WK2, always returns 0.

10: Set WK1 Mode
    Disables pushbutton reporting
11: Set WK2 Mode
    Enables pushbutton reporting, alternate WK status mode is selected.

12: Dump EEPROM
    Dumps all 256 bytes of WK2s internal EEPROM

13: Load EEPROM
    Download all 256 bytes of WK2s internal EEPROM.
14: Send Standalone Message
    Command WK2 to send one of its internal messages. The command syntax is:
       <00><14><msg number> where number is 1 through 6
*/
#define WK_CMD_ADMIN            0x00 // leading byte for all "ADMIN" commands
#define WK_CMD_ADMIN_CALIB         0 // "Calibrate"
#define WK_CMD_ADMIN_RESET         1 // "Reset"
#define WK_CMD_ADMIN_HOST_OPEN     2 // "Host Open"
#define WK_CMD_ADMIN_HOST_CLOSE    3 // "Host Close"
#define WK_CMD_ADMIN_ECHO_TEST     4 // "Echo Test"
#define WK_CMD_ADMIN_PADDLE_A2D    5 // "Paddle A2D"
#define WK_CMD_ADMIN_SPEED_A2D     6 // "Speed A2D"
#define WK_CMD_ADMIN_GET_VALUES    7 // "Get Values"
#define WK_CMD_ADMIN_K1EL_DEBUG    8 // "Reserved, K1EL Debug use only"
#define WK_CMD_ADMIN_GET_CAL       9 // "Get Cal" (later renamed to "Get FW Major Rev" for Winkeyer 3)
#define WK_CMD_ADMIN_SET_WK1_MODE 10 // "Set WK1 Mode. Disables pushbutton reporting". NOTE THAT <nn> MEANS DECIMAL, NOT HEX ! Grrrrr !
#define WK_CMD_ADMIN_SET_WK2_MODE 11   // "Set WK2 Mode. Enables pushbutton reporting".
                                       // (Funny that even after we identified ourselves as "Winkeyer 2",
                                       //  N1MM Logger+ sometimes sent an 'admin command type 0x11 in HEX)
#define WK_CMD_ADMIN_DUMP_EEPROM  12   // "Dump EEPROM. Dumps all 256 bytes of WK2's internal EEPROM".
#define WK_CMD_ADMIN_LOAD_EEPROM  13   // "Load EEPROM. Download all 256 bytes of WK2's internal EEPROM".
#define WK_CMD_ADMIN_SEND_MESSAGE 14   // "Send Standalone Message", later just "Send Message".  <00><14><msg number>
#define WK_CMD_ADMIN_LOAD_X1MODE  15   // "Load mode extension register 1".  Only for Winkeyer 2+3 (and up) ?
                                       //  > "Note that the bit assignments of this register
                                       //  >  are different between WK2 and WK3 mode".
                                       // (indeed, they are VERY different.
                                       //  See 'case WK_CMD_ADMIN_LOAD_XMODE' in the C code)
        /* "End of ADMIN Commands" for Winkeyer 2 */
#define WK_CMD_ADMIN_FIRMWARE_UPLD 16  // "This command initiates an image upload". (we'll never support this)
#define WK_CMD_ADMIN_SET_LOW_BAUD  17  // "Set Low Baud: Change serial comm. Baud Rate to 1200 (default)"
#define WK_CMD_ADMIN_SET_HIGH_BAUD 18  // "Set High Baud: Change serial comm. Baud Rate to 9600"
#define WK_CMD_ADMIN_SET_RTTY_REGS 19  // "Set RTTY Mode Registers". <0x00><19d><P1><P2>. "WK3.1 only".
#define WK_CMD_ADMIN_SET_WK3_MODE  20  // "Set WK3 Mode: Enables WinKeyer 3 functions; expanded X1MODE and additional X2MODE register"
#define WK_CMD_ADMIN_GET_V_SUPPLY  21  // "Read Back Vcc: Return WK IC power supply voltage."
#define WK_CMD_ADMIN_LOAD_X2MODE   22  // "Load mode extension register 2".  Only for Winkeyer 3 (and up) ?
#define WK_CMD_ADMIN_GET_FW_MINOR  23  // "Get FW Minor Rev: Returns the minor firmware revision, 03 for version 31.03"
#define WK_CMD_ADMIN_GET_IC_TYPE   24  // "Returns the WK IC type: 0x1 for SMT, 0x0 for DIP"
#define WK_CMD_ADMIN_SET_SIDETN_VOL 25  // "Set Sidetone Volume <00><24><n> where n =0x1 for low and n=0x4 for high"
        /* "End of ADMIN Commands" for Winkeyer 3 */


/*
Immediate Host Mode Commands
----------------------------
 * Sidetone Control <01><nn>     [YHF: WK_CMD_SET_SIDETONE_IMMEDIATE]
    nn is a value determined from Table 1 & 2
    In WK2 sidetone is always enabled and pin 8 functions as the sidetone
    square wave output. The following tables define the format of value nn.
    Note that these frequencies are slightly different than WK1 (...)

    The most significant bit of the frequency byte controls the paddle only
    sidetone feature. In WK2 you can choose to only use sidetone for paddle entry
    and mute it for CW sourced fron the host port. This is called
        Paddle Only Sidetone
    and is selected by setting the MSB of the sidetone control value.

  * Set WPM Speed    <02><nn>         [YHF: WK_CMD_SET_MORSE_SPEED_IMM]
    nn is in the range of 5-99 WPM
        Example: <02><12> set 18 WPM
    Set a new Morse operating speed, this command takes effect as soon as
    Winkeyer2 receives it. If speed is set to zero then Winkeyer2 will take
    its speed setting directly from the speed pot., this is the reset default

  * Set Weighting    <03><nn>         [YHF: WK_CMD_SET_KEY_WEIGHTING]
    nn is in the range of 10-90%
       Example: <03><32> for weight=50           (...)

  * Set PTT Lead/Tail   <04><nn1><nn2>     [YHF: WK_CMD_SET_PTT_DELAYS]
    nn1 sets lead in time, nn2 sets tail time
    both values range 0 to 250 in 10 mSecs steps
       Example: <04><01><A0>
                lead-in = 10 mSecs, tail = 1.6 sec
    Winkeyer2 provides a transmitter PTT output for each key output that can be
    used to switch a transmitter or linear amplifier over to transmit mode
    in advance of actual CW keying. You have control over the time delay
    between when PTT is asserted and when CW keying will start, this is lead-in.
    You also have control over how long the transmitter will stay in transmit
    after keying has stopped; this is the tail delay.     (...)

  * Setup Speed Pot <05><nn1><nn2><nn3>   [YHF: WK_CMD_SET_SPEED_POT_RANGE]
            nn1 = MIN,
            nn2 = RANGE,
            nn3 = dont care
    This command sets the limits for the speed pot        (...)

  * Set Pause State <06><nn>              [YHF: WK_CMD_PAUSE_MORSE_IMM]
            nn = 01 pause, value = 00 unpause
    When Winkeyer2 is paused, sending will stop immediately and will not resume
    until an unpause state is set.
    The current character being sent in Morse will be completed before pause.
    Note that the Clear Buffer command will cancel pause..

  * Get Speed Pot <07>      (no parameter)   [YHF: WK_CMD_GET_SPEED_POT_IMM]
    Request to Winkeyer2 to return current speed pot setting.
    This command will cause a speed pot command request to be queued
    in Winkeyer2 and it will be acted on as soon as possible.  (...)

  * Backspace <08>          (no parameters)  [YHF: WK_CMD_BACKUP_INPUT_POINTER]
    Backup the input buffer pointer by one character. This command is only
    meaningful if there is something in the serial input buffer,
    otherwise it is ignored.

  * Set PinConfig <09><nn>                   [YHF: WK_CMD_SET_OUTPUT_PIN_CONFIG]
        low nibble determines how output pins are mapped
        high nibble controls ultimatic mode and hang time      (...)

  * Clear Buffer <0A>       (no parameters)  [YHF: WK_CMD_CLEAR_INPUT_BUFFER]
    This command will reset the input buffer pointers to an empty state.
    It is a general clear also in that Tune and Pause are also cancelled by this command.
    Clear Buffer can be sent at any time to abort a message, abort a command,
    or to clear the serial buffer. It will cancel any Morse character in progress
    immediately ending it in midstream if necessary.

  * Key Immediate <0B><nn>    nn = 01 keydown, n = 00 keyup  [YHF: WK_CMD_DIRECT_CTL_KEY_OUTPUT]
    Use this command to implement a tune function. Once asserted, key down
    will remain in effect until either a key immediate with a zero value
    is received or the internal tune watchdog timer expires.
    The tune timer is hard coded to a value of 100 seconds and cannot be disabled.
    The key down can be aborted either by the paddles or by a clear buffer command.


   * Set HSCW        <0C><nn> nn = the lpm rate divided by 100  (...) [YHF: WK_CMD_SET_HSCW_SPEED_IMM]

   * Set Farns WPM   <0D><nn> nn is in the range of 10-99 [YHF: WK_CMD_SET_FARNSWORTH_IMM]
          Example: <0D><12> for Farnsworth=18 WPM
          Farnsworth spacing is useful for CW practice          (...)

   * Set Winkeyer2 Mode    <0E><nn> nn = Mode bit field in binary [YHF: WK_CMD_SET_WK2_MODE_BYTE_IMM]
     Example: <0E><13> set bits 4,1,0, clear the rest
     The operational mode of Winkeyer2 can be modified by directly altering
     its internal mode register. This register is made up of eight bits
     which each control a particular mode.                      (...)

   * Load Defaults      <0F><value list>   [YHF: WK_CMD_LOAD_DEFAULTS_IMM]
                             value list is a set of 15 binary values
     This command is provided to allow all the operating parameters
     to be loaded into Winkeyer2 in one block transfer          (...)

   * Set 1 st Extension  <10><nn>          [YHF: WK_CMD_SET_FIRST_ELEM_CORR]
                              nn is in the range of (0 to 250)  1 mSecs
     Example: <04><80> sets lead in to 80 mSecs
     Winkeyer2 addresses a problem often encountered when keying older
     transceivers that have a slow break-in response. Due to a slow
     receive to transmit changeover time, the first dit or dah of a letter
     sequence can be chopped and reduced in length. Adding a fixed amount
     to the first element of a sequence can compensate for this  (...)

   * Set Key Comp     <11><nn> nn is in the range of (0 to 250)  1 mSecs [YHF: WK_CMD_SET_KEYING_COMPENS]
     Example: <11><B4> sets key comp to 180 mSecs
     Keying Compensation allows a fixed amount to be added to the length
     of all dits and dahs. QSK keying on modern transceivers can cause
     shortening of the dit and dah elements which is especially noticeable
     at high speeds.                                             (...)

   * (...)
*/
   //   WK_CMD_ADMIN                 0x00 // leading byte for all "ADMIN" commands, variable number of params
#define WK_CMD_SET_SIDETONE_IMMEDIATE 0x01 // "Set sidetone frequency, IMMEDIATE" : <01><freq>
#define WK_CMD_SET_MORSE_SPEED_IMM   0x02 // "Set Morse sending speed, IMMEDIATE"  <02><WPM>
#define WK_CMD_SET_KEY_WEIGHTING     0x03 // "Set key weighting, IMMEDIATE"      <03><weight>
#define WK_CMD_SET_PTT_DELAYS        0x04 // "Set up PTT delays, IMMEDIATE"      <04><leadin><tail>
#define WK_CMD_SET_SPEED_POT_RANGE   0x05 // "Set up speed pot range, IMMEDIATE" <05><m><wr><pr>
#define WK_CMD_PAUSE_MORSE_IMM       0x06 // "Pause Morse output, IMMEDIATE"     <06><0 or 1>
#define WK_CMD_GET_SPEED_POT_IMM     0x07 // "Request speed pot value, IMMEDIATE" <07> (without parameters but a RESPONSE)
#define WK_CMD_BACKUP_INPUT_POINTER  0x08 // "Backup input pointer (aka BACKSPACE), IMMEDIATE" <08> (without parameters)
#define WK_CMD_SET_OUTPUT_PIN_CONFIG 0x09 // "Set output pin configuration, IMMEDIATE"  <09><config>
#define WK_CMD_CLEAR_INPUT_BUFFER    0x0A // "Clear input buffer, IMMEDIATE" <0A>  (without parameters)
#define WK_CMD_DIRECT_CTL_KEY_OUTPUT 0x0B // "Direct control of key output, IMMEDIATE"  <0B><0 or 1>
#define WK_CMD_SET_HSCW_SPEED_IMM    0x0C // "Set HSCW speed, IMMEDIATE"              <0C><lpm/100>
#define WK_CMD_SET_FARNSWORTH_IMM    0x0D // "Set Farnsworth speed, IMMEDIATE"        <0D><WPM>
#define WK_CMD_SET_WK2_MODE_BYTE_IMM 0x0E // "Load Winkeyer2 mode byte, IMMEDIATE"    <0E><mode>
#define WK_CMD_LOAD_DEFAULTS_IMM     0x0F // "Load Defaults, IMMEDIATE", aka "Download WK state block" <0F><15 values>
#define WK_CMD_SET_FIRST_ELEM_CORR   0x10 // "Setup first element correction, IMMEDIATE"  <10><msec>
#define WK_CMD_SET_KEYING_COMPENS    0x11 // "Set Keying Compensation, IMMEDIATE"    <11><comp>
#define WK_CMD_SET_PADDLE_SENS_IMM   0x12 // "Setup paddle sensitivity, IMMEDIATE"   <12><sens>
#define WK_CMD_NO_OPERATION_IMM      0x13 // "Null Command, NOP, IMMEDIATE"    <13>  (without parameters)
#define WK_CMD_SW_PADDLE_CTRL_IMM    0x14 // "Software Paddle Control, IMMEDIATE" <14><paddle select> 11
#define WK_CMD_GET_STATUS_BYTE_IMM   0x15 // "Request Winkeyer2 status, IMMEDIATE"  <15> (no parameters, single-byte-command)
#define WK_CMD_BUFFER_POINTER        0x16 // "Buffer pointer commands, IMMEDIATE" <16><cmd>....
#define WK_CMD_DIT_DAH_RATIO_IMM     0x17 // "Set Dit/Dah Ratio, IMMEDIATE"       <17><ratio>
        // Note: The next possible command code, 0x18, is a BUFFERED command. See next chapter !


// Buffered Commands
// ===================
// These commands go into the input buffer maintaining their
// positional relationship to data.
//
// * PTT On/Off     <18><nn>    nn = 01 PTT on, n = 00 PTT off
//    This command allows the PTT output to be used for a custom purpose.
//    The command is operational only when sidetone and PTT are diabled
//   (See PINCFG command) PTT can be turned on or off at will and will be
//   unaffected by all other commands including Clear Buffer.
//   Typical applications could be as a power level control, antenna selector,
//   or to turn on a cooling fan. Since this is a buffered command,
//   the on/off will happen at the commands position in the buffer
//   and remain in effect until the next PTT ON/OFF command is encountered.
//   This command will not stall the output buffer.
//
// * Key Buffered           <19><nn>    nn = 0 to 99 seconds
//   Use this command to assert the key output for a specific period of time.
//   Since this is a buffered command, the keydown will begin at the commands
//   position in the buffer and will stall the buffer until the timeout has been
//   satisfied. The keydown can be aborted either by the paddles or by a
//   Clear Buffer command. The maximum allowable key down time is 99 seconds.
//
// * Wait for nn Seconds    <1A><nn>   nn = 0 to 99 seconds
//   This command is used to insert a fixed pause into a message.
//   Since this is a buffered command, the pause will begin at the commands
//   position in the buffer and will stall the buffer until the timeout
//   has been satisfied.
//
// * Merge Letters   <1B>[C][C]      Merge Two Letters into a Prosign .
//   You can build "on the fly" prosigns with this command. Issue the command
//   followed by two letters or numbers and they will be merged together:
//       <1B>[A][R] is sent as AR.
//   Note that nothing will be sent until both letters have been received.
//   Several common prosigns such as AR, SK, BT, And DN are already assigned
//   (see page 13) so you don't have to build these. One application of this
//   feature is to send special European language characters.
//
// * Change Speed Buffered    <1C><nn>    nn is in the range of 5-99 WPM
//     Example: <02><23> set 35 WPM  ( ? ? Where is the 0x1C in that example ? ?)
//   This command places a speed change command into the serial buffer that will
//   be acted upon when it is taken out of the buffer. The current speed in force
//   will be stored and will be reinstated when the buffered speed change
//   is cancelled by a Cancel Speed Change command or any of the following:
//     Unbuffered Speed change, Weight change, Farnsworth change,
//     Ratio change, Compensation change, or Mode change.
//   This command is useful for building messages with embedded speed changes.
//   In this example the first part of the message will be sent at 5 WPM,
//   the second at 25 WPM and the end at whatever the current speed is:
//       <1C><05>VVV DE K1EL <1C><19> VVV DE K1EL<1E><END DE K1EL>
//   (NOTE, DL4YHF: In the 128-byte buffer, this command appears to occupy
//                  only ONE BYTE - because otherwise, the "buffer pointer commands"
//                  failed. See implementation of WK_CMD_SET_MORSE_SPEED_BUFF )
//
// * HSCW Speed Change     <1D><nn> nn = (lpm/100)
//   This command acts the same as the immediate HSCW command. This allows you
//   to insert an HSCW burst in a regular CW message or to put HSCW bursts
//   of two different rates into the same message.
//    ( .. YHF: Should be banned.. but often heard in contests for the RST)
//
// * Cancel Buffered Speed Change     <1E>
//   This command will cancel any buffered speed change command that is in force.
//   The sending speed that was in force before any buffered speed change
//   was encountered will be restored. Several buffered speed changes can be issued
//   within a message but none will alter the original sending speed.
//
// * Buffered NOP                     <1F>
//   This command will occupy a position in the input buffer
//   but will result in no action when it is processed.
//
#define WK_CMD_PTT_ON_OFF_BUFFERED    0x18 // "PTT On/Off, BUFFERED" <18><0 or 1> 0=PTT on, 0=PTT off
#define WK_CMD_TIMED_KEY_DOWN_BUFF    0x19 // "Timed Key Down, BUFFERED"     <19><secs>
#define WK_CMD_WAIT_N_SECONDS_BUFF    0x1A // "Wait for N seconds, BUFFERED" <1A><secs>
#define WK_CMD_MERGE_TWO_CHARS_BUFF   0x1B // "Merge two characters into a prosign, BUFFERED <1B>[c][c]
#define WK_CMD_SET_MORSE_SPEED_BUFF   0x1C // "Change Morse speed in WPM, BUFFERED <1C><WPM>
#define WK_CMD_SET_HSCW_SPEED_BUFF    0x1D // "HSCW Speed Change, BUFFERED"        <1D><lpm/100>
#define WK_CMD_CANCEL_BUFFERED_SPEED_CHANGE 0x1E // "Cancel Buffered Speed Change, BUFFERED" <1E> without parameters
#define WK_CMD_NO_OPERATION_BUFFERED  0x1F // "Buffered NOP"  <1F> without parameters



/*
Unsolicited Status Transmission
================================
Winkeyer2 will send two types of unsolicited status to the host:
speed pot change, and a change in status byte. Whenever the speed pot is moved
its new value will be sent to the host. Likewise whenever there is a change
to the internal status register inside Winkeyer2, a copy of it will be sent
to the host. In WK2 mode the WK status byte is expanded to include an alternate
status byte that returns pushbutton status.
The status byte will be in the same format as previously described in the
"Get Status" command. Likewise the speed pot status will be as described
in the Get Pot command.
Since these bytes can arrive at any time and potentially can be mixed with
echo back bytes, they have identifying tags. If the MSB is set that identifies
the byte as unsolicited, bit 6 then identifies either a speed pot byte
or a status byte. Echo back bytes will always have the MSB=0.
The host can force either of these bytes to be returned by using their respective
"Get Pot" or "Get Status" commands. Due to the parallel task handling nature
of Winkeyer2 a response may not be immediate, there may be another bytes
in the return queue that need to be sent to the host first.
Worst case latency will be 200 milliseconds. It is not advisable for the host
to wait for a response, it is better to handle it as illustrated
by the code fragment shown in an earlier section of this document.

Prosign Key Assignments
========================
Winkeyer2 has mapped several unused character codes to standard prosigns.
Table 5 shows the mappings.
Any additional prosigns can easily be generated using the merge character command.

   ASCII Hex  Prosign           ASCII Hex  Prosign
     "  0x22  RR                  +  0x2B  AR
     #  0x23  EE (null)           -  0x2D  DU
     $  0x24  SX                  /  0x2F  DN
     %  0x25  EE (null)           :  0x3A  KN
     &  0x26  EE (null)           ;  0x3B  AA
       0x27  WG                  <  0x3C  AR
     (  0x28  KN                  =  0x3D  BT
     )  0x29  KK                  >  0x3E  SK
     *  0x2A  EE (null)           @  0x40  AC


WK2 Host Mode Command Table
============================
  .. see macro constants listed further above.
*/


/*** Winkey EEPROM Block ***/
typedef union{   // [Offset] Info
  struct {
   BYTE magic;   // [0x00]    > "Magic value is a constant 0xA5."
   // > "Other parameters match up with values as described in the WK2 manual."
   // The meaning of the next FIFTEEN bytes seem to be compatible with
   // what the "Winkeyer2 IC Interface & Operation Manual 4/15/2008", page 10,
   // specifies under "Load Default" (same 15 bytes as for "Get Values"):
   // > This command is provided to allow all the operating parameters
   // > to be loaded into Winkeyer2 in one block transfer. The values are
   // > binary and must be loaded in order. The values are exactly the same
   // > as those loaded for the individual commands.
   // A-a. The COMMENTS below are copied from the document mentioned above,
   //      "Figure 13 - Default Value List in order of issuance" .
   //      Note the somehow different terms used "for the EEPROM" and in "Load Defaults".
   BYTE modereg; // [0x01] Mode Register (set via command 0x0E, WK_CMD_SET_WK2_MODE_BYTE_IMM ?)
                 //          Contains a bitwise combination specified in
                 //          Winkeyer2 IC Interface & Operation Manual 4/15/2008 Page 9 :
#       define WK_MODE_MASK_CONTEST_SPACING 0x01 // contest spacing reduces the wordspace time by one dit.
                      // > Instead of 7 dits per wordspace, Contest spacing
                      // > selects SIX dits per wordspace.
                      //   (omg... what a big time saver. Not supported here.)
#       define WK_MODE_MASK_AUTO_SPACE   0x02 // > Here is how autospace works:
                      // > If you pause for more than one dit time between a dit or dah
                      // > Winkeyer2 will interpret this as a letter-space
                      // > and will not send the next dit or dah
                      // > until full letter-space time has been met.
                      // > The normal letter-space is 3 dit spaces.
#       define WK_MODE_MASK_SERIAL_ECHO  0x04 // Bit 2 : "Serial Echoback": 1=Enabled, 0=Disabled :
                      // > Echo back is a feature that is included to allow
                      // > a host application to stay exactly in sync
                      // > with Morse letters sent.
                      // > When this mode is enabled all data taken out of the
                      // > serial buffer is sent to the host after it has been
                      // > sent in Morse. This allows the host to reconcile
                      // > differences in timing introduced by Winkeyer2s
                      // > internal 32 byte serial buffer. Note that only
                      // > letters, and not buffered commands with their parameters
                      // > or wordspaces, are echoed back to the host.
                      // Slightly different for Winkeyer 3 (can you spot the incompatibility?) :
                      // > Serial Echo Back tells WK3 to echo each Morse letter
                      // > that originated at the host. It can be used to allow a
                      // > host application to stay exactly in sync with Morse
                      // > letters as they are sent. Each letter is sent to the host
                      // > after it has been sent in Morse.
                      // > This permits the host to track WK3s progress
                      // > in real time. Note that buffered commands and their
                      // > parameters are not echoed back to the host.
#       define WK_MODE_MASK_PADDLE_SWAP  0x08 // Bit 3 : "Paddle Swap": 1=Swap, 0=Normal :
                      // > this is a nice feature to have when right
                      // > and left handed ops want to share the same keyer.
#       define WK_MODE_MASK_KEYER           0x30 // Bits 5+4 define the KEYER MODE :
#       define WK_MODE_MASK_KEYER_IAMBIC_B  0x00 // Bits 5+4 : 0b00 = IAMBIC B
#       define WK_MODE_MASK_KEYER_IAMBIC_A  0x10 // Bits 5+4 : 0b01 = IAMBIC A
#       define WK_MODE_MASK_KEYER_ULTIMATIC 0x20 // Bits 5+4 : 0b10 = Ultimatic
#       define WK_MODE_MASK_KEYER_BUG       0x30 // Bits 5+4 : 0b11 = Bug Mode
#       define WK_MODE_MASK_PADDLE_ECHO  0x40 // Bit 6 : "Paddle Echoback": 1=Enabled, 0=Disabled :
                      // > When this bit is set to one all characters entered on
                      // > the paddles will be echoed back to the host. From the
                      // > host perspective paddle echo and serial echo are the same,
                      // > in either case the letter sent in Morse by Winkeyer2
                      // > is echoed back to the host. The echo occurs after the
                      // > letter has been completely sent. The host can determine
                      // > the source by the sense of the "break-in" status bit.
                      // > If the bit is high when the echoed letter comes in
                      // > then the letters source was from the paddles,
                      // > if break-in is low the source if from the serial port.
#       define WK_MODE_MASK_DISABLE_WDOG 0x80 // Bit 7 : "Disable Paddle Watchdog" : 0=enabled, 1=disabled.
                      // > Winkeyer2 has a paddle watchdog counter that will disable
                      // > the key output after 128 consecutive dits or dahs.
                      // > This is to guard against the paddles being accidentally
                      // > keyed continuously. By default the paddle watchdog
                      // > is on but it can be turned off by setting this mode bit.
   BYTE speed_wpm;    // [0x02] Speed in WPM (ex: "oprrate", too confusing)
   BYTE sidetone;     // [0x03] Sidetone Frequency + Flags (see cmd 0x01, WK_CMD_SET_SIDETONE_IMMEDIATE)
   BYTE weight;       // [0x04] Weight (length of dits/dahs versus gaps)
   BYTE lead_time;    // [0x05] PTT Lead-In Time (1st parameter from cmd 0x04, WK_CMD_SET_PTT_DELAYS)
   BYTE tail_time;    // [0x06] PTT Tail Time    (2nd parameter from cmd 0x04, WK_CMD_SET_PTT_DELAYS)
   BYTE pot_min_wpm;  // [0x07] MinWPM for pot   (1st parameter from cmd 0x05, WK_CMD_SET_SPEED_POT_RANGE)
   BYTE pot_wpm_range;// [0x08] WPM Range for pot (2nd parameter from cmd 0x05, WK_CMD_SET_SPEED_POT_RANGE)
   BYTE xtnd;         // [0x09] "first Extension" (parameter from cmd 0x10, WK_CMD_SET_FIRST_ELEM_CORR)
   BYTE kcomp;        // [0x0a] Key Compensation
   BYTE farnswpm;     // [0x0b] Farnsworth speed (WPM)
   BYTE sampadj;      // [0x0c] Paddle Setpoint (aka "Paddle Switchpoint", aka "Sensitivity" ?)
   BYTE ratio;        // [0x0d] Dit/Dah Ratio (don't use anything but "50 %" for dit/dah = 1/3)
   BYTE pincfg;       // [0x0e] Pin Configuration from cmd 0x09 = WK_CMD_SET_OUTPUT_PIN_CONFIG.
        // For Winkeyer 1 (using a small 8-pin-PIC), a bitwise combination of the following:
#       define WK1_PINCFG_ENABLE_PTT        (1<<0)
#       define WK1_PINCFG_ENABLE_SIDETONE   (1<<1)
#       define WK1_PINCFG_MASK_PTT_HANGTIME (3<<4)  // "word space + 1/2/4/8 dits" ?
#       define WK1_PINCFG_SHIFT_PTT_HANGTIME   (4)
   BYTE k12mode;      // [0x0f] WK2: "Dont care, set to 0".  WK3: additional stuff, see below
   // > "additional bytes that are used by standalone operation only:
     // '--> k12mode is formatted as follows:
     //        Bit 2 : Enable msg brk by paddle (standalone only)
     //        Bit 3 : Select 0 and 9 cut for serial number (standalone only)
   BYTE cmdwpm;       // [0x10] "sets the WPM rate used for paddle entry commands in standalone mode"
   BYTE freeptr;      // [0x11] "points to the first location of free message memory"
     //  '--> When all memory is used, freeptr will be set to zero.
   BYTE msgptr1;      // [0x12] "point to the location of a message in the storage array"
   BYTE msgptr2;      // [0x13] (zero if a message is empty)
   BYTE msgptr3;      // [0x14]
   BYTE msgptr4;      // [0x15]
   BYTE msgptr5;      // [0x16]
   BYTE msgptr6;      // [0x17]
    //   '--> > "If a message pointer is set to zero the message slot is empty.
    //        > Otherwise the location of the message is that value offset
    //        > from the msgs array start. --------,
   BYTE msgs[232];    // [0x18] .. [0xFF]  <-------'      ("MAXMSG"=232)
   //    '--> K1EL: I discourage modifying the message store from a
   //         > host application. The message store is small
   //         > and there are only six (four directly accessed in WKUSB)
   //         > messages. Better to store messages on the PC where you have
   //         > more space and flexibility.
  } s;        // Winkeyer_EEPROM.s accesses the EEPROM as a STRUCTURE,
 BYTE b[256]; // Winkeyer_EEPROM.s accesses the EEPROM as an old-school BYTE ARRAY.

} T_Winkeyer_EEPROM;
static T_Winkeyer_EEPROM Winkeyer_EEPROM; // all THREE(?) 'Aux COM Ports' share this SINGLE WINKEYER EEPROM

// Local options for compilation:
#define FLUSH_RX_HEX_DUMP_BEFORE_SHOWING_DECODED_COMMAND 1

// Internal function prototypes ("forward references" for stuff called before being implemented)
static void WK_Emu_StartCwGenerator( T_CwGen *pCwGen ); // .. if not already running


// Implementation of functions :

//----------------------------------------------------------------------------
static void Winkeyer_EnterByteInDataFIFO( T_AuxComPortInstance *pInstance,
                          BYTE bData )
{
#if(0) // old, simple, reliable, before the introduction of the bizarre "Buffer Pointer Commands",
       // with what K1EL called "buffer append mode" versus "buffer overwrite mode",
       // requiring TWO DIFFERENT "pointers" (byte indices) into b128Fifo[] :
  BYTE bNewHeadIndex = (pInstance->bFifoHead + 1 ) & 127;

  if( bNewHeadIndex != pInstance->bFifoTail ) // only if the FIFO isn't completely full yet:
   { pInstance->b128Fifo[ pInstance->bFifoHead ] = bData;
     pInstance->bFifoHead = bNewHeadIndex;
   }
#else  // drilled up in 2025-10-11 to support the bizarre "Buffer Pointer Commands"
       // (to save contesters / N1MM Logger+ users a few milliseconds
       //  by STARTING TO send the "exchange" (RST " 5nn ") BEFORE the call has
       //  been entered completely).
  // Since 2025-10-11, Winkeyer_EnterByteInDataFIFO() is aware of
  // pInstance->fWKOverwriteMode and pInstance->bFifoOverwritePos . Example:
  //   ______________________________________________                         .
  //  /<---------- "principally valid for TX" ------>\       b128Fifo[]       .
  // ,------------------------------------------------------------------------,
  // |1C|0A|'A|'B|'C|oo|oo|oo|oo|oo|oo|oo|oo|oo|oo|oo| <----- undefined ----> |
  // '--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+---'
  //  00 01 02 03 04 05 06 07 08 09 10 11 12 13 14 15 16 17 18 19 20 21 22 24 <- DECIMAL, zero-based buffer indices
  //                /|\                              /|\                      .
  //                 |                                |                       .
  //     bFifoOverwritePos                        bFifoHead                   .
  //                 \<-- "NULLs", overwriteable ---->/                       .
  //                  \______________________________/                        .
  //   * In "Overwrite"-mode, bData is principally written to b128Fifo[bFifoOverwritePos++]
  //   * In "Append"-mode,    bData is principally written to b128Fifo[bFifoHead++]
  //   * Winkeyer_ReadByteFromDataFIFO() only cares for bFifoHead, not for bFifoOverwritePos
  //   * Thus, if bFifoOverwritePos exceeds bFifoHead, bFifoHead is set to bFifoOverwritePos
  //           (which allows writing into the buffer part with 'undefined' content,
  //            just like the ordinary APPEND-mode, even in buffer-OVERWRITE-mode)
  //   * This not only works for "normal text to send", but also for those
  //     special "buffered commands" like 0x1C <WPM> ("Set CW Speed, BUFFERED").

  BYTE bNewHeadIndex;

  if( pInstance->fWKOverwriteMode ) // the emulated Winkeyer is currently in "buffer OVERWRITE mode" :
   { // In "buffer Overwrite mode", pInstance->b128Fifo[] may already contain some text,
     // and a couple of "added nulls" that have been *appended* via cmd 0x16 0x03 <NumberOfNulls>, e.g.:
     bNewHeadIndex = (pInstance->bFifoOverwritePos + 1 ) & 127;

     if( bNewHeadIndex != pInstance->bFifoTail ) // only if the FIFO isn't completely full yet:
      { pInstance->b128Fifo[ pInstance->bFifoOverwritePos ] = bData;
        pInstance->bFifoOverwritePos = bNewHeadIndex;
        // As explained above, let pInstance->bFifoTail catch up with bFifoOverwritePos (not vice versa):
        if( pInstance->bFifoHead < bNewHeadIndex )
         {  pInstance->bFifoHead = bNewHeadIndex;  // "buffer OVERWRITE mode" acting like "buffer APPEND mode" (special case)
         }
      }
   }
  else // The emulated Winkeyer is currently in "buffer APPEND mode" :
   {   // This is more or less the NORMAL MODE, where text and "buffered commands"
       // are simple APPEND to our thread-safe, lock-free, circular 128-byte FIFO:
     bNewHeadIndex = (pInstance->bFifoHead + 1 ) & 127;

     if( bNewHeadIndex != pInstance->bFifoTail ) // only if the FIFO isn't completely full yet:
      { pInstance->b128Fifo[ pInstance->bFifoHead ] = bData;
        pInstance->bFifoHead = bNewHeadIndex;
        // THERE'S NO NEED TO MODIFY pInstance->bFifoOverwritePos,
        // because Winkeyer_ReadByteFromDataFIFO() only cares for bFifoHead and bFifoTail !
      }
   }
#endif // variant with or without support for the two Winkeyer "buffer modes" (APPEND vs OVERWRITE) ?

} // end Winkeyer_EnterByteInDataFIFO()


//----------------------------------------------------------------------------
static int Winkeyer_WrapDataFifoIndex( int iIncrementedByteIndex ) // .. for pInstance->b128Fifo[], which is a CIRCULAR FIFO
{ return ( iIncrementedByteIndex + 128 ) & 127;
}

//----------------------------------------------------------------------------
static void Winkeyer_ClearInputBuffer( T_AuxComPortInstance *pInstance )
  // Implements Winkeyer command 0x0A = "Clear input buffer, IMMEDIATE" .
  // (That's not the low-level, tiny SERIAL PORT BUFFER but (for Winkeyer 2)
  //  the emulated 128-byte FIFO for MORSE OUTPUT and "buffered commands".)
  //
  // Received THIS command when opening N1MM Logger's "Send CW" window
  //  by pressing CTRL-K in the logger's main window.
  //  Purpose (example): Interrupt/abort a long CQ loop already in the buffer,
  //                     and send something completely different instead.
{ pInstance->bFifoHead = pInstance->bFifoTail = 0;
} // end Winkeyer_ClearInputBuffer()

//----------------------------------------------------------------------------
static void Winkeyer_DeleteLastCharInInputBuffer( T_AuxComPortInstance *pInstance )
  // Implements what K1EL describes as "Backup the input buffer pointer by one character",
  //            aka "BACKSPACE" (we backup files on our harddisks).
  // > "This command is only meaningful if there is something in the
  // >  serial input buffer, otherwise it is ignored."
  // Operates on the simulated 128-byte FIFO in Winkeyer 2 .
{ if( pInstance->bFifoHead != pInstance->bFifoTail ) // 128-byte FIFO not empty ?
   { pInstance->bFifoTail = ( pInstance->bFifoTail - 1 ) & 127;
   }
} // end Winkeyer_ClearInputBuffer()

//----------------------------------------------------------------------------
static void Winkeyer_GetInputBufferUsage_nBytes( T_AuxComPortInstance *pInstance )
{ int nBytesBuffered = (int)pInstance->bFifoHead - (int)pInstance->bFifoTail;
  if( nBytesBuffered < 0 ) // circular wrap between "head" and "tail" ->
   {  nBytesBuffered += C_WINKEYER_INPUT_BUFFER_FIFO_SIZE;
   }
} // end Winkeyer_GetInputBufferUsage_Percent()


//----------------------------------------------------------------------------
static BOOL Winkeyer_ReadByteFromDataFIFO( T_AuxComPortInstance *pInstance,
                          BYTE *pbData )
{ if( pInstance->bFifoHead != pInstance->bFifoTail )
   { *pbData = pInstance->b128Fifo[ pInstance->bFifoTail ];
     pInstance->bFifoTail = (pInstance->bFifoTail+1) & 127;
     return TRUE;
   }
  else
   { return FALSE;
   }
} // end Winkeyer_ReadByteFromDataFIFO()


//----------------------------------------------------------------------------
static void Winkeyer_SendByte( T_AuxComPortInstance *pInstance, BYTE bData,
                               char *pszInfoForDebugLog )
  // Sends a single byte to this instance's serial port
{
  CFIFO_Write( &pInstance->sTxFifo.fifo, &bData, sizeof(BYTE), 0.0/*dblTimestamp_s*/ );
  if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
   { if( pszInfoForDebugLog == NULL )
      {  pszInfoForDebugLog = "(hex)";
      }
     ShowError( ERROR_CLASS_TX_TRAFFIC | SHOW_ERROR_TIMESTAMP, "WK-EMU: Tx %s 0x%02X",
                pszInfoForDebugLog, (unsigned int)bData );
   }
} // end Winkeyer_SendByte()

//----------------------------------------------------------------------------
static void Winkeyer_SendBytes( T_AuxComPortInstance *pInstance, BYTE* pbData, int nBytes,
                               char *pszInfoForDebugLog )
  // Sends multiple bytes to this instance's serial port,
  // with optional output into the "Debug" message list (for troubleshooting).
{ char sz80HexDump[84];
  char *pszDest    = sz80HexDump;
  char *pszEndstop = sz80HexDump+79;
  int  i;
  BYTE bData;

  sz80HexDump[0] = '\0';  // make sure "C" strings are ALWAYS terminated
  for(i=0; i<nBytes; ++i)
   { bData = *(pbData++);
     CFIFO_Write( &pInstance->sTxFifo.fifo, &bData, sizeof(BYTE), 0.0/*dblTimestamp_s*/ );
     if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
      { SL_AppendPrintf(&pszDest,pszEndstop, "%02X ", (unsigned int)bData );
      }
   }
  if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
   { if( pszInfoForDebugLog == NULL )
      {  pszInfoForDebugLog = "(hex)";
      }
     ShowError( ERROR_CLASS_TX_TRAFFIC | SHOW_ERROR_TIMESTAMP, "WK-EMU: Tx[%02d] %s 0x%s",
                (int)nBytes, pszInfoForDebugLog, (unsigned int)bData, sz80HexDump );
   }

} // end Winkeyer_SendBytes()


//----------------------------------------------------------------------------
static void Winkeyer_DumpDataFIFOToLog( T_AuxComPortInstance *pInstance )
  // [in] pInstance->b128Fifo[], bFifoHead, bFifoOverwritePos, fWKOverwriteMode.
  // [out] RCW-Keyer's "traffic log", via ShowError()
{
  char sz128HexDump[128];
  char *pszDest    = sz128HexDump;
  char *pszEndstop = sz128HexDump+sizeof(sz128HexDump)-1;
  BYTE bData;
  BOOL fShowPointer;

  int  i, nBytes   = pInstance->bFifoHead; // <- "initial assumption" (when not in "BUFFER-OVERWRITE"-mode)
  if( pInstance->fWKOverwriteMode )
   { if( pInstance->bFifoOverwritePos > pInstance->bFifoHead ) // have "overwritten" more bytes in the buffer than the normal "append"-position (FIFO HEAD INDEX) ?
      { nBytes = pInstance->bFifoOverwritePos; // .. then show as many bytes as indicated by K1EL's "buffer overwrite position"
      }
   }

  sz128HexDump[0] = '\0';  // make sure "C" strings are ALWAYS terminated
  if( nBytes > 0 )
   { for(i=0; i<=/*!*/nBytes && i<=22; ++i)
      {         // '--> Running through <nBytes+1> to show e.g. "head->" pointing to an EMPTY buffer location,
        //      after CW transmission is finshed. Example (with N1MM sending a contest report, with "multiple nulls" in between):
        // > 13:49:57.1 WK-BUF: 1C 0A 44 4C 34 59 48 46 00 00 00 00 00 00 00 00 20 35 4E 4E 20 head,tail->
        //   ,------------------------------------------|_____________________| \__" 5NN "___/
        //   '--> "Multiple Nulls" that MAY(!) be OVERWRITTEN while already TRANSMITTING (from N1MM Logger+)
        //        if the contest operator can type sufficiently fast ...
        bData = pInstance->b128Fifo[i];
        // Mark the FIFO-head-index and/or the confusing "overwrite position" for K1EL's "Buffer Pointer Commnds".
        // Hypothetic example with pInstance->b128Fifo[i] = i:
        // > WK-BUF: 00 tail->01 02 ovwr->03 04 05 06 07 08 09 0A 0B 0C 0D 0E 0F 10 11 12 13 14
        fShowPointer = FALSE;
        if( pInstance->fWKOverwriteMode && (i==pInstance->bFifoOverwritePos) )
         { SL_AppendString(&pszDest,pszEndstop, "ovwr" );
           fShowPointer = TRUE;
         }
        if( i==pInstance->bFifoHead )
         { if( fShowPointer )
            { SL_AppendChar(&pszDest,pszEndstop, ',' ); // multiple pointer pointing to the same location are separated by COMMA
            }
           SL_AppendString(&pszDest,pszEndstop, "head" );
           fShowPointer = TRUE;
         }
        if( i==pInstance->bFifoTail )
         { if( fShowPointer )
            { SL_AppendChar(&pszDest,pszEndstop, ',' );
            }
           SL_AppendString(&pszDest,pszEndstop, "tail" );
           fShowPointer = TRUE;
         }
        if( fShowPointer )
         { SL_AppendString(&pszDest,pszEndstop, "->" ); // with some imagination, that's a "pointer" in ASCII
         }
        if( i</*!*/nBytes )
         { SL_AppendPrintf(&pszDest,pszEndstop, "%02X ", (unsigned int)bData );
         }
      }
   }
  else // no hex-dump to generate at all ?
   { SL_AppendPrintf(&pszDest,pszEndstop, "empty (FIFO head=%d, tail=%d, ovwr=%d, mode=%s)",
         (int)pInstance->bFifoHead, (int)pInstance->bFifoTail, (int)pInstance->bFifoOverwritePos,
          pInstance->fWKOverwriteMode ? "OVERWRITE" : "APPEND" );
     // Still not sure about the reason for "OVERWRITE" / "APPEND"
     //               versus a more logical "OVERWRITE" / "INSERT".
     // But keep these names, since K1EL uses them for his undocumented "pointer commands" !
   }
  ShowError( ERROR_CLASS_RX_TRAFFIC | SHOW_ERROR_TIMESTAMP, "WK-BUF: %s", sz128HexDump );
} // end Winkeyer_DumpDataFIFOToLog()


//----------------------------------------------------------------------------
static const char *WK_CommandToString(BYTE bCmd) // [in] bCmd: 0x00 .. 0x1F
{
  switch( bCmd )
   { case WK_CMD_ADMIN                 : return "Admin";
     case WK_CMD_SET_SIDETONE_IMMEDIATE: return "SetSidetone";
     case WK_CMD_SET_MORSE_SPEED_IMM   : return "SetCwSpeed";
     case WK_CMD_SET_KEY_WEIGHTING     : return "SetKeyWeight";
     case WK_CMD_SET_PTT_DELAYS        : return "SetPttDelays";
     case WK_CMD_SET_SPEED_POT_RANGE   : return "SetPotRange";
     case WK_CMD_PAUSE_MORSE_IMM       : return "PauseOutImm";
     case WK_CMD_GET_SPEED_POT_IMM     : return "GetSpeedPot";
     case WK_CMD_BACKUP_INPUT_POINTER  : return "Backspace";
     case WK_CMD_SET_OUTPUT_PIN_CONFIG : return "SetPinConfig";
     case WK_CMD_CLEAR_INPUT_BUFFER    : return "ClearInputBuf";
     case WK_CMD_DIRECT_CTL_KEY_OUTPUT : return "DirectOutput";
     case WK_CMD_SET_HSCW_SPEED_IMM    : return "SetHighSpeed";
     case WK_CMD_SET_FARNSWORTH_IMM    : return "SetFarnsworth";
     case WK_CMD_SET_WK2_MODE_BYTE_IMM : return "SetModeByte";
     case WK_CMD_LOAD_DEFAULTS_IMM     : return "LoadDefaults";
     case WK_CMD_SET_FIRST_ELEM_CORR   : return "SetFirstElCorr";
     case WK_CMD_SET_KEYING_COMPENS    : return "SetKeyingComp";
     case WK_CMD_SET_PADDLE_SENS_IMM   : return "SetPaddleSense";
     case WK_CMD_NO_OPERATION_IMM      : return "NoOpImmediate";
     case WK_CMD_SW_PADDLE_CTRL_IMM    : return "SoftwarePaddleCtrl";
     case WK_CMD_GET_STATUS_BYTE_IMM   : return "GetWinkeyStatus";
     case WK_CMD_BUFFER_POINTER        : return "SetBufferPointer";
     case WK_CMD_DIT_DAH_RATIO_IMM     : return "SetDitDahRatio";
     case WK_CMD_PTT_ON_OFF_BUFFERED   : return "SetPttBuffered";  // 0x18 "PTT On/Off, BUFFERED"
     case WK_CMD_TIMED_KEY_DOWN_BUFF   : return "TimedKeyDownBuf"; // 0x19 "Timed Key Down, BUFFERED"
     case WK_CMD_WAIT_N_SECONDS_BUFF   : return "WaitSecondsBuf";  // 0x1A "Wait for N seconds, BUFFERED"
     case WK_CMD_MERGE_TWO_CHARS_BUFF  : return "MergeCharsBuf";   // 0x1B "Merge two characters into a prosign"
     case WK_CMD_SET_MORSE_SPEED_BUFF  : return "SetCwSpeedBuf";   // 0x1C "Buffered Change Morse speed"
     case WK_CMD_SET_HSCW_SPEED_BUFF   : return "SetHighSpeedBuf"; // 0x1D "HSCW Speed Change, buffered"
     case WK_CMD_CANCEL_BUFFERED_SPEED_CHANGE: return "CancelBufSpeed"; // 0x1E "Cancel Buffered Speed Change"
     case WK_CMD_NO_OPERATION_BUFFERED : return "NoOpBuffered";    // 0x1F "Buffered NOP"
     default: return "UNKNOWN_CMD";
   }
} // end WK_CommandToString()

//----------------------------------------------------------------------------
static const char *WK_AdminCommandToString(BYTE bAdminCmd) // [in] bAdminCmd: 0..25 (DECIMAL!)
{
  switch( bAdminCmd )
   { case WK_CMD_ADMIN_CALIB        : return "Calibrate";
     case WK_CMD_ADMIN_RESET        : return "Reset";
     case WK_CMD_ADMIN_HOST_OPEN    : return "HostOpen";
     case WK_CMD_ADMIN_HOST_CLOSE   : return "HostClose";
     case WK_CMD_ADMIN_ECHO_TEST    : return "EchoTest";
     case WK_CMD_ADMIN_PADDLE_A2D   : return "PaddleA2D";
     case WK_CMD_ADMIN_SPEED_A2D    : return "SpeedA2D";
     case WK_CMD_ADMIN_GET_VALUES   : return "GetValues";
     case WK_CMD_ADMIN_K1EL_DEBUG   : return "K1EL_Debug";
     case WK_CMD_ADMIN_GET_CAL      : return "GetCal";
     case WK_CMD_ADMIN_SET_WK1_MODE : return "SetWK1Mode";
     case WK_CMD_ADMIN_SET_WK2_MODE : return "SetWK2Mode";
     case WK_CMD_ADMIN_DUMP_EEPROM  : return "DumpEEPROM";
     case WK_CMD_ADMIN_LOAD_EEPROM  : return "LoadEEPROM";
     case WK_CMD_ADMIN_SEND_MESSAGE : return "SendMessage"; // aka "Send Standalong Message" in older docs
     case WK_CMD_ADMIN_LOAD_X1MODE  : return "LoadX1Mode";
     case WK_CMD_ADMIN_FIRMWARE_UPLD: return "FwUpdate";
     case WK_CMD_ADMIN_SET_LOW_BAUD : return "SetLowBaud";
     case WK_CMD_ADMIN_SET_HIGH_BAUD: return "SetHighBaud";
     case WK_CMD_ADMIN_SET_RTTY_REGS: return "SetRTTYRegs";
     case WK_CMD_ADMIN_SET_WK3_MODE : return "SetWK3Mode";
     case WK_CMD_ADMIN_GET_V_SUPPLY : return "ReadVSupply";
     case WK_CMD_ADMIN_LOAD_X2MODE  : return "LoadX2Mode";
     case WK_CMD_ADMIN_GET_FW_MINOR : return "GetFWMinor";
     case WK_CMD_ADMIN_GET_IC_TYPE  : return "GetICType"; // "Returns the WK IC type: 0x1 for SMT, 0x0 for DIP"
     case WK_CMD_ADMIN_SET_SIDETN_VOL: return "SetSToneVol";
     default: return "UNKNOWN_ADMIN_CMD";
   } // end switch( bAdminCmd )
} // end WK_AdminCommandToString()


//----------------------------------------------------------------------------
int WK_SidetoneCodeToFrequency_Hz( BYTE bWKSidetoneCode )
{ // From the Winkeyer2 IC Interface & Operation Manual 4/15/2008 Page 6:
  // > In WK2 sidetone is always enabled and pin 8 functions as the
  // > sidetone square wave output. The following tables define the
  // > format of value nn. Note that these frequencies are slightly
  // > different than WK1 .
  // >  Bit 7 (MSB) : "Enable Paddle Only Sidetone" when = 1 .
  // >  Bits 6..4   : Unused, set to zero.
  // >  Bits 3..0   : Sidetone frequency N (see table below)
  // >  N   | Frequency || N   | Frequency |
  // >  ----+-----------++-----+-----------+
  // >  0x1 | 4000 Hz   || 0x6 | 666 Hz    |
  // >  0x2 | 2000 Hz   || 0x7 | 571 Hz    |
  // >  0x3 | 1333 Hz   || 0x8 | 500 Hz    |
  // >  0x4 | 1000 Hz   || 0x9 | 444 Hz    |
  // >  0x5 |  800 Hz   || 0xA | 400 Hz    |
  //    ("Table 2 - Sidetone Selection Table" for Winkeyer 2)
  switch( bWKSidetoneCode & 0x0F ) // only bits 3..0 are relevant here :
   { case 0x1: return 4000;
     case 0x2: return 2000;
     case 0x3: return 1333;
     case 0x4: return 1000;
     case 0x5: return  800;
     case 0x6: return  666;
     case 0x7: return  571;
     case 0x8: return  500;
     case 0x9: return  444;
     case 0xA:
     default : return  400;
   }
} // end WK_SidetoneCodeToFrequency_Hz()

//----------------------------------------------------------------------------
void AuxCom_RunWinkeyerHost( T_AuxComPortInstance *pInstance )
  // Called periodically (every few milliseconds) from AuxComThread(),
  // if pInstance->iComPortUsage == RIGCTRL_PORT_USAGE_WINKEYER_HOST .
{
} // end AuxComPorts_RunWinkeyerHost()

//----------------------------------------------------------------------------
void AuxCom_RunWinkeyerEmulator( T_AuxComPortInstance *pInstance )
  // Called periodically (every few milliseconds) from AuxComThread(),
  // if pInstance->iComPortUsage == RIGCTRL_PORT_USAGE_WINKEYER_EMULATOR .
{
  BYTE bRcvdChar, bSendData;
  char sz80[84], sz80Comment[84];
  char *pszToken = "";
  BOOL cmd_finished, pass_to_fifo;
  BOOL fDataFIFOModified = FALSE;
  int i;
  int iErrorClass = ERROR_CLASS_RX_TRAFFIC | SHOW_ERROR_TIMESTAMP; // <- for display on the "Debug" tab (optional)

  // Make sure the simulated Winkeyer EEPROM contains valid data,
  // because some of those fields are used as sources for "read"-commands,
  // or are the target for "write"-commands, just to make the Winkeyer-host
  // in N1MM Logger+ "happy" (instead of trying to initialize the Winkeyer
  // over in over, in an endless loop, when it wasn't happy with a particular
  // setting and the way that our Winkeyer firmware emulator responded):
  if( Winkeyer_EEPROM.s.magic != 0xA5 ) // "Magic value is a constant 0xA5" ..
   { memset( &Winkeyer_EEPROM, 0, sizeof(Winkeyer_EEPROM) );
     Winkeyer_EEPROM.s.magic     = 0xA5; // don't go through this "cold reset" again
     Winkeyer_EEPROM.s.speed_wpm = Elbug_DotTimeInMillisecondsToWordsPerMinute( CwKeyer_Elbug.cfg.iDotTime_ms );
   } // end if < bad "magic value" in the simulated Winkeyer EEPROM > ?

  while( CFIFO_Read( &pInstance->sRxFifo.fifo, &bRcvdChar, sizeof(bRcvdChar), NULL/*no timestamp*/ ) > 0 )
   {
     if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
      { AuxCom_Debug_AppendRcvdCharToHexDump( bRcvdChar ); // .. but not necessarily FLUSH to the display yet
      }

     // Process the next received SINGLE BYTE (bRcvdChar) in a 'streaming parser',
     //  implemented as a STATE MACHINE that started beautifully simple,
     //  then grew into a monster (the usual fate of switch/case-based state machines):
     if( pInstance->iWKImmediateCmdParserState < 0 ) // NOT parsing anything HERE yet, so...
      { // ,---------'
        // '--> While parsing, contains e.g. WK_CMD_ADMIN (0x00), WK_CMD_SET_SIDETONE_IMMEDIATE,
        //      WK_CMD_SET_MORSE_SPEED_IMM, ..., WK_CMD_NO_OPERATION_BUFFERED (0x1F).
        //  ALL commands (IMMEDIATE as well as BUFFERED commands!) must be recognized
        //  here, but only the IMMEDIATE commands will be 'executed' in THIS loop.
        //  Hold your breath; this STREAMING PARSER is ugly but sufficiently simple
        //  to be implemented even on an ancient PIC (when removing the pInstance).
        if( bRcvdChar < 0x20 ) // if not "normal ASCII", it's a Winkeyer command !
         { pInstance->iWKImmediateCmdParserState = (int)bRcvdChar; // kick the PARSER STATE MACHINE alive
           pInstance->iByteOfImmediateCommand = 0;
           pInstance->nBytesOfImmediateCommand= 1; // "initial guess" (possibly more in the switch/case below)
         }
        else // bRcvdChar is NOT a command (neither IMMEDIATE nor BUFFERED) -> streaming parser remains RESET
         { pInstance->nBytesOfImmediateCommand = pInstance->iByteOfImmediateCommand = 0;
         }
      }
     else // already parsing an IMMEDIATE command (or passing through a BUFFERED command), so...
      { ++pInstance->iByteOfImmediateCommand; // just got the next byte BELONGING TO AN ALREADY RECEIVED COMMAND-BYTE
      }
     cmd_finished = (pInstance->iByteOfImmediateCommand >= pInstance->nBytesOfImmediateCommand);
        // '--> that's an assumption, may be CLEARED in the switch/case below..
     pass_to_fifo = FALSE;
        // '--> that's an assumption, may be SET in the switch/case below if it's a "buffered" command,
        //      or a normal ASCII character intended for transmission in CW
     switch( pInstance->iWKImmediateCmdParserState )
      { // Keep it simple.. iWKImmediateParserState is simply the code of the HOST COMMAND,
        // listed in "WinkeyUSBman.pdf" in chapter "WK2 Host Mode Command Table" :
        case WK_CMD_ADMIN: // "Administative Commands, IMMEDIATE"   <00><type> + sometimes more..
           // > While WK2 is attached to a PC com port in the closed state
           // > it will accept ADMIN commands. This allows an application
           // > to set WK2/WK1 mode, upload or download standalone messages
           // > and standalone settings.
           switch( pInstance->iByteOfImmediateCommand )
            { case 0:  // not finished yet, wait for the <type> of the 'Admin Command'.
                 // 'Admin Commands' consist of at least TWO bytes : <00><type>, so :
                 cmd_finished = FALSE;
                 pInstance->nBytesOfImmediateCommand = 2; // <- initial assumption, for MOST but not ALL "ADMIN" commands
                 // (Got here when N1MM Logger+ sent its FIRST BYTE to what it thought was a Winkeyer chip.
                 //  At pInstance->iByteOfImmediateCommand=1 (i.e. the SECOND BYTE),
                 //  bRcvdChar was 0x02 = WK_CMD_ADMIN_HOST_OPEN = "Host Open".)
                 break;
              case 1: // iByteOfImmediateCommand=1 : Now bRcvdChar is the "Type"
                      //  of the "Admin" command (we call it SUB-COMMAND here),
                      //  so we know how many parameter bytes will follow (if any):
                 pInstance->bImmediateSubcommand = bRcvdChar;
                 switch( pInstance->bImmediateSubcommand )
                  { case WK_CMD_ADMIN_CALIB: // "Calibrate" : <00><00> pause 100 ms <FF>
                       if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
                        {
#                        if( FLUSH_RX_HEX_DUMP_BEFORE_SHOWING_DECODED_COMMAND )
                          AuxCom_Debug_FlushRcvdCharHexDump();
#                        endif
                          ShowError( iErrorClass, "WK-EMU: Rcvd Admin:Calibrate" );
                        }
                       pInstance->nBytesOfImmediateCommand = 3; // after "Calibrate", expect a 0xFF (100 ms later)
                       // (N1MM Logger+ sent this AFTER "Host Open",
                       //  and then waited for something else,
                       //  instead of sending e.g. ASCII for a "CQ call".)
                       break;
                    case WK_CMD_ADMIN_RESET: // 0x01 = "Reset"
                       if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
                        {
#                        if( FLUSH_RX_HEX_DUMP_BEFORE_SHOWING_DECODED_COMMAND )
                          AuxCom_Debug_FlushRcvdCharHexDump();
#                        endif
                          ShowError( iErrorClass, "WK-EMU: Rcvd Admin:Reset" );
                        }
                       cmd_finished = TRUE;
                       break;
                    case WK_CMD_ADMIN_HOST_OPEN: // 0x02 = "Host Open" :
                       // From K1EL:
                       // > Upon power-up, Winkeyer2 initializes with the host mode
                       // > turned off. To enable host mode, the PC host must issue
                       // > the admin:open command. Upon open, Winkeyer2 will respond by
                       // > sending the revision code back to the host.
                       // > The host must wait for this return code before
                       // > any other commands or data can be sent to Winkeyer2.
                       // > Upon open, WK1 mode is set.
                       // The short example code accidentally found
                       // in the "WK_USB Application Interface Guide v1.1",
                       // > "WK Init Psuedo Code, error handling not shown for clarity",
                       // confirms the response for "ADMIN_CMD.ADMIN_OPEN" is
                       // a single BYTE; the value is ignored by the simple example.
                       pInstance->fWKHostMode = TRUE;
                       bSendData = 22;   // 23 (decimal) = "v22" as in "Winkeyer2 v22" ?
                       if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
                        {
#                        if( FLUSH_RX_HEX_DUMP_BEFORE_SHOWING_DECODED_COMMAND )
                          AuxCom_Debug_FlushRcvdCharHexDump();
#                        endif
                          ShowError( iErrorClass, "WK-EMU: Rcvd Admin:Host Open, answer=\"%d\"", (int)bSendData );
                        }
                       Winkeyer_SendByte( pInstance, bSendData, "WK-Rev-Code" );
                       cmd_finished = TRUE;
                       break; // end case <received "Admin: Host Open">
                    case WK_CMD_ADMIN_HOST_CLOSE: // "Host Close"
                       // > Use this command to turn off the host interface.
                       // > Winkeyer2 will return to standby mode
                       // > after this command is issued.
                       // > Standby settings will be restored.
                       pInstance->fWKHostMode = FALSE;
                       cmd_finished = TRUE;
                       if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
                        {
#                        if( FLUSH_RX_HEX_DUMP_BEFORE_SHOWING_DECODED_COMMAND )
                          AuxCom_Debug_FlushRcvdCharHexDump();
#                        endif
                          ShowError( iErrorClass, "WK-EMU: Rcvd Admin:Host Close" );
                        }
                       break; // end case <received "Admin: Host Close">
                    case WK_CMD_ADMIN_ECHO_TEST: // "Echo Test"
                       // > Used to test the serial interface.
                       // > The next character sent to Winkeyer2 after this command
                       // > will be echoed back to the host.
                       pInstance->nBytesOfImmediateCommand = 3; // need ANOTHER BYTE for the "Echo Test" !
                       // NOT YET :  cmd_finished = TRUE;
                       // NOT YET :  if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
                       //             { ShowError( iErrorClass, "WK-EMU: Rcvd Admin:EchoTest" );
                       //             }
                       break; // end case <received "Admin: Echo Test">
                    case WK_CMD_ADMIN_PADDLE_A2D: // 0x05 "Paddle A2D"
                       // > "Historical command not supported in WK2, always returns 0."
                       // A-ha. So the host (expecting a "historical Winkeyer 1")
                       //       also expects a RESPONSE from us. Send it:
                       bSendData = 0;
                       if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
                        {
#                        if( FLUSH_RX_HEX_DUMP_BEFORE_SHOWING_DECODED_COMMAND )
                          AuxCom_Debug_FlushRcvdCharHexDump();
#                        endif
                          ShowError( iErrorClass, "WK-EMU: Rcvd Admin:Paddle A2D, answer=\"%d\" ", (int)bSendData );
                        }
                       Winkeyer_SendByte( pInstance, bSendData, "Paddle A2D Response" ); // response from "Paddle A2D", whatever that means
                       cmd_finished = TRUE;
                       break;
                    case WK_CMD_ADMIN_SPEED_A2D: // 0x06 "Speed A2D"
                       // > "Historical command not supported in WK2, always returns 0."
                       bSendData = 0;
                       if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
                        {
#                        if( FLUSH_RX_HEX_DUMP_BEFORE_SHOWING_DECODED_COMMAND )
                          AuxCom_Debug_FlushRcvdCharHexDump();
#                        endif
                          ShowError( iErrorClass, "WK-EMU: Rcvd Admin:Speed A2D, answer=\"%d\" ", (int)bSendData );
                        }
                       Winkeyer_SendByte( pInstance, 0, "Speed A2D Response" ); // response from "Speed A2D", whatever that means
                       cmd_finished = TRUE;
                       break;
                    case WK_CMD_ADMIN_GET_VALUES: // 0x07 "Get Values"
                       // Winkeyer2 IC Interface & Operation Manual 4/15/2008 Page 5:
                       // > "Returns all of the internal setup parameters.
                       // >  They are sent back in the same order as
                       // >  issued by the Load Defaults command.
                       // >  Again, this command is a diagnostic aid.
                       // >  Only issue this command when host interface is closed.
                       // From the specification of "Load Default",
                       //    Winkeyer2 IC Interface & Operation Manual 4/15/2008 page 10:
                       // > "value list is a set of 15 binary values" ...
                       if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
                        {
#                        if( FLUSH_RX_HEX_DUMP_BEFORE_SHOWING_DECODED_COMMAND )
                          AuxCom_Debug_FlushRcvdCharHexDump();
#                        endif
                          ShowError( iErrorClass, "WK-EMU: Rcvd Admin:Get Values, answer=15 bytes" );
                        }
                       Winkeyer_SendBytes( pInstance, &Winkeyer_EEPROM.s.modereg, 15, "Value List" ); // response from "Get Values"
                       // ,----------------------------------------------'
                       // '--> Actually sends .modereg (offset 0x01 within the 256-byte EEPROM)
                       //                to   .k12mode (offset 0x0F within the 256-byte EEPROM).
                       cmd_finished = TRUE;
                       break;
                    case WK_CMD_ADMIN_K1EL_DEBUG: // 0x08 "Reserved, K1EL Debug use only"
                       if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
                        {
#                        if( FLUSH_RX_HEX_DUMP_BEFORE_SHOWING_DECODED_COMMAND )
                          AuxCom_Debug_FlushRcvdCharHexDump();
#                        endif
                          ShowError( iErrorClass, "WK-EMU: Rcvd Admin:K1EL Debug" );
                        }
                       // Someone speculated that FOUR BYTES are sent in response.
                       // Anyway, N1MM Logger+ and similar applications won't send "K1EL DEBUG".
                       cmd_finished = TRUE;
                       break;
                    case WK_CMD_ADMIN_GET_CAL: // 0x09 "Get Cal"
                       // From Winkeyer2 IC Interface & Operation Manual 4/15/2008 Page 5 :
                       // > Historical command not supported in WK2, always returns 0
                       bSendData = 0;
                       if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
                        {
#                        if( FLUSH_RX_HEX_DUMP_BEFORE_SHOWING_DECODED_COMMAND )
                          AuxCom_Debug_FlushRcvdCharHexDump();
#                        endif
                          ShowError( iErrorClass, "WK-EMU: Rcvd Admin:Get Cal, answer \"%d\"", (int)bSendData);
                        }
                       Winkeyer_SendByte( pInstance, bSendData, "Get Cal Response" ); // response from "Paddle A2D", whatever that means
                       cmd_finished = TRUE;
                       break;
                    case WK_CMD_ADMIN_SET_WK1_MODE : // "Set WK1 Mode. Disables pushbutton reporting".
                       // Are we expected to send a response for this ? Guess not.
                       cmd_finished = TRUE;
                       if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
                        {
#                        if( FLUSH_RX_HEX_DUMP_BEFORE_SHOWING_DECODED_COMMAND )
                          AuxCom_Debug_FlushRcvdCharHexDump();
#                        endif
                          ShowError( iErrorClass, "WK-EMU: Rcvd Admin:Set WK1 Mode" );
                        }
                       break;
                    case WK_CMD_ADMIN_SET_WK2_MODE: // "Set WK2 Mode. Enables pushbutton reporting".
                       // Are we expected to send a response for this ? Guess not.
                       cmd_finished = TRUE;
                       if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
                        {
#                        if( FLUSH_RX_HEX_DUMP_BEFORE_SHOWING_DECODED_COMMAND )
                          AuxCom_Debug_FlushRcvdCharHexDump();
#                        endif
                          ShowError( iErrorClass, "WK-EMU: Rcvd Admin:Set WK2 Mode" );
                        }
                       break;
                    case WK_CMD_ADMIN_DUMP_EEPROM: // "Dump EEPROM. Dumps all 256 bytes of WK2's internal EEPROM".
                       if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
                        {
#                        if( FLUSH_RX_HEX_DUMP_BEFORE_SHOWING_DECODED_COMMAND )
                          AuxCom_Debug_FlushRcvdCharHexDump();  // FIRST show the received command as RAW BYTES, ...
#                        endif
                          // ... THEN show the "interpretation" of the recived commands...
                          ShowError( iErrorClass, "WK-EMU: Rcvd Admin:Dump EEPROM, answer=256 bytes" );
                          // ... and FINALLY let Winkeyer_SendBytes() also show the RESPONSE as hex-dump !
                        }
                       Winkeyer_SendBytes( pInstance, (BYTE*)&Winkeyer_EEPROM, 256, "EEPROM Dump" ); // response from "Dump EEPROM"
                       cmd_finished = TRUE;
                       break;
                    case WK_CMD_ADMIN_LOAD_EEPROM: // "Load EEPROM. Download all 256 bytes of WK2's internal EEPROM".
                       if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
                        {
#                        if( FLUSH_RX_HEX_DUMP_BEFORE_SHOWING_DECODED_COMMAND )
                          AuxCom_Debug_FlushRcvdCharHexDump();  // FIRST show the received command as RAW BYTES, ...
#                        endif
                          ShowError( iErrorClass, "WK-EMU: Rcvd Admin:Load EEPROM" );
                        }
                       pInstance->nBytesOfImmediateCommand = 2+256; // 1 byte for "ADMIN" + 1 byte for "LOAD EEPROM" + 256 bytes for the DATA
                       break;
                    case WK_CMD_ADMIN_SEND_MESSAGE: // "Send [Standalone] Message".  <00><14><msg number>
                       if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
                        {
#                        if( FLUSH_RX_HEX_DUMP_BEFORE_SHOWING_DECODED_COMMAND )
                          AuxCom_Debug_FlushRcvdCharHexDump();  // FIRST show the received command as RAW BYTES, ...
#                        endif
                          ShowError( iErrorClass, "WK-EMU: Rcvd Admin:Send Msg" );
                        }
                       pInstance->nBytesOfImmediateCommand = 3; // 1 byte for "ADMIN"
                          // + 1 byte for the sub-command + 1 byte for the PARAMETER VALUE
                       break;
                    case WK_CMD_ADMIN_LOAD_X1MODE: // 15(decimal!) ? "Load XMODE"
                       if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
                        {
#                        if( FLUSH_RX_HEX_DUMP_BEFORE_SHOWING_DECODED_COMMAND )
                          AuxCom_Debug_FlushRcvdCharHexDump();  // FIRST show the received command as RAW BYTES, ...
#                        endif
                          ShowError( iErrorClass, "WK-EMU: Rcvd Admin:Load XMODE" );
                        }
                       pInstance->nBytesOfImmediateCommand = 3; // 1 byte for "ADMIN" + 1 byte for the sub-command + 1 byte for the PARAMETER VALUE
                       break;
                    // The above "Admin commands" should be sufficient for Winkeyer 2.
                    // But when tested with "WK3 Demo 4.2", got the following trouble
                    // reported in the RCW Keyer's "Debug" tab..
                    // > 20:13:43.0 WK-EMU: Rcvd Admin:EchoTest, echoed 0x55
                    // > 20:13:43.0 WK-EMU: Rcvd Admin:Host Open, answer="22"
                    // > 20:13:43.0 WK-EMU: Rcvd UNKNOWN Admin cmd (0x17)
                    // > 20:13:44.0 AuxCom: RX(hex) 00 04 55 00 02 00 17
                    //                              |______| |___| |___|
                    //                              EchoTest HOpen  ???
                    // Consulted a newer datasheet,
                    //  "WinKeyer 3.1 IC Interface & Operation Manual 3/19/2019 Rev 1.3", page 5:
                    // > * Admin Command   <00><nn>   nn is a (*) value from 0 to 25
                    //    ,--------------------|__|-------------------------------|_|
                    //    '--> One should emphasize that despite the introduction..
                    //     > "In this document a hex value will be presented
                    //     >   in angle brackets, for example <02>"
                    //   ..  the <nn> in angle bracket are NOT HEXADECIMAL BUT DECIMAL !
                    // Below: "ADMIN Commands exclusively for Winkeyer 3" !
                    case WK_CMD_ADMIN_FIRMWARE_UPLD: // 16 (DECIMAL!) "This command initiates an image upload". (we'll never support this)
                    case WK_CMD_ADMIN_SET_LOW_BAUD : // 17 (DECIMAL!) "Set Low Baud: Change serial comm. Baud Rate to 1200 (default)"
                    case WK_CMD_ADMIN_SET_HIGH_BAUD: // 18 (DECIMAL!) "Set High Baud: Change serial comm. Baud Rate to 9600"
                    case WK_CMD_ADMIN_SET_RTTY_REGS: // 19 (DECIMAL!) "Set RTTY Mode Registers". <0x00><19d><P1><P2>. "WK3.1 only".
                    case WK_CMD_ADMIN_SET_WK3_MODE : // 20 (DECIMAL!) "Set WK3 Mode: Enables WinKeyer 3 functions; expanded X1MODE and additional X2MODE register"
                    case WK_CMD_ADMIN_GET_V_SUPPLY : // 21 (DECIMAL!) "Read Back Vcc: Return WK IC power supply voltage."
                    case WK_CMD_ADMIN_LOAD_X2MODE  : // 22 (DECIMAL!) "Load mode extension register 2".  Only for Winkeyer 3 (and up) ?
                    // ex: case WK_CMD_ADMIN_GET_FW_MINOR : // 23 (DECIMAL!) "Get FW Minor Rev: Returns the minor firmware revision, 03 for version 31.03"
                    // ex: case WK_CMD_ADMIN_GET_IC_TYPE  : // 24 (DECIMAL!) "Returns the WK IC type: 0x1 for SMT, 0x0 for DIP"
                    case WK_CMD_ADMIN_SET_SIDETN_VOL: // 25 (DECIMAL!) "Set Sidetone Volume <00><24><n> where n =0x1 for low and n=0x4 for high"
                       if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
                        {
#                        if( FLUSH_RX_HEX_DUMP_BEFORE_SHOWING_DECODED_COMMAND )
                          AuxCom_Debug_FlushRcvdCharHexDump();  // FIRST show the received command as RAW BYTES, ...
#                        endif
                          ShowError( ERROR_CLASS_WARNING | SHOW_ERROR_TIMESTAMP,
                               "WK-EMU: Unsupported UNKNOWN Admin cmd (#%d = %s)",
                               (unsigned int)bRcvdChar,
                               WK_AdminCommandToString(bRcvdChar) );
                        }
                       cmd_finished = TRUE; // whatever it was, IGNORE IT, and don't assume it has "parameters"
                       break;
                    case WK_CMD_ADMIN_GET_FW_MINOR : // 23 (DECIMAL!) "Get FW Minor Rev: Returns the minor firmware revision, 03 for version 31.03"
                       // Without the following, "WK3 Demo 4.2" bailed out early with "Attach WK: Open Failed".
                       // In RCW-Keyer : > WK-EMU: Unsupported UNKNOWN Admin cmd (#23 = GetFWMinor)
                       bSendData = 3; // > "Returns the minor firmware revision, 03 for version 31.03"
                       if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
                        {
#                        if( FLUSH_RX_HEX_DUMP_BEFORE_SHOWING_DECODED_COMMAND )
                          AuxCom_Debug_FlushRcvdCharHexDump();  // FIRST show the received command as RAW BYTES, ...
#                        endif
                          ShowError( iErrorClass, "WK-EMU: Rcvd Admin:%s, answer=\"%d\" ",
                                     WK_AdminCommandToString(bRcvdChar), (int)bSendData );
                        }
                       Winkeyer_SendByte( pInstance, bSendData, "FW Minor" );
                       cmd_finished = TRUE;
                       break; // end case WK_CMD_ADMIN_GET_FW_MINOR
                    case WK_CMD_ADMIN_GET_IC_TYPE : // 24 (DECIMAL!) "Returns the WK IC type: 0x1 for SMT, 0x0 for DIP"
                       // Without the following, "WK3 Demo 4.2" bailed out early :
                       // In RCW-Keyer : > WK-EMU: Unsupported UNKNOWN Admin cmd (#24 = GetICType)
                       bSendData = 0; // > "Returns the WK IC type 0x1 for SMT, 0x0 for DIP"
                       if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
                        {
#                        if( FLUSH_RX_HEX_DUMP_BEFORE_SHOWING_DECODED_COMMAND )
                          AuxCom_Debug_FlushRcvdCharHexDump();  // FIRST show the received command as RAW BYTES, ...
#                        endif
                          ShowError( iErrorClass, "WK-EMU: Rcvd Admin:%s, answer=\"%d\" ",
                                     WK_AdminCommandToString(bRcvdChar), (int)bSendData );
                        }
                       Winkeyer_SendByte( pInstance, bSendData, "IC-Type" );
                       cmd_finished = TRUE;
                       // After adding the above, "WK3 Demo 4.2" bailed out with:
                       //  "This version of WKdemo is for WK3 only".  A-ha.
                       // In 2025-06-09, only got this far in the 'Debug' window:
                       // > WK-EMU: Rcvd Admin:EchoTest, echoed 0x55
                       // > WK-EMU: Rcvd Admin:Host Open, answer="22" (that's the "revision code")
                       // > WK-EMU: Rcvd Admin:GetFWMinor, answer="3"
                       // > WK-EMU: Rcvd Admin:GetICType, answer="0"
                       // > AuxCom: RX(hex) 00 04 55 00 02 00 17 00 18
                       //  Gave up, but left these "first steps towards Winkeyer 3" in the code.
                       //  Back to the main problem: Why does N1MM Logger+ get stuck
                       //  in an endless loop, repeating its "Winkeyer init sequence"
                       //  over and over again ? Does it expect a response we don't send ?
                       //  And why does it always steal the keyboard focus from other apps ?
                       break; // end case WK_CMD_ADMIN_GET_IC_TYPE
                    default:
                       if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
                        {
#                        if( FLUSH_RX_HEX_DUMP_BEFORE_SHOWING_DECODED_COMMAND )
                          AuxCom_Debug_FlushRcvdCharHexDump();  // FIRST show the received command as RAW BYTES, ...
#                        endif
                          ShowError( ERROR_CLASS_WARNING | SHOW_ERROR_TIMESTAMP, "WK-EMU: Rcvd UNKNOWN Admin cmd (0x%02X)",
                                     (unsigned int)bRcvdChar );
                        }
                       cmd_finished = TRUE; // whatever it was, IGNORE IT, and don't assume it has "parameters"
                       break;
                  } // end switch( pInstance->bImmediateSubcommand ) ["type" of a received "Admin" command]
                 break; // end case iByteOfImmediateCommand==1 (for ADMIN commands)
              case 2: // iByteOfImmediateCommand=2 : Now bRcvdChar is the THIRD byte,
                      //    following after the "Type" of the "Admin" command.
                      // Only a few "Admin" command need this (see switch/case below).
                      // Example with pInstance->bImmediateSubcommand=0x04="Echo Test":
                      //  At this point, bRcvdChar is byte that we are supposed to send back to the host.
                 switch( pInstance->bImmediateSubcommand )
                  { case WK_CMD_ADMIN_CALIB: // "Calibrate" (hopefully with bRcvdChar=0xFF now) : <00><00> pause 100 ms <FF>
                       // Send a REPONSE ? ? e.g. Winkeyer_SendByte( pInstance, 1 ); // for what did N1MM Logger+ wait now ?
                       cmd_finished = TRUE;
                       break;
                    case WK_CMD_ADMIN_ECHO_TEST: // "Echo Test" . Definitely needs a response: The single "echoed" byte
                       if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
                        {
#                        if( FLUSH_RX_HEX_DUMP_BEFORE_SHOWING_DECODED_COMMAND )
                          AuxCom_Debug_FlushRcvdCharHexDump();  // FIRST show the received command as RAW BYTES, ...
#                        endif
                          ShowError( iErrorClass, "WK-EMU: Rcvd Admin:EchoTest, echoed 0x%02X ",
                                                  (int)bRcvdChar );
                        }
                       Winkeyer_SendByte( pInstance, bRcvdChar, "Echo-Test" ); // send back the "echoed" byte for the ECHO TEST (not the "Echo back morse in ASCII"-thing)
                       cmd_finished = TRUE;
                       break;
                    default:
                       cmd_finished = TRUE;
                       break;
                  }
                 break; // end case iByteOfImmediateCommand==2 (for ADMIN commands)
              default:
                 break;
            } // end switch( pInstance->iByteOfImmediateCommand ) for "Admin" commands
           break;  // end case < ALL(!) "Administative Commands, IMMEDIATE" >

        case WK_CMD_SET_SIDETONE_IMMEDIATE: // 0x01: "Set sidetone frequency, IMMEDIATE" : <01><freq>
           if( pInstance->iByteOfImmediateCommand == 0 ) // just received the COMMAND, so set the LENGTH:
            { pInstance->nBytesOfImmediateCommand = 2;
            }
           else // Now bRcvdChar is the new "sidetone" config, so we're 'finished'
            { // From the Winkeyer2 IC Interface & Operation Manual 4/15/2008 Page 6:
              // > In WK2 sidetone is always enabled and pin 8 functions as the
              // > sidetone square wave output. The following tables define the
              // > format of value nn. Note that these frequencies are slightly
              // > different than WK1 .
              // >  Bit 7 (MSB) : "Enable Paddle Only Sidetone" when = 1 .
              // >  Bits 6..4   : Unused, set to zero.
              // >  Bits 3..0   : Sidetone frequency N (see table below)
              // >  N   | Frequency || N   | Frequency |
              // >  ----+-----------++-----+-----------+
              // >  0x1 | 4000 Hz   || 0x6 | 666 Hz    |
              // >  0x2 | 2000 Hz   || 0x7 | 571 Hz    |
              // >  0x3 | 1333 Hz   || 0x8 | 500 Hz    |
              // >  0x4 | 1000 Hz   || 0x9 | 444 Hz    |
              // >  0x5 |  800 Hz   || 0xA | 400 Hz    |
              //    ("Table 2 - Sidetone Selection Table" for Winkeyer 2)
              Winkeyer_EEPROM.s.sidetone = bRcvdChar;
              // ToDo: Pass on the new sidetone frequency to Remote CW Keyer ?
              cmd_finished = TRUE;
              if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
               {
#               if( FLUSH_RX_HEX_DUMP_BEFORE_SHOWING_DECODED_COMMAND )
                 AuxCom_Debug_FlushRcvdCharHexDump();  // FIRST show the received command as RAW BYTES, ...
#               endif
                 ShowError( iErrorClass, "WK-EMU: Rcvd SetSidetone(%d Hz)",
                  (int)WK_SidetoneCodeToFrequency_Hz( Winkeyer_EEPROM.s.sidetone) );
               }
            }
           break; // end case WK_CMD_SET_SIDETONE_IMMEDIATE
        case WK_CMD_SET_MORSE_SPEED_IMM: // 0x02 "Set Morse sending speed, IMMEDIATE"  <02><WPM>
           if( pInstance->iByteOfImmediateCommand == 0 ) // just received the COMMAND, so set the LENGTH:
            { pInstance->nBytesOfImmediateCommand = 2;
            }
           else // Now bRcvdChar is the "Morse sending speed" in WPM, so we're 'finished' .
            { cmd_finished = TRUE;
              if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
               {
#               if( FLUSH_RX_HEX_DUMP_BEFORE_SHOWING_DECODED_COMMAND )
                 AuxCom_Debug_FlushRcvdCharHexDump();  // FIRST show the received command as RAW BYTES, ...
#               endif
                 ShowError( iErrorClass, "WK-EMU: Rcvd SetMorseSpeed(%d WPM). Previous setting was %d WPM.",
                         (int)bRcvdChar, (int)Winkeyer_EEPROM.s.speed_wpm );
                 // Got the following when clicking the 'speed up/down' control
                 // in N1MM Logger+ main window :
                 // > 10:23:24.7 WK-EMU: Rcvd SetMorseSpeed(97 WPM). Previous setting was 95 WPM.
                 // Since this an IMMEDIATE command, users will expect it
                 // to be passed on to the CW GENERATOR (module CwGen.c) IMMEDIATELY,
                 // and (to prevent confusion from users) also update the 'speed control'
                 // in the Remote CW Keyer GUI (Keyer_Main.cpp: Ed_WPM + TrackBar_WPM).
                 // Since the VCL isn't (and never will be) thread-safe,
               }
              Winkeyer_EEPROM.s.speed_wpm = bRcvdChar;
              // Pass on the new CW speed to the Remote CW Keyer, and "inform" the GUI ?
              i = Elbug_WordsPerMinuteToDotTimeInMilliseconds( (int)Winkeyer_EEPROM.s.speed_wpm );
              if( i != CwKeyer_Elbug.cfg.iDotTime_ms )
               { CwKeyer_Elbug.cfg.iDotTime_ms = CwKeyer_Gen.cfg.iDotTime_ms = i;
                 KeyerGUI_fUpdateWPMIndicator  = TRUE; // flag to update the GUI on the next occasion
                 // Example: Winkeyer_EEPROM.s.speed_wpm = 97 (~~ the maximum)
                 //  -> i = 15 milliseconds per dot ("dit"),
                 //            previous CwKeyer_Elbug.cfg.iDotTime_ms = 48
                 //   -> KeyerGUI_fUpdateWPMIndicator was SET here
                 //    -> indicator "Speed (WPM)" on the RCWK's "Keyer Settings" tab changed
                 //       (both the decimal input field, and the horizontal track bar)
               } // end if( i != CwKeyer_Elbug.cfg.iDotTime_ms ) || ...
            }   // end if < received the "Morse sending speed" in WPM for the IMMEDIATE command >
           break; // end case WK_CMD_SET_MORSE_SPEED_IMM
        case WK_CMD_SET_KEY_WEIGHTING:   // 0x03 "Set key weighting, IMMEDIATE"      <03><weight>
           if( pInstance->iByteOfImmediateCommand == 0 ) // just received the COMMAND, so set the LENGTH:
            { pInstance->nBytesOfImmediateCommand = 2;
            }
           else // Now bRcvdChar is the new "weight", so we're 'finished' .
            { // We'll most likely never support non-standard weighting (only dit/dash = 1/3)
              cmd_finished = TRUE;
              if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
               {
#               if( FLUSH_RX_HEX_DUMP_BEFORE_SHOWING_DECODED_COMMAND )
                 AuxCom_Debug_FlushRcvdCharHexDump();  // FIRST show the received command as RAW BYTES, ...
#               endif
                 ShowError( iErrorClass, "WK-EMU: Rcvd SetWeight(%d)", (int)bRcvdChar );
               }
            }
           break; // end case WK_CMD_SET_KEY_WEIGHTING
        case WK_CMD_SET_PTT_DELAYS:      // 0x04 "Set up PTT delays, IMMEDIATE"      <04><leadin><tail>
           switch( pInstance->iByteOfImmediateCommand )
            { case 0 : // just received the COMMAND, so set the LENGTH:
                 pInstance->nBytesOfImmediateCommand = 3;
                 break;
              case 1 : // just received the FIRST parameter ("leadin")..
                 Winkeyer_EEPROM.s.lead_time = bRcvdChar;
                 break;
              case 2 : // just received the SECOND parameter ("tail")..
                 Winkeyer_EEPROM.s.tail_time = bRcvdChar;
                 cmd_finished = TRUE; // <- for clarity; should already be set
                 if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
                  {
#                  if( FLUSH_RX_HEX_DUMP_BEFORE_SHOWING_DECODED_COMMAND )
                    AuxCom_Debug_FlushRcvdCharHexDump();  // FIRST show the received command as RAW BYTES, ...
#                  endif
                    ShowError( iErrorClass, "WK-EMU: Rcvd SetPttDelays(lead=%d, tail=%d ms)",
                       (int)Winkeyer_EEPROM.s.lead_time, (int)Winkeyer_EEPROM.s.tail_time );
                  }
                 break;
            }
           break; // end case WK_CMD_SET_PTT_DELAYS
        case WK_CMD_SET_SPEED_POT_RANGE: // 0x05 "Set up speed pot range, IMMEDIATE" <05><m><wr><pr>
           switch( pInstance->iByteOfImmediateCommand )
            { case 0 : // just received the COMMAND, so set the LENGTH:
                 pInstance->nBytesOfImmediateCommand = 4;
                 break;
              case 1 : // just received the FIRST parameter ("<m>")..
                 Winkeyer_EEPROM.s.pot_min_wpm   = bRcvdChar;  // kind of "scaling offset"
                 break;
              case 2 : // just received the SECOND parameter ("<wr>")..
                 Winkeyer_EEPROM.s.pot_wpm_range = bRcvdChar;  // kind of "scaling factor"
                 break;
              case 3 : // just received the THIRD parameter ("<pr>").. ignore; has no function anymore
                 if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
                  {
#                  if( FLUSH_RX_HEX_DUMP_BEFORE_SHOWING_DECODED_COMMAND )
                    AuxCom_Debug_FlushRcvdCharHexDump();  // FIRST show the received command as RAW BYTES, ...
#                  endif
                    ShowError( iErrorClass, "WK-EMU: Rcvd SetSpeedPotRange(min=%d,range=%d WPM)",
                               (int)Winkeyer_EEPROM.s.pot_min_wpm, (int)Winkeyer_EEPROM.s.pot_wpm_range);
                  }
                 cmd_finished = TRUE;
                 break;
            }
           break; // end case WK_CMD_SET_SPEED_POT_RANGE
        case WK_CMD_PAUSE_MORSE_IMM:     // 0x06 "Pause Morse output, IMMEDIATE"     <06><0 or 1>
           if( pInstance->iByteOfImmediateCommand == 0 ) // just received the COMMAND, so set the LENGTH:
            { pInstance->nBytesOfImmediateCommand = 2;
            }
           else // Now bRcvdChar is the new "Pause Morse" flag, so we're 'finished' .
            { pInstance->fWKImmediatePauseState = bRcvdChar;
              cmd_finished = TRUE;
              if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
               {
#               if( FLUSH_RX_HEX_DUMP_BEFORE_SHOWING_DECODED_COMMAND )
                 AuxCom_Debug_FlushRcvdCharHexDump();  // FIRST show the received command as RAW BYTES, ...
#               endif
                 ShowError( iErrorClass, "WK-EMU: Rcvd PauseMorseOutput(%d)",(int)bRcvdChar );
               }
            }
           break; // end case WK_CMD_PAUSE_MORSE_IMM
        case WK_CMD_GET_SPEED_POT_IMM:   // 0x07 "Request speed pot value, IMMEDIATE" <07> (without parameters but a RESPONSE)
           // ex: bSendData = Elbug_DotTimeInMillisecondsToWordsPerMinute( CwKeyer_Elbug.cfg.iDotTime_ms );
           // The response for "Get Speed Pot" is not just the speed in WPM,
           // but the usual bit-fiddling that you only discover on third sight
           // in the "WinKeyer IC Interface & Operation Manual" :
           //  > "The returned value will range from 0 to 31 and will be the
           //  >  actual speed pot value minus the MIN_WPM setting.
           //  >  The speed pot is windowed into a 32 step range between 5
           //  >  to 99 WPM (see Setup SpeedPot command). The two MS Bits
           //  >  of a Speed Pot status byte will always be 0b10 .
           //    bit 7  bit 6  bits 5.. 0
           //      ,------------------------------------,
           //      |  1   |  0   | "6 bit value in WPM" |
           //      '------------------------------------'
           //     "Table 8 - Speed Pot Status Byte Format"
           // Hmmm. A SIX BIT VALUE IN WPM (thus good enough for 0..63 WPM)
           //   but a "returned value that will range from 0 to 31" only ?!
           // This WINKEYER EMULATOR has no "speed pot", and because the
           // response byte is NOT a "value in WPM", fake it:
           i = (int)Winkeyer_EEPROM.s.speed_wpm - (int)Winkeyer_EEPROM.s.pot_min_wpm;
           if( i > 63 )
            {  i = 63;  // hmmm... "Table 8" shows a SIX BIT VALUE so why limit to 31 ?
            }
           bSendData = 0x80/*"pot report Tag"*/ | (BYTE)i;
           if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
            {
#            if( FLUSH_RX_HEX_DUMP_BEFORE_SHOWING_DECODED_COMMAND )
              AuxCom_Debug_FlushRcvdCharHexDump();  // FIRST show the received command as RAW BYTES, ...
#            endif
              ShowError( iErrorClass, "WK-EMU: Rcvd 'GetSpeedPot', answer=\"%d WPM minus MIN_WPM=%d\"",
                         (int)Winkeyer_EEPROM.s.speed_wpm, (int)Winkeyer_EEPROM.s.pot_min_wpm );
              // ,------------|_________________________|  ,----|___________________________|
              // '--> That's not the "speed pot",          '--> Strange but as explained
              //      but we don't have anything else.          in the WK datasheet !
            }
           Winkeyer_SendByte( pInstance, bSendData, "GetSpeedPot-Response" );
           cmd_finished = TRUE;
           break; // end case WK_CMD_GET_SPEED_POT_IMM
        case WK_CMD_BACKUP_INPUT_POINTER: // 0x08 "Backup input pointer (aka BACKSPACE), IMMEDIATE" <08> (without parameters)
           Winkeyer_DeleteLastCharInInputBuffer( pInstance );
           if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
            {
#            if( FLUSH_RX_HEX_DUMP_BEFORE_SHOWING_DECODED_COMMAND )
              AuxCom_Debug_FlushRcvdCharHexDump();  // FIRST show the received command as RAW BYTES, ...
#            endif
              ShowError( iErrorClass, "WK-EMU: Rcvd 'BACKSPACE'" );
            }
           cmd_finished = TRUE;
           break; // end case WK_CMD_BACKUP_INPUT_POINTER
        case WK_CMD_SET_OUTPUT_PIN_CONFIG: // 0x09 "Set output pin configuration, IMMEDIATE"  <09><config>
           if( pInstance->iByteOfImmediateCommand == 0 ) // just received the COMMAND, so set the LENGTH:
            { pInstance->nBytesOfImmediateCommand = 2;
            }
           else // Now bRcvdChar is the new "output pin config" flag, so we're 'finished' .
            { Winkeyer_EEPROM.s.pincfg = bRcvdChar;
              // > WK1s 8 pin package forced Pin 5 to be a shared resource,
              // > it could be assigned as a PTT output, a Sidetone output,
              // > or a secondary Key output: If it was assigned as a PTT output,
              // > that meant it was not possible to output sidetone.
              // > Likewise if it was assigned as a secondary Key output,
              // > sidetone or PTT were not allowed.
              //   ,--------------------------------------------------------,
              //   | Bit 7-4   | Bit 3      | Bit 2      | Bit 1    | Bit 0 |
              //   |-----------+------------+------------+----------+-------|
              //   | See Below |Pin5 KeyOut |Pin5 KeyOut | Pin 5    | Pin 5 |
              //   | (WK docs) | Enable     |  Enable    | Sidetone | PTT   |
              //   '--------------------------------------------------------'
              // (etc etc, see Winkeyer2 IC Interface & Operation Manual 4/15/2008 Page 8)
              // > WK2 has a dedicated sidetone pin, two PTT outputs,
              // > and two Key outputs. No sacrifices are necessary. The
              // > pin config assignments are compatible with WK1 for backwards
              // > software compatibility but allow much more flexibility ....
              if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
               {
#               if( FLUSH_RX_HEX_DUMP_BEFORE_SHOWING_DECODED_COMMAND )
                 AuxCom_Debug_FlushRcvdCharHexDump();  // FIRST show the received command as RAW BYTES, ...
#               endif
                 ShowError( iErrorClass, "WK-EMU: Rcvd SetOutputPinConfig(0x%02X)", (unsigned int)bRcvdChar );
               }
              cmd_finished = TRUE;
            }
           break; // end case WK_CMD_SET_OUTPUT_PIN_CONFIG
        case WK_CMD_CLEAR_INPUT_BUFFER: // 0x0A "Clear input buffer, IMMEDIATE"  (without parameters)
           // (Received THIS command when opening N1MM Logger's "Send CW" window
           //  by pressing CTRL-K in the logger's main window)
           Winkeyer_ClearInputBuffer( pInstance );
           if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
            {
#            if( FLUSH_RX_HEX_DUMP_BEFORE_SHOWING_DECODED_COMMAND )
              AuxCom_Debug_FlushRcvdCharHexDump();  // FIRST show the received command as RAW BYTES, ...
#            endif
              ShowError( iErrorClass, "WK-EMU: Rcvd cmd ClearInputBuffer" );
            }
           cmd_finished = TRUE;  // only a SINGLE BYTE (no params) -> already finished
           break; // end case WK_CMD_CLEAR_INPUT_BUFFER
        case WK_CMD_DIRECT_CTL_KEY_OUTPUT: // 0x0B "Direct control of key output, IMMEDIATE"  <0B><0 or 1>
           if( pInstance->iByteOfImmediateCommand == 0 ) // just received the COMMAND, so set the LENGTH:
            { pInstance->nBytesOfImmediateCommand = 2;
            }
           else // Now bRcvdChar is the "direct control" parmeter value, so we're 'finished' .
            {
              if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
               {
#               if( FLUSH_RX_HEX_DUMP_BEFORE_SHOWING_DECODED_COMMAND )
                 AuxCom_Debug_FlushRcvdCharHexDump();  // FIRST show the received command as RAW BYTES, ...
#               endif
                 ShowError( iErrorClass, "WK-EMU: Rcvd Direct Key Output(%d)",(int)bRcvdChar );
               }
              cmd_finished = TRUE;
            }
           break; // end case WK_CMD_DIRECT_CTL_KEY_OUTPUT
        case WK_CMD_SET_HSCW_SPEED_IMM:  // 0x0C "Set HSCW speed, IMMEDIATE"              <0C><lpm/100>
           if( pInstance->iByteOfImmediateCommand == 0 ) // just received the COMMAND, so set the LENGTH:
            { pInstance->nBytesOfImmediateCommand = 2;
            }
           else // Now bRcvdChar is the "HSCW lpm" parmeter value, so we're 'finished' .
            {
              if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
               {
#               if( FLUSH_RX_HEX_DUMP_BEFORE_SHOWING_DECODED_COMMAND )
                 AuxCom_Debug_FlushRcvdCharHexDump();  // FIRST show the received command as RAW BYTES, ...
#               endif
                 ShowError( iErrorClass, "WK-EMU: Rcvd SetHSCW(%d)",(int)bRcvdChar );
               }
              cmd_finished = TRUE;
            }
           break; // end case WK_CMD_SET_HSCW_SPEED_IMM
        case WK_CMD_SET_FARNSWORTH_IMM:  // 0x0D "Set Farnsworth speed, IMMEDIATE"        <0D><WPM>
           if( pInstance->iByteOfImmediateCommand == 0 ) // just received the COMMAND, so set the LENGTH:
            { pInstance->nBytesOfImmediateCommand = 2;
            }
           else // Now bRcvdChar is the "Farnsworth speed" parmeter value, so we're 'finished' .
            {
              if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
               {
#               if( FLUSH_RX_HEX_DUMP_BEFORE_SHOWING_DECODED_COMMAND )
                 AuxCom_Debug_FlushRcvdCharHexDump();  // FIRST show the received command as RAW BYTES, ...
#               endif
                 ShowError( iErrorClass, "WK-EMU: Rcvd SetFarnsworth(%d WPM)",(int)bRcvdChar );
               }
              cmd_finished = TRUE;
            }
           break; // end case WK_CMD_SET_FARNSWORTH_IMM
        case WK_CMD_SET_WK2_MODE_BYTE_IMM: // 0x0E "Load Winkeyer2 mode byte, IMMEDIATE"    <0E><mode>
           if( pInstance->iByteOfImmediateCommand == 0 ) // just received the COMMAND, so set the LENGTH:
            { pInstance->nBytesOfImmediateCommand = 2;
            }
           else // Now bRcvdChar is the "Winkeyer2 mode byte" parmeter value, so we're 'finished' .
            {
              Winkeyer_EEPROM.s.modereg = bRcvdChar;
              if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
               {
#               if( FLUSH_RX_HEX_DUMP_BEFORE_SHOWING_DECODED_COMMAND )
                 AuxCom_Debug_FlushRcvdCharHexDump();  // FIRST show the received command as RAW BYTES, ...
#               endif
                 ShowError( iErrorClass, "WK-EMU: Rcvd SetWK2ModeByte(0x%02X)",(unsigned int)bRcvdChar );
               }
              cmd_finished = TRUE;
            }
           break; // end case WK_CMD_SET_WK2_MODE_BYTE_IMM
        case WK_CMD_LOAD_DEFAULTS_IMM:   // 0x0F "Load Defaults, IMMEDIATE", aka "Download WK state block" <0F><15 values>
           if( pInstance->iByteOfImmediateCommand == 0 ) // just received the COMMAND, so set the LENGTH:
            { pInstance->nBytesOfImmediateCommand = 16;  // 16 bytes INCLUDING the command itself !
            }
           else if( pInstance->iByteOfImmediateCommand <= 15 )
            { // Keep it simple, only store the next "Default"-value in our simulated EEPROM (RAM):
              Winkeyer_EEPROM.b[ pInstance->iByteOfImmediateCommand ] = bRcvdChar;
              // ,---------------|________________________________|
              // '--> Skips Winkeyer_EEPROM.b[0] = Winkeyer_EEPROM.s.magic   !
              //   First is Winkeyer_EEPROM.b[1] = Winkeyer_EEPROM.s.modereg !
              // See Winkeyer2 IC Interface & Operation Manual 4/15/2008 Page 10,
              //   > "value list is a set of 15 binary values"
              // Seen on the Remote CW Keyer's 'Debug' tab (sent by WKdemo):
              //   0F 00 10 05 32 00 00 05 19 00 00 00 00 32 06 FF
              //  /|\ '------------------------------------------'
              //   |   :       "value list, 15 binary values"   :
              //   |  EEPROM.modereg (*)                      EEPROM.k12mode ?!
              //   command WK_CMD_LOAD_DEFAULTS_IMM
              // (*) WATCH these new settings : inspect Winkeyer_EEPROM.s.k12mode
              //     It was indeed 0xFF after after executing the LOAD DEFAULTS command,
              //     even though the "Winkeyer2 IC Interface & Operation Manual"
              //     says about the 15-th VALUE after "Load Defaults": "Don't care, set to 0".
              //     Obviously, "WKdemo" knows an undocumented little secret.
            }
           if( pInstance->iByteOfImmediateCommand == 15 ) // end of the "value list with 15 binary values" ->
            {
              // which -again- happens AFTER this monsterous switch/case list !
              if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
               {
#               if( FLUSH_RX_HEX_DUMP_BEFORE_SHOWING_DECODED_COMMAND )
                 AuxCom_Debug_FlushRcvdCharHexDump();  // FIRST show the received command as RAW BYTES, ...
#               endif
                 ShowError( iErrorClass, "WK-EMU: Rcvd LoadDefaults(1+15 bytes).." );
               }
              cmd_finished = TRUE;
            }
           break; // end case WK_CMD_LOAD_DEFAULTS_IMM
        case WK_CMD_SET_FIRST_ELEM_CORR: // 0x10 "Setup first element correction, IMMEDIATE"  <10><msec>
           if( pInstance->iByteOfImmediateCommand == 0 ) // just received the COMMAND, so set the LENGTH:
            { pInstance->nBytesOfImmediateCommand = 2;
            }
           else // Now bRcvdChar is the parameter value, and we're 'finished' ->
            { Winkeyer_EEPROM.s.xtnd = bRcvdChar;
              if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
               {
#               if( FLUSH_RX_HEX_DUMP_BEFORE_SHOWING_DECODED_COMMAND )
                 AuxCom_Debug_FlushRcvdCharHexDump();  // FIRST show the received command as RAW BYTES, ...
#               endif
                 ShowError( iErrorClass, "WK-EMU: Rcvd SetFirstElemCorr(%d)",(int)bRcvdChar );
               }
              cmd_finished = TRUE;
            }
           break; // end case WK_CMD_SET_FIRST_ELEM_CORR
        case WK_CMD_SET_KEYING_COMPENS:  // 0x11 "Set Keying Compensation, IMMEDIATE"    <11><comp>
           if( pInstance->iByteOfImmediateCommand == 0 ) // just received the COMMAND, so set the LENGTH:
            { pInstance->nBytesOfImmediateCommand = 2;
            }
           else // Now bRcvdChar is the parameter value, so again we're 'finished' ->
            { Winkeyer_EEPROM.s.kcomp = bRcvdChar;
              cmd_finished = TRUE;
              if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
               {
#               if( FLUSH_RX_HEX_DUMP_BEFORE_SHOWING_DECODED_COMMAND )
                 AuxCom_Debug_FlushRcvdCharHexDump();  // FIRST show the received command as RAW BYTES, ...
#               endif
                 ShowError( iErrorClass, "WK-EMU: Rcvd SetKeyingCompensation(%d)",(int)bRcvdChar );
               }
            }
           break; // end case WK_CMD_SET_KEYING_COMPENS
        case WK_CMD_SET_PADDLE_SENS_IMM: // 0x12 "Setup paddle sensitivity, IMMEDIATE"   <12><sens>
           // > "This controls when Winkeyer2 will start looking for a new
           // > paddle press after sensing the current one. If there is not
           // > enough delay the keyer will send unwanted dits or dahs,
           // > if there is too much delay it bogs you down because you can't
           // > get ahead of the keyer. The default value is one dit time (50)
           // > and is adjustable in percent of a dit time.
           // > Faster operators report a setting somewhat less than default
           // > is more pleasing. If the paddle sensitivity is set to zero,
           // > dit and dah paddle memory is disabled. The delay is calculated
           // > with this formula:
           // >
           // > DELAY_TIME = (SWITCHPOINTDIT_TIME)/50 where Switchpoint
           // >                            is a value between 10 and 90.
           if( pInstance->iByteOfImmediateCommand == 0 ) // just received the COMMAND, so set the LENGTH:
            { pInstance->nBytesOfImmediateCommand = 2;
            }
           else // Now bRcvdChar is the parameter value, so again we're 'finished' ->
            { Winkeyer_EEPROM.s.sampadj = bRcvdChar;
              cmd_finished = TRUE;
              if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
               {
#               if( FLUSH_RX_HEX_DUMP_BEFORE_SHOWING_DECODED_COMMAND )
                 AuxCom_Debug_FlushRcvdCharHexDump();  // FIRST show the received command as RAW BYTES, ...
#               endif
                 ShowError( iErrorClass, "WK-EMU: Rcvd SetPaddleSwitchpoint(%d)",(int)bRcvdChar );
               }
            }
           break; // end case WK_CMD_SET_PADDLE_SENS_IMM
        case WK_CMD_NO_OPERATION_IMM:    // 0x13 "Null Command, NOP, IMMEDIATE"    <13>  (without parameters)
           // This command, sent THREE TIMES, should be the first we receive after power-on
           // from a well-behaving host as recommended by K1EL in the
           // "WK_USB Application Interface Guide v1.1" (targeted to PC software authors):
           // > // Send three null commands to resync host to WK2
           // > wk_send (NULLIMM_CMD);
           // > wk_send (NULLIMM_CMD);
           // > wk_send (NULLIMM_CMD); .
           // Indeed, when testing our "Winkeyer Emulation" with the K1EL "WKdemo" (for WK1 + WK2),
           // the traffic decoded by Remote CW Keyer (shown on the "Debug" tab) was:
           // > 16:14:27.2 WK-EMU: Rcvd 'Null Command' (0x13)
           // > 16:14:27.2 WK-EMU: Rcvd 'Null Command' (0x13)
           // > 16:14:27.2 AuxCom: RX(hex)  13
           // > 16:14:27.2 WK-EMU: Rcvd 'Null Command' (0x13)
           // > 16:14:27.2 AuxCom: RX(hex)  13
           // > 16:14:27.2 WK-EMU: Rcvd Admin:EchoTest, echoed 0x55
           // > 16:14:27.2 AuxCom: RX(hex)  00 04 55
           // > 16:14:27.2 WK-EMU: Rcvd Admin:Host Open, answered "22" (WKdemo accepted this as "WinKey2 V22 OnLine")
           // > 16:14:27.2 AuxCom: RX(hex)  00 02
           // > 16:14:27.2 WK-EMU: Rcvd LoadDefaults(.. 16 bytes ..)
           //                                | ,-----"value list of 15 binary values"-----,
           // > 16:14:27.2 AuxCom: RX(hex)  0F 00 10 05 32 00 00 05 19 00 00 00 00 32 06 FF
           // > 16:14:27.2 AuxCom: RX(hex)  07 (should have been interpreted as "Get Speed Pot")
           // > 16:14:27.2 WK-EMU: Rcvd cmd 'GetStatus', answered "0xC0"
           // > 16:14:27.2 AuxCom: RX(hex)  15 ---'
           // > 16:14:27.2 WK-EMU: Rcvd Admin:Set WK2 Mode
           // > 16:14:27.2 AuxCom: RX(hex)  00 0B -'
           //
           // When clicking the "Close"-button in "WKdemo":
           // > 16:09:46.2 WK-EMU: Rcvd Admin:Set WK1 Mode
           // > 16:09:46.2 AuxCom: RX(hex)  00 0A --'
           // > 16:09:46.2 WK-EMU: Rcvd Admin:Host Close
           // > 16:09:46.2 AuxCom: RX(hex)  00 03 --'
           cmd_finished = TRUE;
           if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
            {
#            if( FLUSH_RX_HEX_DUMP_BEFORE_SHOWING_DECODED_COMMAND )
              AuxCom_Debug_FlushRcvdCharHexDump();  // FIRST show the received command as RAW BYTES, ...
#            endif
              ShowError( iErrorClass, "WK-EMU: Rcvd 'Null Command' (0x%02X)",(unsigned int)bRcvdChar );
            }
           break; // end case WK_CMD_NO_OPERATION_IMM
        case WK_CMD_SW_PADDLE_CTRL_IMM: // 0x14 "Software Paddle Control, IMMEDIATE" <14><paddle select>
           // > "This command provides a way to assert paddle inputs from the host.
           // > The PC application would convert keydown codes to Software Paddle
           // > commands. Due to the limited response time of the keyboard,
           // > operating system, and serial communication the best you can do
           // > is around 20 WPM. The paddle watchdog will be used for this
           // > interface as if it were a normal paddle input.
           // >
           // Parameter value : nn = 00 paddle up, n=01 dit, n=02 dah, n=03 both
           if( pInstance->iByteOfImmediateCommand == 0 ) // just received the COMMAND, so set the LENGTH:
            { pInstance->nBytesOfImmediateCommand = 2;
            }
           else // Now bRcvdChar is the "Software Paddle" parameter value, thus we're 'finished' ->
            { cmd_finished = TRUE;
              if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
               {
#               if( FLUSH_RX_HEX_DUMP_BEFORE_SHOWING_DECODED_COMMAND )
                 AuxCom_Debug_FlushRcvdCharHexDump();  // FIRST show the received command as RAW BYTES, ...
#               endif
                 ShowError( iErrorClass, "WK-EMU: Rcvd 'SoftwarePaddle(0x%02X)'",(unsigned int)bRcvdChar );
               }
            }
           break; // end case WK_CMD_SW_PADDLE_CTRL_IMM
        case WK_CMD_GET_STATUS_BYTE_IMM: // 0x15 "Request Winkeyer2 status, IMMEDIATE"  <15> (no parameters, single-byte-command)
           // > "This command is used to queue a request to Winkeyer2 to send
           // > its current operating state. The status byte returned consists of
           // > a bit field that is defined by the following table.
           // > The three MSBs of the the status byte are always 110.
           // > Note that in WK2 mode bit 3 identifies the status byte as a
           // > pushbutton status byte. WK2 mode is set by the ADMIN command 11
           // > before WK is opened.
           // > K1EL Winkeyer2 CW Keyer for Windows
           // ,--------------------------------------------------------------,
           // | Status | Name  | Definition                                  |
           // |  Bit   |       |                                             |
           // |--------+-------+---------------------------------------------|
           // | 7 (MSB)| Tag   | 1                                           |
           // | 6      | Tag   | 1                                           |
           // | 5      | Tag   | 0                                           |
           // | 4      | WAIT  | WK is waiting for an internally timed event |
           // | 3      |KEYDOWN| Keydown status (Tune) 1 = keydown           |
           // | 2      | BUSY  | Keyer is busy sending Morse when = 1        |
           // | 1      |BREAKIN| Paddle break-in active when = 1             |
           // | 0 (LSB)| XOFF  | Buffer is more than 2/3 full when = 1       |
           // '--------------------------------------------------------------'
           //  ( Figure 16  Winkeyer2 Status Definition WK1 compatible mode,
           //    from Winkeyer2 IC Interface & Operation Manual 4/15/2008 Page 12)
           //
           // Three upper bits = "Tag" (0b110), the other bits are set or cleared all over the place :)
           if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
            {
#            if( FLUSH_RX_HEX_DUMP_BEFORE_SHOWING_DECODED_COMMAND )
              AuxCom_Debug_FlushRcvdCharHexDump();  // FIRST show the received command as RAW BYTES, ...
#            endif
              ShowError( iErrorClass, "WK-EMU: Rcvd cmd 'GetStatus', answered \"0x%02X\"",
                                      (unsigned int)pInstance->bWKStatus  );
            }
           Winkeyer_SendByte( pInstance, pInstance->bWKStatus, "GetStatus-Response" );
           pInstance->bReportedWKStatus = pInstance->bWKStatus; // no need for an UNSOLICITED report now
           cmd_finished = TRUE; // single-byte command, thus "immediately finished"
           break; // end case WK_CMD_GET_STATUS_BYTE_IMM
        case WK_CMD_BUFFER_POINTER: // 0x16 "Buffer pointer commands, IMMEDIATE" <16><cmd>....
           // > This command allows the host app to manipulate the input buffer
           // > for special situations such as "on the fly" callsign correction.
           // > Four commands make up the pointer command set:
           // >   nn=00 Reset input buffer pointers to start of buffer,
           // >          only issue this when buffer is empty. (hmm..)       : <0x16><0x00>
           // >   nn=01 Move input pointer to new position in overwrite mode : <0x16><0x01><MANUAL_OMITTED_THIS_PARAMETER>
           // >   nn=02 Move input pointer to new position in append mode    : <0x16><0x02><MANUAL_OMITTED_THIS_PARAMETER>
           // >   nn=03 Add multiple nulls to the buffer                     : <0x16><0x03><number of nulls>
           // > A detailed description of the pointer command is detailed in a separate application note.
           //     '--> A-ha. Maybe that "application note" makes it clearer that
           //          not only subcommand "nn=03" expects an extra parameter (thus THREE BYTES IN TOTAL)
           //          but also subcommands "nn=01" and "nn=02".
           // > I just discovered a bug in N1MM keyer control.
           // > When N1MM starts sending a text, e.g. CQ or exchange, it sends
           // > "buffered speed change" command before the actual text (command 0x1C)
           // > but there is always an extra 0x00 byte before this command
           // > which is deviation from the Winkeyer protocol:
           // > 0x00 means that Admin command follows, and besides there is
           // > no such admin command 0x1C, it is not what is specified in the
           // > Winkeyer manual.                     (Response from N2IC):
           // > This is not a bug.
           // > This is the third byte in the command to reset the Winkey buffer
           // >  to position 0:  PutInt(POINTER_CMD)
           // >                  PutInt(PTR_SETINS)
           // >                  PutInt(0)
           // >  No change will be made.             (Response from HA5SE):
           // > Upon the manual, for me it seemed clear, that only
           // > ["Buffer Pointer"] nn=03 is three bytes long,
           // > nn=00, 01 and 02 are only length two. Accordingly, upon the manual
           // > nn=00 and nn=02 seemed for me to be the same length.
           // > I stay corrected, thank you.
           // > Well, I admit, the reason for using that nn=02 was anyway unclear to me,
           // > immediately after clear-buf and reset-buffer-pointer.
           // > Unfortunately, the functions of 01, 02 and 03 are unclear to me,
           // > without having access to the
           // >  "detailed description of the pointer command is detailed in a separate application note". :)
           // EXACTLY the same happened here: Stumbled over the documentation
           //     (which failed to mention that   "<0x16><01>" and "<0x16><02>"
           //      expect a THIRD BYTE, just like "<0x16><03><number of nulls>"),
           //     and WB also did not have acccess to that mysterious
           //       "detailed description of the pointer command is detailed in a separate application note".
           //     Unfortunately, K1EL never jumped in to clarify what he meant.
           // ,-----------------------------------------------------------------------------------,
           // | Short form: A CORRECT specification of the "Buffer Pointer" commands would be:    |
           // |  <0x16><0x00>  (no further argument) : "Reset input buffer pointer" (no PLURAL)   |
           // |  <0x16><0x01><new_position> : Move input pointer to new position in overwrite mode|
           // |  <0x16><0x02><new_position> : Move input pointer to new position in append mode   |
           // |  <0x16><03><number of nulls>: Add multiple nulls to the buffer                    |
           // '-----------------------------------------------------------------------------------'
           // Continuing the thread at groups.io / N1MMLoggerPlus:
           // >
           // > And what is not very clear to me is why N1MM first resets the buffer
           // > to zero (0x16 0x00) and follows by "append" at the start of the buffer.
           // > I would suppose that 0x16 0x00 would clear the buffer and then
           // > there is no need for "append" at position zero.
           // > I assume this comes from your testing with a real Winkeyer,
           // > but in that case its internal implemenation must be pretty confusing.
           // > FYI, misunderstanding the WK manual (which, BTW,
           // >            is not very specific in some points)
           // > I implemented 0x16 0x00 as "reset buffer" (which is a circular
           // > buffer to make things simple) and that by default means
           // > the buffer contains zero bytes and so the only possibility
           // > is to append new characters at the end of empty string.
           // > Now I understand that maybe Winkeyer internally has fixed start,
           // > end pointer and current pointer, way too complicated for the
           // > simple task one needs from the keyer.
           //
           //   DL4YHF : CAN ONLY SECOND THAT. Confusing, and way too complicated.
           //            But N1MM Logger+ uses this feature to start sending a
           //            contest report as follows:
           //              * type in the first characters of the call
           //              * press the INSERT key to already start sending <call 5nn>
           //              * quickly type in the missing characters of the call,
           //                before the report ("exchange") has been finished.
           // Full story (and a good explanation by Brian, N3OC) in
           // Remote_CW_Keyer\Bugs\2025_08_16_N1MM and RCWK with winkeyer emulation.txt !
           //
           if( pInstance->iByteOfImmediateCommand == 0 ) // just received the COMMAND, so set the LENGTH:
            { pInstance->nBytesOfImmediateCommand = 3; // <- That's only correct if the not-yet-received <cmd> is NOT ZERO. Buah.
            }
           else // Now bRcvdChar is the "Pointer command" parameter value, any we MAY be 'finished' ->
           if( pInstance->iByteOfImmediateCommand == 1 )
            {
              pInstance->bImmediateSubcommand = bRcvdChar; // save this for the next call, we don't have the PARAMETER BYTE yet !
              if( bRcvdChar == 0x00 ) // > "Add multiple nulls to the buffer <16><03><number of nulls>"
               { pInstance->nBytesOfImmediateCommand = 2;
                 cmd_finished = TRUE;
               }
              else // now expecting to receive the NUMBER OF NULLS (or, as discovered later, the NEW POSITION)
               { // NOT finished yet (because pInstance->nBytesOfImmediateCommand=3,
                 //                      but  pInstance->iByteOfImmediateCommand =1)
               }
            }
           else // here, bRcvdChar is either the "number of nulls" (or, as discovered later, the NEW POSITION)
            {
              cmd_finished = TRUE;
            }
           if( cmd_finished )  // got all we need to "interpret" these "Buffer Pointer"-commands now ?
            { // [in]  bRcvdChar : command-depending meaning (number of "multiple nulls" or "new buffer position")
              // [out] the buffer that would normally be filled (written to)
              //       via Winkeyer_EnterByteInDataFIFO( pInstance, bRcvdChar ) :
              //         pInstance->b128Fifo[ pInstance->bFifoHead++ ] = bData;
              //       To SIMPLIFY THIS: Our b128Fifo[] is a CIRCULAR BUFFER by nature.
              //       But since N1MM always seems to send cmd 0x16 0x00 = "Reset input buffer pointer",
              //       and in many cases even cmd 0x0A = "Clear input buffer" in addition,
              //       assume all this 'buffer pointer trickery' happens LONG BEFORE
              //       pInstance->bFifoHead reaches the wrapping point (128 bytes).
              pszToken = sz80Comment;
              switch( pInstance->bImmediateSubcommand ) // .. after "Pointer Command" (0x16) :
               { case 0x00: // <0x16><0x00> (no further argument) : "Reset input buffer pointer"
                    pszToken = "Reset Buffer";
                    // K1EL uses the PLURAL FORM as quoted below ..
                    //  > Reset input buffer pointers to start of buffer
                    // What he possibly meant (with his own (PIC?-)firmware in mind) was:
                    // Reset both the HEAD- and TAIL INDEX to zero, which essentially
                    // CLEARS the entire 128-byte circular buffer, and avoids having
                    // to screw around with CIRCULAR BUFFER INDICES when implementing
                    // the "move input pointer"-commands implemented further below.
                    Winkeyer_ClearInputBuffer(pInstance); // -> e.g. pInstance->bFifoHead = pInstance->bFifoTail = 0;
                    // At this point, pInstance->b128Fifo[ pInstance->bFifoHead ] is where the next CHARACTER TO SEND will be stored,
                    // unless the two confusing "Move input pointers" implemented below modify pInstance->bFifoHead .
                    pInstance->bFifoOverwritePos = 0; // also clear the esoteric position of the "input pointer in/for(?) K1EL OVERWRITE-mode"
                    // Lacking that mysterious document mentioned in MANY Winkeyer manuals,
                    //  > "A detailed description of the pointer command is detailed in a separate application note",
                    // we can only speculate about being in "APPEND" or "OVERWRITE" mode after this "pointer command".
                    // But since BOTH POINTERS (here: pInstance->bFifoHead *and* pInstance->fWKOverwriteMode)
                    // are ZERO at this point, the input buffer is EMPTY, so it appeared logic that
                    // and 'sendable character' will be APPENDED to the empty buffer. Thus:
                    pInstance->fWKOverwriteMode = FALSE; // FALSE = normal "buffer APPEND mode", writing to b128Fifo[ bFifoHead++ ]
                    break;
                 case 0x01: // <0x16><0x01><new_position> : "Move input pointer to new position in overwrite mode" (O-tone K1EL)
                    // (see musings in C:\cbproj\Remote_CW_Keyer\Bugs\2025_08_16_N1MM and RCWK with winkeyer emulation.txt
                    //  about the speculated difference between "overwrite" and "append" )
                    // [in]  bRcvdChar = new_position
                    // [out] pInstance->bFifoOverwritePos, pInstance->fWKOverwriteMode=TRUE .
                    if( bRcvdChar < 127 )  // respect the size of pInstance->b128Fifo[] ..
                     { // lacking ANY documentation about the WK "pointer commands",
                       // not sure what this "position" means.. the following is
                       // a result of TRIAL AND ERROR !
                       pInstance->bFifoOverwritePos = bRcvdChar;
                     }
                    pInstance->fWKOverwriteMode = TRUE; // here: SET after receiving command 0x16 0x01 <new_position> ("overwrite mode")
                    sprintf(sz80Comment, "Move to pos %d / OVERWRITE", (int)pInstance->bFifoOverwritePos );
                    break;
                 case 0x02: // <0x16><0x02><new_position> : "Move input pointer to new position in append mode" (O-tone K1EL)
                    // [in]  bRcvdChar = new_position
                    // [out] pInstance->bFifoHead (that's the "normal" index to APPEND to the FIFO), pInstance->fWKOverwriteMode=FALSE .
                    if( bRcvdChar < 127 )  // respect the size of pInstance->b128Fifo[] ..
                     { pInstance->bFifoHead = bRcvdChar;
                     }
                    pInstance->fWKOverwriteMode = FALSE; // here: CLEARED after receiving command 0x16 0x02 <new_position> ("append mode")
                    sprintf(sz80Comment, "Move to pos %d / APPEND", (int)pInstance->bFifoHead );
                    break;
                 case 0x03: // <0x16><03><number of nulls>: "Add multiple nulls to the buffer" (O-tone K1EL)..
                    // but guess "add" means APPEND here, or is is OVERWRITE ?
                    // Which of the two "Buffer Pointers" are involved, the one to APPEND or the one to OVERWRITE ?
                    sprintf(sz80Comment, "Append %d Nulls", (int)bRcvdChar );
                    for( i=0; i<bRcvdChar; ++i )
                     { Winkeyer_EnterByteInDataFIFO( pInstance, 0x00 ); // -> pInstance->b128Fifo[ pInstance->bFifoHead++ ] = 0x00 (circular wrapping)
                       //  '--> Since 2025-10-11, aware of pInstance->fWKOverwriteMode and pInstance->bFifoOverwritePos !
                       // See log from RCW-Keyer's "Debug" tab with the WINKEYER traffic
                       // when sending "ABC" <insert to start sending report> "DEF"
                       // in C:\cbproj\Remote_CW_Keyer\Bugs\2025_08_16_N1MM and RCWK with winkeyer emulation.txt :
                       // > 19:08:11.6 AuxCom: RX[01] 0x0A       ; Rcvd cmd ClearInputBuffer
                       // > 19:08:11.6 AuxCom: RX[02] 0x16 00    ; Rcvd 'Pointer Command' 0x00=Reset Buffer
                       // > 19:08:11.6 AuxCom: RX[03] 0x16 02 00 ; Rcvd 'Pointer Command' 0x02=MoveTo/Overwrite-mode (!)
                       // > 19:08:11.6 AuxCom: RX[02] 0x1C 0A    ; "Change Morse speed in WPM, BUFFERED <0x1C><WPM> : 10 WPM (emitted into the CW-generator-feeding-buffer ?)
                       // > 19:08:11.6 AuxCom: RX[06] 0x41 42 43 16 03 0B       ; text "ABC" followed by cmd "Append ELEVEN Nulls" (why ELEVEN ?) [*]
                       // > 19:08:12.1 AuxCom: RX[08] 0x20 35 4E 4E 20 16 01 04 ; text " 5NN " followed by cmd "Move to pos 0x04 / Overwrite")
                       // > 19:08:12.1 AuxCom: RX[04] 0x44 16 02 14 ; text "D" followed by cmd "Move to pos 0x14 / Append mode"
                       // > 19:08:12.4 AuxCom: RX[03] 0x16 01 05    ; Rcvd 'Pointer Command' 0x01=Move/Overwrite
                       // > 19:08:12.4 AuxCom: RX[04] 0x45 16 02 14 ; text "E" followed by cmd "Move to pos 0x14 / Append mode"
                       // > 19:08:12.7 AuxCom: RX[03] 0x16 01 06    ; Rcvd 'Pointer Command' 0x01=Move/Overwrite
                       // > 19:08:12.7 AuxCom: RX[04] 0x46 16 02 14 ; text "F" followed by cmd "Move to pos 0x14 / Append mode"
                       //
                       // [*] Contents of pInstance->b128Fifo[] immediately after "Add multiple nulls"-command:
                       //                 ,--- "Buffer Pointer" (here: bFifoOverwritePos) after entering the 0x1C 0x0A "ABC" : 5
                       //                \|/
                       // ,------------------------------------------------------------------------,
                       // |1C|0A|'A|'B|'C|oo|oo|oo|oo|oo|oo|oo|oo|oo|oo|oo| <----- undefined ----> |
                       // '--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+---'
                       //  00 01 02 03 04 05 06 07 08 09 10 11 12 13 14 15 16 17 18 19 20 21 22 24 <- DECIMAL, zero-based byte indices
                       //                                                 /|\   .
                       //                                                  |    .
                       //   ,----------------------------------------------'    .
                       //  bFifoOverwritePos for the next *input* --' = 5 + 11 = 16 .
                       //  Because bFifoOverwritePos had EXCEEDED bFifoHead,
                       //  bFifoHead has ALSO been incremented in Winkeyer_EnterByteInDataFIFO() !
                     }
                    break;
                 default:   // ignore this sub-command, whatever it is
                    pszToken = "UNKNOWN";
                   break;
               } // end switch( pInstance->bImmediateSubcommand )
            }
           if( cmd_finished && (AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC) )
            {
#            if( FLUSH_RX_HEX_DUMP_BEFORE_SHOWING_DECODED_COMMAND )
              AuxCom_Debug_FlushRcvdCharHexDump();  // FIRST show the received command as RAW BYTES, ...
#            endif
              ShowError( iErrorClass, "WK-EMU: Rcvd 'Pointer Command' 0x%02X=%s",
                         (unsigned int)pInstance->bImmediateSubcommand, pszToken  );
              // To understand what went wrong / what was not expained in the non-existing
              // "detailed description of the pointer command is detailed in a separate application note" (by K1EL),
              // dump the current state of the "input buffer" (as many bytes as the
              // TWO buffer-head-indices, pInstance->bFifoHead and pInstance->bFifoOverwritePos, indicate:
              Winkeyer_DumpDataFIFOToLog( pInstance );
            }
           break; // end case WK_CMD_BUFFER_POINTER
        case WK_CMD_DIT_DAH_RATIO_IMM:  // 0x17 "Set Dit/Dah Ratio, IMMEDIATE"       <17><ratio>
           // From K1EL:
           //  > Some ops use this option to make their CW sound less "machine like".
           // From DL4YHF:
           //  > Pfui Deibel !
           if( pInstance->iByteOfImmediateCommand == 0 ) // just received the COMMAND, so set the LENGTH:
            { pInstance->nBytesOfImmediateCommand = 2;
            }
           else // Now bRcvdChar is the "ratio" parameter value, thus we're 'finished' ->
            { cmd_finished = TRUE;
              if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
               {
#               if( FLUSH_RX_HEX_DUMP_BEFORE_SHOWING_DECODED_COMMAND )
                 AuxCom_Debug_FlushRcvdCharHexDump();  // FIRST show the received command as RAW BYTES, ...
#               endif
                 ShowError( iErrorClass, "WK-EMU: Rcvd SetDitDahRatio(%d)",(int)bRcvdChar );
               }
            }
           break; // end case WK_CMD_DIT_DAH_RATIO_IMM
        // - - - Above: "IMMEDIATE" commands. Below: "BUFFERED" commands. - - -
        // BUFFERED commands (like 0x18=WK_CMD_PTT_ON_OFF_BUFFERED .. 0x1F=WK_CMD_NO_OPERATION_BUFFERED)
        //       must be passed on from the serial-port-RX-buffer-draining loop
        //       into the emulated "128 byte FIFO" in Winkeyer 2.
        //       But even though they are not PROCESSED here yet,
        //       the streaming parser for IMMEDIATE commands must keep track
        //       of those BUFFERED commands, to avoid misinterpreting the
        //       parameter (2nd byte) of a BUFFERED command as the next IMMEDIATE command !
        // Thus the following... : on reception of a TWO-BYTE(!) BUFFERED command,
        //       set the byte-number (e.g. 2 bytes for )
        //       to pInstance->nBytesOfImmediate(!)Command :
        case WK_CMD_PTT_ON_OFF_BUFFERED:  // 0x18 "PTT On/Off, BUFFERED" <18><0 or 1> 0=PTT on, 0=PTT off
           // no break ! Like many other BUFFERED commands, passed on to Winkeyer's output FIFO further below
        case WK_CMD_TIMED_KEY_DOWN_BUFF:  // 0x19 "Timed Key Down, BUFFERED"     <19><secs>
           // no break !
        case WK_CMD_WAIT_N_SECONDS_BUFF:  // 0x1A "Wait for N seconds, BUFFERED" <1A><secs>
           // no break !
        // No-No: case WK_CMD_MERGE_TWO_CHARS_BUFF: // 0x1B "Merge two characters into a prosign <1B>[c][c]
  //EX: case WK_CMD_SET_MORSE_SPEED_BUFF: // 0x1C "Speed Change 1C Buff Change Morse speed <1C><WPM> (for the PARSER, not for the "CW Buffer"?)
           // no break !
        case WK_CMD_SET_HSCW_SPEED_BUFF:  // 0x1D "HSCW Speed Change, buffered"     <1D><lpm/100>
           // no break !
        // No-No: case WK_CMD_CANCEL_BUFFERED_SPEED_CHANGE: // 0x1E "Cancel Buffered Speed Change", <1E> without parameters
        // No-No: case WK_CMD_NO_OPERATION_BUFFERED: // 0x1F "Buffered NOP"  <1F> without parameters
           if( pInstance->iByteOfImmediateCommand == 0 ) // just received the COMMAND, so set the LENGTH:
            {  pInstance->nBytesOfImmediateCommand = 2; // <- ".. of IMMEDIATE command", even though it's a BUFFERED command !
            }
           pass_to_fifo = TRUE;
           cmd_finished = ( pInstance->iByteOfImmediateCommand == (pInstance->nBytesOfImmediateCommand-1) );
           if( cmd_finished )
            { if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
               {
#               if( FLUSH_RX_HEX_DUMP_BEFORE_SHOWING_DECODED_COMMAND )
                 AuxCom_Debug_FlushRcvdCharHexDump();  // FIRST show the received command as RAW BYTES, ...
#               endif
                 ShowError( iErrorClass, "WK-EMU: Passed BUFFERED cmd 0x%02X (%s, %d bytes) to FIFO",
                   (unsigned int)pInstance->iWKImmediateCmdParserState,
                    WK_CommandToString(pInstance->iWKImmediateCmdParserState),
                   (int)pInstance->nBytesOfImmediateCommand );
               }
            }
           break;  // end case < encountered a TWO-BYTE BUFFERED command in the serial-port-draining loop >
        case WK_CMD_SET_MORSE_SPEED_BUFF: // 0x1C "Speed Change 1C Buff Change Morse speed <1C><WPM> (for the PARSER, not for the "CW Buffer"?)
           // Since 2025-10-14, this TWO-BYTE-COMMAND (for the "input parser")
           // seems to be converted into a SINGLE-BYTE-COMMAND for the "128-byte buffer"
           // because otherwise, K1EL's *undocumented* "buffer pointer commands"
           // would fail. See reverse-engineered tests and logs documented in
           //  Bugs/2025_08_16_N1MM and RCWK with winkeyer emulation.txt !
           if( pInstance->iByteOfImmediateCommand == 0 ) // just received the COMMAND, so set the LENGTH:
            {  pInstance->nBytesOfImmediateCommand = 2; // <- ".. of IMMEDIATE command", even though it's a BUFFERED command !
            }
           else // Now bRcvdChar is the "Morse sending speed" in WPM, so we're 'finished' ...
            {
              // For reasons already explained (undocumented "buffer pointer commands"),
              // THIS PARTICULAR COMMAND ("Change Speed Buffered") and its parameter value
              // must be encoded IN A SINGLE BYTE - because N1MM Logger+ sends it
              // at the begin of a contest exchange, after typing e.g. the first
              // two characters of the caller's call, then pressing INSERT,
              // then typing in the rest of the call.
              if( bRcvdChar > 63 )  // can only afford SIX bits for the speed in WPM..
               {  bRcvdChar = 63;   // so stick with a MAXIMUM SPEED of 64 WPM, even for a "contest exchange" (the '5nn')
               }
              bRcvdChar |= 0xC0;  // set bits 7 and 6 for the SINGLE-BYTE variant of "Change Speed Buffered" in pInstance->b128Fifo[] !
              cmd_finished = TRUE;
              pass_to_fifo = TRUE;  // ! (bRcvdChar, now with 0xC0 + the new speed in WPM)
              if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
               {
#               if( FLUSH_RX_HEX_DUMP_BEFORE_SHOWING_DECODED_COMMAND )
                 AuxCom_Debug_FlushRcvdCharHexDump();  // FIRST show the received command as RAW BYTES, ...
#               endif
                 ShowError( iErrorClass, "WK-EMU: Convert 'Change Speed Buffered' (%d WPM) to single byte (0x%02X)",
                     (int)( bRcvdChar & 0x3F ), (unsigned int)bRcvdChar );
               }
            } // end if < received the 'WPM' for WK_CMD_SET_MORSE_SPEED_BUFF >
           break; // end case WK_CMD_SET_MORSE_SPEED_BUFF (0x1C <WPM>)

        // Similar for the reception of any THREE-BYTE(!) buffered command:
        case WK_CMD_MERGE_TWO_CHARS_BUFF: // 0x1B "Merge two characters into a prosign <1B>[c][c]
           if( pInstance->iByteOfImmediateCommand == 0 ) // just received the COMMAND, so set the LENGTH:
            { pInstance->nBytesOfImmediateCommand = 3;  // <- ".. of IMMEDIATE command", even though it's a BUFFERED command !
            }
           cmd_finished = ( pInstance->iByteOfImmediateCommand == (pInstance->nBytesOfImmediateCommand-1) );
           if( cmd_finished )
            { if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
               {
#               if( FLUSH_RX_HEX_DUMP_BEFORE_SHOWING_DECODED_COMMAND )
                 AuxCom_Debug_FlushRcvdCharHexDump();  // FIRST show the received command as RAW BYTES, ...
#               endif
                 ShowError( iErrorClass, "WK-EMU: Passed BUFFERED cmd 0x%02X (%s, %d bytes) to FIFO",
                   (unsigned int)pInstance->iWKImmediateCmdParserState,
                    WK_CommandToString(pInstance->iWKImmediateCmdParserState),
                   (int)pInstance->nBytesOfImmediateCommand );
                }
            }
           pass_to_fifo = TRUE;
           break;  // end case < encountered a THREE-BYTE BUFFERED command in the serial-port-draining loop >
        case WK_CMD_CANCEL_BUFFERED_SPEED_CHANGE: // 0x1E = "Cancel Buffered Speed Change", without parameters
           // no break !
        case WK_CMD_NO_OPERATION_BUFFERED: // 0x1F = "Buffered NOP", without parameters
           cmd_finished = TRUE;
           pass_to_fifo = TRUE;
           if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
            {
#            if( FLUSH_RX_HEX_DUMP_BEFORE_SHOWING_DECODED_COMMAND )
              AuxCom_Debug_FlushRcvdCharHexDump();  // FIRST show the received command as RAW BYTES, ...
#            endif
              ShowError( iErrorClass, "WK-EMU: Passed BUFFERED cmd 0x%02X (%s) to FIFO",
                        (unsigned int)bRcvdChar, WK_CommandToString(bRcvdChar) );
            }
           break;  // end case < encountered a SINGLE-BYTE BUFFERED command in the serial-port-draining loop >

        default :  // e.g. iWKImmediateParserState < 0 :
           // Obviously not a "command" (neither "Immediate" nor "Buffered")
           //   but "data" so enter in Figure 3's "128 byte FIFO":
           pass_to_fifo = TRUE; // pass whatever is in bRcvdChar now into the 128-byte FIFO (further below)
           // with pInstance->iWKImmediateParserState < 0, bRcvdChar may be
           // a 'normal transmitable character' or a 'buffered command'.
           // They will be processed LATER (further below), when they pour
           // out of Winkeyer's 128-byte-FIFO .
           break;
      } // end switch( pInstance->iWKParserState )
     if( cmd_finished )
      { pInstance->iWKImmediateCmdParserState = -1;
      } // end if( cmd_finished )
     if( pass_to_fifo )
      { Winkeyer_EnterByteInDataFIFO( pInstance, bRcvdChar ); // -> basically, pInstance->b128Fifo[ pInstance->bFifoHead++ ], plus circular buffer index wrap
        fDataFIFOModified = TRUE; // <-- a reason to call Winkeyer_DumpDataFIFOToLog() further below ?
      } // end if( pass_to_fifo )

   } // end while < more bytes in the 'auxiliary COM port's RX buffer >

  if( ! pInstance->fWKImmediatePauseState ) // FALSE=not paused, TRUE=paused from cmd "Set Pause State" <06><nn>
   { // To properly implement the "buffered speed changes" in this FIFO,
     // do NOT fill the buffer in the Remote CW Keyer's own CW generator (CwGen.c)
     // is already busy from a transmission:
     if( pInstance->pCwGenerator != NULL )   // got an associated "CW generator" at all ?
      { T_CwGen *pCwGen = (T_CwGen*)pInstance->pCwGenerator;
        BOOL fCwGenBusy = CwGen_IsTxBusy( pCwGen );
        // About the purpose of CwGen_IsTxBusy() [called below] :
        //  > If this function indicates "CW generator NOT busy from transmission",
        //  > the Winkeyer-emulation may e.g. change the keying speed between two
        //  > characters, without spoiling the CW timing.
        // Since 2025-10-12, Winkeyer's "Echo Morse in ASCII" is sent
        // in the very moment that CwGen_IsTxBusy() changes from TRUE to FALSE again !
        if( (!fCwGenBusy ) && (pInstance->bCharacterCurrentlySent != 0x00) )
         {
           //
           // Depending on what K1EL calls "Mode Register",
           // we may have to 'echo back' the finished character(s) to the
           // Winkeyer host (depends on what the "WinKeyer3 IC Interface & Operation Manual 3/19/2019"
           // explains on page 11, "Table 12  WK3 Mode Selection Table",
           // with stuff like "Paddle Echoback" (bit 6 in the "mode register")
           //             and "Serial Echoback" (bit 2 in the "mode register").
           // In THIS WINKEYER EMULATION, that's in Winkeyer_EEPROM.s.modereg :
           if( Winkeyer_EEPROM.s.modereg & WK_MODE_MASK_SERIAL_ECHO )
            {
              sprintf( sz80Comment, "Echo ASCII '%c', hex", (char)pInstance->bCharacterCurrentlySent );
              Winkeyer_SendByte( pInstance, pInstance->bCharacterCurrentlySent, sz80Comment );
              // 2025-10-12 : The question was what N1MM Logger 'makes of this'.
              //              See speculations about the undocumented "Pointer Commands",
              //              and how N1MM tosses the dice for the "OVERWRITE POSITION"
              // - grep for the date (2025-10-12) in C:\cbproj\Remote_CW_Keyer\Bugs\2025_08_16_N1MM and RCWK with winkeyer emulation.txt !
              // MAYBE, in the original K1EL Winkeyer chip, the NEXT character
              // is already moved from the 128-byte "input buffer" into
              // the CW-transmit-shift register when the chip sends the
              // 'Echo Morse in ASCII' for the PREVIOUS character ?
              // Google's AI speculated when asked about "Winkeyer Echo Morse in ASCII":
              //  > The "echo" part of the term means that after the
              //  > WinKeyer sends the Morse code, it also echoes back
              //  > to the host computer (the PC) the same character
              //  > that was sent. This allows the computer software
              //  > to display what the operator is sending in a visible
              //  > text format, with the echo occurring after (!)
              //  > the Morse code has been completely (!) transmitted.
              // Thus moved the transmission of the "Echo Morse in ASCII" HERE
              //  (executed when CwGen_IsTxBusy( pCwGen ) JUST returned FALSE).
            } // end if < "modereg" configured for "Serial Echoback" > ?
           pInstance->bCharacterCurrentlySent = 0x00;  // don't send the ECHO twice
         } // end if( (!fCwGenBusy ) && (pInstance->bCharacterCurrentlySent != 0x00) )
        if( (!fCwGenBusy )  // what's going on in Remote_CW_Keyer/CwGen.c ?
          || ( pCwGen->iState==CW_GEN_OFF)   // CwGen_Handler() won't do anything yet [bugfix 2025-10-04]
          || ( pCwGen->iState==CW_GEN_START) // CwGen_Handler() will START if there's something to send
          )
         { // Ok... time to pull the next character from Winkeyer's "Data FIFO" :
           if( Winkeyer_ReadByteFromDataFIFO( pInstance, &bRcvdChar ) )
            { // Here, bRcvdChar may be a CHARACTER TO SEND, but also a "buffered command" !
              // As shown in WinkeyUSBman.pdf (v22) page 2, Figure 3,
              // a byte has been pulled from the 128-byte FIFO "to input processing".
              // That doesn't mean it's a CHARACTER TO SEND !
              // Similar as bytes entering the "input parser" (first stage, above),
              // bRcvdChar may be "command" or "a character to send" !
              // But in THIS case, we only need to process what the manual calls
              // "Buffered Commands" (listed in WinkeyUSBman.pdf on pages 13 .. 14) :
              if( pInstance->iWKBufferedCmdParserState < 0 ) // NOT parsing a BUFFERED command yet, so...
               { if( (bRcvdChar>=0x18) && (bRcvdChar<=0x1F) ) // is this a BUFFERED command a la Winkeyer ?
                  { pInstance->iWKBufferedCmdParserState = (int)bRcvdChar; // kick the PARSER STATE MACHINE alive
                    pInstance->iByteOfBufferedCommand = 0;
                    pInstance->nBytesOfBufferedCommand= 1; // "initial guess" (more in the switch/case below)
                    // 2025-10-04: Got here with bRcvdChar=0x1C=WK_CMD_SET_MORSE_SPEED_BUFF
                    //   when clicking one of N1MM's "Morse-sending hotkeys", e.g. "F1 Cq" .
                    // Further below, after 'switch( pInstance->iWKBufferedCmdParserState )',
                    // the 'initial guess'   (pInstance->nBytesOfBufferedCommand=1)
                    // was correctly revised (pInstance->nBytesOfBufferedCommand=2) .
                  }
               }
              else // ALREADY PARSING a buffered command (that has been started in a previous loop) ->
               { ++pInstance->iByteOfBufferedCommand; // just got the next byte BELONGING TO AN ALREADY RECEIVED COMMAND-BYTE
               }
              cmd_finished = (pInstance->iByteOfBufferedCommand >= pInstance->nBytesOfBufferedCommand);
                // '--> also for the BUFFER-streaming-parser, that's just an assumption
              switch( pInstance->iWKBufferedCmdParserState )
               { // Keep it simple.. iWKBufferedParserState also uses the code of the HOST COMMAND,
                 // but in this case we only need to look for BUFFERED commands,
                 // beginning with code 0x18 ("<18>" is hexadecimal, as specified on page 5:
                 // > In this document a hex value will be presented in angle brackets, for example <02>
                 case WK_CMD_PTT_ON_OFF_BUFFERED:  // 0x18 "PTT On/Off, BUFFERED" <18><0 or 1> 0=PTT on, 0=PTT off
                    if( pInstance->iByteOfBufferedCommand == 0 ) // just received the COMMAND, so set the LENGTH:
                     { pInstance->nBytesOfBufferedCommand = 2;
                     }
                    else // Now bRcvdChar is the "PTT on/off flag" -> time to switch the PTT...
                     { cmd_finished = TRUE;
                       if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
                        { // No need for AuxCom_Debug_FlushRcvdCharHexDump() here
                          // for this (and the other) BUFFERED commands !
                          ShowError( iErrorClass, "WK-EMU: Buffered 'Set PTT %s'",
                              (char*)( bRcvdChar == 0 ) ? "off" : "ON" );
                        }
                     }
                    break;
                 case WK_CMD_TIMED_KEY_DOWN_BUFF:  // 0x19 "Timed Key Down, BUFFERED"     <19><secs>
                    if( pInstance->iByteOfBufferedCommand == 0 ) // just received the COMMAND, so set the LENGTH:
                     { pInstance->nBytesOfBufferedCommand = 2;
                     }
                    else // Now bRcvdChar is the "PTT on/off flag" -> time to switch the PTT...
                     { cmd_finished = TRUE;
                       if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
                        { ShowError( iErrorClass, "WK-EMU: Buffered 'Timed KEY DOWN %d seconds'",
                                                  (int)bRcvdChar );
                        }
                     }
                    break;
                 case WK_CMD_WAIT_N_SECONDS_BUFF:  // 0x1A "Wait for N seconds, BUFFERED" <1A><secs>
                    if( pInstance->iByteOfBufferedCommand == 0 ) // just received the COMMAND, so set the LENGTH:
                     { pInstance->nBytesOfBufferedCommand = 2;
                     }
                    else // Now bRcvdChar is the "PTT on/off flag" -> time to switch the PTT...
                     { cmd_finished = TRUE;
                       if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
                        { ShowError( iErrorClass, "WK-EMU: Buffered 'Wait for %d seconds'",
                                                  (int)bRcvdChar );
                        }
                     }
                    break;
                 case WK_CMD_MERGE_TWO_CHARS_BUFF: // 0x1B "Merge two characters into a prosign <1B>[c][c]
                    switch( pInstance->iByteOfBufferedCommand )
                     { case 0 : // just received the COMMAND, so set the LENGTH:
                            pInstance->nBytesOfBufferedCommand = 3;
                            break;
                       case 1 : // just received the FIRST character that shall be merged into a prosign:
                            pInstance->c4CharsForProsign[0] = '^'; // <- this is for the "Icom-style" PROSIGN SYNTAX
                            pInstance->c4CharsForProsign[1] = bRcvdChar;
                            break;
                       case 2 : // just received the SECOND character that shall be merged into a prosign
                            pInstance->c4CharsForProsign[2] = bRcvdChar;
                            pInstance->c4CharsForProsign[3] = '\0'; // <- this is because CwGen_AppendForReplay() needs a C-string
                            cmd_finished = TRUE;
                            if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
                             { ShowError( iErrorClass, "WK-EMU: Merge %s into a PROSIGN",
                                                  (char*)pInstance->c4CharsForProsign );
                             }
                            CwGen_AppendForReplay( pCwGen, pInstance->c4CharsForProsign );
                            WK_Emu_StartCwGenerator( pCwGen ); // .. if not already running
                            // WB: The question is how to indicate the completion of this
                            //     using WK's 8-bit 'Echo Morse in ASCII'-feature !
                            pInstance->bCharacterCurrentlySent = pInstance->c4CharsForProsign[0];  // only "echo" the FIRST character of a PROSIGN (better than NOT SENDING 'Echo Morse in ASCI' at all)
                            break;
                     } // end switch( pInstance->iByteOfBufferedCommand )
                    break; // end case WK_CMD_MERGE_TWO_CHARS_BUFF
                 case WK_CMD_SET_MORSE_SPEED_BUFF: // 0x1C "Buffered Change Morse speed <1C><WPM>
                    if( pInstance->iByteOfBufferedCommand == 0 ) // just received the COMMAND, so set the LENGTH:
                     { pInstance->nBytesOfBufferedCommand = 2;
                     }
                    else // Now bRcvdChar is the PARAMETER VALUE (here: Words Per Minite)
                     { cmd_finished = TRUE;
                       // RCWK's own 'CW generator' has its own command to temporarily
                       // switch the speed while sending a MESSAGE, but the format
                       // is naturally different from the Winkeyer protocol:
                       sprintf(sz80, "<s%02d>", (int)(bRcvdChar & 0x7F) );
                       CwGen_AppendForReplay( pCwGen, sz80 );
                       // No need to START the generator here yet because eg "<s40>" alone isn't "sendable text".
                       // The "command appended for replay" will occupy a few bytes
                       // in pCwGen->nCharsRemainingToPlay, but that's no problem
                       // because with pCwGen->iState==CW_GEN_OFF or CW_GEN_START,
                       // the value returned by CwGen_IsTxBusy() is irrelevant.
                       if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
                        { ShowError( iErrorClass, "WK-EMU: Buffered 'Change Speed to %d WPM'",
                                                  (int)bRcvdChar );
                        }
                     }
                    break; // end case WK_CMD_SET_MORSE_SPEED_BUFF
                 case WK_CMD_SET_HSCW_SPEED_BUFF:  // 0x1D "HSCW Speed Change, buffered"     <1D><lpm/100>
                    if( pInstance->iByteOfBufferedCommand == 0 ) // just received the COMMAND, so set the LENGTH:
                     { pInstance->nBytesOfBufferedCommand = 2;
                     }
                    else // Now bRcvdChar is the PARAMETER VALUE...
                     { cmd_finished = TRUE;
                       // How to convert Winkeyer's "lpm/100" into WORDS PER MINUTE ?
                       // Drive to PARIS, and remember in some language,
                       //  a "letter" isn't what you put in an envelope
                       //  but a single CHARACTER .. Aaaah ! So ONE "lpm/100"
                       //  from Winkeyer is 100 letters, ahem, characters per minute,
                       //  and those are 100/strlen("Paris") = 20 WORDS per minute . Thus:
                       i = 20 * (int)(bRcvdChar & 0x7F);
                       sprintf(sz80, "<s%02d>", (int)i );
                       CwGen_AppendForReplay( pCwGen, sz80 );
                       if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
                        { ShowError( iErrorClass, "WK-EMU: Buffered 'HSCW Change Speed to %d WPM",(int)i);
                        }
                     }
                    break; // end case WK_CMD_SET_HSCW_SPEED_BUFF
                 case WK_CMD_CANCEL_BUFFERED_SPEED_CHANGE: // 0x1E "Cancel Buffered Speed Change", <1E> without parameters
                    cmd_finished = TRUE;
                    CwGen_AppendForReplay( pCwGen, "</s>" );
                    if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
                     { ShowError( iErrorClass, "WK-EMU: Buffered 'Cancel Speed Change'" );
                     }
                    break; // end case WK_CMD_CANCEL_BUFFERED_SPEED_CHANGE
                 case WK_CMD_NO_OPERATION_BUFFERED: // 0x1F "Buffered NOP"  <1F> without parameters
                    cmd_finished = TRUE;
                    if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
                     { ShowError( iErrorClass, "WK-EMU: Buffered 'NO OPERATION'" );
                       // (Still no idea about the purpose, but anyway.. show what's going on)
                     }
                    break; // end case WK_CMD_NO_OPERATION_BUFFERED
                 default :  // e.g. iWKImmediateParserState < 0 :
                    if( bRcvdChar == 0x00 ) // obviously one of those "multiple nulls" from cmd 0x16 0x03 <number of nulls> ..
                     { //  '--> originates from Winkeyer_ReadByteFromDataFIFO() ~~ pInstance->b128Fifo[ pInstance->bFifoTail++ ] (ignoring the CIRCULAR nature of the FIFO)
                       // Do NOT pass this to the CW generator, because it's nothing "sendable" !
                       if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
                        { ShowError( ERROR_CLASS_TX_TRAFFIC | SHOW_ERROR_TIMESTAMP, "WK-EMU: Skipping 0x%02X in input buffer[%d]",
                                     (unsigned int)bRcvdChar, Winkeyer_WrapDataFifoIndex( (int)pInstance->bFifoTail-1 ) );
                        }
                     }
                    else if( bRcvdChar & 0x80 ) // MSBit in the BUFFERED byte set ? Must be a special SINGLE-BYTE BUFFERED command,
                     { // with the parameter value encoded in the lower bits, e.g. "Change Speed Buffered" (ex: 0x1C <NN>)
                       // As a future reserve for OTHER SINGLE-BYTE BUFFERED COMMANDS WITH PARAMETER IN THE LOWER BITS,
                       // bit SIX tells the "Change Speed Buffered"-command from others:
                       switch( bRcvdChar & 0xC0 )
                        { case 0xC0:   // 0b11xx xxxx : Switch speed in WPM to 'xx xxxx' (6-bit value)
                             sprintf(sz80, "<s%02d>", (int)(bRcvdChar & 0x3F) ); // -> 0..63 WPM 
                             CwGen_AppendForReplay( pCwGen, sz80 );
                             // No need to START the generator here yet because eg "<s40>" alone isn't "sendable text".
                             // The "command appended for replay" will occupy a few bytes
                             // in pCwGen->nCharsRemainingToPlay, but that's no problem
                             // because with pCwGen->iState==CW_GEN_OFF or CW_GEN_START,
                             // the value returned by CwGen_IsTxBusy() is irrelevant.
                             if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
                              { ShowError( iErrorClass, "WK-EMU: Buffered 'Change Speed to %d WPM'", (int)bRcvdChar );
                              }
                             break;
                          case 0x80:  // 0b10xx .... : future reserve for other "single-byte commands" in WK's 128-byte buffer:
                             break;
                          default:
                             break;
                        } // end switch( bRcvdChar & 0xC0 )
                     }
                    else
                     { // Obviously not a "command", and not a "null byte" (filler)
                       //   but "ASCII data to send", so let the RCW Keyer's own CW generator
                       //   start transmission for a single character:
                       sz80[0] = (char)bRcvdChar;
                       sz80[1] = '\0';  // provide a zero-terminated C-string
                       CwGen_AppendForReplay( pCwGen, sz80 ); // here: send 'normal ASCII character' in Morse code
                       WK_Emu_StartCwGenerator( pCwGen );     // start the CW generator if not already running
                       pInstance->bCharacterCurrentlySent = sz80[0];  // remember the currently-being-transmitted for WK's 'Echo Morse in ASCII')

                       // Call Winkeyer_DumpDataFIFOToLog(), after pulling out another 'normal ASCII character' ?
                       if( fDataFIFOModified && (AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC) )
                        {
#                        if( FLUSH_RX_HEX_DUMP_BEFORE_SHOWING_DECODED_COMMAND )
                          AuxCom_Debug_FlushRcvdCharHexDump();
#                        endif
                          Winkeyer_DumpDataFIFOToLog( pInstance );
                        }
                     }
                    break; // end default ("transmittable character in bRcvdChar) ?
               } // end switch( pInstance->iWKBufferedCmdParserState )
              if( cmd_finished ) // here: for the streaming parser AFTER the 128-byte FIFO ..
               { pInstance->iWKBufferedCmdParserState = -1;
               }
            }    // end if( Winkeyer_ReadByteFromDataFIFO( pInstance, &bRcvdChar ) )
         }      // end if( ! CwGen_IsTxBusy( .. ) )
        else   // CwGen_IsTxBusy() returned TRUE = "already busy from an ongoing transmission" ->
         {     // simply DO NOT pull the next "buffered command" or "character to send"
               // out of Winkeyer_ReadByteFromDataFIFO() ? ?
           switch( pCwGen->iState ) // what's going on in Remote_CW_Keyer/CwGen.c ?
            { case CW_GEN_OFF  : // (0) CwGen_Handler() won't do anything
                 break;
              case CW_GEN_START: // (1) CwGen_Handler() will START if there's something to send
                 break;
              case CW_GEN_SEND_DASH_OR_DOT: // (2) sending a dash or dot ->
                 // Don't interfere, because sooner or later, CwGen_IsTxBusy() will retrurn FALSE,
                 // and AuxCom_Winkeyer.c will read the next byte via Winkeyer_ReadByteFromDataFIFO() !
                 break;
              case CW_GEN_SEND_GAP: // sending a gap with ONE to SEVEN dot lengths
                 break;
              case CW_GEN_FINISHED_SENDING: // reached the end of text in szTxMemory[], "nothing more to send", application may decide WHAT ELSE to send.
                 break;
              case CW_GEN_WAITING_FOR_MACRO:// waiting for more text fed into szTxMemory[] before a MACRO like "<sWPM>" can be parsed
                 break;
              default:
                 break;
            } // end switch( pCwGen->iState )
         } // end else < CwGen_IsTxBusy() > ?
      }   // end if( pInstance->pCwGenerator != NULL )
   }     // end if < ok to process bytes from Winkeyer's "128 byte FIFO" >

  // Near the END of the periodically called AuxComPorts_RunWinkeyerEmulator() .. :
  // Periodically update the "Winkeyer Status Byte", because it must be reported
  // to the host autonomously (on change) but also "on request":
  //  [in,out] pInstance->bWKStatus : bitwise combination of C_WINKEYER_STATUS_...
  //             '--> compared against bWKReportedStatus for what the WK docu
  //                  calls "Unsolicited Status Transmission" .
  if( pInstance->pCwGenerator != NULL ) // is there an associated "CW generator" (struct defined in CwGen.h) ?
   { T_CwGen *pCwGen = (T_CwGen*)pInstance->pCwGenerator;
     switch( pCwGen->iState ) // State of DL4YHF's CW generator state machine ?
      { case CW_GEN_OFF  : // CwGen_Handler() won't do anything, so for Winkey, we're "not busy":
        case CW_GEN_START: // CwGen_Handler() would START if there was something to send -> also "not busy"
           pInstance->bWKStatus &= ~C_WINKEYER_WK_STATUS_BUSY; // clear bit 2 ("Keyer is busy sending Morse..")
           break;
        case CW_GEN_SEND_DASH_OR_DOT: // currently sending a dash or dot
           pInstance->bWKStatus |= C_WINKEYER_WK_STATUS_BUSY; // set bit 2 ("Keyer is busy sending Morse..")
           break;
        case CW_GEN_SEND_GAP: // currently sending a gap with ONE to SEVEN dot lengths
           // Set Winkeyer's emulated "BUSY"-bit but not the WK1-"KEYDOWN"-bit (if we emulated a WINKEYER 1, which we don't)
           pInstance->bWKStatus |= C_WINKEYER_WK_STATUS_BUSY; // set bit 2 ("Keyer is busy sending Morse..")
           break;
        case CW_GEN_FINISHED_SENDING: // reached the end of text in szTxMemory[],
           // there is "nothing more to send" .. so "not BUSY" in Winkeyer's status byte:
           pInstance->bWKStatus &= ~C_WINKEYER_WK_STATUS_BUSY; // clear bit 2 ("Keyer is busy sending Morse..")
           break;
        default:   // oops.. a new CW generator state that we don't support here yet ?
           break;
      }  // end switch( pInstance->pCwGenerator->iState )
   }    // end if( pInstance->pCwGenerator != NULL )

  if( pInstance->fWKHostMode )
   { if( pInstance->bWKStatus != pInstance->bReportedWKStatus ) // "Unsolicited Status Transmission" ?
      { // Make sure the three "Tag bits" (7,6,5) are properly set for a WINKEY STATUS byte:
        pInstance->bWKStatus &= ~(C_WINKEYER_WK_STATUS_TAG_MSK    // bits 7..5 fixed to 0b110 ..
                                 |C_WINKEYER2_WK_STATUS_PUSHBTN); // .. and the "PUSHBUTTON message indictor bit" CLEARED
        pInstance->bWKStatus |=  C_WINKEYER_WK_STATUS_TAG_VAL;
        Winkeyer_SendByte( pInstance, pInstance->bWKStatus, "Unsolicited WK Status" );
        // '--> "Since these bytes can arrive at any time and potentially can
        //       be mixed with echo back bytes, they have identifying tags.
        //       If the MSB is set that identifies the byte as unsolicited,
        //       if bit 6 is clear its a speed pot byte or if bit 6 is set
        //       its a status byte. In WK2 mode, bit 3 identifies a pushbutton
        //       status byte."
        if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
         {
#         if( FLUSH_RX_HEX_DUMP_BEFORE_SHOWING_DECODED_COMMAND )
           AuxCom_Debug_FlushRcvdCharHexDump();
#         endif
           ShowError( ERROR_CLASS_TX_TRAFFIC | SHOW_ERROR_TIMESTAMP,
            "WK-EMU: Sent \"Unsolicited WK Status\", new=0x%02X, prev=0x%02X",
            (unsigned int)pInstance->bWKStatus, (unsigned int)pInstance->bReportedWKStatus );
           // To troubleshoot the "Pointer Command" (2025-10), show the new "Data FIFO" state:
           Winkeyer_DumpDataFIFOToLog( pInstance );
         }
        pInstance->bReportedWKStatus = pInstance->bWKStatus;
      }
     // Similar also for the unsolicited "PUSHBUTTON Status" (even-driven transmission) :
     if( pInstance->bPBStatus != pInstance->bReportedPBStatus ) // "Unsolicited Pushbutton Status" ?
      { // Also here, three "Tag bits" (7,6,5) and a
        // are properly set for a WINKEY STATUS byte:
        pInstance->bPBStatus &= ~C_WINKEYER_WK_STATUS_TAG_MSK; // bits 7..5 fixed to 0b110 ..
        pInstance->bPBStatus |=  C_WINKEYER_WK_STATUS_TAG_VAL | C_WINKEYER2_WK_STATUS_PUSHBTN; // .. and the "PUSHBUTTON message indictor bit" SET
        Winkeyer_SendByte( pInstance, pInstance->bPBStatus, "Unsolicited PB Status" );
        // '--> "Since these bytes can arrive at any time and potentially can
        //       be mixed with echo back bytes, they have identifying tags.
        //       If the MSB is set that identifies the byte as unsolicited,
        //       if bit 6 is clear its a speed pot byte or if bit 6 is set
        //       its a status byte. In WK2 mode, bit 3 identifies a pushbutton
        //       status byte."
        if( AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
         { ShowError( iErrorClass, "WK-EMU: Sent \"Unsolicited PB Status\", new=0x%02X, prev=0x%02X",
            (unsigned int)pInstance->bPBStatus, (unsigned int)pInstance->bReportedPBStatus );
         }
        pInstance->bReportedPBStatus = pInstance->bPBStatus;
      }
   } // end if < Winkeyer in "Host Mode" > ?



  if( (AuxCom_iDiagnosticFlags & AUX_COM_DIAG_FLAGS_SHOW_WINKEY_TRAFFIC )
    &&(AuxCom_Debug_GetTimeElapsedSinceLastHexDump_ms() >= 1000 ) )
   { // When non-empty, it's time to emit (flush) the last received bytes
     // as Hex dump for the display:
     AuxCom_Debug_FlushRcvdCharHexDump();
   }


} // end AuxComPorts_RunWinkeyerEmulator()

//---------------------------------------------------------------------------
static void WK_Emu_StartCwGenerator( T_CwGen *pCwGen ) // .. if not already running
{
  if( (pCwGen->iState == CW_GEN_OFF ) // CW-generator not active yet ?
   || (pCwGen->iState == CW_GEN_FINISHED_SENDING) ) // or already ENDED ?
   {   pCwGen->iState = CW_GEN_START; // start it for the buffered character
       // To let the Remote-CW-Keyer-GUI indicate
       //   "Sending from Winkeyer" in the STATUS LINE:
       KeyerGUI_iStatusIndicatorMemIdx = KEYER_MEMORY_INDEX_WINKEYER_EMU;
   }
} // end WK_Emu_StartCwGenerator()

